#include <stdio.h>
#include <unistd.h>
#include <pwd.h>
#include <grp.h>
#include "soccmd.h"
#include "socdefs.h"
#include "socketio.h"
#include "socdemon.h"

/*
 * who_am_i()
 *   - Simulates the unix 'whoami' command.
*/
void who_am_i()
{
  struct passwd *pw_entry;

   pw_entry = getpwuid(getuid());
   tell_user(client_descriptor,"%s (uid:%d)\n",pw_entry->pw_name,pw_entry->pw_uid);
}

/*
 * get_id()
 *  - Simulates the unix 'id' command.
*/
void get_id()
{
   struct passwd *pw_entry;
   struct group  *grp_entry;
   uid_t uid, euid;
   gid_t gid, egid;
   char *username   = NULL;
   char *groupname  = NULL;
   char *eusername  = NULL;
   char *egroupname = NULL;

   uid  = getuid();
   euid = geteuid();
   gid  = getgid();
   egid = getegid();

   pw_entry = getpwuid(uid);
   if (pw_entry != NULL)
      username = pw_entry->pw_name;

   grp_entry = getgrgid(gid);
   if (grp_entry != NULL)
      groupname = grp_entry->gr_name;

   tell_user(client_descriptor, "uid=%d(%s) gid=%d(%s)",
      uid,username,gid,groupname);

   if (uid != euid) {
      /* Effective User ID != User ID (program running setuid) */
      pw_entry = getpwuid(euid);
      if (pw_entry != NULL)
         eusername = pw_entry->pw_name;
      tell_user(client_descriptor," euid=%d(%s)",euid,eusername);
   }
   if (gid != egid) {
      /* Effective Group ID != Group ID (program  running setgid) */
      grp_entry = getgrgid(egid);
      if (grp_entry != NULL)
         egroupname = grp_entry->gr_name;
      tell_user(client_descriptor," egid=%d(%s)",egid,egroupname);
   }
   tell_user(client_descriptor,"\n");
}

/*
 * ch_dir()
 *   - Changes the working directory..
 *   - Note that cd's executed by 'cmd' will only be
 *     effective for that specific command, using this
 *     routine, we have the cd stay after the call to
 *     popen() unlike 'cmd cd XX'
*/
void ch_dir(directory)
char *directory;
{
  struct passwd *pw_entry;
  char   src_dir[MAX_INPUT_LEN];

   if (directory[0] == 0) {
      /* no directory was supplied. Presumably the user wants to */
      /* change back to the user id's home directory, which a    */
      /* call the chdir() with no parms *will not do*. So we     */
      /* gotta get the current userid's home dir from the passwd */
      /* file..                                                  */
      pw_entry = getpwuid(getuid());
      strcpy(src_dir,pw_entry->pw_dir);
   } else {
      strcpy(src_dir, directory);
   }
   if (chdir(src_dir) != 0) {
      tell_user(client_descriptor,"%3.3d - %s", command_failed,
                out_table[command_failed]);
   } else {
      if (debug_mode) {
         tell_user(client_descriptor,"%3.3d - %s", cd_successful,
                   out_table[cd_successful]);
      }
   }
}

/*
 * get_pwd()
 *   - prints out the current working directory.
*/
void get_pwd()
{
  char this_directory[MAX_OUTPUT_LEN];

  if (getcwd(this_directory, MAX_OUTPUT_LEN) == NULL) {
      tell_user(client_descriptor,"%3.3d - %s", command_failed,
                out_table[command_failed]);
  } else {
      tell_user(client_descriptor,"%s\n",
                this_directory);
  }
}

/*
 * report_version()
 *   - This routine reports the program name, date, and version.
*/
void report_version()
{
   tell_user(client_descriptor, "%s version %s, %s.\n",
             SOCDEMON_NAME, SOCDEMON_VER, SOCDEMON_DATE);
}

/*
 * do_unix_command()
 *   - This procedure passes whatever string it gets as 'parm' to
 *     the remote system using popen(), and then returns the output
 *     back to the user.
 *     NOTE: You can't do interactive commands using popen(), so
 *           all though you can see the output, you cannot input
 *           anything besides what you pass on the command line.
 *   - If a nil string is passed, the the routine will prompt the
 *     user for a command to execute.
*/
void do_unix_command(parm)
char *parm;
{
 FILE *cmd_output;
 char in_msg[MAX_INPUT_LEN];
 char cmd_text[MAX_OUTPUT_LEN];
 int  msg_len = 0;

   strcpy(in_msg,parm);
   if (parm[0] == 0) {
      tell_user(client_descriptor,"Command string to execute? : ");
      msg_len = soc_read(client_descriptor, in_msg, MAX_INPUT_LEN);
      if (msg_len < 0) {
         soc_fprintf(stderr,"ERROR(%d): Error reading socket input [%d:%d].\n",
                     SOCKET_INPUT_ERROR, msg_len, errno);
         return;
      }
      if (in_msg[strlen(in_msg)-1] == '\n') {
         in_msg[strlen(in_msg)-2] = 0;
      }
   }
   if (debug_mode) {
      tell_user(client_descriptor,"Passing \"%s\" to remote system.\n", in_msg);
   }
   if ((cmd_output = popen(in_msg,"r")) == NULL) {
      tell_user(client_descriptor,"%3.3d - %s", didnt_execute,
                out_table[didnt_execute]);
   } else {
      while(fgets(cmd_text,MAX_OUTPUT_LEN,cmd_output) != NULL) {
         tell_user(client_descriptor,cmd_text);
      }
      if (ferror(cmd_output)) {
         tell_user(client_descriptor, "%3.3d - %s",
                   command_failed, out_table[command_failed]);
      }
      pclose(cmd_output);
   }
   if (debug_mode) {
      tell_user(client_descriptor, "%3.3d - %s", command_complete,
                out_table[command_complete]);
   }
}
void do_rm(args)
char *args;
{
  char rm_command[MAX_OUTPUT_LEN];

   strcpy(rm_command,"rm ");
   strcat(rm_command,args);
   do_unix_command(rm_command);
}

void do_mv(args)
char *args;
{
  char mv_command[MAX_OUTPUT_LEN];

   strcpy(mv_command,"mv ");
   strcat(mv_command,args);
   do_unix_command(mv_command);
}

void do_cp(args)
char *args;
{
  char cp_command[MAX_OUTPUT_LEN];

   strcpy(cp_command,"cp ");
   strcat(cp_command,args);
   do_unix_command(cp_command);
}

void do_shell()
{
  int childpid;

   tell_user(client_descriptor,"You have entered a unix shell.\n");
   tell_user(client_descriptor,"No prompt is printed, and all commands must have a ';' appended to them\n");
   tell_user(client_descriptor,"(ie: ls -al;) ignore the \"sh: ^M: not found\" messages.\n");
   tell_user(client_descriptor,"enter 'exit;' to return to %s\n",SOCDEMON_NAME);
 
   childpid = fork();
   if (childpid == 0) {
      execl("/bin/sh","sh",(char *) 0);
      fprintf(stderr,"ERROR: Fatal error\n");
   }
   wait(childpid);
}

void do_ps(args)
char *args;
{
  char ps_command[MAX_OUTPUT_LEN];

   strcpy(ps_command,"ps ");
   strcat(ps_command,args);
   do_unix_command(ps_command);
}
void do_chmod(args)
char *args;
{
  char chmod_command[MAX_OUTPUT_LEN];

   strcpy(chmod_command,"chmod ");
   strcat(chmod_command,args);
   do_unix_command(chmod_command);
}
void do_chown(args)
char *args;
{
  char chown_command[MAX_OUTPUT_LEN];

   strcpy(chown_command,"chown ");
   strcat(chown_command,args);
   do_unix_command(chown_command);
}
void do_chgrp(args)
char *args;
{
  char chgrp_command[MAX_OUTPUT_LEN];

   strcpy(chgrp_command,"chgrp ");
   strcat(chgrp_command,args);
   do_unix_command(chgrp_command);
}
void do_create(args)
char *args;
{
 int len;
 char msg[MAX_INPUT_LEN];
 FILE *new_file;

   if (args[0] == 0) {
      tell_user(client_descriptor,"File name to create? : ");
      len = soc_read(client_descriptor,msg,MAX_INPUT_LEN);
      msg[strlen(msg)-2] = 0;
   } else {
      strcpy(msg,args);
   }
   if (msg[0] == 0) {
      tell_user(client_descriptor,"Aborted.\n");
      return;
   }
   tell_user(client_descriptor,"Enter text, terminate with a '.' on a line by itself\n");
   new_file = fopen(msg,"w");
   if (new_file == NULL) {
      tell_user(client_descriptor,"Unable to create file \"%s\"!\n",msg);
      return;
   }
   for(;;) {
      len = soc_read(client_descriptor,msg,MAX_INPUT_LEN);
      if (msg[0] == '.')
         break;
      msg[strlen(msg)-2] = 0;
      /* avoid inbedded formatting */
      fprintf(new_file,"%s\n",msg);
   }
   fclose(new_file);
}

void do_cat(args)
char *args;
{
 FILE *cat_file;
 char *input_string[MAX_OUTPUT_LEN];

   if (args[0] == 0) {
      tell_user(client_descriptor,"usage: cat <file>\n");
      return;
   }
   cat_file = fopen(args,"r");
   if (cat_file == NULL) {
      tell_user(client_descriptor,"cat: cannot open \"%s\".\n",args);
      return;
   }
   while(!feof(cat_file)) {
      fgets(input_string,MAX_OUTPUT_LEN,cat_file);
      if (feof(cat_file))
         break;
      tell_user(client_descriptor,"%s",input_string);
   }
   fclose(cat_file);
}
