/********************************************************************
*   UnixCompatibility.h
*
*   Used to adapt the posix (unix) calls to mac calls.
*
*   Copyright (c) 1995, Willows Software Inc.  All rights reserved.
********************************************************************/
#ifndef UnixCompatibility__h
#define UnixCompatibility__h

/* "@(#)UnixCompatibility.h	1.9  2/4/97 14:51:30" */

#include <unix.h>
#include <stat.h>
#include <utime.h>
#include <fcntl.h>
#include <Types.h>

/* Redirect some of the unix calls to our "wrapper" calls */
#ifndef NO_POSIX_REDIRECT
#define mkdir		UNIX_mkdir
#define fopen		UNIX_fopen
#define freopen		UNIX_freopen
#define getcwd		UNIX_getcwd
#define chdir		UNIX_chdir
#define remove		UNIX_remove
#define rename		UNIX_rename
#define open			UNIX_open
#define create		UNIX_create

#define unlink		UNIX_remove
#define link			UNIX_rename
#endif


#ifndef M_PI
#define M_PI _PI
#endif

/** stdio.h support		*******************************/
FILE *UNIX_fopen(const char *fname, const char *mode);
FILE *UNIX_freopen(const char *fname, const char *mode, FILE *file);
int UNIX_remove(const char *fname);
int UNIX_rename(const char *oldname, const char *newname);

/** stat.h support		*******************************/
#ifndef S_IFBLK
#define	S_IFBLK		0x00600	/* block special, we do not have an equivalent */
#endif
												/* The unix version has this which is the character and directory bits */
#define	S_IREAD		0x00040	/* read permission, owner */
#define	S_IWRITE		0x00020	/* write permission, owner */
#define	S_IEXEC		0x00010	/* execute/search permission, owner */

int UNIX_stat(const char *path, struct stat *buf);
int UNIX_mkdir(const char *path, int mode);


/** unistd.h 				*******************************/
char *UNIX_getcwd(char *buf, int size);
int UNIX_chdir(const char *path);

/** fcntl.h 					*******************************/
int UNIX_open(const char *path, int oflag);
int UNIX_creat(const char *path, mode_t mode);


/** sys/time.h 			*******************************/

struct timezone {
	int     tz_minuteswest; 			/* minutes west of Greenwich */
	int     tz_dsttime;     				/* type of dst correction */
};

int gettimeofday(struct timeval *tp, struct timezone *tzp);


/** sys/vfs.h 				*******************************/

typedef struct {
	long    val[2];
} fsid_t;

struct statfs {
	long    f_type;     /* type of info, zero for now */
	long    f_bsize;    /* fundamental file system block size */
	long    f_blocks;   /* total blocks in file system */
	long    f_bfree;    /* free blocks */
	long    f_bavail;   /* free blocks available to non-super-user */
	long    f_files;    /* total file nodes in file system */
	long    f_ffree;    /* free file nodes in fs */
	fsid_t  f_fsid;     /* file system id */
	long    f_spare[7]; /* spare for later */
};

int statfs(char *path, struct statfs *buf);


/** sys/types.h 			*******************************/

typedef	long	off_t;

/* Provide the error codes */
#include <Errors.h>

#define	EACCES		wPrErr			/* Permission denied (Diskette is write protected)*/
#define	ENOENT		fnfErr			/* No such file or directory */
#define	ENAMETOOLONG	63		/* File name too long */
#define	ENOTDIR		20		/* Not a directory*/

#define	ENOSPC		dskFulErr		/* No space left on device (Disk full) */
#define	EIO			ioErr				/* I/O error */
#define	EMFILE		tmfoErr		/* Too many open files */
#define	EFBIG		mFulErr		/* File too large (Memory full (open) or file won't fit (load))*/
#define	ENODEV		nsvErr			/* No such device (No such volume) */
#define	ENXIO		volOnLinErr	/* No such device or address (No such volume)*/
#define	EEXIST		fidExists		/* File exists */
#define	EBADF		rfNumErr		/* Bad file number (Ref Num error)*/

#if 0
#define	EPERM		1		/* Not owner */
#define	ESRCH		3		/* No such process */
#define	EINTR		4		/* Interrupted system call */
#define	E2BIG		7		/* Arg list too long */
#define	ENOEXEC		8		/* Exec format error */
#define	ECHILD		10		/* No children */
#define	EAGAIN		11		/* No more processes */
#define	ENOMEM		12		/* Not enough core */
#define	EFAULT		14		/* Bad address */
#define	ENOTBLK		15		/* Block device required */
#define	EBUSY		16		/* Mount device busy */
#define	EXDEV		18		/* Cross-device link */
#define	EISDIR		21		/* Is a directory */
#define	EINVAL		22		/* Invalid argument */
#define	ENFILE		23		/* File table overflow */
#define	ENOTTY		25		/* Not a typewriter */
#define	ETXTBSY		26		/* Text file busy */
#define	ESPIPE		29		/* Illegal seek */
#define	EROFS		30		/* Read-only file system */
#define	EMLINK		31		/* Too many links */
#define	EPIPE		32		/* Broken pipe */
#endif


/** translation utilities	*****************************/
char *MacToUnixPath(char *path, int maxlen);
char *UnixToMacPath(char *path, int maxlen);
void PathNameFromDirID(long dirID, short vRefNum, StringPtr fullPathName);
void PathNameFromWD(long vRefNum, StringPtr pathName);



#endif		/* UnixCompatibility__h */
