/********************************************************************
*   DrvText.c
*
*   Macintosh device specific text sub-system implementation.
*
*   Copyright (c) 1994-1997, Willows Software Inc.  All rights reserved.  
********************************************************************/

/* System includes */
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

/* Includes */
#include "DrvHook.h"
#include "DrvDC.h"
#include "DeviceData.h"
#include "DrvText.h"
#include "DrvGraphics.h"
#include "DrvSystem.h"
#include "DrvUtils.h"

/* Structure to hold font alias information */
typedef struct {
	ATOM faceAtom;
	char *oldname;
	char *newname;
	int  minsize;/* minimal size we can map to FOR SCALABLE FONTS ONLY! */
	int  maxsize;/* maximal size we can map to FOR SCALABLE FONTS ONLY! */
} FONTALIAS;

/* Structure to hold additional (more detailed) information about a font */
typedef struct tagDRVFONTMETRIC {
	int		maxCharWidth;
	int		avgCharWidth;
	char		firstChar, lastChar, defaultChar;
	int		charWidth[256];
}	DRVFONTMETRIC, *LPDRVFONTMETRIC;


#define NUMLOGFONTSYTLES		4		/* Norm, ital, bold, bold-ital  (underline?)*/

/* Local Prototypes */
static DWORD DrvInitText(UINT, BOOL);
static LPDRVFONTDATA DrvRealizeFont(LPDRIVERDC lpddc, LSDE_REALIZEFONT *lpRealizeFont, DWORD dwMapperFlags, BOOL bRescale);
static DWORD DrvGetTextMetrics(LPDRIVERDC lpddc , LPDRVFONTDATA, LPNEWTEXTMETRIC lpNTM);
static DWORD DrvGetCharWidth(LPDRIVERDC lpddc, LPINT lpWidths, DWORD dwLength);
static DWORD DrvExtTextOut(LPDRIVERDC lpddc,LPLSDS_PARAMS lpStruct,BOOL bExtended);
static DWORD DrvBuildFontCache(LPNEWLOGFONT, LPNEWTEXTMETRIC, LPSTR, DWORD);
static DWORD DrvGetFontCacheSize(LPSTR);
static DWORD DrvGetTextFace(LPDRIVERDC lpddc, LPLSDS_PARAMS lpStruct, int len);
static LPDRVFONTDATA NewFontData(void);
static DWORD  FreeFontData(LPDRVFONTDATA);
static void BuildMetrics(LPNEWLOGFONT lplf, LPDRVFONTDATA lpdata, LPNEWTEXTMETRIC lpNTM);

static void DrawStrikethrough(LPDRIVERDC lpddc, int startX, int startY, int endX, int endY);
static short DrvGetFontNumber(char *fontName, short *fontNum);
static void DrvGetFontMetrics(LPDRIVERDC lpddc, LPDRVFONTDATA drvFont, LPDRVFONTMETRIC fontInfo);
static char *GetMacFontAlias(char* winFontName);
static char *GetWinFontAlias(char* macFontName);
static void LogFontHeightToPoints(LPDRVFONTDATA);

/* Some font macros */
#define ISTRUETYPE(fontID) RealFont(fontID, 30)		/* No bitmap will have this size? */
#define SMALLESTFONTSIZE	8
#define LARGESTFONTSIZE	72

/* Local variables */
static int  uiCompatibility;

/* Default font mapping */
static FONTALIAS defaultFontAlias[] = {
	{ 0,"Helv","Helvetica",			0, 0 },
	{ 0,"Symbol","Symbol",			0, 0 },
	{ 0,"Arial","Geneva",			0, 0 },
	{ 0,"Courier","Courier",		0, 0 },
	{ 0,"Times New Roman","Times",		0, 0 },
	{ 0,"Times New Roman Bold","Times",	0, 0 },
	{ 0,"Tms Rmn","Times",			0, 0 },
	{ 0,"MS Sans Serif","Helvetica",	0, 0 },
	{ 0,"MS Serif","New York",		0, 0 },
	{ 0,"Fixedsys","Monoco",		0, 0 },
	{ 0, 0, 0,				0, 0 }
};

FONTALIAS *userFontAlias = NULL;

/* Limit the number of sizes we check for with bitmap fonts */
static short macFontSizes[] = {
	8,9,10,12,18,24,36,0
};

/********************************************************************
*	PrivateTextHook
*
********************************************************************/
DWORD PrivateTextHook(WORD msg, LPARAM lp, LPARAM dwParam, LPVOID lpVoid)
{
    LPLSDS_PARAMS lpStruct = (LPLSDS_PARAMS)lpVoid;
    LPDRIVERDC lpddc = (LPDRIVERDC)lp;

    switch (msg) {
	/* lp - compatibility mask */
	/* dwParam - init/exit flag */
	case DSUBSYSTEM_INIT:
	    return DrvInitText((UINT)lp, (BOOL)dwParam);

	case DSUBSYSTEM_GETCAPS:
	    return (DWORD)0;

	case DSUBSYSTEM_EVENTS:
	    return (DWORD)0;

	/* dwParam -- hDC32->dwMapperFlags */
	case PTH_RESCALE:
	    return (DWORD)DrvRealizeFont(lpddc, &lpStruct->lsde.realizefont, dwParam, TRUE);

	case PTH_REALIZE:
	    return (DWORD)DrvRealizeFont(lpddc, &lpStruct->lsde.realizefont, dwParam, FALSE);

	case PTH_GETMETRICS:
		if(lpddc == NULL)
		    return (DWORD)FALSE;
	    else
	    	return DrvGetTextMetrics(lpddc,	NULL, (LPNEWTEXTMETRIC)lpStruct->lsde.lpmetric);

	/* dwParam -- length of the buffer in lpDX in bytes */
	case PTH_GETCHARWIDTH:
	    return DrvGetCharWidth(lpddc, lpStruct->lsde.text.lpDX, dwParam);

	/* dwParam -- flag TEXTOUT(0)/EXTTEXTOUT(1) */
	case PTH_EXTTEXTOUT:
	    return DrvExtTextOut(lpddc,lpStruct,(BOOL)dwParam);

	/* dwParam -- length of the buffer in lpStruct in bytes */
	case PTH_GETTEXTFACE:
	    return DrvGetTextFace(lpddc, lpStruct, dwParam);

	case PTH_GETENUMCNT:
		return DrvGetFontCacheSize((LPSTR)lpVoid); 	/* family name */

	case PTH_ENUMFONTS:
		return DrvBuildFontCache((LPNEWLOGFONT)lpStruct->lsde.enumfonts.lplfCache, 
                                     (LPNEWTEXTMETRIC)lpStruct->lsde.enumfonts.lpntmCache, 
                                     lpStruct->lsde.enumfonts.lpszFamily, 
                                     lpStruct->lsde.enumfonts.dwCacheSize);

	case PTH_DELETEFONT:
		/* dwParam -- is 0L (as GdiDeleteFont has no info about hDC) */
		/* lpVoid -- the lpMagic pointer to last realized font.				*/
		return FreeFontData((LPDRVFONTDATA)lpVoid);

	case PTH_GETGLYPHOUTLINE:
		return TWIN_GetGlyphOutline(lpddc,
					lpStruct->lsde.getglyphoutline.uChar,
					lpStruct->lsde.getglyphoutline.fuFormat,
					lpStruct->lsde.getglyphoutline.lpgm,
					lpStruct->lsde.getglyphoutline.cbBuffer,
					lpStruct->lsde.getglyphoutline.lpvBuffer,
					lpStruct->lsde.getglyphoutline.lpmat2);

	default:
	    return (DWORD)0;

	}
}


/********************************************************************
*	Convert from font height to point height.
*	Serves a dual role of setting up the metrics while we are at it!
********************************************************************/
static void LogFontHeightToPoints(LPDRVFONTDATA lpdrvFontData)
{
int lineHeight, delta;
short matched = FALSE;
FontInfo fInfo;

	/* Negative numbers specify the cell size, which is points */
	/* Height of zero specifies the system height */
	if (lpdrvFontData->tmHeight <= 0)
		matched = TRUE;

	/* Starting point */
	lpdrvFontData->fontSize = abs(lpdrvFontData->tmHeight) + 1;
	
	/* Look for the best match */
	do {
		lpdrvFontData->fontSize--;
		TextSize(lpdrvFontData->fontSize);
	
		/* This call uses the current port settings for the font info. */
		GetFontInfo(&fInfo);
		lpdrvFontData->ascent = fInfo.ascent;
		lpdrvFontData->descent = fInfo.descent;
		lpdrvFontData->extLeading = fInfo.leading;
	
		/* Try to guess at the internal leading */
		if (lpdrvFontData->ascent < lpdrvFontData->fontSize)
			lpdrvFontData->intLeading = lpdrvFontData->fontSize - lpdrvFontData->ascent;
		else
			lpdrvFontData->intLeading = 0;
	
		/* Determine how close we are */
		lineHeight = lpdrvFontData->ascent + lpdrvFontData->descent + lpdrvFontData->intLeading;
		delta = lineHeight - lpdrvFontData->tmHeight;
	
		if (delta <= 0)
			matched = TRUE;
			
	} while (!matched);
	
}

/********************************************************************
*	PointsToLogFontHeight
*
*	Convert from points to the log font height.
********************************************************************/
static int PointsToLogFontHeight(int points)
{
double fSize = points;

	fSize *= (double) 72;
	fSize /= (double) LOGPIXELSY;

	return(fSize);
}


/********************************************************************
*	DrvRealizeFont
*
********************************************************************/
static LPDRVFONTDATA DrvRealizeFont(LPDRIVERDC lpddc, LSDE_REALIZEFONT *lpRealizeFont, DWORD dwMapperFlags, BOOL bRescale)
{
#pragma unused(dwMapperFlags)
LPDRVFONTDATA	lpdrvFontData = (LPDRVFONTDATA)lpRealizeFont->lpMagic;
short found;
GrafPtr savePort;

	/* If the library is trying to cache a font, it may already be valid. */
	if(lpdrvFontData) {
		/* Check to see if the heights are the same, if they are, re-use the font as is. */
		if ((lpRealizeFont->LogFont.lfHeight == lpdrvFontData->tmHeight) && !bRescale) {
			FreeFontData(lpddc->lpFontData);		/* Remove drvDC's and old drvFont link */
			lpddc->lpFontData = lpdrvFontData;	/* Select into the DC */
			lpddc->lpFontData->nLock++;				/* Link drvDC and drvFont */
			return (lpdrvFontData);
		}
		if (bRescale) {
			 FreeFontData(lpddc->lpFontData);		/* Remove drvDC and old drvFont link */
			 FreeFontData(lpdrvFontData);			/* Free the old drvFont */
		 	/* Now, the two links are removed, create the new font */
			lpdrvFontData = NewFontData();
		}
	}
	else {
		/* Allocate the font structure since one does not exist. */
		if (lpddc->lpFontData)
			FreeFontData(lpddc->lpFontData);
		lpdrvFontData = NewFontData();
	}


	/* See if we are getting the same font family, we already have the number */
	if (strcmp(lpdrvFontData->familyName, lpRealizeFont->LogFont.lfFaceName) != 0) {
		/* Find the font family id for face name */
		/* If not found, it defaults to the system font */
		found = DrvGetFontNumber(lpRealizeFont->LogFont.lfFaceName, &(lpdrvFontData->fontID));
		strncpy(lpdrvFontData->familyName, lpRealizeFont->LogFont.lfFaceName, 254);
	}

	/* Save the text metrics */
	lpdrvFontData->tmHeight			= lpRealizeFont->LogFont.lfHeight;
    lpdrvFontData->tmWeight 		= lpRealizeFont->LogFont.lfWeight;		
    lpdrvFontData->tmItalic 			= lpRealizeFont->LogFont.lfItalic;
   	lpdrvFontData->tmUnderlined 	= lpRealizeFont->LogFont.lfUnderline;
   	lpdrvFontData->tmStruckOut 	= lpRealizeFont->LogFont.lfStrikeOut;
	lpdrvFontData->isTrueType		= FALSE;				/* For now */

	/* Determine the text face from the font text metrics */
	lpdrvFontData->face = lpdrvFontData->tmItalic ? italic: 0;						/* Boolean flag */
	lpdrvFontData->face |= lpdrvFontData->tmWeight > 400 ? bold: 0;			/* bold fonts have weights > 400 */
	lpdrvFontData->face |= lpdrvFontData->tmUnderlined ? underline: 0;		/* Boolean flag */

	/* Now get some of the font info */
	/* Set the font characteristics in the dc's grafport to get font info.*/
	GetPort(&savePort);
	SetPort(lpddc->grafPort);
	TextFont(lpdrvFontData->fontID);
	TextFace(lpdrvFontData->face);

	/* Calclate the desired height in points */
	LogFontHeightToPoints(lpdrvFontData);

	SetPort(savePort);
	
	lpddc->lpFontData = lpdrvFontData;		/* Select into the DC */
	lpddc->lpFontData->nLock++;					/* Link drvDC and drvFont */
	return (lpdrvFontData);
}


/********************************************************************
*	NewFontData
*
*	Allocate a new DRVFONTDATA structure.
*	The nLock variable will be one, representing the link to the library.
********************************************************************/
static LPDRVFONTDATA NewFontData(void)
{
LPDRVFONTDATA	lpNewFont;

	lpNewFont = DrvMalloc(sizeof(DRVFONTDATA));
	memset(lpNewFont, '\0', sizeof(DRVFONTDATA));		/* Clear memory */
	lpNewFont->familyName = DrvMalloc(256);				/* Hold the font name */
	lpNewFont->familyName[0] = '\0';
	
	lpNewFont->nLock = 1;												/* Single link to libFont */

	return(lpNewFont);
}


/********************************************************************
*   FreeFontData
*
*	Free the font structure if no one else is using it.
* The drvDC may have links, so they can be removed by calling this as well.
********************************************************************/
static DWORD FreeFontData(LPDRVFONTDATA lpFont)
{
	/*	Since GdiDeleteFont has no idea about hDC, it passes 0L instead of an actual pointer.  */
	/*	It means the lpddc parameter here is useless.*/

	if (!lpFont)
		return(0);

	/* Check the lock to see if any one else is using before destruction */
	if (--lpFont->nLock <= 0) {
		DrvFree(lpFont->familyName);			/* Free the font name */
		DrvFree(lpFont);								/* Free the structure */
	}
	
	return (1);
}



/********************************************************************
*   DrvGetFontMetrics
*
*	Update the DRVFONTDATA structure.
********************************************************************/
static void DrvGetFontMetrics(LPDRIVERDC lpddc, LPDRVFONTDATA lpdrvFontData, LPDRVFONTMETRIC lpdrvFontMetric)
{
GrafPtr oldPort, grafPort;
short saveFont, saveSize;
Style saveFace;
FMetricRec fMetric;
int iWidth, i;
WidthTable **wTableH;
char alphabet[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";

#if USEOUTLINEMETRICS
char tallChars[] = "{[";
Point pt;
short yMin, yMax
#endif

	/* We may be called without a valid dc, make adjustments accordingly */
	if (lpddc != NULL)
		grafPort = lpddc->grafPort;			/* Use the dc's port if available */
	else {
		grafPort = FrontWindow();
		if (grafPort == NULL)					/* Use one of our front windows. */
			GetWMgrPort(&grafPort);		/* Desperation, use the wMgr port. */
	}
	
	/* Set the port for us to temporarily use */
	GetPort(&oldPort);
	SetPort(grafPort);

	/* Save current settings */
	saveFont = grafPort->txFont;
	saveSize = grafPort->txSize;
	saveFace = grafPort->txFace;
	
	/* Set the DC's current settings */
	grafPort->txFont = lpdrvFontData->fontID;
	grafPort->txSize = lpdrvFontData->fontSize;
	grafPort->txFace = lpdrvFontData->face;

	/* Get font metrics from current font in port */
	FontMetrics(&fMetric);	
	lpdrvFontData->ascent = FixRound(fMetric.ascent);
	lpdrvFontData->descent = FixRound(fMetric.descent);
	lpdrvFontData->extLeading = FixRound(fMetric.leading);
	lpdrvFontMetric->maxCharWidth = FixRound(fMetric.widMax);

	/* Get the character widths for the font */
	wTableH = (WidthTable **) fMetric.wTabHandle;
	for (i=0; i<256; i++)
		lpdrvFontMetric->charWidth[i]=FixRound((**wTableH).tabData[i]);

	lpdrvFontMetric->firstChar = 0;				/* Allow all characters, the font info did not work */
	lpdrvFontMetric->lastChar = 255;
	lpdrvFontMetric->defaultChar =255;

	/* Calculate the average character width based on the upper and lowercase alphabet */
	iWidth = TextWidth(alphabet, 0, 52);
	lpdrvFontMetric->avgCharWidth = iWidth / 52;

#if USEOUTLINEMETRICS
	/* Now, try to figure out what the internal leading is */
	pt.h = 1;	pt.v = 1;				/* Scale by one */
	OutlineMetrics(strlen(tallChars), tallChars, pt, pt, &yMax, &yMin, NULL, NULL, NULL);
	/* We have one of the tallest fonts, see what the difference is from the ascent */
	if (yMax > lpdrvFontData->ascent)
		lpdrvFontMetric->intLeading = yMax - lpdrvFontData->ascent;
	else
		lpdrvFontMetric->intLeading = 0;
#else
	/* This is an approximation based on observations of the font implementations */
	if (lpdrvFontData->ascent < lpdrvFontData->fontSize)
		lpdrvFontData->intLeading = lpdrvFontData->fontSize - lpdrvFontData->ascent;
	else
		lpdrvFontData->intLeading = 0;
#endif

	/* Restore settings */
	grafPort->txFont = saveFont;
	grafPort->txSize = saveSize;
	grafPort->txFace = saveFace;
	SetPort(oldPort);

}


/********************************************************************
*	DrvGetTextMetrics
*
*	Update both the driver DRVFONTDATA and NEWTEXTMETRIC structures.
********************************************************************/
static DWORD DrvGetTextMetrics(LPDRIVERDC lpddc , LPDRVFONTDATA drvFont, LPNEWTEXTMETRIC lpNTM)
{
LPDRVFONTDATA lpdrvFontData;
DRVFONTMETRIC drvFontMetric;

	if (lpddc != NULL)
		lpdrvFontData = lpddc->lpFontData;		/* Use the dc's font if it is available */
	else if (drvFont)
		lpdrvFontData = drvFont;						/* No dc, but we have a font to get info for */
	else
		return(FALSE);										/* Bail, someone is screwed up! */
		
	/* Get the mac font metrics */
	DrvGetFontMetrics(lpddc, lpdrvFontData, &drvFontMetric);

    lpNTM->tmAscent 	= lpdrvFontData->ascent + lpdrvFontData->intLeading; 
    lpNTM->tmDescent 	= lpdrvFontData->descent;
    lpNTM->tmHeight 		= lpNTM->tmAscent + lpNTM->tmDescent;
    lpNTM->tmInternalLeading = lpdrvFontData->intLeading;
    lpNTM->tmExternalLeading = lpdrvFontData->extLeading;

    lpNTM->tmFirstChar 	= drvFontMetric.firstChar;
    lpNTM->tmLastChar 		= drvFontMetric.lastChar;
	lpNTM->tmDefaultChar	=drvFontMetric.defaultChar;

	lpNTM->tmMaxCharWidth	= drvFontMetric.maxCharWidth;
    lpNTM->tmAveCharWidth 	= drvFontMetric.avgCharWidth;

	/* Get the font modes from the original LogFont values */
    lpNTM->tmWeight 			= lpdrvFontData->tmWeight;			
    lpNTM->tmItalic 			= lpdrvFontData->tmItalic;
    lpNTM->tmUnderlined 	= lpdrvFontData->tmUnderlined;
    lpNTM->tmStruckOut 	= lpdrvFontData->tmStruckOut;

    lpNTM->tmBreakChar 			= ' ';
    lpNTM->tmCharSet 				= ANSI_CHARSET;
    lpNTM->tmOverhang 				= 0;
    lpNTM->tmDigitizedAspectX 	= 72;
    lpNTM->tmDigitizedAspectY 	= 72;
    lpNTM->ntmFlags 					= 0;
    lpNTM->ntmSizeEM 				= 0;
    lpNTM->ntmCellHeight 			= 0;
    lpNTM->ntmAvgWidth 			= 0;

    lpNTM->tmPitchAndFamily 	= FF_ROMAN;
    if (lpdrvFontData->isTrueType)
		lpNTM->tmPitchAndFamily |= TMPF_TRUETYPE;
	else
		lpNTM->tmPitchAndFamily |= TMPF_FIXED_PITCH;

    return (DWORD)TRUE;
}


/********************************************************************
*	DrvGetCharWidth
*
********************************************************************/
static DWORD DrvGetCharWidth(LPDRIVERDC lpddc, LPINT lpWidths, DWORD dwLength)
{
LPDRVFONTDATA lpdrvFontData = lpddc->lpFontData;
LPINT lpInt;
int nCount, i;
DRVFONTMETRIC drvFontMetric;

	if(lpddc == NULL)
    	return (DWORD)FALSE;
  
	/* Get the font metrics from the mac driver */
	DrvGetFontMetrics(lpddc, lpdrvFontData, &drvFontMetric);

    if (lpdrvFontData->isTrueType) {
#if LATER
LPABC lpABC;
		nCount = min(dwLength / sizeof(ABC), 256);
		for (i = 0,lpABC = (LPABC)lpWidths; i < nCount; i++,lpABC++) {
	    	lpABC->abcA = xfs->per_char[i].lbearing;
	    	lpABC->abcB = xfs->per_char[i].width;
	    	lpABC->abcA = xfs->per_char[i].rbearing;
		}
#endif
    }
    else {
		nCount = min(dwLength / sizeof(int), 256);
		/* The widths are stored in the font info structure, copy them */
		for (i = 0,	lpInt = lpWidths; i < nCount; i++, lpInt++) {
	    	*lpInt = drvFontMetric.charWidth[i];
	    }
    }

    return (DWORD)TRUE;

}


/********************************************************************
*	SelectTextPen
*
* Setup the port to reflect the dc's setting for style, color, etc.
********************************************************************/
static void SelectTextPen(LPDRIVERDC lpddc)
{
RGBColor rgbColor;
LPDRVFONTDATA lpdrvFontData = lpddc->lpFontData;
CGrafPtr grafPort = lpddc->grafPort;

	/* Set the foreground text color */
	COLORREFTOMACRGB(lpddc->textColor, &rgbColor);
	if (!EQUALRGBCOLORS(&grafPort->rgbFgColor, &rgbColor))
		RGBForeColor(&rgbColor);			/* Set color in grafport */

	/* Set the font characteristics of the graf port */
	grafPort->txFont = lpdrvFontData->fontID;
	grafPort->txSize = lpdrvFontData->fontSize;
	grafPort->txFace = lpdrvFontData->face;

	/* Now set the pen mode according to the bk fill mode */
	if (lpddc->bkFillMode == TRANSPARENT)
		grafPort->txMode = srcOr;
	else {
		grafPort->txMode = srcCopy;
		/* Set the background so that the OPAQUE function works properly */
		COLORREFTOMACRGB(lpddc->backColor, &rgbColor);
		if (!EQUALRGBCOLORS(&grafPort->rgbBkColor, &rgbColor))
			RGBBackColor(&rgbColor);			/* Set color in grafport */
	}


}

/********************************************************************
*	DrvExtTextOut
*
*	Output the text.
********************************************************************/
static DWORD DrvExtTextOut(LPDRIVERDC lpddc,LPLSDS_PARAMS lpStruct,BOOL bExtended)
{
LPDRVFONTDATA lpdrvFontData = lpddc->lpFontData;
LSDE_TEXT *lptxt = &lpStruct->lsde.text;
LPINT lpDX;
UINT xExtent;
int x, y, i, j;

    if (lpStruct->lsde_validate.dwInvalidMask)
		DrvValidate(lpddc, lpStruct);

	DrvSelectDC(SAVE, lpddc);				/* Save old port, set new port */
	SelectTextPen(lpddc);						/* Setup the text pen into the current port */

    lpDX = (bExtended) ? lptxt->lpDX : 0;
     
    if (lptxt->TextAlign & TA_UPDATECP) {
		x = lpddc->cpt.x;
		y = lpddc->cpt.y;
    } else {
		x = lptxt->x;
		y = lptxt->y;
   }

    /* TA_LEFT == 0, so check for others */ 
    /* also if TA_UPDATECP get extents now */
    if (lptxt->TextAlign & (TA_CENTER|TA_RIGHT | TA_UPDATECP)) {
		if (lpDX) {
	    	for (i = 0,xExtent = 0; i < lptxt->nCnt; i++)
				xExtent += abs(lpDX[i]);
		}
		else {
			xExtent = TextWidth(lptxt->lpStr, 0, lptxt->nCnt);

		    if (lptxt->lpExtraSpace) {
               /*   Adjust xExtent to account text justification...       */  
               for (i = j = 0; i < lptxt->nCnt; i++) {
                   if (lptxt->lpStr[i] == lptxt->chBreak && j < lptxt->nBreakCount)
                      xExtent += lptxt->lpExtraSpace[j++];
                   }
		    }
		}

		if ((lptxt->TextAlign & TA_CENTER) == TA_CENTER)
		    x -= xExtent/2;
		if((lptxt->TextAlign & TA_CENTER) == TA_RIGHT)
		    x -= xExtent;
		if(lptxt->TextAlign & TA_UPDATECP)
		    lpddc->cpt.x += xExtent;
    }

    /* TA_TOP == 0, so check for others */
    if (lptxt->TextAlign & (TA_BASELINE | TA_BOTTOM)) {
		if ((lptxt->TextAlign & TA_BASELINE) == TA_BOTTOM)
	    	y -= lpdrvFontData->descent - 1; 
	} 
	else {
		/* Try adding in the internal leading. */
		y += lpdrvFontData->ascent + lpdrvFontData->intLeading;
	}
	
    if (lpDX) {
		for (i = 0; i < lptxt->nCnt; x += lpDX[i++]) {
			MoveTo(x, y);
			DrawText(&lptxt->lpStr[i], 0, 1);
		}
	}
    else {
	    if (lptxt->lpExtraSpace) {
	    	/* Justified TextOut */
	    	register ii;
	    	char *lpsz = DrvMalloc(lptxt->nCnt + 1);

            for (xExtent = ii = i = j = 0; i < lptxt->nCnt; i++) {
                lpsz[ii++] = lptxt->lpStr[i];  lpsz[ii] = '\0';
				xExtent += TextWidth(&lptxt->lpStr[i], 0, 1);

				if (lptxt->lpStr[i] == lptxt->chBreak && j < lptxt->nBreakCount) {
					xExtent += lptxt->lpExtraSpace[j++];
					/* Draw current word, including trailing break character, 	*/
					/* then reset xExtent and character counter.			            */
					MoveTo(x, y);
					DrawText(lpsz, 0, strlen(lpsz));
					x += xExtent;
					xExtent = ii = 0;
				}
			}
			if (ii > 0) { /* draw the rest of the string (if any) */
				MoveTo(x, y);
				DrawText(lpsz, 0 , strlen(lpsz));
			}
            DrvFree(lpsz);
	    }
	    else {
			MoveTo(x,y);
			DrawText(lptxt->lpStr, 0, lptxt->nCnt);
		}
    }

    /* STRIKEOUT: special case for now... */
    if (lpdrvFontData->tmStruckOut) {
		xExtent = TextWidth(lptxt->lpStr, 0, lptxt->nCnt);

       if (lptxt->lpExtraSpace) {
          for (i = j = 0;   i < lptxt->nCnt;  i++)
              if (lptxt->lpStr[i] == lptxt->chBreak && j < lptxt->nBreakCount)
                 xExtent += lptxt->lpExtraSpace[j++];
       }

		y -= lpdrvFontData->descent;
		DrawStrikethrough(lpddc, x, y, x + xExtent, y);
    }

	DrvSelectDC(RESTORE, NULL);		/* Restore old port */
    return (DWORD)TRUE;
}

/********************************************************************
*	Draws the strikethrough line in text.
*	NOTE!! This call assumes the grafport and all of the text related colors 
*	           and transfer modes have been setup.
********************************************************************/
static void DrawStrikethrough(LPDRIVERDC lpddc, int startX, int startY, int endX, int endY)
{
CGrafPtr grafPort = lpddc->grafPort;
short fudge;

	grafPort->txMode =  (lpddc->bkFillMode == TRANSPARENT) ? patOr : patCopy;

	/* Hack to attempt to draw a larger strikethrough based on the size */
	/* 32 roughly corresponds to Windows size 24, where I noticed the change */
	/* 16 is roughly Windows size 12 (*96/72) for bitmap fonts */
	fudge = (ISTRUETYPE(lpddc->lpFontData->fontID)) ? 32 : 16;
	grafPort->pnSize.v = lpddc->lpFontData->fontSize / fudge + 1;

	MoveTo(startX, startY);					/* Position at start point */
	LineTo(endX, endY);							/* Draw the line */

	/* Reset the pen size, just to be safe */
	grafPort->pnSize.v = 1;	grafPort->pnSize.h = 1;

}


/********************************************************************
*	DrvInitText
*
********************************************************************/
static DWORD DrvInitText(UINT uiCompat, BOOL bInit)
{
Boolean aBool;

    if (bInit) {
		uiCompatibility = uiCompat;	
		aBool = GetPreserveGlyph();			/* See what it was */
		SetPreserveGlyph(TRUE);	
		SetOutlinePreferred(TRUE);
		return 1L;
    }
    else
		return 0L;
}


/********************************************************************
*   GetWinFontAlias
*
*	Checks for any aliases for the macintosh font.
********************************************************************/
static char *GetWinFontAlias(char *macFontName)
{
FONTALIAS *lpfa;

	/* Check the default alias table */
   for (lpfa = defaultFontAlias; lpfa->oldname && lpfa->newname;  lpfa++) 
		if (strncmp(macFontName, lpfa->newname, strlen(lpfa->newname)) == 0)
			return lpfa->oldname;

	/* Now check the user defined alias table */
	/* LATER: Add user alias table */
	if (userFontAlias)
		for (lpfa = userFontAlias;  lpfa->oldname && lpfa->newname; lpfa++) 
			if (strncmp(macFontName, lpfa->newname, strlen(lpfa->newname)) == 0)
				return lpfa->oldname;

	/* No alias found, return the original */
	return (macFontName);
}


/********************************************************************
*   GetMacFontAlias
*
*	Checks for any aliases for the windows font.
********************************************************************/
static char *GetMacFontAlias(char *winFontName)
{
FONTALIAS *lpfa;

	/* Check the default alias table */
   for (lpfa = defaultFontAlias; lpfa->oldname && lpfa->newname;  lpfa++) 
		if (strncmp(winFontName, lpfa->oldname, strlen(lpfa->oldname)) == 0)
			return lpfa->newname;


	/* Now check the user defined alias table */
	/* LATER: Add user alias table */
	if (userFontAlias)
		for (lpfa = userFontAlias;  lpfa->oldname && lpfa->newname; lpfa++) 
			if (strncmp(winFontName, lpfa->oldname, strlen(lpfa->oldname)) == 0)
				return lpfa->newname;

	/* No alias found, return the original */
	return (winFontName);
}

/********************************************************************
*   DrvGetFontNumber
*
*	Returns in the fontNum parameter the number for the font with the given 
*	font name. If theres no such font, it returns FALSE.
********************************************************************/
static short DrvGetFontNumber(char *fontName, short *fontNum)
{	
Str255 pFontName;
Str255 systemFontName;

	/* Get any aliases for the fonts */
	c2pstrcpy(pFontName, GetMacFontAlias(fontName));
	
	/* Try to get the requested font */
	GetFNum(pFontName, fontNum);
	
	if (*fontNum == 0) {
		/* either the font was not found, or it is the system font */
		GetFontName(0, systemFontName);
		return(EqualString(pFontName, systemFontName, FALSE, FALSE));
	}
	
	return(TRUE);	
}


/********************************************************************
*   BuildMetrics
*
*	Fill the LOGFONT and NEWTEXTMETRIC for the given driver font (mac font).
*	The DRVFONTDATA must have the fontID, fontSize and fontFace fields set.
********************************************************************/
static void BuildMetrics(LPNEWLOGFONT lpnlf, LPDRVFONTDATA drvFont, LPNEWTEXTMETRIC lpNTM)
{
	/* Fill int the tmFields as some of the calls use them */
	drvFont->tmWeight 		= ((drvFont->face & bold) ? FW_BOLD : FW_REGULAR);       
	drvFont->tmItalic 			= ((drvFont->face & italic) ? TRUE : FALSE);                 
	drvFont->tmUnderlined 	= ((drvFont->face & underline) ? TRUE : FALSE);               
	drvFont->tmStruckOut	= 0;

	/* Determine the actual type of font */
	drvFont->isTrueType = ISTRUETYPE(drvFont->fontID);

	/* Get all the info about this font */
	DrvGetTextMetrics(NULL, drvFont, lpNTM);

	/* Now fill out the LOGFONT information from the NEWTEXTMETRIC info */

	/* Get the windows name for now, some binarys are looking for a */
	/*  specific named font, ie "Arial" */
	strcpy(lpnlf->lf.lfFaceName, GetWinFontAlias(drvFont->familyName));

	lpnlf->lf.lfHeight 			= lpNTM->tmHeight;
	lpnlf->lf.lfWidth 			= lpNTM->tmMaxCharWidth;	//??
	lpnlf->lf.lfWeight 			= lpNTM->tmWeight;       
	lpnlf->lf.lfItalic 				= lpNTM->tmItalic;                 
	lpnlf->lf.lfUnderline 		= lpNTM->tmUnderlined;               
	lpnlf->lf.lfStrikeOut 		= lpNTM->tmStruckOut;                

	lpnlf->lf.lfEscapement 	= 0;          
	lpnlf->lf.lfOrientation 	= 0;          
	lpnlf->lf.lfCharSet 			= ANSI_CHARSET;
	lpnlf->lf.lfOutPrecision 	= OUT_DEFAULT_PRECIS;  
	lpnlf->lf.lfClipPrecision 	= CLIP_DEFAULT_PRECIS;
	lpnlf->lf.lfQuality 			= DEFAULT_QUALITY;          

	if (drvFont->isTrueType) {
		lpnlf->lf.lfPitchAndFamily = VARIABLE_PITCH; 

		/* Construct style text, easier this way to get the spaces correct. */
		if ((lpnlf->lf.lfWeight == FW_BOLD) && lpnlf->lf.lfItalic)
			strcpy((LPSTR)lpnlf->lfStyle, "Bold Italic");
		else if (lpnlf->lf.lfWeight == FW_BOLD)
			strcpy((LPSTR)lpnlf->lfStyle, "Bold");
		else if (lpnlf->lf.lfItalic)
			strcpy((LPSTR)lpnlf->lfStyle, "Italic");
		else
			strcpy((LPSTR)lpnlf->lfStyle, "Normal");
		/* Construct full name */	
		strcpy((LPSTR)lpnlf->lfFullName, lpnlf->lf.lfFaceName);		/* "Geneva" */
		strcat((LPSTR)lpnlf->lfFullName, " ");									/* "Geneva " */
		strcat((LPSTR)lpnlf->lfFullName, (LPSTR)lpnlf->lfStyle);		/* "Geneva Bold Italic" */ 
	}
	else {
		lpnlf->lf.lfPitchAndFamily = FIXED_PITCH;
		lpnlf->lfStyle[0] = '\0';					/* Valid for true type only */
		lpnlf->lfFullName[0] = '\0';			/* Valid for true type only */
	}
				
}


/********************************************************************
*   DrvBuildFontCache
*
*	Stores LOGFONT info for each font in the given family.
*	The memory block lpLogFont should alread be allocated by the calling function.
*	The calling function may call DrvGetFontCachSize() to determine how many LOGFONTS
*	  have been requested.
********************************************************************/
static DWORD DrvBuildFontCache(LPNEWLOGFONT lplfCache, LPNEWTEXTMETRIC lpntmCache, LPSTR lpszFamily, DWORD dwCount)
{
char szAlias[256];
DWORD           		dwTotal = 0L;
NEWLOGFONT        *lpnlf = lplfCache;
NEWTEXTMETRIC  *lpntm = lpntmCache;
LPDRVFONTDATA	drvFont;
int fontStyle, numStyles;
Handle	rHandle;
ResType	rType;
Str255	pFontName;
Boolean enumAllFamilies, doneEnumSizes, isTrueType, validFontName;
short	rID, macFamilyIndex, *size, count = dwCount;

	/* Verify information */
   if (lplfCache == NULL || lpntmCache == NULL || count == 0L)
      return dwTotal;

	/* Create a temporary driver font */
	if ((drvFont = NewFontData()) == NULL)
		return dwTotal;
		
	/* Enum families or fonts? */
	if (lpszFamily == NULL || *lpszFamily == '\0')
		enumAllFamilies = TRUE;
	else {
		enumAllFamilies = FALSE;
	   /* Try to find an alias for the given font name. */
		strcpy(szAlias, GetMacFontAlias(lpszFamily));
	}
	
	/* Now initialize LOGFONT cache. */
	lpnlf = lplfCache;  lpntm = lpntmCache;
	macFamilyIndex = 1;							/* For enumerating all font families */
	while (count > 0) {
		validFontName = FALSE;
		if (enumAllFamilies) {
			/* We are indexing down the families, so get this index families name. */
			SetResLoad(FALSE);			/* do not need resource, just info */
			/* Implement a filter of sorts, to weed out fonts with % in the name */
			while ((count > 0) && !validFontName) {
				rHandle = GetIndResource('FOND', macFamilyIndex++);
				GetResInfo(rHandle, &rID, &rType, pFontName);
				/* Check to see that the first letter of the name is an alphabet character */
				if (isalpha(pFontName[1])) {			/* Pascal, first element is the length */
					validFontName = TRUE;
					GetFNum(pFontName, &drvFont->fontID);
				}
				else
					count--;						/* Decrement count as this was counted in the DrvGetFontCacheSize() call */
			}
			SetResLoad(TRUE);				/* better do this! */
			p2cstrcpy(drvFont->familyName, pFontName);
		}
		else {
			DrvGetFontNumber(szAlias, &drvFont->fontID);
			strcpy(drvFont->familyName, szAlias);
			validFontName = TRUE;
		}
		
		/* Make sure we actually found a valid font name. */
		if (validFontName) {
			isTrueType = ISTRUETYPE(drvFont->fontID);

			if (isTrueType && !enumAllFamilies)
				/* Enum each style in the true type family */
				numStyles = NUMLOGFONTSYTLES;
			else
				/* Enum one style per family when enumerating all font families */
				/* Enum one style when enumerating a single bitmap font family */
				numStyles = 1;
						
			/* Fill in the return structure for each style, if we are not enumerating families */
			for (fontStyle=0; fontStyle < numStyles; fontStyle++) {
				doneEnumSizes = FALSE;
				size = macFontSizes;				/* The default bitmap font sizes */
				while (!doneEnumSizes) {
					if (isTrueType)
						drvFont->fontSize = 0;				/* The mac will interpret this as the default system type (12) */
					else
						drvFont->fontSize = *size++;
					
					if (RealFont(drvFont->fontID, drvFont->fontSize) && (lpnlf != NULL)) {
						drvFont->face = fontStyle;		/* Let it count, norm, bold, italic, bold-italic */
	
						/* Fill the LOGFONT and NEWTEXTMETRIC structures */
						BuildMetrics(lpnlf, drvFont, lpntm); /* stores NEWTEXTMETRIC into memory block, specified by lpntm */
	
						/* Override the text metrics, as they think size zero is for system font size */
						if (drvFont->fontSize == 0) {
							lpnlf->lf.lfHeight = 0;				/* Indicate it is a TrueType font */
							lpntm->tmHeight = 0;
						}
					
						lpnlf++;  lpntm++; 		/* point to next slot in cache */
						count--;
						dwTotal++;
						
						/* When enum families, we are done when the first one is found */
						if (enumAllFamilies)
							doneEnumSizes = TRUE;
					}
	
					/* Only check additional sizes if we are enumerating a bitmap font family */
					if ((*size == 0) || isTrueType)
						doneEnumSizes = TRUE;
				}
			}		/* fontStyle loop */
		}			/* validFont */
	}				/* count > 0 */

	FreeFontData(drvFont);
	
	FlushFonts();			/* Will this clean up the heap?? */

	return dwTotal;
}


/********************************************************************
*   DrvGetFontCacheSize
*
*	Returns the number of LOGFONTs for the given family.
*	The calling function can then allocate a font cache with a size = (count * sizeof(LOGFONT)).  
*  Then DrvBuildFontCache() can be called to fill the cache.
********************************************************************/
static DWORD DrvGetFontCacheSize(LPSTR lpszFamily)
{
char szAlias[256];
int nCount = 0;
short found, fontID, *size;

   if (lpszFamily && *lpszFamily) {
		/* Enumerating a specific family */
		strcpy(szAlias, GetMacFontAlias(lpszFamily));
		/* Check to see if the font actually exists */
		found = DrvGetFontNumber(lpszFamily, &fontID);
		if (found) {
			if (ISTRUETYPE(fontID)) {
				nCount = NUMLOGFONTSYTLES;		/* One size per style */
			}
			else {												/* Find the sizes for this font family */
				size = macFontSizes;
				while (*size)
					if (RealFont(fontID, *size++))
						nCount++;
			}
		}
   }
	else	{
		/* Enumerating all available font families */
		/* Determine how many different font families exist at all */
		nCount = CountResources( 'FOND' );
	}
	
	return (nCount);
}


/********************************************************************
*   DrvGetTextFace
*
*	Get the name of the font.
********************************************************************/
static DWORD DrvGetTextFace(LPDRIVERDC lpddc, LPLSDS_PARAMS lpStruct, int len)
{
LSDE_TEXT *lptxt = &lpStruct->lsde.text;
LPDRVFONTDATA lpdrvFontData = lpddc->lpFontData;

    if (lpStruct->lsde_validate.dwInvalidMask)
		DrvValidate(lpddc,lpStruct);

	if (lptxt->lpStr == NULL)
		return(0);

	/* Copy the name that we have into to buffer */
    strncpy(lptxt->lpStr, lpdrvFontData->familyName, len);
    lptxt->lpStr[len - 1] = '\0';

    return (DWORD)strlen(lptxt->lpStr);
}
