/*
 * Electric(tm) VLSI Design System
 *
 * File: dbcontrol.c
 * Database system controller
 * Written by: Steven M. Rubin, Electric Editor Incorporated
 *
 * Copyright (c) 1998 Electric Editor Incorporated.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Electric Editor Incorporated
 * 23470 Sunset Drive, Suite 108
 * Los Gatos, California 95033
 * support@electriceditor.com
 */

#include "global.h"
#include "database.h"
#include "tech.h"
#include <signal.h>
#ifdef HAVE_SYS_WAIT_H
#  include <sys/wait.h>
#endif
#ifdef HAVE_UNISTD_H
#  include <unistd.h>
#endif
#ifdef	MACOS
#  include <Events.h>
#  include <Dialogs.h>
#endif

/* when changing the version, also update:
 * "graphmac.r"   (for Macintosh)
 * "electric.rc"  (for Windows)
 * "README"
 * "ChangeLog"
 * top-level "Makefile" for packaging releases
 */
char *el_version = "5.4g3";		/* the current Electric version */

/* miscellaneous defaults */
#define	DEFTECH     "nmos"        /* default technology */
#define	DEFCONSTR   "layout"      /* default constraint system */

/* prototypes for local routines */
void myexit(char *str, int status);
void el_primaryinit(void);
void el_secondaryinit(INTBIG, char*[]);
void el_slice(void);

/*
 * the primary initialization of Electric
 */
void el_primaryinit(void)
{
	REGISTER INTSML i;
	REGISTER TECHNOLOGY *tech, *lasttech, *realtech;
	REGISTER CONSTRAINT *constr;
	REGISTER CLUSTER *clus;
	REGISTER LIBRARY *lib;
	extern TECHNOLOGY el_technologylist[];

	/* initialize the memory allocation system */
	db_initclusters();

	/* internal initialization of the constraint solvers */
	el_curconstraint = &el_constraints[0];
	for(i=0; el_constraints[i].conname != 0; i++)
	{
		constr = &el_constraints[i];
		(void)initinfstr();
		(void)addstringtoinfstr("constraint:");
		(void)addstringtoinfstr(constr->conname);
		constr->cluster = alloccluster(returninfstr());
		if (namesame(constr->conname, DEFCONSTR) == 0) el_curconstraint = constr;
	}

	/* internal initialization of the aids */
	for(el_maxaid=0; el_aids[el_maxaid].aidname != 0; el_maxaid++)
	{
		(void)initinfstr();
		(void)addstringtoinfstr("aid:");
		(void)addstringtoinfstr(el_aids[el_maxaid].aidname);
		el_aids[el_maxaid].cluster = alloccluster(returninfstr());
		el_aids[el_maxaid].index = el_maxaid;
	}

	/* initialize the database */
	db_initdatabase();

	/* internal initialization of the technologies */
	lasttech = NOTECHNOLOGY;
	for(el_maxtech=0; el_technologylist[el_maxtech].techname != 0; el_maxtech++)
	{
		tech = &el_technologylist[el_maxtech];

		/* create the real technology object */
		(void)initinfstr();
		(void)addstringtoinfstr("tech:");
		(void)addstringtoinfstr(tech->techname);
		clus = alloccluster(returninfstr());
		realtech = alloctechnology(clus);
		if (realtech == NOTECHNOLOGY) error("No memory for technologies");

		/* link it in */
		if (lasttech == NOTECHNOLOGY) el_technologies = realtech; else
			lasttech->nexttechnology = realtech;
		lasttech = realtech;

		/* initialize the real technology object */
		(void)allocstring(&realtech->techname, tech->techname, clus);
		realtech->index = el_maxtech;
		realtech->deflambda = tech->deflambda;
		realtech->parse = tech->parse;
		realtech->cluster = clus;
		(void)allocstring(&realtech->techdescript, tech->techdescript, clus);
		realtech->layercount = tech->layercount;
		realtech->layers = tech->layers;
		realtech->arcprotocount = tech->arcprotocount;
		realtech->arcprotos = tech->arcprotos;
		realtech->nodeprotocount = tech->nodeprotocount;
		realtech->nodeprotos = tech->nodeprotos;
		realtech->variables = tech->variables;
		realtech->init = tech->init;
		realtech->setmode = tech->setmode;
		realtech->paramnode = tech->paramnode;
		realtech->nodepolys = tech->nodepolys;
		realtech->nodeEpolys = tech->nodeEpolys;
		realtech->shapenodepoly = tech->shapenodepoly;
		realtech->shapeEnodepoly = tech->shapeEnodepoly;
		realtech->shapeportpoly = tech->shapeportpoly;
		realtech->arcpolys = tech->arcpolys;
		realtech->shapearcpoly = tech->shapearcpoly;
		realtech->userbits = tech->userbits;
	}

	/* select a current technology */
	el_curtech = el_technologies;
	for(tech = el_technologies; tech != NOTECHNOLOGY; tech = tech->nexttechnology)
		if (namesame(tech->techname, DEFTECH) == 0) el_curtech = tech;

	/* setup a change batch for initialization work */
	/* db_preparenewbatch(&el_aids[0]); */

	/* pass 1 initialization of the constraint solvers */
	for(i=0; el_constraints[i].conname != 0; i++)
		(*el_constraints[i].init)(&el_constraints[i]);

	/* pass 1 initialization of technologies */
	for(tech = el_technologies; tech != NOTECHNOLOGY; tech = tech->nexttechnology)
	{
		if (tech->init != 0)
		{
			if ((*tech->init)(tech, 0) != 0)
				error("User pass 1 error initializing %s technology", tech->techname);
		}
		if (tech_doinitprocess(tech) != 0)
			error("Pass 1 error initializing %s technology", tech->techname);
	}

	/* pass 1 initialization of the aids */
	for(i=0; i<el_maxaid; i++)
		(*el_aids[i].init)(0, 0, &el_aids[i]);

	/* create the first library */
	el_curlib = NOLIBRARY;
	lib = newlibrary("noname", "noname");
	selectlibrary(lib);

	/* establish the library and documentation locations */
	setupenvironment();
}

/*
 * the secondary initialization of Electric
 */
void el_secondaryinit(INTBIG argc, char *argv[])
{
	REGISTER INTSML i;
	REGISTER TECHNOLOGY *tech;

	/* pass 2 initialization of the constraint solvers */
	for(i=0; el_constraints[i].conname != 0; i++)
		(*el_constraints[i].init)(NOCONSTRAINT);

	/* pass 2 initialization of technologies */
	for(tech = el_technologies; tech != NOTECHNOLOGY; tech = tech->nexttechnology)
	{
		if (tech->init != 0)
		{
			if ((*tech->init)(tech, 1) != 0)
				error("User pass 2 error initializing %s technology", tech->techname);
		}
		if (tech_doaddportsandvars(tech))
			error("Pass 2 error initializing %s technology", tech->techname);
	}

	/* pass 2 initialization of the aids */
	for(i=0; i<el_maxaid; i++) (*el_aids[i].init)(&argc, argv, NOAID);

	/* pass 3 initialization of technologies */
	for(tech = el_technologies; tech != NOTECHNOLOGY; tech = tech->nexttechnology)
	{
		if (tech->init != 0)
		{
			if ((*tech->init)(tech, 2) != 0)
				error("User pass 3 error initializing %s technology", tech->techname);
		}
	}

	/* pass 3 initialization of the aids */
	for(i=0; i<el_maxaid; i++) (*el_aids[i].init)(&argc, argv, 0);

	/* register and initialize technology-variable caching functions */
	db_inittechcache();

	/* delete initialization change batches so it is ready for real changes */
	noundoallowed();
}

void el_slice(void)
{
	REGISTER INTSML s;
	REGISTER AIDENTRY *aid;

	/* the main loop of Electric */
#if !defined(WIN32) || defined(USETK)
	for(;;)
#endif
	{
		for(s=0; s<el_maxaid; s++)
		{
			aid = &el_aids[s];
			if ((aid->aidstate & AIDON) == 0) continue;

			/* let the aid have a whack */
			db_setcurrenttool(aid);
			(*aid->slice)();

			/* announce end of broadcast of changes */
			db_endbatch();
		}
	}
}

INTSML tellaid(AIDENTRY *aid, INTSML count, char *par[])
{
	return((*aid->setmode)(count, par));
}

INTBIG askaid(AIDENTRY *aid, char *command, ...)
{
	va_list ap;
	INTBIG result;

	var_start(ap, command);
	result = (*aid->request)(command, ap);
	va_end(ap);
	return(result);
}

/*
 * routine to turn aid "aid" back on.  It will also inform the aid of each
 * facet that changed while the aid was off (via "examinenodeproto").  If
 * "nocatchup" is nonzero, this informing will not occur.
 */
void aidturnon(AIDENTRY *aid, INTSML nocatchup)
{
	REGISTER INTBIG bit;
	REGISTER NODEPROTO *np;

	/* turn on the aid */
	(void)setval((INTBIG)aid, VAID, "aidstate", aid->aidstate|AIDON, VINTEGER);

	/* if the aid is not incremental, don't inform it of changes while off */
	if ((aid->aidstate & AIDINCREMENTAL) == 0) return;

	/* if catch-up action is supressed, quit now */
	if (nocatchup != 0) return;

	/* look through all facets and update those that need it */
	bit = 1 << aid->index;
	for(np=el_curlib->firstnodeproto; np!=NONODEPROTO; np=np->nextnodeproto)
	{
		if (np->adirty & bit) (*aid->examinenodeproto)(np);
		np->adirty &= ~bit;
	}
}

/*
 * routine to turn aid "aid" off.  If "permanently" is nonzero, this change
 * will not be undoable
 */
void aidturnoff(AIDENTRY *aid, INTSML permanently)
{
	/* turn off the aid */
	if (permanently != 0) aid->aidstate &= ~AIDON; else
		(void)setval((INTBIG)aid, VAID, "aidstate", aid->aidstate & ~AIDON, VINTEGER);
}

/*
 * shutdown the design aid
 */
void bringdown(void)
{
	REGISTER INTSML i;

	/* go backwards through list to shut down in proper order */
	for(i = el_maxaid-1; i >= 0; i--) (*el_aids[i].done)();

	/* terminate any language interpreters */
	db_termlanguage();

	/* shut down the constraint solvers */
	for(i=0; el_constraints[i].conname != 0; i++) (*el_constraints[i].term)();
	exitprogram();
}

void error(char *s, ...)
{
	va_list ap;
	char line[256];

#ifdef	MACOS
	tty_close();
	var_start(ap, s);
	(void)vsprintf(&line[1], s, ap);
	va_end(ap);
	line[0] = strlen(&line[1]);
	ParamText("\pERROR: ", (unsigned char *)line, "\p", "\p");
	StopAlert(130, 0L);
	exitprogram();
#else
	REGISTER INTSML count;
	char *pars[10];
	extern COMCOMP us_noyesp;

	tty_close();
	var_start(ap, s);
	(void)vsprintf(line, s, ap);
	va_end(ap);
	(void)printf("electric: %s\n", line);

	ttyputmsg("Exiting system: status - 1, reason - call to error");
	count = ttygetparam("Exit system? [y] ", &us_noyesp, 2, pars);
	if (count <= 0 || namesamen(pars[0], "yes", (INTSML)strlen(pars[0])) != 0) exitprogram();
#endif
}

INTSML stopping(char *who)
{
	checkforinterrupt();
	if (el_pleasestop != 1) return(el_pleasestop);
	el_pleasestop = 0;
	flushscreen();
	ttyputmsg("%s stopped", who);
	el_pleasestop = 2;
	return(el_pleasestop);
}

INTBIG myfork(void)
{
#ifdef ONUNIX
	return(fork());
#else
	return(0);
#endif
}

/*
 * routine to wait for the completion of child process "process"
 */
void waitfor(INTBIG process)
{
#ifdef ONUNIX
	REGISTER INTBIG pid;

	for(;;)
	{
		pid = wait((int *)0);
		if (pid == process) return;
		if (pid == -1)
		{
			perror("Waiting");
			return;
		}
	}
#endif
}

/*
 * internal routine to offer a core dump when an error has occurred
 */
void db_offercoredump(void)
{
#ifdef	MACOS
	error("Program exiting due to severe error");
#else
	REGISTER INTSML count;
	char *pars[10];
	extern COMCOMP us_noyesp;

	count = ttygetparam("Would you like a core-dump? [n] ", &us_noyesp, 2, pars);
	if (count <= 0 || namesamen(pars[0], "yes", (INTSML)strlen(pars[0])) != 0)
		return;

	/* turn off the catching of this error */
	(void)signal(SIGILL, SIG_DFL);

	/* generate an error */
	abort();
#endif
}

void myexit(char *str, int status)
{
	REGISTER INTSML count;
	char *pars[10], msg[300];
	extern COMCOMP us_noyesp;

	sprintf(msg, "ERROR %s (status %d).  Exit? [y] ", str, status);
	count = ttygetparam(msg, &us_noyesp, 2, pars);
	if (count <= 0 || namesamen(pars[0], "yes", (INTSML)strlen(pars[0])) != 0)
		exitprogram();
}
