/*
 * Electric VLSI Design System
 *
 * file: dblangelk.c
 * ELK Lisp interface module
 * Written by: Steven M. Rubin, Apple Computer Inc.
 *
 * (c) Electric Editor, Incorporated 1994.
 * All or a portion of this program is subject to the restrictions
 * contained in a license granted by, and remains the unpublished
 * property of, Electric Editor, Incorporated.
 */

#include "config.h"
#if LANGLISP

#include "global.h"
#include "dblang.h"
Object lsp_displayablesym;
INTSML lsp_strportinited = 0;
Object lsp_strport;

#define Ensure_Type(x,t) if (TYPE(x) != t) { Wrong_Type(x, t); return(Null); }

/* prototypes for local routines */
int EElectric_Equal(Object, Object);
int ENodeInst_Print(Object, Object, int, int, int);
int ENodeProto_Print(Object, Object, int, int, int);
int EPortArcInst_Print(Object, Object, int, int, int);
int EPortExpInst_Print(Object, Object, int, int, int);
int EPortProto_Print(Object, Object, int, int, int);
int EArcInst_Print(Object, Object, int, int, int);
int EArcProto_Print(Object, Object, int, int, int);
int EGeom_Print(Object, Object, int, int, int);
int ELibrary_Print(Object, Object, int, int, int);
int ETechnology_Print(Object, Object, int, int, int);
int EAid_Print(Object, Object, int, int, int);
int ERTNode_Print(Object, Object, int, int, int);
int ENetwork_Print(Object, Object, int, int, int);
int ECell_Print(Object, Object, int, int, int);
int EView_Print(Object, Object, int, int, int);
int EWindow_Print(Object, Object, int, int, int);
int EGraphics_Print(Object, Object, int, int, int);
int EConstraint_Print(Object, Object, int, int, int);
Object P_ENodeInstP(Object);
Object P_ENodeProtoP(Object);
Object P_EPortArcInstP(Object);
Object P_EPortExpInstP(Object);
Object P_EPortProtoP(Object);
Object P_EArcInstP(Object);
Object P_EArcProtoP(Object);
Object P_EGeomP(Object);
Object P_ELibraryP(Object);
Object P_ETechnologyP(Object);
Object P_EAidP(Object);
Object P_ERTNodeP(Object);
Object P_ENetworkP(Object);
Object P_ECellP(Object);
Object P_EViewP(Object);
Object P_EWindowP(Object);
Object P_EGraphicsP(Object);
Object P_EConstraintP(Object);
Object Make_EElectric(INTBIG, INTSML);
void init_lib_electric(void);
INTSML lsp_getnumericobject(Object, INTBIG*);
char *lsp_getstringobject(Object);
void lsp_getaddrandtype(Object, INTBIG*, INTBIG*);
Object lsp_makevarobject(INTBIG, INTBIG);
Object lsp_curlib(void);
Object lsp_curtech(void);
Object lsp_getval(Object, Object);
Object lsp_setval(Object, Object, Object, Object);
Object lsp_setind(Object, Object, Object, Object);
Object lsp_delval(Object, Object);
Object lsp_initsearch(Object, Object, Object, Object, Object);
Object lsp_nextobject(Object);
Object lsp_getaid(Object);
Object lsp_maxaid(void);
Object lsp_indexaid(Object);
Object lsp_aidturnon(Object, Object);
Object lsp_aidturnoff(Object);
Object lsp_getlibrary(Object);
Object lsp_newlibrary(Object, Object);
Object lsp_killlibrary(Object);
Object lsp_eraselibrary(Object);
Object lsp_selectlibrary(Object);
Object lsp_getnodeproto(Object);
Object lsp_newnodeproto(Object, Object);
Object lsp_killnodeproto(Object);
Object lsp_copynodeproto(Object, Object, Object);
Object lsp_iconview(Object);
Object lsp_contentsview(Object);
Object lsp_gettraversalpath(void);
Object lsp_newnodeinst(Object, Object, Object, Object, Object, Object, Object, Object);
Object lsp_modifynodeinst(Object, Object, Object, Object, Object, Object, Object);
Object lsp_killnodeinst(Object);
Object lsp_replacenodeinst(Object, Object);
Object lsp_nodefunction(Object);
Object lsp_newarcinst(Object, Object, Object, Object, Object, Object, Object, Object, Object, Object, Object, Object);
Object lsp_modifyarcinst(Object, Object, Object, Object, Object, Object);
Object lsp_killarcinst(Object);
Object lsp_replacearcinst(Object, Object);
Object lsp_newportproto(Object, Object, Object, Object);
Object lsp_portposition(Object, Object);
Object lsp_getportproto(Object, Object);
Object lsp_killportproto(Object, Object);
Object lsp_moveportproto(Object, Object, Object, Object);
Object lsp_undoabatch(void);
Object lsp_noundoallowed(void);
Object lsp_getview(Object);
Object lsp_newview(Object, Object);
Object lsp_killview(Object);
Object lsp_changelambda(Object, Object, Object);
Object lsp_tellaid(long, Object[]);
Object lsp_getarcproto(Object);
Object lsp_getcell(Object);
Object lsp_gettechnology(Object);
Object lsp_getpinproto(Object);
Object lsp_getnetwork(Object, Object);

/****************************** ELECTRIC OBJECTS ******************************/

/* the internal LISP types */
long T_ENodeInst, T_ENodeProto, T_EPortArcInst, T_EPortExpInst, T_EPortProto,
	T_EArcInst, T_EArcProto, T_EGeom, T_ELibrary, T_ETechnology, T_EAid,
	T_ERTNode, T_ENetwork, T_ECell, T_EView, T_EWindow, T_EGraphics, T_EConstraint;

/* the LISP structures */
struct S_Electric     { INTBIG        handle; };
struct S_ENodeInst    { NODEINST    *handle; };
struct S_ENodeProto   { NODEPROTO   *handle; };
struct S_EPortArcInst { PORTARCINST *handle; };
struct S_EPortExpInst { PORTEXPINST *handle; };
struct S_EPortProto   { PORTPROTO   *handle; };
struct S_EArcInst     { ARCINST     *handle; };
struct S_EArcProto    { ARCPROTO    *handle; };
struct S_EGeom        { GEOM        *handle; };
struct S_ELibrary     { LIBRARY     *handle; };
struct S_ETechnology  { TECHNOLOGY  *handle; };
struct S_EAid         { AIDENTRY    *handle; };
struct S_ERTNode      { RTNODE      *handle; };
struct S_ENetwork     { NETWORK     *handle; };
struct S_ECell        { CELL        *handle; };
struct S_EView        { VIEW        *handle; };
struct S_EWindow      { WINDOW      *handle; };
struct S_EGraphics    { GRAPHICS    *handle; };
struct S_EConstraint  { CONSTRAINT  *handle; };

/* macros for accessing Electric objects from the Lisp structures */
#define	EELECTRIC(obj)    ((struct S_Electric *)POINTER(obj))
#define	ENODEINST(obj)    ((struct S_ENodeInst *)POINTER(obj))
#define	ENODEPROTO(obj)   ((struct S_ENodeProto *)POINTER(obj))
#define	EPORTARCINST(obj) ((struct S_EPortArcInst *)POINTER(obj))
#define	EPORTEXPINST(obj) ((struct S_EPortExpInst *)POINTER(obj))
#define	EPORTPROTO(obj)   ((struct S_EPortProto *)POINTER(obj))
#define	EARCINST(obj)     ((struct S_EArcInst *)POINTER(obj))
#define	EARCPROTO(obj)    ((struct S_EArcProto *)POINTER(obj))
#define	EGEOM(obj)        ((struct S_EGeom *)POINTER(obj))
#define	ELIBRARY(obj)     ((struct S_ELibrary *)POINTER(obj))
#define	ETECHNOLOGY(obj)  ((struct S_ETechnology *)POINTER(obj))
#define	EAID(obj)         ((struct S_EAid *)POINTER(obj))
#define	ERTNODE(obj)      ((struct S_ERTNode *)POINTER(obj))
#define	ENETWORK(obj)     ((struct S_ENetwork *)POINTER(obj))
#define	ECELL(obj)        ((struct S_ECell *)POINTER(obj))
#define	EVIEW(obj)        ((struct S_EView *)POINTER(obj))
#define	EWINDOW(obj)      ((struct S_EWindow *)POINTER(obj))
#define	EGRAPHICS(obj)    ((struct S_EGraphics *)POINTER(obj))
#define	ECONSTRAINT(obj)  ((struct S_EConstraint *)POINTER(obj))

/* equality routine */
int EElectric_Equal(Object a, Object b) { return(EELECTRIC(a)->handle == EELECTRIC(b)->handle); }

/* print routines */
int ENodeInst_Print(Object w, Object port, int raw, int depth, int len)
{ Printf(port, "#[nodeinst %u]", ENODEINST(w)->handle);  return(0); }
int ENodeProto_Print(Object w, Object port, int raw, int depth, int len)
{ Printf(port, "#[nodeproto %s]", describenodeproto(ENODEPROTO(w)->handle));  return(0); }
int EPortArcInst_Print(Object w, Object port, int raw, int depth, int len)
{ Printf(port, "#[portarcinst %u]", EPORTARCINST(w)->handle);  return(0); }
int EPortExpInst_Print(Object w, Object port, int raw, int depth, int len)
{ Printf(port, "#[portexpinst %u]", EPORTEXPINST(w)->handle);  return(0); }
int EPortProto_Print(Object w, Object port, int raw, int depth, int len)
{ Printf(port, "#[portproto %s]", EPORTPROTO(w)->handle->protoname);  return(0); }
int EArcInst_Print(Object w, Object port, int raw, int depth, int len)
{ Printf(port, "#[arcinst %u]", EARCINST(w)->handle);  return(0); }
int EArcProto_Print(Object w, Object port, int raw, int depth, int len)
{ Printf(port, "#[arcproto %s]", EARCPROTO(w)->handle->protoname);  return(0); }
int EGeom_Print(Object w, Object port, int raw, int depth, int len)
{ Printf(port, "#[geom %u]", EGEOM(w)->handle);  return(0); }
int ELibrary_Print(Object w, Object port, int raw, int depth, int len)
{ Printf(port, "#[library %s]", ELIBRARY(w)->handle->libname);  return(0); }
int ETechnology_Print(Object w, Object port, int raw, int depth, int len)
{ Printf(port, "#[technology %s]", ETECHNOLOGY(w)->handle->techname);  return(0); }
int EAid_Print(Object w, Object port, int raw, int depth, int len)
{ Printf(port, "#[aid %s]", EAID(w)->handle->aidname);  return(0); }
int ERTNode_Print(Object w, Object port, int raw, int depth, int len)
{ Printf(port, "#[r-tree %u]", ERTNODE(w)->handle);  return(0); }
int ENetwork_Print(Object w, Object port, int raw, int depth, int len)
{ Printf(port, "#[network %u]", ENETWORK(w)->handle);  return(0); }
int ECell_Print(Object w, Object port, int raw, int depth, int len)
{ Printf(port, "#[cell %s]", ECELL(w)->handle->cellname);  return(0); }
int EView_Print(Object w, Object port, int raw, int depth, int len)
{ Printf(port, "#[view %s]", EVIEW(w)->handle->viewname);  return(0); }
int EWindow_Print(Object w, Object port, int raw, int depth, int len)
{ Printf(port, "#[window %u]", EWINDOW(w)->handle);  return(0); }
int EGraphics_Print(Object w, Object port, int raw, int depth, int len)
{ Printf(port, "#[graphics %u]", EGRAPHICS(w)->handle);  return(0); }
int EConstraint_Print(Object w, Object port, int raw, int depth, int len)
{ Printf(port, "#[constraint %s]", ECONSTRAINT(w)->handle->conname);  return(0); }

/* type query routines */
Object P_ENodeInstP(Object x)    { return(TYPE(x) == T_ENodeInst    ? True : False); }
Object P_ENodeProtoP(Object x)   { return(TYPE(x) == T_ENodeProto   ? True : False); }
Object P_EPortArcInstP(Object x) { return(TYPE(x) == T_EPortArcInst ? True : False); }
Object P_EPortExpInstP(Object x) { return(TYPE(x) == T_EPortExpInst ? True : False); }
Object P_EPortProtoP(Object x)   { return(TYPE(x) == T_EPortProto   ? True : False); }
Object P_EArcInstP(Object x)     { return(TYPE(x) == T_EArcInst     ? True : False); }
Object P_EArcProtoP(Object x)    { return(TYPE(x) == T_EArcProto    ? True : False); }
Object P_EGeomP(Object x)        { return(TYPE(x) == T_EGeom        ? True : False); }
Object P_ELibraryP(Object x)     { return(TYPE(x) == T_ELibrary     ? True : False); }
Object P_ETechnologyP(Object x)  { return(TYPE(x) == T_ETechnology  ? True : False); }
Object P_EAidP(Object x)         { return(TYPE(x) == T_EAid         ? True : False); }
Object P_ERTNodeP(Object x)      { return(TYPE(x) == T_ERTNode      ? True : False); }
Object P_ENetworkP(Object x)     { return(TYPE(x) == T_ENetwork     ? True : False); }
Object P_ECellP(Object x)        { return(TYPE(x) == T_ECell        ? True : False); }
Object P_EViewP(Object x)        { return(TYPE(x) == T_EView        ? True : False); }
Object P_EWindowP(Object x)      { return(TYPE(x) == T_EWindow      ? True : False); }
Object P_EGraphicsP(Object x)    { return(TYPE(x) == T_EGraphics    ? True : False); }
Object P_EConstraintP(Object x)  { return(TYPE(x) == T_EConstraint  ? True : False); }

Object Make_EElectric(INTBIG obj, INTSML type)
{
	char *p;
	Object w;

	p = (char *)emalloc(sizeof (struct S_Electric), el_tempcluster);
	SET(w, type, p);
	EELECTRIC(w)->handle = obj;
	return(w);
}

/****************************** UTILITIES ******************************/

void lsp_init(void)
{
	char *av[1], progname[30];

	if (lsp_strportinited != 0) return;
	lsp_strportinited = 1;

	/* initialize Lisp (must do it this way to give it stack base info) */
	strcpy(progname, "electric");
	av[0] = progname;
	Elk_Init (1, av, 0, "toplevel.scm");
	init_lib_electric();

	/* load the top-level interpreter */
/*	(void)General_Load(Make_String("toplevel.scm", 12), The_Environment); */

	/* create the dummy string port */
	lsp_strport = Make_Port(0, (FILE *)0, Make_String ((char *)0, 0));
	Global_GC_Link(lsp_strport);
}

/*
 * routine to convert a "C" string into a lisp Object
 */
Object lsp_makeobject(char *str)
{
	Object ret;
	register c, konst = 1;
	Object Read_Sequence (Object port, int vec, int konst);
	Object Read_Atom (Object port, int konst);
	int String_Getc(Object port);
	int Skip_Comment(Object port);
	int String_Ungetc(Object port, int c);

	/* place the string in the port */
	PORT(lsp_strport)->ptr = 0;
	PORT(lsp_strport)->name = Make_String(str, strlen(str));
	PORT(lsp_strport)->flags = P_STRING|P_INPUT|P_OPEN;

	/* read from the string port into an object */
	ret = Eof;
	for(;;)
	{
		c = String_Getc(lsp_strport);
		if (c == EOF) break;
		if (Whitespace(c)) continue;
		if (c == ';')
		{
			if (Skip_Comment(lsp_strport) == EOF) break;
			continue;
		}
		if (c == '(')
		{
			ret = Read_Sequence(lsp_strport, 0, konst);
		} else
		{
			String_Ungetc(lsp_strport, c);
			ret = Read_Atom(lsp_strport, konst);
		}
		break;
	}

	/* free the string pointer */
	PORT(lsp_strport)->name = Null;
	return(ret);
}

/*
 * routine to convert Lisp Object "obj" into an Electric address of type "type" and return
 * it in "retval".  Returns nonzero on error.
 */
INTSML lsp_describeobject(Object obj, INTBIG type, INTBIG *retval)
{
	static char retstr[100];
	static INTBIG retarray[1];
	INTSML len, t;
	Object strobj;
	char *str, *p;
	void Print_Bignum(Object port, Object x);
	void Pr_List(Object port, Object list, int raw, int depth, int length);
	void Pr_Vector(Object port, Object vec, int raw, int depth, int length);
	int Print_Char(Object port, int c);

	t = TYPE(obj);
	switch (type&VTYPE)
	{
		case VINTEGER:
		case VSHORT:
		case VADDRESS:
		case VFRACT:
			switch (t)
			{
				case T_Fixnum:
				case T_Boolean:
					*retval = FIXNUM(obj);
					break;
				case T_Bignum:
					PORT(lsp_strport)->ptr = 0;
					PORT(lsp_strport)->name = Make_String(0, 0);
					PORT(lsp_strport)->flags = P_STRING|P_OPEN;
					Print_Bignum(lsp_strport, obj);
					strobj = PORT(lsp_strport)->name;
					str = STRING(strobj)->data;
					len = PORT(lsp_strport)->ptr;
					(void)strncpy(retstr, str, len);
					retstr[len] = 0;
					*retval = myatoi(retstr);
					break;
				case T_Flonum:
					*retval = (INTBIG)FLONUM(obj)->val;
					break;
				default:
					return(1);
			}
			if ((type&VTYPE) == VFRACT) *retval *= WHOLE;
			break;

		case VFLOAT:
		case VDOUBLE:
			switch (t)
			{
				case T_Fixnum:
				case T_Boolean:
					*retval = castint((float)FIXNUM(obj));
					break;
				case T_Bignum:
					PORT(lsp_strport)->ptr = 0;
					PORT(lsp_strport)->name = Make_String(0, 0);
					PORT(lsp_strport)->flags = P_STRING|P_OPEN;
					Print_Bignum(lsp_strport, obj);
					strobj = PORT(lsp_strport)->name;
					str = STRING(strobj)->data;
					len = PORT(lsp_strport)->ptr;
					(void)strncpy(retstr, str, len);
					retstr[len] = 0;
					*retval = castint(atof(retstr));
					break;
				case T_Flonum:
					*retval = castint(FLONUM(obj)->val);
					break;
				default:
					return(1);
			}
			break;

		case VCHAR:
			if (t != T_Character) return(1);
			*retval = CHAR(obj);
			break;

		case VSTRING:
			*retval = (INTBIG)"";
			switch (t)
			{
				case T_Null:
					*retval = (INTBIG)"()";
					break;
				case T_Fixnum:
					(void)sprintf(retstr, "%d", (int)FIXNUM(obj));
					*retval = (INTBIG)retstr;
					break;
				case T_Bignum:
					PORT(lsp_strport)->ptr = 0;
					PORT(lsp_strport)->name = Make_String(0, 0);
					PORT(lsp_strport)->flags = P_STRING|P_OPEN;
					Print_Bignum(lsp_strport, obj);
					strobj = PORT(lsp_strport)->name;
					str = STRING(strobj)->data;
					len = PORT(lsp_strport)->ptr;
					(void)strncpy(retstr, str, len);
					retstr[len] = 0;
					*retval = (INTBIG)retstr;
					break;
#if 0
				case T_Flonum:
					(void)sprintf(retstr, FLONUM_FORMAT, FLONUM(obj)->val);
					*retval = (INTBIG)retstr;
					break;
#endif
				case T_Boolean:
					if (FIXNUM(obj) != 0) *retval = (INTBIG)"t"; else
						*retval = (INTBIG)"f";
					break;
				case T_Unbound:
					*retval = (INTBIG)"#[unbound]";
					break;
				case T_Special:
					*retval = (INTBIG)"#[special]";
					break;
				case T_Character:
					retstr[0] = CHAR(obj);
					retstr[1] = 0;
					*retval = (INTBIG)retstr;
					break;
				case T_Symbol:
					strobj = SYMBOL(obj)->name;
					str = STRING(strobj)->data;
					len = STRING(strobj)->size;
					(void)strncpy(retstr, str, len);
					retstr[len] = 0;
					*retval = (INTBIG)retstr;
					break;
				case T_Pair:
					PORT(lsp_strport)->ptr = 0;
					PORT(lsp_strport)->name = Make_String(0, 0);
					PORT(lsp_strport)->flags = P_STRING|P_OPEN;
					Pr_List(lsp_strport, obj, 1, DEF_PRINT_DEPTH, DEF_PRINT_LEN);
					strobj = PORT(lsp_strport)->name;
					str = STRING(strobj)->data;
					len = PORT(lsp_strport)->ptr;
					(void)strncpy(retstr, str, len);
					retstr[len] = 0;
					*retval = (INTBIG)retstr;
					break;
				case T_Environment:
					(void)sprintf(retstr, "#[environment %u]", (unsigned int)POINTER(obj));
					*retval = (INTBIG)retstr;
					break;
				case T_String:
					str = STRING(obj)->data;
					len = STRING(obj)->size;
					(void)strncpy(retstr, str, len);
					retstr[len] = 0;
					*retval = (INTBIG)retstr;
					break;
				case T_Vector:
					PORT(lsp_strport)->ptr = 0;
					PORT(lsp_strport)->name = Make_String(0, 0);
					PORT(lsp_strport)->flags = P_STRING|P_OPEN;
					Pr_Vector(lsp_strport, obj, 1, DEF_PRINT_DEPTH, DEF_PRINT_LEN);
					strobj = PORT(lsp_strport)->name;
					str = STRING(strobj)->data;
					len = PORT(lsp_strport)->ptr;
					(void)strncpy(retstr, str, len);
					retstr[len] = 0;
					*retval = (INTBIG)retstr;
					break;
				case T_Primitive:
					(void)sprintf(retstr, "#[primitive %s]", PRIM(obj)->name);
					*retval = (INTBIG)retstr;
					break;
				case T_Compound:
					if (Nullp(COMPOUND(obj)->name))
					{
						(void)sprintf(retstr, "#[compound %u]", (unsigned int)POINTER(obj));
						*retval = (INTBIG)retstr;
						break;
					}
					PORT(lsp_strport)->ptr = 0;
					PORT(lsp_strport)->name = Make_String(0, 0);
					PORT(lsp_strport)->flags = P_STRING|P_OPEN;
					Printf(lsp_strport, "#[compound ");
					Print_Object(COMPOUND(obj)->name, lsp_strport, 1, DEF_PRINT_DEPTH, DEF_PRINT_LEN);
					Print_Char(lsp_strport, ']');
					strobj = PORT(lsp_strport)->name;
					str = STRING(strobj)->data;
					len = PORT(lsp_strport)->ptr;
					(void)strncpy(retstr, str, len);
					retstr[len] = 0;
					*retval = (INTBIG)retstr;
					break;
				case T_Control_Point:
					(void)sprintf(retstr, "#[control-point %u]", (unsigned int)POINTER(obj));
					*retval = (INTBIG)retstr;
					break;
				case T_Promise:
					(void)sprintf(retstr, "#[promise %u]", (unsigned int)POINTER(obj));
					*retval = (INTBIG)retstr;
					break;
				case T_Port:
					switch (PORT(obj)->flags & (P_INPUT|P_BIDIR))
					{
						case 0:       p = "output";       break;
						case P_INPUT: p = "input";        break;
						default:      p = "input-output"; break;
					}
					if ((PORT(obj)->flags&P_STRING) != 0)
						(void)sprintf(retstr, "#[string-%s-port %u]", p, (unsigned int)POINTER(obj)); else
					{
						strobj = PORT(obj)->name;
						str = STRING(strobj)->data;
						len = STRING(strobj)->size;
						(void)sprintf(retstr, "#[file-%s-port ", p);
						(void)strncat(retstr, str, len);
						(void)strcat(retstr, "]");
					}
					*retval = (INTBIG)retstr;
					break;
				case T_End_Of_File:
					*retval = (INTBIG)"#[end-of-file]";
					break;
				case T_Autoload:
					PORT(lsp_strport)->ptr = 0;
					PORT(lsp_strport)->name = Make_String(0, 0);
					PORT(lsp_strport)->flags = P_STRING|P_OPEN;
					Printf(lsp_strport, "#[autoload ");
					Print_Object(AUTOLOAD(obj)->files, lsp_strport, 1, DEF_PRINT_DEPTH, DEF_PRINT_LEN);
					Print_Char(lsp_strport, ']');
					strobj = PORT(lsp_strport)->name;
					str = STRING(strobj)->data;
					len = PORT(lsp_strport)->ptr;
					(void)strncpy(retstr, str, len);
					retstr[len] = 0;
					*retval = (INTBIG)retstr;
					break;
				case T_Macro:
					if (Nullp(MACRO(obj)->name))
					{
						(void)sprintf(retstr, "#[macro %u]", (unsigned int)POINTER(obj));
						*retval = (INTBIG)retstr;
						break;
					}
					PORT(lsp_strport)->ptr = 0;
					PORT(lsp_strport)->name = Make_String(0, 0);
					PORT(lsp_strport)->flags = P_STRING|P_OPEN;
					Printf(lsp_strport, "#[macro ");
					Print_Object(MACRO(obj)->name, lsp_strport, 1, DEF_PRINT_DEPTH, DEF_PRINT_LEN);
					Print_Char(lsp_strport, ']');
					strobj = PORT(lsp_strport)->name;
					str = STRING(strobj)->data;
					len = PORT(lsp_strport)->ptr;
					(void)strncpy(retstr, str, len);
					retstr[len] = 0;
					*retval = (INTBIG)retstr;
					break;
				case T_Broken_Heart:
					*retval = (INTBIG)"!!broken-heart!!";
					break;
				default:
					if (t < 0 || t >= MAX_TYPE || !Types[t].name) break;
					PORT(lsp_strport)->ptr = 0;
					PORT(lsp_strport)->name = Make_String(0, 0);
					PORT(lsp_strport)->flags = P_STRING|P_OPEN;
					(*Types[t].print)(obj, lsp_strport, 1, DEF_PRINT_DEPTH, DEF_PRINT_LEN);
					strobj = PORT(lsp_strport)->name;
					str = STRING(strobj)->data;
					len = PORT(lsp_strport)->ptr;
					(void)strncpy(retstr, str, len);
					retstr[len] = 0;
					*retval = (INTBIG)retstr;
					break;
			}
			break;

		default:
			return(1);
	}

	if ((type&VISARRAY) != 0)
	{
		retarray[0] = *retval;
		*retval = (INTBIG)retarray;
	}
	return(0);
}

/*
 * routine to convert a numeric Lisp Object into an integer.  Returns nonzero on error
 */
INTSML lsp_getnumericobject(Object obj, INTBIG *val)
{
	int Bignum_To_Integer (Object x);

	switch (TYPE(obj))
	{
		case T_Fixnum: *val = FIXNUM(obj);              return(0);
		case T_Bignum: *val = Bignum_To_Integer(obj);   return(0);
		case T_Flonum: *val = (INTBIG)FLONUM(obj)->val; return(0);
		case T_Symbol:
			if (EQ(obj, lsp_displayablesym)) { *val = VDISPLAY;   return(0); }
			break;
	}
	return(1);
}

/*
 * routine to convert a numeric Lisp Object into a string.  Returns zero on error
 */
char *lsp_getstringobject(Object obj)
{
	static char retval[256];
	char *str;
	INTBIG len;

	switch (TYPE(obj))
	{
		case T_Symbol:
			obj = SYMBOL(obj)->name;
		case T_String:
			str = STRING(obj)->data;
			len = STRING(obj)->size;
			(void)strncpy(retval, str, len);
			retval[len] = 0;
			return(retval);
	}
	return(0);
}


/*
 * routine to convert a Lisp Object into an Electric addr/type pair
 */
void lsp_getaddrandtype(Object oaddr, INTBIG *addr, INTBIG *type)
{
	INTSML otype;
	static char retval[256];
	char *str;
	INTBIG len;

	*type = VUNKNOWN;
	otype = TYPE(oaddr);
	if (otype == T_Fixnum)
	{
		*type = VINTEGER;
		*addr = FIXNUM(oaddr);
		return;
	}
	if (otype == T_Bignum)
	{
		*type = VINTEGER;
		*addr = Bignum_To_Integer(oaddr);
		return;
	}
	if (otype == T_Flonum)
	{
		*type = VFLOAT;
		*addr = castint(FLONUM(oaddr)->val);
		return;
	}
	if (otype == T_String)
	{
		*type = VSTRING;
		str = STRING(oaddr)->data;
		len = STRING(oaddr)->size;
		(void)strncpy(retval, str, len);
		retval[len] = 0;
		*addr = (INTBIG)retval;
		return;
	}
	if (otype == T_ENodeInst)    *type = VNODEINST; else
	if (otype == T_ENodeProto)   *type = VNODEPROTO; else
	if (otype == T_EPortArcInst) *type = VPORTARCINST; else
	if (otype == T_EPortExpInst) *type = VPORTEXPINST; else
	if (otype == T_EPortProto)   *type = VPORTPROTO; else
	if (otype == T_EArcInst)     *type = VARCINST; else
	if (otype == T_EArcProto)    *type = VARCPROTO; else
	if (otype == T_EGeom)        *type = VGEOM; else
	if (otype == T_ELibrary)     *type = VLIBRARY; else
	if (otype == T_ETechnology)  *type = VTECHNOLOGY; else
	if (otype == T_EAid)         *type = VAID; else
	if (otype == T_ERTNode)      *type = VRTNODE; else
	if (otype == T_ENetwork)     *type = VNETWORK; else
	if (otype == T_ECell)        *type = VCELL; else
	if (otype == T_EView)        *type = VVIEW; else
	if (otype == T_EWindow)      *type = VWINDOW; else
	if (otype == T_EGraphics)    *type = VGRAPHICS; else
	if (otype == T_EConstraint)  *type = VCONSTRAINT; else return;
	*addr = EELECTRIC(oaddr)->handle;
}

Object lsp_makevarobject(INTBIG type, INTBIG addr)
{
	Object ret;

	/* convert back to a Lisp object */
	switch (type&VTYPE)
	{
		case VINTEGER:     return(Make_Integer(addr));
		case VSHORT:       return(Make_Fixnum(addr));
		case VADDRESS:     return(Make_Unsigned(addr));
/*	case VCHAR: */        /* character variable */
/*	case VFRACT: */       /* fractional integer (scaled by WHOLE) */
		case VSTRING:      return(Make_String((char *)addr, strlen((char *)addr)));
		case VFLOAT:
		case VDOUBLE:      return(Make_Reduced_Flonum((double)castfloat(addr)));
		case VNODEINST:
			ret = Make_EElectric(addr, (INTSML)T_ENodeInst);
			if (ENODEINST(ret)->handle == NONODEINST) return(Null); else return(ret);
		case VNODEPROTO:
			ret = Make_EElectric(addr, (INTSML)T_ENodeProto);
			if (ENODEPROTO(ret)->handle == NONODEPROTO) return(Null); else return(ret);
		case VPORTARCINST:
			ret = Make_EElectric(addr, (INTSML)T_EPortArcInst);
			if (EPORTARCINST(ret)->handle == NOPORTARCINST) return(Null); else return(ret);
		case VPORTEXPINST:
			ret = Make_EElectric(addr, (INTSML)T_EPortExpInst);
			if (EPORTEXPINST(ret)->handle == NOPORTEXPINST) return(Null); else return(ret);
		case VPORTPROTO:
			ret = Make_EElectric(addr, (INTSML)T_EPortProto);
			if (EPORTPROTO(ret)->handle == NOPORTPROTO) return(Null); else return(ret);
		case VARCINST:
			ret = Make_EElectric(addr, (INTSML)T_EArcInst);
			if (EARCINST(ret)->handle == NOARCINST) return(Null); else return(ret);
		case VARCPROTO:
			ret = Make_EElectric(addr, (INTSML)T_EArcProto);
			if (EARCPROTO(ret)->handle == NOARCPROTO) return(Null); else return(ret);
		case VGEOM:
			ret = Make_EElectric(addr, (INTSML)T_EGeom);
			if (EGEOM(ret)->handle == NOGEOM) return(Null); else return(ret);
		case VLIBRARY:
			ret = Make_EElectric(addr, (INTSML)T_ELibrary);
			if (ELIBRARY(ret)->handle == NOLIBRARY) return(Null); else return(ret);
		case VTECHNOLOGY:
			ret = Make_EElectric(addr, (INTSML)T_ETechnology);
			if (ETECHNOLOGY(ret)->handle == NOTECHNOLOGY) return(Null); else return(ret);
		case VAID:
			ret = Make_EElectric(addr, (INTSML)T_EAid);
			if (EAID(ret)->handle == NOAID) return(Null); else return(ret);
		case VRTNODE:
			ret = Make_EElectric(addr, (INTSML)T_ERTNode);
			if (ERTNODE(ret)->handle == NORTNODE) return(Null); else return(ret);
		case VNETWORK:
			ret = Make_EElectric(addr, (INTSML)T_ENetwork);
			if (ENETWORK(ret)->handle == NONETWORK) return(Null); else return(ret);
		case VCELL:
			ret = Make_EElectric(addr, (INTSML)T_ECell);
			if (ECELL(ret)->handle == NOCELL) return(Null); else return(ret);
		case VVIEW:
			ret = Make_EElectric(addr, (INTSML)T_EView);
			if (EVIEW(ret)->handle == NOVIEW) return(Null); else return(ret);
		case VWINDOW:
			ret = Make_EElectric(addr, (INTSML)T_EWindow);
			if (EWINDOW(ret)->handle == NOWINDOW) return(Null); else return(ret);
		case VGRAPHICS:
			ret = Make_EElectric(addr, (INTSML)T_EGraphics);
			if (EGRAPHICS(ret)->handle == NOGRAPHICS) return(Null); else return(ret);
		case VCONSTRAINT:
			ret = Make_EElectric(addr, (INTSML)T_EConstraint);
			if (ECONSTRAINT(ret)->handle == NOCONSTRAINT) return(Null); else return(ret);
	}
	return(Null);
}

/************************* DATABASE EXAMINATION ROUTINES *************************/

Object lsp_curlib(void)
{
	return(Make_EElectric((INTBIG)el_curlib, (INTSML)T_ELibrary));
}

Object lsp_curtech(void)
{
	return(Make_EElectric((INTBIG)el_curtech, (INTSML)T_ETechnology));
}

Object lsp_getval(Object oaddr, Object oattr)
{
	INTBIG type, addr, len, i;
	char *name;
	Object v;
	VARIABLE *var;
	GC_Node;

	/* get inputs from LISP */
	lsp_getaddrandtype(oaddr, &addr, &type);
	if (type == VUNKNOWN) return(Null);
	name = lsp_getstringobject(oattr);
	if (name == 0) return(Null);

	/* get the variable */
	var = getval(addr, type, -1, name);
	if (var == NOVARIABLE) return(Null);
	if ((var->type&VISARRAY) == 0)
		return(lsp_makevarobject(var->type, var->addr));
	len = getlength(var);
	v = Make_Vector(len, Null);
	GC_Link(v);
	for(i=0; i<len; i++)
		VECTOR(v)->data[i] = lsp_makevarobject(var->type, ((INTBIG *)var->addr)[i]);
	GC_Unlink;
	return(v);
}

Object lsp_setval(Object oaddr, Object oname, Object onaddr, Object ontypebits)
{
	INTBIG type, addr, len, i, ntype, naddr, ntypebits, lasttype, thisaddr, ifloat;
	char *name;
	VARIABLE *var;
	float f;

	/* get inputs from LISP */
	lsp_getaddrandtype(oaddr, &addr, &type);
	if (type == VUNKNOWN) return(Null);
	name = lsp_getstringobject(oname);
	if (name == 0) return(Null);
	if (TYPE(onaddr) == T_Vector)
	{
		/* setting an array */
		len = VECTOR(onaddr)->size;
		naddr = (INTBIG)emalloc(len * SIZEOFINTBIG, el_tempcluster);
		if (naddr == 0) return(Null);
		for(i=0; i<len; i++)
		{
			lsp_getaddrandtype(VECTOR(onaddr)->data[i], &thisaddr, &ntype);
			if (ntype == VSTRING)
				(void)allocstring(&((char **)naddr)[i], (char *)thisaddr, el_tempcluster); else
					((INTBIG *)naddr)[i] = thisaddr;
			if (i != 0 && lasttype == VINTEGER && ntype == VFLOAT)
			{
				f = castfloat(thisaddr);
				ifloat = (INTBIG)f;
				if (ifloat == (INTBIG)f)
				{
					((INTBIG *)naddr)[i] = ifloat;
					ntype = VINTEGER;
				}
			}
			if (i != 0 && lasttype == VFLOAT && ntype == VINTEGER)
			{
				f = castfloat(((INTBIG *)naddr)[i-1]);
				ifloat = (INTBIG)f;
				if (ifloat == (INTBIG)f)
				{
					((INTBIG *)naddr)[i-1] = ifloat;
					lasttype = VINTEGER;
				}
			}
			if (i != 0 && ntype != lasttype)
			{
				ttyputerr("Inconsistent type in array");
				return(Null);
			}
			lasttype = ntype;
		}
		ntype |= VISARRAY | (len << VLENGTHSH);
	} else
	{
		/* setting a scalar */
		lsp_getaddrandtype(onaddr, &naddr, &ntype);
		if (ntype == VUNKNOWN) return(Null);
	}
	if (lsp_getnumericobject(ontypebits, &ntypebits) != 0) return(Null);
	ntype |= ntypebits;

	/* set the variable */
	var = setval(addr, type, name, naddr, ntype);
	if ((ntype&VISARRAY) != 0)
	{
		if ((ntype&VTYPE) == VSTRING)
			for(i=0; i<len; i++) efree(((char **)naddr)[i]);
		efree((char *)naddr);
	}
	return(Make_Fixnum(var != NOVARIABLE));
}

Object lsp_setind(Object oaddr, Object oname, Object oindex, Object onaddr)
{
	INTBIG type, addr, ntype, naddr, index;
	char *name;

	/* get inputs from LISP */
	lsp_getaddrandtype(oaddr, &addr, &type);
	if (type == VUNKNOWN) return(Null);
	name = lsp_getstringobject(oname);
	if (name == 0) return(Null);
	if (lsp_getnumericobject(oindex, &index) != 0) return(Null);
	lsp_getaddrandtype(onaddr, &naddr, &ntype);
	if (ntype == VUNKNOWN) return(Null);

	/* set the variable */
	return(Make_Fixnum(setind(addr, type, name, index, naddr)));
}

Object lsp_delval(Object oaddr, Object oname)
{
	INTBIG type, addr;
	char *name;

	/* get inputs from LISP */
	lsp_getaddrandtype(oaddr, &addr, &type);
	if (type == VUNKNOWN) return(Null);
	name = lsp_getstringobject(oname);
	if (name == 0) return(Null);

	/* delete the variable */
	return(Make_Fixnum(delval(addr, type, name)));
}

Object lsp_initsearch(Object olx, Object ohx, Object oly, Object ohy, Object onp)
{
	INTBIG lx, hx, ly, hy, sea;
	REGISTER NODEPROTO *np;

	/* get inputs from LISP */
	if (lsp_getnumericobject(olx, &lx) != 0) return(Null);
	if (lsp_getnumericobject(ohx, &hx) != 0) return(Null);
	if (lsp_getnumericobject(oly, &ly) != 0) return(Null);
	if (lsp_getnumericobject(ohy, &hy) != 0) return(Null);
	Ensure_Type(onp, T_ENodeProto);   np = ENODEPROTO(onp)->handle;

	sea = initsearch(lx, hx, ly, hy, np);
	if (sea == -1) return(Null);
	return(Make_Integer(sea));
}

Object lsp_nextobject(Object osea)
{
	INTBIG sea;
	REGISTER GEOM *g;

	/* get inputs from LISP */
	if (lsp_getnumericobject(osea, &sea) != 0) return(Null);

	g = nextobject(sea);
	if (g == NOGEOM) return(Null);
	return(Make_EElectric((INTBIG)g, (INTSML)T_EGeom));
}

/****************************** AID ROUTINES ******************************/

Object lsp_getaid(Object oname)
{
	char *name;
	REGISTER AIDENTRY *aid;

	/* get inputs from LISP */
	name = lsp_getstringobject(oname);
	if (name == 0) return(Null);

	aid = getaid(name);
	if (aid == NOAID) return(Null);
	return(Make_EElectric((INTBIG)aid, (INTSML)T_EAid));
}

Object lsp_maxaid(void)
{
	return(Make_Fixnum(el_maxaid));
}

Object lsp_indexaid(Object oindex)
{
	INTBIG index;

	/* get inputs from LISP */
	if (lsp_getnumericobject(oindex, &index) != 0) return(Null);

	if (index < 0 || index >= el_maxaid) return(Null);
	return(Make_EElectric((INTBIG)&el_aids[index], (INTSML)T_EAid));
}

Object lsp_aidturnon(Object oaid, Object onocatchup)
{
	REGISTER AIDENTRY *aid;
	INTBIG nocatchup;

	/* get inputs from LISP */
	Ensure_Type(oaid, T_EAid);   aid = EAID(oaid)->handle;
	if (lsp_getnumericobject(onocatchup, &nocatchup) != 0) return(Null);

	aidturnon(aid, (INTSML)nocatchup);
	return(Null);
}

Object lsp_aidturnoff(Object oaid)
{
	REGISTER AIDENTRY *aid;

	/* get inputs from LISP */
	Ensure_Type(oaid, T_EAid);   aid = EAID(oaid)->handle;

	aidturnoff(aid, 1);
	return(Null);
}

/****************************** LIBRARY ROUTINES ******************************/

Object lsp_getlibrary(Object oname)
{
	char *name;
	REGISTER LIBRARY *lib;

	/* get inputs from LISP */
	name = lsp_getstringobject(oname);
	if (name == 0) return(Null);

	lib = getlibrary(name);
	if (lib == NOLIBRARY) return(Null);
	return(Make_EElectric((INTBIG)lib, (INTSML)T_ELibrary));
}

Object lsp_newlibrary(Object oname, Object ofile)
{
	char *name, *file;
	REGISTER LIBRARY *lib;

	/* get inputs from LISP */
	name = lsp_getstringobject(oname);
	if (name == 0) return(Null);
	file = lsp_getstringobject(ofile);
	if (file == 0) return(Null);

	lib = newlibrary(name, file);
	if (lib == NOLIBRARY) return(Null);
	return(Make_EElectric((INTBIG)lib, (INTSML)T_ELibrary));
}

Object lsp_killlibrary(Object olib)
{
	REGISTER LIBRARY *lib;

	/* get inputs from LISP */
	Ensure_Type(olib, T_ELibrary);   lib = ELIBRARY(olib)->handle;

	killlibrary(lib);
	return(Null);
}

Object lsp_eraselibrary(Object olib)
{
	REGISTER LIBRARY *lib;

	/* get inputs from LISP */
	Ensure_Type(olib, T_ELibrary);   lib = ELIBRARY(olib)->handle;

	eraselibrary(lib);
	return(Null);
}

Object lsp_selectlibrary(Object olib)
{
	REGISTER LIBRARY *lib;

	/* get inputs from LISP */
	Ensure_Type(olib, T_ELibrary);   lib = ELIBRARY(olib)->handle;

	selectlibrary(lib);
	return(Null);
}

/****************************** NODEPROTO ROUTINES ******************************/

Object lsp_getnodeproto(Object oname)
{
	char *name;
	REGISTER NODEPROTO *np;

	/* get inputs from LISP */
	name = lsp_getstringobject(oname);
	if (name == 0) return(Null);

	np = getnodeproto(name);
	if (np == NONODEPROTO) return(Null);
	return(Make_EElectric((INTBIG)np, (INTSML)T_ENodeProto));
}

Object lsp_newnodeproto(Object oname, Object olib)
{
	char *name;
	REGISTER LIBRARY *lib;
	REGISTER NODEPROTO *np;

	/* get inputs from LISP */
	Ensure_Type(olib, T_ELibrary);   lib = ELIBRARY(olib)->handle;
	name = lsp_getstringobject(oname);
	if (name == 0) return(Null);

	np = newnodeproto(name, lib);
	if (np == NONODEPROTO) return(Null);
	return(Make_EElectric((INTBIG)np, (INTSML)T_ENodeProto));
}

Object lsp_killnodeproto(Object onp)
{
	REGISTER NODEPROTO *np;

	/* get inputs from LISP */
	Ensure_Type(onp, T_ENodeProto);   np = ENODEPROTO(onp)->handle;

	return(Make_Fixnum(killnodeproto(np)));
}

Object lsp_copynodeproto(Object onp, Object otlib, Object otname)
{
	char *tname;
	REGISTER LIBRARY *tlib;
	REGISTER NODEPROTO *np, *nnp;

	/* get inputs from LISP */
	Ensure_Type(otlib, T_ELibrary);   tlib = ELIBRARY(otlib)->handle;
	Ensure_Type(onp, T_ENodeProto);   np = ENODEPROTO(onp)->handle;
	tname = lsp_getstringobject(otname);
	if (tname == 0) return(Null);

	nnp = copynodeproto(np, tlib, tname);
	if (nnp == NONODEPROTO) return(Null);
	return(Make_EElectric((INTBIG)nnp, (INTSML)T_ENodeProto));
}

Object lsp_iconview(Object onp)
{
	REGISTER NODEPROTO *np, *inp;

	/* get inputs from LISP */
	Ensure_Type(onp, T_ENodeProto);   np = ENODEPROTO(onp)->handle;

	inp = iconview(np);
	if (inp == NONODEPROTO) return(Null);
	return(Make_EElectric((INTBIG)inp, (INTSML)T_ENodeProto));
}

Object lsp_contentsview(Object onp)
{
	REGISTER NODEPROTO *np, *cnp;

	/* get inputs from LISP */
	Ensure_Type(onp, T_ENodeProto);   np = ENODEPROTO(onp)->handle;

	cnp = contentsview(np);
	if (cnp == NONODEPROTO) return(Null);
	return(Make_EElectric((INTBIG)cnp, (INTSML)T_ENodeProto));
}

Object lsp_gettraversalpath(void)
{
	NODEINST **nilist;
	INTSML depth;
	REGISTER INTSML i;
	Object v;
	GC_Node;

	/* call Electric */
	gettraversalpath(&nilist, &depth);

	v = Make_Vector(depth, Null);
	GC_Link(v);
	for(i=0; i<depth; i++)
		VECTOR(v)->data[i] = lsp_makevarobject(VNODEINST, (INTBIG)nilist[i]);
	GC_Unlink;
	return(v);
}

/****************************** NODEINST ROUTINES ******************************/

Object lsp_newnodeinst(Object opro, Object olx, Object ohx, Object oly, Object ohy,
	Object otr, Object orot, Object onp)
{
	REGISTER NODEPROTO *pro, *np;
	INTBIG lx, hx, ly, hy, tr, rot;
	REGISTER NODEINST *ni;

	/* get inputs from LISP */
	Ensure_Type(opro, T_ENodeProto);   pro = ENODEPROTO(opro)->handle;
	Ensure_Type(onp, T_ENodeProto);    np = ENODEPROTO(onp)->handle;
	if (lsp_getnumericobject(olx, &lx) != 0) return(Null);
	if (lsp_getnumericobject(ohx, &hx) != 0) return(Null);
	if (lsp_getnumericobject(oly, &ly) != 0) return(Null);
	if (lsp_getnumericobject(ohy, &hy) != 0) return(Null);
	if (lsp_getnumericobject(otr, &tr) != 0) return(Null);
	if (lsp_getnumericobject(orot, &rot) != 0) return(Null);

	ni = newnodeinst(pro, lx, hx, ly, hy, (INTSML)tr, (INTSML)rot, np);
	if (ni == NONODEINST) return(Null);
	return(Make_EElectric((INTBIG)ni, (INTSML)T_ENodeInst));
}

Object lsp_modifynodeinst(Object oni, Object odlx, Object odly, Object odhx, Object odhy,
	Object odrot, Object odtr)
{
	INTBIG dlx, dly, dhx, dhy, drot, dtr;
	REGISTER NODEINST *ni;

	/* get inputs from LISP */
	Ensure_Type(oni, T_ENodeInst);   ni = ENODEINST(oni)->handle;
	if (lsp_getnumericobject(odlx, &dlx) != 0) return(Null);
	if (lsp_getnumericobject(odly, &dly) != 0) return(Null);
	if (lsp_getnumericobject(odhx, &dhx) != 0) return(Null);
	if (lsp_getnumericobject(odhy, &dhy) != 0) return(Null);
	if (lsp_getnumericobject(odrot, &drot) != 0) return(Null);
	if (lsp_getnumericobject(odtr, &dtr) != 0) return(Null);

	modifynodeinst(ni, dlx, dly, dhx, dhy, (INTSML)drot, (INTSML)dtr);
	return(Null);
}

Object lsp_killnodeinst(Object oni)
{
	REGISTER NODEINST *ni;

	/* get inputs from LISP */
	Ensure_Type(oni, T_ENodeInst);   ni = ENODEINST(oni)->handle;

	return(Make_Fixnum(killnodeinst(ni)));
}

Object lsp_replacenodeinst(Object oni, Object opr)
{
	REGISTER NODEINST *ni, *nni;
	REGISTER NODEPROTO *pr;

	/* get inputs from LISP */
	Ensure_Type(oni, T_ENodeInst);   ni = ENODEINST(oni)->handle;
	Ensure_Type(opr, T_ENodeProto);  pr = ENODEPROTO(opr)->handle;

	nni = replacenodeinst(ni, pr);
	if (nni == NONODEINST) return(Null);
	return(Make_EElectric((INTBIG)nni, (INTSML)T_ENodeInst));
}

Object lsp_nodefunction(Object oni)
{
	char *dummy;
	REGISTER NODEINST *ni;

	/* get inputs from LISP */
	Ensure_Type(oni, T_ENodeInst);   ni = ENODEINST(oni)->handle;

	return(Make_Integer(nodefunction(ni, &dummy)));
}

/****************************** ARCINST ROUTINES ******************************/

Object lsp_newarcinst(Object opro, Object owid, Object obit, Object ona, Object opa,
	Object oxa, Object oya, Object onb, Object opb, Object oxb, Object oyb, Object onp)
{
	REGISTER ARCPROTO *pro;
	REGISTER NODEPROTO *np;
	REGISTER NODEINST *na, *nb;
	REGISTER PORTPROTO *pa, *pb;
	INTBIG wid, bit, xa, ya, xb, yb;
	REGISTER ARCINST *ai;

	/* get inputs from LISP */
	Ensure_Type(opro, T_EArcProto);   pro = EARCPROTO(opro)->handle;
	if (lsp_getnumericobject(owid, &wid) != 0) return(Null);
	if (lsp_getnumericobject(obit, &bit) != 0) return(Null);
	Ensure_Type(ona, T_ENodeInst);    na = ENODEINST(ona)->handle;
	Ensure_Type(opa, T_EPortProto);   pa = EPORTPROTO(opa)->handle;
	if (lsp_getnumericobject(oxa, &xa) != 0) return(Null);
	if (lsp_getnumericobject(oya, &ya) != 0) return(Null);
	Ensure_Type(onb, T_ENodeInst);    nb = ENODEINST(onb)->handle;
	Ensure_Type(opb, T_EPortProto);   pb = EPORTPROTO(opb)->handle;
	if (lsp_getnumericobject(oxb, &xb) != 0) return(Null);
	if (lsp_getnumericobject(oyb, &yb) != 0) return(Null);
	Ensure_Type(onp, T_ENodeProto);   np = ENODEPROTO(onp)->handle;

	ai = newarcinst(pro, wid, bit, na, pa, xa, ya, nb, pb, xb, yb, np);
	if (ai == NOARCINST) return(Null);
	return(Make_EElectric((INTBIG)ai, (INTSML)T_EArcInst));
}

Object lsp_modifyarcinst(Object oai, Object odw, Object odx1, Object ody1, Object odx2,
	Object ody2)
{
	INTBIG dw, dx1, dy1, dx2, dy2;
	REGISTER ARCINST *ai;

	/* get inputs from LISP */
	Ensure_Type(oai, T_EArcInst);   ai = EARCINST(oai)->handle;
	if (lsp_getnumericobject(odw, &dw) != 0) return(Null);
	if (lsp_getnumericobject(odx1, &dx1) != 0) return(Null);
	if (lsp_getnumericobject(ody1, &dy1) != 0) return(Null);
	if (lsp_getnumericobject(odx2, &dx2) != 0) return(Null);
	if (lsp_getnumericobject(ody2, &dy2) != 0) return(Null);

	return(Make_Fixnum(modifyarcinst(ai, dw, dx1, dy1, dx2, dy2)));
}

Object lsp_killarcinst(Object oai)
{
	REGISTER ARCINST *ai;

	/* get inputs from LISP */
	Ensure_Type(oai, T_EArcInst);   ai = EARCINST(oai)->handle;

	return(Make_Fixnum(killarcinst(ai)));
}

Object lsp_replacearcinst(Object oai, Object opr)
{
	REGISTER ARCINST *ai, *nai;
	REGISTER ARCPROTO *pr;

	/* get inputs from LISP */
	Ensure_Type(oai, T_EArcInst);    ai = EARCINST(oai)->handle;
	Ensure_Type(opr, T_EArcProto);   pr = EARCPROTO(opr)->handle;

	nai = replacearcinst(ai, pr);
	if (nai == NOARCINST) return(Null);
	return(Make_EElectric((INTBIG)nai, (INTSML)T_EArcInst));
}

/****************************** PORTPROTO ROUTINES ******************************/

Object lsp_newportproto(Object onp, Object oni, Object opp, Object oname)
{
	char *name;
	REGISTER NODEPROTO *np;
	REGISTER NODEINST *ni;
	REGISTER PORTPROTO *pp;

	/* get inputs from LISP */
	Ensure_Type(onp, T_ENodeProto);   np = ENODEPROTO(onp)->handle;
	Ensure_Type(oni, T_ENodeInst);    ni = ENODEINST(oni)->handle;
	Ensure_Type(opp, T_EPortProto);   pp = EPORTPROTO(opp)->handle;
	name = lsp_getstringobject(oname);
	if (name == 0) return(Null);

	pp = newportproto(np, ni, pp, name);
	if (pp == NOPORTPROTO) return(Null);
	return(Make_EElectric((INTBIG)pp, (INTSML)T_EPortProto));
}

Object lsp_portposition(Object oni, Object opp)
{
	REGISTER NODEINST *ni;
	REGISTER PORTPROTO *pp;
	Object v;
	INTBIG x, y;
	GC_Node;

	/* get inputs from LISP */
	Ensure_Type(oni, T_ENodeInst);    ni = ENODEINST(oni)->handle;
	Ensure_Type(opp, T_EPortProto);   pp = EPORTPROTO(opp)->handle;

	portposition(ni, pp, &x, &y);
	v = Make_Vector(2, Null);
	GC_Link(v);
	VECTOR(v)->data[0] = Make_Integer(x);
	VECTOR(v)->data[1] = Make_Integer(y);
	GC_Unlink;
	return(v);
}

Object lsp_getportproto(Object onp, Object oname)
{
	char *name;
	REGISTER NODEPROTO *np;
	REGISTER PORTPROTO *pp;

	/* get inputs from LISP */
	Ensure_Type(onp, T_ENodeProto);   np = ENODEPROTO(onp)->handle;
	name = lsp_getstringobject(oname);
	if (name == 0) return(Null);

	pp = getportproto(np, name);
	if (pp == NOPORTPROTO) return(Null);
	return(Make_EElectric((INTBIG)pp, (INTSML)T_EPortProto));
}

Object lsp_killportproto(Object onp, Object opp)
{
	REGISTER PORTPROTO *pp;
	REGISTER NODEPROTO *np;

	/* get inputs from LISP */
	Ensure_Type(opp, T_EPortProto);   pp = EPORTPROTO(opp)->handle;
	Ensure_Type(onp, T_ENodeProto);   np = ENODEPROTO(onp)->handle;

	return(Make_Fixnum(killportproto(np, pp)));
}

Object lsp_moveportproto(Object onp, Object oopp, Object onni, Object onpp)
{
	REGISTER NODEPROTO *np;
	REGISTER PORTPROTO *opp, *npp;
	REGISTER NODEINST *nni;

	/* get inputs from LISP */
	Ensure_Type(onp, T_ENodeProto);    np = ENODEPROTO(onp)->handle;
	Ensure_Type(oopp, T_EPortProto);   opp = EPORTPROTO(oopp)->handle;
	Ensure_Type(onni, T_ENodeInst);    nni = ENODEINST(onni)->handle;
	Ensure_Type(onpp, T_EPortProto);   npp = EPORTPROTO(onpp)->handle;

	return(Make_Fixnum(moveportproto(np, opp, nni, npp)));
}

/*************************** CHANGE CONTROL ROUTINES ***************************/

Object lsp_undoabatch(void)
{
	INTBIG aid;

	if (undoabatch((AIDENTRY **)&aid) == 0) return(Null);
	return(Make_Fixnum(aid));
}

Object lsp_noundoallowed(void)
{
	noundoallowed();
	return(Null);
}

/****************************** VIEW ROUTINES ******************************/

Object lsp_getview(Object oname)
{
	char *name;
	REGISTER VIEW *v;

	/* get inputs from LISP */
	name = lsp_getstringobject(oname);
	if (name == 0) return(Null);

	v = getview(name);
	if (v == NOVIEW) return(Null);
	return(Make_EElectric((INTBIG)v, (INTSML)T_EView));
}

Object lsp_newview(Object oname, Object osname)
{
	char *name, *sname;
	REGISTER VIEW *v;

	/* get inputs from LISP */
	name = lsp_getstringobject(oname);
	if (name == 0) return(Null);
	sname = lsp_getstringobject(osname);
	if (sname == 0) return(Null);

	v = newview(name, sname);
	if (v == NOVIEW) return(Null);
	return(Make_EElectric((INTBIG)v, (INTSML)T_EView));
}

Object lsp_killview(Object ov)
{
	REGISTER VIEW *v;

	/* get inputs from LISP */
	Ensure_Type(ov, T_EView);   v = EVIEW(ov)->handle;

	return(Make_Fixnum(killview(v)));
}

/*************************** MISCELLANEOUS ROUTINES ***************************/

Object lsp_changelambda(Object ooldl, Object onewl, Object otech)
{
	REGISTER TECHNOLOGY *tech;
	INTBIG oldl, newl;

	/* get inputs from LISP */
	if (Nullp(otech)) tech = NOTECHNOLOGY; else
	{
		Ensure_Type(otech, T_ETechnology);   tech = ETECHNOLOGY(otech)->handle;
	}
	if (lsp_getnumericobject(ooldl, &oldl) != 0) return(Null);
	if (lsp_getnumericobject(onewl, &newl) != 0) return(Null);

	changelambda(oldl, newl, tech);
	return(Null);
}

Object lsp_tellaid(long argc, Object argv[])
{
	REGISTER AIDENTRY *aid;
	REGISTER INTSML i, val;
	char *par[20], *ret;

	/* get inputs from LISP */
	if (argc < 1) return(Null);
	Ensure_Type(argv[0], T_EAid);   aid = EAID(argv[0])->handle;
	argc--;   argv++;
	for(i=0; i<argc; i++)
	{
		ret = lsp_getstringobject(argv[i]);
		if (ret == 0) return(Null);
		(void)allocstring(&par[i], ret, el_tempcluster);
	}

	val = tellaid(aid, (INTSML)argc, par);
	for(i=0; i<argc; i++) efree(par[i]);
	return(Make_Fixnum(val));
}

Object lsp_getarcproto(Object oname)
{
	char *name;
	REGISTER ARCPROTO *ap;

	/* get inputs from LISP */
	name = lsp_getstringobject(oname);
	if (name == 0) return(Null);

	ap = getarcproto(name);
	if (ap == NOARCPROTO) return(Null);
	return(Make_EElectric((INTBIG)ap, (INTSML)T_EArcProto));
}

Object lsp_getcell(Object oname)
{
	char *name;
	REGISTER CELL *c;

	/* get inputs from LISP */
	name = lsp_getstringobject(oname);
	if (name == 0) return(Null);

	c = getcell(name);
	if (c == NOCELL) return(Null);
	return(Make_EElectric((INTBIG)c, (INTSML)T_ECell));
}

Object lsp_gettechnology(Object oname)
{
	char *name;
	REGISTER TECHNOLOGY *tech;

	/* get inputs from LISP */
	name = lsp_getstringobject(oname);
	if (name == 0) return(Null);

	tech = gettechnology(name);
	if (tech == NOTECHNOLOGY) return(Null);
	return(Make_EElectric((INTBIG)tech, (INTSML)T_ETechnology));
}

Object lsp_getpinproto(Object oap)
{
	REGISTER ARCPROTO *ap;
	REGISTER NODEPROTO *np;

	/* get inputs from LISP */
	Ensure_Type(oap, T_EArcProto);   ap = EARCPROTO(oap)->handle;

	np = getpinproto(ap);
	if (np == NONODEPROTO) return(Null);
	return(Make_EElectric((INTBIG)np, (INTSML)T_ENodeProto));
}

Object lsp_getnetwork(Object onp, Object oname)
{
	char *name;
	REGISTER NETWORK *net;
	REGISTER NODEPROTO *np;

	/* get inputs from LISP */
	Ensure_Type(onp, T_ENodeProto);   np = ENODEPROTO(onp)->handle;
	name = lsp_getstringobject(oname);
	if (name == 0) return(Null);

	net = getnetwork(name, np);
	if (net == NONETWORK) return(Null);
	return(Make_EElectric((INTBIG)net, (INTSML)T_ENetwork));
}

/****************************** INITIALIZATION ******************************/

void init_lib_electric(void)
{
	T_ENodeInst = Define_Type(0, "nodeinst", NOFUNC, sizeof (struct S_ENodeInst),
		EElectric_Equal, EElectric_Equal, ENodeInst_Print, NOFUNC);
	T_ENodeProto = Define_Type(0, "nodeproto", NOFUNC, sizeof (struct S_ENodeProto),
		EElectric_Equal, EElectric_Equal, ENodeProto_Print, NOFUNC);
	T_EPortArcInst = Define_Type(0, "portarcinst", NOFUNC, sizeof (struct S_EPortArcInst),
		EElectric_Equal, EElectric_Equal, EPortArcInst_Print, NOFUNC);
	T_EPortExpInst = Define_Type(0, "portexpinst", NOFUNC, sizeof (struct S_EPortExpInst),
		EElectric_Equal, EElectric_Equal, EPortExpInst_Print, NOFUNC);
	T_EPortProto = Define_Type(0, "portproto", NOFUNC, sizeof (struct S_EPortProto),
		EElectric_Equal, EElectric_Equal, EPortProto_Print, NOFUNC);
	T_EArcInst = Define_Type(0, "arcinst", NOFUNC, sizeof (struct S_EArcInst),
		EElectric_Equal, EElectric_Equal, EArcInst_Print, NOFUNC);
	T_EArcProto = Define_Type(0, "arcproto", NOFUNC, sizeof (struct S_EArcProto),
		EElectric_Equal, EElectric_Equal, EArcProto_Print, NOFUNC);
	T_EGeom = Define_Type(0, "geom", NOFUNC, sizeof (struct S_EGeom),
		EElectric_Equal, EElectric_Equal, EGeom_Print, NOFUNC);
	T_ELibrary = Define_Type(0, "library", NOFUNC, sizeof (struct S_ELibrary),
		EElectric_Equal, EElectric_Equal, ELibrary_Print, NOFUNC);
	T_ETechnology = Define_Type(0, "technology", NOFUNC, sizeof (struct S_ETechnology),
		EElectric_Equal, EElectric_Equal, ETechnology_Print, NOFUNC);
	T_EAid = Define_Type(0, "aid", NOFUNC, sizeof (struct S_EAid),
		EElectric_Equal, EElectric_Equal, EAid_Print, NOFUNC);
	T_ERTNode = Define_Type(0, "rtnode", NOFUNC, sizeof (struct S_ERTNode),
		EElectric_Equal, EElectric_Equal, ERTNode_Print, NOFUNC);
	T_ENetwork = Define_Type(0, "network", NOFUNC, sizeof (struct S_ENetwork),
		EElectric_Equal, EElectric_Equal, ENetwork_Print, NOFUNC);
	T_ECell = Define_Type(0, "cell", NOFUNC, sizeof (struct S_ECell),
		EElectric_Equal, EElectric_Equal, ECell_Print, NOFUNC);
	T_EView = Define_Type(0, "view", NOFUNC, sizeof (struct S_EView),
		EElectric_Equal, EElectric_Equal, EView_Print, NOFUNC);
	T_EWindow = Define_Type(0, "window", NOFUNC, sizeof (struct S_EWindow),
		EElectric_Equal, EElectric_Equal, EWindow_Print, NOFUNC);
	T_EGraphics = Define_Type(0, "graphics", NOFUNC, sizeof (struct S_EGraphics),
		EElectric_Equal, EElectric_Equal, EGraphics_Print, NOFUNC);
	T_EConstraint = Define_Type(0, "constraint", NOFUNC, sizeof (struct S_EConstraint),
		EElectric_Equal, EElectric_Equal, EConstraint_Print, NOFUNC);

	/* define the query predicates */
	Define_Primitive((Object(*)(ELLIPSIS))P_ENodeInstP, "nodeinst?", 1, 1, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))P_ENodeProtoP, "nodeproto?", 1, 1, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))P_EPortArcInstP, "portarcinst?", 1, 1, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))P_EPortExpInstP, "portexpinst?", 1, 1, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))P_EPortProtoP, "portproto?", 1, 1, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))P_EArcInstP, "arcinst?", 1, 1, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))P_EArcProtoP, "arcproto?", 1, 1, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))P_EGeomP, "geom?", 1, 1, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))P_ELibraryP, "library?", 1, 1, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))P_ETechnologyP, "technology?", 1, 1, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))P_EAidP, "aid?", 1, 1, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))P_ERTNodeP, "rtnode?", 1, 1, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))P_ENetworkP, "network?", 1, 1, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))P_ECellP, "cell?", 1, 1, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))P_EViewP, "view?", 1, 1, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))P_EWindowP, "window?", 1, 1, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))P_EGraphicsP, "graphics?", 1, 1, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))P_EConstraintP, "constraint?", 1, 1, EVAL);

	/* define symbols */
	Define_Symbol(&lsp_displayablesym, "displayable");

	/* define the database examination predicates */
	Define_Primitive(lsp_curlib, "curlib", 0, 0, EVAL);
	Define_Primitive(lsp_curtech, "curtech", 0, 0, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))lsp_getval, "getval", 2, 2, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))lsp_setval, "setval", 4, 4, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))lsp_setind, "setind", 4, 4, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))lsp_delval, "delval", 2, 2, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))lsp_initsearch, "initsearch", 5, 5, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))lsp_nextobject, "nextobject", 1, 1, EVAL);

	/* define the aid predicates */
	Define_Primitive((Object(*)(ELLIPSIS))lsp_getaid, "getaid", 1, 1, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))lsp_maxaid, "maxaid", 0, 0, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))lsp_indexaid, "indexaid", 1, 1, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))lsp_aidturnon, "aidturnon", 2, 2, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))lsp_aidturnoff, "aidturnoff", 1, 1, EVAL);

	/* define the library predicates */
	Define_Primitive((Object(*)(ELLIPSIS))lsp_getlibrary, "getlibrary", 1, 1, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))lsp_newlibrary, "newlibrary", 2, 2, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))lsp_killlibrary, "killlibrary", 1, 1, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))lsp_eraselibrary, "eraselibrary", 1, 1, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))lsp_selectlibrary, "selectlibrary", 1, 1, EVAL);

	/* define the nodeproto predicates */
	Define_Primitive((Object(*)(ELLIPSIS))lsp_getnodeproto, "getnodeproto", 1, 1, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))lsp_newnodeproto, "newnodeproto", 2, 2, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))lsp_killnodeproto, "killnodeproto", 1, 1, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))lsp_copynodeproto, "copynodeproto", 3, 3, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))lsp_iconview, "iconview", 1, 1, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))lsp_contentsview, "contentsview", 1, 1, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))lsp_gettraversalpath, "gettraversalpath", 0, 0, EVAL);

	/* define the nodeinst predicates */
	Define_Primitive((Object(*)(ELLIPSIS))lsp_newnodeinst, "newnodeinst", 8, 8, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))lsp_modifynodeinst, "modifynodeinst", 7, 7, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))lsp_killnodeinst, "killnodeinst", 1, 1, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))lsp_replacenodeinst, "replacenodeinst", 2, 2, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))lsp_nodefunction, "nodefunction", 1, 1, EVAL);

	/* define the arcinst predicates */
	Define_Primitive((Object(*)(ELLIPSIS))lsp_newarcinst, "newarcinst", 12, 12, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))lsp_modifyarcinst, "modifyarcinst", 6, 6, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))lsp_killarcinst, "killarcinst", 1, 1, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))lsp_replacearcinst, "replacearcinst", 2, 2, EVAL);

	/* define the portproto predicates */
	Define_Primitive((Object(*)(ELLIPSIS))lsp_getportproto, "getportproto", 2, 2, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))lsp_newportproto, "newportproto", 4, 4, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))lsp_portposition, "portposition", 2, 2, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))lsp_killportproto, "killportproto", 2, 2, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))lsp_moveportproto, "moveportproto", 4, 4, EVAL);

	/* define the change control predicates */
	Define_Primitive((Object(*)(ELLIPSIS))lsp_undoabatch, "undoabatch", 0, 0, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))lsp_noundoallowed, "noundoallowed", 0, 0, EVAL);

	/* define the view predicates */
	Define_Primitive((Object(*)(ELLIPSIS))lsp_getview, "getview", 1, 1, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))lsp_newview, "newview", 2, 2, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))lsp_killview, "killview", 1, 1, EVAL);

	/* define the miscellaneous predicates */
	Define_Primitive((Object(*)(ELLIPSIS))lsp_changelambda, "changelambda", 3, 3, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))lsp_tellaid, "tellaid", 1, MANY, VARARGS);
	Define_Primitive((Object(*)(ELLIPSIS))lsp_getarcproto, "getarcproto", 1, 1, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))lsp_getcell, "getcell", 1, 1, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))lsp_gettechnology, "gettechnology", 1, 1, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))lsp_getpinproto, "getpinproto", 1, 1, EVAL);
	Define_Primitive((Object(*)(ELLIPSIS))lsp_getnetwork, "getnetwork", 2, 2, EVAL);
}

#endif
