/*
 * Electric(tm) VLSI Design System
 *
 * File: dbmerge.c
 * Box merging subsystem
 * Written by Philip Attfield, Queens University, Kingston Ontario.
 * Revised by Sid Penstone, Queens University, Kingston Ontario.
 *
 * Copyright (c) 1998 Electric Editor Incorporated.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Electric Editor Incorporated
 * 23470 Sunset Drive, Suite 108
 * Los Gatos, California 95033
 * support@electriceditor.com
 */

/***************************************************************************
 * Initially, call:
 *    mrginit()
 *
 * For every polygon, call:
 *    mrgstorebox(layer, tech, xsize, ysize, xcenter, ycenter)
 * where "layer" is an integer layer number from technology "tech"; "xsize"
 * and "ysize" are size of the box; "xcenter" and "ycenter" are the center
 * of the box
 *
 * At end of facet, call:
 *    mrgdonefacet(setlayer, writepolygon)
 * where "setlayer" is a routine that will be called before each change of
 * layer with two arguments: the integer layer number and the technology,
 * and "writepolygon" is a routine that will be called for every polygon with
 * four parameters: the X coordinate array (integer array); Y coordinate
 * array; the number of coordinates in each array; and the area of the polygon
 * (float).
 *
 * When done with merging, call:
 *    mrgterm()
 ***************************************************************************
 */
#include "global.h"
#include "database.h"

#define MAXNUMPTS 106   /* maximum allowable number of points on a polygon */
static INTBIG maxnumpts = MAXNUMPTS; /* apply default (PJA:920527) */
# define PTMATCH 2      /* point match */

/* polygon direction and box edge identifiers */
#define	UP          0
#define	RIGHT       1
#define	DOWN        2
#define	LEFT        3

POLYLISTHEAD db_mrg_polycoordlist;
POLYCOORD   *db_mrg_free_poly_list;
BOXPOLY     *db_mrg_free_box_list;
BOXLISTHEAD *db_mrg_free_boxhead_list;
BOXLISTHEAD *db_mrg_curr_facet;

/* prototypes for local routines */
INTBIG mrg_set_maxnumpts(INTBIG);
INTSML db_mrg_included(BOXPOLY*, BOXPOLY*);
void db_mrg_remove_inclusion(BOXLISTHEAD*);
void db_mrg_remove_overlap(BOXLISTHEAD*);
BOXPOLY *db_mrg_breakbox(BOXPOLY*, BOXPOLY*, INTSML*);
BOXPOLY *db_mrg_do_break(BOXPOLY*, BOXPOLY*, INTSML*);
void db_mrg_write_lists(void(*)(INTSML, TECHNOLOGY*), void(*)(INTBIG*, INTBIG*, INTSML, float));
POLYCOORD *db_mrg_getpoly(void);
INTSML db_mrg_mergeboxwithpolygon(BOXPOLY*);
INTSML db_mrg_check_intersect(BOXPOLY*, POLYCOORD*, INTSML*);
INTSML db_mrg_check_point(INTBIG[], INTBIG[], INTBIG[], INTBIG[]);
void db_mrg_insert1(INTSML, BOXPOLY*);
void db_mrg_insert2(INTSML, BOXPOLY*);
void db_mrg_insert3(INTSML, BOXPOLY*);
void db_mrg_insert4(INTSML, BOXPOLY*);
void db_mrg_write_polygon(void(*)(INTBIG*, INTBIG*, INTSML, float), float);
INTSML db_mrg_pred(INTSML);
INTSML db_mrg_succ(INTSML);
INTSML db_mrg_nxtside(INTSML);
INTSML db_mrg_prevside(INTSML);
void db_mrg_assign_edge(POLYCOORD*, INTBIG[], INTBIG[]);
void db_mrg_housekeeper(void);
void db_mrg_insert5(INTSML, INTSML, BOXPOLY*);
BOXPOLY *db_mrg_getbox(void);
void db_mrg_trash(BOXPOLY*);
void db_mrg_dump_free_box_list(void);
void db_mrg_assign_box(BOXPOLY*, INTBIG, INTBIG, INTBIG, INTBIG);
void db_mrg_killpoly(POLYCOORD*);
void db_mrg_dump_free_poly_list(void);
void db_mrg_remove_poly_list(void);
BOXLISTHEAD *db_mrg_get_lay_head(void);
void db_mrg_dump_free_boxhead_list(void);
void db_mrg_remove_boxhead_list(void);

INTBIG mrg_set_maxnumpts(INTBIG npts) /* set maximum number of vertices allowed on polygon (PJA:920527) */
{
	INTBIG cmax;

	if (npts <= 0)
	{ /* nonsense */
		return(0);
	}
	cmax = maxnumpts; /* will return current maximum */
	maxnumpts = npts; /* set as new maximum */
	return(cmax); /* and return the old max */
}

void mrginit(void)
{
}

void mrgterm(void)
{
	db_mrg_dump_free_poly_list();
	db_mrg_dump_free_box_list();
	db_mrg_dump_free_boxhead_list();
}

void mrgdonefacet(void (*setlayer)(INTSML, TECHNOLOGY*),
	void (*writepolygon)(INTBIG*, INTBIG*, INTSML, float))
{
	db_mrg_write_lists(setlayer, writepolygon);
	db_mrg_remove_boxhead_list();
}

void mrgstorebox(INTSML layer, TECHNOLOGY *tech, INTBIG length, INTBIG width,
	INTBIG xc, INTBIG yc)
{
	BOXLISTHEAD *layhead;
	BOXPOLY *creatptr;
	INTBIG left, right, top, bot;

	for(layhead = db_mrg_curr_facet; layhead != NULL; layhead = layhead->nextlayer)
		if (layhead->layer == layer && layhead->tech == tech) break;

	if (layhead == NULL)
	{
		layhead = db_mrg_get_lay_head();
		layhead->nextlayer = db_mrg_curr_facet; /* form link */
		db_mrg_curr_facet = layhead; /* modify facet layer list header */
		layhead->layer = layer;
		layhead->tech = tech;
		layhead->box = NULL;
	}

	/* layhead now points to the correct layer head descriptor */
	creatptr = db_mrg_getbox();
	creatptr->nextbox = layhead->box;
	layhead->box = creatptr; /* insert new descriptor at top of list */
	left = xc - (length/2);
	right = xc + (length/2);
	top = yc + (width/2);
	bot = yc - (width/2);

	/* assign box descriptor values */
	db_mrg_assign_box(creatptr, left, right, top, bot);
}

/************************* INTERNAL ROUTINES *************************/

/*
 * called once at initialization to set globals
 */
void db_mrgdatainit(void)
{
	db_mrg_free_poly_list = NULL;
	db_mrg_free_box_list = NULL;
	db_mrg_free_boxhead_list = NULL;
	db_mrg_curr_facet = NULL;
}

/* checks inclusion */
INTSML db_mrg_included(BOXPOLY *me, BOXPOLY *him)
{
	if (him->left >= me->left && him->right <= me->right &&
		him->top <= me->top && him->bot >= me->bot)
			return(1); /* him is an inclusion */
	return(0); /* him is NOT an inclusion */
}

/* removes inclusions which follow on list */
void db_mrg_remove_inclusion(BOXLISTHEAD *lay_box)
{
	BOXPOLY *me, *him, *oldhim, *garbage;

	me = lay_box->box;
	garbage = NULL;

	if (me->nextbox != NULL) /* if not a singular list */
	{
		for(; me->nextbox != NULL;)
		{
			oldhim = me;
			for(him = me->nextbox; him != NULL;)
			{
				if (db_mrg_included(me, him)) /* if him is an inclusion of me */
				{
					garbage = him;
					oldhim->nextbox = him->nextbox; /* bypass him on list */
					him = him->nextbox;
					garbage->nextbox = NULL;
					db_mrg_trash(garbage);
				} else /* not an inclusion */
				{
					oldhim = him;
					him = him->nextbox;
				}
			} /* end for */

			/* catches case where everyone on list after me was an */
			me = me->nextbox;
			if (me == NULL) break;   /* inclusion */
		} /* end for */
	} /* end if */
}

void db_mrg_remove_overlap(BOXLISTHEAD *layhead)
{
	BOXPOLY *me, *him, *oldme, *garbage, *broken, *temp,
		*endptr;
	INTSML bugflag, rtc;

	oldme = layhead->box;
	me = oldme;
	bugflag  = 0;
	garbage = NULL;

	if (me->nextbox != NULL)
	{
		for(him = me->nextbox; me->nextbox != NULL;)
		{
			if (bugflag)
			{
				oldme = me;
				garbage->nextbox = NULL;
				db_mrg_trash(garbage);
				bugflag = 0;
			}

			/* now check rtc for results */
			broken = db_mrg_breakbox(me,him,&rtc);
			if (rtc == 1) /* if inclusion, remove me from list */
			{
				garbage = me;
				if (me == layhead->box) /* if me is first on list */
				{
					layhead->box = me->nextbox; /* bypass me on list */
					oldme = me;
					bugflag = 1;
				} else /* me is not the first on list */
				{
					oldme->nextbox = me->nextbox;

					/* back me up on list for correct entry on next iter. */
					me = oldme;
					garbage->nextbox = NULL;
					db_mrg_trash(garbage);
				}
			} else if (rtc == 2) /* independent... do nothing */
			{
				;
			} else
			{
				/* rtc = 0, replace me on list with list returned by break */
				garbage = me;
				for(temp=broken; temp!=NULL; )  /*find eol */
				{
					endptr = temp;
					temp = temp->nextbox;
				}
				if (me == layhead->box) /* if me is first on list */
					layhead->box = broken;
						else oldme->nextbox = broken;
				endptr->nextbox = me->nextbox;
				me = endptr;
				garbage->nextbox = NULL;
				db_mrg_trash(garbage);
			}
			oldme=me;
			me=me->nextbox;
			him=me->nextbox;
		} /* end for */
	} /* end if */
}

BOXPOLY *db_mrg_breakbox(BOXPOLY *me, BOXPOLY *him, INTSML *rtc)
{
	BOXPOLY *list, *oldme, *newlist;
	BOXPOLY *garbage, *temp, *endptr, *broken;
	INTSML retc, calrtc, notfinished=1;

	newlist = NULL;
	while (notfinished)
	{
		list = db_mrg_do_break(me, him, &retc);
		if (retc == 2) /* if independent, do again with him->next */
			if (him->nextbox != NULL) him = him->nextbox; else
		{
			/* we are at the end of the him list and have nothing to return */
			if (newlist == NULL)
			{
				*rtc = 2;
				return(NULL);
			}
		} else if (retc == 1)
		{
			/* if an inclusion */
			if (newlist == NULL)
			{
				/* if empty list, let this guy be eaten up */
				*rtc = 1;
				return(NULL);
			} else
			{
				/* if we get here, we are in deep trouble */
			}
		} else /* if we got here, a break occurred..do something with list */
		{
			if (newlist == NULL) /* it has to be anyway */
			{
				newlist = list;
				if (him->nextbox == NULL) /* at end of hims, return code 0 */
				{
					*rtc = 0;
					return(newlist);
				} else /* must recurse across the list */
				{
					oldme = NULL;
					for (me=newlist; me!=NULL; )
					{
						/* do a break */
						broken = db_mrg_breakbox(me, him->nextbox, &calrtc);
						if (calrtc == 2) /* no break done, look at next guy */
						{
							oldme = me;
							me = me->nextbox;
						} else if (calrtc == 1)
						{
							/* me is an inclusion... remove me */
							garbage = me;
							if (me == newlist) /* if me is first on list */
							{
								newlist = me->nextbox;
								me = newlist; /* put me at top of loop again */
								if (me == NULL)
								{
									/* if me was only one left and got eaten */
									*rtc = 1;
									garbage->nextbox = NULL;
									db_mrg_trash(garbage);
									return(NULL);
								}
							} else /* me was not first on list */
							{
								/* oldme still behind me */
								oldme->nextbox = me->nextbox;
								me = me->nextbox;
							}
							garbage->nextbox = NULL;
							db_mrg_trash(garbage);
						} else
						{
							/* calrct=0, add list after newlist or after oldme */
							for(temp=broken;temp!=NULL;) /*get eol */
							{
								endptr=temp;
								temp=temp->nextbox;
							}
							endptr->nextbox = me->nextbox;
							if (me == newlist) /* if me is first on list */
							{
								garbage = me;
								newlist = broken;
								oldme = endptr;
								me = endptr->nextbox;
								garbage->nextbox = NULL;
								db_mrg_trash(garbage);
							} else /* me is not first on list */
							{
								garbage = me;
								oldme->nextbox = broken;
								oldme = endptr;
								me = endptr->nextbox;
								garbage->nextbox = NULL;
								db_mrg_trash(garbage);
							}
						} /* end if calrtc=0 */
					} /* end for */
					*rtc = 0;
					return(newlist);
				}
			}
		}
	} /* end while */
	return(NULL);
}

BOXPOLY *db_mrg_do_break(BOXPOLY *me, BOXPOLY *him, INTSML *rtc)
{
	BOXPOLY *broken, *breakk, *temp;

	/* check for disjointedness */
	if (me->right <= him->left || me->top <= him->bot || me->left >= him->right ||
		me->bot >= him->top)
	{
		*rtc = 2;
		return(NULL);
	}

	/* check for inclusion */
	if (me->top <= him->top && me->left >= him->left && me->bot >= him->bot &&
		me->right <= him->right)
	{
		*rtc = 1;
		return(NULL);
	}

	/* check for identical boxes */
	if (me->top == him->top && me->bot == him->bot && me->left == him->left &&
		me->right == him->right)
	{
		*rtc = 1;
		return(NULL);
	}

	if (me->top > him->top && me->left < him->left && me->bot < him->bot
		&& me->right < him->right && me->right > him->left)
	{
		temp = db_mrg_getbox();
		db_mrg_assign_box(temp,me->left,me->right,me->top,him->top);
		broken = temp;
		temp = db_mrg_getbox();
		broken->nextbox = temp;
		db_mrg_assign_box(temp,me->left,him->left,him->top,him->bot);
		breakk = db_mrg_getbox();
		temp->nextbox = breakk;
		db_mrg_assign_box(breakk,me->left,me->right,him->bot,me->bot);
		*rtc = 0;
		return(broken);
	}

	if (me->bot < him->bot && me->left < him->left && me->right > him->right
		&& me->top > him->bot && me->top < him->top)
	{
		temp = db_mrg_getbox();
		db_mrg_assign_box(temp,me->left,him->left,me->top,me->bot);
		broken = temp;
		temp = db_mrg_getbox();
		broken->nextbox = temp;
		db_mrg_assign_box(temp,him->left,him->right,him->bot,me->bot);
		breakk = db_mrg_getbox();
		temp->nextbox = breakk;
		db_mrg_assign_box(breakk,him->right,me->right,me->top,me->bot);
		*rtc = 0;
		return(broken);
	}

	if (me->right > him->right && me->top > him->top && me->bot < him->bot
		&& me->left < him->right && me->left > him->left)
	{
		temp = db_mrg_getbox();
		db_mrg_assign_box(temp,me->left,me->right,me->top,him->top);
		broken = temp;
		temp = db_mrg_getbox();
		broken->nextbox = temp;
		db_mrg_assign_box(temp,him->right,me->right,him->top,him->bot);
		breakk = db_mrg_getbox();
		temp->nextbox = breakk;
		db_mrg_assign_box(breakk,me->left,me->right,him->bot,me->bot);
		*rtc = 0;
		return(broken);
	}

	if (me->left < him->left && me->right > him->right && me->top > him->top
		&& me->bot > him->bot && me->bot < him->top)
	{
		temp = db_mrg_getbox();
		db_mrg_assign_box(temp,me->left,him->left,me->top,me->bot);
		broken = temp;
		temp = db_mrg_getbox();
		broken->nextbox = temp;
		db_mrg_assign_box(temp,him->left,him->right,me->top,him->top);
		breakk = db_mrg_getbox();
		temp->nextbox = breakk;
		db_mrg_assign_box(breakk,him->right,me->right,me->top,me->bot);
		*rtc = 0;
		return(broken);
	}

	if (him->left <= me->left && him->top >= me->top && him->bot <= me->bot
		&& me->right > him->right && me->left < him->right)
	{
		broken = db_mrg_getbox();
		db_mrg_assign_box(broken,him->right,me->right,me->top,me->bot);
		*rtc = 0;
		return(broken);
	}

	if (me->left < him->left && him->top >= me->top && him->bot <= me->bot
		&& him->right >= me->right && me->right > him->left)
	{
		broken = db_mrg_getbox();
		db_mrg_assign_box(broken,me->left,him->left,me->top,me->bot);
		*rtc = 0;
		return(broken);
	}

	if (him->left <= me->left && him->right >= me->right && him->bot <= me->bot
		&& me->bot < him->top && me->top > him->top)
	{
		broken = db_mrg_getbox();
		db_mrg_assign_box(broken,me->left,me->right,me->top,him->top);
		*rtc = 0;
		return(broken);
	}

	if (him->left <= me->left && him->right >= me->right && him->top >= me->top
		&& me->top > him->bot && me->bot < him->bot)
	{
		broken = db_mrg_getbox();
		db_mrg_assign_box(broken,me->left,me->right,him->bot,me->bot);
		*rtc = 0;
		return(broken);
	}

	if (me->top > him->top && me->left < him->left && me->bot < him->top &&
		me->bot >= him->bot && me->right > him->left && me->right <= him->right)
	{
		temp = db_mrg_getbox();
		broken = temp;
		db_mrg_assign_box(temp,me->left,me->right,me->top,him->top);
		temp = db_mrg_getbox();
		broken->nextbox = temp;
		db_mrg_assign_box(temp,me->left,him->left,him->top,me->bot);
		*rtc = 0;
		return(broken);
	}

	if (me->top > him->top && me->right > him->right && me->left >= him->left
		&& me->left < him->right && me->bot < him->top && me->bot >= him->bot)
	{
		temp = db_mrg_getbox();
		broken = temp;
		db_mrg_assign_box(temp,me->left,me->right,me->top,him->top);
		temp = db_mrg_getbox();
		broken->nextbox = temp;
		db_mrg_assign_box(temp,him->right,me->right,him->top,me->bot);
		*rtc = 0;
		return(broken);
	}

	if (me->right > him->right && me->bot < him->bot && me->left >= him->left
		&& me->left < him->right && me->top > him->bot && me->top <= him->top)
	{
		temp = db_mrg_getbox();
		broken = temp;
		db_mrg_assign_box(temp,him->right,me->right,me->top,him->bot);
		temp = db_mrg_getbox();
		broken->nextbox = temp;
		db_mrg_assign_box(temp,me->left,me->right,him->bot,me->bot);
		*rtc = 0;
		return(broken);
	}

	if (me->left < him->left && me->bot < him->bot && me->top > him->bot &&
		me->top <= him->top && me->right > him->left && me->right <= him->right)
	{
		temp = db_mrg_getbox();
		broken = temp;
		db_mrg_assign_box(temp,me->left,him->left,me->top,him->bot);
		temp = db_mrg_getbox();
		broken->nextbox = temp;
		db_mrg_assign_box(temp,me->left,me->right,him->bot,me->bot);
		*rtc = 0;
		return(broken);
	}

	if (me->left < him->left && me->right > him->right
		&& him->top >= me->top && him->bot <= me->bot)
	{
		temp = db_mrg_getbox();
		broken = temp;
		db_mrg_assign_box(temp,me->left,him->left,me->top,me->bot);
		temp = db_mrg_getbox();
		broken->nextbox = temp;
		db_mrg_assign_box(temp,him->right,me->right,me->top,me->bot);
		*rtc = 0;
		return(broken);
	}

	if (me->top > him->top && me->bot < him->bot
		&& me->left >= him->left && me->right <= him->right)
	{
		temp = db_mrg_getbox();
		broken = temp;
		db_mrg_assign_box(temp,me->left,me->right,me->top,him->top);
		temp = db_mrg_getbox();
		broken->nextbox = temp;
		db_mrg_assign_box(temp,me->left,me->right,him->bot,me->bot);
		*rtc = 0;
		return(broken);
	}

	/* give the user a warning if this routine did not catch some case */
	ttyputerr("WARNING: MERGING DONE INCORRECTLY");
	ttyputerr("me: left = %d, right = %d, top = %d, bot = %d",
		me->left, me->right, me->top, me->bot);
	ttyputerr("him: left = %d, right = %d, top = %d, bot = %d",
		him->left, him->right, him->top, him->bot);

	/* although this is in error, it will enable a completion of the write */
	*rtc = 1;
	return(NULL);
}

void db_mrg_write_lists(void (*setlayer)(INTSML, TECHNOLOGY*),
	void (*writepolygon)(INTBIG*, INTBIG*, INTSML, float))
{
	BOXLISTHEAD *lay_box;
	BOXPOLY *boxscanptr, *garbage, *oldboxscan;
	INTSML lay_not_done, merged;
	float area;         /* SRP920622 */

	for(lay_box=db_mrg_curr_facet; lay_box!=NULL; lay_box=lay_box->nextlayer) /* for each layer... */
	{
		db_mrg_remove_inclusion(lay_box); /* remove inclusions down list */
		db_mrg_remove_overlap(lay_box); /* break into nonoverlapping boxes */
		db_mrg_remove_inclusion(lay_box);
		lay_not_done = 1;
		(*setlayer)(lay_box->layer, lay_box->tech);
		db_mrg_polycoordlist.firstpoly = NULL;
		db_mrg_polycoordlist.numedge = 0;
		area = 0.0;     /* was inside the while()   (SRP) */
		while (lay_not_done)
		{
			db_mrg_polycoordlist.modified = 0;
			for(boxscanptr=lay_box->box; boxscanptr != NULL; )
			{
				/* passing through list of boxes */
				merged = db_mrg_mergeboxwithpolygon(boxscanptr);
				if (merged) /* if list was modified */
				{
					/* set flag that we made a modification */
					db_mrg_polycoordlist.modified = 1;
					merged = 0; /* reset indicator flag */

					/* update area in (float) SRP920622 */
					area += (float)(boxscanptr->top - boxscanptr->bot) *
						(float)(boxscanptr->right - boxscanptr->left);

					/* if box is first on list */
					if (boxscanptr == lay_box->box)
					{
						garbage = boxscanptr;
						lay_box->box = boxscanptr->nextbox;
						boxscanptr = lay_box->box;
						garbage->nextbox = NULL;
						db_mrg_trash(garbage);

						/* if NULL list, we are done */
						if (lay_box->box == NULL)
						{
							lay_not_done = 0;
							break;
						}
					} else
					{
						/* not the first on list */
						garbage = boxscanptr;
						oldboxscan->nextbox = boxscanptr->nextbox;
						boxscanptr = boxscanptr->nextbox;
						garbage->nextbox = NULL;
						db_mrg_trash(garbage);
					}
				} else
				{
					/* no merge done, check next box */
					oldboxscan = boxscanptr;
					boxscanptr = boxscanptr->nextbox;
				}

				/* allow maxnumpts as a max */
				if (db_mrg_polycoordlist.numedge > maxnumpts) break;
			} /* end for */

			/*
			 * write out box if: (1) entire pass yielded no merges
			 * (2) end of list (3) too many points on list
			 */
			if ((!db_mrg_polycoordlist.modified) || (!lay_not_done) ||
				(db_mrg_polycoordlist.numedge > maxnumpts))
			{
				db_mrg_write_polygon(writepolygon, area);
				db_mrg_remove_poly_list(); /* dispose of polygon records */

				/* re-initialize for next pass */
				area = 0.0;     /* since the caller is accumulating areas (SRP) */
				db_mrg_polycoordlist.firstpoly = NULL;
				db_mrg_polycoordlist.numedge = 0;
				db_mrg_polycoordlist.modified = 0;
			}
		} /* end while */
	}
}

/* Macro to insert edge on polygon */
# define INSERT_EDGE(pred,tmp,copy,succ,dmaep1,dmaep2) \
  tmp = db_mrg_getpoly(); \
  tmp->nextpoly = succ; \
  if (pred != NULL) {pred->nextpoly = tmp;} \
  db_mrg_assign_edge(tmp, dmaep1, dmaep2); \
  copy = tmp;

/* Macro to remove n edges (nedge) from polygon */
# define REMOVE_EDGE(pred,start,tmp,nedge) \
  for (i = 0; i < nedge; i++) { \
	  tmp = start->nextpoly; \
	  pred->nextpoly = tmp; \
	  if (start == db_mrg_polycoordlist.firstpoly) { \
		  db_mrg_polycoordlist.firstpoly = tmp; \
	  } \
	  db_mrg_killpoly(start); \
	  start = tmp; \
  }

INTSML db_mrg_mergeboxwithpolygon(BOXPOLY *boxptr)
{
	POLYCOORD *polyptr,*creatptr,*backptr;
	INTSML pair, ok, one_con, i, firstcon, err, ed_con;
	INTBIG connect, auxx, ed_chk, j;

	pair = 0; /* flag to indicate a pair */
	one_con = 0; /* flas to intdicate that intersection was found */
	boxptr->numsides = 0;
	ok = 0;
	backptr = NULL;

	/* initialise seen field if list exists */
	if (db_mrg_polycoordlist.firstpoly != NULL)
	{
		/* initialize pointer to list */
		polyptr = db_mrg_polycoordlist.firstpoly;
		for(i=0; i!=db_mrg_polycoordlist.numedge;)
		{
			polyptr->seen = 1;
			i++;
			polyptr = polyptr->nextpoly;
		}
	}
	for(i=0; i<4; i++)
	{
		boxptr->numint[i] = 0;
		boxptr->polyint[i]=boxptr->auxpolyint[i] = NULL;
	}
	if (db_mrg_polycoordlist.firstpoly == NULL) /* if nothing on list */
	{
		/* make first edge (at head) */
		INSERT_EDGE(backptr,creatptr,backptr,NULL,boxptr->ll,boxptr->ul)
		db_mrg_polycoordlist.firstpoly = creatptr;

		/* second edge */
		INSERT_EDGE(backptr,creatptr,backptr,NULL,boxptr->ul,boxptr->ur)

		/* third edge */
		INSERT_EDGE(backptr,creatptr,backptr,NULL,boxptr->ur,boxptr->lr)

		/* fourth edge (and form ring) */
		INSERT_EDGE(backptr,creatptr,backptr,db_mrg_polycoordlist.firstpoly,boxptr->lr,boxptr->ll)

		db_mrg_housekeeper(); /* set up edge numbers */;
		return(1);  /* all done */
	} else /* we have some edges already */
	{
		for(polyptr=db_mrg_polycoordlist.firstpoly; polyptr->seen; /* check for intersection to all edges */
			polyptr=polyptr->nextpoly)
		{
			polyptr->seen = 0; /* indicate that we have been here */
			connect = db_mrg_check_intersect(boxptr, polyptr, &ed_con);
			if (connect == PTMATCH) /* if a point intersection was found */
				return(0); /* point intersections are not permitted */
			if (connect) /* if connection was found */
			{
				one_con = 1; /* found at least 1 connection */

				/* > 2 connections on one side */
				if (boxptr->numint[ed_con] == 2) return(0);
				if (pair && boxptr->polyint[ed_con] != NULL) /* 2 pairs found */
					return(0);
				if (boxptr->polyint[ed_con] == NULL) /* if first connection */
				{
					/* link edge to polygon descriptor */
					boxptr->polyint[ed_con] = polyptr;
					boxptr->numint[ed_con] += 1; /* increment count for side */

					/* increment number sides intersected */
					boxptr->numsides += 1;
				} else /* second connection for one "first" edge */
				{
					/* link to poly descriptor */
					boxptr->auxpolyint[ed_con] = polyptr;

					/* increment side connection count */
					boxptr->numint[ed_con] += 1;
					pair = 1; /* we have found a pair */
				}
			} /* end if connect */
		} /* end for */

		if (!one_con) /* if no connections found */
			return(0);
		else /* found at least one connection */
		{
			/* based on #sides of box which touched polygon */
			switch (boxptr->numsides)
			{
				case 4: /* intersects on 4 edges */
					if (pair) /* intersects 1 edge twice */
					{
						/* find edge which intersects twice */
						for(i=0; boxptr->numint[i]!=2; i++)
							;
						firstcon = i;
						if ((boxptr->polyint[db_mrg_nxtside(i)]->indx) ==
							db_mrg_succ((INTSML)(boxptr->polyint[i]->indx)))
								auxx = 0;  /* check the statement below */
						else if((boxptr->polyint[db_mrg_nxtside(i)]->indx) ==
							db_mrg_succ((INTSML)(boxptr->auxpolyint[i]->indx)))
								auxx = 1;
						else /* neither edge connection had next side with correct index number */
							return(0);
						if (auxx) /* if auxiliary pointer is the start */
						{
							for(ed_chk=db_mrg_nxtside(i); ed_chk!=i;
								ed_chk=db_mrg_nxtside((INTSML)ed_chk))
									if ((boxptr->polyint[db_mrg_nxtside((INTSML)ed_chk)]->indx) ==
										db_mrg_succ((INTSML)(boxptr->polyint[ed_chk]->indx)))
											continue;
									else
										/* successor had wrong index number */
										return(0);

							/* if we got here all is well */
							db_mrg_insert5(firstcon, (INTSML)auxx, boxptr);
						} else /* auxx = 0 */
						{
							for(ed_chk=db_mrg_nxtside(i); ed_chk!=i;
								ed_chk=db_mrg_nxtside((INTSML)ed_chk))
							{
								/* check for last edge */
								if (ed_chk != db_mrg_prevside(firstcon))
									if ((boxptr->polyint[db_mrg_nxtside(
										(INTSML)ed_chk)]->indx) == (db_mrg_succ(
											(INTSML)(boxptr->polyint[ed_chk]->indx))))
												continue;
									else
										/* index numbers out of sequence */
										return(0);
								else if ((boxptr->auxpolyint[db_mrg_nxtside(
									(INTSML)ed_chk)]->indx) == (db_mrg_succ(
										(INTSML)(boxptr->polyint[ed_chk]->indx))))
											continue;
								else
									/* index numbers out of sequence */
									return(0);
							}
							db_mrg_insert5(firstcon, (INTSML)auxx, boxptr);
						}
					} else /* intersects 4 edges, 1 time each edge */
					{
						ok = 0;

						/* find "start" edge of intersect sequence */
						for(i=0;i<4;i++)
						{
							err = 0;
							ed_chk = i;
							for(j = 0; j < 3; j++)
							{
								if ((boxptr->polyint[db_mrg_nxtside(
									(INTSML)ed_chk)]->indx) != (db_mrg_succ(
										(INTSML)(boxptr->polyint[ed_chk]->indx))))
											err = 1; /* error found */
								ed_chk = db_mrg_nxtside((INTSML)ed_chk);
							}
							if (!err)
							{
								ok = 1;
								break;
							}
						} /* end for */
						if (ok) /* check if we found a correct sequence */
						{
							firstcon = i;
							db_mrg_insert4(firstcon,boxptr);
						} else
							return(0); /* index out of sequence */
					}
					break;

				case 3:
					/* if a pair of contacts on 1 edge of box ... loop found */
					if (pair) return(0);
						else
					{
						/* i is index of edge which doesn't intersect polygon */
						for(i = 0; i < 4; i++)
							if (boxptr->polyint[i] == NULL) break;

						/* index of first non null side */
						firstcon = db_mrg_nxtside(i);
						for(j = firstcon; j != db_mrg_prevside(i);
							j = db_mrg_nxtside((INTSML)j))
								if ((boxptr->polyint[db_mrg_nxtside((INTSML)j)]->indx) !=
									(db_mrg_succ((INTSML)(boxptr->polyint[j]->indx))))
										return(0);
						db_mrg_insert3(firstcon,boxptr);
					}
					break;

				case 2: /* box has 2 edges which intersect polygon */
					if (pair) /* the usual */
						return(0);
					else
					{
						for(i = 0; i < 4; i++) /* find a NULL edge */
							if (boxptr->polyint[i] == NULL) break;
						for(j = i; boxptr->polyint[j] == NULL;
							j = db_mrg_nxtside((INTSML)j))
								;
						firstcon = j; /* first edge connection */

						/* have detected nonsadjacent pair of intersections */
						if (boxptr->polyint[db_mrg_nxtside(firstcon)] == NULL)
							return(0);
						if ((boxptr->polyint[db_mrg_nxtside((INTSML)j)]->indx) !=
							(db_mrg_succ((INTSML)(boxptr->polyint[j]->indx))))
								return(0); /* index numbers out of sequence... */
						db_mrg_insert2(firstcon,boxptr);
					}
					break;

				case 1: /* one edge intersects */
					if (pair) /* if a pair, we have a loop */
						return(0);
					else
					{
						/* find edge which intersects */
						for(i=0; boxptr->polyint[i]==NULL; i++)
							;
						firstcon = i;
						db_mrg_insert1(firstcon,boxptr);
					}
					break;

				default:
					break;
			} /* end switch */

			db_mrg_housekeeper();
			return(1); /* successful addition of box to polygon */
		} /* end else */
	} /* end have edges already */
}

/* returns 0 if no intersection,else 1 (line intersection) or 2 (point intersection) */
INTSML db_mrg_check_intersect(BOXPOLY *boxptr, POLYCOORD *polyptr, INTSML *ed_con)
{
	switch (polyptr->ed_or)
	{
		case UP:
			if (boxptr->right == polyptr->ed_val) /* may be colinear */
			{
				if ((boxptr->bot >= polyptr->ed_start[1] &&
					boxptr->bot < polyptr->ed_end[1]) ||
						(boxptr->top <= polyptr->ed_end[1] &&
							boxptr->top > polyptr->ed_start[1]) ||
								(boxptr->bot < polyptr->ed_start[1] &&
									boxptr->top > polyptr->ed_end[1]))
				{
					*ed_con = DOWN;
					return(1);
				}
				return(db_mrg_check_point(polyptr->ed_start,boxptr->ur,
					polyptr->ed_end,boxptr->lr)); /* check for pt connect */
			}
			return(0);

		case RIGHT:
			if (boxptr->bot == polyptr->ed_val)
			{
				if ((boxptr->left >= polyptr->ed_start[0] &&
					boxptr->left < polyptr->ed_end[0]) ||
						(boxptr->right > polyptr->ed_start[0] &&
							boxptr->right <= polyptr->ed_end[0]) ||
								(boxptr->left < polyptr->ed_start[0] &&
									boxptr->right > polyptr->ed_end[0]))
				{
					*ed_con = LEFT;
					return(1);
				}
				return(db_mrg_check_point(polyptr->ed_start,boxptr->lr,
					polyptr->ed_end,boxptr->ll));
			}
			return(0);

		case DOWN:
			if (boxptr->left == polyptr->ed_val)
			{
				if ((boxptr->bot >= polyptr->ed_end[1] &&
					boxptr->bot < polyptr->ed_start[1]) ||
						(boxptr->top > polyptr->ed_end[1] &&
							boxptr->top <= polyptr->ed_start[1]) ||
								(boxptr->top > polyptr->ed_start[1] &&
									boxptr->bot < polyptr->ed_end[1]))
				{
					*ed_con = UP;
					return(1);
				}
				return(db_mrg_check_point(polyptr->ed_start,boxptr->ll,
					polyptr->ed_end,boxptr->ul));
			}
			return(0);

		case LEFT:
			if (boxptr->top == polyptr->ed_val)
			{
				if ((boxptr->left >= polyptr->ed_end[0] &&
					boxptr->left < polyptr->ed_start[0]) ||
						(boxptr->right > polyptr->ed_end[0] &&
							boxptr->right <= polyptr->ed_start[0]) ||
								(boxptr->left < polyptr->ed_end[0] &&
									boxptr->right > polyptr->ed_start[0]))
				{
					*ed_con = RIGHT;
					return(1);
				}
				return(db_mrg_check_point(polyptr->ed_start,boxptr->ul,
					polyptr->ed_end,boxptr->ur));
			}
			return(0);
	} /* end switch */
	return(0);
}

/* PoinT EQual macro */

# define PTEQ(x,y) \
(((x)[0] == (y)[0]) && ((x)[1] == (y)[1]))

/* checks if points a1, a2 match or if points b1, b2 match */
INTSML db_mrg_check_point(INTBIG pointa1[], INTBIG pointa2[], INTBIG pointb1[],
	INTBIG pointb2[])
{
	if (PTEQ(pointa1, pointa2) || PTEQ(pointb1, pointb2)) {
		return(PTMATCH); /* at least one pair matches */
	}
	return(0); /* neither pair matches */
}

void db_mrg_insert1(INTSML firstcon, BOXPOLY *boxptr)
{
	POLYCOORD *polyptr, *nextedge, *creatptr, *backptr;
	INTBIG *p0, *p1, *p2, *p3;

	/* get pointer to edge of intersection */
	polyptr = boxptr->polyint[firstcon];
	nextedge = polyptr->nextpoly;
	for(backptr = db_mrg_polycoordlist.firstpoly; backptr->nextpoly != polyptr;
		backptr = backptr->nextpoly) /* gat pointer to edge before polyptr */
			;

	switch (polyptr->ed_or) { /* based on polygon edge orientation */
		case UP:
			/*  p3 --- p1  |
				|      \/  ^
				|      \/  ^
				p2 --- p0  |
			*/
			p0 = boxptr->lr;
			p1 = boxptr->ur;
			p2 = boxptr->ll;
			p3 = boxptr->ul;
			break;

		case RIGHT:
			/*  p2 --- p3
				|      |
				|      |
				p0 -<- p1
				--->>>---
			 */
			p0 = boxptr->ll;
			p1 = boxptr->lr;
			p2 = boxptr->ul;
			p3 = boxptr->ur;
			break;

		case DOWN:
			/*  | p0 --- p2
			   \/ ^      |
			   \/ ^      |
				| p1 --- p3
			 */
			p0 = boxptr->ul;
			p1 = boxptr->ll;
			p2 = boxptr->ur;
			p3 = boxptr->lr;
			break;

		case LEFT:
			/* ---<<<---
			   p1 ->- p0
			   |      |
			   |      |
			   p3 --- p2
			 */
			p0 = boxptr->ur;
			p1 = boxptr->ul;
			p2 = boxptr->lr;
			p3 = boxptr->ll;
			break;

		default:
			break;
	} /* end switch */
	/*
	  Generic code to handle case of 1 box edge touching 1 polygon edge
	  p0: point on box which might intersect start point of edge
	  p1: point on box which might intersect end point of edge
	  p2: point on box which is diagonally opposite p1
	  p3: point on box which is diagonally opposite p0
	  */

	/* if first point matches */
	if PTEQ(polyptr->ed_start, p0) {
		/* if second point matches */
		if PTEQ(polyptr->ed_end, p1) {
			/* both points match (adjust edge start/end points) */
			db_mrg_assign_edge(backptr, backptr->ed_start, p2);
			db_mrg_assign_edge(polyptr, p2, p3);
			db_mrg_assign_edge(nextedge, p3, nextedge->ed_end);
		}
		else { /* only first point matches (adjust 2 edges, add 2 edges) */
			db_mrg_assign_edge(backptr, backptr->ed_start, p2);
			db_mrg_assign_edge(polyptr, p2, p3);
			INSERT_EDGE(polyptr,creatptr,polyptr,nextedge, p3, p1)
			INSERT_EDGE(polyptr,creatptr,polyptr,nextedge, p1, nextedge->ed_start)
		}
	}
	else { /* first point does not match */
		/* if second point matches */
		if PTEQ(polyptr->ed_end, p1) {
			/* only second point matches (adjust 2 edges, add 2 edges) */
			db_mrg_assign_edge(polyptr,polyptr->ed_start, p0);
			db_mrg_assign_edge(nextedge, p3,nextedge->ed_end);
			INSERT_EDGE(polyptr,creatptr,polyptr,nextedge, p0, p2)
			INSERT_EDGE(polyptr,creatptr,polyptr,nextedge, p2, p3)
		}
		else {
			/* neither point matches (adjust first edge, add 4 new edges) */
			db_mrg_assign_edge(polyptr,polyptr->ed_start, p0);
			INSERT_EDGE(polyptr,creatptr,polyptr,nextedge, p0, p2)
			INSERT_EDGE(polyptr,creatptr,polyptr,nextedge, p2, p3)
			INSERT_EDGE(polyptr,creatptr,polyptr,nextedge, p3, p1)
			INSERT_EDGE(polyptr,creatptr,polyptr,nextedge, p1, nextedge->ed_start)
		}
	}
}

/* inserts box into polygon where 2 edges touched */
void db_mrg_insert2(INTSML firstcon, BOXPOLY *boxptr)
{
	POLYCOORD *polyptr, *nextedge, *creatptr, *backptr;
	INTBIG temp[2];
	INTBIG *p0, *p1, *p2;

	/* assign polyptr to first edge where intersection occurs */
	polyptr = boxptr->polyint[firstcon];
	temp[0] = polyptr->ed_end[0];
	temp[1] = polyptr->ed_end[1];
	nextedge = polyptr->nextpoly; /* get pointer to next edge */

	switch (polyptr->ed_or) { /* based on polygon edge direction */
		case RIGHT:
		/*
			p2-p1 |
			|  |  ^
			p0-|  ^
				  |
			-->>--|
		 */
			p0 = boxptr->ll;
			p1 = boxptr->ur;
			p2 = boxptr->ul;
			break;

		case LEFT:
		/*
			--<<--
			| |--p0
		   \/ |   |
		   \/ |   |
			| p1-p2
		 */
			p0 = boxptr->ur;
			p1 = boxptr->ll;
			p2 = boxptr->lr;
			break;

		case UP:
		/*
			--<<--|
				  |
			p1--| ^
			|   | ^
			|   | |
			p2-p0 |
		 */
			p0 = boxptr->lr;
			p1 = boxptr->ul;
			p2 = boxptr->ll;
			break;

		case DOWN:
		/*
			 | p0--p2
			\/ |   |
			\/ |   |
			 | |---p1
			 |
			 -->>----
		 */
			p0 = boxptr->ul;
			p1 = boxptr->lr;
			p2 = boxptr->ur;
			break;

		default:
			break;
	} /* end switch */
	/*
	  Generic code to handle case of 2 box edges touching 2 polygon edges
	  p0: point on box which might intersect start point of firsst polygon edge
	  p1: point on box which might intersect end point of second polygon edge
	  p2: point on box which is diagonally opposite "corner" of intersection of polygon edges
	  */

	/* if first point matches */
	if PTEQ(polyptr->ed_start, p0) {
		/* if second point matches */
		if PTEQ(nextedge->ed_end, p1) {
			/* both points match */
			db_mrg_assign_edge(polyptr,polyptr->ed_start,p2);
			db_mrg_assign_edge(nextedge,p2,nextedge->ed_end);
		}
		else {
			/* first matches only */
			for(backptr = db_mrg_polycoordlist.firstpoly;
				backptr->nextpoly != polyptr;
				backptr = backptr->nextpoly)
					;
			db_mrg_assign_edge(backptr,backptr->ed_start,p2);
			db_mrg_assign_edge(polyptr,p2,p1);
			db_mrg_assign_edge(nextedge,p1,nextedge->ed_end);
		}
	}
	else { /* first point doesnt match */
		/* second point matches */
		if PTEQ(nextedge->ed_end, p1) {
			db_mrg_assign_edge(polyptr,polyptr->ed_start,p0);
			db_mrg_assign_edge(nextedge,p0,p2);
			nextedge = nextedge->nextpoly;
			db_mrg_assign_edge(nextedge,p2,nextedge->ed_end);
		}
		else {
			/* neither point touches */
			db_mrg_assign_edge(polyptr,polyptr->ed_start,p0);
			INSERT_EDGE(polyptr,creatptr,polyptr,nextedge,p0,p2)
			INSERT_EDGE(polyptr,creatptr,polyptr,nextedge,p2,p1)
			db_mrg_assign_edge(nextedge,p1,nextedge->ed_end);
		}
	}
}

void db_mrg_insert3(INTSML firstcon, BOXPOLY *boxptr)
{
	POLYCOORD *polyptr, *nextedge, *garbage, *backptr, *temp;
	INTSML i;
	INTBIG *p0, *p1;

	/* assign polyptr to first edge */
	polyptr = boxptr->polyint[firstcon];
	nextedge = polyptr->nextpoly;
	nextedge = nextedge->nextpoly; /* third edge which intersects */
	for(backptr = db_mrg_polycoordlist.firstpoly; backptr->nextpoly != polyptr;
		backptr = backptr->nextpoly)  /* get back pointer */
			;

	switch (polyptr->ed_or) { /* based on polygon edge direction */
		case RIGHT:
		/*
			---<<---|
					|
			p1----| |
			|     | ^
			|     | ^
			p0----| |
					|
			--->>---|
		 */
			p0 = boxptr->ll;
			p1 = boxptr->ul;
			break;

		case LEFT:
		/*
		   |---<<---
		   |
		   | |----p0
		  \/ |     |
		  \/ |     |
		   | |----p1
		   |
		   |--->>---
		 */
			p0 = boxptr->ur;
			p1 = boxptr->lr;
			break;

		case UP:
		/*
		  |---<<----|
		  |         |
		  | |-----| |
		 \/ |     | ^
		 \/ |     | ^
		  | p1---p0 |
		  |         |
		 */
			p0 = boxptr->lr;
			p1 = boxptr->ll;
			break;

		case DOWN:
		/*
		  |         |
		  | p0---p1 |
		 \/ |     | ^
		 \/ |     | ^
		  | |-----| |
		  |         |
		  |--->>----|
		 */
			p0 = boxptr->ul;
			p1 = boxptr->ur;
			break;

		default:
			break;
	} /* end switch */
	/*
	  Generic code to handle case of 3 box edges touching 3 polygon edges
	  p0: point on box which might intersect start point of first polygon edge
	  p1: point on box which might intersect end point of third polygon edge
	  */

	/* if first point matches */
	if PTEQ(polyptr->ed_start, p0) {
		/* if second point matches */
		if PTEQ(nextedge->ed_end, p1) {
		  /* both points match */
			nextedge = nextedge->nextpoly;
			db_mrg_assign_edge(backptr,backptr->ed_start,
							   nextedge->ed_end);
			REMOVE_EDGE(backptr,polyptr,garbage,4) /* remove 4 edges beginning with polyptr */
		}
		else { /* first point matches only */
			db_mrg_assign_edge(backptr,backptr->ed_start,p1);
			db_mrg_assign_edge(nextedge,p1,nextedge->ed_end);
			REMOVE_EDGE(backptr,polyptr,garbage,2) /* remove 2 edges beginning with polyptr */
		}
	}
	else { /* first point doesn't match */
		if PTEQ(nextedge->ed_end, p1) {
			/* second point matches only */
			garbage = polyptr->nextpoly;
			nextedge = nextedge->nextpoly;
			db_mrg_assign_edge(polyptr,polyptr->ed_start,p0);
			db_mrg_assign_edge(nextedge,p0,nextedge->ed_end);
			REMOVE_EDGE(polyptr,garbage,temp,2) /* remove 2 edges beginning with garbage */
		}
		else { /* neither point matches */
			db_mrg_assign_edge(polyptr,polyptr->ed_start,p0);
			temp = polyptr->nextpoly;
			db_mrg_assign_edge(temp,p0,p1);
			db_mrg_assign_edge(nextedge,p1,nextedge->ed_end);
		}
	}
}

/* inserts box where each edge intersected once */
void db_mrg_insert4(INTSML firstcon, BOXPOLY *boxptr)
{
	INTSML i;
	POLYCOORD *polyptr, *nextedge, *garbage, *temp, *backptr;
	INTBIG *p0;
	INTBIG descendp; /* for the below/above/rightof/leftof case */

	polyptr = boxptr->polyint[firstcon];
	nextedge = boxptr->polyint[db_mrg_prevside(firstcon)];
	for(backptr = db_mrg_polycoordlist.firstpoly; backptr->nextpoly != polyptr;
		backptr = backptr->nextpoly)  /* get back pointer */
			;

	switch (polyptr->ed_or) { /* based on polygon edge direction */
		case RIGHT:
		/*
		   |----<<---|
		   |         |
		  \/ |-----| |
		  \/ |     | ^
		   | |     | ^
			 p0----| |
					 |
			 --->>---|
		 */
			p0 = boxptr->ll;
			descendp = ((nextedge->ed_end[1] < polyptr->ed_start[1]) ? 1 : 0);
			break;

		case LEFT:
		/*
		   |---<<---
		   |
		   | |----p0
		  \/ |     | |
		  \/ |     | ^
		   | |-----| ^
		   |         |
		   |--->>----|
		 */
			p0 = boxptr->ur;
			descendp = ((nextedge->ed_end[1] > polyptr->ed_start[1]) ? 1 : 0);
			break;

		case UP:
		/*
		  |---<<----|
		  |         |
		  | |-----| |
		 \/ |     | ^
		 \/ |     | ^
		  | |----p0 |
		  |         |
		  |--->>-
		 */
			p0 = boxptr->lr;
			descendp = ((nextedge->ed_end[0] > polyptr->ed_start[0]) ? 1 : 0);
			break;

		case DOWN:
		/*
			  -<<---|
		  |         |
		  | p0---p1 |
		 \/ |     | ^
		 \/ |     | ^
		  | |-----| |
		  |         |
		  |--->>----|
		 */
			p0 = boxptr->ul;
			descendp = ((nextedge->ed_end[0] < polyptr->ed_start[0]) ? 1 : 0);
			break;

		default:
			break;
	} /* end switch */
	/*
	  Generic code to handle case where 4 polygon edges perfectly enclose
	  (not necessarily fullythough) box
	  p0: start point on box which intersects first edge on polygon
	  */

	/* if first point matches */
	if PTEQ(polyptr->ed_start, p0) {
		db_mrg_assign_edge(backptr,backptr->ed_start,nextedge->ed_end);
		REMOVE_EDGE(backptr,polyptr,garbage,4) /* remove 4 edges beginning with polyptr */
	}
	else if PTEQ(nextedge->ed_end, p0) { /* if second point */
		nextedge = nextedge->nextpoly;
		db_mrg_assign_edge(polyptr,polyptr->ed_start,nextedge->ed_end);
		temp = polyptr->nextpoly;
		REMOVE_EDGE(polyptr,temp,garbage,4) /* remove 4 edges following polyptr */
	}
	else if (descendp) {
		/* this is a weird case (next->end "below"/"above"/"rightof"/"leftof" p0) */
		db_mrg_assign_edge(nextedge,p0,nextedge->ed_end);
		db_mrg_assign_edge(polyptr,polyptr->ed_start,p0);
		temp = polyptr->nextpoly;
		REMOVE_EDGE(polyptr,temp,garbage,2) /* remove 2 edges following polyptr */
	}
	else { /* more normal case */
		db_mrg_assign_edge(polyptr,polyptr->ed_start,p0);
		temp = polyptr->nextpoly;
		db_mrg_assign_edge(nextedge,p0,nextedge->ed_end);
		REMOVE_EDGE(polyptr,temp,garbage,2) /* remove 2 edges following polyptr */
	}
}

/* write polygon */
void db_mrg_write_polygon(void (*writepolygon)(INTBIG*, INTBIG*, INTSML, float), float area)
{
	static INTBIG *xbuf, *ybuf;
	static INTSML buflen = 0;
	POLYCOORD *polyptr;
	INTSML i;

	if (db_mrg_polycoordlist.numedge > buflen) { /* verify adequate buffer space */
		if (buflen != 0) { /* free and allocate new, larger buffer */
			efree((char *)xbuf);
			efree((char *)ybuf);
		}
		buflen = db_mrg_polycoordlist.numedge;
		xbuf = (INTBIG *)emalloc((buflen * SIZEOFINTBIG), db_cluster);
		ybuf = (INTBIG *)emalloc((buflen * SIZEOFINTBIG), db_cluster);
	}

	polyptr = db_mrg_polycoordlist.firstpoly;
	for(i=0; i<db_mrg_polycoordlist.numedge; i++) {
		xbuf[i] = polyptr->ed_start[0];
		ybuf[i] = polyptr->ed_start[1];
		polyptr = polyptr->nextpoly; /* increment pointer */
	}
	(*writepolygon)(xbuf, ybuf, db_mrg_polycoordlist.numedge, area);
}

/* returns successor polygon edge # */
INTSML db_mrg_succ(INTSML index)
{
	if (index == (db_mrg_polycoordlist.numedge - 1))
		return(0);
	return(index + 1);
}

/* returns index number of next edge on box, in CCW order */
INTSML db_mrg_nxtside(INTSML index)
{
	if (index == 0) return(3);
	return(index - 1);
}

/* returns index number of previous edge on box */
INTSML db_mrg_prevside(INTSML index)
{
	if (index == 3) return(0);
	return(index+1);
}

/* assigns characteristics to polygon edge */
void db_mrg_assign_edge(POLYCOORD *polyptr, INTBIG start[], INTBIG end[])
{
	INTSML i;

	if (start[0] == end[0]) /* if same in X */
	{
		polyptr->ed_val = start[0];
		if (start[1] < end[1]) polyptr->ed_or = UP;
			else polyptr->ed_or = DOWN;
	} else /* same in Y */
	{
		polyptr->ed_val = start[1];
		if (start[0] < end[0]) polyptr->ed_or = RIGHT;
			else polyptr->ed_or = LEFT;
	}
	for(i = 0; i < 2; i++)
	{
		polyptr->ed_start[i] = start[i];
		polyptr->ed_end[i] = end[i];
	}
}

/* corrects for co-linear edges, and updates coordinate indices */
void db_mrg_housekeeper(void)
{
	POLYCOORD *polyptr, *garbage, *nextedge;
	INTSML indx;

	for(polyptr = db_mrg_polycoordlist.firstpoly;
		polyptr->nextpoly != db_mrg_polycoordlist.firstpoly;
			polyptr = polyptr->nextpoly)
	polyptr->seen = 1;
	polyptr->seen = 1; /* do last on one ring */
	polyptr = db_mrg_polycoordlist.firstpoly;
	for(nextedge = polyptr->nextpoly; polyptr->seen;)
	{
		if (polyptr->ed_or == nextedge->ed_or) /* if edge orientation is same */
		{
			garbage = nextedge;
			polyptr->ed_end[0] = nextedge->ed_end[0]; /* copy end point */
			polyptr->ed_end[1] = nextedge->ed_end[1];

			/* if about to remove list head */
			if (nextedge == db_mrg_polycoordlist.firstpoly)
				db_mrg_polycoordlist.firstpoly = nextedge->nextpoly;
			polyptr->nextpoly = nextedge->nextpoly;
			nextedge = nextedge->nextpoly;
			garbage->nextpoly = NULL;
			db_mrg_killpoly(garbage);
		} else
		{
			/* not colinear, set as seen */
			polyptr->seen = 0;
			polyptr = nextedge;
			nextedge = nextedge->nextpoly;
		}
	} /* colinearities removed */

	/* update sequence numbers */
	indx = 0;
	for(polyptr = db_mrg_polycoordlist.firstpoly; polyptr->nextpoly !=
		db_mrg_polycoordlist.firstpoly; polyptr = polyptr->nextpoly)
	{
		polyptr->indx = indx;
		indx += 1;
	}
	polyptr->indx = indx; /* do last one */
	db_mrg_polycoordlist.numedge = indx + 1; /* set number of edges */
}

/* inserts box into polygon where intersected 5* */
void db_mrg_insert5(INTSML firstcon, INTSML auxx, BOXPOLY *boxptr)
{
	POLYCOORD *polyptr, *nextedge, *garbage, *temp;
	INTSML i;

	if (auxx) { /* if auxptr points to first intersection */
		polyptr = boxptr->auxpolyint[firstcon];
		nextedge = boxptr->polyint[firstcon];
	} else {
		polyptr = boxptr->polyint[firstcon];
		nextedge = boxptr->auxpolyint[firstcon];
	}

	/* insertion position is independent of edge orientation */
	db_mrg_assign_edge(polyptr,polyptr->ed_start,nextedge->ed_end);
	temp = polyptr->nextpoly;
	REMOVE_EDGE(polyptr,temp,garbage,4) /* remove 4 edges following polyptr */
}

/* some support stuff to do box poly descriptor mem management. */
BOXPOLY *db_mrg_getbox(void)
{
	BOXPOLY *b;

	if (db_mrg_free_box_list == NULL) {
		b = (BOXPOLY *)emalloc(sizeof(BOXPOLY), db_cluster);
		if (b == 0) return(NULL);
		b->nextbox = NULL;
		return(b);
	}
	b = db_mrg_free_box_list;
	db_mrg_free_box_list = b->nextbox;
	b->nextbox = NULL;
	return(b);
}

void db_mrg_trash(BOXPOLY *b)
{
	b->nextbox = db_mrg_free_box_list;
	db_mrg_free_box_list = b;
}

void db_mrg_dump_free_box_list(void)
{
	BOXPOLY *b;

	while (db_mrg_free_box_list != NULL) {
		b = db_mrg_free_box_list;
		db_mrg_free_box_list = b->nextbox;
		b->nextbox = NULL;
		efree((char *)b);
	}
}

void db_mrg_assign_box(BOXPOLY *ptr, INTBIG left, INTBIG right, INTBIG top, INTBIG bot)
{
	ptr->ul[0] = ptr->ll[0] = ptr->left = left;
	ptr->ur[0] = ptr->lr[0] = ptr->right = right;
	ptr->ul[1] = ptr->ur[1] = ptr->top = top;
	ptr->ll[1] = ptr->lr[1] = ptr->bot = bot;
	if ((ptr->right - ptr->left) > (ptr->top - ptr->bot)) {
		ptr->ishor = 1;
	}
	else {
		ptr->ishor = 0;
	}
}

POLYCOORD *db_mrg_getpoly(void)
{
	POLYCOORD *p;

	if (db_mrg_free_poly_list == NULL) {
		p = (POLYCOORD *)emalloc(sizeof(POLYCOORD), db_cluster);
		if (p == 0) return(NULL);
		p->nextpoly = NULL;
		return(p);
	}
	p = db_mrg_free_poly_list;
	db_mrg_free_poly_list = p->nextpoly;
	p->nextpoly = NULL;
	return(p);
}

void db_mrg_killpoly(POLYCOORD *p)
{
	p->nextpoly = db_mrg_free_poly_list;
	db_mrg_free_poly_list = p;
}

void db_mrg_dump_free_poly_list(void)
{
	POLYCOORD *p;

	while (db_mrg_free_poly_list != NULL) {
		p = db_mrg_free_poly_list;
		db_mrg_free_poly_list = p->nextpoly;
		p->nextpoly = NULL;
		efree((char *)p);
	}
}

void db_mrg_remove_poly_list(void)
{
	POLYCOORD *polyptr,*endptr;

	/* find end of list and break ring */
	for(endptr = db_mrg_polycoordlist.firstpoly;
		endptr->nextpoly != db_mrg_polycoordlist.firstpoly;
		endptr = endptr->nextpoly)
			;

	/* ring is broken now */
	endptr->nextpoly = NULL;
	while (db_mrg_polycoordlist.firstpoly != NULL) {
		polyptr = db_mrg_polycoordlist.firstpoly;
		db_mrg_polycoordlist.firstpoly = polyptr->nextpoly;
		polyptr->nextpoly = NULL;
		db_mrg_killpoly(polyptr);
	}
}

BOXLISTHEAD *db_mrg_get_lay_head(void)
{
	BOXLISTHEAD *bl;

	if (db_mrg_free_boxhead_list == NULL) {
		bl = (BOXLISTHEAD *)emalloc(sizeof(BOXLISTHEAD), db_cluster);
		if (bl == 0) return(NULL);
		bl->nextlayer = NULL;
		return(bl);
	}
	bl = db_mrg_free_boxhead_list;
	db_mrg_free_boxhead_list = bl->nextlayer;
	bl->nextlayer = NULL;
	return(bl);
}

void db_mrg_dump_free_boxhead_list(void)
{
	BOXLISTHEAD *bl;

	while (db_mrg_free_boxhead_list != NULL) {
		bl = db_mrg_free_boxhead_list;
		db_mrg_free_boxhead_list = bl->nextlayer;
		bl->nextlayer = NULL;
		bl->box = NULL;
		efree((char *)bl);
	}
}

void db_mrg_remove_boxhead_list(void)
{
	BOXLISTHEAD *bl,*endptr;

	if (db_mrg_curr_facet != NULL) {
		endptr = NULL;
		for(bl = db_mrg_curr_facet; bl != NULL;) {
			bl->box = NULL;
			endptr = bl;
			bl = bl->nextlayer;
		}
		endptr->nextlayer = db_mrg_free_boxhead_list;
		db_mrg_free_boxhead_list = db_mrg_curr_facet;
		db_mrg_curr_facet = NULL;
	}
}
