/*
 * Electric(tm) VLSI Design System
 *
 * File: dbvars.c
 * Database object variables module
 * Written by: Steven M. Rubin, Electric Editor Incorporated
 *
 * Copyright (c) 1998 Electric Editor Incorporated.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Electric Editor Incorporated
 * 23470 Sunset Drive, Suite 108
 * Los Gatos, California 95033
 * support@electriceditor.com
 */

#include "global.h"
#include "database.h"
#include "egraphics.h"
#include <ctype.h>

#define	SPECNAME  0		/* look for special names in structure */
#define	VARNAME   1		/* look for variable names in structure */
#define	PORTCNAME 2		/* look for port names on nodeproto */
#define	ARCINAME  3		/* look for nodeinst/arcinst names in facet */
#define	NODEINAME 4		/* look for nodeinst/arcinst names in facet */
#define	PORTANAME 5		/* look for port arc names on nodeinst */
#define	PORTENAME 6		/* look for port export names on nodeinst */
#define	ARCNAME   7		/* look for arcproto names in technology */
#define	NODENAME  8		/* look for nodeproto names in technology */
#define	NULLNAME  9		/* no more names in structure */

typedef struct
{
	char  *name;
	UINTBIG type;
	INTBIG *ptr;
} NAMEVARS;

#define	MAXAPACK	5				/* maximum nested attribute searches */
struct attrsearch
{
	INTSML      *numvar;			/* address of object variable count */
	VARIABLE   **firstvar;			/* address of object variable list */
	NAMEVARS    *vars;				/* special names in this object */
	INTBIG       proto;				/* address of object prototype */
	INTBIG       object;			/* address of this object */
	INTBIG       state;				/* current name type under test */
	INTSML       varindex;			/* current index for variable search */
	INTSML       specindex;			/* current index for name search */
	NODEPROTO   *nodeprotoname;		/* current facet object */
	ARCPROTO    *arcprotoname;		/* current arcproto object */
	PORTARCINST *portarcinstname;	/* current portarcinst object */
	PORTEXPINST *portexpinstname;	/* current portexpinst object */
	PORTPROTO   *portprotoname;		/* current portproto object */
	ARCINST     *arciname;			/* current arcinst object */
	NODEINST    *nodeiname;			/* current nodeinst object */
} db_attrpacket[MAXAPACK];
static INTSML db_apackindex = 0;
static struct attrsearch *db_thisapack;
static INTBIG db_realaddress;

INTSML db_dovarchangequietly = 0;	/* nonzero to do next variable change quietly */

/* prototypes for local routines */
void db_initnodeinstlist(NODEINST*);
void db_initnodeprotolist(NODEPROTO*);
void db_initportarcinstlist(PORTARCINST*);
void db_initportexpinstlist(PORTEXPINST*);
void db_initportprotolist(PORTPROTO*);
void db_initarcinstlist(ARCINST*);
void db_initarcprotolist(ARCPROTO*);
void db_initgeomlist(GEOM*);
void db_initliblist(LIBRARY*);
void db_inittechlist(TECHNOLOGY*);
void db_initaidlist(AIDENTRY*);
void db_initrtnodelist(RTNODE*);
void db_initnetworklist(NETWORK*);
void db_initcelllist(CELL*);
void db_initviewlist(VIEW*);
void db_initwindowlist(WINDOW*);
void db_initgraphicslist(GRAPHICS*);
void db_initconstraintlist(CONSTRAINT*);
INTSML db_getkey(char*);
VARIABLE *db_evalvar(VARIABLE*);
CLUSTER *db_whichcluster(INTBIG, INTBIG);
INTSML db_setvalkey(INTSML*, VARIABLE**, INTSML, INTBIG, INTBIG, INTBIG, CLUSTER*);
INTSML db_setvalvar(VARIABLE*, INTBIG, INTBIG, CLUSTER*);
INTSML db_getindvar(VARIABLE*, INTBIG, INTBIG*);
INTSML db_setindvar(INTBIG, INTBIG, VARIABLE*, INTBIG, INTBIG);
INTSML db_insindvar(INTBIG, INTBIG, VARIABLE*, INTBIG, INTBIG);
INTSML db_delindvar(INTBIG, INTBIG, VARIABLE*, INTBIG);
INTSML db_setval(char*, char*, INTBIG, CLUSTER*);
INTSML db_binarysearch(INTSML, VARIABLE*, INTBIG);
void db_renamevar(INTSML, VARIABLE*, INTBIG, INTBIG);
INTSML db_getdatasize(INTBIG);
char *db_describetype(INTBIG type);

/************************* ACCESSING VARIABLES *************************/

/*
 * routine to find the variable with name "name", which must be of type
 * "want", on the object whose address is "addr" and type is "type".  If
 * "want" is -1 then any type will do.  If the variable does not exist or
 * is the wrong type, NOVARIABLE is returned.
 */
VARIABLE *getval(INTBIG addr, INTBIG type, INTBIG want, char *name)
{
	static VARIABLE *var;
	REGISTER char *pp;
	REGISTER INTBIG search;

	search = initobjlist(addr, type, 0);
	if (search == 0) return(NOVARIABLE);
	for(;;)
	{
		pp = nextobjectlist(&var, search);
		if (pp == 0) break;
		if (namesame(pp, name) == 0)
		{
			/* variable found: make sure it is right type */
			if (want != -1)
			{
				if ((var->type & (VTYPE|VISARRAY)) != want) return(NOVARIABLE);
			}
			return(db_evalvar(var));
		}
	}
	return(NOVARIABLE);
}

/*
 * routine to find the variable with key "key", which must be of type "want",
 * on the object whose address is "addr" and type is "type".  If the
 * variable does not exist or is the wrong type, NOVARIABLE is returned.
 */
VARIABLE *getvalkey(INTBIG addr, INTBIG type, INTBIG want, INTBIG key)
{
	REGISTER INTSML med;
	INTSML *mynumvar;
	REGISTER VARIABLE *var;
	VARIABLE **myfirstvar;

	/* get the firstvar/numvar into the globals */
	if (db_getvarptr(addr, type, &myfirstvar, &mynumvar) != 0) return(NOVARIABLE);

	/* binary search variable space for the key */
	med = db_binarysearch(*mynumvar, *myfirstvar, key);
	if (med < 0) return(NOVARIABLE);

	/* get the variable */
	var = &(*myfirstvar)[med];

	/* make sure it is right type if a type was specified */
	if (want != -1 && (var->type & (VTYPE|VISARRAY)) != want) return(NOVARIABLE);
	return(db_evalvar(var));
}

/*
 * routine to get an entry in the array variable "name" in object "addr"
 * which is of type "type".  Entry "index" is placed into "value".  Returns
 * nonzero on error.
 */
INTSML getind(INTBIG addr, INTBIG type, char *name, INTBIG index, INTBIG *value)
{
	VARIABLE *var;
	REGISTER char *pp;
	REGISTER INTBIG search;

	search = initobjlist(addr, type, 0);
	if (search == 0) return(1);
	for(;;)
	{
		pp = nextobjectlist(&var, search);
		if (pp == 0) break;
		if (namesame(pp, name) == 0) break;
	}

	/* variable must exist */
	if (pp == 0) return(1);

	if ((var->type&VCANTSET) != 0) return(1);

	return(db_getindvar(var, index, value));
}

/*
 * routine to get an entry in the array variable whose key is "key" in object
 * "addr" which is of type "type".  Entry "index" is placed in "value".
 * The routine returns nonzero upon error.
 */
INTSML getindkey(INTBIG addr, INTBIG type, INTBIG key, INTBIG index, INTBIG *value)
{
	VARIABLE **myfirstvar;
	INTSML *mynumvar;
	REGISTER INTSML med;

	/* get the attribute list into the globals */
	if (db_getvarptr(addr, type, &myfirstvar, &mynumvar) != 0) return(1);

	/* binary search variable space for the key */
	med = db_binarysearch(*mynumvar, *myfirstvar, key);

	/* variable must exist */
	if (med < 0) return(1);

	if (((*myfirstvar)[med].type&VCANTSET) != 0) return(1);

	/* get the variable */
	return(db_getindvar(&(*myfirstvar)[med], index, value));
}

/*
 * routine to modify the text descriptor on variable "var".  The new
 * descriptor is set to "newdescript"
 */
void modifydescript(VARIABLE *var, INTBIG newdescript)
{
	REGISTER INTBIG olddescript;

	if (var != NOVARIABLE)
	{
		olddescript = var->textdescript;
		var->textdescript = newdescript;
	}

	/* tell constraint system about killed variable */
	(*el_curconstraint->modifydescript)(var, olddescript);

	/* record the change */
	(void)db_change((INTBIG)var, DESCRIPTMOD, olddescript, 0, 0, 0, 0, 0);
}

/*
 * routine to return the length of the array in variable "var".  If it is not
 * an array, the value -1 is returned.
 */
INTBIG getlength(VARIABLE *var)
{
	REGISTER INTBIG len;

	if ((var->type&VISARRAY) == 0) return(-1);
	len = (var->type&VLENGTH) >> VLENGTHSH;
	if (len != 0) return(len);

	if ((var->type&VTYPE) == VCHAR)
	{
		for(len=0; ((((char *)var->addr)[len])&0377) != 0377; len++) ;
	} else if ((var->type&VTYPE) == VDOUBLE)
	{
		for(len=0; ((double *)var->addr)[len] != -1; len++) ;
	} else if ((var->type&VTYPE) == VSHORT)
	{
		for(len=0; ((INTSML *)var->addr)[len] != -1; len++) ;
	} else
	{
		for(len=0; ((INTBIG *)var->addr)[len] != -1; len++) ;
	}
	return(len);
}

/*
 * routine to convert a variable name in "name" to a key and return it.
 * If an error is detected, the routine returns -1.
 */
INTBIG makekey(char *name)
{
	REGISTER char **newname;
	REGISTER INTSML med, i;

	/* search namespace for this name */
	med = db_getkey(name);
	if (med >= 0) return((INTBIG)el_namespace[med]);

	/* convert return value to proper index for new name */
	med = -med-1;

	/* make sure the name is valid */
	for(i=0; name[i] != 0; i++) if (name[i] == ' ' || name[i] == '\t')
		return(db_error(DBBADNAME|DBMAKEKEY));

	/* allocate space for new list */
	newname = (char **)emalloc(((el_numnames+1)*(sizeof (char *))), db_cluster);
	if (newname == 0) return(db_error(DBNOMEM|DBMAKEKEY));

	/* copy old list up to the new entry */
	for(i=0; i<med; i++) newname[i] = el_namespace[i];

	/* add the new entry */
	if (allocstring(&newname[med], name, db_cluster) != 0) return(-1);

	/* copy old list after the new entry */
	for(i=med; i<el_numnames; i++) newname[i+1] = el_namespace[i];

	/* clean-up */
	if (el_numnames != 0) efree((char *)el_namespace);
	el_namespace = newname;
	el_numnames++;
	return((INTBIG)el_namespace[med]);
}

/*
 * routine to convert a variable key in "key" to a name and return it.
 * Because of the techniques used, this is trivial.
 */
char *makename(INTBIG key)
{
	return((char *)key);
}

/*
 * routine to initialize a search of the variable names on object "addr"
 * of type "type".  Subsequent calls to "nextobjectlist" will return
 * variable names and fill the parameter with the variable addresses.
 * If "restrict" is nonzero, only list those attributes directly on the
 * object (and not those in linked lists).  The routine returns zero upon
 * error, and otherwise returns a value that must be passed to the subsequent
 * "nextobjectlist" calls
 */
INTBIG initobjlist(INTBIG addr, INTBIG type, INTSML restrict)
{
	db_thisapack = &db_attrpacket[db_apackindex++];
	if (db_apackindex >= MAXAPACK) db_apackindex = 0;
	switch (type&VTYPE)
	{
		case VNODEINST:   db_initnodeinstlist((NODEINST *)addr);       break;
		case VNODEPROTO:  db_initnodeprotolist((NODEPROTO *)addr);     break;
		case VPORTARCINST:db_initportarcinstlist((PORTARCINST *)addr); break;
		case VPORTEXPINST:db_initportexpinstlist((PORTEXPINST *)addr); break;
		case VPORTPROTO:  db_initportprotolist((PORTPROTO *)addr);     break;
		case VARCINST:    db_initarcinstlist((ARCINST *)addr);         break;
		case VARCPROTO:   db_initarcprotolist((ARCPROTO *)addr);       break;
		case VGEOM:       db_initgeomlist((GEOM *)addr);               break;
		case VLIBRARY:    db_initliblist((LIBRARY *)addr);             break;
		case VTECHNOLOGY: db_inittechlist((TECHNOLOGY *)addr);         break;
		case VAID:        db_initaidlist((AIDENTRY *)addr);            break;
		case VRTNODE:     db_initrtnodelist((RTNODE *)addr);           break;
		case VNETWORK:    db_initnetworklist((NETWORK *)addr);         break;
		case VCELL:       db_initcelllist((CELL *)addr);               break;
		case VVIEW:       db_initviewlist((VIEW *)addr);               break;
		case VWINDOW:     db_initwindowlist((WINDOW *)addr);           break;
		case VGRAPHICS:   db_initgraphicslist((GRAPHICS *)addr);       break;
		case VCONSTRAINT: db_initconstraintlist((CONSTRAINT *)addr);   break;
		default:    (void)db_error(DBBADOBJECT|DBINITOBJLIST);         return(0);
	}
	if (restrict != 0)
		if (db_thisapack->state != NULLNAME) db_thisapack->state = SPECNAME;
	return((INTBIG)db_thisapack);
}

/*
 * routine to return the next name in the list (and the associated variable)
 */
char *nextobjectlist(VARIABLE **var, INTBIG ain)
{
	struct attrsearch *apack;
	REGISTER INTSML datasize, indir;
	REGISTER char *retval;
	REGISTER NODEINST *ni;
	REGISTER ARCINST *ai;
	static char line[50];
	static VARIABLE v;
	REGISTER VARIABLE *nvar;

	apack = (struct attrsearch *)ain;
	for(;;) switch (apack->state)
	{
		case SPECNAME:	/* looking for special names in object structure */
			if (apack->vars[apack->specindex].name != 0)
			{
				/* get next special name */
				*var = &v;
				v.key = apack->specindex;
				v.type = apack->vars[apack->specindex].type;
				v.textdescript = TXTSMALL << VTSIZESH;
				v.addr = (INTBIG)apack->vars[apack->specindex].ptr - apack->proto + apack->object;
				db_realaddress = v.addr;
				indir = 0;
				if ((v.type&VISARRAY) == 0) indir++; else
					if ((v.type&VCREF) == 0) indir++;
				if (indir != 0)
				{
					datasize = db_getdatasize(v.type);
					switch (datasize)
					{
						case 1:           v.addr = *(char *)v.addr;    break;
						case SIZEOFINTBIG: v.addr = *(INTBIG *)v.addr;   break;
						case SIZEOFINTSML: v.addr = *(INTSML *)v.addr;   break;
					}
				}
				v.type |= VCREF;
				return(apack->vars[apack->specindex++].name);
			}
			/* no more special names, go to next case */
			apack->state = VARNAME;
			break;
		case VARNAME:	/* looking for variable names in object */
			if (apack->varindex < *apack->numvar)
			{
				/* get next variable name */
				*var = &(*apack->firstvar)[apack->varindex];
				return((char *)(*apack->firstvar)[apack->varindex++].key);
			}
			/* no more variable names, go to terminal case */
			apack->state = NULLNAME;
			break;
		case ARCINAME:	/* looking for arcs in the nodeproto */
			if (apack->arciname != NOARCINST)
			{
				/* get next arcinst name */
				ai = apack->arciname;
				apack->arciname = apack->arciname->nextarcinst;
				*var = &v;   v.key = -1;
				v.type = VARCINST;
				v.textdescript = TXTSMALL << VTSIZESH;
				v.addr = (INTBIG)ai;
				nvar = getvalkey((INTBIG)ai, VARCINST, VSTRING, el_arc_name);
				if (nvar != NOVARIABLE) return((char *)nvar->addr);
				(void)sprintf(line, "arc%d", (INTBIG)ai);
				return(line);
			}
			apack->state = NODEINAME;
			break;
		case NODEINAME:	/* looking for nodes in the nodeproto */
			if (apack->nodeiname != NONODEINST)
			{
				/* get next nodeinst name */
				ni = apack->nodeiname;
				apack->nodeiname = apack->nodeiname->nextnodeinst;
				*var = &v;   v.key = -1;
				v.type = VNODEINST;
				v.textdescript = TXTSMALL << VTSIZESH;
				v.addr = (INTBIG)ni;
				nvar = getvalkey((INTBIG)ni, VNODEINST, VSTRING, el_node_name);
				if (nvar != NOVARIABLE) return((char *)nvar->addr);
				(void)sprintf(line, "node%d", (INTBIG)ni);
				return(line);
			}
			/* no more nodeinst names, go to next case */
			apack->state = PORTCNAME;
			break;
		case PORTCNAME:	/* looking for port names on nodeproto */
			if (apack->portprotoname != NOPORTPROTO)
			{
				/* get next port name */
				*var = &v;   v.key = -1;
				v.textdescript = TXTSMALL << VTSIZESH;
				v.type = VPORTPROTO;   v.addr = (INTBIG)apack->portprotoname;
				retval = apack->portprotoname->protoname;
				apack->portprotoname = apack->portprotoname->nextportproto;
				return(retval);
			}
			/* no more port names, go to next case */
			apack->state = SPECNAME;
			break;
		case PORTANAME:	/* looking for portarcinst names on nodeinst */
			if (apack->portarcinstname != NOPORTARCINST)
			{
				/* get next portarcinst name */
				*var = &v;   v.key = -1;
				v.type = VPORTARCINST;
				v.textdescript = TXTSMALL << VTSIZESH;
				v.addr = (INTBIG)apack->portarcinstname;
				retval = apack->portarcinstname->proto->protoname;
				apack->portarcinstname = apack->portarcinstname->nextportarcinst;
				return(retval);
			}
			/* no more portarcinst names, go to next case */
			apack->state = PORTENAME;
			break;
		case PORTENAME:	/* looking for portexpinst names on nodeinst */
			if (apack->portexpinstname != NOPORTEXPINST)
			{
				/* get next portexpinst name */
				*var = &v;   v.key = -1;
				v.type = VPORTEXPINST;
				v.textdescript = TXTSMALL << VTSIZESH;
				v.addr = (INTBIG)apack->portexpinstname;
				retval = apack->portexpinstname->proto->protoname;
				apack->portexpinstname = apack->portexpinstname->nextportexpinst;
				return(retval);
			}
			/* no more portexpinst names, go to next case */
			apack->state = SPECNAME;
			break;
		case ARCNAME:	/* looking for arcs in the technology */
			if (apack->arcprotoname != NOARCPROTO)
			{
				/* get next arcproto name */
				*var = &v;   v.key = -1;
				v.textdescript = TXTSMALL << VTSIZESH;
				v.type = VARCPROTO;   v.addr = (INTBIG)apack->arcprotoname;
				retval = apack->arcprotoname->protoname;
				apack->arcprotoname = apack->arcprotoname->nextarcproto;
				return(retval);
			}
			/* no more arcproto names, go to next case */
			apack->state = NODENAME;
			break;
		case NODENAME:	/* looking for facets in the tech/lib */
			if (apack->nodeprotoname != NONODEPROTO)
			{
				/* get next facet name */
				*var = &v;   v.key = -1;
				v.textdescript = TXTSMALL << VTSIZESH;
				v.type = VNODEPROTO;   v.addr = (INTBIG)apack->nodeprotoname;
				retval = describenodeproto(apack->nodeprotoname);
				apack->nodeprotoname = apack->nodeprotoname->nextnodeproto;
				return(retval);
			}
			/* no more nodeproto names, go to next case */
			apack->state = SPECNAME;
			break;
		case NULLNAME:
			*var = NOVARIABLE;
			return(0);
	}
}

void nextvarchangequiet(void)
{
	db_dovarchangequietly = 1;
}

/********************** DELETING AND RENAMING VARIABLES *********************/

/*
 * delete the variable "name" from object whose address is "addr" and type
 * is "type".  Returns nonzero if there is an error.
 */
INTSML delval(INTBIG addr, INTBIG type, char *name)
{
	REGISTER INTBIG key;

	key = makekey(name);
	if (key == -1)
	{
		db_dovarchangequietly = 0;
		return(1);
	}

	return(delvalkey(addr, type, key));
}

/*
 * delete the variable with the key "key" from object whose address is
 * "addr" and type is "type".  Returns nonzero if there is an error.
 */
INTSML delvalkey(INTBIG addr, INTBIG type, INTBIG key)
{
	REGISTER VARIABLE *var;
	VARIABLE **myfirstvar;
	INTSML *mynumvar;
	REGISTER INTSML i, j;
	REGISTER INTBIG oldaddr, oldtype, olddescript;

	/* get the firstvar/numvar pointers */
	if (db_getvarptr(addr, type, &myfirstvar, &mynumvar) != 0)
	{
		db_dovarchangequietly = 0;
		return(1);
	}

	/* search for the varaible in the list */
	for(i=0; i<*mynumvar; i++)
		if ((*myfirstvar)[i].key == key) break;
	if (i >= *mynumvar)
	{
		ttyputmsg("Internal error: tried to delete %s on object of type %s.  No worry.",
			makename(key), db_describetype(type));
		db_dovarchangequietly = 0;
		return(db_error(DBNOVAR|DBDELVALKEY));
	}

	/* delete the variable */
	var = &(*myfirstvar)[i];
	oldaddr = var->addr;
	oldtype = var->type;
	olddescript = var->textdescript;

	/* delete the entry in the variable list */
	(*mynumvar)--;
	for(j = i; j < *mynumvar; j++)
	{
		(*myfirstvar)[j].key  = (*myfirstvar)[j+1].key;
		(*myfirstvar)[j].type = (*myfirstvar)[j+1].type;
		(*myfirstvar)[j].textdescript = (*myfirstvar)[j+1].textdescript;
		(*myfirstvar)[j].addr = (*myfirstvar)[j+1].addr;
	}
	if (*mynumvar == 0) efree((char *)*myfirstvar);

	if (db_dovarchangequietly == 0)
	{
		/* tell constraint system about killed variable */
		(*el_curconstraint->killvariable)(addr, type, key, oldaddr, oldtype, olddescript);

		/* record the change */
		(void)db_change((INTBIG)addr, VARIABLEKILL, type, key, oldaddr, oldtype, olddescript, 0);
	} else
	{
		/* not going through change control, so delete the memory now */
		db_dovarchangequietly = 0;
		db_freevar(oldaddr, oldtype);
	}
	return(0);
}

/*
 * routine to rename variable "oldname" to "newname" everywhere in the
 * database.  Returns negative if the operation cannot be done.
 * This routine does not delete the old name from the namespace but it does
 * rename the variable wherever it exists, so the old name is no longer
 * in use.  NOTE: this does not check variables on R-tree nodes.
 */
INTSML renameval(char *oldname, char *newname)
{
	REGISTER INTBIG oldkey, newkey;
	REGISTER INTSML i;
	REGISTER NODEPROTO *np;
	REGISTER CELL *c;
	REGISTER ARCPROTO *ap;
	REGISTER PORTPROTO *pp;
	REGISTER NETWORK *net;
	REGISTER VIEW *v;
	REGISTER NODEINST *ni;
	REGISTER ARCINST *ai;
	REGISTER LIBRARY *lib;
	REGISTER TECHNOLOGY *tech;

	/* get key of old name */
	oldkey = db_getkey(oldname);
	if (oldkey < 0) return(-1);
	oldkey = (INTBIG)el_namespace[oldkey];

	/* make sure new name is different */
	if (namesame(oldname, newname) == 0) return(-1);

	/* make new name key */
	newkey = makekey(newname);
	if (newkey < 0) return(-1);

	/* search the libraries */
	for(lib = el_curlib; lib != NOLIBRARY; lib = lib->nextlibrary)
	{
		db_renamevar(lib->numvar, lib->firstvar, oldkey, newkey);
		for(c = lib->firstcell; c != NOCELL; c = c->nextcell)
			db_renamevar(c->numvar, c->firstvar, oldkey, newkey);
		for(np=lib->firstnodeproto; np!=NONODEPROTO; np=np->nextnodeproto)
		{
			db_renamevar(np->numvar, np->firstvar, oldkey, newkey);
			for(pp=np->firstportproto; pp!=NOPORTPROTO; pp=pp->nextportproto)
			{
				db_renamevar(pp->numvar, pp->firstvar, oldkey, newkey);
				db_renamevar(pp->subportexpinst->numvar,
					pp->subportexpinst->firstvar, oldkey, newkey);
			}
			for(ni=np->firstnodeinst; ni!=NONODEINST; ni=ni->nextnodeinst)
			{
				db_renamevar(ni->numvar, ni->firstvar, oldkey, newkey);
				db_renamevar(ni->geom->numvar, ni->geom->firstvar, oldkey, newkey);
			}
			for(ai=np->firstarcinst; ai!=NOARCINST; ai=ai->nextarcinst)
			{
				db_renamevar(ai->numvar, ai->firstvar, oldkey, newkey);
				db_renamevar(ai->geom->numvar, ai->geom->firstvar, oldkey, newkey);
				db_renamevar(ai->end[0].portarcinst->numvar,
					ai->end[0].portarcinst->firstvar, oldkey, newkey);
				db_renamevar(ai->end[1].portarcinst->numvar,
					ai->end[1].portarcinst->firstvar, oldkey, newkey);
			}
			for(net = np->firstnetwork; net != NONETWORK; net = net->nextnetwork)
				db_renamevar(net->numvar, net->firstvar, oldkey, newkey);
		}
	}

	/* search the aids */
	for(i=0; i<el_maxaid; i++)
		db_renamevar(el_aids[i].numvar, el_aids[i].firstvar, oldkey, newkey);

	/* search the views */
	for(v = el_views; v != NOVIEW; v = v->nextview)
		db_renamevar(v->numvar, v->firstvar, oldkey, newkey);

	/* search the technologies */
	for(tech = el_technologies; tech != NOTECHNOLOGY; tech = tech->nexttechnology)
	{
		db_renamevar(tech->numvar, tech->firstvar, oldkey, newkey);
		for(ap=tech->firstarcproto; ap!=NOARCPROTO; ap=ap->nextarcproto)
			db_renamevar(ap->numvar, ap->firstvar, oldkey, newkey);
		for(np=tech->firstnodeproto; np!=NONODEPROTO; np=np->nextnodeproto)
			db_renamevar(np->numvar, np->firstvar, oldkey, newkey);
	}
	return(0);
}

/************************* SETTING ENTIRE VARIABLES *************************/

/*
 * routine to set the value of entry "name" in object "addr" which is of
 * type "type".  The entry is set to "newaddr" which has type "newtype".
 * The routine returns the variable address (NOVARIABLE on error).
 */
VARIABLE *setval(INTBIG addr, INTBIG type, char *name, INTBIG newaddr, INTBIG newtype)
{
	VARIABLE *var;
	REGISTER char *pp;
	REGISTER INTBIG key, search;

	/* look for an attribute that already has this name */
	search = initobjlist(addr, type, 0);
	if (search == 0)
	{
		db_dovarchangequietly = 0;
		return(NOVARIABLE);
	}
	for(;;)
	{
		pp = nextobjectlist(&var, search);
		if (pp == 0) break;
		if (namesame(pp, name) == 0) break;
	}

	/* if the attribute exists, change its value */
	if (pp != 0)
	{
		if ((var->type&VCANTSET) != 0)
		{
			db_dovarchangequietly = 0;
			return((VARIABLE *)db_error(DBVARFIXED|DBSETVAL));
		}

		if (db_dovarchangequietly == 0)
		{
			/* tell constraint system about killed variable */
			(*el_curconstraint->killvariable)(addr, type, var->key, var->addr, var->type, var->textdescript);

			/* record a change for removal of the old variable */
			(void)db_change((INTBIG)addr, VARIABLEKILL, type, var->key,
				var->addr, var->type, var->textdescript, 0);
		}

		if (db_setvalvar(var, newaddr, newtype, db_whichcluster(addr, type)) != 0)
		{
			db_dovarchangequietly = 0;
			return((VARIABLE *)db_error(DBNOMEM|DBSETVAL));
		}

		if (db_dovarchangequietly == 0)
		{
			/* tell constraint system about new variable */
			(*el_curconstraint->newvariable)(addr, type, var->key, var->type);

			(void)db_change((INTBIG)addr, VARIABLENEW, type, var->key, var->type, 0, 0, 0);
		} else db_dovarchangequietly = 0;
		return(var);
	}

	/* create new variable: first ensure the name starts with a letter */
	if (!isalpha(*name))
	{
		db_dovarchangequietly = 0;
		return((VARIABLE *)db_error(DBBADNAME|DBSETVAL));
	}

	/* get the key of the new name */
	key = makekey(name);
	if (key == -1)
	{
		db_dovarchangequietly = 0;
		return(NOVARIABLE);
	}

	/* set the variable by its key */
	return(setvalkey(addr, type, key, newaddr, newtype));
}

/*
 * routine to set the variable on the object whose address is "addr", type
 * is "type", and key is "key".  The variable is set to "newaddr" with type
 * "newtype".  The routine returns the variable address (NOVARIABLE on error).
 */
VARIABLE *setvalkey(INTBIG addr, INTBIG type, INTBIG key, INTBIG newaddr,
	INTBIG newtype)
{
	REGISTER INTSML med;
	VARIABLE **myfirstvar;
	INTSML *mynumvar;

	/* get the attribute list pointers */
	if (db_getvarptr(addr, type, &myfirstvar, &mynumvar) != 0)
	{
		db_dovarchangequietly = 0;
		return(NOVARIABLE);
	}

	/* binary search variable space for the key */
	med = db_binarysearch(*mynumvar, *myfirstvar, key);
	if (med >= 0)
	{
		if (((*myfirstvar)[med].type&VCANTSET) != 0)
		{
			db_dovarchangequietly = 0;
			return((VARIABLE *)db_error(DBVARFIXED|DBSETVALKEY));
		}

		if (db_dovarchangequietly == 0)
		{
			/* tell constraint system about killed variable */
			(*el_curconstraint->killvariable)(addr, type, key,
				(*myfirstvar)[med].addr, (*myfirstvar)[med].type, (*myfirstvar)[med].textdescript);

			/* record a change for removal of the old variable */
			(void)db_change((INTBIG)addr, VARIABLEKILL, type, key,
				(*myfirstvar)[med].addr, (*myfirstvar)[med].type, (*myfirstvar)[med].textdescript, 0);
		}
	}

	/* set the new variable */
	if (db_setvalkey(mynumvar, myfirstvar, med, key, newaddr, newtype, db_whichcluster(addr, type)) != 0)
	{
		db_dovarchangequietly = 0;
		return((VARIABLE *)db_error(DBNOMEM|DBSETVALKEY));
	}

	if (db_dovarchangequietly == 0)
	{
		/* tell constraint system about new variable */
		(*el_curconstraint->newvariable)(addr, type, key, newtype);

		/* record the change */
		(void)db_change((INTBIG)addr, VARIABLENEW, type, key, newtype, 0, 0, 0);
	} else db_dovarchangequietly = 0;
	if (med < 0) med = -1 - med;
	return(&(*myfirstvar)[med]);
}

/*
 * routine to copy the variables from object "fromaddr" (which has type
 * "fromtype") to object "toaddr" (which has type "totype").  Returns nonzero
 * on error.
 */
INTSML copyvars(INTBIG fromaddr, INTBIG fromtype, INTBIG toaddr, INTBIG totype)
{
	REGISTER INTSML i;
	REGISTER INTBIG key, addr, type;
	INTBIG lx, hx, ly, hy;
	REGISTER GEOM *geom;
	REGISTER NODEPROTO *np;
	REGISTER NODEINST *ni;
	REGISTER ARCINST *ai;
	INTSML *numvar;
	VARIABLE **firstvar;
	REGISTER VARIABLE *var;

	if (db_getvarptr(fromaddr, fromtype, &firstvar, &numvar) != 0) return(1);

	for(i=0; i<(*numvar); i++)
	{
		key = (*firstvar)[i].key;
		addr = (*firstvar)[i].addr;
		type = (*firstvar)[i].type;

		var = setvalkey(toaddr, totype, key, addr, type);
		if (var == NOVARIABLE) return(1);
		var->textdescript = (*firstvar)[i].textdescript;
	}
	
	/* variables may affect geometry size */
	if (totype == VNODEINST || totype == VARCINST)
	{
		if (totype == VNODEINST)
		{
			ni = (NODEINST *)toaddr;
			geom = ni->geom;
			np = ni->parent;
		} else
		{
			ai = (ARCINST *)toaddr;
			geom = ai->geom;
			np = ai->parent;
		}
		boundobj(geom, &lx, &hx, &ly, &hy);
		if (lx != geom->lowx || hx != geom->highx ||
			ly != geom->lowy || hy != geom->highy)
				updategeom(geom, np);
	}
	return(0);
}

/*********************** SETTING ENTRIES IN VARIABLES ***********************/

/*
 * routine to set an entry in the array variable "name" in object "addr"
 * which is of type "type".  Entry "index" is set to "newaddr".
 * The routine returns nonzero upon error.
 */
INTSML setind(INTBIG addr, INTBIG type, char *name, INTBIG index, INTBIG newaddr)
{
	VARIABLE *var;
	REGISTER char *pp;
	REGISTER INTBIG search;
	REGISTER INTSML retval;

	search = initobjlist(addr, type, 0);
	if (search == 0)
	{
		db_dovarchangequietly = 0;
		return(1);
	}
	for(;;)
	{
		pp = nextobjectlist(&var, search);
		if (pp == 0) break;
		if (namesame(pp, name) == 0) break;
	}

	/* variable must exist */
	if (pp == 0)
	{
		db_dovarchangequietly = 0;
		return(db_error(DBNOVAR|DBSETIND));
	}

	if ((var->type&VCANTSET) != 0)
	{
		db_dovarchangequietly = 0;
		return(db_error(DBVARFIXED|DBSETIND));
	}

	retval = db_setindvar(addr, type, var, index, newaddr);
	db_dovarchangequietly = 0;
	if (retval != 0) return(db_error(DBNOMEM|DBSETIND));
	return(0);
}

/*
 * routine to set an entry in the array variable whose key is "key" in object
 * "addr" which is of type "type".  Entry "index" is set to "newaddr".
 * The routine returns nonzero upon error.
 */
INTSML setindkey(INTBIG addr, INTBIG type, INTBIG key, INTBIG index, INTBIG newaddr)
{
	VARIABLE **myfirstvar;
	INTSML *mynumvar;
	REGISTER INTSML med, retval;

	/* get the attribute list into the globals */
	if (db_getvarptr(addr, type, &myfirstvar, &mynumvar) != 0)
	{
		db_dovarchangequietly = 0;
		return(1);
	}

	/* binary search variable space for the key */
	med = db_binarysearch(*mynumvar, *myfirstvar, key);

	/* variable must exist */
	if (med < 0)
	{
		db_dovarchangequietly = 0;
		return(db_error(DBNOVAR|DBSETINDKEY));
	}

	if (((*myfirstvar)[med].type&VCANTSET) != 0)
	{
		db_dovarchangequietly = 0;
		return(db_error(DBVARFIXED|DBSETINDKEY));
	}

	/* set the variable */
	retval = db_setindvar(addr, type, &(*myfirstvar)[med], index, newaddr);
	db_dovarchangequietly = 0;
	if (retval != 0) return(db_error(DBNOMEM|DBSETINDKEY));
	return(0);
}

/*
 * routine to insert an entry in the array variable "name" in object "addr"
 * which is of type "type".  Entry "index" is set to "newaddr" and all entries
 * equal to or greater than "index" are moved up (i.e. "newaddr" is inserted
 * before entry "index").  The routine returns nonzero upon error.
 */
INTSML insind(INTBIG addr, INTBIG type, char *name, INTBIG index, INTBIG newaddr)
{
	VARIABLE *var;
	REGISTER char *pp;
	REGISTER INTBIG search;
	REGISTER INTSML retval;

	search = initobjlist(addr, type, 0);
	if (search == 0)
	{
		db_dovarchangequietly = 0;
		return(1);
	}
	for(;;)
	{
		pp = nextobjectlist(&var, search);
		if (pp == 0) break;
		if (namesame(pp, name) == 0) break;
	}

	/* variable must exist */
	if (pp == 0) 
	{
		db_dovarchangequietly = 0;
		return(db_error(DBNOVAR|DBINSIND));
	}

	/* cannot insert if variable is frozen or is in C data structures */
	if ((var->type&VCANTSET) != 0 || (var->type&VCREF) != 0)
	{
		db_dovarchangequietly = 0;
		return(db_error(DBVARFIXED|DBINSIND));
	}

	retval = db_insindvar(addr, type, var, index, newaddr);
	db_dovarchangequietly = 0;
	if (retval != 0) return(db_error(DBNOMEM|DBINSIND));
	return(0);
}

/*
 * routine to insert an entry in the array variable whose key is "key" in object
 * "addr" which is of type "type".  Entry "index" is set to "newaddr" and all entries
 * equal to or greater than "index" are moved up (i.e. "newaddr" is inserted
 * before entry "index").  The routine returns nonzero upon error.
 */
INTSML insindkey(INTBIG addr, INTBIG type, INTBIG key, INTBIG index, INTBIG newaddr)
{
	VARIABLE **myfirstvar;
	INTSML *mynumvar;
	REGISTER INTSML med, retval;

	/* get the attribute list into the globals */
	if (db_getvarptr(addr, type, &myfirstvar, &mynumvar) != 0)
	{
		db_dovarchangequietly = 0;
		return(1);
	}

	/* binary search variable space for the key */
	med = db_binarysearch(*mynumvar, *myfirstvar, key);

	/* variable must exist */
	if (med < 0)
	{
		db_dovarchangequietly = 0;
		return(db_error(DBNOVAR|DBINSINDKEY));
	}

	/* cannot insert if variable is frozen or is in C data structures */
	if (((*myfirstvar)[med].type&VCANTSET) != 0 || ((*myfirstvar)[med].type&VCREF) != 0)
	{
		db_dovarchangequietly = 0;
		return(db_error(DBVARFIXED|DBINSINDKEY));
	}

	/* set the variable */
	retval = db_insindvar(addr, type, &(*myfirstvar)[med], index, newaddr);
	db_dovarchangequietly = 0;
	if (retval != 0) return(db_error(DBNOMEM|DBINSINDKEY));
	return(0);
}

/*
 * routine to delete entry "index" in the array variable "name" in object "addr"
 * which is of type "type".  The routine returns nonzero upon error.
 */
INTSML delind(INTBIG addr, INTBIG type, char *name, INTBIG index)
{
	VARIABLE *var;
	REGISTER char *pp;
	REGISTER INTBIG search;
	REGISTER INTSML retval;

	search = initobjlist(addr, type, 0);
	if (search == 0)
	{
		db_dovarchangequietly = 0;
		return(1);
	}
	for(;;)
	{
		pp = nextobjectlist(&var, search);
		if (pp == 0) break;
		if (namesame(pp, name) == 0) break;
	}

	/* variable must exist */
	if (pp == 0)
	{
		db_dovarchangequietly = 0;
		return(db_error(DBNOVAR|DBDELIND));
	}

	if ((var->type&VCANTSET) != 0)
	{
		db_dovarchangequietly = 0;
		return(db_error(DBVARFIXED|DBDELIND));
	}

	retval = db_delindvar(addr, type, var, index);
	db_dovarchangequietly = 0;
	if (retval != 0) return(db_error(DBNOMEM|DBDELIND));
	return(0);
}

/*
 * routine to delete entry "index" in the array variable whose key is "key" in object
 * "addr" which is of type "type".  The routine returns nonzero upon error.
 */
INTSML delindkey(INTBIG addr, INTBIG type, INTBIG key, INTBIG index)
{
	VARIABLE **myfirstvar;
	INTSML *mynumvar;
	REGISTER INTSML med, retval;

	/* get the attribute list into the globals */
	if (db_getvarptr(addr, type, &myfirstvar, &mynumvar) != 0)
	{
		db_dovarchangequietly = 0;
		return(1);
	}

	/* binary search variable space for the key */
	med = db_binarysearch(*mynumvar, *myfirstvar, key);

	/* variable must exist */
	if (med < 0)
	{
		db_dovarchangequietly = 0;
		return(db_error(DBNOVAR|DBDELINDKEY));
	}

	if (((*myfirstvar)[med].type&VCANTSET) != 0)
	{
		db_dovarchangequietly = 0;
		return(db_error(DBVARFIXED|DBDELINDKEY));
	}

	/* set the variable */
	retval = db_delindvar(addr, type, &(*myfirstvar)[med], index);
	db_dovarchangequietly = 0;
	if (retval != 0) return(db_error(DBNOMEM|DBDELINDKEY));
	return(0);
}

/************************* DATA FOR TYPES *************************/

/*
 * In the "type" field of a variable, the VCREF bit is typically set
 * to indicate that the variable points into the fixed C structures, and
 * is not part of the extendable attributes in "firstvar/numvar".  However,
 * in the tables below, the VCREF bit is used to indicate dereferencing
 * in array attributes.  Those arrays that are fixed in size or otherwise
 * part of the C structure need no dereferencing, and have the VCREF bit set
 * (see RTNODE->pointers, GRAPHICS->raster, and GRAPHICS->style).  Those
 * arrays that vary in length have only pointers in their C structures, and
 * need dereferencing (see PORTPROTO->connects, LIBRARY->lambda,
 * TECHNOLOGY->layers, NETWORK->arcaddr, and NETWORK->networklist).  These
 * do not have the VCREF bit set.
 */

static NODEINST db_ptni;
static NAMEVARS db_nodevars[] =
{
	{"firstportarcinst",VPORTARCINST|VCANTSET,(INTBIG*)&db_ptni.firstportarcinst},
	{"firstportexpinst",VPORTEXPINST|VCANTSET,(INTBIG*)&db_ptni.firstportexpinst},
	{"geom",            VGEOM|VCANTSET,       (INTBIG*)&db_ptni.geom},
	{"highx",           VINTEGER|VCANTSET,    (INTBIG*)&db_ptni.highx},
	{"highy",           VINTEGER|VCANTSET,    (INTBIG*)&db_ptni.highy},
	{"lastinst",        VNODEINST|VCANTSET,   (INTBIG*)&db_ptni.lastinst},
	{"lastnodeinst",    VNODEINST|VCANTSET,   (INTBIG*)&db_ptni.lastnodeinst},
	{"lowx",            VINTEGER|VCANTSET,    (INTBIG*)&db_ptni.lowx},
	{"lowy",            VINTEGER|VCANTSET,    (INTBIG*)&db_ptni.lowy},
	{"nextinst",        VNODEINST|VCANTSET,   (INTBIG*)&db_ptni.nextinst},
	{"nextnodeinst",    VNODEINST|VCANTSET,   (INTBIG*)&db_ptni.nextnodeinst},
	{"parent",          VNODEPROTO|VCANTSET,  (INTBIG*)&db_ptni.parent},
	{"proto",           VNODEPROTO|VCANTSET,  (INTBIG*)&db_ptni.proto},
	{"rotation",        VSHORT|VCANTSET,      (INTBIG*)&db_ptni.rotation},
	{"temp1",           VINTEGER,             (INTBIG*)&db_ptni.temp1},
	{"temp2",           VINTEGER,             (INTBIG*)&db_ptni.temp2},
	{"textdescript",    VINTEGER,             (INTBIG*)&db_ptni.textdescript},
	{"transpose",       VSHORT|VCANTSET,      (INTBIG*)&db_ptni.transpose},
	{"userbits",        VINTEGER,             (INTBIG*)&db_ptni.userbits},
	{NULL, 0, NULL}  /* 0 */
};
static NODEPROTO db_ptnp;
static NAMEVARS db_nodeprotovars[] =
{
	{"adirty",          VINTEGER|VCANTSET,    (INTBIG*)&db_ptnp.adirty},
	{"cell",            VCELL|VCANTSET,       (INTBIG*)&db_ptnp.cell},
	{"cellview",        VVIEW,                (INTBIG*)&db_ptnp.cellview},
	{"creationdate",    VINTEGER|VCANTSET,    (INTBIG*)&db_ptnp.creationdate},
	{"firstarcinst",    VARCINST|VCANTSET,    (INTBIG*)&db_ptnp.firstarcinst},
	{"firstinst",       VNODEINST|VCANTSET,   (INTBIG*)&db_ptnp.firstinst},
	{"firstnetwork",    VNETWORK|VCANTSET,    (INTBIG*)&db_ptnp.firstnetwork},
	{"firstnodeinst",   VNODEINST|VCANTSET,   (INTBIG*)&db_ptnp.firstnodeinst},
	{"firstportproto",  VPORTPROTO|VCANTSET,  (INTBIG*)&db_ptnp.firstportproto},
	{"highx",           VINTEGER,             (INTBIG*)&db_ptnp.highx},
	{"highy",           VINTEGER,             (INTBIG*)&db_ptnp.highy},
	{"index",           VSHORT|VCANTSET,      (INTBIG*)&db_ptnp.index},
	{"lastnodeproto",   VNODEPROTO|VCANTSET,  (INTBIG*)&db_ptnp.lastnodeproto},
	{"lastversion",     VNODEPROTO|VCANTSET,  (INTBIG*)&db_ptnp.lastversion},
	{"lowx",            VINTEGER,             (INTBIG*)&db_ptnp.lowx},
	{"lowy",            VINTEGER,             (INTBIG*)&db_ptnp.lowy},
	{"newestversion",   VNODEPROTO|VCANTSET,  (INTBIG*)&db_ptnp.newestversion},
	{"nextincell",      VNODEPROTO|VCANTSET,  (INTBIG*)&db_ptnp.nextincell},
	{"nextnodeproto",   VNODEPROTO|VCANTSET,  (INTBIG*)&db_ptnp.nextnodeproto},
	{"primname",        VSTRING,              (INTBIG*)&db_ptnp.primname},
	{"revisiondate",    VINTEGER|VCANTSET,    (INTBIG*)&db_ptnp.revisiondate},
	{"rtree",           VRTNODE|VCANTSET,     (INTBIG*)&db_ptnp.rtree},
	{"tech",            VTECHNOLOGY|VCANTSET, (INTBIG*)&db_ptnp.tech},
	{"temp1",           VINTEGER,             (INTBIG*)&db_ptnp.temp1},
	{"temp2",           VINTEGER,             (INTBIG*)&db_ptnp.temp2},
	{"userbits",        VINTEGER,             (INTBIG*)&db_ptnp.userbits},
	{"version",         VSHORT|VCANTSET,      (INTBIG*)&db_ptnp.version},
	{NULL, 0, NULL}  /* 0 */
};
static PORTARCINST db_ptpi;
static NAMEVARS db_portavars[] =
{
	{"conarcinst",      VARCINST|VCANTSET,    (INTBIG*)&db_ptpi.conarcinst},
	{"nextportarcinst", VPORTARCINST|VCANTSET,(INTBIG*)&db_ptpi.nextportarcinst},
	{"proto",           VPORTPROTO|VCANTSET,  (INTBIG*)&db_ptpi.proto},
	{NULL, 0, NULL}  /* 0 */
};
static PORTEXPINST db_ptpe;
static NAMEVARS db_portevars[] =
{
	{"exportproto",     VPORTPROTO|VCANTSET,  (INTBIG*)&db_ptpe.exportproto},
	{"nextportexpinst", VPORTEXPINST|VCANTSET,(INTBIG*)&db_ptpe.nextportexpinst},
	{"proto",           VPORTPROTO|VCANTSET,  (INTBIG*)&db_ptpe.proto},
	{NULL, 0, NULL}  /* 0 */
};

static PORTPROTO db_ptpp;
static NAMEVARS db_portprotovars[] =
{
	{"connects",        VARCPROTO|VISARRAY|VCANTSET, (INTBIG*)&db_ptpp.connects},
	{"network",         VNETWORK|VCANTSET,           (INTBIG*)&db_ptpp.network},
	{"nextportproto",   VPORTPROTO|VCANTSET,         (INTBIG*)&db_ptpp.nextportproto},
	{"parent",          VNODEPROTO|VCANTSET,         (INTBIG*)&db_ptpp.parent},
	{"protoname",       VSTRING,                     (INTBIG*)&db_ptpp.protoname},
	{"subnodeinst",     VNODEINST|VCANTSET,          (INTBIG*)&db_ptpp.subnodeinst},
	{"subportexpinst",  VPORTEXPINST|VCANTSET,       (INTBIG*)&db_ptpp.subportexpinst},
	{"subportproto",    VPORTPROTO|VCANTSET,         (INTBIG*)&db_ptpp.subportproto},
	{"temp1",           VINTEGER,                    (INTBIG*)&db_ptpp.temp1},
	{"temp2",           VINTEGER,                    (INTBIG*)&db_ptpp.temp2},
	{"textdescript",    VINTEGER,                    (INTBIG*)&db_ptpp.textdescript},
	{"userbits",        VINTEGER,                    (INTBIG*)&db_ptpp.userbits},
	{NULL, 0, NULL}  /* 0 */
};

static ARCINST db_ptai;
static NAMEVARS db_arcvars[] =
{
	{"endshrink",      VINTEGER|VCANTSET,    (INTBIG*)&db_ptai.endshrink},
	{"geom",           VGEOM|VCANTSET,       (INTBIG*)&db_ptai.geom},
	{"lastarcinst",    VARCINST|VCANTSET,    (INTBIG*)&db_ptai.lastarcinst},
	{"length",         VINTEGER|VCANTSET,    (INTBIG*)&db_ptai.length},
	{"network",        VNETWORK|VCANTSET,    (INTBIG*)&db_ptai.network},
	{"nextarcinst",    VARCINST|VCANTSET,    (INTBIG*)&db_ptai.nextarcinst},
	{"nodeinst1",      VNODEINST|VCANTSET,   (INTBIG*)&db_ptai.end[0].nodeinst},
	{"nodeinst2",      VNODEINST|VCANTSET,   (INTBIG*)&db_ptai.end[1].nodeinst},
	{"parent",         VNODEPROTO|VCANTSET,  (INTBIG*)&db_ptai.parent},
	{"portarcinst1",   VPORTARCINST|VCANTSET,(INTBIG*)&db_ptai.end[0].portarcinst},
	{"portarcinst2",   VPORTARCINST|VCANTSET,(INTBIG*)&db_ptai.end[1].portarcinst},
	{"proto",          VARCPROTO|VCANTSET,   (INTBIG*)&db_ptai.proto},
	{"temp1",          VINTEGER,             (INTBIG*)&db_ptai.temp1},
	{"temp2",          VINTEGER,             (INTBIG*)&db_ptai.temp2},
	{"userbits",       VINTEGER,             (INTBIG*)&db_ptai.userbits},
	{"width",          VINTEGER|VCANTSET,    (INTBIG*)&db_ptai.width},
	{"xpos1",          VINTEGER|VCANTSET,    (INTBIG*)&db_ptai.end[0].xpos},
	{"xpos2",          VINTEGER|VCANTSET,    (INTBIG*)&db_ptai.end[1].xpos},
	{"ypos1",          VINTEGER|VCANTSET,    (INTBIG*)&db_ptai.end[0].ypos},
	{"ypos2",          VINTEGER|VCANTSET,    (INTBIG*)&db_ptai.end[1].ypos},
	{NULL, 0, NULL}  /* 0 */
};

static ARCPROTO db_ptap;
static NAMEVARS db_arcprotovars[] =
{
	{"index",           VSHORT|VCANTSET,      (INTBIG*)&db_ptap.index},
	{"nextarcproto",    VARCPROTO|VCANTSET,   (INTBIG*)&db_ptap.nextarcproto},
	{"nominalwidth",    VINTEGER,             (INTBIG*)&db_ptap.nominalwidth},
	{"protoname",       VSTRING,              (INTBIG*)&db_ptap.protoname},
	{"tech",            VTECHNOLOGY|VCANTSET, (INTBIG*)&db_ptap.tech},
	{"temp1",           VINTEGER,             (INTBIG*)&db_ptap.temp1},
	{"temp2",           VINTEGER,             (INTBIG*)&db_ptap.temp2},
	{"userbits",        VINTEGER,             (INTBIG*)&db_ptap.userbits},
	{NULL, 0, NULL}  /* 0 */
};

static GEOM db_ptgeom;
static NAMEVARS db_geomvars[] =
{
	{"entrytype",       VSHORT|VCANTSET,      (INTBIG*)&db_ptgeom.entrytype},
	{"entryaddr",       VINTEGER|VCANTSET,    (INTBIG*)&db_ptgeom.entryaddr},
	{"highx",           VINTEGER|VCANTSET,    (INTBIG*)&db_ptgeom.highx},
	{"highy",           VINTEGER|VCANTSET,    (INTBIG*)&db_ptgeom.highy},
	{"lowx",            VINTEGER|VCANTSET,    (INTBIG*)&db_ptgeom.lowx},
	{"lowy",            VINTEGER|VCANTSET,    (INTBIG*)&db_ptgeom.lowy},
	{NULL, 0, NULL}  /* 0 */
};

static LIBRARY db_ptlib;
static NAMEVARS db_libvars[] =
{
	{"curnodeproto",    VNODEPROTO,                 (INTBIG*)&db_ptlib.curnodeproto},
	{"firstcell",       VCELL|VCANTSET,             (INTBIG*)&db_ptlib.firstcell},
	{"firstnodeproto",  VNODEPROTO|VCANTSET,        (INTBIG*)&db_ptlib.firstnodeproto},
	{"lambda",          VINTEGER|VISARRAY|VCANTSET, (INTBIG*)&db_ptlib.lambda},
	{"libname",         VSTRING,                    (INTBIG*)&db_ptlib.libname},
	{"libfile",         VSTRING,                    (INTBIG*)&db_ptlib.libfile},
	{"nextlibrary",     VLIBRARY|VCANTSET,          (INTBIG*)&db_ptlib.nextlibrary},
	{"temp1",           VINTEGER,                   (INTBIG*)&db_ptlib.temp1},
	{"temp2",           VINTEGER,                   (INTBIG*)&db_ptlib.temp2},
	{"userbits",        VINTEGER,                   (INTBIG*)&db_ptlib.userbits},
	{NULL, 0, NULL}  /* 0 */
};

/*
 * entry 6 of the table below ("layers") gets changed by "db_inittechlist"
 */
static TECHNOLOGY db_pttech;
static NAMEVARS db_techvars[] =
{
	{"arcprotocount",     VSHORT|VCANTSET,             (INTBIG*)&db_pttech.arcprotocount},
	{"deflambda",         VINTEGER|VCANTSET,           (INTBIG*)&db_pttech.deflambda},
	{"firstarcproto",     VARCPROTO|VCANTSET,          (INTBIG*)&db_pttech.firstarcproto},
	{"firstnodeproto",    VNODEPROTO|VCANTSET,         (INTBIG*)&db_pttech.firstnodeproto},
	{"index",             VSHORT|VCANTSET,             (INTBIG*)&db_pttech.index},
	{"layercount",        VSHORT|VCANTSET,             (INTBIG*)&db_pttech.layercount},
	{"layers",            VGRAPHICS|VISARRAY|VCANTSET, (INTBIG*)&db_pttech.layers},
	{"nexttechnology",    VTECHNOLOGY|VCANTSET,        (INTBIG*)&db_pttech.nexttechnology},
	{"nodeprotocount",    VSHORT|VCANTSET,             (INTBIG*)&db_pttech.nodeprotocount},
	{"techdescript",      VSTRING|VCANTSET,            (INTBIG*)&db_pttech.techdescript},
	{"techname",          VSTRING,                     (INTBIG*)&db_pttech.techname},
	{"temp1",             VINTEGER,                    (INTBIG*)&db_pttech.temp1},
	{"temp2",             VINTEGER,                    (INTBIG*)&db_pttech.temp2},
	{"userbits",          VINTEGER,                    (INTBIG*)&db_pttech.userbits},
	{NULL, 0, NULL}  /* 0 */
};

static AIDENTRY db_ptaid;
static NAMEVARS db_aidvars[] =
{
	{"aidcap",          VSHORT|VCANTSET,      (INTBIG*)&db_ptaid.aidcap},
	{"aidname",         VSTRING|VCANTSET,     (INTBIG*)&db_ptaid.aidname},
	{"aidstate",        VINTEGER,             (INTBIG*)&db_ptaid.aidstate},
	{"index",           VSHORT|VCANTSET,      (INTBIG*)&db_ptaid.index},
	{NULL, 0, NULL}  /* 0 */
};

/*
 * entry 6 of the table below ("pointers") gets changed by "db_initrtnodelist"
 */
static RTNODE db_ptrtn;
static NAMEVARS db_rtnvars[] =
{
	{"flag",            VSHORT|VCANTSET,                 (INTBIG*)&db_ptrtn.flag},
	{"highx",           VINTEGER|VCANTSET,               (INTBIG*)&db_ptrtn.highx},
	{"highy",           VINTEGER|VCANTSET,               (INTBIG*)&db_ptrtn.highy},
	{"lowx",            VINTEGER|VCANTSET,               (INTBIG*)&db_ptrtn.lowx},
	{"lowy",            VINTEGER|VCANTSET,               (INTBIG*)&db_ptrtn.lowy},
	{"parent",          VRTNODE|VCANTSET,                (INTBIG*)&db_ptrtn.parent},
	{"pointers",        VRTNODE|VISARRAY|VCREF|VCANTSET, (INTBIG*)db_ptrtn.pointers},
	{"total",           VSHORT|VCANTSET,                 (INTBIG*)&db_ptrtn.total},
	{NULL, 0, NULL}  /* 0 */
};

/*
 * entry 0 of the table below ("arcaddr") gets changed by "db_initnetworklist"
 * entry 6 of the table below ("networklist") also gets changed
 */
static NETWORK db_ptnet;
static NAMEVARS db_netvars[] =
{
	{"arcaddr",         VARCINST|VISARRAY|VCANTSET,  (INTBIG*)&db_ptnet.arcaddr},
	{"arccount",        VSHORT|VCANTSET,             (INTBIG*)&db_ptnet.arccount},
	{"buslinkcount",    VSHORT|VCANTSET,             (INTBIG*)&db_ptnet.buslinkcount},
	{"lastnetwork",     VNETWORK|VCANTSET,           (INTBIG*)&db_ptnet.lastnetwork},
	{"namecount",       VSHORT|VCANTSET,             (INTBIG*)&db_ptnet.namecount},
	{"netname",         VSTRING|VCANTSET,            (INTBIG*)&db_ptnet.netname},
	{"networklist",     VNETWORK|VISARRAY|VCANTSET,  (INTBIG*)&db_ptnet.networklist},
	{"nextnetwork",     VNETWORK|VCANTSET,           (INTBIG*)&db_ptnet.nextnetwork},
	{"parent",          VNODEPROTO|VCANTSET,         (INTBIG*)&db_ptnet.parent},
	{"portcount",       VSHORT|VCANTSET,             (INTBIG*)&db_ptnet.portcount},
	{"refcount" ,       VSHORT|VCANTSET,             (INTBIG*)&db_ptnet.refcount},
	{"signals",         VSHORT|VCANTSET,             (INTBIG*)&db_ptnet.signals},
	{"temp1",           VINTEGER,                    (INTBIG*)&db_ptnet.temp1},
	{"temp2",           VINTEGER,                    (INTBIG*)&db_ptnet.temp2},
	{NULL, 0, NULL}  /* 0 */
};

static CELL db_ptcell;
static NAMEVARS db_cellvars[] =
{
	{"cellname",        VSTRING,              (INTBIG*)&db_ptcell.cellname},
	{"firstincell",     VNODEPROTO|VCANTSET,  (INTBIG*)&db_ptcell.firstincell},
	{"freenetwork",     VNETWORK|VCANTSET,    (INTBIG*)&db_ptcell.freenetwork},
	{"lib",             VLIBRARY|VCANTSET,    (INTBIG*)&db_ptcell.lib},
	{"nextcell",        VCELL|VCANTSET,       (INTBIG*)&db_ptcell.nextcell},
	{"temp1",           VINTEGER,             (INTBIG*)&db_ptcell.temp1},
	{"temp2",           VINTEGER,             (INTBIG*)&db_ptcell.temp2},
	{NULL, 0, NULL}  /* 0 */
};

static VIEW db_ptview;
static NAMEVARS db_viewvars[] =
{
	{"nextview",        VVIEW|VCANTSET,       (INTBIG*)&db_ptview.nextview},
	{"sviewname",       VSTRING|VCANTSET,     (INTBIG*)&db_ptview.sviewname},
	{"temp1",           VINTEGER,             (INTBIG*)&db_ptview.temp1},
	{"temp2",           VINTEGER,             (INTBIG*)&db_ptview.temp2},
	{"viewname",        VSTRING|VCANTSET,     (INTBIG*)&db_ptview.viewname},
	{"viewstate",       VINTEGER|VCANTSET,    (INTBIG*)&db_ptview.viewstate},
	{NULL, 0, NULL}  /* 0 */
};

static WINDOW db_ptwin;
static NAMEVARS db_winvars[] =
{
	{"buttonhandler",   VADDRESS,             (INTBIG*)&db_ptwin.buttonhandler},
	{"changehandler",   VADDRESS,             (INTBIG*)&db_ptwin.changehandler},
	{"charhandler",     VADDRESS,             (INTBIG*)&db_ptwin.charhandler},
	{"curnodeproto",    VNODEPROTO,           (INTBIG*)&db_ptwin.curnodeproto},
	{"editor",          VADDRESS,             (INTBIG*)&db_ptwin.editor},
	{"gridx",           VINTEGER,             (INTBIG*)&db_ptwin.gridx},
	{"gridy",           VINTEGER,             (INTBIG*)&db_ptwin.gridy},
	{"lastwindow",      VWINDOW|VCANTSET,     (INTBIG*)&db_ptwin.lastwindow},
	{"location",        VSTRING|VCANTSET,     (INTBIG*)&db_ptwin.location},
	{"nextwindow",      VWINDOW|VCANTSET,     (INTBIG*)&db_ptwin.nextwindow},
	{"redisphandler",   VADDRESS,             (INTBIG*)&db_ptwin.redisphandler},
	{"scalex",          VFLOAT,               (INTBIG*)&db_ptwin.scalex},
	{"scaley",          VFLOAT,               (INTBIG*)&db_ptwin.scaley},
	{"screenhx",        VINTEGER,             (INTBIG*)&db_ptwin.screenhx},
	{"screenhy",        VINTEGER,             (INTBIG*)&db_ptwin.screenhy},
	{"screenlx",        VINTEGER,             (INTBIG*)&db_ptwin.screenlx},
	{"screenly",        VINTEGER,             (INTBIG*)&db_ptwin.screenly},
	{"state",           VINTEGER,             (INTBIG*)&db_ptwin.state},
	{"termhandler",     VADDRESS,             (INTBIG*)&db_ptwin.termhandler},
	{"usehx",           VSHORT,               (INTBIG*)&db_ptwin.usehx},
	{"usehy",           VSHORT,               (INTBIG*)&db_ptwin.usehy},
	{"uselx",           VSHORT,               (INTBIG*)&db_ptwin.uselx},
	{"usely",           VSHORT,               (INTBIG*)&db_ptwin.usely},
	{NULL, 0, NULL}  /* 0 */
};

static GRAPHICS db_ptgra;
static NAMEVARS db_gravars[] =
{
	{"bits",            VSHORT,                               (INTBIG*)&db_ptgra.bits},
	{"col",             VSHORT,                               (INTBIG*)&db_ptgra.col},
	{"raster",          VSHORT|VISARRAY|VCREF|(8<<VLENGTHSH), (INTBIG*)db_ptgra.raster},
	{"style",           VSHORT|VISARRAY|VCREF|(4<<VLENGTHSH), (INTBIG*)db_ptgra.style},
	{NULL, 0, NULL}  /* 0 */
};

static CONSTRAINT db_ptcon;
static NAMEVARS db_convars[] =
{
	{"conname",         VSTRING|VCANTSET,     (INTBIG*)&db_ptcon.conname},
	{NULL, 0, NULL}  /* 0 */
};

/************************* CODE FOR TYPES *************************/

void db_initnodeinstlist(NODEINST *ni)
{
	if (ni == NONODEINST)
	{
		db_thisapack->state = NULLNAME;
		return;
	}
	db_thisapack->state = PORTANAME;		/* then PORTENAME, SPECNAME, VARNAME */
	db_thisapack->portarcinstname = ni->firstportarcinst;	/* PORTANAME */
	db_thisapack->portexpinstname = ni->firstportexpinst;	/* PORTENAME */
	db_thisapack->vars = db_nodevars;						/* SPECNAME */
	db_thisapack->proto = (INTBIG)&db_ptni;					/* SPECNAME */
	db_thisapack->object = (INTBIG)ni;						/* SPECNAME */
	db_thisapack->specindex = 0;							/* SPECNAME */
	db_thisapack->numvar = &ni->numvar;						/* VARNAME */
	db_thisapack->firstvar = &ni->firstvar;					/* VARNAME */
	db_thisapack->varindex = 0;								/* VARNAME */
}

void db_initnodeprotolist(NODEPROTO *np)
{
	if (np == NONODEPROTO)
	{
		db_thisapack->state = NULLNAME;
		return;
	}
	if (np->index == 0)
	{
		db_thisapack->state = ARCINAME;		/* then NODEINAME, PORTCNAME, SPECNAME, VARNAME */
		db_thisapack->arciname  = np->firstarcinst;			/* ARCINAME */
		db_thisapack->nodeiname  = np->firstnodeinst;		/* NODEINAME */
	} else db_thisapack->state = PORTCNAME;	/* then SPECNAME, VARNAME */
	db_thisapack->portprotoname = np->firstportproto;		/* PORTCNAME */
	db_thisapack->vars = db_nodeprotovars;					/* SPECNAME */
	db_thisapack->proto = (INTBIG)&db_ptnp;					/* SPECNAME */
	db_thisapack->object = (INTBIG)np;						/* SPECNAME */
	db_thisapack->specindex = 0;							/* SPECNAME */
	db_thisapack->numvar = &np->numvar;						/* VARNAME */
	db_thisapack->firstvar = &np->firstvar;					/* VARNAME */
	db_thisapack->varindex = 0;								/* VARNAME */
}

void db_initportarcinstlist(PORTARCINST *pi)
{
	if (pi == NOPORTARCINST)
	{
		db_thisapack->state = NULLNAME;
		return;
	}
	db_thisapack->state = SPECNAME;			/* then VARNAME */
	db_thisapack->vars = db_portavars;					/* SPECNAME */
	db_thisapack->proto = (INTBIG)&db_ptpi;				/* SPECNAME */
	db_thisapack->object = (INTBIG)pi;					/* SPECNAME */
	db_thisapack->specindex = 0;						/* SPECNAME */
	db_thisapack->numvar = &pi->numvar;					/* VARNAME */
	db_thisapack->firstvar = &pi->firstvar;				/* VARNAME */
	db_thisapack->varindex = 0;							/* VARNAME */
}

void db_initportexpinstlist(PORTEXPINST *pe)
{
	if (pe == NOPORTEXPINST)
	{
		db_thisapack->state = NULLNAME;
		return;
	}
	db_thisapack->state = SPECNAME;			/* then VARNAME */
	db_thisapack->vars = db_portevars;					/* SPECNAME */
	db_thisapack->proto = (INTBIG)&db_ptpe;				/* SPECNAME */
	db_thisapack->object = (INTBIG)pe;					/* SPECNAME */
	db_thisapack->specindex = 0;						/* SPECNAME */
	db_thisapack->numvar = &pe->numvar;					/* VARNAME */
	db_thisapack->firstvar = &pe->firstvar;				/* VARNAME */
	db_thisapack->varindex = 0;							/* VARNAME */
}

void db_initportprotolist(PORTPROTO *pp)
{
	if (pp == NOPORTPROTO)
	{
		db_thisapack->state = NULLNAME;
		return;
	}
	db_thisapack->state = SPECNAME;			/* then VARNAME */
	db_thisapack->vars = db_portprotovars;				/* SPECNAME */
	db_thisapack->proto = (INTBIG)&db_ptpp;				/* SPECNAME */
	db_thisapack->object = (INTBIG)pp;					/* SPECNAME */
	db_thisapack->specindex = 0;						/* SPECNAME */
	db_thisapack->numvar = &pp->numvar;					/* VARNAME */
	db_thisapack->firstvar = &pp->firstvar;				/* VARNAME */
	db_thisapack->varindex = 0;							/* VARNAME */
}

void db_initarcinstlist(ARCINST *ai)
{
	if (ai == NOARCINST)
	{
		db_thisapack->state = NULLNAME;
		return;
	}
	db_thisapack->state = SPECNAME;			/* then VARNAME */
	db_thisapack->vars = db_arcvars;					/* SPECNAME */
	db_thisapack->proto = (INTBIG)&db_ptai;				/* SPECNAME */
	db_thisapack->object = (INTBIG)ai;					/* SPECNAME */
	db_thisapack->specindex = 0;						/* SPECNAME */
	db_thisapack->numvar = &ai->numvar;					/* VARNAME */
	db_thisapack->firstvar = &ai->firstvar;				/* VARNAME */
	db_thisapack->varindex = 0;							/* VARNAME */
}

void db_initarcprotolist(ARCPROTO *ap)
{
	if (ap == NOARCPROTO)
	{
		db_thisapack->state = NULLNAME;
		return;
	}
	db_thisapack->state = SPECNAME;			/* then VARNAME */
	db_thisapack->vars = db_arcprotovars;				/* SPECNAME */
	db_thisapack->proto = (INTBIG)&db_ptap;				/* SPECNAME */
	db_thisapack->object = (INTBIG)ap;					/* SPECNAME */
	db_thisapack->specindex = 0;						/* SPECNAME */
	db_thisapack->numvar = &ap->numvar;					/* VARNAME */
	db_thisapack->firstvar = &ap->firstvar;				/* VARNAME */
	db_thisapack->varindex = 0;							/* VARNAME */
}

void db_initgeomlist(GEOM *geom)
{
	if (geom == NOGEOM)
	{
		db_thisapack->state = NULLNAME;
		return;
	}
	switch(geom->entrytype)
	{
		case OBJNODEINST: db_geomvars[1].type = VNODEINST|VCANTSET;   break;
		case OBJARCINST:  db_geomvars[1].type = VARCINST|VCANTSET;    break;
	}
	db_thisapack->state = SPECNAME;			/* then VARNAME */
	db_thisapack->vars = db_geomvars;					/* SPECNAME */
	db_thisapack->proto = (INTBIG)&db_ptgeom;			/* SPECNAME */
	db_thisapack->object = (INTBIG)geom;				/* SPECNAME */
	db_thisapack->specindex = 0;						/* SPECNAME */
	db_thisapack->numvar = &geom->numvar;				/* VARNAME */
	db_thisapack->firstvar = &geom->firstvar;			/* VARNAME */
	db_thisapack->varindex = 0;							/* VARNAME */
}

void db_initliblist(LIBRARY *lib)
{
	if (lib == NOLIBRARY)
	{
		db_thisapack->state = NULLNAME;
		return;
	}
	db_thisapack->state = NODENAME;			/* then SPECNAME, VARNAME */
	db_thisapack->nodeprotoname = lib->firstnodeproto;	/* NODENAME */
	db_thisapack->vars = db_libvars;					/* SPECNAME */
	db_thisapack->proto = (INTBIG)&db_ptlib;			/* SPECNAME */
	db_thisapack->object = (INTBIG)lib;					/* SPECNAME */
	db_thisapack->specindex = 0;						/* SPECNAME */
	db_thisapack->numvar = &lib->numvar;				/* VARNAME */
	db_thisapack->firstvar = &lib->firstvar;			/* VARNAME */
	db_thisapack->varindex = 0;							/* VARNAME */
}

void db_inittechlist(TECHNOLOGY *tech)
{
	if (tech == NOTECHNOLOGY)
	{
		db_thisapack->state = NULLNAME;
		return;
	}
	db_techvars[6].type = VGRAPHICS|VISARRAY|(tech->layercount<<VLENGTHSH)|VCANTSET;
	db_thisapack->state = ARCNAME;		/* then NODENAME, SPECNAME, VARNAME */
	db_thisapack->arcprotoname = tech->firstarcproto;	/* ARCNAME */
	db_thisapack->nodeprotoname = tech->firstnodeproto;	/* NODENAME */
	db_thisapack->vars = db_techvars;					/* SPECNAME */
	db_thisapack->proto = (INTBIG)&db_pttech;			/* SPECNAME */
	db_thisapack->object = (INTBIG)tech;				/* SPECNAME */
	db_thisapack->specindex = 0;						/* SPECNAME */
	db_thisapack->numvar = &tech->numvar;				/* VARNAME */
	db_thisapack->firstvar = &tech->firstvar;			/* VARNAME */
	db_thisapack->varindex = 0;							/* VARNAME */
}

void db_initaidlist(AIDENTRY *aid)
{
	if (aid == NOAID)
	{
		db_thisapack->state = NULLNAME;
		return;
	}
	db_thisapack->state = SPECNAME;			/* then VARNAME */
	db_thisapack->vars = db_aidvars;					/* SPECNAME */
	db_thisapack->proto = (INTBIG)&db_ptaid;			/* SPECNAME */
	db_thisapack->object = (INTBIG)aid;					/* SPECNAME */
	db_thisapack->specindex = 0;						/* SPECNAME */
	db_thisapack->numvar = &aid->numvar;				/* VARNAME */
	db_thisapack->firstvar = &aid->firstvar;			/* VARNAME */
	db_thisapack->varindex = 0;							/* VARNAME */
}

void db_initrtnodelist(RTNODE *rtn)
{
	if (rtn == NORTNODE)
	{
		db_thisapack->state = NULLNAME;
		return;
	}
	db_rtnvars[6].type = VISARRAY | (rtn->total<<VLENGTHSH) | VCANTSET | VCREF;
	if (rtn->flag == 0) db_rtnvars[6].type |= VRTNODE; else
		db_rtnvars[6].type |= VGEOM;
	db_thisapack->state = SPECNAME;			/* then VARNAME */
	db_thisapack->vars = db_rtnvars;					/* SPECNAME */
	db_thisapack->proto = (INTBIG)&db_ptrtn;			/* SPECNAME */
	db_thisapack->object = (INTBIG)rtn;					/* SPECNAME */
	db_thisapack->specindex = 0;						/* SPECNAME */
	db_thisapack->numvar = &rtn->numvar;				/* VARNAME */
	db_thisapack->firstvar = &rtn->firstvar;			/* VARNAME */
	db_thisapack->varindex = 0;							/* VARNAME */
}

void db_initnetworklist(NETWORK *net)
{
	if (net == NONETWORK)
	{
		db_thisapack->state = NULLNAME;
		return;
	}
	if (net->arccount < 1) db_netvars[0].type = VADDRESS|VCANTSET; else
		if (net->arccount == 1) db_netvars[0].type = VARCINST|VCANTSET; else
			db_netvars[0].type = VARCINST|VISARRAY|(net->arccount<<VLENGTHSH)|VCANTSET;
	if (net->signals <= 1) db_netvars[6].type = VADDRESS|VCANTSET; else
		db_netvars[6].type = VNETWORK|VISARRAY|(net->signals<<VLENGTHSH)|VCANTSET;
	db_thisapack->state = SPECNAME;			/* then VARNAME */
	db_thisapack->vars = db_netvars;					/* SPECNAME */
	db_thisapack->proto = (INTBIG)&db_ptnet;			/* SPECNAME */
	db_thisapack->object = (INTBIG)net;					/* SPECNAME */
	db_thisapack->specindex = 0;						/* SPECNAME */
	db_thisapack->numvar = &net->numvar;				/* VARNAME */
	db_thisapack->firstvar = &net->firstvar;			/* VARNAME */
	db_thisapack->varindex = 0;							/* VARNAME */
}

void db_initcelllist(CELL *cell)
{
	if (cell == NOCELL)
	{
		db_thisapack->state = NULLNAME;
		return;
	}
	db_thisapack->state = SPECNAME;			/* then VARNAME */
	db_thisapack->vars = db_cellvars;					/* SPECNAME */
	db_thisapack->proto = (INTBIG)&db_ptcell;			/* SPECNAME */
	db_thisapack->object = (INTBIG)cell;				/* SPECNAME */
	db_thisapack->specindex = 0;						/* SPECNAME */
	db_thisapack->numvar = &cell->numvar;				/* VARNAME */
	db_thisapack->firstvar = &cell->firstvar;			/* VARNAME */
	db_thisapack->varindex = 0;							/* VARNAME */
}

void db_initviewlist(VIEW *view)
{
	if (view == NOVIEW)
	{
		db_thisapack->state = NULLNAME;
		return;
	}
	db_thisapack->state = SPECNAME;			/* then VARNAME */
	db_thisapack->vars = db_viewvars;					/* SPECNAME */
	db_thisapack->proto = (INTBIG)&db_ptview;			/* SPECNAME */
	db_thisapack->object = (INTBIG)view;				/* SPECNAME */
	db_thisapack->specindex = 0;						/* SPECNAME */
	db_thisapack->numvar = &view->numvar;				/* VARNAME */
	db_thisapack->firstvar = &view->firstvar;			/* VARNAME */
	db_thisapack->varindex = 0;							/* VARNAME */
}

void db_initwindowlist(WINDOW *win)
{
	if (win == NOWINDOW)
	{
		db_thisapack->state = NULLNAME;
		return;
	}
	db_thisapack->state = SPECNAME;			/* then VARNAME */
	db_thisapack->vars = db_winvars;					/* SPECNAME */
	db_thisapack->proto = (INTBIG)&db_ptwin;			/* SPECNAME */
	db_thisapack->object = (INTBIG)win;					/* SPECNAME */
	db_thisapack->specindex = 0;						/* SPECNAME */
	db_thisapack->numvar = &win->numvar;				/* VARNAME */
	db_thisapack->firstvar = &win->firstvar;			/* VARNAME */
	db_thisapack->varindex = 0;							/* VARNAME */
}

void db_initgraphicslist(GRAPHICS *gra)
{
	if (gra == NOGRAPHICS)
	{
		db_thisapack->state = NULLNAME;
		return;
	}
	db_thisapack->state = SPECNAME;			/* then VARNAME */
	db_thisapack->vars = db_gravars;					/* SPECNAME */
	db_thisapack->proto = (INTBIG)&db_ptgra;			/* SPECNAME */
	db_thisapack->object = (INTBIG)gra;					/* SPECNAME */
	db_thisapack->specindex = 0;						/* SPECNAME */
	db_thisapack->numvar = &gra->numvar;				/* VARNAME */
	db_thisapack->firstvar = &gra->firstvar;			/* VARNAME */
	db_thisapack->varindex = 0;							/* VARNAME */
}

void db_initconstraintlist(CONSTRAINT *con)
{
	if (con == NOCONSTRAINT)
	{
		db_thisapack->state = NULLNAME;
		return;
	}
	db_thisapack->state = SPECNAME;			/* then VARNAME */
	db_thisapack->vars = db_convars;					/* SPECNAME */
	db_thisapack->proto = (INTBIG)&db_ptcon;			/* SPECNAME */
	db_thisapack->object = (INTBIG)con;					/* SPECNAME */
	db_thisapack->specindex = 0;						/* SPECNAME */
	db_thisapack->numvar = &con->numvar;				/* VARNAME */
	db_thisapack->firstvar = &con->firstvar;			/* VARNAME */
	db_thisapack->varindex = 0;							/* VARNAME */
}

/************************* HELPER ROUTINES *************************/

/*
 * routine to convert a variable name in "name" to a key and return it.
 * If the name is not in the database, the routine returns a negative value
 * that is one less than the negative table index entry.
 */
INTSML db_getkey(char *name)
{
	REGISTER INTSML hi, lo, med, lastmed, i;

	/* binary search name space for the variable name */
	i = lo = 0;   hi = lastmed = el_numnames;
	for(;;)
	{
		/* find mid-point: quit if done */
		med = (hi + lo) / 2;
		if (med == lastmed) break;
		lastmed = med;

		/* test the entry: return the key if a match */
		i = namesame(name, el_namespace[med]);
		if (i == 0) return(med);

		/* decide which way to search in list */
		if (i < 0) hi = med; else lo = med;
	}

	/* create a new position: adjust for position location */
	if (i > 0) med++;
	return(-med-1);
}

/*
 * routine to evaluate variable "var" and replace it with its true value if it is code
 */
VARIABLE *db_evalvar(VARIABLE *var)
{
	REGISTER INTBIG language;
	static VARIABLE retvar;

	if (var == NOVARIABLE) return(var);
	language = var->type & (VCODE1|VCODE2);
	if (language == 0) return(var);
	retvar.key = var->key;
	retvar.type = var->type & ~(VCODE1|VCODE2);
	retvar.textdescript = var->textdescript;
	if (doquerry((char *)var->addr, language, retvar.type, &retvar.addr) != 0)
		return(var);
	return(&retvar);
}

/*
 * routine to determine the appropriate memory cluster to use for the
 * variable whose address is "addr" and type is "type".
 */
CLUSTER *db_whichcluster(INTBIG addr, INTBIG type)
{
	REGISTER GEOM *geom;
	REGISTER NODEPROTO *np;
	extern AIDENTRY *us_aid;

	switch (type&VTYPE)
	{
		case VNODEINST: return(((NODEINST *)addr)->parent->cell->cluster);
		case VNODEPROTO:
			np = (NODEPROTO *)addr;
			if (np->index == 0) return(np->cell->cluster);
			return(np->tech->cluster);
		case VPORTARCINST:
			return(((PORTARCINST *)addr)->conarcinst->parent->cell->cluster);
		case VPORTEXPINST:
			return(((PORTEXPINST *)addr)->exportproto->parent->cell->cluster);
		case VPORTPROTO:
			np = ((PORTPROTO *)addr)->parent;
			if (np->index == 0) return(np->cell->cluster);
			return(np->tech->cluster);
		case VARCINST: return(((ARCINST *)addr)->parent->cell->cluster);
		case VARCPROTO: return(((ARCPROTO *)addr)->tech->cluster);
		case VGEOM: geom = (GEOM *)addr;
			if (geom->entrytype == OBJNODEINST)
				return(geom->entryaddr.ni->parent->cell->cluster);
			return(geom->entryaddr.ai->parent->cell->cluster);
		case VTECHNOLOGY: return(((TECHNOLOGY *)addr)->cluster);
		case VAID: return(((AIDENTRY *)addr)->cluster);
		case VNETWORK: return(((NETWORK *)addr)->parent->cell->cluster);
		case VCELL: return(((CELL *)addr)->cluster);
		case VWINDOW: return(us_aid->cluster);
	}
	return(db_cluster);
}

/*
 * routine to return the name of the variable whose key is "key" and whose
 * type is "type"
 */
char *changedvariablename(INTBIG type, INTBIG key, INTBIG subtype)
{
	if ((subtype&VCREF) == 0)
	{
		if (key == -1) return("NULL");
		return(makename(key));
	}

	switch (type&VTYPE)
	{
		case VNODEINST:    return(db_nodevars[key].name);
		case VNODEPROTO:   return(db_nodeprotovars[key].name);
		case VPORTARCINST: return(db_portavars[key].name);
		case VPORTEXPINST: return(db_portevars[key].name);
		case VPORTPROTO:   return(db_portprotovars[key].name);
		case VARCINST:     return(db_arcvars[key].name);
		case VARCPROTO:    return(db_arcprotovars[key].name);
		case VGEOM:        return(db_geomvars[key].name);
		case VRTNODE:      return(db_rtnvars[key].name);
		case VLIBRARY:     return(db_libvars[key].name);
		case VTECHNOLOGY:  return(db_techvars[key].name);
		case VAID:         return(db_aidvars[key].name);
		case VVIEW:        return(db_viewvars[key].name);
		case VCELL:        return(db_cellvars[key].name);
		case VNETWORK:     return(db_netvars[key].name);
		case VWINDOW:      return(db_winvars[key].name);
		case VGRAPHICS:    return(db_gravars[key].name);
		case VCONSTRAINT:  return(db_convars[key].name);
	}
	return("NULL");
}

/*
 * routine to determine the appropriate facet associated with the
 * variable whose address is "addr" and type is "type".
 */
NODEPROTO *db_whichnodeproto(INTBIG addr, INTBIG type)
{
	REGISTER GEOM *geom;
	REGISTER NODEPROTO *np;

	switch (type&VTYPE)
	{
		case VNODEINST: return(((NODEINST *)addr)->parent);
		case VNODEPROTO:
			np = (NODEPROTO *)addr;
			if (np->index == 0) return(np); else return(NONODEPROTO);
		case VPORTARCINST: return(((PORTARCINST *)addr)->conarcinst->parent);
		case VPORTEXPINST: return(((PORTEXPINST *)addr)->exportproto->parent);
		case VPORTPROTO: return(((PORTPROTO *)addr)->parent);
		case VARCINST: return(((ARCINST *)addr)->parent);
		case VGEOM: geom = (GEOM *)addr;
			if (geom->entrytype == OBJNODEINST) return(geom->entryaddr.ni->parent);
			return(geom->entryaddr.ai->parent);
		case VNETWORK: return(((NETWORK *)addr)->parent);
	}
	return(NONODEPROTO);
}

/*
 * routine to determine the appropriate library associated with the
 * variable whose address is "addr" and type is "type".
 */
LIBRARY *db_whichlibrary(INTBIG addr, INTBIG type)
{
	REGISTER GEOM *geom;
	REGISTER NODEPROTO *np;

	switch (type&VTYPE)
	{
		case VNODEINST: return(((NODEINST *)addr)->parent->cell->lib);
		case VNODEPROTO:
			np = (NODEPROTO *)addr;
			if (np->index == 0) return(np->cell->lib);
			return(NOLIBRARY);
		case VPORTARCINST:
			return(((PORTARCINST *)addr)->conarcinst->parent->cell->lib);
		case VPORTEXPINST:
			return(((PORTEXPINST *)addr)->exportproto->parent->cell->lib);
		case VPORTPROTO:
			np = ((PORTPROTO *)addr)->parent;
			if (np->index == 0) return(np->cell->lib);
			return(NOLIBRARY);
		case VARCINST: return(((ARCINST *)addr)->parent->cell->lib);
		case VGEOM: geom = (GEOM *)addr;
			if (geom->entrytype == OBJNODEINST)
				return(geom->entryaddr.ni->parent->cell->lib);
			return(geom->entryaddr.ai->parent->cell->lib);
		case VLIBRARY: return((LIBRARY *)addr);
		case VNETWORK: return(((NETWORK *)addr)->parent->cell->lib);
		case VCELL: return(((CELL *)addr)->lib);
	}
	return(NOLIBRARY);
}

/*
 * routine to adjust the coordinate values in "poly" to account for the
 * display offset in the type field "textdescription".  Assumes that the
 * initial values are the center of the object.  The object on which this
 * text resides is in "geom".  Routine also sets the style and the font of
 * text message to use with this coordinate.
 */
void adjustdisoffset(GEOM *geom, POLYGON *poly, INTBIG textdescription)
{
	REGISTER INTBIG distx, disty, lambda;
	INTBIG plx, phx, ply, phy;
	REGISTER INTSML i;
	XARRAY trans;

	distx = (textdescription&VTXOFF) >> VTXOFFSH;
	if ((textdescription&VTXOFFNEG) != 0) distx = -distx;
	lambda = figurelambda(geom);
	distx = distx * lambda / 4;
	disty = (textdescription&VTYOFF) >> VTYOFFSH;
	if ((textdescription&VTYOFFNEG) != 0) disty = -disty;
	disty = disty * lambda / 4;
	if ((textdescription&VTPOSITION) == VTPOSBOXED)
	{
		getbbox(poly, &plx, &phx, &ply, &phy);
		if (distx > 0) plx += distx; else
			if (distx < 0) phx -= distx;
		if (disty > 0) ply += disty; else
			if (disty < 0) phy -= disty;
		makerectpoly(plx, phx, ply, phy, poly);
	} else
	{
		/* just shift the polygon */
		for(i=0; i<poly->count; i++)
		{
			poly->xv[i] += distx;
			poly->yv[i] += disty;
		}
	}

	/* determine the text message style */
	switch (textdescription&VTPOSITION)
	{
		case VTPOSCENT:      poly->style = TEXTCENT;      break;
		case VTPOSBOXED:     poly->style = TEXTBOX;       break;
		case VTPOSUP:        poly->style = TEXTBOT;       break;
		case VTPOSDOWN:      poly->style = TEXTTOP;       break;
		case VTPOSLEFT:      poly->style = TEXTRIGHT;     break;
		case VTPOSRIGHT:     poly->style = TEXTLEFT;      break;
		case VTPOSUPLEFT:    poly->style = TEXTBOTRIGHT;  break;
		case VTPOSUPRIGHT:   poly->style = TEXTBOTLEFT;   break;
		case VTPOSDOWNLEFT:  poly->style = TEXTTOPRIGHT;  break;
		case VTPOSDOWNRIGHT: poly->style = TEXTTOPLEFT;   break;
	}
	if (geom->entrytype == OBJNODEINST)
	{
		makeangle(geom->entryaddr.ni->rotation, geom->entryaddr.ni->transpose, trans);
		poly->style = rotatelabel(poly->style, trans);
	}
	poly->font = (textdescription & VTSIZE) >> VTSIZESH;
}

/*
 * internal routine to set entry "med" in the list of variables at "*firstvar"
 * with "*numvar" entries (this entry has the key "key").  If "med" is negative
 * then the entry does not exist in the list and must be created.  The entry is
 * set to the address "newaddr" and type "newtype".  Memory is allocated from
 * cluster "cluster".  Returns nonzero if there is an error.
 */
INTSML db_setvalkey(INTSML *numvar, VARIABLE **firstvar, INTSML med, INTBIG key,
	INTBIG newaddr, INTBIG newtype, CLUSTER *cluster)
{
	REGISTER VARIABLE *newv, *var;
	REGISTER INTSML i;

	/* if there is no existing variable, create it */
	if (med < 0)
	{
		/* get the entry position in the list */
		med = -med - 1;

		/* allocate space for new list */
		newv = (VARIABLE *)emalloc(((*numvar+1)*(sizeof (VARIABLE))), cluster);
		if (newv == 0) return(1);

		/* copy old list up to the new entry */
		for(i=0; i<med; i++)
		{
			newv[i].key  = (*firstvar)[i].key;
			newv[i].type = (*firstvar)[i].type;
			newv[i].textdescript = (*firstvar)[i].textdescript;
			newv[i].addr = (*firstvar)[i].addr;
		}

		/* add the new entry */
		newv[med].key = key;
		newv[med].type = VUNKNOWN|VDONTSAVE;
		newv[med].textdescript = TXTSMALL << VTSIZESH;
		newv[med].addr = 0;
		var = &newv[med];

		/* copy old list after the new entry */
		for(i = med; i < *numvar; i++)
		{
			newv[i+1].key  = (*firstvar)[i].key;
			newv[i+1].type = (*firstvar)[i].type;
			newv[i+1].textdescript = (*firstvar)[i].textdescript;
			newv[i+1].addr = (*firstvar)[i].addr;
		}

		/* clean-up */
		if (*numvar != 0) efree((char *)*firstvar);
		*firstvar = newv;
		(*numvar)++;
	} else var = &(*firstvar)[med];

	/* set the variable */
	if (db_setvalvar(var, newaddr, newtype, cluster) != 0) return(1);

	return(0);
}

/*
 * routine to set the value of the variable "var" to the value "newaddr" and
 * type "newtype".  Memory is allocated from cluster "cluster".  Returns
 * nonzero upon error.
 */
INTSML db_setvalvar(VARIABLE *var, INTBIG newaddr, INTBIG newtype,
	CLUSTER *cluster)
{
	REGISTER INTBIG i, len;
	REGISTER INTSML datasize;
	INTBIG longval;
	REGISTER char *inaddr;

	/* if the variable or key is not valid then ignore this */
	if (var == NOVARIABLE)
	{
		ttyputerr("No valid variable");
		return(1);
	}
	if ((var->type&VCREF) != 0)
	{
		/* setting a fixed attribute on an object */
		if ((newtype&VISARRAY) != 0)
		{
			ttyputmsg("Cannot set array in C structure");
			return(1);
		}
		if ((var->type&(VTYPE|VISARRAY)) != (newtype&(VTYPE|VISARRAY)))
		{
			ttyputerr("Type mismatch");
			return(1);
		}
		if (db_setval((char *)&newaddr, (char *)db_realaddress, newtype, cluster) != 0) return(1);
		return(0);
	}

	/* change the variable type */
	var->type = newtype;

	/* allocate and fill space on the new variables */
	if ((newtype&(VCODE1|VCODE2)) != 0)
	{
		if (db_setval((char *)&newaddr, (char *)&var->addr, VSTRING, cluster) != 0) return(1);
	} else if ((newtype&VISARRAY) != 0)
	{
		var->addr = newaddr;
		len = getlength(var);
		datasize = db_getdatasize(newtype);
		if ((newtype&VLENGTH) == 0)
		{
			var->addr = (INTBIG)emalloc((len+1)*datasize, cluster);
			if (var->addr == 0) return(1);
			for(i=0; i<datasize; i++)
				((char *)var->addr)[len*datasize+i] = -1;
		} else
		{
			var->addr = (INTBIG)emalloc(len*datasize, cluster);
			if (var->addr == 0) return(1);
		}
		for(i=0; i<len; i++)
		{
			if ((newtype&VTYPE) == VSHORT)
			{
				longval = ((INTSML *)newaddr)[i];
				if (db_setval((char *)&longval, &((char *)var->addr)[i*datasize], newtype,
					cluster) != 0) return(1);
			} else
			{
				inaddr = (char *)(newaddr + i*datasize);
				if (db_setval(inaddr, &((char *)var->addr)[i*datasize], newtype, cluster) != 0)
					return(1);
			}
		}
	} else
		if (db_setval((char *)&newaddr, (char *)&var->addr, newtype, cluster)
			!= 0) return(1);

	return(0);
}

/*
 * routine to get entry "index" of array variable "var" and place it in "value".
 * Returns nonzero upon error.
 */
INTSML db_getindvar(VARIABLE *var, INTBIG index, INTBIG *value)
{
	INTBIG type, len, datasize, *loc;

	/* if the variable or key is not valid then ignore this */
	if (var == NOVARIABLE) return(1);

	/* if the variable is not an array, quit now */
	type = var->type;
	if ((type&VISARRAY) == 0) return(1);

	/* ensure that the index is within the range of the array */
	len = getlength(var);
	if (index < 0) index = 0;
	if (index >= len) index = len - 1;

	/* determine the size of the array entries */
	datasize = db_getdatasize(type);

	/* set the entry */
	if ((type&VCREF) != 0)
	{
		/* get the address of the old array entry (fixed attributes) */
		loc = (INTBIG *)(db_realaddress + index*datasize);
	} else
	{
		/* get the address of the old array entry (variable attributes) */
		loc = (INTBIG *)(&((char *)var->addr)[index*datasize]);
	}

	/* set an arbitrary attribute on an object */
	switch (datasize)
	{
		case 1:  *value = *((char *)loc);    break;
		case 2:  *value = *((INTSML *)loc);  break;
		default: *value = *loc;              break;
	}
	return(0);
}

/*
 * routine to set entry "index" of array variable "var" (which is on object
 * "objaddr" of type "objtype") to the value "newaddr".  Returns nonzero
 * upon error.
 */
INTSML db_setindvar(INTBIG objaddr, INTBIG objtype, VARIABLE *var, INTBIG index,
	INTBIG newaddr)
{
	REGISTER INTSML datasize;
	REGISTER INTBIG len, *loc, oldvalue, type;

	/* if the variable or key is not valid then ignore this */
	if (var == NOVARIABLE) return(1);

	/* if the variable is not an array, quit now */
	type = var->type;
	if ((type&VISARRAY) == 0) return(1);

	/* ensure that the index is within the range of the array */
	len = getlength(var);
	if (index < 0) index = 0;
	if (index >= len) index = len - 1;

	/* determine the size of the array entries */
	datasize = db_getdatasize(type);

	/* set the entry */
	if ((type&VCREF) != 0)
	{
		/* get the address of the old array entry (fixed attributes) */
		loc = (INTBIG *)(db_realaddress + index*datasize);
	} else
	{
		/* get the address of the old array entry (variable attributes) */
		loc = (INTBIG *)(&((char *)var->addr)[index*datasize]);
	}

	/* set an arbitrary attribute on an object */
	switch (datasize)
	{
		case 1:  oldvalue = *((char *)loc);    break;
		case 2:  oldvalue = *((INTSML *)loc);  break;
		default: oldvalue = *loc;              break;
	}
	if (db_setval((char *)&newaddr, (char *)loc, type,
		db_whichcluster(objaddr, objtype)) != 0) return(1);

	if (db_dovarchangequietly == 0)
	{
		/* tell constraint system about modified variable */
		(*el_curconstraint->modifyvariable)(objaddr, objtype, var->key, type, index, oldvalue);

		/* mark a change */
		(void)db_change((INTBIG)objaddr, VARIABLEMOD, objtype, var->key, type, index, oldvalue, 0);
	}

	return(0);
}

/*
 * routine to insert the value "newaddr" before entry "index" of array
 * variable "var" (which is on object "objaddr" of type "objtype").
 * Returns nonzero upon error.
 */
INTSML db_insindvar(INTBIG objaddr, INTBIG objtype, VARIABLE *var, INTBIG index,
	INTBIG newaddr)
{
	REGISTER INTBIG i, j, len, truelen, type, *newdata;
	REGISTER INTSML datasize;
	CLUSTER *cluster;

	/* if the variable or key is not valid then ignore this */
	if (var == NOVARIABLE) return(1);

	/* if the variable is not an array, quit now */
	type = var->type;
	if ((type&VISARRAY) == 0) return(1);

	/* ensure that the index is within the range of the array */
	truelen = len = getlength(var);
	if (index < 0) index = 0;
	if (index > len) index = len;

	/* if this is a variable-length array, include the terminator */
	if ((var->type&VLENGTH) == 0) len++;

	/* determine the size of the array entries */
	datasize = db_getdatasize(type);

	/* allocate a new array */
	cluster = db_whichcluster(objaddr, objtype);
	newdata = (INTBIG *)emalloc(datasize * (len+1), cluster);
	if (newdata == 0) return(1);

	/* copy, shifting */
	for(i=j=0; i<len; i++)
	{
		if (i == index) j++;
		switch (type&VTYPE)
		{
			case VCHAR:
				((char *)newdata)[j] = ((char *)var->addr)[i];
				break;
			case VDOUBLE:
				((double *)newdata)[j] = ((double *)var->addr)[i];
				break;
			case VSHORT:
				((INTSML *)newdata)[j] = ((INTSML *)var->addr)[i];
				break;
			default:
				((INTBIG *)newdata)[j] = ((INTBIG *)var->addr)[i];
				break;
		}
		j++;
	}
	if (db_setval((char *)&newaddr, &((char *)newdata)[index*datasize], type, cluster) != 0)
		return(1);
	efree((char *)var->addr);
	var->addr = (INTBIG)newdata;

	/* bump the array size */
	if ((var->type&VLENGTH) != 0)
		var->type = (var->type & ~VLENGTH) | ((truelen+1) << VLENGTHSH);

	if (db_dovarchangequietly == 0)
	{
		/* tell constraint system about inserted variable */
		(*el_curconstraint->insertvariable)(objaddr, objtype, var->key, index);

		/* mark a change */
		(void)db_change((INTBIG)objaddr, VARIABLEINS, objtype, var->key, type, index, 0, 0);
	}

	return(0);
}

/*
 * routine to delete entry "index" of array variable "var" (which is on object
 * "objaddr" of type "objtype").  Returns nonzero upon error.
 */
INTSML db_delindvar(INTBIG objaddr, INTBIG objtype, VARIABLE *var, INTBIG index)
{
	REGISTER INTBIG i, len, truelen, *loc, oldvalue, type;
	REGISTER INTSML datasize;

	/* if the variable or key is not valid then ignore this */
	if (var == NOVARIABLE) return(1);

	/* if the variable is not an array, quit now */
	type = var->type;
	if ((type&VISARRAY) == 0) return(1);

	/* ensure that the index is within the range of the array */
	truelen = len = getlength(var);

	/* can only delete valid line number */
	if (index < 0 || index >= len) return(1);

	/* cannot delete last entry */
	if (truelen == 1) return(1);

	/* if this is a variable-length array, include the terminator */
	if ((var->type&VLENGTH) == 0) len++;

	/* determine the size of the array entries */
	datasize = db_getdatasize(type);

	/* get the address of the old array entry */
	loc = (INTBIG *)(&((char *)var->addr)[index*datasize]);
	switch (datasize)
	{
		case 1:  oldvalue = *((char *)loc);   break;
		case 2:  oldvalue = *((INTSML *)loc);  break;
		default: oldvalue = *loc;             break;
	}

	/* shift the data */
	for(i=index; i<len-1; i++)
	{
		switch (type&VTYPE)
		{
			case VCHAR:
				((char *)var->addr)[i] = ((char *)var->addr)[i+1];
				break;
			case VDOUBLE:
				((double *)var->addr)[i] = ((double *)var->addr)[i+1];
				break;
			case VSHORT:
				((INTSML *)var->addr)[i] = ((INTSML *)var->addr)[i+1];
				break;
			default:
				((INTBIG *)var->addr)[i] = ((INTBIG *)var->addr)[i+1];
				break;
		}
	}

	/* decrement the array size */
	if ((var->type&VLENGTH) != 0)
		var->type = (var->type & ~VLENGTH) | ((truelen-1) << VLENGTHSH);

	if (db_dovarchangequietly == 0)
	{
		/* tell constraint system about deleted variable */
		(*el_curconstraint->deletevariable)(objaddr, objtype, var->key, index, oldvalue);

		/* mark a change */
		(void)db_change((INTBIG)objaddr, VARIABLEDEL, objtype, var->key, type, index, oldvalue, 0);
	}

	return(0);
}

/*
 * routine to set an individual variable entry at address "to" to the value
 * at address "from".  This type is of type "type" and if allocatable, is
 * filled from cluster "cluster".
 */
INTSML db_setval(char *from, char *to, INTBIG type, CLUSTER *cluster)
{
	/* character sized variable */
	if ((type&VTYPE) == VCHAR)
	{
		*to = *from;
		return(0);
	}

	/* double sized variable */
	if ((type&VTYPE) == VDOUBLE)
	{
		((double *)to)[0] = ((double *)from)[0];
		return(0);
	}

	/* short sized variable */
	if ((type&VTYPE) == VSHORT)
	{
		((INTSML *)to)[0] = ((INTBIG *)from)[0];
		return(0);
	}

	/* integer sized variable */
	if ((type&(VCODE1|VCODE2)) == 0 && (type&VTYPE) != VSTRING)
	{
		((INTBIG *)to)[0] = ((INTBIG *)from)[0];
		return(0);
	}

	/* string or code variable */
	return(allocstring((char **)to, ((char **)from)[0], cluster));
}

/*
 * routine to erase a list of variables when the object is deleted
 */
void db_freevars(VARIABLE **firstvar, INTSML *numvar)
{
	REGISTER INTSML i;
	REGISTER VARIABLE *var;

	if (*numvar == 0) return;
	for(i = 0; i < *numvar; i++)
	{
		var = &(*firstvar)[i];
		db_freevar(var->addr, var->type);
	}
	efree((char *)*firstvar);
	*numvar = 0;
}

/*
 * routine to free any memory allocated to the variable whose address is "addr"
 * and whose type is "type"
 */
void db_freevar(INTBIG addr, INTBIG type)
{
	REGISTER INTBIG i, len;

	/* determine whether individual elements are allocated */
	if ((type&(VCODE1|VCODE2)) != 0 || (type&VTYPE) == VSTRING)
	{
		/* the old variable was allocated */
		if ((type&VISARRAY) != 0)
		{
			len = (type&VLENGTH) >> VLENGTHSH;
			if (len == 0)
			{
				for(i=0; ((INTBIG *)addr)[i] != -1; i++)
					efree((char *)((INTBIG *)addr)[i]);
			} else for(i=0; i<len; i++) efree((char *)((INTBIG *)addr)[i]);
		} else efree((char *)addr);
	}
	if ((type&VISARRAY) != 0) efree((char *)addr);
}

/*
 * routine to do a binary search on the variables in "numvar" and "firstvar"
 * for an entry with the key "key".  If found, its index is returned,
 * otherwise a negative value is returned that, when negated, is the position
 * in the list BEFORE which the key should be inserted (i.e. the value -1
 * means that the key belongs in the first entry and the value -3 means that
 * the key belongs after the second entry).
 */
INTSML db_binarysearch(INTSML numvar, VARIABLE *firstvar, INTBIG key)
{
	REGISTER INTSML hi, lo, med, lastmed;
	REGISTER INTBIG i;

	if (numvar == 0) return(-1);
	lo = 0;   hi = lastmed = numvar;
	for(;;)
	{
		/* find mid-point: quit if done */
		med = (hi + lo) / 2;
		if (med == lastmed) break;
		lastmed = med;

		/* test the entry: return the value if a match */
		i = firstvar[med].key;
		if (i == key) return(med);

		/* decide which way to search in list */
		if (i > key) hi = med; else lo = med;
	}
	if (i < key) med++;
	return(-med-1);
}

void db_renamevar(INTSML numvar, VARIABLE *firstvar, INTBIG oldkey,
	INTBIG newkey)
{
	REGISTER INTSML i, j, k;
	REGISTER INTBIG oldaddr, oldtype, olddescript;
	REGISTER VARIABLE *var;

	if (numvar == 0) return;

	for(i=0; i<numvar; i++)
	{
		var = &firstvar[i];
		if (var->key != oldkey) continue;

		/* save the old information about this variable */
		oldtype = var->type;
		olddescript = var->textdescript;
		oldaddr = var->addr;

		/* compact out the old key */
		for(j=i+1; j<numvar; j++)
		{
			firstvar[j-1].key  = firstvar[j].key;
			firstvar[j-1].type = firstvar[j].type;
			firstvar[j-1].textdescript = firstvar[j].textdescript;
			firstvar[j-1].addr = firstvar[j].addr;
		}

		/* now insert the new key */
		for(j=0; j<numvar-1; j++) if (newkey < firstvar[j].key) break;
		for(k=numvar-1; k>j; k--)
		{
			firstvar[k].key  = firstvar[k-1].key;
			firstvar[k].type = firstvar[k-1].type;
			firstvar[k].textdescript = firstvar[k-1].textdescript;
			firstvar[k].addr = firstvar[k-1].addr;
		}

		/* insert the renamed variable in the right place */
		firstvar[j].key = newkey;
		firstvar[j].type = oldtype;
		firstvar[j].textdescript = olddescript;
		firstvar[j].addr = oldaddr;
	}
}

/*
 * routine to fill the parameters "fir" and "num" with the "firstvar" and
 * "numvar" attributes on object "addr", type "type".  Returns zero if
 * successful, nonzero if the object has no "firstvar/numvar".
 */
INTSML db_getvarptr(INTBIG addr, INTBIG type, VARIABLE ***fir, INTSML **num)
{
	REGISTER NODEINST *ni;
	REGISTER NODEPROTO *np;
	REGISTER PORTARCINST *pi;
	REGISTER PORTEXPINST *pe;
	REGISTER PORTPROTO *pp;
	REGISTER ARCINST *ai;
	REGISTER ARCPROTO *ap;
	REGISTER GEOM *g;
	REGISTER LIBRARY *lib;
	REGISTER TECHNOLOGY *tech;
	REGISTER AIDENTRY *aid;
	REGISTER RTNODE *rtn;
	REGISTER NETWORK *net;
	REGISTER CELL *c;
	REGISTER VIEW *v;

	switch (type&VTYPE)
	{
		case VNODEINST:
			ni = (NODEINST *)addr;
			*fir = &ni->firstvar;    *num = &ni->numvar;    break;
		case VNODEPROTO:
			np = (NODEPROTO *)addr;
			*fir = &np->firstvar;    *num = &np->numvar;    break;
		case VPORTARCINST:
			pi = (PORTARCINST *)addr;
			*fir = &pi->firstvar;    *num = &pi->numvar;    break;
		case VPORTEXPINST:
			pe = (PORTEXPINST *)addr;
			*fir = &pe->firstvar;    *num = &pe->numvar;    break;
		case VPORTPROTO:
			pp = (PORTPROTO *)addr;
			*fir = &pp->firstvar;    *num = &pp->numvar;    break;
		case VARCINST:
			ai = (ARCINST *)addr;
			*fir = &ai->firstvar;    *num = &ai->numvar;    break;
		case VARCPROTO:
			ap = (ARCPROTO *)addr;
			*fir = &ap->firstvar;    *num = &ap->numvar;    break;
		case VGEOM:
			g = (GEOM *)addr;
			*fir = &g->firstvar;     *num = &g->numvar;     break;
		case VLIBRARY:
			lib = (LIBRARY *)addr;
			*fir = &lib->firstvar;   *num = &lib->numvar;   break;
		case VTECHNOLOGY:
			tech = (TECHNOLOGY *)addr;
			*fir = &tech->firstvar;  *num = &tech->numvar;  break;
		case VAID:
			aid = (AIDENTRY *)addr;
			*fir = &aid->firstvar;   *num = &aid->numvar;   break;
		case VRTNODE:
			rtn = (RTNODE *)addr;
			*fir = &rtn->firstvar;   *num = &rtn->numvar;   break;
		case VNETWORK:
			net = (NETWORK *)addr;
			*fir = &net->firstvar;   *num = &net->numvar;   break;
		case VCELL:
			c = (CELL *)addr;
			*fir = &c->firstvar;     *num = &c->numvar;     break;
		case VVIEW:
			v = (VIEW *)addr;
			*fir = &v->firstvar;     *num = &v->numvar;     break;
		default: return(1);
	}
	return(0);
}

INTSML db_getdatasize(INTBIG type)
{
	if ((type&VTYPE) == VCHAR) return(1);
	if ((type&VTYPE) == VSHORT) return(SIZEOFINTSML);
	if ((type&VTYPE) == VDOUBLE) return(SIZEOFINTBIG * 2);
	return(SIZEOFINTBIG);
}

char *db_describetype(INTBIG type)
{
	(void)initinfstr();
	switch (type&VTYPE)
	{
		case VINTEGER:     (void)addstringtoinfstr("Integer");       break;
		case VADDRESS:     (void)addstringtoinfstr("Address");       break;
		case VCHAR:        (void)addstringtoinfstr("Character");     break;
		case VSTRING:      (void)addstringtoinfstr("String");        break;
		case VFLOAT:       (void)addstringtoinfstr("Float");         break;
		case VDOUBLE:      (void)addstringtoinfstr("Double");        break;
		case VNODEINST:    (void)addstringtoinfstr("NodeInst");      break;
		case VNODEPROTO:   (void)addstringtoinfstr("NodeProto");     break;
		case VPORTARCINST: (void)addstringtoinfstr("PortArcInst");   break;
		case VPORTEXPINST: (void)addstringtoinfstr("PortExpInst");   break;
		case VPORTPROTO:   (void)addstringtoinfstr("PortProto");     break;
		case VARCINST:     (void)addstringtoinfstr("ArcInst");       break;
		case VARCPROTO:    (void)addstringtoinfstr("ArcProto");      break;
		case VGEOM:        (void)addstringtoinfstr("Geom");          break;
		case VLIBRARY:     (void)addstringtoinfstr("Library");       break;
		case VTECHNOLOGY:  (void)addstringtoinfstr("Technology");    break;
		case VAID:         (void)addstringtoinfstr("Aid");           break;
		case VRTNODE:      (void)addstringtoinfstr("RTNode");        break;
		case VFRACT:       (void)addstringtoinfstr("Fixed-Point");   break;
		case VNETWORK:     (void)addstringtoinfstr("Network");       break;
		case VCELL:        (void)addstringtoinfstr("Cell");          break;
		case VVIEW:        (void)addstringtoinfstr("View");          break;
		case VWINDOW:      (void)addstringtoinfstr("Window");        break;
		case VGRAPHICS:    (void)addstringtoinfstr("Graphics");      break;
		case VSHORT:       (void)addstringtoinfstr("Short");         break;
		case VCONSTRAINT:  (void)addstringtoinfstr("Constraint");    break;
		case VGENERAL:     (void)addstringtoinfstr("General");       break;
		default:           (void)addstringtoinfstr("Unknown");       break;
	}
	if ((type&VISARRAY) != 0) (void)addstringtoinfstr(" array");
	return(returninfstr());
}
