/*
 * Electric(tm) VLSI Design System
 *
 * File: terminal.c
 * General status terminal output handler
 * Written by: Steven M. Rubin, Electric Editor Incorporated
 *
 * Copyright (c) 1998 Electric Editor Incorporated.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Electric Editor Incorporated
 * 23470 Sunset Drive, Suite 108
 * Los Gatos, California 95033
 * support@electriceditor.com
 */

#include "global.h"
#include "usr.h"
#include "edialogs.h"

INTSML   tty_mute;					/* nonzero to supress unimportant messages */
FILE	*tty_audit;					/* disk channel for saving terminal output */
INTSML	 tty_pamyItemHit;			/* item hit during parsing */
INTSML	 tty_ttywriteseparator;		/* nonzero to write separator before next text */
INTSML   tty_ttyinited;

/* general ttygetparam() Dialog */
extern DIALOG tty_listdialog;

/* ttygetline() Dialog */
DIALOGITEM tty_inputdialogitems[] =
{
 /*  1 */ {0, {96,200,120,264}, BUTTON, "OK"},
 /*  2 */ {0, {96,24,120,88}, BUTTON, "Cancel"},
 /*  3 */ {0, {58,38,74,246}, EDITTEXT, ""},
 /*  4 */ {0, {3,9,51,281}, MESSAGE, ""}
};
DIALOG tty_inputdialog = {{50,75,188,371}, "", 4, tty_inputdialogitems};

/* Full Input Dialog */
DIALOGITEM tty_fulldialogitems[] =
{
 /*  1 */ {0, {160,328,184,392}, BUTTON, "OK"},
 /*  2 */ {0, {104,328,128,392}, BUTTON, "Cancel"},
 /*  3 */ {0, {24,8,56,408}, EDITTEXT, ""},
 /*  4 */ {0, {3,8,19,408}, MESSAGE, ""},
 /*  5 */ {0, {88,8,212,294}, SCROLL, ""},
 /*  6 */ {0, {72,56,88,270}, MESSAGE, "Type '?' for a list of options"}
};
DIALOG tty_fulldialog = {{50,75,267,493}, "Command Completion", 6, tty_fulldialogitems};

/* prototypes for local routines */
INTSML tty_inmacchar(COMCOMP**, INTSML*);
void tty_outmacchar(char*);
void tty_erasemacchar(void);
void tty_killmacchar(void);

/************************* INITIALIZATION/TERMINATION *************************/

/*
 * initialize the text terminal and saves state information
 * so that we can restore its characteristics later.  This
 * routine is called once at the start of the program.
 */
void tty_init(void)
{
	/* allocate character arrays that are as wide as the terminal */
	us_pattyline = us_paambiguous = 0;
	tty_mute = 0;
	tty_audit = NULL;
	tty_ttywriteseparator = 0;
	tty_ttyinited = 1;
}

/*
 * close the terminal (called once at the end of the program)
 */
void tty_close(void)
{
	/* close up any audit file */
	if (tty_audit != NULL) xclose(tty_audit);

	if (us_pattyline != 0) efree(us_pattyline);
	if (us_paambiguous != 0) efree(us_paambiguous);
}

/***************************** TERMINAL CONTROL *****************************/

/*
 * routine to mute nonerror messages if flag is nonzero
 * (returns previous state)
 */
INTSML ttyquiet(INTSML flag)
{
	REGISTER INTSML prev;

	prev = tty_mute;
	tty_mute = flag;
	return(prev);
}

/*
 * routine to set auditing of terminal output.  If "state" is nonzero, begin
 * saving output.  Otherwise, terminate auditing.  Returns nonzero on error.
 */
INTSML ttysaveoutput(INTSML state)
{
	if (state != 0)
	{
		if (tty_audit != NULL) return(0);
		tty_audit = xappend(ELECTRICAUD);
		if (tty_audit == NULL)
		{
			ttyputerr("Cannot create %s", ELECTRICAUD);
			return(1);
		}
		return(0);
	}
	if (tty_audit == NULL) return(0);
	xclose(tty_audit);
	tty_audit = NULL;
	return(0);
}

/***************************** TERMINAL OUTPUT *****************************/

void ttynewcommand(void)
{
	tty_ttywriteseparator = 1;
}

/*
 * routine to output a normal message into the scrolling area of the status
 * screen in the style of "printf"
 */
void ttyputmsg(char *msg, ...)
{
	va_list ap;

	/* don't print or save this message if muted */
	if (tty_mute != 0) return;

	/* don't print this message if quit or aborted */
	if (el_pleasestop != 0) return;

	if (tty_ttyinited == 0)
	{
		var_start(ap, msg);
		vprintf(msg, ap);
		va_end(ap);
		putchar('\n');
	} else
	{
		var_start(ap, msg);
		tty_print(msg, ap);
		va_end(ap);
	}
}

/*
 * routine to output a "fact" message (those that provide unimportant information
 * that can be done without) into the scrolling area of the status
 * screen in the style of "printf"
 */
void ttyputmsgf(char *msg, ...)
{
	va_list ap;

	/* ignore this message if facts are turned off */
	if ((us_aid->aidstate&JUSTTHEFACTS) != 0) return;

	/* don't print or save this message if muted */
	if (tty_mute != 0) return;

	/* don't print this message if quit or aborted */
	if (el_pleasestop != 0) return;

	if (tty_ttyinited == 0)
	{
		var_start(ap, msg);
		vprintf(msg, ap);
		va_end(ap);
		putchar('\n');
	} else {
		var_start(ap, msg);
		tty_print(msg, ap);
		va_end(ap);
	}
}

/*
 * routine to output an error message into the scrolling area of the status
 * screen in the style of "printf"
 */
void ttyputerr(char *msg, ...)
{
	va_list ap;

	if (tty_ttyinited == 0)
	{
		fprintf(stderr, msg);
		(void)fprintf(stderr,"\n");
	} else
	{
		ttybeep();
		var_start(ap, msg);
		tty_print(msg, ap);
		va_end(ap);
	}
}

/****************************** TERMINAL INPUT ******************************/

/*
 * ttygetchar gets the next character from the text keyboard.
 * The routine must call the graphics module to do input.
 */
INTSML ttygetchar(void)
{
	return(us_getnxtchar());
}

/*
 * routine to print "prompt" and then read a line of text from the terminal
 * The address of the text line is returned.
 */
char *ttygetline(char *prompt)
{
	INTSML itemHit;
	char *line, *pt, *defaultval, localprompt[256];

	/* parse default value */
	strcpy(localprompt, prompt);
	defaultval = "";
	for(pt=localprompt; *pt != 0; pt++) if (*pt == '[') break;
	if (*pt != 0)
	{
		*pt++ = 0;
		defaultval = pt;
		for( ; *pt != 0; pt++) if (*pt == ']') break;
		if (*pt == ']') *pt = 0;
	}

	/* display the dialog box */
	if (DiaInitDialog(&tty_inputdialog) != 0) return("");
	DiaSetText(4, localprompt);
	DiaSetText(-3, defaultval);

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == CANCEL) break;
		if (itemHit == OK) break;
	}

	line = "";
	if (itemHit != CANCEL)
		line = us_putintoinfstr(DiaGetText(3));

	/* terminate the dialog */
	DiaDoneDialog();
	return(line);
}

/*
 * routine to display "prompt" and then accept a line of text from the
 * status window.  The address of the text line is returned.  Returns
 * zero if end-of-file is typed (^D).
 */
char *ttygetlinemessages(char *prompt)
{
	return(gra_getstring(prompt));
}

/*
 * routine to get a string with "parameter" information (does command completion)
 */
INTSML ttygetparam(char *prompt, COMCOMP *parameter, INTSML keycount, char *paramstart[])
{
	REGISTER INTSML itemHit;

	/* is this necessary?  if not, us_pathiskey does not need to be global */
	us_pathiskey = parameter->ifmatch;

	/* use dialogs if requested */
	if ((us_aid->aidstate & USEDIALOGS) != 0)
	{
		return(tty_specialparse(prompt, parameter, keycount, paramstart));
	}

	/* the general case: display the dialog box */
	if (DiaInitDialog(&tty_listdialog) != 0) return(0);
	DiaInitTextDialog(3, parameter->toplist, parameter->nextcomcomp, tty_nulldlogdone,
		0, SCSELMOUSE|SCSELKEY|SCDOUBLEQUIT);
	DiaSetText(4, prompt);

	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == OK || itemHit == CANCEL) break;
	}
	paramstart[0] = us_putintoinfstr(DiaGetScrollLine(3, DiaGetCurLine(3)));
	DiaDoneDialog();
	if (itemHit == CANCEL) return(0);
	return(1);
}

/*
 * routine to get a string with "parameter" information (does command completion)
 */
INTSML ttygetfullparam(char *prompt, COMCOMP *parameter, INTSML keycount,
	char *paramstart[])
{
	INTSML i;
 
	/* initialize for command completion */
    us_expandttybuffers(80);
	us_pacurchar = 0;
	us_pattyline[0] = 0;
	us_palinestart = us_pattyline;
	us_paparamstart = paramstart;
	us_pakeycount = MAXPARS;
	us_paparamstart[us_paparamcount = 0] = us_palinestart;
	us_paparamtype[us_paparamcount] = parameter;
	us_paparamtype[us_paparamcount+1] = NOCOMCOMP;
 
	/* display the dialog box */
	if (DiaInitDialog(&tty_fulldialog) != 0) return(0);
	DiaInitTextDialog(5, tty_nulldloglist, tty_nulldlogitem, tty_nulldlogdone, -1, 0);
 	DiaCharacterEdit(3);

	/* load header message */
	(void)initinfstr();
	if (*parameter->noise != 0)
	{
		(void)addstringtoinfstr(parameter->noise);
		if (parameter->def != 0)
		{
			(void)addstringtoinfstr(" (");
			(void)addstringtoinfstr(parameter->def);
			(void)addstringtoinfstr(")");
		}
	} else (void)addstringtoinfstr(prompt);
	DiaSetText(4, returninfstr());
 
	for(;;)
	{
		tty_pamyItemHit = 0;
		if (us_pagetword(tty_inmacchar, tty_outmacchar, tty_erasemacchar,
			tty_killmacchar, 0) != 0) break;
		if (tty_pamyItemHit == CANCEL || tty_pamyItemHit == OK) break;
	}
 
	/* parse the line if not cancelled */
	if (tty_pamyItemHit == CANCEL) us_paparamcount = 0; else
	{
		us_paparamcount++;
		for(i=1; i<us_paparamcount; i++) (us_paparamstart[i])[-1] = 0;
		if (*us_paparamstart[us_paparamcount-1] == 0) us_paparamcount--;
	}
	DiaDoneDialog();
	return(us_paparamcount);
}

/***************************** INTERNAL SUPPORT *****************************/

/*
 * internal routine to print a message in the scrolling area with the
 * style of "printf"
 */
void tty_print(char *msg, va_list ap)
{
	char line[8192];
	REGISTER INTSML i, j, k;
	static INTBIG sepcount = 0;

	if (tty_ttywriteseparator != 0)
	{
		sepcount++;
		sprintf(line,
			"================================= %d =================================",
				sepcount);
		gra_putstring(line);
		tty_ttywriteseparator = 0;
	}

	/* build the output line */
	(void)vsprintf(line, msg, ap);

    /* handle audit of terminal output */
    if (tty_audit != NULL) xprintf(tty_audit, "%s\n", line);

	/* remove excessive trailing space */
	i = strlen(line);
	while (i > 1 && (line[i-1] == ' ' || line[i-1] == '\t') &&
		(line[i-2] == ' ' || line[i-2] == '\t')) line[--i] = 0;

	/* break the line at newline characters */
	for(k=j=0; j<i; j++)
	{
		if (line[j] == '\n')
		{
			line[j] = 0;
			gra_putstring(&line[k]);
			k = j + 1;
		}
	}

	if (k < i) gra_putstring(&line[k]);
}

INTSML tty_inmacchar(COMCOMP **newcomcomp, INTSML *reset)
{
	INTSML chr;
 
	*reset = 0;
	for(;;)
	{
		chr = DiaGetNextCharacter(&tty_pamyItemHit);
		if (chr == -1) continue;
		if (chr == -2)
		{
			if (tty_pamyItemHit == 1 || tty_pamyItemHit == 2) return(EOF);
			continue;
		}
		break;
	}
	if (chr == 03) chr = '\n';
	return(chr);
}

void tty_outmacchar(char *t)
{
	INTBIG len;
	char *newmsg, *line;
 
	len = strlen(t);
	if (len == 0) return;
	line = DiaGetText(3);
	newmsg = (char *)emalloc(strlen(line)+len+1, el_tempcluster);
	(void)strcpy(newmsg, line);
	(void)strcat(newmsg, t);
	DiaSetText(3, newmsg);
	efree(newmsg);
}

void tty_erasemacchar(void)
{
	char *line;
 
	line = DiaGetText(3);
	line[strlen(line)-1] = 0;
	DiaSetText(3, line);
}
 
void tty_killmacchar(void)
{
	DiaSetText(3, "");
}
