/*
 * Electric(tm) VLSI Design System
 *
 * File: drc.c
 * Design-rule check aid
 * Written by: Steven M. Rubin, Electric Editor Incorporated
 *
 * Copyright (c) 1998 Electric Editor Incorporated.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Electric Editor Incorporated
 * 23470 Sunset Drive, Suite 108
 * Los Gatos, California 95033
 * support@electriceditor.com
 */

#include "config.h"
#if DRCAID

#include "global.h"
#include "egraphics.h"
#include "drc.h"
#include "efunction.h"

/* the DRC aid table */
static COMCOMP drccp = {NOKEYWORD, us_topoffacets, us_nextparse, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "facet to be rechecked", "check current facet"};
static COMCOMP drctcp = {NOKEYWORD, us_topoffacets, us_nextparse, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "facet to be totally rechecked", "check current facet"};
static KEYWORD drcnotopt[] =
{
	{"verbose",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"pointout",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP drcnotp = {drcnotopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "Design Rule Checker negative action", 0};
static COMCOMP drcfip = {NOKEYWORD, us_topoffacets, us_nextparse, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "facet to ignored for short check DRC", 0};
static COMCOMP drcfup = {NOKEYWORD, us_topoffacets, us_nextparse, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "facet to un-ignored for short check DRC", 0};
static KEYWORD drcfopt[] =
{
	{"run",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"ignore",       1,{&drcfip,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"unignore",     1,{&drcfup,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP drcfp = {drcfopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "Short Detecting Design Rule Checker action", 0};
static KEYWORD drcnopt[] =
{
	{"run",				0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"all",				0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"verbose",			0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"visualdebug",		0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"first-error",		0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"current-error",	0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"next-error",		0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"previous-error",	0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"print-hierarchy",	0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"highlight-step",	0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"count-shapes",	0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP dr_batchp = {drcnopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "Batch Design Rule Checker action", 0};
static KEYWORD drcopt[] =
{
	{"check",          1,{&drccp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"total-check",    1,{&drctcp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"not",            1,{&drcnotp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"verbose",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"pointout",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"batch",          1,{&dr_batchp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"shortcheck",     1,{&drcfp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"ecadcheck",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP dr_drcp = {drcopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "Design Rule Checker action", "show defaults"};

/* dcheck modules */
#define	NODCHECK ((DCHECK *)-1)

#define	NODENEW    1			/* this nodeinst was created */
#define	ARCNEW     2			/* this arcinst was created */
#define	NODEMOD    3			/* this nodeinst was modified */
#define	ARCMOD     4			/* this arcinst was modified */
#define	NODEKILL   5			/* this nodeinst was deleted */
#define	ARCKILL    6			/* this arcinst was deleted */

typedef struct Idcheck
{
	INTBIG    entrytype;				/* type of object being checked */
	union    dr_entry
	{
		NODEINST *ni;
		ARCINST  *ai;
		INTBIG    blind;
	} entryaddr;					/* object being checked */
	struct Idcheck *nextdcheck;		/* next in list */
} DCHECK;
static DCHECK *dr_firstdcheck;
static DCHECK *dr_dcheckfree;

/*
 * The meaning of "aid:drc.DRC_pointout" is as follows:
 * 0: don't identify errors
 * 1: identify errors
 * >1: identify errors (and some have already been identified)
 * -2: identify errors, but no more in this session
 */
INTBIG dr_pointout;					/* key for aid:drc.DRC_pointout */
static INTBIG dr_verbose;			/* key for aid:drc.DRC_verbose */
static INTBIG dr_ignore_list;		/* key for aid:drc.DRC_ignore_list */

struct plist
{
	POLYGON **polygons;				/* polygons to be checked on node or arc */
	INTSML polylistsize;				/* size of above list */
} dr_polylist, dr_subpolylist;

static INTBIG dr_pseudonet = 1;
static NODEINST *dr_tinynodeinst;
static GEOM *dr_tinyobj;
AIDENTRY *dr_aid;					/* the DRC aid object */

#define	DCHECKBLOCKSIZE	50			/* number to "malloc" at a time */

/* prototypes for local routines */
DCHECK *dr_allocdcheck(void);
void dr_freedcheck(DCHECK*);
void dr_queuecheck(GEOM*, INTSML);
void dr_unqueuecheck(LIBRARY *lib);
INTSML dr_checknodeinst(NODEINST*, INTSML);
INTSML dr_checkarcinst(ARCINST*, INTSML);
INTSML dr_badbox(GEOM*, INTSML, TECHNOLOGY*, NODEPROTO*, POLYGON*, NETWORK*, INTSML);
INTSML dr_activeontransistor(GEOM*, INTSML, NETWORK*, INTBIG, INTBIG, INTBIG, INTBIG, GEOM*, INTSML, NETWORK*, INTBIG, INTBIG, INTBIG, INTBIG);
INTSML dr_polyintersect(POLYGON*, POLYGON*);
INTSML dr_lineintersect(INTBIG, INTBIG, INTBIG, INTBIG, POLYGON*);
INTBIG dr_finddistance(POLYGON*, POLYGON*);
INTSML dr_check(INTSML, NETWORK*, GEOM*, POLYGON*, INTSML, NETWORK*, GEOM*, POLYGON*, INTBIG);
void dr_setignore(GEOM*, GEOM*);
void dr_unignore(GEOM*);
INTSML dr_cropnodeinst(NODEINST*, INTSML, NETWORK*, GEOM*, INTBIG*, INTBIG*, INTBIG*, INTBIG*);
INTSML dr_croparcinst(ARCINST*, INTSML, INTBIG*, INTBIG*, INTBIG*, INTBIG*);
char *dr_describeerror(INTSML, GEOM*, INTSML, GEOM*);
INTSML dr_objtouch(GEOM*, GEOM*);
NODEPROTO *dr_checkthisfacet(INTSML, char*);
INTSML dr_ensurepolylist(struct plist*, INTSML);

/*
 * routine to allocate a new dcheck module from the pool (if any) or memory
 */
DCHECK *dr_allocdcheck(void)
{
	REGISTER DCHECK *d;
	REGISTER INTSML i, j;

	if (dr_dcheckfree == NODCHECK)
	{
		for(i=DCHECKBLOCKSIZE; i>0; i /= 2)
		{
			d = (DCHECK *)emalloc(((sizeof (DCHECK)) * i), dr_aid->cluster);
			if (d != 0) break;
		}
		if (d == 0) return(NODCHECK);
		for(j=0; j<i; j++)
			if (j == 0) d[j].nextdcheck = NODCHECK; else
				d[j].nextdcheck = &d[j-1];
		dr_dcheckfree = &d[i-1];
	}

	/* take module from free list */
	d = dr_dcheckfree;
	dr_dcheckfree = (DCHECK *)d->nextdcheck;
	return(d);
}

/*
 * routine to return dcheck module "d" to the pool of free modules
 */
void dr_freedcheck(DCHECK *d)
{
	d->nextdcheck = dr_dcheckfree;
	dr_dcheckfree = d;
}

void dr_queuecheck(GEOM *geom, INTSML function)
{
	REGISTER DCHECK *d;

	d = dr_allocdcheck();
	if (d == NODCHECK)
	{
		ttyputerr("DRC: out of space");
		return;
	}
	d->entrytype = function;
	d->entryaddr.blind = geom->entryaddr.blind;
	d->nextdcheck = dr_firstdcheck;
	dr_firstdcheck = d;
}

/*
 * Routine to remove all check objects that are in library "lib"
 * (because the library has been deleted)
 */
void dr_unqueuecheck(LIBRARY *lib)
{
	REGISTER DCHECK *d, *nextd, *lastd;
	REGISTER LIBRARY *rlib;

	lastd = NODCHECK;
	for(d = dr_firstdcheck; d != NODCHECK; d = nextd)
	{
		nextd = d->nextdcheck;
		if (d->entrytype == OBJNODEINST) rlib = d->entryaddr.ni->parent->cell->lib; else
			rlib = d->entryaddr.ai->parent->cell->lib;
		if (rlib == lib)
		{
			if (lastd == NODCHECK) dr_firstdcheck = nextd; else
				lastd->nextdcheck = nextd;
			dr_freedcheck(d);
			continue;
		}
		lastd = d;
	}
}

/******************** CONTROL ********************/

void dr_init(INTBIG *argc, char *argv[], AIDENTRY *thisaid)
{
	/* only initialize during pass 1 */
	if (thisaid == NOAID || thisaid == 0) return;

	dr_aid = thisaid;

	/* initialize the design-rule checker lists */
	dr_polylist.polylistsize = dr_subpolylist.polylistsize = 0;
	dr_dcheckfree = NODCHECK;
	dr_firstdcheck = NODCHECK;
	dr_ignore_list = makekey("DRC_ignore_list");
	dr_pointout = makekey("DRC_pointout");
	dr_verbose = makekey("DRC_verbose");
	nextvarchangequiet();
	(void)setvalkey((INTBIG)dr_aid, VAID, dr_pointout, 0, VINTEGER|VDONTSAVE);
}

void dr_done(void) {}

INTSML dr_set(INTSML count, char *par[])
{
	REGISTER INTSML l, negate;
	REGISTER INTBIG i;
	REGISTER char *pp;
	REGISTER FILE *f;
	REGISTER NODEPROTO *np;
	REGISTER VARIABLE *var;
	REGISTER TECHNOLOGY *tech;
	char *newpar[3], *truename;
	extern AIDENTRY *io_aid, *net_aid;

	if (count == 0)
	{
		var = getvalkey((INTBIG)dr_aid, VAID, VINTEGER, dr_verbose);
		if (var == NOVARIABLE) i = 0; else i = var->addr;
		if (i == 0) ttyputmsg("Design rule checker is brief"); else
			ttyputmsg("Design rule checker is verbose");
		var = getvalkey((INTBIG)dr_aid, VAID, VINTEGER, dr_pointout);
		if (var == NOVARIABLE) i = 0; else i = var->addr;
		if (i == 0) ttyputmsg("Errors will not be highlighted"); else
			ttyputmsg("Errors will be highlighted");
		return(0);
	}

	l = strlen(pp = par[0]);
	negate = 0;
	if (namesamen(pp, "not", l) == 0)
	{
		negate++;
		if (count <= 1)
		{
			count = ttygetparam("DRC negate option:", &drcnotp, MAXPARS-1, &par[1]) + 1;
			if (count <= 1)
			{
				ttyputerr("Aborted");
				return(1);
			}
		}
		l = strlen(pp = par[1]);
	}

	if (namesamen(pp, "total-check", l) == 0)
	{
		/* quit if incremental DRC not available */
		if (checkcap(dr_aid, DRCHASINC) == 0)
		{
			ttyputerr("Sorry, incremental DRC is not available");
			return(1);
		}

		/* make sure network tool is on */
		if ((net_aid->aidstate&AIDON) == 0)
		{
			ttyputerr("Network tool must be running...turning it on");
			aidturnon(net_aid, 0);
			ttyputerr("...now reissue the DRC command");
			return(1);
		}

		np = dr_checkthisfacet(count, par[1]);
		if (np == NONODEPROTO) return(1);

		/* erase all ignore lists on all objects */
		var = getvalkey((INTBIG)np, VNODEPROTO, VGEOM|VISARRAY, dr_ignore_list);
		if (var != NOVARIABLE)
			(void)delvalkey((INTBIG)np, VNODEPROTO, dr_ignore_list);

		ttyputmsgf("Ignored violations turned back on.  Now checking...");
		dr_examinenodeproto(np);
		ttyputmsgf("Facet %s totally checked", describenodeproto(np));
		return(0);
	}
	if (namesamen(pp, "check", l) == 0)
	{
		/* quit if incremental DRC not available */
		if (checkcap(dr_aid, DRCHASINC) == 0)
		{
			ttyputerr("Sorry, incremental DRC is not available");
			return(1);
		}

		/* make sure network tool is on */
		if ((net_aid->aidstate&AIDON) == 0)
		{
			ttyputerr("Network tool must be running...turning it on");
			aidturnon(net_aid, 0);
			ttyputerr("...now reissue the DRC command");
			return(1);
		}

		np = dr_checkthisfacet(count, par[1]);
		if (np == NONODEPROTO) return(1);
		dr_examinenodeproto(np);
		ttyputmsgf("Facet %s checked", describenodeproto(np));
		return(0);
	}

	if (namesamen(pp, "ecadcheck", l) == 0)
	{
		/* quit if Dracula DRC not available */
		if (checkcap(dr_aid, DRCHASDRAC) == 0)
		{
			ttyputerr("Sorry, ECAD DRC is not available");
			return(1);
		}

		np = getcurfacet();
		if (np == NONODEPROTO)
		{
			ttyputerr("Must be editing a facet to check with ECAD's Dracula");
			return(1);
		}
		tech = whattech(np);
		var = getval((INTBIG)tech, VTECHNOLOGY, VSTRING|VISARRAY, "DRC_ecad_deck");
		if (var == NOVARIABLE)
		{
			ttyputerr("Cannot find an ECAD deck in the %s technology", tech->techname);
			return(1);
		}

		/* create the file */
		f = xcreate("ecaddrc.RUL", FILETYPEDRAC, "ECAD DRC control deck", &truename);
		if (f == NULL)
		{
			if (truename != 0) ttyputerr("Cannot write %s", truename);
			return(1);
		}

		/* write the deck */
		l = getlength(var);
		for(i=0; i<l; i++)
		{
			pp = ((char **)var->addr)[i];
			if (strncmp(pp, " PRIMARY =", 10) == 0)
			{
				xprintf(f, " PRIMARY = %s\n", np->cell->cellname);
				continue;
			}
			if (strncmp(pp, " INDISK =", 9) == 0)
			{
				xprintf(f, " INDISK = %s.cif\n", np->cell->cellname);
				continue;
			}
			xprintf(f, "%s\n", pp);
		}

		/* finished with control deck */
		xclose(f);
		ttyputmsgf("Wrote 'ecaddrc.RUL'.  Now generating CIF for facet %s", describenodeproto(np));

		/* tell I/O to write CIF */
		newpar[0] = "cif";
		newpar[1] = "output";
		newpar[2] = "include-cloak-layer";
		(void)tellaid(io_aid, 3, newpar);
		(void)askaid(io_aid, "write", (INTBIG)np->cell->lib, (INTBIG)"cif");
		newpar[0] = "cif";
		newpar[1] = "output";
		newpar[2] = "ignore-cloak-layer";
		(void)tellaid(io_aid, 3, newpar);
		return(0);
	}

	if (namesamen(pp, "batch", l) == 0)
	{
		/* quit if batch DRC not available */
		if (checkcap(dr_aid, DRCHASBATCH) == 0)
		{
			ttyputerr("Sorry, batch DRC is not available");
			return(1);
		}

		if (count == 1)
		{
			count = ttygetparam("Batch DRC option: ", &dr_batchp, 3, &par[1]) + 1;
			if (count <= 1) return(1);
		}
		l = strlen(pp = par[1]);
		if (namesamen(pp, "run", l) == 0 || namesamen(pp, "all", l) == 0 ||
			namesamen(pp, "check", l) == 0 || namesamen(pp, "checkall", l) == 0)
		{
			/* make sure network tool is on */
			if ((net_aid->aidstate&AIDON) == 0)
			{
				ttyputerr("Network tool must be running...turning it on");
				aidturnon(net_aid, 0);
				ttyputerr("...now reissue the DRC command");
				return(1);
			}

			if (namesamen(pp, "run", l) == 0)
			{
				np = getcurfacet();
				if (np == NONODEPROTO)
				{
					ttyputerr("No current facet");
					return(0);
				}
				ttyputmsg("%d errors", drcb_check(np,1));
				return(0);
			}
			if (namesamen(pp, "all", l) == 0)
			{
				ttyputmsg("%d errors", drcb_check(NONODEPROTO, 1));
				return(0);
			}
		}
		if (namesamen(pp, "verbose", l) == 0)
			return(drcb_debug());
		if (namesamen(pp, "visualdebug", l) == 0)
			return(drcb_vdebug());
		if (namesamen(pp, "next-error", l) == 0)
		{
			ttyputmsg("%s", drcb_next_error());
			return(0);
		}
		if (namesamen(pp, "previous-error", l) == 0)
		{
			ttyputmsg("%s", drcb_prev_error());
			return(0);
		}
		if (namesamen(pp, "current-error", l) == 0)
		{
			ttyputmsg("%s", drcb_curr_error());
			return(0);
		}
		if (namesamen(pp, "first-error", l) == 0)
		{
			ttyputmsg("%s", drcb_first_error());
			return(0);
		}
		if (namesamen(pp, "print-hierarchy", l) == 0)
			return(drcb_print_hierarchy(getcurfacet()));
		if (namesamen(pp, "highlight-step", l) == 0)
			return(drcb_highlight_step(getcurfacet()));
		if (namesamen(pp, "count-shapes", l) == 0)
			return(drcb_nodestats(getcurfacet()));
		ttyputerr("Bad TELLAID DRC BATCH option: %s", pp);
		return(1);
	}

	if (namesamen(pp, "shortcheck", l) == 0)
	{
		if (count < 2)
		{
			ttyputerr("Usage: tellaid drc shortcheck (run|ignore|unignore)");
			return(1);
		}
		l = strlen(pp = par[1]);
		if (namesamen(pp, "run", l) == 0)
		{
			/* make sure network tool is on */
			if ((net_aid->aidstate&AIDON) == 0)
			{
				ttyputerr("Network tool must be running...turning it on");
				aidturnon(net_aid, 0);
				ttyputerr("...now reissue the DRC command");
				return(1);
			}
			return(drc_flatwrite(getcurfacet()));
		}
		if (namesamen(pp, "ignore", l) == 0)
		{
			if (count < 3)
			{
				ttyputerr("Usage: tellaid drc shortcheck ignore FACET");
				return(1);
			}
			return(drc_flatignore(par[2]));
		}
		if (namesamen(pp, "unignore", l) == 0)
		{
			if (count < 3)
			{
				ttyputerr("Usage: tellaid drc shortcheck unignore FACET");
				return(1);
			}
			return(drc_flatunignore(par[2]));
		}
		ttyputerr("Invalid shortcheck DRC option: %s", pp);
		return(1);
	}

	if (namesamen(pp, "verbose", l) == 0)
	{
		if (negate == 0)
		{
			(void)setvalkey((INTBIG)dr_aid, VAID, dr_verbose, 1, VINTEGER|VDONTSAVE);
			ttyputmsgf("Design rule checker verbose");
		} else
		{
			if (getvalkey((INTBIG)dr_aid, VAID, VINTEGER, dr_verbose) != NOVARIABLE)
				(void)delvalkey((INTBIG)dr_aid, VAID, dr_verbose);
			ttyputmsgf("Design rule checker brief");
		}
		return(0);
	}

	if (namesamen(pp, "pointout", l) == 0)
	{
		if (negate == 0)
		{
			(void)setvalkey((INTBIG)dr_aid, VAID, dr_pointout, 1, VINTEGER|VDONTSAVE);
			ttyputmsgf("Errors will be identified");
		} else
		{
			(void)setvalkey((INTBIG)dr_aid, VAID, dr_pointout, 0, VINTEGER|VDONTSAVE);
			ttyputmsgf("Errors not identified");
		}
		return(0);
	}
	ttyputerr("Bad DRC option: %s", pp);
	return(1);
}

/*
 * make requests of the design-rule checking aid:
 *
 ***************** BATCH DRC ****************
 *  "batch-run" RETURNS: number of errors found
 *  "batch-all" RETURNS: number of errors found
 *  "batch-check" RETURNS: number of errors found
 *  "batch-checkall" RETURNS: number of errors found
 *  "batch-error-count" RETURNS: number of error messages
 *  "batch-next-error" RETURNS: string error message
 *  "batch-previous-error" RETURNS: string error message
 *  "batch-current-error" RETURNS: string error message
 *  "batch-first-error" RETURNS: string error message
 */
INTBIG dr_request(char *command, va_list ap)
{
	INTSML l;

	if (namesamen(command, "batch-", 6) == 0)
	{
		/* quit if batch DRC not available */
		if (checkcap(dr_aid, DRCHASBATCH) == 0)
		{
			ttyputerr("Sorry, batch DRC is not available");
			return(0);
		}

		l = strlen(&command[6]);
		if (namesamen(&command[6], "run", l) == 0)
			return(drcb_check(getcurfacet(), 1));
		if (namesamen(&command[6], "all", l) == 0)
			return(drcb_check(NONODEPROTO, 1));
		if (namesamen(&command[6], "check", l) == 0)
			return(drcb_check(getcurfacet(), 0));
		if (namesamen(&command[6], "checkall", l) == 0)
			return(drcb_check(NONODEPROTO, 0));

		if (namesamen(&command[6], "error-count", l) == 0)
			return(drcb_errorcount());
		if (namesamen(&command[6], "next-error", l) == 0)
			return((INTBIG)drcb_next_error());
		if (namesamen(&command[6], "previous-error", l) == 0)
			return((INTBIG)drcb_prev_error());
		if (namesamen(&command[6], "current-error", l) == 0)
			return((INTBIG)drcb_curr_error());
		if (namesamen(&command[6], "first-error", l) == 0)
			return((INTBIG)drcb_first_error());
	}
	return(0);
}

void dr_examinenodeproto(NODEPROTO *np)
{
	REGISTER NODEINST *ni;
	REGISTER ARCINST *ai;
	extern AIDENTRY *net_aid;
	REGISTER VARIABLE *var;
	REGISTER INTSML i, ret;

	/* quit if incremental DRC not available */
	if (checkcap(dr_aid, DRCHASINC) == 0) return;

	if (stopping("DRC") != 0)
	{
		var = getvalkey((INTBIG)dr_aid, VAID, VINTEGER, dr_pointout);
		if (var != NOVARIABLE && var->addr != 0)
			(void)setvalkey((INTBIG)dr_aid, VAID, dr_pointout, 1, VINTEGER|VDONTSAVE);
		return;
	}

	/* make sure network tool is on */
	if ((net_aid->aidstate&AIDON) == 0) return;

	/* now check every object */
	ret = 0;
	for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		ret = dr_checknodeinst(ni, 1);
		if (ret != 0) break;
		if (stopping("DRC") == 0) continue;
		ret = 1;
		break;
	}
	if (ret == 0)
		for(ai = np->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
	{
		if (dr_checkarcinst(ai, 1) != 0) break;
		if (stopping("DRC") != 0) break;
	}

	/* restore highlighting features */
	var = getvalkey((INTBIG)dr_aid, VAID, VINTEGER, dr_pointout);
	if (var != NOVARIABLE) i = abs(var->addr); else i = 0;
	if (i > 1) i = 1;
	(void)setvalkey((INTBIG)dr_aid, VAID, dr_pointout, i, VINTEGER|VDONTSAVE);
}

void dr_slice(void)
{
	REGISTER DCHECK *d, *nextd;
	REGISTER INTSML nomore, i;
	REGISTER VARIABLE *var;
	extern AIDENTRY *net_aid;

	if (dr_firstdcheck == NODCHECK) return;

	/* quit if incremental DRC not available */
	if (checkcap(dr_aid, DRCHASINC) == 0)
	{
		ttyputerr("Sorry, batch DRC is not available");
		aidturnoff(dr_aid, 0);
		return;
	}

	/* make sure network tool is on */
	if ((net_aid->aidstate&AIDON) == 0)
	{
		ttyputerr("Network tool must be running...turning it on");
		aidturnon(net_aid, 0);
		return;
	}

	/* mark this activity */
	setactivity("DRC");

	/* first clear the ignore information on any objects that changed */
	for(d = dr_firstdcheck; d != NODCHECK; d = d->nextdcheck)
	{
		if (d->entrytype == NODEMOD || d->entrytype == NODEKILL)
			dr_unignore(d->entryaddr.ni->geom);
		if (d->entrytype == ARCMOD || d->entrytype == ARCKILL)
			dr_unignore(d->entryaddr.ai->geom);
	}

	/* now check the objects */
	nomore = 0;
	for(d = dr_firstdcheck; d != NODCHECK; d = nextd)
	{
		nextd = d->nextdcheck;
		if (stopping("DRC") == 0 && nomore == 0)
		{
			if (d->entrytype == NODEMOD || d->entrytype == NODENEW)
			{
				if (dr_checknodeinst(d->entryaddr.ni, 0) != 0) nomore++;
			}
			if (d->entrytype == ARCMOD || d->entrytype == ARCNEW)
			{
				if (dr_checkarcinst(d->entryaddr.ai, 0) != 0) nomore++;
			}
		}
		dr_freedcheck(d);
	}
	dr_firstdcheck = NODCHECK;

	(void)stopping("DRC");

	/* restore highlighting features */
	var = getvalkey((INTBIG)dr_aid, VAID, VINTEGER, dr_pointout);
	if (var == NOVARIABLE) return;
	i = abs(var->addr);
	if (i > 1)
	{
		ttyputmsgf("DRC done");
		i = 1;
	}
	if (i != var->addr)
		(void)setvalkey((INTBIG)dr_aid, VAID, dr_pointout, i, VINTEGER|VDONTSAVE);
}

/******************** DATABASE CHANGES ********************/

void dr_startbatch(AIDENTRY *source) {}

void dr_endbatch(void) {}

void dr_startobjectchange(INTBIG addr, INTBIG type) {}

void dr_endobjectchange(INTBIG addr, INTBIG type) {}

void dr_modifynodeinst(NODEINST *ni,INTBIG oldlx,INTBIG oldly,INTBIG oldhx,INTBIG oldhy,
	INTSML oldrot,INTSML oldtran)
{
	dr_queuecheck(ni->geom, NODEMOD);
}

void dr_modifyarcinst(ARCINST *ai,INTBIG oldxA, INTBIG oldyA, INTBIG oldxB, INTBIG oldyB,
	INTBIG oldwid, INTBIG oldlen)
{
	dr_queuecheck(ai->geom, ARCMOD);
}

void dr_modifyportproto(PORTPROTO *pp, NODEINST *oni, PORTPROTO *opp) {}

void dr_modifynodeproto(NODEPROTO *np) {}

void dr_modifydescript(VARIABLE *var, INTBIG odes) {}

void dr_newobject(INTBIG addr, INTBIG type)
{
	if ((type&VTYPE) == VNODEINST)
		dr_queuecheck(((NODEINST *)addr)->geom, NODENEW); else
			if ((type&VTYPE) == VARCINST)
				dr_queuecheck(((ARCINST *)addr)->geom, ARCNEW);
}

void dr_killobject(INTBIG addr, INTBIG type)
{
	if ((type&VTYPE) == VNODEINST)
		dr_queuecheck(((NODEINST *)addr)->geom, NODEKILL); else
			if ((type&VTYPE) == VARCINST)
				dr_queuecheck(((ARCINST *)addr)->geom, ARCKILL);
}

void dr_newvariable(INTBIG addr, INTBIG type, INTBIG key, INTBIG newtype) {}
void dr_killvariable(INTBIG addr, INTBIG type, INTBIG key, INTBIG oldaddr, INTBIG oldtype,
	INTBIG olddescript) {}
void dr_modifyvariable(INTBIG addr, INTBIG type, INTBIG key, INTBIG vartype, INTBIG index,
	INTBIG oldvalue) {}
void dr_insertvariable(INTBIG addr, INTBIG type, INTBIG key, INTBIG vartype, INTBIG index) {}
void dr_deletevariable(INTBIG addr, INTBIG type, INTBIG key, INTBIG vartype, INTBIG index,
	INTBIG oldvalue) {}
void dr_readlibrary(LIBRARY *lib) {}

void dr_eraselibrary(LIBRARY *lib)
{
	dr_unqueuecheck(lib);
}

void dr_writelibrary(LIBRARY *lib, INTSML pass) {}

/******************** SUPPORT ROUTINES ********************/

/*
 * routine to check the design rules about nodeinst "ni".  If "partial"
 * is nonzero, only check those other objects whose geom pointers are
 * greater than this one (i.e. only check any pair of objects once).
 * The routine returns nonzero if checking is to be terminated.
 */
INTSML dr_checknodeinst(NODEINST *ni, INTSML partial)
{
	REGISTER INTSML tot, j, ret;
	REGISTER NODEPROTO *np;
	XARRAY trans;
	REGISTER NETWORK *net;

	np = ni->proto;
	if (np->index == 0) return(0);
	makerot(ni, trans);

	/* get number of polygons on this node */
	tot = nodeEpolys(ni);

	/* make sure there is space for the polygons */
	if (dr_ensurepolylist(&dr_polylist, tot) != 0) return(1);

	/* get all of the polygons on this node */
	for(j=0; j<tot; j++) shapeEnodepoly(ni, j, dr_polylist.polygons[j]);

	/* examine the polygons on this node */
	for(j=0; j<tot; j++)
	{
		if (dr_polylist.polygons[j]->layer < 0) continue;
		if (dr_polylist.polygons[j]->portproto == NOPORTPROTO) net = NONETWORK; else
			net = dr_network(ni, dr_polylist.polygons[j]->portproto);
		xformpoly(dr_polylist.polygons[j], trans);
		ret = dr_badbox(ni->geom, dr_polylist.polygons[j]->layer, np->tech,
			ni->parent, dr_polylist.polygons[j], net, partial);
		if (ret < 0) return(1);
		if (ret > 0) break;
	}
	return(0);
}

/*
 * routine to check the design rules about arcinst "ai".  If "partial"
 * is nonzero, only check those other objects whose geom pointers are
 * greater than this one (i.e. only check any pair of objects once).
 * The routine returns nonzero if checking is to be terminated.
 */
INTSML dr_checkarcinst(ARCINST *ai, INTSML partial)
{
	REGISTER INTSML tot, j, ret;

	/* see how many polygons there are */
	tot = arcpolys(ai);

	/* make sure there is space for the polygons */
	if (dr_ensurepolylist(&dr_polylist, tot) != 0) return(1);

	/* get all of the polygons on this arc */
	for(j=0; j<tot; j++) shapearcpoly(ai, j, dr_polylist.polygons[j]);

	/* examine the polygons on this arc */
	for(j=0; j<tot; j++)
	{
		if (dr_polylist.polygons[j]->layer < 0) continue;
		ret = dr_badbox(ai->geom, dr_polylist.polygons[j]->layer,
			ai->proto->tech, ai->parent, dr_polylist.polygons[j], ai->network, partial);
		if (ret < 0) return(1);
		if (ret > 0) break;
	}
	return(0);
}

/*
 * routine to make a design-rule check on a piece of silicon which is
 * described in polygon "obj".  This silicon is on layer "nlayer" in
 * technology "tech", is part of the object at geometry module "geom", is on
 * electrical layer "nindex" and is in facet "facet".  If "partial" is nonzero
 * then only objects whose geometry modules are greater than this one will
 * be checked (to prevent any pair of objects from being checked twice).
 * A positive return indicates that an error has been detected and reported.
 * A negative return indicates that checking should be terminated.
 */
INTSML dr_badbox(GEOM *geom, INTSML nlayer, TECHNOLOGY *tech, NODEPROTO *facet,
	POLYGON *obj, NETWORK *nnet, INTSML partial)
{
	REGISTER GEOM *pi;
	REGISTER NODEINST *ni;
	REGISTER NODEPROTO *np;
	REGISTER ARCINST *ai;
	REGISTER NETWORK *net;
	REGISTER INTBIG bound, search, dist;
	REGISTER INTSML j, tot, touch, ret;
	static POLYGON *poly = NOPOLYGON;
	INTBIG lx, hx, ly, hy;
	XARRAY trans;
	INTSML con;

	/* see how far around the box it is necessary to search */
	bound = maxdrcsurround(tech, nlayer);
	if (bound < 0) return(0);

	/* get bounds */
	getbbox(obj, &lx, &hx, &ly, &hy);

	/* make sure there is a polygon */
	if (poly == NOPOLYGON) poly = allocpolygon(4, dr_aid->cluster);

	/* search in the area surrounding the box */
	search = initsearch(lx-bound,hx+bound, ly-bound,hy+bound, facet);
	for(;;)
	{
		if ((pi = nextobject(search)) == NOGEOM) break;
		if (pi == geom) continue;
		if (partial != 0 && pi < geom) continue;

		/* see if the objects directly touch */
		touch = dr_objtouch(pi, geom);

		if (pi->entrytype == OBJNODEINST)
		{
			ni = pi->entryaddr.ni;   np = ni->proto;

			/* no hierarchical DRC: ignore nodes that are not primitive */
			if (np->index == 0) continue;

			/* don't check between technologies */
			if (np->tech != tech) continue;

			/* prepare to examine every layer in this nodeinst */
			makerot(ni, trans);
			tot = nodeEpolys(ni);

			/* make sure there is space for the polygons */
			if (dr_ensurepolylist(&dr_subpolylist, tot) != 0) return(1);

			/* get the shape of each nodeinst layer */
			for(j=0; j<tot; j++)
				shapeEnodepoly(ni, j, dr_subpolylist.polygons[j]);

			for(j=0; j<tot; j++)
			{
				if (dr_subpolylist.polygons[j]->layer < 0) continue;

				/* find the global electrical net for this layer */
				net = NONETWORK;
				if (dr_subpolylist.polygons[j]->portproto != NOPORTPROTO)
					net = dr_network(ni, dr_subpolylist.polygons[j]->portproto);

				/* see whether the two objects are electrically connected */
				if (net == NONETWORK || nnet == NONETWORK || net == nnet) con = 1; else
					con = 0;

				/* if they connect electrically and adjoin, don't check */
				if (con != 0 && touch != 0) continue;

				/* see how close they can get */
				dist = drcmindistance(tech, nlayer, dr_subpolylist.polygons[j]->layer, con);
				if (dist < 0) continue;

				xformpoly(dr_subpolylist.polygons[j], trans);

				/* check the distance */
				ret = dr_check(nlayer, nnet, geom, obj, dr_subpolylist.polygons[j]->layer, net, pi,
					dr_subpolylist.polygons[j], dist);
				if (ret != 0) { termsearch(search);   return(ret); }
			}
		} else
		{
			ai = pi->entryaddr.ai;

			/* don't check between technologies */
			if (ai->proto->tech != tech) continue;

			/* see whether the two objects are electrically connected */
			net = ai->network;
			if (nnet == NONETWORK || net == nnet) con = 1; else con = 0;

			/* if they connect electrically and adjoin, don't check */
			if (con != 0 && touch != 0) continue;

			/* prepare to examine every layer in this arcinst */
			tot = arcpolys(ai);

			/* make sure there is space for the polygons */
			if (dr_ensurepolylist(&dr_subpolylist, tot) != 0) return(1);

			/* get the shape of each arcinst layer */
			for(j=0; j<tot; j++) shapearcpoly(ai, j, dr_subpolylist.polygons[j]);

			for(j=0; j<tot; j++)
			{
				if (dr_subpolylist.polygons[j]->layer < 0) continue;

				/* see how close they can get */
				dist = drcmindistance(tech, nlayer, dr_subpolylist.polygons[j]->layer, con);
				if (dist < 0) continue;

				/* check the distance */
				ret = dr_check(nlayer, nnet, geom, obj, dr_subpolylist.polygons[j]->layer, net, pi,
					dr_subpolylist.polygons[j], dist);
				if (ret != 0) { termsearch(search);   return(ret); }
			}
		}
	}
	return(0);
}

/*
 * routine to see if the polygon "poly1" in "pos1" on layer "layer1" with electrical
 * network "net1" comes within "dist" from the polygon "poly2" in "pos2" on layer
 * "layer2" with electrical network "net2".
 * A positive return indicates that an error has been detected and reported.
 * A negative return indicates that checking should be terminated.
 */
INTSML dr_check(INTSML layer1, NETWORK *net1, GEOM *pos1, POLYGON *poly1, INTSML layer2,
	NETWORK *net2, GEOM *pos2, POLYGON *poly2, INTBIG dist)
{
	REGISTER TECHNOLOGY *tech1, *tech2;
	REGISTER INTSML i, len, isbox1, isbox2;
	REGISTER INTBIG pd;
	REGISTER VARIABLE *var;
	REGISTER char *ret;
	extern AIDENTRY *us_aid;
	INTBIG lx1, hx1, ly1, hy1, lx2, hx2, ly2, hy2;
	char *pars[2];
	extern COMCOMP us_drcerrp;

	/* turn off flag that the nodeinst may be undersized */
	dr_tinynodeinst = NONODEINST;

	isbox1 = isbox(poly1, &lx1, &hx1, &ly1, &hy1);
	if (isbox1 == 0) getbbox(poly1, &lx1, &hx1, &ly1, &hy1);
	isbox2 = isbox(poly2, &lx2, &hx2, &ly2, &hy2);
	if (isbox2 == 0) getbbox(poly2, &lx2, &hx2, &ly2, &hy2);

	/* special code if both polygons are manhattan */
	if (isbox1 != 0 && isbox2 != 0)
	{
		/* crop out parts of any arc that is covered by an adjoining node */
		if (pos1->entrytype == OBJARCINST)
		{
			if (dr_croparcinst(pos1->entryaddr.ai, layer1, &lx1, &hx1, &ly1, &hy1)) return(0);
		}
		if (pos2->entrytype == OBJARCINST)
		{
			if (dr_croparcinst(pos2->entryaddr.ai, layer2, &lx2, &hx2, &ly2, &hy2)) return(0);
		}

		/* crop out parts of a box covered by a similar layer on the other node */
		if (pos1->entrytype == OBJNODEINST)
		{
			if (dr_cropnodeinst(pos1->entryaddr.ni, layer2, net2, pos2, &lx2,&hx2, &ly2, &hy2))
				return(0);
		}
		if (pos2->entrytype == OBJNODEINST)
		{
			if (dr_cropnodeinst(pos2->entryaddr.ni, layer1, net1, pos1, &lx1,&hx1, &ly1, &hy1))
				return(0);
		}

		/* now compare the box extents */
		if (hx1+dist <= lx2 || lx1-dist >= hx2 || hy1+dist <= ly2 || ly1-dist >= hy2)
			return(0);
	} else
	{
		switch (poly1->style)
		{
			case FILLEDRECT:
			case CLOSEDRECT:
				maketruerect(poly1);
				break;
			case FILLED:
			case CLOSED:
			case CROSSED:
			case OPENED:
			case OPENEDT1:
			case OPENEDT2:
			case OPENEDT3:
			case VECTORS:
				break;
			default:
				return(0);
		}

		switch (poly2->style)
		{
			case FILLEDRECT:
			case CLOSEDRECT:
				maketruerect(poly2);
				break;
			case FILLED:
			case CLOSED:
			case CROSSED:
			case OPENED:
			case OPENEDT1:
			case OPENEDT2:
			case OPENEDT3:
			case VECTORS:
				break;
			default:
				return(0);
		}

		/* make sure polygons don't intersect */
		if (dr_polyintersect(poly1, poly2) == 0)
		{
			/* find distance between polygons */
			pd = dr_finddistance(poly1, poly2);
			if (pd >= dist) return(0);
		}
	}

	/*
	 * special case: ignore errors between two active layers connected
	 * to either side of a field-effect transistor that is inside of
	 * the error area.
	 */
	if (isbox1 != 0 && isbox2 != 0 &&
		dr_activeontransistor(pos1, layer1, net1, lx1,hx1,ly1,hy1,
			pos2, layer2, net2, lx2,hx2,ly2,hy2) != 0) return(0);

	/* check/describe the error */
	ret = dr_describeerror(layer1, pos1, layer2, pos2);
	if (ret == 0) return(0);
	(void)initinfstr();
	(void)addstringtoinfstr(ret);

	/* get technology for each box */
	if (pos1->entrytype == OBJARCINST)
		tech1 = pos1->entryaddr.ai->proto->tech; else
			tech1 = pos1->entryaddr.ni->proto->tech;
	if (pos2->entrytype == OBJARCINST)
		tech2 = pos2->entryaddr.ai->proto->tech; else
			tech2 = pos2->entryaddr.ni->proto->tech;

	/* add in verbosity if requested */
	var = getvalkey((INTBIG)dr_aid, VAID, VINTEGER, dr_verbose);
	if (var != NOVARIABLE && var->addr != 0)
	{
		if (net1 == NONETWORK || net2 == NONETWORK || net1 == net2)
			(void)addstringtoinfstr(".  Connected "); else
				(void)addstringtoinfstr(".  Unconnected ");
		(void)addstringtoinfstr("bad boxes are (");
		(void)addstringtoinfstr(latoa(lx1));
		(void)addstringtoinfstr("-");
		(void)addstringtoinfstr(latoa(hx1));
		(void)addstringtoinfstr(", ");
		(void)addstringtoinfstr(latoa(ly1));
		(void)addstringtoinfstr("-");
		(void)addstringtoinfstr(latoa(hy1));
		(void)addstringtoinfstr(") on layer ");
		(void)addstringtoinfstr(layername(tech1,layer1));
		(void)addstringtoinfstr(" of ");
		(void)addstringtoinfstr(geomname(pos1));
		(void)addstringtoinfstr(" AND (");
		(void)addstringtoinfstr(latoa(lx2));
		(void)addstringtoinfstr("-");
		(void)addstringtoinfstr(latoa(hx2));
		(void)addstringtoinfstr(", ");
		(void)addstringtoinfstr(latoa(ly2));
		(void)addstringtoinfstr("-");
		(void)addstringtoinfstr(latoa(hy2));
		(void)addstringtoinfstr(") on layer ");
		(void)addstringtoinfstr(layername(tech2,layer2));
		(void)addstringtoinfstr(" of ");
		(void)addstringtoinfstr(geomname(pos2));
	}

	/* if a node was too small to crop an arc, it may explain the error */
	if (dr_tinynodeinst != NONODEINST)
	{
		/* see if the node/arc that failed was involved in the error */
		if ((dr_tinynodeinst->geom==pos1 || dr_tinynodeinst->geom==pos2) &&
			(dr_tinyobj == pos1 || dr_tinyobj == pos2))
		{
			(void)addstringtoinfstr(".  ");
			(void)addstringtoinfstr(describenodeinst(dr_tinynodeinst));
			(void)addstringtoinfstr(" is too small for the ");
			(void)addstringtoinfstr(geomname(dr_tinyobj));
		}
	}

	var = getvalkey((INTBIG)dr_aid, VAID, VINTEGER, dr_pointout);
	if (var == NOVARIABLE || var->addr <= 0)
	{
		ttyputerr("%s", returninfstr());
	} else
	{
		/* save highlight features if this is the first error */
		(void)askaid(us_aid, "clear");
		(void)askaid(us_aid, "show-area", lx1,hx1,ly1,hy1);     /*SRP01*/
		(void)askaid(us_aid, "show-area", lx2,hx2,ly2,hy2);
		flushscreen();
		(void)setvalkey((INTBIG)dr_aid, VAID, dr_pointout, var->addr+1, VINTEGER|VDONTSAVE);
		i = ttygetparam(returninfstr(), &us_drcerrp, 1, pars);
		if (i <= 0) return(1);
		len = strlen(pars[0]);
		if (namesamen(pars[0], "quit-checking", len) == 0) return(-1);
		if (namesamen(pars[0], "continue-silently", len) == 0)
		{
			(void)setvalkey((INTBIG)dr_aid, VAID, dr_pointout, -2, VINTEGER|VDONTSAVE);
			return(1);
		}
		if (namesamen(pars[0], "ignore-error", len) == 0)
		{
			/* mark objects "pos1" and "pos2" as ok */
			dr_setignore(pos1, pos2);
			dr_setignore(pos2, pos1);
			ttyputmsgf("Ignored");
			return(1);
		}
	}
	return(1);
}

/*
 * routine to see if the two boxes are active elements, connected to opposite
 * sides of a field-effect transistor that resides inside of the box area.
 * Returns nonzero if so.
 */
INTSML dr_activeontransistor(GEOM *pos1, INTSML layer1, NETWORK *net1, INTBIG lx1,
	INTBIG hx1, INTBIG ly1, INTBIG hy1, GEOM *pos2, INTSML layer2, NETWORK *net2,
	INTBIG lx2, INTBIG hx2, INTBIG ly2, INTBIG hy2)
{
	REGISTER INTBIG fun, sea;
	REGISTER INTSML on1, on2;
	REGISTER GEOM *g;
	REGISTER NODEINST *ni;
	REGISTER PORTARCINST *pi;
	REGISTER TECHNOLOGY *tech1, *tech2;

	/* networks must be different */
	if (net1 == net2) return(0);

	/* get technology for each box */
	if (pos1->entrytype == OBJARCINST)
		tech1 = pos1->entryaddr.ai->proto->tech; else
			tech1 = pos1->entryaddr.ni->proto->tech;
	if (pos2->entrytype == OBJARCINST)
		tech2 = pos2->entryaddr.ai->proto->tech; else
			tech2 = pos2->entryaddr.ni->proto->tech;

	/* layers must be active */
	fun = layerfunction(tech1, layer1) & LFTYPE;
	if (fun != APDIFF && fun != APDIFFP && fun != APDIFFN && fun != APDIFFS && fun != APDIFFW)
		return(0);
	fun = layerfunction(tech2, layer2) & LFTYPE;
	if (fun != APDIFF && fun != APDIFFP && fun != APDIFFN && fun != APDIFFS && fun != APDIFFW)
		return(0);

	/* both layers are active: see if there is a field-effect transistor on them */
	sea = initsearch(mini(lx1,lx2), maxi(hx1,hx2), mini(ly1,ly2), maxi(hy1,hy2), geomparent(pos1));
	if (sea < 0) return(0);
	for(;;)
	{
		g = nextobject(sea);
		if (g == NOGEOM) break;
		if (!isfet(g)) continue;

		/* got a transistor */
		ni = g->entryaddr.ni;
		on1 = on2 = 0;
		for(pi=ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
		{
			if (pi->conarcinst->network == net1) on1++;
			if (pi->conarcinst->network == net2) on2++;
		}
		if (on1 == 0 || on2 == 0) continue;

		/* transistor found that connects to both actives */
		termsearch(sea);
		return(1);
	}
	return(0);
}

/*
 * routine to return nonzero if polygons "poly1" and "poly2" intersect (that is,
 * if any of their lines intersect).
 */
INTSML dr_polyintersect(POLYGON *poly1, POLYGON *poly2)
{
	INTBIG lx1, hx1, ly1, hy1, lx2, hx2, ly2, hy2;
	REGISTER INTBIG px, py, tx, ty;
	REGISTER INTSML i;

	/* quit now if bounding boxes don't overlap */
	getbbox(poly1, &lx1, &hx1, &ly1, &hy1);
	getbbox(poly2, &lx2, &hx2, &ly2, &hy2);
	if (hx1 < lx2 || hx2 < lx1 || hy1 < ly2 || hy2 < ly1) return(0);

	/* check each line in polygon 1 */
	for(i=0; i<poly1->count; i++)
	{
		if (i == 0)
		{
			if (poly1->style == OPENED || poly1->style == OPENEDT1 || poly1->style == OPENEDT2 ||
				poly1->style == OPENEDT3 || poly1->style == VECTORS) continue;
			px = poly1->xv[poly1->count-1];
			py = poly1->yv[poly1->count-1];
		} else
		{
			px = poly1->xv[i-1];
			py = poly1->yv[i-1];
		}
		tx = poly1->xv[i];
		ty = poly1->yv[i];
		if (poly1->style == VECTORS && (i&1) != 0) i++;
		if (px == tx && py == ty) continue;

		/* compare this line with polygon 2 */
		if (mini(px,tx) > hx2 || maxi(px,tx) < lx2 ||
			mini(py,ty) > hy2 || maxi(py,ty) < ly2) continue;
		if (dr_lineintersect(px,py, tx,ty, poly2) != 0) return(1);
	}
	return(0);
}

/*
 * routine to return nonzero if the line segment from (px1,py1) to (tx1,ty1)
 * intersects any line in polygon "poly"
 */
INTSML dr_lineintersect(INTBIG px1, INTBIG py1, INTBIG tx1, INTBIG ty1, POLYGON *poly)
{
	REGISTER INTBIG px2, py2, tx2, ty2, ang, ang1, ang2;
	INTBIG ix, iy;
	REGISTER INTSML i;

	for(i=0; i<poly->count; i++)
	{
		if (i == 0)
		{
			if (poly->style == OPENED || poly->style == OPENEDT1 || poly->style == OPENEDT2 ||
				poly->style == OPENEDT3 || poly->style == VECTORS) continue;
			px2 = poly->xv[poly->count-1];
			py2 = poly->yv[poly->count-1];
		} else
		{
			px2 = poly->xv[i-1];
			py2 = poly->yv[i-1];
		}
		tx2 = poly->xv[i];
		ty2 = poly->yv[i];
		if (poly->style == VECTORS && (i&1) != 0) i++;
		if (px2 == tx2 && py2 == ty2) continue;

		/* special case: this line is vertical */
		if (px2 == tx2)
		{
			/* simple bounds check */
			if (mini(px1,tx1) > px2 || maxi(px1,tx1) < px2) continue;

			if (px1 == tx1)
			{
				if (mini(py1,ty1) > maxi(py2,ty2) || maxi(py1,ty1) < mini(py2,ty2)) continue;
				return(1);
			}
			if (py1 == ty1)
			{
				if (mini(py2,ty2) > py1 || maxi(py2,ty2) < py1) continue;
				return(1);
			}
			ang = figureangle(px1,py1, tx1,ty1);
			(void)intersect(px2,py2, 900, px1,py1, ang, &ix, &iy);
			if (ix != px2 || iy < mini(py2,ty2) || iy > maxi(py2,ty2)) continue;
			return(1);
		}

		/* special case: this line is horizontal */
		if (py2 == ty2)
		{
			/* simple bounds check */
			if (mini(py1,ty1) > py2 || maxi(py1,ty1) < py2) continue;

			if (py1 == ty1)
			{
				if (mini(px1,tx1) > maxi(px2,tx2) || maxi(px1,tx1) < mini(px2,tx2)) continue;
				return(1);
			}
			if (px1 == tx1)
			{
				if (mini(px2,tx2) > px1 || maxi(px2,tx2) < px1) continue;
				return(1);
			}
			ang = figureangle(px1,py1, tx1,ty1);
			(void)intersect(px2,py2, 0, px1,py1, ang, &ix, &iy);
			if (iy != py2 || ix < mini(px2,tx2) || ix > maxi(px2,tx2)) continue;
			return(1);
		}

		/* simple bounds check */
		if (mini(px1,tx1) > maxi(px2,tx2) || maxi(px1,tx1) < mini(px2,tx2) ||
			mini(py1,ty1) > maxi(py2,ty2) || maxi(py1,ty1) < mini(py2,ty2)) continue;

		/* general case of line intersection */
		ang1 = figureangle(px1,py1, tx1,ty1);
		ang2 = figureangle(px2,py2, tx2,ty2);
		(void)intersect(px2,py2, ang2, px1,py1, ang1, &ix, &iy);
		if (ix < mini(px2,tx2) || ix > maxi(px2,tx2) || iy < mini(py2,ty2) || iy > maxi(py2,ty2) ||
			ix < mini(px1,tx1) || ix > maxi(px1,tx1) || iy < mini(py1,ty1) || iy > maxi(py1,ty1))
				continue;
		return(1);
	}
	return(0);
}

INTBIG dr_finddistance(POLYGON *poly1, POLYGON *poly2)
{
	REGISTER INTSML i;
	INTBIG cx, cy;
	REGISTER INTBIG pd, minpd;

	/* look at all points on polygon 1 */
	for(i=0; i<poly1->count; i++)
	{
		cx = poly1->xv[i];   cy = poly1->yv[i];
		closestpoint(poly2, &cx, &cy);
		pd = computedistance(poly1->xv[i], poly1->yv[i], cx, cy);
		if (pd <= 0) return(0);
		if (i == 0) minpd = pd; else
		{
			if (pd < minpd) minpd = pd;
		}
	}

	/* look at all points on polygon 2 */
	for(i=0; i<poly2->count; i++)
	{
		cx = poly2->xv[i];   cy = poly2->yv[i];
		closestpoint(poly1, &cx, &cy);
		pd = computedistance(poly2->xv[i], poly2->yv[i], cx, cy);
		if (pd <= 0) return(0);
		if (pd < minpd) minpd = pd;
	}
	return(minpd);
}

/*
 * routine to add DRC ignore information to object "pos1" so that it will
 * ignore errors with object "pos2"
 */
void dr_setignore(GEOM *pos1, GEOM *pos2)
{
	REGISTER GEOM **ignorelist, *p1, *p2;
	GEOM *dummylist[2];
	REGISTER VARIABLE *var;
	REGISTER INTSML i, len;
	REGISTER NODEPROTO *np;

	np = geomparent(pos1);
	var = getvalkey((INTBIG)np, VNODEPROTO, VGEOM|VISARRAY, dr_ignore_list);

	/* if the list is empty: create it with one entry */
	if (var == NOVARIABLE)
	{
		dummylist[0] = pos1;   dummylist[1] = pos2;
		(void)setvalkey((INTBIG)np, VNODEPROTO, dr_ignore_list, (INTBIG)dummylist,
			VGEOM|VISARRAY|(2<<VLENGTHSH)|VDONTSAVE);
		return;
	}

	/* search list to see if this entry is on it */
	len = getlength(var);
	for(i=0; i<len; i += 2)
	{
		p1 = ((GEOM **)var->addr)[i];
		p2 = ((GEOM **)var->addr)[i+1];
		if (pos1 == p1 && pos2 == p2) return;
		if (pos1 == p2 && pos2 == p1) return;
	}

	/* entry not in list: add it */
	ignorelist = (GEOM **)emalloc(((len+2) * (sizeof (GEOM *))), el_tempcluster);
	if (ignorelist == 0)
	{
		ttyputerr("No memory for error ignoring");
		return;
	}

	/* update the list */
	for(i=0; i<len; i++) ignorelist[i] = ((GEOM **)var->addr)[i];
	ignorelist[len] = pos1;   ignorelist[len+1] = pos2;
	(void)setvalkey((INTBIG)np, VNODEPROTO, dr_ignore_list, (INTBIG)ignorelist,
		(INTBIG)(VGEOM|VISARRAY|((len+2)<<VLENGTHSH)|VDONTSAVE));
	efree((char *)ignorelist);
}

/*
 * routine to delete DRC ignore information on object "pos"
 */
void dr_unignore(GEOM *pos)
{
	REGISTER GEOM **ignorelist, *p1, *p2;
	REGISTER VARIABLE *var;
	REGISTER INTSML i, len, pt;
	REGISTER NODEPROTO *np;

	np = geomparent(pos);
	var = getvalkey((INTBIG)np, VNODEPROTO, VGEOM|VISARRAY, dr_ignore_list);

	/* if the list is empty there is nothing to do */
	if (var == NOVARIABLE) return;

	/* see if this entry is mentioned in the list */
	len = getlength(var);
	for(i=0; i<len; i++) if (((GEOM **)var->addr)[i] == pos) break;
	if (i >= len) return;

	/* create a new list without the entry */
	ignorelist = (GEOM **)emalloc((len * (sizeof (GEOM *))), el_tempcluster);
	if (ignorelist == 0)
	{
		ttyputerr("No memory for error ignoring");
		return;
	}

	/* search list and remove entries that mention this module */
	pt = 0;
	for(i=0; i<len; i += 2)
	{
		p1 = ((GEOM **)var->addr)[i];
		p2 = ((GEOM **)var->addr)[i+1];
		if (p1 == pos || p2 == pos) continue;
		ignorelist[pt++] = p1;   ignorelist[pt++] = p2;
	}

	/* set the list back in place */
	if (pt > 0) (void)setvalkey((INTBIG)np, VNODEPROTO, dr_ignore_list,
		(INTBIG)ignorelist, (INTBIG)(VGEOM|VISARRAY|(pt<<VLENGTHSH)|VDONTSAVE)); else
			(void)delvalkey((INTBIG)np, VNODEPROTO, dr_ignore_list);
	efree((char *)ignorelist);
}

/*
 * routine to crop the box on layer "nlayer", electrical index "nindex"
 * and bounds (lx-hx, ly-hy) against the nodeinst "ni".  Only those layers
 * in the nodeinst that are the same layer and the same electrical network
 * are checked.  The routine returns nonzero if the bounds are reduced
 * to nothing.
 */
INTSML dr_cropnodeinst(NODEINST *ni, INTSML nlayer, NETWORK *nnet, GEOM *npos, INTBIG *lx,
	INTBIG *hx, INTBIG *ly, INTBIG *hy)
{
	XARRAY trans;
	INTBIG xl, xh, yl, yh, bx, ux, by, uy;
	REGISTER INTSML tot, j, isconnected;
	REGISTER INTBIG temp;
	REGISTER NETWORK *net;
	static POLYGON *poly = NOPOLYGON;

	/* make sure there is a polygon */
	if (poly == NOPOLYGON) poly = allocpolygon(4, dr_aid->cluster);

	tot = nodeEpolys(ni);
	isconnected = 0;
	for(j=0; j<tot; j++)
	{
		shapeEnodepoly(ni, j, poly);
		if (poly->layer != nlayer) continue;
		if (nnet != NONETWORK)
		{
			if (poly->portproto == NOPORTPROTO) continue;
			net = dr_network(ni, poly->portproto);
			if (net != NONETWORK && net != nnet) continue;
		}
		isconnected++;
		break;
	}
	if (isconnected == 0) return(0);

	tot = nodepolys(ni);
	makerot(ni, trans);
	for(j=0; j<tot; j++)
	{
		/* get the description of the nodeinst layer */
		shapenodepoly(ni, j, poly);
		if (poly->layer != nlayer) continue;

		/* warning: does not handle arbitrary polygons, only boxes */
		if (isbox(poly, &xl, &xh, &yl, &yh) == 0) continue;
		xform(xl,yl, &bx,&by, trans);
		xform(xh,yh, &ux,&uy, trans);
		if (bx > ux) { temp = bx; bx = ux; ux = temp; }
		if (by > uy) { temp = by; by = uy; uy = temp; }
		temp = cropbox(lx,hx,ly,hy, bx,ux,by,uy);
		if (temp > 0) return(1);
		if (temp < 0) { dr_tinynodeinst = ni;   dr_tinyobj = npos; }
	}
	return(0);
}

/*
 * routine to crop away any part of layer "lay" of arcinst "ai" that coincides
 * with a similar layer on a connecting nodeinst.  The bounds of the arcinst
 * are in the reference parameters (lx-hx, ly-hy).  The routine returns zero
 * normally, 1 if the arcinst is cropped into oblivion.
 */
INTSML dr_croparcinst(ARCINST *ai, INTSML lay, INTBIG *lx, INTBIG *hx, INTBIG *ly, INTBIG *hy)
{
	INTBIG bx, by, ux, uy, xl, xh, yl, yh;
	XARRAY trans;
	REGISTER INTSML i, j, tot;
	REGISTER INTBIG temp;
	REGISTER NODEINST *ni;
	REGISTER NODEPROTO *np;
	REGISTER PORTPROTO *pp;
	static POLYGON *poly = NOPOLYGON;

	/* make sure there is a polygon */
	if (poly == NOPOLYGON) poly = allocpolygon(4, dr_aid->cluster);

	for(i=0; i<2; i++)
	{
		/* find the primitive nodeinst at the true end of the portinst */
		ni = ai->end[i].nodeinst;   np = ni->proto;
		pp = ai->end[i].portarcinst->proto;
		while (np->index == 0)
		{
			ni = pp->subnodeinst;   np = ni->proto;
			pp = pp->subportproto;
		}

		makerot(ni, trans);
		tot = nodepolys(ni);
		for(j=0; j<tot; j++)
		{
			shapenodepoly(ni, j, poly);
			if (poly->layer != lay) continue;

			/* warning: does not handle arbitrary polygons, only boxes */
			if (isbox(poly, &xl, &xh, &yl, &yh) == 0) continue;
			xform(xl,yl, &bx,&by, trans);
			xform(xh,yh, &ux,&uy, trans);
			if (bx > ux) { temp = bx; bx = ux; ux = temp; }
			if (by > uy) { temp = by; by = uy; uy = temp; }

			temp = cropbox(lx,hx,ly,hy, bx,ux,by,uy);
			if (temp > 0) return(1);
			if (temp < 0) { dr_tinynodeinst = ni;   dr_tinyobj = ai->geom; }
		}
	}
	return(0);
}

char *dr_describeerror(INTSML layer1, GEOM *pos1, INTSML layer2, GEOM *pos2)
{
	static char line[200];
	REGISTER TECHNOLOGY *tech1, *tech2;
	REGISTER INTSML len, i, dontsaylayer;
	REGISTER VARIABLE *var;
	REGISTER NODEPROTO *np;
	REGISTER GEOM *p1, *p2;

	/* first see if this is to be ignored */
	np = geomparent(pos1);
	var = getvalkey((INTBIG)np, VNODEPROTO, VGEOM|VISARRAY, dr_ignore_list);
	if (var != NOVARIABLE)
	{
		len = getlength(var);
		for(i=0; i<len; i += 2)
		{
			p1 = ((GEOM **)var->addr)[i];
			p2 = ((GEOM **)var->addr)[i+1];
			if (p1 == pos1 && p2 == pos2) return(0);
			if (p1 == pos2 && p2 == pos1) return(0);
		}
	}

	/* now determine technologies of these objects */
	if (pos1->entrytype == OBJARCINST)
		tech1 = pos1->entryaddr.ai->proto->tech; else
			tech1 = pos1->entryaddr.ni->proto->tech;
	if (pos2->entrytype == OBJARCINST)
		tech2 = pos2->entryaddr.ai->proto->tech; else
			tech2 = pos2->entryaddr.ni->proto->tech;

	/* layers too close: error */
	if (pos1->entrytype == OBJNODEINST) (void)strcpy(line, "Node "); else
		(void)strcpy(line, "Arc ");
	(void)strcat(line, geomname(pos1));
	if (layer1 == layer2)
	{
		(void)strcat(line, " and ");
		if (pos2->entrytype == OBJNODEINST) (void)strcat(line, "node "); else
			(void)strcat(line, "arc ");
		(void)strcat(line, geomname(pos2));
		(void)strcat(line, " too close");
		if (pos1->entrytype != OBJARCINST && pos2->entrytype != OBJARCINST) dontsaylayer = 0; else
			dontsaylayer = 1;
		if (dontsaylayer != 0)
		{
			if (pos1->entrytype == OBJARCINST && arcpolys(pos1->entryaddr.ai) != 1)
				dontsaylayer = 0;
			if (pos2->entrytype == OBJARCINST && arcpolys(pos2->entryaddr.ai) != 1)
				dontsaylayer = 0;
		}
		if (dontsaylayer == 0)
		{
			(void)strcat(line, " on layer ");
			(void)strcat(line, layername(tech1, layer1));
		}
	} else
	{
		if (pos1->entrytype != OBJARCINST || arcpolys(pos1->entryaddr.ai) != 1)
		{
			(void)strcat(line, ", layer ");
			(void)strcat(line, layername(tech1, layer1));
		}
		(void)strcat(line, " too close to ");
		if (pos2->entrytype == OBJNODEINST) (void)strcat(line, "node "); else
			(void)strcat(line, "arc ");
		(void)strcat(line, geomname(pos2));
		if (pos2->entrytype != OBJARCINST || arcpolys(pos2->entryaddr.ai) != 1)
		{
			(void)strcat(line, ", layer ");
			(void)strcat(line, layername(tech2, layer2));
		}
	}
	return(line);
}

/*
 * routine to tell whether the objects at geometry modules "po1" and "po2"
 * touch directly (that is, an arcinst connected to a nodeinst).  The routine
 * returns nonzero if they touch
 */
INTSML dr_objtouch(GEOM *po1, GEOM *po2)
{
	REGISTER GEOM *temp;
	REGISTER NODEINST *ni;
	REGISTER ARCINST *ai;
	REGISTER INTSML i;

	if (po1->entrytype == OBJNODEINST)
	{
		if (po2->entrytype == OBJNODEINST) return(0);
		temp = po1;   po1 = po2;   po2 = temp;
	}
	if (po2->entrytype == OBJARCINST) return(0);

	/* see if the arcinst at "po1" touches the nodeinst at "po2" */
	ni = po2->entryaddr.ni;
	ai = po1->entryaddr.ai;
	for(i=0; i<2; i++) if (ai->end[i].nodeinst == ni) return(1);
	return(0);
}

/*
 * routine to tell the network associated with nodeinst "ni", portproto "ppt"
 */
NETWORK *dr_network(NODEINST *ni, PORTPROTO *ppt)
{
	REGISTER NETWORK *net;

	net = getnetonport(ni, ppt);
	if (net != NONETWORK) return(net);
	return((NETWORK *)(dr_pseudonet++));
}

/*
 * routine to determine the node prototype specified in "name" (given that
 * there were "count" parameters to the command and that "name" should be in
 * the second).  Returns NONODEPROTO if there is an error.
 */
NODEPROTO *dr_checkthisfacet(INTSML count, char *name)
{
	REGISTER NODEPROTO *np;

	if (count < 2)
	{
		np = getcurfacet();
		if (np == NONODEPROTO) ttyputerr("No current facet");
		return(np);
	}
	np = getnodeproto(name);
	if (np == NONODEPROTO)
	{
		ttyputerr("No facet named %s", name);
		return(NONODEPROTO);
	}
	if (np->index != 0)
	{
		ttyputerr("Can only check facets, not primitives");
		return(NONODEPROTO);
	}
	if (np->cell->lib != el_curlib)
	{
		ttyputerr("Can only check facets in the current library");
		return(NONODEPROTO);
	}
	return(np);
}

/*
 * routine to accumulate a list of polygons at least "tot" long in the
 * polygon structure "list".  The routine returns nonzero if there is no
 * more memory.
 */
INTSML dr_ensurepolylist(struct plist *list, INTSML tot)
{
	REGISTER POLYGON **newpolylist;
	REGISTER INTSML j;

	/* make sure the list is the right size */
	if (list->polylistsize < tot)
	{
		newpolylist = (POLYGON **)emalloc((tot * (sizeof (POLYGON *))), dr_aid->cluster);
		if (newpolylist == 0)
		{
			ttyputerr("No memory for arc DRC");
			return(1);
		}
		for(j=0; j<tot; j++) newpolylist[j] = NOPOLYGON;
		if (list->polylistsize != 0)
		{
			for(j=0; j<list->polylistsize; j++)
				newpolylist[j] = list->polygons[j];
			efree((char *)list->polygons);
		}
		list->polygons = newpolylist;
		list->polylistsize = tot;
	}

	/* make sure there is a polygon for each entry in the list */
	for(j=0; j<tot; j++)
	{
		if (list->polygons[j] != NOPOLYGON) continue;
		list->polygons[j] = allocpolygon(4, dr_aid->cluster);
	}
	return(0);
}

#endif  /* DRCAID - at top */
