/*
 * Electric(tm) VLSI Design System
 *
 * File: drcbatch.c
 * Batch design-rule check aid
 * Written by: Mark Moraes, University of Toronto
 *
 * Copyright (c) 1998 Electric Editor Incorporated.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Electric Editor Incorporated
 * 23470 Sunset Drive, Suite 108
 * Los Gatos, California 95033
 * support@electriceditor.com
 */

/*
 * !! The enclosure rules aren't proper. Need additional flags to
 * indicate whether a layer MUST be enclosed by another layer, or
 * whether the rule applies only IF it is enclosed by another layer. For
 * example, any contact cut MUST be enclosed by metal. However, it may
 * be enclosed by either diffusion or polysilicon or device-well, and
 * the rules for any of those only apply if it is enclosed by one of
 * them. Similarly, a via MUST be enclosed by metal and by metal2.
 * Dealing with enclosures implies dealing with spurious errors properly
 * -- a shape on one layer may have it's enclosure requirements
 * satisfied by two or more shapes on a different layer.
 *
 * !! From an error reporting viewpoint, it would be nice if we reported
 * not just the global network number, which is likely meaningless to
 * the user, but the network name, filtered down from the highest
 * level.  Would be interesting to keep a global array of those names --
 * for a large circuit, that table would be large -- of the order of
 * 20000 names.
 */

#include "config.h"
#if DRCAID

#include "global.h"
#include "efunction.h"
#include "drc.h"
#include "database.h"
#include "tech.h"
#include "egraphics.h"
#include "usr.h"

/* #define OTHERRULES 1 */		/* uncomment for min/max size and enclosure rules */

/*********************************** HASH TABLE ***********************************/

typedef struct
{
	char *key;
	char *datum;
} Hashrec;

typedef struct
{
	Hashrec *table;
	int size;
	int probe;
} Hashtable;

#define HASH_FULL   -2
#define HASH_EXISTS -1

#define HASH_NULLPROC	((char * (*)(char*, char*, char*)) 0)
#define HASH_NULL	((char *) 0)

/*
 * The four Hash functions consider key to be an arbitrary pointer --
 * the hash function is simply a modulo of the hash table size.
 * hashinsert will cause the table to grow if it isn't large enough.
 */

/*
 * The next two insert character strings by applying Ron Unrau's hash
 * function to the string.  These will eventually vanish, and the hash
 * function will be exported, and/or the hashinsert/hashsearch
 * interfaces modified to handle this case.
 */
extern char *hashstrsearch(/* Hashtable hash, char *key */);
extern int hashstrinsert(/* Hashtable hash,char *key,char *datum,int replace */);

/******************************** END HASH TABLE ********************************/

typedef struct
{
	POLYGON **polygons;	/* polygons to be checked on node or arc */
	INTSML polylistsize;	/* size of above list */
} POLYLIST;

/*
 * The STATE structure contains all the global/static data that used to
 * be scattered around in the DRC procedures. We need one STATE per
 * processor, and all procedures with static writable data of any form
 * must get it from here, so they must be passed this structure. The
 * ONLY safe alternative to this is to lock on every access to those
 * structures. Multiprocessors are such fun... Also fix alloc_state() in
 * ndrc.c
 */
typedef struct
{
	/*
	 * used by checkarcinst and checknodeinst, also by
	 * getnodeinstshapes and getarcinstshapes
	 */
	POLYLIST *polylist;
	/*
	 * used by badbox, which is called by checkarcinst, checknodeinst,
	 * and check_shape
	 */
	POLYLIST *subpolylist;
	/*
	 * cropnodeinst and croparcinst can share one polylist, because they
	 * don't call each other.  These two are called by checkdist, which
	 * is called by badbox.
	 */
	POLYLIST *croppolylist;
	NODEINST *tinynodeinst;
	GEOM *tinyobj;
	INTBIG netindex;
	/* just so we keep count how many tasks each processor finishes */
	INTBIG ntasks;
	/* So we can debug individual threads without turning on the whole web */
	int debughook;
	/* Reduces the number of args we pass drcb_badbox */
	Hashtable *hni, *hai;
	Hashtable *nhpp;
} STATE;

/*
 * temp1 in a NODEPROTO is non-zero if the proto has been checked, zero
 * if it has not been checked yet. temp1 in the intersect lists (stored
 * in a dummy NODEPROTO structure) has a count of the number of
 * intersected elements.
 */
typedef struct Ishape
{
	INTSML layer;
	INTSML entrytype;
	union
	{
		NODEINST *ni;
		ARCINST *ai;
		INTBIG blind;
	} entryaddr;
	INTBIG net;
	Hashtable *hpp;
} SHAPE;

/*
 * For now, we ignore any layer interactions greater than 50 microns.
 * For ncmos3, this cuts out overglass, which has an abnormally large
 * interaction distance, and is only needed on the pads. Perhaps a
 * separate pass just to check overglass? (Overglass to metal is 44
 * microns, overglass to overglass is 135 microns, all other rules are 5
 * - 10 microns, except P-Well to P+Mask in N-Sub which is 14 microns)
 */
#define MAX_DRID_HACK 50.0  	/* microns */

/*
 * If a port is unconnected, then we can't call it NONETWORK, but we
 * want to give it a unique network number as a NETWORK * so routines
 * can distinguish between shapes on different networks -- so we start
 * from 1 and go upto MAX_PSEUDONET. Note that dereferencing these
 * pseudo-network pointers is strictly verboten since they point into
 * text space, usually. Maybe we should use etext or edata instead?
 */
#define MAX_PSEUDONET ((NETWORK *) 1000000)

#define NONET ((INTBIG) -1)

/*
 * Initial sizes for various hashtables.  This is chosen so it will not
 * have to grow the hashtable for the common case, only for cases with a
 * lot more entries than normal
 */
#define HTABLE_NI	    71
#define HTABLE_PRIMITIVES   71
#define HTABLE_ARC	    71
#define HTABLE_PORT	    113


typedef struct
{
	STATE *state;
	NODEINST *ni;
	XARRAY vtrans;
} CheckShapeArg;

typedef struct
{
	INTBIG lx, hx, ly, hy;
} Box;

extern AIDENTRY *dr_aid;		/* the DRC aid object */
static char *debugstates[] = {"off", "on"};
static INTSML debugflag = 0;
static INTSML vdebugflag = 0;
static TECHNOLOGY *drcb_maintech = NOTECHNOLOGY;
static CELL *drcb_cell = NOCELL;
static INTBIG drcb_surround;		/* Cached valued of maxdrid() */

/* prototypes for local routines */
static void dec_indent(void);
static INTSML get_indent(void);
static void set_indent(INTSML);
static char *print_inst_count(char*, char*, char*);
static void print_proto(NODEPROTO*, XARRAY);
static char *getnetname(NETWORK*);
static void drcb_report_error(INTBIG, INTBIG, INTBIG, INTBIG, GEOM*, GEOM*, INTSML, INTSML, INTBIG, INTBIG, INTSML, char*);
static void drcb_free_errors(void);
static void count_shapes(NODEPROTO*);
INTSML drcb_init(NODEPROTO*);
static void check_proto(STATE*, NODEPROTO*);
void drcb_check_all_protos(STATE*, NODEPROTO*);
static void check_node(STATE*, NODEINST*, XARRAY, Hashtable*, Hashtable*, NODEPROTO*);
static INTBIG maxdrid(TECHNOLOGY*);
static NODEPROTO *find_intersecting_elements(STATE*, NODEINST*, INTBIG, INTBIG, INTBIG, INTBIG, XARRAY, Hashtable*, Hashtable*, NODEPROTO*);
static void xformbox(INTBIG*, INTBIG*, INTBIG*, INTBIG*, XARRAY);
static void intersect_subtree(STATE*, NODEPROTO*, NODEPROTO*, XARRAY, XARRAY, INTBIG, INTBIG, INTBIG, INTBIG, Hashtable*);
static void getnodeinstshapes(STATE*, NODEPROTO*, NODEINST*, XARRAY, INTBIG, INTBIG, INTBIG, INTBIG, Hashtable*);
static void getarcinstshapes(STATE*, NODEPROTO*, ARCINST*, XARRAY, INTBIG, INTBIG, INTBIG, INTBIG, Hashtable*);
char *drcb_pr_nodenettable(char*, char*, char*);
char *drcb_pr_arcnettable(char*, char*, char*);
char *drcb_pr_ninettable(char*, char*, char*);
static INTBIG drcb_network(Hashtable*, char*, char*);
static void free_intersecting_elements(NODEPROTO*);
static void drcb_linkgeom(GEOM*, NODEPROTO*);
static void check_shape(GEOM*, char*);
static void do_ndrcinst(STATE*, NODEINST*, NODEPROTO*, Hashtable*, Hashtable*);
static char *freehpp(char*, char*, char*);
static void do_ndrcproto(STATE*, NODEPROTO*, Hashtable*, Hashtable*);
static INTSML drcb_checknodeinst(STATE*, NODEINST*, INTSML, Hashtable*);
static INTSML drcb_checkarcinst(STATE*, ARCINST*, INTSML);
static INTSML drcb_badbox(STATE*, GEOM*, INTSML, TECHNOLOGY*, NODEPROTO*, Box*, Box*, INTBIG, INTSML);
static INTSML drcb_checkdist(STATE*, INTSML, INTBIG, GEOM*, Box*, Box*, INTSML, INTBIG, GEOM*, Box*, INTBIG, Hashtable*, Hashtable*);
static INTSML drcb_cropnodeinst(STATE*, NODEINST*, INTSML, INTBIG, GEOM*, Box*, Hashtable*);
static INTSML drcb_croparcinst(STATE*, ARCINST*, INTSML, Box*);
static INTSML drcb_compareboxes(STATE*, INTSML, INTBIG, GEOM*, Box*, Box*, INTSML, INTBIG, GEOM*, Box*, INTBIG);
static INTSML drcb_objtouch(GEOM*, GEOM*);
static void drcb_box(NODEINST*, INTBIG*, INTBIG*, INTBIG*, INTBIG*);
static INTSML allocstate(STATE*);
static void freestate(STATE*);
static void freertree(RTNODE*);
static void walkrtree(RTNODE*, void(*)(GEOM*, char*), char*);
static INTSML intersectbox(INTBIG*, INTBIG*, INTBIG*, INTBIG*, INTBIG, INTBIG, INTBIG, INTBIG);
static INTSML drcb_ensurepolylist(POLYLIST*, INTSML);
static NODEPROTO *drcb_ensurewindow(NODEPROTO*);
void tech_initdrcminsizes(void);
INTSML drcminenclosure(TECHNOLOGY*, INTSML, INTSML, INTBIG*);
INTBIG drcminsize(TECHNOLOGY*, INTSML);
INTBIG drcmaxsize(TECHNOLOGY*, INTSML);
void drcb_print_nets2(NODEPROTO*, INTBIG*);
void drcb_get_init_nets(NODEPROTO*, Hashtable*, INTBIG*);
void drcb_get_arc_networks(NODEPROTO*, Hashtable*, Hashtable*, INTBIG*);
void drcb_get_nodeinst_networks(NODEINST*, Hashtable*, Hashtable*, Hashtable*, INTBIG*);
void drcb_flatnet2(NODEPROTO*, Hashtable*, INTBIG*);
void drcb_flatprop2(ARCINST*, Hashtable*, char*);
static INTSML get_nodeEpolys(NODEINST*, POLYLIST*);
static INTSML get_nodepolys(NODEINST*, POLYLIST*);
static INTSML get_arcpolys(ARCINST*, POLYLIST*);
static int hashindex(char*);
char *hashwalk(Hashtable*, char*(*)(char*, char*, char*), char*);
void hashdestroy(Hashtable*, char*(*)(char*, char*, char*), char*);
void hashclear(Hashtable*);
static char *grow_hook(char*, char*, char*);
INTSML hashgrow(Hashtable*, INTSML);
Hashtable *hashcopy(Hashtable*);
Hashtable *hashcreate(INTSML);
static char *strgrow_hook(char*, char*, char*);
int hashstrgrow(Hashtable*, INTSML);
int hashstrinsert(Hashtable*, char*, char*, int);
char *hashstrsearch(Hashtable*, char*);
int hashinsert(Hashtable*, char*, char*, int);
static Hashrec *findhashrec(Hashtable*, char*);
char *hashsearch (Hashtable*, char*);
char *hashdelete(Hashtable*, char*, char**);
INTSML isprime(INTBIG);
INTBIG pick_prime(INTBIG);

/***************** RECURSIVE INDENT - FOR DEBUGGING ******************/

/* !! Should be an infstring or String - for now, restrict tree to MAXDEPTH */
#define MAXDEPTH 128
static char indent[128];
static INTSML nindent;

/* indent one level deeper */
static void inc_indent(void)
{
	if (nindent >= 0 && nindent < MAXDEPTH)
	{
		indent[nindent] = ' ';
	}
	nindent++;
}

/* back up one level of indentation */
static void dec_indent(void)
{
	nindent--;
	if (nindent >= 0 && nindent < MAXDEPTH)
	{
		indent[nindent] = 0;
	}
}

/*
 * returns indent value - why not use nindent? because this may break
 * out into a library later
 */
static INTSML get_indent(void)
{
	return nindent;
}

/* Sets indentation string to a new value, appropriately padding with NUL */
static void set_indent(INTSML i)
{
	INTSML newdep;

	if (i < 0) i = 0;
	newdep = i;
	if (newdep > MAXDEPTH)
		newdep = MAXDEPTH;
	if (nindent > i)
	{
		if (nindent < MAXDEPTH)
		{
			for(; nindent > newdep; nindent--)
				indent[nindent-1] = 0;
		}
	} else
	{
		for(; nindent < newdep; indent[nindent++] = ' ')
			;
	}
	nindent = i;
}

/************************ NDRC ************************/
/************************ DEBUGGING ************************/

static char *print_inst_count(char *key, char *datum, char *arg)
{
	unsigned long n = ((unsigned long) datum) >> 15;
	unsigned long nshapes = ((unsigned long) datum) - (n << 15);

	if (nshapes == 0)
	{
		ttyputmsg("%s%lu instance%s of %s", indent, n,
				  (n == 1) ? "" : "s", key);
	} else
	{
		ttyputmsg("%s%lu instance%s of %s, %lu shape%s", indent, n,
				  (n == 1) ? "" : "s", key, nshapes, (nshapes == 1) ? "": "s");
	}
	return NULL;
}

static void print_proto(NODEPROTO *np, XARRAY mytrans)
{
	NODEINST *ni;
	ARCINST *ai;
	XARRAY temptrans1, temptrans2, subtrans;
	Hashtable *ni_ht;
	Hashtable *ai_ht;
	INTBIG total = 0;
	unsigned long nelec;
	INTSML j;
	unsigned long  ntot;
	POLYGON *poly;
	char *npname = describenodeproto(np);

	if ((ni_ht = hashcreate(HTABLE_NI)) == NULL)
	{
		ttyputerr("cannot create ni hashtable for %s", describenodeproto(np));
		return;
	}
	if ((ai_ht = hashcreate(HTABLE_ARC)) == NULL)
	{
		ttyputerr("cannot create ai hashtable for %s", describenodeproto(np));
		hashdestroy(ni_ht, HASH_NULLPROC, HASH_NULL);
		return;
	}
	poly = allocpolygon(4, el_tempcluster);
	np->temp1++;    /* indicate that we've printed this cell and it's kids */
	/* first make sure all the child nodes have been printed already */
	for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		if (ni->proto->index == 0)
		{
			/* Complex node - recurse */
			if (ni->proto->temp1 == 0)
			{
				/*
				 * Compute transformation matrix for this instance that maps
				 * objects back to the prototype being displayed.
				 */
				makerot(ni, temptrans1);
				transmult(temptrans1, mytrans, temptrans2);
				maketrans(ni, temptrans1);
				transmult(temptrans1, temptrans2, subtrans);
				print_proto(ni->proto, subtrans);
			}
		}
	}
	/* now print this nodeproto */
	ttyputmsg("%sprototype %s", indent, describenodeproto(np));
	inc_indent();
	total = 0;
	/* go through counting instances */
	for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		char *key = describenodeinst(ni);
		char *cp = hashstrsearch(ni_ht, key);
		unsigned long num = ((unsigned long) cp) >> 15;
		unsigned long nshapes = ((unsigned long) cp) - (num << 15);
		INTSML ret;

		if (ni->proto->index == 0)
		{
			/* Complex node */
			total += ni->proto->temp2;
		} else
		{
			/* primitive node, count only shapes with a layer number */
			ntot = nodeEpolys(ni);
			for(nelec = 0, j = 0; j < ntot; j++)
			{
				shapeEnodepoly(ni, j, poly);
				if (poly->layer < 0) continue;
				nelec++;
			}
			nshapes += nelec;
			total += nelec;
		}
		/*
		 * Printing out each nodeinst is much too expensive. (well,
		 * takes forever, and doesn't give much info) So we count
		 * elements of each type and print that count later
		 */
		ret = hashstrinsert(ni_ht, key, (char *) (((num + 1) << 15) + nshapes), 1);
		if (ret != 0)
		{
			ttyputerr("unable to insert %s in ni_ht -- ret = %d, cp = %d", key, ret, (INTBIG) cp);
			return;
		}
	}
	/* Now count arc shapes */
	for(ai = np->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
	{
		char *key = describearcinst(ai);
		char *cp = hashstrsearch(ai_ht, key);
		unsigned long num = ((unsigned long) cp) >> 15;
		unsigned long nshapes = ((unsigned long) cp) - (num << 15);
		INTSML ret;

		ntot = arcpolys(ai);
		for(nelec = 0, j = 0; j < ntot; j++)
		{
			shapearcpoly(ai, j, poly);
			if (poly->layer < 0) continue;
			nelec++;
		}
		nshapes += nelec;
		total += nelec;
		ret = hashstrinsert(ai_ht, key, (char *) (((num + 1) << 15) + nshapes), 1);
		if (ret != 0)
		{
			ttyputerr("unable to insert %s in ai_ht -- ret = %d, cp = %d", key, ret, (INTBIG) cp);
			return;
		}
	}
	/*
	 * store the total electrical shapes in temp2 for future use, print
	 * the scoop
	 */
	np->temp2 = total;
	ttyputmsg("%snode instances:", indent);
	(void) hashwalk(ni_ht, print_inst_count, NULL);
	hashdestroy(ni_ht, HASH_NULLPROC, HASH_NULL);
	ttyputmsg("%sarc instances:", indent);
	(void) hashwalk(ai_ht, print_inst_count, NULL);
	hashdestroy(ai_ht, HASH_NULLPROC, HASH_NULL);
	freepolygon(poly);

	ttyputmsg("%sprototype %s has %d electrical shapes", indent, npname, np->temp2);
	dec_indent();
	ttyputmsg("%send", indent);
}

/* walks the hierarchy printing cells along the way */
INTSML drcb_print_hierarchy(NODEPROTO *node)
{
	NODEPROTO *np;
	INTBIG netindex;

	if (node == NONODEPROTO)
	{
		ttyputerr("No facet to print");
		return(1);
	}
	for(np = el_curlib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
	{
		np->temp1 = 0; /* Indicates that nodeproto has not been printed */
	}
	print_proto(node, el_matid);
	for(np = el_curlib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
	{
		np->temp1 = 0; /* Indicates that nodeproto has not been printed */
	}
	netindex = 1;
	drcb_print_nets2(node, &netindex);
	return 0;
}

static char *getnetname(NETWORK *net)
{
	static char netnamebuf[32];

	if (net < MAX_PSEUDONET)
	{
		(void) sprintf(netnamebuf, "#%08d", (INTBIG) net);
	} else
	{
		(void) sprintf(netnamebuf, "%s (0x%lx)", describenetwork(net),
					   (unsigned long) net);
	}
	return netnamebuf;
}

INTSML drcb_highlight_step(NODEPROTO *node)
{
	NODEINST *ni;
	ARCINST *ai;
	XARRAY trans;
	INTBIG lx, hx, ly, hy;
	INTSML j, n;
	NETWORK *net;
	char *netname;
	POLYGON *poly;

	if (node == NONODEPROTO)
	{
		ttyputerr("No facet to print");
		return(1);
	}
	(void) askaid(us_aid, "down-stack");
	(void) askaid(us_aid, "clear");
	poly = allocpolygon(4, el_tempcluster);
	drcb_maintech = whattech(node);

	for(ni = node->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		if (ni->proto->index == 0)
		{
			/* complex node - skip */
			continue;
		}
		makerot(ni, trans);
		n = nodeEpolys(ni);
		ttyputmsg("node instance %s (node proto %s, geom %s, %d polys",
			describenodeinst(ni), describenodeproto(ni->proto), geomname(ni->geom), n);
		(void) askaid(us_aid, "clear");
		(void) askaid(us_aid, "show-object", (INTBIG) ni->geom);
		flushscreen();
		(void) ttygetchar();
		for(j = 0; j < n; j++)
		{
			shapeEnodepoly(ni, j, poly);
			if (poly->layer < 0)
			{
				ttyputmsg("non-layer %d", poly->layer);
				continue;
			}
			/* find the global electrical net for this layer */
			if (poly->portproto != NOPORTPROTO)
			{
				net = dr_network(ni, poly->portproto);
			} else
				net = NONETWORK;
			netname = getnetname(net);

			/* !! warning: doesn't handle arbitrary polygons, only boxes */
			if (isbox(poly, &lx, &hx, &ly, &hy) == 0)
			{
				ttyputmsg("Warning: drcb_highlight: non-manhattan shape");
				continue;
			}
			xformbox(&lx, &hx, &ly, &hy, trans);
			(void) askaid(us_aid, "clear");
			(void) askaid(us_aid, "show-area", lx, hx, ly, hy);
			ttyputmsg("layer %s, box %s<=X<=%s, %s<=Y<=%s, network %s",
					  layername(drcb_maintech, poly->layer),
					  latoa(lx), latoa(ly), latoa(hx), latoa(hy), netname);
			flushscreen();
			(void) ttygetchar();
		}
	}
	for(ai = node->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
	{
		n = arcpolys(ai);
		ttyputmsg("arc instance %s, arc proto %s, geom %s, %d polys",
			describearcinst(ai), describearcproto(ai->proto), geomname(ai->geom), n);
		(void) askaid(us_aid, "clear");
		(void) askaid(us_aid, "show-object", (INTBIG) ai->geom);
		flushscreen();
		(void) ttygetchar();
		for(j = 0; j < n; j++)
		{
			shapearcpoly(ai, j, poly);
			if (poly->layer < 0)
			{
				ttyputmsg("non-layer %d", poly->layer);
				continue;
			}
			net = ai->network;
			netname = getnetname(ai->network);
			/* !! warning: doesn't handle arbitrary polygons, only boxes */
			if (isbox(poly, &lx, &hx, &ly, &hy) == 0)
			{
				ttyputmsg("Warning: drcb_highlight: non-manhattan shape");
				continue;
			}
			xformbox(&lx, &hx, &ly, &hy, trans);
			(void) askaid(us_aid, "clear");
			(void) askaid(us_aid, "show-area", lx, hx, ly, hy);
			ttyputmsg("layer %s, box %s<=X<=%s, %s<=Y<=%s, network %s",
					  layername(drcb_maintech, poly->layer),
					  latoa(lx), latoa(ly), latoa(hx), latoa(hy), netname);
			flushscreen();
			(void) ttygetchar();
		}
	}

	(void) askaid(us_aid, "up-stack");
	freepolygon(poly);
	return 0;
}

/*
 * Toggles debugging on and off. Invoked as "tellaid drc hierarchical verbose"
 */
INTSML drcb_debug(void)
{
	debugflag = 1 - debugflag;
	ttyputmsg("NDRC tracing turned %s", debugstates[debugflag]);
	return 0;
}

/*
 * Toggles status of visual debugging. Very slow. Invoked as "tellaid
 * drc hierarchical visualdebug"
 */
INTSML drcb_vdebug(void)
{
	vdebugflag = 1 - vdebugflag;
	ttyputmsg("NDRC visual debugging turned %s", debugstates[vdebugflag]);
	return 0;
}

/************************ ERROR REPORTING ************************/

/*
 * Errors are stored in a doubly linked list -- they are added by
 * pushing them on the front of the list.
 */
typedef struct Idrerror
{
	INTBIG lx, hx;
	INTBIG ly, hy;
	GEOM *geom1;
	GEOM *geom2;
	INTSML layer1;
	INTSML layer2;
	INTBIG net1;
	INTBIG net2;
	INTSML errorcode;
	char *msg;
	struct Idrerror *nexterror;
	struct Idrerror *lasterror;
} DRERROR;

#define NODRERROR ((DRERROR *) -1)

static DRERROR *errlist = NODRERROR;
static DRERROR *curerror = NODRERROR;
static INTBIG n_errors = 0;

#define ERR_SPACING	((INTSML) 0)
#define ERR_MINSIZE	((INTSML) (ERR_SPACING + 1))
#define ERR_MAXSIZE	((INTSML) (ERR_MINSIZE + 1))
#define ERR_ENCLOSURE	((INTSML) (ERR_MAXSIZE + 1))
#define ERR_SPECIAL	((INTSML) (ERR_ENCLOSURE + 1))

#define N_ERRS		((INTSML) (ERR_SPECIAL + 1))

static char *dr_errstrings[N_ERRS + 1] = {
	"Spacing", "Minsize", "Maxsize", "Enclosure", "Special", 0
};

/* Adds details about an error to the error list */
static void drcb_report_error(INTBIG lx, INTBIG hx, INTBIG ly, INTBIG hy, GEOM *geom1,
	GEOM *geom2, INTSML layer1, INTSML layer2, INTBIG net1, INTBIG net2, INTSML errorcode,
	char *msg)
{
	DRERROR *ep = (DRERROR *) emalloc(sizeof(DRERROR), dr_aid->cluster);

	if (ep == 0)
	{
		ttyputerr("Warning: couldn't allocate memory for drc error");
		return;
	}
	ep->lx = lx;
	ep->hx = hx;
	ep->ly = ly;
	ep->hy = hy;
	ep->geom1 = geom1;
	ep->geom2 = geom2;
	ep->layer1 = layer1;
	ep->layer2 = layer2;
	ep->net1 = net1;
	ep->net2 = net2;
	ep->errorcode = errorcode;
	ep->msg = msg;
	ep->nexterror = errlist;
	ep->lasterror = NODRERROR;
	if (errlist != NODRERROR)
		errlist->lasterror = ep;
	errlist = ep;
	n_errors++;
	if(debugflag)
	{
		ttyputmsg("%s error (%s<=X<=%s, %s<=Y<=%s), layer %s, object %s, network %d",
			dr_errstrings[ep->errorcode], latoa(ep->lx), latoa(ep->hx), latoa(ep->ly),
				latoa(ep->hy), layername(drcb_maintech, ep->layer1),  geomname(ep->geom1), ep->net1);
		ttyputmsg("          to layer %s, object %s, network %d",
			layername(drcb_maintech, ep->layer2), geomname(ep->geom2), ep->net2);
		if (ep->msg != NULL)
			ttyputmsg("Additional information: %s", ep->msg);
	}
}

/* Frees all stored errors. */
static void drcb_free_errors(void)
{
	DRERROR *ep, *next;

	for(ep = errlist; ep != NODRERROR; ep = next)
	{
		next = ep->nexterror;
		if (ep->msg != NULL) efree(ep->msg);
		efree((char *) ep);
	}
	errlist = NODRERROR;
	curerror = NODRERROR;
	n_errors = 0;
}

/*
 * Redraws current error in the error list. Invoked by "tellaid drc
 * hierarchical current-error"
 */
char *drcb_curr_error(void)
{
	DRERROR *ep = curerror;
	NODEPROTO *np1, *np2;

	/*
	 * !! make sure appropriate nodeproto is displayed. Highlight the
	 * error box and print the message.
	 */
	if (ep == NODRERROR && (ep = errlist) == NODRERROR)
		return("No errors");

	(void) askaid(us_aid, "clear");
	np1 = geomparent(ep->geom1);
	np2 = geomparent(ep->geom2);
	(void)askaid(us_aid, "show-area", ep->lx, ep->hx, ep->ly, ep->hy);

	/* build the error message */
	(void)initinfstr();
	(void)addstringtoinfstr(dr_errstrings[ep->errorcode]);
	(void)addstringtoinfstr(" error between ");
	if (ep->geom1->entrytype == OBJNODEINST)
	{
		(void)addstringtoinfstr("node ");
		(void)addstringtoinfstr(describenodeinst(ep->geom1->entryaddr.ni));
	}
	if (ep->geom1->entrytype == OBJARCINST)
	{
		(void)addstringtoinfstr("arc ");
		(void)addstringtoinfstr(describearcinst(ep->geom1->entryaddr.ai));
	}
	if (ep->layer1 != ep->layer2)
	{
		(void)addstringtoinfstr(", layer ");
		(void)addstringtoinfstr(layername(drcb_maintech, ep->layer1));
	}
	if (np1 != np2)
	{
		(void)addstringtoinfstr(" in facet ");
		(void)addstringtoinfstr(describenodeproto(np1));
	}
	(void)addstringtoinfstr(" and ");
	if (ep->net1 != ep->net2) (void)addstringtoinfstr("un");
	(void)addstringtoinfstr("connected ");
	if (ep->geom2->entrytype == OBJNODEINST)
	{
		(void)addstringtoinfstr("node ");
		(void)addstringtoinfstr(describenodeinst(ep->geom2->entryaddr.ni));
	}
	if (ep->geom2->entrytype == OBJARCINST)
	{
		(void)addstringtoinfstr("arc ");
		(void)addstringtoinfstr(describearcinst(ep->geom2->entryaddr.ai));
	}
	(void)addstringtoinfstr(", layer ");
	(void)addstringtoinfstr(layername(drcb_maintech, ep->layer2));
	if (np1 != np2)
	{
		(void)addstringtoinfstr(" in facet ");
		(void)addstringtoinfstr(describenodeproto(np2));
	}
	if (ep->msg != NULL)
	{
		(void)addstringtoinfstr("; ");
		(void)addstringtoinfstr(ep->msg);
	}
	return(returninfstr());
}

/*
 * returns the current number of errors
 */
INTSML drcb_errorcount(void)
{
	INTSML errcount;
	DRERROR *err;

	errcount = 0;
	for(err = errlist; err != NODRERROR; err = err->nexterror) errcount++;
	return(errcount);
}

/*
 * Resets pointer to the first error in the error list for stepping
 * through the errors. Invoked by "tellaid drc hierarchical first-error"
 */
char *drcb_first_error(void)
{
	curerror = errlist;
	if (curerror == NODRERROR) return("No errors");

	return(drcb_curr_error());
}

/*
 * Sets pointer to the next error in the error list for stepping
 * through the errors. Invoked by "tellaid drc hierarchical next-error"
 */
char *drcb_next_error(void)
{
	if (curerror == NODRERROR)
		return(drcb_first_error());

	if (curerror->nexterror == NODRERROR)
		return("You are at the last error");

	/* !! undraw the last error */
	curerror = curerror->nexterror;
	return(drcb_curr_error());
}

/*
 * Sets pointer to the previous error in the error list for stepping
 * through the errors. Invoked by "tellaid drc hierarchical previous-error"
 */
char *drcb_prev_error(void)
{
	if (curerror == NODRERROR)
		return(drcb_first_error());

	if (curerror->lasterror == NODRERROR)
		return("You're at the first error");

	/* !! undraw the last error */
	curerror = curerror->lasterror;
	return(drcb_curr_error());
}

static POLYLIST *stplist;
static INTBIG *shapecounts;
static INTSML ncounts;
static INTBIG sumdepth;
static INTBIG ndepths;
static INTBIG depth;
static INTBIG maxdepth;
static INTBIG widths[MAXDEPTH];

static void count_shapes(NODEPROTO *np)
{
	NODEINST *ni;
	ARCINST *ai;
	INTSML i;
	INTBIG subnodes = 0;

	if (depth < MAXDEPTH)
	{
		widths[depth]++;
	} else
	{
		ttyputerr("depth %d exceeded MAXDEPTH %d", depth, MAXDEPTH);
	}
	if (depth > maxdepth)
		maxdepth = depth;
	depth++;
	for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		if (ni->proto->index == 0)
		{
			subnodes++;
			count_shapes(ni->proto);
		} else
		{
			INTSML tot;
			if ((tot = get_nodeEpolys(ni, stplist)) < 0)
			{
				ttyputerr("couldn't get node polys");
				continue;
			}
			for(i = 0; i < tot; i++)
			{
				POLYGON *poly = stplist->polygons[i];

				if (poly->layer < 0) continue;
				if (poly->layer > ncounts)
				{
					ttyputerr("layer too big -- %d", poly->layer);
					continue;
				}
				shapecounts[poly->layer]++;
			}
		}
	}
	for(ai = np->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
	{
		INTSML tot;
		if ((tot = get_arcpolys(ai, stplist)) < 0)
		{
			ttyputerr("couldn't get arc polys");
			continue;
		}
		for(i = 0; i < tot; i++)
		{
			POLYGON *poly = stplist->polygons[i];

			if (poly->layer < 0) continue;
			if (poly->layer > ncounts)
			{
				ttyputerr("layer too big -- %d", poly->layer);
				continue;
			}
			shapecounts[poly->layer]++;
		}
	}
	if (subnodes == 0 && np->index == 0)
	{
		/* leaf complex node, only primitive subnodes */
		sumdepth += depth;
		ndepths++;
	}
	depth--;
}

INTSML drcb_nodestats(NODEPROTO *np)
{
	INTSML i;

	if (np == NONODEPROTO)
	{
		ttyputerr("No current node proto to count");
		return 1;
	}
	ncounts = el_curtech->layercount;
	stplist = (POLYLIST *) emalloc(sizeof(POLYLIST), el_tempcluster);
	if (stplist == 0) return 1;
	stplist->polylistsize = 0;
	shapecounts = (INTBIG *) emalloc(sizeof(INTBIG) * ncounts, el_tempcluster);
	if (shapecounts == 0) return 1;
	for(i = 0; i < ncounts; i++) shapecounts[i] = 0;
	ndepths = depth = sumdepth = 0;
	for(i = 0; i < MAXDEPTH; i++) widths[i] = 0;
	count_shapes(np);
	ttyputmsg("Layer counts:");
	for(i = 0; i < ncounts; i++)
		if (shapecounts[i] != 0)
			ttyputmsg("    %s : %d", layername(el_curtech, i), shapecounts[i]);
	ttyputmsg("%d leaf nodes, maximum depth = %d average depth = %d",
			  ndepths, maxdepth, roundfloat(((float)sumdepth)/((float)ndepths)));
	ttyputmsg("Widths:");
	for(i = 0; i < MAXDEPTH; i++)
		if (widths[i] != 0)
			ttyputmsg("    %d : %d", i, widths[i]);
	efree((char *)stplist);
	stplist = NULL;
	efree((char *)shapecounts);
	shapecounts = NULL;
	ncounts = 0;
	return 0;
}


/************************ CHECKING ************************/

INTSML drcb_init(NODEPROTO *node)
{
	drcb_maintech = (node != NONODEPROTO) ? whattech(node) : el_curtech;
	drcb_surround = maxdrid(drcb_maintech);
	/*
	 * fake cell - used because rtree's use nodeproto->cell->cluster for
	 * allocation.
	 */
	if (drcb_cell == NOCELL)
	{
		drcb_cell = alloccell(dr_aid->cluster);
		if (drcb_cell == NOCELL) return(1);
		drcb_cell->cluster = dr_aid->cluster;
	}
	if (debugflag)
		ttyputmsg("Maximum distance for technology %s is %s",
			drcb_maintech->techname, latoa(drcb_surround));
	drcb_free_errors();
	return 0;
}

/*
 * Main entry point - initializes all variables, caches some
 * information, and starts the drc. Invoked by "tellaid drc hierarchical
 * run" to run hierarchically on the current node and by "tellaid drc
 * hierarchical all" to check all nodes in the library hierarchically.  If node is
 * NONODEPROTO, it means run on all cells in the library.
 * Returns number of errors found.
 */
INTSML drcb_check(NODEPROTO *node, INTSML hier)
{
	NODEPROTO *np;
	STATE drcb_state;	/* !! One for each processor */

	if (drcb_init(node)) return(0);
	if (allocstate(&drcb_state) == 0) return(0);
	if (node == NONODEPROTO)
	{
		for(np = el_curlib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		{
			if (debugflag)
				ttyputmsg("Checking prototype %s", describenodeproto(np));
			/* !! allocate parcel of net number space to each processor */
			drcb_state.netindex = 1;
			check_proto(&drcb_state, np);
		}
	} else
	{
		np = node;
		/* !! allocate parcel of net number space to each processor */
		drcb_state.netindex = 1;
		check_proto(&drcb_state, np);
	}
	freestate(&drcb_state);
	return(n_errors);
}

/*
 * Checks a node prototype in isolation. Spurious errors are put in a
 * list in the prototype. Definitely non-spurious errrors (i.e. which
 * cannot be corrected by any interactions with the rest of the
 * hierarchy) are reported.  We recurse down in check_proto, checking
 * each child node instance. Each node instance then recurses down
 * further. If the node instances' prototype isn't checked yet, it comes
 * back here to check that, and so on. No circular loops are possible
 * since a proto's children may not refer to it.
 */
static void check_proto(STATE *state, NODEPROTO *np)
{
	NODEINST top;	/* Dummy node instance for node */
	GEOM topgeom;	/* Dummy geometry module for dummy top node */
	INTSML saveindent;
	Hashtable *ht;
	Hashtable *hai;
	NODEPROTO *oldnp;	/* last nodeproto displayed in the current window. */
	NODEPROTO *tmpnp;

	for(tmpnp = el_curlib->firstnodeproto; tmpnp != NONODEPROTO; tmpnp = tmpnp->nextnodeproto)
		tmpnp->temp1 = 0; /* Indicates that nodeproto is not checked */
	np->temp1 = 0;

	if (debugflag)
	{
		/* Save current indentation on program stack and push to left margin */
		saveindent = get_indent();
		set_indent(0);
		ttyputmsg("Checking prototype %s", describenodeproto(np));
	}
	if (vdebugflag)
	{
		/*
		 * Save presently highlighted errors and clear the decks for
		 * action. Make sure this prototype is in a window.
		 */
		(void) askaid(us_aid, "down-stack");
		(void) askaid(us_aid, "clear");
		oldnp = drcb_ensurewindow(np);
		flushscreen();
	}

	drcb_check_all_protos(state, np);

	/*
	 * Create dummy node instance for top level cell. It has the same
	 * coordinates as the nodeproto, and no parent. Those are the only
	 * relevant members for it.
	 */
	top = *dummynode();
	topgeom.firstvar = NOVARIABLE;
	topgeom.numvar = 0;
	topgeom.entrytype = OBJNODEINST;
	topgeom.entryaddr.ni = &top;

	top.lowx = topgeom.lowx = np->lowx;
	top.highx = topgeom.highx = np->highx;
	top.lowy = topgeom.lowy = np->lowy;
	top.highy = topgeom.highy = np->highy;
	top.proto = np;
	top.geom = &topgeom;

	ht = hashcreate(HTABLE_PORT); /* check error return !! */
	hai = hashcreate(HTABLE_ARC); /* check error return !! */
	drcb_get_init_nets(np, ht, &state->netindex);
	drcb_get_arc_networks(np, ht, hai, &state->netindex);

	/* Start the checking process for this proto */
	check_node(state, &top, el_matid, ht, hai, NONODEPROTO);

	if (debugflag)
	{
		ttyputmsg("Finished checking prototype %s", describenodeproto(np));
		/* Restore old indentation */
		set_indent(saveindent);
	}
	if (vdebugflag)
	{
		/* Cleanup and restore last proto */
		(void) drcb_ensurewindow(oldnp);
		(void) askaid(us_aid, "clear");
		(void) askaid(us_aid, "up-stack");
		flushscreen();
	}
}

/*
 * recursively walks through a prototype, checking all prototypes of
 * instances and sub-instances
 */
void drcb_check_all_protos(STATE *state, NODEPROTO *np)
{
	Hashtable *nhpp;
	Hashtable *nhai;
	NODEINST *ni;

	state->netindex = 1;    /* !! per-processor initialization */
	nhpp = hashcreate(HTABLE_PORT); /* check error return !! */
	nhai = hashcreate(HTABLE_ARC); /* check error return !! */
	drcb_get_init_nets(np, nhpp, &state->netindex);
	drcb_get_arc_networks(np, nhpp, nhai, &state->netindex);
	do_ndrcproto(state, np, nhpp, nhai);
	hashdestroy(nhpp, HASH_NULLPROC, HASH_NULL);
	hashdestroy(nhai, HASH_NULLPROC, HASH_NULL);
	/* now check that all our children have been checked */
	for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		if (ni->proto->index == 0 && ni->proto->temp1 == 0)
		{
			/* If node prototype hasn't been checked yet, check it first */
			ni->proto->temp1++;
			drcb_check_all_protos(state, ni->proto);
		}
	}
}

/* Recursive check of a node instance (node prototype + transformation). */
static void check_node(STATE *state, NODEINST *nodeinst, XARRAY mytrans,
	Hashtable *hpp, Hashtable *hai, NODEPROTO *intersect_list)
{
	NODEINST *ni;
	NODEPROTO *np = nodeinst->proto;
	NODEPROTO *child_intersects;
	XARRAY subtrans, temptrans1, temptrans2;
	INTBIG lx, hx, ly, hy;
	Hashtable *nhai;
	Hashtable *nhpp;

	if (debugflag)
	{
		ttyputmsg("starting %scheck_node(%s)", indent, describenodeinst(nodeinst));
		inc_indent();
	}
	if (vdebugflag && nodeinst->parent != NONODEPROTO)
	{
		(void) askaid(us_aid, "show-object", (INTBIG) nodeinst->geom);
		flushscreen();
		(void) ttygetchar();
	}

	if(debugflag)
	{
		if (intersect_list != NONODEPROTO)
		{
			ttyputmsg("%sintersected %d shapes in total", indent, intersect_list->temp1);
		} else
		{
			ttyputmsg("%sNo intersected shapes", indent);
		}
	}

	/*
	 * Check each intersecting element with any interacting primitive
	 * nodes in the nodeinst.
	 */
	if (intersect_list != NONODEPROTO)
		do_ndrcinst(state, nodeinst, intersect_list, hpp, hai);

	for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		if (ni->proto->index == 0)
		{
			/* Complex node - recursively check it */

			nhpp = hashcreate(HTABLE_PORT); /* check error return !! */
			nhai = hashcreate(HTABLE_ARC); /* check error return !! */

			drcb_get_nodeinst_networks(ni, hpp, hai, nhpp, &state->netindex);
			drcb_get_arc_networks(ni->proto, nhpp, nhai, &state->netindex);

			/*
			 * Compute transformation matrix for this instance that
			 * maps objects back to the prototype being displayed.
			 */
			makerot(ni, temptrans1);
			transmult(temptrans1, mytrans, temptrans2);
			maketrans(ni, temptrans1);
			transmult(temptrans1, temptrans2, subtrans);

			/*
			 * drcb_box returns the bounding box of nodeinst - this box is
			 * relative to the parent prototype, which is fine, because that's
			 * what we'll be searching for intersecting elements
			 */
			drcb_box(ni, &lx, &hx, &ly, &hy);

			/*
			 * find intersecting elements returns elements relative to
			 * nodeinst.  all checking of the nodeinst can then be done
			 * in the nodeinst's coordinate system
			 */
			child_intersects = find_intersecting_elements(state, ni,
														  lx, hx, ly, hy,
														  subtrans, hpp, hai,
														  intersect_list);
			/*
			 * child_intersects, nhpp and nhai will be freed after the
			 * check_node call
			 */
			check_node(state, ni, subtrans, nhpp, nhai, child_intersects);
		}
	}

	if (debugflag)
	{
		ttyputmsg("finished check_node(%s)", describenodeinst(nodeinst));
		dec_indent();
	}
	free_intersecting_elements(intersect_list);
	hashdestroy(hpp, HASH_NULLPROC, HASH_NULL);
	hashdestroy(hai, HASH_NULLPROC, HASH_NULL);
}


/*
 * Returns the maximum drc interaction distance on any layer for a given
 * technology. We only need to check objects within maxdrid() of a complex
 * node.
 */
static INTBIG maxdrid(TECHNOLOGY *tech)
{
	INTSML i;
	INTBIG drid = 0;
	INTBIG l, max_drid_hack;

	max_drid_hack = scalefromdispunit(MAX_DRID_HACK, DISPUNITMIC);
	for(i = 0; i < tech->layercount; i++)
	{
		l = maxdrcsurround(tech, i);
		if (l > drid && l < max_drid_hack)
		{
			drid = l;
		}
	}
	return drid;
}

/*
 * To intersect elements, we call find_intersecting_elements with a
 * transformation that describes how to go from nodeinst->parent
 * (prototype of the instances' parent) to nodeinst->proto (prototype of
 * the instance). This is the inverse of the matrix obtained by makerot
 * and maketrans, i.e. use makerotI and maketransI.
 * find_intersecting_elements then transforms all intersecting elements
 * it finds by that matrix. For child node instances,
 * find_intersecting_elements calls intersect_subtree on them, giving
 * them a new transformation matrix. This matrix is the product of the
 * transformation to go from the child node instance's prototype to the
 * parent (i.e. the result of makerot/maketrans on the child instance)
 * and the current transformation. Thus we hand the transformation down
 * the hierarchy.
 */
/*
 * We also hand a clip box down to find_intersecting_elements.
 * The box is inverse transformed from the coordinate
 * system of the parent to the coordinates of the child instance's
 * prototype, and handed in to the intersect procedure for the prototype
 * along with the matrix above. The child instance's prototype is
 * searched using the transformed clip box -- this may recurse.
 */
/*
 * All transformation matrices and clip boxes that are passed down via
 * recursion are on the stack.
 */
/*
 * !! Try to avoid having find_intersecting_elements AND intersect_subtree
 * even if it means passing in a nodeinst value to each level?
 */

/*
 * Computes the intersecting elements above a node instance, and store
 * it in the instance.
 */
static NODEPROTO *find_intersecting_elements(STATE *state, NODEINST *nodeinst,
	INTBIG lowx, INTBIG highx, INTBIG lowy, INTBIG highy, XARRAY mytrans,
	Hashtable *hpp, Hashtable *hai, NODEPROTO *pi)
{
	Hashtable *nhpp = hashcreate(HTABLE_PORT); /* check error return!! */
	INTBIG searchkey;
	NODEPROTO *intersect_list;
	GEOM *geom;
	XARRAY itrans, subtrans, temptrans1, temptrans2;
	XARRAY temptrans3;
	INTBIG lx, hx, ly, hy;

	if (nodeinst->parent == NONODEPROTO)
	{
		return NONODEPROTO;
	}
	if (debugflag)
	{
		ttyputmsg("%sfind_intersecting_elements(%s)", indent, describenodeinst(nodeinst));
	}
	if (vdebugflag)
	{
		(void) askaid(us_aid, "down-stack");
		(void) askaid(us_aid, "clear");
		(void) askaid(us_aid, "show-area", lowx, highx, lowy, highy);
		flushscreen();
		(void) ttygetchar();
	}
	intersect_list = allocnodeproto(dr_aid->cluster);
	if (intersect_list == NONODEPROTO) return NONODEPROTO;
	intersect_list->cell = drcb_cell;
	intersect_list->temp1 = 0;	/* stores a count of number of intersections */
	if (geomstructure(intersect_list) != 0)
	{
		freenodeproto(intersect_list);
		return NONODEPROTO;
	}
	/*
	 * compute trans - matrix to transform the shapes back to nodeinst's
	 * prototype coordinate space
	 */
	makerotI(nodeinst, temptrans1);
	maketransI(nodeinst, temptrans2);
	transmult(temptrans1, temptrans2, itrans);

	searchkey = initsearch(lowx, highx, lowy, highy, nodeinst->parent);
	while((geom = nextobject(searchkey)) != NOGEOM)
	{
		if (geom->entrytype == OBJNODEINST)
		{
			NODEINST *ni;

			ni = geom->entryaddr.ni;
			if (ni == nodeinst)
				/* skip the instance which "called" this */
				continue;
			/* Compute networks for the nodeinst */
			drcb_get_nodeinst_networks(ni, hpp, hai, nhpp, &state->netindex);
			if (ni->proto->index == 0)
			{
				/*
				 * transform the intersecting box to the child
				 * nodeproto's coordinate system so we can recurse down
				 * the tree still intersecting with the same box.
				 */
				makerotI(ni, temptrans1);
				maketransI(ni, temptrans2);
				transmult(temptrans1, temptrans2, temptrans3);
				lx = lowx;
				hx = highx;
				ly = lowy;
				hy = highy;
				xformbox(&lx, &hx, &ly, &hy, temptrans3);
				/*
				 * Compute the inverse transform to apply to objects
				 * that intersect so they will get transformed back to
				 * the original nodeinst's coordinate space. Make a new
				 * itrans for ni in subtrans.
				 */
				makerot(ni, temptrans1);
				maketrans(ni, temptrans2);
				transmult(temptrans1, temptrans2, temptrans3);
				transmult(temptrans3, itrans, subtrans);

				/*
				 * mytrans transforms to the displayed prototype's
				 * coordinate system. Make a new mytrans for the ni in
				 * temptrans1 so we can pass it down as we recurse.
				 */
				transmult(temptrans3, mytrans, temptrans1);
				intersect_subtree(state, intersect_list, ni->proto, subtrans,
								  temptrans1, lx, hx, ly, hy, nhpp);
			} else
			{
				getnodeinstshapes(state, intersect_list, ni, itrans,
								  lowx, highx, lowy, highy, nhpp);
			}
			hashclear(nhpp);
		} else if (geom->entrytype == OBJARCINST)
		{
			getarcinstshapes(state, intersect_list, geom->entryaddr.ai, itrans,
							 lowx, highx, lowy, highy, hai);
		}
	}

	if(debugflag)
	{
		ttyputmsg("%sintersected %d shapes from siblings", indent, intersect_list->temp1);
	}
	/*
	 * !! Now go through the parent's intersect list 'pi' and select
	 * those shapes that intersect. Transform by itrans and push onto
	 * intersect_list. We allocate new copies -- if we used only
	 * pointers, then it would be faster but we'd have to keep refcnts
	 * to keep track of them when freeing.
	 */
	if (pi != NONODEPROTO)
	{
		searchkey = initsearch(lowx, highx, lowy, highy, pi);
		while((geom = nextobject(searchkey)) != NOGEOM)
		{
			GEOM *g;
			SHAPE *shape;

			allocgeom(1, &g, dr_aid->cluster);
			if (g == NOGEOM) continue;
			*g = *geom;
			/*
			 * we're only interested in the part of the shape that
			 * intersects the drc box
			 */
			if (intersectbox(&(g->lowx), &(g->highx), &(g->lowy), &(g->highy),
				lowx, highx, lowy, highy) != 0)
			{
				(void) freegeom(g);
				continue;
			}
			shape = (SHAPE *) emalloc(sizeof(SHAPE), dr_aid->cluster);
			if (shape == 0)
			{
				ttyputerr("Cannot allocate memory for shape");
				continue;
			}
			*shape = *((SHAPE *) (geom->entryaddr.ni));
			g->entryaddr.ni = (NODEINST *) shape;
			if (shape->hpp)
				shape->hpp = hashcopy(shape->hpp); /* check error return !! */
			xformbox(&(g->lowx), &(g->highx), &(g->lowy), &(g->highy), itrans);
			drcb_linkgeom(g, intersect_list);
			intersect_list->temp1++;
		}
	}
	hashdestroy(nhpp, HASH_NULLPROC, HASH_NULL);
	/* !! How about the error list ? */

	if (vdebugflag)
	{
		(void) askaid(us_aid, "clear");
		(void) askaid(us_aid, "up-stack");
		flushscreen();
		(void) ttygetchar();
	}

	return intersect_list;
}

/*
 * Takes a box specified by lx, hx, ly, hy, transforms it by trans, and
 * returns the box in lx, hx, ly, hy. The relationship between lx, hx,
 * ly, hy is preserved - i.e hx > lx, hy > ly
 */
static void xformbox(INTBIG *lx, INTBIG *hx, INTBIG *ly, INTBIG *hy, XARRAY trans)
{
	INTBIG tx, ty;
	INTBIG lowx = *lx;
	INTBIG highx = *hx;
	INTBIG lowy = *ly;
	INTBIG highy = *hy;

	xform(lowx, lowy, lx,ly, trans);   *hx = *lx;  *hy = *ly;
	xform(lowx, highy, &tx,&ty, trans);
	if (tx < *lx) *lx = tx;   if (tx > *hx) *hx = tx;
	if (ty < *ly) *ly = ty;   if (ty > *hy) *hy = ty;
	xform(highx, highy, &tx,&ty, trans);
	if (tx < *lx) *lx = tx;   if (tx > *hx) *hx = tx;
	if (ty < *ly) *ly = ty;   if (ty > *hy) *hy = ty;
	xform(highx, lowy, &tx,&ty, trans);
	if (tx < *lx) *lx = tx;   if (tx > *hx) *hx = tx;
	if (ty < *ly) *ly = ty;   if (ty > *hy) *hy = ty;
}


/*
 * Descends through a complex node prototype and it's instances, finding
 * all elements intersecting the box described by lowx, highx, lowy,
 * highy and adding their polygons to plpp. itrans is the matrix
 * accumulated through the recursion down the tree that maps objects
 * back to the nodeinst that the search started from. mytrans is the
 * matrix accumulated at the same time describing the transformation
 * that maps objects back to the nodeproto being displayed. mytrans is
 * only needed (and computed) if vdebugflag is on.
 */
static void intersect_subtree(STATE *state, NODEPROTO *intersect_list, NODEPROTO *np,
	XARRAY itrans, XARRAY mytrans, INTBIG lowx, INTBIG highx, INTBIG lowy,
	INTBIG highy, Hashtable *hpp)
{
	Hashtable *nhpp = hashcreate(HTABLE_PORT); /* check error return!! */
	Hashtable *hai = hashcreate(HTABLE_ARC); /* check error return!! */
	INTBIG searchkey;
	GEOM *geom;
	XARRAY subtrans, temptrans1, temptrans2;
	XARRAY temptrans3;
	INTBIG lx, hx, ly, hy;

	if (debugflag)
	{
		inc_indent();
		ttyputmsg("%sintersect_subtree(%s)", indent, describenodeproto(np));
	}
	drcb_get_arc_networks(np, hpp, hai, &state->netindex);
	searchkey = initsearch(lowx, highx, lowy, highy, np);
	while((geom = nextobject(searchkey)) != NOGEOM)
	{
		if (geom->entrytype == OBJNODEINST)
		{
			NODEINST *ni;

			ni = geom->entryaddr.ni;
			/* Compute networks for the child */
			drcb_get_nodeinst_networks(ni, hpp, hai, nhpp, &state->netindex);
			if (ni->proto->index == 0)
			{
				/*
				 * transform the intersecting box to the child
				 * nodeproto's coordinate system so we can recurse down
				 * the tree still intersecting with the same box.
				 */
				makerotI(ni, temptrans1);
				maketransI(ni, temptrans2);
				transmult(temptrans1, temptrans2, temptrans3);
				lx = lowx;
				hx = highx;
				ly = lowy;
				hy = highy;
				xformbox(&lx, &hx, &ly, &hy, temptrans3);
				/*
				 * Compute the inverse transform to apply to objects
				 * that intersect so they will get transformed back to
				 * the original nodeinst's coordinate space. Make a new
				 * itrans for ni in subtrans.
				 */
				makerot(ni, temptrans1);
				maketrans(ni, temptrans2);
				transmult(temptrans1, temptrans2, temptrans3);
				transmult(temptrans3, itrans, subtrans);
				/*
				 * mytrans transforms to the displayed prototype's
				 * coordinate system. Make a new mytrans for the ni in
				 * temptrans1 so we can pass it down as we recurse.
				 */

				transmult(temptrans3, mytrans, temptrans1);
				intersect_subtree(state, intersect_list, ni->proto, subtrans,
								  temptrans1, lx, hx, ly, hy, nhpp);
			} else
			{
				getnodeinstshapes(state, intersect_list, ni, itrans,
								  lowx, highx, lowy, highy, nhpp);
			}
			hashclear(nhpp);
		} else if (geom->entrytype == OBJARCINST)
		{
			getarcinstshapes(state, intersect_list, geom->entryaddr.ai, itrans,
							 lowx, highx, lowy, highy, hai);
		}
	}
	hashdestroy(nhpp, HASH_NULLPROC, HASH_NULL);
	hashdestroy(hai, HASH_NULLPROC, HASH_NULL);
	if (debugflag)
	{
		dec_indent();
	}
	return;
}


/*
 * Extracts the electrical shapes (presently only boxes) from ni,
 * transforms them by itrans, and inserts them into intersect_list
 */
static void getnodeinstshapes(STATE *state, NODEPROTO *intersect_list, NODEINST *ni,
	XARRAY itrans, INTBIG lowx, INTBIG highx, INTBIG lowy, INTBIG highy, Hashtable *hpp)
{
	XARRAY rtrans;
	XARRAY trans;
	INTBIG lx, hx, ly, hy;
	INTSML j, n;
	INTBIG net;
	POLYLIST *polylist = state->polylist;
	POLYGON *poly;
	GEOM *geom;
	SHAPE *shape;

	makerot(ni, rtrans);
	transmult(itrans, rtrans, trans);
	n = get_nodeEpolys(ni, polylist);
	for(j = 0; j < n; j++)
	{
		poly = polylist->polygons[j];
		if (poly->layer < 0 || maxdrcsurround(drcb_maintech, poly->layer) < 0)
			continue;
		/* find the global electrical net for this layer */
		if (poly->portproto != NOPORTPROTO)
			net = drcb_network(hpp, (char *) poly->portproto, "node");
		else
			net = NONET;
		/* !! warning: doesn't handle arbitrary polygons, only boxes */
		if (isbox(poly, &lx, &hx, &ly, &hy) == 0)
		{
			ttyputmsg("Warning: getnodeinstshapes non-manhattan shape");
			continue;
		}
		/* extract those parts of the box that intersect the drc box */
		if (intersectbox(&lx, &hx, &ly, &hy, lowx, highx, lowy, highy) != 0)
		{
			continue;
		}
		/*
		 * transform the box to the coordinates of the nodeinstance that
		 * initiated the search
		 */
		xformbox(&lx, &hx, &ly, &hy, trans);
		shape = (SHAPE *) emalloc(sizeof(SHAPE), dr_aid->cluster);
		if (shape == 0)
		{
			ttyputerr("Cannot allocate memory for shape");
			continue;
		}
		shape->layer = poly->layer;
		shape->entrytype = OBJNODEINST;
		shape->entryaddr.ni = ni;
		shape->net = net;
		/*
		 * we're wasting a fair bit of memory by copying an entire
		 * network table for a nodeinst for each shape in the intersect
		 * list.  Unfortunately, there's no simple way out of it; any
		 * more memory efficient scheme involves keeping reference
		 * counts for freeing, and sharing the tables as far as possible.
		 */
		shape->hpp = hashcopy(hpp); /* check error return !! */
		allocgeom(1, &geom, dr_aid->cluster);
		if (geom == NOGEOM)
			continue;
		geom->entrytype = 0;
		/* !! Wish geom->entryaddr had a generic pointer as well */
		geom->entryaddr.ni = (NODEINST *) shape;
		geom->lowx = lx;
		geom->highx = hx;
		geom->lowy = ly;
		geom->highy = hy;
		drcb_linkgeom(geom, intersect_list);
		intersect_list->temp1++;
		geom = NOGEOM;
	}
}


/*
 * Extracts the electrical shapes (presently only boxes) from ai,
 * transforms them by trans, and inserts them into intersect_list
 */
static void getarcinstshapes(STATE *state, NODEPROTO *intersect_list, ARCINST *ai,
	XARRAY trans, INTBIG lowx, INTBIG highx, INTBIG lowy, INTBIG highy, Hashtable *hai)
{
	INTBIG lx, hx, ly, hy;
	INTSML j, n;
	POLYGON *poly;
	POLYLIST *polylist = state->polylist;
	GEOM *geom;
	SHAPE *shape;

	n = get_arcpolys(ai, polylist);
	if (n < 0) return;
	for(j = 0; j < n; j++)
	{
		poly = polylist->polygons[j];
		if (poly->layer < 0 || maxdrcsurround(drcb_maintech, poly->layer) < 0) continue;
		/* !! warning: doesn't not handle arbitrary polygons, only boxes */
		if (isbox(poly, &lx, &hx, &ly, &hy) == 0)
		{
			ttyputmsg("Warning: getarcinstshapes non-manhattan shape");
			continue;
		}
		/* extract those parts of the box that intersect the drc box */
		if (intersectbox(&lx, &hx, &ly, &hy, lowx, highx, lowy, highy) != 0)
		{
			continue;
		}
		/*
		 * transform the box to the coordinates of the nodeinstance that
		 * initiated the search
		 */
		xformbox(&lx, &hx, &ly, &hy, trans);
		shape = (SHAPE *) emalloc(sizeof(SHAPE), dr_aid->cluster);
		if (shape == 0)
		{
			ttyputerr("Cannot allocate memory for shape");
			continue;
		}
		shape->layer = poly->layer;
		shape->entrytype = OBJARCINST;
		shape->entryaddr.ai = ai;
		shape->net = drcb_network(hai, (char *) ai, "arc");
		shape->hpp = NULL;
		allocgeom(1, &geom, dr_aid->cluster);
		if (geom == NOGEOM) continue;
		geom->entrytype = 0;
		/* !! Wish geom->entryaddr had a generic pointer as well */
		geom->entryaddr.ni = (NODEINST *) shape;
		geom->lowx = lx;
		geom->highx = hx;
		geom->lowy = ly;
		geom->highy = hy;
		drcb_linkgeom(geom, intersect_list);
		intersect_list->temp1++;
		geom = NOGEOM;
	}
}

char *drcb_pr_nodenettable(char *key, char *datum, char *arg)
{
	ttyputmsg(" 0x%x (%s) network %d", key, ((PORTPROTO *) key)->protoname, datum);
	return 0;
}

char *drcb_pr_arcnettable(char *key, char *datum, char *arg)
{
	ttyputmsg(" 0x%x (%s) network %d", key, describearcinst((ARCINST *) key), datum);
	return 0;
}

static INTBIG drcb_network(Hashtable *ht, char *cp, char *name)
{
	char *net;

	if (ht)
		net = hashsearch(ht, cp);
	else
		return 0;

	if (net == NULL)
	{
		ttyputerr("cannot find %s 0x%lx in hashtable 0x%lx", name, (long) cp, (long) ht);
		if (strcmp(name, "node") == 0)
		{
			ttyputmsg("portproto %s", ((PORTPROTO *)cp)->protoname);
			hashwalk(ht, drcb_pr_nodenettable, 0);
		} else
		{
			ttyputmsg("arcinst %s", describearcinst((ARCINST *)cp));
			hashwalk(ht, drcb_pr_arcnettable, 0);
		}
		return 0;
	}
	if (net == 0)
		ttyputerr("network numbered 0!");
	return (INTBIG) net;
}


/* Frees the list of intersecting elements in a node instance */
static void free_intersecting_elements(NODEPROTO *np)
{
	if (np != NONODEPROTO)
	{
		freertree(np->rtree);
		np->rtree = NORTNODE;
		freenodeproto(np);
	}
}

/*
 * routine to link geometry module "geom" into the R-tree.  The parent
 * nodeproto is in "parnt". (slightly modified from linkgeom -- this one
 * doesn't call boundobj to get the geom bounds -- it expects the ones in
 * geom to be valid when it is passed in.
 */
static void drcb_linkgeom(GEOM *geom, NODEPROTO *parnt)
{
	REGISTER RTNODE *rtn;
	REGISTER INTSML i, bestsubnode;
	REGISTER INTBIG bestexpand, area, newarea, expand, scaledown;
	INTBIG lxv, hxv, lyv, hyv;

	/* find the leaf that would expand least by adding this node */
	rtn = parnt->rtree;
	for(;;)
	{
		/* if R-tree node contains primitives, exit loop */
		if (rtn->flag != 0) break;

		/* compute scaling factor */
		scaledown = HUGEINT;
		for(i=0; i<rtn->total; i++)
		{
			db_rtnbbox(rtn, i, &lxv, &hxv, &lyv, &hyv);
			if (hxv-lxv < scaledown) scaledown = hxv - lxv;
			if (hyv-lyv < scaledown) scaledown = hyv - lyv;
		}
		if (scaledown <= 0) scaledown = 1;

		/* find sub-node that would expand the least */
		bestexpand = HUGEINT;
		for(i=0; i<rtn->total; i++)
		{
			/* get bounds and area of sub-node */
			db_rtnbbox(rtn, i, &lxv, &hxv, &lyv, &hyv);
			area = ((hxv - lxv) / scaledown) * ((hyv - lyv) / scaledown);

			/* get area of sub-node with new element */
			newarea = ((maxi(hxv, geom->highx) - mini(lxv, geom->lowx)) / scaledown) *
				((maxi(hyv, geom->highy) - mini(lyv, geom->lowy)) / scaledown);

			/* accumulate the least expansion */
			expand = newarea - area;
			if (expand > bestexpand) continue;
			bestexpand = expand;
			bestsubnode = i;
		}

		/* recurse down to sub-node that expanded least */
		rtn = (RTNODE *)rtn->pointers[bestsubnode];
	}

	/* add this geometry element to the correct leaf R-tree node */
	(void)db_addtortnode((unsigned)geom, rtn, parnt);
}

static void check_shape(GEOM *g, char *arg)
{
	CheckShapeArg *csap = (CheckShapeArg *) arg;
	NODEINST *ni = csap->ni;
	SHAPE *shape = (SHAPE *) g->entryaddr.ni;
	Box b, topbox;

	b.lx = g->lowx;
	b.hx = g->highx;
	b.ly = g->lowy;
	b.hy = g->highy;
	topbox = b;
	xformbox(&topbox.lx, &topbox.hx, &topbox.ly, &topbox.hy, csap->vtrans);
	if (vdebugflag)
	{
		(void) askaid(us_aid, "show-area", topbox.lx, topbox.hx, topbox.ly, topbox.hy);
		flushscreen();
		(void) ttygetchar();
	}

	/*
	 * Check shape against all intersecting elements in the nodeinst.
	 * We don't need to transform anything.
	 */
	if (shape->entrytype == OBJNODEINST)
	{
		NODEINST *sni = shape->entryaddr.ni;

		csap->state->nhpp = shape->hpp;
		drcb_badbox(csap->state, sni->geom, shape->layer, sni->proto->tech, ni->proto,
			&b, &topbox, shape->net, 0);
		csap->state->nhpp = NULL;
	} else if (shape->entrytype == OBJARCINST)
	{
		ARCINST *sai = shape->entryaddr.ai;
		csap->state->nhpp = NULL;
		drcb_badbox(csap->state, sai->geom, shape->layer, sai->proto->tech, ni->proto,
			&b, &topbox, shape->net, 0);
	}
}


/*
 * Actual check of a node instance - goes through each intersecting
 * element in the instance and makes sure the node elements that
 * intersect with that element are drc correct. Checks if any spurious
 * errors are removed or confirmed.
 */
static void do_ndrcinst(STATE *state, NODEINST *nip, NODEPROTO *intersect_list,
	Hashtable *hpp, Hashtable *hai)
{
	XARRAY temptrans1, temptrans2;
	NODEINST *ni;
	NODEPROTO *niproto = nip->proto;
	Hashtable *hni = hashcreate(HTABLE_NI); /* check error return !! */
	Hashtable *nhpp = hashcreate(HTABLE_PORT); /* check error return !! */
	Hashtable *h;
	CheckShapeArg csa;

	if (debugflag)
	{
		ttyputmsg("%sdo_ndrcinst(%s)", indent, describenodeinst(nip));
	}
	csa.ni = nip;
	csa.state = state;
	state->hni = hni;
	state->hai = hai;

	/* build network tables for all objects in nip's prototype */
	for(ni = niproto->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		if (ni->proto->index != 0)
		{
			drcb_get_nodeinst_networks(ni, hpp, hai, nhpp, &state->netindex);
			h = hashcopy(nhpp); /* check error return !! */
			if (hashinsert(hni, (char *) ni, (char *)h, 0) != 0)
			{
				ttyputerr("couldn't add h 0x%lx, ni %s (0x%lx), hni 0x%lx",
						  (long) h, describenodeinst(ni), (long) ni, (long) hni);
			}
			hashclear(nhpp);
		}
	}

	/*
	 * transformation matrix to for visual debugging. This transforms
	 * the shape from the coordinatespace of nip's nodeproto to the
	 * coordinate space of nip's parent, which is what is being
	 * displayed.
	 */
	makerot(nip, temptrans1);
	maketrans(nip, temptrans2);
	transmult(temptrans1, temptrans2, csa.vtrans);

	/* check each geom in the intersect list */
	if (intersect_list != NONODEPROTO)
		walkrtree(intersect_list->rtree, check_shape, (char *) &csa);

	hashwalk(hni, freehpp, HASH_NULL);
	hashdestroy(hni, HASH_NULLPROC, HASH_NULL);
	hashdestroy(nhpp, HASH_NULLPROC, HASH_NULL);
	state->hni = NULL;
	state->hai = NULL;
}


static char *freehpp(char *key, char *datum, char *arg)
{
	hashdestroy((Hashtable *) datum, HASH_NULLPROC, HASH_NULL);
	return 0;
}

/*
 * Actual check of a node prototype - goes through all primitive
 * nodeinsts and all arcinsts in a nodeproto and checks that they are
 * design rule correct within the cell. Any errors are caused by excess
 * paint are reported. Any errors caused by the absence of paint are
 * possibly spurious -- they're stored in case a subcell gets rid of
 * them. Even if there are no subcells within interacting distance of an
 * "absence of paint" error, it can be spurious if it gets removed by
 * some overlapping instance. !! If it isn't removed in any instance of
 * the cell, then it should be reported as being a proto error, if it is
 * removed in some instances but not in others, then it should be
 * reported as an instance error. How do we tell?
 */
static void do_ndrcproto(STATE *state, NODEPROTO *np, Hashtable *hpp, Hashtable *hai)
{
	REGISTER NODEINST *ni;
	REGISTER ARCINST *ai;
	REGISTER INTSML freq, stopped;
	Hashtable *hni = hashcreate(HTABLE_NI); /* check error return !! */
	Hashtable *nhpp = hashcreate(HTABLE_PORT); /* check error return !! */
	Hashtable *h;

	if(debugflag)
	{
		ttyputmsg("%sdo_ndrcproto(%s)", indent, describenodeproto(np));
	}
	/* build network tables for all objects */
	for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		if (ni->proto->index != 0)
		{
			drcb_get_nodeinst_networks(ni, hpp, hai, nhpp, &state->netindex);
			h = hashcopy(nhpp); /* check error return !! */
			if (hashinsert(hni, (char *) ni, (char *)h, 0) != 0)
			{
				ttyputerr("couldn't add h 0x%lx, ni %s (0x%lx), hni 0x%lx",
						  (long) h, describenodeinst(ni), (long) ni, (long) hni);
			}
			hashclear(nhpp);
		}
	}
	state->hni = hni;
	state->hai = hai;

	/* now check every primitive node object */
	freq = stopped = 0;
	for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		if (((++freq)%50) == 0)
		{
			stopped = stopping("DRC");
			if (stopped != 0) break;
		}
		if (ni->proto->index != 0)
		{
			h = (Hashtable *) hashsearch(hni, (char *) ni);
			if (h == NULL)
			{
				ttyputerr("couldn't find nhpp for ni %s (0x%lx), hni 0x%lx",
						  describenodeinst(ni), (long) ni, (long) hni);
			}
			state->nhpp = h;
			(void) drcb_checknodeinst(state, ni, 1, h);
		}
	}

	/* check all arcs */
	state->nhpp = NULL;
	if (stopped == 0)
		for(ai = np->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
	{
		if (((++freq)%50) == 0)
			if (stopping("DRC") != 0) break;
		(void) drcb_checkarcinst(state, ai, 1);
	}
	hashwalk(hni, freehpp, HASH_NULL);
	hashdestroy(hni, HASH_NULLPROC, HASH_NULL);
	hashdestroy(nhpp, HASH_NULLPROC, HASH_NULL);
	state->hni = NULL;
	state->hai = NULL;
}

/*
 * routine to check the design rules about nodeinst "ni".  If "partial"
 * is nonzero, only check those other objects whose geom pointers are
 * greater than this one (i.e. only check any pair of objects once).
 * The routine returns nonzero if checking is to be terminated.
 */
static INTSML drcb_checknodeinst(STATE *state, NODEINST *ni, INTSML partial, Hashtable *nhpp)
{
	REGISTER INTSML tot, j, ret;
	XARRAY trans;
	REGISTER INTBIG net;
	POLYLIST *polylist = state->polylist;
	Box b;

	makerot(ni, trans);

	/* get all of the polygons on this node */
	tot = get_nodeEpolys(ni, polylist);
	if (tot < 0) return 1;

	/* examine the polygons on this node */
	for(j=0; j<tot; j++)
	{
		if (polylist->polygons[j]->layer < 0) continue;
		if (polylist->polygons[j]->portproto == NOPORTPROTO)
			net = NONET;
		else
			net = drcb_network(nhpp, (char *) polylist->polygons[j]->portproto, "node");
		xformpoly(polylist->polygons[j], trans);
		/* can only handle orthogonal rectangles for now */
		if (isbox(polylist->polygons[j], &b.lx, &b.hx, &b.ly, &b.hy) == 0)
			continue;
		ret = drcb_badbox(state, ni->geom, polylist->polygons[j]->layer,
						  ni->proto->tech, ni->parent, &b, &b, net, partial);
	}
	return(0);
}

/*
 * routine to check the design rules about arcinst "ai".  If "partial"
 * is nonzero, only check those other objects whose geom pointers are
 * greater than this one (i.e. only check any pair of objects once).
 * The routine returns nonzero if checking is to be terminated.
 */
static INTSML drcb_checkarcinst(STATE *state, ARCINST *ai, INTSML partial)
{
	REGISTER INTSML tot, j, ret;
	POLYLIST *polylist = state->polylist;
	Box b;

	/* get all of the polygons on this arc */
	tot = get_arcpolys(ai, polylist);
	if (tot < 0)
		return(1);

	/* examine the polygons on this arc */
	for(j=0; j<tot; j++)
	{
		if (polylist->polygons[j]->layer < 0) continue;
		/* can only handle orthogonal rectangles for now */
		if (isbox(polylist->polygons[j], &b.lx, &b.hx, &b.ly, &b.hy) == 0)
			continue;
		ret = drcb_badbox(state, ai->geom, polylist->polygons[j]->layer, ai->proto->tech,
			ai->parent, &b, &b, drcb_network(state->hai, (char *) ai, "arc"), partial);
	}
	return(0);
}

/*
 * routine to make a design-rule check on a piece of silicon which is
 * described in polygon "obj".  This silicon is on layer "nlayer" in
 * technology "tech", is part of the object at geometry module "geom",
 * is on electrical layer "nindex" and is in facet "facet".  If
 * "partial" is nonzero then only objects whose geometry modules are
 * greater than this one will be checked (to prevent any pair of objects
 * from being checked twice).  A positive return indicates that an error
 * has been detected and reported.  A negative return indicates that
 * checking should be terminated.  nhpp has the network tables for geom
 * if geom is a NODEINST, and can be NULL if geom is an ARCINST.  hni is
 * a hash table containing the network tables for all nodeinsts in
 * facet, keyed by the nodeinst address.
 */
static INTSML drcb_badbox(STATE *state, GEOM *geom, INTSML nlayer, TECHNOLOGY *tech,
	NODEPROTO *facet, Box *bp, Box *topbox, INTBIG nnet, INTSML partial)
{
	REGISTER GEOM *pi;
	REGISTER NODEINST *ni;
	REGISTER NODEPROTO *np;
	REGISTER ARCINST *ai;
	REGISTER INTBIG net;
	REGISTER INTBIG bound, search, dist;
	REGISTER INTSML j, tot, touch, ret;
	POLYLIST *subpolylist = state->subpolylist;
	XARRAY trans;
	Box b;
	INTSML con;
	Hashtable *pihpp;

	/* see how far around the box it is necessary to search */
	bound = maxdrcsurround(tech, nlayer);
	if (bound < 0) return(0);

	/* search in the area surrounding the box */
	search = initsearch(bp->lx-bound, bp->hx+bound, bp->ly-bound, bp->hy+bound, facet);
	for(;;)
	{
		if ((pi = nextobject(search)) == NOGEOM) break;
		if (pi == geom) continue;
		if (partial != 0 && pi < geom) continue;

		/* see if the objects directly touch */
		touch = drcb_objtouch(pi, geom);

		if (pi->entrytype == OBJNODEINST)
		{
			ni = pi->entryaddr.ni;   np = ni->proto;

			/* no hierarchical DRC here: ignore nodes that are not primitive */
			if (np->index == 0) continue;

			/* don't check between technologies */
			if (np->tech != tech) continue;

			/* prepare to examine every layer in this nodeinst */
			makerot(ni, trans);

			if (state->hni == NULL) continue;
			pihpp = (Hashtable *) hashsearch(state->hni, (char *) ni);
			if (pihpp == NULL)
			{
				ttyputerr("couldn't find pihpp for ni %s (0x%lx), hni 0x%lx",
						  describenodeinst(ni), (long) ni, (long) state->hni);
				continue;
			}

			/* get the shape of each nodeinst layer */
			tot = get_nodeEpolys(ni, subpolylist);
			if (tot < 0) return(-1);

			for(j=0; j<tot; j++)
			{
				if (subpolylist->polygons[j]->layer < 0) continue;

				/* find the global electrical net for this layer */
				if (subpolylist->polygons[j]->portproto != NOPORTPROTO)
				{
					net = drcb_network(pihpp, (char *)
									   subpolylist->polygons[j]->portproto, "node");
					if (net == 0)
					{
						continue;
					}
				} else
				{
					net = NONET;
				}

				/* see whether the two objects are electrically connected */
				con = (net == NONET || nnet == NONET || net == nnet);

				/* if they connect electrically and adjoin, don't check */
				if (con != 0 && touch != 0) continue;

				/* see how close they can get */
				dist = drcmindistance(tech, nlayer, subpolylist->polygons[j]->layer, con);
				if (dist < 0) continue;

				/* warning: does not handle arbitrary polygons, only boxes */
				xformpoly(subpolylist->polygons[j], trans);
				if (isbox(subpolylist->polygons[j], &b.lx, &b.hx, &b.ly, &b.hy) == 0)
					continue;

				/* check the distance */
				ret = drcb_checkdist(state, nlayer,nnet,geom, bp, topbox,
					subpolylist->polygons[j]->layer, net, pi, &b, dist, state->nhpp, pihpp);
				if (ret != 0) return(ret);
			}
		} else
		{
			ai = pi->entryaddr.ai;

			/* don't check between technologies */
			if (ai->proto->tech != tech) continue;

			if (state->hai == NULL)
				continue;
			/* see whether the two objects are electrically connected */
			net = drcb_network(state->hai, (char *) ai, "arc");
			if (nnet == NONET || net == nnet) con = 1; else con = 0;

			/* if they connect electrically and adjoin, don't check */
			if (con != 0 && touch != 0) continue;

			/* get the shape of each arcinst layer */
			tot = get_arcpolys(ai, subpolylist);
			if (tot < 0) return(-1);

			for(j=0; j<tot; j++)
			{
				if (subpolylist->polygons[j]->layer < 0) continue;

				/* see how close they can get */
				dist = drcmindistance(tech, nlayer, subpolylist->polygons[j]->layer, con);
				if (dist < 0) continue;

				/* warning: nonmanhattan arcs are ignored here */
				if (isbox(subpolylist->polygons[j], &b.lx, &b.hx, &b.ly, &b.hy) == 0)
					continue;

				/* check the distance */
				ret = drcb_checkdist(state, nlayer,nnet,geom, bp, topbox,
					subpolylist->polygons[j]->layer, net, pi, &b, dist, state->nhpp,
						(Hashtable *) NULL);
				if (ret != 0) return(ret);
			}
		}
	}
	return(0);
}


/*
 * routine to see if the object in "pos1" on layer "layer1" with
 * electrical network "net1" comes within "dist" from the object in
 * "pos2" on layer "layer2" with electrical network "net2".  The bounds
 * of object "pos1" are (lx1-hx1,ly1-hy1), and the bounds of object
 * "pos2" are (lx2-hx2,ly2-hy2).  A positive return indicates that an
 * error has been detected and reported.  A negative return indicates
 * that checking should be terminated.  hpp1 is the network table for
 * pos1, hpp2 is the network table for pos2, needed only if pos1 or pos2
 * respectively are NODEINSTs.  (for cropping with shapes on the same
 * networks) If pos1 or pos2 is an ARCINST, we don't need the
 * corresponding hpp, it can be NULL.
 */
static INTSML drcb_checkdist(STATE *state, INTSML layer1, INTBIG net1, GEOM *pos1,
	Box *bp1, Box *topbp1, INTSML layer2, INTBIG net2, GEOM *pos2, Box *bp2, INTBIG dist,
	Hashtable *hpp1, Hashtable *hpp2)
{
	extern AIDENTRY *us_aid;

	/* turn off flag that the nodeinst may be undersized */
	state->tinynodeinst = NONODEINST;

	/* crop out parts of any arc that is covered by an adjoining node */
	if (pos1->entrytype == OBJARCINST)
	{
		if (drcb_croparcinst(state, pos1->entryaddr.ai, layer1, bp1)) return(0);
	}
	if (pos2->entrytype == OBJARCINST)
	{
		if (drcb_croparcinst(state, pos2->entryaddr.ai, layer2, bp2)) return(0);
	}

	/* crop out parts of a box covered by a similar layer on the other node */
	if (pos1->entrytype == OBJNODEINST)
	{
		if (drcb_cropnodeinst(state, pos1->entryaddr.ni, layer2, net2, pos2, bp2, hpp1))
			return(0);
	}
	if (pos2->entrytype == OBJNODEINST)
	{
		if (drcb_cropnodeinst(state, pos2->entryaddr.ni, layer1, net1, pos1, bp1, hpp2))
			return(0);
	}

	/* now compare the box extents */
	return(drcb_compareboxes(state, layer1, net1, pos1, bp1, topbp1, layer2, net2, pos2, bp2, dist));
}


/*
 * routine to crop the box on layer "nlayer", electrical index "nindex"
 * and bounds (lx-hx, ly-hy) against the nodeinst "ni".  Only those layers
 * in the nodeinst that are the same layer and the same electrical network
 * are checked.  The routine returns nonzero if the bounds are reduced
 * to nothing.
 */
static INTSML drcb_cropnodeinst(STATE *state, NODEINST *ni, INTSML nlayer, INTBIG nnet,
	GEOM *npos, Box *bp, Hashtable *hpp)
{
	XARRAY trans;
	INTBIG  xl, xh, yl, yh, bx, ux, by, uy;
	REGISTER INTSML tot, j, isconnected;
	REGISTER INTBIG temp;
	REGISTER INTBIG net;
	POLYLIST *polylist = state->croppolylist;
	POLYGON *poly;

	tot = get_nodeEpolys(ni, polylist);
	if (tot < 0) return 1;
	isconnected = 0;
	for(j=0; j<tot; j++)
	{
		poly = polylist->polygons[j];
		if (poly->layer != nlayer) continue;
		if (nnet != NONET)
		{
			if (poly->portproto == NOPORTPROTO) continue;
			net = drcb_network(hpp, (char *) poly->portproto, "node");
			if (net != NONET && net != nnet) continue;
		}
		isconnected++;
		break;
	}
	if (isconnected == 0) return(0);

	/* get the description of the nodeinst layers */
	tot = get_nodepolys(ni, polylist);
	if (tot < 0) return 1;
	makerot(ni, trans);
	for(j=0; j<tot; j++)
	{
		poly = polylist->polygons[j];
		if (poly->layer != nlayer) continue;

		/* warning: does not handle arbitrary polygons, only boxes */
		if (isbox(poly, &xl, &xh, &yl, &yh) == 0) continue;
		xform(xl,yl, &bx,&by, trans);
		xform(xh,yh, &ux,&uy, trans);
		if (bx > ux) { temp = bx; bx = ux; ux = temp; }
		if (by > uy) { temp = by; by = uy; uy = temp; }
		temp = cropbox(&bp->lx,&bp->hx,&bp->ly,&bp->hy, bx,ux,by,uy);
		if (temp > 0) return(1);
		if (temp < 0)
		{
			state->tinynodeinst = ni;
			state->tinyobj = npos;
		}
	}
	return(0);
}

/*
 * routine to crop away any part of layer "lay" of arcinst "ai" that coincides
 * with a similar layer on a connecting nodeinst.  The bounds of the arcinst
 * are in the reference parameters (lx-hx, ly-hy).  The routine returns zero
 * normally, 1 if the arcinst is cropped into oblivion.
 */
static INTSML drcb_croparcinst(STATE *state, ARCINST *ai, INTSML lay, Box *bp)
{
	INTBIG bx, by, ux, uy, xl, xh, yl, yh;
	XARRAY trans;
	REGISTER INTSML i, j, tot;
	REGISTER INTBIG temp;
	REGISTER NODEINST *ni;
	REGISTER NODEPROTO *np;
	REGISTER PORTPROTO *pp;
	POLYLIST *polylist = state->croppolylist;
	POLYGON *poly;

	for(i=0; i<2; i++)
	{
		/* find the primitive nodeinst at the true end of the portinst */
		ni = ai->end[i].nodeinst;   np = ni->proto;
		pp = ai->end[i].portarcinst->proto;
		while (np->index == 0)
		{
			ni = pp->subnodeinst;   np = ni->proto;
			pp = pp->subportproto;
		}

		makerot(ni, trans);
		tot = get_nodepolys(ni, polylist);
		for(j=0; j<tot; j++)
		{
			poly = polylist->polygons[j];
			if (poly->layer != lay) continue;

			/* warning: does not handle arbitrary polygons, only boxes */
			if (isbox(poly, &xl, &xh, &yl, &yh) == 0) continue;
			xform(xl,yl, &bx,&by, trans);
			xform(xh,yh, &ux,&uy, trans);
			if (bx > ux) { temp = bx; bx = ux; ux = temp; }
			if (by > uy) { temp = by; by = uy; uy = temp; }

			temp = cropbox(&bp->lx,&bp->hx,&bp->ly,&bp->hy, bx,ux,by,uy);
			if (temp > 0) return(1);
			if (temp < 0)
			{
				state->tinynodeinst = ni;
				state->tinyobj = ai->geom;
			}
		}
	}
	return(0);
}

/* compares two boxes, and reports the error if they intersect */
static INTSML drcb_compareboxes(STATE *state, INTSML layer1, INTBIG net1, GEOM *pos1,
	Box *bp1, Box *topbp1, INTSML layer2, INTBIG net2, GEOM *pos2, Box *bp2, INTBIG dist)
{
	char *s1, *s2, *msg;

	msg = NULL;
	if (bp1->hx+dist>bp2->lx && bp1->lx-dist<bp2->hx && bp1->hy+dist>bp2->ly &&
		bp1->ly-dist<bp2->hy)
	{
		/* if a node was too small to crop an arc, it may explain the error */
		if (state->tinynodeinst != NONODEINST)
		{
			/* see if the node/arc that failed was involved in the error */
			if ((state->tinynodeinst->geom==pos1 || state->tinynodeinst->geom==pos2) &&
				 (state->tinyobj == pos1 || state->tinyobj == pos2))
			{
				s1 = describenodeinst(state->tinynodeinst);
				s2 = geomname(state->tinyobj);
#define ERRFMT	"%s is too small for the %s"
				msg = (char *) emalloc(strlen(s1) + strlen(s2) +
										sizeof(ERRFMT) + 1, dr_aid->cluster);
				(void) sprintf(msg, ERRFMT, s1, s2);
			}
		}
		if (intersectbox(&bp2->lx, &bp2->hx, &bp2->ly, &bp2->hy,
						 bp1->lx-dist, bp1->hx+dist, bp1->ly-dist, bp1->hy+dist) != 0)
		{
			ttyputerr("Something broke -- intersectbox returned non-zero");
		}
		drcb_report_error(topbp1->lx, topbp1->hx, topbp1->ly, topbp1->hy, pos1, pos2,
			layer1, layer2, net1, net2, ERR_SPACING, msg);
		return(1);
	}
	return(0);
}



/*
 * routine to tell whether the objects at geometry modules "po1" and "po2"
 * touch directly (that is, an arcinst connected to a nodeinst).  The routine
 * returns nonzero if they touch
 */
static INTSML drcb_objtouch(GEOM *po1, GEOM *po2)
{
	REGISTER GEOM *temp;
	REGISTER NODEINST *ni;
	REGISTER ARCINST *ai;
	REGISTER INTSML i;

	if (po1->entrytype == OBJNODEINST)
	{
		if (po2->entrytype == OBJNODEINST) return(0);
		temp = po1;   po1 = po2;   po2 = temp;
	}
	if (po2->entrytype == OBJARCINST)
		return(0);

	/* see if the arcinst at "po1" touches the nodeinst at "po2" */
	ni = po2->entryaddr.ni;
	ai = po1->entryaddr.ai;
	for(i=0; i<2; i++)
		if (ai->end[i].nodeinst == ni)
			return(1);
	return(0);
}

/*
 * Computes the drc box for a node instance - a drc box is the bounding
 * box of the node, extended outward by the maximum drid
 */
static void drcb_box(NODEINST *ni, INTBIG *lxp, INTBIG *hxp, INTBIG *lyp, INTBIG *hyp)
{
	*lxp = ni->geom->lowx - drcb_surround;
	*lyp = ni->geom->lowy - drcb_surround;
	*hxp = ni->geom->highx + drcb_surround;
	*hyp = ni->geom->highy + drcb_surround;

	if (debugflag)
	{
		ttyputmsg("%sndrcbox(%s) is %s<=X<=%s, %s<=Y<=%s", indent,
			describenodeinst(ni), latoa(*lxp), latoa(*lyp), latoa(*hxp), latoa(*hyp));
	}
}


/*
 * Allocate the contents of a state structure. If any of the allocations
 * fail, we don't bother to free all previous allocations -- too much
 * pain!!
 */
static INTSML allocstate(STATE *state)
{
	state->polylist = (POLYLIST *) emalloc(sizeof(POLYLIST), dr_aid->cluster);
	if (state->polylist == 0) return 0;
	state->polylist->polylistsize = 0;
	state->subpolylist = (POLYLIST *) emalloc(sizeof(POLYLIST), dr_aid->cluster);
	if (state->subpolylist == 0) return 0;
	state->subpolylist->polylistsize = 0;
	state->croppolylist = (POLYLIST *) emalloc(sizeof(POLYLIST),dr_aid->cluster);
	if (state->croppolylist == 0) return 0;
	state->croppolylist->polylistsize = 0;
	state->ntasks = 0;
	return 1;
}


static void freestate(STATE *state)
{
	efree((char *) state->polylist);
	efree((char *) state->subpolylist);
	efree((char *) state->croppolylist);
}

/*
 * Frees an R Tree hierarchy - traversing the tree in a depth first
 * search free'ing all nodes and their contents. This should probably be
 * an Electric database routine, but right now, it has knowledge of the
 * shape wired in here.  Should be abstracted out into a hook routine!!
 */
static void freertree(RTNODE *rtn)
{
	REGISTER INTSML j;
	SHAPE *shape;

	if (rtn->flag != 0)
	{
		for(j = 0; j < rtn->total; j++)
		{
			/* free shape */
			shape = (SHAPE *) (((GEOM *)(rtn->pointers[j]))->entryaddr.ni);
			if (shape->hpp)
				hashdestroy(shape->hpp, HASH_NULLPROC, HASH_NULL);
			efree((char *) shape);
			freegeom((GEOM *)(rtn->pointers[j]));
			rtn->pointers[j] = 0;
		}
		freertnode(rtn);
		return;
	}

	for (j = 0; j < rtn->total; j++)
	{
		/*
		 * !! Why are RTNODE pointers unsigned instead of generic
		 * pointers?
		 */
		freertree((RTNODE *)rtn->pointers[j]);
		rtn->pointers[j] = 0;
	}
	freertnode(rtn);
}

/*
 * Walks an R Tree hierarchy - traversing the tree in a depth first
 * search running the supplied function 'func' on all terminal nodes.
 * This should probably be an Electric database routine. 'func' is called
 * as (*func)(GEOM *geom, char *arg) where geom is a pointer to the
 * geometry object stored in that R tree node. 'func' may NOT modify the
 * geometry related contents of geom (since that would mean a change in
 * the R tree structure)
 */
static void walkrtree(RTNODE *rtn, void (*func)(GEOM*, char*), char *arg)
{
	REGISTER INTSML j;

	if (rtn->flag != 0)
	{
		for(j = 0; j < rtn->total; j++)
		{
			(*func)((GEOM *)(rtn->pointers[j]), arg);
		}
		return;
	}

	for (j = 0; j < rtn->total; j++)
	{
		/*
		 * !! Why are RTNODE pointers unsigned instead of generic
		 * pointers?
		 */
		walkrtree((RTNODE *)rtn->pointers[j], func, arg);
	}
}

/*
 * Computes the intersection of the two boxes and returns the
 * intersection in box1. Returns 0 for success, 1 for failure. (The
 * latter is if the boxes do not intersect - in the latter case, the
 * contents of box1 remain unchanged.) Intersection is strict - if the
 * boxes touch, the check returns 0.
 */
static INTSML intersectbox(INTBIG *lx1p, INTBIG *hx1p, INTBIG *ly1p, INTBIG *hy1p,
	INTBIG lx2, INTBIG hx2, INTBIG ly2, INTBIG hy2)
{
	INTBIG nlx, nhx, nly, nhy;

	nlx = (*lx1p > lx2) ? *lx1p : lx2;	/* MAX(lx1, lx2) */
	nhx = (*hx1p < hx2) ? *hx1p : hx2;	/* MIN(hx1, hx2) */
	if (nhx <= nlx) return 1;
	nly = (*ly1p > ly2) ? *ly1p : ly2;	/* MAX(ly1, ly2) */
	nhy = (*hy1p < hy2) ? *hy1p : hy2;	/* MAX(hy1, hy2) */
	if (nhy <= nly) return 1;
	*lx1p = nlx;
	*hx1p = nhx;
	*ly1p = nly;
	*hy1p = nhy;
	return 0;
}

/*
 * routine to accumulate a list of polygons at least "tot" long in the
 * polygon structure "list".  The routine returns nonzero if there is no
 * more memory.
 */
static INTSML drcb_ensurepolylist(POLYLIST *list, INTSML tot)
{
	REGISTER POLYGON **newpolylist;
	REGISTER INTSML j;

	/* make sure the list is the right size */
	if (list->polylistsize < tot)
	{
		newpolylist = (POLYGON **)emalloc(tot * (sizeof (POLYGON *)), dr_aid->cluster);
		if (newpolylist == 0)
		{
			ttyputerr("No memory for DRC");
			return(1);
		}
		for(j=0; j<tot; j++) newpolylist[j] = NOPOLYGON;
		if (list->polylistsize != 0)
		{
			for(j=0; j<list->polylistsize; j++)
				newpolylist[j] = list->polygons[j];
			efree((char *)list->polygons);
		}
		list->polygons = newpolylist;
		list->polylistsize = tot;
	}

	/* make sure there is a polygon for each entry in the list */
	for(j=0; j<tot; j++) if (list->polygons[j] == NOPOLYGON)
		list->polygons[j] = allocpolygon(4, dr_aid->cluster);
	return(0);
}


/*
 * Make sure the current window shows the node prototype 'np'. If not,
 * show np in the current window and return a pointer to the nodeproto
 * that was being displayed before.
 */
static NODEPROTO *drcb_ensurewindow(NODEPROTO *np)
{
	char *par;
	NODEPROTO *oldnp;

	if (np == NONODEPROTO || el_curwindow == NOWINDOW)
		return NONODEPROTO;

	oldnp = getcurfacet();
	if (oldnp != np)
	{
		par = describenodeproto(np);
		us_editfacet(1, &par);
	}
	return oldnp;
}

#ifdef	OTHERRULES
/*
 * These routines should be in dbtech.c. In particular,
 * tech_initdrcminsizes() should be merged into
 * tech_initmaxdrcsurround().
 */

static INTBIG *tech_drcmaxsizes = 0;	/* cache for "drcmaxsize" */
static INTBIG *tech_drcminsizes = 0;	/* cache for "drcminsize" */
static INTBIG *tech_drcenclosures = 0;	/* cache for "drcminenclose" */

/*
 * this should be called whenever "DRC_min_sizes" or "DRC_max_sizes" or
 * "DRC_min_enclosures" changes!
 */
/*
 * routine to initialize the database variables "DRC_max_sizes",
 * "DRC_min_sizes", and "DRC_min_enclosures".  This is called once at
 * initialization and again whenever the arrays are changed.
 */
void tech_initdrcminsizes(void)
{
	REGISTER VARIABLE *var;
	REGISTER TECHNOLOGY *t;
	static INTBIG DRC_min_sizes = 0, DRC_max_sizes = 0, DRC_min_enclosures = 0;

	if (DRC_max_sizes == 0) DRC_max_sizes = makekey("DRC_max_sizes");
	if (DRC_min_sizes == 0) DRC_min_sizes = makekey("DRC_min_sizes");
	if (DRC_min_enclosures == 0) DRC_min_enclosures = makekey("DRC_min_enclosures");

	if (tech_drcmaxsizes != 0) efree((char *)tech_drcmaxsizes);
	if (tech_drcminsizes != 0) efree((char *)tech_drcminsizes);
	if (tech_drcenclosures != 0) efree((char *)tech_drcenclosures);

	tech_drcmaxsizes = emalloc(el_maxtech * SIZEOFINTBIG, db_cluster);
	if (tech_drcmaxsizes == 0) return;
	tech_drcminsizes = emalloc(el_maxtech * SIZEOFINTBIG, db_cluster);
	if (tech_drcminsizes == 0) return;
	tech_drcenclosures = emalloc(el_maxtech * SIZEOFINTBIG, db_cluster);
	if (tech_drcenclosures == 0) return;

	for(t = el_technologies; t != NOTECHNOLOGY; t = t->nexttechnology)
	{
		tech_drcmaxsizes[t->index] = 0;
		tech_drcminsizes[t->index] = 0;
		tech_drcenclosures[t->index] = 0;

		var = getvalkey((INTBIG)t, VTECHNOLOGY, VFRACT|VISARRAY, DRC_max_sizes);
		if (var != NOVARIABLE) tech_drcmaxsizes[t->index] = var->addr;
		var = getvalkey((INTBIG)t, VTECHNOLOGY, VFRACT|VISARRAY, DRC_min_sizes);
		if (var != NOVARIABLE) tech_drcminsizes[t->index] = var->addr;
		var = getvalkey((INTBIG)t, VTECHNOLOGY, VFRACT|VISARRAY, DRC_min_enclosures);
		if (var != NOVARIABLE) tech_drcenclosures[t->index] = var->addr;
	}
}

/*
 * routine to tell the minimum enclosure of "layer1" by "layer2". A
 * positive distance means that layer2 must surround layer1 by at least
 * that value, a negative distance means that layer1 must surround layer2
 * by that value. (Zero means the two layers must at least be the same
 * size) The distance is returned in *dist. The function returns 0 to
 * indicate a valid distance, and -1 to indicate a don't care (i.e. -1
 * means no enclosure exists for these two layers)
 */
INTSML drcminenclosure(TECHNOLOGY *tech, INTSML layer1, INTSML layer2, INTBIG *dist)
{
	REGISTER INTBIG i;
	REGISTER INTSML index, temp;
	REGISTER INTBIG addr;

	if (layer1 < 0 || layer2 < 0) return(-1);

	/* make sure cache of information is valid */
	if (tech_drcenclosures == 0)
	{
		tech_initdrcminsizes();
		if (tech_drcenclosures == 0) return(-1);
	}
	addr = tech_drcenclosures[tech->index];
	if (addr == 0) return(-1);

	/* compute index into upper triangular matrix representing enclosures */
	if (layer1 > layer2)
	{
		temp = layer1;
		layer1 = layer2;
		layer2 = temp;
	}
	index = (layer1+1) * (layer1/2) + (layer1&1) * ((layer1+1)/2);
	index = layer2 + tech->layercount * layer1 - index;

	/* get the un-scaled distance */
	i = ((INTBIG *)addr)[index];

	/* scale result for current lambda value */
	*dist = i * tech->deflambda / WHOLE;
	return(0);
}

/*
 * routine to tell the minimum size that a shape on layer "layer" must be
 * This routine accesses the database variable "DRC_min_sizes" in the
 * technologies. Returns XX if the rule doesn't apply to the layer.
 */
INTBIG drcminsize(TECHNOLOGY *tech, INTSML layer)
{
	REGISTER INTBIG i, addr;

	if (layer < 0 || layer >= tech->layercount) return(XX);

	/* make sure cache of information is valid */
	if (tech_drcminsizes == 0)
	{
		tech_initdrcminsizes();
		if (tech_drcminsizes == 0) return(XX);
	}

	addr = tech_drcminsizes[tech->index];
	if (addr == 0) return(XX);
	i = ((INTBIG *)addr)[layer];
	if (i < 0) return(XX);
	i = i*tech->deflambda/WHOLE;
	return(i);
}

/*
 * routine to tell the maximum size that a shape on layer "layer" must be
 * This routine accesses the database variable "DRC_max_sizes" in the
 * technologies. Returns XX if the rule doesn't apply to the layer.
 */
INTBIG drcmaxsize(TECHNOLOGY *tech, INTSML layer)
{
	REGISTER INTBIG i, addr;

	if (layer < 0 || layer >= tech->layercount) return(XX);

	/* make sure cache of information is valid */
	if (tech_drcmaxsizes == 0)
	{
		tech_initdrcminsizes();
		if (tech_drcmaxsizes == 0) return(XX);
	}

	addr = tech_drcmaxsizes[tech->index];
	if (addr == 0) return(XX);
	i = ((INTBIG *)addr)[layer];
	if (i < 0) return(XX);
	i = i*tech->deflambda/WHOLE;
	return(i);
}
#endif

/**********************************************************************/

/*
 * The following are versions of the same network functions as above, but
 * implemented using hash tables to make them re-entrant, and usable on
 * a multiprocessor.  They don't use the temp2 variables on the
 * PORTPROTOs and ARCINSTs.
 */
void drcb_print_nets2(NODEPROTO *np, INTBIG *netindex)
{
	Hashtable *ht;

	ht = hashcreate(HTABLE_PORT);
	drcb_get_init_nets(np, ht, netindex);
	drcb_flatnet2(np, ht, netindex);
	hashdestroy(ht, HASH_NULLPROC, HASH_NULL);
	(*netindex)--;
	ttyputmsg("%ld network%s", (long) *netindex, (*netindex == 1)? "" : "s");
	(*netindex)++;
}


/*
 * Initializes the port numbers for a toplevel nodeproto np, starting
 * the net numbers from *netindex, and storing all portprotos and their
 * nets in ht
 */
void drcb_get_init_nets(NODEPROTO *np, Hashtable *ht, INTBIG *netindex)
{
	PORTPROTO *pp, *spp;
	char *cp;

	/* re-assign net-list values, starting at each port of the top facet */
	for(pp=np->firstportproto; pp!=NOPORTPROTO; pp=pp->nextportproto)
	{
		for(spp=np->firstportproto; spp!=pp; spp=spp->nextportproto)
		{
			if (spp->network != pp->network) continue;
			/* this port is the same as another one, use same net */
			cp = hashsearch(ht, (char *) spp);
			if (cp == NULL)
			{
				ttyputerr("couldn't find portproto 0x%lx in hash", (long) spp);
				return;
			}
			if (hashinsert(ht, (char *)pp, cp, 0) != 0)
			{
				ttyputerr("couldn't insert portproto 0x%lx, net %ld in hash", (long) pp, (long) cp);
				return;
			}
			break;
		}
		/* assign a new net number if the loop terminated normally */
		if (spp == pp)
		{
			cp = (char *) *netindex;
			if (hashinsert(ht, (char *)pp, cp, 0) != 0)
			{
				ttyputerr("couldn't insert portproto 0x%lx, net %ld in hash", (long) pp, (long) cp);
				return;
			}
			(*netindex)++;
		}
	}
}

/*
 * generate a hash table of (arcinst, network numbers) pairs, keyed by
 * arcinst for a NODEPROTO from the hash table for portprotos (hpp) in
 * the parent NODEPROTO of the NODEINST.  The net numbers are propagated
 * to all arcs.  Returns the table in hai, a hash table passed in by
 * the parent.
 */
void drcb_get_arc_networks(NODEPROTO *np, Hashtable *hpp, Hashtable *hai, INTBIG *netindex)
{
	char *cp;
	REGISTER ARCINST *ai;
	REGISTER PORTPROTO *pp;
	REGISTER PORTARCINST *pi;

	/* propagate port numbers to arcs in the nodeproto */
	for(pp = np->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
	{
		/* find a connecting arc on the node that this port comes from */
		for(pi = pp->subnodeinst->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
		{

			ai = pi->conarcinst;
			if (ai->network != pp->network) continue;
			if (hashsearch(hai, (char *)ai) != 0) continue;
			cp = hashsearch(hpp, (char *) pp);
			if (cp == NULL)
			{
				ttyputerr("couldn't find portproto 0x%lx in hpp", (long) hpp);
				return;
			}
			drcb_flatprop2(ai, hai, cp);
			break;
		}
	}

	/* set node numbers on arcs that are local */
	for(ai = np->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
	{
		cp = hashsearch(hai, (char *)ai);
		if (cp == NULL)
		{
			drcb_flatprop2(ai, hai, (char *) *netindex);
			(*netindex)++;
		}
	}
}

/*
 * generate a hash table of (portproto, network numbers) pairs, keyed by
 * portproto for a NODEINST from corresponding hash tables for
 * portprotos (hpp) and arcs (hai) in the NODEINST's parent proto.  The
 * net numbers are derived from the arcs and exported ports in the
 * NODEINST.  Returns the table in nhpp, a hash table passed in by the
 * parent.
 */
void drcb_get_nodeinst_networks(NODEINST *ni, Hashtable *hpp, Hashtable *hai,
	Hashtable *nhpp, INTBIG *netindex)
{
	char *cp;
	REGISTER PORTPROTO *pp, *spp;
	REGISTER PORTARCINST *pi;
	REGISTER PORTEXPINST *pe;

	for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
	{
		pp = pi->proto;
		if (hashsearch(nhpp, (char *) pp) != 0) continue;
		cp = hashsearch(hai, (char *) pi->conarcinst);
		if (cp == 0)
		{
			ttyputerr("couldn't find conarcinst 0x%lx in hai", (long) pi->conarcinst);
			return;
		}
		if (hashinsert(nhpp, (char *) pp, cp, 0) != 0)
		{
			ttyputerr("couldn't insert portproto 0x%lx, net %ld in nhpp", (long) pp, (long) cp);
			return;
		}
	}
	for(pe = ni->firstportexpinst; pe != NOPORTEXPINST; pe = pe->nextportexpinst)
	{
		pp = pe->proto;
		if (hashsearch(nhpp, (char *) pp) != 0) continue;
		cp = hashsearch(hpp, (char *) pe->exportproto);
		if (cp == 0)
		{
			ttyputerr("couldn't find exportproto 0x%lx in hpp", (long) pe->exportproto);
			return;
		}
		if (hashinsert(nhpp, (char *) pp, cp, 0) != 0)
		{
			ttyputerr("couldn't insert pe->proto 0x%lx, net %ld in nhpp", (long) pp, (long) cp);
			return;
		}
	}

	/* look for unassigned ports and give new or copied network numbers */
	for(pp = ni->proto->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
	{
		if (hashsearch(nhpp, (char *) pp) != 0) continue;
		cp = 0;
		for(spp = ni->proto->firstportproto; spp != NOPORTPROTO;
			spp = spp->nextportproto)
		{
			if (spp->network != pp->network) continue;
			cp = hashsearch(nhpp, (char *) spp);
			if (cp == 0) continue;
			if (hashinsert(nhpp, (char *) pp, cp, 0) != 0)
			{
				ttyputerr("couldn't insert unassigned pp 0x%lx, net %ld in nhpp", (long) pp, (long) cp);
				return;
			}
			break;
		}
		if (cp == 0)
		{
			cp = (char *) *netindex;
			if (hashinsert(nhpp, (char *) pp, cp, 0) != 0)
			{
				ttyputerr("couldn't insert unassigned pp 0x%lx, net %ld in nhpp", (long) pp, (long) cp);
				return;
			}
			(*netindex)++;
		}
	}
}

/*
 * recursive routine to walk the hierarchy, flattening the network,
 * numbering all portprotos and counting them.
 */
void drcb_flatnet2(NODEPROTO *np, Hashtable *hpp, INTBIG *netindex)
{
	REGISTER NODEINST *ni;
	INTSML printing_proto = 0;
	INTSML narcs, npp, npa, npe, ninst;
	Hashtable *hai;
	Hashtable *nhpp = hashcreate(HTABLE_PORT);

	inc_indent();

	if (nhpp == NULL)
	{
		ttyputerr("couldn't allocate nhpp for nodeproto %s", describenodeproto(np));
		return;
	}

	/*
	 * use the net numbers for portprotos computed by the parent
	 * nodeproto to number all arcs
	 */
	hai = hashcreate(HTABLE_ARC);
	if (hai == NULL)
	{
		ttyputerr("couldn't allocate hai for nodeproto %s", describenodeproto(np));
		hashdestroy(nhpp, HASH_NULLPROC, HASH_NULL);
		return;
	}
	drcb_get_arc_networks(np, hpp, hai, netindex);

	if (np->temp1 == 0)
	{
		REGISTER ARCINST *ai;
		REGISTER PORTPROTO *pp;

		np->temp1++;
		printing_proto++;
		/* Just gather some stats for inquiring minds */
		narcs = 0;
		for(ai = np->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst) narcs++;
		npp = 0;
		for(pp = np->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto) npp++;
		ttyputmsg("%s%s, %d arcs, %d portprotos", indent, describenodeproto(np), narcs, npp);
	}

	/* write every node in the nodeproto */
	ninst = 0;
	for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		ninst++;

		/* number networks for the child nodeinst */
		drcb_get_nodeinst_networks(ni, hpp, hai, nhpp, netindex);

		if (ni->proto->index == 0)
		{
			if (printing_proto)
			{
				REGISTER PORTARCINST *pi;
				REGISTER PORTEXPINST *pe;

				npa = 0;
				for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
					npa++;
				npe = 0;
				for(pe = ni->firstportexpinst; pe != NOPORTEXPINST; pe = pe->nextportexpinst)
					npe++;
				ttyputmsg("%s%d: nodeinst %s, npa = %d, npe = %d",
						  indent, ninst, describenodeinst(ni), npa, npe);
			}
			/* recurse */
			drcb_flatnet2(ni->proto, nhpp, netindex);
		}
		/*
		 * clear the nhpp hashtable for the next nodeinst.  Saves having
		 * to allocate one every time.
		 */
		hashclear(nhpp);
	}
	hashdestroy(hai, HASH_NULLPROC, HASH_NULL);
	hashdestroy(nhpp, HASH_NULLPROC, HASH_NULL);
	dec_indent();
}

/*
 * routine to propagate the node number "index" to the "temp2" of arcs
 * connected to arc "ai".
 */
void drcb_flatprop2(ARCINST *ai, Hashtable *hai, char *index)
{
	REGISTER ARCINST *oai;
	REGISTER NODEINST *ni;
	REGISTER PORTARCINST *pi;
	REGISTER INTSML i;

	/* set this arcinst to the current node number */
	if (hashinsert(hai, (char *) ai, index, 0) != 0)
	{
		ttyputerr("couldn't insert arc 0x%lx, net %ld in hai", (long) ai, (long) index);
		return;
	}

	/* if signals do not flow through this arc, do not propagate */
	if (((ai->proto->userbits&AFUNCTION) >> AFUNCTIONSH) == APNONELEC) return;

	/* recursively set all arcs and nodes touching this */
	for(i=0; i<2; i++)
	{
		if ((ai->end[i].portarcinst->proto->userbits&PORTISOLATED) != 0) continue;
		ni = ai->end[i].nodeinst;
		for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
		{
			/* select an arc that has not been examined */
			oai = pi->conarcinst;
			if (hashsearch(hai, (char *) oai) != 0) continue;

			/* see if the two ports connect electrically */
			if (ai->end[i].portarcinst->proto->network != pi->proto->network) continue;

			/* recurse on the nodes of this arc */
			drcb_flatprop2(oai, hai, index);
		}
	}
}

/***************** PRIMITIVE NODE/ARCINST TO SHAPES  ******************/

/*
 * These routines get all the polygons in a primitive instance, and
 * store them in the POLYLIST structure, returning a count of the number
 * of polygons.   They all return -1 if an allocation fails.
 */
static INTSML get_nodeEpolys(NODEINST *ni, POLYLIST *plist)
{
	INTSML tot;
	INTSML j;

	tot = nodeEpolys(ni);
	if (drcb_ensurepolylist(plist, tot) != 0) return -1;
	for(j = 0; j < tot; j++)
		shapeEnodepoly(ni, j, plist->polygons[j]);
	return tot;
}

static INTSML get_nodepolys(NODEINST *ni, POLYLIST *plist)
{
	INTSML tot;
	INTSML j;

	tot = nodepolys(ni);
	if (drcb_ensurepolylist(plist, tot) != 0) return -1;
	for(j = 0; j < tot; j++)
		shapenodepoly(ni, j, plist->polygons[j]);
	return tot;
}

static INTSML get_arcpolys(ARCINST *ai, POLYLIST *plist)
{
	INTSML tot;
	INTSML j;

	tot = arcpolys(ai);
	if (drcb_ensurepolylist(plist, tot) != 0) return -1;
	for(j = 0; j < tot; j++)
		shapearcpoly(ai, j, plist->polygons[j]);
	return tot;
}

/*********************************** HASH TABLE ***********************************/

static int hashindex (char *s)
{
	register int shift = 0;
	register int index = 0;

	while (*s != 0)
	{
		index ^= *(s++) << shift;
		shift = (shift == 0) ? 8 : 0;
	}

	return index;
}


/*
 * calls proc(key, datum, arg) for each item in the hash table. proc may
 * return a char *, in which case the walk terminates and the value is
 * returned to the caller
 */
char *hashwalk(Hashtable *hash, char *(*proc)(char*, char*, char*), char *arg)
{
	int i;
	char *ret;

	for(i = 0; i < hash->size; i++)
	{
		if (hash->table[i].key != NULL)
		{
			ret = (* proc)(hash->table[i].key, hash->table[i].datum, arg);
			if (ret != NULL) return ret;
		}
	}
	return NULL;
}


/*
 * Frees a created hash table. It first walks the hash table, calling
 * free_proc for every element in the table. free_proc should return
 * NULL on each element or the free walk will terminate. The table will
 * be freed anyway.
 */
void hashdestroy(Hashtable *hash, char *(*free_hook)(char*, char*, char*), char *arg)
{
	if (free_hook != NULL)
		(void) hashwalk(hash, free_hook, arg);
	efree((char *) hash->table);
	efree((char *) hash);
}

/*
 * Even though this function can be accomplished with hashwalk, it's
 * useful and small enough to deserve special-casing.
 */
void hashclear(Hashtable *hash)
{
	register int i;

	for(i = hash->size - 1; i >= 0; i--)
	{
		hash->table[i].key = NULL;
		hash->table[i].datum = NULL;
	}
}


static char *grow_hook(char *key, char *datum, char *arg)
{
	/*
	 * !! we need a version for hashstr insert.  Why not export hashindex,
	 * and let the user hash everything?  Much more general.
	 */
	(void) hashinsert((Hashtable *) arg, key, datum, 0);
	return 0;
}

/*
 * grows a table to a new size.  returns 0 if it failed, in which case
 * the old table is still intact, 1 if it succeeds, in which case hash
 * will contain the new table
 */
INTSML hashgrow(Hashtable *hash, INTSML newsize)
{
	Hashtable *newh = hashcreate(newsize);

	if (newh == NULL) return 0;
	(void) hashwalk(hash, grow_hook, (char *) newh);
	(void) efree((char *) hash->table);
	*hash = *newh;
	(void) efree((char *) newh);
	return 1;
}


/*
 * copies a table to a new table.  returns 0 if it failed, or the new
 * table if it succeeded.
 */
Hashtable *hashcopy(Hashtable *hash)
{
	Hashtable *newh = hashcreate((INTSML)hash->size);
	register int n;
	register Hashrec *dhrp;
	register Hashrec *shrp;

	if (newh == NULL) return 0;
	for(dhrp = newh->table, shrp = hash->table, n = hash->size; n > 0; n--)
	{
		dhrp->key = shrp->key;
		dhrp->datum = shrp->datum;
		dhrp++;
		shrp++;
	}
	return newh;
}


/*
 * Creates a hashtable 'size' elements long. size should be a prime
 * number. The user should not poke around inside the hash table
 */
Hashtable *hashcreate(INTSML size)
{
	Hashtable *newh;
	int i;

	if (size < 5)	/* Minimum size for this to work */
		return NULL;
	newh = (Hashtable *) emalloc (sizeof (Hashtable), dr_aid->cluster);
	if (newh == NULL) return NULL;
	newh->table = (Hashrec *) emalloc(size * (sizeof (Hashrec)), dr_aid->cluster);
	if (newh->table == NULL)
	{
		efree((char *) newh);
		return NULL;
	}
	for(i=0; i<size; i++) newh->table[i].key = newh->table[i].datum = 0;
	newh->size = size;

	newh->probe = size / 3;
	if (newh->probe * 3 == size) newh->probe--;
	if (newh->probe % 2 == 0) newh->probe--;

	return newh;
}

static char *strgrow_hook(char *key, char *datum, char *arg)
{
	(void) hashstrinsert((Hashtable *) arg, key, datum, 0);
	return 0;
}

/*
 * grows a table to a new size.  returns 0 if it failed, in which case
 * the old table is still intact, 1 if it succeeds, in which case hash
 * will contain the new table
 */
int hashstrgrow(Hashtable *hash, INTSML newsize)
{
	Hashtable *newh = hashcreate(newsize);

	if (newh == NULL) return 0;
	(void) hashwalk(hash, strgrow_hook, (char *) newh);
	(void) efree((char *) hash->table);
	*hash = *newh;
	(void) efree((char *) newh);
	return 1;
}


/*
 * inserts an item in the hash table.returns 0 if it succeeds, If
 * replace is true, and the key already exists in the table, then it
 * replaces the old datum with the new one. If replace is false, the it
 * only inserts the datum if the key is not found in the table and
 * there's enough space. if replace is false, it returns HASH_EXISTS if
 * it finds the item in the table already, HASH_FULL if the table couldn't
 * be grown.  key is assumed to be a null terminated character string.
 */
int hashstrinsert (Hashtable *hash, char *key, char *datum, int replace)
{
	register int index;
	register Hashrec *hashptr = hash->table;
	register int countdown = hash->size;

	index = hashindex (key) % hash->size;
	hashptr += index;

	while (hashptr->key && --countdown)
	{
		if (strcmp (hashptr->key, key) == 0)
		{
			if (replace)
			{
				hashptr->datum = datum;
				return 0;
			} else
			{
				return HASH_EXISTS;
			}
		}
		index = (index + hash->probe) % hash->size;
		hashptr = hash->table + index;
	}

	if (countdown)
	{
		hashptr->key = key;
		hashptr->datum = datum;
	} else
	{
		int newsize = pick_prime(hash->size * 3);
		if (newsize == 0) return HASH_FULL;
		if (hashstrgrow(hash, (INTSML)newsize) == 0) return HASH_FULL;
		return hashstrinsert(hash, key, datum, replace);
	}
	return 0;
}

/*
 * looks for key in the hash table.  key is assumed to be a null
 * terminated character string.
 */
char *hashstrsearch (Hashtable *hash, char *key)
{
	register int index;
	register Hashrec *hashptr = hash->table;
	register int countdown = hash->size;

	index = hashindex (key) % hash->size;
	hashptr += index;

	while (hashptr->key && strcmp (hashptr->key, key) && --countdown)
	{
		index = (index + hash->probe) % hash->size;
		hashptr = hash->table + index;
	}

	if (hashptr->key && countdown)
		return (hashptr->datum);
	else
		return ((char *) 0);
}

/*
 * inserts a pointer item in the hash table. returns 0 if it succeeds, If
 * replace is true, and the key already exists in the table, then it
 * replaces the old datum with the new one. If replace is false, the it
 * only inserts the datum if the key is not found in the table and
 * there's enough space. if replace is false, it returns HASH_EXISTS if
 * it finds the item in the table already, HASH_FULL if the table has no
 * room in it and it was unable to grow it to a new size.
 * !! (hashstrinsert does not grow the table yet)
 */
int hashinsert (Hashtable *hash, char *key, char *datum, int replace)
{
	register int index;
	register Hashrec *hashptr = hash->table;
	/*
	 * we should probably make countdown smaller so we don't walk
	 * through the entire table when it gets too full
	 */
	register int countdown = hash->size;

	index = ((long) key) % hash->size;
	hashptr += index;

	while (hashptr->key && --countdown)
	{
		if (hashptr->key == key)
		{
			if (replace)
			{
				hashptr->datum = datum;
				return 0;
			} else
			{
				return HASH_EXISTS;
			}
		}
		index = (index + hash->probe) % hash->size;
		hashptr = hash->table + index;
	}

	if (countdown)
	{
		hashptr->key = key;
		hashptr->datum = datum;
	} else
	{
		int newsize = pick_prime(hash->size * 3);
		if (newsize == 0) return HASH_FULL;
		if (hashgrow(hash, (INTSML)newsize) == 0) return HASH_FULL;
		return hashinsert(hash, key, datum, replace);
	}
	return 0;
}

/*
 * looks for key in the hash table.  Key is treated as an opaque pointer
 * value
 */
static Hashrec *findhashrec(Hashtable *hash, char *key)
{
	register int index;
	register Hashrec *hashptr = hash->table;
	register int countdown = hash->size;

	index = ((long) key) % hash->size;
	hashptr += index;

	while (hashptr->key && hashptr->key != key && --countdown)
	{
		index = (index + hash->probe) % hash->size;
		hashptr = hash->table + index;
	}

	if (hashptr->key && countdown)
		return hashptr;
	else
		return 0;
}

char *hashsearch (Hashtable *hash, char *key)
{
	Hashrec *hashptr = findhashrec(hash, key);

	if (hashptr)
		return hashptr->datum;
	else
		return 0;
}


/*
 * This will find key in hash, return the datum, return the key in
 * keyret, and delete the entry from the hash table.
 */
char *hashdelete(Hashtable *hash, char *key, char **keyret)
{
	register int index;
	register int hval;
	char *ret;
	register int countdown = hash->size;
	Hashrec *hashptr = findhashrec(hash, key);

	if (hashptr == 0) return 0;

	ret = hashptr->datum;
	*keyret = hashptr->key;
	hval = index = ((long) key) % hash->size;

	while (--countdown)
	{
		Hashrec *nhp;

		index = (index + hash->probe) % hash->size;
		nhp = hash->table + index;
		if (nhp && (((long) nhp->key) % hash->size) == hval)
		{
			*hashptr = *nhp;
		} else
		{
			hashptr->key = hashptr->datum = NULL;
			break;
		}
	}
	return ret;
}

/*
 * max_checkable is as high as we'd ever want to check for
 * primes.  If you raise it, you have to generate more primes for
 * the small_primes table.  max_checkable should not be a prime.
 */
static long max_checkable = 20000000L;

/* This table contains all primes under sqrt(max_checkable) */
static int small_primes[] = {
	2, 3, 5, 7, 11, 13, 17, 19, 23, 29, 31, 37, 41, 43, 47, 53, 59,
	61, 67, 71, 73, 79, 83, 89, 97, 101, 103, 107, 109, 113, 127,
	131, 137, 139, 149, 151, 157, 163, 167, 173, 179, 181, 191,
	193, 197, 199, 211, 223, 227, 229, 233, 239, 241, 251, 257,
	263, 269, 271, 277, 281, 283, 293, 307, 311, 313, 317, 331,
	337, 347, 349, 353, 359, 367, 373, 379, 383, 389, 397, 401,
	409, 419, 421, 431, 433, 439, 443, 449, 457, 461, 463, 467,
	479, 487, 491, 499, 503, 509, 521, 523, 541, 547, 557, 563,
	569, 571, 577, 587, 593, 599, 601, 607, 613, 617, 619, 631,
	641, 643, 647, 653, 659, 661, 673, 677, 683, 691, 701, 709,
	719, 727, 733, 739, 743, 751, 757, 761, 769, 773, 787, 797,
	809, 811, 821, 823, 827, 829, 839, 853, 857, 859, 863, 877,
	881, 883, 887, 907, 911, 919, 929, 937, 941, 947, 953, 967,
	971, 977, 983, 991, 997, 1009, 1013, 1019, 1021, 1031, 1033,
	1039, 1049, 1051, 1061, 1063, 1069, 1087, 1091, 1093, 1097,
	1103, 1109, 1117, 1123, 1129, 1151, 1153, 1163, 1171, 1181,
	1187, 1193, 1201, 1213, 1217, 1223, 1229, 1231, 1237, 1249,
	1259, 1277, 1279, 1283, 1289, 1291, 1297, 1301, 1303, 1307,
	1319, 1321, 1327, 1361, 1367, 1373, 1381, 1399, 1409, 1423,
	1427, 1429, 1433, 1439, 1447, 1451, 1453, 1459, 1471, 1481,
	1483, 1487, 1489, 1493, 1499, 1511, 1523, 1531, 1543, 1549,
	1553, 1559, 1567, 1571, 1579, 1583, 1597, 1601, 1607, 1609,
	1613, 1619, 1621, 1627, 1637, 1657, 1663, 1667, 1669, 1693,
	1697, 1699, 1709, 1721, 1723, 1733, 1741, 1747, 1753, 1759,
	1777, 1783, 1787, 1789, 1801, 1811, 1823, 1831, 1847, 1861,
	1867, 1871, 1873, 1877, 1879, 1889, 1901, 1907, 1913, 1931,
	1933, 1949, 1951, 1973, 1979, 1987, 1993, 1997, 1999, 2003,
	2011, 2017, 2027, 2029, 2039, 2053, 2063, 2069, 2081, 2083,
	2087, 2089, 2099, 2111, 2113, 2129, 2131, 2137, 2141, 2143,
	2153, 2161, 2179, 2203, 2207, 2213, 2221, 2237, 2239, 2243,
	2251, 2267, 2269, 2273, 2281, 2287, 2293, 2297, 2309, 2311,
	2333, 2339, 2341, 2347, 2351, 2357, 2371, 2377, 2381, 2383,
	2389, 2393, 2399, 2411, 2417, 2423, 2437, 2441, 2447, 2459,
	2467, 2473, 2477, 2503, 2521, 2531, 2539, 2543, 2549, 2551,
	2557, 2579, 2591, 2593, 2609, 2617, 2621, 2633, 2647, 2657,
	2659, 2663, 2671, 2677, 2683, 2687, 2689, 2693, 2699, 2707,
	2711, 2713, 2719, 2729, 2731, 2741, 2749, 2753, 2767, 2777,
	2789, 2791, 2797, 2801, 2803, 2819, 2833, 2837, 2843, 2851,
	2857, 2861, 2879, 2887, 2897, 2903, 2909, 2917, 2927, 2939,
	2953, 2957, 2963, 2969, 2971, 2999, 3001, 3011, 3019, 3023,
	3037, 3041, 3049, 3061, 3067, 3079, 3083, 3089, 3109, 3119,
	3121, 3137, 3163, 3167, 3169, 3181, 3187, 3191, 3203, 3209,
	3217, 3221, 3229, 3251, 3253, 3257, 3259, 3271, 3299, 3301,
	3307, 3313, 3319, 3323, 3329, 3331, 3343, 3347, 3359, 3361,
	3371, 3373, 3389, 3391, 3407, 3413, 3433, 3449, 3457, 3461,
	3463, 3467, 3469, 3491, 3499, 3511, 3517, 3527, 3529, 3533,
	3539, 3541, 3547, 3557, 3559, 3571, 3581, 3583, 3593, 3607,
	3613, 3617, 3623, 3631, 3637, 3643, 3659, 3671, 3673, 3677,
	3691, 3697, 3701, 3709, 3719, 3727, 3733, 3739, 3761, 3767,
	3769, 3779, 3793, 3797, 3803, 3821, 3823, 3833, 3847, 3851,
	3853, 3863, 3877, 3881, 3889, 3907, 3911, 3917, 3919, 3923,
	3929, 3931, 3943, 3947, 3967, 3989, 4001, 4003, 4007, 4013,
	4019, 4021, 4027, 4049, 4051, 4057, 4073, 4079, 4091, 4093,
	4099, 4111, 4127, 4129, 4133, 4139, 4153, 4157, 4159, 4177,
	4201, 4211, 4217, 4219, 4229, 4231, 4241, 4243, 4253, 4259,
	4261, 4271, 4273, 4283, 4289, 4297, 4327, 4337, 4339, 4349,
	4357, 4363, 4373, 4391, 4397, 4409, 4421, 4423, 4441, 4447,
	4451, 4457, 4463,
};
static int n_small_primes = sizeof(small_primes) / sizeof(int);

/*
 * A fast and simple way to check if a number is prime - it
 * shouldn't be divisible by any prime less than it's square
 * root.
 */
INTSML isprime(INTBIG n)
{
	register int i;

	if (n % 2 == 0) return 0;
	/* if we can't check it, assume it is false */
	if (n >= max_checkable) return 0;

	for(i = 0; i < n_small_primes; i++)
	{
		register long m = small_primes[i];
		if (m * m >= n) break;
		if (n % m == 0) return 0;
	}
	return 1;
}

/*
 * returns the first prime over N, by checking all odd numbers
 * over N till it finds one
 */
INTBIG pick_prime(INTBIG n)
{
	if (n >= max_checkable) return 0;
	if (n % 2 == 0) n++;
	while(!isprime(n))
	{
		n += 2;
	}
	return n;
}

#endif  /* DRCAID - at top */
