/*
 * Electric(tm) VLSI Design System
 *
 * File: graphmac.c
 * Interface for Apple Macintosh computers
 * Written by: Steven M. Rubin, Electric Editor Incorporated
 *
 * Copyright (c) 1998 Electric Editor Incorporated.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Electric Editor Incorporated
 * 23470 Sunset Drive, Suite 108
 * Los Gatos, California 95033
 * support@electriceditor.com
 */

/* #define INVERTHIGHLIGHT 1 /* */

#include "global.h"
#include "egraphics.h"
#include "usr.h"
#include "efunction.h"
#include "edialogs.h"
#include "usrdiacom.h"
#include "dblang.h"

#include <AppleEvents.h>
#include <Desk.h>
#include <Errors.h>
#include <Events.h>
#include <Files.h>
#include <Gestalt.h>
#include <LowMem.h>
#include <QDOffscreen.h>
#include <Retrace.h>
#include <Scrap.h>
#include <Script.h>
#include <SegLoad.h>
#include <StandardFile.h>
#include <time.h>
#include <ToolUtils.h>
#include <unix.h>
#ifdef	USETK
#  include "tk.h"
#  include "tkInt.h"
#  include "tkMacInt.h"
#endif

/****** window frames ******/
#define	NOWINDOWFRAME ((WINDOWFRAME *)-1)

typedef struct IWindowFrame
{
	GWorldPtr            window;				/* the offscreen buffer for the editing window */
	CGrafPtr             realwindow;			/* the real window */
#ifdef	USETK
	Tk_Window            tkwin;					/* the TK window */
#else
	void                *tkwin;					/* the TK window placeholder */
#endif
	INTSML               floating;				/* nonzero for floating palette window */
	INTSML               swid, shei;			/* screen dimensions */
	INTSML               revy;					/* for reversing Y coordinates */
	Rect                 copyrect;				/* rectangle to copy to screen */
	UINTBIG              starttime;				/* time when this window was refreshed from offscreen */
	INTSML               offscreendirty;		/* nonzero if offscreen area is "dirty" */
	INTSML               screenleft;			/* left bound of editing window screen */
	INTSML				 screenright;			/* right bound of editing window screen */
	INTSML				 screentop;				/* top bound of editing window screen */
	INTSML				 screenbottom;			/* bottom bound of editing window screen */
	char               **rowstart;				/* start of each row */
	struct IWindowFrame *nextwindowframe;		/* next in list */
} WINDOWFRAME;

static WINDOWFRAME  *gra_firstwindowframe = NOWINDOWFRAME;
static WINDOWFRAME  *gra_curwindowframe;		/* current window frame */

/****** the messages window ******/
#define	BFONT	     courier					/* font in the messages windows */

static WindowPtr     gra_messageswindow;		/* the scrolled messages window */
static INTSML        gra_messagesinfront;		/* nonzero if messages is frontmost */
static INTSML        gra_linesInFolder;			/* lines in text folder */
static INTSML        gra_messagesfont;			/* font in messages window */
static INTSML        gra_messagesfontsize;		/* size of font in messages window */
static INTSML        gra_messagesleft;			/* left bound of messages window screen */
static INTSML        gra_messagesright;			/* right bound of messages window screen */
static INTSML        gra_messagestop;			/* top bound of messages window screen */
static INTSML        gra_messagesbottom;		/* bottom bound of messages window screen */
static ControlHandle gra_vScroll;				/* vertical scroll control in messages window */
static ControlHandle gra_hScroll;				/* horizontal scroll control in messages window */
static TEHandle      gra_TEH;					/* text editing handle in messages window */
static ControlActionUPP gra_scrollvprocUPP;		/* UPP for "gra_scrollvproc" */
static ControlActionUPP gra_scrollhprocUPP;		/* UPP for "gra_scrollhproc" */
static GWorldPtr     gra_textbuf = 0;			/* temp buffer for displaying text */
static INTSML        gra_textbufwid = 0;		/* width of text temp buffer */
static INTSML        gra_textbufhei = 0;		/* height of text temp buffer */
static char        **gra_textbufrowstart;		/* row start array for text temp buffer */

/****** events ******/
#define	CHARREAD          0177					/* character that was read */
#define	ISBUTTON          0200					/* set if button pushed (or released) */
#define	BUTTONUP          0400					/* set if button was released */
#define	SHIFTDOWN        01000					/* set if shift key was held down */
#define	COMMANDDOWN      02000					/* set if command key was held down */
#define	OPTIONDOWN       04000					/* set if option key was held down */
#define	CONTROLDOWN     010000					/* set if control key was held down */
#define	DOUBLECLICK     020000					/* set if this is second click */
#define	MOTION          040000					/* set if mouse motion detected */
#define	FILEREPLY      0100000					/* set if standard file reply detected */
#define	WINDOWCHANGE   0100400					/* set if window moved and/or grown */
#define	NOEVENT             -1					/* set if nothing happened */
#define	MENUEVENT     BUTTONUP					/* set if menu entry selected (values in cursor) */

struct
{
	INTSML x;
	INTSML y;
	INTSML kind;
} gra_action;

static INTBIG        gra_inputstate;			/* current state of device input */
static INTBIG        gra_lastclick;				/* time of last click */
static INTSML        gra_lstcurx, gra_lstcury;	/* current position of mouse */
static INTSML        gra_cursorx, gra_cursory;	/* current position of mouse */

#define	EVENTQUEUESIZE	100

typedef struct
{
	INTSML           cursorx, cursory;			/* current position of mouse */
	INTBIG           inputstate;				/* current state of device input */
} MYEVENTQUEUE;

static MYEVENTQUEUE  gra_eventqueue[EVENTQUEUESIZE];
static MYEVENTQUEUE *gra_eventqueuehead;		/* points to next event in queue */
static MYEVENTQUEUE *gra_eventqueuetail;		/* points to first free event in queue */

/****** timing ******/
#define FLUSHTICKS   60							/* ticks between display flushes */
#define MOTIONCHECK  2							/* ticks between mouse motion checks */
#define INTCHECK     30							/* MOTIONCHECKs between interrupt checks */

typedef struct VBLRect
{
	VBLTask myVBLTask;
	long    vblA5;
} VBLRec;

static VBLRec        gra_vblrec;
static INTSML        gra_checkcountdown;		/* countdown to interrupt checks */
static INTSML        gra_motioncheck;			/* nonzero if mouse motion can be checked */
	   INTSML        gra_cancheck;				/* nonzero if interrupt can be checked */
unsigned long        gra_timestart;

/****** stuff for TCL/TK ******/
#ifdef	USETK
  static Tk_Window   gra_tkmessageswindow;		/* TK handle to messages window */

  void TkMacHandleMenuSelect(long mResult, int optionKeyPressed);
  void gra_messageEventProc(ClientData clientData, XEvent *eventPtr);
  Tk_Window MakeTKWindow(short type, short left, short right, short top, short bottom,
  	void (*eventproc)(ClientData, XEvent*));
  void gra_raisewindow(Tk_Window win);
  void gra_namewindow(Tk_Window win, char *name);
  void gra_clearwindowframe(WINDOWFRAME *frame);
#else
  static void       *gra_tkmessageswindow;		/* dummy TK messages window handle */
#endif

/****** pulldown menus ******/
#define	MENUSIZE              19				/* size of menu bar */
#define	MFONT	               0				/* font in menus */
#define	appleMENU            128				/* resource ID for Apple menu */
#define	USERMENUBASE         130				/* base resource ID for other menus */
#define	aboutMeCommand         1				/* menu entry for "About Electric..." item */

static INTSML        gra_lowmenu;				/* low word of selected pulldown menu */
static INTSML        gra_highmenu;				/* high word of selected pulldown menu */
static INTSML        gra_pulldownmenucount;		/* number of pulldown menus */
static INTSML        gra_tkmenusloaded = 0;		/* nonzero if TK menus are built */
static MenuHandle   *gra_pulldownmenus;			/* the current pulldown menu handles */
static POPUPMENU   **gra_pulldowns;				/* the current pulldown menus */
static MenuHandle    gra_appleMenu;				/* the Apple menu */

/****** mouse buttons ******/
#define	BUTTONS      17							/* cannot exceed NUMBUTS in "usr.h" */

typedef struct
{
	char *name;									/* button name */
	INTSML unique;								/* number of letters that make it unique */
} BUTTONNAMES;

static BUTTONNAMES   gra_buttonname[BUTTONS] =
{						/* Shift Command Option Control */
	"Button",      1,   /*                              */
	"SButton",     2,	/* Shift                        */
	"MButton",     2,	/*       Command                */
	"SMButton",    3,	/* Shift Command                */
	"OButton",     2,	/*               Option         */
	"SOButton",    3,	/* Shift         Option         */
	"MOButton",    3,	/*       Command Option         */
	"SMOButton",   4,	/* Shift Command Option         */
	"CButton",     2,	/*                      Control */
	"SCButton",    3,	/* Shift                Control */
	"CMButton",    3,	/*       Command        Control */
	"SCMButton",   4,	/* Shift Command        Control */
	"COButton",    3,	/*               Option Control */
	"SCOButton",   4,	/* Shift         Option Control */
	"CMOButton",   4,	/*       Command Option Control */
	"SCMOButton",  5,	/* Shift Command Option Control */
	"DButton",     2
};

static INTSML        gra_doubleclick;					/* interval between double clicks */

/****** cursors ******/
#define	NORMALCURSOR       0					/* a normal arrow cursor */
#define	WANTTTYCURSOR      1					/* a "use the TTY" cursor */
#define	PENCURSOR          2					/* a "draw with pen" cursor */
#define	NULLCURSOR         3					/* a null cursor */
#define	MENUCURSOR         4					/* a menu selection cursor */
#define	HANDCURSOR         5					/* a hand dragging cursor */
#define	TECHCURSOR         6					/* the technology edit cursor ("T") */
#define	IBEAMCURSOR        7					/* the text edit cursor ("I") */
enum {wantttyCURSOR = 129, penCURSOR, nullCURSOR, menulCURSOR, handCURSOR, techCURSOR};

static INTSML        gra_cursorstate;			/* see defines above */
static INTSML        gra_normalcursor;			/* default cursor to use */
static CursHandle    gra_wantttyCurs;			/* a "use the TTY" cursor */
static CursHandle    gra_penCurs;				/* a "draw with pen" cursor */
static CursHandle    gra_nullCurs;				/* a null cursor */
static CursHandle    gra_menuCurs;				/* a menu selection cursor */
static CursHandle    gra_handCurs;				/* a hand dragging cursor */
static CursHandle    gra_techCurs;				/* a technology edit cursor */
static CursHandle    gra_ibeamCurs;				/* a text edit cursor */

/****** rectangle saving ******/
#define	NOSAVEDBOX ((SAVEDBOX *)-1)

typedef struct Isavedbox
{
	char             *pix;
	WINDOW           *win;
	INTSML            lx, hx, ly, hy;
	INTSML            truelx, truehx;
	struct Isavedbox *nextsavedbox;
} SAVEDBOX;

SAVEDBOX *gra_firstsavedbox = NOSAVEDBOX;

/****** polygon decomposition ******/
#define NOPOLYSEG ((POLYSEG *)-1)

typedef struct Isegment
{
	INTBIG fx,fy, tx,ty, direction, increment, index;
	struct Isegment *nextedge;
	struct Isegment *nextactive;
} POLYSEG;

static POLYSEG      *gra_polysegs;
static INTSML        gra_polysegcount = 0;

/****** curve drawing ******/
#define MODM(x) ((x<1) ? x+8 : x)
#define MODP(x) ((x>8) ? x-8 : x)
#define ODD(x)  (x&1)

static INTSML        gra_arcocttable[] = {0,0,0,0,0,0,0,0,0};
static INTBIG        gra_arccenterx, gra_arccentery, gra_arcradius, gra_curvemaxx, gra_curvemaxy;
static INTBIG        gra_arclx, gra_archx, gra_arcly, gra_archy;
static INTSML        gra_curvecol, gra_curvemask, gra_curvestyle, gra_arcfirst;
static WINDOWFRAME  *gra_arcframe;				/* window frame for arc drawing */
static void        (*gra_arcpixeldraw)(INTBIG x, INTBIG y);
static void        (*gra_discrowdraw)(WINDOWFRAME *frame, INTBIG thisy, INTBIG startx, INTBIG endx,
						GRAPHICS *desc);

/****** dialogs ******/
#define	aboutMeDLOG          128				/* resource ID for "About Electric" alert dialog */
#define noscreenALERT        129				/* resource ID for "no valid screen" alert dialog */
#define errorALERT           130				/* resource ID for "error" alert dialog */
#define	FILEINDIALOG         136				/* resource ID for "file input" dialog */
#define	NAMERSRC            2000				/* resource ID for user name */
#define	COMPANYRSRC         2001				/* resource ID for company name */
#define	SPECIALRSRC         2002				/* resource ID for special instructions */
#define	CHECKRSRC           2003				/* resource ID for checksum */

static DIALOGITEM gra_fontdialogitems[] =
{
 /*  1 */ {0, {136,192,160,256}, BUTTON, "OK"},
 /*  2 */ {0, {88,192,112,256}, BUTTON, "Cancel"},
 /*  3 */ {0, {8,24,24,56}, MESSAGE, "Font"},
 /*  4 */ {0, {24,200,40,232}, MESSAGE, "Size"},
 /*  5 */ {0, {32,24,160,184}, SCROLL, ""},
 /*  6 */ {0, {48,192,64,248}, EDITTEXT, ""}
};
static DIALOG gra_fontdialog = {{50,75,219,338}, "Messages Window Font", 6, gra_fontdialogitems};

static FileFilterYDUPP gra_fileFilterProcUPP;

pascal Boolean gra_fileFilterProc(CInfoPBPtr pb, Ptr mydata);

/****** network control ******/
/* #define CHECKNETWORK 1 */

#ifdef CHECKNETWORK
# define LOOKUPBUFSIZE   2000					/* for duplicate name checking */
  static MPPParamBlock gra_mppb;				/* for duplicate name checking */
  static short         gra_nameregistered = 0;	/* for duplicate name checking */
#endif

/****** miscellaneous ******/
#define	EFONT	            helvetica			/* font in the editing window */
#define	TFONT               courier				/* font in text editing */
#define	SFONT	            monaco				/* font for status at the bottom of editing windows */
#define	SBARWIDTH           15					/* width of scroll bars */
#ifdef USETK
#  define	PALETTEWIDTH     0					/* width of palette */
#else
#  define	PALETTEWIDTH    80					/* width of palette */
#endif
#define	FLOATINGHEADERSIZE  10					/* size of tool palette drag bar */
#define MAXSTATUSLINES       1					/* lines in status bar */

static INTSML        gra_curfontsize;			/* size of current font */
static FontInfo      gra_curfontinfo;			/* current font information */
static INTSML        gra_screenleft;			/* left bound of any editor window screen */
static INTSML        gra_screenright;			/* right bound of any editor window screen */
static INTSML        gra_screentop;				/* top bound of any editor window screen */
static INTSML        gra_screenbottom;			/* bottom bound of any editor window screen */
static GDHandle      gra_origgdevh;				/* the original graphics device */
static char          gra_localstring[256];		/* local string */
static INTSML        gra_logrecordindex = 0;	/* count for session flushing */
static INTSML        gra_playbackmultiple = 0;	/* count for multi-key playback */
static INTSML        gra_multiwindow;			/* nonzero if using multiple windows */
static void         *gra_fileliststringarray = 0;

/****** prototypes for local routines ******/
void         gra_addeventtoqueue(INTBIG state, INTSML x, INTSML y);
INTSML       gra_addfiletolist(char *file);
void         gra_adjusthtext(INTSML);
void         gra_adjustvtext(void);
void         gra_applemenu(INTSML);
void         gra_arcbresccw(INTBIG x, INTBIG y, INTBIG x1, INTBIG y1);
void         gra_arcbrescw(INTBIG x, INTBIG y, INTBIG x1, INTBIG y1);
void         gra_arcbresmidccw(INTBIG x, INTBIG y);
void         gra_arcbresmidcw(INTBIG x, INTBIG y);
void         gra_arcdopixel(INTBIG x, INTBIG y);
INTSML       gra_arcfindoctant(INTBIG x, INTBIG y);
void         gra_arcoutxform(INTBIG x, INTBIG y);
void         gra_arcxformoctant(INTBIG x, INTBIG y, INTBIG oct, INTBIG *ox, INTBIG *oy);
void         gra_backup(void);
INTSML       gra_buildwindow(WINDOWFRAME*, INTSML);
void         gra_centermessage(char*, INTSML, INTSML, INTSML);
void         gra_drawdiscrow(WINDOWFRAME *frame, INTBIG thisy, INTBIG startx, INTBIG endx, GRAPHICS *desc);
void         gra_drawline(WINDOWFRAME*, INTSML, INTSML, INTSML, INTSML, INTSML, INTSML);
void         gra_drawpatline(WINDOWFRAME*, INTSML, INTSML, INTSML, INTSML, INTSML, INTSML, INTSML);
INTSML       gra_enternetwork(char*, char*);
void         gra_fakemouseup(void);
void         gra_getdevices(void);
Rect        *gra_geteditorwindowlocation(void);
CGrafPtr     gra_getoffscreen(WINDOW *win);
CGrafPtr     gra_getwindow(WINDOW *win);
pascal OSErr gra_handleodoc(AppleEvent*, AppleEvent*, long);
pascal OSErr gra_handlequit(AppleEvent*, AppleEvent*, long);
void         gra_hidemessageswindow(void);
void         gra_initfilelist(void);
void         gra_initializemenus(void);
void         gra_installvbl(void);
UINTBIG      gra_machinetimeoffset(void);
INTSML       gra_makebutton(INTBIG);
INTSML       gra_makeeditwindow(WINDOWFRAME*);
char        *gra_makefullnamefromFSS(FSSpec theFSS);
MenuHandle   gra_makepdmenu(POPUPMENU*, INTSML);
void         gra_mygrowwindow(WindowPtr, INTSML, INTSML, Rect*);
void         gra_nativemenudoone(INTSML low, INTSML high);
void         gra_nextevent(INTSML, EventRecord*);
INTSML       gra_pulldownindex(POPUPMENU*);
void         gra_rebuildrowstart(WINDOWFRAME*);
void         gra_recachepixmap(PixPatHandle);
void         gra_redrawdisplay(WINDOWFRAME*);
void         gra_reloadmap(void);
INTSML       gra_remakeeditwindow(WINDOWFRAME*);
void         gra_savewindowsettings(void);
pascal void  gra_scrollhproc(ControlHandle, INTSML);
pascal void  gra_scrollvproc(ControlHandle, INTSML);
void         gra_setcurrentwindowframe(void);
void         gra_setrect(WINDOWFRAME*, INTSML, INTSML, INTSML, INTSML);
void         gra_setuptextbuffer(INTSML width, INTSML height);
void         gra_setview(WindowPtr);
void         gra_setvscroll(void);
INTSML       gra_showmessageswindow(void);
void         gra_showselect(void);
void         gra_waitforaction(INTSML, EventRecord*);

#ifndef USETK
  void       gra_activatewindow(WindowPtr, INTSML);
  void       gra_disposeoswindow(WindowPtr, void*);
  void       gra_dragfloatingwindow(WindowPtr, Point);
  void       gra_drawosgraphics(WINDOWFRAME*);
  void       gra_drawpalettedragbar(WINDOWFRAME *mw);
  void       gra_frontnonfloatingwindow(WindowPtr *);
  WindowPtr  gra_lastfloatingwindow(void);
  void       gra_selectoswindow(WindowPtr);
  void       gra_showallfloaters(INTSML);
#endif
#ifdef	INVERTHIGHLIGHT
  void       gra_arcinvpixel(INTBIG x, INTBIG y);
  void       gra_invdiscrow(WINDOWFRAME *frame, INTBIG thisy, INTBIG startx, INTBIG endx, GRAPHICS *desc);
  void       gra_invertcircle(WINDOW *win, INTBIG atx, INTBIG aty, INTBIG radius);
#endif

/******************** INITIALIZATION ********************/

/*
 * routine to establish the default display
 */
void us_setdisplay(char *name, INTBIG *argc, char **argv)
{
	/* set multiple window factor */
	if (*argc > 0 && argv[0][strlen(argv[0])-1] == '1') gra_multiwindow = 0; else
		gra_multiwindow = 1;
}

/*
 * routine to initialize the display device.  Creates status window if "messages" is nonzero.
 */
INTSML us_initgraphics(INTSML messages)
{
	void *mw;
	char fontname[100];
	Rect rstat, r;
	INTSML err;
	long response;
	Handle f;
	CTabHandle iclut;
	REGISTER INTSML i;
	RgnHandle grayrgn;

	/* get double-click interval */
	gra_doubleclick = GetDblTime();
	gra_lastclick = 0;
	gra_lstcurx = -1;

	/* setup UPPs */
	gra_scrollvprocUPP = NewControlActionProc(gra_scrollvproc);
	gra_scrollhprocUPP = NewControlActionProc(gra_scrollhproc);
	gra_fileFilterProcUPP = NewFileFilterYDProc(gra_fileFilterProc);

	/* initialize menus */
	gra_initializemenus();

	/* get cursors */
	gra_wantttyCurs = GetCursor(wantttyCURSOR);
	gra_penCurs = GetCursor(penCURSOR);
	gra_nullCurs = GetCursor(nullCURSOR);
	gra_menuCurs = GetCursor(menulCURSOR);
	gra_handCurs = GetCursor(handCURSOR);
	gra_techCurs = GetCursor(techCURSOR);
	gra_ibeamCurs = GetCursor(iBeamCursor);
	gra_normalcursor = NORMALCURSOR;

	/* setup globals that describe location of windows */
	gra_getdevices();

	/* determine font and size of text in messages window */
	f = GetResource('MPSR', 1004);
	if (f == 0)
	{
		/* no window resource: use defaults */
		gra_messagesfont = BFONT;
		gra_messagesfontsize = 12;
	} else
	{
		HLock(f);

		/* get messages window extent */
		rstat.left = ((INTSML *)(*f))[0];
		rstat.right = ((INTSML *)(*f))[1];
		rstat.top = ((INTSML *)(*f))[2];
		rstat.bottom = ((INTSML *)(*f))[3];

		/* is the top of the messages window visible on this display? */
		r.left = rstat.left;   r.right = rstat.right;
		r.top = rstat.top;     r.bottom = rstat.top + MENUSIZE;
		grayrgn = GetGrayRgn();
		if (RectInRgn(&r, grayrgn))
		{
			gra_messagesleft = rstat.left;   gra_messagesright = rstat.right;
			gra_messagestop = rstat.top;     gra_messagesbottom = rstat.bottom;
		}

		/* get font and size */
		gra_messagesfontsize = ((INTSML *)(*f))[8];
		(void)strcpy(&fontname[1], &((char *)(*f))[18]);
		fontname[0] = strlen(&fontname[1]);
		GetFNum((unsigned char *)fontname, &gra_messagesfont);
		HUnlock(f);
	}

	/* create the scrolling messages window */
	gra_messageswindow = 0;
	gra_TEH = 0;
	if (messages != 0)
	{
		if (gra_showmessageswindow() != 0) error("Cannot create messages window");
	}

	/* initialize the mouse */
	if (gra_nullCurs != 0L) SetCursor(&(**gra_nullCurs)); else
		SetCursor(&qd.arrow);
	gra_cursorstate = NULLCURSOR;
	gra_inputstate = NOEVENT;
	gra_eventqueuehead = gra_eventqueuetail = gra_eventqueue;

	/* create the CLUT for identity mapping on color displays */
	iclut = (CTabHandle)NewHandle(2056);
	if (iclut == 0) error("Cannot allocate identity color lookup table");
	(*iclut)->ctSeed = GetCTSeed();
	(*iclut)->ctFlags = 0;
	(*iclut)->ctSize = 255;
	for(i=0; i<256; i++)
	{
		(*iclut)->ctTable[i].value = i;
		(*iclut)->ctTable[i].rgb.red = i<<8;
		(*iclut)->ctTable[i].rgb.green = i<<8;
		(*iclut)->ctTable[i].rgb.blue = i<<8;
	}

	/* setup to handle input files via apple events */
	err = Gestalt(gestaltAppleEventsAttr, &response);
	if (err == 0 && ((response >> gestaltAppleEventsPresent) & 1) != 0)
	{
		/* apple events method */
		err = AEInstallEventHandler(kCoreEventClass, kAEOpenDocuments,
			NewAEEventHandlerProc(gra_handleodoc), 0, 0);
		err = AEInstallEventHandler(kCoreEventClass, kAEQuitApplication,
			NewAEEventHandlerProc(gra_handlequit), 0, 0);
	}

	/* initialize vertical retrace manager to check for interrupts */
	gra_installvbl();

	/* create the first window frame */
	if (gra_multiwindow == 0)
	{
		mw = (WINDOWFRAME *)emalloc((sizeof (WINDOWFRAME)), us_aid->cluster);
		if (mw == 0) error("Cannot create Macintosh window info structure");
		gra_firstwindowframe = gra_curwindowframe = mw;
		gra_firstwindowframe->nextwindowframe = NOWINDOWFRAME;

		/* load an editor window into this frame */
		if (gra_buildwindow(gra_curwindowframe, 0) != 0)
			error("Cannot create Macintosh window info structure");
	}

	return(0);
}

/*
 * Routine to establish the library directories from the environment.
 */
void setupenvironment(void)
{
	/* initialize system directories for library files */
	(void)initinfstr();
	(void)addstringtoinfstr(mac_systemfoldername());
	(void)addstringtoinfstr(LIBDIR);
	(void)allocstring(&el_libdir, returninfstr(), db_cluster);

	(void)initinfstr();
	(void)addstringtoinfstr(mac_systemfoldername());
	(void)addstringtoinfstr(LISPDIR);
	(void)allocstring(&el_lispdir, returninfstr(), db_cluster);

#if LANGTCL
	(void)initinfstr();
	(void)addstringtoinfstr(mac_systemfoldername());
	(void)addstringtoinfstr(TCLDIR);
	(void)allocstring(&el_tcldir, returninfstr(), db_cluster);
	if (el_tcldir[strlen(el_tcldir)-1] == ':') el_tcldir[strlen(el_tcldir)-1] = 0;

# ifdef USETK
	(void)initinfstr();
	(void)addstringtoinfstr(mac_systemfoldername());
	(void)addstringtoinfstr(TKDIR);
	(void)allocstring(&el_tkdir, returninfstr(), db_cluster);
	if (el_tkdir[strlen(el_tkdir)-1] == ':') el_tkdir[strlen(el_tkdir)-1] = 0;
# endif
#endif

	/* set machine name */
	nextvarchangequiet();
	(void)setvalkey((INTBIG)us_aid, VAID, makekey("USER_machine"),
		(INTBIG)"Macintosh", VSTRING|VDONTSAVE);
}

/*
 * Routine to examine the display devices available and to setup globals that describe
 * the editing windows and messages window extents.  On exit, the globals "gra_screenleft",
 * "gra_screenright", "gra_screentop", and "gra_screenbottom" will describe the area
 * for the editing windows and the variables "gra_messagesleft", "gra_messagesright",
 * "gra_messagestop", and "gra_messagesbottom" will describe the messages window.
 */
void gra_getdevices(void)
{
	GDHandle gdevh, bestcgdevh, bestbwgdevh, curgdevh;
	INTSML totaldevices, iscolor, anycolor;
	INTBIG bestcpixels, bestbwpixels, pixels, ys;

	/* obtain the current device */
	gra_origgdevh = GetGDevice();

	/* find the devices */
	bestcpixels = bestbwpixels = 0L;
	totaldevices = anycolor = 0;
	for(gdevh = GetDeviceList(); gdevh != 0L; gdevh = GetNextDevice(gdevh))
	{
		totaldevices++;

		/* determine size of this display */
		pixels = (*gdevh)->gdRect.right - (*gdevh)->gdRect.left;
		ys = (*gdevh)->gdRect.bottom - (*gdevh)->gdRect.top;
		pixels *= ys;

		/* The low bit is set if the display is color */
		iscolor = 1;
		if (((*gdevh)->gdFlags&1) == 0) iscolor = 0;

		/* color displays must be at least 8 bits deep */
		if ((*(*gdevh)->gdPMap)->pixelSize < 8) iscolor = 0;

		/* accumulate the best of each display type */
		if (iscolor != 0)
		{
			if (pixels > bestcpixels || (pixels == bestcpixels && gdevh == gra_origgdevh))
			{
				bestcgdevh = gdevh;
				bestcpixels = pixels;
			}
		} else
		{
			if (pixels > bestbwpixels || (pixels == bestbwpixels && gdevh == gra_origgdevh))
			{
				bestbwgdevh = gdevh;
				bestbwpixels = pixels;
			}
		}
	}

	/* if there is a color device, choose it */
	if (bestcpixels != 0) curgdevh = bestcgdevh; else
		if (bestbwpixels != 0) curgdevh = bestbwgdevh; else
	{
		StopAlert(noscreenALERT, 0L);
		exitprogram();
	}

	/* set the extent of the editing windows */
	gra_screenleft   = (*curgdevh)->gdRect.left;
	gra_screenright  = (*curgdevh)->gdRect.right;
	gra_screentop    = (*curgdevh)->gdRect.top;
	gra_screenbottom = (*curgdevh)->gdRect.bottom;

	/* set the extent of the messages window */
	gra_messagesleft   = gra_screenleft + PALETTEWIDTH + 1;
	gra_messagesright  = gra_screenright - PALETTEWIDTH - 1;
	gra_messagestop    = gra_screentop + MENUSIZE*3 + 2 +
		(gra_screenbottom-(gra_screentop+MENUSIZE*2)) * 3 / 5;
	gra_messagesbottom = gra_screenbottom;

	/* if multiple displays exist, choose another for the messages window */
	if (totaldevices > 1)
	{
		/* look for another screen to be used for the messages display */
		for(gdevh = GetDeviceList(); gdevh != 0L; gdevh = GetNextDevice(gdevh))
			if (gdevh != curgdevh)
		{
			INTSML i, siz;
			gra_messagesleft = (*gdevh)->gdRect.left;
			gra_messagesright = (*gdevh)->gdRect.right;
			gra_messagestop = (*gdevh)->gdRect.top;
			gra_messagesbottom = (*gdevh)->gdRect.bottom;

			i = (gra_messagesleft + gra_messagesright) / 2;
			siz = (gra_messagesright - gra_messagesleft) * 2 / 5;
			gra_messagesleft = i - siz;     gra_messagesright = i + siz;
			i = (gra_messagestop + gra_messagesbottom) / 2;
			siz = (gra_messagesbottom - gra_messagestop) / 4;
			gra_messagestop = i - siz;      gra_messagesbottom = i + siz;
			return;
		}
	}
}

/*
 * this routine returns the host ID for this computer in "name"
 */
void getmachineid(char *name)
{
	INTSML checksum, i;
	Handle uname, company, special, check;

	/* make sure user name hasn't been tampered */
	uname = GetResource('STR ', NAMERSRC);
	company = GetResource('STR ', COMPANYRSRC);
	special = GetResource('STR ', SPECIALRSRC);
	check = GetResource('STR ', CHECKRSRC);

	/* compute the checksum */
	checksum = 5001;
	if (uname != 0) for(i=0; i<(*uname)[0]; i++) checksum += (*uname)[i+1]*7;
	if (company != 0) for(i=0; i<(*company)[0]; i++) checksum += (*company)[i+1]*7;
	if (special != 0) for(i=0; i<(*special)[0]; i++) checksum += (*special)[i+1]*7;
	checksum = checksum & 0x7FFF;
	if (check == 0) (void)strcpy(name, "0"); else
	{
		for(i=0; i<(*check)[0]; i++) name[i] = (*check)[i+1];
		name[i] = 0;
	}

	/* trap tampering */
	i = myatoi(name);
	if (i != checksum) error("Illegal tampering with Electric!");

#ifdef CHECKNETWORK
	/* make sure no other user is running this copy of Electric */
	i = gra_enternetwork(name, "ElectricCAD");
	if (i < 0) ttyputerr("Error checking the network");
	if (i > 0) error("This copy of Electric is already being used on another machine");
#endif
}

#ifdef CHECKNETWORK
/*
 * routine to determine whether the program "ourname", of type "type" is the only
 * one running on the network.  Returns zero if so, positive if there are other
 * copies already running, and negative on error.
 */
INTSML gra_enternetwork(char *ourname, char *type)
{
	short mpp, index, err, len;
	Handle buffer;
	NamesTableEntry *NTPtr;

	/* open the AppleTalk driver */
	err = OpenDriver("\p.MPP", &mpp);
	if (err != 0) return(-1);

	/* make the name of this program */
	NTPtr = (NamesTableEntry *)NewPtrClear(sizeof(NamesTableEntry));
	if (NTPtr == 0) return(-1);
	index = 0;
	(NTPtr->nt.entityData[index]) = len = strlen(ourname);   index++;
	(void)strcpy((char *)&(NTPtr->nt.entityData[index]), ourname);   index += len;
	(NTPtr->nt.entityData[index]) = len = strlen(type);      index++;
	(void)strcpy((char *)&(NTPtr->nt.entityData[index]), type);      index += len;
	(NTPtr->nt.entityData[index]) = 1;                       index++;
	(void)strcpy((char *)&(NTPtr->nt.entityData[index]), "*");

	/* look for others with this name */
	gra_mppb.MPPioCompletion = 0L;
	gra_mppb.MPPioRefNum = mpp;
	buffer = NewHandle(LOOKUPBUFSIZE);
	if (buffer == 0) return(-1);
	HLock(buffer);
	gra_mppb.NBPinterval = 5;
	gra_mppb.NBPcount = 4;
	gra_mppb.NBPentityPtr = (Ptr)&(NTPtr->nt.entityData[0]);
	gra_mppb.NBPretBuffPtr = *buffer;
	gra_mppb.NBPretBuffSize = LOOKUPBUFSIZE;
	gra_mppb.NBPmaxToGet = LOOKUPBUFSIZE / 110;
	gra_mppb.NBPnumGotten = 0;
	err = PLookupName(&gra_mppb, false);
	DisposHandle(buffer);
	if (err != 0) return(-1);
	err = gra_mppb.MPPioResult;
	if (err != 0) return(-1);
	if (gra_mppb.NBPnumGotten > 0) return(gra_mppb.NBPnumGotten);

	/* register our name on the network */
	gra_mppb.NBPinterval = 3;
	gra_mppb.NBPcount = 3;
	gra_mppb.NBPverifyFlag = true;
	gra_mppb.NBPntQElPtr = (Ptr)NTPtr;
	err = PRegisterName(&gra_mppb, true);
	if (err != 0) return(-1);
	while (gra_mppb.MPPioResult == 1) ;
	err = gra_mppb.MPPioResult;
	/* if (err == nbpDuplicate) SysBeep(40); */
	gra_mppb.NBPntQElPtr = (Ptr)&(NTPtr->nt.entityData[0]);
	gra_nameregistered++;
	return(0);
}
#endif

/******************** TERMINATION ********************/

void us_unsetdisplay(void) {}

void us_termgraphics(INTSML final)
{
	if (final != 0) return;
#ifdef USETK
	while (gra_firstwindowframe != NOWINDOWFRAME)
		(void)Tk_DestroyWindow(gra_firstwindowframe->tkwin);
	if (gra_messageswindow != 0) (void)Tk_DestroyWindow(gra_tkmessageswindow);
#else
	while (gra_firstwindowframe != NOWINDOWFRAME)
		gra_disposeoswindow((WindowPtr)gra_firstwindowframe->realwindow, gra_firstwindowframe->tkwin);
	if (gra_messageswindow != 0)
		gra_disposeoswindow(gra_messageswindow, gra_tkmessageswindow);
#endif
}

/*
 * routine to handle a "quit" Apple Event
 */
pascal OSErr gra_handlequit(AppleEvent *theAppleEvent, AppleEvent *reply, long handlerRefCon)
{
	char *par[MAXPARS];

	if (us_preventloss(0, par, NOLIBRARY, "Quit")) return(noErr);
	bringdown();
	return(noErr);
}

/*
 * routine to quit the program.  It unregisters the name first and then exits.
 */
void exitprogram(void)
{
#ifdef CHECKNETWORK
	if (gra_nameregistered != 0) (void)PRemoveName(&gra_mppb, false);
#endif
	ExitToShell();
}

/******************** WINDOW CONTROL ********************/

/*
 * Routine to create a new window frame (floating if "floating" nonzero).
 */
void *us_newwindowframe(INTSML floating)
{
	WINDOWFRAME *mw, *oldlisthead;

	if (gra_multiwindow == 0) return(0);

#ifdef USETK
	if (floating != 0) return(0);
#endif

	/* allocate one */
	mw = (WINDOWFRAME *)emalloc((sizeof (WINDOWFRAME)), us_aid->cluster);
	if (mw == 0) return(0);

	/* insert window-frame in linked list */
	oldlisthead = gra_firstwindowframe;
	mw->nextwindowframe = gra_firstwindowframe;
	gra_firstwindowframe = mw;

	/* load an editor window into this frame */
	if (gra_buildwindow(mw, floating) != 0)
	{
		efree((char *)mw);
		gra_firstwindowframe = oldlisthead;
		return(0);
	}

	/* remember that this is the current window frame */
	if (floating == 0) gra_curwindowframe = mw;

	return((void *)mw);
}

void us_killwindowframe(void *frame)
{
	WINDOWFRAME *mw;

	if (gra_multiwindow != 0)
	{
		/* find this frame in the list */
		mw = (WINDOWFRAME *)frame;

		/* kill the actual window, that will remove the frame, too */
#ifdef USETK
		(void)Tk_DestroyWindow(mw->tkwin);
#else
		gra_disposeoswindow((WindowPtr)mw->realwindow, mw->tkwin);
#endif
	}
}

/*
 * Routine to return the current window frame.
 */
void *us_getwindowframe(void)
{
	if (gra_curwindowframe == NOWINDOWFRAME) return(0);
	return((void *)gra_curwindowframe);
}

/*
 * Routine to return the first in the list of window frames.
 */
void *us_firstwindowframe(void)
{
	return((void *)gra_firstwindowframe);
}

/*
 * Routine to return the next in the list of window frames (after "frame").
 */
void *us_nextwindowframe(void *frame)
{
	WINDOWFRAME *mw;

	mw = (WINDOWFRAME *)frame;
	if (mw == 0 || mw == NOWINDOWFRAME) return(0);
	if (mw->nextwindowframe == NOWINDOWFRAME) return(0);
	return((void *)mw->nextwindowframe);
}

/*
 * routine to return size of window "win" in "wid" and "hei"
 */
void us_getwindowsize(void *frame, INTSML *wid, INTSML *hei)
{
	WINDOWFRAME *mw;

	mw = (WINDOWFRAME *)frame;
	*wid = mw->swid;
	*hei = mw->shei;
}

/*
 * Routine to return the current Macintosh window associated with Electric window "win".
 * This is only called from "ioplotmac.c"
 */
CGrafPtr gra_getwindow(WINDOW *win)
{
	WINDOWFRAME *mw;

	mw = (WINDOWFRAME *)win->frame;
	return(mw->realwindow);
}

/*
 * Routine to return the offscreen Macintosh buffer associated with Electric window "win".
 * This is only called from "ioplotmac.c"
 */
CGrafPtr gra_getoffscreen(WINDOW *win)
{
	WINDOWFRAME *mw;

	mw = (WINDOWFRAME *)win->frame;
	return(mw->window);
}

void us_sizewindowframe(void *frame, INTSML wid, INTSML hei)
{
	WINDOWFRAME *mw;
	Rect         fr;

	mw = (WINDOWFRAME *)frame;
	fr = (*((WindowPeek)mw->realwindow)->strucRgn)->rgnBBox;

	/* determine new window size */
	wid++;
	hei += 2;
	if (mw != NOWINDOWFRAME && mw->floating != 0) hei += FLOATINGHEADERSIZE;
	if (wid+2 == fr.right-fr.left && hei+2 == fr.bottom-fr.top) return;

	/* resize the window */
#ifdef	USETK
	(void)Tk_ResizeWindow(mw->tkwin, wid, hei);
#else
	SizeWindow((WindowRef)mw->realwindow, wid, hei, 1);

	/* rebuild the offscreen windows */
	if (gra_remakeeditwindow(mw) > 0)
	{
		fr.left += 1;         fr.right -= 2;
		fr.top += MENUSIZE;   fr.bottom -= 2;
		SizeWindow((WindowRef)mw->realwindow, fr.right-fr.left, fr.bottom-fr.top, 1);
		return;
	}
	gra_reloadmap();

	SetPort((WindowPtr)mw->realwindow);
	EraseRect(&mw->realwindow->portRect);
	gra_drawosgraphics(mw);
#endif
}

void us_movewindowframe(void *frame, INTSML top, INTSML left)
{
	WINDOWFRAME *mw;
	INTSML        oleft, otop;
	Rect         fr;

	mw = (WINDOWFRAME *)frame;
	fr = (*((WindowPeek)mw->realwindow)->strucRgn)->rgnBBox;

	/* determine new window location */
	top += MENUSIZE+1;
	if (left == fr.left && top == fr.top) return;
	oleft = fr.left;   otop = fr.top;

#ifdef	USETK
	(void)Tk_MoveWindow(mw->tkwin, left, top);
#else
	MoveWindow((WindowRef)mw->realwindow, left, top, 0);
#endif
}

/*
 * routine to grow window "w" to the new size of "wid" by "hei".  If the grow fails,
 * reset the window to its former size in "formerrect".
 */
void gra_mygrowwindow(WindowPtr w, INTSML wid, INTSML hei, Rect *formerrect)
{
	REGISTER WINDOWFRAME *mw;
	Rect r;
	REGISTER INTSML ret, top;

	if (w == gra_messageswindow && gra_messageswindow != 0)
	{
		SetPort(w);
		InvalRect(&w->portRect);

		gra_setview(w);
		HidePen();
		top = w->portRect.top;
		MoveControl(gra_vScroll, w->portRect.right-SBARWIDTH, top);
		SizeControl(gra_vScroll, SBARWIDTH+1, w->portRect.bottom-top-(SBARWIDTH-2));
		MoveControl(gra_hScroll, w->portRect.left-1, w->portRect.bottom-SBARWIDTH);
		SizeControl(gra_hScroll, w->portRect.right-w->portRect.left-(SBARWIDTH-2), SBARWIDTH+1);
		ShowPen();

		gra_setvscroll();
		gra_adjustvtext();

		r = (*((WindowPeek)w)->strucRgn)->rgnBBox;
		gra_messagesleft = r.left;  gra_messagesright = r.right;
		gra_messagestop = r.top;    gra_messagesbottom = r.bottom;

		/* remember window settings */
		gra_savewindowsettings();
		return;
	}

	/* find the window */
	for(mw = gra_firstwindowframe; mw != NOWINDOWFRAME; mw = mw->nextwindowframe)
		if (w == (WindowPtr)mw->realwindow) break;
	if (mw == NOWINDOWFRAME) return;

	/* rebuild the window pointers */
	ret = gra_remakeeditwindow(mw);

	/* no change necessary */
	if (ret < 0) return;

	/* change failed: move it back */
	if (ret > 0)
	{
#ifndef USETK
		formerrect->left += 1;         formerrect->right -= 2;
		formerrect->top += MENUSIZE;   formerrect->bottom -= 2;
		MoveWindow((WindowPtr)mw->realwindow, formerrect->left, formerrect->top, 0);
		SizeWindow((WindowPtr)mw->realwindow, formerrect->right-formerrect->left,
			formerrect->bottom-formerrect->top, 1);
#endif
		return;
	}

	/* change succeeded, redraw the window */
	gra_reloadmap();
	SetPort((WindowPtr)mw->realwindow);
	EraseRect(&mw->realwindow->portRect);
#ifdef	USETK
	DrawGrowIcon((WindowPtr)mw->realwindow);
#else
	gra_drawosgraphics(mw);
#endif
	gra_redrawdisplay(mw);
}

/*
 * Routine to close the messages window if it is in front.  Returns nonzero if the
 * window was closed.
 */
INTSML us_closefrontmostmessages(void)
{
	if (gra_messagesinfront != 0)
	{
		gra_hidemessageswindow();
		return(1);
	}
	return(0);
}

/*
 * routine to return the important pieces of information in placing the floating
 * palette with the fixed menu.  The maximum screen size is placed in "wid" and
 * "hei", and the amount of space that is being left for this palette on the
 * side of the screen is placed in "palettewidth".
 */
void us_getpaletteparameters(INTSML *wid, INTSML *hei, INTSML *palettewidth)
{
	*hei = gra_screenbottom - (gra_screentop+MENUSIZE) - FLOATINGHEADERSIZE;
	*wid = gra_screenright - gra_screenleft;
	*palettewidth = PALETTEWIDTH;
}

Rect *gra_geteditorwindowlocation(void)
{
	static Rect redit;
	static INTSML offset = 0;

	if (gra_multiwindow != 0)
	{
		redit.top = offset + gra_screentop + MENUSIZE;
		redit.bottom = redit.top + (gra_screenbottom-(gra_screentop+MENUSIZE)) * 3 / 4;
		redit.left = offset + gra_screenleft + PALETTEWIDTH + 1;
		redit.right = redit.left + (gra_screenright-gra_screenleft-PALETTEWIDTH-1) * 4 / 5;

		/* is the editing window visible on this display? */
		if (redit.bottom > gra_screenbottom || redit.right > gra_screenright)
		{
			offset = 0;
			redit.top = gra_screentop + MENUSIZE;
			redit.bottom = redit.top + (gra_screenbottom-(gra_screentop+MENUSIZE)) * 3 / 4;
			redit.left = gra_screenleft + PALETTEWIDTH + 1;
			redit.right = redit.left + (gra_screenright-gra_screenleft-PALETTEWIDTH-1) * 4 / 5;
		}
		offset += 30;
	} else
	{
		redit.top = gra_screentop + MENUSIZE;
		redit.bottom = gra_screenbottom;
		redit.left = gra_screenleft;
		redit.right = gra_screenright;
	}
#ifndef USETK
	redit.top += MENUSIZE;
#endif
	return(&redit);
}

/*
 * routine to build a new window frame
 */
INTSML gra_buildwindow(WINDOWFRAME *mw, INTSML floating)
{
	Rect          redit;
#ifndef	USETK
	WindowPtr     frontmost, wBehind;
#endif
	static INTSML  first = 1;
	WStateData   *wst;
	INTSML         cy, ty, left, right, top, bottom;
	char          line[200];
	Handle        name, company, special;

	mw->floating = floating;
	if (floating == 0)
	{
		/* get editor window location */
		redit = *gra_geteditorwindowlocation();

#ifdef	USETK
		/* create the editing window */
		mw->tkwin = MakeTKWindow(2, redit.left, redit.right, redit.top, redit.bottom,
			gra_messageEventProc);
		if (mw->tkwin == NULL) return(1);
		mw->realwindow = (CGrafPtr)TkMacGetDrawablePort(Tk_WindowId(mw->tkwin));
		if (gra_tkmenusloaded != 0)
		{
			if (Tcl_VarEval(tcl_interp, Tk_PathName(mw->tkwin), " configure -menu .menus",
				0) == TCL_ERROR) ttyputerr("Tcl_VarEval returned %s", tcl_interp->result);
		}
#else
		/* deactivate any other nonfloating window that is current */
		gra_frontnonfloatingwindow(&frontmost);
		if (frontmost != 0) gra_activatewindow(frontmost, 0);

		/* figure out which window to put this behind */
		wBehind = gra_lastfloatingwindow();
		if (wBehind == 0) wBehind = (WindowPtr)-1;

		/* create the editing window */
		mw->realwindow = (CGrafPtr)NewCWindow(0L, &redit, "\p", 0, zoomDocProc, wBehind, 1, 0L);
#endif
	} else
	{
		/* default palette window location */
		redit.top = gra_screentop + MENUSIZE + 2;
		redit.bottom = (gra_screentop+gra_screenbottom)/2;
		redit.left = gra_screenleft + 1;
		redit.right = redit.left + PALETTEWIDTH/2;

		/* create the editing window */
#ifdef	USETK
		mw->tkwin = MakeTKWindow(2, redit.left, redit.right, redit.top, redit.bottom,
			gra_messageEventProc);
		if (mw->tkwin == NULL) return(1);
		mw->realwindow = (CGrafPtr)TkMacGetDrawablePort(Tk_WindowId(mw->tkwin));
		if (gra_tkmenusloaded != 0)
		{
			if (Tcl_VarEval(tcl_interp, Tk_PathName(mw->tkwin), " configure -menu .menus",
				0) == TCL_ERROR) ttyputerr("Tcl_VarEval returned %s", tcl_interp->result);
		}
#else
		mw->realwindow = (CGrafPtr)NewCWindow(0L, &redit, "\p", 0, plainDBox, (WindowRef)(-1L),
			1, 0L);
#endif
	}
	if (mw->realwindow == 0) return(1);
#ifdef	USETK
	SetPort((WindowPtr)mw->realwindow);
	EraseRect(&mw->realwindow->portRect);
	DrawGrowIcon((WindowPtr)mw->realwindow);
#else
	ShowHide((WindowRef)mw->realwindow, 1);
	gra_frontnonfloatingwindow(&frontmost);
	gra_activatewindow((WindowRef)mw->realwindow, 0);
	gra_activatewindow(frontmost, 1);
	SetPort((WindowPtr)mw->realwindow);
	EraseRect(&mw->realwindow->portRect);
	gra_drawosgraphics(mw);
#endif
	if (floating == 0)
	{
		if (first != 0)
		{
			/* get user name/company/special message */
			name = GetResource('STR ', NAMERSRC);
			company = GetResource('STR ', COMPANYRSRC);
			special = GetResource('STR ', SPECIALRSRC);

			(void)strcpy(line, " Version ");
			(void)strcat(line, el_version);
			line[0] = strlen(&line[1]);

			/* display the welcome message */
			TextFont(0);
			TextSize(12);
			left = mw->realwindow->portRect.left;
			right = mw->realwindow->portRect.right;
			top = mw->realwindow->portRect.top;
			bottom = mw->realwindow->portRect.bottom;
			cy = (top+bottom) / 2;
			ty = (bottom-top) / 5;
			gra_centermessage((char *)"\pElectric Design System", left, right, ty);
			gra_centermessage(line, left, right, ty+15);
			gra_centermessage((char *)"\pProprietary Software from Electric Editor Incorporated",
				left, right, ty+40);
			if (name == 0)
				gra_centermessage((char *)"\p***** UNKNOWN USER *****", left, right, cy); else
			{
				HLock(name);
				gra_centermessage(*name, left, right, cy);
				HUnlock(name);
			}
			if (company == 0)
				gra_centermessage((char *)"\p***** UNKNOWN LOCATION *****", left, right, cy+20); else
			{
				HLock(company);
				gra_centermessage(*company, left, right, cy+20);
				HUnlock(company);
			}
			if (special != 0)
			{
				HLock(special);
				gra_centermessage(*special, left, right, cy+40);
				HUnlock(special);
			}

			gra_centermessage((char *)"\pPlease wait for loading...", left, right, bottom-30);
		}
		wst = (WStateData *) *(((WindowPeek)mw->realwindow)->dataHandle);
		wst->stdState.top = gra_screentop + MENUSIZE*2;
		wst->stdState.bottom = gra_screenbottom;
		wst->stdState.left = gra_screenleft + PALETTEWIDTH;
		wst->stdState.right = gra_screenright;
	}

	/* build the offscreen buffer for the window */
	if (gra_makeeditwindow(mw) != 0)
	{
#ifdef	USETK
		(void)Tk_DestroyWindow(mw->tkwin);
#else
		DisposeWindow((WindowPtr)mw->realwindow);
#endif
		return(1);
	}

	/* load any map the first time to establish the map segment length */
	el_maplength = 256;
	SetGWorld(mw->realwindow, gra_origgdevh);
	if (floating == 0 && first != 0) us_getcolormap(el_curtech, COLORSDEFAULT, 0); else
		gra_reloadmap();
	first = 0;

	/* want the editing window to be the primary one */
	SetPort((WindowPtr)mw->realwindow);

	return(0);
}

void gra_centermessage(char *msg, INTSML left, INTSML right, INTSML y)
{
	INTSML wid;

	wid = StringWidth((unsigned char *)msg);
	MoveTo((left+right-wid)/2, y);
	DrawString((unsigned char *)msg);
}

/*
 * Routine to redraw editing window "mw" due to a drag or grow
 */
void gra_redrawdisplay(WINDOWFRAME *mw)
{
#ifndef	USETK
	us_beginchanges();
#endif
	if (mw == NOWINDOWFRAME || mw->floating != 0) us_drawmenu(0, (void *)mw); else
		us_drawmenu(-1, (void *)mw);
	us_redostatus((void *)mw);
#ifndef	USETK
	us_endchanges(NOWINDOW);
#endif
	us_state |= HIGHLIGHTSET;
	us_showallhighlight();
}

/*
 * Routine to allocate the offscreen buffer that corresponds with the window
 * "mw->realwindow".  The buffer is 8-bits deep.
 * Returns nonzero if memory cannot be allocated.
 */
INTSML gra_makeeditwindow(WINDOWFRAME *mw)
{
	INTSML height, bytes, i;
	char *addr;
	Rect r;

	r = mw->realwindow->portRect;
	if (mw->floating == 0) r.bottom -= SBARWIDTH; else
		r.bottom -= FLOATINGHEADERSIZE;
	if (NewGWorld(&mw->window, 8, &r, 0L, 0L, 0) != noErr)
	{
		ParamText("\pCannot create the offscreen window", "\p", "\p", "\p");
		Alert(errorALERT, 0L);
		return(1);
	}

	SetGWorld(mw->window, 0L);
	(void)LockPixels(mw->window->portPixMap);
	PenMode(patCopy);
	BackColor(0);
	EraseRect(&mw->window->portRect);
	UnlockPixels(mw->window->portPixMap);

	height = r.bottom - r.top;
	mw->rowstart = (char **)emalloc(height * SIZEOFINTBIG, us_aid->cluster);
	if (mw->rowstart == 0)
	{
		DisposeGWorld(mw->window);
		SetGWorld(mw->realwindow, gra_origgdevh);
		ParamText("\pCannot create the offscreen pointers", "\p", "\p", "\p");
		Alert(errorALERT, 0L);
		return(1);
	}

	bytes = (*mw->window->portPixMap)->rowBytes & 0x7FFF;
	addr = GetPixBaseAddr(mw->window->portPixMap);
	for(i=0; i<height; i++)
	{
		mw->rowstart[i] = addr;
		addr += bytes;
	}
	SetGWorld(mw->realwindow, gra_origgdevh);

	mw->swid = mw->window->portRect.right - mw->window->portRect.left;
	mw->shei = mw->window->portRect.bottom - mw->window->portRect.top;
	mw->revy = mw->shei - 1;
	mw->offscreendirty = 0;
	return(0);
}

/*
 * Routine to rebuild the offscreen buffer when its size or depth has changed.
 * Returns -1 if no change is necessary, 0 if the change was successful, and
 * 1 if the change failed (memory error).
 */
INTSML gra_remakeeditwindow(WINDOWFRAME *mw)
{
	INTSML res, bytes, i, height;
	char *addr, **newrowstart;
	Rect r;

	/* quit now if there was no change */
	r = mw->realwindow->portRect;
	if (mw->floating == 0) r.bottom -= SBARWIDTH; else
		r.bottom -= FLOATINGHEADERSIZE;
	height = r.bottom - r.top;
	if (mw->swid == r.right - r.left && mw->shei == height) return(-1);

	/* reallocate rowstart array */
	newrowstart = (char **)emalloc(height * SIZEOFINTBIG, us_aid->cluster);
	if (newrowstart == 0)
	{
		ParamText("\pNot enough memory to modify the window", "\p", "\p", "\p");
		Alert(errorALERT, 0L);
		return(1);
	}

	res = UpdateGWorld(&mw->window, 8, &r, 0L, 0L, 0);
	if (res != noErr)
	{
		efree((char *)newrowstart);
		ParamText("\pNot enough memory to modify the window", "\p", "\p", "\p");
		Alert(errorALERT, 0L);
		return(1);
	}

	/* this next line should not be needed (fixed in 7.0) */
	RectRgn(mw->window->clipRgn, &mw->window->portRect);

	/* clear the new buffer */
	SetGWorld(mw->window, 0L);
	(void)LockPixels(mw->window->portPixMap);
	PenMode(patCopy);
	BackColor(0);
	EraseRect(&mw->window->portRect);
	UnlockPixels(mw->window->portPixMap);
	SetGWorld(mw->realwindow, gra_origgdevh);

	/* load new row start array */
	bytes = (*mw->window->portPixMap)->rowBytes & 0x7FFF;
	addr = GetPixBaseAddr(mw->window->portPixMap);
	for(i=0; i<height; i++)
	{
		newrowstart[i] = addr;
		addr += bytes;
	}
	efree((char *)mw->rowstart);
	mw->rowstart = newrowstart;

	mw->swid = mw->window->portRect.right - mw->window->portRect.left;
	mw->shei = mw->window->portRect.bottom - mw->window->portRect.top;
	mw->revy = mw->shei - 1;
	mw->offscreendirty = 0;
	return(0);
}

/*
 * routine to rebuild the array "mw->rowstart".  Since it points to the
 * image buffer, and since that buffer is a handle, the buffer can be moved by
 * the system for no good reason.  Before using the array, each
 * routine should see if it is valid and call this to repair the array.
 */
void gra_rebuildrowstart(WINDOWFRAME *mw)
{
	INTSML height, bytes, i;
	char *addr;
	Rect r;

	/* the array has moved: recompute it */
	addr = GetPixBaseAddr(mw->window->portPixMap);
	r = mw->realwindow->portRect;
	if (mw->floating == 0) r.bottom -= SBARWIDTH; else
		r.bottom -= FLOATINGHEADERSIZE;
	height = r.bottom - r.top;
	bytes = (*mw->window->portPixMap)->rowBytes & 0x7FFF;
	for(i=0; i<height; i++)
	{
		mw->rowstart[i] = addr;
		addr += bytes;
	}
}

#ifndef USETK
void gra_drawosgraphics(WINDOWFRAME *mw)
{
	WindowPtr win;

	/* figure out which window is being drawn */
	win = (WindowPtr)mw->realwindow;
	if (mw == NOWINDOWFRAME)
	{
		if (gra_messageswindow == 0) return;
		win = gra_messageswindow;
	}

	if (mw == NOWINDOWFRAME || mw->floating == 0)
	{
		/* for all but menu, draw the grow icon */
		DrawGrowIcon(win);
	}

	if (mw != NOWINDOWFRAME && mw->floating != 0)
	{
		/* for floating windows, draw the small drag bar */
		gra_drawpalettedragbar(mw);
	}
}

void gra_drawpalettedragbar(WINDOWFRAME *mw)
{
	Rect r;
	Pattern thePat;
	WindowPtr win, frontWindow;

	win = (WindowPtr)mw->realwindow;
	if (mw == NOWINDOWFRAME)
	{
		if (gra_messageswindow == 0) return;
		win = gra_messageswindow;
	}

	SetPort(win);
	r = win->portRect;
	r.bottom = FLOATINGHEADERSIZE;
	EraseRect(&r);
	MoveTo(0, FLOATINGHEADERSIZE);
	LineTo(r.right, FLOATINGHEADERSIZE);
	gra_frontnonfloatingwindow(&frontWindow);
	if (win != gra_messageswindow || win == frontWindow)
	{
		/* draw the small drag bar */
		r = win->portRect;
		r.bottom = FLOATINGHEADERSIZE;
		GetIndPattern(&thePat, 0, 24);
		PenPat(&thePat);
		PaintRect(&r);
		PenPat(&qd.black);

		/* now draw the close box */
		r.top = 2;
		r.bottom = FLOATINGHEADERSIZE-1;
		r.left = 4;
		r.right = FLOATINGHEADERSIZE+1;
		PenMode(patBic);
		PaintRect(&r);
		PenMode(patCopy);
		FrameRect(&r);
	}
}
#endif

/******************** FLOATING WINDOW ROUTINES ********************/

#ifndef USETK
/*
 * Routine to return the frontmost window that is NOT floating (zero if none).
 */
void gra_frontnonfloatingwindow(WindowPtr *result)
{
	WindowPeek   theWindow, firstWindow;
	WINDOWFRAME *mw;

	firstWindow = (WindowPeek)LMGetWindowList();
	for(theWindow = firstWindow; theWindow != 0; theWindow = theWindow->nextWindow)
	{
		/* see if this window is in the edit-window list */
		for(mw = gra_firstwindowframe; mw != NOWINDOWFRAME; mw = mw->nextwindowframe)
			if (mw->realwindow == (CGrafPtr)theWindow) break;

		/* if not in the list, it must be messages or a dialog, and therefore valid */
		if (mw == NOWINDOWFRAME) break;

		/* accept editor window only if it is nonfloating */
		if (mw->floating == 0) break;
	}

	/* return what was found */
	*result = (WindowPtr)theWindow;
}

/*
 * Routine to make all floating windows visible/invisible.
 */
void gra_showallfloaters(INTSML vis)
{
	WINDOWFRAME *mw;

	for(mw = gra_firstwindowframe; mw != NOWINDOWFRAME; mw = mw->nextwindowframe)
	{
		if (mw->floating == 0) continue;
		if (vis) ShowHide((WindowPtr)mw->realwindow, 1); else
			ShowHide((WindowPtr)mw->realwindow, 0);
	}
}

/*
 * Routine to return the address of the last floating window.  All normal
 * windows must go behind this.
 */
WindowPtr gra_lastfloatingwindow(void)
{
	WindowPeek   theWindow, firstWindow;
	WindowPtr    last;
	WINDOWFRAME *mw;

	last = 0;
	firstWindow = (WindowPeek)LMGetWindowList();
	for(theWindow = firstWindow; theWindow != 0; theWindow = theWindow->nextWindow)
	{
		/* see if this window is in the list */
		for(mw = gra_firstwindowframe; mw != NOWINDOWFRAME; mw = mw->nextwindowframe)
			if (mw->realwindow == (CGrafPtr)theWindow) break;
		if (mw == NOWINDOWFRAME) continue;

		/* make sure it is floating */
		if (mw->floating == 0) continue;

		/* floating window found: save its address */
		last = (WindowPtr)mw->realwindow;
	}
	return(last);
}

void gra_dragfloatingwindow(WindowPtr windowToDrag, Point startPoint)
{
	Rect        dragRect, r;
	GrafPtr     savePort, windowManagerPort;
	RgnHandle   dragRegion;
	INTBIG       dragResult;
	INTSML       topLimit, newHorizontalWindowPosition, newVerticalWindowPosition,
				horizontalOffset, verticalOffset;
	WINDOWFRAME *mw;

	/* see if this is an editor window */
	for(mw = gra_firstwindowframe; mw != NOWINDOWFRAME; mw = mw->nextwindowframe)
		if ((WindowPtr)mw->realwindow == windowToDrag) break;

	/* adjust the top of the dragging rectangle so that its below the menu bar */
	dragRect = qd.screenBits.bounds;
	topLimit = GetMBarHeight();
	if (dragRect.top < topLimit) dragRect.top = topLimit;
	InsetRect(&dragRect, 4, 4);

	/* Set up the Window Manager port */
	GetPort(&savePort);
	GetWMgrPort(&windowManagerPort);
	SetPort(windowManagerPort);
	SetClip(GetGrayRgn());

	/* Create a region to drag */
	dragRegion = NewRgn();

	r = (*((WindowPeek)windowToDrag)->strucRgn)->rgnBBox;
	RectRgn(dragRegion, &r);

	/* Drag the window around */
	dragResult = DragGrayRgn(dragRegion, startPoint, &dragRect, &dragRect, noConstraint, nil);

	/* Restore the port for coordinate conversion */
	SetPort(savePort);

	if (dragResult != 0)
	{
		horizontalOffset = dragResult & 0xFFFF;
		verticalOffset = dragResult >> 16;

		/* Only move it if it stayed inside the dragging box */
		if (verticalOffset != -32768)
		{
			r = (*((WindowPeek)windowToDrag)->strucRgn)->rgnBBox;
			newHorizontalWindowPosition = r.left + horizontalOffset + 1;
			newVerticalWindowPosition = r.top + verticalOffset;
			if (mw == NOWINDOWFRAME || mw->floating == 0)
				newVerticalWindowPosition += MENUSIZE;
			MoveWindow(windowToDrag, newHorizontalWindowPosition, newVerticalWindowPosition, false);
		}
	}

	/* Get rid of the dragging region */
	DisposeRgn(dragRegion);

	/* if this is the messages window, adjust the bits */
	if (mw == NOWINDOWFRAME)
	{
		r = (*((WindowPeek)windowToDrag)->strucRgn)->rgnBBox;
		gra_messagesleft = r.left;   gra_messagesright = r.right;
		gra_messagestop = r.top;     gra_messagesbottom = r.bottom;
	}
}

void gra_selectoswindow(WindowPtr windowToSelect)
{
	WindowPtr    currentFrontWindow;
	WINDOWFRAME *mw;
	WindowPtr    lastFloatingWindow;

	/* handle floating windows specially */
	for(mw = gra_firstwindowframe; mw != NOWINDOWFRAME; mw = mw->nextwindowframe)
		if (mw->realwindow == (CGrafPtr)windowToSelect)
	{
		if (mw->floating != 0)
		{
			BringToFront(windowToSelect);
			return;
		}
		break;
	}

	/* quit now if this is already selected */
	gra_frontnonfloatingwindow(&currentFrontWindow);
	if (currentFrontWindow == windowToSelect) return;

	/* deactivate current frontmost window */
	if (currentFrontWindow != 0) gra_activatewindow(currentFrontWindow, 0);

	/* bring this window to the front */
	lastFloatingWindow = gra_lastfloatingwindow();
	if (lastFloatingWindow == 0) BringToFront(windowToSelect); else
		SendBehind(windowToSelect, lastFloatingWindow);

	/* activate this one */
	gra_activatewindow(windowToSelect, 1);
}

/*
 * Routine to highlight and activate a window.
 */
void gra_activatewindow(WindowPtr win, INTSML active)
{
	HiliteWindow(win, active);
	if (win == gra_messageswindow && gra_messageswindow != 0)
	{
		if (active != 0)
		{
			ShowControl(gra_vScroll);
			ShowControl(gra_hScroll);
			gra_messagesinfront = 1;
		} else
		{
			HideControl(gra_vScroll);
			HideControl(gra_hScroll);
			gra_messagesinfront = 0;
		}
	}
}

/*
 * Routine to close a window.
 */
void gra_disposeoswindow(WindowPtr win, void *tkwin)
{
	WINDOWFRAME *mw, *lastmw;
	WindowPtr front;

	/* deactivate, make invisible, and dispose */
	gra_activatewindow(win, 0);
	ShowHide(win, 0);
	DisposeWindow(win);

	/* activate the next one (if there is) */
	gra_frontnonfloatingwindow(&front);
	if (front != 0) gra_activatewindow(front, 1);

	/* if this is an editor window, remove the structure */
	lastmw = NOWINDOWFRAME;
	for(mw = gra_firstwindowframe; mw != NOWINDOWFRAME; mw = mw->nextwindowframe)
	{
		if (mw->realwindow == (CGrafPtr)win)
		{
			if (lastmw == NOWINDOWFRAME) gra_firstwindowframe = mw->nextwindowframe; else
				lastmw->nextwindowframe = mw->nextwindowframe;
			DisposeGWorld(mw->window);
			efree((char *)mw);
			break;
		}
		lastmw = mw;
	}

	/* determine current window frame */
	gra_setcurrentwindowframe();
}
#endif

/*
 * routine to determine the current window frame and load the global "gra_curwindowframe".
 */
void gra_setcurrentwindowframe(void)
{
	WindowPeek   theWindow, firstWindow;
	WINDOWFRAME *mw;
	WINDOW *w;

	gra_curwindowframe = NOWINDOWFRAME;
	firstWindow = (WindowPeek)LMGetWindowList();
	for(theWindow = firstWindow; theWindow != 0; theWindow = theWindow->nextWindow)
	{
		/* see if this window is in the edit-window list */
		for(mw = gra_firstwindowframe; mw != NOWINDOWFRAME; mw = mw->nextwindowframe)
			if (mw->realwindow == (CGrafPtr)theWindow) break;

#ifdef USETK
		/* if not in the list, it must be a dialog */
		if (mw == NOWINDOWFRAME) continue;
#else
		/* if not in the list, it must be a dialog */
		if (mw == NOWINDOWFRAME)
		{
			gra_curwindowframe = mw;
			break;
		}
#endif

		/* ignore floating windows */
		if (mw->floating != 0) continue;

		gra_curwindowframe = mw;

		/* see if the change of window frame invalidates the current window */
		if (el_curwindow == NOWINDOW || el_curwindow->frame != mw)
		{
			/* must choose new window */
			for(w = el_topwindow; w != NOWINDOW; w = w->nextwindow)
			{
				if (w->frame == mw)
				{
					(void)setvalkey((INTBIG)us_aid, VAID, us_current_window, (INTBIG)w,
						VWINDOW|VDONTSAVE);
					(void)setval((INTBIG)el_curlib, VLIBRARY, "curnodeproto",
						(INTBIG)w->curnodeproto, VNODEPROTO);
					break;
				}
			}
		}
		break;
	}
}

void gra_savewindowsettings(void)
{
	Handle f;
	INTSML i;
	char line[256];

	/* get the current font name and messages window location */
	GetFontName(gra_messagesfont, (unsigned char *)line);

	/* remove any previous resource */
	f = GetResource('MPSR', 1004);
	if (f != 0) RmveResource(f);

	/* make a handle for this information */
	f = NewHandle(line[0]+19);

	/* load the window location information */
	((INTSML *)(*f))[0] = gra_messagesleft;
	((INTSML *)(*f))[1] = gra_messagesright;
	((INTSML *)(*f))[2] = gra_messagestop;
	((INTSML *)(*f))[3] = gra_messagesbottom;
	((INTSML *)(*f))[4] = 0;
	((INTSML *)(*f))[5] = 0;
	((INTSML *)(*f))[6] = 0;
	((INTSML *)(*f))[7] = 0;

	/* load the font information */
	((INTSML *)(*f))[8] = gra_messagesfontsize;
	for(i=0; i<line[0]; i++) (*f)[i+18] = line[i+1];
	(*f)[line[0]+18] = 0;

	/* write the resource */
	AddResource(f, 'MPSR', 1004, "\pWindow");
	f = GetResource('MPSR', 1004);
	if (f != 0) WriteResource(f);
}

/******************** EXTERNAL ROUTINES ********************/

/*
 * routine to return display style of window "win"
 */
short us_getdispstyle(WINDOW *win)
{
	return(COLORMAP);
}

void us_startbatching(void) {}

void us_endbatching(void)
{
	flushscreen();
}

void us_setspy(INTSML mag) {}

/*
 * return nonzero if the capabilities in "want" are present
 */
INTSML us_graphicshas(INTSML want)
{
	/* cannot do "spyglass" magnification or native popups */
	if ((want&(CANMAGNIFY|CANSHOWPOPUP)) != 0) return(0);
	if (gra_multiwindow == 0)
	{
		if ((want&CANUSEFRAMES) != 0) return(0);
	}

	return(1);
}

/*
 * internal routine to make a beep
 */
void ttybeep(void)
{
	SysBeep(20);
}

/******************** TIMING ROUTINES ********************/

#if defined(powerc) || defined(__powerc)
static void gra_dovbl(VBLRec *recPtr)
{
	long curA5;

	curA5 = SetA5(recPtr->vblA5);

	/* do the task */
	gra_motioncheck = 1;
	if (--gra_checkcountdown <= 0)
	{
		gra_checkcountdown = INTCHECK;
		gra_cancheck = 1;
	}

	/* reset counter and global pointers */
	recPtr->myVBLTask.vblCount = MOTIONCHECK;
	curA5 = SetA5(curA5);
}
#else
static pascal long GetVBLRec(void) = 0x2E88;

static void gra_dovbl(void)
{
	long curA5;
	VBLRec *recPtr;

	recPtr = (VBLRec *)GetVBLRec();
	curA5 = SetA5(recPtr->vblA5);

	/* do the task */
	gra_motioncheck = 1;
	if (--gra_checkcountdown <= 0)
	{
		gra_checkcountdown = INTCHECK;
		gra_cancheck = 1;
	}

	/* reset counter and global pointers */
	recPtr->myVBLTask.vblCount = MOTIONCHECK;
	curA5 = SetA5(curA5);
}
#endif

void gra_installvbl(void)
{
	OSErr err;

	gra_vblrec.myVBLTask.qType = vType;
	gra_vblrec.myVBLTask.vblAddr = NewVBLProc(gra_dovbl);
	gra_vblrec.myVBLTask.vblCount = MOTIONCHECK;
	gra_vblrec.myVBLTask.vblPhase = 0;
	gra_vblrec.vblA5 = SetCurrentA5();
	err = VInstall((QElemPtr)&gra_vblrec.myVBLTask);
	if (err == noErr) gra_cancheck = 1;
	gra_cancheck = 0;
	gra_motioncheck = 0;
	gra_checkcountdown = INTCHECK;
}

/******************** MESSAGES WINDOW ROUTINES ********************/

/*
 * Routine to delete the messages window.
 */
void gra_hidemessageswindow(void)
{
	if (gra_messageswindow == 0) return;
#ifdef USETK
	(void)Tk_DestroyWindow(gra_tkmessageswindow);
#else
	gra_disposeoswindow(gra_messageswindow, gra_tkmessageswindow);
#endif
	gra_messagesinfront = 0;
	gra_messageswindow = 0;
}

/*
 * Routine to create the messages window.
 */
INTSML gra_showmessageswindow(void)
{
	Rect r;

	if (gra_messageswindow == 0)
	{
#ifdef	USETK
		gra_tkmessageswindow = MakeTKWindow(1, gra_messagesleft, gra_messagesright-3,
			gra_messagestop, gra_messagesbottom-21, gra_messageEventProc);
		if (gra_tkmessageswindow == NULL) return(1);
		gra_messageswindow = (WindowPtr)TkMacGetDrawablePort(Tk_WindowId(gra_tkmessageswindow));
		gra_namewindow(gra_tkmessageswindow, "Electric Messages");
		if (gra_tkmenusloaded != 0)
		{
			if (Tcl_VarEval(tcl_interp, Tk_PathName(gra_tkmessageswindow),
				" configure -menu .menus", 0) == TCL_ERROR)
					ttyputerr("Tcl_VarEval returned %s", tcl_interp->result);
		}
#else
		r.left = gra_messagesleft;   r.right = gra_messagesright;
		r.top = gra_messagestop;     r.bottom = gra_messagesbottom;
		gra_messageswindow = (WindowPtr)NewWindow(0L, &r, "\pElectric Messages", 1, documentProc,
			(WindowRef)(-1L), 1, 0L);
		if (gra_messageswindow == 0) return(1);
#endif
		SetPort(gra_messageswindow);
		EraseRect(&gra_messageswindow->portRect);
		TextFont(gra_messagesfont);
		TextSize(gra_messagesfontsize);
		r = gra_messageswindow->portRect;
		r.left = r.right - SBARWIDTH;
		r.right++;
		r.bottom -= 14;
		r.top += 1;
		gra_vScroll = NewControl((WindowRef)gra_messageswindow, &r, "\p", 1, 0, 0, 0,
			scrollBarProc, 0L);
		if (gra_vScroll == 0) return(1);
		r = gra_messageswindow->portRect;
		r.top = r.bottom - SBARWIDTH;
		r.bottom++;
		r.right -= 14;
		r.left--;
		gra_hScroll = NewControl((WindowRef)gra_messageswindow, &r, "\p", 1, 0, 0, 0,
			scrollBarProc, 0L);
		if (gra_hScroll == 0) return(1);
		HideControl(gra_vScroll);
		HideControl(gra_hScroll);
		DrawGrowIcon((WindowRef)gra_messageswindow);
		SetCtlMin(gra_hScroll, 0);
		SetCtlMax(gra_hScroll, 90);
	}
	if (gra_TEH == 0)
	{
		r = gra_messageswindow->portRect;
		gra_TEH = TENew(&r, &r);
		if (gra_TEH == 0) return(1);
	}
	gra_setview(gra_messageswindow);
	TEActivate(gra_TEH);
	return(0);
}

void gra_setview(WindowPtr w)
{
	INTSML width;

	(*gra_TEH)->viewRect = w->portRect;
	(*gra_TEH)->viewRect.right -= SBARWIDTH;
	(*gra_TEH)->viewRect.bottom -= SBARWIDTH;
	width = (*gra_TEH)->viewRect.right - (*gra_TEH)->viewRect.left;
	InsetRect(&(*gra_TEH)->viewRect, 4, 4);
	(*gra_TEH)->destRect = (*gra_TEH)->viewRect;
	gra_linesInFolder = ((*gra_TEH)->destRect.bottom - (*gra_TEH)->destRect.top) /
		(*gra_TEH)->lineHeight;
	(*gra_TEH)->destRect.bottom = (*gra_TEH)->destRect.top + (*gra_TEH)->lineHeight *
		gra_linesInFolder;
	(*gra_TEH)->destRect.left += 4;
	(*gra_TEH)->destRect.right = (*gra_TEH)->destRect.left + width*10;
	TECalText(gra_TEH);
}

pascal void gra_scrollvproc(ControlHandle theControl, INTSML theCode)
{
	INTSML pageSize, scrollAmt;

	if (theCode == 0) return;
	pageSize = ((*gra_TEH)->viewRect.bottom-(*gra_TEH)->viewRect.top) / (*gra_TEH)->lineHeight - 1;
	switch (theCode)
	{
		case inUpButton:   scrollAmt = -1;          break;
		case inDownButton: scrollAmt = 1;           break;
		case inPageUp:     scrollAmt = -pageSize;   break;
		case inPageDown:   scrollAmt = pageSize;    break;
	}
	SetCtlValue(theControl, GetCtlValue(theControl)+scrollAmt);
	gra_adjustvtext();
}

pascal void gra_scrollhproc(ControlHandle theControl, INTSML theCode)
{
	INTSML scrollAmt, pos, oldpos;

	if (theCode == 0) return;
	switch (theCode)
	{
		case inUpButton:   scrollAmt = -1;    break;
		case inDownButton: scrollAmt = 1;     break;
		case inPageUp:     scrollAmt = -10;   break;
		case inPageDown:   scrollAmt = 10;    break;
	}
	oldpos = GetCtlValue(theControl);
	pos = oldpos + scrollAmt;
	if (pos < 0) pos = 0;
	if (pos > 90) pos = 90;
	SetCtlValue(theControl, pos);
	gra_adjusthtext(oldpos);
}

void gra_adjustvtext(void)
{
	INTSML oldScroll, newScroll, delta;

	oldScroll = (*gra_TEH)->viewRect.top - (*gra_TEH)->destRect.top;
	newScroll = GetCtlValue(gra_vScroll) * (*gra_TEH)->lineHeight;
	delta = oldScroll - newScroll;
	if (delta != 0) TEScroll(0, delta, gra_TEH);
}

void gra_adjusthtext(INTSML oldpos)
{
	INTSML pos, wid, delta;

	pos = GetCtlValue(gra_hScroll);
	wid = ((*gra_TEH)->viewRect.right - (*gra_TEH)->viewRect.left) / 10;
	delta = wid * (pos - oldpos);
	if (delta != 0) TEScroll(-delta, 0, gra_TEH);
}

void gra_setvscroll(void)
{
	INTSML n;

	n = (*gra_TEH)->nLines - gra_linesInFolder + 1;
	if ((*gra_TEH)->teLength > 0 && (*((*gra_TEH)->hText))[(*gra_TEH)->teLength-1] != '\r') n++;
	SetCtlMax(gra_vScroll, n > 0 ? n : 0);
}

void gra_showselect(void)
{
	INTSML topLine, bottomLine, theLine;

	gra_setvscroll();
	gra_adjustvtext();

	topLine = GetCtlValue(gra_vScroll);
	bottomLine = topLine + gra_linesInFolder;

	if ((*gra_TEH)->selStart < (*gra_TEH)->lineStarts[topLine] ||
		(*gra_TEH)->selStart >= (*gra_TEH)->lineStarts[bottomLine])
	{
		for (theLine = 0; (*gra_TEH)->selStart >= (*gra_TEH)->lineStarts[theLine]; theLine++) ;
		SetCtlValue(gra_vScroll, theLine - gra_linesInFolder / 2);
		gra_adjustvtext();
	}
}

/*
 * routine to cut text from the messages window if it is current.  Returns nonzero
 * if sucessful.
 */
INTSML tty_cut(void)
{
	/* if a desk accessory wants the "cut" command, stop now */
	if (SystemEdit(2) != 0) return(1);

	if (gra_messagesinfront != 0)
	{
		TECut(gra_TEH);
		ZeroScrap();
		TEToScrap();
		return(1);
	}
	return(0);
}

/*
 * routine to copy text from the messages window if it is current.  Returns nonzero
 * if sucessful.
 */
INTSML tty_copy(void)
{
	/* if a desk accessory wants the "copy" command, stop now */
	if (SystemEdit(3) != 0) return(1);

	if (gra_messagesinfront != 0)
	{
		TECopy(gra_TEH);
		ZeroScrap();
		TEToScrap();
		return(1);
	}
	return(0);
}

/*
 * routine to paste text to the messages window if it is current.  Returns nonzero
 * if sucessful.
 */
INTSML tty_paste(void)
{
	/* if a desk accessory wants the "paste" command, stop now */
	if (SystemEdit(4) != 0) return(1);

	if (gra_messagesinfront != 0)
	{
		TEFromScrap();
		TEPaste(gra_TEH);
		return(1);
	}
	return(0);
}

/*
 * routine to get the contents of the system cut buffer
 */
char *tty_getcutbuffer(void)
{
	Handle sc;
	INTBIG scoffset, len, i;

	/* get cut buffer */
	sc = NewHandle(0);
	len = GetScrap(sc, 'TEXT', (long *)&scoffset);
	(void)initinfstr();
	for(i=0; i<len; i++) (void)addtoinfstr((*sc)[i]);
	return(returninfstr());
}

/*
 * routine to set the contents of the system cut buffer to "msg"
 */
void tty_setcutbuffer(char *msg)
{
	ZeroScrap();
	PutScrap(strlen(msg), 'TEXT', msg);
}

/*
 * routine to select fonts in the messages window
 * Font list = 5 (user item)
 * Size      = 6 (edit text)
 */
void gra_setfont(void)
{
	INTSML itemHit, i, tot, id, which, fontnum;
	INTBIG typ, fonttype;
	Handle f;
	char line[256];
	FontInfo finfo;
	extern TEHandle tty_listteh;

	/* display the font dialog box */
	DiaInitDialog(&gra_fontdialog);
	DiaInitTextDialog(5, tty_nulldloglist, tty_nulldlogitem, tty_nulldlogdone, -1,
		SCSELMOUSE|SCDOUBLEQUIT);

	/* if there are 'FONT's, then no names will be obtained by "GetResInfo" below!!! */
	fonttype = 'FOND';
	tot = CountResources(fonttype);
	if (tot == 0)
	{
		fonttype = 'FONT';
		tot = CountResources(fonttype);
	}
	which = fontnum = 0;
	for(i=1; i<=tot; i++)
	{
		SetResLoad(0);
		f = GetIndResource(fonttype, i);
		GetResInfo(f, &id, (ResType *)&typ, (unsigned char *)line);
		SetResLoad(1);
		if (line[0] == 0) continue;
		GetFNum((unsigned char *)line, &id);
		if (id == gra_messagesfont) which = fontnum;
		line[line[0]+1] = 0;
		DiaStuffLine(5, &line[1]);
		fontnum++;
	}
	DiaSelectLine(5, which);
	(void)sprintf(line, "%d", gra_messagesfontsize);
	DiaSetText(-6, line);

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == OK || itemHit == CANCEL) break;
	}

	if (itemHit != CANCEL)
	{
		(void)strcpy(&line[1], DiaGetScrollLine(5, DiaGetCurLine(5)));
		line[0] = strlen(&line[1]);
		GetFNum((unsigned char *)line, &id);
		i = myatoi(DiaGetText(6));
		if (i != gra_messagesfontsize || id != gra_messagesfont)
		{
			if (gra_messageswindow == 0)
			{
				if (gra_showmessageswindow() != 0) return;
			}

			gra_messagesfontsize = i;
			gra_messagesfont = id;
			SetPort(gra_messageswindow);
			TextSize(i);
			TextFont(id);
			GetFontInfo(&finfo);
			(*gra_TEH)->txSize = i;
			(*gra_TEH)->txFont = id;
			(*gra_TEH)->fontAscent = finfo.ascent;
			(*gra_TEH)->lineHeight = finfo.ascent + finfo.descent + finfo.leading;
			gra_setview(gra_messageswindow);
			EraseRect(&gra_messageswindow->portRect);
			DrawControls(gra_messageswindow);
			DrawGrowIcon(gra_messageswindow);
			TEUpdate(&gra_messageswindow->portRect, gra_TEH);

			/* save new messages window settings */
			gra_savewindowsettings();
		}
	}
	DiaDoneDialog();
}

/*
 * Routine to put the string "s" into the scrolling messages window.  Called from
 * "terminal.c"
 */
void gra_putstring(char *s)
{
	INTBIG len, insert;
	Rect r;
	GrafPtr savePort;
	WINDOWFRAME *mw;

	len = strlen(s);
	GetPort(&savePort);
	if (gra_messageswindow == 0)
	{
		if (gra_showmessageswindow() != 0) return;
	}

	/* see if the messages window occludes any edit window */
	for(mw = gra_firstwindowframe; mw != NOWINDOWFRAME; mw = mw->nextwindowframe)
	{
		if (SectRect(&(*((WindowPeek)gra_messageswindow)->strucRgn)->rgnBBox,
			&(*((WindowPeek)mw->realwindow)->strucRgn)->rgnBBox, &r) != 0)
		{
#ifdef	USETK
			gra_raisewindow(gra_tkmessageswindow);
#else
			gra_selectoswindow(gra_messageswindow);
#endif
			break;
		}
	}
	if ((*gra_TEH)->teLength > 30000)
	{
		/* cut out the top half of the buffer before it overflows */
		TESetSelect(0, 15000, gra_TEH);
		TEKey(010, gra_TEH);
		insert = 32767;
		TESetSelect(insert, insert, gra_TEH);
	}
	TEInsert(s, len, gra_TEH);
	TEKey('\r', gra_TEH);
	insert = 32767;
	TESetSelect(insert, insert, gra_TEH);
	gra_showselect();
	SetPort(savePort);
}

/*
 * Routine to get a string from the scrolling messages window.  Returns zero if end-of-file
 * (^D) is typed.
 */
char *gra_getstring(char *prompt)
{
	EventRecord theEvent;
	INTSML start, end, i, j, ch;
	static char outline[256];

	if (gra_messageswindow == 0)
	{
		if (gra_showmessageswindow() != 0) return("");
	}
#ifdef	USETK
	gra_raisewindow(gra_tkmessageswindow);
#else
	gra_selectoswindow(gra_messageswindow);
#endif

	/* show the prompt */
	TESetSelect(32767, 32767, gra_TEH);
	TEInsert(prompt, strlen(prompt), gra_TEH);
	TESetSelect(32767, 32767, gra_TEH);
	start = (*gra_TEH)->selStart;
	for(;;)
	{
		gra_waitforaction(0, &theEvent);
		end = (*gra_TEH)->selStart-1;
		if (end < start) continue;
		ch = (*(*gra_TEH)->hText)[end];
		if (ch == '\r' || ch == 4) break;
	}
	TEKey(010, gra_TEH);
	TEKey('\r', gra_TEH);
	if (ch == 4) return(0);
	j = 0;
	for(i=start; i<end; i++) outline[j++] = (*(*gra_TEH)->hText)[i];
	outline[j] = 0;
	return(outline);
}

/*
 * Routine to remove the last character from the scrolling messages window.
 * Called from "terminal.c"
 */
void gra_backup(void)
{
	GrafPtr savePort;

	GetPort(&savePort);
	if (gra_messageswindow == 0)
	{
		if (gra_showmessageswindow() != 0) return;
	}
	TEKey(010, gra_TEH);
	gra_showselect();
	SetPort(savePort);
}

/******************** STATUS BAR ROUTINES ********************/

/*
 * Routine to return the number of status lines on the display.
 */
INTSML ttynumstatuslines(void)
{
	return(MAXSTATUSLINES);
}

/*
 * Routine to display "message" in the status "field" of window "frame" (uses all windows
 * if "frame" is zero).  If "cancrop" is zero, field cannot be cropped and should be
 * replaced with "*" if there isn't room.
 */
void ttysetstatusfield(void *frame, STATUSFIELD *sf, char *message, INTSML cancrop)
{
	INTSML len, i, width, winwid, startx, endx;
	Rect clear;
	REGISTER WINDOWFRAME *mw, *mwwant;

	if (sf == 0) return;
	if (frame == 0) mwwant = NOWINDOWFRAME; else
		mwwant = (WINDOWFRAME *)frame;

	for(mw = gra_firstwindowframe; mw != NOWINDOWFRAME; mw = mw->nextwindowframe)
	{
		if (mw->floating != 0) continue;
		if (mwwant != NOWINDOWFRAME && mwwant != mw) continue;

		/* construct the status line */
		(void)strcpy(&gra_localstring[1], sf->label);
		(void)strcat(&gra_localstring[1], message);
		len = strlen(&gra_localstring[1]);
		while (len > 0 && gra_localstring[len] == ' ') gra_localstring[len--] = 0;

		/* special case for window title */
		if (sf->line == 0)
		{
#ifdef	USETK
			gra_namewindow(mw->tkwin, &gra_localstring[1]);
#else
			gra_localstring[0] = strlen(&gra_localstring[1]);
			SetWTitle((WindowPtr)mw->realwindow, (unsigned char *)gra_localstring);
#endif
			return;
		}

		/* determine how much room there is for the status field */
		SetPort((WindowPtr)mw->realwindow);
		winwid = mw->realwindow->portRect.right - mw->realwindow->portRect.left - SBARWIDTH;
		startx = winwid * sf->startper / 100 + mw->realwindow->portRect.left;
		endx = winwid * sf->endper / 100 + mw->realwindow->portRect.left;

		/* make sure the message fits */
		width = TextWidth(&gra_localstring[1], 0, len);
		if (width > endx-startx && cancrop == 0)
		{
			for(i=strlen(sf->label); gra_localstring[i+1] != 0; i++)
				gra_localstring[i+1] = '*';
		}
		while (len > 0 && width > endx-startx)
		{
			len--;
			width = TextWidth(&gra_localstring[1], 0, len);
		}

		/* display the field */
		TextFont(SFONT);
		TextSize(9);
		TextMode(srcOr);
		clear.left = startx;
		clear.right = endx;
		clear.bottom = mw->realwindow->portRect.bottom;
		clear.top = clear.bottom - SBARWIDTH+1;
		EraseRect(&clear);
		if (len > 0)
		{
			MoveTo(startx, mw->realwindow->portRect.bottom-3);
			DrawText(&gra_localstring[1], 0, len);
		}
	}
}

/*
 * Routine to free status field object "sf".
 */
void ttyfreestatusfield(STATUSFIELD *sf)
{
	efree(sf->label);
	efree((char *)sf);
}

/******************** GRAPHICS CONTROL ROUTINES ********************/

void flushscreen(void)
{
	WINDOWFRAME *mw;
	Rect dr;

	for(mw = gra_firstwindowframe; mw != NOWINDOWFRAME; mw = mw->nextwindowframe)
	{
		/* if screen has not changed, stop now */
		if (mw->offscreendirty == 0) continue;
		mw->offscreendirty = 0;

		/* make sure region falls inside screen */
		SetPort((WindowPtr)mw->realwindow);
		if (mw->copyrect.left < 0) mw->copyrect.left = 0;
		if (mw->copyrect.right > (*mw->window->portPixMap)->bounds.right)
			mw->copyrect.right = (*mw->window->portPixMap)->bounds.right;
		if (mw->copyrect.top < 0) mw->copyrect.top = 0;
		if (mw->copyrect.bottom > (*mw->window->portPixMap)->bounds.bottom)
			mw->copyrect.bottom = (*mw->window->portPixMap)->bounds.bottom;

		(void)LockPixels(mw->window->portPixMap);
		if (mw->floating == 0)
		{
			CopyBits((BitMap *)*(mw->window->portPixMap), (BitMap *)*mw->realwindow->portPixMap,
				&mw->copyrect, &mw->copyrect, srcCopy, 0L);
		} else
		{
			dr = mw->copyrect;
			dr.top += FLOATINGHEADERSIZE;
			dr.bottom += FLOATINGHEADERSIZE;
			CopyBits((BitMap *)*(mw->window->portPixMap), (BitMap *)*mw->realwindow->portPixMap,
				&mw->copyrect, &dr, srcCopy, 0L);
		}
		UnlockPixels(mw->window->portPixMap);
	}
}

/*
 * Routine to accumulate the rectangle of change to the offscreen PixMap
 */
void gra_setrect(WINDOWFRAME *mw, INTSML lx, INTSML hx, INTSML ly, INTSML hy)
{
	UINTBIG thistime;

	thistime = TickCount();
	if (mw->offscreendirty == 0)
	{
		mw->copyrect.left = lx;   mw->copyrect.right = hx;
		mw->copyrect.top = ly;    mw->copyrect.bottom = hy;
		mw->offscreendirty = 1;
		mw->starttime = thistime;
	} else
	{
		if (lx < mw->copyrect.left) mw->copyrect.left = lx;
		if (hx > mw->copyrect.right) mw->copyrect.right = hx;
		if (ly < mw->copyrect.top) mw->copyrect.top = ly;
		if (hy > mw->copyrect.bottom) mw->copyrect.bottom = hy;
	}

	/* flush the screen every two seconds */
	if (thistime - mw->starttime > FLUSHTICKS) flushscreen();
}

void gra_reloadmap(void)
{
	REGISTER VARIABLE *varred, *vargreen, *varblue;

	varred = getvalkey((INTBIG)us_aid, VAID, VINTEGER|VISARRAY, us_colormap_red);
	vargreen = getvalkey((INTBIG)us_aid, VAID, VINTEGER|VISARRAY, us_colormap_green);
	varblue = getvalkey((INTBIG)us_aid, VAID, VINTEGER|VISARRAY, us_colormap_blue);
	if (varred == NOVARIABLE || vargreen == NOVARIABLE || varblue == NOVARIABLE) return;
	us_loadmap((INTBIG *)varred->addr, (INTBIG *)vargreen->addr, (INTBIG *)varblue->addr, 0, 255);
}

void us_loadmap(INTBIG *red, INTBIG *green, INTBIG *blue, INTSML low, INTSML high)
{
	INTSML i;
	CTabHandle clut;
	REGISTER WINDOWFRAME *mw;

	for(mw = gra_firstwindowframe; mw != NOWINDOWFRAME; mw = mw->nextwindowframe)
	{
		clut = (*mw->window->portPixMap)->pmTable;
		for(i=low; i<=high; i++)
		{
			(*clut)->ctTable[i].rgb.red = red[i-low] << 8;
			(*clut)->ctTable[i].rgb.green = green[i-low] << 8;
			(*clut)->ctTable[i].rgb.blue = blue[i-low] << 8;
			if (i == 255)
			{
				(*clut)->ctTable[i].rgb.red = (255-red[i-low]) << 8;
				(*clut)->ctTable[i].rgb.green = (255-green[i-low]) << 8;
				(*clut)->ctTable[i].rgb.blue = (255-blue[i-low]) << 8;
			}
		}
		(*clut)->ctSeed++;

		/* mark the entire screen for redrawing */
		if (low == 0 && high == 255)
			gra_setrect(mw, 0, (*mw->window->portPixMap)->bounds.right, 0,
				(*mw->window->portPixMap)->bounds.bottom);
	}

	/* recache table and all PixMaps */
	CTabChanged(clut);
}

/*
 * helper routine to set the cursor shape to "state"
 */
void gra_set_cursorstate(INTSML state)
{
	if (gra_cursorstate == state) return;

	switch (state)
	{
		case NORMALCURSOR:
			SetCursor(&qd.arrow);
			break;
		case WANTTTYCURSOR:
			if (gra_wantttyCurs != 0L) SetCursor(&(**gra_wantttyCurs)); else
			SetCursor(&qd.arrow);
			break;
		case PENCURSOR:
			if (gra_penCurs != 0L) SetCursor(&(**gra_penCurs)); else
				SetCursor(&qd.arrow);
			break;
		case NULLCURSOR:
			if (gra_nullCurs != 0L) SetCursor(&(**gra_nullCurs)); else
				SetCursor(&qd.arrow);
			break;
		case MENUCURSOR:
			if (gra_menuCurs != 0L) SetCursor(&(**gra_menuCurs)); else
				SetCursor(&qd.arrow);
			break;
		case HANDCURSOR:
			if (gra_handCurs != 0L) SetCursor(&(**gra_handCurs)); else
				SetCursor(&qd.arrow);
			break;
		case TECHCURSOR:
			if (gra_techCurs != 0L) SetCursor(&(**gra_techCurs)); else
				SetCursor(&qd.arrow);
			break;
		case IBEAMCURSOR:
			if (gra_ibeamCurs != 0L) SetCursor(&(**gra_ibeamCurs)); else
				SetCursor(&qd.arrow);
			break;
	}
	gra_cursorstate = state;
}

/*
 * Routine to change the default cursor (to indicate modes)
 * If "curs" is 0, use the basic cursor.
 * If "curs" is 1, use the pen cursor (polygon edit mode).
 * If "curs" is 2, use the "T" cursor (technology edit mode).
 */
INTSML us_setnormalcursor(INTSML curs)
{
	INTSML lastcurs;

	switch (gra_normalcursor)
	{
		case NORMALCURSOR: lastcurs = 0;   break;
		case PENCURSOR:    lastcurs = 1;   break;
		case TECHCURSOR:   lastcurs = 2;   break;
	}
	switch (curs)
	{
		case 0: gra_normalcursor = NORMALCURSOR;   break;
		case 1: gra_normalcursor = PENCURSOR;      break;
		case 2: gra_normalcursor = TECHCURSOR;     break;
	}
	return(lastcurs);
}

/******************** LINE DRAWING ********************/

void us_invertline(WINDOW *win, INTSML x1, INTSML y1, INTSML x2, INTSML y2)
{
	REGISTER INTSML lx, hx, ly, hy, dx, dy, d, incr1, incr2, x, y, xend, yend, yincr, xincr;
	REGISTER WINDOWFRAME *frame;

	/* make sure the image buffer has not moved */
	frame = (WINDOWFRAME *)win->frame;
	if (GetPixBaseAddr(frame->window->portPixMap) != frame->rowstart[0])
		gra_rebuildrowstart(frame);

	/* get line type parameters */
	y1 = frame->revy - y1;   y2 = frame->revy - y2;

	/* initialize the Bresenham algorithm */
	dx = abs(x2-x1);
	dy = abs(y2-y1);
	if (dx > dy)
	{
		/* initialize for lines that increment along X */
		incr1 = 2 * dy;
		d = incr2 = 2 * (dy - dx);
		if (x1 > x2)
		{
			x = x2;   y = y2;   xend = x1;   yend = y1;
		} else
		{
			x = x1;   y = y1;   xend = x2;   yend = y2;
		}
		if (yend < y) yincr = -1; else yincr = 1;
		frame->rowstart[y][x] = ~frame->rowstart[y][x];

		/* draw line that increments along X */
		while (x < xend)
		{
			x++;
			if (d < 0) d += incr1; else
			{
				y += yincr;   d += incr2;
			}
			frame->rowstart[y][x] = ~frame->rowstart[y][x];
		}
	} else
	{
		/* initialize for lines that increment along Y */
		incr1 = 2 * dx;
		d = incr2 = 2 * (dx - dy);
		if (y1 > y2)
		{
			x = x2;   y = y2;   xend = x1;   yend = y1;
		} else
		{
			x = x1;   y = y1;   xend = x2;   yend = y2;
		}
		if (xend < x) xincr = -1; else xincr = 1;
		frame->rowstart[y][x] = ~frame->rowstart[y][x];

		/* draw line that increments along X */
		while (y < yend)
		{
			y++;
			if (d < 0) d += incr1; else
			{
				x += xincr;   d += incr2;
			}
			frame->rowstart[y][x] = ~frame->rowstart[y][x];
		}
	}
	if (x1 < x2) { lx = x1;   hx = x2; } else { lx = x2;   hx = x1; }
	if (y1 < y2) { ly = y1;   hy = y2; } else { ly = y2;   hy = y1; }
	gra_setrect(frame, lx, hx+1, ly, hy+1);
}

void us_drawline(WINDOW *win, INTSML x1, INTSML y1, INTSML x2, INTSML y2, GRAPHICS *desc, INTSML texture)
{
	REGISTER INTSML col, mask, lx, hx, ly, hy;
	static INTSML pat[] = {0xFF, 0x88, 0xE7, 0x80};
	REGISTER WINDOWFRAME *frame;

#ifdef	INVERTHIGHLIGHT
	/* if layer is HIGHLIGHT, invert */
	if (desc->bits == LAYERH)
	{
		us_invertline(win, x1, y1, x2, y2);
		return;
	}
#endif

	/* make sure the image buffer has not moved */
	frame = (WINDOWFRAME *)win->frame;
	if (GetPixBaseAddr(frame->window->portPixMap) != frame->rowstart[0])
		gra_rebuildrowstart(frame);

	/* get line type parameters */
	col = desc->col;         mask = ~desc->bits;
	y1 = frame->revy - y1;   y2 = frame->revy - y2;

	if (texture != 0) gra_drawpatline(frame, x1, y1, x2, y2, col, mask, pat[texture]); else
		gra_drawline(frame, x1, y1, x2, y2, col, mask);
	if (x1 < x2) { lx = x1;   hx = x2; } else { lx = x2;   hx = x1; }
	if (y1 < y2) { ly = y1;   hy = y2; } else { ly = y2;   hy = y1; }
	gra_setrect(frame, lx, hx+1, ly, hy+1);
}

void gra_drawpatline(WINDOWFRAME *frame, INTSML x1, INTSML y1, INTSML x2, INTSML y2, INTSML col,
	INTSML mask, INTSML pattern)
{
	INTSML dx, dy, d, incr1, incr2, x, y, xend, yend, yincr, xincr, i;

	/* initialize counter for line style */
	i = 0;

	/* initialize the Bresenham algorithm */
	dx = abs(x2-x1);
	dy = abs(y2-y1);
	if (dx > dy)
	{
		/* initialize for lines that increment along X */
		incr1 = 2 * dy;
		d = incr2 = 2 * (dy - dx);
		if (x1 > x2)
		{
			x = x2;   y = y2;   xend = x1;   yend = y1;
		} else
		{
			x = x1;   y = y1;   xend = x2;   yend = y2;
		}
		if (yend < y) yincr = -1; else yincr = 1;
		frame->rowstart[y][x] = (frame->rowstart[y][x]&mask) | col;

		/* draw line that increments along X */
		while (x < xend)
		{
			x++;
			if (d < 0) d += incr1; else
			{
				y += yincr;   d += incr2;
			}
			if (i == 7) i = 0; else i++;
			if ((pattern & (1 << i)) == 0) continue;
			frame->rowstart[y][x] = (frame->rowstart[y][x]&mask) | col;
		}
	} else
	{
		/* initialize for lines that increment along Y */
		incr1 = 2 * dx;
		d = incr2 = 2 * (dx - dy);
		if (y1 > y2)
		{
			x = x2;   y = y2;   xend = x1;   yend = y1;
		} else
		{
			x = x1;   y = y1;   xend = x2;   yend = y2;
		}
		if (xend < x) xincr = -1; else xincr = 1;
		frame->rowstart[y][x] = (frame->rowstart[y][x]&mask) | col;

		/* draw line that increments along X */
		while (y < yend)
		{
			y++;
			if (d < 0) d += incr1; else
			{
				x += xincr;   d += incr2;
			}
			if (i == 7) i = 0; else i++;
			if ((pattern & (1 << i)) == 0) continue;
			frame->rowstart[y][x] = (frame->rowstart[y][x]&mask) | col;
		}
	}
}

void gra_drawline(WINDOWFRAME *frame, INTSML x1, INTSML y1, INTSML x2, INTSML y2, INTSML col, INTSML mask)
{
	INTSML dx, dy, d, incr1, incr2, x, y, xend, yend, yincr, xincr;

	/* initialize the Bresenham algorithm */
	dx = abs(x2-x1);
	dy = abs(y2-y1);
	if (dx > dy)
	{
		/* initialize for lines that increment along X */
		incr1 = 2 * dy;
		d = incr2 = 2 * (dy - dx);
		if (x1 > x2)
		{
			x = x2;   y = y2;   xend = x1;   yend = y1;
		} else
		{
			x = x1;   y = y1;   xend = x2;   yend = y2;
		}
		if (yend < y) yincr = -1; else yincr = 1;
		frame->rowstart[y][x] = (frame->rowstart[y][x]&mask) | col;

		/* draw line that increments along X */
		while (x < xend)
		{
			x++;
			if (d < 0) d += incr1; else
			{
				y += yincr;   d += incr2;
			}
			frame->rowstart[y][x] = (frame->rowstart[y][x]&mask) | col;
		}
	} else
	{
		/* initialize for lines that increment along Y */
		incr1 = 2 * dx;
		d = incr2 = 2 * (dx - dy);
		if (y1 > y2)
		{
			x = x2;   y = y2;   xend = x1;   yend = y1;
		} else
		{
			x = x1;   y = y1;   xend = x2;   yend = y2;
		}
		if (xend < x) xincr = -1; else xincr = 1;
		frame->rowstart[y][x] = (frame->rowstart[y][x]&mask) | col;

		/* draw line that increments along X */
		while (y < yend)
		{
			y++;
			if (d < 0) d += incr1; else
			{
				x += xincr;   d += incr2;
			}
			frame->rowstart[y][x] = (frame->rowstart[y][x]&mask) | col;
		}
	}
}

/******************** POLYGON DRAWING ********************/

/*
 * routine to draw a polygon in the arrays (x, y) with "count" points.
 */
void us_drawpolygon(WINDOW *win, INTBIG *x, INTBIG *y, INTSML count, GRAPHICS *desc)
{
	REGISTER INTBIG i, j, k, l, ycur, yrev, wrap, lx, hx, ly, hy;
	REGISTER char *row;
	REGISTER INTSML col, mask, style, pat;
	REGISTER POLYSEG *a, *active, *edge, *lastedge, *left, *edgelist;
	REGISTER WINDOWFRAME *frame;

	/* make sure the image buffer has not moved */
	frame = (WINDOWFRAME *)win->frame;
	if (GetPixBaseAddr(frame->window->portPixMap) != frame->rowstart[0])
		gra_rebuildrowstart(frame);

	/* get parameters */
	col = desc->col;   mask = ~desc->bits;
	style = desc->style[COLORMAP] & NATURE;

	/* set redraw area */
	for(i=0; i<count; i++)
	{
		if (i == 0)
		{
			lx = hx = x[i];
			ly = hy = y[i];
		} else
		{
			lx = mini(lx, x[i]);
			hx = maxi(hx, x[i]);
			ly = mini(ly, y[i]);
			hy = maxi(hy, y[i]);
		}
	}

	/* color: make sure there is room in internal structures */
	if (count > gra_polysegcount)
	{
		if (gra_polysegcount > 0) efree((char *)gra_polysegs);
		gra_polysegcount = 0;
		gra_polysegs = (POLYSEG *)emalloc(count * (sizeof (POLYSEG)), us_aid->cluster);
		if (gra_polysegs == 0) return;
		gra_polysegcount = count;
	}

	/* fill in internal structures */
	edgelist = NOPOLYSEG;
	for(i=0; i<count; i++)
	{
		if (i == 0)
		{
			gra_polysegs[i].fx = x[count-1];
			gra_polysegs[i].fy = y[count-1];
		} else
		{
			gra_polysegs[i].fx = x[i-1];
			gra_polysegs[i].fy = y[i-1];
		}
		gra_polysegs[i].tx = x[i];   gra_polysegs[i].ty = y[i];
		gra_polysegs[i].index = i;

		/* compute the direction of this edge */
		j = gra_polysegs[i].ty - gra_polysegs[i].fy;
		if (j > 0) gra_polysegs[i].direction = 1; else
			if (j < 0) gra_polysegs[i].direction = -1; else
				gra_polysegs[i].direction = 0;

		/* compute the X increment of this edge */
		if (j == 0) gra_polysegs[i].increment = 0; else
		{
			gra_polysegs[i].increment = gra_polysegs[i].tx - gra_polysegs[i].fx;
			if (gra_polysegs[i].increment != 0) gra_polysegs[i].increment =
				(gra_polysegs[i].increment * 65536 - j + 1) / j;
		}
		gra_polysegs[i].tx <<= 16;   gra_polysegs[i].fx <<= 16;

		/* make sure "from" is above "to" */
		if (gra_polysegs[i].fy > gra_polysegs[i].ty)
		{
			j = gra_polysegs[i].tx;
			gra_polysegs[i].tx = gra_polysegs[i].fx;
			gra_polysegs[i].fx = j;
			j = gra_polysegs[i].ty;
			gra_polysegs[i].ty = gra_polysegs[i].fy;
			gra_polysegs[i].fy = j;
		}

		/* insert this edge into the edgelist, sorted by ascending "fy" */
		if (edgelist == NOPOLYSEG)
		{
			edgelist = &gra_polysegs[i];
			gra_polysegs[i].nextedge = NOPOLYSEG;
		} else
		{
			/* insert by ascending "fy" */
			if (edgelist->fy > gra_polysegs[i].fy)
			{
				gra_polysegs[i].nextedge = edgelist;
				edgelist = &gra_polysegs[i];
			} else for(a = edgelist; a != NOPOLYSEG; a = a->nextedge)
			{
				if (a->nextedge == NOPOLYSEG || a->nextedge->fy > gra_polysegs[i].fy)
				{
					/* insert after this */
					gra_polysegs[i].nextedge = a->nextedge;
					a->nextedge = &gra_polysegs[i];
					break;
				}
			}
		}
	}

	/* scan polygon and render */
	active = NOPOLYSEG;
	while (active != NOPOLYSEG || edgelist != NOPOLYSEG)
	{
		if (active == NOPOLYSEG)
		{
			active = edgelist;
			active->nextactive = NOPOLYSEG;
			edgelist = edgelist->nextedge;
			ycur = active->fy;
		}

		/* introduce edges from edge list into active list */
		while (edgelist != NOPOLYSEG && edgelist->fy <= ycur)
		{
			/* insert "edgelist" into active list, sorted by "fx" coordinate */
			if (active->fx > edgelist->fx ||
				(active->fx == edgelist->fx && active->increment > edgelist->increment))
			{
				edgelist->nextactive = active;
				active = edgelist;
				edgelist = edgelist->nextedge;
			} else for(a = active; a != NOPOLYSEG; a = a->nextactive)
			{
				if (a->nextactive == NOPOLYSEG || a->nextactive->fx > edgelist->fx ||
					(a->nextactive->fx == edgelist->fx && a->nextactive->increment > edgelist->increment))
				{
					/* insert after this */
					edgelist->nextactive = a->nextactive;
					a->nextactive = edgelist;
					edgelist = edgelist->nextedge;
					break;
				}
			}
		}

		/* generate regions to be filled in on current scan line */
		wrap = 0;
		left = active;
		for(edge = active; edge != NOPOLYSEG; edge = edge->nextactive)
		{
			wrap = wrap + edge->direction;
			if (wrap == 0)
			{
				j = (left->fx + 32768) >> 16;
				k = (edge->fx + 32768) >> 16;
				yrev = frame->revy - ycur;
				row = frame->rowstart[yrev];
				if (style == PATTERNED)
				{
					/* patterned fill */
					pat = desc->raster[yrev&7];
					if (pat != 0)
					{
#ifdef	INVERTHIGHLIGHT
						/* if layer is HIGHLIGHT, invert */
						if (desc->bits == LAYERH)
						{
							for(l=j; l<=k; l++)
							{
								if ((pat & (1 << (15-(l&15)))) != 0) row[l] = ~row[l];
							}
						} else
#endif
						{
							for(l=j; l<=k; l++)
							{
								if ((pat & (1 << (15-(l&15)))) != 0) row[l] = (row[l] & mask) | col;
							}
						}
					}
				} else
				{
					/* solid fill */
#ifdef	INVERTHIGHLIGHT
					/* if layer is HIGHLIGHT, invert */
					if (desc->bits == LAYERH)
					{
						for(l=j; l<=k; l++) row[l] = ~row[l];
					} else
#endif
					{
						for(l=j; l<=k; l++) row[l] = (row[l] & mask) | col;
					}
				}
				left = edge->nextactive;
			}
		}
		ycur++;

		/* update edges in active list */
		lastedge = NOPOLYSEG;
		for(edge = active; edge != NOPOLYSEG; edge = edge->nextactive)
		{
			if (ycur >= edge->ty)
			{
				if (lastedge == NOPOLYSEG) active = edge->nextactive; else
					lastedge->nextactive = edge->nextactive;
			} else
			{
				edge->fx += edge->increment;
				lastedge = edge;
			}
		}
	}
	gra_setrect(frame, lx, hx + 1, frame->revy-hy, frame->revy-ly + 1);
}

/******************** BOX DRAWING ********************/

void us_drawbox(WINDOW *win, INTSML lowx, INTSML highx, INTSML lowy, INTSML highy, GRAPHICS *desc)
{
	Rect r;
	INTSML col, mask, style, x, y, pat;
	REGISTER char *thisrow;
	REGISTER WINDOWFRAME *frame;

#ifdef	INVERTHIGHLIGHT
	/* if layer is HIGHLIGHT, invert */
	if (desc->bits == LAYERH)
	{
		us_invertbox(win, lowx, highx, lowy, highy);
		return;
	}
#endif

	/* make sure the image buffer has not moved */
	frame = (WINDOWFRAME *)win->frame;
	if (GetPixBaseAddr(frame->window->portPixMap) != frame->rowstart[0])
		gra_rebuildrowstart(frame);

	/* get graphics parameters */
	col = desc->col;   mask = ~desc->bits;
	style = desc->style[COLORMAP] & NATURE;
	r.left = lowx;                     r.right = highx + 1;
	r.bottom = frame->revy-lowy + 1;   r.top = frame->revy-highy;

	/* handle color drawing */
	if (style == PATTERNED)
	{
		/* special case the patterned fill */
		for(y=r.top; y<r.bottom; y++)
		{
			pat = desc->raster[y&7];
			if (pat == 0) continue;
			thisrow = frame->rowstart[y];
			for(x=r.left; x<r.right; x++)
			{
				if ((pat & (1 << (15-(x&15)))) != 0)
					thisrow[x] = (thisrow[x] & mask) | col;
			}
		}
	} else
	{
		for(y=r.top; y<r.bottom; y++)
		{
			thisrow = frame->rowstart[y];
			for(x=r.left; x<r.right; x++)
				thisrow[x] = (thisrow[x] & mask) | col;
		}
	}
	gra_setrect(frame, r.left, r.right, r.top, r.bottom);
}

/*
 * routine to invert the bits in the box from (lowx, lowy) to (highx, highy)
 */
void us_invertbox(WINDOW *win, INTSML lowx, INTSML highx, INTSML lowy, INTSML highy)
{
	Rect r;
	REGISTER WINDOWFRAME *frame;

	frame = (WINDOWFRAME *)win->frame;

	r.left = lowx;                     r.right = highx + 1;
	r.bottom = frame->revy-lowy + 1;   r.top = frame->revy-highy;

	/* prepare for graphics */
	SetGWorld(frame->window, 0L);
	(void)LockPixels(frame->window->portPixMap);

	PenMode(patXor);
	PenPat((ConstPatternParam)&qd.black);
	PaintRect(&r);

	UnlockPixels(frame->window->portPixMap);
	SetGWorld(frame->realwindow, gra_origgdevh);

	gra_setrect(frame, lowx, highx + 1, frame->revy-highy, frame->revy-lowy + 1);
}

/*
 * routine to move bits on the display starting with the area at
 * (sx,sy) and ending at (dx,dy).  The size of the area to be
 * moved is "wid" by "hei".
 */
void us_movebox(WINDOW *win, INTSML sx, INTSML sy, INTSML wid, INTSML hei, INTSML dx, INTSML dy)
{
	Rect from, to;
	INTSML xsize, ysize, x, y, dir, fromstart, frominc, tostart, toinc;
	REGISTER char *frombase, *tobase;
	REGISTER WINDOWFRAME *frame;

	frame = (WINDOWFRAME *)win->frame;

	/* make sure the image buffer has not moved */
	if (GetPixBaseAddr(frame->window->portPixMap) != frame->rowstart[0])
		gra_rebuildrowstart(frame);

	/* setup source rectangle */
	from.left = sx;
	from.right = from.left + wid;
	from.top = frame->revy + 1 - sy - hei;
	from.bottom = from.top + hei;

	/* setup destination rectangle */
	to.left = dx;
	to.right = to.left + wid;
	to.top = frame->revy + 1 - dy - hei;
	to.bottom = to.top + hei;

	/* determine size of bits to move */
	xsize = wid;   ysize = hei;

	/* determine direction of bit copy */
	if (from.left < to.left) dir = 1; else dir = 0;
	if (from.top < to.top)
	{
		fromstart = from.bottom-1;   frominc = -1;
		tostart = to.bottom-1;       toinc = -1;
	} else
	{
		fromstart = from.top;   frominc = 1;
		tostart = to.top;       toinc = 1;
	}

	/* move the bits */
	if (dir == 0)
	{
		/* normal forward copy in X */
		for(y = 0; y < ysize; y++)
		{
			frombase = frame->rowstart[fromstart] + from.left;
			fromstart += frominc;
			tobase = frame->rowstart[tostart] + to.left;
			tostart += toinc;
			for(x = 0; x < xsize; x++) *tobase++ = *frombase++;
		}
	} else
	{
		/* reverse copy in X */
		for(y = 0; y < ysize; y++)
		{
			frombase = frame->rowstart[fromstart] + from.right;
			fromstart += frominc;
			tobase = frame->rowstart[tostart] + to.right;
			tostart += toinc;
			for(x = 0; x < xsize; x++) *tobase-- = *frombase--;
		}
	}
	gra_setrect(frame, to.left, to.right, to.top, to.bottom);
}

/*
 * routine to save the contents of the box from "lx" to "hx" in X and from
 * "ly" to "hy" in Y.  A code is returned that identifies this box for
 * overwriting and restoring.  The routine returns -1 if there is a error.
 */
INTBIG us_savebox(WINDOW *win, INTSML lx, INTSML hx, INTSML ly, INTSML hy)
{
	SAVEDBOX *box;
	REGISTER INTBIG toindex, xsize, ysize;
	REGISTER INTSML i, x, y, truelx, truehx;
	REGISTER WINDOWFRAME *frame;

	frame = (WINDOWFRAME *)win->frame;

	/* make sure the image buffer has not moved */
	if (GetPixBaseAddr(frame->window->portPixMap) != frame->rowstart[0])
		gra_rebuildrowstart(frame);

	truelx = lx;   truehx = hx;
	i = ly;   ly = frame->revy-hy;   hy = frame->revy-i;
	xsize = hx-lx+1;
	ysize = hy-ly+1;

	box = (SAVEDBOX *)emalloc((sizeof (SAVEDBOX)), us_aid->cluster);
	if (box == 0) return(-1);
	box->pix = (char *)emalloc(xsize * ysize, us_aid->cluster);
	if (box->pix == 0) return(-1);
	box->win = win;
	box->nextsavedbox = gra_firstsavedbox;
	gra_firstsavedbox = box;
	box->lx = lx;           box->hx = hx;
	box->ly = ly;           box->hy = hy;
	box->truelx = truelx;   box->truehx = truehx;

	/* move the bits */
	toindex = 0;
	for(y = ly; y <= hy; y++)
	{
		for(x = lx; x <= hx; x++)
			box->pix[toindex++] = frame->rowstart[y][x];
	}

	return((INTBIG)box);
}

/*
 * routine to shift the saved box "code" so that it is restored in a different location,
 * offset by (dx,dy)
 */
void us_movesavedbox(INTBIG code, INTSML dx, INTSML dy)
{
	REGISTER SAVEDBOX *box;
	REGISTER WINDOWFRAME *frame;

	if (code == -1) return;
	box = (SAVEDBOX *)code;
	frame = (WINDOWFRAME *)box->win->frame;
	box->truelx += dx;   box->truehx += dx;
	box->lx += dx;       box->hx += dx;
	box->ly -= dy;       box->hy -= dy;
}

/*
 * routine to restore saved box "code" to the screen.  "destroy" is:
 *  0   restore box, do not free memory
 *  1   restore box, free memory
 * -1   free memory
 * Returns nonzero if there is an error.
 */
INTSML us_restorebox(INTBIG code, INTSML destroy)
{
	REGISTER SAVEDBOX *box, *lbox, *tbox;
	REGISTER INTBIG fromindex;
	REGISTER INTSML x, y;
	REGISTER WINDOWFRAME *frame;

	/* get the box */
	if (code == -1) return(1);
	box = (SAVEDBOX *)code;
	frame = (WINDOWFRAME *)box->win->frame;

	/* move the bits */
	if (destroy >= 0)
	{
		/* make sure the image buffer has not moved */
		if (GetPixBaseAddr(frame->window->portPixMap) != frame->rowstart[0])
			gra_rebuildrowstart(frame);
		fromindex = 0;
		for(y = box->ly; y <= box->hy; y++)
			for(x = box->lx; x <= box->hx; x++)
				frame->rowstart[y][x] = box->pix[fromindex++];
		gra_setrect(frame, box->truelx, box->truehx+1, box->ly, box->hy+1);
	}

	/* destroy this box's memory if requested */
	if (destroy != 0)
	{
		lbox = NOSAVEDBOX;
		for(tbox = gra_firstsavedbox; tbox != NOSAVEDBOX; tbox = tbox->nextsavedbox)
		{
			if (tbox == box) break;
			lbox = tbox;
		}
		if (lbox == NOSAVEDBOX) gra_firstsavedbox = box->nextsavedbox; else
			lbox->nextsavedbox = box->nextsavedbox;
		efree((char *)box->pix);
		efree((char *)box);
	}
	return(0);
}

/******************** TEXT DRAWING ********************/

void us_settextsize(WINDOW *win, INTSML fnt)
{
	/* geneva has no 6, 7, 12 */
	/* helvetica has no 6, 7, 20 */
	static convert[9] = {6, 7, 9, 10, 12, 14, 18, 20, 24};
	REGISTER WINDOWFRAME *frame;

	/* make sure the temp buffer exists */
	if (gra_textbuf == 0) gra_setuptextbuffer(100, 25);

	/* set sizes in this buffer */
	SetGWorld(gra_textbuf, 0L);
	if (fnt == TXTEDITOR)
	{
		TextFont(TFONT);
		gra_curfontsize = 10;
	} else if (fnt == TXTMENU)
	{
		TextFont(MFONT);
		gra_curfontsize = 12;
	} else
	{
		TextFont(EFONT);
		gra_curfontsize = convert[fnt];
	}
	TextSize(gra_curfontsize);
	GetFontInfo(&gra_curfontinfo);

	/* restore the world */
	frame = (WINDOWFRAME *)win->frame;
	SetGWorld(frame->realwindow, gra_origgdevh);
}

void us_textsize(WINDOW *win, char *str, INTSML *x, INTSML *y)
{
	REGISTER INTSML len;
	REGISTER WINDOWFRAME *frame;

	/* make sure the temp buffer exists */
	if (gra_textbuf == 0) gra_setuptextbuffer(100, 25);

	/* determine text size in this buffer */
	SetGWorld(gra_textbuf, 0L);
	len = strlen(str);
	*x = TextWidth(str, 0, len);
	*y = gra_curfontinfo.ascent + gra_curfontinfo.descent;

	/* restore the world */
	frame = (WINDOWFRAME *)win->frame;
	SetGWorld(frame->realwindow, gra_origgdevh);
}

void us_puttext(WINDOW *win, INTSML atx, INTSML aty, char *s, GRAPHICS *desc)
{
	REGISTER INTSML i, len, col, bits, mask, xp, yp, width, height;
	Rect dr, sr;
	REGISTER WINDOWFRAME *frame;
	REGISTER char *sourcerow, *destrow;

	frame = (WINDOWFRAME *)win->frame;

	/* copy the string and correct it */
	len = strlen(s);
	for(i=0; i<len; i++)
	{
		gra_localstring[i] = s[i] & 0177;
		if (gra_localstring[i] < ' ') gra_localstring[i] = ' ';
	}
	col = desc->col;   bits = desc->bits;

	/* make sure the temp buffer exists */
	if (gra_textbuf == 0) gra_setuptextbuffer(100, 25);

	/* determine string size */
	SetGWorld(gra_textbuf, 0L);
	width = TextWidth(gra_localstring, 0, len);
	height = gra_curfontinfo.ascent + gra_curfontinfo.descent;

	/* make sure the temp buffer is big enough */
	gra_setuptextbuffer(width, height);

	/* write to the temp buffer */
	sr.left = 0;   sr.right = width;
	sr.top = 0;    sr.bottom = height;
	EraseRect(&sr);
	TextMode(srcCopy);
	MoveTo(0, gra_curfontinfo.ascent);
	DrawText(gra_localstring, 0, len);
	SetGWorld(frame->realwindow, gra_origgdevh);

	/* set destination area */
	dr.left = atx;
	dr.right = dr.left + width;
	dr.bottom = frame->revy - aty;
	dr.top = dr.bottom - gra_curfontinfo.ascent - gra_curfontinfo.descent;

	/* clip to the window */
	if (dr.left < 0)
	{
		sr.left -= dr.left;
		dr.left = 0;
	}
	if (dr.right > (*frame->window->portPixMap)->bounds.right)
	{
		sr.right -= dr.right - (*frame->window->portPixMap)->bounds.right;
		dr.right = (*frame->window->portPixMap)->bounds.right;
	}
	if (dr.top < 0)
	{
		sr.top -= dr.top;
		dr.top = 0;
	}
	if (dr.bottom > (*frame->window->portPixMap)->bounds.bottom)
	{
		sr.bottom -= dr.bottom - (*frame->window->portPixMap)->bounds.bottom;
		dr.bottom = (*frame->window->portPixMap)->bounds.bottom;
	}

	/* make sure the image buffer has not moved */
	if (GetPixBaseAddr(frame->window->portPixMap) != frame->rowstart[0])
		gra_rebuildrowstart(frame);

	/* handle offscreen display */
	mask = ~bits;
	for(yp=sr.top; yp<sr.bottom; yp++)
	{
		sourcerow = &gra_textbufrowstart[yp][sr.left];
		destrow = &frame->rowstart[dr.top+yp][dr.left];
#ifdef	INVERTHIGHLIGHT
		/* if layer is HIGHLIGHT, invert */
		if (bits == LAYERH)
		{
			for(xp=sr.left; xp<sr.right; xp++)
			{
				if (*sourcerow++ != 0) *destrow = ~(*destrow);
				destrow++;
			}
		} else
#endif
		{
			for(xp=sr.left; xp<sr.right; xp++)
			{
				if (*sourcerow++ != 0) *destrow = (*destrow & mask) | col;
				destrow++;
			}
		}
	}
	SetFontLock(1);
	gra_setrect(frame, dr.left, dr.right, dr.top, dr.bottom);
}

void gra_setuptextbuffer(INTSML width, INTSML height)
{
	Rect r;
	REGISTER INTSML i, bytes;
	REGISTER char *addr;

	if (width <= gra_textbufwid && height <= gra_textbufhei) return;

	r.left = 0;
	r.right = width;
	r.top = 0;
	r.bottom = height;
	if (gra_textbuf == 0)
	{
		NewGWorld(&gra_textbuf, 8, &r, 0L, 0L, 0);
		(void)LockPixels(gra_textbuf->portPixMap);
	} else
	{
		UpdateGWorld(&gra_textbuf, 8, &r, 0L, 0L, 0);
		(void)LockPixels(gra_textbuf->portPixMap);
		DisposPtr((Ptr)gra_textbufrowstart);
	}
	gra_textbufrowstart = (char **)NewPtr(height * (sizeof (char *)));
	bytes = (*gra_textbuf->portPixMap)->rowBytes & 0x7FFF;
	addr = GetPixBaseAddr(gra_textbuf->portPixMap);
	for(i=0; i<height; i++)
	{
		gra_textbufrowstart[i] = addr;
		addr += bytes;
	}
	gra_textbufwid = width;   gra_textbufhei = height;
}

/******************** CIRCLE DRAWING ********************/

void us_drawcircle(WINDOW *win, INTBIG atx, INTBIG aty, INTBIG radius, GRAPHICS *desc)
{
	Rect r;
	REGISTER INTSML col, mask;
	REGISTER INTBIG x, y, d, maxx, maxy, thisx, thisy;
	REGISTER WINDOWFRAME *frame;
	REGISTER char *thisrow;

#ifdef	INVERTHIGHLIGHT
	/* if layer is HIGHLIGHT, invert */
	if (desc->bits == LAYERH)
	{
		gra_invertcircle(win, atx, aty, radius);
		return;
	}
#endif

	/* make sure the image buffer has not moved */
	frame = (WINDOWFRAME *)win->frame;
	if (GetPixBaseAddr(frame->window->portPixMap) != frame->rowstart[0])
		gra_rebuildrowstart(frame);

	/* get parameters */
	col = desc->col;   mask = ~desc->bits;
	aty = frame->revy - aty;

	maxx = (*frame->window->portPixMap)->bounds.right;
	maxy = (*frame->window->portPixMap)->bounds.bottom;
	x = 0;   y = radius;
	d = 3 - 2 * radius;
	if (r.left >= 0 && r.right < maxx && r.top >= 0 && r.bottom < maxy)
	{
		/* no clip version is faster */
		while (x <= y)
		{
			thisrow = frame->rowstart[aty + y];
			thisrow[atx + x] = (thisrow[atx + x] & mask) | col;
			thisrow[atx - x] = (thisrow[atx - x] & mask) | col;

			thisrow = frame->rowstart[aty - y];
			thisrow[atx + x] = (thisrow[atx + x] & mask) | col;
			thisrow[atx - x] = (thisrow[atx - x] & mask) | col;

			thisrow = frame->rowstart[aty + x];
			thisrow[atx + y] = (thisrow[atx + y] & mask) | col;
			thisrow[atx - y] = (thisrow[atx - y] & mask) | col;

			thisrow = frame->rowstart[aty - x];
			thisrow[atx + y] = (thisrow[atx + y] & mask) | col;
			thisrow[atx - y] = (thisrow[atx - y] & mask) | col;

			if (d < 0) d += 4*x + 6; else
			{
				d += 4 * (x-y) + 10;
				y--;
			}
			x++;
		}
	} else
	{
		/* clip version */
		while (x <= y)
		{
			thisy = aty + y;
			if (thisy >= 0 && thisy < maxy)
			{
				thisrow = frame->rowstart[thisy];
				thisx = atx + x;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = (thisrow[thisx] & mask) | col;
				thisx = atx - x;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = (thisrow[thisx] & mask) | col;
			}

			thisy = aty - y;
			if (thisy >= 0 && thisy < maxy)
			{
				thisrow = frame->rowstart[thisy];
				thisx = atx + x;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = (thisrow[thisx] & mask) | col;
				thisx = atx - x;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = (thisrow[thisx] & mask) | col;
			}

			thisy = aty + x;
			if (thisy >= 0 && thisy < maxy)
			{
				thisrow = frame->rowstart[thisy];
				thisx = atx + y;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = (thisrow[thisx] & mask) | col;
				thisx = atx - y;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = (thisrow[thisx] & mask) | col;
			}

			thisy = aty - x;
			if (thisy >= 0 && thisy < maxy)
			{
				thisrow = frame->rowstart[thisy];
				thisx = atx + y;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = (thisrow[thisx] & mask) | col;
				thisx = atx - y;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = (thisrow[thisx] & mask) | col;
			}

			if (d < 0) d += 4*x + 6; else
			{
				d += 4 * (x-y) + 10;
				y--;
			}
			x++;
		}
	}

	/* set redraw area */
	r.left = atx - radius;
	r.right = atx + radius + 1;
	r.top = aty - radius;
	r.bottom = aty + radius + 1;
	gra_setrect(frame, r.left, r.right, r.top, r.bottom);
}

#ifdef	INVERTHIGHLIGHT
void gra_invertcircle(WINDOW *win, INTBIG atx, INTBIG aty, INTBIG radius)
{
	Rect r;
	REGISTER INTBIG x, y, d, maxx, maxy, thisx, thisy;
	REGISTER WINDOWFRAME *frame;
	REGISTER char *thisrow;

	/* make sure the image buffer has not moved */
	frame = (WINDOWFRAME *)win->frame;
	if (GetPixBaseAddr(frame->window->portPixMap) != frame->rowstart[0])
		gra_rebuildrowstart(frame);

	/* get parameters */
	aty = frame->revy - aty;

	maxx = (*frame->window->portPixMap)->bounds.right;
	maxy = (*frame->window->portPixMap)->bounds.bottom;
	x = 0;   y = radius;
	d = 3 - 2 * radius;
	if (r.left >= 0 && r.right < maxx && r.top >= 0 && r.bottom < maxy)
	{
		/* no clip version is faster */
		while (x <= y)
		{
			thisrow = frame->rowstart[aty + y];
			thisrow[atx + x] = ~thisrow[atx + x];
			thisrow[atx - x] = ~thisrow[atx - x];

			thisrow = frame->rowstart[aty - y];
			thisrow[atx + x] = ~thisrow[atx + x];
			thisrow[atx - x] = ~thisrow[atx - x];

			thisrow = frame->rowstart[aty + x];
			thisrow[atx + y] = ~thisrow[atx + y];
			thisrow[atx - y] = ~thisrow[atx - y];

			thisrow = frame->rowstart[aty - x];
			thisrow[atx + y] = ~thisrow[atx + y];
			thisrow[atx - y] = ~thisrow[atx - y];

			if (d < 0) d += 4*x + 6; else
			{
				d += 4 * (x-y) + 10;
				y--;
			}
			x++;
		}
	} else
	{
		/* clip version */
		while (x <= y)
		{
			thisy = aty + y;
			if (thisy >= 0 && thisy < maxy)
			{
				thisrow = frame->rowstart[thisy];
				thisx = atx + x;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = ~thisrow[thisx];
				thisx = atx - x;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = ~thisrow[thisx];
			}

			thisy = aty - y;
			if (thisy >= 0 && thisy < maxy)
			{
				thisrow = frame->rowstart[thisy];
				thisx = atx + x;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = ~thisrow[thisx];
				thisx = atx - x;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = ~thisrow[thisx];
			}

			thisy = aty + x;
			if (thisy >= 0 && thisy < maxy)
			{
				thisrow = frame->rowstart[thisy];
				thisx = atx + y;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = ~thisrow[thisx];
				thisx = atx - y;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = ~thisrow[thisx];
			}

			thisy = aty - x;
			if (thisy >= 0 && thisy < maxy)
			{
				thisrow = frame->rowstart[thisy];
				thisx = atx + y;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = ~thisrow[thisx];
				thisx = atx - y;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = ~thisrow[thisx];
			}

			if (d < 0) d += 4*x + 6; else
			{
				d += 4 * (x-y) + 10;
				y--;
			}
			x++;
		}
	}

	/* set redraw area */
	r.left = atx - radius;
	r.right = atx + radius + 1;
	r.top = aty - radius;
	r.bottom = aty + radius + 1;
	gra_setrect(frame, r.left, r.right, r.top, r.bottom);
}
#endif

/******************** DISC DRAWING ********************/

/*
 * routine to draw row "thisy" of a disc from "startx" <= X <= "endx"
 */
void gra_drawdiscrow(WINDOWFRAME *frame, INTBIG thisy, INTBIG startx, INTBIG endx, GRAPHICS *desc)
{
	REGISTER char *thisrow;
	REGISTER INTBIG x;
	REGISTER INTSML pat;

	if (thisy < 0 || thisy >= gra_curvemaxy) return;
	thisrow = frame->rowstart[thisy];
	if (startx < 0) startx = 0;
	if (endx >= gra_curvemaxx) endx = gra_curvemaxy - 1;
	if (gra_curvestyle == PATTERNED)
	{
		pat = desc->raster[thisy&7];
		if (pat != 0)
		{
			for(x=startx; x<=endx; x++)
				if ((pat & (1 << (15-(x&15)))) != 0)
					thisrow[x] = (thisrow[x] & gra_curvemask) | gra_curvecol;
		}
	} else
	{
		for(x=startx; x<=endx; x++)
			thisrow[x] = (thisrow[x] & gra_curvemask) | gra_curvecol;
	}
}

#ifdef	INVERTHIGHLIGHT
/*
 * routine to invert row "thisy" of a disc from "startx" <= X <= "endx"
 */
void gra_invdiscrow(WINDOWFRAME *frame, INTBIG thisy, INTBIG startx, INTBIG endx, GRAPHICS *desc)
{
	REGISTER char *thisrow;
	REGISTER INTBIG x;
	REGISTER INTSML pat;

	if (thisy < 0 || thisy >= gra_curvemaxy) return;
	thisrow = frame->rowstart[thisy];
	if (startx < 0) startx = 0;
	if (endx >= gra_curvemaxx) endx = gra_curvemaxy - 1;
	if (gra_curvestyle == PATTERNED)
	{
		pat = desc->raster[thisy&7];
		if (pat != 0)
		{
			for(x=startx; x<=endx; x++)
				if ((pat & (1 << (15-(x&15)))) != 0) thisrow[x] = ~thisrow[x];
		}
	} else
	{
		for(x=startx; x<=endx; x++) thisrow[x] = ~thisrow[x];
	}
}
#endif

/*
 * routine to draw a filled-in circle at (atx,aty) with radius "radius"
 */
void us_drawdisc(WINDOW *win, INTBIG atx, INTBIG aty, INTBIG radius, GRAPHICS *desc)
{
	Rect r;
	REGISTER WINDOWFRAME *frame;
	REGISTER INTBIG x, y, d;

	/* make sure the image buffer has not moved */
	frame = (WINDOWFRAME *)win->frame;
	if (GetPixBaseAddr(frame->window->portPixMap) != frame->rowstart[0])
		gra_rebuildrowstart(frame);

	/* get parameters */
	gra_curvestyle = desc->style[COLORMAP] & NATURE;
	gra_curvecol = desc->col;   gra_curvemask = ~desc->bits;
	aty = frame->revy - aty;

#ifdef	INVERTHIGHLIGHT
	if (desc->bits == LAYERH) gra_discrowdraw = gra_invdiscrow; else
#endif
		gra_discrowdraw = gra_drawdiscrow;

	gra_curvemaxx = (*frame->window->portPixMap)->bounds.right;
	gra_curvemaxy = (*frame->window->portPixMap)->bounds.bottom;
	x = 0;   y = radius;
	d = 3 - 2 * radius;
	while (x <= y)
	{
		(*gra_discrowdraw)(frame, aty+y, atx-x, atx+x, desc);
		(*gra_discrowdraw)(frame, aty-y, atx-x, atx+x, desc);
		(*gra_discrowdraw)(frame, aty+x, atx-y, atx+y, desc);
		(*gra_discrowdraw)(frame, aty-x, atx-y, atx+y, desc);

		if (d < 0) d += 4*x + 6; else
		{
			d += 4 * (x-y) + 10;
			y--;
		}
		x++;
	}

	/* set redraw area */
	r.left = atx - radius;
	r.right = atx + radius + 1;
	r.top = aty - radius;
	r.bottom = aty + radius + 1;
	gra_setrect(frame, r.left, r.right, r.top, r.bottom);
}

/******************** ARC DRAWING ********************/

INTSML gra_arcfindoctant(INTBIG x, INTBIG y)
{
	if (x > 0)
		if (y >= 0)
			if (y >= x)	 return 7;
			else         return 8;
		else
			if (x >= -y) return 1;
			else         return 2;
	else
		if (y > 0)
			if (y > -x)  return 6;
			else         return 5;
		else
			if (y > x)   return 4;
			else         return 3;
}

void gra_arcxformoctant(INTBIG x, INTBIG y, INTBIG oct, INTBIG *ox, INTBIG *oy)
{
	switch (oct)
	{
		case 1 : *ox = -y;   *oy = x;   break;
		case 2 : *ox = x;    *oy = -y;  break;
		case 3 : *ox = -x;   *oy = -y;  break;
		case 4 : *ox = -y;   *oy = -x;  break;
		case 5 : *ox = y;    *oy = -x;  break;
		case 6 : *ox = -x;   *oy = y;   break;
		case 7 : *ox = x;    *oy = y;   break;
		case 8 : *ox = y;    *oy = x;   break;
	}
}

#ifdef	INVERTHIGHLIGHT
void gra_arcinvpixel(INTBIG x, INTBIG y)
{
	if (x < 0 || x >= gra_curvemaxx || y < 0 || y >= gra_curvemaxy) return;
	if (gra_arcfirst != 0)
	{
		gra_arcfirst = 0;
		gra_arclx = gra_archx = x;
		gra_arcly = gra_archy = y;
	} else
	{
		if (x < gra_arclx) gra_arclx = x;
		if (x > gra_archx) gra_archx = x;
		if (y < gra_arcly) gra_arcly = y;
		if (y > gra_archy) gra_archy = y;
	}
	gra_arcframe->rowstart[y][x] = ~gra_arcframe->rowstart[y][x];
}
#endif

void gra_arcdopixel(INTBIG x, INTBIG y)
{
	if (x < 0 || x >= gra_curvemaxx || y < 0 || y >= gra_curvemaxy) return;
	if (gra_arcfirst != 0)
	{
		gra_arcfirst = 0;
		gra_arclx = gra_archx = x;
		gra_arcly = gra_archy = y;
	} else
	{
		if (x < gra_arclx) gra_arclx = x;
		if (x > gra_archx) gra_archx = x;
		if (y < gra_arcly) gra_arcly = y;
		if (y > gra_archy) gra_archy = y;
	}
	gra_arcframe->rowstart[y][x] = (gra_arcframe->rowstart[y][x] & gra_curvemask) | gra_curvecol;
}

void gra_arcoutxform(INTBIG x, INTBIG y)
{
	if (gra_arcocttable[1]) (*gra_arcpixeldraw)( y + gra_arccenterx, -x + gra_arccentery);
	if (gra_arcocttable[2]) (*gra_arcpixeldraw)( x + gra_arccenterx, -y + gra_arccentery);
	if (gra_arcocttable[3]) (*gra_arcpixeldraw)(-x + gra_arccenterx, -y + gra_arccentery);
	if (gra_arcocttable[4]) (*gra_arcpixeldraw)(-y + gra_arccenterx, -x + gra_arccentery);
	if (gra_arcocttable[5]) (*gra_arcpixeldraw)(-y + gra_arccenterx,  x + gra_arccentery);
	if (gra_arcocttable[6]) (*gra_arcpixeldraw)(-x + gra_arccenterx,  y + gra_arccentery);
	if (gra_arcocttable[7]) (*gra_arcpixeldraw)( x + gra_arccenterx,  y + gra_arccentery);
	if (gra_arcocttable[8]) (*gra_arcpixeldraw)( y + gra_arccenterx,  x + gra_arccentery);
}

void gra_arcbrescw(INTBIG x, INTBIG y, INTBIG x1, INTBIG y1)
{
	REGISTER INTBIG d;

	d = 3 - 2 * y + 4 * x;
	while (x < x1 && y > y1)
	{
		gra_arcoutxform(x, y);
		if (d < 0) d += 4*x+6; else
		{
			d += 4*(x-y)+10;
			y--;
		}
		x++;
	}

	/* get to the end */
	for ( ; x < x1; x++) gra_arcoutxform(x, y);
	for ( ; y > y1; y--) gra_arcoutxform(x, y);
   gra_arcoutxform(x1, y1);
}

void gra_arcbresmidcw(INTBIG x, INTBIG y)
{
	REGISTER INTBIG d;

	d = 3 - 2 * y + 4 * x;
	while (x < y)
	{
		gra_arcoutxform(x, y);
		if (d < 0) d += 4*x+6; else
		{
			d += 4*(x-y)+10;
			y--;
		}
		x++;
   }
   if (x == y) gra_arcoutxform(x, y);
}

void gra_arcbresmidccw(INTBIG x, INTBIG y)
{
	REGISTER INTBIG d;

	d = 3 + 2 * y - 4 * x;
	while (x > 0)
	{
		gra_arcoutxform(x, y);
		if (d > 0) d += 6-4*x; else
		{
			d += 4*(y-x)+10;
			y++;
		}
		x--;
   }
   gra_arcoutxform(0, gra_arcradius);
}

void gra_arcbresccw(INTBIG x, INTBIG y, INTBIG x1, INTBIG y1)
{
	REGISTER INTBIG d;

	d = 3 + 2 * y + 4 * x;
	while(x > x1 && y < y1)
	{
		/* not always correct */
		gra_arcoutxform(x, y);
		if (d > 0) d += 6 - 4*x; else
		{
			d += 4*(y-x)+10;
			y++;
		}
		x--;
	}

	/* get to the end */
	for ( ; x > x1; x--) gra_arcoutxform(x, y);
	for ( ; y < y1; y++) gra_arcoutxform(x, y);
	gra_arcoutxform(x1, y1);
}

/*
 * draws an arc centered at (centerx, centery), clockwise,
 * passing by (x1,y1) and (x2,y2)
 */
void us_drawcirclearc(WINDOW *win, INTBIG centerx, INTBIG centery, INTBIG p1_x, INTBIG p1_y,
	INTBIG p2_x, INTBIG p2_y, GRAPHICS *desc)
{
	REGISTER INTBIG alternate, pa_x, pa_y, pb_x, pb_y, i, diff;
	INTBIG x, y;
	REGISTER INTSML start_oct, end_oct;

	/* ignore tiny arcs */
	if (p1_x == p2_x && p1_y == p2_y) return;

	/* make sure the image buffer has not moved */
	gra_arcframe = (WINDOWFRAME *)win->frame;
	if (GetPixBaseAddr(gra_arcframe->window->portPixMap) != gra_arcframe->rowstart[0])
		gra_rebuildrowstart(gra_arcframe);

	/* get parameters */
	gra_curvecol = desc->col;   gra_curvemask = ~desc->bits;
	gra_curvemaxx = (*gra_arcframe->window->portPixMap)->bounds.right;
	gra_curvemaxy = (*gra_arcframe->window->portPixMap)->bounds.bottom;
	p1_y = gra_arcframe->revy - p1_y;
	p2_y = gra_arcframe->revy - p2_y;
	gra_arccentery = gra_arcframe->revy - centery;
	gra_arccenterx = centerx;
	pa_x = p2_x - gra_arccenterx;
	pa_y = p2_y - gra_arccentery;
	pb_x = p1_x - gra_arccenterx;
	pb_y = p1_y - gra_arccentery;
	gra_arcradius = computedistance(gra_arccenterx, gra_arccentery, p2_x, p2_y);
	alternate = computedistance(gra_arccenterx, gra_arccentery, p1_x, p1_y);
	start_oct = gra_arcfindoctant(pa_x, pa_y);
	end_oct   = gra_arcfindoctant(pb_x, pb_y);
	gra_arcfirst = 1;
#ifdef	INVERTHIGHLIGHT
	if (desc->bits == LAYERH) gra_arcpixeldraw = gra_arcinvpixel; else
#endif
		gra_arcpixeldraw = gra_arcdopixel;

	/* move the point */
	if (gra_arcradius != alternate)
	{
		diff = gra_arcradius-alternate;
		switch (end_oct)
		{
			case 6:
			case 7: /*  y >  x */ pb_y += diff;  break;
			case 8: /*  x >  y */
			case 1: /*  x > -y */ pb_x += diff;  break;
			case 2: /* -y >  x */
			case 3: /* -y > -x */ pb_y -= diff;  break;
			case 4: /* -y < -x */
			case 5: /*  y < -x */ pb_x -= diff;  break;
		}
	}

	for(i=1; i<9; i++) gra_arcocttable[i] = 0;

	if (start_oct == end_oct)
	{
		INTBIG x1, y1, x2, y2;

		gra_arcocttable[start_oct] = 1;
		gra_arcxformoctant(pa_x, pa_y, start_oct, &x1, &y1);
		gra_arcxformoctant(pb_x, pb_y, start_oct, &x2 ,&y2);

		if (ODD(start_oct)) gra_arcbrescw(x1, y1, x2, y2);
		else				gra_arcbresccw(x1 ,y1, x2, y2);
		gra_arcocttable[start_oct] = 0;
	} else
	{
		gra_arcocttable[start_oct] = 1;
		gra_arcxformoctant(pa_x, pa_y, start_oct, &x, &y);
		if (ODD(start_oct)) gra_arcbresmidcw(x, y);
		else				gra_arcbresmidccw(x, y);
		gra_arcocttable[start_oct] = 0;

		gra_arcocttable[end_oct] = 1;
		gra_arcxformoctant(pb_x, pb_y, end_oct, &x, &y);
		if (ODD(end_oct)) gra_arcbresmidccw(x, y);
		else			  gra_arcbresmidcw(x, y);
		gra_arcocttable[end_oct] = 0;

		if (MODP(start_oct+1) != end_oct)
		{
			if (MODP(start_oct+1) == MODM(end_oct-1)) gra_arcocttable[MODP(start_oct+1)] = 1; else
				for(i = MODP(start_oct+1); i != end_oct; i = MODP(i+1)) gra_arcocttable[i] = 1;
			gra_arcbresmidcw(0, gra_arcradius);
		}
	}

	/* set redraw area */
	if (gra_arcfirst == 0)
		gra_setrect(gra_arcframe, gra_arclx, gra_archx+1, gra_arcly, gra_archy+1);
}

/******************** GRID DRAWING ********************/

/*
 * grid drawing routine
 */
void us_drawgrid(WINDOW *win, POLYGON *obj)
{
	REGISTER INTBIG i, j, xnum, xden, ynum, yden, x0,y0, x1,y1, x2,y2, x3,y3,
		x4,y4, x5,y5, x10, y10, y10mod;
	REGISTER INTSML x, y;
	REGISTER WINDOWFRAME *frame;

	frame = (WINDOWFRAME *)win->frame;

	/* make sure the image buffer has not moved */
	if (GetPixBaseAddr(frame->window->portPixMap) != frame->rowstart[0])
		gra_rebuildrowstart(frame);

	x0 = obj->xv[0];   y0 = obj->yv[0];		/* screen space grid spacing */
	x1 = obj->xv[1];   y1 = obj->yv[1];		/* screen space grid start */
	x2 = obj->xv[2];   y2 = obj->yv[2];		/* display space low */
	x3 = obj->xv[3];   y3 = obj->yv[3];		/* display space high */
	x4 = obj->xv[4];   y4 = obj->yv[4];		/* screen space low */
	x5 = obj->xv[5];   y5 = obj->yv[5];		/* screen space high */
	xnum = x3 - x2;
	xden = x5 - x4;
	ynum = y3 - y2;
	yden = y5 - y4;
	x10 = x0*10;       y10 = y0*10;

	/* draw the grid to the offscreen PixMap */
	for(i = y1; i < y5; i += y0)
	{
		y = muldiv(i-y4, ynum, yden) + y2;
		if (y < y2 || y > y3) continue;
		y = frame->revy - y;
		y10mod = i % y10;

		for(j = x1; j < x5; j += x0)
		{
			x = muldiv(j-x4, xnum, xden) + x2;
			if (x >= x2 && x <= x3) frame->rowstart[y][x] |= GRID;

			/* special case every 10 grid points in each direction */
			if ((j%x10) == 0 && y10mod == 0)
			{
				if (x > x2) frame->rowstart[y][x-1] |= GRID;
				if (x < x3) frame->rowstart[y][x+1] |= GRID;
				if (y > y2) frame->rowstart[y-1][x] |= GRID;
				if (y < y3) frame->rowstart[y+1][x] |= GRID;
			}
		}
	}

	/* copy it back to the screen */
	gra_setrect(frame, x2, x3, frame->revy-y3, frame->revy-y2);
}

/******************** MOUSE CONTROL ********************/

/*
 * routine to return the number of buttons on the mouse
 */
INTSML buttoncount(void)
{
	return(mini(BUTTONS, NUMBUTS));
}

/*
 * routine to tell whether button "but" is a double-click
 */
INTSML doublebutton(INTSML b)
{
	if (b == 16) return(1);
	return(0);
}

/*
 * routine to tell whether button "but" has the "shift" key held
 */
INTSML shiftbutton(INTSML b)
{
	if ((b%2) == 1) return(1);
	return(0);
}

/*
 * routine to return the name of button "b" (from 0 to "buttoncount()").
 * The number of letters unique to the button is placed in "important".
 */
char *buttonname(INTSML b, INTSML *important)
{
	*important = gra_buttonname[b].unique;
	return(gra_buttonname[b].name);
}

/*
 * routine to convert from "gra_inputstate" (the typical input parameter)
 * to button numbers (the table "gra_buttonname")
 */
INTSML gra_makebutton(INTBIG state)
{
	INTSML base;

	if ((state&DOUBLECLICK) != 0) return(16);
	base = 0;
	if ((state&CONTROLDOWN) != 0) base += 8;
	if ((state&OPTIONDOWN) != 0) base += 4;
	if ((state&COMMANDDOWN) != 0) base += 2;
	if ((state&SHIFTDOWN) != 0) base++;
	return(base);
}

/*
 * routine to wait for a button push and return its index (0 based) in "*but".
 * The coordinates of the cursor are placed in "*x" and "*y".  If there is no
 * button push, the value of "*but" is negative.
 */
void waitforbutton(INTSML *x, INTSML *y, INTSML *but)
{
	EventRecord theEvent;

	if (gra_inputstate != NOEVENT && (gra_inputstate&(ISBUTTON|BUTTONUP)) == ISBUTTON)
	{
		*but = gra_makebutton(gra_inputstate);
		*x = gra_cursorx;
		*y = gra_cursory;
		gra_inputstate = NOEVENT;
		if (gra_cursorstate != IBEAMCURSOR) gra_set_cursorstate(NULLCURSOR);
		return;
	}

	gra_waitforaction(1, &theEvent);

	if (gra_inputstate != NOEVENT && (gra_inputstate&(ISBUTTON|BUTTONUP)) == ISBUTTON)
	{
		*but = gra_makebutton(gra_inputstate);
		*x = gra_cursorx;
		*y = gra_cursory;
		gra_inputstate = NOEVENT;
		if (gra_cursorstate != IBEAMCURSOR) gra_set_cursorstate(NULLCURSOR);
		return;
	}
	*but = -1;
}

/*
 * routine to track the cursor until a button is released, calling "whileup" for
 * each co-ordinate when the mouse moves before the first button push, calling
 * "whendown" once when the button goes down, calling "eachdown" for each
 * co-ordinate when the mouse moves after the button is pushed, calling
 * "eachchar" for each key that is typed at any time, and calling "done" once
 * when done.  The "whendown" and "done" routines are called with no parameters;
 * "whileup" and "eachdown" are called with the X and Y coordinates of the
 * cursor; and "eachchar" is called with the X, Y, and character value that was
 * typed.  The "whileup", "eachdown", and "eachchar" routines return nonzero to
 * abort tracking.
 * If "waitforpush" is nonzero then the routine will wait for a button to
 * actually be pushed before tracking (otherwise it will begin tracking
 * immediately).  The value of "purpose" determines what the cursor will look
 * like during dragging: 0 for normal (the standard cursor), 1 for drawing (a pen),
 * 2 for dragging (a hand), 3 for popup menu selection (a horizontal arrow), 4 for
 * hierarchical popup menu selection (arrow, stays at end).
 */
void trackcursor(INTSML waitforpush, INTSML (*whileup)(INTBIG, INTBIG), void (*whendown)(void),
	INTSML (*eachdown)(INTBIG, INTBIG), INTSML (*eachchar)(INTBIG, INTBIG, INTSML), void (*done)(void),
		INTSML purpose)
{
	REGISTER INTSML keepon;
	EventRecord theEvent;

	/* change the cursor to an appropriate icon */
	switch (purpose)
	{
		case TRACKDRAWING:    gra_set_cursorstate(PENCURSOR);    break;
		case TRACKDRAGGING:   gra_set_cursorstate(HANDCURSOR);   break;
		case TRACKSELECTING:
		case TRACKHSELECTING: gra_set_cursorstate(MENUCURSOR);   break;
	}

	/* now wait for a button to go down, if requested */
	keepon = 0;
	if (waitforpush != 0)
	{
		while (keepon == 0)
		{
			gra_waitforaction(2, &theEvent);
			if (gra_inputstate == NOEVENT) continue;

			/* if button just went down, stop this loop */
			if ((gra_inputstate&ISBUTTON) != 0 && (gra_inputstate&BUTTONUP) == 0) break;

			if ((gra_inputstate&MOTION) != 0)
			{
				keepon = (*whileup)(gra_cursorx, gra_cursory);
			} else if ((gra_inputstate&CHARREAD) != 0)
			{
				keepon = (*eachchar)(gra_cursorx, gra_cursory, gra_inputstate&CHARREAD);
			}
			if (el_pleasestop != 0) keepon = 1;
		}
	}

	/* button is now down, real tracking begins */
	if (keepon == 0)
	{
		(*whendown)();
		keepon = (*eachdown)(gra_cursorx, gra_cursory);
	}

	/* now track while the button is down */
	while (keepon == 0)
	{
		gra_waitforaction(2, &theEvent);

		/* for each motion, report the coordinates */
		if (gra_inputstate == NOEVENT) continue;

		if ((gra_inputstate&ISBUTTON) != 0 && (gra_inputstate&BUTTONUP) != 0) break;
		if ((gra_inputstate&MOTION) != 0)
		{
			keepon = (*eachdown)(gra_cursorx, gra_cursory);
		} else if ((gra_inputstate&CHARREAD) != 0)
		{
			keepon = (*eachchar)(gra_cursorx, gra_cursory, gra_inputstate&CHARREAD);
		}
		if (el_pleasestop != 0) keepon = 1;
	}
	gra_inputstate = NOEVENT;

	/* inform the user that all is done */
	(*done)();

	/* restore the state of the world */
	if (purpose != TRACKHSELECTING) gra_set_cursorstate(NULLCURSOR);
}

/*
 * routine to read the current co-ordinates of the tablet and return them
 * in "*x" and "*y".
 */
void readtablet(INTSML *x, INTSML *y)
{
	*x = gra_cursorx;   *y = gra_cursory;
}

/*
 * routine to turn off the cursor tracking if it is on
 */
void stoptablet(void)
{
	if (gra_cursorstate != IBEAMCURSOR) gra_set_cursorstate(NULLCURSOR);
}

/******************** KEYBOARD CONTROL ********************/

/*
 * routine to get the next character from the keyboard
 */
INTSML us_getnxtchar(void)
{
	REGISTER INTSML i;
	EventRecord theEvent;

	if (gra_inputstate != NOEVENT && (gra_inputstate&CHARREAD) != 0)
	{
		i = gra_inputstate & CHARREAD;
		if ((gra_inputstate&COMMANDDOWN) != 0) i |= 0200;
		gra_inputstate = NOEVENT;
		return(i);
	}
	if (gra_cursorstate != IBEAMCURSOR) gra_set_cursorstate(WANTTTYCURSOR);
	for(;;)
	{
		gra_waitforaction(0, &theEvent);
		if (gra_inputstate != NOEVENT && (gra_inputstate&CHARREAD) != 0) break;
	}
	i = gra_inputstate & CHARREAD;
	if ((gra_inputstate&COMMANDDOWN) != 0) i |= 0200;
	gra_inputstate = NOEVENT;
	if (gra_cursorstate != IBEAMCURSOR) gra_set_cursorstate(NULLCURSOR);
	return(i);
}

void checkforinterrupt(void)
{
	EventRecord theEvent;
	short oak;
	extern INTSML gra_cancheck;

	if (el_pleasestop == 0 && gra_cancheck != 0)
	{
		gra_cancheck = 0;
		oak = EventAvail(keyDownMask, &theEvent);
		if (oak != 0)
		{
			(void)GetNextEvent(keyDownMask, &theEvent);
			if (theEvent.what == keyDown)
				if ((theEvent.modifiers & cmdKey) != 0)
					if ((theEvent.message & charCodeMask) == '.')
			{
				ttyputmsg("Interrupted...");
				el_pleasestop = 1;
			}
		}
	}
}

/*
 * routine to tell whether data is waiting at the terminal.  Returns nonzero
 * if data is ready.
 */
INTSML ttydataready(void)
{
	EventRecord theEvent;

	/* see if something is already pending */
	if (gra_inputstate != NOEVENT)
	{
		if ((gra_inputstate&CHARREAD) != 0) return(1);
		return(0);
	}

	/* wait for something and analyze it */
	gra_waitforaction(1, &theEvent);
	if (gra_inputstate != NOEVENT && (gra_inputstate&CHARREAD) != 0) return(1);
	return(0);
}

/****************************** FILES ******************************/

/*
 * Routine to set the type and creator of file "name" to "type" and "creator"
 */
void mac_settypecreator(char *name, INTBIG type, INTBIG creator)
{
	FInfo finfo;
	char pname[256];

	(void)strcpy(&pname[1], name);
	pname[0] = strlen(name);
	(void)GetFInfo((unsigned char *)pname, 0, &finfo);
	finfo.fdType = type;
	finfo.fdCreator = creator;
	(void)SetFInfo((unsigned char *)pname, 0, &finfo);
}

/*
 * Macintosh routine to return the name of the system folder
 */
char *mac_systemfoldername(void)
{
	static char foldername[256];
	char thisname[256];
	INTSML i, j, len;
	WDPBRec wpb;
	CInfoPBRec cpb;
	SysEnvRec sysenv;

	/* get system folder reference number */
	SysEnvirons(2, &sysenv);

	/* determine directory ID of system folder */
	wpb.ioNamePtr = (unsigned char *)foldername;
	wpb.ioVRefNum = sysenv.sysVRefNum;
	wpb.ioWDIndex = 0;
	wpb.ioWDProcID = 0L;
	wpb.ioWDVRefNum = wpb.ioVRefNum;
	i = PBGetWDInfo(&wpb, 0);
	if (i != noErr) return("");

	/* find the name of the system folder */
	foldername[0] = 0;
	cpb.dirInfo.ioDrParID = wpb.ioWDDirID;
	for(j=0; ; j++)
	{
		cpb.dirInfo.ioNamePtr = (unsigned char *)thisname;
		cpb.dirInfo.ioVRefNum = wpb.ioWDVRefNum;
		cpb.dirInfo.ioFDirIndex = -1;
		cpb.dirInfo.ioDrDirID = cpb.dirInfo.ioDrParID;
		i = PBGetCatInfo(&cpb, 0);
		if (i != noErr) return("");
		len = thisname[0] + 1;
		if (j != 0) thisname[len++] = ':';
		for(i=0; i<foldername[0]; i++) thisname[len++] = foldername[i+1];
		thisname[len] = 0;
		for(i=0; i<len; i++) foldername[i+1] = thisname[i+1];
		foldername[0] = len-1;
		if (cpb.dirInfo.ioDrDirID == fsRtDirID) break;
	}

	/* see if there is a folder with the name ":Preferences:Electric Files" added on */
	for(i=0; i<foldername[0]; i++) thisname[i+1] = foldername[i+1];
	thisname[foldername[0]+1] = 0;
	(void)strcat(&thisname[1], ":Preferences:Electric Files");
	thisname[0] = strlen(&thisname[1]);
	cpb.dirInfo.ioNamePtr = (unsigned char *)thisname;
	cpb.dirInfo.ioVRefNum = wpb.ioWDVRefNum;
	cpb.dirInfo.ioFDirIndex = 0;
	i = PBGetCatInfo(&cpb, 0);
	if (i != noErr) return(&foldername[1]);
	if ((cpb.dirInfo.ioFlAttrib&16) == 0) return(&foldername[1]);
	(void)strcpy(foldername, &thisname[1]);
	return(foldername);
}

/*
 * Routine to convert a Pascal string file name in "thisname" and its volume
 * reference number in "refnum" into a full path name (and a C string).
 */
char *mac_makefullname(char *thisname, INTSML refnum)
{
	INTSML err, len, i;
	CInfoPBRec cpb;
	char line[256];
	static char sofar[256];

	len = thisname[0];
	for(i=0; i<len; i++) sofar[i] = thisname[i+1];
	sofar[len] = 0;
	cpb.hFileInfo.ioVRefNum = refnum;
	cpb.hFileInfo.ioDirID = 0;
	cpb.hFileInfo.ioCompletion = 0L;
	cpb.hFileInfo.ioNamePtr = (StringPtr)line;
	cpb.hFileInfo.ioFDirIndex = -1;
	for(;;)
	{
		err = PBGetCatInfo(&cpb, 0);
		if (err != noErr) break;
		line[line[0]+1] = 0;
		strcat(line, ":");
		strcat(line, sofar);
		strcpy(sofar, &line[1]);
		if (cpb.hFileInfo.ioFlParID == 0) break;
		cpb.hFileInfo.ioDirID = cpb.hFileInfo.ioFlParID;
	}
	return(sofar);
}

/*
 * Routine to convert a FSS specification in "theFSS" into a full path name
 * (and a C string).
 */
char *gra_makefullnamefromFSS(FSSpec theFSS)
{
	DirInfo block;
	short len;
	static char fileName[256];
	char dirName[256];
	OSErr err;

	if (theFSS.parID != 0)
	{
		theFSS.name[theFSS.name[0]+1] = 0;
		strcpy(fileName, (char *)&theFSS.name[1]);
		block.ioDrParID = theFSS.parID;
		block.ioNamePtr = (StringPtr)dirName;
		do {
			block.ioVRefNum = theFSS.vRefNum;
			block.ioFDirIndex = -1;
			block.ioDrDirID = block.ioDrParID;
			err = PBGetCatInfo((CInfoPBPtr)&block, 0);
			dirName[dirName[0]+1] = 0;
			len = strlen(&dirName[1]);
			BlockMove(fileName, fileName + len+1, strlen(fileName)+1);
			strcpy(fileName, &dirName[1]);
			fileName[len] = ':';
		} while (block.ioDrDirID != 2);
		return(fileName);
	}

	/* no directory ID specified in FSS, use old name/vrefnum method */
	return(mac_makefullname((char *)theFSS.name, theFSS.vRefNum));
}

/*
 * File filter for binary library files.  Accepts "Elec" type files or those
 * that end with ".elib".
 */
pascal Boolean gra_fileFilterProc(CInfoPBPtr pb, Ptr mydata)
{
	INTSML pos;
	unsigned char *str;

	if ((pb->hFileInfo.ioFlAttrib&ioDirMask) != 0) return(0);
	str = pb->hFileInfo.ioNamePtr;
	pos = str[0];
	if (str[pos-4] == '.' && str[pos-3] == 'e' && str[pos-2] == 'l' && str[pos-1] == 'i' &&
		str[pos] == 'b') return(0);
	if (pb->hFileInfo.ioFlFndrInfo.fdType == 'Elec') return(0);
	return(1);
}

/*
 * Routine to display a standard file prompt dialog and return the selected file.
 * The prompt message is in "msg" and the kind of file is in "filetype".  The default
 * output file name is in "defofile" (only used if "filetype" is negative).
 */
char *gra_fileselect(char *msg, INTSML filetype, char *defofile)
{
	SFTypeList myTypes;
	static Point SFwhere = {90, 82};
	INTBIG len;
	REGISTER INTSML i, save;
	StandardFileReply reply;
	WindowPtr savewin;
	WDPBRec wdpb;
	char leng, ofile[256], prompt[256];

	if (us_logplay != NULL)
	{
		(void)xfread((char *)&gra_action, 1, sizeof (gra_action), us_logplay);
		if ((gra_action.kind&0xFFFF) != FILEREPLY) gra_localstring[0] = 0; else
		{
			(void)xfread(&leng, 1, 1, us_logplay);
			len = leng;
			(void)xfread(gra_localstring, 1, len, us_logplay);
			gra_localstring[len] = 0;
		}
	} else
	{
		strcpy(&prompt[1], msg);
		prompt[0] = strlen(msg);
		GetPort(&savewin);
		if ((filetype&FILETYPEWRITE) == 0)
		{
			/* input file selection */
			ParamText((unsigned char *)prompt, "\p", "\p", "\p");
			if ((filetype&FILETYPE) == FILETYPEBLIB)
			{
				/* special case for Electric binary libraries */
				CustomGetFile(gra_fileFilterProcUPP, -1, myTypes, &reply, FILEINDIALOG, SFwhere,
					0, 0, 0, 0, 0);
			} else
			{
				/* standard file input */
				CustomGetFile(0, -1, myTypes, &reply, FILEINDIALOG, SFwhere, 0, 0, 0, 0, 0);
			}
		} else
		{
			/* output file selection */
			for(i = strlen(defofile)-1; i > 0; i--)
				if (defofile[i] == ':') break;
			if (i > 0)
			{
				/* there is a ":" in the path, set the default directory */
				i++;
				save = defofile[i];
				defofile[i] = 0;
				(void)strcpy(&ofile[1], defofile);
				ofile[0] = strlen(defofile);
				wdpb.ioNamePtr = (StringPtr)ofile;
				PBHSetVol(&wdpb, 0);
				defofile[i] = save;
				defofile = &defofile[i];
			}
			(void)strcpy(&ofile[1], defofile);
			ofile[0] = strlen(defofile);
			StandardPutFile((unsigned char *)prompt, (unsigned char *)ofile, &reply);
		}
		SetPort(savewin);
		if (reply.sfGood == 0) gra_localstring[0] = 0; else
			(void)strcpy(gra_localstring, gra_makefullnamefromFSS(reply.sfFile));
	}

	/* log this result if logging */
	if (us_logrecord != NULL)
	{
		gra_action.kind = FILEREPLY;
		(void)xfwrite((char *)&gra_action, 1, sizeof (gra_action), us_logrecord);
		leng = strlen(gra_localstring);
		(void)xfwrite(&leng, 1, 1, us_logrecord);
		len = leng;
		(void)xfwrite(gra_localstring, 1, len, us_logrecord);
		gra_logrecordindex++;
		if (gra_logrecordindex >= us_logflushfreq)
		{
			/* flush the session log file */
			gra_logrecordindex = 0;
			xflushbuf(us_logrecord);
		}
	}
	return(gra_localstring);
}

/*
 * routine to handle an "open document" Apple Event and read a library
 */
pascal OSErr gra_handleodoc(AppleEvent *theAppleEvent, AppleEvent *reply, long handlerRefCon)
{
	FSSpec myFSS;
	AEDescList docList;
	OSErr err;
	char *argv[3];
	long index, itemsInList;
	Size actualSize;
	AEKeyword keywd;
	DescType returnedType;

	/* get the direct parameter, a descriptor list, and put it into a doclist */
	err = AEGetParamDesc(theAppleEvent, keyDirectObject, typeAEList, &docList);
	if (err) return(err);

	/* check for missing parameters */
	err = AEGetAttributePtr(theAppleEvent, keyMissedKeywordAttr, typeWildCard,
		&returnedType, nil, 0, &actualSize);
	if (err != errAEDescNotFound)
	{
		if (!err) return(errAEEventNotHandled);
		return(err);
	}

	/* count the number of descriptor records in the list */
	err = AECountItems(&docList, &itemsInList);

	/*
	 * now get each descriptor record from the list, coerce the returned data to
	 * an FSSpec record, and open the associated file
	 */
	for(index = 1; index <= itemsInList; index++)
	{
		err = AEGetNthPtr(&docList, index, typeFSS, &keywd, &returnedType,
			(Ptr)&myFSS, sizeof(myFSS), &actualSize);
		if (err) return(err);
		us_beginchanges();
		argv[0] = "read";
		argv[1] = gra_makefullnamefromFSS(myFSS);
		argv[2] = "make-current";
		us_library(3, argv);
		us_endchanges(NOWINDOW);
	}
	err = AEDisposeDesc(&docList);

	return(noErr);
}

/*
 * Helper routine to initialize the list of files in a directory.
 */
void gra_initfilelist(void)
{
	if (gra_fileliststringarray == 0)
	{
		gra_fileliststringarray = newstringarray(db_cluster);
		if (gra_fileliststringarray == 0) return;
	}
	clearstrings(gra_fileliststringarray);
}

/*
 * Helper routine to add "file" to the list of files in a directory.
 * Returns nonzero on error.
 */
INTSML gra_addfiletolist(char *file)
{
	addtostringarray(gra_fileliststringarray, file);
	return(0);
}

/*
 * Routine to search for all of the files/directories in directory "directory" and
 * return them in the array of strings "filelist".  Returns the number of files found.
 */
INTSML filesindirectory(char *directory, char ***filelist)
{
	INTSML err, i;
	INTBIG dirid, len;
	char file[256];
	CInfoPBRec cinfo;

	if (*directory == 0) dirid = 0; else
	{
		(void)strcpy(&file[1], directory);
		file[0] = strlen(directory);
		SetVol((unsigned char *)file, 0);
		cinfo.hFileInfo.ioCompletion = 0L;
		cinfo.hFileInfo.ioNamePtr = (StringPtr)file;
		cinfo.hFileInfo.ioVRefNum = 0;
		cinfo.hFileInfo.ioFDirIndex = 0;
		cinfo.hFileInfo.ioDirID = 0L;
		err = PBGetCatInfo(&cinfo, 0);
		if (err != noErr) return(0);
		dirid = cinfo.hFileInfo.ioDirID;
	}

	gra_initfilelist();
	for(i=1; ; i++)
	{
		cinfo.hFileInfo.ioCompletion = 0L;
		cinfo.hFileInfo.ioNamePtr = (StringPtr)file;
		cinfo.hFileInfo.ioVRefNum = 0;
		cinfo.hFileInfo.ioFDirIndex = i;
		cinfo.hFileInfo.ioDirID = dirid;
		err = PBGetCatInfo(&cinfo, 0);
		if (err != noErr) break;
		file[file[0]+1] = 0;
		if (gra_addfiletolist(&file[1]) != 0) return(0);
	}

	*filelist = getstringarray(gra_fileliststringarray, &len);
	return(len);
}

/* routine to convert a path name with "~" to a real path */
char *truepath(char *line)
{
	/* only have tilde parsing on UNIX */
	return(line);
}

/*
 * Routine to return information about the file or directory "name":
 *  0: does not exist
 *  1: is a file
 *  2: is a directory
 */
INTSML fileexistence(char *name)
{
	struct stat buf;

	if (stat(name, &buf) < 0) return(0);
	if ((buf.st_mode & S_IFMT) == S_IFDIR) return(2);
	return(1);
}

/*
 * Routine to create a directory.
 * Returns nonzero on error.
 */
INTSML createdirectory(char *dirname)
{
	FSSpec fsp;
	long dir;
	INTSML err;
	char pname[256];

	strcpy(&pname[1], dirname);
	pname[0] = strlen(dirname);
	err = FSMakeFSSpec(0, 0, (unsigned char *)pname, &fsp);
	if (err != noErr && err != fnfErr) return(1);
	err = FSpDirCreate(&fsp, smSystemScript, &dir);
	if (err != noErr && err != dupFNErr) return(1);
	return(0);
}

/*
 * Routine to return the current directory name
 */
char *currentdirectory(void)
{
	return(mac_makefullname((char *)"\p", 0));
}

/*
 * routine to return the date of the last modification to file "filename"
 */
UINTBIG filedate(char *filename)
{
	struct stat buf;

	stat(filename, &buf);
	buf.st_mtime -= gra_machinetimeoffset();
	return(buf.st_mtime);
}

/*
 * Routine to lock a resource called "lockfilename" by creating such a file
 * if it doesn't exist.  Returns nonzero if successful, zero if unable to
 * lock the file.
 */
INTSML lockfile(char *lockfilename)
{
	INTSML err;
	FSSpec fsp;
	char pname[256];

	strcpy(&pname[1], lockfilename);
	pname[0] = strlen(lockfilename);
	err = FSMakeFSSpec(0, 0, (unsigned char *)pname, &fsp);
	err = FSpCreate(&fsp, 'Elec', 'Lock', smSystemScript);
	if (err == noErr) return(1);
	return(0);
}

/*
 * Routine to unlock a resource called "lockfilename" by deleting such a file.
 */
void unlockfile(char *lockfilename)
{
	FSSpec fsp;
	char pname[256];

	strcpy(&pname[1], lockfilename);
	pname[0] = strlen(lockfilename);
	FSMakeFSSpec(0, 0, (unsigned char *)pname, &fsp);
	FSpDelete(&fsp);
}

/*************************** TIME ROUTINES ***************************/

/*
 * The Macintosh uses an "epoch" of January 1, 1904.
 * All other machines use an "epoch" of January 1, 1970.
 * This means that time on the Macintosh is off by 66 years.
 * Specifically, there were 17 leap years between 1904 and 1970, so
 * there were  66*365+17 = 24107 days
 * Since each day has 24*60*60 = 86400 seconds in it, there were
 * 24107 * 86400 = 2,082,844,800 seconds difference between epochs.
 *
 * However, since Macintosh systems deal with local time and other
 * systems deal with GMT time, the Mac must also offset by the time zone.
 */
#define	MACEPOCHOFFSET 2082844800

/*
 * This routine returns the amount to add to the operating-system time
 * to adjust for a common time format.
 */
UINTBIG gra_machinetimeoffset(void)
{
	static UINTBIG offset;
	static INTSML offsetcomputed = 0;
	UINTBIG timezoneoffset;
	MachineLocation ml;

	if (offsetcomputed == 0)
	{
		offsetcomputed = 1;
		offset = MACEPOCHOFFSET;
		ReadLocation(&ml);
		timezoneoffset = ml.u.gmtDelta & 0xFFFFFF;
		if ((timezoneoffset & 0x800000) != 0)
			timezoneoffset = -(((~timezoneoffset) & 0xFFFFFF) + 1);
		offset += timezoneoffset;
	}
	return(offset);
}

UINTBIG ticktime(void)
{
	return(TickCount());
}

/*
 * Routine to start counting time.
 */
void starttimer(void)
{
	gra_timestart = clock();
}

/*
 * Routine to stop counting time and return the number of elapsed seconds
 * since the last call to "starttimer()".
 */
float endtimer(void)
{
	float seconds;
	UINTBIG thistime;

	thistime = clock();
	seconds = ((float)(thistime - gra_timestart)) / 60.0;
	return(seconds);
}

/*
 * Routine to wait "ticks" sixtieths of a second and then return.
 */
void gotosleep(INTBIG ticks)
{
	long l;
	Delay(ticks, (long *)&l);
}

/*************************** EVENT ROUTINES ***************************/

/*
 * Hack routine that writes a fake "mouse up" event to the session logging file.  This
 * is necessary because certain toolbox routines (i.e. TEClick) track the
 * mouse while it is down in order to handle selection of text.  Since this tracking
 * cannot be obtained and logged, a fake "SHIFT mouse" is logged after the routine finishes
 * so that the text selection is effected in the same way
 */
void gra_fakemouseup(void)
{
	Point p;

	if (us_logrecord == NULL) return;
	GetMouse(&p);
	gra_action.kind = ISBUTTON | SHIFTDOWN;
	gra_action.x = p.h;
	gra_action.y = p.v;
	if (xfwrite((char *)&gra_action, 1, sizeof (gra_action), us_logrecord) == 0)
	{
		ttyputerr("Error writing session log file: recording disabled");
		us_logfinishrecord();
	}
	gra_logrecordindex++;
	if (gra_logrecordindex >= us_logflushfreq)
	{
		/* flush the session log file */
		gra_logrecordindex = 0;
		xflushbuf(us_logrecord);
	}
}

/*
 * helper routine to wait for some keyboard or mouse input.  The value of "nature" is:
 *   0  allow mouse, keyboard
 *   1  allow mouse, keyboard, pulldown menus
 *   2  allow mouse, keyboard, pulldown menus, motion
 */
void gra_waitforaction(INTSML nature, EventRecord *theEvent)
{
	REGISTER INTBIG err;
	static INTBIG last_cursorx, last_cursory;
	REGISTER INTSML saveevent;
	static INTBIG fakewhen = 0;
	Rect r, fr;

	if (us_logplay != NULL)
	{
		if (EventAvail(updateMask | activMask, theEvent) != 0)
		{
			gra_nextevent(4, theEvent);
			return;
		}
		flushscreen();
		err = xfread((char *)&gra_action, 1, sizeof (gra_action), us_logplay);
		if (stopping("Playback") != 0) err = 0;
		if (err != 0)
		{
			if (gra_playbackmultiple <= 0)
			{
				gra_playbackmultiple = 0;
				for(;;)
				{
					gra_inputstate = NOEVENT;
					gra_nextevent(0, theEvent);
					if (gra_inputstate == NOEVENT) continue;
					if ((gra_inputstate&CHARREAD) == 0) continue;
					if ((gra_inputstate&CHARREAD) < '0' || (gra_inputstate&CHARREAD) > '9') break;
					gra_playbackmultiple = gra_playbackmultiple * 10 + (gra_inputstate&CHARREAD) - '0';
				}
				if (gra_inputstate != NOEVENT && (gra_inputstate&CHARREAD) == 'q')
					err = 0;
			}
			gra_playbackmultiple--;

			/* allow Command-. to interrupt long playbacks */
			if ((gra_playbackmultiple%10) == 9)
			{
				if (EventAvail(keyDownMask, theEvent) != 0)
				{
					(void)WaitNextEvent(keyDownMask, theEvent, 0, 0);
					if ((theEvent->modifiers & cmdKey) != 0 &&
						(theEvent->message & charCodeMask) == '.')
							gra_playbackmultiple = 0;
				}
			}

			gra_inputstate = gra_action.kind & 0xFFFF;
			gra_cursorx = gra_action.x;
			gra_cursory = gra_action.y;
			if (gra_inputstate == MENUEVENT)
			{
				(void)xfread((char *)&gra_action, 1, sizeof (gra_action), us_logplay);
				gra_lowmenu = gra_action.x;
				gra_highmenu = gra_action.y;
			} else if (gra_inputstate == WINDOWCHANGE)
			{
				(void)xfread((char *)&r, 1, sizeof (r), us_logplay);
				fr = (*((WindowPeek)gra_curwindowframe->realwindow)->strucRgn)->rgnBBox;
				r.left += 1;         r.right -= 2;
				r.top += MENUSIZE;   r.bottom -= 2;
				MoveWindow((WindowPtr)gra_curwindowframe->realwindow, r.left, r.top, 0);
				SizeWindow((WindowPtr)gra_curwindowframe->realwindow, r.right-r.left,
					r.bottom-r.top, 1);
				gra_mygrowwindow((WindowPtr)gra_curwindowframe->realwindow, r.right-r.left,
					r.bottom-r.top, &fr);
			}
		}

		/* convert to an event */
		fakewhen += gra_doubleclick + 1;
		theEvent->what = nullEvent;
		theEvent->when = fakewhen;
		theEvent->modifiers = 0;
		if ((gra_inputstate&SHIFTDOWN) != 0) theEvent->modifiers |= shiftKey;
		if ((gra_inputstate&COMMANDDOWN) != 0) theEvent->modifiers |= cmdKey;
		if ((gra_inputstate&OPTIONDOWN) != 0) theEvent->modifiers |= optionKey;
		if ((gra_inputstate&CONTROLDOWN) != 0) theEvent->modifiers |= controlKey;
		if ((gra_inputstate&BUTTONUP) != 0) theEvent->modifiers |= btnState;
		theEvent->where.h = gra_cursorx;
		theEvent->where.v = gra_curwindowframe->revy - gra_cursory;
		if ((gra_inputstate&CHARREAD) != 0)
		{
			theEvent->what = keyDown;
			theEvent->message = gra_inputstate & CHARREAD;
		} else if ((gra_inputstate&ISBUTTON) != 0)
		{
			if ((gra_inputstate&BUTTONUP) == 0) theEvent->what = mouseDown; else
				theEvent->what = mouseUp;
			if ((gra_inputstate&DOUBLECLICK) != 0)
			{
				theEvent->when--;
				fakewhen--;
			}
		}

		/* stop now if end of playback file */
		if (err == 0)
		{
			ttyputmsg("End of session playback file");
			xclose(us_logplay);
			us_logplay = NULL;
			return;
		}
	} else
	{
		flushscreen();
		gra_nextevent(nature, theEvent);
	}

	if (us_logrecord != NULL && gra_inputstate != NOEVENT)
	{
		saveevent = 1;
		if ((gra_inputstate&MOTION) != 0)
		{
			if (gra_cursorx == last_cursorx && gra_cursory == last_cursory) saveevent = 0;
		}
		if (saveevent != 0)
		{
			gra_action.kind = gra_inputstate;
			gra_action.x = last_cursorx = gra_cursorx;
			gra_action.y = last_cursory = gra_cursory;
			if (xfwrite((char *)&gra_action, 1, sizeof (gra_action), us_logrecord) == 0)
			{
				ttyputerr("Error writing session log file: recording disabled");
				us_logfinishrecord();
			}
			if (gra_inputstate == MENUEVENT)
			{
				gra_action.x = gra_lowmenu;
				gra_action.y = gra_highmenu;
				(void)xfwrite((char *)&gra_action, 1, sizeof (gra_action), us_logrecord);
			} else if (gra_inputstate == WINDOWCHANGE)
			{
				if (gra_curwindowframe != NOWINDOWFRAME)
				{
					r = (*((WindowPeek)gra_curwindowframe->realwindow)->strucRgn)->rgnBBox;
					(void)xfwrite((char *)&r, 1, sizeof (r), us_logrecord);
				}
			}
			gra_logrecordindex++;
			if (gra_logrecordindex >= us_logflushfreq)
			{
				/* flush the session log file */
				gra_logrecordindex = 0;
				xflushbuf(us_logrecord);
			}
		}
	}

	/* deal with special event types */
	if (gra_inputstate == MENUEVENT)
	{
		if (gra_highmenu == appleMENU && gra_lowmenu == 1)
		{
			gra_applemenu(gra_lowmenu);
		} else gra_nativemenudoone(gra_lowmenu, gra_highmenu);
		gra_inputstate = NOEVENT;
	} else if (gra_inputstate == WINDOWCHANGE) gra_inputstate = NOEVENT;
}

/*
 * Routine to get the next Electric input action and set the global "gra_inputstate"
 * accordingly.  The value of "nature" is:
 *   0  allow mouse, keyboard (no window switching)
 *   1  allow mouse, keyboard, pulldown menus
 *   2  allow mouse, keyboard, motion
 *   4  allow update and activate events only
 */
void gra_nextevent(INTSML nature, EventRecord *theEvent)
{
#ifdef	USETK
	if (gra_messageswindow != 0) TEIdle(gra_TEH);
	(void)Tcl_DoOneEvent(TCL_DONT_WAIT);
	if (gra_eventqueuehead != gra_eventqueuetail)
	{
		gra_inputstate = gra_eventqueuehead->inputstate;
		gra_cursorx = gra_eventqueuehead->cursorx;
		gra_cursory = gra_eventqueuehead->cursory;
		gra_eventqueuehead++;
		if (gra_eventqueuehead >= &gra_eventqueue[EVENTQUEUESIZE])
			gra_eventqueuehead = gra_eventqueue;
	}
#else
	INTSML oak, cntlCode, x, y, stroke, oldpos, findres;
	INTBIG key, theResult, xv, yv;
	WStateData *wst;
	REGISTER WINDOW *w;
	REGISTER EDITOR *e;
	char *par[1];
	static INTSML firstwupdate = 1, firstsupdate = 1;
	WindowPtr theWindow, win, frontWindow;
	Rect r, fr;
	ControlHandle theControl;
	WINDOWFRAME *mw;

	gra_inputstate = NOEVENT;
	HiliteMenu(0);
	if (gra_messageswindow != 0) TEIdle(gra_TEH);
	if (nature == 4) oak = WaitNextEvent(updateMask | activMask, theEvent, 0, 0); else
		oak = WaitNextEvent(everyEvent, theEvent, 0, 0);
	if (oak == 0 && nature != 4 && gra_motioncheck != 0)
	{
		gra_motioncheck = 0;
		oak = theEvent->what = app3Evt;
		if (Button())
		{
			theEvent->modifiers |= btnState;
		} else
		{
			theEvent->modifiers &= ~btnState;
		}
	}
	if (oak == 0)
	{
		if (nature != 2)
		{
			if (FindWindow(theEvent->where, &theWindow) != inContent)
				gra_set_cursorstate(NORMALCURSOR);
		}
		return;
	}
	if (oak != 0) switch (theEvent->what)
	{
		case mouseUp:
			if (gra_curwindowframe != NOWINDOWFRAME)
			{
				SetPort((WindowPtr)gra_curwindowframe->realwindow);
				GlobalToLocal(&theEvent->where);
				gra_cursorx = theEvent->where.h;
				gra_cursory = gra_curwindowframe->revy - theEvent->where.v;
			}
			gra_inputstate = ISBUTTON | BUTTONUP;
			us_state |= DIDINPUT;
			break;
		case mouseDown:
			switch (FindWindow(theEvent->where, &theWindow))
			{
				case inMenuBar:
					if (nature != 1) break;
					key = MenuSelect(theEvent->where);
					if (HiWord(key) == appleMENU && LoWord(key) != 1)
					{
						gra_applemenu(LoWord(key));
						return;
					}
					gra_highmenu = HiWord(key);
					gra_lowmenu = LoWord(key);
					gra_inputstate = MENUEVENT;
					return;
				case inSysWindow:
					SystemClick(theEvent, theWindow);
					break;
				case inContent:
					SetPort(theWindow);
					GlobalToLocal(&theEvent->where);

					/* determine which editor window was hit */
					for(mw = gra_firstwindowframe; mw != NOWINDOWFRAME; mw = mw->nextwindowframe)
						if ((WindowPtr)mw->realwindow == theWindow) break;

					/* handle clicks in title bar of floating windows */
					if (mw != NOWINDOWFRAME && mw->floating != 0 && theEvent->where.v < FLOATINGHEADERSIZE)
					{
						if (theEvent->where.v >= 2 && theEvent->where.v <= 8 &&
							theEvent->where.h >= 4 && theEvent->where.h <= 10)
						{
							/* click in close box: turn off floating window */
							par[0] = "off";
							us_menu(1, par);
						} else
						{
							/* click in drag bar: move it */
							LocalToGlobal(&theEvent->where);
							gra_dragfloatingwindow(theWindow, theEvent->where);
						}
						break;
					}

					gra_frontnonfloatingwindow(&frontWindow);
					if (theWindow != frontWindow)
					{
						/* click in new window: see if it is allowed */
						if (nature == 0)
						{
							ttybeep();
							break;
						}

						/* select this one as the frontmost window */
						gra_selectoswindow(theWindow);
						gra_setcurrentwindowframe();

						/* when reentering edit window, force proper cursor */
						if (mw == NOWINDOWFRAME) break;
						oak = gra_cursorstate;
						gra_cursorstate++;
						gra_set_cursorstate(oak);
					}

					/* special case to ensure that "gra_curwindowframe" gets floating window */
					if (gra_curwindowframe != NOWINDOWFRAME && gra_curwindowframe->floating != 0)
						gra_setcurrentwindowframe();
					if (mw != NOWINDOWFRAME && mw->floating != 0) gra_curwindowframe = mw;

					if (theWindow == gra_messageswindow && gra_messageswindow != 0)
					{
						cntlCode = FindControl(theEvent->where, theWindow, &theControl);
						if (cntlCode == inThumb)
						{
							oldpos = GetCtlValue(theControl);
							TrackControl(theControl, theEvent->where, 0L);
							if (theControl == gra_vScroll) gra_adjustvtext(); else
								gra_adjusthtext(oldpos);
							break;
						}
						if (cntlCode == inUpButton || cntlCode == inDownButton ||
							cntlCode == inPageUp || cntlCode == inPageDown)
						{
							if (theControl == gra_vScroll)
								TrackControl(theControl, theEvent->where, gra_scrollvprocUPP); else
									TrackControl(theControl, theEvent->where, gra_scrollhprocUPP);
							break;
						}
						TEClick(theEvent->where, (theEvent->modifiers&shiftKey) != 0, gra_TEH);
						gra_fakemouseup();
						break;
					}

					/* ignore clicks in the messages area at the bottom */
					if (mw != NOWINDOWFRAME && mw->floating == 0 &&
						theEvent->where.v >= theWindow->portRect.bottom - SBARWIDTH) break;

					/* create the "click" event */
					gra_inputstate = ISBUTTON;
					if ((theEvent->modifiers&shiftKey) != 0) gra_inputstate |= SHIFTDOWN;
					if ((theEvent->modifiers&cmdKey) != 0) gra_inputstate |= COMMANDDOWN;
					if ((theEvent->modifiers&optionKey) != 0) gra_inputstate |= OPTIONDOWN;
					if ((theEvent->modifiers&controlKey) != 0) gra_inputstate |= CONTROLDOWN;
					gra_cursorx = theEvent->where.h;
					if (mw == NOWINDOWFRAME) gra_cursory = theEvent->where.v; else
					{
						gra_cursory = mw->revy - theEvent->where.v;
						if (mw->floating != 0) gra_cursory += FLOATINGHEADERSIZE;
					}
					if (theEvent->when - gra_doubleclick < gra_lastclick &&
						(gra_inputstate & (SHIFTDOWN|COMMANDDOWN|OPTIONDOWN|CONTROLDOWN)) == 0 &&
						abs(gra_cursorx-gra_lstcurx) < 5 && abs(gra_cursory-gra_lstcury) < 5)
					{
						gra_inputstate |= DOUBLECLICK;
						gra_lastclick = theEvent->when - gra_doubleclick - 1;
					} else gra_lastclick = theEvent->when;
					gra_lstcurx = gra_cursorx;   gra_lstcury = gra_cursory;
					us_state |= DIDINPUT;
					break;
				case inDrag:
					if (theWindow != FrontWindow() && nature == 0)
					{
						ttybeep();
						break;
					}
					gra_selectoswindow(theWindow);
					gra_setcurrentwindowframe();
					gra_dragfloatingwindow(theWindow, theEvent->where);
					break;
				case inGrow:
					SetPort(theWindow);
					SetRect(&r, 80, 80, qd.screenBits.bounds.right+500,
						qd.screenBits.bounds.bottom+500);
					fr = (*((WindowPeek)theWindow)->strucRgn)->rgnBBox;
					theResult = GrowWindow(theWindow, theEvent->where, &r);
					if (theResult != 0)
					{
						SizeWindow(theWindow, LoWord(theResult), HiWord(theResult), 1);
						gra_mygrowwindow(theWindow, LoWord(theResult), HiWord(theResult), &fr);
					}
					gra_inputstate = WINDOWCHANGE;
					break;
				case inZoomIn:
					if (TrackBox(theWindow, theEvent->where, inZoomIn) == 0) break;
					SetPort(theWindow);
					EraseRect(&theWindow->portRect);
					wst = (WStateData *) *(((WindowPeek)theWindow)->dataHandle);
					fr = (*((WindowPeek)theWindow)->strucRgn)->rgnBBox;
					MoveWindow(theWindow, wst->userState.left, wst->userState.top, 0);
					SizeWindow(theWindow, wst->userState.right-wst->userState.left,
						wst->userState.bottom-wst->userState.top, 1);
					gra_mygrowwindow(theWindow, wst->userState.right-wst->userState.left,
						wst->userState.bottom-wst->userState.top, &fr);
					gra_inputstate = WINDOWCHANGE;
					break;
				case inZoomOut:
					if (TrackBox(theWindow, theEvent->where, inZoomOut) == 0) break;
					SetPort(theWindow);
					EraseRect(&theWindow->portRect);
					wst = (WStateData *) *(((WindowPeek)theWindow)->dataHandle);
					fr = (*((WindowPeek)theWindow)->strucRgn)->rgnBBox;
					MoveWindow(theWindow, wst->stdState.left, wst->stdState.top, 0);
					SizeWindow(theWindow, wst->stdState.right-wst->stdState.left,
						wst->stdState.bottom-wst->stdState.top, 1);
					gra_mygrowwindow(theWindow, wst->stdState.right-wst->stdState.left,
						wst->stdState.bottom-wst->stdState.top, &fr);
					gra_inputstate = WINDOWCHANGE;
					break;
				case inGoAway:
					SetPort(theWindow);
					if (TrackGoAway(theWindow, theEvent->where) == 0) break;
					if (theWindow == gra_messageswindow && gra_messageswindow != 0)
					{
						gra_hidemessageswindow();
					} else if (gra_curwindowframe != NOWINDOWFRAME &&
						theWindow == (WindowPtr)gra_curwindowframe->realwindow)
					{
						par[0] = "delete";
						us_window(1, par);
					} else
					{
						/* determine which editor window was hit */
						for(mw = gra_firstwindowframe; mw != NOWINDOWFRAME; mw = mw->nextwindowframe)
							if ((WindowPtr)mw->realwindow == theWindow) break;
						if (mw != NOWINDOWFRAME) gra_disposeoswindow(theWindow, mw->tkwin); else
							gra_disposeoswindow(theWindow, 0);
					}
					break;
			}
			break;

		case keyDown:
		case autoKey:
			stroke = theEvent->message & charCodeMask;
			if (stroke == 0) break;
			if (gra_curwindowframe != NOWINDOWFRAME)
			{
				SetPort((WindowPtr)gra_curwindowframe->realwindow);
				GlobalToLocal(&theEvent->where);
				gra_cursorx = theEvent->where.h;
				gra_cursory = gra_curwindowframe->revy - theEvent->where.v;
			}
			if ((theEvent->modifiers & cmdKey) != 0 && stroke == '.')
			{
				el_pleasestop = 2;
				return;
			}
			if ((theEvent->modifiers & cmdKey) != 0 && nature == 1)
			{
				key = MenuKey((char)stroke);
				gra_highmenu = HiWord(key);
				gra_lowmenu = -LoWord(key);
				gra_inputstate = MENUEVENT;
				if (gra_highmenu != 0) return;
			}
			gra_frontnonfloatingwindow(&frontWindow);
			if (gra_messageswindow == frontWindow && gra_messageswindow != 0)
			{
				TESetSelect(32767, 32767, gra_TEH);
				TEKey(stroke, gra_TEH);
				gra_showselect();
				break;
			}
			gra_inputstate = stroke & CHARREAD;
			us_state |= DIDINPUT;
			break;

		case activateEvt:
			win = (WindowPtr)theEvent->message;

			/* check for an editor window frame */
			for(mw = gra_firstwindowframe; mw != NOWINDOWFRAME; mw = mw->nextwindowframe)
				if ((WindowPtr)mw->realwindow == win) break;

			if (mw != NOWINDOWFRAME)
			{
				if ((theEvent->modifiers&activeFlag) != 0)
				{
					SetPort((WindowPtr)mw->realwindow);
					r = (*mw->window->portPixMap)->bounds;
					InvalRect(&r);
				}
				break;
			}

			/* probably a dialog, just process it */
			SetPort(win);
			InvalRect(&win->portRect);
			break;

		case updateEvt:
			win = (WindowPtr)theEvent->message;
			SetPort(win);
			BeginUpdate(win);
			if (win == gra_messageswindow && gra_messageswindow != 0)
			{
				if (firstsupdate == 0)
				{
					EraseRect(&win->portRect);
					DrawControls(win);
					DrawGrowIcon(win);
					TEUpdate(&win->portRect, gra_TEH);
				}
				firstsupdate = 0;
			} else
			{
				for(mw = gra_firstwindowframe; mw != NOWINDOWFRAME; mw = mw->nextwindowframe)
				{
					if ((WindowPtr)mw->realwindow == win) break;
				}
				if (mw != NOWINDOWFRAME)
				{
					if (firstwupdate == 0)
					{
						gra_drawosgraphics(mw);
						if (mw->floating == 0)
							gra_setrect(mw, (*mw->window->portPixMap)->bounds.right-SBARWIDTH-1,
								(*mw->window->portPixMap)->bounds.right-SBARWIDTH+1,
									0, (*mw->window->portPixMap)->bounds.bottom);
						r = (*mw->realwindow->visRgn)->rgnBBox;
						if (mw->floating != 0)
						{
							r.top -= FLOATINGHEADERSIZE;
							r.bottom -= FLOATINGHEADERSIZE;
						}
						gra_setrect(mw, r.left, r.right, r.top, r.bottom);

						if (mw->floating == 0) us_redostatus((void *)mw);
					}
					firstwupdate = 0;
				}
			}

			/* probably a dialog, just process it */
			SetPort(win);
			DiaRedrawDialogWindow();
			EndUpdate(win);
			break;

		case app3Evt:
			/* see if this happened in an editor window */
			findres = FindWindow(theEvent->where, &theWindow);
			if (theWindow == 0) break;
			for(mw = gra_firstwindowframe; mw != NOWINDOWFRAME; mw = mw->nextwindowframe)
				if ((WindowPtr)mw->realwindow == theWindow) break;
			SetPort(theWindow);
			GlobalToLocal(&theEvent->where);

			if (nature == 2)
			{
				/* handle cursor motion */
				gra_cursorx = theEvent->where.h;
				if (mw == NOWINDOWFRAME) gra_cursory = theEvent->where.v; else
					gra_cursory = mw->revy - theEvent->where.v;
				gra_inputstate = MOTION;
				if ((theEvent->modifiers&btnState) != 0) gra_inputstate |= BUTTONUP;
				return;
			}

			/* checkout the cursor position */
			if (findres == inContent)
			{
				if (theWindow == gra_messageswindow && gra_messageswindow != 0)
				{
					if (theEvent->where.h > theWindow->portRect.right - SBARWIDTH ||
						theEvent->where.v > theWindow->portRect.bottom - SBARWIDTH)
							gra_set_cursorstate(NORMALCURSOR); else
								gra_set_cursorstate(IBEAMCURSOR);
					return;
				}
				if (mw == NOWINDOWFRAME || theWindow != (WindowPtr)mw->realwindow)
				{
					gra_set_cursorstate(NORMALCURSOR);
					return;
				}
				x = theEvent->where.h;
				if (mw == NOWINDOWFRAME) y = theEvent->where.v; else
					y = mw->revy - theEvent->where.v;
				for(w = el_topwindow; w != NOWINDOW; w = w->nextwindow)
				{
					if (w->frame != mw) continue;
					if (x < w->uselx || x > w->usehx || y < w->usely || y > w->usehy) continue;
					if ((w->state&WINDOWTYPE) == POPTEXTWINDOW ||
						(w->state&WINDOWTYPE) == TEXTWINDOW)
					{
						e = w->editor;
						if ((e->state&EDITORTYPE) == PACEDITOR)
						{
							if (x > w->usehx - SBARWIDTH || y < w->usely + SBARWIDTH || y >= e->revy)
								gra_set_cursorstate(NORMALCURSOR); else
									gra_set_cursorstate(IBEAMCURSOR);
							return;
						}
					} else if ((us_aid->aidstate&SHOWXY) != 0)
					{
						xv = x;   yv = y;
						xv = muldiv(xv - w->uselx, w->screenhx - w->screenlx,
							w->usehx - w->uselx) + w->screenlx;
						yv = muldiv(yv - w->usely, w->screenhy - w->screenly,
							w->usehy - w->usely) + w->screenly;
						gridalign(&xv, &yv, us_alignment);
						us_setcursorpos((void *)mw, xv, yv);
					}
				}
				gra_set_cursorstate(gra_normalcursor);
				return;
			}
			gra_set_cursorstate(NORMALCURSOR);
			return;

		case kHighLevelEvent:
			(void)AEProcessAppleEvent(theEvent);
			break;

		case osEvt:
			switch ((theEvent->message >> 24) & 0xFF)
			{
				case suspendResumeMessage:
					if ((theEvent->message&resumeFlag) == 0)
					{
						/* suspend the application */
						gra_frontnonfloatingwindow(&theWindow);
						if (theWindow != 0) gra_activatewindow(theWindow, 0);
						gra_showallfloaters(0);
					} else
					{
						/* resume the application */
						gra_frontnonfloatingwindow(&theWindow);
						if (theWindow != 0) gra_activatewindow(theWindow, 1);
						gra_showallfloaters(1);
					}
					break;
			}
	}
#endif
}

/*************************** SESSION LOGGING ROUTINES ***************************/

/*
 * routine to begin playback of session logging file "file".  The routine
 * returns nonzero if there is an error.  If "all" is nonzero, playback
 * the entire file with no prompt.
 */
INTSML us_logplayback(char *file, INTSML all)
{
	REGISTER INTBIG comcount;
	char *filename;

	us_logplay = xopen(file, FILETYPELOG, "", &filename);
	if (us_logplay == NULL) return(1);
	ttyputmsg("Type any key to playback the next step in the log file");
	ttyputmsg("Type a number followed by 'x' to playback that many steps");
	ttyputmsg("Type 'q' to terminate playback");

	comcount = filesize(us_logplay) / (sizeof (gra_action));
	ttyputmsg("There are no more than %d steps to playback", comcount);
	if (all != 0) gra_playbackmultiple = comcount;
	return(0);
}

/*
 * routine to create a session logging file
 */
void us_logstartrecord(void)
{
#if 0
	us_logrecord = xcreate(ELECTRICLOG, FILETYPELOG, 0, 0);
	mac_settypecreator(ELECTRICLOG, 'LOG ', 'Elec');
#else
	us_logrecord = NULL;
#endif
}

/*
 * routine to terminate session logging
 */
void us_logfinishrecord(void)
{
	if (us_logrecord != NULL) xclose(us_logrecord);
	us_logrecord = NULL;
}

/****************************** MENUS ******************************/

void gra_initializemenus(void)
{
#ifdef USETK
	tcl_nativemenuinitialize();
#else
	gra_pulldownmenucount = 0;
	gra_appleMenu = GetMenu(appleMENU);
	if (gra_appleMenu == 0) error("Cannot find menu!");
	InsertMenu(gra_appleMenu, 0);
	DrawMenuBar();
	AddResMenu(gra_appleMenu, 'DRVR');
#endif
}

/*
 * routine to handle the Apple menu, including the "About Electric..." dialog
 */
void gra_applemenu(INTSML index)
{
	Str255 name;
	GrafPtr savePort;
	char line[256];
	INTSML i, j;
	Handle pname, company;

	GetPort(&savePort);
	if (index == aboutMeCommand)
	{
		pname = GetResource('STR ', NAMERSRC);
		company = GetResource('STR ', COMPANYRSRC);
		if (pname != 0 && company != 0)
		{
			HLock(pname);   HLock(company);
			(void)strcpy(line, "Licensed to ");
			j = strlen(line);
			for(i = 1; i <= (*pname)[0]; i++) line[j++] = (*pname)[i];
			line[j++] = ' ';   line[j++] = 'a';
			line[j++] = 't';   line[j++] = ' ';
			for(i = 1; i <= (*company)[0]; i++) line[j++] = (*company)[i];
			line[j] = 0;
			HUnlock(pname);   HUnlock(company);
		} else line[0] = 0;
		tty_aboutdlog(line);
	} else
	{
		GetItem(gra_appleMenu, index, name);
		(void)OpenDeskAcc(name);
	}
	SetPort(savePort);
}

INTSML gra_popupmenu(POPUPMENU *menu, INTSML header, INTSML left, INTSML top)
{
	return(-1);
}

void gra_nativemenudoone(INTSML low, INTSML high)
{
	INTSML i, j;
	POPUPMENU *pm;

	i = high - USERMENUBASE;
	if (i >= 0 && i < gra_pulldownmenucount)
	{
		pm = gra_pulldowns[i];
		j = abs(low) - 1;
		if (j >= 0 && j < pm->total)
		{
			us_state |= DIDINPUT;
			us_state &= ~GOTXY;
			gra_set_cursorstate(NULLCURSOR);
			us_forceeditchanges();
			us_execute(pm->list[j].response, us_aid->aidstate&ECHOBIND, 1, 1);
			setactivity(pm->list[j].attribute);
		}
	}
	HiliteMenu(0);
}

/* routine to redraw entry "index" of popupmenu "pm" because it changed */
void gra_nativemenurename(POPUPMENU *pm, INTSML index)
{
#ifdef USETK
	void tcl_nativemenurename(POPUPMENU*, INTSML);

	tcl_nativemenurename(pm, index);
#else
	INTSML i, submenuindex;
	char line[100], *pt;
	USERCOM *uc;

	for(i=0; i<gra_pulldownmenucount; i++)
		if (gra_pulldowns[i] == pm)
	{
		uc = pm->list[index].response;
		if (uc->active < 0)
		{
			if (gra_pulldownmenus[i] != 0)
				DisableItem(gra_pulldownmenus[i], index+1);
			if (*pm->list[i].attribute == 0) (void)strcpy(line, " -");
		} else
		{
			if (gra_pulldownmenus[i] != 0)
				EnableItem(gra_pulldownmenus[i], index+1);
		}

		(void)strcpy(&line[1], pm->list[index].attribute);
		line[0] = strlen(&line[1]);
		pt = line;
		if (pt[1] != '>' && pt[2] != ' ')
		{
			if (gra_pulldownmenus[i] != 0)
				CheckItem(gra_pulldownmenus[i], index+1, 0);
		} else
		{
			pt[2] = pt[0] - 4;
			pt += 2;
			if (gra_pulldownmenus[i] != 0)
				CheckItem(gra_pulldownmenus[i], index+1, 1);
		}
		if (pt[pt[0]-1] == '/') pt[0] -= 2;
		if (gra_pulldownmenus[i] != 0)
			SetItem(gra_pulldownmenus[i], index+1, (unsigned char *)pt);

		/* see if this command is another menu */
		if (uc->menu != NOPOPUPMENU)
		{
			for(submenuindex=0; submenuindex<gra_pulldownmenucount; submenuindex++)
				if (gra_pulldowns[submenuindex] == uc->menu) break;
			if (submenuindex >= gra_pulldownmenucount) continue;
			if (gra_pulldownmenus[i] != 0)
			{
				SetItemCmd(gra_pulldownmenus[i], index+1, 0x1B);
				SetItemMark(gra_pulldownmenus[i], index+1, USERMENUBASE+submenuindex);
			}
			continue;
		}
		break;
	}
#endif
}

/*
 * Routine to establish the "count" pulldown menu names in "par" as the pulldown menu bar.
 * Returns nonzero on error.
 */
INTSML gra_nativemenuload(INTSML count, char *par[])
{
#ifdef USETK
	WINDOWFRAME *mw;

	/* create the TK menubar */
	if (tcl_nativemenuload(count, par) != 0) return(1);
	gra_tkmenusloaded = 1;

	/* attach it to all windows */
	if (Tcl_VarEval(tcl_interp, Tk_PathName(gra_tkmessageswindow), " configure -menu .menus",
		0) == TCL_ERROR) ttyputerr("Tcl_VarEval returned %s", tcl_interp->result);
	for(mw = gra_firstwindowframe; mw != NOWINDOWFRAME; mw = mw->nextwindowframe)
		if (Tcl_VarEval(tcl_interp, Tk_PathName(mw->tkwin), " configure -menu .menus",
			0) == TCL_ERROR) ttyputerr("Tcl_VarEval returned %s", tcl_interp->result);
	return(0);
#else
	REGISTER INTSML i, menuindex;
	REGISTER POPUPMENU *pm;
	POPUPMENU *pulls[25];

	/* build the pulldown menu bar */
	for(i=0; i<count; i++)
	{
		pm = us_getpopupmenu(par[i]);
		if (pm == NOPOPUPMENU) continue;
		pulls[i] = pm;
		menuindex = gra_pulldownindex(pm);
		if (menuindex < 0) continue;
		InsertMenu(gra_pulldownmenus[menuindex], 0);
	}
	DrawMenuBar();
	return(0);
#endif
}

/*
 * Routine to create a pulldown menu from popup menu "pm".
 * Returns an index to the table of pulldown menus (-1 on error).
 */
INTSML gra_pulldownindex(POPUPMENU *pm)
{
	REGISTER INTSML i, index;
	MenuHandle *newpulldownmenus;
	POPUPMENU **newpulldowns;

	/* see if it is in the list already */
	for(i=0; i<gra_pulldownmenucount; i++)
		if (gra_pulldowns[i] == pm) return(i);

	/* allocate new space with one more */
	newpulldownmenus = (MenuHandle *)emalloc((gra_pulldownmenucount+1) *
		(sizeof (MenuHandle)), us_aid->cluster);
	if (newpulldownmenus == 0) return(-1);
	newpulldowns = (POPUPMENU **)emalloc((gra_pulldownmenucount+1) *
		(sizeof (POPUPMENU *)), us_aid->cluster);
	if (newpulldowns == 0) return(-1);

	/* copy former arrays then delete them */
	for(i=0; i<gra_pulldownmenucount; i++)
	{
		newpulldownmenus[i] = gra_pulldownmenus[i];
		newpulldowns[i] = gra_pulldowns[i];
	}
	if (gra_pulldownmenucount != 0)
	{
		efree((char *)gra_pulldownmenus);
		efree((char *)gra_pulldowns);
	}
	gra_pulldownmenus = newpulldownmenus;
	gra_pulldowns = newpulldowns;

	index = gra_pulldownmenucount++;
	gra_pulldownmenus[index] = gra_makepdmenu(pm, USERMENUBASE+index);
	if (gra_pulldownmenus[index] == 0) return(-1);
	gra_pulldowns[index] = pm;
	return(index);
}

/*
 * Routine to create pulldown menu number "value" from the popup menu in "pm" and return
 * the menu handle.
 */
MenuHandle gra_makepdmenu(POPUPMENU *pm, INTSML value)
{
	REGISTER INTSML i, submenuindex;
	char myline[256], *pt;
	REGISTER USERCOM *uc;
	REGISTER POPUPMENUITEM *mi;
	MenuHandle thismenu;
	REGISTER INTSML len;
	char submenu[4];

	strcpy(&myline[1], pm->header);
	myline[0] = strlen(pm->header);
	thismenu = NewMenu(value, (unsigned char *)myline);
	if (thismenu == 0) return(0);

	/* build the actual menu */
	for(i=0; i<pm->total; i++)
	{
		mi = &pm->list[i];
		uc = mi->response;
		if (uc->active < 0)
		{
			(void)strcpy(myline, " (");
			if (*mi->attribute == 0) (void)strcat(myline, "-"); else
				(void)strcat(myline, mi->attribute);
			myline[0] = strlen(&myline[1]);
			AppendMenu(thismenu, (unsigned char *)myline);
			continue;
		}

		/* see if this command is another menu */
		if (uc->menu != NOPOPUPMENU)
		{
			submenuindex = gra_pulldownindex(uc->menu);
			if (submenuindex < 0) continue;
			InsertMenu(gra_pulldownmenus[submenuindex], -1);
			myline[1] = '!';   myline[2] = USERMENUBASE+submenuindex;
			(void)strcpy(&myline[3], mi->attribute);
			submenu[0] = '/';   submenu[1] = 0x1B;   submenu[2] = 0;
			(void)strcat(&myline[1], submenu);
			myline[0] = strlen(&myline[1]);
			AppendMenu(thismenu, (unsigned char *)myline);
			continue;
		}

		/* insert command title */
		pt = mi->attribute;
		if (pt[0] == '>' && pt[1] == ' ')
		{
			myline[1] = '!';
			myline[2] = 022;
			(void)strcpy(&myline[3], &pt[2]);
			len = strlen(myline);
			if (myline[len-2] == ' ' && myline[len-1] == '<') myline[len-2] = 0;
		} else (void)strcpy(&myline[1], pt);
		myline[0] = strlen(&myline[1]);
		AppendMenu(thismenu, (unsigned char *)myline);
	}
	return(thismenu);
}

/******************** TCL/TK ********************/

#ifdef	USETK

/*
 * This is a hack to take over menu selection from TK.
 * The module "tkMacMenus.c" has this routine for handling
 * a limited set of pulldown menus.  By yanking that routine
 * from the TK build, this routine provides full menu access
 * to Electric.
 */
void TkMacHandleMenuSelect(long mResult, int optionKeyPressed)
{
    short theItem, theMenu;
	Str255 name;
	GrafPtr savePort;

	if (mResult == 0)
	{
		TkMacHandleTearoffMenu();
		return;
	}

    theItem = LoWord(mResult);
    theMenu = HiWord(mResult);
	if (theMenu == appleMENU)
	{
		GetPort(&savePort);
		GetItem(tkAppleMenu, theItem, name);
		(void)OpenDeskAcc(name);
		SetPort(savePort);
		return;
	}
    TkMacDispatchMenuEvent(theMenu, theItem);
    TkMacClearMenubarActive();
    HiliteMenu(0);
}

void gra_addeventtoqueue(INTBIG state, INTSML x, INTSML y)
{
	MYEVENTQUEUE *next;

	next = gra_eventqueuetail + 1;
	if (next >= &gra_eventqueue[EVENTQUEUESIZE]) next = gra_eventqueue;
	if (next == gra_eventqueuehead)
	{
		/* queue is full */
		SysBeep(100);
		SysBeep(100);
		return;
	}

	gra_eventqueuetail->inputstate = state;
	gra_eventqueuetail->cursorx = x;
	gra_eventqueuetail->cursory = y;
	gra_eventqueuetail = next;
}

void gra_clearwindowframe(WINDOWFRAME *frame)
{
	REGISTER WINDOW *w, *neww, *nextw;
	REGISTER NODEPROTO *np;
	REGISTER WINDOWFRAME *lastmw, *mw;

	/* save highlighting and turn it off */
	us_pushhighlight();
	us_clearhighlightcount();

	startobjectchange((INTBIG)us_aid, VAID);

	/* kill all editor windows on this frame */
	neww = NOWINDOW;
	for(w = el_topwindow; w != NOWINDOW; w = nextw)
	{
		nextw = w->nextwindow;
		if (w->frame != frame)
		{
			neww = w;
			continue;
		}

		/* kill this window */
		killwindow(w);
	}
	endobjectchange((INTBIG)us_aid, VAID);

	(void)setvalkey((INTBIG)us_aid, VAID, us_current_window, (INTBIG)neww, VWINDOW|VDONTSAVE);
	if (neww != NOWINDOW) np = neww->curnodeproto; else np = NONODEPROTO;
	(void)setval((INTBIG)el_curlib, VLIBRARY, "curnodeproto", (INTBIG)np, VNODEPROTO);

	/* restore highlighting */
	(void)us_pophighlight(0);

	/* remove the window frame object */
	lastmw = NOWINDOWFRAME;
	for(mw = gra_firstwindowframe; mw != NOWINDOWFRAME; mw = mw->nextwindowframe)
	{
		if (mw == frame)
		{
			if (lastmw == NOWINDOWFRAME) gra_firstwindowframe = mw->nextwindowframe; else
				lastmw->nextwindowframe = mw->nextwindowframe;
			DisposeGWorld(mw->window);
			efree((char *)mw);
			break;
		}
		lastmw = mw;
	}
}

void gra_raisewindow(Tk_Window win)
{
	char *argv[2];
	extern Tk_Window gra_tktopwindow;
	int ret;

	argv[0] = "raise";
	argv[1] = Tk_PathName(win);
	ret = Tk_RaiseCmd((ClientData)gra_tktopwindow, tcl_interp, 2, argv);
	if (ret != TCL_OK)
	{
		ttyputerr("ERROR from Tk_RaiseCmd: %s", tcl_interp->result);
		Tcl_ResetResult(tcl_interp);
	}
}

void gra_namewindow(Tk_Window win, char *name)
{
	char *argv[4];
	extern Tk_Window gra_tktopwindow;
	int ret;

	argv[0] = "wmcmd";
	argv[1] = "title";
	argv[2] = Tk_PathName(win);
	argv[3] = name;
	ret = Tk_WmCmd((ClientData)gra_tktopwindow, tcl_interp, 4, argv);
	if (ret != TCL_OK)
	{
		ttyputerr("ERROR from Tk_WmCmd: %s", tcl_interp->result);
		Tcl_ResetResult(tcl_interp);
	}
}

void gra_messageEventProc(ClientData clientData, XEvent *eventPtr)
{
	INTBIG which, key, inputstate, theTime;
	INTSML cntlCode, oldpos, x, y;
	static XWindowAttributes xwa;
	static INTSML update = 1;
	WindowPtr win;
	WINDOWFRAME *mw;
	Rect r;
	Point p;
	ControlHandle theControl;

	inputstate = NOEVENT;
	win = (WindowPtr)TkMacGetDrawablePort(eventPtr->xany.window);
	switch (eventPtr->type)
	{
		case KeyPress:
			which = ((XKeyEvent *)eventPtr)->keycode & charCodeMask;
			if (gra_curwindowframe != NOWINDOWFRAME)
			{
				SetPort((WindowPtr)gra_curwindowframe->realwindow);
				x = eventPtr->xkey.x;
				y = gra_curwindowframe->revy - eventPtr->xkey.y;
			}
			if ((eventPtr->xbutton.state&Mod1Mask) != 0 && which == '.')
			{
				el_pleasestop = 2;
				break;
			}
			if ((eventPtr->xbutton.state&Mod1Mask) != 0)
			{
				key = MenuKey((char)which);
				gra_highmenu = HiWord(key);
				gra_lowmenu = -LoWord(key);
				inputstate = MENUEVENT;
				if (gra_highmenu != 0) break;
			}
			if (gra_messagesinfront != 0)
			{
				TESetSelect(32767, 32767, gra_TEH);
				TEKey(which, gra_TEH);
				gra_showselect();
				break;
			}
			inputstate = which & CHARREAD;
			us_state |= DIDINPUT;
			gra_addeventtoqueue(inputstate, x, y);
			break;

		case ButtonPress:
			p.h = eventPtr->xbutton.x;	p.v = eventPtr->xbutton.y;
			for(mw = gra_firstwindowframe; mw != NOWINDOWFRAME; mw = mw->nextwindowframe)
				if ((WindowPtr)mw->realwindow == win) break;

			if (gra_messagesinfront != 0)
			{
				cntlCode = FindControl(p, win, &theControl);
				if (cntlCode == inThumb)
				{
					oldpos = GetCtlValue(theControl);
					TrackControl(theControl, p, 0L);
					if (theControl == gra_vScroll) gra_adjustvtext(); else
						gra_adjusthtext(oldpos);
					break;
				}
				if (cntlCode == inUpButton || cntlCode == inDownButton ||
					cntlCode == inPageUp || cntlCode == inPageDown)
				{
					if (theControl == gra_vScroll)
						TrackControl(theControl, p, gra_scrollvprocUPP); else
							TrackControl(theControl, p, gra_scrollhprocUPP);
					break;
				}
				TEClick(p, (eventPtr->xbutton.state&ShiftMask) != 0, gra_TEH);
				break;
			}

			/* ignore clicks in the messages area at the bottom */
			if (mw != NOWINDOWFRAME && eventPtr->xbutton.y >= win->portRect.bottom - SBARWIDTH)
				break;

			/* compute event information */
			inputstate = ISBUTTON;
			if ((eventPtr->xbutton.state&(ShiftMask|LockMask)) != 0) inputstate |= SHIFTDOWN;
			if ((eventPtr->xbutton.state&ControlMask) != 0) inputstate |= CONTROLDOWN;
			if ((eventPtr->xbutton.state&Mod1Mask) != 0) inputstate |= COMMANDDOWN;
			if ((eventPtr->xbutton.state&Mod2Mask) != 0) inputstate |= OPTIONDOWN;
			x = eventPtr->xbutton.x;
			if (mw == NOWINDOWFRAME) y = eventPtr->xbutton.y; else
			{
				y = mw->revy - eventPtr->xbutton.y;
			}
			theTime = LMGetTicks();
			if (theTime - gra_doubleclick < gra_lastclick &&
				(gra_inputstate & (SHIFTDOWN|COMMANDDOWN|OPTIONDOWN|CONTROLDOWN)) == 0 &&
				abs(x-gra_lstcurx) < 5 && abs(y-gra_lstcury) < 5)
			{
				inputstate |= DOUBLECLICK;
				gra_lastclick = theTime - gra_doubleclick - 1;
			} else gra_lastclick = theTime;
			gra_lstcurx = x;   gra_lstcury = y;
			us_state |= DIDINPUT;
			gra_addeventtoqueue(inputstate, x, y);
			break;

		case ButtonRelease:
			inputstate = ISBUTTON|BUTTONUP;
			if ((eventPtr->xbutton.state&(ShiftMask|LockMask)) != 0) inputstate |= SHIFTDOWN;
			if ((eventPtr->xbutton.state&ControlMask) != 0) inputstate |= CONTROLDOWN;
			if ((eventPtr->xbutton.state&Mod1Mask) != 0) inputstate |= COMMANDDOWN;
			if ((eventPtr->xbutton.state&Mod2Mask) != 0) inputstate |= OPTIONDOWN;
			x = eventPtr->xbutton.x;
			if (gra_curwindowframe == NOWINDOWFRAME) y = eventPtr->xbutton.y; else
				y = gra_curwindowframe->revy - eventPtr->xbutton.y;
			gra_addeventtoqueue(inputstate, x, y);
			break;

		case MotionNotify:
			for(mw = gra_firstwindowframe; mw != NOWINDOWFRAME; mw = mw->nextwindowframe)
				if ((WindowPtr)mw->realwindow == win) break;
			inputstate = MOTION;
			if ((eventPtr->xmotion.state&Button1Mask) == 0) inputstate |= BUTTONUP;
			x = eventPtr->xmotion.x;
			if (mw == NOWINDOWFRAME) y = eventPtr->xmotion.y; else
				y = mw->revy - eventPtr->xmotion.y;
			gra_addeventtoqueue(inputstate, x, y);
			break;

		case ConfigureNotify:
			gra_mygrowwindow(win, win->portRect.right-win->portRect.left,
				win->portRect.bottom-win->portRect.top, &win->portRect);
			break;

		case FocusIn:
			if (win == gra_messageswindow && gra_messageswindow != 0)
			{
				gra_messagesinfront = 1;
				ShowControl(gra_vScroll);
				ShowControl(gra_hScroll);
			}
			gra_setcurrentwindowframe();
			break;

		case FocusOut:
			if (win == gra_messageswindow && gra_messageswindow != 0)
			{
				gra_messagesinfront = 0;
				HideControl(gra_vScroll);
				HideControl(gra_hScroll);
			}
			break;

		case DestroyNotify:
			if (win == gra_messageswindow && gra_messageswindow != 0)
			{
				/* messages window deleted */
				gra_messagesinfront = 0;
				gra_messageswindow = 0;
				break;
			}

			/* determine which editor window was hit */
			for(mw = gra_firstwindowframe; mw != NOWINDOWFRAME; mw = mw->nextwindowframe)
				if ((WindowPtr)mw->realwindow == win) break;
			if (mw != NOWINDOWFRAME) gra_clearwindowframe(mw);
			break;

		case Expose:
			if (eventPtr->xexpose.count == 0)
			{
				SetPort(win);
				if (win == gra_messageswindow && gra_messageswindow != 0)
				{
					EraseRect(&win->portRect);
					DrawGrowIcon(win);
					DrawControls(win);
					TEUpdate(&win->portRect, gra_TEH);
				} else
				{
					for(mw = gra_firstwindowframe; mw != NOWINDOWFRAME; mw = mw->nextwindowframe)
						if ((WindowPtr)mw->realwindow == win) break;
					if (mw != NOWINDOWFRAME)
					{
						DrawGrowIcon(win);
						r = (*mw->realwindow->visRgn)->rgnBBox;
						gra_setrect(mw, r.left, r.right, r.top, r.bottom);
						us_redostatus((void *)mw);
					} else
					{
						/* probably a dialog, just process it */
						DiaRedrawDialogWindow();
					}
				}
			}
			break;
	}
}
#endif
