/*
 * Electric(tm) VLSI Design System
 *
 * File: graphpccode.cpp
 * Interface for Win32 (Win95 & WinNT) computers
 * Written by: Steven M. Rubin, Electric Editor Incorporated
 *
 * Copyright (c) 1998 Electric Editor Incorporated.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Electric Editor Incorporated
 * 23470 Sunset Drive, Suite 108
 * Los Gatos, California 95033
 * support@electriceditor.com
 */

#include "graphpcstdafx.h"
#include "graphpc.h"
#include "graphpcdoc.h"
#include "graphpcview.h"
#include "graphpcmainframe.h"
#include "graphpcchildframe.h"
#include "graphpcmsgview.h"
#include "graphpcdialog.h"

extern "C"
{
#include "global.h"
#include "egraphics.h"
#include "usr.h"
#include "edialogs.h"
#if LANGTCL
#  include "dblang.h"
#endif
}

#include <io.h>
#include <sys/stat.h>
#include <direct.h>
#include <sys/timeb.h>

/****** window frames ******/
#define	NOWINDOWFRAME ((WINDOWFRAME *)-1)

typedef struct IWindowFrame
{
	CChildFrame         *wndframe;
	HWND				 realwindow;		/* the real window */
	char                *data;				/* raster data for window */
	CDC					*hDC;				/* device context for real window */
	HBITMAP				 hBitmap;			/* 8-bit offscreen bitmap */
	BITMAPINFO          *bminfo;			/* bit map info structure */
	CFont               *hFont;				/* current font in bitmap */
	CDC					*hDCOff;          	/* device context for the offscreeen bitmap */
	CPalette			*hPalette;
	LOGPALETTE			*pColorPalette;
	INTSML               floating;			/* nonzero for floating palette window */
	INTSML               swid, shei;		/* screen dimensions for drawing */
	INTSML               revy;				/* for reversing Y coordinates */
	RECT                 copyrect;			/* rectangle to copy to screen */
	INTSML               offscreendirty;	/* nonzero if offscreen area is "dirty" */
	INTSML               screenleft,		/* left bound of editing window screen */
                         screenright,		/* right bound of editing window screen */
                         screentop,			/* top bound of editing window screen */
                         screenbottom;		/* bottom bound of editing window screen */
	char               **rowstart;			/* start of each row */
	struct IWindowFrame *nextwindowframe;	/* next in list */
} WINDOWFRAME;

static WINDOWFRAME *gra_firstwindowframe = NOWINDOWFRAME;
static WINDOWFRAME *gra_curwindowframe;
static WINDOWFRAME *gra_cureditwindowframe;
static WINDOWFRAME *gra_palettewindowframe = NOWINDOWFRAME;

/****** the messages window ******/
static CMenu       *gra_hMenu;				/* System Menu */
	   CChildFrame *gra_messageswindow;		/* the scrolled status window */
static CEdit       *gra_editCtrl;
static INTSML       gra_messagescurrent;	/* nonzero if messages window is current */
static INTSML       gra_messagesleft;		/* left bound of messages window screen */
static INTSML       gra_messagesright;		/* right bound of messages window screen */
static INTSML       gra_messagestop;		/* top bound of messages window screen */
static INTSML       gra_messagesbottom;		/* bottom bound of messages window screen */
static char         gra_msgwindowchartyped;	/* key typed in messages window */
static INTSML       gra_wantmsgwindowcharacters;	/* nonzero to accept text from msgs */
static CDC         *gra_texthdc;			/* device context for text buffer */
static HBITMAP      gra_textbitmap;			/* bitmap for text buffer */
static BITMAPINFO  *gra_textbitmapinfo;		/* bitmap information structure */
static char        *gra_textdatabuffer;		/* data in text buffer */
static char       **gra_textrowstart;		/* row starts for text buffer */
static INTSML       gra_textbufwid, gra_textbufhei;	/* size of text buffer */
static INTSML       gra_textbufinited = 0;	/* nonzero if text buffer is initialized */
extern CElectricApp theApp;

/****** the status bar ******/
#define MAXSTATUSLINES        2

STATUSFIELD *gra_statusfields[100];
char *gra_statusfieldtext[100];
INTSML gra_indicatorcount = 0;
static UINT gra_indicators[] = {ID_INDICATOR_ELECTRIC01, ID_INDICATOR_ELECTRIC02,
	ID_INDICATOR_ELECTRIC03, ID_INDICATOR_ELECTRIC04, ID_INDICATOR_ELECTRIC05,
	ID_INDICATOR_ELECTRIC06, ID_INDICATOR_ELECTRIC07, ID_INDICATOR_ELECTRIC08,
	ID_INDICATOR_ELECTRIC09, ID_INDICATOR_ELECTRIC10, ID_INDICATOR_ELECTRIC11,
	ID_INDICATOR_ELECTRIC12, ID_INDICATOR_ELECTRIC13, ID_INDICATOR_ELECTRIC14,
	ID_INDICATOR_ELECTRIC15, ID_INDICATOR_ELECTRIC16, ID_INDICATOR_ELECTRIC17};

/****** the dialogs ******/
#define PROGRESSOFFSET   100

int              gra_dialoghit;
int              gra_dialoghitchar;
int              gra_dialogdefaultbutton;
int              gra_dialogredrawitem;
CElectricDialog *gra_dialogwindow;
DIALOG          *gra_dialogitemdesc;
int              gra_dialogdbnx, gra_dialogdbny;
int              gra_dialogdbux, gra_dialogdbuy;
CBrush          *gra_dialogonbrush = 0, *gra_dialogoffbrush = 0;
int              gra_dialogeditline;
char             gra_dialoglastedittext[300];

/****** events ******/
#define	CHARREAD           0177				/* character that was read */
#define	ISBUTTON           0200				/* set if button pushed (or released) */
#define	BUTTONUP           0400				/* set if button was released */
#define	SHIFTDOWN         01000				/* set if shift key was held down */
#define	COMMANDDOWN       02000				/* set if command key was held down */
#define	ALTDOWN           04000				/* set if alt key was held down */
#define	CONTROLDOWN      010000				/* set if control key was held down */
#define	DOUBLECLICK      020000				/* set if this is second click */
#define	MOTION           040000				/* set if mouse motion detected */
#define	FILEREPLY       0100000				/* set if standard file reply detected */
#define	WINDOWCHANGE    0100400				/* set if window moved and/or grown */
#define ISLEFT         01000000				/* set if left button */
#define ISMIDDLE       02000000				/* set if middle button */
#define ISRIGHT        04000000				/* set if right button */
#define	NOEVENT              -1				/* set if nothing happened */
#define	MENUEVENT      BUTTONUP				/* set if menu entry selected (values in cursor) */
#define	MENUEVENT1    010000000				/* set if menu entry selected (values in cursor) */

static INTBIG       gra_inputstate;			/* current state of device input */
static INTSML       gra_cursorx, gra_cursory;	/* current position of mouse */

struct
{
	INTSML x;
	INTSML y;
	INTSML kind;
} gra_action;

typedef struct
{
	INTSML  message;
	INTSML  what;
	UINTBIG when;
	INTSML  where;
	INTSML  modifiers;
} EventRecord;

typedef struct
{
	INTSML                cursorx, cursory;	/* current position of mouse */
	INTBIG                inputstate;		/* current state of device input */
} MYEVENTQUEUE;

#define	EVENTQUEUESIZE	100

static MYEVENTQUEUE  gra_eventqueue[EVENTQUEUESIZE];
static MYEVENTQUEUE *gra_eventqueuehead;	/* points to next event in queue */
static MYEVENTQUEUE *gra_eventqueuetail;	/* points to first free event in queue */

/****** pulldown menus ******/
#define IDR_MENU           3000				/* base ID for menus */

static INTSML       gra_lowmenu;			/* low word of selected pulldown menu */
static INTSML       gra_highmenu;			/* high word of selected pulldown menu */
static int			gra_menustate;
	   CMenu      **gra_pulldownmenus;		/* list of Windows pulldown menus */
	   POPUPMENU  **gra_pulldowns;			/* list of Electric pulldown menus */
	   INTSML       gra_pulldownmenucount;	/* number of pulldown menus */
int gra_menures[] = {ID_ELECTRIC_MENU01, ID_ELECTRIC_MENU02, ID_ELECTRIC_MENU03,
	ID_ELECTRIC_MENU04, ID_ELECTRIC_MENU05, ID_ELECTRIC_MENU06, ID_ELECTRIC_MENU07,
	ID_ELECTRIC_MENU08, ID_ELECTRIC_MENU09, ID_ELECTRIC_MENU10, ID_ELECTRIC_MENU11,
	ID_ELECTRIC_MENU12, ID_ELECTRIC_MENU13, ID_ELECTRIC_MENU14, ID_ELECTRIC_MENU15,
	ID_ELECTRIC_MENU16, ID_ELECTRIC_MENU17, ID_ELECTRIC_MENU18, ID_ELECTRIC_MENU19,
	ID_ELECTRIC_MENU20, ID_ELECTRIC_MENU21, ID_ELECTRIC_MENU22, ID_ELECTRIC_MENU23,
	ID_ELECTRIC_MENU24, ID_ELECTRIC_MENU25, ID_ELECTRIC_MENU26, ID_ELECTRIC_MENU27,
	ID_ELECTRIC_MENU28, ID_ELECTRIC_MENU29};

/****** mouse buttons ******/
#define	BUTTONS              27				/* cannot exceed NUMBUTS in "usr.h" */

struct
{
	char  *name;			/* button name */
	INTSML unique;			/* number of letters that make it unique */
} gra_buttonname[BUTTONS] =
{
	{"LEFT" ,1},  {"MIDDLE",  1}, {"RIGHT",  1},	/* 0: unshifted */
	{"SLEFT",2},  {"SMIDDLE", 2}, {"SRIGHT", 2},	/* 3: shift held down */
	{"CLEFT",2},  {"CMIDDLE", 2}, {"CRIGHT", 2},	/* 6: control held down */
	{"ALEFT",2},  {"AMIDDLE", 2}, {"ARIGHT", 2},	/* 9: alt held down */
	{"SCLEFT",3}, {"SCMIDDLE",3}, {"SCRIGHT",3},	/* 12: control/shift held down */
	{"SALEFT",3}, {"SAMIDDLE",3}, {"SARIGHT",3},	/* 15: shift/alt held down */
	{"CALEFT",3}, {"CAMIDDLE",3}, {"CARIGHT",3},	/* 18: control/alt held down */
	{"SCALEFT",4},{"SCAMIDDLE",4},{"SCARIGHT",4},	/* 21: shift/control/alt held down */
	{"DLEFT",2},  {"DMIDDLE", 2}, {"DRIGHT", 2}		/* 24: double-click */
};

/****** cursors ******/
#define	NORMALCURSOR          0				/* a normal arrow cursor */
#define	WANTTTYCURSOR         1				/* a "use the TTY" cursor */
#define	PENCURSOR             2				/* a "draw with pen" cursor */
#define	NULLCURSOR            3				/* a null cursor */
#define	MENUCURSOR            4				/* a menu selection cursor */
#define	HANDCURSOR            5				/* a hand dragging cursor */
#define	TECHCURSOR            6				/* the technology edit cursor ("T") */
#define	IBEAMCURSOR           7				/* the text edit cursor ("I") */

static INTSML       gra_cursorstate;		/* see defines above */
HCURSOR	            gra_wantttyCurs,		/* a "use the TTY" cursor */
                    gra_penCurs,			/* a "draw with pen" cursor */
                    gra_menuCurs,			/* a menu selection cursor */
                    gra_handCurs,			/* a hand dragging cursor */
                    gra_techCurs,			/* a technology edit cursor */
                    gra_ibeamCurs,			/* a text edit cursor */
                    gra_normalCurs;
static INTSML       gra_normalcursor;		/* default cursor to use */

/****** rectangle saving ******/
#define	NOSAVEDBOX ((SAVEDBOX *)-1)
typedef struct Isavedbox
{
	HBITMAP           hBox;
	BITMAPINFO       *bminfo;
	CDC              *hMemDC;
	char             *data;
	char            **rowstart;
	WINDOW           *win;
	INTSML            lx, hx, ly, hy;
	struct Isavedbox *nextsavedbox;
} SAVEDBOX;

SAVEDBOX *gra_firstsavedbox = NOSAVEDBOX;

/****** polygon decomposition ******/
#define NOPOLYSEG ((POLYSEG *)-1)

typedef struct Isegment
{
	INTBIG fx,fy, tx,ty, direction, increment, index;
	struct Isegment *nextedge;
	struct Isegment *nextactive;
} POLYSEG;

static POLYSEG     *gra_polysegs;
static INTSML       gra_polysegcount = 0;

/****** curve drawing ******/
#define MODM(x) ((x<1) ? x+8 : x)
#define MODP(x) ((x>8) ? x-8 : x)
#define ODD(x)  (x&1)

static INTSML       gra_arcocttable[] = {0,0,0,0,0,0,0,0,0};
static INTBIG       gra_arccenterx, gra_arccentery, gra_arcradius, gra_curvemaxx, gra_curvemaxy;
static INTBIG       gra_arclx, gra_archx, gra_arcly, gra_archy;
static INTSML       gra_curvecol, gra_curvemask, gra_curvestyle, gra_arcfirst;
static WINDOWFRAME *gra_arcframe;			/* window frame for arc drawing */

/****** miscellaneous ******/
#define	PALETTEWIDTH        114				/* width of palette */
#define MAXPATHLEN          256				/* max chars in file path */
#define MAXTYPEDLINE        256				/* max chars on input line */
#define	SBARWIDTH            15				/* width of scroll bars */

/* font specification from registry */
#define USESYSFONT -1000	/* use system specified font */
struct {
	INTSML eFontIndex;		/* font index in Electric */
	char  *eFontName;		/* font name in Electric */
	char  *regFontName;		/* registry key name for font name */
	char  *regFontSize;		/* registry key name for font size */
	char  *fontName;		/* default font - will be overwritten by registry entries */
	INTSML fontSize;		/* default size - will be overwritten by registry entries */
} fontList[] = {
	// Small Fonts: 6 is good but tiny, 7,8,9 are good, 10,12 are bad
	{TXT4P,     "4point",      "4p_font",     "4p_size", "Small Fonts", -7},
	{TXT6P,     "6point",      "6p_font",     "6p_size", "Small Fonts", -9},
	{TXT8P,     "8point",      "8p_font",     "8p_size", "MS Sans Serif", -12},
	{TXT10P,    "10point",    "10p_font",    "10p_size", "MS Sans Serif", -13},
	{TXT12P,    "12point",    "12p_font",    "12p_size", "MS Sans Serif", -16},
	{TXT14P,    "14point",    "14p_font",    "14p_size", "MS Sans Serif", -19},
	{TXT16P,    "16point",    "16p_font",    "16p_size", "MS Sans Serif", -24},
	{TXT18P,    "18point",    "18p_font",    "18p_size", "MS Sans Serif", -27},
	{TXT20P,    "20point",    "20p_font",    "20p_size", "MS Sans Serif", -32},
	{TXTSMALL,  "small",    "small_font",  "small_size", "MS Sans Serif", -12},
	{TXTMEDIUM, "medium",  "medium_font", "medium_size", "MS Sans Serif", -19},
	{TXTLARGE,  "large",    "large_font",  "large_size", "MS Sans Serif", -32},
	{TXTEDITOR, "editor",  "editor_font", "editor_size", "Lucida Console", -12},
	{TXTMENU,   "menu",      "menu_font",   "menu_size", "MS Sans Serif", -13},
	{TXTSTATUS, "status",  "status_font", "status_size", "MS Sans Serif", -13},
	{-1, NULL, NULL, NULL}
};

static INTSML       gra_screenleft,			/* left bound of any editor window screen */
                    gra_screenright,		/* right bound of any editor window screen */
                    gra_screentop,			/* top bound of any editor window screen */
                    gra_screenbottom;		/* bottom bound of any editor window screen */
static char         gra_localstring[256];	/* local string */
static INTSML       gra_logrecordindex = 0;	/* count for session flushing */
static INTSML       gra_playbackmultiple = 0;	/* count for multi-key playback */
static void        *gra_fileliststringarray = 0;
static time_t       gra_timebasesec;
static INTBIG       gra_timebasems;

/****** prototypes for local routines ******/
void   gra_activateframe(CChildFrame *frame, BOOL bActivate);
void   gra_addeventtoqueue(INTBIG state, INTSML x, INTSML y);
INTSML gra_addfiletolist(char *file);
INTSML gra_buildoffscreenbuffer(WINDOWFRAME *frame, INTSML wid, INTSML hei, CDC **hdc,
		HBITMAP *bitmap, BITMAPINFO **bminfo, char **databuffer, char ***rowstart);
INTSML gra_buildwindow(WINDOWFRAME*, INTSML);
void   gra_buttonaction(int state, UINT nFlags, CPoint point, CWnd *frm);
int    gra_closeframe(CChildFrame *frame);
int    gra_closeworld(void);
CFont *gra_createtextfont(INTSML);
void   gra_diaredrawitem(void);
INTSML gra_dodialogisinsideuserdrawn(int x, int y);
int    gra_dodialoglistkey(UINT nKey, CListBox* pListBox, UINT nIndex);
void   gra_dodialogtextchange(int nID);
void   gra_drawdiscrow(WINDOWFRAME *frame, INTBIG thisy, INTBIG startx, INTBIG endx, GRAPHICS *desc);
void   gra_drawline(WINDOWFRAME *frame, INTSML x1, INTSML y1, INTSML x2, INTSML y2, INTSML col, INTSML mask);
void   gra_drawpatline(WINDOWFRAME *frame, INTSML x1, INTSML y1, INTSML x2, INTSML y2, INTSML col,
		INTSML mask, INTSML pattern);
RECT  *gra_geteditorwindowlocation(void);
void   gra_getdevices(void);
HWND   gra_getwindow(void *frame);
void   gra_initfilelist(void);
void   gra_itemclicked(int nID);
void   gra_itemdoubleclicked(int nID);
void   gra_keyaction(UINT nChar, UINT nFlags, UINT nRepCnt);
INTSML gra_makebutton(INTBIG);
INTSML gra_makeeditwindow(WINDOWFRAME*);
HICON  gra_makeicon(char *data);
INTSML gra_makemessageswindow(void);
CMenu *gra_makepdmenu(POPUPMENU *, INTSML );
INTSML gra_messagesnotvisible(void);
void   gra_mouseaction(UINT nFlags, CPoint point, CWnd *frm);
int    gra_msgchartyped(UINT nChar);
void   gra_msgwindowactive(BOOL bActivate);
void   gra_msgwindowchange(void);
void   gra_nativemenudoone(INTSML low, INTSML high);
void   gra_nextevent(INTSML);
INTSML gra_pulldownindex(POPUPMENU *);
void   gra_redrawdisplay(WINDOWFRAME*);
void   gra_redrawstatusindicators(void);
void   gra_reloadmap(void);
INTSML gra_remakeeditwindow(WINDOWFRAME*);
void   gra_repaint(CChildFrame *frame, CPaintDC *dc);
void   gra_resize(CChildFrame *frame, int cx, int cy);
void   gra_resizemain(int cx, int cy);
void   gra_setdefaultcursor(void);
void   gra_setrect(WINDOWFRAME*, INTSML, INTSML, INTSML, INTSML);
void   gra_waitforaction(INTSML, EventRecord*);
char  *gra_waitforSF(char *msg, INTSML kind, char *defofile);
#if LANGTCL
int    InitializeTCL(void);
#endif

extern "C"
{
	void  (*gra_dialogredrawroutine)(RECTAREA*);
	void    gra_dodialogcharacteredit(int item);
	void    gra_dodialogdimitem(int item, int enabled);
	void    gra_dodialogdone(void);
	void    gra_dodialogdrawrect(RECT *r, int on);
	void    gra_dodialogframerect(RECT *r);
	int     gra_dodialoggetcontrol(int item, int type);
	char   *gra_dodialoggetline(int item, int line);
	void    gra_dodialoggetmouse(int *x, int *y);
	int     gra_dodialoggetpopupentry(int item);
	int     gra_dodialoggetselectedline(int item);
	char   *gra_dodialoggettext(int item, int type);
	long    gra_dodialoginit(DIALOG *dialog);
	void    gra_dodialoginitscroll(int item, int flags);
	void    gra_dodialogloadscroll(int item, int sort, short (*toplist)(char **),
				char *(*nextinlist)(void), void (*donelist)(void));
	int     gra_dodialognextchar(short *itemHit);
	int     gra_dodialognexthit(void);
	void    gra_dodialogopaqueitem(int item);
	void    gra_dodialogpercent(int item, int percent);
	void    gra_dodialogredisproutine(int item, void (*routine)(RECTAREA*));
	void    gra_dodialogselectline(int item, int line);
	void    gra_dodialogsetcontrol(int item, int type, int value);
	void    gra_dodialogsetdefault(int formeritem, int item);
	void    gra_dodialogsetline(int item, int line, char *msg);
	void    gra_dodialogsetpopup(int item, int count, char **names);
	void    gra_dodialogsetpopupentry(int item, int entry);
	void    gra_dodialogsettext(int item, int type, char *msg, int highlight);
	void    gra_dodialogstuffline(int item, char *line);
	void    gra_dopopdialog(long dia, DIALOG *desc);
	UINTBIG gra_machinetimeoffset(void);
};

/******************** INITIALIZATION ********************/

/*
 * routines to establish the default display
 */
void us_setdisplay(char *name, INTBIG *argc, char **argv) {}

/*
 * routine to initialize the display device
 */
INTSML us_initgraphics(INTSML messages)
{
	void *mw;

	gra_inputstate = NOEVENT;
	gra_pulldownmenucount = 0;	/* number of pulldown menus */
	gra_hMenu = 0;
	gra_dialogwindow = 0;

	/* get cursors */
	gra_normalCurs = theApp.LoadStandardCursor(IDC_ARROW);
	gra_wantttyCurs = theApp.LoadCursor(IDC_CURSORTTY);
	if (gra_wantttyCurs == 0) gra_wantttyCurs = gra_normalCurs;
	gra_penCurs = theApp.LoadCursor(IDC_CURSORPEN);
	if (gra_penCurs == 0) gra_penCurs = gra_normalCurs;
	gra_handCurs = theApp.LoadCursor(IDC_CURSORHAND);
	if (gra_handCurs == 0) gra_handCurs = gra_normalCurs;
	gra_menuCurs = theApp.LoadCursor(IDC_CURSORMENU);
	if (gra_menuCurs == 0) gra_menuCurs = gra_normalCurs;
	gra_techCurs = theApp.LoadCursor(IDC_CURSORTECH);
	if (gra_techCurs == 0) gra_techCurs = gra_normalCurs;
	gra_ibeamCurs = theApp.LoadStandardCursor(IDC_IBEAM);
	gra_normalcursor = NORMALCURSOR;

	/* setup globals that describe location of windows */
	gra_getdevices();

	/* create the scrolling messages window */
	if (messages == 0) gra_messageswindow = 0; else
	{
	   if (gra_makemessageswindow() != 0) error("Cannot create messages window");
	}
 
	/* initialize the cursor */
	SetCursor(0);
	gra_cursorstate = NULLCURSOR;

	gra_eventqueuehead = gra_eventqueuetail = gra_eventqueue;
	gra_firstwindowframe = gra_curwindowframe = gra_cureditwindowframe = NOWINDOWFRAME;

	return(0);
}

INTSML gra_makemessageswindow(void)
{
	RECT rmsg;
	CCreateContext context;

	/* create a context to tell this window to be an EditView */
	context.m_pNewViewClass = RUNTIME_CLASS(CElectricMsgView);
	context.m_pCurrentDoc = NULL;
	context.m_pNewDocTemplate = NULL;
	context.m_pLastView = NULL;
	context.m_pCurrentFrame = NULL;

	/* set messages window location */
	rmsg.left = gra_messagesleft;
	rmsg.right = gra_messagesright;
	rmsg.top = gra_messagestop;
	rmsg.bottom = gra_messagesbottom;

	/* create the window */
	gra_messageswindow = new CChildFrame();
	gra_messageswindow->Create(NULL, "Electric Messages",
		WS_CHILD|WS_OVERLAPPEDWINDOW|WS_VISIBLE, rmsg, NULL, &context);
	gra_messageswindow->InitialUpdateFrame(NULL, TRUE);
	CEditView *msgView = (CEditView *)gra_messageswindow->GetActiveView();
	gra_editCtrl = &msgView->GetEditCtrl();

	gra_messagescurrent = 0;
	gra_wantmsgwindowcharacters = 0;
	return(0);
}

/*
 * Routine to examine the display devices available and to setup globals that describe
 * the editing windows and messages window extents.  On exit, the globals "gra_screenleft",
 * "gra_screenright", "gra_screentop", and "gra_screenbottom" will describe the area
 * for the editing windows and the variables "gra_messagesleft", "gra_messagesright",
 * "gra_messagestop", and "gra_messagesbottom" will describe the messages window.
 */
void gra_getdevices(void)
{
	RECT rDesktop;

	/* obtain the current device */
	theApp.m_pMainWnd->GetClientRect(&rDesktop);
	rDesktop.bottom -= 22;		/* status bar height? */

	gra_screenleft   = (INTSML)0;
	gra_screenright  = (INTSML)rDesktop.right-rDesktop.left;
	gra_screentop    = (INTSML)0;
	gra_screenbottom = (INTSML)rDesktop.bottom-rDesktop.top;

	gra_messagesleft   = gra_screenleft + PALETTEWIDTH;
	gra_messagesright  = gra_screenright - PALETTEWIDTH;
	gra_messagestop    = gra_screentop - 2 + (gra_screenbottom-gra_screentop) * 4 / 5;
	gra_messagesbottom = gra_screenbottom;
}

/*
 * this routine returns the host ID for this computer in "name"
 */
void getmachineid(char *name)
{
	(void)strcpy(name, "52001ae4");	/* just fake a valid key */
}

/******************** TERMINATION ********************/

void us_unsetdisplay(void) {}

void us_termgraphics(INTSML final)
{
	if (final != 0) return;
	while (gra_firstwindowframe != NOWINDOWFRAME)
	{
		delete gra_firstwindowframe->wndframe;
	}
	if (gra_messageswindow != 0)
		delete gra_messageswindow;
}

void exitprogram(void)
{
	exit(0);
}

/******************** WINDOW CONTROL ********************/

void *us_newwindowframe(INTSML floating)
{
	WINDOWFRAME *mw, *oldlisthead;

	/* allocate one */
	mw = (WINDOWFRAME *)emalloc((sizeof (WINDOWFRAME)), us_aid->cluster);
	if (mw == 0) return(0);

	/* insert window-frame in linked list */
	oldlisthead = gra_firstwindowframe;
	mw->nextwindowframe = gra_firstwindowframe;
	gra_firstwindowframe = mw;
	gra_curwindowframe = mw;
	if (floating == 0) gra_cureditwindowframe = mw; else
		gra_palettewindowframe = mw;

	/* load an editor window into this frame */
	mw->pColorPalette = 0;
	if (gra_buildwindow(mw, floating) != 0)
	{
		efree((char *)mw);
		gra_firstwindowframe = oldlisthead;
		return(0);
	}

	return((void *)mw);
}

void us_killwindowframe(void *frame)
{
	WINDOWFRAME *mw, *omw, *lastmw;

	/* kill the actual window */
	omw = (WINDOWFRAME *)frame;
	omw->wndframe->DestroyWindow();

	/* find this frame in the list */
	lastmw = NOWINDOWFRAME;
	for(mw = gra_firstwindowframe; mw != NOWINDOWFRAME; mw = mw->nextwindowframe)
	{
		if (mw == omw) break;
		lastmw = mw;
	}
	if (mw == NOWINDOWFRAME) return;
	if (lastmw == NOWINDOWFRAME) gra_firstwindowframe = omw->nextwindowframe; else
		lastmw->nextwindowframe = omw->nextwindowframe;
	if (gra_curwindowframe == omw) gra_curwindowframe = NOWINDOWFRAME;
	if (gra_cureditwindowframe == omw) gra_cureditwindowframe = NOWINDOWFRAME;
	free((char *)omw);
}

void *us_getwindowframe(void)
{
	if (gra_curwindowframe != NOWINDOWFRAME)
		return((void *)gra_curwindowframe);
	if (gra_cureditwindowframe != NOWINDOWFRAME)
		return((void *)gra_cureditwindowframe);
	return(0);
}

void *us_firstwindowframe(void)
{
	return((void *)gra_firstwindowframe);
}

void *us_nextwindowframe(void *frame)
{
	WINDOWFRAME *mw;

	mw = (WINDOWFRAME *)frame;
	if (mw == 0 || mw == NOWINDOWFRAME) return(0);
	if (mw->nextwindowframe == NOWINDOWFRAME) return(0);
	return((void *)mw->nextwindowframe);
}

/*
 * routine to return size of window "win" in "wid" and "hei"
 */
void us_getwindowsize(void *frame, INTSML *wid, INTSML *hei)
{
	WINDOWFRAME *mw;

	mw = (WINDOWFRAME *)frame;
	*wid = mw->swid;
	*hei = mw->shei;
}

HWND gra_getwindow(void *frame)
{
	WINDOWFRAME *mw;

	mw = (WINDOWFRAME *)frame;
	return(mw->realwindow);
}

void us_sizewindowframe(void *frame, INTSML wid, INTSML hei)
{
	WINDOWFRAME *mw;
	RECT         fr, cr;
	int          framewid, framehei;

	mw = (WINDOWFRAME *)frame;

	/* if window frame already this size, stop now */
	if (wid == mw->swid && hei == mw->shei) return;

	/* resize window if needed */
	mw->wndframe->GetClientRect(&cr);
	if (wid != cr.right-cr.left || hei != cr.bottom-cr.top)
	{
		mw->wndframe->GetWindowRect(&fr);

		framewid = (fr.right - fr.left) - (cr.right - cr.left);
		framehei = (fr.bottom - fr.top) - (cr.bottom - cr.top);

		/* determine new window size */
		wid += framewid;
		hei += framehei;

		/* resize the window */
		if (mw->wndframe->SetWindowPos(0, fr.left, fr.top, wid, hei, 0) == 0)
		{
			// failed to resize window
			return;
		}
	}

	/* rebuild the offscreen windows */
	if (gra_remakeeditwindow(mw) != 0)
	{
//		fr.left += 1;   fr.right -= 2;   fr.bottom -= 2;
//		SetWindowPos(mw->realwindow, NULL, 0, 0, fr.right-fr.left, fr.bottom-fr.top,SWP_NOZORDER | SWP_NOMOVE);
		return;
	}
	gra_reloadmap();
}

void us_movewindowframe(void *frame, INTSML top, INTSML left)
{
	WINDOWFRAME *mw;
	RECT         fr;

	mw = (WINDOWFRAME *)frame;
	mw->wndframe->GetWindowRect(&fr);

	/* determine new window location */
	if (left == fr.left && top == fr.top) return;
	mw->wndframe->SetWindowPos(0, left, top, fr.right-fr.left, fr.bottom-fr.top, 0);
}

/*
 * Routine to close the messages window if it is in front.  Returns nonzero if the
 * window was closed.
 */
INTSML us_closefrontmostmessages(void)
{
	if (gra_messagescurrent != 0 && gra_messageswindow != 0)
	{
		gra_messageswindow->DestroyWindow();
		gra_messagescurrent = 0;
		gra_messageswindow = 0;
		return(1);
	}
	return(0);
}

/*
 * routine to return the important pieces of information in placing the floating
 * palette with the fixed menu.  The maximum screen size is placed in "wid" and
 * "hei", and the amount of space that is being left for this palette on the
 * side of the screen is placed in "palettewidth".
 */
void us_getpaletteparameters(INTSML *wid, INTSML *hei, INTSML *palettewidth)
{
	RECT r;
	CMainFrame *wnd;

	wnd = (CMainFrame *)AfxGetMainWnd();
	wnd->GetClientRect(&r);
	*hei = r.bottom - r.top - 30;
	*wid = r.right - r.left;
	*palettewidth = PALETTEWIDTH;
}

RECT *gra_geteditorwindowlocation(void)
{
	static RECT redit;
	static INTSML offset = 0;

	redit.top = offset + gra_screentop;
	redit.bottom = redit.top + (gra_screenbottom-gra_screentop) * 4 / 5 - 1;
	redit.left = offset + gra_screenleft + PALETTEWIDTH;
	redit.right = redit.left + (gra_screenright-gra_screenleft-PALETTEWIDTH) * 4 / 5;

	/* is the editing window visible on this display? */
	if ((redit.bottom > gra_screenbottom) || (redit.right > gra_screenright))
	{
		offset = 0;
		redit.top = gra_screentop;
		redit.bottom = redit.top + (gra_screenbottom-gra_screentop) * 3 / 5;
		redit.left = gra_screenleft + PALETTEWIDTH;
		redit.right = redit.left + (gra_screenright-gra_screenleft-PALETTEWIDTH) * 4 / 5;
	}
	offset += 30;
	return(&redit);
}

/*
 * routine to build a new window frame
 */
INTSML gra_buildwindow(WINDOWFRAME *mw, INTSML floating)
{
	RECT redit;
	int i;
	REGISTER VARIABLE *varred, *vargreen, *varblue;

	mw->floating = floating;
	mw->wndframe = 0;
	CChildFrame *tempFrame = new CChildFrame();
	if (floating == 0)
	{
		/* get editor window location */
		redit = *gra_geteditorwindowlocation();

		/* create the editing window */
		tempFrame->Create(NULL, "Electric",
			WS_CHILD|WS_OVERLAPPEDWINDOW|WS_VISIBLE, redit, NULL, NULL);
	} else
	{
		/* default palette window location */
		redit.top = gra_screentop + 2;
		redit.bottom = (gra_screentop+gra_screenbottom)/2;
		redit.left = gra_screenleft + 1;
		redit.right = redit.left + PALETTEWIDTH/2;

		/* create the floating window */
		BOOL good = tempFrame->Create(NULL, "Palette",
			WS_CHILD|WS_VISIBLE|WS_DLGFRAME, redit, NULL, NULL);
	}
	mw->wndframe = tempFrame;
	mw->realwindow = tempFrame->m_hWnd;    
	if (mw->realwindow == 0) return(1);

	mw->hDC = mw->wndframe->GetDC();
	mw->wndframe->ReleaseDC(mw->hDC);
	mw->wndframe->ShowWindow(SW_SHOW);
	mw->wndframe->SetActiveWindow();
	mw->hPalette = 0;
	
	/* load any map the first time to establish the map segment length */
	el_maplength = 256;

	/* create a color palette for the buffer */
	varred   = getvalkey((INTBIG)us_aid, VAID, VINTEGER|VISARRAY, us_colormap_red);
	vargreen = getvalkey((INTBIG)us_aid, VAID, VINTEGER|VISARRAY, us_colormap_green);
	varblue  = getvalkey((INTBIG)us_aid, VAID, VINTEGER|VISARRAY, us_colormap_blue);
	if (varred != NOVARIABLE && vargreen != NOVARIABLE && varblue != NOVARIABLE)
	{
		mw->pColorPalette = (LOGPALETTE *)emalloc(256 * (sizeof (PALETTEENTRY)) + 2 * (sizeof (WORD)),
			db_cluster);
		if (mw->pColorPalette == 0) return(1);
		mw->pColorPalette->palVersion = 0x300;
		mw->pColorPalette->palNumEntries = 256;
		for(i=0; i<256; i++)
		{
			mw->pColorPalette->palPalEntry[i].peRed   = (unsigned char)((INTBIG *)varred->addr)[i];
			mw->pColorPalette->palPalEntry[i].peGreen = (unsigned char)((INTBIG *)vargreen->addr)[i];
			mw->pColorPalette->palPalEntry[i].peBlue  = (unsigned char)((INTBIG *)varblue->addr)[i];
			mw->pColorPalette->palPalEntry[i].peFlags = 0;
		}
		mw->pColorPalette->palPalEntry[255].peRed   = (unsigned char)(255-((INTBIG *)varred->addr)[255]);
		mw->pColorPalette->palPalEntry[255].peGreen = (unsigned char)(255-((INTBIG *)vargreen->addr)[255]);
		mw->pColorPalette->palPalEntry[255].peBlue  = (unsigned char)(255-((INTBIG *)varblue->addr)[255]);
		mw->hPalette = new CPalette();
		mw->hPalette->CreatePalette(mw->pColorPalette);
	}

	/* build the offscreen buffer for the window */
	if (gra_makeeditwindow(mw) != 0)
	{
//		DestroyWindow(mw->realwindow);
		return(1);
	}

	return(0);
}

/*
 * Routine to redraw editing window "mw" due to a drag or grow
 */
void gra_redrawdisplay(WINDOWFRAME *mw)
{
	us_beginchanges();
	us_drawmenu(-1, (void *)mw);
	us_endchanges(NOWINDOW);
	us_state |= HIGHLIGHTSET;
	us_showallhighlight();
}

/*
 *  Routine to allocate the offscreen buffer that corresponds with the window
 *  "mw->realwindow".  The buffer is 8-bits deep and is stored in "mw->window".
 *  Returns nonzero if memory cannot be allocated.
 */
INTSML gra_makeeditwindow(WINDOWFRAME *mw)
{
	RECT r;
		
	/* get information about the window that was just created */
	mw->wndframe->GetClientRect(&r);

	/* set frame characteristics */
	mw->swid = (INTSML)r.right - r.left;
	mw->shei = r.bottom - r.top;
	mw->revy = mw->shei - 1;
	mw->offscreendirty = 0;
	mw->screenleft   = 0;
	mw->screenright  = mw->swid;
	mw->screentop    = 0;
	mw->screenbottom = mw->shei;

	/* allocate the main offscreen buffer */
	if (gra_buildoffscreenbuffer(mw, mw->swid, mw->shei, &mw->hDCOff, &mw->hBitmap,
		&mw->bminfo, &mw->data, &mw->rowstart) != 0) return(1);

	/* attach the color palette to drawing window section */
	if (mw->hPalette != 0)
	{
		(void)mw->hDCOff->SelectPalette(mw->hPalette, TRUE);
		(void)mw->hDCOff->RealizePalette();
	}

	return(0);
}

/*
 * Routine to build an offscreen buffer that is "wid" by "hei" and based on window frame "frame".
 * The device context is placed in "hdc", the bitmap in "bitmap", the actual offscreen memory in
 * "databuffer" and the row start array in "rowstart".  Returns nonzero on error.
 */
INTSML gra_buildoffscreenbuffer(WINDOWFRAME *frame, INTSML wid, INTSML hei, CDC **hdc,
	HBITMAP *bitmap, BITMAPINFO **bmiInfo, char **databuffer, char ***rowstart)
{
	BITMAPINFOHEADER bmiHeader;
	REGISTER VARIABLE *varred, *vargreen, *varblue;
	RGBQUAD bmiColors[256];
	REGISTER INTSML i, j, retval;
	REGISTER INTBIG size;
	REGISTER char *ptr;

	/* make the info structure */
	*bmiInfo = (BITMAPINFO *)emalloc(sizeof(bmiHeader) + sizeof(bmiColors), db_cluster);
	if (*bmiInfo == 0) return(1);

	/* grab the device context */
	frame->hDC = frame->wndframe->GetDC();

	/* setup the header block */
	bmiHeader.biSize = (DWORD)sizeof(bmiHeader); 
	bmiHeader.biWidth = (LONG)wid; 
	bmiHeader.biHeight = (LONG)-hei; 
	bmiHeader.biPlanes = 1;
	bmiHeader.biBitCount = 8;
	bmiHeader.biCompression = (DWORD)BI_RGB; 
	bmiHeader.biSizeImage = 0; 
	bmiHeader.biXPelsPerMeter = 0; 
	bmiHeader.biYPelsPerMeter = 0; 
	bmiHeader.biClrUsed = 256;
	bmiHeader.biClrImportant = 0; 

	/* get color map information */
	varred = getvalkey((INTBIG)us_aid, VAID, VINTEGER|VISARRAY, us_colormap_red);
	vargreen = getvalkey((INTBIG)us_aid, VAID, VINTEGER|VISARRAY, us_colormap_green);
	varblue = getvalkey((INTBIG)us_aid, VAID, VINTEGER|VISARRAY, us_colormap_blue);
	if (varred != NOVARIABLE && vargreen != NOVARIABLE && varblue != NOVARIABLE)
	{
		for(i=0; i<255; i++)
		{
			bmiColors[i].rgbRed = (unsigned char)((INTBIG *)varred->addr)[i];
			bmiColors[i].rgbGreen = (unsigned char)((INTBIG *)vargreen->addr)[i];
			bmiColors[i].rgbBlue = (unsigned char)((INTBIG *)varblue->addr)[i];
			bmiColors[i].rgbReserved = 0;
		}
		bmiColors[255].rgbRed = (unsigned char)(255-((INTBIG *)varred->addr)[255]);
		bmiColors[255].rgbGreen = (unsigned char)(255-((INTBIG *)vargreen->addr)[255]);
		bmiColors[255].rgbBlue = (unsigned char)(255-((INTBIG *)varblue->addr)[255]);
	}

	/* create a device context for this offscreen buffer */
	retval = 1;
	*hdc = new CDC();
	BOOL result = (*hdc)->CreateCompatibleDC(frame->hDC);
	if (result)
	{
		/* prepare structure describing offscreen buffer */
		ptr =  (char *)*bmiInfo;
		memcpy(ptr, &bmiHeader, sizeof(bmiHeader));
		ptr += sizeof(bmiHeader);
		memcpy(ptr, &bmiColors, sizeof(bmiColors));

		/* allocate offscreen buffer */
		*bitmap = CreateDIBSection((*hdc)->m_hDC, *bmiInfo, DIB_RGB_COLORS, (void **)databuffer, NULL, 0);
		if (*bitmap != 0)
		{
			(*hdc)->SelectObject(CBitmap::FromHandle(*bitmap));

			/* setup row pointers to offscreen buffer */
			*rowstart = (char **)emalloc(hei * (sizeof (char *)), db_cluster);
			if (*rowstart == 0) return(1);
			size = (wid+3)/4*4;
			for(j=0; j<hei; j++) (*rowstart)[j] = (*databuffer) + (size * j);
			retval = 0;
		}
	}

	/* release the device context */
	frame->wndframe->ReleaseDC(frame->hDC);
	return(retval);
}

/*
 * Routine to rebuild the offscreen buffer when its size or depth has changed.
 * Returns nonzero if the window cannot be rebuilt.
 */
INTSML gra_remakeeditwindow(WINDOWFRAME *mw)
{
	/* free memory associated with the frame */
	efree((char *)mw->rowstart);
	efree((char *)mw->bminfo);
	DeleteObject(mw->hBitmap);
	delete mw->hDCOff;

	return(gra_makeeditwindow(mw));
}

/******************** EXTERNAL ROUTINES ********************/

/*
 * routine to return display style of window "win"
 */
short us_getdispstyle(WINDOW *win)
{
	return(COLORMAP);
}

void us_startbatching(void) {}

void us_endbatching(void)
{
	flushscreen();
}

void us_setspy(INTSML mag) {}

/*
 * return nonzero if the capabilities in "want" are present
 */
INTSML us_graphicshas(INTSML want)
{
	/* cannot do "spyglass" magnification */
	if ((want&CANMAGNIFY) != 0) return(0);
	return(1);
}

/*
 * internal routine to beep the status display
 */
void ttybeep(void)
{
	if (us_aid->aidstate & TERMBEEP) MessageBeep(MB_ICONASTERISK);
}

/******************** MESSAGES WINDOW ROUTINES ********************/

/*
 * Routine to put the string "s" into the scrolling messages window.  Called from "pcterminal.c"
 */
void gra_putstring(char *s)
{
	int pos, line;

	/* ensure window exists, is noniconic, and is on top */
	if (gra_messageswindow == 0)
	   (void)gra_makemessageswindow();
	if (gra_messageswindow->IsIconic())
		gra_messageswindow->ShowWindow(SW_RESTORE);
	if (gra_messagesnotvisible())
		gra_messageswindow->BringWindowToTop();	

	/* get next line and load it up */
	line = gra_editCtrl->GetLineCount();
	pos = gra_editCtrl->LineIndex(line-1);
	gra_editCtrl->SetSel(pos, pos);
	gra_editCtrl->ReplaceSel(s);
	gra_editCtrl->ReplaceSel("\r\n");
}

int gra_msgchartyped(UINT nChar)
{
	if (gra_wantmsgwindowcharacters == 0)
	{
		gra_keyaction(nChar, 0, 1);
		return(0);
	}
	gra_msgwindowchartyped = nChar;
	return(1);
}

/*
 * Routine to get a string from the scrolling messages window.  Returns zero if end-of-file
 * (^D) is typed.
 */
char *gra_getstring(char *prompt)
{
	int line, pos, startpos;
	char cTmpCh, *pt;
	static char typedline[MAXTYPEDLINE];

	/* ensure window exists, is noniconic, and is on top */
	if (gra_messageswindow == 0)
	   (void)gra_makemessageswindow();
	if (gra_messageswindow->IsIconic())
		gra_messageswindow->ShowWindow(SW_RESTORE);
	if (gra_messagesnotvisible())
		gra_messageswindow->BringWindowToTop();	
	flushscreen();

	/* advance to next line and allocate maximum number of characters */
	gra_messageswindow->MDIActivate();
	line = gra_editCtrl->GetLineCount() - 1;
	pos = gra_editCtrl->LineIndex(line);
	pos += gra_editCtrl->LineLength(line);
	gra_editCtrl->SetSel(pos, pos);
	gra_editCtrl->ReplaceSel(prompt);
	startpos = gra_editCtrl->GetLine(line, typedline, MAXTYPEDLINE);

	CElectricMsgView *msgView = (CElectricMsgView *)gra_messageswindow->GetActiveView();
	msgView->m_forceToBottom = 1;
	gra_msgwindowchartyped = 0;
	gra_wantmsgwindowcharacters = 1;
	for(;;)
	{
		gra_nextevent(0);
		if (gra_msgwindowchartyped != 0)
		{
			cTmpCh = gra_msgwindowchartyped;
			if (cTmpCh == 0x04)
			{
				gra_editCtrl->ReplaceSel("\r\n");
				if (startpos == gra_editCtrl->GetLine(line, typedline, MAXTYPEDLINE))
				{
					msgView->m_forceToBottom = 0;
					gra_wantmsgwindowcharacters = 0;
					return(NULL);
				}
				cTmpCh = 0x0D;
			}
			if (cTmpCh == 0x0D)
			{
				gra_editCtrl->GetLine(line, typedline, MAXTYPEDLINE);
				pt = &typedline[startpos];
				msgView->m_forceToBottom = 0;
				gra_wantmsgwindowcharacters = 0;
				return(pt);
			}
		}
	}
}

/*
 * routine to select fonts in the messages window
 */
void gra_setfont(void)
{
	CFont *fnt;
	CFontDialog dlg;
	LOGFONT lf;

	if (dlg.DoModal() == IDOK)
	{
		/* Retrieve the dialog data */
		dlg.GetCurrentFont(&lf);
		fnt = new CFont();
		fnt->CreateFontIndirect(&lf);
		gra_editCtrl->SetFont(fnt);
	}
}

/*
 * routine to cut text from the messages window if it is current.  Returns nonzero
 * if sucessful.
 */
INTSML tty_cut(void)
{
	if (gra_messageswindow == 0) return(0);
	if (gra_messageswindow->IsIconic()) return(0);
	CMDIFrameWnd *parent = gra_messageswindow->GetMDIFrame();
	CWnd *wnd = parent->MDIGetActive();
	if (gra_messageswindow != wnd) return(0);	

	/* do the copy */
	gra_editCtrl->Cut();
	return(1);
}

/*
 * routine to copy text from the messages window if it is current.  Returns nonzero
 * if sucessful.
 */
INTSML tty_copy(void)
{
	if (gra_messageswindow == 0) return(0);
	if (gra_messageswindow->IsIconic()) return(0);
	CMDIFrameWnd *parent = gra_messageswindow->GetMDIFrame();
	CWnd *wnd = parent->MDIGetActive();
	if (gra_messageswindow != wnd) return(0);	

	/* do the copy */
	gra_editCtrl->Copy();
	return(1);
}

/*
 * routine to paste text to the messages window if it is current.  Returns nonzero
 * if sucessful.
 */
INTSML tty_paste(void)
{
	if (gra_messageswindow == 0) return(0);
	if (gra_messageswindow->IsIconic()) return(0);
	CMDIFrameWnd *parent = gra_messageswindow->GetMDIFrame();
	CWnd *wnd = parent->MDIGetActive();
	if (gra_messageswindow != wnd) return(0);	

	/* do the copy */
	gra_editCtrl->Paste();
	return(1);
}

/*
 * routine to get the contents of the system cut buffer
 */
char *tty_getcutbuffer(void)
{
	char *ptr;

	if (OpenClipboard(NULL) == 0) return("");
	ptr = (char *)GetClipboardData(CF_TEXT);
	(void)initinfstr();
	if (ptr != 0)
		(void)addstringtoinfstr(ptr);
	CloseClipboard();
	return(returninfstr());
}

/*
 * routine to set the contents of the system cut buffer to "msg"
 */
void tty_setcutbuffer(char *msg)
{
	HGLOBAL hnd;

	if (OpenClipboard(NULL) == 0) return;

	/* Remove the current Clipboard contents */
	if (EmptyClipboard() == 0)
	{
		CloseClipboard();
		return;
	}

	hnd = GlobalAlloc(GHND, strlen(msg)+1);
	strcpy(*((char **)hnd), msg);
	(void)SetClipboardData(CF_TEXT, hnd);

	CloseClipboard();
}

/*
 * Routine to return nonzero if the messages window is obscured by
 * an editor window.  Should ignore edit windows below this!!!
 */
INTSML gra_messagesnotvisible(void)
{
	RECT mr, r;
	CWnd *wnd;

	gra_messageswindow->GetWindowRect(&mr);
	mr.top += 2;   mr.bottom -= 2;
	mr.left += 2;  mr.right -= 2;

	/* look for "previous" windows that may obscure this one */
	wnd = gra_messageswindow;
	for(;;)
	{
		wnd = wnd->GetNextWindow(GW_HWNDPREV);
		if (wnd == 0) break;
		wnd->GetWindowRect(&r);
		if (r.left < mr.right && r.right > mr.left &&
			r.top < mr.bottom && r.bottom > mr.top) return(1);
	}
	return(0);
}

/******************** STATUS BAR ROUTINES ********************/

/*
 * Routine to return the number of status lines on the display.
 */
INTSML ttynumstatuslines(void)
{
	return(MAXSTATUSLINES);
}

/*
 * Routine to free status field object "sf".
 */
void ttyfreestatusfield(STATUSFIELD *sf)
{
	INTSML i, j;

	for(i=0; i<gra_indicatorcount; i++)
	{
		if (gra_statusfields[i] != sf) continue;

		/* remove the field */
		efree(gra_statusfieldtext[i]);
		for(j=i+1; j<gra_indicatorcount; j++)
		{
			gra_statusfields[j-1] = gra_statusfields[j];
			gra_statusfieldtext[j-1] = gra_statusfieldtext[j];
		}
		gra_indicatorcount--;
		gra_redrawstatusindicators();
		break;
	}

	efree(sf->label);
	efree((char *)sf);
}

/*
 * Routine to display "message" in the status "field" of window "frame" (uses all windows
 * if "frame" is zero).  If "cancrop" is zero, field cannot be cropped and should be
 * replaced with "*" if there isn't room.
 */
void ttysetstatusfield(void *frame, STATUSFIELD *field, char *message, INTSML cancrop)
{
	INTSML len, i, j, pos, newpos;
	REGISTER WINDOWFRAME *mw;
	CMainFrame *wnd;

	/* figure out which indicator this is */
	if (field == 0) return;

	/* construct the status line */
	gra_localstring[0] = 0;
	(void)strcat(gra_localstring, field->label);
	(void)strcat(gra_localstring, message);
	len = strlen(gra_localstring);
	while (len > 0 && gra_localstring[len-1] == ' ') gra_localstring[--len] = 0;

	/* if this is a title bar setting, do it now */
	if (field->line == 0)
	{
		if (frame == 0) mw = gra_cureditwindowframe; else
			mw = (WINDOWFRAME *)frame;
		if (mw == NOWINDOWFRAME) return;
		if (*gra_localstring == 0) strcpy(gra_localstring, "*** NO FACET ***");
		mw->wndframe->SetWindowText(gra_localstring);
		return;
	}

	/* ignore null fields */
	if (*field->label == 0) return;

	/* see if this indicator is in the list */
	wnd = (CMainFrame *)AfxGetMainWnd();
	for(i=0; i<gra_indicatorcount; i++)
	{
		if (gra_statusfields[i]->line != field->line || 
			gra_statusfields[i]->startper != field->startper || 
			gra_statusfields[i]->endper != field->endper) continue; 

		/* load the string */
		(void)reallocstring(&gra_statusfieldtext[i], gra_localstring, db_cluster);
		wnd->m_wndStatusBar.SetPaneText(i+1, gra_localstring);
		return;
	}

	/* not found: find where this field fits in the order */
	newpos = (field->line-1) * 100 + field->startper;
	for(i=0; i<gra_indicatorcount; i++)
	{
		pos = (gra_statusfields[i]->line-1) * 100 + gra_statusfields[i]->startper;
		if (newpos < pos) break;
	}
	for(j=gra_indicatorcount; j>i; j--)
	{
		gra_statusfieldtext[j] = gra_statusfieldtext[j-1];
		gra_statusfields[j] = gra_statusfields[j-1];
	}
	gra_statusfields[i] = field;
	(void)allocstring(&gra_statusfieldtext[i], gra_localstring, db_cluster);
	gra_indicatorcount++;

	/* reload all indicators */
	gra_redrawstatusindicators();
}

void gra_redrawstatusindicators(void)
{
	INTSML i, ind, wid, screenwid, totalwid;
	RECT r;
	CMainFrame *wnd;

	wnd = (CMainFrame *)AfxGetMainWnd();
	wnd->GetClientRect(&r);
	screenwid = r.right - r.left - gra_indicatorcount * 5;
	wnd->m_wndStatusBar.SetIndicators(gra_indicators, gra_indicatorcount+1);
	totalwid = 0;
	for(i=0; i<gra_indicatorcount; i++)
		totalwid += gra_statusfields[i]->endper - gra_statusfields[i]->startper;
	for(i=0; i<=gra_indicatorcount; i++)
	{
		if (i == 0)
		{
			wnd->m_wndStatusBar.SetPaneInfo(i, gra_indicators[i], SBPS_NOBORDERS, 0);
			wnd->m_wndStatusBar.SetPaneText(i, "");
		} else
		{
			wid = gra_statusfields[i-1]->endper - gra_statusfields[i-1]->startper;
			wid = wid * screenwid / totalwid;
			wnd->m_wndStatusBar.SetPaneInfo(i, gra_indicators[i], SBPS_NORMAL, wid);
			wnd->m_wndStatusBar.SetPaneText(i, gra_statusfieldtext[i-1]);
		}
	}
}

/******************** GRAPHICS CONTROL ROUTINES ********************/

void flushscreen(void)
{
	WINDOWFRAME *mw;

	for(mw = gra_firstwindowframe; mw != NOWINDOWFRAME; mw = mw->nextwindowframe)
	{
		/* if screen has not changed, stop now */
		if (mw->offscreendirty == 0) continue;
		mw->offscreendirty = 0;

		/* refresh screen */
		mw->hDC = mw->wndframe->GetDC();
		mw->hDC->BitBlt(mw->copyrect.left, mw->copyrect.top, mw->copyrect.right-mw->copyrect.left,
			   mw->copyrect.bottom-mw->copyrect.top, mw->hDCOff, mw->copyrect.left, mw->copyrect.top, SRCCOPY);
		mw->wndframe->ReleaseDC(mw->hDC);
	}
}

/*
 * Routine to accumulate the rectangle of change to the offscreen PixMap
 */
void gra_setrect(WINDOWFRAME *mw, INTSML lx, INTSML hx, INTSML ly, INTSML hy)
{
	if (mw->offscreendirty == 0)
	{
		mw->copyrect.left = lx;   
		mw->copyrect.right = hx;
		mw->copyrect.top = ly;    
		mw->copyrect.bottom = hy;
		mw->offscreendirty = 1;
	} else
	{
		if (lx < mw->copyrect.left)   mw->copyrect.left = lx;
		if (hx > mw->copyrect.right)  mw->copyrect.right = hx;
		if (ly < mw->copyrect.top)    mw->copyrect.top = ly;
		if (hy > mw->copyrect.bottom) mw->copyrect.bottom = hy;
	}
}

void gra_reloadmap(void)
{
	REGISTER VARIABLE *varred, *vargreen, *varblue;

	varred = getvalkey((INTBIG)us_aid, VAID, VINTEGER|VISARRAY, us_colormap_red);
	vargreen = getvalkey((INTBIG)us_aid, VAID, VINTEGER|VISARRAY, us_colormap_green);
	varblue = getvalkey((INTBIG)us_aid, VAID, VINTEGER|VISARRAY, us_colormap_blue);
	if (varred == NOVARIABLE || vargreen == NOVARIABLE || varblue == NOVARIABLE) return;
	us_loadmap((INTBIG *)varred->addr, (INTBIG *)vargreen->addr, (INTBIG *)varblue->addr, 0, 255);
}

void us_loadmap(INTBIG *red, INTBIG *green, INTBIG *blue, INTSML low, INTSML high)
{
	INTSML i;
	REGISTER WINDOWFRAME *mw;
	RGBQUAD bmiColors[256];

	for(mw = gra_firstwindowframe; mw != NOWINDOWFRAME; mw = mw->nextwindowframe)
	{
		/* do not touch the color map when in B&W mode */
		if (mw->pColorPalette == 0) continue;

		i = GetDIBColorTable(mw->hDCOff->m_hDC, 0, 256, bmiColors);
		for(i=low; i<=high; i++)
		{
			bmiColors[i].rgbRed = (unsigned char)red[i-low];
			bmiColors[i].rgbGreen = (unsigned char)green[i-low];
			bmiColors[i].rgbBlue = (unsigned char)blue[i-low];
			bmiColors[i].rgbReserved = 0;
			if (i == 255)
			{
				bmiColors[i].rgbRed = (unsigned char)(255-red[i-low]);
				bmiColors[i].rgbGreen = (unsigned char)(255-green[i-low]);
				bmiColors[i].rgbBlue = (unsigned char)(255-blue[i-low]);
			}

			mw->pColorPalette->palPalEntry[i].peRed = bmiColors[i].rgbRed;
			mw->pColorPalette->palPalEntry[i].peGreen = bmiColors[i].rgbGreen;
			mw->pColorPalette->palPalEntry[i].peBlue = bmiColors[i].rgbBlue;
			mw->pColorPalette->palPalEntry[i].peFlags = 0;
		}		
		i = SetDIBColorTable(mw->hDCOff->m_hDC, 0, 256, bmiColors);

		if (mw->hPalette != 0)
			delete mw->hPalette;

		mw->hPalette = new CPalette();
		mw->hPalette->CreatePalette(mw->pColorPalette);

		/* graphics window */
		mw->hDCOff->SelectPalette(mw->hPalette, TRUE);
		mw->hDCOff->RealizePalette();
		gra_setrect(mw, 0, mw->swid-1, 0, mw->shei-1);
	}
}

/*
 * helper routine to set the cursor shape to "state"
 */
void gra_set_cursorstate(INTSML state)
{
	if (gra_cursorstate == state) return;

	switch (state)
	{
		case NORMALCURSOR:  SetCursor(gra_normalCurs);   break;
		case WANTTTYCURSOR: SetCursor(gra_wantttyCurs);  break;
		case PENCURSOR:     SetCursor(gra_penCurs);      break;
		case NULLCURSOR:    SetCursor(gra_normalCurs);   break;
		case MENUCURSOR:    SetCursor(gra_menuCurs);     break;
		case HANDCURSOR:    SetCursor(gra_handCurs);     break;
		case TECHCURSOR:    SetCursor(gra_techCurs);     break;
		case IBEAMCURSOR:   SetCursor(gra_ibeamCurs);    break;
	}
	ShowCursor(TRUE);
	gra_cursorstate = state;
}

/*
 * Routine to change the default cursor (to indicate modes)
 * If "curs" is 0, use the basic cursor.
 * If "curs" is 1, use the pen cursor (polygon edit mode).
 * If "curs" is 2, use the "T" cursor (technology edit mode).
 */
INTSML us_setnormalcursor(INTSML curs)
{
	INTSML lastcurs;

	switch (gra_normalcursor)
	{
		case NORMALCURSOR: lastcurs = 0;   break;
		case PENCURSOR:    lastcurs = 1;   break;
		case TECHCURSOR:   lastcurs = 2;   break;
	}
	switch (curs)
	{
		case 0: gra_normalcursor = NORMALCURSOR;   break;
		case 1: gra_normalcursor = PENCURSOR;      break;
		case 2: gra_normalcursor = TECHCURSOR;     break;
	}
	gra_set_cursorstate(gra_normalcursor);
	return(lastcurs);
}

/******************** GRAPHICS LINES ********************/

void us_drawline(WINDOW *win, INTSML x1, INTSML y1, INTSML x2, INTSML y2, GRAPHICS *desc, INTSML texture)
{
	REGISTER INTSML col, mask, lx, hx, ly, hy;
	static INTSML pat[] = {0xFF, 0x88, 0xE7, 0x80};
	REGISTER WINDOWFRAME *frame;

	/* make sure the image buffer has not moved */
	frame = (WINDOWFRAME *)win->frame;

	/* get line type parameters */
	col = desc->col;         mask = ~desc->bits;
	y1 = frame->revy - y1;   y2 = frame->revy - y2;

	if (texture != 0) gra_drawpatline(frame, x1, y1, x2, y2, col, mask, pat[texture]); else
		gra_drawline(frame, x1, y1, x2, y2, col, mask);
	if (x1 < x2) { lx = x1;   hx = x2; } else { lx = x2;   hx = x1; }
	if (y1 < y2) { ly = y1;   hy = y2; } else { ly = y2;   hy = y1; }
	gra_setrect(frame, lx, (INTSML)(hx+1), ly, (INTSML)(hy+1));
}

void us_invertline(WINDOW *win, INTSML x1, INTSML y1, INTSML x2, INTSML y2)
{
	REGISTER INTSML lx, hx, ly, hy, dx, dy, d, incr1, incr2, x, y, xend, yend, yincr, xincr;
	REGISTER WINDOWFRAME *frame;

	/* get line type parameters */
	frame = (WINDOWFRAME *)win->frame;
	y1 = frame->revy - y1;   y2 = frame->revy - y2;

	/* initialize the Bresenham algorithm */
	dx = abs(x2-x1);
	dy = abs(y2-y1);
	if (dx > dy)
	{
		/* initialize for lines that increment along X */
		incr1 = 2 * dy;
		d = incr2 = 2 * (dy - dx);
		if (x1 > x2)
		{
			x = x2;   y = y2;   xend = x1;   yend = y1;
		} else
		{
			x = x1;   y = y1;   xend = x2;   yend = y2;
		}
		if (yend < y) yincr = -1; else yincr = 1;
		frame->rowstart[y][x] = ~frame->rowstart[y][x];

		/* draw line that increments along X */
		while (x < xend)
		{
			x++;
			if (d < 0) d += incr1; else
			{
				y += yincr;   d += incr2;
			}
			frame->rowstart[y][x] = ~frame->rowstart[y][x];
		}
	} else
	{
		/* initialize for lines that increment along Y */
		incr1 = 2 * dx;
		d = incr2 = 2 * (dx - dy);
		if (y1 > y2)
		{
			x = x2;   y = y2;   xend = x1;   yend = y1;
		} else
		{
			x = x1;   y = y1;   xend = x2;   yend = y2;
		}
		if (xend < x) xincr = -1; else xincr = 1;
		frame->rowstart[y][x] = ~frame->rowstart[y][x];

		/* draw line that increments along X */
		while (y < yend)
		{
			y++;
			if (d < 0) d += incr1; else
			{
				x += xincr;   d += incr2;
			}
			frame->rowstart[y][x] = ~frame->rowstart[y][x];
		}
	}
	if (x1 < x2) { lx = x1;   hx = x2; } else { lx = x2;   hx = x1; }
	if (y1 < y2) { ly = y1;   hy = y2; } else { ly = y2;   hy = y1; }
	gra_setrect(frame, lx, (INTSML)(hx+1), ly, (INTSML)(hy+1));
}

void gra_drawpatline(WINDOWFRAME *frame, INTSML x1, INTSML y1, INTSML x2, INTSML y2, INTSML col,
	INTSML mask, INTSML pattern)
{
	short dx, dy, d, incr1, incr2, x, y, xend, yend, yincr, xincr, i;

	/* initialize counter for line style */
	i = 0;

	/* initialize the Bresenham algorithm */
	dx = abs(x2-x1);
	dy = abs(y2-y1);
	if (dx > dy)
	{
		/* initialize for lines that increment along X */
		incr1 = 2 * dy;
		d = incr2 = 2 * (dy - dx);
		if (x1 > x2)
		{
			x = x2;   y = y2;   xend = x1;   yend = y1;
		} else
		{
			x = x1;   y = y1;   xend = x2;   yend = y2;
		}
		if (yend < y) yincr = -1; else yincr = 1;
		frame->rowstart[y][x] = (frame->rowstart[y][x]&mask) | col;

		/* draw line that increments along X */
		while (x < xend)
		{
			x++;
			if (d < 0) d += incr1; else
			{
				y += yincr;
				d += incr2;
			}
			if (i == 7) i = 0; else i++;
			if ((pattern & (1 << i)) == 0) continue;
			frame->rowstart[y][x] = (frame->rowstart[y][x]&mask) | col;
		}
	} else
	{
		/* initialize for lines that increment along Y */
		incr1 = 2 * dx;
		d = incr2 = 2 * (dx - dy);
		if (y1 > y2)
		{
			x = x2;   y = y2;   xend = x1;   yend = y1;
		} else
		{
			x = x1;   y = y1;   xend = x2;   yend = y2;
		}
		if (xend < x) xincr = -1; else xincr = 1;
		frame->rowstart[y][x] = (frame->rowstart[y][x]&mask) | col;

		/* draw line that increments along X */
		while (y < yend)
		{
			y++;
			if (d < 0) d += incr1; else
			{
				x += xincr;
				d += incr2;
			}
			if (i == 7) i = 0; else i++;
			if ((pattern & (1 << i)) == 0) continue;
			frame->rowstart[y][x] = (frame->rowstart[y][x]&mask) | col;
		}
	}
}

void gra_drawline(WINDOWFRAME *frame, INTSML x1, INTSML y1, INTSML x2, INTSML y2, INTSML col, INTSML mask)
{
	short dx, dy, d, incr1, incr2, x, y, xend, yend, yincr, xincr;

	/* initialize the Bresenham algorithm */
	dx = abs(x2-x1);
	dy = abs(y2-y1);
	if (dx > dy)
	{
		/* initialize for lines that increment along X */
		incr1 = 2 * dy;
		d = incr2 = 2 * (dy - dx);
		if (x1 > x2)
		{
			x = x2;   y = y2;   xend = x1;   yend = y1;
		} else
		{
			x = x1;   y = y1;   xend = x2;   yend = y2;
		}
		if (yend < y) yincr = -1; else yincr = 1;
		frame->rowstart[y][x] = (frame->rowstart[y][x]&mask) | col;

		/* draw line that increments along X */
		while (x < xend)
		{
			x++;
			if (d < 0) d += incr1; else
			{
				y += yincr;
				d += incr2;
			}
			frame->rowstart[y][x] = (frame->rowstart[y][x]&mask) | col;
		}
	} else
	{
		/* initialize for lines that increment along Y */
		incr1 = 2 * dx;
		d = incr2 = 2 * (dx - dy);
		if (y1 > y2)
		{
			x = x2;   y = y2;   xend = x1;   yend = y1;
		} else
		{
			x = x1;   y = y1;   xend = x2;   yend = y2;
		}
		if (xend < x) xincr = -1; else xincr = 1;
		frame->rowstart[y][x] = (frame->rowstart[y][x]&mask) | col;

		/* draw line that increments along X */
		while (y < yend)
		{
			y++;
			if (d < 0) d += incr1; else
			{
				x += xincr;
				d += incr2;
			}
			frame->rowstart[y][x] = (frame->rowstart[y][x]&mask) | col;
		}
	}
}

/******************** GRAPHICS POLYGONS ********************/

void us_drawpolygon(WINDOW *win, INTBIG *x, INTBIG *y, INTSML count, GRAPHICS *desc)
{
	REGISTER INTBIG i, j, k, l, ycur, yrev, wrap, lx, hx, ly, hy;
	REGISTER char *row;
	REGISTER INTSML col, mask, style, pat;
	REGISTER POLYSEG *a, *active, *edge, *lastedge, *left, *edgelist;
	REGISTER WINDOWFRAME *frame;

	/* get parameters */
	frame = (WINDOWFRAME *)win->frame;
	col = desc->col;   mask = ~desc->bits;
	style = desc->style[COLORMAP] & NATURE;

	/* set redraw area */
	for(i=0; i<count; i++)
	{
		if (i == 0)
		{
			lx = hx = x[i];
			ly = hy = y[i];
		} else
		{
			lx = mini(lx, x[i]);
			hx = maxi(hx, x[i]);
			ly = mini(ly, y[i]);
			hy = maxi(hy, y[i]);
		}
	}
	gra_setrect(frame, (INTSML)lx, (INTSML)(hx + 1), (INTSML)(frame->revy-hy),
		(INTSML)(frame->revy-ly + 1));

	/* make sure there is room in internal structures */
	if (count > gra_polysegcount)
	{
		if (gra_polysegcount > 0) efree((char *)gra_polysegs);
		gra_polysegcount = 0;
		gra_polysegs = (POLYSEG *)emalloc(count * (sizeof (POLYSEG)), us_aid->cluster);
		if (gra_polysegs == 0) return;
		gra_polysegcount = count;
	}

	/* fill in internal structures */
	edgelist = NOPOLYSEG;
	for(i=0; i<count; i++)
	{
		if (i == 0)
		{
			gra_polysegs[i].fx = x[count-1];
			gra_polysegs[i].fy = y[count-1];
		} else
		{
			gra_polysegs[i].fx = x[i-1];
			gra_polysegs[i].fy = y[i-1];
		}
		gra_polysegs[i].tx = x[i];   gra_polysegs[i].ty = y[i];
		gra_polysegs[i].index = i;

		/* compute the direction of this edge */
		j = gra_polysegs[i].ty - gra_polysegs[i].fy;
		if (j > 0) gra_polysegs[i].direction = 1; else
			if (j < 0) gra_polysegs[i].direction = -1; else
				gra_polysegs[i].direction = 0;

		/* compute the X increment of this edge */
		if (j == 0) gra_polysegs[i].increment = 0; else
		{
			gra_polysegs[i].increment = gra_polysegs[i].tx - gra_polysegs[i].fx;
			if (gra_polysegs[i].increment != 0) gra_polysegs[i].increment =
				(gra_polysegs[i].increment * 65536 - j + 1) / j;
		}
		gra_polysegs[i].tx <<= 16;   gra_polysegs[i].fx <<= 16;

		/* make sure "from" is above "to" */
		if (gra_polysegs[i].fy > gra_polysegs[i].ty)
		{
			j = gra_polysegs[i].tx;
			gra_polysegs[i].tx = gra_polysegs[i].fx;
			gra_polysegs[i].fx = j;
			j = gra_polysegs[i].ty;
			gra_polysegs[i].ty = gra_polysegs[i].fy;
			gra_polysegs[i].fy = j;
		}

		/* insert this edge into the edgelist, sorted by ascending "fy" */
		if (edgelist == NOPOLYSEG)
		{
			edgelist = &gra_polysegs[i];
			gra_polysegs[i].nextedge = NOPOLYSEG;
		} else
		{
			/* insert by ascending "fy" */
			if (edgelist->fy > gra_polysegs[i].fy)
			{
				gra_polysegs[i].nextedge = edgelist;
				edgelist = &gra_polysegs[i];
			} else for(a = edgelist; a != NOPOLYSEG; a = a->nextedge)
			{
				if (a->nextedge == NOPOLYSEG ||
					a->nextedge->fy > gra_polysegs[i].fy)
				{
					/* insert after this */
					gra_polysegs[i].nextedge = a->nextedge;
					a->nextedge = &gra_polysegs[i];
					break;
				}
			}
		}
	}

	/* scan polygon and render */
	active = NOPOLYSEG;
	while (active != NOPOLYSEG || edgelist != NOPOLYSEG)
	{
		if (active == NOPOLYSEG)
		{
			active = edgelist;
			active->nextactive = NOPOLYSEG;
			edgelist = edgelist->nextedge;
			ycur = active->fy;
		}

		/* introduce edges from edge list into active list */
		while (edgelist != NOPOLYSEG && edgelist->fy <= ycur)
		{
			/* insert "edgelist" into active list, sorted by "fx" coordinate */
			if (active->fx > edgelist->fx ||
				(active->fx == edgelist->fx && active->increment > edgelist->increment))
			{
				edgelist->nextactive = active;
				active = edgelist;
				edgelist = edgelist->nextedge;
			} else for(a = active; a != NOPOLYSEG; a = a->nextactive)
			{
				if (a->nextactive == NOPOLYSEG ||
					a->nextactive->fx > edgelist->fx ||
						(a->nextactive->fx == edgelist->fx &&
							a->nextactive->increment > edgelist->increment))
				{
					/* insert after this */
					edgelist->nextactive = a->nextactive;
					a->nextactive = edgelist;
					edgelist = edgelist->nextedge;
					break;
				}
			}
		}

		/* generate regions to be filled in on current scan line */
		wrap = 0;
		left = active;
		for(edge = active; edge != NOPOLYSEG; edge = edge->nextactive)
		{
			wrap = wrap + edge->direction;
			if (wrap == 0)
			{
				j = (left->fx + 32768) >> 16;
				k = (edge->fx + 32768) >> 16;
				yrev = frame->revy - ycur;
				row = frame->rowstart[yrev];
				if (style == PATTERNED)
				{
					/* patterned fill */
					pat = desc->raster[yrev&7];
					if (pat != 0)
					{
						for(l=j; l<=k; l++)
						{
							if ((pat & (1 << (15-(l&15)))) != 0)
								row[l] = (row[l] & mask) | col;
						}
					}
				} else
				{
					/* solid fill */
					for(l=j; l<=k; l++)
						row[l] = (row[l] & mask) | col;
				}
				left = edge->nextactive;
			}
		}
		ycur++;

		/* update edges in active list */
		lastedge = NOPOLYSEG;
		for(edge = active; edge != NOPOLYSEG; edge = edge->nextactive)
		{
			if (ycur >= edge->ty)
			{
				if (lastedge == NOPOLYSEG) active = edge->nextactive;
					else lastedge->nextactive = edge->nextactive;
			} else
			{
				edge->fx += edge->increment;
				lastedge = edge;
			}
		}
	}
}

/******************** GRAPHICS BOXES ********************/

void us_drawbox(WINDOW *win, INTSML lowx, INTSML highx, INTSML lowy, INTSML highy, GRAPHICS *desc)
{
	REGISTER INTSML col, mask, style, x, y, pat, top, bottom, left, right;
	REGISTER char *thisrow;
	REGISTER WINDOWFRAME *frame;

	frame = (WINDOWFRAME *)win->frame;
	
	/* get graphics parameters */
	col = desc->col;   mask = ~desc->bits;
	style = desc->style[COLORMAP] & NATURE;
	left = lowx;                     right = highx + 1;
	bottom = frame->revy-lowy + 1;   top = frame->revy-highy;

	if (style == PATTERNED)
	{
		/* special case the patterned fill */
		for(y=top; y<bottom; y++)
		{
			pat = desc->raster[y&7];
			if (pat == 0) continue;
			thisrow = frame->rowstart[y];
			for(x=left; x<right; x++)
			{
				if ((pat & (1 << (15-(x&15)))) != 0)
					thisrow[x] = (thisrow[x] & mask) | col;
			}
		}
	} else
	{
		for(y=top; y<bottom; y++)
		{
			thisrow = frame->rowstart[y];
			for(x=left; x<right; x++)
				thisrow[x] = (thisrow[x] & mask) | col;
		}
	}

	gra_setrect(frame, left, right, top, bottom);
}

/*
 * routine to invert the bits in the box from (lowx, lowy) to (highx, highy)
 */
void us_invertbox(WINDOW *win, INTSML lowx, INTSML highx, INTSML lowy, INTSML highy)
{
	REGISTER INTSML top, bottom, left, right, x, y;
	REGISTER char *thisrow;
	REGISTER WINDOWFRAME *frame;

	frame = (WINDOWFRAME *)win->frame;
	left = lowx;                  
	right = highx + 1;
	bottom = frame->revy - lowy + 1;
	top = frame->revy - highy;

	for(y=top; y<bottom; y++)
	{
		thisrow = frame->rowstart[y];
		for(x=left; x<right; x++) thisrow[x] = ~thisrow[x];
	}

	gra_setrect(frame, left, right, top, bottom);
}

/*
 * routine to move bits on the display starting with the area at
 * (sx,sy) and ending at (dx,dy).  The size of the area to be
 * moved is "wid" by "hei".
 */
void us_movebox(WINDOW *win, INTSML sx, INTSML sy, INTSML wid, INTSML hei, INTSML dx, INTSML dy)
{
	REGISTER INTSML totop, tobottom, toleft, toright, fromleft, fromtop;
	REGISTER WINDOWFRAME *frame;

	frame = (WINDOWFRAME *)win->frame;

	/* setup source rectangle */
	fromleft = sx;
	fromtop = frame->revy + 1 - sy - hei;

	/* setup destination rectangle */
	toleft = dx;
	toright = toleft + wid;
	totop = frame->revy + 1 - dy - hei;
	tobottom = totop + hei;

	frame->hDCOff->BitBlt(toleft, totop, wid, hei, frame->hDCOff, 
		fromleft, fromtop, SRCCOPY);

	gra_setrect(frame, toleft, toright, totop, tobottom);
}

/*
 * routine to save the contents of the box from "lx" to "hx" in X and from
 * "ly" to "hy" in Y.  A code is returned that identifies this box for
 * overwriting and restoring.  The routine returns -1 if there is a error.
 */
INTBIG us_savebox(WINDOW *win, INTSML lx, INTSML hx, INTSML ly, INTSML hy)
{
	SAVEDBOX *box;
	REGISTER INTBIG xsize, ysize;
	REGISTER INTSML i, x, y;
	REGISTER WINDOWFRAME *frame;
	REGISTER char *source, *dest;

	frame = (WINDOWFRAME *)win->frame;
	i = ly;   
	ly = frame->revy-hy;   
	hy = frame->revy-i;
	xsize = hx-lx+1;
	ysize = hy-ly+1;
	box = (SAVEDBOX *)emalloc((sizeof (SAVEDBOX)), us_aid->cluster);
	if (box == 0) return(-1);

	if (gra_buildoffscreenbuffer(frame, (INTSML)xsize, (INTSML)ysize, &box->hMemDC,
		&box->hBox, &box->bminfo, &box->data, &box->rowstart) != 0) return(-1);

	box->win = win;
	box->nextsavedbox = gra_firstsavedbox;
	gra_firstsavedbox = box;
	box->lx = lx;           box->hx = hx;
	box->ly = ly;           box->hy = hy;

	/* move the bits */
	for(y=0; y<ysize; y++)
	{
		source = &frame->rowstart[ly+y][lx];
		dest = box->rowstart[y];
		for(x=0; x<xsize; x++) *dest++ = *source++;
	}

	return((INTBIG)box);
}

/*
 * routine to shift the saved box "code" so that it is restored in a different location,
 * offset by (dx,dy)
 */
void us_movesavedbox(INTBIG code, INTSML dx, INTSML dy)
{
	REGISTER SAVEDBOX *box;

	if (code == -1) return;
	box = (SAVEDBOX *)code;
	box->lx += dx;       box->hx += dx;
	box->ly -= dy;       box->hy -= dy;
}

/*
 * routine to restore saved box "code" to the screen.  "destroy" is:
 *  0   restore box, do not free memory
 *  1   restore box, free memory
 * -1   free memory
 * Returns nonzero if there is an error.
 */
INTSML us_restorebox(INTBIG code, INTSML destroy)
{
	REGISTER SAVEDBOX *box, *lbox, *tbox;
	REGISTER INTSML x, y, xsize, ysize;
	REGISTER char *source, *dest;
	REGISTER WINDOWFRAME *frame;

	/* get the box */
	if (code == -1) return(1);
	box = (SAVEDBOX *)code;

	/* move the bits */
	if (destroy >= 0)
	{
		frame = (WINDOWFRAME *)box->win->frame;
		xsize = box->hx - box->lx + 1;
		ysize = box->hy - box->ly + 1;
		for(y=0; y<ysize; y++)
		{
			dest = &frame->rowstart[box->ly+y][box->lx];
			source = box->rowstart[y];
			for(x=0; x<xsize; x++) *dest++ = *source++;
		}
		efree((char *)box->rowstart);
		efree((char *)box->bminfo);
		DeleteObject(box->hBox);
		delete box->hMemDC;
		gra_setrect(frame, box->lx, (INTSML)(box->hx+1), box->ly, (INTSML)(box->hy+1));
	}

	/* destroy this box's memory if requested */
	if (destroy != 0)
	{
		lbox = NOSAVEDBOX;
		for(tbox = gra_firstsavedbox; tbox != NOSAVEDBOX; tbox = tbox->nextsavedbox)
		{
			if (tbox == box) 
				break;
			lbox = tbox;
		}
		if (lbox == NOSAVEDBOX) 
			gra_firstsavedbox = box->nextsavedbox; 
		else
			lbox->nextsavedbox = box->nextsavedbox;
		efree((char *)box);
	}
	return(0);
}

/******************** GRAPHICS TEXT ********************/

void us_settextsize(WINDOW *win, INTSML fnt)
{
	REGISTER WINDOWFRAME *frame;
	static CFont *txt4p = 0, *txt6p = 0, *txt8p = 0, *txt10p = 0,
		*txt12p = 0, *txt14p = 0, *txt16p = 0, *txt18p = 0, *txt20p = 0,
		*txtsmall = 0, *txtmedium = 0, *txtlarge = 0,
		*txtstatus = 0, *txteditor = 0, *txtmenu = 0;

	frame = (WINDOWFRAME *)win->frame;
	frame->hFont = 0;
	switch (fnt)
	{
		case TXT4P:
			if (txt4p == 0)     txt4p = gra_createtextfont(TXT4P);
			frame->hFont = txt4p;         break;
		case TXT6P:
			if (txt6p == 0)     txt6p = gra_createtextfont(TXT6P);
			frame->hFont = txt6p;         break;
		case TXT8P:
			if (txt8p == 0)     txt8p = gra_createtextfont(TXT8P);
			frame->hFont = txt8p;         break;
		case TXT10P:
			if (txt10p == 0)    txt10p = gra_createtextfont(TXT10P);
			frame->hFont = txt10p;        break;
		case TXT12P:
			if (txt12p == 0)    txt12p = gra_createtextfont(TXT12P);
			frame->hFont = txt12p;        break;
		case TXT14P:
			if (txt14p == 0)    txt14p = gra_createtextfont(TXT14P);
			frame->hFont = txt14p;        break;
		case TXT16P:
			if (txt16p == 0)    txt16p = gra_createtextfont(TXT16P);
			frame->hFont = txt16p;        break;
		case TXT18P:
			if (txt18p == 0)    txt18p = gra_createtextfont(TXT18P);
			frame->hFont = txt18p;        break;
		case TXT20P:
			if (txt20p == 0)    txt20p = gra_createtextfont(TXT20P);
			frame->hFont = txt20p;        break;

		case TXTSMALL:
			if (txtsmall == 0)  txtsmall = gra_createtextfont(TXTSMALL);
			frame->hFont = txtsmall;      break;
		case TXTMEDIUM:
			if (txtmedium == 0) txtmedium = gra_createtextfont(TXTMEDIUM);
			frame->hFont = txtmedium;     break;
		case TXTLARGE:
			if (txtlarge == 0)  txtlarge = gra_createtextfont(TXTLARGE);
			frame->hFont = txtlarge;      break;

		case TXTSTATUS:
			if (txtstatus == 0) txtstatus = gra_createtextfont(TXTSTATUS);
			frame->hFont = txtstatus;     break;
		case TXTEDITOR:
			if (txteditor == 0) txteditor = gra_createtextfont(TXTEDITOR);
			frame->hFont = txteditor;     break;
		case TXTMENU:
			if (txtmenu == 0)   txtmenu = gra_createtextfont(TXTMENU);
			frame->hFont = txtmenu;       break;
	}

	if (frame->hFont == 0) return;   
	frame->hDCOff->SelectObject(frame->hFont);
}

CFont *gra_createtextfont(INTSML whichFont)
{
	LOGFONT lf;
	CFont *hf;
	INTSML i;

	/* special case for text editor font */
	for(i=0; fontList[i].eFontIndex != -1; i++)
	{
		if (fontList[i].eFontIndex == whichFont)
		{
			lf.lfHeight = fontList[i].fontSize;
			strcpy(lf.lfFaceName, fontList[i].fontName);
			break;
		}
	}

	lf.lfWidth = 0;
	lf.lfEscapement = 0;
	lf.lfOrientation = 0;
	lf.lfWeight = FW_NORMAL;
	lf.lfItalic = 0;
	lf.lfUnderline = 0;
	lf.lfStrikeOut = 0;
	lf.lfCharSet = 0;
	lf.lfOutPrecision = OUT_STROKE_PRECIS;
	lf.lfClipPrecision = CLIP_STROKE_PRECIS;
	lf.lfQuality = 1;
	lf.lfPitchAndFamily = FF_DONTCARE | FF_SWISS;

	hf = new CFont();
	hf->CreateFontIndirect(&lf);

	return(hf);
}

void us_textsize(WINDOW *win, char *str, INTSML *x, INTSML *y)
{
	INTSML len;
	CSize textSize;
	REGISTER WINDOWFRAME *frame;

	frame = (WINDOWFRAME *)win->frame;
	len = strlen(str);
	textSize = frame->hDCOff->GetTextExtent((LPCTSTR)str, len);
	*x = (INTSML)textSize.cx;
	*y = (INTSML)textSize.cy+1;
}

void us_puttext(WINDOW *win, INTSML atx, INTSML aty, char *s, GRAPHICS *desc)
{
	REGISTER INTSML i, len, col, top, bottom, left, right, mask, j, wid, hei, pos;
	CSize textSize;
	REGISTER WINDOWFRAME *frame;
	REGISTER char *ptr, *dest;

	/* copy the string and correct it */
	len = strlen(s);
	if (len == 0) return;
	for(i=0; i<len; i++)
	{
		gra_localstring[i] = s[i] & 0177;
		if (gra_localstring[i] < ' ') gra_localstring[i] = ' ';
	}
	gra_localstring[i] = 0;

	/* get parameters */
	frame = (WINDOWFRAME *)win->frame;
	col = desc->col;   mask = ~desc->bits;
	aty = frame->revy - aty;

	/* determine size of string */
	textSize = frame->hDCOff->GetTextExtent((LPCTSTR)gra_localstring, len);
	wid = (INTSML)textSize.cx;
	hei = (INTSML)textSize.cy;

	/* text is too high - rescale it? */
	if (aty-hei <= 0) return;

	/* see if text buffer needs to be (re)initialized */
	if (gra_textbufinited != 0)
	{
		if (wid > gra_textbufwid || hei > gra_textbufhei)
		{
			efree((char *)gra_textrowstart);
			efree((char *)gra_textbitmapinfo);
			DeleteObject(gra_textbitmap);
			delete gra_texthdc;
			gra_textbufinited = 0;
		}
	}

	/* allocate text buffer if needed */
	if (gra_textbufinited == 0)
	{
		if (gra_buildoffscreenbuffer(frame, wid, hei, &gra_texthdc, &gra_textbitmap,
			&gra_textbitmapinfo, &gra_textdatabuffer, &gra_textrowstart) != 0) return;
		gra_texthdc->SetTextColor(PALETTEINDEX(1));
		gra_texthdc->SetROP2(R2_COPYPEN);
		gra_texthdc->SetBkMode(TRANSPARENT);
		gra_texthdc->SetTextAlign(TA_TOP<<8);

		/* remember information about text buffer */
		gra_textbufwid = wid;   gra_textbufhei = hei;
		gra_textbufinited = 1;
	}

	/* clear the text buffer */
	for(i=0; i<hei; i++)
	{
		ptr = gra_textrowstart[i];
		for(j=0; j<wid; j++) *ptr++ = 0;
	}

	/* write to the text buffer */
	gra_texthdc->SelectObject(frame->hFont);
	gra_texthdc->TextOut(0, 0, (LPCTSTR)gra_localstring, len);

	/* copy text buffer to the main offscreen buffer */
	pos = aty - hei;
	for(i=0; i<hei; i++)
	{
		ptr = gra_textrowstart[i];
		dest = frame->rowstart[pos+i];
		for(j=0; j<wid; j++)
		{
			if (ptr[j] == 0) continue;
			dest[atx + j] = (dest[atx + j] & mask) | col;
		}
	}

	/* set redraw area */
	left = atx;     right = left + wid;
	bottom = aty;   top = bottom - hei;
	gra_setrect(frame, left, right, top, bottom);
}

/******************** CIRCLES ********************/

void us_drawcircle(WINDOW *win, INTBIG atx, INTBIG aty, INTBIG radius, GRAPHICS *desc)
{
	REGISTER INTSML col, mask, top, bottom, left, right;
	REGISTER INTBIG x, y, d, maxx, maxy, thisx, thisy;
	REGISTER WINDOWFRAME *frame;
	REGISTER char *thisrow;

	/* get parameters */
	frame = (WINDOWFRAME *)win->frame;
	col = desc->col;   mask = ~desc->bits;
	aty = frame->revy - aty;

	/* set redraw area */
	left = (INTSML)(atx - radius);
	right = (INTSML)(atx + radius + 1);
	top = (INTSML)(aty - radius);
	bottom = (INTSML)(aty + radius + 1);
	gra_setrect(frame, left, right, top, bottom);

	maxx = frame->swid;
	maxy = frame->shei;
	x = 0;   y = radius;
	d = 3 - 2 * radius;
	if (left >= 0 && right < maxx && top >= 0 && bottom < maxy)
	{
		/* no clip version is faster */
		while (x <= y)
		{
			thisrow = frame->rowstart[aty + y];
			thisrow[atx + x] = (thisrow[atx + x] & mask) | col;
			thisrow[atx - x] = (thisrow[atx - x] & mask) | col;

			thisrow = frame->rowstart[aty - y];
			thisrow[atx + x] = (thisrow[atx + x] & mask) | col;
			thisrow[atx - x] = (thisrow[atx - x] & mask) | col;

			thisrow = frame->rowstart[aty + x];
			thisrow[atx + y] = (thisrow[atx + y] & mask) | col;
			thisrow[atx - y] = (thisrow[atx - y] & mask) | col;

			thisrow = frame->rowstart[aty - x];
			thisrow[atx + y] = (thisrow[atx + y] & mask) | col;
			thisrow[atx - y] = (thisrow[atx - y] & mask) | col;

			if (d < 0) d += 4*x + 6; else
			{
				d += 4 * (x-y) + 10;
				y--;
			}
			x++;
		}
	} else
	{
		/* clip version */
		while (x <= y)
		{
			thisy = aty + y;
			if (thisy >= 0 && thisy < maxy)
			{
				thisrow = frame->rowstart[thisy];
				thisx = atx + x;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = (thisrow[thisx] & mask) | col;
				thisx = atx - x;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = (thisrow[thisx] & mask) | col;
			}

			thisy = aty - y;
			if (thisy >= 0 && thisy < maxy)
			{
				thisrow = frame->rowstart[thisy];
				thisx = atx + x;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = (thisrow[thisx] & mask) | col;
				thisx = atx - x;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = (thisrow[thisx] & mask) | col;
			}

			thisy = aty + x;
			if (thisy >= 0 && thisy < maxy)
			{
				thisrow = frame->rowstart[thisy];
				thisx = atx + y;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = (thisrow[thisx] & mask) | col;
				thisx = atx - y;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = (thisrow[thisx] & mask) | col;
			}

			thisy = aty - x;
			if (thisy >= 0 && thisy < maxy)
			{
				thisrow = frame->rowstart[thisy];
				thisx = atx + y;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = (thisrow[thisx] & mask) | col;
				thisx = atx - y;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = (thisrow[thisx] & mask) | col;
			}

			if (d < 0) d += 4*x + 6; else
			{
				d += 4 * (x-y) + 10;
				y--;
			}
			x++;
		}
	}
}

/*
 * routine to draw a filled-in circle of radius "radius"
 */
void gra_drawdiscrow(WINDOWFRAME *frame, INTBIG thisy, INTBIG startx, INTBIG endx, GRAPHICS *desc)
{
	REGISTER char *thisrow;
	REGISTER INTBIG x;
	REGISTER INTSML pat;

	if (thisy < 0 || thisy >= gra_curvemaxy) return;
	thisrow = frame->rowstart[thisy];
	if (startx < 0) startx = 0;
	if (endx >= gra_curvemaxx) endx = gra_curvemaxy - 1;
	if (gra_curvestyle == PATTERNED)
	{
		pat = desc->raster[thisy&7];
		if (pat != 0)
		{
			for(x=startx; x<=endx; x++)
				if ((pat & (1 << (15-(x&15)))) != 0)
					thisrow[x] = (thisrow[x] & gra_curvemask) | gra_curvecol;
		}
	} else
	{
		for(x=startx; x<=endx; x++)
			thisrow[x] = (thisrow[x] & gra_curvemask) | gra_curvecol;
	}
}

/*
 * routine to draw a filled-in circle of radius "radius"
 */
void us_drawdisc(WINDOW *win, INTBIG atx, INTBIG aty, INTBIG radius, GRAPHICS *desc)
{
	REGISTER WINDOWFRAME *frame;
	REGISTER INTBIG x, y, d;
	REGISTER INTSML top, bottom, left, right;

	/* get parameters */
	frame = (WINDOWFRAME *)win->frame;
	gra_curvestyle = desc->style[COLORMAP] & NATURE;
	gra_curvecol = desc->col;   gra_curvemask = ~desc->bits;

	/* set redraw area */
	aty = frame->revy - aty;
	left = (INTSML)(atx - radius);
	right = (INTSML)(atx + radius + 1);
	top = (INTSML)(aty - radius);
	bottom = (INTSML)(aty + radius + 1);
	gra_setrect(frame, left, right, top, bottom);

	gra_curvemaxx = frame->swid;
	gra_curvemaxy = frame->shei;
	x = 0;   y = radius;
	d = 3 - 2 * radius;
	while (x <= y)
	{
		gra_drawdiscrow(frame, aty+y, atx-x, atx+x, desc);
		gra_drawdiscrow(frame, aty-y, atx-x, atx+x, desc);
		gra_drawdiscrow(frame, aty+x, atx-y, atx+y, desc);
		gra_drawdiscrow(frame, aty-x, atx-y, atx+y, desc);

		if (d < 0) d += 4*x + 6; else
		{
			d += 4 * (x-y) + 10;
			y--;
		}
		x++;
	}
}

/******************** ARC DRAWING ********************/

INTSML gra_arcfindoctant(INTBIG x, INTBIG y)
{
	if (x > 0)
		if (y >= 0)
			if (y >= x)	 return 7;
			else         return 8;
		else
			if (x >= -y) return 1;
			else         return 2;
	else
		if (y > 0)
			if (y > -x)  return 6;
			else         return 5;
		else
			if (y > x)   return 4;
			else         return 3;
}

void gra_arcxformoctant(INTBIG x, INTBIG y, INTBIG oct, INTBIG *ox, INTBIG *oy)
{
	switch (oct)
	{
		case 1 : *ox = -y;   *oy = x;   break;
		case 2 : *ox = x;    *oy = -y;  break;
		case 3 : *ox = -x;   *oy = -y;  break;
		case 4 : *ox = -y;   *oy = -x;  break;
		case 5 : *ox = y;    *oy = -x;  break;
		case 6 : *ox = -x;   *oy = y;   break;
		case 7 : *ox = x;    *oy = y;   break;
		case 8 : *ox = y;    *oy = x;   break;
	}
}

void gra_arcdopixel(INTBIG x, INTBIG y)
{
	if (x < 0 || x >= gra_curvemaxx || y < 0 || y >= gra_curvemaxy) return;
	if (gra_arcfirst != 0)
	{
		gra_arcfirst = 0;
		gra_arclx = gra_archx = x;
		gra_arcly = gra_archy = y;
	} else
	{
		if (x < gra_arclx) gra_arclx = x;
		if (x > gra_archx) gra_archx = x;
		if (y < gra_arcly) gra_arcly = y;
		if (y > gra_archy) gra_archy = y;
	}
	gra_arcframe->rowstart[y][x] = (gra_arcframe->rowstart[y][x] & gra_curvemask) | gra_curvecol;
}

void gra_arcoutxform(INTBIG x, INTBIG y)
{
	if (gra_arcocttable[1]) gra_arcdopixel( y + gra_arccenterx, -x + gra_arccentery);
	if (gra_arcocttable[2]) gra_arcdopixel( x + gra_arccenterx, -y + gra_arccentery);
	if (gra_arcocttable[3]) gra_arcdopixel(-x + gra_arccenterx, -y + gra_arccentery);
	if (gra_arcocttable[4]) gra_arcdopixel(-y + gra_arccenterx, -x + gra_arccentery);
	if (gra_arcocttable[5]) gra_arcdopixel(-y + gra_arccenterx,  x + gra_arccentery);
	if (gra_arcocttable[6]) gra_arcdopixel(-x + gra_arccenterx,  y + gra_arccentery);
	if (gra_arcocttable[7]) gra_arcdopixel( x + gra_arccenterx,  y + gra_arccentery);
	if (gra_arcocttable[8]) gra_arcdopixel( y + gra_arccenterx,  x + gra_arccentery);
}

void gra_arcbrescw(INTBIG x, INTBIG y, INTBIG x1, INTBIG y1)
{
	REGISTER INTBIG d;

	d = 3 - 2 * y + 4 * x;
	while (x < x1 && y > y1)
	{
		gra_arcoutxform(x, y);
		if (d < 0) d += 4*x+6; else
		{
			d += 4*(x-y)+10;
			y--;
		}
		x++;
	}

	/* get to the end */
	for ( ; x < x1; x++) gra_arcoutxform(x, y);
	for ( ; y > y1; y--) gra_arcoutxform(x, y);
   gra_arcoutxform(x1, y1);
}

void gra_arcbresmidcw(INTBIG x, INTBIG y)
{
	REGISTER INTBIG d;

	d = 3 - 2 * y + 4 * x;
	while (x < y)
	{
		gra_arcoutxform(x, y);
		if (d < 0) d += 4*x+6; else
		{
			d += 4*(x-y)+10;
			y--;
		}
		x++;
   }
   if (x == y) gra_arcoutxform(x, y);
}

void gra_arcbresmidccw(INTBIG x, INTBIG y)
{
	REGISTER INTBIG d;

	d = 3 + 2 * y - 4 * x;
	while (x > 0)
	{
		gra_arcoutxform(x, y);
		if (d > 0) d += 6-4*x; else
		{
			d += 4*(y-x)+10;
			y++;
		}
		x--;
   }
   gra_arcoutxform(0, gra_arcradius);
}

void gra_arcbresccw(INTBIG x, INTBIG y, INTBIG x1, INTBIG y1)
{
	REGISTER INTBIG d;

	d = 3 + 2 * y + 4 * x;
	while(x > x1 && y < y1)
	{
		/* not always correct */
		gra_arcoutxform(x, y);
		if (d > 0) d += 6 - 4*x; else
		{
			d += 4*(y-x)+10;
			y++;
		}
		x--;
	}

	/* get to the end */
	for ( ; x > x1; x--) gra_arcoutxform(x, y);
	for ( ; y < y1; y++) gra_arcoutxform(x, y);
	gra_arcoutxform(x1, y1);
}

/*
 * draws an arc centered at (centerx, centery), clockwise,
 * passing by (x1,y1) and (x2,y2)
 */
void us_drawcirclearc(WINDOW *win, INTBIG centerx, INTBIG centery, INTBIG p1_x, INTBIG p1_y,
	INTBIG p2_x, INTBIG p2_y, GRAPHICS *desc)
{
	REGISTER INTBIG alternate, pa_x, pa_y, pb_x, pb_y, i, diff;
	INTBIG x, y;
	REGISTER INTSML start_oct, end_oct;

	/* ignore tiny arcs */
	if (p1_x == p2_x && p1_y == p2_y) return;

	/* get parameters */
	gra_arcframe = (WINDOWFRAME *)win->frame;
	gra_curvecol = desc->col;   gra_curvemask = ~desc->bits;
	gra_curvemaxx = gra_arcframe->swid;
	gra_curvemaxy = gra_arcframe->shei;
	p1_y = gra_arcframe->revy - p1_y;
	p2_y = gra_arcframe->revy - p2_y;
	gra_arccentery = gra_arcframe->revy - centery;
	gra_arccenterx = centerx;
	pa_x = p2_x - gra_arccenterx;
	pa_y = p2_y - gra_arccentery;
	pb_x = p1_x - gra_arccenterx;
	pb_y = p1_y - gra_arccentery;
	gra_arcradius = computedistance(gra_arccenterx, gra_arccentery, p2_x, p2_y);
	alternate = computedistance(gra_arccenterx, gra_arccentery, p1_x, p1_y);
	start_oct = gra_arcfindoctant(pa_x, pa_y);
	end_oct   = gra_arcfindoctant(pb_x, pb_y);
	gra_arcfirst = 1;

	/* move the point */
	if (gra_arcradius != alternate)
	{
		diff = gra_arcradius-alternate;
		switch (end_oct)
		{
			case 6:
			case 7: /*  y >  x */ pb_y += diff;  break;
			case 8: /*  x >  y */ 
			case 1: /*  x > -y */ pb_x += diff;  break;
			case 2: /* -y >  x */
			case 3: /* -y > -x */ pb_y -= diff;  break;
			case 4: /* -y < -x */
			case 5: /*  y < -x */ pb_x -= diff;  break;
		}
	}

	for(i=1; i<9; i++) gra_arcocttable[i] = 0;

	if (start_oct == end_oct)
	{
		INTBIG x1, y1, x2, y2;

		gra_arcocttable[start_oct] = 1;
		gra_arcxformoctant(pa_x, pa_y, start_oct, &x1, &y1);
		gra_arcxformoctant(pb_x, pb_y, start_oct, &x2 ,&y2);

		if (ODD(start_oct)) gra_arcbrescw(x1, y1, x2, y2);
		else				gra_arcbresccw(x1 ,y1, x2, y2);
		gra_arcocttable[start_oct] = 0;
	} else
	{
		gra_arcocttable[start_oct] = 1;
		gra_arcxformoctant(pa_x, pa_y, start_oct, &x, &y);		
		if (ODD(start_oct)) gra_arcbresmidcw(x, y);
		else				gra_arcbresmidccw(x, y);
		gra_arcocttable[start_oct] = 0;

		gra_arcocttable[end_oct] = 1;
		gra_arcxformoctant(pb_x, pb_y, end_oct, &x, &y);
		if (ODD(end_oct)) gra_arcbresmidccw(x, y);
		else			  gra_arcbresmidcw(x, y);
		gra_arcocttable[end_oct] = 0;
	 
		if (MODP(start_oct+1) != end_oct)
		{
			if (MODP(start_oct+1) == MODM(end_oct-1))
				gra_arcocttable[MODP(start_oct+1)] = 1; else
					for(i = MODP(start_oct+1); i != end_oct; i = MODP(i+1))
						gra_arcocttable[i] = 1;
			gra_arcbresmidcw(0, gra_arcradius);
		}
	}

	/* set redraw area */
	if (gra_arcfirst == 0)
		gra_setrect(gra_arcframe, (INTSML)(gra_arclx), (INTSML)(gra_archx+1), (INTSML)(gra_arcly),
			(INTSML)(gra_archy+1));
}

/******************** GRID CONTROL ********************/

/*
 * grid drawing routine
 */
void us_drawgrid(WINDOW *win, POLYGON *obj)
{
	REGISTER INTBIG i, j, xnum, xden, ynum, yden, x0,y0, x1,y1, x2,y2, x3,y3,
		x4,y4, x5,y5, x10, y10, y10mod;
	REGISTER INTSML x, y;
	REGISTER WINDOWFRAME *frame;

	frame = (WINDOWFRAME *)win->frame;
	x0 = obj->xv[0];   y0 = obj->yv[0];		/* screen space grid spacing */
	x1 = obj->xv[1];   y1 = obj->yv[1];		/* screen space grid start */
	x2 = obj->xv[2];   y2 = obj->yv[2];		/* display space low */
	x3 = obj->xv[3];   y3 = obj->yv[3];		/* display space high */
	x4 = obj->xv[4];   y4 = obj->yv[4];		/* screen space low */
	x5 = obj->xv[5];   y5 = obj->yv[5];		/* screen space high */
	
	xnum = x3 - x2;
	xden = x5 - x4;
	ynum = y3 - y2;
	yden = y5 - y4;
	x10 = x0*10;       y10 = y0*10;

	/* draw the grid to the offscreen buffer */
	for(i = y1; i < y5; i += y0)
	{
		y = (INTSML)(muldiv(i-y4, ynum, yden) + y2);
		if (y < y2 || y > y3) continue;
		y = frame->revy - y;
		y10mod = i % y10;
		for(j = x1; j < x5; j += x0)
		{
			x = (INTSML)(muldiv(j-x4, xnum, xden) + x2);
			if (x >= x2 && x <= x3) frame->rowstart[y][x] |= GRID;

			/* special case every 10 grid points in each direction */
			if ((j%x10) == 0 && y10mod == 0)
			{
				if (x > x2) frame->rowstart[y][x-1] |= GRID;
				if (x < x3) frame->rowstart[y][x+1] |= GRID;
				if (y > y2) frame->rowstart[y-1][x] |= GRID;
				if (y < y3) frame->rowstart[y+1][x] |= GRID;
			}
		}
	}

	/* copy it back to the screen */
	gra_setrect(frame, (INTSML)x2, (INTSML)x3, (INTSML)(frame->revy-y3), (INTSML)(frame->revy-y2));
}

/******************** MOUSE CONTROL ********************/

/*
 * routine to return the number of buttons on the mouse
 */
INTSML buttoncount(void)
{
	return((INTSML)mini(BUTTONS, NUMBUTS));
}

/*
 * routine to tell whether button "but" is a double-click
 */
INTSML doublebutton(INTSML b)
{
	if (b >= 24 && b <= 26) return(1);
	return(0);
}

/*
 * routine to tell whether button "but" has the "shift" key held
 */
INTSML shiftbutton(INTSML b)
{
	if ((b >= 3 && b <= 5) || (b >= 12 && b <= 17) ||
		(b >= 21 && b <= 23)) return(1);
	return(0);
}

/*
 * routine to return the name of button "b" (from 0 to "buttoncount()").
 * The number of letters unique to the button is placed in "important".
 */
char *buttonname(INTSML b, INTSML *important)
{
	*important = gra_buttonname[b].unique;
	return(gra_buttonname[b].name);
}

/*
 * routine to convert from "gra_inputstate" (the typical input parameter)
 * to button numbers (the table "gra_buttonname")
 */
INTSML gra_makebutton(INTBIG state)
{
	REGISTER INTSML base;

	if ((state&ISLEFT) != 0) base = 0; else
		if ((state&ISMIDDLE) != 0) base = 1; else
			base = 2;

	if ((state&DOUBLECLICK) != 0) return(base+24);
	switch (state & (SHIFTDOWN|CONTROLDOWN|ALTDOWN))
	{
		case SHIFTDOWN                    : return(base+3);
		case           CONTROLDOWN        : return(base+6);
		case                       ALTDOWN: return(base+9);
		case SHIFTDOWN|CONTROLDOWN        : return(base+12);
		case SHIFTDOWN|            ALTDOWN: return(base+15);
		case           CONTROLDOWN|ALTDOWN: return(base+18);
		case SHIFTDOWN|CONTROLDOWN|ALTDOWN: return(base+21);
	}
	return(base);
}

/*
 * routine to wait for a button push and return its index (0 based) in "*but".
 * The coordinates of the cursor are placed in "*x" and "*y".  If there is no
 * button push, the value of "*but" is negative.
 */
void waitforbutton(INTSML *x, INTSML *y, INTSML *but)
{
	EventRecord theEvent;

	if (gra_inputstate != NOEVENT && (gra_inputstate&(ISBUTTON|BUTTONUP)) == ISBUTTON)
	{
		*but = gra_makebutton(gra_inputstate);
		*x = gra_cursorx;
		*y = gra_cursory;
		gra_inputstate = NOEVENT;
		if (gra_cursorstate != IBEAMCURSOR) gra_set_cursorstate(NULLCURSOR);
		return;
	}
	gra_waitforaction(1, &theEvent);
	if (gra_inputstate != NOEVENT && (gra_inputstate&(ISBUTTON|BUTTONUP)) == ISBUTTON)
	{
		*but = gra_makebutton(gra_inputstate);
		*x = gra_cursorx;
		*y = gra_cursory;
		gra_inputstate = NOEVENT;
		if (gra_cursorstate != IBEAMCURSOR) gra_set_cursorstate(NULLCURSOR);
		return;
	}
	*but = -1;
}

/*
 * routine to track the cursor until a button is released, calling "whileup" for
 * each co-ordinate when the mouse moves before the first button push, calling
 * "whendown" once when the button goes down, calling "eachdown" for each
 * co-ordinate when the mouse moves after the button is pushed, calling
 * "eachchar" for each key that is typed at any time, and calling "done" once
 * when done.  The "whendown" and "done" routines are called with no parameters;
 * "whileup" and "eachdown" are called with the X and Y coordinates of the
 * cursor; and "eachchar" is called with the X, Y, and character value that was
 * typed.  The "whileup", "eachdown", and "eachchar" routines return nonzero to
 * abort tracking.
 * If "waitforpush" is nonzero then the routine will wait for a button to
 * actually be pushed before tracking (otherwise it will begin tracking
 * immediately).  The value of "purpose" determines what the cursor will look
 * like during dragging: 0 for normal (the standard cursor), 1 for drawing (a pen),
 * 2 for dragging (a hand), 3 for popup menu selection (a horizontal arrow), 4 for
 * hierarchical popup menu selection (arrow, stays at end).
 */
void trackcursor(INTSML waitforpush, INTSML (*whileup)(INTBIG, INTBIG),
	void (*whendown)(void), INTSML (*eachdown)(INTBIG, INTBIG),
	INTSML (*eachchar)(INTBIG, INTBIG, INTSML), void (*done)(void), INTSML purpose)
{
	REGISTER INTSML keepon;
	EventRecord theEvent;
	INTBIG action;

	/* change the cursor to an appropriate icon */
	switch (purpose)
	{
		case TRACKDRAWING:    
			gra_set_cursorstate(PENCURSOR);    
			break;
		case TRACKDRAGGING:   
			gra_set_cursorstate(HANDCURSOR);   
			break;
		case TRACKSELECTING:
		case TRACKHSELECTING: 
			gra_set_cursorstate(MENUCURSOR);   
			break;
	}

	/* now wait for a button to go down, if requested */
	keepon = 0;
	if (waitforpush != 0)
	{
		while (keepon == 0)
		{
			gra_waitforaction(2, &theEvent);
			if (gra_inputstate == NOEVENT) continue;
			action = gra_inputstate;
			gra_inputstate = NOEVENT;

			/* if button just went down, stop this loop */
			if ((action&ISBUTTON) != 0 && (action&BUTTONUP) == 0) break;
			if ((action&MOTION) != 0)
			{
				keepon = (*whileup)(gra_cursorx, gra_cursory);
			} else if ((action&CHARREAD) != 0)
			{
				keepon = (*eachchar)(gra_cursorx, gra_cursory, (INTSML)(action&CHARREAD));
			}
			if (el_pleasestop != 0) keepon = 1;
		}
	}

	/* button is now down, real tracking begins */
	if (keepon == 0)
	{
		(*whendown)();
		keepon = (*eachdown)(gra_cursorx, gra_cursory);
	}
 
	/* now track while the button is down */
	while (keepon == 0)
	{
		gra_waitforaction(2, &theEvent);

		/* for each motion, report the coordinates */
		if (gra_inputstate == NOEVENT) continue;
		action = gra_inputstate;
		gra_inputstate = NOEVENT;
		if ((action&ISBUTTON) != 0 && (action&BUTTONUP) != 0) break;
	
		if ((action&MOTION) != 0)
		{
			keepon = (*eachdown)(gra_cursorx, gra_cursory);
		} else if ((action&CHARREAD) != 0)
		{
			keepon = (*eachchar)(gra_cursorx, gra_cursory, (INTSML)(action&CHARREAD));
		}
		if (el_pleasestop != 0) keepon = 1;
	}

	/* inform the user that all is done */
	(*done)();

	/* restore the state of the world */
	if (purpose != TRACKHSELECTING) gra_set_cursorstate(NULLCURSOR);
}

/*
 * routine to read the current co-ordinates of the tablet and return them
 * in "*x" and "*y".
 */
void readtablet(INTSML *x, INTSML *y)
{
	*x = gra_cursorx;   
	*y = gra_cursory;
}

/*
 * routine to turn off the cursor tracking if it is on
 */
void stoptablet(void)
{
	if (gra_cursorstate != IBEAMCURSOR) 
		gra_set_cursorstate(NULLCURSOR);
}

/******************** KEYBOARD CONTROL ********************/

/*
 * routine to get the next character from the keyboard
 */
INTSML us_getnxtchar(void)
{
	REGISTER INTSML i;
	EventRecord theEvent;

	if (gra_inputstate != NOEVENT && (gra_inputstate&CHARREAD) != 0)
	{
		i = gra_inputstate & CHARREAD;
		if ((gra_inputstate&COMMANDDOWN) != 0) i |= 0200;
		gra_inputstate = NOEVENT;
		return(i);
	}
	if (gra_cursorstate != IBEAMCURSOR) 
		gra_set_cursorstate(WANTTTYCURSOR);
	for(;;)
	{
		gra_waitforaction(0, &theEvent);
		if (gra_inputstate != NOEVENT && (gra_inputstate&CHARREAD) != 0)
			break;
	}
	i = gra_inputstate & CHARREAD;
	if ((gra_inputstate&COMMANDDOWN) != 0) i |= 0200;
	gra_inputstate = NOEVENT;

	if (gra_cursorstate != IBEAMCURSOR) 
		gra_set_cursorstate(NULLCURSOR);
	return(i);
}

void checkforinterrupt(void)
{
	gra_nextevent(0);
}

/*
 * routine to tell whether data is waiting at the terminal.  Returns nonzero
 * if data is ready.
 */
INTSML ttydataready(void)
{
	EventRecord theEvent;

	/* see if something is already pending */
	if (gra_inputstate != NOEVENT)
	{
		if ((gra_inputstate&CHARREAD) != 0) return(1);
	}
 
	/* wait for something and analyze it */
	gra_waitforaction(1, &theEvent);
	if (gra_inputstate != NOEVENT && (gra_inputstate&CHARREAD) != 0) return(1);
	return(0);
}

/*
 * routine to set the proper keyboard mode for input.  Returns nonzero
 * upon error.
 */
INTSML gra_onttyinput(void)
{
	return(0);
}

/*
 * routine to turn off the keyboard mode for input (back to normal)
 */
void gra_offttyinput(void) {}

/****************************** FILES ******************************/

/*
 * Routine to display a standard file prompt dialog and return the selected file.
 * The prompt message is in "msg" and the kind of file is in "filetype".  The default
 * output file name is in "defofile" (only used if "filetype" is negative).
 */
char *gra_fileselect(char *msg, INTSML filetype, char *defofile)
{
	INTSML kind = 0;
	INTBIG len;
	REGISTER INTSML i, j;
	char leng, ofile[256], fs[256], *extension, *filter, *shortname, *longname;
	char *af1 = "All Files (*.*)";
	char *af2 = "*.*";
	OPENFILENAME ofn;

	if (filetype & FILETYPEWRITE) kind = 1;	/* save file dialog */

	/* build filter string */
	describefiletype(filetype, &extension, &filter, &shortname, &longname);
	j = 0;
	for (i=0; i<strlen(longname); i++) fs[j++] = longname[i];
	fs[j++] = ' ';
	fs[j++] = '(';
	for (i=0; i<strlen(filter); i++) fs[j++] = filter[i];
	fs[j++] = ')';
	fs[j++] = 0;										/* terminate description */
	for (i=0; i<strlen(filter); i++) fs[j++] = filter[i];
	fs[j++] = 0;										/* terminate filter */
	for (i=0; i<strlen(af1); i++) fs[j++] = af1[i];
	fs[j++] = 0;										/* terminate description */
	for (i=0; i<strlen(af2); i++) fs[j++] = af2[i];
	fs[j++] = 0;										/* terminate filter */
	fs[j] = 0;											/* terminate complete filter string */

	if (us_logplay != NULL)
	{
		(void)xfread((char *)&gra_action, 1, sizeof (gra_action), us_logplay);
		if ((gra_action.kind&0xFFFF) != FILEREPLY) 
			gra_localstring[0] = 0; 
		else
		{
			(void)xfread(&leng, 1, 1, us_logplay);
			len = leng;
			(void)xfread(gra_localstring, 1, len, us_logplay);
			gra_localstring[len] = 0;
		}
	} else
	{
		switch (kind)
		{
			case 0:  /* open file dialog */
				gra_localstring[0] = 0;
				ofn.lStructSize = sizeof(OPENFILENAME); 
				ofn.hwndOwner = AfxGetMainWnd()->m_hWnd;
				ofn.hInstance = 0; 
				ofn.lpstrFilter = (LPCTSTR)fs; 
				ofn.lpstrCustomFilter = 0; 
				ofn.nMaxCustFilter = 0; 
				ofn.nFilterIndex = 0; 
				ofn.lpstrFile = gra_localstring; /* answer goes here */
				ofn.nMaxFile = 256; 
				ofn.lpstrFileTitle = ofile; 
				ofn.nMaxFileTitle = 256; 
				ofn.lpstrInitialDir = 0; 
				ofn.lpstrTitle = msg; 
				ofn.Flags = OFN_HIDEREADONLY | OFN_FILEMUSTEXIST; 
				ofn.nFileOffset = 0; 
				ofn.nFileExtension = 0; 
				ofn.lpstrDefExt = 0; 
				ofn.lCustData = 0; 
				ofn.lpfnHook = 0; 
				ofn.lpTemplateName = 0; 
				i = GetOpenFileName(&ofn);
				if (i == 0) gra_localstring[0] = 0;
				break;

			case 1:  /* save file dialog */
				for(i = strlen(defofile)-1; i > 0; i--)
					if (defofile[i] == ':' || defofile[i] == '/' || defofile[i] == '\\') break;
				if (i > 0) i++;
				(void)strcpy(gra_localstring, &defofile[i]);
				strcpy(ofile, "huh?");
				ofn.lStructSize = sizeof(OPENFILENAME); 
				ofn.hwndOwner = AfxGetMainWnd()->m_hWnd;
				ofn.hInstance = 0; 
				ofn.lpstrFilter = (LPCTSTR)fs; 
				ofn.lpstrCustomFilter = 0; 
				ofn.nMaxCustFilter = 0; 
				ofn.nFilterIndex = 0; 
				ofn.lpstrFile = gra_localstring; /* answer goes here */
				ofn.nMaxFile = 256; 
				ofn.lpstrFileTitle = ofile; 
				ofn.nMaxFileTitle = 256; 
				ofn.lpstrInitialDir = 0; 
				ofn.lpstrTitle = msg; 
				ofn.Flags = OFN_HIDEREADONLY | OFN_OVERWRITEPROMPT; 
				ofn.nFileOffset = 0; 
				ofn.nFileExtension = 0; 
				ofn.lpstrDefExt = 0; 
				ofn.lCustData = 0; 
				ofn.lpfnHook = 0; 
				ofn.lpTemplateName = 0; 
				i = GetSaveFileName(&ofn);
				if (i == 0) gra_localstring[0] = 0;
				break;
		}
		
	}

	/* log this result if logging */
	if (us_logrecord != NULL)
	{
		gra_action.kind = (INTSML)FILEREPLY;
		(void)xfwrite((char *)&gra_action, 1, sizeof (gra_action), us_logrecord);
		leng = strlen(gra_localstring);
		(void)xfwrite(&leng, 1, 1, us_logrecord);
		len = leng;
		(void)xfwrite(gra_localstring, 1, len, us_logrecord);
		gra_logrecordindex++;
		if (gra_logrecordindex >= us_logflushfreq)
		{
			/* flush the session log file */
			gra_logrecordindex = 0;
			xflushbuf(us_logrecord);
		}
	}

	if (gra_localstring[0] == 0) return ((char *)NULL); else
		return(gra_localstring);
}

/*
 * Helper routine to initialize the list of files in a directory.
 */
void gra_initfilelist(void)
{
	if (gra_fileliststringarray == 0)
	{
		gra_fileliststringarray = newstringarray(db_cluster);
		if (gra_fileliststringarray == 0) return;
	}
	clearstrings(gra_fileliststringarray);
}

/*
 * Helper routine to add "file" to the list of files in a directory.
 * Returns nonzero on error.
 */
INTSML gra_addfiletolist(char *file)
{
	addtostringarray(gra_fileliststringarray, file);
	return(0);
}

/*
 * Routine to search for all of the files/directories in directory "directory" and
 * return them in the array of strings "filelist".  Returns the number of files found.
 */
INTSML filesindirectory(char *directory, char ***filelist)
{
	HANDLE handle;
	WIN32_FIND_DATA data;
	BOOL found;
	char *dir;
	INTBIG len;

	/* search for all files in directory */
	(void)initinfstr();
	(void)addstringtoinfstr(directory);
	(void)addstringtoinfstr("*.*");
	dir = returninfstr();

	handle = FindFirstFile(dir, &data);
	if (handle == INVALID_HANDLE_VALUE) return(0);

	gra_initfilelist();
	for (found = 1; found; found = FindNextFile(handle, &data))
	{
		if (gra_addfiletolist(data.cFileName) != 0) return(0);
	}
	FindClose(handle);

	*filelist = getstringarray(gra_fileliststringarray, &len);
	return(len);
}

/* routine to convert a path name with "~" to a real path */
char *truepath(char *line)
{
	/* only have tilde parsing on UNIX */
	return(line);
}

/*
 * Routine to return information about the file or directory "name":
 *  0: does not exist
 *  1: is a file
 *  2: is a directory
 */
INTSML fileexistence(char *name)
{
	struct stat buf;

	if (stat(name, &buf) < 0) return(0);
	if ((buf.st_mode & S_IFMT) == S_IFDIR) return(2);
	return(1);
}

/*
 * Routine to create a directory.
 * Returns nonzero on error.
 */
INTSML createdirectory(char *dirname)
{
	if (mkdir(dirname) == 0) return(0);
	return(1);
}

/*
 * Routine to return the current directory name
 */
char *currentdirectory(void)
{
	char line[MAXPATHLEN];

	(void)initinfstr();
	(void)addstringtoinfstr(getcwd(line, MAXPATHLEN+1));
	return(returninfstr());
}

UINTBIG filedate(char *filename)
{
	struct stat buf;

	stat(filename, &buf);
	buf.st_mtime -= gra_machinetimeoffset();
	return(buf.st_mtime);
}

/*
 * Routine to lock a resource called "lockfilename" by creating such a file
 * if it doesn't exist.  Returns nonzero if successful, zero if unable to
 * lock the file.
 */
INTSML lockfile(char *lockfilename)
{
	int fd;

	fd = creat(lockfilename, 0);
	if (fd == -1) return(0);
	if (close(fd) == -1) return(0);
	return(1);
}

/*
 * Routine to unlock a resource called "lockfilename" by deleting such a file.
 */
void unlockfile(char *lockfilename)
{
	INTBIG attrs;
	attrs = GetFileAttributes(lockfilename);
	if ((attrs & FILE_ATTRIBUTE_READONLY) != 0)
	{
		if (SetFileAttributes(lockfilename, attrs & ~FILE_ATTRIBUTE_READONLY) == 0)
		{
			ttyputerr("Error removing readonly bit on %s (error %d)",
				lockfilename, GetLastError());
		}
	}

	if (DeleteFile(lockfilename) == 0)
		ttyputerr("Error unlocking %s (error %d)", lockfilename, GetLastError());
}

/*************************** TIME ROUTINES ***************************/

UINTBIG gra_machinetimeoffset(void)
{
	return(0);
}

UINTBIG ticktime(void)
{
	UINTBIG msTime;

	/* convert milliseconds to ticks */
	msTime = GetTickCount();
	return(msTime*6/100);
}

/*
 * Routine to start counting time.
 */
void starttimer(void)
{
	struct timeb timeptr;

	ftime(&timeptr);
	gra_timebasesec = timeptr.time;
	gra_timebasems = timeptr.millitm;
}

/*
 * Routine to stop counting time and return the number of elapsed seconds
 * since the last call to "starttimer()".
 */
float endtimer(void)
{
	float seconds;
	INTBIG milliseconds;
	struct timeb timeptr;

	ftime(&timeptr);
	milliseconds = (timeptr.time - gra_timebasesec) * 1000 + (timeptr.millitm-gra_timebasems);
	seconds = ((float)milliseconds) / 1000.0;
	return(seconds);
}

/*
 * Routine to wait "ticks" sixtieths of a second and then return.
 */
void gotosleep(INTBIG ticks)
{
	Sleep(ticks * 100 / 6);
}

/*************************** EVENT ROUTINES ***************************/

/*
 * helper routine to wait for some keyboard or mouse input.  The value of "nature" is:
 *   0  allow mouse, keyboard
 *   1  allow mouse, keyboard, pulldown menus
 *   2  allow mouse, keyboard, pulldown menus, motion
 */
void gra_waitforaction(INTSML nature, EventRecord *theEvent)
{
	REGISTER INTBIG err;
	static INTBIG fakewhen = 0;
	RECT r, fr;

	/* deal with special event types */
	if (gra_menustate == MENUEVENT)
	{
		gra_menustate = NOEVENT;
		gra_nativemenudoone(gra_lowmenu, gra_highmenu);
	} else if (gra_inputstate == WINDOWCHANGE) 
		gra_inputstate = NOEVENT;

	if (us_logplay != NULL)
	{
		gra_nextevent(4);
		flushscreen();
		err = xfread((char *)&gra_action, 1, sizeof (gra_action), us_logplay);
		if (stopping("Playback") != 0) err = 0;
		if (err != 0)
		{
			if (gra_playbackmultiple <= 0)
			{
				gra_playbackmultiple = 0;
				for(;;)
				{
					gra_inputstate = NOEVENT;
					gra_nextevent(0);
					if (gra_inputstate == NOEVENT) continue;
					if ((gra_inputstate&CHARREAD) == 0) continue;
					if ((gra_inputstate&CHARREAD) < '0' ||
						(gra_inputstate&CHARREAD) > '9') break;
					gra_playbackmultiple = gra_playbackmultiple * 10 + (gra_inputstate&CHARREAD) - '0';
				}
				if (gra_inputstate != NOEVENT && (gra_inputstate&CHARREAD) == 'q') err = 0;
			}
			gra_playbackmultiple--;

			/* allow Command-. to interrupt long playbacks */
			if ((gra_playbackmultiple%10) == 9)
			{
			}
			gra_inputstate = gra_action.kind & 0xFFFF;
			gra_cursorx = gra_action.x;
			gra_cursory = gra_action.y;
			if (gra_inputstate == MENUEVENT)
			{
				(void)xfread((char *)&gra_action, 1, sizeof (gra_action), us_logplay);
				gra_lowmenu = gra_action.x;
				gra_highmenu = gra_action.y;
			} else if (gra_inputstate == WINDOWCHANGE)
			{
				(void)xfread((char *)&r, 1, sizeof (r), us_logplay);
				GetWindowRect(gra_curwindowframe->realwindow, &fr);
				r.left += 1;         
				r.right -= 2;
				r.bottom -= 2;
			
				SetWindowPos(gra_curwindowframe->realwindow, NULL, r.left, r.top, 0,0, SWP_NOZORDER | SWP_NOSIZE);
				/* gra_MyGrowWindow(gra_curwindowframe->realwindow, (INTSML)(r.right-r.left), (INTSML)(r.bottom-r.top), &fr); */
			}
		}

		/* convert to an event */
#if 0
		fakewhen += gra_doubleclick + 1;
		theEvent->what = nullEvent;
		theEvent->when = fakewhen;
		theEvent->modifiers = 0;
		if ((gra_inputstate & SHIFTDOWN) != 0) 
			theEvent->modifiers |= shiftKey;
		if ((gra_inputstate & COMMANDDOWN) != 0) 
			theEvent->modifiers |= cmdKey;
		if ((gra_inputstate & ALTDOWN) != 0) 
			theEvent->modifiers |= optionKey;
		if ((gra_inputstate & CONTROLDOWN) != 0) 
		theEvent->modifiers |= controlKey;
		if ((gra_inputstate & BUTTONUP) != 0) 
			theEvent->modifiers |= btnState;
		theEvent->where.h = gra_cursorx;
		theEvent->where.v = gra_curwindowframe->revy - gra_cursory;
		if ((gra_inputstate & CHARREAD) != 0)
		{
			theEvent->what = keyDown;
			theEvent->message = gra_inputstate & CHARREAD;
		} else if ((gra_inputstate & ISBUTTON) != 0)
		{
			if ((gra_inputstate & BUTTONUP) == 0) 
				theEvent->what = mouseDown; 
			else
				theEvent->what = mouseUp;
			if ((gra_inputstate & DOUBLECLICK) != 0)
			{
				theEvent->when--;
				fakewhen--;
			}
		}
#endif

		/* stop now if end of playback file */
		if (err == 0)
		{
			ttyputmsg("End of session playback file");
			xclose(us_logplay);
			us_logplay = NULL;
			return;
		}
	} else
	{
		flushscreen();
		gra_nextevent(nature);
	}
	if (us_logrecord != NULL && gra_inputstate != NOEVENT)
	{
		/* shouldn't log this stuff - log file fills up real fast! */
		gra_action.kind = (INTSML)gra_inputstate;
		gra_action.x = gra_cursorx;
		gra_action.y = gra_cursory;
		if (xfwrite((char *)&gra_action, 1, sizeof (gra_action), us_logrecord) == 0)
		{
			ttyputerr("Error writing session log file: recording disabled");
				us_logfinishrecord();
		}

		if (gra_inputstate == MENUEVENT)
		{
			gra_action.x = gra_lowmenu;
			gra_action.y = gra_highmenu;
			(void)xfwrite((char *)&gra_action, 1, sizeof (gra_action), us_logrecord);
			gra_logrecordindex++;
		} else if (gra_inputstate == WINDOWCHANGE)
		{
			GetWindowRect(gra_curwindowframe->realwindow, &r);
			(void)xfwrite((char *)&r, 1, sizeof (r), us_logrecord);
			gra_logrecordindex++;
		}
		if (gra_logrecordindex >= us_logflushfreq)
		{
			/* flush the session log file */
			gra_logrecordindex = 0;
			xflushbuf(us_logrecord);
		}
	}
}

void gra_addeventtoqueue(INTBIG state, INTSML x, INTSML y)
{
	MYEVENTQUEUE *next;

	next = gra_eventqueuetail + 1;
	if (next >= &gra_eventqueue[EVENTQUEUESIZE])
		next = gra_eventqueue;
	if (next == gra_eventqueuehead)
	{
		/* queue is full */
		MessageBeep(MB_ICONASTERISK);
		return;
	}

	gra_eventqueuetail->inputstate = state;
	gra_eventqueuetail->cursorx = x;
	gra_eventqueuetail->cursory = y;
	gra_eventqueuetail = next;
}

/*
 * Routine to get the next Electric input action and set the global "gra_inputstate"
 * accordingly.  The value of "nature" is:
 *   0  allow mouse, keyboard (no window switching)
 *   1  allow mouse, keyboard, pulldown menus
 *   2  allow mouse, keyboard, motion
 *   4  allow update and activate events only
 */
void gra_nextevent(INTSML nature)
{
	MSG msg;

	while ( ::PeekMessage(&msg, NULL, 0, 0, PM_NOREMOVE))
	{
		if (!theApp.PumpMessage())
		{
			::PostQuitMessage(0);
			break;
		}
	}

	if (gra_eventqueuehead != gra_eventqueuetail)
	{
		gra_inputstate = gra_eventqueuehead->inputstate;
		gra_cursorx = gra_eventqueuehead->cursorx;
		gra_cursory = gra_eventqueuehead->cursory;
		gra_eventqueuehead++;
		if (gra_eventqueuehead >= &gra_eventqueue[EVENTQUEUESIZE])
			gra_eventqueuehead = gra_eventqueue;
	}
}

void gra_buttonaction(int state, UINT nFlags, CPoint point, CWnd *frm)
{
	WINDOWFRAME *mw;
	INTBIG event;
	INTSML x, y, item;

	for(mw = gra_firstwindowframe; mw != NOWINDOWFRAME; mw = mw->nextwindowframe)
		if (mw->wndframe == frm) break;
	if (mw == NOWINDOWFRAME)
	{
		if (frm != gra_dialogwindow) return;
		item = gra_dodialogisinsideuserdrawn(point.x, point.y);
		if (item == 0) return;
		if (state != 1)
		{
			gra_itemclicked(item-1+ID_DIALOGITEM_0);
			return;
		}
	}

	/* set appropriate button */
	if ((nFlags & MK_LBUTTON) != 0) event = ISBUTTON|ISLEFT;
	if ((nFlags & MK_MBUTTON) != 0) event = ISBUTTON|ISMIDDLE;
	if ((nFlags & MK_RBUTTON) != 0) event = ISBUTTON|ISRIGHT;

	/* add in extras */
	if ((nFlags & MK_SHIFT) != 0) event |= SHIFTDOWN;
	if ((nFlags & MK_CONTROL) != 0) event |= CONTROLDOWN;
	if ((GetKeyState(VK_MENU)&0x8000) != 0) event |= ALTDOWN;
	if (state == 2 && (event&(SHIFTDOWN|CONTROLDOWN|ALTDOWN)) == 0)
		event |= DOUBLECLICK;
	if (state == 1) event |= BUTTONUP;

	x = point.x;
	if (mw == NOWINDOWFRAME) y = point.y; else
		y = mw->revy - point.y;
	us_state |= DIDINPUT;
	gra_addeventtoqueue(event, x, y);
}

void gra_mouseaction(UINT nFlags, CPoint point, CWnd *frm)
{
	WINDOWFRAME *mw;
	INTBIG event;
	INTSML x, y, setcursor;
	REGISTER WINDOW *w;
	REGISTER EDITOR *e;

	if (gra_inputstate != NOEVENT) return;
	for(mw = gra_firstwindowframe; mw != NOWINDOWFRAME; mw = mw->nextwindowframe)
		if (mw->wndframe == frm) break;
	if (mw == NOWINDOWFRAME)
	{
		if (frm != gra_dialogwindow) return;
		if (gra_dodialogisinsideuserdrawn(point.x, point.y) == 0) return;
	}

	x = point.x;
	if (mw == NOWINDOWFRAME) y = point.y; else
	{
		y = mw->revy - point.y;
		setcursor = 0;
		for(w = el_topwindow; w != NOWINDOW; w = w->nextwindow)
		{
			if (w->frame != mw) continue;
			if (x < w->uselx || x > w->usehx || y < w->usely || y > w->usehy) continue;
			if ((w->state&WINDOWTYPE) == POPTEXTWINDOW ||
				(w->state&WINDOWTYPE) == TEXTWINDOW)
			{
				e = w->editor;
				if ((e->state&EDITORTYPE) == PACEDITOR)
				{
					if (x > w->usehx - SBARWIDTH || y < w->usely + SBARWIDTH || y >= e->revy)
						gra_set_cursorstate(NORMALCURSOR); else
							gra_set_cursorstate(IBEAMCURSOR);
					setcursor = 1;
				}
			}
		}
		if (setcursor == 0)
		{
			if ((gra_cursorstate == NORMALCURSOR || gra_cursorstate == PENCURSOR ||
				gra_cursorstate == TECHCURSOR) && gra_cursorstate != gra_normalcursor)
					gra_set_cursorstate(gra_normalcursor);
		}
	}
	event = MOTION;

	if ((nFlags & (MK_LBUTTON|MK_MBUTTON|MK_RBUTTON)) == 0)
		event |= BUTTONUP;
	gra_addeventtoqueue(event, x, y);
}

void gra_keyaction(UINT nChar, UINT nFlags, UINT nRepCnt)
{
	POINT pt;
	INTBIG event;
	INTSML x, y;

	if (nChar == 015) nChar = 012;
	GetCursorPos(&pt);
	x = pt.x;
	if (gra_curwindowframe == NOWINDOWFRAME) y = pt.y; else
		y = gra_curwindowframe->revy - pt.y;
	event = nChar & CHARREAD;
	if ((nChar&0200) != 0) event |= COMMANDDOWN;
	us_state |= DIDINPUT;
	gra_addeventtoqueue(event, x, y);
}

void gra_setdefaultcursor(void)
{
	int curstate;

	curstate = gra_cursorstate;
	gra_cursorstate++;
	gra_set_cursorstate(curstate);
}

void gra_activateframe(CChildFrame *frame, BOOL bActivate)
{
	WINDOWFRAME *mw;
	WINDOW *w;

	if (!bActivate)
	{
		gra_curwindowframe = NOWINDOWFRAME;
		return;
	}

	for(mw = gra_firstwindowframe; mw != NOWINDOWFRAME; mw = mw->nextwindowframe)
	{
		if (mw->wndframe != frame) continue;
		gra_curwindowframe = mw;
		if (mw->floating != 0) continue;
		gra_cureditwindowframe = mw;

		/* see if the change of window frame invalidates the current window */
		if (el_curwindow == NOWINDOW || el_curwindow->frame != mw)
		{
			/* must choose new window */
			for(w = el_topwindow; w != NOWINDOW; w = w->nextwindow)
			{
				if (w->frame == mw)
				{
					(void)setvalkey((INTBIG)us_aid, VAID, us_current_window, (INTBIG)w,
						VWINDOW|VDONTSAVE);
					(void)setval((INTBIG)el_curlib, VLIBRARY, "curnodeproto",
						(INTBIG)w->curnodeproto, VNODEPROTO);
					break;
				}
			}
		}
	}
}

int gra_closeframe(CChildFrame *frame)
{
	WINDOWFRAME *mw;
	char *par[2];

	if (frame == gra_messageswindow)
	{
		gra_messagescurrent = 0;
		gra_messageswindow = 0;
		return(1);
	}

	for(mw = gra_firstwindowframe; mw != NOWINDOWFRAME; mw = mw->nextwindowframe)
	{
		if (frame == mw->wndframe) break;
	}
	if (mw == NOWINDOWFRAME) return(0);

	mw->wndframe->SetActiveWindow();
	par[0] = "delete";
	us_window(1, par);
	return(0);
}

int gra_closeworld(void)
{
	char *par[20];

	if (us_preventloss(0, par, NOLIBRARY, "Quit")) return(1);  /* keep working */
	return(0);
}

void gra_repaint(CChildFrame *frame, CPaintDC *dc)
{
	WINDOWFRAME *mw;

	for(mw = gra_firstwindowframe; mw != NOWINDOWFRAME; mw = mw->nextwindowframe)
	{
		if (frame == mw->wndframe)
		{
			dc->BitBlt(0, 0, mw->swid, mw->shei, mw->hDCOff, 0, 0, SRCCOPY);
			break;
		}
	}
}

void gra_resize(CChildFrame *frame, int cx, int cy)
{
	WINDOWFRAME *mw;

	if (frame->IsIconic()) return;
	for(mw = gra_firstwindowframe; mw != NOWINDOWFRAME; mw = mw->nextwindowframe)
	{
		if (frame == mw->wndframe)
		{
			us_sizewindowframe(mw, cx, cy);
			gra_redrawdisplay(mw);
			break;
		}
	}
}

void gra_resizemain(int cx, int cy)
{
	gra_redrawstatusindicators();
	if (gra_palettewindowframe != NOWINDOWFRAME)
		us_drawmenu(-1, (void *)gra_palettewindowframe);
}

void gra_msgwindowactive(BOOL bActivate)
{
	if (bActivate) gra_messagescurrent = 1; else
		gra_messagescurrent = 0;
}

void gra_msgwindowchange(void)
{
	RECT r;
	WINDOWPLACEMENT wp;

	gra_messageswindow->GetWindowPlacement(&wp);
	r = wp.rcNormalPosition;
	gra_messagesleft = r.left;
	gra_messagesright = r.right;
	gra_messagestop = r.top;
	gra_messagesbottom = r.bottom;
}

/*************************** SESSION LOGGING ROUTINES ***************************/

/*
 * routine to begin playback of session logging file "file".  The routine
 * returns nonzero if there is an error.  If "all" is nonzero, playback
 * the entire file with no prompt.
 */
INTSML us_logplayback(char *file, INTSML all)
{
	REGISTER INTBIG comcount;
	char *filename;
	us_logplay = xopen(file, FILETYPETEXT, "", &filename);
	if (us_logplay == NULL) return(1);
	ttyputmsg("Type any key to playback the next step in the log file");
	ttyputmsg("Type a number followed by 'x' to playback that many steps");
	ttyputmsg("Type 'q' to terminate playback");
	comcount = filesize(us_logplay) / (sizeof (gra_action));
	ttyputmsg("There are no more than %d steps to playback", comcount);
	if (all != 0) gra_playbackmultiple = (INTSML)comcount;
	return(0);
}

/*
 * routine to create a session logging file
 */
void us_logstartrecord(void)
{
#if 0
	us_logrecord = xcreate(ELECTRICLOG, 2, 0, 0);
#else
	us_logrecord = NULL;
#endif
}

/*
 * routine to terminate session logging
 */
void us_logfinishrecord(void)
{
	if (us_logrecord != NULL) xclose(us_logrecord);
	us_logrecord = NULL;
}

/****************************** MENUS ******************************/

INTSML gra_popupmenu(POPUPMENU *menu, INTSML header, INTSML left, INTSML top)
{
	CMainFrame *wnd;
	int j, index;
	POPUPMENUITEM *mi;
	RECT r;
	HMENU popmenu;
	POINT p, p2;
	UINT flags;

	wnd = (CMainFrame *)AfxGetMainWnd();
	flags = TPM_NONOTIFY|TPM_RETURNCMD|TPM_LEFTALIGN|TPM_LEFTBUTTON;
	if (left < 0 || top < 0)
	{
		p2.x = p2.y = 0;
		wnd->MapWindowPoints(0, &p2, 1);
		GetCursorPos(&p);
		left = p.x - p2.x;
		top = p.y - p2.y;
		flags = TPM_NONOTIFY|TPM_RETURNCMD|TPM_CENTERALIGN|TPM_VCENTERALIGN|TPM_LEFTBUTTON;
	}
	popmenu = CreatePopupMenu();
	if (popmenu == 0) return(-1);
	if (header != 0)
	{
		if (AppendMenu(popmenu, MF_STRING, -1, menu->header) == 0) return(-1);
		if (AppendMenu(popmenu, MF_SEPARATOR, 0, 0) == 0) return(-1);
	}
	for(j=0; j < menu->total; j++)
	{
		mi = &menu->list[j];
		mi->changed = 0;
		if (AppendMenu(popmenu, MF_STRING, j, mi->attribute) == 0) return(-1);
	}
	index = TrackPopupMenu(popmenu, flags, left, top, 0, wnd->m_hWnd, 0); 
 	DestroyMenu(popmenu);
	return(index);
}

/*
 * routine to handle the menu
 */
void gra_nativemenudoone(INTSML item, INTSML menuindex)
{
	INTSML i, j, trueItem;
	POPUPMENU *pm;
	REGISTER POPUPMENUITEM *mi;
	extern int el_inslice;
	int saveslice;

	i = menuindex;
	if (i >= 0 && i < gra_pulldownmenucount)
	{
		pm = gra_pulldowns[i];
		for(trueItem=j=0; j < pm->total; j++)
		{
			mi = &pm->list[j];
			if (mi->response->active < 0 && *mi->attribute == 0) continue;
			trueItem++;
			if (trueItem != item) continue;
			us_state |= DIDINPUT;
			us_state &= ~GOTXY;
			gra_set_cursorstate(NULLCURSOR);

			/* special case for "system" commands": don't erase */
			us_forceeditchanges();
			saveslice = el_inslice;
			el_inslice = 1;
			us_execute(mi->response, us_aid->aidstate&ECHOBIND, 1, 1);
			el_inslice = saveslice;
			setactivity(mi->attribute);
			break;
		}
	}
}

/*
 * Routine to establish the "count" pulldown menu names in "par" as the pulldown menu bar
 */
INTSML gra_nativemenuload(INTSML count, char *par[])
{
	REGISTER INTSML i, menuindex;
	REGISTER POPUPMENU *pm;
	POPUPMENU *pulls[25];
	void *curwindowframe;
	HWND wind;
	WINDOWFRAME *frame;

	curwindowframe = us_getwindowframe();
	if (curwindowframe == 0) return(1);

	if (gra_hMenu == 0)
	{
		gra_hMenu = new CMenu();
		gra_hMenu->CreateMenu();
	}
	frame = (WINDOWFRAME *)curwindowframe;

	for(i=0; i<count; i++)
	{
		pm = us_getpopupmenu(par[i]);
		if (pm == NOPOPUPMENU) continue;
		pulls[i] = pm;

		menuindex = gra_pulldownindex(pm);
		if (menuindex < 0) continue;
		if (gra_hMenu->InsertMenu(menuindex, MF_ENABLED | MF_POPUP | MF_BYPOSITION,
			(DWORD)gra_pulldownmenus[menuindex]->m_hMenu, pm->header) == 0)
				return(1);
	}
	CWnd* parent = AfxGetMainWnd();
	CMenu* lastMenu = parent->GetMenu();
	if (parent->SetMenu(gra_hMenu) == 0) return(1);
	gra_hMenu->Detach();
	parent->DrawMenuBar();
	return(0);
}

/*
 * Routine to create a pulldown menu from popup menu "pm".  Returns an index to
 * the table of pulldown menus (-1 on error).
 */
INTSML gra_pulldownindex(POPUPMENU *pm)
{
	REGISTER INTSML i, index;
	CMenu **newpulldownmenus;
	POPUPMENU **newpulldowns;

	for(i=0; i<gra_pulldownmenucount; i++)
		if (gra_pulldowns[i] == pm) return(i);

	/* allocate new space with one more */
	newpulldownmenus = (CMenu **)emalloc((gra_pulldownmenucount+1) *
		(sizeof (CMenu *)), us_aid->cluster);
	if (newpulldownmenus == 0) return(-1);
	newpulldowns = (POPUPMENU **)emalloc((gra_pulldownmenucount+1) *
		(sizeof (POPUPMENU *)), us_aid->cluster);
	if (newpulldowns == 0) return(-1);

	/* copy former arrays then delete them */
	for(i=0; i<gra_pulldownmenucount; i++)
	{
		newpulldownmenus[i] = gra_pulldownmenus[i];
		newpulldowns[i] = gra_pulldowns[i];
	}
	if (gra_pulldownmenucount != 0)
	{
		efree((char *)gra_pulldownmenus);
		efree((char *)gra_pulldowns);
	}

	gra_pulldownmenus = newpulldownmenus;
	gra_pulldowns = newpulldowns;

	index = gra_pulldownmenucount++;
	gra_pulldownmenus[index] = gra_makepdmenu(pm, index);
	if (gra_pulldownmenus[index] == 0)
		return(-1);
	gra_pulldowns[index] = pm;
	return(index);
}

CMenu *gra_makepdmenu(POPUPMENU *pm, INTSML index)
{
	CMenu *hPrevMenu, *hMenu;
	REGISTER USERCOM *uc;
	REGISTER POPUPMENUITEM *mi;
	REGISTER INTSML j, submenuindex, key, len, idIndex;
	REGISTER char *pt, save;
	char keychr, myline[100];

	hMenu = new CMenu();
	if (hMenu->CreatePopupMenu() == 0) return(0);

	idIndex = 0;
	for(j=0; j < pm->total; j++)
	{
		mi = &pm->list[j];
		uc = mi->response;
		if (uc->active < 0)
		{
			if (*mi->attribute == 0)
			{
				if (hMenu->AppendMenu(MF_SEPARATOR) == 0) return(0);
			} else
			{
				if (hMenu->AppendMenu(MF_STRING | MF_DISABLED, gra_menures[idIndex++], mi->attribute) == 0) return(0);
			}
			continue;
		}

		if (uc->menu != NOPOPUPMENU)
		{
			hPrevMenu = hMenu;
			idIndex++;
			submenuindex = gra_pulldownindex(uc->menu);
			if (hPrevMenu->InsertMenu(-1, MF_POPUP | MF_BYPOSITION,
				(DWORD)gra_pulldownmenus[submenuindex]->m_hMenu, mi->attribute) == 0)
					return(0);
			continue;
		}
		strcpy(myline, mi->attribute);
		len = strlen(myline);
		if (myline[len-1] == '<') myline[len-1] = 0;
		for(pt = myline; *pt != 0; pt++) if (*pt == '/') break;
		if (*pt != 0)
		{
			*pt++ = 0;
			keychr = *pt;
			len = strlen(myline);
			sprintf(&myline[len], "\tCtrl-%c", keychr);
			if (hMenu->AppendMenu(MF_STRING, gra_menures[idIndex++], myline) == 0) return(0);
		} else if (myline[0] == '>')
		{
			if (hMenu->AppendMenu(MF_CHECKED | MF_STRING, gra_menures[idIndex++], &myline[1]) == 0) return(0);
		} else
		{
			if (hMenu->AppendMenu(MF_STRING, gra_menures[idIndex++], myline) == 0) return(0);
		}
	}
	return(hMenu);
}

/* routine to redraw entry "index" of popupmenu "pm" because it changed */
void gra_nativemenurename(POPUPMENU *pm, INTSML index)
{
	INTSML i, j, submenuindex, len, key, idIndex;
	char line[100], *pt;
	USERCOM *uc;
	REGISTER POPUPMENUITEM *mi;

	for(i=0; i<gra_pulldownmenucount; i++)
		if (gra_pulldowns[i] == pm)
	{
		idIndex = 0;
		for(j=0; j < pm->total; j++)
		{
			mi = &pm->list[j];
			uc = mi->response;
			if (j == index) break;
			if (uc->active < 0 && *mi->attribute == 0) continue;
			idIndex++;
		}

		if (uc->active < 0)
		{
			if (*pm->list[i].attribute == 0)
			{
				gra_pulldownmenus[i]->ModifyMenu(index, MF_BYPOSITION|MF_SEPARATOR);
			} else
			{
				gra_pulldownmenus[i]->EnableMenuItem(index, MF_BYPOSITION|MF_GRAYED);
			}
		} else
		{
			gra_pulldownmenus[i]->EnableMenuItem(index, MF_BYPOSITION|MF_ENABLED);
		}

		/* copy and examine the menu string */
		(void)strcpy(line, pm->list[index].attribute);
		len = strlen(line);
		if (line[len-1] == '<') line[len-1] = 0;
		for(pt = line; *pt != 0; pt++) if (*pt == '/') break;

		/* handle single-key equivalents */
		if (*pt == '/')
		{
			*pt++ = 0;
			key = *pt;
			len = strlen(line);
			sprintf(&line[len], "\tCtrl-%c", key);
			gra_pulldownmenus[i]->ModifyMenu(index, MF_BYPOSITION, gra_menures[idIndex], line);
		} else
		{
			/* see if entry is checked */
			if (line[0] != '>')
			{
				/* not checked: reload the entry */
				gra_pulldownmenus[i]->ModifyMenu(index, MF_BYPOSITION, gra_menures[idIndex], line);
			} else
			{
				/* remove "check" marks and check the entry properly */
				gra_pulldownmenus[i]->ModifyMenu(index, MF_CHECKED | MF_BYPOSITION,
					gra_menures[idIndex], &line[1]);
			}
		}

		/* see if this command is another menu */
		if (uc->menu != NOPOPUPMENU)
		{
			for(submenuindex=0; submenuindex<gra_pulldownmenucount; submenuindex++)
				if (gra_pulldowns[submenuindex] == uc->menu) break;
			if (submenuindex >= gra_pulldownmenucount) continue;
			gra_pulldownmenus[i]->ModifyMenu(index, MF_POPUP | MF_BYPOSITION,
				(DWORD) gra_pulldownmenus[submenuindex], pm->list[index].attribute);
			continue;
		}
		break;
	}
}

/****************************** SYSTEM CALLS ******************************/

int wait(void)
{ 
	return(-1);
}

unsigned short getpid(void)
{ 
	return((unsigned short)GetCurrentProcessId());
}

/*
 * Routine to establish the library directories from the environment.
 *
 * Registry entires: 
 *   HKEY_LOCAL_MACHINE -> Software -> Electric Editor -> Electric
 *                          -> EEI_LIBDIR
 *                          -> EEI_LISPDIR
 */
#define HKLM HKEY_LOCAL_MACHINE
void setupenvironment(void)
{
#if 0
	HKEY hKey;
	DWORD size;
	char tmp[256], kVal[256];
	INTSML i;
	extern char *el_version;

	(void)sprintf(kVal, "Software\\Electric Editor\\Electric");	
	if (RegOpenKeyEx(HKLM, kVal, 0, KEY_READ, &hKey) == ERROR_SUCCESS)
	{
		if (RegQueryValueEx(hKey, "EEI_LIBDIR", NULL, NULL, NULL, &size) == ERROR_SUCCESS)
			if (RegQueryValueEx(hKey, "EEI_LIBDIR", NULL, NULL, tmp, &size) == ERROR_SUCCESS)
				(void)allocstring(&el_libdir, tmp, db_cluster);	
			else
				(void)allocstring(&el_libdir, "nodir", db_cluster);

		if (RegQueryValueEx(hKey, "EEI_LISPDIR", NULL, NULL, NULL, &size) == ERROR_SUCCESS)
			if (RegQueryValueEx(hKey, "EEI_LISPDIR", NULL, NULL, tmp, &size) == ERROR_SUCCESS)
				(void)allocstring(&el_lispdir, tmp, db_cluster);
			else
				(void)allocstring(&el_lispdir, "nodir", db_cluster);
		RegCloseKey(hKey);
	}


	/* load font information */
	(void)sprintf(kVal, "Software\\Electric Editor\\Electric\\Fonts");
	if (RegOpenKeyEx(HKLM, kVal, 0, KEY_READ, &hKey) == ERROR_SUCCESS)
	{
		i=0;
		while (fontList[i].eFontIndex != -1)
		{
			if (RegQueryValueEx(hKey, fontList[i].regFontName, NULL, NULL, NULL, &size) == ERROR_SUCCESS)
				if (RegQueryValueEx(hKey, fontList[i].regFontName, NULL, NULL, tmp, &size) == ERROR_SUCCESS)
					strcpy(fontList[i].fontName, tmp);
			if (RegQueryValueEx(hKey, fontList[i].regFontSize, NULL, NULL, NULL, &size) == ERROR_SUCCESS)
				if (RegQueryValueEx(hKey, fontList[i].regFontSize, NULL, NULL, tmp, &size) == ERROR_SUCCESS)
					fontList[i].fontSize = myatoi(tmp);
			i++;
		}

		RegCloseKey(hKey);
	}
#endif

#if LANGTCL
	(void)initinfstr();
	(void)addstringtoinfstr("lib\\tcl");
	(void)allocstring(&el_tcldir, returninfstr(), db_cluster);
#endif

	/* set machine name */
	nextvarchangequiet();
	(void)setvalkey((INTBIG)us_aid, VAID, makekey("USER_machine"),
		(INTBIG)"Windows", VSTRING|VDONTSAVE);
}

/****************************** DIALOGS ******************************/

long gra_dodialoginit(DIALOG *dialog)
{
	DLGTEMPLATE *dtheader;
	WORD *dtmenu, *dtclass, *dttitle;
	int headerlength, menulength, classlength, titlelength, itemslength,
		totallen;
	DLGITEMTEMPLATE **dtitems;
	RECT r;
	char *descblock, *title, *pt;
	int *itemlengths, i, j, len, itemtype, style, itemclass, iconxpos;
	WORD output[100];
	CProgressCtrl *prog;
	CListBox *list;

#if 0
	long dbu = GetDialogBaseUnits();
	gra_dialogdbnx = 4;   gra_dialogdbny = 8;
	gra_dialogdbux = dbu & 0xFFFF;
	gra_dialogdbuy = (dbu >> 16) & 0xFFFF;
#else
	gra_dialogdbnx = 9;     gra_dialogdbux = 14;
	gra_dialogdbny = 131;   gra_dialogdbuy = 210;
#endif
	gra_dialogredrawroutine = 0;
	gra_dialogwindow = new CElectricDialog();
	gra_dialogitemdesc = dialog;
	gra_dialogeditline = -1;
	gra_dialoglastedittext[0] = 0;

	/* determine item lengths */
	itemlengths = (int *)emalloc(dialog->items * (sizeof (int)), us_aid->cluster);
	if (itemlengths == 0) return(0);
	dtitems = (DLGITEMTEMPLATE **)emalloc(dialog->items * (sizeof (DLGITEMTEMPLATE *)), us_aid->cluster);
	if (dtitems == 0) return(0);
	itemslength = 0;
	for(i=0; i<dialog->items; i++)
	{
		itemlengths[i] = (sizeof DLGITEMTEMPLATE) + 3 * (sizeof (WORD));
		itemtype = dialog->list[i].type;
		switch (itemtype&ITEMTYPE)
		{
			case SCROLL:
			case ICON:
			case POPUP:
				itemlengths[i] += 1 * (sizeof (WORD));
				break;
			default:
				itemlengths[i] += (strlen(dialog->list[i].msg) + 1) * (sizeof (WORD));
				break;
		}
		itemlengths[i] = (itemlengths[i] + 3) & ~3;
		itemslength += itemlengths[i];
	}

	/* allocate space for entire dialog template */
	headerlength = sizeof (DLGTEMPLATE);
	menulength = sizeof (WORD);
	classlength = sizeof (WORD);
	if (dialog->movable != 0) title = dialog->movable; else
		title = "";
	titlelength = (strlen(title) + 1) * (sizeof (WORD));
	i = headerlength + menulength + classlength + titlelength;
	totallen = (i + 3) & ~3;
	descblock = (char *)emalloc(totallen + itemslength, us_aid->cluster);
	if (descblock == 0) return(0);
	pt = descblock;
	dtheader = (DLGTEMPLATE *)pt;  pt += headerlength;
	dtmenu = (WORD *)pt;           pt += menulength;
	dtclass = (WORD *)pt;          pt += classlength;
	dttitle = (WORD *)pt;          pt += titlelength;
	for(i=0; i<dialog->items; i++)
	{
		pt = (char *)(((DWORD)pt + 3) & ~3);
		dtitems[i] = (DLGITEMTEMPLATE *)pt;
		pt += itemlengths[i];
	}

	/* load dialog template header */
	dtheader->style = WS_VISIBLE | WS_DLGFRAME | WS_POPUP /* | DS_ABSALIGN */;
	if (dialog->movable != 0) dtheader->style |= WS_CAPTION | DS_MODALFRAME;
	dtheader->dwExtendedStyle = 0;
	dtheader->cdit = dialog->items;
	dtheader->x = dialog->windowRect.left * gra_dialogdbnx / gra_dialogdbux;
	dtheader->y = dialog->windowRect.top * gra_dialogdbny / gra_dialogdbuy;
	dtheader->cx = (dialog->windowRect.right - dialog->windowRect.left) * gra_dialogdbnx / gra_dialogdbux;
	dtheader->cy = (dialog->windowRect.bottom - dialog->windowRect.top) * gra_dialogdbny / gra_dialogdbuy;

	/* no menu or class in this dialog */
	dtmenu[0] = 0;
	dtclass[0] = 0;

	/* load the dialog title */
	len = MultiByteToWideChar(CP_ACP, 0, title, strlen(title), output, 100);
	for(j=0; j<len; j++) *dttitle++ = output[j];
	*dttitle++ = 0;

	/* load the items */
	iconxpos = -1;
	for(i=0; i<dialog->items; i++)
	{
		dtitems[i]->x = dialog->list[i].r.left * gra_dialogdbnx / gra_dialogdbux;
		dtitems[i]->y = dialog->list[i].r.top * gra_dialogdbny / gra_dialogdbuy; 
		dtitems[i]->cx = (dialog->list[i].r.right - dialog->list[i].r.left) * gra_dialogdbnx / gra_dialogdbux;
		dtitems[i]->cy = (dialog->list[i].r.bottom - dialog->list[i].r.top) * gra_dialogdbny / gra_dialogdbuy;
		dtitems[i]->dwExtendedStyle = 0;
		dtitems[i]->id = i+ID_DIALOGITEM_0;
		itemtype = dialog->list[i].type;
		switch (itemtype&ITEMTYPE)
		{
			case BUTTON:
				itemclass = 0x80;
				style = WS_VISIBLE | WS_CHILD | BS_PUSHBUTTON;
				if (i == 0) style |= BS_DEFPUSHBUTTON;
				break;
			case CHECK:
				itemclass = 0x80;
				style = WS_VISIBLE | WS_CHILD | BS_CHECKBOX;
				break;
			case RADIO:
				itemclass = 0x80;
				style = WS_VISIBLE | WS_CHILD | BS_RADIOBUTTON;
				break;
			case EDITTEXT:
				itemclass = 0x81;
				style = WS_CHILD | WS_VISIBLE | WS_BORDER | WS_TABSTOP | ES_LEFT;
				if (dtitems[i]->cy > 10) style |= ES_MULTILINE;
				dtitems[i]->y--;
				dtitems[i]->cy += 2;
				break;
			case MESSAGE:
				itemclass = 0x82;
				style = WS_CHILD | WS_VISIBLE | SS_LEFT;
				if ((itemtype&INACTIVE) == 0) style |= SS_NOTIFY;
				break;
			case PROGRESS:
				itemclass = 0x82;
				style = WS_CHILD | WS_VISIBLE;
				dtitems[i]->id += PROGRESSOFFSET;
				break;
			case USERDRAWN:
				itemclass = 0x82;
				style = WS_CHILD | WS_VISIBLE;
				dtitems[i]->x += 1000;
				dtitems[i]->y += 1000;
				break;
			case POPUP:
				itemclass = 0x85;
				dtitems[i]->cy *= 8;
				style = WS_CHILD | WS_VISIBLE | WS_VSCROLL | CBS_DROPDOWNLIST;
				break;
			case ICON:
				itemclass = 0x82;
				style = WS_CHILD | WS_VISIBLE | SS_ICON;
				if ((itemtype&INACTIVE) == 0) style |= SS_NOTIFY;
				break;
			case SCROLL:
				itemclass = 0x83;
				style = WS_BORDER | WS_CHILD | WS_VISIBLE | LBS_USETABSTOPS |
					WS_VSCROLL | WS_HSCROLL | LBS_WANTKEYBOARDINPUT | LBS_SORT;
				if ((itemtype&INACTIVE) == 0) style |= LBS_NOTIFY;
				break;
			default:
				itemclass = 0x82;
				style = WS_CHILD | SS_LEFT;
				break;
		}
#if 1
		dtitems[i]->style = style;
#else
		dtitems[i]->style = style | DS_ABSALIGN;
#endif
		pt = ((char *)dtitems[i]) + (sizeof (DLGITEMTEMPLATE));
		dtclass = (WORD *)pt;
		*dtclass++ = 0xFFFF;
		*dtclass++ = itemclass;
		switch (itemtype&ITEMTYPE)
		{
			case POPUP:
			case ICON:
			case SCROLL:
				*dtclass++ = 0;
				break;
			default:
				len = MultiByteToWideChar(CP_ACP, 0, dialog->list[i].msg,
					strlen(dialog->list[i].msg), output, 100);
				for(j=0; j<len; j++) *dtclass++ = output[j];
				*dtclass++ = 0;
				break;
		}
		*dtclass++ = 0;
	}

	/* create the dialog */
	if (gra_dialogwindow->CreateIndirect(dtheader) == 0)
	{
		return(0);
	}

	/* finish initialization */
	CStatic *stat;
	HICON map;
	for(i=0; i<dialog->items; i++)
	{
		itemtype = dialog->list[i].type;
		switch (itemtype&ITEMTYPE)
		{
			case ICON:
				stat = (CStatic *)gra_dialogwindow->GetDlgItem(i+ID_DIALOGITEM_0);
				if (stat == 0) break;
				map = gra_makeicon(dialog->list[i].msg);
				stat->SetIcon(map);
				break;
			case PROGRESS:
				stat = (CStatic *)gra_dialogwindow->GetDlgItem(i+ID_DIALOGITEM_0+PROGRESSOFFSET);
				stat->GetWindowRect(&r);
				gra_dialogwindow->ScreenToClient(&r);
				prog = new CProgressCtrl();
				prog->Create(WS_CHILD | WS_VISIBLE, r, gra_dialogwindow, i+ID_DIALOGITEM_0);
				break;
			case SCROLL:
				list = (CListBox *)gra_dialogwindow->GetDlgItem(i+ID_DIALOGITEM_0);
				if (list == 0) break;
				list->SetTabStops(1);
				break;
		}
	}

	gra_dialoghit = -1;
	gra_dialogdefaultbutton = 1;
	return((long)gra_dialogwindow);
}

void gra_dodialogdone(void)
{
	gra_dialogwindow->EndDialog(0);
	gra_dialogwindow = 0;
}

void gra_dopopdialog(long dia, DIALOG *desc)
{
	gra_dialogwindow = (CElectricDialog *)dia;
	gra_dialogitemdesc = desc;
}

INTSML gra_dodialogisinsideuserdrawn(int x, int y)
{
	int i, itemtype;

	if (gra_dialogwindow == 0) return(0);
	for(i=0; i<gra_dialogitemdesc->items; i++)
	{
		itemtype = gra_dialogitemdesc->list[i].type;
		if ((itemtype&ITEMTYPE) != USERDRAWN) continue;
		if (x < gra_dialogitemdesc->list[i].r.left) continue;
		if (x > gra_dialogitemdesc->list[i].r.right) continue;
		if (y < gra_dialogitemdesc->list[i].r.top) continue;
		if (y > gra_dialogitemdesc->list[i].r.bottom) continue;
		return(i+1);
	}
	return(0);
}

/*
 * Routine called when a character is typed
 */
void gra_dodialogtextchange(int nID)
{
	char line[300];

	/* ignore if this field isn't being character edited */
	gra_dialoghit = nID + 1;
	if (nID != gra_dialogeditline) return;

	/* get current, with changes */
	CWnd *wnd = gra_dialogwindow->GetDlgItem(nID+ID_DIALOGITEM_0);
	if (wnd == 0) return;
	wnd->GetWindowText(line, 300);

	/* determine which character was typed (hack!!!) */
	if (strlen(line) > strlen(gra_dialoglastedittext))
	{
		gra_dialoghitchar = line[strlen(line)-1];
	} else if (strlen(line) < strlen(gra_dialoglastedittext))
	{
		gra_dialoghitchar = 010;
	}

	/* reset to previous state */
	wnd->SetWindowText(gra_dialoglastedittext);
	CEdit *edit = (CEdit *)wnd;
	int len = strlen(gra_dialoglastedittext);
	edit->SetSel(len, len);
}

int gra_dodialognextchar(short *itemHit)
{
	int chr;

	gra_nextevent(0);
	if (gra_dialoghit == -1) return(-1);
	*itemHit = gra_dialoghit;
	if (gra_dialoghitchar == 0) return(-2);
	chr = gra_dialoghitchar;
	gra_dialoghitchar = 0;
	return(chr);
}

/*
 * Routine called when an item is clicked
 */
void gra_itemclicked(int nID)
{
	int itemtype;

	gra_dialoghitchar = 0;
	if (nID == 1)
	{
		gra_dialoghit = gra_dialogdefaultbutton;
		return;
	}
	if (nID == 2)
	{
		gra_dialoghit = 2;
		return;
	}

	// ignore clicks in scroll areas that do not want hits reported
	itemtype = gra_dialogitemdesc->list[nID-ID_DIALOGITEM_0].type;
	if ((itemtype&ITEMTYPE) == SCROLL &&
		(gra_dialogitemdesc->list[nID-ID_DIALOGITEM_0].data&SCREPORT) == 0)
			return;

	gra_dialoghit = nID - ID_DIALOGITEM_0 + 1;
}

void gra_itemdoubleclicked(int nID)
{
	int i, itemtype;

	if (gra_dialogwindow == 0) return;
	itemtype = gra_dialogitemdesc->list[nID].type;
	if ((itemtype&ITEMTYPE) != SCROLL) return;
	if ((gra_dialogitemdesc->list[nID].data&SCDOUBLEQUIT) == 0) return;

	gra_dialoghit = gra_dialogdefaultbutton;
	gra_dialoghitchar = 0;
}

/*
 * Called when a key is typed to a list box.  Returns nonzero to accept the
 * keystroke, zero to ignore typed keys in the list box.
 */
int gra_dodialoglistkey(UINT nKey, CListBox* pListBox, UINT nIndex)
{
	CListBox *list;
	int i, itemtype;

	for(i=0; i<gra_dialogitemdesc->items; i++)
	{
		itemtype = gra_dialogitemdesc->list[i].type;
		if ((itemtype&ITEMTYPE) != SCROLL) continue;
		list = (CListBox *)gra_dialogwindow->GetDlgItem(i+ID_DIALOGITEM_0);
		if (list != pListBox) continue;
		if ((gra_dialogitemdesc->list[i].data&SCSELKEY) == 0) return(0);
		break;
	}

	return(1);
}

int gra_dodialognexthit(void)
{
	short item, ret;

	for(;;)
	{
		flushscreen();
		gra_nextevent(0);
		if (gra_dialoghit != -1)
		{
			item = gra_dialoghit;
			gra_dialoghit = -1;
			return(item);
		}
	}
}

void gra_dodialogsetdefault(int formeritem, int item)
{
	CButton *but;
	UINT style;

	/* turn default style on for the new button */
	but = (CButton *)gra_dialogwindow->GetDlgItem(item+ID_DIALOGITEM_0);
	if (but != 0)
	{
		style = but->GetButtonStyle();
		but->SetButtonStyle(style | BS_DEFPUSHBUTTON);
		gra_dialogwindow->GotoDlgCtrl(but);
	}

	/* turn default style off for the old button */
	but = (CButton *)gra_dialogwindow->GetDlgItem(formeritem+ID_DIALOGITEM_0);
	if (but != 0)
	{
		style = but->GetButtonStyle();
		but->SetButtonStyle(style & ~BS_DEFPUSHBUTTON);
	}

	gra_dialogwindow->SetDefID(item+ID_DIALOGITEM_0);
	gra_dialogdefaultbutton = item + 1;
}

void gra_dodialogsettext(int item, int type, char *msg, int highlight)
{
	CWnd *wnd = gra_dialogwindow->GetDlgItem(item+ID_DIALOGITEM_0);
	if (wnd == 0) return;
	wnd->SetWindowText(msg);
	if ((type&ITEMTYPE) == EDITTEXT)
	{
		CEdit *edit = (CEdit *)wnd;
		int len = strlen(msg);
		if (highlight != 0) edit->SetSel(0, len); else
			edit->SetSel(len, len);
		if (item == gra_dialogeditline)
			strcpy(gra_dialoglastedittext, msg);
	}
}

char *gra_dodialoggettext(int item, int type)
{
	static int bufnum = 0;
	static char line[10][300];

	CWnd *wnd = gra_dialogwindow->GetDlgItem(item+ID_DIALOGITEM_0);
	if (wnd == 0) return("");
	bufnum++;
	if (bufnum >= 10) bufnum = 0;
	wnd->GetWindowText(line[bufnum], 300);
	return(line[bufnum]);
}

int gra_dodialoggetcontrol(int item, int type)
{
	CButton *but = (CButton *)gra_dialogwindow->GetDlgItem(item+ID_DIALOGITEM_0);
	if (but == 0) return(0);
	return(but->GetCheck());
}

void gra_dodialogsetcontrol(int item, int type, int value)
{
	CButton *but = (CButton *)gra_dialogwindow->GetDlgItem(item+ID_DIALOGITEM_0);
	if (but == 0) return;
	but->SetCheck(value);
}

void gra_dodialogdimitem(int item, int enabled)
{
	CWnd *wnd = gra_dialogwindow->GetDlgItem(item+ID_DIALOGITEM_0);
	wnd->EnableWindow(enabled);
}

void gra_dodialogopaqueitem(int item)
{
	CEdit *edit = (CEdit *)gra_dialogwindow->GetDlgItem(item+ID_DIALOGITEM_0);
	edit->ModifyStyle(0, ES_PASSWORD);
	edit->SetPasswordChar('*');
}

void gra_dodialogpercent(int item, int percent)
{
	RECT r;

	CProgressCtrl *prog = (CProgressCtrl *)gra_dialogwindow->GetDlgItem(item+ID_DIALOGITEM_0);
	prog->SetPos(percent);
}

void gra_dodialogcharacteredit(int item)
{
	gra_dialogeditline = item;
}

void gra_dodialoginitscroll(int item, int flags)
{
	long add, remove;
	static CFont *fnt = 0;
	LOGFONT lf;

	CListBox *list = (CListBox *)gra_dialogwindow->GetDlgItem(item+ID_DIALOGITEM_0);
	if (list == 0) return;

	add = remove = 0;
	if ((flags&SCHORIZBAR) == 0) remove |= WS_HSCROLL; else
	{
		add |= WS_HSCROLL;
		list->SetHorizontalExtent(1000);
	}
	list->ModifyStyle(remove, add, SWP_NOSIZE);
	gra_dialogitemdesc->list[item].data = flags;
	if ((flags&SCFIXEDWIDTH) != 0)
	{
		if (fnt == 0)
		{
			lf.lfHeight = -10;
			strcpy(lf.lfFaceName, "Lucida Console");
			lf.lfWidth = 0;
			lf.lfEscapement = 0;
			lf.lfOrientation = 0;
			lf.lfWeight = FW_NORMAL;
			lf.lfItalic = 0;
			lf.lfUnderline = 0;
			lf.lfStrikeOut = 0;
			lf.lfCharSet = 0;
			lf.lfOutPrecision = OUT_STROKE_PRECIS;
			lf.lfClipPrecision = CLIP_STROKE_PRECIS;
			lf.lfQuality = 1;
			lf.lfPitchAndFamily = FF_DONTCARE | FF_SWISS;

			fnt = new CFont();
			fnt->CreateFontIndirect(&lf);
		}
		list->SetFont(fnt);
	}
}

void gra_dodialogloadscroll(int item, int sort, short (*toplist)(char **),
	char *(*nextinlist)(void), void (*donelist)(void))
{
	char *next, line[256];
	int i;

	CListBox *list = (CListBox *)gra_dialogwindow->GetDlgItem(item+ID_DIALOGITEM_0);
	if (list == 0) return;

	/* clear the list */
	list->ResetContent();

	/* load the list */
	line[0] = 0;
	next = line;
	(void)(*toplist)(&next);
	for(i=0; ; i++)
	{
		next = (*nextinlist)();
		if (next == 0) break;
		if (sort < 0) list->InsertString(-1, next); else
			list->AddString(next);
	}
	(*donelist)();
	if (i > 0) list->SetCurSel(0);
}

void gra_dodialogstuffline(int item, char *line)
{
	CListBox *list = (CListBox *)gra_dialogwindow->GetDlgItem(item+ID_DIALOGITEM_0);
	if (list == 0) return;
	list->InsertString(-1, line);
}

void gra_dodialogselectline(int item, int line)
{
	CListBox *list = (CListBox *)gra_dialogwindow->GetDlgItem(item+ID_DIALOGITEM_0);
	if (list == 0) return;
	list->SetCurSel(line);
}

int gra_dodialoggetselectedline(int item)
{
	int line;

	CListBox *list = (CListBox *)gra_dialogwindow->GetDlgItem(item+ID_DIALOGITEM_0);
	if (list == 0) return(-1);
	line = list->GetCurSel();
	if (line == LB_ERR) return(-1);
	return(line);
}

char *gra_dodialoggetline(int item, int line)
{
	static char text[300];

	CListBox *list = (CListBox *)gra_dialogwindow->GetDlgItem(item+ID_DIALOGITEM_0);
	if (list == 0) return("");
	if (list->GetText(line, text) == -1) return("");
	return(text);
}

void gra_dodialogsetline(int item, int line, char *msg)
{
	CListBox *list = (CListBox *)gra_dialogwindow->GetDlgItem(item+ID_DIALOGITEM_0);
	if (list == 0) return;
	list->DeleteString(line);
	list->InsertString(line, msg);
	list->SetCurSel(line);
}

void gra_dodialogsetpopup(int item, int count, char **names)
{
	int i;

	CComboBox *cb = (CComboBox *)gra_dialogwindow->GetDlgItem(item+ID_DIALOGITEM_0);
	if (cb == 0) return;
	cb->ResetContent();
	for(i=0; i<count; i++) cb->AddString(names[i]);
	cb->SetCurSel(0);
}

void gra_dodialogsetpopupentry(int item, int entry)
{
	CComboBox *cb = (CComboBox *)gra_dialogwindow->GetDlgItem(item+ID_DIALOGITEM_0);
	if (cb == 0) return;
	cb->SetCurSel(entry);
}

int gra_dodialoggetpopupentry(int item)
{
	CComboBox *cb = (CComboBox *)gra_dialogwindow->GetDlgItem(item+ID_DIALOGITEM_0);
	if (cb == 0) return(0);
	return(cb->GetCurSel());
}

void gra_dodialoggetmouse(int *x, int *y)
{
	POINT p, p2;

	p2.x = p2.y = 0;
	gra_dialogwindow->MapWindowPoints(0, &p2, 1);
	GetCursorPos(&p);
	*x = p.x - p2.x;   *y = p.y - p2.y;
}

void gra_dodialogredisproutine(int item, void (*routine)(RECTAREA*))
{
	gra_dialogredrawroutine = routine;
	gra_dialogredrawitem = item;
}

void gra_diaredrawitem(void)
{
	RECTAREA r;

	if (gra_dialogredrawroutine != 0)
	{
		DiaItemRect(gra_dialogredrawitem, &r);
		(*gra_dialogredrawroutine)(&r);
	}
}

void gra_dodialogframerect(RECT *ur)
{
	RECT r;

	r.left = ur->left;
	r.right = ur->right;
	r.top = ur->top;
	r.bottom = ur->bottom;

	if (gra_dialogoffbrush == 0) gra_dialogoffbrush = new CBrush((COLORREF)0xFFFFFF);
	CDC *dc = gra_dialogwindow->GetDC();
	dc->FillRect(&r, gra_dialogoffbrush);
	dc->MoveTo(r.left, r.top);
	dc->LineTo(r.right, r.top);
	dc->LineTo(r.right, r.bottom);
	dc->LineTo(r.left, r.bottom);
	dc->LineTo(r.left, r.top);
	gra_dialogwindow->ReleaseDC(dc);
}

void gra_dodialogdrawrect(RECT *ur, int on)
{
	RECT r;

	r.left = ur->left;
	r.right = ur->right;
	r.top = ur->top;
	r.bottom = ur->bottom;

	if (gra_dialogonbrush == 0) gra_dialogonbrush = new CBrush((COLORREF)0x000000);
	if (gra_dialogoffbrush == 0) gra_dialogoffbrush = new CBrush((COLORREF)0xFFFFFF);
	CDC *dc = gra_dialogwindow->GetDC();
	dc->FillRect(&r, (on ? gra_dialogonbrush : gra_dialogoffbrush));
	gra_dialogwindow->ReleaseDC(dc);
}

HICON gra_makeicon(char *data)
{
	unsigned char zero[128];
	int i;

	for(i=0; i<128; i++)
	{
		zero[i] = 0;
		data[i] = ~data[i];
	}
	HICON icon = CreateIcon(0, 32, 32, 1, 1, (unsigned char *)data, zero);
	for(i=0; i<128; i++) data[i] = ~data[i];
	return(icon);
}

#if LANGTCL
int InitializeTCL(void)
{
	int err;
	char *newArgv[2];

	/* set the program name/path */
	newArgv[0] = "Electric";
	newArgv[1] = NULL;
	(void)Tcl_FindExecutable(newArgv[0]);

	tcl_interp = Tcl_CreateInterp();
	if (tcl_interp == 0) error("from Tcl_CreateInterp");

	/* tell Electric the TCL interpreter handle */
	el_tclinterpreter(tcl_interp);

	/* Make command-line arguments available in the Tcl variables "argc" and "argv" */
	Tcl_SetVar(tcl_interp, "argv", "", TCL_GLOBAL_ONLY);
	Tcl_SetVar(tcl_interp, "argc", "0", TCL_GLOBAL_ONLY);
	Tcl_SetVar(tcl_interp, "argv0", "electric", TCL_GLOBAL_ONLY);

	/* Set the "tcl_interactive" variable */
	Tcl_SetVar(tcl_interp, "tcl_interactive", "1", TCL_GLOBAL_ONLY);

	/* initialize the interpreter */
	err = Tcl_Init(tcl_interp);
	if (err != TCL_OK) error("(from Tcl_Init) %s", tcl_interp->result);

	return(err);
}
#endif

/****************************** PRINTING ******************************/

extern "C"
{
#  include "eio.h"

	void gra_printwindow(void);
}

/* call this routine to print the current window */
void gra_printwindow(void)
{
	INTBIG pagewid, pagehei, pagehpixperinch, pagevpixperinch,
		marginx, marginy, slx, shx, sly, shy, hlx, hhx, hly, hhy,
		ulx, uhx, uly, uhy, width, height, centerx, centery, prod1, prod2;
	HCURSOR hCursorBusy, hCursorOld;
	static DOCINFO di = {sizeof(DOCINFO), "Electric", NULL};
	CDC printDC, *pDCPrint;
	HDC hPrnDC;
	WINDOW *win;
	NODEPROTO *np;
	REGISTER VARIABLE *varstate;
	WINDOWFRAME *mw;

	/* get facet to plot */
	win = el_curwindow;
	if (win == NOWINDOW || win->curnodeproto == NONODEPROTO)
	{
		ttyputerr("No current facet to plot");
		return;
	}
	mw = (WINDOWFRAME *)win->frame;
	np = win->curnodeproto;

	/* determine area to plot (in database units) */
	if (framesize(&shx, &shy, np) == 0)
	{
		shx /= 2;   slx = -shx;
		shy /= 2;   sly = -shy;
	} else
	{
		slx = np->lowx;   shx = np->highx;
		sly = np->lowy;   shy = np->highy;
	}

	/* see if focusing on highlighted area */
	varstate = getvalkey((INTBIG)io_aid, VAID, VINTEGER, io_state);
	if (varstate != NOVARIABLE && (varstate->addr&PLOTFOCUS) != 0)
	{
		if (askaid(us_aid, "get-highlighted-area", (INTBIG)&hlx, (INTBIG)&hhx, (INTBIG)&hly, (INTBIG)&hhy) != 0)
			ttyputerr("Warning: no highlighted area; printing entire facet"); else
		{
			slx = hlx;   shx = hhx;
			sly = hly;   shy = hhy;
		}
	}

	/* convert to screen units */
	(void)us_makescreen(&slx, &sly, &shx, &shy, win);

	/* Let user select printer */
	CPrintDialog printDlg(FALSE);
	if (printDlg.DoModal() != IDOK) return;

	/* Set cursor to busy */
	hCursorBusy = LoadCursor(NULL, IDC_WAIT);
	hCursorOld = SetCursor(hCursorBusy);

	/* Collect size information about the Printer DC */
	hPrnDC = printDlg.GetPrinterDC();
	pDCPrint = printDC.FromHandle(hPrnDC);
	pagewid = pDCPrint->GetDeviceCaps(HORZRES);
	pagehei = pDCPrint->GetDeviceCaps(VERTRES);
	pagehpixperinch = pDCPrint->GetDeviceCaps(LOGPIXELSX);
	pagevpixperinch = pDCPrint->GetDeviceCaps(LOGPIXELSY);
	marginx = pagehpixperinch / 2;
	marginy = pagevpixperinch / 2;

	/* make the plot have square pixels */
	ulx = marginx;
	uly = marginy;
	uhx = pagewid-marginx;
	uhy = pagehei-marginy;
	prod1 = (shx - slx) * (uhy - uly);
	prod2 = (shy - sly) * (uhx - ulx);
	if (prod1 != prod2)
	{
		/* adjust the scale */
		if (prod1 > prod2)
		{
			/* make it fill the width of the screen */
			height = muldiv(uhx - ulx, shy - sly, shx - slx);
			centery = (uly + uhy) / 2;
			uly = centery = height/2;
			uhy = uly + height;
		} else
		{
			/* make it fill the height of the screen */
			width = muldiv(uhy - uly, shx - slx, shy - sly);
			centerx = (ulx + uhx) / 2;
			ulx = centerx = width/2;
			uhx = ulx + width;
		}
	}

	/* Start printing */
	pDCPrint->StartDoc(&di);
	pDCPrint->StartPage();

	/* print the window */
	StretchDIBits(hPrnDC, ulx, uly, uhx-ulx, uhy-uly, 
		slx, sly, shx-slx, shy-sly,
		mw->data, mw->bminfo, DIB_RGB_COLORS, SRCCOPY);

	/* finish printing */
	pDCPrint->EndPage();
	pDCPrint->EndDoc();
	pDCPrint->Detach();
	DeleteDC(hPrnDC);

	/* Restore original cursor */
	SetCursor(hCursorOld);
}
