/*
 * Electric(tm) VLSI Design System
 *
 * File: graphunixsys.cpp
 * Interface for Unix computers.
 * Written by: Russell Wright, Electric Editor Incorporated
 *
 * Copyright (c) 1998 Electric Editor Incorporated.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Electric Editor Incorporated
 * 23470 Sunset Drive, Suite 108
 * Los Gatos, California 95033
 * support@electriceditor.com
 */

#include "global.h"
#include "egraphics.h"
#include "usr.h"
#include "dblang.h"
#include "database.h"
#if HAVE_SYS_WAIT_H
#  include <sys/wait.h>
#endif
#include <errno.h>
#include <signal.h>

#if LANGTCL
# include "tcl.h"
#endif

#ifdef USETK
# include "tk.h"
# define MYEVENTS (ExposureMask | ButtonPressMask | ButtonReleaseMask | KeyPressMask | EnterWindowMask | LeaveWindowMask |\
                   VisibilityChangeMask | PointerMotionMask | StructureNotifyMask | FocusChangeMask)
  Tk_Window gra_tktopwindow;
  void InitializeTK(void);
  Tk_Window MakeTKWindow(short type, short left, short right, short top, short bottom,
                         void (*eventproc)(ClientData, XEvent*));
#else
#  if LANGTCL
#    include "tcl.h"
#  endif
#endif  /* USETK */

#if LANGTCL
  Tcl_Interp *myTCLInterp;
  int InitializeTCL(void);
#endif

/****** prototypes for local routines ******/
void db_bombout_error(char*);
RETSIGTYPE db_sigill_trap(void);
RETSIGTYPE db_sigfpe_trap(void);
RETSIGTYPE db_sigbus_trap(void);
RETSIGTYPE db_sigsegv_trap(void);
void el_primaryinit(void);
void el_secondaryinit(INTBIG, char*[]);
void el_slice(void);

/* PAC 4/5/94  - supply definition of SIG_PF */
/*               as it seems to be missing from signal.h */
typedef void (*SIG_PF)();

int main(int argc, char *argv[])
{
    /* on A/UX machines, simulate Berkeley signal catching */
#ifdef aux
    set42sig();
#endif

    /* catch signals if Electric trys to bomb out */
    (void)signal(SIGILL, (SIG_PF)db_sigill_trap);
    (void)signal(SIGFPE, (SIG_PF)db_sigfpe_trap);
    (void)signal(SIGBUS, (SIG_PF)db_sigbus_trap);
    (void)signal(SIGSEGV, (SIG_PF)db_sigsegv_trap);

    /* primary initialization of Electric */
    el_primaryinit();
 
#if LANGTCL
    /* initialize TCL here */
    if (InitializeTCL() == TCL_ERROR)
        error("InitializeTCL failed: %s\n", myTCLInterp->result);
 
#ifdef USETK
    /* initialize TK here */
    InitializeTK();
#endif
#endif  /* LANGTCL */
 
    /* secondary initialization of Electric */
    el_secondaryinit(argc, argv);

    /* now run Electric */
    for(;;)
    {
        el_slice();
    }
}

#ifdef USETK
void InitializeTK(void)
{
    int code;
 
    if (Tk_Init(myTCLInterp) == TCL_ERROR)
    {
        ttyputerr("Tk_Init failed: %s", tcl_interp->result);
        return;
    }
 
    /* get the top Tk window handle */
    gra_tktopwindow = Tk_MainWindow(myTCLInterp);
    if (gra_tktopwindow == NULL)
        error("(from Tk_MainWindow) %s\n", myTCLInterp->result);
 
    /* withdraw the top window */
    if (Tcl_VarEval(myTCLInterp, "wm withdraw .", (char *)NULL) == TCL_ERROR)
        error("(from Tcl_VarEval) %s\n", myTCLInterp->result);
 
    (void)Tk_SetClass(gra_tktopwindow, "Electric");
    (void)Tcl_ResetResult(myTCLInterp);
 
    /* initialize tk package */
    if (Tcl_PkgRequire(myTCLInterp, "Tcl", TCL_VERSION, 1) == NULL)
        error("(from Tcl_PkgRequire) %s", myTCLInterp->result);
    code = Tcl_PkgProvide(myTCLInterp, "Tk", TK_VERSION);
    if (code != TCL_OK) error("(from Tcl_PkgProvide) %s", myTCLInterp->result);
}

/*
 * Routine to create a TK window of type "type":
 *  1  Scrolling text window
 *  2  Graphic edit window
 *  3  Dialog window with title banner
 *  4  Dialog windiw with no title banner
 * The window runs from "left" to "right" and from "top" to "bottom".
 * The event proc to use for this window is in "eventproc"
 */
Tk_Window MakeTKWindow(short type, short left, short right, short top, short bottom,
        void (*eventproc)(ClientData, XEvent*))
{
    Tk_Window win;
    char windowname[100], unsupportedcmd[200], *argv[3];
    static INTSML windowcount = 1;
 
    switch (type) {
        case 1:
            sprintf(windowname, ".status");
            break;

        case 2:
            sprintf(windowname, ".graphics");
            break;

        case 3:
        case 4:
            sprintf(windowname, ".electric%d", windowcount);
            windowcount++;
            break;
    }

#if 0
    win = Tk_CreateWindowFromPath(myTCLInterp, gra_tktopwindow, windowname, "");
    if (win == NULL)
    {
        ttyputerr("ERROR from Tk_CreateWindowFromPath: %s", myTCLInterp->result);
        Tcl_ResetResult(myTCLInterp);
        return(0);
    }
#else
        sprintf(unsupportedcmd, "etoplevel %s", windowname);
        if (Tcl_VarEval(myTCLInterp, unsupportedcmd, 0) == TCL_ERROR)
                ttyputerr("Tcl_VarEval returned %s", myTCLInterp->result);
        win = Tk_NameToWindow(myTCLInterp, windowname, gra_tktopwindow);
#endif
 
    (void)Tk_CreateEventHandler(win, MYEVENTS, eventproc, (ClientData)NULL);
    (void)Tk_GeometryRequest(win, right-left, bottom-top);
    (void)Tk_MoveToplevelWindow(win, left, top);
#ifdef STANDALONE
    (void)Tk_MapWindow(win);
#endif
    (void)Tk_MakeWindowExist(win);
 
    if (type == 3 || type == 4)
    {
        argv[0] = "grab";
        argv[1] = "-global";
        argv[2] = windowname;
        if (Tk_GrabCmd((ClientData)gra_tktopwindow, myTCLInterp, 3, argv) != TCL_OK)
        {
                ttyputmsg("ERROR from Tk_GrabCmd: %s", myTCLInterp->result);
                Tcl_ResetResult(myTCLInterp);
        }
    }
 
    return(win);
}
#endif  /* USETK */

#if LANGTCL
int InitializeTCL(void)
{
        int err;
        char *newArgv[2];
 
        /* set the program name/path */
        newArgv[0] = "Electric";
        newArgv[1] = NULL;
        (void)Tcl_FindExecutable(newArgv[0]);
 
        myTCLInterp = Tcl_CreateInterp();
        if (myTCLInterp == 0) error("from Tcl_CreateInterp");
 
        /* tell Electric the TCL interpreter handle */
        el_tclinterpreter(myTCLInterp);
 
        /* Make command-line arguments available in the Tcl variables "argc" and "argv" */
        Tcl_SetVar(myTCLInterp, "argv", "", TCL_GLOBAL_ONLY);
        Tcl_SetVar(myTCLInterp, "argc", "0", TCL_GLOBAL_ONLY);
        Tcl_SetVar(myTCLInterp, "argv0", "electric", TCL_GLOBAL_ONLY);
 
        /* Set the "tcl_interactive" variable */
        Tcl_SetVar(myTCLInterp, "tcl_interactive", "1", TCL_GLOBAL_ONLY);
 
        /* initialize the interpreter */
        err = Tcl_Init(myTCLInterp);
        if (err != TCL_OK) error("(from Tcl_Init) %s", myTCLInterp->result);
 
        return(err);
}
#endif  /* LANGTCL */

RETSIGTYPE db_sigill_trap(void)
{
        (void)signal(SIGILL, (SIG_PF)db_sigill_trap);
        db_bombout_error("An illegal instruction has been trapped");
}
 
RETSIGTYPE db_sigfpe_trap(void)
{
        (void)signal(SIGFPE, (SIG_PF)db_sigfpe_trap);
        db_bombout_error("A numerical error has occurred");
}
 
RETSIGTYPE db_sigbus_trap(void)
{
        (void)signal(SIGBUS, (SIG_PF)db_sigbus_trap);
        db_bombout_error("A bus error has occurred");
}

RETSIGTYPE db_sigsegv_trap(void)
{
        (void)signal(SIGSEGV, (SIG_PF)db_sigsegv_trap);
        db_bombout_error("A segmentation violation has occurred");
}

void db_bombout_error(char *message)
{
        extern COMCOMP us_userp;
        extern AIDENTRY *us_aid;
        char *pars[10];
        REGISTER INTSML count;
 
        ttyputerr("ERROR: %s", message);
#ifndef vms
        db_offercoredump();
#endif
        ttyputmsg("Execution proceeding at your own risk");
        ttyputmsg("'debug check-database, library write, and quit' are recommended");
        for(;;)
        {
                count = ttygetfullparam("> ", &us_userp, 10, pars);
                if (count != 0) (void)tellaid(us_aid, count, pars);
        }
}

