/*
 * Electric(tm) VLSI Design System
 *
 * File: graphunixx11.c
 * X Window System, version 11 graphics and keyboard input manager
 * Written by: Steven M. Rubin, Electric Editor Incorporated
 *
 * Copyright (c) 1998 Electric Editor Incorporated.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Electric Editor Incorporated
 * 23470 Sunset Drive, Suite 108
 * Los Gatos, California 95033
 * support@electriceditor.com
 */

/*
 ***************************** Defines ***************************************
 * Define Options: (default shown in [] )
 * SWAPPED [depends on platform] - byte order in bit maps
 * SHOWCURSOR [defined] - continuously shows position of cursor in top
 * of window
 * Electric, instead of using terminal window from which program starts
 * HASSPYGLASS [depends on system]- operates magnifier within Electric
 *************************** Revisions ****************************************
 * SRP930805 fixed crash bug in gra_showcoords() when us_alignment is zero
 * Revision July 1993 to remove failure to call us_squarescreen()
 * when window is partly off the screen. SRP930713
 * Revision June 1993 to create separate X messages window by: Sid Penstone.
 * It uses backing store on the messages window to avoid buffering
 * of the window data.
 *
 * Sept.1992: Change fonts, use resource file, use other R4 functions
 * Add separate font for editor use
 * This version has option to use backing store, if you have enough
 * X-Server resources. Declare the X resource "Electric.retained:True"
 * in the .Xdefaults or equivalent file before starting.
 * This disables the magnifier option
 *
 * Dec/91: Sid Penstone, Queen's University
 * Added cursor display on top line of window gra_showcoords()
 * Comment out SHOWCURSOR to remove it (SRP)
 */

#include "global.h"
#include "egraphics.h"
#include "usr.h"
#include "edialogs.h"
#include "config.h"
#include <signal.h>
#include <errno.h>
#include <pwd.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include <X11/keysym.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/cursorfont.h>
#include <X11/Shell.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Text.h>
#include <X11/Xaw/AsciiText.h>

#ifdef HAVE_SYS_TYPES_H
#  include <sys/types.h>
#endif

#ifdef HAVE_UNISTD_H
#  include <unistd.h>
#endif

#ifdef HAVE_FCNTL_H
#  include <fcntl.h>
#endif

#ifdef TIME_WITH_SYS_TIME
#  include <sys/time.h>
#  include <time.h>
#else
#  ifdef HAVE_SYS_TIME_H
#    include <sys/time.h>
#  else
#    include <time.h>
#  endif
#endif

#ifdef HAVE_SYS_TIMEB_H
#  include <sys/timeb.h>
#endif

#ifdef HAVE_DIRENT_H
#  include <dirent.h>
#  define NAMELEN(dirent) strlen((dirent)->d_name)
#else
#  define dirent direct
#  define NAMELEN(dirent) (dirent)->d_namelen
#  ifdef HAVE_SYS_NDIR_H
#    include <sys/ndir.h>
#  endif
#  ifdef HAVE_SYS_DIR_H
#    include <sys/dir.h>
#  endif
#  ifdef HAVE_NDIR_H
#    include <ndir.h>
#  endif
#endif

#ifdef HAVE_TERMIO_H
#  include <termio.h>
#else
#  include <sgtty.h>
#endif

/****** windows ******/

#define MAXSTATUSLINES 1						/* number of status lines */
#define USE8PLANES  1							/* use 8 graphics planes */

/* #define STATIC_CMAP 1 */						/* static colormap */
/* #define BLKBGD  1 */							/* black background */

#ifdef __cplusplus
#  define VisualClass(v) v->c_class
#else
#  define VisualClass(v) v->class
#endif

static Display      *gra_dpy;
       Window        gra_win = (Window)NULL;
       INTBIG        gra_screen_num;
static Display      *gra_topdpy;
static Window        gra_topwin;
static Display      *gra_topmsgdpy;
static Window        gra_topmsgwin;
       Widget        gra_toplevelwidget, gra_graphicswidget;
       Widget        gra_msgtoplevelwidget, gra_messageswidget;
       INTSML        gra_status_height;			/* space for lines at bottom of screen */
       INTSML        gra_shei;					/* real height of main window */
static XColor        gra_xfc, gra_xbc;			/* cursor color structure  */
extern INTSML        gra_fgd, gra_bgd;			/* the monochrome colors */
       INTSML        gra_use_backing;			/* flag if backing store used */
static Atom          gra_wm_delete_window;
static Atom          gra_wm_protocols;
       XTextProperty wintitle;
       INTSML        us_dispstyle, us_swid, us_shei;
static GC            gra_gc;
static GC            gra_gcinv;
static GC            gra_gcstip;
static Colormap      gra_colormap;				/* system color map */
static INTSML        gra_offx;					/* for offsetting X coordinates */
static INTSML        gra_revy;					/* for reversing Y coordinates */
static INTSML        gra_maphigh;				/* highest entry to use in color map */
static INTSML        gra_screenw, gra_screenh;	/* screen size */
static INTBIG        gra_sdep;					/* depth of frame buffer in bits */
static INTBIG        gra_ttydevice;				/* input device number */
#ifdef HAVE_TERMIO_H
  struct termio us_sttybuf;
#else
  struct sgttyb us_sttybuf;
#endif
static INTSML        gra_logrecordindex = 0;	/* count for session flushing */
static INTSML        gra_playbackmultiple=0;	/* count for multi-key playback */
static INTBIG        gra_argc;					/* "argc" for display */
static char        **gra_argv;					/* "argv" for display */
extern char         *el_version;
       char          gra_localstring[256];
extern GRAPHICS      us_box, us_ebox, us_menutext;

Display   *gra_finddisplay(void);
void       gra_loadstipple(UINTSML raster[]);
INTBIG     gra_errors(Display *dpy, XErrorEvent *err);

/****** the messages window ******/
#define MESSAGESWIDTH        97					/* magic value for gra_stringput() */
#define MESSAGEBUFFERSIZE 10000					/* size of XT messages window */
#define MESLEADING            1					/* extra character spacing */

extern FILE   *tty_audit;						/* disk channel for saving terminal output */
static Window  gra_msgwin;						/* messages window */
static GC      gra_gc_messages;
static INTSML  gra_messages = 0;				/* 0 if no messages window */
       INTSML  gra_messages_obscured;			/* nonzero if messages window covered */

/****** events ******/
/* the meaning of "gra_inputstate" */
#define CHARREAD        0177	/* character that was read */
#define ISCHAR          0200	/* set if character read */
#define ISBUTTON       01400	/* set if button pushed (or released) */
#define ISLEFT          0400	/* set if left button */
#define ISMIDDLE       01000	/* set if middle button */
#define ISRIGHT        01400	/* set if right button */
#define BUTTONUP       02000	/* set if button was released */
#define SHIFTDOWN      04000	/* set if shift key was held down */
#define CONTROLDOWN   010000	/* set if control key was held down */
#define METADOWN      020000	/* set if meta key was held down */
#define DOUBLECL      040000	/* set if double-click */
#define MOTION       0100000	/* set if mouse motion detected */
#define NOEVENT           -1	/* set if nothing happened */

#define CTRLC   03
#define CTRLD   04

static INTBIG    gra_inputstate;			/* current state of device input */
static INTSML    gra_doublethresh;			/* threshold of double-click */
static INTSML    gra_ignoremotion;			/* nonzero to ignore motion events */
static INTSML    gra_cursorx, gra_cursory;	/* current position of mouse */
static INTSML    gra_lstcurx, gra_lstcury;	/* former position of mouse */

#if 0
static char *eventNames[] =
{
	"",
	"",
	"KeyPress",
	"KeyRelease",
	"ButtonPress",
	"ButtonRelease",
	"MotionNotify",
	"EnterNotify",
	"LeaveNotify",
	"FocusIn",
	"FocusOut",
	"KeymapNotify",
	"Expose",
	"GraphicsExpose",
	"NoExpose",
	"VisibilityNotify",
	"CreateNotfiy",
	"DestroyNotfiy",
	"UnmapNotify",
	"MapNotify",
	"MapRequest",
	"ReparentNotfiy",
	"ConfigureNotfiy",
	"ConfigureRequest",
	"GravityNotfiy",
	"ResizeRequest",
	"CirculateNotify",
	"CirculateRequest",
	"PropertyNotfiy",
	"SelectionClear",
	"SelectionRequest",
	"SelectionNotify",
	"ColormapNotify",
	"ClientMessage",
	"MappingNotify"
};
#endif

void       gra_handlemsgevent(XEvent *event);
void       gra_handleeditevent(XEvent *event);
void       gra_nextevent(void);
void       gra_repaint(INTSML redo);
void       gra_waitforaction(void);
RETSIGTYPE us_noint(int sig);
void       gra_destroygraphics(Widget w, XtPointer data, XEvent *event, Boolean *cont);
void       gra_graphics_event_handler(Widget w, XtPointer data, XEvent *event,
			Boolean *cont);
void       gra_messages_event_handler(Widget w, XtPointer data, XEvent *event,
			Boolean *cont);

/****** mouse buttons ******/
#define BUTTONS     27		/* cannot exceed NUMBUTS in "usr.h" */
#define REALBUTS    3		/* actual number of buttons */

struct {
	char  *name;				/* button name */
	INTSML unique;				/* number of letters that make it unique */
} gra_buttonname[BUTTONS] =
{
	{"LEFT", 1},    {"MIDDLE", 2},    {"RIGHT", 1},		/* unshifted */

	{"SLEFT", 2},   {"SMIDDLE", 3},   {"SRIGHT", 2},	/* shift held down */
	{"CLEFT", 2},   {"CMIDDLE", 3},   {"CRIGHT", 2},	/* control held down */
	{"MLEFT", 2},   {"MMIDDLE", 3},   {"MRIGHT", 2},	/* meta held down */

	{"SCLEFT", 3},  {"SCMIDDLE", 4},  {"SCRIGHT", 3},	/* shift and control held down*/
	{"SMLEFT", 3},  {"SMMIDDLE", 4},  {"SMRIGHT", 3},	/* shift and meta held down */
	{"CMLEFT", 3},  {"CMMIDDLE", 4},  {"CMRIGHT", 3},	/* control and meta held down */

	{"SCMLEFT", 4}, {"SCMMIDDLE", 4}, {"SCMRIGHT", 4},	/* shift, control, and meta */

	{"DLEFT", 2},   {"DMIDDLE", 2},   {"DRIGHT", 2}		/* double-click */
};

INTSML     gra_makebutton(INTBIG);

/****** cursors and icons ******/
#define SHOWCURSOR       1		/* show cursor co-ordinates in top right corner */

/* the meaning of "gra_cursorstate" */
#define NORMALCURSOR     0		/* a normal "X" cursor */
#define WANTTTYCURSOR    1		/* a "use the TTY" cursor */
#define PENCURSOR        2		/* a "draw with pen" cursor */
#define NULLCURSOR       3		/* a null cursor */
#define MENUCURSOR       4		/* a menu selection cursor */
#define HANDCURSOR       5		/* a hand dragging cursor */

static INTSML gra_cursorstate;	/* see defines above */
       Cursor gra_realcursor;
       Cursor gra_nomousecursor;
       Cursor gra_drawcursor;
       Cursor gra_nullcursor;
       Cursor gra_menucursor;
       Cursor gra_handcursor;

/* SWAPPED means that the bytes are LSB first */
#ifdef i386
#  define SWAPPED 1
#endif

#ifndef SWAPPED

/* the icon for Electric */
static UINTSML gra_icon[256] =
{
	0x1111, 0x1111, 0x1111, 0x1111, 0x1111, 0x1111, 0x1111, 0x1111,
	0x4444, 0x4444, 0x4444, 0x4444, 0x4444, 0x4444, 0x4444, 0x4444,
	0x11FD, 0xFFFF, 0xFFFF, 0x3F11, 0x1105, 0x0000, 0x0000, 0x2011,
	0x4402, 0x0000, 0x0000, 0x4044, 0x4401, 0x0000, 0x0000, 0x8044,
	0x9100, 0x0000, 0x0000, 0x0011, 0x9100, 0x0000, 0x0000, 0x0011,
	0x4400, 0x0000, 0x0000, 0x0046, 0x6400, 0x0000, 0x0000, 0x0044,
	0x3100, 0x0000, 0x0000, 0x0014, 0x1100, 0x0000, 0x0000, 0x0018,
	0x1480, 0x1F00, 0x00FC, 0x0048, 0x1480, 0x1F00, 0x00FC, 0x0048,
	0x0980, 0x1F00, 0x00FC, 0x0010, 0x0980, 0x1F00, 0x00FC, 0x0010,
	0x0480, 0x1F00, 0x00FC, 0x0060, 0x0480, 0x1F00, 0x00FC, 0x0060,
	0x0580, 0x1F00, 0x00FC, 0x0020, 0x0380, 0x1F00, 0x00FC, 0x0040,
	0x0280, 0x1F00, 0x00FC, 0x0040, 0x0280, 0x1F00, 0x00FC, 0x0040,
	0x0380, 0x1F00, 0x00FC, 0x0040, 0x0380, 0x1F00, 0x00FC, 0x0040,
	0x0180, 0x1F00, 0x00FC, 0x00C0, 0x0180, 0x1F00, 0x00FC, 0x00C0,
	0x0180, 0x1F00, 0x00FC, 0x0080, 0x0180, 0x1F00, 0x00FC, 0x0080,
	0x0180, 0x1F00, 0x00FC, 0x0080, 0x0180, 0x1F00, 0x00FC, 0x0080,
	0x0100, 0x0000, 0x0000, 0x0080, 0x0100, 0x0000, 0x0000, 0x0080,
	0x0100, 0x0000, 0x0000, 0x0080, 0x0100, 0x0000, 0x0000, 0x0080,
	0x0100, 0x0000, 0x0000, 0x0080, 0x0100, 0x0000, 0x0000, 0x0080,
	0x0100, 0x00F0, 0x0700, 0x00C0, 0x0200, 0x00F8, 0x0F00, 0x0040,
	0x0300, 0x00FC, 0x1F00, 0x0040, 0x0300, 0x00FE, 0x3F00, 0x0040,
	0x0600, 0x00FE, 0x3F00, 0x0040, 0x0600, 0x00FE, 0x3F00, 0x0040,
	0x0500, 0x00FE, 0x3F00, 0x0020, 0x0500, 0x00FE, 0x3F00, 0x0020,
	0x0400, 0x00FE, 0x3F00, 0x0060, 0x0C00, 0x00FE, 0x3F00, 0x0050,
	0x1900, 0x00FE, 0x3F00, 0x0010, 0x1100, 0x00FE, 0x3F00, 0x0018,
	0x1400, 0x00FE, 0x3F00, 0x0048, 0x1400, 0x00FE, 0x3F00, 0x0048,
	0x3100, 0x0000, 0x0000, 0x0014, 0x3100, 0x0000, 0x0000, 0x0014,
	0x4400, 0x0000, 0x0000, 0x0046, 0xC400, 0x0000, 0x0000, 0x0045,
	0x9101, 0x0000, 0x0000, 0x0011, 0x1103, 0x0000, 0x0000, 0x8011,
	0x4406, 0x0000, 0x0000, 0x4044, 0x44FC, 0xFFFF, 0xFFFF, 0x7F44,
	0x1111, 0x1111, 0x1111, 0x1111, 0x1111, 0x1111, 0x1111, 0x1111,
	0x4444, 0x4444, 0x4444, 0x4444, 0x4444, 0x4444, 0x4444, 0x4444
};

/* the normal cursor (NORMALCURSOR, an "X") */
static UINTSML gra_realcursordata[16] = {
	0x03C0, 0x07E0, 0x0E70, 0x1C38, 0x381C, 0x700E, 0xE007, 0x4002,
	0x4002, 0xE007, 0x700E, 0x381C, 0x1C38, 0x0E70, 0x07E0, 0x03C0
};

/* the "use the TTY" cursor (WANTTTYCURSOR, the word "tty" above keyboard) */
static UINTSML gra_nomousecursordata[16] = {
	0x5777, 0x5215, 0x7237, 0x2211, 0x2271, 0x0000, 0xFFFF, 0x0180,
	0xA98A, 0x5195, 0xA98A, 0x5195, 0x0180, 0xF99F, 0x0180, 0xFFFF
};

/* the "draw with pen" cursor (PENCURSOR, a pen) */
static UINTSML gra_drawcursordata[16] = {
	0x0030, 0x0078, 0x00F4, 0x00E2, 0x0041, 0x8020, 0x4010, 0x2008,
	0x1004, 0x0802, 0x0C01, 0x8C00, 0x7E00, 0x3E00, 0x0F00, 0x0300
};

/* the null cursor (NULLCURSOR, an egg timer) */
static UINTSML gra_nullcursordata[16] = {
	0xF81F, 0x0420, 0x0420, 0x0420, 0xA81A, 0xF00F, 0xE007, 0xC003,
	0x4002, 0x2004, 0x1008, 0x0810, 0xA42A, 0xFC3F, 0xFC3F, 0xF81F
};

/* the menu selection cursor (MENUCURSOR, a sideways arrow) */
static UINTSML gra_menucursordata[16] = {
	0x0000, 0x0000, 0x0004, 0x000C, 0x001C, 0x003C, 0xFF7F, 0xFFFF,
	0xFF7F, 0x003C, 0x001C, 0x000C, 0x0004, 0x0000, 0x0000, 0x0000
};

/* the "move" command dragging cursor (HANDCURSOR, a dragging hand) */
static UINTSML gra_handcursordata[16] = {
	0x0000, 0x803F, 0x40C0, 0x8003, 0x4000, 0x2000, 0xC001, 0x2000,
	0x1000, 0xFE03, 0x0100, 0x0100, 0xFE1F, 0x8080, 0x0041, 0x003E
};

#else  /* the byte swapped cursor definitions */

/* the icon for Electric  */
static UINTSML gra_icon[256] =
{
	0x1111, 0x1111, 0x1111, 0x1111, 0x1111, 0x1111, 0x1111, 0x1111,
	0x4444, 0x4444, 0x4444, 0x4444, 0x4444, 0x4444, 0x4444, 0x4444,
	0xFD11, 0xFFFF, 0xFFFF, 0x113F, 0x0511, 0x0000, 0x0000, 0x1120,
	0x0244, 0x0000, 0x0000, 0x4440, 0x0144, 0x0000, 0x0000, 0x4480,
	0x0091, 0x0000, 0x0000, 0x1100, 0x0091, 0x0000, 0x0000, 0x1100,
	0x0044, 0x0000, 0x0000, 0x4600, 0x0064, 0x0000, 0x0000, 0x4400,
	0x0031, 0x0000, 0x0000, 0x1400, 0x0011, 0x0000, 0x0000, 0x1800,
	0x8014, 0x001F, 0xFC00, 0x4800, 0x8014, 0x001F, 0xFC00, 0x4800,
	0x8009, 0x001F, 0xFC00, 0x1000, 0x8009, 0x001F, 0xFC00, 0x1000,
	0x8004, 0x001F, 0xFC00, 0x6000, 0x8004, 0x001F, 0xFC00, 0x6000,
	0x8005, 0x001F, 0xFC00, 0x2000, 0x8003, 0x001F, 0xFC00, 0x4000,
	0x8002, 0x001F, 0xFC00, 0x4000, 0x8002, 0x001F, 0xFc00, 0x4000,
	0x8003, 0x001F, 0xFC00, 0x4000, 0x8003, 0x001F, 0xFC00, 0x4000,
	0x8001, 0x001F, 0xFC00, 0xC000, 0x8001, 0x001F, 0xFC00, 0xC000,
	0x8001, 0x001F, 0xFC00, 0x8000, 0x8001, 0x001F, 0xFC00, 0x8000,
	0x8001, 0x001F, 0xFC00, 0x8000, 0x8001, 0x001F, 0xFC00, 0x8000,
	0x0001, 0x0000, 0x0000, 0x8000, 0x0001, 0x0000, 0x0000, 0x8000,
	0x0001, 0x0000, 0x0000, 0x8000, 0x0001, 0x0000, 0x0000, 0x8000,
	0x0001, 0x0000, 0x0000, 0x8000, 0x0001, 0x0000, 0x0000, 0x8000,
	0x0001, 0xF000, 0x0007, 0xC000, 0x0002, 0xF800, 0x000F, 0x4000,
	0x0003, 0xFC00, 0x001F, 0x4000, 0x0003, 0xFE00, 0x003F, 0x4000,
	0x0006, 0xFE00, 0x003F, 0x4000, 0x0006, 0xFE00, 0x003F, 0x4000,
	0x0005, 0xFE00, 0x003F, 0x2000, 0x0005, 0xFE00, 0x003F, 0x2000,
	0x0004, 0xFE00, 0x003F, 0x6000, 0x000C, 0xFE00, 0x003F, 0x5000,
	0x0019, 0xFE00, 0x003F, 0x1000, 0x0011, 0xFE00, 0x003F, 0x1800,
	0x0014, 0xFE00, 0x003F, 0x4800, 0x0014, 0xFE00, 0x003F, 0x4800,
	0x0031, 0x0000, 0x0000, 0x1400, 0x0031, 0x0000, 0x0000, 0x1400,
	0x0044, 0x0000, 0x0000, 0x4600, 0x00C4, 0x0000, 0x0000, 0x4500,
	0x0191, 0x0000, 0x0000, 0x1100, 0x0311, 0x0000, 0x0000, 0x1180,
	0x0644, 0x0000, 0x0000, 0x4440, 0xFC44, 0xFFFF, 0xFFFF, 0x447F,
	0x1111, 0x1111, 0x1111, 0x1111, 0x1111, 0x1111, 0x1111, 0x1111,
	0x4444, 0x4444, 0x4444, 0x4444, 0x4444, 0x4444, 0x4444, 0x4444
};

/* the normal cursor (NORMALCURSOR, an "X") */
static UINTSML gra_realcursordata[16] = {
	0xC003, 0xE007, 0x700E, 0x381C, 0x1C38, 0x0E70, 0x07E0, 0x0240,
	0x0240, 0x07E0, 0x0E70, 0x1C38, 0x381C, 0x700E, 0xE007, 0xC003
};

/* the "use the TTY" cursor (WANTTTYCURSOR, the word "tty" above keyboard) */
static UINTSML gra_nomousecursordata[16] = {
	0x7757, 0x1552, 0x3772, 0x1122, 0x7122, 0x0000, 0xFFFF, 0x8001,
	0x8AA9, 0x9551, 0x8AA9, 0x9551, 0x8001, 0x9FF9, 0x8001, 0xFFFF
};

/* the "draw with pen" cursor (PENCURSOR, a pen) */
static UINTSML gra_drawcursordata[16] = {
	0x3000, 0x7800, 0xF400, 0xE200, 0x4100, 0x2080, 0x1040, 0x0820,
	0x0410, 0x0208, 0x010C, 0x008C, 0x007E, 0x003E, 0x000F, 0x0003
};

/* the null cursor (NULLCURSOR, an egg timer) */
static UINTSML gra_nullcursordata[16] = {
	0x1FF8, 0x2004, 0x2004, 0x2004, 0x1AA8, 0x0FF0, 0x07E0, 0x03C0,
	0x0240, 0x0420, 0x0810, 0x1008, 0x2AA4, 0x3FFC, 0x3FFC, 0x1FF8
};

/* the menu selection cursor (MENUCURSOR, a sideways arrow) */
static UINTSML gra_menucursordata[16] = {
	0x0000, 0x0000, 0x0400, 0x0C00, 0x1C00, 0x3C00, 0x7FFF, 0xFFFF,
	0x7FFF, 0x3C00, 0x1C00, 0x0C00, 0x0400, 0x0000, 0x0000, 0x0000
};

/* the "move" command dragging cursor (HANDCURSOR, a dragging hand) */
static UINTSML gra_handcursordata[16] = {
	0x0000, 0x3F80, 0xC040, 0x0380, 0x0040, 0x0020, 0x01C0, 0x0020,
	0x0010, 0x03FE, 0x0001, 0x0001, 0x1FFE, 0x8080, 0x4100, 0x3E00
};

#endif  /* !SWAPPED */

void       gra_recolorcursor(XColor *fg, XColor *bg);
#ifdef SHOWCURSOR
void       gra_showcoords(INTSML x, INTSML y);
#endif

/****** fonts ******/
/* font searches with XListFonts */
#define FONTLISTSIZE 10

typedef struct
{
	XFontStruct *font;
	char *fontname;
} FontRec;

FontRec gra_messages_font = {(XFontStruct *)0, "fixed"};

/*
 * Default font - this font MUST exist, or Electric will die.  Best to choose a
 * font that is commonly used, so the X server is likely to have it loaded.
 */
#define DEFAULTFONTNAME "fixed"

static FontRec      gra_defaultFont;
static XFontStruct *gra_curfont;			/* current writing font */

char *gra_resource_fontname[] =
{
	"font0", "font1", "font2", "font3", "font4", "font5",
	"font6", "font7", "font8", "fontmenu", "fontedit", "fontstatus"
};

FontRec gra_font[] =
{
#ifdef sun
	{(XFontStruct *)0, "5x8"},                    /* 0 */
	{(XFontStruct *)0, "6x10"},                   /* 1 */
	{(XFontStruct *)0, "6x12"},                   /* 2 */
	{(XFontStruct *)0, "6x13"},                   /* 3 */
	{(XFontStruct *)0, "7x13"},                   /* 4 */
	{(XFontStruct *)0, "8x13"},                   /* 5 */
	{(XFontStruct *)0, "9x15"},                   /* 6 */
	{(XFontStruct *)0, "*courier-*-180-*"},       /* 7 */
	{(XFontStruct *)0, "-misc-fixed-*-200-*"},    /* 8 */
	{(XFontStruct *)0, "lucidasans-bold"},        /* MENU */
	{(XFontStruct *)0, "*fixed-bold-r-*-140-*"},  /* EDIT */
	{(XFontStruct *)0, "fixed"},                  /* STATUS */
#else /* for everyone else; these should be part of X distribution */
#if 0
	{(XFontStruct *)0, "-*-helvetica-medium-r-normal-*-*-80-*-*-*"},	/* 0 */
	{(XFontStruct *)0, "-*-fixed-medium-r-normal-*-*-90-*-*-*"},		/* 1 */
	{(XFontStruct *)0, "-*-helvetica-medium-r-normal-*-*-100-*-*-*"},	/* 2 */
	{(XFontStruct *)0, "-adobe-*-medium-r-normal-*-*-110-*-*-*"},		/* 3 */
	{(XFontStruct *)0, "-*-helvetica-medium-r-normal-*-*-120-*-*-*"},	/* 4 */
	{(XFontStruct *)0, "-*-clean-medium-r-normal-*-*-130-*-*-*"},		/* 5 */
	{(XFontStruct *)0, "-*-helvetica-medium-r-normal-*-*-140-*-*-*"},	/* 6 */
	{(XFontStruct *)0, "-*-clean-medium-r-normal-*-*-150-*-*-*"},		/* 7 */
	{(XFontStruct *)0, "-*-helvetica-medium-r-normal-*-*-180-*-*-*"},	/* 8 */
#else
	{(XFontStruct *)0, "-*-*-medium-r-normal-*-9-*-*-*-*"},				/* 0 */
	{(XFontStruct *)0, "-*-helvetica-medium-r-normal-*-10-*-*-*-*"},	/* 1 */
	{(XFontStruct *)0, "-*-helvetica-medium-r-normal-*-12-*-*-*-*"},	/* 2 */
	{(XFontStruct *)0, "-*-helvetica-medium-r-normal-*-14-*-*-*-*"},	/* 3 */
	{(XFontStruct *)0, "-*-clean-medium-r-normal-*-16-*-*-*-*"},		/* 4 */
	{(XFontStruct *)0, "-*-helvetica-medium-r-normal-*-18-*-*-*-*"},	/* 5 */
	{(XFontStruct *)0, "-*-*-medium-r-normal-*-20-*-*-*-*"},			/* 6 */
	{(XFontStruct *)0, "-*-helvetica-medium-r-normal-*-24-*-*-*-*"},	/* 7 */
	{(XFontStruct *)0, "-*-helvetica-bold-r-normal-*-24-*-*-*-*"},		/* 8 */
#endif
	{(XFontStruct *)0, "-*-helvetica-bold-r-normal-*-*-120-*-*-*"},		/* MENU */
	{(XFontStruct *)0, "*fixed-*-normal-*-*-120-*"},					/* EDIT */
	{(XFontStruct *)0, "fixed"},										/* STATUS */
#endif
	{(XFontStruct *)0, NULL}  /* terminator */
};

/****** rectangle saving ******/
#define NOSAVEDBOX ((SAVEDBOX *)-1)
typedef struct Isavedbox
{
	Pixmap  pix;
	INTBIG  code;
	INTSML  lx, hx, ly, hy;
	struct Isavedbox *nextsavedbox;
} SAVEDBOX;

SAVEDBOX *gra_firstsavedbox = NOSAVEDBOX;
INTSML    gra_savedboxcodes = 0;

/****** spyglass ******/
#define HASSPYGLASS 1  /* magnifying glass */

#ifdef HASSPYGLASS
#  define MAXMAG 8
#  define SMALLSIZE 16
#  define MAXSPYSIZE (SMALLSIZE*MAXMAG + 2)

static INTSML    gra_spyx, gra_spyy;		/* position of spyglass magnifier */
static INTSML    gra_spysize;				/* the size of the spyglass */
static XImage   *gra_spyrect;				/* the image under the spyglass */
static XImage   *gra_spyrect2;				/* the image under the spyglass */
static INTSML    gra_spymag;				/* the magnification of the spyglass */
static Pixmap    gra_spymap;				/* to save the spyglass */
static INTSML    gra_xmin, gra_ymin;		/* actual position of edge of the window */
static INTSML    gra_xmax, gra_ymax;		/* actual position of edge of the window */

void gra_getboundaries(Window win);
void gra_advancespy(void);
void gra_resetspy(void);
void gra_writespy(void);
#endif

/****** time ******/
static time_t    gra_timebasesec;
static INTBIG    gra_timebasems;
static Time      gra_lasttime = 0;			/* time of last click */

UINTBIG gra_machinetimeoffset(void);

/****** files ******/
void   *gra_fileliststringarray = 0;
char    tty_curpath[255] = {0};

/* File Input */
DIALOGITEM usr_fileindialogitems[] =
{
 /*  1 */ {0, {128,256,152,336}, BUTTON, "Open"},
 /*  2 */ {0, {176,256,200,336}, BUTTON, "Cancel"},
 /*  3 */ {0, {64,8,216,240}, SCROLL, ""},
 /*  4 */ {0, {8,8,24,336}, MESSAGE, ""},
 /*  5 */ {0, {32,48,64,336}, MESSAGE, ""},
 /*  6 */ {0, {80,256,104,336}, BUTTON, "Up"}
};
DIALOG usr_fileindialog = {{50,75,278,421}, 0, 6, usr_fileindialogitems};

/* File Output */
DIALOGITEM usr_fileoutdialogitems[] =
{
 /*  1 */ {0, {88,232,112,304}, BUTTON, "OK"},
 /*  2 */ {0, {88,32,112,104}, BUTTON, "Cancel"},
 /*  3 */ {0, {8,12,24,340}, MESSAGE, ""},
 /*  4 */ {0, {40,16,72,336}, EDITTEXT, ""}
};
DIALOG usr_fileoutdialog = {{50,75,172,421}, 0, 4, usr_fileoutdialogitems};

INTBIG gra_fileselectall(struct dirent *);
INTSML gra_addfiletolist(char *file);
void   gra_initfilelist(void);

/******************** INITIALIZATION ********************/

/*
 * Routine to establish the default display define tty codes.
 */
void us_setdisplay(char *name, INTBIG *argc, char **argv)
{
	/* save command arguments */
	gra_argc = *argc;
	gra_argv = argv;

	/* initialize keyboard input */
	gra_ttydevice = fileno(stdin);
#ifdef HAVE_TERMIO_H
	(void)ioctl(gra_ttydevice, TCGETA, &us_sttybuf);
#else
	(void)gtty(gra_ttydevice, &us_sttybuf);
#endif

	us_noint(0);
	return;
}

/* dummy interrupt handler to use during initialization*/
RETSIGTYPE us_noint(int sig)
{
	static INTSML start = 0;

#ifndef __SVR4
	(void)signal(SIGINT, us_noint);
#endif
	if (start) (void)fprintf(stderr, "Interrupted during initialization");
	start = 1;
}

/*
 * Routine to initialize the display device.
 */
INTSML us_initgraphics(INTSML messages)
{
#ifdef USE8PLANES
	INTBIG planes[8], pixels[256];
#else
	INTBIG planes[7], pixels[128];
#endif
	REGISTER INTBIG i, j, bg, bd, gra_blk, gra_wht;
	XSizeHints *xsh;
	XWMHints *xwmh;
	XClassHint *clsh;
	XWindowAttributes xwa;
	XSetWindowAttributes xswa;
	Pixmap pixm;
	XGCValues gcv;
	XEvent event;
	char *geomSpec = NULL;
	INTSML full_screen, does_backing, does_saveunders;
	char *ptr, **fontlist;
	Visual *visual;
	INTBIG bitmask, namecount;
	unsigned long valuemask;
	XVisualInfo vinfo;
	INTBIG classnew;
	XSizeHints *xsh_messages;
	XWMHints *xwmh_messages;
	XClassHint *clsh_messages;
	XSetWindowAttributes xswa_messages;
	unsigned long messagesvaluemask;
	XTextProperty swintitle, sicontitle;
	XTextProperty icontitle;
	char *iconname = "Electric";
	char *sswinname = "Electric Messages";
	char *ssiconname = "Messages";
	Arg arg[5];
	XTextProperty wintitle;
	char *buffer, *title;
	int graphicsheight, graphicswidth, msgwidth, msgheight;
	Cursor gra_ibeam;

	gra_messages = messages;

	/* initialize error handling */
	(void)XSetErrorHandler(gra_errors);

	/* double-click threshold (in milliseconds) */
	gra_doublethresh = 300;

	/* get the display */
	gra_dpy = gra_finddisplay();
	gra_screen_num = DefaultScreen(gra_dpy); /* We should use this everywhere */

	/* initial map length  and number of planes: */
#ifdef USE8PLANES
	el_maplength = (INTSML)DisplayCells(gra_dpy, gra_screen_num);
#else
	el_maplength = 128;
#endif
	gra_sdep = DefaultDepth(gra_dpy, gra_screen_num);

	full_screen = 0;
	for(i=1; i < gra_argc; i++)
	{
		char *opt = gra_argv[i];
		if (opt[0] == '-' )
		{
			switch (opt[1])
			{
				case 'f':
					full_screen = 1;
					continue;

				case 'm':
					if (gra_sdep > 1)
					{
						j = myatoi(&opt [2]);
						if (j <= 4 || j > 8)
							error("Map length (-mM) must be from 5 to 8\n");
						el_maplength = 1 << j;
					}
					continue;

				case 'g':       /* Geometry */
					if (++i >= gra_argc) continue;
					geomSpec = gra_argv[i];
					continue;
			}
		}
	}
	gra_maphigh = el_maplength - 1;

	/* check for backing store option */
	gra_use_backing = 0;
	does_backing = DoesBackingStore(DefaultScreenOfDisplay(gra_dpy));
	ptr = XGetDefault(gra_dpy, "Electric", "retained");
	if (ptr != NULL)
	{
		if (namesamen(ptr, "True", 4) == 0)
		{
			if (does_backing)
			{
				gra_use_backing = 1;
			} else
			{
				(void)fprintf(stderr, "Backing Store is not available.\n");
			}
		} else
		{
			gra_use_backing = 0;
		}
	}

	/* get fonts */
	gra_defaultFont.fontname = XGetDefault(gra_dpy, "Electric", "font");
	if (gra_defaultFont.fontname == NULL) gra_defaultFont.fontname = DEFAULTFONTNAME;

	gra_defaultFont.font = XLoadQueryFont(gra_dpy, gra_defaultFont.fontname);
	if (gra_defaultFont.font == NULL)
		(void)fprintf(stderr, "Cannot open default font \"%s\"\n", gra_defaultFont.fontname);

	/* Check the .Xdefaults or .Xresources file for a value for each of the
	 * requested fonts; they will have resource names .Font4, .Font6,...
	 * .FontStatus. Look for a matching font, if found, use it; otherwise look
	 * for the font that is hard-coded.
	 */

	for(i=0; gra_font[i].fontname != 0; i++)
	{
		/* check the resource database: */
		if ((ptr = XGetDefault(gra_dpy, "Electric", gra_resource_fontname[i])) != NULL)
		{
			fontlist = XListFonts(gra_dpy, ptr, 10, &namecount);
			if (namecount == 0)
			{
				/* server could not find a match */
				(void)fprintf(stderr, "Cannot find font '%s', using '%s'\n",
					ptr, gra_font[i].fontname);
			} else
			{
				/* replace the default (this is for debugging only) */
				(void)allocstring(&gra_font[i].fontname, fontlist[0], db_cluster);
			}
		} else
		{
			fontlist = XListFonts(gra_dpy, gra_font[i].fontname, 1, &namecount);
		}

		if (namecount != 0)
		{
			/* at least one version should have worked */
			gra_font[i].font = XLoadQueryFont(gra_dpy, fontlist[0]);
			XFreeFontNames(fontlist);
		}

		if (gra_font[i].font == NULL)
		{
			(void)fprintf(stderr, "Cannot find font '%s', using default\n",
				gra_font[i].fontname);
			gra_font[i].font = gra_defaultFont.font;
		}
	}

	if (gra_font[11].font != NULL)
	{
		gra_messages_font.font = gra_font[11].font;
		gra_messages_font.fontname = gra_font[11].fontname;
	}

	clsh = XAllocClassHint();
	xsh = XAllocSizeHints();
	xwmh = XAllocWMHints();

	clsh_messages = XAllocClassHint();
	xsh_messages = XAllocSizeHints();
	xwmh_messages = XAllocWMHints();

	/*
	 * Deal with providing the window with an initial position & size.
	 * If -f option was used, then go to full screen.  If a -geometry was used,
	 * then use it.  Fill out the XSizeHints struct to inform the window
	 * manager.
	 */
	gra_screenh = XHeightOfScreen(XDefaultScreenOfDisplay(gra_dpy));
	gra_screenw = XWidthOfScreen(XDefaultScreenOfDisplay(gra_dpy));

	 /* any geometry from command line ? */
	if (geomSpec == NULL) geomSpec = XGetDefault(gra_dpy, "Electric", "geometry");

	if (geomSpec == NULL || full_screen != 0)
	{
		/*
		 * The defaults database doesn't contain a specification of the initial
		 * size & position: use full screen if requested, or fit the window
		 * in the screen.
		 */
		xsh->flags = (PPosition | PSize);
		xsh->height = gra_screenh - 2;   /* room for window manager border */
		xsh->width = gra_screenw - 8;    /* room for openlook window manager */
		if (full_screen == 0)
		{
			xsh->height = xsh->height / 8 * 6;
		} else
		{
			xsh->height = gra_screenh - 24; /* room for a border */
		}

		xsh->x = 0;
		xsh->y = 0;
	} else
	{
		bitmask = XWMGeometry(gra_dpy, gra_screen_num, geomSpec, NULL, 1, xsh,
			&xsh->x, &xsh->y, &xsh->width, &xsh->height, &xsh->win_gravity);
		if (bitmask & (XValue | YValue)) xsh->flags |= USPosition;
		if (bitmask & (WidthValue | HeightValue)) xsh->flags |= USSize;
	}

	/*
	 * calculate the messages window for 15 by 97 at startup, at lower edge of of main window
	 */
	i = gra_messages_font.font->max_bounds.width;
	j = gra_messages_font.font->descent + gra_messages_font.font->ascent + MESLEADING;

	/* room for MAXSTATUSLINES lines - with padding*/
	gra_status_height = MAXSTATUSLINES * j + 4; 
	xsh_messages->width = i * MESSAGESWIDTH;
	if (full_screen != 0)
	{
		xsh_messages->height = 15 * j;
		xsh_messages->y = gra_screenh - xsh_messages->height - 50;
	} else
	{
		xsh_messages->height = gra_screenh - xsh->height - 80;
		xsh_messages->y = gra_screenh - xsh_messages->height - 52;
	}
	xsh_messages->min_width = 0;
	xsh_messages->min_height = 0;
	xsh_messages->x = xsh->x;  /* under the other window */
	xsh_messages->flags = (PPosition | PSize | PMinSize);

	gra_blk = BlackPixel(gra_dpy, gra_screen_num);  /* need these later */
	gra_wht = WhitePixel(gra_dpy, gra_screen_num);

	/* Electric assumes that it can load a color map that makes pixel 0
	 * the background pixel. On mono stations, the map is usually fixed,
	 * so erasing will always draw in whatever the station uses for pixel 0.
	 * Set up the monochrome drawing colors to match.
	 */

	if (gra_sdep > 1)
	{
		/* use the grid and background pixels */
		bd = 1;  /* window border */
		bg = 0;  /* start with  default background on a color screen */
	} else
	{
		/* monochrome definitions */
		bg = gra_wht;   /* others have 1 for black */
		bd = gra_blk;   /* white background on monochrome */;
		gra_fgd = bd;
		gra_bgd = bg;
	}

	/* define the cursors initially (This does not matter, anyhow!) */
	if (gra_sdep > 1)
	{
		gra_xfc.pixel = 1;  /* current map */
		gra_xbc.pixel = 0;  /* Electric uses 0 as background */
	} else
	{
		/* monochrome */
		gra_xfc.pixel = gra_fgd;
		gra_xbc.pixel = gra_bgd;
	}

	gra_xfc.flags = gra_xbc.flags = DoRed | DoGreen | DoBlue;

	XQueryColor(gra_dpy, XDefaultColormapOfScreen(XDefaultScreenOfDisplay(gra_dpy)),
		&gra_xfc);
	XQueryColor(gra_dpy, XDefaultColormapOfScreen(XDefaultScreenOfDisplay(gra_dpy)),
		&gra_xbc);

	/* Some servers change the default screen to StaticColor class, so that
	 * Electric cannot change the color map.  Try to find one that can be used.
	 */
	visual = DefaultVisual(gra_dpy, gra_screen_num);
	classnew = PseudoColor;
	if (gra_sdep > 1)
	{
		if (VisualClass(visual) != classnew)
		{
			/* look for pseudo color only for now */
			if (XMatchVisualInfo(gra_dpy, gra_screen_num, gra_sdep, classnew, &vinfo))
			{
				visual = vinfo.visual;
			} else
			{
				ttyputerr("Cannot find desired visual, carrying on anyhow.");
			}
		}
	}

	/* now we can create the window */
	xswa.background_pixel = bg;
	xswa.border_pixel = bd;
	valuemask = CWBackPixel | CWBorderPixel | CWColormap;

	if (gra_use_backing)
	{
		xswa.backing_store = WhenMapped;
		xswa.backing_pixel = bg;
		xswa.backing_planes = AllPlanes;
		valuemask = valuemask |= (CWBackingStore | CWBackingPixel | CWBackingPlanes);
	}

	xswa_messages.background_pixel = bg;
	xswa_messages.border_pixel = bd;
	xswa_messages.colormap = gra_colormap;
	xswa_messages.bit_gravity = NorthWestGravity;
	messagesvaluemask = CWBackPixel | CWBorderPixel | CWColormap | CWBitGravity;

	if (does_backing)
	{
		xswa_messages.backing_store = Always;
		xswa_messages.backing_pixel = bg;
		xswa_messages.backing_planes = AllPlanes;
		messagesvaluemask |= CWBackingStore | CWBackingPixel | CWBackingPlanes;
	} else
	{
		(void)fprintf(stderr, "Backing Store is not available\n");
	}

	/* try to save under the messages window... may not always work */
	does_saveunders = DoesSaveUnders(DefaultScreenOfDisplay(gra_dpy));
	if (does_saveunders)
	{
		xswa_messages.save_under = True;
		messagesvaluemask |= CWSaveUnder;
	}

	/* make top-level widget */
	gra_toplevelwidget = XtInitialize("electric", "Electric", NULL, 0, &gra_argc, gra_argv);
	XtSetArg(arg[0], XtNx, 1);
	XtSetArg(arg[1], XtNy, 0);
	XtSetValues(gra_toplevelwidget, arg, 2);

	/* get display size */
	graphicswidth = gra_screenw - 10;
	graphicsheight = (gra_screenh-2)/8*6;
	msgwidth = 600;
	msgheight = gra_screenh - graphicsheight - 80;

	/* make graphics widget */
	gra_graphicswidget = XtVaCreateManagedWidget("graphics", boxWidgetClass,
		gra_toplevelwidget,
		XtNwidth,    graphicswidth,
		XtNheight,   graphicsheight,
		NULL);
	XtAddEventHandler(gra_graphicswidget, ButtonPressMask | ButtonReleaseMask |
		KeyPressMask | PointerMotionMask | ExposureMask, FALSE,
		gra_graphics_event_handler, NULL);
	XtRealizeWidget(gra_toplevelwidget);

	/* get info about this widget */
	gra_topwin = XtWindow(gra_toplevelwidget);
	gra_topdpy = XtDisplay(gra_toplevelwidget);
	gra_dpy = XtDisplay(gra_graphicswidget);
	gra_win = XtWindow(gra_graphicswidget);

	/* load window manager information */
	(void)strcpy(gra_localstring, "Electric");
	ptr = gra_localstring;
	XStringListToTextProperty(&ptr, 1, &wintitle);
	XStringListToTextProperty(&iconname, 1, &icontitle);
	xwmh->input = True;
	xwmh->initial_state = NormalState;
	xwmh->icon_pixmap = xwmh->icon_mask = XCreateBitmapFromData(gra_topdpy, gra_topwin,
		(const char *)gra_icon, 64, 64);
	xwmh->flags = InputHint | StateHint | IconPixmapHint;
	XSetWMProperties(gra_topdpy, gra_topwin, &wintitle, &icontitle,
		gra_argv, gra_argc, xsh, xwmh, NULL);

#ifndef STATIC_CMAP
	gra_colormap = XCreateColormap(gra_topdpy, RootWindow(gra_topdpy, gra_screen_num),
		visual, AllocNone);
#else
	gra_colormap = XDefaultColormap(gra_topdpy, gra_screen_num);
#endif
	xswa.colormap = gra_colormap;
	XChangeWindowAttributes(gra_topdpy, gra_topwin, CWColormap, &xswa);

	if (gra_messages != 0)
	{
		/* make second top-level shell for messages */
		gra_msgtoplevelwidget = XtCreateApplicationShell(NULL, topLevelShellWidgetClass,
			0, 0);
		XtSetArg(arg[0], XtNx, 100);
		XtSetArg(arg[1], XtNy, graphicsheight+28);
		XtSetArg(arg[2], XtNwidth, msgwidth);
		XtSetArg(arg[3], XtNheight, msgheight);
		XtSetValues(gra_msgtoplevelwidget, arg, 4);

		/* make text widget for messages */
		buffer = (char *)emalloc(MESSAGEBUFFERSIZE, db_cluster);
		buffer[0] = 0;
		gra_messageswidget = XtVaCreateManagedWidget("messages", asciiTextWidgetClass,
			gra_msgtoplevelwidget,
			XtNstring,           buffer,
			XtNbackground,       WHITE,
			XtNborderColor,      BLACK,
			XtNforeground,       BLACK,
			XtNlength,           MESSAGEBUFFERSIZE,
			XtNeditType,         XawtextEdit,
			XtNscrollHorizontal, XawtextScrollAlways,
			XtNscrollVertical,   XawtextScrollAlways,
			NULL);
		XtAddEventHandler(gra_messageswidget, ButtonPressMask | KeyPressMask |
			VisibilityChangeMask, FALSE, gra_messages_event_handler, NULL);
		XtRealizeWidget(gra_msgtoplevelwidget);
		gra_msgwin = XtWindow(gra_messageswidget);
		gra_messages_obscured = 0;

		/* label the messages widget */
		gra_topmsgwin = XtWindow(gra_msgtoplevelwidget);
		gra_topmsgdpy = XtDisplay(gra_msgtoplevelwidget);
		title = "Electric Messages";
		XStringListToTextProperty(&title, 1, &wintitle);
		XSetWMName(gra_topmsgdpy, gra_topmsgwin, &wintitle);
		XChangeWindowAttributes(gra_topmsgdpy, gra_topmsgwin, CWColormap, &xswa);

		/* set I-beam cursor */
		gra_ibeam = XCreateFontCursor(gra_topmsgdpy, XC_xterm);
		XDefineCursor(gra_topmsgdpy, gra_topmsgwin, gra_ibeam);

		/* load window manager information */
		XStringListToTextProperty(&sswinname, 1, &swintitle);
		XStringListToTextProperty(&ssiconname, 1, &sicontitle);
		xwmh_messages->input = True;
		xwmh_messages->initial_state = NormalState;
		xwmh_messages->icon_pixmap = xwmh->icon_mask = xwmh->icon_pixmap;
		xwmh_messages->icon_x = xsh_messages->x;
		xwmh_messages->icon_y = xsh_messages->y;
		xwmh_messages->flags = InputHint | StateHint | IconPixmapHint | IconPositionHint;
		XSetWMProperties(gra_topmsgdpy, gra_topmsgwin, &swintitle, &sicontitle,
			gra_argv, gra_argc, xsh_messages, xwmh_messages, NULL);
	}

	/* set up to trap window kills */
	gra_wm_delete_window = XInternAtom(gra_topdpy, "WM_DELETE_WINDOW", False);
	gra_wm_protocols = XInternAtom(gra_topdpy, "WM_PROTOCOLS", False);
	XtAddEventHandler(gra_toplevelwidget, NoEventMask, TRUE, gra_destroygraphics, NULL);
	XSetWMProtocols(gra_topdpy, gra_topwin, &gra_wm_delete_window, 1);
	if (gra_messages != 0)
	{
		XtAddEventHandler(gra_msgtoplevelwidget, NoEventMask, TRUE, gra_destroygraphics,
			NULL);
		XSetWMProtocols(gra_topmsgdpy, gra_topmsgwin, &gra_wm_delete_window, 1);
	}

	pixm = XCreateBitmapFromData(gra_dpy, gra_win, (const char *)gra_realcursordata, 16, 16);
	gra_realcursor = XCreatePixmapCursor(gra_dpy, pixm, pixm, &gra_xfc, &gra_xbc, 8, 8);

	pixm = XCreateBitmapFromData(gra_dpy, gra_win, (const char *)gra_nomousecursordata,
		16, 16);
	gra_nomousecursor = XCreatePixmapCursor(gra_dpy, pixm, pixm, &gra_xfc, &gra_xbc, 8, 8);

	pixm = XCreateBitmapFromData(gra_dpy, gra_win, (const char *)gra_drawcursordata, 16, 16);
	gra_drawcursor = XCreatePixmapCursor(gra_dpy, pixm, pixm, &gra_xfc,  &gra_xbc, 0, 16);

	pixm = XCreateBitmapFromData(gra_dpy, gra_win, (const char *)gra_nullcursordata, 16, 16);
	gra_nullcursor = XCreatePixmapCursor(gra_dpy, pixm, pixm, &gra_xfc, &gra_xbc, 8, 8);

	pixm = XCreateBitmapFromData(gra_dpy, gra_win, (const char *)gra_menucursordata, 16, 16);
	gra_menucursor = XCreatePixmapCursor(gra_dpy, pixm, pixm, &gra_xfc, &gra_xbc, 16, 8);

	pixm = XCreateBitmapFromData(gra_dpy, gra_win, (const char *)gra_handcursordata, 16, 16);
	gra_handcursor = XCreatePixmapCursor(gra_dpy, pixm, pixm, &gra_xfc, &gra_xbc, 0, 10);

	/* build graphics contexts */
	gcv.foreground = bd;
	gcv.graphics_exposures = False;     /* no event after XCopyArea SRP */
	gra_gc = XCreateGC(gra_dpy, gra_win, GCForeground | GCGraphicsExposures, &gcv);

	gcv.function = GXinvert;
	gra_gcinv = XCreateGC(gra_dpy, gra_win, GCFunction | GCGraphicsExposures, &gcv);

	gcv.fill_style = FillStippled;
	gra_gcstip = XCreateGC(gra_dpy, gra_win, GCFillStyle | GCGraphicsExposures, &gcv);

	/* get window size and depth after creation */
	XGetWindowAttributes(gra_dpy, gra_win, &xwa);
	us_swid = xwa.width;
	gra_shei = xwa.height;  /* we keep the true height locally*/
	us_shei = gra_shei - gra_status_height;
	gra_offx = 0;
	gra_revy = us_shei-1;
	gcv.foreground = gra_blk;
	gcv.background = gra_wht;
	gcv.graphics_exposures = False;

	if (gra_messages != 0)
	{
		gra_gc_messages = XCreateGC(gra_dpy, gra_msgwin,
			GCForeground | GCForeground | GCGraphicsExposures, &gcv);
		XSetFont(gra_dpy, gra_gc_messages, gra_messages_font.font->fid);
	}

	/* fill the color map  */
	if (gra_sdep != 1)
	{
		if (XAllocColorCells(gra_dpy, gra_colormap, 1, (unsigned long *)planes, 0,
			(unsigned long *)pixels, el_maplength) == 0)
				printf("XAllocColorCells failed\n");
		us_dispstyle = COLORMAP;

		/* load any map the first time to establish the map segment length */
		us_getcolormap(el_curtech, COLORSDEFAULT, 0);
  		el_colcursor = CURSOR;	/* also done in usr.c later */
		gra_recolorcursor(&gra_xfc, &gra_xbc);
	} else
	{
		us_dispstyle = BWRASTER;
	}

	/* initialize the mouse */
	gra_cursorstate = -1;
	gra_set_cursorstate(NULLCURSOR);
	gra_inputstate = NOEVENT;
	gra_ignoremotion = 1;

#ifdef HASSPYGLASS
	/* initialize the spyglass and turn it off */
	gra_spymag = 1;
	gra_spyx = -1;
	gra_xmin = 0;
	gra_xmax = us_swid - 1;
	gra_ymin = 0;
	gra_ymax = us_shei - 1;
#endif

	/* wait for the first exposure event to force window placement */
	if (gra_use_backing == 0)
	{
		/* avoid double repaint at startup */
		while (XCheckTypedWindowEvent(gra_dpy, gra_win, Expose, &event) == False);
	} else
	{
		while (XCheckTypedWindowEvent(gra_dpy, gra_win, MapNotify, &event) == False);
	}

	return(0);
}

Display *gra_finddisplay(void)
{
	Display *cur_dpy;

	/* find the current display */
	if ((cur_dpy = XOpenDisplay(NULL)) == NULL)
		error("Cannot open display %s\n", XDisplayName(NULL));

	/* Map R14 (down arrow) to "2" */
	XRebindKeysym(cur_dpy, XK_Down, (KeySym *)XK_VoidSymbol, 0, (unsigned char *)"\037", 1);

	/* Map R10 (left arrow) to "4" */
	XRebindKeysym(cur_dpy, XK_Left, (KeySym *)XK_VoidSymbol, 0, (unsigned char *)"\034", 1);

	/* Map R12 (right arrow) to "6" */
	XRebindKeysym(cur_dpy, XK_Right, (KeySym *)XK_VoidSymbol, 0, (unsigned char *)"\035", 1);

	/* Map R8 (up arrow) to "8" */
	XRebindKeysym(cur_dpy, XK_Up, (KeySym *)XK_VoidSymbol, 0, (unsigned char *)"\036", 1);

	return(cur_dpy);
}

/*
 * Routine to establish the library directories from the environment.
 */
void setupenvironment(void)
{
	REGISTER char *pt;

	pt = getenv("EEI_LIBDIR");
	if (pt == NULL) pt = LIBDIR;
	(void)allocstring(&el_libdir, pt, db_cluster);

	pt = getenv("EEI_LISPDIR");
	if (pt == NULL) pt = LISPDIR;
	(void)allocstring(&el_lispdir, pt, db_cluster);

#if LANGTCL
	pt = getenv("EEI_TCLDIR");
	if (pt == NULL) pt = TCLDIR;
	(void)allocstring(&el_tcldir, pt, db_cluster);

#  ifdef USETK
	pt = getenv("EEI_TKDIR");
	if (pt == NULL) pt = TKDIR;
	(void)allocstring(&el_tkdir, pt, db_cluster);
#  endif
#endif

	/* set machine name */
	nextvarchangequiet();
	(void)setvalkey((INTBIG)us_aid, VAID, makekey("USER_machine"),
		(INTBIG)"UNIX", VSTRING|VDONTSAVE);
}

/*
 * this routine returns the host ID for this computer in "name"
 */
void getmachineid(char *name)
{
#if 0		/* was #ifdef HAVE_GETHOSTID, but this function is unimportant */
	INTBIG hid;

	hid = gethostid();
	(void)sprintf(name, "%x", hid);
#else
	(void)strcpy(name, "52001ae4"); /* just fake a valid key */
#endif
}

/******************** TERMINATION ********************/

void us_unsetdisplay(void) {}

void us_termgraphics(INTSML final)
{
	/* quit if no graphics being used */
	if (us_dispstyle < 0) return;

	XCloseDisplay(gra_dpy);
}

void exitprogram(void)
{
	exit(0);
}

/******************** WINDOW CONTROL ********************/

/*
 * These routines implement multiple windows on the display (each display window
 * is called a "windowframe".  Currently these routines are unimplemented.
 */

void *us_newwindowframe(INTSML floating)
{
	return(0);
}

void us_killwindowframe(void *frame)
{
}

void *us_getwindowframe(void)
{
	return((void *)1);
}

void *us_firstwindowframe(void)
{
	return((void *)1);
}

void *us_nextwindowframe(void *frame)
{
	return(0);
}

/*
 * routine to return size of window "win" in "wid" and "hei"
 */
void us_getwindowsize(void *frame, INTSML *wid, INTSML *hei)
{
	*wid = us_swid;
	*hei = us_shei;
}

void us_sizewindowframe(void *frame, INTSML wid, INTSML hei)
{
}

void us_movewindowframe(void *frame, INTSML top, INTSML left)
{
}

/*
 * Routine to close the messages window if it is in front.  Returns nonzero if the
 * window was closed.
 */
INTSML us_closefrontmostmessages(void)
{
	return(0);
}

void us_getpaletteparameters(INTSML *wid, INTSML *hei, INTSML *palettewidth)
{
	*hei = us_shei;
	*wid = us_swid;
	*palettewidth = 70;
}

void us_settargetframe(void *frame)
{
}

/******************** EXTERNAL ROUTINES ********************/

/*
 * routine to return display style of window "win"
 */
short us_getdispstyle(WINDOW *win)
{
	return(us_dispstyle);
}

void us_startbatching(void) {}

void us_endbatching(void) {}

/*
 * routine to set the spyglass magnification to "mag", which must be either 1
 * (no magnification, off), 2, 4, or 8.
 */
void us_setspy(INTSML mag)
{
#ifdef HASSPYGLASS
	/* quit if no graphics being used */
	if (us_dispstyle < 0) return;

	if (gra_use_backing)
	{
		/* conflict with backing */
		gra_resetspy();
		ttyputmsg("Magnifier disabled by retained option");
		return;
	}

	/* create the storage area */
	if (gra_spymap == 0)
		gra_spymap = XCreatePixmap(gra_dpy, gra_win, MAXSPYSIZE, MAXSPYSIZE, gra_sdep);

	/* switch magnification off */
	gra_resetspy();

	if (mag >= 8) gra_spymag = 8; else
		if (mag >= 4) gra_spymag = 4; else
			if (mag >= 2) gra_spymag = 2; else
	{
		gra_spymag = 1;
		return;
	}
	gra_spysize = SMALLSIZE*gra_spymag + 2;
	gra_writespy();
#endif
}

#ifdef HASSPYGLASS
void gra_advancespy(void)
{
	REGISTER INTSML newspyx, newspyy;

	/* if spyglass is not on, cannot slide it.  just turn it on */
	if (gra_spyx < 0)
	{
		gra_writespy();
		return;
	}

	/* compute new location of spyglass */
	newspyx = mini(gra_xmax - gra_spysize - 1, maxi(gra_cursorx - gra_spysize/2, gra_xmin));
	newspyy = mini(gra_ymax - gra_spysize - 1, maxi(gra_revy - gra_cursory - gra_spysize/2,
		gra_ymin));

	/* if new location does not overlap old location, sliding cannot be done */
	if ((abs(newspyx - gra_spyx) >= gra_spysize) || (abs(newspyy - gra_spyy) >= gra_spysize))
	{
		gra_resetspy();
		gra_writespy();
		return;
	}

	/* well!  this is simpleminded!  should do the hard code here */
	gra_resetspy();
	gra_writespy();
}

/* restore the old image that was under the spyglass, and set spyx as a flag */
void gra_resetspy(void)
{
	if (gra_spyx >= 0)
	{
		if (us_dispstyle == COLORMAP) XSetPlaneMask(gra_dpy, gra_gc, AllPlanes);
		XCopyArea(gra_dpy, gra_spymap, gra_win, gra_gc, 0, 0, gra_spysize, gra_spysize,
			gra_spyx, gra_spyy);
		gra_spyx = -1;
	}
}

/*
 * Version for X11, Sid Penstone;
 * We will have to check all bounds of the spyglass image because X11 will
 * not handle an image that extends beyond the window.
 */
void gra_writespy(void)
{
	REGISTER INTSML x, y, xout, yout, smallx, smally, i;
	char *ptr1, *ptr2;
	UINTBIG pxl;

	/* initialize */
	gra_spyx = mini(gra_xmax - gra_spysize - 1, maxi(gra_cursorx - gra_spysize/2, gra_xmin));
	gra_spyy = mini(gra_ymax - gra_spysize - 1, maxi(gra_revy - gra_cursory - gra_spysize/2,
		gra_ymin));

	if (us_dispstyle == COLORMAP) XSetPlaneMask(gra_dpy, gra_gc, AllPlanes);

	/* save what is under the spyglass */
	XCopyArea(gra_dpy, gra_win, gra_spymap, gra_gc, gra_spyx, gra_spyy, gra_spysize,
		gra_spysize, 0, 0);

	/* get the area to be magnified */
	smallx = mini(gra_xmax - SMALLSIZE - 1, maxi(gra_xmin + 1, gra_cursorx - SMALLSIZE/2));
	smally = mini(gra_ymax - SMALLSIZE - 1, maxi(gra_ymin + 1,
		gra_revy - gra_cursory - SMALLSIZE/2));
	gra_spyrect = XGetImage(gra_dpy, gra_win, smallx, smally, SMALLSIZE, SMALLSIZE,
		AllPlanes, ZPixmap);

	/* report the error, do not crash */
	if (gra_spyrect == 0 ) return;

	/* and the image to be replaced */
	gra_spyrect2 = XGetImage(gra_dpy, gra_win, gra_spyx + 1, gra_spyy + 1, gra_spysize - 2,
		gra_spysize - 2, AllPlanes, ZPixmap);
	if (gra_spyrect2 == 0) return;

	/* draw the border */
	XSetLineAttributes(gra_dpy, gra_gc, 0, LineSolid, CapButt, JoinMiter);
	XSetForeground(gra_dpy, gra_gc, el_colmenbor);
	XDrawRectangle(gra_dpy, gra_win, gra_gc, gra_spyx, gra_spyy, gra_spysize - 1,
		gra_spysize - 1);

	/* and move magnified pixels into the image area
	 * Hacked by Sid Penstone, Sept, Nov. 1990:
	 * Dissect the image over SMALLSIZE by SMALLSIZE pixels;
	 * There will be gra_spymag times as many output bytes.
	 * Each new scan line is found (bytes_per_line) further into the image data.
	 * There are several ways of doing this. Byte pushing is very fast when
	 * it is an 8 plane system, but it is messier for other formats. In
	 * the interest of portable code, use the X11 pixel routines.
	 * Here is the direct version and the pixel version
	 */

	for (y=0; y<SMALLSIZE; y++)
	{
		ptr2 = gra_spyrect2->data + ( y * gra_spymag * gra_spyrect2->bytes_per_line);

		if (gra_spyrect->bits_per_pixel == 8)
		{
			/* 8 plane system */
			ptr1 = gra_spyrect->data + (y * gra_spyrect->bytes_per_line);
			for ( x = 0; x < SMALLSIZE; x++)
				for (i = 0; i < gra_spymag; i++)
					ptr2[(x * gra_spymag) + i] = ptr1[x];
		} else
		{
			/* any other number of planes */
			yout = y*gra_spymag;
			for (x = 0; x<SMALLSIZE; x++)
			{
				xout = x*gra_spymag;
				pxl = XGetPixel(gra_spyrect, x, y);
				for (i = 0; i < gra_spymag; i++)
					(void)XPutPixel(gra_spyrect2, xout + i, yout, pxl);
			}
		}

		/* now duplicate the line */
		if (gra_spymag > 1)
			bcopy(ptr2, ptr2 + gra_spyrect2->bytes_per_line, gra_spyrect2->bytes_per_line);
		if (gra_spymag > 2)  /* and another 2 lines */
			bcopy(ptr2, ptr2 + 2 * gra_spyrect2->bytes_per_line,
				2 * gra_spyrect2->bytes_per_line);
		if (gra_spymag > 4)  /* double again, for eight in total */
			bcopy(ptr2, ptr2 + 4 * gra_spyrect2->bytes_per_line,
				4 * gra_spyrect2->bytes_per_line);
	} /* end of one scanline */

	XPutImage(gra_dpy, gra_win, gra_gc, gra_spyrect2, 0, 0, gra_spyx + 1, gra_spyy + 1,
		gra_spysize - 2, gra_spysize - 2);
	(void)XDestroyImage(gra_spyrect);
	(void)XDestroyImage(gra_spyrect2);
}
#endif  /* HASSPYGLASS */

/*
 * return nonzero if the capabilities in "want" are present
 */
INTSML us_graphicshas(INTSML want)
{
	/* capability not there if no graphics being used */
	if (us_dispstyle < 0) return(0);
	if ((want & CANUSEFRAMES) != 0) return(0);
	if ((want & CANSHOWPOPUP) != 0) return(0);
#ifndef HASSPYGLASS
	if ((want & CANMAGNIFY) != 0) return(0);
#endif

	return(1);
}

void ttybeep(void)
{
    if (us_aid->aidstate & TERMBEEP)
	{
    	XBell(gra_dpy, 100);
    }
}

/******************** STATUS BAR ROUTINES ********************/

/*
 * Routine to return the number of status lines on the display.
 */
INTSML ttynumstatuslines(void)
{
	return(MAXSTATUSLINES);
}

/*
 * Routine to free status field object "sf".
 */
void ttyfreestatusfield(STATUSFIELD *sf)
{
	efree(sf->label);
	efree((char *)sf);
}

/*
 * Routine to display "message" in the status "field" of window "frame" (uses all windows
 * if "frame" is zero).  If "cancrop" is zero, field cannot be cropped and should be
 * replaced with "*" if there isn't room.
 */
void ttysetstatusfield(void *frame, STATUSFIELD *field, char *message, INTSML cancrop)
{
	INTSML len, lx, ly, hx, hy;
	XFontStruct *oldfont;
	INTSML sizey, sizex;
	char *ptr;

	/* draw the separator line between drawing and status areas */
	us_drawline(NULL, 1, -2, us_swid - 1, -2, &us_box, 0);

	if (field == 0) return;
	if (field->line == 0)
	{
		/* should set title bar here */
		(void)strcpy(gra_localstring, "Electric");
		if (strlen(message) > 0)
		{
			(void)strcat(gra_localstring, " (");
			(void)strcat(gra_localstring, field->label);
			(void)strcat(gra_localstring, message);
			(void)strcat(gra_localstring, ")");
		}
		len = strlen(gra_localstring);
		while (len > 0 && gra_localstring[len-1] == ' ') gra_localstring[--len] = 0;
		ptr = gra_localstring;
		XStringListToTextProperty(&ptr, 1, &wintitle);
		XSetWMName(gra_topdpy, gra_topwin, &wintitle);
		return;
	}

	/* construct the status line */
	(void)strcpy(gra_localstring, field->label);
	(void)strcat(gra_localstring, message);
	len = strlen(gra_localstring);
	while (len > 0 && gra_localstring[len-1] == ' ') gra_localstring[--len] = 0;

	oldfont = gra_curfont;
	us_settextsize(NULL, TXT6P);
	sizey = gra_curfont->ascent + gra_curfont->descent;
	sizex = gra_curfont->max_bounds.rbearing - gra_curfont->max_bounds.lbearing;

	lx = us_swid * field->startper / 100;
	hx = us_swid * field->endper / 100;
	ly = - (gra_status_height / field->line);

	hy = ly + sizey;
	us_drawbox(NULL, lx, hx - 1, ly, hy, &us_ebox);
	us_puttext(el_curwindow, lx, ly, gra_localstring, &us_menutext);

	gra_curfont = oldfont;
}

/******************** MESSAGES WINDOW ROUTINES ********************/

/*
 * internal routine to print one line onto the scrolling area
 */
void gra_putstring(char *line)
{
	XawTextBlock tb;
	char endline[2];
	XWindowAttributes xwa;

	/* bring window to top if not */
	if (XGetWindowAttributes(gra_topmsgdpy, gra_topmsgwin, &xwa) == 0) return;
	if (xwa.map_state != IsViewable || gra_messages_obscured != 0)
		XMapRaised(gra_topmsgdpy, gra_topmsgwin);

	tb.firstPos = 0;
	tb.length = strlen(line);
	tb.ptr = line;
	tb.format = FMT8BIT;
	XawTextReplace(gra_messageswidget, MESSAGEBUFFERSIZE, MESSAGEBUFFERSIZE, &tb);
	endline[0] = '\n';   endline[1] = 0;
	tb.length = 1;
	tb.ptr = endline;
	XawTextReplace(gra_messageswidget, MESSAGEBUFFERSIZE, MESSAGEBUFFERSIZE, &tb);
	XawTextSetInsertionPoint(gra_messageswidget, MESSAGEBUFFERSIZE);
}

char *gra_getstring(char *prompt)
{
	char c;
	int i;
	XawTextPosition posstart, posend;
	Arg arg;
	char *str, ch[3];
	XawTextBlock tb;

	/* show the prompt */
	tb.firstPos = 0;
	tb.length = strlen(prompt);
	tb.ptr = prompt;
	tb.format = FMT8BIT;
	XawTextReplace(gra_messageswidget, MESSAGEBUFFERSIZE, MESSAGEBUFFERSIZE, &tb);

	/* get initial text position */
	XawTextSetInsertionPoint(gra_messageswidget, MESSAGEBUFFERSIZE);
	posstart = XawTextGetInsertionPoint(gra_messageswidget);

	/* loop while typing is done */
	tb.ptr = ch;
	for(;;)
	{
		c = us_getnxtchar();
		if (c == '\n' || c == '\r' || c == CTRLD) break;
		if (c == 0177 || c == 010)
		{
			posend = XawTextGetInsertionPoint(gra_messageswidget);
			if (posend > posstart)
			{
				tb.length = 0;
				XawTextReplace(gra_messageswidget, posend-1, posend, &tb);
				XawTextSetInsertionPoint(gra_messageswidget, MESSAGEBUFFERSIZE);
			}
			continue;
		}
		if (c == KILLCH)
		{
			posend = XawTextGetInsertionPoint(gra_messageswidget);
			if (posend > posstart)
			{
				tb.length = 0;
				XawTextReplace(gra_messageswidget, posstart, posend, &tb);
				XawTextSetInsertionPoint(gra_messageswidget, MESSAGEBUFFERSIZE);
			}
			continue;
		}
		tb.length = 1;
		ch[0] = c;
		XawTextReplace(gra_messageswidget, MESSAGEBUFFERSIZE, MESSAGEBUFFERSIZE, &tb);
		XawTextSetInsertionPoint(gra_messageswidget, MESSAGEBUFFERSIZE);
	}

	/* get current text position */
	XawTextSetInsertionPoint(gra_messageswidget, MESSAGEBUFFERSIZE);
	posend = XawTextGetInsertionPoint(gra_messageswidget);

	/* get typed string */
	XtSetArg(arg, XtNstring, &str);
	XtGetValues(gra_messageswidget, &arg, 1);
	(void)initinfstr();
	for(i=posstart; i<posend; i++) (void)addtoinfstr(str[i]);
	str = returninfstr();

	/* add in final carriage return */
	ch[0] = '\n';
	tb.length = 1;
	XawTextReplace(gra_messageswidget, MESSAGEBUFFERSIZE, MESSAGEBUFFERSIZE, &tb);
	if (*str == 0 && c == 4) return(0);
	return(str);
}

/*
 * routine to cut text from the messages window if it is current.  Returns nonzero
 * if sucessful.
 */
INTSML tty_cut(void)
{
	Window win;
	int revert, length, i;
	XawTextPosition pos1, pos2;
	unsigned char *str;
	Arg arg;
	XawTextBlock tb;

	/* stop now if messages window is not on top */
	XGetInputFocus(gra_topmsgdpy, &win, &revert);
	if (win != gra_topmsgwin) return(0);

	/* store selection in cut buffer */
	XawTextGetSelectionPos(gra_messageswidget, &pos1, &pos2);
	length = pos2 - pos1;
	if (length <= 0) return(0);
	XtSetArg(arg, XtNstring, &str);
	XtGetValues(gra_messageswidget, &arg, 1);
	(void)initinfstr();
	for(i=pos1; i<pos2; i++) (void)addtoinfstr(str[i]);
	tty_setcutbuffer(returninfstr());

	/* remove selected text */
	tb.firstPos = 0;
	tb.length = 0;
	tb.ptr = 0;
	tb.format = FMT8BIT;
	XawTextReplace(gra_messageswidget, pos1, pos2, &tb);

	return(1);
}

/*
 * routine to copy text from the messages window if it is current.  Returns nonzero
 * if sucessful.
 */
INTSML tty_copy(void)
{
	Window win;
	int revert, length, i;
	XawTextPosition pos1, pos2;
	unsigned char *str;
	Arg arg;

	/* stop now if messages window is not on top */
	XGetInputFocus(gra_topmsgdpy, &win, &revert);
	if (win != gra_topmsgwin) return(0);

	/* store selection in cut buffer */
	XawTextGetSelectionPos(gra_messageswidget, &pos1, &pos2);
	length = pos2 - pos1;
	if (length <= 0) return(0);
	XtSetArg(arg, XtNstring, &str);
	XtGetValues(gra_messageswidget, &arg, 1);
	(void)initinfstr();
	for(i=pos1; i<pos2; i++) (void)addtoinfstr(str[i]);
	tty_setcutbuffer(returninfstr());
	return(1);
}

/*
 * routine to paste text to the messages window if it is current.  Returns nonzero
 * if sucessful.
 */
INTSML tty_paste(void)
{
	Window win;
	char *string;
	int revert;
	XawTextBlock tb;
	XawTextPosition pos1, pos2;

	/* stop now if messages window is not on top */
	XGetInputFocus(gra_topmsgdpy, &win, &revert);
	if (win != gra_topmsgwin) return(0);

	/* get cut buffer */
	string = tty_getcutbuffer();

	/* insert cut buffer into messages window */
	tb.firstPos = 0;
	tb.length = strlen(string);
	tb.ptr = string;
	tb.format = FMT8BIT;
	XawTextGetSelectionPos(gra_messageswidget, &pos1, &pos2);
	XawTextReplace(gra_messageswidget, pos1, pos2, &tb);
	XawTextSetInsertionPoint(gra_messageswidget, pos1+tb.length);
	return(1);
}

/*
 * routine to get the contents of the system cut buffer
 */
char *tty_getcutbuffer(void)
{
	int returned;
	char *string;

	/* get cut buffer */
	string = XFetchBuffer(gra_topmsgdpy, &returned, 0);
	(void)initinfstr();
	if (*string != 0) (void)addstringtoinfstr(string);
	XFree(string);
	return(returninfstr());
}

/*
 * routine to set the contents of the system cut buffer to "msg"
 */
void tty_setcutbuffer(char *msg)
{
	XStoreBuffer(gra_topmsgdpy, msg, strlen(msg), 0);
}

/******************** GRAPHICS CONTROL ROUTINES ********************/

void flushscreen(void) {}

/*
 * Routine to change the default cursor (to indicate modes).
 */
INTSML us_setnormalcursor(INTSML curs)
{
	return(curs);
}

void us_loadmap(INTBIG *red, INTBIG *green, INTBIG *blue, INTSML low, INTSML high)
{
#ifndef STATIC_CMAP
	REGISTER INTSML i;
	XColor xc[256];

	if (us_dispstyle != COLORMAP) return;

	/* clip to the number of map entries requested */
	if (high > gra_maphigh) high = gra_maphigh;

	for(i=low; i<=high; i++)
	{
		xc[i-low].flags = DoRed | DoGreen | DoBlue;
		xc[i-low].pixel = i;
		xc[i-low].red   =   red[i-low] << 8;
		xc[i-low].green = green[i-low] << 8;
		xc[i-low].blue  =  blue[i-low] << 8;

		/* always complement the highest color */
		if (low < 255 && i == 255)
		{
			xc[i-low].red   = (255 -   red[i-low]) << 8;
			xc[i-low].green = (255 - green[i-low]) << 8;
			xc[i-low].blue  = (255 -  blue[i-low]) << 8;
		}
	}

	XStoreColors(gra_dpy, gra_colormap, xc, high-low + 1);

	if (low == 0 || (low <= el_colcursor && high >= el_colcursor))
		gra_recolorcursor(&gra_xfc, &gra_xbc);
#endif
}


/*
 * Helper routine to change the cursor color.
 * fg and bg are the XColor structures to be used. We will look up the current
 * values for the cursor and the background from the installed color map, and
 * copy them into bg and fg.
 */
void gra_recolorcursor(XColor *fg, XColor *bg)
{
	fg->pixel = el_colcursor;
	XQueryColor(gra_dpy, gra_colormap, fg);
	bg->pixel = 0;
	XQueryColor(gra_dpy, gra_colormap, bg);
	XRecolorCursor(gra_dpy, gra_realcursor, fg, bg);
	XRecolorCursor(gra_dpy, gra_nomousecursor, fg, bg);
	XRecolorCursor(gra_dpy, gra_drawcursor, fg, bg);
	XRecolorCursor(gra_dpy, gra_nullcursor, fg, bg);
	XRecolorCursor(gra_dpy, gra_menucursor, fg, bg);
	XRecolorCursor(gra_dpy, gra_handcursor, fg, bg);
}

/*
 * Helper routine to set the cursor shape to "state".
 */
void gra_set_cursorstate(INTSML state)
{
	if (gra_cursorstate == state) return;

	switch (state)
	{
		case NORMALCURSOR:
			XDefineCursor(gra_topdpy, gra_topwin, gra_realcursor);
			break;
		case WANTTTYCURSOR:
			XDefineCursor(gra_topdpy, gra_topwin, gra_nomousecursor);
			break;
		case PENCURSOR:
			XDefineCursor(gra_topdpy, gra_topwin, gra_drawcursor);
			break;
		case NULLCURSOR:
			XDefineCursor(gra_topdpy, gra_topwin, gra_nullcursor);
			break;
		case MENUCURSOR:
			XDefineCursor(gra_topdpy, gra_topwin, gra_menucursor);
			break;
		case HANDCURSOR:
			XDefineCursor(gra_topdpy, gra_topwin, gra_handcursor);
			break;
	}

	gra_cursorstate = state;
}

/******************** GRAPHICS LINES ********************/

void us_drawline(WINDOW *win, INTSML x1, INTSML y1, INTSML x2, INTSML y2, GRAPHICS *desc,
	INTSML texture)
{
	static INTSML on[] =  {0, 1, 6, 1};
	static INTSML off[] = {0, 3, 2, 7};
	static INTBIG linestyle[] = {LineSolid, LineOnOffDash, LineOnOffDash, LineOnOffDash};
	char dashes[2];

	/* quit if no graphics being used */
	if (us_dispstyle < 0) return;

#ifdef HASSPYGLASS
	/* turn off magnification if it is on */
	if (gra_spymag != 1) gra_resetspy();
#endif

	/* adjust the co-ordinates for display mapping */
	y1 = gra_revy - y1;
	y2 = gra_revy - y2;
	x1 += gra_offx;
	x2 += gra_offx;

	if (us_dispstyle == COLORMAP)
	{
		XSetPlaneMask(gra_dpy, gra_gc, desc->bits & gra_maphigh);
#ifdef sun
		/*
		 * strange OpenWindows bug requires that the width change in order to
		 * also change the line style
		 */
		XSetLineAttributes(gra_dpy, gra_gc, 1, linestyle[texture], CapButt, JoinMiter);
#endif
		XSetLineAttributes(gra_dpy, gra_gc, 0, linestyle[texture], CapButt, JoinMiter);

		if (texture != 0)
		{
			dashes[0] = on[texture];
			dashes[1] = off[texture];
			XSetDashes(gra_dpy, gra_gc, 0, dashes, 2);
		}

		XSetForeground(gra_dpy, gra_gc, desc->col);
		XDrawLine(gra_dpy, gra_win, gra_gc, x1, y1, x2, y2);
		return;
	}

	/* the rest is for monochrome */
	if ((desc->style[us_dispstyle] & NATURE) == PATTERNED)
	{
		gra_loadstipple(desc->raster);
		XSetForeground(gra_dpy, gra_gcstip, (desc->col) ? gra_fgd : gra_bgd);
		XDrawLine(gra_dpy, gra_win, gra_gcstip, x1, y1, x2, y2);
		return;
	}

	if (desc->bits == LAYERH)
	{
		XSetLineAttributes(gra_dpy, gra_gcinv, 0, linestyle[texture], CapButt, JoinMiter);
		if (texture != 0)
		{
			dashes[0] = on[texture];
			dashes[1] = off[texture];
			XSetDashes(gra_dpy, gra_gcinv, 0, dashes, 2);
		}
		XSetForeground(gra_dpy, gra_gcinv, (desc->col) ? gra_fgd : gra_bgd);
		XDrawLine(gra_dpy, gra_win, gra_gcinv, x1, y1, x2, y2);
		return;
	}

	XSetLineAttributes(gra_dpy, gra_gc, 0, linestyle[texture], CapButt, JoinMiter);
	if (texture != 0)
	{
		dashes[0] = on[texture];
		dashes[1] = off[texture];
		XSetDashes(gra_dpy, gra_gc, 0, dashes, 2);
	}
	XSetForeground(gra_dpy, gra_gc, (desc->col) ? gra_fgd : gra_bgd);
	XDrawLine(gra_dpy, gra_win, gra_gc, x1, y1, x2, y2);
}

void us_invertline(WINDOW *win, INTSML x1, INTSML y1, INTSML x2, INTSML y2)
{
	/* quit if no graphics being used */
	if (us_dispstyle < 0) return;

#ifdef HASSPYGLASS
	/* turn off magnification if it is on */
	if (gra_spymag != 1) gra_resetspy();
#endif

	/* adjust the co-ordinates for display mapping */
	y1 = gra_revy - y1;
	y2 = gra_revy - y2;
	x1 += gra_offx;
	x2 += gra_offx;

	XSetPlaneMask(gra_dpy, gra_gcinv, LAYERH & gra_maphigh);
	XSetLineAttributes(gra_dpy, gra_gcinv, 0, LineSolid, CapButt, JoinMiter);
	XSetForeground(gra_dpy, gra_gcinv, HIGHLIT);
	XDrawLine(gra_dpy, gra_win, gra_gcinv, x1, y1, x2, y2);
	return;
}

/******************** GRAPHICS POLYGONS ********************/

void us_drawpolygon(WINDOW *win, INTBIG *x, INTBIG *y, INTSML count, GRAPHICS *desc)
{
	static XPoint *pointlist;
	static INTSML pointcount = 0;
	REGISTER INTSML i;

	/* quit if no graphics being used */
	if (us_dispstyle < 0) return;

#ifdef HASSPYGLASS
	/* turn off magnification if it is on */
	if (gra_spymag != 1) gra_resetspy();
#endif

	if (count > pointcount)
	{
		if (pointcount != 0) efree((char *)pointlist);
		pointcount = 0;
		pointlist = (XPoint *)emalloc(count * (sizeof (XPoint)), us_aid->cluster);
		if (pointlist == 0) return;
		pointcount = count;
	}

	for(i=0; i<count; i++)
	{
		pointlist[i].x = gra_offx + x[i];
		pointlist[i].y = gra_revy - y[i];
	}

	if (((desc->style[us_dispstyle] & NATURE) == PATTERNED) && (desc->col != 0))
	{
		gra_loadstipple(desc->raster);
		if (us_dispstyle == COLORMAP)
		{
			XSetPlaneMask(gra_dpy, gra_gcstip, desc->bits & gra_maphigh);
			XSetForeground(gra_dpy, gra_gcstip, desc->col);
		} else
		{
			XSetForeground(gra_dpy, gra_gcstip, (desc->col) ? gra_fgd : gra_bgd);
		}
		XFillPolygon(gra_dpy, gra_win, gra_gcstip, pointlist, count, Complex,
			CoordModeOrigin);
	} else
	{
		if (us_dispstyle == COLORMAP)
		{
			XSetPlaneMask(gra_dpy, gra_gc, desc->bits & gra_maphigh);
			XSetForeground(gra_dpy, gra_gc, desc->col);
		} else
		{
			XSetForeground(gra_dpy, gra_gc, (desc->col) ? gra_fgd : gra_bgd);
		}
		XFillPolygon(gra_dpy, gra_win, gra_gc, pointlist, count, Complex, CoordModeOrigin);
	}
}

/******************** GRAPHICS BOXES ********************/

void us_drawbox(WINDOW *win, INTSML lowx, INTSML highx, INTSML lowy, INTSML highy,
	GRAPHICS *desc)
{
	/* quit if no graphics being used */
	if (us_dispstyle < 0) return;

#ifdef HASSPYGLASS
	/* turn off magnification if it is on */
	if (gra_spymag != 1) gra_resetspy();
#endif

	 /* special code for patterned writes */
	if (((desc->style[us_dispstyle] & NATURE) == PATTERNED) && (desc->col != 0))
	{
		gra_loadstipple(desc->raster);
		if (us_dispstyle == COLORMAP)
		{
			XSetPlaneMask(gra_dpy, gra_gcstip, desc->bits & gra_maphigh);
			XSetForeground(gra_dpy, gra_gcstip, desc->col);
		} else
		{
			XSetForeground(gra_dpy, gra_gcstip, (desc->col) ? gra_fgd : gra_bgd);
		}
		XFillRectangle(gra_dpy, gra_win, gra_gcstip, gra_offx + lowx, gra_revy - highy,
			highx - lowx + 1, highy-lowy + 1);
#ifdef sun
		/*
		 * there seems to be a bug in OpenWindows: the right edge of stipple
		 * patterns goes haywire if the pattern is more than 32 pixels wide
		 */
		if (highx - lowx + 1 > 32)
		{
			XFillRectangle(gra_dpy, gra_win, gra_gcstip, gra_offx + highx - 31,
				gra_revy - highy, 32, highy - lowy + 1);
		}
#endif
	} else
	{
		if (us_dispstyle == COLORMAP)
		{
			XSetPlaneMask(gra_dpy, gra_gc, desc->bits & gra_maphigh);
			XSetForeground(gra_dpy, gra_gc, desc->col);
		} else
		{
			XSetForeground(gra_dpy, gra_gc, (desc->col) ? gra_fgd : gra_bgd);
		}
		XFillRectangle(gra_dpy, gra_win, gra_gc, gra_offx + lowx, gra_revy - highy,
			highx - lowx + 1, highy - lowy + 1);
	}
}

/*
 * routine to invert the bits in the box from (lowx, lowy) to (highx, highy)
 */
void us_invertbox(WINDOW *win, INTSML lowx, INTSML highx, INTSML lowy, INTSML highy)
{
	/* quit if no graphics being used */
	if (us_dispstyle < 0) return;

#ifdef HASSPYGLASS
	/* turn off magnification if it is on */
	if (gra_spymag != 1) gra_resetspy();
#endif

	XSetPlaneMask(gra_dpy, gra_gcinv, AllPlanes);
	XFillRectangle(gra_dpy, gra_win, gra_gcinv, gra_offx + lowx, gra_revy - highy,
		highx - lowx + 1, highy - lowy + 1);
}

/*
 * routine to move bits on the display starting with the area at
 * (sx,sy) and ending at (dx,dy).  The size of the area to be
 * moved is "wid" by "hei".
 */
void us_movebox(WINDOW *win, INTSML sx, INTSML sy, INTSML wid, INTSML hei,
	INTSML dx, INTSML dy)
{
	/* quit if no graphics being used */
	if (us_dispstyle < 0) return;

#ifdef HASSPYGLASS
	/* turn off magnification if it is on */
	if (gra_spymag != 1) gra_resetspy();
#endif

	/* make sure all bit planes are copied */
	if (us_dispstyle == COLORMAP) XSetPlaneMask(gra_dpy, gra_gc, AllPlanes);

	XCopyArea(gra_dpy, gra_win, gra_win, gra_gc, gra_offx + sx, gra_revy + 1 - sy - hei,
			  wid, hei, gra_offx + dx, gra_revy + 1 - dy - hei);
}

/*
 * routine to save the contents of the box from "lx" to "hx" in X and from
 * "ly" to "hy" in Y.  A code is returned that identifies this box for
 * overwriting and restoring.  The routine returns negative if there is a error.
 */
INTBIG us_savebox(WINDOW *win, INTSML lx, INTSML hx, INTSML ly, INTSML hy)
{
	SAVEDBOX *box;
	REGISTER INTSML i;

	/* quit if no graphics being used */
	if (us_dispstyle < 0) return(-1);

#ifdef HASSPYGLASS
	/* turn off magnification if it is on */
	if (gra_spymag != 1) gra_resetspy();
#endif

	lx += gra_offx;
	hx += gra_offx;
	i = ly;
	ly = gra_revy - hy;
	hy = gra_revy - i;

	box = (SAVEDBOX *)emalloc((sizeof (SAVEDBOX)), us_aid->cluster);
	if (box == 0) return(-1);

	box->pix = XCreatePixmap(gra_dpy, gra_win, hx - lx + 1, hy - ly + 1, gra_sdep);
	if (box->pix == 0) return(-1);

	box->nextsavedbox = gra_firstsavedbox;
	gra_firstsavedbox = box;
	gra_savedboxcodes++;
	box->code = gra_savedboxcodes;
	box->lx = lx;
	box->hx = hx;
	box->ly = ly;
	box->hy = hy;

	/* make sure the whole buffer is accessable */
	if (us_dispstyle == COLORMAP) XSetPlaneMask(gra_dpy, gra_gc, AllPlanes);

	XCopyArea(gra_dpy, gra_win, box->pix, gra_gc, lx, ly, hx - lx + 1, hy - ly + 1, 0, 0);

	return(box->code);
}

/*
 * routine to shift the saved box "code" so that it is restored in a different
 * lcoation, offset by (dx,dy)
 */
void us_movesavedbox(INTBIG code, INTSML dx, INTSML dy)
{
	REGISTER SAVEDBOX *box;

	for(box = gra_firstsavedbox; box != NOSAVEDBOX; box = box->nextsavedbox)
		if (box->code == code) break;

	if (box == NOSAVEDBOX) return;

	box->lx += dx;       box->hx += dx;
	box->ly -= dy;       box->hy -= dy;
}

/*
 * routine to restore saved box "code" to the screen.  "destroy" is:
 *  0   restore box, do not free memory
 *  1   restore box, free memory
 * -1   free memory
 * Returns nonzero if there is an error.
 */
INTSML us_restorebox(INTBIG code, INTSML destroy)
{
	REGISTER SAVEDBOX *box, *lbox;

	/* quit if no graphics being used */
	if (us_dispstyle < 0) return(1);

	/* find the box corresponding to the "code" */
	lbox = NOSAVEDBOX;
	for(box = gra_firstsavedbox; box != NOSAVEDBOX; box = box->nextsavedbox)
	{
		if (box->code == code) break;
		lbox = box;
	}
	if (box == NOSAVEDBOX) return(1);

	if (destroy >= 0)
	{
#ifdef HASSPYGLASS
		/* turn off magnification if it is on */
		if (gra_spymag != 1) gra_resetspy();
#endif
		/* make sure the whole buffer is accessable */
		if (us_dispstyle == COLORMAP) XSetPlaneMask(gra_dpy, gra_gc, AllPlanes);

		/* restore the bits in the box */
		XCopyArea(gra_dpy, box->pix, gra_win, gra_gc, 0, 0, box->hx - box->lx + 1,
			box->hy - box->ly + 1, box->lx, box->ly);
	}

	/* destroy this box's memory if requested */
	if (destroy != 0)
	{
		if (lbox == NOSAVEDBOX)
		{
			gra_firstsavedbox = box->nextsavedbox;
		} else
		{
			lbox->nextsavedbox = box->nextsavedbox;
		}
		XFreePixmap(gra_dpy, box->pix);
		efree((char *)box);
	}
	return(0);
}

/******************** GRAPHICS TEXT ********************/

void us_settextsize(WINDOW *win, INTSML fnt)
{
	int i;

	if (us_dispstyle < 0) return;

	if (fnt > TXT20P)
	{
		switch (fnt)
		{
			case TXTSMALL:  i = 1;   break;
			case TXTMEDIUM: i = 4;   break;
			case TXTLARGE:  i = 7;   break;
			case TXTMENU:   i = 9;   break;
			case TXTEDITOR: i = 10;  break;
			case TXTSTATUS: i = 11;  break;
			default:        i = 0;   break;
		}
	} else
	{
		i = fnt;
	}
	gra_curfont = gra_font[i].font;
}

void us_textsize(WINDOW *win, char *str, INTSML *x, INTSML *y)
{
	INTBIG direction, asc, desc, len;
	XCharStruct xcs;

	if (us_dispstyle < 0) return;

	len = strlen(str);
	XTextExtents(gra_curfont, str, len, &direction, &asc, &desc, &xcs);
	*x = xcs.width;
	*y = gra_curfont->ascent + gra_curfont->descent;
}

void us_puttext(WINDOW *win, INTSML atx, INTSML aty, char *s, GRAPHICS *desc)
{
	INTSML i, l;

	/* quit if no graphics being used */
	if (us_dispstyle < 0) return;

#ifdef HASSPYGLASS
	/* turn off magnification if it is on */
	if (gra_spymag != 1) gra_resetspy();
#endif

	l = strlen(s);
	for(i=0; i<l; i++)
	{
		gra_localstring[i] = s[i] & 0177;
		if (gra_localstring[i] < ' ') gra_localstring[i] = ' ';
	}

	/* we have to wait for the window to be visible */

	/* special case for highlight layer: always complement the data */
	if (desc->bits == LAYERH)
	{
		XSetPlaneMask(gra_dpy, gra_gc, AllPlanes);
		XSetFont(gra_dpy, gra_gcinv, gra_curfont->fid);
		XDrawString(gra_dpy, gra_win, gra_gcinv, gra_offx + atx,
			gra_revy - aty - gra_curfont->descent, gra_localstring, l);
		return;
	}

	if (us_dispstyle == COLORMAP)
	{
		XSetPlaneMask(gra_dpy, gra_gc, desc->bits & gra_maphigh);
		XSetForeground(gra_dpy, gra_gc, desc->col);
	} else
	{
		XSetForeground(gra_dpy, gra_gc, (desc->col) ? gra_fgd : gra_bgd);
	}
	XSetFont(gra_dpy, gra_gc, gra_curfont->fid);
	XDrawString(gra_dpy, gra_win, gra_gc, gra_offx + atx,
		gra_revy - aty - gra_curfont->descent, gra_localstring, l);
}

/******************** CIRCLES ********************/

void us_drawcircle(WINDOW *win, INTBIG atx, INTBIG aty, INTBIG radius, GRAPHICS *desc)
{
	/* quit if no graphics being used */
	if (us_dispstyle < 0) return;

#ifdef HASSPYGLASS
	/* turn off magnification if it is on */
	if (gra_spymag != 1) gra_resetspy();
#endif

	if (us_dispstyle == COLORMAP)
	{
		XSetPlaneMask(gra_dpy, gra_gc, desc->bits & gra_maphigh);
		XSetForeground(gra_dpy, gra_gc, desc->col);
	} else
	{
		XSetForeground(gra_dpy, gra_gc, (desc->col) ? gra_fgd : gra_bgd);
	}
	XDrawArc(gra_dpy, gra_win, gra_gc, gra_offx + atx - radius, gra_revy - aty - radius,
			 radius*2, radius*2, 0, 360*64);
}

/*
 * routine to draw a filled-in circle of radius "radius"
 */
void us_drawdisc(WINDOW *win, INTBIG atx, INTBIG aty, INTBIG radius, GRAPHICS *desc)
{
	/* quit if no graphics being used */
	if (us_dispstyle < 0) return;

#ifdef HASSPYGLASS
	/* turn off magnification if it is on */
	if (gra_spymag != 1) gra_resetspy();
#endif

	if (((desc->style[us_dispstyle] & NATURE) == PATTERNED) && (desc->col != 0))
	{
		gra_loadstipple(desc->raster);
		if (us_dispstyle == COLORMAP)
		{
			XSetPlaneMask(gra_dpy, gra_gcstip, desc->bits & gra_maphigh);
			XSetForeground(gra_dpy, gra_gcstip, desc->col);
		} else
		{
			XSetForeground(gra_dpy, gra_gcstip, (desc->col) ? gra_fgd : gra_bgd);
		}
		XFillArc(gra_dpy, gra_win, gra_gcstip, gra_offx + atx - radius,
			gra_revy - aty - radius, radius*2, radius*2, 0, 360*64);
	} else
	{
		if (us_dispstyle == COLORMAP)
		{
			XSetPlaneMask(gra_dpy, gra_gc, desc->bits & gra_maphigh);
			XSetForeground(gra_dpy, gra_gc, desc->col);
		} else
		{
			XSetForeground(gra_dpy, gra_gc, (desc->col) ? gra_fgd : gra_bgd);
		}
		XFillArc(gra_dpy, gra_win, gra_gc, gra_offx + atx - radius, gra_revy - aty - radius,
			radius*2, radius*2, 0, 360*64);
	}
}

/******************** ARC DRAWING ********************/

/*
 * draws an arc centered at (centerx, centery), clockwise, passing by (x1,y1) and (x2,y2)
 */
void us_drawcirclearc(WINDOW *win, INTBIG centerx, INTBIG centery, INTBIG x1, INTBIG y1,
	INTBIG x2, INTBIG y2, GRAPHICS *desc)
{
	REGISTER INTSML radius, startangle, endangle;

	/* quit if no graphics being used */
	if (us_dispstyle < 0) return;

#ifdef HASSPYGLASS
	/* turn off magnification if it is on */
	if (gra_spymag != 1) gra_resetspy();
#endif

	if (us_dispstyle == COLORMAP)
	{
		XSetPlaneMask(gra_dpy, gra_gc, desc->bits & gra_maphigh);
		XSetForeground(gra_dpy, gra_gc, desc->col);
	} else
	{
		XSetForeground(gra_dpy, gra_gc, (desc->col) ? gra_fgd : gra_bgd);
	}

	radius = computedistance(gra_offx + centerx, gra_revy - centery, gra_offx + x1,
		gra_revy - y1);
	startangle = figureangle(centerx, centery, x1, y1);
	endangle = figureangle(centerx, centery, x2, y2);
	if (startangle < endangle) startangle += 3600;
	XDrawArc(gra_dpy, gra_win, gra_gc, gra_offx + centerx - radius,
		gra_revy - centery - radius, radius*2, radius*2, (endangle*64 + 5) / 10,
		((startangle - endangle)*64 + 5) / 10);
}

/******************** GRID CONTROL ********************/

/*
 * fast grid drawing routine
 */
void us_drawgrid(WINDOW *win, POLYGON *obj)
{
	REGISTER INTBIG i, gridsize;
	REGISTER INTSML x, y;
	static Pixmap grid_pix;
	static GC grid_gc;
	XGCValues gcv;
	static INTBIG bestsize = 0;

	/* quit if no graphics being used */
	if (us_dispstyle < 0) return;

#ifdef HASSPYGLASS
	/* turn off magnification if it is on */
	if (gra_spymag != 1) gra_resetspy();
#endif

	/* grid pixrect */
	gridsize = obj->xv[3] - obj->xv[2];

	/* for some reason, must always reallocate SEEMS OK NOW: SRP */
	if (us_swid > bestsize)
	{
		if (bestsize != 0)
		{
			XFreePixmap(gra_dpy, grid_pix);
			XFreeGC(gra_dpy, grid_gc);
		}
		grid_pix = XCreatePixmap(gra_dpy, gra_win, bestsize = us_swid, 1, 1);
		gcv.background = 0;
		gcv.foreground = 0;
		grid_gc = XCreateGC(gra_dpy, grid_pix, GCBackground | GCForeground, &gcv);
	}

	XSetForeground(gra_dpy, grid_gc, 0);
	XFillRectangle(gra_dpy, grid_pix, grid_gc, 0, 0, bestsize, 1);

	/* calculate the horizontal dots position */
	XSetForeground(gra_dpy, grid_gc, 1);
	for(i = obj->xv[1]; i < obj->xv[5]; i += obj->xv[0])
	{
		x = muldiv(i - obj->xv[4], gridsize, obj->xv[5] - obj->xv[4]) + obj->xv[2];
		XDrawPoint(gra_dpy, grid_pix, grid_gc, x, 0);
	}

	/* draw the vertical columns */
	XSetStipple(gra_dpy, gra_gcstip, grid_pix);
	if (us_dispstyle == COLORMAP)
	{
		XSetPlaneMask(gra_dpy, gra_gcstip, obj->desc->bits & gra_maphigh);
		XSetForeground(gra_dpy, gra_gcstip, obj->desc->col);
	} else
	{
		XSetForeground(gra_dpy, gra_gcstip, (obj->desc->col) ? gra_fgd : gra_bgd);
	}

	for(i = obj->yv[1]; i < obj->yv[5]; i += obj->yv[0])
	{
		y = muldiv(i - obj->yv[4], obj->yv[3] - obj->yv[2], obj->yv[5] - obj->yv[4]) +
			obj->yv[2];
		XFillRectangle(gra_dpy, gra_win, gra_gcstip, gra_offx + obj->xv[2],
			gra_revy - y, gridsize, 1);
	}
}

/******************** MOUSE CONTROL ********************/

/*
 * routine to return the number of buttons on the mouse
 */
INTSML buttoncount(void)
{
	if (us_dispstyle < 0) return(0);
	return(mini(BUTTONS, NUMBUTS));
}

/*
 * routine to tell whether button "but" is a double-click
 */
INTSML doublebutton(INTSML b)
{
	if (b >= BUTTONS - REALBUTS) return(1);
	return(0);
}

/*
 * routine to tell whether button "but" has the "shift" key held
 */
INTSML shiftbutton(INTSML b)
{
	REGISTER INTSML index;

	index = b / REALBUTS;
	if (index == 1 || index == 4 || index == 5 || index == 7) return(1);
	return(0);
}

/*
 * routine to return the name of button "b" (from 0 to "buttoncount()").
 * The number of letters unique to the button is placed in "important".
 */
char *buttonname(INTSML b, INTSML *important)
{
	*important = gra_buttonname[b].unique;
	return(gra_buttonname[b].name);
}

/*
 * routine to convert from "gra_inputstate" (the typical input parameter)
 * to button numbers (the table "gra_buttonname")
 */
INTSML gra_makebutton(INTBIG state)
{
	REGISTER INTSML base;

	base = 0;
	if ((state & DOUBLECL) != 0)
	{
		base += REALBUTS*8;
	} else
	{
		if ((state & SHIFTDOWN) != 0)
		{
			/* shift down */
			if ((state & CONTROLDOWN) != 0)
			{
				/* shift and control down */
				if ((state & METADOWN) != 0) base += REALBUTS*7; else
					base += REALBUTS*4;
			} else
			{
				/* shift down, control up */
				if ((state & METADOWN) != 0) base += REALBUTS*5; else
					base += REALBUTS;
			}
		} else
		{
			/* shift up */
			if ((state & CONTROLDOWN) != 0)
			{
				/* shift up, control down */
				if ((state & METADOWN) != 0) base += REALBUTS*6; else
					base += REALBUTS*2;
			} else
			{
				/* shift and control up */
				if ((state & METADOWN) != 0) base += REALBUTS*3;
			}
		}
	}

	switch (state & ISBUTTON)
	{
		case ISLEFT:                break;
		case ISMIDDLE: base++;      break;
		case ISRIGHT:  base += 2;   break;
	}
	return(base);
}

/*
 * routine to wait for a button push and return its index (0 based) in "*but".
 * The coordinates of the cursor are placed in "*x" and "*y".  If there is no
 * button push, the value of "*but" is negative.
 */
void waitforbutton(INTSML *x, INTSML *y, INTSML *but)
{
	/* quit if no graphics being used */
	if (us_dispstyle < 0)
	{
		*but = -1;
		return;
	}

	/* if there is a button push pending, return it */
	if ((gra_inputstate != NOEVENT) && ((gra_inputstate & ISBUTTON) != 0) &&
		((gra_inputstate & BUTTONUP) == 0))
	{
		*but = gra_makebutton(gra_inputstate);
		*x = gra_cursorx;
		*y = gra_cursory;
		gra_inputstate = NOEVENT;
		gra_set_cursorstate(NULLCURSOR);
		return;
	}

	gra_set_cursorstate(NORMALCURSOR);

	/* get some input (keyboard or mouse) */
	gra_waitforaction();

	/* if a button push was read, return it */
	if ((gra_inputstate != NOEVENT) && ((gra_inputstate & ISBUTTON) != 0) &&
		((gra_inputstate & BUTTONUP) == 0))
	{
		*but = gra_makebutton(gra_inputstate);
		*x = gra_cursorx;
		*y = gra_cursory;
		gra_inputstate = NOEVENT;
		gra_set_cursorstate(NULLCURSOR);
		return;
	}

	/* no button input yet */
	*but = -1;
}

/*
 * routine to track the cursor until a button is released, calling "whileup" for
 * each co-ordinate when the mouse moves before the first button push, calling
 * "whendown" once when the button goes down, calling "eachdown" for each
 * co-ordinate when the mouse moves after the button is pushed, calling
 * "eachchar" for each key that is typed at any time, and calling "done" once
 * when done.  The "whendown" and "done" routines are called with no parameters;
 * "whileup" and "eachdown" are called with the X and Y coordinates of the
 * cursor; and "eachchar" is called with the X, Y, and character value that was
 * typed.  The "whileup", "eachdown", and "eachchar" routines return nonzero to
 * abort tracking.
 * If "waitforpush" is nonzero then the routine will wait for a button to
 * actually be pushed before tracking (otherwise it will begin tracking
 * immediately).  The value of "purpose" determines what the cursor will look
 * like during dragging: 0 for normal (the X cursor), 1 for drawing (a pen),
 * 2 for dragging (a hand), 3 for popup menu selection (a horizontal arrow), 4 for
 * hierarchical popup menu selection (arrow, stays at end).
 */
void trackcursor(INTSML waitforpush, INTSML (*whileup)(INTBIG, INTBIG),
	void (*whendown)(void), INTSML (*eachdown)(INTBIG, INTBIG),
	INTSML (*eachchar)(INTBIG, INTBIG, INTSML), void (*done)(void), INTSML purpose)
{
	REGISTER INTSML keepon, chr;

	/* quit if no graphics being used */
	if (us_dispstyle < 0) return;

	/* change the cursor to an appropriate icon */
	switch (purpose)
	{
		case TRACKDRAWING:
			gra_set_cursorstate(PENCURSOR);
			break;

		case TRACKDRAGGING:
			gra_set_cursorstate(HANDCURSOR);
			break;

		case TRACKSELECTING:
		case TRACKHSELECTING:
			gra_set_cursorstate(MENUCURSOR);
			break;
	}

	/* now wait for a button to go down, if requested */
	keepon = 0;
	if (waitforpush != 0)
	{
		while (keepon == 0)
		{
			gra_ignoremotion = 0;
			gra_waitforaction();
			if (gra_inputstate == NOEVENT) continue;

			/* if button just went down, stop this loop */
			if (((gra_inputstate & ISBUTTON) != 0) && ((gra_inputstate & BUTTONUP) == 0))
				break;

			if ((gra_inputstate & MOTION) != 0)
			{
				keepon = (*whileup)(gra_cursorx, gra_cursory);
			} else if ((gra_inputstate & ISCHAR) != 0)
			{
				chr = gra_inputstate & CHARREAD;
				if ((gra_inputstate & METADOWN) != 0) chr |= 0200;
				keepon = (*eachchar)(gra_cursorx, gra_cursory, chr);
			}
		}
	}

	/* button is now down, real tracking begins */
	if (keepon == 0)
	{
		(*whendown)();
		keepon = (*eachdown)(gra_cursorx, gra_cursory);
	}

	/* now track while the button is down */
	while (keepon == 0)
	{
		gra_ignoremotion = 0;
		gra_waitforaction();

		/* for each motion, report the coordinates */
		if (gra_inputstate == NOEVENT) continue;
		if (((gra_inputstate & ISBUTTON) != 0) && ((gra_inputstate & BUTTONUP) != 0)) break;

		if ((gra_inputstate & MOTION) != 0)
		{
			keepon = (*eachdown)(gra_cursorx, gra_cursory);
		} else if ((gra_inputstate & ISCHAR) != 0)
		{
			chr = gra_inputstate & CHARREAD;
			if ((gra_inputstate & METADOWN) != 0) chr |= 0200;
			keepon = (*eachchar)(gra_cursorx, gra_cursory, chr);
		}
	}
	gra_inputstate = NOEVENT;

	/* inform the user that all is done */
	(*done)();

	/* restore the state of the world */
	if (purpose != TRACKHSELECTING) gra_set_cursorstate(NULLCURSOR);

	/* no longer accept motion events */
	gra_ignoremotion = 1;
}

/*
 * routine to read the current co-ordinates of the tablet and return them in "*x" and "*y"
 */
void readtablet(INTSML *x, INTSML *y)
{
	*x = gra_cursorx;
	*y = gra_cursory;
}

/*
 * routine to turn off the cursor tracking if it is on
 */
void stoptablet(void)
{
	/* quit if no graphics being used */
	if (us_dispstyle < 0) return;

	gra_set_cursorstate(NULLCURSOR);
}

/******************** KEYBOARD CONTROL ********************/

/*
 * routine to get the next character from the keyboard
 */
INTSML us_getnxtchar(void)
{
	REGISTER INTSML i;

	/* quit if no graphics being used */
	if (us_dispstyle < 0) return(getchar());

	if ((gra_inputstate != NOEVENT) && ((gra_inputstate & ISCHAR) == ISCHAR))
	{
		i = gra_inputstate & CHARREAD;
		if ((gra_inputstate & METADOWN) != 0) i |= 0200;
		gra_inputstate = NOEVENT;
		return(i);
	}
	gra_set_cursorstate(WANTTTYCURSOR);

	for(;;)
	{
		gra_waitforaction();
		if ((gra_inputstate != NOEVENT) && ((gra_inputstate & ISCHAR) == ISCHAR)) break;
	}

	i = gra_inputstate & CHARREAD;
	if ((gra_inputstate & METADOWN) != 0) i |= 0200;

	gra_inputstate = NOEVENT;
	gra_set_cursorstate(NULLCURSOR);

	return(i);
}

/* not required with X11 */
void checkforinterrupt(void) {}

/*
 * routine to tell whether data is waiting at the terminal.  Returns nonzero
 * if data is ready.
 */
INTSML ttydataready(void)
{
	/* quit if no graphics being used */
	if (us_dispstyle < 0) return(1);

	/* see if something is already pending */
	if (gra_inputstate != NOEVENT)
	{
		if ((gra_inputstate & ISCHAR) == ISCHAR) return(1);
		return(0);
	}

	/* wait for something and analyze it */
	gra_set_cursorstate(NORMALCURSOR);
	gra_waitforaction();
	if ((gra_inputstate != NOEVENT) && ((gra_inputstate & ISCHAR) == ISCHAR)) return(1);
	return(0);
}

/*
 * routine to set the proper keyboard mode for input.  Returns nonzero upon error.
 */
INTSML gra_onttyinput(void)
{
	return(0);
}

/*
 * routine to turn off the keyboard mode for input (back to normal)
 */
void gra_offttyinput(void) {}

/****************************** FILES ******************************/

char tty_requiredextension[20];

INTSML tty_topoffile(char **a)
{
	char *b, **c;
	INTSML retval;

	b = tty_curpath;
	c = &b;
	retval = topoffile(c);
	requiredextension(tty_requiredextension);
	return(retval);
}

/*
 * Routine to display a standard file prompt dialog and return the selected file.
 * The prompt message is in "msg" and the kind of file is in "filetype".  The default
 * output file name is in "defofile" (only used if "filetype" is for output files).
 */
char *gra_fileselect(char *msg, INTSML filetype, char *defofile)
{
	INTSML itemHit, l, oldlen;
	char *ths, sep[2], *extension, *filter, *shortname, *longname;
	static INTSML curlinevalid = 0;
	static INTSML properpathlen;

	if ((filetype&FILETYPEWRITE) == 0)
	{
		/* input file selection: display the file input dialog box */
		describefiletype(filetype, &extension, &filter, &shortname, &longname);
		strcpy(tty_requiredextension, extension);
		if (DiaInitDialog(&usr_fileindialog) != 0) return(0);
		sep[0] = DIRSEP;   sep[1] = 0;
		if (curlinevalid == 0)
		{
			(void)strcpy(tty_curpath, currentdirectory());
			if (tty_curpath[strlen(tty_curpath)-1] != DIRSEP)
				(void)strcat(tty_curpath, sep);
			curlinevalid = 1;
			properpathlen = strlen(tty_curpath);
		} else tty_curpath[properpathlen] = 0;
		DiaInitTextDialog(3, tty_topoffile, nextfile, tty_nulldlogdone, 0,
			SCSELKEY|SCSELMOUSE|SCDOUBLEQUIT);
		DiaSetText(4, msg);
		DiaSetText(5, tty_curpath);

		/* loop until done */
		for(;;)
		{
			itemHit = DiaNextHit();
			if (itemHit == CANCEL) break;
			if (itemHit == OK)
			{
				oldlen = strlen(tty_curpath);
				ths = DiaGetScrollLine(3, DiaGetCurLine(3));
				(void)strcat(tty_curpath, ths);
				if (fileexistence(tty_curpath) != 2) break;
				properpathlen = strlen(tty_curpath);
				DiaSetText(5, tty_curpath);
				DiaLoadTextDialog(3, tty_topoffile, nextfile, tty_nulldlogdone, 0);
				continue;
			}
			if (itemHit == 6)
			{
				l = strlen(tty_curpath);
				if (l > 0 && tty_curpath[l-1] == DIRSEP) l--;
				if (l <= 0) continue;
				for(l--; l>=0; l--) if (tty_curpath[l] == DIRSEP) break;
				tty_curpath[l+1] = 0;
				properpathlen = strlen(tty_curpath);
				DiaSetText(5, tty_curpath);
				DiaLoadTextDialog(3, tty_topoffile, nextfile, tty_nulldlogdone, 0);
				continue;
			}
		}

		gra_set_cursorstate(3);		/* the hourglass cursor */
		DiaDoneDialog();
		if (itemHit == CANCEL)
		{
			el_pleasestop = 2;
			return((char *)NULL);
		}
		return(tty_curpath);
	} else
	{
		/* output file selection: display the file output dialog box */
		if (DiaInitDialog(&usr_fileoutdialog) != 0) return(0);
		DiaSetText(3, msg);
		if (defofile[0] != 0) DiaSetText(-4, defofile);

		/* loop until done */
		for(;;)
		{
			itemHit = DiaNextHit();
			if (itemHit == CANCEL) break;
			if (itemHit == OK && DiaValidEntry(4)) break;
		}

		gra_set_cursorstate(3);		/* the hourglass cursor */
		strcpy(tty_curpath, DiaGetText(4));
		DiaDoneDialog();

		if (itemHit == CANCEL)
		{
			tty_curpath[0] = 0;
			el_pleasestop = 2;
			return((char *)NULL);
		}
		return(tty_curpath);
	}
}

/*
 * Helper routine to initialize the list of files in a directory.
 */
void gra_initfilelist(void)
{
	if (gra_fileliststringarray == 0)
	{
		gra_fileliststringarray = newstringarray(db_cluster);
		if (gra_fileliststringarray == 0) return;
	}
	clearstrings(gra_fileliststringarray);
}

/*
 * Helper routine to add "file" to the list of files in a directory.
 * Returns nonzero on error.
 */
INTSML gra_addfiletolist(char *file)
{
	addtostringarray(gra_fileliststringarray, file);
	return(0);
}

INTBIG gra_fileselectall(struct dirent *a)
{
	REGISTER char *pt;

	pt = a->d_name;
	if (strcmp(pt, "..") == 0) return(0);
	if (strcmp(pt, ".") == 0) return(0);
	return(1);
}

/*
 * Routine to search for all of the files/directories in directory "directory" and
 * return them in the array of strings "filelist".  Returns the number of files found.
 */
INTSML filesindirectory(char *directory, char ***filelist)
{
	INTBIG len;
	char localname[256];
	struct dirent **filenamelist;
	struct dirent *dir;
	INTSML total, i;
#ifndef HAVE_SCANDIR
#  define DIRFILECOUNT 2048  /* max number of entries in a directory */
#  ifndef _POSIX_PATH_MAX
#    define _POSIX_PATH_MAX 255
#  endif
	DIR *dirp;
	register int tdirsize;
#endif

	/* make sure there is no ending "/" */
	strcpy(localname, directory);
	i = strlen(localname) - 1;
	if (i > 0 && localname[i] == '/') localname[i] = 0;
	if (i == -1) sprintf(localname, ".");

	/* scan the directory */
#ifdef HAVE_SCANDIR
	total = scandir(localname, &filenamelist, gra_fileselectall,
		(int (*)(const __ptr_t, const __ptr_t))0);
#else
	/* get the directory */
	dirp = opendir(localname);
	if (dirp == NULL) return(-1);

	/* get space for directory */
	filenamelist = (struct dirent **)calloc(DIRFILECOUNT, sizeof(struct dirent *));
	if (filenamelist == 0) return(0);
	tdirsize = sizeof (struct dirent);
	dir = (struct dirent *)malloc(tdirsize + _POSIX_PATH_MAX);
	if (dir == 0) return(0);

	/* construct the list */
	total = 0;
	while ((struct dirent *)readdir_r(dirp, dir) != NULL)
	{
		if (gra_fileselectall(dir))
		{
			filenamelist[total] = (struct dirent *)malloc(tdirsize + _POSIX_PATH_MAX);
			if (filenamelist[total] == NULL) return(0);
			memcpy(filenamelist[total], dir, sizeof(dir)+_POSIX_PATH_MAX);
			total++;
		}
	}
#endif

	gra_initfilelist();
	for(i=0; i<total; i++)
	{
		dir = filenamelist[i];
		if (gra_addfiletolist(dir->d_name) != 0) return(0);
		free((char *)dir);
	}
	free((char *)filenamelist);
	*filelist = getstringarray(gra_fileliststringarray, &len);
	return(len);
}

/*
 * routine to return the date of the last modification to file "filename"
 */
UINTBIG filedate(char *filename)
{
	struct stat buf;

	stat(filename, &buf);
	buf.st_mtime -= gra_machinetimeoffset();
	return(buf.st_mtime);
}

/* routine to convert a path name with "~" to a real path */
char *truepath(char *line)
{
	static char outline[100], home[50];
	REGISTER char save, *ch;
	static INTSML gothome = 0;
	struct passwd *tmp;

	if (line[0] != '~') return(line);

	/* if it is the form "~username", figure it out */
	if (line[1] != '/')
	{
		for(ch = &line[1]; *ch != 0 && *ch != '/'; ch++);
		save = *ch;   *ch = 0;
		tmp = getpwnam(&line[1]);
		if (tmp == 0) return(line);
		*ch = save;
		(void)strcpy(outline, tmp->pw_dir);
		(void)strcat(outline, ch);
		return(outline);
	}

	/* get the user's home directory once only */
	if (gothome == 0)
	{
		/* get name of user */
		tmp = getpwuid(getuid());
		if (tmp == 0) return(line);
		(void)strcpy(home, tmp->pw_dir);
		gothome++;
	}

	/* substitute home directory for the "~" */
	(void)strcpy(outline, home);
	(void)strcat(outline, &line[1]);
	return(outline);
}

/*
 * Routine to return information about the file or directory "name":
 *  0: does not exist
 *  1: is a file
 *  2: is a directory
 */
INTSML fileexistence(char *name)
{
	struct stat buf;

	if (stat(name, &buf) < 0) return(0);
	if ((buf.st_mode & S_IFMT) == S_IFDIR) return(2);
	return(1);
}

/*
 * Routine to create a directory.
 * Returns nonzero on error.
 */
INTSML createdirectory(char *dirname)
{
	char cmd[256];

	(void)sprintf(cmd, "mkdir %s", dirname);
	if (system(cmd) == 0) return(0);
	return(1);
}

/*
 * Routine to return the current directory name
 */
char *currentdirectory(void)
{
	char line[MAXPATHLEN];
#ifdef HAVE_GETCWD
	(void)getcwd(line, MAXPATHLEN);
#else
#  ifdef HAVE_GETWD
	(void)getwd(line);
#  else
	line[0] = 0;
#  endif
#endif
	(void)initinfstr();
	(void)addstringtoinfstr(line);
	return(returninfstr());
}

/*
 * Routine to lock a resource called "lockfilename" by creating such a file
 * if it doesn't exist.  Returns nonzero if successful, zero if unable to
 * lock the file.
 */
INTSML lockfile(char *lockfilename)
{
	int fd;

	fd = creat(lockfilename, 0);
	if (fd == -1 && errno == EACCES) return(0);
	if (fd == -1 || close(fd) == -1) return(0);
	return(1);
}

/*
 * Routine to unlock a resource called "lockfilename" by deleting such a file.
 */
void unlockfile(char *lockfilename)
{
	if (unlink(lockfilename) == -1)
		ttyputerr("Error unlocking %s", lockfilename);
}

/******************** TIME ROUTINES ********************/

/*
 * This routine returns the amount to add to the operating-system time
 * to adjust for a common time format.
 */
UINTBIG gra_machinetimeoffset(void)
{
	return(0);
}

UINTBIG ticktime(void)
{
#ifdef HAVE_GETTIMEOFDAY
	struct timeval tp;
	struct timezone tzp;

	gettimeofday(&tp, &tzp);
	return(tp.tv_sec * 60 + tp.tv_usec * 6 / 100000);
#else
	return(1);
#endif
}

/*
 * Routine to start counting time.
 */
void starttimer(void)
{
#ifdef HAVE_FTIME
	struct timeb tp;

	ftime(&tp);
	gra_timebasesec = tp.time;
	gra_timebasems = tp.millitm;
#else
	gra_timebasesec = time(0);
	gra_timebasems = 0;
#endif
}

/*
 * Routine to stop counting time and return the number of elapsed seconds
 * since the last call to "starttimer()".
 */
float endtimer(void)
{
	float seconds;
#ifdef HAVE_FTIME
	INTBIG milliseconds;
	struct timeb timeptr;

	ftime(&timeptr);
	milliseconds = (timeptr.time - gra_timebasesec) * 1000 +
		(timeptr.millitm-gra_timebasems);
	seconds = ((float)milliseconds) / 1000.0;
#else
	seconds = time(0) - gra_timebasesec;
#endif
	return(seconds);
}

/*
 * Routine to wait "ticks" sixtieths of a second and then return.
 */
void gotosleep(INTBIG ticks)
{
	sleep(ticks);
}

/*************************** EVENT ROUTINES ***************************/

/*
 * helper routine to wait for some keyboard or mouse input
 */

void gra_destroygraphics(Widget w, XtPointer data, XEvent *event, Boolean *cont)
{
	if (event->type == ClientMessage &&
		event->xclient.message_type == gra_wm_protocols)
	{
		if (event->xclient.data.l[0] == gra_wm_delete_window)
		{
			if (w == gra_msgtoplevelwidget)
			{
				/* iconify the messages window */
				XIconifyWindow(gra_topmsgdpy, gra_topmsgwin, gra_screen_num);
			} else
			{
				/* simply refuse to do anything when edit window is closed */
				ttybeep();
				ttyputerr("WARNING! An attempt was made to kill your graphics window.");
				ttyputmsg("Always exit from within Electric to prevent loss of data.");
			}
		}
	}
}

void gra_graphics_event_handler(Widget w, XtPointer data, XEvent *event, Boolean *cont)
{
	gra_handleeditevent(event);
}

void gra_messages_event_handler(Widget w, XtPointer data, XEvent *event, Boolean *cont)
{
	gra_handlemsgevent(event);
}


void gra_nextevent(void)
{
	XEvent event;

	XtNextEvent(&event);
	XtDispatchEvent(&event);
}

void gra_handlemsgevent(XEvent *event)
{
	char buffer[2];
	REGISTER INTBIG which, ret;
	void us_onint(void);

	switch (event->type)
	{
		case VisibilityNotify:
			if (event->xvisibility.state == VisibilityUnobscured)
				gra_messages_obscured = 0; else
					gra_messages_obscured = 1;
			break;

#if 0
		case MappingNotify:
			XRefreshKeyboardMapping(event);
			break;
#endif

		case KeyPress:
			ret = XLookupString((XKeyEvent *)event, buffer, 2, NULL, NULL);
			if (ret < 1) which = 0; else
				which = buffer[0];
			if (which == CTRLC)
			{
				/* we have to do interrupts here */
				us_onint();
				which = 0;
			}
			if (which == 0) break;
			gra_inputstate = (which & CHARREAD) | ISCHAR;
			us_state |= DIDINPUT;
			if ((event->xbutton.state & Mod1Mask) != 0)
				gra_inputstate |= METADOWN;
			event->type = 0;     /* disable this event so key doesn't go into window */
			break;

#if 0
		default:
			printf("Unhandled event %s in messages window\n",
				eventNames[event->type]);
			break;
#endif
	}
}

void gra_handleeditevent(XEvent *event)
{
	INTSML redo;
	char buffer[2];
	REGISTER INTBIG which, ret, count;
	XWindowAttributes xwa;
	void us_onint(void);
	XEvent peekevent;

	switch (event->type)
	{
		case Expose:
			/* ignore partial events */
			if (event->xexpose.count != 0) break;

			/* ignore if further Expose events are queued */
			count = XEventsQueued(gra_dpy, QueuedAlready);
			if (count > 0)
			{
				XPeekEvent(gra_dpy, &peekevent);
				if (peekevent.type == Expose) break;
			}

			if (XGetWindowAttributes(gra_topdpy, gra_topwin, &xwa) == 0) exit(1);
			if (xwa.width != us_swid || xwa.height != us_shei + gra_status_height)
				redo = 1; else
					redo = 0;
			gra_repaint(redo);
			break;

		case ClientMessage:
			/* watch for deletes of our windows */
			if (event->xclient.message_type == gra_wm_protocols)
			{
				if (event->xclient.data.l[0] == gra_wm_delete_window)
				{
					ttybeep();
					ttyputerr("WARNING! An attempt was made to kill your graphics window.");
					ttyputmsg("Always exit from within Electric to prevent loss of data.");
				}
			}
			break;

		case ConfigureNotify:
#ifdef HASSPYGLASS
			gra_getboundaries(gra_win);
#endif
			break;

		case KeyPress:
			ret = XLookupString((XKeyEvent *)event, buffer, 2, NULL, NULL);
			if (ret < 1) which = 0; else
				which = buffer[0];
			if (which == CTRLC)
			{
				/* we have to do interrupts here */
				us_onint();
				which = 0;
			}
			if (which == 0) break;
			gra_inputstate = (which & CHARREAD) | ISCHAR;
			us_state |= DIDINPUT;
			if ((event->xbutton.state & Mod1Mask) != 0)
				gra_inputstate |= METADOWN;

			/* don't track in the status window */
			gra_cursorx = event->xkey.x;
			gra_cursory = maxi(gra_revy - event->xkey.y, 0);
			break;

		case ButtonPress:  /* these are ignored in the messages window */
			switch (event->xbutton.button)
			{
				case 1: gra_inputstate = ISLEFT;   break;
				case 2: gra_inputstate = ISMIDDLE; break;
				case 3: gra_inputstate = ISRIGHT;  break;
			}
			if ((event->xbutton.state & ShiftMask) != 0)
				gra_inputstate |= SHIFTDOWN;
			if ((event->xbutton.state & LockMask) != 0)
				gra_inputstate |= SHIFTDOWN;
			if ((event->xbutton.state & ControlMask) != 0)
				gra_inputstate |= CONTROLDOWN;
			if ((event->xbutton.state & Mod1Mask) != 0)
				gra_inputstate |= METADOWN;
			gra_cursorx = event->xbutton.x;
			gra_cursory = maxi(gra_revy - event->xbutton.y, 0);
			if (((event->xbutton.time - gra_lasttime) <= gra_doublethresh) &&
				(gra_inputstate & (SHIFTDOWN|CONTROLDOWN|METADOWN)) == 0 &&
				(gra_cursorx == gra_lstcurx) && (gra_cursory == gra_lstcury))
			{
				gra_inputstate |= DOUBLECL;
				gra_lasttime = event->xbutton.time - gra_doublethresh - 1;
			} else
			{
				gra_lasttime = event->xbutton.time;
			}
			gra_lstcurx = gra_cursorx;
			gra_lstcury = gra_cursory;
			us_state |= DIDINPUT;
			break;

		case ButtonRelease:
			switch (event->xbutton.button)
			{
				case 1: gra_inputstate = ISLEFT | BUTTONUP;   break;
				case 2: gra_inputstate = ISMIDDLE | BUTTONUP; break;
				case 3: gra_inputstate = ISRIGHT | BUTTONUP;  break;
			}
			if ((event->xbutton.state & ShiftMask) != 0)
				gra_inputstate |= SHIFTDOWN;
			if ((event->xbutton.state & LockMask) != 0)
				gra_inputstate |= SHIFTDOWN;
			if ((event->xbutton.state & ControlMask) != 0)
				gra_inputstate |= CONTROLDOWN;
			if ((event->xbutton.state & Mod1Mask) != 0)
				gra_inputstate |= METADOWN;
			gra_cursorx = event->xbutton.x;
			gra_cursory = maxi(gra_revy - event->xbutton.y, 0);
			us_state |= DIDINPUT;
			break;

		case MotionNotify:
#ifdef SHOWCURSOR
			gra_showcoords(event->xmotion.x, gra_revy - event->xmotion.y);
#endif
			if (gra_ignoremotion == 0)
			{
				gra_inputstate = MOTION;
				if ((event->xmotion.state & (Button1Mask | Button2Mask | Button3Mask)) == 0)
					gra_inputstate |= BUTTONUP;
				gra_cursorx = event->xmotion.x;
				gra_cursory = gra_revy - event->xmotion.y;
				if (gra_cursory < 0) gra_cursory = 0;
			}
#ifdef HASSPYGLASS
			else if (gra_spyx >= 0)
			{
				if ((event->xmotion.state & (Button1Mask | Button2Mask | Button3Mask)) == 0)
				{
					/* skip over all of the movements up to the last one */
					while(XCheckTypedEvent(gra_dpy, MotionNotify, event));
					gra_cursorx = event->xmotion.x;
					gra_cursory = gra_revy - event->xmotion.y;
					if (gra_cursory < 0) gra_cursory = 0;
					gra_advancespy();
				}
			}
#endif
			break;

#if 0
		case MappingNotify:
			XRefreshKeyboardMapping(event);
			break;
#endif

		case ReparentNotify:
#ifdef HASSPYGLASS
			gra_getboundaries(gra_win);
#endif
			break;

#if 0
		default:
			(void)fprintf(stderr, "Unhandled event %s in graphics window.\n",
				eventNames[event->type]);
			break;
#endif
	}
}

#ifdef HASSPYGLASS
/*
 * Helper routine to update the current boundaries of the window on the screen
 * call it with the current window
 */
void gra_getboundaries(Window win)
{
	Window root, parent, *childlist;
	UINTBIG  nchild;
	INTSML xmin, ymin;
	XWindowAttributes xwa, xwdummy;

	if (XGetWindowAttributes(gra_dpy, gra_win, &xwa) == 0) exit(1);

	xmin = xwa.x;
	ymin = xwa.y;

	if (XQueryTree(gra_dpy, win, &root, &parent, &childlist, &nchild))
	{
		/* if we are using window manager, then we have to find where the window really is */
		if (nchild) XFree((char *)childlist);
		while (root != parent && parent != None)
		{
			XGetWindowAttributes(gra_dpy, parent, &xwdummy);
			xmin += xwdummy.x;
			ymin += xwdummy.y;
			if(!XQueryTree(gra_dpy, parent, &root, &parent, &childlist, &nchild)) break;
			if (nchild) XFree((char *)childlist);
		}
	}

	gra_xmin = maxi(0, -xmin);
	gra_ymin = maxi(0, -ymin);
	gra_xmax = mini(xwa.width, gra_screenw - xmin - 1);
	gra_ymax = mini(xwa.height, gra_screenh - ymin - 1);
}
#endif

void gra_repaint(INTSML redo)
{
	XWindowAttributes xwa;

	XClearWindow(gra_dpy, gra_win);
	us_pushhighlight();

	/* we could use a global xwa */
	XGetWindowAttributes(gra_topdpy, gra_topwin, &xwa);

	us_swid = xwa.width;
	gra_shei = xwa.height;
	us_shei = gra_shei - gra_status_height;
	gra_offx = 0;
	gra_revy = us_shei - 1;
#ifdef HASSPYGLASS
	gra_getboundaries(gra_win);
#endif
	us_clearhighlightcount();

	us_redostatus(0);

	/* draw the separator line between drawing and status areas */
	us_drawline(NULL, 1, -2, us_swid - 1, -2, &us_box, 0);
	if (redo != 0) us_drawmenu(1, NULL); else
		us_drawmenu(-1, NULL);

	/* redraw any popup menus */
	DiaRedrawDialogWindow();
	(void)us_pophighlight(0);
	setactivity("WINDOW RESIZE");
}

struct {
	INTSML x;
	INTSML y;
	INTSML kind;
} gra_action;

/*
 * helper routine to wait for some keyboard or mouse input
 */
void gra_waitforaction(void)
{
	REGISTER INTBIG err;

#ifdef HASSPYGLASS
	/* turn magnification back on if it is supposed to be */
	if ((gra_spymag != 1) && (gra_spyx < 0)) gra_writespy();
#endif

	if (us_logplay != NULL)
	{
		err = fread((char *)&gra_action, sizeof(gra_action), 1, us_logplay);
		if (stopping("Playback") != 0) err = 0;
		if (err != 0)
		{
			if (gra_playbackmultiple <= 0)
			{
				gra_playbackmultiple = 0;
				for(;;)
				{
					gra_nextevent();
					if (gra_inputstate == NOEVENT) continue;
					if ((gra_inputstate & ISCHAR) != ISCHAR) break;
					if ((gra_inputstate & CHARREAD) < '0' ||
						(gra_inputstate & CHARREAD) > '9') break;
					gra_playbackmultiple = gra_playbackmultiple * 10 +
						(gra_inputstate & CHARREAD) - '0';
				}
				if ((gra_inputstate != NOEVENT) &&
					((gra_inputstate & (ISCHAR | CHARREAD)) == (ISCHAR | 'q'))) err = 0;
			}
			gra_playbackmultiple--;
			gra_inputstate = gra_action.kind & 0xFFFF;
			gra_cursorx = gra_action.x;
			gra_cursory = gra_action.y;
			if (gra_cursory < 0) gra_cursory = 0;
		}
		if (err == 0)
		{
			ttyputmsg("End of session playback file");
			(void)fclose(us_logplay);
			us_logplay = NULL;
			return;
		}
	} else
	{
		gra_nextevent();
	}

	if ((us_logrecord != NULL) && (gra_inputstate != NOEVENT))
	{
		gra_action.kind = gra_inputstate;
		gra_action.x = gra_cursorx;
		gra_action.y = gra_cursory;
		if (fwrite((char *)&gra_action, sizeof (gra_action), 1, us_logrecord) == 0)
		{
			ttyputerr("Error writing session log file: recording disabled");
			us_logfinishrecord();
		}
		gra_logrecordindex++;
		if (gra_logrecordindex >= us_logflushfreq)
		{
			/* flush the session log file */
			gra_logrecordindex = 0;
			(void)fflush(us_logrecord);
		}
	}
}

/*************************** SESSION LOGGING ROUTINES ***************************/

/*
 * routine to begin playback of session logging file "file".  The routine
 * returns nonzero if there is an error.  If "all" is nonzero, playback
 * the entire file with no prompt.
 */
INTSML us_logplayback(char *file, INTSML all)
{
	struct stat buf;
	REGISTER INTSML comcount;

	us_logplay = fopen(truepath(file), "r");
	if (us_logplay == NULL) return(1);

	ttyputmsg("Type any key to playback the next command in the log file");
	ttyputmsg("Type a number followed by 'x' to execute that many commands");
	ttyputmsg("Type 'q' to terminate playback");

	if (stat(file, &buf) >= 0)
	{
		comcount = buf.st_size / (sizeof(gra_action));
		ttyputmsg(" (there are %d commands to playback)", comcount);
	} else
	{
		comcount = 0;
	}

	if (all != 0) gra_playbackmultiple = comcount;
	return(0);
}

/*
 * routine to create a session logging file
 */
void us_logstartrecord(void)
{
#if 0
	us_logrecord = xcreate(ELECTRICLOG, FILETYPELOG, 0, 0);
#else
	us_logrecord = NULL;
#endif
}

/*
 * routine to terminate session logging
 */
void us_logfinishrecord(void)
{
	if (us_logrecord != NULL) xclose(us_logrecord);
	us_logrecord = NULL;
}

/****************************** MENUS ******************************/

/* not required with X11 */
INTSML gra_popupmenu(POPUPMENU *menu, INTSML header, INTSML left, INTSML top)
{
	return(-1);
}

/* not required with X11 */
void gra_nativemenudoone(INTSML low, INTSML high) {}

/* not required with X11 */
INTSML gra_nativemenuload(INTSML count, char *par[])
{
	return(1);
}

/* not required with X11 */
void gra_nativemenurename(POPUPMENU *pm, INTSML index) {}

/****************************** SYSTEM CALLS ******************************/

/*
 * routine to load the global Graphic Context "gra_gcstip" with the
 * stipple pattern in "raster"
 */
void gra_loadstipple(UINTSML raster[])
{
	static Pixmap buildup = (Pixmap)NULL;
	static XImage ximage;
	static INTBIG data[8];
	static GC pgc;
	XGCValues gcv;
	REGISTER INTBIG y, pattern;
	static INTSML reverse = 0;   /* for debugging and testing */

	/* initialize raster structure for patterned write */
	if (buildup == (Pixmap)NULL)
	{
		buildup = XCreatePixmap(gra_dpy, gra_win, 32, 8, 1);
		if (gra_sdep != 1)
		{  /* choose polarity of stipples */
#ifdef BLKBGD
			gcv.foreground = WhitePixel(gra_dpy, DefaultScreen(gra_dpy));
			/* was black*/
			gcv.background = BlackPixel(gra_dpy, DefaultScreen(gra_dpy));
#else
#  ifdef i386
			gcv.background = BlackPixel(gra_dpy, DefaultScreen(gra_dpy));
			gcv.foreground = WhitePixel(gra_dpy, DefaultScreen(gra_dpy));
#  else
			gcv.foreground = BlackPixel(gra_dpy, DefaultScreen(gra_dpy));
			gcv.background = WhitePixel(gra_dpy, DefaultScreen(gra_dpy));
#  endif
#endif
		} else
		{  /* monochrome */
			gcv.foreground = gra_fgd;
			gcv.background = gra_bgd;
		}
		if (reverse)
		{
			y = gcv.foreground;
			gcv.foreground = gcv.background;
			gcv.background = y;
		}
		gcv.fill_style = FillStippled;
		pgc = XCreateGC(gra_dpy, buildup, GCForeground | GCBackground | GCFillStyle, &gcv);
		ximage.width = 32;
		ximage.height = 8;
		ximage.xoffset = 0;
		ximage.format = XYBitmap;
		ximage.data = (char *)data;
		ximage.byte_order = XImageByteOrder(gra_dpy);
		ximage.bitmap_unit = XBitmapUnit(gra_dpy);
		ximage.bitmap_bit_order = XBitmapBitOrder(gra_dpy);
		ximage.bitmap_pad = XBitmapPad(gra_dpy);
		ximage.depth = 1;
		ximage.bytes_per_line = 4;
	}

	for(y=0; y<8; y++)
	{
		pattern = raster[y] & 0xFFFF;
		data[y] = pattern | (pattern << 16);
	}

	XPutImage(gra_dpy, buildup, pgc, &ximage, 0, 0, 0, 0, 32, 8);
	XSetStipple(gra_dpy, gra_gcstip, buildup);
}

/*
 * support routine to print any internal errors
 */
INTBIG gra_errors(Display *dpy, XErrorEvent *err)
{
	char buffer[100];

	XGetErrorText(dpy, err->error_code, buffer, 100);
	ttyputerr("ERROR: X Window System routine %d has %s", err->request_code, buffer);
	return(0);
}

#ifdef SHOWCURSOR
/*
 * Function to show the cursor coordinate in the current window
 * enter with cursor coordinates in x, y, display in current lambda.
 */
void gra_showcoords(INTSML x, INTSML y)
{
	INTBIG icx = x, icy = y;
	WINDOW *w;
	char temp[30];
	INTSML lx, ly, hx, hy;
	static POLYGON  *poly = NOPOLYGON;
	INTSML wid, hei;
	static INTSML maxwid;
	XFontStruct *oldfont;

	if (poly == NOPOLYGON) poly = allocpolygon(4, us_aid->cluster);

	for (w = el_topwindow; w!= NOWINDOW; w = w->nextwindow)
	{
		if ((x >= w->uselx) && (x <= w->usehx) && (y >= w->usely) && (y <= w->usehy))
		{
			us_scaletowindow(&icx, &icy, w);
			if (us_alignment != 0) gridalign(&icx, &icy, us_alignment);
			(void)sprintf(temp, "(%s, %s)", latoa(icx), latoa(icy));
			oldfont = gra_curfont;  /* save current environment */
			us_settextsize(NULL, TXTMENU);
			us_textsize(NULL, temp, &wid, &hei);
			maxwid = maxi(wid, maxwid);
			hx = us_swid - 5; lx = hx - maxwid;
			hy = us_shei - 5; ly = hy - hei;
			us_drawbox(NULL, hx-maxwid, hx, ly, hy, &us_ebox);  /* erase old value */
			us_puttext(w, hx-wid, ly, temp, &us_menutext);
			gra_curfont = oldfont;
		}
	}
}
#endif
