/*
 * Electric(tm) VLSI Design System
 *
 * File: global.h
 * Definitions of interest to all modules
 * Written by: Steven M. Rubin, Electric Editor Incorporated
 *
 * Copyright (c) 1998 Electric Editor Incorporated.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Electric Editor Incorporated
 * 23470 Sunset Drive, Suite 108
 * Los Gatos, California 95033
 * support@electriceditor.com
 */

#include "config.h"

/* system includes */
#include <stdio.h>
#if STDC_HEADERS
#  include <stdlib.h>
#  include <string.h>
#  include <stdarg.h>
#  define var_start(x, y) va_start(x, y)
#else
#  ifndef HAVE_STRCHR
#    define strchr index
#  endif
#  include <varargs.h>
#  define var_start(x, y) va_start(x)
#endif
#ifdef HAVE_STRINGS_H
#  include <strings.h>
#endif

/* define the basic integer types */
#ifdef WIN32
#  include "windows.h"
#  undef TECHNOLOGY
   typedef LONG            INTBIG;
   typedef SHORT           INTSML;
   typedef ULONG          UINTBIG;
   typedef USHORT         UINTSML;
#else
   typedef          int    INTBIG;
   typedef          short  INTSML;
   typedef unsigned int   UINTBIG;
   typedef unsigned short UINTSML;
#endif

typedef INTBIG XARRAY[3][3];
#define	SIZEOFINTBIG (sizeof (INTBIG))		/* bytes per long integer */
#define	SIZEOFINTSML (sizeof (INTSML))		/* bytes per short integer */

/* SUN and MIPS systems don't handle "register" properly */
#if defined(sun) || defined(mips)
#  define REGISTER
#else
#  define REGISTER register
#endif

/* forward declarations for structures */
struct Icomcomp;
struct Inodeinst;
struct Iportarcinst;
struct Iportexpinst;
struct Inodeproto;
struct Iportproto;
struct Iarcinst;
struct Iarcproto;
struct Inetwork;
struct Igeom;
struct Irtnode;
struct Ilibrary;
struct Itechnology;

/*************************** VARIABLES ****************************/

#define	NOVARIABLE  ((VARIABLE *)-1)

/*
 * when adding objects to this list, also update:
 *   "dbchange.c:db_namevariable"
 *   "dbvars.c"
 *   "dbtext.c:db_makestringvar"
 *   "usrstatus.c:us_variableattributes"
 *   "usrcom1.c:us_nextvars"
 */
/* the value of VARIABLE->type */
#define	VUNKNOWN                 0			/* undefined variable */
#define	VINTEGER                01			/* 32-bit integer variable */
#define	VADDRESS                02			/* unsigned address */
#define	VCHAR                   03			/* character variable */
#define	VSTRING                 04			/* string variable */
#define	VFLOAT                  05			/* floating point variable */
#define	VDOUBLE                 06			/* double-precision floating point */
#define	VNODEINST               07			/* nodeinst pointer */
#define	VNODEPROTO             010			/* nodeproto pointer */
#define	VPORTARCINST           011			/* portarcinst pointer */
#define	VPORTEXPINST           012			/* portexpinst pointer */
#define	VPORTPROTO             013			/* portproto pointer */
#define	VARCINST               014			/* arcinst pointer */
#define	VARCPROTO              015			/* arcproto pointer */
#define	VGEOM                  016			/* geometry pointer */
#define	VLIBRARY               017			/* library pointer */
#define	VTECHNOLOGY            020			/* technology pointer */
#define	VAID                   021			/* aid pointer */
#define	VRTNODE                022			/* R-tree pointer */
#define	VFRACT                 023			/* fractional integer (scaled by WHOLE) */
#define	VNETWORK               024			/* network pointer */
#define	VCELL                  025			/* cell pointer */
#define	VVIEW                  026			/* view pointer */
#define	VWINDOW                027			/* window pointer */
#define	VGRAPHICS              030			/* graphics object pointer */
#define	VSHORT                 031			/* 16-bit integer */
#define	VCONSTRAINT            032			/* constraint solver */
#define	VGENERAL               033			/* general address/type pairs (used only in fixed-length arrays) */
#define	VTYPE                  037			/* all above type fields */
#define	VCODE1                 040			/* variable is interpreted code (with VCODE2) */
#define	VDISPLAY              0100			/* display variable (uses textdescript field) */
#define	VISARRAY              0200			/* set if variable is array of above objects */
#define VCREF                 0400			/* variable points into C structure */
#define	VLENGTH        03777777000			/* array length (0: array is -1 terminated) */
#define	VLENGTHSH                9			/* right shift for VLENGTH */
#define	VCODE2         04000000000			/* variable is interpreted code (with VCODE1) */
#define	VLISP               VCODE1			/* variable is LISP */
#define	VTCL                VCODE2			/* variable is TCL */
#define	VMATHEMATICA VCODE1|VCODE2			/* variable is Mathematica */
#define	VDONTSAVE     010000000000			/* set to prevent saving on disk */
#define	VCANTSET      020000000000			/* set to prevent changing value */

/* the value of VARIABLE->textdescript */
#define	VTPOSITION         017				/* position of text relative to point */
#define	VTPOSCENT            0				/* text centered about point */
#define	VTPOSUP              1				/* text centered above point */
#define	VTPOSDOWN            2				/* text centered below point */
#define	VTPOSLEFT            3				/* text centered to left of point */
#define	VTPOSRIGHT           4				/* text centered to right of point */
#define	VTPOSUPLEFT          5				/* text centered to upper-left of point */
#define	VTPOSUPRIGHT         6				/* text centered to upper-right of point */
#define	VTPOSDOWNLEFT        7				/* text centered to lower-left of point */
#define	VTPOSDOWNRIGHT       8				/* text centered to lower-right of point */
#define	VTPOSBOXED           9				/* text centered and limited to object size */
#define	VTROTATION         060				/* rotation of text (90 degree increments) */
#define	VTROTATIONSH         4				/* right shift of VTROTATION */
#define	VTSIZE           01700				/* size of text (in points) */
#define	VTSIZESH             6				/* right shift of VTSIZE */
#define	VTFONT           06000				/* text font (2 bits only) */
#define	VTFONTSH            10				/* right shift of VTFONT */
#define	VTXOFF        07770000				/* X offset of text */
#define	VTXOFFSH            12				/* right shift of VTXOFF */
#define	VTXOFFNEG    010000000				/* set if X offset is negative */
#define	VTYOFF    017760000000				/* Y offset of text */
#define	VTYOFFSH            22				/* right shift of VTYOFF */
#define	VTYOFFNEG 020000000000				/* set if Y offset is negative */

typedef struct
{
	INTBIG    key;							/* library-specific key to this name */
	UINTBIG   type;							/* the type of variables (see above) */
	UINTBIG   textdescript;					/* nature of text that displays variable */
	INTBIG    addr;							/* contents of variable */
} VARIABLE;

extern char  **el_namespace;				/* names in name space */
extern INTSML  el_numnames;					/* number of names in name space */

/* some keys to commonly used variable names */
extern INTBIG  el_node_name;				/* variable "NODE_name" */
extern INTBIG  el_arc_name;					/* variable "ARC_name" */
extern INTBIG  el_arc_radius;				/* variable "ARC_radius" */
extern INTBIG  el_trace;					/* variable "trace" */
extern INTBIG  el_facet_message;			/* variable "FACET_message" */
extern INTBIG  el_transistor_width;			/* variable "transistor_width" */
extern INTBIG  el_prototype_center;			/* variable "prototype_center" */

/*************************** MEMORY ALLOCATION ****************************/

#define	NOCLUSTER	((CLUSTER *)-1)

#define	CLUSTERFILLING   1					/* set if no need to search entire cluster */

typedef struct Icluster
{
	INTBIG           address;				/* base address of this cluster */
	INTBIG           flags;					/* information bits about this cluster */
	char             clustername[20];		/* for debugging only */
	INTBIG           clustersize;			/* number of pages to allocate at a time */
	struct Icluster *nextcluster;			/* next in linked list */
} CLUSTER;

extern CLUSTER *el_tempcluster;				/* cluster for temporary allocation */
extern CLUSTER *db_cluster;					/* database general allocation */

/************************** COMMAND COMPLETION ***************************/

#define	TEMPLATEPARS  5						/* maximum parameters in a template */
#define	MAXPARS      20						/* maximum parameters in a command */
#define	NOCOMCOMP    ((COMCOMP *)-1)
#define	NOKEYWORD    ((KEYWORD *)-1)
#define	NOKEY        NOCOMCOMP				/* short form for null keyword */
#define	TERMKEY      {NULL, 0, {NULL, NULL, NULL, NULL, NULL}}
#define	NOTOPLIST    (us_patoplist)			/* no routine for the keyword table */
#define	NONEXTLIST   (us_panextinlist)		/* no routine for table slot */
#define	NOPARAMS     (us_paparams)			/* no routine for table slot */
#define	NOBACKUP     (us_pabackupto)		/* no routine for entry backing up */

/* bits in COMCOMP->interpret */
#define	NOFILL         01					/* if set, don't fill out keyword */
#define	NOSHOALL       02					/* if set, suppress options list when null */
#define	INPUTOPT       04					/* if set, this can be input option on popup */
#define	INCLUDENOISE  010					/* if set, include "noise" option in list */
#define	MULTIOPT      020					/* if set, allow multiple menu picks */

/*
 * tables of command options use this structure
 */
typedef struct Ikeyword
{
	char            *name;					/* name of this command */
	INTSML           params;				/* number of parameters to command */
	struct Icomcomp *par[TEMPLATEPARS];		/* parameter types */
} KEYWORD;

/*
 * this structure defines the basic command parameter
 */
typedef struct Icomcomp
{
	KEYWORD *ifmatch;						/* list of keywords to search if it matches */
	INTSML (*toplist)(char**);				/* reset to top of list of keywords */
	char  *(*nextcomcomp)(void);			/* give next keyword in list */
	INTSML (*params)(char*, struct Icomcomp*[], char);	/* set parameters to keyword */
	void   (*backupto)(INTSML, char*[]);	/* backup to a keyword position */
	INTSML   interpret;						/* bits for interpretation */
	char    *breakchrs;						/* keyword separation characters */
	char    *noise;							/* describe list */
	char    *def;							/* default value */
} COMCOMP;

/*************************** AIDS ****************************/

#define	NOAID	((AIDENTRY *)-1)

/* capabilities of each aid (in AIDENTRY->aidcap) */
#define	USERHASTEXTED  01	/* User interface: text edit available */
#define	USERHASTECHED  02	/* User interface: technology edit available */
#define	USERHASLANG    04	/* User interface: Language interface available */

#define	IOHASIC        01	/* I/O: GDS, CIF, EDIF, L, VHDL available */
#define	IOHASMECH      02	/* I/O: DXF available */
#define	IOCANWRITE     04	/* I/O: can write libraries (demo) */
#define	IOHASPRINT    010	/* I/O: Postscript, HPGL, QuickDraw available */

#define	COMPAVAIL      01	/* Compaction: available */

#define	PLAHASNMOS     01	/* PLA generator: nMOS generator available */
#define	PLAHASCMOS     02	/* PLA generator: CMOS generator available */

#define	ROUTHASSTITCH  01	/* Router: stitcher/mimic available */
#define	ROUTHASRIVER   02	/* Router: river available */
#define	ROUTHASMAZE    04	/* Router: maze available */

#define	SILCOMPAVAIL   01	/* Silicon compiler: available */

#define	NETHASNCC      01	/* Network: NCC available */

#define	DRCHASINC      01	/* DRC: incremental check available */
#define	DRCHASBATCH    02	/* DRC: batch check available */
#define	DRCHASDRAC     04	/* DRC: Dracula check available */

#define	VHDLHASGEN     01	/* VHDL compiler: can generate VHDL */
#define	VHDLHASCOMP    02	/* VHDL compiler: can compile VHDL */

#define	SIMHASSPICE    01	/* Simulation: SPICE available */
#define	SIMHASALS      02	/* Simulation: ALS available */
#define	SIMHASCOMM     04	/* Simulation: Commercial interfaces available */
#define	SIMHASUNIV    010	/* Simulation: University interfaces available */

#define	MISCHASCOMPEN  01	/* Miscellaneous: Compensation available */
#define MISCHASPROJ    02	/* Miscellaneous: Project management available */
#define MISCHASMAPPER  04	/* Miscellaneous: Mapper available */

/* aid descriptors */
typedef struct Iaidentry
{
	char    *aidname;						/* name of aid */
	INTBIG   aidstate;						/* state of aid */
	INTBIG   aidcap;						/* capabilities of aid */
	INTSML   index;							/* aid index */
	COMCOMP *parse;							/* parsing structure for aid direction */
	CLUSTER *cluster;						/* virtual memory cluster for this aid */

	void   (*init)(INTBIG*, char*[], struct Iaidentry*);	/* initialization */
	void   (*done)(void);									/* completion */
	INTSML (*setmode)(INTSML, char*[]);						/* user-command options */
	INTBIG (*request)(char*, va_list);						/* direct call options */
	void   (*examinenodeproto)(struct Inodeproto*);			/* to examine an entire facet at once */
	void   (*slice)(void);									/* time slice */

	void   (*startbatch)(struct Iaidentry*);				/* start change broadcast */
	void   (*endbatch)(void);								/* end change broadcast */

	void   (*startobjectchange)(INTBIG, INTBIG);			/* broadcast that object about to be changed */
	void   (*endobjectchange)(INTBIG, INTBIG);				/* broadcast that object done being changed */

	void   (*modifynodeinst)(struct Inodeinst*,INTBIG,INTBIG,INTBIG,INTBIG,INTSML,INTSML);	/* broadcast modified nodeinst */
	void   (*modifyarcinst)(struct Iarcinst*,INTBIG,INTBIG,INTBIG,INTBIG,INTBIG,INTBIG);	/* broadcast modified arcinst */
	void   (*modifyportproto)(struct Iportproto*, struct Inodeinst*, struct Iportproto*);	/* broadcast modified portproto */
	void   (*modifynodeproto)(struct Inodeproto*);			/* broadcast modified nodeproto */
	void   (*modifydescript)(VARIABLE*, INTBIG);			/* broadcast modified descriptor */

	void   (*newobject)(INTBIG, INTBIG);					/* broadcast new object */
	void   (*killobject)(INTBIG, INTBIG);					/* broadcast deleted object */
	void   (*newvariable)(INTBIG, INTBIG, INTBIG, INTBIG);	/* broadcast new variable */
	void   (*killvariable)(INTBIG, INTBIG, INTBIG, INTBIG, INTBIG, INTBIG);	/* broadcast deleted variable */
	void   (*modifyvariable)(INTBIG, INTBIG, INTBIG, INTBIG, INTBIG, INTBIG);	/* broadcast modified array variable entry */
	void   (*insertvariable)(INTBIG, INTBIG, INTBIG, INTBIG, INTBIG);			/* broadcast inserted array variable entry */
	void   (*deletevariable)(INTBIG, INTBIG, INTBIG, INTBIG, INTBIG, INTBIG);	/* broadcast deleted array variable entry */

	void   (*readlibrary)(struct Ilibrary*);				/* broadcast newly read library */
	void   (*eraselibrary)(struct Ilibrary*);				/* broadcast library to be erased */
	void   (*writelibrary)(struct Ilibrary*, INTSML);		/* broadcast library writing */

	/* variables */
	VARIABLE *firstvar;						/* first variable in list */
	INTSML    numvar;						/* number of variables in list */
} AIDENTRY;

extern AIDENTRY el_aids[];					/* defined in "aidtable.c" */
extern INTSML   el_maxaid;					/* current number of aids */

/* the value of AIDENTRY->aidstate */
#define	AIDON              01				/* set if aid is on */
#define	AIDBG              02				/* set if aid is running in background */
#define	AIDFIX             04				/* set if aid will fix errors */
#define	AIDLISP           010				/* set if aid is coded in Lisp */
#define	AIDINCREMENTAL    020				/* set if aid functions incrementally */
#define	AIDANALYSIS       040				/* set if aid does analysis */
#define	AIDSYNTHESIS     0100				/* set if aid does synthesis */

/*************************** VIEWS ****************************/

#define	NOVIEW ((VIEW *)-1)

/* the value of VIEW->viewstate */
#define	TEXTVIEW         01					/* view contains only text */
#define	MULTIPAGEVIEW    02					/* view is one of multiple pages */
#define	PERMANENTVIEW    04					/* view is statically defined and cannot be deleted */

typedef struct Iview
{
	char         *viewname;					/* name of this view */
	char         *sviewname;				/* short name of this view */
	struct Iview *nextview;					/* next in linked list */
	INTBIG        temp1, temp2;				/* working variables */
	INTBIG        viewstate;				/* information about the view */

	/* variables */
	VARIABLE     *firstvar;					/* first variable in list */
	INTSML        numvar;					/* number of variables in list */
} VIEW;

extern VIEW *el_views;						/* list of existing view */
extern VIEW *el_unknownview;				/* the unknown view */
extern VIEW *el_layoutview;					/* the layout view */
extern VIEW *el_schematicview;				/* the schematic view */
extern VIEW *el_iconview;					/* the icon view */
extern VIEW *el_simsnapview;				/* the simulation-snapshot view */
extern VIEW *el_skeletonview;				/* the skeleton view */
extern VIEW *el_compview;					/* the compensated view */
extern VIEW *el_vhdlview;					/* the VHDL view (text) */
extern VIEW *el_netlistview;				/* the netlist view, generic (text) */
extern VIEW *el_netlistnetlispview;			/* the netlist view, netlisp (text) */
extern VIEW *el_netlistalsview;				/* the netlist view, als (text) */
extern VIEW *el_netlistquiscview;			/* the netlist view, quisc (text) */
extern VIEW *el_netlistrsimview;			/* the netlist view, rsim (text) */
extern VIEW *el_netlistsilosview;			/* the netlist view, silos (text) */
extern VIEW *el_docview;					/* the documentation view (text) */

/*************************** CELLS ****************************/

#define	NOCELL ((CELL *)-1)

typedef struct Icell
{
	char              *cellname;			/* name of this cell */
	struct Icell      *nextcell;			/* next in list of cells */
	struct Inodeproto *firstincell;			/* first cell in list */
	struct Ilibrary   *lib;					/* library containing this cell */
	CLUSTER           *cluster;				/* virtual memory cluster for this cell */
	struct Inetwork   *freenetwork;			/* free nets in this cell */
	INTBIG             temp1, temp2;		/* working variables */

	/* variables */
	VARIABLE          *firstvar;			/* first variable in list */
	INTSML             numvar;				/* number of variables in list */
} CELL;

/*************************** NODE INSTANCES ****************************/

#define	NONODEINST ((NODEINST *)-1)

/* the value of NODEINST->userbits (bits 5, 6 (0140) used by database) */
#define	DEADN         01					/* node is not in use */

#define	NEXPAND       04					/* if on, draw node expanded */
#define	WIPED        010					/* set if node not drawn due to wiping arcs */
#define	NSHORT       020					/* set if node is to be drawn shortened */

/* node instances */
typedef struct Inodeinst
{
	INTBIG               lowx, highx;		/* bounding X box of nodeinst */
	INTBIG               lowy, highy;		/* bounding Y box of nodeinst */
	INTSML               transpose;			/* nonzero to transpose before rotation */
	INTSML               rotation;			/* angle from normal (0 to 359) */
	struct Inodeproto   *proto;				/* current nodeproto */
	struct Inodeproto   *parent;			/* facet that contains this nodeinst */
	struct Inodeinst    *lastnodeinst;		/* list of instances in parent facet */
	struct Inodeinst    *nextnodeinst;
	struct Igeom        *geom;				/* relative geometry list pointer */
	struct Inodeinst    *lastinst;			/* list of others of this type */
	struct Inodeinst    *nextinst;
	struct Iportarcinst *firstportarcinst;	/* first portarcinst on node */
	struct Iportexpinst *firstportexpinst;	/* first portexpinst on node */
	UINTBIG              textdescript;		/* nature of text that displays facet name */

	/* change information */
	char                *changeaddr;		/* change associated with this nodeinst */
	INTBIG               changed;			/* clock entry for changes to this nodeinst */

	/* aid information */
	UINTBIG              userbits;			/* state flags */
	INTBIG               temp1, temp2;		/* temporaries */

	/* variables */
	VARIABLE            *firstvar;			/* first variable in list */
	INTSML               numvar;			/* number of variables in list */
} NODEINST;

/*************************** PORT ARC INSTANCES ****************************/

#define	NOPORTARCINST ((PORTARCINST *)-1)

typedef struct Iportarcinst
{
	struct Iportproto   *proto;				/* portproto of this portarcinst */
	struct Iarcinst     *conarcinst;		/* arcinst connecting to this portarcinst */
	struct Iportarcinst *nextportarcinst;	/* next portarcinst in list */

	/* variables */
	VARIABLE            *firstvar;			/* first variable in list */
	INTSML               numvar;			/* number of variables in list */
} PORTARCINST;

/************************** PORT EXPORT INSTANCES ***************************/

#define	NOPORTEXPINST ((PORTEXPINST *)-1)

typedef struct Iportexpinst
{
	struct Iportproto   *proto;				/* portproto of this portexpinst */
	struct Iportexpinst *nextportexpinst;	/* next portexpinst in list */
	struct Iportproto   *exportproto;		/* portproto on parent facet */

	/* variables */
	VARIABLE            *firstvar;			/* first variable in list */
	INTSML               numvar;			/* number of variables in list */
} PORTEXPINST;

/*************************** NODE PROTOTYPES ****************************/

#define	NONODEPROTO ((NODEPROTO *)-1)

/* the value of NODEPROTO->userbits (bits 17, 18 (01400000) used by database) */
#define	NODESHRINK          01				/* set if nonmanhattan instances shrink */
#define	WANTNEXPAND         02				/* set if instances should be expanded */
#define	NFUNCTION         0374				/* node function (from efunction.h) */
#define	NFUNCTIONSH          2				/* right shift for NFUNCTION */
#define	ARCSHRINK         0400				/* set if nonmanhattan arcs on this shrink */
#define	ARCSWIPE         01000				/* set if instances can be wiped */
#define	NSQUARE          02000				/* set if node is to be kept square in size */
#define	HOLDSTRACE       04000				/* primitive can hold trace information */
#define	REDOFACETNET    010000				/* set to reevaluate this facet's network */
#define	WIPEON1OR2      020000				/* set to erase if connected to 1 or 2 arcs */
#define	LOCKEDPRIM      040000				/* set if primitive is lockable (cannot move) */
#define	NEDGESELECT    0100000				/* set if primitive is selectable by edge, not area */
#define	HASPARAMS      0200000				/* set if parameterizable */

typedef struct Inodeproto
{
	/* used by all aids */
	char               *primname;			/* primitive nodeproto name */
	INTSML              index;				/* nonzero if primitive */
	INTBIG              lowx, highx;		/* bounds in X */
	INTBIG              lowy, highy;		/* bounds in Y */
	NODEINST           *firstinst;			/* first in list of instances of this kind */
	NODEINST           *firstnodeinst;		/* head of list of nodeinsts in this facet */
	struct Iarcinst    *firstarcinst;		/* head of list of arcinsts in this facet */
	struct Itechnology *tech;				/* technology with this nodeproto (prim only) */
	struct Inodeproto  *lastnodeproto;		/* next in library/technology list */
	struct Inodeproto  *nextnodeproto;		/* last in library/technology list */
	struct Iportproto  *firstportproto;		/* list of ports */

	CELL               *cell;				/* cell in which this facet belongs */
	VIEW               *cellview;			/* view of this facet */
	INTSML              version;			/* version number of this facet */
	struct Inodeproto  *lastversion;		/* earlier version of this facet */
	struct Inodeproto  *newestversion;		/* most recent version of this facet */
	struct Inodeproto  *nextincell;			/* next in list of facets in this cell */
	UINTBIG             creationdate;		/* date facet was created */
	UINTBIG             revisiondate;		/* date facet was last changed */

	struct Inetwork    *firstnetwork;		/* nets in this facet */
	struct Irtnode     *rtree;				/* top of geometric tree of objects in facet */

	/* change information */
	char               *changeaddr;			/* change associated with this facet */

	/* aid specific */
	unsigned            adirty;				/* "dirty" bit for each aid */
	UINTBIG             userbits;			/* state flags */
	INTBIG              temp1, temp2;		/* temporaries */

	/* variables */
	VARIABLE           *firstvar;			/* first variable in list */
	INTSML              numvar;				/* number of variables in list */
} NODEPROTO;

/*************************** PORT PROTOTYPES ****************************/

#define	NOPORTPROTO ((PORTPROTO *)-1)

/* the value of PORTPROTO->userbits */
#define	PORTANGLE          0777				/* angle of this port from node center */
#define	PORTANGLESH           0				/* right shift of PORTANGLE field */
#define	PORTARANGE      0377000				/* range of valid angles about port angle */
#define	PORTARANGESH          9				/* right shift of PORTARANGE field */
#define	PORTNET      0177400000				/* electrical net of primitive port (0-31) */
#define	PORTNETSH            17				/* right shift of PORTNET field */
#define	PORTISOLATED 0200000000				/* set if arcs to this port do not connect */
#define	PORTDRAWN    0400000000				/* set if this port should always be drawn */
#define	BODYONLY    01000000000				/* set to exclude this port from the icon */
#define	STATEBITS  036000000000				/* input/output/power/ground/clock state: */
#define	CLKPORT     02000000000				/* un-phased clock port */
#define	C1PORT      04000000000				/* clock phase 1 */
#define	C2PORT      06000000000				/* clock phase 2 */
#define	C3PORT     010000000000				/* clock phase 3 */
#define	C4PORT     012000000000				/* clock phase 4 */
#define	C5PORT     014000000000				/* clock phase 5 */
#define	C6PORT     016000000000				/* clock phase 6 */
#define	INPORT     020000000000				/* input port */
#define	OUTPORT    022000000000				/* output port */
#define	BIDIRPORT  024000000000				/* bidirectional port */
#define	PWRPORT    026000000000				/* power port */
#define	GNDPORT    030000000000				/* ground port */
#define	REFOUTPORT 032000000000				/* bias-level reference output port */
#define	REFINPORT  034000000000				/* bias-level reference input port */

typedef struct Iportproto
{
	struct Iarcproto **connects;			/* arc prototypes that can touch this port */
	NODEPROTO         *parent;				/* nodeproto that this portproto resides in */
	NODEINST          *subnodeinst;			/* subnodeinst that portproto comes from */
	PORTEXPINST       *subportexpinst;		/* portexpinst in subnodeinst */
	struct Iportproto *subportproto;		/* portproto in above subnodeinst */
	struct Iportproto *nextportproto;		/* next in list of port prototypes */
	char              *protoname;			/* name of this port prototype */
	UINTBIG            textdescript;		/* nature of text that displays proto name */

	/* change information */
	char              *changeaddr;			/* change associated with this port */

	/* for the aids */
	UINTBIG            userbits;			/* state flags */
	struct Inetwork   *network;				/* network object within facet */
	INTBIG             temp1, temp2;		/* temporaries */

	/* variables */
	VARIABLE          *firstvar;			/* first variable in list */
	INTSML             numvar;				/* number of variables in list */
} PORTPROTO;

/*************************** ARC INSTANCES ****************************/

#define	NOARCINST ((ARCINST *)-1)

/* the value of ARCINST->userbits (bit 24 (0100000000) used by database) */
#define	FIXED               01				/* fixed-length arc */
#define	FIXANG              02				/* fixed-angle arc */

#define	DEADA              020				/* arc is not in use */
#define	AANGLE          037740				/* angle of arc from end 0 to end 1 */
#define	AANGLESH             5				/* bits of right shift for AANGLE field */
#define	ASHORT          040000				/* set if arc is to be drawn shortened */
#define	NOEXTEND       0400000				/* set if ends do not extend by half width */
#define	ISNEGATED     01000000				/* set if ends are negated */
#define	ISDIRECTIONAL 02000000				/* set if arc aims from end 0 to end 1 */
#define	NOTEND0       04000000				/* no extension/negation/arrows on end 0 */
#define	NOTEND1      010000000				/* no extension/negation/arrows on end 1 */
#define	REVERSEEND   020000000				/* reverse extension/negation/arrow ends */
#define	CANTSLIDE    040000000				/* set if arc can't slide around in ports */

typedef struct Iarcinst
{
	/* physical description of arcinst */
	struct Iarcproto *proto;				/* arc prototype of this arcinst */
	INTBIG            length;				/* length of arcinst */
	INTBIG            width;				/* width of arcinst */
	INTBIG            endshrink;			/* shrinkage factor on ends */
	struct
	{
		INTBIG        xpos, ypos;			/* position of arcinst end */
		NODEINST     *nodeinst;				/* connecting nodeinst */
		PORTARCINST  *portarcinst;			/* portarcinst in the connecting nodeinst */
	} end[2];								/* for each end of the arcinst */
	struct Iarcinst  *lastarcinst;			/* list of arcs in the parent facet */
	struct Iarcinst  *nextarcinst;
	struct Igeom     *geom;					/* geometry entry */
	NODEPROTO        *parent;				/* parent facet */

	/* change information */
	char             *changeaddr;			/* change associated with this nodeinst */
	INTBIG            changed;				/* clock entry for changes to this nodeinst */

	/* aid specific */
	UINTBIG           userbits;				/* state flags */
	struct Inetwork  *network;				/* network object within facet */
	INTBIG            temp1, temp2;			/* temporaries */

	/* variables */
	VARIABLE         *firstvar;				/* first variable in list */
	INTSML            numvar;				/* number of variables in list */
} ARCINST;

/*************************** ARC PROTOTYPES ****************************/

#define	NOARCPROTO ((ARCPROTO *)-1)

/* the value of ARCPROTO->userbits */
#define	WANTFIX               01			/* these arcs are fixed-length */
#define	WANTFIXANG            02			/* these arcs are fixed-angle */
#define	WANTCANTSLIDE         04			/* set if arcs should not slide in ports */
#define	WANTNOEXTEND         010			/* set if ends do not extend by half width */
#define	WANTNEGATED          020			/* set if arcs should be negated */
#define	WANTDIRECTIONAL      040			/* set if arcs should be directional */
#define	CANWIPE             0100			/* set if arcs can wipe wipable nodes */
#define	CANCURVE            0200			/* set if arcs can curve */
#define	AFUNCTION         017400			/* arc function (from efunction.h) */
#define	AFUNCTIONSH            8			/* right shift for AFUNCTION */
#define	AANGLEINC      017760000			/* angle increment for this type of arc */
#define	AANGLEINCSH           13			/* right shift for AANGLEINC */
#define	AEDGESELECT    020000000			/* set if arc is selectable by edge, not area */

typedef struct Iarcproto
{
	/* information for all aids */
	char               *protoname;			/* full arcproto name */
	INTBIG              nominalwidth;		/* default width of arcs */
	INTSML              index;				/* index number of arcproto in technology */
	struct Itechnology *tech;				/* technology this arcproto is in */
	struct Iarcproto   *nextarcproto;		/* next in list for this technology */

	/* information for specific aid */
	UINTBIG             userbits;			/* state flags */
	INTBIG              temp1, temp2;		/* temporaries */

	/* variables */
	VARIABLE           *firstvar;			/* first variable in list */
	INTSML              numvar;				/* number of variables in list */
} ARCPROTO;

/**************************** NETWORKS ****************************/

#define	NONETWORK ((NETWORK *)-1)

typedef struct Inetwork
{
	char             *netname;				/* name of this network */
	INTSML            namecount;			/* number of names */
	INTSML            tempname;				/* nonzero if name is temporary (for back annotation) */
	INTBIG            arcaddr;				/* address of arc(s) on this network */
	INTSML            arccount;				/* number of arcs on this network */
	INTSML            refcount;				/* number of arcs on network */
	INTSML            portcount;			/* number of ports on this network */
	INTSML            buslinkcount;			/* number of busses referencing this network */
	NODEPROTO        *parent;				/* facet that has this network */
	INTSML            signals;				/* width of bus or index into bus */
	struct Inetwork **networklist;			/* list of single-wire networks on bus */
	struct Inetwork  *nextnetwork;			/* next in linked list */
	struct Inetwork  *lastnetwork;			/* previous in linked list */
	INTBIG            temp1, temp2;			/* temporaries */

	/* variables */
	VARIABLE         *firstvar;				/* first variable in list */
	INTSML            numvar;				/* number of variables in list */
} NETWORK;

/**************************** GEOMETRY POINTERS ****************************/

#define	NOGEOM ((GEOM *)-1)

/* the value of GEOM->entrytype */
#define	OBJNODEINST  1						/* entrytype value for nodes */
#define	OBJARCINST   2						/* entrytype value for arcs */

/*
 * Each nodeinst and arcinst points to a geometry module.  The module points
 * back to the nodeinst or arcinst and describes the position relative to
 * other nodes and arcs.  Geometry modules are also at the leaf positions
 * in R-trees, which organize them spatially.
 */
typedef struct Igeom
{
	INTSML         entrytype;				/* OBJNODEINST, OBJARCINST, etc. */
	union          u_entry
	{
		NODEINST  *ni;
		ARCINST   *ai;
		INTBIG     blind;
	} entryaddr;							/* pointer to the element */
	INTBIG         lowx, highx;				/* horizontal bounds */
	INTBIG         lowy, highy;				/* vertical bounds */

	/* variables */
	VARIABLE      *firstvar;				/* first variable in list */
	INTSML         numvar;					/* number of variables in list */
} GEOM;

#define	MINRTNODESIZE	4					/* lower bound on R-tree node size */
#define	MAXRTNODESIZE	(MINRTNODESIZE*2)

#define	NORTNODE ((RTNODE *)-1)

typedef struct Irtnode
{
	INTBIG          lowx, highx;			/* X bounds of this node */
	INTBIG          lowy, highy;			/* Y bounds of this node */
	INTSML          total;					/* number of pointers */
	INTSML          flag;					/* nonzero if children are terminal */
	UINTBIG         pointers[MAXRTNODESIZE];/* pointers */
	struct Irtnode *parent;					/* parent node */

	/* variables */
	VARIABLE       *firstvar;				/* first variable in list */
	INTSML          numvar;					/* number of variables in list */
} RTNODE;

/*************************** LIBRARIES ****************************/

#define	NOLIBRARY ((LIBRARY *)-1)

/* the value of LIBRARY->userbits */
#define	LIBCHANGED     01					/* library has changed */
#define	REDOFACETLIB   02					/* recheck networks in library */
#define	READFROMDISK   04					/* set if library came from disk */
#define LIBUNITS      070					/* internal units in library (see INTERNALUNITS) */
#define LIBUNITSSH      3					/* right shift for LIBUNITS */

typedef struct Ilibrary
{
	NODEPROTO       *firstnodeproto;		/* list of nodeprotos in this library */
	CELL            *firstcell;				/* list of cells in this library */
	char            *libname;				/* name of this library */
	char            *libfile;				/* file this library comes from */
	INTBIG          *lambda;				/* half-millimicrons per unit in this library */
	NODEPROTO       *curnodeproto;			/* top facet of this library (if any) */
	struct Ilibrary *nextlibrary;			/* next library in list */
	UINTBIG          userbits;				/* state flags */
	INTBIG           temp1, temp2;			/* temporaries */
	CLUSTER         *cluster;				/* memory cluster for whole library */

	/* variables */
	VARIABLE        *firstvar;				/* first variable in list */
	INTSML           numvar;				/* number of variables in list */
} LIBRARY;

extern LIBRARY  *el_curlib;					/* pointer to current library (list head) */

/*************************** GRAPHICS and POLYGONS ****************************/

#define	NOGRAPHICS ((GRAPHICS *)-1)

/* the meaning of GRAPHICS->style */
#define	MAXSTYLES    4						/* number of display styles */
#define	COLORMAP     0						/* color display with 4 to 9 bit map */
#define	BWRASTER     1						/* Black&White 1-bit display */
#define	BWVIDEO      2						/* Black&White video display */
#define	COLORNMAP    3						/* color display with no map */

typedef struct
{
	INTSML    bits;							/* bit planes to use (color displays) */
	INTSML    col;							/* color to draw */
	INTSML    style[MAXSTYLES];				/* drawing style for different displays */
	UINTSML   raster[8];					/* 16x8 raster pattern (for stippling) */

	/* variables */
	VARIABLE *firstvar;						/* first variable in list */
	INTSML    numvar;						/* number of variables in list */
} GRAPHICS;


#define	NOPOLYGON ((POLYGON *)-1)

typedef struct Ipolygon
{
	INTBIG          *xv, *yv;				/* the polygon coordinates */
	INTSML           limit;					/* maximum number of points in polygon */
	INTSML           count;					/* current number of points in polygon */
	CLUSTER         *cluster;				/* virtual memory cluster with this polygon */
	INTSML           style;					/* the polygon style */
	char            *string;				/* string (if text graphics) */
	INTSML           font;					/* font type (if text graphics) */
	GRAPHICS        *desc;					/* graphical description of polygon */
	INTSML           layer;					/* layer number of this polygon */
	PORTPROTO       *portproto;				/* port prototype associated with this polygon */
	struct Ipolygon *nextpolygon;			/* for linking them into a list */
} POLYGON;

/******************** EDITOR MODULES ********************/

#define NOEDITOR             ((EDITOR *)-1)

/* editor state (in EDITOR->state) */
#define	EDITORTYPE       07					/* type of editor */
#define	PACEDITOR        01					/* editor is point-and-click */
#define	EMACSEDITOR      02					/* editor is EMACS-like */
#define	EDITORINITED    010					/* set if editor has been initialized */
#define	EGRAPHICSOFF    020					/* set if editor graphics is disabled */
#define	LINESFIXED      040					/* set if editor disallows line insertion/deletion */
#define	TEXTTYPING     0100					/* set if typing into the editor (text undo) */
#define	TEXTTYPED      0200					/* set if typed into the editor (text undo) */

/* an overloaded entry */
#define	charposition highlightedline

typedef struct Ieditor
{
	INTSML    state;						/* status bits (see above) */
	INTSML    swid, shei;					/* screen width and height */
	INTSML    revy;							/* for reversing Y coordinates */
	INTSML    offx;							/* for adjusting X coordinates */
	INTBIG    maxlines;						/* maximum lines in buffer */
	INTBIG    linecount;					/* actual number of lines in buffer */
	INTSML    screenlines;					/* number of lines on screen */
	INTBIG    working;						/* line number being changed */
	INTBIG    firstline;					/* top line on screen */
	INTSML    screenchars;					/* maximum characters across screen */
	INTSML    mostchars;					/* most characters on any line */
	INTBIG    curline, curchar;				/* current start position */
	INTBIG    endline, endchar;				/* current end position */
	INTBIG    highlightedline;				/* line being highlighted (EMACS only) */
	INTSML    horizfactor;					/* position of 1st character on left side */
	INTSML    vthumbpos;					/* Y position of vertical thumb slider */
	INTSML    hthumbpos;					/* X position of horizontal thumb slider */
	INTSML    dirty;						/* nonzero if buffer has changed */
	INTSML   *maxchars;						/* number of characters in each line */
	char     *formerline;					/* line being changed: former state */
	char    **textarray;					/* image of screen contents */
	INTBIG    savedbox;						/* where the bits are kept for popups */
	char     *header;						/* header text */
	char     *editobjaddr;					/* variable editing: parent object address */
	INTBIG    editobjtype;					/* variable editing: parent object type */
	char     *editobjqual;					/* variable editing: attribute name on parent */
	char     *editobjname;					/* variable editing: name of variable */
	VARIABLE *editobjvar;					/* variable editing: actual variable */
	struct Ieditor *nexteditor;				/* next in linked list */
} EDITOR;

/******************** WINDOW MODULES ********************/

#define NOWINDOW             ((WINDOW *)-1)

/* the meaning of WINDOW->state */
#define GRIDON               1				/* grid is to be displayed */
#define GRIDTOOSMALL         2				/* grid is too small to display */
#define PORTLABELS         070				/* bits for port labels */
#define PORTSOFF             0				/* ports not drawn */
#define PORTSCROSS         010				/* ports drawn as crosses */
#define PORTSFULL          020				/* ports drawn as complete text */
#define PORTSSHORT         030				/* ports drawn as shortened text */
#define	WINDOWTYPE        0700				/* type of this window */
#define	DISPWINDOW        0100				/* this is a normal display window */
#define	TEXTWINDOW        0200				/* this is a text editor window */
#define	POPTEXTWINDOW     0300				/* this is a popup text editor window */
#define	PALETTEWINDOW     0400				/* this is a color mixing window */
#define	WAVEFORMWINDOW    0500				/* this is a signal waveform window */
#define	HASOWNFRAME      01000				/* set if window was in its own frame */
#define	WINDOWSIMULATING 02000				/* set if window is simulating */

#define SIMULATINGBORDERSIZE 3				/* size of red border when simulating */

/* the meaning of the second argument to "WINDOW->changehandler" */
#define REPLACETEXTLINE 1
#define DELETETEXTLINE  2
#define INSERTTEXTLINE  3
#define REPLACEALLTEXT  4

typedef struct Iwindow
{
	INTSML          uselx, usehx;			/* X: bounds of window drawing area */
	INTSML          usely, usehy;			/* Y: bounds of window drawing area */
	INTBIG          screenlx,screenhx;		/* X: low and high of window */
	INTBIG          screenly,screenhy;		/* Y: low and high of window */
	INTSML          framelx, framehx;		/* X: bounds of window frame (global coords) */
	INTSML          framely, framehy;		/* Y: bounds of window frame (global coords) */
	float           scalex, scaley;			/* X and Y scale from window to drawing area */
	NODEPROTO      *curnodeproto;			/* facet in window */
	INTBIG          gridx,gridy;			/* size of grid in window */
	INTBIG          state;					/* miscellaneous state bits (see above) */
	char           *location;				/* string describing location */
	void           *frame;					/* real graphics window that contains this */
	EDITOR         *editor;					/* editor running in this window */
	INTSML        (*charhandler)(struct Iwindow*, INTSML);							/* routine for characters in window */
	void          (*buttonhandler)(struct Iwindow*, INTSML, INTSML, INTSML);		/* routine for buttons in window */
	void          (*changehandler)(struct Iwindow*, INTSML, char*, char*, INTBIG);	/* routine for changes to window */
	void          (*termhandler)(struct Iwindow*);									/* routine for termination of window */
	void          (*redisphandler)(struct Iwindow*);								/* routine for redisplay of window */
	struct Iwindow *nextwindow;				/* next window in list */
	struct Iwindow *lastwindow;				/* previous window in list */

	/* variables */
	VARIABLE       *firstvar;				/* first variable in list */
	INTSML          numvar;					/* number of variables in list */
} WINDOW;

extern WINDOW *el_topwindow;				/* top window in list */
extern WINDOW *el_curwindow;				/* current window */

/*************************** TECHNOLOGIES ****************************/

/* ===== LAYER DESCRIPTIONS ===== */

/* definition of a color value */
typedef struct Itech_colormap
{
	INTSML red, green, blue;				/* color values from 0 to 255 */
} TECH_COLORMAP;

/* the value of TECH_POLYGON->representation */
#define	POINTS    0							/* list of scalable points */
#define	BOX       1							/* a rectangle */
#define	ABSPOINTS 2							/* list of absolute points */
#define	MINBOX    3							/* minimum sized rectangle */

/* the structure for layers of a node prototype */
typedef struct Itech_polygon
{
	INTSML  layernum;						/* drawing layer in technology */
	INTSML  portnum;						/* the port number in this technology */
	INTSML  count;							/* number of points in polygon */
	INTSML  style;							/* polygon style */
	INTSML  representation;					/* see above list */
	INTBIG *points;							/* data list */
} TECH_POLYGON;

/* ===== ARC DESCRIPTIONS ===== */

/* the structure for layers of an arc prototype */
typedef struct Itech_arclay
{
	INTSML lay;								/* index of this layer */
	INTBIG off;								/* width offset for this layer */
	INTSML style;							/* polygon style */
} TECH_ARCLAY;

/* the structure for an arc prototype */
typedef struct Itech_arcs
{
	char        *arcname;					/* layer name */
	INTBIG       arcwidth;					/* default layer width */
	INTSML       arcindex;					/* index of this arcinst */
	INTSML       laycount;					/* number of layers */
	TECH_ARCLAY *list;						/* list of layers that compose arc */
	UINTBIG      initialbits;				/* initial userbits for this arcproto */
} TECH_ARCS;

/* ===== PORT DESCRIPTIONS ===== */

/* the structure for ports of a node prototype */
typedef struct Itech_ports
{
	INTBIG     *portarcs;					/* allowable arcs (list ends with -1) */
	char      *protoname;					/* name of this port */
	PORTPROTO *addr;						/* address used by later routines */
	UINTBIG    initialbits;					/* initial userbits for this port */
	INTSML     lowxmul,  lowxsum;			/* defines low X of portinst area */
	INTSML     lowymul,  lowysum;			/* defines low Y of portinst area */
	INTSML     highxmul, highxsum;			/* defines high X of portinst area */
	INTSML     highymul, highysum;			/* defines high Y of portinst area */
} TECH_PORTS;

/* ===== NODE DESCRIPTIONS ===== */

/* the structure for serpentine MOS transistor description */
typedef struct Itech_serpent
{
	TECH_POLYGON basics;					/* the basic information */
	INTSML       lwidth;					/* the extension of width on the left */
	INTSML       rwidth;					/* the extension of width on the right */
	INTSML       extend;					/* the extension of length on the ends */
} TECH_SERPENT;

/* the value of TECH_NODES->special */
#define	SERPTRANS     1						/* serpentine transistor */
#define	POLYGONAL     2						/* polygonally defined transistor */
#define	MULTICUT      3						/* multi-cut contact */
#define	MOSTRANS      4						/* MOS transistor (nonserpentine) */

/* the structure for a node prototype */
typedef struct Itech_nodes
{
	char         *nodename;
	INTSML        index;
	NODEPROTO    *creation;
	INTBIG        xsize, ysize;
	INTSML        portcount;
	TECH_PORTS   *portlist;
	INTSML        layercount;
	TECH_POLYGON *layerlist;
	INTBIG        initialbits;
	INTSML        special;
	INTSML        f1, f2, f3, f4;
	TECH_SERPENT *gra, *ele;
} TECH_NODES;

/* ===== VARIABLE DESCRIPTIONS ===== */

/* the structure for a variable */
typedef struct Itech_variables
{
	char *name;								/* attribute name to set */
	char  *value;							/* initial (address/integer) value for attribute */
	float  fvalue;							/* initial (floating) value for attribute */
	INTBIG type;							/* initial type of attribute */
} TECH_VARIABLES;

/* ===== MAIN STRUCTURE ===== */

#define	NOTECHNOLOGY ((TECHNOLOGY *)-1)

/* the value of TECHNOLOGY->userbits */
#define	NONELECTRICAL      01				/* technology is not electrical  */
#define	NODIRECTIONALARCS  02				/* has no directional arcs */
#define	NONEGATEDARCS      04				/* has no negated arcs */
#define	NONSTANDARD       010				/* nonstandard technology (cannot be edited) */

/* scaling factor for fixed-point numbers (used in technologies) */
#define	WHOLE   120

typedef struct Itechnology
{
	char               *techname;			/* name of this technology */
	INTSML              index;				/* 0-based index of this technology */
	INTBIG              deflambda;			/* the default size of a unit */
	NODEPROTO          *firstnodeproto;		/* list of primitive nodeprotos */
	ARCPROTO           *firstarcproto;		/* pointer to type description */
	VARIABLE           *firstvar;			/* first variable in list */
	INTSML              numvar;				/* number of variables in list */
	COMCOMP            *parse;				/* parsing structure for tech direction */
	CLUSTER            *cluster;			/* virtual memory cluster for technology */
	char               *techdescript;		/* description of this technology */

	INTSML              layercount;			/* number of layers */
	GRAPHICS          **layers;				/* layer descriptions */
	INTSML              arcprotocount;		/* number of arcs */
	TECH_ARCS         **arcprotos;			/* raw data for arcs */
	INTSML              nodeprotocount;		/* number of nodes */
	TECH_NODES        **nodeprotos;			/* raw data for nodes */
	TECH_VARIABLES     *variables;			/* variable descriptions */

	INTSML            (*init)(struct Itechnology*, INTSML);			/* process initialization */
	void              (*setmode)(INTSML, char *[]);					/* set operation mode */
	void              (*paramnode)(NODEINST*);						/* top-down parameterize */

	/* node description */
	INTSML            (*nodepolys)(NODEINST*);						/* returns total real polygons in node */
	INTSML            (*nodeEpolys)(NODEINST*);						/* returns total electrical polys in node */
	void              (*shapenodepoly)(NODEINST*, INTSML, POLYGON*);	/* returns real polygon shape in node */
	void              (*shapeEnodepoly)(NODEINST*, INTSML, POLYGON*);	/* returns electrical polygon shape in node */

	/* node port description */
	void              (*shapeportpoly)(NODEINST*, PORTPROTO*, POLYGON*, XARRAY, INTSML);/* returns port shape on a node */

	/* arc description */
	INTSML            (*arcpolys)(ARCINST*);						/* returns total polygons in arc */
	void              (*shapearcpoly)(ARCINST*, INTSML, POLYGON*);	/* returns polygon shape in arc */
	struct Itechnology *nexttechnology;		/* next in linked list */
	UINTBIG             userbits;			/* state flags */
	INTBIG              temp1, temp2;		/* temporaries */
} TECHNOLOGY;

extern TECHNOLOGY *el_technologies;			/* defined in "tectable.c" */
extern TECHNOLOGY *el_curtech;				/* pointer to current technology */
extern INTSML      el_maxtech;				/* current number of technologies */

/*************************** CONSTRAINT SYSTEMS ****************************/

#define	NOCONSTRAINT ((CONSTRAINT *)-1)

typedef struct Iconstraint
{
	char     *conname;						/* constraint system name */
	char     *condesc;						/* constraint system description */
	COMCOMP  *parse;						/* parsing structure for constraint direction */
	CLUSTER  *cluster;						/* virtual memory cluster for constraint */
	VARIABLE *firstvar;						/* first variable in list */
	INTSML    numvar;						/* number of variables in list */
	void    (*init)(struct Iconstraint*);	/* initialize a constraint system */
	void    (*term)(void);					/* terminate a constraint system */
	void    (*setmode)(INTSML, char*[]);	/* user-command options */
	INTBIG  (*request)(char*, INTBIG);		/* direct call options */
	void    (*solve)(NODEPROTO*);			/* solve a batch of changes */

	void    (*newobject)(INTBIG,INTBIG);				/* constraint newly created object */
	void    (*killobject)(INTBIG,INTBIG);				/* constraint deleted object */
	INTSML  (*setobject)(INTBIG,INTBIG,INTBIG,INTBIG);	/* set constraint properties on object */

	void    (*modifynodeinst)(NODEINST*,INTBIG,INTBIG,INTBIG,INTBIG,INTSML,INTSML);		/* constrain a modified node instance */
	void    (*modifyarcinst)(ARCINST*, INTBIG, INTBIG, INTBIG, INTBIG, INTBIG, INTBIG);	/* constrain a modified arc instance */
	void    (*modifyportproto)(PORTPROTO*, NODEINST*, PORTPROTO*);					/* constrain a modified port prototype */
	void    (*modifynodeproto)(NODEPROTO*);											/* constrain a modified node prototype */
	void    (*modifydescript)(VARIABLE*, INTBIG);									/* constrain a modified descriptor */

	void    (*newlib)(LIBRARY*);			/* constrain a newly created library */
	void    (*killlib)(LIBRARY*);			/* constrain a deleted library */

	void    (*newvariable)(INTBIG, INTBIG, INTBIG, INTBIG);						/* constrain a newly created variable */
	void    (*killvariable)(INTBIG, INTBIG, INTBIG, INTBIG, INTBIG, INTBIG);	/* constrain a deleted variable */
	void    (*modifyvariable)(INTBIG, INTBIG, INTBIG, INTBIG, INTBIG, INTBIG);	/* constrain a modified array variable entry */
	void    (*insertvariable)(INTBIG, INTBIG, INTBIG, INTBIG);					/* constrain an inserted array variable entry */
	void    (*deletevariable)(INTBIG, INTBIG, INTBIG, INTBIG, INTBIG);			/* constrain a deleted array variable entry */
	void    (*setvariable)(void);												/* set constraint properties on variable */
} CONSTRAINT;

extern CONSTRAINT *el_curconstraint;		/* current constraint solver */
extern CONSTRAINT  el_constraints[];		/* list of constraint solvers */

/*************************** MISCELLANEOUS ****************************/

#define EPI          3.14159265				/* Pi */
#define	HUGEINT      0x7FFFFFFF				/* largest possible integer */
#define	NOSTRING     ((char *)-1)

/* the meaning of "el_units" */
#define DISPLAYUNITS       017				/**** the display units ****/
#define	DISPUNITLAMBDA       0				/* lambda display (12.3) */
#define DISPUNITINCH        01				/* inch display (12.3") */
#define DISPUNITCM          02				/* centimeter display (12.3cm) */
#define DISPUNITMM          04				/* millimeter display (12.3mm) */
#define DISPUNITMIL        010				/* mil display (12.3mil) */
#define DISPUNITMIC        011				/* micron display (12.3u) */
#define DISPUNITCMIC       012				/* centimicron display (12.3cu) */
#define DISPUNITMMIC       013				/* millimicron display (12.3mu) */
#define INTERNALUNITS     0160				/**** the internal units ****/
#define INTERNALUNITSSH      4				/* right shift for INTERNALUNITS */
#define INTUNITHMMIC         0				/* half-millimicron units */
#define INTUNITHDMIC       020				/* half-decimicron units */

/*
 * the file types for "xopen" and "xcreate"
 * (update table "dbtext.c:db_filetypeinfo[]" when changing this)
 */
#define FILETYPE           077				/* type of file */
#define FILETYPEACTELTAB    01				/* Actel tables */
#define FILETYPEBLIB        02				/* Binary library */
#define FILETYPECADRC       03				/* "cadrc" startup */
#define FILETYPECIF         04				/* CIF */
#define FILETYPECOLORMAP    05				/* Color map */
#define FILETYPEDRAC        06				/* Dracula */
#define FILETYPEDXF         07				/* DXF */
#define FILETYPEEDIF       010				/* EDIF */
#define FILETYPEESIM       011				/* ESIM netlist */
#define FILETYPEFPGA       012				/* FPGA architecture */
#define FILETYPEGDS        013				/* GDS */
#define FILETYPEHELP       014				/* Help */
#define FILETYPEHPGL       015				/* HPGL */
#define FILETYPEL          016				/* L */
#define FILETYPELOG        017				/* Log */
#define FILETYPEMACRO      020				/* Macro */
#define FILETYPEMOSSIM     021				/* MOSSIM netlist */
#define FILETYPENETLISP    022				/* Netlisp netlist */
#define FILETYPENEWS       023				/* News */
#define FILETYPEPAL        024				/* PAL netlist */
#define FILETYPEPLATAB     025				/* PLA table */
#define FILETYPEPROJ       026				/* Project */
#define FILETYPEPROJUSERS  027				/* Project users */
#define FILETYPEPS         030				/* PostScript */
#define FILETYPEALS        031				/* ALS netlist */
#define FILETYPEALSVEC     032				/* ALS vectors */
#define FILETYPEQUISC      033				/* QUISC netlist */
#define FILETYPERSIM       034				/* RSIM netlist */
#define FILETYPESCTAB      035				/* Silicon compiler table */
#define FILETYPESCSIM      036				/* Silicon compiler simulation */
#define FILETYPESDF        037				/* SDF */
#define FILETYPESILOS      040				/* Silos netlist */
#define FILETYPESPICE      041				/* Spice */
#define FILETYPESPICEOUT   042				/* Spice output */
#define FILETYPETEGAS      043				/* Tegas netlist */
#define FILETYPETEGASTAB   044				/* Tegas table */
#define FILETYPETEXT       045				/* Plain text */
#define FILETYPETLIB       046				/* Text library */
#define FILETYPEVERILOG    047				/* Verilog */
#define FILETYPEVHDL       050				/* VHDL */
#define FILETYPEWRITE     0100				/* bit set to write file (instead of read) */
#define FILETYPEAPPEND    0200				/* bit set to append to file (instead of read) */

/* the "purpose" parameter (cursor shape) for "trackcursor()" */
#define	TRACKNORMAL          0				/* standard cursor */
#define	TRACKDRAWING         1				/* pen cursor */
#define	TRACKDRAGGING        2				/* hand cursor */
#define	TRACKSELECTING       3				/* horizontal arrow cursor */
#define	TRACKHSELECTING      4				/* arrow cursor */

extern XARRAY el_matid;						/* identity matrix */
extern INTSML el_pleasestop;				/* nonzero if abort is requested */
extern INTSML el_units;						/* display and internal units */
extern char  *el_libdir;					/* pointer to library directory */
extern char  *el_lispdir;					/* pointer to lisp directory */
extern char  *el_tcldir;					/* pointer to Tcl directory */
extern char  *el_tkdir;						/* pointer to Tk directory */
extern char  *el_version;					/* current version string */

/*************************** DECLARATIONS ****************************/

#ifdef __cplusplus
extern "C"
{
#endif

/* node routines */
NODEINST *newnodeinst(NODEPROTO *proto, INTBIG lowx, INTBIG highx, INTBIG lowy, INTBIG highy,
	INTSML transposition, INTSML rotation, NODEPROTO *parent);
INTSML killnodeinst(NODEINST *node);
void modifynodeinst(NODEINST *node, INTBIG Dlx, INTBIG Dly, INTBIG Dhx,
	INTBIG Dhy, INTSML Drotation, INTSML Dtransposition);
NODEINST *replacenodeinst(NODEINST *oldnode, NODEPROTO *newproto);
INTSML nodepolys(NODEINST *node);
void shapenodepoly(NODEINST *node, INTSML box, POLYGON *poly);
INTSML nodeEpolys(NODEINST *node);
void shapeEnodepoly(NODEINST *node, INTSML box, POLYGON *poly);
void nodesizeoffset(NODEPROTO *node, INTBIG *lowx, INTBIG *lowy, INTBIG *highx, INTBIG *highy);
INTSML nodefunction(NODEINST *node, char **extra);
char *nodefunctionname(INTSML function, NODEINST *ni);
char *describenodeinst(NODEINST *node);
void transistorsize(NODEINST *node, INTBIG *length, INTBIG *width);
void getarcdegrees(NODEINST *ni, double *startoffset, double *endangle);
void setarcdegrees(NODEINST *ni, double startoffset, double endangle);
void getarcendpoints(NODEINST *ni, double startoffset, double endangle, INTBIG *fx, INTBIG *fy,
	INTBIG *tx, INTBIG *ty);
void transistorports(NODEINST *node, PORTPROTO **gateleft, PORTPROTO **gateright,
	PORTPROTO **activetop, PORTPROTO **activebottom);
VARIABLE *gettrace(NODEINST *node);
void nodearea(NODEINST *node, PORTPROTO *port, INTBIG *lowx, INTBIG *highx, INTBIG *lowy,
	INTBIG *highy);
void corneroffset(NODEINST *node, NODEPROTO *proto, INTSML rotation, INTSML transpose,
	INTBIG *xoff, INTBIG *yoff);
NODEINST *dummynode(void);
INTSML isfet(GEOM *pos);

/* arc routines */
ARCINST *newarcinst(ARCPROTO *proto, INTBIG width, INTBIG initialuserbits, NODEINST *nodeA,
	PORTPROTO *portA, INTBIG xA, INTBIG yA, NODEINST *nodeB, PORTPROTO *portB, INTBIG xB,
	INTBIG yB, NODEPROTO *parent);
INTSML killarcinst(ARCINST *arc);
INTSML modifyarcinst(ARCINST *arc, INTBIG Dwidth, INTBIG Dx1, INTBIG Dy1, INTBIG Dx2, INTBIG Dy2);
ARCINST *aconnect(GEOM *fromgeom, PORTPROTO *fromport, GEOM *togeom, PORTPROTO *toport,
	ARCPROTO *arc, INTBIG x, INTBIG y, ARCINST **arc2, ARCINST **arc3, INTSML ang);
ARCINST *replacearcinst(ARCINST *oldarc, ARCPROTO *newproto);
INTSML arcpolys(ARCINST *arc);
void shapearcpoly(ARCINST *arc, INTSML box, POLYGON *poly);
char *arcfunctionname(INTSML function);
char *describearcinst(ARCINST *arc);
char *describearcproto(ARCPROTO *arcproto);
ARCPROTO *getarcproto(char *specification);
INTBIG arcwidthoffset(ARCPROTO *arc);
NODEPROTO *getpinproto(ARCPROTO *arc);
void makearcpoly(INTBIG length, INTBIG width, ARCINST *arc, POLYGON *poly, INTSML style);
INTSML curvedarcoutline(ARCINST *arc, POLYGON *poly, INTSML style, INTBIG width);
void setshrinkvalue(ARCINST *arc);
void determineangle(ARCINST *arc);
INTSML arcconnects(INTSML ang, INTBIG lx1, INTBIG hx1, INTBIG ly1, INTBIG hy1, INTBIG lx2, INTBIG hx2,
	INTBIG ly2, INTBIG hy2, INTBIG *x1, INTBIG *y1, INTBIG *x2, INTBIG *y2);
ARCINST *dummyarc(void);

/* facet routines */
NODEPROTO *newnodeproto(char *name, LIBRARY *library);
INTSML killnodeproto(NODEPROTO *facet);
NODEPROTO *copynodeproto(NODEPROTO *originalfacet, LIBRARY *destlibrary, char *name);
NODEPROTO *getnodeproto(char *specification);
char *describenodeproto(NODEPROTO *nodeproto);
char *nldescribenodeproto(NODEPROTO *nodeproto);
CELL *getcell(char *specification);
void grabpoint(NODEPROTO *nodeproto, INTBIG *grabx, INTBIG *graby);
INTSML isachildof(NODEPROTO *parent, NODEPROTO *child);
INTSML framesize(INTBIG *x, INTBIG *y, NODEPROTO *np);
INTSML framepolys(NODEPROTO *np);
void framepoly(INTSML box, POLYGON *poly, NODEPROTO *np);
void begintraversehierarchy(void);
void downhierarchy(NODEINST *ni);
void uphierarchy(void);
void gettraversalpath(NODEINST ***nilist, INTSML *depth);
NODEPROTO *getcurfacet(void);

/* port routines */
PORTPROTO *newportproto(NODEPROTO *facet, NODEINST *nodeinfacet, PORTPROTO *portinnode, char *name);
INTSML killportproto(NODEPROTO *facet, PORTPROTO *port);
INTSML moveportproto(NODEPROTO *facet, PORTPROTO *port, NODEINST *newnode, PORTPROTO *newnodeport);
PORTPROTO *getportproto(NODEPROTO *facet, char *portname);
void shapeportpoly(NODEINST *node, PORTPROTO *port, POLYGON *poly, INTSML purpose);
void shapetransportpoly(NODEINST *node, PORTPROTO *port, POLYGON *poly, XARRAY trans);
void portposition(NODEINST *node, PORTPROTO *port, INTBIG *x, INTBIG *y);
void reduceportpoly(POLYGON *poly, NODEINST *node, PORTPROTO *port, INTBIG width);
PORTPROTO *equivalentport(NODEPROTO *facet, PORTPROTO *port, NODEPROTO *equivalentfacet);
INTSML portispower(PORTPROTO *port);
INTSML portisground(PORTPROTO *port);
void changeallports(PORTPROTO *port);

/* view routines */
VIEW *newview(char *viewname, char *abbreviation);
INTSML killview(VIEW *view);
INTSML changefacetview(NODEPROTO *facet, VIEW *view);
VIEW *getview(char *specification);
NODEPROTO *contentsview(NODEPROTO *facet);
NODEPROTO *iconview(NODEPROTO *facet);
NODEPROTO *layoutview(NODEPROTO *facet);
NODEPROTO *vhdlview(NODEPROTO *facet);

/* window routines */
WINDOW *newewindow(char *location, WINDOW *protowindow, INTSML newframe);
void killwindow(WINDOW *window);
void copywindow(WINDOW *dwindow, WINDOW *swindow);
INTSML truefontsize(INTSML font, WINDOW *window, TECHNOLOGY *tech);
void computewindowscale(WINDOW *window);
INTBIG applyxscale(WINDOW *window, INTBIG value);
INTBIG applyyscale(WINDOW *window, INTBIG value);

/* variable routines */
VARIABLE *getval(INTBIG addr, INTBIG  type, INTBIG want, char *name);
VARIABLE *setval(INTBIG addr, INTBIG type, char *name, INTBIG newaddr, INTBIG newtype);
INTSML delval(INTBIG addr, INTBIG type, char *name);
INTSML getind(INTBIG addr, INTBIG type, char *name, INTBIG index, INTBIG *value);
INTSML setind(INTBIG addr, INTBIG type, char *name, INTBIG index, INTBIG newaddr);
INTSML insind(INTBIG addr, INTBIG type, char *name, INTBIG index, INTBIG newaddr);
INTSML delind(INTBIG addr, INTBIG type, char *name, INTBIG index);
INTSML renameval(char *oldname, char *newname);
INTBIG makekey(char *name);
char *makename(INTBIG key);
VARIABLE *getvalkey(INTBIG addr, INTBIG type, INTBIG want, INTBIG key);
VARIABLE *setvalkey(INTBIG addr, INTBIG type, INTBIG key, INTBIG newaddr, INTBIG newtype);
INTSML getindkey(INTBIG addr, INTBIG type, INTBIG key, INTBIG index, INTBIG *value);
INTSML setindkey(INTBIG addr, INTBIG type, INTBIG key, INTBIG index, INTBIG newaddr);
INTSML insindkey(INTBIG addr, INTBIG type, INTBIG key, INTBIG index, INTBIG newaddr);
INTSML delindkey(INTBIG addr, INTBIG type, INTBIG key, INTBIG index);
INTSML delvalkey(INTBIG addr, INTBIG type, INTBIG key);
void nextvarchangequiet(void);
INTBIG initobjlist(INTBIG addr, INTBIG type, INTSML restrict);
char *nextobjectlist(VARIABLE **var, INTBIG search);
INTSML copyvars(INTBIG fromaddr, INTBIG fromtype, INTBIG toaddr, INTBIG totype);
void modifydescript(VARIABLE *var, INTBIG newdescript);
INTBIG getlength(VARIABLE *var);
char *describevariable(VARIABLE *var, INTSML index, INTSML purpose);

/* change control routines */
void startobjectchange(INTBIG address, INTBIG type);
void endobjectchange(INTBIG address, INTBIG type);
INTSML undonature(INTSML undo);
INTSML undoabatch(AIDENTRY **aid);
INTSML redoabatch(AIDENTRY **aid);
void noundoallowed(void);
void noredoallowed(void);
INTSML historylistsize(INTSML newsize);
void setactivity(char *message);
void showhistorylist(INTSML batch);
char *changedvariablename(INTBIG type, INTBIG key, INTBIG subtype);
INTBIG getcurrentbatchnumber(void);

/* layer routines */
char *layername(TECHNOLOGY *tech, INTSML layer);
INTBIG layerfunction(TECHNOLOGY *tech, INTSML layer);
INTBIG maxdrcsurround(TECHNOLOGY *tech, INTSML layer);
INTBIG drcmindistance(TECHNOLOGY *tech, INTSML layer1, INTSML layer2, INTSML connected);
INTSML getecolor(char *colorname);
INTSML ecolorname(INTSML color, char **colorname, char **colorsymbol);

/* geometric routines */
INTBIG initsearch(INTBIG lowx, INTBIG highx, INTBIG lowy, INTBIG highy, NODEPROTO *facet);
GEOM *nextobject(INTBIG initsearchvalue);
void termsearch(INTBIG initsearchvalue);
void boundobj(GEOM *object, INTBIG *lowx, INTBIG *highx, INTBIG *lowy, INTBIG *highy);
char *geomname(GEOM *geom);
NODEPROTO *geomparent(GEOM *geom);
void linkgeom(GEOM *geom, NODEPROTO *facet);
void undogeom(GEOM *geom, NODEPROTO *facet);
void updategeom(GEOM *geom, NODEPROTO *facet);
INTSML geomstructure(NODEPROTO *facet);

/* library routines */
LIBRARY *getlibrary(char *libname);
LIBRARY *newlibrary(char *name, char *diskfile);
void selectlibrary(LIBRARY *library);
void killlibrary(LIBRARY *library);
void eraselibrary(LIBRARY *library);

/* technology routines */
TECHNOLOGY *gettechnology(char *techname);
TECHNOLOGY *whattech(NODEPROTO *facet);
INTSML addtechnology(TECHNOLOGY *technology);
INTSML killtechnology(TECHNOLOGY *technology);
void registertechnologycache(void (*routine)(void));

/* aid routines */
void aidturnon(AIDENTRY *aid, INTSML nocatchup);
void aidturnoff(AIDENTRY *aid, INTSML permanently);
INTSML tellaid(AIDENTRY *aid, INTSML count, char *args[]);
INTBIG askaid(AIDENTRY *aid, char *command, ...);
AIDENTRY *getaid(char *aidname);
INTSML checkcap(AIDENTRY *aid, INTSML bit);

/* transformations and mathematics */
void makeangle(INTSML rotation, INTSML transposition, XARRAY matrix);
INTBIG sine(INTSML angle);
INTBIG cosine(INTSML angle);
void makerot(NODEINST *node, XARRAY matrix);
void makerotI(NODEINST *node, XARRAY matrix);
void maketrans(NODEINST *node, XARRAY matrix);
void maketransI(NODEINST *node, XARRAY matrix);
void transid(XARRAY matrix);
void transcpy(XARRAY matsource, XARRAY matdest);
void transmult(XARRAY mata, XARRAY matb, XARRAY matdest);
INTSML ismanhattan(XARRAY trans);
void xform(INTBIG x, INTBIG y, INTBIG *newx, INTBIG *newy, XARRAY matrix);
INTBIG muldiv(INTBIG a, INTBIG b, INTBIG c);
INTBIG mult(INTBIG a, INTBIG b);
INTBIG mini(INTBIG a, INTBIG b);
INTBIG maxi(INTBIG a, INTBIG b);
INTBIG roundfloat(float v);
INTBIG rounddouble(double v);
INTBIG castint(float f);
float castfloat(INTBIG i);
INTBIG intsqrt(INTBIG v);
INTBIG scalefromdispunit(float value, INTSML dispunit);
float scaletodispunit(INTBIG value, INTSML dispunit);
float scaletodispunitsq(INTBIG value, INTSML dispunit);
INTBIG computedistance(INTBIG x1, INTBIG y1, INTBIG x2, INTBIG y2);
INTSML figureangle(INTBIG x1, INTBIG y1, INTBIG x2, INTBIG y2);
INTSML intersect(INTBIG x1, INTBIG y1, INTBIG angle1, INTBIG x2, INTBIG y2, INTBIG angle2,
	INTBIG *x, INTBIG *y);
INTBIG closestpointtosegment(INTBIG x1, INTBIG y1, INTBIG x2, INTBIG y2, INTBIG *x, INTBIG *y);
INTSML findcenters(INTBIG r, INTBIG x1, INTBIG y1, INTBIG x2, INTBIG y2, INTBIG d,
	INTBIG *ix1, INTBIG *iy1, INTBIG *ix2, INTBIG *iy2);
INTSML circlelineintersection(INTBIG icx, INTBIG icy, INTBIG isx, INTBIG isy, INTBIG lx1,
	INTBIG ly1, INTBIG lx2, INTBIG ly2, INTBIG *ix1, INTBIG *iy1, INTBIG *ix2, INTBIG *iy2,
	INTBIG tolerance);
INTSML circletangents(INTBIG x, INTBIG y, INTBIG cx, INTBIG cy, INTBIG sx, INTBIG sy,
	INTBIG *ix1, INTBIG *iy1, INTBIG *ix2, INTBIG *iy2);
void arcbbox(INTBIG xs, INTBIG ys, INTBIG xe, INTBIG ye, INTBIG xc, INTBIG yc,
	INTBIG *lx, INTBIG *hx, INTBIG *ly, INTBIG *hy);
INTSML clipline(INTBIG *fx, INTBIG *fy, INTBIG *tx, INTBIG *ty, INTBIG lx, INTBIG hx,
	INTBIG ly, INTBIG hy);
void cliparc(POLYGON *in, INTBIG lx, INTBIG hx, INTBIG ly, INTBIG hy);
void subrange(INTBIG low, INTBIG high, INTBIG lowmul, INTBIG lowsum, INTBIG highmul, INTBIG highsum,
	INTBIG *newlow, INTBIG *newhigh, INTBIG lambda);
INTSML polyinrect(POLYGON *poly, INTBIG lx, INTBIG hx, INTBIG ly, INTBIG hy);
void clippoly(POLYGON *poly, INTBIG lx, INTBIG hx, INTBIG ly, INTBIG hy);
void closestpoint(POLYGON *poly, INTBIG *x, INTBIG *y);
INTBIG getrange(INTBIG low, INTBIG high, INTBIG mul,INTBIG sum, INTBIG lambda);
INTSML rotatelabel(INTSML oldstyle, XARRAY trans);
INTSML cropbox(INTBIG *lx, INTBIG *hx, INTBIG *ly, INTBIG *hy, INTBIG bx, INTBIG ux, INTBIG by,
	INTBIG uy);
void arctopoly(INTBIG cx, INTBIG cy, INTBIG sx, INTBIG sy, INTBIG ex, INTBIG ey, POLYGON *poly,
	INTBIG arcres, INTBIG arcsag);
void circletopoly(INTBIG cx, INTBIG cy, INTBIG radius, POLYGON *poly, INTBIG arcres, INTBIG arcsag);
void adjustdisoffset(GEOM *geom, POLYGON *poly, INTBIG textdescription);

/* box merging routines */
void mrginit(void);
void mrgterm(void);
void mrgstorebox(INTSML layer, TECHNOLOGY *tech, INTBIG length, INTBIG width, INTBIG xc, INTBIG yc);
void mrgdonefacet(void (*setlayer)(INTSML layernum, TECHNOLOGY *tech),
	void (*writepolygon)(INTBIG *x, INTBIG *y, INTSML count, float area));

/* polygon routines */
POLYGON *allocpolygon(INTSML points, CLUSTER *cluster);
void freepolygon(POLYGON *poly);
INTSML extendpolygon(POLYGON *poly, INTSML newcount);
INTSML polysame(POLYGON *poly1, POLYGON *poly2);
void polycopy(POLYGON *dpoly, POLYGON *spoly);
INTSML isinside(INTBIG x, INTBIG y, POLYGON *poly);
INTBIG areapoly(POLYGON *poly);
void xformpoly(POLYGON *poly, XARRAY trans);
INTSML isbox(POLYGON *poly, INTBIG *xl, INTBIG *xh, INTBIG *yl, INTBIG *yh);
void maketruerect(POLYGON *poly);
INTBIG polydistance(POLYGON *poly, INTBIG x, INTBIG y);
void makerectpoly(INTBIG lx, INTBIG ux, INTBIG ly, INTBIG uy, POLYGON *poly);
void maketruerectpoly(INTBIG lx, INTBIG ux, INTBIG ly, INTBIG uy, POLYGON *poly);
void getcenter(POLYGON *poly, INTBIG *x, INTBIG *y);
void getbbox(POLYGON *poly, INTBIG *lx, INTBIG *hx, INTBIG *ly, INTBIG *hy);
INTSML polysplithoriz(INTBIG yl, POLYGON *which, POLYGON **abovep, POLYGON **belowp);
INTSML polysplitvert(INTBIG xl, POLYGON *which, POLYGON **leftp, POLYGON **rightp);

/* string manipulation routines */
INTSML allocstring(char **addr, char *str, CLUSTER *cluster);
INTSML reallocstring(char **addr, char *str, CLUSTER *cluster);
char *latoa(INTBIG value);
INTBIG atola(char *str);
char *frtoa(INTBIG value);
INTBIG atofr(char *str);
INTBIG myatoi(char *str);
INTSML isanumber(char *str);
char *makeabbrev(char *str, INTSML upper);
INTSML namesame(char *str1, char *str2);
INTSML namesamen(char *str1, char *str2, INTSML length);
INTSML stringmatch(char *str1, char *str2);
INTSML parse(char *keyword, COMCOMP *list, INTSML noise);
INTSML initinfstr(void);
INTSML addtoinfstr(char c);
INTSML addstringtoinfstr(char *str);
char *returninfstr(void);
void *newstringarray(CLUSTER *cluster);
void killstringarray(void *sa);
void clearstrings(void *sa);
void addtostringarray(void *sa, char *string);
void stringarraytotextfacet(void *sa, NODEPROTO *np, INTSML permanent);
char **getstringarray(void *sa, INTBIG *count);
char *getkeyword(char **ptin, char *brk);
char tonextchar(char **ptin);
char seenextchar(char **ptin);
INTBIG defaulttextdescript(void);
char *unitsname(INTSML units);
void myencrypt(char *text, char *key);

/* language interface routines */
char *languagename(void);
INTSML languageconverse(INTBIG language);
INTSML doquerry(char *code, INTBIG language, INTBIG type, INTBIG *retval);
INTSML loadcode(char *program, INTBIG language);

/* memory allocation routines */
/* #define DEBUGMEMORY 1 */			/* uncomment to debug memory */
#ifdef DEBUGMEMORY
  INTBIG *_emalloc(INTBIG size, CLUSTER *cluster, char*, INTBIG);
# define emalloc(x, c) _emalloc((x), (c), (char *)__FILE__, (INTBIG)__LINE__)
#else
  INTBIG *emalloc(INTBIG size, CLUSTER *cluster);
#endif
void efree(char *addr);
CLUSTER *alloccluster(char *name);
void freecluster(CLUSTER *cluster);
void allocnodeinst(INTBIG count, NODEINST **nilist, CLUSTER *cluster);
void freenodeinst(NODEINST *ni);
void allocarcinst(INTBIG count, ARCINST **ailist, CLUSTER *cluster);
void freearcinst(ARCINST *ai);
void allocportproto(INTBIG count, PORTPROTO **pplist, CLUSTER *cluster);
void freeportproto(PORTPROTO *pp);
void allocportarcinst(INTBIG count, PORTARCINST **pilist, CLUSTER *cluster);
void freeportarcinst(PORTARCINST *pi);
void allocportexpinst(INTBIG count, PORTEXPINST **pelist, CLUSTER *cluster);
void freeportexpinst(PORTEXPINST *pe);
NODEPROTO *allocnodeproto(CLUSTER *cluster);
void freenodeproto(NODEPROTO *np);
CELL *alloccell(CLUSTER *cluster);
void freecell(CELL *cell);
VIEW *allocview(void);
void freeview(VIEW *view);
void allocgeom(INTBIG count, GEOM **geomlist, CLUSTER *cluster);
void freegeom(GEOM *geom);
RTNODE *allocrtnode(CLUSTER *cluster);
void freertnode(RTNODE *rtnode);
LIBRARY *alloclibrary(void);
void freelibrary(LIBRARY *lib);
TECHNOLOGY *alloctechnology(CLUSTER *cluster);
void freetechnology(TECHNOLOGY *tech);
void freewindow(WINDOW *win);

/* terminal output routines */
void ttynewcommand(void);
void ttyputmsg(char *format, ...);
void ttyputmsgf(char *format, ...);
void ttyputerr(char *format, ...);
void error(char *format, ...);
INTSML ttyquiet(INTSML flag);
void ttybeep(void);
INTSML ttysaveoutput(INTSML state);
void telldatabaseerror(void);

/* terminal input routines */
INTSML ttygetchar(void);
char *ttygetline(char *prompt);
char *ttygetlinemessages(char *prompt);
INTSML ttygetparam(char *prompt, COMCOMP *parameter, INTSML keycount, char *paramstart[]);
INTSML ttygetfullparam(char *prompt, COMCOMP *parameter, INTSML keycount, char *paramstart[]);
INTSML ttydataready(void);
void checkforinterrupt(void);

/* mouse/tablet input routines */
INTSML getxy(INTBIG *x, INTBIG *y);
void gridalign(INTBIG *xcur, INTBIG *ycur, INTBIG alignment);
INTSML buttoncount(void);
char *buttonname(INTSML but, INTSML *important);
INTSML doublebutton(INTSML but);
INTSML shiftbutton(INTSML but);
void waitforbutton(INTSML *x, INTSML *y, INTSML *but);
void trackcursor(INTSML waitforpush, INTSML (*whileup)(INTBIG x, INTBIG y), void (*whendown)(void),
	INTSML (*eachdown)(INTBIG x, INTBIG y), INTSML (*eachchar)(INTBIG x, INTBIG y, INTSML chr),
	void (*done)(void), INTSML purpose);
void readtablet(INTSML *x, INTSML *y);
void stoptablet(void);
UINTBIG ticktime(void);

/* disk file routines */
char *truepath(char *path);
INTSML fileexistence(char *file);
char *currentdirectory(void);
INTSML createdirectory(char *directory);
INTSML filesindirectory(char *directory, char ***filelist);
INTSML lockfile(char *file);
void unlockfile(char *file);
UINTBIG getcurrenttime(void);
char *timetostring(UINTBIG time);
void parsetime(UINTBIG time, INTSML *year, INTSML *month, INTSML *mday);
UINTBIG filedate(char *file);
INTBIG filesize(FILE *stream);
void describefiletype(INTSML filetype, char **extension, char **filter, char **shortname, char **longname);
INTSML getfiletype(char *shortname);
FILE *xopen(char *file, INTSML filetype, char *otherdir, char **truename);
FILE *xcreate(char *name, INTSML filetype, char *prompt, char **truename);
FILE *xappend(char *file);
void xclose(FILE *stream);
void xflushbuf(FILE *stream);
INTSML xeof(FILE *stream);
void xseek(FILE *stream, INTBIG pos, INTSML nature);
INTBIG xtell(FILE *f);
void xprintf(FILE *stream, char *format, ...);
INTSML xgetc(FILE *stream);
void xungetc(char chr, FILE *stream);
void xputc(char chr, FILE *stream);
INTSML xputs(char *str, FILE *stream);
INTBIG xfread(char *str, INTBIG size, INTBIG count, FILE *stream);
INTBIG xfwrite(char *str, INTBIG size, INTBIG count, FILE *stream);
INTSML xfgets(char *str, INTSML limit, FILE *stream);

/* miscellaneous routines */
void changelambda(INTBIG oldlam, INTBIG lam, TECHNOLOGY *tech);
INTBIG figurelambda(GEOM *geom);
void changeinternalunits(LIBRARY *whichlib, INTSML oldunits, INTSML newunits);
NETWORK *getnetwork(char *specification, NODEPROTO *facet);
NETWORK *getnetonport(NODEINST *node, PORTPROTO *port);
char *describenetwork(NETWORK *net);
void securitycheck(void);
void bringdown(void);
INTSML stopping(char *cause);
void waitfor(INTBIG process);
INTBIG myfork(void);
void flushscreen(void);
void exitprogram(void);
void starttimer(void);
float endtimer(void);
void gotosleep(INTBIG ticks);

/* graphics (not documented in Internals Manual) */
void us_setdisplay(char *name, INTBIG *argc, char **argv);
void us_unsetdisplay(void);
void setupenvironment(void);
void getmachineid(char *name);
INTSML us_initgraphics(INTSML messages);
void us_termgraphics(INTSML final);
void *us_newwindowframe(INTSML floating);
void us_killwindowframe(void*);
void us_getpaletteparameters(INTSML *wid, INTSML *hei, INTSML *palettewidth);
void us_sizewindowframe(void *frame, INTSML wid, INTSML hei);
void us_movewindowframe(void *frame, INTSML top, INTSML left);
void *us_getwindowframe(void);
void *us_firstwindowframe(void);
void *us_nextwindowframe(void *frame);
void us_getwindowsize(void *frame, INTSML *wid, INTSML *hei);
short us_getdispstyle(WINDOW *win);
void us_startbatching(void);
void us_endbatching(void);
void us_setspy(INTSML mag);
INTSML us_graphicshas(INTSML want);
void gra_set_cursorstate(INTSML state);
char *gra_getstring(char *prompt);
void gra_putstring(char *str);
INTSML us_setnormalcursor(INTSML curs);
void us_loadmap(INTBIG *red, INTBIG *green, INTBIG *blue, INTSML low, INTSML high);
void us_drawline(WINDOW *win, INTSML x1, INTSML y1, INTSML x2, INTSML y2, GRAPHICS *desc,
	INTSML texture);
void us_invertline(WINDOW *win, INTSML x1, INTSML y1, INTSML x2, INTSML y2);
void us_drawpolygon(WINDOW *win, INTBIG *x, INTBIG *y, INTSML count, GRAPHICS *desc);
void us_drawbox(WINDOW *win, INTSML lowx, INTSML highx, INTSML lowy, INTSML highy, GRAPHICS *desc);
void us_invertbox(WINDOW *win, INTSML lowx, INTSML highx, INTSML lowy, INTSML highy);
void us_movebox(WINDOW *win, INTSML sx, INTSML sy, INTSML wid, INTSML hei, INTSML dx, INTSML dy);
INTBIG us_savebox(WINDOW *win, INTSML lx, INTSML hx, INTSML ly, INTSML hy);
void us_movesavedbox(INTBIG code, INTSML dx, INTSML dy);
INTSML us_restorebox(INTBIG code, INTSML destroy);
void us_settextsize(WINDOW *win, INTSML fnt);
void us_textsize(WINDOW *win, char *str, INTSML *x, INTSML *y);
void us_puttext(WINDOW *win, INTSML atx, INTSML aty, char *s, GRAPHICS *desc);
void us_drawcircle(WINDOW *win, INTBIG atx, INTBIG aty, INTBIG radius, GRAPHICS *desc);
void us_drawdisc(WINDOW *win, INTBIG atx, INTBIG aty, INTBIG radius, GRAPHICS *desc);
void us_drawcirclearc(WINDOW *win, INTBIG centerx, INTBIG centery, INTBIG p1x, INTBIG p1y,
	INTBIG p2x, INTBIG p2y, GRAPHICS *desc);
void us_drawgrid(WINDOW *win, POLYGON *obj);
INTSML us_getnxtchar(void);
INTSML us_logplayback(char *file, INTSML all);
void us_logstartrecord(void);
void us_logfinishrecord(void);
char *gra_fileselect(char*, INTSML, char*);

/* terminal I/O (not documented in Internals Manual) */
void tty_close(void);
char *nextaids(void);
char *nextcells(void);
char *nextfile(void);
char *nextlibs(void);
char *nextnets(void);
char *nexttechs(void);
char *nextviews(void);
char *us_nextparse(void);
void requiredextension(char *extension);
INTSML topofaids(char **chrpos);
INTSML topofcells(char **chrpos);
INTSML topoffile(char **chrpos);
INTSML topoflibfile(char **chrpos);
INTSML topoflibs(char **chrpos);
INTSML topofnets(char **chrpos);
INTSML topoftechs(char **chrpos);
INTSML topofviews(char **chrpos);
INTSML us_topoffacets(char **chrpos);
INTSML us_topofarcs(char **chrpos);
INTSML us_patoplist(char **chrpos);
INTSML us_paparams(char *word, COMCOMP *arr[], char breakc);
void us_pabackupto(INTSML a, char *b[]);
char *us_panextinlist(void);

#ifdef __cplusplus
}
#endif
