/*
 * Electric(tm) VLSI Design System
 *
 * File: simals.h
 * Header file for asynchronous logic simulator
 * From algorithms by: Brent Serbin and Peter J. Gallant 
 * Last maintained by: Steven M. Rubin
 *
 * Copyright (c) 1998 Electric Editor Incorporated.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Electric Editor Incorporated
 * 23470 Sunset Drive, Suite 108
 * Los Gatos, California 95033
 * support@electriceditor.com
 */

typedef enum {DELAY_MIN=0, DELAY_TYP, DELAY_MAX} DELAY_TYPES;

/* The trace buffer in ALS can currently hold 10000 events.
 * More events can be accommodated in the trace buffer by
 * increasing its size.
 */
#define TRACE_SIZE	10000

typedef struct model_ds
{
	INTBIG              num;
	char               *name;
	char                type;
	char               *ptr;	/* may be CONPTR or ROWPTR or FUNCPTR */
	struct export_ds   *exptr;
	struct io_ds       *setptr;
	struct load_ds     *loadptr;
	char                fanout;
	INTSML              priority;
	struct model_ds    *next;
    char               *level;  /* hierarchical level */
} MODEL;
typedef MODEL *MODPTR;

typedef struct row_ds
{
	struct io_ds   *inptr;
	struct io_ds   *outptr;
	float           delta;
	float           linear;
	float           exp;
	float           random;
    float           abs;    /* BA delay - SDF absolute port delay */
	struct row_ds  *next;
    char           *delay;  /* delay transition name (01, 10, etc) */
} ROW;
typedef ROW *ROWPTR;

typedef struct io_ds
{
	struct node_ds *nodeptr;
	unsigned char   operatr;
	char           *operand;
	INTSML          strength;
	struct io_ds   *next;
} IO;
typedef IO *IOPTR;

typedef struct connect_ds
{
	char               *inst_name;
	char               *model_name;
	struct export_ds   *exptr;
	struct connect_ds  *parent;
	struct connect_ds  *child;
	struct connect_ds  *next;
	struct channel_ds  *display_page;  /* pointer to the display page */
	INTSML              num_chn;       /* number of exported channels in this level */
	INTSML	            level_plot;    /* flag to determine if included in monster plot */
	INTSML	            level_mixed;   /* flag to determine if included in mixed signal display */
	INTSML	            mix_flag;      /* flag is this is the [MIXED_SIGNAL_LEVEL] */
} CONNECT;
typedef CONNECT *CONPTR;

typedef struct export_ds
{
	char              *node_name;
	struct node_ds    *nodeptr;
	struct export_ds  *next;
    INTBIG            td[12];  /* transition delays */
} EXPORT;
typedef EXPORT *EXPTR;

typedef struct load_ds
{
	char            *ptr;
	float            load;
	struct load_ds  *next;
} LOAD;
typedef LOAD *LOADPTR;

typedef struct func_ds
{
	void             (*procptr)(MODPTR);
	struct export_ds  *inptr;
	float              delta;
	float              linear;
	float              exp;
	float              abs;    /* absolute delay for back annotation */
	float              random;
	char              *userptr;
	INTBIG             userint;
	float              userfloat;
} FUNC;
typedef FUNC *FUNCPTR;

typedef struct node_ds
{
	struct connect_ds  *cellptr;
	INTBIG              num;
	INTBIG              sum_state;
	INTSML              sum_strength;
	INTBIG              new_state;
	INTSML              new_strength;
	INTBIG              plot_node;
	struct stat_ds     *statptr;
	struct load_ds     *pinptr;
	float               load;
	INTBIG              visit;
	INTBIG              maxsize;
	INTBIG              arrive;
	INTBIG              depart;
	float               tk_sec;
	float               t_last;
	struct node_ds     *next;
} NODE;
typedef NODE *NODEPTR;

typedef struct stat_ds
{
	struct model_ds  *primptr;
	struct node_ds   *nodeptr;
	INTBIG            new_state;
	INTSML            new_strength;
	unsigned char     sched_op;
	INTBIG            sched_state;
	INTSML            sched_strength;
	struct stat_ds   *next;
} STAT;
typedef STAT *STATPTR;

typedef struct link_ds
{
	struct link_ds  *left;
	struct link_ds  *right;
	struct link_ds  *up;
	struct link_ds  *down;
	char            *ptr;
	char             type;
	unsigned char    operatr;
	INTBIG           state;
	INTSML           strength;
	INTSML           priority;
	float            time;
} LINK;
typedef LINK *LINKPTR;

typedef struct trak_ds
{
	struct node_ds  *ptr;
	INTBIG           state;
	INTSML           strength;
	float            time;
} TRAK;
typedef TRAK *TRAKPTR;

typedef struct channel_ds
{
	char             name[15];
	struct node_ds  *nodeptr;
	INTBIG           displayptr;
} CHANNEL;
typedef CHANNEL *CHNPTR;

/*
 * Now come all the global variables declared extern for reference by other files
 */
extern MODPTR      simals_modroot, simals_modptr2, simals_primroot;
extern ROWPTR      simals_rowptr2;
extern IOPTR       simals_ioptr2;
extern CONPTR      simals_conptr2, simals_levelptr, simals_cellroot;
extern EXPTR       simals_exptr2;
extern NODEPTR     simals_noderoot, simals_drive_node;
extern LINKPTR     simals_linkfront, simals_linkback, simals_setroot;
extern TRAKPTR     simals_trakroot;
extern LOADPTR     simals_chekroot;
extern char      **simals_rowptr1, **simals_ioptr1;
extern char       *simals_instbuf;
extern INTBIG      simals_loop_count, simals_pseq, simals_nseq, *simals_instptr,
				   simals_ibufsize, simals_iptrsize, simals_trakfull, simals_trakptr;
extern INTBIG    **simals_poolroot, simals_poolsize, simals_poolptr;
extern INTBIG      simals_trace;
extern INTSML      simals_seed_flag;
extern float       simals_time_abs, simals_delta_def,
				   simals_linear_def, simals_exp_def, simals_random_def, simals_abs_def;
extern NODEPROTO  *simals_mainproto;

/* prototypes for intratool interface */
INTSML simals_set_current_level(void);
float simals_initialize_simulator(void);
void simals_convert_to_upper(char*);
NODEPTR simals_find_node(char*);
LINKPTR simals_alloc_link_mem(void);
char *simals_alloc_mem(INTBIG);
char **simals_insert_set_list(LINKPTR);
INTBIG simals_atoi(char*);
INTBIG simals_trans_state_to_number(char*);
INTSML simals_free_link_mem(LINKPTR);
CONPTR simals_find_level(char*);
void simals_trans_number_to_state(INTBIG, char*);
void simals_compute_path_name(CONPTR, char*);
INTSML simals_fragment_command(char*);
void simals_compute_node_name(NODEPTR, char*);
void simals_build_actel_command(INTSML, char*[]);
void simals_clock_command(INTSML, char*[]);
void simals_erase_model(void);
void simals_go_command(INTSML, char*[]);
void simals_level_command(INTSML, char*[]);
void simals_help_command(void);
void simals_print_command(INTSML, char*[]);
void simals_seed_command(INTSML, char*[]);
void simals_set_command(INTSML, char*[]);
void simals_trace_command(INTSML, char*[]);
void simals_vector_command(INTSML, char*[]);
INTBIG *simals_get_function_address(char*);
void simals_fill_display_arrays(void);
INTSML simals_alloc_ptr_buffer(INTBIG**, INTBIG*, INTBIG);
void simals_insert_link_list(LINKPTR);
INTSML sim_alstopofinstances(char**);
char *sim_alsnextinstance(void);
void simals_init(void);
void simals_init_display(void);
void simals_com_comp(INTSML, char*[10]);
INTSML simals_read_net_desc(NODEPROTO*);
INTSML simals_flatten_network(void);
void simals_annotate_command(INTSML, char*[]);
void simals_order_command(INTSML, char*[]);
