/*
 * Electric(tm) VLSI Design System
 *
 * File: io.c
 * Input/output aid: controller module
 * Written by: Steven M. Rubin, Electric Editor Incorporated
 *
 * Copyright (c) 1998 Electric Editor Incorporated.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Electric Editor Incorporated
 * 23470 Sunset Drive, Suite 108
 * Los Gatos, California 95033
 * support@electriceditor.com
 */

#include "global.h"
#include "database.h"
#include "dbcontour.h"
#include "eio.h"
#include "usredtec.h"
#include "edialogs.h"
#include "tecart.h"
#include "tecschem.h"
#include <ctype.h>
#include <math.h>

/* the command parsing table */
static KEYWORD iocifoopt[] =
{
	{"fully-instantiated",    0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"exactly-as-displayed",  0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"individual-boxes",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"merge-boxes",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"include-cloak-layer",   0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"ignore-cloak-layer",    0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"centered",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"not-centered",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP iocifop = {iocifoopt,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "style of CIF output", 0};
static KEYWORD iocifiopt[] =
{
	{"rounded-wires",    0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"squared-wires",    0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP iocifip = {iocifiopt,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "style of CIF input", 0};
static KEYWORD iocifopt[] =
{
	{"input",     1,{&iocifip,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"output",    1,{&iocifop,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP iocifp = {iocifopt,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "control of CIF", 0};
static KEYWORD ioplotnopt[] =
{
	{"focus",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"include-date",0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP ioplotnp = {ioplotnopt,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "negating control of plot output", 0};
static KEYWORD ioplotopt[] =
{
	{"not",         1,{&ioplotnp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"focus",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"include-date",0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP ioplotp = {ioplotopt,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "control of plot output", 0};

static COMCOMP iogdsoarp = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "maximum number of degrees per arc segment", 0};
static COMCOMP iogdsoasp = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "maximum sag distance for an arc segment", 0};
static KEYWORD iogdsoopt[] =
{
	{"individual-boxes",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"merge-boxes",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"include-cloak-layer",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"ignore-cloak-layer",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"arc-resolution",           1,{&iogdsoarp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"arc-sag",                  1,{&iogdsoasp,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP iogdsop = {iogdsoopt,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "style of GDS output", 0};
static KEYWORD iogdsopt[] =
{
	{"output",   1,{&iogdsop,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP iogdsp = {iogdsopt,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "control of GDS", 0};
static KEYWORD iopostopt[] =
{
	{"plain",            0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"encapsulated",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP iopostp = {iopostopt,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "control of PostScript format", 0};
static KEYWORD ioveropt[] =
{
	{"off",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"on",               0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"graphical",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP ioverp = {ioveropt,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "verbose option", 0};
static COMCOMP iohpgl2sp = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "internal units per pixel", 0};
static KEYWORD iohpgl2opt[] =
{
	{"scale",           1,{&iohpgl2sp,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP iohpgl2p = {iohpgl2opt,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "HPGL/2 scaling option", 0};
static KEYWORD iohpglopt[] =
{
	{"1",               0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"2",               1,{&iohpgl2p,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP iohpglp = {iohpglopt,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "HPGL version", 0};
static KEYWORD ioedifopt[] =
{
	{"schematic",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"netlist",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP ioedifp = {ioedifopt,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "EDIF output option", 0};
static KEYWORD iodxfaopt[] =
{
	{"all",                  0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"restrict",             0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP iodxfap = {iodxfaopt,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "DXF layer restriction options", 0};
static KEYWORD iodxfopt[] =
{
	{"acceptable-layers",   1,{&iodxfap,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"flatten-input",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"not-flatten-input",   0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP iodxfp = {iodxfopt,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "DXF options", 0};
static COMCOMP iobloatlp = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "Layer name to bloat", 0};
static COMCOMP iobloatap = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	0, " \t", "amount to bloat layer (in internal units)", 0};
static KEYWORD iosdfopt[] =
{
    {"verbose",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
    {"noverbose",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
    TERMKEY
};
static COMCOMP iosdfp = {iosdfopt,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
    INPUTOPT, " \t", "SDF input option", 0};
static KEYWORD ioopt[] =
{
	{"cif",            1,{&iocifp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"dxf",            1,{&iodxfp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"gds",            1,{&iogdsp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"plot",           1,{&ioplotp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"verbose",        1,{&ioverp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"postscript",     1,{&iopostp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"hpgl",           1,{&iohpglp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"edif",           1,{&ioedifp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"sdf",            1,{&iosdfp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"bloat-output",   2,{&iobloatlp,&iobloatap,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP io_iop = {ioopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
		0, " \t", "Input/Output action", 0};

struct
{
	char *name;
	INTSML required, bits;
} io_formatlist[] =
{
	{"binary",     1, FBINARY},
#if IOCIF
	{"cif",        1, FCIF},
#endif
#if IODXF
	{"dxf",        1, FDXF},
#endif
#if IOEDIF
	{"edif",       1, FEDIF},
#endif
#if IOGDS
	{"gds",        1, FGDS},
#endif
#if IOHPGL
	{"hpgl",       1, FHPGL},
#endif
#if IOL
	{"l",          1, FL},
#endif
#if IOPOSTSCRIPT
	{"postscript", 1, FPOSTSCRIPT},
#endif
#if IOQUICKDRAW
	{"quickdraw",  1, FQUICKDRAW},
#endif
#if IOSDF
    {"sdf",        1, FSDF},
#endif
	{"text",       1, FTEXT},
#if VHDLAID
	{"vhdl",       1, FVHDL},
#endif
	{NULL, 0, 0}  /* 0 */
};

#define	NOBLOAT ((BLOAT *)-1)

typedef struct Ibloat
{
	char  *layer;
	INTBIG  amount;
	struct Ibloat *nextbloat;
} BLOAT;

BLOAT *io_curbloat = NOBLOAT;

/* miscellaneous */
FILE        *io_fileout;		/* channel for output */
FILE        *io_filein;			/* channel for input */
char        *io_namesave;		/* for saving the global namespace */
INTBIG      *io_newnames;		/* for adding new names to global namespace */
jmp_buf      io_filerror;		/* nonlocal jump when I/O fails */
INTSML       io_cifbase;		/* index used when writing CIF */
INTBIG       io_state;			/* key for "IO_state" */
INTSML       io_verbose;		/* 0: silent  1:chattier  -1:graphical */
NODEINST   **io_nodelist;		/* list of nodes for readin */
INTBIG      *io_nodecount;		/* number of nodeinsts in each facet */
ARCINST    **io_arclist;		/* list of arcs for readin */
INTBIG      *io_arccount;		/* number of arcinsts in each facet */
NODEPROTO  **io_nodeprotolist;	/* list of facets for readin */
CELL       **io_celllist;		/* list of cells for readin */
TECHNOLOGY **io_techlist;		/* list of technologies for readin */
INTBIG      *io_aidlist;		/* list of aids for readin */
PORTPROTO  **io_portprotolist;	/* list of portprotos for readin */
INTBIG      *io_portcount;		/* number of portprotos in each facet */
PORTPROTO  **io_portpprotolist;	/* list of primitive portprotos for readin */
NODEPROTO  **io_nodepprotolist;	/* list of primitive nodeprotos for readin */
ARCPROTO   **io_arcprotolist;	/* list of arcprotos for readin */
AIDENTRY    *io_aid;			/* the I/O aid object */

/* prototypes for local routines */
void io_fixrtree(RTNODE*);
void io_convert_serplib(LIBRARY*);
INTSML io_isserp(NODEINST*, PORTPROTO**, PORTPROTO**);
void io_gdsarcconversiondlg(void);
double io_calc_angle(double r, double dx, double dy);
void io_compute_center(INTBIG xc, INTBIG yc, INTBIG x1, INTBIG y1,
	INTBIG x2, INTBIG y2, INTBIG *cx, INTBIG *cy);

/* dummy routines for the I/O aid */
void io_examinenodeproto(NODEPROTO *np) {}

void io_slice(void)
{
	ttyputmsg("Input and Output are performed with the 'library' command");
	ttyputmsg("...I/O aid turned off");
	aidturnoff(io_aid, 0);
}

void io_done(void) {}
void io_startbatch(AIDENTRY *source) {}
void io_endbatch(void) {}
void io_startobjectchange(INTBIG addr, INTBIG type) {}
void io_endobjectchange(INTBIG addr, INTBIG type) {}
void io_modifynodeinst(NODEINST *ni, INTBIG olx, INTBIG oly, INTBIG ohx, INTBIG ohy, INTSML orot, INTSML otran) {}
void io_modifyarcinst(ARCINST *ai, INTBIG oxA, INTBIG oyA, INTBIG oxB, INTBIG oyB, INTBIG owid, INTBIG olen) {}
void io_modifyportproto(PORTPROTO *pp, NODEINST *oni, PORTPROTO *opp) {}
void io_modifynodeproto(NODEPROTO *np) {}
void io_modifydescript(VARIABLE *var, INTBIG odes) {}
void io_newobject(INTBIG addr, INTBIG type) {}
void io_killobject(INTBIG addr, INTBIG type) {}
void io_newvariable(INTBIG addr, INTBIG type, INTBIG key, INTBIG newtype) {}
void io_killvariable(INTBIG addr, INTBIG type, INTBIG key, INTBIG oldaddr, INTBIG oldtype, INTBIG olddescript) {}
void io_modifyvariable(INTBIG addr, INTBIG type, INTBIG key, INTBIG vartype, INTBIG index, INTBIG oldvalue) {}
void io_insertvariable(INTBIG addr, INTBIG type, INTBIG key, INTBIG vartype, INTBIG index) {}
void io_deletevariable(INTBIG addr, INTBIG type, INTBIG key, INTBIG vartype, INTBIG index, INTBIG oldvalue) {}
void io_readlibrary(LIBRARY *lib) {}
void io_eraselibrary(LIBRARY *lib) {}
void io_writelibrary(LIBRARY *lib, INTSML pass) {}

void io_init(INTBIG  *argc, char *argv[], AIDENTRY *thisaid)
{
	/* nothing for pass 2 or 3 initialization */
	if (thisaid == NOAID || thisaid == 0) return;

	/* pass 1 initialization */
	io_aid = thisaid;
	io_state = makekey("IO_state");
	nextvarchangequiet();
	(void)setvalkey((INTBIG)io_aid, VAID, io_state, DXFFLATTENINPUT, VINTEGER|VDONTSAVE);
	io_verbose = 0;
}

INTSML io_set(INTSML count, char *par[])
{
	REGISTER INTSML l;
	REGISTER INTBIG curstate, scale;
	REGISTER char *pp;
	REGISTER VARIABLE *var;
	INTBIG arcres, arcsag;

	if (count == 0)
	{
		count = ttygetparam("IO option:", &io_iop, MAXPARS, par);
		if (count == 0)
		{
			ttyputerr("Aborted");
			return(1);
		}
	}
	l = strlen(pp = par[0]);

	/* get current state of I/O aid */
	var = getvalkey((INTBIG)io_aid, VAID, VINTEGER, io_state);
	if (var == NOVARIABLE) curstate = 0; else curstate = var->addr;

	/* check for bloating specifications */
	if (namesamen(pp, "bloat-output", l) == 0)
	{
		if (count <= 1)
		{
			io_setoutputbloat("", 0);
			return(0);
		}
		if (count < 3)
		{
			ttyputerr("Usage: tellaid io bloat-output LAYER AMOUNT");
			return(1);
		}
		io_setoutputbloat(par[1], myatoi(par[2]));
		return(0);
	}

	/* check for format specifications */
	if (namesamen(pp, "plot", l) == 0)
	{
		if (count <= 1)
		{
			if ((curstate&PLOTFOCUS) != 0)
				ttyputmsg("Plot output will focus on highlighted area"); else
					ttyputmsg("Plot output will display entire facet");
			if ((curstate&PLOTDATES) != 0)
				ttyputmsg("Plot output will include dates"); else
					ttyputmsg("Plot output will not include dates");
			return(0);
		}

		l = strlen(pp = par[1]);
		if (namesamen(pp, "not", l) == 0)
		{
			if (count <= 2)
			{
				ttyputerr("Usage: tellaid io plot not OPTION");
				return(1);
			}
			l = strlen(pp = par[2]);
			if (namesamen(pp, "focus", l) == 0)
			{
				(void)setvalkey((INTBIG)io_aid, VAID, io_state, curstate & ~PLOTFOCUS, VINTEGER|VDONTSAVE);
				ttyputmsgf("Plot output will display entire facet");
				return(0);
			}
			if (namesamen(pp, "include-date", l) == 0)
			{
				(void)setvalkey((INTBIG)io_aid, VAID, io_state, curstate & ~PLOTDATES, VINTEGER|VDONTSAVE);
				ttyputmsgf("Plot output will not include dates");
				return(0);
			}
			ttyputerr("Bad TELLAID IO PLOT NOT option");
			return(1);
		}
		if (namesamen(pp, "focus", l) == 0)
		{
			(void)setvalkey((INTBIG)io_aid, VAID, io_state, curstate | PLOTFOCUS, VINTEGER|VDONTSAVE);
			ttyputmsgf("Plot output will focus on highlighted area");
			return(0);
		}
		if (namesamen(pp, "include-date", l) == 0)
		{
			(void)setvalkey((INTBIG)io_aid, VAID, io_state, curstate | PLOTDATES, VINTEGER|VDONTSAVE);
			ttyputmsgf("Plot output will include dates");
			return(0);
		}
		ttyputerr("Usage: tellaid io plot [not] (focus | include-dates)");
		return(1);
	}

#if IOCIF
	if (namesamen(pp, "cif", l) == 0)
	{
		if (count <= 1)
		{
			if ((curstate&CIFINSQUARE) == 0)
				ttyputmsg("CIF wires will have rounded ends"); else
					ttyputmsg("CIF wires will have squared ends");
			if ((curstate&CIFOUTEXACT) == 0)
				ttyputmsg("CIF generation will include all facets"); else
					ttyputmsg("CIF generation will duplicate screen");
			if ((curstate&CIFOUTMERGE) == 0)
				ttyputmsg("CIF generation will list individual boxes"); else
					ttyputmsg("CIF generation will merge boxes");
			if ((curstate&CIFOUTADDDRC) == 0)
				ttyputmsg("CIF generation will ignore DRC cloak layer"); else
					ttyputmsg("CIF generation will include DRC cloak layer");
			if ((curstate&CIFOUTNOTCEN) == 0)
				ttyputmsg("CIF generation will put origins at center"); else
					ttyputmsg("CIF generation will put origins at lower corner");
			return(0);
		}
		l = strlen(pp = par[1]);
		if (namesamen(pp, "input", l) == 0)
		{
			if (count <= 2)
			{
				if ((curstate&CIFINSQUARE) == 0)
					ttyputmsg("CIF wires will have rounded ends"); else
						ttyputmsg("CIF wires will have squared ends");
				return(0);
			}
			l = strlen(pp = par[2]);
			switch (*pp)
			{
				case 'r':
					(void)setvalkey((INTBIG)io_aid, VAID, io_state,
						curstate & ~CIFINSQUARE, VINTEGER|VDONTSAVE);
					ttyputmsgf("CIF wires will have rounded ends");
					break;
				case 's':
					(void)setvalkey((INTBIG)io_aid, VAID, io_state,
						curstate | CIFINSQUARE, VINTEGER|VDONTSAVE);
					ttyputmsgf("CIF wires will have squared ends");
					break;
				default:
					ttyputerr("Bad TELLAID IO CIF INPUT option");
			}
			return(0);
		}
		if (namesamen(pp, "output", l) == 0)
		{
			if (count <= 2)
			{
				if ((curstate&CIFOUTEXACT) == 0)
					ttyputmsg("CIF generation will include all facets"); else
						ttyputmsg("CIF generation will duplicate screen");
				if ((curstate&CIFOUTMERGE) == 0)
					ttyputmsg("CIF generation will list individual boxes"); else
						ttyputmsg("CIF generation will merge boxes");
				if ((curstate&CIFOUTADDDRC) == 0)
					ttyputmsg("CIF generation will ignore DRC cloak layer"); else
						ttyputmsg("CIF generation will include DRC cloak layer");
				if ((curstate&CIFOUTNOTCEN) == 0)
					ttyputmsg("CIF generation will put origins at center"); else
						ttyputmsg("CIF generation will put origins at lower corner");
				return(0);
			}
			l = strlen(pp = par[2]);
			if (namesamen(pp, "fully-instantiated", l) == 0)
			{
				(void)setvalkey((INTBIG)io_aid, VAID, io_state,
					curstate & ~CIFOUTEXACT, VINTEGER|VDONTSAVE);
				ttyputmsgf("CIF generation will include all facets");
				return(0);
			}
			if (namesamen(pp, "exactly-as-displayed", l) == 0)
			{
				(void)setvalkey((INTBIG)io_aid, VAID, io_state,
					curstate | CIFOUTEXACT, VINTEGER|VDONTSAVE);
				ttyputmsgf("CIF generation will duplicate screen");
				return(0);
			}
			if (namesamen(pp, "merge-boxes", l) == 0)
			{
				(void)setvalkey((INTBIG)io_aid, VAID, io_state,
					curstate | CIFOUTMERGE, VINTEGER|VDONTSAVE);
				ttyputmsgf("CIF generation will merge boxes");
				return(0);
			}
			if (namesamen(pp, "individual-boxes", l) == 0 && l >= 3)
			{
				(void)setvalkey((INTBIG)io_aid, VAID, io_state,
					curstate & ~CIFOUTMERGE, VINTEGER|VDONTSAVE);
				ttyputmsgf("CIF generation will list individual boxes");
				return(0);
			}
			if (namesamen(pp, "include-cloak-layer", l) == 0 && l >= 3)
			{
				(void)setvalkey((INTBIG)io_aid, VAID, io_state,
					curstate | CIFOUTADDDRC, VINTEGER|VDONTSAVE);
				ttyputmsgf("CIF generation will include DRC cloak layer");
				return(0);
			}
			if (namesamen(pp, "ignore-cloak-layer", l) == 0 && l >= 2)
			{
				(void)setvalkey((INTBIG)io_aid, VAID, io_state,
					curstate & ~CIFOUTADDDRC, VINTEGER|VDONTSAVE);
				ttyputmsgf("CIF generation will ignore DRC cloak layer");
				return(0);
			}
			if (namesamen(pp, "not-centered", l) == 0 && l >= 1)
			{
				(void)setvalkey((INTBIG)io_aid, VAID, io_state,
					curstate | CIFOUTNOTCEN, VINTEGER|VDONTSAVE);
				ttyputmsgf("CIF generation will put origins at lower corner");
				return(0);
			}
			if (namesamen(pp, "centered", l) == 0 && l >= 1)
			{
				(void)setvalkey((INTBIG)io_aid, VAID, io_state,
					curstate & ~CIFOUTNOTCEN, VINTEGER|VDONTSAVE);
				ttyputmsgf("CIF generation will put origins in center");
				return(0);
			}
		}
		ttyputerr("Usage: tellaid io cif input|output OPTION");
		return(1);
	}
#endif

#if IOGDS
	if (namesamen(pp, "gds", l) == 0)
	{
		if (count <= 1)
		{
			if ((curstate&GDSOUTMERGE) == 0)
				ttyputmsg("GDS generation will list individual boxes"); else
					ttyputmsg("GDS generation will merge boxes");
			if ((curstate&GDSOUTADDDRC) == 0)
				ttyputmsg("GDS generation will ignore DRC cloak layer"); else
					ttyputmsg("GDS generation will include DRC cloak layer");
			return(0);
		}
		l = strlen(pp = par[1]);
		if (namesamen(pp, "output", l) == 0)
		{
			if (count <= 2)
			{
				if ((curstate&GDSOUTMERGE) == 0)
					ttyputmsg("GDS generation will list individual boxes"); else
						ttyputmsg("GDS generation will merge boxes");
				if ((curstate&GDSOUTADDDRC) == 0)
					ttyputmsg("GDS generation will ignore DRC cloak layer"); else
						ttyputmsg("GDS generation will include DRC cloak layer");
				return(0);
			}
			l = strlen(pp = par[2]);
			if (namesamen(pp, "merge-boxes", l ) == 0)
			{
				(void)setvalkey((INTBIG)io_aid, VAID, io_state,
					curstate | GDSOUTMERGE, VINTEGER|VDONTSAVE);
				ttyputmsgf("GDS generation will merge boxes");
				return(0);
			}
			if (namesamen(pp, "individual-boxes", l) == 0 && l >= 3)
			{
				(void)setvalkey((INTBIG)io_aid, VAID, io_state,
					curstate & ~GDSOUTMERGE, VINTEGER|VDONTSAVE);
				ttyputmsgf("GDS generation will list individual boxes");
				return(0);
			}
			if (namesamen(pp, "include-cloak-layer", l) == 0 && l >= 3)
			{
				(void)setvalkey((INTBIG)io_aid, VAID, io_state,
					curstate | GDSOUTADDDRC, VINTEGER|VDONTSAVE);
				ttyputmsgf("GDS generation will include DRC cloak layer");
				return(0);
			}
			if (namesamen(pp, "ignore-cloak-layer", l) == 0 && l >= 2)
			{
				(void)setvalkey((INTBIG)io_aid, VAID, io_state,
					curstate & ~GDSOUTADDDRC, VINTEGER|VDONTSAVE);
				ttyputmsgf("GDS generation will ignore DRC cloak layer");
				return(0);
			}
			if (namesamen(pp, "arc-resolution", l) == 0 && l >= 5)
			{
				getcontoursegmentparameters(&arcres, &arcsag);
				if (count > 3)
				{
					if (namesamen(par[3], "dialog", (INTSML)strlen(par[3])) == 0)
					{
						io_gdsarcconversiondlg();
						return(0);
					}
					arcres = atofr(par[3]) * 10 / WHOLE;
				}
				ttyputmsgf("GDS arc generation will create a line every %s degrees",
					frtoa(arcres*WHOLE/10));
				setcontoursegmentparameters(arcres, arcsag);
				return(0);
			}
			if (namesamen(pp, "arc-sag", l) == 0 && l >= 5)
			{
				getcontoursegmentparameters(&arcres, &arcsag);
				if (count > 3) arcsag = atola(par[3]);
				ttyputmsgf("GDS arc generation will sag no more than %s", latoa(arcsag));
				setcontoursegmentparameters(arcres, arcsag);
				return(0);
			}
		}
		ttyputerr("Usage: tellaid io gds output OPTION");
		return(1);
	}
#endif

#if IODXF
	if (namesamen(pp, "dxf", l) == 0)
	{
		if (count <= 1)
		{
			if ((curstate&DXFALLLAYERS) == 0)
				ttyputmsg("DXF input will accept only layers listed by technology"); else
					ttyputmsg("DXF input will accept all layers");
			return(0);
		}
		l = strlen(pp = par[1]);
		if (namesamen(pp, "acceptable-layers", l) == 0)
		{
			if (count <= 2)
			{
				if ((curstate&DXFALLLAYERS) == 0)
					ttyputmsg("DXF input will accept only layers listed by technology"); else
						ttyputmsg("DXF input will accept all layers");
				return(0);
			}
			l = strlen(pp = par[2]);
			if (namesamen(pp, "all", l) == 0)
			{
				(void)setvalkey((INTBIG)io_aid, VAID, io_state,
					curstate | DXFALLLAYERS, VINTEGER|VDONTSAVE);
				ttyputmsgf("DXF input will accept all layers");
				return(0);
			}
			if (namesamen(pp, "restrict", l) == 0)
			{
				(void)setvalkey((INTBIG)io_aid, VAID, io_state,
					curstate & ~DXFALLLAYERS, VINTEGER|VDONTSAVE);
				ttyputmsgf("DXF input will accept only layers listed by technology");
				return(0);
			}
		}
		if (namesamen(pp, "flatten-input", l) == 0)
		{
			(void)setvalkey((INTBIG)io_aid, VAID, io_state,
				curstate | DXFFLATTENINPUT, VINTEGER|VDONTSAVE);
			ttyputmsgf("DXF input will flatten input blocks");
			return(0);
		}
		if (namesamen(pp, "not-flatten-input", l) == 0)
		{
			(void)setvalkey((INTBIG)io_aid, VAID, io_state,
				curstate & ~DXFFLATTENINPUT, VINTEGER|VDONTSAVE);
			ttyputmsgf("DXF input will retain input hierarchy");
			return(0);
		}
		ttyputerr("Bad tellaid io dxf OPTION");
		return(1);
	}
#endif

#if IOPOSTSCRIPT
	if (namesamen(pp, "postscript", l) == 0)
	{
		if (count <= 1)
		{
			if ((curstate&EPSPSCRIPT) == 0)
				ttyputmsg("PostScript output is in plain format"); else
					ttyputmsg("PostScript output is in encapsulated format");
			return(0);
		}
		l = strlen(pp = par[1]);
		if (namesamen(pp, "encapsulated", l) == 0)
		{
			(void)setvalkey((INTBIG)io_aid, VAID, io_state, curstate|EPSPSCRIPT, VINTEGER|VDONTSAVE);
			ttyputmsg("PostScript output will be encapsulated format");
			return(0);
		}
		if (namesamen(pp, "plain", l) == 0)
		{
			(void)setvalkey((INTBIG)io_aid, VAID, io_state, curstate & ~EPSPSCRIPT, VINTEGER|VDONTSAVE);
			ttyputmsgf("PostScript output will be plain format");
			return(0);
		}
		ttyputerr("Usage: tellaid io postscript OPTION");
		return(1);
	}
#endif

#if IOHPGL
	if (namesamen(pp, "hpgl", l) == 0)
	{
		if (count <= 1)
		{
			if ((curstate&HPGL2) == 0)
				ttyputmsg("HPGL output uses older (version 1) format"); else
					ttyputmsg("HPGL output uses newer (HPGL/2) format");
			return(0);
		}
		l = strlen(pp = par[1]);
		if (namesamen(pp, "1", l) == 0)
		{
			(void)setvalkey((INTBIG)io_aid, VAID, io_state, curstate & ~HPGL2, VINTEGER|VDONTSAVE);
			ttyputmsg("HPGL output will use older (version 1) format");
			return(0);
		}
		if (namesamen(pp, "2", l) == 0)
		{
			(void)setvalkey((INTBIG)io_aid, VAID, io_state, curstate|HPGL2, VINTEGER|VDONTSAVE);
			ttyputmsgf("HPGL output will use newer (HPGL/2) format");
			if (count >= 4 && namesamen(par[2], "scale", (INTSML)strlen(par[2])) == 0)
			{
				scale = myatoi(par[3]);
				(void)setval((INTBIG)io_aid, VAID, "IO_hpgl2_scale", scale, VINTEGER|VDONTSAVE);
				ttyputmsgf("HPGL/2 plots will scale at %d %ss per pixel", scale, unitsname(el_units));
			}
			return(0);
		}
		ttyputerr("Usage: tellaid io hpgl [1|2]");
		return(1);
	}
#endif

#if IOEDIF
	if (namesamen(pp, "edif", l) == 0)
	{
		if (count <= 1)
		{
			if ((curstate&EDIFSCHEMATIC) == 0)
				ttyputmsg("EDIF output writes schematics"); else
					ttyputmsg("EDIF output writes netlists");
			return(0);
		}
		l = strlen(pp = par[1]);
		if (namesamen(pp, "schematic", l) == 0)
		{
			(void)setvalkey((INTBIG)io_aid, VAID, io_state, curstate | EDIFSCHEMATIC, VINTEGER|VDONTSAVE);
			ttyputmsgf("EDIF output will write schematics");
			return(0);
		}
		if (namesamen(pp, "netlist", l) == 0)
		{
			(void)setvalkey((INTBIG)io_aid, VAID, io_state, curstate & ~EDIFSCHEMATIC, VINTEGER|VDONTSAVE);
			ttyputmsgf("EDIF output will write netlists");
			return(0);
		}
		ttyputerr("Usage: tellaid io edif [schematic|netlist]");
		return(1);
	}
#endif

#if IOSDF
    if (namesamen(pp, "sdf", l) == 0)
	{
		if (count <= 1)
		{
            if ((curstate&SDFVERBOSE) == 0)
                ttyputmsg("SDF verbose input"); else
                    ttyputmsg("SDF no verbose input");
            return(0);
        }
        l = strlen(pp = par[1]);
        if(namesamen(pp, "verbose", l) == 0)
        {
            (void)setvalkey((INTBIG)io_aid, VAID, io_state, curstate | SDFVERBOSE, VINTEGER|VDONTSAVE);
            ttyputmsgf("SDF input will print verbose messages");
            return(0);
        }
        if(namesamen(pp, "noverbose", l) == 0)
        {
            (void)setvalkey((INTBIG)io_aid, VAID, io_state, curstate & ~SDFVERBOSE, VINTEGER|VDONTSAVE);
            ttyputmsgf("SDF input will not print verbose messages");
            return(0);
        }
        ttyputerr("Usage: tellaid io sdf [verbose|noverbose]");
        return(1);
    }
#endif

	if (namesamen(pp, "verbose", l) == 0)
	{
		if (count <= 1)
		{
			ttyputerr("Usage: tellaid io verbose OPTION");
			return(1);
		}
		l = strlen(pp = par[1]);
		if (namesamen(pp, "off", l) == 0 && l >= 2)
		{
			io_verbose = 0;
			ttyputmsgf("I/O done silently");
			return(0);
		}
		if (namesamen(pp, "on", l) == 0 && l >= 2)
		{
			io_verbose = 1;
			ttyputmsgf("I/O prints status");
			return(0);
		}
		if (namesamen(pp, "graphical", l) == 0 && l >= 1)
		{
			io_verbose = -1;
			ttyputmsgf("I/O shows graphical progress");
			return(0);
		}
		ttyputerr("Unknown verbose option: %s", pp);
		return(1);
	}

	ttyputerr("Bad I/O option: %s", par[0]);
	return(1);
}

/*
 * make I/O requests of this aid:
 *
 * "read" TAKES: LIBRARY to read and string style
 * "write" TAKES: LIBRARY to write and string style
 * "verbose" TAKES: new verbose factor, RETURNS: old verbose factor
 */
INTBIG io_request(char *command, va_list ap)
{
	REGISTER LIBRARY *lib;
	REGISTER INTSML i, l, len, format;
	REGISTER INTBIG arg1;
	char *arg2;

	if (namesame(command, "verbose") == 0)
	{
		i = io_verbose;
		io_verbose = va_arg(ap, INTSML);
		return(i);
	}

	/* get the arguments */
	arg1 = va_arg(ap, INTBIG);
	arg2 = va_arg(ap, char*);

	/* find desired library and format */
	lib = (LIBRARY *)arg1;
	format = -1;
	len = strlen(arg2);
	for(i=0; io_formatlist[i].name != 0; i++)
		if (namesamen(arg2, io_formatlist[i].name, len) == 0 && len >= io_formatlist[i].required)
	{
		format = io_formatlist[i].bits;
		break;
	}
	if (format < 0)
	{
		ttyputerr("Unknown I/O format: %s", arg2);
		return(1);
	}

	if (namesame(command, "write") == 0)
	{
		/* announce the beginning of library output */
		for(i=0; i<el_maxaid; i++) (*el_aids[i].writelibrary)(lib, 0);

		switch (format)
		{
			case FBINARY:
				if (checkcap(io_aid, IOCANWRITE) == 0)
				{
					ttyputerr("Sorry, this demo version cannot write libraries");
					return(1);
				}
				l = io_writebinlibrary(lib);
				break;

#if IOCIF
			case FCIF:
				if (checkcap(io_aid, IOHASIC) == 0)
				{
					ttyputerr("Sorry, CIF output is not available");
					return(1);
				}
				l = io_writeciflibrary(lib);
				break;
#endif

#if IODXF
			case FDXF:
				if (checkcap(io_aid, IOHASMECH) == 0)
				{
					ttyputerr("Sorry, DXF output is not available");
					return(1);
				}
				l = io_writedxflibrary(lib);
				break;
#endif

#if IOEDIF
			case FEDIF:
				if (checkcap(io_aid, IOHASIC) == 0)
				{
					ttyputerr("Sorry, EDIF output is not available");
					return(1);
				}
				l = io_writeediflibrary(lib);
				break;
#endif

#if IOGDS
			case FGDS:
				if (checkcap(io_aid, IOHASIC) == 0)
				{
					ttyputerr("Sorry, GDS output is not available");
					return(1);
				}
				l = io_writegdslibrary(lib);
				break;
#endif

#if IOHPGL
			case FHPGL:
				if (checkcap(io_aid, IOHASPRINT) == 0)
				{
					ttyputerr("Sorry, HPGL output is not available");
					return(1);
				}
				l = io_writehpgllibrary(lib);
				break;
#endif

#if IOL
			case FL:
				if (checkcap(io_aid, IOHASIC) == 0)
				{
					ttyputerr("Sorry, L output is not available");
					return(1);
				}
				l = io_writellibrary(lib);
				break;
#endif

#if IOPOSTSCRIPT
			case FPOSTSCRIPT:
				if (checkcap(io_aid, IOHASPRINT) == 0)
				{
					ttyputerr("Sorry, PostScript output is not available");
					return(1);
				}
				l = io_writepostscriptlibrary(lib);
				break;
#endif

#if IOQUICKDRAW
			case FQUICKDRAW:
				if (checkcap(io_aid, IOHASPRINT) == 0)
				{
					ttyputerr("Sorry, Quickdraw output is not available");
					return(1);
				}
				l = io_writequickdrawlibrary(lib);
				break;
#endif

			case FTEXT:
				if (checkcap(io_aid, IOCANWRITE) == 0)
				{
					ttyputerr("Sorry, this demo version cannot write libraries");
					return(1);
				}
				l = io_writetextlibrary(lib);
				break;

			default:
				for(i=0; io_formatlist[i].name != 0; i++)
					if (io_formatlist[i].bits == format)
				{
					ttyputerr("Cannot write %s files", io_formatlist[i].name);
					return(1);
				}
		}

		/* announce the ending of library output */
		if (l == 0)
		{
			/* lib->userbits |= READFROMDISK; */
			for(i=0; i<el_maxaid; i++) (*el_aids[i].writelibrary)(lib, 1);
		}
		return(l);
	}

	if (namesame(command, "read") == 0)
	{
		switch (format)
		{
			case FBINARY:
				l = io_readbinlibrary(lib);
				break;

#if IOCIF
			case FCIF:
				if (checkcap(io_aid, IOHASIC) == 0)
				{
					ttyputerr("Sorry, CIF input is not available");
					return(1);
				}
				l = io_readciflibrary(lib);
				break;
#endif

#if IODXF
			case FDXF:
				if (checkcap(io_aid, IOHASMECH) == 0)
				{
					ttyputerr("Sorry, DXF input is not available");
					return(1);
				}
				l = io_readdxflibrary(lib);
				break;
#endif

#if IOEDIF
			case FEDIF:
				if (checkcap(io_aid, IOHASIC) == 0)
				{
					ttyputerr("Sorry, EDIF input is not available");
					return(1);
				}
				l = io_readediflibrary(lib);
				break;
#endif

#if IOGDS
			case FGDS:
				if (checkcap(io_aid, IOHASIC) == 0)
				{
					ttyputerr("Sorry, GDS input is not available");
					return(1);
				}
				l = io_readgdslibrary(lib);
				break;
#endif

#if IOSDF
            case FSDF:
                if (checkcap(io_aid, IOHASIC) == 0)
                {
                    ttyputerr("Sorry, SDF input is not available");
                    return(1);
                }
                l = io_readsdflibrary(lib);
                break;
#endif

			case FTEXT:
				l = io_readtextlibrary(lib);
				break;

#if VHDLAID
			case FVHDL:
				l = io_readvhdllibrary(lib);
				break;
#endif

			default:
				for(i=0; io_formatlist[i].name != 0; i++)
					if (io_formatlist[i].bits == format)
				{
					ttyputerr("Cannot read %s files", io_formatlist[i].name);
					return(1);
				}
		}

		/* announce the completion of library input */
		if (l == 0)
		{
			for(i=0; i<el_maxaid; i++) (*el_aids[i].readlibrary)(lib);
		}
		return(l);
	}
	return(-1);
}

/******************************* BLOATING *******************************/

/*
 * routine to indicate that CIF layer "layer" is to be bloated by "amount"
 */
void io_setoutputbloat(char *layer, INTBIG amount)
{
	REGISTER BLOAT *bl;

	/* special case when bloating the null layer */
	if (*layer == 0)
	{
		for(bl = io_curbloat; bl != NOBLOAT; bl = bl->nextbloat)
			ttyputmsg("Bloating layer %s by %d %ss", bl->layer, bl->amount, unitsname(el_units));
		return;
	}

	/* first see if this layer is already being bloated */
	for(bl = io_curbloat; bl != NOBLOAT; bl = bl->nextbloat)
		if (namesame(bl->layer, layer) == 0)
	{
		if (bl->amount == amount)
		{
			ttyputmsg("Layer %s is already being bloated by %d %ss",
				bl->layer, amount, unitsname(el_units));
			return;
		}
		ttyputmsg("Layer %s was being bloated by %d %ss, now by %d",
			bl->layer, bl->amount, unitsname(el_units), amount);
		bl->amount = amount;
		return;
	}

	bl = (BLOAT *)emalloc(sizeof (BLOAT), io_aid->cluster);
	if (bl == 0)
	{
		ttyputerr("No bloat memory");
		return;
	}
	(void)allocstring(&bl->layer, layer, io_aid->cluster);
	bl->amount = amount;
	bl->nextbloat = io_curbloat;
	io_curbloat = bl;
	ttyputmsg("Layer %s will be bloated by %d %ss", layer, amount, unitsname(el_units));
}

/*
 * routine to tell the amount to bloat layer "lay"
 */
INTBIG io_getoutputbloat(char *layer)
{
	REGISTER BLOAT *bl;

	for(bl = io_curbloat; bl != NOBLOAT; bl = bl->nextbloat)
		if (namesame(bl->layer, layer) == 0) return(bl->amount);
	return(0);
}

/******************************* LIBRARY REPAIR *******************************/

/*
 * routine to complete formation of library "lib" that has just been read from
 * disk.  Some data is not read, but computed from other data.
 */
void io_fixnewlib(LIBRARY *lib)
{
	REGISTER NODEPROTO *np, *pnt;
	REGISTER NODEINST *ni;
	REGISTER ARCINST *ai;
	REGISTER INTSML i, j, libunits;
	REGISTER INTBIG lam;
	INTBIG lx, ly;
	static POLYGON *poly = NOPOLYGON;
	static INTBIG SIM_behave_rise_delay = 0, SIM_behave_fall_delay = 0,
		SIM_silos_behave_file = 0, SILOS_global_name = 0, SIM_rise_delay = 0,
		SIM_fall_delay = 0, SIM_silos_behavior_file = 0, SIM_silos_global_name = 0,
		SPICE_model_file = 0, SPICE_trailer_file = 0, SIM_spice_model_file = 0,
		SIM_spice_trailer_file = 0, edtec_option = 0;
	char line[50];
	REGISTER VARIABLE *var;
	REGISTER TECHNOLOGY *tech;
	REGISTER PORTPROTO *pp, *inpp;
	REGISTER PORTEXPINST *pe;
	extern AIDENTRY *net_aid;

	/* make sure units are correct */
	libunits = ((lib->userbits & LIBUNITS) >> LIBUNITSSH) << INTERNALUNITSSH;
	if ((libunits & INTERNALUNITS) != (el_units & INTERNALUNITS))
	{
		ttyputmsg("Converting library from %s units to %s units", unitsname(libunits),
			unitsname(el_units));
		changeinternalunits(lib, libunits, el_units);
	}

	/* adjust every facet in the library */
	for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		np->firstinst = NONODEINST;
	for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
	{
		/* examine every nodeinst in the facet */
		for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
		{
			/* compute linked list of primitive nodes */
			pnt = ni->proto;
			if (pnt->firstinst != NONODEINST) pnt->firstinst->lastinst = ni;
			ni->nextinst = pnt->firstinst;
			ni->lastinst = NONODEINST;
			pnt->firstinst = ni;
		}
		for(pp = np->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
			for(pe = pp->subnodeinst->firstportexpinst; pe != NOPORTEXPINST; pe = pe->nextportexpinst)
				if (pe->proto == pp->subportproto)
		{
			pp->subportexpinst = pe;  break;
		}
#ifdef MACOS
		/*
		 * The Macintosh uses a different date epoch than other machines, and this
		 * is handled in "dbtext.c".  However, the adjustment code is recent,
		 * and some old Macintosh libraries may have incorrect date information.
		 *
		 * The Macintosh epoch is 1904, but every other machine uses 1970.
		 * Therefore, the Macintosh numbers are 66 years larger.
		 *
		 * To check for bogus date information, see if the date is larger
		 * than the constant 0x9FE3EB1F.  This constant is 1989 in Mac time
		 * but is 2055 elsewhere (1989 is the earliest possible incarnation
		 * of Electric on the Mac).  If the date is that large, then either it
		 * is 2055 and Macs still exist (unlikely) or the date is bad and needs
		 * to have 66 years (0x7C254E10) subtracted from it.
		 */
		if (np->creationdate > 0x9FE3EB1F) np->creationdate -= 0x7C254E10;
		if (np->revisiondate > 0x9FE3EB1F) np->revisiondate -= 0x7C254E10;
#endif
	}

	/* now fill in the connection lists on the ports */
	for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		for(pp = np->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
	{
		inpp = pp;
		while (inpp->parent->index == 0) inpp = inpp->subportproto;
		pp->connects = inpp->connects;
	}

	/* create network information */
	(void)askaid(net_aid, "total-re-number", (INTBIG)lib);

	/*
	 * see if this is version 4 or earlier...
	 * In version 4 and earlier, the basic unit was the centimicron and rotation
	 * was specified in degrees.  From version 5 and up, the basic unit is scaled
	 * by 20 to the half-millimicron and the rotation fields are scaled by 10 to
	 * tenth-degrees.  Also, must convert some variable names.
	 */
	var = getval((INTBIG)lib, VLIBRARY, VSTRING, "LIB_former_version");
	if (var == NOVARIABLE) return;
	i = atoi((char *)var->addr);
	if (i >= 5) return;

	/* cache old variable names */
	if (SIM_behave_rise_delay == 0) SIM_behave_rise_delay = makekey("SIM_behave_rise_delay");
	if (SIM_behave_fall_delay == 0) SIM_behave_fall_delay = makekey("SIM_behave_fall_delay");
	if (SIM_silos_behave_file == 0) SIM_silos_behave_file = makekey("SIM_silos_behave_file");
	if (SILOS_global_name == 0) SILOS_global_name = makekey("SILOS_global_name");
	if (SPICE_model_file == 0) SPICE_model_file = makekey("SPICE_model_file");
	if (SPICE_trailer_file == 0) SPICE_trailer_file = makekey("SPICE_trailer_file");

	/* cache new variable names */
	if (SIM_rise_delay == 0) SIM_rise_delay = makekey("SIM_rise_delay");
	if (SIM_fall_delay == 0) SIM_fall_delay = makekey("SIM_fall_delay");
	if (SIM_silos_behavior_file == 0) SIM_silos_behavior_file = makekey("SIM_silos_behavior_file");
	if (SIM_silos_global_name == 0) SIM_silos_global_name = makekey("SIM_silos_global_name");
	if (SIM_spice_model_file == 0) SIM_spice_model_file = makekey("SIM_spice_model_file");
	if (SIM_spice_trailer_file == 0) SIM_spice_trailer_file = makekey("SIM_spice_trailer_file");

	/* search technologies and convert variables */
	for(tech = el_technologies; tech != NOTECHNOLOGY; tech = tech->nexttechnology)
	{
		var = getvalkey((INTBIG)tech, VTECHNOLOGY, VSTRING, SPICE_model_file);
		if (var != NOVARIABLE)
		{
			nextvarchangequiet();
			(void)setvalkey((INTBIG)tech, VTECHNOLOGY, SIM_spice_model_file, var->addr, var->type);
			nextvarchangequiet();
			(void)delvalkey((INTBIG)tech, VTECHNOLOGY, SPICE_model_file);
		}

		var = getvalkey((INTBIG)tech, VTECHNOLOGY, VSTRING, SPICE_trailer_file);
		if (var != NOVARIABLE)
		{
			nextvarchangequiet();
			(void)setvalkey((INTBIG)tech, VTECHNOLOGY, SIM_spice_trailer_file, var->addr, var->type);
			nextvarchangequiet();
			(void)delvalkey((INTBIG)tech, VTECHNOLOGY, SPICE_trailer_file);
		}
	}

	/* search the library and convert variables */
	for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
	{
		var = getvalkey((INTBIG)np, VNODEPROTO, VSTRING, SIM_silos_behave_file);
		if (var != NOVARIABLE)
		{
			nextvarchangequiet();
			(void)setvalkey((INTBIG)np, VNODEPROTO, SIM_silos_behavior_file, var->addr, var->type);
			nextvarchangequiet();
			(void)delvalkey((INTBIG)np, VNODEPROTO, SIM_silos_behave_file);
		}
		for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
		{
			if (ni->proto->index == 0) continue;
			var = getvalkey((INTBIG)ni, VNODEINST, -1, SIM_behave_rise_delay);
			if (var != NOVARIABLE)
			{
				nextvarchangequiet();
				(void)setvalkey((INTBIG)ni, VNODEINST, SIM_rise_delay, var->addr, var->type);
				nextvarchangequiet();
				(void)delvalkey((INTBIG)ni, VNODEINST, SIM_behave_rise_delay);
			}

			var = getvalkey((INTBIG)ni, VNODEINST, -1, SIM_behave_fall_delay);
			if (var != NOVARIABLE)
			{
				nextvarchangequiet();
				(void)setvalkey((INTBIG)ni, VNODEINST, SIM_fall_delay, var->addr, var->type);
				nextvarchangequiet();
				(void)delvalkey((INTBIG)ni, VNODEINST, SIM_behave_fall_delay);
			}

			var = getvalkey((INTBIG)ni, VNODEINST, -1, SILOS_global_name);
			if (var != NOVARIABLE)
			{
				nextvarchangequiet();
				(void)setvalkey((INTBIG)ni, VNODEINST, SIM_silos_global_name, var->addr, var->type);
				nextvarchangequiet();
				(void)delvalkey((INTBIG)ni, VNODEINST, SILOS_global_name);
			}
		}
	}

	/* setup for units conversion */
	if (poly == NOPOLYGON) poly = allocpolygon(4, io_aid->cluster);
	if (edtec_option == 0) edtec_option = makekey("EDTEC_option");

	/* must scale, first make sure the technologies agree with this library */
	if (lib != el_curlib)
		for(tech = el_technologies; tech != NOTECHNOLOGY; tech = tech->nexttechnology)
			db_scaletechnology(tech, lib->lambda[tech->index], tech->deflambda);

	/* now scale */
	for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
	{
		var = getvalkey((INTBIG)np, VNODEPROTO, VINTEGER|VISARRAY, el_prototype_center);
		if (var != NOVARIABLE)
		{
			((INTBIG *)var->addr)[0] *= 20;
			((INTBIG *)var->addr)[1] *= 20;
		}
		for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
		{
			ni->lowx *= 20;   ni->highx *= 20;
			ni->lowy *= 20;   ni->highy *= 20;
			ni->rotation *= 10;
			var = getvalkey((INTBIG)ni, VNODEINST, VINTEGER, edtec_option);
			if (var != NOVARIABLE)
			{
				if (var->addr == TECHLAMBDA)
				{
					var = getvalkey((INTBIG)ni, VNODEINST, VSTRING, art_messagekey);
					if (var != NOVARIABLE)
					{
						lam = myatoi(&((char *)var->addr)[8]);
						(void)sprintf(line, "Lambda: %d", lam*20);
						nextvarchangequiet();
						(void)setvalkey((INTBIG)ni, VNODEINST, art_messagekey,
							(INTBIG)line, VSTRING|VDISPLAY);
					}
				}
			}
			var = getvalkey((INTBIG)ni, VNODEINST, VINTEGER|VISARRAY, el_trace);
			if (var != NOVARIABLE)
			{
				i = getlength(var);
				for(j=0; j<i; j++) ((INTBIG *)var->addr)[j] *= 20;
			}
			var = getvalkey((INTBIG)ni, VNODEINST, VINTEGER, art_degreeskey);
			if (var != NOVARIABLE) var->addr *= 10;
			boundobj(ni->geom, &ni->geom->lowx, &ni->geom->highx, &ni->geom->lowy, &ni->geom->highy);
		}
	}

	for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
	{
		for(ai = np->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
		{
			ai->width *= 20;
			ai->end[0].xpos *= 20;   ai->end[0].ypos *= 20;
			ai->end[1].xpos *= 20;   ai->end[1].ypos *= 20;
			setshrinkvalue(ai);
			var = getvalkey((INTBIG)ai, VARCINST, VINTEGER, el_arc_radius);
			if (var != NOVARIABLE) var->addr *= 20;
			for(i=0; i<2; i++)
			{
				/* make sure arcinst connects in portinst area */
				shapeportpoly(ai->end[i].nodeinst, ai->end[i].portarcinst->proto, poly, 0);
				if (isinside(ai->end[i].xpos, ai->end[i].ypos, poly) != 0) continue;
				portposition(ai->end[i].nodeinst, ai->end[i].portarcinst->proto, &lx, &ly);
				if (lx == ai->end[i].xpos && ly == ai->end[i].ypos) continue;

				/* try to make manhattan fix */
				if ((ai->end[0].xpos == ai->end[1].xpos) && isinside(ai->end[i].xpos, ly, poly) != 0)
					ai->end[i].ypos = ly; else
						if ((ai->end[0].ypos == ai->end[1].ypos) &&
							isinside(lx, ai->end[i].ypos, poly) != 0)
								ai->end[i].xpos = lx; else
				{
					ai->end[i].xpos = lx;   ai->end[i].ypos = ly;
				}
			}
			ai->length = computedistance(ai->end[0].xpos,ai->end[0].ypos,
				ai->end[1].xpos,ai->end[1].ypos);
			boundobj(ai->geom, &ai->geom->lowx, &ai->geom->highx, &ai->geom->lowy, &ai->geom->highy);
		}
		db_boundfacet(np, &np->lowx, &np->highx, &np->lowy, &np->highy);
		io_fixrtree(np->rtree);
	}

	/* now restore the technology lambda values */
	if (lib != el_curlib)
		for(tech = el_technologies; tech != NOTECHNOLOGY; tech = tech->nexttechnology)
			db_scaletechnology(tech, el_curlib->lambda[tech->index], tech->deflambda);
}

void io_fixrtree(RTNODE *rtree)
{
	REGISTER INTSML i;
	REGISTER GEOM *geom;
	REGISTER RTNODE *subrt;

	if (rtree->total <= 0) return;
	if (rtree->flag != 0)
	{
		geom = (GEOM *)rtree->pointers[0];
		rtree->lowx = geom->lowx;   rtree->highx = geom->highx;
		rtree->lowy = geom->lowy;   rtree->highy = geom->highy;
		for(i=1; i<rtree->total; i++)
		{
			geom = (GEOM *)rtree->pointers[i];
			if (geom->lowx < rtree->lowx) rtree->lowx = geom->lowx;
			if (geom->highx > rtree->highx) rtree->highx = geom->highx;
			if (geom->lowy < rtree->lowy) rtree->lowy = geom->lowy;
			if (geom->highy > rtree->highy) rtree->highy = geom->highy;
		}
	} else
	{
		subrt = (RTNODE *)rtree->pointers[0];
		io_fixrtree(subrt);
		rtree->lowx = subrt->lowx;   rtree->highx = subrt->highx;
		rtree->lowy = subrt->lowy;   rtree->highy = subrt->highy;
		for(i=1; i<rtree->total; i++)
		{
			subrt = (RTNODE *)rtree->pointers[i];
			io_fixrtree(subrt);
			if (subrt->lowx < rtree->lowx) rtree->lowx = subrt->lowx;
			if (subrt->highx > rtree->highx) rtree->highx = subrt->highx;
			if (subrt->lowy < rtree->lowy) rtree->lowy = subrt->lowy;
			if (subrt->highy > rtree->highy) rtree->highy = subrt->highy;
		}
	}
}

/* former definitions for pre-verision-4 libraries */
#define	OLDPORTLABXOFF  0300000000
#define	OLDPORTLABXOFFSH        24
#define	OLDPORTLABYOFF 01400000000
#define	OLDPORTLABYOFFSH        26
#define	VIEWTYPE            074000	/* former location of facet view type */
#define	VIEWTYPESH              11	/* right shift of VIEWTYPE */

struct
{
	char      *viewname;
	INTBIG      viewkey;
	NODEPROTO *facet;
	char      *realview;
} io_facetviews[8] =
{
	{"",                       0, NONODEPROTO, "unknown"},
	{"CELL_layout",            0, NONODEPROTO, "layout"},
	{"CELL_layout_alt1",       0, NONODEPROTO, "layout"},
	{"CELL_layout_alt2",       0, NONODEPROTO, "layout"},
	{"CELL_schematic",         0, NONODEPROTO, "schematic"},
	{"CELL_icon",              0, NONODEPROTO, "icon"},
	{"CELL_simulation_output", 0, NONODEPROTO, "simulation-snapshot"},
	{"CELL_skeleton",          0, NONODEPROTO, "skeleton"},
};

/*
 * routine to convert pre-version-4 libraries to version 4 libraries.
 * This includes the following:
 * > The date information in the NODEPROTO variable "CELL_dates" is converted
 *   to the NODEPROTO->creationdate and NODEPROTO->revisiondate fields.
 * > The view associations in the NODEPROTO variables "CELL_layout",
 *   "CELL_layout_alt1", "CELL_layout_alt2", "CELL_schematic", "CELL_icon",
 *   "CELL_simulation_output", and "CELL_skeleton" are converted to proper
 *   facet organizations.
 * > The variable "original_cell" on facets is converted to "original_facet".
 * > The network information in "NET_arc_list" and "NET_name_list" is thrown out.
 * > The schematics attributes "LOGIC_function", "LOGIC_capacitance",
 *   "LOGIC_flipflop_type", "LOGIC_inductance", "LOGIC_diode",
 *   "LOGIC_meter_type", "LOGIC_source", "LOGIC_resistance", and
 *   "LOGIC_transistor_type" are made displayable and converted to their
 *   proper names which begin with "SCHEM" rather than "LOGIC".
 * > The schematics attribute "LOGIC_net" is deleted.
 * > The variable "ART_font" is converted to a proper descriptor on the
 *   associated variable "ART_message"
 * > The array "ART_message" is made scalar if its length is 1.
 * > The port offset factors in PORTPROTO->userbits is moved to the text
 *   descriptor field in PORTPROTO->textdescript.
 * > Serpentine transistors have had their ports reversed.  The connections
 *   are redone.
 * > The "transistor_width" field on serpentine transistors is converted from
 *   integer or float to fractional
 */
#define	SHIFTRIGHT 1
#define	SHIFTUP    2
#define	NOTDISPLAY 3
struct
{
	char      *nodename;
	NODEPROTO *type;
	INTSML      special;
	INTSML      xoff, yoff;
	char      *oldname;
	INTBIG      oldkey;
	char      *newname;
	INTBIG      newkey;
} io_convertschem[] =
{
	{"Schematic:Buffer",     NONODEPROTO, 0,          -4, 1, 0, 0, 0, 0},
	{"Schematic:And",        NONODEPROTO, SHIFTRIGHT, -2, 1, 0, 0, 0, 0},
	{"Schematic:Or",         NONODEPROTO, SHIFTRIGHT,  0, 1, 0, 0, 0, 0},
	{"Schematic:Xor",        NONODEPROTO, SHIFTRIGHT,  0, 1, 0, 0, 0, 0},
	{"Schematic:Flip-Flop",  NONODEPROTO, NOTDISPLAY,  1, 0,
		"LOGIC_flipflop_type",   0, "SCHEM_flipflop_type",   0},
	{"Schematic:Bbox",       NONODEPROTO, 0,           0, 0,
		"LOGIC_function",        0, "SCHEM_function",        0},
	{"Schematic:Switch",     NONODEPROTO, 0,           1, 1, 0, 0, 0, 0},
	{"Schematic:Off-Page",   NONODEPROTO, 0,           2, 1,
		"LOGIC_net",             0, 0, 0},
	{"Schematic:Power",      NONODEPROTO, 0,           1, 1, 0, 0, 0, 0},
	{"Schematic:Ground",     NONODEPROTO, 0,           1, 1, 0, 0, 0, 0},
	{"Schematic:Source",     NONODEPROTO, 0,           1, 1,
		"LOGIC_source",          0, "SCHEM_source",          0},
	{"Schematic:Transistor", NONODEPROTO, SHIFTUP,    -1,-1,
		"LOGIC_transistor_type", 0, "SCHEM_transistor_type", 0},
	{"Schematic:Resistor",   NONODEPROTO, 0,           1,-3,
		"LOGIC_resistance",      0, "SCHEM_resistance",      0},
	{"Schematic:Capacitor",  NONODEPROTO, 0,          -1,-1,
		"LOGIC_capacitance",     0, "SCHEM_capacitance",     0},
	{"Schematic:Diode",      NONODEPROTO, 0,          -3,-1,
		"LOGIC_diode",           0, "SCHEM_diode",           0},
	{"Schematic:Inductor",   NONODEPROTO, 0,          -3,-1,
		"LOGIC_inductance",      0, "SCHEM_inductance",      0},
	{"Schematic:Meter",      NONODEPROTO, 0,           1, 1,
		"LOGIC_meter_type",      0, "SCHEM_meter_type",      0},
	{NULL, NULL, 0, 0, 0, NULL, 0, NULL, 0}  /* 0 */
};
void io_yankvariabledata(LIBRARY *lib)
{
	REGISTER NODEPROTO *np, *onp, *lnp;
	REGISTER NODEINST *ni;
	REGISTER PORTPROTO *pp;
	REGISTER CELL *c;
	REGISTER INTSML i, j, l, found, letter, thisone, trynew, gotone;
	REGISTER INTBIG descript, adjust, xoff, yoff, schemlambda, artlambda, twid;
	INTBIG xshift, yshift;
	XARRAY trans;
	REGISTER VARIABLE *var, *var2;
	static INTBIG cell_dates = 0, net_name_list = 0, net_arc_list = 0,
		art_font = 0, original_cell = 0,
		original_facet = 0, original_nodename;
	REGISTER float wid;
	REGISTER char *pt;
	char *common, *oldname;

	ttyputmsg("Converting pre-version-4 library...");

	/* cache variable names */
	if (cell_dates == 0) cell_dates = makekey("CELL_dates");
	if (original_cell == 0) original_cell = makekey("original_cell");
	if (original_nodename == 0) original_nodename = makekey("ORIGINAL_NODENAME");
	if (original_facet == 0) original_facet = makekey("original_facet");
	if (net_name_list == 0) net_name_list = makekey("NET_name_list");
	if (net_arc_list == 0) net_arc_list = makekey("NET_arc_list");
	for(i=0; io_convertschem[i].nodename != 0; i++)
	{
		io_convertschem[i].type = getnodeproto(io_convertschem[i].nodename);
		if (io_convertschem[i].oldname != 0)
			io_convertschem[i].oldkey = makekey(io_convertschem[i].oldname);
		if (io_convertschem[i].newname != 0)
			io_convertschem[i].newkey = makekey(io_convertschem[i].newname);
	}
	if (art_font == 0) art_font = makekey("ART_font");
	for(i=1; i<8; i++)
		if (io_facetviews[i].viewkey == 0)
			io_facetviews[i].viewkey = makekey(io_facetviews[i].viewname);

	/* find the value of lambda in the schematics and artwork technologies */
	schemlambda = lib->lambda[sch_tech->index];
	artlambda = lib->lambda[art_tech->index];

	/* set facet views */
	for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		switch ((np->userbits&VIEWTYPE) >> VIEWTYPESH)
	{
		case 0: np->cellview = el_unknownview;      break;
		case 1: np->cellview = el_layoutview;       break;
		case 2: np->cellview = el_layoutview;       break;
		case 3: np->cellview = el_layoutview;       break;
		case 4: np->cellview = el_schematicview;    break;
		case 5: np->cellview = el_iconview;         break;
		case 6: np->cellview = el_simsnapview;      break;
		case 7: np->cellview = el_skeletonview;     break;
	}

	/* search the library for view links */
	gotone = 1;
	while (gotone != 0)
	{
		gotone = 0;
		for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		{
			/* clear view lists */
			for(i=0; i<8; i++) io_facetviews[i].facet = NONODEPROTO;
			found = 0;
			for(i=1; i<8; i++)
			{
				var = getvalkey((INTBIG)np, VNODEPROTO, VNODEPROTO, io_facetviews[i].viewkey);
				if (var != NOVARIABLE)
				{
					io_facetviews[i].facet = (NODEPROTO *)var->addr;
					found++;
				}
			}

			/* add in this facet in its proper place */
			thisone = (np->userbits&VIEWTYPE) >> VIEWTYPESH;
			if (io_facetviews[thisone].facet == NONODEPROTO)
			{
				found++;
				io_facetviews[thisone].facet = np;
			}

			/* if there is no set to link, go no further */
			if (found <= 1) continue;
			gotone++;

			/* find a common name */
			for(l=0; ; l++)
			{
				letter = -1;
				for(i=0; i<8; i++)
				{
					if (io_facetviews[i].facet == NONODEPROTO) continue;
					np = io_facetviews[i].facet;
					if (letter == -1)
					{
						letter = io_facetviews[i].facet->cell->cellname[l];
						if (isupper(letter)) letter = tolower(letter);
					} else
					{
						j = io_facetviews[i].facet->cell->cellname[l];
						if (isupper(j)) j = tolower(j);
						if (letter != j) break;
					}
				}
				if (i < 8) break;
				if (letter == 0) break;
			}

			if (l == 0)
			{
				/* no common name found */
				if (io_facetviews[1].facet != NONODEPROTO)
				{
					(void)allocstring(&common, io_facetviews[1].facet->cell->cellname, el_tempcluster);
				} else if (io_facetviews[4].facet != NONODEPROTO)
				{
					(void)allocstring(&common, io_facetviews[1].facet->cell->cellname, el_tempcluster);
				} else (void)allocstring(&common, "SORRY", el_tempcluster);
			} else
			{
				/* use common name for facet */
				if (l > 1 && !isalnum(np->cell->cellname[l-1])) l--;
				letter = np->cell->cellname[l];
				np->cell->cellname[l] = 0;
				(void)allocstring(&common, np->cell->cellname, el_tempcluster);
				np->cell->cellname[l] = letter;
			}

			/* make sure common name is unique */
			oldname = (char *)emalloc((strlen(common)+10), el_tempcluster);
			(void)strcpy(oldname, common);
			for(trynew=1; ; trynew++)
			{
				for(c = lib->firstcell; c != NOCELL; c = c->nextcell)
				{
					if (namesame(oldname, c->cellname) != 0) continue;
					for(i=0; i<8; i++)
					{
						if (io_facetviews[i].facet == NONODEPROTO) continue;
						if (io_facetviews[i].facet->cell == c) break;
					}
					if (i < 8) continue;

					/* name already exists, try another */
					(void)sprintf(oldname, "%s%d", common, trynew);
					break;
				}
				if (c == NOCELL) break;
			}
			efree(common);

			(void)initinfstr();
			(void)addstringtoinfstr("The facets");
			found = 0;
			for(i=0; i<8; i++)
			{
				if (io_facetviews[i].facet == NONODEPROTO) continue;
				if (found != 0) (void)addtoinfstr(',');
				(void)addstringtoinfstr(" '");
				(void)addstringtoinfstr(io_facetviews[i].facet->cell->cellname);
				(void)addstringtoinfstr("'");
				found++;
			}
			ttyputmsg("%s have been renamed '%s'", returninfstr(), oldname);

			/* delete the old view links */
			for(i=0; i<8; i++)
			{
				if (io_facetviews[i].facet == NONODEPROTO) continue;
				for(j=1; j<8; j++)
				{
					var = getvalkey((INTBIG)io_facetviews[i].facet, VNODEPROTO,
						VNODEPROTO, io_facetviews[j].viewkey);
					if (var != NOVARIABLE)
					{
						nextvarchangequiet();
						(void)delvalkey((INTBIG)io_facetviews[i].facet,
							VNODEPROTO, io_facetviews[j].viewkey);
					}
				}
			}

			/* reorganize the facets */
			for(i=0; i<8; i++) if (io_facetviews[i].facet != NONODEPROTO)
			{
				c = io_facetviews[i].facet->cell;
				break;
			}
			for(i=0; i<8; i++)
			{
				lnp = io_facetviews[i].facet;
				if (lnp == NONODEPROTO) continue;
				db_retractnodeproto(lnp);
				if (lnp->cell != c)
				{
					if (lnp->cell->firstincell == NONODEPROTO)
					{
						efree(lnp->cell->cellname);
						freecell(lnp->cell);
					} else ttyputmsg("Facet %s did not stand alone!", lnp->cell->cellname);
				}
			}

			/* rename the main facet */
			(void)reallocstring(&c->cellname, oldname, lib->cluster);
			efree(oldname);

			/* place the facets under this cell */
			for(i=0; i<8; i++)
			{
				lnp = io_facetviews[i].facet;
				if (lnp == NONODEPROTO) continue;
				lnp->cell = c;

				/* assign a version number */
				lnp->version = 1;
				for(onp = c->firstincell; onp != NONODEPROTO; onp = onp->nextincell)
					if (onp->cellview == lnp->cellview) lnp->version = onp->version + 1;

				/* insert the facet into the cell list */
				db_insertnodeproto(lnp);
			}
			break;
		}
	}

	/* miscellaneous: facet names, view types, date info, port position, etc */
	adjust = 0;
	for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
	{
		/* search for illegal characters */
		oldname = 0;
		for(pt = np->cell->cellname; *pt != 0; pt++)
			if (*pt == ':' || *pt == ';' || *pt == '{' || *pt == '}')
		{
			if (oldname == 0)
				(void)allocstring(&oldname, np->cell->cellname, el_tempcluster);
			*pt = '_';
		}
		if (oldname != 0)
		{
			ttyputmsg("Cell '%s' renamed to '%s'", oldname, np->cell->cellname);
			efree(oldname);
		}

		/* handle facet date information */
		var = getvalkey((INTBIG)np, VNODEPROTO, VINTEGER|VISARRAY, cell_dates);
		if (var != NOVARIABLE)
		{
			np->creationdate = ((UINTBIG *)var->addr)[0];
			np->revisiondate = ((UINTBIG *)var->addr)[1];
			nextvarchangequiet();
			(void)delvalkey((INTBIG)np, VNODEPROTO, cell_dates);
		}

		/* handle original_cell attribute */
		var = getvalkey((INTBIG)np, VNODEPROTO, VSTRING, original_cell);
		if (var != NOVARIABLE)
		{
			nextvarchangequiet();
			(void)setvalkey((INTBIG)np, VNODEPROTO, original_facet, var->addr, VSTRING);
			nextvarchangequiet();
			(void)delvalkey((INTBIG)np, VNODEPROTO, original_cell);
		}

		/* ignore network information */
		var = getvalkey((INTBIG)np, VNODEPROTO, VSTRING|VISARRAY, net_name_list);
		if (var != NOVARIABLE)
		{
			nextvarchangequiet();
			(void)delvalkey((INTBIG)np, VNODEPROTO, net_name_list);
		}
		var = getvalkey((INTBIG)np, VNODEPROTO, VARCINST|VISARRAY, net_arc_list);
		if (var != NOVARIABLE)
		{
			nextvarchangequiet();
			(void)delvalkey((INTBIG)np, VNODEPROTO, net_arc_list);
		}

		/* convert and make the logic gate attributes displayable */
		for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
		{
			if (ni->proto->index == 0) continue;
			for(i=0; io_convertschem[i].nodename != 0; i++)
			{
				if (ni->proto != io_convertschem[i].type) continue;
				undogeom(ni->geom, np);
				ni->lowx -= io_convertschem[i].xoff * schemlambda / 2;
				ni->highx += io_convertschem[i].xoff * schemlambda / 2;
				ni->lowy -= io_convertschem[i].yoff * schemlambda / 2;
				ni->highy += io_convertschem[i].yoff * schemlambda / 2;
				xshift = yshift = 0;
				switch (io_convertschem[i].special)
				{
					case SHIFTRIGHT: xshift = schemlambda;  break;
					case SHIFTUP:    yshift = schemlambda;  break;
				}
				makeangle(ni->rotation, ni->transpose, trans);
				xform(xshift, yshift, &xshift, &yshift, trans);
				ni->lowx += xshift;   ni->highx += xshift;
				ni->lowy += yshift;   ni->highy += yshift;
				linkgeom(ni->geom, np);
				var = getvalkey((INTBIG)ni, VNODEINST, VSTRING, io_convertschem[i].oldkey);
				if (var != NOVARIABLE)
				{
					if (io_convertschem[i].newkey != 0)
					{
						if (io_convertschem[i].special == NOTDISPLAY)
						{
							nextvarchangequiet();
							(void)setvalkey((INTBIG)ni, VNODEINST, io_convertschem[i].newkey,
								(INTBIG)((char *)var->addr), VSTRING);
						} else
						{
							nextvarchangequiet();
							(void)setvalkey((INTBIG)ni, VNODEINST, io_convertschem[i].newkey,
								(INTBIG)((char *)var->addr), VSTRING|VDISPLAY);
						}
					}
					nextvarchangequiet();
					(void)delvalkey((INTBIG)ni, VNODEINST, io_convertschem[i].oldkey);
				}
				adjust++;
			}

			/* convert "transitor_width" variable on former serpentine nodes */
			var = getvalkey((INTBIG)ni, VNODEINST, -1, el_transistor_width);
			if (var != NOVARIABLE)
			{
				if ((var->type&VTYPE) == VINTEGER)
				{
					nextvarchangequiet();
					(void)setvalkey((INTBIG)ni, VNODEINST,
						el_transistor_width, var->addr * WHOLE, VFRACT);
				} else if ((var->type&VTYPE) == VFLOAT)
				{
					wid = castfloat(var->addr);
					twid = (INTBIG)(wid * WHOLE);
					nextvarchangequiet();
					(void)setvalkey((INTBIG)ni, VNODEINST, el_transistor_width, twid, VFRACT);
				}
			}

			/* convert "ART_font" variable on former text nodes */
			var = getvalkey((INTBIG)ni, VNODEINST, VINTEGER, art_font);
			if (var != NOVARIABLE)
			{
				var2 = getvalkey((INTBIG)ni, VNODEINST, VSTRING|VISARRAY, art_messagekey);
				if (var2 != NOVARIABLE)
				{
					var2->type |= VDISPLAY;
					var2->textdescript = (var2->textdescript & ~VTSIZE) |
						((var->addr << VTSIZESH) & VTSIZE);
				}
				nextvarchangequiet();
				(void)delvalkey((INTBIG)ni, VNODEINST, art_font);
			}

			/* check for "ART_message" with one-long array */
			var = getvalkey((INTBIG)ni, VNODEINST, VSTRING|VISARRAY, art_messagekey);
			if (var != NOVARIABLE)
			{
				descript = var->textdescript;
				var2 = getvalkey((INTBIG)ni, VNODEINST, VSTRING, original_nodename);
				if (var2 != NOVARIABLE)
				{
					pt = (char *)var2->addr;
					if (namesame(pt, "artwork:Left-Message") == 0)
					{
						xoff = (ni->highx-ni->lowx) * 2 / artlambda;
						yoff = (ni->highy-ni->lowy) * 2 / artlambda;
						descript &= ~(VTPOSITION|VTXOFF|VTYOFF);
						descript |= (xoff << VTXOFFSH) | VTXOFFNEG |
							(yoff << VTYOFFSH) | VTYOFFNEG | VTPOSUPRIGHT;
					} else if (namesame(pt, "artwork:Right-Message") == 0)
					{
						descript = (descript & ~VTPOSITION) | VTPOSUPLEFT;
						xoff = (ni->highx-ni->lowx) * 2 / artlambda;
						yoff = (ni->highy-ni->lowy) * 2 / artlambda;
						descript &= ~(VTPOSITION|VTXOFF|VTXOFFNEG|VTYOFF);
						descript |= (xoff << VTXOFFSH) | (yoff << VTYOFFSH) |
							VTYOFFNEG | VTPOSUPLEFT;
					} else if (namesame(pt, "artwork:Message") == 0)
						descript = (descript & ~VTPOSITION) | VTPOSBOXED;
				}
				if (getlength(var) == 1)
				{
					nextvarchangequiet();
					var = setvalkey((INTBIG)ni, VNODEINST, art_messagekey,
						(INTBIG)(((char **)var->addr)[0]), VSTRING|VDISPLAY);
				}
				if (var != NOVARIABLE) var->textdescript = descript;
			}
		}

		/* move port position information */
		for(pp = np->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
		{
			pp->textdescript = defaulttextdescript();
			switch ((pp->userbits&OLDPORTLABXOFF)>>OLDPORTLABXOFFSH)
			{
				case 0:		/* center */
					switch ((pp->userbits&OLDPORTLABYOFF)>>OLDPORTLABYOFFSH)
					{
						case 1:		/* top */
							pp->textdescript |= VTPOSUP;   break;
						case 2:		/* bottom */
							pp->textdescript |= VTPOSDOWN;   break;
					}
					break;
				case 1:		/* left */
					switch ((pp->userbits&OLDPORTLABYOFF)>>OLDPORTLABYOFFSH)
					{
						case 0:		/* left */
							pp->textdescript |= VTPOSLEFT;   break;
						case 1:		/* top left */
							pp->textdescript |= VTPOSUPLEFT;   break;
						case 2:		/* bottom left */
							pp->textdescript |= VTPOSDOWNLEFT;   break;
					}
					break;
				case 2:		/* right */
					switch ((pp->userbits&OLDPORTLABYOFF)>>OLDPORTLABYOFFSH)
					{
						case 0:		/* right */
							pp->textdescript |= VTPOSRIGHT;   break;
						case 1:		/* top right */
							pp->textdescript |= VTPOSUPRIGHT;   break;
						case 2:		/* bottom right */
							pp->textdescript |= VTPOSDOWNRIGHT;   break;
					}
					break;
			}
		}
	}

	/* search for serpentine transistors and convert if any are found */
	found = 0;
	for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
	{
		for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
		{
			if (ni->proto->index == 0) continue;
			if (gettrace(ni) == NOVARIABLE) continue;
			for(pp = ni->proto->firstportproto, i=0; pp!=NOPORTPROTO; pp = pp->nextportproto) i++;
			if (i < 2) continue; /* Must have at least 3 ports */
			found++;
			break;  /* Found at least one */
		}
		if (found != 0) break;
	}
	if (found != 0) io_convert_serplib(lib);
	if (adjust == 0) ttyputmsg("...done"); else
		ttyputmsg("...done (do a '-debug check-database' to correct library)");
}

/*
 * routine to convert old serpentine transistors by changing the portprotos
 * to which the arcs that are connected to the top and bottom ports on a MOS
 * transistor are connected, leaving them with the same geometrical properties
 * and locations.
 */
void io_convert_serplib(LIBRARY *lib)
{
	REGISTER NODEPROTO *np;
	REGISTER NODEINST *ni;
	REGISTER PORTPROTO *ppt, *opp;
	PORTPROTO *pp1, *pp2, *po1, *po2;
	REGISTER PORTARCINST *pi;
	REGISTER ARCINST *ai, *ait;
	REGISTER INTBIG i1, xfrom, xto, yfrom, yto;

	for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
	{
		for(ai = np->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
			ai->temp1 = 0;
		for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
		{
			/* see if it is a serpentine transistor */
			if (io_isserp(ni, &pp1, &pp2) == 0) continue;

			/* Now find the corresponding arcs */
			for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
			{
				if (pi->proto != pp1 && pi->proto != pp2) continue;
				ai = pi->conarcinst;
				if (ai->temp1 != 0) continue;    /* arc already converted */
				if (ai->end[0].nodeinst == ni) i1 = 0; else i1 = 1;
				xfrom = ai->end[i1].xpos;
				xto   = ai->end[1-i1].xpos;
				yfrom = ai->end[i1].ypos;
				yto   = ai->end[1-i1].ypos;
				if (pi->proto == pp1)   /* Supposed to be connected to Port 1? */
					ppt = pp2;
				if (pi->proto == pp2) ppt = pp1;
				opp = ai->end[1-i1].portarcinst->proto;

				/* the other end may be serpentine too! */
				if (io_isserp(ai->end[1-i1].nodeinst, &po1, &po2) != 0)
				{
					if (po1 == opp) opp = po2; else if (po2 == opp) opp = po1;
				}

				/* create the replacement arc */
				if ((ait = newarcinst(ai->proto, ai->width, ai->userbits, ni, ppt, xfrom, yfrom,
					ai->end[1-i1].nodeinst, opp, xto, yto, ni->parent)) == NOARCINST)
				{
					ttyputerr("Problem creating the arc");
					return;
				}
				(void)copyvars((INTBIG)ai, VARCINST, (INTBIG)ait, VARCINST);
				ai->temp1 = 1;      /* Flag for later erase */
				ait->temp1 = 2;
			}   /* end for */

			/* Now delete old arcs */
			for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
			{
				if (pi->proto != pp1 && pi->proto != pp2) continue;
				ai = pi->conarcinst;
				if (ai->temp1 == 1) (void)killarcinst(ai);
			}
		}
	}
}

/*
 * helper routine to determine whether node "ni" is a serpentine transistor.
 * If it is, routine returns nonzero and sets ports "pp1" and "pp2" to the
 * two diffusion ports
 */
INTSML io_isserp(NODEINST *ni, PORTPROTO **pp1, PORTPROTO **pp2)
{
	REGISTER PORTPROTO *pp;
	REGISTER INTSML i;

	/* simple tests */
	if (ni->proto->index == 0) return(0);
	if (gettrace(ni) == NOVARIABLE) return(0);

	/* Get the two connections that are already there */
	*pp1 = *pp2 = NOPORTPROTO;
	for (pp = ni->proto->firstportproto,i = 0; pp != NOPORTPROTO; pp = pp->nextportproto, i++)
	{
		if (i == 1) *pp1 = pp;
		if (i == 3) *pp2 = pp;
	}
	if (*pp1 == NOPORTPROTO || *pp2 == NOPORTPROTO) return(0);
	return(1);
}

#if IOGDS

/* GDS Arc Conversion */
DIALOGITEM io_gdsarcconversiondialogitems[] =
{
 /*  1 */ {0, {80,184,104,248}, BUTTON, "OK"},
 /*  2 */ {0, {80,24,104,88}, BUTTON, "Cancel"},
 /*  3 */ {0, {28,172,44,262}, EDITTEXT, ""},
 /*  4 */ {0, {28,12,44,157}, MESSAGE, "Maximum arc angle:"},
 /*  5 */ {0, {52,172,68,262}, EDITTEXT, ""},
 /*  6 */ {0, {52,12,68,140}, MESSAGE, "Maximum arc sag:"},
 /*  7 */ {0, {4,4,20,294}, MESSAGE, "Rules for converting arcs to line segments"}
};
DIALOG io_gdsarcconversiondialog = {{50,75,165,374}, "Arc Conversion Rules", 7, io_gdsarcconversiondialogitems};

void io_gdsarcconversiondlg(void)
{
	REGISTER INTSML itemHit;
	INTBIG arcres, arcsag;

	if (DiaInitDialog(&io_gdsarcconversiondialog) != 0) return;

	/* default number of changes to undo */
	getcontoursegmentparameters(&arcres, &arcsag);
	DiaSetText(3, frtoa(arcres*WHOLE/10));
	DiaSetText(5, latoa(arcsag));

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == CANCEL) break;
		if (itemHit == OK) break;
	}

	if (itemHit != CANCEL)
	{
		arcres = atofr(DiaGetText(3)) * 10 / WHOLE;
		arcsag = atola(DiaGetText(5));
		setcontoursegmentparameters(arcres, arcsag);
	}
	DiaDoneDialog();
}

void io_compute_center(INTBIG xc, INTBIG yc, INTBIG x1, INTBIG y1,
	INTBIG x2, INTBIG y2, INTBIG *cx, INTBIG *cy)
{
	int r, dx, dy, a1, a2, a;
	double pie, theta, radius, Dx, Dy;

	/* reconstruct angles to p1 and p2 */
	Dx = x1 - xc;
	Dy = y1 - yc;
	radius = sqrt(Dx * Dx + Dy * Dy);
	r = rounddouble(radius);
	a1 = (int)io_calc_angle(r, (double)(x1 - xc), (double)(y1 - yc));
	a2 = (int)io_calc_angle(r, (double)(x2 - xc), (double)(y2 - yc));
	if (a1 < a2) a1 += 3600;
	a = (a1 + a2) >> 1;
	pie = acos(-1.0);
	theta = (double) a *pie / 1800.0;	/* in radians */
	Dx = radius * cos(theta);
	Dy = radius * sin(theta);
	dx = rounddouble(Dx);
	dy = rounddouble(Dy);
	*cx = xc + dx;
	*cy = yc + dy;
}

double io_calc_angle(double r, double dx, double dy)
{
	double ratio, a1, a2;

	ratio = 1800.0 / EPI;
	a1 = acos(dx/r) * ratio;
	a2 = asin(dy/r) * ratio;
	if (a2 < 0.0) return(3600.0 - a1);
	return(a1);
}

#endif
