/*
 * Electric(tm) VLSI Design System
 *
 * File: iobinaryi.c
 * Input/output aid: binary format input
 * Written by: Steven M. Rubin, Electric Editor Incorporated
 *
 * Copyright (c) 1998 Electric Editor Incorporated.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Electric Editor Incorporated
 * 23470 Sunset Drive, Suite 108
 * Los Gatos, California 95033
 * support@electriceditor.com
 */

#include "global.h"
#include "database.h"
#include "eio.h"
#include "tecgen.h"
#include "tecschem.h"
#include "edialogs.h"

#define	REPORTINC   2000				/* bytes between graphical updates */

static INTSML io_aidbitsmessed;			/* 0: aid order is correct */
static char  *io_tempstring;			/* for reading temporary strings */
static INTSML io_tempstringlength = 0;	/* length of temporary string */
static INTBIG io_aabcount;				/* number of aidbits for old files */
static INTBIG io_bytecount;				/* position within input file */
static char **io_realname;				/* variable names */
static INTBIG io_namecount;				/* number of variable names */
static INTBIG io_filelength;			/* length of the file */
static INTBIG io_reported;				/* reported progress in file */

/* totals of objects in the file */
static INTBIG io_magic;					/* file's magic number */
static INTBIG io_aacount;				/* number of aids */
static INTBIG io_techcount;				/* number of technologies */
static INTBIG io_nodepprotoindex;		/* number of primitive node prototypes */
static INTBIG io_portpprotoindex;		/* number of primitive port prototypes */
static INTBIG io_arcprotoindex;			/* number of arc prototypes */
static INTBIG io_nodeprotoindex;		/* number of facets */
static INTBIG io_nodeindex;				/* number of node instances */
static INTBIG io_portprotoindex;		/* number of port prototypes */
static INTBIG io_arcindex;				/* number of arc instances */
static INTBIG io_geomindex;				/* number of geometry modules */
static INTBIG io_cellindex;				/* number of cells */
static INTBIG io_curnodeproto;			/* current facet */

/* input error message arrays */
static char  **io_techerror;			/* name of unknown technologies */
static char  **io_aiderror;				/* name of unknown aids */
static INTSML *io_nodepprotoerror;		/* flag for unknown prim. NODEPROTOs */
static char  **io_nodepprotoorig;		/* name of unknown prim. NODEPROTOs */
static INTBIG *io_nodepprototech;		/* tech. assoc. with prim. NODEPROTOs */
static char  **io_portpprotoerror;		/* name of unknown PORTPROTOs */
static char  **io_arcprotoerror;		/* name of unknown ARCPROTOs */
static INTSML  io_swap_bytes;			/* are the bytes swapped? 1 if so */
static INTBIG  io_swapped_floats;		/* number of swappped floating-point variables */
static INTBIG  io_swapped_doubles;		/* number of swappped double-precision variables */

/* prototypes for local routines */
char *io_doreadlibrary(LIBRARY*);
INTSML io_readnodeproto(NODEPROTO*);
INTSML io_readnodeinst(NODEINST*);
void io_readgeom(INTBIG*, INTBIG*);
INTSML io_readarcinst(ARCINST*);
INTSML io_readnamespace(void);
void io_ignorevariables(void);
INTSML io_readvariables(INTBIG, INTBIG);
INTSML io_getinvar(INTBIG*, INTBIG);
INTBIG io_arrangeaidbits(INTBIG);
char *io_convertnodeproto(NODEPROTO**);
void io_convertportproto(PORTPROTO**);
void io_convertarcproto(ARCPROTO**);
ARCPROTO *io_getarcprotolist(INTSML);
PORTPROTO *io_getportpprotolist(INTBIG);
NODEPROTO *io_getnodepprotolist(INTBIG);
TECHNOLOGY *io_gettechlist(INTSML);
void io_getin(UINTBIG, INTSML);
char *io_getstring(CLUSTER*);
char *io_gettempstring(void);
INTBIG io_swap32(INTBIG*);
INTSML io_swap16(INTSML*);

INTSML io_readbinlibrary(LIBRARY *lib)
{
	REGISTER char *result;
	char *filename;
	extern DIALOG usr_progressdialog;

	result = truepath(lib->libfile);
	io_filein = xopen(result, FILETYPEBLIB, el_libdir, &filename);
	if (io_filein == NULL)
	{
		ttyputerr("File '%s' not found", result);
		return(1);
	}

	io_reported = 0;
	if (io_verbose < 0)
	{
		io_filelength = filesize(io_filein);
		if (io_filelength > 0)
		{
			if (DiaInitDialog(&usr_progressdialog) != 0)
			{
				xclose(io_filein);
				return(1);
			}
			DiaPercent(1, 0);
			(void)initinfstr();
			(void)addstringtoinfstr("Reading library ");
			(void)addstringtoinfstr(lib->libname);
			(void)addstringtoinfstr("...");
			DiaSetText(2, returninfstr());
		}
	} else io_filelength = 0;

	/* now read the file */
	io_bytecount = 0;
	io_swap_bytes = 0;      /* assume regular format */
	io_swapped_floats = io_swapped_doubles = 0;
	result = io_doreadlibrary(lib);
	xclose(io_filein);
	if (io_verbose < 0 && io_filelength > 0) DiaDoneDialog();
	if (io_swapped_floats != 0)
		ttyputmsgf("Read %d swapped float variables; may be incorrect", io_swapped_floats);
	if (io_swapped_doubles != 0)
		ttyputmsgf("Read %d swapped double variables; may be incorrect", io_swapped_doubles);
	if (*result == 0) return(0);
	ttyputerr("%s", result);
	return(1);
}

char *io_doreadlibrary(LIBRARY *lib)
{
	REGISTER INTSML te, imosconv, oldunit;
	REGISTER INTBIG i, thisone, *io_geomtype, *io_geommoreup, top, bot, look;
	INTBIG num, den;
	INTBIG j, count, cou, arcinstpos, nodeinstpos, portprotopos;
	REGISTER char *name, *sname;
	REGISTER NODEPROTO *np, *onp, *lastnp, *ent;
	REGISTER NODEINST *ni;
	GEOM *geom;
	REGISTER ARCPROTO *ap;
	REGISTER ARCINST *ai;
	REGISTER PORTPROTO *pp;
	REGISTER TECHNOLOGY *tech;
	REGISTER PORTEXPINST **io_portexpinstlist;
	REGISTER PORTARCINST **io_portarcinstlist;
	char *version;
	REGISTER VIEW *v;
	REGISTER CELL *c;
	REGISTER LIBRARY *savelib;

	if (setjmp(io_filerror)) return("error reading file");

	/* read magic number */
	io_getin((UINTBIG)&io_magic, SIZEOFINTBIG);
	if (io_magic != MAGIC1 && io_magic != MAGIC2 && io_magic != MAGIC3 &&
		io_magic != MAGIC4 && io_magic != MAGIC5 && io_magic != MAGIC6 &&
		io_magic != MAGIC7 && io_magic != MAGIC8 && io_magic != MAGIC9)
	{
		io_magic = io_swap32(&io_magic); /* try swapping the bytes */
		if (io_magic != MAGIC1 && io_magic != MAGIC2 && io_magic != MAGIC3 &&
			io_magic != MAGIC4 && io_magic != MAGIC5 && io_magic != MAGIC6 &&
			io_magic != MAGIC7 && io_magic != MAGIC8 && io_magic != MAGIC9)
				return("Bad file format"); else
		{
			io_swap_bytes = 1;
			ttyputmsg("File has swapped bytes in the header. Will attempt to read it.");
		}
	}
	if (io_verbose > 0) switch (io_magic)
	{
		case MAGIC1: ttyputmsg("This library is 8 versions old");   break;
		case MAGIC2: ttyputmsg("This library is 7 versions old");   break;
		case MAGIC3: ttyputmsg("This library is 6 versions old");   break;
		case MAGIC4: ttyputmsg("This library is 5 versions old");   break;
		case MAGIC5: ttyputmsg("This library is 4 versions old");   break;
		case MAGIC6: ttyputmsg("This library is 3 versions old");   break;
		case MAGIC7: ttyputmsg("This library is 2 versions old");   break;
		case MAGIC8: ttyputmsg("This library is 1 version old");    break;
	}

	/* get count of objects in the file */
	io_getin((UINTBIG)&io_aacount, SIZEOFINTBIG);
	io_getin((UINTBIG)&io_techcount, SIZEOFINTBIG);
	io_getin((UINTBIG)&io_nodepprotoindex, SIZEOFINTBIG);
	io_getin((UINTBIG)&io_portpprotoindex, SIZEOFINTBIG);
	io_getin((UINTBIG)&io_arcprotoindex, SIZEOFINTBIG);
	io_getin((UINTBIG)&io_nodeprotoindex, SIZEOFINTBIG);
	io_getin((UINTBIG)&io_nodeindex, SIZEOFINTBIG);
	io_getin((UINTBIG)&io_portprotoindex, SIZEOFINTBIG);
	io_getin((UINTBIG)&io_arcindex, SIZEOFINTBIG);
	io_getin((UINTBIG)&io_geomindex, SIZEOFINTBIG);
	if (io_magic <= MAGIC9) io_getin((UINTBIG)&io_cellindex, SIZEOFINTBIG); else
		io_cellindex = io_nodeprotoindex;
	io_getin((UINTBIG)&io_curnodeproto, SIZEOFINTBIG);

	/* get the Electric version (version 8 and later) */
	if (io_magic <= MAGIC8) version = io_getstring(el_tempcluster); else
		(void)allocstring(&version, "3.35", el_tempcluster);

	/* get the newly created views (version 9 and later) */
	for(v = el_views; v != NOVIEW; v = v->nextview) v->temp1 = 0;
	el_unknownview->temp1 = -1;
	el_layoutview->temp1 = -2;
	el_schematicview->temp1 = -3;
	el_iconview->temp1 = -4;
	el_simsnapview->temp1 = -5;
	el_skeletonview->temp1 = -6;
	el_vhdlview->temp1 = -7;
	el_netlistview->temp1 = -8;
	el_docview->temp1 = -9;
	el_netlistnetlispview->temp1 = -10;
	el_netlistalsview->temp1 = -11;
	el_netlistquiscview->temp1 = -12;
	el_netlistrsimview->temp1 = -13;
	el_netlistsilosview->temp1 = -14;
	if (io_magic <= MAGIC9)
	{
		io_getin((UINTBIG)&j, SIZEOFINTBIG);
		for(i=0; i<j; i++)
		{
			name = io_getstring(db_cluster);
			sname = io_getstring(db_cluster);
			v = getview(name);
			if (v == NOVIEW)
			{
				v = allocview();
				if (v == NOVIEW) return("No memory");
				v->viewname = name;
				v->sviewname = sname;
				v->nextview = el_views;
				el_views = v;
			} else
			{
				efree(name);
				efree(sname);
			}
			v->temp1 = i + 1;
		}
	}

	/* get the number of aidbits to ignore */
	if (io_magic <= MAGIC3 && io_magic >= MAGIC6)
	{
		/* versions 3, 4, 5, and 6 find this in the file */
		io_getin((UINTBIG)&io_aabcount, SIZEOFINTBIG);
	} else
	{
		/* versions 1 and 2 compute this (versions 7 and later ignore it) */
		io_aabcount = io_aacount;
	}

	if (io_verbose > 0)
	{
		ttyputmsg("Reading %d aids, %d technologies", io_aacount, io_techcount);
		ttyputmsg("        %d prim nodes, %d prim ports, %d arc protos",
			io_nodepprotoindex, io_portpprotoindex, io_arcprotoindex);
		ttyputmsg("        %d cells, %d facets, %d ports, %d nodes, %d arcs",
			io_cellindex, io_nodeprotoindex, io_portprotoindex, io_nodeindex,
				io_arcindex);
	}

	/* erase the current database */
	if (io_verbose > 0) ttyputmsg("Erasing old library");
	eraselibrary(lib);

	/* setup pointers */
	io_nodelist = (NODEINST **)emalloc(((sizeof (NODEINST *)) * io_nodeindex), el_tempcluster);
	if (io_nodelist == 0) return("No memory");
	io_nodecount = emalloc((SIZEOFINTBIG * io_nodeprotoindex), el_tempcluster);
	if (io_nodecount == 0) return("No memory");
	io_nodeprotolist = (NODEPROTO **)emalloc(((sizeof (NODEPROTO *)) * io_nodeprotoindex), el_tempcluster);
	if (io_nodeprotolist == 0) return("No memory");
	io_portprotolist = (PORTPROTO **)emalloc(((sizeof (PORTPROTO *)) * io_portprotoindex), el_tempcluster);
	if (io_portprotolist == 0) return("No memory");
	io_portcount = emalloc((SIZEOFINTBIG * io_nodeprotoindex), el_tempcluster);
	if (io_portcount == 0) return("No memory");
	io_portpprotolist = (PORTPROTO **)emalloc(((sizeof (PORTPROTO *)) * io_portpprotoindex), el_tempcluster);
	if (io_portpprotolist == 0) return("No memory");
	io_portpprotoerror = (char **)emalloc(((sizeof (char *)) * io_portpprotoindex), el_tempcluster);
	if (io_portpprotoerror == 0) return("No memory");
	io_portexpinstlist = (PORTEXPINST **)emalloc(((sizeof (PORTEXPINST *)) * io_portprotoindex), el_tempcluster);
	if (io_portexpinstlist == 0) return("No memory");
	io_portarcinstlist = (PORTARCINST **)emalloc(((sizeof (PORTARCINST *)) * io_arcindex * 2), el_tempcluster);
	if (io_portarcinstlist == 0) return("No memory");
	io_arclist = (ARCINST **)emalloc(((sizeof (ARCINST *)) * io_arcindex), el_tempcluster);
	if (io_arclist == 0) return("No memory");
	io_arccount = emalloc((SIZEOFINTBIG * io_nodeprotoindex), el_tempcluster);
	if (io_arccount == 0) return("No memory");
	io_nodepprotolist = (NODEPROTO **)emalloc(((sizeof (NODEPROTO *)) * io_nodepprotoindex), el_tempcluster);
	if (io_nodepprotolist == 0) return("No memory");
	io_nodepprototech = emalloc((SIZEOFINTBIG * io_nodepprotoindex), el_tempcluster);
	if (io_nodepprototech == 0) return("No memory");
	io_nodepprotoerror = (INTSML *)emalloc((SIZEOFINTSML * io_nodepprotoindex), el_tempcluster);
	if (io_nodepprotoerror == 0) return("No memory");
	io_nodepprotoorig = (char **)emalloc(((sizeof (char *)) * io_nodepprotoindex), el_tempcluster);
	if (io_nodepprotoorig == 0) return("No memory");
	io_arcprotolist = (ARCPROTO **)emalloc(((sizeof (ARCPROTO *)) * io_arcprotoindex), el_tempcluster);
	if (io_arcprotolist == 0) return("No memory");
	io_arcprotoerror = (char **)emalloc(((sizeof (char *)) * io_arcprotoindex), el_tempcluster);
	if (io_arcprotoerror == 0) return("No memory");
	io_techlist = (TECHNOLOGY **)emalloc(((sizeof (TECHNOLOGY *)) * io_techcount), el_tempcluster);
	if (io_techlist == 0) return("No memory");
	io_techerror = (char **)emalloc(((sizeof (char *)) * io_techcount), el_tempcluster);
	if (io_techerror == 0) return("No memory");
	io_aidlist = emalloc((SIZEOFINTBIG * io_aacount), el_tempcluster);
	if (io_aidlist == 0) return("No memory");
	io_aiderror = (char **)emalloc(((sizeof (char *)) * io_aacount), el_tempcluster);
	if (io_aiderror == 0) return("No memory");

	/* versions 9 and later get cell information */
	if (io_magic <= MAGIC9)
	{
		io_celllist = (CELL **)emalloc(((sizeof (CELL *)) * io_cellindex), el_tempcluster);
		if (io_celllist == 0) return("No memory");
	}

	if (io_magic > MAGIC5)
	{
		/* versions 4 and earlier must use some geometric data */
		io_geomtype = emalloc((SIZEOFINTBIG * io_geomindex), el_tempcluster);
		if (io_geomtype == 0) return("No memory");
		io_geommoreup = emalloc((SIZEOFINTBIG * io_geomindex), el_tempcluster);
		if (io_geommoreup == 0) return("No memory");
		if (io_geomtype == 0 || io_geommoreup == 0) return("No memory");
	}
	if (io_verbose > 0) ttyputmsg("Allocated pointer memory");

	/* get number of arcinsts and nodeinsts in the facet */
	if (io_magic != MAGIC1)
	{
		/* versions 2 and later find this in the file */
		nodeinstpos = arcinstpos = portprotopos = 0;
		for(i=0; i<io_nodeprotoindex; i++)
		{
			io_getin((UINTBIG)&io_arccount[i], SIZEOFINTBIG);
			arcinstpos += io_arccount[i];
			io_getin((UINTBIG)&io_nodecount[i], SIZEOFINTBIG);
			nodeinstpos += io_nodecount[i];
			io_getin((UINTBIG)&io_portcount[i], SIZEOFINTBIG);
			portprotopos += io_portcount[i];
		}

		/* verify that the number of node instances is equal to the total in the file */
		if (nodeinstpos != io_nodeindex)
		{
			ttyputerr("Error: facets have %d nodes but library has %d", nodeinstpos, io_nodeindex);
			return("Bad file");
		}
		if (arcinstpos != io_arcindex)
		{
			ttyputerr("Error: facets have %d arcs but library has %d", arcinstpos, io_arcindex);
			return("Bad file");
		}
		if (portprotopos != io_portprotoindex)
		{
			ttyputerr("Error: facets have %d ports but library has %d", portprotopos, io_portprotoindex);
			return("Bad file");
		}
	} else
	{
		/* version 1 computes this information */
		io_arccount[0] = io_arcindex;
		io_nodecount[0] = io_nodeindex;
		io_portcount[0] = io_portprotoindex;
		for(i=1; i<io_nodeprotoindex; i++)
			io_arccount[i] = io_nodecount[i] = io_portcount[i] = 0;
	}

	/* allocate all cells and facets in the library */
	lib->firstcell = NOCELL;

	/* versions 9 and later allocate cells */
	if (io_magic <= MAGIC9)
	{
		for(i=0; i<io_cellindex; i++)
		{
			io_celllist[i] = alloccell(lib->cluster);
			if (io_celllist[i] == NOCELL) return("No memory");
			io_celllist[i]->cluster = lib->cluster;
			io_celllist[i]->lib = lib;
			if (i > 0) io_celllist[i-1]->nextcell = io_celllist[i]; else
				lib->firstcell = io_celllist[0];
		}
	}
	for(i=0; i<io_nodeprotoindex; i++)
	{
		io_nodeprotolist[i] = allocnodeproto(lib->cluster);
		if (io_nodeprotolist[i] == NONODEPROTO) return("No memory");

		/* versions 8 and earlier assume each facet is a cell */
		if (io_magic >= MAGIC8)
		{
			c = alloccell(lib->cluster);
			if (c == NOCELL) return("No memory");
			c->cluster = lib->cluster;
			io_nodeprotolist[i]->cell = c;
			c->firstincell = io_nodeprotolist[i];
			c->lib = lib;
			if (i > 0) io_nodeprotolist[i-1]->cell->nextcell = c; else
				lib->firstcell = io_nodeprotolist[0]->cell;
		}
		io_nodeprotolist[i]->cellview = el_unknownview;
		io_nodeprotolist[i]->newestversion = io_nodeprotolist[i];
	}
	if (io_verbose > 0) ttyputmsg("Allocated facets");

	/* allocate the objects in the library, facet by facet */
	nodeinstpos = arcinstpos = portprotopos = 0;
	for(i=0; i<io_nodeprotoindex; i++)
	{
		np = io_nodeprotolist[i];

		/* allocate node instances in this facet */
		allocnodeinst(io_nodecount[i], &io_nodelist[nodeinstpos], lib->cluster);
		if (io_nodecount[i] == 0) np->firstnodeinst = NONODEINST; else
			np->firstnodeinst = io_nodelist[nodeinstpos];
		for(j=0; j<io_nodecount[i]; j++)
		{
			ni = io_nodelist[j+nodeinstpos];
			if (ni == NONODEINST) return("No memory");
			allocgeom(1, &geom, lib->cluster);
			if (geom == NOGEOM) return("No memory");
			ni->geom = geom;

			/* compute linked list of nodes in this facet */
			if (j == 0) ni->lastnodeinst = NONODEINST; else
				ni->lastnodeinst = io_nodelist[j+nodeinstpos-1];
			if (j+1 == io_nodecount[i]) ni->nextnodeinst = NONODEINST; else
				ni->nextnodeinst = io_nodelist[j+nodeinstpos+1];
		}
		nodeinstpos += io_nodecount[i];

		/* allocate port prototypes and port exp instances in this facet */
		allocportproto(io_portcount[i], &io_portprotolist[portprotopos], lib->cluster);
		allocportexpinst(io_portcount[i], &io_portexpinstlist[portprotopos], lib->cluster);
		for(j=0; j<io_portcount[i]; j++)
		{
			thisone = j + portprotopos;
			if (io_portexpinstlist[thisone] == NOPORTEXPINST ||
				io_portprotolist[thisone] == NOPORTPROTO) return("No memory");
			io_portprotolist[thisone]->subportexpinst = io_portexpinstlist[thisone];
		}
		portprotopos += io_portcount[i];

		/* allocate arc instances and port arc instances in this facet */
		allocarcinst(io_arccount[i], &io_arclist[arcinstpos], lib->cluster);
		allocportarcinst(io_arccount[i]*2, &io_portarcinstlist[arcinstpos*2], lib->cluster);
		if (io_arccount[i] == 0) np->firstarcinst = NOARCINST; else
			np->firstarcinst = io_arclist[arcinstpos];
		for(j=0; j<io_arccount[i]; j++)
		{
			thisone = j + arcinstpos;
			ai = io_arclist[thisone];
			if (ai == NOARCINST || io_portarcinstlist[thisone*2] == NOPORTARCINST ||
				io_portarcinstlist[thisone*2+1] == NOPORTARCINST) return("No memory");
			allocgeom(1, &geom, lib->cluster);
			if (geom == NOGEOM) return("No memory");
			ai->geom = geom;
			ai->end[0].portarcinst = io_portarcinstlist[thisone*2];
			ai->end[1].portarcinst = io_portarcinstlist[thisone*2+1];

			/* compute linked list of arcs in this facet */
			if (j == 0) ai->lastarcinst = NOARCINST; else
				ai->lastarcinst = io_arclist[j+arcinstpos-1];
			if (j+1 == io_arccount[i]) ai->nextarcinst = NOARCINST; else
				ai->nextarcinst = io_arclist[j+arcinstpos+1];
		}
		arcinstpos += io_arccount[i];

		if (io_verbose > 0)
			ttyputmsg("Allocated %d arcs, %d nodes, %d ports for facet %d",
				io_arccount[i], io_nodecount[i], io_portcount[i], i);
	}

	efree((char *)io_portarcinstlist);
	efree((char *)io_portexpinstlist);

	if (io_verbose > 0) ttyputmsg("Setting technology pointers");

	/* setup pointers for technologies and primitives */
	io_nodepprotoindex = 0;
	io_portpprotoindex = 0;
	io_arcprotoindex = 0;
	for(te=0; te<io_techcount; te++)
	{
		/* associate the technology name with the true technology */
		name = io_gettempstring();
		if (name == 0) return("No memory");
		tech = gettechnology(name);

		/* conversion code for old technologies */
		imosconv = 0;
		if (tech == NOTECHNOLOGY)
		{
			if (namesame(name, "imos") == 0)
			{
				tech = gettechnology("mocmos");
				if (tech != NOTECHNOLOGY) imosconv++;
			} else if (namesame(name, "logic") == 0) tech = sch_tech;
		}

		if (tech == NOTECHNOLOGY)
		{
			tech = el_technologies;
			(void)allocstring(&io_techerror[te], name, el_tempcluster);
		} else io_techerror[te] = 0;
		io_techlist[te] = tech;

		/* get the number of primitive node prototypes */
		io_getin((UINTBIG)&count, SIZEOFINTBIG);
		for(j=0; j<count; j++)
		{
			io_nodepprotoorig[io_nodepprotoindex] = 0;
			io_nodepprotoerror[io_nodepprotoindex] = 0;
			name = io_gettempstring();
			if (name == 0) return("No memory");
			if (imosconv != 0) name += 6;
			for(np = tech->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
				if (strcmp(np->primname, name) == 0) break;
			if (np == NONODEPROTO)
			{
				if (io_verbose > 0)
					ttyputmsg("No node exactly named %s in technology %s", name, tech->techname);

				/* look for substring name match at start of name */
				i = strlen(name);
				for(np = tech->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
					if (namesamen(np->primname, name, (INTSML)(mini(strlen(np->primname), i))) == 0)
						break;

				/* look for substring match at end of name */
				if (np == NONODEPROTO)
				{
					for(np = tech->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
					{
						thisone = strlen(np->primname);
						if (i >= thisone) continue;
						if (namesame(&np->primname[thisone-i], name) == 0) break;
					}
				}

				/* special cases: convert "message" and "cell-center" nodes */
				if (np == NONODEPROTO)
				{
					if (namesame(name, "Cell-Center") == 0) np = gen_facetcenterprim; else
					if (namesame(name, "Message") == 0 ||
						namesame(name, "Centered-Message") == 0 ||
						namesame(name, "Left-Message") == 0 ||
						namesame(name, "Right-Message") == 0) np = gen_invispinprim;
				}

				/* give up and use first primitive in this technology */
				if (np == NONODEPROTO) np = tech->firstnodeproto;

				/* construct the error message */
				(void)initinfstr();
				if (io_techerror[te] != 0) (void)addstringtoinfstr(io_techerror[te]); else
					(void)addstringtoinfstr(tech->techname);
				(void)addtoinfstr(':');
				(void)addstringtoinfstr(name);
				(void)allocstring(&io_nodepprotoorig[io_nodepprotoindex], returninfstr(), el_tempcluster);
				io_nodepprotoerror[io_nodepprotoindex] = 1;
			}
			io_nodepprototech[io_nodepprotoindex] = te;
			io_nodepprotolist[io_nodepprotoindex] = np;

			/* get the number of primitive port prototypes */
			io_getin((UINTBIG)&cou, SIZEOFINTBIG);
			for(i=0; i<cou; i++)
			{
				io_portpprotoerror[io_portpprotoindex] = 0;
				name = io_gettempstring();
				if (name == 0) return("No memory");
				for(pp = np->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
					if (strcmp(pp->protoname, name) == 0) break;
				if (pp == NOPORTPROTO)
				{
					pp = np->firstportproto;
					if (io_nodepprotoerror[io_nodepprotoindex] == 0)
					{
						(void)initinfstr();
						(void)addstringtoinfstr(name);
						(void)addstringtoinfstr(" on ");
						if (io_nodepprotoorig[io_nodepprotoindex] != 0)
							(void)addstringtoinfstr(io_nodepprotoorig[io_nodepprotoindex]); else
						{
							if (io_techerror[te] != 0)
								(void)addstringtoinfstr(io_techerror[te]); else
									(void)addstringtoinfstr(tech->techname);
							(void)addtoinfstr(':');
							(void)addstringtoinfstr(np->primname);
						}
						(void)allocstring(&io_portpprotoerror[io_portpprotoindex],
							returninfstr(), el_tempcluster);
					}
				}
				io_portpprotolist[io_portpprotoindex++] = pp;
			}
			io_nodepprotoindex++;
		}

		/* get the number of arc prototypes */
		io_getin((UINTBIG)&count, SIZEOFINTBIG);
		for(j=0; j<count; j++)
		{
			io_arcprotoerror[io_arcprotoindex] = 0;
			name = io_gettempstring();
			if (name == 0) return("No memory");
			if (imosconv != 0) name += 6;
			for(ap=tech->firstarcproto; ap!=NOARCPROTO; ap=ap->nextarcproto)
				if (strcmp(ap->protoname, name) == 0) break;
			if (ap == NOARCPROTO)
			{
				ap = tech->firstarcproto;
				(void)initinfstr();
				if (io_techerror[te] != 0)
					(void)addstringtoinfstr(io_techerror[te]); else
						(void)addstringtoinfstr(tech->techname);
				(void)addtoinfstr(':');
				(void)addstringtoinfstr(name);
				(void)allocstring(&io_arcprotoerror[io_arcprotoindex], returninfstr(), el_tempcluster);
			}
			io_arcprotolist[io_arcprotoindex++] = ap;
		}
	}

	if (io_verbose > 0) ttyputmsg("Setting Tool information");

	/* setup pointers for aids */
	io_aidbitsmessed = 0;
	for(i=0; i<io_aacount; i++)
	{
		name = io_gettempstring();
		if (name == 0) return("No memory");
		io_aiderror[i] = 0;
		for(j=0; j<el_maxaid; j++)
			if (strcmp(name, el_aids[j].aidname) == 0) break;
		if (j >= el_maxaid)
		{
			(void)allocstring(&io_aiderror[i], name, el_tempcluster);
			j = -1;
		}
		if (i != j) io_aidbitsmessed++;
		io_aidlist[i] = j;
	}

	/* versions 3, 4, 5, and 6 must ignore aidbits associations */
	if (io_magic <= MAGIC3 && io_magic >= MAGIC6)
		for(i=0; i<io_aabcount; i++) (void)io_gettempstring();

	/* get the library userbits */
	if (io_magic <= MAGIC7)
	{
		/* version 7 and later simply read the relevant data */
		io_getin((UINTBIG)&lib->userbits, SIZEOFINTBIG);
	} else
	{
		/* version 6 and earlier must sift through the information */
		if (io_aabcount >= 1) io_getin((UINTBIG)&lib->userbits, SIZEOFINTBIG);
		for(i=1; i<io_aabcount; i++) io_getin((UINTBIG)&j, SIZEOFINTBIG);
	}
	lib->userbits &= ~LIBCHANGED;
	lib->userbits |= READFROMDISK;

	if (io_verbose > 0) ttyputmsg("Reading lambda values");

	/* initialize all lambda values */
	for(tech = el_technologies; tech != NOTECHNOLOGY; tech = tech->nexttechnology)
		lib->lambda[tech->index] = tech->deflambda;

	/* set the lambda values in the library */
	for(i=0; i<io_techcount; i++)
	{
		io_getin((UINTBIG)&j, SIZEOFINTBIG);
		if (io_techerror[i] != 0) continue;
		tech = io_techlist[i];

		/* for version 4 or earlier, scale lambda by 20 */
		if (atoi(version) <= 4) j *= 20;

		/* account for any differences of internal unit in this library */
		oldunit = ((lib->userbits & LIBUNITS) >> LIBUNITSSH) << INTERNALUNITSSH;
		if (oldunit != (el_units&INTERNALUNITS))
		{
			db_getinternalunitscale(&num, &den, el_units, oldunit);
			j = muldiv(j, den, num);
		}

		/* if this is to be the current library, adjust technologies */
		if (lib == el_curlib) changelambda(tech->deflambda, j, tech);
		lib->lambda[tech->index] = j;
	}

	/* read the global namespace */
	io_realname = 0;
	io_newnames = 0;
	if (io_readnamespace() != 0) return("No memory");

	/* read the library variables */
	if (io_verbose > 0) ttyputmsg("Reading library variables");
	if (io_readvariables((INTBIG)lib, VLIBRARY) < 0) return("No memory");

	/* read the aid variables */
	if (io_verbose > 0) ttyputmsg("Reading aid variables");
	for(i=0; i<io_aacount; i++)
	{
		j = io_aidlist[i];
		if (j < 0) io_ignorevariables(); else
			if (io_readvariables((INTBIG)&el_aids[j], VAID) < 0)
				return("No memory");
	}

	/* read the technology variables */
	if (io_verbose > 0) ttyputmsg("Reading technology/primitive variables");
	for(te=0; te<io_techcount; te++)
	{
		tech = io_techlist[te];
		j = io_readvariables((INTBIG)tech, VTECHNOLOGY);
		if (j < 0) return("No memory");
		if (j > 0) (void)io_gettechlist(te);
	}

	/* read the arcproto variables */
	for(i=0; i<io_arcprotoindex; i++)
	{
		ap = io_arcprotolist[i];
		j = io_readvariables((INTBIG)ap, VARCPROTO);
		if (j < 0) return("No memory");
		if (j > 0) (void)io_getarcprotolist((INTSML)i);
	}

	/* read the primitive nodeproto variables */
	for(i=0; i<io_nodepprotoindex; i++)
	{
		np = io_nodepprotolist[i];
		j = io_readvariables((INTBIG)np, VNODEPROTO);
		if (j < 0) return("No memory");
		if (j > 0) (void)io_getnodepprotolist(i);
	}

	/* read the primitive portproto variables */
	for(i=0; i<io_portpprotoindex; i++)
	{
		pp = io_portpprotolist[i];
		j = io_readvariables((INTBIG)pp, VPORTPROTO);
		if (j < 0) return("No memory");
		if (j > 0) (void)io_getportpprotolist(i);
	}

	/* read the view variables (version 9 and later) */
	if (io_magic <= MAGIC9)
	{
		io_getin((UINTBIG)&count, SIZEOFINTBIG);
		for(i=0; i<count; i++)
		{
			io_getin((UINTBIG)&j, SIZEOFINTBIG);
			for(v = el_views; v != NOVIEW; v = v->nextview)
				if (v->temp1 == j) break;
			if (v == NOVIEW)
			{
				ttyputmsg("View index %d not found", j);
				io_ignorevariables();
				continue;
			}
			if (io_readvariables((INTBIG)v, VVIEW) < 0) return("No memory");
		}
	}

	/* read the cells (version 9 and later) */
	if (io_magic <= MAGIC9)
	{
		for(i=0; i<io_cellindex; i++)
		{
			c = io_celllist[i];
			c->lib = lib;
			c->cellname = io_getstring(lib->cluster);

			/* read variable information */
			if (io_readvariables((INTBIG)c, VCELL) < 0) return("No memory");
		}
	}

	/* read the facets */
	io_portprotoindex = 0;
	for(i=0; i<io_nodeprotoindex; i++)
	{
		np = io_nodeprotolist[i];
		if (io_readnodeproto(np) != 0) return("No memory");
		if (i == 0) np->lastnodeproto = NONODEPROTO; else
			np->lastnodeproto = io_nodeprotolist[i-1];
		if (i == io_nodeprotoindex-1) np->nextnodeproto = NONODEPROTO; else
			np->nextnodeproto = io_nodeprotolist[i+1];
	}
	if (io_verbose > 0) ttyputmsg("Done reading facets");

	/* read the facet contents: arcs, nodes, geoms */
	io_nodeindex = io_arcindex = io_geomindex = 0;
	for(i=0; i<io_nodeprotoindex; i++)
	{
		if (stopping("Binary input") != 0) return("Library incomplete");
		np = io_nodeprotolist[i];
		if (io_verbose != 0)
		{
			savelib = el_curlib;   el_curlib = lib;
			if (io_verbose < 0 && io_filelength > 0)
			{
				(void)initinfstr();
				(void)addstringtoinfstr("Reading ");
				(void)addstringtoinfstr(describenodeproto(np));
				DiaSetText(2, returninfstr());
			} else ttyputmsg("Reading %s", describenodeproto(np));
			el_curlib = savelib;
		}
		if (io_magic > MAGIC5)
		{
			/* versions 4 and earlier must read some geometric information */
			j = io_geomindex;
			io_readgeom(&io_geomtype[j], &io_geommoreup[j]);   j++;
			io_readgeom(&io_geomtype[j], &io_geommoreup[j]);   j++;
			top = j;   io_readgeom(&io_geomtype[j], &io_geommoreup[j]);   j++;
			bot = j;   io_readgeom(&io_geomtype[j], &io_geommoreup[j]);   j++;
			for(;;)
			{
				io_readgeom(&io_geomtype[j], &io_geommoreup[j]);   j++;
				if (io_geommoreup[j-1] == top) break;
			}
			io_geomindex = j;
			for(look = bot; look != top; look = io_geommoreup[look])
				if (io_geomtype[look] == OBJARCINST)
			{
				io_arclist[io_arcindex]->parent = np;
				if (io_readarcinst(io_arclist[io_arcindex]) != 0) return("No memory");
				io_arcindex++;
			} else if (io_geomtype[look] == OBJNODEINST)
			{
				io_nodelist[io_nodeindex]->parent = np;
				if (io_readnodeinst(io_nodelist[io_nodeindex]) != 0) return("No memory");
				io_nodeindex++;
			}
		} else
		{
			/* version 5 and later find the arcs and nodes in linear order */
			for(j=0; j<io_arccount[i]; j++)
			{
				io_arclist[io_arcindex]->parent = np;
				if (io_readarcinst(io_arclist[io_arcindex]) != 0) return("No memory");
				io_arcindex++;
			}
			for(j=0; j<io_nodecount[i]; j++)
			{
				io_nodelist[io_nodeindex]->parent = np;
				if (io_readnodeinst(io_nodelist[io_nodeindex]) != 0) return("No memory");
				io_nodeindex++;
			}
		}
	}

	if (io_verbose < 0 && io_filelength > 0)
	{
		DiaPercent(1, 100);
		DiaSetText(2, "Cleaning up...");
	}

	/* transform indices to pointers */
	if (io_nodeprotoindex <= 0) lib->firstnodeproto = NONODEPROTO; else
		lib->firstnodeproto = io_nodeprotolist[0];
	if (io_curnodeproto != -1 && io_curnodeproto < io_nodeprotoindex)
		lib->curnodeproto = io_nodeprotolist[io_curnodeproto]; else
			lib->curnodeproto = NONODEPROTO;

	/* create proper cell lists (version 9 and later) */
	if (io_magic <= MAGIC9)
	{
		for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		{
			c = np->cell;

			/* see if there is already a facet with this view */
			lastnp = NONODEPROTO;
			for(onp = c->firstincell; onp != NONODEPROTO; onp = onp->nextincell)
			{
				if (onp->cellview == np->cellview) break;
				lastnp = onp;
			}
			if (onp != NONODEPROTO)
			{
				/* new version of a cell/view, find place in list */
				if (np->version >= onp->version)
				{
					/* place on top of list */
					if (lastnp == NONODEPROTO) c->firstincell = np; else
						lastnp->nextincell = np;
					np->nextincell = onp->nextincell;
					onp->nextincell = NONODEPROTO;
					np->lastversion = onp;
					for(; onp != NONODEPROTO; onp = onp->lastversion)
						onp->newestversion = np;
					np->newestversion = np;
				} else
				{
					/* find place in list of versions */
					np->newestversion = onp;
					ent = NONODEPROTO;
					for(; onp != NONODEPROTO; onp = onp->lastversion)
					{
						if (np->version >= onp->version) break;
						ent = onp;
					}
					ent->lastversion = np;
					np->lastversion = onp;
				}
			} else
			{
				/* this view is new */
				np->lastversion = NONODEPROTO;
				np->newestversion = np;
				np->nextincell = c->firstincell;
				c->firstincell = np;
			}
		}
	}

	/* link up the nodes and arcs to their geometry modules */
	for(i=0; i<io_nodeindex; i++)
	{
		ni = io_nodelist[i];
		geom = ni->geom;
		geom->entrytype = OBJNODEINST;  geom->entryaddr.ni = ni;
		linkgeom(geom, ni->parent);
	}
	for(i=0; i<io_arcindex; i++)
	{
		ai = io_arclist[i];
		setshrinkvalue(ai);
		geom = ai->geom;
		geom->entrytype = OBJARCINST;  geom->entryaddr.ai = ai;
		linkgeom(geom, ai->parent);
	}

	/* create clusters for the cells */
	for(c = lib->firstcell; c != NOCELL; c = c->nextcell)
	{
		c->cluster = alloccluster(c->cellname);
		if (c->cluster == NOCLUSTER) return("No memory");
	}

	/* transform data formerly stored in variables (versions 8 and earlier) */
	if (io_magic >= MAGIC8) io_yankvariabledata(lib);

	/* store the version number in the library */
	nextvarchangequiet();
	(void)setvalkey((INTBIG)lib, VLIBRARY, makekey("LIB_former_version"),
		(INTBIG)version, VSTRING|VDONTSAVE);
	efree(version);

	/* create those pointers not on the disk file */
	io_fixnewlib(lib);

	/* look for any database changes that did not matter */
	for(te=0; te<io_techcount; te++)
		if (io_techerror[te] != 0)
	{
		if (io_verbose > 0)
			ttyputmsg("Unknown technology %s not used so all is well", io_techerror[te]);
		efree(io_techerror[te]);
	}
	for(i=0; i<io_aacount; i++)
		if (io_aiderror[i] != 0)
	{
		if (io_verbose > 0)
			ttyputmsg("Unknown aid %s not used so all is well", io_aiderror[i]);
		efree(io_aiderror[i]);
	}
	for(i=0; i<io_nodepprotoindex; i++)
		if (io_nodepprotoorig[i] != 0)
	{
		if (io_verbose > 0 && io_nodepprotoerror[i] != 0)
			ttyputmsg("Unknown node %s not used so all is well", io_nodepprotoorig[i]);
		efree(io_nodepprotoorig[i]);
	}
	for(i=0; i<io_portpprotoindex; i++)
		if (io_portpprotoerror[i] != 0)
	{
		if (io_verbose > 0) ttyputmsg("Unknown port %s not used so all is well",
			io_portpprotoerror[i]);
		efree(io_portpprotoerror[i]);
	}
	for(i=0; i<io_arcprotoindex; i++)
		if (io_arcprotoerror[i] != 0)
	{
		if (io_verbose > 0) ttyputmsg("Unknown arc %s not used so all is well",
			io_arcprotoerror[i]);
		efree(io_arcprotoerror[i]);
	}

	/* free the memory used here */
	if (io_realname != 0)
	{
		for(i=0; i<io_namecount; i++) efree(io_realname[i]);
		efree((char *)io_realname);
	}
	if (io_newnames != 0) efree((char *)io_newnames);
	if (io_magic > MAGIC5)
	{
		/* versions 4 and earlier used geometric data */
		efree((char *)io_geomtype);
		efree((char *)io_geommoreup);
	}
	efree((char *)io_nodelist);
	efree((char *)io_nodecount);
	efree((char *)io_nodeprotolist);
	efree((char *)io_arclist);
	efree((char *)io_arccount);
	efree((char *)io_nodepprotolist);
	efree((char *)io_portprotolist);
	efree((char *)io_portcount);
	efree((char *)io_portpprotolist);
	efree((char *)io_arcprotolist);
	efree((char *)io_techlist);
	efree((char *)io_aidlist);
	efree((char *)io_techerror);
	efree((char *)io_aiderror);
	efree((char *)io_nodepprotoerror);
	efree((char *)io_nodepprotoorig);
	efree((char *)io_portpprotoerror);
	efree((char *)io_arcprotoerror);
	if (io_magic <= MAGIC9) efree((char *)io_celllist);

	return("");
}

/******************** COMPONENT INPUT ********************/

/* routine to read node prototype.  returns nonzero upon error */
INTSML io_readnodeproto(NODEPROTO *np)
{
	INTBIG i, portcount, k;
	REGISTER INTSML j;
	REGISTER PORTPROTO *pp, *lpt;
	REGISTER VIEW *v;

	/* read the cell information (version 9 and later) */
	if (io_magic <= MAGIC9)
	{
		io_getin((UINTBIG)&np->cell, SIZEOFINTBIG);
		np->cell = io_celllist[(INTBIG)np->cell];
		io_getin((UINTBIG)&np->cellview, SIZEOFINTBIG);
		for(v = el_views; v != NOVIEW; v = v->nextview)
			if (v->temp1 == (INTBIG)np->cellview) break;
		if (v == NOVIEW) v = el_unknownview;
		np->cellview = v;
		io_getin((UINTBIG)&i, SIZEOFINTBIG);
		np->version = i;
		io_getin((UINTBIG)&np->creationdate, SIZEOFINTBIG);
		io_getin((UINTBIG)&np->revisiondate, SIZEOFINTBIG);
	}

	/* versions 8 and earlier read a cell name */
	if (io_magic >= MAGIC8)
	{
		np->cell->cellname = io_getstring(np->cell->lib->cluster);
		if (np->cell->cellname == 0) return(1);
		if (io_verbose > 0) ttyputmsg("Reading cell %s", np->cell->cellname);
	}

	/* set the nodeproto primitive index */
	np->index = 0;

	/* read the nodeproto bounding box */
	io_getin((UINTBIG)&np->lowx, SIZEOFINTBIG);
	io_getin((UINTBIG)&np->highx, SIZEOFINTBIG);
	io_getin((UINTBIG)&np->lowy, SIZEOFINTBIG);
	io_getin((UINTBIG)&np->highy, SIZEOFINTBIG);

	/* reset the first nodeinst index */
	np->firstinst = NONODEINST;

	/* zap the technology field */
	np->tech = NOTECHNOLOGY;

	/* read the library list of nodeproto indices (versions 5 or older) */
	if (io_magic >= MAGIC5)
	{
		io_getin((UINTBIG)&np->lastnodeproto, SIZEOFINTBIG);
		(void)io_convertnodeproto(&np->lastnodeproto);
		io_getin((UINTBIG)&np->nextnodeproto, SIZEOFINTBIG);
		(void)io_convertnodeproto(&np->nextnodeproto);
	}

	/* read the number of portprotos on this nodeproto */
	io_getin((UINTBIG)&portcount, SIZEOFINTBIG);

	/* read the portprotos on this nodeproto */
	lpt = NOPORTPROTO;
	for(j=0; j<portcount; j++)
	{
		/* set pointers to portproto */
		pp = io_portprotolist[io_portprotoindex++];
		if (lpt == NOPORTPROTO) np->firstportproto = pp; else
			lpt->nextportproto = pp;
		lpt = pp;

		/* set the parent pointer */
		pp->parent = np;

		/* read the connecting subnodeinst for this portproto */
		io_getin((UINTBIG)&i, SIZEOFINTBIG);
		pp->subnodeinst = io_nodelist[i];

		/* read the sub-port prototype of the subnodeinst */
		io_getin((UINTBIG)&pp->subportproto, SIZEOFINTBIG);
		io_convertportproto(&pp->subportproto);

		/* read the portproto name and text descriptor */
		pp->protoname = io_getstring(np->cell->lib->cluster);
		if (pp->protoname == 0) return(1);
		if (io_magic <= MAGIC9) io_getin((UINTBIG)&pp->textdescript, SIZEOFINTBIG);

		/* ignore the "seen" bits (versions 8 and older) */
		if (io_magic > MAGIC9) io_getin((UINTBIG)&k, SIZEOFINTBIG);

		/* read the portproto aid information */
		if (io_magic <= MAGIC7)
		{
			/* version 7 and later simply read the relevant data */
			io_getin((UINTBIG)&pp->userbits, SIZEOFINTBIG);

			/* versions 7 and 8 ignore net number */
			if (io_magic >= MAGIC8) io_getin((UINTBIG)&k, SIZEOFINTBIG);
		} else
		{
			/* version 6 and earlier must sift through the information */
			if (io_aabcount >= 1) io_getin((UINTBIG)&pp->userbits, SIZEOFINTBIG);
			for(i=1; i<io_aabcount; i++) io_getin((UINTBIG)&k, SIZEOFINTBIG);
		}

		/* read variable information */
		if (io_readvariables((INTBIG)pp, VPORTPROTO) < 0) return(1);
	}

	/* set final pointer in portproto list */
	if (lpt == NOPORTPROTO) np->firstportproto = NOPORTPROTO; else
		lpt->nextportproto = NOPORTPROTO;

	/* read the facet's geometry modules */
	if (io_magic > MAGIC5)
	{
		/* versions 4 and older have geometry module pointers (ignore it) */
		io_getin((UINTBIG)&i, SIZEOFINTBIG);
		io_getin((UINTBIG)&i, SIZEOFINTBIG);
		io_getin((UINTBIG)&i, SIZEOFINTBIG);
		io_getin((UINTBIG)&i, SIZEOFINTBIG);
		io_getin((UINTBIG)&i, SIZEOFINTBIG);
	}

	/* read aid information */
	io_getin((UINTBIG)&np->adirty, SIZEOFINTBIG);
	np->adirty = io_arrangeaidbits((INTBIG)np->adirty);
	if (io_magic <= MAGIC7)
	{
		/* version 7 and later simply read the relevant data */
		io_getin((UINTBIG)&np->userbits, SIZEOFINTBIG);

		/* versions 7 and 8 ignore net number */
		if (io_magic >= MAGIC8) io_getin((UINTBIG)&k, SIZEOFINTBIG);
	} else
	{
		/* version 6 and earlier must sift through the information */
		if (io_aabcount >= 1) io_getin((UINTBIG)&np->userbits, SIZEOFINTBIG);
		for(i=1; i<io_aabcount; i++) io_getin((UINTBIG)&k, SIZEOFINTBIG);
	}

	/* build the dummy geometric structure for this facet */
	if (geomstructure(np) != 0) return(1);

	/* read variable information */
	if (io_readvariables((INTBIG)np, VNODEPROTO) < 0) return(1);
	return(0);
}

/* routine to read a node instance.  returns nonzero upon error */
INTSML io_readnodeinst(NODEINST *ni)
{
	INTBIG i, k;
	REGISTER INTSML j;
	REGISTER char *inst;
	REGISTER PORTARCINST *pi, *lpo;
	REGISTER PORTEXPINST *pe, *lpe;
	PORTPROTO *pp;
	REGISTER ARCINST *ai;
	static INTBIG orignodename = 0;

	/* read the nodeproto index */
	io_getin((UINTBIG)&ni->proto, SIZEOFINTBIG);
	inst = io_convertnodeproto(&ni->proto);
	if (inst != 0)
	{
		if (orignodename == 0) orignodename = makekey("ORIGINAL_NODENAME");
		nextvarchangequiet();
		(void)setvalkey((INTBIG)ni, VNODEINST, orignodename, (INTBIG)inst, VSTRING|VDONTSAVE);
	}

	/* read the descriptive information */
	io_getin((UINTBIG)&ni->lowx, SIZEOFINTBIG);
	io_getin((UINTBIG)&ni->lowy, SIZEOFINTBIG);
	io_getin((UINTBIG)&ni->highx, SIZEOFINTBIG);
	io_getin((UINTBIG)&ni->highy, SIZEOFINTBIG);
	io_getin((UINTBIG)&i, SIZEOFINTBIG);
	ni->transpose = i;
	io_getin((UINTBIG)&i, SIZEOFINTBIG);
	ni->rotation = i;

	/* versions 9 and later get text descriptor for facet name */
	if (io_magic <= MAGIC9) io_getin((UINTBIG)&ni->textdescript, SIZEOFINTBIG);

	/* read the nodeinst name (versions 1, 2, or 3 only) */
	if (io_magic >= MAGIC3)
	{
		inst = io_getstring(el_tempcluster);
		if (inst == 0) return(1);
		if (*inst != 0)
		{
			nextvarchangequiet();
			if (setvalkey((INTBIG)ni, VNODEINST, el_node_name,
				(INTBIG)inst, VSTRING|VDISPLAY) == NOVARIABLE) return(1);
		}
		efree(inst);
	}

	/* ignore the geometry index (versions 4 or older) */
	if (io_magic > MAGIC5) io_getin((UINTBIG)&i, SIZEOFINTBIG);

	/* read the arc ports */
	io_getin((UINTBIG)&i, SIZEOFINTBIG);
	lpo = NOPORTARCINST;
	for(j=0; j<i; j++)
	{
		/* read the arcinst information (and the particular end on the arc) */
		io_getin((UINTBIG)&k, SIZEOFINTBIG);
		ai = io_arclist[k>>1];
		pi = ai->end[k&1].portarcinst;
		pi->conarcinst = ai;

		/* link in the portarcinst */
		if (j == 0) ni->firstportarcinst = pi; else lpo->nextportarcinst = pi;
		lpo = pi;

		/* read the arcinst index */
		io_getin((UINTBIG)&pi->proto, SIZEOFINTBIG);
		io_convertportproto(&pi->proto);

		/* read variable information */
		if (io_readvariables((INTBIG)pi, VPORTARCINST) < 0) return(1);
	}

	/* setup the last portinst pointer */
	if (i == 0) ni->firstportarcinst = NOPORTARCINST; else
		lpo->nextportarcinst = NOPORTARCINST;

	/* read the exported ports */
	io_getin((UINTBIG)&i, SIZEOFINTBIG);
	lpe = NOPORTEXPINST;
	for(j=0; j<i; j++)
	{
		/* read the exported port prototype index */
		io_getin((UINTBIG)&pp, SIZEOFINTBIG);
		io_convertportproto(&pp);
		pe = pp->subportexpinst;
		pe->exportproto = pp;

		if (j == 0) ni->firstportexpinst = pe; else lpe->nextportexpinst = pe;
		lpe = pe;

		/* read the export prototype */
		io_getin((UINTBIG)&pe->proto, SIZEOFINTBIG);
		io_convertportproto(&pe->proto);

		/* read variable information */
		if (io_readvariables((INTBIG)pe, VPORTEXPINST) < 0) return(1);
	}

	/* setup the last portinst pointer */
	if (i == 0) ni->firstportexpinst = NOPORTEXPINST; else
		lpe->nextportexpinst = NOPORTEXPINST;

	/* ignore the "seen" bits (versions 8 and older) */
	if (io_magic > MAGIC9) io_getin((UINTBIG)&k, SIZEOFINTBIG);

	/* read the aid information */
	if (io_magic <= MAGIC7)
	{
		/* version 7 and later simply read the relevant data */
		io_getin((UINTBIG)&ni->userbits, SIZEOFINTBIG);
	} else
	{
		/* version 6 and earlier must sift through the information */
		if (io_aabcount >= 1) io_getin((UINTBIG)&ni->userbits, SIZEOFINTBIG);
		for(i=1; i<io_aabcount; i++) io_getin((UINTBIG)&k, SIZEOFINTBIG);
	}

	/* read variable information */
	if (io_readvariables((INTBIG)ni, VNODEINST) < 0) return(1);

	return(0);
}

/* routine to read (and mostly ignore) a geometry module */
void io_readgeom(INTBIG *type, INTBIG *moreup)
{
	INTBIG i;

	io_getin((UINTBIG)type, SIZEOFINTBIG);	/* read entrytype */
	if (*type != 0) io_getin((UINTBIG)&i, SIZEOFINTBIG);/* skip entryaddr */
	io_getin((UINTBIG)&i, SIZEOFINTBIG);	/* skip lowx */
	io_getin((UINTBIG)&i, SIZEOFINTBIG);	/* skip highx */
	io_getin((UINTBIG)&i, SIZEOFINTBIG);	/* skip lowy */
	io_getin((UINTBIG)&i, SIZEOFINTBIG);	/* skip highy */
	io_getin((UINTBIG)&i, SIZEOFINTBIG);	/* skip moreleft */
	io_getin((UINTBIG)&i, SIZEOFINTBIG);	/* skip ll */
	io_getin((UINTBIG)&i, SIZEOFINTBIG);	/* skip moreright */
	io_getin((UINTBIG)&i, SIZEOFINTBIG);	/* skip lr */
	io_getin((UINTBIG)moreup, SIZEOFINTBIG);	/* read moreup */
	io_getin((UINTBIG)&i, SIZEOFINTBIG);	/* skip lu */
	io_getin((UINTBIG)&i, SIZEOFINTBIG);	/* skip moredown */
	io_getin((UINTBIG)&i, SIZEOFINTBIG);	/* skip ld */
	io_ignorevariables();			/* skip variables */
}

INTSML io_readarcinst(ARCINST *ai)
{
	INTBIG i, j;
	REGISTER char *inst;

	/* read the arcproto pointer */
	io_getin((UINTBIG)&ai->proto, SIZEOFINTBIG);
	io_convertarcproto(&ai->proto);

	/* read the arc length (versions 5 or older) */
	if (io_magic >= MAGIC5) io_getin((UINTBIG)&ai->length, SIZEOFINTBIG);

	/* read the arc width */
	io_getin((UINTBIG)&ai->width, SIZEOFINTBIG);

	/* ignore the signals value (versions 6, 7, or 8) */
	if (io_magic <= MAGIC6 && io_magic >= MAGIC8)
		io_getin((UINTBIG)&i, SIZEOFINTBIG);

	/* read the arcinst name (versions 3 or older) */
	if (io_magic >= MAGIC3)
	{
		inst = io_getstring(el_tempcluster);
		if (inst == 0) return(1);
		if (*inst != 0)
		{
			nextvarchangequiet();
			if (setvalkey((INTBIG)ai, VARCINST, el_arc_name,
				(INTBIG)inst, VSTRING|VDISPLAY) == NOVARIABLE) return(1);
		}
		efree(inst);
	}

	/* read the arcinst end information */
	for(i=0; i<2; i++)
	{
		io_getin((UINTBIG)&ai->end[i].xpos, SIZEOFINTBIG);
		io_getin((UINTBIG)&ai->end[i].ypos, SIZEOFINTBIG);

		/* read the arcinst's connecting nodeinst index */
		io_getin((UINTBIG)&ai->end[i].nodeinst, SIZEOFINTBIG);
		ai->end[i].nodeinst = io_nodelist[(INTBIG)ai->end[i].nodeinst];
	}

	/* compute the arc length (versions 6 or newer) */
	if (io_magic <= MAGIC6) ai->length = computedistance(ai->end[0].xpos,
		ai->end[0].ypos, ai->end[1].xpos, ai->end[1].ypos);

	/* ignore the geometry index (versions 4 or older) */
	if (io_magic > MAGIC5) io_getin((UINTBIG)&i, SIZEOFINTBIG);

	/* ignore the "seen" bits (versions 8 and older) */
	if (io_magic > MAGIC9) io_getin((UINTBIG)&i, SIZEOFINTBIG);

	/* read the arcinst's aid information */
	if (io_magic <= MAGIC7)
	{
		/* version 7 and later simply read the relevant data */
		io_getin((UINTBIG)&ai->userbits, SIZEOFINTBIG);

		/* versions 7 and 8 ignore net number */
		if (io_magic >= MAGIC8) io_getin((UINTBIG)&j, SIZEOFINTBIG);
	} else
	{
		/* version 6 and earlier must sift through the information */
		if (io_aabcount >= 1) io_getin((UINTBIG)&ai->userbits, SIZEOFINTBIG);
		for(i=1; i<io_aabcount; i++) io_getin((UINTBIG)&j, SIZEOFINTBIG);
	}

	/* read variable information */
	if (io_readvariables((INTBIG)ai, VARCINST) < 0) return(1);
	return(0);
}

/******************** VARIABLE ROUTINES ********************/

/* routine to read the global namespace.  returns nonzero upon error */
INTSML io_readnamespace(void)
{
	REGISTER INTSML i;

	io_getin((UINTBIG)&io_namecount, SIZEOFINTBIG);
	if (io_verbose > 0) ttyputmsg("Reading %d variable names", io_namecount);
	if (io_namecount == 0) return(0);

	/* read in the namespace */
	io_newnames = (INTBIG *)emalloc((SIZEOFINTBIG * io_namecount), el_tempcluster);
	if (io_newnames == 0) return(1);
	io_realname = (char **)emalloc(((sizeof (char *)) * io_namecount), el_tempcluster);
	if (io_realname == 0) return(1);
	for(i=0; i<io_namecount; i++)
	{
		io_realname[i] = io_getstring(el_tempcluster);
		if (io_realname[i] == 0) return(1);
		io_newnames[i] = 0;
	}
	return(0);
}

/*
 * routine to ignore one set of object variables on readin
 */
void io_ignorevariables(void)
{
	REGISTER NODEINST *ni;

	ni = dummynode();
	(void)io_readvariables((INTBIG)ni, VNODEINST);

	/* this next line is not strictly legal!!! */
	if (ni->firstvar != NOVARIABLE) db_freevars(&ni->firstvar, &ni->numvar);
}

/*
 * routine to read a set of object variables.  returns negative upon error and
 * otherwise returns the number of variables read
 */
INTSML io_readvariables(INTBIG addr, INTBIG type)
{
	REGISTER INTSML i, j, datasize;
	REGISTER INTBIG ty;
	REGISTER VARIABLE *var;
	INTBIG count, newtype, len, newaddr, cou, newdescript;
	INTSML key;

	io_getin((UINTBIG)&count, SIZEOFINTBIG);
	if (io_verbose > 0 && count > 0)
		 ttyputmsg("Reading %d variables on %d object", count, type);
	for(i=0; i<count; i++)
	{
		io_getin((UINTBIG)&key, SIZEOFINTSML);
		io_getin((UINTBIG)&newtype, SIZEOFINTBIG);
		ty = newtype;

		/* version 9 and later reads text description on displayable variables */
		if ((ty&VDISPLAY) != 0 && io_magic <= MAGIC9)
			io_getin((UINTBIG)&newdescript, SIZEOFINTBIG); else
				newdescript = defaulttextdescript();
		if ((ty&VISARRAY) != 0)
		{
			io_getin((UINTBIG)&len, SIZEOFINTBIG);
			cou = len;
			if ((ty&VLENGTH) == 0) cou++;
			if ((ty&VTYPE) == VCHAR) datasize = 1; else
				if ((ty&VTYPE) == VDOUBLE) datasize = SIZEOFINTBIG*2; else
					if ((ty&VTYPE) == VSHORT) datasize = 2; else
						datasize = SIZEOFINTBIG;
			newaddr = (INTBIG)emalloc((cou*datasize), el_tempcluster);
			if (newaddr == 0) return(-1);
			if ((ty&VTYPE) == VGENERAL)
			{
				for(j=0; j<len; j += 2)
				{
					io_getin((UINTBIG)(INTBIG *)(newaddr + (j+1)*datasize), SIZEOFINTBIG);
					if (io_getinvar((INTBIG *)(newaddr + j*datasize),
						*(INTBIG *)(newaddr + (j+1)*datasize)) != 0) return(-1);
				}
			} else
			{
				for(j=0; j<len; j++)
					if (io_getinvar((INTBIG *)(newaddr + j*datasize), ty) != 0) return(-1);
			}
			if ((ty&VLENGTH) == 0)
				for(j=0; j<datasize; j++)
					((char *)newaddr)[len*datasize+j] = -1;
		} else if (io_getinvar(&newaddr, ty) != 0) return(-1);

		/* copy this variable into database */
		if (io_newnames[key] == 0) io_newnames[key] = makekey(io_realname[key]);
		nextvarchangequiet();
		var = setvalkey(addr, type, io_newnames[key], newaddr, newtype);
		if (var == NOVARIABLE) return(-1);
		var->textdescript = newdescript;

		/* free the memory allocated for the creation of this variable */
		if ((ty&VTYPE) == VSTRING || (ty&(VCODE1|VCODE2)) != 0)
		{
			if ((ty&VISARRAY) == 0) efree((char *)newaddr); else
				for(j=0; j<len; j++) efree((char *)((INTBIG *)newaddr)[j]);
		}
		if ((ty&VISARRAY) != 0) efree((char *)newaddr);
	}
	return(count);
}

INTSML io_getinvar(INTBIG *addr, INTBIG ty)
{
	INTBIG i;
	REGISTER char *pp;
	REGISTER ARCINST *ai;
	PORTPROTO *ppr;

	if ((ty&(VCODE1|VCODE2)) != 0) ty = VSTRING;
	switch (ty&VTYPE)
	{
		case VADDRESS:
		case VINTEGER:
		case VFRACT:
		case VFLOAT:
			io_getin((UINTBIG)addr, SIZEOFINTBIG);
			if ((ty&VTYPE) == VFLOAT)
			{
				if (io_swap_bytes != 0) io_swapped_floats++;
			}
			break;
		case VDOUBLE:
			io_getin((UINTBIG)addr, SIZEOFINTBIG*2);
			if (io_swap_bytes != 0) io_swapped_doubles++;
			break;
		case VSHORT:
			io_getin((UINTBIG)addr, 2);
			break;
		case VCHAR:
			io_getin((UINTBIG)addr, 1);
			break;
		case VSTRING:
			pp = io_getstring(el_tempcluster);
			if (pp == 0) return(1);
			*addr = (INTBIG)pp;
			break;
		case VNODEINST:
			io_getin((UINTBIG)addr, SIZEOFINTBIG);
			if (*addr < 0 || *addr >= io_nodeindex) *addr = -1; else
				*addr = (INTBIG)io_nodelist[*addr];
			break;
		case VNODEPROTO:
			io_getin((UINTBIG)addr, SIZEOFINTBIG);
			(void)io_convertnodeproto((NODEPROTO **)addr);
			break;
		case VARCPROTO:
			io_getin((UINTBIG)addr, SIZEOFINTBIG);
			io_convertarcproto((ARCPROTO **)addr);
			break;
		case VPORTPROTO:
			io_getin((UINTBIG)addr, SIZEOFINTBIG);
			io_convertportproto((PORTPROTO **)addr);
			break;
		case VARCINST:
			io_getin((UINTBIG)addr, SIZEOFINTBIG);
			if (*addr < 0 || *addr >= io_arcindex) *addr = -1; else
				*addr = (INTBIG)io_arclist[*addr];
			break;
		case VGEOM:
			*addr = -1;
			io_getin((UINTBIG)&i, SIZEOFINTBIG);
			if (io_magic <= MAGIC5)
			{
				/* versions 5 and later store extra information */
				if (i == OBJNODEINST)
				{
					io_getin((UINTBIG)addr, SIZEOFINTBIG);
					if (*addr < 0 || *addr >= io_nodeindex) *addr = -1;
						else *addr = (INTBIG)io_nodelist[*addr]->geom;
				} else if (i == OBJARCINST)
				{
					io_getin((UINTBIG)addr, SIZEOFINTBIG);
					if (*addr < 0 || *addr >= io_arcindex) *addr = -1;
						else *addr = (INTBIG)io_arclist[*addr]->geom;
				}
			}
			break;
		case VTECHNOLOGY:
			io_getin((UINTBIG)addr, SIZEOFINTBIG);
			if (*addr != -1) *addr = (INTBIG)io_gettechlist((INTSML)(*addr));
			break;
		case VPORTARCINST:
			io_getin((UINTBIG)addr, SIZEOFINTBIG);
			if (*addr != -1)
			{
				ai = io_arclist[(*addr) >> 1];
				i = (*addr) & 1;
				*addr = (INTBIG)ai->end[i].portarcinst;
			}
			break;
		case VPORTEXPINST:
			io_getin((UINTBIG)addr, SIZEOFINTBIG);
			if (*addr != -1)
			{
				ppr = (PORTPROTO *)*addr;
				io_convertportproto(&ppr);
				*addr = (INTBIG)ppr->subportexpinst;
			}
			break;
		case VLIBRARY:
			pp = io_getstring(el_tempcluster);
			if (pp == 0) return(1);
			*addr = (INTBIG)getlibrary(pp);
			break;
		case VAID:
			io_getin((UINTBIG)addr, SIZEOFINTBIG);
			if (*addr < 0 || *addr >= io_aacount) *addr = -1; else
			{
				i = io_aidlist[*addr];
				if (i < 0 || i >= el_maxaid)
				{
					i = 0;
					if (io_aiderror[*addr] != 0)
					{
						ttyputerr("WARNING: no aid called '%s', using 'user'", io_aiderror[*addr]);
						efree(io_aiderror[*addr]);
						io_aiderror[*addr] = 0;
					}
				}
				*addr = (INTBIG)&el_aids[i];
			}
			break;
		case VRTNODE:
			*addr = -1;
			break;
   }
   return(0);
}

/******************** INPUT HELPER ROUTINES ********************/

/*
 * routine to return the proper aid bits word given the arrangement
 * of aids.  These words have one bit per aid according to
 * the aid's position which may get re-arranged.
 */
INTBIG io_arrangeaidbits(INTBIG val)
{
	REGISTER INTBIG out;
	REGISTER INTSML i, j;

	/* if the bits are not re-arranged, result is simple */
	if (io_aidbitsmessed == 0) return(val);

	/* re-arrange the bits in the word */
	out = 0;
	for(i=0; i<io_aacount; i++)
	{
		j = io_aidlist[i];
		if (j < 0) continue;
		if ((val & (1 << i)) != 0) out |= 1 << j;
	}
	return(out);
}

/*
 * routine to convert the nodeproto index at "np" to a true nodeproto pointer.
 * returns a string with the original name of this prototype (if there were
 * conversion problems in the file).  Returns zero if the pointer is fine.
 */
char *io_convertnodeproto(NODEPROTO **np)
{
	REGISTER INTBIG i, index;
	REGISTER char *origname;

	if (*np == NONODEPROTO) return(0);
	i = (INTBIG)*np;
	origname = 0;
	if (i < 0)
	{
		index = -i - 2;
		if (index >= io_nodepprotoindex)
		{
			ttyputerr("Want primitive node index %d when limit is %d", index, io_nodepprotoindex);
			index = 0;
		}
		*np = io_getnodepprotolist(index);
		origname = io_nodepprotoorig[index];
	} else *np = io_nodeprotolist[i];
	return(origname);
}

void io_convertportproto(PORTPROTO **pp)
{
	REGISTER INTBIG i, index;

	if (*pp == NOPORTPROTO) return;
	i = (INTBIG)*pp;
	if (i < 0)
	{
		index = -i - 2;
		if (index >= io_portpprotoindex)
		{
			ttyputerr("Want primitive port index %d when limit is %d", index, io_portpprotoindex);
			index = 0;
		}
		*pp = io_getportpprotolist(index);
	} else *pp = io_portprotolist[i];
}

void io_convertarcproto(ARCPROTO **ap)
{
	REGISTER INTBIG i, index;

	if (*ap == NOARCPROTO) return;
	i = (INTBIG)*ap;
	index = -i - 2;
	if (index >= io_arcprotoindex || index < 0)
	{
		ttyputerr("Want primitive arc index %d when range is 0 to %d", index, io_arcprotoindex);
		index = 0;
	}
	*ap = io_getarcprotolist((INTSML)index);
}

/******************** DATA ACCESS ROUTINES ********************/

ARCPROTO *io_getarcprotolist(INTSML i)
{
	REGISTER ARCPROTO *ap;
	REGISTER char *name;

	if (io_arcprotoerror[i] != 0)
	{
		ap = NOARCPROTO;
		while (ap == NOARCPROTO)
		{
			(void)initinfstr();
			(void)addstringtoinfstr("Cannot find arc '");
			(void)addstringtoinfstr(io_arcprotoerror[i]);
			(void)addstringtoinfstr("', use which instead [");
			(void)addstringtoinfstr(io_arcprotolist[i]->protoname);
			(void)addstringtoinfstr("] ");
			name = ttygetline(returninfstr());
			if (*name == 0) break;
			for(ap = io_arcprotolist[i]->tech->firstarcproto; ap != NOARCPROTO; ap = ap->nextarcproto)
				if (namesame(ap->protoname, name) == 0) break;
		}
		if (ap != NOARCPROTO) io_arcprotolist[i] = ap;
		efree(io_arcprotoerror[i]);
		io_arcprotoerror[i] = 0;
	}
	return(io_arcprotolist[i]);
}

PORTPROTO *io_getportpprotolist(INTBIG i)
{
	if (io_portpprotoerror[i] != 0)
	{
		ttyputerr("WARNING: port %s not found, using %s", io_portpprotoerror[i],
			io_portpprotolist[i]->protoname);
		efree(io_portpprotoerror[i]);
		io_portpprotoerror[i] = 0;
	}
	return(io_portpprotolist[i]);
}

NODEPROTO *io_getnodepprotolist(INTBIG i)
{
	REGISTER NODEPROTO *np;
	REGISTER char *name;

	(void)io_gettechlist((INTSML)(io_nodepprototech[i]));
	if (io_nodepprotoerror[i] != 0)
	{
		np = NONODEPROTO;
		while (np == NONODEPROTO)
		{
			(void)initinfstr();
			(void)addstringtoinfstr("Cannot find primitive '");
			(void)addstringtoinfstr(io_nodepprotoorig[i]);
			(void)addstringtoinfstr("', use which instead [");
			(void)addstringtoinfstr(io_nodepprotolist[i]->primname);
			(void)addstringtoinfstr("] ");
			name = ttygetline(returninfstr());
			if (*name == 0) break;
			for(np = io_nodepprotolist[i]->tech->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
				if (namesame(np->primname, name) == 0) break;
		}
		if (np != NONODEPROTO) io_nodepprotolist[i] = np;
		io_nodepprotoerror[i] = 0;
	}
	return(io_nodepprotolist[i]);
}

TECHNOLOGY *io_gettechlist(INTSML i)
{
	extern COMCOMP us_noyesp;
	REGISTER INTSML count;
	char *pars[10];

	if (io_techerror[i] != 0)
	{
		ttyputerr("WARNING: technology '%s' does not exist, using '%s'",
			io_techerror[i], io_techlist[i]->techname);
		efree(io_techerror[i]);
		io_techerror[i] = 0;
		count = ttygetparam("Is this OK? [n] ", &us_noyesp, 2, pars);
		if (count <= 0 || namesamen(pars[0], "yes", (INTSML)(strlen(pars[0]))) != 0)
			longjmp(io_filerror, 1);
		ttyputerr("WARNING: saving this library with a substitute technology may corrupt it!");
	}
	return(io_techlist[i]);
}

/******************** I/O ROUTINES ********************/

void io_getin(UINTBIG data, INTSML size)
{
	REGISTER INTBIG ret;
	INTSML *ptr16;
	INTBIG *ptr32;

	if (size == 0)
	{
		ttyputmsg("Warning: null length data; database may be bad (byte %d)", io_bytecount);
		return;
	}
	ret = xfread((char *)data, 1, size, io_filein);
	if (ret == 0) longjmp(io_filerror, 1);
	if (io_swap_bytes > 0 && size > 1)
	{
		switch (size)
		{
			case 2:
				ptr16 = (INTSML *)data;
				*ptr16 = io_swap16(ptr16);
				break;
			case 4:
				ptr32 = (INTBIG *)data;
				*ptr32 = io_swap32(ptr32);
				break;
			default:
				break;  /* needs to be worked on */
		}
	}
	io_bytecount += size;
	if (io_verbose < 0 && io_filelength > 0)
	{
		if (io_bytecount > io_reported + REPORTINC)
		{
			DiaPercent(1, io_bytecount*100/io_filelength);
			io_reported = io_bytecount;
		}
	}
}

char *io_getstring(CLUSTER *cluster)
{
	INTBIG len;
	char *name;

	io_getin((UINTBIG)&len, SIZEOFINTBIG);
	name = (char *)emalloc((len+1), cluster);
	if (name == 0) return(0);
	if (io_swap_bytes != 0 && len != 0) io_swap_bytes = -1;     /* prevent swapping */
	if (len != 0) io_getin((UINTBIG)name, (INTSML)len);
	name[len] = 0;
	if (io_swap_bytes != 0) io_swap_bytes = 1;
	return(name);
}

char *io_gettempstring(void)
{
	INTBIG len;

	io_getin((UINTBIG)&len, SIZEOFINTBIG);
	if (len+1 > io_tempstringlength)
	{
		if (io_tempstringlength != 0) efree(io_tempstring);
		io_tempstringlength = len+1;
		io_tempstring = (char *)emalloc(io_tempstringlength, io_aid->cluster);
		if (io_tempstring == 0) { io_tempstringlength = 0; return(0); }
	}
	if (io_swap_bytes != 0 && len != 0) io_swap_bytes = -1;
	if (len != 0) io_getin((UINTBIG)io_tempstring, (INTSML)len);
	io_tempstring[len] = 0;
	if (io_swap_bytes != 0) io_swap_bytes = 1;
	return(io_tempstring);
}

/***************** Byte swapping routines ***********************/

INTBIG io_swap32(INTBIG *i)
{
	return( ((*i<<24)&0xFF000000) | ((*i<<8)&0x00FF0000) |
		((*i>>8)&0x0000FF00) | ((*i>>24)&0x000000FF) );
}

INTSML io_swap16(INTSML *s)
{
	return( ((*s<<8)&0xFF00) | ((*s>>8)&0x00FF) );
}
