/*
 * Electric(tm) VLSI Design System
 *
 * File: iobinaryo.c
 * Input/output aid: binary format output
 * Written by: Steven M. Rubin, Electric Editor Incorporated
 *
 * Copyright (c) 1998 Electric Editor Incorporated.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Electric Editor Incorporated
 * 23470 Sunset Drive, Suite 108
 * Los Gatos, California 95033
 * support@electriceditor.com
 */

#include "global.h"
#include "eio.h"
#include "usr.h"
#ifdef	MACOS
#  include <Files.h>
#endif
#ifdef HAVE_SYS_TYPES_H
#  include <sys/types.h>
#endif
#ifdef	ONUNIX
#  include <sys/stat.h>				/* UNIX */
#  define	RENAME 1
#endif
#ifdef HAVE_UNISTD_H
#  include <unistd.h>
#endif

static INTSML  io_outputstage;		/* progress of database (aborting) */

/* prototypes for local routines */
void io_cleanup(void);
void io_writenodeproto(NODEPROTO*);
void io_writenodeinst(NODEINST*);
void io_writearcinst(ARCINST*);
void io_writenamespace(void);
void io_restorenamespace(void);
void io_writevariables(VARIABLE*, INTSML);
void io_putoutvar(INTBIG*, INTBIG);
void io_putout(UINTBIG, INTSML);
void io_putstring(char*);

INTSML io_writebinlibrary(LIBRARY *lib)
{
	INTBIG i, a, n, p, magic, aacount, techcount, nodepprotoindex,
		portpprotoindex, arcprotoindex, nodeprotoindex, nodeindex,
		portprotoindex, arcindex, geomindex, curnodeproto, cellindex;
	REGISTER NODEPROTO *np;
	REGISTER ARCPROTO *ap;
	REGISTER PORTPROTO *pp;
	REGISTER NODEINST *ni;
	REGISTER ARCINST *ai;
	REGISTER TECHNOLOGY *tech;
	REGISTER char *rename, *name, *istrue, *dotaddr, *msg;
	char *truename, *arg[3];
	REGISTER CELL *c;
	REGISTER VIEW *v;
#ifdef	RENAME
	struct stat buf;
#endif

	io_outputstage = 0;
	i = initinfstr();
	istrue = &lib->libfile[strlen(lib->libfile)-5];
	if (strcmp(istrue, ".elib") == 0)
	{
		dotaddr = istrue;
		*istrue = 0;
	} else dotaddr = 0;
	istrue = truepath(lib->libfile);
	i += addstringtoinfstr(istrue);
	i += addstringtoinfstr(".elib");
	if (dotaddr != 0) *dotaddr = '.';
	if (i != 0)
	{
		ttyputerr("No memory! saving to 'PANIC'");
		name = "PANIC";
	} else name = returninfstr();

	rename = 0;
	if ((lib->userbits&READFROMDISK) == 0) msg = "Library file"; else
	{
		msg = 0;
#ifdef	RENAME
		/* if the file already exists, rename it first */
		if (stat(name, &buf) >= 0) rename = (char *)emalloc((strlen(name)+8), el_tempcluster);
		if (rename != 0)
		{
			(void)strcpy(rename, name);
			(void)strcat(rename, ".XXXXXX");
			(void)mktemp(rename);
			if (link(name, rename) < 0)
			{
				ttyputerr("Cannot rename old file to %s", rename);
				efree(rename);
				rename = 0;
			} else
			{
				if (unlink(name) < 0)
					ttyputerr("Cannot complete rename of old %s", name);
			}
		}
#endif
	}
	io_fileout = xcreate(name, FILETYPEBLIB, msg, &truename);
	if (io_fileout == NULL)
	{
		if (truename != 0) ttyputerr("Cannot write %s", truename);
		return(1);
	}

	/* if the user got a dialog, replace the library and file name */
	if (msg != 0)
	{
		arg[0] = lib->libname;
		arg[1] = truename;
		arg[2] = "library";
		us_rename(3, arg);
		name = truename;
	}

	if (setjmp(io_filerror))
	{
		ttyputerr("Error writing %s", name);
		xclose(io_fileout);
		io_cleanup();
		if (rename != 0) ttyputmsg("Old binary file saved in %s", rename);
		return(1);
	}

	/* first flush all changes so that database is clean */
	noundoallowed();

	/* initialize the file (writing version 9) */
	magic = MAGIC9;
	io_putout((UINTBIG)&magic, SIZEOFINTBIG);
	aacount = el_maxaid;
	io_putout((UINTBIG)&aacount, SIZEOFINTBIG);
	techcount = el_maxtech;
	io_putout((UINTBIG)&techcount, SIZEOFINTBIG);

	/* transform database from pointer base to index base */
	nodeindex = portprotoindex = nodeprotoindex = arcindex = nodepprotoindex =
		portpprotoindex = arcprotoindex = cellindex = 0;

	/* count and number the objects in the library */
	for(c = lib->firstcell; c != NOCELL; c = c->nextcell)
		c->temp1 = cellindex++;
	for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
	{
		np->temp1 = nodeprotoindex++;
		for(pp=np->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
			pp->temp1 = portprotoindex++;
		for(ai=np->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
			ai->temp1 = arcindex++;
		for(ni=np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
			ni->temp1 = nodeindex++;
	}

	/* count and number the primitive node and port prototypes */
	for(tech = el_technologies; tech != NOTECHNOLOGY; tech = tech->nexttechnology)
	{
		for(np = tech->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		{
			np->temp1 = -2 - nodepprotoindex++;
			for(pp = np->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
				pp->temp1 = -2 - portpprotoindex++;
		}
		for(ap = tech->firstarcproto; ap != NOARCPROTO; ap = ap->nextarcproto)
			ap->temp1 = -2 - arcprotoindex++;
	}

	/* write number of objects */
	io_putout((UINTBIG)&nodepprotoindex, SIZEOFINTBIG);
	io_putout((UINTBIG)&portpprotoindex, SIZEOFINTBIG);
	io_putout((UINTBIG)&arcprotoindex, SIZEOFINTBIG);
	io_putout((UINTBIG)&nodeprotoindex, SIZEOFINTBIG);
	io_putout((UINTBIG)&nodeindex, SIZEOFINTBIG);
	io_putout((UINTBIG)&portprotoindex, SIZEOFINTBIG);
	io_putout((UINTBIG)&arcindex, SIZEOFINTBIG);
	io_putout((UINTBIG)&geomindex, SIZEOFINTBIG);
	io_putout((UINTBIG)&cellindex, SIZEOFINTBIG);

	/* write the current facet */
	if (lib->curnodeproto == NONODEPROTO) curnodeproto = -1; else
		curnodeproto = lib->curnodeproto->temp1;
	io_putout((UINTBIG)&curnodeproto, SIZEOFINTBIG);

	if (io_verbose > 0)
	{
		ttyputmsg("Writing %d aids, %d technologies", aacount, techcount);
		ttyputmsg("        %d prim nodes, %d prim ports, %d arc protos",
			nodepprotoindex, portpprotoindex, arcprotoindex);
		ttyputmsg("        %d cells %d facets, %d ports, %d nodes, %d arcs",
			cellindex, nodeprotoindex, portprotoindex, nodeindex, arcindex);
	}

	/* write the version number */
	io_putstring(el_version);

	/* number the views and write nonstandard ones */
	for(v = el_views; v != NOVIEW; v = v->nextview) v->temp1 = 0;
	el_unknownview->temp1 = -1;
	el_layoutview->temp1 = -2;
	el_schematicview->temp1 = -3;
	el_iconview->temp1 = -4;
	el_simsnapview->temp1 = -5;
	el_skeletonview->temp1 = -6;
	el_vhdlview->temp1 = -7;
	el_netlistview->temp1 = -8;
	el_docview->temp1 = -9;
	el_netlistnetlispview->temp1 = -10;
	el_netlistalsview->temp1 = -11;
	el_netlistquiscview->temp1 = -12;
	el_netlistrsimview->temp1 = -13;
	el_netlistsilosview->temp1 = -14;
	i = 1;
	for(v = el_views; v != NOVIEW; v = v->nextview)
		if (v->temp1 == 0) v->temp1 = i++;
	i--;
	io_putout((UINTBIG)&i, SIZEOFINTBIG);
	for(v = el_views; v != NOVIEW; v = v->nextview)
	{
		if (v->temp1 < 0) continue;
		io_putstring(v->viewname);
		io_putstring(v->sviewname);
	}

	/* write total number of arcinsts, nodeinsts, and ports in each facet */
	for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
	{
		for(a=0, ai=np->firstarcinst; ai!=NOARCINST; ai=ai->nextarcinst) a++;
		io_putout((UINTBIG)&a, SIZEOFINTBIG);
		for(n=0, ni=np->firstnodeinst; ni!=NONODEINST; ni=ni->nextnodeinst) n++;
		io_putout((UINTBIG)&n, SIZEOFINTBIG);
		for(p=0, pp=np->firstportproto; pp!=NOPORTPROTO; pp=pp->nextportproto) p++;
		io_putout((UINTBIG)&p, SIZEOFINTBIG);
		if (io_verbose > 0) ttyputmsg("Facet %s has %d arcs, %d nodes, %d ports",
			describenodeproto(np), a, n, p);
	}

	/* write the names of technologies and primitive prototypes */
	for(tech = el_technologies; tech != NOTECHNOLOGY; tech = tech->nexttechnology)
	{
		/* write the technology name */
		io_putstring(tech->techname);

		/* count and write the number of primitive node prototypes */
		for(np = tech->firstnodeproto, i=0; np != NONODEPROTO; np = np->nextnodeproto) i++;
		io_putout((UINTBIG)&i, SIZEOFINTBIG);

		for(np = tech->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		{
			/* write the primitive node prototype name */
			io_putstring(np->primname);
			for(pp = np->firstportproto, i=0; pp != NOPORTPROTO; pp = pp->nextportproto) i++;
			io_putout((UINTBIG)&i, SIZEOFINTBIG);
			for(pp = np->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
				io_putstring(pp->protoname);
		}

		/* count and write the number of arc prototypes */
		for(ap = tech->firstarcproto, i=0; ap != NOARCPROTO; ap = ap->nextarcproto) i++;
		io_putout((UINTBIG)&i, SIZEOFINTBIG);

		/* write the primitive arc prototype names */
		for(ap = tech->firstarcproto; ap != NOARCPROTO; ap = ap->nextarcproto)
			io_putstring(ap->protoname);
	}

	/* write the names of the aids */
	for(i=0; i<el_maxaid; i++) io_putstring(el_aids[i].aidname);

	/* write the userbits for the library */
	io_putout((UINTBIG)&lib->userbits, SIZEOFINTBIG);

	/* write the aid lambda values */
	for(tech = el_technologies; tech != NOTECHNOLOGY; tech = tech->nexttechnology)
		io_putout((UINTBIG)&lib->lambda[tech->index], SIZEOFINTBIG);

	/* write the global namespace */
	io_writenamespace();
	io_outputstage = 1;

	/* write the library variables */
	io_writevariables(lib->firstvar, lib->numvar);

	/* write the aid variables */
	for(i=0; i<el_maxaid; i++)
		io_writevariables(el_aids[i].firstvar, el_aids[i].numvar);

	/* write the variables on technologies */
	for(tech = el_technologies; tech != NOTECHNOLOGY; tech = tech->nexttechnology)
		io_writevariables(tech->firstvar, tech->numvar);

	/* write the arcproto variables */
	for(tech = el_technologies; tech != NOTECHNOLOGY; tech = tech->nexttechnology)
		for(ap = tech->firstarcproto; ap != NOARCPROTO; ap = ap->nextarcproto)
			io_writevariables(ap->firstvar, ap->numvar);

	/* write the variables on primitive node prototypes */
	for(tech = el_technologies; tech != NOTECHNOLOGY; tech = tech->nexttechnology)
		for(np = tech->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
			io_writevariables(np->firstvar, np->numvar);

	/* write the variables on primitive port prototypes */
	for(tech = el_technologies; tech != NOTECHNOLOGY; tech = tech->nexttechnology)
		for(np = tech->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
			for(pp = np->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
				io_writevariables(pp->firstvar, pp->numvar);

	/* write the view variables */
	i = 0;
	for(v = el_views; v != NOVIEW; v = v->nextview) i++;
	io_putout((UINTBIG)&i, SIZEOFINTBIG);
	for(v = el_views; v != NOVIEW; v = v->nextview)
	{
		io_putout((UINTBIG)&v->temp1, SIZEOFINTBIG);
		io_writevariables(v->firstvar, v->numvar);
	}

	/* write the rest of the database */
	for(c = lib->firstcell; c != NOCELL; c = c->nextcell)
	{
		if (stopping("Binary output") != 0) longjmp(io_filerror, 1);

		/* write the cell name */
		io_putstring(c->cellname);

		/* write variable information */
		io_writevariables(c->firstvar, c->numvar);

		if (io_verbose > 0) ttyputmsg("Wrote cell %s", c->cellname);
	}
	for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
	{
		if (stopping("Binary output") != 0) longjmp(io_filerror, 1);
		io_writenodeproto(np);
	}
	for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
	{
		if (stopping("Binary output") != 0) longjmp(io_filerror, 1);
		a = n = 0;
		for(ai = np->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
		{
			io_writearcinst(ai);
			a++;
		}
		for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
		{
			io_writenodeinst(ni);
			n++;
		}
		if (io_verbose > 0) ttyputmsg("Wrote %d arcs and %d nodes in facet %s",
			a, n, describenodeproto(np));
	}

	/* close the output file */
	xclose(io_fileout);

	/* restore any damage to the database */
	io_cleanup();

	/* if the file was renamed, kill the backup copy */
#ifdef	RENAME
	if (rename != 0)
	{
		if (unlink(rename) < 0) ttyputerr("Cannot delete backup file %s", rename);
		efree(rename);
	}
#endif

#ifdef MACOS
	ttyputmsgf("%s written (%d facets)", name, nodeprotoindex);
#else
	ttyputmsg("%s written (%d facets)", name, nodeprotoindex);
#endif
	lib->userbits = (lib->userbits & ~LIBCHANGED) | READFROMDISK;
	return(0);
}

/*
 * routine to clean-up the internal database after it has been modified
 * for binary output
 */
void io_cleanup(void)
{
	/* if binary output didn't get far, no cleanup needed */
	if (io_outputstage == 0) return;

	/* restore damage to the global namespace */
	io_restorenamespace();
}

/******************** COMPONENT OUTPUT ********************/

void io_writenodeproto(NODEPROTO *np)
{
	INTBIG i;
	REGISTER PORTPROTO *pp;

	/* write facet information */
	io_putout((UINTBIG)&np->cell->temp1, SIZEOFINTBIG);
	io_putout((UINTBIG)&np->cellview->temp1, SIZEOFINTBIG);
	i = np->version;   io_putout((UINTBIG)&i, SIZEOFINTBIG);
	io_putout((UINTBIG)&np->creationdate, SIZEOFINTBIG);
	io_putout((UINTBIG)&np->revisiondate, SIZEOFINTBIG);

	/* write the nodeproto bounding box */
	io_putout((UINTBIG)&np->lowx, SIZEOFINTBIG);
	io_putout((UINTBIG)&np->highx, SIZEOFINTBIG);
	io_putout((UINTBIG)&np->lowy, SIZEOFINTBIG);
	io_putout((UINTBIG)&np->highy, SIZEOFINTBIG);

	/* write the number of portprotos on this nodeproto */
	i = 0;
	for(pp=np->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto) i++;
	io_putout((UINTBIG)&i, SIZEOFINTBIG);

	/* write the portprotos on this nodeproto */
	for(pp=np->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
	{
		/* write the connecting subnodeinst for this portproto */
		if (pp->subnodeinst == NONODEINST) i = -1; else
			i = pp->subnodeinst->temp1;
		io_putout((UINTBIG)&i, SIZEOFINTBIG);

		/* write the portproto index in the subnodeinst */
		if (pp->subnodeinst == NONODEINST || pp->subportproto == NOPORTPROTO) i = -1; else
			i = pp->subportproto->temp1;
		io_putout((UINTBIG)&i, SIZEOFINTBIG);

		/* write the portproto name and text descriptor */
		io_putstring(pp->protoname);
		io_putout((UINTBIG)&pp->textdescript, SIZEOFINTBIG);

		/* write the portproto aid information */
		io_putout((UINTBIG)&pp->userbits, SIZEOFINTBIG);

		/* write variable information */
		io_writevariables(pp->firstvar, pp->numvar);
	}

	/* write aid information */
	io_putout((UINTBIG)&np->adirty, SIZEOFINTBIG);
	io_putout((UINTBIG)&np->userbits, SIZEOFINTBIG);

	/* write variable information */
	io_writevariables(np->firstvar, np->numvar);

	if (io_verbose > 0) ttyputmsg("Wrote facet %s", describenodeproto(np));
}

void io_writenodeinst(NODEINST *ni)
{
	INTBIG i;
	REGISTER ARCINST *ai;
	REGISTER PORTARCINST *pi;
	REGISTER PORTEXPINST *pe;

	/* write the nodeproto pointer */
	io_putout((UINTBIG)&ni->proto->temp1, SIZEOFINTBIG);

	/* write descriptive information */
	io_putout((UINTBIG)&ni->lowx, SIZEOFINTBIG);
	io_putout((UINTBIG)&ni->lowy, SIZEOFINTBIG);
	io_putout((UINTBIG)&ni->highx, SIZEOFINTBIG);
	io_putout((UINTBIG)&ni->highy, SIZEOFINTBIG);
	i = ni->transpose;   io_putout((UINTBIG)&i, SIZEOFINTBIG);
	i = ni->rotation;    io_putout((UINTBIG)&i, SIZEOFINTBIG);
	io_putout((UINTBIG)&ni->textdescript, SIZEOFINTBIG);

	/* count the arc ports */
	i = 0;
	for(pi=ni->firstportarcinst; pi!=NOPORTARCINST; pi=pi->nextportarcinst) i++;
	io_putout((UINTBIG)&i, SIZEOFINTBIG);

	/* write the arc ports */
	for(pi=ni->firstportarcinst; pi!=NOPORTARCINST; pi=pi->nextportarcinst)
	{
		/* write the arcinst index (and the particular end on that arc) */
		ai = pi->conarcinst;
		if (ai->end[0].portarcinst == pi) i = 0; else i = 1;
		i = (ai->temp1 << 1) + (ai->end[0].portarcinst == pi ? 0 : 1);
		io_putout((UINTBIG)&i, SIZEOFINTBIG);

		/* write the portinst prototype */
		io_putout((UINTBIG)&pi->proto->temp1, SIZEOFINTBIG);

		/* write the variable information */
		io_writevariables(pi->firstvar, pi->numvar);
	}

	/* count the exported ports */
	i = 0;
	for(pe=ni->firstportexpinst; pe!=NOPORTEXPINST; pe=pe->nextportexpinst) i++;
	io_putout((UINTBIG)&i, SIZEOFINTBIG);

	/* write the exported ports */
	for(pe=ni->firstportexpinst; pe!=NOPORTEXPINST; pe=pe->nextportexpinst)
	{
		io_putout((UINTBIG)&pe->exportproto->temp1, SIZEOFINTBIG);

		/* write the portinst prototype */
		io_putout((UINTBIG)&pe->proto->temp1, SIZEOFINTBIG);

		/* write the variable information */
		io_writevariables(pe->firstvar, pe->numvar);
	}

	/* write the aid information */
	io_putout((UINTBIG)&ni->userbits, SIZEOFINTBIG);

	/* write variable information */
	io_writevariables(ni->firstvar, ni->numvar);
}

void io_writearcinst(ARCINST *arcinst)
{
	INTBIG i;

	/* write the arcproto pointer */
	io_putout((UINTBIG)&arcinst->proto->temp1, SIZEOFINTBIG);

	/* write basic arcinst information */
	io_putout((UINTBIG)&arcinst->width, SIZEOFINTBIG);

	/* write the arcinst end information */
	for(i=0; i<2; i++)
	{
		io_putout((UINTBIG)&arcinst->end[i].xpos, SIZEOFINTBIG);
		io_putout((UINTBIG)&arcinst->end[i].ypos, SIZEOFINTBIG);

		/* write the arcinst's connecting nodeinst index */
		io_putout((UINTBIG)&arcinst->end[i].nodeinst->temp1, SIZEOFINTBIG);
	}

	/* write the arcinst's aid information */
	io_putout((UINTBIG)&arcinst->userbits, SIZEOFINTBIG);

	/* write variable information */
	io_writevariables(arcinst->firstvar, arcinst->numvar);
}

/******************** VARIABLE ROUTINES ********************/

/* routine to write the global namespace */
void io_writenamespace(void)
{
	REGISTER INTSML i;
	INTBIG val;

	val = el_numnames;
	io_putout((UINTBIG)&val, SIZEOFINTBIG);
	if (el_numnames == 0) return;

	for(i=0; i<el_numnames; i++) io_putstring(el_namespace[i]);

	io_namesave = (char *)emalloc((el_numnames*2), el_tempcluster);
	if (io_namesave == 0)
	{
		ttyputerr("No memory! destroying internal variables to save file");
		ttyputerr("Saved file is good, memory NOT: quit when save finishes!");
		for(i=0; i<el_numnames; i++) *((INTSML *)el_namespace[i]) = i;
	} else for(i=0; i<el_numnames; i++)
	{
		io_namesave[2*i] = el_namespace[i][0];
		io_namesave[2*i+1] = el_namespace[i][1];
		*((INTSML *)el_namespace[i]) = i;
	}
}

void io_restorenamespace(void)
{
	REGISTER INTSML i;

	if (el_numnames == 0) return;
	for(i=0; i<el_numnames; i++)
	{
		el_namespace[i][0] = io_namesave[2*i];
		el_namespace[i][1] = io_namesave[2*i+1];
	}
	efree(io_namesave);
}

void io_writevariables(VARIABLE *firstvar, INTSML numvar)
{
	REGISTER INTSML i, j, datasize;
	INTBIG num, len, addr, ty;

	for(num=i=0; i<numvar; i++)
		if ((firstvar[i].type&VDONTSAVE) == 0) num++;
	io_putout((UINTBIG)&num, SIZEOFINTBIG);
	for(i=0; i<numvar; i++)
	{
		if ((firstvar[i].type&VDONTSAVE) != 0) continue;
		io_putout((UINTBIG)firstvar[i].key, SIZEOFINTSML);
		ty = firstvar[i].type;
		io_putout((UINTBIG)&ty, SIZEOFINTBIG);
		if ((ty&VDISPLAY) != 0) io_putout((UINTBIG)&firstvar[i].textdescript, SIZEOFINTBIG);
		addr = firstvar[i].addr;
		if ((ty&VISARRAY) != 0)
		{
			len = (ty&VLENGTH) >> VLENGTHSH;
			if ((ty&VTYPE) == VCHAR)
			{
				datasize = 1;
				if (len == 0)
					for(len=0; ((((char *)addr)[len])&0377) != 0377; len++)
						;
			} else if ((ty&VTYPE) == VDOUBLE)
			{
				datasize = SIZEOFINTBIG*2;
				if (len == 0)
					for(len=0; ((INTBIG *)addr)[len*2] != -1 &&
						((INTBIG *)addr)[len*2+1] != -1; len++)
							;
			} else if ((ty&VTYPE) == VSHORT)
			{
				datasize = 2;
				if (len == 0)
					for(len=0; ((INTSML *)addr)[len] != -1; len++)
						;
			} else
			{
				datasize = SIZEOFINTBIG;
				if (len == 0) for(len=0; ((INTBIG *)addr)[len] != -1; len++)
					;
			}
			io_putout((UINTBIG)&len, SIZEOFINTBIG);
			if ((ty&VTYPE) == VGENERAL)
			{
				for(j=0; j<len; j += 2)
				{
					io_putout((UINTBIG)((INTBIG *)(addr + (j+1)*datasize)), SIZEOFINTBIG);
					io_putoutvar((INTBIG *)(addr + j*datasize), *(INTBIG *)(addr + (j+1)*datasize));
				}
			} else
			{
				for(j=0; j<len; j++) io_putoutvar((INTBIG *)(addr + j*datasize), ty);
			}
		} else io_putoutvar(&addr, ty);
	}
}

void io_putoutvar(INTBIG *addr, INTBIG ty)
{
	REGISTER ARCINST *ai;
	REGISTER PORTARCINST *pi;
	REGISTER GEOM *geom;
	INTBIG j;
	static INTBIG nullptr = -1;

	if ((ty&(VCODE1|VCODE2)) != 0) ty = VSTRING;
	switch (ty&VTYPE)
	{
		case VADDRESS:
		case VFRACT:
		case VINTEGER:
		case VFLOAT:
			io_putout((UINTBIG)addr, SIZEOFINTBIG);
			break;
		case VDOUBLE:
			io_putout((UINTBIG)addr, SIZEOFINTBIG*2);
			break;
		case VCHAR:
			io_putout((UINTBIG)addr, 1);
			break;
		case VSTRING:
			io_putstring((char *)*addr);
			break;
		case VTECHNOLOGY:
			if ((TECHNOLOGY *)*addr != NOTECHNOLOGY)
			{
				j = ((TECHNOLOGY *)*addr)->index;
				io_putout((UINTBIG)&j, SIZEOFINTBIG);
			} else io_putout((UINTBIG)&nullptr, SIZEOFINTBIG);
			break;
		case VNODEINST:
			if ((NODEINST *)*addr != NONODEINST)
				io_putout((UINTBIG)&((NODEINST *)*addr)->temp1, SIZEOFINTBIG); else
					io_putout((UINTBIG)&nullptr, SIZEOFINTBIG);
			break;
		case VNODEPROTO:
			if ((NODEPROTO *)*addr != NONODEPROTO)
				io_putout((UINTBIG)&((NODEPROTO *)*addr)->temp1, SIZEOFINTBIG); else
					io_putout((UINTBIG)&nullptr, SIZEOFINTBIG);
			break;
		case VARCPROTO:
			if ((ARCPROTO *)*addr != NOARCPROTO)
				io_putout((UINTBIG)&((ARCPROTO *)*addr)->temp1, SIZEOFINTBIG); else
					io_putout((UINTBIG)&nullptr, SIZEOFINTBIG);
			break;
		case VPORTPROTO:
			if ((PORTPROTO *)*addr != NOPORTPROTO)
				io_putout((UINTBIG)&((PORTPROTO *)*addr)->temp1, SIZEOFINTBIG); else
					io_putout((UINTBIG)&nullptr, SIZEOFINTBIG);
			break;
		case VARCINST:
			if ((ARCINST *)*addr != NOARCINST)
				io_putout((UINTBIG)&((ARCINST *)*addr)->temp1, SIZEOFINTBIG); else
					io_putout((UINTBIG)&nullptr, SIZEOFINTBIG);
			break;
		case VGEOM:
			geom = (GEOM *)*addr;
			io_putout((UINTBIG)&geom->entrytype, SIZEOFINTBIG);
			if ((INTBIG)geom->entryaddr.blind == -1) io_putout((UINTBIG)&nullptr, SIZEOFINTBIG); else
			{
				if (geom->entrytype == OBJNODEINST)
					io_putout((UINTBIG)&geom->entryaddr.ni->temp1, SIZEOFINTBIG); else
						if (geom->entrytype == OBJARCINST)
							io_putout((UINTBIG)&geom->entryaddr.ai->temp1, SIZEOFINTBIG);
			}
			break;
		case VPORTARCINST:
			pi = (PORTARCINST *)*addr;
			if (pi != NOPORTARCINST)
			{
				ai = pi->conarcinst;
				j = (ai->temp1 << 1) + (ai->end[0].portarcinst == pi ? 0 : 1);
				io_putout((UINTBIG)&j, SIZEOFINTBIG);
			} else io_putout((UINTBIG)&nullptr, SIZEOFINTBIG);
			break;
		case VPORTEXPINST:
			if ((PORTEXPINST *)*addr != NOPORTEXPINST)
				io_putout((UINTBIG)&((PORTEXPINST *)*addr)->exportproto->temp1, SIZEOFINTBIG); else
					io_putout((UINTBIG)&nullptr, SIZEOFINTBIG);
			break;
		case VLIBRARY:
			if ((LIBRARY *)*addr != NOLIBRARY)
				io_putstring(((LIBRARY *)*addr)->libname); else
					io_putstring("noname");
			break;
		case VAID:
			if ((AIDENTRY *)*addr != NOAID)
				io_putout((UINTBIG)&((AIDENTRY *)*addr)->index, SIZEOFINTBIG); else
					io_putout((UINTBIG)&nullptr, SIZEOFINTBIG);
			break;
		case VSHORT:
			io_putout((UINTBIG)addr, 2);
			break;
	}
}

/******************** I/O ROUTINES ********************/

void io_putout(UINTBIG data, INTSML size)
{
	REGISTER INTBIG ret;

	if (size == 0)
	{
		ttyputmsg("Warning: null length data item; database may be bad");
		return;
	}
	ret = xfwrite((char *)data, size, 1, io_fileout);
	if (ret == 0) longjmp(io_filerror, 1);
}

void io_putstring(char *name)
{
	INTBIG len;

	len = strlen(name);
	io_putout((UINTBIG)&len, SIZEOFINTBIG);
	if (len != 0) io_putout((UINTBIG)name, (INTSML)len);
}
