/*
 * Electric(tm) VLSI Design System
 *
 * File: iocifin.c
 * Input/output aid: CIF input
 * Written by: Robert Winstanley, University of Calgary
 *
 * Copyright (c) 1998 Electric Editor Incorporated.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Electric Editor Incorporated
 * 23470 Sunset Drive, Suite 108
 * Los Gatos, California 95033
 * support@electriceditor.com
 */

#include "config.h"
#if IOCIF

#include "global.h"
#include "eio.h"
#include "iocifpars.h"
#include "edialogs.h"
#include <math.h>

#define	RTOD (45. / atan(1.))

/**************************** CIF CELLS ****************************/

#define NOCIFCELL ((CIFCELL *) -1)

typedef struct Icifcell
{
	INTSML           index;			/* cell index given in the define statement */
	INTBIG           l, r, t, b;	/* bounding box of cell */
	NODEPROTO       *addr;			/* the address of the cif cell */
	struct Icifcell *nextcell;		/* the next cif cell in the list */
} CIFCELL;

CIFCELL *io_cifcells;			/* head of list of 'cifcells' */
CIFCELL *io_curcell;			/* the current cell */
static INTSML roundwires;		/* nonzero if wires should be rounded */

/**************************** CIF LISTS ****************************/

#define NOCIFLIST ((CIFLIST *) -1)

/* values for CIFLIST->identity */
#define C_START		0
#define C_END		1
#define C_WIRE		2
#define C_FLASH		3
#define C_BOX		4
#define C_POLY		5
#define C_COMMAND	6
#define C_GNAME		7
#define C_LABEL		8
#define C_CALL		9

typedef struct Iciflist
{
	INTSML           identity;		/* specifies the nature of the entry */
	char            *member;		/* will point to member's structure */
	struct Iciflist *next;			/* next entry in list */
} CIFLIST;

CIFLIST *io_ciflist;				/* head of the list */
CIFLIST *io_curlist;				/* current location in list */

/**************************** MEMBER STRUCTURES ****************************/

typedef struct Icfstart
{
	INTSML index;					/* cell index */
	char  *name;					/* cell name */
	INTBIG l, r, t, b;				/* bounding box of cell */
} CFSTART;

typedef struct Icbox
{
	INTSML lay;						/* the corresponding layer number */
	INTBIG length, width;			/* dimensions of box */
	INTBIG cenx, ceny;				/* center point of box */
	INTSML xrot, yrot;				/* box direction */
} CBOX;

typedef struct Icpolygon
{
	INTSML  lay;					/* the corresponding layer number */
	INTBIG *x, *y;					/* list of points */
	INTSML  lim;					/* number of points in list */
} CPOLY;

typedef struct Icgeoname
{
	INTSML lay;						/* the corresponding layer number */
	INTBIG x, y;					/* location of name */
	char  *geoname;					/* the geo name */
} CGNAME;

typedef struct Iclabel
{
	INTBIG x, y;					/* location of label */
	char  *label;					/* the label */
} CLABEL;

typedef struct Iccall
{
	INTSML          index;			/* index of cell called */
	char           *name;			/* name of cell called */
	struct Ictrans *list;			/* list of transformations */
} CCALL;

/* values for the transformation type */
#define MIRX	1					/* mirror in x */
#define MIRY	2					/* mirror in y */
#define TRANS	3					/* translation */
#define ROT		4					/* rotation */

#define NOCTRANS ((CTRANS *) -1)

typedef struct Ictrans
{
	INTSML          type;			/* type of transformation */
	INTBIG          x, y;			/* not required for the mirror types */
	struct Ictrans *next;			/* next element in list */
} CTRANS;

CTRANS *io_curctrans;				/* current transformation description */

/**************************** MISCELLANEOUS ****************************/

char       *io_curnodeprotoname;	/* name of the current cell */
NODEPROTO  *io_incell;				/* address of facet being defined */
NODEPROTO **io_cifnodes;			/* nodeprotos associated with CIF names */
INTSML      io_varlength;			/* the length of all attribute arrays */
VARIABLE   *io_cifnames;			/* cif names for the current technology */

/* prototypes for local routines */
INTSML io_interpret(FILE*);
INTSML io_listtonodes(LIBRARY*);
NODEPROTO *io_nodes_start(LIBRARY*);
INTSML io_nodes_box(void);
INTSML io_nodes_poly(void);
INTSML io_nodes_call(void);
void io_rotatelayer(INTBIG*, INTBIG*, INTSML);
INTSML io_initfind(void);
NODEPROTO *io_findprotonode(INTSML);
CIFCELL *io_findcifcell(INTSML);
CIFCELL *io_newcifcell(void);
CIFLIST *io_newciflist(INTSML);
void io_placeciflist(INTSML);
CTRANS *io_newctrans(void);
INTSML io_outwindow(INTBIG, INTBIG, INTBIG, INTBIG);

INTSML io_readciflibrary(LIBRARY *lib)
{
	REGISTER FILE *f;
	char *filename;
	REGISTER VARIABLE *var;

	var = getvalkey((INTBIG)io_aid, VAID, VINTEGER, io_state);
	if (var != NOVARIABLE && (var->addr&CIFINSQUARE) != 0) roundwires = 0;
		else roundwires = 1;

	/* initialize all lists and the searching routines */
	io_cifcells = NOCIFCELL;
	io_ciflist = NOCIFLIST;
	if (io_initfind()) return(1);

	/* get the cif file */
	f = xopen(lib->libfile, FILETYPECIF, "", &filename);
	if (f == NULL)
	{
		ttyputerr("File %s not found", lib->libfile);
		return(1);
	}

	if (setjmp(io_filerror))
	{
		ttyputerr("Error reading CIF");
		return(1);
	}

	/* parse the cif and create a listing */
	if (io_interpret(f)) return(1);

	/* instantiate the cif as nodes */
	if (io_listtonodes(lib)) return(1);

	/* clean up */
	(void)io_doneinterpreter();
	return(0);
}

INTSML io_interpret(FILE *file)
{
	INTBIG comcount, left, right, bottom, top;
	float fleft, fright, fbottom, ftop, lambda;

	if (io_initparser())
	{
		ttyputerr("Error initializing parser");
		return(1);
	}
	if (io_initinterpreter())
	{
		ttyputerr("Error initializing interpreter");
		return(1);
	}
	if (io_infromfile(file)) return(1);
	comcount = io_parsefile();		/* read in the cif */
	(void)io_doneparser();
	if (stopping("CIF input") != 0) return(1);
	ttyputmsgf("Total CIF commands: %d",comcount);

	if (io_fatalerrors() > 0) return(1);

	lambda = (float)el_curtech->deflambda;
	io_iboundbox(&left, &right, &bottom, &top);

	(void)io_outwindow(left, right, bottom, top);

	/* construct a list: first step in the conversion */
	(void)io_createlist();

	fleft   = ((float)left)   / lambda;
	fright  = ((float)right)  / lambda;
	fbottom = ((float)bottom) / lambda;
	ftop    = ((float)top)    / lambda;
	ttyputmsgf("The bounding box is %d<=X<=%d, %d<=Y<=%d", scalefromdispunit(fleft, DISPUNITCMIC),
		scalefromdispunit(fright, DISPUNITCMIC), scalefromdispunit(fbottom, DISPUNITCMIC),
			scalefromdispunit(ftop, DISPUNITCMIC));
	return(0);
}

/**************************** NODE CONVERSION ****************************/

INTSML io_listtonodes(LIBRARY *lib)
{
	extern DIALOG usr_progressdialog;

	if (io_verbose < 0)
	{
		if (DiaInitDialog(&usr_progressdialog) != 0) return(1);
		DiaPercent(1, 100);
	}
	io_incell = NONODEPROTO;
	for(io_curlist = io_ciflist; io_curlist != NOCIFLIST; io_curlist = io_curlist->next)
	{
		if (stopping("CIF input") != 0)
		{
			if (io_verbose < 0) DiaDoneDialog();
			return(1);
		}
		if (io_incell != NONODEPROTO || io_curlist->identity == C_START)
			switch (io_curlist->identity)
		{
			case C_START:
				io_incell = io_nodes_start(lib);
				if (io_incell == NONODEPROTO)
				{
					if (io_verbose < 0) DiaDoneDialog();
					return(1);
				}
				break;
			case C_END:
				/* make cell size right */
				(*el_curconstraint->solve)(io_incell);
				io_incell = NONODEPROTO;
				break;
			case C_BOX:
				if (io_nodes_box())
				{
					if (io_verbose < 0) DiaDoneDialog();
					return(1);
				}
				break;
			case C_POLY:
				if (io_nodes_poly())
				{
					if (io_verbose < 0) DiaDoneDialog();
					return(1);
				}
				break;
			case C_CALL:
				if (io_nodes_call())
				{
					if (io_verbose < 0) DiaDoneDialog();
					return(1);
				}
				break;
		}
	}
	if (io_verbose < 0) DiaDoneDialog();
	return(0);
}

NODEPROTO *io_nodes_start(LIBRARY *lib)
{
	CIFCELL *cifcell;
	CFSTART *cs;
	char *opt;

	if ((cifcell = io_newcifcell()) == NOCIFCELL) return(NONODEPROTO);
	cs = (CFSTART *)io_curlist->member;
	cifcell->index = cs->index;
	cifcell->l = cs->l;   cifcell->r = cs->r;
	cifcell->b = cs->b;   cifcell->t = cs->t;
	io_curnodeprotoname = cs->name;

	/* remove illegal characters */
	for(opt = io_curnodeprotoname; *opt != 0; opt++)
		if (*opt <= ' ' || *opt == ':' || *opt == ';' || *opt >= 0177)
			*opt = 'X';
	cifcell->addr = newnodeproto(io_curnodeprotoname, lib);
	if (cifcell->addr == NONODEPROTO)
	{
		ttyputerr("Cannot create the cell %s", io_curnodeprotoname);
		return(NONODEPROTO);
	}

	if (io_verbose < 0)
	{
		(void)initinfstr();
		(void)addstringtoinfstr("Reading ");
		(void)addstringtoinfstr(io_curnodeprotoname);
		DiaSetText(2, returninfstr());
	} else if (io_verbose > 0) ttyputmsg("Reading %s", io_curnodeprotoname);
	return(cifcell->addr);
}

INTSML io_nodes_box(void)
{
	NODEPROTO *node;
	CBOX *cb;
	INTBIG l, w, lx, ly, hx, hy, r;

	cb = (CBOX *)io_curlist->member;
	node = io_findprotonode(cb->lay);
	l = cb->length;        w = cb->width;
	lx = cb->cenx - l/2;   ly = cb->ceny - w/2;
	hx = cb->cenx + l/2;   hy = cb->ceny + w/2;
	r = figureangle(0, 0, (INTBIG)cb->xrot, (INTBIG)cb->yrot);
	if (newnodeinst(node, scalefromdispunit((float)lx, DISPUNITCMIC),
		scalefromdispunit((float)hx, DISPUNITCMIC), scalefromdispunit((float)ly, DISPUNITCMIC),
		scalefromdispunit((float)hy, DISPUNITCMIC), 0, (INTSML)r, io_curcell->addr) == NONODEINST)
	{
		ttyputerr("Problems creating a box on layer %d in facet %s", cb->lay,
			describenodeproto(io_curcell->addr));
		return(1);
	}
	return(0);
}

INTSML io_nodes_poly(void)
{
	REGISTER INTBIG lx, ly, hx, hy, i, *trace, cx, cy, *pt;
	REGISTER NODEPROTO *np;
	REGISTER NODEINST *newni;
	REGISTER CPOLY *cp;

	cp = (CPOLY *)io_curlist->member;
	if (cp->lim == 0) return(0);
	np = io_findprotonode(cp->lay);
	lx = hx = cp->x[0];   ly = hy = cp->y[0];
	for(i=1; i<cp->lim; i++)
	{
		if (cp->x[i] < lx) lx = cp->x[i];
		if (cp->x[i] > hx) hx = cp->x[i];
		if (cp->y[i] < ly) ly = cp->y[i];
		if (cp->y[i] > hy) hy = cp->y[i];
	}
	newni = newnodeinst(np, scalefromdispunit((float)lx, DISPUNITCMIC),
		scalefromdispunit((float)hx, DISPUNITCMIC), scalefromdispunit((float)ly, DISPUNITCMIC),
			scalefromdispunit((float)hy, DISPUNITCMIC), 0, 0, io_curcell->addr);
	if (newni == NONODEINST)
	{
		ttyputerr("Problems creating a polygon on layer %d in facet %s", cp->lay,
			describenodeproto(io_curcell->addr));
		return(1);
	}

	/* store the trace information */
	pt = trace = emalloc((cp->lim*2*SIZEOFINTBIG), el_tempcluster);
	if (trace == 0) return(1);
	cx = (hx + lx) / 2;   cy = (hy + ly) / 2;
	for(i=0; i<cp->lim; i++)
	{
		*pt++ = scalefromdispunit((float)(cp->x[i] - cx), DISPUNITCMIC);
		*pt++ = scalefromdispunit((float)(cp->y[i] - cy), DISPUNITCMIC);
	}

	/* store the trace information */
	(void)setvalkey((INTBIG)newni, VNODEINST, el_trace, (INTBIG)trace,
		VINTEGER|VISARRAY|((cp->lim*2)<<VLENGTHSH));

	/* free the polygon memory */
	efree((char *)trace);
	efree((char *)cp->x);   efree((char *)cp->y);
	cp->lim = 0;
	return(0);
}

INTSML io_nodes_call(void)
{
	CIFCELL *cell;
	CCALL *cc;
	CTRANS *ctrans;
	INTBIG l, r, t, b, rot, trans, hlen, hwid, cenx, ceny, temp;
	INTSML deg;

	cc = (CCALL *)io_curlist->member;
	if ((cell = io_findcifcell(cc->index)) == NOCIFCELL)
	{
		ttyputerr("Referencing an undefined cell");
		return(1);
	}
	rot = trans = 0;
	l = cell->l;    r = cell->r;    b = cell->b;    t = cell->t;
	for(ctrans = cc->list; ctrans != NOCTRANS; ctrans = ctrans->next)
		switch (ctrans->type)
	{
		case MIRX:
			temp = l;   l = -r;   r = -temp;
			rot = (trans) ? ((rot+2700) % 3600) : ((rot+900) % 3600);
			trans = 1 - trans;
			break;
		case MIRY:
			temp = t;   t = -b;   b = -temp;
			rot = (trans) ? ((rot+900) % 3600) : ((rot+2700) % 3600);
			trans = 1 - trans;
			break;
		case TRANS:
			l += ctrans->x;   r += ctrans->x;
			b += ctrans->y;   t += ctrans->y;
			break;
		case ROT:
			deg = figureangle(0L, 0L, ctrans->x, ctrans->y);
			if (deg != 0)
			{
				hlen = abs(((l-r)/2));   hwid = abs(((b-t)/2));
				cenx = (l+r)/2;   ceny = (b+t)/2;
				io_rotatelayer(&cenx,&ceny, deg);
				l = cenx - hlen;   r = cenx + hlen;
				b = ceny - hwid;   t = ceny + hwid;
				rot += ((trans) ? -deg : deg);
			}
	}
	while(rot >= 3600) rot -= 3600;
	while(rot < 0) rot += 3600;
	if (newnodeinst((NODEPROTO *)(cell->addr), scalefromdispunit((float)l, DISPUNITCMIC),
		scalefromdispunit((float)r, DISPUNITCMIC), scalefromdispunit((float)b, DISPUNITCMIC),
			scalefromdispunit((float)t, DISPUNITCMIC), (INTSML)trans, (INTSML)rot, io_curcell->addr) ==
				NONODEINST)
	{
		ttyputerr("Problems creating an instance of cell %s in facet %s",
			describenodeproto(cell->addr), describenodeproto(io_curcell->addr));
		return(1);
	}
	return(0);
}

void io_rotatelayer(INTBIG *cenx, INTBIG *ceny, INTSML deg)
{
	double vlen, vang, fx, fy, factx, facty, fact;
	INTBIG temp;

	/* trivial test to prevent atan2 domain errors */
	if (*cenx == 0 && *ceny == 0) return;
	switch (deg)	/* do the manhattan cases directly (SRP)*/
	{
		case 0:
		case 3600:	/* just in case */
			break;
		case 900:
			temp = *cenx;   *cenx = -*ceny;   *ceny = temp;
			break;
		case 1800:
			*cenx = -*cenx;   *ceny = -*ceny;
			break;
		case 2700:
			temp = *cenx;   *cenx = *ceny;   *ceny = -temp;
			break;
		default: /* this old code only permits rotation by integer angles (SRP)*/
			for(factx=1.; fabs((*cenx/factx)) > 1000; factx *= 10.) ;
			for(facty=1.; fabs((*ceny/facty)) > 1000; facty *= 10.) ;
			fact = (factx > facty) ? facty : factx;
			fx = *cenx / fact;		  fy = *ceny / fact;
			vlen = fact * sqrt((double)(fx*fx + fy*fy));
			vang = (deg + figureangle(0L,0L,*cenx, *ceny)) / 10.0 / RTOD;
			*cenx = (INTBIG)(vlen * cos(vang));
			*ceny = (INTBIG)(vlen * sin(vang));
			break;
	}
}

/**************************** SEARCHING ROUTINES ****************************/

INTSML io_initfind(void)
{
	REGISTER char *ch, *ch2;
	REGISTER NODEPROTO *np;
	REGISTER NODEINST *nodeinst;
	REGISTER INTSML total, i, j, k;
	static POLYGON *poly = NOPOLYGON;

	/* get the array of CIF names */
	io_cifnames = getval((INTBIG)el_curtech, VTECHNOLOGY, VSTRING|VISARRAY, "IO_cif_layer_names");
	if (io_cifnames == NOVARIABLE)
	{
		ttyputerr("There are no CIF layer names assigned in the %s technology", el_curtech->techname);
		return(-1);
	}
	io_varlength = getlength(io_cifnames);

	/* create the array of nodes associated with the array of CIF names */
	total = el_curtech->layercount;
	io_cifnodes = (NODEPROTO **)emalloc((total * (sizeof (NODEPROTO *))), el_curtech->cluster);
	if (io_cifnodes == 0) return(-1);
	for(i=0; i<total; i++) io_cifnodes[i] = NONODEPROTO;

	/* create the polygon */
	if (poly == NOPOLYGON) poly = allocpolygon(4, io_aid->cluster);

	/* run through the node prototypes in this technology */
	for(np = el_curtech->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
	{
		nodeinst = dummynode();
		nodeinst->proto = np;
		nodeinst->lowx  = np->lowx;
		nodeinst->highx = np->highx;
		nodeinst->lowy  = np->lowy;
		nodeinst->highy = np->highy;
		i = nodepolys(nodeinst);
		if (i != 1) continue;
		shapenodepoly(nodeinst, 0, poly);
		if (poly->layer < 0) continue;
		ch = ((char **)io_cifnames->addr)[poly->layer];
		if (*ch != 0) io_cifnodes[poly->layer] = np;
	}

	/* make sure every CIF string has an equivalent node */
	for(i=0; i<total; i++)
	{
		ch = ((char **)io_cifnames->addr)[i];
		if (*ch == 0) continue;
		if (io_cifnodes[i] != NONODEPROTO) continue;

		/* first look for a layer with the same CIF name */
		for (k=0; k<total; k++)
		{
			np = io_cifnodes[k];
			if (np == NONODEPROTO) continue;
			nodeinst = dummynode();
			nodeinst->proto = np;
			nodeinst->lowx  = np->lowx;
			nodeinst->highx = np->highx;
			nodeinst->lowy  = np->lowy;
			nodeinst->highy = np->highy;
			(void)nodepolys(nodeinst);
			shapenodepoly(nodeinst, 0, poly);
			ch2 = ((char **)io_cifnames->addr)[poly->layer];
			if (strcmp(ch, ch2) != 0) continue;
			io_cifnodes[i] = np;
			break;
		}
		if (k < total) continue;

		/* search for ANY node with this layer */
		for(np = el_curtech->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		{
			nodeinst = dummynode();
			nodeinst->proto = np;
			nodeinst->lowx  = np->lowx;
			nodeinst->highx = np->highx;
			nodeinst->lowy  = np->lowy;
			nodeinst->highy = np->highy;
			j = nodepolys(nodeinst);
			for(k=0; k<j; k++)
			{
				shapenodepoly(nodeinst, 0, poly);
				if (poly->layer == i) break;
			}
			if (k >= j) continue;
			io_cifnodes[i] = np;
			break;
		}
	}

	return(0);
}

INTSML io_findlayernum(char *name)
{
	INTSML i;

	for(i=0; i<io_varlength; i++)
		if (strcmp(name, ((char **)io_cifnames->addr)[i]) == 0) return(i);

	/* CIF name not found */
	ttyputmsg("Layer %s not found", name);
	return(-1);
}

NODEPROTO *io_findprotonode(INTSML num)
{
	return(io_cifnodes[num]);
}

CIFCELL *io_findcifcell(INTSML index)
{
	CIFCELL *cifcell;

	for(cifcell=io_cifcells; cifcell->index!=index; cifcell=cifcell->nextcell)
		if (cifcell->nextcell == NOCIFCELL) return(NOCIFCELL);
	return(cifcell);
}

CIFCELL *io_newcifcell(void)
{
	CIFCELL *newcc, *endcc;

	newcc = (CIFCELL *)emalloc((sizeof (CIFCELL)), el_tempcluster);
	if (newcc == (CIFCELL *) 0)
	{
		ttyputerr("Not enough memory allocated for CIFCELL");
		return(NOCIFCELL);
	}
	newcc->nextcell = NOCIFCELL;
	if (io_cifcells == NOCIFCELL) io_cifcells = newcc; else
	{
		endcc = io_cifcells;
		while(endcc->nextcell != NOCIFCELL) endcc = endcc->nextcell;
		endcc->nextcell = newcc;
	}
	io_curcell = newcc;
	newcc->addr = NONODEPROTO;
	newcc->index = 0;
	return(newcc);
}

CIFLIST *io_newciflist(INTSML id)
{
	CIFLIST *newcl;

	newcl = (CIFLIST *)emalloc((sizeof (CIFLIST)), el_tempcluster);
	if (newcl == (CIFLIST *) 0)
	{
		ttyputerr("Not enough memory allocated for CIFLIST");
		return(NOCIFLIST);
	}
	newcl->next = NOCIFLIST;
	newcl->identity = id;
	switch (id)
	{
		case C_START:
			newcl->member = (char *)emalloc((sizeof (CFSTART)), el_tempcluster);
			if (newcl->member == (char *) 0) return(NOCIFLIST);
			break;
		case C_BOX:
			newcl->member = (char *)emalloc((sizeof (CBOX)), el_tempcluster);
			if (newcl->member == (char *) 0) return(NOCIFLIST);
			break;
		case C_POLY:
			newcl->member = (char *)emalloc((sizeof (CPOLY)), el_tempcluster);
			if (newcl->member == (char *) 0) return(NOCIFLIST);
			break;
		case C_GNAME:
			newcl->member = (char *)emalloc((sizeof (CGNAME)), el_tempcluster);
			if (newcl->member == (char *) 0) return(NOCIFLIST);
			break;
		case C_LABEL:
			newcl->member = (char *)emalloc((sizeof (CLABEL)), el_tempcluster);
			if (newcl->member == (char *) 0) return(NOCIFLIST);
			break;
		case C_CALL:
			newcl->member = (char *)emalloc((sizeof (CCALL)), el_tempcluster);
			if (newcl->member == (char *) 0) return(NOCIFLIST);
			break;
	}
	return(newcl);
}

void io_placeciflist(INTSML id)
{
	CIFLIST *cl;

	if ((cl = io_newciflist(id)) == NOCIFLIST) return;
	if (io_ciflist == NOCIFLIST) io_ciflist = io_curlist = cl; else
	{
		while(io_curlist->next != NOCIFLIST)
			io_curlist = io_curlist->next;
		io_curlist->next = cl;
		io_curlist = io_curlist->next;
	}
}

CTRANS *io_newctrans(void)
{
	CTRANS *newct;
	CCALL *cc;

	newct = (CTRANS *)emalloc((sizeof (CTRANS)), el_tempcluster);
	if (newct == (CTRANS *) 0)
	{
		ttyputerr("Not enough memory allocated for CTRANS");
		return(NOCTRANS);
	}
	newct->next = NOCTRANS;
	cc = (CCALL *)io_curlist->member;
	if (cc->list == NOCTRANS) cc->list = newct;
	else io_curctrans->next = newct;
	io_curctrans = newct;
	return(newct);
}

static INTBIG lt, rt, bm, tp;		/* window limits */

INTSML io_outwindow(INTBIG l, INTBIG r, INTBIG b, INTBIG t)
{
	lt = l;   rt = r;   bm = b;   tp = t;
	return(0);
}

void io_outputwire(INTSML lay, INTBIG width, char *wpath)
{
	/* convert wires to boxes and flashes  */
	INTSML i;
	INTBIG llt, lrt, lbm, ltp, lim;
	point prev, curr;

	lim = io_pathlength((path)wpath);
	prev = io_removepoint((path)wpath);

	/* do not use roundflashes with zero-width wires */
	if (width != 0 && roundwires != 0)
	{
		io_bbflash(lay, width, prev, &llt, &lrt, &lbm, &ltp);
		io_outputflash(lay, width, prev);
	}
	for (i = 1; i < lim; i++)
	{
		INTBIG len;
		INTBIG xr, yr;
		point center;

		curr = io_removepoint((path)wpath);

		/* do not use roundflashes with zero-width wires */
		if (width != 0 && roundwires != 0)
		{
			io_bbflash(lay, width, curr, &llt, &lrt, &lbm, &ltp);
			io_outputflash(lay, width, curr);
		}
		xr = curr.x-prev.x;   yr = curr.y-prev.y;
		len = computedistance(0, 0, xr, yr);
		if (roundwires == 0) len += width;
		center.x = (curr.x+prev.x)/2;   center.y = (curr.y+prev.y)/2;
		io_bbbox(lay, len, width, center, xr, yr, &llt, &lrt, &lbm, &ltp);
		io_outputbox(lay, len, width, center, (INTSML)xr, (INTSML)yr);
		prev = curr;
	}
}

void io_outputflash(INTSML lay, INTBIG diameter, point center)
{
	/* flash approximated by an octagon */
	INTBIG radius = diameter/2;
	float fcx = (float)center.x;
	float fcy = (float)center.y;
	float offset = (((float) diameter)/2.0)*0.414213;
	char *fpath = io_makepath();
	point temp;

	temp.x = center.x-radius;
	temp.y = roundfloat(fcy+offset);
	io_appendpoint((path)fpath,temp);
	temp.y = roundfloat(fcy-offset);
	io_appendpoint((path)fpath,temp);
	temp.x = roundfloat(fcx-offset);
	temp.y = center.y-radius;
	io_appendpoint((path)fpath,temp);
	temp.x = roundfloat(fcx+offset);
	io_appendpoint((path)fpath,temp);
	temp.x = center.x+radius;
	temp.y = roundfloat(fcy-offset);
	io_appendpoint((path)fpath,temp);
	temp.y = roundfloat(fcy+offset);
	io_appendpoint((path)fpath,temp);
	temp.x = roundfloat(fcx+offset);
	temp.y = center.y+radius;
	io_appendpoint((path)fpath,temp);
	temp.x = roundfloat(fcx-offset);
	io_appendpoint((path)fpath,temp);

	io_outputpolygon(lay,fpath);
	io_freepath((path)fpath);
}

void io_outputbox(INTSML lay, INTBIG length, INTBIG width, point center,
	INTSML xrotation, INTSML yrotation)
{
	CBOX *cb;

	if (length == 0 && width == 0) return;	/* ignore null boxes */
	io_placeciflist(C_BOX);
	cb = (CBOX *)io_curlist->member;
	cb->lay = lay;
	cb->length = length;	cb->width = width;
	cb->cenx = center.x;	cb->ceny = center.y;
	cb->xrot = xrotation;	cb->yrot = yrotation;
}

void io_outputpolygon(INTSML lay, char *ppath)
{
	INTSML i;
	INTBIG lim;
	CPOLY *cp;
	point temp;

	lim = io_pathlength((path)ppath);
	if (lim < 3) return;

	io_placeciflist(C_POLY);
	cp = (CPOLY *)io_curlist->member;
	cp->lay = lay;
	cp->x = emalloc((lim * SIZEOFINTBIG), el_tempcluster);
	if (cp->x == 0)
	{
		ttyputerr("No memory for polygon");
		cp->lim = 0;
		return;
	}
	cp->y = emalloc((lim * SIZEOFINTBIG), el_tempcluster);
	if (cp->y == 0)
	{
		ttyputerr("No memory for polygon");
		cp->lim = 0;
		return;
	}

	cp->lim = lim;
	for (i = 0; i < lim; i++)
	{
		temp = io_removepoint((path)ppath);
		cp->x[i] = temp.x;
		cp->y[i] = temp.y;
	}
}

void io_outputusercommand(INTSML command, char *text)
{
}

void io_outputgeoname(char *name, point pt, INTSML lay)
{
	CGNAME *cg;

	io_placeciflist(C_GNAME);
	cg = (CGNAME *)io_curlist->member;
	cg->lay = lay;
	cg->geoname = io_storename(name);
	cg->x = pt.x;   cg->y = pt.y;
}

void io_outputlabel(char *name, point pt)
{
	CLABEL *cl;

	io_placeciflist(C_LABEL);
	cl = (CLABEL *)io_curlist->member;
	cl->label = io_storename(name);
	cl->x = pt.x;   cl->y = pt.y;
}

void io_outputcall(INTSML number, char *name, char *list)
{
	CCALL *cc;
	tentry temp;
	INTSML i;

	io_placeciflist(C_CALL);
	cc = (CCALL *)io_curlist->member;
	cc->index = number;
	cc->name = io_storename(name);
	cc->list = io_curctrans = NOCTRANS;
	for(i = io_tlistlength((tlist)list); i>0; i--)
	{
		if (io_newctrans() == NOCTRANS) return;
		temp = io_removetentry((tlist)list);
		switch (temp.kind)
		{
			case MIRROR:
				if (temp.guts.mi.xcoord) io_curctrans->type = MIRX; else
					io_curctrans->type = MIRY;
				break;
			case TRANSLATE:
				io_curctrans->type = TRANS;
				io_curctrans->x = temp.guts.tr.xt;
				io_curctrans->y = temp.guts.tr.yt;
				break;
			case ROTATE:
				io_curctrans->type = ROT;
				io_curctrans->x = temp.guts.ro.xrot;
				io_curctrans->y = temp.guts.ro.yrot;
				break;
		}
	}
}

void io_outputds(INTSML number, char *name, INTBIG l, INTBIG r, INTBIG b, INTBIG t)
{
	CFSTART *cs;

	io_placeciflist(C_START);
	cs = (CFSTART *)io_curlist->member;
	cs->index = number;
	cs->name = io_storename(name);
	cs->l = l;   cs->r = r;
	cs->b = b;   cs->t = t;
}

void io_outputdf(void)
{
	io_placeciflist(C_END);
}

#endif  /* IOCIF - at top */
