/*
 * Electric(tm) VLSI Design System
 *
 * File: iogdsin.c
 * Input/output aid: GDSII stream
 * Written by: Glen Lawson, S-MOS Systems, Inc.
 *
 * Copyright (c) 1998 Electric Editor Incorporated.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Electric Editor Incorporated
 * 23470 Sunset Drive, Suite 108
 * Los Gatos, California 95033
 * support@electriceditor.com
 */

/*
 * Notes:
 *	Unsupported features and assumptions:
 *	Map layers with the IO_gds_layer_numbers variable on the current
 *		technology.
 *	The pure-layer node must have the HOLDSTRACE option, if not will CRASH
 *              for boundary types.  Will map to the generic:DRC-node if not found.
 *	Case sensitive.
 *	Creates structures with a Facet-Center.
 *	SUBLAYERS or XXXTYPE fields are not supported, perhaps a variable
 *		could be attached to the object with this value.
 *	NODEs and TEXTNODEs - don't have an example
 *	PROPERTIES - could add as a variable to the structure.
 *	PATHTYPE 1 - rounded ends on paths, not supported
 *	Path dogears - no cleanup yet, simpler to map paths into arcs
 *	Absolute angle - ???
 *	REFLIBS - not supported.
 *	AREFS - only support the 8 standard transforms.
 *	I have not tried this on non-manhattan geometries.
 *	I have no BOX type examples.
 *
 *	Nice to have:
 *	PATH-ARC mapping - should be done, problem is that any layer can
 *		be a path, only connection layers in electric are arcs.
 *	Someone could make a GDS mapping technology for this purpose,
 *		defaults could be taken from this technology.
 *	Misc. fields mapped to variables - should be done.
 *	Units - could auto scale units to a technology.
 *	AREFs - could build into a separate NODEPROTO, and instance, thus
 *		preserving the original intent and space.  With a timestamp,
 *		can determine if the structure has been disturbed for output.
 *	MAG - no scaling is possible, must create a separate object for each
 *		value, don't scale.  (TEXT does scale)
 *
 *	Dated:	5/14/91
 *
 * FIXES BY STEVEN RUBIN:
 * Rewrote "io_gdsIinit()" to be more robust in finding pure-layer nodes.
 * In "gdsl_error()", added index specification "[n]" to the end of "io_gdsIerror_string"
 *    in "ttyputerr()" call.
 * Converted calls to "pow" to use "mypow" instead since "pow" is not
 *    available on Macintosh.
 * In "io_gdsIinit()", initialized "record_count" to zero.
 * In "io_gdsIgetrawbyte()", reversed 2nd and 3rd parameters to "xfread" call.
 * In "io_gdsIdetermine_layer()", added code to detect unknown GDS layers.
 *
 * 11 FEB 92 - changed the variables "register1" and "register2" in "io_gdsIgetdouble()"
 *             to be of type "UINTBIG"
 *           - added io_gds_scale, a global variable ot type "double" defined in
 *             "io_gdsIunits()", used to scale the coordinate points read in from
 *             the GDS file.  Affected routines are:
 *                              "io_gdsIdetermine_points()"
 *                              "io_gdsIdetermine_path()"
 *                              "io_gdsIdetermine_text()"
 *             (RLW - Queen's)
 *
 * 12 FEB 92 - redefined equation for io_gds_scale in "io_gdsIunits()" to prevent
 *             roundoff error
 *             (RLW - Queen's)
 *
 */

#include "config.h"
#if IOGDS

#include "global.h"
#include "egraphics.h"
#include "efunction.h"
#include "database.h"
#include "eio.h"
#include "iogdsi.h"
#include "tecgen.h"
#include "edialogs.h"
#ifdef	ultrix
#  ifndef sigsetjmp
#    include <setjmp.h>
#  endif
#else
#  include <setjmp.h>
#endif
#include <math.h>
#include <ctype.h>

/* electric specific globals */
static LIBRARY   *library = NOLIBRARY;
static jmp_buf    env;
static NODEPROTO *io_gdsstructure = NONODEPROTO;
static NODEPROTO *io_gdssref = NONODEPROTO;
static NODEINST  *instance = NONODEINST;
static NODEPROTO *layer_node = NONODEPROTO;
static NODEPROTO *layer_nodes[MAXLAYERS];
static INTBIG     io_gdsrecomputed;

/* globals ... */
static STREAM		stream;
static INTBIG       bytes_read;
static INTBIG		record_count;
static INTBIG		buffer_index;
static INTBIG		block_count;
static UINTSML		record_type;
static datatype_symbol	valuetype;
static gsymbol		token;
static bit_array	tokenflags;
static INTSML		tokenvalue_16;
static INTBIG		tokenvalue_32;
static float		realtoken_sp;
static double		realtoken_dp;
static char			tokenstring[TEXTLINE+1];
static INTSML		bad_opcode;
static INTBIG		err_count;
static char		    gds_version_string[IDSTRING+1];
static char		    libraryname[IDSTRING+1];
static INTBIG		technology_scale_factor = 1000;
static point_type	vertex[MAXPOINTS];
static point_type	boundarydelta[MAXPOINTS];
static INTBIG		cur_layer, cur_sublayer;
static double       io_gds_scale;
static INTBIG       file_size;

static char *io_gdsIerror_string[] =
{
	"success",											/* 0 */
	"Invalid GDS II datatype",							/* 1 */
	"GDS II version number is not decipherable",		/* 2 */
	"GDS II header statement is missing",				/* 3 */
	"Library name is missing",							/* 4 */
	"Begin library statement is missing",				/* 5 */
	"Date value is not a valid number",					/* 6 */
	"Libname statement is missing",						/* 7 */
	"Generations value is invalid",						/* 8 */
	"Units statement has invalid number format",		/* 9 */
	"Units statement is missing",						/* 10 */
	"Warning this structure is empty",					/* 11 */
	"Structure name is missing",						/* 12 */
	"Strname statement is missing",						/* 13 */
	"Begin structure statement is missing",				/* 14 */
	"Element end statement is missing",					/* 15 */
	"Structure reference name is missing",				/* 16 */
	"Structure reference has no translation value",		/* 17 */
	"Boundary has no points",							/* 18 */
	"Structure reference is missing its flags field",	/* 19 */
	"Not enough points in the shape",					/* 20 */
	"Too many points in the shape",						/* 21 */
	"Invalid layer number",								/* 22 */
	"Layer statement is missing",						/* 23 */
	"No datatype field",								/* 24 */
	"Path element has no points",						/* 25 */
	"Text element has no reference point",				/* 26 */
	"Array reference has no parameters",				/* 27 */
	"Array reference name is missing",					/* 28 */
	"Property has no value",							/* 29 */
	"Array orientation oblique"							/* 30 */
};

/* prototypes for local routines */
double mypow(INTBIG, INTBIG);
INTSML io_gdsImember(gsymbol, gsymbol[]);
void io_gdsIcopyset(gsymbol[], gsymbol[]);
void io_gdsIonlyone(gsymbol[], gsymbol);
void io_gdsIappendset(gsymbol[], gsymbol);
byte_unsigned io_gdsIgetrawbyte(void);
void io_gdsIgetword(UINTSML*);
void io_gdsIgetinteger(INTBIG*);
void io_gdsIgetstring(char*);
void io_gdsIitoa(char*, INTBIG);
void io_gdsIgetfloat(float*);
void io_gdsIgetdouble(double*);
datatype_symbol io_gdsIgetrecordtype(void);
gsymbol io_gdsIgetrecord(datatype_symbol*);
void io_gdsIerror(INTBIG);
void io_gdsIgettoken(void);
void io_gdsItranslate(INTBIG*, INTBIG*, double, INTSML);
void io_gdsItransform(point_type, INTBIG, INTBIG);
void io_gdsIstep(INTBIG, INTBIG, INTBIG, INTBIG);
void io_gdsIdetermine_boundary(INTBIG, shape_type*, shape_type*);
void io_gdsIbox(INTBIG*);
void io_gdsIshape(INTBIG, shape_type, shape_type);
void io_gdsItext(char*, INTBIG, INTBIG, INTBIG, INTBIG, double);
void io_gdsIinit(void);
void io_gdsIdetermine_time(char*);
void io_gdsIheader(void);
void io_gdsIlibrary(void);
void io_gdsIreflibs(void);
void io_gdsIfonts(void);
void io_gdsIbackup(void);
void io_gdsIattrtable(void);
void io_gdsIunits(void);
void io_gdsIunsupported(gsymbol[]);
void io_gdsIdetermine_points(INTBIG, INTBIG, INTBIG*);
void io_gdsIdetermine_orientation(INTBIG*, INTBIG*, double*);
void io_gdsIdetermine_layer(INTBIG*, INTBIG*);
void io_gdsIbeginstructure(void);
void io_gdsIdetermine_property(void);
void io_gdsIdetermine_node(void);
void io_gdsIdetermine_box(void);
void io_gdsIdetermine_shape(void);
void io_gdsIdetermine_path(void);
void io_gdsIgetproto(char*);
void io_gdsIdetermine_sref(void);
void io_gdsIdetermine_aref(void);
void io_gdsIdetermine_justification(INTBIG*, INTBIG*);
void io_gdsIdetermine_text(void);
void io_gdsIelement(INTSML*);
void io_gdsIstructure(void);
INTSML io_gdsIload(FILE*);
void io_recompute(NODEPROTO*);
void io_gdsIcompute_size(NODEPROTO*);

/* modules ... */

double mypow(INTBIG val, INTBIG power)
{
	double result;
	REGISTER INTBIG count;

	result = 1.0;
	for(count=0; count<power; count++) result *= (double)val;
	return(result);
}

INTSML io_gdsImember(gsymbol tok, gsymbol set[])
{
	REGISTER gsymbol *ctok;

	for (ctok = set; *ctok != GDS_BADFIELD; ctok++)
		if (*ctok == tok) return(1);
	return(0);
}

void io_gdsIcopyset(gsymbol set1[], gsymbol set2[])
{
	REGISTER gsymbol *item1, *item2;
	for (item1 = set1, item2 = set2; *item1 != GDS_BADFIELD; item1++, item2++)
		*item2 = *item1;
	*item2 = GDS_BADFIELD;
}

void io_gdsIonlyone(gsymbol set[], gsymbol item)
{
	set[0] = item;
	set[1] = GDS_BADFIELD;
}

void io_gdsIappendset(gsymbol set[], gsymbol item)
{
	REGISTER gsymbol *pos;

	pos = set;
	while (*pos != GDS_BADFIELD) pos++;
	*pos++ = item; *pos = GDS_BADFIELD;
}

byte_unsigned io_gdsIgetrawbyte(void)
{
	if (buffer_index >= BLOCKSIZE || buffer_index < 0)
	{
		(void)xfread(stream.buf, 1, BLOCKSIZE, stream.fp);
		buffer_index = 0;
		block_count = block_count + 1;
		bytes_read += BLOCKSIZE;
		if (bytes_read/5000 != (bytes_read-BLOCKSIZE)/5000)
		{
			if (io_verbose < 0 && file_size > 0)
			DiaPercent(1, bytes_read*100/file_size); else
			  ttyputmsg("%d bytes read (%d%% done)", bytes_read, bytes_read*100/file_size);
		}
	}
	record_count = record_count - 1;
	return(stream.buf[buffer_index++]);
}

void io_gdsIgetword(UINTSML *dataword)
{
	byte_unsigned	lowbyte, highbyte;

	highbyte = io_gdsIgetrawbyte();
	lowbyte = io_gdsIgetrawbyte();
	(*dataword) = (highbyte << 8) + lowbyte;
}

void io_gdsIgetinteger(INTBIG *datavalue)
{
	UINTSML lowword, highword;

	io_gdsIgetword(&highword);
	io_gdsIgetword(&lowword);
	(*datavalue) = (highword << 16) + lowword;
}

void io_gdsIgetstring(char *streamstring)
{
	char letter, *pos;

	pos = streamstring;
	while (record_count != 0)
	{
		letter = io_gdsIgetrawbyte();
		*pos++ = letter;
	}
	*pos = '\0';
}

void io_gdsIitoa(char *nstring, INTBIG number)
{
	(void)sprintf(nstring, "%d", number);
}

void io_gdsIgetfloat(float *realvalue)
{
	unsigned long reg;
	UINTSML	dataword;
	INTBIG	sign;
	INTBIG	binary_exponent;
	INTBIG	shift_count;

	reg = io_gdsIgetrawbyte();
	if (reg & 0x00000080) sign = -1;
	else sign = 1;
	reg = reg & 0x0000007F;

	/* generate the exponent, currently in Excess-64 representation */
	binary_exponent = (reg - 64) << 2;
	reg = (unsigned long)io_gdsIgetrawbyte() << 16;
	io_gdsIgetword(&dataword);
	reg = reg + dataword;
	shift_count = 0;

	/* normalize the matissa */
	while ((reg & 0x00800000) == 0)
	{
		reg = reg << 1;
		shift_count++;
	}
	/* this is the exponent + normalize shift - precision of matissa */
	binary_exponent = binary_exponent + shift_count - 24;

	if (binary_exponent > 0)
		(*realvalue) = (float)sign * (float) reg * mypow(2, binary_exponent); else
			if (binary_exponent < 0)
				(*realvalue) = (float)sign * (float) reg / mypow(2, -binary_exponent); else
					(*realvalue) = (float)sign * (float) reg;
}

void io_gdsIgetdouble(double	*realvalue)
{
	UINTBIG register1, register2;
	UINTSML	dataword;
	INTBIG	sign;
	INTBIG	binary_exponent;
	INTBIG	long_integer;
	INTBIG	shift_count;

	register1 = io_gdsIgetrawbyte();
	if (register1 & 0x00000080) sign = -1; else
	sign = 1;
	register1 = register1 & 0x0000007F;
	binary_exponent = 4 * (register1 - 64);
	register1 = (UINTBIG)io_gdsIgetrawbyte() << 16;
	io_gdsIgetword(&dataword);
	register1 = register1 + dataword;
	io_gdsIgetinteger(&long_integer);
	register2 = (UINTBIG)long_integer;
	shift_count = 0;
	if (register1 != 0)
	{
		while ((register1 & 0x00800000) == 0)
		{
			register1 = register1 * 2;
			if ((register2 & 0x80000000) != 0)
			register1 = register1 + 1;
			register2 = register2 * 2;
			shift_count = shift_count + 1;
		}
	}
	binary_exponent = binary_exponent + shift_count - 56;
	if (binary_exponent > 0)
	{
		(*realvalue) = (double)sign * ((double) register1 * mypow(2, 32) + (double) register2) *
			mypow(2, binary_exponent);
	} else if (binary_exponent < 0)
	{
		(*realvalue) = (double)sign * ((double) register1 *
			mypow(2, 32) + (double) register2) / mypow(2, -binary_exponent);
	} else
		(*realvalue) = (double)sign * ((double) register1 * mypow(2, 32) + (double) register2);
}

datatype_symbol io_gdsIgetrecordtype(void)
{
	switch (io_gdsIgetrawbyte())
	{
		case 0:  return(GDS_NONE);
		case 1:  return(GDS_FLAGS);
		case 2:  return(GDS_SHORT_WORD);
		case 3:  return(GDS_LONG_WORD);
		case 4:  return(GDS_SHORT_REAL);
		case 5:  return(GDS_LONG_REAL);
		case 6:  return(GDS_ASCII_STRING);
		default: return(GDS_ERR);
	}
}

gsymbol io_gdsIgetrecord(datatype_symbol *gds_type)
{
	gsymbol temp;
	UINTSML dataword;

	io_gdsIgetword(&dataword);
	record_count = dataword - 2;
	record_type = io_gdsIgetrawbyte();
	temp = GDS_NULLSYM;
	switch (record_type)
	{
		case  0: temp = GDS_HEADER;        break;
		case  1: temp = GDS_BGNLIB;        break;
		case  2: temp = GDS_LIBNAME;       break;
		case  3: temp = GDS_UNITS;         break;
		case  4: temp = GDS_ENDLIB;        break;
		case  5: temp = GDS_BGNSTR;        break;
		case  6: temp = GDS_STRNAME;       break;
		case  7: temp = GDS_ENDSTR;        break;
		case  8: temp = GDS_BOUNDARY;      break;
		case  9: temp = GDS_PATH;          break;
		case 10: temp = GDS_SREF;          break;
		case 11: temp = GDS_AREF;          break;
		case 12: temp = GDS_TEXTSYM;       break;
		case 13: temp = GDS_LAYER;         break;
		case 14: temp = GDS_DATATYPSYM;    break;
		case 15: temp = GDS_WIDTH;         break;
		case 16: temp = GDS_XY;            break;
		case 17: temp = GDS_ENDEL;         break;
		case 18: temp = GDS_SNAME;         break;
		case 19: temp = GDS_COLROW;        break;
		case 20: temp = GDS_TEXTNODE;      break;
		case 21: temp = GDS_NODE;          break;
		case 22: temp = GDS_TEXTTYPE;      break;
		case 23: temp = GDS_PRESENTATION;  break;
		case 24: temp = GDS_SPACING;       break;
		case 25: temp = GDS_STRING;        break;
		case 26: temp = GDS_STRANS;        break;
		case 27: temp = GDS_MAG;           break;
		case 28: temp = GDS_ANGLE;         break;
		case 29: temp = GDS_UINTEGER;      break;
		case 30: temp = GDS_USTRING;       break;
		case 31: temp = GDS_REFLIBS;       break;
		case 32: temp = GDS_FONTS;         break;
		case 33: temp = GDS_PATHTYPE;      break;
		case 34: temp = GDS_GENERATIONS;   break;
		case 35: temp = GDS_ATTRTABLE;     break;
		case 36: temp = GDS_STYPTABLE;     break;
		case 37: temp = GDS_STRTYPE;       break;
		case 38: temp = GDS_ELFLAGS;       break;
		case 39: temp = GDS_ELKEY;         break;
		case 40: temp = GDS_LINKTYPE;      break;
		case 41: temp = GDS_LINKKEYS;      break;
		case 42: temp = GDS_NODETYPE;      break;
		case 43: temp = GDS_PROPATTR;      break;
		case 44: temp = GDS_PROPVALUE;     break;
		case 45: temp = GDS_BOX;           break;
		case 46: temp = GDS_BOXTYPE;       break;
		case 47: temp = GDS_PLEX;          break;
		case 48: temp = GDS_BGNEXTN;       break;
		case 49: temp = GDS_ENDEXTN;       break;
		case 50: temp = GDS_TAPENUM;       break;
		case 51: temp = GDS_TAPECODE;      break;
		case 52: temp = GDS_STRCLASS;      break;
		case 53: temp = GDS_NUMTYPES;      break;
		case 54: temp = GDS_FORMAT;        break;
		case 55: temp = GDS_MASK;          break;
		case 56: temp = GDS_ENDMASKS;      break;
		default: temp = GDS_BADFIELD;
	}
	*gds_type = io_gdsIgetrecordtype();
	return(temp);
}

void io_gdsIerror(INTBIG n)
{
	ttyputerr("Error #%d: at block number %d index = %d\n", n, block_count, buffer_index);
	longjmp(env, (INTBIG)io_gdsIerror_string[n]);
}

void io_gdsIgettoken(void)
{
	if (record_count == 0)
	{
		token = io_gdsIgetrecord(&valuetype);
	} else
	{
		switch (valuetype)
		{
			case GDS_NONE:
				break;
			case GDS_FLAGS:
				io_gdsIgetword(&tokenflags.u.value);
				token = GDS_FLAGSYM;
				break;
			case GDS_SHORT_WORD:
				io_gdsIgetword((UINTSML *)&tokenvalue_16);
				token = GDS_SHORT_NUMBER;
				break;
			case GDS_LONG_WORD:
				io_gdsIgetinteger(&tokenvalue_32);
				token = GDS_NUMBER;
				break;
			case GDS_SHORT_REAL:
				io_gdsIgetfloat(&realtoken_sp);
				token = GDS_REALNUM;
				break;
			case GDS_LONG_REAL:
				io_gdsIgetdouble(&realtoken_dp);
				token = GDS_REALNUM;
				break;
			case GDS_ASCII_STRING:
				io_gdsIgetstring(tokenstring);
				token = GDS_IDENT;
				break;
			case GDS_ERR:
				io_gdsIerror(1);
				bad_opcode = 1;
				break;
		}
	}
}

void io_gdsItranslate(INTBIG *ang, INTBIG *trans, double angle, INTSML reflected)
{
	*ang = rounddouble(angle);
	*trans = (reflected != 0 ? 1 : 0);
	if (*trans)
	{
		*ang = 6300 - *ang;
		if (*ang >= 3600) *ang -= 3600;
	}
}

void io_gdsItransform(point_type delta, INTBIG angle, INTBIG trans)
{
	XARRAY matrix;

	makeangle((INTSML)angle, (INTSML)trans, matrix);
	xform(delta.px, delta.py, &delta.px, &delta.py, matrix);
}

void io_gdsIstep(INTBIG nc, INTBIG nr, INTBIG angle, INTBIG trans)
{
	point_type rowinterval, colinterval, ptc, pt;
	INTBIG ic, ir;

	if (nc != 1)
	{
		colinterval.px = (vertex[1].px - vertex[0].px) / nc;
		colinterval.py = (vertex[1].py - vertex[0].py) / nc;
		io_gdsItransform(colinterval, angle, trans);
	}
	if (nr != 1)
	{
		rowinterval.px = (vertex[2].px - vertex[0].px) / nr;
		rowinterval.py = (vertex[2].py - vertex[0].py) / nr;
		io_gdsItransform(rowinterval, angle, trans);
	}

	/* now generate the array */
	ptc.px = vertex[0].px;   ptc.py = vertex[0].py;
	for (ic = 0; ic < nc; ic++)
	{
		pt = ptc;
		for (ir = 0; ir < nr; ir++)
		{
			/* create the node */
			if ((instance = newnodeinst(io_gdssref, pt.px + io_gdssref->lowx,
				pt.px + io_gdssref->highx, pt.py + io_gdssref->lowy,
					pt.py + io_gdssref->highy, (INTSML)trans, (INTSML)angle,
						io_gdsstructure)) == NONODEINST)
							longjmp(env, (INTBIG)"failed to create AREF");

			/* add the row displacement */
			pt.px += rowinterval.px;   pt.py += rowinterval.py;
		}

		/* add displacement */
		ptc.px += colinterval.px;   ptc.py += colinterval.py;
	}
}

void io_gdsIdetermine_boundary(INTBIG npts, shape_type *perimeter, shape_type *oclass)
{
	INTSML is90, is45;
	REGISTER INTBIG i;

	is90 = 1;
	is45 = 1;
	if (((vertex[0].px == vertex[npts - 1].px) && (vertex[0].py == vertex[npts - 1].py)))
		(*perimeter) = GDS_CLOSED; else
			(*perimeter) = GDS_LINE;

	for (i=0; i<npts-1 && i<MAXPOINTS-1; i++)
	{
		boundarydelta[i].px = vertex[i+1].px - vertex[i].px;
		boundarydelta[i].py = vertex[i+1].py - vertex[i].py;
	}
	for (i=0; i<npts-1 && i<MAXPOINTS-1; i++)
	{
		if (((boundarydelta[i].px != 0) && (boundarydelta[i].py != 0)))
		{
			is90 = 0;
			if (abs(boundarydelta[i].px) != abs(boundarydelta[i].py))
			is45 = 0;
		}
	}
	if ((((*perimeter) == GDS_CLOSED) && (is90 != 0 || is45 != 0))) (*oclass) = GDS_POLY; else
		(*oclass) = GDS_OBLIQUE;
	if (((npts == 5) && is90 != 0 && ((*perimeter) == GDS_CLOSED))) (*oclass) = GDS_RECTANGLE;
}

void io_gdsIbox(INTBIG *npts)
{
	REGISTER INTBIG i;
	INTBIG pxm;
	INTBIG pym;
	INTBIG pxs;
	INTBIG pys;

	pxm = vertex[4].px;
	pxs = vertex[4].px;
	pym = vertex[4].py;
	pys = vertex[4].py;
	for (i = 0; i<4; i++)
	{
		if ((vertex[i].px > pxm))
		{
			pxm = vertex[i].px;
		}
		if ((vertex[i].px < pxs))
		{
			pxs = vertex[i].px;
		}
		if ((vertex[i].py > pym))
		{
			pym = vertex[i].py;
		}
		if ((vertex[i].py < pys))
		{
			pys = vertex[i].py;
		}
	}
	vertex[0].px = pxs;
	vertex[0].py = pys;
	vertex[1].px = pxm;
	vertex[1].py = pym;
	(*npts) = 2;
}

void io_gdsIshape(INTBIG npts, shape_type perimeter, shape_type oclass)
{
	INTBIG lx, ly, hx, hy, cx, cy, i;
	INTBIG *trace, *pt;
	NODEINST *ni;

	switch (oclass)
	{
		case GDS_RECTANGLE:
			io_gdsIbox(&npts);
			/* create the rectangle */
			if (newnodeinst(layer_node, vertex[0].px, vertex[1].px, vertex[0].py, vertex[1].py, 0, 0,
				io_gdsstructure) == NONODEINST)
					longjmp(env, (INTBIG)"failed to create RECTANGLE");
			break;

		case GDS_OBLIQUE:
		case GDS_POLY:
			/* determine the bounds of the polygon */
			lx = hx = vertex[0].px;
			ly = hy = vertex[0].py;
			for (i=1; i<npts;i++)
			{
				if (lx > vertex[i].px) lx = vertex[i].px;
				if (hx < vertex[i].px) hx = vertex[i].px;
				if (ly > vertex[i].py) ly = vertex[i].py;
				if (hy < vertex[i].py) hy = vertex[i].py;
			}
			/* now create the node */
			if ((ni = newnodeinst(layer_node, lx, hx, ly, hy, 0, 0, io_gdsstructure)) == NONODEINST)
				longjmp(env, (INTBIG)"failed to create POLYGON");
			/* now allocate the trace */
			pt = trace = emalloc((npts*2*SIZEOFINTBIG), el_tempcluster);
			if (trace == 0) longjmp(env, (INTBIG)"no memory");
			cx = (hx + lx) / 2;   cy = (hy + ly) / 2;
			for (i=0; i<npts; i++)
			{
				*pt++ = vertex[i].px - cx;
				*pt++ = vertex[i].py - cy;
			}
			/* store the trace information */
			if (setvalkey((INTBIG)ni, VNODEINST, el_trace, (INTBIG)trace,
				VINTEGER|VISARRAY|((npts*2)<<VLENGTHSH)) == NOVARIABLE)
					longjmp(env, (INTBIG)"failed to create POLYGON points");
			/* free the polygon memory */
			efree((char *)trace);
			break;
		default:
			break;
	}
}

void io_gdsItext(char *charstring, INTBIG vjust, INTBIG hjust, INTBIG angle,
	INTBIG trans, double scale)
{
	NODEINST *ni;
	VARIABLE *var;

	vertex[1].px = vertex[0].px + MINFONTWIDTH * strlen(charstring);
	vertex[1].py = vertex[0].py + MINFONTHEIGHT;

	/* create a holding node */
	if ((ni = newnodeinst(layer_node, vertex[0].px, vertex[0].px, vertex[0].py, vertex[0].py,
		(INTSML)trans, (INTSML)angle, io_gdsstructure)) == NONODEINST)
			longjmp(env, (INTBIG)"Could not create text marker");

	/* now add the text */
	if ((var = setvalkey((INTBIG)ni, VNODEINST, el_node_name, (INTBIG)charstring,
		VSTRING|VDISPLAY)) == NOVARIABLE)
			longjmp(env, (INTBIG)"Could not create text string");

	/* set the text size and orientation */
	switch (rounddouble (scale))
	{
		case 0:
		case 1: var->textdescript = (TXT4P << VTSIZESH); break;
		case 2: var->textdescript = (TXT6P << VTSIZESH); break;
		case 3: var->textdescript = (TXT8P << VTSIZESH); break;
		case 4: var->textdescript = (TXT10P << VTSIZESH); break;
		case 5: var->textdescript = (TXT12P << VTSIZESH); break;
		case 6: var->textdescript = (TXT14P << VTSIZESH); break;
		case 7: var->textdescript = (TXT16P << VTSIZESH); break;
		case 8: var->textdescript = (TXT18P << VTSIZESH); break;
		default: var->textdescript = (TXT20P << VTSIZESH);
	}

	/* determine the presentation */
	var->textdescript = (var->textdescript & ~VTPOSITION);
	switch (vjust)
	{
		case 1:	/* top */
			switch (hjust)
			{
				case 1: var->textdescript |= VTPOSUPRIGHT; break;
				case 2: var->textdescript |= VTPOSUPLEFT; break;
				default: var->textdescript |= VTPOSUP;
			}
			break;
		case 2:	/* bottom */
			switch (hjust)
			{
				case 1: var->textdescript |= VTPOSDOWNRIGHT; break;
				case 2: var->textdescript |= VTPOSDOWNLEFT; break;
				default: var->textdescript |= VTPOSDOWN;
			}
			break;
		default:	/* centered */
			switch (hjust)
			{
				case 1: var->textdescript |= VTPOSRIGHT; break;
				case 2: var->textdescript |= VTPOSLEFT; break;
				default: var->textdescript |= VTPOSCENT;
			}
	}
}

/*
 * This routine rewritten by Steven Rubin to be more robust in finding
 * pure-layer nodes associated with the GDS layer numbers.
 */
void io_gdsIinit(void)
{
	INTSML i;
	INTBIG *layer_numbers, j, count, which;
	VARIABLE *var;
	static POLYGON *poly = NOPOLYGON;
	REGISTER NODEINST *ni;
	REGISTER NODEPROTO *np;

	/* create the polygon */
	if (poly == NOPOLYGON) poly = allocpolygon(4, io_aid->cluster);

	/* run through the node prototypes in this technology */
	ni = dummynode();

	/* create the layer mapping table */
	for (i=0; i<MAXLAYERS; i++) layer_nodes[i] = NONODEPROTO;
	if ((var = getval((INTBIG)el_curtech, VTECHNOLOGY, VINTEGER|VISARRAY,
		"IO_gds_layer_numbers")) != NOVARIABLE)
	{
		layer_numbers = (INTBIG *)var->addr;
		count = getlength(var);
		for (i=0; i<count; i++)
		{
			which = layer_numbers[i];
			if (which >= 0 && which < MAXLAYERS)
			{
				for(np = el_curtech->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
				{
					if (((np->userbits&NFUNCTION)>>NFUNCTIONSH) != NPNODE) continue;
					ni->proto = np;
					ni->lowx  = np->lowx;
					ni->highx = np->highx;
					ni->lowy  = np->lowy;
					ni->highy = np->highy;
					j = nodepolys(ni);
					if (j != 1) continue;
					shapenodepoly(ni, 0, poly);
					if (i == poly->layer) break;
				}
				if (np != NONODEPROTO)
				{
					layer_nodes[which] = np;
					continue;
				}
				if (layer_nodes[which] != NONODEPROTO) continue;
				for(j=i+1; j<count; j++) if (which == layer_numbers[j]) break;
				if (j < count) continue;
				ttyputmsg("Cannot read GDS layer %d (%s)", which, layername(el_curtech, i));
			}
		}
	} else ttyputmsg("Technology has no GDSII layer numbers");

	buffer_index = -1;
	record_count = 0;
	block_count = 1;
	cur_layer = cur_sublayer = 0;
}

void io_gdsIdetermine_time(char *time_string)
{
	REGISTER INTBIG i;
	char time_array[7][IDSTRING+1];

	for (i = 0; i < 6; i++)
	{
		if (token == GDS_SHORT_NUMBER)
		{
			(void)sprintf(time_array[i],"%02d", tokenvalue_16);
			io_gdsIgettoken();
		} else
		{
			io_gdsIonlyone(io_gdsIrecovery_set, GDS_SHORT_NUMBER);
			io_gdsIerror(6);
		}
	}
	(void)sprintf(time_string,"%s-%s-%s at %s:%s:%s", time_array[1], time_array[2],
		time_array[0], time_array[3], time_array[4], time_array[5]);
}

void io_gdsIheader(void)
{
	if (token == GDS_HEADER)
	{
		io_gdsIgettoken();
		if (token == GDS_SHORT_NUMBER)
		{
			io_gdsIitoa(gds_version_string, tokenvalue_16);
		} else
		{
			io_gdsIonlyone(io_gdsIrecovery_set, GDS_BGNLIB);
			io_gdsIerror(2);
		}
	} else
	{
		io_gdsIonlyone(io_gdsIrecovery_set, GDS_BGNLIB);
		io_gdsIerror(3);
	}
}

void io_gdsIlibrary(void)
{
	char mod_time[IDSTRING+1], create_time[IDSTRING+1];

	if (token == GDS_BGNLIB)
	{
		io_gdsIgettoken();
		io_gdsIdetermine_time(create_time);
		io_gdsIdetermine_time(mod_time);
		if (token == GDS_LIBNAME)
		{
			io_gdsIgettoken();
			if (token == GDS_IDENT)
			{
				(void)strcpy(libraryname, tokenstring);
			} else
			{
				io_gdsIonlyone(io_gdsIrecovery_set, GDS_UNITS);
				io_gdsIerror(4);
			}
		} else
		{
			io_gdsIonlyone(io_gdsIrecovery_set, GDS_UNITS);
		}
	} else
	{
		io_gdsIonlyone(io_gdsIrecovery_set, GDS_UNITS);
		io_gdsIerror(5);
	}
}

void io_gdsIreflibs(void)
{
	io_gdsIgettoken();
	io_gdsIgettoken();
}

void io_gdsIfonts(void)
{
	io_gdsIgettoken();
	io_gdsIgettoken();
}

void io_gdsIbackup(void)
{
	char backup_string[IDSTRING+1];

	io_gdsIgettoken();
	if (token == GDS_SHORT_NUMBER)
	{
		io_gdsIitoa(backup_string, tokenvalue_16);
	} else
	{
		io_gdsIcopyset(io_gdsIbackup_set, io_gdsIrecovery_set);
		io_gdsIerror(8);
	}
	io_gdsIgettoken();
}

void io_gdsIattrtable(void)
{
	io_gdsIgettoken();
	if (token == GDS_IDENT)
	{
		io_gdsIgettoken();
	}
}

void io_gdsIunits(void)
{
	double meter_unit, db_unit;
	double microns_per_user_unit;
	char realstring[IDSTRING+1];

	if (token == GDS_UNITS)
	{
		io_gdsIgettoken();
		if (token == GDS_REALNUM)
		{
			db_unit = realtoken_dp;
			technology_scale_factor = rounddouble(1.0 / realtoken_dp);
			io_gdsIgettoken();
			meter_unit = realtoken_dp;
			microns_per_user_unit = (meter_unit / db_unit) * 1.0e6;
			(void)sprintf(realstring, "%6.3f", microns_per_user_unit);

			/* don't change the cast in this equation - roundoff error! */
			io_gds_scale = meter_unit  * (double)(1000000 * scalefromdispunit(1.0, DISPUNITMIC));
		} else
		{
			io_gdsIonlyone(io_gdsIrecovery_set, GDS_BGNSTR);
			io_gdsIerror(9);
		}
	} else
	{
		io_gdsIonlyone(io_gdsIrecovery_set, GDS_BGNSTR);
		io_gdsIerror(10);
	}
}

void io_gdsIunsupported(gsymbol bad_op_set[])
{
	if (io_gdsImember(token, bad_op_set) != 0)
	do
	{
		io_gdsIgettoken();
	} while (io_gdsImember(token, io_gdsIgood_op_set) == 0);
}

void io_gdsIdetermine_points(INTBIG min_points, INTBIG max_points, INTBIG *point_counter)
{
	(*point_counter) = 0;
	while (token == GDS_NUMBER)
	{
		vertex[*point_counter].px = rounddouble((double)tokenvalue_32 * io_gds_scale);
		io_gdsIgettoken();
		vertex[*point_counter].py = rounddouble((double)tokenvalue_32 * io_gds_scale);
		(*point_counter)++;
		if (*point_counter > max_points)
		{
			ttyputmsg("Found %d points", *point_counter);
			io_gdsIerror(21);
		}
		io_gdsIgettoken();
	}
	if (*point_counter < min_points)
	{
		ttyputmsg("Found %d points", *point_counter);
		io_gdsIerror(20);
	}
}

void io_gdsIdetermine_orientation(INTBIG *angle, INTBIG *trans, double *scale)
{
	INTSML mirror_x, abs_scale, abs_angle;
	double anglevalue;

	anglevalue = 0.0;
	io_gdsIgettoken();
	if (token == GDS_FLAGSYM)
	{
		if (tokenflags.u.f.flag0) mirror_x = 1; else
			mirror_x = 0;
		if (tokenflags.u.f.flag13) abs_scale = 1; else
			abs_scale = 0;
		if (tokenflags.u.f.flag14) abs_angle = 1; else
			abs_angle = 0;
		io_gdsIgettoken();
	} else
	{
		io_gdsIonlyone(io_gdsIrecovery_set, GDS_XY);
		io_gdsIerror(19);
	}
	if (token == GDS_MAG)
	{
		io_gdsIgettoken();
		*scale = realtoken_dp;
		io_gdsIgettoken();
	}
	if (token == GDS_ANGLE)
	{
		io_gdsIgettoken();
		anglevalue = realtoken_dp * 10;
		io_gdsIgettoken();
	}
	io_gdsItranslate(angle, trans, anglevalue, mirror_x);
}

void io_gdsIdetermine_layer(INTBIG *layer, INTBIG *sublayer)
{
	*layer = cur_layer;
	*sublayer = cur_sublayer;
	if (token == GDS_LAYER)
	{
		io_gdsIgettoken();
		if (token == GDS_SHORT_NUMBER)
		{
			cur_layer = *layer = tokenvalue_16;
			if (layer_nodes[cur_layer] == NONODEPROTO)
			{
				ttyputmsg("GDS layer %d unknown, using Generic:DRC", cur_layer);
				layer_nodes[cur_layer] = gen_drcprim;
			}
			layer_node = layer_nodes[cur_layer];

			io_gdsIgettoken();
			if (io_gdsImember(token, io_gdsImask_set) != 0)
			{
				io_gdsIgettoken();
				if (tokenvalue_16 != 0)
				{
					cur_sublayer = *sublayer = tokenvalue_16;
				}
			} else
			{
				io_gdsIonlyone(io_gdsIrecovery_set, GDS_XY);
				io_gdsIerror(24);
			}
		} else
		{
			io_gdsIonlyone(io_gdsIrecovery_set, GDS_XY);
			io_gdsIerror(22);
		}
	} else
	{
		io_gdsIonlyone(io_gdsIrecovery_set, GDS_XY);
		io_gdsIerror(23);
	}
}

void io_gdsIbeginstructure(void)
{
	char create_time[IDSTRING+1], mod_time[IDSTRING+1];

	if (token == GDS_BGNSTR)
	{
		io_gdsIgettoken();
		io_gdsIdetermine_time(create_time);
		io_gdsIdetermine_time(mod_time);
		if (token == GDS_STRNAME)
		{
			io_gdsIgettoken();
			if (token == GDS_IDENT)
			{
				/* look for this nodeproto */
				for (io_gdsstructure = library->firstnodeproto; io_gdsstructure != NONODEPROTO;
					 io_gdsstructure = io_gdsstructure->nextnodeproto)
				if (!strcmp(io_gdsstructure->cell->cellname, tokenstring)) break;
				if (io_gdsstructure == NONODEPROTO)
				{
					/* create the proto */
					if ((io_gdsstructure = newnodeproto(tokenstring, library)) == NONODEPROTO)
						longjmp(env, (INTBIG)"failed to create structure");
					if (io_verbose < 0 && file_size > 0)
					{
						(void)initinfstr();
						(void)addstringtoinfstr("Reading ");
						(void)addstringtoinfstr(tokenstring);
						DiaSetText(2, returninfstr());
					} else if (io_verbose != 0) ttyputmsg("Reading %s", tokenstring);

					/* now add the facet center */
					if (newnodeinst(gen_facetcenterprim, gen_facetcenterprim->lowx,
						gen_facetcenterprim->highx, gen_facetcenterprim->lowy,
						gen_facetcenterprim->highy, 0, 0, io_gdsstructure) == NONODEINST)
							longjmp(env, (INTBIG)"failed to create facet center");
					if (library->curnodeproto == NONODEPROTO)
						library->curnodeproto = io_gdsstructure;
				}
			} else
			{
				io_gdsIcopyset(io_gdsIshape_set, io_gdsIrecovery_set);
				io_gdsIappendset(io_gdsIrecovery_set, GDS_ENDSTR);
				io_gdsIerror(12);
			}
		} else
		{
			io_gdsIcopyset(io_gdsIshape_set, io_gdsIrecovery_set);
			io_gdsIappendset(io_gdsIrecovery_set, GDS_ENDSTR);
			io_gdsIerror(13);
		}
	} else
	{
		io_gdsIcopyset(io_gdsIshape_set, io_gdsIrecovery_set);
		io_gdsIappendset(io_gdsIrecovery_set, GDS_ENDSTR);
		io_gdsIerror(14);
	}
}

void io_gdsIdetermine_property(void)
{
	INTSML propattr;
	char propvalue[IDSTRING+1];

	io_gdsIgettoken();
	propattr = tokenvalue_16;
	io_gdsIgettoken();
	if (token == GDS_PROPVALUE)
	{
		io_gdsIgettoken();
		(void)strcpy(propvalue, tokenstring);
		/* add to the current structure as a variable ? ... */
		io_gdsIgettoken();
	} else
	{
		io_gdsIonlyone(io_gdsIrecovery_set, GDS_ENDEL);
		io_gdsIerror(29);
	}
	/* close Attribute */
}

/* need more info ... */
void io_gdsIdetermine_node(void)
{
	INTBIG n, layer, nodetype;

	io_gdsIgettoken();
	io_gdsIunsupported (io_gdsIunsupported_set);
	layer = cur_layer;
	if (token == GDS_LAYER)
	{
		io_gdsIgettoken();
		if (token == GDS_SHORT_NUMBER)
		{
			cur_layer = layer = tokenvalue_16;
			if (layer_nodes[cur_layer] == NONODEPROTO)
			{
				ttyputmsg("GDS layer %d unknown, using Generic:DRC", cur_layer);
				layer_nodes[cur_layer] = gen_drcprim;
			}
			layer_node = layer_nodes[cur_layer];

			io_gdsIgettoken();
		}
	} else
	{
		io_gdsIonlyone(io_gdsIrecovery_set, GDS_ENDEL);
		io_gdsIerror(18);
	}

	/* should do something with node type??? */
	if (token == GDS_NODETYPE)
	{
		io_gdsIgettoken();
		if (token == GDS_SHORT_NUMBER)
		{
			nodetype = tokenvalue_16;
		}
		io_gdsIgettoken();
	}

	/* make a dot */
	if (token == GDS_XY)
	{
		io_gdsIgettoken();
		io_gdsIdetermine_points (1,1, &n);

		/* create the node */
		if (newnodeinst (layer_node, vertex[0].px, vertex[0].px, vertex[0].py, vertex[0].py, 0, 0,
			io_gdsstructure) == NONODEINST)
				longjmp(env, (INTBIG) "failed to create NODE");
	} else
	{
		io_gdsIonlyone(io_gdsIrecovery_set, GDS_ENDEL);
		io_gdsIerror(18);
	}
}

/* untested feature, I don't have a box type */
void io_gdsIdetermine_box(void)
{
	INTBIG n, layer, sublayer;

	io_gdsIgettoken();
	io_gdsIunsupported(io_gdsIunsupported_set);
	io_gdsIdetermine_layer(&layer, &sublayer);
	if (token == GDS_XY)
	{
		io_gdsIgettoken();
		io_gdsIdetermine_points(2, MAXPOINTS, &n);
		/* create the box */
		if (newnodeinst(layer_node, vertex[0].px, vertex[1].px, vertex[0].py, vertex[1].py , 0, 0,
			io_gdsstructure) == NONODEINST)
				longjmp(env, (INTBIG) "failed to create box");
	} else
	{
		io_gdsIonlyone(io_gdsIrecovery_set, GDS_ENDEL);
		io_gdsIerror(18);
	}
}

void io_gdsIdetermine_shape(void)
{
	INTBIG n, layer, sublayer;
	shape_type perimeter, oclass;

	io_gdsIgettoken();
	io_gdsIunsupported(io_gdsIunsupported_set);
	io_gdsIdetermine_layer(&layer, &sublayer);
	io_gdsIgettoken();
	if (token == GDS_XY)
	{
		io_gdsIgettoken();
		io_gdsIdetermine_points(3, MAXPOINTS, &n);
		io_gdsIdetermine_boundary(n, &perimeter, &oclass);
		io_gdsIshape(n, perimeter, oclass);
	} else
	{
		io_gdsIonlyone(io_gdsIrecovery_set, GDS_ENDEL);
		io_gdsIerror(18);
	}
}

void io_gdsIdetermine_path(void)
{
	INTBIG endcode, n, layer, sublayer;
	INTBIG width, lx, ly, hx, hy, i;

	endcode = 0;
	io_gdsIgettoken();
	io_gdsIunsupported(io_gdsIunsupported_set);
	io_gdsIdetermine_layer(&layer, &sublayer);
	io_gdsIgettoken();
	if (token == GDS_PATHTYPE)
	{
		io_gdsIgettoken();
		endcode = tokenvalue_16;
		io_gdsIgettoken();
	}
	if (token == GDS_WIDTH)
	{
		io_gdsIgettoken();
		width = rounddouble((double)tokenvalue_32 * io_gds_scale);
		io_gdsIgettoken();
	} else
	{
		width = 0;
	}

	if (token == GDS_XY)
	{
		io_gdsIgettoken();
		io_gdsIdetermine_points(2, MAXPOINTS, &n);
		/* construct the path */
		for (i=0; i < n-1; i++)
		{
			lx = hx = vertex[i].px; ly = hy = vertex[i].py;
			if (lx > vertex[i+1].px) lx = vertex[i+1].px;
			if (hx < vertex[i+1].px) hx = vertex[i+1].px;
			if (ly > vertex[i+1].py) ly = vertex[i+1].py;
			if (hy < vertex[i+1].py) hy = vertex[i+1].py;
			/* 10 MAR 93 - RLW, Queen's */
			/* handle all paths, except two segment paths with flush end type */
			if (n != 3 || endcode == 2)
			{
				if (ly == hy || endcode == 2 || (endcode == 0 && i != 0 && i+1 < n-1))
				{
					ly -= width/2;
					hy += width/2;
				}
				if (lx == hx || endcode == 2 || (endcode == 0 && i != 0 && i+1 < n-1))
				{
					lx -= width/2;
					hx += width/2;
				}
			} else
			{
				/* special case for a two segment path with a flush end type */
				if (endcode == 0)
				{
					if (vertex[0].px < vertex[1].px && i == 0) hx += width/2;
					if (vertex[1].px < vertex[2].px && i == 1) lx -= width/2;
					if (vertex[1].px > vertex[2].px && i == 1) hx += width/2;
					if (vertex[0].px > vertex[1].px && i == 0) lx -= width/2;
					if (lx == hx)
					{
						lx -= width/2;
						hx += width/2;
					}
					if (ly == hy)
					{
						ly -= width/2;
						hy += width/2;
					}
				}
			}
			if (newnodeinst(layer_node, lx, hx, ly, hy, 0, 0, io_gdsstructure) == NONODEINST)
				longjmp(env, (INTBIG)"failed to create trace");
		}
	} else
	{
		io_gdsIonlyone(io_gdsIrecovery_set, GDS_ENDEL);
		io_gdsIerror(25);
	}
}

void io_gdsIgetproto(char *name)
{
	NODEPROTO *np;

	/* scan for this proto */
	for (np = library->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		if (!strcmp (np->cell->cellname, name)) break;
	if (np == NONODEPROTO)
	{
		/* FILO order, create this nodeproto */
		if ((np = newnodeproto(tokenstring, library)) == NONODEPROTO)
			longjmp(env, (INTBIG)"failed to create SREF proto");
		if (io_verbose < 0 && file_size > 0)
		{
			(void)initinfstr();
			(void)addstringtoinfstr("Reading ");
			(void)addstringtoinfstr(tokenstring);
			DiaSetText(2, returninfstr());
		} else if (io_verbose != 0) ttyputmsg("Reading %s", tokenstring);

		/* now add the facet center */
		if (newnodeinst(gen_facetcenterprim, gen_facetcenterprim->lowx,
			gen_facetcenterprim->highx, gen_facetcenterprim->lowy,
				gen_facetcenterprim->highy, 0, 0, np) == NONODEINST)
					longjmp(env, (INTBIG)"failed to create facet center");
	}

	/* set the reference node proto */
	io_gdssref = np;
}

void io_gdsIdetermine_sref(void)
{
	INTBIG n, angle, trans;
	double scale;

	io_gdsIgettoken();
	io_gdsIunsupported(io_gdsIunsupported_set);
	if (token == GDS_SNAME)
	{
		io_gdsIgettoken();
		io_gdsIgetproto(tokenstring);
		io_gdsIgettoken();
		if (token == GDS_STRANS)
			io_gdsIdetermine_orientation(&angle, &trans, &scale); else
		{
			angle = 0; trans = 0;
			scale = 1.0;
		}
		if (token == GDS_XY)
		{
			io_gdsIgettoken();
			io_gdsIdetermine_points(1, 1, &n);
			/* close Translate */
		} else
		{
			io_gdsIonlyone(io_gdsIrecovery_set, GDS_ENDEL);
			io_gdsIerror(17);
		}
		/* create the node */
		if ((instance = newnodeinst(io_gdssref, vertex[0].px + io_gdssref->lowx,
			vertex[0].px+io_gdssref->highx,vertex[0].py+io_gdssref->lowy,
				vertex[0].py+io_gdssref->highy, (INTSML)trans,
					(INTSML)angle, io_gdsstructure)) == NONODEINST)
						longjmp(env, (INTBIG)"failed to create SREF");
	} else
	{
		io_gdsIonlyone(io_gdsIrecovery_set, GDS_ENDEL);
		io_gdsIerror(16);
	}
}

void io_gdsIdetermine_aref(void)
{
	INTBIG n, ncols, nrows, angle, trans;
	double scale;

	io_gdsIgettoken();
	io_gdsIunsupported(io_gdsIunsupported_set);
	if (token == GDS_SNAME)
	{
		io_gdsIgettoken();
		/* get this nodeproto */
		io_gdsIgetproto(tokenstring);
		io_gdsIgettoken();
		if (token == GDS_STRANS) io_gdsIdetermine_orientation(&angle, &trans, &scale); else
		{
			angle = trans = 0;
			scale = 1.0;
		}
		if (token == GDS_COLROW)
		{
			io_gdsIgettoken();
			ncols = tokenvalue_16;
			io_gdsIgettoken();
			nrows = tokenvalue_16;
			io_gdsIgettoken();
		}
		if (token == GDS_XY)
		{
			io_gdsIgettoken();
			io_gdsIdetermine_points(3, 3, &n);
			io_gdsIstep(ncols, nrows, angle, trans);
		} else
		{
			io_gdsIonlyone(io_gdsIrecovery_set, GDS_ENDEL);
			io_gdsIerror(27);
		}
	} else
	{
		io_gdsIonlyone(io_gdsIrecovery_set, GDS_ENDEL);
		io_gdsIerror(28);
	}
}

void io_gdsIdetermine_justification(INTBIG *vert_just, INTBIG *horiz_just)
{
	INTBIG font_libno;

	io_gdsIgettoken();
	if (token == GDS_FLAGSYM)
	{
		font_libno = tokenflags.u.value & 0x0030;
		font_libno = font_libno >> 4;
		(*vert_just) = tokenflags.u.value & 0x000C;
		(*vert_just) = (*vert_just) >> 2;
		(*horiz_just) = tokenflags.u.value & 0x0003;
		io_gdsIgettoken();
	} else
	{
		io_gdsIonlyone(io_gdsIrecovery_set, GDS_XY);
		io_gdsIerror(27);
	}
}

void io_gdsIdetermine_text(void)
{
	INTBIG vert_just, horiz_just, layer, sublayer;
	char textstring[TEXTLINE+1];
	INTBIG n, angle, trans, pathcode, pathwidth;
	double scale;

	io_gdsIgettoken();
	io_gdsIunsupported(io_gdsIunsupported_set);
	io_gdsIdetermine_layer(&layer, &sublayer);
	io_gdsIgettoken();
	vert_just = -1;
	horiz_just = -1;
	if (token == GDS_PRESENTATION)
	io_gdsIdetermine_justification(&vert_just, &horiz_just);
	if (token == GDS_PATHTYPE)
	{
		io_gdsIgettoken();
		pathcode = tokenvalue_16;
		io_gdsIgettoken();
	}
	if (token == GDS_WIDTH)
	{
		io_gdsIgettoken();
		pathwidth = rounddouble((double)tokenvalue_32 * io_gds_scale);
		io_gdsIgettoken();
	}
	angle = trans = 0;
	if (token == GDS_STRANS)
	io_gdsIdetermine_orientation(&angle, &trans, &scale);
	if (token == GDS_XY)
	{
		io_gdsIgettoken();
		io_gdsIdetermine_points(1, 1, &n);
	} else
	{
		io_gdsIonlyone(io_gdsIrecovery_set, GDS_ENDEL);
		io_gdsIerror(26);
	}
	if (token == GDS_STRING)
	{
		if (record_count |= 0)
		{
			io_gdsIgettoken();
			(void)strcpy(textstring,tokenstring);
		}
		else textstring[0] = '\0';
		io_gdsIgettoken();
	}
	io_gdsItext(textstring, vert_just, horiz_just, angle, trans, scale);
}

void io_gdsIelement(INTSML *empty_structure)
{
	while (io_gdsImember(token, io_gdsIshape_set) != 0)
		switch (token)
	{
		case GDS_AREF:     io_gdsIdetermine_aref();    break;
		case GDS_SREF:     io_gdsIdetermine_sref();    break;
		case GDS_BOUNDARY: io_gdsIdetermine_shape();   break;
		case GDS_PATH:     io_gdsIdetermine_path();    break;
		case GDS_NODE:     io_gdsIdetermine_node();    break;
		case GDS_TEXTSYM:  io_gdsIdetermine_text();    break;
		case GDS_BOX:      io_gdsIdetermine_box();     break;
		default:                                       break;
	}

	if (token == GDS_PROPATTR)
		io_gdsIdetermine_property();
	if (token != GDS_ENDEL)
	{
		io_gdsIonlyone(io_gdsIrecovery_set, GDS_BGNSTR);
		io_gdsIerror(15);
	}
	(*empty_structure) = 0;
}

void io_gdsIstructure(void)
{
	INTSML empty_structure;

	io_gdsIbeginstructure();
	empty_structure = 1;
	io_gdsIgettoken();
	while (token != GDS_ENDSTR)
	{
		io_gdsIelement(&empty_structure);
		io_gdsIgettoken();
	}
	if (empty_structure != 0)
	{
		ttyputmsg("%s", io_gdsIerror_string[11]);
		ttyputerr("Error at block number %5d and offset %4d\n",
			block_count, buffer_index);
	}

	db_boundfacet(io_gdsstructure, &io_gdsstructure->lowx, &io_gdsstructure->highx,
		&io_gdsstructure->lowy, &io_gdsstructure->highy);
}

INTSML io_gdsIload(FILE *input)
{
	io_gdsIinit();
	io_gdsIgettoken();
	io_gdsIheader();
	io_gdsIgettoken();
	io_gdsIlibrary();
	io_gdsIgettoken();
	while (io_gdsImember(token, io_gdsIoption_set) != 0) switch (token)
	{
		case GDS_REFLIBS:
			io_gdsIreflibs();     break;
		case GDS_FONTS:
			io_gdsIfonts();       break;
		case GDS_ATTRTABLE:
			io_gdsIattrtable();   break;
		case GDS_GENERATIONS:
			io_gdsIbackup();      break;
		default:                  break;
	}
	while (token != GDS_UNITS)
		io_gdsIgettoken();
	io_gdsIunits();
	io_gdsIgettoken();
	while (token != GDS_ENDLIB)
	{
		io_gdsIstructure();
		io_gdsIgettoken();
	}
	if (err_count) return(1);
	return(0);
}

/* module: io_readgdslibrary
 * function:  Electric entry-point in gdsII stream library read.
 * inputs:
 * lib - the library to read
 * returns 0 on success, 1 on error
 */
INTSML io_readgdslibrary(LIBRARY *lib)
{
	FILE *input;
	INTSML err;
	char *msg, *filename;
	NODEPROTO *np;
	extern DIALOG usr_progressdialog;

	library = lib;

	/* get the gds file */
	if ((input = xopen(library->libfile, FILETYPEGDS, "", &filename)) == NULL)
	{
		ttyputerr("File %s not found", library->libfile);
		return(1);
	}

	stream.fp = input;
	file_size = filesize(input);
	if (io_verbose < 0 && file_size > 0)
	{
		if (DiaInitDialog(&usr_progressdialog) != 0)
		{
			xclose(input);
			return(1);
		}
		DiaPercent(1, 0);
	}
	bytes_read = 0;

	/* setup the error return location */
	if ((msg = (char *)setjmp(env)) != NULL)
	{
		ttyputerr("Error: %s", msg);
		xclose(input);
		if (io_verbose < 0 && file_size > 0) DiaDoneDialog();
		return(1);
	}

	/* set default objects */
	layer_node = gen_drcprim;

	err = io_gdsIload(input);

	/* finish up */
	xclose(input);
	if (io_verbose < 0 && file_size > 0)
	{
		DiaSetText(2, "Cleaning up...");
		DiaPercent(1, 100);
	}

	/* determine which facets need to be recomputed */
	for(np = library->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		np->temp1 = 0;
	io_gdsrecomputed = 0;
	for(np = library->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		if (np->temp1 == 0) io_recompute(np);
	if (io_verbose < 0 && file_size > 0) DiaDoneDialog();
	return(err);
}

void io_recompute(NODEPROTO *np)
{
	REGISTER INTBIG cx, cy;
	char numsofar[50];
	INTBIG ox, oy;
	XARRAY trans;
	REGISTER NODEINST *ni;

	for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		if (ni->proto->index != 0) continue;
		if (ni->proto->temp1 == 0) io_recompute(ni->proto);
		if (ni->highx-ni->lowx == ni->proto->highx-ni->proto->lowx &&
			ni->highx-ni->lowx == ni->proto->highx-ni->proto->lowx) continue;

		makeangle(ni->rotation, ni->transpose, trans);

		/* now transform the center */
		cx = (ni->proto->highx + ni->proto->lowx) / 2;
		cy = (ni->proto->highy + ni->proto->lowy) / 2;
		xform(cx, cy, &ox, &oy, trans);

		/* calculate the offset from the original center */
		ox -= cx;   oy -= cy;
		ni->lowx += (ni->proto->lowx + ox);
		ni->highx += (ni->proto->highx + ox);
		ni->lowy += (ni->proto->lowy + oy);
		ni->highy += (ni->proto->highy + oy);

		/* now correct the dimensions of this np */
		updategeom(ni->geom, np);
		io_gdsrecomputed++;
		if ((io_gdsrecomputed%1000) == 0 && io_verbose < 0 && file_size > 0)
		{
			(void)sprintf(numsofar, "Cleaned up %d instances...", io_gdsrecomputed);
			DiaSetText(2, numsofar);
		}
	}
	db_boundfacet(np, &np->lowx, &np->highx, &np->lowy, &np->highy);
}

#endif  /* IOGDS - at top */
