/*
 * Electric(tm) VLSI Design System
 *
 * File: iopsout.c
 * Input/output analysis aid: PostScript generation
 * Written by: Steven M. Rubin, Electric Editor Incorporated
 *
 * Copyright (c) 1998 Electric Editor Incorporated.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Electric Editor Incorporated
 * 23470 Sunset Drive, Suite 108
 * Los Gatos, California 95033
 * support@electriceditor.com
 */

/*
 * Rules used in creating the PostScript file.
 *
 * 1) Plain PostScript - the image will be rotated on the page if appropriate
 *                     - when using an area-highlight objects completely
 *                         outside the highlight area are not printed
 *                     - file created with a '.ps' extension
 *
 * 2) Encapsulated PostScript - the image is NOT rotated to fit the page
 *                            - when using an area-highlight the image clip
 *                                path is set to the highlight area size
 *                            - file created with a '.eps' extension
 *
 * The rules for EPS are such because in most cases the EPS file will be used
 * inside a publishing package.  The publishing package can take care of any
 * rotation, etc. that may be necessary.
 */

#include "config.h"
#if IOPOSTSCRIPT

#include "global.h"
#include "egraphics.h"
#include "eio.h"

#define CLIP 1  /* set if the bounding box is used as the clip path, for EPS only */

#define	THRESH	      2000000
#define	UNITSX        (562*4)
#define	UNITSY        (742*4)

static FILE      *io_psout;
static WINDOW     io_pswindow;
static INTBIG     io_psoffset, io_psscale;
static XARRAY     io_psmatrix = {{0,0,0},{0,0,0},{0,0,0}};
static GRAPHICS **io_psgraphicsseen;
static INTSML     io_psgraphicsseenlimit = 0, io_psgraphicsseencount;
static INTSML     io_psepsformat;
static INTSML     io_psdotput;				/* PostScript for dot put out */
static INTSML     io_psdrawlineput;			/* PostScript for line put out */
static INTSML     io_pspolygonput;			/* PostScript for polygon put out */
static INTSML     io_psfilledpolygonput;	/* PostScript for filled polygon put out */
static INTSML     io_psstringput;			/* PostScript for strings put out */

GRAPHICS io_psblack = {LAYERO, BLACK, {SOLIDC, SOLIDC, SOLIDC, SOLIDC},
	{0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF}, NOVARIABLE, 0};
static io_psconvert[9] = {4, 6, 8, 10, 12, 14, 16, 18, 20};

/* prototypes for local routines */
void   io_psarc(INTBIG, INTBIG, INTBIG, INTBIG, INTBIG, INTBIG);
void   io_pscircle(INTBIG, INTBIG, INTBIG, INTBIG);
void   io_psdisc(INTBIG, INTBIG, INTBIG, INTBIG, GRAPHICS*);
void   io_psdot(INTBIG, INTBIG);
void   io_psline(INTBIG, INTBIG, INTBIG, INTBIG, INTSML);
INTSML io_pspattern(GRAPHICS*);
INTSML io_pspoly(POLYGON*, WINDOW*);
void   io_pspolygon(INTBIG*, INTBIG*, INTSML, GRAPHICS*);
void   io_pstext(INTSML, INTBIG, INTBIG, INTBIG, INTBIG, INTSML, char*);
void   io_pswrite(char *s, ...);
void   io_pswritestring(char*);
void   io_psxform(INTBIG, INTBIG, INTBIG*, INTBIG*);

/*
 * routine to write out a PostScript file
 */
INTSML io_writepostscriptlibrary(LIBRARY *lib)
{
	char file[100], *facetname, *creation, *username, *truename;
	REGISTER NODEPROTO *np;
	INTBIG lx, hx, ly, hy, hlx, hhx, hly, hhy, gridlx, gridly, gridx, gridy,
		cx, cy, i, j, psulx, psuhx, psuly, psuhy, sulx, suhx,
		suly, suhy, sslx, sshx, ssly, sshy;
	INTBIG bblx, bbhx, bbly, bbhy, t, t1, t2;
	UINTBIG curdate;
	VARIABLE *var;
	REGISTER VARIABLE *varstate;
	static POLYGON *poly = NOPOLYGON;
	extern AIDENTRY *us_aid;

	np = lib->curnodeproto;
	if (np == NONODEPROTO)
	{
		ttyputerr("No current facet to plot");
		return(1);
	}

	/* clear flags that tell whether headers have been included */
	io_psdotput = 0;
	io_psdrawlineput = 0;
	io_pspolygonput = 0;
	io_psfilledpolygonput = 0;
	io_psstringput = 0;

	/* See if encapsulated output is requested */
	varstate = getvalkey((INTBIG)io_aid, VAID, VINTEGER, io_state);
	if (varstate != NOVARIABLE && (varstate->addr&EPSPSCRIPT) != 0) io_psepsformat = 1; else
		io_psepsformat = 0;

	/* create the PostScript file */
	/* use file name if set, otherwise use current cell name */
	var = getval((INTBIG)lib->curnodeproto, VNODEPROTO, VSTRING, "IO_postscript_filename");
	if (var != NOVARIABLE)
	{
		(void)sprintf(file, "%s", (char *)var->addr);
	} else
	{
		(void)sprintf(file, "%s.%s", np->cell->cellname, io_psepsformat ? "eps" : "ps");
	}
	io_psout = xcreate(file, FILETYPEPS, "PostScript file", &truename);
	if (io_psout == NULL)
	{
		if (truename != 0) ttyputerr("Cannot create %s", truename);
		return(1);
	}

	/* determine area to plot */
	if (framesize(&hx, &hy, np) == 0)
	{
		hx /= 2;   lx = -hx;
		hy /= 2;   ly = -hy;
	} else
	{
		lx = np->lowx;   hx = np->highx;
		ly = np->lowy;   hy = np->highy;
	}

	/* Get name and date */
	(void)initinfstr();
	(void)addstringtoinfstr(describenodeproto(np));
	facetname = returninfstr();
	curdate = getcurrenttime();
	(void)initinfstr();
	(void)addstringtoinfstr(timetostring(curdate));
	creation = returninfstr();
	(void)initinfstr();
	(void)addstringtoinfstr("%%%%For: ");
	(void)addstringtoinfstr(getenv("USER"));
	username = returninfstr();

	/* see if focusing on highlighted area */
	if (varstate != NOVARIABLE && (varstate->addr&PLOTFOCUS) != 0)
	{
		if (askaid(us_aid, "get-highlighted-area", (INTBIG)&hlx, (INTBIG)&hhx, (INTBIG)&hly, (INTBIG)&hhy) != 0)
			ttyputerr("Warning: no highlighted area; plotting entire facet"); else
		{
			if (hhx == hlx || hhy == hly)
			{
				/* possibly put this code just before the disk file is opened
				   on second thought leave it here - it's OK that an empty file
				   is written because the user obviously wants to create a PS file
				   and just accidentally highlighted the wrong item
				*/
				ttyputerr("Warning: no highlighted area; highlight area and reissue command");
				xclose(io_psout);
				ttyputmsg("Null PostScript file written");
				return(1);
			}
			lx = hlx;   hx = hhx;
			ly = hly;   hy = hhy;
		}
	}

	/* build pseudowindow for text scaling */
	psulx = psuly = 0;
	psuhx = psuhy = 1000;
	io_pswindow.uselx = psulx;
	io_pswindow.usely = psuly;
	io_pswindow.usehx = psuhx;
	io_pswindow.usehy = psuhy;

	io_pswindow.screenlx = lx;
	io_pswindow.screenhx = hx;
	io_pswindow.screenly = ly;
	io_pswindow.screenhy = hy;
	computewindowscale(&io_pswindow);

	/* add a "border" around image to prevent "cropping" by the printer
	   don't need a border for EPS because it's not usually printed by itself
	*/
	if (!io_psepsformat)
	{
		/* increase the x size by 5% */
		i = (hx - lx) / 20;
		if (i == 0) i++;
		hx += i;   lx -= i;

		/* increase the y size by 5% */
		i = (hy - ly) / 20;
		if (i == 0) i++;
		hy += i;   ly -= i;
	}

	/* set the bounding box */
	bblx = lx; bbhx = hx; bbly = ly; bbhy = hy;

	/* PostScript: compute the transformation matrix */
	cx = (hx + lx) / 2;
	cy = (hy + ly) / 2;
	i = mini(((UNITSX-10) << 16) / (hx - lx), ((UNITSY-10) << 16) / (hy - ly));
	j = mini(((UNITSX-10) << 16) / (hy - ly), ((UNITSY-10) << 16) / (hx - lx));
	io_psmatrix[0][0] = i;   io_psmatrix[0][1] = 0;
	io_psmatrix[1][0] = 0;   io_psmatrix[1][1] = i;
	io_psmatrix[2][0] = - (i*cx >> 16) + UNITSX / 2;
	io_psmatrix[2][1] = - (i*cy >> 16) + UNITSY / 2;
	io_psscale = 4;
	io_psoffset = 25*io_psscale;

	/* PostScript: %! */
	if (io_psepsformat) io_pswrite("%%!PS-Adobe-2.0 EPSF-2.0\n"); else
		io_pswrite("%%!PS-Adobe-1.0\n");

	io_pswrite("%%%%Title: %s\n", facetname);
	io_pswrite("%%%%Creator: Electric Design System v%s\n", el_version);
	io_pswrite("%%%%CreationDate: %s\n", creation);
	if (io_psepsformat) io_pswrite("%%%%Pages: 0\n"); else
		io_pswrite("%%%%Pages: 1\n");

	io_psxform(bblx, bbly, &bblx, &bbly);
	io_psxform(bbhx, bbhy, &bbhx, &bbhy);

	if (j > i && !io_psepsformat)
	{
		/*
		 * fiddle with the bbox if image rotated on page
		 * remember bbox coordinates are absolute and are not scaled by the printer
		 */
		t1 = bblx;
		t2 = bbhx;
		bblx = -bbhy + (INTBIG)(9.75 * 300);
		bbhx = -bbly + (INTBIG)(9.75 * 300);
		bbly = t1 + (INTBIG)(1.25 * 300);
		bbhy = t2 + (INTBIG)(1.25 * 300);
	}

	if (bblx>bbhx) { t = bblx; bblx = bbhx; bbhx =t; }
	if (bbly>bbhy) { t = bbly; bbly = bbhy; bbhy =t; }
	bblx = roundfloat(((float)bblx) / (((float)io_psscale)*75.0) * 72.0) * (bblx>=0 ? 1 : -1);
	bbly = roundfloat(((float)bbly) / (((float)io_psscale)*75.0) * 72.0) * (bbly>=0 ? 1 : -1);
	bbhx = roundfloat(((float)bbhx) / (((float)io_psscale)*75.0) * 72.0) * (bbhx>=0 ? 1 : -1);
	bbhy = roundfloat(((float)bbhy) / (((float)io_psscale)*75.0) * 72.0) * (bbhy>=0 ? 1 : -1);
	t =  roundfloat(((float)io_psoffset) / (((float)io_psscale)*75.0) * 72.0);

	/*
	 * SRP920115 Write in EPS format:
	 * if the image is rotated on the page then subtract the offset from bblx and bbhx,
	 * otherwise add the offset.  Also increase the size of the bbox by one "pixel" to
	 * prevent the edges from being obscured by some drawing tools
	 */
	if (j > i && !io_psepsformat)
		io_pswrite("%%%%BoundingBox: %d %d %d %d\n",bblx-t-1, bbly+t-1, bbhx-t+1, bbhy+t+1); else
			io_pswrite("%%%%BoundingBox: %d %d %d %d\n",bblx+t-1, bbly+t-1, bbhx+t+1, bbhy+t+1);
	io_pswrite("%%%%DocumentFonts: Times-Roman\n");
	io_pswrite("%%%%EndComments\n");

	/* PostScript: add some debugging and EPS info */
	if (np != NONODEPROTO)
	{
		/* leave this in for debugging */
		io_pswrite("%% facet dimensions: %d wide x %d high (database units)\n",
			np->highx-np->lowx, np->highy-np->lowy);
		io_pswrite("%% origin: %d %d\n", np->lowx, np->lowy);
	}
	io_pswrite("%% Format modified to match EPSF-2.0, January 1992 SRP\n");
	io_pswrite("%% The non-EPS header does not claim conformance to Adobe-2.0\n");
	io_pswrite("%% because the structure may not be exactly correct. The EPS\n");
	io_pswrite("%% version differs from the .ps version only in the first\n");
	io_pswrite("%% line.  A later revision should declare a private\n");
	io_pswrite("%% dictionary for the EPS form.\n");
	io_pswrite("%% \n");

#ifdef CLIP
	/* set the clip path to be equal to the bounding box, only for EPS */
	if (io_psepsformat)
	{
		io_pswrite("gsave\n");
		io_pswrite("newpath\n");
		io_pswrite("%d %d moveto %d %d lineto %d %d lineto %d %d lineto\n",
			bblx+t-1, bbly+t-1, bbhx+t+1, bbly+t-1, bbhx+t+1, bbhy+t+1, bblx+t-1, bbhy+t+1);
		io_pswrite("closepath\n");
		io_pswrite("clip\n");
		io_pswrite("newpath\n");
	}
#endif

	/* PostScript: make the scale be exactly equal to one page pixel */
	io_pswrite("72 %d div 72 %d div scale\n", io_psscale*75, io_psscale*75);

	/* PostScript: rotate the image if sensible */
	if (j > i && !io_psepsformat)
		io_pswrite("8.5 1.25 add 300 mul 1.25 300 mul translate 90 rotate\n");

	/* PostScript: set the proper typeface */
	io_pswrite("/scaleFont {\n");
	io_pswrite("    /Times-Roman findfont\n");
	io_pswrite("    exch scalefont setfont} def\n");

	/* PostScript: make the line width proper */
	io_pswrite("%d setlinewidth\n", io_psscale/2);

	/* PostScript: make the line ends look right */
	io_pswrite("1 setlinecap\n");

	/* draw the grid if requested */
	if ((el_curwindow->state&(GRIDON|GRIDTOOSMALL)) == GRIDON)
	{
		gridx = el_curwindow->gridx;
		gridy = el_curwindow->gridy;
		grabpoint(np, &gridlx, &gridly);
		var = getval((INTBIG)us_aid, VAID, VINTEGER, "USER_alignment_obj");
		if (var != NOVARIABLE) gridalign(&gridlx, &gridly, var->addr);
		gridlx += gridlx / gridx * gridx;
		gridly += gridly / gridy * gridy;

		/* adjust to ensure that the first point is inside the range */
		while (gridlx < lx) gridlx += gridx;
		while (gridly < ly) gridly += gridy;

		/* plot the loop to the printer */
		for (cx = gridlx; cx <= hx; cx += gridx)
			for (cy = gridly; cy <= hy; cy += gridy)
				io_psdot(cx, cy);

		/* PostScript: write the grid loop */
		io_pswrite("%d %d %d\n{\n", gridlx, gridx, hx);
		io_pswrite("    %d %d %d\n    {\n", gridly, gridy, hy); /* x y */
		io_pswrite("        dup 3 -1 roll dup dup\n");		/* y y x x x */
		io_pswrite("        5 1 roll 3 1 roll\n");		/* x y x y x */
		io_pswrite("        %d mul exch %d mul add 65536 div %d add\n",
			io_psmatrix[0][0], io_psmatrix[1][0], io_psmatrix[2][0]+io_psoffset); /* x y x x' */
		io_pswrite("        3 1 roll\n");			/* x x' y x */
		io_pswrite("        %d mul exch %d mul add 65536 div %d add\n",
			io_psmatrix[0][1], io_psmatrix[1][1], io_psmatrix[2][1]+io_psoffset); /* x x' y' */
		io_pswrite("        newpath moveto 0 0 rlineto stroke\n");
		io_pswrite("    } for\n");
		io_pswrite("} for\n");
	}

	/* initialize list of GRAPHICS modules that have been put out */
	io_psgraphicsseencount = 0;

	/* plot everything */
	sulx = el_curwindow->uselx;      suhx = el_curwindow->usehx;
	suly = el_curwindow->usely;      suhy = el_curwindow->usehy;
	sslx = el_curwindow->screenlx;   sshx = el_curwindow->screenhx;
	ssly = el_curwindow->screenly;   sshy = el_curwindow->screenhy;
	el_curwindow->uselx = io_pswindow.uselx;   el_curwindow->usehx = io_pswindow.usehx;
	el_curwindow->usely = io_pswindow.usely;   el_curwindow->usehy = io_pswindow.usehy;
	el_curwindow->screenlx = io_pswindow.screenlx;
	el_curwindow->screenhx = io_pswindow.screenhx;
	el_curwindow->screenly = io_pswindow.screenly;
	el_curwindow->screenhy = io_pswindow.screenhy;
	computewindowscale(el_curwindow);
	(void)askaid(us_aid, "display-to-routine", io_pspoly);
	el_curwindow->uselx = sulx;      el_curwindow->usehx = suhx;
	el_curwindow->usely = suly;      el_curwindow->usehy = suhy;
	el_curwindow->screenlx = sslx;   el_curwindow->screenhx = sshx;
	el_curwindow->screenly = ssly;   el_curwindow->screenhy = sshy;
	computewindowscale(el_curwindow);

	/* put out dates if requested */
	if (varstate != NOVARIABLE && (varstate->addr&PLOTDATES) != 0)
	{
		/* create the polygon if it doesn't exist */
		if (poly == NOPOLYGON) poly = allocpolygon(1, io_aid->cluster);

		/* plot facet name */
		poly->string = facetname;
		poly->xv[0] = np->highx;
		poly->yv[0] = np->lowy;
		poly->count = 1;
		poly->style = TEXTBOTRIGHT;
		poly->font = TXTMEDIUM;
		poly->desc = &io_psblack;
		(void)io_pspoly(poly, el_curwindow);

		/* plot creation date */
		(void)initinfstr();
		(void)addstringtoinfstr("Created: ");
		(void)addstringtoinfstr(timetostring(np->creationdate));
		poly->string = returninfstr();
		poly->yv[0] = np->lowy + (np->highy-np->lowy) / 20;
		(void)io_pspoly(poly, el_curwindow);

		/* plot revision date */
		(void)initinfstr();
		(void)addstringtoinfstr("Revised: ");
		(void)addstringtoinfstr(timetostring(np->revisiondate));
		poly->string = returninfstr();
		poly->yv[0] = np->lowy + (np->highy-np->lowy) / 10;
		(void)io_pspoly(poly, el_curwindow);
	}

	io_pswrite("showpage\n");
#ifdef CLIP /* restore the graphics state after clipping */
	if (io_psepsformat) io_pswrite("grestore\n");
#endif
	io_pswrite("%%%%Trailer\n");
	xclose(io_psout);
	ttyputmsgf("Wrote PostScript file '%s'", file);
	return(0);
}

/*
 * routine to plot the polygon "poly"
 */
INTSML io_pspoly(POLYGON *poly, WINDOW *win)
{
	REGISTER INTSML k, type, font;
	INTBIG xl, xh, yl, yh, x, y, listx[4], listy[4];

	/* ignore null layers */
	if (poly->desc->bits == LAYERN || poly->desc->col == ALLOFF) return(1);

	/* ignore grids */
	if (poly->style == GRIDDOTS) return(0);

	switch (poly->style)
	{
		case FILLED:
		case FILLEDRECT:
			if (isbox(poly, &xl, &xh, &yl, &yh))
			{
				if (xl == xh)
				{
					if (yl == yh) io_psdot(xl, yl); else
						io_psline(xl, yl, xl, yh, 0);
					break;
				} else if (yl == yh)
				{
					io_psline(xl, yl, xh, yl, 0);
					break;
				}
				listx[0] = xl;   listy[0] = yl;
				listx[1] = xl;   listy[1] = yh;
				listx[2] = xh;   listy[2] = yh;
				listx[3] = xh;   listy[3] = yl;
				io_pspolygon(listx, listy, 4, poly->desc);
			} else
			{
				if (poly->count == 1)
				{
					io_psdot(poly->xv[0], poly->yv[0]);
					break;
				}
				if (poly->count == 2)
				{
					io_psline(poly->xv[0], poly->yv[0], poly->xv[1], poly->yv[1], 0);
					break;
				}
				io_pspolygon(poly->xv, poly->yv, poly->count, poly->desc);
			}
			break;

		case CLOSED:
		case CLOSEDRECT:
			if (isbox(poly, &xl, &xh, &yl, &yh))
			{
				io_psline(xl, yl, xl, yh, 0);
				io_psline(xl, yh, xh, yh, 0);
				io_psline(xh, yh, xh, yl, 0);
				io_psline(xh, yl, xl, yl, 0);
				break;
			}

		case OPENED:
		case OPENEDT1:
		case OPENEDT2:
		case OPENEDT3:
			switch (poly->style)
			{
				case OPENEDT1: type = 1; break;
				case OPENEDT2: type = 2; break;
				case OPENEDT3: type = 3; break;
				default:       type = 0; break;
			}
			for (k = 1; k < poly->count; k++)
				io_psline(poly->xv[k-1], poly->yv[k-1], poly->xv[k], poly->yv[k], type);
			if (poly->style == CLOSED)
			{
				k = poly->count - 1;
				io_psline(poly->xv[k], poly->yv[k], poly->xv[0], poly->yv[0], type);
			}
			break;

		case VECTORS:
			for(k=0; k<poly->count; k += 2)
				io_psline(poly->xv[k], poly->yv[k], poly->xv[k+1], poly->yv[k+1], 0);
			break;

		case CROSS:
		case BIGCROSS:
			getcenter(poly, &x, &y);
			io_psline(x-5, y, x+5, y, 0);
			io_psline(x, y+5, x, y-5, 0);
			break;

		case CROSSED:
			getbbox(poly, &xl, &xh, &yl, &yh);
			io_psline(xl, yl, xl, yh, 0);
			io_psline(xl, yh, xh, yh, 0);
			io_psline(xh, yh, xh, yl, 0);
			io_psline(xh, yl, xl, yl, 0);
			io_psline(xh, yh, xl, yl, 0);
			io_psline(xh, yl, xl, yh, 0);
			break;

		case DISC:
			io_psdisc(poly->xv[0], poly->yv[0], poly->xv[1], poly->yv[1], poly->desc);

		case CIRCLE:
			io_pscircle(poly->xv[0], poly->yv[0], poly->xv[1], poly->yv[1]);
			break;

		case CIRCLEARC:
			io_psarc(poly->xv[0], poly->yv[0], poly->xv[1], poly->yv[1], poly->xv[2], poly->yv[2]);
			break;

		case TEXTCENT:
		case TEXTTOP:
		case TEXTBOT:
		case TEXTLEFT:
		case TEXTRIGHT:
		case TEXTTOPLEFT:
		case TEXTBOTLEFT:
		case TEXTTOPRIGHT:
		case TEXTBOTRIGHT:
		case TEXTBOX:
			font = truefontsize(poly->font, &io_pswindow, el_curtech);
			getbbox(poly, &xl, &xh, &yl, &yh);
			io_pstext(poly->style, xl, xh, yl, yh, font, poly->string);
			break;
	}
	return(0);
}

void io_psdot(INTBIG x, INTBIG y)
{
	INTSML i;
	INTBIG psx, psy;
	static char *putdot[] =
	{
		"/Putdot {",				/* print dot at stack pos */
		"    newpath moveto 0 0 rlineto stroke} def",
	0};

	io_psxform(x, y, &psx, &psy);

	if (io_psdotput == 0)
	{
		io_psdotput++;
		for(i=0; putdot[i] != 0; i++) io_pswrite("%s\n", putdot[i]);
	}
	io_pswrite("%d %d Putdot\n", psx+io_psoffset, psy+io_psoffset);
}

/* draw a line */
void io_psline(INTBIG x1, INTBIG y1, INTBIG x2, INTBIG y2, INTSML pattern)
{
	INTBIG psx1, psy1, psx2, psy2, i;
	static char *drawline[] =
	{
		"/Drawline {",				/* draw line on stack */
		"    newpath moveto lineto stroke} def",
	0};

	io_psxform(x1, y1, &psx1, &psy1);
	io_psxform(x2, y2, &psx2, &psy2);

	if (io_psdrawlineput == 0)
	{
		io_psdrawlineput++;
		for(i=0; drawline[i] != 0; i++)
			io_pswrite("%s\n", drawline[i]);
	}
	i = io_psscale / 2;
	switch (pattern)
	{
		case 1: io_pswrite("[%d %d] 0 setdash ", i, i*3);    break;
		case 2: io_pswrite("[%d %d] 6 setdash ", i*6, i*3);  break;
		case 3: io_pswrite("[%d %d] 0 setdash ", i, i*7);    break;
	}
	io_pswrite("%d %d %d %d Drawline", psx1+io_psoffset, psy1+io_psoffset, psx2+io_psoffset, psy2+io_psoffset);
	if (pattern != 0) io_pswrite(" [] 0 setdash");
	io_pswrite("\n");
}

void io_psarc(INTBIG centerx,INTBIG centery, INTBIG x1, INTBIG y1, INTBIG x2, INTBIG y2)
{
	INTBIG radius, pscx, pscy, psx1, psy1, psx2, psy2;
	INTSML startangle, endangle;

	io_psxform(centerx, centery, &pscx, &pscy);
	io_psxform(x1, y1, &psx1, &psy1);
	io_psxform(x2, y2, &psx2, &psy2);

	radius = computedistance(pscx, pscy, psx1, psy1);
	startangle = (figureangle(pscx, pscy, psx2, psy2) + 5) / 10;
	endangle = (figureangle(pscx, pscy, psx1, psy1) + 5) / 10;
	io_pswrite("newpath %d %d %d %d %d arc stroke\n", pscx+io_psoffset,
		pscy+io_psoffset, radius, startangle, endangle);
}

void io_pscircle(INTBIG atx, INTBIG aty, INTBIG ex, INTBIG ey)
{
	INTBIG radius, pscx, pscy, psex, psey;

	io_psxform(atx, aty, &pscx, &pscy);
	io_psxform(ex, ey, &psex, &psey);

	radius = computedistance(pscx, pscy, psex, psey);
	io_pswrite("newpath %d %d %d 0 360 arc stroke\n", pscx+io_psoffset,
		pscy+io_psoffset, radius);
}

void io_psdisc(INTBIG atx, INTBIG aty, INTBIG ex, INTBIG ey, GRAPHICS *desc)
{
	INTBIG radius, pscx, pscy, psex, psey;

	io_psxform(atx, aty, &pscx, &pscy);
	io_psxform(ex, ey, &psex, &psey);

	radius = computedistance(pscx, pscy, psex, psey);
	io_pswrite("newpath %d %d %d 0 360 arc fill\n", pscx+io_psoffset, pscy+io_psoffset, radius);
}

void io_pspolygon(INTBIG *x, INTBIG *y, INTSML count, GRAPHICS *desc)
{
	INTSML i;
	INTBIG lx, hx, ly, hy, psx, psy;
	static char *polygon[] =
	{
		"/Polygon {",			/* put array into path */
		"    aload",
		"    length 2 idiv /len exch def",
		"    newpath",
		"    moveto",
		"    len 1 sub {lineto} repeat",
		"    closepath",
		"} def",
	0};
	static char *filledpolygon[] =
	{
		"/BuildCharDict 10 dict def",	/* ref Making a User Defined (PostScript Cookbook) */

		"/StippleFont1 7 dict def",
		"StippleFont1 begin",
		"    /FontType 3 def",
		"    /FontMatrix [1 0 0 1 0 0] def",
		"    /FontBBox [0 0 1 1] def",
		"    /Encoding 256 array def",
		"    0 1 255 {Encoding exch /.notdef put} for",
		"    /CharacterDefs 40 dict def",
		"    CharacterDefs /.notdef {} put",
		"    /BuildChar",
		"        { BuildCharDict begin",
		"            /char exch def",
		"            /fontdict exch def",
		"            /charname fontdict /Encoding get",
		"            char get def",
		"            /charproc fontdict /CharacterDefs get",
		"            charname get def",
		"            1 0 0 0 1 1 setcachedevice",
		"            gsave charproc grestore",
		"        end",
		"    } def",
		"end",

		"/StippleFont StippleFont1 definefont pop",

		"/StippleCharYSize 128 def",
		"/StippleCharXSize StippleCharYSize def",

		"/Filledpolygon {",
		"    gsave",
		"    /StippleFont findfont StippleCharYSize scalefont setfont",
		"    /LowY exch def /LowX exch def",
		"    /HighY exch LowY add def /HighX exch LowX add def",
		"    Polygon clip",
		"    /Char exch def",
		"    /LowY LowY StippleCharYSize div truncate StippleCharYSize mul def",
		"    /LowX LowX StippleCharXSize div truncate StippleCharXSize mul def",
		"    /HighY HighY StippleCharYSize div 1 add truncate StippleCharYSize mul def",
		"    /HighX HighX StippleCharXSize div 1 add truncate StippleCharXSize mul def",
		"    LowY StippleCharYSize HighY ",
		"    { LowX exch moveto ",
		"        LowX StippleCharXSize HighX ",
		"        { Char show pop ",
		"        } for ",
		"    } for",
		"    grestore",
		"} def",
	0};

	if (count == 0) return;

	if (io_pspolygonput == 0)
	{
		io_pspolygonput++;
		for(i=0; polygon[i] != 0; i++) io_pswrite("%s\n", polygon[i]);
	}
	for(i=0; i<8; i++)
		if ((desc->raster[i]&0xFFFF) != 0xFFFF) break;

	/* use solid black if solid pattern, high-resolution mode, or no pattern */
	if (i >= 8 || io_psscale > 4 || desc->style[BWRASTER] != PATTERNED)
	{
		/* solid fill: simply blacken the area */
		io_pswrite("[");
		for(i=0; i<count; i++)
		{
			if (i != 0) io_pswrite(" ");
			io_psxform(x[i], y[i], &psx, &psy);
			io_pswrite("%d %d", psx+io_psoffset, psy+io_psoffset);
		}
		io_pswrite("] Polygon fill\n");
		return;
	}

	/* patterned fill: the hard one */
	if (io_psfilledpolygonput == 0)
	{
		io_psfilledpolygonput++;
		for(i=0; filledpolygon[i] != 0; i++)
			io_pswrite("%s\n", filledpolygon[i]);
	}
	/* CS900228 Generate filled polygons by defining a stipple font,
				and then tiling the polygon to fill with 128x128 pixel
				characters, clipping to the polygon edge.
	*/
	io_pswrite("(%c) [", io_pspattern(desc));
	io_psxform(x[0], y[0], &psx, &psy);
	lx = hx = psx;
	ly = hy = psy;
	for(i=0; i<count; i++)
	{
		if (i != 0) io_pswrite(" ");
		io_psxform(x[i], y[i], &psx, &psy);
		io_pswrite("%d %d", psx+io_psoffset, psy+io_psoffset);
		if (psx < lx) lx = psx;   if (psx > hx) hx = psx;
		if (psy < ly) ly = psy;   if (psy > hy) hy = psy;
	}
	io_pswrite("] %d %d %d %d Filledpolygon\n", hx-lx+1, hy-ly+1, lx+io_psoffset, ly+io_psoffset);
}

INTSML io_pspattern(GRAPHICS *col)
{
	INTSML i;
	INTSML j, k, bl, bh, bld, bhd;
	GRAPHICS **newgraphicsseen;

	/* see if this graphics has been seen already */
	for(i=0; i<io_psgraphicsseencount; i++)
		if (io_psgraphicsseen[i] == col) return(i+'A');

	/* add to list */
	if (io_psgraphicsseencount >= io_psgraphicsseenlimit)
	{
		newgraphicsseen = (GRAPHICS **)emalloc((io_psgraphicsseenlimit + 50) *
			(sizeof (GRAPHICS *)), io_aid->cluster);
		if (newgraphicsseen == 0) return(0);
		for(i=0; i<io_psgraphicsseencount; i++)
			newgraphicsseen[i] = io_psgraphicsseen[i];
		if (io_psgraphicsseenlimit != 0) efree((char *)io_psgraphicsseen);
		io_psgraphicsseen = newgraphicsseen;
		io_psgraphicsseenlimit += 50;
	}
	io_psgraphicsseen[io_psgraphicsseencount++] = col;

	/* CS900228 Generate filled polygons by defining a stipple font,
				and then tiling the polygon to fill with 128x128 pixel
				characters, clipping to the polygon edge.

				Take ELECTRIC's 16x8 bit images, double each bit,
				and then output 4 times to get 128 bit wide image.
				Double vertically by outputting each row twice.
				Note that full vertical size need not be generated,
				as PostScript will just reuse the lines until the 128
				size is reached.

				ref Making a User Defined Font:  PostScript Cookbook
	*/
	io_pswrite("StippleFont1 begin\n");
	io_pswrite("    Encoding (%c) 0 get /Stipple%c put\n", i+'A', i+'A');
	io_pswrite("    CharacterDefs /Stipple%c  { 128 128 true [128 0 0 -128 0 128]\n", i+'A');
	io_pswrite("        { <\n");
	for(i=0; i<8; i++)
	{
		bl = col->raster[i] & 0x00FF;
		bh = (col->raster[i] & 0xFF00) >> 8;
		bld = bhd = 0;
		for (k=0; k<8; ++k)
		{
			bld = (bld << 1);
			bld |= (bl & 0x1);
			bld = (bld << 1);
			bld |= (bl & 0x1);
			bl = (bl >> 1);
			bhd = (bhd << 1);
			bhd |= (bh & 0x1);
			bhd = (bhd << 1);
			bhd |= (bh & 0x1);
			bh = (bh >> 1);
		}
		for (k=0; k<2; ++k)
		{
			io_pswrite("             ");
			for(j=0; j<4; j++)
				io_pswrite("%04x %04x ", bhd&0xFFFF, bld&0xFFFF);
			io_pswrite("\n");
		}
	}
	io_pswrite("                 >\n");
	io_pswrite("            } imagemask\n");
	io_pswrite("        } put\n");
	io_pswrite("end\n");
	return(io_psgraphicsseencount+'A'-1);
}

void io_pstext(INTSML type, INTBIG lx, INTBIG ux, INTBIG ly, INTBIG uy, INTSML fnt,
	char *text)
{
	INTSML i;
	INTBIG pslx, pshx, psly, pshy;
	static char *stringheader[] =
	{
		/*
		 * CS901126  Added code to do super and subscripts:
		 *
		 * example:
		 *	"NORMAL\dSUB\}   NORMAL\uSUP\}"
		 *
		 * will subscript "SUB" and superscript "SUP", so "\d"  starts a
		 * subscript, "\u" starts a superscript, "\}" returns to
		 * normal.  Sub-subscripts, and super-superscripts are not
		 * supported.  To print a "\", use "\\".
		 *
		 * changes:
		 *
		 * all calls to stringwidth were changed to calls to StringLength,
		 *    which returns the same info (assumes non-rotated text), but
		 *    takes sub- and super-scripts into account.
		 * all calls to show were changes to calls to StringShow, which
		 *    handles sub- and super-scripts.
		 * note that TSize is set to the text height, and is passed to
		 *    StringLength and StringShow.
		 */
		"/ComStart 92 def",								/* "\", enter command mode */
		"/ComSub  100 def",								/* "d", start subscript */
		"/ComSup  117 def",								/* "u", start superscript */
		"/ComNorm 125 def",								/* "}", return to normal */
		"/SSSize .70 def",								/* sub- and super-script size */
		"/SubDy  -.20 def",								/* Dy for sub-script */
		"/SupDy   .40 def",								/* Dy for super-script*/

		"/StringShow {",								/* str size StringShow */
		"    /ComMode 0 def",							/* command mode flag */
		"    /TSize exch def",							/* text size */
		"    /TString exch def",						/* string to draw */
		"    /NormY currentpoint exch pop def",			/* save Y coord of string */
		"    TSize scaleFont",
		"    TString {",								/* scan string char by char */
		"        /CharCode exch def",					/* save char */
		"        ComMode 1 eq {",
		"            /ComMode 0 def",					/* command mode */
		"            CharCode ComSub eq {",				/* start subscript */
		"                TSize SSSize mul scaleFont",
		"                currentpoint pop NormY TSize SubDy mul add moveto",
		"            } if",
		"            CharCode ComSup eq {",				/* start superscript */
		"                TSize SSSize mul scaleFont",
		"                currentpoint pop NormY TSize SupDy mul add moveto",
		"            } if",
		"            CharCode ComNorm eq {",			/* end sub- or super-script */
		"                TSize scaleFont",
		"                currentpoint pop NormY moveto",
		"            } if",
		"            CharCode ComStart eq {",			/* print a "\" */
		"                ( ) dup 0 CharCode put show",
		"            } if",
		"        }",
		"        {",
		"            CharCode ComStart eq {",
		"                /ComMode 1 def",				/* enter command mode */
		"            }",
		"            {",
		"                ( ) dup 0 CharCode put show",	/* print char */
		"            } ifelse",
		"        } ifelse",
		"    } forall ",
		"} def",

		"/StringLength {",								/* str size StringLength */
		"    /ComMode 0 def",							/* command mode flag */
		"    /StrLen 0 def",							/* total string length */
		"    /TSize exch def",							/* text size */
		"    /TString exch def",						/* string to draw */
		"    TSize scaleFont",
		"    TString {",								/* scan string char by char */
		"        /CharCode exch def",					/* save char */
		"        ComMode 1 eq {",
		"            /ComMode 0 def",					/* command mode */
		"            CharCode ComSub eq {",				/* start subscript */
		"                TSize SSSize mul scaleFont",
		"            } if",
		"            CharCode ComSup eq {",				/* start superscript */
		"                TSize SSSize mul scaleFont",
		"            } if",
		"            CharCode ComNorm eq {",			/* end sub- or super-script */
		"                TSize scaleFont",
		"            } if",
		"            CharCode ComStart eq {",			/* add "\" to length */
		"                ( ) dup 0 CharCode put stringwidth pop StrLen add",
		"                /StrLen exch def",
		"            } if",
		"        }",
		"        {",
		"            CharCode ComStart eq {",
		"                /ComMode 1 def",				/* enter command mode */
		"            }",
		"            {",								/* add char to length */
		"                ( ) dup 0 CharCode put stringwidth pop StrLen add",
		"                /StrLen exch def",
		"            } ifelse",
		"        } ifelse",
		"    } forall ",
		"    StrLen 0",									/* return info like stringwidth */
		"} def",

		"/Centerstring {",								/* x y str sca */
		"    dup /TSize exch def",						/* CS901126 save size */
		"    dup scaleFont exch dup TSize StringLength", /* x y sca str xw yw */
		"    pop 3 -1 roll .8 mul",						/* x y str xw sca*.8 */
		"    exch 5 -1 roll exch 2 div sub",			/* y str sca*.8 x-xw/2 */
		"    exch 4 -1 roll exch 2 div sub",			/* str x-xw/2 y-sca*.8/2 */
		"    moveto TSize StringShow",
		"} def",

		"/Topstring {",									/* x y str sca */
		"    dup /TSize exch def",						/* CS901126 save size */
		"    dup scaleFont exch dup TSize StringLength", /* x y sca str xw yw */
		"    pop 3 -1 roll .8 mul",						/* x y str xw sca*.8 */
		"    exch 5 -1 roll exch 2 div sub",			/* y str sca*.8 x-xw/2 */
		"    exch 4 -1 roll exch sub",					/* str x-xw/2 y-sca*.8 */
		"    moveto TSize StringShow",
		"} def",

		"/Botstring {",									/* x y str sca */
		"    dup /TSize exch def",						/* CS901126 save size */
		"    scaleFont dup TSize StringLength pop",		/* x y str xw */
		"    4 -1 roll exch 2 div sub",					/* y str x-xw/2 */
		"    3 -1 roll moveto TSize StringShow",
		"} def",

		"/Leftstring {",								/* x y str sca */
		"    dup /TSize exch def",						/* CS901126 save size */
		"    dup scaleFont .4 mul",						/* x y str sca*.4 */
		"    3 -1 roll exch sub",						/* x str y-sca*.4 */
		"    3 -1 roll exch",							/* str x y-sca*.4 */
		"    moveto TSize StringShow",
		"} def",

		"/Rightstring {",								/* x y str sca */
		"    dup /TSize exch def",						/* CS901126 save size */
		"    dup scaleFont exch dup TSize StringLength", /* x y sca str xw yw */
		"    pop 3 -1 roll .4 mul",						/* x y str xw sca*.4 */
		"    exch 5 -1 roll exch sub",					/* y str sca*.4 x-xw */
		"    exch 4 -1 roll exch sub",					/* str x-xw y-sca*.4 */
		"    moveto TSize StringShow",
		"} def",

		"/Topleftstring {",								/* x y str sca */
		"    dup /TSize exch def",						/* CS901126 save size */
		"    dup scaleFont .8 mul",						/* x y str sca*.8 */
		"    3 -1 roll exch sub",						/* x str y-sca*.8 */
		"    3 -1 roll exch",							/* str x y-sca*.8 */
		"    moveto TSize StringShow",
		"} def",

		"/Toprightstring {",							/* x y str sca */
		"    dup /TSize exch def",						/* CS901126 save size */
		"    dup scaleFont exch dup TSize StringLength", /* x y sca str xw yw */
		"    pop 3 -1 roll .8 mul",						/* x y str xw sca*.8 */
		"    exch 5 -1 roll exch sub",					/* y str sca*.8 x-xw */
		"    exch 4 -1 roll exch sub",					/* str x-xw y-sca*.8 */
		"    moveto TSize StringShow",
		"} def",

		"/Botleftstring {",								/* x y str sca */
		"    dup /TSize exch def",						/* CS901126 save size */
		"    scaleFont 3 1 roll moveto TSize StringShow",
		"} def",

		"/Botrightstring {",							/* x y str sca */
		"    dup /TSize exch def",						/* CS901126 save size */
		"    scaleFont dup TSize StringLength",
		"    pop 4 -1 roll exch",
		"    sub 3 -1 roll",
		"    moveto TSize StringShow",
		"} def",

		"/Min {",										/* leave minimum of top two */
		"    dup 3 -1 roll dup",
		"    3 1 roll gt",
		"    {exch} if pop",
		"} def",

		"/Boxstring {",									/* x y mx my str sca */
		"    dup /TSize exch def",						/* CS 901126 save size */
		"    dup scaleFont",							/* x y mx my str sca */
		"    exch dup TSize StringLength pop",			/* x y mx my sca str xw */
		"    3 -1 roll dup",							/* x y mx my str xw sca sca */
		"    6 -1 roll mul",							/* x y my str xw sca sca*mx */
		"    3 -1 roll div",							/* x y my str sca sca*mx/xw */
		"    4 -1 roll",								/* x y str sca sca*mx/xw my */
		"    Min Min",									/* x y str minsca */
		"    Centerstring",
		"} def",
	0};

	io_psxform(lx, ly, &pslx, &psly);
	io_psxform(ux, uy, &pshx, &pshy);

	if (io_psstringput == 0)
	{
		io_psstringput++;
		for(i=0; stringheader[i] != 0; i++)
			io_pswrite("%s\n", stringheader[i]);
	}
	switch (type)
	{
		case TEXTCENT:
			io_pswrite("%d %d ", (pslx+pshx)/2+io_psoffset, (psly+pshy)/2+io_psoffset);
			io_pswritestring(text);
			io_pswrite(" %d Centerstring\n", io_psconvert[fnt]*io_psscale);
			break;
		case TEXTTOP:
			io_pswrite("%d %d ", (pslx+pshx)/2+io_psoffset, pshy+io_psoffset);
			io_pswritestring(text);
			io_pswrite(" %d Topstring\n", io_psconvert[fnt]*io_psscale);
			break;
		case TEXTBOT:
			io_pswrite("%d %d ", (pslx+pshx)/2+io_psoffset, psly+io_psoffset);
			io_pswritestring(text);
			io_pswrite(" %d Botstring\n", io_psconvert[fnt]*io_psscale);
			break;
		case TEXTLEFT:
			io_pswrite("%d %d ", pslx+io_psoffset, (psly+pshy)/2+io_psoffset);
			io_pswritestring(text);
			io_pswrite(" %d Leftstring\n", io_psconvert[fnt]*io_psscale);
			break;
		case TEXTRIGHT:
			io_pswrite("%d %d ", pshx+io_psoffset, (psly+pshy)/2+io_psoffset);
			io_pswritestring(text);
			io_pswrite(" %d Rightstring\n", io_psconvert[fnt]*io_psscale);
			break;
		case TEXTTOPLEFT:
			io_pswrite("%d %d ", pslx+io_psoffset, pshy+io_psoffset);
			io_pswritestring(text);
			io_pswrite(" %d Topleftstring\n", io_psconvert[fnt]*io_psscale);
			break;
		case TEXTTOPRIGHT:
			io_pswrite("%d %d ", pshx+io_psoffset, pshy+io_psoffset);
			io_pswritestring(text);
			io_pswrite(" %d Toprightstring\n", io_psconvert[fnt]*io_psscale);
			break;
		case TEXTBOTLEFT:
			io_pswrite("%d %d ", pslx+io_psoffset, psly+io_psoffset);
			io_pswritestring(text);
			io_pswrite(" %d Botleftstring\n", io_psconvert[fnt]*io_psscale);
			break;
		case TEXTBOTRIGHT:
			io_pswrite("%d %d ", pshx+io_psoffset, psly+io_psoffset);
			io_pswritestring(text);
			io_pswrite(" %d Botrightstring\n", io_psconvert[fnt]*io_psscale);
			break;
		case TEXTBOX:
			io_pswrite("%d %d %d %d ", (pslx+pshx)/2+io_psoffset,
				(psly+pshy)/2+io_psoffset, pshx-pslx, pshy-psly);
			io_pswritestring(text);
			io_pswrite(" %d Boxstring\n", io_psconvert[fnt]*io_psscale);
			break;
	}
}

/*
 * Routine to convert the coordinates (x,y) for display.  The coordinates for
 * printing are placed back into (x,y) and the PostScript coordinates are placed
 * in (psx,psy).
 */
void io_psxform(INTBIG x, INTBIG y, INTBIG *psx, INTBIG *psy)
{
	*psx = ((x * io_psmatrix[0][0] + y * io_psmatrix[1][0]) >> 16) + io_psmatrix[2][0];
	*psy = ((x * io_psmatrix[0][1] + y * io_psmatrix[1][1]) >> 16) + io_psmatrix[2][1];
}

void io_pswritestring(char *str)
{
	io_pswrite("(");
	for( ; *str != 0; str++)
	{
		if (*str == '(' || *str == ')' || *str == '\\') io_pswrite("\\");
		io_pswrite("%c", *str);
	}
	io_pswrite(")");
}

void io_pswrite(char *s, ...)
{
	char theline[100];
	va_list ap;

	var_start(ap, s);
	(void)vsprintf(theline, s, ap);
	va_end(ap);

	xprintf(io_psout, "%s", theline);
}

#endif  /* IOPOSTSCRIPT - at top */
