/*
 * Electric(tm) VLSI Design System
 *
 * File: iotext.c
 * Input/output aid: textual format
 * Written by: Steven M. Rubin, Electric Editor Incorporated
 *
 * Copyright (c) 1998 Electric Editor Incorporated.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Electric Editor Incorporated
 * 23470 Sunset Drive, Suite 108
 * Los Gatos, California 95033
 * support@electriceditor.com
 */

#include "global.h"
#include "database.h"
#include "eio.h"
#include "tecgen.h"
#include "tecschem.h"
#include "edialogs.h"

/****************************** OUTPUT ******************************/

INTSML io_facetnumber;
INTBIG io_nodeinsterror, io_portarcinsterror, io_portexpinsterror,
	  io_portprotoerror, io_arcinsterror, io_geomerror, io_rtnodeerror, io_libraryerror;

/* prototypes for local routines */
void io_textrecurse(NODEPROTO*);
INTSML io_countvars(INTSML, VARIABLE*);
void io_writevars(INTSML, VARIABLE*, NODEPROTO*);
char *io_makestring(VARIABLE*, NODEPROTO*);
INTSML io_makestringvar(INTBIG, INTBIG, NODEPROTO*);
INTSML io_addstring(char*);
void io_printname(char*, FILE*);
void io_newlib(void);
INTSML io_getkeyword(void);
INTSML io_grabbuffers(INTSML);
PORTPROTO *io_getport(char*, NODEPROTO*);
void io_null(void);
void io_newlib(void);
void io_versn(void);
void io_libkno(void);
void io_libain(void);
void io_libaib(void);
void io_libbit(void);
void io_libusb(void);
void io_libte(void);
void io_libten(void);
void io_lambda(void);
void io_getvar(void);
INTBIG io_decode(char*, INTBIG);
void io_libcc(void);
void io_libms(void);
void io_libvie(void);
void io_newcel(void);
void io_celnam(void);
void io_celver(void);
void io_celcre(void);
void io_celrev(void);
void io_cellx(void);
void io_celhx(void);
void io_celly(void);
void io_celhy(void);
void io_tech(void);
void io_celaad(void);
void io_celbit(void);
void io_celusb(void);
void io_celnet(void);
void io_celnoc(void);
void io_celarc(void);
void io_celptc(void);
void io_celdon(void);
void io_newno(void);
void io_nodtyp(void);
void io_nodlx(void);
void io_nodhx(void);
void io_nodly(void);
void io_nodhy(void);
void io_nodnam(void);
void io_noddes(void);
void io_nodrot(void);
void io_nodtra(void);
void io_nodkse(void);
void io_nodpoc(void);
void io_nodbit(void);
void io_nodusb(void);
void io_newpor(void);
void io_porarc(void);
void io_porexp(void);
void io_newar(void);
void io_arctyp(void);
void io_arcnam(void);
void io_arcwid(void);
void io_arclen(void);
void io_arcsig(void);
void io_newend(void);
void io_endnod(void);
void io_endpt(void);
void io_endxp(void);
void io_endyp(void);
void io_arckse(void);
void io_arcbit(void);
void io_arcusb(void);
void io_arcnet(void);
void io_newpt(void);
void io_ptnam(void);
void io_ptdes(void);
void io_ptsno(void);
void io_ptspt(void);
void io_ptkse(void);
void io_ptbit(void);
void io_ptusb(void);
void io_ptnet(void);
TECHNOLOGY *io_gettechnology(char*);

INTSML io_writetextlibrary(LIBRARY *lib)
{
	REGISTER char *name;
	char file[256], *truename;
	REGISTER INTBIG i, j, noc, arcinst, poc, facet;
	REGISTER NODEPROTO *np;
	REGISTER PORTPROTO *pp;
	REGISTER NODEINST *ni;
	REGISTER PORTARCINST *pi;
	REGISTER PORTEXPINST *pe;
	REGISTER ARCINST *ai;
	REGISTER TECHNOLOGY *tech;
	REGISTER VIEW *v;

	(void)strcpy(file, lib->libfile);
	name = &file[strlen(file)-5];
	if (strcmp(name, ".elib") == 0) *name = 0;
	(void)strcat(file, ".txt");
	name = truepath(file);
	io_fileout = xcreate(name, FILETYPETLIB, "Textual library file", &truename);
	if (io_fileout == NULL)
	{
		if (truename != 0) ttyputerr("Cannot write %s", truename);
		return(1);
	}

	/* clear error counters */
	io_nodeinsterror = io_portarcinsterror = io_portexpinsterror = 0;
	io_portprotoerror = io_arcinsterror = io_geomerror = io_libraryerror = 0;
	io_rtnodeerror = 0;

	/* determine proper library order */
	for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		np->temp1 = -1;
	io_facetnumber = 0;
	for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		if (np->firstinst == NONODEINST) io_textrecurse(np);

	/* write header information */
	xprintf(io_fileout, "****library: \"%s\"\n", lib->libname);
	xprintf(io_fileout, "version: %s\n", el_version);
	xprintf(io_fileout, "aids: %d\n", el_maxaid);
	for(i=0; i<el_maxaid; i++)
	{
		xprintf(io_fileout, "aidname: %s\n", el_aids[i].aidname);
		if (io_countvars(el_aids[i].numvar, el_aids[i].firstvar) != 0)
			io_writevars(el_aids[i].numvar, el_aids[i].firstvar, NONODEPROTO);
	}
	xprintf(io_fileout, "userbits: %d\n", lib->userbits);
	xprintf(io_fileout, "techcount: %d\n", el_maxtech);
	for(tech = el_technologies; tech != NOTECHNOLOGY; tech = tech->nexttechnology)
	{
		xprintf(io_fileout, "techname: %s lambda: %d\n", tech->techname, lib->lambda[tech->index]);
		io_writevars(tech->numvar, tech->firstvar, NONODEPROTO);
	}
	for(v = el_views; v != NOVIEW; v = v->nextview)
		xprintf(io_fileout, "view: %s{%s}\n", v->viewname, v->sviewname);
	xprintf(io_fileout, "cellcount: %d\n", io_facetnumber);
	if (lib->curnodeproto != NONODEPROTO)
		xprintf(io_fileout, "maincell: %d\n", lib->curnodeproto->temp1);

	/* write variables on the library */
	io_writevars(lib->numvar, lib->firstvar, NONODEPROTO);

	/* write the rest of the database */
	for(facet = 0; facet < io_facetnumber; facet++)
	{
		/* write the nodeproto name */
		for(np=lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
			if (np->temp1 == facet) break;
		xprintf(io_fileout, "***cell: %d\n", np->temp1);
		xprintf(io_fileout, "name: %s", np->cell->cellname);
		if (*np->cellview->sviewname != 0)
			xprintf(io_fileout, "{%s}", np->cellview->sviewname);
		xprintf(io_fileout, "\n");
		xprintf(io_fileout, "version: %d\n", np->version);
		xprintf(io_fileout, "creationdate: %d\n", np->creationdate);
		xprintf(io_fileout, "revisiondate: %d\n", np->revisiondate);

		/* write the nodeproto bounding box */
		xprintf(io_fileout, "lowx: %d highx: %d lowy: %d highy: %d\n",
			np->lowx, np->highx, np->lowy, np->highy);

		/* write aid information */
		xprintf(io_fileout, "aadirty: %d\n", np->adirty);
		xprintf(io_fileout, "userbits: %d\n", np->userbits);

		/* count and number the nodes, arcs, and ports */
		noc = arcinst = poc = 0;
		for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
			ni->temp1 = noc++;
		for(ai = np->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
			ai->temp1 = arcinst++;
		for(pp=np->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
			pp->temp1 = poc++;
		xprintf(io_fileout, "nodes: %d arcs: %d porttypes: %d\n", noc, arcinst, poc);

		/* write variables on the facet */
		io_writevars(np->numvar, np->firstvar, np);

		/* write the nodes in this facet */
		for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
		{
			xprintf(io_fileout, "**node: %d\n", ni->temp1);
			if (ni->proto->index == 0)
				xprintf(io_fileout, "type: [%d]\n", ni->proto->temp1); else
					xprintf(io_fileout, "type: %s:%s\n",
						ni->proto->tech->techname, ni->proto->primname);
			xprintf(io_fileout, "lowx: %d highx: %d lowy: %d highy: %d\n", ni->lowx,
				ni->highx, ni->lowy, ni->highy);
			xprintf(io_fileout, "rotation: %d transpose: %d\n", ni->rotation, ni->transpose);
			if (ni->proto->index == 0)
				xprintf(io_fileout, "descript: %d\n", ni->textdescript);
			xprintf(io_fileout, "userbits: %d\n", ni->userbits);
			io_writevars(ni->numvar, ni->firstvar, np);

			pi = ni->firstportarcinst;   pe = ni->firstportexpinst;
			for(pp = ni->proto->firstportproto, i=0; pp != NOPORTPROTO; pp = pp->nextportproto, i++)
			{
				j = 0;
				while (pi != NOPORTARCINST && pi->proto == pp)
				{
					if (j == 0) xprintf(io_fileout, "*port: %s\n", pp->protoname);
					j++;
					xprintf(io_fileout, "arc: %d\n", pi->conarcinst->temp1);
					io_writevars(pi->numvar, pi->firstvar, np);
					pi = pi->nextportarcinst;
				}
				while (pe != NOPORTEXPINST && pe->proto == pp)
				{
					if (j == 0) xprintf(io_fileout, "*port: %s\n", pp->protoname);
					j++;
					xprintf(io_fileout, "exported: %d\n", pe->exportproto->temp1);
					io_writevars(pe->numvar, pe->firstvar, np);
					pe = pe->nextportexpinst;
				}
			}
		}

		/* write the portprotos in this facet */
		poc = 0;
		for(pp=np->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
		{
			xprintf(io_fileout, "**porttype: %d\n", poc++);
			xprintf(io_fileout, "subnode: %d\n", pp->subnodeinst->temp1);
			xprintf(io_fileout, "subport: %s\n", pp->subportproto->protoname);
			xprintf(io_fileout, "name: %s\n", pp->protoname);
			xprintf(io_fileout, "descript: %d\n", pp->textdescript);
			xprintf(io_fileout, "userbits: %d\n", pp->userbits);
			io_writevars(pp->numvar, pp->firstvar, np);
		}

		/* write the arcs in this facet */
		for(ai = np->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
		{
			xprintf(io_fileout, "**arc: %d\n", ai->temp1);
			xprintf(io_fileout, "type: %s:%s\n", ai->proto->tech->techname, ai->proto->protoname);
			xprintf(io_fileout, "width: %d length: %d\n", ai->width, ai->length);
			xprintf(io_fileout, "userbits: %d\n", ai->userbits);
			for(i=0; i<2; i++)
			{
				xprintf(io_fileout, "*end: %d\n", i);
				xprintf(io_fileout, "node: %d\n", ai->end[i].nodeinst->temp1);
				xprintf(io_fileout, "nodeport: %s\n", ai->end[i].portarcinst->proto->protoname);
				xprintf(io_fileout, "xpos: %d ypos: %d\n", ai->end[i].xpos, ai->end[i].ypos);
			}
			io_writevars(ai->numvar, ai->firstvar, np);
		}
		xprintf(io_fileout, "celldone: %s\n", np->cell->cellname);
	}

	/* print any variable-related error messages */
	if (io_nodeinsterror != 0)
		ttyputmsg("Warning: %d NODEINST pointers point outside facet: not saved", io_nodeinsterror);
	if (io_arcinsterror != 0)
		ttyputmsg("Warning: %d ARCINST pointers point outside facet: not saved", io_arcinsterror);
	if (io_portprotoerror != 0)
		ttyputmsg("Warning: %d PORTPROTO pointers point outside facet: not saved", io_portprotoerror);
	if (io_portarcinsterror != 0)
		ttyputmsg("Warning: %d PORTARCINST pointers could not be saved", io_portarcinsterror);
	if (io_portexpinsterror != 0)
		ttyputmsg("Warning: %d PORTEXPINST pointers could not be saved", io_portexpinsterror);
	if (io_geomerror != 0)
		ttyputmsg("Warning: %d GEOM pointers could not be saved", io_geomerror);
	if (io_rtnodeerror != 0)
		ttyputmsg("Warning: %d RTNODE pointers could not be saved", io_rtnodeerror);
	if (io_libraryerror != 0)
		ttyputmsg("Warning: LIBRARY pointers could not be saved", io_libraryerror);

	/* clean up and return */
	xclose(io_fileout);
	ttyputmsgf("%s written", name);
	lib->userbits &= ~LIBCHANGED;
	return(0);
}

/*
 * routine to help order the library for proper nonforward references
 * in the outout
 */
void io_textrecurse(NODEPROTO *np)
{
	REGISTER NODEINST *ni;

	for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		if (ni->proto->index != 0) continue;
		if (ni->proto->temp1 == -1) io_textrecurse(ni->proto);
	}

	/* add this facet to the list */
	np->temp1 = io_facetnumber++;
}

/*
 * routine to return the number of permanent variables on an object
 */
INTSML io_countvars(INTSML numvar, VARIABLE *firstvar)
{
	REGISTER INTSML i, j;

	i = 0;
	for(j=0; j<numvar; j++)
		if ((firstvar[j].type & VDONTSAVE) == 0) i++;
	return(i);
}

/*
 * routine to write the variables on an object.  The current facet is
 * "curnodeproto" such that any references to objects in a facet must be in
 * this facet.
 */
void io_writevars(INTSML numvar, VARIABLE *firstvar, NODEPROTO *curnodeproto)
{
	REGISTER char *pt;
	REGISTER INTSML i;
	REGISTER VARIABLE *var;

	i = io_countvars(numvar, firstvar);
	if (i == 0) return;

	xprintf(io_fileout, "variables: %d\n", i);
	for(i=0; i<numvar; i++)
	{
		var = &firstvar[i];
		if ((var->type & VDONTSAVE) != 0) continue;
		pt = io_makestring(var, curnodeproto);
		if (pt == 0) return;
		io_printname((char *)var->key, io_fileout);
		if ((var->type&(VLENGTH|VISARRAY)) != VISARRAY)
			xprintf(io_fileout, "[0%o,0%o]: ", var->type, var->textdescript); else
				xprintf(io_fileout, "(%d)[0%o,0%o]: ", getlength(var), var->type, var->textdescript);
		xprintf(io_fileout, "%s\n", pt);
	}
}

/*
 * routine to convert variable "var" to a string for printing in the text file.
 * returns zero on error
 */
char *io_makestring(VARIABLE *var, NODEPROTO *curnodeproto)
{
	REGISTER INTSML err, i, len;
	char line[50];

	if (var == NOVARIABLE) return(0);

	err = initinfstr();
	if ((var->type&VISARRAY) != 0)
	{
		len = getlength(var);
		for(i=0; i<len; i++)
		{
			if (i == 0) err += addtoinfstr('['); else
				err += addtoinfstr(',');

			if ((var->type&VTYPE) == VGENERAL)
			{
				if ((i&1) == 0)
				{
					(void)sprintf(line, "0%o", ((INTBIG *)var->addr)[i+1]);
					err += addstringtoinfstr(line);
				} else
				{
					err += io_makestringvar(((INTBIG *)var->addr)[i], ((INTBIG *)var->addr)[i-1],
						curnodeproto);
				}
			} else
			{
				switch ((var->type&VTYPE))
				{
					case VCHAR:
						err += io_makestringvar(var->type, ((INTBIG)((char *)var->addr)[i]),
							curnodeproto);
						break;

					case VDOUBLE:
						err += io_makestringvar(var->type, ((INTBIG)((double *)var->addr)[i]),
							curnodeproto);
						break;

					default:
						err += io_makestringvar(var->type, ((INTBIG *)var->addr)[i], curnodeproto);
						break;
				}
			}
		}
		err += addtoinfstr(']');
	} else err += io_makestringvar(var->type, var->addr, curnodeproto);
	if (err != 0) return(0);
	return(returninfstr());
}

/*
 * routine to make a string from the value in "addr" which has a type in
 * "type".  Returns nonzero if there is a memory allocation error.
 */
INTSML io_makestringvar(INTBIG type, INTBIG addr, NODEPROTO *curnodeproto)
{
	char line[100];
	REGISTER INTSML err;
	REGISTER INTBIG index;
	REGISTER NODEINST *ni;
	REGISTER NODEPROTO *np;
	REGISTER PORTARCINST *pi;
	REGISTER PORTEXPINST *pe;
	REGISTER PORTPROTO *pp;
	REGISTER ARCINST *ai;
	REGISTER ARCPROTO *ap;
	REGISTER GEOM *geom;
	REGISTER RTNODE *rtn;
	REGISTER LIBRARY *lib;
	REGISTER TECHNOLOGY *tech;
	REGISTER AIDENTRY *aid;

	if ((type&(VCODE1|VCODE2)) != 0) type = VSTRING;
	switch (type&VTYPE)
	{
		case VINTEGER:
		case VFRACT:
			(void)sprintf(line, "%d", addr);
			return(addstringtoinfstr(line));
		case VADDRESS:
			(void)sprintf(line, "0%o", addr);
			err = addstringtoinfstr(line);
			return(err);
		case VCHAR:
			return(addtoinfstr((char)addr));
		case VSTRING:
			err = addtoinfstr('"');
			err += io_addstring((char *)addr);
			err += addtoinfstr('"');
			return(err);
		case VFLOAT:
			(void)sprintf(line, "%f", castfloat(addr));
			return(addstringtoinfstr(line));
		case VDOUBLE:
			(void)sprintf(line, "%f", castfloat(addr));
			return(addstringtoinfstr(line));
		case VNODEINST:
			ni = (NODEINST *)addr;
			index = -1;
			if (ni != NONODEINST)
			{
				if (ni->parent == curnodeproto) index = ni->temp1; else
					io_nodeinsterror++;
			}
			(void)sprintf(line, "%d", index);
			return(addstringtoinfstr(line));
		case VNODEPROTO:
			np = (NODEPROTO *)addr;
			if (np == NONODEPROTO) return(addstringtoinfstr("-1"));
			if (np->index == 0)
			{
				(void)sprintf(line, "%d", np->temp1);
				return(addstringtoinfstr(line));
			} else
			{
				err = io_addstring(np->tech->techname);
				err += addtoinfstr(':');
				err += io_addstring(np->primname);
				return(err);
			}
		case VPORTARCINST:
			io_portarcinsterror++;
			pi = (PORTARCINST *)addr;
			if (pi == NOPORTARCINST) return(addstringtoinfstr("NOPORTARCINST"));
			(void)sprintf(line, "portarc%d", (INTBIG)pi);
			return(addstringtoinfstr(line));
		case VPORTEXPINST:
			io_portexpinsterror++;
			pe = (PORTEXPINST *)addr;
			if (pe == NOPORTEXPINST) return(addstringtoinfstr("NOPORTEXPINST"));
			(void)sprintf(line, "portexp%d", (INTBIG)pe);
			return(addstringtoinfstr(line));
		case VPORTPROTO:
			pp = (PORTPROTO *)addr;
			index = -1;
			if (pp != NOPORTPROTO)
			{
				if (pp->parent == curnodeproto) index = pp->temp1; else
					io_portprotoerror++;
			}
			(void)sprintf(line, "%d", index);
			return(addstringtoinfstr(line));
		case VARCINST:
			ai = (ARCINST *)addr;
			index = -1;
			if (ai != NOARCINST)
			{
				if (ai->parent == curnodeproto) index = ai->temp1; else
					io_arcinsterror++;
			}
			(void)sprintf(line, "%d", index);
			return(addstringtoinfstr(line));
		case VARCPROTO:
			ap = (ARCPROTO *)addr;
			if (ap == NOARCPROTO) return(addstringtoinfstr("NOARCPROTO"));
			err = io_addstring(ap->tech->techname);
			err += addtoinfstr(':');
			err += io_addstring(ap->protoname);
			return(err);
		case VGEOM:
			io_geomerror++;
			geom = (GEOM *)addr;
			if (geom == NOGEOM) return(addstringtoinfstr("NOGEOM"));
			(void)sprintf(line, "geom%d", (INTBIG)geom);
			return(addstringtoinfstr(line));
		case VLIBRARY:
			io_libraryerror++;
			lib = (LIBRARY *)addr;
			if (lib == NOLIBRARY) return(addstringtoinfstr("NOLIBRARY"));
			err = addtoinfstr('"');
			err += io_addstring(lib->libname);
			err += addtoinfstr('"');
			return(err);
		case VTECHNOLOGY:
			tech = (TECHNOLOGY *)addr;
			if (tech == NOTECHNOLOGY) return(addstringtoinfstr("NOTECHNOLOGY"));
			return(io_addstring(tech->techname));
		case VAID:
			aid = (AIDENTRY *)addr;
			if (aid == NOAID) return(addstringtoinfstr("NOAID"));
			return(io_addstring(aid->aidname));
		case VRTNODE:
			io_rtnodeerror++;
			rtn = (RTNODE *)addr;
			if (rtn == NORTNODE) return(addstringtoinfstr("NORTNODE"));
			(void)sprintf(line, "rtn%d", (INTBIG)rtn);
			return(addstringtoinfstr(line));
	}
	return(0);
}

/*
 * routine to add the string "str" to the infinite string and to quote the
 * special characters '[', ']', '"', and '^'.  The routine returns nonzero
 * if there is memory problem with the infinite string package.
 */
INTSML io_addstring(char *str)
{
	REGISTER INTSML err;

	err = 0;
	while (*str != 0)
	{
		if (*str == '[' || *str == ']' || *str == '"' || *str == '^')
			err += addtoinfstr('^');
		err += addtoinfstr(*str++);
	}
	return(err);
}

/*
 * routine to print the variable name in "name" on file "file".  The
 * conversion performed is to quote with a backslash any of the characters
 * '(', '[', or '^'.
 */
void io_printname(char *name, FILE *file)
{
	REGISTER char *pt;

	for(pt = name; *pt != 0; pt++)
	{
		if (*pt == '^' || *pt == '[' || *pt == '(') xputc('^', file);
		xputc(*pt, file);
	}
}

/****************************** INPUT ******************************/

#define	REPORTRATE  100		/* number of elements between graphical update */

/* state of input (value of "io_textlevel") */
#define	INLIB       1
#define	INFACET     2
#define	INPORTPROTO 3
#define	INNODEINST  4
#define	INPOR       5
#define	INARCINST   6
#define	INARCEND    7

/* state of variable reading (value of "io_varpos") */
#define	INVAID         1
#define	INVTECHNOLOGY  2
#define	INVLIBRARY     3
#define	INVNODEPROTO   4
#define	INVNODEINST    5
#define	INVPORTARCINST 6
#define	INVPORTEXPINST 7
#define	INVPORTPROTO   8
#define	INVARCINST     9

static INTBIG io_facetcount, io_textlevel, io_aacount, io_bitcount,
	  io_techcount, io_mainfacet, io_curarcend, io_nodeinstcount,
	  io_arcinstcount, io_varaddr, io_portprotocount, io_linenumber,
	  io_curportcount, io_varpos, io_filelength, io_fileposition;
static char *io_version;
static NODEINST *io_curnodeinst;
static ARCINST *io_curarcinst;
static PORTPROTO *io_curportproto;
static NODEPROTO *io_curnodeproto;
static LIBRARY *io_curlib;
static TECHNOLOGY *io_curtech;

/* input string buffer */
static INTSML io_maxinputline = 0;
static char *io_keyword;

struct keyroutine
{
	char *matchstring;
	void (*routine)(void);
};

/* possible keywords while reading libraries */
struct keyroutine io_libraryroutines[] =
{
	{"****library",io_newlib},
	{"bits",       io_libbit},
	{"lambda",     io_lambda},
	{"version",    io_versn},
	{"aids",       io_libkno},
	{"aidname",    io_libain},
	{"aidbits",    io_libaib},
	{"userbits",   io_libusb},
	{"techcount",  io_libte},
	{"techname",   io_libten},
	{"cellcount",  io_libcc},
	{"maincell",   io_libms},
	{"view",       io_libvie},
	{"***cell",    io_newcel},
	{"variables",  io_getvar},
   {NULL, NULL}  /* 0 */
};

/* possible keywords while reading facets */
struct keyroutine io_facetroutines[] =
{
	{"bits",         io_celbit},
	{"userbits",     io_celusb},
	{"netnum",       io_celnet},
	{"name",         io_celnam},
	{"version",      io_celver},
	{"creationdate", io_celcre},
	{"revisiondate", io_celrev},
	{"lowx",         io_cellx},
	{"highx",        io_celhx},
	{"lowy",         io_celly},
	{"highy",        io_celhy},
	{"aadirty",      io_celaad},
	{"nodes",        io_celnoc},
	{"arcs",         io_celarc},
	{"porttypes",    io_celptc},
	{"celldone",     io_celdon},
	{"technology",   io_tech},
	{"**node",       io_newno},
	{"**arc",        io_newar},
	{"***cell",      io_newcel},
	{"variables",    io_getvar},
   {NULL, NULL}  /* 0 */
};

/* possible keywords while reading ports */
struct keyroutine io_portroutines[] =
{
	{"bits",       io_ptbit},
	{"userbits",   io_ptusb},
	{"netnum",     io_ptnet},
	{"subnode",    io_ptsno},
	{"subport",    io_ptspt},
	{"name",       io_ptnam},
	{"descript",   io_ptdes},
	{"aseen",      io_ptkse},
	{"**porttype", io_newpt},
	{"**arc",      io_newar},
	{"**node",     io_newno},
	{"celldone",   io_celdon},
	{"***cell",    io_newcel},
	{"variables",  io_getvar},
	{"arcbits",    io_null},	/* used no more: ignored for compatibility */
   {NULL, NULL}  /* 0 */
};

/* possible keywords while reading nodes */
struct keyroutine io_noderoutines[] =
{
	{"bits",       io_nodbit},
	{"userbits",   io_nodusb},
	{"type",       io_nodtyp},
	{"lowx",       io_nodlx},
	{"highx",      io_nodhx},
	{"lowy",       io_nodly},
	{"highy",      io_nodhy},
	{"rotation",   io_nodrot},
	{"transpose",  io_nodtra},
	{"aseen",      io_nodkse},
	{"name",       io_nodnam},
	{"descript",   io_noddes},
	{"*port",      io_newpor},
	{"**node",     io_newno},
	{"**porttype", io_newpt},
	{"**arc",      io_newar},
	{"celldone",   io_celdon},
	{"variables",  io_getvar},
	{"***cell",    io_newcel},
	{"ports",      io_nodpoc},
	{"exportcount",io_null},	/* used no more: ignored for compatibility */
   {NULL, NULL}  /* 0 */
};

/* possible keywords while reading portinsts */
struct keyroutine io_portinstroutines[] =
{
	{"arc",        io_porarc},
	{"exported",   io_porexp},
	{"*port",      io_newpor},
	{"**node",     io_newno},
	{"**porttype", io_newpt},
	{"**arc",      io_newar},
	{"variables",  io_getvar},
	{"celldone",   io_celdon},
	{"***cell",    io_newcel},
   {NULL, NULL}  /* 0 */
};

/* possible keywords while reading arcs */
struct keyroutine io_arcroutines[] =
{
	{"bits",       io_arcbit},
	{"userbits",   io_arcusb},
	{"netnum",     io_arcnet},
	{"type",       io_arctyp},
	{"width",      io_arcwid},
	{"length",     io_arclen},
	{"signals",    io_arcsig},
	{"aseen",      io_arckse},
	{"name",       io_arcnam},
	{"*end",       io_newend},
	{"**arc",      io_newar},
	{"**node",     io_newno},
	{"variables",  io_getvar},
	{"celldone",   io_celdon},
	{"***cell",    io_newcel},
   {NULL, NULL}  /* 0 */
};

/* possible keywords while reading arc ends */
struct keyroutine io_arcendroutines[] =
{
	{"node",       io_endnod},
	{"nodeport",   io_endpt},
	{"xpos",       io_endxp},
	{"ypos",       io_endyp},
	{"*end",       io_newend},
	{"**arc",      io_newar},
	{"**node",     io_newno},
	{"variables",  io_getvar},
	{"celldone",   io_celdon},
	{"***cell",    io_newcel},
   {NULL, NULL}  /* 0 */
};

INTSML io_readtextlibrary(LIBRARY *lib)
{
	struct keyroutine *table;
	REGISTER char *pp;
	REGISTER INTBIG i, j, version, total, clustersize, largestcluster, assoc,
		reportcount, lambda;
	INTBIG num, den;
	REGISTER INTSML oldunit;
	REGISTER NODEPROTO *np, **list, *onp;
	REGISTER TECHNOLOGY *tech;
	REGISTER CELL *c;
	extern DIALOG usr_progressdialog;
	char *filename;

	/* initialize the input string buffers */
	if (io_maxinputline == 0)
		if (io_grabbuffers(1000) != 0) return(1);

	io_filein = xopen(truepath(lib->libfile), FILETYPETLIB, el_libdir, &filename);
	if (io_filein == NULL)
	{
		ttyputerr("File %s not found", lib->libfile);
		return(1);
	}

	reportcount = io_fileposition = 0;
	if (io_verbose < 0)
	{
		io_filelength = filesize(io_filein);
		if (io_filelength > 0)
		{
			if (DiaInitDialog(&usr_progressdialog) != 0)
			{
				xclose(io_filein);
				return(1);
			}
			DiaPercent(1, 0);
		}
	} else io_filelength = 0;

	/* the error recovery point */
	pp = (char *)setjmp(io_filerror);
	if (pp != 0)
	{
		ttyputerr("Error: %s on line %d, keyword '%s'", pp, io_linenumber, io_keyword);
		xclose(io_filein);
		if (io_verbose < 0 && io_filelength > 0) DiaDoneDialog();
		return(1);
	}

	/* force the library name to be the proper file name without the "txt" */
	(void)initinfstr();
	for(i=strlen(lib->libfile)-1; i>=0; i--) if (lib->libfile[i] == DIRSEP) break;
	pp = &lib->libfile[i+1];
	j = strlen(pp);
	if (j > 4 && namesame(&pp[j-4], ".txt") == 0) j -= 4;
	for(i=0; i<j; i++) (void)addtoinfstr(pp[i]);
	if (allocstring(&lib->libname, returninfstr(), lib->cluster) != 0)
		longjmp(io_filerror, (INTBIG)"No memory");

	io_curlib = lib;
	eraselibrary(lib);
	lib->userbits &= ~LIBCHANGED;

	/* clear error counters */
	io_nodeinsterror = io_portarcinsterror = io_portexpinsterror = 0;
	io_portprotoerror = io_arcinsterror = io_geomerror = io_libraryerror = 0;
	io_rtnodeerror = 0;

	io_textlevel = INLIB;
	io_linenumber = 0;
	for(;;)
	{
		/* get keyword from file */
		if (io_getkeyword()) break;
		pp = io_keyword;   while (*pp != 0) pp++;
		if (pp[-1] != ':') longjmp(io_filerror, (INTBIG)"Missing colon");
		pp[-1] = 0;

		if (reportcount++ > REPORTRATE)
		{
			reportcount = 0;
			if (io_verbose < 0 && io_filelength > 0)
				DiaPercent(1, io_fileposition*100/io_filelength);
		}

		/* determine which keyword table to use */
		switch (io_textlevel)
		{
			case INLIB:       table = io_libraryroutines;   break;
			case INFACET:     table = io_facetroutines;     break;
			case INPORTPROTO: table = io_portroutines;      break;
			case INNODEINST:  table = io_noderoutines;      break;
			case INPOR:       table = io_portinstroutines;  break;
			case INARCINST:   table = io_arcroutines;       break;
			case INARCEND:    table = io_arcendroutines;    break;
		}

		/* parse keyword and argument */
		for(i=0; table[i].matchstring != 0; i++)
			if (strcmp(table[i].matchstring, io_keyword) == 0) break;
		if (table[i].matchstring == 0)
			longjmp(io_filerror, (INTBIG)"Unknown keyword");

		/* get argument to routine and call it */
		if (io_getkeyword()) longjmp(io_filerror, (INTBIG)"EOF too soon");
		(*(table[i].routine))();
	}
	xclose(io_filein);
	if (io_verbose < 0 && io_filelength > 0)
	{
		DiaPercent(1, 100);
		DiaSetText(2, "Cleaning up...");
	}

	/* set the former version number on the library */
	version = atoi(io_version);
	nextvarchangequiet();
	(void)setvalkey((INTBIG)io_curlib, VLIBRARY, makekey("LIB_former_version"),
		(INTBIG)io_version, VSTRING|VDONTSAVE);
	efree(io_version);

	/* fill in any lambda values that are not specified */
	oldunit = ((io_curlib->userbits & LIBUNITS) >> LIBUNITSSH) << INTERNALUNITSSH;
	if (oldunit == (el_units&INTERNALUNITS)) num = den = 1; else
		db_getinternalunitscale(&num, &den, el_units, oldunit);
	for(tech = el_technologies; tech != NOTECHNOLOGY; tech = tech->nexttechnology)
	{
		lambda = io_curlib->lambda[tech->index];
		if (lambda == 0) lambda = tech->deflambda; else
			lambda = muldiv(lambda, den, num);
		io_curlib->lambda[tech->index] = lambda;
	}

	/* if this is to be the current library, adjust technologies */
	if (io_curlib == el_curlib)
		for(tech = el_technologies; tech != NOTECHNOLOGY; tech = tech->nexttechnology)
			changelambda(tech->deflambda, io_curlib->lambda[tech->index], tech);

	/* create clusters for the cells */
	for(c = io_curlib->firstcell; c != NOCELL; c = c->nextcell)
	{
		c->cluster = alloccluster(c->cellname);
		if (c->cluster == NOCLUSTER)
			longjmp(io_filerror, (INTBIG)"No cluster memory");
	}

	/* convert libraries from before version 4 */
	if (version < 4) io_yankvariabledata(io_curlib); else
	{
		/* coalesce similar cell names */
		for(np = io_curlib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
			np->temp1 = 0;

		/* associate cell names that are the same */
		assoc = 1;
		largestcluster = 0;
		for(np = io_curlib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		{
			if (np->temp1 != 0) continue;
			np->temp1 = assoc;
			clustersize = 1;
			for(onp = np->nextnodeproto; onp != NONODEPROTO; onp = onp->nextnodeproto)
				if (namesame(onp->cell->cellname, np->cell->cellname) == 0)
			{
				onp->temp1 = assoc;
				clustersize++;
			}
			if (clustersize > largestcluster) largestcluster = clustersize;
			assoc++;
		}

		/* get memory for groups of cells */
		list = (NODEPROTO **)emalloc(((sizeof (NODEPROTO *)) * largestcluster), el_tempcluster);
		if (list == 0) longjmp(io_filerror, (INTBIG)"No memory");

		/* combine names that are the same */
		for(i=1; i<assoc; i++)
		{
			/* see how many have this name */
			total = 0;
			for(np = io_curlib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
				if (np->temp1 == i) list[total++] = np;
			if (total <= 1) continue;

			/* move all but the first */
			for(j=1; j<total; j++)
			{
				np = list[j];

				/* remove the facet from the cell list */
				db_retractnodeproto(np);

				/* delete the cell */
				if (np->cell->firstincell == NONODEPROTO)
				{
					efree(np->cell->cellname);
					freecell(np->cell);
				} else ttyputmsg("Facet %s did not stand alone!", np->cell->cellname);

				/* set the cell pointer correctly */
				np->cell = list[0]->cell;

				/* insert the facet into the new cell list */
				db_insertnodeproto(np);
			}
		}

		efree((char *)list);
	}

	/* clean up the library */
	io_fixnewlib(io_curlib);

	/* print any variable-related error messages */
	if (io_nodeinsterror != 0)
		ttyputmsg("Warning: %d invalid NODEINST pointers", io_nodeinsterror);
	if (io_arcinsterror != 0)
		ttyputmsg("Warning: %d invalid ARCINST pointers", io_arcinsterror);
	if (io_portprotoerror != 0)
		ttyputmsg("Warning: %d invalid PORTPROTO pointers", io_portprotoerror);
	if (io_portarcinsterror != 0)
		ttyputmsg("Warning: %d PORTARCINST pointers not restored", io_portarcinsterror);
	if (io_portexpinsterror != 0)
		ttyputmsg("Warning: %d PORTEXPINST pointers not restored", io_portexpinsterror);
	if (io_geomerror != 0)
		ttyputmsg("Warning: %d GEOM pointers not restored", io_geomerror);
	if (io_rtnodeerror != 0)
		ttyputmsg("Warning: %d RTNODE pointers not restored", io_rtnodeerror);
	if (io_libraryerror != 0)
		ttyputmsg("Warning: LIBRARY pointers not restored", io_libraryerror);

	if (io_mainfacet != -1 && io_mainfacet < io_facetcount)
		io_curlib->curnodeproto = io_nodeprotolist[io_mainfacet]; else
			io_curlib->curnodeproto = NONODEPROTO;
	if (io_facetcount != 0) efree((char *)io_nodeprotolist);
	if (io_verbose < 0 && io_filelength > 0) DiaDoneDialog();
	return(0);
}

/******************** GENERAL PARSING ROUTINES ********************/

INTSML io_getkeyword(void)
{
	REGISTER INTSML c, index, inquote;

	/* skip leading blanks */
	for(;;)
	{
		c = xgetc(io_filein);
		io_fileposition++;
		if (c == '\n')
		{
			io_linenumber++;
			if (io_verbose == 0 && io_linenumber%1000 == 0)
				ttyputmsg("%d lines read", io_linenumber);
			continue;
		}
		if (c != ' ') break;
	}

	/* if the file ended, quit now */
	if (c == EOF) return(1);

	/* collect the word */
	index = 0;
	if (c == '"') inquote = 1; else inquote = 0;
	io_keyword[index++] = c;
	for(;;)
	{
		c = xgetc(io_filein);
		if (c == EOF) return(1);
		io_fileposition++;
		if (c == '\n' || (c == ' ' && inquote == 0)) break;
		if (index >= io_maxinputline)
		{
			/* try to allocate more space */
			if (io_grabbuffers((INTSML)(io_maxinputline*2)) != 0) break;
		}
		if (c == '"' && (index == 0 || io_keyword[index-1] != '^'))
			inquote = 1 - inquote;
		io_keyword[index++] = c;
	}
	if (c == '\n')
	{
		io_linenumber++;
		if (io_verbose == 0 && io_linenumber%1000 == 0)
			ttyputmsg("%d lines read", io_linenumber);
	}
	io_keyword[index] = 0;
	return(0);
}

/*
 * routine to allocate the keyword buffer to be the specified size
 */
INTSML io_grabbuffers(INTSML size)
{
	REGISTER char *p;
	REGISTER INTSML i;

	/* first get the new buffer */
	p = (char *)emalloc((size+1), io_aid->cluster);
	if (p == 0)
	{
		ttyputerr("No memory for input buffer");
		return(1);
	}

	/* if there was an old buffer, copy it */
	if (io_maxinputline != 0)
	{
		for(i=0; i<io_maxinputline; i++) p[i] = io_keyword[i];
		efree(io_keyword);
	}

	/* set the new size and buffer */
	io_maxinputline = size;
	io_keyword = p;
	return(0);
}

/*
 * helper routine to parse a port prototype name "line" that should be
 * in node prototype "np".  The routine returns NOPORTPROTO if it cannot
 * figure out what port this name refers to.
 */
PORTPROTO *io_getport(char *line, NODEPROTO *np)
{
	REGISTER PORTPROTO *pp;
	REGISTER INTSML i;

	for(pp = np->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
		if (namesame(line, pp->protoname) == 0) return(pp);

	/* try to parse version 1 port names */
	if (io_version[0] == '1')
	{
		/* see if database uses shortened name */
		for(pp=np->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
		{
			i = strlen(pp->protoname);
			if (namesamen(line, pp->protoname, i) == 0) return(pp);
		}

		/* see if the port name ends in a digit, fake with that */
		i = strlen(line);
		if (line[i-2] == '-' && line[i-1] >= '0' && line[i-1] <= '9')
		{
			i = (atoi(&line[i-1])-1) / 3;
			for(pp=np->firstportproto; pp!=NOPORTPROTO; pp=pp->nextportproto)
				if (i-- == 0) return(pp);
		}
	}

	/* sorry, cannot figure out what port prototype this is */
	return(NOPORTPROTO);
}

/*
 * null routine for ignoring keywords
 */
void io_null(void) {}

/******************** LIBRARY PARSING ROUTINES ********************/

/*
 * a new library is introduced (keyword "****library")
 * This should be the first keyword in the file
 */
void io_newlib(void)
{
	REGISTER TECHNOLOGY *tech;

	/* set defaults */
	io_mainfacet = -1;
	(void)allocstring(&io_version, "1.00", el_tempcluster);
	io_varpos = INVAID;
	io_curtech = NOTECHNOLOGY;
	for(tech = el_technologies; tech != NOTECHNOLOGY; tech = tech->nexttechnology)
		io_curlib->lambda[tech->index] = 0;
	io_textlevel = INLIB;
}

/*
 * get the file's Electric version number (keyword "version")
 */
void io_versn(void)
{
	(void)reallocstring(&io_version, io_keyword, el_tempcluster);
}

/*
 * get the number of aids (keyword "aids")
 */
void io_libkno(void)
{
	io_aacount = atoi(io_keyword);
	io_bitcount = 0;
}

/*
 * get the name of the aid (keyword "aidname")
 */
void io_libain(void)
{
	REGISTER AIDENTRY *aid;

	aid = getaid(io_keyword);
	if (aid == NOAID) io_bitcount = -1; else
		io_bitcount = aid->index + 1;
}

/*
 * get the number of aidbits (keyword "aidbits")
 */
void io_libaib(void)
{
	io_aacount = atoi(io_keyword);
	io_bitcount = 0;
}

/*
 * get aid information for the library (keyword "bits")
 */
void io_libbit(void)
{
	if (io_bitcount == 0) io_curlib->userbits = atoi(io_keyword);
	io_bitcount++;
}

/*
 * get the number of aidbits (keyword "userbits")
 */
void io_libusb(void)
{
	io_curlib->userbits = atoi(io_keyword);
}

/*
 * get the number of technologies (keyword "techcount")
 */
void io_libte(void)
{
	io_techcount = atoi(io_keyword);
	io_varpos = INVTECHNOLOGY;
	io_bitcount = 0;
}

/*
 * get the name of the technology (keyword "techname")
 */
void io_libten(void)
{
	REGISTER TECHNOLOGY *tech;

	tech = gettechnology(io_keyword);
	if (tech == NOTECHNOLOGY) io_bitcount = -1; else
		io_bitcount = tech->index;
}

/*
 * get lambda values for each technology in library (keyword "lambda")
 */
void io_lambda(void)
{
	REGISTER INTBIG lam;

	if (io_bitcount >= el_maxtech || io_bitcount < 0) return;
	lam = atoi(io_keyword);

	/* for version 4.0 and earlier, scale lambda by 20 */
	if (atoi(io_version) <= 4) lam *= 20;
	io_curlib->lambda[io_bitcount++] = lam;
}

/*
 * get variables on current object (keyword "variables")
 */
void io_getvar(void)
{
	REGISTER INTBIG i, j, count, type, len, naddr, ntype, *store, key, ret,
		descript;
	REGISTER char *pt, *start;
	REGISTER VARIABLE *var;
	REGISTER TECHNOLOGY *t;

	naddr = -1;
	switch (io_varpos)
	{
		case INVAID:				/* keyword applies to aids */
			if (io_bitcount < 0) break;
			naddr = (INTBIG)&el_aids[io_bitcount-1];
			ntype = VAID;
			break;
		case INVTECHNOLOGY:			/* keyword applies to technologies */
			if (io_bitcount < 0) break;
			for(t = el_technologies; t != NOTECHNOLOGY; t = t->nexttechnology)
				if (t->index == io_bitcount-1) break;
			naddr = (INTBIG)t;
			ntype = VTECHNOLOGY;
			break;
		case INVLIBRARY:			/* keyword applies to library */
			naddr = (INTBIG)io_curlib;
			ntype = VLIBRARY;
			break;
		case INVNODEPROTO:			/* keyword applies to nodeproto */
			naddr = (INTBIG)io_curnodeproto;    ntype = VNODEPROTO;
			break;
		case INVNODEINST:			/* keyword applies to nodeinst */
			naddr = (INTBIG)io_curnodeinst;    ntype = VNODEINST;
			break;
		case INVPORTARCINST:		/* keyword applies to portarcinsts */
			naddr = (INTBIG)io_varaddr;    ntype = VPORTARCINST;
			break;
		case INVPORTEXPINST:		/* keyword applies to portexpinsts */
			naddr = (INTBIG)io_varaddr;    ntype = VPORTEXPINST;
			break;
		case INVPORTPROTO:			/* keyword applies to portproto */
			naddr = (INTBIG)io_curportproto;    ntype = VPORTPROTO;
			break;
		case INVARCINST:			/* keyword applies to arcinst */
			naddr = (INTBIG)io_curarcinst;    ntype = VARCINST;
			break;
	}

	/* find out how many variables to read */
	count = atoi(io_keyword);
	for(i=0; i<count; i++)
	{
		/* read the first keyword with the name, type, and descriptor */
		if (io_getkeyword()) longjmp(io_filerror, (INTBIG)"EOF in variables");
		if (io_keyword[strlen(io_keyword)-1] != ':')
			longjmp(io_filerror, (INTBIG)"Missing ':' in variable");

		/* get the variable name */
		j = initinfstr();
		for(pt = io_keyword; *pt != 0; pt++)
		{
			if (*pt == '^' && pt[1] != 0)
			{
				pt++;
				j += addtoinfstr(*pt);
			}
			if (*pt == '(' || *pt == '[' || *pt == ':') break;
			j += addtoinfstr(*pt);
		}
		if (j != 0) longjmp(io_filerror, (INTBIG)"No memory");
		key = makekey(returninfstr());

		/* get optional length */
		if (*pt == '(')
		{
			len = myatoi(&pt[1]);
			while (*pt != '[' && *pt != ':' && *pt != 0) pt++;
		} else len = -1;

		/* get type */
		if (*pt != '[') longjmp(io_filerror, (INTBIG)"Missing '[' in variable");
		type = myatoi(&pt[1]);

		/* get the descriptor */
		while (*pt != ',' && *pt != ':' && *pt != 0) pt++;
		if (*pt == ',') descript = myatoi(&pt[1]); else descript = 0;

		/* get value */
		if (io_getkeyword()) longjmp(io_filerror, (INTBIG)"EOF in variables");
		if ((type&VISARRAY) != 0)
		{
			if (len < 0)
			{
				len = (type&VLENGTH) >> VLENGTHSH;
				store = emalloc((len * SIZEOFINTBIG), el_tempcluster);
				if (store == 0) longjmp(io_filerror, (INTBIG)"No memory");
			} else
			{
				store = emalloc(((len+1) * SIZEOFINTBIG), el_tempcluster);
				if (store == 0) longjmp(io_filerror, (INTBIG)"No memory");
				store[len] = -1;
			}
			pt = io_keyword;
			if (*pt++ != '[') longjmp(io_filerror, (INTBIG)"Missing '[' in variable array");
			for(j=0; j<len; j++)
			{
				/* string arrays must be handled specially */
				if ((type&VTYPE) == VSTRING)
				{
					while (*pt != '"' && *pt != 0) pt++;
					if (*pt != 0)
					{
						start = pt++;
						for(;;)
						{
							if (pt[0] == '^' && pt[1] != 0)
							{
								pt += 2;
								continue;
							}
							if (pt[0] == '"' || pt[0] == 0) break;
							pt++;
						}
						if (*pt != 0) pt++;
					}
				} else
				{
					start = pt;
					while (*pt != ',' && *pt != ']' && *pt != 0) pt++;
				}
				if (*pt == 0) longjmp(io_filerror, (INTBIG)"Short array specification");
				*pt++ = 0;
				if ((type&VTYPE) == VGENERAL)
				{
					if ((j&1) == 0)
					{
						store[j+1] = myatoi(start);
					} else
					{
						store[j-1] = io_decode(start, store[j]);
					}
				} else
				{
					store[j] = io_decode(start, type);
				}
			}
			if (naddr != -1)
			{
				nextvarchangequiet();
				var = setvalkey(naddr, ntype, key, (INTBIG)store, type);
				if (var == NOVARIABLE) longjmp(io_filerror, (INTBIG)"No memory");
				var->textdescript = descript;
			}
			efree((char *)store);
		} else
		{
			ret = io_decode(io_keyword, type);
			if (naddr != -1)
			{
				nextvarchangequiet();
				var = setvalkey(naddr, ntype, key, ret, type);
				if (var == NOVARIABLE) longjmp(io_filerror, (INTBIG)"No memory");
				var->textdescript = descript;
			}
		}
	}
}

INTBIG io_decode(char *name, INTBIG type)
{
	REGISTER INTBIG thistype, index;
	REGISTER char *out, *retur;
	REGISTER NODEPROTO *np;

	thistype = type;
	if ((thistype&(VCODE1|VCODE2)) != 0) thistype = VSTRING;

	switch (thistype&VTYPE)
	{
		case VINTEGER:
		case VFRACT:
		case VADDRESS:
			return(myatoi(name));
		case VCHAR:
			return((INTBIG)name);
		case VSTRING:
			if (*name == '"') name++;
			retur = name;
			for(out = name; *name != 0; name++)
			{
				if (*name == '^' && name[1] != 0)
				{
					name++;
					*out++ = *name;
					continue;
				}
				if (*name == '"') break;
				*out++ = *name;
			}
			*out = 0;
			return((INTBIG)retur);
		case VFLOAT:
		case VDOUBLE:
			return(castint(atof(name)));
		case VNODEINST:
			index = myatoi(name);
			if (index >= 0 && index < io_nodeinstcount) return((INTBIG)io_nodelist[index]);
			io_nodeinsterror++;
			return(-1);
		case VNODEPROTO:
			index = myatoi(name);
			if (index == -1) return(-1);

			/* see if there is a ":" in the type */
			for(out = name; *out != 0; out++) if (*out == ':') break;
			if (*out == 0)
			{
				index = atoi(name);
				if (index == -1) return(-1);
				return((INTBIG)io_nodeprotolist[index]);
			}

			/* parse primitive nodeproto name */
			np = getnodeproto(name);
			if (np == NONODEPROTO) longjmp(io_filerror, (INTBIG)"Unknown node prototype");
			if (np->index == 0) longjmp(io_filerror, (INTBIG)"Unknown node prototype");
			return((INTBIG)np);
		case VPORTARCINST:
			io_portarcinsterror++;   break;
		case VPORTEXPINST:
			io_portexpinsterror++;   break;
		case VPORTPROTO:
			index = myatoi(name);
			if (index >= 0 && index < io_portprotocount) return((INTBIG)io_portprotolist[index]);
			io_portprotoerror++;
			return(-1);
		case VARCINST:
			index = myatoi(name);
			if (index >= 0 && index < io_arcinstcount) return((INTBIG)io_arclist[index]);
			io_arcinsterror++;
			return(-1);
		case VARCPROTO:
			return((INTBIG)getarcproto(name));
		case VGEOM:
			io_geomerror++;   break;
		case VLIBRARY:
			io_libraryerror++;   break;
		case VTECHNOLOGY:
			return((INTBIG)io_gettechnology(name));
		case VAID:
			return((INTBIG)getaid(name));
		case VRTNODE:
			io_rtnodeerror++;   break;
	}
	return(-1);
}

/*
 * get the number of facets in this library (keyword "cellcount")
 */
void io_libcc(void)
{
	REGISTER INTSML i;
	REGISTER CELL *c;

	io_facetcount = atoi(io_keyword);
	if (io_facetcount == 0)
	{
		io_curlib->firstnodeproto = NONODEPROTO;
		return;
	}

	io_varpos = INVLIBRARY;

	/* allocate a list of node prototypes for this library */
	io_nodeprotolist = (NODEPROTO **)emalloc(((sizeof (NODEPROTO *)) * io_facetcount),
		el_tempcluster);
	if (io_nodeprotolist == 0) longjmp(io_filerror, (INTBIG)"No memory");
	for(i=0; i<io_facetcount; i++)
	{
		c = alloccell(io_curlib->cluster);
		if (c == NOCELL) longjmp(io_filerror, (INTBIG)"No memory for facets");
		io_nodeprotolist[i] = allocnodeproto(io_curlib->cluster);
		if (io_nodeprotolist[i] == NONODEPROTO)
			longjmp(io_filerror, (INTBIG)"No memory for facets");
		io_nodeprotolist[i]->cell = c;
		c->firstincell = io_nodeprotolist[i];
		c->nextcell = NOCELL;
		c->cluster = io_curlib->cluster;
		if (i > 0) io_nodeprotolist[i-1]->cell->nextcell = c;
		c->lib = io_curlib;
		io_nodeprotolist[i]->cellview = el_unknownview;
		io_nodeprotolist[i]->newestversion = io_nodeprotolist[i];
		io_nodeprotolist[i]->tech = NOTECHNOLOGY;
		io_nodeprotolist[i]->index = 0;
		io_nodeprotolist[i]->firstportproto = NOPORTPROTO;
		io_nodeprotolist[i]->firstnodeinst = NONODEINST;
		io_nodeprotolist[i]->firstarcinst = NOARCINST;

		/* create geometry structure in the facet */
		if (geomstructure(io_nodeprotolist[i]) != 0)
			longjmp(io_filerror, (INTBIG)"No memory for geometry structure");
	}
	if (io_facetcount == 0) io_curlib->firstcell = NOCELL; else
		io_curlib->firstcell = io_nodeprotolist[0]->cell;

	/* create the linked list of node prototypes in this library */
	for(i=0; i<io_facetcount; i++)
	{
		if (i != 0) io_nodeprotolist[i]->lastnodeproto = io_nodeprotolist[i-1]; else
			io_nodeprotolist[i]->lastnodeproto = NONODEPROTO;
		if (i != io_facetcount-1)
			io_nodeprotolist[i]->nextnodeproto = io_nodeprotolist[i+1]; else
				io_nodeprotolist[i]->nextnodeproto = NONODEPROTO;
	}
	io_curlib->firstnodeproto = io_nodeprotolist[0];
}

/*
 * get the main facet of this library (keyword "maincell")
 */
void io_libms(void)
{
	io_mainfacet = atoi(io_keyword);
}

/*
 * get a view (keyword "view")
 */
void io_libvie(void)
{
	REGISTER char *pt;
	REGISTER INTSML len;
	REGISTER VIEW *v;

	pt = io_keyword;
	while (*pt != 0 && *pt != '{') pt++;
	if (*pt != '{') longjmp(io_filerror, (INTBIG)"Missing view abbreviation");
	*pt++ = 0;
	len = strlen(pt);
	if (pt[len-1] != '}') longjmp(io_filerror, (INTBIG)"View abbreviation bad");
	pt[len-1] = 0;
	v = getview(io_keyword);
	if (v == NOVIEW)
	{
		v = allocview();
		if (v == NOVIEW) longjmp(io_filerror, (INTBIG)"No memory");
		(void)allocstring(&v->viewname, io_keyword, db_cluster);
		(void)allocstring(&v->sviewname, pt, db_cluster);
		v->nextview = el_views;
		el_views = v;
	} else
	{
		if (namesame(v->sviewname, pt) != 0)
			longjmp(io_filerror, (INTBIG)"View abbreviation inconsistent");
	}
	pt[-1] = '{';   pt[len-1] = '}';
}

/******************** FACET PARSING ROUTINES ********************/

/*
 * initialize for a new facet (keyword "***cell")
 */
void io_newcel(void)
{
	io_curnodeproto = io_nodeprotolist[atoi(io_keyword)];
	io_textlevel = INFACET;
	io_varpos = INVNODEPROTO;
}

/*
 * get the name of the current facet (keyword "name")
 */
void io_celnam(void)
{
	REGISTER char *pt;
	REGISTER VIEW *v;

	if (io_verbose != 0)
	{
		if (io_verbose < 0 && io_filelength > 0)
		{
			(void)initinfstr();
			(void)addstringtoinfstr("Reading ");
			(void)addstringtoinfstr(io_keyword);
			DiaSetText(2, returninfstr());
		} else ttyputmsg("Reading %s", io_keyword);
	}
	io_curnodeproto->cellview = el_unknownview;
	pt = io_keyword;
	while (*pt != 0 && *pt != '{') pt++;
	if (*pt == '{')
	{
		*pt = 0;
		if (allocstring(&io_curnodeproto->cell->cellname, io_keyword, io_curlib->cluster) != 0)
			longjmp(io_filerror, (INTBIG)"No memory for facet name");
		pt++;
		pt[strlen(pt)-1] = 0;
		for(v = el_views; v != NOVIEW; v = v->nextview)
			if (*v->sviewname != 0 && namesame(pt, v->sviewname) == 0) break;
		if (v != NOVIEW) io_curnodeproto->cellview = v;
	} else
	{
		if (allocstring(&io_curnodeproto->cell->cellname, io_keyword, io_curlib->cluster) != 0)
			longjmp(io_filerror, (INTBIG)"No memory for facet name");
	}
}

/*
 * get the version of the current facet (keyword "version")
 */
void io_celver(void)
{
	io_curnodeproto->version = atoi(io_keyword);
}

/*
 * get the creation date of the current facet (keyword "creationdate")
 */
void io_celcre(void)
{
	io_curnodeproto->creationdate = atoi(io_keyword);
}

/*
 * get the revision date of the current facet (keyword "revisiondate")
 */
void io_celrev(void)
{
	io_curnodeproto->revisiondate = atoi(io_keyword);
}

/*
 * get the boundary of the current facet
 */
void io_cellx(void)
{
	io_curnodeproto->lowx = atoi(io_keyword);
}

void io_celhx(void)
{
	io_curnodeproto->highx = atoi(io_keyword);
}

void io_celly(void)
{
	io_curnodeproto->lowy = atoi(io_keyword);
}

void io_celhy(void)
{
	io_curnodeproto->highy = atoi(io_keyword);
}

/*
 * get the default technology for objects in this facet (keyword "technology")
 */
void io_tech(void)
{
	REGISTER TECHNOLOGY *tech;

	tech = io_gettechnology(io_keyword);
	if (tech == NOTECHNOLOGY) longjmp(io_filerror, (INTBIG)"Unknown technology");
	io_curtech = tech;
}

/*
 * get the aid dirty word for the current facet (keyword "aadirty")
 */
void io_celaad(void)
{
	io_curnodeproto->adirty = atoi(io_keyword);
	io_bitcount = 0;
}

/*
 * get aid information for current facet (keyword "bits")
 */
void io_celbit(void)
{
	if (io_bitcount == 0) io_curnodeproto->userbits = atoi(io_keyword);
	io_bitcount++;
}

/*
 * get aid information for current facet (keyword "userbits")
 */
void io_celusb(void)
{
	io_curnodeproto->userbits = atoi(io_keyword);
}

/*
 * get aid information for current facet (keyword "netnum")
 */
void io_celnet(void) {}

/*
 * get the number of node instances in the current facet (keyword "nodes")
 */
void io_celnoc(void)
{
	REGISTER INTSML i;
	REGISTER NODEINST *ni;
	REGISTER GEOM **geomlist;

	io_nodeinstcount = atoi(io_keyword);
	if (io_nodeinstcount == 0) return;
	io_nodelist = (NODEINST **)emalloc(((sizeof (NODEINST *)) * io_nodeinstcount), el_tempcluster);
	if (io_nodelist == 0) longjmp(io_filerror, (INTBIG)"No memory for nodeinsts");
	geomlist = (GEOM **)emalloc(((sizeof (GEOM *)) * io_nodeinstcount), el_tempcluster);
	if (geomlist == 0) longjmp(io_filerror, (INTBIG)"No memory for nodeinsts");
	for(i=0; i<io_nodeinstcount; i++)
	{
		allocnodeinst(1, &io_nodelist[i], io_curlib->cluster);
		allocgeom(1, &geomlist[i], io_curlib->cluster);
		ni = io_nodelist[i];
		if (ni == NONODEINST || geomlist[i] == NOGEOM)
			longjmp(io_filerror, (INTBIG)"No memory for nodeinsts");
		ni->parent = io_curnodeproto;
		ni->firstportarcinst = NOPORTARCINST;
		ni->firstportexpinst = NOPORTEXPINST;
		ni->geom = geomlist[i];

		/* compute linked list of nodes in this facet */
		if (io_curnodeproto->firstnodeinst != NONODEINST)
			io_curnodeproto->firstnodeinst->lastnodeinst = ni;
		ni->nextnodeinst = io_curnodeproto->firstnodeinst;
		ni->lastnodeinst = NONODEINST;
		io_curnodeproto->firstnodeinst = ni;
	}
	efree((char *)geomlist);
}

/*
 * get the number of arc instances in the current facet (keyword "arcs")
 */
void io_celarc(void)
{
	REGISTER INTSML i;
	REGISTER ARCINST *ai;
	REGISTER GEOM **geomlist;

	io_arcinstcount = atoi(io_keyword);
	if (io_arcinstcount == 0) return;
	io_arclist = (ARCINST **)emalloc(((sizeof (ARCINST *)) * io_arcinstcount), el_tempcluster);
	if (io_arclist == 0) longjmp(io_filerror, (INTBIG)"No memory for arcinsts");
	geomlist = (GEOM **)emalloc(((sizeof (GEOM *)) * io_arcinstcount), el_tempcluster);
	if (geomlist == 0) longjmp(io_filerror, (INTBIG)"No memory for arcinsts");
	for(i=0; i<io_arcinstcount; i++)
	{
		allocarcinst(1, &io_arclist[i], io_curlib->cluster);
		allocgeom(1, &geomlist[i], io_curlib->cluster);
		ai = io_arclist[i];
		if (ai == NOARCINST || geomlist[i] == NOGEOM)
			longjmp(io_filerror, (INTBIG)"No memory for arcinsts");
		ai->parent = io_curnodeproto;
		ai->geom = geomlist[i];

		/* compute linked list of arcs in this facet */
		if (io_curnodeproto->firstarcinst != NOARCINST)
			io_curnodeproto->firstarcinst->lastarcinst = ai;
		ai->nextarcinst = io_curnodeproto->firstarcinst;
		ai->lastarcinst = NOARCINST;
		io_curnodeproto->firstarcinst = ai;
	}
	efree((char *)geomlist);
}

/*
 * get the number of port prototypes in the current facet (keyword "porttypes")
 */
void io_celptc(void)
{
	REGISTER INTSML i;

	io_portprotocount = atoi(io_keyword);
	if (io_portprotocount == 0) return;
	io_portprotolist = (PORTPROTO **)emalloc(((sizeof (PORTPROTO *))* io_portprotocount),
		el_tempcluster);
	if (io_portprotolist == 0) longjmp(io_filerror, (INTBIG)"No memory for port prototypes");
	for(i=0; i<io_portprotocount; i++)
	{
		allocportproto(1, &io_portprotolist[i], io_curlib->cluster);
		if (io_portprotolist[i] == NOPORTPROTO)
			longjmp(io_filerror, (INTBIG)"No memory for port prototypes");
		io_portprotolist[i]->parent = io_curnodeproto;
	}

	/* link the portprotos */
	io_curnodeproto->firstportproto = io_portprotolist[0];
	for(i=1; i<io_portprotocount; i++)
		io_portprotolist[i-1]->nextportproto = io_portprotolist[i];
	io_portprotolist[io_portprotocount-1]->nextportproto = NOPORTPROTO;
}

/*
 * close the current facet (keyword "celldone")
 */
void io_celdon(void)
{
	REGISTER INTSML i;
	REGISTER NODEINST *ni;
	REGISTER ARCINST *ai;
	REGISTER PORTARCINST *pi;
	REGISTER GEOM *geom;

	/* verify facet name */
	if (namesame(io_keyword, io_curnodeproto->cell->cellname) != 0)
		ttyputmsg("Warning: facet '%s' wants to be '%s'", io_curnodeproto->cell->cellname,
			io_keyword);

	/* silly hack: convert arcinst->end->portarcinst pointers */
	for(i=0; i<io_nodeinstcount; i++)
	{
		ni = io_nodelist[i];
		for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
		{
			ai = pi->conarcinst;
			if (ai->end[0].nodeinst != ni)
			{
				ai->end[1].portarcinst = pi;
				continue;
			}
			if (ai->end[1].nodeinst != ni)
			{
				ai->end[0].portarcinst = pi;
				continue;
			}
			if ((PORTPROTO *)ai->end[0].portarcinst == pi->proto) ai->end[0].portarcinst = pi;
			if ((PORTPROTO *)ai->end[1].portarcinst == pi->proto) ai->end[1].portarcinst = pi;
		}
	}

	/* fill geometry module for each nodeinst */
	for(i=0; i<io_nodeinstcount; i++)
	{
		ni = io_nodelist[i];
		geom = ni->geom;
		geom->entrytype = OBJNODEINST;  geom->entryaddr.ni = ni;
		linkgeom(geom, io_curnodeproto);
	}

	/* fill geometry modules for each arcinst */
	for(i=0; i<io_arcinstcount; i++)
	{
		ai = io_arclist[i];
		setshrinkvalue(ai);
		geom = ai->geom;
		geom->entrytype = OBJARCINST;  geom->entryaddr.ai = ai;
		linkgeom(geom, io_curnodeproto);
	}

	/* free the lists of objects in this facet */
	if (io_nodeinstcount != 0) efree((char *)io_nodelist);
	if (io_portprotocount != 0) efree((char *)io_portprotolist);
	if (io_arcinstcount != 0) efree((char *)io_arclist);
}

/******************** NODE INSTANCE PARSING ROUTINES ********************/

/*
 * initialize for a new node instance (keyword "**node")
 */
void io_newno(void)
{
	io_curnodeinst = io_nodelist[atoi(io_keyword)];
	io_textlevel = INNODEINST;
	io_varpos = INVNODEINST;
}

/*
 * get the type of the current nodeinst (keyword "type")
 */
void io_nodtyp(void)
{
	REGISTER NODEPROTO *np;
	REGISTER TECHNOLOGY *tech;
	REGISTER char *pt, *line;
	char orig[50];
	static INTBIG orignodename = 0;

	line = io_keyword;
	if (*line == '[')
	{
		io_curnodeinst->proto = io_nodeprotolist[atoi(&line[1])];
	} else
	{
		for(pt = line; *pt != 0; pt++) if (*pt == ':') break;
		if (*pt == ':')
		{
			*pt = 0;
			tech = io_gettechnology(line);
			if (tech == NOTECHNOLOGY) longjmp(io_filerror, (INTBIG)"Unknown technology");
			*pt++ = ':';
			line = pt;
		} else tech = io_curtech;
		if (io_curtech == NOTECHNOLOGY) io_curtech = tech;
		for(np = tech->firstnodeproto; np != NONODEPROTO;
			np = np->nextnodeproto) if (namesame(np->primname, line) == 0)
		{
			io_curnodeinst->proto = np;
			return;
		}

		/* convert "message" and "cell-center" nodes */
		if (namesame(line, "Cell-Center") == 0) io_curnodeinst->proto = gen_facetcenterprim; else
		if (namesame(line, "Message") == 0 || namesame(line, "Centered-Message") == 0 ||
			namesame(line, "Left-Message") == 0 || namesame(line, "Right-Message") == 0)
		{
			io_curnodeinst->proto = gen_invispinprim;
			(void)strcpy(orig, "artwork:");
			(void)strcat(orig, line);
			if (orignodename == 0) orignodename = makekey("ORIGINAL_NODENAME");
			nextvarchangequiet();
			(void)setvalkey((INTBIG)io_curnodeinst, VNODEINST,
				orignodename, (INTBIG)orig, VSTRING|VDONTSAVE);
		}
		if (io_curnodeinst->proto != NONODEPROTO) return;

		longjmp(io_filerror, (INTBIG)"Unknown node prototype");
	}
}

/*
 * get the bounding box information for the current node instance
 */
void io_nodlx(void)
{
	io_curnodeinst->lowx = atoi(io_keyword);
}

void io_nodhx(void)
{
	io_curnodeinst->highx = atoi(io_keyword);
}

void io_nodly(void)
{
	io_curnodeinst->lowy = atoi(io_keyword);
}

void io_nodhy(void)
{
	io_curnodeinst->highy = atoi(io_keyword);
}

/*
 * get the instance name of the current node instance (keyword "name")
 */
void io_nodnam(void)
{
	nextvarchangequiet();
	(void)setvalkey((INTBIG)io_curnodeinst, VNODEINST, el_node_name,
		(INTBIG)io_keyword, VSTRING|VDISPLAY);
}

/*
 * get the text descriptor of the current node instance (keyword "descript")
 */
void io_noddes(void)
{
	io_curnodeinst->textdescript = atoi(io_keyword);
}

/*
 * get the rotation for the current nodeinst (keyword "rotation");
 */
void io_nodrot(void)
{
	io_curnodeinst->rotation = atoi(io_keyword);
}

/*
 * get the transposition for the current nodeinst (keyword "transpose")
 */
void io_nodtra(void)
{
	io_curnodeinst->transpose = atoi(io_keyword);
}

/*
 * get the aid seen bits for the current nodeinst (keyword "aseen")
 */
void io_nodkse(void)
{
	io_bitcount = 0;
}

/*
 * get the port count for the current nodeinst (keyword "ports")
 */
void io_nodpoc(void)
{
	io_curportcount = atoi(io_keyword);
}

/*
 * get aid information for current nodeinst (keyword "bits")
 */
void io_nodbit(void)
{
	if (io_bitcount == 0) io_curnodeinst->userbits = atoi(io_keyword);
	io_bitcount++;
}

/*
 * get aid information for current nodeinst (keyword "userbits")
 */
void io_nodusb(void)
{
	io_curnodeinst->userbits = atoi(io_keyword);
}

/*
 * initialize for a new portinst on the current nodeinst (keyword "*port")
 */
void io_newpor(void)
{
	REGISTER INTSML i, index;
	REGISTER PORTPROTO *pp;

	if (io_version[0] == '1')
	{
		/* version 1 files used an index here */
		index = atoi(io_keyword);

		/* dividing is a heuristic used to decypher version 1 files */
		for(pp = io_curnodeinst->proto->firstportproto, i=0; pp != NOPORTPROTO;
			pp = pp->nextportproto) i++;
		if (i < io_curportcount)
		{
			for(;;)
			{
				index /= 3;
				if (index < i) break;
			}
		}

		for(io_curportproto = io_curnodeinst->proto->firstportproto, i=0;
			io_curportproto != NOPORTPROTO; io_curportproto = io_curportproto->nextportproto, i++)
				if (i == index) break;
	} else
	{
		/* version 2 and later use port prototype names */
		io_curportproto = io_getport(io_keyword, io_curnodeinst->proto);
	}
	io_textlevel = INPOR;
}

/*
 * get an arc connection for the current nodeinst (keyword "arc")
 */
void io_porarc(void)
{
	PORTARCINST *pi;

	allocportarcinst(1, &pi, io_curlib->cluster);
	if (pi == NOPORTARCINST) longjmp(io_filerror, (INTBIG)"No memory");
	pi->proto = io_curportproto;
	db_addportarcinst(io_curnodeinst, pi);
	pi->conarcinst = io_arclist[atoi(io_keyword)];
	io_varpos = INVPORTARCINST;
	io_varaddr = (INTBIG)pi;
}

/*
 * get an export site for the current nodeinst (keyword "exported")
 */
void io_porexp(void)
{
	PORTEXPINST *pe;

	allocportexpinst(1, &pe, io_curlib->cluster);
	if (pe == NOPORTEXPINST) longjmp(io_filerror, (INTBIG)"No memory");
	pe->proto = io_curportproto;
	db_addportexpinst(io_curnodeinst, pe);
	pe->exportproto = io_portprotolist[atoi(io_keyword)];
	io_varpos = INVPORTEXPINST;
	io_varaddr = (INTBIG)pe;
}

/******************** ARC INSTANCE PARSING ROUTINES ********************/

/*
 * initialize for a new arc instance (keyword "**arc")
 */
void io_newar(void)
{
	io_curarcinst = io_arclist[atoi(io_keyword)];
	io_textlevel = INARCINST;
	io_varpos = INVARCINST;
}

/*
 * get the type of the current arc instance (keyword "type")
 */
void io_arctyp(void)
{
	REGISTER ARCPROTO *ap;
	REGISTER char *pt, *line;
	REGISTER TECHNOLOGY *tech;

	line = io_keyword;
	for(pt = line; *pt != 0; pt++) if (*pt == ':') break;
	if (*pt == ':')
	{
		*pt = 0;
		tech = io_gettechnology(line);
		if (tech == NOTECHNOLOGY) longjmp(io_filerror, (INTBIG)"Unknown technology");
		*pt++ = ':';
		line = pt;
	} else tech = io_curtech;
	if (io_curtech == NOTECHNOLOGY) io_curtech = tech;
	for(ap = tech->firstarcproto; ap != NOARCPROTO; ap = ap->nextarcproto)
		if (namesame(line, ap->protoname) == 0)
	{
		io_curarcinst->proto = ap;
		return;
	}

	/* special hack: try the generic technology if name is not found */
	for(ap = gen_tech->firstarcproto; ap != NOARCPROTO; ap = ap->nextarcproto)
		if (namesame(line, ap->protoname) == 0)
	{
		io_curarcinst->proto = ap;
		return;
	}

	longjmp(io_filerror, (INTBIG)"Invalid arc prototype");
}

/*
 * get the instance name of the current arc instance (keyword "name")
 */
void io_arcnam(void)
{
	nextvarchangequiet();
	(void)setvalkey((INTBIG)io_curarcinst, VARCINST, el_arc_name,
		(INTBIG)io_keyword, VSTRING|VDISPLAY);
}

/*
 * get the width of the current arc instance (keyword "width")
 */
void io_arcwid(void)
{
	io_curarcinst->width = atoi(io_keyword);
}

/*
 * get the length of the current arc instance (keyword "length")
 */
void io_arclen(void)
{
	io_curarcinst->length = atoi(io_keyword);
}

/*
 * get the signals information of the current arc instance (keyword "signals")
 */
void io_arcsig(void) {}

/*
 * initialize for an end of the current arcinst (keyword "*end")
 */
void io_newend(void)
{
	io_curarcend = atoi(io_keyword);
	io_textlevel = INARCEND;
}

/*
 * get the node at the current end of the current arcinst (keyword "node")
 */
void io_endnod(void)
{
	io_curarcinst->end[io_curarcend].nodeinst = io_nodelist[atoi(io_keyword)];
}

/*
 * get the porttype at the current end of current arcinst (keyword "nodeport")
 */
void io_endpt(void)
{
	REGISTER PORTPROTO *pp;

	pp = io_getport(io_keyword,io_curarcinst->end[io_curarcend].nodeinst->proto);
	if (pp == NOPORTPROTO) longjmp(io_filerror, (INTBIG)"Unknown port prototype name");
	io_curarcinst->end[io_curarcend].portarcinst = (PORTARCINST *)pp;
}

/*
 * get the coordinates of the current end of the current arcinst
 */
void io_endxp(void)
{
	io_curarcinst->end[io_curarcend].xpos = atoi(io_keyword);
}

void io_endyp(void)
{
	io_curarcinst->end[io_curarcend].ypos = atoi(io_keyword);
}

/*
 * get the aid information for the current arcinst (keyword "aseen")
 */
void io_arckse(void)
{
	io_bitcount = 0;
}

/*
 * get aid information for current arcinst (keyword "bits")
 */
void io_arcbit(void)
{
	if (io_bitcount == 0) io_curarcinst->userbits = atoi(io_keyword);
	io_bitcount++;
}

/*
 * get aid information for current arcinst (keyword "userbits")
 */
void io_arcusb(void)
{
	io_curarcinst->userbits = atoi(io_keyword);
}

/*
 * get aid information for current arcinst (keyword "netnum")
 */
void io_arcnet(void) {}

/******************** PORT PROTOTYPE PARSING ROUTINES ********************/

/*
 * initialize for a new port prototype (keyword "**porttype")
 */
void io_newpt(void)
{
	io_curportproto = io_portprotolist[atoi(io_keyword)];
	io_textlevel = INPORTPROTO;
	io_varpos = INVPORTPROTO;
}

/*
 * get the name for the current port prototype (keyword "name")
 */
void io_ptnam(void)
{
	if (allocstring(&io_curportproto->protoname, io_keyword, io_curlib->cluster) != 0)
		longjmp(io_filerror, (INTBIG)"No memory for portproto name");
}

/*
 * get the text descriptor for the current port prototype (keyword "descript")
 */
void io_ptdes(void)
{
	io_curportproto->textdescript = atoi(io_keyword);
}

/*
 * get the sub-nodeinst for the current port prototype (keyword "subnode")
 */
void io_ptsno(void)
{
	io_curportproto->subnodeinst = io_nodelist[atoi(io_keyword)];
}

/*
 * get the sub-portproto for the current port prototype (keyword "subport")
 */
void io_ptspt(void)
{
	REGISTER PORTPROTO *pp;

	pp = io_getport(io_keyword, io_curportproto->subnodeinst->proto);
	if (pp == NOPORTPROTO) longjmp(io_filerror, (INTBIG)"Unknown port prototype name");
	io_curportproto->subportproto = pp;
}

/*
 * get the aid seen for the current port prototype (keyword "aseen")
 */
void io_ptkse(void)
{
	io_bitcount = 0;
}

/*
 * get the aid data for the current port prototype (keyword "bits")
 */
void io_ptbit(void)
{
	if (io_bitcount == 0) io_curportproto->userbits = atoi(io_keyword);
	io_bitcount++;
}

/*
 * get the aid data for the current port prototype (keyword "userbits")
 */
void io_ptusb(void)
{
	io_curportproto->userbits = atoi(io_keyword);
}

/*
 * get the aid data for the current port prototype (keyword "netnum")
 */
void io_ptnet(void) {}

/*
 * routine to convert the technology name in "line" to a technology.
 * also handles conversion of the old technology name "logic"
 */
TECHNOLOGY *io_gettechnology(char *line)
{
	REGISTER TECHNOLOGY *tech;

	tech = gettechnology(line);
	if (tech != NOTECHNOLOGY) return(tech);
	if (namesame(line, "logic") == 0) tech = sch_tech;
	return(tech);
}
