/*
 * Electric(tm) VLSI Design System
 *
 * File: rout.c
 * Routines for the wire routing aid
 * Written by: Steven M. Rubin, Electric Editor Incorporated
 *
 * Copyright (c) 1998 Electric Editor Incorporated.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Electric Editor Incorporated
 * 23470 Sunset Drive, Suite 108
 * Los Gatos, California 95033
 * support@electriceditor.com
 */

#include "config.h"
#if ROUTAID

#include "global.h"
#include "rout.h"
#include "usr.h"
#include "tecschem.h"
#include "tecgen.h"

/* the ROUTER aid table */
static COMCOMP routerfacetp = {NOKEYWORD, us_topoffacets, us_nextparse, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "Facet to stitch", 0};
COMCOMP ro_routerarcp = {NOKEYWORD, us_topofarcs, us_nextparse, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "Arc prototype to use in stitching (* to reset)", 0};
static KEYWORD routerautoopt[] =
{
	{"stitch-now",             1,{&routerfacetp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"highlighted-stitch-now", 0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP routerautop = {routerautoopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "Auto-stitching action", 0};
static COMCOMP routerriverp = {NOKEYWORD, us_topoffacets, us_nextparse, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "Facet to river-route", 0};
static COMCOMP routermazesetboundaryp = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	0, " \t", "routing boundary", 0};
static COMCOMP routermazesetgridxp = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	0, " \t", "routing grid x", 0};
static COMCOMP routermazesetgridyp = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	0, " \t", "routing grid y", 0};
static COMCOMP routermazesetoffsetxp = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	0, " \t", "routing offset x", 0};
static COMCOMP routermazesetoffsetyp = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	0, " \t", "routing offset y", 0};
static KEYWORD routermazesetopt[] =
{
	{"boundary",			 1,{&routermazesetboundaryp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"grid",				 2,{&routermazesetgridxp,&routermazesetgridyp,NOKEY,NOKEY,NOKEY}},
	{"offset",			     2,{&routermazesetoffsetxp,&routermazesetoffsetyp,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP routermazesetp = {routermazesetopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "set operation", 0 };
static KEYWORD routermazerouteopt[] =
{
	{"cell",	 0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"selected", 0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP routermazeroutep = {routermazerouteopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "routing operation", 0 };
static KEYWORD routermazeopt[] =
{
	{"set",	    1, {&routermazesetp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"route",   1, {&routermazeroutep,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP routermazep = {routermazeopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "Maze routing operation", 0 };
static KEYWORD routernopt[] =
{
	{"select",             0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP routernp = {routernopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "Negating action", 0};
static KEYWORD routeropt[] =
{
	{"auto-stitch",    1,{&routerautop,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"mimic-stitch",   0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"river-route",    1,{&routerriverp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"maze-route",     1,{&routermazep,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"use-arc",        1,{&ro_routerarcp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"select",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"unroute",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"not",            1,{&routernp,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP ro_routerp = {routeropt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "Routing action", 0};

RCHECK *ro_firstrcheck;
RCHECK *ro_rcheckfree;

AIDENTRY *ro_source;		/* source of batch of changes */
INTBIG    ro_state;			/* cached key for "ROUT_state" */
INTSML    ro_count;			/* number of stitches made */
INTBIG    ro_prefered;		/* cached key for "ROUT_prefered_arc" */
ARCINST  *ro_lastarc;		/* last created arc for mimicing */
AIDENTRY *ro_aid;			/* the Router aid object */

/* prototypes for local routines */
RCHECK *ro_allocrcheck(void);
void ro_freercheck(RCHECK*);
void ro_queuercheck(NODEINST*);
void ro_unroutecurrent(void);
INTSML ro_unroutenet(NETWORK *net);

/*
 * routine to allocate a new check module from the pool (if any) or memory
 */
#define	RCHECKBLOCKSIZE	50	/* number to "malloc" at a time */
RCHECK *ro_allocrcheck(void)
{
	REGISTER RCHECK *r;
	REGISTER INTSML i, j;

	if (ro_rcheckfree == NORCHECK)
	{
		for(i=RCHECKBLOCKSIZE; i>0; i /= 2)
		{
			r = (RCHECK *)emalloc(((sizeof (RCHECK)) * i), ro_aid->cluster);
			if (r != 0) break;
		}
		if (r == 0) return(NORCHECK);
		for(j=0; j<i; j++)
			if (j == 0) r[j].nextcheck = NORCHECK; else
				r[j].nextcheck = &r[j-1];
		ro_rcheckfree = &r[i-1];
	}

	/* take module from free list */
	r = ro_rcheckfree;
	ro_rcheckfree = (RCHECK *)r->nextcheck;
	return(r);
}

/*
 * routine to return check module "r" to the pool of free modules
 */
void ro_freercheck(RCHECK *r)
{
	r->nextcheck = ro_rcheckfree;
	ro_rcheckfree = r;
}

/*
 * routine to queue nodeinst "ni" to be checked during the next slice of
 * the router
 */
void ro_queuercheck(NODEINST *ni)
{
	REGISTER RCHECK *r;

	r = ro_allocrcheck();
	if (r == NORCHECK)
	{
		ttyputerr("ROUTING: out of space");
		return;
	}
	r->entity = ni;
	r->nextcheck = ro_firstrcheck;
	ro_firstrcheck = r;
}

void ro_init(INTBIG *argc, char *argv[], AIDENTRY *thisaid)
{
	if (thisaid == 0)
	{
		/* pass 3: nothing to do */
	} else if (thisaid == NOAID)
	{
		/* pass 2: nothing to do */
	} else
	{
		/* pass 1: initialize */
		ro_aid = thisaid;

		/* set default mode for router: facet stitching */
		ro_state = makekey("ROUT_state");
		ro_prefered = makekey("ROUT_prefered_arc");
		nextvarchangequiet();
		(void)setvalkey((INTBIG)ro_aid, VAID, ro_state, STITCH, VINTEGER|VDONTSAVE);

		/* initialize the free lists */
		ro_rcheckfree = NORCHECK;
		ro_firstrcheck = NORCHECK;

		/* no last arc was created */
		ro_lastarc = NOARCINST;
	}
}

INTSML ro_set(INTSML count, char *par[])
{
	REGISTER INTBIG l;
	REGISTER NODEPROTO *np;
	REGISTER ARCPROTO *ap;
	REGISTER NODEINST *ni;
	REGISTER char *pp;
	REGISTER GEOM **list;
	REGISTER VARIABLE *var;

	if (count == 0)
	{
		var = getvalkey((INTBIG)ro_aid, VAID, VINTEGER, ro_state);
		if (var != NOVARIABLE)
		{
			switch (var->addr&STYLE)
			{
				case STITCH:
					ttyputmsg("Default is auto-stitch routing");   break;
				case MIMIC:
					ttyputmsg("Default is mimic-stitch routing");  break;
			}
		}
		var = getvalkey((INTBIG)ro_aid, VAID, VARCPROTO, ro_prefered);
		if (var != NOVARIABLE)
			ttyputmsg("Default arc for river routing is %s",
				describearcproto(((ARCPROTO *)var->addr)));
		return(0);
	}

	l = strlen(pp = par[0]);

	if (namesamen(pp, "auto-stitch", (INTSML)l) == 0)
	{
		(void)setvalkey((INTBIG)ro_aid, VAID, ro_state, STITCH, VINTEGER|VDONTSAVE);
		if (count <= 1) return(0);
		l = strlen(pp = par[1]);
		if (namesamen(pp, "stitch-now", (INTSML)l) == 0)
		{
			if (count >= 3)
			{
				np = getnodeproto(par[2]);
				if (np == NONODEPROTO)
				{
					ttyputerr("No facet named %s", par[2]);
					return(1);
				}
				if (np->index != 0)
				{
					ttyputerr("Can only stitch facets, not primitives");
					return(1);
				}
				if (np->cell->lib != el_curlib)
				{
					ttyputerr("Can only stitch facets in the current library");
					return(1);
				}
			} else
			{
				np = getcurfacet();
				if (np == NONODEPROTO)
				{
					ttyputerr("No current facet");
					return(1);
				}
			}

			for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
				ro_queuercheck(ni);
			ro_slice();
			ttyputmsgf("Facet %s stitched", describenodeproto(np));
			return(0);
		}
		if (namesamen(pp, "highlighted-stitch-now", (INTSML)l) == 0)
		{
			list = (GEOM **)askaid(us_aid, "get-all-nodes");
			if (list[0] == NOGEOM)
			{
				ttyputerr("Select an area to be stitched");
				return(1);
			}
			for(l=0; list[l] != NOGEOM; l++)
				ro_queuercheck(list[l]->entryaddr.ni);
			ro_slice();
			ttyputmsgf("Stitching complete");
			return(0);
		}
		ttyputerr("Bad AUTO-STITCH option: %s", pp);
		return(1);
	}

	if (namesamen(pp, "river-route", (INTSML)l) == 0)
	{
		/* quit if river routing not available */
		if (checkcap(ro_aid, ROUTHASRIVER) == 0)
		{
			ttyputerr("Sorry, river routing is not available");
			return(1);
		}

		if (count >= 2)
		{
			np = getnodeproto(par[1]);
			if (np == NONODEPROTO)
			{
				ttyputerr("No facet named %s", par[1]);
				return(1);
			}
			if (np->index != 0)
			{
				ttyputerr("Can only river-route facets, not primitives");
				return(1);
			}
		} else
		{
			np = getcurfacet();
			if (np == NONODEPROTO)
			{
				ttyputerr("No current facet");
				return(1);
			}
		}

		if (ro_river(np) == 0) ttyputmsg("Routing not successful"); else
			ttyputmsgf("Facet %s river-routed", describenodeproto(np));
		return(0);
	}

	if (namesamen(pp, "maze-route", (INTSML)l) == 0 && l > 1)
	{
		/* quit if river routing not available */
		if (checkcap(ro_aid, ROUTHASMAZE) == 0)
		{
			ttyputerr("Sorry, maze routing is not available");
			return(1);
		}

		l = strlen(pp = par[1]);
		if (namesamen(pp, "route", (INTSML)l) == 0)
		{
			if (count <= 2) return(0);
			l = strlen(pp = par[2]);
			if (namesamen(pp, "cell", (INTSML)l) == 0)
			{
				ro_mazeroutecell();
				return(0);
			}
			if (namesamen(pp, "selected", (INTSML)l) == 0)
			{
				ro_mazerouteselected();
				return(0);
			}
			ttyputerr("Bad maze-route option: %s", pp);
			return(1);
		}
		if (namesamen(pp, "set", (INTSML)l) == 0)
		{
			if (count <= 2)
			{
				ttyputmsg("Routing grid (%3.1f,%3.1f)", (double)ro_mazegridx/(double)sch_tech->deflambda,
					(double)ro_mazegridy/(double)sch_tech->deflambda);
				ttyputmsg("Routing offset (%3.1f,%3.1f)", (double)ro_mazeoffsetx/(double)sch_tech->deflambda,
					(double)ro_mazeoffsety/(double)sch_tech->deflambda);
				ttyputmsg("Routing boundary (%3.1f)", (double)ro_mazeboundary/(double)sch_tech->deflambda);
				return(0);
			}

			l = strlen(pp = par[2]);
			if (namesamen(pp, "grid", (INTSML)l) == 0)
			{
				ro_mazegridx = (INTBIG)(atof(par[3]) * sch_tech->deflambda);
				ro_mazegridy = (INTBIG)(atof(par[4]) * sch_tech->deflambda);
				ttyputmsg("Routing grid (%3.1f,%3.1f)", (double)ro_mazegridx/(double)sch_tech->deflambda,
					(double)ro_mazegridy/(double)sch_tech->deflambda);
				return 0;
			}
			if (namesamen(pp, "offset", (INTSML)l) == 0)
			{
				ro_mazeoffsetx = (INTBIG)(atof(par[3]) * sch_tech->deflambda);
				ro_mazeoffsety = (INTBIG)(atof(par[4]) * sch_tech->deflambda);
				ttyputmsg("Routing offset (%3.1f,%3.1f)", (double)ro_mazeoffsetx/(double)sch_tech->deflambda,
					(double)ro_mazeoffsety/(double)sch_tech->deflambda);
				return 0;
			}
			if (namesamen(pp, "boundary", (INTSML)l) == 0)
			{
				ro_mazeboundary = (INTBIG)(atof(par[3]) * sch_tech->deflambda);
				ttyputmsg("Routing boundary (%3.1f)", (double)ro_mazeboundary/(double)sch_tech->deflambda);
				return 0;
			}

			ttyputerr("Bad maze-route set option: %s", pp);
			return(1);
		}
	}

	if (namesamen(pp, "mimic-stitch", (INTSML)l) == 0 && l > 1)
	{
		(void)setvalkey((INTBIG)ro_aid, VAID, ro_state, MIMIC, VINTEGER|VDONTSAVE);
		return(0);
	}

	if (namesamen(pp, "use-arc", (INTSML)l) == 0)
	{
		if (count < 2)
		{
			ttyputerr("Usage: tellaid routing use-arc ARCPROTO");
			return(1);
		}
		if (strcmp(par[1], "*") == 0)
		{
			if (getvalkey((INTBIG)ro_aid, VAID, VARCPROTO, ro_prefered) != NOVARIABLE)
				(void)delvalkey((INTBIG)ro_aid, VAID, ro_prefered);
			ttyputmsgf("No arc will be presumed for stitching");
			return(0);
		}
		ap = getarcproto(par[1]);
		if (ap == NOARCPROTO)
		{
			ttyputerr("No arc prototype called %s", par[1]);
			return(1);
		}
		ttyputmsgf("Default stitching arc will be %s", describearcproto(ap));
		(void)setvalkey((INTBIG)ro_aid, VAID, ro_prefered, (INTBIG)ap, VARCPROTO|VDONTSAVE);
		return(0);
	}

	if (namesamen(pp, "unroute", (INTSML)l) == 0)
	{
		ro_unroutecurrent();
		return(0);
	}

	if (namesamen(pp, "not", (INTSML)l) == 0)
	{
		if (count <= 1)
		{
			ttyputerr("Usage: tellaid routing not OPTION");
			return(1);
		}
		l = strlen(pp = par[1]);
		if (namesamen(pp, "select", (INTSML)l) == 0)
		{
			var = getvalkey((INTBIG)ro_aid, VAID, VINTEGER, ro_state);
			if (var == NOVARIABLE) l = STITCH; else l = var->addr;
			(void)setvalkey((INTBIG)ro_aid, VAID, ro_state, l & ~SELECT, VINTEGER|VDONTSAVE);
			ttyputmsgf("Wire placement not subject to approval");
			return(0);
		}
		ttyputerr("Bad ROUTING NOT option: %s", pp);
		return(1);
	}

	if (namesamen(pp, "select", (INTSML)l) == 0)
	{
		var = getvalkey((INTBIG)ro_aid, VAID, VINTEGER, ro_state);
		if (var == NOVARIABLE) l = STITCH; else l = var->addr;
		(void)setvalkey((INTBIG)ro_aid, VAID, ro_state, l | SELECT, VINTEGER|VDONTSAVE);
		ttyputmsgf("Wire placement subject to approval");
		return(0);
	}

	ttyputerr("Bad ROUTING option: %s", pp);
	return(1);
}

/* inquire the routing aid */
INTBIG ro_request(char *command, va_list ap) { return(0); }

void ro_slice(void)
{
	REGISTER VARIABLE *var;
	REGISTER INTBIG state;

	/* zero the count of stitches made */
	ro_count = 0;

	var = getvalkey((INTBIG)ro_aid, VAID, VINTEGER, ro_state);
	if (var == NOVARIABLE) state = STITCH; else state = var->addr;
	if ((state&(SELSKIP|SELDONE)) != 0)
	{
		state &= ~(SELSKIP|SELDONE);
		(void)setvalkey((INTBIG)ro_aid, VAID, ro_state, state, VINTEGER|VDONTSAVE);
	}

	switch (state&STYLE)
	{
		case STITCH:
			if (checkcap(ro_aid, ROUTHASSTITCH) == 0)
			{
				ttyputerr("Sorry, stitching router is not available");
				aidturnoff(ro_aid, 0);
				return;
			}
			ro_autostitch();
			break;
		case MIMIC:
			if (checkcap(ro_aid, ROUTHASSTITCH) == 0)
			{
				ttyputerr("Sorry, mimic router is not available");
				aidturnoff(ro_aid, 0);
				return;
			}
			ro_mimicstitch();
			break;
	}
	(void)stopping("Routing");

	/* if any stitches were made, say so */
	if (ro_count != 0)
	{
		if (ro_count == 1) ttyputmsg("ROUTING: added 1 stitch"); else
			ttyputmsg("ROUTING: added %d stitches", ro_count);
	}

	/* set the message of what happened */
	setactivity("Facets stitched");
}

void ro_examinenodeproto(NODEPROTO *np) {}

void ro_done(void) {}

/******************** DATABASE CHANGES ********************/

void ro_startbatch(AIDENTRY *source)
{
	ro_source = source;
	ro_lastarc = NOARCINST;
}

void ro_endbatch() {}

void ro_startobjectchange(INTBIG addr, INTBIG type) {}

void ro_endobjectchange(INTBIG addr, INTBIG type) {}

void ro_modifynodeinst(NODEINST *ni, INTBIG oldlx, INTBIG oldly, INTBIG oldhx,
	INTBIG oldhy, INTSML oldrot, INTSML oldtran)
{
	if (ro_source != ro_aid) ro_queuercheck(ni);
	ro_lastarc = NOARCINST;
}

void ro_modifyarcinst(ARCINST *ai, INTBIG oldxA, INTBIG oldyA, INTBIG oldxB,
	INTBIG oldyB, INTBIG oldwid, INTBIG oldlen)
{
	ro_lastarc = NOARCINST;
}

void ro_modifyportproto(PORTPROTO *pp, NODEINST *oldsubni, PORTPROTO *oldsubpp)
{
	ro_lastarc = NOARCINST;
}

void ro_modifynodeproto(NODEPROTO *np)
{
	ro_lastarc = NOARCINST;
}

void ro_modifydescript(VARIABLE *var, INTBIG odes) {}

void ro_newobject(INTBIG addr, INTBIG type)
{
	REGISTER NODEINST *ni;
	REGISTER ARCINST *ai;

	if (ro_source == ro_aid) return;
	if ((type&VTYPE) == VNODEINST)
	{
		ni = (NODEINST *)addr;
		ro_queuercheck(ni);
		ro_lastarc = NOARCINST;
	} else if ((type&VTYPE) == VARCINST)
	{
		ai = (ARCINST *)addr;
		if (ro_lastarc != NOARCINST) ro_lastarc = NOARCINST; else
			ro_lastarc = ai;
	} else if ((type&VTYPE) == VPORTPROTO || (type&VTYPE) == VNODEPROTO)
	{
		ro_lastarc = NOARCINST;
	}
}

void ro_killobject(INTBIG type, INTBIG addr)
{
	if (ro_source == ro_aid) return;
	if ((type&VTYPE) == VNODEINST || (type&VTYPE) == VARCINST ||
		(type&VTYPE) == VPORTPROTO || (type&VTYPE) == VNODEPROTO)
			ro_lastarc = NOARCINST;
}

void ro_newvariable(INTBIG addr, INTBIG type, INTBIG key, INTBIG newtype) {}
void ro_killvariable(INTBIG addr, INTBIG type, INTBIG key, INTBIG oldaddr, INTBIG oldtype,
	INTBIG olddescript) {}
void ro_modifyvariable(INTBIG addr, INTBIG type, INTBIG key, INTBIG vartype, INTBIG index,
	INTBIG oldvalue) {}
void ro_insertvariable(INTBIG addr, INTBIG type, INTBIG key, INTBIG vartype, INTBIG index) {}
void ro_deletevariable(INTBIG addr, INTBIG type, INTBIG key, INTBIG vartype, INTBIG index,
	INTBIG oldvalue) {}
void ro_readlibrary(LIBRARY *lib) {}
void ro_eraselibrary(LIBRARY *lib) {}
void ro_writelibrary(LIBRARY *lib, INTSML pass) {}

/*********************** CODE TO UNROUTE (CONVERT TO UNROUTED WIRE) ***********************/

/* Routine to convert the current network(s) to an unrouted wire */
void ro_unroutecurrent(void)
{
	GEOM **list;
	REGISTER NETWORK *net;
	REGISTER NODEPROTO *np;
	REGISTER ARCINST *ai;
	REGISTER INTBIG i, selectcount;
	REGISTER INTSML found;

	/* see what is highlighted */
	list = us_gethighlighted(OBJARCINST);
	for(selectcount=0; list[selectcount] != NOGEOM; selectcount++) ;
	if (selectcount == 0)
	{
		ttyputerr("Must select arcs to unroute");
		return;
	}
	np = geomparent(list[0]);

	/* convert requested nets */
	us_clearhighlightcount();
	found = 1;
	while(found != 0)
	{
		/* no net found yet */
		found = 0;

		/* find a net to be routed */
		for(net = np->firstnetwork; net != NONETWORK;
			net = net->nextnetwork)
		{
			/* is the net included in the selection list? */
			for(i=0; i<selectcount; i++)
			{
				if (list[i] == NOGEOM) continue;
				if (list[i]->entrytype != OBJARCINST) continue;
				ai = list[i]->entryaddr.ai;
				if (ai->network == net) break;
			}
			if (i >= selectcount) continue;

			/* yes: remove all arcs that have this net from the selection */
			for(i=0; i<selectcount; i++)
			{
				if (list[i] == NOGEOM) continue;
				if (list[i]->entrytype != OBJARCINST) continue;
				ai = list[i]->entryaddr.ai;
				if (ai->network == net) list[i] = NOGEOM;
			}

			/* now unroute the net */
			if (ro_unroutenet(net) != 0) return;
			found = 1;
			break;
		}
	}
}

INTSML ro_unroutenet(NETWORK *net)
{
	INTBIG *xlist, *ylist, count;
	NODEINST **nilist;
	PORTPROTO **pplist;
	REGISTER ARCINST *ai, *nextai;
	REGISTER NODEINST *ni, *nextni;
	REGISTER NODEPROTO *np;
	REGISTER INTBIG bits;

	/* convert this net and mark arcs and nodes on it */
	count = ro_findnetends(net, &nilist, &pplist, &xlist, &ylist);
	if (count != 2)
	{
		ttyputerr("Network must have 2 ends: this has %d", count);
		return(0);
	}

	/* remove marked nodes and arcs */
	np = net->parent;
	for(ai = np->firstarcinst; ai != NOARCINST; ai = nextai)
	{
		nextai = ai->nextarcinst;
		if (ai->temp1 == 0) continue;
		startobjectchange((INTBIG)ai, VARCINST);
		if (killarcinst(ai))
		{
			ttyputerr("Error deleting arc");
			return(1);
		}
	}
	for(ni = np->firstnodeinst; ni != NONODEINST; ni = nextni)
	{
		nextni = ni->nextnodeinst;
		if (ni->temp1 == 0) continue;
		startobjectchange((INTBIG)ni, VNODEINST);
		if (killnodeinst(ni))
		{
			ttyputerr("Error deleting intermediate node");
			return(1);
		}
	}

	/* now create the new unrouted wire */
	bits = us_makearcuserbits(gen_unroutedarc);
	ai = newarcinst(gen_unroutedarc, gen_unroutedarc->nominalwidth, bits,
		nilist[0], pplist[0], xlist[0], ylist[0], nilist[1], pplist[1], xlist[1], ylist[1], np);
	if (ai == NOARCINST)
	{
		ttyputerr("Could not create unrouted arc");
		return(1);
	}
	endobjectchange((INTBIG)ai, VARCINST);
	(void)askaid(us_aid, "show-object", (INTBIG)ai->geom);
	return(0);
}

/*
 * Routine to find the endpoints of network "net" and store them in the array
 * "ni/pp/xp/yp".  Returns the number of nodes in the array.
 * As a side effect, sets "temp1" on nodes and arcs to nonzero if they are part
 * of the network.
 */
INTBIG ro_findnetends(NETWORK *net, NODEINST ***nilist, PORTPROTO ***pplist,
	INTBIG **xplist, INTBIG **yplist)
{
	REGISTER NODEPROTO *np;
	REGISTER NODEINST *ni;
	REGISTER ARCINST *ai;
	REGISTER PORTARCINST *pi, *thispi;
	REGISTER INTSML term;
	static INTBIG listtotal = 0;
	INTBIG listcount, newtotal, i, j;
	static NODEINST **listni;
	static PORTPROTO **listpp;
	static INTBIG *listx;
	static INTBIG *listy;
	NODEINST **newlistni;
	PORTPROTO **newlistpp;
	INTBIG *newlistx;
	INTBIG *newlisty;

	/* initialize */
	np = net->parent;
	listcount = 0;
	for(ai = np->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst) ai->temp1 = 0;
	for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst) ni->temp1 = 0;

	/* look at every arc and see if it is part of the network */
	for(ai = np->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
	{
		if (ai->network != net) continue;
		ai->temp1 = 1;

		/* see if an end of the arc is a network "end" */
		for(i=0; i<2; i++)
		{
			ni = ai->end[i].nodeinst;
			thispi = ai->end[i].portarcinst;
			term = 0;
			for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
				if (pi != thispi && pi->conarcinst->network == net) break;
			if (pi == NOPORTARCINST) term = 1;
			if (term != 0)
			{
				/* valid network end: add it to the list */
				if (listcount >= listtotal)
				{
					newtotal = listcount * 2;
					if (newtotal == 0) newtotal = 10;
					newlistni = (NODEINST **)emalloc(newtotal * (sizeof (NODEINST *)),
						ro_aid->cluster);
					newlistpp = (PORTPROTO **)emalloc(newtotal * (sizeof (PORTPROTO *)),
						ro_aid->cluster);
					newlistx = (INTBIG *)emalloc(newtotal * (sizeof (INTBIG)),
						ro_aid->cluster);
					newlisty = (INTBIG *)emalloc(newtotal * (sizeof (INTBIG)),
						ro_aid->cluster);
					for(j=0; j<listcount; j++)
					{
						newlistni[j] = listni[j];
						newlistpp[j] = listpp[j];
						newlistx[j] = listx[j];
						newlisty[j] = listy[j];
					}
					if (listtotal > 0)
					{
						efree((char *)listni);
						efree((char *)listpp);
						efree((char *)listx);
						efree((char *)listy);
					}
					listni = newlistni;
					listpp = newlistpp;
					listx = newlistx;
					listy = newlisty;
					listtotal = newtotal;
				}
				listni[listcount] = ni;
				listpp[listcount] = thispi->proto;
				listx[listcount] = ai->end[i].xpos;
				listy[listcount] = ai->end[i].ypos;
				listcount++;
			} else
			{
				/* not a network end: mark the node for removal */
				ni->temp1 = 1;
			}
		}
	}
	*nilist = listni;
	*pplist = listpp;
	*xplist = listx;
	*yplist = listy;
	return(listcount);
}

#endif  /* ROUTAID - at top */
