/*
 * Electric(tm) VLSI Design System
 *
 * File: routmimic.c
 * Arc mimic code for the wire routing aid
 * Written by: Brian W. Thomson, University of Toronto
 *
 * Copyright (c) 1998 Electric Editor Incorporated.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Electric Editor Incorporated
 * 23470 Sunset Drive, Suite 108
 * Los Gatos, California 95033
 * support@electriceditor.com
 */

#include "config.h"
#if ROUTAID

#include "global.h"
#include "rout.h"

struct portpos
{
	NODEINST  *node;
	PORTPROTO *port;
	INTBIG     lx, hx, ly, hy;
};

/* prototypes for local routines */
PORTPROTO *ro_findport(NODEINST*, PORTPROTO*, ARCPROTO*);
struct portpos *ro_portposlist(NODEINST*, ARCINST*);
void ro_portbounds(NODEINST*, PORTPROTO*, INTBIG*, INTBIG*, INTBIG*, INTBIG*);

void ro_mimicstitch(void)
{
	REGISTER ARCINST *arc;
	REGISTER NODEINST *n, *n1, *n2;
	NODEPROTO *endfacet[2], *facet;
	INTSML samenode;
	INTBIG flx, fhx, fly, fhy, tlx, thx, tly, thy, fx, tx, fy, ty,
		fxoff, fyoff, txoff, tyoff, xp, yp;
	struct portpos *n1pos, *n2pos;
	REGISTER PORTARCINST *pi;
	REGISTER ARCINST *ai;
	REGISTER struct portpos *pp1, *pp2;

	/* get the single arc that was just created */
	arc = ro_lastarc;
	if (arc == NOARCINST) return;
	ro_lastarc = NOARCINST;

	/* Currently only handle single new arc */
	facet = arc->parent;
	endfacet[0] = arc->end[0].nodeinst->proto;
	endfacet[1] = arc->end[1].nodeinst->proto;
	samenode = arc->end[0].nodeinst == arc->end[1].nodeinst;
	n1 = NONODEINST;   n2 = NONODEINST;
	for(n = facet->firstnodeinst; n != NONODEINST; n = n->nextnodeinst)
	{
		if (n->proto == endfacet[0])
		{
			n->temp2 = (INTBIG)n1;
			n1 = n;
		} else if (n->proto == endfacet[1])
		{
			n->temp2 = (INTBIG)n2;
			n2 = n;
		}
	}

	/* combine lists if source and dest are same type */
	if (endfacet[1] == endfacet[0]) n2 = n1;

	n1pos = ro_portposlist(n1, arc);
	if (n1pos == (struct portpos *)0) return;
	n2pos = ro_portposlist(n2, arc);
	if (n2pos == (struct portpos *)0)
	{
		efree((char *)n1pos);
		return;
	}

	/* find offset of arc end and port center */
	portposition(arc->end[0].nodeinst, arc->end[0].portarcinst->proto, &xp, &yp);
	fxoff = arc->end[0].xpos - xp;
	fyoff = arc->end[0].ypos - yp;
	portposition(arc->end[1].nodeinst, arc->end[1].portarcinst->proto, &xp, &yp);
	txoff = arc->end[1].xpos - xp;
	tyoff = arc->end[1].ypos - yp;

	/* For each appropriate port on each n1 connect it to an n2 */
	for(pp1 = n1pos; pp1->node != NONODEINST; pp1++)
	{
		if (stopping("Routing") != 0) break;
		flx = pp1->lx;   fhx = pp1->hx;
		fly = pp1->ly;   fhy = pp1->hy;
		for(pp2 = n2pos; pp2->node != NONODEINST; pp2++)
		{
			if (stopping("Routing") != 0) break;
			if (pp2->port == NOPORTPROTO) continue;
			if ((pp1->node == pp2->node) != samenode) continue;

			/* look for an arc that already makes this connection */
			for(pi = pp1->node->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
			{
				ai = pi->conarcinst;
				if (ai->end[0].nodeinst == pp2->node && ai->end[0].portarcinst->proto == pp2->port)
					break;
				if (ai->end[1].nodeinst == pp2->node && ai->end[1].portarcinst->proto == pp2->port)
					break;
			}
			if (pi != NOPORTARCINST) continue;

			tlx = pp2->lx;   thx = pp2->hx;
			tly = pp2->ly;   thy = pp2->hy;

			/* see if arc is null length */
			if (arc->end[0].xpos == arc->end[1].xpos && arc->end[0].ypos == arc->end[1].ypos)
			{
				fx = (flx+fhx) / 2;   fy = (fly+fhy) / 2;
				tx = (tlx+thx) / 2;   ty = (tly+thy) / 2;
			} else
			{
				if (arcconnects((INTSML)(((arc->userbits&AANGLE) >> AANGLESH) * 10),
					flx, fhx, fly, fhy, tlx, thx, tly, thy, &fx, &fy, &tx, &ty) == 0) continue;
			}

			/*
			 * Make sure we aren't connecting to a port
			 * coincident with one of the sample ports!
			 */
			if ((fx+fxoff == arc->end[0].xpos && fy+fyoff == arc->end[0].ypos) ||
				(tx+txoff == arc->end[1].xpos && ty+tyoff == arc->end[1].ypos)) continue;

			/*
			 * We should now be prepared to create an
			 * arc of the same length as the original.
			 * If not, we have chosen ports poorly.
			 */
			if (abs(tx-fx) != abs((arc->end[1].xpos-txoff)-(arc->end[0].xpos-fxoff))) continue;
			if (abs(ty-fy) != abs((arc->end[1].ypos-tyoff)-(arc->end[0].ypos-fyoff))) continue;

			/* create the arc */
			ai = newarcinst(arc->proto, arc->width, arc->userbits, pp1->node,
				pp1->port, fx, fy, pp2->node,  pp2->port, tx, ty, facet);
			if (ai == NOARCINST) ttyputerr("Problem creating arc"); else
			{
				pp2->port = NOPORTPROTO;
				(void)ro_didaccept(ai, NOARCINST, NOARCINST);
				endobjectchange((INTBIG)ai, VARCINST);
			}
		}
	}
	efree((char *)n1pos);
	efree((char *)n2pos);
}

/*
 * Search nodeinst 'ni' for a port that can connect to an arc of type 'at'.
 * Returns the found port or returns NOPORTPROTO if search fails.
 * If 'pp' is not NOPORTPROTO, then begin searching only after
 * encountering port 'pp' in the list of ports on node 'ni'.
 */
PORTPROTO *ro_findport(NODEINST *ni, PORTPROTO *pp, ARCPROTO *at)
{
	REGISTER PORTPROTO *mypt;
	REGISTER INTSML i;

	for(mypt = ni->proto->firstportproto; mypt != NOPORTPROTO; mypt = mypt->nextportproto)
	{
		if (pp != NOPORTPROTO)
		{
			if (pp == mypt) pp = NOPORTPROTO;
			continue;
		}
		for(i=0; mypt->connects[i] != NOARCPROTO; i++)
			if (mypt->connects[i] == at) return(mypt);
	}
	return(NOPORTPROTO);
}

/*
 * Find bounds of all possible connecting ports on nodes in "nodes" list.
 * Remove multiple coincident ports from lists, so we don't make more than
 * one connection per net.  We do this for efficiency's sake, so we only call
 * ro_portbounds() once for each port we investigate.
 */
struct portpos *ro_portposlist(NODEINST *nodes, ARCINST *arc)
{
	REGISTER NODEINST *n;
	REGISTER PORTPROTO *po;
	struct portpos *pp;
	REGISTER struct portpos *ppos;
	INTSML i;
	INTBIG lx, hx, ly, hy;

	i = 1;
	for(n = nodes; n != NONODEINST; n = (NODEINST *)n->temp2)
	{
		po = NOPORTPROTO;
		for(;;)
		{
			po = ro_findport(n, po, arc->proto);
			if (po == NOPORTPROTO) break;
			i++;
		}
	}

	pp = (struct portpos *)emalloc((i * sizeof(struct portpos)), el_tempcluster);
	if (pp == (struct portpos *)0)
	{
		ttyputerr("routmimic: No memory");
		return(pp);
	}

	/*
	 * We have arranged that the pp array is at least
	 * one entry longer than necessary, so we can use
	 * a pp->node value of NONODEINST as a terminator and
	 * it will never be overwritten.
	 */
	for(ppos = pp; i; i--, ppos++) ppos->node = NONODEINST;

	i = 0;
	for(n = nodes; n != NONODEINST; n = (NODEINST *)n->temp2)
	{
		po = NOPORTPROTO;
		for(;;)
		{
			po = ro_findport(n, po, arc->proto);
			if (po == NOPORTPROTO) break;
			ro_portbounds(n, po, &lx, &hx, &ly, &hy);
			for(ppos = pp; ; ppos++)
			{
				if (ppos->node == NONODEINST)
				{
					ppos->port = po;
					ppos->node = n;
					ppos->lx = lx;
					ppos->hx = hx;
					ppos->ly = ly;
					ppos->hy = hy;
					i++;
					break;
				}
				if (ppos->lx == lx && ppos->hx == hx && ppos->ly == ly && ppos->hy == hy &&
					ppos->node == n) break;
			}
		}
	}
	return(pp);
}

void ro_portbounds(NODEINST *ni, PORTPROTO *pt, INTBIG *lxp, INTBIG *hxp, INTBIG *lyp, INTBIG *hyp)
{
	static POLYGON *poly = NOPOLYGON;

	if (poly == NOPOLYGON) poly = allocpolygon(4, ro_aid->cluster);

	shapeportpoly(ni, pt, poly, 0);
	getbbox(poly, lxp, hxp, lyp, hyp);
}

#endif  /* ROUTAID - at top */
