/*
 * Electric(tm) VLSI Design System
 *
 * File: sc1electric.c
 * Modules for the QUISC Silicon Compiler to interface it with Electric
 * Written by: Andrew R. Kostiuk, Queen's University
 *
 * Copyright (c) 1998 Electric Editor Incorporated.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Electric Editor Incorporated
 * 23470 Sunset Drive, Suite 108
 * Los Gatos, California 95033
 * support@electriceditor.com
 */

#include "config.h"
#if SCAID

#include "global.h"
#include "efunction.h"
#include "sc1.h"
#include "usr.h"

/***********************************************************************
	Global Variables
------------------------------------------------------------------------
*/

AIDENTRY         *sc_aid;	/* the Silicon Compiler aid object */
static INTBIG     sc_simkey;	/* key for the ALS info */
static INTBIG     sc_silkey;	/* key for the silos information */
static INTBIG     sc_bitskey;
static INTBIG     sc_numskey;
static NODEPROTO *layer1_proto = NONODEPROTO;
static NODEPROTO *layer2_proto = NONODEPROTO;
static NODEPROTO *via_proto = NONODEPROTO;
static NODEPROTO *pwell_proto = NONODEPROTO;
static ARCPROTO	 *layer1_arc = NOARCPROTO;
static ARCPROTO	 *layer2_arc = NOARCPROTO;

extern AIDENTRY  *us_aid;
extern char      *sc_errmsg;
extern int       sc_sim_format;
extern SCMAKERCONTROL maker_control;

/* prototypes for local routines */
int Sc_redraw_graphics(void);
void Sc_set_leaf_port_type(PORTPROTO*, int);

/***********************************************************************
Module:  sc_init
------------------------------------------------------------------------
Description:
	Initialize routine for the Silicon Compiler aid.
------------------------------------------------------------------------
*/

void sc_init(INTBIG *argc, char *argv[], AIDENTRY *thisaid)
{
	/* only initialize during pass 1 */
	if (thisaid == NOAID || thisaid == 0) return;
	sc_aid = thisaid;
	Sc_initialize();
	if ((sc_simkey = makekey("SC_sim")) == -1) {
		ttyputmsg("ERROR making SIM key in SC Aid");
	}
	if ((sc_bitskey = makekey("SC_bits")) == -1) {
		ttyputmsg("ERROR making BITS key in SC Aid");
	}
	if ((sc_numskey = makekey("SC_nums")) == -1) {
		ttyputmsg("ERROR making NUMS key in SC Aid");
	}
	if ((sc_silkey = makekey("SC_silos")) == -1 ) {
		ttyputmsg("ERROR making SILOS key in SC Aid.");
	}
}

/***********************************************************************
Module:  sc_slice
------------------------------------------------------------------------
Description:
	Main module of the Silicon Compiler aid.  Invoked by the onaid
	command.
------------------------------------------------------------------------
*/

void sc_slice(void)
{
	el_pleasestop = 0;

	/* quit if silicon compiler not available */
	if (checkcap(sc_aid, SILCOMPAVAIL) == 0)
		ttyputerr("Sorry, silicon compiler is not available"); else
			Sc_main();
	aidturnoff(sc_aid, 0);
}

/***********************************************************************
Module:  sc_set
------------------------------------------------------------------------
Description:
	Module invoked by the tellaid command.
------------------------------------------------------------------------
*/

INTSML sc_set(INTSML count, char *pars[])
{
	/* quit if silicon compiler not available */
	if (checkcap(sc_aid, SILCOMPAVAIL) == 0)
		ttyputerr("Sorry, silicon compiler is not available"); else
			Sc_one_command(count, pars);
	return(1);
}

/***********************************************************************
Default Silicon Compiler aid routines.
------------------------------------------------------------------------
*/

INTBIG sc_request(char *command, va_list ap) { return(0); }
void sc_examinenodeproto(NODEPROTO *np) {}
void sc_done(void) {}
void sc_startbatch(AIDENTRY *source) {}
void sc_endbatch(void) {}
void sc_startobjectchange(INTBIG addr, INTBIG type) {}
void sc_endobjectchange(INTBIG addr, INTBIG type) {}
void sc_modifynodeinst(NODEINST *ni, INTBIG olx, INTBIG oly, INTBIG ohx, INTBIG ohy,
	INTSML orot, INTSML otran) {}
void sc_modifyarcinst(ARCINST *ai, INTBIG oxA, INTBIG oyA, INTBIG oxB, INTBIG oyB,
	INTBIG owid, INTBIG olen) {}
void sc_modifyportproto(PORTPROTO *pp, NODEINST *oni, PORTPROTO *opp) {}
void sc_modifynodeproto(NODEPROTO *np) {}
void sc_modifydescript(VARIABLE *var, INTBIG odes) {}
void sc_newobject(INTBIG addr, INTBIG type) {}
void sc_killobject(INTBIG addr, INTBIG type) {}
void sc_newvariable(INTBIG addr, INTBIG type, INTBIG key, INTBIG newtype) {}
void sc_killvariable(INTBIG addr, INTBIG type, INTBIG key, INTBIG oldaddr, INTBIG oldtype,
	INTBIG olddescript) {}
void sc_modifyvariable(INTBIG addr, INTBIG type, INTBIG key, INTBIG vartype, INTBIG index,
	INTBIG oldvalue) {}
void sc_insertvariable(INTBIG addr, INTBIG type, INTBIG key, INTBIG vartype, INTBIG index) {}
void sc_deletevariable(INTBIG addr, INTBIG type, INTBIG key, INTBIG vartype, INTBIG index,
	INTBIG oldvalue) {}
void sc_readlibrary(LIBRARY *lib) {}
void sc_eraselibrary(LIBRARY *lib) {}
void sc_writelibrary(LIBRARY *lib, INTSML pass) {}

/***********************************************************************
Module:  Sc_clear
------------------------------------------------------------------------
Description:
	Clear the status terminal screen.
------------------------------------------------------------------------
Calling Sequence:  Sc_clear();
------------------------------------------------------------------------
*/

void Sc_clear(void)
{
}

/***********************************************************************
Module:  Sc_redraw_graphics
------------------------------------------------------------------------
Description:
	Clear the graphics terminal screen.
------------------------------------------------------------------------
Calling Sequence:  Sc_redraw_graphics();
------------------------------------------------------------------------
*/

int Sc_redraw_graphics(void)
{
	char	*params[PARS] ;

	params[0] = "*" ;
	us_redraw(1, params) ;
	return(0);
}

/***********************************************************************
Module:  Sc_stop and Sc_clear_stop
------------------------------------------------------------------------
Description:
	Sc_stop returns TRUE if the stopping condition is TRUE.  Sc_clear_stop
	clears the stopping condition.
------------------------------------------------------------------------
*/

int Sc_stop(void)
{
	if (stopping("VHDL compiler")) {
		return(TRUE);
	}
	else {
		return(FALSE);
	}
}

void Sc_clear_stop(void)
{
	el_pleasestop = 0;
}

/***********************************************************************
Module:  Sc_tell_environment
------------------------------------------------------------------------
Description:
	Send a command to the operating environment (shell).
------------------------------------------------------------------------
Calling Sequence:  Sc_tell_environment(count, pars);

Name		Type		Description
----		----		-----------
count		int		Number of parameters.
pars		*char[]		Array of pointers to parameters.
------------------------------------------------------------------------
*/

void Sc_tell_environment(int count, char *pars[])
{
	(void)tellaid(us_aid, (INTSML)count, pars);
	return;
}

/***********************************************************************
Module:  Sc_system
------------------------------------------------------------------------
Description:
	Send a command to the operating system.
------------------------------------------------------------------------
Calling Sequence:  Sc_system(count, pars);

Name		Type		Description
----		----		-----------
count		int		Number of parameters.
pars		*char[]		Array of pointers to parameters.
------------------------------------------------------------------------
*/

void Sc_system(int count, char *pars[])
{
	us_system((INTSML)count, pars);
}

/***********************************************************************
Module:  Scmalloc
------------------------------------------------------------------------
Description:
	Allocate memory of the indicated size from the system resources.
------------------------------------------------------------------------
Calling Sequence:  space = Scmalloc(size);

Name		Type		Description
----		----		-----------
size		int		Number of bytes to allocate.
space		*char		Pointer to allocated space, NULL if
								allocation error occurs.
------------------------------------------------------------------------
*/

char  *Scmalloc(int size)
{
	return ((char *)emalloc(size, sc_aid->cluster));
}

/***********************************************************************
Module:  Scfree
------------------------------------------------------------------------
Description:
	Free the memory at the given address back to the system resources.
------------------------------------------------------------------------
Calling Sequence:  Scfree(addr);

Name		Type		Description
----		----		-----------
addr		*char		Address of block of memory to free.
------------------------------------------------------------------------
*/

void Scfree(char *addr)
{
	efree(addr);
}

/***********************************************************************
Module:  Scallocstring
------------------------------------------------------------------------
Description:
	Allocate memory for a copy of the string passed and set the pointer
	to that memory.  Return !NULL if allocation error.
------------------------------------------------------------------------
Calling Sequence:  error = Scallocstring(addr, str);

Name		Type		Description
----		----		-----------
addr		**char		Address of pointer to result copy.
str		*char		Pointer to string to be copied.
error		int		!NULL if error occurs.
------------------------------------------------------------------------
*/

int Scallocstring(char **addr, char *str)
{
	char	*temp;

	if ((temp = (char *)emalloc(strlen(str) + 1, sc_aid->cluster)) == NULL) {
		return(1);
	}
	else {
		(void)strcpy(temp, str);
		*addr = temp;
		return(0);
	}
}

/***********************************************************************

		D A T A B A S E     I N T E R F A C I N G     M O D U L E S

------------------------------------------------------------------------
*/

/***********************************************************************
Module:  Sc_find_leaf_facet
------------------------------------------------------------------------
Description:
	Return a pointer to the named leaf facet.  If facet is not found,
	return NULL.
------------------------------------------------------------------------
Calling Sequence:  leaffacet = Sc_find_leaf_facet(name);

Name		Type		Description
----		----		-----------
name		*char		String name of leaf facet.
leaffacet	*char		Generic pointer to found facet,
								NULL if not found.
------------------------------------------------------------------------
*/

char  *Sc_find_leaf_facet(char *name)
{
	NODEPROTO	*np;

	np = getnodeproto(name);
	np = layoutview(np);
	if (np != NONODEPROTO && np->index == 0)
		return((char *)np);
	return((char *)NULL);
}

/***********************************************************************
Module:  Sc_leaf_facet_name
------------------------------------------------------------------------
Description:
	Return a pointer to the name of a leaf facet.
------------------------------------------------------------------------
Calling Sequence:  leaffacetname = Sc_leaf_facet_name(leaffacet);

Name		Type		Description
----		----		-----------
leaffacet	*char		Pointer to the leaf facet.
leaffacetname	*char		String of leaf facet name.
------------------------------------------------------------------------
*/

char  *Sc_leaf_facet_name(char *leaffacet)
{
	NODEPROTO *np;

	np = (NODEPROTO *)leaffacet;
	if (np->index != 0) return(np->primname);
	return(np->cell->cellname);
}

/***********************************************************************
Module:  Sc_first_leaf_facet
------------------------------------------------------------------------
Description:
	Return a pointer to the first leaf facet in the current library.
	Return NULL if error, eg. no library, no leaf facets, etc.
------------------------------------------------------------------------
Calling Sequence:  leaffacet = Sc_first_leaf_facet();

Name		Type		Description
----		----		-----------
leaffacet	*char		Pointer to the first leaf facet, NULL if
								no leaf facet found.
------------------------------------------------------------------------
*/

char  *Sc_first_leaf_facet(void)
{
	if (el_curlib == NOLIBRARY) {
		return((char *)NULL);
	}
	if (el_curlib->firstnodeproto == NONODEPROTO) {
		return((char *)NULL);
	}
	else {
		return((char *)el_curlib->firstnodeproto);
	}
}

/***********************************************************************
Module:  Sc_next_leaf_facet
------------------------------------------------------------------------
Description:
	Return a pointer to the next leaf facet in the current library.
	Return NULL when end of list is reached.
------------------------------------------------------------------------
Calling Sequence:  nextleaffacet = Sc_next_leaf_facet(leaffacet);

Name		Type		Description
----		----		-----------
leaffacet	*char		Pointer to the current leaf facet.
nextleaffacet	*char		Returned pointer to next leaf facet.
------------------------------------------------------------------------
*/

char  *Sc_next_leaf_facet(char *leaffacet)
{
	if (((NODEPROTO *)leaffacet)->nextnodeproto == NONODEPROTO)
		return((char *)NULL);
	return((char *)(((NODEPROTO *)leaffacet)->nextnodeproto));
}

/***********************************************************************
Module:  Sc_leaf_facet_bits
------------------------------------------------------------------------
Description:
	Return the value of the bits field of the leaf facet for the
	Silicon Compiler.
------------------------------------------------------------------------
Calling Sequence:  bits = Sc_leaf_facet_bits(leaffacet);

Name		Type		Description
----		----		-----------
leaffacet	*char		Pointer to the leaf facet.
bits		int		Value of the SC bit field.
------------------------------------------------------------------------
*/

int Sc_leaf_facet_bits(char *leaffacet)
{
	VARIABLE	*var;

	/* check if variable exits */
	if ((var = getvalkey((INTBIG)leaffacet, VNODEPROTO, VINTEGER, sc_bitskey))
		== NOVARIABLE)
	{
		if (setvalkey((INTBIG)leaffacet, VNODEPROTO, sc_bitskey, 0, VINTEGER) ==
			NOVARIABLE) ttyputmsg("ERROR creating SCBITS variable.");
		return(0);
	}
	return(var->addr);
}

/***********************************************************************
Module:  Sc_leaf_facet_bits_address
------------------------------------------------------------------------
Description:
	Return the address of the bits field of the leaf facet for the
	Silicon Compiler.
------------------------------------------------------------------------
Calling Sequence:  bits_addr = Sc_leaf_facet_bits_address(leaffacet);

Name		Type		Description
----		----		-----------
leaffacet	*char		Pointer to the leaf facet.
bits_addr	*int		Address of the SC bits field.
------------------------------------------------------------------------
*/

int  *Sc_leaf_facet_bits_address(char *leaffacet)
{
	VARIABLE	*var;
	static int retval;

	/* check if variable exits */
	if ((var = getvalkey((INTBIG)leaffacet, VNODEPROTO, VINTEGER, sc_bitskey))
		== NOVARIABLE)
	{
		if (setvalkey((INTBIG)leaffacet, VNODEPROTO, sc_bitskey, 0, VINTEGER) ==
			NOVARIABLE) ttyputmsg("ERROR creating SCBITS variable.");
		if ((var = getvalkey((INTBIG)leaffacet, VNODEPROTO, VINTEGER,
			sc_bitskey)) == NOVARIABLE)
		{
			ttyputmsg("ERROR creating SCBITS variable.");
			return((int *)NULL);
		}
	}
	retval = var->addr;
	return(&retval);
}

/***********************************************************************
Module:  Sc_leaf_facet_xsize
------------------------------------------------------------------------
Description:
	Return the size in the x direction for the indicated leaf facet.
------------------------------------------------------------------------
Calling Sequence:  xsize = Sc_leaf_facet_xsize(leaffacet);

Name		Type		Description
----		----		-----------
leaffacet	*char		Pointer to the leaf facet.
xsize		int		Returned size in x direction.
------------------------------------------------------------------------
*/

int Sc_leaf_facet_xsize(char *leaffacet)
{
	return(((NODEPROTO *)leaffacet)->highx - ((NODEPROTO *)leaffacet)->lowx);
}

/***********************************************************************
Module:  Sc_leaf_facet_ysize
------------------------------------------------------------------------
Description:
	Return the size in the y direction for the indicated leaf facet.
------------------------------------------------------------------------
Calling Sequence:  xsize = Sc_leaf_facet_ysize(leaffacet);

Name		Type		Description
----		----		-----------
leaffacet	*char		Pointer to the leaf facet.
xsize		int		Returned size in y direction.
------------------------------------------------------------------------
*/

int Sc_leaf_facet_ysize(char *leaffacet)
{
	return(((NODEPROTO *)leaffacet)->highy - ((NODEPROTO *)leaffacet)->lowy);
}

/***********************************************************************
Module:  Sc_leaf_facet_sim_info
------------------------------------------------------------------------
Description:
	Return an array of pointers to strings for the leaf facet's
	simulation information.  Return NULL on error.  Note that the
	array should be terminated by a NULL.
------------------------------------------------------------------------
Calling Sequence:  simlist = Sc_leaf_facet_sim_info(leaffacet);

Name		Type		Description
----		----		-----------
leaffacet	*char		Pointer to the leaf facet.
simlist		**char		Array of pointers to simulation info,
								NULL on error.
------------------------------------------------------------------------
*/

char  **Sc_leaf_facet_sim_info(char *leaffacet)
{
	VARIABLE	*simvar;
	char	**simlist, **simlist2;
	int		i, numsim, use_key;

	if (sc_sim_format == SC_ALS_FORMAT)
		use_key = sc_simkey;
	else if (sc_sim_format == SC_SILOS_FORMAT)
		use_key = sc_silkey;
	if ((simvar = getvalkey((INTBIG)leaffacet, VNODEPROTO, VSTRING | VISARRAY,
		use_key)) == NOVARIABLE) return((char **)NULL);
	simlist2 = (char **)simvar->addr;

	/* count the number of simulation lines */
	numsim = 0;
	for (i = 0; simlist2[i] != NOSTRING; i++) numsim++;

	/* create array of pointers */
	if ((simlist = (char **)emalloc(sizeof(char *)*(numsim+1),sc_aid->cluster))
		== 0) return((char **)NULL);
	for (i = 0; i < numsim; i++) simlist[i] = simlist2[i];
	simlist[numsim] = NULL;
	return(simlist);
}

/***********************************************************************
Module:  Sc_leaf_facet_set_sim
------------------------------------------------------------------------
Description:
	Set the passed simulation information to the indicated leaf facet.
------------------------------------------------------------------------
Calling Sequence:  err = Sc_leaf_facet_set_sim(simline, leaffacet);

Name		Type		Description
----		----		-----------
simline		*SCSIM		Lines of simulation information.
leaffacet	*char		Pointer to the leaf facet.
err		int		Returned error code, 0 = no error.
------------------------------------------------------------------------
*/

int Sc_leaf_facet_set_sim(SCSIM *simline, char *leaffacet)
{
	char	**simlist;
	SCSIM	*nextsim;
	int		i, numsim, use_key;

	if (sc_sim_format == SC_ALS_FORMAT)
		use_key = sc_simkey;
	else if (sc_sim_format == SC_SILOS_FORMAT)
		use_key = sc_silkey;
	/* count the number of simulation lines */
	numsim = 0;
	for (nextsim = simline; nextsim; nextsim = nextsim->next) numsim++;

	/* create array of pointers */
	if ((simlist = (char **)emalloc(sizeof(char *)*(numsim+1),sc_aid->cluster))
		== 0) {
		return(Sc_seterrmsg(SC_NOMEMORY));
	}
	for (i = 0; i < numsim; i++)
	{
		simlist[i] = simline->model;
		simline = simline->next;
	}
	simlist[numsim] = NOSTRING;

	/* set variable */
	if (setvalkey((INTBIG)leaffacet, VNODEPROTO, use_key, (INTBIG)simlist,
		VSTRING | VISARRAY) == NOVARIABLE) return(Sc_seterrmsg(SC_SIMXSETVAL));
	return(0);
}

/***********************************************************************
Module:  Sc_leaf_facet_get_nums
------------------------------------------------------------------------
Description:
	Fill in the facet_nums structure for the indicated leaf facet.
------------------------------------------------------------------------
Calling Sequence:  Sc_leaf_facet_get_nums(leaffacet, nums);

Name		Type		Description
----		----		-----------
leaffacet	*char		Pointer to the leaf facet.
nums		*SCCELLNUMS	Address of structure to fill.
------------------------------------------------------------------------
*/

void Sc_leaf_facet_get_nums(char *leaffacet, SCCELLNUMS *nums)
{
	VARIABLE	*var;
	int		i, j, *iarray, *jarray;

	i = sizeof(SCCELLNUMS) / sizeof(int);

	/* check if variable exits */
	var = getvalkey((INTBIG)leaffacet, VNODEPROTO, VINTEGER|VISARRAY, sc_numskey);
	if (var == NOVARIABLE)
	{
		iarray = (int *)nums;
		for (j = 0; j < i; j++) iarray[j] = 0;
		return;
	}

	iarray = (int *)nums;
	jarray = (int *)var->addr;
	for (j = 0; j < i; j++) iarray[j] = jarray[j];
}

/***********************************************************************
Module:  Sc_leaf_facet_set_nums
------------------------------------------------------------------------
Description:
	Set the facet_nums variable for the indicated leaf facet.
------------------------------------------------------------------------
Calling Sequence:  err = Sc_leaf_facet_set_nums(leaffacet, nums);

Name		Type		Description
----		----		-----------
leaffacet	*char		Pointer to the leaf facet.
nums		*SCCELLNUMS	Pointer to facet nums structure.
err		int		Returned error code, 0 = no error.
------------------------------------------------------------------------
*/

int Sc_leaf_facet_set_nums(char *leaffacet, SCCELLNUMS *nums)
{
	VARIABLE	*var;
	int		i, j, *iarray;
	INTBIG   *jarray;

	i = sizeof(SCCELLNUMS) / sizeof(int);

	/* check if variable exits */
	var = getvalkey((INTBIG)leaffacet, VNODEPROTO, VINTEGER|VISARRAY, sc_numskey);
	if (var == NOVARIABLE)
	{
		if ((jarray = emalloc((i + 1) * sizeof(INTBIG), sc_aid->cluster)) == 0)
			return(Sc_seterrmsg(SC_NOMEMORY));
		iarray = (int *)nums;
		for (j = 0; j < i; j++) jarray[j] = iarray[j];
		jarray[j] = -1;
		if (setvalkey((INTBIG)leaffacet, VNODEPROTO, sc_numskey, (INTBIG)jarray,
			VINTEGER | VISARRAY) == NOVARIABLE)
				return(Sc_seterrmsg(SC_NOSET_CELL_NUMS));
		return(SC_NOERROR);
	}
	iarray = (int *)nums;
	jarray = (INTBIG *)var->addr;
	for (j = 0; j < i; j++) jarray[j] = iarray[j];
	return(SC_NOERROR);
}

/***********************************************************************
Module:  Sc_find_leaf_port
------------------------------------------------------------------------
Description:
	Return a pointer to the named port on the facet.  If no port found,
	return NULL.
------------------------------------------------------------------------
Calling Sequence:  leafport = Sc_find_leaf_port(leaffacet, portname);

Name		Type		Description
----		----		-----------
leaffacet	*char		Pointer to leaf facet.
portname	*char		String name of port.
leafport	*char		Generic pointer to first port,
								NULL if not found.
------------------------------------------------------------------------
*/

char  *Sc_find_leaf_port(char *leaffacet, char *portname)
{
	PORTPROTO	*pp;

	for (pp = ((NODEPROTO *)leaffacet)->firstportproto; pp != NOPORTPROTO;
		pp = pp->nextportproto)
	{
		if (namesame(pp->protoname, portname) == 0)
			return((char *)pp);
	}
	return((char *)NULL);
}

/***********************************************************************
Module:  Sc_leaf_port_name
------------------------------------------------------------------------
Description:
	Return a pointer to the name of a leaf port.
------------------------------------------------------------------------
Calling Sequence:  portname = Sc_leaf_port_name(leafport);

Name		Type		Description
----		----		-----------
leafport	*char		Pointer to port.
portname	*char		String of port name.
------------------------------------------------------------------------
*/

char  *Sc_leaf_port_name(char *leafport)
{
	return(((PORTPROTO *)leafport)->protoname);
}

/***********************************************************************
Module:  Sc_first_leaf_port
------------------------------------------------------------------------
Description:
	Return a pointer to the first port of the facet.  If no ports,
	return NULL.
------------------------------------------------------------------------
Calling Sequence:  leafport = Sc_first_leaf_port(leaffacet);

Name		Type		Description
----		----		-----------
leaffacet	*char		Pointer to leaf facet.
leafport	*char		Generic pointer to first port,
								NULL if not found.
------------------------------------------------------------------------
*/

char  *Sc_first_leaf_port(char *leaffacet)
{
	PORTPROTO	*pp;

	pp = ((NODEPROTO *)leaffacet)->firstportproto;
	if (pp == NOPORTPROTO) return((char *)NULL);
	return((char *)pp);
}

/***********************************************************************
Module:  Sc_next_leaf_port
------------------------------------------------------------------------
Description:
	Return a pointer to the next port of the facet.  If no ports left,
	return NULL.
------------------------------------------------------------------------
Calling Sequence:  nextport = Sc_next_leaf_port(curport);

Name		Type		Description
----		----		-----------
curport		*char		Pointer to current port.
nextport	*char		Generic pointer to next port,
								NULL if not found.
------------------------------------------------------------------------
*/

char  *Sc_next_leaf_port(char *curport)
{
	PORTPROTO	*pp;

	pp = ((PORTPROTO *)curport)->nextportproto;
	if (pp == NOPORTPROTO) return((char *)NULL);
	return((char *)pp);
}

/***********************************************************************
Module:  Sc_leaf_port_bits
------------------------------------------------------------------------
Description:
	Return the bits field of a leaf port.
------------------------------------------------------------------------
Calling Sequence:  bits = Sc_leaf_port_bits(leafport);

Name		Type		Description
----		----		-----------
leafport	*char		Pointer to port.
bits		int		Value of the bits field.
------------------------------------------------------------------------
*/

int Sc_leaf_port_bits(char *leafport)
{
	VARIABLE	*var;

	/* check if variable exits */
	var = getvalkey((INTBIG)leafport, VPORTPROTO, VINTEGER, sc_bitskey);
	if (var == NOVARIABLE)
	{
		if (setvalkey((INTBIG)leafport, VPORTPROTO, sc_bitskey, 0, VINTEGER) ==
			NOVARIABLE) ttyputmsg("ERROR creating SCBITS variable.");
		return(0);
	}
	return(var->addr);
}

/***********************************************************************
Module:  Sc_leaf_port_bits_address
------------------------------------------------------------------------
Description:
	Return the address of the bits field of a leaf port.
------------------------------------------------------------------------
Calling Sequence:  bits_a = Sc_leaf_port_bits_address(leafport);

Name		Type		Description
----		----		-----------
leafport	*char		Pointer to port.
bits_a		*int		Address of the bits field.
------------------------------------------------------------------------
*/

int  *Sc_leaf_port_bits_address(char *leafport)
{
	VARIABLE	*var;
	static int retval;

	/* check if variable exits */
	var = getvalkey((INTBIG)leafport, VPORTPROTO, VINTEGER, sc_bitskey);
	if (var == NOVARIABLE)
	{
		if (setvalkey((INTBIG)leafport, VPORTPROTO, sc_bitskey, 0, VINTEGER) ==
			NOVARIABLE) ttyputmsg("ERROR creating SCBITS variable.");
		if ((var = getvalkey((INTBIG)leafport, VPORTPROTO, VINTEGER, sc_bitskey))
			== NOVARIABLE)
		{
			ttyputmsg("ERROR creating SCBITS variable.");
			return((int *)NULL);
		}
	}
	retval = var->addr;
	return(&retval);
}

/***********************************************************************
Module:  Sc_leaf_port_type
------------------------------------------------------------------------
Description:
	Return the type of the leaf port.
------------------------------------------------------------------------
Calling Sequence:  type = Sc_leaf_port_type(leafport);

Name		Type		Description
----		----		-----------
leafport	*char		Pointer to leaf port.
type		int		Returned type.
------------------------------------------------------------------------
*/

int Sc_leaf_port_type(char *leafport)
{
	if (portispower((PORTPROTO *)leafport) != 0) return(SCPWRPORT);
	if (portisground((PORTPROTO *)leafport) != 0) return(SCGNDPORT);

	switch (((PORTPROTO *)leafport)->userbits & STATEBITS)
	{
		case BIDIRPORT: return(SCBIDIRPORT);
		case OUTPORT:   return(SCOUTPORT);
		case INPORT:    return(SCINPORT);
	}
	return(SCUNPORT);
}

/***********************************************************************
Module:  Sc_set_leaf_port_type
------------------------------------------------------------------------
Description:
	Set the type of the leaf port.
------------------------------------------------------------------------
Calling Sequence:  Sc_set_leaf_port_type(leafport, type);

Name		Type		Description
----		----		-----------
leafport	*PORTPROTO	Pointer to leaf port.
type		int		Type to set (eg. input, output, etc.).
------------------------------------------------------------------------
*/

void Sc_set_leaf_port_type(PORTPROTO *leafport, int type)
{
	leafport->userbits &= ~STATEBITS;
	switch (type) {
	case SCGNDPORT:
		leafport->userbits |= GNDPORT;
		break;
	case SCPWRPORT:
		leafport->userbits |= PWRPORT;
		break;
	case SCBIDIRPORT:
		leafport->userbits |= BIDIRPORT;
		break;
	case SCOUTPORT:
		leafport->userbits |= OUTPORT;
		break;
	case SCINPORT:
		leafport->userbits |= INPORT;
		break;
	default:
		leafport->userbits |= GNDPORT;
		break;
	}
	return;
}

/***********************************************************************
Module:  Sc_leaf_port_direction
------------------------------------------------------------------------
Description:
	Return the directions that a port can be attached to (i.e. up,
	down, left, right).
------------------------------------------------------------------------
Calling Sequence:  direct = Sc_leaf_port_direction(port);

Name		Type		Description
----		----		-----------
port		*char		Pointer to port.
direct		int		Returned direction for attaching.
------------------------------------------------------------------------
*/

int Sc_leaf_port_direction(char *port)
{
	int		direct, bits;

	direct = 0;
	bits = Sc_leaf_port_bits(port);
	if (bits & SCPORTDIRMASK) direct = bits & SCPORTDIRMASK;
		else
	{
		/* if no direction specified, assume both up and down */
		direct |= SCPORTDIRUP;
		direct |= SCPORTDIRDOWN;
	}

	return(direct);
}

/***********************************************************************
Module:  Sc_leaf_port_xpos
------------------------------------------------------------------------
Description:
	Return the xpos of the indicated leaf port from the left side of
	it's parent leaf facet.
------------------------------------------------------------------------
Calling Sequence:  xpos = Sc_leaf_port_xpos(port);

Name		Type		Description
----		----		-----------
port		*char		Pointer to leaf port.
xpos		int		Returned position from left side of facet.
------------------------------------------------------------------------
*/

int Sc_leaf_port_xpos(char *port)
{
	INTBIG		x, y;

	if (port != 0)
	{
		portposition(((PORTPROTO *)port)->subnodeinst,
			((PORTPROTO *)port)->subportproto, &x, &y);
		return(x - ((PORTPROTO *)port)->parent->lowx);
	}
	return(0);
}

/***********************************************************************
Module:  Sc_leaf_port_ypos
------------------------------------------------------------------------
Description:
	Return the ypos of the indicated leaf port from the bottom of
	it's parent leaf facet.
------------------------------------------------------------------------
Calling Sequence:  ypos = Sc_leaf_port_ypos(port);

Name		Type		Description
----		----		-----------
port		*char		Pointer to leaf port.
ypos		int		Returned position from bottom of facet.
------------------------------------------------------------------------
*/

int Sc_leaf_port_ypos(char *port)
{
	INTBIG		x, y;

	if (port != 0)
	{
		portposition(((PORTPROTO *)port)->subnodeinst,
			((PORTPROTO *)port)->subportproto, &x, &y);
		return(y - ((PORTPROTO *)port)->parent->lowy);
	}
	return(0);
}

/***********************************************************************
Module:  Sc_leaf_port_set_first
------------------------------------------------------------------------
Description:
	Set the first port of the indicated leaf facet to the indicated
	leaf port.
------------------------------------------------------------------------
Calling Sequence:  Sc_leaf_port_set_first(leaffacet, leafport);

Name		Type		Description
----		----		-----------
leaffacet	*char		Pointer to leaf facet.
leafport	*char		Pointer to leaf port.
------------------------------------------------------------------------
*/

void Sc_leaf_port_set_first(char *leaffacet, char *leafport)
{
	((NODEPROTO *)leaffacet)->firstportproto = (PORTPROTO *)leafport;
}

/***********************************************************************
Module:  Sc_leaf_port_set_next
------------------------------------------------------------------------
Description:
	Set the next port of the indicated leaf port to the indicated
	leaf port.  If the second leaf port is NULL, set to NOPORTPROTO.
------------------------------------------------------------------------
Calling Sequence:  Sc_leaf_port_set_next(leafport1, leafport2);

Name		Type		Description
----		----		-----------
leafport1	*char		Pointer to first leaf port.
leafport2	*char		Pointer to second leaf port.
------------------------------------------------------------------------
*/

void Sc_leaf_port_set_next(char *leafport1, char *leafport2)
{
	if (leafport2 != 0) {
		((PORTPROTO *)leafport1)->nextportproto = (PORTPROTO *)leafport2;
	}
	else {
		((PORTPROTO *)leafport1)->nextportproto = NOPORTPROTO;
	}
}

/***********************************************************************
Module:  Sc_library_read
------------------------------------------------------------------------
Description:
	Read the specified library.  Set error codes.
------------------------------------------------------------------------
Calling Sequence:  err = Sc_library_read(libname);

Name		Type		Description
----		----		-----------
libname		*char		String giving name of library.
err		int		Returned error code, 0 = no error.
------------------------------------------------------------------------
*/

int Sc_library_read(char *libname)
{
	LIBRARY    *lib;
	char       *newpars[PARS];
	extern AIDENTRY   *io_aid;

	/* check to see if library already exists */
	for (lib = el_curlib; lib != NOLIBRARY; lib = lib->nextlibrary) {
		if (namesame(libname, lib->libname) == 0)
			break;
	}
	if (lib == NOLIBRARY) {
		if ((lib = newlibrary(libname, libname)) == NOLIBRARY) {
			return(Sc_seterrmsg(SC_LIBNOMAKE, libname));
		}
	}
	else {
		eraselibrary(lib);
		if (reallocstring(&lib->libfile, libname, lib->cluster)) {
			return(Sc_seterrmsg(SC_NOMEMORY));
		}
	}
	newpars[0] = "binary";
	if (askaid(io_aid, "read", (INTBIG)lib, "binary") != 0) {
		eraselibrary(lib);
		return(Sc_seterrmsg(SC_LIBNOREAD, libname));
	}
	selectlibrary(lib);
	return(0);
}

/***********************************************************************
Module:  Sc_setup_for_maker
------------------------------------------------------------------------
Description:
	Set up the future creation of the maker by setting the appropriate
	prototypes.
------------------------------------------------------------------------
Calling Sequence:  err = Sc_setup_for_maker();

Name		Type		Description
----		----		-----------
err		int		Returned error code, 0 = no error.
------------------------------------------------------------------------
*/

int Sc_setup_for_maker(char *layer1, char *layer2)
{
	REGISTER ARCPROTO *ap, *alternatearc, **arclist;
	REGISTER NODEINST *ni;
	REGISTER int fun, i, j;
	static POLYGON *poly = NOPOLYGON;

	/* make sure there is a polygon */
	if (poly == NOPOLYGON) poly = allocpolygon(4, sc_aid->cluster);

	/* find the horizontal (layer 1) and vertical (layer 2) arcs */
	layer1_arc = layer2_arc = alternatearc = NOARCPROTO;
	for(ap = el_curtech->firstarcproto; ap != NOARCPROTO; ap = ap->nextarcproto)
	{
		switch ((ap->userbits&AFUNCTION)>>AFUNCTIONSH)
		{
			case APMETAL1: layer1_arc = ap;   break;
			case APMETAL2: case APMETAL3: case APMETAL4: case APMETAL5:
			case APMETAL6: case APMETAL7: case APMETAL8:
				layer2_arc = ap;   break;
			case APPOLY1: case APPOLY2: case APPOLY3:
				alternatearc = ap;   break;
		}
	}
	if (layer1_arc == NOARCPROTO) layer1_arc = alternatearc;

	/* allow overrides */
	if (*layer1 != 0)
	{
		ap = getarcproto(layer1);
		if (ap != NOARCPROTO) layer1_arc = ap;
	}
	if (*layer2 != 0)
	{
		ap = getarcproto(layer2);
		if (ap != NOARCPROTO) layer2_arc = ap;
	}
	if (layer1_arc == NOARCPROTO) return(Sc_seterrmsg(SC_NO_LAYER1_ARC));
	if (layer2_arc == NOARCPROTO) return(Sc_seterrmsg(SC_NO_LAYER2_ARC));

	/* find the contact between the two layers */
	for(via_proto = el_curtech->firstnodeproto; via_proto != NONODEPROTO;
		via_proto = via_proto->nextnodeproto)
	{
		fun = (via_proto->userbits&NFUNCTION) >> NFUNCTIONSH;
		if (fun != NPCONTACT && fun != NPCONNECT) continue;
		arclist = via_proto->firstportproto->connects;
		for(i=0; arclist[i] != NOARCPROTO; i++)
			if (arclist[i] == layer1_arc) break;
		if (arclist[i] == NOARCPROTO) continue;
		for(i=0; arclist[i] != NOARCPROTO; i++)
			if (arclist[i] == layer2_arc) break;
		if (arclist[i] != NOARCPROTO) break;
	}
	if (via_proto == NONODEPROTO) return(Sc_seterrmsg(SC_NO_VIA));

	/* find the pin nodes on the connecting layers */
	layer1_proto = getpinproto(layer1_arc);
	if (layer1_proto == NONODEPROTO) return(Sc_seterrmsg(SC_NO_LAYER1_NODE));
	layer2_proto = getpinproto(layer2_arc);
	if (layer2_proto == NONODEPROTO) return(Sc_seterrmsg(SC_NO_LAYER2_NODE));

	/*
	 * find the pure-layer node on the P-well layer
	 * if the p-well size is zero don't look for the node
	 * allows technologies without p-wells to be routed (i.e. GaAs)
	 */
	if (maker_control.pwell_size == 0) pwell_proto = NONODEPROTO; else
	{
		for(pwell_proto = el_curtech->firstnodeproto; pwell_proto != NONODEPROTO;
			pwell_proto = pwell_proto->nextnodeproto)
		{
			fun = (pwell_proto->userbits&NFUNCTION) >> NFUNCTIONSH;
			if (fun != NPNODE) continue;
			ni = dummynode();
			ni->proto = pwell_proto;
			ni->lowx = pwell_proto->lowx;   ni->highx = pwell_proto->highx;
			ni->lowy = pwell_proto->lowy;   ni->highy = pwell_proto->highy;
			j = nodepolys(ni);
			for(i=0; i<j; i++)
			{
				shapenodepoly(ni, (INTSML)i, poly);
				fun = layerfunction(el_curtech, poly->layer);
				if ((fun&LFTYPE) == LFWELL) break;
			}
			if (i < j) break;
		}
		if (pwell_proto == NONODEPROTO) return(Sc_seterrmsg(SC_NO_LAYER_PWELL));
	}

	return(SC_NOERROR);
}

/***********************************************************************
Module:  Sc_create_leaf_facet
------------------------------------------------------------------------
Description:
	Create a new leaf facet of the indicated name in the VLSI Layout Tool
	Environment.  Return NULL on error.
------------------------------------------------------------------------
Calling Sequence:  bfacet = Sc_create_leaf_facet(name);

Name		Type		Description
----		----		-----------
name		*char		Name of new leaf facet.
bfacet		*char		Pointer to created leaf facet.
------------------------------------------------------------------------
*/

char  *Sc_create_leaf_facet(char *name)
{
	NODEPROTO	*bfacet;
	char	*view, layoutname[256];

								/*
								 * use layoutname because we don't know how
								 * long "name" is
								 */
	(void) strcpy (layoutname, name);

	if ((view = strchr (layoutname, '{')) == NULL)
		(void) strcat (layoutname, "{lay}");
	else if (strncmp (view, "{lay}", 5) != 0){
		view = '\0';
		(void) strcat (layoutname, "{lay}");
	}

	bfacet = newnodeproto(layoutname, el_curlib);
	if (bfacet == NONODEPROTO) {
		return((char *)NULL);
	}
	else {
		return((char *)bfacet);
	}
}

/***********************************************************************
Module:  Sc_create_leaf_instance
------------------------------------------------------------------------
Description:
	Create a new leaf instance of the indicated type in the VLSI Layout
	Tool Environment.  Return NULL on error.
------------------------------------------------------------------------
Calling Sequence:  binst = Sc_create_leaf_instance(name, type, minx, maxx,
								miny, maxy, transpose, rotation, parent);

Name		Type		Description
----		----		-----------
name		*char		Name of new leaf instance.
type		*char		Pointer to leaf facet type.
minx, maxx	int		Bounds in X.
miny, maxy	int		Bounds in Y.
transpose	int		Transposition flag (TRUE == transpose).
rotation	int		Degrees of rotation.
parent		*char		Facet in which to create instance.
binst		*char		Pointer to created leaf instance.
------------------------------------------------------------------------
*/

char  *Sc_create_leaf_instance(char *name, char *type, int minx, int maxx, int miny,
	int maxy, int transpose, int rotation, char *parent)
{
	NODEINST	*binst;

	binst = newnodeinst((NODEPROTO *)type, minx, maxx, miny, maxy, (INTSML)transpose,
		(INTSML)rotation, (NODEPROTO *)parent);
	if (binst == NONODEINST) return((char *)NULL);

	if (setvalkey((INTBIG)binst, VNODEINST, el_node_name, (INTBIG)name,
		VSTRING|VDISPLAY) == NOVARIABLE)
			ttyputmsg("ERROR creating name of instance '%s'.", name);
	return((char *)binst);
}

/***********************************************************************
Module:  Sc_create_layer1_node
------------------------------------------------------------------------
Description:
	Create a node in layer1 in the indicated facet at the given position.
------------------------------------------------------------------------
Calling Sequence:  binst = Sc_create_layer1_node(xpos, ypos, xsize,
								ysize, bfacet);

Name		Type		Description
----		----		-----------
xpos		int		Centre X position.
ypos		int		Centre Y position.
xsize		int		Size in X.
ysize		int		Size in Y.
bfacet		*char		Facet in which to create feed.
binst		*char		Returned pointer to created instance.
------------------------------------------------------------------------
*/

char  *Sc_create_layer1_node(int xpos, int ypos, int xsize, int ysize, char *bfacet)
{
	NODEINST	*binst;
	int		lowx, highx, lowy, highy;

	lowx = xpos - (xsize >> 1);
	highx = lowx + xsize;
	lowy = ypos - (ysize >> 1);
	highy = lowy + ysize;
	binst = newnodeinst(layer1_proto, lowx, highx, lowy, highy, 0, 0,
		(NODEPROTO *)bfacet);
	if (binst == NONODEINST) return((char *)NULL);
	return((char *)binst);
}

/***********************************************************************
Module:  Sc_create_layer2_node
------------------------------------------------------------------------
Description:
	Create a layer2 node in the indicated facet at the given position.
------------------------------------------------------------------------
Calling Sequence:  binst = Sc_create_layer2_node(xpos, ypos, xsize,
								ysize, bfacet);

Name		Type		Description
----		----		-----------
xpos		int		Centre X position.
ypos		int		Centre Y position.
xsize		int		Width in X direction.
ysize		int		Size in Y direction.
bfacet		*char		Facet in which to create feed.
binst		*char		Returned pointer to created instance.
------------------------------------------------------------------------
*/

char  *Sc_create_layer2_node(int xpos, int ypos, int xsize, int ysize, char *bfacet)
{
	NODEINST	*binst;
	int		lowx, highx, lowy, highy;

	lowx =  xpos - (xsize >> 1);
	highx = lowx + xsize;
	lowy = ypos - (ysize >> 1);
	highy = lowy + ysize;
	binst = newnodeinst(layer2_proto, lowx, highx, lowy, highy, 0, 0,
		(NODEPROTO *)bfacet);
	if (binst == NONODEINST) return((char *)NULL);
	return((char *)binst);
}

/***********************************************************************
Module:  Sc_create_via
------------------------------------------------------------------------
Description:
	Create a via in the indicated facet at the given position.
------------------------------------------------------------------------
Calling Sequence:  binst = Sc_create_via(xpos, ypos, bfacet);

Name		Type		Description
----		----		-----------
xpos		int		Centre X position.
ypos		int		Centre Y position.
bfacet		*char		Facet in which to create feed.
binst		*char		Returned pointer to created instance.
------------------------------------------------------------------------
*/

char  *Sc_create_via(int xpos, int ypos, char *bfacet)
{
	NODEINST	*binst;
	int		lowx, highx, lowy, highy;

	lowx = xpos - ((via_proto->highx - via_proto->lowx) >> 1);
	highx = lowx + (via_proto->highx - via_proto->lowx);
	lowy = ypos - ((via_proto->highy - via_proto->lowy) >> 1);
	highy = lowy + (via_proto->highy - via_proto->lowy);
	binst = newnodeinst(via_proto, lowx, highx, lowy, highy, 0, 0,
		(NODEPROTO *)bfacet);
	if (binst == NONODEINST) return((char *)NULL);
	return((char *)binst);
}

/***********************************************************************
Module:  Sc_create_pwell
------------------------------------------------------------------------
Description:
	Create a pwell in the indicated facet at the given position.
------------------------------------------------------------------------
Calling Sequence:  binst = Sc_create_pwell(xpos, ypos, xsize, ysize, bfacet);

Name		Type		Description
----		----		-----------
xpos		int		Centre X position.
ypos		int		Centre Y position.
xsize		int		Size in X.
ysize		int		Size in Y.
bfacet		*char		Facet in which to create feed.
binst		*char		Returned pointer to created instance.
------------------------------------------------------------------------
*/

char  *Sc_create_pwell(int xpos, int ypos, int xsize, int ysize, char *bfacet)
{
	NODEINST	*binst;
	int		lowx, highx, lowy, highy;

	if (pwell_proto == NONODEPROTO) return((char *)NULL);

	lowx = xpos - (xsize >> 1);
	highx = lowx + xsize;
	lowy = ypos - (ysize >> 1);
	highy = lowy + ysize;
	binst = newnodeinst(pwell_proto, lowx, highx, lowy, highy, 0, 0,
		(NODEPROTO *)bfacet);
	if (binst == NONODEINST) return((char *)NULL);
	return((char *)binst);
}

/***********************************************************************
Module:  Sc_create_track_layer1
------------------------------------------------------------------------
Description:
	Create a track between the two given ports on the first routing
	layer.  Note that ports of primitive instances are passed as NULL
	and must be determined.
------------------------------------------------------------------------
Calling Sequence:  inst = Sc_create_track_layer1(instA, portA,
								instB, portB, width, bfacet);

Name		Type		Description
----		----		-----------
instA		*char		Pointer to first instance.
portA		*char		Pointer to first port.
instB		*char		Pointer to second instance.
portB		*char		Pointer to second port.
width		int		Width of track.
bfacet		*char		Pointer to facet in which to create.
inst		*char		Returned pointer to created track.
------------------------------------------------------------------------
*/

char  *Sc_create_track_layer1(char *instA, char *portA, char *instB, char *portB,
	int width, char *bfacet)
{
	ARCINST	*inst;
	INTBIG		xA, yA, xB, yB;

	/* check if first is primative */
	if ((PORTPROTO *)portA == NULL) {
		portA = (char *)(((NODEINST *)instA)->proto->firstportproto);
	}
	/* find centre position of portA */
	portposition((NODEINST *)instA, (PORTPROTO *)portA, &xA, &yA);

	/* check if second is primative */
	if ((PORTPROTO *)portB == NULL) {
		portB = (char *)(((NODEINST *)instB)->proto->firstportproto);
	}
	/* find centre position of portB */
	portposition((NODEINST *)instB, (PORTPROTO *)portB, &xB, &yB);

	inst = newarcinst(layer1_arc, width, FIXANG, (NODEINST *)instA,
		(PORTPROTO *)portA, xA, yA, (NODEINST *)instB, (PORTPROTO *)portB, xB,
			yB, (NODEPROTO *)bfacet);
	if (inst == NOARCINST) return((char *)NULL);
	return((char *)inst);
}

/***********************************************************************
Module:  Sc_create_track_layer2
------------------------------------------------------------------------
Description:
	Create a track between the two given ports on the second routing
	layer.  Note that ports of primitive instances are passed as NULL
	and must be determined.
------------------------------------------------------------------------
Calling Sequence:  inst = Sc_create_track_layer2(instA, portA,
								instB, portB, width, bfacet);

Name		Type		Description
----		----		-----------
instA		*char		Pointer to first instance.
portA		*char		Pointer to first port.
instB		*char		Pointer to second instance.
portB		*char		Pointer to second port.
width		int		Width of track.
bfacet		*char		Pointer to facet in which to create.
inst		*char		Returned pointer to created track.
------------------------------------------------------------------------
*/

char  *Sc_create_track_layer2(char *instA, char *portA, char *instB, char *portB,
	int width, char *bfacet)
{
	ARCINST	*inst;
	INTBIG		xA, yA, xB, yB;

	/* check if first is primative */
	if ((PORTPROTO *)portA == NULL)
		portA = (char *)(((NODEINST *)instA)->proto->firstportproto);

	/* find centre position of portA */
	portposition((NODEINST *)instA, (PORTPROTO *)portA, &xA, &yA);

	/* check if second is primative */
	if ((PORTPROTO *)portB == NULL)
		portB = (char *)(((NODEINST *)instB)->proto->firstportproto);

	/* find centre position of portB */
	portposition((NODEINST *)instB, (PORTPROTO *)portB, &xB, &yB);

	inst = newarcinst(layer2_arc, width, FIXANG, (NODEINST *)instA,
		(PORTPROTO *)portA, xA, yA, (NODEINST *)instB, (PORTPROTO *)portB, xB,
			yB, (NODEPROTO *)bfacet);
	if (inst == NOARCINST) return((char *)NULL);
	return((char *)inst);
}

/***********************************************************************
Module:  Sc_create_export_port
------------------------------------------------------------------------
Description:
	Create an export port at the given instance at the given port.
	Note that ports of primitive instances are passed as NULL and must
	be determined.
------------------------------------------------------------------------
Calling Sequence:  xport = Sc_create_export_port(inst, port, name,
								type, bfacet);

Name		Type		Description
----		----		-----------
instA		*char		Pointer to instance.
port		*char		Pointer to port.
name		*char		Pointer to name.
type		int		Type of port (eg. input, output, etc.)
bfacet		*char		Pointer to facet in which to create.
xport		*char		Returned pointer to created port.
------------------------------------------------------------------------
*/

char  *Sc_create_export_port(char *inst, char *port, char *name, int type, char *bfacet)
{
	PORTPROTO	*xport;

	/* check if primative */
	if ((PORTPROTO *)port == NULL)
		port = (char *)(((NODEINST *)inst)->proto->firstportproto);

	xport = newportproto((NODEPROTO *)bfacet, (NODEINST *)inst,
		(PORTPROTO *)port, name);
	if (xport == NOPORTPROTO) return((char *)NULL);
	Sc_set_leaf_port_type(xport, type);
	return((char *)xport);
}

/*
 * routine to examine a schematic from the current facet
 */
void Sc_schematic(void)
{
	ttyputmsg("Sorry, cannot handle schematics yet");
}

#endif  /* SCAID - at top */
