/*
 * Electric(tm) VLSI Design System
 *
 * File: sim.c
 * Simulation aid: main module
 * Written by: Steven M. Rubin, Electric Editor Incorporated
 *
 * Copyright (c) 1998 Electric Editor Incorporated.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Electric Editor Incorporated
 * 23470 Sunset Drive, Suite 108
 * Los Gatos, California 95033
 * support@electriceditor.com
 */

#include "config.h"
#if SIMAID

#include "global.h"
#include "egraphics.h"
#include "sim.h"
#if SIMALS
#  include "simals.h"
#endif
#include <signal.h>

/* prototypes for local routines */
#if SIMALS
static INTSML sim_nextcustomclock(char*, COMCOMP*[], char);
#endif
void sim_checktostopsimulation(NODEPROTO *np);

/************************** ALS COMMANDS **************************/

#if SIMALS

/* ALS's build-actel-models command */
static COMCOMP qbuildp = {NOKEYWORD, topoffile, nextfile, NOPARAMS, NOBACKUP,
	0, " \t", "Actel table file to convert to library", 0};

/* ALS's clock command */
static COMCOMP clockcdurp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, sim_nextcustomclock, NOBACKUP,
	0, " \t", "clock: duration of this phase", 0};
static COMCOMP clockclevp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "clock: level for this phase", 0};
static INTSML sim_nextcustomclock(char *i, COMCOMP *j[], char c)
{ j[0] = &clockclevp; j[1] = &clockcdurp; return(2); }
static COMCOMP clockccyp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, sim_nextcustomclock, NOBACKUP,
	0, " \t", "clock: number of cycles (0 for infinite)", 0};
static COMCOMP clockcstrp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "clock: strength", 0};
static COMCOMP clockclp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "clock: random distribution", 0};
static COMCOMP clockfp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "50/50 duty cycle frequency", 0};
static COMCOMP clockpp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "50/50 duty cycle period", 0};
static KEYWORD clockopt[] =
{
	{"frequency",     1,{&clockfp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"period",        1,{&clockpp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"custom",        3,{&clockclp,&clockcstrp,&clockccyp,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP sim_clockop = {clockopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "clock options", 0};
static COMCOMP clocknp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "node name", 0};

/* ALS's go command */
static COMCOMP gop = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "maximum simulation time (in seconds)", 0};

/* ALS's level command */
COMCOMP sim_alslevelsp = {NOKEYWORD, sim_alstopofinstances, sim_alsnextinstance,
	NOPARAMS, NOBACKUP, 0, " \t", "instance to set", 0};
static KEYWORD levelopt[] =
{
	{"up",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"set",        1,{&sim_alslevelsp,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP levelp = {levelopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "level change option", 0};

/* ALS's print command */
static COMCOMP printsp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "node to print", 0};
static KEYWORD printopt[] =
{
	{"display",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"instances",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"netlist",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"size",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"state",          1,{&printsp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"vector",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"xref",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP printp = {printopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "printing options", 0};

/* ALS's seed command */
static KEYWORD seedopt[] =
{
	{"reset",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"no-reset",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP seedp = {seedopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "random number seed setting", 0};

/* ALS's set command */
static COMCOMP settp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "time of node set (in seconds)", 0};
static COMCOMP setsp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "strength of node set", 0};
static KEYWORD setlopt[] =
{
	{"H",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"L",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"X",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP setlp = {setlopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "state level", 0};
static COMCOMP setnp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "node name to set", 0};

/* ALS's trace command */
static KEYWORD traceopt[] =
{
	{"on",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"off",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP tracep = {traceopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "trace state", 0};

/* ALS's vector command */
static COMCOMP vectordtp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "time at which to delete", 0};
static KEYWORD vectordopt[] =
{
	{"time",       1,{&vectordtp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"all",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP vectordop = {vectordopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "vector deletion option", 0};
static COMCOMP vectordp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "node name to delete", 0};
static COMCOMP vectorlp = {NOKEYWORD, topoffile, nextfile, NOPARAMS, NOBACKUP,
	0, " \t", "file name with vectors", 0};
static COMCOMP vectorsp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "file name to save vectors", 0};
static KEYWORD vectoropt[] =
{
	{"delete",      2,{&vectordp,&vectordop,NOKEY,NOKEY,NOKEY}},
	{"load",        1,{&vectorlp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"save",        1,{&vectorsp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"new",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP vectorp = {vectoropt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "vector options", 0};
static KEYWORD annotateopt[] =
{
    {"minimum",    0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
    {"typical",    0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
    {"maximum",    0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
    TERMKEY
};
static COMCOMP annotatep = {annotateopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
    0, " \t", "annotate options", 0};
static COMCOMP orderrp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
    0, " \t", "ordered list of trace names", 0};
static KEYWORD orderopt[] =
{
    {"save",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
    {"restore",    1,{&orderrp,NOKEY,NOKEY,NOKEY,NOKEY}},
    TERMKEY
};
static COMCOMP orderp = {orderopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
    0, " \t", "order options", 0};

/* ALS's command table */
static KEYWORD sim_alsopt[] =
{
    {"annotate",           1,{&annotatep,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"build-actel-models", 1,{&qbuildp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"clock",              2,{&clocknp,&sim_clockop,NOKEY,NOKEY,NOKEY}},
	{"erase",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"go",                 1,{&gop,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"help",               0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"level",              1,{&levelp,NOKEY,NOKEY,NOKEY,NOKEY}},
    {"order",              1,{&orderp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"print",              1,{&printp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"seed",               1,{&seedp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"set",                4,{&setnp,&setlp,&setsp,&settp,NOKEY}},
	{"trace",              1,{&tracep,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"vector",             1,{&vectorp,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP sim_alsp = {sim_alsopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "Simulation command", 0};

#endif  /* SIMALS */

/************************** SIMULATION WINDOW COMMANDS **************************/

static COMCOMP cursorgp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "time at the extension cursor", 0};
static COMCOMP cursorwp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "time at the main cursor", 0};
static KEYWORD cursoropt[] =
{
	{"extension",    1,{&cursorgp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"main",         1,{&cursorwp,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP cursorp = {cursoropt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "cursor color", 0};

static COMCOMP moveap = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "distance to move (in seconds)", 0};
static COMCOMP movesp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "distance to move (in signals)", 0};
static KEYWORD moveopt[] =
{
	{"left",         1,{&moveap,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"right",        1,{&moveap,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"up",           1,{&movesp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"down",         1,{&movesp,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP movep = {moveopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "window move option", 0};

static COMCOMP zoomup = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "upper time of zoom", 0};
static COMCOMP zoomlp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "lower time of zoom", 0};
static COMCOMP zoomap = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "amount to zoom", 0};
static KEYWORD zoomopt[] =
{
	{"window",      2,{&zoomlp,&zoomup,NOKEY,NOKEY,NOKEY}},
	{"in",          1,{&zoomap,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"out",         1,{&zoomap,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"cursor",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP zoomp = {zoomopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "window zoom options", 0};
static COMCOMP tracessp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "number of traces to show on the display", 0};
static KEYWORD tracesopt[] =
{
	{"more",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"less",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"set",       1,{&tracessp,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP tracesp = {tracesopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "window traces options", 0};
static KEYWORD dispcolorsopt[] =
{
	{"white",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"black",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"red",            0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"blue",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"green",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"cyan",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"magenta",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"yellow",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"gray",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"orange",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"purple",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"brown",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"light-gray",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"dark-gray",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"light-red",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"dark-red",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"light-green",    0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"dark-green",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"light-blue",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"dark-blue",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP dispcolorsp = {dispcolorsopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "color for this strength/level", 0};
static KEYWORD dispcoloropt[] =
{
	{"off",       1,{&dispcolorsp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"node",      1,{&dispcolorsp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"gate",      1,{&dispcolorsp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"power",     1,{&dispcolorsp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"low",       1,{&dispcolorsp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"high",      1,{&dispcolorsp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"undefined", 1,{&dispcolorsp,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP dispcolorp = {dispcoloropt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "window strength color options", 0};
static KEYWORD sim_windowopt[] =
{
	{"cursor",           1,{&cursorp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"move",             1,{&movep,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"traces",           1,{&tracesp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"zoom",             1,{&zoomp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"color",            1,{&dispcolorp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"2-state-display",  0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"12-state-display", 0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"advance-time",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"freeze-time",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"display-waveform", 0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"ignore-waveform",  0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP sim_windowp = {sim_windowopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "Simulation window command", 0};

/************************** SPICE COMMANDS **************************/

#if SIMSPICE

static KEYWORD sim_spicelevelopt[] =
{
	{"1",    0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"2",    0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"3",    0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP sim_spicelevelp = {sim_spicelevelopt, NOTOPLIST, NONEXTLIST,
	NOPARAMS, NOBACKUP, INPUTOPT, " \t", "Spice simulation level", "show current"};
static KEYWORD sim_spiceformopt[] =
{
	{"2",                  0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"3",                  0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"hspice",             0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP sim_spiceformp = {sim_spiceformopt, NOTOPLIST, NONEXTLIST,
	NOPARAMS, NOBACKUP, INPUTOPT, " \t", "Spice format", 0};
static KEYWORD sim_spicenotopt[] =
{
	{"resistance",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"capacitances",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"plot",               0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"use-nodenames",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP sim_spicenotp = {sim_spicenotopt, NOTOPLIST, NONEXTLIST,
	NOPARAMS, NOBACKUP, INPUTOPT, " \t", "Spice simulation NOT option", 0};
COMCOMP sim_spicereadp = {NOKEYWORD, topoffile, nextfile, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "File containing spice output", 0};
static COMCOMP sim_spicesavep = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "File in which to save spice output", 0};
static KEYWORD sim_spiceopt[] =
{
	{"level",              1,{&sim_spicelevelp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"save-output",        1,{&sim_spicesavep,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"format",             1,{&sim_spiceformp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"parse-output",       1,{&sim_spicereadp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"debug-toggle",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"not",                1,{&sim_spicenotp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"resistance",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"capacitances",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"plot",               0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"use-nodenames",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP sim_spicep = {sim_spiceopt, NOTOPLIST, NONEXTLIST,
	NOPARAMS, NOBACKUP, INPUTOPT, " \t", "Spice simulation option", 0};

#endif  /* SIMSPICE */

/************************** SIMULATOR COMMANDS **************************/

static KEYWORD simulatorsimulateopt[] =
{
#if SIMSIM
	{"esim",             0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"rsim",             0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"rnl",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"cosmos",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
#endif
#if SIMSPICE
	{"spice",            0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
#endif
#if SIMMOSSIM
	{"mossim",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
#endif
#if SIMTEXSIM
	{"texsim",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
#endif
#if SIMALS
	{"als",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
#endif
#if SIMVERILOG
	{"verilog",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
#endif
#if SIMPAL
	{"abel-pal",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
#endif
#if SIMSILOS
	{"silos",            0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
#endif
	TERMKEY
};
static COMCOMP simsimulatep = {simulatorsimulateopt, NOTOPLIST, NONEXTLIST,
	NOPARAMS, NOBACKUP, INPUTOPT, " \t", "Netlist format to generate", 0};
static COMCOMP simulatornetp = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "net or component to point out", 0};
static KEYWORD simulatornotopt[] =
{
	{"execute",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP simulatornotp = {simulatornotopt, NOTOPLIST, NONEXTLIST,
	NOPARAMS, NOBACKUP, INPUTOPT, " \t", "Simulation NOT option", 0};
static KEYWORD simulatoropt[] =
{
#if SIMSPICE
	{"spice",            1,{&sim_spicep,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"pointout",         1,{&simulatornetp,NOKEY,NOKEY,NOKEY,NOKEY}},
#endif
#if SIMALS
	{"als",              1,{&sim_alsp,NOKEY,NOKEY,NOKEY,NOKEY}},
#endif
	{"execute",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"not",              1,{&simulatornotp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"resume",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"simulate",         1,{&simsimulatep,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"window",           1,{&sim_windowp,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP sim_simulatorp = {simulatoropt,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
		0, " \t", "Simulator action", "show current simulator"};

struct
{
	char *simname;
	INTSML min;
	INTSML format;
} sim_simtable[] =
{
#if SIMPAL
	{"abel-pal", 1, ABEL},
#endif
#if SIMSIM
	{"cosmos",	 1, COSMOS},
	{"esim",     1, ESIM},
	{"rnl",      2, RNL},
	{"rsim",     2, RSIM},
#endif
#if SIMMOSSIM
	{"mossim",   1, MOSSIM},
#endif
#if SIMALS
	{"als",      1, ALS},
#endif
#if SIMTEXSIM
	{"texsim",   1, TEXSIM},
#endif
#if SIMSILOS
	{"silos",    2, SILOS},
#endif
#if SIMSPICE
	{"spice",    2, SPICE},
#endif
#if SIMVERILOG
	{"verilog",  1, VERILOG},
#endif
	{NULL, 0, 0}  /* 0 */
};

AIDENTRY  *sim_aid;				/* the Simulator aid object */
INTBIG     sim_format;			/* key for "SIM_format" (ESIM, etc) */
INTBIG     sim_netfile;			/* key for "SIM_netfile" */
INTBIG     sim_dontrun;			/* key for "SIM_dontrun" */
NODEPROTO *sim_simnt;			/* facet being simulated */
INTBIG     sim_tosim[2];		/* pipe to the simulator process */
INTBIG     sim_fromsim[2];		/* pipe from the simulator process */
INTBIG     sim_process;			/* process number of simulator */
INTBIG     sim_pseudonet;		/* for internal networks */
INTSML     sim_circuitchanged;	/* nonzero if circuit being simulated was changed */
ARCINST   *sim_modifyarc;		/* arc whose userbits are being changed */
INTBIG     sim_modifyarcbits;	/* former value of userbits on changed arc */

void sim_init(INTBIG *argc, char *argv[], AIDENTRY *thisaid)
{
	/* nothing on pass 2 or 3 of initialization */
	if (thisaid == NOAID || thisaid == 0) return;

	/* pass 1 initialization */
	sim_aid = thisaid;
	sim_format = makekey("SIM_format");
	sim_dontrun = makekey("SIM_dontrun");
	sim_netfile = makekey("SIM_netfile");
#if SIMSPICE
	sim_listingfile = makekey("SIM_listingfile");
	sim_spice_level = makekey("SIM_spice_level");
	sim_spice_state = makekey("SIM_spice_state");
#endif

	/* initialize default simulator format */
	nextvarchangequiet();
	(void)setvalkey((INTBIG)sim_aid, VAID, sim_format, ALS, VINTEGER|VDONTSAVE);

	/* do not have the simulator automatically run */
	nextvarchangequiet();
	(void)setvalkey((INTBIG)sim_aid, VAID, sim_dontrun, 1, VINTEGER|VDONTSAVE);

	/* ESIM/RSIM/RNL initialization */
	nextvarchangequiet();
	(void)setvalkey((INTBIG)sim_aid, VAID, sim_netfile, (INTBIG)"", VSTRING|VDONTSAVE);

	/* SPICE initialization */
#if SIMSPICE
	nextvarchangequiet();
	(void)setvalkey((INTBIG)sim_aid, VAID, sim_listingfile, (INTBIG)"", VSTRING|VDONTSAVE);
	nextvarchangequiet();
	(void)setvalkey((INTBIG)sim_aid, VAID, sim_spice_state, SPICERESISTANCE, VINTEGER|VDONTSAVE);
	nextvarchangequiet();
	(void)setvalkey((INTBIG)sim_aid, VAID, sim_spice_level, 1, VINTEGER|VDONTSAVE);
#endif

	/* initialize the simulation window system */
	sim_window_init();

	/* miscellaneous initialization */
	sim_process = -1;
	sim_simnt = NONODEPROTO;
	sim_circuitchanged = 0;
}

void sim_done(void)
{
	REGISTER VARIABLE *var;

	if (sim_process < 0) return;
#if !defined(MACOS) && !defined(WIN32)
    (void)kill(sim_process, SIGKILL);
	waitfor(sim_process);
#endif

	var = getvalkey((INTBIG)sim_aid, VAID, VSTRING, sim_netfile);
	if (var != NOVARIABLE)
		ttyputmsg("Simulation net list saved in '%s'", (char *)var->addr);
}

INTSML sim_set(INTSML count, char *par[])
{
	REGISTER INTSML i, l, newformat, amount, signals;
	REGISTER INTBIG spicestate;
	REGISTER char *pp;
	REGISTER VARIABLE *var;
	NODEPROTO *np;
#if SIMALS
	NODEPROTO *simnp;
	REGISTER INTSML active;
	extern AIDENTRY *vhdl_aid;
#endif
	float time, size, factor, maxtime, mintime, main, extension;

	if (count == 0)
	{
		var = getvalkey((INTBIG)sim_aid, VAID, VINTEGER, sim_format);
		if (var == NOVARIABLE) l = -1; else l = var->addr;
		for(i=0; sim_simtable[i].simname != 0; i++)
			if (l == sim_simtable[i].format)
		{
			ttyputmsg("Current simulator is %s", sim_simtable[i].simname);
			if (l == ESIM || l == RSIM || l == RNL || l == SPICE)
			{
				var = getvalkey((INTBIG)sim_aid, VAID, VINTEGER, sim_dontrun);
				if (var != NOVARIABLE && var->addr == 0)
				{
					ttyputmsg("After writing netlist, simulator will be run");
					return(0);
				}
			}
			ttyputmsg("Only netlist will be written, no simulator will be run");
			return(0);
		}
		ttyputmsg("No current simulator");
		return(0);
	}
	l = strlen(pp = par[0]);

	if (namesamen(pp, "simulate", l) == 0 && l >= 2)
	{
		if (count < 2)
		{
			ttyputerr("Usage: tellaid simulation simulate FORMAT");
			return(1);
		}

		/* get the simulator name */
		l = strlen(pp = par[1]);
		for(i=0; sim_simtable[i].simname != 0; i++)
			if (namesamen(pp, sim_simtable[i].simname, l) == 0 && l >= sim_simtable[i].min) break;
		if (sim_simtable[i].simname == 0)
		{
			ttyputerr("Unknown simulation format: %s", pp);
			return(1);
		}
		newformat = sim_simtable[i].format;

		/* see if there is already a simulation running */
		if (sim_process != -1)
		{
			var = getvalkey((INTBIG)sim_aid, VAID, VINTEGER, sim_format);
			if (var != NOVARIABLE && newformat == var->addr)
			{
				ttyputmsg("This simulator is already running");
				return(0);
			}
			ttyputerr("Cannot switch simulators while one is running");
			return(1);
		}

		/* make sure there is a facet to simulate */
		np = getcurfacet();
		if (np == NONODEPROTO)
		{
			ttyputerr("No current facet to simulate");
			return(1);
		}

		/* remember the currently running format */
		(void)setvalkey((INTBIG)sim_aid, VAID, sim_format, newformat, VINTEGER|VDONTSAVE);

		/* write the appropriate simulation file */
		switch (newformat)
		{
#if SIMPAL
			case ABEL:
				if (checkcap(sim_aid, SIMHASCOMM) == 0)
				{
					ttyputerr("Sorry, ABEL interface is not available");
					break;
				}
				sim_writepalnetlist(np);
				break;
#endif
#if SIMSIM
			case COSMOS:
			case ESIM:
			case RNL:
			case RSIM:
				if (checkcap(sim_aid, SIMHASUNIV) == 0)
				{
					ttyputerr("Sorry, ESIM/RSIM/RNL/COSMOS interfaces are not available");
					break;
				}
				sim_writesim(np, newformat);
				break;
#endif
#if SIMMOSSIM
			case MOSSIM:
				if (checkcap(sim_aid, SIMHASUNIV) == 0)
				{
					ttyputerr("Sorry, MOSSIM interface is not available");
					break;
				}
				sim_writemossim(np);
				break;
#endif
#if SIMALS
			case ALS:
				if (checkcap(sim_aid, SIMHASALS) == 0)
				{
					ttyputerr("Sorry, simulation is not available");
					break;
				}

				active = sim_window_isactive(&simnp);
				if (active != 0)
				{
					if (simnp == np)
					{
						/* display schematic window if it is not up */
						if ((active&SIMWINDOWSCHEMATIC) == 0)
						{
							(void)sim_window_create(0, simnp, 0, simals_charhandlerschem);
							break;
						}

						/* display waveform window if it is not up */
						if ((active&SIMWINDOWWAVEFORM) == 0 && (sim_window_state&SHOWWAVEFORM) != 0)
						{
							(void)sim_window_create(0, simnp, simals_charhandlerwave, 0);
							break;
						}
						ttyputmsg("Simulation is already running");
						break;
					}
					ttyputerr("Close all simulation windows before simulating another facet");
					break;
				}

				/* initialize memory */
				simals_init();

				/* demand a ALS netlist */
				(void)askaid(vhdl_aid, "want-netlist-input", (INTBIG)np, FILETYPEALS);

				/* read netlist */
				simals_erase_model();
				if (simals_read_net_desc(np) != 0) break;
				if (simals_flatten_network() != 0) break;

				/* initialize display */
				simals_init_display();

				/* turn on this tool so that it can track changes to original circuit */
				aidturnon(sim_aid, 1);
				break;
#endif
#if SIMSILOS
			case SILOS:
				if (checkcap(sim_aid, SIMHASCOMM) == 0)
				{
					ttyputerr("Sorry, SILOS interface is not available");
					break;
				}
				sim_writesilnetlist(np);
				break;
#endif
#if SIMSPICE
			case SPICE:
				if (checkcap(sim_aid, SIMHASSPICE) == 0)
				{
					ttyputerr("Sorry, SPICE interface is not available");
					break;
				}
				sim_writespice(np);
				break;
#endif
#if SIMTEXSIM
			case TEXSIM:
				if (checkcap(sim_aid, SIMHASCOMM) == 0)
				{
					ttyputerr("Sorry, TEXSIM interface is not available");
					break;
				}
				sim_writetexnetlist(np);
				break;
#endif
#if SIMVERILOG
			case VERILOG:
				if (checkcap(sim_aid, SIMHASCOMM) == 0)
				{
					ttyputerr("Sorry, VERILOG interface is not available");
					break;
				}
				sim_writevernetlist(np);
				break;
#endif
		}
		return(0);
	}

	if (namesamen(pp, "resume", l) == 0 && l >= 1)
	{
		var = getvalkey((INTBIG)sim_aid, VAID, VINTEGER, sim_format);
		if (var == NOVARIABLE)
		{
			ttyputerr("No simulation is running");
			return(1);
		}
		switch (var->addr)
		{
#if SIMSIM
			case RNL:
			case RSIM:
			case ESIM:
				sim_resumesim(0);
				break;
#endif
			default:
				ttyputerr("Cannot resume this simulator");
				break;
		}
		return(0);
	}

	if (namesamen(pp, "not", l) == 0 && l >= 1)
	{
		if (count <= 1)
		{
			ttyputerr("Usage: tellaid simulation not OPTION");
			return(1);
		}
		l = strlen(pp = par[1]);
		if (namesamen(pp, "execute", l) == 0 && l >= 1)
		{
			(void)setvalkey((INTBIG)sim_aid, VAID, sim_dontrun, 1, VINTEGER|VDONTSAVE);
			ttyputmsg("Simulator will not be invoked (deck generation only)");
			return(0);
		}
		ttyputerr("Bad SIMULATION NOT option: %s", pp);
		return(1);
	}

	if (namesamen(pp, "execute", l) == 0 && l >= 1)
	{
		(void)setvalkey((INTBIG)sim_aid, VAID, sim_dontrun, 0, VINTEGER|VDONTSAVE);
		ttyputmsg("Simulator will be invoked");
		return(0);
	}

	if (namesamen(pp, "pointout", l) == 0 && l >= 1)
	{
		if (count <= 1)
		{
			ttyputerr("Usage: tellaid simulation pointout ELEMENT");
			return(1);
		}
		var = getvalkey((INTBIG)sim_aid, VAID, VINTEGER, sim_format);
		if (var != NOVARIABLE) switch (var->addr)
		{
#if SIMSIM
			case RNL:
			case RSIM:
			case ESIM:
			case COSMOS: return(sim_simpointout(par[1], (INTSML)(var->addr)));
#endif
		}
		ttyputerr("Current simulator cannot pointout netlist objects");
		return(1);
	}

	if (namesamen(pp, "window", l) == 0 && l >= 1)
	{
		if (count <= 1)
		{
			ttyputerr("Usage: tellaid simulation window COMMAND");
			return(1);
		}
		l = strlen(pp = par[1]);
		if (namesamen(pp, "2-state-display", l) == 0 && l >= 1)
		{
			sim_window_state &= ~FULLSTATE;
			ttyputmsgf("Simulation will show only 2 states in schematics window");
			return(0);
		}
		if (namesamen(pp, "12-state-display", l) == 0 && l >= 1)
		{
			sim_window_state |= FULLSTATE;
			ttyputmsgf("Simulation will show all 12 states in schematics window");
			return(0);
		}
		if (namesamen(pp, "advance-time", l) == 0 && l >= 1)
		{
			sim_window_state |= ADVANCETIME;
			ttyputmsgf("State changes will advance time to end of activity");
			return(0);
		}
		if (namesamen(pp, "freeze-time", l) == 0 && l >= 1)
		{
			sim_window_state &= ~ADVANCETIME;
			ttyputmsgf("State changes will not advance time");
			return(0);
		}
		if (namesamen(pp, "display-waveform", l) == 0 && l >= 1)
		{
			sim_window_state |= SHOWWAVEFORM;
			ttyputmsgf("Simulation will show waveform window");
			return(0);
		}
		if (namesamen(pp, "ignore-waveform", l) == 0 && l >= 1)
		{
			sim_window_state &= ~SHOWWAVEFORM;
			ttyputmsgf("Simulation will not show waveform window");
			return(0);
		}
		if (namesamen(pp, "color", l) == 0 && l >= 2)
		{
			if (count <= 2)
			{
				sim_window_displaycolor(99, 0);
				return(0);
			}
			l = strlen(pp = par[2]);
			if (count < 4)
			{
				ttyputerr("Usage: tellaid simulation window color %s COLORNAME", pp);
				return(1);
			}
			i = getecolor(par[3]);
			if (i < 0)
			{
				ttyputerr("Unknown color: %s", par[3]);
				return(1);
			}
			if (namesamen(pp, "off", l) == 0)
			{
				sim_window_displaycolor(OFF_STRENGTH, i);
				return(0);
			}
			if (namesamen(pp, "node", l) == 0)
			{
				sim_window_displaycolor(NODE_STRENGTH, i);
				return(0);
			}
			if (namesamen(pp, "gate", l) == 0)
			{
				sim_window_displaycolor(GATE_STRENGTH, i);
				return(0);
			}
			if (namesamen(pp, "power", l) == 0)
			{
				sim_window_displaycolor(VDD_STRENGTH, i);
				return(0);
			}
			if (namesamen(pp, "low", l) == 0)
			{
				sim_window_displaycolor(LOGIC_LOW, i);
				return(0);
			}
			if (namesamen(pp, "high", l) == 0)
			{
				sim_window_displaycolor(LOGIC_HIGH, i);
				return(0);
			}
			if (namesamen(pp, "undefined", l) == 0)
			{
				sim_window_displaycolor(LOGIC_X, i);
				return(0);
			}
			ttyputerr("Unknown strength: %s", pp);
			return(1);
		}
		if (namesamen(pp, "cursor", l) == 0 && l >= 2)
		{
			if (sim_window_isactive(&np) == 0)
			{
				ttyputerr("No simulator active");
				return(1);
			}
			if (count < 4)
			{
				ttyputerr("Usage: tellaid simulation window cursor WHICH TIME");
				return(1);
			}
			time = atof(par[3]);
			if (time < 0.0)
			{
				ttyputerr("Warning: time cannot be negative, set to 0 sec.");
				time = 0.0;
			}
			if (namesamen(par[2], "main", (INTSML)(strlen(par[2]))) == 0)
			{
				sim_window_setmaincursor(time);
				return(0);
			}
			if (namesamen(par[2], "extension", (INTSML)(strlen(par[2]))) == 0)
			{
				sim_window_setextensioncursor(time);
				return(0);
			}
			ttyputerr("Unknown cursor name: %s", par[2]);
			return(1);
		}
		if (namesamen(pp, "move", l) == 0 && l >= 1)
		{
			if ((sim_window_isactive(&np)&SIMWINDOWWAVEFORM) == 0)
			{
				ttyputerr("No simulator waveform window is active");
				return(1);
			}
			if (count < 3)
			{
				ttyputerr("Usage: tellaid simulation window move (left | right | up | down) [AMOUNT]");
				return(1);
			}

			l = strlen(pp = par[2]);
			if (namesamen(pp, "left", l) == 0 || namesamen(pp, "right", l) == 0)
			{
				sim_window_gettimerange(&mintime, &maxtime);
				size = maxtime - mintime;
				if (count < 2) time = size / 2.0; else
				{
					time = atof(par[3]);
					if (time <= 0.0)
					{
						ttyputerr("Window movement time must be greater than 0 seconds");
						return(1);
					}
				}

				if (namesamen(pp, "left", l) == 0)
				{
					mintime -= time;
					if (mintime < 0.0) mintime = 0.0;
				} else
				{
					mintime += time;
				}
				maxtime = mintime + size;
				sim_window_settimerange(mintime, maxtime);
				sim_window_redraw();
				return(0);
			}
			if (namesamen(pp, "up", l) == 0 || namesamen(pp, "down", l) == 0)
			{
				signals = sim_window_getlines();
				if (count < 2) amount = signals / 2; else
				{
					amount = myatoi(par[3]);
					if (amount <= 0)
					{
						ttyputerr("Window movement time must be greater than 0 signals");
						return(1);
					}
				}
#if SIMALS
				simals_set_current_level();
#endif
				sim_window_redraw();
				return(0);
			}

			ttyputerr("Invalid movement direction: %s", pp);
			return(1);
		}
		if (namesamen(pp, "traces", l) == 0 && l >= 1)
		{
			if ((sim_window_isactive(&np)&SIMWINDOWWAVEFORM) == 0)
			{
				ttyputerr("No simulator waveform window is active");
				return(1);
			}
			if (count < 3)
			{
				ttyputerr("Usage: tellaid simulation window traces (more | less | set A)");
				return(1);
			}
			l = strlen(pp = par[2]);

			if (namesamen(pp, "more", l) == 0)
			{
				sim_window_setlines((INTSML)(sim_window_getlines()+1));
				sim_window_redraw();
				return(0);
			}
			if (namesamen(pp, "less", l) == 0)
			{
				i = sim_window_getlines();
				if (i <= 1)
				{
					ttyputerr("Must be at least 1 signal on the display");
					return(1);
				}
				sim_window_setlines((INTSML)(i+1));
				sim_window_redraw();
				return(0);
			}
			if (namesamen(pp, "set", l) == 0)
			{
				if (count < 4)
				{
					ttyputerr("Usage: tellaid simulation window traces set AMOUNT");
					return(1);
				}
				i = myatoi(par[3]);
				if (i < 1)
				{
					ttyputerr("Must be at least 1 signal on the display");
					return(1);
				}
				sim_window_setlines(i);
				sim_window_redraw();
				return(0);
			}
			ttyputerr("Bad tellaid simulation window TRACES option: %s", pp);
			return(1);
		}
		if (namesamen(pp, "zoom", l) == 0 && l >= 1)
		{
			if ((sim_window_isactive(&np)&SIMWINDOWWAVEFORM) == 0)
			{
				ttyputerr("No simulator waveform window is active");
				return(1);
			}
			if (count < 3)
			{
				ttyputerr("Usage: tellaid simulation window zoom (in | out | window | cursor)");
				return(1);
			}
			l = strlen(pp = par[2]);

			if (namesamen(pp, "cursor", l) == 0)
			{
				main = sim_window_getmaincursor();
				extension = sim_window_getextensioncursor();
				if (main == extension) return(0);
				if (main > extension)
				{
					size = (main-extension) / 20.0;
					maxtime = main + size;
					mintime = extension - size;
				} else
				{
					size = (extension-main) / 20.0;
					maxtime = extension + size;
					mintime = main - size;
				}
				sim_window_settimerange(mintime, maxtime);
				sim_window_redraw();
				return(0);
			}

			if (namesamen(pp, "in", l) == 0)
			{
				if (count < 4) factor = 2.0; else
				{
					factor = (float)myatoi(par[3]);
					if (factor <= 1)
					{
						ttyputerr("Zoom factor must be integer greater than 1");
						return(1);
					}
				}

				sim_window_gettimerange(&mintime, &maxtime);
				size = (maxtime - mintime) / factor;
				mintime = sim_window_getmaincursor() - (size / 2.0);
				if (mintime < 0.0) mintime = 0.0;
				maxtime = mintime + size;
				sim_window_settimerange(mintime, maxtime);
				sim_window_redraw();
				return(0);
			}

			if (namesamen(pp, "out", l) == 0)
			{
				if (count < 4) factor = 2.0; else
				{
					factor = (float)myatoi(par[3]);
					if (factor <= 1)
					{
						ttyputerr("Zoom factor must be integer greater than 1");
						return(1);
					}
				}

				sim_window_gettimerange(&mintime, &maxtime);
				size = (maxtime - mintime) * factor;
				mintime = sim_window_getmaincursor() - (size / 2.0);
				if (mintime < 0.0) mintime = 0.0;
				maxtime = mintime + size;
				sim_window_settimerange(mintime, maxtime);
				sim_window_redraw();
				return(0);
			}

			if (namesamen(pp, "window", l) == 0)
			{
				if (count < 4)
				{
					ttyputerr("Usage: tellaid simulation window zoom window MIN [MAX]");
					return(1);
				}

				mintime = atof(par[3]);
				maxtime = 0.0;
				if (count > 4) maxtime = atof(par[4]);
				if (mintime == maxtime)
				{
					ttyputerr("ERROR: Window size must be greater than 0");
					return(1);
				}
				if (mintime > maxtime)
				{
					time = maxtime;
					maxtime = mintime;
					mintime = time;
				}
				sim_window_settimerange(mintime, maxtime);
				sim_window_redraw();
				return(0);
			}

			ttyputerr("Bad tellaid simulation window ZOOM option: %s", pp);
			return(1);
		}
		ttyputerr("Unknown simulation window command: %s", pp);
		return(1);
	}

#if SIMALS
	if (namesamen(pp, "als", l) == 0 && l >= 1)
	{
		/* quit if ALS not available */
		if (checkcap(sim_aid, SIMHASALS) == 0)
		{
			ttyputerr("Sorry, simulation is not available");
			return(1);
		}

		simals_com_comp((INTSML)(count-1), &par[1]);
		return(0);
	}
#endif  /* SIMALS */

#if SIMSPICE
	if (namesamen(pp, "spice", l) == 0 && l >= 2)
	{
		/* quit if SPICE not available */
		if (checkcap(sim_aid, SIMHASSPICE) == 0)
		{
			ttyputerr("Sorry, SPICE simulation is not available");
			return(1);
		}

		l = strlen(pp = par[1]);
		if (namesamen(pp, "level", l) == 0)
		{
			if (count >= 3)
			{
				i = atoi(par[2]);
				if (i <= 0 || i > MAXSPICELEVEL) i = 1;
				(void)setvalkey((INTBIG)sim_aid, VAID, sim_spice_level, i, VINTEGER|VDONTSAVE);
			}
			var = getvalkey((INTBIG)sim_aid, VAID, VINTEGER, sim_spice_level);
			if (var == NOVARIABLE) i = 1; else i = var->addr;
			ttyputmsgf("Simulation level set to %d", i);
			return(0);
		}
		if (namesamen(pp, "save-output", l) == 0)
		{
			if (count < 3)
			{
				ttyputerr("Should supply a file name to write");
				par[2] = "electric_def.spo";
			}
			(void)setvalkey((INTBIG)sim_aid, VAID, sim_listingfile, (INTBIG)par[2], VSTRING|VDONTSAVE);
			ttyputmsgf("Simulation output will go to file %s", par[2]);
			return(0);
		}
		if (namesamen(pp, "format", l) == 0)
		{
			var = getvalkey((INTBIG)sim_aid, VAID, VINTEGER, sim_spice_state);
			if (var != NOVARIABLE) spicestate = var->addr; else spicestate = 0;
			if (count <= 2)
			{
				switch(spicestate&SPICETYPE)
				{
					case SPICE2:      ttyputmsg("SPICE decks for SPICE 2");   break;
					case SPICE3:      ttyputmsg("SPICE decks for SPICE 3");   break;
					case SPICEHSPICE: ttyputmsg("SPICE decks for HSPICE");    break;
				}
				return(0);
			}
			l = strlen(pp = par[2]);
			if (namesamen(pp, "2", l) == 0)
			{
				(void)setvalkey((INTBIG)sim_aid, VAID, sim_spice_state,
					(spicestate & ~SPICETYPE) | SPICE2, VINTEGER|VDONTSAVE);
				return(0);
			}
			if (namesamen(pp, "3", l) == 0)
			{
				(void)setvalkey((INTBIG)sim_aid, VAID, sim_spice_state,
					(spicestate & ~SPICETYPE) | SPICE3, VINTEGER|VDONTSAVE);
				return(0);
			}
			if (namesamen(pp, "hspice", l) == 0)
			{
				(void)setvalkey((INTBIG)sim_aid, VAID, sim_spice_state,
					(spicestate & ~SPICETYPE) | SPICEHSPICE, VINTEGER|VDONTSAVE);
				return(0);
			}
		}
		if (namesamen(pp, "parse-output", l) == 0)
		{
			if (count < 3)
			{
				ttyputerr("Must supply a SPICE output file to read");
				return(1);
			}
			sim_spice_execute("", par[2]);
			return(0);
		}
		if (namesamen(pp, "not", l) == 0)
		{
			if (count <= 2)
			{
				ttyputerr("Usage: tellaid simulation spice not OPTION");
				return(1);
			}
			l = strlen(pp = par[2]);
			if (namesamen(pp, "resistance", l) == 0 || namesamen(pp, "capacitances", l) == 0)
			{
				var = getvalkey((INTBIG)sim_aid, VAID, VINTEGER, sim_spice_state);
				if (var != NOVARIABLE)
					(void)setvalkey((INTBIG)sim_aid, VAID, sim_spice_state,
						var->addr & ~SPICERESISTANCE, VINTEGER|VDONTSAVE);
				ttyputmsgf("SPICE decks will not print parasitics");
				return(0);
			}
			if (namesamen(pp, "plot", l) == 0)
			{
				var = getvalkey((INTBIG)sim_aid, VAID, VINTEGER, sim_spice_state);
				if (var != NOVARIABLE)
					(void)setvalkey((INTBIG)sim_aid, VAID, sim_spice_state,
						var->addr & ~SPICEPLOT, VINTEGER|VDONTSAVE);
				ttyputmsgf("SPICE decks will not do plots");
				return(0);
			}
			if (namesamen(pp, "use-nodenames", l) == 0)
			{
				var = getvalkey((INTBIG)sim_aid, VAID, VINTEGER, sim_spice_state);
				if (var != NOVARIABLE)
					(void)setvalkey((INTBIG)sim_aid, VAID, sim_spice_state,
						var->addr & ~SPICENODENAMES, VINTEGER|VDONTSAVE);
				ttyputmsgf("SPICE decks will contain node numbers");
				return(0);
			}
			ttyputerr("Bad SPICE NOT option: %s", pp);
			return(1);
		}
		if (namesamen(pp, "resistance", l) == 0 || namesamen(pp, "capacitances", l) == 0)
		{
			var = getvalkey((INTBIG)sim_aid, VAID, VINTEGER, sim_spice_state);
			if (var != NOVARIABLE)
				(void)setvalkey((INTBIG)sim_aid, VAID, sim_spice_state,
					var->addr | SPICERESISTANCE, VINTEGER|VDONTSAVE);
			ttyputmsgf("SPICE decks will print parasitics");
			return(0);
		}
		if (namesamen(pp, "plot", l) == 0)
		{
			var = getvalkey((INTBIG)sim_aid, VAID, VINTEGER, sim_spice_state);
			if (var != NOVARIABLE)
				(void)setvalkey((INTBIG)sim_aid, VAID, sim_spice_state,
					var->addr | SPICEPLOT, VINTEGER|VDONTSAVE);
			ttyputmsgf("SPICE decks will do plots");
			return(0);
		}
		if (namesamen(pp, "debug-toggle", l) == 0)
		{
			if ((sim_spice_debug&SPICEDEBUG) == 0)
			{
				sim_spice_debug |= SPICEDEBUG;
				ttyputmsgf("SPICE deck generation debugging output on");
			} else
			{
				sim_spice_debug &= ~SPICEDEBUG;
				ttyputmsgf("SPICE deck generation debugging output off");
			}
			return(0);
		}
		if (namesamen(pp, "use-nodenames", l) == 0)
		{
			var = getvalkey((INTBIG)sim_aid, VAID, VINTEGER, sim_spice_state);
			if (var != NOVARIABLE)
				(void)setvalkey((INTBIG)sim_aid, VAID, sim_spice_state,
					var->addr | SPICENODENAMES, VINTEGER|VDONTSAVE);
			ttyputmsgf("SPICE decks will contain node names");
			return(0);
		}
		ttyputerr("Bad SPICE option: %s", pp);
		return(1);
	}
#endif  /* SIMSPICE */

	/* unknown command */
	ttyputerr("Unknown simulation option: %s", pp);
	return(1);
}

/* inquire the simulation aid */
INTBIG sim_request(char *command, va_list ap) { return(0); }
void sim_slice(void)
{
	NODEPROTO *np;

	if (sim_circuitchanged != 0)
	{
		if (sim_window_isactive(&np) != 0)
		{
			sim_window_stopsimulation();
			aidturnoff(sim_aid, 0);
			ttyputmsg("Circuit changed: simulation stopped");
		}
		sim_circuitchanged = 0;
	}
}

void sim_checktostopsimulation(NODEPROTO *np)
{
	NODEPROTO *onp;

	if (sim_window_isactive(&onp) == 0) return;
	if (onp != np) return;
	sim_circuitchanged = 1;
}

void sim_examinenodeproto(NODEPROTO *np) {}
void sim_startbatch(AIDENTRY *source) {}
void sim_endbatch(void) {}
void sim_startobjectchange(INTBIG addr, INTBIG type) {}
void sim_endobjectchange(INTBIG addr, INTBIG type) {}

void sim_modifynodeinst(NODEINST *ni, INTBIG olx, INTBIG oly, INTBIG ohx, INTBIG ohy,
	INTSML orot, INTSML otran)
{
	sim_checktostopsimulation(ni->parent);
}

void sim_modifyarcinst(ARCINST *ai, INTBIG oxA, INTBIG oyA, INTBIG oxB, INTBIG oyB,
	INTBIG owid, INTBIG olen)
{
	sim_checktostopsimulation(ai->parent);
}

void sim_modifyportproto(PORTPROTO *pp, NODEINST *oni, PORTPROTO *opp)
{
	sim_checktostopsimulation(pp->parent);
}

void sim_modifynodeproto(NODEPROTO *np) {}
void sim_modifydescript(VARIABLE *var, INTBIG odes) {}

void sim_newobject(INTBIG addr, INTBIG type)
{
	switch (type&VTYPE)
	{
		case VNODEINST:  sim_checktostopsimulation(((NODEINST *)addr)->parent);   break;
		case VARCINST:   sim_checktostopsimulation(((ARCINST *)addr)->parent);    break;
		case VPORTPROTO: sim_checktostopsimulation(((PORTPROTO *)addr)->parent);  break;
	}
}

void sim_killobject(INTBIG addr, INTBIG type)
{
	switch (type&VTYPE)
	{
		case VNODEINST:  sim_checktostopsimulation(((NODEINST *)addr)->parent);   break;
		case VARCINST:   sim_checktostopsimulation(((ARCINST *)addr)->parent);    break;
		case VPORTPROTO: sim_checktostopsimulation(((PORTPROTO *)addr)->parent);  break;
	}
}

void sim_newvariable(INTBIG addr, INTBIG type, INTBIG key, INTBIG newtype)
{
	REGISTER char *name;
	ARCINST *ai;

	/* detect change to "ai->userbits" to see if arc negation changed */
	if ((newtype&VCREF) == 0) return;
	if ((type&VTYPE) != VARCINST) return;
	name = changedvariablename(type, key, newtype);
	if (namesame(name, "userbits") != 0) return;

	ai = (ARCINST *)addr;
	if (ai != sim_modifyarc) return;
	if ((ai->userbits&ISNEGATED) != (sim_modifyarcbits&ISNEGATED))
		sim_checktostopsimulation(ai->parent);
}

void sim_killvariable(INTBIG addr, INTBIG type, INTBIG key, INTBIG oldaddr, INTBIG oldtype,
	INTBIG olddescript)
{
	REGISTER char *name;

	/* detect change to "ai->userbits" and save former value */
	if ((oldtype&VCREF) == 0) return;
	if ((type&VTYPE) != VARCINST) return;
	name = changedvariablename(type, key, oldtype);
	if (namesame(name, "userbits") != 0) return;

	sim_modifyarc = (ARCINST *)addr;
	sim_modifyarcbits = sim_modifyarc->userbits;
}

void sim_modifyvariable(INTBIG addr, INTBIG type, INTBIG key, INTBIG vartype, INTBIG index,
	INTBIG oldvalue) {}
void sim_insertvariable(INTBIG addr, INTBIG type, INTBIG key, INTBIG vartype, INTBIG index) {}
void sim_deletevariable(INTBIG addr, INTBIG type, INTBIG key, INTBIG vartype, INTBIG index,
	INTBIG oldvalue) {}
void sim_readlibrary(LIBRARY *lib) {}
void sim_eraselibrary(LIBRARY *lib) {}
void sim_writelibrary(LIBRARY *lib, INTSML pass) {}

#endif  /* SIMAID */
