/*
 * Electric(tm) VLSI Design System
 *
 * File: simspicerun.c
 * Companion file to simspice.c
 * Written by: Steven M. Rubin, Electric Editor Incorporated
 *
 * Copyright (c) 1998 Electric Editor Incorporated.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Electric Editor Incorporated
 * 23470 Sunset Drive, Suite 108
 * Los Gatos, California 95033
 * support@electriceditor.com
 */

#include "config.h"
#if SIMSPICE & SIMAID

#include "global.h"
#include "sim.h"
#include "efunction.h"
#include "usr.h"
#include "edialogs.h"

#include <ctype.h>
#include <math.h>
#include <signal.h>
#include <errno.h>
#ifdef HAVE_VFORK_H
#  include <vfork.h>		/* needed by SUN optimiser */
#endif
#ifdef HAVE_UNISTD_H
#  include <unistd.h>
#endif
#ifdef HAVE_FCNTL_H
#  include <fcntl.h>
#endif

#ifdef WIN32
#  define pipe _pipe
#endif

/******************** SPICE NUMBER OUTPUT QUEUE ********************/

#define NONUMBERS  ((NUMBERS *)-1)
#define NUMBERSBLOCKSIZE       50	/* number to "emalloc" at a time */

typedef struct Inumbers
{
	float  time;
	float *list;
	INTSML count;
	struct Inumbers *nextnumbers;
} NUMBERS;

NUMBERS *sim_numbersfree = NONUMBERS;

extern INTSML spice_state;			/* value of "SIM_spice_state" */
extern char sim_deck_facetname[];	/* name extracted from .SPO file */
#ifdef MACOS
  static FILE *sim_macfromsim;
#endif

/* the raw data from the SPICE simulator */
static NUMBERS *sim_window_numbers = NONUMBERS;		/* current list */
static char   **sim_window_signames;
static INTSML  *sim_window_sigtypes;
static float   *sim_window_time, *sim_window_val;
static INTSML   sim_window_signals = 0;				/* entries in "signames/sigtypes" */
static INTSML   sim_window_iter;
static INTSML   sim_window_limit = 0;				/* entries in "time/val" */
static float    sim_window_lowy, sim_window_highy;	/* range of an entry */
static INTBIG   sim_window_filepos;

/* prototypes for local routines */
NUMBERS *sim_allocnumbers(INTSML);
void sim_freenumbers(NUMBERS*);
void sim_spice_terminate(FILE*, char*, char*);
INTSML sim_getfromsimulator(void);
INTSML sim_window_topofsignals(char**);
char *sim_window_nextsignals(void);
INTSML sim_spice_plotvalues(INTSML);
INTSML sim_window_ensurespace(INTSML);

/*
 * routine to allocate a new numbers module from the pool (if any) or memory
 */
NUMBERS *sim_allocnumbers(INTSML total)
{
	REGISTER NUMBERS *d;
	REGISTER INTSML i, j;

	if (sim_numbersfree == NONUMBERS)
	{
		for(i=NUMBERSBLOCKSIZE; i>0; i /= 2)
		{
			d = (NUMBERS *)emalloc(((sizeof (NUMBERS)) * i), sim_aid->cluster);
			if (d != 0) break;
		}
		if (d == 0) return(NONUMBERS);
		for(j=0; j<i; j++)
			if (j == 0) d[j].nextnumbers = NONUMBERS; else
				d[j].nextnumbers = &d[j-1];
		sim_numbersfree = &d[i-1];
	}

	/* take module from free list */
	d = sim_numbersfree;
	sim_numbersfree = (NUMBERS *)d->nextnumbers;
	d->list = (float *)emalloc(((sizeof (float)) * total), sim_aid->cluster);
	if (d->list == 0) return(NONUMBERS);
	d->count = total;
	return(d);
}

/*
 * routine to return numbers module "d" to the pool of free modules
 */
void sim_freenumbers(NUMBERS *d)
{
	efree((char *)d->list);
	d->nextnumbers = sim_numbersfree;
	sim_numbersfree = d;
}

/******************** SPICE EXECUTION AND PLOTTING ********************/

#define MAXTRACES 7

/*
 * routine to execute SPICE with "infile" as the input deck file, "outfile"
 * as the output listing file, and "spicetype" as either 2 for SPICE2 or
 * 3 for SPICE3 execution.  If "infile" is null, do not run the simulator,
 * but just read "outfile" as the output listing.  If "infile" is not null,
 * run SPICE on that file.  If both "infile" and "outfile" are not null,
 * run SPICE on the "infile" and save the output in "outfile".
 */
void sim_spice_execute(char *infile, char *outfile)
{
	char line[200], *ptr;
#ifdef MACOS
	char *truename;
#endif
	REGISTER INTSML j, k, l, datamode, knows, past_end, important,
		sweepcnt, nodcnt, numnoi, cndcnt, first;
	REGISTER INTBIG i, filelen;
	float numbers[20], lastt;
	NODEPROTO *np;
	NUMBERS *num, *numend;
	REGISTER FILE *outputfile;
	VARIABLE *var;
	extern DIALOG usr_progressdialog;

	outputfile = NULL;
	filelen = -1;
	if (*infile == 0)
	{
		/* parsing output into simulation window: make sure none is active */
		if (sim_window_isactive(&np) != 0)
		{
			ttyputerr("Only one simulation window can run at a time");
			ttyputmsg("Terminate the current one before reading SPICE output");
			return;
		}

		/* no input deck, simply use output listing */
#ifdef  MACOS
		sim_macfromsim = xopen(truepath(outfile), FILETYPESPICEOUT, "", &truename);
		if (sim_macfromsim == NULL)
		{
			ttyputerr("Cannot read %s", truename);
			return;
		}
		filelen = filesize(sim_macfromsim);
#else
		i = open(truepath(outfile), 0);
		if (i < 0)
		{
			ttyputerr("Cannot read %s", outfile);
			return;
		}
		filelen = lseek(i, 0, 2);
		(void)lseek(i, 0, 0);
		sim_fromsim[0] = i;
#endif
	} else
	{
		/* input deck exists, run SPICE on it */
#if defined(MACOS) || defined(WIN32)
		ttyputmsg("Run SPICE directly please");
		return;
#endif
#ifdef ONUNIX
		/* for communication with simulator, read fromsim[0] */
		(void)pipe(sim_fromsim);

		if ((sim_process = myfork()) == 0)
		{
			REGISTER INTBIG save0, save1, save2;

			save0 = dup(0);  (void)close(0);  (void)open(infile, 0);
			save1 = dup(1);  (void)close(1);  (void)dup(sim_fromsim[1]);
			save2 = dup(2);  (void)close(2);  (void)open("/dev/null", 1);
#  ifdef REMOTESPICE
			/* spice on a remote machine */
			execl("/usr/ucb/rsh", "rsh", SPICEHOST, SPICENAME, (char *)0);
#  else
			ptr = getenv("SPICELOC");
			if (ptr == NULL) ptr = SPICELOC;
			execl(ptr, SPICENAME, 0);
#  endif

			/* try using the search path to find an executable spice */
			execlp(SPICENAME, SPICENAME, 0);

			(void)close(0);   (void)dup(save0);
			(void)close(1);   (void)dup(save1);
			(void)close(2);   (void)dup(save2);
			ttyputerr("Cannot run spice");
			_exit(1);
		}

		/* close the file descriptor so the pipe will terminate when done */
		(void)close(sim_fromsim[1]);

		/* prepare output file if requested */
		if (*outfile != 0)
		{
			char *truename;

			outputfile = xcreate(outfile, FILETYPESPICEOUT, "SPICE output file", &truename);
			if (outputfile == NULL)
			{
				if (truename != 0) ttyputerr("Cannot write SPICE output file %s", truename);
			}
		}
#endif
	}

	/*
	 * Determine the "spice_state".  If we're doing a write-execute-parse,
	 * then we've already done this.  If we're just doing a parse, we have not.
	 */
	var = getvalkey((INTBIG)sim_aid, VAID, VINTEGER, sim_spice_state);
	if (var != NOVARIABLE) spice_state = var->addr; else spice_state = 0;

	/* free any former data */
	if (sim_window_signals > 0)
	{
		for(i=0; i<sim_window_signals; i++) efree(sim_window_signames[i]);
		efree((char *)sim_window_signames);
		efree((char *)sim_window_sigtypes);
		sim_window_signals = 0;
	}
	while (sim_window_numbers != NONUMBERS)
	{
		num = sim_window_numbers;
		sim_window_numbers = sim_window_numbers->nextnumbers;
		sim_freenumbers(num);
	}

	/* show progress if possible */
	if (filelen > 0)
	{
		if (DiaInitDialog(&usr_progressdialog) != 0) return;
		DiaPercent(1, 0);
	}
	sim_window_filepos = 0;

	/* read from the simulator and list on the status terminal */
	if ((spice_state & SPICETYPE) == SPICE2)
	{
		/* parse SPICE 2 format output */
		past_end = datamode = 0;
		first = 1;
		numend = NONUMBERS;
		ptr = line;
		for(;;)
		{
			if (stopping("SPICE") != 0) break;
			i = sim_getfromsimulator();
			if (i == EOF) break;
			*ptr = i;
			if (*ptr == '\n')
			{
				*ptr++ = 0;   *ptr = 0;
				if (filelen > 0) DiaPercent(1, sim_window_filepos*100/filelen);
				if (first != 0)
				{
					/* check the first line for HSPICE format possibility */
					first = 0;
					if (strlen(line) >= 20 && line[16] == '9' && line[17] == '0' &&
						line[18] == '0' && line[19] == '7')
					{
						ttyputerr("This is an HSPICE file, not a SPICE2 file");
						ttyputerr("Change the SPICE format and reread");
						if (filelen > 0) DiaDoneDialog();
						(void)sim_spice_terminate(outputfile, outfile, infile);
						return;
					}
				}
				if (*infile != 0)
				{
					if (outputfile == NULL) ttyputmsg("%s", (isprint(*line) ? line : &line[1]));
						else sim_spice_xprintf(outputfile, "%s\n", line);
				}
												/*look for a facet name */
				if ((sim_deck_facetname[0] == '\0') && (namesamen(line, "*** FACET",9) == 0))
					if ((i = sscanf(line+9,"%s",sim_deck_facetname)) != 1)
						sim_deck_facetname[0] = '\0';
				if (namesamen(line, ".END", 4) == 0 && namesamen(line, ".ENDS", 5) != 0)
				{
					past_end = 1;
					ptr = line;
					continue;
				}
				if (past_end && datamode == 0)
				{
					if ((isspace(line[0]) || line[0] == '-') && isdigit(line[1]))
						datamode = 1;
				}
				if (past_end && datamode == 1)
				{
					if (!((isspace(line[0]) || line[0] == '-') && isdigit(line[1])))
					{
						datamode = 0;
						past_end = 0;
					}
				}
				if (datamode != 0)
				{
					ptr = line;
					sim_window_signals = 0;
					for(;;)
					{
						while (isspace(*ptr)) ptr++;
						if (*ptr == 0) break;
						numbers[sim_window_signals++] = atof(ptr);
						while (*ptr != ' ' && *ptr != 0) ptr++;
					}
					if (sim_window_signals > 1)
					{
						sim_window_signals--;
						num = sim_allocnumbers(sim_window_signals);
						if (num == NONUMBERS) break;
						num->time = numbers[0];
						for(i=0; i<sim_window_signals; i++) num->list[i] = numbers[i+1];
						if (numend == NONUMBERS) sim_window_numbers = num; else
							numend->nextnumbers = num;
						num->nextnumbers = NONUMBERS;
						numend = num;
					}
				}
				ptr = line;
				continue;
			}
			ptr++;
		}

		/* generate dummy names */
		important = sim_window_signals;
		sim_window_signames = (char **)emalloc(sim_window_signals * (sizeof (char *)), sim_aid->cluster);
		sim_window_sigtypes = (INTSML *)emalloc(sim_window_signals * SIZEOFINTSML, sim_aid->cluster);
		if (sim_window_signames == 0 || sim_window_sigtypes == 0)
		{
			/* terminate execution so we can restart simulator */
			if (filelen > 0) DiaDoneDialog();
			(void)sim_spice_terminate(outputfile, outfile, infile);
			return;
		}
		for(i=0; i<sim_window_signals; i++)
		{
			(void)sprintf(line, "Signal %d", i+1);
			(void)allocstring(&sim_window_signames[i], line, sim_aid->cluster);
		}
	} else if ((spice_state & SPICETYPE) == SPICE3)
	{
		/* parse SPICE 3 format output */
		numend = NONUMBERS;
		ptr = line;
		first = 1;
		for(;;)
		{
			if (stopping("SPICE") != 0) break;
			i = sim_getfromsimulator();
			if (i == EOF) break;
			*ptr = i;
			if (*ptr == '\n')
			{
				*ptr = 0;
				if (filelen > 0) DiaPercent(1, sim_window_filepos*100/filelen);

				if (first != 0)
				{
					/* check the first line for HSPICE format possibility */
					first = 0;
					if (strlen(line) >= 20 && line[16] == '9' && line[17] == '0' &&
						line[18] == '0' && line[19] == '7')
					{
						ttyputerr("This is an HSPICE file, not a SPICE3 file");
						ttyputerr("Change the SPICE format and reread");
						if (filelen > 0) DiaDoneDialog();
						(void)sim_spice_terminate(outputfile, outfile, infile);
						return;
					}
				}

				/* skip first word if there is an "=" in the line */
				for(ptr = line; *ptr != 0; ptr++) if (*ptr == '=') break;
				if (*ptr == 0) ptr = line; else ptr += 3;

				/* read the data values */
				knows = 0;
				lastt = 0.0;
				for(;;)
				{
					while (*ptr == ' ' || *ptr == '\t') ptr++;
					if (*ptr == 0 || *ptr == ')') break;
					if (sim_window_signals == 0)
					{
						num = sim_allocnumbers(MAXTRACES);
						num->time = atof(ptr);
						if (numend == NONUMBERS) sim_window_numbers = num; else
						{
							numend->nextnumbers = num;
							if (num->time <= lastt && knows == 0)
							{
								ttyputerr("First trace not increasing values");
								ttyputmsg("   this should be 'time'");
								knows++;
							}
						}
						lastt = num->time;
						num->nextnumbers = NONUMBERS;
						numend = num;
					} else
					{
						if (num == NONUMBERS) ttyputmsg("Line %d of data has too many values",
							sim_window_signals); else
						{
							if (sim_window_signals <= MAXTRACES)
								num->list[sim_window_signals-1] = atof(ptr);
							num = num->nextnumbers;
						}
					}
					while (*ptr != ' ' && *ptr != '\t' && *ptr != 0) ptr++;
				}

				/* see if there is an ")" at the end of the line */
				if (line[strlen(line)-1] == ')')
				{
					/* advance to the next value for subsequent reads */
					if (sim_window_signals != 0 && num != NONUMBERS)
						ttyputmsg("Line %d of data has too few values", sim_window_signals);
					sim_window_signals++;
					num = sim_window_numbers;
				}

				if (*infile != 0)
				{
					if (outputfile == NULL)
						ttyputmsg("%s", (isprint(*line) ? line : &line[1])); else
							sim_spice_xprintf(outputfile, "%s\n", line);
				}
				ptr = line;
				continue;
			}
			ptr++;
		}

		/* generate dummy names */
		important = sim_window_signals;
		sim_window_signames = (char **)emalloc(sim_window_signals * (sizeof (char *)), sim_aid->cluster);
		sim_window_sigtypes = (INTSML *)emalloc(sim_window_signals * SIZEOFINTSML, sim_aid->cluster);
		if (sim_window_signames == 0 || sim_window_sigtypes == 0)
		{
			/* terminate execution so we can restart simulator */
			if (filelen > 0) DiaDoneDialog();
			(void)sim_spice_terminate(outputfile, outfile, infile);
			return;
		}
		for(i=0; i<sim_window_signals; i++)
		{
			(void)sprintf(line, "Signal %d", i+1);
			(void)allocstring(&sim_window_signames[i], line, sim_aid->cluster);
		}
	} else if ((spice_state & SPICETYPE) == SPICEHSPICE)
	{
		/* parse HSPICE format output */
		/* get number of nodes, special items, and conditions */
		line[4] = 0;
		for(j=0; j<4; j++) line[j] = sim_getfromsimulator();
		nodcnt = atoi(line);
		for(j=0; j<4; j++) line[j] = sim_getfromsimulator();
		numnoi = atoi(line);
		for(j=0; j<4; j++) line[j] = sim_getfromsimulator();
		cndcnt = atoi(line);
		for(j=0; j<4; j++) line[j] = sim_getfromsimulator();
		sim_window_signals = numnoi + nodcnt - 1;

		/* get version number */
		for(j=0; j<4; j++) line[j] = sim_getfromsimulator();
		j = atoi(line);
		if (j != 9007)
		{
			ttyputerr("Sorry, cannot read this version of HSPICE output");
			if (filelen > 0) DiaDoneDialog();

			/* terminate execution so we can restart simulator */
			(void)sim_spice_terminate(outputfile, outfile, infile);
			return;
		}

		/* ignore the unused/title information (4+72 characters over line break) */
		for(j=0; j<76; j++)
		{
			k = sim_getfromsimulator();
			if (k == '\n') j--;
		}

		/* ignore the date/time information (16 characters) */
		for(j=0; j<16; j++) (void)sim_getfromsimulator();

		/* ignore the copywrite information (72 characters over line break) */
		for(j=0; j<72; j++)
		{
			k = sim_getfromsimulator();
			if (k == '\n') j--;
		}

		/* get number of sweeps */
		line[4] = 0;
		for(j=0; j<4; j++) line[j] = sim_getfromsimulator();
		sweepcnt = atoi(line);

		/* ignore the Monticarlo information (76 characters over line break) */
		for(j=0; j<76; j++)
		{
			k = sim_getfromsimulator();
			if (k == '\n') j--;
		}
		if (filelen > 0) DiaPercent(1, sim_window_filepos*100/filelen);

		/* get the type of each signal */
		important = numnoi;
		sim_window_signames = (char **)emalloc(sim_window_signals * (sizeof (char *)), sim_aid->cluster);
		sim_window_sigtypes = (INTSML *)emalloc(sim_window_signals * SIZEOFINTSML, sim_aid->cluster);
		if (sim_window_signames == 0 || sim_window_sigtypes == 0)
		{
			/* terminate execution so we can restart simulator */
			if (filelen > 0) DiaDoneDialog();
			(void)sim_spice_terminate(outputfile, outfile, infile);
			return;
		}
		line[8] = 0;
		for(k=0; k<=sim_window_signals; k++)
		{
			for(j=0; j<8; j++)
			{
				l = sim_getfromsimulator();
				line[j] = l;
				if (l == '\n') j--;
			}
			if (k == 0) continue;
			if (k < nodcnt) l = k + numnoi - 1; else l = k - nodcnt;
			sim_window_sigtypes[l] = atoi(line);
		}
		line[16] = 0;
		for(k=0; k<=sim_window_signals; k++)
		{
			for(j=0; j<16; j++)
			{
				l = sim_getfromsimulator();
				line[j] = l;
				if (l == '\n') j--;
			}
			for(j=15; j>0; j--) if (line[j] != ' ') break;
			line[j+1] = 0;
			if (k == 0) continue;
			if (k < nodcnt) l = k + numnoi - 1; else l = k - nodcnt;
			(void)allocstring(&sim_window_signames[l], line, sim_aid->cluster);
		}

		/* ensure the end-of-header */
		for(j=0; j<16; j++)
		{
			l = sim_getfromsimulator();
			line[j] = l;
			if (l == '\n') j--;
		}
		for(j=15; j>0; j--) if (line[j] != ' ') break;
		line[j+1] = 0;
		if (strcmp(line, "$&%#") != 0)
		{
			ttyputerr("HSPICE header improperly terminated");
			if (filelen > 0) DiaDoneDialog();

			/* terminate execution so we can restart simulator */
			(void)sim_spice_terminate(outputfile, outfile, infile);
			return;
		}
		for(;;)
		{
			l = sim_getfromsimulator();
			if (l == '\n') break;
		}

		/* now read the data */
		sim_window_numbers = numend = NONUMBERS;
		for(;;)
		{
			if (stopping("SPICE") != 0) break;
			if (filelen > 0) DiaPercent(1, sim_window_filepos*100/filelen);

			/* get the first number, see if it terminates */
			line[11] = 0;
			for(j=0; j<11; j++)
			{
				l = sim_getfromsimulator();
				line[j] = l;
				if (l == '\n') j--;
			}
			if (strcmp(line, "0.10000E+31") == 0) break;

			/* save a row of numbers */
			num = sim_allocnumbers(sim_window_signals);
			if (numend == NONUMBERS) sim_window_numbers = num; else
				numend->nextnumbers = num;
			num->nextnumbers = NONUMBERS;
			numend = num;

			/* load the numbers */
			num->time = atof(line);
			for(k=1; k<=sim_window_signals; k++)
			{
				for(j=0; j<11; j++)
				{
					l = sim_getfromsimulator();
					line[j] = l;
					if (l == '\n') j--;
				}
				if (k < nodcnt) l = k + numnoi - 1; else l = k - nodcnt;
				num->list[l] = atof(line);
			}
		}
	}
	if (filelen > 0) DiaDoneDialog();

	(void)sim_spice_terminate(outputfile, outfile, infile);

	/* give list of numbers */
	if (stopping("SPICE") == 0)
		if (sim_spice_plotvalues(important) != 0) ttyputerr("Problem making the plot");
}

/* terminate SPICE execution */
void sim_spice_terminate(FILE *outputfile, char *outfile, char *infile)
{
#ifdef  MACOS
	xclose(sim_macfromsim);
#else
	(void)close(sim_fromsim[0]);
#endif
	if (outputfile != NULL)
	{
		xclose(outputfile);
		ttyputmsgf("%s written", outfile);
	}
#ifdef ONUNIX
	if (*infile != 0)
	{
		if (kill(sim_process, SIGKILL) == 0)
		{
			waitfor(sim_process);
			if (errno >= 0)
				ttyputmsg("SPICE execution reports error %d", errno);
					else ttyputmsg("SPICE execution terminated");
		} else ttyputmsg("SPICE execution complete");
		sim_process = -1;
	}
#endif
}

INTSML sim_getfromsimulator(void)
{
	INTSML i;
	char val;

#ifdef  MACOS
	i = xfread(&val, 1, 1, sim_macfromsim);
#else
	i = read(sim_fromsim[0], &val, 1);
#endif
	if (i != 1) return(EOF);
	sim_window_filepos++;
	return(val);
}

INTSML sim_window_topofsignals(char **c)
{
	sim_window_iter = 0;
	return(1);
}

char *sim_window_nextsignals(void)
{
	char *nextname;

	if (sim_window_iter >= sim_window_signals) return(0);
	nextname = sim_window_signames[sim_window_iter];
	sim_window_iter++;
	return(nextname);
}

static COMCOMP sim_window_picktrace = {NOKEYWORD, sim_window_topofsignals,
	sim_window_nextsignals, NOPARAMS, NOBACKUP, 0, " \t", "pick a signal to display", ""};

/*
 * routine to display the numbers in "sim_window_numbers".  For each entry of the
 * linked list is another time sample, with "sim_window_signals" trace points.  The
 * names of these signals is in "sim_window_signames".  Only the first "important"
 * entries should be displayed initially.  Returns nonzero on error.
 */
INTSML sim_spice_plotvalues(INTSML important)
{
	float lowy, highy, r, mul, tl,th, min, max;
	REGISTER NUMBERS *num;
	REGISTER INTSML i, numtotal;
	char *pars[3];
	REGISTER INTBIG ysteps, j, tr;
	REGISTER NODEPROTO *plotnp;
	extern COMCOMP us_yesnop, us_showdp;

	/* count the number of values */
	for(num = sim_window_numbers, numtotal=0; num != NONUMBERS; num = num->nextnumbers)
		numtotal++;
	if (numtotal == 0) return(0);

	/* We have to establish a link for the plot facet */
	plotnp = NONODEPROTO;

	/* if we're already simulating a facet, use that one */
	if (sim_simnt != NONODEPROTO) plotnp = sim_simnt; else
		if (sim_deck_facetname[0] != '\0') plotnp = getnodeproto(sim_deck_facetname);
	if (plotnp != NONODEPROTO && plotnp->index != 0) plotnp = NONODEPROTO;
	if (plotnp != NONODEPROTO)
	{
		(void)initinfstr();
		(void)addstringtoinfstr("Is this a simulation of facet ");
		(void)addstringtoinfstr(describenodeproto(plotnp));
		(void)addstringtoinfstr("? ");
		i = ttygetparam(returninfstr(), &us_yesnop, 3, pars);
		if (i == 1)
		{
			if (pars[0][0] == 'n') plotnp = NONODEPROTO;
		}
	}

	/* one final chance to pick a facet */
	if (plotnp == NONODEPROTO)
	{
		i = ttygetparam("Please select a facet to associate with this plot: ", &us_showdp, 3, pars);
		if (i > 0) plotnp = getnodeproto(pars[0]);
	}
	if (plotnp == NONODEPROTO) return(1);

	/* get memory for the values */
	if (sim_window_ensurespace(numtotal) != 0) return(1);

	/* get Y range of values */
	lowy = highy = sim_window_numbers->time;
	j = 0;
	for(num = sim_window_numbers; num != NONUMBERS; num = num->nextnumbers)
	{
		sim_window_time[j++] = num->time;
		for(i=0; i<sim_window_signals; i++)
		{
			if (num->list[i] < lowy) lowy = num->list[i];
			if (num->list[i] > highy) highy = num->list[i];
		}
	}

	/* spread the Y values if they are all equal */
	if (lowy == highy)
	{
		highy = lowy * 2.0;   lowy = 0.0;
		if (highy == 0.0) highy = 1.0;
	}

	/* determine range of values, multiply units by "mul" to get magnitude */
	r = highy - lowy;
	mul = 1.0;
	while (r > 10.0) { r /= 10.0;   mul *= 10.0; }
	while (r < 1.0)  { r *= 10.0;   mul /= 10.0; }

	/* range selected, step lower bound to a half-step below */
	ysteps = (INTBIG)(r-1);
	j = (INTBIG)((lowy+highy) / 2.0 / mul);
	do
	{
		ysteps++;
		tl = j*mul - ysteps*mul/2.0;
	} while (tl > lowy);

	/* adjust upper bound to a half step above */
	th = tl + ysteps*mul;
	while (th < highy)
	{
		ysteps++;
		th = tl + ysteps*mul;
	}

	/* set the globals with the range of the simulation window */
	sim_window_lowy = tl;   sim_window_highy = th;

	/* make the simulation window */
	if (sim_window_create(1, plotnp, sim_spice_charhandler, 0) != 0) return(1);

	/* create the trace nodes */
	sim_window_setanarange(sim_window_lowy, sim_window_highy);
	for(i=0; i<important; i++)
	{
		tr = sim_window_newtrace(0, sim_window_signames[i], 0);
		j = 0;
		for(num = sim_window_numbers; num != NONUMBERS; num = num->nextnumbers)
			sim_window_val[j++] = (num->list[i] - sim_window_lowy) /
				(sim_window_highy - sim_window_lowy);
		sim_window_loadanatrace(tr, numtotal, sim_window_time, sim_window_val);
	}

	/* clean up */
	min = sim_window_time[0];   max = sim_window_time[numtotal-1];
	sim_window_settimerange(min, max);
	sim_window_setmaincursor((max-min)*0.25 + min);
	sim_window_setextensioncursor((max-min)*0.75 + min);
	sim_window_redraw();
	return(0);
}

/*
 * routine to ensure that there is space for "count" time samples in the global arrays
 * "sim_window_time" and "sim_window_val".  Returns nonzero on error.
 */
INTSML sim_window_ensurespace(INTSML count)
{
	REGISTER INTSML take;

	if (count <= sim_window_limit) return(0);
	if (sim_window_limit > 0)
	{
		efree((char *)sim_window_time);
		efree((char *)sim_window_val);
		sim_window_limit = 0;
	}
	take = count + 10;
	sim_window_time = (float *)emalloc(take * (sizeof (float)), sim_aid->cluster);
	if (sim_window_time == 0) return(1);
	sim_window_val = (float *)emalloc(take * (sizeof (float)), sim_aid->cluster);
	if (sim_window_val == 0) return(1);
	sim_window_limit = take;
	return(0);
}

INTSML sim_spice_charhandler(WINDOW *w, INTSML chr)
{
	char *par[3];
	float v, time;
	REGISTER INTBIG tr, trl;
	REGISTER INTSML i, j, thispos, pos, lines, traces;
	REGISTER NUMBERS *num;

	/* see if there are special functions for SPICE waveform simulation */
	switch (chr)
	{
		case '?':		/* help */
			ttyputmsg(" i    show value of selected signal at main cursor");
			ttyputmsg(" a    add signal to simulation window");
			ttyputmsg(" o    overlay signal on top of current signal line");
			ttyputmsg(" r    remove selected signal from simulation window");
			ttyputmsg(" s    save snapshot of simulation window in database");
			return(0);
		case 'i':		/* show trace value */
			tr = sim_window_gethighlighttrace();
			if (tr == 0) return(0);
			time = sim_window_getmaincursor();
			v = sim_window_getanatracevalue(tr, time);
			ttyputmsg("Signal %s has value %g at time %g", sim_window_gettracename(tr), v, time);
			return(0);
		case 's':		/* save image in {simulation_snapshot} view */
			sim_window_savegraph();
			return(0);
		case 'a':		/* add trace */
			i = ttygetparam("Signal to add", &sim_window_picktrace, 3, par);
			if (i == 0) return(0);
			for(i=0; i<sim_window_signals; i++)
				if (namesame(par[0], sim_window_signames[i]) == 0) break;
			if (i >= sim_window_signals) return(0);

			/* count the number of traces */
			sim_window_inittraceloop();
			for(traces=0; ; traces++) if (sim_window_nexttraceloop() == 0) break;

			/* if there are no traces, put new trace on line zero */
			if (traces == 0) j = 0; else
			{
				/* other traces exist, make a new line in the plot */
				j = sim_window_getlines();
				sim_window_setlines((INTSML)(j+1));
			}

			/* create a new trace in the last slot */
			tr = sim_window_newtrace(j, sim_window_signames[i], 0);
			j = 0;
			for(num = sim_window_numbers; num != NONUMBERS; num = num->nextnumbers)
				sim_window_val[j++] = (num->list[i] - sim_window_lowy) /
					(sim_window_highy - sim_window_lowy);
			sim_window_loadanatrace(tr, j, sim_window_time, sim_window_val);
			sim_window_sethighlighttrace(tr);
			sim_window_redraw();
			return(0);
		case 'o':		/* overlap trace */
			tr = sim_window_gethighlighttrace();
			if (tr == 0) return(0);
			thispos = sim_window_gettraceline(tr);
			i = ttygetparam("Signal to overlay", &sim_window_picktrace, 3, par);
			if (i == 0) return(0);
			for(i=0; i<sim_window_signals; i++)
				if (namesame(par[0], sim_window_signames[i]) == 0) break;
			if (i >= sim_window_signals) return(0);

			/* create a new trace in this slot */
			tr = sim_window_newtrace(thispos, sim_window_signames[i], 0);
			j = 0;
			for(num = sim_window_numbers; num != NONUMBERS; num = num->nextnumbers)
				sim_window_val[j++] = (num->list[i] - sim_window_lowy) /
					(sim_window_highy - sim_window_lowy);
			sim_window_loadanatrace(tr, j, sim_window_time, sim_window_val);
			sim_window_sethighlighttrace(tr);
			sim_window_redraw();
			return(0);
		case 'r':		/* remove trace */
			tr = sim_window_gethighlighttrace();
			if (tr == 0) return(0);
			sim_window_sethighlighttrace(0);

			/* see if any other traces use this line */
			thispos = sim_window_gettraceline(tr);
			sim_window_inittraceloop();
			for(;;)
			{
				trl = sim_window_nexttraceloop();
				if (trl == tr) continue;
				if (trl == 0) break;
				pos = sim_window_gettraceline(trl);
				if (pos == thispos) break;
			}
			lines = sim_window_getlines();
			if (trl == 0 && lines > 1)
			{
				/* no other traces on this line, delete it */
				sim_window_inittraceloop();
				for(;;)
				{
					trl = sim_window_nexttraceloop();
					if (trl == 0) break;
					pos = sim_window_gettraceline(trl);
					if (pos > thispos) sim_window_settraceline(trl, (INTSML)(pos-1));
				}
				sim_window_setlines((INTSML)(lines-1));
			}

			/* kill trace, redraw */
			sim_window_killtrace(tr);
			sim_window_redraw();
			return(0);
	}
	return(us_charhandler(w, chr));
}

#endif  /* SIMSPICE - at top */
