/*
 * Electric(tm) VLSI Design System
 *
 * File: simwindow.c
 * Simulation aid: signal window control
 * Written by: Steven M. Rubin, Electric Editor Incorporated
 *
 * Copyright (c) 1998 Electric Editor Incorporated.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Electric Editor Incorporated
 * 23470 Sunset Drive, Suite 108
 * Los Gatos, California 95033
 * support@electriceditor.com
 */

#include "config.h"
#if SIMAID & (SIMALS | SIMSPICE)

#include "global.h"
#include "egraphics.h"
#include "sim.h"
#if SIMALS
#  include "simals.h"
#endif
#include "tecgen.h"
#include "tecschem.h"
#include "tecart.h"
#include "usr.h"

/*
 *     WINDOW CONTROL:
 * sim_window_init()                            Initialize simulation window system (do only once)
 * INTSML sim_window_create(l, np, chw, chs)    Make simulation window with "l" lines
 *                                              in facet "np" charhandlers "chw/s" (l=0 to restore)
 * void sim_window_stopsimulation()             Stops simulation
 * INTSML sim_window_isactive(np)               Returns nonzero if simulating, returns facet if so
 * sim_window_redraw()                          Redraws simulation window
 * sim_window_setlines(count)                   Sets number of lines in simulation window
 * INTSML sim_window_getlines()                 Returns number of lines in sim window
 * sim_window_savegraph()                       Preserves plot in database
 * sim_window_displaycolor(s, c)                Set color of strength/state "s" to "c"
 *
 *     CONTROL OF TRACES:
 * INTBIG sim_window_newtrace(line, name, data) Creates trace at "line" called "name" with user "data"
 * sim_window_loaddigtrace(tr, num, time, sta)  Loads trace "tr" with "num" digital signals "time/sta"
 * sim_window_loadanatrace(tr, num, time, val)  Loads trace "tr" with "num" analog signals "time/val"
 * sim_window_setanarange(low, high)            Sets analog range from "low" to "high"
 * sim_window_settraceline(tr, line)            Sets line number for trace "tr"
 * sim_window_killtrace(tr)                     Deletes trace "tr"
 * sim_window_killalltraces()                   Deletes all traces
 *
 *     TRACE INFORMATION:
 * INTSML sim_window_gettraceline(tr)           Returns line number for trace "tr"
 * char *sim_window_gettracename(tr)            Returns name for trace "tr"
 * INTBIG sim_window_gettracedata(tr)           Returns user data for trace "tr"
 * float sim_window_getanatracevalue(tr, time)  Returns analog value for trace "tr" at time "time"
 * INTBIG sim_window_findtrace(name)            Locates the trace with this name
 * sim_window_inittraceloop()                   Begin search of all traces
 * sim_window_inittraceloop2()                  Begin search of all traces
 * INTBIG sim_window_nexttraceloop()            Returns next trace (0 when done)
 * INTBIG sim_window_nexttraceloop2()           Returns next trace (0 when done)
 *
 *     TRACE HIGHLIGHTING:
 * sim_window_sethighlighttrace(tr)             Highlights trace "tr" (0 for none)
 * INTBIG sim_window_gethighlighttrace()        Returns highlighted trace (0 if none)
 *
 *     TIME AND CURSOR CONTROL:
 * sim_window_setmaincursor(time)               Sets position of main cursor
 * float sim_window_getmaincursor()             Returns position of main cursor
 * sim_window_setextensioncursor(time)          Sets position of extension cursor
 * float sim_window_getextensioncursor()        Returns position of extension cursor
 * sim_window_settimerange(min, max)            Sets window time range from "min" to "max"
 * sim_window_gettimerange(min, max)            Returns window time range
 */

#define	NODISCHANNEL ((DISCHANNEL *)-1)

typedef struct Idischannel
{
	char     *name;			/* name of this trace */
	INTBIG    nodeptr;		/* "user" data for this trace */
	INTSML    status;		/* state of this trace (0 for digital, color for analog) */
	INTSML    nameoff;		/* offset of trace name (when many per line) */
	INTSML    position;		/* line number of this trace in window */
	INTSML    numsteps;		/* number of steps along this trace */
	INTSML    timelimit;	/* maximum times allocated for this trace */
	INTSML    statelimit;	/* maximum states allocated for this trace */
	INTSML    valuelimit;	/* maximum values allocated for this trace */
	float    *timearray;	/* time steps along this trace */
	INTSML   *statearray;	/* states along this trace (digital) */
	float    *valuearray;	/* values along this trace (analog) */
	struct Idischannel *nextdischannel;
} DISCHANNEL;

static GRAPHICS sim_window_desc = {LAYERA, ALLOFF, {SOLIDC, SOLIDC, SOLIDC, SOLIDC},
	{0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF}, NOVARIABLE, 0};
static INTSML sim_window_anacolor[] = {RED, GREEN, BLUE, CYAN, MAGENTA, YELLOW, GRAY, ORANGE,
	PURPLE, BROWN, LGRAY, DGRAY, LRED, DRED, LGREEN, DGREEN, LBLUE, DBLUE, 0};

/*
 * In X:
 *    0 to 112 is signal name
 *      113    is bar
 *   114-767 is signal waveform
 * In Y:
 *    0 to  30 is axis label
 *   31 to 570 is signal waveforms
 *  571 to 600 is cursor value
 */
static INTSML        sim_window_curx, sim_window_cury;
static INTSML        sim_window_textsize;
static INTSML        sim_window_lines;
static INTSML        sim_window_txthei;
static INTSML        sim_window_offx, sim_window_offy;
static INTSML        sim_window_basex, sim_window_basey;
       INTSML        sim_window_state;
static WINDOW       *sim_window;			/* waveform window */
static WINDOW       *sim_schemwindow;		/* associated schematics window */
static NODEPROTO    *sim_window_np;
static float         sim_window_maincursor, sim_window_extensioncursor;
static float         sim_window_mintime, sim_window_maxtime;
static float         sim_window_analow, sim_window_anahigh;
static DISCHANNEL   *sim_window_highlightedtrace;
static DISCHANNEL   *sim_window_plot;
static DISCHANNEL   *sim_window_plotfree;
static DISCHANNEL   *sim_window_loop;
static DISCHANNEL   *sim_window_loop2;
static INTSML        sim_colorstrengthoff;		/* color of off-strength */
static INTSML        sim_colorstrengthnode;		/* color of node-strength */
static INTSML        sim_colorstrengthgate;		/* color of gate-strength */
static INTSML        sim_colorstrengthpower;	/* color of power-strength */
static INTSML        sim_colorlevellow;			/* color of low-levels */
static INTSML        sim_colorlevelhigh;		/* color of high-levels */
static INTSML        sim_colorlevelundef;		/* color of power-levels */
static INTSML      (*sim_window_origschemcharhandler)(WINDOW*, INTSML);
static void        (*sim_window_origschemtermhandler)(WINDOW*);

/* prototypes for local routines */
void   sim_window_addsegment(INTBIG*, INTBIG*, INTSML, INTSML, INTSML, INTSML);
void   sim_window_buttonhandler(WINDOW*, INTSML, INTSML, INTSML);
void   sim_window_drawbox(INTSML, INTSML, INTSML, INTSML);
void   sim_window_drawcstring(char*);
void   sim_window_drawcursors(void);
void   sim_window_drawgraph(void);
void   sim_window_drawlstring(char*, INTSML);
void   sim_window_drawrstring(char*);
void   sim_window_drawtimescale(void);
void   sim_window_drawto(INTSML, INTSML);
void   sim_window_drawtracedrag(INTSML on);
INTSML sim_window_eachbdown(INTBIG, INTBIG);
INTSML sim_window_eachdown(INTBIG x, INTBIG y);
INTSML sim_window_eachwdown(INTBIG, INTBIG);
INTSML sim_window_ignorechar(INTBIG, INTBIG, INTSML);
void   sim_window_ignoredone(void);
void   sim_window_ignoredown(void);
INTSML sim_window_ignoreup(INTBIG, INTBIG);
void   sim_window_mapcoord(INTSML*, INTSML*);
void   sim_window_moveto(INTSML, INTSML);
void   sim_window_redisphandler(WINDOW*);
void   sim_window_scaletowindow(INTBIG*, INTBIG*);
void   sim_window_setcolor(INTSML);
void   sim_window_setmask(INTSML);
void   sim_window_setviewport(WINDOW*);
void   sim_window_termhandler(WINDOW*);
void   sim_window_termschemhandler(WINDOW*);
INTBIG sim_window_timetoxpos(float);
void   sim_window_writetracename(DISCHANNEL*);
float  sim_window_xpostotime(INTBIG);

/********************************* WINDOW CONTROL *********************************/

/*
 * routine to initialize the simulation window
 */
void sim_window_init(void)
{
	sim_window_mintime = 0.0;
	sim_window_maxtime = 0.0000005f;
	sim_window_maincursor = 0.0000002f;
	sim_window_extensioncursor = 0.0000003f;
	sim_window_lines = 0;
	sim_window_plot = NODISCHANNEL;
	sim_window_plotfree = NODISCHANNEL;
	sim_window = NOWINDOW;
	sim_schemwindow = NOWINDOW;
	sim_window_np = NONODEPROTO;
	sim_window_highlightedtrace = NODISCHANNEL;
	sim_window_state = FULLSTATE | SHOWWAVEFORM;
	sim_colorstrengthoff = BLUE;
	sim_colorstrengthnode = GREEN;
	sim_colorstrengthgate = MAGENTA;
	sim_colorstrengthpower = BLACK;
	sim_colorlevellow = BLUE;
	sim_colorlevelhigh = MAGENTA;
	sim_colorlevelundef = BLACK;
}

/*
 * routine to start a simulation window in "win" (NOWINDOW if no waveform window is to
 * be displayed) with room for "linecount" traces (if "linecount" is negative, restore
 * the last window data as is).  Associated facet is "np", the character handler to use
 * in the waveform window is in "charhandlerwave", the character handler to use in the
 * schematic/layout window is in "charhandlerschem".  Returns nonzero on error.
 */
INTSML sim_window_create(INTSML linecount, NODEPROTO *np, INTSML (*charhandlerwave)(WINDOW*, INTSML),
	INTSML (*charhandlerschem)(WINDOW*, INTSML))
{
	REGISTER WINDOW *win;

	/* sorry, can't have more than 1 simulation window at a time */
	if (sim_window_np != NONODEPROTO && sim_window_np != np)
	{
		ttyputerr("Sorry, already simulating facet %s", describenodeproto(sim_window_np));
		return(1);
	}

	/* reinitialize waveform display if number of lines is specified */
	if (linecount > 0)
	{
		sim_window_lines = linecount;
		sim_window_np = np;
		if (sim_window_lines < 1) sim_window_lines = 1;
		sim_window_killalltraces();
	}

	/* create waveform window if requested */
	if (charhandlerwave != 0)
	{
		if (sim_window != NOWINDOW) win = sim_window; else
		{
			if (us_graphicshas(CANUSEFRAMES))
			{
				win = (WINDOW *)askaid(us_aid, "window-new");
			} else
			{
				win = (WINDOW *)askaid(us_aid, "window-horiz-new");
			}
			if (win == NOWINDOW) return(1);
		}
		sim_window_setviewport(win);

		startobjectchange((INTBIG)win, VWINDOW);
		(void)setval((INTBIG)win, VWINDOW, "state", (win->state & ~WINDOWTYPE) | WAVEFORMWINDOW |
			WINDOWSIMULATING, VINTEGER);
		(void)setval((INTBIG)win, VWINDOW, "curnodeproto", (INTBIG)sim_window_np, VNODEPROTO);
		(void)setval((INTBIG)win, VWINDOW, "buttonhandler", (INTBIG)sim_window_buttonhandler, VADDRESS);
		(void)setval((INTBIG)win, VWINDOW, "charhandler", (INTBIG)charhandlerwave, VADDRESS);
		(void)setval((INTBIG)win, VWINDOW, "termhandler", (INTBIG)sim_window_termhandler, VADDRESS);
		(void)setval((INTBIG)win, VWINDOW, "redisphandler", (INTBIG)sim_window_redisphandler, VADDRESS);
		endobjectchange((INTBIG)win, VWINDOW);
	}

	/* find and setup associated schematics/layout window if requested */
	if (charhandlerschem != 0)
	{
		for(sim_schemwindow = el_topwindow; sim_schemwindow != NOWINDOW;
			sim_schemwindow = sim_schemwindow->nextwindow)
		{
			if ((sim_schemwindow->state&WINDOWTYPE) == WAVEFORMWINDOW) continue;
			if (sim_schemwindow->curnodeproto == sim_window_np) break;
		}
		if (sim_schemwindow != NOWINDOW)
		{
			startobjectchange((INTBIG)sim_schemwindow, VWINDOW);
			sim_window_origschemcharhandler = sim_schemwindow->charhandler;
			sim_window_origschemtermhandler = sim_schemwindow->termhandler;
			(void)setval((INTBIG)sim_schemwindow, VWINDOW, "state", sim_schemwindow->state |
				WINDOWSIMULATING, VINTEGER);
			(void)setval((INTBIG)sim_schemwindow, VWINDOW, "charhandler",
				(INTBIG)charhandlerschem, VADDRESS);
			(void)setval((INTBIG)sim_schemwindow, VWINDOW, "termhandler",
				(INTBIG)sim_window_termschemhandler, VADDRESS);
			endobjectchange((INTBIG)sim_schemwindow, VWINDOW);
		}
	}
	return(0);
}

/*
 * Routine to stop simulation.
 */
void sim_window_stopsimulation(void)
{
	sim_window_np = NONODEPROTO;

	/* disable simulation control in schematics/layout window */
	if (sim_schemwindow != NOWINDOW)
	{
		startobjectchange((INTBIG)sim_schemwindow, VWINDOW);
		(void)setval((INTBIG)sim_schemwindow, VWINDOW, "state", sim_schemwindow->state &
			~WINDOWSIMULATING, VINTEGER);
		(void)setval((INTBIG)sim_schemwindow, VWINDOW, "charhandler",
			(INTBIG)sim_window_origschemcharhandler, VADDRESS);
		(void)setval((INTBIG)sim_schemwindow, VWINDOW, "termhandler",
			(INTBIG)sim_window_origschemtermhandler, VADDRESS);
		endobjectchange((INTBIG)sim_schemwindow, VWINDOW);
		sim_schemwindow = NOWINDOW;
	}

	/* disable waveform window */
	if (sim_window != NOWINDOW)
	{
		startobjectchange((INTBIG)sim_window, VWINDOW);
		(void)setval((INTBIG)sim_window, VWINDOW, "state", sim_window->state &
			~WINDOWSIMULATING, VINTEGER);
		endobjectchange((INTBIG)sim_window, VWINDOW);
		/* sim_window_redisphandler(sim_window); */
	}
}

/*
 * routine that returns:
 *  0                                    if there is no active simulation
 *  SIMWINDOWWAVEFORM                    if simulating and have a waveform window
 *  SIMWINDOWSCHEMATIC                   if simulating and have a schematics window
 *  SIMWINDOWWAVEFORM|SIMWINDOWSCHEMATIC if simulating and have both windows
 * If there is active simulation, the facet being simulated is returned in "np"
 */
INTSML sim_window_isactive(NODEPROTO **np)
{
	REGISTER INTSML activity;

	*np = sim_window_np;
	if (sim_window_np == NONODEPROTO) return(0);

	activity = 0;
	if (sim_window != NOWINDOW) activity |= SIMWINDOWWAVEFORM;
	if (sim_schemwindow != NOWINDOW) activity |= SIMWINDOWSCHEMATIC;
	return(activity);
}

/*
 * this procedure redraws everything, including the waveforms.
 */
void sim_window_redraw(void)
{
	sim_window_setcolor(ALLOFF);
	sim_window_setmask(LAYERA);
	sim_window_drawbox(0, 0, 767, 600);
	sim_window_drawtimescale();
	sim_window_drawgraph();
	sim_window_drawcursors();
}

/*
 * routine that changes the number of trace lines in the simulation window to "count"
 */
void sim_window_setlines(INTSML count)
{
	if (count <= 0) count = 1;
	sim_window_lines = count;
}

/*
 * routine that returns the number of signals in the simulation window
 */
INTSML sim_window_getlines(void)
{
	return(sim_window_lines);
}

/********************************* CONTROL OF TRACES *********************************/

/*
 * routine to create a new trace to be displayed in window location
 * "position".  Its name is set to "name", and "userdata" is
 * associated with it.  A pointer to the trace is returned (zero on error).
 */
INTBIG sim_window_newtrace(INTSML position, char *name, INTBIG userdata)
{
	REGISTER DISCHANNEL *tr;

	if (sim_window_plotfree == NODISCHANNEL)
	{
		tr = (DISCHANNEL *)emalloc(sizeof (DISCHANNEL), sim_aid->cluster);
		if (tr == 0) return(0);
		tr->timelimit = 0;
		tr->statelimit = 0;
		tr->valuelimit = 0;
	} else
	{
		tr = sim_window_plotfree;
		sim_window_plotfree = tr->nextdischannel;
	}

	/* put in active list, load data */
	tr->nextdischannel = sim_window_plot;
	sim_window_plot = tr;
	(void)allocstring(&tr->name, name, sim_aid->cluster);
	tr->nodeptr = userdata;
	tr->position = position;
	tr->numsteps = 0;
	tr->nameoff = 0;
	return((INTBIG)tr);
}

/*
 * routine to load trace "tri" with "count" digital events.  The events occur
 * at time "time" and have state "state".  "State" is encoded with a
 * level in the upper 8 bits (LOGIC_LOW, LOGIC_HIGH, or LOGIC_X) and a
 * strength in the lower 8 bits (OFF_STRENGTH, NODE_STRENGTH, GATE_STRENGTH,
 * or VDD_STRENGTH).
 */
void sim_window_loaddigtrace(INTBIG tri, INTSML count, float *time, INTSML *state)
{
	REGISTER INTSML wanted, i;
	REGISTER DISCHANNEL *tr;

	/* ensure space in the arrays */
	tr = (DISCHANNEL *)tri;
	if (tr == NODISCHANNEL || tr == 0) return;
	if (count > tr->timelimit)
	{
		if (tr->timelimit > 0)
		{
			efree((char *)tr->timearray);
			tr->timelimit = 0;
		}
		wanted = count + 10;
		tr->timearray = (float *)emalloc(wanted * (sizeof (float)), sim_aid->cluster);
		if (tr->timearray == 0) return;
		tr->timelimit = wanted;
	}
	if (count > tr->statelimit)
	{
		if (tr->statelimit > 0)
		{
			efree((char *)tr->statearray);
			tr->statelimit = 0;
		}
		wanted = count + 10;
		tr->statearray = (INTSML *)emalloc(wanted * SIZEOFINTSML, sim_aid->cluster);
		if (tr->statearray == 0) return;
		tr->statelimit = wanted;
	}

	/* load the data */
	for(i=0; i<count; i++)
	{
		tr->timearray[i] = time[i];
		tr->statearray[i] = state[i];
	}
	tr->status = 0;
	tr->numsteps = count;
}

/*
 * routine to load trace "tri" with "count" analog events.  The events occur
 * at time "time" and have value "value".
 */
void sim_window_loadanatrace(INTBIG tri, INTSML count, float *time, float *value)
{
	REGISTER INTSML wanted, i;
	REGISTER DISCHANNEL *tr;

	/* ensure space in the arrays */
	tr = (DISCHANNEL *)tri;
	if (tr == NODISCHANNEL || tr == 0) return;
	if (count > tr->timelimit)
	{
		if (tr->timelimit > 0)
		{
			efree((char *)tr->timearray);
			tr->timelimit = 0;
		}
		wanted = count + 10;
		tr->timearray = (float *)emalloc(wanted * (sizeof (float)), sim_aid->cluster);
		if (tr->timearray == 0) return;
		tr->timelimit = wanted;
	}
	if (count > tr->valuelimit)
	{
		if (tr->valuelimit > 0)
		{
			efree((char *)tr->valuearray);
			tr->valuelimit = 0;
		}
		wanted = count + 10;
		tr->valuearray = (float *)emalloc(wanted * (sizeof (float)), sim_aid->cluster);
		if (tr->valuearray == 0) return;
		tr->valuelimit = wanted;
	}

	/* load the data */
	for(i=0; i<count; i++)
	{
		tr->timearray[i] = time[i];
		tr->valuearray[i] = value[i];
	}
	tr->status = RED;
	tr->numsteps = count;
}

void sim_window_setanarange(float low, float high)
{
	sim_window_analow = low;
	sim_window_anahigh = high;
}

void sim_window_settraceline(INTBIG tri, INTSML line)
{
	REGISTER DISCHANNEL *tr;

	tr = (DISCHANNEL *)tri;
	if (tr == NODISCHANNEL || tr == 0) return;
	tr->position = line;
}

/*
 * routine to delete the trace "tri"
 */
void sim_window_killtrace(INTBIG tri)
{
	REGISTER DISCHANNEL *tr, *str, *ltr;

	tr = (DISCHANNEL *)tri;
	if (tr == NODISCHANNEL || tr == 0) return;
	ltr = NODISCHANNEL;
	for(str = sim_window_plot; str != NODISCHANNEL; str = str->nextdischannel)
	{
		if (str == tr) break;
		ltr = str;
	}
	if (ltr == NODISCHANNEL) sim_window_plot = tr->nextdischannel; else
		ltr->nextdischannel = tr->nextdischannel;
	tr->nextdischannel = sim_window_plotfree;
	sim_window_plotfree = tr;
	efree(tr->name);
}

/*
 * routine to delete all traces
 */
void sim_window_killalltraces(void)
{
	REGISTER DISCHANNEL *tr, *nexttr;

	/* return all plot channels to the free list */
	for(tr = sim_window_plot; tr != NODISCHANNEL; tr = nexttr)
	{
		nexttr = tr->nextdischannel;
		tr->nextdischannel = sim_window_plotfree;
		sim_window_plotfree = tr;
		efree(tr->name);
	}
	sim_window_plot = NODISCHANNEL;
	sim_window_highlightedtrace = NODISCHANNEL;
}

/********************************* TRACE INFORMATION *********************************/

/*
 * routine to return the line number associated with trace "tri"
 */
INTSML sim_window_gettraceline(INTBIG tri)
{
	REGISTER DISCHANNEL *tr;

	tr = (DISCHANNEL *)tri;
	if (tr == NODISCHANNEL || tr == 0) return(0);
	return(tr->position);
}

/*
 * routine to return the name of trace "tri".
 */
char *sim_window_gettracename(INTBIG tri)
{
	REGISTER DISCHANNEL *tr;

	tr = (DISCHANNEL *)tri;
	if (tr == NODISCHANNEL || tr == 0) return("");
	return(tr->name);
}

/*
 * routine to return the "user" data associated with trace "tri"
 */
INTBIG sim_window_gettracedata(INTBIG tri)
{
	REGISTER DISCHANNEL *tr;

	tr = (DISCHANNEL *)tri;
	if (tr == NODISCHANNEL || tr == 0) return(0);
	return(tr->nodeptr);
}

float sim_window_getanatracevalue(INTBIG tri, float time)
{
	REGISTER DISCHANNEL *tr;
	REGISTER INTSML j;
	REGISTER float lasttime, lastvalue, thistime, thisvalue, range, v;

	tr = (DISCHANNEL *)tri;
	if (tr == NODISCHANNEL || tr == 0) return(0.0);

	/* must be an analog trace */
	if (tr->status == 0) return(0.0);

	range = sim_window_anahigh - sim_window_analow;
	for(j=0; j<tr->numsteps; j++)
	{
		thistime = tr->timearray[j];
		thisvalue = tr->valuearray[j] * range + sim_window_analow;
		if (thistime == time) return(thisvalue);
		if (time > thistime)
		{
			lasttime = thistime;
			lastvalue = thisvalue;
			continue;
		}
		if (j == 0) return(thisvalue);
		v = lastvalue + (time-lasttime) * (thisvalue-lastvalue) / (thistime-lasttime);
		return(v);
	}

	return(lastvalue);
}

INTBIG sim_window_findtrace(char *name)
{
	REGISTER DISCHANNEL *tr;

	for(tr = sim_window_plot; tr != NODISCHANNEL; tr = tr->nextdischannel)
	{
		if (namesame(tr->name, name) == 0) return((INTBIG)tr);
	}

	/* handle "v(" and "l(" prefixes of HSPICE (see also "network.c:net_parsenetwork()") */
	for(tr = sim_window_plot; tr != NODISCHANNEL; tr = tr->nextdischannel)
	{
		if ((tr->name[0] == 'l' || tr->name[0] == 'v') && tr->name[1] == '(')
		{
			if (namesame(&tr->name[2], name) == 0) return((INTBIG)tr);
		}
	}

	/* name not found in simulator */
	return(0);
}

void sim_window_inittraceloop(void)
{
	sim_window_loop = sim_window_plot;
}

void sim_window_inittraceloop2(void)
{
    sim_window_loop2 = sim_window_plot;
}

INTBIG sim_window_nexttraceloop(void)
{
	REGISTER DISCHANNEL *tr;

	tr = sim_window_loop;
	if (tr == NODISCHANNEL) return(0);
	sim_window_loop = tr->nextdischannel;
	return((INTBIG)tr);
}

INTBIG sim_window_nexttraceloop2(void)
{
    REGISTER DISCHANNEL *tr;
 
    tr = sim_window_loop2;
    if (tr == NODISCHANNEL) return(0);
    sim_window_loop2 = tr->nextdischannel;
    return((INTBIG)tr);
}

/********************************* TRACE HIGHLIGHTING *********************************/

/*
 * routine that sets trace "tri" to be highlighted in the simulation window
 */
void sim_window_sethighlighttrace(INTBIG tri)
{
	REGISTER DISCHANNEL *tr, *oldtr;

	tr = (DISCHANNEL *)tri;
	if (tr == 0) tr = NODISCHANNEL;
	oldtr = sim_window_highlightedtrace;
	sim_window_highlightedtrace = tr;
	if (oldtr != NODISCHANNEL) sim_window_writetracename(oldtr);
	if (tr != NODISCHANNEL) sim_window_writetracename(tr);
}

/*
 * routine that returns the highlighted trace in the simulation window
 */
INTBIG sim_window_gethighlighttrace(void)
{
	if (sim_window_highlightedtrace == NODISCHANNEL) return(0);
	return((INTBIG)sim_window_highlightedtrace);
}

/*
 * Routine to update associated layout windows when the main cursor changes
 */
void sim_window_updatelayoutwindow(void)
{
	REGISTER DISCHANNEL *tr;
	REGISTER NETWORK *net;
	REGISTER ARCINST *ai;
	REGISTER INTBIG j, addr, fx, fy, tx, ty;
	REGISTER INTSML texture;
	static POLYGON *poly = NOPOLYGON;
	void us_wanttodraw(INTBIG, INTBIG, INTBIG, INTBIG, WINDOW*, GRAPHICS*, INTSML);

	/* make sure simulation is running */
	if (sim_window_np == NONODEPROTO) return;

	/* make sure there is an associated layout/schematic facet in a window */
	if (sim_schemwindow == NOWINDOW) return;

	/* reset all values on networks */
	for(net = sim_window_np->firstnetwork; net != NONETWORK; net = net->nextnetwork)
		net->temp1 = -1;

	/* assign values from simulation window traces to networks */
	for(tr = sim_window_plot; tr != NODISCHANNEL; tr = tr->nextdischannel)
	{
		/* can only handle digital waveforms right now */
		if (tr->status != 0) continue;

		/* find the proper data for the time of the main cursor */
		for(j=1; j<tr->numsteps; j++)
			if (sim_window_maincursor < tr->timearray[j]) break;
		j--;

		/* set simulation value on the network in the associated layout/schematic window */
		net = getnetwork(tr->name, sim_window_np);
		if (net == NONETWORK)
		{
			if (tr->name[0] == 'N' && tr->name[1] == 'E' && tr->name[2] == 'T')
			{
				addr = atoi(&tr->name[3]);
				for(net = sim_window_np->firstnetwork; net != NONETWORK; net = net->nextnetwork)
					if (net == (NETWORK *)addr) break;
			}
		}
		if (net == NONETWORK)
		{
			/* check for prefix "v(" and "l(" HSPICE prefixes */
			if ((tr->name[0] == 'l' || tr->name[0] == 'v') && tr->name[1] == '(')
				net = getnetwork(&tr->name[2], sim_window_np);
		}
		if (net != NONETWORK && j < tr->numsteps) net->temp1 = tr->statearray[j];
	}

	/* redraw all arcs in the layout/schematic window */
	sim_window_desc.bits = LAYERA;
	for(ai = sim_window_np->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
	{
		net = ai->network;
		if (net == NONETWORK) continue;
		if (net->temp1 == -1) continue;

		/* get extent of arc */
		fx = ai->end[0].xpos;   fy = ai->end[0].ypos;
		tx = ai->end[1].xpos;   ty = ai->end[1].ypos;
		if (ai->proto == sch_wirearc)
		{
			/* for schematic wires, ask technology about drawing so negating bubbles work */
			if (poly == NOPOLYGON) poly = allocpolygon(4, sim_aid->cluster);
			(void)arcpolys(ai);
			shapearcpoly(ai, 0, poly);
			fx = poly->xv[0];   fy = poly->yv[0];
			tx = poly->xv[1];   ty = poly->yv[1];
		}

		if ((sim_window_state&FULLSTATE) != 0)
		{
			/* 12-state display: determine trace texture */
			/* determine trace texture */
			switch (net->temp1 & 0377)
			{
				case NODE_STRENGTH: texture = 1;    break;
				case GATE_STRENGTH: texture = 0;    break;
				case VDD_STRENGTH:  texture = 2;    break;
				default:            texture = -1;   break;
			}

			/* reset background arc if trace is textured */
			if (texture != 0)
			{
				sim_window_desc.col = ALLOFF;
				us_wanttodraw(fx, fy, tx, ty, sim_schemwindow, &sim_window_desc, 0);
				if (texture < 0) texture = 0;
			}

			/* determine trace color */
			switch (net->temp1 >> 8)
			{
				case LOGIC_LOW:  sim_window_desc.col = sim_colorlevellow;     break;
				case LOGIC_X:    sim_window_desc.col = sim_colorlevelundef;   break;
				case LOGIC_HIGH: sim_window_desc.col = sim_colorlevelhigh;    break;
			}
		} else
		{
			/* 2-state display */
			texture = 0;
			if ((net->temp1 >> 8) == LOGIC_HIGH) sim_window_desc.col = sim_colorstrengthgate; else
				sim_window_desc.col = sim_colorstrengthoff;
		}

		/* draw the trace on the arc */
		us_wanttodraw(fx, fy, tx, ty, sim_schemwindow, &sim_window_desc, texture);
	}
}

/********************************* TIME AND CURSOR CONTROL *********************************/

/*
 * routine to set the time for the "main" cursor in the simulation window
 */
void sim_window_setmaincursor(float time)
{
	sim_window_maincursor = time;
	sim_window_updatelayoutwindow();
	sim_window_drawcursors();
}

/*
 * routine to return the time of the "main" cursor in the simulation window
 */
float sim_window_getmaincursor(void)
{
	return(sim_window_maincursor);
}

/*
 * routine to set the time for the "extension" cursor in the simulation window
 */
void sim_window_setextensioncursor(float time)
{
	sim_window_extensioncursor = time;
	sim_window_drawcursors();
}

/*
 * routine to return the time of the "extension" cursor in the simulation window
 */
float sim_window_getextensioncursor(void)
{
	return(sim_window_extensioncursor);
}

/*
 * routine to set the simulation window to run from time "min" to time "max"
 */
void sim_window_settimerange(float min, float max)
{
	sim_window_mintime = min;
	sim_window_maxtime = max;
}

/*
 * routine to return the range of time displayed in the simulation window
 */
void sim_window_gettimerange(float *min, float *max)
{
	*min = sim_window_mintime;
	*max = sim_window_maxtime;
}

/********************************* WINDOW METHODS *********************************/

INTSML sim_window_ignoreup(INTBIG x, INTBIG y) { return(0); }
void sim_window_ignoredown(void) {}
INTSML sim_window_ignorechar(INTBIG x, INTBIG y, INTSML chr) { return(0); }
void sim_window_ignoredone(void) {}
INTSML sim_window_eachwdown(INTBIG x, INTBIG y)
{
	sim_window_scaletowindow(&x, &y);
	sim_window_maincursor = sim_window_xpostotime(x);
	if (sim_window_maincursor < sim_window_mintime)
		sim_window_maincursor = sim_window_mintime;
	if (sim_window_maincursor > sim_window_maxtime)
		sim_window_maincursor = sim_window_maxtime;
	sim_window_drawcursors();
	sim_window_updatelayoutwindow();
	return(0);
}
INTSML sim_window_eachbdown(INTBIG x, INTBIG y)
{
	sim_window_scaletowindow(&x, &y);
	sim_window_extensioncursor = sim_window_xpostotime(x);
	if (sim_window_extensioncursor < sim_window_mintime)
		sim_window_extensioncursor = sim_window_mintime;
	if (sim_window_extensioncursor > sim_window_maxtime)
		sim_window_extensioncursor = sim_window_maxtime;
	sim_window_drawcursors();
	return(0);
}

INTBIG sim_window_tracedragyoff, sim_window_initialy;
DISCHANNEL *sim_window_tracedragtr;

/*
 * Routine for drawing highlight around dragged trace name.
 */
void sim_window_drawtracedrag(INTSML on)
{
	INTSML y_pos, temp_trace_spacing, line, lx, hx, ly, hy;

	/* compute positions */
	temp_trace_spacing = 540 / sim_window_lines;
	line = sim_window_tracedragtr->position;
	y_pos = (sim_window_lines - 1 - line) * temp_trace_spacing + 31 + sim_window_tracedragyoff;
	ly = y_pos + sim_window_tracedragtr->nameoff - 1;
	hy = ly + sim_window_txthei+2;
	lx = 0;
	hx = 112;

	sim_window_setmask(LAYERH);
	if (on == 0) sim_window_setcolor(ALLOFF); else
		sim_window_setcolor(HIGHLIT);
	sim_window_moveto(lx, ly);
	sim_window_drawto(hx, ly);
	sim_window_drawto(hx, hy);
	sim_window_drawto(lx, hy);
	sim_window_drawto(lx, ly);
}

/*
 * Routine for tracking the drag of signal names in the waveform window.
 */
INTSML sim_window_eachdown(INTBIG x, INTBIG y)
{
	sim_window_scaletowindow(&x, &y);
	sim_window_drawtracedrag(0);
	sim_window_tracedragyoff = y - sim_window_initialy;
	sim_window_drawtracedrag(1);
	return(0);
}

void sim_window_buttonhandler(WINDOW *w, INTSML button, INTSML inx, INTSML iny)
{
	INTBIG x, y, xpos, ypos;
	INTSML ind, temp_trace_spacing, tracelinesmoved;
	char *par[2];
	extern AIDENTRY *net_aid;
	REGISTER DISCHANNEL *tr, *otr, *trprev, *otrprev;

	sim_window_setviewport(w);
	x = inx;   y = iny;
	sim_window_scaletowindow(&x, &y);

	/* see if a signal name was hit */
	if (x < 114)
	{
		sim_window_sethighlighttrace(0);
		temp_trace_spacing = 540 / sim_window_lines;
		ind = sim_window_lines - 1 - (y - 31) / temp_trace_spacing;
		trprev = NODISCHANNEL;
		for(tr = sim_window_plot; tr != NODISCHANNEL; tr = tr->nextdischannel)
		{
			if (tr->position == ind)
			{
				ypos = (sim_window_lines - 1 - ind) * temp_trace_spacing + 31 + tr->nameoff;
				if (y >= ypos && y <= ypos + sim_window_txthei) break;
			}
			trprev = tr;
		}
		if (tr == NODISCHANNEL) return;

		/* see if the signal shares a line with any others */
		for(otr = sim_window_plot; otr != NODISCHANNEL; otr = otr->nextdischannel)
			if (otr != tr && otr->position == tr->position) break;
		if (otr == NODISCHANNEL)
		{
			/* handle dragging of the signal name */
			sim_window_tracedragyoff = 0;
			sim_window_initialy = y;
			sim_window_tracedragtr = tr;
			sim_window_drawtracedrag(1);
			trackcursor(0, sim_window_ignoreup, sim_window_ignoredown, sim_window_eachdown,
				sim_window_ignorechar, sim_window_ignoredone, TRACKDRAGGING);
			sim_window_drawtracedrag(0);

			/* see if the signal name was dragged to a new location */
			if (sim_window_tracedragyoff < 0) sim_window_tracedragyoff -= temp_trace_spacing/2; else
				sim_window_tracedragyoff += temp_trace_spacing/2;
			tracelinesmoved = sim_window_tracedragyoff / temp_trace_spacing;
			if (tracelinesmoved != 0)
			{
				/* find which signal name it was dragged to */
				otrprev = NODISCHANNEL;
				for(otr = sim_window_plot; otr != NODISCHANNEL; otr = otr->nextdischannel)
				{
					if (otr->position == tr->position - tracelinesmoved) break;
					otrprev = otr;
				}
				if (otr != NODISCHANNEL)
				{
					/* remember number of screen lines */
					ind = sim_window_plot->position;

					/* pull out the selected signal */
					if (trprev == NODISCHANNEL) sim_window_plot = tr->nextdischannel; else
						trprev->nextdischannel = tr->nextdischannel;
					if (tr->position > otr->position)
					{
						/* shift this signal down the screen */
						tr->nextdischannel = otr->nextdischannel;
						otr->nextdischannel = tr;
					} else
					{
						/* shift this signal up the screen */
						tr->nextdischannel = otr;
						if (otrprev == NODISCHANNEL) sim_window_plot = tr; else
							otrprev->nextdischannel = tr;
					}
					for(otr = sim_window_plot; otr != NODISCHANNEL; otr = otr->nextdischannel, ind--)
						otr->position = ind;

					/* redisplay window */
					sim_window_redraw();
				}
			}
		}

		/* highlight the selected signal */
		sim_window_sethighlighttrace((INTBIG)tr);
		par[0] = "highlight";
		tellaid(net_aid, 1, par);
		return;
	}

	/* see if a cursor was hit */
	xpos = sim_window_timetoxpos(sim_window_maincursor);
	if (abs(x-xpos) < 10)
	{
		/* main cursor hit */
		trackcursor(0, sim_window_ignoreup, sim_window_ignoredown, sim_window_eachwdown,
			sim_window_ignorechar, sim_window_ignoredone, TRACKDRAGGING);
		return;
	}
	xpos = sim_window_timetoxpos(sim_window_extensioncursor);
	if (abs(x-xpos) < 10)
	{
		/* extension cursor hit */
		trackcursor(0, sim_window_ignoreup, sim_window_ignoredown, sim_window_eachbdown,
			sim_window_ignorechar, sim_window_ignoredone, TRACKDRAGGING);
		return;
	}
}

/*
 * Routine called when the associated schematics/layout window is closed.
 */
void sim_window_termschemhandler(WINDOW *w)
{
	sim_schemwindow = NOWINDOW;

	/* if there is no waveform window, stop simulation */
	if (sim_window == NOWINDOW) sim_window_stopsimulation();
}

/*
 * Routine called when the waveform window is closed.
 */
void sim_window_termhandler(WINDOW *w)
{
	sim_window_setviewport(NOWINDOW);

	/* if there is no schematics/layout window, stop simulation */
	if (sim_schemwindow == NOWINDOW) sim_window_stopsimulation();
}

/*
 * This procedure redraws the display, allowing for resize.
 */
void sim_window_redisphandler(WINDOW *w)
{
	INTSML lx, hx, ly, hy;

	if (sim_window == NOWINDOW)
	{
		/* reestablish this as an active simulation waveform window */
		if (sim_window_np != NONODEPROTO) return;

		/* enable waveform window */
		sim_window_np = w->curnodeproto;
		w->state |= WINDOWSIMULATING;
		w->uselx += SIMULATINGBORDERSIZE;   w->usehx -= SIMULATINGBORDERSIZE;
		w->usely += SIMULATINGBORDERSIZE;   w->usehy -= SIMULATINGBORDERSIZE;
	}
	sim_window_setviewport(w);

	/* clear the window */
	sim_window_desc.bits = LAYERA;
	sim_window_desc.col = ALLOFF;
	lx = w->uselx;   hx = w->usehx;
	ly = w->usely;   hy = w->usehy;
	us_drawbox(w, lx, hx, ly, hy, &sim_window_desc);

	/* draw red outline */
	if ((w->state&WINDOWSIMULATING) != 0)
	{
		sim_window_desc.col = RED;
		us_drawbox(w, (INTSML)(lx-SIMULATINGBORDERSIZE), (INTSML)(hx+SIMULATINGBORDERSIZE), (INTSML)(ly-SIMULATINGBORDERSIZE), ly,
			&sim_window_desc);
		us_drawbox(w, (INTSML)(lx-SIMULATINGBORDERSIZE), (INTSML)(hx+SIMULATINGBORDERSIZE), hy, (INTSML)(hy+SIMULATINGBORDERSIZE),
			&sim_window_desc);
		us_drawbox(w, (INTSML)(lx-SIMULATINGBORDERSIZE), lx, ly, hy, &sim_window_desc);
		us_drawbox(w, hx, (INTSML)(hx+SIMULATINGBORDERSIZE), ly, hy, &sim_window_desc);
	}

	/* write banner at top of window */
	us_settextsize(w, TXT10P);
	sim_window_desc.col = el_colmentxt;
	(void)initinfstr();
	if ((w->state&WINDOWSIMULATING) == 0) (void)addstringtoinfstr("Inactive ");
	(void)addstringtoinfstr("Simulation Window, ? for help");
	us_puttext(w, (INTSML)(lx+5), (INTSML)(hy-18), returninfstr(), &sim_window_desc);
	us_invertbox(w, lx, hx, (INTSML)(hy-20), hy);
	sim_window_redraw();
}

/****************************** SUPPORT FUNCTIONS ******************************/

/*
 * Routine to convert from simulation time to the proper X position in the waveform window.
 */
INTBIG sim_window_timetoxpos(float time)
{
	float result;

	result = 653.0 * (time - sim_window_mintime) / (sim_window_maxtime - sim_window_mintime) + 114.0;
	return(roundfloat(result));
}

/*
 * Routine to convert from the X position in the waveform window to simulation time.
 */
float sim_window_xpostotime(INTBIG xpos)
{
	float result;

	result = (xpos-114.0) * (sim_window_maxtime-sim_window_mintime) / 653.0 + sim_window_mintime;
	return(result);
}

/*
 * This procedure prints out the time scale on the graphics terminal.
 */
void sim_window_drawtimescale(void)
{
	INTBIG i, x_pos;
	float time;
	char s2[20];

	sim_window_setmask(LAYERA);
	sim_window_setcolor(MENTXT);
	for(i=1; i<5; i++)
	{
		time = i * 0.2 * (sim_window_maxtime - sim_window_mintime) + sim_window_mintime;
		x_pos = sim_window_timetoxpos(time);
		sim_windowconvertengineeringnotation(time, s2);
		sim_window_moveto((INTSML)x_pos, 0);
		sim_window_drawcstring(s2);
	}
}

void sim_window_writetracename(DISCHANNEL *tr)
{
	INTSML y_pos, temp_trace_spacing, line, lx, hx, ly, hy;

	/* compute positions */
	temp_trace_spacing = 540 / sim_window_lines;
	line = tr->position;
	y_pos = (sim_window_lines - 1 - line) * temp_trace_spacing + 31;
	ly = y_pos + tr->nameoff-1;   hy = ly + sim_window_txthei+2;
	lx = 0;                       hx = 112;

	/* erase the box where the text will go */
	sim_window_setmask(LAYERA);
	sim_window_setcolor(ALLOFF);
	sim_window_drawbox(lx, ly, hx, hy);

	/* draw the vertical line between the label and the trace */
	sim_window_setcolor(MENGLY);
	sim_window_moveto(113, y_pos);
	sim_window_drawto(113, (INTSML)(y_pos+temp_trace_spacing-10));

	/* write the text */
	sim_window_moveto(1, (INTSML)(y_pos + tr->nameoff));
	if (tr->status == 0) sim_window_setcolor(MENTXT); else
		sim_window_setcolor(tr->status);
	sim_window_drawlstring(tr->name, 111);

	/* outline it if it is highlighted */
	if (sim_window_highlightedtrace == tr)
	{
		sim_window_setmask(LAYERH);
		sim_window_setcolor(HIGHLIT);
		sim_window_moveto(lx, ly);
		sim_window_drawto(hx, ly);
		sim_window_drawto(hx, hy);
		sim_window_drawto(lx, hy);
		sim_window_drawto(lx, ly);
	}
}

/*
 * This procedure draws the cursors on the timing diagram.  The other color
 * planes are write protected so that the cursor doesn't alter any of the
 * timing diagram information.
 */
void sim_window_drawcursors(void)
{
	INTBIG x_pos;
	char s2[40];
	float v;

	/* erase all cursors */
	sim_window_setcolor(ALLOFF);
	sim_window_setmask(LAYERH);
	sim_window_drawbox(114, 0, 767, 570);
	sim_window_setcolor(HIGHLIT);

	/* draw main cursor */
	x_pos = sim_window_timetoxpos(sim_window_maincursor);
	if (x_pos >= 114 && x_pos <= 767)
	{
		sim_window_moveto((INTSML)x_pos, 31);
		sim_window_drawto((INTSML)x_pos, 570);
	}

	/* draw extension cursor */
	x_pos = sim_window_timetoxpos(sim_window_extensioncursor);
	if (x_pos >= 114 && x_pos <= 767)
	{
		sim_window_moveto((INTSML)x_pos, 31);
		sim_window_drawto((INTSML)x_pos, 570);
		sim_window_moveto((INTSML)(x_pos-5), 558);
		sim_window_drawto((INTSML)(x_pos+5), 568);
		sim_window_moveto((INTSML)(x_pos+5), 558);
		sim_window_drawto((INTSML)(x_pos-5), 568);
	}

	/* draw cursor locations textually */
	sim_window_setmask(LAYERA);
	sim_window_setcolor(ALLOFF);
	sim_window_drawbox(114, 571, 329, 600);		/* main cursor text area */
	sim_window_drawbox(333, 571, 548, 600);		/* extension cursor text area */
	sim_window_drawbox(552, 571, 767, 600);		/* delta text area */
	sim_window_setcolor(MENTXT);
	sim_window_moveto(114, 571);
	strcpy(s2, "Main: ");
	sim_windowconvertengineeringnotation(sim_window_maincursor, &s2[6]);
	sim_window_drawlstring(s2, 215);
	sim_window_moveto(333, 571);
	strcpy(s2, "Ext: ");
	sim_windowconvertengineeringnotation(sim_window_extensioncursor, &s2[5]);
	sim_window_drawlstring(s2, 215);
	sim_window_moveto(552, 571);
	strcpy(s2, "Delta: ");
	v = sim_window_extensioncursor - sim_window_maincursor;
	if (v < 0.0) v = -v;
	sim_windowconvertengineeringnotation(v, &s2[7]);
	sim_window_drawlstring(s2, 215);
}

void sim_window_drawgraph(void)
{
	INTSML x_min, x_max, y_min, y_max, j, maxoff, state, laststate, color, first,
		slot, temp_trace_spacing, range, hasana, fx, fy, tx, ty;
	char s2[15];
	REGISTER DISCHANNEL *tr, *otr;
	float time, position;

	/* set name offsets */
	for(tr = sim_window_plot; tr != NODISCHANNEL; tr = tr->nextdischannel) tr->nameoff = -1;
	temp_trace_spacing = 540 / sim_window_lines;
	for(tr = sim_window_plot; tr != NODISCHANNEL; tr = tr->nextdischannel)
	{
		/* determine position of trace label on the line */
		if (tr->nameoff < 0)
		{
			hasana = 0;
			maxoff = 0;
			for(otr = sim_window_plot; otr != NODISCHANNEL; otr = otr->nextdischannel)
			{
				if (otr->position != tr->position) continue;
				maxoff++;
				if (otr->status != 0) hasana++;
			}
			if (maxoff == 1)
			{
				/* only one trace: center it */
				tr->nameoff = (temp_trace_spacing - 10 - sim_window_txthei) / 2;
				if (tr->nameoff < 0) tr->nameoff = 0;
			} else
			{
				/* multiple traces: compute them */
				range = (temp_trace_spacing - 10 - sim_window_txthei*3) / (maxoff-1);
				if (range <= 0) range = 1;
				slot = 0;
				for(otr = sim_window_plot; otr != NODISCHANNEL; otr = otr->nextdischannel)
				{
					if (otr->position != tr->position) continue;
					otr->nameoff = slot * range + sim_window_txthei;
					slot++;
				}
			}

			/* if there are analog traces on this line, show height values */
			if (hasana != 0)
			{
				y_min = (sim_window_lines - 1 - tr->position) * temp_trace_spacing + 31;
				y_max = y_min + temp_trace_spacing - 10;
				sim_window_setmask(LAYERA);
				sim_window_setcolor(MENTXT);
				sim_window_moveto(100, y_min);
				sim_window_drawto(113, y_min);
				sim_window_moveto(112, y_min);
				(void)sprintf(s2, "%g", sim_window_analow);
				sim_window_drawrstring(s2);

				sim_window_moveto(100, y_max);
				sim_window_drawto(113, y_max);
				sim_window_moveto(112, (INTSML)(y_max-sim_window_txthei));
				(void)sprintf(s2, "%g", sim_window_anahigh);
				sim_window_drawrstring(s2);
			}
		}
	}

	/* set colors for analog traces */
	j = 0;
	for(tr = sim_window_plot; tr != NODISCHANNEL; tr = tr->nextdischannel)
	{
		/* handle analog trace color */
		if (tr->status != 0)
		{
			tr->status = sim_window_anacolor[j];
			j++;
			if (sim_window_anacolor[j] == 0) j = 0;
		}
	}

	/* plot it all */
	for(tr = sim_window_plot; tr != NODISCHANNEL; tr = tr->nextdischannel)
	{
		if (stopping("Simulation Redraw") != 0) break;

		slot = tr->position;
		sim_window_writetracename(tr);
		y_min = (sim_window_lines - 1 - slot) * temp_trace_spacing + 31;
		y_max = y_min + temp_trace_spacing - 10;

		first = 1;
		for(j=0; j<tr->numsteps; j++)
		{
			time = tr->timearray[j];
			if (tr->status == 0)
			{
				/* digital waveform */
				if (j == tr->numsteps-1) x_max = 767; else
				{
					x_max = sim_window_timetoxpos(tr->timearray[j+1]);
					if (x_max < 114) continue;
				}
				x_min = sim_window_timetoxpos(time);
				if (x_min < 114) x_min = 114;
				if (x_min > 767) continue;

				state = tr->statearray[j] >> 8;
				switch (tr->statearray[j] & 0377)
				{
					case NODE_STRENGTH: color = sim_colorstrengthnode;    break;
					case GATE_STRENGTH: color = sim_colorstrengthgate;    break;
					case VDD_STRENGTH:  color = sim_colorstrengthpower;   break;
					default:            color = sim_colorstrengthoff;     break;
				}
				sim_window_setcolor(color);
				if (first == 0 && state != laststate)
				{
					sim_window_moveto(x_min, y_min);
					sim_window_drawto(x_min, y_max);
				}
				first = 0;
				switch (state)
				{
					case LOGIC_LOW:
						sim_window_moveto(x_min, y_min);
						sim_window_drawto(x_max, y_min);
						break;
					case LOGIC_X:
						sim_window_drawbox(x_min, y_min, x_max, y_max);
						break;
					case LOGIC_HIGH:
						sim_window_moveto(x_min, y_max);
						sim_window_drawto(x_max, y_max);
						break;
					default:
						sim_window_drawbox(x_min, y_min, x_max, y_max);
						if ((x_max - x_min) <= 1) break;
						sim_window_setcolor(ALLOFF);
						sim_window_drawbox((INTSML)(x_min+1), (INTSML)(y_min+1), (INTSML)(x_max-1), (INTSML)(y_max-1));
						(void)sprintf(s2, "0x%X", state);
						sim_window_moveto((INTSML)((x_min + x_max) / 2), (INTSML)(y_min+3));
						sim_window_setcolor(color);
						sim_window_drawcstring(s2);
						break;
				}
				laststate = state;
			} else
			{
				/* analog waveform */
				tx = sim_window_timetoxpos(time);
				position = (float)(y_max - y_min);
				position *= tr->valuearray[j];
				ty = (INTSML)(position + y_min);
				if (j != 0 && tx >= 114)
				{
					if (tx > 767)
					{
						/* "to" data point off the screen: interpolate */
						ty = (ty-fy) * (767-fx) / (tx-fx) + fy;
						tx = 767;
					}

					if (first != 0 && fx < 114)
					{
						/* "from" data point off the screen: interpolate */
						fy = (ty-fy) * (114-fx) / (tx-fx) + fy;
						fx = 114;
					}
					first = 0;
					sim_window_setcolor(tr->status);
					sim_window_moveto(fx, fy);
					sim_window_drawto(tx, ty);
				}
				fx = tx;   fy = ty;
				if (fx >= 767) break;
			}
		}
	}
}

/*
 * routine to snapshot the simulation window in a facet with artwork primitives
 */
void sim_window_savegraph(void)
{
	INTSML x_min, x_max, y_min, y_max, j, maxoff, state, laststate, color, first, x,
		slot, temp_trace_spacing, range, hasana, lastcolor, tx1, ty1, tx2, ty2, fx, fy, tx, ty;
	char s2[15];
	REGISTER INTBIG *data, maxsteps;
	INTBIG datapos;
	REGISTER NODEPROTO *np, *plotnp;
	REGISTER NODEINST *ni;
	REGISTER VARIABLE *var;
	REGISTER DISCHANNEL *tr, *otr;
	float time, position;

	/* get facet being saved */
	np = getcurfacet();
	if (np == NONODEPROTO) return;

	/* create the plot facet */
	(void)initinfstr();
	(void)addstringtoinfstr(np->cell->cellname);
	(void)addstringtoinfstr("{sim}");
	plotnp = newnodeproto(returninfstr(), el_curlib);
	if (plotnp == NONODEPROTO) return;
	setactivity("MAKE SIMULATION PLOT");

	/* determine maximum number of steps */
	maxsteps = 1;
	for(tr = sim_window_plot; tr != NODISCHANNEL; tr = tr->nextdischannel)
	{
		if (tr->numsteps > maxsteps) maxsteps = tr->numsteps;
	}

	/* make space */
	data = (INTBIG *)emalloc(maxsteps * 4 * SIZEOFINTBIG, el_tempcluster);
	if (data == 0) return;

	/* compute transformation for points in the plot area */
	tx1 = 114;   ty1 = 31;    sim_window_mapcoord(&tx1, &ty1);
	tx2 = 767;   ty2 = 570;   sim_window_mapcoord(&tx2, &ty2);
	sim_window_offx = (tx2 - tx1) / 2;   sim_window_basex = tx1;
	sim_window_offy = (ty2 - ty1) / 2;   sim_window_basey = ty1;

	/* write the header */
	tx1 = 383;   ty1 = 600;   sim_window_mapcoord(&tx1, &ty1);
	ni = newnodeinst(gen_invispinprim, tx1, tx1, ty1, ty1, 0, 0, plotnp);
	if (ni == NONODEINST) { efree((char *)data);  return; }
	(void)initinfstr();
	(void)addstringtoinfstr("Simulation snapshot of facet ");
	(void)addstringtoinfstr(describenodeproto(np));
	var = setvalkey((INTBIG)ni, VNODEINST, art_messagekey, (INTBIG)returninfstr(), VSTRING|VDISPLAY);
	if (var != NOVARIABLE) var->textdescript = (TXT14P << VTSIZESH) | VTPOSUP;

	/* set name offsets */
	for(tr = sim_window_plot; tr != NODISCHANNEL; tr = tr->nextdischannel) tr->nameoff = -1;
	temp_trace_spacing = 540 / sim_window_lines;
	for(tr = sim_window_plot; tr != NODISCHANNEL; tr = tr->nextdischannel)
	{
		/* determine position of trace label on the line */
		if (tr->nameoff < 0)
		{
			hasana = 0;
			maxoff = 0;
			for(otr = sim_window_plot; otr != NODISCHANNEL; otr = otr->nextdischannel)
			{
				if (otr->position != tr->position) continue;
				maxoff++;
				if (otr->status != 0) hasana++;
			}
			if (maxoff == 1)
			{
				/* only one trace: center it */
				tr->nameoff = (temp_trace_spacing - 10 - sim_window_txthei) / 2;
				if (tr->nameoff < 0) tr->nameoff = 0;
			} else
			{
				/* multiple traces: compute them */
				range = (temp_trace_spacing - 10 - sim_window_txthei*3) / (maxoff-1);
				if (range <= 0) range = 1;
				slot = 0;
				for(otr = sim_window_plot; otr != NODISCHANNEL; otr = otr->nextdischannel)
				{
					if (otr->position != tr->position) continue;
					otr->nameoff = slot * range + sim_window_txthei;
					slot++;
				}
			}

			/* if there are analog traces on this line, show height values */
			if (hasana != 0)
			{
				y_min = (sim_window_lines - 1 - tr->position) * temp_trace_spacing + 31;
				y_max = y_min + temp_trace_spacing - 10;

				tx1 = 101;   ty1 = y_min;   sim_window_mapcoord(&tx1, &ty1);
				tx2 = 113;   ty2 = y_min;   sim_window_mapcoord(&tx2, &ty2);
				data[0] = -(tx2-tx1)/2;  data[1] = 0;
				data[2] = (tx2-tx1)/2;   data[3] = 0;
				ni = newnodeinst(art_openedpolygonprim, tx1,tx2, ty1,ty2, 0, 0, plotnp);
				if (ni == NONODEINST) { efree((char *)data);  return; }
				(void)setvalkey((INTBIG)ni, VNODEINST, el_trace, (INTBIG)data,
					VINTEGER|VISARRAY|(4<<VLENGTHSH));

				(void)sprintf(s2, "%g", sim_window_analow);
				tx1 = 112;   ty1 = y_min;   sim_window_mapcoord(&tx1, &ty1);
				ni = newnodeinst(gen_invispinprim, tx1, tx1, ty1, ty1, 0, 0, plotnp);
				if (ni == NONODEINST) { efree((char *)data);  return; }
				var = setvalkey((INTBIG)ni, VNODEINST, art_messagekey, (INTBIG)s2, VSTRING|VDISPLAY);
				if (var != NOVARIABLE) var->textdescript = (TXT10P << VTSIZESH) | VTPOSUPLEFT;

				tx1 = 101;   ty1 = y_max;   sim_window_mapcoord(&tx1, &ty1);
				tx2 = 113;   ty2 = y_max;   sim_window_mapcoord(&tx2, &ty2);
				data[0] = -(tx2-tx1)/2;  data[1] = 0;
				data[2] = (tx2-tx1)/2;   data[3] = 0;
				ni = newnodeinst(art_openedpolygonprim, tx1,tx2, ty1,ty2, 0, 0, plotnp);
				if (ni == NONODEINST) { efree((char *)data);  return; }
				(void)setvalkey((INTBIG)ni, VNODEINST, el_trace, (INTBIG)data,
					VINTEGER|VISARRAY|(4<<VLENGTHSH));

				(void)sprintf(s2, "%g", sim_window_anahigh);
				tx1 = 112;   ty1 = y_max;   sim_window_mapcoord(&tx1, &ty1);
				ni = newnodeinst(gen_invispinprim, tx1, tx1, ty1, ty1, 0, 0, plotnp);
				if (ni == NONODEINST) { efree((char *)data);  return; }
				var = setvalkey((INTBIG)ni, VNODEINST, art_messagekey, (INTBIG)s2, VSTRING|VDISPLAY);
				if (var != NOVARIABLE) var->textdescript = (TXT10P << VTSIZESH) | VTPOSDOWNLEFT;
			}
		}
	}

	/* set colors for analog traces */
	j = 0;
	for(tr = sim_window_plot; tr != NODISCHANNEL; tr = tr->nextdischannel)
	{
		/* handle analog trace color */
		if (tr->status != 0)
		{
			tr->status = sim_window_anacolor[j];
			j++;
			if (sim_window_anacolor[j] == 0) j = 0;
		}
	}

	/* plot time values */
	for(j=1; j<5; j++)
	{
		time = j * 0.2 * (sim_window_maxtime - sim_window_mintime) + sim_window_mintime;
		x = sim_window_timetoxpos(time);
		sim_windowconvertengineeringnotation(time, s2);
		tx1 = x;   ty1 = 0;   sim_window_mapcoord(&tx1, &ty1);
		ni = newnodeinst(gen_invispinprim, tx1, tx1, ty1, ty1, 0, 0, plotnp);
		if (ni == NONODEINST) { efree((char *)data);  return; }
		var = setvalkey((INTBIG)ni, VNODEINST, art_messagekey, (INTBIG)s2, VSTRING|VDISPLAY);
		if (var != NOVARIABLE) var->textdescript = (TXT10P << VTSIZESH) | VTPOSUP;
	}

	/* plot it all */
	for(tr = sim_window_plot; tr != NODISCHANNEL; tr = tr->nextdischannel)
	{
		if (stopping("Simulation Redraw") != 0) break;

		/* compute positions */
		slot = tr->position;
		y_min = (sim_window_lines - 1 - slot) * temp_trace_spacing + 31;

		/* draw the vertical line between the label and the trace */
		tx1 = 113;   ty1 = y_min;   sim_window_mapcoord(&tx1, &ty1);
		tx2 = 113;   ty2 = y_min+temp_trace_spacing-10;   sim_window_mapcoord(&tx2, &ty2);
		data[0] = 0;   data[1] = -(ty2-ty1)/2;
		data[2] = 0;   data[3] = (ty2-ty1)/2;
		ni = newnodeinst(art_openedpolygonprim, tx1, tx2, ty1, ty2, 0, 0, plotnp);
		if (ni == NONODEINST) { efree((char *)data);  return; }
		(void)setvalkey((INTBIG)ni, VNODEINST, el_trace, (INTBIG)data,
			VINTEGER|VISARRAY|(4<<VLENGTHSH));

		/* write the text */
		(void)sprintf(s2, "%g", sim_window_anahigh);
		tx1 = 1;   ty1 = y_min + tr->nameoff;   sim_window_mapcoord(&tx1, &ty1);
		ni = newnodeinst(gen_invispinprim, tx1, tx1, ty1, ty1, 0, 0, plotnp);
		if (ni == NONODEINST) { efree((char *)data);  return; }
		var = setvalkey((INTBIG)ni, VNODEINST, art_messagekey, (INTBIG)tr->name, VSTRING|VDISPLAY);
		if (var != NOVARIABLE) var->textdescript = (TXT10P << VTSIZESH) | VTPOSUPRIGHT;
		if (tr->status != 0)
		{
			tx1 = 0;     ty1 = y_min + tr->nameoff;   sim_window_mapcoord(&tx1, &ty1);
			tx2 = 100;   ty2 = y_min + tr->nameoff;   sim_window_mapcoord(&tx2, &ty2);
			data[0] = -(tx2-tx1)/2;  data[1] = 0;
			data[2] = (tx2-tx1)/2;   data[3] = 0;
			ni = newnodeinst(art_openedpolygonprim, tx1, tx2, ty1, ty2, 0, 0, plotnp);
			if (ni == NONODEINST) { efree((char *)data);  return; }
			(void)setvalkey((INTBIG)ni, VNODEINST, el_trace, (INTBIG)data,
				VINTEGER|VISARRAY|(4<<VLENGTHSH));
			(void)setvalkey((INTBIG)ni, VNODEINST, art_colorkey, tr->status, VINTEGER);
		}

		y_min = (sim_window_lines - 1 - slot) * temp_trace_spacing + 31;
		y_max = y_min + temp_trace_spacing - 10;

		first = 1;
		datapos = 0;
		laststate = -1;
		for(j=0; j<tr->numsteps; j++)
		{
			time = tr->timearray[j];
			if (tr->status == 0)
			{
				/* digital waveform */
				if (j == tr->numsteps-1) x_max = 767; else
				{
					x_max = sim_window_timetoxpos(tr->timearray[j+1]);
					if (x_max < 114) continue;
				}
				x_min = sim_window_timetoxpos(time);
				if (x_min < 114) x_min = 114;
				if (x_min > 767) continue;

				switch (tr->statearray[j] & 0377)
				{
					case NODE_STRENGTH: color = sim_colorstrengthnode;    break;
					case GATE_STRENGTH: color = sim_colorstrengthgate;    break;
					case VDD_STRENGTH:  color = sim_colorstrengthpower;   break;
					default:            color = sim_colorstrengthoff;     break;
				}
				if (datapos > 0 && color != lastcolor)
				{
					/* strength has changed, dump out data so far */
					tx1 = 114;   ty1 = 31;    sim_window_mapcoord(&tx1, &ty1);
					tx2 = 767;   ty2 = 570;   sim_window_mapcoord(&tx2, &ty2);
					ni = newnodeinst(art_openedpolygonprim, tx1, tx2, ty1, ty2, 0, 0, plotnp);
					if (ni == NONODEINST) { efree((char *)data);  return; }
					(void)setvalkey((INTBIG)ni, VNODEINST, el_trace, (INTBIG)data,
						VINTEGER|VISARRAY|(datapos<<VLENGTHSH));
					(void)setvalkey((INTBIG)ni, VNODEINST, art_colorkey, lastcolor, VINTEGER);
					datapos = 0;
				}
				lastcolor = color;

				state = tr->statearray[j] >> 8;
				switch (state)
				{
					case LOGIC_LOW:
						if (laststate == LOGIC_HIGH)
							sim_window_addsegment(data, &datapos, x_min, y_max, x_min, y_min);
						sim_window_addsegment(data, &datapos, x_min, y_min, x_max, y_min);
						break;
					case LOGIC_HIGH:
						if (laststate == LOGIC_LOW)
							sim_window_addsegment(data, &datapos, x_min, y_min, x_min, y_max);
						sim_window_addsegment(data, &datapos, x_min, y_max, x_max, y_max);
						break;
					default:
						if (datapos > 0)
						{
							/* doing unknown block, dump out any line data */
							tx1 = 114;   ty1 = 31;    sim_window_mapcoord(&tx1, &ty1);
							tx2 = 767;   ty2 = 570;   sim_window_mapcoord(&tx2, &ty2);
							ni = newnodeinst(art_openedpolygonprim, tx1, tx2, ty1, ty2, 0, 0, plotnp);
							if (ni == NONODEINST) { efree((char *)data);  return; }
							(void)setvalkey((INTBIG)ni, VNODEINST, el_trace, (INTBIG)data,
								VINTEGER|VISARRAY|(datapos<<VLENGTHSH));
							(void)setvalkey((INTBIG)ni, VNODEINST, art_colorkey, lastcolor,
								VINTEGER);
						}
						datapos = 0;
						tx1 = x_min;   ty1 = y_min;   sim_window_mapcoord(&tx1, &ty1);
						tx2 = x_max;   ty2 = y_max;   sim_window_mapcoord(&tx2, &ty2);
						ni = newnodeinst(art_boxprim, tx1, tx2, ty1, ty2, 0, 0, plotnp);
						if (ni == NONODEINST) { efree((char *)data);  return; }
						(void)setvalkey((INTBIG)ni, VNODEINST, art_colorkey, lastcolor, VINTEGER);
						break;
				}
				laststate = state;
				first = 0;
			} else
			{
				/* analog waveform */
				tx = sim_window_timetoxpos(time);
				position = (float)(y_max - y_min);
				position *= tr->valuearray[j];
				ty = (INTSML)(position + y_min);
				if (j != 0 && tx >= 114)
				{
					if (tx > 767)
					{
						/* "to" data point off the screen: interpolate */
						ty = (ty-fy) * (767-fx) / (tx-fx) + fy;
						tx = 767;
					}

					if (first != 0 && fx < 114)
					{
						/* "from" data point off the screen: interpolate */
						fy = (ty-fy) * (114-fx) / (tx-fx) + fy;
						fx = 114;
					}
					first = 0;
					sim_window_addsegment(data, &datapos, fx, fy, tx, ty);
				}
				fx = tx;   fy = ty;
				if (fx >= 767) break;
			}
		}
		if (datapos > 0)
		{
			tx1 = 114;   ty1 = 31;    sim_window_mapcoord(&tx1, &ty1);
			tx2 = 767;   ty2 = 570;   sim_window_mapcoord(&tx2, &ty2);
			ni = newnodeinst(art_openedpolygonprim, tx1, tx2, ty1, ty2, 0, 0, plotnp);
			if (ni == NONODEINST) { efree((char *)data);  return; }
			(void)setvalkey((INTBIG)ni, VNODEINST, el_trace, (INTBIG)data,
				VINTEGER|VISARRAY|(datapos<<VLENGTHSH));
			if (tr->status != 0)
				(void)setvalkey((INTBIG)ni, VNODEINST, art_colorkey, tr->status, VINTEGER); else
					(void)setvalkey((INTBIG)ni, VNODEINST, art_colorkey, lastcolor, VINTEGER);
		}
	}

	/* ensure that the facet has the right size */
	(*el_curconstraint->solve)(plotnp);

	/* clean up */
	ttyputmsg("Created facet %s", describenodeproto(plotnp));
	efree((char *)data);
}

/*
 * Set the display color for strength "strength" to "color" (if "strength" is -1, show all colors).
 */
void sim_window_displaycolor(INTSML strength, INTSML color)
{
	char *colorname, *colorsymbol;

	switch (strength)
	{
		case OFF_STRENGTH:
			sim_colorstrengthoff = color;
			break;
		case NODE_STRENGTH:
			sim_colorstrengthnode = color;
			break;
		case GATE_STRENGTH:
			sim_colorstrengthgate = color;
			break;
		case VDD_STRENGTH:
			sim_colorstrengthpower = color;
			break;
		case LOGIC_LOW:
			sim_colorlevellow = color;
			break;
		case LOGIC_X:
			sim_colorlevelundef = color;
			break;
		case LOGIC_HIGH:
			sim_colorlevelhigh = color;
			break;
		default:
			if (ecolorname(sim_colorstrengthoff, &colorname, &colorsymbol) != 0)
				colorname = "**UNKNOWN**";
			ttyputmsg("Off-strength signals are colored %s", colorname);
			if (ecolorname(sim_colorstrengthnode, &colorname, &colorsymbol) != 0)
				colorname = "**UNKNOWN**";
			ttyputmsg("Node-strength signals are colored %s", colorname);
			if (ecolorname(sim_colorstrengthgate, &colorname, &colorsymbol) != 0)
				colorname = "**UNKNOWN**";
			ttyputmsg("Gate-strength signals are colored %s", colorname);
			if (ecolorname(sim_colorstrengthpower, &colorname, &colorsymbol) != 0)
				colorname = "**UNKNOWN**";
			ttyputmsg("Power-strength signals are colored %s", colorname);
			if (ecolorname(sim_colorlevellow, &colorname, &colorsymbol) != 0)
				colorname = "**UNKNOWN**";
			ttyputmsg("Low-level signals are colored %s", colorname);
			if (ecolorname(sim_colorlevelhigh, &colorname, &colorsymbol) != 0)
				colorname = "**UNKNOWN**";
			ttyputmsg("High-level signals are colored %s", colorname);
			if (ecolorname(sim_colorlevelundef, &colorname, &colorsymbol) != 0)
				colorname = "**UNKNOWN**";
			ttyputmsg("Undefined-level signals are colored %s", colorname);
			break;
	}
}

void sim_window_addsegment(INTBIG *data, INTBIG *datapos, INTSML fx, INTSML fy,
	INTSML tx, INTSML ty)
{
	REGISTER INTSML pos, truefx, truefy;
	INTSML x, y;

	x = fx;   y = fy;   sim_window_mapcoord(&x, &y);
	truefx = (x-sim_window_basex) - sim_window_offx;
	truefy = (y-sim_window_basey) - sim_window_offy;

	pos = *datapos;
	if (pos < 2 || data[pos-2] != truefx || data[pos-1] != truefy)
	{
		data[pos++] = truefx;
		data[pos++] = truefy;
	}

	x = tx;   y = ty;   sim_window_mapcoord(&x, &y);
	data[pos++] = (x-sim_window_basex) - sim_window_offx;
	data[pos++] = (y-sim_window_basey) - sim_window_offy;
	*datapos = pos;
}

/********************************* LOW LEVEL GRAPHICS *********************************/

void sim_window_setviewport(WINDOW *w)
{
	float scalex, scaley;
	INTSML x, y;

	sim_window = w;
	if (w == NOWINDOW) return;
	w->screenlx = 0;   w->screenhx = 767;
	w->screenly = 0;   w->screenhy = 600;
	computewindowscale(w);

	scalex = (w->usehx - w->uselx) / 767.0;
	scaley = ((w->usehy-20) - w->usely) / 600.0;
	if (scalex > 2.0 && scaley > 2.0) sim_window_textsize = TXT20P; else
		if (scalex > 1.0 && scaley > 1.0) sim_window_textsize = TXT16P; else
			if (scalex > 0.75 && scaley > 0.75) sim_window_textsize = TXT12P; else
				sim_window_textsize = TXT8P;

	/* determine height of text in the window */
	us_settextsize(w, sim_window_textsize);
	us_textsize(w, "X", &x, &y);
	sim_window_txthei = y * (w->screenhy - w->screenly) / (w->usehy - w->usely - 20);
}

void sim_window_mapcoord(INTSML *x, INTSML *y)
{
	REGISTER INTSML newx, newy;

	if (sim_window == NOWINDOW) return;
	newx = sim_window->uselx + applyxscale(sim_window, *x - sim_window->screenlx);
	newy = sim_window->usely + (*y - sim_window->screenly) *
		((sim_window->usehy-20) - (sim_window->usely)) / (sim_window->screenhy - sim_window->screenly);
	*x = newx;   *y = newy;
}

void sim_window_scaletowindow(INTBIG *x, INTBIG *y)
{
	if (sim_window == NOWINDOW) return;
	*x = muldiv(*x - sim_window->uselx, sim_window->screenhx - sim_window->screenlx,
		sim_window->usehx - sim_window->uselx) + sim_window->screenlx;
	*y = muldiv(*y - sim_window->usely, sim_window->screenhy - sim_window->screenly,
		(sim_window->usehy-20) - sim_window->usely) + sim_window->screenly;
}

/* Some Control Sequences for the graphics terminal */
void sim_window_setcolor(INTSML color)
{
	sim_window_desc.col = color;
}

void sim_window_setmask(INTSML mask)
{
	sim_window_desc.bits = mask;
}

/* Graphics Commands */
void sim_window_moveto(INTSML x, INTSML y)
{
	sim_window_curx = x;
	sim_window_cury = y;
}

void sim_window_drawto(INTSML x, INTSML y)
{
	INTSML fx, fy, tx, ty;
	INTBIG cfx, cfy, ctx, cty;

	if (sim_window == NOWINDOW) return;
	fx = sim_window_curx;   fy = sim_window_cury;
	tx = x;   ty = y;
	sim_window_mapcoord(&fx, &fy);
	sim_window_mapcoord(&tx, &ty);
	cfx = fx;   cfy = fy;   ctx = tx;   cty = ty;
	if (clipline(&cfx, &cfy, &ctx, &cty, sim_window->uselx, sim_window->usehx, sim_window->usely,
		sim_window->usehy) == 0)
	{
		fx = cfx;   fy = cfy;   tx = ctx;   ty = cty;
		us_drawline(sim_window, fx, fy, tx, ty, &sim_window_desc, 0);
	}
	sim_window_curx = x;
	sim_window_cury = y;
}

void sim_window_drawbox(INTSML xll, INTSML yll, INTSML xur, INTSML yur)
{
	INTSML lx, ly, hx, hy, swap;

	if (sim_window == NOWINDOW) return;
	lx = xll;   ly = yll;
	hx = xur;   hy = yur;
	sim_window_mapcoord(&lx, &ly);
	sim_window_mapcoord(&hx, &hy);
	if (lx > hx) { swap = lx;   lx = hx;   hx = swap; }
	if (ly > hy) { swap = ly;   ly = hy;   hy = swap; }
	us_drawbox(sim_window, lx, hx, ly, hy, &sim_window_desc);
}

void sim_window_drawcstring(char *s)
{
	INTSML x, y, px, py;

	if (sim_window == NOWINDOW) return;
	px = sim_window_curx;   py = sim_window_cury;
	sim_window_mapcoord(&px, &py);
	us_settextsize(sim_window, sim_window_textsize);
	us_textsize(sim_window, s, &x, &y);
	us_puttext(sim_window, (INTSML)(px-x/2), py, s, &sim_window_desc);
}

void sim_window_drawlstring(char *s, INTSML maxx)
{
	INTSML x, y, save, hch, ux, uy, px, py;

	if (sim_window == NOWINDOW) return;
	ux = maxx;
	uy = sim_window_cury;
	sim_window_mapcoord(&ux, &uy);
	px = sim_window_curx;   py = sim_window_cury;
	sim_window_mapcoord(&px, &py);

	us_settextsize(sim_window, sim_window_textsize);
	for(hch = strlen(s); hch > 0; hch--)
	{
		save = s[hch];
		s[hch] = 0;
		us_textsize(sim_window, s, &x, &y);
		if (x <= ux - px) break;
		s[hch] = save;
	}

	us_puttext(sim_window, px, py, s, &sim_window_desc);
	s[hch] = save;
}

void sim_window_drawrstring(char *s)
{
	INTSML x, y, px, py;

	if (sim_window == NOWINDOW) return;
	us_settextsize(sim_window, sim_window_textsize);
	us_textsize(sim_window, s, &x, &y);
	px = sim_window_curx;   py = sim_window_cury;
	sim_window_mapcoord(&px, &py);
	us_puttext(sim_window, (INTSML)(px-x), py, s, &sim_window_desc);
}

/*
 * Name: sim_windowconvertengineeringnotation
 *
 * Description:
 *	This procedure converts a floating point number and represents time
 * in engineering units such as pico, micro, milli, etc.
 *
 * Calling Arguments:
 *	time = floating point value to be converted to Engineering notation
 *	s1   = pointer to a string that will be used to print the converted value
 */
void sim_windowconvertengineeringnotation(float time, char *s1)
{
	if ((time < 10.0e-15) || (time >= 9999.5))
	{
		(void)sprintf(s1, "%8.2e", time);
		return;
	}
	if ((time >= 10.0e-15) && (time < 9999.5e-15))
	{
		time = (time / 1.0e-15) + 0.5;
		(void)sprintf(s1, "%4dfsec", (INTSML)time);
		return;
	}
	if ((time >= 1.0e-12) && (time < 9999.5e-12))
	{
		time = (time / 1.0e-12) + 0.5;
		(void)sprintf(s1, "%4dpsec", (INTSML)time);
		return;
	}
	if ((time >= 1.0e-9) && (time < 9999.5e-9))
	{
		time = (time / 1.0e-9) + 0.5;
		(void)sprintf(s1, "%4dnsec", (INTSML)time);
		return;
	}
	if ((time >= 1.0e-6) && (time < 9999.5e-6))
	{
		time = (time / 1.0e-6) + 0.5;
		(void)sprintf(s1, "%4dusec", (INTSML)time);
		return;
	}
	if ((time >= 1.0e-3) && (time < 9999.5e-3))
	{
		time = (time / 1.0e-3) + 0.5;
		(void)sprintf(s1, "%4dmsec", (INTSML)time);
		return;
	}
	if ((time >= 1.0) && (time < 9999.5))
	{
		time = time + 0.5;
		(void)sprintf(s1, "%4d sec", (INTSML)time);
		return;
	}
}

#endif  /* SIMAID - at top */
