/*
 * Electric(tm) VLSI Design System
 *
 * File: tecgen.c
 * Generic technology description
 * Written by: Steven M. Rubin, Electric Editor Incorporated
 *
 * Copyright (c) 1998 Electric Editor Incorporated.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Electric Editor Incorporated
 * 23470 Sunset Drive, Suite 108
 * Los Gatos, California 95033
 * support@electriceditor.com
 */

#include "global.h"
#include "egraphics.h"
#include "tech.h"
#include "efunction.h"
#include "tecgen.h"
#include "tecart.h"

TECHNOLOGY *gen_tech;
NODEPROTO  *gen_univpinprim, *gen_invispinprim, *gen_unroutedpinprim, *gen_facetcenterprim,
	*gen_portprim, *gen_drcprim;
ARCPROTO   *gen_universalarc, *gen_invisiblearc, *gen_unroutedarc;

/* prototypes for local routines */
void gen_makeunivlist(void);

/******************** LAYERS ********************/

#define	MAXLAYERS       5		/* total layers below */
#define	LUNIV           0		/* universal layer (connects all) */
#define	LINVIS          1		/* invisible layer (nonelectrical) */
#define	LUNROUTED       2		/* unrouted layer (for routers) */
#define	LGLYPH          3		/* glyph layer (for menu icons) */
#define	LDRC            4		/* drc ignore layer */

static GRAPHICS gen_un_lay = {LAYERO, MENTXT, {SOLIDC,SOLIDC,SOLIDC,SOLIDC},
/* univ */					{0,0,0,0,0,0,0,0}, NOVARIABLE, 0};
static GRAPHICS gen_in_lay = {LAYERO, GRAY,  {SOLIDC,SOLIDC,SOLIDC,SOLIDC},
/* invis */					{0,0,0,0,0,0,0,0}, NOVARIABLE, 0};
static GRAPHICS gen_int_lay = {LAYERO, BLACK,  {SOLIDC,SOLIDC,SOLIDC,SOLIDC},
/* invis text */			{0,0,0,0,0,0,0,0}, NOVARIABLE, 0};
static GRAPHICS gen_ur_lay = {LAYERO, DGRAY, {SOLIDC,SOLIDC,SOLIDC,SOLIDC},
/* unrouted */				{0,0,0,0,0,0,0,0}, NOVARIABLE, 0};
static GRAPHICS gen_gl_lay = {LAYERO, MENGLY, {SOLIDC,SOLIDC,SOLIDC,SOLIDC},
/* glyph */					{0,0,0,0,0,0,0,0}, NOVARIABLE, 0};
static GRAPHICS gen_dr_lay = {LAYERO, ORANGE, {SOLIDC,SOLIDC,SOLIDC,SOLIDC},
/* drc */					{0,0,0,0,0,0,0,0}, NOVARIABLE, 0};

/* these tables must be updated togehter */
GRAPHICS *gen_layers[MAXLAYERS+1] = {&gen_un_lay, &gen_in_lay,
	&gen_ur_lay, &gen_gl_lay, &gen_dr_lay, NOGRAPHICS};
static char *gen_layer_names[MAXLAYERS] = {"Universal", "Invisible",
	"Unrouted", "Glyph", "DRC"};
static char *gen_cif_layers[MAXLAYERS] = {"", "", "", "", "DRC"};
static INTBIG gen_layer_function[MAXLAYERS] = {LFUNKNOWN, LFUNKNOWN|LFNONELEC,
	LFUNKNOWN, LFART|LFNONELEC, LFART|LFNONELEC};
static char *gen_layer_letters[MAXLAYERS] = {"u", "i", "r", "g", "d"};

/******************** ARCS ********************/

#define	ARCPROTOCOUNT     3
#define GENAUNIV          0
#define GENAINVIS         1
#define GENAUNROUTED      2

/* universal arc */
static TECH_ARCLAY gen_al_u[] = {{LUNIV,0,FILLED}};
static TECH_ARCS gen_a_u = {
	"Universal",0,GENAUNIV,									/* name */
	1,gen_al_u,												/* layers */
	(APUNKNOWN<<AFUNCTIONSH)|WANTFIXANG|(45<<AANGLEINCSH)};	/* userbits */

/* invisible arc */
static TECH_ARCLAY gen_al_i[] = {{LINVIS,0,FILLED}};
static TECH_ARCS gen_a_i = {
	"Invisible",0,GENAINVIS,								/* name */
	1,gen_al_i,												/* layers */
	(APNONELEC<<AFUNCTIONSH)|WANTFIXANG|(45<<AANGLEINCSH)};	/* userbits */

/* unrouted arc */
static TECH_ARCLAY gen_al_r[] = {{LUNROUTED,0,FILLED}};
static TECH_ARCS gen_a_r = {
	"Unrouted",0,GENAUNROUTED,								/* name */
	1,gen_al_r,												/* layers */
	(APUNROUTED<<AFUNCTIONSH)|(0<<AANGLEINCSH)};			/* userbits */

TECH_ARCS *gen_arcprotos[ARCPROTOCOUNT+1] = {
	&gen_a_u, &gen_a_i, &gen_a_r, ((TECH_ARCS *)-1)};

/******************** PORT CONNECTIONS **************************/

/* these values are replaced with actual arcproto addresses */
static INTBIG gen_pc_u[] = {-1, GENAUNIV, -1};
static INTBIG gen_pc_i[] = {-1, GENAINVIS, GENAUNIV, -1};
static INTBIG gen_pc_r[] = {-1, GENAUNROUTED, GENAUNIV, -1};

/******************** NODES ********************/

#define	NODEPROTOCOUNT    25
#define	NUNIV              1		/* universal pin */
#define	NINVIS             2		/* invisible pin */
#define	NUNROUTED          3		/* unrouted pin */
#define	NCENTER            4		/* facet center */
#define	NPORT              5		/* port declaration (for tech edit) */
#define	NDRC               6		/* drc ignore mask (for ECAD's DRC) */
#define	NDOWNV             7		/* window down glyph */
#define	NERASEV            8		/* erase glyph */
#define	NEXPANDV           9		/* node expand glyph */
#define	NGRIDV            10		/* grid glyph */
#define	NLEFTV            11		/* window left glyph */
#define	NOUTHIERV         12		/* outhier glyph */
#define	NQUITV            13		/* quit glyph */
#define	NREDRAWV          14		/* redraw glyph */
#define	NRIGHTV           15		/* window right glyph */
#define	NROTATEV          16		/* rotate glyph */
#define	NSAVEV            17		/* save glyph */
#define	NSYSTEMV          18		/* system glyph */
#define	NUNDOV            19		/* undo glyph */
#define	NUNEXPANDV        20		/* node unexpand glyph */
#define	NUPV              21		/* window up glyph */
#define	NVIEWALLV         22		/* window viewall glyph */
#define	NVIEWV            23		/* window view glyph */
#define	NZOOMOUTV         24		/* window zoomout glyph */
#define	NZOOMINV          25		/* window zoomin glyph */

static INTBIG gen_fullbox[]    = {LEFTEDGE, BOTEDGE, RIGHTEDGE, TOPEDGE};
static INTBIG gen_disccenter[] = {CENTER,   CENTER,  RIGHTEDGE, CENTER};
static INTBIG gen_in2box[]     = {LEFTIN2,  BOTIN2,  RIGHTIN2,  TOPIN2};
static INTBIG gen_center_v2[]  = {CENTER,   CENTER,  CENTER,    CENTER};
static INTBIG gen_portabox[]   = {CENTER,   CENTER,  CENTER,    CENTER,
								 CENTER,   CENTER};

/* Universal pin */
static TECH_PORTS gen_u_p[] = {						/* ports */
	{(INTBIG *)0, "univ", NOPORTPROTO, (180<<PORTARANGESH),
		CENTER, CENTER, CENTER, CENTER}};
static TECH_POLYGON gen_u_l[] = {{					/* layers */
	LUNIV, 0, 2, DISC, POINTS, gen_disccenter}};
static TECH_NODES gen_u = {
	"Universal-Pin",NUNIV,NONODEPROTO,				/* name */
	K1,K1,											/* size */
	1,gen_u_p,										/* ports */
	1,gen_u_l,										/* layers */
	(NPPIN<<NFUNCTIONSH)|HOLDSTRACE|WIPEON1OR2,		/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* Invisible Pin */
static TECH_PORTS gen_i_p[] = {{						/* ports */
	gen_pc_i, "center", NOPORTPROTO, (180<<PORTARANGESH),
		CENTER, CENTER, CENTER, CENTER}};
static TECH_POLYGON gen_i_l[] = {{					/* layers */
	LINVIS, 0, 4, CLOSEDRECT, BOX, gen_fullbox}};
static TECH_NODES gen_i = {
	"Invisible-Pin",NINVIS,NONODEPROTO,				/* name */
	K1,K1,											/* size */
	1,gen_i_p,										/* ports */
	1,gen_i_l,										/* layers */
	(NPPIN<<NFUNCTIONSH)|WIPEON1OR2,				/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* Unrouted Pin */
static TECH_PORTS gen_r_p[] = {{						/* ports */
	gen_pc_r, "unrouted", NOPORTPROTO, (180<<PORTARANGESH),
		CENTER, CENTER, CENTER, CENTER}};
static TECH_POLYGON gen_r_l[] = {{					/* layers */
	LUNROUTED, 0, 2, DISC, POINTS, gen_disccenter}};
static TECH_NODES gen_r = {
	"Unrouted-Pin",NUNROUTED,NONODEPROTO,			/* name */
	K1,K1,											/* size */
	1,gen_r_p,										/* ports */
	1,gen_r_l,										/* layers */
	(NPPIN<<NFUNCTIONSH)|WIPEON1OR2,				/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* Facet Center */
static TECH_PORTS gen_c_p[] = {{						/* ports */
	gen_pc_u, "center", NOPORTPROTO, (180<<PORTARANGESH),
		CENTER, CENTER, CENTER, CENTER}};
static TECH_POLYGON gen_c_l[] = {{					/* layers */
	LGLYPH, 0, 2, BIGCROSS, POINTS, gen_center_v2}};
static TECH_NODES gen_c = {
	"Facet-Center",NCENTER,NONODEPROTO,				/* name */
	K0,K0,											/* size */
	1,gen_c_p,										/* ports */
	1,gen_c_l,										/* layers */
	(NPART<<NFUNCTIONSH),							/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* Port */
static TECH_POLYGON gen_p_l[] = {					/* layers */
	{LGLYPH, 0,         4, CLOSED,   BOX,    gen_in2box},
	{LGLYPH, 0,         3, OPENED,   POINTS, gen_portabox}};
static TECH_NODES gen_p = {
	"Port",NPORT,NONODEPROTO,						/* name */
	K6,K6,											/* size */
	1,gen_c_p,										/* ports */
	2,gen_p_l,										/* layers */
	(NPART<<NFUNCTIONSH),							/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* DRC Node */
static TECH_POLYGON gen_d_l[] = {					/* layers */
	{LDRC, 0, 4, CLOSEDRECT, BOX, gen_fullbox}};
static TECH_NODES gen_d = {
	"DRC-Node",NDRC,NONODEPROTO,					/* name */
	K2,K2,											/* size */
	1,gen_c_p,										/* ports */
	1,gen_d_l,										/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE,				/* userbits */
	POLYGONAL,0,0,0,0,0,0};							/* characteristics */

/* window down glyph */
static INTBIG gen_down_v[] = {0,40*K1,0,-40*K1, 0,-40*K1,-25*K1,-15*K1,
	0,-40*K1,25*K1,-15*K1};
static TECH_POLYGON gen_down_l[] = {				/* layers */
	{LGLYPH, 0, 6, VECTORS, ABSPOINTS, gen_down_v}};
static TECH_NODES gen_downv = {
	"Window-Down",NDOWNV,NONODEPROTO,				/* name */
	K1*100,K1*100,									/* size */
	1,gen_u_p, 1,gen_down_l,						/* detail */
	(NPART<<NFUNCTIONSH),							/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* erase glyph */
static INTBIG gen_erase_v[] = {50*K1,25*K1,-10*K1,-35*K1,
	-10*K1,-35*K1,-20*K1,-40*K1, -20*K1,-40*K1,-30*K1,-40*K1,
	-30*K1,-40*K1,-40*K1,-30*K1, -40*K1,-30*K1,-40*K1,-20*K1,
	-40*K1,-20*K1,-35*K1,-10*K1, -35*K1,-10*K1,25*K1,50*K1, 0,-25*K1,-25*K1,0,
	5*K1,-20*K1,-20*K1,5*K1, 20*K1,-5*K1,-5*K1,20*K1, 25*K1,0,0,25*K1};
static TECH_POLYGON gen_erase_l[] = {				/* layers */
	{LGLYPH, 0, 22, VECTORS, ABSPOINTS, gen_erase_v}};
static TECH_NODES gen_erasev = {
	"Erase",NERASEV,NONODEPROTO,					/* name */
	K1*100,K1*100,									/* size */
	1,gen_u_p, 1,gen_erase_l,						/* detail */
	(NPART<<NFUNCTIONSH),							/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* node expand glyph */
static INTBIG gen_expand1_v[] = {-50*K1,0,-40*K1,10*K1,
	-40*K1,10*K1,-25*K1,20*K1, -25*K1,20*K1,-10*K1,25*K1,
	-10*K1,25*K1,10*K1,25*K1, 10*K1,25*K1,25*K1,20*K1,
	25*K1,20*K1,40*K1,10*K1, 40*K1,10*K1,50*K1,0*K1,
	50*K1,0,40*K1,-10*K1, 40*K1,-10*K1,25*K1,-20*K1,
	25*K1,-20*K1,10*K1,-25*K1, 10*K1,-25*K1,-10*K1,-25*K1,
	-10*K1,-25*K1,-25*K1,-20*K1, -25*K1,-20*K1,-40*K1,-10*K1,
	-40*K1,-10*K1,-50*K1,0, 5*K1,25*K1,15*K1,20*K1,
	15*K1,20*K1,20*K1,15*K1, 20*K1,15*K1,25*K1,5*K1, 25*K1,5*K1,25*K1,-5*K1,
	25*K1,-5*K1,20*K1,-15*K1, 20*K1,-15*K1,15*K1,-20*K1};
static INTBIG gen_expand2_v[] = {15*K1,-20*K1,5*K1,-25*K1,
	-5*K1,-25*K1,-15*K1,-20*K1, -15*K1,-20*K1,-20*K1,-15*K1,
	-20*K1,-15*K1,-25*K1,-5*K1, -25*K1,-5*K1,-25*K1,5*K1,
	-25*K1,5*K1,-20*K1,15*K1, -20*K1,15*K1,-15*K1,20*K1,
	-15*K1,20*K1,-5*K1,25*K1, 5*K1,10*K1,10*K1,5*K1,
	10*K1,5*K1,10*K1,-5*K1, 10*K1,-5*K1,5*K1,-10*K1, 5*K1,-10*K1,-5*K1,-10*K1,
	-5*K1,-10*K1,-10*K1,-5*K1, -10*K1,-5*K1,-10*K1,5*K1,
	-10*K1,5*K1,-5*K1,10*K1, -5*K1,10*K1,5*K1,10*K1};
static TECH_POLYGON gen_expand_l[] = {				/* layers */
	{LGLYPH, 0, 40, VECTORS, ABSPOINTS, gen_expand1_v},
	{LGLYPH, 0, 32, VECTORS, ABSPOINTS, gen_expand2_v}};
static TECH_NODES gen_expandv = {
	"Node-Expand",NEXPANDV,NONODEPROTO,				/* name */
	K1*100,K1*100,									/* size */
	1,gen_u_p, 2,gen_expand_l,						/* detail */
	(NPART<<NFUNCTIONSH),							/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* grid glyph */
static INTBIG gen_grid_v[] = {-20*K1,-40*K1,-20*K1,40*K1,
	20*K1,-40*K1,20*K1,40*K1, -40*K1,-20*K1,40*K1,-20*K1,
	-40*K1,20*K1,40*K1,20*K1, 0,-40*K1,0,40*K1, -40*K1,0,40*K1,0};
static TECH_POLYGON gen_grid_l[] = {				/* layers */
	{LGLYPH, 0, 12, VECTORS, ABSPOINTS, gen_grid_v}};
static TECH_NODES gen_gridv = {
	"Grid",NGRIDV,NONODEPROTO,						/* name */
	K1*100,K1*100,									/* size */
	1,gen_u_p, 1,gen_grid_l,						/* detail */
	(NPART<<NFUNCTIONSH),							/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* window left glyph */
static INTBIG gen_left_v[] = {40*K1,0,-40*K1,0, -40*K1,0,-15*K1,25*K1,
	-40*K1,0,-15*K1,-25*K1};
static TECH_POLYGON gen_left_l[] = {				/* layers */
	{LGLYPH, 0, 6, VECTORS, ABSPOINTS, gen_left_v}};
static TECH_NODES gen_leftv = {
	"Window-Left",NLEFTV,NONODEPROTO,				/* name */
	K1*100,K1*100,									/* size */
	1,gen_u_p, 1,gen_left_l,						/* detail */
	(NPART<<NFUNCTIONSH),							/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* outhier glyph */
static INTBIG gen_outhier_v[] = {-5*K1,45*K1,5*K1,45*K1, 5*K1,45*K1,5*K1,35*K1,
	5*K1,35*K1,-5*K1,35*K1, -5*K1,35*K1,-5*K1,45*K1,
	-45*K1,-45*K1,-45*K1,-35*K1, -45*K1,-35*K1,-35*K1,-35*K1,
	-35*K1,-35*K1,-35*K1,-45*K1, -35*K1,-45*K1,-45*K1,-45*K1,
	45*K1,-45*K1,45*K1,-35*K1, 45*K1,-35*K1,35*K1,-35*K1,
	35*K1,-35*K1,35*K1,-45*K1, 35*K1,-45*K1,45*K1,-45*K1,
	-5*K1,-45*K1,5*K1,-45*K1, 5*K1,-45*K1,5*K1,-35*K1,
	5*K1,-35*K1,-5*K1,-35*K1, -5*K1,-35*K1,-5*K1,-45*K1, 0,35*K1,0,-35*K1,
	0,35*K1,-40*K1,-35*K1, 0,35*K1,40*K1,-35*K1, 10*K1,-30*K1,10*K1,-5*K1,
	10*K1,-5*K1,15*K1,-10*K1, 10*K1,-5*K1,5*K1,-10*K1};
static TECH_POLYGON gen_outhier_l[] = {				/* layers */
	{LGLYPH, 0, 44, VECTORS, ABSPOINTS, gen_outhier_v}};
static TECH_NODES gen_outhierv = {
	"Outhier",NOUTHIERV,NONODEPROTO,				/* name */
	K1*100,K1*100,									/* size */
	1,gen_u_p, 1,gen_outhier_l,						/* detail */
	(NPART<<NFUNCTIONSH),							/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* quit glyph */
static INTBIG gen_quit_v[] = {-25*K1,0,25*K1,0, 25*K1,0,25*K1,-45*K1,
	25*K1,-45*K1,-25*K1,-45*K1, -25*K1,-45*K1,-25*K1,0, -5*K1,0,-5*K1,30*K1,
	-5*K1,30*K1,-20*K1,30*K1, -20*K1,30*K1,-20*K1,35*K1,
	-20*K1,35*K1,20*K1,35*K1, 20*K1,35*K1,20*K1,30*K1, 20*K1,30*K1,5*K1,30*K1,
	5*K1,30*K1,5*K1,0};
static TECH_POLYGON gen_quit_l[] = {				/* layers */
	{LGLYPH, 0, 22, VECTORS, ABSPOINTS, gen_quit_v}};
static TECH_NODES gen_quitv = {
	"Quit",NQUITV,NONODEPROTO,						/* name */
	K1*100,K1*100,									/* size */
	1,gen_u_p, 1,gen_quit_l,						/* detail */
	(NPART<<NFUNCTIONSH),							/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* redraw glyph */
static INTBIG gen_redraw_v[] = {50*K1,25*K1,5*K1,-20*K1,
	5*K1,-20*K1,-45*K1,-45*K1, -45*K1,-45*K1,-20*K1,5*K1,
	-20*K1,5*K1,25*K1,50*K1, -25*K1,-35*K1,-35*K1,-25*K1,
	-20*K1,5*K1,-10*K1,5*K1, -10*K1,5*K1,-10*K1,-5*K1, -10*K1,-5*K1,0,-5*K1,
	0,-5*K1,0,-15*K1, 0,-15*K1,10*K1,-15*K1};
static TECH_POLYGON gen_redraw_l[] = {				/* layers */
	{LGLYPH, 0, 20, VECTORS, ABSPOINTS, gen_redraw_v}};
static TECH_NODES gen_redrawv = {
	"Redraw",NREDRAWV,NONODEPROTO,					/* name */
	K1*100,K1*100,									/* size */
	1,gen_u_p, 1,gen_redraw_l,						/* detail */
	(NPART<<NFUNCTIONSH),							/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* window right glyph */
static INTBIG gen_right_v[] = {-40*K1,0,40*K1,0, 40*K1,0,15*K1,25*K1,
	40*K1,0,15*K1,-25*K1};
static TECH_POLYGON gen_right_l[] = {				/* layers */
	{LGLYPH, 0, 6, VECTORS, ABSPOINTS, gen_right_v}};
static TECH_NODES gen_rightv = {
	"Window-Right",NRIGHTV,NONODEPROTO,				/* name */
	K1*100,K1*100,									/* size */
	1,gen_u_p, 1,gen_right_l,						/* detail */
	(NPART<<NFUNCTIONSH),							/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* rotate glyph */
static INTBIG gen_rotate_v[] = {25*K1,-35*K1,25*K1,-10*K1,
	25*K1,-10*K1,20*K1,5*K1, 20*K1,5*K1,15*K1,10*K1, 15*K1,10*K1,0,15*K1,
	0,15*K1,-25*K1,15*K1, -25*K1,15*K1,-5*K1,35*K1, -25*K1,15*K1,-5*K1,-5*K1};
static TECH_POLYGON gen_rotate_l[] = {				/* layers */
	{LGLYPH, 0, 14, VECTORS, ABSPOINTS, gen_rotate_v}};
static TECH_NODES gen_rotatev = {
	"Rotate",NROTATEV,NONODEPROTO,					/* name */
	K1*100,K1*100,									/* size */
	1,gen_u_p, 1,gen_rotate_l,						/* detail */
	(NPART<<NFUNCTIONSH),							/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* save glyph */
static INTBIG gen_save_v[] = {-5*K1,-40*K1,40*K1,-40*K1,
	40*K1,-40*K1,40*K1,40*K1, 40*K1,40*K1,-5*K1,40*K1,
	-5*K1,40*K1,-5*K1,-40*K1, 0,-35*K1,35*K1,-35*K1,
	35*K1,-35*K1,35*K1,-15*K1, 35*K1,-15*K1,0,-15*K1, 0,-15*K1,0,-35*K1,
	0,-10*K1,35*K1,-10*K1, 35*K1,-10*K1,35*K1,10*K1, 35*K1,10*K1,0,10*K1,
	0,10*K1,0,-10*K1, 0,15*K1,35*K1,15*K1, 35*K1,15*K1,35*K1,35*K1,
	35*K1,35*K1,0,35*K1, 0,35*K1,0,15*K1, -40*K1,0,-10*K1,0,
	-10*K1,0,-25*K1,15*K1, -10*K1,0,-25*K1,-15*K1};
static TECH_POLYGON gen_save_l[] = {				/* layers */
	{LGLYPH, 0, 38, VECTORS, ABSPOINTS, gen_save_v}};
static TECH_NODES gen_savev = {
	"Save",NSAVEV,NONODEPROTO,						/* name */
	K1*100,K1*100,									/* size */
	1,gen_u_p, 1,gen_save_l,						/* detail */
	(NPART<<NFUNCTIONSH),							/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* system glyph */
static INTBIG gen_system_v[] = {-20*K1,-35*K1,-30*K1,-35*K1,
	-30*K1,-35*K1,-30*K1,-45*K1, -30*K1,-45*K1,30*K1,-45*K1,
	30*K1,-45*K1,30*K1,-35*K1, 30*K1,-35*K1,20*K1,-35*K1,
	20*K1,-35*K1,35*K1,-5*K1, 35*K1,-5*K1,35*K1,15*K1,
	35*K1,15*K1,30*K1,30*K1, 30*K1,30*K1,20*K1,40*K1, 20*K1,40*K1,5*K1,45*K1,
	5*K1,45*K1,-5*K1,45*K1, -5*K1,45*K1,-20*K1,40*K1,
	-20*K1,40*K1,-30*K1,30*K1, -30*K1,30*K1,-35*K1,15*K1,
	-35*K1,15*K1,-35*K1,-5*K1, -35*K1,-5*K1,-20*K1,-35*K1};
static TECH_POLYGON gen_system_l[] = {				/* layers */
	{LGLYPH, 0, 32, VECTORS, ABSPOINTS, gen_system_v}};
static TECH_NODES gen_systemv = {
	"System",NSYSTEMV,NONODEPROTO,					/* name */
	K1*100,K1*100,									/* size */
	1,gen_u_p, 1,gen_system_l,						/* detail */
	(NPART<<NFUNCTIONSH),							/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* undo glyph */
static INTBIG gen_undo_v[] = {-15*K1,0,-45*K1,20*K1, -45*K1,20*K1,-45*K1,40*K1,
	-45*K1,40*K1,0,10*K1, 0,10*K1,45*K1,40*K1, 45*K1,40*K1,45*K1,20*K1,
	45*K1,20*K1,15*K1,0, 15*K1,0,45*K1,-20*K1, 45*K1,-20*K1,45*K1,-40*K1,
	45*K1,-40*K1,0,-10*K1, 0,-10*K1,-45*K1,-40*K1,
	-45*K1,-40*K1,-45*K1,-20*K1, -45*K1,-20*K1,-15*K1,0};
static TECH_POLYGON gen_undo_l[] = {				/* layers */
	{LGLYPH, 0, 24, VECTORS, ABSPOINTS, gen_undo_v}};
static TECH_NODES gen_undov = {
	"Undo",NUNDOV,NONODEPROTO,						/* name */
	K1*100,K1*100,									/* size */
	1,gen_u_p, 1,gen_undo_l,						/* detail */
	(NPART<<NFUNCTIONSH),							/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* node unexpand glyph */
static INTBIG gen_unexpand_v[] = {-50*K1,0,-40*K1,10*K1,
	-40*K1,10*K1,-25*K1,20*K1, -25*K1,20*K1,-10*K1,25*K1,
	-10*K1,25*K1,10*K1,25*K1, 10*K1,25*K1,25*K1,20*K1,
	25*K1,20*K1,40*K1,10*K1, 40*K1,10*K1,50*K1,0,
	50*K1,0,40*K1,-10*K1, 40*K1,-10*K1,25*K1,-20*K1,
	25*K1,-20*K1,10*K1,-25*K1, 10*K1,-25*K1,-10*K1,-25*K1,
	-10*K1,-25*K1,-25*K1,-20*K1, -25*K1,-20*K1,-40*K1,-10*K1,
	-40*K1,-10*K1,-50*K1,0, -40*K1,-10*K1,-40*K1,-20*K1,
	40*K1,-10*K1,40*K1,-20*K1, -20*K1,-21*K1,-20*K1,-31*K1,
	20*K1,-21*K1,20*K1,-31*K1, 0,-25*K1,0,-35*K1};
static TECH_POLYGON gen_unexpand_l[] = {			/* layers */
	{LGLYPH, 0, 38, VECTORS, ABSPOINTS, gen_unexpand_v}};
static TECH_NODES gen_unexpandv = {
	"Node-Unexpand",NUNEXPANDV,NONODEPROTO,			/* name */
	K1*100,K1*100,									/* size */
	1,gen_u_p, 1,gen_unexpand_l,					/* detail */
	(NPART<<NFUNCTIONSH),							/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* window up glyph */
static INTBIG gen_up_v[] = {0,-40*K1,0,40*K1, 0,40*K1,-25*K1,15*K1,
	0,40*K1,25*K1,15*K1};
static TECH_POLYGON gen_up_l[] = {					/* layers */
	{LGLYPH, 0, 6, VECTORS, ABSPOINTS, gen_up_v}};
static TECH_NODES gen_upv = {
	"Window-Up",NUPV,NONODEPROTO,					/* name */
	K1*100,K1*100,									/* size */
	1,gen_u_p, 1,gen_up_l,							/* detail */
	(NPART<<NFUNCTIONSH),							/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* window viewall glyph */
static INTBIG gen_viewall_v[] = {-20*K1,0,-45*K1,0, -45*K1,0,-35*K1,10*K1,
	-45*K1,0,-35*K1,-10*K1, 20*K1,0,45*K1,0, 45*K1,0,35*K1,10*K1,
	45*K1,0,35*K1,-10*K1, 0,-20*K1,0,-45*K1, 0,-45*K1,10*K1,-35*K1,
	0,-45*K1,-10*K1,-35*K1, 0,20*K1,0,45*K1, 0,45*K1,10*K1,35*K1,
	0,45*K1,-10*K1,35*K1};
static TECH_POLYGON gen_viewall_l[] = {				/* layers */
	{LGLYPH, 0, 24, VECTORS, ABSPOINTS, gen_viewall_v}};
static TECH_NODES gen_viewallv = {
	"Window-All-Displayed",NVIEWALLV,NONODEPROTO,	/* name */
	K1*100,K1*100,									/* size */
	1,gen_u_p, 1,gen_viewall_l,						/* detail */
	(NPART<<NFUNCTIONSH),							/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* window view glyph */
static INTBIG gen_view_v[] = {-20*K1,-20*K1,20*K1,-20*K1,
	20*K1,-20*K1,20*K1,20*K1, 20*K1,20*K1,-20*K1,20*K1,
	-20*K1,20*K1,-20*K1,-20*K1, -20*K1,0,-45*K1,0,
	-45*K1,0,-35*K1,10*K1, -45*K1,0,-35*K1,-10*K1, 20*K1,0,45*K1,0,
	45*K1,0,35*K1,10*K1, 45*K1,0,35*K1,-10*K1, 0,-20*K1,0,-45*K1,
	0,-45*K1,10*K1,-35*K1, 0,-45*K1,-10*K1,-35*K1, 0,20*K1,0,45*K1,
	0,45*K1,10*K1,35*K1, 0,45*K1,-10*K1,35*K1};
static TECH_POLYGON gen_view_l[] = {				/* layers */
	{LGLYPH, 0, 32, VECTORS, ABSPOINTS, gen_view_v}};
static TECH_NODES gen_viewv = {
	"Window-Highlight-Displayed",NVIEWV,NONODEPROTO,/* name */
	K1*100,K1*100,									/* size */
	1,gen_u_p, 1,gen_view_l,						/* detail */
	(NPART<<NFUNCTIONSH),							/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* window zoomout glyph */
static INTBIG gen_zoomout_v[] = {-45*K1,0,-20*K1,0, -20*K1,0,-30*K1,10*K1,
	-20*K1,0,-30*K1,-10*K1, 45*K1,0,20*K1,0, 20*K1,0,30*K1,10*K1,
	20*K1,0,30*K1,-10*K1, 0,-45*K1,0,-20*K1, 0,-20*K1,10*K1,-30*K1,
	0,-20*K1,-10*K1,-30*K1, 0,45*K1,0,20*K1, 0,20*K1,10*K1,30*K1,
	0,20*K1,-10*K1,30*K1};
static TECH_POLYGON gen_zoomout_l[] = {				/* layers */
	{LGLYPH, 0, 24, VECTORS, ABSPOINTS, gen_zoomout_v}};
static TECH_NODES gen_zoomoutv = {
	"Window-Out-Zoom",NZOOMOUTV,NONODEPROTO,		/* name */
	K1*100,K1*100,									/* size */
	1,gen_u_p, 1,gen_zoomout_l,						/* detail */
	(NPART<<NFUNCTIONSH),							/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* window zoomin glyph */
static INTBIG gen_zoomin_v[] = {-30*K1,0,-20*K1,10*K1, -30*K1,0,-20*K1,-10*K1,
	-30*K1,0,-5*K1,0, 30*K1,0,20*K1,10*K1, 30*K1,0,20*K1,-10*K1,
	30*K1,0,5*K1,0, 0,30*K1,10*K1,20*K1, 0,30*K1,-10*K1,20*K1,
	0,30*K1,0,5*K1, 0,-30*K1,10*K1,-20*K1, 0,-30*K1,-10*K1,-20*K1,
	0,-30*K1, 0,-5*K1};
static TECH_POLYGON gen_zoomin_l[] = {				/* layers */
	{LGLYPH, 0, 24, VECTORS, ABSPOINTS, gen_zoomin_v}};
static TECH_NODES gen_zoominv = {
	"Window-In-Zoom",NZOOMINV,NONODEPROTO,			/* name */
	K1*100,K1*100,									/* size */
	1,gen_u_p, 1,gen_zoomin_l,						/* detail */
	(NPART<<NFUNCTIONSH),							/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

TECH_NODES *gen_nodeprotos[NODEPROTOCOUNT+1] = {&gen_u, &gen_i,
	&gen_r, &gen_c, &gen_p, &gen_d, &gen_downv, &gen_erasev, &gen_expandv,
	&gen_gridv, &gen_leftv, &gen_outhierv, &gen_quitv, &gen_redrawv, &gen_rightv,
	&gen_rotatev, &gen_savev, &gen_systemv, &gen_undov, &gen_unexpandv, &gen_upv,
	&gen_viewallv, &gen_viewv, &gen_zoomoutv, &gen_zoominv, ((TECH_NODES *)-1)};

static INTBIG gen_node_widoff[NODEPROTOCOUNT*4] = {0,0,0,0, 0,0,0,0, 0,0,0,0,
	0,0,0,0, K2,K2,K2,K2, 0,0,0,0, 0,0,0,0, 0,0,0,0, 0,0,0,0, 0,0,0,0,
	0,0,0,0, 0,0,0,0, 0,0,0,0, 0,0,0,0, 0,0,0,0, 0,0,0,0,
	0,0,0,0, 0,0,0,0, 0,0,0,0, 0,0,0,0, 0,0,0,0,
	0,0,0,0, 0,0,0,0, 0,0,0,0, 0,0,0,0};

/******************** VARIABLE AGGREGATION ********************/

TECH_VARIABLES gen_variables[] =
{
	/* set general information about the technology */
	{"TECH_layer_names", (char *)gen_layer_names, 0.0,
		VSTRING|VDONTSAVE|VISARRAY|(MAXLAYERS<<VLENGTHSH)},
	{"TECH_layer_function", (char *)gen_layer_function, 0.0,
		VINTEGER|VDONTSAVE|VISARRAY|(MAXLAYERS<<VLENGTHSH)},
	{"TECH_node_width_offset", (char *)gen_node_widoff, 0.0,
		VFRACT|VDONTSAVE|VISARRAY|((NODEPROTOCOUNT*4)<<VLENGTHSH)},

	/* set information for the I/O aid */
	{"IO_cif_layer_names", (char *)gen_cif_layers, 0.0,
		VSTRING|VDONTSAVE|VISARRAY|(MAXLAYERS<<VLENGTHSH)},

	/* set information for the USER aid */
	{"USER_layer_letters", (char *)gen_layer_letters, 0.0,
		VSTRING|VDONTSAVE|VISARRAY|(MAXLAYERS<<VLENGTHSH)},
	{NULL, NULL, 0.0, 0}
};

/******************** ROUTINES ********************/

INTSML gen_initprocess(TECHNOLOGY *tech, INTSML pass)
{
	switch (pass)
	{
		case 0:
			/* initialize the technology variable */
			gen_tech = tech;
			break;

		case 1:
			/* create list of ALL arcs (now and when technologies change) */
			registertechnologycache(gen_makeunivlist);

			gen_univpinprim = getnodeproto("Generic:Universal-Pin");
			gen_invispinprim = getnodeproto("Generic:Invisible-Pin");
			gen_unroutedpinprim = getnodeproto("Generic:Unrouted-Pin");
			gen_facetcenterprim = getnodeproto("Generic:Facet-Center");
			gen_portprim = getnodeproto("Generic:Port");
			gen_drcprim = getnodeproto("Generic:DRC-Node");

			gen_universalarc = getarcproto("Generic:Universal");
			gen_invisiblearc = getarcproto("Generic:Invisible");
			gen_unroutedarc = getarcproto("Generic:Unrouted");
			break;

		case 2:
			/* set colors properly */
			gen_gl_lay.col = el_colmengly;
			break;
	}
	return(0);
}

/*
 * routine to update the connecitivity list for universal pins so that
 * they can connect to ALL arcs.  This is called at initialization and again
 * whenever the number of technologies changes
 */
void gen_makeunivlist(void)
{
	REGISTER INTSML tot;
	REGISTER ARCPROTO **upconn, *ap;
	REGISTER TECHNOLOGY *t;

	/* count the number of arcs in all technologies */
	tot = 0;
	for(t = el_technologies; t != NOTECHNOLOGY; t = t->nexttechnology)
	{
		for(ap = t->firstarcproto; ap != NOARCPROTO; ap = ap->nextarcproto) tot++;
	}

	/* make an array for each arc */
	upconn = (ARCPROTO **)emalloc(((tot+2) * (sizeof (ARCPROTO *))), gen_tech->cluster);
	if (upconn == 0) return;

	/* fill the array */
	tot = 0;
	upconn[tot++] = (ARCPROTO *)0;
	for(t = el_technologies; t != NOTECHNOLOGY; t = t->nexttechnology)
	{
		for(ap = t->firstarcproto; ap != NOARCPROTO; ap = ap->nextarcproto) upconn[tot++] = ap;
	}
	upconn[tot] = NOARCPROTO;

	/* store the array in this technology */
	gen_u_p[0].portarcs = (INTBIG *)upconn;
}

INTSML gen_nodepolys(NODEINST *ni)
{
	REGISTER INTSML index, count, start, end;
	static INTBIG portangle = 0, portrange = 0;
	REGISTER VARIABLE *var, *var2;

	index = ni->proto->index;
	count = gen_nodeprotos[index-1]->layercount;
	if (index == NUNIV || index == NINVIS || index == NUNROUTED)
	{
		if (tech_pinusecount(ni, NOARCPROTO) != 0) count = 0;
		if (index == NINVIS)
		{
			gen_in_lay.bits = LAYERO;
			gen_in_lay.col = GRAY;
			gen_int_lay.bits = LAYERO;
			gen_int_lay.col = BLACK;
			var = getvalkey((INTBIG)ni, VNODEINST, VINTEGER, art_colorkey);
			if (var != NOVARIABLE)
			{
				switch (var->addr)
				{
					case LAYERT1: gen_in_lay.col = COLORT1;  gen_in_lay.bits = LAYERT1;  break;
					case LAYERT2: gen_in_lay.col = COLORT2;  gen_in_lay.bits = LAYERT2;  break;
					case LAYERT3: gen_in_lay.col = COLORT3;  gen_in_lay.bits = LAYERT3;  break;
					case LAYERT4: gen_in_lay.col = COLORT4;  gen_in_lay.bits = LAYERT4;  break;
					case LAYERT5: gen_in_lay.col = COLORT5;  gen_in_lay.bits = LAYERT5;  break;
					default:
						if ((var->addr&(LAYERG|LAYERH|LAYEROE)) == LAYEROE)
							gen_in_lay.bits = LAYERO; else gen_in_lay.bits = LAYERA;
						gen_in_lay.col = var->addr;
						break;
				}
				gen_int_lay.bits = gen_in_lay.bits;
				gen_int_lay.col = gen_in_lay.col;
			}
		}
	} else if (index == NPORT)
	{
		if (portangle == 0) portangle = makekey("EDTEC_portangle");
		if (portrange == 0) portrange = makekey("EDTEC_portrange");

		/* port node becomes a cross when it is 1x1 */
		if (ni->highx-ni->lowx == gen_tech->deflambda*2 &&
			ni->highy-ni->lowy == gen_tech->deflambda*2)
				gen_p_l[0].style = CROSS; else gen_p_l[0].style = CLOSED;

		var = getvalkey((INTBIG)ni, VNODEINST, VINTEGER, portangle);
		var2 = getvalkey((INTBIG)ni, VNODEINST, VINTEGER, portrange);
		if (var == NOVARIABLE || var2 == NOVARIABLE) count--; else
		{
			start = (var->addr - var2->addr) * 10;
			end = (var->addr + var2->addr) * 10;
			while (start < 0) start += 3600;
			while (start > 3600) start -= 3600;
			while (end < 0) end += 3600;
			while (end > 3600) end -= 3600;
			gen_portabox[1] = mult(cosine(start), K2);
			gen_portabox[3] = mult(sine(start), K2);
			gen_portabox[9] = mult(cosine(end), K2);
			gen_portabox[11] = mult(sine(end), K2);
		}
	}

	/* add in displayable variables */
	tech_realpolys = count;
	count += tech_displayablenvars(ni);
	return(count);
}

void gen_shapenodepoly(NODEINST *ni, INTSML box, POLYGON *poly)
{
	REGISTER TECH_POLYGON *lay;
	REGISTER INTSML index, count, i, j;
	REGISTER INTBIG x, y, xoff, yoff, cross;
	REGISTER VARIABLE *var;

	/* handle displayable variables */
	index = ni->proto->index;
	if (box >= tech_realpolys)
	{
		(void)tech_filldisplayablenvar(ni, poly);
		if (index == NINVIS) poly->desc = &gen_int_lay;
		return;
	}

	lay = &gen_nodeprotos[index - 1]->layerlist[box];

	/* universal pins may have trace information */
	if (index == NUNIV)
	{
		var = getvalkey((INTBIG)ni, VNODEINST, VINTEGER|VISARRAY, el_trace);
		if (var != NOVARIABLE)
		{
			/* make sure polygon can hold this description */
			count = getlength(var) / 2;
			j = count*4 + (count-1)*2;
			if (poly->limit < j) (void)extendpolygon(poly, j);

			/* fill the polygon */
			xoff = (ni->highx + ni->lowx) / 2;
			yoff = (ni->highy + ni->lowy) / 2;
			cross = ni->proto->tech->deflambda / 4;
			j = 0;
			for(i=0; i<count; i++)
			{
				x = ((INTBIG *)var->addr)[i*2];
				y = ((INTBIG *)var->addr)[i*2+1];
				poly->xv[j] = x-cross+xoff;
				poly->yv[j] = y-cross+yoff;   j++;
				poly->xv[j] = x+cross+xoff;
				poly->yv[j] = y+cross+yoff;   j++;
				poly->xv[j] = x-cross+xoff;
				poly->yv[j] = y+cross+yoff;   j++;
				poly->xv[j] = x+cross+xoff;
				poly->yv[j] = y-cross+yoff;   j++;
			}
			for(i=1; i<count; i++)
			{
				poly->xv[j] = ((INTBIG *)var->addr)[(i-1)*2]+xoff;
				poly->yv[j] = ((INTBIG *)var->addr)[(i-1)*2+1]+yoff;   j++;
				poly->xv[j] = ((INTBIG *)var->addr)[i*2]+xoff;
				poly->yv[j] = ((INTBIG *)var->addr)[i*2+1]+yoff;   j++;
			}

			/* add in peripheral information */
			poly->layer = lay->layernum;
			poly->style = VECTORS;
			poly->count = j;
			poly->desc = gen_layers[poly->layer];
			return;
		}
	}

	/* nontrace pins draw the normal way */
	tech_fillpoly(poly, lay, ni, gen_tech->deflambda, FILLED);
	poly->desc = gen_layers[poly->layer];
}

void gen_shapeportpoly(NODEINST *ni, PORTPROTO *pp, POLYGON *poly, XARRAY trans,
	INTSML purpose)
{
	REGISTER INTSML index;

	index = ni->proto->index;
	tech_fillportpoly(ni, pp, poly, trans, gen_nodeprotos[index-1], OPENED);
}

INTSML gen_arcpolys(ARCINST *ai)
{
	REGISTER INTSML i;

	i = gen_arcprotos[ai->proto->index]->laycount;

	/* add in displayable variables */
	tech_realpolys = i;
	i += tech_displayableavars(ai);
	return(i);
}

void gen_shapearcpoly(ARCINST *ai, INTSML box, POLYGON *poly)
{
	REGISTER INTSML index;
	REGISTER TECH_ARCLAY *thista;

	/* handle displayable variables */
	if (box >= tech_realpolys)
	{
		(void)tech_filldisplayableavar(ai, poly);
		return;
	}

	index = ai->proto->index;
	thista = &gen_arcprotos[index]->list[box];
	makearcpoly(ai->length, ai->width-thista->off*gen_tech->deflambda/WHOLE, ai, poly, thista->style);
	poly->layer = thista->lay;
	poly->desc = gen_layers[poly->layer];
}
