/*
 * Electric(tm) VLSI Design System
 *
 * File: tecmocmossub.c
 * MOSIS CMOS Submicron technology description
 * Written by: Steven M. Rubin, Electric Editor Incorporated
 * The MOSIS 4-metal, 2-poly submicron rules,
 * interpreted by Richard "Bic" Schediwy
 *
 * Copyright (c) 1998 Electric Editor Incorporated.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Electric Editor Incorporated
 * 23470 Sunset Drive, Suite 108
 * Los Gatos, California 95033
 * support@electriceditor.com
 */

/*
 * Assumptions made from the MOSIS submicron design rules
 * (MOSIS rule is in square brackets []):
 *
 * Arc widths (minimum sizes):
 *   metal1: 3 [7.1]
 *   metal2: 3 [9.1]
 *   metal3: 5 (if 3-metal process) [15.1]
 *   metal3: 3 (if 4-metal process) [15.1]
 *   metal4: 6 [22.1]
 *   poly1:  2 [3.1]
 *   poly2:  3 [11.1]
 *   p/n active (active&select&well):
 *     active: 3 [2.1], select extends by 2 [4.2], well extends by 6 [2.3]
 *   active: 3 [2.1]
 *
 * Pin/Node sizes (minimum sizes):
 *   metal1: 3 [7.1]
 *   metal2: 3 [9.1]
 *   metal3: 5 (if 3-metal process) [15.1]
 *   metal3: 3 (if 4-metal process) [15.1]
 *   metal4: 6 [22.1]
 *   poly1:  2 [3.1]
 *   poly2:  3 [11.1]
 *   active: 3 [2.1]
 *   select: 2 [4.4]
 *   well:  12 [1.1]
 *
 * Special nodes:
 *   p/n active-to-metal1 contact:
 *     cuts 2x2 [6.1], separated 3 [6.3]
 *     metal1 extends around cut by 1 (4x4) [7.3]
 *     active extends around cut by 1.5 (5x5) [6.2]
 *     select extends around active by 1 (7x7) [4.3]
 *     well extends around active by 6 (17x17) [2.3]
 *   poly1-to-metal1 contact:
 *     cuts 2x2 [5.1], separated 3 [5.3]
 *     metal1 extends around cut by 1 (4x4) [7.3]
 *     poly1 extends around cut by 1.5 (5x5) [5.2]
 *   poly2-to-metal1 contact:
 *     cuts 2x2 [5.1], separated 3 [5.3]
 *     metal1 extends around cut by 1 (4x4) [7.3]
 *     poly2 size: 3 (3x3) [11.1]
 *   poly1-to-poly2 (capacitor) contact:
 *     cuts 2x2 [5.1], separated 3 [5.3]
 *     poly2 size: 3 (3x3) [11.1]
 *     poly1 extends around poly2 by 2 (7x7) [11.3]
 *   Transistors:
 *     active is 3 wide [2.1] and sticks out by 3 (3x8) [3.4]
 *     poly1 is 2 wide [3.1] and sticks out by 2 (7x2) [3.3]
 *     transistor area is 3x2
 *     select surrounds active by 2 (7x12) [4.2]
 *     well surrounds active by 6 (15x20) [2.3]
 *   Via1:
 *     cuts 2x2 [8.1], separated 3 [8.2]
 *     metal1 extends around cut by 1 (4x4) [8.3]
 *     metal2 extends around cut by 1 (4x4) [9.3]
 *   Via2:
 *     cuts 2x2 [14.1], separated 3 [14.2]
 *     metal2 extends around cut by 1 (4x4) [14.3]
 *     metal3 extends around cut by: 2 (6x6) (if 3-metal process) [15.3]
 *     metal3 extends around cut by: 1 (4x4) (if 4-metal process) [15.3]
 *   Via3:
 *     cuts 2x2 [21.1], separated 4 [21.2]
 *     metal3 extends around cut by: 1 (4x4) [21.3]
 *     metal4 extends around cut by: 2 (6x6) [22.3]
 *   Substrate/well contact:
 *     well extends around active by 6 [2.3]
 *
 * DRC:
 *   metal1-to-metal1: 3 [7.2]
 *   metal2-to-metal2: 4 [9.2]
 *   metal3-to-metal3: 3 [15.2]
 *   metal4-to-metal4: 6 [22.2]
 *   poly1-to-poly1: 3 [3.2]
 *   poly1-to-active: 1 [3.5]
 *   poly2-to-poly2: 3 [11.2]
 *   poly2-to-active: 1 [3.5]
 *   poly2-to-polyCut: 3 [11.5]
 *   active-to-active: 3 [2.2]
 *   select-to-trans: 3 [4.1]
 *   polyCut/actCut-to-polyCut/actCut: 3 [5.3]
 *   polyCut/actCut-to-via1: 2 [8.4]
 *   polyCut-to-transistor: 2 [5.4]
 *   actCut-to-transistor: 2 [6.4]
 *   via1-to-via1: 3 [8.2]
 *   via1-to-via2: 2 [14.4]
 *   via2-to-via2: 2 [14.2]
 *   via3-to-via3: 4 [21.2]
 */

#include "config.h"
#if TECMOCMOSSUB

#include "global.h"
#include "egraphics.h"
#include "tech.h"
#include "tecmocmossub.h"
#include "efunction.h"

/*
 * Can switch from P-well (the default) to N-well
 * Can switch from 4-metal rules (the default) to 3-metal rules
 */

/* the options table */
static KEYWORD mocmossubopt[] =
{
	{"p-well",                     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"n-well",                     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"3-metal-rules",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"4-metal-rules",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP mocmossub_parse = {mocmossubopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, 0, " \t", "MOSIS CMOS Submicron option", "show current options"};

#define	NWELL      1		/* set for n-well, clear for p-well */
#define	THREEMETAL 2		/* set for 3-metal rules, clear for 4-metal rules */

static INTSML mocmossub_state = 0;		/* p-well 4-metal */
static TECHNOLOGY *mocmossub_tech;

void mocmossub_setdescript(void);

/******************** LAYERS ********************/

#define	MAXLAYERS   34		/* total layers below         */
#define	LMETAL1      0		/* metal layer 1              */
#define	LMETAL2      1		/* metal layer 2              */
#define	LMETAL3      2		/* metal layer 3              */
#define	LMETAL4      3		/* metal layer 4              */
#define	LPOLY1       4		/* polysilicon                */
#define	LPOLY2       5		/* polysilicon 2 (electrode)  */
#define	LSACT        6		/* P (or N) active            */
#define	LDACT        7		/* N (or P) active            */
#define	LSELECTP     8		/* P-type select              */
#define	LSELECTN     9		/* N-type select              */
#define	LWELLP      10		/* P-type well                */
#define	LWELLN      11		/* N-type well                */
#define	LPOLYCUT    12		/* poly contact cut           */
#define	LACTCUT     13		/* active contact cut         */
#define	LVIA1       14		/* metal1-to-metal2 via       */
#define	LVIA2       15		/* metal2-to-metal3 via       */
#define	LVIA3       16		/* metal3-to-metal4 via       */
#define	LPASS       17		/* passivation (overglass)    */
#define	LTRANS      18		/* transistor                 */
#define	LPOLYCAP    19		/* polysilicon capacitor      */
#define	LSACTWELL   20		/* P active in well           */
#define	LMET1P      21		/* pseudo metal 1             */
#define	LMET2P      22		/* pseudo metal 2             */
#define	LMET3P      23		/* pseudo metal 3             */
#define	LMET4P      24		/* pseudo metal 4             */
#define	LPOLY1P     25		/* pseudo polysilicon 1       */
#define	LPOLY2P     26		/* pseudo polysilicon 2       */
#define	LSACTP      27		/* pseudo P (or N) active     */
#define	LDACTP      28		/* pseudo N (or P) active     */
#define	LSELECTPP   29		/* pseudo P-type select       */
#define	LSELECTNP   30		/* pseudo N-type select       */
#define	LWELLPP     31		/* pseudo P-type well         */
#define	LWELLNP     32		/* pseudo N-type well         */
#define	LFRAME      33		/* pad frame boundary         */

static GRAPHICS mocmossub_m1_lay = {LAYERT1,COLORT1, {SOLIDC,PATTERNED,SOLIDC,SOLIDC},
/* metal-1 layer */		{0x2222, /*   X   X   X   X  */
						0x0000,  /*                  */
						0x8888,  /* X   X   X   X    */
						0x0000,  /*                  */
						0x2222,  /*   X   X   X   X  */
						0x0000,  /*                  */
						0x8888,  /* X   X   X   X    */
						0x0000}, /*                  */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_m2_lay = {LAYERT4,COLORT4, {SOLIDC,PATTERNED,SOLIDC,SOLIDC},
/* metal-2 layer */		{0x1010, /*    X       X     */
						0x2020,  /*   X       X      */
						0x4040,  /*  X       X       */
						0x8080,  /* X       X        */
						0x0101,  /*        X       X */
						0x0202,  /*       X       X  */
						0x0404,  /*      X       X   */
						0x0808}, /*     X       X    */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_m3_lay = {LAYERT5,COLORT5, {SOLIDC,PATTERNED,SOLIDC,SOLIDC},
/* metal-3 layer */		{0x2222, /*   X   X   X   X  */
						0x0000,  /*                  */
						0x8888,  /* X   X   X   X    */
						0x0000,  /*                  */
						0x2222,  /*   X   X   X   X  */
						0x0000,  /*                  */
						0x8888,  /* X   X   X   X    */
						0x0000}, /*                  */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_m4_lay = {LAYERO,LBLUE, {PATTERNED,PATTERNED,PATTERNED,SOLIDC},
/* metal-4 layer */		{0x0808, /*     X       X    */
						0x1818,  /*    XX      XX    */
						0x2828,  /*   X X     X X    */
						0x4848,  /*  X  X    X  X    */
						0xFCFC,  /* XXXXXX  XXXXXXX  */
						0x0808,  /*     X       X    */
						0x0808,  /*     X       X    */
						0x0000}, /*                  */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_p1_lay = {LAYERT2,COLORT2, {SOLIDC,PATTERNED,SOLIDC,SOLIDC},
/* poly layer */		{0x1111, /*    X   X   X   X */
						0xFFFF,  /* XXXXXXXXXXXXXXXX */
						0x1111,  /*    X   X   X   X */
						0x5555,  /*  X X X X X X X X */
						0x1111,  /*    X   X   X   X */
						0xFFFF,  /* XXXXXXXXXXXXXXXX */
						0x1111,  /*    X   X   X   X */
						0x5555}, /*  X X X X X X X X */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_p2_lay = {LAYERO,ORANGE, {PATTERNED,PATTERNED,PATTERNED,SOLIDC},
/* poly2 layer */		{0xAFAF, /* X X XXXXX X XXXX */
						0x8888,  /* X   X   X   X    */
						0xFAFA,  /* XXXXX X XXXXX X  */
						0x8888,  /* X   X   X   X    */
						0xAFAF,  /* X X XXXXX X XXXX */
						0x8888,  /* X   X   X   X    */
						0xFAFA,  /* XXXXX X XXXXX X  */
						0x8888}, /* X   X   X   X    */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_sa_lay = {LAYERT3,COLORT3, {SOLIDC,PATTERNED,SOLIDC,SOLIDC},
/* P active layer */	{0x0000, /*                  */
						0x0303,  /*       XX      XX */
						0x4848,  /*  X  X    X  X    */
						0x0303,  /*       XX      XX */
						0x0000,  /*                  */
						0x3030,  /*   XX      XX     */
						0x8484,  /* X    X  X    X   */
						0x3030}, /*   XX      XX     */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_da_lay = {LAYERT3,COLORT3, {SOLIDC,PATTERNED,SOLIDC,SOLIDC},
/* N active layer */	{0x0000, /*                  */
						0x0303,  /*       XX      XX */
						0x4848,  /*  X  X    X  X    */
						0x0303,  /*       XX      XX */
						0x0000,  /*                  */
						0x3030,  /*   XX      XX     */
						0x8484,  /* X    X  X    X   */
						0x3030}, /*   XX      XX     */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_ssp_lay = {LAYERO,YELLOW, {PATTERNED,PATTERNED,PATTERNED,SOLIDC},
/* P Select layer */	{0x1010, /*    X       X     */
						0x2020,  /*   X       X      */
						0x4040,  /*  X       X       */
						0x8080,  /* X       X        */
						0x0101,  /*        X       X */
						0x0202,  /*       X       X  */
						0x0404,  /*      X       X   */
						0x0808}, /*     X       X    */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_ssn_lay = {LAYERN,YELLOW, {PATTERNED,PATTERNED,PATTERNED,SOLIDC},
/* N Select layer */	{0x1010, /*    X       X     */
						0x2020,  /*   X       X      */
						0x4040,  /*  X       X       */
						0x8080,  /* X       X        */
						0x0101,  /*        X       X */
						0x0202,  /*       X       X  */
						0x0404,  /*      X       X   */
						0x0808}, /*     X       X    */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_wp_lay = {LAYERO,BROWN, {PATTERNED,PATTERNED,PATTERNED,SOLIDC},
/* P Well layer */		{0x0202, /*       X       X  */
						0x0101,  /*        X       X */
						0x8080,  /* X       X        */
						0x4040,  /*  X       X       */
						0x2020,  /*   X       X      */
						0x1010,  /*    X       X     */
						0x0808,  /*     X       X    */
						0x0404}, /*      X       X   */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_wn_lay = {LAYERN,BROWN, {PATTERNED,PATTERNED,PATTERNED,SOLIDC},
/* N Well implant */	{0x0202, /*       X       X  */
						0x0101,  /*        X       X */
						0x8080,  /* X       X        */
						0x4040,  /*  X       X       */
						0x2020,  /*   X       X      */
						0x1010,  /*    X       X     */
						0x0808,  /*     X       X    */
						0x0404}, /*      X       X   */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_pc_lay = {LAYERO,BLACK, {SOLIDC,SOLIDC,SOLIDC,SOLIDC},
/* poly cut layer */	{0,0,0,0,0,0,0,0}, NOVARIABLE, 0};
static GRAPHICS mocmossub_ac_lay = {LAYERO,BLACK, {SOLIDC,SOLIDC,SOLIDC,SOLIDC},
/* active cut layer */	{0,0,0,0,0,0,0,0}, NOVARIABLE, 0};
static GRAPHICS mocmossub_v1_lay = {LAYERO,LGRAY, {SOLIDC,SOLIDC,SOLIDC,SOLIDC},
/* via1 layer */	{0,0,0,0,0,0,0,0}, NOVARIABLE, 0};
static GRAPHICS mocmossub_v2_lay = {LAYERO,LGRAY, {SOLIDC,SOLIDC,SOLIDC,SOLIDC},
/* via2 layer */	{0,0,0,0,0,0,0,0}, NOVARIABLE, 0};
static GRAPHICS mocmossub_v3_lay = {LAYERO,LGRAY, {SOLIDC,SOLIDC,SOLIDC,SOLIDC},
/* via3 layer */	{0,0,0,0,0,0,0,0}, NOVARIABLE, 0};
static GRAPHICS mocmossub_ovs_lay = {LAYERO,DGRAY, {PATTERNED,PATTERNED,PATTERNED,
	SOLIDC},
/* passivation layer */	{0x1C1C, /*    XXX     XXX   */
						0x3E3E,  /*   XXXXX   XXXXX  */
						0x3636,  /*   XX XX   XX XX  */
						0x3E3E,  /*   XXXXX   XXXXX  */
						0x1C1C,  /*    XXX     XXX   */
						0x0000,  /*                  */
						0x0000,  /*                  */
						0x0000}, /*                  */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_tr_lay = {LAYERN,ALLOFF, {SOLIDC,SOLIDC,SOLIDC,SOLIDC},
/* transistor layer */	{0,0,0,0,0,0,0,0}, NOVARIABLE, 0};
static GRAPHICS mocmossub_cp_lay = {LAYERO,BLACK, {SOLIDC,SOLIDC,SOLIDC,SOLIDC},
/* poly cap layer */	{0,0,0,0,0,0,0,0}, NOVARIABLE, 0};
static GRAPHICS mocmossub_saw_lay = {LAYERT3,COLORT3, {SOLIDC,PATTERNED,SOLIDC,SOLIDC},
/* P act well layer */	{0x0000, /*                  */
						0x0303,  /*       XX      XX */
						0x4848,  /*  X  X    X  X    */
						0x0303,  /*       XX      XX */
						0x0000,  /*                  */
						0x3030,  /*   XX      XX     */
						0x8484,  /* X    X  X    X   */
						0x3030}, /*   XX      XX     */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_pm1_lay ={LAYERT1,COLORT1, {SOLIDC,PATTERNED,SOLIDC,SOLIDC},
/* pseudo metal 1 */	{0x2222, /*   X   X   X   X  */
						0x0000,  /*                  */
						0x8888,  /* X   X   X   X    */
						0x0000,  /*                  */
						0x2222,  /*   X   X   X   X  */
						0x0000,  /*                  */
						0x8888,  /* X   X   X   X    */
						0x0000}, /*                  */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_pm2_lay = {LAYERT4,COLORT4, {SOLIDC,PATTERNED,SOLIDC,SOLIDC},
/* pseudo metal-2 */	{0x1010, /*    X       X     */
						0x2020,  /*   X       X      */
						0x4040,  /*  X       X       */
						0x8080,  /* X       X        */
						0x0101,  /*        X       X */
						0x0202,  /*       X       X  */
						0x0404,  /*      X       X   */
						0x0808}, /*     X       X    */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_pm3_lay = {LAYERT5,COLORT5, {SOLIDC,PATTERNED,SOLIDC,SOLIDC},
/* pseudo metal-3 */	{0x1010, /*    X       X     */
						0x2020,  /*   X       X      */
						0x4040,  /*  X       X       */
						0x8080,  /* X       X        */
						0x0101,  /*        X       X */
						0x0202,  /*       X       X  */
						0x0404,  /*      X       X   */
						0x0808}, /*     X       X    */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_pm4_lay = {LAYERO,LBLUE, {PATTERNED,PATTERNED,PATTERNED,SOLIDC},
/* pseudo metal-4 */	{0x0808, /*     X       X    */
						0x1818,  /*    XX      XX    */
						0x2828,  /*   X X     X X    */
						0x4848,  /*  X  X    X  X    */
						0xFCFC,  /* XXXXXX  XXXXXXX  */
						0x0808,  /*     X       X    */
						0x0808,  /*     X       X    */
						0x0000}, /*                  */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_pp1_lay = {LAYERT2,COLORT2, {SOLIDC,PATTERNED,SOLIDC,SOLIDC},
/* pseudo poly layer */	{0x1111, /*    X   X   X   X */
						0xFFFF,  /* XXXXXXXXXXXXXXXX */
						0x1111,  /*    X   X   X   X */
						0x5555,  /*  X X X X X X X X */
						0x1111,  /*    X   X   X   X */
						0xFFFF,  /* XXXXXXXXXXXXXXXX */
						0x1111,  /*    X   X   X   X */
						0x5555}, /*  X X X X X X X X */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_pp2_lay = {LAYERO,ORANGE, {PATTERNED,PATTERNED,PATTERNED,SOLIDC},
/* pseudo poly2 layer */{0xAFAF, /* X X XXXXX X XXXX */
						0x8888,  /* X   X   X   X    */
						0xFAFA,  /* XXXXX X XXXXX X  */
						0x8888,  /* X   X   X   X    */
						0xAFAF,  /* X X XXXXX X XXXX */
						0x8888,  /* X   X   X   X    */
						0xFAFA,  /* XXXXX X XXXXX X  */
						0x8888}, /* X   X   X   X    */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_psa_lay = {LAYERT3,COLORT3, {SOLIDC,PATTERNED,SOLIDC,SOLIDC},
/* pseudo P active */	{0x0000, /*                  */
						0x0303,  /*       XX      XX */
						0x4848,  /*  X  X    X  X    */
						0x0303,  /*       XX      XX */
						0x0000,  /*                  */
						0x3030,  /*   XX      XX     */
						0x8484,  /* X    X  X    X   */
						0x3030}, /*   XX      XX     */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_pda_lay = {LAYERT3,COLORT3, {SOLIDC,PATTERNED,SOLIDC,SOLIDC},
/* pseudo N active */	{0x0000, /*                  */
						0x0303,  /*       XX      XX */
						0x4848,  /*  X  X    X  X    */
						0x0303,  /*       XX      XX */
						0x0000,  /*                  */
						0x3030,  /*   XX      XX     */
						0x8484,  /* X    X  X    X   */
						0x3030}, /*   XX      XX     */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_pssp_lay = {LAYERO,YELLOW,{PATTERNED,PATTERNED,PATTERNED,SOLIDC},
/* pseudo P Select */	{0x1010, /*    X       X     */
						0x2020,  /*   X       X      */
						0x4040,  /*  X       X       */
						0x8080,  /* X       X        */
						0x0101,  /*        X       X */
						0x0202,  /*       X       X  */
						0x0404,  /*      X       X   */
						0x0808}, /*     X       X    */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_pssn_lay = {LAYERN,YELLOW,{PATTERNED,PATTERNED,PATTERNED,SOLIDC},
/* pseudo N Select */	{0x1010, /*    X       X     */
						0x2020,  /*   X       X      */
						0x4040,  /*  X       X       */
						0x8080,  /* X       X        */
						0x0101,  /*        X       X */
						0x0202,  /*       X       X  */
						0x0404,  /*      X       X   */
						0x0808}, /*     X       X    */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_pwp_lay = {LAYERO,BROWN, {PATTERNED,PATTERNED,PATTERNED,SOLIDC},
/* pseudo P Well */		{0x0202, /*       X       X  */
						0x0101,  /*        X       X */
						0x8080,  /* X       X        */
						0x4040,  /*  X       X       */
						0x2020,  /*   X       X      */
						0x1010,  /*    X       X     */
						0x0808,  /*     X       X    */
						0x0404}, /*      X       X   */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_pwn_lay = {LAYERN,BROWN, {PATTERNED,PATTERNED,PATTERNED,SOLIDC},
/* pseudo N Well */		{0x0202, /*       X       X  */
						0x0101,  /*        X       X */
						0x8080,  /* X       X        */
						0x4040,  /*  X       X       */
						0x2020,  /*   X       X      */
						0x1010,  /*    X       X     */
						0x0808,  /*     X       X    */
						0x0404}, /*      X       X   */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_pf_lay = {LAYERO, RED, {SOLIDC,PATTERNED,SOLIDC,SOLIDC},
/* pad frame */		{0,0,0,0,0,0,0,0}, NOVARIABLE, 0};

/* these tables must be updated together */
GRAPHICS *mocmossub_layers[MAXLAYERS+1] = {
	&mocmossub_m1_lay, &mocmossub_m2_lay,						/* metal 1/2 */
	&mocmossub_m3_lay, &mocmossub_m4_lay,						/* metal 3/4 */
	&mocmossub_p1_lay, &mocmossub_p2_lay,						/* poly 1/2 */
	&mocmossub_sa_lay, &mocmossub_da_lay,						/* P/N active */
	&mocmossub_ssp_lay, &mocmossub_ssn_lay,						/* P/N select */
	&mocmossub_wp_lay, &mocmossub_wn_lay,						/* P/N well */
	&mocmossub_pc_lay, &mocmossub_ac_lay,						/* poly/act cut */
	&mocmossub_v1_lay, &mocmossub_v2_lay, &mocmossub_v3_lay,	/* via1/2/3 */
	&mocmossub_ovs_lay,											/* overglass */
	&mocmossub_tr_lay,											/* transistor */
	&mocmossub_cp_lay,											/* poly cap */
	&mocmossub_saw_lay,											/* P active well */
	&mocmossub_pm1_lay, &mocmossub_pm2_lay,						/* pseudo metal 1/2 */
	&mocmossub_pm3_lay, &mocmossub_pm4_lay,						/* pseudo metal 3/4 */
	&mocmossub_pp1_lay, &mocmossub_pp2_lay,						/* pseudo poly 1/2 */
	&mocmossub_psa_lay, &mocmossub_pda_lay,						/* pseudo P/N active */
	&mocmossub_pssp_lay, &mocmossub_pssn_lay,					/* pseudo P/N select */
	&mocmossub_pwp_lay, &mocmossub_pwn_lay,						/* pseudo P/N well */
	&mocmossub_pf_lay, NOGRAPHICS};								/* pad frame */
static char *mocmossub_layer_names[MAXLAYERS] = {
	"Metal-1", "Metal-2", "Metal-3", "Metal-4",					/* metal 1/2/3/4 */
	"Polysilicon-1", "Polysilicon-2",							/* poly 1/2 */
	"P-Active", "N-Active",										/* P/N active */
	"P-Select", "N-Select",										/* P/N select */
	"P-Well", "N-Well",											/* P/N well */
	"Poly-Cut", "Active-Cut",									/* poly/act cut */
	"Via1", "Via2", "Via3",										/* via1/2/3 */
	"Passivation",												/* overglass */
	"Transistor",												/* transistor */
	"Poly-Cap",													/* poly cap */
	"P-Active-Well",											/* P active well */
	"Pseudo-Metal-1", "Pseudo-Metal-2",							/* pseudo metal 1/2 */
	"Pseudo-Metal-3", "Pseudo-Metal-4",							/* pseudo metal 3/4 */
	"Pseudo-Polysilicon", "Pseudo-Electrode",					/* pseudo poly 1/2 */
	"Pseudo-P-Active", "Pseudo-N-Active",						/* pseudo P/N active */
	"Pseudo-P-Select", "Pseudo-N-Select",						/* pseudo P/N select */
	"Pseudo-P-Well", "Pseudo-N-Well",							/* pseudo P/N well */
	"Pad-Frame"};												/* pad frame */
static INTBIG mocmossub_layer_function[MAXLAYERS] = {
	LFMETAL1|LFTRANS1, LFMETAL2|LFTRANS4,						/* metal 1/2 */
	LFMETAL3|LFTRANS5, LFMETAL4,								/* metal 3/4 */
	LFPOLY1|LFTRANS2, LFPOLY2,									/* poly 1/2 */
	LFDIFF|LFPTYPE|LFTRANS3, LFDIFF|LFNTYPE|LFTRANS3,			/* P/N active */
	LFIMPLANT|LFPTYPE, LFIMPLANT|LFNTYPE,						/* P/N select */
	LFWELL|LFPTYPE, LFWELL|LFNTYPE,								/* P/N well */
	LFCONTACT1, LFCONTACT1,										/* poly/act cut */
	LFCONTACT2, LFCONTACT3, LFCONTACT4,							/* via1/2/3 */
	LFOVERGLASS,												/* overglass */
	LFTRANSISTOR|LFPSEUDO,										/* transistor */
	LFCAP,														/* poly cap */
	LFDIFF|LFPTYPE|LFTRANS3,									/* P active well */
	LFMETAL1|LFPSEUDO|LFTRANS1,LFMETAL2|LFPSEUDO|LFTRANS4,		/* pseudo metal 1/2 */
	LFMETAL3|LFPSEUDO|LFTRANS5, LFMETAL4|LFPSEUDO,				/* pseudo metal 3/4 */
	LFPOLY1|LFPSEUDO|LFTRANS2, LFPOLY2|LFPSEUDO,				/* pseudo poly 1/2 */
	LFDIFF|LFPTYPE|LFPSEUDO|LFTRANS3,							/* pseudo P/N active */
		LFDIFF|LFNTYPE|LFPSEUDO|LFTRANS3,
	LFIMPLANT|LFPTYPE|LFPSEUDO, LFIMPLANT|LFNTYPE|LFPSEUDO,		/* pseudo P/N select */
	LFWELL|LFPTYPE|LFPSEUDO, LFWELL|LFNTYPE|LFPSEUDO,			/* pseudo P/N well */
	LFART};														/* pad frame */
static char *mocmossub_cif_layers[MAXLAYERS] = {
	"CMF", "CMS", "CMT", "CMQ",									/* metal 1/2/3/4 */
	"CPG", "CEL",												/* poly 1/2 */
	"CAA", "CAA",												/* P/N active */
	"CSP", "CSN",												/* P/N select */
	"CWP", "CWN",												/* P/N well */
	"CCG", "CCG",												/* poly/act cut */
	"CVA", "CVS", "CVT",										/* via1/2/3 */
	"COG",														/* overglass */
	"",															/* transistor */
	"CPC",														/* poly cap */
	"CAA",														/* P active well */
	"", "", "", "",												/* pseudo metal 1/2/3/4 */
	"", "",														/* pseudo poly 1/2 */
	"", "",														/* pseudo P/N active */
	"", "",														/* pseudo P/N select */
	"", "",														/* pseudo P/N well */
	"CX"};														/* pad frame */
static INTBIG mocmossub_gds_layers[MAXLAYERS] = {
	49, 51, 62, 31,												/* metal 1/2/3/4 */
	46, 56,														/* poly 1/2 */
	43, 43,														/* P/N active */
	44, 45,														/* P/N select */
	41, 42,														/* P/N well */
	25, 25,														/* poly/act cut */
	50, 61, 30,													/* via1/2/3 */
	52,															/* overglass */
	-1,															/* transistor */
	28,															/* poly cap */
	43,															/* P active well */
	-1, -1, -1, -1,												/* pseudo metal 1/2/3/4 */
	-1, -1,														/* pseudo poly 1/2 */
	-1, -1,														/* pseudo P/N active */
	-1, -1,														/* pseudo P/N select */
	-1, -1,														/* pseudo P/N well */
	-1};														/* pad frame */
/* the only available letters are "a", "c", "i", and "y" */
static char *mocmossub_layer_letters[MAXLAYERS] = {
	"m", "h", "r", "q",											/* metal 1/2/3/3 */
	"p", "l",													/* poly 1/2 */
	"s", "d",													/* P/N active */
	"e", "f",													/* P/N select */
	"w", "n",													/* P/N well */
	"j", "k",													/* poly/act cut */
	"v", "u", "z",												/* via1/2/3 */
	"o",														/* overglass */
	"t",														/* transistor */
	"g",														/* poly cap */
	"x",														/* P active well */
	"M", "H", "R", "Q",											/* pseudo metal 1/2/3/4 */
	"P", "L",													/* pseudo poly 1/2 */
	"S", "D",													/* pseudo P/N active */
	"E", "F",													/* pseudo P/N select */
	"W", "N",													/* pseudo P/N well */
	"b"};														/* pad frame */
/* The low 5 bits map Metal-1, Poly-1, Active, Metal-2, and Metal-3 */
static TECH_COLORMAP mocmossub_colmap[32] =
{                  /*     metal3 metal2 active poly1 metal1 */
	{200,200,200}, /* 0:                                    */
	{ 96,209,255}, /* 1:                             metal1 */
	{255,155,192}, /* 2:                       poly1        */
	{ 96,127,192}, /* 3:                       poly1+metal1 */
	{107,226, 96}, /* 4:                active              */
	{ 40,186, 96}, /* 5:                active+      metal1 */
	{107,137, 72}, /* 6:                active+poly1        */
	{ 40,113, 72}, /* 7:                active+poly1+metal1 */
	{224, 95,255}, /* 8:         metal2                     */
	{ 85, 78,255}, /* 9:         metal2+             metal1 */
	{224, 57,192}, /* 10:        metal2+       poly1        */
	{ 85, 47,192}, /* 11:        metal2+       poly1+metal1 */
	{ 94, 84, 96}, /* 12:        metal2+active              */
	{ 36, 69, 96}, /* 13:        metal2+active+      metal1 */
	{ 94, 51, 72}, /* 14:        metal2+active+poly1        */
	{ 36, 42, 72}, /* 15:        metal2+active+poly1+metal1 */
	{247,251, 20}, /* 16: metal3                            */
	{ 77,251,222}, /* 17: metal3+                    metal1 */
	{251,195, 54}, /* 18: metal3+              poly1        */
	{ 91,111,136}, /* 19: metal3+              poly1+metal1 */
	{169,230, 16}, /* 20: metal3+       active              */
	{ 38,161, 68}, /* 21: metal3+       active+      metal1 */
	{101,119, 51}, /* 22: metal3+       active+poly1        */
	{ 38, 98, 51}, /* 23: metal3+       active+poly1+metal1 */
	{236,156,247}, /* 24: metal3+metal2                     */
	{ 80, 68,181}, /* 25: metal3+metal2+             metal1 */
	{211, 50,136}, /* 26: metal3+metal2+       poly1        */
	{ 80, 41,136}, /* 27: metal3+metal2+       poly1+metal1 */
	{ 89, 73, 68}, /* 28: metal3+metal2+active              */
	{ 33, 60, 68}, /* 29: metal3+metal2+active+      metal1 */
	{ 89, 44, 51}, /* 30: metal3+metal2+active+poly1        */
	{ 33, 36, 51}  /* 31: metal3+metal2+active+poly1+metal1 */
};

/******************** DESIGN RULES ********************/

#define	X	XX
#define	A	K1
#define	B	K2
#define	C	K3
#define	D	K4
#define	E	K5
#define	F	K6
#define	R	K18

/* layers that can connect to other layers when electrically disconnected */
static INTBIG mocmossub_unconnectedtable[] = {
/*          M M M M P P P N S S W W P A V V V P T P P M M M M P P P N S S W W P */
/*          e e e e o o A A e e e e o c i i i a r C a e e e e o o A A e e e e a */
/*          t t t t l l c c l l l l l t a a a s a a c t t t t l l c c l l l l d */
/*          1 2 3 4 y y t t P N l l y C 1 2 3 s n p t 1 2 3 4 1 2 t t P N P N F */
/*                  1 2         P N C           s   W P P P P P P P P P P P P r */
/* Met1  */ C,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Met2  */   D,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Met3  */     C,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Met4  */       F,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Poly1 */         C,X,A,A,X,X,X,X,X,X,X,X,X,X,X,X,A,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Poly2 */           C,A,A,X,X,X,X,C,X,X,X,X,X,X,X,A,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* PAct  */             C,C,X,X,X,X,X,X,X,X,X,X,X,X,C,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* NAct  */               C,X,X,X,X,X,X,X,X,X,X,X,X,C,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* SelP  */                 X,X,X,X,X,X,X,X,X,X,C,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* SelN  */                   X,X,X,X,X,X,X,X,X,C,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* WellP */                     X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* WellN */                       X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* PolyC */                         C,C,B,X,X,X,B,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* ActC  */                           C,B,X,X,X,B,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Via1  */                             C,B,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Via2  */                               B,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Via3  */                                 D,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Pass  */                                   X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Trans */                                     X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* PCap  */                                       X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* PactW */                                         X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Met1P */                                           X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Met2P */                                             X,X,X,X,X,X,X,X,X,X,X,X,
/* Met3P */                                               X,X,X,X,X,X,X,X,X,X,X,
/* Met4P */                                                 X,X,X,X,X,X,X,X,X,X,
/* Poly1P */                                                  X,X,X,X,X,X,X,X,X,
/* Poly2P */                                                    X,X,X,X,X,X,X,X,
/* PActP */                                                       X,X,X,X,X,X,X,
/* NActP */                                                         X,X,X,X,X,X,
/* SelPP */                                                           X,X,X,X,X,
/* SelNP */                                                             X,X,X,X,
/* WelPP */                                                               X,X,X,
/* WelNP */                                                                 X,X,
/* PadFr */                                                                   X,
};

/* layers that can connect to other layers when electrically connected */
static INTBIG mocmossub_connectedtable[] = {
/*          M M M M P P P N S S W W P A V V V P T P P M M M M P P P N S S W W P */
/*          e e e e o o A A e e e e o c i i i a r C a e e e e o o A A e e e e a */
/*          t t t t l l c c l l l l l t a a a s a a c t t t t l l c c l l l l d */
/*          1 2 3 4 y y t t P N l l y C 1 2 3 s n p t 1 2 3 4 1 2 t t P N P N F */
/*                  1 2         P N C           s   W P P P P P P P P P P P P r */
/* Met1  */ X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Met2  */   X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Met3  */     X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Met4  */       X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Poly1 */         X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Poly2 */           X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* PAct  */             X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* NAct  */               X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* SelP  */                 X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* SelN  */                   X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* WellP */                     X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* WellN */                       X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* PolyC */                         X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* ActC  */                           X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Via1  */                             X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Via2  */                               X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Via3  */                                 X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Pass  */                                   X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Trans */                                     X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* PCap  */                                       X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* PactW */                                         X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Met1P */                                           X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Met2P */                                             X,X,X,X,X,X,X,X,X,X,X,X,
/* Met3P */                                               X,X,X,X,X,X,X,X,X,X,X,
/* Met4P */                                                 X,X,X,X,X,X,X,X,X,X,
/* Poly1P */                                                  X,X,X,X,X,X,X,X,X,
/* Poly2P */                                                    X,X,X,X,X,X,X,X,
/* PActP */                                                       X,X,X,X,X,X,X,
/* NActP */                                                         X,X,X,X,X,X,
/* SelPP */                                                           X,X,X,X,X,
/* SelNP */                                                             X,X,X,X,
/* WelPP */                                                               X,X,X,
/* WelNP */                                                                 X,X,
/* PadFr */                                                                   X,
};

/******************** ARCS ********************/

#define	ARCPROTOCOUNT  9
#define	AMETAL1        0	/* metal-1                   */
#define	AMETAL2        1	/* metal-2                   */
#define	AMETAL3        2	/* metal-3                   */
#define	AMETAL4        3	/* metal-4                   */
#define	APOLY1         4	/* polysilicon-1             */
#define	APOLY2         5	/* polysilicon-2             */
#define	ASACT          6	/* P-active (or N)           */
#define	ADACT          7	/* N-active (or P)           */
#define	AACT           8	/* General active            */

/* metal 1 arc */
static TECH_ARCLAY mocmossub_al_m1[] = {{LMETAL1,0,FILLED }};
static TECH_ARCS mocmossub_a_m1 = {
	"Metal-1",K3,AMETAL1,											/* name */
	1,mocmossub_al_m1,												/* layers */
	(APMETAL1<<AFUNCTIONSH)|WANTFIXANG|CANWIPE|(90<<AANGLEINCSH)};	/* userbits */

/* metal 2 arc */
static TECH_ARCLAY mocmossub_al_m2[] = {{LMETAL2,0,FILLED }};
static TECH_ARCS mocmossub_a_m2 = {
	"Metal-2",K3,AMETAL2,											/* name */
	1,mocmossub_al_m2,												/* layers */
	(APMETAL2<<AFUNCTIONSH)|WANTFIXANG|CANWIPE|(90<<AANGLEINCSH)};	/* userbits */

/* metal 3 arc */
static TECH_ARCLAY mocmossub_al_m3[] = {{LMETAL3,0,FILLED }};
static TECH_ARCS mocmossub_a_m3 = {
	"Metal-3",K3,AMETAL3,											/* name */
	1,mocmossub_al_m3,												/* layers */
	(APMETAL3<<AFUNCTIONSH)|WANTFIXANG|CANWIPE|(90<<AANGLEINCSH)};	/* userbits */

/* metal 4 arc */
static TECH_ARCLAY mocmossub_al_m4[] = {{LMETAL4,0,FILLED }};
static TECH_ARCS mocmossub_a_m4 = {
	"Metal-4",K6,AMETAL4,											/* name */
	1,mocmossub_al_m4,												/* layers */
	(APMETAL4<<AFUNCTIONSH)|WANTFIXANG|CANWIPE|(90<<AANGLEINCSH)};	/* userbits */

/* polysilicon 1 arc */
static TECH_ARCLAY mocmossub_al_p1[] = {{LPOLY1,0,FILLED }};
static TECH_ARCS mocmossub_a_po1 = {
	"Polysilicon-1",K2,APOLY1,										/* name */
	1,mocmossub_al_p1,												/* layers */
	(APPOLY1<<AFUNCTIONSH)|WANTFIXANG|CANWIPE|(90<<AANGLEINCSH)};	/* userbits */

/* polysilicon 2 arc */
static TECH_ARCLAY mocmossub_al_p2[] = {{LPOLY2,0,FILLED }};
static TECH_ARCS mocmossub_a_po2 = {
	"Polysilicon-2",K3,APOLY2,										/* name */
	1,mocmossub_al_p2,												/* layers */
	(APPOLY2<<AFUNCTIONSH)|WANTFIXANG|CANWIPE|(90<<AANGLEINCSH)};	/* userbits */

/* P-active arc */
static TECH_ARCLAY mocmossub_al_pa[] = {{LSACT,K12,FILLED}, {LWELLNP,0,FILLED},
	{LSELECTP,K8,FILLED}};
static TECH_ARCS mocmossub_a_pa = {
	"P-Active",K15,ASACT,											/* name */
	3,mocmossub_al_pa,												/* layers */
	(APDIFFP<<AFUNCTIONSH)|WANTFIXANG|CANWIPE|(90<<AANGLEINCSH)};	/* userbits */

/* N-active arc */
static TECH_ARCLAY mocmossub_al_na[] = {{LDACT,K12,FILLED}, {LWELLP,0,FILLED},
	{LSELECTNP,K8,FILLED}};
static TECH_ARCS mocmossub_a_na = {
	"N-Active",K15,ADACT,											/* name */
	3,mocmossub_al_na,												/* layers */
	(APDIFFN<<AFUNCTIONSH)|WANTFIXANG|CANWIPE|(90<<AANGLEINCSH)};	/* userbits */

/* General active arc */
static TECH_ARCLAY mocmossub_al_a[] = {{LDACT,0,FILLED}, {LSACT,0,FILLED}};
static TECH_ARCS mocmossub_a_a = {
	"Active",K3,AACT,												/* name */
	2,mocmossub_al_a,												/* layers */
	(APDIFF<<AFUNCTIONSH)|WANTFIXANG|CANWIPE|(90<<AANGLEINCSH)};	/* userbits */

TECH_ARCS *mocmossub_arcprotos[ARCPROTOCOUNT+1] = {
	&mocmossub_a_m1, &mocmossub_a_m2, &mocmossub_a_m3, &mocmossub_a_m4,
	&mocmossub_a_po1, &mocmossub_a_po2,
	&mocmossub_a_pa, &mocmossub_a_na,
	&mocmossub_a_a, ((TECH_ARCS *)-1)};

static INTBIG mocmossub_arc_widoff[ARCPROTOCOUNT] = {0,0,0,0, 0,0, K12,K12, 0};

/******************** PORTINST CONNECTIONS ********************/

static INTBIG mocmossub_pc_m1[]   = {-1, AMETAL1, ALLGEN, -1};
static INTBIG mocmossub_pc_m1a[]  = {-1, AMETAL1, AACT, ALLGEN, -1};
static INTBIG mocmossub_pc_m2[]   = {-1, AMETAL2, ALLGEN, -1};
static INTBIG mocmossub_pc_m3[]   = {-1, AMETAL3, ALLGEN, -1};
static INTBIG mocmossub_pc_m4[]   = {-1, AMETAL4, ALLGEN, -1};
static INTBIG mocmossub_pc_p1[]   = {-1, APOLY1, ALLGEN, -1};
static INTBIG mocmossub_pc_p2[]   = {-1, APOLY2, ALLGEN, -1};
static INTBIG mocmossub_pc_pa[]   = {-1, ASACT, ALLGEN, -1};
static INTBIG mocmossub_pc_a[]    = {-1, AACT, ASACT, ADACT, ALLGEN,-1};
static INTBIG mocmossub_pc_na[]   = {-1, ADACT, ALLGEN, -1};
static INTBIG mocmossub_pc_pam1[] = {-1, ASACT, AMETAL1, ALLGEN, -1};
static INTBIG mocmossub_pc_nam1[] = {-1, ADACT, AMETAL1, ALLGEN, -1};
static INTBIG mocmossub_pc_pm1[]  = {-1, APOLY1, AMETAL1, ALLGEN, -1};
static INTBIG mocmossub_pc_pm2[]  = {-1, APOLY2, AMETAL1, ALLGEN, -1};
static INTBIG mocmossub_pc_pm12[] = {-1, APOLY1, APOLY2, AMETAL1, ALLGEN, -1};
static INTBIG mocmossub_pc_m1m2[] = {-1, AMETAL1, AMETAL2, ALLGEN, -1};
static INTBIG mocmossub_pc_m2m3[] = {-1, AMETAL2, AMETAL3, ALLGEN, -1};
static INTBIG mocmossub_pc_m3m4[] = {-1, AMETAL3, AMETAL4, ALLGEN, -1};
static INTBIG mocmossub_pc_null[] = {-1, ALLGEN, -1};

/******************** NODES ********************/

#define	NODEPROTOCOUNT 40
#define	NMETAL1P        1	/* metal-1 pin */
#define	NMETAL2P        2	/* metal-2 pin */
#define	NMETAL3P        3	/* metal-3 pin */
#define	NMETAL4P        4	/* metal-4 pin */
#define	NPOLY1P         5	/* polysilicon-1 pin */
#define	NPOLY2P         6	/* polysilicon-2 pin */
#define	NSACTP          7	/* P-active (or N) pin */
#define	NDACTP          8	/* N-active (or P) pin */
#define	NACTP           9	/* General active pin */
#define	NMETSACTC      10	/* metal-1-P-active (or N) contact */
#define	NMETDACTC      11	/* metal-1-N-active (or P) contact */
#define	NMETPOLY1C     12	/* metal-1-polysilicon-1 contact */
#define	NMETPOLY2C     13	/* metal-1-polysilicon-2 contact */
#define	NMETPOLY12C    14	/* metal-1-polysilicon-1-2 capacitor/contact */
#define	NSTRANS        15	/* P-transistor (or N) */
#define	NDTRANS        16	/* N-transistor (or P) */
#define	NVIA1          17	/* metal-1-metal-2 contact */
#define	NVIA2          18	/* metal-2-metal-3 contact */
#define	NVIA3          19	/* metal-3-metal-4 contact */
#define	NWBUT          20	/* metal-1-Well contact */
#define	NSBUT          21	/* metal-1-Substrate contact */
#define	NMETAL1N       22	/* metal-1 node */
#define	NMETAL2N       23	/* metal-2 node */
#define	NMETAL3N       24	/* metal-3 node */
#define	NMETAL4N       25	/* metal-4 node */
#define	NPOLY1N        26	/* polysilicon-1 node */
#define	NPOLY2N        27	/* polysilicon-2 node */
#define	NACTIVEN       28	/* active(P) node */
#define	NDACTIVEN      29	/* N-active node */
#define	NSELECTPN      30	/* P-select node */
#define	NSELECTNN      31	/* N-select node */
#define	NPCUTN         32	/* poly cut node */
#define	NACUTN         33	/* active cut node */
#define	NVIA1N         34	/* via-1 node */
#define	NVIA2N         35	/* via-2 node */
#define	NVIA3N         36	/* via-3 node */
#define	NWELLPN        37	/* P-well node */
#define	NWELLNN        38	/* N-well node */
#define	NPASSN         39	/* passivation node */
#define	NPADFRN        40	/* pad frame node */

/* for geometry */
static INTBIG mocmossub_cutbox[8]  = {LEFTIN1,  BOTIN1,   LEFTIN3,   BOTIN3};/*adjust*/
static INTBIG mocmossub_fullbox[8] = {LEFTEDGE, BOTEDGE,  RIGHTEDGE, TOPEDGE};
static INTBIG mocmossub_in0hbox[8] = {LEFTIN0H, BOTIN0H,  RIGHTIN0H, TOPIN0H};
static INTBIG mocmossub_in1box[8]  = {LEFTIN1,  BOTIN1,   RIGHTIN1,  TOPIN1};
static INTBIG mocmossub_in2box[8]  = {LEFTIN2,  BOTIN2,   RIGHTIN2,  TOPIN2};
static INTBIG mocmossub_in4box[8]  = {LEFTIN4,  BOTIN4,   RIGHTIN4,  TOPIN4};
static INTBIG mocmossub_in5box[8]  = {LEFTIN5,  BOTIN5,   RIGHTIN5,  TOPIN5};
static INTBIG mocmossub_in6box[8]  = {LEFTIN6,  BOTIN6,   RIGHTIN6,  TOPIN6};
static INTBIG mocmossub_in6hbox[8] = {LEFTIN6H, BOTIN6H,  RIGHTIN6H, TOPIN6H};
static INTBIG mocmossub_min5box[16]= {LEFTIN5,  BOTIN5,   RIGHTIN5,  TOPIN5,
									 CENTERL2, CENTERD2, CENTERR2,  CENTERU2};
static INTBIG mocmossub_trnpbox[8] = {LEFTIN4,  BOTIN9,   RIGHTIN4,  TOPIN9};
static INTBIG mocmossub_trntbox[8] = {LEFTIN6,  BOTIN9,   RIGHTIN6,  TOPIN9};
static INTBIG mocmossub_trna1box[8]= {LEFTIN6,  BOTIN9,   RIGHTIN6,  TOPIN9};
static INTBIG mocmossub_trna2box[8]= {LEFTIN6,  TOPIN9,   RIGHTIN6,  TOPIN6};
static INTBIG mocmossub_trna3box[8]= {LEFTIN6,  BOTIN6,   RIGHTIN6,  BOTIN9};

/* metal-1-pin */
static TECH_PORTS mocmossub_pm1_p[] = {				/* ports */
	{mocmossub_pc_m1, "metal-1", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN1H, BOTIN1H, RIGHTIN1H, TOPIN1H}};
static TECH_POLYGON mocmossub_pm1_l[] = {			/* layers */
	{LMET1P, 0, 4, CROSSED, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_pm1 = {
	"Metal-1-Pin",NMETAL1P,NONODEPROTO,				/* name */
	K3,K3,											/* size */
	1,mocmossub_pm1_p,								/* ports */
	1,mocmossub_pm1_l,								/* layers */
	(NPPIN<<NFUNCTIONSH)|ARCSWIPE|ARCSHRINK,		/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* metal-2-pin */
static TECH_PORTS mocmossub_pm2_p[] = {				/* ports */
	{mocmossub_pc_m2, "metal-2", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN1H, BOTIN1H, RIGHTIN1H, TOPIN1H}};
static TECH_POLYGON mocmossub_pm2_l[] = {			/* layers */
	{LMET2P, 0, 4, CROSSED, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_pm2 = {
	"Metal-2-Pin",NMETAL2P,NONODEPROTO,				/* name */
	K3,K3,											/* size */
	1,mocmossub_pm2_p,								/* ports */
	1,mocmossub_pm2_l,								/* layers */
	(NPPIN<<NFUNCTIONSH)|ARCSWIPE|ARCSHRINK,		/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* metal-3-pin */
static TECH_PORTS mocmossub_pm3_p[] = {				/* ports */
	{mocmossub_pc_m3, "metal-3", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN1H, BOTIN1H, RIGHTIN1H, TOPIN1H}};
static TECH_POLYGON mocmossub_pm3_l[] = {			/* layers */
	{LMET3P, 0, 4, CROSSED, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_pm3 = {
	"Metal-3-Pin",NMETAL3P,NONODEPROTO,				/* name */
	K3,K3,											/* size */
	1,mocmossub_pm3_p,								/* ports */
	1,mocmossub_pm3_l,								/* layers */
	(NPPIN<<NFUNCTIONSH)|ARCSWIPE|ARCSHRINK,		/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* metal-4-pin */
static TECH_PORTS mocmossub_pm4_p[] = {				/* ports */
	{mocmossub_pc_m4, "metal-4", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN3, BOTIN3, RIGHTIN3, TOPIN3}};
static TECH_POLYGON mocmossub_pm4_l[] = {			/* layers */
	{LMET4P, 0, 4, CROSSED, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_pm4 = {
	"Metal-4-Pin",NMETAL4P,NONODEPROTO,				/* name */
	K6,K6,											/* size */
	1,mocmossub_pm4_p,								/* ports */
	1,mocmossub_pm4_l,								/* layers */
	(NPPIN<<NFUNCTIONSH)|ARCSWIPE|ARCSHRINK,		/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* polysilicon-1-pin */
static TECH_PORTS mocmossub_pp1_p[] = {				/* ports */
	{mocmossub_pc_p1, "polysilicon-1", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN1, BOTIN1, RIGHTIN1, TOPIN1}};
static TECH_POLYGON mocmossub_pp1_l[] = {			/* layers */
	{LPOLY1P, 0, 4, CROSSED, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_pp1 = {
	"Polysilicon-1-Pin",NPOLY1P,NONODEPROTO,		/* name */
	K2,K2,											/* size */
	1,mocmossub_pp1_p,								/* ports */
	1,mocmossub_pp1_l,								/* layers */
	(NPPIN<<NFUNCTIONSH)|ARCSWIPE|ARCSHRINK,		/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* polysilicon-2-pin */
static TECH_PORTS mocmossub_pp2_p[] = {				/* ports */
	{mocmossub_pc_p2, "polysilicon-2", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN1H, BOTIN1H, RIGHTIN1H, TOPIN1H}};
static TECH_POLYGON mocmossub_pp2_l[] = {			/* layers */
	{LPOLY2P, 0, 4, CROSSED, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_pp2 = {
	"Polysilicon-2-Pin",NPOLY2P,NONODEPROTO,		/* name */
	K3,K3,											/* size */
	1,mocmossub_pp2_p,								/* ports */
	1,mocmossub_pp2_l,								/* layers */
	(NPPIN<<NFUNCTIONSH)|ARCSWIPE|ARCSHRINK,		/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* P-active-pin (or N) */
static TECH_PORTS mocmossub_psa_p[] = {				/* ports */
	{mocmossub_pc_pa, "p-active", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN7H, BOTIN7H, RIGHTIN7H, TOPIN7H}};
static TECH_POLYGON mocmossub_psa_l[] = {			/* layers */
	{LSACTP,    0, 4, CROSSED, BOX, mocmossub_in6box},
	{LWELLNP,  -1, 4, CROSSED, BOX, mocmossub_fullbox},
	{LSELECTPP,-1, 4, CROSSED, BOX, mocmossub_in4box}};
static TECH_NODES mocmossub_psa = {
	"P-Active-Pin",NSACTP,NONODEPROTO,				/* name */
	K15,K15,										/* size */
	1,mocmossub_psa_p,								/* ports */
	3,mocmossub_psa_l,								/* layers */
	(NPPIN<<NFUNCTIONSH)|ARCSWIPE|ARCSHRINK,		/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* N-active-pin (or P) */
static TECH_PORTS mocmossub_pda_p[] = {				/* ports */
	{mocmossub_pc_na, "n-active", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN7H, BOTIN7H, RIGHTIN7H, TOPIN7H}};
static TECH_POLYGON mocmossub_pda_l[] = {			/* layers */
	{LDACTP,    0, 4, CROSSED, BOX, mocmossub_in6box},
	{LWELLPP,  -1, 4, CROSSED, BOX, mocmossub_fullbox},
	{LSELECTNP,-1, 4, CROSSED, BOX, mocmossub_in4box}};
static TECH_NODES mocmossub_pda = {
	"N-Active-Pin",NDACTP,NONODEPROTO,				/* name */
	K15,K15,										/* size */
	1,mocmossub_pda_p,								/* ports */
	3,mocmossub_pda_l,								/* layers */
	(NPPIN<<NFUNCTIONSH)|ARCSWIPE|ARCSHRINK,		/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* General active-pin */
static TECH_PORTS mocmossub_pa_p[] = {				/* ports */
	{mocmossub_pc_a, "active", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN1H, BOTIN1H, RIGHTIN1H, TOPIN1H}}	;
static TECH_POLYGON mocmossub_pa_l[] = {			/* layers */
	{LDACTP, 0, 4, CROSSED, BOX, mocmossub_fullbox},
	{LSACTP, 0, 4, CROSSED, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_pa = {
	"Active-Pin",NACTP,NONODEPROTO,					/* name */
	K3,K3,											/* size */
	1,mocmossub_pa_p,								/* ports */
	2,mocmossub_pa_l,								/* layers */
	(NPPIN<<NFUNCTIONSH)|ARCSWIPE|ARCSHRINK,		/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* metal-1-P-active-contact (or N) */
static TECH_PORTS mocmossub_mpa_p[] = {				/* ports */
	{mocmossub_pc_pam1, "metal-1-p-act", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN8, BOTIN8, RIGHTIN8, TOPIN8}};
static TECH_POLYGON mocmossub_mpa_l[] = {			/* layers */
	{LMETAL1,  0, 4, FILLEDRECT, BOX, mocmossub_in6hbox},
	{LSACT,    0, 4, FILLEDRECT, BOX, mocmossub_in6box},
	{LWELLNP, -1, 4, FILLEDRECT, BOX, mocmossub_fullbox},
	{LSELECTP,-1, 4, FILLEDRECT, BOX, mocmossub_in5box},
	{LACTCUT, -1, 4, FILLEDRECT, BOX, mocmossub_cutbox}};
static TECH_NODES mocmossub_mpa = {
	"Metal-1-P-Active-Con",NMETSACTC,NONODEPROTO,	/* name */
	K17,K17,										/* size */
	1,mocmossub_mpa_p,								/* ports */
	5,mocmossub_mpa_l,								/* layers */
	(NPCONTACT<<NFUNCTIONSH),						/* userbits */
	MULTICUT,K2,K2,H1,K3,0,0};						/* characteristics */

/* metal-1-N-active-contact (or P) */
static TECH_PORTS mocmossub_mna_p[] = {				/* ports */
	{mocmossub_pc_nam1, "metal-1-n-act", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN8, BOTIN8, RIGHTIN8, TOPIN8}};
static TECH_POLYGON mocmossub_mna_l[] = {			/* layers */
	{LMETAL1,   0, 4, FILLEDRECT, BOX, mocmossub_in6hbox},
	{LDACT,     0, 4, FILLEDRECT, BOX, mocmossub_in6box},
	{LWELLP,   -1, 4, FILLEDRECT, BOX, mocmossub_fullbox},
	{LSELECTNP,-1, 4, FILLEDRECT, BOX, mocmossub_in5box},
	{LACTCUT,  -1, 4, FILLEDRECT, BOX, mocmossub_cutbox}};
static TECH_NODES mocmossub_mna = {
	"Metal-1-N-Active-Con",NMETDACTC,NONODEPROTO,	/* name */
	K17,K17,										/* size */
	1,mocmossub_mna_p,								/* ports */
	5,mocmossub_mna_l,								/* layers */
	(NPCONTACT<<NFUNCTIONSH),						/* userbits */
	MULTICUT,K2,K2,H1,K3,0,0};						/* characteristics */

/* metal-1-polysilicon-1-contact */
static TECH_PORTS mocmossub_mp1_p[] = {				/* ports */
	{mocmossub_pc_pm1, "metal-1-polysilicon-1", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN2, BOTIN2, RIGHTIN2, TOPIN2}};
static TECH_POLYGON mocmossub_mp1_l[] = {			/* layers */
	{LMETAL1,  0, 4, FILLEDRECT, BOX, mocmossub_in0hbox},
	{LPOLY1,   0, 4, FILLEDRECT, BOX, mocmossub_fullbox},
	{LPOLYCUT,-1, 4, FILLEDRECT, BOX, mocmossub_cutbox}};
static TECH_NODES mocmossub_mp1 = {
	"Metal-1-Polysilicon-1-Con",NMETPOLY1C,NONODEPROTO,/* name */
	K5,K5,											/* size */
	1,mocmossub_mp1_p,								/* ports */
	3,mocmossub_mp1_l,								/* layers */
	(NPCONTACT<<NFUNCTIONSH),						/* userbits */
	MULTICUT,K2,K2,H1,K3,0,0};						/* characteristics */

/* metal-1-polysilicon-2-contact */
static TECH_PORTS mocmossub_mp2_p[] = {				/* ports */
	{mocmossub_pc_pm2, "metal-1-polysilicon-2", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN1H, BOTIN1H, RIGHTIN1H, TOPIN1H}};
static TECH_POLYGON mocmossub_mp2_l[] = {			/* layers */
	{LMETAL1,  0, 4, FILLEDRECT, BOX, mocmossub_fullbox},
	{LPOLY2,   0, 4, FILLEDRECT, BOX, mocmossub_in0hbox},
	{LPOLYCUT,-1, 4, FILLEDRECT, BOX, mocmossub_cutbox}};
static TECH_NODES mocmossub_mp2 = {
	"Metal-1-Polysilicon-2-Con",NMETPOLY2C,NONODEPROTO,/* name */
	K4,K4,											/* size */
	1,mocmossub_mp2_p,								/* ports */
	3,mocmossub_mp2_l,								/* layers */
	(NPCONTACT<<NFUNCTIONSH),						/* userbits */
	MULTICUT,K2,K2,K1,K3,0,0};						/* characteristics */

/* metal-1-polysilicon-1-2-contact */
static TECH_PORTS mocmossub_mp12_p[] = {				/* ports */
	{mocmossub_pc_pm12, "metal-1-polysilicon-1-2", NOPORTPROTO, (180<<PORTARANGESH),
		CENTER, CENTER, CENTER, CENTER}};
static TECH_POLYGON mocmossub_mp12_l[] = {			/* layers */
	{LPOLY1,   0, 4, FILLEDRECT, BOX, mocmossub_fullbox},
	{LPOLY2,   0, 4, FILLEDRECT, BOX, mocmossub_in2box},
	{LPOLYCUT,-1, 4, FILLEDRECT, BOX, mocmossub_cutbox}};
static TECH_NODES mocmossub_mp12 = {
	"Metal-1-Polysilicon-1-2-Con",NMETPOLY12C,NONODEPROTO,/* name */
	K7,K7,											/* size */
	1,mocmossub_mp12_p,								/* ports */
	3,mocmossub_mp12_l,								/* layers */
	(NPCONTACT<<NFUNCTIONSH),						/* userbits */
	MULTICUT,K2,K2,H2,K3,0,0};						/* characteristics */

/* P-transistor (or N) */
static TECH_PORTS mocmossub_tpa_p[] = {				/* ports */
	{mocmossub_pc_p1, "p-trans-poly-left",  NOPORTPROTO, (180<<PORTANGLESH)|
		(90<<PORTARANGESH),                LEFTIN4,  BOTIN10, LEFTIN4,   TOPIN10},
	{mocmossub_pc_pa, "p-trans-diff-top",   NOPORTPROTO, (90<<PORTANGLESH)|
		(90<<PORTARANGESH)|(1<<PORTNETSH), LEFTIN7H,  TOPIN6H, RIGHTIN7H,  TOPIN6},
	{mocmossub_pc_p1, "p-trans-poly-right", NOPORTPROTO, (0<<PORTANGLESH)|
		(90<<PORTARANGESH),                RIGHTIN4, BOTIN10, RIGHTIN4,  TOPIN10},
	{mocmossub_pc_pa, "p-trans-diff-bottom",NOPORTPROTO, (270<<PORTANGLESH)|
		(90<<PORTARANGESH)|(2<<PORTNETSH), LEFTIN7H,  BOTIN6, RIGHTIN7H,  BOTIN6}};
static TECH_SERPENT mocmossub_tpa_l[] = {			/* graphical layers */
	{{LSACT,    1, 4, FILLEDRECT, BOX,    mocmossub_in6box},  K3, K3,  0},
	{{LPOLY1,   0, 4, FILLEDRECT, BOX,    mocmossub_trnpbox}, K1, K1, K2},
	{{LWELLNP, -1, 4, FILLEDRECT, BOX,    mocmossub_fullbox}, K7, K7, K4},
	{{LSELECTP,-1, 4, FILLEDRECT, BOX,    mocmossub_in4box},  K5, K5, K2},
	{{LTRANS,  -1, 4, FILLEDRECT, BOX,    mocmossub_trntbox}, K1, K1,  0}};
static TECH_SERPENT mocmossub_tpaE_l[] = {			/* electric layers */
	{{LSACT,    1, 4, FILLEDRECT, BOX,    mocmossub_trna2box}, K3,-K1,  0},
	{{LSACT,    3, 4, FILLEDRECT, BOX,    mocmossub_trna3box},-K1, K3,  0},
	{{LSACT,   -1, 4, FILLEDRECT, BOX,    mocmossub_trna1box}, K1, K1,  0},
	{{LPOLY1,   0, 4, FILLEDRECT, BOX,    mocmossub_trnpbox},  K1, K1, K2},
	{{LWELLNP, -1, 4, FILLEDRECT, BOX,    mocmossub_fullbox},  K7, K7, K4},
	{{LSELECTP,-1, 4, FILLEDRECT, BOX,    mocmossub_in4box},   K5, K5, K2},
	{{LTRANS,  -1, 4, FILLEDRECT, BOX,    mocmossub_trntbox},  K1, K1,  0}};
static TECH_NODES mocmossub_tpa = {
	"P-Transistor",NSTRANS,NONODEPROTO,				/* name */
	K15,K20,										/* size */
	4,mocmossub_tpa_p,								/* ports */
	5,(TECH_POLYGON *)0,							/* layers */
	NODESHRINK|(NPTRAPMOS<<NFUNCTIONSH)|HOLDSTRACE,	/* userbits */
	SERPTRANS,K1,K1,K2,0,mocmossub_tpa_l,mocmossub_tpaE_l};/* characteristics */

/* N-transistor (or P) */
static TECH_PORTS mocmossub_tna_p[] = {				/* ports */
	{mocmossub_pc_p1, "n-trans-poly-left", NOPORTPROTO, (180<<PORTANGLESH)|
		(90<<PORTARANGESH),                LEFTIN4,  BOTIN10, LEFTIN4,   TOPIN10},
	{mocmossub_pc_na, "n-trans-diff-top",  NOPORTPROTO, (90<<PORTANGLESH)|
		(90<<PORTARANGESH)|(1<<PORTNETSH), LEFTIN7H,  TOPIN6H, RIGHTIN7H,  TOPIN6},
	{mocmossub_pc_p1, "n-trans-poly-right",NOPORTPROTO, (0<<PORTANGLESH)|
		(90<<PORTARANGESH),                RIGHTIN4, BOTIN10, RIGHTIN4,  TOPIN10},
	{mocmossub_pc_na, "n-trans-diff-bot",  NOPORTPROTO, (270<<PORTANGLESH)|
		(90<<PORTARANGESH)|(2<<PORTNETSH), LEFTIN7H,  BOTIN6, RIGHTIN7H,  BOTIN6}};
static TECH_SERPENT mocmossub_tna_l[] = {			/* graphical layers */
	{{LDACT,     1, 4, FILLEDRECT, BOX,    mocmossub_in6box}, K3, K3,  0},
	{{LPOLY1,    0, 4, FILLEDRECT, BOX,    mocmossub_trnpbox}, K1, K1, K2},
	{{LWELLP,   -1, 4, FILLEDRECT, BOX,    mocmossub_fullbox}, K7, K7, K4},
	{{LSELECTNP,-1, 4, FILLEDRECT, BOX,    mocmossub_in4box},  K5, K5, K2},
	{{LTRANS,   -1, 4, FILLEDRECT, BOX,    mocmossub_trntbox}, K1, K1,  0}};
static TECH_SERPENT mocmossub_tnaE_l[] = {			/* electric layers */
	{{LDACT,     1, 4, FILLEDRECT, BOX,    mocmossub_trna2box}, K3,-K1,  0},
	{{LDACT,     3, 4, FILLEDRECT, BOX,    mocmossub_trna3box},-K1, K3,  0},
	{{LDACT,    -1, 4, FILLEDRECT, BOX,    mocmossub_trna1box}, K1, K1,  0},
	{{LPOLY1,    0, 4, FILLEDRECT, BOX,    mocmossub_trnpbox}, K1, K1, K2},
	{{LWELLP,   -1, 4, FILLEDRECT, BOX,    mocmossub_fullbox}, K7, K7, K4},
	{{LSELECTNP,-1, 4, FILLEDRECT, BOX,    mocmossub_in4box},  K5, K5, K2},
	{{LTRANS,   -1, 4, FILLEDRECT, BOX,    mocmossub_trntbox}, K1, K1,  0}};
static TECH_NODES mocmossub_tna = {
	"N-Transistor",NDTRANS,NONODEPROTO,				/* name */
	K15,K20,										/* size */
	4,mocmossub_tna_p,								/* ports */
	5,(TECH_POLYGON *)0,							/* layers */
	NODESHRINK|(NPTRANMOS<<NFUNCTIONSH)|HOLDSTRACE,	/* userbits */
	SERPTRANS,K1,K1,K2,0,mocmossub_tna_l,mocmossub_tnaE_l};/* characteristics */

/* metal-1-metal-2-contact */
static TECH_PORTS mocmossub_m1m2_p[] = {				/* ports */
	{mocmossub_pc_m1m2, "metal-1-metal-2", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN1H, BOTIN1H, RIGHTIN1H, TOPIN1H}};
static TECH_POLYGON mocmossub_m1m2_l[] = {			/* layers */
	{LMETAL1, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox},
	{LMETAL2, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox},
	{LVIA1,  -1, 4, FILLEDRECT, BOX, mocmossub_cutbox}};
static TECH_NODES mocmossub_m1m2 = {
	"Metal-1-Metal-2-Con",NVIA1,NONODEPROTO,		/* name */
	K4,K4,											/* size */
	1,mocmossub_m1m2_p,								/* ports */
	3,mocmossub_m1m2_l,								/* layers */
	(NPCONTACT<<NFUNCTIONSH),						/* userbits */
	MULTICUT,K2,K2,K1,K3,0,0};						/* characteristics */

/* metal-2-metal-3-contact */
static TECH_PORTS mocmossub_m2m3_p[] = {			/* ports */
	{mocmossub_pc_m2m3, "metal-2-metal-3", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN2H, BOTIN2H, RIGHTIN2H, TOPIN2H}};
static TECH_POLYGON mocmossub_m2m3_l[] = {			/* layers */
	{LMETAL2, 0, 4, FILLEDRECT, BOX, mocmossub_in1box},
	{LMETAL3, 0, 4, FILLEDRECT, BOX, mocmossub_in1box},
	{LVIA2,  -1, 4, FILLEDRECT, BOX, mocmossub_cutbox}};
static TECH_NODES mocmossub_m2m3 = {
	"Metal-2-Metal-3-Con",NVIA2,NONODEPROTO,		/* name */
	K6,K6,											/* size */
	1,mocmossub_m2m3_p,								/* ports */
	3,mocmossub_m2m3_l,								/* layers */
	(NPCONTACT<<NFUNCTIONSH),						/* userbits */
	MULTICUT,K2,K2,K1,K3,0,0};						/* characteristics */

/* metal-3-metal-4-contact */
static TECH_PORTS mocmossub_m3m4_p[] = {			/* ports */
	{mocmossub_pc_m3m4, "metal-3-metal-4", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN2H, BOTIN2H, RIGHTIN2H, TOPIN2H}};
static TECH_POLYGON mocmossub_m3m4_l[] = {			/* layers */
	{LMETAL3, 0, 4, FILLEDRECT, BOX, mocmossub_in1box},
	{LMETAL4, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox},
	{LVIA3,  -1, 4, FILLEDRECT, BOX, mocmossub_cutbox}};
static TECH_NODES mocmossub_m3m4 = {
	"Metal-3-Metal-4-Con",NVIA3,NONODEPROTO,		/* name */
	K6,K6,											/* size */
	1,mocmossub_m3m4_p,								/* ports */
	3,mocmossub_m3m4_l,								/* layers */
	(NPCONTACT<<NFUNCTIONSH),						/* userbits */
	MULTICUT,K2,K2,K2,K4,0,0};						/* characteristics */

/* Metal-1-Well Contact */
static TECH_PORTS mocmossub_psub_p[] = {			/* ports */
	{mocmossub_pc_m1a, "metal-1-well", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN5H, BOTIN5H, RIGHTIN5H, TOPIN5H}};
static TECH_POLYGON mocmossub_psub_l[] = {			/* layers */
	{LMETAL1,   0, 4, FILLEDRECT, MINBOX, mocmossub_min5box},
	{LSACTWELL, 0, 4, FILLEDRECT, BOX,    mocmossub_in4box},
	{LWELLP,   -1, 4, FILLEDRECT, BOX,    mocmossub_fullbox},
	{LSELECTP, -1, 4, FILLEDRECT, MINBOX, mocmossub_min5box},
	{LACTCUT,  -1, 4, FILLEDRECT, BOX,    mocmossub_cutbox}};
static TECH_NODES mocmossub_psub = {
	"Metal-1-Well-Con",NWBUT,NONODEPROTO,			/* name */
	K14,K14,										/* size */
	1,mocmossub_psub_p,								/* ports */
	5,mocmossub_psub_l,								/* layers */
	(NPWELL<<NFUNCTIONSH),							/* userbits */
	MULTICUT,K2,K2,H1,K3,0,0};						/* characteristics */

/* Metal-1-Substrate Contact */
static TECH_PORTS mocmossub_nsub_p[] = {			/* ports */
	{mocmossub_pc_m1a, "metal-1-substrate", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN5H, BOTIN5H, RIGHTIN5H, TOPIN5H}};
static TECH_POLYGON mocmossub_nsub_l[] = {			/* layers */
	{LMETAL1,   0, 4, FILLEDRECT, MINBOX, mocmossub_min5box},
	{LDACT,     0, 4, FILLEDRECT, BOX,    mocmossub_in4box},
	{LWELLNP,  -1, 4, FILLEDRECT, BOX,    mocmossub_fullbox},
	{LSELECTNP,-1, 4, FILLEDRECT, BOX,    mocmossub_in5box},
	{LACTCUT,  -1, 4, FILLEDRECT, BOX,    mocmossub_cutbox}};
static TECH_NODES mocmossub_nsub = {
	"Metal-1-Substrate-Con",NSBUT,NONODEPROTO,		/* name */
	K14,K14,										/* size */
	1,mocmossub_nsub_p,								/* ports */
	5,mocmossub_nsub_l,								/* layers */
	(NPSUBSTRATE<<NFUNCTIONSH),						/* userbits */
	MULTICUT,K2,K2,H1,K3,0,0};						/* characteristics */

/* Metal-1-Node */
static TECH_PORTS mocmossub_m1_p[] = {				/* ports */
	{mocmossub_pc_m1, "metal-1", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN1H, BOTIN1H, RIGHTIN1H, TOPIN1H}};
static TECH_POLYGON mocmossub_m1_l[] = {			/* layers */
	{LMETAL1, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_m1 = {
	"Metal-1-Node",NMETAL1N,NONODEPROTO,			/* name */
	K3,K3,											/* size */
	1,mocmossub_m1_p,								/* ports */
	1,mocmossub_m1_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE,				/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* Metal-2-Node */
static TECH_PORTS mocmossub_m2_p[] = {				/* ports */
	{mocmossub_pc_m2, "metal-2", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN1H, BOTIN1H, RIGHTIN1H, TOPIN1H}};
static TECH_POLYGON mocmossub_m2_l[] = {			/* layers */
	{LMETAL2, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_m2 = {
	"Metal-2-Node",NMETAL2N,NONODEPROTO,			/* name */
	K3,K3,											/* size */
	1,mocmossub_m2_p,								/* ports */
	1,mocmossub_m2_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE,				/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* Metal-3-Node */
static TECH_PORTS mocmossub_m3_p[] = {				/* ports */
	{mocmossub_pc_m3, "metal-3", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN2H, BOTIN2H, RIGHTIN2H, TOPIN2H}};
static TECH_POLYGON mocmossub_m3_l[] = {			/* layers */
	{LMETAL3, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_m3 = {
	"Metal-3-Node",NMETAL3N,NONODEPROTO,			/* name */
	K5,K5,											/* size */
	1,mocmossub_m3_p,								/* ports */
	1,mocmossub_m3_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE,				/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* Metal-4-Node */
static TECH_PORTS mocmossub_m4_p[] = {				/* ports */
	{mocmossub_pc_m4, "metal-4", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN3, BOTIN3, RIGHTIN3, TOPIN3}};
static TECH_POLYGON mocmossub_m4_l[] = {			/* layers */
	{LMETAL4, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_m4 = {
	"Metal-4-Node",NMETAL4N,NONODEPROTO,			/* name */
	K6,K6,											/* size */
	1,mocmossub_m4_p,								/* ports */
	1,mocmossub_m4_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE,				/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* Polysilicon-1-Node */
static TECH_PORTS mocmossub_p1_p[] = {				/* ports */
	{mocmossub_pc_p1, "polysilicon-1", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN1, BOTIN1, RIGHTIN1, TOPIN1}};
static TECH_POLYGON mocmossub_p1_l[] = {			/* layers */
	{LPOLY1, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_p1 = {
	"Polysilicon-1-Node",NPOLY1N,NONODEPROTO,		/* name */
	K2,K2,											/* size */
	1,mocmossub_p1_p,								/* ports */
	1,mocmossub_p1_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE,				/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* Polysilicon-2-Node */
static TECH_PORTS mocmossub_p2_p[] = {				/* ports */
	{mocmossub_pc_p2, "polysilicon-2", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN1H, BOTIN1H, RIGHTIN1H, TOPIN1H}};
static TECH_POLYGON mocmossub_p2_l[] = {			/* layers */
	{LPOLY2, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_p2 = {
	"Polysilicon-2-Node",NPOLY2N,NONODEPROTO,		/* name */
	K3,K3,											/* size */
	1,mocmossub_p2_p,								/* ports */
	1,mocmossub_p2_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE,				/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* Active-Node */
static TECH_PORTS mocmossub_a_p[] = {				/* ports */
	{mocmossub_pc_a, "active", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN1H, BOTIN1H, RIGHTIN1H, TOPIN1H}};
static TECH_POLYGON mocmossub_a_l[] = {				/* layers */
	{LSACT, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_a = {
	"Active-Node",NACTIVEN,NONODEPROTO,				/* name */
	K3,K3,											/* size */
	1,mocmossub_a_p,								/* ports */
	1,mocmossub_a_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE,				/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* N-Active-Node (or P) */
static TECH_PORTS mocmossub_da_p[] = {				/* ports */
	{mocmossub_pc_a, "active", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN1H, BOTIN1H, RIGHTIN1H, TOPIN1H}};
static TECH_POLYGON mocmossub_da_l[] = {			/* layers */
	{LDACT, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_da = {
	"N-Active-Node",NDACTIVEN,NONODEPROTO,			/* name */
	K3,K3,											/* size */
	1,mocmossub_da_p,								/* ports */
	1,mocmossub_da_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE,				/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* P-Select-Node */
static TECH_PORTS mocmossub_sp_p[] = {				/* ports */
	{mocmossub_pc_null, "select", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTEDGE, BOTEDGE, RIGHTEDGE, TOPEDGE}};
static TECH_POLYGON mocmossub_sp_l[] = {			/* layers */
	{LSELECTP, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_sp = {
	"P-Select-Node",NSELECTPN,NONODEPROTO,			/* name */
	K4,K4,											/* size */
	1,mocmossub_sp_p,								/* ports */
	1,mocmossub_sp_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE,				/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* N-Select-Node */
static TECH_PORTS mocmossub_sn_p[] = {				/* ports */
	{mocmossub_pc_null, "select", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTEDGE, BOTEDGE, RIGHTEDGE, TOPEDGE}};
static TECH_POLYGON mocmossub_sn_l[] = {			/* layers */
	{LSELECTNP, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_sn = {
	"N-Select-Node",NSELECTNN,NONODEPROTO,			/* name */
	K4,K4,											/* size */
	1,mocmossub_sn_p,								/* ports */
	1,mocmossub_sn_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE,				/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* PolyCut-Node */
static TECH_PORTS mocmossub_gc_p[] = {				/* ports */
	{mocmossub_pc_null, "polycut", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTEDGE, BOTEDGE, RIGHTEDGE, TOPEDGE}};
static TECH_POLYGON mocmossub_gc_l[] = {			/* layers */
	{LPOLYCUT, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_gc = {
	"Poly-Cut-Node",NPCUTN,NONODEPROTO,				/* name */
	K2,K2,											/* size */
	1,mocmossub_gc_p,								/* ports */
	1,mocmossub_gc_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE,				/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* ActiveCut-Node */
static TECH_PORTS mocmossub_ac_p[] = {				/* ports */
	{mocmossub_pc_null, "activecut", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTEDGE, BOTEDGE, RIGHTEDGE, TOPEDGE}};
static TECH_POLYGON mocmossub_ac_l[] = {			/* layers */
	{LACTCUT, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_ac = {
	"Active-Cut-Node",NACUTN,NONODEPROTO,			/* name */
	K2,K2,											/* size */
	1,mocmossub_ac_p,								/* ports */
	1,mocmossub_ac_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE,				/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* Via-1-Node */
static TECH_PORTS mocmossub_v1_p[] = {				/* ports */
	{mocmossub_pc_null, "via-1", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTEDGE, BOTEDGE, RIGHTEDGE, TOPEDGE}};
static TECH_POLYGON mocmossub_v1_l[] = {			/* layers */
	{LVIA1, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_v1 = {
	"Via-1-Node",NVIA1N,NONODEPROTO,				/* name */
	K2,K2,											/* size */
	1,mocmossub_v1_p,								/* ports */
	1,mocmossub_v1_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE,				/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* Via-2-Node */
static TECH_PORTS mocmossub_v2_p[] = {				/* ports */
	{mocmossub_pc_null, "via-2", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTEDGE, BOTEDGE, RIGHTEDGE, TOPEDGE}};
static TECH_POLYGON mocmossub_v2_l[] = {			/* layers */
	{LVIA2, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_v2 = {
	"Via-2-Node",NVIA2N,NONODEPROTO,				/* name */
	K2,K2,											/* size */
	1,mocmossub_v2_p,								/* ports */
	1,mocmossub_v2_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE,				/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* Via-3-Node */
static TECH_PORTS mocmossub_v3_p[] = {				/* ports */
	{mocmossub_pc_null, "via-3", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTEDGE, BOTEDGE, RIGHTEDGE, TOPEDGE}};
static TECH_POLYGON mocmossub_v3_l[] = {			/* layers */
	{LVIA3, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_v3 = {
	"Via-3-Node",NVIA3N,NONODEPROTO,				/* name */
	K2,K2,											/* size */
	1,mocmossub_v3_p,								/* ports */
	1,mocmossub_v3_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE,				/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* P-Well-Node */
static TECH_PORTS mocmossub_wp_p[] = {				/* ports */
	{mocmossub_pc_pa, "well", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN3, BOTIN3, RIGHTIN3, TOPIN3}};
static TECH_POLYGON mocmossub_wp_l[] = {			/* layers */
	{LWELLP, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_wp = {
	"P-Well-Node",NWELLPN,NONODEPROTO,				/* name */
	K12,K12,										/* size */
	1,mocmossub_wp_p,								/* ports */
	1,mocmossub_wp_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE,				/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* N-Well-Node */
static TECH_PORTS mocmossub_wn_p[] = {				/* ports */
	{mocmossub_pc_pa, "well", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN3, BOTIN3, RIGHTIN3, TOPIN3}};
static TECH_POLYGON mocmossub_wn_l[] = {			/* layers */
	{LWELLNP, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_wn = {
	"N-Well-Node",NWELLNN,NONODEPROTO,				/* name */
	K12,K12,										/* size */
	1,mocmossub_wn_p,								/* ports */
	1,mocmossub_wn_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE,				/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* Passivation-node */
static TECH_PORTS mocmossub_o_p[] = {				/* ports */
	{mocmossub_pc_null, "passivation", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTEDGE, BOTEDGE, RIGHTEDGE, TOPEDGE}};
static TECH_POLYGON mocmossub_o_l[] = {				/* layers */
	{LPASS, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_o = {
	"Passivation-Node",NPASSN,NONODEPROTO,			/* name */
	K8,K8,											/* size */
	1,mocmossub_o_p,								/* ports */
	1,mocmossub_o_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE,				/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

/* Pad-Frame-node */
static TECH_PORTS mocmossub_pf_p[] = {				/* ports */
	{mocmossub_pc_null, "pad-frame", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTEDGE, BOTEDGE, RIGHTEDGE, TOPEDGE}};
static TECH_POLYGON mocmossub_pf_l[] = {			/* layers */
	{LFRAME, 0, 4, CLOSEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_pf = {
	"Pad-Frame-Node",NPADFRN,NONODEPROTO,			/* name */
	K8,K8,											/* size */
	1,mocmossub_pf_p,								/* ports */
	1,mocmossub_pf_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE,				/* userbits */
	0,0,0,0,0,0,0};									/* characteristics */

TECH_NODES *mocmossub_nodeprotos[NODEPROTOCOUNT+1] = {
	&mocmossub_pm1, &mocmossub_pm2,					/* metal 1/2 pin */
	&mocmossub_pm3, &mocmossub_pm4,					/* metal 3/4 pin */
	&mocmossub_pp1, &mocmossub_pp2,					/* polysilicon 1/2 pin */
	&mocmossub_psa, &mocmossub_pda,					/* P/N active pin */
	&mocmossub_pa,									/* active pin */
	&mocmossub_mpa, &mocmossub_mna,					/* metal 1 to P/N active contact */
	&mocmossub_mp1, &mocmossub_mp2,					/* metal 1 to polysilicon 1/2 contact */
	&mocmossub_mp12,								/* poly capacitor */
	&mocmossub_tpa, &mocmossub_tna,					/* P/N transistor */
	&mocmossub_m1m2, &mocmossub_m2m3, &mocmossub_m3m4,	/* via 1/2/3 */
	&mocmossub_psub, &mocmossub_nsub,				/* well / substrate contact */
	&mocmossub_m1, &mocmossub_m2,					/* metal 1/2 node */
	&mocmossub_m3, &mocmossub_m4,					/* metal 3/4 node */
	&mocmossub_p1, &mocmossub_p2,					/* polysilicon 1/2 node */
	&mocmossub_a, &mocmossub_da,					/* active N-Active node */
	&mocmossub_sp, &mocmossub_sn,					/* P/N select node */
	&mocmossub_gc, &mocmossub_ac,					/* poly cut / active cut */
	&mocmossub_v1, &mocmossub_v2, &mocmossub_v3,	/* via 1/2/3 node */
	&mocmossub_wp, &mocmossub_wn,					/* P/N well node */
	&mocmossub_o,									/* overglass node */
	&mocmossub_pf, ((TECH_NODES *)-1)};				/* pad frame node */

/* this table must correspond with the above table */
static INTBIG mocmossub_node_widoff[NODEPROTOCOUNT*4] = {
	0,0,0,0, 0,0,0,0, 0,0,0,0, 0,0,0,0,				/* metal 1/2/3/4 pin */
	0,0,0,0, 0,0,0,0,								/* polysilicon 1/2 pin */
	K6,K6,K6,K6, K6,K6,K6,K6,						/* P/N active pin */
	0,0,0,0,										/* active pin */
	K6,K6,K6,K6, K6,K6,K6,K6,						/* metal 1 to P/N active contact */
	0,0,0,0, 0,0,0,0,								/* metal 1 to polysilicon 1/2 contact */
	0,0,0,0,										/* poly capacitor */
	K6,K6,K9,K9, K6,K6,K9,K9,						/* P/N transistor */
	0,0,0,0, K1,K1,K1,K1, 0,0,0,0,					/* via 1/2/3 */
	K4,K4,K4,K4, K4,K4,K4,K4,						/* well / substrate contact */
	0,0,0,0, 0,0,0,0, 0,0,0,0, 0,0,0,0,				/* metal 1/2/3/4 node */
	0,0,0,0, 0,0,0,0,								/* polysilicon 1/2 node */
	0,0,0,0, 0,0,0,0,								/* active N-Active node */
	0,0,0,0, 0,0,0,0,								/* P/N select node */
	0,0,0,0, 0,0,0,0,								/* poly cut / active cut */
	0,0,0,0, 0,0,0,0,								/* via 1/2 node */
	0,0,0,0, 0,0,0,0,								/* P/N well node */
	0,0,0,0,										/* overglass node */
	0,0,0,0};										/* pad frame node */

/******************** SIMULATION VARIABLES ********************/

/* for SPICE simulation */
#define	MOCMOSSUB_MIN_RESIST	50.0f	/* minimum resistance consider */
#define	MOCMOSSUB_MIN_CAPAC	     0.04f	/* minimum capacitance consider */
float mocmossub_sim_spice_resistance[MAXLAYERS] = {  /* per square micron */
	0.03f, 0.03f, 0.03f, 0.03f,			/* metal 1/2/3/4 */
	50.0f, 50.0f,						/* poly 1/2 */
	10.0f, 10.0f,						/* P/N active */
	0.0, 0.0,							/* P/N select */
	0.0, 0.0,							/* P/N well */
	0.0, 0.0,							/* poly/act cut */
	0.0, 0.0, 0.0,						/* via1/2/3 */
	0.0,								/* overglass */
	0.0,								/* transistor */
	0.0,								/* poly cap */
	0.0,								/* P active well */
	0.0, 0.0, 0.0, 0.0,					/* pseudo metal 1/2/3/4 */
	0.0, 0.0,							/* pseudo poly 1/2 */
	0.0, 0.0,							/* pseudo P/N active */
	0.0, 0.0,							/* pseudo P/N select */
	0.0, 0.0,							/* pseudo P/N well */
	0.0};								/* pad frame */
float mocmossub_sim_spice_capacitance[MAXLAYERS] = { /* per square micron */
	0.03f, 0.03f, 0.03f, 0.03f,			/* metal 1/2/3/4 */
	0.04f, 0.04f,						/* poly 1/2 */
	0.1f, 0.1f,							/* P/N active */
	0.0, 0.0,							/* P/N select */
	0.0, 0.0,							/* P/N well */
	0.0, 0.0,							/* poly/act cut */
	0.0, 0.0, 0.0,						/* via1/2/3 */
	0.0,								/* overglass */
	0.0,								/* transistor */
	0.0,								/* poly cap */
	0.0,								/* P active well */
	0.0, 0.0, 0.0, 0.0,					/* pseudo metal 1/2/3/4 */
	0.0, 0.0,							/* pseudo poly 1/2 */
	0.0, 0.0,							/* pseudo P/N active */
	0.0, 0.0,							/* pseudo P/N select */
	0.0, 0.0,							/* pseudo P/N well */
	0.0};								/* pad frame */
char *mocmossub_sim_spice_header_level1[] = {
	"*CMOS/BULK-NWELL (PRELIMINARY PARAMETERS)",
	".OPTIONS NOMOD DEFL=3UM DEFW=3UM DEFAD=70P DEFAS=70P LIMPTS=1000",
	"+ITL4=1000 ITL5=0 RELTOL=0.01 ABSTOL=500PA VNTOL=500UV LVLTIM=2",
	"+LVLCOD=1",
	".MODEL N NMOS LEVEL=1",
	"+KP=60E-6 VTO=0.7 GAMMA=0.3 LAMBDA=0.05 PHI=0.6",
	"+LD=0.4E-6 TOX=40E-9 CGSO=2.0E-10 CGDO=2.0E-10 CJ=.2MF/M^2",
	".MODEL P PMOS LEVEL=1",
	"+KP=20E-6 VTO=0.7 GAMMA=0.4 LAMBDA=0.05 PHI=0.6",
	"+LD=0.6E-6 TOX=40E-9 CGSO=3.0E-10 CGDO=3.0E-10 CJ=.2MF/M^2",
	".MODEL DIFFCAP D CJO=.2MF/M^2",
	NOSTRING};
char *mocmossub_sim_spice_header_level2[] = {
	"* MOSIS 3u CMOS PARAMS",
	".OPTIONS NOMOD DEFL=2UM DEFW=6UM DEFAD=100P DEFAS=100P",
	"+LIMPTS=1000 ITL4=1000 ITL5=0 ABSTOL=500PA VNTOL=500UV",
	"* Note that ITL5=0 sets ITL5 to infinity",
	".MODEL N NMOS LEVEL=2 LD=0.3943U TOX=502E-10",
	"+NSUB=1.22416E+16 VTO=0.756 KP=4.224E-05 GAMMA=0.9241",
	"+PHI=0.6 UO=623.661 UEXP=8.328627E-02 UCRIT=54015.0",
	"+DELTA=5.218409E-03 VMAX=50072.2 XJ=0.4U LAMBDA=2.975321E-02",
	"+NFS=4.909947E+12 NEFF=1.001E-02 NSS=0.0 TPG=1.0",
	"+RSH=20.37 CGDO=3.1E-10 CGSO=3.1E-10",
	"+CJ=3.205E-04 MJ=0.4579 CJSW=4.62E-10 MJSW=0.2955 PB=0.7",
	".MODEL P PMOS LEVEL=2 LD=0.2875U TOX=502E-10",
	"+NSUB=1.715148E+15 VTO=-0.7045 KP=1.686E-05 GAMMA=0.3459",
	"+PHI=0.6 UO=248.933 UEXP=1.02652 UCRIT=182055.0",
	"+DELTA=1.0E-06 VMAX=100000.0 XJ=0.4U LAMBDA=1.25919E-02",
	"+NFS=1.0E+12 NEFF=1.001E-02 NSS=0.0 TPG=-1.0",
	"+RSH=79.10 CGDO=2.89E-10 CGSO=2.89E-10",
	"+CJ=1.319E-04 MJ=0.4125 CJSW=3.421E-10 MJSW=0.198 PB=0.66",
	".TEMP 25.0",
	NOSTRING};


/******************** VARIABLE AGGREGATION ********************/

TECH_VARIABLES mocmossub_variables[] =
{
	/* set general information about the technology */
	{"TECH_layer_names", (char *)mocmossub_layer_names, 0.0,
		VSTRING|VDONTSAVE|VISARRAY|(MAXLAYERS<<VLENGTHSH)},
	{"TECH_layer_function", (char *)mocmossub_layer_function, 0.0,
		VINTEGER|VDONTSAVE|VISARRAY|(MAXLAYERS<<VLENGTHSH)},
	{"TECH_arc_width_offset", (char *)mocmossub_arc_widoff, 0.0,
		VFRACT|VDONTSAVE|VISARRAY|(ARCPROTOCOUNT<<VLENGTHSH)},
	{"TECH_node_width_offset", (char *)mocmossub_node_widoff, 0.0,
		VFRACT|VDONTSAVE|VISARRAY|((NODEPROTOCOUNT*4)<<VLENGTHSH)},

	/* set information for the USER aid */
	{"USER_color_map", (char *)mocmossub_colmap, 0.0,
		VCHAR|VDONTSAVE|VISARRAY|((sizeof mocmossub_colmap)<<VLENGTHSH)},
	{"USER_layer_letters", (char *)mocmossub_layer_letters, 0.0,
		VSTRING|VDONTSAVE|VISARRAY|(MAXLAYERS<<VLENGTHSH)},

	/* set information for the I/O aid */
	{"IO_cif_layer_names", (char *)mocmossub_cif_layers, 0.0,
		VSTRING|VDONTSAVE|VISARRAY|(MAXLAYERS<<VLENGTHSH)},
	{"IO_gds_layer_numbers", (char *)mocmossub_gds_layers, 0.0,
		VINTEGER|VDONTSAVE|VISARRAY|(MAXLAYERS<<VLENGTHSH)},

	/* set information for the DRC aid */
	{"DRC_min_unconnected_distances", (char *)mocmossub_unconnectedtable, 0.0,
		VFRACT|VDONTSAVE|VISARRAY|
			(((sizeof mocmossub_unconnectedtable)/SIZEOFINTBIG)<<VLENGTHSH)},
	{"DRC_min_connected_distances", (char *)mocmossub_connectedtable, 0.0,
		VFRACT|VDONTSAVE|VISARRAY|
			(((sizeof mocmossub_connectedtable)/SIZEOFINTBIG)<<VLENGTHSH)},

	/* set information for the SIM aid (SPICE) */
	{"SIM_spice_min_resistance", 0, MOCMOSSUB_MIN_RESIST, VFLOAT|VDONTSAVE},
	{"SIM_spice_min_capacitance", 0, MOCMOSSUB_MIN_CAPAC, VFLOAT|VDONTSAVE},
	{"SIM_spice_resistance", (char *)mocmossub_sim_spice_resistance, 0.0,
		VFLOAT|VISARRAY|(MAXLAYERS<<VLENGTHSH)|VDONTSAVE},
	{"SIM_spice_capacitance", (char *)mocmossub_sim_spice_capacitance, 0.0,
		VFLOAT|VISARRAY|(MAXLAYERS<<VLENGTHSH)|VDONTSAVE},
	{"SIM_spice_header_level1", (char *)mocmossub_sim_spice_header_level1, 0.0,
		VSTRING|VDONTSAVE|VISARRAY},
	{"SIM_spice_header_level2", (char *)mocmossub_sim_spice_header_level2, 0.0,
		VSTRING|VDONTSAVE|VISARRAY},
	{NULL, NULL, 0.0, 0}
};

/******************** ROUTINES ********************/

INTSML mocmossub_initprocess(TECHNOLOGY *tech, INTSML pass)
{
	/* initialize the technology variable */
	if (pass == 0) mocmossub_tech = tech;
	return(0);
}

void mocmossub_setmode(INTSML count, char *par[])
{
	REGISTER INTSML l;
	REGISTER char *pp;
	REGISTER ARCPROTO *ap;
	REGISTER VARIABLE *var;

	if (count == 0)
	{
		if ((mocmossub_state&NWELL) == 0) ttyputmsg("Technology currently set for p-well"); else
			ttyputmsg("Technology currently set for n-well");
		return;
	}

	l = strlen(pp = par[0]);

	if (namesamen(pp, "3-metal-rules", l) == 0)
	{
		/* make metal-3 arcs be 5 wide */
		ap = mocmossub_tech->firstarcproto;	/* metal-1 */
		ap = ap->nextarcproto;			/* metal-2 */
		ap = ap->nextarcproto;			/* metal-3 */
		ap->nominalwidth = K5 * mocmossub_tech->deflambda / WHOLE;

		/* make metal-3 pin be 5x5 default, port 2.5 in from edges */
		mocmossub_pm3.creation->lowx = -K5 * mocmossub_tech->deflambda / WHOLE / 2;
		mocmossub_pm3.creation->highx = K5 * mocmossub_tech->deflambda / WHOLE / 2;
		mocmossub_pm3.creation->lowy = -K5 * mocmossub_tech->deflambda / WHOLE / 2;
		mocmossub_pm3.creation->highy = K5 * mocmossub_tech->deflambda / WHOLE / 2;
		mocmossub_pm3_p[0].lowxsum = H2;
		mocmossub_pm3_p[0].lowysum = H2;
		mocmossub_pm3_p[0].highxsum = -H2;
		mocmossub_pm3_p[0].highysum = -H2;

		/* make metal-2-metal-3 via have metal-3 that is full to edge */
		mocmossub_m2m3_l[1].points = mocmossub_fullbox;
		mocmossub_m2m3.f3 = K2;
		var = getval((INTBIG)mocmossub_tech, VTECHNOLOGY, VFRACT|VISARRAY, "TECH_node_width_offset");
		if (var != NOVARIABLE)
		{
			((INTBIG *)var->addr)[(NVIA2-1)*4] = 0;
			((INTBIG *)var->addr)[(NVIA2-1)*4+1] = 0;
			((INTBIG *)var->addr)[(NVIA2-1)*4+2] = 0;
			((INTBIG *)var->addr)[(NVIA2-1)*4+3] = 0;
		}

		mocmossub_state |= THREEMETAL;
		mocmossub_setdescript();
		ttyputmsgf("MOSIS CMOS Submicron technology uses 3-metal rules");
		return;
	}

	if (namesamen(pp, "4-metal-rules", l) == 0)
	{
		/* make metal-3 arcs be 3 wide */
		ap = mocmossub_tech->firstarcproto;	/* metal-1 */
		ap = ap->nextarcproto;			/* metal-2 */
		ap = ap->nextarcproto;			/* metal-3 */
		ap->nominalwidth = K3 * mocmossub_tech->deflambda / WHOLE;

		/* make metal-3 pin be 3x3 default, port 1.5 in from edges */
		mocmossub_pm3.creation->lowx = -K3 * mocmossub_tech->deflambda / WHOLE / 2;
		mocmossub_pm3.creation->highx = K3 * mocmossub_tech->deflambda / WHOLE / 2;
		mocmossub_pm3.creation->lowy = -K3 * mocmossub_tech->deflambda / WHOLE / 2;
		mocmossub_pm3.creation->highy = K3 * mocmossub_tech->deflambda / WHOLE / 2;
		mocmossub_pm3_p[0].lowxsum = H1;
		mocmossub_pm3_p[0].lowysum = H1;
		mocmossub_pm3_p[0].highxsum = -H1;
		mocmossub_pm3_p[0].highysum = -H1;

		/* make metal-2-metal-3 via have metal-3 that is inset by 1 to edge */
		mocmossub_m2m3_l[1].points = mocmossub_in1box;
		mocmossub_m2m3.f3 = K1;
		var = getval((INTBIG)mocmossub_tech, VTECHNOLOGY, VFRACT|VISARRAY, "TECH_node_width_offset");
		if (var != NOVARIABLE)
		{
			((INTBIG *)var->addr)[(NVIA2-1)*4] = K1;
			((INTBIG *)var->addr)[(NVIA2-1)*4+1] = K1;
			((INTBIG *)var->addr)[(NVIA2-1)*4+2] = K1;
			((INTBIG *)var->addr)[(NVIA2-1)*4+3] = K1;
		}

		mocmossub_state &= ~THREEMETAL;
		mocmossub_setdescript();
		ttyputmsgf("MOSIS CMOS Submicron technology uses 4-metal rules");
		return;
	}

	if (namesamen(pp, "p-well", l) == 0)		/* this is the default */
	{
		/* rename the P/N pin nodes */
		reallocstring(&mocmossub_psa.creation->primname, "P-Active-Pin",
			mocmossub_tech->cluster);
		reallocstring(&mocmossub_pda.creation->primname, "N-Active-Pin",
			mocmossub_tech->cluster);

		/* rename the P/N contact nodes */
		reallocstring(&mocmossub_mpa.creation->primname, "Metal-1-P-Active-Con",
			mocmossub_tech->cluster);
		reallocstring(&mocmossub_mna.creation->primname, "Metal-1-N-Active-Con",
			mocmossub_tech->cluster);

		/* rename the P/N pure-layer node */
		reallocstring(&mocmossub_da.creation->primname, "N-Active-Node",
			mocmossub_tech->cluster);

		/* set the P-transistor to be N */
		mocmossub_tna.creation->userbits &= ~NFUNCTION;
		mocmossub_tna.creation->userbits |= (NPTRANMOS<<NFUNCTIONSH);
		reallocstring(&mocmossub_tna.creation->primname, "N-Transistor",
			mocmossub_tech->cluster);

		/* set the N-transistor to be P */
		mocmossub_tpa.creation->userbits &= ~NFUNCTION;
		mocmossub_tpa.creation->userbits |= (NPTRAPMOS<<NFUNCTIONSH);
		reallocstring(&mocmossub_tpa.creation->primname, "P-Transistor",
			mocmossub_tech->cluster);

		/* set the Metal-1-Well-Contact to be of type WELL */
		mocmossub_psub.creation->userbits &= ~NFUNCTION;
		mocmossub_psub.creation->userbits |= (NPWELL<<NFUNCTIONSH);

		/* set the Metal-1-Substrate-Contact to be of type SUBSTRATE */
		mocmossub_nsub.creation->userbits &= ~NFUNCTION;
		mocmossub_nsub.creation->userbits |= (NPSUBSTRATE<<NFUNCTIONSH);

		/* set the P-active arc to P-well and N-active arc to N-well */
		ap = mocmossub_tech->firstarcproto;	/* metal-1 */
		ap = ap->nextarcproto;			/* metal-2 */
		ap = ap->nextarcproto;			/* metal-3 */
		ap = ap->nextarcproto;			/* metal-4 */
		ap = ap->nextarcproto;			/* polysilicon-1 */
		ap = ap->nextarcproto;			/* polysilicon-2 */
		ap = ap->nextarcproto;			/* P-active */
		ap->userbits &= ~AFUNCTION;
		ap->userbits |= (APDIFFP << AFUNCTIONSH);
		reallocstring(&ap->protoname, "P-Active", mocmossub_tech->cluster);
		ap = ap->nextarcproto;			/* N-active */
		ap->userbits &= ~AFUNCTION;
		ap->userbits |= (APDIFFN << AFUNCTIONSH);
		reallocstring(&ap->protoname, "N-Active", mocmossub_tech->cluster);

		/* change the layer functions */
		var = getval((INTBIG)mocmossub_tech, VTECHNOLOGY, VINTEGER|VISARRAY, "TECH_layer_function");
		if (var != NOVARIABLE)
		{
			/* change layers */
			((INTBIG *)var->addr)[LSACT] = LFDIFF|LFPTYPE|LFTRANS3;
			((INTBIG *)var->addr)[LDACT] = LFDIFF|LFNTYPE|LFTRANS3;
			((INTBIG *)var->addr)[LSACTWELL] = LFDIFF|LFPTYPE|LFTRANS3;

			/* change pseudo-layers */
			((INTBIG *)var->addr)[LSACTP] = LFDIFF|LFPTYPE|LFPSEUDO|LFTRANS3;
			((INTBIG *)var->addr)[LDACTP] = LFDIFF|LFNTYPE|LFPSEUDO|LFTRANS3;
		}

		/* change the layer names */
		var = getval((INTBIG)mocmossub_tech, VTECHNOLOGY, VSTRING|VISARRAY, "TECH_layer_names");
		if (var != NOVARIABLE)
		{
			/* change layers */
			((char **)var->addr)[LSACT] = "P-Active";
			((char **)var->addr)[LDACT] = "N-Active";
			((char **)var->addr)[LSACTWELL] = "P-Active-Well";
			((char **)var->addr)[LSACTP] = "Pseudo-P-Active";
			((char **)var->addr)[LDACTP] = "Pseudo-N-Active";
		}

		/* change the layer visibility */
		mocmossub_wp_lay.bits = mocmossub_pwp_lay.bits = LAYERO;	/* P Well visible */
		mocmossub_ssp_lay.bits = mocmossub_pssp_lay.bits = LAYERO;	/* P Select visible */
		mocmossub_wn_lay.bits = mocmossub_pwn_lay.bits = LAYERN;	/* N Well invisible */
		mocmossub_ssn_lay.bits = mocmossub_pssn_lay.bits = LAYERN;	/* N Select invisible */

		/* make P-active arc be P-Select, N-active arc be P-Well */
		mocmossub_al_pa[1].lay = LWELLNP;
		mocmossub_al_pa[2].lay = LSELECTP;
		mocmossub_al_na[1].lay = LWELLP;
		mocmossub_al_na[2].lay = LSELECTNP;

		/* make Metal-1-P-active-contact be P-Select, Metal-1-N-active-contact be P-Well */
		mocmossub_mpa_l[2].layernum = LWELLNP;
		mocmossub_mpa_l[3].layernum = LSELECTP;
		mocmossub_mna_l[2].layernum = LWELLP;
		mocmossub_mna_l[3].layernum = LSELECTNP;

		/* make P-Transistor be P-Select, N-Transistor be P-Well */
		mocmossub_tpa_l[2].basics.layernum = LWELLNP;
		mocmossub_tpa_l[3].basics.layernum = LSELECTP;
		mocmossub_tpaE_l[4].basics.layernum = LWELLNP;
		mocmossub_tpaE_l[5].basics.layernum = LSELECTP;
		mocmossub_tna_l[2].basics.layernum = LWELLP;
		mocmossub_tna_l[3].basics.layernum = LSELECTNP;
		mocmossub_tnaE_l[4].basics.layernum = LWELLP;
		mocmossub_tnaE_l[5].basics.layernum = LSELECTNP;

		/* make Metal-1-Substrate Contact be empty, Metal-1-Well Contact be P-Well/P-Select */
		mocmossub_nsub_l[2].layernum = LWELLNP;
		mocmossub_nsub_l[3].layernum = LSELECTNP;
		mocmossub_psub_l[2].layernum = LWELLP;
		mocmossub_psub_l[3].layernum = LSELECTP;

		/* make N-Well-Node be empty, P-Well-Node be P-Well */
		mocmossub_wn_l[0].layernum = LWELLNP;
		mocmossub_wp_l[0].layernum = LWELLP;

		/* make N-Select-Node be empty, P-Select-Node be P-Select */
		mocmossub_sn_l[0].layernum = LSELECTNP;
		mocmossub_sp_l[0].layernum = LSELECTP;

		mocmossub_state &= ~NWELL;
		mocmossub_setdescript();
		ttyputmsgf("MOSIS CMOS Submicron technology is P-well");
		return;
	}

	if (namesamen(pp, "n-well", l) == 0)
	{
		/* rename the P/N pin nodes */
		reallocstring(&mocmossub_psa.creation->primname, "N-Active-Pin",
			mocmossub_tech->cluster);
		reallocstring(&mocmossub_pda.creation->primname, "P-Active-Pin",
			mocmossub_tech->cluster);

		/* rename the P/N contact nodes */
		reallocstring(&mocmossub_mpa.creation->primname, "Metal-1-N-Active-Con",
			mocmossub_tech->cluster);
		reallocstring(&mocmossub_mna.creation->primname, "Metal-1-P-Active-Con",
			mocmossub_tech->cluster);

		/* rename the P/N pure-layer node */
		reallocstring(&mocmossub_da.creation->primname, "P-Active-Node",
			mocmossub_tech->cluster);

		/* set the N-transistor to be P */
		mocmossub_tna.creation->userbits &= ~NFUNCTION;
		mocmossub_tna.creation->userbits |= (NPTRAPMOS<<NFUNCTIONSH);
		reallocstring(&mocmossub_tna.creation->primname, "P-Transistor",
			mocmossub_tech->cluster);

		/* set the P-transistor to be N */
		mocmossub_tpa.creation->userbits &= ~NFUNCTION;
		mocmossub_tpa.creation->userbits |= (NPTRANMOS<<NFUNCTIONSH);
		reallocstring(&mocmossub_tpa.creation->primname, "N-Transistor",
			mocmossub_tech->cluster);

		/* set the Metal-1-Well-Contact to be of type SUBSTRATE */
		mocmossub_psub.creation->userbits &= ~NFUNCTION;
		mocmossub_psub.creation->userbits |= (NPSUBSTRATE<<NFUNCTIONSH);

		/* set the Metal-1-Substrate-Contact to be of type WELL */
		mocmossub_nsub.creation->userbits &= ~NFUNCTION;
		mocmossub_nsub.creation->userbits |= (NPWELL<<NFUNCTIONSH);

		/* set the P-active arc to N-well and N-active arc to P-well */
		ap = mocmossub_tech->firstarcproto;	/* metal-1 */
		ap = ap->nextarcproto;			/* metal-2 */
		ap = ap->nextarcproto;			/* metal-3 */
		ap = ap->nextarcproto;			/* metal-4 */
		ap = ap->nextarcproto;			/* polysilicon-1 */
		ap = ap->nextarcproto;			/* polysilicon-2 */
		ap = ap->nextarcproto;			/* P-active */
		ap->userbits &= ~AFUNCTION;
		ap->userbits |= (APDIFFN << AFUNCTIONSH);
		reallocstring(&ap->protoname, "N-Active", mocmossub_tech->cluster);
		ap = ap->nextarcproto;			/* N-active */
		ap->userbits &= ~AFUNCTION;
		ap->userbits |= (APDIFFP << AFUNCTIONSH);
		reallocstring(&ap->protoname, "P-Active", mocmossub_tech->cluster);

		/* change the layer functions */
		var = getval((INTBIG)mocmossub_tech, VTECHNOLOGY, VINTEGER|VISARRAY, "TECH_layer_function");
		if (var != NOVARIABLE)
		{
			/* change layers */
			((INTBIG *)var->addr)[LSACT] = LFDIFF|LFNTYPE|LFTRANS3;
			((INTBIG *)var->addr)[LDACT] = LFDIFF|LFPTYPE|LFTRANS3;
			((INTBIG *)var->addr)[LSACTWELL] = LFSUBSTRATE|LFNTYPE|LFTRANS3;

			/* change pseudo-layers */
			((INTBIG *)var->addr)[LSACTP] = LFDIFF|LFNTYPE|LFPSEUDO|LFTRANS3;
			((INTBIG *)var->addr)[LDACTP] = LFDIFF|LFPTYPE|LFPSEUDO|LFTRANS3;
		}

		/* change the layer names */
		var = getval((INTBIG)mocmossub_tech, VTECHNOLOGY, VSTRING|VISARRAY, "TECH_layer_names");
		if (var != NOVARIABLE)
		{
			/* change layers */
			((char **)var->addr)[LSACT] = "N-Active";
			((char **)var->addr)[LDACT] = "P-Active";
			((char **)var->addr)[LSACTWELL] = "N-Active-Well";
			((char **)var->addr)[LSACTP] = "Pseudo-N-Active";
			((char **)var->addr)[LDACTP] = "Pseudo-P-Active";
		}

		/* change the layer visibility */
		mocmossub_wp_lay.bits = mocmossub_pwp_lay.bits = LAYERN;	/* P Well invisible */
		mocmossub_ssp_lay.bits = mocmossub_pssp_lay.bits = LAYERN;	/* P Select invisible */
		mocmossub_wn_lay.bits = mocmossub_pwn_lay.bits = LAYERO;	/* N Well visible */
		mocmossub_ssn_lay.bits = mocmossub_pssn_lay.bits = LAYERO;	/* N Select visible */

		/* make P-active arc be N-Well, N-active arc be N-Select */
		mocmossub_al_pa[1].lay = LWELLN;
		mocmossub_al_pa[2].lay = LSELECTPP;
		mocmossub_al_na[1].lay = LWELLPP;
		mocmossub_al_na[2].lay = LSELECTN;

		/* make Metal-1-P-active-contact be N-Well, Metal-1-N-active-contact be N-Select */
		mocmossub_mpa_l[2].layernum = LWELLN;
		mocmossub_mpa_l[3].layernum = LSELECTPP;
		mocmossub_mna_l[2].layernum = LWELLPP;
		mocmossub_mna_l[3].layernum = LSELECTN;

		/* make P-Transistor be N-Well, N-Transistor be N-Select */
		mocmossub_tpa_l[2].basics.layernum = LWELLN;
		mocmossub_tpa_l[3].basics.layernum = LSELECTPP;
		mocmossub_tpaE_l[4].basics.layernum = LWELLN;
		mocmossub_tpaE_l[5].basics.layernum = LSELECTPP;
		mocmossub_tna_l[2].basics.layernum = LWELLPP;
		mocmossub_tna_l[3].basics.layernum = LSELECTN;
		mocmossub_tnaE_l[4].basics.layernum = LWELLPP;
		mocmossub_tnaE_l[5].basics.layernum = LSELECTN;

		/* make Metal-1-Substrate Contact be N-Well/N-Select, Metal-1-Well Contact be empty */
		mocmossub_nsub_l[2].layernum = LWELLN;
		mocmossub_nsub_l[3].layernum = LSELECTN;
		mocmossub_psub_l[2].layernum = LWELLPP;
		mocmossub_psub_l[3].layernum = LSELECTPP;

		/* make N-Well-Node be N-Well, P-Well-Node be empty */
		mocmossub_wn_l[0].layernum = LWELLN;
		mocmossub_wp_l[0].layernum = LWELLPP;

		/* make N-Select-Node be N-Select, P-Select-Node be empty */
		mocmossub_sn_l[0].layernum = LSELECTN;
		mocmossub_sp_l[0].layernum = LSELECTPP;

		mocmossub_state |= NWELL;
		mocmossub_setdescript();
		ttyputmsgf("MOSIS CMOS technology is N-well");
		return;
	}
	ttyputerr("Unknown MOSIS CMOS Submicron option: %s", pp);
}

void mocmossub_setdescript(void)
{
	switch (mocmossub_state)
	{
		case NWELL|THREEMETAL:
			mocmossub_tech->techdescript =
				"Complementary MOS (from MOSIS, Submicron, N-Well, triple metal, double poly)";
			break;

		case NWELL:
			mocmossub_tech->techdescript =
				"Complementary MOS (from MOSIS, Submicron, N-Well, quad metal, double poly)";
			break;

		case THREEMETAL:
			mocmossub_tech->techdescript =
				"Complementary MOS (from MOSIS, Submicron, P-Well, triple metal, double poly)";
			break;

		case 0:
			mocmossub_tech->techdescript =
				"Complementary MOS (from MOSIS, Submicron, P-Well, quad metal, double poly)";
			break;
	}
}

#endif  /* TECMOCOS - at top */
