/*
 * Electric(tm) VLSI Design System
 *
 * File: usr.c
 * User interface aid: main module
 * Written by: Steven M. Rubin, Electric Editor Incorporated
 *
 * Copyright (c) 1998 Electric Editor Incorporated.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Electric Editor Incorporated
 * 23470 Sunset Drive, Suite 108
 * Los Gatos, California 95033
 * support@electriceditor.com
 */

/* #define DECW 1 */	  /* use Decwindows */

#include "global.h"
#include "egraphics.h"
#include "tech.h"
#include "usr.h"
#include "usrdiacom.h"
#include "tecgen.h"
#include <signal.h>		/* for interrupt handling */
#include <setjmp.h>		/* for nonlocal goto's */

#ifdef WIN32
  typedef void (*SIG_PF)(int);
#  define SIGNALCAST SIG_PF
#endif
#ifdef ONUNIX
  typedef void (*SIG_PF)();
#  define SIGNALCAST SIG_PF
#endif
#ifdef MACOS
#  define SIGNALCAST _Sigfun*
#endif

AIDENTRY *us_aid;					/* the USER aid object */
INTSML    gra_fgd=1, gra_bgd=0;		/* global monochrome colors, modified in "graphx11.c" */
INTSML    us_ignore_cadrc;			/* set when user wants to ignore the cadrc file */
jmp_buf   us_cmdone;				/* when command is done, user slice quits */
char     *us_firstlibrary;			/* library to read upon startup */
char     *us_firstmacrofile;		/* macro file to read upon startup */
INTSML    us_lastcol;				/* width of status terminal */
INTSML    us_longcount;				/* number of long commands */
INTSML  (*us_displayroutine)(POLYGON*, WINDOW*); /* routine for graphics display */
INTSML    us_currenteditor;			/* the current editor (index to "us_editortable") */
INTBIG    us_layer_letters;			/* cached key for "USER_layer_letters" */
INTBIG    us_highlighted;			/* cached key for "USER_highlighted" */
INTBIG    us_highlightstack;		/* cached key for "USER_highlightstack" */
INTBIG    us_facet_stack;			/* cached key for "USER_facet_stack" */
INTBIG    us_binding_keys;			/* cached key for "USER_binding_keys" */
INTBIG    us_binding_buttons;		/* cached key for "USER_binding_buttons" */
INTBIG    us_binding_menu;			/* cached key for "USER_binding_menu" */
INTBIG    us_current_node;			/* cached key for "USER_current_node" */
INTBIG    us_current_arc;			/* cached key for "USER_current_arc" */
INTBIG    us_placement_angle;		/* cached key for "USER_placement_angle" */
INTBIG    us_alignment_obj;			/* cached key for "USER_alignment_obj" */
INTBIG    us_alignment_edge;		/* cached key for "USER_alignment_edge" */
INTBIG    us_arcstyle;				/* cached key for "USER_arc_style" */
INTBIG    us_current_technology;	/* cached key for "USER_current_technology" */
INTBIG    us_current_window;		/* cached key for "USER_current_window" */
INTBIG    us_current_constraint;	/* cached key for "USER_current_constraint" */
INTBIG    us_colormap_red;			/* cached key for "USER_colormap_red" */
INTBIG    us_colormap_green;		/* cached key for "USER_colormap_green" */
INTBIG    us_colormap_blue;			/* cached key for "USER_colormap_blue" */
INTBIG    us_menu_position;			/* cached key for "USER_menu_position" */
INTBIG    us_menu_x;				/* cached key for "USER_menu_x" */
INTBIG    us_menu_y;				/* cached key for "USER_menu_y" */
INTBIG    us_macrorunning;			/* cached key for "USER_macrorunning" */
INTBIG    us_macrobuilding;			/* cached key for "USER_macrobuilding" */
INTBIG    us_spyglassfactor;		/* cached key for "USER_spyglassfactor" */
INTBIG    us_text_editor;			/* cached key for "USER_text_editor" */

/* internal state */
INTBIG     us_state;				/* miscellaneous state of user interface */
EDITOR    *us_editorfree;			/* top of free list of editors */

STATUSFIELD *us_statusalign;		/* current node alignment */
STATUSFIELD *us_statusangle;		/* current placement angle */
STATUSFIELD *us_statusarc;			/* current arc prototype */
STATUSFIELD *us_statusfacet;		/* current facet */
STATUSFIELD *us_statusfacetsize;	/* current facet size */
STATUSFIELD *us_statusgridsize;		/* current grid size */
STATUSFIELD *us_statuslambda;		/* current lambda */
STATUSFIELD *us_statusnetwork;		/* current network */
STATUSFIELD *us_statusnode;			/* current node prototype */
STATUSFIELD *us_statustechnology;	/* current technology */
STATUSFIELD *us_statusxpos;			/* current x position */
STATUSFIELD *us_statusypos;			/* current y position */
STATUSFIELD *us_statusproject;		/* current project */
STATUSFIELD *us_statusroot;			/* current root */
STATUSFIELD *us_statuspart;			/* current part */
STATUSFIELD *us_statuspackage;		/* current package */
STATUSFIELD *us_statusselection;	/* current selection */

/* user interface state */
NODEPROTO *us_curnodeproto;			/* current nodeproto */
ARCPROTO  *us_curarcproto;			/* current arcproto */
INTBIG     us_alignment;			/* current alignment in lambda */
INTBIG     us_edgealignment;		/* current edge alignment in lambda */
void      *us_menuframe;			/* window frame on which menu resides */
INTSML     us_menux, us_menuy;		/* number of menu elements on the screen */
INTSML     us_menupos;				/* position: 0:top 1:bottom 2:left 3:right */
INTSML     us_menuxsz,us_menuysz;	/* size of menu elements */
INTSML     us_menuhnx,us_menuhny;	/* highlighted nodeinst menu entry */
INTSML     us_menuhax,us_menuhay;	/* highlighted arcinst menu entry */
INTSML     us_menulx,us_menuhx;		/* X: low and high of menu area */
INTSML     us_menuly,us_menuhy;		/* Y: low and high of menu area */
FILE      *us_logrecord;			/* logging record file */
FILE      *us_logplay;				/* logging playback file */
INTSML     us_logflushfreq;			/* session logging flush frequency */
char      *us_desiredlibrary;		/* library to be read */

MACROPACK *us_macropacktop;			/* top of list of defined macro packages */
MACROPACK *us_curmacropack;			/* current macro package */

USERCOM   *us_usercomfree;			/* list of free user commands */
USERCOM   *us_lastcom;				/* last command and arguments */

POPUPMENU *us_firstpopupmenu;		/* list of existing pop-up menus */
INTSML     us_menubarsize;			/* actual size (0 if no menu bar) */
INTSML     us_pulldownmenucount;	/* number of pulldown menus */
INTSML    *us_pulldownmenupos;		/* position of pulldown menus */
POPUPMENU **us_pulldowns;			/* the current pulldown menus */

/* prototypes for local routines */
INTSML us_do2init(INTBIG*, char*[]);
INTSML us_do3init(void);
void us_onint(void);
void us_ontablet(INTSML, INTSML, INTSML);
void us_oncommand(void);
void us_causeofslice(USERCOM*);
void us_setcommand(char*, USERCOM*, INTSML, NODEPROTO*, ARCPROTO*, char*, POPUPMENU*, INTSML);
INTSML us_options(INTBIG*, char*[]);
INTSML us_docadrc(char*);
void us_findcadrc(void);

/******************** CONTROL ********************/

/* initialize the user interface aid */
void us_init(INTBIG *argc, char *argv[], AIDENTRY *thisaid)
{
	if (thisaid == 0)
	{
		/* pass 3 */
		if (us_do3init() != 0)
			error("No memory to run the user interface");
	} else if (thisaid == NOAID)
	{
		/* pass 2 */
		if (us_do2init(argc, argv) != 0)
			error("No memory to run the user interface");
	} else
	{
		/* pass 1 */
		us_aid = thisaid;
	}
}

/* pass 2 of initialization */
INTSML us_do2init(INTBIG *argc, char *argv[])
{
	char *newpar[3];
	REGISTER INTSML i;
	INTSML swid, shei;
	REGISTER TECHNOLOGY *tech;
	extern GRAPHICS us_gbox;

	/* initialize lists */
	us_editorfree = NOEDITOR;		/* no free editors */
	us_usercomfree = NOUSERCOM;		/* no free command modules */

	/* define the default macro called "macro" */
	newpar[0] = newpar[1] = newpar[2] = "";
	nextvarchangequiet();
	(void)setvalkey((INTBIG)us_aid, VAID, makekey("USER_macro_macro"),
		(INTBIG)newpar, VSTRING|VISARRAY|(3<<VLENGTHSH)|VDONTSAVE);

	/* basic initialization */
	us_logplay = NULL;					/* no session playback */
	us_logrecord = NULL;				/* no session recording */
	el_curwindow = NOWINDOW;
	us_logflushfreq = 10;				/* flush session recording every 10 */
	us_lastcom = NOUSERCOM;				/* no last command issued */
	us_macropacktop = NOMACROPACK;		/* no defined macro packages */
	us_curmacropack = NOMACROPACK;		/* no current macro package */
	us_menuhnx = us_menuhax = -1;		/* no menu highlighting */
	us_menupos = 0;						/* menus on the top */
	us_firstpopupmenu = NOPOPUPMENU;	/* no popup menus yet */
	us_menubarsize = 0;					/* no menubar */
	us_pulldownmenucount = 0;			/* no pulldown menus in menubar */
	us_state = 0;						/* all is well */
	us_currenteditor = 0;				/* use the first editor */
	us_aid->aidstate |= INTERACTIVE;	/* use interactive cursor commands */
	us_desiredlibrary = 0;				/* no library to read */
	us_layer_letters = makekey("USER_layer_letters");
	us_highlighted = makekey("USER_highlighted");
	us_highlightstack = makekey("USER_highlightstack");
	us_facet_stack = makekey("USER_facet_stack");
	us_binding_keys = makekey("USER_binding_keys");
	us_binding_buttons = makekey("USER_binding_buttons");
	us_binding_menu = makekey("USER_binding_menu");
	us_current_node = makekey("USER_current_node");
	us_current_arc = makekey("USER_current_arc");
	us_placement_angle = makekey("USER_placement_angle");
	us_alignment_obj = makekey("USER_alignment_obj");
	us_alignment_edge = makekey("USER_alignment_edge");
	us_arcstyle = makekey("USER_arc_style");
	us_current_technology = makekey("USER_current_technology");
	us_current_window = makekey("USER_current_window");
	us_current_constraint = makekey("USER_current_constraint");
	us_colormap_red = makekey("USER_colormap_red");
	us_colormap_green = makekey("USER_colormap_green");
	us_colormap_blue = makekey("USER_colormap_blue");
	us_menu_position = makekey("USER_menu_position");
	us_menu_x = makekey("USER_menu_x");
	us_menu_y = makekey("USER_menu_y");
	us_macrorunning = makekey("USER_macrorunning");
	us_macrobuilding = makekey("USER_macrobuilding");
	us_spyglassfactor = makekey("USER_spyglassfactor");
	us_text_editor = makekey("USER_text_editor");

	/* initially ignore all graphics */
	us_displayroutine = us_nulldisplayroutine;

	/* set opaque graphics data */
	for(tech = el_technologies; tech != NOTECHNOLOGY; tech = tech->nexttechnology)
		us_figuretechopaque(tech);

	/* register technology-caching routines */
	registertechnologycache(us_initlayerletters);

	/* count the number of commands */
	us_longcount = 0;
	for(i=0; us_lcommand[i].name != 0; i++) us_longcount++;

	/* get switches to the program */
	us_firstlibrary = 0;
	us_firstmacrofile = 0;
	us_setdisplay("display", argc, argv);
	if (us_options(argc, argv) != 0) return(1);

	/* initialize status output and graphics */
	if (us_initgraphics(1)) exit(1);
	us_getcolormap(el_curtech, COLORSDEFAULT, 0);

	/* establish the colors to use for peripheral graphics */
	if (el_maplength == 2)      /* monochrome; us_initgraphics should have set gra_fgd, etc. */
	{
		el_colfacettxt = gra_fgd;
		el_colfacet = gra_fgd;
		el_colwinbor = gra_fgd;
		el_colhwinbor = gra_bgd;
		el_colmenbor = gra_fgd;
		el_colhmenbor = gra_bgd;
		el_colmentxt = gra_fgd;
		el_colmengly = gra_fgd;
		el_colcursor = gra_fgd;         /* this doesn't mean much on X versions */
	} else
	{
		if (el_maplength < 256)
		{
			us_gbox.bits = LAYERA;
			us_gbox.col = BLACK;
		} else
		{
			us_gbox.bits = LAYERG;
			us_gbox.col = GRID;
		}
		el_colfacettxt = FACETTXT;
		el_colfacet = FACETOUT;
		el_colwinbor = WINBOR;
		el_colhwinbor = HWINBOR;
		el_colmenbor = MENBOR;
		el_colhmenbor = HMENBOR;
		el_colmentxt = MENTXT;
		el_colmengly = MENGLY;
	}

	tty_init();
	us_lastcol = 80;
	us_initstatus();

	/* catch interrupts */
	(void)signal(SIGINT, (SIGNALCAST)us_onint);

	/* create the window frame that has the palette */
	us_menuframe = us_newwindowframe(1);

	/* create the initial window structure */
	el_topwindow = NOWINDOW;
	el_curwindow = newewindow("entire", NOWINDOW, 1);
	if (el_curwindow == NOWINDOW) return(1);
	nextvarchangequiet();
	(void)setvalkey((INTBIG)us_aid, VAID, us_current_window,
		(INTBIG)el_curwindow, VWINDOW|VDONTSAVE);
	el_curwindow->buttonhandler = DEFAULTBUTTONHANDLER;
	el_curwindow->charhandler = DEFAULTCHARHANDLER;
	el_curwindow->changehandler = DEFAULTCHANGEHANDLER;
	el_curwindow->termhandler = DEFAULTTERMHANDLER;
	el_curwindow->redisphandler = DEFAULTREDISPHANDLER;

	/* set shadows */
	nextvarchangequiet();
	(void)setvalkey((INTBIG)us_aid, VAID, us_current_constraint,
		(INTBIG)el_curconstraint, VCONSTRAINT|VDONTSAVE);
	nextvarchangequiet();
	(void)setvalkey((INTBIG)us_aid, VAID, us_spyglassfactor, 0, VINTEGER|VDONTSAVE);
	us_edgealignment = 0;
	nextvarchangequiet();
	(void)setvalkey((INTBIG)us_aid, VAID, us_alignment_edge,
		us_edgealignment, VINTEGER|VDONTSAVE);

	/* if the screen is seriously wide, put menus on the left */
	us_getwindowsize(el_curwindow->frame, &swid, &shei);
	if (swid*2 > shei*3) us_menupos = 2;
	return(0);
}

/* pass 3 of initialization */
INTSML us_do3init(void)
{
	char *libn, *newpar[3];
	extern AIDENTRY *io_aid;

	/* read an initial library if it was specified in program parameters */
	if (us_firstlibrary != 0)
	{
		if (reallocstring(&el_curlib->libfile, us_firstlibrary, el_curlib->cluster) != 0) return(1);
		libn = us_skippath(us_firstlibrary);
		if (reallocstring(&el_curlib->libname, libn, el_curlib->cluster) != 0) return(1);
		if (askaid(io_aid, "read", (INTBIG)el_curlib, (INTBIG)"binary") != 0)
		{
			ttyputerr("Could not read %s", el_curlib->libfile);
			el_curlib->curnodeproto = NONODEPROTO;
		} else if (el_curlib->curnodeproto != NONODEPROTO)
			el_curtech = whattech(el_curlib->curnodeproto);
	} else el_curlib->curnodeproto = NONODEPROTO;

	/* set the user's shadow of the current technology */
	(void)setvalkey((INTBIG)us_aid, VAID, us_current_technology,
		(INTBIG)el_curtech, VTECHNOLOGY|VDONTSAVE);
	us_getcolormap(el_curtech, COLORSDEFAULT, 0);
	el_curwindow->gridx = el_curwindow->gridy = el_curtech->deflambda;
	us_alignment = el_curtech->deflambda;
	nextvarchangequiet();
	(void)setvalkey((INTBIG)us_aid, VAID, us_alignment_obj, us_alignment, VINTEGER|VDONTSAVE);

	us_curarcproto = el_curtech->firstarcproto;
	us_curnodeproto = el_curtech->firstnodeproto;
	if (us_initialbinding() != 0) return(1);	/* default command binding */

	/* initial display on status terminal */
	us_redostatus(0);

	/* set application environment */
#if EWORLD
	nextvarchangequiet();
	(void)setvalkey((INTBIG)us_aid, VAID, makekey("USER_application"),
		(INTBIG)"EWORLD", VSTRING|VDONTSAVE);
#endif
#if EGNU
	nextvarchangequiet();
	(void)setvalkey((INTBIG)us_aid, VAID, makekey("USER_application"),
		(INTBIG)"EGNU", VSTRING|VDONTSAVE);
#endif
#if EGNUL
	nextvarchangequiet();
	(void)setvalkey((INTBIG)us_aid, VAID, makekey("USER_application"),
		(INTBIG)"EGNUL", VSTRING|VDONTSAVE);
#endif
#if EVE
	nextvarchangequiet();
	(void)setvalkey((INTBIG)us_aid, VAID, makekey("USER_application"),
		(INTBIG)"EVE", VSTRING|VDONTSAVE);
#endif
#if ELM
	nextvarchangequiet();
	(void)setvalkey((INTBIG)us_aid, VAID, makekey("USER_application"),
		(INTBIG)"ELM", VSTRING|VDONTSAVE);
#endif
#if ELF
	nextvarchangequiet();
	(void)setvalkey((INTBIG)us_aid, VAID, makekey("USER_application"),
		(INTBIG)"ELF", VSTRING|VDONTSAVE);
#endif
#if ELFA
	nextvarchangequiet();
	(void)setvalkey((INTBIG)us_aid, VAID, makekey("USER_application"),
		(INTBIG)"ELFA", VSTRING|VDONTSAVE);
#endif

	/* read "cadrc" files */
	if (us_ignore_cadrc == 0) us_findcadrc();
	if (us_firstmacrofile != 0)
	{
		if (us_docadrc(us_firstmacrofile) != 0)
			ttyputerr("Cannot find startup file '%s'", us_firstmacrofile);
	}

	/* now allow graphics to reach the screen */
	us_displayroutine = us_showpoly;

	/* now draw everything */
	us_drawmenu(1, 0);	/* draw basic screen layout */

	securitycheck();	/* make sure the user can run the program */

#if !defined(MACOS)
	if ((us_aid->aidstate&USEDIALOGS) != 0)
	{
		(void)ttydataready();
		tty_aboutdlog("Protected by License");
		(void)ttydataready();
	} else
		ttyputmsg("Electric is Proprietary Software from Electric Editor Incorporated");
#endif

	/* if there is a current facet in the library, edit it */
	if (el_curlib->curnodeproto != NONODEPROTO)
	{
		newpar[0] = "editfacet";
		newpar[1] = describenodeproto(el_curlib->curnodeproto);
		(void)tellaid(us_aid, 2, newpar);
	}
	return(0);
}

/* handle interrupts */
RETSIGTYPE us_onint(void)
{
	(void)signal(SIGINT, (SIGNALCAST)us_onint);
	el_pleasestop = 1;
	ttyputerr("Interrupted...");
}

/* terminate the user interface aid */
void us_done(void)
{
	tty_close();
	us_logfinishrecord();
	us_termgraphics(1);
}

/* set the nature of the user interface aid */
INTSML us_set(INTSML count, char *par[])
{
	REGISTER USERCOM *com;
	REGISTER INTSML i;

	if (count == 0) return(0);

	(void)initinfstr();
	for(i=0; i<count; i++)
	{
		(void)addstringtoinfstr(par[i]);
		(void)addtoinfstr(' ');
	}
	com = us_makecommand(returninfstr());
	us_execute(com, 0, 1, 0);
	us_freeusercom(com);
	return(0);
}

/*
 * make requests of the user interface aid:
 *
 ***************** CONTROL OF HIGHLIGHTING ****************
 *  "show-object" TAKES: the GEOM module to highlight
 *  "show-port" TAKES: the GEOM module and PORTPROTO to highlight
 *  "show-area" TAKES: the lowx, highx, lowy, highy of an area
 *  "show-line" TAKES: the x1, y1, x2, y2 of a line
 *  "show-multiple" TAKES: string of GEOM addresses separated by slashes
 *  "clear" clears all highlighting
 *  "down-stack" saves the currently highlighted objects on a stack
 *  "up-stack" restores the stacked highlighted objects
 *
 ***************** INFORMATION ABOUT HIGHLIGHTED OBJECTS ****************
 *  "get-node" RETURNS: the currently highlighted NODEINST
 *  "get-port" RETURNS: the currently highlighted PORTPROTO
 *  "get-arc" RETURNS: the currently highlighted ARCINST
 *  "get-object" RETURNS: the currently highlighted GEOM
 *  "get-all-nodes" RETURNS: a list of GEOMs
 *  "get-all-arcs" RETURNS: a list of GEOMs
 *  "get-all-objects" RETURNS: a list of GEOMs
 *  "get-highlighted-area" TAKES: reference lx, hx, ly, hy bounds
 *
 ***************** WINDOW CONTROL ****************
 *  "window-new" RETURNS: newly created window
 *  "window-horiz-new" RETURNS: newly created window (prefers horizontal split)
 *  "window-vert-new" RETURNS: newly created window (prefers vertical split)
 *  "display-to-routine" TAKES: routine to call with polygons
 *  "display-highlighted-to-routine" TAKES: routine to call with polygons
 *  "flush-changes" forces display changes to be visible
 *
 ***************** EDITOR CONTROL ****************
 *  "edit-starteditor" TAKES: window, header, &chars, &lines RETURNS: zero: ok
 *  "edit-totallines" TAKES: window RETURNS: number of lines of text
 *  "edit-getline" TAKES: window and line index RETURNS: string
 *  "edit-addline" TAKES: window and line index and string to insert
 *  "edit-replaceline" TAKES: window and line index and new string
 *  "edit-deleteline" TAKES: window and line index to delete
 *  "edit-highlightline" TAKES: window and line index to highlight
 *  "edit-suspendgraphics" TAKES: window in which to suspends display changes
 *  "edit-resumegraphics" TAKES: window in which to resumes display of changes
 *  "edit-describe" RETURNS: name of editor
 *  "edit-readfile" TAKES: window and file name
 *  "edit-writefile" TAKES: window and file name
 *
 ***************** MISCELLANEOUS ****************
 *  "make-icon" TAKES: portlist, icon name, facet name, technology, library
 *              RETURNS: facet
 */
INTBIG us_request(char *command, va_list ap)
{
	HIGHLIGHT newhigh;
	REGISTER GEOM *from;
	REGISTER PORTPROTO *pp;
	REGISTER VARIABLE *var;
	REGISTER NODEPROTO *np;
	REGISTER INTSML len, i;
	REGISTER INTSML (*curdisplay)(POLYGON*, WINDOW*);
	REGISTER INTBIG arg1, arg2, arg3, arg4, arg5;
	REGISTER GEOM **list;
	REGISTER WINDOW *w;

	if (namesamen(command, "get-", 4) == 0)
	{
		if (namesame(&command[4], "node") == 0)
			return((INTBIG)us_getobject(OBJNODEINST, 0));
		if (namesame(&command[4], "arc") == 0)
			return((INTBIG)us_getobject(OBJARCINST, 0));
		if (namesame(&command[4], "object") == 0)
		{
			var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_highlighted);
			if (var == NOVARIABLE) return(-1);
			len = getlength(var);
			if (len > 1) return(-1);
			(void)us_makehighlight(((char **)var->addr)[0], &newhigh);
			if ((newhigh.status&HIGHTYPE) != HIGHFROM) return(-1);
			return((INTBIG)newhigh.fromgeom);
		}
		if (namesame(&command[4], "port") == 0)
		{
			var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_highlighted);
			if (var == NOVARIABLE) return(-1);
			len = getlength(var);
			if (len > 1) return(-1);
			(void)us_makehighlight(((char **)var->addr)[0], &newhigh);
			if ((newhigh.status&HIGHTYPE) != HIGHFROM || newhigh.fromgeom->entrytype != OBJNODEINST)
				return(-1);
			return((INTBIG)newhigh.fromport);
		}
		if (namesame(&command[4], "all-nodes") == 0)
			return((INTBIG)us_gethighlighted(OBJNODEINST));
		if (namesame(&command[4], "all-arcs") == 0)
			return((INTBIG)us_gethighlighted(OBJARCINST));
		if (namesame(&command[4], "all-objects") == 0)
			return((INTBIG)us_gethighlighted(OBJNODEINST|OBJARCINST));
		if (namesame(&command[4], "highlighted-area") == 0)
		{
			/* get the arguments */
			arg1 = va_arg(ap, INTBIG);
			arg2 = va_arg(ap, INTBIG);
			arg3 = va_arg(ap, INTBIG);
			arg4 = va_arg(ap, INTBIG);
			return(us_getareabounds((INTBIG *)arg1, (INTBIG *)arg2, (INTBIG *)arg3, (INTBIG *)arg4));
		}
	}

	if (namesamen(command, "edit-", 5) == 0)
	{
		if (namesame(&command[5], "getline") == 0)
		{
			/* get the arguments */
			arg1 = va_arg(ap, INTBIG);
			arg2 = va_arg(ap, INTBIG);
			return((INTBIG)us_getline((WINDOW *)arg1, arg2));
		}
		if (namesame(&command[5], "totallines") == 0)
		{
			/* get the arguments */
			arg1 = va_arg(ap, INTBIG);
			return(us_totallines((WINDOW *)arg1));
		}
		if (namesame(&command[5], "addline") == 0)
		{
			/* get the arguments */
			arg1 = va_arg(ap, INTBIG);
			arg2 = va_arg(ap, INTBIG);
			arg3 = va_arg(ap, INTBIG);
			us_addline((WINDOW *)arg1, arg2, (char *)arg3);
			return(0);
		}
		if (namesame(&command[5], "replaceline") == 0)
		{
			/* get the arguments */
			arg1 = va_arg(ap, INTBIG);
			arg2 = va_arg(ap, INTBIG);
			arg3 = va_arg(ap, INTBIG);
			us_replaceline((WINDOW *)arg1, arg2, (char *)arg3);
			return(0);
		}
		if (namesame(&command[5], "deleteline") == 0)
		{
			/* get the arguments */
			arg1 = va_arg(ap, INTBIG);
			arg2 = va_arg(ap, INTBIG);
			us_deleteline((WINDOW *)arg1, arg2);
			return(0);
		}
		if (namesame(&command[5], "highlightline") == 0)
		{
			/* get the arguments */
			arg1 = va_arg(ap, INTBIG);
			arg2 = va_arg(ap, INTBIG);
			us_highlightline((WINDOW *)arg1, arg2);
			return(0);
		}
		if (namesame(&command[5], "starteditor") == 0)
		{
			/* get the arguments */
			arg1 = va_arg(ap, INTBIG);
			arg2 = va_arg(ap, INTBIG);
			arg3 = va_arg(ap, INTBIG);
			arg4 = va_arg(ap, INTBIG);
			if (us_makeeditor((WINDOW *)arg1, (char *)arg2, (INTSML *)arg3,
				(INTSML *)arg4) == NOWINDOW) return(-1);
			return(0);
		}
		if (namesame(&command[5], "suspendgraphics") == 0)
		{
			/* get the arguments */
			arg1 = va_arg(ap, INTBIG);
			us_suspendgraphics((WINDOW *)arg1);
			return(0);
		}
		if (namesame(&command[5], "resumegraphics") == 0)
		{
			/* get the arguments */
			arg1 = va_arg(ap, INTBIG);
			us_resumegraphics((WINDOW *)arg1);
			return(0);
		}
		if (namesame(&command[5], "describe") == 0)
		{
			/* get the arguments */
			arg1 = va_arg(ap, INTBIG);
			us_describeeditor((char **)arg1);
			return(0);
		}
		if (namesame(&command[5], "readfile") == 0)
		{
			/* get the arguments */
			arg1 = va_arg(ap, INTBIG);
			arg2 = va_arg(ap, INTBIG);
			us_readtextfile((WINDOW *)arg1, (char *)arg2);
			return(0);
		}
		if (namesame(&command[5], "writefile") == 0)
		{
			/* get the arguments */
			arg1 = va_arg(ap, INTBIG);
			arg2 = va_arg(ap, INTBIG);
			us_writetextfile((WINDOW *)arg1, (char *)arg2);
			return(0);
		}
	}

	if (namesamen(command, "show-", 5) == 0)
	{
		if (namesame(&command[5], "object") == 0)
		{
			/* get the arguments */
			arg1 = va_arg(ap, INTBIG);

			/* fill in the highlight information */
			from = (GEOM *)arg1;
			us_ensurewindow(from);
			newhigh.status = HIGHFROM;
			newhigh.facet = geomparent(from);
			newhigh.fromgeom = from;
			newhigh.fromport = NOPORTPROTO;
			newhigh.frompoint = 0;
			(void)us_addhighlight(&newhigh);
			us_showallhighlight();
			return(0);
		}
		if (namesame(&command[5], "port") == 0)
		{
			/* get the arguments */
			arg1 = va_arg(ap, INTBIG);
			arg2 = va_arg(ap, INTBIG);

			/* fill in the highlight information */
			from = (GEOM *)arg1;
			pp = (PORTPROTO *)arg2;
			us_ensurewindow(from);
			newhigh.status = HIGHFROM;
			newhigh.facet = geomparent(from);
			newhigh.fromgeom = from;
			newhigh.fromport = pp;
			newhigh.frompoint = 0;
			(void)us_addhighlight(&newhigh);
			us_showallhighlight();
			return(0);
		}
		if (namesame(&command[5], "line") == 0)
		{
			np = getcurfacet();
			if (np == NONODEPROTO) return(0);

			/* get the arguments */
			arg1 = va_arg(ap, INTBIG);
			arg2 = va_arg(ap, INTBIG);
			arg3 = va_arg(ap, INTBIG);
			arg4 = va_arg(ap, INTBIG);

			/* fill in the highlight information */
			newhigh.status = HIGHLINE;
			newhigh.facet = np;
			newhigh.stalx = arg1;
			newhigh.staly = arg2;
			newhigh.stahx = arg3;
			newhigh.stahy = arg4;
			(void)us_addhighlight(&newhigh);
			us_showallhighlight();
			return(0);
		}
		if (namesame(&command[5], "area") == 0)
		{
			np = getcurfacet();
			if (np == NONODEPROTO) return(0);

			/* get the arguments */
			arg1 = va_arg(ap, INTBIG);
			arg2 = va_arg(ap, INTBIG);
			arg3 = va_arg(ap, INTBIG);
			arg4 = va_arg(ap, INTBIG);

			/* fill in the highlight information */
			newhigh.status = HIGHBBOX;
			newhigh.facet = np;
			newhigh.stalx = arg1;
			newhigh.stahx = arg2;
			newhigh.staly = arg3;
			newhigh.stahy = arg4;
			(void)us_addhighlight(&newhigh);
			us_showallhighlight();
			return(0);
		}
		if (namesame(&command[5], "multiple") == 0)
		{
			/* get the arguments */
			arg1 = va_arg(ap, INTBIG);

			/* fill in the highlight information */
			us_setmultiplehighlight((char *)arg1, 0);
			us_showallhighlight();
			return(0);
		}
	}

	if (namesame(command, "flush-changes") == 0)
	{
		us_endchanges(NOWINDOW);
		us_showallhighlight();
		us_beginchanges();
		return(0);
	}

	if (namesame(command, "display-to-routine") == 0)
	{
		if (el_curwindow == NOWINDOW) return(0);

		/* get the arguments */
		arg1 = va_arg(ap, INTBIG);

		curdisplay = us_displayroutine;
		us_displayroutine = (INTSML (*)(POLYGON*, WINDOW*))arg1;
		us_redisplaynow(el_curwindow, 1);
		us_displayroutine = curdisplay;
		return(0);
	}

	if (namesame(command, "display-highlighted-to-routine") == 0)
	{
		if (el_curwindow == NOWINDOW) return(0);

		/* get the arguments */
		arg1 = va_arg(ap, INTBIG);

		curdisplay = us_displayroutine;
		us_displayroutine = (INTSML (*)(POLYGON*, WINDOW*))arg1;

		/* get the objects to be displayed */
		list = us_gethighlighted(OBJNODEINST | OBJARCINST);
		begintraversehierarchy();
		for(i=0; list[i] != NOGEOM; i++)
		{
			if (list[i]->entrytype == OBJNODEINST)
			{
				if (us_drawfacet(list[i]->entryaddr.ni, LAYERA, el_matid, 3,
					el_curwindow) < 0) break;
			} else
			{
				if (us_drawarcinst(list[i]->entryaddr.ai, LAYERA,
					el_matid, 3, el_curwindow) < 0) break;
			}
		}
		us_displayroutine = curdisplay;
		return(0);
	}

	if (namesame(command, "make-icon") == 0)
	{
		/* get the arguments */
		arg1 = va_arg(ap, INTBIG);
		arg2 = va_arg(ap, INTBIG);
		arg3 = va_arg(ap, INTBIG);
		arg4 = va_arg(ap, INTBIG);
		arg5 = va_arg(ap, INTBIG);

		return((INTBIG)us_makeiconfacet((PORTPROTO *)arg1, (char *)arg2,
			(char *)arg3, (TECHNOLOGY *)arg4, (LIBRARY *)arg5));
	}

	if (namesamen(command, "window-", 7) == 0)
	{
		/* first see if the window can be placed in a separate frame */
		if (us_graphicshas(CANUSEFRAMES))
		{
			w = us_makenewwindow();
			if (w == NOWINDOW) us_abortcommand("Could not create new window");
			return((INTBIG)w);
		}

		/* get a new window */
		if (namesame(&command[7], "new") == 0)
		{
			return((INTBIG)us_splitcurrentwindow(0, 0));
		}
		if (namesame(&command[7], "horiz-new") == 0)
		{
			if (us_needwindow()) return(0);
			return((INTBIG)us_splitcurrentwindow(1, 0));
		}
		if (namesame(&command[7], "vert-new") == 0)
		{
			if (us_needwindow()) return(0);
			return((INTBIG)us_splitcurrentwindow(2, 0));
		}
		return(0);
	}

	if (namesame(command, "down-stack") == 0)
	{
		us_pushhighlight();
		return(0);
	}

	if (namesame(command, "up-stack") == 0)
	{
		(void)us_pophighlight(0);
		return(0);
	}

	if (namesame(command, "clear") == 0)
	{
		us_clearhighlightcount();
		return(0);
	}
	return(-1);
}

/* examine an entire facet */
void us_examinenodeproto(NODEPROTO *np) {}

/* one time slice for the user aid: process a command */
void us_slice(void)
{
	INTSML but, x, y;
	char *args[3];

	if (us_desiredlibrary != 0)
	{
		args[0] = "read";
		args[1] = us_desiredlibrary;
		args[2] = "make-current";
		us_library(3, args);
		efree(us_desiredlibrary);
		us_desiredlibrary = 0;
	}

	/* handle language loops */
	if ((us_state&LANGLOOP) != 0)
	{
		if (languageconverse(0) != 0)
		{
			us_state &= ~LANGLOOP;
			ttyputmsg("Back to Electric");
		}

		setactivity("INTERPRETER");

		/* set all facet center changes detected during the last broadcast */
		us_doubchanges();
		return;
	}

	/* set all facet center changes detected during the last broadcast */
	us_doubchanges();

	/* remove menu highlighting if not temporary */
	if ((us_aid->aidstate&MENUON) != 0 &&
		(us_aid->aidstate&ONESHOTMENU) != 0 && us_menuhnx >= 0)
	{
		us_highlightmenu(us_menuhnx, us_menuhny, el_colmenbor);
		us_menuhnx = -1;
	}

	/* if keyboard data arrived, quit */
	if (setjmp(us_cmdone)) return;

	/* first see if there is any pending input */
	el_pleasestop = 0;
	if (ttydataready() != 0) us_oncommand();
	if (el_pleasestop != 0) return;

	/* get a button */
	el_pleasestop = 0;
	waitforbutton(&x, &y, &but);

	if (el_pleasestop != 0) return;
	if (but < 0) return;

	/* execute the button */
	us_ontablet(x, y, but);
}

/*
 * handle button pushes
 */
void us_ontablet(INTSML x, INTSML y, INTSML but)
{
	REGISTER USERCOM *item;
	REGISTER char *str;
	POPUPMENU *popup, *cpopup;
	INTSML butstate, swid, shei;
	REGISTER INTSML e, lx;
	REGISTER VARIABLE *var;
	REGISTER POPUPMENUITEM *retmi;
	REGISTER WINDOW *w;
	REGISTER void *curframe;
	COMMANDBINDING commandbinding;

	/* reset single-key suspension flag */
	if ((us_state&SKIPKEYS) != 0)
	{
		ttyputmsg("Single-key command suspension now lifted");
		us_state &= ~SKIPKEYS;
	}

	/* first switch windows to the current frame */
	curframe = us_getwindowframe();
	for(w = el_topwindow; w != NOWINDOW; w = w->nextwindow)
	{
		if (w->frame != curframe) continue;
		if (x >= w->uselx && x <= w->usehx && y >= w->usely && y <= w->usehy)
		{
			/* make this window the current one */
			if (w != el_curwindow)
			{
				(void)setvalkey((INTBIG)us_aid, VAID, us_current_window, (INTBIG)w,
					VWINDOW|VDONTSAVE);
				(void)setval((INTBIG)el_curlib, VLIBRARY, "curnodeproto",
					(INTBIG)w->curnodeproto, VNODEPROTO);
			}
			break;
		}
	}

	/* see if it is a pulldown menu hit */
	us_getwindowsize(curframe, &swid, &shei);
	if (us_menubarsize != 0 && y > shei-us_menubarsize)
	{
		for(;;)
		{
			lx = 0;
			for(e=0; e<us_pulldownmenucount; e++)
			{
				if (e == us_pulldownmenucount-1) break;
				if (x < us_pulldownmenupos[e]) break;
				lx = us_pulldownmenupos[e];
			}
			popup = us_pulldowns[e];

			/* highlight the pulldown title */
			us_invertbox(el_curwindow, lx, us_pulldownmenupos[e], (INTSML)(shei-us_menubarsize+1), shei);

			/* initialize the popup menu for pulldown use */
			us_initpulldownmenu(popup);

			/* determine whether menus drop-down or pop-up */
			if ((us_aid->aidstate&DROPMENUS) != 0) butstate = 1; else butstate = 0;

			/* display and select from the menu */
			cpopup = popup;
			retmi = us_popupmenu(&cpopup, &butstate, 0, lx, (INTSML)(shei-us_menubarsize), 3);

			/* restore pulldown menu reconfiguration */
			us_termpulldownmenu(popup);

			/* unhighlight the pulldown title */
			us_invertbox(el_curwindow, lx, us_pulldownmenupos[e], (INTSML)(shei-us_menubarsize+1), shei);

			/* if another menu is desired, do it */
			if (butstate != 0)
			{
				readtablet(&x, &y);
				continue;
			}

			/* now execute the selected command if it wasn't already done */
			if (retmi == NOPOPUPMENUITEM || retmi == 0) return;
			item = retmi->response;
			if (item == NOUSERCOM) return;
			us_forceeditchanges();
			us_execute(item, 0, 0, 0);
			return;
		}
	}

	/* see if it is a fixed menu hit */
	if ((us_aid->aidstate&MENUON) != 0 && (us_menuframe == 0 || curframe == us_menuframe) &&
		y >= us_menuly && y <= us_menuhy && x >= us_menulx && x <= us_menuhx)
	{
		x = (x-us_menulx) / us_menuxsz;
		y = (y-us_menuly) / us_menuysz;
		if (x < 0 || y < 0 || x >= us_menux || y >= us_menuy) return;
		var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_binding_menu);
		if (var == NOVARIABLE) return;
		if ((us_aid->aidstate&ONESHOTMENU) != 0)
			us_highlightmenu(us_menuhnx = x, us_menuhny = y, el_colhmenbor);
		if (us_menupos <= 1) str = ((char **)var->addr)[y * us_menux + x]; else
			str = ((char **)var->addr)[x * us_menuy + y];
		us_parsebinding(str, &commandbinding);
		item = us_makecommand(commandbinding.command);
		us_freebindingparse(&commandbinding);
		if (item == NOUSERCOM || item->active < 0)
		{
			ttyputerr("No command is bound to this menu entry");
			if (item != NOUSERCOM) us_freeusercom(item);
			return;
		}
		us_state &= ~GOTXY;
		us_forceeditchanges();
		us_execute(item, us_aid->aidstate&ECHOBIND, 1, 1);
		us_causeofslice(item);
		us_freeusercom(item);
		flushscreen();
	} else
	{
		/* get cursor position and check windows */
		if (us_setxy(x, y))
		{
			ttyputerr("Cursor off the screen");
			return;
		}
		if (el_curwindow != NOWINDOW && el_curwindow->buttonhandler != 0)
			(*el_curwindow->buttonhandler)(el_curwindow, but, x, y);
	}
}

/*
 * default button handler for buttons pushed in normal windows
 */
void us_buttonhandler(WINDOW *w, INTSML but, INTSML x, INTSML y)
{
	REGISTER USERCOM *item;
	REGISTER VARIABLE *var;
	COMMANDBINDING commandbinding;
	INTSML unimportant;

	/* get the command attached to the button */
	var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_binding_buttons);
	if (var == NOVARIABLE) return;
	us_parsebinding(((char **)var->addr)[but], &commandbinding);
	item = us_makecommand(commandbinding.command);
	us_freebindingparse(&commandbinding);
	if (item == NOUSERCOM || item->active < 0)
	{
		ttyputerr("No command is bound to the %s button", buttonname(but, &unimportant));
		if (item != NOUSERCOM) us_freeusercom(item);
		return;
	}
	us_forceeditchanges();
	us_execute(item, us_aid->aidstate&ECHOBIND, 1, 1);
	us_causeofslice(item);
	us_freeusercom(item);
	flushscreen();
}

/* routine that is called when keyboard input is ready */
void us_oncommand(void)
{
	INTSML cmd;
	INTBIG x, y;

	/* Read and execute the command */
	el_pleasestop = 0;
	stoptablet();
	cmd = ttygetchar();
	us_state &= ~GOTXY;
	(void)getxy(&x, &y);

	if (el_curwindow != NOWINDOW && el_curwindow->charhandler != 0)
	{
		if ((*el_curwindow->charhandler)(el_curwindow, cmd) != 0)
			us_killcurrentwindow(1);
	} else
	{
		(void)(*DEFAULTCHARHANDLER)(el_curwindow, cmd);
	}
	longjmp(us_cmdone, 1);
}

/*
 * default character handler for keys typed in normal windows
 */
INTSML us_charhandler(WINDOW *w, INTSML cmd)
{
	REGISTER INTSML count, longintro;
	static char prompt[] = "-";
	REGISTER USERCOM *uc;
	char *paramstart[MAXPARS];
	extern COMCOMP us_userp;
	REGISTER VARIABLE *var;
	COMMANDBINDING commandbinding;

	/* get the command attached to the key */
	var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_binding_keys);
	if (var == NOVARIABLE) return(0);
	us_parsebinding(((char **)var->addr)[cmd&255], &commandbinding);

	/* see if the key introduces a long command */
	longintro = 0;
	if (namesame(commandbinding.command, "tellaid user") == 0) longintro = 1;

	/* if single-key commands were suspended, wait for carriage-return */
	if ((us_state&SKIPKEYS) != 0)
	{
		if ((cmd&127) == '\r' || (cmd&127) == '\n' || longintro != 0)
		{
			ttyputmsg("Single-key command suspension now lifted");
			us_state &= ~SKIPKEYS;
		}
		if (longintro == 0)
		{
			us_freebindingparse(&commandbinding);
			return(0);
		}
	}

	/* if this is a "tellaid user" character, fill out the command */
	if (longintro != 0)
	{
		us_freebindingparse(&commandbinding);

		/* get the full command */
		prompt[0] = cmd;
		count = ttygetfullparam(prompt, &us_userp, MAXPARS, paramstart);
		if (count == 0) return(0);

		/* parse into fields and execute */
		uc = us_buildcommand(count, paramstart);
		if (uc == NOUSERCOM)
		{
			/* this is very bad!  code should deal with it better!!! */
			ttyputerr("No memory to execute command!");
			return(0);
		}
		us_forceeditchanges();
		us_execute(uc, 0, 1, 1);
		us_causeofslice(uc);
		us_freeusercom(uc);
	} else
	{
		/* check for new windows and get set tablet position for command */
		if (*commandbinding.command == 0)
		{
			us_abortcommand("The '%s' key has no meaning in this window", us_keyname(cmd));
			us_freebindingparse(&commandbinding);
			return(0);
		}
		uc = us_makecommand(commandbinding.command);
		us_freebindingparse(&commandbinding);
		if (uc == NOUSERCOM) return(0);
		if (uc->active >= 0)
		{
			us_forceeditchanges();
			us_execute(uc, us_aid->aidstate&ECHOBIND, 1, 1);
			us_causeofslice(uc);
		}
		us_freeusercom(uc);
	}
	flushscreen();
	return(0);
}

/*
 * routine to save the command in "uc" as the one that produced this slice
 */
void us_causeofslice(USERCOM *uc)
{
	if (uc->active < 0) return;
	(void)initinfstr();
	(void)addstringtoinfstr(uc->comname);
	(void)us_appendargs(uc);
	setactivity(returninfstr());
}

void us_forceeditchanges(void)
{
	REGISTER WINDOW *w;

	for(w = el_topwindow; w != NOWINDOW; w = w->nextwindow)
		if ((w->state&WINDOWTYPE) == TEXTWINDOW) us_shipchanges(w);
}

/******************** CHANGES TO THE DISPLAY ********************/

static WINDOW us_oldwindow, *us_firstnewwindow, *us_secondnewwindow, *us_killedwindow;
static INTSML us_maplow, us_maphigh, us_menuchanged, us_newwindowcount;
static INTBIG us_oldstate;

void us_startbatch(AIDENTRY *source)
{
	us_state &= ~HIGHLIGHTSET;
	us_beginchanges();
	us_menuchanged = 0;
}

void us_endbatch(void)
{
	us_endchanges(NOWINDOW);
	us_showallhighlight();
}

/*
 * routine to display any delayed highlighting
 */
void us_showallhighlight(void)
{
	REGISTER INTSML i, len;
	REGISTER VARIABLE *var;
	HIGHLIGHT high;

	/* stop now if highlighting has not been delayed */
	if ((us_state&HIGHLIGHTSET) == 0) return;

	/* show all highlighting */
	var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_highlighted);
	if (var != NOVARIABLE)
	{
		len = getlength(var);
		for(i=0; i<len; i++)
		{
			if (us_makehighlight(((char **)var->addr)[i], &high) != 0) continue;
			us_sethighlight(&high, HIGHLIT);
		}
	}

	/* show current network */
	us_shownetworkname(0);

	/* mark that highlighting is not delayed */
	us_state &= ~HIGHLIGHTSET;
}

void us_startobjectchange(INTBIG addr, INTBIG type)
{
	switch (type&VTYPE)
	{
		case VNODEINST:
			us_undisplayobject(((NODEINST *)addr)->geom);   break;
		case VARCINST:
			us_undisplayobject(((ARCINST *)addr)->geom);    break;
		case VWINDOW:
			copywindow(&us_oldwindow, (WINDOW *)addr);      break;
		case VAID:
			if ((AIDENTRY *)addr == us_aid) us_maplow = -1;
			us_oldstate = us_aid->aidstate;
			us_menuchanged = us_newwindowcount = 0;
			us_firstnewwindow = us_secondnewwindow = NOWINDOW;
			us_killedwindow = NOWINDOW;
			break;
	}
}

void us_endobjectchange(INTBIG addr, INTBIG type)
{
	REGISTER WINDOW *w, *ow, *w1, *w2, *w3;
	REGISTER INTBIG slx, shx, sly, shy, dist, l;
	REGISTER INTSML ulx, uhx, uly, uhy;
	static POLYGON *poly = NOPOLYGON;
	REGISTER NODEPROTO *np;
	INTBIG dummy, cenx, ceny;
	extern GRAPHICS us_egbox, us_gbox;
	REGISTER VARIABLE *varred, *vargreen, *varblue;

	switch (type&VTYPE)
	{
		case VNODEINST:
			us_queueredraw(((NODEINST *)addr)->geom, 0);   break;
		case VARCINST:
			us_queueredraw(((ARCINST *)addr)->geom, 0);    break;
		case VAID:
			if ((AIDENTRY *)addr != us_aid) break;

			/* handle changes to the menu */
			if ((us_oldstate&MENUON) != (us_aid->aidstate&MENUON) || us_menuchanged != 0)
			{
				/* destroy or create the menu if requested */
				us_menuchanged = 0;
				if (us_menuframe != 0 && (us_aid->aidstate&MENUON) == 0)
				{
					us_killwindowframe(us_menuframe);
					us_menuframe = 0;
					return;
				}
				if (us_menuframe == 0 && (us_aid->aidstate&MENUON) != 0)
					us_menuframe = us_newwindowframe(1);
				us_drawmenu(1, us_menuframe);
				return;
			}

			/* handle changes to the color map */
			if (us_maplow != -1)
			{
				varred = getvalkey((INTBIG)us_aid, VAID, VINTEGER|VISARRAY, us_colormap_red);
				vargreen = getvalkey((INTBIG)us_aid, VAID, VINTEGER|VISARRAY, us_colormap_green);
				varblue = getvalkey((INTBIG)us_aid, VAID, VINTEGER|VISARRAY, us_colormap_blue);
				if (varred != NOVARIABLE && vargreen != NOVARIABLE && varblue != NOVARIABLE)
					us_loadmap(&((INTBIG *)varred->addr)[us_maplow],
						&((INTBIG *)vargreen->addr)[us_maplow], &((INTBIG *)varblue->addr)[us_maplow],
							us_maplow, us_maphigh);
			}

			/* rewrite window information in header if configuration changed */
			if (us_newwindowcount != 0 || us_killedwindow != NOWINDOW)
			{
				for(w = el_topwindow; w != NOWINDOW; w = w->nextwindow)
				{
					us_setfacetname(w);
					us_setgridsize(w);
					us_setfacetsize(w);
				}
			}

			/* see if windows were created */
			if (us_newwindowcount != 0)
			{
				/* special case for split edit windows */
				if (us_killedwindow != NOWINDOW && us_newwindowcount == 2 &&
					(us_firstnewwindow->state&WINDOWTYPE) == DISPWINDOW &&
						(us_secondnewwindow->state&WINDOWTYPE) == DISPWINDOW &&
							us_firstnewwindow->curnodeproto == us_secondnewwindow->curnodeproto &&
								us_firstnewwindow->curnodeproto == us_killedwindow->curnodeproto)
				{
					/* see if block transfer can be used */
					if (us_graphicshas(CANMOVEBOX) != 0)
					{
						/* block transfer available */
						w1 = &us_oldwindow;
						w2 = us_firstnewwindow;
						w3 = us_secondnewwindow;

						/* make sure the windows have the same scale */
						if (us_windowcansplit(w1, w2, w3) != 0)
						{
							/* make sure that "w3" is the larger window */
							if (w2->usehx - w2->uselx > w3->usehx - w3->uselx ||
								w2->usehy - w2->usely > w3->usehy - w3->usely)
							{
								w = w2;   w2 = w3;   w3 = w;
							}
							us_movebox(w2, (INTSML)((w1->usehx+w1->uselx - w3->usehx+w3->uselx) / 2),
								(INTSML)((w1->usehy+w1->usely - w3->usehy+w3->usely) / 2),
									(INTSML)(w3->usehx-w3->uselx+1), (INTSML)(w3->usehy-w3->usely+1),
										w3->uselx, w3->usely);
							if (w2->curnodeproto != NONODEPROTO && w2->curnodeproto == w3->curnodeproto)
								us_movebox(w2, w3->uselx, w3->usely, (INTSML)(w2->usehx-w2->uselx+1),
									(INTSML)(w2->usehy-w2->usely+1), w2->uselx, w2->usely);
							us_drawwindow(w2, el_colwinbor);
							us_drawwindow(w3, el_colwinbor);
							break;
						}
					}
				}

				/* simple window creation: draw and outline */
				if (us_newwindowcount > 2)
				{
					/* many windows created: redraw all of them */
					for(w = el_topwindow; w != NOWINDOW; w = w->nextwindow)
					{
						us_drawwindow(w, el_colwinbor);
						if (w->redisphandler != 0) (*w->redisphandler)(w);
					}
				} else
				{
					/* one or two windows created: draw them */
					us_drawwindow(us_firstnewwindow, el_colwinbor);
					if (us_firstnewwindow->redisphandler != 0)
						(*us_firstnewwindow->redisphandler)(us_firstnewwindow);
					if (us_secondnewwindow != NOWINDOW)
					{
						us_drawwindow(us_secondnewwindow, el_colwinbor);
						if (us_secondnewwindow->redisphandler != 0)
							(*us_secondnewwindow->redisphandler)(us_secondnewwindow);
					}
				}
				break;
			}
			break;

		case VWINDOW:
			w = (WINDOW *)addr;
			ow = &us_oldwindow;

			/* adjust window if simulating info changed */
			if ((w->state&WINDOWSIMULATING) != (ow->state&WINDOWSIMULATING))
			{
				if ((w->state&WINDOWSIMULATING) != 0)
				{
					/* started simulating */
					w->uselx += SIMULATINGBORDERSIZE;   w->usehx -= SIMULATINGBORDERSIZE;
					w->usely += SIMULATINGBORDERSIZE;   w->usehy -= SIMULATINGBORDERSIZE;
				} else
				{
					/* stopped simulating */
					w->uselx -= SIMULATINGBORDERSIZE;   w->usehx += SIMULATINGBORDERSIZE;
					w->usely -= SIMULATINGBORDERSIZE;   w->usehy += SIMULATINGBORDERSIZE;
				}
				computewindowscale(w);
				if (w->redisphandler != 0) (*w->redisphandler)(w);
				break;
			}

			/* see if the facet changed */
			if (w->curnodeproto != ow->curnodeproto || w->buttonhandler != ow->buttonhandler ||
				w->charhandler != ow->charhandler || w->termhandler != ow->termhandler ||
					w->redisphandler != ow->redisphandler || w->changehandler != ow->changehandler)
			{
				us_setfacetname(w);
				us_setfacetsize(w);
				us_setgridsize(w);
				if (w->redisphandler != 0) (*w->redisphandler)(w);
				break;
			}

			/* see if the window slid vertically */
			if (w->screenlx == ow->screenlx && w->screenhx == ow->screenhx &&
				w->screenly - ow->screenly == w->screenhy - ow->screenhy &&
					w->screenly != ow->screenly)
			{
				/* window slid vertically */
				sly = ow->screenly;   shy = ow->screenhy;
				ulx = w->uselx;       uhx = w->usehx;
				uly = w->usely;       uhy = w->usehy;
				ow->uselx = ulx;      ow->usehx = uhx;
				ow->usely = uly;      ow->usehy = uhy;
				dist = shy - w->screenhy;

				/* for less than a full screen, try bit pushing */
				if (abs(dist) < shy - sly)
				{
					l = muldiv(shy - sly - abs(dist), uhy - uly, shy - sly);
					if (us_graphicshas(CANMOVEBOX) != 0)
					{
						if (dist < 0)
						{
							us_movebox(w, ulx, (INTSML)(uhy-l), (INTSML)(uhx-ulx+1), (INTSML)l, ulx, uly);
							ow->usely += l;
							ow->screenly = muldiv(shy-sly, l, uhy-uly) + sly;
						} else
						{
							us_movebox(w, ulx, uly, (INTSML)(uhx-ulx+1), (INTSML)l, ulx, (INTSML)(uhy-l));
							ow->usehy -= l;
							ow->screenhy = shy - muldiv(shy-sly, l, uhy-uly);
						}
						ow->screenly -= dist;
						ow->screenhy -= dist;
						us_redisplay(&us_oldwindow);
						break;
					}
				}

				/* bit sliding didn't work, redisplay entire window */
				if (w->redisphandler != 0) (*w->redisphandler)(w);
				break;
			}

			/* see if the window slid horizontally */
			if (w->screenly == ow->screenly && w->screenhy == ow->screenhy &&
				w->screenlx - ow->screenlx == w->screenhx - ow->screenhx &&
					w->screenlx != ow->screenlx)
			{
				/* window slid horizontally */
				slx = ow->screenlx;   shx = ow->screenhx;
				ulx = w->uselx;       uhx = w->usehx;
				uly = w->usely;       uhy = w->usehy;
				ow->uselx = ulx;      ow->usehx = uhx;
				ow->usely = uly;      ow->usehy = uhy;
				computewindowscale(ow);
				dist = w->screenhx - shx;

				if (abs(dist) < shx - slx)
				{
					l = muldiv(shx - slx - abs(dist), uhx - ulx, shx - slx);
					if (us_graphicshas(CANMOVEBOX) != 0)
					{
						if (dist < 0)
						{
							us_movebox(w, ulx, uly, (INTSML)l, (INTSML)(uhy-uly+1), (INTSML)(uhx-l), uly);
							ow->usehx -= l;
							ow->screenhx = shx - muldiv(shx-slx, l, uhx-ulx);
						} else
						{
							us_movebox(w, (INTSML)(uhx-l), uly, (INTSML)l, (INTSML)(uhy-uly+1), ulx, uly);
							ow->uselx += l;
							ow->screenlx = muldiv(shx-slx, l, uhx-ulx) + slx;
						}
						ow->screenlx += dist;
						ow->screenhx += dist;
						computewindowscale(ow);
						us_redisplay(&us_oldwindow);
						break;
					}
				}

				/* bit sliding didn't work, redisplay entire window */
				if (w->redisphandler != 0) (*w->redisphandler)(w);
				break;
			}

			/* handle grid going off or changing size while on */
			if (((ow->state&GRIDON) != 0 && (w->state&GRIDON) == 0) ||
				((w->state&GRIDON) != 0 &&
					(ow->gridx != w->gridx || ow->gridy != w->gridy)))
			{
				/* must redraw everything on B&W displays or those with less than 8-bits */
				us_setgridsize(w);
				if (us_getdispstyle(w) == BWRASTER || el_maplength < 256)
				{
					if (w->redisphandler != 0) (*w->redisphandler)(w);
					break;
				}

				/* get polygon */
				if (poly == NOPOLYGON) poly = allocpolygon(6, us_aid->cluster);

				/* erase the grid layer */
				maketruerectpoly(w->screenlx, w->screenhx, w->screenly, w->screenhy, poly);
				poly->desc = &us_egbox;
				poly->style = FILLEDRECT;
				(*us_displayroutine)(poly, w);

				/* if grid went off and nothing else changed, all is done */
				if ((w->state&GRIDON) == 0 && w->screenlx == ow->screenlx &&
					w->screenhx == ow->screenhx && w->screenly == ow->screenly &&
						w->screenhy == ow->screenhy) break;
			}

			/* handle grid going on or staying on and changing size */
			if (w->screenlx == ow->screenlx && w->screenhx == ow->screenhx &&
				w->screenly == ow->screenly && w->screenhy == ow->screenhy)
					if (((ow->state & (GRIDTOOSMALL|GRIDON)) != GRIDON &&
						(w->state & (GRIDTOOSMALL|GRIDON)) == GRIDON) ||
							((ow->state&GRIDON) != 0 && (w->state&GRIDON) != 0 &&
								(ow->gridx != w->gridx || ow->gridy != w->gridy)))
			{
				np = w->curnodeproto;
				if (np == NONODEPROTO) break;

				/* get polygon */
				if (poly == NOPOLYGON) poly = allocpolygon(6, us_aid->cluster);

				/* get facet center */
				grabpoint(np, &cenx, &ceny);

				/* grid spacing */
				poly->xv[0] = w->gridx;
				poly->yv[0] = w->gridy;

				/* initial grid location */
				poly->xv[1] = us_alignvalue(cenx, us_alignment, &dummy);
				poly->xv[1] += (w->screenlx-poly->xv[1]) / w->gridx * w->gridx;
				poly->yv[1] = us_alignvalue(ceny, us_alignment, &dummy);
				poly->yv[1] += (w->screenly-poly->yv[1]) / w->gridy * w->gridy;

				/* display screen extent */
				poly->xv[2] = w->uselx;     poly->yv[2] = w->usely;
				poly->xv[3] = w->usehx;     poly->yv[3] = w->usehy;

				/* object space extent */
				poly->xv[4] = w->screenlx;  poly->yv[4] = w->screenly;
				poly->xv[5] = w->screenhx;  poly->yv[5] = w->screenhy;

				poly->count = 6;
				poly->style = GRIDDOTS;
				poly->desc = &us_gbox;
				(*us_displayroutine)(poly, w);
				flushscreen();
				break;
			}

			/* no sliding or grid change: simply redisplay the window */
			if (w->redisphandler != 0) (*w->redisphandler)(w);
			break;
	}
}

void us_modifynodeinst(NODEINST *ni, INTBIG oldlx, INTBIG oldly, INTBIG oldhx, INTBIG oldhy,
	INTSML oldrot, INTSML oldtran)
{
	/* look for the special "facet-center" node in the "generic" technology */
	if (ni->proto == gen_facetcenterprim)
		us_setnodeprotocenter((ni->highx+ni->lowx)/2, (ni->highy+ni->lowy)/2, ni->parent);
}

void us_modifyarcinst(ARCINST *ai, INTBIG oldx0, INTBIG oldy0, INTBIG oldx1,
	INTBIG oldy1, INTBIG oldwid, INTBIG oldlen)
{
	setshrinkvalue(ai);
}

void us_modifyportproto(PORTPROTO *pp, NODEINST *oni, PORTPROTO *opp) {}

void us_modifynodeproto(NODEPROTO *np)
{
	REGISTER WINDOW *w;

	for(w = el_topwindow; w != NOWINDOW; w = w->nextwindow)
		if (np == w->curnodeproto) us_setfacetsize(w);
}

void us_modifydescript(VARIABLE *var, INTBIG odes) {}

void us_newobject(INTBIG addr, INTBIG type)
{
	REGISTER NODEINST *ni;
	REGISTER NODEPROTO *np;
	REGISTER ARCINST *ai;
	REGISTER PORTPROTO *pp;
	REGISTER INTSML i;
	REGISTER WINDOW *w;
	REGISTER NODEPROTO *onp;
	REGISTER EDITOR *ed;

	switch (type&VTYPE)
	{
		case VNODEINST:
			ni = (NODEINST *)addr;

			/* look for the "facet-center" node in the "generic" technology */
			if (ni->proto == gen_facetcenterprim)
				us_setnodeprotocenter((ni->highx+ni->lowx)/2, (ni->highy+ni->lowy)/2, ni->parent);
			break;

		case VARCINST:
			ai = (ARCINST *)addr;
			setshrinkvalue(ai);

			/* see if this arcinst wipes out the visibility of some pins */
			if ((ai->proto->userbits&CANWIPE) != 0) for(i=0; i<2; i++)
			{
				ni = ai->end[i].nodeinst;
				if ((ni->proto->userbits&ARCSWIPE) == 0) continue;

				/* if nodeinst was visible, erase it before setting WIPED bit */
				if ((ni->userbits&WIPED) == 0) us_undisplayobject(ni->geom);
				ni->userbits |= WIPED;
			}
			break;

		case VNODEPROTO:
			np = (NODEPROTO *)addr;

			/* update status display if this is new version of whats displayed */
			for(w = el_topwindow; w != NOWINDOW; w = w->nextwindow)
			{
				if (w->curnodeproto == NONODEPROTO) continue;
				if (w->curnodeproto->cell != np->cell) continue;
				if (w->curnodeproto->cellview != np->cellview) continue;
				if ((w->state&WINDOWTYPE) == TEXTWINDOW)
				{
					ed = w->editor;
					if (ed != NOEDITOR) (void)reallocstring(&ed->header,
						describenodeproto(w->curnodeproto), us_aid->cluster);
					w->redisphandler(w);
				}
				us_setfacetname(w);
			}

			onp = us_curnodeproto;
			if (onp == NONODEPROTO) return;
			if (onp->index != 0) return;
			if (onp->cell != np->cell) return;
			if (onp->cellview != np->cellview) return;
			us_curnodeproto = NONODEPROTO;
			us_shadownodeproto(0, onp);
			break;

		case VPORTPROTO:
			pp = (PORTPROTO *)addr;

			/* look at all instances of this nodeproto for use on display */
			for(ni = pp->parent->firstinst; ni != NONODEINST; ni = ni->nextinst)
				if ((ni->userbits & NEXPAND) == 0 || (pp->userbits&PORTDRAWN) != 0)
					us_queueredraw(ni->geom, 0);
			break;

		case VWINDOW:
			w = (WINDOW *)addr;
			us_newwindowcount++;
			if (us_firstnewwindow == NOWINDOW) us_firstnewwindow = w; else
				us_secondnewwindow = w;
			break;
	}
}

void us_killobject(INTBIG addr, INTBIG type)
{
	REGISTER INTSML i;
	REGISTER NODEINST *ni;
	REGISTER NODEPROTO *np, *onp;
	REGISTER PORTPROTO *pp;
	REGISTER ARCINST *ai;
	REGISTER VARIABLE *var;
	REGISTER PORTARCINST *pi;
	REGISTER WINDOW *w;
	static INTBIG edtec_layer = 0;

	switch (type&VTYPE)
	{
		case VNODEINST:
			ni = (NODEINST *)addr;
			ni->userbits |= (RETDONN|REODONN);

			/* look for the "facet-center" node in the "generic" technology */
			if (ni->proto == gen_facetcenterprim)
				us_delnodeprotocenter(ni->parent);
			break;

		case VARCINST:
			ai = (ARCINST *)addr;

			/* check the shortening of other arcs on the end nodes */
			for(i=0; i<2; i++)
			{
				ni = ai->end[i].nodeinst;
				if ((ni->userbits&DEADN) != 0) continue;
				for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
					setshrinkvalue(pi->conarcinst);
			}

			/* see if this arcinst wiped out the visibility of some pins */
			if ((ai->proto->userbits&CANWIPE) != 0) for(i=0; i<2; i++)
			{
				ni = ai->end[i].nodeinst;
				if ((ni->proto->userbits&ARCSWIPE) == 0) continue;

				/* if nodeinst still live and has other arcs, leave it */
				if ((ni->userbits&DEADN) != 0) continue;

				if (ni->firstportarcinst != NOPORTARCINST) continue;

				/* if node was invisible, display before resetting WIPED bit */
				if ((ni->userbits&WIPED) != 0) us_queueredraw(ni->geom, 0);
				ni->userbits &= ~WIPED;
			}
			break;

		case VNODEPROTO:
			np = (NODEPROTO *)addr;
			if (namesamen(np->cell->cellname, "layer-", 6) == 0)
			{
				/* may have deleted layer facet in technology library */
				if (edtec_layer == 0) edtec_layer = makekey("EDTEC_layer");
				for(onp = np->cell->lib->firstnodeproto; onp != NONODEPROTO; onp = onp->nextnodeproto)
				{
					for(ni = onp->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
					{
						var = getvalkey((INTBIG)ni, VNODEINST, VNODEPROTO, edtec_layer);
						if (var == NOVARIABLE) continue;
						if ((NODEPROTO *)var->addr == np) break;
					}
					if (ni != NONODEINST)
						ttyputerr("Warning: layer %s is used in %s", &np->cell->cellname[6],
							describenodeproto(onp));
				}
			}
			us_removeubchange(np);
			break;

		case VPORTPROTO:
			pp = (PORTPROTO *)addr;

			/* look at all instances of this nodeproto for use on display */
			for(ni = pp->parent->firstinst; ni != NONODEINST; ni = ni->nextinst)
				if ((ni->userbits & NEXPAND) == 0 || (pp->userbits&PORTDRAWN) != 0)
			{
				us_undisplayobject(ni->geom);
				us_queueredraw(ni->geom, 0);
			}
			break;

		case VWINDOW:
			w = (WINDOW *)addr;
			copywindow(&us_oldwindow, w);
			us_killedwindow = w;
			break;
	}
}

void us_newvariable(INTBIG addr, INTBIG type, INTBIG key, INTBIG newtype)
{
	REGISTER INTSML len, i;
	REGISTER VARIABLE *var;
	REGISTER WINDOW *w;
	REGISTER POPUPMENU *pm;
	REGISTER POPUPMENUITEM *mi;
	REGISTER CELL *c;
	REGISTER LIBRARY *lib;
	REGISTER TECHNOLOGY *tech;
	REGISTER char *name;
	REGISTER USERCOM *rb;
	COMMANDBINDING commandbinding;

	if ((newtype&VCREF) != 0)
	{
		name = changedvariablename(type, key, newtype);

		/* see if cell name changed */
		if ((type&VTYPE) == VCELL && namesame(name, "cellname") == 0)
		{
			c = (CELL *)addr;
			for(w = el_topwindow; w != NOWINDOW; w = w->nextwindow)
				if (w->curnodeproto != NONODEPROTO && w->curnodeproto->cell == c)
					us_setfacetname(w);
			return;
		}

		/* see if technology name changed */
		if ((type&VTYPE) == VTECHNOLOGY && namesame(name, "techname") == 0)
		{
			tech = (TECHNOLOGY *)addr;
			if (tech == el_curtech) us_settechname(0);
			return;
		}

		/* see if window extent changed */
		if ((type&VTYPE) == VWINDOW && (namesamen(name, "use", 3) == 0 ||
			namesamen(name, "screen", 6) == 0))
		{
			w = (WINDOW *)addr;
			computewindowscale(w);
			return;
		}

		/* see if current facet changed */
		if ((type&VTYPE) == VLIBRARY && namesame(name, "curnodeproto") == 0)
		{
			lib = (LIBRARY *)addr;
			if (lib == el_curlib)
			{
				if (el_curwindow != NOWINDOW) us_setfacetname(el_curwindow);
			}
			return;
		}
		return;
	}

	/* if layer letters changed, recache the data */
	if (key == us_layer_letters)
	{
		us_initlayerletters();
		return;
	}

	/* handle changes to objects on the user interface */
	if (addr == (INTBIG)us_aid)
	{
		if (key == us_spyglassfactor)
		{
			var = getvalkey(addr, type, VINTEGER, key);
			if (var == NOVARIABLE) return;
			us_setspy((INTSML)abs(var->addr));
			return;
		}

		/* see if default text editor changed */
		if (key == us_text_editor)
		{
			var = getvalkey(addr, type, VSTRING, key);
			if (var == NOVARIABLE) return;
			for(i=0; us_editortable[i].editorname != 0; i++)
				if (namesame((char *)var->addr, us_editortable[i].editorname) == 0) break;
			if (us_editortable[i].editorname == 0) return;
			us_currenteditor = i;
			return;
		}

		/* see if menu factors changed */
		if (key == us_menu_x)
		{
			var = getvalkey(addr, type, VINTEGER, key);
			if (var == NOVARIABLE) return;
			us_menux = var->addr;
			us_menuchanged++;
			return;
		}
		if (key == us_menu_y)
		{
			var = getvalkey(addr, type, VINTEGER, key);
			if (var == NOVARIABLE) return;
			us_menuy = var->addr;
			us_menuchanged++;
			return;
		}
		if (key == us_menu_position)
		{
			var = getvalkey(addr, type, VINTEGER, key);
			if (var == NOVARIABLE) return;
			us_menupos = var->addr;
			us_menuchanged++;
			return;
		}

		/* set current window if it changed */
		if (key == us_current_window)
		{
			var = getvalkey(addr, type, VWINDOW, key);
			if (var == NOVARIABLE) return;
			us_highlightwindow((WINDOW *)var->addr, 0);
			return;
		}

		/* set colormap updating if it changed */
		if (key == us_colormap_red || key == us_colormap_green ||
			key == us_colormap_blue)
		{
			us_maplow = 0;
			var = getvalkey(addr, type, VINTEGER|VISARRAY, key);
			if (var == NOVARIABLE) return;
			len = getlength(var);
			if (us_maplow == -1) us_maphigh = len-1; else
				if (len-1 > us_maphigh) us_maphigh = len-1;
			return;
		}

		/* show highlight if it changed */
		if (key == us_highlighted)
		{
			us_state |= HIGHLIGHTSET;
			return;
		}

		/* shadow current nodeproto or arcproto if it changed */
		if (key == us_current_node)
		{
			var = getvalkey(addr, type, VNODEPROTO, key);
			if (var != NOVARIABLE) us_shadownodeproto(0, (NODEPROTO *)var->addr);
			return;
		}
		if (key == us_current_arc)
		{
			var = getvalkey(addr, type, VARCPROTO, key);
			if (var != NOVARIABLE) us_shadowarcproto(0, (ARCPROTO *)var->addr);
			return;
		}

		/* shadow placement angle if it changed */
		if (key == us_placement_angle)
		{
			us_setnodeangle(0);
			return;
		}

		/* switch technology if it changed */
		if (key == us_current_technology)
		{
			var = getvalkey(addr, type, VTECHNOLOGY, key);
			if (var == NOVARIABLE) return;
			el_curtech = (TECHNOLOGY *)var->addr;
			us_settechname(0);
			us_setlambda(0);
			for(w = el_topwindow; w != NOWINDOW; w = w->nextwindow) us_setfacetsize(w);
			return;
		}

		/* switch constraint solver if it changed */
		if (key == us_current_constraint)
		{
			var = getvalkey(addr, type, VCONSTRAINT, key);
			if (var == NOVARIABLE) return;
			el_curconstraint = (CONSTRAINT *)var->addr;
			return;
		}

		/* shadow alignment if it changed */
		if (key == us_alignment_obj)
		{
			var = getvalkey(addr, type, VINTEGER, key);
			if (var != NOVARIABLE)
			{
				us_alignment = var->addr;
				us_setalignment(0);
			}
			return;
		}

		if (key == us_alignment_edge)
		{
			var = getvalkey(addr, type, VINTEGER, key);
			if (var != NOVARIABLE)
			{
				us_edgealignment = var->addr;
				us_setalignment(0);
			}
			return;
		}

		/* see if popup menu was created */
		if (namesamen(makename(key), "USER_binding_popup_", 19) == 0)
		{
			var = getvalkey(addr, type, VSTRING|VISARRAY, key);
			if (var == NOVARIABLE) return;
			len = getlength(var);

			/* create the popup menu */
			pm = (POPUPMENU *)emalloc(sizeof(POPUPMENU), us_aid->cluster);
			if (pm == 0)
			{
				ttyputerr("No memory for popup menu");
				return;
			}
			mi = (POPUPMENUITEM *)emalloc((len-1) * sizeof(POPUPMENUITEM), us_aid->cluster);
			if (mi == 0)
			{
				ttyputerr("No memory for popup menu");
				return;
			}
			pm->nextpopupmenu = us_firstpopupmenu;
			us_firstpopupmenu = pm;
			(void)allocstring(&pm->name, &makename(key)[19], us_aid->cluster);
			(void)allocstring(&pm->header, ((char **)var->addr)[0], us_aid->cluster);
			pm->list = mi;
			pm->total = len-1;

			/* fill the menu */
			for(i=1; i<len; i++)
			{
				mi[i-1].response = rb = us_allocusercom();
				rb->active = -1;
				(void)allocstring(&mi[i-1].attribute, "", us_aid->cluster);
				mi[i-1].value = 0;
				us_parsebinding(((char **)var->addr)[i], &commandbinding);
				us_setcommand(commandbinding.command, rb, (INTSML)(i-1), commandbinding.nodeglyph,
					commandbinding.arcglyph, commandbinding.menumessage,
						commandbinding.popup, commandbinding.inputpopup);
				us_freebindingparse(&commandbinding);
			}
			return;
		}
	}
}

void us_killvariable(INTBIG addr, INTBIG type, INTBIG key, INTBIG oldaddr,
	INTBIG oldtype, INTBIG olddescript)
{
	REGISTER INTSML len, i;
	REGISTER char *name;
	REGISTER POPUPMENU *pm, *lastpm;
	HIGHLIGHT high;
	REGISTER WINDOW *w;
	REGISTER EDITOR *e;

	if ((oldtype&VCREF) != 0) return;

	/* close any text editor windows that are examining the variable */
	if ((oldtype&VTYPE) == VSTRING && (oldtype&VISARRAY) != 0)
	{
		for(w = el_topwindow; w != NOWINDOW; w = w->nextwindow)
			if ((w->state&WINDOWTYPE) == TEXTWINDOW)
		{
			e = w->editor;
			if (e == NOEDITOR) continue;
			if (e->editobjvar == NOVARIABLE) continue;
			if (e->editobjvar->addr != oldaddr || e->editobjvar->type != oldtype) continue;
			if ((INTBIG)e->editobjaddr != addr || e->editobjtype != type) continue;
			us_delfacetmessage(w->curnodeproto);
		}
	}

	/* handle changes to objects on the user interface */
	if (addr == (INTBIG)us_aid)
	{
		/* show highlight if it changed */
		if (key == us_highlighted && (us_state&HIGHLIGHTSET) == 0)
		{
			len = (oldtype&VLENGTH) >> VLENGTHSH;
			for(i=0; i<len; i++)
			{
				if (us_makehighlight(((char **)oldaddr)[i], &high) != 0) continue;
				us_sethighlight(&high, ALLOFF);
				us_shownetworkname(0);
			}
			return;
		}

		/* shadow placement angle if it changed */
		if (key == us_placement_angle)
		{
			us_setnodeangle(0);
			return;
		}

		/* see if popup menu was deleted */
		if (namesamen(makename(key), "USER_binding_popup_", 19) == 0)
		{
			name = &makename(key)[19];
			lastpm = NOPOPUPMENU;
			for(pm = us_firstpopupmenu; pm != NOPOPUPMENU; pm = pm->nextpopupmenu)
			{
				if (namesame(name, pm->name) == 0) break;
				lastpm = pm;
			}
			if (pm == NOPOPUPMENU) return;
			if (lastpm == NOPOPUPMENU) us_firstpopupmenu = pm->nextpopupmenu; else
				lastpm->nextpopupmenu = pm->nextpopupmenu;
			for(i=0; i<pm->total; i++)
			{
				efree(pm->list[i].attribute);
				us_freeusercom(pm->list[i].response);
			}
			efree(pm->name);
			efree(pm->header);
			efree((char *)pm->list);
			efree((char *)pm);
			return;
		}
	}
}

void us_modifyvariable(INTBIG addr, INTBIG type, INTBIG key, INTBIG vartype,
	INTBIG index, INTBIG oldvalue)
{
	REGISTER VARIABLE *var;
	REGISTER INTSML x, y;
	REGISTER char *str;
	REGISTER USERCOM *rb;
	POPUPMENU *pm;
	COMMANDBINDING commandbinding;

	if ((vartype&VCREF) != 0) return;

	/* handle changes to objects on the user interface */
	if (addr == (INTBIG)us_aid)
	{
		/* set colormap updating if it changed */
		if (key == us_colormap_red || key == us_colormap_green || key == us_colormap_blue)
		{
			if (us_maplow == -1) us_maplow = us_maphigh = index; else
			{
				if (index < us_maplow) us_maplow = index;
				if (index > us_maphigh) us_maphigh = index;
			}
		}

		/* set menu binding if it changed */
		if (key == us_binding_menu)
		{
			if (us_menuchanged != 0) return;
			if ((us_aid->aidstate&MENUON) == 0) return;
			var = getvalkey(addr, type, VSTRING|VISARRAY, key);
			if (var == NOVARIABLE) return;
			us_parsebinding(((char **)var->addr)[index], &commandbinding);
			if (us_menupos <= 1)
			{
				y = index / us_menux;
				x = index % us_menux;
			} else
			{
				x = index / us_menuy;
				y = index % us_menuy;
			}
			if (*commandbinding.command == 0) us_drawmenuentry(x, y, ""); else
				us_drawmenuentry(x, y, ((char **)var->addr)[index]);
			us_freebindingparse(&commandbinding);
			return;
		}

		/* set popup menu binding if it changed */
		if (namesamen(makename(key), "USER_binding_popup_", 19) == 0)
		{
			var = getvalkey(addr, type, VSTRING|VISARRAY, key);
			if (var == NOVARIABLE) return;
			if (index == 0)
			{
				/* special case: set menu header */
				str = &makename(key)[19];
				for(pm = us_firstpopupmenu; pm != NOPOPUPMENU; pm = pm->nextpopupmenu)
					if (namesame(str, pm->name) == 0) break;
				if (pm != NOPOPUPMENU)
					(void)allocstring(&pm->header, ((char **)var->addr)[0], us_aid->cluster);
				return;
			}
			us_parsebinding(((char **)var->addr)[index], &commandbinding);
			if (commandbinding.popup != NOPOPUPMENU)
			{
				rb = commandbinding.popup->list[index-1].response;
				us_setcommand(commandbinding.command, rb, (INTSML)(index-1), commandbinding.nodeglyph,
					commandbinding.arcglyph, commandbinding.menumessage,
						commandbinding.popup, commandbinding.inputpopup);
				gra_nativemenurename(commandbinding.popup, (INTSML)(index-1));
			}
			us_freebindingparse(&commandbinding);
			return;
		}
	}
}

void us_insertvariable(INTBIG addr, INTBIG type, INTBIG key, INTBIG vartype, INTBIG index) {}

void us_deletevariable(INTBIG addr, INTBIG type, INTBIG key, INTBIG vartype, INTBIG index,
	INTBIG oldvalue) {}

/*
 * routine to place the command "str" in the command object "rb".  Also sets
 * the node glyph if "nodeglyph" is not NONODEPROTO, the arc glyph if
 * "arcglyph" is not NOARCPROTO and the menu message if "menumessage" is nonzero.
 * If "popup" is not NOPOPUPMENU, this is a popup (and is input-popup if
 * "inputpopup" is nonzero).  For popup menus, "index" is the menu entry.
 */
void us_setcommand(char *str, USERCOM *rb, INTSML index,
	NODEPROTO *nodeglyph, ARCPROTO *arcglyph, char *menumessage,
	POPUPMENU *popup, INTSML inputpopup)
{
	char *a[MAXPARS+1];
	REGISTER INTSML newcount, j, i;
	COMCOMP *carray[MAXPARS];
	extern COMCOMP us_userp;

	/* handle un-binding */
	if (*str == 0)
	{
		if (popup != NOPOPUPMENU)
		{
			if (menumessage != 0)
			{
				if (rb->message != 0) efree(rb->message);
				(void)allocstring(&rb->message, menumessage, us_aid->cluster);
			}
			if (rb->message == 0)
				(void)reallocstring(&popup->list[index].attribute, "", us_aid->cluster); else
					(void)reallocstring(&popup->list[index].attribute, rb->message, us_aid->cluster);
		}
		if (rb->active >= 0)
		{
			for(j=0; j<rb->count; j++) efree(rb->word[j]);
			rb->count = 0;
			if (rb->message != 0) efree(rb->message);
			rb->message = 0;
		}
		rb->active = -1;
		return;
	}

	/* parse this command */
	newcount = us_parsecommand(str, a);
	if (newcount <= 0) return;
	i = parse(a[0], &us_userp, 1);
	if (i < 0) return;

	/* remove former command if it exists */
	if (rb->active >= 0)
	{
		for(j=0; j<rb->count; j++) efree(rb->word[j]);
		if (rb->message != 0) efree(rb->message);
		rb->message = 0;
	}

	/* set the new command */
	for(j=0; j<newcount-1; j++)
	{
		if (allocstring(&rb->word[j], a[j+1], us_aid->cluster) != 0)
		{
			ttyputerr("No memory for binding");
			rb->active = -1;
			return;
		}
	}

	/* set all other information */
	if (menumessage == 0) rb->message = 0; else
		(void)allocstring(&rb->message, menumessage, us_aid->cluster);
	rb->count = newcount-1;
	rb->active = i;
	rb->menu = us_getpopupmenu(a[0]);
	(void)allocstring(&rb->comname, a[0], us_aid->cluster);
	rb->nodeglyph = nodeglyph;
	rb->arcglyph = arcglyph;
	if (popup != NOPOPUPMENU)
	{
		/* load popup menu with full command string */
		if (rb->message != 0)
			(void)reallocstring(&popup->list[index].attribute, rb->message, us_aid->cluster); else
		{
			(void)initinfstr();
			(void)addstringtoinfstr(rb->comname);
			(void)us_appendargs(rb);
			(void)reallocstring(&popup->list[index].attribute, returninfstr(), us_aid->cluster);
		}
		popup->list[index].valueparse = NOCOMCOMP;
		if (inputpopup != 0)
		{
			/* determine command completion for input portion */
			i = us_fillcomcomp(rb, carray);
			if (i > rb->count && carray[rb->count] != NOCOMCOMP)
				popup->list[index].valueparse = carray[rb->count];
		}
		if (popup->list[index].valueparse == NOCOMCOMP)
			popup->list[index].maxlen = -1; else popup->list[index].maxlen = 20;
	}
}

/*
 * routine to parse the binding string in "st" and fill in the binding structure
 * in "commandbinding".
 */
void us_parsebinding(char *st, COMMANDBINDING *commandbinding)
{
	char *pt;
	REGISTER char *str;

	commandbinding->nodeglyph = NONODEPROTO;
	commandbinding->arcglyph = NOARCPROTO;
	commandbinding->backgroundcolor = 0;
	commandbinding->menumessagesize = 0;
	commandbinding->menumessage = 0;
	commandbinding->popup = NOPOPUPMENU;
	commandbinding->inputpopup = 0;
	commandbinding->command = "";

	pt = st;
	for(;;)
	{
		str = getkeyword(&pt, "= ");
		if (str == NOSTRING || *str == 0) break;
		(void)tonextchar(&pt);

		if (namesame(str, "node") == 0)
		{
			str = getkeyword(&pt, " ");
			if (str == NOSTRING) break;
			commandbinding->nodeglyph = getnodeproto(str);
			continue;
		}
		if (namesame(str, "arc") == 0)
		{
			str = getkeyword(&pt, " ");
			if (str == NOSTRING) break;
			commandbinding->arcglyph = getarcproto(str);
			continue;
		}
		if (namesame(str, "background") == 0)
		{
			str = getkeyword(&pt, " ");
			if (str == NOSTRING) break;
			commandbinding->backgroundcolor = myatoi(str);
			continue;
		}
		if (namesame(str, "popup") == 0)
		{
			str = getkeyword(&pt, " ");
			if (str == NOSTRING) break;
			commandbinding->popup = us_getpopupmenu(str);
			continue;
		}
		if (namesame(str, "inputpopup") == 0)
		{
			str = getkeyword(&pt, " ");
			if (str == NOSTRING) break;
			commandbinding->popup = us_getpopupmenu(str);
			commandbinding->inputpopup = 1;
			continue;
		}
		if (namesame(str, "message") == 0)
		{
			commandbinding->menumessagesize = TXTLARGE;
			(void)tonextchar(&pt);
			(void)initinfstr();
			for(str = pt; *str != 0; str++)
			{
				if (*str == '"') break;
				if (*str == '^') str++;
				(void)addtoinfstr(*str);
			}
			pt = str+1;
			(void)allocstring(&commandbinding->menumessage, returninfstr(), us_aid->cluster);
			continue;
		}
		if (namesame(str, "messagesize") == 0)
		{
			str = getkeyword(&pt, " ");
			if (str == NOSTRING) break;
			commandbinding->menumessagesize = myatoi(str);
		}
		if (namesame(str, "command") == 0) break;
	}
	(void)allocstring(&commandbinding->command, pt, us_aid->cluster);
}

/*
 * routine to free any allocated things in the "commandbinding" structure that was filled-in
 * by "us_parsebinding()".
 */
void us_freebindingparse(COMMANDBINDING *commandbinding)
{
	if (commandbinding->menumessage != 0) efree(commandbinding->menumessage);
	commandbinding->menumessage = 0;
	if (*commandbinding->command != 0) efree(commandbinding->command);
	commandbinding->command = "";
}

void us_readlibrary(LIBRARY *lib)
{
	REGISTER VARIABLE *var;
	REGISTER char *pt, *name;
	char *par[MAXPARS];
	REGISTER INTSML majversion, minversion, found, i, j, nvers, len;
	REGISTER TECHNOLOGY *mocmostech;
	REGISTER POPUPMENU *pm;
	REGISTER POPUPMENUITEM *mi;
	REGISTER NODEPROTO *np;
	REGISTER NODEINST *ni;
	REGISTER ARCINST *ai;
	REGISTER USERCOM *rb;
	COMMANDBINDING commandbinding;
	extern COMCOMP us_yesnop;

	/* create any popup menus found in the library */
	for(i=0; i<us_aid->numvar; i++)
	{
		/* only want permanent user-interface variables (ones just read in) */
		var = &us_aid->firstvar[i];
		if ((var->type&VDONTSAVE) != 0) continue;

		/* handle popup menus */
		if (namesamen(makename(var->key), "USER_binding_popup_", 19) == 0)
		{
			name = &makename(var->key)[19];
			if (us_getpopupmenu(name) != NOPOPUPMENU) continue;

			/* create the popup menu */
			len = getlength(var);
			pm = (POPUPMENU *)emalloc(sizeof(POPUPMENU), us_aid->cluster);
			if (pm == 0)
			{
				ttyputerr("No memory for popup menu");
				break;
			}
			mi = (POPUPMENUITEM *)emalloc((len-1) * sizeof(POPUPMENUITEM), us_aid->cluster);
			if (mi == 0)
			{
				ttyputerr("No memory for popup menu");
				return;
			}
			pm->nextpopupmenu = us_firstpopupmenu;
			us_firstpopupmenu = pm;
			(void)allocstring(&pm->name, name, us_aid->cluster);
			(void)allocstring(&pm->header, ((char **)var->addr)[0], us_aid->cluster);
			pm->list = mi;
			pm->total = len-1;

			/* initialize the menu */
			for(j=1; j<len; j++)
			{
				mi[j-1].response = rb = us_allocusercom();
				rb->active = -1;
				(void)allocstring(&mi[j-1].attribute, "", us_aid->cluster);
				mi[j-1].value = 0;
			}
			us_scanforkeyequiv(pm);
		}
	}

	/* fill any popup menus found in the library */
	for(i=0; i<us_aid->numvar; i++)
	{
		/* only want permanent user-interface variables (ones just read in) */
		var = &us_aid->firstvar[i];
		if ((var->type&VDONTSAVE) != 0) continue;
		var->type |= VDONTSAVE;

		/* handle popup menus */
		if (namesamen(makename(var->key), "USER_binding_popup_", 19) == 0)
		{
			name = &makename(var->key)[19];
			pm = us_getpopupmenu(name);
			mi = pm->list;
			len = getlength(var);

			/* fill the menu */
			for(j=1; j<len; j++)
			{
				rb = mi[j-1].response;
				us_parsebinding(((char **)var->addr)[j], &commandbinding);
				us_setcommand(commandbinding.command, rb, (INTSML)(j-1), commandbinding.nodeglyph,
					commandbinding.arcglyph, commandbinding.menumessage,
						pm, commandbinding.inputpopup);
				us_freebindingparse(&commandbinding);
			}
		}
	}

	/*
	 * see if the library is old enough to require MOCMOS conversion
	 * or serpentine port conversion
	 */
	var = getval((INTBIG)lib, VLIBRARY, VSTRING, "LIB_former_version");
	if (var == NOVARIABLE) return;

	/* parse the former Electric version number */
	pt = (char *)var->addr;
	majversion = atoi(pt);
	while (*pt != 0 && *pt != '.') pt++;
	if (*pt == 0) minversion = 0; else minversion = atoi(pt+1);
	nvers = majversion*100 + minversion;
	if (nvers > 335) return;

	/* versions 3.35 or earlier may require MOCMOS conversion */
	mocmostech = gettechnology("mocmos");
	if (mocmostech == NOTECHNOLOGY) return;

	/* see if there is any MOCMOS data in the library */
	found = 0;
	for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
	{
		for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
			if (ni->proto->index != 0 && ni->proto->tech == mocmostech)
		{
			found++;
			break;
		}
		for(ai = np->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
			if (ai->proto->tech == mocmostech)
		{
			found++;
			break;
		}
		if (found != 0) break;
	}
	if (found == 0) return;

	/* there are MOCMOS elements in an old library: offer conversion */
	ttyputmsg("This library contains old format MOSIS CMOS (mocmos) facets");
	i = ttygetparam("Would you like to convert them? [y] ", &us_yesnop, MAXPARS, par);
	if (i > 0 && namesamen(par[0], "no", (INTSML)(strlen(par[0]))) == 0)
	{
		ttyputmsg("No conversion done.  To do this later, use:");
		ttyputmsg("    -technology tell mocmos convert-old-format-library");
		return;
	}

	/* do the conversion */
	tech_convertmocmoslib(lib);
}

void us_eraselibrary(LIBRARY *lib)
{
	us_unqueueredraw(lib);
}

void us_writelibrary(LIBRARY *lib, INTSML pass) {}

/*
 * handle options. Returns nonzero upon error
 */
INTSML us_options(INTBIG *argc, char *argv[])
{
	REGISTER INTSML nolog;
	REGISTER char *retur;
	extern char *el_version;

	us_ignore_cadrc = 0;
	nolog = 0;

	while (*argc > 1 && argv[1][0] == '-')
	{
		switch (argv[1][1])
		{
			case '-':			/* long options */
				if (strcmp(&argv[1][2], "help") == 0)
				{
					error("Usage: electric [-o] [-n] [-mN] [-c] [-f] [-geom WxH+X+Y] [-t TECHNOLOGY] [LIBRARY]");
				}
				if (strcmp(&argv[1][2], "version") == 0)
				{
					error("Electric version %s", el_version);
				}
				break;

			case 'c':			/* ignore cadrc file */
				us_ignore_cadrc = 1;
				break;

			case 'f':			/* full screen usage, used in graphX11.c */
				break;

			case 'g':			/* X11 Geometry spec SRP 2-19-91 */
				(*argc)--; argv++;	/* bump past the geometry string */
				break;

			case 'i':			/* initial macro file to load after cadrc */
				us_firstmacrofile = argv[2];
				(*argc)--;   argv++;
				break;

			case 'm':			/* bitmap length, used in graphXXX.c */
				break;

			case 'n':			/* no session recording */
				nolog++;
				break;

			case 't':
				if (argv[1][2] != 0) retur = &argv[1][2]; else
				{
					retur = argv[2];
					(*argc)--;   argv++;
				}
				el_curtech = gettechnology(retur);
				if (el_curtech == NOTECHNOLOGY) error("Unknown technology: '%s'", retur);
				break;

#ifdef	sun
			case 'W':			/* SUN window switches must be ignored */
				switch (argv[1][2])
				{
					case 'g':	/* default color */
					case 'H':	/* help */
					case 'i':	/* iconic */
					case 'n':	/* no label */
						break;
					case 'I':	/* icon image */
					case 'l':	/* label */
					case 'L':	/* icon label */
					case 't':	/* font */
					case 'T':	/* icon font */
					case 'w':	/* width */
						(*argc)--;   argv++;   break;
					case 'p':	/* position */
					case 'P':	/* closed position */
					case 's':	/* size */
						(*argc) -= 2;   argv += 2;   break;
					case 'b':	/* background color */
					case 'f':	/* foreground color */
						(*argc) -= 3;   argv += 3;   break;
				}
				break;
#endif

			default:
				error("Unrecognized switch: %s", argv[1]);
		}
		(*argc)--;
		argv++;
	}

	/* remember the initial library if specified */
	if (*argc == 2)
	{
		(*argc)--;
		us_firstlibrary = argv[1];
	}

	/* check the arguments */
	if (*argc > 1)
		error("Usage: electric [-o] [-n] [-mN] [-c] [-f] [-geom WxH+X+Y] [-t TECHNOLOGY] [LIBRARY]");

	/* see if tracing should be done */
	if (nolog == 0) us_logstartrecord();
	return(0);
}

void us_findcadrc(void)
{
	char *suf, *cadrc;

	/* establish proper file name */
#ifdef ONUNIX
	cadrc = ".cadrc";
#else
	cadrc = "cadrc";
#endif

	/* try "cadrc" in current (and home on UNIX) directory */
	if (us_docadrc(cadrc) == 0) return;
#ifdef ONUNIX
	if (us_docadrc("~/.cadrc") == 0) return;
#endif

	/* not found there: try library directory */
	if (initinfstr()) return;
	if (addstringtoinfstr(el_libdir)) return;
	if (addstringtoinfstr(cadrc)) return;
	suf = returninfstr();
	if (us_docadrc(suf) == 0) return;

	ttyputerr("Cannot find '%s' startup file: installation may be incorrect",
		cadrc);
}

/*
 * get the commands in the system's or user's "cadrc" file: "name".
 * Returns nonzero if the file cannot be found.
 */
INTSML us_docadrc(char *name)
{
	FILE *in;
	char *filename;

	/* look for startup file */
	in = xopen(name, FILETYPECADRC, "", &filename);
	if (in == NULL) return(1);

	/* create a new macro package */
	us_curmacropack = us_newmacropack("CADRC");

	/* execute commands beginning with "electric" in this file */
	us_docommands(in, 0, "electric");
	xclose(in);

	/* now there is no macro package */
	us_curmacropack = NOMACROPACK;
	return(0);
}

void us_wanttoread(char *name)
{
	(void)allocstring(&us_desiredlibrary, name, us_aid->cluster);
}
