/*
 * Electric(tm) VLSI Design System
 *
 * File: usrcomcd.c
 * User interface aid: command handler for C through D
 * Written by: Steven M. Rubin, Electric Editor Incorporated
 *
 * Copyright (c) 1998 Electric Editor Incorporated.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Electric Editor Incorporated
 * 23470 Sunset Drive, Suite 108
 * Los Gatos, California 95033
 * support@electriceditor.com
 */

#include "global.h"
#include "egraphics.h"
#include "usr.h"
#include "usrtrack.h"
#include "database.h"
#include "tecgen.h"
#include "tecart.h"
#include "tecschem.h"
#include "efunction.h"
#include <math.h>

#define	MAXLINE	   200		/* max characters on color map input line */
struct {
	INTSML bits;
	INTSML set;
} us_overlappables[] =
{
	{LAYERT1, 0},
	{LAYERT2, 0},
	{LAYERT3, 0},
	{LAYERT4, 0},
	{LAYERT5, 0},
	{0, 0}
};

void us_color(INTSML count, char *par[])
{
	REGISTER INTSML i, j, k, l, high, totalcolor, max, red, green, blue,
		newmax, style, newraster;
	char line[MAXLINE], *layerlabel[5], *layerabbrev[5], *filename, *truename;
	INTBIG redt[256], greent[256], bluet[256], rr, rg, rb;
	float hue, sat, inten, amt;
	REGISTER VARIABLE *varred, *vargreen, *varblue;
	REGISTER char *pp, *orig, *s, *la;
	GRAPHICS *desc;
	REGISTER FILE *f;
	extern COMCOMP us_colorp, us_colorentryp, us_colorreadp, us_colorwritep;

	if (count == 0)
	{
		count = ttygetparam("COLOR option: ", &us_colorp, MAXPARS, par);
		if (count == 0)
		{
			us_abortedmsg();
			return;
		}
	}
	l = strlen(pp = par[0]);

	if (namesamen(pp, "black-background", l) == 0 && l >= 1)
	{
		us_getcolormap(el_curtech, COLORSBLACK, 1);
		ttyputmsgf("Black-background colormap loaded");
		return;
	}

	if (namesamen(pp, "white-background", l) == 0 && l >= 1)
	{
		us_getcolormap(el_curtech, COLORSWHITE, 1);
		ttyputmsgf("White-background colormap loaded");
		return;
	}

	if (namesamen(pp, "default", l) == 0 && l >= 1)
	{
		us_getcolormap(el_curtech, COLORSDEFAULT, 1);
		ttyputmsgf("Default colormap loaded");
		return;
	}

	if (namesamen(pp, "highlight", l) == 0 && l >= 1)
	{
		if (us_needwindow()) return;
		if (us_getdispstyle(el_curwindow) != COLORMAP)
		{
			us_abortcommand("Cannot highlight colors on Black&White display");
			return;
		}
		if (count < 2)
		{
			us_abortcommand("Usage: color highlight LAYERLETTERS [AMOUNT]");
			return;
		}
		amt = 0.2f;
		if (count >= 3)
		{
			amt = (float)atofr(par[2]);
			amt /= WHOLE;
		}
		if (amt < 0.0 || amt > 1.0)
		{
			us_abortcommand("Highlight amount must be from 0 to 1");
			return;
		}

		/* check the list of letters */
		for(k=0; us_overlappables[k].bits != 0; k++)
			us_overlappables[k].set = 0;
		for(i=0; par[1][i] != 0 && par[1][i] != '('; i++)
		{
			for(j=0; j<el_curtech->layercount; j++)
			{
				la = us_layerletters(el_curtech, j);
				for(k=0; la[k] != 0; k++) if (par[1][i] == la[k]) break;
				if (la[k] != 0) break;
			}
			if (j >= el_curtech->layercount)
			{
				us_abortcommand("Unknown layer letter: %c", par[1][i]);
				return;
			}
			for(k=0; us_overlappables[k].bits != 0; k++)
				if (el_curtech->layers[j]->bits == us_overlappables[k].bits) break;
			if (us_overlappables[k].bits == 0)
			{
				us_abortcommand("Layer %s(%c) is not overlappable", layername(el_curtech, j), par[1][i]);
				return;
			}
			us_overlappables[k].set = 1;
		}

		/* get the color map */
		varred = getvalkey((INTBIG)us_aid, VAID, VINTEGER|VISARRAY, us_colormap_red);
		vargreen = getvalkey((INTBIG)us_aid, VAID, VINTEGER|VISARRAY, us_colormap_green);
		varblue = getvalkey((INTBIG)us_aid, VAID, VINTEGER|VISARRAY, us_colormap_blue);
		if (varred == NOVARIABLE || vargreen == NOVARIABLE || varblue == NOVARIABLE)
		{
			ttyputerr("Cannot get current color map");
			return;
		}

		/* copy the color arrays from the variables */
		for(i=0; i<256; i++)
		{
			redt[i] = ((INTBIG *)varred->addr)[i];
			greent[i] = ((INTBIG *)vargreen->addr)[i];
			bluet[i] = ((INTBIG *)varblue->addr)[i];
		}

		/* adjust the color arrays */
		for(i=1; i<256; i++)
		{
			/* ignore if nonoverlappable, grid, or highlight */
			if ((i&(LAYERG|LAYERH|LAYEROE)) != 0) continue;

			/* skip if it is one of the highlighted layers */
			for(k=0; us_overlappables[k].bits != 0; k++)
				if ((us_overlappables[k].bits&i) != 0 && us_overlappables[k].set != 0) break;
			if (us_overlappables[k].bits != 0) continue;

			/* dim the layer */
			us_rgbtohsv((INTSML)redt[i], (INTSML)greent[i], (INTSML)bluet[i], &hue, &sat, &inten);
			sat *= amt;
			us_hsvtorgb(hue, sat, inten, &rr, &rg, &rb);
			redt[i] = rr;   greent[i] = rg;   bluet[i] = rb;
		}

		/* set the new color map */
		startobjectchange((INTBIG)us_aid, VAID);
		(void)setvalkey((INTBIG)us_aid, VAID, us_colormap_red, (INTBIG)redt,
			VINTEGER|VISARRAY|(256<<VLENGTHSH)|VDONTSAVE);
		(void)setvalkey((INTBIG)us_aid, VAID, us_colormap_green, (INTBIG)greent,
			VINTEGER|VISARRAY|(256<<VLENGTHSH)|VDONTSAVE);
		(void)setvalkey((INTBIG)us_aid, VAID, us_colormap_blue, (INTBIG)bluet,
			VINTEGER|VISARRAY|(256<<VLENGTHSH)|VDONTSAVE);
		endobjectchange((INTBIG)us_aid, VAID);
		return;
	}

	if ((namesamen(pp, "entry", l) == 0 ||
		namesamen(pp, "pattern", l) == 0) && l >= 1)
	{
		/* force pattern references on black&white display */
		style = *pp;
		if (us_needwindow()) return;
		if (us_getdispstyle(el_curwindow) == BWRASTER) style = 'p';

		/* get the entry number */
		if (count < 2)
		{
			count = ttygetparam("Entry: ", &us_colorentryp, MAXPARS-1, &par[1]) + 1;
			if (count == 1)
			{
				us_abortedmsg();
				return;
			}
		}

		if (style == 'p')
		{
			/* pattern specification */
			j = el_curtech->layercount;
			totalcolor = us_getcolormapentry(par[1], 1);
		} else
		{
			/* color map entry specification */
			j = el_maplength;
			totalcolor = us_getcolormapentry(par[1], 0);
		}
		if (totalcolor < 0) return;
		if (totalcolor >= j)
		{
			us_abortcommand("Entry must be from 0 to %d", j-1);
			return;
		}

		/* get color arrays */
		varred = getvalkey((INTBIG)us_aid, VAID, VINTEGER|VISARRAY, us_colormap_red);
		vargreen = getvalkey((INTBIG)us_aid, VAID, VINTEGER|VISARRAY, us_colormap_green);
		varblue = getvalkey((INTBIG)us_aid, VAID, VINTEGER|VISARRAY, us_colormap_blue);
		if (varred == NOVARIABLE || vargreen == NOVARIABLE || varblue == NOVARIABLE)
		{
			ttyputerr("Cannot get current color map");
			return;
		}

		/* if more values specified, get new entry */
		if (count >= 3)
		{
			if (style == 'p')
			{
				/* get a stipple pattern for this layer */
				desc = el_curtech->layers[totalcolor];
				pp = par[2];
				for(j=0; j<8; j++)
				{
					if (*pp == 0) pp = par[2];
					if (*pp == '/')
					{
						pp++;
						continue;
					}
					newraster = myatoi(pp);
					(void)setind((INTBIG)desc, VGRAPHICS, "raster", j, newraster);
					while (*pp != '/' && *pp != 0) pp++;
					if (*pp == '/') pp++;
				}
			} else
			{
				/* get three colors for the map */
				if (count < 5)
				{
					us_abortcommand("Usage: color entry INDEX [RED GREEN BLUE]");
					return;
				}
				us_setcolorentry(totalcolor, (INTSML)myatoi(par[2]), (INTSML)myatoi(par[3]),
					(INTSML)myatoi(par[4]), 0, 1);
			}
		}

		/* report the current color values */
		if (style == 'p')
		{
			ttyputmsg("Entry %d (%s) is bits:", totalcolor,
				layername(el_curtech, totalcolor));
			desc = el_curtech->layers[totalcolor];
			for(j=0; j<8; j++)
			{
				(void)sprintf(line, "0x%04x |", desc->raster[j]&0xFFFF);
				for(k=0; k<16; k++)
					if ((desc->raster[j] & (1<<(15-k))) != 0)
						(void)strcat(line, "X"); else
							(void)strcat(line, " ");
				(void)strcat(line, "|");
				ttyputmsg("%s", line);
			}
		} else us_printcolorvalue(totalcolor);
		return;
	}

	if (namesamen(pp, "mix", l) == 0 && l >= 1)
	{
		if (us_needwindow()) return;
		if (us_getdispstyle(el_curwindow) != COLORMAP)
		{
			us_abortcommand("Cannot edit color map on Black&White display");
			return;
		}
		for(i=0; i<5; i++) layerlabel[i] = 0;
		j = el_curtech->layercount;
		for(i=0; i<j; i++)
		{
			desc = el_curtech->layers[i];
			switch (desc->bits)
			{
				case LAYERT1: k = 0;   break;
				case LAYERT2: k = 1;   break;
				case LAYERT3: k = 2;   break;
				case LAYERT4: k = 3;   break;
				case LAYERT5: k = 4;   break;
				default:      k = -1;  break;
			}
			if (k < 0) continue;
			if (layerlabel[k] != 0) continue;
			layerlabel[k] = layername(el_curtech, i);
			layerabbrev[k] = (char *)emalloc(2, el_tempcluster);
			layerabbrev[k][0] = *us_layerletters(el_curtech, i);
			layerabbrev[k][1] = 0;
		}
		for(i=0; i<5; i++) if (layerlabel[i] == 0)
		{
			layerlabel[i] = "UNUSED";
			(void)allocstring(&layerabbrev[i], "?", el_tempcluster);
		}
		us_palette(el_curwindow, layerlabel, layerabbrev);
		for(i=0; i<5; i++) efree(layerabbrev[i]);
		return;
	}

	if (namesamen(pp, "read", l) == 0 && l >= 1)
	{
		if (count < 2)
		{
			count = ttygetparam("color/Color map file name: ", &us_colorreadp, MAXPARS-1, &par[1]) + 1;
			if (count == 1)
			{
				us_abortedmsg();
				return;
			}
		}
		orig = par[1];

		/* get color map from file */
		f = xopen(orig, FILETYPECOLORMAP, el_libdir, &filename);
		if (f == NULL)
		{
			us_abortcommand("Cannot find %s", orig);
			return;
		}
		ttyputmsg("Reading %s", orig);

		/* see if technology name is right */
		(void)xfgets(line, MAXLINE, f);
		if (strncmp(line, "technology=", 11) != 0)
		{
			us_abortcommand("Invalid color map file");
			xclose(f);
			return;
		}
		pp = &line[11];
		if (strcmp(pp, el_curtech->techname) != 0)
			ttyputmsgf("Warning: color map is for %s technology", pp);

		max = 0;
		for(;;)
		{
			if (xfgets(line, MAXLINE, f))
			{
				us_abortcommand("Map ends prematurely");
				return;
			}
			(void)strcat(line, "\t\t\t\t");
			pp = line;   red = atoi(pp);
			while (*pp != '\t' && *pp != 0) pp++;  green = atoi(++pp);
			while (*pp != '\t' && *pp != 0) pp++;  blue = atoi(++pp);
			while (*pp != '\t' && *pp != 0) pp++;  newmax = atoi(++pp);
			if (newmax < max || newmax >= 256)
			{
				us_abortcommand("Bad map indices: %s", line);
				xclose(f);
				return;
			}
			for(; max <= newmax; max++)
			{
				redt[max] = red&0377;
				greent[max] = green&0377;
				bluet[max] = blue&0377;
			}
			if (max >= 256) break;
		}

		/* set the color map */
		startobjectchange((INTBIG)us_aid, VAID);
		(void)setvalkey((INTBIG)us_aid, VAID, us_colormap_red, (INTBIG)redt,
			VINTEGER|VISARRAY|(256<<VLENGTHSH)|VDONTSAVE);
		(void)setvalkey((INTBIG)us_aid, VAID, us_colormap_green, (INTBIG)greent,
			VINTEGER|VISARRAY|(256<<VLENGTHSH)|VDONTSAVE);
		(void)setvalkey((INTBIG)us_aid, VAID, us_colormap_blue, (INTBIG)bluet,
			VINTEGER|VISARRAY|(256<<VLENGTHSH)|VDONTSAVE);
		endobjectchange((INTBIG)us_aid, VAID);

		/* now get the bit-map information */
		if (xfgets(line, MAXLINE, f))
		{
			ttyputmsg("Warning: no bit-map information in file");
			return;
		}
		j = atoi(line);
		high = el_curtech->layercount;
		for(i=0; i<j; i++)
		{
			if (xfgets(line, MAXLINE, f))
			{
				ttyputmsg("Warning: EOF during bit-map data");
				break;
			}
			for(k=0; k<high; k++)
			{
				pp = us_layerletters(el_curtech, k);
				for(l=0; pp[l] != 0; l++) if (pp[l] == line[0]) break;
				if (pp[l] != 0)
				{
					desc = el_curtech->layers[k];
					pp = &line[1];
					for(l=0; l<8; l++)
					{
						while (*pp != ' ' && *pp != 0) pp++;
						newraster = myatoi(++pp);
						(void)setind((INTBIG)desc, VGRAPHICS, "raster", l, newraster);
					}
					break;
				}
			}
			if (k >= high) ttyputmsg("Warning: no layer '%c' in this technology", line[0]);
		}
		xclose(f);

		ttyputmsg("Color map in %s read", orig);
		return;
	}

	if (namesamen(pp, "write", l) == 0 && l >= 1)
	{
		if (count < 2)
		{
			count = ttygetparam("color/Color map file name: ", &us_colorwritep, MAXPARS-1, &par[1]) + 1;
			if (count == 1)
			{
				us_abortedmsg();
				return;
			}
		}
		orig = par[1];

		/* get color arrays */
		varred = getvalkey((INTBIG)us_aid, VAID, VINTEGER|VISARRAY, us_colormap_red);
		vargreen = getvalkey((INTBIG)us_aid, VAID, VINTEGER|VISARRAY, us_colormap_green);
		varblue = getvalkey((INTBIG)us_aid, VAID, VINTEGER|VISARRAY, us_colormap_blue);
		if (varred == NOVARIABLE || vargreen == NOVARIABLE || varblue == NOVARIABLE)
		{
			ttyputerr("Cannot get current color map");
			return;
		}

		/* write color map to file */
		f = xcreate(orig, FILETYPECOLORMAP, "Color map file", &truename);
		if (f == NULL)
		{
			if (truename != 0) us_abortcommand("Cannot write %s", truename);
			return;
		}
		xprintf(f, "technology=%s\n", el_curtech->techname);
		for(i=0; i<256; i++)
			if (i == 255 || ((INTBIG *)varred->addr)[i] != ((INTBIG *)varred->addr)[i+1] ||
				((INTBIG *)vargreen->addr)[i] != ((INTBIG *)vargreen->addr)[i+1] ||
					((INTBIG *)varblue->addr)[i] != ((INTBIG *)varblue->addr)[i+1])
		{
			xprintf(f, "%d\t%d\t%d\t%d\n", ((INTBIG *)varred->addr)[i],
				((INTBIG *)vargreen->addr)[i], ((INTBIG *)varblue->addr)[i], i);
		}

		/* write the bit maps */
		high = el_curtech->layercount;
		xprintf(f, "%d\n", high);
		for(i=0; i<high; i++)
		{
			desc = el_curtech->layers[i];
			s = us_layerletters(el_curtech, i);
			xprintf(f, "%c", s[0]);
			for(j=0; j<8; j++)
				xprintf(f, " 0%o", desc->raster[j] & 0xFFFF);
			xprintf(f, "\n");
		}
		xclose(f);
		ttyputmsgf("%s written", truepath(orig));
		return;
	}

	us_abortcommand("Bad COLOR option: %s", par[0]);
}

void us_commandfile(INTSML count, char *par[])
{
	REGISTER FILE *in;
	REGISTER INTSML verbose;
	REGISTER char *pp;
	char *filename;
	REGISTER MACROPACK *lastmacropack;
	extern COMCOMP us_colorreadp;

	if (count >= 2 && namesamen(par[1], "verbose", (INTSML)(strlen(par[1]))) == 0) verbose = 1; else
		verbose = 0;

	if (count == 0)
	{
		count = ttygetparam("macro/Command file: ", &us_colorreadp, MAXPARS, par);
		if (count == 0)
		{
			us_abortedmsg();
			return;
		}
	}
	pp = par[0];

	in = xopen(pp, FILETYPEMACRO, el_libdir, &filename);
	if (in == NULL)
	{
		us_abortcommand("Command file %s not found", pp);
		return;
	}

	/* create a new macro package entry */
	lastmacropack = us_curmacropack;
	us_curmacropack = us_newmacropack(pp);

	/* read the commands */
	us_docommands(in, verbose, "");
	xclose(in);

	/* now there is no macro package */
	us_curmacropack = lastmacropack;
}

void us_constraint(INTSML count, char *par[])
{
	REGISTER INTSML l, c;
	REGISTER char *pp;

	/* get the name of the constraint solver */
	if (count <= 1)
	{
		us_abortcommand("Usage: constraint (use | tell) SOLVER [OPTIONS]");
		return;
	}
	for(c=0; el_constraints[c].conname != 0; c++)
		if (namesame(el_constraints[c].conname, par[1]) == 0) break;
	if (el_constraints[c].conname == 0)
	{
		us_abortcommand("No constraint solver called %s", par[1]);
		return;
	}

	l = strlen(pp = par[0]);

	if (namesamen(pp, "tell", l) == 0 && l >= 1)
	{
		(*(el_constraints[c].setmode))((INTSML)(count-2), &par[2]);
		return;
	}

	if (namesamen(pp, "use", l) == 0 && l >= 1)
	{
		if (el_curconstraint == &el_constraints[c])
		{
			ttyputerr("Already using that constraint solver");
			return;
		}
		us_clearhighlightcount();
		(void)setvalkey((INTBIG)us_aid, VAID, us_current_constraint,
			(INTBIG)&el_constraints[c], VCONSTRAINT|VDONTSAVE);
		ttyputmsg("Switching to %s", el_constraints[c].condesc);
		return;
	}

	us_abortcommand("Bad CONSTRAINT option: %s", pp);
}

void us_copyfacet(INTSML count, char *par[])
{
	REGISTER NODEPROTO *np, *onp, *pnp;
	REGISTER char *pt, *ppt, *lastch;
	extern COMCOMP us_copyfacetp;
	REGISTER PORTPROTO *pp, *rpp;
	REGISTER NODEINST *ni, *newni;
	REGISTER VIEW *nview;
	REGISTER INTBIG xc, yc, lowx, highx, lowy, highy, i;
	REGISTER INTSML newang, newtran, save, quiet;
	XARRAY localtrans, ntrans, trans;
	INTBIG newx, newy;

	/* get name of old facet */
	if (count == 0)
	{
		count = ttygetparam("Old facet name: ", &us_copyfacetp, MAXPARS, par);
		if (count == 0)
		{
			us_abortedmsg();
			return;
		}
	}
	np = getnodeproto(par[0]);
	if (np == NONODEPROTO)
	{
		us_abortcommand("Cannot find source facet %s", par[0]);
		return;
	}
	if (np->index != 0)
	{
		us_abortcommand("Cannot copy primitives");
		return;
	}

	/* get name of new facet */
	quiet = 0;
	if (count <= 1) pt = np->cell->cellname; else
	{
		pt = par[1];
		if (namesame(pt, "{ic}") == 0 || namesame(pt, "{sk}") == 0)
		{
			(void)initinfstr();
			(void)addstringtoinfstr(np->cell->cellname);
			(void)addstringtoinfstr(par[1]);
			pt = returninfstr();
		}
		if (count > 2 && namesamen(par[2], "quiet", (INTSML)strlen(par[2])) == 0)
			quiet = 1;
	}

	/* see if icon generation is requested */
	i = strlen(pt);
	if (count > 1 && i > 4 && namesame(&pt[i-4], "{ic}") == 0 && np->cellview != el_iconview)
	{
		/* cannot iconize text-only views */
		if ((np->cellview->viewstate&TEXTVIEW) != 0)
		{
			us_abortcommand("Cannot iconize textual views: only layout and schematic");
			return;
		}

		/* generate the icon */
		onp = us_makeiconfacet(np->firstportproto, np->cell->cellname, pt, whattech(np), el_curlib);
		if (onp == NONODEPROTO) return;
		if (quiet == 0)
			ttyputmsg("Facet %s created with an iconic representation of %s",
				describenodeproto(onp), describenodeproto(np));
		return;
	}

	/* see if skeletonization is requested */
	i = strlen(pt);
	if (count > 1 && i > 4 && namesame(&pt[i-4], "{sk}") == 0 && np->cellview != el_skeletonview)
	{
		/* cannot skeletonize text-only views */
		if ((np->cellview->viewstate&TEXTVIEW) != 0)
		{
			us_abortcommand("Cannot skeletonize textual views: only layout");
			return;
		}

		/* warn if skeletonizing nonlayout views */
		if (np->cellview != el_unknownview && np->cellview != el_layoutview)
			ttyputmsg("Warning: skeletonization only makes sense for layout facets, not %s",
				np->cellview->viewname);

		onp = np;
		np = newnodeproto(pt, el_curlib);
		if (np == NONODEPROTO)
		{
			us_abortcommand("Cannot create %s", pt);
			return;
		}

		/* place all exported ports in the new facet */
		lowx = highx = (onp->lowx + onp->highx) / 2;
		lowy = highy = (onp->lowy + onp->highy) / 2;
		for(pp = onp->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
		{
			/* make a transformation matrix for the node that is exported */
			ni = pp->subnodeinst;
			rpp = pp->subportproto;
			newang = ni->rotation;
			newtran = ni->transpose;
			makerot(ni, trans);
			while (ni->proto->index == 0)
			{
				maketrans(ni, localtrans);
				transmult(localtrans, trans, ntrans);
				ni = rpp->subnodeinst;
				rpp = rpp->subportproto;
				if (ni->transpose == 0) newang = ni->rotation + newang; else
					newang = ni->rotation + 3600 - newang;
				newtran = (newtran + ni->transpose) & 1;
				makerot(ni, localtrans);
				transmult(localtrans, ntrans, trans);
			}

			/* create this node */
			xc = (ni->lowx + ni->highx) / 2;   yc = (ni->lowy + ni->highy) / 2;
			xform(xc, yc, &newx, &newy, trans);
			newx -= (ni->highx - ni->lowx) / 2;
			newy -= (ni->highy - ni->lowy) / 2;
			newang = newang % 3600;   if (newang < 0) newang += 3600;
			newni = newnodeinst(ni->proto, newx, newx+ni->highx-ni->lowx,
				newy, newy+ni->highy-ni->lowy, newtran, newang, np);
			if (newni == NONODEINST)
			{
				us_abortcommand("Cannot create node in this facet");
				return;
			}
			endobjectchange((INTBIG)newni, VNODEINST);
			lowx = mini(lowx, newx);
			highx = maxi(highx, newx+ni->highx-ni->lowx);
			lowy = mini(lowy, newy);
			highy = maxi(highy, newy+ni->highy-ni->lowy);

			/* export the port from the node */
			(void)newportproto(np, newni, rpp, pp->protoname);
		}

		/* make sure facet is the same size */
		if (lowx > onp->lowx)
		{
			i = (onp->highy+onp->lowy)/2 - (gen_invispinprim->highy-gen_invispinprim->lowy)/2;
			(void)newnodeinst(gen_invispinprim, onp->lowx, onp->lowx+gen_invispinprim->highx-gen_invispinprim->lowx,
				i, i+gen_invispinprim->highy-gen_invispinprim->lowy, 0, 0, np);
		}
		if (highx < onp->highx)
		{
			i = (onp->highy+onp->lowy)/2 - (gen_invispinprim->highy-gen_invispinprim->lowy)/2;
			(void)newnodeinst(gen_invispinprim, onp->highx-(gen_invispinprim->highx-gen_invispinprim->lowx), onp->highx,
				i, i+gen_invispinprim->highy-gen_invispinprim->lowy, 0, 0, np);
		}
		if (lowy > onp->lowy)
		{
			i = (onp->highx+onp->lowx)/2 - (gen_invispinprim->highx-gen_invispinprim->lowx)/2;
			(void)newnodeinst(gen_invispinprim, i, i+gen_invispinprim->highx-gen_invispinprim->lowx,
				onp->lowy, onp->lowy+gen_invispinprim->highy-gen_invispinprim->lowy, 0, 0, np);
		}
		if (highy < onp->highy)
		{
			i = (onp->highx+onp->lowx)/2 - (gen_invispinprim->highx-gen_invispinprim->lowx)/2;
			(void)newnodeinst(gen_invispinprim, i, i+gen_invispinprim->highx-gen_invispinprim->lowx,
				onp->highy-(gen_invispinprim->highy-gen_invispinprim->lowy), onp->highy, 0, 0,np);
		}

		/* place the actual origin of the facet inside */
		(void)initinfstr();
		(void)addstringtoinfstr(onp->cell->lib->libfile);
		(void)addtoinfstr(':');
		(void)addstringtoinfstr(nldescribenodeproto(onp));
		(void)setval((INTBIG)np, VNODEPROTO, "original_facet",
			(INTBIG)returninfstr(), VSTRING);

		if (quiet == 0)
			ttyputmsg("Facet %s created with a skeletal representation of %s",
				describenodeproto(np), describenodeproto(onp));
		return;
	}

	/* if going within the same library, let the database do it */
	if (np->cell->lib == el_curlib)
	{
		onp = copynodeproto(np, el_curlib, pt);
		if (onp == NONODEPROTO)
		{
			ttyputerr("Error copying facet");
			return;
		}
		if (onp->cell != np->cell || onp->cellview != np->cellview)
		{
			if (quiet == 0)
				ttyputmsg("Facet %s copied to %s", describenodeproto(np),
					describenodeproto(onp));
		} else
		{
			if (quiet == 0)
				ttyputmsg("New version of facet %s created, old is %s",
					describenodeproto(onp), describenodeproto(np));

			/* set current nodeproto to itself to redisplay its name */
			if (np == getcurfacet())
				(void)setval((INTBIG)el_curlib, VLIBRARY, "curnodeproto", (INTBIG)np, VNODEPROTO);

			/* update display of any unexpanded old versions */
			for(ni = np->firstinst; ni != NONODEINST; ni = ni->nextinst)
			{
				if ((ni->userbits&NEXPAND) != 0) continue;
				startobjectchange((INTBIG)ni, VNODEINST);
				endobjectchange((INTBIG)ni, VNODEINST);
			}
		}
		return;
	}

	/* mark those facets that already exist in this library */
	for(pnp = el_curlib->firstnodeproto; pnp != NONODEPROTO; pnp = pnp->nextnodeproto) pnp->temp1 = 1;

	/* get the new view type */
	nview = np->cellview;
	for(ppt = pt; *ppt != 0; ppt++) if (*ppt == '{') break;
	if (*ppt == '{')
	{
		lastch = &pt[strlen(pt)-1];
		if (*lastch != '}')
		{
			us_abortcommand("View name '%s'must end with a '}'", &ppt[1]);
			return;
		}
		*lastch = 0;
		nview = getview(&ppt[1]);
		*lastch = '}';
		if (nview == NOVIEW)
		{
			us_abortcommand("Unknown view abbreviation: %s", ppt);
			return;
		}
	}

	save = *ppt;
	*ppt = 0;
	onp = us_copyrecursively(np, pt, nview);
	*ppt = save;
	if (onp == NONODEPROTO) us_abortcommand("Cannot copy facet"); else
	{
		if (quiet == 0)
			ttyputmsg("Facet %s copied to %s", describenodeproto(np), describenodeproto(onp));
	}
}

void us_create(INTSML count, char *par[])
{
	REGISTER NODEPROTO *np, *rnp, *curfacet;
	REGISTER NODEINST *ni, *newno, *ni2;
	REGISTER PORTPROTO *pp, *ppt, *fpt, *pp1, *pp2;
	PORTPROTO *fromport, *toport;
	REGISTER ARCPROTO *ap;
	REGISTER ARCINST *ai, *newai;
	REGISTER TECHNOLOGY *tech;
	NODEINST *ani[2];
	REGISTER INTSML ang, l, remainhighlighted, ishor, isver, getcontents,
		pangle, join, waitfordown, angledcreate;
	REGISTER INTBIG px, py, bestdist, bestdist1, bestdist2, wid, truewid, i, k,
		bits1, bits2, dist;
	INTBIG nx, ny, cx, cy, x, y, xp[2], yp[2], xcur, ycur, otheralign;
	PORTPROTO *app[2];
	REGISTER VARIABLE *var;
	static POLYGON *poly = NOPOLYGON;
	ARCINST *alt1, *alt2;
	extern COMCOMP us_createxp, us_createyp;
	char *oldarcname;
	HIGHLIGHT newhigh;
	extern AIDENTRY *net_aid;
	REGISTER GEOM *foundgeom, *geom;
	GEOM *fromgeom, *togeom;

	/* get polygon */
	if (poly == NOPOLYGON) poly = allocpolygon(4, us_aid->cluster);

	/* make sure there is a facet being edited */
	if (us_needwindow() != 0) return;
	if ((el_curwindow->state&WINDOWTYPE) != DISPWINDOW)
	{
		us_abortcommand("Can only place components in graphical editing windows");
		return;
	}
	curfacet = us_needfacet();
	if (curfacet == NONODEPROTO) return;

	remainhighlighted = getcontents = waitfordown = angledcreate = join = 0;
	while (count > 0)
	{
		l = strlen(par[0]);
		if (namesamen(par[0], "remain-highlighted", l) == 0 && l >= 1)
		{
			remainhighlighted++;
			count--;
			par++;
			continue;
		}
		if (namesamen(par[0], "contents", l) == 0 && l >= 1)
		{
			getcontents++;
			count--;
			par++;
			continue;
		}
		if (namesamen(par[0], "wait-for-down", l) == 0 && l >= 1)
		{
			waitfordown++;
			count--;
			par++;
			continue;
		}
		if ((namesamen(par[0], "angle", l) == 0 ||
			namesamen(par[0], "join-angle", l) == 0) && l >= 1)
		{
			if (namesamen(par[0], "join-angle", l) == 0) join++;
			angledcreate++;
			count--;
			par++;
			if (count >= 1)
			{
				ang = atofr(par[0])*10/WHOLE;
				count--;
				par++;
			} else ang = ((us_curarcproto->userbits&AANGLEINC) >> AANGLEINCSH) * 10;
		}
		break;
	}

	/* create a node or arc */
	if (count == 0)
	{
		/* see if two objects are selected (and arc should connect them) */
		if (us_gettwoobjects(&fromgeom, &fromport, &togeom, &toport) == 0)
		{
			/* two objects highlighted: create an arc between them */
			if (us_demandxy(&xcur, &ycur)) return;
			gridalign(&xcur, &ycur, us_alignment);
			if (geomparent(fromgeom) != geomparent(togeom))
			{
				us_abortcommand("Cannot run an arc between different facets");
				return;
			}

			/* run an arc from "fromgeom" to "togeom" */
			if (remainhighlighted != 0) us_pushhighlight();
			us_clearhighlightcount();
			if (angledcreate == 0) ang = 900;
			if (aconnect(fromgeom, fromport, togeom, toport, us_curarcproto,
				xcur, ycur, &alt1, &alt2, ang) == NOARCINST) return;

			/* if an arc was highlighted and it got broken, fix the highlight */
			geom = fromgeom;
			if (geom->entrytype == OBJARCINST && (geom->entryaddr.ai->userbits&DEADA) != 0)
			{
				geom = togeom;
				if (geom->entrytype == OBJARCINST && (geom->entryaddr.ai->userbits&DEADA) != 0) return;
			}

			/* unless indicated, leave only one object highlighted */
			if (remainhighlighted != 0) (void)us_pophighlight(0); else
			{
				newhigh.status = HIGHFROM;
				newhigh.facet = geomparent(geom);
				newhigh.fromgeom = geom;
				newhigh.fromport = NOPORTPROTO;
				(void)us_addhighlight(&newhigh);
			}
			return;
		}

		/* if no "angle" specified, simply create a node */
		if (angledcreate == 0)
		{
			/* determine the node type being created */
			np = us_curnodeproto;
			if (np == NONODEPROTO)
			{
				us_abortcommand("No selected node proto to create");
				return;
			}

			/* disallow creating if lock is on */
			if (us_protolocked(np) != 0) return;

			/* use the icon facet if one exists and contents wasn't requested */
			if (getcontents == 0)
			{
				rnp = iconview(np);
				if (rnp != NONODEPROTO) np = rnp;
			}

			/* create a node instance: check for recursion first */
			if (isachildof(curfacet, np))
			{
				us_abortcommand("Cannot create the node: it would be recursive");
				return;
			}

			/* disallow creation of second facet center */
			if (np == gen_facetcenterprim && us_alreadyfacetcenter(curfacet) != 0) return;

			/* adjust the cursor position if selecting interactively */
			us_clearhighlightcount();
			if ((us_state&NONPERSISTENTCURNODE) != 0) us_setnodeproto(np);

			/* get placement angle */
			var = getvalkey((INTBIG)us_aid, VAID, VINTEGER, us_placement_angle);
			if (var == NOVARIABLE) pangle = 0; else pangle = var->addr;

			if ((us_aid->aidstate&INTERACTIVE) != 0 && us_graphicshas(CANTRACKCURSOR) != 0)
			{
				us_createinit(0, 0, np, pangle, 0);
				trackcursor(waitfordown, us_ignoreup, us_createbegin, us_dragdown,
					us_stoponchar, us_dragup, TRACKDRAGGING);
				if (el_pleasestop != 0) return;
			}
			if (us_demandxy(&xcur, &ycur)) return;
			gridalign(&xcur, &ycur, us_alignment);

			/* create and display the nodeinst */
			corneroffset(NONODEINST, np, (INTSML)(pangle%3600), (INTSML)(pangle/3600), &cx, &cy);
			xcur -= cx;   ycur -= cy;
			newno = newnodeinst(np, xcur, xcur+np->highx-np->lowx, ycur,
				ycur+np->highy-np->lowy, (INTSML)(pangle/3600), (INTSML)(pangle%3600), curfacet);
			if (newno == NONODEINST)
			{
				us_abortcommand("Cannot create node");
				return;
			}
			if ((np->userbits&WANTNEXPAND) != 0) newno->userbits |= NEXPAND;
			endobjectchange((INTBIG)newno, VNODEINST);

			newhigh.status = HIGHFROM;
			newhigh.facet = newno->parent;
			newhigh.fromgeom = newno->geom;
			newhigh.fromport = NOPORTPROTO;
			newhigh.frompoint = 0;
			(void)us_addhighlight(&newhigh);
			return;
		}

		/* "create angle": get cursor co-ordinates */
		if (us_demandxy(&xcur, &ycur)) return;
		gridalign(&xcur, &ycur, us_alignment);

		/* get nodeinst from which to draw */
		ni = (NODEINST *)us_getobject(OBJNODEINST, 0);
		if (ni == NONODEINST) return;

		/* ensure that cursor is in proper window */
		if (ni->parent != curfacet)
		{
			us_abortcommand("Cursor must be in the same facet as the highlighted node");
			return;
		}

		/* find the closest port to the cursor */
		var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_highlighted);
		if (var != NOVARIABLE)
			(void)us_makehighlight(((char **)var->addr)[0], &newhigh);
		bestdist1 = bestdist2 = HUGEINT;
		ppt = fpt = NOPORTPROTO;
		for(pp = ni->proto->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
		{
			if (newhigh.fromport != NOPORTPROTO && newhigh.fromport != pp) continue;
			shapeportpoly(ni, pp, poly, 0);
			k = polydistance(poly, xcur, ycur);

			/* find the closest port to the cursor */
			if (k < bestdist1)
			{
				bestdist1 = k;
				ppt = pp;  /* pp is the closest port to (xcur, ycur) */
			}

			/* find the closest port that can connect with the current arc */
			if (k < bestdist2)
			{
				for (i=0; pp->connects[i] != NOARCPROTO; i++)
				{
					if (pp->connects[i] == us_curarcproto)
					{
						/* can connect */
						bestdist2 = k;
						fpt = pp;
					}
				}
			}
		}

		/* error if there is no port */
		if (ppt == NOPORTPROTO)
		{
			us_abortcommand("Cannot run wires out of this node");
			return;
		}

		/*
		 * now ppt is the closest port, and fpt is the closest port that can
		 * connect with the current arc.
		 */
		if (fpt == NOPORTPROTO)
		{
			/* no arcproto of correct type: first reset all arcprotos */
			for(tech = el_technologies; tech != NOTECHNOLOGY; tech = tech->nexttechnology)
				for(ap = tech->firstarcproto; ap != NOARCPROTO; ap = ap->nextarcproto)
					ap->userbits &= ~CANCONNECT;

			/* now set all arcs that can connect this node's closest port */
			for(i=0; ppt->connects[i] != NOARCPROTO; i++) ppt->connects[i]->userbits |= CANCONNECT;

			/* see if current arcproto is acceptable */
			if ((us_curarcproto->userbits&CANCONNECT) == 0)
			{
				/* search this technology for another arcproto */
				for(ap = el_curtech->firstarcproto; ap != NOARCPROTO; ap = ap->nextarcproto)
				{
					if ((ap->userbits&CANCONNECT) == 0) continue;
					if (getpinproto(ap) == NONODEPROTO) return;
					us_setarcproto(ap, 0);
					break;
				}

				/* if nothing in this technology can connect, see what can */
				if (ap == NOARCPROTO && ppt->connects[0] != NOARCPROTO)
				{
					ap = ppt->connects[0];
					if (getpinproto(ap) == NONODEPROTO) return;
					us_setarcproto(ap, 0);
				}

				/* error if no acceptable arcprotos */
				if (ap == NOARCPROTO)
				{
					us_abortcommand("No ports on this node");
					return;
				}
			}
		} else ppt = fpt;

		/* turn off highlighting */
		us_clearhighlightcount();

		/* compute widest other arc on this node */
		truewid = maxi(us_curarcproto->nominalwidth, us_widestarcinst(us_curarcproto, ni, ppt));
		wid = truewid - arcwidthoffset(us_curarcproto);

		/* specify precise location for new port */
		poly->xv[0] = xcur;   poly->yv[0] = ycur;   poly->count = 1;
		shapeportpoly(ni, ppt, poly, 1);

		/* adjust port area to account for width of arc */
		reduceportpoly(poly, ni, ppt, wid);

		/* determine pin for making the connection */
		np = getpinproto(us_curarcproto);
		if (np == NONODEPROTO) return;

		/* get placement angle */
		var = getvalkey((INTBIG)us_aid, VAID, VINTEGER, us_placement_angle);
		if (var == NOVARIABLE) pangle = 0; else pangle = var->addr;

		/* determine location of pin */
		px = (np->highx - np->lowx) / 2;   py = (np->highy - np->lowy) / 2;
		corneroffset(NONODEINST, np, (INTSML)(pangle%3600), (INTSML)(pangle/3600), &cx, &cy);

		/* adjust the cursor position if selecting interactively */
		switch (poly->style)
		{
			case FILLED:
			case FILLEDRECT:
			case CROSSED:
			case CROSS:
			case BIGCROSS:
			case TEXTCENT:
			case TEXTTOP:
			case TEXTBOT:
			case TEXTLEFT:
			case TEXTRIGHT:
			case TEXTTOPLEFT:
			case TEXTBOTLEFT:
			case TEXTTOPRIGHT:
			case TEXTBOTRIGHT:
			case TEXTBOX:
			case DISC:
				getcenter(poly, &x, &y);
				break;

			default:
				x = xcur;   y = ycur;
				closestpoint(poly, &x, &y);
				break;
		}
		if ((us_aid->aidstate&INTERACTIVE) != 0 && us_graphicshas(CANTRACKCURSOR) != 0)
		{
			us_createinit(x-px+cx, y-py+cy, np, ang, join);
			trackcursor(waitfordown, us_ignoreup, us_createabegin, us_createadown,
				us_stoponchar, us_createaup, TRACKDRAGGING);
			if (el_pleasestop != 0) return;
			if (us_demandxy(&xcur, &ycur) != 0) return;
			gridalign(&xcur, &ycur, us_alignment);
		}

		/* determine the proper point on the polygon to draw radials */
		us_getslide(ang, x-px+cx, y-py+cy, xcur, ycur, &nx, &ny);

		/* adjust for the box corner */
		nx -= cx;
		ny -= cy;

		/* find the object under the cursor */
		if (join == 0) foundgeom = NOGEOM; else
			foundgeom = us_getclosest(nx+px, ny+py, ni->parent);
		if (foundgeom != NOGEOM)
		{
			/* found another object: connect to it */
			fpt = NOPORTPROTO;
			bestdist = HUGEINT;
			if (foundgeom->entrytype == OBJNODEINST)
			{
				/* find the closest port */
				ni = foundgeom->entryaddr.ni;
				for(pp = ni->proto->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
				{
					shapeportpoly(ni, pp, poly, 0);
					dist = polydistance(poly, xcur, ycur);
					if (dist > bestdist) continue;
					bestdist = dist;   fpt = pp;
				}
			}
			(void)aconnect(newhigh.fromgeom, newhigh.fromport, foundgeom,
				fpt, us_curarcproto, xcur, ycur, &alt1, &alt2, ang);

			/* if an arc will be highlighted and it broke, fix the highlight */
			if (foundgeom->entrytype == OBJARCINST &&
				(foundgeom->entryaddr.ai->userbits&DEADA) != 0) return;

			/* highlight the found object */
			newhigh.status = HIGHFROM;
			newhigh.facet = ni->parent;
			newhigh.fromgeom = foundgeom;
			newhigh.fromport = fpt;
			newhigh.frompoint = 0;
			(void)us_addhighlight(&newhigh);
			return;
		}

		/* see if arc edge alignment is appropriate */
		if (us_edgealignment != 0 && (x == nx || y == ny))
		{
			if (x != nx) isver = 0; else
			{
				isver = 1;
				ny = us_alignvalue(ny + wid/2, us_edgealignment, &otheralign) - wid/2;
			}
			if (y != ny) ishor = 0; else
			{
				ishor = 1;
				nx = us_alignvalue(nx + wid/2, us_edgealignment, &otheralign) - wid/2;
			}

			shapeportpoly(ni, ppt, poly, 0);
			i = us_alignvalue(x + wid/2, us_edgealignment, &otheralign) - wid/2;
			otheralign -= wid/2;
			if (isinside(i, y, poly) != 0)
			{
				if (isver != 0) nx = i;
				x = i;
			} else if (isinside(otheralign, y, poly) != 0)
			{
				if (isver != 0) nx = otheralign;
				x = otheralign;
			}

			i = us_alignvalue(y + wid/2, us_edgealignment, &otheralign) - wid/2;
			otheralign -= wid/2;
			if (isinside(x, i, poly) != 0)
			{
				if (ishor != 0) ny = i;
				y = i;
			} else if (isinside(x, otheralign, poly) != 0)
			{
				if (ishor != 0) ny = otheralign;
				y = otheralign;
			}
		}

		/* create the pin and the arcinst to it */
		newno = newnodeinst(np, nx, nx+np->highx-np->lowx, ny,
			ny+np->highy-np->lowy, (INTSML)(pangle/3600), (INTSML)(pangle%3600), curfacet);
		if (newno == NONODEINST)
		{
			us_abortcommand("Cannot create pin");
			return;
		}
		nx += px;   ny += py;
		if ((np->userbits&WANTNEXPAND) != 0) newno->userbits |= NEXPAND;
		endobjectchange((INTBIG)newno, VNODEINST);
		if (((ppt->userbits&PORTARANGE) >> PORTARANGESH) == 180)
		{
			(void)us_runarcinst(ni, ppt, x, y, newno,
				newno->proto->firstportproto, nx, ny, us_curarcproto, truewid);
		} else
		{
			k = us_bottomrecurse(ni, ppt, 0) % 3600;
			if (k < 0) k += 3600;
			xcur = x + mult(cosine((INTSML)k), ni->highx-ni->lowx);
			ycur = y + mult(sine((INTSML)k), ni->highy-ni->lowy);
			(void)aconnect(ni->geom, ppt, newno->geom, newno->proto->firstportproto,
				us_curarcproto, xcur, ycur, &alt1, &alt2, ang);
		}

		newhigh.status = HIGHFROM;
		newhigh.facet = newno->parent;
		newhigh.fromgeom = newno->geom;
		newhigh.fromport = NOPORTPROTO;
		newhigh.frompoint = 0;
		(void)us_addhighlight(&newhigh);
		return;
	}

	/* handle the absolute placement option */
	if (namesamen(par[0], "to", l) == 0 && l >= 1)
	{
		/* get angle from the nodeinst */
		if (count < 2)
		{
			count = ttygetparam("X position: ", &us_createxp, MAXPARS-1, &par[1]) + 1;
			if (count == 1)
			{
				us_abortedmsg();
				return;
			}
		}
		nx = atola(par[1]);
		if (count < 3)
		{
			count = ttygetparam("Y position: ", &us_createyp, MAXPARS-2, &par[2]) + 2;
			if (count == 2)
			{
				us_abortedmsg();
				return;
			}
		}
		ny = atola(par[2]);

		/* disallow copying if lock is on */
		np = us_curnodeproto;
		if (us_protolocked(np) != 0) return;

		/* disallow creation of second facet center */
		if (np == gen_facetcenterprim && us_alreadyfacetcenter(curfacet) != 0) return;

		/* get placement angle */
		var = getvalkey((INTBIG)us_aid, VAID, VINTEGER, us_placement_angle);
		if (var == NOVARIABLE) pangle = 0; else pangle = var->addr;

		corneroffset(NONODEINST, np, (INTSML)(pangle%3600), (INTSML)(pangle/3600), &cx, &cy);
		nx -= cx;   ny -= cy;
		newno = newnodeinst(np, nx, nx+np->highx-np->lowx, ny, ny+np->highy-np->lowy,
			(INTSML)(pangle/3600), (INTSML)(pangle%3600), curfacet);
		if (newno == NONODEINST)
		{
			us_abortcommand("Cannot create pin");
			return;
		}
		if ((np->userbits&WANTNEXPAND) != 0) newno->userbits |= NEXPAND;
		endobjectchange((INTBIG)newno, VNODEINST);

		us_clearhighlightcount();
		newhigh.status = HIGHFROM;
		newhigh.facet = newno->parent;
		newhigh.fromgeom = newno->geom;
		newhigh.fromport = NOPORTPROTO;
		newhigh.frompoint = 0;
		(void)us_addhighlight(&newhigh);
		return;
	}

	if (namesamen(par[0], "insert", l) == 0 && l >= 1)
	{
		if (us_demandxy(&xcur, &ycur)) return;
		gridalign(&xcur, &ycur, us_alignment);

		/* get the highlighted arc */
		ai = (ARCINST *)us_getobject(OBJARCINST, 0);
		if (ai == NOARCINST) return;
		ap = ai->proto;

		/* get the node to insert */
		np = us_curnodeproto;
		if (np == NONODEPROTO)
		{
			us_abortcommand("Must get a node prototype for insertion on arc");
			return;
		}

		/* disallow creating if lock is on */
		if (us_protolocked(np) != 0) return;

		/* disallow creation of second facet center */
		if (np == gen_facetcenterprim && us_alreadyfacetcenter(curfacet) != 0) return;

		/* turn off highlighting */
		us_clearhighlightcount();
		if ((us_state&NONPERSISTENTCURNODE) != 0) us_setnodeproto(np);

		/* adjust position interactively if possible */
		if ((us_aid->aidstate&INTERACTIVE) != 0 &&
			us_graphicshas(CANTRACKCURSOR) != 0)
		{
			us_createinsinit(ai, np);
			trackcursor(waitfordown, us_ignoreup, us_createabegin, us_createinsdown,
				us_stoponchar, us_dragup, TRACKDRAGGING);
			if (el_pleasestop != 0) return;
		}
		if (us_demandxy(&xcur, &ycur)) return;
		gridalign(&xcur, &ycur, us_alignment);

		/* find two ports on this node that connect to the arc */
		pp1 = pp2 = NOPORTPROTO;
		for(pp = np->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
		{
			/* make sure this port can connect to this arc */
			for(i=0; pp->connects[i] != NOARCPROTO; i++)
				if (pp->connects[i] == ap) break;
			if (pp->connects[i] == NOARCPROTO) continue;

			/* save the port */
			if (pp1 == NOPORTPROTO) pp1 = pp; else
				if (pp2 == NOPORTPROTO) pp2 = pp;
		}

		/* make sure there are two ports for connection */
		if (pp1 == NOPORTPROTO)
		{
			us_abortcommand("No ports on %s node connect to %s arcs", describenodeproto(np),
				describearcproto(ap));
			return;
		}
		if (pp2 == NOPORTPROTO) pp2 = pp1;

		/* determine position of the new node */
		ang = ((ai->userbits&AANGLE) >> AANGLESH) * 10;
		(void)intersect(ai->end[0].xpos, ai->end[0].ypos, ang, xcur,
			ycur, (ang+900)%3600, &nx, &ny);
		cx = mini(ai->end[0].xpos, ai->end[1].xpos);
		if (nx < cx) nx = cx;
		cx = maxi(ai->end[0].xpos, ai->end[1].xpos);
		if (nx > cx) nx = cx;
		cy = mini(ai->end[0].ypos, ai->end[1].ypos);
		if (ny < cy) ny = cy;
		cy = maxi(ai->end[0].ypos, ai->end[1].ypos);
		if (ny > cy) ny = cy;
		px = np->highx - np->lowx;   py = np->highy - np->lowy;
		x = nx - px/2;   y = ny - py/2;

		/* create the new node */
		ni = newnodeinst(np, x,x+px, y,y+py, 0, 0, curfacet);
		if (ni == NONODEINST)
		{
			us_abortcommand("Cannot create node %s", describenodeproto(np));
			return;
		}
		endobjectchange((INTBIG)ni, VNODEINST);
		portposition(ni, pp1, &nx, &ny);
		portposition(ni, pp2, &cx, &cy);

		/* see if any rotation needs to be applied to this node */
		if (pp1 != pp2)
		{
			k = figureangle(nx, ny, cx, cy);
			if (k != ang)
			{
				if ((k+1800)%3600 == ang)
				{
					/* simple 180 degree rotation: reverse the ports */
					pp = pp1;   pp1 = pp2;   pp2 = pp;
				} else
				{
					/* complex rotation: rotate the node */
					ang -= k;   if (ang < 0) ang += 3600;
					startobjectchange((INTBIG)ni, VNODEINST);
					modifynodeinst(ni, 0, 0, 0, 0, ang, 0);
					endobjectchange((INTBIG)ni, VNODEINST);
					portposition(ni, pp1, &nx, &ny);
					portposition(ni, pp2, &cx, &cy);
				}
			}
		}

		/* see if node edge alignment is appropriate */
		if (us_edgealignment != 0 && (ai->end[0].xpos == ai->end[1].xpos ||
			ai->end[0].ypos == ai->end[1].ypos))
		{
			px = us_alignvalue(x, us_edgealignment, &otheralign);
			py = us_alignvalue(y, us_edgealignment, &otheralign);
			if (px != x || py != y)
			{
				/* shift the node and make sure the ports are still valid */
				startobjectchange((INTBIG)ni, VNODEINST);
				modifynodeinst(ni, px-x, py-y, px-x, py-y, 0, 0);
				endobjectchange((INTBIG)ni, VNODEINST);
				(void)shapeportpoly(ni, pp1, poly, 0);
				if (isinside(nx, ny, poly) == 0) getcenter(poly, &nx, &ny);
				(void)shapeportpoly(ni, pp2, poly, 0);
				if (isinside(cx, cy, poly) == 0) getcenter(poly, &cx, &cy);
			}
		}

		/* now save the arc information and delete it */
		for(i=0; i<2; i++)
		{
			xp[i] = ai->end[i].xpos;
			yp[i] = ai->end[i].ypos;
			app[i] = ai->end[i].portarcinst->proto;
			ani[i] = ai->end[i].nodeinst;
		}
		wid = ai->width;
		bits1 = bits2 = ai->userbits;
		if ((bits1&ISNEGATED) != 0)
		{
			if ((bits1&REVERSEEND) == 0) bits2 &= ~ISNEGATED; else
				bits1 &= ~ISNEGATED;
		}
		var = getvalkey((INTBIG)ai, VARCINST, VSTRING, el_arc_name);
		if (var == NOVARIABLE) oldarcname = 0; else
		{
			(void)allocstring(&oldarcname, (char *)var->addr, el_tempcluster);
			(void)askaid(net_aid, "delname", (INTBIG)oldarcname,
				(INTBIG)curfacet);
		}
		startobjectchange((INTBIG)ai, VARCINST);
		(void)killarcinst(ai);

		/* create the two new arcs */
		newai = newarcinst(ap, wid, bits1, ani[0], app[0], xp[0], yp[0], ni, pp1, nx, ny, curfacet);
		if (oldarcname != 0 && newai != NOARCINST)
		{
			(void)askaid(net_aid, "setname", (INTBIG)newai, (INTBIG)oldarcname);
			(void)setvalkey((INTBIG)newai, VARCINST, el_arc_name, (INTBIG)oldarcname, VSTRING|VDISPLAY);
			efree(oldarcname);
		}
		if (newai != NOARCINST) endobjectchange((INTBIG)newai, VARCINST);
		newai = newarcinst(ap, wid, bits2, ni, pp2, cx, cy, ani[1],app[1],xp[1],yp[1], curfacet);
		if (newai != NOARCINST) endobjectchange((INTBIG)newai, VARCINST);

		/* highlight the node */
		newhigh.status = HIGHFROM;
		newhigh.facet = curfacet;
		newhigh.fromgeom = ni->geom;
		newhigh.fromport = NOPORTPROTO;
		newhigh.frompoint = 0;
		(void)us_addhighlight(&newhigh);
		return;
	}

	if (namesamen(par[0], "breakpoint", l) == 0 && l >= 1)
	{
		if (us_demandxy(&xcur, &ycur)) return;
		gridalign(&xcur, &ycur, us_alignment);

		/* get the highlighted arc */
		ai = (ARCINST *)us_getobject(OBJARCINST, 0);
		if (ai == NOARCINST) return;
		ap = ai->proto;

		/* get the pin to insert */
		np = getpinproto(ap);
		if (np == NONODEPROTO)
		{
			us_abortcommand("Cannot determine pin type to insert in arc");
			return;
		}
		ppt = np->firstportproto;

		/* turn off highlighting */
		us_clearhighlightcount();
		if ((us_state&NONPERSISTENTCURNODE) != 0) us_setnodeproto(np);

		/* adjust position interactively if possible */
		if ((us_aid->aidstate&INTERACTIVE) != 0 &&
			us_graphicshas(CANTRACKCURSOR) != 0)
		{
			us_createinsinit(ai, np);
			trackcursor(waitfordown, us_ignoreup, us_createabegin, us_createinsdown,
				us_stoponchar, us_dragup, TRACKDRAGGING);
			if (el_pleasestop != 0) return;
		}
		if (us_demandxy(&xcur, &ycur)) return;
		gridalign(&xcur, &ycur, us_alignment);

		/* determine position of the new pins */
		ang = ((ai->userbits&AANGLE) >> AANGLESH) * 10;
		(void)intersect(ai->end[0].xpos, ai->end[0].ypos, ang, xcur, ycur, (ang+900)%3600, &nx, &ny);
		cx = mini(ai->end[0].xpos, ai->end[1].xpos);
		if (nx < cx) nx = cx;
		cx = maxi(ai->end[0].xpos, ai->end[1].xpos);
		if (nx > cx) nx = cx;
		cy = mini(ai->end[0].ypos, ai->end[1].ypos);
		if (ny < cy) ny = cy;
		cy = maxi(ai->end[0].ypos, ai->end[1].ypos);
		if (ny > cy) ny = cy;
		px = np->highx - np->lowx;   py = np->highy - np->lowy;
		x = nx - px/2;   y = ny - py/2;

		/* create the break pins */
		ni = newnodeinst(np, x,x+px, y,y+py, 0, 0, curfacet);
		if (ni == NONODEINST)
		{
			us_abortcommand("Cannot create pin %s", describenodeproto(np));
			return;
		}
		endobjectchange((INTBIG)ni, VNODEINST);
		ni2 = newnodeinst(np, x,x+px, y,y+py, 0, 0, curfacet);
		if (ni2 == NONODEINST)
		{
			us_abortcommand("Cannot create pin %s", describenodeproto(np));
			return;
		}
		endobjectchange((INTBIG)ni2, VNODEINST);

		/* get location of connection to these pins */
		portposition(ni, ppt, &nx, &ny);

		/* see if edge alignment is appropriate */
		if (us_edgealignment != 0 && (ai->end[0].xpos == ai->end[1].xpos ||
			ai->end[0].ypos == ai->end[1].ypos))
		{
			px = us_alignvalue(x, us_edgealignment, &otheralign);
			py = us_alignvalue(y, us_edgealignment, &otheralign);
			if (px != x || py != y)
			{
				/* shift the nodes and make sure the ports are still valid */
				startobjectchange((INTBIG)ni, VNODEINST);
				modifynodeinst(ni, px-x, py-y, px-x, py-y, 0, 0);
				endobjectchange((INTBIG)ni, VNODEINST);
				startobjectchange((INTBIG)ni2, VNODEINST);
				modifynodeinst(ni2, px-x, py-y, px-x, py-y, 0, 0);
				endobjectchange((INTBIG)ni2, VNODEINST);
				(void)shapeportpoly(ni, ppt, poly, 0);
				if (isinside(nx, ny, poly) == 0) getcenter(poly, &nx, &ny);
			}
		}

		/* now save the arc information and delete it */
		for(i=0; i<2; i++)
		{
			xp[i] = ai->end[i].xpos;
			yp[i] = ai->end[i].ypos;
			app[i] = ai->end[i].portarcinst->proto;
			ani[i] = ai->end[i].nodeinst;
		}
		wid = ai->width;
		bits1 = bits2 = ai->userbits;
		if ((bits1&ISNEGATED) != 0)
		{
			if ((bits1&REVERSEEND) == 0) bits2 &= ~ISNEGATED; else
				bits1 &= ~ISNEGATED;
		}
		var = getvalkey((INTBIG)ai, VARCINST, VSTRING, el_arc_name);
		if (var == NOVARIABLE) oldarcname = 0; else
		{
			(void)allocstring(&oldarcname, (char *)var->addr, el_tempcluster);
			(void)askaid(net_aid, "delname", (INTBIG)oldarcname,
				(INTBIG)curfacet);
		}
		startobjectchange((INTBIG)ai, VARCINST);
		(void)killarcinst(ai);

		/* create the new arcs */
		newai = newarcinst(ap, wid, bits1, ani[0], app[0], xp[0], yp[0], ni, ppt, nx, ny, curfacet);
		if (oldarcname != 0 && newai != NOARCINST)
		{
			(void)askaid(net_aid, "setname", (INTBIG)newai, (INTBIG)oldarcname);
			(void)setvalkey((INTBIG)newai, VARCINST, el_arc_name, (INTBIG)oldarcname, VSTRING|VDISPLAY);
			efree(oldarcname);
		}
		if (newai != NOARCINST) endobjectchange((INTBIG)newai, VARCINST);
		newai = newarcinst(ap, wid, bits2, ni2, ppt, nx, ny, ani[1],app[1],xp[1],yp[1], curfacet);
		if (newai != NOARCINST) endobjectchange((INTBIG)newai, VARCINST);
		bits2 &= ~ISNEGATED;
		ang = (ang + 900) % 3600;
		bits2 = (bits2 & ~AANGLE) | (((ang+5)/10) << AANGLESH);
		newai = newarcinst(ap, wid, bits2, ni, ppt, nx, ny, ni2, ppt, nx, ny, curfacet);
		if (newai != NOARCINST) endobjectchange((INTBIG)newai, VARCINST);

		/* highlight the arc */
		newhigh.status = HIGHFROM;
		newhigh.facet = curfacet;
		newhigh.fromgeom = newai->geom;
		newhigh.fromport = NOPORTPROTO;
		newhigh.frompoint = 0;
		(void)us_addhighlight(&newhigh);
		return;
	}

	us_abortcommand("Bad CREATE option: %s", par[0]);
}

void us_debug(INTSML count, char *par[])
{
	REGISTER NODEPROTO *np;
	REGISTER PORTPROTO *pp;
	REGISTER NODEINST *ni;
	REGISTER PORTARCINST *pi;
	REGISTER ARCINST *ai;
	REGISTER LIBRARY *lib;
	REGISTER VARIABLE *var;
	REGISTER INTSML i, j, l, let1, let2, let3;
	char line[20], *pt;
	extern INTSML db_printerrors, db_minrtnodesize, db_maxrtnodesize;
	HIGHLIGHT newhigh;

	if (count == 0) return;
	l = strlen(par[0]);

	if (namesamen(par[0], "out-of-bounds", l) == 0)
	{
		np = us_needfacet();
		if (np == NONODEPROTO) return;
		i = 0;
		for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
		{
			if (ni->geom->lowx > -HUGEINT/2 && ni->geom->highx < HUGEINT/2 &&
				ni->geom->lowy > -HUGEINT/2 && ni->geom->highy < HUGEINT/2) continue;
			if (i == 0) us_clearhighlightcount();
			newhigh.status = HIGHFROM;
			newhigh.facet = np;
			newhigh.fromgeom = ni->geom;
			newhigh.fromport = NOPORTPROTO;
			newhigh.frompoint = 0;
			(void)us_addhighlight(&newhigh);
			i++;
		}
		for(ai = np->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
		{
			if (ai->geom->lowx > -HUGEINT/2 && ai->geom->highx < HUGEINT/2 &&
				ai->geom->lowy > -HUGEINT/2 && ai->geom->highy < HUGEINT/2) continue;
			newhigh.status = HIGHFROM;
			newhigh.facet = np;
			newhigh.fromgeom = ai->geom;
			newhigh.fromport = NOPORTPROTO;
			newhigh.frompoint = 0;
			(void)us_addhighlight(&newhigh);
			i++;
		}
		if (i == 0) ttyputmsg("No objects are out of bounds"); else
			ttyputmsg("%d objects are out of bounds", i);
		return;
	}

	if (namesamen(par[0], "label-facet", l) == 0 && l >= 1)
	{
		np = us_needfacet();
		if (np == NONODEPROTO) return;
		us_clearhighlightcount();
		let1 = 'A';   let2 = let3 = 0;
		for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
		{
			line[0] = let1;   line[1] = let2;   line[2] = let3;   line[3] = 0;
			(void)setvalkey((INTBIG)ni, VNODEINST, el_node_name, (INTBIG)line, VSTRING|VDISPLAY);
			let1++;
			if (let1 > 'Z')
			{
				let1 = 'A';
				if (let2 == 0) let2 = 'A'; else let2++;
				if (let2 > 'Z')
				{
					let2 = 'A';
					if (let3 == 0) let3 = 'A'; else let3++;
				}
			}
		}
		i = 1;
		for(ai = np->firstarcinst; ai != NOARCINST;
			ai = ai->nextarcinst)
		{
			(void)sprintf(line, "%d", i++);
			(void)setvalkey((INTBIG)ai, VARCINST, el_arc_name, (INTBIG)line,
				VSTRING|VDISPLAY);
		}
		us_redisplay(el_curwindow);
		return;
	}

	if (namesamen(par[0], "scale-facet", l) == 0 && l >= 2)
	{
		np = us_needfacet();
		if (np == NONODEPROTO) return;
		if (count != 3)
		{
			us_abortcommand("Usage: debug scale-facet NUMERATOR DENOMINATOR");
			return;
		}
		db_scaleonefacet(np, myatoi(par[1]), myatoi(par[2]));
		ttyputmsg("Facet %s scaled...recommend '-debug check-database'",
			describenodeproto(np));
		return;
	}

	if (namesamen(par[0], "highlight-nonmanhattan", l) == 0 && l >= 1)
	{
		np = us_needfacet();
		if (np == NONODEPROTO) return;
		i = 0;
		for(ai = np->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
		{
			var = getvalkey((INTBIG)ai, VARCINST, VINTEGER, el_arc_radius);
			if (var == NOVARIABLE && (ai->end[0].xpos == ai->end[1].xpos ||
				ai->end[0].ypos == ai->end[1].ypos)) continue;
			if (i == 0) us_clearhighlightcount();
			newhigh.status = HIGHFROM;
			newhigh.facet = np;
			newhigh.fromgeom = ai->geom;
			newhigh.fromport = NOPORTPROTO;
			newhigh.frompoint = 0;
			(void)us_addhighlight(&newhigh);
			i++;
		}
		if (i == 0) ttyputmsg("No nonmanhattan arcs in this facet"); else
			ttyputmsg("%d arcs are not manhattan in this facet", i);
		return;
	}

	if (namesamen(par[0], "erase-bits", l) == 0 && l >= 1)
	{
		for(lib = el_curlib; lib != NOLIBRARY; lib = lib->nextlibrary)
		{
			/* clear the library bits */
			lib->temp1 = lib->temp2 = 0;
			lib->userbits &= (LIBCHANGED|LIBUNITS);

			/* look at every facet in the library */
			for(np=lib->firstnodeproto; np!=NONODEPROTO; np=np->nextnodeproto)
			{
				/* clear the node prototype bits */
				np->temp1 = np->temp2 = 0;
				np->userbits &= WANTNEXPAND;

				/* clear the port prototype bits */
				for(pp = np->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
				{
					pp->temp1 = pp->temp2 = 0;
					pp->userbits &= (STATEBITS|PORTANGLE|PORTARANGE);
				}

				/* clear the node instance bits */
				for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
				{
					ni->temp1 = ni->temp2 = 0;
					ni->userbits &= NEXPAND;

					/* compute WIPED bit on node instance */
					if ((ni->proto->userbits&ARCSWIPE) != 0)
					{
						for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
						{
							if ((pi->conarcinst->proto->userbits&CANWIPE) != 0)
							{
								ni->userbits |= WIPED;
								break;
							}
						}
					}
				}

				/* clear the arc instance bits */
				for(ai=np->firstarcinst; ai!=NOARCINST; ai=ai->nextarcinst)
				{
					ai->temp1 = ai->temp2 = 0;
					ai->userbits &= (FIXED|FIXANG|ISNEGATED|NOEXTEND|NOTEND0|NOTEND1|REVERSEEND|CANTSLIDE);
					determineangle(ai);
					setshrinkvalue(ai);
				}
			}
		}
		ttyputmsg("All aid words reset");
		return;
	}

	/********** THESE ARE STANDARD **********/

	if (namesamen(par[0], "check-database", l) == 0 && l >= 1)
	{
		if (count > 1 && namesamen(par[1], "verbose", (INTSML)(strlen(par[1]))) == 0)
			us_checkdatabase(1); else
				us_checkdatabase(0);
		return;
	}

	if (namesamen(par[0], "undo", l) == 0 && l >= 1)
	{
		db_undodlog();
		return;
	}

	if (namesamen(par[0], "freeze-user-interface", l) == 0 && l >= 1)
	{
		j = l = 0;
		for(i=0; i<us_aid->numvar; i++)
		{
			var = &us_aid->firstvar[i];
			pt = makename(var->key);
			if (namesamen(pt, "USER_binding_", 13) == 0)
			{
				if (namesame(pt, "USER_binding_menu") == 0) continue;
				var->type &= ~VDONTSAVE;
				j++;
			} else if (namesamen(pt, "USER_macro_", 11) == 0)
			{
				var->type &= ~VDONTSAVE;

				/*
				 * when saving macros to disk, must erase entry 2, which
				 * is the address of the parsing structures: these cannot
				 * be valid when read back in.
				 */
				setindkey((INTBIG)us_aid, VAID, var->key, 2, (INTBIG)"");
				l++;
			}
		}
		ttyputmsg("Made %d macros and %d bindings permanent", l, j);
		return;
	}

	if (namesamen(par[0], "rtree", l) == 0 && l >= 2)
	{
		np = us_needfacet();
		if (np == NONODEPROTO) return;
		ttyputmsg("R-tree for facet %s (nodes hold from %d to %d entries)",
			describenodeproto(np), db_minrtnodesize, db_maxrtnodesize);
		db_printrtree(NORTNODE, np->rtree, 0);
		return;
	}

	if (namesamen(par[0], "internal-errors", l) == 0 && l >= 1)
	{
		db_printerrors = !db_printerrors;
		ttyputmsg("Internal database errors will %sbe printed", (db_printerrors==0 ? "not " : ""));
		return;
	}

	if (namesamen(par[0], "namespace", l) == 0 && l >= 1)
	{
		ttyputmsg("%d names in global namespace:", el_numnames);
		for(i=0; i<el_numnames; i++) ttyputmsg("'%s'", el_namespace[i]);
		return;
	}

	if (namesamen(par[0], "arena", l) == 0 && l >= 1)
	{
		if (count > 1) db_printclusterarena(par[1]); else
			db_printclusterarena("");
		return;
	}

	us_abortcommand("Bad DEBUG option");
}

void us_defarc(INTSML count, char *par[])
{
	REGISTER char *aname, *pt;
	char line[20];
	REGISTER INTSML l, negate;
	REGISTER INTBIG i, wid, style, bits;
	REGISTER ARCPROTO *ap;
	REGISTER PORTPROTO *pp;
	REGISTER NODEPROTO *np;
	extern COMCOMP us_defarcsp, us_defarcnotp, us_defarcwidp, us_defarcpinp, us_defarcangp;
	REGISTER VARIABLE *var;

	if (count == 0)
	{
		count = ttygetparam("Arc prototype: ", &us_defarcsp, MAXPARS, par);
		if (count == 0)
		{
			us_abortedmsg();
			return;
		}
	}

	/* get arc prototype name */
	if (strcmp(par[0], "*") == 0)
	{
		ap = NOARCPROTO;
		aname = "all";
	} else
	{
		ap = getarcproto(par[0]);
		if (ap == NOARCPROTO)
		{
			us_abortcommand("Unknown arc prototype: %s", par[0]);
			return;
		}
		aname = ap->protoname;
	}

	if (count <= 1)
	{
		/* get style to report */
		if (ap != NOARCPROTO) style = ap->userbits; else
		{
			var = getvalkey((INTBIG)us_aid, VAID, VINTEGER, us_arcstyle);
			if (var != NOVARIABLE) style = var->addr; else style = -1;
			if (style == -1)
			{
				ttyputmsg("No overriding defaults set for arcs");
				return;
			}
		}
		(void)initinfstr();
		(void)addstringtoinfstr(aname);
		(void)addstringtoinfstr(" arcs drawn ");
		if ((style&WANTFIX) != 0) (void)addstringtoinfstr("rigid"); else
		{
			(void)addstringtoinfstr("nonrigid, ");
			if ((style&WANTFIXANG) == 0) (void)addstringtoinfstr("not ");
			(void)addstringtoinfstr("fixed angle, ");
			if ((style&WANTCANTSLIDE) != 0) (void)addstringtoinfstr("non");
			(void)addstringtoinfstr("slidable");
		}
		if ((style&WANTDIRECTIONAL) != 0) (void)addstringtoinfstr(", directional");
		if ((style&WANTNEGATED) != 0) (void)addstringtoinfstr(", negated");
		(void)addstringtoinfstr(", ");
		if ((style&WANTNOEXTEND) != 0) (void)addstringtoinfstr("not ");
		(void)addstringtoinfstr("ends-extended");
		if (ap != NOARCPROTO)
		{
			(void)addstringtoinfstr(", width ");
			(void)addstringtoinfstr(latoa(ap->nominalwidth-arcwidthoffset(ap)));
			(void)sprintf(line, ", angle %d", (ap->userbits&AANGLEINC) >> AANGLEINCSH);
			(void)addstringtoinfstr(line);
		}
		ttyputmsg("%s", returninfstr());
		return;
	}

	l = strlen(pt = par[1]);
	negate = 0;

	if (namesamen(pt, "not", l) == 0 && l >= 2)
	{
		if (count <= 2)
		{
			count = ttygetparam("Defarc option: ", &us_defarcnotp, MAXPARS-2, &par[2])+2;
			if (count == 2)
			{
				us_abortedmsg();
				return;
			}
		}
		count--;
		par++;
		negate++;
		l = strlen(pt = par[1]);
	}

	if (namesamen(pt, "default", l) == 0 && l >= 2)
	{
		if (ap != NOARCPROTO)
		{
			us_abortcommand("Can only set default on every arc (*)");
			return;
		}
		(void)delvalkey((INTBIG)us_aid, VAID, us_arcstyle);
		ttyputmsgf("All arcs drawn in default style");
		return;
	}

	if ((namesamen(pt, "manhattan", l) == 0 || namesamen(pt, "fixed-angle", l) == 0) && l >= 1)
	{
		if (namesamen(pt, "manhattan", l) == 0)
			ttyputmsg("It is now proper to use 'defarc fixed-angle' instead of 'defarc manhattan'");
		if (ap != NOARCPROTO)
		{
			if (negate != 0) style = ap->userbits & ~WANTFIXANG; else
				style = ap->userbits | WANTFIXANG;
			(void)setval((INTBIG)ap, VARCPROTO, "userbits", style, VINTEGER);
		} else
		{
			var = getvalkey((INTBIG)us_aid, VAID, VINTEGER, us_arcstyle);
			if (var != NOVARIABLE) style = var->addr; else style = 0;
			if (negate != 0) style &= ~WANTFIXANG; else style |= WANTFIXANG;
			(void)setvalkey((INTBIG)us_aid, VAID, us_arcstyle, style, VINTEGER|VDONTSAVE);
		}
		ttyputmsgf("%s arcs drawn %sfixed-angle", aname,
			(negate!=0 ? "not " : ""));
		return;
	}

	if (namesamen(pt, "rigid", l) == 0 && l >= 1)
	{
		if (ap != NOARCPROTO)
		{
			if (negate != 0) style = ap->userbits & ~WANTFIX; else
				style = ap->userbits | WANTFIX;
			(void)setval((INTBIG)ap, VARCPROTO, "userbits", style, VINTEGER);
		} else
		{
			var = getvalkey((INTBIG)us_aid, VAID, VINTEGER, us_arcstyle);
			if (var != NOVARIABLE) style = var->addr; else style = WANTFIXANG;
			if (negate != 0) style &= ~WANTFIX; else style |= WANTFIX;
			(void)setvalkey((INTBIG)us_aid, VAID, us_arcstyle, style, VINTEGER|VDONTSAVE);
		}
		ttyputmsgf("%s arcs drawn %srigid", aname, (negate!=0 ? "un-" : ""));
		return;
	}

	if (namesamen(pt, "directional", l) == 0 && l >= 2)
	{
		if (ap != NOARCPROTO)
		{
			if (negate != 0) style = ap->userbits & ~WANTDIRECTIONAL; else
				style = ap->userbits | WANTDIRECTIONAL;
			(void)setval((INTBIG)ap, VARCPROTO, "userbits", style, VINTEGER);
		} else
		{
			var = getvalkey((INTBIG)us_aid, VAID, VINTEGER, us_arcstyle);
			if (var != NOVARIABLE) style = var->addr; else style = WANTFIXANG;
			if (negate != 0) style &= ~WANTDIRECTIONAL; else
				style |= WANTDIRECTIONAL;
			(void)setvalkey((INTBIG)us_aid, VAID, us_arcstyle, style, VINTEGER|VDONTSAVE);
		}
		ttyputmsgf("%s arcs drawn %sdirectional", aname, (negate!=0 ? "non" : ""));
		return;
	}

	if (namesamen(pt, "slide", l) == 0 && l >= 1)
	{
		if (ap != NOARCPROTO)
		{
			if (negate == 0) style = ap->userbits & ~WANTCANTSLIDE; else
				style = ap->userbits | WANTCANTSLIDE;
			(void)setval((INTBIG)ap, VARCPROTO, "userbits", style, VINTEGER);
		} else
		{
			var = getvalkey((INTBIG)us_aid, VAID, VINTEGER, us_arcstyle);
			if (var != NOVARIABLE) style = var->addr; else style = WANTFIXANG;
			if (negate == 0) style &= ~WANTCANTSLIDE; else
				style |= WANTCANTSLIDE;
			(void)setvalkey((INTBIG)us_aid, VAID, us_arcstyle, style, VINTEGER|VDONTSAVE);
		}
		ttyputmsgf("%s arcs can %sslide in their ports", aname, (negate!=0 ? "not " : ""));
		return;
	}

	if (namesamen(pt, "negated", l) == 0 && l >= 2)
	{
		if (ap != NOARCPROTO)
		{
			if (negate != 0) style = ap->userbits & ~WANTNEGATED; else
				style = ap->userbits | WANTNEGATED;
			(void)setval((INTBIG)ap, VARCPROTO, "userbits", style, VINTEGER);
		} else
		{
			var = getvalkey((INTBIG)us_aid, VAID, VINTEGER, us_arcstyle);
			if (var != NOVARIABLE) style = var->addr; else style = WANTFIXANG;
			if (negate != 0) style &= ~WANTNEGATED; else style |= WANTNEGATED;
			(void)setvalkey((INTBIG)us_aid, VAID, us_arcstyle, style, VINTEGER|VDONTSAVE);
		}
		ttyputmsgf("%s arcs drawn %snegated", aname, (negate!=0 ? "un-" : ""));
		return;
	}

	if (namesamen(pt, "ends-extend", l) == 0 && l >= 1)
	{
		if (ap != NOARCPROTO)
		{
			if (negate != 0) style = ap->userbits | WANTNOEXTEND; else
				style = ap->userbits &= ~WANTNOEXTEND;
			(void)setval((INTBIG)ap, VARCPROTO, "userbits", style, VINTEGER);
		} else
		{
			var = getvalkey((INTBIG)us_aid, VAID, VINTEGER, us_arcstyle);
			if (var != NOVARIABLE) style = var->addr; else style = WANTFIXANG;
			if (negate != 0) style |= WANTNOEXTEND; else style &= ~WANTNOEXTEND;
			(void)setvalkey((INTBIG)us_aid, VAID, us_arcstyle, style, VINTEGER|VDONTSAVE);
		}
		ttyputmsgf("%s arcs drawn with ends %s", aname,
			(negate!=0 ? "not extended" : "extended by half-width"));
		return;
	}

	if (namesamen(pt, "width", l) == 0 && l >= 1)
	{
		if (ap == NOARCPROTO)
		{
			us_abortcommand("Default width requires specific arc prototypes");
			return;
		}
		if (count <= 2)
		{
			count = ttygetparam("Width: ", &us_defarcwidp, MAXPARS-2, &par[2]) + 2;
			if (count == 2)
			{
				us_abortedmsg();
				return;
			}
		}
		pt = par[2];
		i = atola(pt);
		if (i&1) i++;
		if (i < 0)
		{
			us_abortcommand("Width must not be negative");
			return;
		}
		wid = arcwidthoffset(ap) + i;
		(void)setval((INTBIG)ap, VARCPROTO, "nominalwidth", wid, VINTEGER);
		ttyputmsgf("%s arcs default to %s wide", aname, latoa(i));
		return;
	}

	if (namesamen(pt, "angle", l) == 0 && l >= 1)
	{
		if (ap == NOARCPROTO)
		{
			us_abortcommand("Default angle increment requires specific arc prototypes");
			return;
		}
		if (count <= 2)
		{
			count = ttygetparam("Angle: ", &us_defarcangp, MAXPARS-2, &par[2]) + 2;
			if (count == 2)
			{
				us_abortedmsg();
				return;
			}
		}
		pt = par[2];
		i = atoi(pt) % 360;
		if (i < 0)
		{
			us_abortcommand("Angle increment must not be negative");
			return;
		}
		bits = (ap->userbits & ~AANGLEINC) | (i << AANGLEINCSH);
		(void)setval((INTBIG)ap, VARCPROTO, "userbits", bits, VINTEGER);
		ttyputmsgf("%s arcs will run at %d degree angle increments", aname, i);
		return;
	}

	if (namesamen(pt, "pin", l) == 0 && l >= 1)
	{
		if (ap == NOARCPROTO)
		{
			us_abortcommand("Default pin requires specific arc prototypes");
			return;
		}
		if (count <= 2)
		{
			count = ttygetparam("Pin: ", &us_defarcpinp, MAXPARS-2, &par[2]) + 2;
			if (count == 2)
			{
				us_abortedmsg();
				return;
			}
		}
		np = getnodeproto(par[2]);
		if (np == NONODEPROTO)
		{
			us_abortcommand("Cannot find primitive %s", par[2]);
			return;
		}
		pp = np->firstportproto;
		for(i=0; pp->connects[i] != NOARCPROTO; i++)
			if (pp->connects[i] == ap) break;
		if (pp->connects[i] == NOARCPROTO)
		{
			us_abortcommand("Cannot: the %s node must connect to %s arcs on its first port (%s)",
				describenodeproto(np), aname, pp->protoname);
			return;
		}
		(void)setval((INTBIG)ap, VARCPROTO, "ARC_Default_Pin", (INTBIG)np, VNODEPROTO|VDONTSAVE);
		ttyputmsgf("Default pins for arc %s will be %s", aname, describenodeproto(np));
		return;
	}

	us_abortcommand("Bad DEFARC option: %s", par[1]);
}

void us_defnode(INTSML count, char *par[])
{
	REGISTER char *pp, *nname;
	REGISTER INTSML l, negated, pangle;
	REGISTER INTBIG i, j, lx, hx, ly, hy;
	INTBIG plx, ply, phx, phy;
	REGISTER VARIABLE *var;
	REGISTER NODEPROTO *np;
	extern COMCOMP us_defnodesp, us_defnodexsp, us_defnodeysp;

	if (count == 0)
	{
		count = ttygetparam("Node prototype: ", &us_defnodesp, MAXPARS, par);
		if (count == 0)
		{
			us_abortedmsg();
			return;
		}
	}

	/* get node prototype name */
	if (strcmp(par[0], "*") == 0)
	{
		np = NONODEPROTO;
		nname = "All complex";
	} else
	{
		np = getnodeproto(par[0]);
		if (np == NONODEPROTO)
		{
			us_abortcommand("Unknown node prototype: %s", par[0]);
			return;
		}
		nname = describenodeproto(np);
	}

	if (count <= 1)
	{
		if (np == NONODEPROTO)
		{
			var = getvalkey((INTBIG)us_aid, VAID, VINTEGER, us_placement_angle);
			if (var == NOVARIABLE) pangle = 0; else pangle = var->addr;
			ttyputmsg("Nodes are created with %s degrees rotation%s",
				frtoa(pangle%3600*WHOLE/10), (pangle >= 3600 ? ", transposed" : ""));
			ttyputmsg("Facet revision dates are %sbeing checked",
				(us_aid->aidstate&CHECKDATE) == 0 ? "not " : "");
			ttyputmsg("Facet changes are %sallowed",
				(us_aid->aidstate&NOFACETCHANGES) != 0 ? "dis" : "");
			ttyputmsg("Locked primitive changes are %sallowed",
				(us_aid->aidstate&NOPRIMCHANGES) != 0 ? "dis" : "");
		} else
		{
			if (np->index == 0)
			{
				ttyputmsg("Newly created %s facets are %sexpanded", describenodeproto(np),
					(np->userbits&WANTNEXPAND) == 0 ? "un" : "");
			} else
			{
				nodesizeoffset(np, &plx, &ply, &phx, &phy);
				lx = np->lowx + plx;   hx = np->highx - phx;
				ly = np->lowy + ply;   hy = np->highy - phy;
				ttyputmsg("Default %s size is %sx%s", np->primname, latoa(hx-lx), latoa(hy-ly));
			}
		}
		return;
	}

	l = strlen(pp = par[1]);

	/* handle negation */
	if (namesamen(pp, "not", l) == 0 && l >= 1)
	{
		negated = 1;
		count--;
		par++;
		l = strlen(pp = par[1]);
	} else negated = 0;

	if (namesamen(pp, "check-dates", l) == 0 && l >= 1)
	{
		if (np != NONODEPROTO)
		{
			us_abortcommand("Date checking must be done for all facets (*)");
			return;
		}
		if (negated == 0)
		{
			if ((us_aid->aidstate&CHECKDATE) != 0)
				ttyputmsgf("Facet revision dates already being checked"); else
					ttyputmsgf("Facet revision date errors will be identified");
			(void)setval((INTBIG)us_aid, VAID, "aidstate", us_aid->aidstate | CHECKDATE, VINTEGER);
		} else
		{
			if ((us_aid->aidstate&CHECKDATE) == 0)
				ttyputmsgf("Facet revision dates already being ignored"); else
					ttyputmsgf("Facet revision date errors will be ignored");
			(void)setval((INTBIG)us_aid, VAID, "aidstate", us_aid->aidstate & ~CHECKDATE, VINTEGER);
		}
		return;
	}

	if (namesamen(pp, "alterable", l) == 0 && l >= 1)
	{
		if (np != NONODEPROTO)
		{
			us_abortcommand("Alteration lock must be done for all facets (*)");
			return;
		}
		if (negated == 0)
		{
			if ((us_aid->aidstate&NOFACETCHANGES) == 0)
				ttyputmsgf("Facets are already able to be freely altered"); else
					ttyputmsgf("Facets may now be freely edited");
			(void)setval((INTBIG)us_aid, VAID, "aidstate", us_aid->aidstate & ~NOFACETCHANGES, VINTEGER);
		} else
		{
			if ((us_aid->aidstate&NOFACETCHANGES) != 0)
				ttyputmsgf("Facet changes are already being disallowed"); else
					ttyputmsgf("Facet changes will not be allowed");
			(void)setval((INTBIG)us_aid, VAID, "aidstate", us_aid->aidstate | NOFACETCHANGES, VINTEGER);
		}
		return;
	}

	if (namesamen(pp, "locked-primitives", l) == 0 && l >= 1)
	{
		if (np != NONODEPROTO)
		{
			us_abortcommand("Primitive lock must be done for all prototypes (*)");
			return;
		}
		if (negated != 0)
		{
			if ((us_aid->aidstate&NOPRIMCHANGES) == 0)
				ttyputmsgf("Locked primitives are already able to be freely altered"); else
					ttyputmsgf("Locked primitives may now be freely edited");
			(void)setval((INTBIG)us_aid, VAID, "aidstate", us_aid->aidstate & ~NOPRIMCHANGES, VINTEGER);
		} else
		{
			if ((us_aid->aidstate&NOPRIMCHANGES) != 0)
				ttyputmsgf("Locked primitive changes are already being disallowed"); else
					ttyputmsgf("Locked primitive changes will not be allowed");
			(void)setval((INTBIG)us_aid, VAID, "aidstate", us_aid->aidstate | NOPRIMCHANGES, VINTEGER);
		}
		return;
	}

	if (namesamen(pp, "expanded", l) == 0 && l >= 1)
	{
		if (np == NONODEPROTO)
		{
			for(np = el_curlib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
			{
				if (negated == 0) (void)setval((INTBIG)np, VNODEPROTO, "userbits",
					np->userbits | WANTNEXPAND, VINTEGER); else
						(void)setval((INTBIG)np, VNODEPROTO, "userbits",
							np->userbits & ~WANTNEXPAND, VINTEGER);
			}
		} else
		{
			if (np->index != 0)
			{
				if (negated == 0) us_abortcommand("Primitives are expanded by definition"); else
					us_abortcommand("Cannot un-expand primitives");
				return;
			}
			if (negated == 0) (void)setval((INTBIG)np, VNODEPROTO, "userbits",
				np->userbits | WANTNEXPAND, VINTEGER); else
					(void)setval((INTBIG)np, VNODEPROTO, "userbits",
						np->userbits & ~WANTNEXPAND, VINTEGER);
		}
		ttyputmsgf("%s node prototypes will be %sexpanded on creation", nname, (negated==0 ? "" : "un-"));
		return;
	}

	if (namesamen(pp, "size", l) == 0 && l >= 1)
	{
		if (np == NONODEPROTO)
		{
			us_abortcommand("Must change size on a single node");
			return;
		}
		if (np->index == 0)
		{
			us_abortcommand("Can only change default size on primitives");
			return;
		}
		if (count <= 2)
		{
			count = ttygetparam("X size: ", &us_defnodexsp, MAXPARS-2, &par[2]) + 2;
			if (count == 2)
			{
				us_abortedmsg();
				return;
			}
		}
		pp = par[2];
		i = atola(pp);
		if (i&1) i++;
		if (count <= 3)
		{
			count = ttygetparam("Y size: ", &us_defnodeysp, MAXPARS-3, &par[3]) + 3;
			if (count == 3)
			{
				us_abortedmsg();
				return;
			}
		}
		pp = par[3];
		j = atola(pp);
		if (j&1) j++;
		if (i < 0 || j < 0)
		{
			us_abortcommand("Sizes must not be negative");
			return;
		}
		nodesizeoffset(np, &plx, &ply, &phx, &phy);
		lx = -(i+plx+phx) / 2;   hx = lx+i+plx+phx;
		ly = -(j+ply+phy) / 2;   hy = ly+j+ply+phy;
		(void)setval((INTBIG)np, VNODEPROTO, "lowx", lx, VINTEGER);
		(void)setval((INTBIG)np, VNODEPROTO, "highx", hx, VINTEGER);
		(void)setval((INTBIG)np, VNODEPROTO, "lowy", ly, VINTEGER);
		(void)setval((INTBIG)np, VNODEPROTO, "highy", hy, VINTEGER);
		ttyputmsgf("%s nodes will be created %sx%s in size", describenodeproto(np), latoa(i), latoa(j));
		return;
	}

	if (namesamen(pp, "placement", l) == 0 && l >= 1)
	{
		if (np != NONODEPROTO)
		{
			us_abortcommand("Default placement must be done for all nodes (*)");
			return;
		}

		/* get placement angle */
		var = getvalkey((INTBIG)us_aid, VAID, VINTEGER, us_placement_angle);
		if (var == NOVARIABLE) pangle = 0; else pangle = var->addr;

		if (count > 2)
		{
			pangle = (atofr(par[2])*10/WHOLE) % 3600;
			if (pangle < 0) pangle += 3600;
			if (namesame(&par[2][strlen(par[2])-1], "t") == 0) pangle += 3600;
		} else if (pangle >= 6300) pangle = 0; else
			pangle += 900;
		(void)setvalkey((INTBIG)us_aid, VAID, us_placement_angle, pangle, VINTEGER|VDONTSAVE);
		return;
	}

	us_abortcommand("Bad DEFNODE option: %s", par[1]);
}

void us_duplicate(INTSML count, char *par[])
{
	REGISTER NODEPROTO *np;
	REGISTER NODEINST *ni, *newni, **nodelist;
	REGISTER ARCINST *ai, *newar;
	REGISTER INTSML i, j;
	REGISTER INTBIG wid, dx, dy;
	INTBIG xcur, ycur, lx, hx, ly, hy, bestlx, bestly;
	INTSML total;
	HIGHLIGHT newhigh;
	REGISTER GEOM **list;
	static POLYGON *poly = NOPOLYGON;

	/* get polygon */
	if (poly == NOPOLYGON) poly = allocpolygon(4, us_aid->cluster);

	/* get object */
	np = us_needfacet();
	if (np == NONODEPROTO) return;
	list = us_gethighlighted(OBJNODEINST | OBJARCINST);
	if (list[0] == NOGEOM)
	{
		us_abortcommand("First select objects to duplicate");
		return;
	}

	/* make sure they are all in the same facet */
	for(i=1; list[i] != NOGEOM; i++) if (np != geomparent(list[i]))
	{
		us_abortcommand("All duplicated objects must be in the same facet");
		return;
	}

	/* mark all nodes (including those touched by highlighted arcs) */
	for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
		ni->temp1 = 0;
	for(i=0; list[i] != NOGEOM; i++) if (list[i]->entrytype == OBJARCINST)
	{
		ai = list[i]->entryaddr.ai;
		ai->end[0].nodeinst->temp1 = ai->end[1].nodeinst->temp1 = 1;
	}
	for(i=0; list[i] != NOGEOM; i++) if (list[i]->entrytype == OBJNODEINST)
		list[i]->entryaddr.ni->temp1 = 1;

	/* count the number of nodes, excluding locked ones */
	i = 0;
	for(total=0, ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		if (ni->temp1 == 0) continue;
		i++;
		if (us_islocked(ni->proto) != 0)
		{
			ni->temp1 = 0;
			continue;
		}
		total++;
	}
	if (total == 0)
	{
		if (i != 0) us_abortcommand("All selected objects are locked");
		return;
	}

	/* build a list that includes all nodes touching copyed arcs */
	nodelist = (NODEINST **)emalloc((total * (sizeof (NODEINST *))), el_tempcluster);
	if (nodelist == 0) return;
	for(i=0, ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
		if (ni->temp1 != 0) nodelist[i++] = ni;

	/* check for recursion */
	for(i=0; i<total; i++) if (nodelist[i]->proto->index == 0)
		if (nodelist[i]->proto->index == 0)
	{
		if (isachildof(np, nodelist[i]->proto))
		{
			us_abortcommand("Cannot: that would be recursive");
			efree((char *)nodelist);
			return;
		}
	}

	/* figure out lower-left corner of this collection of objects */
	us_getlowleft(nodelist[0], &bestlx, &bestly);
	for(i=1; i<total; i++)
	{
		us_getlowleft(nodelist[i], &lx, &ly);
		if (lx < bestlx) bestlx = lx;
		if (ly < bestly) bestly = ly;
	}
	for(i=0; list[i] != NOGEOM; i++) if (list[i]->entrytype == OBJARCINST)
	{
		ai = list[i]->entryaddr.ai;
		if (us_islocked(ai->end[0].nodeinst->proto) != 0 ||
			us_islocked(ai->end[1].nodeinst->proto) != 0) continue;
		wid = ai->width - arcwidthoffset(ai->proto);
		makearcpoly(ai->length, wid, ai, poly, FILLED);
		getbbox(poly, &lx, &hx, &ly, &hy);
		if (lx < bestlx) bestlx = lx;
		if (ly < bestly) bestly = ly;
	}

	/* special case when moving one node: account for facet center */
	if (total == 1 && list[1] == NOGEOM)
	{
		ni = nodelist[0];
		corneroffset(ni, ni->proto, ni->rotation, ni->transpose, &lx, &ly);
		bestlx = ni->lowx + lx;
		bestly = ni->lowy + ly;
	}
	us_clearhighlightcount();

	/* adjust the cursor position if selecting interactively */
	if ((us_aid->aidstate&INTERACTIVE) != 0 && us_graphicshas(CANTRACKCURSOR) != 0)
	{
		us_multidraginit(bestlx, bestly, list, nodelist, total, 0, 0);
		trackcursor(0, us_ignoreup, us_multidragbegin, us_multidragdown,
			us_stoponchar, us_multidragup, TRACKDRAGGING);
		if (el_pleasestop != 0)
		{
			efree((char *)nodelist);
			return;
		}
	}

	/* get aligned cursor co-ordinates */
	if (us_demandxy(&xcur, &ycur)) return;
	gridalign(&xcur, &ycur, us_alignment);

	dx = xcur-bestlx;
	dy = ycur-bestly;

	/* create the new objects */
	for(i=0; i<total; i++)
	{
		ni = nodelist[i];
		newni = newnodeinst(ni->proto, ni->lowx+dx, ni->highx+dx, ni->lowy+dy,
			ni->highy+dy, ni->transpose, ni->rotation, np);
		if (newni == NONODEINST)
		{
			us_abortcommand("Cannot create node");
			return;
		}
		newni->userbits = ni->userbits & ~(WIPED|NSHORT);
		(void)copyvars((INTBIG)ni, VNODEINST, (INTBIG)newni, VNODEINST);
		endobjectchange((INTBIG)newni, VNODEINST);
		ni->temp1 = (INTBIG)newni;
	}
	for(i=0; list[i] != NOGEOM; i++) if (list[i]->entrytype == OBJARCINST)
	{
		ai = list[i]->entryaddr.ai;
		if (us_islocked(ai->end[0].nodeinst->proto) != 0 ||
			us_islocked(ai->end[1].nodeinst->proto) != 0) continue;
		newar = newarcinst(ai->proto, ai->width, ai->userbits,
			(NODEINST *)ai->end[0].nodeinst->temp1, ai->end[0].portarcinst->proto, ai->end[0].xpos+dx,
				ai->end[0].ypos+dy, (NODEINST *)ai->end[1].nodeinst->temp1,
					ai->end[1].portarcinst->proto, ai->end[1].xpos+dx, ai->end[1].ypos+dy, np);
		if (newar == NOARCINST)
		{
			us_abortcommand("Cannot create arc");
			return;
		}
		(void)copyvars((INTBIG)ai, VARCINST, (INTBIG)newar, VARCINST);
		for(j=0; j<2; j++)
			(void)copyvars((INTBIG)ai->end[j].portarcinst, VPORTARCINST,
				(INTBIG)newar->end[j].portarcinst, VPORTARCINST);
		endobjectchange((INTBIG)newar, VARCINST);
	}

	/* highlight the one node (if there is only one copied) */
	if (total == 1)
	{
		ni = (NODEINST *)nodelist[0]->temp1;
		newhigh.status = HIGHFROM;
		newhigh.facet = ni->parent;
		newhigh.fromgeom = ni->geom;
		newhigh.fromport = NOPORTPROTO;
		newhigh.frompoint = 0;
		(void)us_addhighlight(&newhigh);
	}
}
