/*
 * Electric(tm) VLSI Design System
 *
 * File: usrcomln.c
 * User interface aid: command handler for L through N
 * Written by: Steven M. Rubin, Electric Editor Incorporated
 *
 * Copyright (c) 1998 Electric Editor Incorporated.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Electric Editor Incorporated
 * 23470 Sunset Drive, Suite 108
 * Los Gatos, California 95033
 * support@electriceditor.com
 */

#include "global.h"
#include "egraphics.h"
#include "usr.h"
#include "usrtrack.h"
#include "efunction.h"
#include "edialogs.h"
#include "tecart.h"

void us_lambda(INTSML count, char *par[])
{
	REGISTER char *pp;
	REGISTER INTBIG lam, oldlam;
	REGISTER INTSML l;
	REGISTER WINDOW *w;
	REGISTER NODEPROTO *np;
	REGISTER TECHNOLOGY *tech;
	extern COMCOMP us_lambdachp, us_noyesp;

	if (count > 0)
	{
		l = strlen(pp = par[0]);

		/* handle display unit setting */
		if (namesamen(pp, "display-units", l) == 0)
		{
			if (count >= 2)
			{
				l = strlen(pp = par[1]);
				if (namesamen(pp, "lambda", l) == 0)
				{
					el_units = (el_units & ~DISPLAYUNITS) | DISPUNITLAMBDA;
				} else if (namesamen(pp, "inch", l) == 0)
				{
					el_units = (el_units & ~DISPLAYUNITS) | DISPUNITINCH;
				} else if (namesamen(pp, "centimeter", l) == 0 && l >= 7)
				{
					el_units = (el_units & ~DISPLAYUNITS) | DISPUNITCM;
				} else if (namesamen(pp, "millimeter", l) == 0 && l >= 7)
				{
					el_units = (el_units & ~DISPLAYUNITS) | DISPUNITMM;
				} else if (namesamen(pp, "mil", l) == 0 && l >= 3)
				{
					el_units = (el_units & ~DISPLAYUNITS) | DISPUNITMIL;
				} else if (namesamen(pp, "micron", l) == 0 && l >= 3)
				{
					el_units = (el_units & ~DISPLAYUNITS) | DISPUNITMIC;
				} else if (namesamen(pp, "centimicron", l) == 0 && l >= 7)
				{
					el_units = (el_units & ~DISPLAYUNITS) | DISPUNITCMIC;
				} else if (namesamen(pp, "millimicron", l) == 0 && l >= 7)
				{
					el_units = (el_units & ~DISPLAYUNITS) | DISPUNITMMIC;
				} else
				{
					us_abortcommand("Unknown display unit: %s", pp);
					return;
				}
			}
			switch (el_units&DISPLAYUNITS)
			{
				case DISPUNITLAMBDA:
					ttyputmsgf("Distance expressed in lambda (currently %d %ss)",
						el_curtech->deflambda, unitsname(el_units));
					break;
				case DISPUNITINCH:
					ttyputmsgf("Distance expressed in inches");
					break;
				case DISPUNITCM:
					ttyputmsgf("Distance expressed in centimeters");
					break;
				case DISPUNITMM:
					ttyputmsgf("Distance expressed in millimeters");
					break;
				case DISPUNITMIL:
					ttyputmsgf("Distance expressed in mils");
					break;
				case DISPUNITMIC:
					ttyputmsgf("Distance expressed in microns");
					break;
				case DISPUNITCMIC:
					ttyputmsgf("Distance expressed in centimicrons");
					break;
				case DISPUNITMMIC:
					ttyputmsgf("Distance expressed in millimicrons");
					break;
			}
			return;
		}

		/* handle internal unit setting */
		if (namesamen(pp, "internal-units", l) == 0)
		{
			if (count >= 2)
			{
				l = strlen(pp = par[1]);
				if (namesamen(pp, "half-decimicron", l) == 0 && l >= 6)
				{
					changeinternalunits(NOLIBRARY, el_units, INTUNITHDMIC);
				} else if (namesamen(pp, "half-millimicron", l) == 0 && l >= 6)
				{
					changeinternalunits(NOLIBRARY, el_units, INTUNITHMMIC);
				} else
				{
					us_abortcommand("Unknown internal unit: %s", pp);
					return;
				}
			}
			ttyputmsgf("Smallest internal unit is %s", unitsname(el_units));
			return;
		}

		if (namesamen(pp, "change", l) == 0 && l >= 1) tech = NOTECHNOLOGY; else
			if (namesamen(pp, "no-scale-change", l) == 0 && l >= 1) tech = el_curtech; else
		{
			us_abortcommand("Usage: lambda change|no-scale-change VALUE");
			return;
		}

		if (count <= 1)
		{
			count = ttygetparam("Lambda value: ", &us_lambdachp, MAXPARS-1, &par[1]) + 1;
			if (count == 1)
			{
				us_abortedmsg();
				return;
			}
		}

		l = strlen(par[1]);
		{
			lam = atoi(par[1]);
			if (lam <= 0)
			{
				us_abortcommand("Lambda value must be positive");
				return;
			}
			if (lam / 4 * 4 != lam)
			{
				count = ttygetparam("Instability may result if lambda is not divisible by four.  Continue? [n]",
					&us_noyesp, MAXPARS, par);
				if (count <= 0 || namesamen(par[0], "yes", (INTSML)(strlen(par[0]))) != 0)
				{
					ttyputmsgf("No change made");
					return;
				}
			}
		}

		/* save highlighting if redisplaying */
		if (tech != NOTECHNOLOGY)
		{
			us_pushhighlight();
			us_clearhighlightcount();
		}

		oldlam = el_curtech->deflambda;
		changelambda(oldlam, lam, tech);
		us_setlambda(0);
		us_adjustlambda(oldlam, lam);
		for(w = el_topwindow; w != NOWINDOW; w = w->nextwindow)
		{
			if (w->curnodeproto == NONODEPROTO) continue;
			if (whattech(w->curnodeproto) != el_curtech) continue;
			if (tech == NOTECHNOLOGY)
			{
				w->screenlx = muldiv(w->screenlx, lam, el_curtech->deflambda);
				w->screenhx = muldiv(w->screenhx, lam, el_curtech->deflambda);
				w->screenly = muldiv(w->screenly, lam, el_curtech->deflambda);
				w->screenhy = muldiv(w->screenhy, lam, el_curtech->deflambda);
				computewindowscale(w);
			} else us_redisplay(w);
		}

		/* restore highlighting if redisplayed */
		if (tech != NOTECHNOLOGY) (void)us_pophighlight(0);
	}

	/* determine which technologys to report */
	for(tech = el_technologies; tech != NOTECHNOLOGY; tech = tech->nexttechnology) tech->temp1 = 0;
	for(np=el_curlib->firstnodeproto; np!=NONODEPROTO; np=np->nextnodeproto) whattech(np)->temp1++;
	el_curtech->temp1++;
	for(tech = el_technologies; tech != NOTECHNOLOGY; tech = tech->nexttechnology)
	{
		if (tech->temp1 == 0) continue;
		ttyputmsgf("Lambda size for %s is %s", tech->techname, latoa(tech->deflambda));
	}
}

void us_library(INTSML count, char *par[])
{
	REGISTER LIBRARY *lib, *ol, *lastlib, *mergelib;
	REGISTER INTSML i, l, makecurrent, found, total, retval;
	REGISTER char *pp, *style, *libf, *oldlibfile;
	REGISTER NODEPROTO *np;
	REGISTER VARIABLE *var;
	REGISTER EDITOR *ed;
	REGISTER WINDOW *w;
	INTBIG lx, hx, ly, hy;
	INTSML dummy;
	char *newpar[2], *libn, *extn;
	extern COMCOMP us_libraryp;
	extern AIDENTRY *io_aid;
#ifdef WIN32
	char abspath[256];
#endif

	if (count == 0)
	{
		count = ttygetparam("Library option: ", &us_libraryp, MAXPARS, par);
		if (count == 0)
		{
			us_abortedmsg();
			return;
		}
	}
	l = strlen(pp = par[0]);

	if (namesamen(pp, "default-path", l) == 0 && l >= 9)
	{
		if (count < 2)
		{
			us_abortcommand("Usage: library default-path PATHNAME");
			return;
		}
		libf = par[1];
		for(pp = libf; *pp != 0; pp++)
			if (*pp == '/') *pp = DIRSEP;
		(void)initinfstr();
#ifdef	MACOS
		if (libf[0] == ':')
		{
			pp = mac_systemfoldername();
			(void)addstringtoinfstr(pp);
		} else (void)addtoinfstr(':');
#endif
#ifdef WIN32
		libf = _fullpath(abspath, libf, 256);
#endif
		(void)addstringtoinfstr(libf);
		if (libf[strlen(libf)-1] != DIRSEP) (void)addtoinfstr(DIRSEP);
		(void)reallocstring(&el_libdir, returninfstr(), db_cluster);
		ttyputmsgf("Default library path is now '%s'", el_libdir);
		return;
	}

	if (namesamen(pp, "default-lisp-path", l) == 0 && l >= 9)
	{
		if (count < 2)
		{
			us_abortcommand("Usage: library default-lisp-path PATHNAME");
			return;
		}
		libf = par[1];
		for(pp = libf; *pp != 0; pp++)
			if (*pp == '/') *pp = DIRSEP;
		(void)initinfstr();
#ifdef	MACOS
		if (libf[0] == ':')
		{
			pp = mac_systemfoldername();
			(void)addstringtoinfstr(pp);
		} else (void)addtoinfstr(':');
#endif
#ifdef WIN32
		libf = _fullpath(abspath, libf, 256);
#endif
		(void)addstringtoinfstr(libf);
		if (libf[strlen(libf)-1] != DIRSEP) (void)addtoinfstr(DIRSEP);
		(void)reallocstring(&el_lispdir, returninfstr(), db_cluster);
		ttyputmsgf("LISP-file directory is now '%s'", el_lispdir);
		return;
	}

	if (namesamen(pp, "purge", l) == 0 && l >= 1)
	{
		/* delete all facets that are not the most recent and are unused */
		found = 1;
		total = 0;
		while (found != 0)
		{
			found = 0;
			for(np = el_curlib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
			{
				if (np->newestversion == np) continue;
				if (np->firstinst != NONODEINST) continue;
				newpar[0] = describenodeproto(np);
				us_killfacet(1, newpar);
				found++;
				total++;
				break;
			}
		}
		if (total == 0) ttyputmsg("No unused old facet versions to delete"); else
			ttyputmsg("Deleted %d facets", total);
		return;
	}

	if (namesamen(pp, "kill", l) == 0 && l >= 1)
	{
		if (count <= 1) lib = el_curlib; else
		{
			lib = getlibrary(par[1]);
			if (lib == NOLIBRARY)
			{
				us_abortcommand("No library called %s", par[1]);
				return;
			}
		}
		if (lib == el_curlib && lib->nextlibrary == NOLIBRARY)
		{
			us_abortcommand("Cannot kill last library");
			return;
		}

		/* make sure it hasn't changed (unless "safe" option is given) */
		if (count <= 2 || namesamen(par[2], "safe", (INTSML)strlen(par[2])) != 0)
		{
			if (us_preventloss(0, par, lib, "Kill")) return;
		}

		/* switch to another library if killing current one */
		if (lib == el_curlib)
		{
			us_switchtolibrary(el_curlib->nextlibrary);
			ttyputmsgf("Current library is now %s", el_curlib->libname);
		} else ttyputmsgf("Library %s killed", lib->libname);

		/* remove display of all facets in this library */
		for(w = el_topwindow; w != NOWINDOW; w = w->nextwindow)
		{
			if (w->curnodeproto == NONODEPROTO) continue;
			if (w->curnodeproto->cell->lib == lib) us_clearwindow(w);
		}

		/* kill the library */
		killlibrary(lib);
		return;
	}

	if (namesamen(pp, "read", l) == 0 && l >= 1)
	{
		if (count < 2)
		{
			us_abortcommand("Usage: library read FILENAME [options]");
			return;
		}

		/* get library and file name */
		libf = par[1];
		(void)allocstring(&libn, us_skippath(libf), el_tempcluster);

		/* get style of input */
		makecurrent = 0;
		style = "binary";
		extn = ".elib";
		mergelib = NOLIBRARY;
		while (count >= 3)
		{
			l = strlen(pp = par[2]);
			if (namesamen(pp, "binary", l) == 0 && l >= 1) { style = "binary"; extn = ".elib"; } else
			if (namesamen(pp, "cif",    l) == 0 && l >= 1) { style = "cif";    extn = ".cif";  } else
			if (namesamen(pp, "dxf",    l) == 0 && l >= 1) { style = "dxf";    extn = ".dxf";  } else
			if (namesamen(pp, "edif",   l) == 0 && l >= 1) { style = "edif";   extn = ".edif"; } else
			if (namesamen(pp, "gds",    l) == 0 && l >= 1) { style = "gds";    extn = ".gds";  } else
			if (namesamen(pp, "sdf",    l) == 0 && l >= 1) { style = "sdf";    extn = ".sdf"; } else
			if (namesamen(pp, "text",   l) == 0 && l >= 1) { style = "text";   extn = ".txt";  } else
			if (namesamen(pp, "vhdl",   l) == 0 && l >= 1) { style = "vhdl";   extn = ".vhdl";  } else
			if (namesamen(pp, "make-current", l) == 0 && l >= 2) makecurrent++; else
			if (namesamen(pp, "merge", l) == 0 && l >= 2)
			{
				if (count < 4)
				{
					us_abortcommand("Usage: library read FILENAME merge LIBRARYNAME");
					efree(libn);
					return;
				}
				mergelib = getlibrary(par[3]);
				if (mergelib == NOLIBRARY)
				{
					us_abortcommand("Cannot find merge library %s", par[3]);
					efree(libn);
					return;
				}
				count--;
				par++;
			} else
			{
				us_abortcommand("Read options: binary|cif|dxf|edif|gds|sdf|text|vhdl|make-current|(merge LIB)");
				efree(libn);
				return;
			}
			count--;
			par++;
		}

		/* remove the extension from the library name */
		l = strlen(libn) - strlen(extn);
		if (namesame(&libn[l], extn) == 0)
		{
			libn[l] = 0;

			/* remove extension from library file if not binary */
			if (strcmp(extn, ".elib") != 0) libf[strlen(libf) - strlen(extn)] = 0;
		}

		/* turn off highlighting */
		us_clearhighlightcount();

		if (mergelib != NOLIBRARY)
		{
			/* place the new library file name onto the merge library */
			oldlibfile = mergelib->libfile;
			mergelib->libfile = libf;
			lib = mergelib;
		} else
		{
			/* reading a new library (or replacing an old one) */
			lib = getlibrary(libn);
			if (lib == NOLIBRARY)
			{
				/* create the library if it does not already exist in memory */
				lib = newlibrary(libn, libf);
				if (lib == NOLIBRARY)
				{
					us_abortcommand("Cannot create library %s", libn);
					efree(libn);
					return;
				}
			} else
			{
				/* not a new library: make sure the old library is saved */
				if (us_preventloss(0, newpar, lib, "Replace") != 0)
				{
					efree(libn);
					return;
				}

				/* erase the old library */
				eraselibrary(lib);
				if (reallocstring(&lib->libfile, libf, lib->cluster)) ttyputerr("No memory!");
				for(w = el_topwindow; w != NOWINDOW; w = w->nextwindow) w->curnodeproto = NONODEPROTO;
			}

			/* remove any GETPROTOs to facets in this library */
			us_removegetfacets();
		}

		/* read the library */
		retval = askaid(io_aid, "read", (INTBIG)lib, (INTBIG)style);
		if (mergelib != NOLIBRARY) mergelib->libfile = oldlibfile;
		if (retval == 0)
		{
			ttyputmsgf("Library %s read", libn);
			if (makecurrent != 0 || lib == el_curlib)
			{
				/* these operations are done directly to prevent undo */
				us_alignment = muldiv(us_alignment, lib->lambda[el_curtech->index], el_curtech->deflambda);
				nextvarchangequiet();
				(void)setvalkey((INTBIG)us_aid, VAID, us_alignment_obj, us_alignment,
					VINTEGER|VDONTSAVE);
				us_edgealignment = muldiv(us_edgealignment, lib->lambda[el_curtech->index], el_curtech->deflambda);
				nextvarchangequiet();
				(void)setvalkey((INTBIG)us_aid, VAID, us_alignment_edge, us_edgealignment,
					VINTEGER|VDONTSAVE);
				us_setalignment(0);

				selectlibrary(lib);
				us_setlambda(0);
				if (us_curnodeproto == NONODEPROTO || us_curnodeproto->index == 0)
				{
					nextvarchangequiet();
					(void)setvalkey((INTBIG)us_aid, VAID, us_current_node,
						(INTBIG)el_curtech->firstnodeproto, VNODEPROTO|VDONTSAVE);
					us_shadownodeproto(0, el_curtech->firstnodeproto);
				}
				np = el_curlib->curnodeproto;
				if (np != NONODEPROTO && el_curwindow != NOWINDOW)
				{
					if ((np->cellview->viewstate&TEXTVIEW) != 0)
					{
						/* text window: make an editor */
						if (us_makeeditor(el_curwindow, describenodeproto(np), &dummy, &dummy) == NOWINDOW)
						{
							efree(libn);
							return;
						}

						/* get the text that belongs here */
						var = getvalkey((INTBIG)np, VNODEPROTO, VSTRING|VISARRAY, el_facet_message);
						if (var == NOVARIABLE)
						{
							newpar[0] = "";
							var = setvalkey((INTBIG)np, VNODEPROTO, el_facet_message,
								(INTBIG)newpar, VSTRING|VISARRAY|(1<<VLENGTHSH));
							if (var == NOVARIABLE)
							{
								efree(libn);
								return;
							}
						}

						/* setup for editing */
						el_curwindow->curnodeproto = np;
						ed = el_curwindow->editor;
						ed->editobjaddr = (char *)np;
						ed->editobjtype = VNODEPROTO;
						ed->editobjqual = "FACET_message";
						ed->editobjvar = var;

						/* load the text into the window */
						us_suspendgraphics(el_curwindow);
						l = getlength(var);
						for(i=0; i<l; i++)
						{
							pp = ((char **)var->addr)[i];
							if (i == l-1 && *pp == 0) continue;
							us_addline(el_curwindow, i, pp);
						}
						us_resumegraphics(el_curwindow);
						el_curwindow->changehandler = us_textfacetchanges;
					} else
					{
						el_curwindow->curnodeproto = np;
						el_curwindow->state = (el_curwindow->state & ~(WINDOWTYPE|WINDOWSIMULATING)) | DISPWINDOW;
						el_curwindow->editor = NOEDITOR;
						el_curwindow->buttonhandler = DEFAULTBUTTONHANDLER;
						el_curwindow->charhandler = DEFAULTCHARHANDLER;
						el_curwindow->changehandler = DEFAULTCHANGEHANDLER;
						el_curwindow->termhandler = DEFAULTTERMHANDLER;
						el_curwindow->redisphandler = DEFAULTREDISPHANDLER;
						us_fullview(np, &lx, &hx, &ly, &hy);
						us_squarescreen(el_curwindow, NOWINDOW, 0, &lx, &hx, &ly, &hy);
						el_curwindow->screenlx = lx;
						el_curwindow->screenhx = hx;
						el_curwindow->screenly = ly;
						el_curwindow->screenhy = hy;
						computewindowscale(el_curwindow);
						us_setfacetname(el_curwindow);
						us_setfacetsize(el_curwindow);
						us_setgridsize(el_curwindow);
						if (el_curwindow->redisphandler != 0)
							(*el_curwindow->redisphandler)(el_curwindow);
					}
				}
			}
		} else if (mergelib == NOLIBRARY)
		{
			/* library is incomplete: remove it from the list */
			if (el_curlib != lib)
			{
				/* this is not the current library: safe to delete */
				lastlib = NOLIBRARY;
				for(ol = el_curlib; ol != NOLIBRARY; ol = ol->nextlibrary)
				{
					if (ol == lib) break;
					lastlib = ol;
				}
				if (lastlib != NOLIBRARY) lastlib->nextlibrary = lib->nextlibrary;
			} else
			{
				/* current library is incomplete */
				if (el_curlib->nextlibrary == NOLIBRARY)
				{
					/* must create a new library: it is the only one */
					el_curlib = alloclibrary();
					(void)allocstring(&el_curlib->libname, lib->libname, el_curlib->cluster);
					(void)allocstring(&el_curlib->libfile, lib->libfile, el_curlib->cluster);
					el_curlib->nextlibrary = NOLIBRARY;
				} else el_curlib = el_curlib->nextlibrary;

				us_clearhighlightcount();
				us_adjustlambda(el_curtech->deflambda, el_curlib->lambda[el_curtech->index]);
				us_setlambda(0);
				if (us_curnodeproto == NONODEPROTO || us_curnodeproto->index == 0)
				{
					if ((us_state&NONPERSISTENTCURNODE) != 0) us_setnodeproto(NONODEPROTO); else
						us_setnodeproto(el_curtech->firstnodeproto);
				}
				if (el_curlib->curnodeproto != NONODEPROTO)
				{
					newpar[0] = describenodeproto(el_curlib->curnodeproto);
					us_editfacet(1, newpar);
				}
			}
		}
		efree(libn);
		return;
	}

	if (namesamen(pp, "use", l) == 0 && l >= 1)
	{
		if (count <= 1)
		{
			us_abortcommand("Usage: library use LIBNAME");
			return;
		}

		/* find actual library name */
		libn = us_skippath(par[1]);

		lib = getlibrary(libn);
		if (lib != NOLIBRARY)
		{
			if (el_curlib == lib) return;
			us_switchtolibrary(lib);
			return;
		}

		/* create new library */
		lib = newlibrary(libn, par[1]);
		if (lib == NOLIBRARY)
		{
			us_abortcommand("Cannot create library %s", libn);
			return;
		}

		/* make this library the current one */
		us_switchtolibrary(lib);
		ttyputmsgf("New library: %s", libn);
		return;
	}

	if (namesamen(pp, "new", l) == 0 && l >= 1)
	{
		if (count <= 1)
		{
			us_abortcommand("Usage: library new LIBNAME");
			return;
		}

		/* find actual library name */
		libn = us_skippath(par[1]);

		lib = getlibrary(libn);
		if (lib != NOLIBRARY)
		{
			us_abortcommand("Already a library called '%s'", libn);
			return;
		}

		/* create new library */
		lib = newlibrary(libn, par[1]);
		if (lib == NOLIBRARY)
		{
			us_abortcommand("Cannot create library %s", libn);
			return;
		}

		ttyputmsgf("New library: %s", libn);
		return;
	}

	if (namesamen(pp, "save", l) == 0 && l >= 2)
	{
		/* save every library that has been modified */
		for(lib = el_curlib; lib != NOLIBRARY; lib = lib->nextlibrary)
		{
			if ((lib->userbits&LIBCHANGED) == 0) continue;

			/* save the library in binary format */
			if (askaid(io_aid, "write", (INTBIG)lib, (INTBIG)"binary") != 0) return;
		}

		/* rewind session logging if it is on */
		if (us_logrecord != NULL)
		{
			us_logfinishrecord();
			us_logstartrecord();
		}
		return;
	}

	if (namesamen(pp, "write", l) == 0 && l >= 1)
	{
		if (count < 2)
		{
			us_abortcommand("Usage: library write LIBNAME [options]");
			return;
		}

		/* get style of output */
		style = "binary";
		if (count >= 3)
		{
			l = strlen(pp = par[2]);
			if (l >= 1 && namesamen(pp, "binary", l) == 0) style = "binary"; else
			if (l >= 1 && namesamen(pp, "cif", l) == 0) style = "cif"; else
			if (l >= 1 && namesamen(pp, "dxf", l) == 0) style = "dxf"; else
			if (l >= 1 && namesamen(pp, "edif", l) == 0) style = "edif"; else
			if (l >= 1 && namesamen(pp, "gds", l) == 0) style = "gds"; else
			if (l >= 1 && namesamen(pp, "hpgl", l) == 0) style = "hpgl"; else
			if (l >= 1 && namesamen(pp, "l", l) == 0) style = "l"; else
			if (l >= 2 && namesamen(pp, "postscript", l) == 0) style = "postscript"; else
			if (l >= 2 && namesamen(pp, "quickdraw", l) == 0) style = "quickdraw"; else
			if (l >= 1 && namesamen(pp, "text", l) == 0) style = "text"; else
			{
				us_abortcommand("File formats: binary|cif|dxf|edif|gds|hpgl|l|postscript|quickdraw|text");
				return;
			}
		}

		/* get library to write */
		lib = getlibrary(par[1]);
		if (lib == NOLIBRARY)
		{
			us_abortcommand("No library called %s", par[1]);
			return;
		}

		/* do it */
		if (askaid(io_aid, "write", (INTBIG)lib, (INTBIG)style) != 0) return;

		/* rewind session recording if writing in permanent format */
		if ((*style == 'b' || *style == 't') && us_logrecord != NULL)
		{
			us_logfinishrecord();
			us_logstartrecord();
		}
		return;
	}

	us_abortcommand("Bad LIBRARY option: %s", par[0]);
}

void us_macbegin(INTSML count, char *par[])
{
	REGISTER INTSML i, l, verbose, execute;
	REGISTER INTBIG key;
	REGISTER char *ch, *pp;
	char *newmacro[3];
	extern COMCOMP us_macbeginnp;
	REGISTER VARIABLE *var;

	var = getvalkey((INTBIG)us_aid, VAID, VSTRING, us_macrobuilding);
	if (var != NOVARIABLE)
	{
		us_abortcommand("Already defining a macro");
		return;
	}

	/* get macro name */
	if (count == 0)
	{
		count = ttygetparam("Macro name: ", &us_macbeginnp, MAXPARS, par);
		if (count == 0)
		{
			us_abortedmsg();
			return;
		}
	}
	pp = par[0];

	/* get switches */
	verbose = 0;   execute = 2;
	for(i=1; i<count; i++)
	{
		ch = par[i];
		l = strlen(ch);
		if (namesamen(ch, "verbose", l) == 0) verbose = 1; else
		if (namesamen(ch, "no-execute", l) == 0) execute = 0; else
		{
			us_abortcommand("Usage: macbegin MACRONAME [verbose|no-execute]");
			return;
		}
	}

	/* make sure macro name isn't overloading existing command */
	for(i=0; us_lcommand[i].name != 0; i++)
		if (namesame(pp, us_lcommand[i].name) == 0)
	{
		us_abortcommand("There is a command with that name");
		return;
	}

	/* see if macro name already exists */
	if (us_getmacro(pp) == NOVARIABLE)
	{
		/* new macro name, check for validity */
		for(ch = pp; *ch != 0; ch++) if (*ch == ' ' || *ch == '\t')
		{
			us_abortcommand("Macro name must not have embedded spaces");
			return;
		}
	}

	/* create the macro variable */
	(void)initinfstr();
	(void)addstringtoinfstr("USER_macro_");
	(void)addstringtoinfstr(pp);
	key = makekey(returninfstr());

	/* build the first two lines of the macro */
	(void)initinfstr();
	if (us_curmacropack != NOMACROPACK)
	{
		(void)addstringtoinfstr("macropack=");
		(void)addstringtoinfstr(us_curmacropack->packname);
	}
	if (verbose != 0) (void)addstringtoinfstr(" verbose");
	if (execute == 0) (void)addstringtoinfstr(" noexecute");
	newmacro[0] = returninfstr();
	newmacro[1] = "";		/* no parameters yet */
	newmacro[2] = "";		/* no parsing structures yet */
	(void)setvalkey((INTBIG)us_aid, VAID, key, (INTBIG)newmacro,
		VSTRING|VISARRAY|(3<<VLENGTHSH)|VDONTSAVE);

	/* turn on macro remembering */
	(void)setvalkey((INTBIG)us_aid, VAID, us_macrobuilding, (INTBIG)pp, VSTRING|VDONTSAVE);
	(void)setvalkey((INTBIG)us_aid, VAID, us_macrorunning, (INTBIG)pp, VSTRING|VDONTSAVE);

	ttyputmsgf("Remembering %s...", pp);
}

void us_macdone(INTSML count, char *par[])
{
	REGISTER VARIABLE *var;

	var = getvalkey((INTBIG)us_aid, VAID, VSTRING, us_macrorunning);
	if (var == NOVARIABLE)
	{
		us_abortcommand("Can only stop execution of a macro if one is running");
		return;
	}
	us_state |= MACROTERMINATED;
}

void us_macend(INTSML count, char *par[])
{
	REGISTER VARIABLE *var;

	var = getvalkey((INTBIG)us_aid, VAID, VSTRING, us_macrobuilding);
	if (var == NOVARIABLE)
	{
		us_abortcommand("Not in a macro");
		return;
	}
	ttyputmsgf("%s defined", (char *)var->addr);
	(void)delvalkey((INTBIG)us_aid, VAID, us_macrobuilding);
	if (getvalkey((INTBIG)us_aid, VAID, VSTRING, us_macrorunning) != NOVARIABLE)
		(void)delvalkey((INTBIG)us_aid, VAID, us_macrorunning);
}

void us_menu(INTSML count, char *par[])
{
	REGISTER INTSML position, x, y, l, s, i, total, coms, changed,
		j, len, arctot, pintot, comptot, puretot, fun, autoset;
	REGISTER INTBIG varkey;
	INTSML butstate;
#ifndef	MACOS
	INTSML wid, hei;
#endif
	REGISTER char *pp, **temp;
	char number[10], *pt, *implicit[6];
	POPUPMENU *pm, *cpm;
	REGISTER POPUPMENUITEM *mi, *miret;
	NODEPROTO *np;
	ARCPROTO *ap;
	REGISTER USERCOM *uc;
	REGISTER VARIABLE *var;
	COMCOMP *comarray[MAXPARS];
	COMMANDBINDING commandbinding;
	extern COMCOMP us_menup;
	extern KEYWORD *us_pathiskey;

	/* with no arguments, toggle the menu state */
	if (count == 0)
	{
		count = ttygetparam("Menu display option: ", &us_menup, MAXPARS, par);
		if (count == 0)
		{
			us_abortedmsg();
			return;
		}
	}

	/* check for simple changes of the menu display */
	l = strlen(pp = par[0]);
	if (namesamen(pp, "on", l) == 0 && l >= 2)
	{
		if ((us_aid->aidstate&MENUON) != 0)
		{
			us_abortcommand("Menu is already displayed");
			return;
		}

		/* save highlighting */
		us_pushhighlight();
		us_clearhighlightcount();

		startobjectchange((INTBIG)us_aid, VAID);
		(void)setval((INTBIG)us_aid, VAID, "aidstate", us_aid->aidstate|MENUON,
			VINTEGER);
		endobjectchange((INTBIG)us_aid, VAID);

		/* restore highlighting */
		(void)us_pophighlight(0);
		return;
	}

	if (namesamen(pp, "off", l) == 0 && l >= 2)
	{
		if ((us_aid->aidstate&MENUON) == 0)
		{
			us_abortcommand("Menu is not displayed");
			return;
		}

		/* save highlighting */
		us_pushhighlight();
		us_clearhighlightcount();

		startobjectchange((INTBIG)us_aid, VAID);
		(void)setval((INTBIG)us_aid, VAID, "aidstate", us_aid->aidstate & ~MENUON, VINTEGER);
		endobjectchange((INTBIG)us_aid, VAID);

		/* restore highlighting */
		(void)us_pophighlight(0);
		return;
	}

	if (namesamen(pp, "dopopup", l) == 0 && l >= 1)
	{
		if (count <= 1)
		{
			implicit[0] = "tellaid";
			implicit[1] = "user";
			uc = us_buildcommand(2, implicit);
		} else uc = us_buildcommand((INTSML)(count-1), &par[1]);
		if (uc == NOUSERCOM)
		{
			us_abortcommand("Command '%s' not found", par[1]);
			return;
		}

		/* loop through the popup menus of this command */
		for(;;)
		{
			/* convert to a string of command completion objects */
			total = us_fillcomcomp(uc, comarray);
			if (total <= uc->count) break;

			/* count the number of options */
			us_pathiskey = comarray[uc->count]->ifmatch;
			pt = "";
			(void)(*comarray[uc->count]->toplist)(&pt);
			for(coms=0; (pp = (*comarray[uc->count]->nextcomcomp)()); coms++) ;
			if ((comarray[uc->count]->interpret&INCLUDENOISE) != 0) coms++;

			/* build the prompt */
			(void)initinfstr();
			(void)addstringtoinfstr(comarray[uc->count]->noise);

			/* if there are no choices, prompt directly */
			if (coms == 0)
			{
				(void)addstringtoinfstr(": ");
				pp = ttygetline(returninfstr());
				if (*pp == 0) break;
				(void)allocstring(&uc->word[uc->count], pp, us_aid->cluster);
				uc->count++;
				continue;
			}

			/* create the popup menu */
			pm = (POPUPMENU *)emalloc(sizeof(POPUPMENU), el_tempcluster);
			if (pm == 0) return;
			pm->name = "noname";
			pm->header = returninfstr();
			pm->total = coms;
			mi = (POPUPMENUITEM *)emalloc((coms * (sizeof (POPUPMENUITEM))), el_tempcluster);
			if (mi == 0) return;
			pm->list = mi;

			/* fill the popup menu */
			pt = "";
			(void)(*comarray[uc->count]->toplist)(&pt);
			for(i=0; i<coms; i++)
			{
				mi[i].valueparse = NOCOMCOMP;
				mi[i].response = NOUSERCOM;

				/* if noise is includable, do it first */
				if (i == 0 && (comarray[uc->count]->interpret&INCLUDENOISE) != 0)
				{
					mi[i].attribute = comarray[uc->count]->noise;
					if (comarray[uc->count]->def != 0)
					{
						mi[i].maxlen = maxi(20, strlen(comarray[uc->count]->def));
						mi[i].value = (char *)emalloc((mi[i].maxlen+1), el_tempcluster);
						(void)strcpy(mi[i].value, comarray[uc->count]->def);
					} else
					{
						mi[i].maxlen = 20;
						mi[i].value = (char *)emalloc((mi[i].maxlen+1), el_tempcluster);
						(void)strcpy(mi[i].value, "*");
					}
					continue;
				}

				/* normal entry */
				mi[i].attribute = (*comarray[uc->count]->nextcomcomp)();
				mi[i].value = 0;
				mi[i].maxlen = -1;

				/* see what would come next with this keyword */
				(void)allocstring(&uc->word[uc->count], mi[i].attribute, el_tempcluster);
				uc->count++;
				total = us_fillcomcomp(uc, comarray);
				uc->count--;
				efree(uc->word[uc->count]);
				if (total > uc->count+1)
				{
					if ((comarray[uc->count+1]->interpret&INPUTOPT) != 0)
					{
						mi[i].valueparse = comarray[uc->count+1];
						if (comarray[uc->count+1]->def != 0)
						{
							mi[i].maxlen = maxi(20, strlen(comarray[uc->count+1]->def));
							mi[i].value = (char *)emalloc((mi[i].maxlen+1), el_tempcluster);
							(void)strcpy(mi[i].value, comarray[uc->count+1]->def);
						} else
						{
							mi[i].maxlen = 20;
							mi[i].value = (char *)emalloc((mi[i].maxlen+1), el_tempcluster);
							(void)strcpy(mi[i].value, "*");
						}
					}
				}
			}

			/* invoke the popup menu */
			butstate = 1;
			cpm = pm;
			miret = us_popupmenu(&cpm, &butstate, 1, -1, -1, 0);
			if (miret == 0) us_abortcommand("Sorry, this display cannot support popup menus");
			if (miret == NOPOPUPMENUITEM || miret == 0)
			{
				efree((char *)mi);
				efree((char *)pm);
				break;
			}

			/* see if multiple answers were given */
			changed = 0;
			for(i=0; i<pm->total; i++) if (mi[i].changed != 0) changed++;
			if (changed > 1 || (changed == 1 && miret->changed == 0))
			{
				if ((comarray[uc->count]->interpret&MULTIOPT) == 0)
				{
					us_abortcommand("Multiple commands cannot be selected");
					return;
				}

				/* tack on all of the answers */
				for(i=0; i<pm->total; i++) if (mi[i].changed != 0)
				{
					if (uc->count >= MAXPARS-1) break;
					(void)allocstring(&uc->word[uc->count++], mi[i].attribute, us_aid->cluster);
					(void)allocstring(&uc->word[uc->count++], mi[i].value, us_aid->cluster);
				}
				if (miret->changed == 0 && uc->count < MAXPARS)
					(void)allocstring(&uc->word[uc->count++], miret->attribute, us_aid->cluster);

				for(i=0; i<pm->total; i++)
					if (mi[i].maxlen >= 0) efree(mi[i].value);
				efree((char *)mi);
				efree((char *)pm);
				break;
			}

			/* if noise was mentioned, copy the value */
			if (miret == mi && miret->changed != 0 &&
				(comarray[uc->count]->interpret&INCLUDENOISE) != 0)
			{
				(void)allocstring(&uc->word[uc->count++], miret->value, us_aid->cluster);
			} else
			{
				(void)allocstring(&uc->word[uc->count++], miret->attribute, us_aid->cluster);
				if (miret->changed != 0)
					(void)allocstring(&uc->word[uc->count++], miret->value, us_aid->cluster);
			}
			for(i=0; i<pm->total; i++)
				if (mi[i].maxlen >= 0) efree(mi[i].value);
			efree((char *)mi);
			efree((char *)pm);
		}

		/* issue the completed command */
		us_execute(uc, us_aid->aidstate&ECHOBIND, 0, 1);
		us_freeusercom(uc);
		return;
	}

	if (namesamen(pp, "popup", l) == 0 && l >= 1)
	{
		if (count <= 2)
		{
			us_abortcommand("Usage: menu popup NAME OPTIONS");
			return;
		}

		/* check out the name of the pop-up menu */
		pm = us_getpopupmenu(par[1]);

		/* determine what to do with the menu */
		l = strlen(pp = par[2]);
		if (namesamen(pp, "header", l) == 0 && l >= 1)
		{
			/* set header: ensure it is specified */
			if (count <= 3)
			{
				us_abortcommand("Usage: menu popup header MESSAGE");
				return;
			}
			if (pm == NOPOPUPMENU)
			{
				us_abortcommand("No popup menu called %s", par[1]);
				return;
			}
			(void)initinfstr();
			(void)addstringtoinfstr("USER_binding_popup_");
			(void)addstringtoinfstr(par[1]);
			varkey = makekey(returninfstr());
			var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, varkey);
			if (var == NOVARIABLE)
			{
				us_abortcommand("Cannot find the menu");
				return;
			}
			(void)setindkey((INTBIG)us_aid, VAID, varkey, 0, (INTBIG)par[3]);
			return;
		}
		if (namesamen(pp, "size", l) == 0 && l >= 1)
		{
			/* set size: ensure it is specified */
			if (count <= 3)
			{
				us_abortcommand("Usage: menu popup NAME size SIZE");
				return;
			}
			s = myatoi(par[3]);
			if (s <= 0)
			{
				us_abortcommand("Popup menu size must be positive");
				return;
			}

			/* set size of menu */
			if (pm == NOPOPUPMENU)
			{
				/* create the variable with the popup menu */
				temp = (char **)emalloc(((s+1) * (sizeof (char *))), el_tempcluster);
				(void)allocstring(&temp[0], "MENU!", el_tempcluster);
				for(i=0; i<s; i++)
				{
					(void)initinfstr();
					(void)addstringtoinfstr("inputpopup=");
					(void)addstringtoinfstr(par[1]);
					(void)addstringtoinfstr(" command=bind set popup ");
					(void)addstringtoinfstr(par[1]);
					(void)sprintf(number, " %d", i);
					(void)addstringtoinfstr(number);
					(void)allocstring(&temp[i+1], returninfstr(), el_tempcluster);
				}
				(void)initinfstr();
				(void)addstringtoinfstr("USER_binding_popup_");
				(void)addstringtoinfstr(par[1]);
				varkey = makekey(returninfstr());
				(void)setvalkey((INTBIG)us_aid, VAID, varkey, (INTBIG)temp,
					VSTRING|VISARRAY|VDONTSAVE|((s+1)<<VLENGTHSH));
				for(i=0; i<=s; i++) efree(temp[i]);
				efree((char *)temp);
			} else
			{
				/* get the former popup menu */
				(void)initinfstr();
				(void)addstringtoinfstr("USER_binding_popup_");
				(void)addstringtoinfstr(par[1]);
				varkey = makekey(returninfstr());
				var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, varkey);
				if (var == NOVARIABLE)
				{
					us_abortcommand("Cannot find popup menu %s", par[1]);
					return;
				}
				len = getlength(var);

				/* create the new popup menu */
				temp = (char **)emalloc(((s+1) * (sizeof (char *))), el_tempcluster);
				if (temp == 0) return;
				(void)allocstring(&temp[0], ((char **)var->addr)[0], el_tempcluster);

				/* copy the existing menu entries */
				for(i=1; i<mini((s+1), len); i++)
				{
					us_parsebinding(((char **)var->addr)[i], &commandbinding);
					(void)initinfstr();
					if (commandbinding.inputpopup != 0) (void)addstringtoinfstr("input");
					(void)addstringtoinfstr("popup=");
					(void)addstringtoinfstr(par[1]);
					if (commandbinding.menumessage != 0)
					{
						(void)addstringtoinfstr(" message=\"");
						(void)addstringtoinfstr(commandbinding.menumessage);
						(void)addtoinfstr('"');
					}
					(void)addstringtoinfstr(" command=");
					(void)addstringtoinfstr(commandbinding.command);
					(void)allocstring(&temp[i], returninfstr(), el_tempcluster);
					us_freebindingparse(&commandbinding);
				}

				for(j=i; j<s+1; j++)
				{
					(void)initinfstr();
					(void)addstringtoinfstr("inputpopup=");
					(void)addstringtoinfstr(par[1]);
					(void)addstringtoinfstr(" command=bind set popup ");
					(void)addstringtoinfstr(par[1]);
					(void)sprintf(number, " %d", j);
					(void)addstringtoinfstr(number);
					(void)allocstring(&temp[j], returninfstr(), el_tempcluster);
				}
				(void)setvalkey((INTBIG)us_aid, VAID, varkey, (INTBIG)temp,
					VSTRING|VISARRAY|VDONTSAVE|((s+1)<<VLENGTHSH));
				for(i=0; i<=s; i++) efree(temp[i]);
				efree((char *)temp);
			}
			return;
		}
		us_abortcommand("Unknown popup menu option: %s", pp);
		return;
	}

	if (namesamen(pp, "setmenubar", l) == 0 && l >= 2)
	{
		/* check the menus for validity */
		for(i=1; i<count; i++)
		{
			/* check out the name of the pop-up menu */
			if (us_getpopupmenu(par[i]) == NOPOPUPMENU)
			{
				us_abortcommand("Unknown popup menu: %s", par[i]);
				return;
			}
		}
#if defined(MACOS) || defined(USETK) || defined(WIN32)
		gra_nativemenuload((INTSML)(count-1), &par[1]);
#else
		if (count > 1)
		{
			if (us_needwindow()) return;

			/* make sure menu bar can be handled */
			if (us_graphicshas(CANINVERTBOX|CANSAVEBOX|CANTRACKCURSOR) == 0)
			{
				us_abortcommand("Insufficient graphics capability for a menu bar");
				return;
			}

			/* turn on the menu bar with these popups */
			us_menubarsize = MENUBARHEIGHT;

			/* deallocate any former menu bar */
			if (us_pulldownmenucount != 0)
			{
				efree((char *)us_pulldowns);
				efree((char *)us_pulldownmenupos);
			}
			us_pulldownmenucount = 0;

			/* allocate space for new menu bar */
			us_pulldowns = (POPUPMENU **)emalloc((count-1) * (sizeof (POPUPMENU *)), us_aid->cluster);
			us_pulldownmenupos = (INTSML *)emalloc((count-1) * SIZEOFINTSML, us_aid->cluster);
			if (us_pulldowns == 0 || us_pulldownmenupos == 0) return;
			us_pulldownmenucount = count-1;

			/* load the information */
			us_settextsize(el_curwindow, TXTMENU);
			x = 0;
			for(i=1; i<count; i++)
			{
				pm = us_getpopupmenu(par[i]);
				us_pulldowns[i-1] = pm;
				us_textsize(el_curwindow, pm->header, &wid, &hei);
				x += wid + 12;
				us_pulldownmenupos[i-1] = x;
			}

			/* display the pulldown menu */
			us_drawmenu(0, 0);
		} else
		{
			/* turn off the menu bar */
			if (us_menubarsize == 0) return;
			us_menubarsize = 0;

			/* deallocate any space */
			if (us_pulldownmenucount != 0)
			{
				efree((char *)us_pulldowns);
				efree((char *)us_pulldownmenupos);
			}
			us_pulldownmenucount = 0;
			us_drawmenu(0, 0);
		}
#endif
		return;
	}

	/* see if new fixed menu location has been specified */
	position = us_menupos;
	x = us_menux;   y = us_menuy;
	if (namesamen(pp, "top", l) == 0 && l >= 1)
	{
		position = 0;
		count--;   par++;
	} else if (namesamen(pp, "bottom", l) == 0 && l >= 1)
	{
		position = 1;
		count--;   par++;
	} else if (namesamen(pp, "left", l) == 0 && l >= 1)
	{
		position = 2;
		count--;   par++;
	} else if (namesamen(pp, "right", l) == 0 && l >= 1)
	{
		position = 3;
		count--;   par++;
	}

	/* check for new fixed menu size */
	autoset = 0;
	if (count > 0)
	{
		if (namesamen(par[0], "size", (INTSML)(strlen(par[0]))) != 0)
		{
			us_abortcommand("Bad MENU option: %s", par[0]);
			return;
		}
		if (count == 2 && namesamen(par[1], "auto", (INTSML)(strlen(par[1]))) == 0)
		{
			autoset = 1;
			arctot = pintot = comptot = puretot = 0;
			for(ap = el_curtech->firstarcproto; ap != NOARCPROTO; ap = ap->nextarcproto) arctot++;
			for(np = el_curtech->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
			{
				fun = (np->userbits&NFUNCTION) >> NFUNCTIONSH;
				if (fun == NPPIN) pintot++; else
					if (fun == NPNODE) puretot++; else
						comptot++;
			}
			if (pintot + comptot == 0) pintot = puretot;
			x = arctot + pintot + comptot;   y = 1;
			if (x > 40)
			{
				x = (x+2) / 3;
				y = 3;
			} else if (x > 20)
			{
				x = (x+1) / 2;
				y = 2;
			}
			if (position > 1) { i = x;   x = y;   y = i; }
		} else if (count == 3)
		{
			x = atoi(par[1]);
			y = atoi(par[2]);
			if (x <= 0 || y <= 0)
			{
				us_abortcommand("Menu sizes must be positive numbers");
				return;
			}
		} else
		{
			us_abortcommand("Usage: menu size (ROWS COLUMNS) | auto");
			return;
		}
	}

	/* set the fixed menu up */
	if (x == us_menux && y == us_menuy && position == us_menupos &&
		(us_aid->aidstate&MENUON) != 0 && autoset == 0)
	{
		ttyputmsgf("Menu has not changed");
		return;
	}

	startobjectchange((INTBIG)us_aid, VAID);
	if (x != us_menux || y != us_menuy || position != us_menupos || autoset != 0)
		(void)us_setmenusize(x, y, position, autoset);
	if ((us_aid->aidstate&MENUON) == 0)
		(void)setval((INTBIG)us_aid, VAID, "aidstate", us_aid->aidstate|MENUON, VINTEGER);
	endobjectchange((INTBIG)us_aid, VAID);
}

void us_mirror(INTSML count, char *par[])
{
	REGISTER NODEINST *ni;
	REGISTER INTSML amt, gamt, i;
	REGISTER char *pp;
	REGISTER VARIABLE *var;
	REGISTER HIGHLIGHT *high;
	XARRAY transtz, rot, transfz, t1, t2;
	INTBIG gx, gy, cx, cy;
	extern COMCOMP us_mirrorp;

	ni = (NODEINST *)us_getobject(OBJNODEINST, 0);
	if (ni == NONODEINST) return;

	/* disallow mirroring if lock is on */
	if (us_protolocked(ni->proto) != 0) return;

	if (count == 0)
	{
		count = ttygetparam("Horizontally or vertically? ", &us_mirrorp, MAXPARS, par);
		if (count == 0)
		{
			us_abortedmsg();
			return;
		}
	}
	pp = par[0];
	if (*pp != 'h' && *pp != 'v')
	{
		us_abortcommand("Usage: mirror horizontal|vertical");
		return;
	}

	/* determine amount of rotation to effect the mirror */
	if (*pp == 'h')
	{
		if (ni->transpose == 0) amt = 2700; else amt = 900;
	} else
	{
		if (ni->transpose == 0) amt = 900; else amt = 2700;
	}

	/* determine translation when mirroring about grab or trace point */
	if (count >= 2) i = strlen(par[1]);
	if (count >= 2 && namesamen(par[1], "about-grab-point", i) == 0 && i >= 7)
	{
		/* find the grab point */
		corneroffset(ni, ni->proto, ni->rotation, ni->transpose, &gx, &gy);
		gx += ni->lowx;   gy += ni->lowy;

		/* build transformation for this operation */
		transid(transtz);   transtz[2][0] = -gx;   transtz[2][1] = -gy;
		if (*pp == 'h') gamt = 2700; else gamt = 900;
		makeangle(gamt, 1, rot);
		transid(transfz);   transfz[2][0] = gx;   transfz[2][1] = gy;
		transmult(transtz, rot, t1);
		transmult(t1, transfz, t2);

		/* find out where true center moves */
		cx = (ni->lowx+ni->highx)/2;   cy = (ni->lowy+ni->highy)/2;
		xform(cx, cy, &gx, &gy, t2);
		gx -= cx;   gy -= cy;
	} else if (count >= 2 && namesamen(par[1], "about-trace-point", i) == 0 && i >= 7)
	{
		/* get the trace information */
		var = gettrace(ni);
		if (var == NOVARIABLE)
		{
			us_abortcommand("Highlighted node must have trace information");
			return;
		}

		/* find the pivot point */
		high = us_getonehighlight();
		i = high->frompoint;   if (i != 0) i--;
		makerot(ni, t1);
		gx = (ni->highx + ni->lowx) / 2;
		gy = (ni->highy + ni->lowy) / 2;
		xform(((INTBIG *)var->addr)[i*2]+gx, ((INTBIG *)var->addr)[i*2+1]+gy, &gx, &gy, t1);

		/* build transformation for this operation */
		transid(transtz);   transtz[2][0] = -gx;   transtz[2][1] = -gy;
		if (*pp == 'h') gamt = 2700; else gamt = 900;
		makeangle(gamt, 1, rot);
		transid(transfz);   transfz[2][0] = gx;   transfz[2][1] = gy;
		transmult(transtz, rot, t1);
		transmult(t1, transfz, t2);

		/* find out where true center moves */
		cx = (ni->lowx+ni->highx)/2;   cy = (ni->lowy+ni->highy)/2;
		xform(cx, cy, &gx, &gy, t2);
		gx -= cx;   gy -= cy;
	} else gx = gy = 0;

	/* save highlighting */
	us_pushhighlight();
	us_clearhighlightcount();

	/* mirror the node */
	startobjectchange((INTBIG)ni, VNODEINST);
	modifynodeinst(ni, gx, gy, gx, gy, amt, (INTSML)(1-ni->transpose*2));
	endobjectchange((INTBIG)ni, VNODEINST);

	/* restore highlighting */
	(void)us_pophighlight(1);

	if (*pp == 'h') ttyputmsgf("Node mirrored horizontally"); else
		ttyputmsgf("Node mirrored vertically");
}

static DIALOGITEM usr_movetodialogitems[] =
{
 /*  1 */ {0, {64,96,88,160}, BUTTON, "OK"},
 /*  2 */ {0, {64,8,88,72}, BUTTON, "Cancel"},
 /*  3 */ {0, {8,48,24,142}, EDITTEXT, ""},
 /*  4 */ {0, {8,20,24,42}, MESSAGE, "X:"},
 /*  5 */ {0, {32,48,48,142}, EDITTEXT, ""},
 /*  6 */ {0, {32,20,48,42}, MESSAGE, "Y:"}
};
DIALOG usr_movetodialog = {{50,75,147,244}, "Move To Location", 6, usr_movetodialogitems};

void us_move(INTSML count, char *par[])
{
	REGISTER NODEINST *ni, **nodelist;
	REGISTER NODEPROTO *np;
	REGISTER ARCINST *ai;
	REGISTER INTBIG amt, i, j, wid, len;
	REGISTER INTSML l, total, itemHit;
	INTBIG dx, dy, xcur, ycur, lx, hx, ly, hy, bestlx, bestly, p1x, p1y, p2x, p2y, snapx, snapy;
	HIGHLIGHT high;
	REGISTER GEOM **list;
	HIGHLIGHT thishigh, otherhigh;
	REGISTER char *pp;
	static POLYGON *poly = NOPOLYGON;
	REGISTER VARIABLE *var;
	/* get polygon */
	if (poly == NOPOLYGON) poly = allocpolygon(4, us_aid->cluster);

	/* make sure there is a current facet */
	np = us_needfacet();
	if (np == NONODEPROTO) return;

	/* get the objects to be moved */
	list = us_gethighlighted(OBJNODEINST | OBJARCINST);

	/* see if a text object is highlighted */
	var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_highlighted);
	if (var != NOVARIABLE)
	{
		if (getlength(var) == 1)
		{
			(void)us_makehighlight(((char **)var->addr)[0], &high);
			if ((high.status&HIGHTYPE) == HIGHTEXT)
			{
				us_movetext(&high, count, par);
				return;
			}
			if ((high.status&HIGHTYPE) == HIGHBBOX) list = us_gethighlighted(OBJNODEINST);
		}
	}

	if (list[0] == NOGEOM)
	{
		us_abortcommand("First select objects to move");
		return;
	}

	/* make sure they are all in the same facet */
	for(i=1; list[i] != NOGEOM; i++) if (np != geomparent(list[i]))
	{
		us_abortcommand("All moved objects must be in the same facet");
		return;
	}

	/* mark all nodes (including those touched by highlighted arcs) */
	for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
		ni->temp1 = 0;
	for(i=0; list[i] != NOGEOM; i++) if (list[i]->entrytype == OBJARCINST)
	{
		ai = list[i]->entryaddr.ai;
		ai->end[0].nodeinst->temp1 = ai->end[1].nodeinst->temp1 = 1;
	}
	for(i=0; list[i] != NOGEOM; i++) if (list[i]->entrytype == OBJNODEINST)
		list[i]->entryaddr.ni->temp1 = 1;

	/* count the number of nodes, excluding locked ones */
	i = 0;
	for(total=0, ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		if (ni->temp1 == 0) continue;
		i++;
		if (us_islocked(ni->proto) != 0)
		{
			ni->temp1 = 0;
			continue;
		}
		total++;
	}
	if (total == 0)
	{
		if (i != 0) us_abortcommand("All selected objects are locked");
		return;
	}

	/* remove from list all arcs touching locked nodes */
	for(i=j=0; list[i] != NOGEOM; i++)
	{
		if (list[i]->entrytype == OBJARCINST)
		{
			ai = list[i]->entryaddr.ai;
			if (us_islocked(ai->end[0].nodeinst->proto) != 0 ||
				us_islocked(ai->end[1].nodeinst->proto) != 0) continue;
		}
		list[j++] = list[i];
	}
	list[j] = NOGEOM;

	/* build a list that includes all nodes touching moved arcs */
	nodelist = (NODEINST **)emalloc((total * (sizeof (NODEINST *))), el_tempcluster);
	if (nodelist == 0) return;
	for(i=0, ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
		if (ni->temp1 != 0) nodelist[i++] = ni;

	/* figure out lower-left corner of this collection of objects */
	us_getlowleft(nodelist[0], &bestlx, &bestly);
	for(i=1; i<total; i++)
	{
		us_getlowleft(nodelist[i], &lx, &ly);
		if (lx < bestlx) bestlx = lx;
		if (ly < bestly) bestly = ly;
	}
	for(i=0; list[i] != NOGEOM; i++) if (list[i]->entrytype == OBJARCINST)
	{
		ai = list[i]->entryaddr.ai;
		wid = ai->width - arcwidthoffset(ai->proto);
		makearcpoly(ai->length, wid, ai, poly, FILLED);
		getbbox(poly, &lx, &hx, &ly, &hy);
		if (lx < bestlx) bestlx = lx;
		if (ly < bestly) bestly = ly;
	}

	/* special case when moving one node: account for facet center */
	if (total == 1 && list[1] == NOGEOM)
	{
		/* get standard corner offset */
		ni = nodelist[0];
		corneroffset(ni, ni->proto, ni->rotation, ni->transpose, &lx, &ly);
		bestlx = ni->lowx + lx;
		bestly = ni->lowy + ly;
	}

	/* special case if there is exactly one snap point */
	if (us_getonesnappoint(&snapx, &snapy) != 0)
	{
		bestlx = snapx;   bestly = snapy;
	}

	/* special case if two objects are selected with snap points */
	if (count == 0)
	{
		var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_highlighted);
		if (var != NOVARIABLE)
		{
			len = getlength(var);
			if (len == 2)
			{
				if (us_makehighlight(((char **)var->addr)[0], &thishigh) == 0 &&
					us_makehighlight(((char **)var->addr)[1], &otherhigh) == 0)
				{
					/* describe these two objects */
					if ((thishigh.status&HIGHSNAP) != 0 && (otherhigh.status&HIGHSNAP) != 0)
					{
						/* move first to second */
						us_getsnappoint(&thishigh, &p1x, &p1y);
						us_getsnappoint(&otherhigh, &p2x, &p2y);
						dx = p2x - p1x;
						dy = p2y - p1y;
						if (dx == 0 && dy == 0)
						{
							us_abortcommand("Points are already aligned");
							return;
						}
						ni = thishigh.fromgeom->entryaddr.ni;
						us_pushhighlight();
						us_clearhighlightcount();
						startobjectchange((INTBIG)ni, VNODEINST);
						modifynodeinst(ni, dx, dy, dx, dy, 0, 0);
						endobjectchange((INTBIG)ni, VNODEINST);
						(void)us_pophighlight(1);
						return;
					}
				}
			}
		}
	}

	/* save and turn off all highlighting */
	us_pushhighlight();
	us_clearhighlightcount();

	/* no arguments: simple motion */
	if (count == 0)
	{
		if ((us_aid->aidstate&INTERACTIVE) != 0 &&
			us_graphicshas(CANTRACKCURSOR) != 0)
		{
			/* interactive motion: track cursor */
			us_multidraginit(bestlx, bestly, list, nodelist, total, 0, 0);
			trackcursor(0, us_ignoreup, us_multidragbegin, us_multidragdown,
				us_stoponchar, us_multidragup, TRACKDRAGGING);
			if (el_pleasestop != 0)
			{
				efree((char *)nodelist);
				(void)us_pophighlight(0);
				return;
			}
		}

		/* get co-ordinates of cursor */
		if (us_demandxy(&xcur, &ycur))
		{
			efree((char *)nodelist);
			(void)us_pophighlight(0);
			return;
		}
		gridalign(&xcur, &ycur, us_alignment);

		/* make the move if it is valid */
		if (xcur == bestlx && ycur == bestly) ttyputmsgf("Null motion"); else
			us_manymove(list, nodelist, total, xcur-bestlx, ycur-bestly);
		efree((char *)nodelist);
		(void)us_pophighlight(1);
		return;
	}

	l = strlen(pp = par[0]);

	/* handle absolute motion option "move to X Y" */
	if (namesamen(pp, "to", l) == 0 && l >= 1)
	{
		/* get absolute position to place object */
		if (count == 2 && namesamen(par[1], "dialog", (INTSML)strlen(par[1])) == 0)
		{
			/* get coordinates from dialog */
			if (DiaInitDialog(&usr_movetodialog) != 0) return;
			DiaSetText(3, latoa(bestlx));
			DiaSetText(5, latoa(bestly));
			for(;;)
			{
				itemHit = DiaNextHit();
				if (itemHit == CANCEL || itemHit == OK) break;
			}
			xcur = atola(DiaGetText(3));
			ycur = atola(DiaGetText(5));
			DiaDoneDialog();
			if (itemHit == CANCEL) return;
		} else
		{
			/* get coordinates from command line */
			if (count < 3)
			{
				us_abortcommand("Usage: move to X Y");
				efree((char *)nodelist);
				(void)us_pophighlight(0);
				return;
			}
			xcur = atola(par[1]);
			ycur = atola(par[2]);
		}

		/* make the move if it is valid */
		if (xcur == bestlx && ycur == bestly) ttyputmsgf("Null motion"); else
			us_manymove(list, nodelist, total, xcur-bestlx, ycur-bestly);
		efree((char *)nodelist);
		(void)us_pophighlight(1);
		return;
	}

	/* handle motion to cursor along an angle: "move angle ANG" */
	if (namesamen(pp, "angle", l) == 0 && l >= 1)
	{
		/* get co-ordinates of cursor */
		if (us_demandxy(&xcur, &ycur))
		{
			efree((char *)nodelist);
			(void)us_pophighlight(0);
			return;
		}
		gridalign(&xcur, &ycur, us_alignment);

		/* get angle to align along */
		if (count < 2)
		{
			us_abortcommand("Usage: move angle ANGLE");
			efree((char *)nodelist);
			(void)us_pophighlight(0);
			return;
		}
		amt = atofr(par[1])*10/WHOLE;

		/* adjust the cursor position if selecting interactively */
		if ((us_aid->aidstate&INTERACTIVE) != 0 && us_graphicshas(CANTRACKCURSOR) != 0)
		{
			us_multidraginit(bestlx, bestly, list, nodelist, total, (INTSML)amt, 0);
			trackcursor(0, us_ignoreup, us_multidragbegin, us_multidragdown,
				us_stoponchar, us_multidragup, TRACKDRAGGING);
			if (el_pleasestop != 0)
			{
				efree((char *)nodelist);
				(void)us_pophighlight(0);
				return;
			}
			if (us_demandxy(&xcur, &ycur) != 0) return;
			gridalign(&xcur, &ycur, us_alignment);
		}

		/* compute sliding for this highlighting */
		us_getslide((INTSML)amt, bestlx, bestly, xcur, ycur, &dx, &dy);

		/* make the move if it is nonnull */
		if (dx == bestlx && dy == bestly) ttyputmsgf("Null motion"); else
			us_manymove(list, nodelist, total, dx-bestlx, dy-bestly);
		efree((char *)nodelist);
		(void)us_pophighlight(1);
		return;
	}

	/* if direction is specified, get it */
	dx = dy = 0;
	if (namesamen(pp, "up", l) == 0 && l >= 1)
	{
		if (count < 2) amt = el_curtech->deflambda; else amt = atola(par[1]);
		dy += amt;
	} else if (namesamen(pp, "down", l) == 0 && l >= 1)
	{
		if (count < 2) amt = el_curtech->deflambda; else amt = atola(par[1]);
		dy -= amt;
	} else if (namesamen(pp, "left", l) == 0 && l >= 1)
	{
		if (count < 2) amt = el_curtech->deflambda; else amt = atola(par[1]);
		dx -= amt;
	} else if (namesamen(pp, "right", l) == 0 && l >= 1)
	{
		if (count < 2) amt = el_curtech->deflambda; else amt = atola(par[1]);
		dx += amt;
	} else
	{
		us_abortcommand("Invalid MOVE option: %s", par[0]);
		efree((char *)nodelist);
		(void)us_pophighlight(0);
		return;
	}

	/* now move the object */
	if (dx != 0 || dy != 0) us_manymove(list, nodelist, total, dx, dy);
	efree((char *)nodelist);
	(void)us_pophighlight(1);
}

void us_node(INTSML count, char *par[])
{
	REGISTER INTBIG i, amt, *newlist, x, y, curlamb, inside, outside;
	REGISTER INTSML l, negated, size, nogood, total, waitfordown, addpoint,
		deletepoint, whichpoint, p, movepoint, pointgiven, segs, degs;
	XARRAY trans;
	INTBIG d[2], xcur, ycur;
	REGISTER char *pt, *ch;
	REGISTER GEOM **list;
	extern COMCOMP us_nodep;
	REGISTER NODEINST *ni, *store;
	REGISTER HIGHLIGHT *high;
	HIGHLIGHT newhigh;
	REGISTER VARIABLE *var;

	if (count == 0)
	{
		count = ttygetparam("Node option: ", &us_nodep, MAXPARS, par);
		if (count == 0)
		{
			us_abortedmsg();
			return;
		}
	}
	l = strlen(pt = par[0]);

	/* handle negation */
	if (namesamen(pt, "not", l) == 0 && l >= 2)
	{
		negated = 1;
		count--;
		par++;
		l = strlen(pt = par[0]);
	} else negated = 0;

	if (namesamen(pt, "expand", l) == 0 && l >= 1)
	{
		if (count < 2) amt = HUGEINT; else amt = atoi(par[1]);
		list = us_gethighlighted(OBJNODEINST);
		if (list[0] == NOGEOM)
		{
			us_abortcommand("Select nodes to be expanded");
			return;
		}

		/* save highlighting */
		us_pushhighlight();
		us_clearhighlightcount();

		/* pre-compute when un-expanding */
		if (negated) for(i=0; list[i] != NOGEOM; i++)
		{
			ni = list[i]->entryaddr.ni;
			if ((ni->userbits & NEXPAND) != 0) (void)us_setunexpand(ni, (INTSML)amt);
		}

		/* do the change */
		for(i=0; list[i] != NOGEOM; i++)
		{
			ni = list[i]->entryaddr.ni;
			if (negated == 0) us_doexpand(ni, (INTSML)amt, 0); else
				us_dounexpand(ni);
		}

		/* restore highlighting */
		(void)us_pophighlight(0);
		return;
	}

	if (namesamen(pt, "name", l) == 0 && l >= 2)
	{
		ni = (NODEINST *)us_getobject(OBJNODEINST, 0);
		if (ni == NONODEINST) return;

		if (negated == 0)
		{
			if (count < 2)
			{
				us_abortcommand("Usage: node name NODENAME");
				return;
			}
			ch = par[1];

			/* sensibility check for multiple nodes with the same name */
			for(store = ni->parent->firstnodeinst; store != NONODEINST; store = store->nextnodeinst)
			{
				if (store == ni) continue;
				var = getvalkey((INTBIG)store, VNODEINST, VSTRING, el_node_name);
				if (var == NOVARIABLE) continue;
				if (namesame(ch, (char *)var->addr) == 0)
				{
					ttyputmsg("Warning: already a node in this facet called %s", ch);
					break;
				}
			}
		}

		/* save highlighting */
		us_pushhighlight();
		us_clearhighlightcount();

		/* change the name of the nodeinst */
		startobjectchange((INTBIG)ni, VNODEINST);
		if (negated != 0) (void)delvalkey((INTBIG)ni, VNODEINST, el_node_name); else
		{
			var = setvalkey((INTBIG)ni, VNODEINST, el_node_name, (INTBIG)ch, VSTRING|VDISPLAY);

			/* shift text down if on a facet instance */
			if (var != NOVARIABLE && ni->proto->index == 0)
			{
				var->textdescript = us_setdescriptoffset(var->textdescript,
					0, (ni->highy-ni->lowy) / el_curtech->deflambda);
			}
		}
		endobjectchange((INTBIG)ni, VNODEINST);

		/* restore highlighting */
		(void)us_pophighlight(0);

		/* report results and quit */
		if (negated != 0) ttyputmsgf("Node name removed"); else
			ttyputmsgf("Node name is '%s'", ch);
		return;
	}

	if (namesamen(pt, "trace", l) == 0 && l >= 1)
	{
		/* special case for filleting */
		if (count >= 2 && namesamen(par[1], "fillet", (INTSML)(strlen(par[1]))) == 0)
		{
			us_dofillet();
			return;
		}

		/* special case for annulus construction */
		if (count >= 2 && namesamen(par[1], "construct-annulus", (INTSML)(strlen(par[1]))) == 0)
		{
			if (count < 4)
			{
				us_abortcommand("Usage: node trace construct-annulus INSIDE OUTSIDE [RESOLUTION] [DEGREES]");
				return;
			}
			inside = atola(par[2]);
			outside = atola(par[3]);
			segs = 16;
			if (count >= 5) segs = myatoi(par[4]);
			if (segs < 4) segs = 4;
			degs = 3600;
			if (count >= 6) degs = atofr(par[5])*10/WHOLE;
			if (degs <= 0) degs = 3600;
			if (degs > 3600) degs = 3600;

			/* make sure node can handle trace information */
			store = (NODEINST *)us_getobject(OBJNODEINST, 0);
			if (store == NONODEINST) return;
			if ((store->proto->userbits&HOLDSTRACE) == 0)
			{
				us_abortcommand("Sorry, %s nodes cannot hold trace information",
					describenodeproto(store->proto));
				return;
			}

			/* allocate space for the trace */
			newlist = emalloc(((segs+1)*4*SIZEOFINTBIG), el_tempcluster);
			if (newlist == 0)
			{
				us_abortcommand("No memory!");
				return;
			}

			l = 0;
			for(i=0; i<=segs; i++)
			{
				p = degs*i/segs;
				x = mult(inside, cosine(p)) + (store->lowx+store->highx)/2;
				y = mult(inside, sine(p)) + (store->lowy+store->highy)/2;
				newlist[l++] = x;   newlist[l++] = y;
			}
			for(i=segs; i>=0; i--)
			{
				p = degs*i/segs;
				x = mult(outside, cosine(p)) + (store->lowx+store->highx)/2;
				y = mult(outside, sine(p)) + (store->lowy+store->highy)/2;
				newlist[l++] = x;   newlist[l++] = y;
			}

			/* save highlighting, set trace, restore highlighting */
			us_pushhighlight();
			us_clearhighlightcount();
			us_settrace(store, newlist, (INTSML)((segs+1)*2));
			(void)us_pophighlight(0);
			efree((char *)newlist);
			return;
		}

		/* parse the options */
		store = NONODEINST;
		waitfordown = addpoint = deletepoint = movepoint = pointgiven = 0;
		for(i=1; i<count; i++)
		{
			l = strlen(pt = par[i]);

			if (namesamen(pt, "store-trace", l) == 0 && l >= 1)
			{
				store = (NODEINST *)us_getobject(OBJNODEINST, 0);
				if (store == NONODEINST) return;
			} else if (namesamen(pt, "add-point", l) == 0 && l >= 1)
			{
				addpoint++;
				store = (NODEINST *)us_getobject(OBJNODEINST, 0);
				if (store == NONODEINST) return;
				high = us_getonehighlight();
				whichpoint = high->frompoint;
				if (count > i+2 && isanumber(par[i+1]) != 0 && isanumber(par[i+2]) != 0)
				{
					xcur = atola(par[i+1]);
					ycur = atola(par[i+2]);
					i += 2;
					pointgiven++;
				}
			} else if (namesamen(pt, "move-point", l) == 0 && l >= 1)
			{
				movepoint++;
				store = (NODEINST *)us_getobject(OBJNODEINST, 0);
				if (store == NONODEINST) return;
				high = us_getonehighlight();
				whichpoint = high->frompoint;
				if (count > i+2 && isanumber(par[i+1]) != 0 && isanumber(par[i+2]) != 0)
				{
					xcur = atola(par[i+1]);
					ycur = atola(par[i+2]);
					i += 2;
					pointgiven++;
				}
			} else if (namesamen(pt, "delete-point", l) == 0 && l >= 1)
			{
				deletepoint++;
				store = (NODEINST *)us_getobject(OBJNODEINST, 0);
				if (store == NONODEINST) return;
				high = us_getonehighlight();
				whichpoint = high->frompoint;
			} else if (namesamen(pt, "wait-for-down", l) == 0 && l >= 1)
			{
				waitfordown++;
			} else
			{
				us_abortcommand("Invalid NODE TRACE option: %s", pt);
				return;
			}
		}

		if (addpoint + deletepoint + movepoint > 1)
		{
			us_abortcommand("Can only add OR delete OR move");
			return;
		}

		/* make sure node can handle trace information */
		if (store != NONODEINST)
		{
			if ((store->proto->userbits&HOLDSTRACE) == 0)
			{
				us_abortcommand("Sorry, %s nodes cannot hold trace information", describenodeproto(store->proto));
				return;
			}
		}

		/* handle freeform cursor traces */
		if (addpoint == 0 && deletepoint == 0 && movepoint == 0)
		{
			/* adjust the cursor position if selecting interactively */
			if (us_graphicshas(CANTRACKCURSOR) == 0)
			{
				us_abortcommand("Sorry, this display cannot do cursor tracking");
				return;
			}

			/* just do tracking if no storage requested */
			if (store == NONODEINST)
			{
				/* save highlighting */
				us_pushhighlight();
				us_clearhighlightcount();

				trackcursor(waitfordown, us_ignoreup, us_tracebegin,
					us_tracedown, us_stoponchar, us_traceup, TRACKDRAWING);

				/* restore highlighting */
				(void)us_pophighlight(0);
				return;
			}

			/* clear highlighting */
			us_clearhighlightcount();

			/* get the trace */
			trackcursor(waitfordown, us_ignoreup, us_tracebegin, us_tracedown,
				us_stoponchar, us_traceup, TRACKDRAWING);
			if (el_pleasestop != 0) return;

			/* get the trace data in the "%T" variable */
			var = getval((INTBIG)us_aid, VAID, VINTEGER|VISARRAY, us_commandvarname('T'));
			if (var == NOVARIABLE)
			{
				us_abortcommand("Invalid trace information");
				return;
			}
			size = getlength(var) / 2;

			/* create a place to keep this data */
			newlist = emalloc((size*2*SIZEOFINTBIG), el_tempcluster);
			if (newlist == 0)
			{
				us_abortcommand("No memory!");
				return;
			}

			/* copy the trace from "%T" to "newlist" */
			curlamb = el_curtech->deflambda;
			el_curtech->deflambda = 4;
			nogood = 0;
			for(i=0; i<size; i++)
			{
				x = ((INTBIG *)var->addr)[i*2];
				y = ((INTBIG *)var->addr)[i*2+1];
				nogood += us_setxy((INTSML)x, (INTSML)y);
				(void)getxy(&xcur, &ycur);
				newlist[i*2] = xcur;   newlist[i*2+1] = ycur;
			}
			el_curtech->deflambda = curlamb;

			/* if data is valid, store it in the node */
			if (nogood != 0) ttyputerr("Trace not inside window"); else
			{
				us_settrace(store, newlist, size);

				/* highlighting the node */
				newhigh.status = HIGHFROM;
				newhigh.fromgeom = store->geom;
				newhigh.fromport = NOPORTPROTO;
				newhigh.frompoint = size+1;
				newhigh.facet = store->parent;
				(void)us_addhighlight(&newhigh);
			}
			efree((char *)newlist);
			return;
		}

		/* add a point to a trace on this object */
		if (addpoint != 0)
		{
			/* clear highlighting */
			us_clearhighlightcount();

			if (pointgiven == 0)
			{
				if (us_demandxy(&xcur, &ycur)) return;
				gridalign(&xcur, &ycur, us_alignment);

				/* adjust the cursor position if selecting interactively */
				if ((us_aid->aidstate&INTERACTIVE) != 0 && us_graphicshas(CANTRACKCURSOR) != 0)
				{
					us_pointinit(store, whichpoint);
					trackcursor(0, us_ignoreup, us_pointbegin, us_addpdown,
						us_stoponchar, us_dragup, TRACKDRAGGING);
					if (el_pleasestop != 0) return;
					if (us_demandxy(&xcur, &ycur) != 0) return;
					gridalign(&xcur, &ycur, us_alignment);
				}
			}

			var = gettrace(store);
			if (var == NOVARIABLE)
			{
				d[0] = xcur;   d[1] = ycur;
				us_settrace(store, d, 1);
				whichpoint = 1;
			} else
			{
				size = getlength(var) / 2;
				newlist = emalloc(((size+1)*2*SIZEOFINTBIG), el_tempcluster);
				if (newlist == 0)
				{
					us_abortcommand("No memory!");
					return;
				}
				p = 0;
				makerot(store, trans);
				x = (store->highx + store->lowx) / 2;
				y = (store->highy + store->lowy) / 2;
				for(i=0; i<size; i++)
				{
					xform(((INTBIG *)var->addr)[i*2]+x, ((INTBIG *)var->addr)[i*2+1]+y, &newlist[p],
						&newlist[p+1], trans);
					p += 2;
					if (i+1 == whichpoint)
					{
						newlist[p++] = xcur;
						newlist[p++] = ycur;
					}
				}

				/* now re-draw this trace */
				us_settrace(store, newlist, (INTSML)(size+1));
				whichpoint++;
				efree((char *)newlist);
			}

			/* highlighting the node */
			newhigh.status = HIGHFROM;
			newhigh.fromgeom = store->geom;
			newhigh.fromport = NOPORTPROTO;
			newhigh.frompoint = whichpoint;
			newhigh.facet = store->parent;
			(void)us_addhighlight(&newhigh);
			return;
		}

		/* delete the current point */
		if (deletepoint != 0)
		{
			var = gettrace(store);
			if (var == NOVARIABLE)
			{
				us_abortcommand("No trace data to delete");
				return;
			}
			if (whichpoint == 0)
			{
				us_abortcommand("Highlight a point or line to delete");
				return;
			}
			size = getlength(var) / 2;
			if (size <= 1)
			{
				us_abortcommand("Trace must retain at least one point");
				return;
			}
			newlist = emalloc(((size-1)*2*SIZEOFINTBIG), el_tempcluster);
			if (newlist == 0)
			{
				us_abortcommand("No memory!");
				return;
			}
			p = 0;
			makerot(store, trans);
			x = (store->highx + store->lowx) / 2;
			y = (store->highy + store->lowy) / 2;
			total = 0;
			for(i=0; i<size; i++)
			{
				if (i+1 == whichpoint) { total++;   continue;}
				xform(((INTBIG *)var->addr)[i*2]+x, ((INTBIG *)var->addr)[i*2+1]+y,
					&newlist[p], &newlist[p+1], trans);
				p += 2;
			}

			/* clear highlighting */
			us_clearhighlightcount();

			/* now re-draw this trace */
			us_settrace(store, newlist, (INTSML)(size-total));
			whichpoint = maxi(whichpoint-total, 1);
			efree((char *)newlist);

			/* highlighting the node */
			newhigh.status = HIGHFROM;
			newhigh.fromgeom = store->geom;
			newhigh.fromport = NOPORTPROTO;
			newhigh.frompoint = whichpoint;
			newhigh.facet = store->parent;
			(void)us_addhighlight(&newhigh);
			return;
		}

		/* move a point on a trace on this object */
		if (movepoint != 0)
		{
			var = gettrace(store);
			if (var == NOVARIABLE)
			{
				us_abortcommand("No trace data to move");
				return;
			}
			if (whichpoint == 0)
			{
				us_abortcommand("Highlight a point or line to move");
				return;
			}

			/* save highlighting */
			us_pushhighlight();
			us_clearhighlightcount();

			if (pointgiven == 0)
			{
				if (us_demandxy(&xcur, &ycur)) return;
				gridalign(&xcur, &ycur, us_alignment);

				/* adjust the cursor position if selecting interactively */
				if ((us_aid->aidstate&INTERACTIVE) != 0 && us_graphicshas(CANTRACKCURSOR) != 0)
				{
					us_pointinit(store, whichpoint);
					trackcursor(0, us_ignoreup, us_pointbegin, us_movepdown,
						us_stopandpoponchar, us_dragup, TRACKDRAGGING);
					if (el_pleasestop != 0) return;
					if (us_demandxy(&xcur, &ycur) != 0) return;
					gridalign(&xcur, &ycur, us_alignment);
				}
			}

			size = getlength(var) / 2;
			newlist = emalloc((size*2*SIZEOFINTBIG), el_tempcluster);
			if (newlist == 0)
			{
				us_abortcommand("No memory!");
				(void)us_pophighlight(0);
				return;
			}
			makerot(store, trans);
			x = (store->highx + store->lowx) / 2;
			y = (store->highy + store->lowy) / 2;
			for(i=0; i<size; i++)
			{
				if (i+1 == whichpoint)
				{
					newlist[i*2] = xcur;
					newlist[i*2+1] = ycur;
				} else xform(((INTBIG *)var->addr)[i*2]+x, ((INTBIG *)var->addr)[i*2+1]+y,
					&newlist[i*2], &newlist[i*2+1], trans);
			}

			/* now re-draw this trace */
			us_settrace(store, newlist, size);
			efree((char *)newlist);

			/* restore highlighting */
			(void)us_pophighlight(1);
			return;
		}
		return;
	}

	us_abortcommand("Bad NODE option: %s", pt);
}
