/*
 * Electric(tm) VLSI Design System
 *
 * File: usrcomoq.c
 * User interface aid: command handler for O through Q
 * Written by: Steven M. Rubin, Electric Editor Incorporated
 *
 * Copyright (c) 1998 Electric Editor Incorporated.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Electric Editor Incorporated
 * 23470 Sunset Drive, Suite 108
 * Los Gatos, California 95033
 * support@electriceditor.com
 */

#include "global.h"
#include "usr.h"

void us_offaid(INTSML count, char *par[])
{
	REGISTER INTSML permanently;

	if (count == 2 && namesamen(par[1], "permanently", (INTSML)(strlen(par[1]))) == 0)
	{
		count--;
		permanently = 1;
	} else permanently = 0;
	us_setaid(count, par, permanently);
}

void us_onaid(INTSML count, char *par[])
{
	REGISTER INTSML catchup;

	if (count == 2 && namesamen(par[1], "no-catch-up", (INTSML)(strlen(par[1]))) == 0)
	{
		count--;
		catchup = 3;
	} else catchup = 2;
	us_setaid(count, par, catchup);
}

void us_outhier(INTSML count, char *par[])
{
	REGISTER INTSML total, i, len, j;
	INTBIG lx, hx, ly, hy;
	REGISTER char **list, *pt, *opt;
	REGISTER NODEPROTO *parnt, *curnp, *upnp, *np;
	REGISTER VARIABLE *var;
	REGISTER NODEINST *ni;
	NODEINST *hini;
	PORTPROTO *hipp;

	/* make sure repeat count is valid */
	if (count == 0) total = 1; else
	{
		total = atoi(par[0]);
		if (total <= 0)
		{
			us_abortcommand("Specify a positive number");
			return;
		}
	}

	/* must be editing a facet */
	curnp = us_needfacet();
	if (curnp == NONODEPROTO) return;
	upnp = NONODEPROTO;
	hini = NONODEINST;   hipp = NOPORTPROTO;

	/* initialize desired window extent */
	lx = el_curwindow->screenlx;   hx = el_curwindow->screenhx;
	ly = el_curwindow->screenly;   hy = el_curwindow->screenhy;

	/* if one level of highlighting and exported port selected, follow hierarchy */
	if (total == 1)
	{
		upnp = us_followexportedportup(&hini, &hipp);
		if (upnp == 0) return;
		if (upnp != NONODEPROTO)
		{
			lx = upnp->lowx;   hx = upnp->highx;
			ly = upnp->lowy;   hy = upnp->highy;
		}
	}

	us_clearhighlightcount();
	for(i=0; i<total; i++)
	{
		if (i != 0) hini = NONODEINST;

		/* if there is a stack, use it */
		var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_facet_stack);
		if (var != NOVARIABLE)
		{
			/* parse last entry on stack */
			len = getlength(var);
			opt = pt = ((char **)var->addr)[len-1];
			while (*pt != 0 && *pt != ' ') pt++;
			if (*pt != ' ') ttyputerr("Cannot understand facet stack"); else
			{
				*pt = 0;
				np = getnodeproto(opt);
				*pt++ = ' ';
				if (upnp == NONODEPROTO || np == upnp)
				{
					/* use what is on the stack */
					upnp = np;
					hini = (NODEINST *)myatoi(pt);

					/* shorten the stack */
					if (len == 1) (void)delvalkey((INTBIG)us_aid, VAID, us_facet_stack); else
					{
						len--;
						list = (char **)emalloc((len * (sizeof (char *))), el_tempcluster);
						if (list == 0) return;
						for(j=0; j<len; j++)
							(void)allocstring(&list[j], ((char **)var->addr)[j], el_tempcluster);
						(void)setvalkey((INTBIG)us_aid, VAID, us_facet_stack, (INTBIG)list,
							VSTRING|VISARRAY|(len<<VLENGTHSH)|VDONTSAVE);
						for(j=0; j<len; j++) efree(list[j]);
						efree((char *)list);
					}
				}
			}
		}

		/* if nothing on stack and no exported port path, see if upper facet is obvious */
		if (var == NOVARIABLE && upnp == NONODEPROTO)
		{
			/* see if facet "curnp" has an icon facet */
			np = iconview(curnp);
			if (np != NONODEPROTO) curnp = np;

			ni = curnp->firstinst;
			if (ni != NONODEINST) parnt = ni->parent; else parnt = NONODEPROTO;
			for(; ni != NONODEINST; ni = ni->nextinst)
				if (ni->parent != parnt) break;
			if (ni != NONODEINST) parnt = NONODEPROTO;
			if (parnt != NONODEPROTO)
			{
				upnp = parnt;
				hini = NONODEINST;
			}
		}

		/* quit if nothing was found */
		if (upnp == NONODEPROTO)
		{
			if (i == 0)
			{
				us_abortcommand("Not in any facets");
				return;
			}
			ttyputerr("Only in %d deep", i);
			break;
		}

		/* adjust view of the facet */
		if (hini != NONODEINST && hipp == NOPORTPROTO)
		{
			lx += hini->geom->lowx - curnp->lowx;
			hx += hini->geom->lowx - curnp->lowx;
			ly += hini->geom->lowy - curnp->lowy;
			hy += hini->geom->lowy - curnp->lowy;
		} else us_fullview(upnp, &lx, &hx, &ly, &hy);

		curnp = upnp;
		upnp = NONODEPROTO;
	}

	us_switchtofacet(curnp, lx, hx, ly, hy, hini, hipp, 0);
}

void us_package(INTSML count, char *par[])
{
	INTBIG lx, hx, ly, hy;
	REGISTER INTBIG search;
	REGISTER GEOM *look;
	REGISTER NODEPROTO *np, *parnt;
	REGISTER PORTPROTO *ppt, *npt;
	REGISTER NODEINST *ni, *newni;
	REGISTER ARCINST *ai, *newar;
	REGISTER LIBRARY *lib;
	REGISTER char *pt;
	extern COMCOMP us_packagep;

	/* get the specified area */
	if (us_getareabounds(&lx, &hx, &ly, &hy) != 0)
	{
		us_abortcommand("Enclose an area to be packaged");
		return;
	}

	if (count == 0)
	{
		count = ttygetparam("New facet name: ", &us_packagep, MAXPARS, par);
		if (count == 0)
		{
			us_abortedmsg();
			return;
		}
	}

	/* figure out which library to use */
	lib = el_curlib;
	for(pt = par[0]; *pt != 0; pt++) if (*pt == ':') break;
	if (*pt == ':')
	{
		*pt++ = 0;
		lib = getlibrary(par[0]);
		if (lib == NOLIBRARY)
		{
			us_abortcommand("Cannot find library '%s'", par[0]);
			return;
		}
		par[0] = pt;
	}

	/* create the new facet */
	np = newnodeproto(par[0], lib);
	if (np == NONODEPROTO)
	{
		us_abortcommand("Cannot create facet %s", par[0]);
		return;
	}

	/* first zap all nodes touching all arcs in the region */
	parnt = getcurfacet();
	search = initsearch(lx, hx, ly, hy, parnt);
	for(;;)
	{
		look = nextobject(search);
		if (look == NOGEOM) break;
		if (look->entrytype != OBJARCINST) continue;
		ai = look->entryaddr.ai;
		ai->end[0].nodeinst->temp1 = (unsigned)-1;
		ai->end[1].nodeinst->temp1 = (unsigned)-1;
	}

	/* copy the nodes into the new facet */
	search = initsearch(lx, hx, ly, hy, parnt);
	for(;;)
	{
		look = nextobject(search);
		if (look == NOGEOM) break;
		if (look->entrytype != OBJNODEINST) continue;
		ni = look->entryaddr.ni;
		newni = newnodeinst(ni->proto, ni->lowx, ni->highx, ni->lowy,
			ni->highy, ni->transpose, ni->rotation, np);
		if (newni == NONODEINST)
		{
			us_abortcommand("Cannot create node in new facet");
			return;
		}
		ni->temp1 = (INTBIG)newni;
		newni->userbits = ni->userbits;
		newni->textdescript = ni->textdescript;
		if (copyvars((INTBIG)ni, VNODEINST, (INTBIG)newni, VNODEINST) != 0)
		{
			us_abortcommand("Memory low");
			return;
		}

		/* make ports where this nodeinst has them */
		if (ni->firstportexpinst != NOPORTEXPINST)
			for(ppt = parnt->firstportproto; ppt != NOPORTPROTO; ppt = ppt->nextportproto)
		{
			if (ppt->subnodeinst != ni) continue;
			npt = newportproto(np, newni, ppt->subportproto, ppt->protoname);
			npt->userbits = (npt->userbits & ~STATEBITS) | (ppt->userbits & STATEBITS);
			npt->textdescript = ppt->textdescript;
			if (copyvars((INTBIG)ppt, VPORTPROTO, (INTBIG)npt, VPORTPROTO) != 0)
			{
				us_abortcommand("Memory low");
				return;
			}
		}
	}

	/* copy the arcs into the new facet */
	search = initsearch(lx, hx, ly, hy, parnt);
	for(;;)
	{
		look = nextobject(search);
		if (look == NOGEOM) break;
		if (look->entrytype != OBJARCINST) continue;
		ai = look->entryaddr.ai;
		if (ai->end[0].nodeinst->temp1 == (unsigned)-1 ||
			ai->end[1].nodeinst->temp1 == (unsigned)-1) continue;
		newar = newarcinst(ai->proto, ai->width, ai->userbits, (NODEINST *)ai->end[0].nodeinst->temp1,
			ai->end[0].portarcinst->proto, ai->end[0].xpos,ai->end[0].ypos,
				(NODEINST *)ai->end[1].nodeinst->temp1, ai->end[1].portarcinst->proto,
					ai->end[1].xpos,ai->end[1].ypos, np);
		if (newar == NOARCINST)
		{
			us_abortcommand("Cannot create arc in new facet");
			return;
		}
		if (copyvars((INTBIG)ai, VARCINST, (INTBIG)newar, VARCINST) != 0 ||
			copyvars((INTBIG)ai->end[0].portarcinst, VPORTARCINST,
				(INTBIG)newar->end[0].portarcinst, VPORTARCINST) != 0 ||
			copyvars((INTBIG)ai->end[1].portarcinst, VPORTARCINST,
				(INTBIG)newar->end[1].portarcinst, VPORTARCINST) != 0)
		{
			us_abortcommand("Memory low");
			return;
		}
	}

	/* recompute bounds of new facet */
	(*el_curconstraint->solve)(np);

	ttyputmsg("Facet %s created", describenodeproto(np));
}

void us_port(INTSML count, char *par[])
{
	INTBIG lx, hx, ly, hy, mlx, mly, mhx, mhy, bits, mask;
	REGISTER NODEINST *ni, *oni;
	REGISTER NODEPROTO *np;
	REGISTER PORTPROTO *pp, *ppt;
	REGISTER PORTARCINST *pi;
	REGISTER PORTEXPINST *pe;
	GEOM *fromgeom, *togeom;
	PORTPROTO *fromport, *toport;
	REGISTER char *pt;
	REGISTER INTSML i, l, labels, total, reducehighlighting;
	char *portname;
	REGISTER WINDOW *w;
	REGISTER HIGHLIGHT *high;
	HIGHLIGHT newhigh;
	extern COMCOMP us_portp, us_portep;
	REGISTER GEOM **g;
	static POLYGON *poly = NOPOLYGON;

	/* get options */
	if (count == 0)
	{
		count = ttygetparam("Port option: ", &us_portp, MAXPARS, par);
		if (count == 0)
		{
			us_abortedmsg();
			return;
		}
	}
	l = strlen(pt = par[0]);

	if (namesamen(pt, "labels", l) == 0 && l >= 1)
	{
		if (us_needwindow()) return;
		if (count > 1)
		{
			l = strlen(pt = par[1]);
			if (namesamen(pt, "none", l) == 0 && l >= 2) labels = PORTSOFF; else
			if (namesamen(pt, "crosses", l) == 0 && l >= 1) labels = PORTSCROSS; else
			if (namesamen(pt, "short", l) == 0 && l >= 2) labels = PORTSSHORT; else
			if (namesamen(pt, "long", l) == 0 && l >= 1) labels = PORTSFULL; else
			{
				us_abortcommand("Usage: port labels none|short|long|crosses");
				return;
			}
			for(w = el_topwindow; w != NOWINDOW; w = w->nextwindow)
				w->state = (w->state & ~PORTLABELS) | labels;
		}
		switch (el_curwindow->state&PORTLABELS)
		{
			case PORTSOFF:   ttyputmsgf("No ports are indicated");     break;
			case PORTSCROSS: ttyputmsgf("Ports shown as crosses");     break;
			case PORTSFULL:  ttyputmsgf("Long port names displayed");  break;
			case PORTSSHORT: ttyputmsgf("Short port names displayed"); break;
		}
		return;
	}

	if (namesamen(pt, "move", l) == 0 && l >= 1)
	{
		if (us_gettwoobjects(&fromgeom, &fromport, &togeom, &toport) != 0)
		{
			us_abortcommand("Must select two nodes");
			return;
		}
		if (fromport == NOPORTPROTO || toport == NOPORTPROTO)
		{
			us_abortcommand("Must select two nodes AND their ports");
			return;
		}

		/* make sure ports are in the same facet */
		if (geomparent(fromgeom) != geomparent(togeom))
		{
			us_abortcommand("Port motion must be within a single facet");
			return;
		}

		reducehighlighting = 1;
		if (count > 1)
		{
			l = strlen(pt = par[1]);
			if (namesamen(pt, "remain-highlighted", l) == 0) reducehighlighting = 0;
		}

		/* get the "source" node and exported port */
		oni = fromgeom->entryaddr.ni;
		for(pe = oni->firstportexpinst; pe != NOPORTEXPINST; pe = pe->nextportexpinst)
			if (pe->proto == fromport) break;
		if (pe == NOPORTEXPINST)
		{
			us_abortcommand("Port %s of node %s is not exported", fromport->protoname,
				describenodeinst(oni));
			return;
		}

		/* move the port */
		ni = togeom->entryaddr.ni;
		if (ni == oni && fromport == toport)
		{
			us_abortcommand("This does not move the port");
			return;
		}

		/* clear highlighting */
		if (reducehighlighting == 0) us_pushhighlight();
		us_clearhighlightcount();

		np = ni->parent;
		pt = pe->exportproto->protoname;
		startobjectchange((INTBIG)ni, VNODEINST);
		startobjectchange((INTBIG)oni, VNODEINST);
		if (moveportproto(np, pe->exportproto, ni, toport) != 0) ttyputerr("Port not moved"); else
		{
			endobjectchange((INTBIG)ni, VNODEINST);
			endobjectchange((INTBIG)oni, VNODEINST);
			ttyputmsgf("Port %s moved from node %s to node %s", pt, describenodeinst(oni),
				describenodeinst(ni));
		}

		/* restore highlighting */
		if (reducehighlighting == 0) (void)us_pophighlight(0); else
		{
			newhigh.status = HIGHFROM;
			newhigh.facet = np;
			newhigh.fromgeom = fromgeom;
			newhigh.fromport = fromport;
			(void)us_addhighlight(&newhigh);
		}
		return;
	}

	if (namesamen(pt, "re-export-all", l) == 0 && l >= 1)
	{
		/* make sure there is a current facet */
		np = us_needfacet();
		if (np == NONODEPROTO) return;

		/* save highlighting */
		us_pushhighlight();
		us_clearhighlightcount();

		/* look at every node in this facet */
		total = 0;
		for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
		{
			/* only look for facets, not primitives */
			if (ni->proto->index != 0) continue;

			/* clear marks on the ports of this node */
			for(pp = ni->proto->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
				pp->temp1 = 0;

			/* mark the connected and exported ports */
			for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
				pi->proto->temp1++;
			for(pe = ni->firstportexpinst; pe != NOPORTEXPINST; pe = pe->nextportexpinst)
				pe->proto->temp1++;

			/* now export the remaining ports */
			for(pp = ni->proto->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
				if (pp->temp1 == 0)
			{
				if (us_reexportport(pp, ni) != 0)
				{
					(void)us_pophighlight(0);
					return;
				}
				total++;
			}
		}
		if (total == 0) ttyputmsg("No ports to re-export"); else
			ttyputmsg("%d ports re-exported", total);

		/* restore highlighting */
		(void)us_pophighlight(0);
		return;
	}

	if (namesamen(pt, "highlighted-re-export", l) == 0 && l >= 1)
	{
		/* get the necessary polygon */
		if (poly == NOPOLYGON) poly = allocpolygon(4, us_aid->cluster);

		/* make sure there is a current facet */
		if (us_needfacet() == NONODEPROTO) return;

		/* get the bounds of the selected area */
		if (us_getareabounds(&lx, &hx, &ly, &hy) != 0)
		{
			us_abortcommand("Must select an area enclosing ports to re-export");
			return;
		}

		/* get list of nodes in the selected area */
		g = us_gethighlighted(OBJNODEINST);
		if (g[0] == NOGEOM)
		{
			us_abortcommand("Must select some nodes for exporting ports");
			return;
		}

		/* save highlighting */
		us_pushhighlight();
		us_clearhighlightcount();

		/* re-export the requested ports */
		total = 0;
		for(i=0; g[i] != NOGEOM; i++)
		{
			ni = g[i]->entryaddr.ni;

			/* only look for facets, not primitives */
			if (ni->proto->index != 0) continue;

			/* clear marks on the ports of this node */
			for(pp = ni->proto->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
				pp->temp1 = 0;

			/* mark the connected and exported ports */
			for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
				pi->proto->temp1++;
			for(pe = ni->firstportexpinst; pe != NOPORTEXPINST; pe = pe->nextportexpinst)
				pe->proto->temp1++;

			/* now export the remaining ports that are in the area */
			for(pp = ni->proto->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
				if (pp->temp1 == 0)
			{
				shapeportpoly(ni, pp, poly, 0);
				getbbox(poly, &mlx, &mhx, &mly, &mhy);
				if (mlx > hx || mhx < lx || mly > hy || mhy < ly) continue;
				if (us_reexportport(pp, ni) != 0)
				{
					(void)us_pophighlight(0);
					return;
				}
				total++;
			}
		}
		if (total == 0) ttyputmsg("No ports to re-export"); else
			ttyputmsg("%d ports re-exported", total);

		/* restore highlighting */
		(void)us_pophighlight(0);
		return;
	}

	if (namesamen(pt, "not", l) == 0 && l >= 1)
	{
		if (count <= 1 || namesamen(par[1], "export", (INTSML)(strlen(par[1]))) != 0)
		{
			us_abortcommand("Usage: port not export [OPTIONS]");
			return;
		}

		high = us_getonehighlight();
		if (high == NOHIGHLIGHT) return;
		if ((high->status&HIGHTYPE) == HIGHTEXT)
		{
			if (high->fromvar != NOVARIABLE || high->fromport == NOPORTPROTO)
			{
				us_abortcommand("Selected text must be a port name");
				return;
			}
			ni = high->fromgeom->entryaddr.ni;
			ppt = high->fromport;
		} else
		{
			ni = (NODEINST *)us_getobject(OBJNODEINST, 0);
			if (ni == NONODEINST) return;
			ppt = NOPORTPROTO;
		}

		/* first see if there is a port on this nodeinst */
		if (ni->firstportexpinst == NOPORTEXPINST)
		{
			us_abortcommand("No exported ports on this node");
			return;
		}

		/* get the specification details */
		if (count == 3 && namesame(par[2], "all") == 0) pp = NOPORTPROTO; else
		{
			pp = us_portdetails(ppt, (INTSML)(count-2), &par[2], ni, &bits, &mask, 1, "");
			if (pp == NOPORTPROTO) return;
		}

		/* remove highlighting */
		us_clearhighlightcount();

		/* remove the port(s) */
		startobjectchange((INTBIG)ni, VNODEINST);
		us_undoportproto(ni, pp);
		endobjectchange((INTBIG)ni, VNODEINST);

		/* restore highlighting */
		high->fromport = NOPORTPROTO;
		(void)us_addhighlight(high);

		if (pp == NOPORTPROTO) ttyputmsgf("All ports on this node deleted"); else
			ttyputmsgf("Port %s deleted", pp->protoname);
		return;
	}

	if (namesamen(pt, "export", l) == 0 && l >= 1)
	{
		ni = (NODEINST *)us_getobject(OBJNODEINST, 0);
		if (ni == NONODEINST) return;

		if (count <= 1)
		{
			count = ttygetparam("Port name: ", &us_portep, MAXPARS-1, &par[1])+1;
			if (count <= 1)
			{
				us_abortedmsg();
				return;
			}
		}

		/* check name for validity */
		pt = par[1];
		while (*pt == ' ') pt++;
		if (*pt == 0)
		{
			us_abortcommand("Null name");
			return;
		}

		/* find a unique port name */
		portname = us_uniqueportname(pt, ni->parent);
		if (namesame(portname, pt) != 0)
			ttyputmsg("Already a port called %s, calling this %s", pt, portname);

		/* get the specification details */
		pp = us_portdetails(NOPORTPROTO, (INTSML)(count-2), &par[2], ni, &bits, &mask, 0, pt);
		if (pp == NOPORTPROTO) return;

		/* save highlighting */
		us_pushhighlight();
		us_clearhighlightcount();

		/* make the port */
		us_makenewportproto(ni->parent, ni, pp, portname, mask, bits, pp->textdescript);

		/* restore highlighting */
		(void)us_pophighlight(0);

		ttyputmsgf("Port %s created", pp->protoname);
		return;
	}

	if (namesamen(pt, "change", l) == 0 && l >= 1)
	{
		high = us_getonehighlight();
		if (high == NOHIGHLIGHT) return;
		if ((high->status&HIGHTYPE) == HIGHTEXT)
		{
			if (high->fromvar != NOVARIABLE || high->fromport == NOPORTPROTO)
			{
				us_abortcommand("Selected text must be a port name");
				return;
			}
			ni = high->fromgeom->entryaddr.ni;
			ppt = high->fromport;
		} else
		{
			ni = (NODEINST *)us_getobject(OBJNODEINST, 0);
			if (ni == NONODEINST) return;
			ppt = NOPORTPROTO;
		}

		/* get the specification details */
		pp = us_portdetails(ppt, (INTSML)(count-1), &par[1], ni, &bits, &mask, 1, "");
		if (pp == NOPORTPROTO) return;

		/* change the port characteristics */
		if ((mask&STATEBITS) != 0)
			pp->userbits = (pp->userbits & ~STATEBITS) | (bits & STATEBITS);
		if ((mask&PORTDRAWN) != 0)
			pp->userbits = (pp->userbits & ~PORTDRAWN) | (bits & PORTDRAWN);
		if ((mask&BODYONLY) != 0)
			pp->userbits = (pp->userbits & ~BODYONLY) | (bits & BODYONLY);

		/* propagate the change information up the hierarchy */
		changeallports(pp);

		(void)initinfstr();
		(void)addstringtoinfstr("Port '");
		(void)addstringtoinfstr(pp->protoname);
		if ((pp->userbits&STATEBITS) == 0)
			(void)addstringtoinfstr("' has no characteristics "); else
		{
			(void)addstringtoinfstr("' is ");
			(void)addstringtoinfstr(us_describeportbits(pp));
		}
		(void)addstringtoinfstr(" (label ");
		(void)addstringtoinfstr(us_describetextdescript(pp->textdescript));
		if ((pp->userbits&PORTDRAWN) != 0) (void)addstringtoinfstr(",always-drawn");
		if ((pp->userbits&BODYONLY) != 0) (void)addstringtoinfstr(",only-on-body");
		(void)addstringtoinfstr(") ");
		ttyputmsg("%s", returninfstr());
		return;
	}

	us_abortcommand("Bad PORT option: %s", pt);
}

void us_quit(INTSML count, char *par[])
{
	if (us_preventloss(count, par, NOLIBRARY, "Quit")) return;
	bringdown();
}
