/*
 * Electric(tm) VLSI Design System
 *
 * File: usrcomtz.c
 * User interface aid: command handler for W through Z
 * Written by: Steven M. Rubin, Electric Editor Incorporated
 *
 * Copyright (c) 1998 Electric Editor Incorporated.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Electric Editor Incorporated
 * 23470 Sunset Drive, Suite 108
 * Los Gatos, California 95033
 * support@electriceditor.com
 */

#include "global.h"
#include "egraphics.h"
#include "usr.h"
#include "usrtrack.h"
#include "efunction.h"
#include <ctype.h>

struct
{
	char         *keyword;
	short         unique;
	STATUSFIELD **fieldaddr;
} us_statusfields[] =
{
	{"align",      2,  &us_statusalign},
	{"angle",      2,  &us_statusangle},
	{"arc",        2,  &us_statusarc},
	{"facet",      1,  &us_statusfacet},
	{"grid",       1,  &us_statusgridsize},
	{"lambda",     1,  &us_statuslambda},
	{"network",    2,  &us_statusnetwork},
	{"node",       2,  &us_statusnode},
	{"package",    3,  &us_statuspackage},
	{"part",       3,  &us_statuspart},
	{"project",    2,  &us_statusproject},
	{"root",       1,  &us_statusroot},
	{"selection",  2,  &us_statusselection},
	{"size",       2,  &us_statusfacetsize},
	{"technology", 1,  &us_statustechnology},
	{"x",          1,  &us_statusxpos},
	{"y",          1,  &us_statusypos},
	{0,0,0}
};

void us_window(INTSML count, char *par[])
{
	REGISTER WINDOW *w, *oldw, *nextw, *neww;
	REGISTER INTBIG i, dist, curwx, curwy, size, x, y, diffx, diffy;
	REGISTER INTSML l, nogood, splitkey, lineno, startper, endper;
	INTBIG lx, hx, ly, hy, xcur, ycur, schx, schy, windowView[8];
	static POLYGON *poly = NOPOLYGON;
	REGISTER STATUSFIELD *sf, **whichstatus;
	char *newpar[4], *fieldname;
	void *mw;
#if SIMAID
	extern AIDENTRY *sim_aid;
#endif
	REGISTER char *pp, *win;
	REGISTER VARIABLE *var;
	REGISTER HIGHLIGHT *high;
	REGISTER NODEPROTO *np;
	extern GRAPHICS us_arbit;
	extern COMCOMP us_windowp, us_windowup, us_windowmp;

	/* get polygon */
	if (poly == NOPOLYGON) poly = allocpolygon(4, us_aid->cluster);

	if (count == 0)
	{
		count = ttygetparam("Window configuration: ", &us_windowp, MAXPARS, par);
		if (count == 0)
		{
			us_abortedmsg();
			return;
		}
	}
	l = (INTSML)strlen(pp = par[0]);

	if (namesamen(pp, "all-displayed", l) == 0 && l >= 1)
	{
		if (us_needwindow()) return;

		/* use direct methods on nonstandard windows */
		if ((el_curwindow->state&WINDOWTYPE) != DISPWINDOW)
		{
			if (el_curwindow->redisphandler != 0)
				(*el_curwindow->redisphandler)(el_curwindow);
			return;
		}

		np = us_needfacet();
		if (np == NONODEPROTO) return;

		/* save and erase highlighting */
		us_pushhighlight();
		us_clearhighlightcount();

		/* make the facet fill the window */
		us_fullview(np, &lx, &hx, &ly, &hy);
		us_squarescreen(el_curwindow, NOWINDOW, 0, &lx, &hx, &ly, &hy);

		startobjectchange((INTBIG)el_curwindow, VWINDOW);
		(void)setval((INTBIG)el_curwindow, VWINDOW, "screenlx", lx, VINTEGER);
		(void)setval((INTBIG)el_curwindow, VWINDOW, "screenhx", hx, VINTEGER);
		(void)setval((INTBIG)el_curwindow, VWINDOW, "screenly", ly, VINTEGER);
		(void)setval((INTBIG)el_curwindow, VWINDOW, "screenhy", hy, VINTEGER);
		us_gridset(el_curwindow, el_curwindow->state);
		endobjectchange((INTBIG)el_curwindow, VWINDOW);

		/* restore highlighting */
		(void)us_pophighlight(0);
		return;
	}

	if (namesamen(pp, "center-highlight", l) == 0 && l >= 2)
	{
		np = us_needfacet();
		if (np == NONODEPROTO) return;

		/* cannot manipulate a nonstandard window */
		if ((el_curwindow->state&WINDOWTYPE) != DISPWINDOW)
		{
			us_abortcommand("Can only pan circuit editing windows");
			return;
		}

		if (us_getareabounds(&lx, &hx, &ly, &hy) != 0)
		{
			us_abortcommand("Outline an area");
			return;
		}

		/* pre-compute current window size */
		curwx = el_curwindow->screenhx - el_curwindow->screenlx;
		curwy = el_curwindow->screenhy - el_curwindow->screenly;

		/* center about this area without re-scaling */
		x = (hx + lx) / 2;     y = (hy + ly) / 2;
		lx = x - curwx/2;      ly = y - curwy/2;
		hx = lx + curwx;       hy = ly + curwy;

		/* limit schematic windows to their normal extent */
		if (framesize(&schx, &schy, np) == 0)
		{
			if (lx < -schx/2) { lx = -schx/2;   hx = lx + curwx; }
			if (hx >  schx/2) { hx =  schx/2;   lx = hx - curwx; }
			if (ly < -schy/2) { ly = -schy/2;   hy = ly + curwy; }
			if (hy >  schy/2) { hy =  schy/2;   ly = hy - curwy; }
		}

		/* save and erase highlighting */
		us_pushhighlight();
		us_clearhighlightcount();

		startobjectchange((INTBIG)el_curwindow, VWINDOW);
		(void)setval((INTBIG)el_curwindow, VWINDOW, "screenlx", lx, VINTEGER);
		(void)setval((INTBIG)el_curwindow, VWINDOW, "screenhx", hx, VINTEGER);
		(void)setval((INTBIG)el_curwindow, VWINDOW, "screenly", ly, VINTEGER);
		(void)setval((INTBIG)el_curwindow, VWINDOW, "screenhy", hy, VINTEGER);
		us_gridset(el_curwindow, el_curwindow->state);
		endobjectchange((INTBIG)el_curwindow, VWINDOW);

		/* restore highlighting */
		(void)us_pophighlight(0);
		return;
	}

	if (namesamen(pp, "cursor-centered", l) == 0 && l >= 2)
	{
		np = us_needfacet();
		if (np == NONODEPROTO) return;

		/* cannot manipulate a nonstandard window */
		if ((el_curwindow->state&WINDOWTYPE) != DISPWINDOW)
		{
			us_abortcommand("Can only pan circuit editing windows");
			return;
		}

		if (us_demandxy(&xcur, &ycur)) return;

		/* pre-compute current window size */
		curwx = el_curwindow->screenhx - el_curwindow->screenlx;
		curwy = el_curwindow->screenhy - el_curwindow->screenly;

		lx = xcur - curwx/2;   ly = ycur - curwy/2;
		hx = lx + curwx;       hy = ly + curwy;

		/* limit schematic windows to their normal extent */
		if (framesize(&schx, &schy, np) == 0)
		{
			if (lx < -schx/2) { lx = -schx/2;   hx = lx + curwx; }
			if (hx >  schx/2) { hx =  schx/2;   lx = hx - curwx; }
			if (ly < -schy/2) { ly = -schy/2;   hy = ly + curwy; }
			if (hy >  schy/2) { hy =  schy/2;   ly = hy - curwy; }
		}

		/* save and erase highlighting */
		us_pushhighlight();
		us_clearhighlightcount();

		startobjectchange((INTBIG)el_curwindow, VWINDOW);
		(void)setval((INTBIG)el_curwindow, VWINDOW, "screenlx", lx, VINTEGER);
		(void)setval((INTBIG)el_curwindow, VWINDOW, "screenhx", hx, VINTEGER);
		(void)setval((INTBIG)el_curwindow, VWINDOW, "screenly", ly, VINTEGER);
		(void)setval((INTBIG)el_curwindow, VWINDOW, "screenhy", hy, VINTEGER);
		us_gridset(el_curwindow, el_curwindow->state);
		endobjectchange((INTBIG)el_curwindow, VWINDOW);

		/* restore highlighting */
		(void)us_pophighlight(0);
		return;
	}

	/* handle deletion of separate windows */
	if (namesamen(pp, "delete", l) == 0 && l >= 2)
	{
		/* close the messages window if in front and can be closed */
		if (us_closefrontmostmessages() != 0) return;

		/* delete split if there are no multiple window frames */
		if (us_graphicshas(CANUSEFRAMES) == 0)
		{
			if (us_needwindow()) return;
			us_killcurrentwindow(1);
			return;
		}

		/* create a new frame */
		mw = us_getwindowframe();
		if (mw == 0)
		{
			us_abortcommand("No current window to delete");
			return;
		}

		/* save highlighting and turn it off */
		us_pushhighlight();
		us_clearhighlightcount();

		startobjectchange((INTBIG)us_aid, VAID);

		/* kill all editor windows on this frame */
		neww = NOWINDOW;
		for(w = el_topwindow; w != NOWINDOW; w = nextw)
		{
			nextw = w->nextwindow;
			if (w->frame != mw)
			{
				neww = w;
				continue;
			}

			/* kill this window */
			killwindow(w);
		}
		endobjectchange((INTBIG)us_aid, VAID);

		(void)setvalkey((INTBIG)us_aid, VAID, us_current_window, (INTBIG)neww, VWINDOW|VDONTSAVE);
		if (neww != NOWINDOW) np = neww->curnodeproto; else np = NONODEPROTO;
		(void)setval((INTBIG)el_curlib, VLIBRARY, "curnodeproto", (INTBIG)np, VNODEPROTO);

		/* restore highlighting */
		(void)us_pophighlight(0);
		return;
	}

	if (namesamen(pp, "down", l) == 0 && l >= 2)
	{
		if (us_needwindow()) return;

#if SIMAID
		/* special case for waveform window */
		if ((el_curwindow->state&WINDOWTYPE) == WAVEFORMWINDOW)
		{
			newpar[0] = "window";
			newpar[1] = "move";
			newpar[2] = "down";
			(void)tellaid(sim_aid, 3, newpar);
			return;
		}
#endif

		/* cannot pan a nonstandard window */
		if ((el_curwindow->state&WINDOWTYPE) != DISPWINDOW)
		{
			us_abortcommand("Can only pan circuit editing windows");
			return;
		}

		if (us_needfacet() == NONODEPROTO) return;
		if (count < 2) pp = "0.5"; else pp = par[1];
		if (pp[strlen(pp)-1] == 'l') dist = atola(pp); else
			dist = muldiv(atofr(pp), (el_curwindow->screenhy - el_curwindow->screenly), WHOLE);
		us_slideup(-dist);
		return;
	}

	if (namesamen(pp, "dragging", l) == 0 && l >= 3)
	{
		if (count >= 2)
		{
			pp = par[1];
			if (namesame(pp, "on") == 0)
				(void)setval((INTBIG)us_aid, VAID, "aidstate", us_aid->aidstate | INTERACTIVE, VINTEGER);
			else if (namesamen(pp, "of", 2) == 0)
				(void)setval((INTBIG)us_aid, VAID, "aidstate", us_aid->aidstate & ~INTERACTIVE, VINTEGER);
			else
			{
				us_abortcommand("Usage: window dragging on|off");
				return;
			}
		}
		if ((us_aid->aidstate&INTERACTIVE) == 0)
			ttyputmsgf("Cursor-based commands will act immediately"); else
				ttyputmsgf("Cursor-based commands will drag their objects");
		return;
	}

	if (namesamen(pp, "drop-down-menus", l) == 0 && l >= 3)
	{
		(void)setval((INTBIG)us_aid, VAID, "aidstate", us_aid->aidstate | DROPMENUS, VINTEGER);
		ttyputmsgf("Menus will drop-down");
		return;
	}

	if (namesamen(pp, "highlight-displayed", l) == 0 && l >= 1)
	{
		if (us_needwindow()) return;

#if SIMAID
		/* special case for waveform window */
		if ((el_curwindow->state&WINDOWTYPE) == WAVEFORMWINDOW)
		{
			newpar[0] = "window";
			newpar[1] = "zoom";
			newpar[2] = "cursor";
			(void)tellaid(sim_aid, 3, newpar);
			return;
		}
#endif

		/* cannot manipulate a nonstandard window */
		if ((el_curwindow->state&WINDOWTYPE) != DISPWINDOW)
		{
			us_abortcommand("Can only adjust circuit editing windows");
			return;
		}

		if (us_getareabounds(&lx, &hx, &ly, &hy) != 0)
		{
			us_abortcommand("Outline an area");
			return;
		}

		/* pre-compute current window size */
		curwx = el_curwindow->screenhx - el_curwindow->screenlx;
		curwy = el_curwindow->screenhy - el_curwindow->screenly;

		/* limit schematic windows to their normal extent */
		np = us_needfacet();
		if (np == NONODEPROTO) return;
		i = framesize(&schx, &schy, np);
		if (i == 0)
		{
			if (lx < -schx/2) lx = -schx/2;
			if (hx > schx/2) hx = schx/2;
			if (ly < -schy/2) ly = -schy/2;
			if (hy > schy/2) hy = schy/2;
		}

		/* save and erase highlighting */
		us_pushhighlight();
		us_clearhighlightcount();

		/* make sure the new window has square pixels */
		us_squarescreen(el_curwindow, NOWINDOW, 0, &lx, &hx, &ly, &hy);

		/* make sure squaring didn't push extent of schematic frame */
		if (i == 0)
		{
			if (lx < -schx/2)
			{
				hx -= schx/2 + lx;
				lx = -schx/2;
			}
			if (hx > schx/2)
			{
				lx -= hx - schx/2;
				hx = schx/2;
			}
			if (ly < -schy/2)
			{
				hy -= schy/2 + ly;
				ly = -schy/2;
			}
			if (hy > schy/2)
			{
				ly -= hy - schy/2;
				hy = schy/2;
			}
		}

		startobjectchange((INTBIG)el_curwindow, VWINDOW);
		(void)setval((INTBIG)el_curwindow, VWINDOW, "screenlx", lx, VINTEGER);
		(void)setval((INTBIG)el_curwindow, VWINDOW, "screenhx", hx, VINTEGER);
		(void)setval((INTBIG)el_curwindow, VWINDOW, "screenly", ly, VINTEGER);
		(void)setval((INTBIG)el_curwindow, VWINDOW, "screenhy", hy, VINTEGER);
		us_gridset(el_curwindow, el_curwindow->state);
		endobjectchange((INTBIG)el_curwindow, VWINDOW);

		/* restore highlighting */
		(void)us_pophighlight(0);
		return;
	}

	if (namesamen(pp, "in-zoom", l) == 0 && l >= 1)
	{
		if (us_needwindow()) return;

#if SIMAID
		/* special case for waveform window */
		if ((el_curwindow->state&WINDOWTYPE) == WAVEFORMWINDOW)
		{
			newpar[0] = "window";
			newpar[1] = "zoom";
			newpar[2] = "in";
			(void)tellaid(sim_aid, 3, newpar);
			return;
		}
#endif

		/* cannot zoom a nonstandard window */
		if ((el_curwindow->state&WINDOWTYPE) != DISPWINDOW)
		{
			us_abortcommand("Can only zoom circuit editing windows");
			return;
		}

		if (us_needfacet() == NONODEPROTO) return;
		if (count >= 2) dist = atola(par[1]); else
			dist = 2 * el_curtech->deflambda;
		if (dist == 0)
		{
			us_abortcommand("Must zoom by a nonzero amount");
			return;
		}

		/* save and erase highlighting */
		us_pushhighlight();
		us_clearhighlightcount();

		diffx = muldiv(el_curwindow->screenhx - el_curwindow->screenlx,
			el_curtech->deflambda, dist);
		diffy = muldiv(el_curwindow->screenhy - el_curwindow->screenly,
			el_curtech->deflambda, dist);
		lx = el_curwindow->screenlx;   hx = el_curwindow->screenhx;
		ly = el_curwindow->screenly;   hy = el_curwindow->screenhy;
		lx = (hx+lx-diffx)/2;   hx = lx + diffx;
		ly = (hy+ly-diffy)/2;   hy = ly + diffy;
		us_squarescreen(el_curwindow, NOWINDOW, 0, &lx, &hx, &ly, &hy);

		startobjectchange((INTBIG)el_curwindow, VWINDOW);
		(void)setval((INTBIG)el_curwindow, VWINDOW, "screenlx", lx, VINTEGER);
		(void)setval((INTBIG)el_curwindow, VWINDOW, "screenhx", hx, VINTEGER);
		(void)setval((INTBIG)el_curwindow, VWINDOW, "screenly", ly, VINTEGER);
		(void)setval((INTBIG)el_curwindow, VWINDOW, "screenhy", hy, VINTEGER);
		us_gridset(el_curwindow, el_curwindow->state);
		endobjectchange((INTBIG)el_curwindow, VWINDOW);

		/* restore highlighting */
		(void)us_pophighlight(0);
		return;
	}

	/* handle killing of the other window specially */
	if (namesamen(pp, "join", l) == 0 && l >= 1)
	{
		if (us_needwindow()) return;
		us_killcurrentwindow(0);
		return;
	}

	/* handle killing of this window specially */
	if (namesamen(pp, "kill", l) == 0 && l >= 1)
	{
		if (us_needwindow()) return;
		us_killcurrentwindow(1);
		return;
	}

	if (namesamen(pp, "left", l) == 0 && l >= 1)
	{
		if (us_needwindow()) return;

#if SIMAID
		/* special case for waveform window */
		if ((el_curwindow->state&WINDOWTYPE) == WAVEFORMWINDOW)
		{
			newpar[0] = "window";
			newpar[1] = "move";
			newpar[2] = "left";
			(void)tellaid(sim_aid, 3, newpar);
			return;
		}
#endif

		/* cannot pan a nonstandard window */
		if ((el_curwindow->state&WINDOWTYPE) != DISPWINDOW)
		{
			us_abortcommand("Can only pan circuit editing windows");
			return;
		}

		if (us_needfacet() == NONODEPROTO) return;
		if (count < 2) pp = "0.5"; else pp = par[1];
		if (pp[strlen(pp)-1] == 'l') dist = atola(pp); else
			dist = muldiv(atofr(pp), (el_curwindow->screenhx - el_curwindow->screenlx), WHOLE);
		us_slideleft(dist);
		return;
	}

	if (namesamen(pp, "magnification", l) == 0 && l >= 3)
	{
		if (us_graphicshas(CANMAGNIFY) == 0)
		{
			us_abortcommand("Sorry, this display cannot magnify the window");
			return;
		}
		if (count >= 2)
		{
			l = strlen(pp = par[1]);
			if (namesamen(pp, "off", l) == 0 && l >= 2) i = 0; else
			if (namesamen(pp, "temporary", l) == 0 && l >= 1)
			{
				if (count < 3) i = -4; else i = -myatoi(par[2]);
			} else if (namesamen(pp, "on", l) == 0 && l >= 2)
			{
				if (count < 3) i = 4; else i = myatoi(par[2]);
			} else
			{
				us_abortcommand("Usage: window magnification off|on|temporary");
				return;
			}
			(void)setvalkey((INTBIG)us_aid, VAID, us_spyglassfactor, i,
				VINTEGER|VDONTSAVE);
		}
		var = getvalkey((INTBIG)us_aid, VAID, VINTEGER, us_spyglassfactor);
		if (var == NOVARIABLE) i = 0; else i = var->addr;
		if (i == 0) ttyputmsgf("Window magnification is off"); else
			if (i < 0) ttyputmsgf("Window magnification is temporarily on"); else
				ttyputmsgf("Window magnification is on");
		return;
	}

	if (namesamen(pp, "match", l) == 0 && l >= 3)
	{
		/* count the number of windows */
		for(i = 0, w = el_topwindow; w != NOWINDOW; w = w->nextwindow) i++;
		if (i <= 1)
		{
			us_abortcommand("Must be multiple windows to match them");
			return;
		}

		/* if there are two windows, the other to match is obvious */
		if (i == 2)
		{
			if (el_curwindow == el_topwindow) w = el_topwindow->nextwindow; else
				w = el_topwindow;
		} else
		{
			if (count < 2)
			{
				count = ttygetparam("Other window to match: ", &us_windowmp, MAXPARS-1, &par[1]) + 1;
				if (count == 1)
				{
					us_abortedmsg();
					return;
				}
			}
			win = par[1];
			for(w = el_topwindow; w != NOWINDOW; w = w->nextwindow)
				if (namesame(win, w->location) == 0) break;
			if (w == NOWINDOW)
			{
				us_abortcommand("No window named '%s'", win);
				return;
			}
		}

		if (w == el_curwindow)
		{
			us_abortcommand("Choose a window other than the current one to match");
			return;
		}

		/* cannot match if they are editing the same thing */
		if ((w->state&WINDOWTYPE) != (el_curwindow->state&WINDOWTYPE))
		{
			us_abortcommand("Can only match windows that edit the same thing");
			return;
		}

		/* cannot match if they are not normal display windows */
		if ((w->state&WINDOWTYPE) != DISPWINDOW)
		{
			us_abortcommand("Can only match normal editing windows");
			return;
		}

		/* save and erase highlighting */
		us_pushhighlight();
		us_clearhighlightcount();

		/* make window "el_curwindow" match the scale of "w" */
		diffx = muldiv(w->screenhx - w->screenlx, el_curwindow->usehx -
			el_curwindow->uselx, w->usehx - w->uselx);
		diffx = diffx - (el_curwindow->screenhx - el_curwindow->screenlx);
		diffy = muldiv(w->screenhy - w->screenly, el_curwindow->usehy -
			el_curwindow->usely, w->usehy - w->usely);
		diffy = diffy - (el_curwindow->screenhy - el_curwindow->screenly);

		startobjectchange((INTBIG)el_curwindow, VWINDOW);
		(void)setval((INTBIG)el_curwindow, VWINDOW, "screenlx",
			el_curwindow->screenlx - diffx/2, VINTEGER);
		(void)setval((INTBIG)el_curwindow, VWINDOW, "screenhx",
			el_curwindow->screenhx + diffx/2, VINTEGER);
		(void)setval((INTBIG)el_curwindow, VWINDOW, "screenly",
			el_curwindow->screenly - diffy/2, VINTEGER);
		(void)setval((INTBIG)el_curwindow, VWINDOW, "screenhy",
			el_curwindow->screenhy + diffy/2, VINTEGER);
		us_gridset(el_curwindow, el_curwindow->state);
		endobjectchange((INTBIG)el_curwindow, VWINDOW);

		/* restore highlighting */
		(void)us_pophighlight(0);
		return;
	}

	if (namesamen(pp, "measure", l) == 0 && l >= 2)
	{
		us_clearhighlightcount();
		trackcursor(1, us_ignoreup, us_distancebegin, us_distancedown,
			us_stoponchar, us_distanceup, TRACKDRAGGING);
		return;
	}

	if (namesamen(pp, "name", l) == 0 && l >= 2)
	{
		if (us_needwindow()) return;
		if (count <= 1)
		{
			us_abortcommand("Usage: window name VIEWNAME");
			return;
		}
		(void)initinfstr();
		(void)addstringtoinfstr("USER_windowview_");
		(void)addstringtoinfstr(par[1]);
		var = getval((INTBIG)us_aid, VAID, VINTEGER|VISARRAY, returninfstr());
		if (var == NOVARIABLE)
		{
			us_abortcommand("Cannot find saved window view '%s'", par[1]);
			return;
		}

		/* save and erase highlighting */
		us_pushhighlight();
		us_clearhighlightcount();

		lx = ((INTBIG *)var->addr)[0];
		hx = ((INTBIG *)var->addr)[1];
		ly = ((INTBIG *)var->addr)[2];
		hy = ((INTBIG *)var->addr)[3];

		/* if the window extent changed, make sure the pixels are square */
		if (((INTBIG *)var->addr)[5] - ((INTBIG *)var->addr)[4] != el_curwindow->usehx - el_curwindow->uselx ||
			((INTBIG *)var->addr)[7] - ((INTBIG *)var->addr)[6] != el_curwindow->usehy - el_curwindow->usely)
		{
			us_squarescreen(el_curwindow, NOWINDOW, 0, &lx, &hx, &ly, &hy);
		}

		startobjectchange((INTBIG)el_curwindow, VWINDOW);
		(void)setval((INTBIG)el_curwindow, VWINDOW, "screenlx", lx, VINTEGER);
		(void)setval((INTBIG)el_curwindow, VWINDOW, "screenhx", hx, VINTEGER);
		(void)setval((INTBIG)el_curwindow, VWINDOW, "screenly", ly, VINTEGER);
		(void)setval((INTBIG)el_curwindow, VWINDOW, "screenhy", hy, VINTEGER);
		us_gridset(el_curwindow, el_curwindow->state);
		endobjectchange((INTBIG)el_curwindow, VWINDOW);

		/* restore highlighting */
		(void)us_pophighlight(0);
		return;
	}

	/* handle creating of separate windows */
	if (namesamen(pp, "new", l) == 0 && l >= 2)
	{
		/* create a new frame */
		w = us_wantnewwindow();
		if (w == NOWINDOW)
		{
			us_abortcommand("Cannot create new window frame");
			return;
		}
		return;
	}

	if (namesamen(pp, "normal-cursor", l) == 0 && l >= 2)
	{
		if (count < 2)
		{
			us_abortcommand("Usage: window normal-cursor CURSORNAME");
			return;
		}
		l = strlen(pp = par[1]);
		if (namesamen(pp, "standard", l) == 0) (void)us_setnormalcursor(0); else
			if (namesamen(pp, "pen", l) == 0) (void)us_setnormalcursor(1); else
				if (namesamen(pp, "tee", l) == 0) (void)us_setnormalcursor(2); else
					us_abortcommand("Bad window cursor option: %s", pp);
		return;
	}

	if (namesamen(pp, "out-zoom", l) == 0 && l >= 1)
	{
		if (us_needwindow()) return;

#if SIMAID
		/* special case for waveform window */
		if ((el_curwindow->state&WINDOWTYPE) == WAVEFORMWINDOW)
		{
			newpar[0] = "window";
			newpar[1] = "zoom";
			newpar[2] = "out";
			(void)tellaid(sim_aid, 3, newpar);
			return;
		}
#endif

		/* cannot zoom a nonstandard window */
		if ((el_curwindow->state&WINDOWTYPE) != DISPWINDOW)
		{
			us_abortcommand("Can only zoom circuit editing windows");
			return;
		}

		np = us_needfacet();
		if (np == NONODEPROTO) return;
		if (count >= 2) dist = atola(par[1]); else
			dist = 2 * el_curtech->deflambda;
		if (dist == 0)
		{
			us_abortcommand("Must zoom by a nonzero amount");
			return;
		}

		/* save and erase highlighting */
		us_pushhighlight();
		us_clearhighlightcount();

		lx = el_curwindow->screenlx;   hx = el_curwindow->screenhx;
		ly = el_curwindow->screenly;   hy = el_curwindow->screenhy;
		i = muldiv(el_curwindow->screenhx - el_curwindow->screenlx, dist, el_curtech->deflambda);
		lx = (lx+hx-i)/2;   hx = lx + i;
		i = muldiv(el_curwindow->screenhy - el_curwindow->screenly, dist, el_curtech->deflambda);
		ly = (ly+hy-i)/2;   hy = ly + i;

		/* limit schematic windows to their normal extent */
		i = framesize(&schx, &schy, np);
		if (i == 0)
		{
			if (lx < -schx/2) lx = -schx/2;
			if (hx > schx/2) hx = schx/2;
			if (ly < -schy/2) ly = -schy/2;
			if (hy > schy/2) hy = schy/2;
		}
		us_squarescreen(el_curwindow, NOWINDOW, 0, &lx, &hx, &ly, &hy);

		/* make sure squaring didn't push extent of schematic frame */
		if (i == 0)
		{
			if (lx < -schx/2)
			{
				hx -= schx/2 + lx;
				lx = -schx/2;
			}
			if (hx > schx/2)
			{
				lx -= hx - schx/2;
				hx = schx/2;
			}
			if (ly < -schy/2)
			{
				hy -= schy/2 + ly;
				ly = -schy/2;
			}
			if (hy > schy/2)
			{
				ly -= hy - schy/2;
				hy = schy/2;
			}
		}

		startobjectchange((INTBIG)el_curwindow, VWINDOW);
		(void)setval((INTBIG)el_curwindow, VWINDOW, "screenlx", lx, VINTEGER);
		(void)setval((INTBIG)el_curwindow, VWINDOW, "screenhx", hx, VINTEGER);
		(void)setval((INTBIG)el_curwindow, VWINDOW, "screenly", ly, VINTEGER);
		(void)setval((INTBIG)el_curwindow, VWINDOW, "screenhy", hy, VINTEGER);
		us_gridset(el_curwindow, el_curwindow->state);
		endobjectchange((INTBIG)el_curwindow, VWINDOW);

		/* restore highlighting */
		(void)us_pophighlight(0);
		return;
	}

	if (namesamen(pp, "overlappable-display", l) == 0 && l >= 2)
	{
		if (count >= 2)
		{
			pp = par[1];
			l = strlen(pp);
			if (namesamen(pp, "on", l) == 0)
			{
				us_state &= ~NONOVERLAPPABLEDISPLAY;
			} else if (namesamen(pp, "off", l) == 0)
			{
				us_state |= NONOVERLAPPABLEDISPLAY;
			} else
			{
				us_abortcommand("Usage: window overlappable-display [on|off]");
				return;
			}
		}
		if ((us_state&NONOVERLAPPABLEDISPLAY) != 0)
			ttyputmsgf("Overlappable layers will not be handled"); else
				ttyputmsgf("Overlappable layers will be drawn properly");
		return;
	}

	if (namesamen(pp, "peek", l) == 0 && l >= 2)
	{
		if (us_getareabounds(&lx, &hx, &ly, &hy) != 0)
		{
			us_abortcommand("Enclose an area to be peeked");
			return;
		}
		high = us_getonehighlight();
		if (high == NOHIGHLIGHT) return;
		np = us_needfacet();
		if (np == NONODEPROTO) return;
		if (np != high->facet)
		{
			us_abortcommand("Current window must contain highlighted area");
			return;
		}

		/* clip this bounding box to the window extent */
		lx = maxi(lx, el_curwindow->screenlx);
		hx = mini(hx, el_curwindow->screenhx);
		ly = maxi(ly, el_curwindow->screenly);
		hy = mini(hy, el_curwindow->screenhy);

		/* save and erase highlighting */
		us_pushhighlight();
		us_clearhighlightcount();

		/* un-draw the peek area */
		maketruerectpoly(lx, hx, ly, hy, poly);
		poly->desc = &us_arbit;
		us_arbit.col = 0;
		us_arbit.bits = LAYERO;
		poly->style = FILLEDRECT;
		(void)us_showpoly(poly, el_curwindow);

		/* get new window to describe sub-area */
		w = us_subwindow(lx, hx, ly, hy, el_curwindow);

		/* do the peek operation */
		us_dopeek(lx, hx, ly, hy, np, w);

		/* restore highlighting */
		(void)us_pophighlight(0);
		return;
	}

	if (namesamen(pp, "pop-up-menus", l) == 0 && l >= 3)
	{
		(void)setval((INTBIG)us_aid, VAID, "aidstate", us_aid->aidstate & ~DROPMENUS, VINTEGER);
		ttyputmsgf("Menus will pop-up");
		return;
	}

	if (namesamen(pp, "right", l) == 0 && l >= 1)
	{
		if (us_needwindow()) return;

#if SIMAID
		/* special case for waveform window */
		if ((el_curwindow->state&WINDOWTYPE) == WAVEFORMWINDOW)
		{
			newpar[0] = "window";
			newpar[1] = "move";
			newpar[2] = "right";
			(void)tellaid(sim_aid, 3, newpar);
			return;
		}
#endif

		/* cannot pan a nonstandard window */
		if ((el_curwindow->state&WINDOWTYPE) != DISPWINDOW)
		{
			us_abortcommand("Can only pan circuit editing windows");
			return;
		}

		if (us_needfacet() == NONODEPROTO) return;
		if (count < 2) pp = "0.5"; else pp = par[1];
		if (pp[strlen(pp)-1] == 'l') dist = atola(pp); else
			dist = muldiv(atofr(pp), (el_curwindow->screenhx - el_curwindow->screenlx), WHOLE);
		us_slideleft(-dist);
		return;
	}

	if (namesamen(pp, "save", l) == 0 && l >= 2)
	{
		if (us_needwindow()) return;
		if (count <= 1)
		{
			us_abortcommand("Usage: window save VIEWNAME");
			return;
		}
		windowView[0] = el_curwindow->screenlx;
		windowView[1] = el_curwindow->screenhx;
		windowView[2] = el_curwindow->screenly;
		windowView[3] = el_curwindow->screenhy;
		windowView[4] = el_curwindow->uselx;
		windowView[5] = el_curwindow->usehx;
		windowView[6] = el_curwindow->usely;
		windowView[7] = el_curwindow->usehy;
		(void)initinfstr();
		(void)addstringtoinfstr("USER_windowview_");
		(void)addstringtoinfstr(par[1]);
		(void)setval((INTBIG)us_aid, VAID, returninfstr(), (INTBIG)windowView,
			VINTEGER|VISARRAY|(8<<VLENGTHSH)|VDONTSAVE);
		ttyputmsgf("Window view %s saved", par[1]);
		return;
	}

	if (namesamen(pp, "split", l) == 0 && l >= 2)
	{
		if (us_needwindow()) return;
		splitkey = 0;
		if (count > 1)
		{
			l = strlen(pp = par[1]);
			if (namesamen(pp, "horizontal", l) == 0 && l >= 1) splitkey = 1; else
				if (namesamen(pp, "vertical", l) == 0 && l >= 1) splitkey = 2; else
			{
				us_abortcommand("Usage: window split horizontal|vertical");
				return;
			}
		}

		/* split the window */
		(void)us_splitcurrentwindow(splitkey, 1);
		return;
	}

	if (namesamen(pp, "status-bar", l) == 0 && l >= 2)
	{
		if (count < 2)
		{
			/* report all status bar locations */
			for(i=0; us_statusfields[i].fieldaddr != 0; i++)
			{
				sf = *us_statusfields[i].fieldaddr;
				if (sf == 0) continue;
				if (sf->line == 0)
					ttyputmsg("Window title has %s", us_statusfields[i].fieldaddr); else
						ttyputmsg("Line %d from %3d%% to %3d%% is %s", sf->line, sf->startper,
							sf->endper, us_statusfields[i].fieldaddr);
			}
			return;
		}

		if (namesamen(par[1], "current-node", (INTSML)strlen(par[1])) == 0)
		{
			if (count < 3)
			{
				if ((us_state&NONPERSISTENTCURNODE) != 0)
					ttyputmsg("Current node displayed temporarily"); else
						ttyputmsg("Current node display is persistent");
				return;
			}
			l = strlen(pp = par[2]);
			if (namesamen(pp, "persistent", l) == 0)
			{
				us_state &= ~NONPERSISTENTCURNODE;
				ttyputmsgf("Current node display is persistent");
				return;
			}
			if (namesamen(pp, "temporary", l) == 0)
			{
				us_state |= NONPERSISTENTCURNODE;
				ttyputmsgf("Current node displayed temporarily");
				return;
			}
			us_abortcommand("Usage: find current-node [persistent|temporary]");
			return;
		}

		if (count < 3)
		{
			us_abortcommand("Usage: window status-bar %s FIELD...", pp);
			return;
		}

		/* determine area being controlled */
		l = strlen(pp = par[2]);
		for(i=0; us_statusfields[i].keyword != 0; i++)
			if (namesamen(pp, us_statusfields[i].keyword, l) == 0 &&
				l >= us_statusfields[i].unique)
		{
			whichstatus = us_statusfields[i].fieldaddr;
			break;
		}
		if (us_statusfields[i].keyword == 0)
		{
			us_abortcommand("Unknown status-bar location: %s", pp);
			return;
		}

		/* get option */
		l = strlen(pp = par[1]);
		if (namesamen(pp, "delete", l) == 0)
		{
			if (*whichstatus != 0) ttyfreestatusfield(*whichstatus);
			*whichstatus = 0;
			us_redostatus(0);
			return;
		}
		if (namesamen(pp, "add", l) == 0)
		{
			if (count < 6)
			{
				us_abortcommand("Usage: window status-bar add FIELD LINE STARTPER ENDPER [TITLE]");
				return;
			}
			lineno = atoi(par[3]);
			if (lineno < 0 || lineno > ttynumstatuslines())
			{
				us_abortcommand("Line number must range from 0 to %d", ttynumstatuslines());
				return;
			}
			startper = atoi(par[4]);
			if (startper < 0 || startper > 100)
			{
				us_abortcommand("Starting percentage must range from 0 to 100");
				return;
			}
			endper = atoi(par[5]);
			if (endper <= startper || endper > 100)
			{
				us_abortcommand("Ending percentage must range from %d to 100", startper+1);
				return;
			}
			if (count == 7) fieldname = par[6]; else fieldname = "";
			if (*whichstatus != 0) ttyfreestatusfield(*whichstatus);
			*whichstatus = ttydeclarestatusfield(lineno, startper, endper, fieldname);
			us_redostatus(0);
			return;
		}
		us_abortcommand("Usage: window status-bar [add | delete]");
		return;
	}

	if (namesamen(pp, "tiny-facets", l) == 0 && l >= 2)
	{
		if (us_needwindow()) return;
		if (count >= 2)
		{
			l = strlen(pp = par[1]);
			if (namesamen(pp, "draw", l) == 0)
			{
				startobjectchange((INTBIG)us_aid, VAID);
				(void)setval((INTBIG)us_aid, VAID, "aidstate",
					us_aid->aidstate | DRAWTINYFACETS, VINTEGER);
				endobjectchange((INTBIG)us_aid, VAID);
			} else if (namesamen(pp, "hash-out", l) == 0)
			{
				startobjectchange((INTBIG)us_aid, VAID);
				(void)setval((INTBIG)us_aid, VAID, "aidstate",
					us_aid->aidstate & ~DRAWTINYFACETS, VINTEGER);
				endobjectchange((INTBIG)us_aid, VAID);
			} else
			{
				us_abortcommand("Usage: window tiny-facets draw|hash-out");
				return;
			}
		}
		if ((us_aid->aidstate&DRAWTINYFACETS) != 0)
			ttyputmsgf("Tiny facets will be drawn"); else
				ttyputmsgf("Tiny facets will be hashed-out");
		return;
	}

	if (namesamen(pp, "trace-displayed", l) == 0 && l >= 2)
	{
		np = us_needfacet();
		if (np == NONODEPROTO) return;

		/* cannot manipulate a nonstandard window */
		if ((el_curwindow->state&WINDOWTYPE) != DISPWINDOW)
		{
			us_abortcommand("Can only adjust circuit editing windows");
			return;
		}

		/* pre-compute current window size */
		curwx = el_curwindow->screenhx - el_curwindow->screenlx;
		curwy = el_curwindow->screenhy - el_curwindow->screenly;

		var = getval((INTBIG)us_aid, VAID, VINTEGER|VISARRAY, us_commandvarname('T'));
		if (var == NOVARIABLE)
		{
			us_abortcommand("Issue a trace before zooming into that area");
			return;
		}
		size = getlength(var) / 2;
		nogood = 0;
		for(i=0; i<size; i++)
		{
			x = ((INTBIG *)var->addr)[i*2];
			y = ((INTBIG *)var->addr)[i*2+1];
			nogood += us_setxy((INTSML)x, (INTSML)y);
			(void)getxy(&xcur, &ycur);
			if (i == 0)
			{
				lx = hx = xcur;   ly = hy = ycur;
			} else
			{
				lx = mini(lx, xcur);   hx = maxi(hx, xcur);
				ly = mini(ly, ycur);   hy = maxi(hy, ycur);
			}
		}
		if (nogood != 0)
		{
			us_abortcommand("Trace not inside window");
			return;
		}

		/* limit schematic windows to their normal extent */
		i = framesize(&schx, &schy, np);
		if (i == 0)
		{
			if (lx < -schx/2) lx = -schx/2;
			if (hx > schx/2) hx = schx/2;
			if (ly < -schy/2) ly = -schy/2;
			if (hy > schy/2) hy = schy/2;
		}

		/* save and erase highlighting */
		us_pushhighlight();
		us_clearhighlightcount();

		/* set the new window size */
		us_squarescreen(el_curwindow, NOWINDOW, 0, &lx, &hx, &ly, &hy);

		/* make sure squaring didn't push extent of schematic frame */
		if (i == 0)
		{
			if (lx < -schx/2)
			{
				hx -= schx/2 + lx;
				lx = -schx/2;
			}
			if (hx > schx/2)
			{
				lx -= hx - schx/2;
				hx = schx/2;
			}
			if (ly < -schy/2)
			{
				hy -= schy/2 + ly;
				ly = -schy/2;
			}
			if (hy > schy/2)
			{
				ly -= hy - schy/2;
				hy = schy/2;
			}
		}

		startobjectchange((INTBIG)el_curwindow, VWINDOW);
		(void)setval((INTBIG)el_curwindow, VWINDOW, "screenlx", lx, VINTEGER);
		(void)setval((INTBIG)el_curwindow, VWINDOW, "screenhx", hx, VINTEGER);
		(void)setval((INTBIG)el_curwindow, VWINDOW, "screenly", ly, VINTEGER);
		(void)setval((INTBIG)el_curwindow, VWINDOW, "screenhy", hy, VINTEGER);
		us_gridset(el_curwindow, el_curwindow->state);
		endobjectchange((INTBIG)el_curwindow, VWINDOW);

		/* restore highlighting */
		(void)us_pophighlight(0);
		return;
	}

	if (namesamen(pp, "up", l) == 0 && l >= 2)
	{
		if (us_needwindow()) return;

#if SIMAID
		/* special case for waveform window */
		if ((el_curwindow->state&WINDOWTYPE) == WAVEFORMWINDOW)
		{
			newpar[0] = "window";
			newpar[1] = "move";
			newpar[2] = "up";
			(void)tellaid(sim_aid, 3, newpar);
			return;
		}
#endif

		/* cannot pan a nonstandard window */
		if ((el_curwindow->state&WINDOWTYPE) != DISPWINDOW)
		{
			us_abortcommand("Can only pan circuit editing windows");
			return;
		}

		if (us_needfacet() == NONODEPROTO) return;
		if (count < 2) pp = "0.5"; else pp = par[1];
		if (pp[strlen(pp)-1] == 'l') dist = atola(pp); else
			dist = muldiv(atofr(pp), (el_curwindow->screenhy - el_curwindow->screenly), WHOLE);
		us_slideup(dist);
		return;
	}

	if (namesamen(pp, "use", l) == 0 && l >= 2)
	{
		if (count <= 1)
		{
			count = ttygetparam("Window to use: ", &us_windowup, MAXPARS-1, &par[1]) + 1;
			if (count == 1)
			{
				us_abortedmsg();
				return;
			}
		}
		for(w = el_topwindow; w != NOWINDOW; w = w->nextwindow)
			if (namesame(par[1], w->location) == 0) break;
		if (w == NOWINDOW)
		{
			us_abortcommand("No window named '%s'", par[1]);
			return;
		}
		(void)setvalkey((INTBIG)us_aid, VAID, us_current_window, (INTBIG)w, VWINDOW|VDONTSAVE);
		(void)setval((INTBIG)el_curlib, VLIBRARY, "curnodeproto", (INTBIG)w->curnodeproto, VNODEPROTO);
		return;
	}

	if (namesamen(pp, "zoom-scale", l) == 0 && l >= 1)
	{
		if (count >= 2)
		{
			l = strlen(pp = par[1]);
			if (namesamen(pp, "integral", l) == 0 && l >= 1)
				(void)setval((INTBIG)us_aid, VAID, "aidstate", us_aid->aidstate | INTEGRAL, VINTEGER);
			else if (namesamen(pp, "nonintegral", l) == 0 && l >= 1)
				(void)setval((INTBIG)us_aid, VAID, "aidstate", us_aid->aidstate & ~INTEGRAL, VINTEGER);
			else
			{
				us_abortcommand("Usage: window zoom-scale integral|nonintegral");
				return;
			}
		}
		if ((us_aid->aidstate&INTEGRAL) == 0)
			ttyputmsgf("Window scaling will be continuous"); else
				ttyputmsgf("Window scaling will force integral pixel alignment");
		return;
	}

	if (namesamen(pp, "1-window", l) == 0 && l >= 1)
	{
		if (us_needwindow()) return;

		if (strcmp(el_curwindow->location, "entire") == 0)
		{
			ttyputmsg("Already displaying only one window");
			return;
		}

		/* remember the current window */
		oldw = el_curwindow;

		/* turn off highlighting */
		us_pushhighlight();
		us_clearhighlightcount();
		(void)setvalkey((INTBIG)us_aid, VAID, us_current_window, (INTBIG)NOWINDOW, VWINDOW|VDONTSAVE);

		startobjectchange((INTBIG)us_aid, VAID);

		/* create a new window */
		neww = newewindow("entire", oldw, 0);
		if (neww == NOWINDOW)
		{
			ttyputerr("No memory for windows");
			return;
		}

		/* if reducing to an editor window, move the editor structure */
		if ((oldw->state&WINDOWTYPE) == TEXTWINDOW ||
			(oldw->state&WINDOWTYPE) == POPTEXTWINDOW)
		{
			(void)setval((INTBIG)neww, VWINDOW, "editor", (INTBIG)oldw->editor, VADDRESS);
			(void)setval((INTBIG)oldw, VWINDOW, "editor", -1, VADDRESS);
		}

		/* now delete all other windows */
		for(w = el_topwindow; w != NOWINDOW; w = nextw)
		{
			nextw = w->nextwindow;
			if (w != neww) killwindow(w);
		}

		/* set the window extents */
		us_windowfit(0, 0);
		us_squarescreen(neww, NOWINDOW, 1, &neww->screenlx, &neww->screenhx,
			&neww->screenly, &neww->screenhy);
		computewindowscale(neww);

		endobjectchange((INTBIG)us_aid, VAID);

		/* restore highlighting */
		(void)setvalkey((INTBIG)us_aid, VAID, us_current_window, (INTBIG)neww, VWINDOW|VDONTSAVE);
		(void)us_pophighlight(0);
		return;
	}

	us_abortcommand("Bad WINDOW option: %s", par[0]);
}

void us_yanknode(INTSML count, char *par[])
{
	REGISTER INTSML keepports, found, total;
	REGISTER NODEINST *topno;
	REGISTER GEOM **list;

	/* see if the "keep-ports" option is selected */
	if (count == 1 && namesamen(par[0], "keep-ports", (INTSML)(strlen(par[0]))) == 0)
		keepports = 1; else
			keepports = 0;

	/* disallow yanking if lock is on */
	if (us_protolocked(NONODEPROTO) != 0) return;

	list = us_gethighlighted(OBJNODEINST);
	if (list[0] == NOGEOM)
	{
		us_abortcommand("Must highlight facet(s) to be yanked");
		return;
	}

	found = 0;
	for(total=0; list[total] != NOGEOM; total++)
	{
		topno = list[total]->entryaddr.ni;
		if (topno->proto->index != 0) continue;

		/* turn off highlighting for the first facet */
		if (found == 0) us_clearhighlightcount();

		/* yank this facet */
		us_yankonenode(topno, keepports);
		found++;
	}

	if (found == 0) us_abortcommand("Can only yank facets");
}
