/*
 * Electric(tm) VLSI Design System
 *
 * File: usrdialogs.c
 * Graphics for dialog handling
 * Written by: Steven M. Rubin, Electric Editor Incorporated
 *
 * Copyright (c) 1998 Electric Editor Incorporated.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Electric Editor Incorporated
 * 23470 Sunset Drive, Suite 108
 * Los Gatos, California 95033
 * support@electriceditor.com
 */

#include "global.h"
#include "egraphics.h"
#include "usr.h"
#include "edialogs.h"
#include <ctype.h>

/******************************** DIALOG PACKAGE ********************************/

/*
 *    DIALOG CONTROL
 * DiaInitDialog(dia)                 initializes dialog from table "dia"
 * item = DiaNextHit()                obtains next "item" hit in dialog processing
 * chr = DiaGetNextCharacter(&item)   obtains next character (-1 if none, -2 if item)
 * DiaDoneDialog()                    removes and terminates dialog
 * DiaAllowSubDialog()                allows subdialog to be displayed
 * DiaRedrawDialogWindow()            redraws current dialog
 *
 *    CONTROL OF STANDARD DIALOG ITEMS
 * DiaSetText(item, msg)              puts "msg" in "item" in the dialog
 * line = DiaGetText(item)            gets text in "item" in the dialog
 * DiaSetControl(item, value)         puts "value" in "item" in the dialog
 * value = DiaGetControl(item)        gets integer in "item" in the dialog
 * valid = DiaValidEntry(item)        returns nonzero if "item" is nonblank in the dialog
 *
 *    SPECIAL DIALOG CONTROL
 * DiaDimItem(item)                   makes "item" dimmed in the dialog
 * DiaUnDimItem(item)                 makes "item" undimmed in the dialog
 * DiaNoEditControl(item)             makes "item" not editable in the dialog
 * DiaEditControl(item)               makes "item" editable in the dialog
 * DiaOpaqueEdit(item)                makes "item" not display text
 * DiaDefaultButton(item)             makes "item" default button
 * DiaCharacterEdit(item)             makes "item" report keystrokes
 *
 *    CONTROL OF POPUP ITEMS
 * DiaSetPopup(item, count, names)    makes "item" a popup with "count" entries in "names"
 * DiaSetPopupEntry(item, entry)      makes popup "item" use entry "entry" as current
 * entry = DiaGetPopupEntry(item)     gets current entry in popup "item"
 *
 *    CONTROL OF SCROLLABLE ITEMS
 * DiaInitTextDialog(item, top, next, done, sortpos, flags)
 * DiaLoadTextDialog(item, top, next, done, sortpos)
 * DiaStuffLine(item, msg)            add line "msg" to item "item"
 * DiaSelectLine(item, l)             select line "l" in item "item"
 * which = DiaGetCurLine(item)        get current line number in scroll item "item"
 * msg = DiaGetScrollLine(item, l)    get line "l" in scroll item "item"
 * DiaSetScrollLine(item, l, msg)     set line "l" in scroll item "item"
 *
 *    USER-CONTROL OF DIALOG ITEMS
 * DiaItemRect(item, r)               get rectangle for item "item"
 * DiaPercent(item, p)                fill item "item" with "p" percent progress
 * DiaRedispRoutine(item, routine)    call "routine" to redisplay item "item"
 * DiaFrameRect(r)                    draw frame in rectangle "r"
 * DiaDrawRect(r, on)                 fill rectangle "r" with "on"
 * DiaGetMouse(x, y)                  get current mouse position into "x/y"
 */

#ifdef	MACOS
#define	MENUSIZE   19
# include <Fonts.h>
# include <Dialogs.h>
# include <Resources.h>
# include <ToolUtils.h>
# ifdef USETK
#   include "tk.h"
#   include "tkMacInt.h"
# endif
#endif
#ifdef	WIN32
#  define WIN32MFC
#endif

INTSML     dia_allowanother = 1;

#ifdef WIN32MFC

/************************* WINDOWS *************************/

#define	MAXDIALOGS   2			/* maximum subdialogs on screen */

typedef struct
{
	DIALOG   *dlgresaddr;		/* address of this dialog */
	INTBIG    theDialog;
	short     defaultbutton;	/* default button */
} DIALOCAL;

DIALOCAL   dia_it, dia_save[MAXDIALOGS];
INTSML     dia_savepos = 0;

long gra_dodialoginit(DIALOG *dialog);
int gra_dodialognexthit(void);
void gra_dodialogdone(void);
void gra_dopopdialog(long dia, DIALOG *desc);
void gra_dodialogsettext(int item, int type, char *msg, int highlight);
char *gra_dodialoggettext(int item, int type);
void gra_dodialogsetcontrol(int item, int type, int value);
int gra_dodialoggetcontrol(int item, int type);
char *gra_dodialoggettext(int item, int type);
void gra_dodialogdimitem(int item, int enabled);
void gra_dodialogopaqueitem(int item);
void gra_dodialogsetdefault(int formeritem, int item);
void gra_dodialogsetpopup(int item, int count, char **names);
void gra_dodialogsetpopupentry(int item, int entry);
int gra_dodialoggetpopupentry(int item);
void gra_dodialoginitscroll(int item, int flags);
void gra_dodialogloadscroll(int item, int sort, short (*toplist)(char **),
	char *(*nextinlist)(void), void (*donelist)(void));
void gra_dodialogstuffline(int item, char *line);
void gra_dodialogselectline(int item, int line);
int gra_dodialoggetselectedline(int item);
char *gra_dodialoggetline(int item, int line);
void gra_dodialogsetline(int item, int line, char *msg);
void gra_dodialogredisproutine(int item, void (*routine)(RECTAREA*));
void gra_dodialogframerect(RECT *r);
void gra_dodialogdrawrect(RECT *r, int on);
void gra_dodialoggetmouse(int *x, int *y);
int gra_dodialognextchar(short *itemHit);
void gra_dodialogpercent(int item, int percent);
void gra_dodialogcharacteredit(int item);

/*************************** DIALOG CONTROL ***************************/

/*
 * Routine to initialize a dialog described by "dialog".
 * Returns nonzero if dialog cannot be initialized.
 */
INTSML DiaInitDialog(DIALOG *dialog)
{
	/* disallow nested dialogs */
	if (dia_allowanother == 0)
	{
		ttybeep();
		return(1);
	}
	dia_allowanother = 0;

	/* save the current dialog if this is a subdialog */
	if (dia_savepos > 0)
	{
		if (dia_savepos > MAXDIALOGS) return(1);
		dia_save[dia_savepos-1] = dia_it;
	}
	dia_savepos++;

	/* initialize dialog data structures */
	dia_it.defaultbutton = OK;
	dia_it.dlgresaddr = dialog;

	dia_it.theDialog = gra_dodialoginit(dialog);
	return(0);
}

void DiaAllowSubDialog(void)
{
	dia_allowanother = 1;
}

/*
 * Routine to handle actions and return the next item hit.
 */
INTSML DiaNextHit(void)
{
	int ret;

	ret = gra_dodialognexthit();
	return(ret);
}

/*
 * Routine to parse the next input event and return the next character typed.
 * If the routine returns -1, nothing has happened.  If the
 * routine returns -2, an item has been hit (and is in "itemHit").
 */
INTSML DiaGetNextCharacter(INTSML *itemHit)
{
	return(gra_dodialognextchar(itemHit));
}

void DiaDoneDialog(void)
{
	gra_dodialogdone();

	dia_allowanother = 1;
	dia_savepos--;
	if (dia_savepos <= 0) return;
	dia_it = dia_save[dia_savepos-1];
	gra_dopopdialog(dia_it.theDialog, dia_it.dlgresaddr);
}

/*************************** CONTROL OF STANDARD DIALOG ITEMS ***************************/

/*
 * Routine to set the text in item "item" to "msg"
 */
void DiaSetText(INTSML item, char *msg)
{
	int type, highlight;
	char *pt;

	highlight = 0;
	if (item < 0)
	{
		item = -item;
		highlight = 1;
	}
	type = dia_it.dlgresaddr->list[item-1].type;
	for(pt = msg; *pt != 0; pt++) if (strncmp(pt, "(c)", 3) == 0)
	{
		(void)strcpy(pt, "\251");		/* "copyright" character */
		(void)strcpy(&pt[1], &pt[3]);
		break;
	}
	gra_dodialogsettext(item-1, type&ITEMTYPE, msg, highlight);
}

/*
 * Routine to return the text in item "item"
 */
char *DiaGetText(INTSML item)
{
	int type;

	type = dia_it.dlgresaddr->list[item-1].type;
	return(gra_dodialoggettext(item-1, type&ITEMTYPE));
}

/*
 * Routine to set the value in item "item" to "value"
 */
void DiaSetControl(INTSML item, INTSML value)
{
	int type;

	type = dia_it.dlgresaddr->list[item-1].type;
	gra_dodialogsetcontrol(item-1, type&ITEMTYPE, value == 0 ? 0 : 1);
}

/*
 * Routine to return the value in item "item"
 */
INTSML DiaGetControl(INTSML item)
{
	int type;

	type = dia_it.dlgresaddr->list[item-1].type;
	return(gra_dodialoggetcontrol(item-1, type&ITEMTYPE) == 0 ? 0 : 1);
}

/*
 * Routine to check item "item" to make sure that there is
 * text in it.  If so, it returns nonzero.  Otherwise it beeps and returns zero.
 */
INTSML DiaValidEntry(INTSML item)
{
	int type;
	char *msg;

	type = dia_it.dlgresaddr->list[item-1].type;
	msg = gra_dodialoggettext(item-1, type&ITEMTYPE);
	while (*msg == ' ') msg++;
	if (*msg != 0) return(1);
	ttybeep();
	return(0);
}

/*************************** SPECIAL DIALOG CONTROL ***************************/

/*
 * Routine to dim item "item"
 */
void DiaDimItem(INTSML item)
{
	gra_dodialogdimitem(item-1, 0);
}

/*
 * Routine to un-dim item "item"
 */
void DiaUnDimItem(INTSML item)
{
	gra_dodialogdimitem(item-1, 1);
}

/*
 * Routine to change item "item" to be a message rather
 * than editable text
 */
void DiaNoEditControl(INTSML item)
{
	gra_dodialogdimitem(item-1, 0);
}

/*
 * Routine to change item "item" to be editable text rather
 * than a message
 */
void DiaEditControl(INTSML item)
{
	gra_dodialogdimitem(item-1, 1);
}

void DiaOpaqueEdit(INTSML item)
{
	gra_dodialogopaqueitem(item-1);
}

/*
 * Routine to change item "item" to be the default button
 */
void DiaDefaultButton(INTSML item)
{
	int formeritem;

	formeritem = dia_it.defaultbutton;
	dia_it.defaultbutton = item;
	gra_dodialogsetdefault(formeritem-1, item-1);
}

/*
 * Routine to cause item "item" to report character hits
 */
void DiaCharacterEdit(INTSML item)
{
	gra_dodialogcharacteredit(item-1);
}

/*************************** CONTROL OF POPUP ITEMS ***************************/

/*
 * Routine to change item "item" into a popup with "count" entries
 * in "names".
 */
void DiaSetPopup(INTSML item, INTSML count, char **names)
{
	gra_dodialogsetpopup(item-1, count, names);
}

/*
 * Routine to change popup item "item" so that the current entry is "entry".
 */
void DiaSetPopupEntry(INTSML item, INTSML entry)
{
	gra_dodialogsetpopupentry(item-1, entry);
}

/*
 * Routine to return the current item in popup menu item "item".
 */
INTSML DiaGetPopupEntry(INTSML item)
{
	return(gra_dodialoggetpopupentry(item-1));
}

/*************************** CONTROL OF SCROLLABLE ITEMS ***************************/

void DiaInitTextDialog(INTSML item, INTSML (*toplist)(char **), char *(*nextinlist)(void),
	void (*donelist)(void), INTSML sortpos, INTSML flags)
{
	gra_dodialoginitscroll(item-1, flags);
	gra_dodialogloadscroll(item-1, sortpos, toplist, nextinlist, donelist);
}

void DiaLoadTextDialog(INTSML item, INTSML (*toplist)(char **), char *(*nextinlist)(void),
	void (*donelist)(void), INTSML sortpos)
{
	gra_dodialogloadscroll(item-1, sortpos, toplist, nextinlist, donelist);
}

/*
 * Routine to stuff line "line" at the end of the edit buffer.
 */
void DiaStuffLine(INTSML item, char *line)
{
	gra_dodialogstuffline(item-1, line);
}

void DiaSelectLine(INTSML item, INTSML l)
{
	gra_dodialogselectline(item-1, l);
}

INTSML DiaGetCurLine(INTSML item)
{
	return(gra_dodialoggetselectedline(item-1));
}

char *DiaGetScrollLine(INTSML item, INTSML l)
{
	return(gra_dodialoggetline(item-1, l));
}

void DiaSetScrollLine(INTSML item, INTSML l, char *msg)
{
	gra_dodialogsetline(item-1, l, msg);
}

/*************************** USER-CONTROL OF DIALOG ITEMS ***************************/

void DiaItemRect(INTSML item, RECTAREA *rect)
{
	if (item <= 0 || item > dia_it.dlgresaddr->items) return;
	*rect = dia_it.dlgresaddr->list[item-1].r;
}

void DiaPercent(INTSML item, INTBIG p)
{
	gra_dodialogpercent(item-1, p);
}

void DiaRedispRoutine(INTSML item, void (*routine)(RECTAREA*))
{
	gra_dodialogredisproutine(item-1, routine);
}

void DiaFrameRect(RECTAREA *r)
{
	RECT mr;

	mr.left = r->left;   mr.right = r->right;
	mr.top = r->top;   mr.bottom = r->bottom;
	gra_dodialogframerect(&mr);
}

void DiaDrawRect(RECTAREA *r, INTSML on)
{
	RECT mr;

	mr.left = r->left;   mr.right = r->right;
	mr.top = r->top;   mr.bottom = r->bottom;
	gra_dodialogdrawrect(&mr, on);
}

void DiaGetMouse(INTSML *x, INTSML *y)
{
	int ix, iy;

	gra_dodialoggetmouse(&ix, &iy);
	*x = ix;   *y = iy;
}

#else

/************************* MAC AND UNIX *************************/

#define	TAB           011		/* the TAB character */
#define	DELETEKEY     010		/* the DELETE character */
#define	BS           0177		/* the BS character */
#define	LEFTARROWKEY  034		/* the Left Arrow key */
#define	RIGHTARROWKEY 035		/* the Right Arrow key */
#define	UPARROWKEY    036		/* the Up Arrow key */
#define	DOWNARROWKEY  037		/* the Down Arrow key */

/* the four scroller arrows */
#define	UPARROW      0
#define	DOWNARROW    1
#define	LEFTARROW    2
#define	RIGHTARROW   3

#define	THUMBSIZE   16			/* width of the thumb area in scroll slider */
#define	MAXSCROLLS   2			/* maximum scroll items in a dialog */
#define	MAXDIALOGS   2			/* maximum subdialogs on screen */

/* the scroll arrow definition */
#define	ARROWLEN     7
static INTSML dia_arrowx[] = {4, 10, 10, 13, 7, 1, 4};
static INTSML dia_arrowy[] = {12, 12, 8, 8, 2, 8, 8};

typedef struct
{
	INTSML  count;
	INTSML  current;
	char **namelist;
} POPUPDATA;

typedef struct
{
	INTSML   scrollitem;		/* item number of SCROLL area (-1 if none) */
	RECTAREA userrect;			/* position of SCROLL area */
	INTSML   flags;				/* state SCROLL area */
	INTSML   vthumbpos;			/* position of vertical thumb slider  */
	INTSML   hthumbpos;			/* position of horizontal thumb slider */
	INTSML   horizfactor;		/* shift of horizontal text (0 to 100) */
	INTSML   firstline;			/* line number of top line */
	INTSML   linesinfolder;		/* number of lines displayable */
	INTSML   which;				/* currently highlighted line */
	INTSML   lineheight;		/* height of line of text */
	INTSML   lineoffset;		/* offset to baseline for text */
	char   **scrolllist;		/* list of text lines */
	INTSML   scrolllistsize;	/* size of line list */
	INTSML   scrolllistlen;		/* number of valid lines/list */
} DSCROLL;

typedef struct
{
	DIALOG   *dlgresaddr;		/* address of this dialog */
	short     onscreen;			/* nonzero if displayed */
	short     defaultbutton;	/* default button */
#ifdef	MACOS
	WindowPtr theDialog;
# ifdef USETK
	Tk_Window tkwindow;
# else
	void     *tkwindow;
# endif
#else
	INTBIG    theDialog;
#endif

	/* for the scroll item */
	INTSML     scrollcount;			/* number of scroll items */
	INTSML     curscroll;			/* current scroll item */
	DSCROLL    scroll[MAXSCROLLS];	/* data structures for the scroll item(s) */

	/* for the current edit text item */
	INTSML     curitem;				/* current edit item */
	INTSML     editstart, editend;	/* start/end selected text in edit item */
	INTSML     firstch;				/* first displayed character in edit item */
	INTSML     opaqueitem;			/* item number of opaque edit text */
} DIALOCAL;

DIALOCAL   dia_it, dia_save[MAXDIALOGS];
INTSML     dia_active = 0;
INTSML     dia_savepos = 0;
INTSML     dia_slineheight;			/* height of a line of scroll text */
INTSML     dia_slineoffset;			/* scroll text: distance up to baseline */
INTSML     dia_lineheight;			/* height of a line of other text */
INTSML     dia_lineoffset;			/* other text: distance up to baseline */
INTSML     dia_curlineoffset;		/* current distance up to baseline */
INTSML     dia_knowndialogcount = 0;/* number of resource-read dialogs */
DIALOG   **dia_knowndialogs;		/* array of resource-read dialogs */
INTSML    *dia_knowdialognumbers;	/* array of resource-read dialog numbers */
INTSML     dia_firstupdate;
INTSML     dia_wasin, dia_lbase, dia_hbase, dia_offset, dia_ddata;
RECTAREA   dia_rect;				/* the current rectangle being tracked */
char      *dia_msg;
POPUPDATA *dia_pd;					/* the current popup menu */
INTSML     dia_cnt;					/* the current entry count in the popup menu */
INTSML     dia_sta;					/* the current start point in the popup menu */

/* prototypes for local routines */
INTSML Dbuttondown(INTBIG x, INTBIG y);
INTSML Dcheckdown(INTBIG x, INTBIG y);
void Dclear(void);
void Dcorrectxy(INTBIG *x, INTBIG *y);
void Ddonedialogwindow(void);
void Ddoneedit(void);
INTSML Ddownarrow(INTBIG x, INTBIG y);
INTSML Ddownpage(INTBIG x, INTBIG y);
void Ddragwindow(INTSML x, INTSML y);
void Ddrawarrow(INTSML sc, INTSML which, INTSML filled);
void Ddrawbox(INTBIG lx, INTBIG hx, INTBIG ly, INTBIG hy, GRAPHICS *which);
void Ddrawcircle(RECTAREA *r, INTSML dim);
void Ddrawdisc(RECTAREA *r);
void Ddrawhorizslider(INTSML sc);
void Ddrawitem(INTSML type, RECTAREA *r, char *msg, INTSML dim);
void Ddrawline(INTSML xf, INTSML yf, INTSML xt, INTSML yt);
void Ddrawmsg(INTSML sc, char *msg, INTSML which);
void Ddrawpolygon(INTBIG *xv, INTBIG *yv, INTSML count, INTSML filled);
void Ddrawpopupentries(void);
void Ddrawrectframe(RECTAREA *r, INTSML on, INTSML dim);
void Ddrawroundrectframe(RECTAREA *r, INTSML arc, INTSML dim);
void Ddrawtext(char *msg, INTSML len, INTSML x, INTSML y, INTSML dim);
void Ddrawvertslider(INTSML sc);
void Deditbox(RECTAREA *r, INTSML draw, INTSML dim);
INTSML Deditdown(INTBIG x, INTBIG y);
void Dforcedialog(void);
INTSML Dgeteditpos(RECTAREA *r, INTSML x, INTSML y, char *msg);
INTSML Dgettextsize(char *msg, INTSML len);
void Dgetwindowextent(INTSML *ly, INTSML *hy);
void Dgrayrect(RECTAREA *r);
INTSML Dhandlepopup(RECTAREA *r, POPUPDATA *pd);
void Dhighlight(INTSML on);
void Dhighlightrect(RECTAREA *r);
INTSML Dhscroll(INTBIG x, INTBIG y);
void Dinsertstr(char *insmsg);
void Dinsetrect(RECTAREA *r, INTSML amt);
void Dintdrawrect(RECTAREA *r, INTSML on);
void Dinvertbox(INTBIG lx, INTBIG hx, INTBIG ly, INTBIG hy);
void Dinvertentry(INTSML sc, INTSML on);
void Dinvertrectoff(RECTAREA *r);
void Dinvertrecton(RECTAREA *r);
void Dinvertrectframeoff(RECTAREA *r);
void Dinvertrectframeon(RECTAREA *r);
void Dinvertroundrectoff(RECTAREA *r);
void Dinvertroundrecton(RECTAREA *r);
INTSML Dleftarrow(INTBIG x, INTBIG y);
INTSML Dleftpage(INTBIG x, INTBIG y);
void Dnewdialogwindow(RECTAREA *r, char *movable);
void Dnewdialogwindowframe(RECTAREA *r, char *movable);
INTSML Dneweditbase(void);
INTSML Dnullchar(INTBIG x, INTBIG y, INTSML ch);
INTSML Dnullup(INTBIG x, INTBIG y);
void Dnullvoid(void);
void Dputicon(INTSML x, INTSML y, char *data);
void Dredrawscroll(INTSML sc);
void Drestorerect(INTBIG sr);
INTSML Drightarrow(INTBIG x, INTBIG y);
INTSML Drightpage(INTBIG x, INTBIG y);
INTBIG Dsaverect(RECTAREA *r);
void Dsethscroll(INTSML sc);
void Dsettextsmall(INTSML sc);
void Dsetvscroll(INTSML sc);
void Dshiftbits(RECTAREA *sr, RECTAREA *dr);
void Dstuffmessage(char *msg, RECTAREA *r, INTSML dim);
void Dstufftext(char *msg, RECTAREA *r);
void Dtextlocation(char *msg, INTSML len, RECTAREA *r, INTSML *wid, INTSML *line);
void Dtrackcursor(INTSML, INTSML, INTSML (*eachdown)(INTBIG, INTBIG));
INTSML Dtrackpopup(INTBIG x, INTBIG y);
INTSML Duparrow(INTBIG x, INTBIG y);
INTSML Duppage(INTBIG x, INTBIG y);
INTSML Dvscroll(INTBIG x, INTBIG y);
INTSML Dwaitforaction(INTSML *x, INTSML *y, INTSML *chr, UINTBIG *time);
INTSML Dwhichitem(INTSML x, INTSML y);

/*************************** DIALOG CONTROL ***************************/

/*
 * Routine to initialize a dialog described by "dialog"
 */
INTSML DiaInitDialog(DIALOG *dialog)
{
	INTSML itemtype, i;
	RECTAREA r;
	char *save, *line;
	POPUPDATA *pd;

	/* disallow nested dialogs */
	if (dia_allowanother == 0)
	{
		ttybeep();
		return(1);
	}
	dia_allowanother = 0;

	/* save the current dialog if this is a subdialog */
	if (dia_savepos > 0)
	{
		if (dia_savepos > MAXDIALOGS) return(1);
		dia_save[dia_savepos-1] = dia_it;
	}
	dia_savepos++;

	/* initialize dialog data structures */
	dia_it.defaultbutton = OK;
	dia_it.dlgresaddr = dialog;
	dia_it.curitem = 0;
	dia_it.opaqueitem = -1;
	for(i=0; i<MAXSCROLLS; i++)
	{
		dia_it.scroll[i].scrollitem = -1;
		dia_it.scroll[i].horizfactor = 0;
		dia_it.scroll[i].scrolllistlen = 0;
		dia_it.scroll[i].scrolllistsize = 0;
	}
	dia_it.curscroll = 0;
	dia_it.scrollcount = 0;
	dia_it.onscreen = 1;
	dia_it.firstch = 0;

	/* make the window */
	Dnewdialogwindow(&dialog->windowRect, dialog->movable);
	Dnewdialogwindowframe(&dialog->windowRect, dialog->movable);

	/* loop through all of the dialog entries, drawing them */
	for(i=0; i<dialog->items; i++)
	{
		/* draw the item */
		itemtype = dia_it.dlgresaddr->list[i].type;
		line = dia_it.dlgresaddr->list[i].msg;
		r = dia_it.dlgresaddr->list[i].r;

		if ((itemtype&ITEMTYPE) == EDITTEXT)
		{
			if (dia_it.curitem == 0) dia_it.curitem = i+1;
		}
		if ((itemtype&ITEMTYPE) == SCROLL)
		{
			if (dia_it.scrollcount < MAXSCROLLS)
				dia_it.scroll[dia_it.scrollcount++].scrollitem = i+1;
		}
		if ((itemtype&ITEMTYPE) == MESSAGE || (itemtype&ITEMTYPE) == EDITTEXT)
		{
			save = (char *)emalloc(strlen(line)+1, el_tempcluster);
			if (save == 0) return(1);
			(void)strcpy(save, line);
			dia_it.dlgresaddr->list[i].data = (INTBIG)save;
		} else if ((itemtype&ITEMTYPE) == POPUP)
		{
			pd = (POPUPDATA *)emalloc(sizeof (POPUPDATA), el_tempcluster);
			if (pd == 0) return(1);
			pd->count = 0;
			dia_it.dlgresaddr->list[i].data = (INTBIG)pd;
			line = (char *)pd;
		} else dia_it.dlgresaddr->list[i].data = 0;

		Ddrawitem(itemtype, &r, line, 0);

		/* highlight the default button */
		if (i == dia_it.defaultbutton-1 && itemtype == BUTTON) Dhighlightrect(&r);
	}
	if (dia_it.curitem != 0)
	{
		dia_it.editstart = 0;
		dia_it.editend = strlen(dia_it.dlgresaddr->list[dia_it.curitem-1].msg);
		Dhighlight(1);
	}
	dia_active++;
	dia_firstupdate = 1;
	return(0);
}

void DiaAllowSubDialog(void)
{
	dia_allowanother = 1;
}

/*
 * Routine to handle actions and return the next item hit.
 */
INTSML DiaNextHit(void)
{
	INTSML chr, itemHit;
	char ch[2];

	for(;;)
	{
		/* if interrupted, stop dialog */
		if (el_pleasestop != 0) return(CANCEL);

		/* get the next event, ignore fluff */
		chr = DiaGetNextCharacter(&itemHit);
		if (chr == -1) continue;

		/* non-character events return immediately */
		if (chr == -2) break;

		/* handle special character events */
		if (chr == 033) return(CANCEL);
		if (chr == '\n' || chr == '\r' || chr == 03) return(dia_it.defaultbutton);

		/* handle delete/backspace key */
		if ((chr == DELETEKEY || chr == BS) && dia_it.curitem != 0)
		{
			if (dia_it.editstart == dia_it.editend && dia_it.editstart > 0)
				dia_it.editstart--;
			chr = 0;
		}

		ch[0] = chr;   ch[1] = 0;
		Dinsertstr(ch);
		break;
	}
	return(itemHit);
}

void DiaDoneDialog(void)
{
	INTSML i, j, type;
	POPUPDATA *oldpd;

	/* free all the edit text and message buffers */
	for(i=0; i<dia_it.dlgresaddr->items; i++)
	{
		type = dia_it.dlgresaddr->list[i].type;
		if ((type&ITEMTYPE) == POPUP)
		{
			oldpd = (POPUPDATA *)dia_it.dlgresaddr->list[i].data;
			for(j=0; j<oldpd->count; j++) efree((char *)oldpd->namelist[j]);
			efree((char *)oldpd->namelist);
		}
		if ((type&ITEMTYPE) == MESSAGE || (type&ITEMTYPE) == EDITTEXT || (type&ITEMTYPE) == POPUP)
			efree((char *)dia_it.dlgresaddr->list[i].data);
	}

	/* free all items in the scroll area */
	for(i=0; i<dia_it.scrollcount; i++)
	{
		for(j=0; j<dia_it.scroll[i].scrolllistlen; j++)
			efree((char *)dia_it.scroll[i].scrolllist[j]);
		if (dia_it.scroll[i].scrolllistsize > 0) efree((char *)dia_it.scroll[i].scrolllist);
	}
	if (dia_it.onscreen != 0)
	{
		dia_it.onscreen = 0;
		Ddonedialogwindow();
		dia_active--;
	}

	dia_allowanother = 1;
	dia_savepos--;
	if (dia_savepos <= 0) return;
	dia_it = dia_save[dia_savepos-1];
	Dforcedialog();
	DiaRedrawDialogWindow();
}

/*************************** CONTROL OF STANDARD DIALOG ITEMS ***************************/

/*
 * Routine to set the text in item "item" to "msg"
 */
void DiaSetText(INTSML item, char *msg)
{
	INTSML type, high, oldcur;
	INTBIG amt;
	char *save;

	high = 0;
	if (item < 0) { item = -item;   high++; }
	if (item <= 0 || item > dia_it.dlgresaddr->items) return;
	oldcur = dia_it.curitem;   Ddoneedit();
	type = dia_it.dlgresaddr->list[item-1].type;
	if ((type&ITEMTYPE) == MESSAGE || (type&ITEMTYPE) == EDITTEXT)
	{
		if ((type&ITEMTYPE) == MESSAGE)
			Dstuffmessage(msg, &dia_it.dlgresaddr->list[item-1].r, 0); else
				Dstufftext(msg, &dia_it.dlgresaddr->list[item-1].r);
		amt = strlen(msg)+1;
		save = (char *)emalloc(amt, el_tempcluster);
		if (save == 0) return;
		(void)strcpy(save, msg);
		efree((char *)dia_it.dlgresaddr->list[item-1].data);
		dia_it.dlgresaddr->list[item-1].data = (INTBIG)save;
		if ((type&ITEMTYPE) == EDITTEXT)
		{
			if (high != 0)
			{
				Ddoneedit();
				oldcur = item;
				dia_it.editstart = 0;
				dia_it.editend = strlen(msg);
				dia_it.firstch = 0;
			} else if (oldcur == item)
			{
				dia_it.editstart = dia_it.editend = strlen(msg);
			}
		}
	}
	dia_it.curitem = oldcur;
	Dhighlight(1);
}

/*
 * Routine to return the text in item "item"
 */
char *DiaGetText(INTSML item)
{
	INTSML type;
	POPUPDATA *pd;

	if (item <= 0 || item > dia_it.dlgresaddr->items) return("");
	type = dia_it.dlgresaddr->list[item-1].type;
	if ((type&ITEMTYPE) == POPUP)
	{
		pd = (POPUPDATA *)dia_it.dlgresaddr->list[item-1].data;
		return(pd->namelist[pd->current]);
	}
	if ((type&ITEMTYPE) == MESSAGE || (type&ITEMTYPE) == EDITTEXT)
		return((char *)dia_it.dlgresaddr->list[item-1].data);
	return(0);
}

/*
 * Routine to set the value in item "item" to "value"
 */
void DiaSetControl(INTSML item, INTSML value)
{
	INTSML type;
	RECTAREA r;

	if (item <= 0 || item > dia_it.dlgresaddr->items) return;
	type = dia_it.dlgresaddr->list[item-1].type;
	r = dia_it.dlgresaddr->list[item-1].r;
	if ((type&ITEMTYPE) == CHECK)
	{
		/* check box */
		r.right = r.left + 12;
		r.top = (r.top + r.bottom) / 2 - 6;
		r.bottom = r.top + 12;
		Dintdrawrect(&r, 0);
		Ddrawrectframe(&r, 1, 0);
		if (value != 0)
		{
			Ddrawline(r.left, r.top, (INTSML)(r.right-1), (INTSML)(r.bottom-1));
			Ddrawline(r.left, (INTSML)(r.bottom-1), (INTSML)(r.right-1), r.top);
		}
		dia_it.dlgresaddr->list[item-1].data = (INTBIG)value;
	} else if ((type&ITEMTYPE) == RADIO)
	{
		/* radio button */
		r.right = r.left + 12;
		r.top = (r.top + r.bottom) / 2 - 6;
		r.bottom = r.top + 12;
		Dintdrawrect(&r, 0);
		Ddrawcircle(&r, 0);
		if (value != 0)
		{
			Dinsetrect(&r, 3);
			Ddrawdisc(&r);
		}
		dia_it.dlgresaddr->list[item-1].data = (INTBIG)value;
	}
}

/*
 * Routine to return the value in item "item"
 */
INTSML DiaGetControl(INTSML item)
{
	INTSML type;

	if (item <= 0 || item > dia_it.dlgresaddr->items) return(0);
	type = dia_it.dlgresaddr->list[item-1].type;
	if ((type&ITEMTYPE) == CHECK || (type&ITEMTYPE) == RADIO)
		return(dia_it.dlgresaddr->list[item-1].data);
	return(0);
}

/*
 * Routine to check item "item" to make sure that there is
 * text in it.  If so, it returns nonzero.  Otherwise it beeps and returns zero.
 */
INTSML DiaValidEntry(INTSML item)
{
	INTSML type;
	char *msg;

	if (item <= 0 || item > dia_it.dlgresaddr->items) return(0);
	type = dia_it.dlgresaddr->list[item-1].type;
	if ((type&ITEMTYPE) == MESSAGE || (type&ITEMTYPE) == EDITTEXT)
	{
		msg = (char *)dia_it.dlgresaddr->list[item-1].data;
		while (*msg == ' ') msg++;
		if (*msg != 0) return(1);
	}
	ttybeep();
	return(0);
}

/*************************** SPECIAL DIALOG CONTROL ***************************/

/*
 * Routine to dim item "item"
 */
void DiaDimItem(INTSML item)
{
	char *msg;
	INTSML type;
	RECTAREA r;

	if (item <= 0 || item > dia_it.dlgresaddr->items) return;
	if (item == dia_it.curitem) Ddoneedit();
	dia_it.dlgresaddr->list[item-1].type |= INACTIVE;
	type = dia_it.dlgresaddr->list[item-1].type;
	r = dia_it.dlgresaddr->list[item-1].r;
	msg = (char *)dia_it.dlgresaddr->list[item-1].data;
	if ((type&ITEMTYPE) == BUTTON || (type&ITEMTYPE) == CHECK ||
		(type&ITEMTYPE) == RADIO) msg = dia_it.dlgresaddr->list[item-1].msg;
	Ddrawitem(type, &r, msg, 1);
}

/*
 * Routine to un-dim item "item"
 */
void DiaUnDimItem(INTSML item)
{
	char *msg;
	INTSML type;
	RECTAREA r;

	if (item <= 0 || item > dia_it.dlgresaddr->items) return;
	dia_it.dlgresaddr->list[item-1].type &= ~INACTIVE;
	type = dia_it.dlgresaddr->list[item-1].type;
	r = dia_it.dlgresaddr->list[item-1].r;
	msg = (char *)dia_it.dlgresaddr->list[item-1].data;
	if ((type&ITEMTYPE) == BUTTON || (type&ITEMTYPE) == CHECK ||
		(type&ITEMTYPE) == RADIO) msg = dia_it.dlgresaddr->list[item-1].msg;
	Ddrawitem(type, &r, msg, 0);

	/* if undimming selected radio button, redraw disc */
	if ((type&ITEMTYPE) == RADIO && dia_it.dlgresaddr->list[item-1].data != 0)
	{
		r.right = r.left + 9;
		r.left += 3;
		r.top = (r.top + r.bottom) / 2 - 3;
		r.bottom = r.top + 6;
		Ddrawdisc(&r);
	}
}

/*
 * Routine to change item "item" to be a message rather
 * than editable text
 */
void DiaNoEditControl(INTSML item)
{
	if (item <= 0 || item > dia_it.dlgresaddr->items) return;
	if (item == dia_it.curitem) Ddoneedit();
	dia_it.dlgresaddr->list[item-1].type = MESSAGE;
	Deditbox(&dia_it.dlgresaddr->list[item-1].r, 0, 0);
}

/*
 * Routine to change item "item" to be editable text rather
 * than a message
 */
void DiaEditControl(INTSML item)
{
	if (item <= 0 || item > dia_it.dlgresaddr->items) return;
	dia_it.dlgresaddr->list[item-1].type = EDITTEXT;
	Deditbox(&dia_it.dlgresaddr->list[item-1].r, 1, 0);
}

void DiaOpaqueEdit(INTSML item)
{
	if (item <= 0 || item > dia_it.dlgresaddr->items) return;
	if (dia_it.dlgresaddr->list[item-1].type != EDITTEXT) return;
	dia_it.opaqueitem = item;
}

/*
 * Routine to change item "item" to be the default button
 */
void DiaDefaultButton(INTSML item)
{
	INTSML itemtype, formeritem, dim;
	RECTAREA r;
	char *line;

	if (item <= 0 || item > dia_it.dlgresaddr->items) return;
	itemtype = dia_it.dlgresaddr->list[item-1].type;
	if ((itemtype&ITEMTYPE) != BUTTON) return;

	/* turn off former default button */
	formeritem = dia_it.defaultbutton;
	itemtype = dia_it.dlgresaddr->list[formeritem-1].type;
	r = dia_it.dlgresaddr->list[formeritem-1].r;
	line = dia_it.dlgresaddr->list[formeritem-1].msg;
	if ((itemtype&INACTIVE) != 0) dim = 1; else dim = 0;
	Dinsetrect(&r, -4);
	Dintdrawrect(&r, 0);
	Dinsetrect(&r, 4);
	Ddrawitem(itemtype, &r, line, dim);

	/* draw new default button */
	dia_it.defaultbutton = item;
	itemtype = dia_it.dlgresaddr->list[item-1].type;
	r = dia_it.dlgresaddr->list[item-1].r;
	line = dia_it.dlgresaddr->list[item-1].msg;
	if ((itemtype&INACTIVE) != 0) dim = 1; else dim = 0;
	Ddrawitem(itemtype, &r, line, dim);
	Dhighlightrect(&r);
}

/*
 * Routine to cause item "item" to report character hits
 */
void DiaCharacterEdit(INTSML item)
{
}

/*************************** CONTROL OF POPUP ITEMS ***************************/

/*
 * Routine to change item "item" into a popup with "count" entries
 * in "names".
 */
void DiaSetPopup(INTSML item, INTSML count, char **names)
{
	POPUPDATA *pd;
	INTSML i, type;

	if (item <= 0 || item > dia_it.dlgresaddr->items) return;
	type = dia_it.dlgresaddr->list[item-1].type;
	if ((type&ITEMTYPE) != POPUP) return;

	/* copy into a POPUPDATA structure */
	pd = (POPUPDATA *)dia_it.dlgresaddr->list[item-1].data;
	for(i=0; i<pd->count; i++) efree((char *)pd->namelist[i]);
	if (pd->count > 0) efree((char *)pd->namelist);
	pd->count = count;
	pd->current = 0;
	pd->namelist = (char **)emalloc(count * (sizeof (char *)), el_tempcluster);
	if (pd->namelist == 0) return;
	for(i=0; i<count; i++)
	{
		pd->namelist[i] = (char *)emalloc((strlen(names[i])+1) * (sizeof (char)),
			el_tempcluster);
		if (pd->namelist[i] == 0) return;
		(void)strcpy(pd->namelist[i], names[i]);
	}

	/* display the popup */
	Ddrawitem(POPUP, &dia_it.dlgresaddr->list[item-1].r, (char *)pd, 0);
}

/*
 * Routine to change popup item "item" so that the current entry is "entry".
 */
void DiaSetPopupEntry(INTSML item, INTSML entry)
{
	POPUPDATA *pd;
	INTSML type;

	if (item <= 0 || item > dia_it.dlgresaddr->items) return;
	type = dia_it.dlgresaddr->list[item-1].type;
	if ((type&ITEMTYPE) != POPUP) return;
	pd = (POPUPDATA *)dia_it.dlgresaddr->list[item-1].data;
	if (entry < 0 || entry >= pd->count) return;
	pd->current = entry;

	Ddrawitem(POPUP, &dia_it.dlgresaddr->list[item-1].r, (char *)pd,
		(INTSML)(type&INACTIVE));
}

/*
 * Routine to return the current item in popup menu item "item".
 */
INTSML DiaGetPopupEntry(INTSML item)
{
	POPUPDATA *pd;

	if (item <= 0 || item > dia_it.dlgresaddr->items) return(0);
	if ((dia_it.dlgresaddr->list[item-1].type&ITEMTYPE) != POPUP) return(0);
	pd = (POPUPDATA *)dia_it.dlgresaddr->list[item-1].data;
	return(pd->current);
}

/*************************** CONTROL OF SCROLLABLE ITEMS ***************************/

void DiaInitTextDialog(INTSML item, INTSML (*toplist)(char **), char *(*nextinlist)(void),
	void (*donelist)(void), INTSML sortpos, INTSML flags)
{
	DSCROLL *scr;
	INTSML sc;

	if (item <= 0 || item > dia_it.dlgresaddr->items) return;
	for(sc=0; sc<dia_it.scrollcount; sc++) if (dia_it.scroll[sc].scrollitem == item) break;
	if (sc >= dia_it.scrollcount) return;
	scr = &dia_it.scroll[sc];

	/* save information about this scroll area */
	scr->flags = flags;
	if ((scr->flags&SCSMALLFONT) != 0)
	{
		scr->lineheight = dia_slineheight;
		scr->lineoffset = dia_slineoffset;
	} else
	{
		scr->lineheight = dia_lineheight;
		scr->lineoffset = dia_lineoffset;
	}

	/* compute size of actual area (not including scroll bars) */
	scr->userrect = dia_it.dlgresaddr->list[item-1].r;
	scr->userrect.right -= 14;
	if ((scr->flags&SCHORIZBAR) != 0) scr->userrect.bottom -= 14;
	scr->linesinfolder = (scr->userrect.bottom - scr->userrect.top) / scr->lineheight;

	/* draw sliders */
	Ddrawvertslider(sc);
	if ((scr->flags&SCHORIZBAR) != 0) Ddrawhorizslider(sc);

	/* load the text */
	scr->scrolllistlen = 0;
	DiaLoadTextDialog(item, toplist, nextinlist, donelist, sortpos);
}

void DiaLoadTextDialog(INTSML item, INTSML (*toplist)(char **), char *(*nextinlist)(void),
	void (*donelist)(void), INTSML sortpos)
{
	INTSML i, j, sorted, items, *order, sc;
	char *next, **list, line[256];
	DSCROLL *scr;

	if (item <= 0 || item > dia_it.dlgresaddr->items) return;
	for(sc=0; sc<dia_it.scrollcount; sc++) if (dia_it.scroll[sc].scrollitem == item) break;
	if (sc >= dia_it.scrollcount) return;
	scr = &dia_it.scroll[sc];

	/* deallocate all former items in the list */
	for(i=0; i<scr->scrolllistlen; i++) efree((char *)scr->scrolllist[i]);
	scr->scrolllistlen = 0;
	scr->firstline = 0;

	/* count the number of items to be put in the text editor */
	line[0] = 0;
	next = line;
	(void)(*toplist)(&next);
	for(items=0; ; items++) if ((*nextinlist)() == 0) break;
	(*donelist)();

	/* allocate space for the strings */
	if (items > 0)
	{
		list = (char **)emalloc(items * (sizeof (char *)), el_tempcluster);
		if (list == 0) return;
		order = (INTSML *)emalloc(items * (sizeof (INTSML)), el_tempcluster);
		if (order == 0) return;
	}

	/* get the list */
	line[0] = 0;
	next = line;
	(void)(*toplist)(&next);
	for(i=0; i<items; i++)
	{
		next = (*nextinlist)();
		if (next == 0) next = "???";
		list[i] = (char *)emalloc(strlen(next)+1, el_tempcluster);
		if (list[i] == 0) return;
		strcpy(list[i], next);
		order[i] = i;
	}
	(*donelist)();

	/* sort the list */
	if (sortpos >= 0)
	{
		sorted = 0;
		while (sorted == 0)
		{
			sorted = 1;
			for(i=1; i<items; i++)
			{
				if (namesame(&list[order[i-1]][sortpos], &list[order[i]][sortpos]) <= 0)
					continue;
				j = order[i];   order[i] = order[i-1];   order[i-1] = j;
				sorted = 0;
			}
		}
	}

	/* stuff the list into the text editor */
	scr->which = -1;
	Dinsetrect(&scr->userrect, 1);
	Dintdrawrect(&scr->userrect, 0);
	Dinsetrect(&scr->userrect, -1);
	for(i=0; i<items; i++) DiaStuffLine(item, list[order[i]]);
	Dsetvscroll(sc);
	if (scr->scrolllistlen > 0) DiaSelectLine(item, 0);

	/* deallocate the list */
	if (items > 0)
	{
		for(i=0; i<items; i++) efree((char *)list[i]);
		efree((char *)(char *)list);
		efree((char *)(char *)order);
	}
}

/*
 * Routine to stuff line "line" at the end of the edit buffer.
 */
void DiaStuffLine(INTSML item, char *line)
{
	char **newlist, *pt;
	INTSML i, sc;
	DSCROLL *scr;

	if (item <= 0 || item > dia_it.dlgresaddr->items) return;
	for(sc=0; sc<dia_it.scrollcount; sc++) if (dia_it.scroll[sc].scrollitem == item) break;
	if (sc >= dia_it.scrollcount) return;
	scr = &dia_it.scroll[sc];

	if (scr->scrolllistlen >= scr->scrolllistsize)
	{
		newlist = (char **)emalloc((scr->scrolllistsize+10) * (sizeof (char *)),
			el_tempcluster);
		if (newlist == 0) return;
		for(i=0; i<scr->scrolllistlen; i++) newlist[i] = scr->scrolllist[i];
		if (scr->scrolllistsize != 0) efree((char *)scr->scrolllist);
		scr->scrolllist = newlist;
		scr->scrolllistsize += 10;
	}
	pt = (char *)emalloc(strlen(line)+1, el_tempcluster);
	if (pt == 0) return;
	(void)strcpy(pt, line);
	if (scr->scrolllistlen < scr->firstline+scr->linesinfolder)
		Ddrawmsg(sc, line, scr->scrolllistlen);
	scr->scrolllist[scr->scrolllistlen++] = pt;
}

void DiaSelectLine(INTSML item, INTSML l)
{
	INTSML n, sc;
	DSCROLL *scr;

	if (item <= 0 || item > dia_it.dlgresaddr->items) return;
	for(sc=0; sc<dia_it.scrollcount; sc++) if (dia_it.scroll[sc].scrollitem == item) break;
	if (sc >= dia_it.scrollcount) return;
	scr = &dia_it.scroll[sc];

	/* if the new line is visible, simply shift the highlighting */
	if (l-scr->firstline >= 0 && l-scr->firstline < scr->linesinfolder)
	{
		Dinvertentry(sc, 0);
	} else
	{
		/* must shift the buffer */
		n = l - scr->linesinfolder/2;
		if (n > scr->scrolllistlen-scr->linesinfolder)
			n = scr->scrolllistlen-scr->linesinfolder;
		if (n < 0) n = 0;
		scr->firstline = n;
		Dredrawscroll(sc);
	}
	scr->which = l;
	Dinvertentry(sc, 1);
	Dsetvscroll(sc);
}

INTSML DiaGetCurLine(INTSML item)
{
	INTSML sc;

	if (item <= 0 || item > dia_it.dlgresaddr->items) return(-1);
	for(sc=0; sc<dia_it.scrollcount; sc++)
		if (dia_it.scroll[sc].scrollitem == item) return(dia_it.scroll[sc].which);
	return(-1);
}

char *DiaGetScrollLine(INTSML item, INTSML l)
{
	INTSML sc;

	if (item <= 0 || item > dia_it.dlgresaddr->items) return("");
	for(sc=0; sc<dia_it.scrollcount; sc++)
		if (dia_it.scroll[sc].scrollitem == item)
	{
		if (l < 0 || l >= dia_it.scroll[sc].scrolllistlen) break;
		return(dia_it.scroll[sc].scrolllist[l]);
	}
	return("");
}

void DiaSetScrollLine(INTSML item, INTSML l, char *msg)
{
	char *ins;
	INTSML sc;
	DSCROLL *scr;

	if (item <= 0 || item > dia_it.dlgresaddr->items) return;
	for(sc=0; sc<dia_it.scrollcount; sc++) if (dia_it.scroll[sc].scrollitem == item) break;
	if (sc >= dia_it.scrollcount) return;
	scr = &dia_it.scroll[sc];

	if (l >= scr->scrolllistlen) DiaStuffLine(item, msg); else
	{
		ins = (char *)emalloc(strlen(msg)+1, el_tempcluster);
		if (ins == 0) return;
		(void)strcpy(ins, msg);
		efree((char *)scr->scrolllist[l]);
		scr->scrolllist[l] = ins;
	}
	Dredrawscroll(sc);
	Dinvertentry(sc, 1);
	Dsetvscroll(sc);
}

/*************************** USER-CONTROL OF DIALOG ITEMS ***************************/

void DiaItemRect(INTSML item, RECTAREA *rect)
{
	if (item <= 0 || item > dia_it.dlgresaddr->items) return;
	*rect = dia_it.dlgresaddr->list[item-1].r;
	Dforcedialog();
}

void DiaPercent(INTSML item, INTBIG p)
{
	RECTAREA r;
	INTSML type;

	if (item <= 0 || item > dia_it.dlgresaddr->items) return;
	type = dia_it.dlgresaddr->list[item-1].type;
	if ((type&ITEMTYPE) != PROGRESS) return;
	r = dia_it.dlgresaddr->list[item-1].r;
	if (p == 0)
	{
		Dintdrawrect(&r, 0);
		Ddrawrectframe(&r, 1, 0);
		return;
	}
	r.left++;
	r.right = r.left + (r.right-1-r.left)*p/100;
	r.top++;   r.bottom--;
	Dgrayrect(&r);
	flushscreen();
}

void DiaRedispRoutine(INTSML item, void (*routine)(RECTAREA*))
{
	INTSML type;

	if (item <= 0 || item > dia_it.dlgresaddr->items) return;
	type = dia_it.dlgresaddr->list[item-1].type;
	if ((type&ITEMTYPE) != USERDRAWN) return;
	dia_it.dlgresaddr->list[item-1].data = (INTBIG)routine;
}

void DiaFrameRect(RECTAREA *r)
{
	Dintdrawrect(r, 0);
	Ddrawrectframe(r, 1, 0);
}

void DiaDrawRect(RECTAREA *r, INTSML on)
{
	Dintdrawrect(r, on);
}

void DiaGetMouse(INTSML *x, INTSML *y)
{
	INTBIG lx, ly;
	readtablet(x, y);
	lx = *x;   ly = *y;
	Dcorrectxy(&lx, &ly);
	*x = lx;   *y = ly;
}

/************************* CONTROL OF SUB-DIALOGS *************************/

typedef void (*USERTYPE)(RECTAREA*);

void DiaRedrawDialogWindow(void)
{
	INTSML i, itemtype, dim;
	char *line;
	RECTAREA r;
	USERTYPE routine;

	if (dia_active <= 0) return;
	Dclear();
#ifdef	MACOS
	/* do not handle the first macintosh update event */
	if (dia_firstupdate != 0)
	{
		dia_firstupdate = 0;
		return;
	}
#endif

	/* redraw the window frame */
	Dsettextsmall(-1);
	Dnewdialogwindowframe(&dia_it.dlgresaddr->windowRect, dia_it.dlgresaddr->movable);

	for(i=0; i<dia_it.dlgresaddr->items; i++)
	{
		/* draw the item */
		itemtype = dia_it.dlgresaddr->list[i].type;
		r = dia_it.dlgresaddr->list[i].r;
		if ((itemtype&ITEMTYPE) == USERDRAWN)
		{
			routine = (USERTYPE)dia_it.dlgresaddr->list[i].data;
			if (routine != 0) (*routine)(&r);
			continue;
		}
		if ((itemtype&ITEMTYPE) == MESSAGE || (itemtype&ITEMTYPE) == EDITTEXT || (itemtype&ITEMTYPE) == POPUP)
			line = (char *)dia_it.dlgresaddr->list[i].data; else
				line = dia_it.dlgresaddr->list[i].msg;
		if ((itemtype&ITEMTYPE) == CHECK || (itemtype&ITEMTYPE) == RADIO)
			DiaSetControl((INTSML)(i+1), (INTSML)dia_it.dlgresaddr->list[i].data);
		if ((itemtype&INACTIVE) != 0) dim = 1; else dim = 0;
		Ddrawitem(itemtype, &r, line, dim);

		/* highlight the default button */
		if (i == dia_it.defaultbutton-1 && itemtype == BUTTON) Dhighlightrect(&r);
	}

	/* turn on anything being edited */
	if (dia_it.curitem != 0) Dhighlight(1);
	for(i=0; i<dia_it.scrollcount; i++)
	{
		Dredrawscroll(i);
		Dinvertentry(i, 1);      /* restores previous inverted entry */
		Dsetvscroll(i);
		if ((dia_it.scroll[i].flags&SCHORIZBAR) != 0)
		{
			Ddrawhorizslider(i);
			Dsethscroll(i);
		}
	}
}

/************************* INTERNAL ROUTINES *************************/

INTSML Dbuttondown(INTBIG x, INTBIG y)
{
	INTSML in;

	Dcorrectxy(&x, &y);
	if (x < dia_rect.left || x > dia_rect.right || y < dia_rect.top || y > dia_rect.bottom)
		in = 0; else
			in = 1;
	if (in != dia_wasin)
	{
		if (in == 0) Dinvertroundrectoff(&dia_rect); else
			Dinvertroundrecton(&dia_rect);
	}
	dia_wasin = in;
	return(0);
}

INTSML Dtrackpopup(INTBIG x, INTBIG y)
{
	INTSML which;
	RECTAREA r;

	Dcorrectxy(&x, &y);
	if (x < dia_rect.left || x > dia_rect.right || y < dia_rect.top || y > dia_rect.bottom)
		which = -1; else
	{
		which = (y - dia_rect.top - 1) / dia_lineheight;
		if (which >= dia_cnt-dia_sta) which = dia_cnt-dia_sta-1;

	}
	if (which != dia_wasin)
	{
		r.left = dia_rect.left + 1;
		r.right = dia_rect.right - 1;
		if (dia_wasin >= 0)
		{
			r.top = dia_rect.top + (dia_wasin * dia_lineheight) + 1;
			r.bottom = r.top + dia_lineheight;
			Dinvertrectoff(&r);
		}

		/* special case if in the "up" arrow */
		if (which == 0 && dia_sta != 0)
		{
			dia_sta--;   dia_cnt--;
			Ddrawpopupentries();
			dia_wasin = -1;
			return(0);
		}

		/* special case if in the "down" arrow */
		if (which == dia_cnt-dia_sta-1 && dia_cnt != dia_pd->count)
		{
			dia_sta++;   dia_cnt++;
			Ddrawpopupentries();
			dia_wasin = -1;
			return(0);
		}

		if (which >= 0)
		{
			r.top = dia_rect.top + (which * dia_lineheight) + 1;
			r.bottom = r.top + dia_lineheight;
			Dinvertrecton(&r);
		}

		dia_wasin = which;
	}
	return(0);
}

INTSML Dcheckdown(INTBIG x, INTBIG y)
{
	INTSML in;
	RECTAREA r;

	Dcorrectxy(&x, &y);
	if (x < dia_rect.left || x > dia_rect.right || y < dia_rect.top || y > dia_rect.bottom)
		in = 0; else
			in = 1;
	if (in != dia_wasin)
	{
		r = dia_rect;
		r.right = r.left + 11;   r.left++;
		r.top = (r.top + r.bottom) / 2 - 5;
		r.bottom = r.top + 10;
		if (in != 0) Ddrawrectframe(&r, 1, 0); else
		{
			Ddrawrectframe(&r, 0, 0);
			if (dia_ddata != 0)
			{
				Ddrawline(r.left, r.top, (INTSML)(r.right-1), (INTSML)(r.bottom-1));
				Ddrawline(r.left, (INTSML)(r.bottom-1), (INTSML)(r.right-1), r.top);
			}
		}
	}
	dia_wasin = in;
	return(0);
}

INTSML Deditdown(INTBIG x, INTBIG y)
{
	INTSML l, h, pos, wid, len, prevpos, basepos;
	char *msg;

	Dcorrectxy(&x, &y);

	/* shift the text if the cursor moves outside the field */
	if (dia_rect.bottom-dia_rect.top < dia_lineheight*2)
	{
		if (y < dia_rect.top || y > dia_rect.bottom) return(0);

		/* scroll horizontally if there is only 1 line in the edit field */
		if (x < dia_rect.left && dia_it.firstch > 0)
		{
			Dhighlight(0);
			dia_it.firstch--;
			Dstufftext(dia_msg, &dia_rect);
			Dhighlight(1);
			gotosleep(10);
			return(0);
		}
		if (x > dia_rect.right && dia_it.firstch < dia_it.editend-1)
		{
			Dhighlight(0);
			dia_it.firstch++;
			Dstufftext(dia_msg, &dia_rect);
			Dhighlight(1);
			gotosleep(10);
			return(0);
		}
	} else
	{
		if (x < dia_rect.left || x > dia_rect.right) return(0);

		/* scroll vertically if there are multiple lines in the field */
		if (y < dia_rect.top && dia_it.firstch > 0)
		{
			msg = dia_msg;
			prevpos = 0;
			basepos = 0;
			while (*msg != 0)
			{
				for(len = strlen(msg); len>0; len--)
				{
					wid = Dgettextsize(msg, len);
					if (wid < dia_rect.right-dia_rect.left-2) break;
				}
				if (len == 0) break;
				basepos += len;
				if (basepos == dia_it.firstch) break;
				prevpos += len;
				msg += len;
			}
			Dhighlight(0);
			dia_it.firstch = prevpos;
			Dstufftext(dia_msg, &dia_rect);
			Dhighlight(1);
			gotosleep(30);
			return(0);
		}
		if (y > dia_rect.bottom && dia_it.firstch < dia_it.editend-1)
		{
			msg = &dia_msg[dia_it.firstch];
			for(len = strlen(msg); len>0; len--)
			{
				wid = Dgettextsize(msg, len);
				if (wid < dia_rect.right-dia_rect.left-2) break;
			}
			if (len == strlen(msg)) return(0);
			Dhighlight(0);
			dia_it.firstch += len;
			Dstufftext(dia_msg, &dia_rect);
			Dhighlight(1);
			gotosleep(30);
			return(0);
		}
	}

	pos = Dgeteditpos(&dia_rect, (INTSML)x, (INTSML)y, dia_msg);
	l = dia_lbase;   h = dia_hbase;
	if (pos > h) h = pos;
	if (pos < l) l = pos;
	if (l != dia_it.editstart || h != dia_it.editend)
	{
		Dhighlight(0);
		dia_it.editstart = l;   dia_it.editend = h;
		Dhighlight(1);
	}
	return(0);
}

INTSML Ddownarrow(INTBIG x, INTBIG y)
{
	short sc;
	DSCROLL *scr;
#ifdef MACOS
	short i;
	RECTAREA r, fr, tr;
#endif

	scr = &dia_it.scroll[sc=dia_it.curscroll];
	if (scr->scrolllistlen-scr->firstline <= scr->linesinfolder) return(1);
	scr->firstline++;
#ifndef	MACOS
	Dredrawscroll(sc);
	Dinvertentry(sc, 1);
	Dsetvscroll(sc);
#else
	/* shift the window contents up by 1 line (scr->lineheight) */
	r = scr->userrect;
	fr.left = r.left+1;                 fr.right = r.right-1;
	fr.top = r.top+1+scr->lineheight;   fr.bottom = r.bottom-1;
	tr.left = r.left+1;   tr.right = r.right-1;
	tr.top = r.top+1;     tr.bottom = r.bottom-1-scr->lineheight;
	Dshiftbits(&fr, &tr);

	/* fill in a new last line */
	fr.top = r.bottom-1-scr->lineheight;   fr.bottom = r.bottom-1;
	Dintdrawrect(&fr, 0);
	i = scr->firstline + scr->linesinfolder - 1;
	if (i < scr->scrolllistlen)
		Ddrawmsg(sc, scr->scrolllist[i], i-scr->firstline);

	/* invert it if selected */
	if (i == scr->which) Dinvertentry(sc, 1);

	/* redo thumb position */
	Dsetvscroll(sc);
#endif
	return(0);
}

INTSML Duparrow(INTBIG x, INTBIG y)
{
	INTSML sc;
	DSCROLL *scr;
#ifdef MACOS
	RECTAREA r, fr, tr;
#endif

	scr = &dia_it.scroll[sc=dia_it.curscroll];
	if (scr->firstline <= 0) return(1);
	scr->firstline--;
#ifndef	MACOS
	Dredrawscroll(sc);
	Dinvertentry(sc, 1);
	Dsetvscroll(sc);
#else
	/* shift the window contents down by 1 line (scr->lineheight) */
	r = scr->userrect;
	fr.left = r.left+1;   fr.right = r.right-1;
	fr.top = r.top+1;     fr.bottom = r.bottom-1-scr->lineheight;

	tr.left = r.left+1;                 tr.right = r.right-1;
	tr.top = r.top+1+scr->lineheight;   tr.bottom = r.bottom-1;
	Dshiftbits(&fr, &tr);

	/* fill in a new top line */
	fr.top = r.top+1;   fr.bottom = r.top+1+scr->lineheight;
	Dintdrawrect(&fr, 0);
	Ddrawmsg(sc, scr->scrolllist[scr->firstline], 0);

	/* invert it if selected */
	if (scr->firstline == scr->which) Dinvertentry(sc, 1);

	/* redo thumb position */
	Dsetvscroll(sc);
#endif
	return(0);
}

INTSML Ddownpage(INTBIG x, INTBIG y)
{
	INTSML sc;
	DSCROLL *scr;

	scr = &dia_it.scroll[sc=dia_it.curscroll];
	Dcorrectxy(&x, &y);
	if (y <= scr->vthumbpos+THUMBSIZE/2 || y > scr->userrect.bottom-16) return(1);
	if (scr->scrolllistlen-scr->firstline <= scr->linesinfolder) return(1);
	scr->firstline += scr->linesinfolder-1;
	if (scr->scrolllistlen-scr->firstline <= scr->linesinfolder)
		scr->firstline = scr->scrolllistlen-scr->linesinfolder;
	Dredrawscroll(sc);
	Dinvertentry(sc, 1);
	Dsetvscroll(sc);
	return(0);
}

INTSML Duppage(INTBIG x, INTBIG y)
{
	INTSML sc;
	DSCROLL *scr;

	scr = &dia_it.scroll[sc=dia_it.curscroll];
	Dcorrectxy(&x, &y);
	if (y >= scr->vthumbpos-THUMBSIZE/2 || y < scr->userrect.top+16) return(1);
	if (scr->firstline <= 0) return(1);
	scr->firstline -= scr->linesinfolder-1;
	if (scr->firstline < 0) scr->firstline = 0;
	Dredrawscroll(sc);
	Dinvertentry(sc, 1);
	Dsetvscroll(sc);
	return(0);
}

INTSML Dvscroll(INTBIG x, INTBIG y)
{
	INTSML l, sc;
	DSCROLL *scr;

	scr = &dia_it.scroll[sc=dia_it.curscroll];
	Dcorrectxy(&x, &y);
	l = scr->vthumbpos;
	scr->vthumbpos = y + dia_offset;
	if (scr->vthumbpos < scr->userrect.top+16+THUMBSIZE/2)
		scr->vthumbpos = scr->userrect.top+16+THUMBSIZE/2;
	if (scr->vthumbpos > scr->userrect.bottom-16-THUMBSIZE/2)
		scr->vthumbpos = scr->userrect.bottom-16-THUMBSIZE/2;
	if (scr->vthumbpos == l) return(0);
	dia_rect.top = l - THUMBSIZE/2;
	dia_rect.bottom = l + THUMBSIZE/2;
	Dinvertrectframeoff(&dia_rect);
	dia_rect.top = scr->vthumbpos - THUMBSIZE/2;
	dia_rect.bottom = scr->vthumbpos + THUMBSIZE/2;
	Dinvertrectframeon(&dia_rect);
	return(0);
}

INTSML Drightarrow(INTBIG x, INTBIG y)
{
	INTSML sc;
	DSCROLL *scr;

	scr = &dia_it.scroll[sc=dia_it.curscroll];
	if (scr->horizfactor >= 100) return(1);
	scr->horizfactor++;
	Dredrawscroll(sc);
	Dinvertentry(sc, 1);
	Dsethscroll(sc);
	return(0);
}

INTSML Dleftarrow(INTBIG x, INTBIG y)
{
	INTSML sc;
	DSCROLL *scr;

	scr = &dia_it.scroll[sc=dia_it.curscroll];
	if (scr->horizfactor <= 0) return(1);
	scr->horizfactor--;
	Dredrawscroll(sc);
	Dinvertentry(sc, 1);
	Dsethscroll(sc);
	return(0);
}

INTSML Drightpage(INTBIG x, INTBIG y)
{
	INTSML sc;
	DSCROLL *scr;

	scr = &dia_it.scroll[sc=dia_it.curscroll];
	Dcorrectxy(&x, &y);
	if (x <= scr->hthumbpos+THUMBSIZE/2 || x > scr->userrect.right-16) return(1);
	if (scr->horizfactor >= 100) return(1);
	scr->horizfactor += 10;
	if (scr->horizfactor >= 100) scr->horizfactor = 100;
	Dredrawscroll(sc);
	Dinvertentry(sc, 1);
	Dsethscroll(sc);
	return(0);
}

INTSML Dleftpage(INTBIG x, INTBIG y)
{
	INTSML sc;
	DSCROLL *scr;

	scr = &dia_it.scroll[sc=dia_it.curscroll];
	Dcorrectxy(&x, &y);
	if (x >= scr->hthumbpos-THUMBSIZE/2 || x < scr->userrect.left+16) return(1);
	if (scr->horizfactor <= 0) return(1);
	scr->horizfactor -= 10;
	if (scr->horizfactor <= 0) scr->horizfactor = 0;
	Dredrawscroll(sc);
	Dinvertentry(sc, 1);
	Dsethscroll(sc);
	return(0);
}

INTSML Dhscroll(INTBIG x, INTBIG y)
{
	INTSML l, sc;
	DSCROLL *scr;

	scr = &dia_it.scroll[sc=dia_it.curscroll];
	Dcorrectxy(&x, &y);
	l = scr->hthumbpos;
	scr->hthumbpos = x + dia_offset;
	if (scr->hthumbpos < scr->userrect.left+16+THUMBSIZE/2)
		scr->hthumbpos = scr->userrect.left+16+THUMBSIZE/2;
	if (scr->hthumbpos > scr->userrect.right-16-THUMBSIZE/2)
		scr->hthumbpos = scr->userrect.right-16-THUMBSIZE/2;
	if (scr->hthumbpos == l) return(0);
	dia_rect.left = l - THUMBSIZE/2;
	dia_rect.right = l + THUMBSIZE/2;
	Dinvertrectframeoff(&dia_rect);
	dia_rect.left = scr->hthumbpos - THUMBSIZE/2;
	dia_rect.right = scr->hthumbpos + THUMBSIZE/2;
	Dinvertrectframeon(&dia_rect);
	return(0);
}

/*
 * Routine to parse the next input event and return the next character typed into
 * the current edit item.  If the routine returns -1, nothing has happened.  If the
 * routine returns -2, an item has been hit (and is in "itemHit").
 */
INTSML DiaGetNextCharacter(INTSML *itemHit)
{
	RECTAREA r;
	char *msg;
	INTBIG sr, thumbval;
	UINTBIG time;
#define MAXMATCH 50
	static char match[MAXMATCH];
	static INTSML matchpos = 0;
	static UINTBIG lasttime = 0;
	INTSML i, chr, type, which, v, t, n, x, y, sc, oak, newfirst, ly, hy;
	DSCROLL *scr;
	POPUPDATA *pd;

	oak = Dwaitforaction(&x, &y, &chr, &time);
	if (oak == 1 || oak == 5)
	{
		/* hit in the window: find the item */
		*itemHit = Dwhichitem(x, y);
		if (*itemHit == 0) return(-1);
		type = dia_it.dlgresaddr->list[*itemHit-1].type;
		r = dia_it.dlgresaddr->list[*itemHit-1].r;

		if ((type&ITEMTYPE) == MESSAGE || (type&INACTIVE) != 0) return(-1);

		/* if the item is a popup menu, display it and track */
		if ((type&ITEMTYPE) == POPUP)
		{
			pd = (POPUPDATA *)dia_it.dlgresaddr->list[*itemHit-1].data;
			i = Dhandlepopup(&r, pd);
			if (i == pd->current) return(-1);
			if (i < 0)
			{
				/* system can't handle popups automatically: do it by hand */
				dia_cnt = pd->count;    dia_sta = 0;
				Dgetwindowextent(&ly, &hy);
				for(;;)
				{
					i = r.top - ((pd->current-dia_sta) * dia_lineheight) + 1;
					if (i > ly) break;
					dia_sta++;
				}
				for(;;)
				{
					i = r.bottom + ((dia_cnt-pd->current-1) * dia_lineheight) + 1;
					if (i < hy) break;
					dia_cnt--;
				}
				r.top -= ((pd->current-dia_sta) * dia_lineheight) + 1;
				r.bottom += ((dia_cnt-pd->current-1) * dia_lineheight) + 1;
				r.left--;   r.right++;
				sr = Dsaverect(&r);
				dia_pd = pd;
				dia_rect = r;
				Dsettextsmall(0);
				Ddrawpopupentries();
				dia_wasin = -1;
				Dtrackcursor(x, y, Dtrackpopup);
				Drestorerect(sr);
				if (dia_wasin < 0) return(-1);
				i = dia_wasin + dia_sta;
			}
			pd->current = i;
			Ddrawitem(POPUP, &dia_it.dlgresaddr->list[*itemHit-1].r, (char *)pd, 0);
			return(-2);
		}

		/* items under the user's control are given to the user */
		if (type == USERDRAWN) return(-2);

		/* if the item is edit text, make it the current one */
		if (type == EDITTEXT)
		{
			if (dia_it.curitem != *itemHit) Ddoneedit(); else Dhighlight(0);
			dia_it.curitem = *itemHit;
			msg = (char *)dia_it.dlgresaddr->list[*itemHit-1].data;
			i = Dgeteditpos(&r, x, y, msg);
			if (oak == 5)
			{
				/* look for a full word about position "base" */
				for(dia_it.editstart=i-1; dia_it.editstart>=0; dia_it.editstart--)
					if (!isalnum(msg[dia_it.editstart])) break;
				dia_it.editstart++;
				for(dia_it.editend = dia_it.editstart; msg[dia_it.editend] != 0; dia_it.editend++)
					if (!isalnum(msg[dia_it.editend])) break;
				Dhighlight(1);
			} else
			{
				dia_it.editstart = dia_it.editend = i;
				Dhighlight(1);
			}
			dia_lbase = dia_it.editstart;   dia_hbase = dia_it.editend;
			dia_rect = r;
			dia_msg = msg;
			Dtrackcursor(x, y, Deditdown);
			return(-2);
		}

		/* if the item is a button, reverse it and track */
		if (type == BUTTON)
		{
			r.left++;   r.right--;
			r.top++;    r.bottom--;
			Dinvertroundrecton(&r);
			dia_rect = r;
			dia_wasin = 1;
			Dtrackcursor(x, y, Dbuttondown);
			if (dia_wasin == 0) return(-1);
			Dinvertroundrectoff(&r);
			return(-2);
		}

		/* if the item is a check, outline it and track */
		if (type == CHECK)
		{
			dia_rect = r;
			r.right = r.left + 11;   r.left++;
			r.top = (r.top + r.bottom) / 2 - 5;
			r.bottom = r.top + 10;
			Ddrawrectframe(&r, 1, 0);
			dia_wasin = 1;
			dia_ddata = dia_it.dlgresaddr->list[*itemHit-1].data;
			Dtrackcursor(x, y, Dcheckdown);
			if (dia_wasin == 0) return(-1);
			Ddrawrectframe(&r, 0, 0);
			if (dia_ddata != 0)
			{
				Ddrawline(r.left, r.top, (INTSML)(r.right-1), (INTSML)(r.bottom-1));
				Ddrawline(r.left, (INTSML)(r.bottom-1), (INTSML)(r.right-1), r.top);
			}
			return(-2);
		}

		/* if the item is a scroll area, select a line */
		if (type == SCROLL)
		{
			for(sc=0; sc<dia_it.scrollcount; sc++)
				if (dia_it.scroll[sc].scrollitem == *itemHit) break;
			if (sc >= dia_it.scrollcount) return(-1);
			scr = &dia_it.scroll[dia_it.curscroll=sc];

			if (x > scr->userrect.right)
			{
				/* cursor in vertical slider */
				if (scr->scrolllistlen <= scr->linesinfolder) return(-1);
				if (y > scr->userrect.bottom-16)
				{
					/* the down arrow */
					Ddrawarrow(sc, DOWNARROW, 1);
					Dtrackcursor(x, y, Ddownarrow);
					Ddrawarrow(sc, DOWNARROW, 0);
					return(-1);
				}
				if (y < scr->userrect.top+16)
				{
					/* the up arrow */
					Ddrawarrow(sc, UPARROW, 1);
					Dtrackcursor(x, y, Duparrow);
					Ddrawarrow(sc, UPARROW, 0);
					return(-1);
				}
				if (y > scr->vthumbpos+THUMBSIZE/2 && y <= scr->userrect.bottom-16)
				{
					/* scroll down one page */
					Dtrackcursor(x, y, Ddownpage);
					return(-1);
				}
				if (y < scr->vthumbpos-THUMBSIZE/2 && y >= scr->userrect.top+16)
				{
					/* scroll up one page */
					Dtrackcursor(x, y, Duppage);
					return(-1);
				}
				if (y >= scr->vthumbpos-THUMBSIZE/2 && y <= scr->vthumbpos+THUMBSIZE/2)
				{
					/* drag slider appropriately */
					v = y;   t = scr->vthumbpos;
					dia_rect = dia_it.dlgresaddr->list[*itemHit-1].r;
					dia_rect.left = dia_rect.right - 14;
					dia_rect.right--;
					dia_rect.top = scr->vthumbpos - THUMBSIZE/2;
					dia_rect.bottom = scr->vthumbpos + THUMBSIZE/2;
					Dinvertrectframeon(&dia_rect);
					dia_offset = t-v;
					Dtrackcursor(x, y, Dvscroll);
					dia_rect.top = scr->vthumbpos - THUMBSIZE/2;
					dia_rect.bottom = scr->vthumbpos + THUMBSIZE/2;
					Dinvertrectframeoff(&dia_rect);
					r = scr->userrect;
					r.top += 16;   r.bottom -= 16;
					thumbval = scr->vthumbpos - r.top - THUMBSIZE/2;
					thumbval *= scr->scrolllistlen - scr->linesinfolder;
					thumbval /= r.bottom - r.top - THUMBSIZE;
					i = thumbval;
					if (i < 0) i = 0;
					if (i == scr->firstline) return(-1);
					scr->firstline = i;
					Dredrawscroll(sc);
					Dinvertentry(sc, 1);
					Dsetvscroll(sc);
					return(-1);
				}
			} else if (y > scr->userrect.bottom)
			{
				/* cursor in horizontal slider */
				if (x > scr->userrect.right-16)
				{
					/* the right arrow */
					Ddrawarrow(sc, RIGHTARROW, 1);
					Dtrackcursor(x, y, Drightarrow);
					Ddrawarrow(sc, RIGHTARROW, 0);
					return(-1);
				}
				if (x < scr->userrect.left+16)
				{
					/* the left arrow */
					Ddrawarrow(sc, LEFTARROW, 1);
					Dtrackcursor(x, y, Dleftarrow);
					Ddrawarrow(sc, LEFTARROW, 0);
					return(-1);
				}
				if (x > scr->hthumbpos+THUMBSIZE/2 && x <= scr->userrect.right-16)
				{
					/* scroll right one page */
					Dtrackcursor(x, y, Drightpage);
					return(-1);
				}
				if (x < scr->hthumbpos-THUMBSIZE/2 && x >= scr->userrect.left+16)
				{
					/* scroll left one page */
					Dtrackcursor(x, y, Dleftpage);
					return(-1);
				}
				if (x >= scr->hthumbpos-THUMBSIZE/2 && x <= scr->hthumbpos+THUMBSIZE/2)
				{
					/* drag slider appropriately */
					v = x;   t = scr->hthumbpos;
					dia_rect = dia_it.dlgresaddr->list[*itemHit-1].r;
					dia_rect.top = dia_rect.bottom - 14;
					dia_rect.bottom--;
					dia_rect.left = scr->hthumbpos - THUMBSIZE/2;
					dia_rect.right = scr->hthumbpos + THUMBSIZE/2;
					Dinvertrectframeon(&dia_rect);
					dia_offset = t - v;
					Dtrackcursor(x, y, Dhscroll);
					dia_rect.left = scr->hthumbpos - THUMBSIZE/2;
					dia_rect.right = scr->hthumbpos + THUMBSIZE/2;
					Dinvertrectframeoff(&dia_rect);
					r = scr->userrect;
					r.left += 16;   r.right -= 16;
					thumbval = scr->hthumbpos - r.left - THUMBSIZE/2;
					thumbval *= 100;
					thumbval /= r.right - r.left - THUMBSIZE;
					i = thumbval;
					if (i < 0) i = 0;
					if (i == scr->horizfactor) return(-1);
					scr->horizfactor = i;
					Dredrawscroll(sc);
					Dinvertentry(sc, 1);
					Dsethscroll(sc);
					return(-1);
				}
			} else
			{
				/* double click in scroll selects and returns */
				if (oak == 5 && scr->scrolllistlen > 0 && (scr->flags&SCDOUBLEQUIT) != 0)
				{
					*itemHit = OK;
					return(-2);
				}

				if ((scr->flags&SCSELMOUSE) != 0)
				{
					/* cursor in list: select an entry */
					which = y - r.top;
					which /= scr->lineheight;
					which += scr->firstline;
					if (which >= scr->scrolllistlen) which = scr->scrolllistlen - 1;
					DiaSelectLine(*itemHit, which);
					if ((scr->flags&SCREPORT) == 0) return(-1);
				}
			}
		}
		return(-2);
	}

	/* get the character, return immediately if special */
	if (oak != 0) return(-1);
	if (chr == 033 || chr == '\n' || chr == '\r' || chr == 03) return(chr);

	/* handle arrow positioning */
	if (chr == LEFTARROWKEY)
	{
		if (dia_it.curitem == 0) return(-1);
		Dhighlight(0);
		dia_it.editstart--;
		if (dia_it.editstart < 0) dia_it.editstart = 0;
		dia_it.editend = dia_it.editstart;
		newfirst = Dneweditbase();
		if (dia_it.firstch != newfirst)
		{
			dia_it.firstch = newfirst;
			msg = (char *)dia_it.dlgresaddr->list[dia_it.curitem-1].data;
			Dstufftext(msg, &dia_it.dlgresaddr->list[dia_it.curitem-1].r);
		}
		Dhighlight(1);
		return(-1);
	}
	if (chr == RIGHTARROWKEY)
	{
		if (dia_it.curitem == 0) return(-1);
		Dhighlight(0);
		msg = (char *)dia_it.dlgresaddr->list[dia_it.curitem-1].data;
		if (msg[dia_it.editend] != 0) dia_it.editend++;
		dia_it.editstart = dia_it.editend;
		newfirst = Dneweditbase();
		if (dia_it.firstch != newfirst)
		{
			dia_it.firstch = newfirst;
			Dstufftext(msg, &dia_it.dlgresaddr->list[dia_it.curitem-1].r);
		}
		Dhighlight(1);
		return(-1);
	}

	/* tab to next edit text item */
	if (chr == TAB)
	{
		type = 0;
		for(i=dia_it.curitem+1; i<=dia_it.dlgresaddr->items; i++)
		{
			type = dia_it.dlgresaddr->list[i-1].type;
			if (type == EDITTEXT) break;
		}
		if (type != EDITTEXT)
		{
			for(i=1; i<dia_it.curitem; i++)
			{
				type = dia_it.dlgresaddr->list[i-1].type;
				if (type == EDITTEXT) break;
			}
		}
		if (type != EDITTEXT) return(-1);
		Ddoneedit();
		dia_it.curitem = i;
		dia_it.editstart = 0;
		msg = (char *)dia_it.dlgresaddr->list[i-1].data;
		dia_it.editend = strlen(msg);
		dia_it.firstch = 0;
		Dhighlight(1);
		return(-1);
	}

	if (chr == DOWNARROWKEY)
	{
		if (dia_it.scrollcount <= 0) return(-1);
		scr = &dia_it.scroll[sc = dia_it.curscroll];
		if (scr->scrolllistlen <= 0) return(-1);
		if (scr->which >= scr->scrolllistlen-1) return(-1);
		*itemHit = scr->scrollitem;
		DiaSelectLine(*itemHit, (INTSML)(scr->which+1));
		if ((scr->flags&SCREPORT) == 0) return(-1);
		return(-2);
	}
	if (chr == UPARROWKEY)
	{
		if (dia_it.scrollcount <= 0) return(-1);
		scr = &dia_it.scroll[sc = dia_it.curscroll];
		if (scr->scrolllistlen <= 0) return(-1);
		if (scr->which <= 0) return(-1);
		*itemHit = scr->scrollitem;
		DiaSelectLine(*itemHit, (INTSML)(scr->which-1));
		if ((scr->flags&SCREPORT) == 0) return(-1);
		return(-2);
	}

	if (dia_it.curitem == 0 && dia_it.scrollcount > 0 &&
		(dia_it.scroll[dia_it.curscroll].flags&SCSELKEY) != 0)
	{
		/* use key to select line in scroll area */
		scr = &dia_it.scroll[sc=dia_it.curscroll];
		*itemHit = scr->scrollitem;
		if (chr >= 'A' && chr <= 'Z') chr += 040;

		/* if it has been more than a second, reset the match string */
		if (time - lasttime > 60) matchpos = 0;
		lasttime = time;

		/* add this character to the match string */
		if (matchpos < MAXMATCH)
		{
			match[matchpos] = chr;
			matchpos++;
		}

		/* find that string */
		for(which = 0; which < scr->scrolllistlen; which++)
		{
			for(i=0; i<matchpos; i++)
			{
				n = scr->scrolllist[which][i];
				if (n >= 'A' && n <= 'Z') n += 040;
				if (match[i] != n) break;
			}
			if (i >= matchpos)
			{
				DiaSelectLine(*itemHit, which);
				break;
			}
		}
		if ((scr->flags&SCREPORT) == 0) return(-1);
		return(-2);
	}

	/* insert character into edit text */
	*itemHit = dia_it.curitem;
	return(chr);
}

void Dredrawscroll(INTSML sc)
{
	RECTAREA r;
	INTSML i;
	DSCROLL *scr;

	scr = &dia_it.scroll[sc];
	r = scr->userrect;
	r.left++;        r.right--;
	r.top++;         r.bottom--;
	Dintdrawrect(&r, 0);
	for(i=scr->firstline; i<scr->scrolllistlen; i++)
	{
		if (i-scr->firstline >= scr->linesinfolder) break;
		Ddrawmsg(sc, scr->scrolllist[i], (INTSML)(i-scr->firstline));
	}
}

/*
 * Routine to set the vertical scroll bar
 */
void Dsetvscroll(INTSML sc)
{
	RECTAREA r;
	INTBIG f;
	DSCROLL *scr;

	/* first redraw the border */
	Ddrawvertslider(sc);

	/* get the area of the slider without arrows */
	scr = &dia_it.scroll[sc];
	r = scr->userrect;
	r.top += 16;
	r.bottom -= 16;
	r.left = r.right;
	r.right += 14;

	/* if there is nothing to scroll, clear this area */
	if (scr->scrolllistlen <= scr->linesinfolder)
	{
		Dintdrawrect(&r, 0);
		return;
	}

	/* gray the scroll area */
	Dgrayrect(&r);

	/* compute position of vertical thumb area */
	f = scr->firstline;   f *= r.bottom - r.top-THUMBSIZE;
	f /= scr->scrolllistlen - scr->linesinfolder;
	scr->vthumbpos = r.top + THUMBSIZE/2 + f;
	if (scr->vthumbpos > r.bottom-THUMBSIZE/2) scr->vthumbpos = r.bottom-THUMBSIZE/2;

	/* draw the thumb */
	r.top = scr->vthumbpos - THUMBSIZE/2;
	r.bottom = scr->vthumbpos + THUMBSIZE/2;
	Dintdrawrect(&r, 0);
	Ddrawrectframe(&r, 1, 0);
}

/*
 * Routine to set the horizontal scroll bar
 */
void Dsethscroll(INTSML sc)
{
	RECTAREA r;
	INTBIG f;
	DSCROLL *scr;

	/* get the area of the slider without arrows */
	scr = &dia_it.scroll[sc];
	r = scr->userrect;
	r.left += 16;
	r.right -= 16;
	r.top = r.bottom;
	r.bottom += 13;

	/* gray the scroll area */
	Dgrayrect(&r);

	/* compute position of vertical thumb area */
	f = scr->horizfactor;   f *= (INTBIG)(r.right-r.left-THUMBSIZE);   f /= 100;
	scr->hthumbpos = r.left + THUMBSIZE/2 + f;
	if (scr->hthumbpos > r.right-THUMBSIZE/2) scr->hthumbpos = r.right-THUMBSIZE/2;

	/* draw the thumb */
	r.left = scr->hthumbpos - THUMBSIZE/2;
	r.right = scr->hthumbpos + THUMBSIZE/2;
	Dintdrawrect(&r, 0);
	Ddrawrectframe(&r, 1, 0);
}

void Dinvertentry(INTSML sc, INTSML on)
{
	RECTAREA r;
	DSCROLL *scr;

	scr = &dia_it.scroll[sc];
	if (scr->which-scr->firstline >= 0 && scr->which-scr->firstline < scr->linesinfolder)
	{
		r.left = scr->userrect.left+1;
		r.right = scr->userrect.right-1;
		r.top = scr->userrect.top + scr->lineheight*(scr->which-scr->firstline)+1;
		r.bottom = r.top + scr->lineheight;
		if (r.bottom >= scr->userrect.bottom) r.bottom = scr->userrect.bottom-1;
		if (on == 0) Dinvertrectoff(&r); else Dinvertrecton(&r);
	}
}

/*
 * routine to determine which item falls under the coordinates (x, y)
 */
INTSML Dwhichitem(INTSML x, INTSML y)
{
	INTSML i;
	RECTAREA r;

	for(i=0; i<dia_it.dlgresaddr->items; i++)
	{
		r = dia_it.dlgresaddr->list[i].r;
		if (x >= r.left && x <= r.right && y >= r.top && y <= r.bottom) return(i+1);
	}
	return(0);
}

void Dinsertstr(char *insmsg)
{
	INTSML i, j, newcurspos;
	char *oldmsg, *newmsg, *pt;

	if (dia_it.curitem == 0) return;
	Dhighlight(0);
	oldmsg = (char *)dia_it.dlgresaddr->list[dia_it.curitem-1].data;

	/* allocate space for the new message and fill it */
	newmsg = (char *)emalloc(strlen(oldmsg)+strlen(insmsg)+dia_it.editend-dia_it.editstart+1, el_tempcluster);
	if (newmsg == 0) return;
	j = 0;
	for(i=0; i<dia_it.editstart; i++) newmsg[j++] = oldmsg[i];
	for(i=0; insmsg[i] != 0; i++) newmsg[j++] = insmsg[i];
	newcurspos = j;
	for(i=dia_it.editend; oldmsg[i] != 0; i++) newmsg[j++] = oldmsg[i];
	newmsg[j] = 0;
	dia_it.editstart = dia_it.editend = newcurspos;

	/* replace the message */
	efree((char *)dia_it.dlgresaddr->list[dia_it.curitem-1].data);
	dia_it.dlgresaddr->list[dia_it.curitem-1].data = (INTBIG)newmsg;

	/* make sure the cursor is visible */
	dia_it.firstch = Dneweditbase();
	if (dia_it.opaqueitem == dia_it.curitem)
	{
		allocstring(&oldmsg, newmsg, el_tempcluster);
		for(pt = oldmsg; *pt != 0; pt++)
#ifdef	MACOS
			*pt = '\245';		/* bullet */
#else
			*pt = '*';
#endif
		Dstufftext(oldmsg, &dia_it.dlgresaddr->list[dia_it.curitem-1].r);
		efree(oldmsg);
	} else
	{
		Dstufftext(newmsg, &dia_it.dlgresaddr->list[dia_it.curitem-1].r);
	}

	/* set new highlighting */
	Dhighlight(1);
}

INTSML Dneweditbase(void)
{
	INTSML wid, prevpos, basepos, y, firstoff, len, newfirst;
	char *newmsg, *msg;
	RECTAREA r;

	newfirst = dia_it.firstch;
	newmsg = (char *)dia_it.dlgresaddr->list[dia_it.curitem-1].data;
	r = dia_it.dlgresaddr->list[dia_it.curitem-1].r;
	if (r.bottom-r.top < dia_lineheight*2)
	{
		/* if there is only 1 line in the edit field, shift horizontally */
		if (dia_it.editend < newfirst)
		{
			newfirst = dia_it.editend-1;
			if (newfirst < 0) newfirst = 0;
		}
		while (dia_it.editend > newfirst)
		{
			wid = Dgettextsize(&newmsg[newfirst], (INTSML)(dia_it.editend-newfirst));
			if (wid <= r.right-r.left-2) break;
			newfirst++;
		}
	} else
	{
		/* multiple lines in the edit field, shift vertically */
		while (dia_it.editend < newfirst)
		{
			msg = newmsg;
			prevpos = 0;
			basepos = 0;
			while (*msg != 0)
			{
				for(len = strlen(msg); len>0; len--)
				{
					wid = Dgettextsize(msg, len);
					if (wid < r.right-r.left-2) break;
				}
				if (len == 0) break;
				basepos += len;
				if (basepos == newfirst) break;
				prevpos += len;
				msg += len;
			}
			newfirst = prevpos;
		}
		if (dia_it.editend > newfirst)
		{
			y = r.top + dia_lineheight;
			msg = &newmsg[newfirst];
			firstoff = 0;
			basepos = 0;
			while (*msg != 0)
			{
				for(len = strlen(msg); len>0; len--)
				{
					wid = Dgettextsize(msg, len);
					if (wid < r.right-r.left-2) break;
				}
				if (len == 0) break;
				if (dia_it.editend <= newfirst + basepos + len) break;
				if (firstoff == 0) firstoff = len;
				msg += len;
				basepos += len;
				y += dia_lineheight;
				if (y > r.bottom) { newfirst += firstoff;   break; }
			}
		}
	}
	return(newfirst);
}

void Dhighlight(INTSML on)
{
	INTSML i, sx, ex, sline, eline, line, dummy, es, ee;
	char *msg;
	RECTAREA r, itemrect;

	if (dia_it.curitem == 0) return;
	itemrect = dia_it.dlgresaddr->list[dia_it.curitem-1].r;
	msg = (char *)dia_it.dlgresaddr->list[dia_it.curitem-1].data;
	es = dia_it.editstart;   ee = dia_it.editend;
	if (es > ee) { i = es;   es = ee;   ee = i; }
	Dtextlocation(msg, es, &itemrect, &sx, &sline);
	Dtextlocation(msg, ee, &itemrect, &ex, &eline);
	for(i=sline; i<=eline; i++)
	{
		r.top = itemrect.top + i * dia_lineheight;
		if (r.top >= itemrect.bottom) break;
		r.bottom = r.top + dia_lineheight;
		if (r.bottom > itemrect.bottom) r.bottom = itemrect.bottom;

		r.left = itemrect.left;
		if (i == sline)
		{
			Dtextlocation(msg, (INTSML)(es+1), &itemrect, &dummy, &line);
			if (line != sline && sline != eline) continue;
			r.left += sx;
		}
		r.right = itemrect.right;
		if (i == eline)
		{
			if (ex == 0 && sline != eline) continue;
			r.right = itemrect.left + ex + 1;
		}

		if (on == 0) Dinvertrectoff(&r); else Dinvertrecton(&r);
	}
}

/*
 * routine to determine where the cursor is in the edit item "r" given character
 * "len" of "msg".  The horizontal offset is placed in "wid" and the line number
 * in "line".
 */
void Dtextlocation(char *msg, INTSML position, RECTAREA *r, INTSML *x, INTSML *line)
{
	INTSML basepos, len, wid;

	/* if beyond the end, any hit is the end */
	*line = *x = 0;
	if (dia_it.firstch >= strlen(msg)) return;

	/* set initial message and offset in corner of edit box */
	msg = &msg[dia_it.firstch];
	basepos = dia_it.firstch;
	while (*msg != 0)
	{
		for(len = strlen(msg); len>0; len--)
		{
			wid = Dgettextsize(msg, len);
			if (wid < r->right-r->left-2) break;
		}
		if (len <= 0) { *x = 0;   break; }

		if (position - basepos <= len)
		{
			*x = Dgettextsize(msg, (INTSML)(position - basepos));
			break;
		}
		msg += len;
		basepos += len;
		(*line)++;
	}
}

INTSML Dgeteditpos(RECTAREA *r, INTSML x, INTSML y, char *msg)
{
	INTSML pos, i, basepos, lastpos, len, wid;

	/* if beyond the end, any hit is the end */
	if (dia_it.firstch > strlen(msg)) return(strlen(msg));

	/* set initial message and offset in corner of edit box */
	msg = &msg[dia_it.firstch];
	basepos = dia_it.firstch;
	while (*msg != 0)
	{
		for(len = strlen(msg); len>0; len--)
		{
			wid = Dgettextsize(msg, len);
			if (wid < r->right-r->left-2) break;
		}
		if (len <= 0) break;

		if (y > r->top && y <= r->top+dia_lineheight)
		{
			lastpos = 0;
			for(i=0; i<len; i++)
			{
				pos = Dgettextsize(msg, (INTSML)(i+1));
				if (r->left+(lastpos+pos)/2 > x) break;
				lastpos = pos;
			}
			return(basepos+i);
		}

		msg += len;
		basepos += len;
		y -= dia_lineheight;
	}
	return(basepos);
}

void Ddoneedit(void)
{
	INTSML was;

	if (dia_it.curitem == 0) return;
	Dhighlight(0);
	was = dia_it.curitem;
	dia_it.curitem = 0;
	if (dia_it.firstch == 0) return;
	dia_it.firstch = 0;
	Dstufftext((char *)dia_it.dlgresaddr->list[was-1].data, &dia_it.dlgresaddr->list[was-1].r);
}

void Ddrawitem(INTSML type, RECTAREA *r, char *msg, INTSML dim)
{
	INTSML j, save;
	POPUPDATA *pd;
	RECTAREA myrect;
	INTBIG xv[3], yv[3];

	if ((type&ITEMTYPE) == POPUP)
	{
		/* popup menu item */
		pd = (POPUPDATA *)msg;
		if (pd->count <= 0) return;
		Dstuffmessage(pd->namelist[pd->current], r, dim);
		Dinsetrect(r, -1);
		Ddrawrectframe(r, 1, 0);
		Ddrawline((INTSML)(r->left+3), r->bottom, r->right, r->bottom);
		Ddrawline(r->right, (INTSML)(r->top+3), r->right, r->bottom);
		Dinsetrect(r, 1);
		xv[0] = r->right - 18;   yv[0] = r->top + 4;
		xv[1] = r->right - 12;   yv[1] = r->top + 10;
		xv[2] = r->right - 6;    yv[2] = r->top + 4;
		save = r->left;   r->left = r->right - 19;
		Dintdrawrect(r, 0);
		r->left = save;
		Ddrawpolygon(xv, yv, 3, 1);
	} else if ((type&ITEMTYPE) == BUTTON)
	{
		/* button */
		j = Dgettextsize(msg, (INTSML)strlen(msg));
		Ddrawroundrectframe(r, 12, dim);
		Ddrawtext(msg, (INTSML)strlen(msg), (INTSML)((r->left+r->right-j)/2),
			(INTSML)((r->top+r->bottom+dia_lineheight)/2), dim);
	} else if ((type&ITEMTYPE) == CHECK)
	{
		/* check box */
		myrect = *r;
		myrect.right = myrect.left + 12;
		myrect.top = (myrect.top + myrect.bottom) / 2 - 6;
		myrect.bottom = myrect.top + 12;
		Ddrawrectframe(&myrect, 1, dim);
		Ddrawtext(msg, (INTSML)strlen(msg), (INTSML)(myrect.right+4), (INTSML)(r->top+dia_lineheight), dim);
	} else if ((type&ITEMTYPE) == RADIO)
	{
		/* radio button */
		myrect = *r;
		myrect.right = myrect.left + 12;
		myrect.top = (myrect.top + myrect.bottom) / 2 - 6;
		myrect.bottom = myrect.top + 12;
		Ddrawcircle(&myrect, dim);
		Ddrawtext(msg, (INTSML)strlen(msg), (INTSML)(myrect.right+4), (INTSML)(myrect.top+dia_lineheight), dim);
	} else if ((type&ITEMTYPE) == MESSAGE)
	{
		/* message */
		Dstuffmessage(msg, r, dim);
	} else if ((type&ITEMTYPE) == EDITTEXT)
	{
		/* edit text */
		if (dim == 0) Dstufftext(msg, r);
		Deditbox(r, 1, dim);
	} else if ((type&ITEMTYPE) == SCROLL)
	{
		/* scrollable list */
		Ddrawrectframe(r, 1, dim);
	} else if ((type&ITEMTYPE) == ICON)
	{
		/* 32x32 icon */
		Dputicon(r->left, r->top, msg);
	}
}

void Ddrawpopupentries(void)
{
	INTSML i, y;
	INTBIG xv[3], yv[3];

	Dintdrawrect(&dia_rect, 0);
	Ddrawrectframe(&dia_rect, 1, 0);
	for(i=dia_sta; i<dia_cnt; i++)
	{
		if (i == dia_sta && dia_sta != 0)
		{
			y = dia_rect.top+(i-dia_sta+1)*dia_lineheight+1;
			xv[0] = dia_rect.left + 18;   yv[0] = y - 4;
			xv[1] = dia_rect.left + 12;   yv[1] = y - 10;
			xv[2] = dia_rect.left + 6;    yv[2] = y - 4;
			Ddrawpolygon(xv, yv, 3, 1);
			continue;
		}
		if (i == dia_cnt-1 && dia_cnt != dia_pd->count)
		{
			y = dia_rect.top+(i-dia_sta+1)*dia_lineheight+1;
			xv[0] = dia_rect.left + 18;   yv[0] = y - 10;
			xv[1] = dia_rect.left + 12;   yv[1] = y - 4;
			xv[2] = dia_rect.left + 6;    yv[2] = y - 10;
			Ddrawpolygon(xv, yv, 3, 1);
			continue;
		}
		Ddrawtext(dia_pd->namelist[i], (INTSML)(strlen(dia_pd->namelist[i])), (INTSML)(dia_rect.left+1),
			(INTSML)(dia_rect.top+(i-dia_sta+1)*dia_lineheight+1), 0);
	}
}

void Dstufftext(char *msg, RECTAREA *r)
{
	INTSML len, wid, vpos;

	Dintdrawrect(r, 0);

	/* display the message in multiple lines */
	vpos = r->top+dia_lineheight;
	if (dia_it.firstch > strlen(msg)) return;
	msg = &msg[dia_it.firstch];
	while (*msg != 0)
	{
		for(len = strlen(msg); len>0; len--)
		{
			wid = Dgettextsize(msg, len);
			if (wid < r->right-r->left-2) break;
		}
		if (len <= 0) break;
		Ddrawtext(msg, len, (INTSML)(r->left+1), vpos, 0);
		msg += len;
		vpos += dia_lineheight;
		if (vpos > r->bottom) break;
	}
}

void Dstuffmessage(char *msg, RECTAREA *r, INTSML dim)
{
	INTSML len, wid, truelen, vpos, mostch;
#ifdef	MACOS
	char *pt;

	for(pt = msg; *pt != 0; pt++) if (strncmp(pt, "(tm)", 4) == 0)
	{
		(void)strcpy(pt, "\252");		/* "tm" character */
		(void)strcpy(&pt[1], &pt[4]);
		break;
	}
#endif
	Dintdrawrect(r, 0);

	/* see how much fits */
	truelen = strlen(msg);
	for(len = truelen; len > 0; len--)
	{
		wid = Dgettextsize(msg, len);
		if (wid <= r->right-r->left-2) break;
	}
	if (len <= 0) return;

	/* if it all fits or must fit (because there is just 1 line), draw it */
	vpos = r->top+dia_lineheight;
	if (len == truelen || dia_lineheight*2 > r->bottom - r->top)
	{
		/* detect fields that don't fit!!!   if (len != truelen) ttybeep(); */
		Ddrawtext(msg, len, (INTSML)(r->left+1), vpos, dim);
		return;
	}

	/* write the message in multiple lines */
	while (truelen > 0)
	{
		mostch = 0;
		for(len = truelen; len > 0; len--)
		{
			wid = Dgettextsize(msg, len);
			if (wid > r->right-r->left-2) continue;
			if (mostch == 0) mostch = len;
			if (msg[len] == 0 || msg[len] == ' ') break;
		}
		if (len <= 0) len = mostch;
		Ddrawtext(msg, len, (INTSML)(r->left+1), vpos, dim);
		if (msg[len] == ' ') len++;
		msg += len;
		truelen -= len;
		vpos += dia_lineheight;
		if (vpos > r->bottom) break;
	}
}

void Deditbox(RECTAREA *r, INTSML draw, INTSML dim)
{
	Dinsetrect(r, -3);
	Ddrawrectframe(r, draw, dim);
	Dinsetrect(r, 3);
}

/*
 * routine to draw line "msg" in position "which" of scroll area "sc"
 */
void Ddrawmsg(INTSML sc, char *msg, INTSML which)
{
	INTSML wid, len, firstch, offset;
	INTBIG skip;
	DSCROLL *scr;

	Dsettextsmall(sc);
	scr = &dia_it.scroll[sc];
	firstch = offset = 0;
	if (scr->horizfactor != 0)
	{
		skip = scr->userrect.right - scr->userrect.left - 6;
		skip *= (INTBIG)scr->horizfactor;   skip /= 10;
		wid = Dgettextsize(msg, (INTSML)strlen(msg));
		if (skip >= wid) return;
		for(firstch = 0; firstch < strlen(msg); firstch++)
		{
			wid = Dgettextsize(msg, firstch);
			if (wid >= skip) break;
		}
		offset = wid - skip;
	}

	for(len = strlen(&msg[firstch]); len>0; len--)
	{
		wid = Dgettextsize(&msg[firstch], len);
		if (wid+offset < scr->userrect.right-scr->userrect.left-6) break;
	}
	if (len <= 0) return;
	Ddrawtext(&msg[firstch], len, (INTSML)(scr->userrect.left+2+offset),
		(INTSML)(scr->userrect.top + scr->lineheight*(which+1)), 0);
	Dsettextsmall(-1);
}

void Dinsetrect(RECTAREA *r, INTSML amt)
{
	r->left += amt;   r->right -= amt;
	r->top += amt;    r->bottom -= amt;
}

void Ddrawvertslider(INTSML sc)
{
	RECTAREA r;
	DSCROLL *scr;

	scr = &dia_it.scroll[sc];
	r = scr->userrect;
	r.left = r.right-1;   r.right += 14;
	Ddrawrectframe(&r, 1, 0);
	Ddrawline(r.left, (INTSML)(r.top+15), (INTSML)(r.right-1), (INTSML)(r.top+15));
	Ddrawarrow(sc, UPARROW, 0);
	Ddrawline(r.left, (INTSML)(r.bottom-16), (INTSML)(r.right-1), (INTSML)(r.bottom-16));
	Ddrawarrow(sc, DOWNARROW, 0);
}

void Ddrawhorizslider(INTSML sc)
{
	RECTAREA r;
	DSCROLL *scr;

	scr = &dia_it.scroll[sc];
	r = scr->userrect;
	r.top = r.bottom-1;   r.bottom += 14;
	Ddrawrectframe(&r, 1, 0);
	Ddrawline((INTSML)(r.left+15), r.top, (INTSML)(r.left+15), (INTSML)(r.bottom-1));
	Ddrawarrow(sc, LEFTARROW, 0);
	Ddrawline((INTSML)(r.right-16), r.top, (INTSML)(r.right-16), (INTSML)(r.bottom-1));
	Ddrawarrow(sc, RIGHTARROW, 0);
	Dsethscroll(sc);
}

void Ddrawarrow(INTSML sc, INTSML which, INTSML filled)
{
	INTSML i, left, top, bottom, right;
	INTBIG xv[ARROWLEN], yv[ARROWLEN];
	DSCROLL *scr;

	scr = &dia_it.scroll[sc];
	top = scr->userrect.top;
	bottom = scr->userrect.bottom-1;
	right = scr->userrect.right-1;
	left = scr->userrect.left;
	for(i=0; i<ARROWLEN; i++)
	{
		switch (which)
		{
			case UPARROW:
				xv[i] = right+dia_arrowx[i];
				yv[i] = top+dia_arrowy[i];
				break;
			case DOWNARROW:
				xv[i] = right+dia_arrowx[i];
				yv[i] = bottom-dia_arrowy[i];
				break;
			case LEFTARROW:
				xv[i] = left+dia_arrowy[i];
				yv[i] = bottom+dia_arrowx[i];
				break;
			case RIGHTARROW:
				xv[i] = right-dia_arrowy[i];
				yv[i] = bottom+dia_arrowx[i];
				break;
		}
	}
	Ddrawpolygon(xv, yv, ARROWLEN, filled);
}

/*********************** PRIMITIVE GRAPHICS FOR NON-MACINTOSH ***********************/

#ifndef MACOS

INTSML Dnullup(INTBIG x, INTBIG y) { return(0); }
void Dnullvoid(void) {}
INTSML Dnullchar(INTBIG x, INTBIG y, INTSML ch) { return(0); }

extern GRAPHICS us_menufigs, us_menutext, us_ebox;
GRAPHICS dia_drawgray = {LAYERA, DGRAY, {SOLIDC, SOLIDC, SOLIDC, SOLIDC},
	{0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF}, NOVARIABLE, 0};
GRAPHICS dia_drawdim = {LAYERA, GRAY, {SOLIDC, SOLIDC, SOLIDC, SOLIDC},
	{0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF}, NOVARIABLE, 0};

INTSML     dia_xoffset, dia_yrev;
INTSML     dia_revy;					/* for reversing Y coordinates */
POLYGON  *dia_poly = NOPOLYGON;
WINDOW    dia_window;				/* current window in which dialog is drawn */

#define DRAGBARHEIGHT 18

/*
 * routine to prepare a dialog in location "r", draggable if "movable" is nonzero
 */
void Dnewdialogwindow(RECTAREA *r, char *movable)
{
	INTSML x, swid, shei;
	RECTAREA lr;

	dia_window = *el_curwindow;
	us_getwindowsize(el_curwindow->frame, &swid, &shei);

	/* set drawing bounds to address the world */
	dia_window.screenlx = dia_window.screenly = dia_window.uselx = dia_window.usely = 0;
	dia_window.screenhx = dia_window.usehx = swid-1;
	dia_window.screenhy = dia_window.usehy = shei-1;
	dia_window.frame = el_curwindow->frame;
	computewindowscale(&dia_window);

	dia_revy = shei-1;
	gra_set_cursorstate(0);		/* the arrow cursor */
	us_menufigs.col = el_colmengly;
	us_menutext.col = el_colmentxt;

	/* set text sizes */
	us_settextsize(&dia_window, TXT8P);
	us_textsize(&dia_window, "X", &x, &dia_slineheight);
	us_settextsize(&dia_window, TXTMENU);
	us_textsize(&dia_window, "X", &x, &dia_lineheight);

	/* adjust for window borders */
	lr = *r;
	if (movable == 0) Dinsetrect(&lr, -5); else
	{
		Dinsetrect(&lr, -1);
		lr.top -= DRAGBARHEIGHT;
	}

	/* make sure the dialog fits in the edit window */
	us_getwindowsize(el_curwindow->frame, &swid, &shei);
	if (lr.right >= swid) { lr.left -= lr.right-swid+1;   lr.right = swid-1; }
	if (lr.bottom >= shei) { lr.top -= lr.bottom-shei+1;   lr.bottom = shei-1; }
	if (lr.left < 0) { lr.right -= lr.left;   lr.left = 0; }
	if (lr.top < 0) { lr.bottom -= lr.top;   lr.top = 0; }
	if (lr.right >= swid) lr.right = swid-1;
	if (lr.bottom >= shei) lr.bottom = shei-1;

	/* save the display under the dialog */
	dia_it.theDialog = us_savebox(&dia_window, lr.left, lr.right, (INTSML)(dia_revy-lr.bottom),
		(INTSML)(dia_revy-lr.top));

	/* compute the true offsets */
	if (movable == 0)
	{
		dia_xoffset = lr.left+5;    dia_yrev = dia_revy - (lr.top+5);
	} else
	{
		dia_xoffset = lr.left+1;    dia_yrev = dia_revy - (lr.top+DRAGBARHEIGHT+1);
	}
	dia_lineoffset = 0;
	dia_curlineoffset = dia_lineoffset;
}

/*
 * routine to draw the dialog frame in location "r", draggable if "movable" is nonzero
 */
void Dnewdialogwindowframe(RECTAREA *r, char *movable)
{
	RECTAREA lr;
	INTSML realxo, realyo, wid, swid, shei;

	lr = *r;
	if (movable == 0) Dinsetrect(&lr, -5); else
	{
		Dinsetrect(&lr, -1);
		lr.top -= DRAGBARHEIGHT;
	}

	/* make sure the dialog fits in the edit window */
	us_getwindowsize(el_curwindow->frame, &swid, &shei);
	if (lr.right >= swid) { lr.left -= lr.right-swid+1;   lr.right = swid-1; }
	if (lr.bottom >= shei) { lr.top -= lr.bottom-shei+1;   lr.bottom = shei-1; }
	if (lr.left < 0) { lr.right -= lr.left;   lr.left = 0; }
	if (lr.top < 0) { lr.bottom -= lr.top;   lr.top = 0; }
	if (lr.right >= swid) lr.right = swid-1;
	if (lr.bottom >= shei) lr.bottom = shei-1;

	/* compute the true offsets and set temporary ones */
	if (movable == 0)
	{
		realxo = lr.left+5;    realyo = dia_revy - (lr.top+5);
	} else
	{
		realxo = lr.left+1;    realyo = dia_revy - (lr.top+DRAGBARHEIGHT+1);
	}
	dia_xoffset = lr.left;    dia_yrev = dia_revy - lr.top;

	us_drawbox(&dia_window, lr.left, lr.right, (INTSML)(dia_revy-lr.bottom), (INTSML)(dia_revy-lr.top), &us_ebox);
	lr.right -= lr.left;   lr.left = 0;
	lr.bottom -= lr.top;   lr.top = 0;
	if (movable == 0)
	{
		Ddrawrectframe(&lr, 1, 0);
		Dinsetrect(&lr, 3);
		Ddrawrectframe(&lr, 1, 0);
		Dinsetrect(&lr, 1);
		Ddrawrectframe(&lr, 1, 0);
	} else
	{
		Ddrawrectframe(&lr, 1, 0);
		lr.bottom = lr.top + DRAGBARHEIGHT;
		lr.left++;   lr.right--;   lr.top++;
		Dgrayrect(&lr);
		Ddrawline(lr.left,lr.bottom, lr.right,lr.bottom);
		if (*movable != 0)
		{
			wid = Dgettextsize(movable, (INTSML)strlen(movable));
			lr.left = (lr.right+lr.left-wid) / 2-10;
			lr.right = lr.left + wid + 20;
			Dintdrawrect(&lr, 0);
			Ddrawtext(movable, (INTSML)strlen(movable), (INTSML)(lr.left+10), lr.bottom, 0);
		}
	}

	/* set true offsets */
	dia_xoffset = realxo;    dia_yrev = realyo;
}

/*
 * Routine to return the low and high Y coordinates of the WINDOW in dialog space.
 * The function is only needed if "Dhandlepopup()" is not implemented.
 */
void Dgetwindowextent(INTSML *ly, INTSML *hy)
{
	INTSML swid, shei;

	us_getwindowsize(el_curwindow->frame, &swid, &shei);
	*ly = -dia_it.dlgresaddr->windowRect.top;
	*hy = shei-dia_it.dlgresaddr->windowRect.top;
}

/*
 * routine to terminate the current dialog
 */
void Ddonedialogwindow(void)
{
	(void)us_restorebox(dia_it.theDialog, 1);
}

/*
 * routine to erase the dialog window
 */
void Dclear(void)
{
}

/*
 * routine to force the dialog window to be current (on window systems
 * where this is relevant)
 */
void Dforcedialog(void)
{
}

/*
 * routine to wait for the next action from the dialog window.  Returns:
 *  -1  Nothing happened
 *   0  Key typed, character in "chr"
 *   1  Mouse button pushed, coordinates in (x,y), modifiers in "chr"
 *   2  Mouse button released, coordinates in (x,y)
 *   4  Mouse motion while button down, coordinates in (x,y)
 *   5  Double-click of mouse button, coordinates in (x,y), modifiers in "chr"
 */
INTSML Dwaitforaction(INTSML *x, INTSML *y, INTSML *chr, UINTBIG *time)
{
	INTSML but;
	INTBIG lx, ly;

	if (ttydataready() == 0)
	{
		waitforbutton(x, y, &but);
		if (but < 0) return(-1);
		lx = *x;   ly = *y;
		Dcorrectxy(&lx, &ly);
		*x = lx;   *y = ly;

		/* see if the drag bar was hit */
		if (dia_it.dlgresaddr->movable != 0 && *y < 0)
		{
			Ddragwindow(*x, *y);
			return(-1);
		}
		*time = ticktime();
		if (doublebutton(but) != 0) return(5);
		return(1);
	}
	*chr = us_getnxtchar();
	*time = ticktime();
	return(0);
}

/*
 * routine to track mouse movements while the button remains down, calling
 * the routine "eachdown" for each advance.  The routine is given the
 * mouse coordinates as parameters.  The initial mouse position is (fx,fy).
 */
void Dtrackcursor(INTSML fx, INTSML fy, INTSML (*eachdown)(INTBIG, INTBIG))
{
	trackcursor(0, Dnullup, Dnullvoid, eachdown, Dnullchar, Dnullvoid, TRACKNORMAL);
}

/*
 * routine to save the rectangle "r" and return a pointer that can be
 * used to restore it with "Drestorerect()".  Returns zero if this function
 * is not possible (the function is only needed if "Dhandlepopup()" is not
 * implemented).
 */
INTBIG Dsaverect(RECTAREA *r)
{
	return(us_savebox(&dia_window, (INTSML)(r->left+dia_xoffset), (INTSML)(r->right+1+dia_xoffset),
		(INTSML)(dia_yrev-r->bottom), (INTSML)(dia_yrev-r->top)));
}

/*
 * routine to restore the rectangle "sr" (a pointer that came from "Dsaverect()")
 */
void Drestorerect(INTBIG sr)
{
	(void)us_restorebox(sr, 1);
}

/*
 * routine to move the rectangle "sr" to "dr"
 */
void Dshiftbits(RECTAREA *sr, RECTAREA *dr)
{
	us_movebox(&dia_window, sr->left, sr->top, (INTSML)(sr->right-sr->left), (INTSML)(sr->bottom-sr->top),
		dr->left, dr->top);
}

/*
 * routine to handle popup menu "pd" in rectangle "r".  Returns negative if
 * the function is not possible.
 */
INTSML Dhandlepopup(RECTAREA *r, POPUPDATA *pd)
{
	return(-1);
}

void Dcorrectxy(INTBIG *x, INTBIG *y)
{
	*x -= dia_xoffset;
	*y = dia_yrev - *y;
}

/*
 * routine to draw the 32x32 bitmap in "data" at (x,y)
 */
void Dputicon(INTSML x, INTSML y, char *data)
{
	INTSML xp, yp, v;

	for(yp=0; yp<32; yp++)
	{
		for(xp=0; xp<32; xp++)
		{
			v = data[yp*4 + (xp>>3)];
			if ((v & (0200>>(xp&7))) == 0) continue;
			Ddrawbox(x+xp, x+xp, y+yp, y+yp, &us_menufigs);
		}
	}
}

/*
 * routine to draw a line from (xf,yf) to (xt,yt) in the dialog
 */
void Ddrawline(INTSML xf, INTSML yf, INTSML xt, INTSML yt)
{
	INTBIG fx, fy, tx, ty;

	fx = xf + dia_xoffset;   fy = dia_yrev - yf;
	tx = xt + dia_xoffset;   ty = dia_yrev - yt;
	if (clipline(&fx, &fy, &tx, &ty, dia_window.uselx, dia_window.usehx,
		dia_window.usely, dia_window.usehy) != 0) return;
	us_drawline(&dia_window, (INTSML)fx, (INTSML)fy, (INTSML)tx, (INTSML)ty, &us_menufigs, 0);
}

/*
 * routine to draw a filled rectangle at "r" in the dialog.  Erases if "on" is zero
 */
void Dintdrawrect(RECTAREA *r, INTSML on)
{
	GRAPHICS *which;

	if (on == 0) which = &us_ebox; else which = &us_menufigs;
	Ddrawbox(r->left, r->right-1, r->bottom-1, r->top, which);
}

void Ddrawbox(INTBIG lx, INTBIG hx, INTBIG ly, INTBIG hy, GRAPHICS *which)
{
	lx += dia_xoffset;   ly = dia_yrev - ly;
	hx += dia_xoffset;   hy = dia_yrev - hy;
	if (lx < dia_window.uselx) lx = dia_window.uselx;
	if (hx >= dia_window.usehx) hx = dia_window.usehx-1;
	if (ly < dia_window.usely) ly = dia_window.usely;
	if (hy >= dia_window.usehy) hy = dia_window.usehy-1;
	if (lx > hx || ly > hy) return;
	us_drawbox(&dia_window, (INTSML)lx, (INTSML)hx, (INTSML)ly, (INTSML)hy, which);
}

void Dinvertbox(INTBIG lx, INTBIG hx, INTBIG ly, INTBIG hy)
{
	lx += dia_xoffset;   ly = dia_yrev - ly;
	hx += dia_xoffset;   hy = dia_yrev - hy;
	if (lx < dia_window.uselx) lx = dia_window.uselx;
	if (hx >= dia_window.usehx) hx = dia_window.usehx-1;
	if (ly < dia_window.usely) ly = dia_window.usely;
	if (hy >= dia_window.usehy) hy = dia_window.usehy-1;
	if (lx >= hx || ly >= hy) return;
	us_invertbox(&dia_window, (INTSML)lx, (INTSML)hx, (INTSML)ly, (INTSML)hy);
}

/*
 * routine to draw a gray rectangle at "r" in the dialog
 */
void Dgrayrect(RECTAREA *r)
{
	Ddrawbox(r->left, r->right-1, r->bottom-1, r->top, &dia_drawgray);
}

/*
 * routine to invert a rectangle at "r" in the dialog
 */
void Dinvertrecton(RECTAREA *r)
{
	Dinvertbox(r->left, r->right, r->bottom-1, r->top);
}

void Dinvertrectoff(RECTAREA *r)
{
	Dinvertrecton(r);
}

/*
 * routine to invert a rounded rectangle at "r" in the dialog
 */
void Dinvertroundrecton(RECTAREA *r)
{
	Dinvertrecton(r);
}

void Dinvertroundrectoff(RECTAREA *r)
{
	Dinvertrecton(r);
}

/*
 * routine to draw the outline of rectangle "r" in the dialog.  Erases if "on" is zero
 */
void Ddrawrectframe(RECTAREA *r, INTSML on, INTSML dim)
{
	GRAPHICS *which;

	if (on == 0) which = &us_ebox; else
	{
		if (dim == 0) which = &us_menufigs; else which = &dia_drawdim;
	}
	Ddrawline(r->left, r->top, r->left, (INTSML)(r->bottom-1));
	Ddrawline(r->left, (INTSML)(r->bottom-1), r->right, (INTSML)(r->bottom-1));
	Ddrawline(r->right, (INTSML)(r->bottom-1), r->right, r->top);
	Ddrawline(r->right, r->top, r->left, r->top);
}

/*
 * routine to highlight the outline of rectangle "r" in the dialog
 */
void Dhighlightrect(RECTAREA *r)
{
	Dinsetrect(r, -3);
	Ddrawrectframe(r, 1, 0);
	Dinsetrect(r, -1);
	Ddrawrectframe(r, 1, 0);
}

/*
 * routine to draw the outline of rounded rectangle "r" in the dialog
 */
void Ddrawroundrectframe(RECTAREA *r, INTSML arc, INTSML dim)
{
	Ddrawrectframe(r, 1, dim);
}

/*
 * routine to invert the outline of rectangle "r" in the dialog
 */
void Dinvertrectframeon(RECTAREA *r)
{
	Dinvertbox(r->left, r->left+1, r->bottom-1, r->top);
	Dinvertbox(r->right-1, r->right, r->bottom-1, r->top);
	Dinvertbox(r->left+1, r->right-1, r->bottom, r->bottom-1);
	Dinvertbox(r->left+1, r->right-1, r->top+1, r->top);
}

void Dinvertrectframeoff(RECTAREA *r)
{
	Dinvertrectframeon(r);
}

/*
 * routine to draw the polygon in "count" points (xv,yv) and fill it if "filled" is nonzero
 */
void Ddrawpolygon(INTBIG *xv, INTBIG *yv, INTSML count, INTSML filled)
{
	INTSML i, l;

	/* adjust to edit window coordinates */
	if (dia_poly == NOPOLYGON) dia_poly = allocpolygon(count, us_aid->cluster); else
		if (dia_poly->limit < count) (void)extendpolygon(dia_poly, count);
	for(i=0; i<count; i++)
	{
		dia_poly->xv[i] = xv[i] + dia_xoffset;
		dia_poly->yv[i] = dia_yrev - yv[i];
	}
	dia_poly->count = count;
	dia_poly->style = CLOSED;

	clippoly(dia_poly, dia_window.uselx, dia_window.usehx, dia_window.usely, dia_window.usehy);
	if (dia_poly->count == 0) return;

	if (filled == 0)
	{
		us_drawpolygon(&dia_window, dia_poly->xv, dia_poly->yv, dia_poly->count, &us_ebox);
		for(i=0; i<dia_poly->count; i++)
		{
			if (i == 0) l = dia_poly->count-1; else l = i-1;
			us_drawline(&dia_window, (INTSML)dia_poly->xv[l], (INTSML)dia_poly->yv[l],
				(INTSML)dia_poly->xv[i], (INTSML)dia_poly->yv[i], &us_menufigs, 0);
		}
	} else us_drawpolygon(&dia_window, dia_poly->xv, dia_poly->yv, dia_poly->count, &us_menufigs);
}

/*
 * routine to draw a circle outline in rectangle "r" in the dialog
 */
void Ddrawcircle(RECTAREA *r, INTSML dim)
{
	GRAPHICS *which;

	if (dim == 0) which = &us_menufigs; else which = &dia_drawdim;
	if (dia_poly == NOPOLYGON) dia_poly = allocpolygon(2, us_aid->cluster); else
		if (dia_poly->limit < 2) (void)extendpolygon(dia_poly, 2);
	dia_poly->xv[0] = (r->left+r->right)/2+dia_xoffset;
	dia_poly->yv[0] = dia_yrev-(r->top+r->bottom)/2;
	dia_poly->xv[1] = dia_poly->xv[0] + (r->right-r->left)/2;
	dia_poly->yv[1] = dia_poly->yv[0];
	dia_poly->count = 2;
	dia_poly->style = CIRCLE;
	cliparc(dia_poly, dia_window.uselx, dia_window.usehx, dia_window.usely, dia_window.usehy);
	if (dia_poly->style != CIRCLE) return;

	us_drawcircle(&dia_window, dia_poly->xv[0], dia_poly->yv[0], (r->right-r->left)/2, which);
}

/*
 * routine to draw a filled circle in rectangle "r" in the dialog
 */
void Ddrawdisc(RECTAREA *r)
{
	if (dia_poly == NOPOLYGON) dia_poly = allocpolygon(2, us_aid->cluster); else
		if (dia_poly->limit < 2) (void)extendpolygon(dia_poly, 2);
	dia_poly->xv[0] = (r->left+r->right)/2+dia_xoffset;
	dia_poly->yv[0] = dia_yrev-(r->top+r->bottom)/2;
	dia_poly->xv[1] = dia_poly->xv[0] + (r->right-r->left)/2;
	dia_poly->yv[1] = dia_poly->yv[0];
	dia_poly->count = 2;
	dia_poly->style = DISC;
	cliparc(dia_poly, dia_window.uselx, dia_window.usehx, dia_window.usely, dia_window.usehy);
	if (dia_poly->style != DISC) return;

	us_drawdisc(&dia_window, dia_poly->xv[0], dia_poly->yv[0], (r->right-r->left)/2, &us_menufigs);
}

/*
 * routine to determine the width of the "len" characters at "msg"
 */
INTSML Dgettextsize(char *msg, INTSML len)
{
	INTSML wid, hei, save;

	if (len == 0) return(0);
	save = msg[len];   msg[len] = 0;
	us_textsize(&dia_window, msg, &wid, &hei);
	msg[len] = save;
	return(wid);
}

/*
 * Routine to set the current text size to the large or small scroll text.
 */
void Dsettextsmall(INTSML sc)
{
	INTSML smallf;
	DSCROLL *scr;

	if (sc < 0) smallf = 0; else
	{
		scr = &dia_it.scroll[sc];
		smallf = scr->flags & SCSMALLFONT;
	}
	if (smallf != 0)
	{
		dia_curlineoffset = dia_slineoffset;
		us_settextsize(&dia_window, TXT8P);
	} else
	{
		dia_curlineoffset = dia_lineoffset;
		us_settextsize(&dia_window, TXTMENU);
	}
}

/*
 * routine to draw "len" characters of "msg" at (x,y)
 */
void Ddrawtext(char *msg, INTSML len, INTSML x, INTSML y, INTSML dim)
{
	INTSML save, sx, sy;
	GRAPHICS *which;

	if (dim == 0) which = &us_menutext; else which = &dia_drawdim;
	save = msg[len];   msg[len] = 0;
	us_textsize(&dia_window, msg, &sx, &sy);
	if (x+dia_xoffset >= dia_window.uselx && x+dia_xoffset+sx < dia_window.usehx &&
		dia_yrev-y >= dia_window.usely && dia_yrev-y+sy < dia_window.usehy)
	{
		us_puttext(&dia_window, (INTSML)(x+dia_xoffset), (INTSML)(dia_yrev-y), msg, which);
	}
	msg[len] = save;
}

INTSML dia_lastx, dia_lasty;

INTSML Ddragwin(INTBIG x, INTBIG y)
{
	INTSML swid, shei;

	if (x == dia_lastx && y == dia_lasty) return(0);
	us_getwindowsize(el_curwindow->frame, &swid, &shei);
	if (dia_it.dlgresaddr->windowRect.left+x-dia_lastx < 0) return(0);
	if (dia_it.dlgresaddr->windowRect.right+x-dia_lastx >= swid) return(0);
	if (dia_it.dlgresaddr->windowRect.top-y+dia_lasty < 0) return(0);
	if (dia_it.dlgresaddr->windowRect.bottom-y+dia_lasty >= shei) return(0);
	Dinvertrectframeoff(&dia_it.dlgresaddr->windowRect);
	dia_it.dlgresaddr->windowRect.left += x-dia_lastx;
	dia_it.dlgresaddr->windowRect.right += x-dia_lastx;
	dia_it.dlgresaddr->windowRect.top -= y-dia_lasty;
	dia_it.dlgresaddr->windowRect.bottom -= y-dia_lasty;
	dia_lastx = x;   dia_lasty = y;
	Dinvertrectframeon(&dia_it.dlgresaddr->windowRect);
	return(0);
}

void Ddragwindow(INTSML x, INTSML y)
{
	INTBIG save;
	INTSML swid, shei;
	REGISTER INTSML saveleft, saveright, savetop, savebottom;
	RECTAREA lr, oldr;

	if (y <= -DRAGBARHEIGHT && x >= 0 &&
		x <= dia_it.dlgresaddr->windowRect.right-dia_it.dlgresaddr->windowRect.left) return;

	dia_lastx = x + dia_xoffset;
	dia_lasty = dia_yrev - y;
	dia_xoffset = 0;         dia_yrev = dia_revy;
	oldr = dia_it.dlgresaddr->windowRect;

	/* make the window frame true to size */
	Dinsetrect(&dia_it.dlgresaddr->windowRect, -1);
	dia_it.dlgresaddr->windowRect.top -= DRAGBARHEIGHT;
	Dinvertrectframeon(&dia_it.dlgresaddr->windowRect);
	trackcursor(0, Dnullup, Dnullvoid, Ddragwin, Dnullchar, Dnullvoid, TRACKNORMAL);
	Dinvertrectframeoff(&dia_it.dlgresaddr->windowRect);

	/* save the old dialog image */
	Dinsetrect(&oldr, -1);
	oldr.top -= DRAGBARHEIGHT;
	saveleft = oldr.left;
	saveright = oldr.right;
	savetop = dia_revy-oldr.bottom;
	savebottom = dia_revy-oldr.top;

	/* make sure the dialog fits in the edit window */
	us_getwindowsize(el_curwindow->frame, &swid, &shei);
	if (saveright >= swid) { saveleft -= saveright-swid+1;    saveright = swid-1; }
	if (savebottom >= shei) { savetop -= savebottom-shei+1;   savebottom = shei-1; }
	if (saveleft < 0) { saveright -= saveleft;   saveleft = 0; }
	if (savetop < 0) { lr.bottom -= savetop;   savetop = 0; }
	if (saveright >= swid) saveright = swid-1;
	if (savebottom >= shei) savebottom = shei-1;

	save = us_savebox(&dia_window, saveleft, saveright, savetop, savebottom);

	/* move the saved bits to the new location */
	us_movesavedbox(save, (INTSML)(dia_it.dlgresaddr->windowRect.left-oldr.left),
		(INTSML)(-(dia_it.dlgresaddr->windowRect.top-oldr.top)));

	/* restore the bits underneath */
	us_restorebox(dia_it.theDialog, 1);

	/* save the bits under the new dialog area */
	lr = dia_it.dlgresaddr->windowRect;
	saveleft = lr.left;
	saveright = lr.right;
	savetop = dia_revy-lr.bottom;
	savebottom = dia_revy-lr.top;
	if (saveright >= swid) { saveleft -= saveright-swid+1;    saveright = swid-1; }
	if (savebottom >= shei) { savetop -= savebottom-shei+1;   savebottom = shei-1; }
	if (saveleft < 0) { saveright -= saveleft;   saveleft = 0; }
	if (savetop < 0) { lr.bottom -= savetop;   savetop = 0; }
	if (saveright >= swid) saveright = swid-1;
	if (savebottom >= shei) savebottom = shei-1;
	dia_it.theDialog = us_savebox(&dia_window, saveleft, saveright, savetop, savebottom);

	/* restore the dialog to the new location */
	us_restorebox(save, 1);

	/* restore the moved dialog frame to its true size */
	Dinsetrect(&dia_it.dlgresaddr->windowRect, 1);
	dia_it.dlgresaddr->windowRect.top += DRAGBARHEIGHT;

	/* compute proper drawing offsets */
	dia_xoffset = dia_it.dlgresaddr->windowRect.left;
	dia_yrev = dia_revy - dia_it.dlgresaddr->windowRect.top;
}

#else

/****************************** PRIMITIVE GRAPHICS FOR MACINTOSH ******************************/

INTSML Dnullup(INTBIG x, INTBIG y) { return(0); }
void Dnullvoid(void) {}
INTSML Dnullchar(INTBIG x, INTBIG y, INTSML ch) { return(0); }

/* prototypes for window control */
#ifdef	USETK
  void gra_raisewindow(Tk_Window win);
  void gra_namewindow(Tk_Window win, char *name);
  Tk_Window MakeTKWindow(short type, short left, short right, short top, short bottom,
	void (*eventproc)(ClientData, XEvent*));
  void gra_messageEventProc(ClientData clientData, XEvent *eventPtr);
#else
  void gra_disposeoswindow(WindowPtr, void*);
  void gra_selectoswindow(WindowPtr);
  void gra_activatewindow(WindowPtr, INTSML);
  void gra_frontnonfloatingwindow(WindowPtr *);
#endif

/*
 * routine to prepare a dialog in location "r", draggable if "movable" is nonzero
 */
void Dnewdialogwindow(RECTAREA *r, char *movable)
{
	FontInfo fi;

#ifdef	USETK
	gra_set_cursorstate(0);		/* the arrow cursor */

	if (movable != 0)
	{
		dia_it.tkwindow = MakeTKWindow(3, r->left, r->right, r->top, r->bottom, gra_messageEventProc);
		if (dia_it.tkwindow == NULL) return;
		gra_namewindow(dia_it.tkwindow, movable);
	} else
	{
		dia_it.tkwindow = MakeTKWindow(4, r->left, r->right, r->top, r->bottom, gra_messageEventProc);
		if (dia_it.tkwindow == NULL) return;
	}
	dia_it.theDialog = (WindowPtr)TkMacGetDrawablePort(Tk_WindowId(dia_it.tkwindow));
#else
	WindowPtr frontmost;
	INTSML type;
	INTSML i;
	Str255 line;

	gra_set_cursorstate(0);		/* the arrow cursor */

	if (movable != 0)
	{
		for(i=0; movable[i] != 0; i++) line[i+1] = movable[i];
		line[0] = i;
		type = noGrowDocProc;
	} else
	{
		line[0] = 0;
		type = dBoxProc;
	}

	gra_frontnonfloatingwindow(&frontmost);
	if (frontmost != 0) gra_activatewindow(frontmost, 0);

	dia_it.theDialog = NewWindow(0L, (Rect *)r, line, 0, type, (WindowPtr)(-1L), 0, 0L);
	if (dia_it.theDialog == 0) return;

	ShowHide(dia_it.theDialog, 1);
	gra_activatewindow(dia_it.theDialog, 1);
#endif

	SetPort(dia_it.theDialog);

	/* setup for small scroll text */
	TextFont(courier);
	TextSize(10);
	GetFontInfo(&fi);
	dia_slineheight = fi.ascent + fi.descent + fi.leading;
	dia_slineoffset = fi.descent;

	/* setup for all other text */
	TextFont(0);
	TextSize(12);
	GetFontInfo(&fi);
	dia_lineheight = fi.ascent + fi.descent + fi.leading;
	dia_lineoffset = fi.descent;
	dia_curlineoffset = dia_lineoffset;
}

/*
 * routine to draw the dialog frame in location "r", draggable if "movable" is nonzero
 */
void Dnewdialogwindowframe(RECTAREA *r, char *movable)
{
}

/*
 * Routine to return the low and high Y coordinates of the WINDOW in dialog space.
 * The function is only needed if "Dhandlepopup()" is not implemented.
 */
void Dgetwindowextent(INTSML *ly, INTSML *hy)
{
}

/*
 * routine to terminate the current dialog
 */
void Ddonedialogwindow(void)
{
	Rect r;

#ifdef	USETK
	if (dia_it.dlgresaddr->movable != 0)
	{
		r = (*((WindowPeek)dia_it.theDialog)->strucRgn)->rgnBBox;
		dia_it.dlgresaddr->windowRect.left = r.left+1;
		dia_it.dlgresaddr->windowRect.right = r.right-2;
		dia_it.dlgresaddr->windowRect.top = r.top+1;
		dia_it.dlgresaddr->windowRect.bottom = r.bottom-1-MENUSIZE;
	}
	(void)Tk_DestroyWindow(dia_it.tkwindow);
#else
	if (dia_it.dlgresaddr->movable != 0)
	{
		r = (*((WindowPeek)dia_it.theDialog)->strucRgn)->rgnBBox;
		dia_it.dlgresaddr->windowRect.left = r.left+1;
		dia_it.dlgresaddr->windowRect.right = r.right-2;
		dia_it.dlgresaddr->windowRect.top = r.top+MENUSIZE;
		dia_it.dlgresaddr->windowRect.bottom = r.bottom-2;
	}
	gra_disposeoswindow(dia_it.theDialog, dia_it.tkwindow);
#endif
}

/*
 * routine to erase the dialog window
 */
void Dclear(void)
{
#ifdef USETK
	gra_raisewindow(dia_it.tkwindow);
#else
	gra_selectoswindow(dia_it.theDialog);
#endif
	SetPort(dia_it.theDialog);
}

/*
 * routine to force the dialog window to be current (on window systems
 * where this is relevant)
 */
void Dforcedialog(void)
{
#ifdef USETK
	gra_raisewindow(dia_it.tkwindow);
#else
	gra_selectoswindow(dia_it.theDialog);
#endif
	SetPort(dia_it.theDialog);
}

/*
 * routine to wait for the next action from the dialog window.  Returns:
 *  -1  Nothing happened
 *   0  Key typed, character in "chr"
 *   1  Mouse button pushed, coordinates in (x,y), modifiers in "chr"
 *   2  Mouse button released, coordinates in (x,y)
 *   4  Mouse motion while button down, coordinates in (x,y)
 *   5  Double-click of mouse button, coordinates in (x,y), modifiers in "chr"
 */
INTSML Dwaitforaction(INTSML *x, INTSML *y, INTSML *chr, UINTBIG *time)
{
	INTSML but;
	INTBIG lx, ly;

	if (ttydataready() != 0)
	{
		*chr = us_getnxtchar();
		*time = ticktime();
		return(0);
	}

	/* get button */
	waitforbutton(x, y, &but);
	if (but < 0) return(-1);
	lx = *x;   ly = *y;
	Dcorrectxy(&lx, &ly);
	*x = lx;   *y = ly;
	*time = ticktime();
	if (doublebutton(but) != 0) return(5);
	return(1);
}

/*
 * routine to track mouse movements while the button remains down, calling
 * the routine "eachdown" for each advance.  The routine is given the
 * mouse coordinates as parameters.  The initial mouse position is (fx,fy).
 */
void Dtrackcursor(INTSML fx, INTSML fy, INTSML (*eachdown)(INTBIG, INTBIG))
{
	trackcursor(0, Dnullup, Dnullvoid, eachdown, Dnullchar, Dnullvoid, TRACKNORMAL);
}

void Dcorrectxy(INTBIG *x, INTBIG *y)
{
}

/*
 * routine to save the rectangle "r" and return a pointer that can be
 * used to restore it with "Drestorerect()".  Returns zero if this function
 * is not possible (the function is only needed if "Dhandlepopup()" is not
 * implemented).
 */
INTBIG Dsaverect(RECTAREA *r)
{
	return(0);
}

/*
 * routine to restore the rectangle "sr" (a pointer that came from "Dsaverect()")
 */
void Drestorerect(INTBIG sr)
{
}

/*
 * routine to move the rectangle "sr" to "dr"
 */
void Dshiftbits(RECTAREA *sr, RECTAREA *dr)
{
	SetPort(dia_it.theDialog);
	CopyBits(&dia_it.theDialog->portBits, &dia_it.theDialog->portBits,
		(Rect *)sr, (Rect *)dr, srcCopy, 0);
}

/*
 * routine to handle popup menu "pd" in rectangle "r".  Returns negative if
 * the function is not possible.  Returns the entry in "pd" that is selected
 * (returns pd->current if no change is made).
 */
INTSML Dhandlepopup(RECTAREA *r, POPUPDATA *pd)
{
	INTBIG ret;
	INTSML i, j;
	Str255 name;
	MenuHandle menu;
	Point p;

	SetPort(dia_it.theDialog);
	p.h = r->left;   p.v = r->top;
	LocalToGlobal(&p);
	menu = NewMenu(2048, "\p");
	for(i=0; i<pd->count; i++)
	{
		for(j=0; pd->namelist[i][j] != 0; j++) name[j+1] = pd->namelist[i][j];
		name[0] = j;
		AppendMenu(menu, name);
	}
	InsertMenu(menu, -1);
	ret = PopUpMenuSelect(menu, p.v, p.h, pd->current+1);
	DeleteMenu(2048);
	DisposeMenu(menu);
	if (HiWord(ret) == 0) return(pd->current);
	return(LoWord(ret) - 1);
}

/*
 * routine to draw the 32x32 bitmap in "data" at (x,y)
 */
void Dputicon(INTSML x, INTSML y, char *data)
{
	BitMap b;

	b.baseAddr = data;
	b.rowBytes = 4;
	b.bounds.left = x;   b.bounds.top = y;
	b.bounds.right = x+32;   b.bounds.bottom = y+32;
	CopyBits(&b, &dia_it.theDialog->portBits, &b.bounds, &b.bounds, srcCopy, 0L);
}

/*
 * routine to draw a line from (xf,yf) to (xt,yt) in the dialog
 */
void Ddrawline(INTSML xf, INTSML yf, INTSML xt, INTSML yt)
{
	SetPort(dia_it.theDialog);
	MoveTo(xf, yf);
	LineTo(xt, yt);
}

/*
 * routine to draw a filled rectangle at "r" in the dialog.  Erases if "on" is zero
 */
void Dintdrawrect(RECTAREA *r, INTSML on)
{
	SetPort(dia_it.theDialog);
	if (on == 0) EraseRect((Rect *)r); else FillRect((Rect *)r, &qd.black);
}

/*
 * routine to draw a gray rectangle at "r" in the dialog
 */
void Dgrayrect(RECTAREA *r)
{
	SetPort(dia_it.theDialog);
	FillRect((Rect *)r, &qd.ltGray);
}

/*
 * routines to invert a rectangle at "r" in the dialog
 */
void Dinvertrectoff(RECTAREA *r)
{
	SetPort(dia_it.theDialog);
	InvertRect((Rect *)r);
}

void Dinvertrecton(RECTAREA *r)
{
	Dinvertrectoff(r);
}

/*
 * routine to invert a rounded rectangle at "r" in the dialog
 */
void Dinvertroundrecton(RECTAREA *r)
{
	SetPort(dia_it.theDialog);
	InvertRoundRect((Rect *)r, 10, 10);
}

void Dinvertroundrectoff(RECTAREA *r)
{
	Dinvertroundrecton(r);
}

/*
 * routine to draw the outline of rectangle "r" in the dialog.  Erases if "on" is zero
 */
void Ddrawrectframe(RECTAREA *r, INTSML on, INTSML dim)
{
	SetPort(dia_it.theDialog);
	if (on == 0) PenMode(patBic);
	FrameRect((Rect *)r);
	PenMode(patCopy);
	if (dim != 0)
	{
		PenMode(patBic);
		PenPat(&qd.gray);
		PaintRect((Rect *)r);
		PenMode(patCopy);
		PenPat(&qd.black);
	}
}

/*
 * routine to highlight the outline of rectangle "r" in the dialog
 */
void Dhighlightrect(RECTAREA *r)
{
	Dinsetrect(r, -4);
	SetPort(dia_it.theDialog);
	PenSize(3, 3);
	FrameRoundRect((Rect *)r, 16, 16);
	PenSize(1, 1);
}

/*
 * routine to draw the outline of rounded rectangle "r" in the dialog
 */
void Ddrawroundrectframe(RECTAREA *r, INTSML arc, INTSML dim)
{
	SetPort(dia_it.theDialog);
	FrameRoundRect((Rect *)r, arc, arc);
	if (dim != 0)
	{
		PenMode(patBic);
		PenPat(&qd.gray);
		PaintRect((Rect *)r);
		PenMode(patCopy);
		PenPat(&qd.black);
	}
}

/*
 * routine to invert the outline of rectangle "r" in the dialog
 */
void Dinvertrectframeon(RECTAREA *r)
{
	SetPort(dia_it.theDialog);
	PenMode(patXor);
	FrameRect((Rect *)r);
	PenMode(patCopy);
}

void Dinvertrectframeoff(RECTAREA *r)
{
	Dinvertrectframeon(r);
}

/*
 * routine to draw the polygon in "count" points (xv,yv) and fill it if "filled" is nonzero
 */
void Ddrawpolygon(INTBIG *xv, INTBIG *yv, INTSML count, INTSML filled)
{
	PolyHandle ph;
	INTSML i, l;

	SetPort(dia_it.theDialog);
	ph = OpenPoly();
	for(i=0; i<count; i++)
	{
		if (i == 0) l = count-1; else l = i-1;
		MoveTo(xv[l], yv[l]);
		LineTo(xv[i], yv[i]);
	}
	ClosePoly();
	if (filled != 0) FillPoly(ph, &qd.black); else
	{
		FillPoly(ph, &qd.white);
		FramePoly(ph);
	}
	KillPoly(ph);
}

/*
 * routine to draw a circle outline in rectangle "r" in the dialog
 */
void Ddrawcircle(RECTAREA *r, INTSML dim)
{
	SetPort(dia_it.theDialog);
	FrameOval((Rect *)r);
	if (dim != 0)
	{
		PenMode(patBic);
		PenPat(&qd.gray);
		PaintRect((Rect *)r);
		PenMode(patCopy);
		PenPat(&qd.black);
	}
}

/*
 * routine to draw a filled circle in rectangle "r" in the dialog
 */
void Ddrawdisc(RECTAREA *r)
{
	SetPort(dia_it.theDialog);
	FillOval((Rect *)r, &qd.black);
}

/*
 * routine to determine the width of the "len" characters at "msg"
 */
INTSML Dgettextsize(char *msg, INTSML len)
{
	INTSML wid;

	if (len == 0) return(0);
	SetPort(dia_it.theDialog);
	wid = TextWidth(msg, 0, len);
	return(wid);
}

/*
 * Routine to set the current text size to the large or small scroll text.
 */
void Dsettextsmall(INTSML sc)
{
	INTSML smallf;
	DSCROLL *scr;

	if (sc < 0) smallf = 0; else
	{
		scr = &dia_it.scroll[sc];
		smallf = scr->flags & SCSMALLFONT;
	}
	if (smallf != 0)
	{
		dia_curlineoffset = scr->lineoffset;
		TextFont(courier);
		TextSize(10);
	} else
	{
		dia_curlineoffset = dia_lineoffset;
		TextFont(0);
		TextSize(12);
	}
}

/*
 * routine to draw "len" characters of "msg" at (x,y)
 */
void Ddrawtext(char *msg, INTSML len, INTSML x, INTSML y, INTSML dim)
{
	Rect r;

	SetPort(dia_it.theDialog);
	MoveTo(x, y-dia_curlineoffset-1);
	DrawText(msg, 0, len);
	if (dim != 0)
	{
		r.left = x;     r.right = x + TextWidth(msg, 0, len);
		r.bottom = y;   r.top = y - dia_lineheight;
		PenPat(&qd.gray);
		PenMode(patBic);
		PaintRect(&r);
		PenMode(patCopy);
		PenPat(&qd.black);
	}
}

#endif

#endif
