/*
 * Electric(tm) VLSI Design System
 *
 * File: usrdisp.c
 * User interface aid: miscellaneous display control
 * Written by: Steven M. Rubin, Electric Editor Incorporated
 *
 * Copyright (c) 1998 Electric Editor Incorporated.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Electric Editor Incorporated
 * 23470 Sunset Drive, Suite 108
 * Los Gatos, California 95033
 * support@electriceditor.com
 */

#include "global.h"
#include "egraphics.h"
#include "usr.h"
#include "tech.h"
#include "tecgen.h"
#include <ctype.h>

INTBIG us_stopevent = 0;

/* for drawing facet name, outline or instance name (color changes) */
GRAPHICS us_facetgra = {LAYERO, 0, {SOLIDC, SOLIDC, SOLIDC, SOLIDC},
	{0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF}, NOVARIABLE, 0};

/* for drawing outlines (color changes) */
GRAPHICS us_box = {LAYERA, 0, {SOLIDC, SOLIDC, SOLIDC, SOLIDC},
	{0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF}, NOVARIABLE, 0};

/* for drawing highlight layer (color changes) */
GRAPHICS us_hbox = {LAYERH, 0, {SOLIDC, SOLIDC, SOLIDC, SOLIDC},
	{0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF}, NOVARIABLE, 0};

/* for drawing normal menu border on (nothing changes) */
GRAPHICS us_nmbox = {LAYERA, MENBOR, {SOLIDC, SOLIDC, SOLIDC, SOLIDC},
	{0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF}, NOVARIABLE, 0};

/* for drawing arbitrary graphics (bits and color change) */
GRAPHICS us_arbit = {0, 0, {SOLIDC, SOLIDC, SOLIDC, SOLIDC},
	{0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF}, NOVARIABLE, 0};

/* for erasing polygons (nothing changes) */
GRAPHICS us_ebox = {LAYERA, ALLOFF, {SOLIDC, SOLIDC, SOLIDC, SOLIDC},
	{0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF}, NOVARIABLE, 0};

/* for drawing grid lines (nothing changes) */
GRAPHICS us_gbox = {LAYERG, GRID, {SOLIDC, SOLIDC, SOLIDC, SOLIDC},
	{0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF}, NOVARIABLE, 0};

/* for erasing grid layer (nothing changes) */
GRAPHICS us_egbox = {LAYERG, ALLOFF, {SOLIDC, SOLIDC, SOLIDC, SOLIDC},
	{0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF}, NOVARIABLE, 0};

/* for drawing patterned facet contents (when resolution is too small, color changes) */
GRAPHICS us_graybox = {LAYERO, GRAY, {PATTERNED, PATTERNED, PATTERNED, PATTERNED},
					{0x0000, /*                  */
					0x0303,  /*       XX      XX */
					0x4848,  /*  X  X    X  X    */
					0x0303,  /*       XX      XX */
					0x0000,  /*                  */
					0x3030,  /*   XX      XX     */
					0x8484,  /* X    X  X    X   */
					0x3030}, /*   XX      XX     */
					NOVARIABLE, 0};

#define	MAXGRID        75
#define MINGRID         4	/* the minimum grid has 4 pixels spacing */

/* prototypes for local routines */
void us_showemptywindow(WINDOW*);
void us_graphicsarcs(PORTPROTO*, INTSML*, INTSML*);
void us_combinelayers(ARCINST*, INTSML*, INTSML*);
void us_writeprotoname(PORTPROTO*, INTSML, XARRAY, INTSML, INTSML, WINDOW*, INTBIG, INTBIG, INTBIG, INTBIG);
POLYGON *us_makeprotopoly(PORTPROTO*, XARRAY, WINDOW*, INTBIG, INTBIG, INTBIG, INTBIG);
INTSML us_drawall(INTBIG, INTBIG, INTBIG, INTBIG, NODEPROTO*, XARRAY, INTSML);
INTSML us_drawarcinstpeek(ARCINST*, XARRAY, INTSML);
INTSML us_drawnodeinstpeek(NODEINST*, XARRAY, INTSML);
void us_peeksetlayer(INTSML, TECHNOLOGY*);
void us_peekwritepolygon(INTBIG*, INTBIG*, INTSML, float);

/******************** WINDOW CONTROL ********************/

/*
 * routine to draw the border of window "w"
 */
void us_drawwindow(WINDOW *w, INTSML color)
{
	WINDOW ww;
	static POLYGON *poly = NOPOLYGON;

	/* get polygon */
	if (poly == NOPOLYGON) poly = allocpolygon(5, us_aid->cluster);

	/* don't draw window border if it is a whole-screen window */
	if (strcmp(w->location, "entire") == 0) return;

	/* don't draw window border around popup text editors */
#if defined(MACOS) || defined(WIN32)
	if ((w->state & WINDOWTYPE) == POPTEXTWINDOW) return;
#endif

	ww.screenlx = ww.uselx = w->uselx-1;
	ww.screenhx = ww.usehx = w->usehx+1;
	ww.screenly = ww.usely = w->usely-1;
	ww.screenhy = ww.usehy = w->usehy+1;
	ww.frame = w->frame;
	computewindowscale(&ww);
	maketruerectpoly(ww.uselx, ww.usehx, ww.usely, ww.usehy, poly);
	poly->desc = &us_box;
	poly->style = CLOSEDRECT;
	if (us_getdispstyle(w) == BWRASTER && color == el_colhwinbor)
	{
		us_box.col = 0;
		(void)us_showpoly(poly, &ww);
		makerectpoly(ww.uselx,ww.usehx, ww.usely,ww.usehy, poly);
		us_box.col = HIGHLIT;
		poly->style = OPENEDT1;
		poly->xv[4] = poly->xv[0];   poly->yv[4] = poly->yv[0];
		poly->count++;
		(void)us_showpoly(poly, &ww);
	} else
	{
		us_box.col = color;
		(void)us_showpoly(poly, &ww);
	}
}

/*
 * routine to adjust screen boundaries to fill the view
 */
void us_fullview(NODEPROTO *np, INTBIG *screenlx, INTBIG *screenhx, INTBIG *screenly,
	INTBIG *screenhy)
{
	INTBIG x, y;
	REGISTER NODEINST *ni;
	REGISTER ARCINST *ai;
	REGISTER INTSML first;

	if (framesize(&x, &y, np) == 0)
	{
		/* only one size allowed in schematics window */
		*screenlx = -x/2;
		*screenly = -y/2;
		*screenhx = x/2;
		*screenhy = y/2;
	} else
	{
		/* must recompute this by hand because facet bounds don't include facet-centers */
		first = 1;
		for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
		{
			if (first != 0)
			{
				*screenlx = ni->geom->lowx;
				*screenhx = ni->geom->highx;
				*screenly = ni->geom->lowy;
				*screenhy = ni->geom->highy;
				first = 0;
				continue;
			}
			*screenlx = mini(*screenlx, ni->geom->lowx);
			*screenhx = maxi(*screenhx, ni->geom->highx);
			*screenly = mini(*screenly, ni->geom->lowy);
			*screenhy = maxi(*screenhy, ni->geom->highy);
		}

		/* include all arcs in the facet */
		for(ai = np->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
		{
			*screenlx = mini(*screenlx, ai->geom->lowx);
			*screenhx = maxi(*screenhx, ai->geom->highx);
			*screenly = mini(*screenly, ai->geom->lowy);
			*screenhy = maxi(*screenhy, ai->geom->highy);
		}
		if (*screenlx >= *screenhx && *screenly >= *screenhy)
		{
			*screenlx -= 20 * el_curtech->deflambda;
			*screenhx += 20 * el_curtech->deflambda;
			*screenly -= 20 * el_curtech->deflambda;
			*screenhy += 20 * el_curtech->deflambda;
		}
	}
}

/*
 * routine to adjust "screenlx/hx/ly/hy" for window "w" so that it defines
 * a square screen and leaves some extra space around the values.
 * If "formerw" is a valid window, this is the former window from which
 * a new one in "w" was created, so adjust without scaling.
 * Scaling a window means that one dimension gets bigger or the other
 * gets smaller to make the square relationship hold.  If "largescale" is
 * nonzero, the new window will use the larger of the two scales rather
 * than the smaller.
 *
 * There is a threshold used to determine the method of squaring the
 * screen.  If the window extents are wider or taller than this
 * threshold, imprecise methods are used to see if the window must
 * get narrower or shorter.  Otherwise, more exact methods are used.
 * The threshold is set so that when multiplying the number of pixels
 * by this value, it doesn't overflow.  Assuming that the screen is
 * never more than 1000 wide, a threshold of 2,000,000 will ensure
 * that the 32-bit integer representation of 2,000,000,000 will be
 * safe
 */
#define	THRESH	2000000
void us_squarescreen(WINDOW *w, WINDOW *formerw, INTSML largescale, INTBIG *screenlx,
	INTBIG *screenhx, INTBIG *screenly, INTBIG *screenhy)
{
	REGISTER INTBIG i, units, image, design;
	float prod1, prod2, prodswap, fslx, fshx, fsly, fshy, bump, ysize, xsize;

	if (formerw != NOWINDOW)
	{
		*screenlx = muldiv(((formerw->usehx-formerw->uselx) - (w->usehx-w->uselx))/2,
			formerw->screenhx-formerw->screenlx, formerw->usehx-formerw->uselx) + formerw->screenlx;
		*screenly = muldiv(((formerw->usehy-formerw->usely) - (w->usehy-w->usely))/2,
			formerw->screenhy-formerw->screenly, formerw->usehy-formerw->usely) + formerw->screenly;
		*screenhx = w->screenlx + muldiv(w->usehx-w->uselx, formerw->screenhx-formerw->screenlx,
			formerw->usehx-formerw->uselx);
		*screenhy = w->screenly + muldiv(w->usehy-w->usely, formerw->screenhy-formerw->screenly,
			formerw->usehy-formerw->usely);
		return;
	}

	fslx = (float)*screenlx;   fshx = (float)*screenhx;
	fsly = (float)*screenly;   fshy = (float)*screenhy;
	xsize = (float)(w->usehx - w->uselx);
	ysize = (float)(w->usehy - w->usely);
	prod1 = (fshx - fslx) * ysize;
	prod2 = (fshy - fsly) * xsize;
	if (prod1 != prod2)
	{
		/* reverse the sense if the larger scale is desired */
		if (largescale != 0)
		{
			prodswap = prod1;   prod1 = prod2;   prod2 = prodswap;
		}

		/* adjust the scale */
		if (prod1 > prod2)
		{
			/* screen extent is too wide for window */
			bump = (fshx - fslx) / 20.0;
			if (bump == 0.0) bump = 1.0;
			fshx += bump;   fslx -= bump;
			bump = (fshx - fslx) * ysize / xsize - (fshy - fsly);
			fsly -= bump/2.0;
			fshy += bump/2.0;
		} else
		{
			/* screen extent is too tall for window */
			bump = (fshy - fsly) / 20.0;
			if (bump == 0.0) bump = 1.0;
			fshy += bump;   fsly -= bump;
			bump = (fshy - fsly) * xsize / ysize - (fshx - fslx);
			fslx -= bump/2;
			fshx += bump/2;
		}

		/* put it back into the integer extent fields */
		if (fslx < -HUGEINT/2) *screenlx = -HUGEINT/2; else *screenlx = (INTBIG)fslx;
		if (fshx >  HUGEINT/2) *screenhx =  HUGEINT/2; else *screenhx = (INTBIG)fshx;
		if (fsly < -HUGEINT/2) *screenly = -HUGEINT/2; else *screenly = (INTBIG)fsly;
		if (fshy >  HUGEINT/2) *screenhy =  HUGEINT/2; else *screenhy = (INTBIG)fshy;
	}

	if ((us_aid->aidstate&INTEGRAL) != 0)
	{
		/* adjust window so that it matches well with screen pixels */
		if (*screenhx == *screenlx) return;
		design = (*screenhx - *screenlx) / el_curtech->deflambda;
		if (design <= 0) design = 1;
		image = w->usehx - w->uselx;
		if (image > design)
		{
			/* force integral number of pixels per lambda unit */
			i = (muldiv(image, el_curtech->deflambda, image / design) - (*screenhx - *screenlx)) / 2;
			(*screenhx) += i;   (*screenlx) -= i;
			i = muldiv(*screenhx - *screenlx, w->usehy - w->usely,
				w->usehx - w->uselx) - (*screenhy - *screenly);
			(*screenly) -= i/2;
			(*screenhy) += i/2;
		} else
		{
			/* force integral number of lambda units per pixel */
			units = muldiv(design, el_curtech->deflambda, image);
			i = ((units * (w->usehx - w->uselx)) - (*screenhx - *screenlx)) / 2;
			(*screenhx) += i;   (*screenlx) -= i;
			i = muldiv(*screenhx - *screenlx, w->usehy - w->usely,
				w->usehx - w->uselx) - (*screenhy - *screenly);
			(*screenly) -= i/2;
			(*screenhy) += i/2;
		}
	}
}

/*
 * routine to redisplay everything (all the way down to the bottom) in the
 * window "w"
 */
WINDOW *us_subwindow(INTBIG lx, INTBIG hx, INTBIG ly, INTBIG hy, WINDOW *w)
{
	static WINDOW ww;

	/* redefine the window to clip to this boundary */
	ww.uselx = applyxscale(w, lx-w->screenlx) + w->uselx;
	ww.usely = applyyscale(w, ly-w->screenly) + w->usely;
	ww.usehx = applyxscale(w, hx-w->screenlx) + w->uselx;
	ww.usehy = applyyscale(w, hy-w->screenly) + w->usely;
	ww.screenlx = muldiv(w->screenhx-w->screenlx, ww.uselx-w->uselx, w->usehx-w->uselx) +
		w->screenlx;
	ww.screenhx = muldiv(w->screenhx-w->screenlx, ww.usehx-w->uselx, w->usehx-w->uselx) +
		w->screenlx;
	ww.screenly = muldiv(w->screenhy-w->screenly, ww.usely-w->usely, w->usehy-w->usely) +
		w->screenly;
	ww.screenhy = muldiv(w->screenhy-w->screenly, ww.usehy-w->usely, w->usehy-w->usely) +
		w->screenly;
	ww.frame = w->frame;
	computewindowscale(&ww);
	return(&ww);
}

/*
 * routine to erase window "w" and draw everything that should be in it.
 */
void us_redisplay(WINDOW *w)
{
	WINDOW ww;
	static POLYGON *poly = NOPOLYGON;
	REGISTER INTSML lx, hx, ly, hy;

	/* draw fat red border if simulating */
	if ((w->state&WINDOWSIMULATING) != 0)
	{
		/* get polygon */
		if (poly == NOPOLYGON) poly = allocpolygon(5, us_aid->cluster);

		lx = w->uselx - SIMULATINGBORDERSIZE;   hx = w->usehx + SIMULATINGBORDERSIZE;
		ly = w->usely - SIMULATINGBORDERSIZE;   hy = w->usehy + SIMULATINGBORDERSIZE;
		ww.screenlx = ww.uselx = lx;
		ww.screenhx = ww.usehx = hx;
		ww.screenly = ww.usely = ly;
		ww.screenhy = ww.usehy = hy;
		ww.frame = w->frame;
		computewindowscale(&ww);
		poly->desc = &us_box;
		us_box.col = RED;
		poly->style = FILLEDRECT;

		maketruerectpoly(lx, hx, ly, ly+SIMULATINGBORDERSIZE, poly);
		(void)us_showpoly(poly, &ww);
		maketruerectpoly(lx, hx, hy-SIMULATINGBORDERSIZE, hy, poly);
		(void)us_showpoly(poly, &ww);
		maketruerectpoly(lx, lx+SIMULATINGBORDERSIZE, ly, hy, poly);
		(void)us_showpoly(poly, &ww);
		maketruerectpoly(hx-SIMULATINGBORDERSIZE, hx, ly, hy, poly);
		(void)us_showpoly(poly, &ww);
	}

	if ((us_state&NONOVERLAPPABLEDISPLAY) != 0) us_redisplaynow(w, 1); else
		us_redisplaynow(w, 0);
}

/*
 * routine to erase window "w" and draw everything that should be in it.
 * If "now" is nonzero, draw everything immediately.  Otherwise, draw
 * overlapping layers first and queue nonoverlapping layers for later.
 */
void us_redisplaynow(WINDOW *w, INTSML now)
{
	REGISTER NODEPROTO *facet;
	REGISTER NODEINST *ni;
	REGISTER ARCINST *ai;
	REGISTER INTSML i, j, ret;
	INTBIG cenx, ceny, dummy;
	static POLYGON *poly = NOPOLYGON;

	/* erase the window */
	us_erasewindow(w);
	facet = w->curnodeproto;
	if (facet == NONODEPROTO)
	{
		us_showemptywindow(w);
		return;
	}

	/* initialize hierarchy traversal */
	begintraversehierarchy();

	/* draw the frame if there is one */
	j = framepolys(facet);
	if (j != 0)
	{
		/* get polygon */
		if (poly == NOPOLYGON) poly = allocpolygon(6, us_aid->cluster);
		for(i=0; i<j; i++)
		{
			framepoly(i, poly, facet);
			(*us_displayroutine)(poly, w);
		}
	}

	for(ni = facet->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		if (now != 0)
		{
			if (us_drawfacet(ni, LAYERA, el_matid, 3, w) < 0) break;
		} else
		{
			ret = us_drawfacet(ni, LAYERA, el_matid, 1, w);
			if (ret < 0) break;
			if (ret == 2) us_queueopaque(ni->geom, 0);
		}
	}

	for(ai = facet->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
	{
		if (now != 0)
		{
			if (us_drawarcinst(ai, LAYERA, el_matid, 3, w) < 0) break;
		} else
		{
			ret = us_drawarcinst(ai, LAYERA, el_matid, 1, w);
			if (ret < 0) break;
			if (ret == 2) us_queueopaque(ai->geom, 0);
		}
	}

	/* re-draw grid if on */
	if ((w->state&(GRIDON|GRIDTOOSMALL)) == GRIDON)
	{
		/* get polygon */
		if (poly == NOPOLYGON) poly = allocpolygon(6, us_aid->cluster);

		/* get facet center */
		grabpoint(facet, &cenx, &ceny);

		/* grid spacing */
		poly->xv[0] = w->gridx;
		poly->yv[0] = w->gridy;

		/* initial grid location */
		poly->xv[1] = us_alignvalue(cenx, us_alignment, &dummy);
		poly->xv[1] += (w->screenlx-poly->xv[1]) / w->gridx * w->gridx;
		poly->yv[1] = us_alignvalue(ceny, us_alignment, &dummy);
		poly->yv[1] += (w->screenly-poly->yv[1]) / w->gridy * w->gridy;

		/* display screen extent */
		poly->xv[2] = w->uselx;     poly->yv[2] = w->usely;
		poly->xv[3] = w->usehx;     poly->yv[3] = w->usehy;

		/* object space extent */
		poly->xv[4] = w->screenlx;  poly->yv[4] = w->screenly;
		poly->xv[5] = w->screenhx;  poly->yv[5] = w->screenhy;
		poly->count = 6;
		poly->style = GRIDDOTS;
		poly->desc = &us_gbox;
		(*us_displayroutine)(poly, w);
		flushscreen();
	}
}

/* erase the screen area in window frame "mw" (all windows if "mw" is zero) */
void us_erasescreen(void *mw)
{
	WINDOW ww;
	static POLYGON *poly = NOPOLYGON;
	INTSML swid, shei;
	REGISTER void *frame;

	/* get polygon */
	if (poly == NOPOLYGON) poly = allocpolygon(4, us_aid->cluster);

	for(frame = us_firstwindowframe(); frame != 0; frame = us_nextwindowframe(frame))
	{
		if (mw != 0 && frame != mw) continue;
		us_getwindowsize(frame, &swid, &shei);
		ww.screenlx = ww.uselx = 0;   ww.screenhx = ww.usehx = swid-1;
		ww.screenly = ww.usely = 0;   ww.screenhy = ww.usehy = shei-1;
		ww.frame = frame;
		computewindowscale(&ww);
		maketruerectpoly(0, swid-1, 0, shei-1, poly);
		poly->desc = &us_ebox;
		poly->style = FILLEDRECT;
		(*us_displayroutine)(poly, &ww);
	}
}

/*
 * Routine to display a message in windows with no facet
 */
void us_showemptywindow(WINDOW *w)
{
	static POLYGON *poly = NOPOLYGON;

	/* get polygon */
	if (poly == NOPOLYGON) poly = allocpolygon(4, us_aid->cluster);

	makerectpoly(w->screenlx, w->screenhx, w->screenly, w->screenhy, poly);
	poly->style = TEXTBOX;
#if ELFD
	poly->string = " ";
#else
	poly->string = "No facet in this window";
#endif
	poly->font = TXT20P;
	poly->desc = &us_box;
	us_box.col = MENTXT;
	(*us_displayroutine)(poly, w);
}

/******************** GRID CONTROL ********************/

/*
 * routine to turn the grid on or off in window "w", according to "state"
 */
void us_gridset(WINDOW *w, INTBIG state)
{
	REGISTER NODEPROTO *np;
	REGISTER INTBIG x0, y0, x1, y1;

	if (stopping("Grid")) return;

	if ((state&GRIDON) == 0)
	{
		(void)setval((INTBIG)w, VWINDOW, "state", w->state & ~GRIDON, VINTEGER);
		return;
	}

	np = w->curnodeproto;
	if (np == NONODEPROTO)
	{
		ttyputmsg("Edit a facet before manipulating the grid");
		return;
	}

	x0 = applyxscale(w, 0-w->screenlx) + w->uselx;
	x1 = applyxscale(w, w->gridx-w->screenlx) + w->uselx;

	y0 = applyyscale(w, 0-w->screenly) + w->usely;
	y1 = applyyscale(w, w->gridy-w->screenly) + w->usely;

	if (abs(x1-x0) < MINGRID || abs(y1-y0) < MINGRID)
	{
		if ((w->state&GRIDTOOSMALL) == 0)
		{
			ttyputmsgf("Grid too small, turned off");
			(void)setval((INTBIG)w, VWINDOW, "state", w->state | GRIDTOOSMALL | GRIDON, VINTEGER);
		}
		return;
	}

	if ((w->state&GRIDTOOSMALL) != 0) ttyputmsgf("Grid turned back on");
	(void)setval((INTBIG)w, VWINDOW, "state", (w->state & ~GRIDTOOSMALL) | GRIDON, VINTEGER);
}

/******************** PORT SELECTION ******************/

/*
 * routine to identify the portinst locations by number in a nodeinst
 */
void us_identifyports(NODEINST *ni, INTSML on)
{
	REGISTER PORTPROTO *pp, *npt;
	REGISTER INTSML index, same, j;
	REGISTER INTBIG wx, wy, x, y, digitindentx, digitindenty;
	INTBIG bx, by, cx, cy;
	REGISTER float m, b;
	char line[80], build[10];
	REGISTER WINDOW *w;
	static POLYGON *poly = NOPOLYGON, *npoly = NOPOLYGON, *ppoly = NOPOLYGON;

	/* get polygon */
	if (poly == NOPOLYGON) poly = allocpolygon(4, us_aid->cluster);
	if (npoly == NOPOLYGON) npoly = allocpolygon(4, us_aid->cluster);
	if (ppoly == NOPOLYGON) ppoly = allocpolygon(4, us_aid->cluster);

	/* determine the window to use */
	if (ni->parent == getcurfacet()) w = el_curwindow; else
		for(w = el_topwindow; w != NOWINDOW; w = w->nextwindow)
			if (ni->parent == w->curnodeproto) break;
	if (w == NOWINDOW) return;

	index = 0;
	(void)strcpy(line, "");

	/* compute bounds of initial port*/
	shapeportpoly(ni, ni->proto->firstportproto, npoly, 0);

	/* determine spacing of lines from edge of window */
	digitindentx = (w->screenhx - w->screenlx) / 15;
	digitindenty = (w->screenhy - w->screenly) / 15;

	/* look at each port */
	for(pp=ni->proto->firstportproto; pp!=NOPORTPROTO; pp=pp->nextportproto)
	{
		us_stopevent++;
		if ((us_stopevent%25) == 0)
			if (stopping("Highlight")) return;
		index++;

		/* set the bounds of this port to be from the last portinst */
		polycopy(ppoly, npoly);

		/* compute bounds of the next portinst */
		npt = pp->nextportproto;
		if (npt == NOPORTPROTO) same = 0; else
		{
			shapeportpoly(ni, npt, npoly, 0);
			same = polysame(npoly, ppoly);
		}

		/* now see if the two ports are to be combined */
		if (same != 0)
		{
			(void)sprintf(build, "%d,", index);
			(void)strcat(line, build);
			continue;
		}

		/* prepare to draw this port */
		bx = by = 0;
		for(j=0; j<ppoly->count; j++)
		{
			bx += ppoly->xv[j];   by += ppoly->yv[j];
		}
		bx /= ppoly->count;	by /= ppoly->count;
		cx = (ni->highx+ni->lowx) / 2;
		cy = (ni->highy+ni->lowy) / 2;
		if (cx == bx)
		{
			/* special case: line through node center and port is vertical */
			wx = cx;
			if (by > cy) wy = w->screenhy - digitindenty; else
				wy = w->screenly + digitindenty;
		} else
		{
			/* compute slope of line through nodeinst center and portinst */
			m = (float)(by - cy);   m /= bx - cx;
			b = (float)by;   b -= m*bx;
			if (bx > cx)
			{
				/* project to right side of window */
				y = (INTBIG)(m * w->screenhx + b);
				if (y >= w->screenly && y <= w->screenhy)
				{
					wx = w->screenhx - digitindentx;   wy = y;
				}
			}
			if (bx < cx)
			{
				/* project to left side of window */
				y = (INTBIG)(m * w->screenlx + b);
				if (y >= w->screenly && y <= w->screenhy)
				{
					wx = w->screenlx + digitindentx;   wy = y;
				}
			}
			if (by > cy && m != 0.0)
			{
				/* project to top side of window */
				x = (INTBIG)((w->screenhy - b) / m);
				if (x >= w->screenlx && x <= w->screenhx)
				{
					wx = x;   wy = w->screenhy - digitindenty;
				}
			}
			if (by < cy && m != 0.0)
			{
				/* project to bottom side of window */
				x = (INTBIG)((w->screenly - b) / m);
				if (x >= w->screenlx && x <= w->screenhx)
				{
					wx = x;   wy = w->screenly + digitindenty;
				}
			}
		}
		(void)sprintf(build, "%d", index);
		(void)strcat(line, build);

		/* draw the port index number */
		poly->xv[0] = wx;   poly->yv[0] = wy;   poly->count = 1;
		poly->string = line;
		poly->desc = &us_hbox;
		poly->style = TEXTCENT;
		poly->font = TXTLARGE;
		us_hbox.col = HIGHLIT&on;
		(void)us_showpoly(poly, w);
		(void)strcpy(line, "");

		/* draw the line from the port to the index number */
		poly->xv[0] = bx;   poly->yv[0] = by;
		poly->xv[1] = wx;   poly->yv[1] = wy;
		poly->count = 2;
		poly->desc = &us_arbit;
		poly->style = VECTORS;
		us_arbit.bits = LAYERH;   us_arbit.col = HIGHLIT&on;
		(void)us_showpoly(poly, w);

		/* draw the port outline */
		ppoly->style = CLOSED;
		ppoly->desc = &us_arbit;
		(void)us_showpoly(ppoly, w);
	}
	flushscreen();
}

/******************** NODE/ARC DISPLAY ********************/

/*
 * routine to draw nodeinst "ni" when transformed through "prevtrans".
 * If "on" is nonzero, draw it, otherwise erase it.  If "layers" is
 * 1, only transparent layers are drawn.  If "layers" is 2, only
 * opaque layers are drawn.  If "layers" is 3, all layers are drawn.
 * The nodeinst is drawin in window "w".  The routine returns a layers code
 * to indicate any other layers that must be drawn.  It returns negative
 * if the display has been interrupted.  It is assumed that this
 * nodeinst is in the current facet and must be transformed properly first.
 */
INTSML us_drawfacet(NODEINST *ni, INTSML on, XARRAY prevtrans, INTSML layers, WINDOW *w)
{
	XARRAY localtran, trans;
	REGISTER INTSML res;

	/* make transformation matrix within the current nodeinst */
	if (ni->rotation == 0 && ni->transpose == 0)
	{
		res = us_drawnodeinst(ni, on, prevtrans, layers, w);
	} else
	{
		makerot(ni, localtran);
		transmult(localtran, prevtrans, trans);
		res = us_drawnodeinst(ni, on, trans, layers, w);
	}
	if (res == -2) return(-1);
	if (res == -1) res = 0;
	return(res);
}

/*
 * routine to draw nodeinst "ni" when transformed through "prevtrans".
 * If "on" is nonzero, draw it, otherwise erase it.  The parameter
 * "layers" determines which parts of the nodeinst to draw: 1 for
 * transparent layers, 2 for opaque layers, and 3 for both.
 * No assumptions about the location of the nodeinst are made: it is
 * displayed exactly as it is transformed.  The routine returns
 * -2 if display has been interrupted, -1 if the nodeinst is off the screen,
 * 1 if there are transparent layers to be drawn, 2 if there are opaque layers
 * to be drawn, and 0 if nothing else needs to be drawn.
 */
INTSML us_drawnodeinst(NODEINST *ni, INTSML on, XARRAY prevtrans, INTSML layers, WINDOW *w)
{
	REGISTER INTSML i, j, res, displaytotal, dispstyle;
	INTSML color, bits, moretodo, low, high;
	XARRAY localtran, subrot, trans;
	INTBIG bx, by, ux, uy, px, py, portcliplx, portcliphx, portcliply, portcliphy, swap;
	static POLYGON *poly = NOPOLYGON;
	REGISTER GRAPHICS *gra;
	REGISTER NODEPROTO *np;
	REGISTER PORTPROTO *pp;
	REGISTER PORTARCINST *pi;
	REGISTER PORTEXPINST *pe;
	REGISTER NODEINST *ino;
	REGISTER ARCINST *iar;
	REGISTER VARIABLE *var;

	/* initialize for drawing of facets */
	us_stopevent++;
	if ((us_stopevent%25) == 0)
		if (stopping("Display")) return(-2);

	/* get polygon */
	if (poly == NOPOLYGON) poly = allocpolygon(4, us_aid->cluster);

	moretodo = 0;
	np = ni->proto;

	/* get outline of nodeinst in the window */
	if (ismanhattan(prevtrans) != 0)
	{
		/* manhattan orientation, so only examine 2 diagonal corner points */
		xform(ni->lowx, ni->lowy, &bx, &by, prevtrans);
		xform(ni->highx, ni->highy, &ux, &uy, prevtrans);
		if (bx > ux) { swap = bx;   bx = ux;   ux = swap; }
		if (by > uy) { swap = by;   by = uy;   uy = swap; }
	} else
	{
		/* nonmanhattan orientation, must examine all 4 corner points */
		xform(ni->lowx, ni->lowy, &bx, &by, prevtrans);
		ux = bx;   uy = by;
		xform(ni->highx, ni->highy, &px, &py, prevtrans);
		if (px < bx) bx = px;   if (px > ux) ux = px;
		if (py < by) by = py;   if (py > uy) uy = py;
		xform(ni->lowx, ni->highy, &px, &py, prevtrans);
		if (px < bx) bx = px;   if (px > ux) ux = px;
		if (py < by) by = py;   if (py > uy) uy = py;
		xform(ni->highx, ni->lowy, &px, &py, prevtrans);
		if (px < bx) bx = px;   if (px > ux) ux = px;
		if (py < by) by = py;   if (py > uy) uy = py;
	}

	/* check for being off screen and return error message */
	if (bx > w->screenhx || ux < w->screenlx || by > w->screenhy || uy < w->screenly) return(-1);

	/* write port names if appropriate */
	if (ni->firstportexpinst != NOPORTEXPINST && ni->parent == w->curnodeproto)
	{
		if ((layers&2) == 0) moretodo |= 2; else
		{
			for(pe = ni->firstportexpinst; pe != NOPORTEXPINST; pe = pe->nextportexpinst)
				us_writeprotoname(pe->exportproto, on, prevtrans, LAYERO,
					(INTSML)(el_colfacettxt&on), w, 0, 0, 0, 0);
		}
	}

	/* primitive nodeinst: ask the technology how to draw it */
	if (np->index != 0)
	{
		downhierarchy(ni);
		i = nodepolys(ni);
		if ((us_state&NONOVERLAPPABLEDISPLAY) != 0)
		{
			/* no overlappable display: just draw everything */
			low = 0;   high = i;
		} else
		{
			if ((np->userbits&NHASOPA) == 0) j = i; else
			{
				j = (np->userbits&NFIRSTOPA) >> NFIRSTOPASH;
				if (i < j) j = i;
			}
			if ((layers&1) != 0) low = 0; else low = j;
			if ((layers&2) != 0) high = i; else high = j;
			if (low > 0) moretodo |= 1;
			if (high < i) moretodo |= 2;
		}

		/* don't draw invisible pins to alternative output */
		if (us_displayroutine != us_showpoly)
			if (np == gen_invispinprim && low == 0) low++;

		dispstyle = us_getdispstyle(w);
		for(j=low; j<high; j++)
		{
			/* get description of this layer */
			shapenodepoly(ni, j, poly);

			/* ignore if this layer is not being displayed */
			gra = poly->desc;
			if ((gra->style[dispstyle]&INVISIBLE) != 0) continue;

			/* ignore if no bits are to be drawn */
			if ((on&gra->col) == 0 && on != 0) continue;

			/* draw the nodeinst */
			xformpoly(poly, prevtrans);

			/* save the color information and update it */
			i = gra->col;
			gra->col &= on;

			/* draw the nodeinst and restore the color */
			(*us_displayroutine)(poly, w);
			gra->col = i;
		}
		uphierarchy();
	} else
	{
		/* draw a facet */
		if (on == 0)
		{
			/* facet off: undraw the facet center if it is defined */
			poly->desc = &us_facetgra;   us_facetgra.col = ALLOFF;
			var = getvalkey((INTBIG)ni->proto, VNODEPROTO, VINTEGER|VISARRAY, el_prototype_center);
			if (var != NOVARIABLE)
			{
				poly->xv[0] = ((INTBIG *)var->addr)[0] + (ni->lowx+ni->highx)/2 -
					(ni->proto->lowx+ni->proto->highx)/2;
				poly->yv[0] = ((INTBIG *)var->addr)[1] + (ni->lowy+ni->highy)/2 -
					(ni->proto->lowy+ni->proto->highy)/2;
				poly->count = 1;
				poly->style = CROSS;
				xformpoly(poly, prevtrans);
				(*us_displayroutine)(poly, w);
			}

			/* if facet is off, simply blank the area */
			maketruerectpoly(ni->lowx, ni->highx, ni->lowy, ni->highy, poly);
			poly->style = FILLEDRECT;
			xformpoly(poly, prevtrans);
			if ((*us_displayroutine)(poly, w)) return(-1);

			/* why are the next 3 lines needed?  The area is already erased!!! */
			if (poly->style == FILLEDRECT) poly->style = CLOSEDRECT; else
				poly->style = CLOSED;
			if ((*us_displayroutine)(poly, w)) return(-1);
			return(0);
		}

		/* transform into the nodeinst for display of its guts */
		maketrans(ni, localtran);
		transmult(localtran, prevtrans, subrot);

		/* get facet rectangle */
		maketruerectpoly(ni->lowx, ni->highx, ni->lowy, ni->highy, poly);
		poly->style = CLOSEDRECT;
		xformpoly(poly, prevtrans);
		getbbox(poly, &portcliplx, &portcliphx, &portcliply, &portcliphy);
		(void)us_makescreen(&portcliplx, &portcliply, &portcliphx, &portcliphy, w);

		/* if facet is not expanded, draw its outline, name and ports */
		if ((ni->userbits & NEXPAND) == 0)
		{
			/* draw the unexpanded facet */
			if ((layers&2) != 0)
			{
				/* draw the facet outline as four vectors */
				poly->desc = &us_facetgra;   us_facetgra.col = el_colfacet;
				if ((*us_displayroutine)(poly, w)) return(-1);

				/* write the facet name */
				if ((ni->textdescript&VTPOSITION) == VTPOSBOXED)
				{
					makerectpoly(ni->lowx, ni->highx, ni->lowy, ni->highy, poly);
					poly->style = FILLED;
				} else
				{
					poly->count = 1;
					poly->xv[0] = (ni->lowx + ni->highx) / 2;
					poly->yv[0] = (ni->lowy + ni->highy) / 2;
				}
				adjustdisoffset(ni->geom, poly, ni->textdescript);
				xformpoly(poly, prevtrans);
				poly->desc = &us_facetgra;   us_facetgra.col = el_colfacettxt;
				poly->string = describenodeproto(ni->proto);
				(*us_displayroutine)(poly, w);

				/* see if there are displayable variables on the facet */
				displaytotal = tech_displayablenvars(ni);
				for(i = 0; i < displaytotal; i++)
				{
					(void)tech_filldisplayablenvar(ni, poly);
					xformpoly(poly, prevtrans);
					(*us_displayroutine)(poly, w);
				}

				/* draw the facet center if it is defined */
				var = getvalkey((INTBIG)ni->proto, VNODEPROTO, VINTEGER|VISARRAY, el_prototype_center);
				if (var != NOVARIABLE)
				{
					poly->xv[0] = ((INTBIG *)var->addr)[0] + (ni->lowx+ni->highx)/2 -
						(ni->proto->lowx+ni->proto->highx)/2;
					poly->yv[0] = ((INTBIG *)var->addr)[1] + (ni->lowy+ni->highy)/2 -
						(ni->proto->lowy+ni->proto->highy)/2;
					poly->count = 1;
					poly->style = CROSS;
					xformpoly(poly, prevtrans);
					(*us_displayroutine)(poly, w);
				}
			} else moretodo |= 2;
			for(pp = ni->proto->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
			{
				if ((pp->userbits&PORTDRAWN) == 0)
				{
					/* if there is an arc on this port, don't draw the port */
					for (pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
						if (pi->proto == pp) break;
					if (pi != NOPORTARCINST) continue;
				}

				/* don't bother plotting if no bits are to be drawn */
				us_graphicsarcs(pp, &bits, &color);
				if ((bits&LAYEROE) != 0 && (layers&2) == 0)
				{ moretodo |= 2;  continue; }
				if ((bits&LAYEROE) == 0 && (layers&1) == 0)
				{ moretodo |= 1;  continue; }

				if (pp->subnodeinst->rotation == 0 && pp->subnodeinst->transpose == 0)
				{
					us_writeprotoname(pp, LAYERA, subrot, bits, color, w, portcliplx,
						portcliphx, portcliply, portcliphy);
				} else
				{
					makerot(pp->subnodeinst, localtran);
					transmult(localtran, subrot, trans);
					us_writeprotoname(pp, LAYERA, trans, bits, color, w, portcliplx,
						portcliphx, portcliply, portcliphy);
				}
			}
		} else
		{
			/* if the resolution is too fine, just draw texture pattern */
			if ((us_aid->aidstate&DRAWTINYFACETS) == 0 &&
				(w->screenhx - w->screenlx) / (el_curtech->deflambda*4) > w->usehx - w->uselx &&
				(w->screenhy - w->screenly) / (el_curtech->deflambda*4) > w->usehy - w->usely)
			{
				/* cannot sensibly draw the contents at this resolution: just draw a pattern */
				poly->desc = &us_graybox;   us_graybox.col = el_colfacet;
				poly->style = FILLEDRECT;
				if ((*us_displayroutine)(poly, w)) return(-1);
				poly->desc = &us_facetgra;   us_facetgra.col = el_colfacet;
				if (poly->style == FILLEDRECT) poly->style = CLOSEDRECT; else
					poly->style = CLOSED;
				if ((*us_displayroutine)(poly, w)) return(1);
				return(0);
			}

			/* descend hierarchy to this node */
			downhierarchy(ni);

			/* write facet that is expanded */
			if ((layers&2) == 0) moretodo |= 2; else
			{
				/* write ports that must always be displayed */
				for(pp = ni->proto->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
					if ((pp->userbits&PORTDRAWN) != 0)
				{
					if (pp->subnodeinst->rotation == 0 && pp->subnodeinst->transpose == 0)
					{
						us_writeprotoname(pp, LAYERA, subrot, LAYERO, el_colfacettxt, w,
							portcliplx, portcliphx, portcliply, portcliphy);
					} else
					{
						makerot(pp->subnodeinst, localtran);
						transmult(localtran, subrot, trans);
						us_writeprotoname(pp, LAYERA, trans, LAYERO, el_colfacettxt, w,
							portcliplx, portcliphx, portcliply, portcliphy);
					}
				}

				/* see if there are displayable variables on the facet */
				displaytotal = tech_displayablenvars(ni);
				for(i = 0; i < displaytotal; i++)
				{
					(void)tech_filldisplayablenvar(ni, poly);
					xformpoly(poly, prevtrans);
					(*us_displayroutine)(poly, w);
				}
			}

			/* search through facet */
			for(ino = np->firstnodeinst; ino != NONODEINST; ino = ino->nextnodeinst)
			{
				res = us_drawfacet(ino, LAYERA, subrot, layers, w);
				if (res < 0) break;
				moretodo |= res;
			}
			for(iar = np->firstarcinst; iar != NOARCINST; iar = iar->nextarcinst)
			{
				res = us_drawarcinst(iar, LAYERA, subrot, layers, w);
				if (res < 0) { res = 0;   break; }
				moretodo |= res;
			}
			uphierarchy();
		}
	}
	return(moretodo);
}

/*
 * routine to determine the color of port "pp" by combining the colors
 * of all arcs that can connect to that port into the integers
 * "bitplanes" and "color".  The arcs under consideration are kept
 * in the array "connections"
 */
void us_graphicsarcs(PORTPROTO *pp, INTSML *bitplanes, INTSML *color)
{
	REGISTER INTSML i;
	REGISTER ARCPROTO *outside;
	REGISTER ARCINST *ai;
	REGISTER TECHNOLOGY *tech;

	*bitplanes = *color = 0;
	outside = NOARCPROTO;
	ai = dummyarc();
	ai->width = ai->length = 2000;
	ai->end[0].xpos = ai->end[0].ypos = ai->end[1].xpos = 0;
	ai->end[1].ypos = 2000;
	tech = NOTECHNOLOGY;
	for(i=0; pp->connects[i] != NOARCPROTO; i++)
	{
		ai->proto = pp->connects[i];
		if (tech != NOTECHNOLOGY && ai->proto->tech != tech)
		{
			outside = ai->proto;
			continue;
		}
		tech = ai->proto->tech;
		us_combinelayers(ai, bitplanes, color);
	}
	if (*bitplanes == 0 && outside != NOARCPROTO)
	{
		ai->proto = outside;
		us_combinelayers(ai, bitplanes, color);
	}
}

/*
 * helper routine for "us_graphicsarcs" that builds the integer "bitplanes"
 * and "color" with the layers in arcinst "ai".
 */
void us_combinelayers(ARCINST *ai, INTSML *bitplanes, INTSML *color)
{
	REGISTER INTSML j, k, b, c;
	static POLYGON *poly = NOPOLYGON;

	/* get polygon */
	if (poly == NOPOLYGON) poly = allocpolygon(4, us_aid->cluster);

	k = arcpolys(ai);
	for(j=0; j<k; j++)
	{
		shapearcpoly(ai, j, poly);
		b = poly->desc->bits;
		c = poly->desc->col;
		*bitplanes |= b;
		if (b == LAYERO) *color = c; else *color |= c;
	}
}

/*
 * routine to draw an arcinst.  Returns indicator of what else needs to
 * be drawn.  Returns negative if display interrupted
 */
INTSML us_drawarcinst(ARCINST *ai, INTSML on, XARRAY trans, INTSML layers, WINDOW *w)
{
	REGISTER INTSML i, j, moretodo, low, high, dispstyle;
	REGISTER ARCPROTO *ap;
	static POLYGON *poly = NOPOLYGON;

	/* ask the technology how to draw the arcinst */
	us_stopevent++;
	if ((us_stopevent%25) == 0)
		if (stopping("Display")) return(-1);

	/* get polygon */
	if (poly == NOPOLYGON) poly = allocpolygon(4, us_aid->cluster);

	ap = ai->proto;
	i = arcpolys(ai);
	moretodo = 0;
	if ((us_state&NONOVERLAPPABLEDISPLAY) != 0)
	{
		/* no overlappable display: just draw everything */
		low = 0;   high = i;
	} else
	{
		if ((ap->userbits&AHASOPA) == 0) j = i; else
			j = mini((ap->userbits&AFIRSTOPA) >> AFIRSTOPASH, i);
		if (layers&1) low = 0; else low = j;
		if (layers&2) high = i; else high = j;
		if (low > 0) moretodo |= 1;
		if (high < i) moretodo |= 2;
	}

	/* get the endpoints of the arcinst */
	dispstyle = us_getdispstyle(w);
	for(j=low; j<high; j++)
	{
		shapearcpoly(ai, j, poly);

		/* ignore if this layer is not to be drawn */
		if ((poly->desc->style[dispstyle]&INVISIBLE) != 0) continue;

		/* don't bother plotting if no bits are to be drawn */
		if ((on&poly->desc->col) == 0 && on != 0) continue;

		/* draw the arcinst */
		xformpoly(poly, trans);

		/* save the color information and update it */
		i = poly->desc->col;
		poly->desc->col &= on;

		/* draw the nodeinst and restore the color */
		(*us_displayroutine)(poly, w);
		poly->desc->col = i;
	}
	return(moretodo);
}

/*
 * routine to write the name of portproto "pp" at the appropriate location,
 * transformed through "prevtrans".  The color of the text is "color" and
 * the bit planes in which to write the text is in "bits".  The text is
 * written if "on" is nonzero, erased if zero.  The port is written in window
 * "w".  If "portcliplx" is not equal to "portcliphx", then clip the port text
 * to within that range in X and to "portcliply/portcliphy" in Y.
 */
void us_writeprotoname(PORTPROTO *pp, INTSML on, XARRAY prevtrans, INTSML bits,
	INTSML color, WINDOW *w, INTBIG portcliplx, INTBIG portcliphx, INTBIG portcliply,
	INTBIG portcliphy)
{
	REGISTER POLYGON *poly;

	poly = us_makeprotopoly(pp, prevtrans, w, portcliplx, portcliphx, portcliply, portcliphy);
	if (poly == NOPOLYGON) return;

	/* set the graphics information */
	poly->desc = &us_arbit;
	poly->desc->col = color&on;
	poly->desc->bits = bits;
	(*us_displayroutine)(poly, w);
}

/*
 * routine to make a polygon that describes port "pp", transformed through
 * matrix "prevtrans" in window "w".  If "portcliplx" is not equal to "portcliphx",
 * then clip the port text to within that range in X and to "portcliply/portcliphy" in Y
 * (if visible at all).  Returns NOPOLYGON if nothing should be drawn.
 * The style of the port is taken from the window's "state&PORTLABELS" field:
 * PORTSOFF       no port indication drawn
 * PORTSCROSS     port drawn as a cross
 * PORTSFULL      full port name written textually (regardless of size)
 * PORTSSHORT     short port name written textually (regardless of size)
 *
 * !!! THIS ROUTINE IS ONLY CALLED FROM "us_writeprotoname" ABOVE, SO THEY COULD BE COMBINED.
 */
POLYGON *us_makeprotopoly(PORTPROTO *pp, XARRAY prevtrans, WINDOW *w,
	INTBIG portcliplx, INTBIG portcliphx, INTBIG portcliply, INTBIG portcliphy)
{
	XARRAY localtran, tempt1, tempt2, *t1, *t2, *swapt;
	INTBIG px, py;
	INTSML wid, hei;
	REGISTER INTBIG sx, sy;
	REGISTER char *pt;
	static POLYGON *poly = NOPOLYGON;
	REGISTER NODEINST *ni;
	REGISTER PORTPROTO *rpp;
	REGISTER TECHNOLOGY *tech;

	/* quit now if labels are off */
	if ((w->state&PORTLABELS) == PORTSOFF) return(NOPOLYGON);

	/* get polygon */
	if (poly == NOPOLYGON) poly = allocpolygon(4, us_aid->cluster);

	/* find the most primitive node to get its technology */
	ni = pp->subnodeinst;
	rpp = pp->subportproto;
	while (ni->proto->index == 0)
	{
		ni = rpp->subnodeinst;
		rpp = rpp->subportproto;
	}
	tech = ni->proto->tech;

	/* account for port offset */
	t1 = &tempt1;   t2 = &tempt2;
	if ((w->state&PORTLABELS) != PORTSCROSS && (pp->textdescript&(VTXOFF|VTYOFF)) != 0)
	{
		transid(localtran);
		localtran[2][0] = (pp->textdescript&VTXOFF)>>VTXOFFSH;
		if ((pp->textdescript&VTXOFFNEG) != 0) localtran[2][0] = -localtran[2][0];
		localtran[2][0] = localtran[2][0] * tech->deflambda / 4;
		localtran[2][1] = (pp->textdescript&VTYOFF)>>VTYOFFSH;
		if ((pp->textdescript&VTYOFFNEG) != 0) localtran[2][1] = -localtran[2][1];
		localtran[2][1] = localtran[2][1] * tech->deflambda / 4;
		transmult(localtran, prevtrans, *t1);
	} else transcpy(prevtrans, *t1);

	/* build the rest of the transformation to the bottom */
	ni = pp->subnodeinst;
	rpp = pp->subportproto;
	while (ni->proto->index == 0)
	{
		maketrans(ni, localtran);
		transmult(localtran, *t1, *t2);
		swapt = t1;   t1 = t2;   t2 = swapt;
		ni = rpp->subnodeinst;
		rpp = rpp->subportproto;
		if (ni->rotation != 0 || ni->transpose != 0)
		{
			makerot(ni, localtran);
			transmult(localtran, *t1, *t2);
			swapt = t1;   t1 = t2;   t2 = swapt;
		}
		if ((w->state&PORTLABELS) != PORTSCROSS &&
			ni->proto->index == 0 && (rpp->textdescript&(VTXOFF|VTYOFF)) != 0)
		{
			transid(localtran);
			localtran[2][0] = (rpp->textdescript&VTXOFF)>>VTXOFFSH;
			if ((rpp->textdescript&VTXOFFNEG) != 0) localtran[2][0] = -localtran[2][0];
			localtran[2][0] = localtran[2][0] * tech->deflambda / 4;
			localtran[2][1] = (rpp->textdescript&VTYOFF)>>VTYOFFSH;
			if ((rpp->textdescript&VTYOFFNEG) != 0) localtran[2][1] = -localtran[2][1];
			localtran[2][1] = localtran[2][1] * tech->deflambda / 4;
			transmult(localtran, *t1, *t2);
			swapt = t1;   t1 = t2;   t2 = swapt;
		}
	}

	/* get the center position of the port */
	shapetransportpoly(ni, rpp, poly, *t1);
	getcenter(poly, &px, &py);
	poly->xv[0] = px;   poly->yv[0] = py;   poly->count = 1;

	/* simple description if only drawing ports as crosses */
	if ((w->state&PORTLABELS) == PORTSCROSS)
	{
		poly->style = CROSS;
		return(poly);
	}

	/* writing text name: setup polygon */
	poly->font = (pp->textdescript & VTSIZE) >> VTSIZESH;
	poly->font = truefontsize(poly->font, w, tech);


	/* convert text description to polygon description */
	switch (pp->textdescript&VTPOSITION)
	{
		case VTPOSCENT:      poly->style = TEXTCENT;      break;
		case VTPOSBOXED:     poly->style = TEXTBOX;       break;
		case VTPOSUP:        poly->style = TEXTBOT;       break;
		case VTPOSDOWN:      poly->style = TEXTTOP;       break;
		case VTPOSLEFT:      poly->style = TEXTRIGHT;     break;
		case VTPOSRIGHT:     poly->style = TEXTLEFT;      break;
		case VTPOSUPLEFT:    poly->style = TEXTBOTRIGHT;  break;
		case VTPOSUPRIGHT:   poly->style = TEXTBOTLEFT;   break;
		case VTPOSDOWNLEFT:  poly->style = TEXTTOPRIGHT;  break;
		case VTPOSDOWNRIGHT: poly->style = TEXTTOPLEFT;   break;
	}

	/* get shortened name if requested */
	poly->string = pp->protoname;
	if ((w->state&PORTLABELS) == PORTSSHORT)
	{
		for(pt = pp->protoname; *pt != 0; pt++) if (ispunct(*pt)) break;
		if (*pt != 0)
		{
			(void)initinfstr();
			for(pt = pp->protoname; !ispunct(*pt); pt++) (void)addtoinfstr(*pt);
			poly->string = returninfstr();
		}
	}

	/* see if node rotation affects port position */
	if (poly->style != TEXTCENT)
		poly->style = rotatelabel(poly->style, prevtrans);

	/* clip the port text to the facet bounds, if requested */
	if (portcliplx != portcliphx)
	{
		sx = applyxscale(w, px-w->screenlx) + w->uselx;
		sy = applyyscale(w, py-w->screenly) + w->usely;
		if (sx < portcliplx || sx > portcliphx || sy < portcliply || sy > portcliphy)
			return(NOPOLYGON);
		us_settextsize(w, poly->font);
		us_textsize(w, poly->string, &wid, &hei);   wid++;   hei++;
		switch (poly->style)
		{
			case TEXTCENT:
			case TEXTBOT:
			case TEXTTOP:
				if (portcliphx - portcliplx < wid) poly->style = CROSS; else
				{
					if (sx-wid/2 < portcliplx) sx = portcliplx + wid/2;
					if (sx+wid/2 > portcliphx) sx = portcliphx - wid/2;
				}
				break;
			case TEXTRIGHT:
			case TEXTBOTRIGHT:
			case TEXTTOPRIGHT:
				if (portcliphx - portcliplx < wid) poly->style = CROSS; else
				{
					if (sx-wid < portcliplx) sx = portcliplx + wid;
					if (sx > portcliphx) sx = portcliphx;
				}
				break;
			case TEXTLEFT:
			case TEXTBOTLEFT:
			case TEXTTOPLEFT:
				if (portcliphx - portcliplx < wid) poly->style = CROSS; else
				{
					if (sx < portcliplx) sx = portcliplx;
					if (sx+wid > portcliphx) sx = portcliphx - wid;
				}
				break;
		}
		switch (poly->style)
		{
			case TEXTCENT:
			case TEXTRIGHT:
			case TEXTLEFT:
				if (portcliphy - portcliply < hei) poly->style = CROSS; else
				{
					if (sy-hei/2 < portcliply) sy = portcliply + hei/2;
					if (sy+hei/2 > portcliphy) sy = portcliphy - hei/2;
				}
				break;
			case TEXTBOT:
			case TEXTBOTRIGHT:
			case TEXTBOTLEFT:
				if (portcliphy - portcliply < hei) poly->style = CROSS; else
				{
					if (sy < portcliply) sy = portcliply;
					if (sy+hei > portcliphy) sy = portcliphy - hei;
				}
				break;
			case TEXTTOP:
			case TEXTTOPRIGHT:
			case TEXTTOPLEFT:
				if (portcliphy - portcliply < hei) poly->style = CROSS; else
				{
					if (sy-hei < portcliply) sy = portcliply + hei;
					if (sy > portcliphy) sy = portcliphy;
				}
				break;
		}
		poly->xv[0] = muldiv(sx - w->uselx, w->screenhx - w->screenlx,
			w->usehx - w->uselx) + w->screenlx;
		poly->yv[0] = muldiv(sy - w->usely, w->screenhy - w->screenly,
			w->usehy - w->usely) + w->screenly;
	}

	return(poly);
}

/******************** PEEK DISPLAY ********************/

WINDOW *us_peekwindow;

void us_dopeek(INTBIG lx, INTBIG hx, INTBIG ly, INTBIG hy, NODEPROTO *np, WINDOW *w)
{
	us_peekwindow = w;

	/* draw everything in this area */
	begintraversehierarchy();
	if (us_drawall(lx, hx, ly, hy, np, el_matid, 1) == 2)
	{
		(void)us_drawall(lx, hx, ly, hy, np, el_matid, 2);
	}
}

/*
 * routine to draw everything between "lx" and "hx" in X and between "ly"
 * and "hy" in Y of facet "np", given a transformation matrix of "trans".
 * If "layers" is 1 then only the transparent layers will be drawn and if
 * "layers" is 2 then only the opaque layers will be drawn.  Drawing is done
 * in window "w".  The routine returns the other layers that need to be drawn
 * (1 or 2).  If it returns -1, the display has been aborted.
 */
INTSML us_drawall(INTBIG lx, INTBIG hx, INTBIG ly, INTBIG hy, NODEPROTO *np, XARRAY trans,
	INTSML layers)
{
	XARRAY localtrans, localrot, thist, subrot, bound, *tbound, *tthis;
	INTBIG newlx, newhx, newly, newhy, newx, newy;
	REGISTER NODEINST *ni;
	REGISTER NODEPROTO *subnt;
	REGISTER ARCINST *ai;
	REGISTER INTBIG search;
	REGISTER INTSML i, moretodo;
	REGISTER GEOM *look;

	moretodo = 0;
	search = initsearch(lx, hx, ly, hy, np);
	for(;;)
	{
		/* check for interrupts */
		us_stopevent++;
		if ((us_stopevent%25) == 0)
			if (stopping("Display")) { termsearch(search);   break; }

		/* get the next object to draw */
		look = nextobject(search);
		if (look == NOGEOM) break;
		if (look->entrytype == OBJARCINST)
		{
			ai = look->entryaddr.ai;
			i = us_drawarcinstpeek(ai, trans, layers);
			if (i != -1) moretodo |= i;
		} else if (look->entrytype == OBJNODEINST)
		{
			ni = look->entryaddr.ni;
			if (ni->rotation == 0 && ni->transpose == 0)
			{
				transcpy(trans, thist);
				tthis = &thist;
			} else
			{
				makerot(ni, localrot);   transmult(localrot, trans, thist);
				tthis = &thist;
			}
			downhierarchy(ni);
			if (ni->proto->index != 0)
			{
				/* node is primitive: draw it */
				i = us_drawnodeinstpeek(ni, *tthis, layers);
				if (i != -1) moretodo |= i;
			} else
			{
				/* node is complex: recurse */
				subnt = ni->proto;
				maketransI(ni, localtrans);
				if (ni->rotation == 0 && ni->transpose == 0) tbound = &localtrans; else
				{
					makerotI(ni, localrot);
					transmult(localrot, localtrans, bound);
					tbound = &bound;
				}

				/* compute bounding area inside of sub-facet */
				xform(lx, ly, &newlx, &newly, *tbound);   newhx = newlx;   newhy = newly;
				xform(lx, hy, &newx, &newy, *tbound);
				if (newx < newlx) newlx = newx;   if (newx > newhx) newhx = newx;
				if (newy < newly) newly = newy;   if (newy > newhy) newhy = newy;
				xform(hx, ly, &newx, &newy, *tbound);
				if (newx < newlx) newlx = newx;   if (newx > newhx) newhx = newx;
				if (newy < newly) newly = newy;   if (newy > newhy) newhy = newy;
				xform(hx, hy, &newx, &newy, *tbound);
				if (newx < newlx) newlx = newx;   if (newx > newhx) newhx = newx;
				if (newy < newly) newly = newy;   if (newy > newhy) newhy = newy;

				/* compute new matrix for sub-facet display */
				localtrans[2][0] = -localtrans[2][0];
				localtrans[2][1] = -localtrans[2][1];
				transmult(localtrans, *tthis, subrot);
				i = us_drawall(newlx, newhx, newly, newhy, subnt, subrot, layers);
				if (i != -1) moretodo |= i;
			}
			uphierarchy();
		}
	}
	if (stopping("Display")) return(-1);
	return(moretodo);
}

INTSML us_drawarcinstpeek(ARCINST *ai, XARRAY trans, INTSML layers)
{
	REGISTER INTSML i, j, moretodo, low, high, dispstyle;
	REGISTER ARCPROTO *ap;
	REGISTER GRAPHICS *gra;
	static POLYGON *poly = NOPOLYGON;

	/* get polygon */
	if (poly == NOPOLYGON) poly = allocpolygon(4, us_aid->cluster);

	/* determine which polygons to draw */
	ap = ai->proto;
	i = arcpolys(ai);
	moretodo = 0;
	if ((us_state&NONOVERLAPPABLEDISPLAY) != 0)
	{
		/* no overlappable display: just draw everything */
		low = 0;   high = i;
	} else
	{
		if ((ap->userbits&AHASOPA) == 0) j = i; else
		{
			j = (ap->userbits&AFIRSTOPA) >> AFIRSTOPASH;
			if (i < j) j = i;
		}
		if (layers&1) low = 0; else low = j;
		if (layers&2) high = i; else high = j;
		if (low > 0) moretodo |= 1;
		if (high < i) moretodo |= 2;
	}

	/* get the polygons of the arcinst */
	dispstyle = us_getdispstyle(us_peekwindow);
	for(j=low; j<high; j++)
	{
		shapearcpoly(ai, j, poly);

		/* ignore if this layer is not to be drawn */
		gra = poly->desc;
		if ((gra->style[dispstyle]&INVISIBLE) != 0) continue;

		/* don't bother plotting if nothing to be drawn */
		if (gra->col == ALLOFF || gra->bits == LAYERN) continue;

		/* transform the polygon */
		xformpoly(poly, trans);

		/* draw the polygon */
		(*us_displayroutine)(poly, us_peekwindow);
	}
	return(moretodo);
}

INTSML us_drawnodeinstpeek(NODEINST *ni, XARRAY prevtrans, INTSML layers)
{
	REGISTER INTSML i, j, dispstyle;
	INTSML moretodo, low, high;
	static POLYGON *poly = NOPOLYGON;
	REGISTER GRAPHICS *gra;
	REGISTER NODEPROTO *np;

	/* get polygon */
	if (poly == NOPOLYGON) poly = allocpolygon(4, us_aid->cluster);

	/* determine which polygons to draw */
	np = ni->proto;
	i = nodepolys(ni);
	moretodo = 0;
	if ((us_state&NONOVERLAPPABLEDISPLAY) != 0)
	{
		/* no overlappable display: just draw everything */
		low = 0;   high = i;
	} else
	{
		if ((np->userbits&NHASOPA) == 0) j = i; else
		{
			j = (np->userbits&NFIRSTOPA) >> NFIRSTOPASH;
			if (i < j) j = i;
		}
		if ((layers&1) != 0) low = 0; else low = j;
		if ((layers&2) != 0) high = i; else high = j;
		if (low > 0) moretodo |= 1;
		if (high < i) moretodo |= 2;
	}

	/* don't draw invisible pins to alternative output */
	if (np == gen_invispinprim && low == 0) low++;

	/* get the polygons of the nodeinst */
	dispstyle = us_getdispstyle(us_peekwindow);
	for(j=low; j<high; j++)
	{
		/* get description of this layer */
		shapenodepoly(ni, j, poly);

		/* ignore if this layer is not being displayed */
		gra = poly->desc;
		if ((gra->style[dispstyle]&INVISIBLE) != 0) continue;

		/* ignore if no bits are to be drawn */
		if (gra->col == ALLOFF || gra->bits == LAYERN) continue;

		/* transform the polygon */
		xformpoly(poly, prevtrans);

		/* draw the polygon */
		(*us_displayroutine)(poly, us_peekwindow);
	}
	return(moretodo);
}
