/*
 * Electric(tm) VLSI Design System
 *
 * File: usredtecg.c
 * User interface technology editor: conversion from technology to library
 *  and technology writing to "C" source
 * Written by: Steven M. Rubin, Electric Editor Incorporated
 *
 * Copyright (c) 1998 Electric Editor Incorporated.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Electric Editor Incorporated
 * 23470 Sunset Drive, Suite 108
 * Los Gatos, California 95033
 * support@electriceditor.com
 */

#include "global.h"
#include "egraphics.h"
#include "efunction.h"
#include "tech.h"
#include "tecgen.h"
#include "tecart.h"
#include "usr.h"
#include "usredtec.h"
#include <ctype.h>

extern LIST us_teclayer_functions[];
extern LIST us_tecarc_functions[];
extern LIST us_tecnode_functions[];

/* prototypes for local routines */
char *us_tecedstripdash(char*);
NODEINST *us_tecedplacegeom(POLYGON*, NODEPROTO*);
void us_tecedsetlist(NODEINST*, POLYGON*, INTBIG, INTBIG, INTBIG, INTBIG);
void us_teceditdumpdrctab(FILE*, INTBIG*, TECHNOLOGY*);
char *us_tecededgelabel(INTBIG, INTBIG, INTSML);
char *us_tecedmakefract(INTBIG);
char *us_tecedmakeupper(char*);
char *us_tecedmakesymbol(char*);

/*
 * convert technology "tech" into a library and return that library.
 * Returns NOLIBRARY on error
 */
LIBRARY *us_tecedmakelibfromtech(TECHNOLOGY *tech)
{
	REGISTER LIBRARY *lib;
	REGISTER char **drclist, *lay, *dxf, **sequence, **varnames, *fname;
	REGISTER INTBIG i, j, k, l, total, e, xs, ys, oldlam, *newmap, *mapptr,
		drccount, drcindex, tcon, func, serp, gds, nodexpos, portdescript;
	REGISTER INTSML first;
	INTBIG lx, hx, ly, hy, xpos[4], ypos[4], xsc[4], ysc[4], lxo, hxo, lyo, hyo,
		lxp, hxp, lyp, hyp;
	REGISTER NODEPROTO *np, **nplist, *pnp, **aplist;
	REGISTER TECH_COLORMAP *colmap;
	float spires, spicap, spiecap;
	REGISTER VARIABLE *var, *var2, *var3, *var5, *var6, *var7, *var8,
		*varred, *vargreen, *varblue;
	REGISTER NODEINST *ni, *oni, *nni;
	REGISTER ARCINST *ai;
	REGISTER PORTPROTO *pp, *opp;
	REGISTER ARCPROTO *ap;
	REGISTER CELL *c;
	REGISTER GRAPHICS *desc;
	static POLYGON *poly = NOPOLYGON;
	extern AIDENTRY *net_aid;
	REGISTER TECH_POLYGON *ll;

	/* make sure network tool is on */
	if ((net_aid->aidstate&AIDON) == 0)
	{
		ttyputerr("Network tool must be running...turning it on");
		aidturnon(net_aid, 0);
		ttyputerr("...now reissue the technology editing command");
		return(NOLIBRARY);
	}

	/* get polygon */
	if (poly == NOPOLYGON) poly = allocpolygon(4, us_aid->cluster);

	lib = newlibrary(tech->techname, tech->techname);
	if (lib == NOLIBRARY)
	{
		ttyputerr("Cannot create library %s", tech->techname);
		return(NOLIBRARY);
	}
	ttyputmsg("Created library %s...", tech->techname);

	/* create the information node */
	np = newnodeproto("factors", lib);
	if (np == NONODEPROTO) return(NOLIBRARY);
	us_tecedmakeinfo(np, tech->deflambda, tech->techdescript);

	/* copy any miscellaneous variables and make a list of their names */
	j = 0;
	for(i=0; us_knownvars[i].varname != 0; i++)
	{
		us_knownvars[i].ival = 0;
		var = getval((INTBIG)tech, VTECHNOLOGY, -1, us_knownvars[i].varname);
		if (var == NOVARIABLE) continue;
		us_knownvars[i].ival = 1;
		j++;
		(void)setval((INTBIG)lib, VLIBRARY, us_knownvars[i].varname, var->addr, var->type);
	}
	if (j > 0)
	{
		varnames = (char **)emalloc(j * (sizeof (char *)), el_tempcluster);
		if (varnames == 0) return(NOLIBRARY);
		j = 0;
		for(i=0; us_knownvars[i].varname != 0; i++)
			if (us_knownvars[i].ival != 0) varnames[j++] = us_knownvars[i].varname;
		(void)setval((INTBIG)lib, VLIBRARY, "EDTEC_variable_list", (INTBIG)varnames,
			VSTRING|VISARRAY|(j<<VLENGTHSH));
		efree((char *)varnames);
	}

	/* create the layer node names */
	total = tech->layercount;
	nplist = (NODEPROTO **)emalloc((total * (sizeof (NODEPROTO *))), el_tempcluster);
	if (nplist == 0) return(NOLIBRARY);

	/* modify this technology's lambda value to match the current one */
	oldlam = tech->deflambda;
	tech->deflambda = el_curtech->deflambda;

	/* create the layer nodes */
	ttyputmsg("Creating the layers...");
	var2 = getval((INTBIG)tech, VTECHNOLOGY, VSTRING|VISARRAY, "IO_cif_layer_names");
	var3 = getval((INTBIG)tech, VTECHNOLOGY, VINTEGER|VISARRAY, "IO_gds_layer_numbers");
	var8 = getval((INTBIG)tech, VTECHNOLOGY, VSTRING|VISARRAY, "IO_dxf_layer_names");
	var5 = getval((INTBIG)tech, VTECHNOLOGY, VFLOAT|VISARRAY, "SIM_spice_resistance");
	var6 = getval((INTBIG)tech, VTECHNOLOGY, VFLOAT|VISARRAY, "SIM_spice_capacitance");
	var7 = getval((INTBIG)tech, VTECHNOLOGY, VFLOAT|VISARRAY, "SIM_spice_edge_capacitance");
	for(i=0; i<total; i++)
	{
		desc = tech->layers[i];
		(void)initinfstr();
		(void)addstringtoinfstr("layer-");
		(void)addstringtoinfstr(us_tecedstripdash(layername(tech, (INTSML)i)));
		fname = returninfstr();

		/* make sure the layer doesn't exist */
		for(c = lib->firstcell; c != NOCELL; c = c->nextcell)
			if (namesame(fname, c->cellname) == 0)
		{
			ttyputerr("Warning: multiple layers named '%s'", fname);
			break;
		}

		np = newnodeproto(fname, lib);
		if (np == NONODEPROTO) return(NOLIBRARY);
		nplist[i] = np;

		/* compute the CIF layer */
		if (var2 == NOVARIABLE) lay = ""; else
			lay = ((char **)var2->addr)[i];

		/* compute the GDS-II layer */
		if (var3 == NOVARIABLE) gds = -1; else
			gds = ((INTBIG *)var3->addr)[i];

		/* compute the DXF layer name */
		if (var8 == NOVARIABLE) dxf = ""; else
			dxf = ((char **)var8->addr)[i];

		/* compute the SPICE information */
		if (var5 == NOVARIABLE) spires = 0.0; else
			spires = ((float *)var5->addr)[i];
		if (var6 == NOVARIABLE) spicap = 0.0; else
			spicap = ((float *)var6->addr)[i];
		if (var7 == NOVARIABLE) spiecap = 0.0; else
			spiecap = ((float *)var7->addr)[i];

		/* build the layer facet */
		us_tecedmakelayer(np, desc->col, desc->raster,
			(INTSML)(desc->style[COLORMAP]&(NATURE|OUTLINEPAT)), lay, layerfunction(tech, (INTSML)i),
				us_layerletters(tech, (INTSML)i), dxf, gds, spires, spicap, spiecap);
	}

	/* save the layer sequence */
	sequence = (char **)emalloc(total * (sizeof (char *)), el_tempcluster);
	if (sequence == 0) return(NOLIBRARY);
	i = 0;
	for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		if (namesamen(np->cell->cellname, "layer-", 6) == 0)
			sequence[i++] = &np->cell->cellname[6];
	(void)setval((INTBIG)lib, VLIBRARY, "EDTEC_layersequence", (INTBIG)sequence,
		VSTRING|VISARRAY|(total<<VLENGTHSH));
	efree((char *)sequence);

	/* create the color map information */
	ttyputmsg("Adding color map and design rules...");
	var2 = getval((INTBIG)tech, VTECHNOLOGY, VCHAR|VISARRAY, "USER_color_map");
	varred = getvalkey((INTBIG)us_aid, VAID, VINTEGER|VISARRAY, us_colormap_red);
	vargreen = getvalkey((INTBIG)us_aid, VAID, VINTEGER|VISARRAY, us_colormap_green);
	varblue = getvalkey((INTBIG)us_aid, VAID, VINTEGER|VISARRAY, us_colormap_blue);
	if (varred != NOVARIABLE && vargreen != NOVARIABLE && varblue != NOVARIABLE &&
		var2 != NOVARIABLE)
	{
		newmap = emalloc((256*3*SIZEOFINTBIG), el_tempcluster);
		if (newmap == 0) return(NOLIBRARY);
		mapptr = newmap;
		colmap = (TECH_COLORMAP *)var2->addr;
		for(i=0; i<256; i++)
		{
			*mapptr++ = ((INTBIG *)varred->addr)[i];
			*mapptr++ = ((INTBIG *)vargreen->addr)[i];
			*mapptr++ = ((INTBIG *)varblue->addr)[i];
		}
		for(i=0; i<32; i++)
		{
			newmap[(i<<3)*3]   = colmap[i].red;
			newmap[(i<<3)*3+1] = colmap[i].green;
			newmap[(i<<3)*3+2] = colmap[i].blue;
		}
		(void)setval((INTBIG)lib, VLIBRARY, "EDTEC_colormap", (INTBIG)newmap,
			VINTEGER|VISARRAY|((256*3)<<VLENGTHSH));
		efree((char *)newmap);
	}

	/* create the design rule information */
	var2 = getval((INTBIG)tech, VTECHNOLOGY, VFRACT|VISARRAY, "DRC_min_unconnected_distances");
	var3 = getval((INTBIG)tech, VTECHNOLOGY, VFRACT|VISARRAY, "DRC_min_connected_distances");
	if (var2 != NOVARIABLE || var3 != NOVARIABLE)
	{
		/* design rules exist: count them */
		drccount = 0;
		if (var2 != NOVARIABLE)
		{
			l = getlength(var2);
			for(i=0; i<l; i++) if (((INTBIG *)var2->addr)[i] != XX) drccount++;
		}
		if (var3 != NOVARIABLE)
		{
			l = getlength(var3);
			for(i=0; i<l; i++) if (((INTBIG *)var3->addr)[i] != XX) drccount++;
		}

		/* nonzero amount: save it */
		if (drccount != 0)
		{
			drclist = (char **)emalloc((drccount * (sizeof (char *))), el_tempcluster);
			if (drclist == 0) return(NOLIBRARY);

			/* first do the unconnected list */
			drcindex = 0;
			if (var2 != NOVARIABLE)
			{
				k = 0;
				for(i=0; i<total; i++) for(j=i; j<total; j++)
				{
					if (((INTBIG *)var2->addr)[k] != XX)
					{
						(void)initinfstr();
						(void)addstringtoinfstr(frtoa(((INTBIG *)var2->addr)[k]));
						(void)addtoinfstr(' ');
						(void)addstringtoinfstr(us_tecedstripdash(layername(tech, (INTSML)i)));
						(void)addtoinfstr(' ');
						(void)addstringtoinfstr(us_tecedstripdash(layername(tech, (INTSML)j)));
						(void)allocstring(&drclist[drcindex], returninfstr(), el_tempcluster);
						drcindex++;
					}
					k++;
				}
			}

			/* now do the connected list */
			if (var3 != NOVARIABLE)
			{
				k = 0;
				for(i=0; i<total; i++) for(j=i; j<total; j++)
				{
					if (((INTBIG *)var3->addr)[k] != XX)
					{
						(void)initinfstr();
						(void)addtoinfstr('c');
						(void)addstringtoinfstr(frtoa(((INTBIG *)var3->addr)[k]));
						(void)addtoinfstr(' ');
						(void)addstringtoinfstr(us_tecedstripdash(layername(tech, (INTSML)i)));
						(void)addtoinfstr(' ');
						(void)addstringtoinfstr(us_tecedstripdash(layername(tech, (INTSML)j)));
						(void)allocstring(&drclist[drcindex], returninfstr(), el_tempcluster);
						drcindex++;
					}
					k++;
				}
			}

			/* save the design rules in the library object */
			(void)setval((INTBIG)lib, VLIBRARY, "EDTEC_DRC", (INTBIG)drclist,
				VSTRING|VISARRAY|(drccount<<VLENGTHSH));
			for(i=0; i<drccount; i++) efree(drclist[i]);
			efree((char *)drclist);
		}
	}

	/* create the arc nodes */
	ttyputmsg("Creating the arcs...");
	total = 0;
	for(ap = tech->firstarcproto; ap != NOARCPROTO; ap = ap->nextarcproto)
	{
		(void)initinfstr();
		(void)addstringtoinfstr("arc-");
		(void)addstringtoinfstr(us_tecedstripdash(ap->protoname));
		fname = returninfstr();

		/* make sure the arc doesn't exist */
		for(c = lib->firstcell; c != NOCELL; c = c->nextcell)
			if (namesame(fname, c->cellname) == 0)
		{
			ttyputerr("Warning: multiple arcs named '%s'", fname);
			break;
		}

		np = newnodeproto(fname, lib);
		if (np == NONODEPROTO) return(NOLIBRARY);
		ap->temp1 = (INTBIG)np;
		us_tecedmakearc(np, (ap->userbits&AFUNCTION)>>AFUNCTIONSH,
			ap->userbits&WANTFIXANG, ap->userbits&CANWIPE,
				ap->userbits&WANTNOEXTEND, (ap->userbits&AANGLEINC)>>AANGLEINCSH);

		/* now create the arc layers */
		ai = dummyarc();
		ai->proto = ap;
		ai->width = ap->nominalwidth * tech->deflambda / oldlam;
		ai->end[0].xpos = -60000;
		ai->end[0].ypos = -10000;
		ai->end[1].xpos = -30000;
		ai->end[1].ypos = -10000;
		ai->length = 30000;
		j = arcpolys(ai);
		for(i=0; i<j; i++)
		{
			shapearcpoly(ai, (INTSML)i, poly);
			if (poly->layer < 0) continue;
			desc = tech->layers[poly->layer];
			if (desc->bits == LAYERN) continue;

			/* create the node to describe this layer */
			ni = us_tecedplacegeom(poly, np);
			if (ni == NONODEINST) continue;

			/* get graphics for this layer */
			us_teceditsetpatch(ni, desc);
			(void)setval((INTBIG)ni, VNODEINST, "EDTEC_layer", (INTBIG)nplist[poly->layer], VNODEPROTO);
			(void)setval((INTBIG)ni, VNODEINST, "EDTEC_option", LAYERPATCH, VINTEGER);
		}
		i = (ap->nominalwidth*tech->deflambda/oldlam - arcwidthoffset(ap)) / 2;
		ni = newnodeinst(art_boxprim, -60000-i, -30000+i, -10000-i, -10000+i, 0, 0, np);
		if (ni == NONODEINST) return(NOLIBRARY);
		(void)setvalkey((INTBIG)ni, VNODEINST, art_colorkey, HIGHLIT, VINTEGER);
		(void)setval((INTBIG)ni, VNODEINST, "EDTEC_layer", (INTBIG)NONODEPROTO, VNODEPROTO);
		(void)setval((INTBIG)ni, VNODEINST, "EDTEC_option", LAYERPATCH, VINTEGER);
		total++;
	}

	/* save the arc sequence */
	sequence = (char **)emalloc(total * (sizeof (char *)), el_tempcluster);
	if (sequence == 0) return(NOLIBRARY);
	i = 0;
	for(ap = tech->firstarcproto; ap != NOARCPROTO; ap = ap->nextarcproto)
		sequence[i++] = &((NODEPROTO *)ap->temp1)->cell->cellname[4];
	(void)setval((INTBIG)lib, VLIBRARY, "EDTEC_arcsequence", (INTBIG)sequence,
		VSTRING|VISARRAY|(total<<VLENGTHSH));
	efree((char *)sequence);

	/* create the node nodes */
	ttyputmsg("Creating the nodes...");
	portdescript = defaulttextdescript();
	total = 0;
	for(pnp=tech->firstnodeproto; pnp != NONODEPROTO; pnp = pnp->nextnodeproto)
	{
		pnp->temp1 = 0;
		first = 1;

		/* create the node layers */
		oni = dummynode();
		oni->proto = pnp;
		xs = (pnp->highx - pnp->lowx) * 2 * tech->deflambda / oldlam;
		ys = (pnp->highy - pnp->lowy) * 2 * tech->deflambda / oldlam;
		nodexpos = -xs*2;
		xpos[0] = nodexpos - xs;
		xpos[1] = nodexpos + xs;
		xpos[2] = nodexpos - xs;
		xpos[3] = nodexpos + xs;
		ypos[0] = -10000 + ys;
		ypos[1] = -10000 + ys;
		ypos[2] = -10000 - ys;
		ypos[3] = -10000 - ys;
		nodesizeoffset(pnp, &lxp, &lyp, &hxp, &hyp);
		xs = (pnp->highx - pnp->lowx) * tech->deflambda / oldlam - lxp - hxp;
		ys = (pnp->highy - pnp->lowy) * tech->deflambda / oldlam - lyp - hyp;
		xsc[0] = 1;   ysc[0] = 1;
		xsc[1] = 2;   ysc[1] = 1;
		xsc[2] = 1;   ysc[2] = 2;
		xsc[3] = 2;   ysc[3] = 2;
		for(e=0; e<4; e++)
		{
			/* do not create node if main example had no polygons */
			if (e != 0 && first != 0) continue;

			/* square nodes have only two examples */
			if ((pnp->userbits&NSQUARE) != 0 && (e == 1 || e == 2)) continue;
			oni->lowx  = xpos[e] - xs*xsc[e]/2 - lxp;
			oni->lowy  = ypos[e] - ys*ysc[e]/2 - lyp;
			oni->highx = xpos[e] + xs*xsc[e]/2 + hxp;
			oni->highy = ypos[e] + ys*ysc[e]/2 + hyp;

			/* place the layers */
			j = nodepolys(oni);
			for(i=0; i<j; i++)
			{
				shapenodepoly(oni, (INTSML)i, poly);
				if (poly->layer < 0) continue;
				desc = tech->layers[poly->layer];
				if (desc->bits == LAYERN) continue;

				/* create the node facet on the first valid layer */
				if (first != 0)
				{
					first = 0;
					(void)initinfstr();
					(void)addstringtoinfstr("node-");
					(void)addstringtoinfstr(us_tecedstripdash(pnp->primname));
					fname = returninfstr();

					/* make sure the node doesn't exist */
					for(c = lib->firstcell; c != NOCELL; c = c->nextcell)
						if (namesame(fname, c->cellname) == 0)
					{
						ttyputerr("Warning: multiple nodes named '%s'", fname);
						break;
					}

					np = newnodeproto(fname, lib);
					if (np == NONODEPROTO) return(NOLIBRARY);
					func = (pnp->userbits&NFUNCTION)>>NFUNCTIONSH;
					serp = 0;
					if ((func == NPTRANMOS || func == NPTRAPMOS || func == NPTRADMOS) &&
						(pnp->userbits&HOLDSTRACE) != 0)
							serp = 1;
					us_tecedmakenode(np, func, (INTSML)serp, (INTSML)(pnp->userbits&NSQUARE),
						(INTSML)(pnp->userbits&WIPEON1OR2), (INTSML)(pnp->userbits&LOCKEDPRIM));
					total++;
					pnp->temp1 = (INTBIG)np;
				}

				/* create the node to describe this layer */
				ni = us_tecedplacegeom(poly, np);
				if (ni == NONODEINST) return(NOLIBRARY);

				/* get graphics for this layer */
				us_teceditsetpatch(ni, desc);
				(void)setval((INTBIG)ni, VNODEINST, "EDTEC_layer",
					(INTBIG)nplist[poly->layer], VNODEPROTO);
				(void)setval((INTBIG)ni, VNODEINST, "EDTEC_option", LAYERPATCH, VINTEGER);

				/* set minimum polygon factor on smallest example */
				if (e != 0) continue;
				if (i >= tech->nodeprotos[pnp->index-1]->layercount) continue;
				ll = tech->nodeprotos[pnp->index-1]->layerlist;
				if (ll == 0) continue;
				if (ll[i].representation != MINBOX) continue;
				var = setval((INTBIG)ni, VNODEINST, "EDTEC_minbox", (INTBIG)"MIN", VSTRING|VDISPLAY);
				if (var != NOVARIABLE) var->textdescript = TXTSMALL << VTSIZESH;
			}
			if (first != 0) continue;

			/* create the highlight node */
			ni = newnodeinst(art_boxprim, xpos[e]-xs*xsc[e]/2, xpos[e]+xs*xsc[e]/2,
				ypos[e]-ys*ysc[e]/2, ypos[e]+ys*ysc[e]/2, 0, 0, np);
			if (ni == NONODEINST) return(NOLIBRARY);
			(void)setvalkey((INTBIG)ni, VNODEINST, art_colorkey, HIGHLIT, VINTEGER);
			(void)setval((INTBIG)ni, VNODEINST, "EDTEC_layer", (INTBIG)NONODEPROTO, VNODEPROTO);
			(void)setval((INTBIG)ni, VNODEINST, "EDTEC_option", LAYERPATCH, VINTEGER);

			/* create a grab node (only in main example) */
			if (e == 0)
			{
				var = getvalkey((INTBIG)pnp, VNODEPROTO, VINTEGER|VISARRAY, el_prototype_center);
				if (var != NOVARIABLE)
				{
					lx = hx = xpos[0] + ((INTBIG *)var->addr)[0];
					ly = hy = ypos[0] + ((INTBIG *)var->addr)[1];
					nodesizeoffset(gen_facetcenterprim, &lxo, &lyo, &hxo, &hyo);
					ni = newnodeinst(gen_facetcenterprim, lx-lxo, hx+hxo, ly-lyo, hy+hyo, 0, 0, np);
					if (ni == NONODEINST) return(NOLIBRARY);
				}
			}

			/* also draw ports */
			for(pp = pnp->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
			{
				shapeportpoly(oni, pp, poly, 0);
				getbbox(poly, &lx, &hx, &ly, &hy);
				nodesizeoffset(gen_portprim, &lxo, &lyo, &hxo, &hyo);
				ni = newnodeinst(gen_portprim, lx-lxo, hx+hxo, ly-lyo, hy+hyo, 0, 0, np);
				if (ni == NONODEINST) return(NOLIBRARY);
				pp->temp1 = (INTBIG)ni;
				(void)setval((INTBIG)ni, VNODEINST, "EDTEC_option", LAYERPATCH, VINTEGER);
				var = setval((INTBIG)ni, VNODEINST, "EDTEC_portname", (INTBIG)pp->protoname,
					VSTRING|VDISPLAY);
				if (var != NOVARIABLE) var->textdescript = portdescript;

				/* on the first sample, also show angle and connection */
				if (e != 0) continue;
				if (((pp->userbits&PORTANGLE)>>PORTANGLESH) != 0 ||
					((pp->userbits&PORTARANGE)>>PORTARANGESH) != 180)
				{
					(void)setval((INTBIG)ni, VNODEINST, "EDTEC_portangle",
						(pp->userbits&PORTANGLE)>>PORTANGLESH, VINTEGER);
					(void)setval((INTBIG)ni, VNODEINST, "EDTEC_portrange",
						(pp->userbits&PORTARANGE)>>PORTARANGESH, VINTEGER);
				}

				/* add in the "local" port connections (from this tech) */
				for(tcon=i=0; pp->connects[i] != NOARCPROTO; i++)
					if (pp->connects[i]->tech == tech) tcon++;
				if (tcon != 0)
				{
					aplist = (NODEPROTO **)emalloc((tcon * (sizeof (NODEPROTO *))), el_tempcluster);
					if (aplist == 0) return(NOLIBRARY);
					for(j=i=0; pp->connects[i] != NOARCPROTO; i++)
						if (pp->connects[i]->tech == tech)
							aplist[j++] = (NODEPROTO *)pp->connects[i]->temp1;
					(void)setval((INTBIG)ni, VNODEINST, "EDTEC_connects",
						(INTBIG)aplist, VNODEPROTO|VISARRAY|(tcon<<VLENGTHSH));
					efree((char *)aplist);
				}

				/* connect the connected ports */
				for(opp = pnp->firstportproto; opp != pp; opp = opp->nextportproto)
				{
					if (opp->network != pp->network) continue;
					nni = (NODEINST *)opp->temp1;
					if (nni == NONODEINST) continue;
					if (newarcinst(gen_universalarc, 0, 0, ni, ni->proto->firstportproto,
						(ni->highx+ni->lowx)/2, (ni->highy+ni->lowy)/2, nni,
							nni->proto->firstportproto, (nni->highx+nni->lowx)/2,
								(nni->highy+nni->lowy)/2, np) == NOARCINST) return(NOLIBRARY);
					break;
				}
			}
		}
	}

	/* save the node sequence */
	sequence = (char **)emalloc(total * (sizeof (char *)), el_tempcluster);
	if (sequence == 0) return(NOLIBRARY);
	i = 0;
	for(np = tech->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		if (np->temp1 != 0)
			sequence[i++] = &((NODEPROTO *)np->temp1)->cell->cellname[5];
	(void)setval((INTBIG)lib, VLIBRARY, "EDTEC_nodesequence", (INTBIG)sequence,
		VSTRING|VISARRAY|(total<<VLENGTHSH));
	efree((char *)sequence);

	/* clean up */
	ttyputmsg("Done.");
	efree((char *)nplist);
	tech->deflambda = oldlam;
	return(lib);
}

/*************************** FACET CREATION HELPERS ***************************/

/*
 * routine to build the appropriate descriptive information for a layer into
 * facet "np".  The color is "colorindex"; the stipple array is in "stip"; the
 * layer style is in "style", the CIF layer is in "ciflayer"; the function is
 * in "functionindex"; the layer letters are in "layerletters"; the DXF layer name(s)
 * are "dxf"; the Calma GDS-II layer is in "gds"; the SPICE resistance is in "spires",
 * the SPICE capacitance is in "spicap", and the SPICE edge capacitance is in "spiecap".
 */
void us_tecedmakelayer(NODEPROTO *np, INTSML colorindex, UINTSML stip[8], INTSML style,
	char *ciflayer, INTBIG functionindex, char *layerletters, char *dxf,
	INTBIG gds, float spires, float spicap, float spiecap)
{
	REGISTER NODEINST *nicolor, *ni, *laycolor, *laystipple, *laystyle, *laycif,
		*laydxf, *laycalma, *layfunction, *layletters, *layspires, *layspicap, *layspiecap;
	REGISTER INTSML i, x, y;
	REGISTER VARIABLE *var;
	char line[20], *colorname, *colorsymbol;
	UINTBIG pattern[8];
	UINTSML spattern[8];

	laycolor = laystipple = laystyle = laycif = laydxf = laycalma = NONODEINST;
	layfunction = layletters = layspires = layspicap = layspiecap = NONODEINST;
	for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		var = getval((INTBIG)ni, VNODEINST, VINTEGER, "EDTEC_option");
		if (var == NOVARIABLE) continue;
		switch (var->addr)
		{
			case LAYERCOLOR:    laycolor = ni;      break;
			case LAYERPATTERN:  laystipple = ni;    break;
			case LAYERSTYLE:    laystyle = ni;      break;
			case LAYERCIF:      laycif = ni;        break;
			case LAYERDXF:      laydxf = ni;        break;
			case LAYERGDS:      laycalma = ni;      break;
			case LAYERFUNCTION: layfunction = ni;   break;
			case LAYERLETTERS:  layletters = ni;    break;
			case LAYERSPIRES:   layspires = ni;     break;
			case LAYERSPICAP:   layspicap = ni;     break;
			case LAYERSPIECAP:  layspiecap = ni;    break;
		}
	}

	/* create the color information if it is not there */
	if (laycolor == NONODEINST)
	{
		/* create the graphic color object */
		nicolor = newnodeinst(art_filledboxprim, -20000, -10000, 10000, 20000, 0, 0, np);
		if (nicolor == NONODEINST) return;
		(void)setvalkey((INTBIG)nicolor, VNODEINST, art_colorkey, colorindex, VINTEGER);
		(void)setval((INTBIG)np, VNODEPROTO, "EDTEC_colornode", (INTBIG)nicolor, VNODEINST);
		if ((style&NATURE) == PATTERNED)
		{
			if ((style&OUTLINEPAT) == 0)
			{
				for(i=0; i<8; i++) pattern[i] = stip[i];
				(void)setvalkey((INTBIG)nicolor, VNODEINST, art_patternkey, (INTBIG)pattern,
					VINTEGER|VISARRAY|(8<<VLENGTHSH));
			} else
			{
				for(i=0; i<8; i++) spattern[i] = stip[i];
				(void)setvalkey((INTBIG)nicolor, VNODEINST, art_patternkey, (INTBIG)spattern,
					VSHORT|VISARRAY|(8<<VLENGTHSH));
			}
		}

		/* create the text color object */
		ni = newnodeinst(gen_invispinprim, 0, 40000, 22000, 26000, 0, 0, np);
		if (ni == NONODEINST) return;
		(void)initinfstr();
		(void)addstringtoinfstr(TECEDNODETEXTCOLOR);
		if (ecolorname(colorindex, &colorname, &colorsymbol) != 0) colorname = "unknown";
		(void)addstringtoinfstr(colorname);
		var = setvalkey((INTBIG)ni, VNODEINST, art_messagekey, (INTBIG)returninfstr(),
			VSTRING|VDISPLAY);
		if (var != NOVARIABLE) var->textdescript = TXTMEDIUM << VTSIZESH;
		(void)setval((INTBIG)ni, VNODEINST, "EDTEC_option", LAYERCOLOR, VINTEGER);
	}

	/* create the stipple pattern objects if it is not there */
	if (laystipple == NONODEINST)
	{
		for(x=0; x<16; x++) for(y=0; y<8; y++)
		{
			if ((stip[y] & (1 << (15-x))) != 0)
				ni = newnodeinst(art_filledboxprim, x*2000-40000, x*2000-38000,
					4000-y*2000, 6000-y*2000, 0, 0, np); else
				ni = newnodeinst(art_boxprim, x*2000-40000, x*2000-38000, 4000-y*2000,
					6000-y*2000, 0, 0, np);
			if (ni == NONODEINST) return;
			(void)setval((INTBIG)ni, VNODEINST, "EDTEC_option", LAYERPATTERN, VINTEGER);
		}
		ni = newnodeinst(gen_invispinprim, -40000, -8000, 6000, 8000, 0, 0, np);
		if (ni == NONODEINST) return;
		var = setvalkey((INTBIG)ni, VNODEINST, art_messagekey, (INTBIG)"Stipple Pattern",
			VSTRING|VDISPLAY);
		if (var != NOVARIABLE) var->textdescript = TXTMEDIUM << VTSIZESH;
	}

	/* create the color style object */
	if (laystyle == NONODEINST)
	{
		ni = newnodeinst(gen_invispinprim, 0, 40000, 16000, 20000, 0, 0, np);
		if (ni == NONODEINST) return;
		(void)initinfstr();
		(void)addstringtoinfstr(TECEDNODETEXTSTYLE);
		if ((style&NATURE) == SOLIDC) (void)addstringtoinfstr("solid"); else
		{
			if ((style&OUTLINEPAT) != 0) (void)addstringtoinfstr("patterned/outlined"); else
				(void)addstringtoinfstr("patterned");
		}
		var = setvalkey((INTBIG)ni, VNODEINST, art_messagekey, (INTBIG)returninfstr(),
			VSTRING|VDISPLAY);
		if (var != NOVARIABLE) var->textdescript = TXTMEDIUM << VTSIZESH;
		(void)setval((INTBIG)ni, VNODEINST, "EDTEC_option", LAYERSTYLE, VINTEGER);
	}

	/* create the CIF layer object */
	if (laycif == NONODEINST)
	{
		ni = newnodeinst(gen_invispinprim, 0, 40000, 10000, 14000, 0, 0, np);
		if (ni == NONODEINST) return;
		(void)initinfstr();
		(void)addstringtoinfstr(TECEDNODETEXTCIF);
		(void)addstringtoinfstr(ciflayer);
		var = setvalkey((INTBIG)ni, VNODEINST, art_messagekey, (INTBIG)returninfstr(),
			VSTRING|VDISPLAY);
		if (var != NOVARIABLE) var->textdescript = TXTMEDIUM << VTSIZESH;
		(void)setval((INTBIG)ni, VNODEINST, "EDTEC_option", LAYERCIF, VINTEGER);
	}

	/* create the DXF layer object */
	if (laydxf == NONODEINST)
	{
		ni = newnodeinst(gen_invispinprim, 0, 40000, 4000, 8000, 0, 0, np);
		if (ni == NONODEINST) return;
		(void)initinfstr();
		(void)addstringtoinfstr(TECEDNODETEXTDXF);
		(void)addstringtoinfstr(dxf);
		var = setvalkey((INTBIG)ni, VNODEINST, art_messagekey, (INTBIG)returninfstr(),
			VSTRING|VDISPLAY);
		if (var != NOVARIABLE) var->textdescript = TXTMEDIUM << VTSIZESH;
		(void)setval((INTBIG)ni, VNODEINST, "EDTEC_option", LAYERDXF, VINTEGER);
	}

	/* create the Calma GDS-II layer object */
	if (laycalma == NONODEINST)
	{
		ni = newnodeinst(gen_invispinprim, 0, 40000, -2000, 2000, 0, 0, np);
		if (ni == NONODEINST) return;
		(void)initinfstr();
		(void)addstringtoinfstr(TECEDNODETEXTGDS);
		(void)sprintf(line, "%d", gds);
		(void)addstringtoinfstr(line);
		var = setvalkey((INTBIG)ni, VNODEINST, art_messagekey, (INTBIG)returninfstr(),
			VSTRING|VDISPLAY);
		if (var != NOVARIABLE) var->textdescript = TXTMEDIUM << VTSIZESH;
		(void)setval((INTBIG)ni, VNODEINST, "EDTEC_option", LAYERGDS, VINTEGER);
	}

	/* create the Function object */
	if (layfunction == NONODEINST)
	{
		ni = newnodeinst(gen_invispinprim, 0, 40000, -8000, -4000, 0, 0, np);
		if (ni == NONODEINST) return;
		(void)initinfstr();
		(void)addstringtoinfstr(TECEDNODETEXTFUNCTION);
		us_tecedaddfunstring(functionindex);
		var = setvalkey((INTBIG)ni, VNODEINST, art_messagekey, (INTBIG)returninfstr(),
			VSTRING|VDISPLAY);
		if (var != NOVARIABLE) var->textdescript = TXTMEDIUM << VTSIZESH;
		(void)setval((INTBIG)ni, VNODEINST, "EDTEC_option", LAYERFUNCTION, VINTEGER);
	}

	/* create the Layer Letters object */
	if (layletters == NONODEINST)
	{
		ni = newnodeinst(gen_invispinprim, 0, 40000, -14000, -10000, 0, 0, np);
		if (ni == NONODEINST) return;
		(void)initinfstr();
		(void)addstringtoinfstr(TECEDNODETEXTLETTERS);
		(void)addstringtoinfstr(layerletters);
		var = setvalkey((INTBIG)ni, VNODEINST, art_messagekey, (INTBIG)returninfstr(),
			VSTRING|VDISPLAY);
		if (var != NOVARIABLE) var->textdescript = TXTMEDIUM << VTSIZESH;
		(void)setval((INTBIG)ni, VNODEINST, "EDTEC_option", LAYERLETTERS, VINTEGER);
	}

	/* create the SPICE resistance object */
	if (layspires == NONODEINST)
	{
		ni = newnodeinst(gen_invispinprim, 0, 40000, -20000, -16000, 0, 0, np);
		if (ni == NONODEINST) return;
		(void)initinfstr();
		(void)addstringtoinfstr(TECEDNODETEXTSPICERES);
		(void)sprintf(line, "%g", spires);
		(void)addstringtoinfstr(line);
		var = setvalkey((INTBIG)ni, VNODEINST, art_messagekey, (INTBIG)returninfstr(),
			VSTRING|VDISPLAY);
		if (var != NOVARIABLE) var->textdescript = TXTMEDIUM << VTSIZESH;
		(void)setval((INTBIG)ni, VNODEINST, "EDTEC_option", LAYERSPIRES, VINTEGER);
	}

	/* create the SPICE capacitance object */
	if (layspicap == NONODEINST)
	{
		ni = newnodeinst(gen_invispinprim, 0, 40000, -26000, -22000, 0, 0, np);
		if (ni == NONODEINST) return;
		(void)initinfstr();
		(void)addstringtoinfstr(TECEDNODETEXTSPICECAP);
		(void)sprintf(line, "%g", spicap);
		(void)addstringtoinfstr(line);
		var = setvalkey((INTBIG)ni, VNODEINST, art_messagekey, (INTBIG)returninfstr(),
			VSTRING|VDISPLAY);
		if (var != NOVARIABLE) var->textdescript = TXTMEDIUM << VTSIZESH;
		(void)setval((INTBIG)ni, VNODEINST, "EDTEC_option", LAYERSPICAP, VINTEGER);
	}

	/* create the SPICE edge capacitance object */
	if (layspiecap == NONODEINST)
	{
		ni = newnodeinst(gen_invispinprim, 0, 40000, -32000, -28000, 0, 0, np);
		if (ni == NONODEINST) return;
		(void)initinfstr();
		(void)addstringtoinfstr(TECEDNODETEXTSPICEECAP);
		(void)sprintf(line, "%g", spiecap);
		(void)addstringtoinfstr(line);
		var = setvalkey((INTBIG)ni, VNODEINST, art_messagekey, (INTBIG)returninfstr(),
			VSTRING|VDISPLAY);
		if (var != NOVARIABLE) var->textdescript = TXTMEDIUM << VTSIZESH;
		(void)setval((INTBIG)ni, VNODEINST, "EDTEC_option", LAYERSPIECAP, VINTEGER);
	}
}

/*
 * routine to add the text corresponding to the layer function in "func"
 * to the current infinite string
 */
void us_tecedaddfunstring(INTBIG func)
{
	REGISTER INTSML i;

	(void)addstringtoinfstr(us_teclayer_functions[func&LFTYPE].name);
	for(i=0; us_teclayer_functions[i].name != 0; i++)
	{
		if (us_teclayer_functions[i].value <= LFTYPE) continue;
		if ((func&us_teclayer_functions[i].value) == 0) continue;
		func &= ~us_teclayer_functions[i].value;
		(void)addtoinfstr(',');
		(void)addstringtoinfstr(us_teclayer_functions[i].name);
	}
}

/*
 * routine to build the appropriate descriptive information for an arc into
 * facet "np".  The function is in "func"; the arc is fixed-angle if "fixang"
 * is nonzero; the arc wipes pins if "wipes" is nonzero; and the arc does
 * not extend its ends if "noextend" is nonzero.  The angle increment is
 * in "anginc".
 */
void us_tecedmakearc(NODEPROTO *np, INTBIG func, INTBIG fixang, INTBIG wipes, INTBIG noextend,
	INTBIG anginc)
{
	REGISTER NODEINST *ni;
	REGISTER VARIABLE *var;
	REGISTER char *str;
	char line[50];

	/* create the Function object */
	ni = newnodeinst(gen_invispinprim, 0, 40000, -2000, 2000, 0, 0, np);
	if (ni == NONODEINST) return;
	(void)initinfstr();
	(void)addstringtoinfstr(TECEDNODETEXTFUNCTION);
	(void)addstringtoinfstr(us_tecarc_functions[func].name);
	var = setvalkey((INTBIG)ni, VNODEINST, art_messagekey, (INTBIG)returninfstr(), VSTRING|VDISPLAY);
	if (var != NOVARIABLE) var->textdescript = TXTMEDIUM << VTSIZESH;
	(void)setval((INTBIG)ni, VNODEINST, "EDTEC_option", ARCFUNCTION, VINTEGER);

	/* create the Fixed-angle object */
	ni = newnodeinst(gen_invispinprim, 0, 40000, -8000, -4000, 0, 0, np);
	if (ni == NONODEINST) return;
	if (fixang != 0) str = "Fixed-angle: Yes"; else str = "Fixed-angle: No";
	var = setvalkey((INTBIG)ni, VNODEINST, art_messagekey, (INTBIG)str, VSTRING|VDISPLAY);
	if (var != NOVARIABLE) var->textdescript = TXTMEDIUM << VTSIZESH;
	(void)setval((INTBIG)ni, VNODEINST, "EDTEC_option", ARCFIXANG, VINTEGER);

	/* create the Wipe object */
	ni = newnodeinst(gen_invispinprim, 0, 40000, -14000, -10000, 0, 0, np);
	if (ni == NONODEINST) return;
	if (wipes != 0) str = "Wipes pins: Yes"; else str = "Wipes pins: No";
	var = setvalkey((INTBIG)ni, VNODEINST, art_messagekey, (INTBIG)str, VSTRING|VDISPLAY);
	if (var != NOVARIABLE) var->textdescript = TXTMEDIUM << VTSIZESH;
	(void)setval((INTBIG)ni, VNODEINST, "EDTEC_option", ARCWIPESPINS, VINTEGER);

	/* create the Noextend object */
	ni = newnodeinst(gen_invispinprim, 0, 40000, -20000, -16000, 0, 0, np);
	if (ni == NONODEINST) return;
	if (noextend == 0) str = "Extend arcs: Yes"; else str = "Extend arcs: No";
	var = setvalkey((INTBIG)ni, VNODEINST, art_messagekey, (INTBIG)str, VSTRING|VDISPLAY);
	if (var != NOVARIABLE) var->textdescript = TXTMEDIUM << VTSIZESH;
	(void)setval((INTBIG)ni, VNODEINST, "EDTEC_option", ARCNOEXTEND, VINTEGER);

	/* create the Angle Increment object */
	ni = newnodeinst(gen_invispinprim, 0, 40000, -26000, -22000, 0, 0, np);
	if (ni == NONODEINST) return;
	(void)sprintf(line, "Angle increment: %d", anginc);
	var = setvalkey((INTBIG)ni, VNODEINST, art_messagekey, (INTBIG)line, VSTRING|VDISPLAY);
	if (var != NOVARIABLE) var->textdescript = TXTMEDIUM << VTSIZESH;
	(void)setval((INTBIG)ni, VNODEINST, "EDTEC_option", ARCINC, VINTEGER);
}

/*
 * routine to build the appropriate descriptive information for a node into
 * facet "np".  The function is in "func", the serpentine transistor factor
 * is in "serp", the node is square if "square" is nonzero, the node
 * is invisible on 1 or 2 arcs if "wipes" is nonzero, and the node is lockable
 * if "lockable" is nonzero.
 */
void us_tecedmakenode(NODEPROTO *np, INTBIG func, INTSML serp, INTSML square, INTSML wipes,
	INTSML lockable)
{
	REGISTER NODEINST *ni, *nifunc, *niserp, *nisquare, *niwipes, *nilockable;
	REGISTER char *str;
	REGISTER VARIABLE *var;

	nifunc = niserp = nisquare = niwipes = nilockable = NONODEINST;
	for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		var = getval((INTBIG)ni, VNODEINST, VINTEGER, "EDTEC_option");
		if (var == NOVARIABLE) continue;
		switch (var->addr)
		{
			case NODEFUNCTION:   nifunc = ni;       break;
			case NODESERPENTINE: niserp = ni;       break;
			case NODESQUARE:     nisquare = ni;     break;
			case NODEWIPES:      niwipes = ni;      break;
			case NODELOCKABLE:   nilockable = ni;   break;
		}
	}

	/* create the Function object if it is not there */
	if (nifunc == NONODEINST)
	{
		ni = newnodeinst(gen_invispinprim, 0, 40000, -2000, 2000, 0, 0, np);
		if (ni == NONODEINST) return;
		(void)initinfstr();
		(void)addstringtoinfstr(TECEDNODETEXTFUNCTION);
		(void)addstringtoinfstr(us_tecnode_functions[func].name);
		var = setvalkey((INTBIG)ni, VNODEINST, art_messagekey, (INTBIG)returninfstr(),
			VSTRING|VDISPLAY);
		if (var != NOVARIABLE) var->textdescript = TXTMEDIUM << VTSIZESH;
		(void)setval((INTBIG)ni, VNODEINST, "EDTEC_option", NODEFUNCTION, VINTEGER);
	}

	/* create the Serpentine Transistor object if it is not there */
	if (niserp == NONODEINST)
	{
		ni = newnodeinst(gen_invispinprim, 0, 40000, -8000, -4000, 0, 0, np);
		if (ni == NONODEINST) return;
		if (serp != 0) str = "Serpentine transistor: yes"; else
			str = "Serpentine transistor: no";
		var = setvalkey((INTBIG)ni, VNODEINST, art_messagekey, (INTBIG)str, VSTRING|VDISPLAY);
		if (var != NOVARIABLE) var->textdescript = TXTMEDIUM << VTSIZESH;
		(void)setval((INTBIG)ni, VNODEINST, "EDTEC_option", NODESERPENTINE, VINTEGER);
	}

	/* create the Square Node object if it is not there */
	if (nisquare == NONODEINST)
	{
		ni = newnodeinst(gen_invispinprim, 0, 40000, -14000, -10000, 0, 0, np);
		if (ni == NONODEINST) return;
		if (square != 0) str = "Square node: yes"; else
			str = "Square node: no";
		var = setvalkey((INTBIG)ni, VNODEINST, art_messagekey, (INTBIG)str, VSTRING|VDISPLAY);
		if (var != NOVARIABLE) var->textdescript = TXTMEDIUM << VTSIZESH;
		(void)setval((INTBIG)ni, VNODEINST, "EDTEC_option", NODESQUARE, VINTEGER);
	}

	/* create the Wipes Node object if it is not there */
	if (niwipes == NONODEINST)
	{
		ni = newnodeinst(gen_invispinprim, 0, 40000, -20000, -16000, 0, 0, np);
		if (ni == NONODEINST) return;
		if (wipes != 0) str = "Invisible with 1 or 2 arcs: yes"; else
			str = "Invisible with 1 or 2 arcs: no";
		var = setvalkey((INTBIG)ni, VNODEINST, art_messagekey, (INTBIG)str, VSTRING|VDISPLAY);
		if (var != NOVARIABLE) var->textdescript = TXTMEDIUM << VTSIZESH;
		(void)setval((INTBIG)ni, VNODEINST, "EDTEC_option", NODEWIPES, VINTEGER);
	}

	/* create the Lockable Node object if it is not there */
	if (nilockable == NONODEINST)
	{
		ni = newnodeinst(gen_invispinprim, 0, 40000, -26000, -22000, 0, 0, np);
		if (ni == NONODEINST) return;
		if (lockable != 0) str = "Lockable: yes"; else str = "Lockable: no";
		var = setvalkey((INTBIG)ni, VNODEINST, art_messagekey, (INTBIG)str, VSTRING|VDISPLAY);
		if (var != NOVARIABLE) var->textdescript = TXTMEDIUM << VTSIZESH;
		(void)setval((INTBIG)ni, VNODEINST, "EDTEC_option", NODELOCKABLE, VINTEGER);
	}
}

/*
 * routine to build the appropriate descriptive information for the information
 * facet "np".
 */
void us_tecedmakeinfo(NODEPROTO *np, INTBIG lambda, char *description)
{
	REGISTER NODEINST *ni, *nilambda, *nidescript;
	char line[50];
	REGISTER VARIABLE *var;

	nilambda = nidescript = NONODEINST;
	for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		var = getval((INTBIG)ni, VNODEINST, VINTEGER, "EDTEC_option");
		if (var == NOVARIABLE) continue;
		switch (var->addr)
		{
			case TECHLAMBDA:   nilambda = ni;     break;
			case TECHDESCRIPT: nidescript = ni;   break;
		}
	}

	/* create the Lambda object if it is not there */
	if (nilambda == NONODEINST)
	{
		ni = newnodeinst(gen_invispinprim, 0, 40000, -2000, 2000, 0, 0, np);
		if (ni == NONODEINST) return;
		(void)sprintf(line, "Lambda: %d", lambda);
		var = setvalkey((INTBIG)ni, VNODEINST, art_messagekey, (INTBIG)line, VSTRING|VDISPLAY);
		if (var != NOVARIABLE) var->textdescript = TXTMEDIUM << VTSIZESH;
		(void)setval((INTBIG)ni, VNODEINST, "EDTEC_option", TECHLAMBDA, VINTEGER);
	}

	/* create the Description object if it is not there */
	if (nidescript == NONODEINST)
	{
		ni = newnodeinst(gen_invispinprim, 0, 40000, -8000, -4000, 0, 0, np);
		if (ni == NONODEINST) return;
		(void)initinfstr();
		(void)addstringtoinfstr("Description: ");
		(void)addstringtoinfstr(description);
		var = setvalkey((INTBIG)ni, VNODEINST, art_messagekey, (INTBIG)returninfstr(),
			VSTRING|VDISPLAY);
		if (var != NOVARIABLE) var->textdescript = TXTMEDIUM << VTSIZESH;
		(void)setval((INTBIG)ni, VNODEINST, "EDTEC_option", TECHDESCRIPT, VINTEGER);
	}
}

/*
 * routine to convert the dashes and other illegal characters in "str" to
 * the underscore character, then return the string
 */
char *us_tecedstripdash(char *str)
{
	(void)initinfstr();
	for( ; *str != 0; str++)
	{
		if (isalnum(*str)) (void)addtoinfstr(*str); else
		{
			switch (*str)
			{
				case '+': (void)addtoinfstr('P');   break;
				default:  (void)addtoinfstr('_');   break;
			}
		}
	}
	return(returninfstr());
}

NODEINST *us_tecedplacegeom(POLYGON *poly, NODEPROTO *np)
{
	INTBIG lx, hx, ly, hy, dummy;
	REGISTER NODEINST *nni;
	REGISTER VARIABLE *var;

	getbbox(poly, &lx, &hx, &ly, &hy);
	switch (poly->style)
	{
		case FILLED:
			if (isbox(poly, &dummy, &dummy, &dummy, &dummy) == 0)
			{
				nni = newnodeinst(art_filledpolygonprim, lx, hx, ly, hy, 0, 0, np);
				if (nni == NONODEINST) return(NONODEINST);
				us_tecedsetlist(nni, poly, lx, hx, ly, hy);
				endobjectchange((INTBIG)nni, VNODEINST);
				return(nni);
			}
		case FILLEDRECT:
			nni = newnodeinst(art_filledboxprim, lx, hx, ly, hy, 0, 0, np);
			if (nni == NONODEINST) return(NONODEINST);
			endobjectchange((INTBIG)nni, VNODEINST);
			return(nni);
		case CLOSED:
			if (isbox(poly, &dummy, &dummy, &dummy, &dummy) == 0)
			{
				nni = newnodeinst(art_closedpolygonprim, lx, hx, ly, hy, 0, 0, np);
				if (nni == NONODEINST) return(NONODEINST);
				us_tecedsetlist(nni, poly, lx, hx, ly, hy);
				endobjectchange((INTBIG)nni, VNODEINST);
				return(nni);
			}
		case CLOSEDRECT:
			nni = newnodeinst(art_boxprim, lx, hx, ly, hy, 0, 0, np);
			if (nni == NONODEINST) return(NONODEINST);
			endobjectchange((INTBIG)nni, VNODEINST);
			return(nni);
		case CROSSED:
			nni = newnodeinst(art_crossedboxprim, lx, hx, ly, hy, 0,0, np);
			if (nni == NONODEINST) return(NONODEINST);
			endobjectchange((INTBIG)nni, VNODEINST);
			return(nni);
		case OPENED:
			nni = newnodeinst(art_openedpolygonprim, lx, hx, ly, hy, 0, 0, np);
			if (nni == NONODEINST) return(NONODEINST);
			us_tecedsetlist(nni, poly, lx, hx, ly, hy);
			endobjectchange((INTBIG)nni, VNODEINST);
			return(nni);
		case OPENEDT1:
			nni = newnodeinst(art_openeddottedpolygonprim, lx, hx, ly, hy, 0, 0, np);
			if (nni == NONODEINST) return(NONODEINST);
			us_tecedsetlist(nni, poly, lx, hx, ly, hy);
			endobjectchange((INTBIG)nni, VNODEINST);
			return(nni);
		case OPENEDT2:
			nni = newnodeinst(art_openeddashedpolygonprim, lx, hx, ly, hy, 0, 0, np);
			if (nni == NONODEINST) return(NONODEINST);
			us_tecedsetlist(nni, poly, lx, hx, ly, hy);
			endobjectchange((INTBIG)nni, VNODEINST);
			return(nni);
		case OPENEDT3:
			nni = newnodeinst(art_openedfardottepolygonprim, lx, hx, ly, hy, 0, 0, np);
			if (nni == NONODEINST) return(NONODEINST);
			us_tecedsetlist(nni, poly, lx, hx, ly, hy);
			endobjectchange((INTBIG)nni, VNODEINST);
			return(nni);
		case CIRCLE:
			nni = newnodeinst(art_circleprim, lx, hx, ly, hy, 0, 0, np);
			if (nni == NONODEINST) return(NONODEINST);
			endobjectchange((INTBIG)nni, VNODEINST);
			return(nni);
		case DISC:
			nni = newnodeinst(art_filledcircleprim, lx, hx, ly, hy, 0, 0, np);
			if (nni == NONODEINST) return(NONODEINST);
			endobjectchange((INTBIG)nni, VNODEINST);
			return(nni);
		case CIRCLEARC:
			nni = newnodeinst(art_circleprim, lx, hx, ly, hy, 0, 0, np);
			if (nni == NONODEINST) return(NONODEINST);
			setarcdegrees(nni, 0.0, 45.0*EPI/180.0);
			endobjectchange((INTBIG)nni, VNODEINST);
			return(nni);
		case TEXTCENT:
			nni = newnodeinst(gen_invispinprim, lx, hx, ly, hy, 0, 0, np);
			if (nni == NONODEINST) return(NONODEINST);
			var = setvalkey((INTBIG)nni, VNODEINST, art_messagekey,
				(INTBIG)poly->string, VSTRING|VDISPLAY);
			if (var != NOVARIABLE) var->textdescript = VTPOSCENT;
			endobjectchange((INTBIG)nni, VNODEINST);
			return(nni);
		case TEXTBOTLEFT:
			nni = newnodeinst(gen_invispinprim, lx, hx, ly, hy, 0, 0, np);
			if (nni == NONODEINST) return(NONODEINST);
			var = setvalkey((INTBIG)nni, VNODEINST, art_messagekey,
				(INTBIG)poly->string, VSTRING|VDISPLAY);
			if (var != NOVARIABLE) var->textdescript = VTPOSUPRIGHT;
			endobjectchange((INTBIG)nni, VNODEINST);
			return(nni);
		case TEXTBOTRIGHT:
			nni = newnodeinst(gen_invispinprim, lx, hx, ly, hy, 0, 0, np);
			if (nni == NONODEINST) return(NONODEINST);
			var = setvalkey((INTBIG)nni, VNODEINST, art_messagekey,
				(INTBIG)poly->string, VSTRING|VDISPLAY);
			if (var != NOVARIABLE) var->textdescript = VTPOSUPLEFT;
			endobjectchange((INTBIG)nni, VNODEINST);
			return(nni);
		case TEXTBOX:
			nni = newnodeinst(gen_invispinprim, lx, hx, ly, hy, 0, 0, np);
			if (nni == NONODEINST) return(NONODEINST);
			var = setvalkey((INTBIG)nni, VNODEINST, art_messagekey,
				(INTBIG)poly->string, VSTRING|VDISPLAY);
			if (var != NOVARIABLE) var->textdescript = VTPOSBOXED;
			endobjectchange((INTBIG)nni, VNODEINST);
			return(nni);
	}
	return(NONODEINST);
}

void us_tecedsetlist(NODEINST *ni, POLYGON *poly, INTBIG lx, INTBIG hx, INTBIG ly, INTBIG hy)
{
	REGISTER INTBIG *list;
	REGISTER INTSML i;

	list = emalloc((poly->count*2*SIZEOFINTBIG), el_tempcluster);
	if (list == 0) return;
	for(i=0; i<poly->count; i++)
	{
		list[i*2] = poly->xv[i] - (hx+lx)/2;
		list[i*2+1] = poly->yv[i] - (hy+ly)/2;
	}
	(void)setvalkey((INTBIG)ni, VNODEINST, el_trace, (INTBIG)list,
		VINTEGER|VISARRAY|((poly->count*2)<<VLENGTHSH));
	efree((char *)list);
}

/****************************** WRITE TECHNOLOGY ******************************/

/*
 * routine to dump the layer information in technology "tech" to the stream in
 * "f".
 */
void us_teceditdumplayers(FILE *f, TECHNOLOGY *tech)
{
	char *sym, *colorname, *colorsymbol;
	REGISTER INTSML i, j, k, l;
	REGISTER char *l1, *l2, *l3, *l4, *l5;

	/* write information for "tectable.c" */
	xprintf(f, "#ifdef NEVER\n");
	xprintf(f, "/* next 4 lines belong at the top of 'tectable.c': */\n");
	xprintf(f, "extern GRAPHICS *%s_layers[];\n", us_tecedmakesymbol(tech->techname));
	xprintf(f, "extern TECH_ARCS *%s_arcprotos[];\n", us_tecedmakesymbol(tech->techname));
	xprintf(f, "extern TECH_NODES *%s_nodeprotos[];\n", us_tecedmakesymbol(tech->techname));
	xprintf(f, "extern TECH_VARIABLES %s_variables[];\n", us_tecedmakesymbol(tech->techname));
	xprintf(f, "\n/* next 4 lines belong in array of 'tectable.c': */\n");
	xprintf(f, "\t\"%s\", 0, %d, NONODEPROTO,NOARCPROTO,NOVARIABLE,",
		us_tecedmakesymbol(tech->techname), tech->deflambda);
	xprintf(f, "0,NOCOMCOMP,NOCLUSTER,\n");
	xprintf(f, "\t{\"%s\",\n", tech->techdescript);
	xprintf(f, "\t0, %s_layers,", us_tecedmakesymbol(tech->techname));
	xprintf(f, " 0, %s_arcprotos,", us_tecedmakesymbol(tech->techname));
	xprintf(f, " 0, %s_nodeprotos,", us_tecedmakesymbol(tech->techname));
	xprintf(f, " %s_variables,\n", us_tecedmakesymbol(tech->techname));
	xprintf(f, "\t0, 0, 0, 0, 0, 0, 0, 0, 0, NOTECHNOLOGY,");
	xprintf(f, " NONEGATEDARCS, 0, 0,}\n");
	xprintf(f, "#endif\n");

	/* write legal banner */
	xprintf(f, "/*\n");
	xprintf(f, " * Electric Design System\n");
	xprintf(f, " *\n");
	xprintf(f, " * File: TECHNOLOGY.c\n");
	xprintf(f, " * %s technology description\n", tech->techname);
	xprintf(f, " * Generated automatically from a library\n");
	xprintf(f, " *\n");
	xprintf(f, " * (c) Electric Editor, Incorporated 1993-7.\n");
	xprintf(f, " * All or a portion of this program is subject to the restrictions\n");
	xprintf(f, " * contained in a license granted by, and remains the unpublished\n");
	xprintf(f, " * property of, Electric Editor, Incorporated\n");
	xprintf(f, " */\n");

	/* write header */
	xprintf(f, "#include \"global.h\"\n");
	xprintf(f, "#include \"egraphics.h\"\n");
	xprintf(f, "#include \"tech.h\"\n");
	xprintf(f, "#include \"efunction.h\"\n");

	/* write the layer declarations */
	xprintf(f, "\n/******************** LAYERS ********************/\n");
	k = 8;
	for(i=0; i<tech->layercount; i++) k = maxi(k, strlen(us_teclayer_iname[i]));
	k++;
	xprintf(f, "\n#define MAXLAYERS");
	for(j=8; j<k; j++) xprintf(f, " ");
	xprintf(f, "%d\n", tech->layercount);
	for(i=0; i<tech->layercount; i++)
	{
		xprintf(f, "#define L%s", us_teclayer_iname[i]);
		for(j=strlen(us_teclayer_iname[i]); j<k; j++) xprintf(f, " ");
		xprintf(f, "%d\t\t\t\t/* %s */\n", i, us_teclayer_names[i]);
	}
	xprintf(f, "\n");

	/* write the layer descriptions */
	for(i=0; i<tech->layercount; i++)
	{
		xprintf(f, "static GRAPHICS %s_%s_lay = {", us_tecedmakesymbol(tech->techname),
			us_teclayer_iname[i]);
		switch (tech->layers[i]->bits)
		{
			case LAYERT1: xprintf(f, "LAYERT1, ");   break;
			case LAYERT2: xprintf(f, "LAYERT2, ");   break;
			case LAYERT3: xprintf(f, "LAYERT3, ");   break;
			case LAYERT4: xprintf(f, "LAYERT4, ");   break;
			case LAYERT5: xprintf(f, "LAYERT5, ");   break;
			case LAYERO:  xprintf(f, "LAYERO, ");    break;
		}
		if (ecolorname(tech->layers[i]->col, &colorname, &colorsymbol) != 0) colorsymbol = "unknown";
		xprintf(f, "%s, {", colorsymbol);
		for(j=0; j<MAXSTYLES; j++)
		{
			if ((tech->layers[i]->style[j]&NATURE) == SOLIDC) xprintf(f, "SOLIDC"); else
			{
				xprintf(f, "PATTERNED");
				if ((tech->layers[i]->style[j]&OUTLINEPAT) != 0) xprintf(f, "|OUTLINEPAT");
			}
			if (j == MAXSTYLES-1) xprintf(f, "}");
			xprintf(f, ",");
		}
		xprintf(f, "\n");
		for(j=0; j<8; j++) if (tech->layers[i]->raster[j] != 0) break;
		if (j >= 8)
			xprintf(f, "\t{0,0,0,0,0,0,0,0}, NOVARIABLE, 0};\n"); else
		{
			for(j=0; j<8; j++)
			{
				xprintf(f, "\t");
				if (j == 0) xprintf(f, "{");
				xprintf(f, "0x%04x", tech->layers[i]->raster[j]&0xFFFF);
				if (j == 7) xprintf(f, "}");
				xprintf(f, ",");
				if (j > 0 && j < 7) xprintf(f, " ");
				xprintf(f, "  /* ");
				for(k=0; k<16; k++)
					if ((tech->layers[i]->raster[j] & (1 << (15-k))) != 0)
						xprintf(f, "X"); else xprintf(f, " ");
				xprintf(f, " */\n");
			}
			xprintf(f, "\tNOVARIABLE, 0};\n");
		}
	}

	/* write the aggregation of all layers */
	sym = us_tecedmakesymbol(tech->techname);
	l = strlen(sym) + 34;
	xprintf(f, "\nGRAPHICS *%s_layers[MAXLAYERS+1] = {", sym);
	for(i=0; i<tech->layercount; i++)
	{
		sym = us_tecedmakesymbol(tech->techname);
		if (l + strlen(sym) + strlen(us_teclayer_iname[i]) + 8 > 80)
		{
			xprintf(f, "\n\t");
			l = 4;
		}
		xprintf(f, "&%s_%s_lay, ", sym, us_teclayer_iname[i]);
		l += strlen(sym) + strlen(us_teclayer_iname[i]) + 8;
	}
	xprintf(f, "NOGRAPHICS};\n");

	/* write the layer names */
	sym = us_tecedmakesymbol(tech->techname);
	l = strlen(sym) + 40;
	xprintf(f, "static char *%s_layer_names[MAXLAYERS] = {", sym);
	for(i=0; i<tech->layercount; i++)
	{
		if (i != 0) { xprintf(f, ", "); l += 2; }
		if (us_teclayer_names[i] == 0) sym = ""; else sym = us_teclayer_names[i];
		if (l + strlen(sym) + 2 > 80)
		{
			xprintf(f, "\n\t");
			l = 4;
		}
		xprintf(f, "\"%s\"", sym);
		l += strlen(sym) + 2;
	}
	xprintf(f, "};\n");

	/* write the CIF layer names */
	if ((us_tecflags&HASCIF) != 0)
	{
		xprintf(f, "static char *%s_cif_layers[MAXLAYERS] = {", us_tecedmakesymbol(tech->techname));
		for(i=0; i<tech->layercount; i++)
		{
			if (i != 0) xprintf(f, ", ");
			if (us_teccif_layers[i] == 0) xprintf(f, "\"\""); else
				xprintf(f, "\"%s\"", us_teccif_layers[i]);
		}
		xprintf(f, "};\n");
	}

	/* write the DXF layer numbers */
	if ((us_tecflags&HASDXF) != 0)
	{
		xprintf(f, "static char *%s_dxf_layers[MAXLAYERS] = {", us_tecedmakesymbol(tech->techname));
		for(i=0; i<tech->layercount; i++)
		{
			if (i != 0) xprintf(f, ", ");
			xprintf(f, "\"%s\"", us_tecdxf_layers[i]);
		}
		xprintf(f, "};\n");
	}

	/* write the Calma GDS-II layer number */
	if ((us_tecflags&HASGDS) != 0)
	{
		xprintf(f, "static INTBIG %s_gds_layers[MAXLAYERS] = {", us_tecedmakesymbol(tech->techname));
		for(i=0; i<tech->layercount; i++)
		{
			if (i != 0) xprintf(f, ", ");
			xprintf(f, "%d", us_tecgds_layers[i]);
		}
		xprintf(f, "};\n");
	}

	/* write the layer functions */
	sym = us_tecedmakesymbol(tech->techname);
	l = strlen(sym) + 43;
	xprintf(f, "static INTBIG %s_layer_function[MAXLAYERS] = {", sym);
	for(i=0; i<tech->layercount; i++)
	{
		if (i != 0) { xprintf(f, ", "); l += 2; }
		(void)initinfstr();
		(void)addstringtoinfstr(us_teclayer_functions[us_teclayer_function[i]&LFTYPE].constant);
		for(j=0; us_teclayer_functions[j].name != 0; j++)
		{
			if (us_teclayer_functions[j].value <= LFTYPE) continue;
			if ((us_teclayer_function[i]&us_teclayer_functions[j].value) != 0)
			{
				(void)addtoinfstr('|');
				(void)addstringtoinfstr(us_teclayer_functions[j].constant);
			}
		}
		if (tech->layers[i]->bits == LAYERT1) (void)addstringtoinfstr("|LFTRANS1"); else
		if (tech->layers[i]->bits == LAYERT2) (void)addstringtoinfstr("|LFTRANS2"); else
		if (tech->layers[i]->bits == LAYERT3) (void)addstringtoinfstr("|LFTRANS3"); else
		if (tech->layers[i]->bits == LAYERT4) (void)addstringtoinfstr("|LFTRANS4"); else
		if (tech->layers[i]->bits == LAYERT5) (void)addstringtoinfstr("|LFTRANS5");
		sym = returninfstr();
		if (l + strlen(sym) > 80)
		{
			xprintf(f, "\n\t");
			l = 4;
		}
		xprintf(f, "%s", sym);
		l += strlen(sym);
	}
	xprintf(f, "};\n");

	/* write the layer letters */
	xprintf(f, "static char *%s_layer_letters[MAXLAYERS] = {", us_tecedmakesymbol(tech->techname));
	for(i=0; i<tech->layercount; i++)
	{
		if (i != 0) xprintf(f, ", ");
		if (us_teclayer_letters[i] == 0) xprintf(f, "\"\""); else
			xprintf(f, "\"%s\"", us_teclayer_letters[i]);
	}
	xprintf(f, "};\n");

	/* write the SPICE information */
	if ((us_tecflags&HASSPIRES) != 0)
	{
		xprintf(f, "static float %s_sim_spice_resistance[MAXLAYERS] = {",
			us_tecedmakesymbol(tech->techname));
		for(i=0; i<tech->layercount; i++)
		{
			if (i != 0) xprintf(f, ", ");
			xprintf(f, "%g", us_tecspice_res[i]);
		}
		xprintf(f, "};\n");
	}
	if ((us_tecflags&HASSPICAP) != 0)
	{
		xprintf(f, "static float %s_sim_spice_capacitance[MAXLAYERS] = {",
			us_tecedmakesymbol(tech->techname));
		for(i=0; i<tech->layercount; i++)
		{
			if (i != 0) xprintf(f, ", ");
			xprintf(f, "%g", us_tecspice_cap[i]);
		}
		xprintf(f, "};\n");
	}
	if ((us_tecflags&HASSPIECAP) != 0)
	{
		xprintf(f, "static float %s_sim_spice_edge_cap[MAXLAYERS] = {",
			us_tecedmakesymbol(tech->techname));
		for(i=0; i<tech->layercount; i++)
		{
			if (i != 0) xprintf(f, ", ");
			xprintf(f, "%g", us_tecspice_ecap[i]);
		}
		xprintf(f, "};\n");
	}

	/* write the color map */
	if ((us_tecflags&HASCOLORMAP) != 0)
	{
		/* determine the five overlapping layers */
		l1 = l2 = l3 = l4 = l5 = 0;
		for(i=0; i<tech->layercount; i++)
		{
			if (tech->layers[i]->bits == LAYERT1 && l1 == 0)
				l1 = us_teclayer_names[i]; else
			if (tech->layers[i]->bits == LAYERT2 && l2 == 0)
				l2 = us_teclayer_names[i]; else
			if (tech->layers[i]->bits == LAYERT3 && l3 == 0)
				l3 = us_teclayer_names[i]; else
			if (tech->layers[i]->bits == LAYERT4 && l4 == 0)
				l4 = us_teclayer_names[i]; else
			if (tech->layers[i]->bits == LAYERT5 && l5 == 0)
				l5 = us_teclayer_names[i];
		}
		if (l1 == 0) l1 = "layer 1";
		if (l2 == 0) l2 = "layer 2";
		if (l3 == 0) l3 = "layer 3";
		if (l4 == 0) l4 = "layer 4";
		if (l5 == 0) l5 = "layer 5";
		xprintf(f, "\nstatic TECH_COLORMAP %s_colmap[32] =\n{\n", us_tecedmakesymbol(tech->techname));
		for(i=0; i<32; i++)
		{
			xprintf(f, "\t{%3d,%3d,%3d}, /* %2d: ", us_teccolmap[i].red,
				us_teccolmap[i].green, us_teccolmap[i].blue, i);
			if ((i&1) != 0) xprintf(f, "%s", l1); else
				for(j=0; j<strlen(l1); j++) xprintf(f, " ");
			xprintf(f, "+");
			if ((i&2) != 0) xprintf(f, "%s", l2); else
				for(j=0; j<strlen(l2); j++) xprintf(f, " ");
			xprintf(f, "+");
			if ((i&4) != 0) xprintf(f, "%s", l3); else
				for(j=0; j<strlen(l3); j++) xprintf(f, " ");
			xprintf(f, "+");
			if ((i&8) != 0) xprintf(f, "%s", l4); else
				for(j=0; j<strlen(l4); j++) xprintf(f, " ");
			xprintf(f, "+");
			if ((i&16) != 0) xprintf(f, "%s", l5); else
				for(j=0; j<strlen(l5); j++) xprintf(f, " ");
			xprintf(f, " */\n");
		}
		xprintf(f, "};\n");
	}

	/* write design rules */
	if ((us_tecflags&(HASUNCONDRC|HASCONDRC)) != 0)
	{
		xprintf(f, "\n/******************** DESIGN RULES ********************/\n");
		if ((us_tecflags&HASUNCONDRC) != 0)
		{
			xprintf(f, "\nstatic INTBIG %s_unconnectedtable[] = {\n",
				us_tecedmakesymbol(tech->techname));
			us_teceditdumpdrctab(f, us_tecdrc_min_unconnected_dist, tech);
		}
		if ((us_tecflags&HASCONDRC) != 0)
		{
			xprintf(f, "\nstatic INTBIG %s_connectedtable[] = {\n",
				us_tecedmakesymbol(tech->techname));
			us_teceditdumpdrctab(f, us_tecdrc_min_connected_dist, tech);
		}
	}
}

void us_teceditdumpdrctab(FILE *f, INTBIG *distances, TECHNOLOGY *tech)
{
	REGISTER INTSML i, j;
	REGISTER INTBIG amt, mod;
	char shortname[7];

	for(i=0; i<6; i++)
	{
		xprintf(f, "/*            ");
		for(j=0; j<tech->layercount; j++)
		{
			if (strlen(us_teclayer_iname[j]) <= i) xprintf(f, " "); else
				xprintf(f, "%c", us_teclayer_iname[j][i]);
			xprintf(f, "  ");
		}
		xprintf(f, " */\n");
	}
	for(j=0; j<tech->layercount; j++)
	{
		(void)strncpy(shortname, us_teclayer_iname[j], 6);
		shortname[6] = 0;
		xprintf(f, "/* %-6s */ ", shortname);
		for(i=0; i<j; i++) xprintf(f, "   ");
		for(i=j; i<tech->layercount; i++)
		{
			amt = *distances++;
			if (amt < 0) xprintf(f, "XX"); else
			{
				mod = amt % WHOLE;
				if (mod == 0) xprintf(f, "K%d", amt/WHOLE); else
				if (mod == WHOLE/2) xprintf(f, "H%d", amt/WHOLE); else
				if (mod == WHOLE/4) xprintf(f, "Q%d", amt/WHOLE); else
				if (mod == WHOLE/4*3) xprintf(f, "T%d", amt/WHOLE); else
					xprintf(f, "%d", amt);
			}
			if (j != tech->layercount-1 || i != tech->layercount-1)
				xprintf(f, ",");
		}
		xprintf(f, "\n");
	}
	xprintf(f, "};\n");
}

/*
 * routine to dump the arc information in technology "tech" to the stream in
 * "f".
 */
void us_teceditdumparcs(FILE *f, TECHNOLOGY *tech)
{
	REGISTER INTSML i, j, k;

	/* print the header */
	xprintf(f, "\n/******************** ARCS ********************/\n");

	/* compute the width of the widest arc name */
	k = 12;
	for(i=0; i<tech->arcprotocount; i++)
		k = maxi(k, strlen(tech->arcprotos[i]->arcname));
	k++;

	/* write the number of arcs */
	xprintf(f, "\n#define ARCPROTOCOUNT");
	for(j=12; j<k; j++) xprintf(f, " ");
	xprintf(f, "%d\n", tech->arcprotocount);

	/* write defines for each arc */
	for(i=0; i<tech->arcprotocount; i++)
	{
		xprintf(f, "#define A%s", us_tecedmakeupper(tech->arcprotos[i]->arcname));
		for(j=strlen(tech->arcprotos[i]->arcname); j<k; j++)
			xprintf(f, " ");
		xprintf(f, "%d\t\t\t\t/* %s */\n", i, tech->arcprotos[i]->arcname);
	}

	/* now write the arcs */
	for(i=0; i<tech->arcprotocount; i++)
	{
		xprintf(f, "\nstatic TECH_ARCLAY %s_al_%d[] = {",
			us_tecedmakesymbol(tech->techname), i);
		for(k=0; k<tech->arcprotos[i]->laycount; k++)
		{
			if (k != 0) xprintf(f, ", ");
			xprintf(f, "{");
			xprintf(f, "L%s,", us_teclayer_iname[tech->arcprotos[i]->list[k].lay]);
			if (tech->arcprotos[i]->list[k].off == 0) xprintf(f, "0,"); else
				xprintf(f, "%s,", us_tecedmakefract(tech->arcprotos[i]->list[k].off));
			if (tech->arcprotos[i]->list[k].style == FILLED) xprintf(f, "FILLED}"); else
				xprintf(f, "CLOSED}");
		}
		xprintf(f, "};\n");
		xprintf(f, "static TECH_ARCS %s_a_%d = {\n", us_tecedmakesymbol(tech->techname), i);
		xprintf(f, "\t\"%s\", ", tech->arcprotos[i]->arcname);
		xprintf(f, "%s, ", us_tecedmakefract(tech->arcprotos[i]->arcwidth));
		xprintf(f, "A%s,\n", us_tecedmakeupper(tech->arcprotos[i]->arcname));
		xprintf(f, "\t%d, %s_al_%d,\n", tech->arcprotos[i]->laycount,
			us_tecedmakesymbol(tech->techname), i);
		for(j=0; us_tecarc_functions[j].name != 0; j++)
			if (us_tecarc_functions[j].value ==
				((tech->arcprotos[i]->initialbits&AFUNCTION)>>AFUNCTIONSH))
		{
			xprintf(f, "\t(%s<<AFUNCTIONSH)", us_tecarc_functions[j].constant);
			break;
		}
		if (us_tecarc_functions[j].name == 0)                    xprintf(f, "\t(APUNKNOWN<<AFUNCTIONSH)");
		if ((tech->arcprotos[i]->initialbits&WANTFIXANG) != 0)   xprintf(f, "|WANTFIXANG");
		if ((tech->arcprotos[i]->initialbits&CANWIPE) != 0)      xprintf(f, "|CANWIPE");
		if ((tech->arcprotos[i]->initialbits&WANTNOEXTEND) != 0) xprintf(f, "|WANTNOEXTEND");
		xprintf(f, "|(%d<<AANGLEINCSH)", (tech->arcprotos[i]->initialbits&AANGLEINC)>>AANGLEINCSH);
		xprintf(f, "};\n");
	}

	/* print the summary */
	xprintf(f, "\nTECH_ARCS *%s_arcprotos[ARCPROTOCOUNT+1] = {\n\t",
		us_tecedmakesymbol(tech->techname));
	for(i=0; i<tech->arcprotocount; i++)
		xprintf(f, "&%s_a_%d, ", us_tecedmakesymbol(tech->techname), i);
	xprintf(f, "((TECH_ARCS *)-1)};\n");

	/* print the variable with the width offsets */
	if ((us_tecflags&HASARCWID) != 0)
	{
		xprintf(f, "\nstatic INTBIG %s_arc_widoff[ARCPROTOCOUNT] = {",
			us_tecedmakesymbol(tech->techname));
		for(i=0; i<tech->arcprotocount; i++)
		{
			if (i != 0) xprintf(f, ", ");
			if (us_tecarc_widoff[i] == 0) xprintf(f, "0"); else
				xprintf(f, "%s", us_tecedmakefract(us_tecarc_widoff[i]));
		}
		xprintf(f, "};\n");
	}
}

/*
 * routine to dump the node information in technology "tech" to the stream in
 * "f".
 */
void us_teceditdumpnodes(FILE *f, TECHNOLOGY *tech)
{
	REGISTER RULE *r;
	REGISTER INTSML i, j, k, l, tot;
	char *ab, *sym;
	REGISTER PCON *pc;
	REGISTER TECH_POLYGON *plist;
	REGISTER TECH_SERPENT *slist;
	REGISTER TECH_NODES *nlist;

	/* make abbreviations for each node */
	for(i=0; i<tech->nodeprotocount; i++)
	{
		(void)allocstring(&ab, makeabbrev(tech->nodeprotos[i]->nodename, 0), el_tempcluster);
		tech->nodeprotos[i]->creation = (NODEPROTO *)ab;

		/* loop until the name is unique */
		for(;;)
		{
			/* see if a previously assigned abbreviation is the same */
			for(j=0; j<i; j++)
				if (namesame(ab, (char *)tech->nodeprotos[j]->creation) == 0) break;
			if (j == i) break;

			/* name conflicts: change it */
			l = strlen(ab);
			if (ab[l-1] >= '0' && ab[l-1] <= '8') ab[l-1]++; else
			{
				(void)initinfstr();
				(void)addstringtoinfstr(ab);
				(void)addtoinfstr('0');
				(void)reallocstring(&ab, returninfstr(), el_tempcluster);
				tech->nodeprotos[i]->creation = (NODEPROTO *)ab;
			}
		}
	}

	/* write the port lists */
	xprintf(f, "\n/******************** PORT CONNECTIONS ********************/\n\n");
	i = 1;
	for(pc = us_tecedfirstpcon; pc != NOPCON; pc = pc->nextpcon)
	{
		pc->index = i++;
		xprintf(f, "static INTBIG %s_pc_%d[] = {-1, ",
			us_tecedmakesymbol(tech->techname), pc->index);
		for(j=0; j<pc->total; j++)
		{
			k = pc->connects[j+1];
			xprintf(f, "A%s, ", us_tecedmakeupper(tech->arcprotos[k]->arcname));
		}
		xprintf(f, "ALLGEN, -1};\n");
	}

	xprintf(f, "\n/******************** RECTANGLE DESCRIPTIONS ********************/");
	xprintf(f, "\n\n");

	/* print box information */
	i = 1;
	for(r = us_tecedfirstrule; r != NORULE; r = r->nextrule)
	{
		if (r->used == 0) continue;
		r->index = i++;
		xprintf(f, "static INTBIG %s_box%d[%d] = {",
			us_tecedmakesymbol(tech->techname), r->index, r->count);
		for(j=0; j<r->count; j += 2)
		{
			if (j != 0) xprintf(f, ", ");
			xprintf(f, "%s", us_tecededgelabel(r->value[j], r->value[j+1],
				(INTSML)((j%4 == 0 ? 0 : 1))));
		}
		if (r->istext != 0)
			xprintf(f, ", \"%s\"", (char *)r->value[r->count]);
		xprintf(f, "};\n");
	}

	xprintf(f, "\n/******************** NODES ********************/\n");

	/* compute widest node name */
	k = 13;
	for(i=0; i<tech->nodeprotocount; i++)
		k = maxi(k, strlen((char *)tech->nodeprotos[i]->creation));
	k++;

	/* write the total define */
	xprintf(f, "\n#define NODEPROTOCOUNT");
	for(j=13; j<k; j++) xprintf(f, " ");
	xprintf(f, "%d\n", tech->nodeprotocount);

	/* write the other defines */
	for(i=0; i<tech->nodeprotocount; i++)
	{
		ab = (char *)tech->nodeprotos[i]->creation;
		xprintf(f, "#define N%s", us_tecedmakeupper(ab));
		for(j=strlen(ab); j<k; j++) xprintf(f, " ");
		xprintf(f, "%d\t\t\t\t/* %s */\n", i+1, tech->nodeprotos[i]->nodename);
	}

	/* print node information */
	for(i=0; i<tech->nodeprotocount; i++)
	{
		/* header comment */
		nlist = tech->nodeprotos[i];
		ab = (char *)nlist->creation;
		xprintf(f, "\n/* %s */\n", nlist->nodename);

		/* print ports */
		xprintf(f, "static TECH_PORTS %s_%s_p[] = {\n", us_tecedmakesymbol(tech->techname), ab);
		for(j=0; j<nlist->portcount; j++)
		{
			if (j != 0) xprintf(f, ",\n");

			/* the name of the connection structure */
			for(pc = us_tecedfirstpcon; pc != NOPCON; pc = pc->nextpcon)
				if (pc->connects == nlist->portlist[j].portarcs) break;
			if (pc != NOPCON)
				xprintf(f, "\t{%s_pc_%d, ", us_tecedmakesymbol(tech->techname), pc->index);

			/* the port name */
			xprintf(f, "\"%s\", NOPORTPROTO, ", nlist->portlist[j].protoname);

			/* the port userbits */
			xprintf(f, "(%d<<PORTARANGESH)",
				(nlist->portlist[j].initialbits&PORTARANGE)>>PORTARANGESH);
			if ((nlist->portlist[j].initialbits&PORTANGLE) != 0)
				xprintf(f, "|(%d<<PORTANGLESH)",
					(nlist->portlist[j].initialbits&PORTANGLE)>>PORTANGLESH);
			if ((nlist->portlist[j].initialbits&PORTNET) != 0)
				xprintf(f, "|(%d<<PORTNETSH)", (nlist->portlist[j].initialbits&PORTNET)>>PORTNETSH);
			xprintf(f, ",\n");

			/* the port area */
			xprintf(f, "\t\t%s, %s, %s, %s}",
				us_tecededgelabel(nlist->portlist[j].lowxmul, nlist->portlist[j].lowxsum, 0),
				us_tecededgelabel(nlist->portlist[j].lowymul, nlist->portlist[j].lowysum, 1),
				us_tecededgelabel(nlist->portlist[j].highxmul, nlist->portlist[j].highxsum, 0),
				us_tecededgelabel(nlist->portlist[j].highymul, nlist->portlist[j].highysum, 1));
		}
		xprintf(f, "};\n");

		/* print layers */
		for(k=0; k<2; k++)
		{
			if (nlist->special == SERPTRANS)
			{
				if (k == 0)
				{
					xprintf(f, "static TECH_SERPENT %s_%s_l[] = {\n",
						us_tecedmakesymbol(tech->techname), ab);
					tot = nlist->layercount;
				} else
				{
					xprintf(f, "static TECH_SERPENT %s_%sE_l[] = {\n",
						us_tecedmakesymbol(tech->techname), ab);
					tot = nlist->layercount + 2;
				}
			} else
			{
				if (k != 0) continue;
				xprintf(f, "static TECH_POLYGON %s_%s_l[] = {\n",
					us_tecedmakesymbol(tech->techname), ab);
				tot = nlist->layercount;
			}
			for(j=0; j<tot; j++)
			{
				if (j != 0) xprintf(f, ",\n");
				xprintf(f, "\t");
				if (nlist->special == SERPTRANS)
				{
					xprintf(f, "{");
					if (k == 0) plist = &nlist->gra[j].basics;
						else plist = &nlist->ele[j].basics;
				} else plist = &nlist->layerlist[j];
				xprintf(f, "{L%s,", us_teclayer_iname[plist->layernum]);
				xprintf(f, " %d,", plist->portnum);
				xprintf(f, " %d,", plist->count);
				switch (plist->style)
				{
					case FILLEDRECT:  xprintf(f, " FILLEDRECT,");  break;
					case CLOSEDRECT:  xprintf(f, " CLOSEDRECT,");  break;
					case CROSSED:     xprintf(f, " CROSSED,");     break;
					case FILLED:      xprintf(f, " FILLED,");      break;
					case CLOSED:      xprintf(f, " CLOSED,");      break;
					case OPENED:      xprintf(f, " OPENED,");      break;
					case OPENEDT1:    xprintf(f, " OPENEDT1,");    break;
					case OPENEDT2:    xprintf(f, " OPENEDT2,");    break;
					case OPENEDT3:    xprintf(f, " OPENEDT3,");    break;
					case VECTORS:     xprintf(f, " VECTORS,");     break;
					case CIRCLE:      xprintf(f, " CIRCLE,");      break;
					case DISC:        xprintf(f, " DISC,");        break;
					case CIRCLEARC:   xprintf(f, " CIRCLEARC,");   break;
					case TEXTCENT:    xprintf(f, " TEXTCENT,");    break;
					case TEXTTOP:     xprintf(f, " TEXTTOP,");     break;
					case TEXTBOT:     xprintf(f, " TEXTBOT,");     break;
					case TEXTLEFT:    xprintf(f, " TEXTLEFT,");    break;
					case TEXTRIGHT:   xprintf(f, " TEXTRIGHT,");   break;
					case TEXTTOPLEFT: xprintf(f, " TEXTTOPLEFT,"); break;
					case TEXTBOTLEFT: xprintf(f, " TEXTBOTLEFT,"); break;
					case TEXTTOPRIGHT:xprintf(f, " TEXTTOPRIGHT,");break;
					case TEXTBOTRIGHT:xprintf(f, " TEXTBOTRIGHT,");break;
					case TEXTBOX:     xprintf(f, " TEXTBOX,");     break;
					default:          xprintf(f, " ????,");        break;
				}
				switch (plist->representation)
				{
					case BOX:    xprintf(f, " BOX,");     break;
					case MINBOX: xprintf(f, " MINBOX,");  break;
					case POINTS: xprintf(f, " POINTS,");  break;
					default:     xprintf(f, " ????,");    break;
				}
				for(r = us_tecedfirstrule; r != NORULE; r = r->nextrule)
					if (r->value == plist->points) break;
				if (r != NORULE)
					xprintf(f, " %s_box%d", us_tecedmakesymbol(tech->techname), r->index); else
						xprintf(f, " %s_box??", us_tecedmakesymbol(tech->techname));
				xprintf(f, "}");
				if (nlist->special == SERPTRANS)
				{
					if (k == 0) slist = &nlist->gra[j]; else
						slist = &nlist->ele[j];
					xprintf(f, ", %s", us_tecedmakefract(slist->lwidth));
					xprintf(f, ", %s", us_tecedmakefract(slist->rwidth));
					xprintf(f, ", %s}", us_tecedmakefract(slist->extend));
				}
			}
			xprintf(f, "};\n");
		}

		/* print the node information */
		xprintf(f, "static TECH_NODES %s_%s = {\n", us_tecedmakesymbol(tech->techname), ab);
		xprintf(f, "\t\"%s\", N%s, NONODEPROTO,\n", nlist->nodename, us_tecedmakeupper(ab));
		xprintf(f, "\t%s,", us_tecedmakefract(nlist->xsize));
		xprintf(f, " %s,\n", us_tecedmakefract(nlist->ysize));
		xprintf(f, "\t%d, %s_%s_p,\n", nlist->portcount, us_tecedmakesymbol(tech->techname), ab);
		if (nlist->special == SERPTRANS)
			xprintf(f, "\t%d, (TECH_POLYGON *)0,\n", nlist->layercount); else
				xprintf(f, "\t%d, %s_%s_l,\n", nlist->layercount,
					us_tecedmakesymbol(tech->techname), ab);
		for(j=0; us_tecnode_functions[j].name != 0; j++)
			if (us_tecnode_functions[j].value ==
				((nlist->initialbits&NFUNCTION)>>NFUNCTIONSH)) break;
		if (us_tecnode_functions[j].name == 0) j = 0;
		xprintf(f, "\t(%s<<NFUNCTIONSH)", us_tecnode_functions[j].constant);
		if ((nlist->initialbits&WIPEON1OR2) != 0) xprintf(f, "|WIPEON1OR2");
		if ((nlist->initialbits&HOLDSTRACE) != 0) xprintf(f, "|HOLDSTRACE");
		if ((nlist->initialbits&NSQUARE) != 0)    xprintf(f, "|NSQUARE");
		if ((nlist->initialbits&ARCSWIPE) != 0)   xprintf(f, "|ARCSWIPE");
		if ((nlist->initialbits&ARCSHRINK) != 0)  xprintf(f, "|ARCSHRINK");
		if ((nlist->initialbits&NODESHRINK) != 0) xprintf(f, "|NODESHRINK");
		if ((nlist->initialbits&LOCKEDPRIM) != 0) xprintf(f, "|LOCKEDPRIM");
		xprintf(f, ",\n");
		switch (nlist->special)
		{
			case 0:
				xprintf(f, "\t0,0,0,0,0,0,0");
				break;
			case SERPTRANS:
				xprintf(f, "\tSERPTRANS,%s,", us_tecedmakefract(nlist->f1));
				xprintf(f, "%s,", us_tecedmakefract(nlist->f2));
				xprintf(f, "%s,0,", us_tecedmakefract(nlist->f3));
				xprintf(f, "%s_%s_l,", us_tecedmakesymbol(tech->techname), ab);
				xprintf(f, "%s_%sE_l", us_tecedmakesymbol(tech->techname), ab);
				break;
			case MULTICUT:
				xprintf(f, "\tMULTICUT,%s,", us_tecedmakefract(nlist->f1));
				xprintf(f, "%s,", us_tecedmakefract(nlist->f2));
				xprintf(f, "%s,", us_tecedmakefract(nlist->f3));
				xprintf(f, "%s,0,0", us_tecedmakefract(nlist->f4));
				break;
			case POLYGONAL:
				xprintf(f, "\tPOLYGONAL,0,0,0,0,0,0");
				break;
		}
		xprintf(f, "};\n");
	}

	/* print summary of nodes */
	xprintf(f, "\nTECH_NODES *%s_nodeprotos[NODEPROTOCOUNT+1] = {\n\t",
		us_tecedmakesymbol(tech->techname));
	l = 4;
	for(i=0; i<tech->nodeprotocount; i++)
	{
		sym = us_tecedmakesymbol(tech->techname);
		if (l + strlen(sym) + strlen((char *)tech->nodeprotos[i]->creation) + 4 > 80)
		{
			xprintf(f, "\n\t");
			l = 4;
		}
		xprintf(f, "&%s_%s, ", sym, (char *)tech->nodeprotos[i]->creation);
		l += strlen(sym) + strlen((char *)tech->nodeprotos[i]->creation) + 4;
	}
	xprintf(f, "((TECH_NODES *)-1)};\n");

	/* print highlight offset information */
	xprintf(f, "\nstatic INTBIG %s_node_widoff[NODEPROTOCOUNT*4] = {\n\t",
		us_tecedmakesymbol(tech->techname));
	l = 4;
	for(i=0; i<tech->nodeprotocount; i++)
	{
		if (i != 0) { xprintf(f, ", "); l += 2; }
		(void)initinfstr();
		if (us_tecnode_widoff[i*4] == 0) (void)addtoinfstr('0'); else
			(void)addstringtoinfstr(us_tecedmakefract(us_tecnode_widoff[i*4]));
		(void)addtoinfstr(',');
		if (us_tecnode_widoff[i*4+1] == 0) (void)addtoinfstr('0'); else
			(void)addstringtoinfstr(us_tecedmakefract(us_tecnode_widoff[i*4+1]));
		(void)addtoinfstr(',');
		if (us_tecnode_widoff[i*4+2] == 0) (void)addtoinfstr('0'); else
			(void)addstringtoinfstr(us_tecedmakefract(us_tecnode_widoff[i*4+2]));
		(void)addtoinfstr(',');
		if (us_tecnode_widoff[i*4+3] == 0) (void)addtoinfstr('0'); else
			(void)addstringtoinfstr(us_tecedmakefract(us_tecnode_widoff[i*4+3]));
		sym = returninfstr();
		l += strlen(sym);
		if (l > 80)
		{
			xprintf(f, "\n\t");
			l = 4;
		}
		xprintf(f, "%s", sym);
	}
	xprintf(f, "};\n");

	/* print grab point informaton if it exists */
	if ((us_tecflags&HASGRAB) != 0 && us_tecnode_grabcount > 0)
	{
		xprintf(f, "\nstatic INTBIG %s_centergrab[] = {\n", us_tecedmakesymbol(tech->techname));
		for(i=0; i<us_tecnode_grabcount; i += 3)
		{
			ab = (char *)tech->nodeprotos[us_tecnode_grab[i]-1]->creation;
			xprintf(f, "\tN%s, %d, %d", us_tecedmakeupper(ab), us_tecnode_grab[i+1],
				us_tecnode_grab[i+2]);
			if (i != us_tecnode_grabcount-3) xprintf(f, ",\n");
		}
		xprintf(f, "\n};\n");
	}

	/* clean up */
	for(i=0; i<tech->nodeprotocount; i++)
	{
		efree((char *)tech->nodeprotos[i]->creation);
		tech->nodeprotos[i]->creation = NONODEPROTO;
	}
}

/*
 * routine to dump the variable information in technology "tech" to the stream in
 * "f".
 */
void us_teceditdumpvars(FILE *f, TECHNOLOGY *tech)
{
	REGISTER INTSML i, j, k;
	REGISTER char *pt;
	REGISTER VARIABLE *var;

	xprintf(f, "\n/******************** VARIABLE AGGREGATION ********************/\n");

	/* write any miscellaneous string array variables */
	for(i=0; us_knownvars[i].varname != 0; i++)
	{
		var = getval((INTBIG)tech, VTECHNOLOGY, -1, us_knownvars[i].varname);
		if (var == NOVARIABLE) continue;
		if ((var->type&(VTYPE|VISARRAY)) == (VSTRING|VISARRAY))
		{
			xprintf(f, "\nchar *%s_%s[] = {\n", us_tecedmakesymbol(tech->techname),
				us_knownvars[i].varname);
			j = getlength(var);
			for(k=0; k<j; k++)
			{
				for(pt = ((char **)var->addr)[k]; *pt != 0; pt++)
				{
					if (*pt == '"') xprintf(f, "^");
					xprintf(f, "%c", *pt);
				}
			}
			xprintf(f, "};\n");
		}
	}

	xprintf(f, "\nTECH_VARIABLES %s_variables[] =\n{\n", us_tecedmakesymbol(tech->techname));

	xprintf(f, "\t{\"TECH_layer_names\", (char *)%s_layer_names, 0.0,\n",
		us_tecedmakesymbol(tech->techname));
	xprintf(f, "\t\tVSTRING|VDONTSAVE|VISARRAY|(MAXLAYERS<<VLENGTHSH)},\n");

	xprintf(f, "\t{\"TECH_layer_function\", (char *)%s_layer_function, 0.0,\n",
		us_tecedmakesymbol(tech->techname));
	xprintf(f, "\t\tVINTEGER|VDONTSAVE|VISARRAY|(MAXLAYERS<<VLENGTHSH)},\n");

	xprintf(f, "\t{\"TECH_node_width_offset\", (char *)%s_node_widoff, 0.0,\n",
		us_tecedmakesymbol(tech->techname));
	xprintf(f, "\t\tVFRACT|VDONTSAVE|VISARRAY|((NODEPROTOCOUNT*4)<<VLENGTHSH)},\n");
	if ((us_tecflags&HASGRAB) != 0 && us_tecnode_grabcount > 0)
		xprintf(f, "\t{\"prototype_center\", (char *)%s_centergrab, 0.0, %d},\n",
			us_tecedmakesymbol(tech->techname), us_tecnode_grabcount/3);

	if ((us_tecflags&HASARCWID) != 0)
	{
		xprintf(f, "\t{\"TECH_arc_width_offset\", (char *)%s_arc_widoff, 0.0,\n",
			us_tecedmakesymbol(tech->techname));
		xprintf(f, "\t\tVFRACT|VDONTSAVE|VISARRAY|(ARCPROTOCOUNT<<VLENGTHSH)},\n");
	}

	xprintf(f, "\t{\"USER_layer_letters\", (char *)%s_layer_letters, 0.0,\n",
		us_tecedmakesymbol(tech->techname));
	xprintf(f, "\t\tVSTRING|VDONTSAVE|VISARRAY|(MAXLAYERS<<VLENGTHSH)},\n");

	if ((us_tecflags&HASCOLORMAP) != 0)
	{
		xprintf(f, "\t{\"USER_color_map\", (char *)%s_colmap, 0.0,\n",
			us_tecedmakesymbol(tech->techname));
		xprintf(f, "\t\tVCHAR|VDONTSAVE|VISARRAY|((sizeof %s_colmap)<<VLENGTHSH)},\n",
			us_tecedmakesymbol(tech->techname));
	}

	if ((us_tecflags&HASCIF) != 0)
	{
		xprintf(f, "\t{\"IO_cif_layer_names\", (char *)%s_cif_layers, 0.0,\n",
			us_tecedmakesymbol(tech->techname));
		xprintf(f, "\t\tVSTRING|VDONTSAVE|VISARRAY|(MAXLAYERS<<VLENGTHSH)},\n");
	}

	if ((us_tecflags&HASDXF) != 0)
	{
		xprintf(f, "\t{\"IO_dxf_layer_names\", (char *)%s_dxf_layers, 0.0,\n",
			us_tecedmakesymbol(tech->techname));
		xprintf(f, "\t\tVSTRING|VDONTSAVE|VISARRAY|(MAXLAYERS<<VLENGTHSH)},\n");
	}

	if ((us_tecflags&HASGDS) != 0)
	{
		xprintf(f, "\t{\"IO_gds_layer_numbers\", (char *)%s_gds_layers, 0.0,\n",
			us_tecedmakesymbol(tech->techname));
		xprintf(f, "\t\tVINTEGER|VDONTSAVE|VISARRAY|(MAXLAYERS<<VLENGTHSH)},\n");
	}

	if ((us_tecflags&HASUNCONDRC) != 0)
	{
		xprintf(f, "\t{\"DRC_min_unconnected_distances\", (char *)%s_unconnectedtable, 0.0,\n",
			us_tecedmakesymbol(tech->techname));
		xprintf(f, "\t\tVFRACT|VDONTSAVE|VISARRAY|\n");
		xprintf(f, "\t\t   (((sizeof %s_unconnectedtable)/SIZEOFINTBIG)<<VLENGTHSH)},\n",
			us_tecedmakesymbol(tech->techname));
	}

	if ((us_tecflags&HASCONDRC) != 0)
	{
		xprintf(f, "\t{\"DRC_min_connected_distances\", (char *)%s_connectedtable, 0.0,\n",
			us_tecedmakesymbol(tech->techname));
		xprintf(f, "\t\tVFRACT|VDONTSAVE|VISARRAY|\n");
		xprintf(f, "\t\t\t(((sizeof %s_connectedtable)/SIZEOFINTBIG)<<VLENGTHSH)},\n",
			us_tecedmakesymbol(tech->techname));
	}
	if ((us_tecflags&HASSPIRES) != 0)
	{
		xprintf(f, "\t{\"SIM_spice_resistance\", (char *)%s_sim_spice_resistance, 0.0,\n",
			us_tecedmakesymbol(tech->techname));
		xprintf(f, "\t\tVFLOAT|VDONTSAVE|VISARRAY|(MAXLAYERS<<VLENGTHSH)},\n");
	}
	if ((us_tecflags&HASSPICAP) != 0)
	{
		xprintf(f, "\t{\"SIM_spice_capacitance\", (char *)%s_sim_spice_capacitance, 0.0,\n",
			us_tecedmakesymbol(tech->techname));
		xprintf(f, "\t\tVFLOAT|VDONTSAVE|VISARRAY|(MAXLAYERS<<VLENGTHSH)},\n");
	}
	if ((us_tecflags&HASSPIECAP) != 0)
	{
		xprintf(f, "\t{\"SIM_spice_edge_capacitance\", (char *)%s_sim_spice_edge_cap, 0.0,\n",
			us_tecedmakesymbol(tech->techname));
		xprintf(f, "\t\tVFLOAT|VDONTSAVE|VISARRAY|(MAXLAYERS<<VLENGTHSH)},\n");
	}

	/* throw in pointers to any miscellaneous variables */
	for(i=0; us_knownvars[i].varname != 0; i++)
	{
		var = getval((INTBIG)tech, VTECHNOLOGY, -1, us_knownvars[i].varname);
		if (var == NOVARIABLE) continue;
		xprintf(f, "\t{\"%s\", ", us_knownvars[i].varname);
		switch (var->type&(VTYPE|VISARRAY))
		{
			case VINTEGER:
				xprintf(f, "(char *)%d, 0.0, VINTEGER|VDONTSAVE", var->addr);
				break;
			case VFLOAT:
				xprintf(f, "(char *)0, %g, VFLOAT|VDONTSAVE", castfloat(var->addr));
				break;
			case VSTRING:
				xprintf(f, "\"%s\", 0.0, VSTRING|VDONTSAVE", (char *)var->addr);
				break;
			case VSTRING|VISARRAY:
				xprintf(f, "(char *)%s_%s, 0.0,\n\t\tVSTRING|VDONTSAVE|VISARRAY|(MAXLAYERS<<VLENGTHSH)",
					us_tecedmakesymbol(tech->techname), us_knownvars[i].varname);
				break;
		}
		xprintf(f, "},\n");
	}

	xprintf(f, "\t{NULL, NULL, 0.0, 0}\n};\n");
}

/*
 * routine to convert the multiplication and addition factors in "mul" and
 * "add" into proper constant names.  The "axis" is zero for X and 1 for Y
 */
char *us_tecededgelabel(INTBIG mul, INTBIG add, INTSML axis)
{
	char line[20];
	REGISTER INTBIG amt;

	(void)initinfstr();

	/* handle constant distance from center (handles halves up to 5.5) */
	if (mul == 0 && (add%H0) == 0 && abs(add) < K6)
	{
		(void)addstringtoinfstr("CENTER");
		if (add == 0) return(returninfstr());
		if (axis == 0)
		{
			if (add < 0) (void)addtoinfstr('L'); else (void)addtoinfstr('R');
		} else
		{
			if (add < 0) (void)addtoinfstr('D'); else (void)addtoinfstr('U');
		}
		amt = abs(add);
		switch (amt%WHOLE)
		{
			case 0:  (void)sprintf(line, "%d",  amt/WHOLE);   break;
			case H0: (void)sprintf(line, "%dH", amt/WHOLE);   break;
		}
		(void)addstringtoinfstr(line);
		return(returninfstr());
	}

	/* handle constant distance from edge (handles quarters up to 10, halves to 20) */
	if ((mul == H0 || mul == -H0) &&
		(((add%Q0) == 0 && abs(add) < K10) || ((add%H0) == 0 && abs(add) < K20)))
	{
		if (axis == 0)
		{
			if (mul < 0) (void)addstringtoinfstr("LEFT"); else
				(void)addstringtoinfstr("RIGHT");
		} else
		{
			if (mul < 0) (void)addstringtoinfstr("BOT"); else
				(void)addstringtoinfstr("TOP");
		}
		if (add == 0) (void)addstringtoinfstr("EDGE"); else
		{
			amt = abs(add);
			switch (amt%WHOLE)
			{
				case 0:  (void)sprintf(line, "IN%d",  amt/WHOLE);   break;
				case Q0: (void)sprintf(line, "IN%dQ", amt/WHOLE);   break;
				case H0: (void)sprintf(line, "IN%dH", amt/WHOLE);   break;
				case T0: (void)sprintf(line, "IN%dT", amt/WHOLE);   break;
			}
			(void)addstringtoinfstr(line);
		}
		return(returninfstr());
	}

	/* generate two-value description */
	(void)addstringtoinfstr(us_tecedmakefract(mul));
	(void)addtoinfstr(',');
	(void)addstringtoinfstr(us_tecedmakefract(add));
	return(returninfstr());
}

/*
 * routine to convert the fractional value "amt" to a technology constant.
 * The presumption is that quarter values exist from K0 to K10, that
 * half values exist up to K20, that whole values exist up to K30, and
 * that other values are not necessarily defined in "tech.h".
 */
char *us_tecedmakefract(INTBIG amt)
{
	static char line[20];
	REGISTER INTBIG whole;
	REGISTER char *pt;

	pt = line;
	if (amt < 0)
	{
		*pt++ = '-';
		amt = -amt;
	}
	whole = amt/WHOLE;
	switch (amt%WHOLE)
	{
		case 0:
			if (whole <= 30) (void)sprintf(pt, "K%d", whole); else
				(void)sprintf(pt, "%d", amt);
			break;
		case Q0:
			if (whole <= 10) (void)sprintf(pt, "Q%d", whole); else
				(void)sprintf(pt, "%d", amt);
			break;
		case H0:
			if (whole <= 20) (void)sprintf(pt, "H%d", whole); else
				(void)sprintf(pt, "%d", amt);
			break;
		case T0:
			if (whole <= 10) (void)sprintf(pt, "T%d", whole); else
				(void)sprintf(pt, "%d", amt);
			break;
		default:
			(void)sprintf(pt, "%d", amt);
			break;
	}
	return(line);
}

/*
 * routine to convert all characters in string "str" to upper case and to
 * change any nonalphanumeric characters to a "_"
 */
char *us_tecedmakeupper(char *str)
{
	REGISTER INTSML ch;

	(void)initinfstr();
	while (*str != 0)
	{
		ch = *str++;
		if (islower(ch)) ch = toupper(ch);
		if (!isalnum(ch)) ch = '_';
		(void)addtoinfstr((char)ch);
	}
	return(returninfstr());
}

/*
 * routine to change any nonalphanumeric characters in string "str" to a "_"
 */
char *us_tecedmakesymbol(char *str)
{
	REGISTER INTSML ch;

	(void)initinfstr();
	while (*str != 0)
	{
		ch = *str++;
		if (!isalnum(ch)) ch = '_';
		(void)addtoinfstr((char)ch);
	}
	return(returninfstr());
}
