/*
 * Electric(tm) VLSI Design System
 *
 * File: usredtecp.c
 * User interface technology editor: conversion from library to technology
 * Written by: Steven M. Rubin, Electric Editor Incorporated
 *
 * Copyright (c) 1998 Electric Editor Incorporated.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Electric Editor Incorporated
 * 23470 Sunset Drive, Suite 108
 * Los Gatos, California 95033
 * support@electriceditor.com
 */

#include "global.h"
#include "egraphics.h"
#include "efunction.h"
#include "tech.h"
#include "tecgen.h"
#include "tecart.h"
#include "usr.h"
#include "usredtec.h"
#include <ctype.h>

extern LIST us_tecarc_functions[];
extern LIST us_tecnode_functions[];

/* prototypes for local routines */
void us_tecedcheck(TECHNOLOGY*);
INTSML us_tecedmakefactors(LIBRARY**, INTSML, TECHNOLOGY*);
INTSML us_tecedmakelayers(LIBRARY**, INTSML, TECHNOLOGY*);
INTSML us_tecedmakearcs(LIBRARY**, INTSML, TECHNOLOGY*);
INTSML us_tecedmakenodes(LIBRARY**, INTSML, TECHNOLOGY*);
INTSML us_tecedfindport(TECH_NODES*, EXAMPLE*, INTBIG, INTBIG, INTBIG, INTBIG, INTBIG);
INTSML us_tecedassociateexamples(EXAMPLE*, NODEPROTO*);
INTSML us_tecedmakeprim(EXAMPLE*, NODEPROTO*, TECHNOLOGY*);
INTSML us_tecedforcearrays(INTBIG**, INTBIG**, INTBIG**, INTBIG**, INTBIG**, INTBIG**, INTBIG**, INTBIG**, INTBIG**, INTBIG**, INTBIG**, INTBIG**, INTBIG**, INTSML*, INTSML);
INTBIG *us_tecedstretchpoints(INTBIG*, INTBIG*, INTSML, INTBIG*, SAMPLE*, NODEPROTO*, EXAMPLE*);
INTSML us_tecedmulticut(SAMPLE*, EXAMPLE*, NODEPROTO*);
PCON *us_tecedaddportlist(INTSML, INTBIG*);

/*
 * the routine invoked for the "technology edit library-to-tech" command.  Dumps
 * C code if "dumpc" is nonzero
 */
void us_tecfromlibinit(LIBRARY *lib, INTSML dumpc)
{
	REGISTER FILE *f;
	REGISTER TECHNOLOGY *tech;
	REGISTER CLUSTER *clus;
	REGISTER VARIABLE *var, *ovar;
	REGISTER char **varnames;
	char *truename;
	static TECH_VARIABLES us_tecvariables[2] = {{NULL, NULL, 0.0, 0},
                                               {NULL, NULL, 0.0, 0}};
	extern AIDENTRY *net_aid;
	INTBIG oldlam;
	LIBRARY **dependentlibs;
	REGISTER INTSML dependentlibcount, i, j;

	/* make sure network tool is on */
	if ((net_aid->aidstate&AIDON) == 0)
	{
		ttyputerr("Network tool must be running...turning it on");
		aidturnon(net_aid, 0);
		ttyputerr("...now reissue the technology editing command");
		return;
	}

	/* create the technology */
	(void)initinfstr();
	(void)addstringtoinfstr("tech:");
	(void)addstringtoinfstr(lib->libname);
	clus = alloccluster(returninfstr());
	if (clus == NOCLUSTER) return;
	tech = alloctechnology(clus);
	if (tech == NOTECHNOLOGY) return;

	/* loop until the name is valid */
	if (allocstring(&tech->techname, lib->libname, clus) != 0) return;
	for(;;)
	{
		if (gettechnology(tech->techname) == NOTECHNOLOGY) break;
		(void)initinfstr();
		(void)addstringtoinfstr(tech->techname);
		(void)addtoinfstr('X');
		(void)reallocstring(&tech->techname, returninfstr(), clus);
	}
	if (namesame(tech->techname, lib->libname) != 0)
		ttyputmsg("Warning: already a technology called %s.  Naming this %s",
			lib->libname, tech->techname);

	/* set technology description */
	if (allocstring(&tech->techdescript, tech->techname, clus) != 0) return;

	/* get list of dependent libraries */
	dependentlibcount = us_teceditgetdependents(lib, &dependentlibs);

	/* initialize the state of this technology */
	us_tecflags = 0;
	if (us_tecedmakefactors(dependentlibs, dependentlibcount, tech) != 0) return;

	/* build layer structures */
	if (us_tecedmakelayers(dependentlibs, dependentlibcount, tech) != 0) return;

	/* build arc structures */
	if (us_tecedmakearcs(dependentlibs, dependentlibcount, tech) != 0) return;

	/* build node structures */
	if (us_tecedmakenodes(dependentlibs, dependentlibcount, tech) != 0) return;

	/* copy any miscellaneous variables (should use dependent libraries facility) */
	var = getval((INTBIG)lib, VLIBRARY, VSTRING|VISARRAY, "EDTEC_variable_list");
	if (var != NOVARIABLE)
	{
		j = getlength(var);
		varnames = (char **)var->addr;
		for(i=0; i<j; i++)
		{
			ovar = getval((INTBIG)lib, VLIBRARY, -1, varnames[i]);
			if (ovar == NOVARIABLE) continue;
			(void)setval((INTBIG)tech, VTECHNOLOGY, varnames[i], ovar->addr, ovar->type);
		}
	}

	/* check technology for consistency */
	us_tecedcheck(tech);

	if (dumpc != 0)
	{
		/* print the technology */
		f = xcreate("TECHNOLOGY.c", FILETYPETEXT, "Technology code file", &truename);
		if (f == NULL)
		{
			if (truename != 0) ttyputerr("Cannot write %s", truename);
			return;
		}
		ttyputmsgf("Writing: TECHNOLOGY.c");

		/* write the layers, arcs, and nodes */
		us_teceditdumplayers(f, tech);
		us_teceditdumparcs(f, tech);
		us_teceditdumpnodes(f, tech);
		us_teceditdumpvars(f, tech);

		/* clean up */
		xclose(f);
	}

	/* finish initializing the technology */
	if ((us_tecflags&HASGRAB) == 0) us_tecvariables[0].name = 0; else
	{
		us_tecvariables[0].name = "prototype_center";
		us_tecvariables[0].value = (char *)us_tecnode_grab;
		us_tecvariables[0].type = us_tecnode_grabcount/3;
	}
	tech->variables = us_tecvariables;
	if (tech_doinitprocess(tech) != 0) return;
	if (tech_doaddportsandvars(tech) != 0) return;

	/* install the technology fully */
	(void)addtechnology(tech);

	/* let the user interface process it */
	us_figuretechopaque(tech);

	/* switch to this technology */
	ttyputmsg("Technology %s built.  Switching to it.", tech->techname);
	us_setnodeproto(NONODEPROTO);
	us_setarcproto(NOARCPROTO, 1);
	oldlam = el_curtech->deflambda;
	us_getcolormap(tech, COLORSEXISTING, 1);
	(void)setvalkey((INTBIG)us_aid, VAID, us_current_technology, (INTBIG)tech,
		VTECHNOLOGY|VDONTSAVE);
	us_adjustlambda(oldlam, el_curtech->deflambda);

	/* fix up the menu entries */
	us_setmenunodearcs();
	if ((us_state&NONPERSISTENTCURNODE) == 0) us_setnodeproto(tech->firstnodeproto);
	us_setarcproto(tech->firstarcproto, 1);
}

void us_tecedcheck(TECHNOLOGY *tech)
{
	REGISTER INTSML i, j, k, l;
	REGISTER TECH_POLYGON *plist;
	REGISTER TECH_NODES *nlist;

	/* make sure there is a pure-layer node for every nonpseudo layer */
	for(i=0; i<tech->layercount; i++)
	{
		if ((us_teclayer_function[i]&LFPSEUDO) != 0) continue;
		for(j=0; j<tech->nodeprotocount; j++)
		{
			nlist = tech->nodeprotos[j];
			if (((nlist->initialbits&NFUNCTION)>>NFUNCTIONSH) != NPNODE) continue;
			plist = &nlist->layerlist[0];
			if (plist->layernum == i) break;
		}
		if (j < tech->nodeprotocount) continue;
		ttyputmsg("Warning: Layer %s has no associated pure-layer node",
			us_teclayer_names[i]);
	}

	/* make sure there is a pin for every arc and that it uses pseudo-layers */
	for(i=0; i<tech->arcprotocount; i++)
	{
		for(j=0; j<tech->nodeprotocount; j++)
		{
			nlist = tech->nodeprotos[j];
			if (((nlist->initialbits&NFUNCTION)>>NFUNCTIONSH) != NPPIN) continue;
			for(k=0; k<nlist->portcount; k++)
			{
				for(l=1; nlist->portlist[k].portarcs[l] >= 0; l++)
					if (nlist->portlist[k].portarcs[l] == i) break;
				if (nlist->portlist[k].portarcs[l] >= 0) break;
			}
			if (k < nlist->portcount) break;
		}
		if (j < tech->nodeprotocount)
		{
			/* pin found: make sure it uses pseudo-layers */
			nlist = tech->nodeprotos[j];
			for(k=0; k<nlist->layercount; k++)
			{
				plist = &nlist->layerlist[k];
				if ((us_teclayer_function[plist->layernum]&LFPSEUDO) == 0) break;
			}
			if (k < nlist->layercount)
				ttyputmsg("Warning: Pin %s is not composed of pseudo-layers",
					tech->nodeprotos[j]->nodename);
			continue;
		}
		ttyputmsg("Warning: Arc %s has no associated pin node", tech->arcprotos[i]->arcname);
	}
}

/*
 * routine to scan the "dependentlibcount" libraries in "dependentlibs",
 * and get global factors for technology "tech".  Returns nonzero on error.
 */
INTSML us_tecedmakefactors(LIBRARY **dependentlibs, INTSML dependentlibcount, TECHNOLOGY *tech)
{
	REGISTER NODEPROTO *np;
	REGISTER NODEINST *ni;
	REGISTER VARIABLE *var;
	REGISTER INTSML opt;
	REGISTER char *str;
	REGISTER INTSML i;

	np = NONODEPROTO;
	for(i=dependentlibcount-1; i>=0; i--)
	{
		for(np = dependentlibs[i]->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
			if (namesame(np->cell->cellname, "factors") == 0) break;
		if (np != NONODEPROTO) break;
	}
	if (np == NONODEPROTO)
	{
		tech->deflambda = 2000;
		return(0);
	}

	for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		opt = us_tecedgetoption(ni);
		var = getvalkey((INTBIG)ni, VNODEINST, VSTRING, art_messagekey);
		if (var == NOVARIABLE) continue;
		str = (char *)var->addr;
		switch (opt)
		{
			case TECHLAMBDA:	/* lambda */
				tech->deflambda = myatoi(&str[8]);
				break;
			case TECHDESCRIPT:	/* description */
				(void)reallocstring(&tech->techdescript, &str[13], tech->cluster);
				break;
			default:
				us_tecedpointout(ni, np);
				ttyputerr("Unknown object in miscellaneous-information facet");
				return(1);
		}
	}
	return(0);
}

/*
 * routine to scan the "dependentlibcount" libraries in "dependentlibs",
 * and build the layer structures for it in technology "tech".  Returns nonzero on error.
 */
INTSML us_tecedmakelayers(LIBRARY **dependentlibs, INTSML dependentlibcount, TECHNOLOGY *tech)
{
	REGISTER NODEPROTO *np;
	NODEPROTO **sequence;
	REGISTER INTSML i, j, l, total;
	REGISTER INTBIG *drcptr, drcsize;
	INTSML connected, layer1, layer2;
	INTBIG amt;
	REGISTER char *str, *ab;
	REGISTER VARIABLE *var;

	/* first find the number of layers */
	tech->layercount = us_teceditfindsequence(dependentlibs, dependentlibcount, "layer-",
		"EDTEC_layersequence", &sequence);
	if (tech->layercount <= 0)
	{
		ttyputerr("No layers found");
		if ((us_aid->aidstate&NODETAILS) == 0)
			ttyputerr("Create them with the 'edit-layer' option");
		return(1);
	}

	/* allocate the arrays for the layers */
	us_teclayer_iname = (char **)emalloc((tech->layercount * (sizeof (char *))), us_aid->cluster);
	if (us_teclayer_iname == 0) return(1);
	tech->layers = (GRAPHICS **)emalloc(((tech->layercount+1) * (sizeof (GRAPHICS *))), tech->cluster);
	if (tech->layers == 0) return(1);
	tech->layers[tech->layercount] = NOGRAPHICS;
	us_teclayer_names = (char **)emalloc((tech->layercount * (sizeof (char *))), us_aid->cluster);
	if (us_teclayer_names == 0) return(1);
	us_teccif_layers = (char **)emalloc((tech->layercount * (sizeof (char *))), us_aid->cluster);
	if (us_teccif_layers == 0) return(1);
	us_tecdxf_layers = (char **)emalloc((tech->layercount * SIZEOFINTBIG), us_aid->cluster);
	if (us_tecdxf_layers == 0) return(1);
	us_teclayer_function = emalloc((tech->layercount * SIZEOFINTBIG), us_aid->cluster);
	if (us_teclayer_function == 0) return(1);
	us_tecgds_layers = emalloc((tech->layercount * SIZEOFINTBIG), us_aid->cluster);
	if (us_tecgds_layers == 0) return(1);
	us_teclayer_letters = (char **)emalloc((tech->layercount * (sizeof (char *))), us_aid->cluster);
	if (us_teclayer_letters == 0) return(1);
	drcsize = tech->layercount*tech->layercount/2 + (tech->layercount+1)/2;
	us_tecdrc_min_unconnected_dist = emalloc((drcsize * SIZEOFINTBIG), us_aid->cluster);
	if (us_tecdrc_min_unconnected_dist == 0) return(1);
	us_tecdrc_min_connected_dist = emalloc((drcsize * SIZEOFINTBIG), us_aid->cluster);
	if (us_tecdrc_min_connected_dist == 0) return(1);
	us_tecspice_res = (float *)emalloc((tech->layercount * (sizeof (float))), us_aid->cluster);
	if (us_tecspice_res == 0) return(1);
	us_tecspice_cap = (float *)emalloc((tech->layercount * (sizeof (float))), us_aid->cluster);
	if (us_tecspice_cap == 0) return(1);
	us_tecspice_ecap = (float *)emalloc((tech->layercount * (sizeof (float))), us_aid->cluster);
	if (us_tecspice_ecap == 0) return(1);

	/* initialize the layer arrays */
	for(i=0; i<tech->layercount; i++)
	{
		us_teclayer_names[i] = us_teccif_layers[i] = us_teclayer_letters[i] = 0;
		us_tecdxf_layers[i] = 0;
		us_tecgds_layers[i] = -1;
		us_tecspice_res[i] = 0.0;
		us_tecspice_cap[i] = 0.0;
		us_tecspice_ecap[i] = 0.0;
	}
	for(i=0; i<drcsize; i++)
		us_tecdrc_min_unconnected_dist[i] = us_tecdrc_min_connected_dist[i] = -WHOLE;

	/* now scan each layer and fill in the data */
	for(total=0; total<tech->layercount; total++)
	{
		/* set the layer name */
		np = sequence[total];
		(void)allocstring(&us_teclayer_names[total], &np->cell->cellname[6], us_aid->cluster);

		if (us_teceditgetlayerinfo(np, &tech->layers[total], &us_teccif_layers[total],
			&us_teclayer_function[total], &us_teclayer_letters[total], &us_tecdxf_layers[total],
				&us_tecgds_layers[total], &us_tecspice_res[total], &us_tecspice_cap[total],
					&us_tecspice_ecap[total]) != 0) return(1);
		if (us_teccif_layers[total] != 0 && namesame(us_teccif_layers[total], "xx") != 0)
			us_tecflags |= HASCIF;
		if (us_tecdxf_layers[total] != 0) us_tecflags |= HASDXF;
		if (us_tecgds_layers[total] != -1) us_tecflags |= HASGDS;
		if (us_tecspice_res[total] != 0.0) us_tecflags |= HASSPIRES;
		if (us_tecspice_cap[total] != 0.0) us_tecflags |= HASSPICAP;
		if (us_tecspice_ecap[total] != 0.0) us_tecflags |= HASSPIECAP;
		tech->layers[total]->firstvar = NOVARIABLE;
		tech->layers[total]->numvar = 0;
	}

	for(i=0; i<total; i++)
	{
		(void)allocstring(&us_teclayer_iname[i], makeabbrev(us_teclayer_names[i], 1),
			us_aid->cluster);

		/* loop until the name is unique */
		for(;;)
		{
			/* see if a previously assigned abbreviation is the same */
			for(j=0; j<i; j++)
				if (namesame(us_teclayer_iname[i], us_teclayer_iname[j]) == 0)
					break;
			if (j >= i) break;

			/* name conflicts: change it */
			l = strlen(ab = us_teclayer_iname[i]);
			if (ab[l-1] >= '0' && ab[l-1] <= '8') ab[l-1]++; else
			{
				(void)initinfstr();
				(void)addstringtoinfstr(ab);
				(void)addtoinfstr('0');
				(void)reallocstring(&us_teclayer_iname[i], returninfstr(), us_aid->cluster);
			}
		}
	}

	/* get the color map */
	var = NOVARIABLE;
	for(i=dependentlibcount-1; i>=0; i--)
	{
		var = getval((INTBIG)dependentlibs[i], VLIBRARY, VINTEGER|VISARRAY, "EDTEC_colormap");
		if (var != NOVARIABLE) break;
	}
	if (var != NOVARIABLE)
	{
		us_tecflags |= HASCOLORMAP;
		drcptr = (INTBIG *)var->addr;
		for(i=0; i<32; i++)
		{
			us_teccolmap[i].red = drcptr[(i<<3)*3];
			us_teccolmap[i].green = drcptr[(i<<3)*3+1];
			us_teccolmap[i].blue = drcptr[(i<<3)*3+2];
		}
	}

	/* get the design rules */
	var = NOVARIABLE;
	for(i=dependentlibcount-1; i>=0; i--)
	{
		var = getval((INTBIG)dependentlibs[i], VLIBRARY, VSTRING|VISARRAY, "EDTEC_DRC");
		if (var != NOVARIABLE) break;
	}
	if (var != NOVARIABLE)
	{
		l = getlength(var);
		for(i=0; i<l; i++)
		{
			/* parse the DRC rule */
			str = ((char **)var->addr)[i];
			while (*str == ' ') str++;
			if (*str == 0) continue;
			if (us_tecedgetdrc(str, &connected, &amt, &layer1, &layer2, tech->layercount,
				us_teclayer_names) != 0)
			{
				ttyputmsg("DRC line %d is: %s", i+1, str);
				continue;
			}

			/* set the layer spacing */
			if (layer1 > layer2) { j = layer1;  layer1 = layer2;  layer2 = j; }
			j = (layer1+1) * (layer1/2) + (layer1&1) * ((layer1+1)/2);
			j = layer2 + tech->layercount * layer1 - j;
			if (connected == 0)
			{
				us_tecdrc_min_unconnected_dist[j] = amt;
				us_tecflags |= HASUNCONDRC;
			} else
			{
				us_tecdrc_min_connected_dist[j] = amt;
				us_tecflags |= HASCONDRC;
			}
		}
	}

	/* store this information on the technology object */
	(void)setval((INTBIG)tech, VTECHNOLOGY, "TECH_layer_names", (INTBIG)us_teclayer_names,
		VSTRING|VDONTSAVE|VISARRAY|(tech->layercount<<VLENGTHSH));
	(void)setval((INTBIG)tech, VTECHNOLOGY, "TECH_layer_function", (INTBIG)us_teclayer_function,
		VINTEGER|VDONTSAVE|VISARRAY|(tech->layercount<<VLENGTHSH));
	(void)setvalkey((INTBIG)tech, VTECHNOLOGY, us_layer_letters, (INTBIG)us_teclayer_letters,
		VSTRING|VDONTSAVE|VISARRAY|(tech->layercount<<VLENGTHSH));
	if ((us_tecflags&HASCOLORMAP) != 0)
		(void)setval((INTBIG)tech, VTECHNOLOGY, "USER_color_map", (INTBIG)us_teccolmap,
			VCHAR|VDONTSAVE|VISARRAY|((sizeof us_teccolmap)<<VLENGTHSH));
	if ((us_tecflags&HASCIF) != 0)
		(void)setval((INTBIG)tech, VTECHNOLOGY, "IO_cif_layer_names", (INTBIG)us_teccif_layers,
			VSTRING|VDONTSAVE|VISARRAY|(tech->layercount<<VLENGTHSH));
	if ((us_tecflags&HASDXF) != 0)
		(void)setval((INTBIG)tech, VTECHNOLOGY, "IO_dxf_layer_names", (INTBIG)us_tecdxf_layers,
			VSTRING|VDONTSAVE|VISARRAY|(tech->layercount<<VLENGTHSH));
	if ((us_tecflags&HASGDS) != 0)
		(void)setval((INTBIG)tech, VTECHNOLOGY, "IO_gds_layer_numbers", (INTBIG)us_tecgds_layers,
			VINTEGER|VDONTSAVE|VISARRAY|(tech->layercount<<VLENGTHSH));
	if ((us_tecflags&HASUNCONDRC) != 0)
		(void)setval((INTBIG)tech, VTECHNOLOGY, "DRC_min_unconnected_distances",
			(INTBIG)us_tecdrc_min_unconnected_dist, VFRACT|VDONTSAVE|VISARRAY|(drcsize<<VLENGTHSH));
	if ((us_tecflags&HASCONDRC) != 0)
		(void)setval((INTBIG)tech, VTECHNOLOGY, "DRC_min_connected_distances",
			(INTBIG)us_tecdrc_min_connected_dist, VFRACT|VDONTSAVE|VISARRAY|(drcsize<<VLENGTHSH));
	if ((us_tecflags&HASSPIRES) != 0)
		(void)setval((INTBIG)tech, VTECHNOLOGY, "SIM_spice_resistance", (INTBIG)us_tecspice_res,
			VFLOAT|VDONTSAVE|VISARRAY|(tech->layercount<<VLENGTHSH));
	if ((us_tecflags&HASSPICAP) != 0)
		(void)setval((INTBIG)tech, VTECHNOLOGY, "SIM_spice_capacitance", (INTBIG)us_tecspice_cap,
			VFLOAT|VDONTSAVE|VISARRAY|(tech->layercount<<VLENGTHSH));
	if ((us_tecflags&HASSPIECAP) != 0)
		(void)setval((INTBIG)tech, VTECHNOLOGY, "SIM_spice_edge_capacitance", (INTBIG)us_tecspice_ecap,
			VFLOAT|VDONTSAVE|VISARRAY|(tech->layercount<<VLENGTHSH));

	efree((char *)sequence);
	return(0);
}

/*
 * routine to parse DRC line "str" and fill the factors "connected" (set nonzero
 * if rule is for connected layers), "amt" (rule distance), "layer1" and "layer2"
 * (the layers).  Presumes that there are "maxlayers" layer names in the
 * array "layernames".  Returns nonzero on error.
 */
INTSML us_tecedgetdrc(char *str, INTSML *connected, INTBIG *amt, INTSML *layer1,
	INTSML *layer2, INTSML maxlayers, char **layernames)
{
	REGISTER char *pt;
	REGISTER INTSML save;

	*connected = 0;
	if (namesamen(str, "C", 1) == 0)
	{
		str++;
		*connected = 1;
	}

	/* get the distance */
	pt = str;
	while (*pt != 0 && *pt != ' ' && *pt != '\t') pt++;
	while (*pt == ' ' || *pt == '\t') pt++;
	if (*pt == 0)
	{
		ttyputerr("Cannot find layer names on DRC line");
		return(1);
	}
	*amt = atofr(str);

	/* get the first layer */
	str = pt;
	while (*pt != 0 && *pt != ' ' && *pt != '\t') pt++;
	if (*pt == 0)
	{
		ttyputerr("Cannot find layer name on DRC line");
		return(1);
	}
	save = *pt;
	*pt = 0;
	for(*layer1 = 0; *layer1 < maxlayers; (*layer1)++)
		if (namesame(str, layernames[*layer1]) == 0) break;
	*pt++ = save;
	if (*layer1 >= maxlayers)
	{
		ttyputerr("First DRC layer name unknown");
		return(1);
	}
	while (*pt == ' ' || *pt == '\t') pt++;

	/* get the second layer */
	for(*layer2 = 0; *layer2 < maxlayers; (*layer2)++)
		if (namesame(pt, layernames[*layer2]) == 0) break;
	if (*layer2 >= maxlayers)
	{
		ttyputerr("Second DRC layer name unknown");
		return(1);
	}
	return(0);
}

/*
 * routine to scan the "dependentlibcount" libraries in "dependentlibs",
 * and build the arc structures for it in technology "tech".  Returns nonzero on error.
 */
INTSML us_tecedmakearcs(LIBRARY **dependentlibs, INTSML dependentlibcount,
	TECHNOLOGY *tech)
{
	REGISTER NODEPROTO *np;
	NODEPROTO **sequence;
	REGISTER INTSML arcindex, count, j, k, layerindex, typ;
	REGISTER INTBIG maxwid, hwid, wid;
	REGISTER char *str;
	REGISTER NODEINST *ni;
	REGISTER EXAMPLE *nelist;
	REGISTER SAMPLE *ns;
	REGISTER VARIABLE *var;

	/* count the number of arcs in the technology */
	tech->arcprotocount = us_teceditfindsequence(dependentlibs, dependentlibcount, "arc-",
		"EDTEC_arcsequence", &sequence);
	if (tech->arcprotocount <= 0)
	{
		ttyputerr("No arcs found");
		if ((us_aid->aidstate&NODETAILS) == 0)
			ttyputerr("Create them with the 'edit-arc' option");
		return(1);
	}

	/* allocate the arcs */
	tech->arcprotos = (TECH_ARCS **)emalloc(((tech->arcprotocount+1) * (sizeof (TECH_ARCS *))),
		tech->cluster);
	if (tech->arcprotos == 0) return(1);
	tech->arcprotos[tech->arcprotocount] = ((TECH_ARCS *)-1);
	us_tecarc_widoff = emalloc((tech->arcprotocount * SIZEOFINTBIG), us_aid->cluster);
	if (us_tecarc_widoff == 0) return(1);

	/* create the arc structures */
	for(arcindex=0; arcindex<tech->arcprotocount; arcindex++)
	{
		/* build a list of examples found in this arc */
		np = sequence[arcindex];
		nelist = us_tecedgetexamples(np, 0);
		if (nelist == NOEXAMPLE) return(1);
		if (nelist->nextexample != NOEXAMPLE)
		{
			us_tecedpointout(NONODEINST, np);
			ttyputerr("Can only be one example of %s but more were found", describenodeproto(np));
			return(1);
		}

		/* get width and polygon count information */
		count = 0;
		maxwid = hwid = -1;
		for(ns = nelist->firstsample; ns != NOSAMPLE; ns = ns->nextsample)
		{
			wid = mini(ns->node->highx - ns->node->lowx, ns->node->highy - ns->node->lowy);
			if (wid > maxwid) maxwid = wid;
			if (ns->layer == NONODEPROTO) hwid = wid; else count++;
		}

		/* error if there is no highlight box */
		if (hwid < 0)
		{
			us_tecedpointout(NONODEINST, np);
			ttyputerr("No highlight layer found in %s", describenodeproto(np));
			if ((us_aid->aidstate&NODETAILS) == 0)
				ttyputmsg("Use 'place-layer' option to create HIGHLIGHT");
			return(1);
		}

		/* create and fill the basic structure entries for this arc */
		tech->arcprotos[arcindex] = (TECH_ARCS *)emalloc(sizeof (TECH_ARCS), tech->cluster);
		if (tech->arcprotos[arcindex] == 0) return(1);
		(void)allocstring(&tech->arcprotos[arcindex]->arcname, &np->cell->cellname[4],
			tech->cluster);
		tech->arcprotos[arcindex]->arcwidth = maxwid * WHOLE / el_curtech->deflambda;
		tech->arcprotos[arcindex]->arcindex = arcindex;
		tech->arcprotos[arcindex]->laycount = count;
		us_tecarc_widoff[arcindex] = (maxwid - hwid) * WHOLE / el_curtech->deflambda;
		if (us_tecarc_widoff[arcindex] != 0) us_tecflags |= HASARCWID;

		/* look for descriptive nodes in the facet */
		tech->arcprotos[arcindex]->initialbits = 0;
		for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
		{
			typ = us_tecedgetoption(ni);
			var = getvalkey((INTBIG)ni, VNODEINST, VSTRING, art_messagekey);
			if (var == NOVARIABLE) continue;

			/* the "Function:" node */
			if (typ == ARCFUNCTION)
			{
				str = &((char *)var->addr)[10];
				for(j=0; us_tecarc_functions[j].name != 0; j++)
					if (namesame(str, us_tecarc_functions[j].name) == 0)
				{
					tech->arcprotos[arcindex]->initialbits |=
						(us_tecarc_functions[j].value << AFUNCTIONSH);
					break;
				}
			}

			/* the "Fixed-angle:" node */
			if (typ == ARCFIXANG)
			{
				str = (char *)var->addr;
				if (str[9] == ':') str = &str[11]; else str = &str[13];
				if (namesame(str, "yes") == 0)
					tech->arcprotos[arcindex]->initialbits |= WANTFIXANG;
			}

			/* the "Wipes pins:" node */
			if (typ == ARCWIPESPINS)
			{
				str = &((char *)var->addr)[12];
				if (namesame(str, "yes") == 0)
					tech->arcprotos[arcindex]->initialbits |= CANWIPE;
			}

			/* the "Extend arcs:" node */
			if (typ == ARCNOEXTEND)
			{
				str = &((char *)var->addr)[13];
				if (namesame(str, "no") == 0)
					tech->arcprotos[arcindex]->initialbits |= WANTNOEXTEND;
			}

			/* the "Angle increment:" node */
			if (typ == ARCINC)
			{
				j = myatoi(&((char *)var->addr)[17]) % 360;
				if (j < 0) j += 360;
				tech->arcprotos[arcindex]->initialbits &= ~AANGLEINC;
				tech->arcprotos[arcindex]->initialbits |= (j << AANGLEINCSH);
			}
		}

		/* allocate the individual arc layer structures */
		tech->arcprotos[arcindex]->list = (TECH_ARCLAY *)emalloc((count * (sizeof (TECH_ARCLAY))),
			tech->cluster);
		if (tech->arcprotos[arcindex]->list == 0) return(1);

		/* fill the individual arc layer structures */
		layerindex = 0;
		for(k=0; k<2; k++)
			for(ns = nelist->firstsample; ns != NOSAMPLE; ns = ns->nextsample)
		{
			if (ns->layer == NONODEPROTO) continue;

			/* get the layer index */
			for(j=0; j<tech->layercount; j++)
				if (namesame(&ns->layer->cell->cellname[6], us_teclayer_names[j]) == 0) break;
			if (j >= tech->layercount)
			{
				ttyputerr("Cannot find layer %s", describenodeproto(ns->layer));
				return(1);
			}

			/* only add overlappable layers when k=0 */
			if (k == 0)
			{
				if (tech->layers[j]->bits == LAYERO) continue;
			} else
			{
				if (tech->layers[j]->bits != LAYERO) continue;
			}

			tech->arcprotos[arcindex]->list[layerindex].lay = j;

			/* determine the style of this arc layer */
			if (ns->node->proto == art_filledboxprim)
				tech->arcprotos[arcindex]->list[layerindex].style = FILLED; else
					tech->arcprotos[arcindex]->list[layerindex].style = CLOSED;

			/* determine the width offset of this arc layer */
			wid = mini(ns->node->highx-ns->node->lowx, ns->node->highy-ns->node->lowy);
			tech->arcprotos[arcindex]->list[layerindex].off = (maxwid-wid) * WHOLE /
				el_curtech->deflambda;

			layerindex++;
		}
	}

	/* store width offset on the technology */
	if ((us_tecflags&HASARCWID) != 0)
		(void)setval((INTBIG)tech, VTECHNOLOGY, "TECH_arc_width_offset", (INTBIG)us_tecarc_widoff,
			VFRACT|VDONTSAVE|VISARRAY|(tech->arcprotocount<<VLENGTHSH));
	efree((char *)sequence);
	return(0);
}

/*
 * routine to scan the "dependentlibcount" libraries in "dependentlibs",
 * and build the node structures for it in technology "tech".  Returns nonzero on error.
 */
INTSML us_tecedmakenodes(LIBRARY **dependentlibs, INTSML dependentlibcount,
	TECHNOLOGY *tech)
{
	REGISTER NODEPROTO *np;
	NODEPROTO **sequence;
	REGISTER NODEINST *ni;
	REGISTER VARIABLE *var;
	REGISTER char *str;
	REGISTER INTBIG *list, save1, save2, nfunction, x1pos, x2pos, y1pos, y2pos, net;
	REGISTER INTSML i, j, k, l, m, pass, nodeindex, sty, difindex, polindex,
		serpdifind, opt, nsindex, err, portchecked;
	INTSML pol1port, pol2port, dif1port, dif2port;
	INTBIG serprule[8];
	REGISTER EXAMPLE *ne, *nelist, *nextne;
	REGISTER SAMPLE *ns, *nextns, *ons, *diflayer, *pollayer;
	REGISTER PCON *pc;
	REGISTER RULE *r;
	REGISTER TECH_NODES *tlist;

	/* no rectangle rules or port connections */
	us_tecedfirstrule = NORULE;
	us_tecedfirstpcon = NOPCON;

	tech->nodeprotocount = us_teceditfindsequence(dependentlibs, dependentlibcount, "node-",
		"EDTEC_nodesequence", &sequence);
	if (tech->nodeprotocount <= 0)
	{
		ttyputerr("No nodes found");
		if ((us_aid->aidstate&NODETAILS) == 0)
			ttyputerr("Create them with the 'edit-node' option");
		return(1);
	}

	/* allocate the nodes */
	tech->nodeprotos = (TECH_NODES **)emalloc((tech->nodeprotocount+1) *
		(sizeof (TECH_NODES *)), tech->cluster);
	if (tech->nodeprotos == 0) return(1);
	tech->nodeprotos[tech->nodeprotocount] = ((TECH_NODES *)-1);
	us_tecnode_widoff = emalloc((4*tech->nodeprotocount * SIZEOFINTBIG), us_aid->cluster);
	if (us_tecnode_widoff == 0) return(1);
	us_tecnode_grab = emalloc((3*tech->nodeprotocount * SIZEOFINTBIG), us_aid->cluster);
	if (us_tecnode_grab == 0) return(1);
	us_tecnode_grabcount = 0;

	/* get the nodes */
	nodeindex = 0;
	for(pass=0; pass<3; pass++)
		for(m=0; m<tech->nodeprotocount; m++)
	{
		/* make sure this is the right type of node for this pass of the nodes */
		np = sequence[m];
		nfunction = NPUNKNOWN;
		for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
		{
			/* get the node function */
			if (us_tecedgetoption(ni) != NODEFUNCTION) continue;
			var = getvalkey((INTBIG)ni, VNODEINST, VSTRING, art_messagekey);
			if (var == NOVARIABLE) continue;
			str = &((char *)var->addr)[10];
			for(j=0; us_tecnode_functions[j].name != 0; j++)
				if (namesame(str, us_tecnode_functions[j].name) == 0) break;
			if (us_tecnode_functions[j].name != 0)
			{
				nfunction = us_tecnode_functions[j].value;
				break;
			}
		}

		/* only want pins on pass 0, pure-layer nodes on pass 2 */
		if (pass == 0 && nfunction != NPPIN) continue;
		if (pass == 1 && (nfunction == NPPIN || nfunction == NPNODE)) continue;
		if (pass == 2 && nfunction != NPNODE) continue;

		/* build a list of examples found in this node */
		nelist = us_tecedgetexamples(np, 1);
		if (nelist == NOEXAMPLE) return(1);

		/* associate the samples in each example */
		if (us_tecedassociateexamples(nelist, np) != 0) return(1);

		/* derive primitives from the examples */
		if (us_tecedmakeprim(nelist, np, tech) != 0) return(1);

		/* allocate and fill the TECH_NODES structure */
		tlist = (TECH_NODES *)emalloc(sizeof (TECH_NODES), tech->cluster);
		if (tlist == 0) return(1);
		tech->nodeprotos[nodeindex] = tlist;
		(void)allocstring(&tlist->nodename, &np->cell->cellname[5], tech->cluster);
		tlist->index = nodeindex + 1;
		tlist->creation = NONODEPROTO;
		tlist->xsize = (nelist->hx-nelist->lx)*WHOLE/el_curtech->deflambda;
		tlist->ysize = (nelist->hy-nelist->ly)*WHOLE/el_curtech->deflambda;
		tlist->layerlist = 0;
		tlist->layercount = 0;
		tlist->special = 0;
		tlist->f1 = 0;
		tlist->f2 = 0;
		tlist->f3 = 0;
		tlist->f4 = 0;
		tlist->gra = 0;
		tlist->ele = 0;

		/* determine user bits */
		tlist->initialbits = nfunction<<NFUNCTIONSH;
		for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
		{
			opt = us_tecedgetoption(ni);

			/* pick up square node information */
			if (opt == NODESQUARE)
			{
				var = getvalkey((INTBIG)ni, VNODEINST, VSTRING, art_messagekey);
				if (var == NOVARIABLE) continue;
				str = &((char *)var->addr)[13];
				if (namesame(str, "yes") == 0) tlist->initialbits |= NSQUARE;
				continue;
			}

			/* pick up invisible on 1 or 2 arc information */
			if (opt == NODEWIPES)
			{
				var = getvalkey((INTBIG)ni, VNODEINST, VSTRING, art_messagekey);
				if (var == NOVARIABLE) continue;
				str = &((char *)var->addr)[28];
				if (namesame(str, "yes") == 0)
					tlist->initialbits = WIPEON1OR2 | (tlist->initialbits & ~(ARCSWIPE|ARCSHRINK));
				continue;
			}

			/* pick up lockable information */
			if (opt == NODELOCKABLE)
			{
				var = getvalkey((INTBIG)ni, VNODEINST, VSTRING, art_messagekey);
				if (var == NOVARIABLE) continue;
				str = &((char *)var->addr)[10];
				if (namesame(str, "yes") == 0) tlist->initialbits |= LOCKEDPRIM;
				continue;
			}

			/* pick up serpentine transistor information */
			if (opt == NODESERPENTINE)
			{
				var = getvalkey((INTBIG)ni, VNODEINST, VSTRING, art_messagekey);
				if (var == NOVARIABLE) continue;
				str = &((char *)var->addr)[23];
				if (namesame(str, "yes") == 0)
				{
					if (tlist->special != 0)
					{
						us_tecedpointout(ni, np);
						ttyputerr("Serpentine %s must have Transistor function",
							describenodeproto(np));
						return(1);
					}
					tlist->special = SERPTRANS;
					tlist->initialbits |= (NODESHRINK | HOLDSTRACE);
				}
				continue;
			}
		}

		/* analyze special node function circumstances */
		switch (nfunction)
		{
			case NPNODE:
				if (tlist->special != 0)
				{
					us_tecedpointout(NONODEINST, np);
					ttyputerr("Pure layer %s can not be serpentine", describenodeproto(np));
					return(1);
				}
				tlist->special = POLYGONAL;
				tlist->initialbits |= HOLDSTRACE;
				break;
			case NPPIN:
				if ((tlist->initialbits&WIPEON1OR2) == 0)
				{
					tlist->initialbits |= ARCSWIPE;
					tlist->initialbits |= ARCSHRINK;
				}
				break;
		}

		/* count the number of ports on this node */
		tlist->portcount = 0;
		for(ns = nelist->firstsample; ns != NOSAMPLE; ns = ns->nextsample)
			if (ns->layer == gen_portprim) tlist->portcount++;
		if (tlist->portcount == 0)
		{
			us_tecedpointout(NONODEINST, np);
			ttyputerr("No ports found in %s", describenodeproto(np));
			if ((us_aid->aidstate&NODETAILS) == 0)
				ttyputmsg("Use 'place-layer port' option to create one");
			return(1);
		}

		/* allocate space for the ports */
		tlist->portlist = (TECH_PORTS *)emalloc((tlist->portcount * (sizeof (TECH_PORTS))),
			tech->cluster);
		if (tlist->portlist == 0) return(1);

		/* fill the port structures */
		pol1port = pol2port = dif1port = dif2port = -1;
		i = 0;
		for(ns = nelist->firstsample; ns != NOSAMPLE; ns = ns->nextsample)
		{
			if (ns->layer != gen_portprim) continue;

			/* port connections */
			var = getval((INTBIG)ns->node, VNODEINST, VNODEPROTO|VISARRAY, "EDTEC_connects");
			if (var == NOVARIABLE) pc = us_tecedaddportlist(0, (INTBIG *)0); else
			{
				/* convert "arc-FACET" pointers to indices */
				l = getlength(var);
				list = emalloc((l * SIZEOFINTBIG), el_tempcluster);
				if (list == 0) return(1);
				portchecked = 0;
				for(j=0; j<l; j++)
				{
					/* find arc that connects */
					for(k=0; k<tech->arcprotocount; k++)
						if (namesame(tech->arcprotos[k]->arcname,
							&((NODEPROTO **)var->addr)[j]->cell->cellname[4]) == 0) break;
					if (k >= tech->arcprotocount)
					{
						us_tecedpointout(ns->node, ns->node->parent);
						ttyputerr("Invalid connection list on port in %s", describenodeproto(np));
						if ((us_aid->aidstate&NODETAILS) == 0)
							ttyputmsg("Use 'change' option to remove arc %s",
								&((NODEPROTO **)var->addr)[j]->cell->cellname[4]);
						return(1);
					}

					/* fill in the connection information */
					list[j] = k;

					/* find port characteristics for possible transistors */
					if (portchecked != 0) continue;
					switch ((tech->arcprotos[k]->initialbits&AFUNCTION)>>AFUNCTIONSH)
					{
						case APPOLY1:   case APPOLY2:
							if (pol1port < 0)
							{
								pol1port = i;
								portchecked++;
							} else if (pol2port < 0)
							{
								pol2port = i;
								portchecked++;
							}
							break;
						case APDIFF:    case APDIFFP:   case APDIFFN:
						case APDIFFS:   case APDIFFW:
							if (dif1port < 0)
							{
								dif1port = i;
								portchecked++;
							} else if (dif2port < 0)
							{
								dif2port = i;
								portchecked++;
							}
							break;
					}
				}

				/* store complete list in memory */
				pc = us_tecedaddportlist(l, list);
				efree((char *)list);
			}

			/* link connection list to the port */
			if (pc == NOPCON) return(1);
			tlist->portlist[i].portarcs = pc->connects;

			/* port name */
			var = getval((INTBIG)ns->node, VNODEINST, VSTRING, "EDTEC_portname");
			if (var == NOVARIABLE) continue;
			for(str = (char *)var->addr; *str != 0; str++)
				if (*str <= ' ' || *str >= 0177)
			{
				us_tecedpointout(ns->node, np);
				ttyputerr("Invalid port name: '%s'", (char *)var->addr);
				return(1);
			}
			(void)allocstring(&tlist->portlist[i].protoname, (char *)var->addr, tech->cluster);

			/* port angle and range */
			tlist->portlist[i].initialbits = 0;
			var = getval((INTBIG)ns->node, VNODEINST, VINTEGER, "EDTEC_portangle");
			if (var != NOVARIABLE)
				tlist->portlist[i].initialbits |= var->addr << PORTANGLESH;
			var = getval((INTBIG)ns->node, VNODEINST, VINTEGER, "EDTEC_portrange");
			if (var != NOVARIABLE)
				tlist->portlist[i].initialbits |= var->addr << PORTARANGESH; else
					tlist->portlist[i].initialbits |= 180 << PORTARANGESH;

			/* port connectivity */
			net = i;
			if (ns->node->firstportarcinst != NOPORTARCINST)
			{
				j = 0;
				for(ons = nelist->firstsample; ons != ns; ons = ons->nextsample)
				{
					if (ons->layer != gen_portprim) continue;
					if (ons->node->firstportarcinst != NOPORTARCINST)
					{
						if (ns->node->firstportarcinst->conarcinst->network ==
							ons->node->firstportarcinst->conarcinst->network)
						{
							net = j;
							break;
						}
					}
					j++;
				}
			}
			tlist->portlist[i].initialbits |= (net << PORTNETSH);

			/* port area rule */
			r = ns->rule;
			if (r == NORULE) continue;
			tlist->portlist[i].lowxmul = r->value[0];
			tlist->portlist[i].lowxsum = r->value[1];
			tlist->portlist[i].lowymul = r->value[2];
			tlist->portlist[i].lowysum = r->value[3];
			tlist->portlist[i].highxmul = r->value[4];
			tlist->portlist[i].highxsum = r->value[5];
			tlist->portlist[i].highymul = r->value[6];
			tlist->portlist[i].highysum = r->value[7];
			i++;
		}

		/* on FET transistors, make sure ports 0 and 2 are poly */
		if (nfunction == NPTRANMOS || nfunction == NPTRADMOS || nfunction == NPTRAPMOS ||
			nfunction == NPTRADMES || nfunction == NPTRAEMES)
		{
			if (pol1port < 0 || pol2port < 0 || dif1port < 0 || dif2port < 0)
			{
				us_tecedpointout(NONODEINST, np);
				ttyputerr("Need 2 gate and 2 active ports on field-effect transistor %s",
					describenodeproto(np));
				return(1);
			}
			if (pol1port != 0)
			{
				if (pol2port == 0) us_tecedswapports(&pol1port, &pol2port, tlist); else
				if (dif1port == 0) us_tecedswapports(&pol1port, &dif1port, tlist); else
				if (dif2port == 0) us_tecedswapports(&pol1port, &dif2port, tlist);
			}
			if (pol2port != 2)
			{
				if (dif1port == 2) us_tecedswapports(&pol2port, &dif1port, tlist); else
				if (dif2port == 2) us_tecedswapports(&pol2port, &dif2port, tlist);
			}
			if (dif1port != 1) us_tecedswapports(&dif1port, &dif2port, tlist);

			/* also make sure that dif1port is positive and dif2port is negative */
			x1pos = (tlist->portlist[dif1port].lowxmul*tlist->xsize +
				tlist->portlist[dif1port].lowxsum +
					tlist->portlist[dif1port].highxmul*tlist->xsize +
						tlist->portlist[dif1port].highxsum) / 2;
			x2pos = (tlist->portlist[dif2port].lowxmul*tlist->xsize +
				tlist->portlist[dif2port].lowxsum +
					tlist->portlist[dif2port].highxmul*tlist->xsize +
						tlist->portlist[dif2port].highxsum) / 2;
			y1pos = (tlist->portlist[dif1port].lowymul*tlist->ysize +
				tlist->portlist[dif1port].lowysum +
					tlist->portlist[dif1port].highymul*tlist->ysize +
						tlist->portlist[dif1port].highysum) / 2;
			y2pos = (tlist->portlist[dif2port].lowymul*tlist->ysize +
				tlist->portlist[dif2port].lowysum +
					tlist->portlist[dif2port].highymul*tlist->ysize +
						tlist->portlist[dif2port].highysum) / 2;
			if (abs(x1pos-x2pos) > abs(y1pos-y2pos))
			{
				if (x1pos < x2pos)
				{
					us_tecedswapports(&dif1port, &dif2port, tlist);
					j = dif1port;   dif1port = dif2port;   dif2port = j;
				}
			} else
			{
				if (y1pos < y2pos)
				{
					us_tecedswapports(&dif1port, &dif2port, tlist);
					j = dif1port;   dif1port = dif2port;   dif2port = j;
				}
			}

			/* also make sure that pol1port is negative and pol2port is positive */
			x1pos = (tlist->portlist[pol1port].lowxmul*tlist->xsize +
				tlist->portlist[pol1port].lowxsum +
					tlist->portlist[pol1port].highxmul*tlist->xsize +
						tlist->portlist[pol1port].highxsum) / 2;
			x2pos = (tlist->portlist[pol2port].lowxmul*tlist->xsize +
				tlist->portlist[pol2port].lowxsum +
					tlist->portlist[pol2port].highxmul*tlist->xsize +
						tlist->portlist[pol2port].highxsum) / 2;
			y1pos = (tlist->portlist[pol1port].lowymul*tlist->ysize +
				tlist->portlist[pol1port].lowysum +
					tlist->portlist[pol1port].highymul*tlist->ysize +
						tlist->portlist[pol1port].highysum) / 2;
			y2pos = (tlist->portlist[pol2port].lowymul*tlist->ysize +
				tlist->portlist[pol2port].lowysum +
					tlist->portlist[pol2port].highymul*tlist->ysize +
						tlist->portlist[pol2port].highysum) / 2;
			if (abs(x1pos-x2pos) > abs(y1pos-y2pos))
			{
				if (x1pos > x2pos)
				{
					us_tecedswapports(&pol1port, &pol2port, tlist);
					j = pol1port;   pol1port = pol2port;   pol2port = j;
				}
			} else
			{
				if (y1pos > y2pos)
				{
					us_tecedswapports(&pol1port, &pol2port, tlist);
					j = pol1port;   pol1port = pol2port;   pol2port = j;
				}
			}
		}

		/* count the number of layers on the node */
		tlist->layercount = 0;
		for(ns = nelist->firstsample; ns != NOSAMPLE; ns = ns->nextsample)
		{
			if (ns->rule != NORULE && ns->layer != gen_portprim &&
				ns->layer != gen_facetcenterprim && ns->layer != NONODEPROTO)
					tlist->layercount++;
		}

		/* allocate space for the layers */
		if (tlist->special != SERPTRANS)
		{
			tlist->layerlist = (TECH_POLYGON *)emalloc((tlist->layercount *
				(sizeof (TECH_POLYGON))), tech->cluster);
			if (tlist->layerlist == 0) return(1);
		} else
		{
			tlist->gra = (TECH_SERPENT *)emalloc(((sizeof (TECH_SERPENT)) * tlist->layercount),
				tech->cluster);
			if (tlist->gra == 0) return(1);
			tlist->ele = (TECH_SERPENT *)emalloc(((sizeof (TECH_SERPENT)) * (tlist->layercount+2)),
				tech->cluster);
			if (tlist->ele == 0) return(1);
		}

		/* fill the layer structures (3 times: overlappable, opaque, multicut) */
		i = 0;
		pollayer = diflayer = NOSAMPLE;
		for(k=0; k<3; k++)
			for(nsindex=0, ns = nelist->firstsample; ns != NOSAMPLE; nsindex++, ns = ns->nextsample)
		{
			r = ns->rule;
			if (r == NORULE || ns->layer == gen_portprim ||
				ns->layer == gen_facetcenterprim || ns->layer == NONODEPROTO) continue;

			/* add cut layers last (only when k=2) */
			if (k == 2)
			{
				if (r->multicut == 0) continue;
				if (tlist->special != 0)
				{
					us_tecedpointout(ns->node, ns->node->parent);
					ttyputerr("%s is too complex (multiple cuts AND serpentine)",
						describenodeproto(np));
					return(1);
				}
				tlist->special = MULTICUT;
				tlist->f1 = r->multixs*WHOLE/el_curtech->deflambda;
				tlist->f2 = r->multiys*WHOLE/el_curtech->deflambda;
				tlist->f3 = r->multiindent*WHOLE/el_curtech->deflambda;
				tlist->f4 = r->multisep*WHOLE/el_curtech->deflambda;
			} else
			{
				if (r->multicut != 0) continue;
			}

			/* layer number */
			for(j=0; j<tech->layercount; j++)
				if (namesame(&ns->layer->cell->cellname[6], us_teclayer_names[j]) == 0) break;
			if (j >= tech->layercount)
			{
				ttyputerr("Cannot find layer %s", describenodeproto(ns->layer));
				return(1);
			}

			/* only add overlappable layers when k=0 */
			if (k == 0)
			{
				if (tech->layers[j]->bits == LAYERO) continue;
			} else if (k == 1)
			{
				if (tech->layers[j]->bits != LAYERO) continue;
			}

			/* layer style */
			sty = -1;
			if (ns->node->proto == art_filledboxprim)             sty = FILLEDRECT; else
			if (ns->node->proto == art_boxprim)                   sty = CLOSEDRECT; else
			if (ns->node->proto == art_crossedboxprim)            sty = CROSSED; else
			if (ns->node->proto == art_filledpolygonprim)         sty = FILLED; else
			if (ns->node->proto == art_closedpolygonprim)         sty = CLOSED; else
			if (ns->node->proto == art_openedpolygonprim)         sty = OPENED; else
			if (ns->node->proto == art_openeddottedpolygonprim)   sty = OPENEDT1; else
			if (ns->node->proto == art_openeddashedpolygonprim)   sty = OPENEDT2; else
			if (ns->node->proto == art_openedfardottepolygonprim) sty = OPENEDT3; else
			if (ns->node->proto == art_filledcircleprim)          sty = DISC; else
			if (ns->node->proto == art_circleprim)
			{
				var = getvalkey((INTBIG)ns->node, VNODEINST, VINTEGER, art_degreeskey);
				if (var != NOVARIABLE) sty = CIRCLEARC; else sty = CIRCLE;
			} else if (ns->node->proto == gen_invispinprim)
			{
				var = getvalkey((INTBIG)ns->node, VNODEINST, VSTRING|VISARRAY, art_messagekey);
				if (var != NOVARIABLE)
				{
					switch (var->textdescript&VTPOSITION)
					{
						case VTPOSBOXED:     sty = TEXTBOX;       break;
						case VTPOSCENT:      sty = TEXTCENT;      break;
						case VTPOSUP:        sty = TEXTBOT;       break;
						case VTPOSDOWN:      sty = TEXTTOP;       break;
						case VTPOSLEFT:      sty = TEXTRIGHT;     break;
						case VTPOSRIGHT:     sty = TEXTLEFT;      break;
						case VTPOSUPLEFT:    sty = TEXTBOTRIGHT;  break;
						case VTPOSUPRIGHT:   sty = TEXTBOTLEFT;   break;
						case VTPOSDOWNLEFT:  sty = TEXTTOPRIGHT;  break;
						case VTPOSDOWNRIGHT: sty = TEXTTOPLEFT;   break;
					}
				}
			}
			if (sty == -1)
				ttyputmsg("Cannot determine style to use for %s node in %s",
					describenodeproto(ns->node->proto), describenodeproto(np));

			/* load the layer structure(s) */
			if (tlist->special == SERPTRANS)
			{
				/* determine port numbers for serpentine transistors */
				switch (us_teclayer_function[j]&LFTYPE)
				{
					case LFMETAL1: case LFMETAL2: case LFMETAL3: case LFMETAL4:
					case LFMETAL5: case LFMETAL6: case LFMETAL7: case LFMETAL8:
						tlist->gra[i].basics.portnum = 0;
						break;
					case LFPOLY1:  case LFPOLY2:
						pollayer = ns;
						if (pol1port >= 0)
							tlist->gra[i].basics.portnum = pol1port; else
								tlist->gra[i].basics.portnum = 0;
						polindex = i;
						break;
					case LFDIFF:
						diflayer = ns;
						difindex = i;
						tlist->gra[i].basics.portnum = 0;
						break;
					default:
						tlist->gra[i].basics.portnum = -1;
						break;
				}

				tlist->gra[i].basics.layernum = j;
				tlist->gra[i].basics.count = r->count/4;
				tlist->gra[i].basics.style = sty;
				if (tlist->gra[i].basics.count == 2 && (sty == CROSSED ||
					sty == FILLEDRECT || sty == FILLED || sty == CLOSEDRECT || sty == CLOSED))
				{
					tlist->gra[i].basics.representation = BOX;
					tlist->gra[i].basics.count = 4;
				} else tlist->gra[i].basics.representation = POINTS;
				tlist->gra[i].basics.points = r->value;
				tlist->gra[i].lwidth = nsindex;
				tlist->gra[i].rwidth = 0;
				tlist->gra[i].extend = 0;
			} else
			{
				tlist->layerlist[i].portnum = us_tecedfindport(tlist, nelist,
					ns->node->lowx, ns->node->highx, ns->node->lowy, ns->node->highy,
						ns->node->proto->tech->deflambda);
				tlist->layerlist[i].layernum = j;
				tlist->layerlist[i].count = r->count/4;
				tlist->layerlist[i].style = sty;
				tlist->layerlist[i].representation = POINTS;
				if (sty == CROSSED || sty == FILLEDRECT || sty == FILLED || sty == CLOSEDRECT ||
					sty == CLOSED)
				{
					if (r->count == 8)
					{
						tlist->layerlist[i].representation = BOX;
						tlist->layerlist[i].count = 4;
					} else if (r->count == 16)
					{
						tlist->layerlist[i].representation = MINBOX;
						tlist->layerlist[i].count = 4;
					}
				}
				tlist->layerlist[i].points = r->value;
			}

			/* mark this rectangle rule "used" */
			r->used++;
			i++;
		}

		/* finish up serpentine transistors */
		if (tlist->special == SERPTRANS)
		{
			if (diflayer == NOSAMPLE || pollayer == NOSAMPLE || dif1port < 0)
			{
				us_tecedpointout(NONODEINST, np);
				ttyputerr("No diffusion and polysilicon layers in transistor %s",
					describenodeproto(np));
				return(1);
			}

			/* compute port extension factors */
			if (tlist->portlist[dif1port].lowxsum >
				tlist->portlist[dif1port].lowysum)
			{
				/* vertical diffusion layer */
				tlist->f1 = tlist->portlist[dif1port].lowxsum -
					tlist->gra[difindex].basics.points[1];
				tlist->f2 = tlist->gra[polindex].basics.points[3] -
					abs(tlist->portlist[dif1port].lowysum);
				tlist->f3 = tlist->ysize - (tlist->gra[polindex].basics.points[3] -
					tlist->gra[polindex].basics.points[7]);
			} else
			{
				/* horizontal diffusion layer */
				tlist->f1 = tlist->portlist[dif1port].lowysum -
					tlist->gra[difindex].basics.points[3];
				tlist->f2 = tlist->gra[polindex].basics.points[1] -
					abs(tlist->portlist[dif1port].lowxsum);
				tlist->f3 = tlist->xsize - (tlist->gra[polindex].basics.points[1] -
					tlist->gra[polindex].basics.points[5]);
			}

			/* find width and extension from comparison to poly layer */
			for(i=0; i<tlist->layercount; i++)
			{
				for(nsindex=0, ns = nelist->firstsample; ns != NOSAMPLE;
					nsindex++, ns = ns->nextsample)
						if (tlist->gra[i].lwidth == nsindex) break;
				if (ns == NOSAMPLE)
				{
					us_tecedpointout(NONODEINST, np);
					ttyputerr("Internal error in serpentine %s", describenodeproto(np));
					continue;
				}

				if (pollayer->node->highx-pollayer->node->lowx >
					pollayer->node->highy-pollayer->node->lowy)
				{
					/* horizontal layer */
					tlist->gra[i].lwidth = (ns->node->highy - (ns->parent->ly + ns->parent->hy)/2) *
						WHOLE/el_curtech->deflambda;
					tlist->gra[i].rwidth = ((ns->parent->ly + ns->parent->hy)/2 - ns->node->lowy) *
						WHOLE/el_curtech->deflambda;
					tlist->gra[i].extend = (diflayer->node->lowx - ns->node->lowx) * WHOLE /
						el_curtech->deflambda;
				} else
				{
					/* vertical layer */
					tlist->gra[i].lwidth = (ns->node->highx - (ns->parent->lx + ns->parent->hx)/2) *
						WHOLE/el_curtech->deflambda;
					tlist->gra[i].rwidth = ((ns->parent->lx + ns->parent->hx)/2 - ns->node->lowx) *
						WHOLE/el_curtech->deflambda;
					tlist->gra[i].extend = (diflayer->node->lowy - ns->node->lowy) * WHOLE /
						el_curtech->deflambda;
				}
			}

			/* copy basic graphics to electrical version, tripling diffusion */
			i = 0;
			for(j=0; j<tlist->layercount; j++)
			{
				if (j != difindex) k = 1; else
				{
					k = 3;

					/* copy rectangle rule and prepare for electrical layers */
					r = diflayer->rule;
					if (r->count != 8)
					{
						us_tecedpointout(NONODEINST, np);
						ttyputerr("Nonrectangular diffusion in Serpentine %s",
							describenodeproto(np));
						return(1);
					}
					for(l=0; l<r->count; l++) serprule[l] = r->value[l];
					if (serprule[0] != -H0 || serprule[2] != -H0 ||
						serprule[4] != H0 || serprule[6] != H0)
					{
						us_tecedpointout(NONODEINST, np);
						ttyputerr("Unusual diffusion in Serpentine %s", describenodeproto(np));
						return(1);
					}
					if (tlist->xsize - serprule[1] + serprule[5] <
						tlist->ysize - serprule[3] + serprule[7]) serpdifind = 2; else
							serpdifind = 0;
				}
				for(l=0; l<k; l++)
				{
					tlist->ele[i].basics.layernum = tlist->gra[j].basics.layernum;
					tlist->ele[i].basics.count = tlist->gra[j].basics.count;
					tlist->ele[i].basics.style = tlist->gra[j].basics.style;
					tlist->ele[i].basics.representation = tlist->gra[j].basics.representation;
					tlist->ele[i].basics.points = tlist->gra[j].basics.points;
					tlist->ele[i].lwidth = tlist->gra[j].lwidth;
					tlist->ele[i].rwidth = tlist->gra[j].rwidth;
					tlist->ele[i].extend = tlist->gra[j].extend;
					if (k != 3) tlist->ele[i].basics.portnum = tlist->gra[j].basics.portnum; else
						switch (l)
					{
						case 0:
							tlist->ele[i].basics.portnum = dif1port;
							tlist->ele[i].rwidth = -tlist->gra[polindex].lwidth;
							save1 = serprule[serpdifind+1];
							serprule[serpdifind] = H0;
							serprule[serpdifind+1] = pollayer->rule->value[serpdifind+5];
							r = us_tecedaddrule(serprule, 8, 0, (char *)0);
							if (r == NORULE) return(1);
							r->used++;
							tlist->ele[i].basics.points = r->value;
							serprule[serpdifind] = -H0;
							serprule[serpdifind+1] = save1;
							break;
						case 1:
							tlist->ele[i].basics.portnum = -1;
							tlist->ele[i].rwidth = tlist->gra[polindex].rwidth;
							tlist->ele[i].lwidth = tlist->gra[polindex].lwidth;
							save1 = serprule[serpdifind+1];
							save2 = serprule[serpdifind+5];
							serprule[serpdifind+1] = pollayer->rule->value[serpdifind+1];
							serprule[serpdifind+5] = pollayer->rule->value[serpdifind+5];
							r = us_tecedaddrule(serprule, 8, 0, (char *)0);
							if (r == NORULE) return(1);
							r->used++;
							tlist->ele[i].basics.points = r->value;
							serprule[serpdifind+1] = save1;
							serprule[serpdifind+5] = save2;
							break;
						case 2:
							tlist->ele[i].basics.portnum = dif2port;
							tlist->ele[i].lwidth = -tlist->gra[polindex].rwidth;
							save1 = serprule[serpdifind+5];
							serprule[serpdifind+4] = -H0;
							serprule[serpdifind+5] = pollayer->rule->value[serpdifind+1];
							r = us_tecedaddrule(serprule, 8, 0, (char *)0);
							if (r == NORULE) return(1);
							r->used++;
							tlist->ele[i].basics.points = r->value;
							serprule[serpdifind+4] = H0;
							serprule[serpdifind+5] = save1;
							break;
					}
					i++;
				}
			}
		}

		/* extract width offset information */
		us_tecnode_widoff[nodeindex*4] = 0;
		us_tecnode_widoff[nodeindex*4+1] = 0;
		us_tecnode_widoff[nodeindex*4+2] = 0;
		us_tecnode_widoff[nodeindex*4+3] = 0;
		for(ns = nelist->firstsample; ns != NOSAMPLE; ns = ns->nextsample)
			if (ns->layer == NONODEPROTO) break;
		if (ns != NOSAMPLE)
		{
			r = ns->rule;
			if (r != NORULE)
			{
				err = 0;
				switch (r->value[0])		/* left edge offset */
				{
					case -H0:
						us_tecnode_widoff[nodeindex*4] = r->value[1];
						break;
					case H0:
						us_tecnode_widoff[nodeindex*4] = tlist->xsize + r->value[1];
						break;
					default:
						err++;
						break;
				}
				switch (r->value[2])		/* bottom edge offset */
				{
					case -H0:
						us_tecnode_widoff[nodeindex*4+2] = r->value[3];
						break;
					case H0:
						us_tecnode_widoff[nodeindex*4+2] = tlist->ysize + r->value[3];
						break;
					default:
						err++;
						break;
				}
				switch (r->value[4])		/* right edge offset */
				{
					case H0:
						us_tecnode_widoff[nodeindex*4+1] = -r->value[5];
						break;
					case -H0:
						us_tecnode_widoff[nodeindex*4+1] = tlist->xsize - r->value[5];
						break;
					default:
						err++;
						break;
				}
				switch (r->value[6])		/* top edge offset */
				{
					case H0:
						us_tecnode_widoff[nodeindex*4+3] = -r->value[7];
						break;
					case -H0:
						us_tecnode_widoff[nodeindex*4+3] = tlist->ysize - r->value[7];
						break;
					default:
						err++;
						break;
				}
				if (err != 0)
				{
					us_tecedpointout(ns->node, ns->node->parent);
					ttyputmsg("Highlighting cannot scale from center in %s", describenodeproto(np));
					return(1);
				}
			} else
			{
				us_tecedpointout(ns->node, ns->node->parent);
				ttyputerr("No rule found for highlight in %s", describenodeproto(np));
				return(1);
			}
		} else
		{
			us_tecedpointout(NONODEINST, np);
			ttyputerr("No highlight found in %s", describenodeproto(np));
			if ((us_aid->aidstate&NODETAILS) == 0)
				ttyputmsg("Use 'place-layer' option to create HIGHLIGHT");
			return(1);
		}

		/* get grab point information */
		for(ns = nelist->firstsample; ns != NOSAMPLE; ns = ns->nextsample)
			if (ns->layer == gen_facetcenterprim) break;
		if (ns != NOSAMPLE)
		{
			us_tecnode_grab[us_tecnode_grabcount++] = nodeindex+1;
			us_tecnode_grab[us_tecnode_grabcount++] = (ns->node->geom->lowx +
				ns->node->geom->highx - nelist->lx - nelist->hx)/2 * tech->deflambda /
					el_curtech->deflambda;
			us_tecnode_grab[us_tecnode_grabcount++] = (ns->node->geom->lowy +
				ns->node->geom->highy - nelist->ly - nelist->hy)/2 * tech->deflambda /
					el_curtech->deflambda;
			us_tecflags |= HASGRAB;
		}

		/* free all examples */
		for(ne = nelist; ne != NOEXAMPLE; ne = nextne)
		{
			nextne = ne->nextexample;
			for(ns = ne->firstsample; ns != NOSAMPLE; ns = nextns)
			{
				nextns = ns->nextsample;
				efree((char *)ns);
			}
			efree((char *)ne);
		}

		/* advance the fill pointer */
		nodeindex++;
	}

	/* store width offset on the technology */
	(void)setval((INTBIG)tech, VTECHNOLOGY, "TECH_node_width_offset", (INTBIG)us_tecnode_widoff,
		VFRACT|VDONTSAVE|VISARRAY|((tech->nodeprotocount*4)<<VLENGTHSH));
	efree((char *)sequence);

	return(0);
}

/*
 * routine to find the closest port to the layer describe by "lx<=X<=hx" and
 * "ly<+Y<=hy" in the list "nelist".  The ports are listed in "tlist".  The algorithm
 * is to find a port that overlaps this layer.  If there is only one, or if all of
 * them electrically connect, use that.  If there are no such ports, or multiple
 * unconnected ports, presume that the layer is not related to any port.
 */
INTSML us_tecedfindport(TECH_NODES *tlist, EXAMPLE *nelist, INTBIG lx, INTBIG hx, INTBIG ly,
	INTBIG hy, INTBIG lambda)
{
	REGISTER INTSML bestport, l, oldnet, newnet;
	INTBIG portlx, porthx, portly, porthy;
	REGISTER INTBIG swap;

	bestport = -1;
	for(l=0; l<tlist->portcount; l++)
	{
		subrange(nelist->lx, nelist->hx, tlist->portlist[l].lowxmul,
			tlist->portlist[l].lowxsum, tlist->portlist[l].highxmul,
				tlist->portlist[l].highxsum, &portlx, &porthx, lambda);
		if (portlx > porthx)
		{
			swap = portlx;   portlx = porthx;   porthx = swap;
		}
		subrange(nelist->ly, nelist->hy, tlist->portlist[l].lowymul,
			tlist->portlist[l].lowysum, tlist->portlist[l].highymul,
				tlist->portlist[l].highysum, &portly, &porthy, lambda);
		if (portlx > porthx)
		{
			swap = portly;   portly = porthy;   porthy = swap;
		}

		/* ignore the port if there is no intersection */
		if (lx > porthx || hx < portlx || ly > porthy || hy < portly) continue;

		/* if there is no previous overlapping port, use this */
		if (bestport == -1)
		{
			bestport = l;
			continue;
		}

		/* if these two ports connect, all is well */
		newnet = (tlist->portlist[l].initialbits & PORTNET) >> PORTNETSH;
		oldnet = (tlist->portlist[bestport].initialbits & PORTNET) >> PORTNETSH;
		if (newnet == oldnet) continue;

		/* two unconnected ports intersect layer: make it free */
		return(-1);
	}
	return(bestport);
}

/*
 * routine to parse the node examples in facet "np" and return a list of
 * EXAMPLEs (one per example).  "isnode" is nonzero if this is a node
 * being examined.  Returns NOEXAMPLE on error.
 */
EXAMPLE *us_tecedgetexamples(NODEPROTO *np, INTSML isnode)
{
	REGISTER SAMPLE *ns;
	REGISTER EXAMPLE *ne, *nelist, *bestne;
	REGISTER NODEINST *ni, *otherni;
	REGISTER INTBIG sea, sizex, sizey, newsize, locx, locy;
	REGISTER INTSML foundone, hcount, funct, gotbbox;
	INTBIG lx, hx, ly, hy, sflx, sfhx, sfly, sfhy;
	REGISTER GEOM *geom;
	REGISTER VARIABLE *var;

	for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		ni->temp1 = (INTBIG)NOEXAMPLE;

		/* ignore special nodes with function information */
		funct = us_tecedgetoption(ni);
		if (funct != LAYERPATCH && funct != PORTOBJ && funct != HIGHLIGHTOBJ &&
			funct != CENTEROBJ) ni->temp1 = 0;
	}

	nelist = NOEXAMPLE;
	for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		if (ni->temp1 != (INTBIG)NOEXAMPLE) continue;

		/* get a new cluster of nodes */
		ne = (EXAMPLE *)emalloc((sizeof (EXAMPLE)), us_aid->cluster);
		if (ne == 0) return(NOEXAMPLE);
		ne->firstsample = NOSAMPLE;
		gotbbox = 0;
		us_tecedgetbbox(ni, &sflx, &sfhx, &sfly, &sfhy);
		ne->nextexample = nelist;
		nelist = ne;

		/* now find all others that touch this area */
		foundone = 1;
		hcount = 0;
		while (foundone != 0)
		{
			foundone = 0;

			/* begin to search the area so far */
			sea = initsearch(sflx, sfhx, sfly, sfhy, np);
			if (sea == -1) return(NOEXAMPLE);
			for(;;)
			{
				/* get next node in the area */
				geom = nextobject(sea);
				if (geom == NOGEOM) break;
				if (geom->entrytype != OBJNODEINST) continue;
				otherni = geom->entryaddr.ni;

				/* make sure the node is valid */
				if (otherni->temp1 != (INTBIG)NOEXAMPLE)
				{
					if (otherni->temp1 == (INTBIG)ne) continue;
					us_tecedpointout(otherni, np);
					ttyputerr("Examples are too close in %s", describenodeproto(np));
					termsearch(sea);
					return(NOEXAMPLE);
				}
				otherni->temp1 = (INTBIG)ne;

				/* add it to the cluster */
				ns = (SAMPLE *)emalloc((sizeof (SAMPLE)), us_aid->cluster);
				if (ns == 0) return(NOEXAMPLE);
				ns->node = otherni;
				ns->rule = NORULE;
				ns->parent = ne;
				ns->nextsample = ne->firstsample;
				ne->firstsample = ns;
				ns->assoc = NOSAMPLE;
				ns->xpos = (otherni->geom->lowx + otherni->geom->highx) / 2;
				ns->ypos = (otherni->geom->lowy + otherni->geom->highy) / 2;
				if (otherni->proto == gen_portprim)
				{
					if (isnode == 0)
					{
						us_tecedpointout(otherni, np);
						ttyputerr("%s cannot have ports.  Delete this", describenodeproto(np));
						termsearch(sea);
						return(NOEXAMPLE);
					}
					ns->layer = gen_portprim;
				} else if (otherni->proto == gen_facetcenterprim)
				{
					if (isnode == 0)
					{
						us_tecedpointout(otherni, np);
						ttyputerr("%s cannot have a grab point.  Delete this", describenodeproto(np));
						termsearch(sea);
						return(NOEXAMPLE);
					}
					ns->layer = gen_facetcenterprim;
				} else
				{
					var = getval((INTBIG)otherni, VNODEINST, VNODEPROTO,
						"EDTEC_layer");
					if (var == NOVARIABLE)
					{
						us_tecedpointout(otherni, np);
						ttyputerr("No layer information on this sample in %s",
							describenodeproto(np));
						if ((us_aid->aidstate&NODETAILS) == 0)
							ttyputmsg("Use 'change' option or delete it");
						termsearch(sea);
						return(NOEXAMPLE);
					}
					ns->layer = (NODEPROTO *)var->addr;
					if (ns->layer == NONODEPROTO) hcount++;
				}

				/* accumulate state if this is not a "grab point" mark */
				if (otherni->proto != gen_facetcenterprim)
				{
					us_tecedgetbbox(otherni, &lx, &hx, &ly, &hy);
					if (gotbbox == 0)
					{
						ne->lx = lx;   ne->hx = hx;
						ne->ly = ly;   ne->hy = hy;
						gotbbox = 1;
					} else
					{
						if (lx < ne->lx) ne->lx = lx;
						if (hx > ne->hx) ne->hx = hx;
						if (ly < ne->ly) ne->ly = ly;
						if (hy > ne->hy) ne->hy = hy;
					}
					sflx = ne->lx;   sfhx = ne->hx;
					sfly = ne->ly;   sfhy = ne->hy;
				}
				foundone++;
			}
		}
		if (hcount == 0)
		{
			us_tecedpointout(NONODEINST, np);
			ttyputerr("No highlight layer in %s example", describenodeproto(np));
			if ((us_aid->aidstate&NODETAILS) == 0)
				ttyputmsg("Use 'place-layer' option to create HIGHLIGHT");
			return(NOEXAMPLE);
		}
		if (hcount != 1)
		{
			us_tecedpointout(NONODEINST, np);
			ttyputerr("Too many highlight layers in %s example.  Delete some", describenodeproto(np));
			return(NOEXAMPLE);
		}
	}
	if (nelist == NOEXAMPLE)
	{
		us_tecedpointout(NONODEINST, np);
		ttyputerr("No examples found in %s", describenodeproto(np));
		if ((us_aid->aidstate&NODETAILS) == 0)
			ttyputmsg("Use 'place-layer' option to produce some geometry");
		return(nelist);
	}

	/*
	 * now search the list for the smallest, most upper-right example
	 * (the "main" example)
	 */
	sizex = (nelist->hx - nelist->lx) / el_curtech->deflambda;
	sizey = (nelist->hy - nelist->ly) / el_curtech->deflambda;
	locx = (nelist->lx + nelist->hx) / 2;
	locy = (nelist->ly + nelist->hy) / 2;
	bestne = nelist;
	for(ne = nelist; ne != NOEXAMPLE; ne = ne->nextexample)
	{
		newsize = (ne->hx-ne->lx) / el_curtech->deflambda;
		newsize *= (ne->hy-ne->ly) / el_curtech->deflambda;
		if (newsize > sizex*sizey) continue;
		if (newsize == sizex*sizey && (ne->lx+ne->hx)/2 >= locx && (ne->ly+ne->hy)/2 <= locy)
			continue;
		sizex = (ne->hx - ne->lx) / el_curtech->deflambda;
		sizey = (ne->hy - ne->ly) / el_curtech->deflambda;
		locx = (ne->lx + ne->hx) / 2;
		locy = (ne->ly + ne->hy) / 2;
		bestne = ne;
	}

	/* place the main example at the top of the list */
	if (bestne != nelist)
	{
		for(ne = nelist; ne != NOEXAMPLE; ne = ne->nextexample)
			if (ne->nextexample == bestne)
		{
			ne->nextexample = bestne->nextexample;
			break;
		}
		bestne->nextexample = nelist;
		nelist = bestne;
	}

	/* done */
	return(nelist);
}

/*
 * Routine to associate the samples of example "nelist" in facet "np"
 * Returns nonzero if there is an error
 */
INTSML us_tecedassociateexamples(EXAMPLE *nelist, NODEPROTO *np)
{
	REGISTER EXAMPLE *ne;
	REGISTER SAMPLE *ns, *nslist, *nsfound, **listsort, **thissort;
	REGISTER INTSML total, i, found;
	REGISTER char *name, *othername;
	REGISTER VARIABLE *var;

	/* if there is only one example, no association */
	if (nelist->nextexample == NOEXAMPLE) return(0);

	/* associate each example "ne" with the original in "nelist" */
	for(ne = nelist->nextexample; ne != NOEXAMPLE; ne = ne->nextexample)
	{
		/* clear associations for every sample "ns" in the example "ne" */
		for(ns = ne->firstsample; ns != NOSAMPLE; ns = ns->nextsample)
			ns->assoc = NOSAMPLE;

		/* associate every sample "ns" in the example "ne" */
		for(ns = ne->firstsample; ns != NOSAMPLE; ns = ns->nextsample)
		{
			if (ns->assoc != NOSAMPLE) continue;

			/* cannot have center in other examples */
			if (ns->layer == gen_facetcenterprim)
			{
				us_tecedpointout(ns->node, ns->node->parent);
				ttyputerr("Grab point should only be in main example of %s", describenodeproto(np));
				return(1);
			}

			/* count number of similar layers in original example "nelist" */
			for(total = 0, nslist = nelist->firstsample; nslist != NOSAMPLE;
				nslist = nslist->nextsample)
			{
				if (nslist->layer != ns->layer) continue;
				total++;
				nsfound = nslist;
			}

			/* no similar layer found in the original: error */
			if (total == 0)
			{
				us_tecedpointout(ns->node, ns->node->parent);
				ttyputerr("Layer %s not found in main example of %s",
					us_tecedsamplename(ns->layer), describenodeproto(np));
				return(1);
			}

			/* just one in the original: simple association */
			if (total == 1)
			{
				ns->assoc = nsfound;
				continue;
			}

			/* if it is a port, associate by port name */
			if (ns->layer == gen_portprim)
			{
				var = getval((INTBIG)ns->node, VNODEINST, VSTRING, "EDTEC_portname");
				if (var == NOVARIABLE)
				{
					us_tecedpointout(ns->node, ns->node->parent);
					ttyputerr("Port does not have a name name in %s", describenodeproto(np));
					return(1);
				}
				name = (char *)var->addr;

				/* search the original for that port */
				for(nslist = nelist->firstsample; nslist != NOSAMPLE; nslist = nslist->nextsample)
					if (nslist->layer == gen_portprim)
				{
					var = getval((INTBIG)nslist->node, VNODEINST, VSTRING, "EDTEC_portname");
					if (var == NOVARIABLE)
					{
						us_tecedpointout(nslist->node, nslist->node->parent);
						ttyputerr("Port does not have a name in %s", describenodeproto(np));
						return(1);
					}
					othername = (char *)var->addr;
					if (namesame(name, othername) != 0) continue;
					ns->assoc = nslist;
					break;
				}
				if (nslist == NOSAMPLE)
				{
					us_tecedpointout(NONODEINST, np);
					ttyputerr("Could not find port %s in all examples of %s",
						name, describenodeproto(np));
					return(1);
				}
				continue;
			}

			/* count the number of this layer in example "ne" */
			for(i = 0, nslist = ne->firstsample; nslist != NOSAMPLE;
				nslist = nslist->nextsample)
					if (nslist->layer == ns->layer) i++;

			/* if number of similar layers differs: error */
			if (total != i)
			{
				us_tecedpointout(ns->node, ns->node->parent);
				ttyputerr("Layer %s found %d times in main example, %d in other",
					us_tecedsamplename(ns->layer), total, i);
				ttyputmsg("Make the counts consistent");
				return(1);
			}

			/* make a list of samples on this layer in original */
			listsort = (SAMPLE **)emalloc((total * (sizeof (SAMPLE *))), el_tempcluster);
			if (listsort == 0) return(1);
			for(i = 0, nslist = nelist->firstsample; nslist != NOSAMPLE;
				nslist = nslist->nextsample)
					if (nslist->layer == ns->layer) listsort[i++] = nslist;

			/* make a list of samples on this layer in example "ne" */
			thissort = (SAMPLE **)emalloc((total * (sizeof (SAMPLE *))), el_tempcluster);
			if (thissort == 0) return(1);
			for(i = 0, nslist = ne->firstsample; nslist != NOSAMPLE; nslist = nslist->nextsample)
				if (nslist->layer == ns->layer) thissort[i++] = nslist;

			/* sort each list in X/Y/shape */
			found = 1;
			while (found != 0)
			{
				found = 0;
				for(i=1; i<total; i++)
					if (listsort[i]->xpos < listsort[i-1]->xpos ||
						(listsort[i]->xpos == listsort[i-1]->xpos &&
							listsort[i]->ypos < listsort[i-1]->ypos) ||
						(listsort[i]->xpos == listsort[i-1]->xpos &&
							listsort[i]->ypos == listsort[i-1]->ypos &&
								listsort[i]->node->proto < listsort[i-1]->node->proto))
				{
					nsfound = listsort[i];
					listsort[i] = listsort[i-1];
					listsort[i-1] = nsfound;
					found = 1;
				}
			}
			found = 1;
			while (found != 0)
			{
				found = 0;
				for(i=1; i<total; i++)
					if (thissort[i]->xpos < thissort[i-1]->xpos ||
						(thissort[i]->xpos == thissort[i-1]->xpos &&
							thissort[i]->ypos < thissort[i-1]->ypos) ||
						(thissort[i]->xpos == thissort[i-1]->xpos &&
							thissort[i]->ypos == thissort[i-1]->ypos &&
								thissort[i]->node->proto < thissort[i-1]->node->proto))
				{
					nsfound = thissort[i];
					thissort[i] = thissort[i-1];
					thissort[i-1] = nsfound;
					found = 1;
				}
			}

			/* see if the lists have duplication */
			for(i=1; i<total; i++)
				if ((thissort[i]->xpos == thissort[i-1]->xpos &&
					thissort[i]->ypos == thissort[i-1]->ypos &&
						thissort[i]->node->proto == thissort[i-1]->node->proto) ||
					(listsort[i]->xpos == listsort[i-1]->xpos &&
						listsort[i]->ypos == listsort[i-1]->ypos &&
							listsort[i]->node->proto == listsort[i-1]->node->proto)) break;
			if (i >= total)
			{
				/* association can be made in X */
				for(i=0; i<total; i++) thissort[i]->assoc = listsort[i];
				efree((char *)thissort);
				efree((char *)listsort);
				continue;
			}

			/* don't know how to associate this sample */
			us_tecedpointout(thissort[i]->node, thissort[i]->node->parent);
			ttyputerr("Sample %s is unassociated in %s",
				us_tecedsamplename(thissort[i]->layer), describenodeproto(np));
			efree((char *)thissort);
			efree((char *)listsort);
			return(1);
		}

		/* final check: make sure every sample in original example associates */
		for(nslist = nelist->firstsample; nslist != NOSAMPLE;
			nslist = nslist->nextsample) nslist->assoc = NOSAMPLE;
		for(ns = ne->firstsample; ns != NOSAMPLE; ns = ns->nextsample)
			ns->assoc->assoc = ns;
		for(nslist = nelist->firstsample; nslist != NOSAMPLE;
			nslist = nslist->nextsample) if (nslist->assoc == NOSAMPLE)
		{
			if (nslist->layer == gen_facetcenterprim) continue;
			us_tecedpointout(nslist->node, nslist->node->parent);
			ttyputerr("Layer %s found in main example, but not others in %s",
				us_tecedsamplename(nslist->layer), describenodeproto(np));
			return(1);
		}
	}
	return(0);
}

/* flags about the edge positions in the examples */
#define	TOEDGELEFT       01		/* constant to left edge */
#define	TOEDGERIGHT      02		/* constant to right edge */
#define	TOEDGETOP        04		/* constant to top edge */
#define	TOEDGEBOT       010		/* constant to bottom edge */
#define	FROMCENTX       020		/* constant in X to center */
#define	FROMCENTY       040		/* constant in Y to center */
#define	RATIOCENTX     0100		/* fixed ratio from X center to edge */
#define	RATIOCENTY     0200		/* fixed ratio from Y center to edge */

INTSML us_tecedmakeprim(EXAMPLE *nelist, NODEPROTO *np, TECHNOLOGY *tech)
{
	REGISTER SAMPLE *ns, *nso;
	REGISTER EXAMPLE *ne;
	REGISTER INTSML total, count, newcount, i, truecount;
	REGISTER INTBIG *newrule, r, dist;
	XARRAY trans;
	static INTBIG *px, *py, *factor, *leftdist, *rightdist, *botdist,
		*topdist, *centxdist, *centydist, *ratiox, *ratioy, *cx, *cy;
	static INTSML arrlen=0;
	REGISTER char *str;
	REGISTER VARIABLE *var, *var2, *var3;
	REGISTER NODEINST *ni;

	/* look at every sample "ns" in the main example "nelist" */
	for(ns = nelist->firstsample; ns != NOSAMPLE; ns = ns->nextsample)
	{
		/* ignore grab point specification */
		if (ns->layer == gen_facetcenterprim) continue;

		/* if there is only one example: make sample scale with edge */
		if (nelist->nextexample == NOEXAMPLE)
		{
			/* see if there is polygonal information */
			if (ns->node->proto == art_filledpolygonprim ||
				ns->node->proto == art_closedpolygonprim ||
				ns->node->proto == art_openedpolygonprim ||
				ns->node->proto == art_openeddottedpolygonprim ||
				ns->node->proto == art_openeddashedpolygonprim ||
				ns->node->proto == art_openedfardottepolygonprim)
			{
				var = getvalkey((INTBIG)ns->node, VNODEINST, VINTEGER|VISARRAY, el_trace);
			} else var = NOVARIABLE;
			if (var != NOVARIABLE)
			{
				/* make sure the arrays hold "count" points */
				count = getlength(var) / 2;
				(void)us_tecedforcearrays(&px, &py, &cx, &cy, &factor, &leftdist, &rightdist,
					&botdist, &topdist, &centxdist, &centydist, &ratiox, &ratioy, &arrlen, count);

				/* fill the array */
				makerot(ns->node, trans);
				for(i=0; i<count; i++)
				{
					xform((ns->node->geom->lowx + ns->node->geom->highx)/2 +
						((INTBIG *)var->addr)[i*2],
							(ns->node->geom->lowy + ns->node->geom->highy)/2 +
								((INTBIG *)var->addr)[i*2+1], &px[i], &py[i], trans);
					factor[i] = FROMCENTX|FROMCENTY;
				}
			} else
			{
				/* see if it is an arc of a circle */
				count = 2;
				if (ns->node->proto == art_circleprim)
				{
					var = getvalkey((INTBIG)ns->node, VNODEINST, VINTEGER, art_degreeskey);
					if (var != NOVARIABLE) count = 3;
				} else var = NOVARIABLE;

				/* make sure the arrays hold enough points */
				(void)us_tecedforcearrays(&px, &py, &cx, &cy, &factor, &leftdist, &rightdist,
					&botdist, &topdist, &centxdist, &centydist, &ratiox, &ratioy, &arrlen, count);

				/* set sample description */
				if (var != NOVARIABLE)
				{
					/* handle circular arc sample */
					px[0] = (ns->node->geom->lowx + ns->node->geom->highx) / 2;
					py[0] = (ns->node->geom->lowy + ns->node->geom->highy) / 2;
					makerot(ns->node, trans);
					dist = ns->node->geom->highx - px[0];
					xform(px[0] + mult(dist, cosine((INTSML)(var->addr))),
						py[0] + mult(dist, sine((INTSML)(var->addr))), &px[1], &py[1], trans);
					xform(ns->node->geom->highx,
						(ns->node->geom->lowy + ns->node->geom->highy) / 2, &px[2], &py[2], trans);
					factor[0] = FROMCENTX|FROMCENTY;
					factor[1] = RATIOCENTX|RATIOCENTY;
					factor[2] = RATIOCENTX|RATIOCENTY;
				} else if (ns->node->proto == art_circleprim ||
					ns->node->proto == art_filledcircleprim)
				{
					/* handle circular sample */
					px[0] = (ns->node->geom->lowx + ns->node->geom->highx) / 2;
					py[0] = (ns->node->geom->lowy + ns->node->geom->highy) / 2;
					px[1] = ns->node->geom->highx;
					py[1] = (ns->node->geom->lowy + ns->node->geom->highy) / 2;
					factor[0] = FROMCENTX|FROMCENTY;
					factor[1] = TOEDGERIGHT|FROMCENTY;
				} else
				{
					/* rectangular sample: get the bounding box in (px, py) */
					us_tecedgetbbox(ns->node, &px[0], &px[1], &py[0], &py[1]);

					/* preset stretch factors to go to the edges of the box */
					factor[0] = TOEDGELEFT|TOEDGEBOT;
					factor[1] = TOEDGERIGHT|TOEDGETOP;
				}
			}

			/* add the rule to the collection */
			newrule = us_tecedstretchpoints(px, py, count, factor, ns, np,
				nelist);
			if (newrule == 0) return(1);
			var = getvalkey((INTBIG)ns->node, VNODEINST, VSTRING|VISARRAY, art_messagekey);
			if (var == NOVARIABLE) str = (char *)0; else
				str = ((char **)var->addr)[0];
			ns->rule = us_tecedaddrule(newrule, (INTSML)(count*4), 0, str);
			if (ns->rule == NORULE) return(1);
			efree((char *)newrule);
			continue;
		}

		/* look at other examples and find samples associated with this */
		nelist->studysample = ns;
		for(ne = nelist->nextexample; ne != NOEXAMPLE; ne = ne->nextexample)
		{
			/* count number of samples associated with the main sample */
			total = 0;
			for(nso = ne->firstsample; nso != NOSAMPLE; nso = nso->nextsample)
				if (nso->assoc == ns)
			{
				ne->studysample = nso;
				total++;
			}
			if (total == 0)
			{
				us_tecedpointout(ns->node, ns->node->parent);
				ttyputerr("Still unassociated sample in %s (shouldn't happen)",
					describenodeproto(np));
				return(1);
			}

			/* if there are multiple associations, it must be a contact cut */
			if (total > 1)
			{
				/* make sure the layer is real geometry, not highlight or a port */
				if (ns->layer == NONODEPROTO || ns->layer == gen_portprim)
				{
					us_tecedpointout(ns->node, ns->node->parent);
					ttyputerr("Only contact layers may be iterated in examples of %s",
						describenodeproto(np));
					return(1);
				}

				/* make sure the contact cut layer is opaque */
				for(i=0; i<tech->layercount; i++)
					if (namesame(&ns->layer->cell->cellname[6], us_teclayer_names[i]) == 0)
				{
					if (tech->layers[i]->bits != LAYERO)
					{
						us_tecedpointout(ns->node, ns->node->parent);
						ttyputerr("Multiple contact layers must be nonoverlappable in %s",
							describenodeproto(np));
						return(1);
					}
					break;
				}

				/* add the rule */
				if (us_tecedmulticut(ns, nelist, np) != 0) return(1);
				break;
			}
		}
		if (ne != NOEXAMPLE) continue;

		/* associations done for this sample, now analyze them */
		if (ns->node->proto == art_filledpolygonprim ||
			ns->node->proto == art_closedpolygonprim ||
			ns->node->proto == art_openedpolygonprim ||
			ns->node->proto == art_openeddottedpolygonprim ||
			ns->node->proto == art_openeddashedpolygonprim ||
			ns->node->proto == art_openedfardottepolygonprim)
		{
			var = getvalkey((INTBIG)ns->node, VNODEINST, VINTEGER|VISARRAY, el_trace);
		} else var = NOVARIABLE;
		if (var != NOVARIABLE)
		{
			truecount = count = getlength(var) / 2;

			/* make sure the arrays hold "count" points */
			(void)us_tecedforcearrays(&px, &py, &cx, &cy, &factor, &leftdist, &rightdist, &botdist,
				&topdist, &centxdist, &centydist, &ratiox, &ratioy, &arrlen, count);
			makerot(ns->node, trans);
			for(i=0; i<count; i++)
				xform((ns->node->geom->lowx + ns->node->geom->highx)/2 + ((INTBIG *)var->addr)[i*2],
					(ns->node->geom->lowy + ns->node->geom->highy)/2 +
						((INTBIG *)var->addr)[i*2+1], &px[i], &py[i], trans);
		} else
		{
			/* make sure the arrays hold enough points */
			count = 2;
			if (ns->node->proto == art_circleprim)
			{
				var3 = getvalkey((INTBIG)ns->node, VNODEINST, VINTEGER, art_degreeskey);
				if (var3 != NOVARIABLE) count = 3;
			} else var3 = NOVARIABLE;
			truecount = count;
			if (var3 == NOVARIABLE)
			{
				var2 = getval((INTBIG)ns->node, VNODEINST, VSTRING, "EDTEC_minbox");
				if (var2 != NOVARIABLE) count *= 2;
			}
			(void)us_tecedforcearrays(&px, &py, &cx, &cy, &factor, &leftdist, &rightdist, &botdist,
				&topdist, &centxdist, &centydist, &ratiox, &ratioy, &arrlen, count);

			/* set sample description */
			if (var3 != NOVARIABLE)
			{
				/* handle circular arc sample */
				px[0] = (ns->node->geom->lowx + ns->node->geom->highx) / 2;
				py[0] = (ns->node->geom->lowy + ns->node->geom->highy) / 2;
				makerot(ns->node, trans);
				dist = ns->node->geom->highx - px[0];
				xform(px[0] + mult(dist, cosine((INTSML)(var3->addr))),
					py[0] + mult(dist, sine((INTSML)(var3->addr))), &px[1], &py[1], trans);
				xform(ns->node->geom->highx,
					(ns->node->geom->lowy + ns->node->geom->highy) / 2, &px[2], &py[2], trans);
			} else if (ns->node->proto == art_circleprim || ns->node->proto == art_filledcircleprim)
			{
				/* handle circular sample */
				px[0] = (ns->node->geom->lowx + ns->node->geom->highx) / 2;
				py[0] = (ns->node->geom->lowy + ns->node->geom->highy) / 2;
				px[1] = ns->node->geom->highx;
				py[1] = (ns->node->geom->lowy + ns->node->geom->highy) / 2;
			} else
			{
				/* rectangular sample: get the bounding box in (px, py) */
				us_tecedgetbbox(ns->node, &px[0], &px[1], &py[0], &py[1]);
			}
			if (var2 != NOVARIABLE)
			{
				px[2] = px[0];   py[2] = py[0];
				px[3] = px[1];   py[3] = py[1];
			}
		}

		for(i=0; i<count; i++)
		{
			leftdist[i] = px[i] - nelist->lx;
			rightdist[i] = nelist->hx - px[i];
			botdist[i] = py[i] - nelist->ly;
			topdist[i] = nelist->hy - py[i];
			centxdist[i] = px[i] - (nelist->lx+nelist->hx)/2;
			centydist[i] = py[i] - (nelist->ly+nelist->hy)/2;
			if (nelist->hx == nelist->lx) ratiox[i] = 0; else
				ratiox[i] = (px[i] - (nelist->lx+nelist->hx)/2) * WHOLE / (nelist->hx-nelist->lx);
			if (nelist->hy == nelist->ly) ratioy[i] = 0; else
				ratioy[i] = (py[i] - (nelist->ly+nelist->hy)/2) * WHOLE / (nelist->hy-nelist->ly);
			if (i < truecount)
				factor[i] = TOEDGELEFT | TOEDGERIGHT | TOEDGETOP | TOEDGEBOT | FROMCENTX |
					FROMCENTY | RATIOCENTX | RATIOCENTY; else
						factor[i] = FROMCENTX | FROMCENTY;
		}
		for(ne = nelist->nextexample; ne != NOEXAMPLE; ne = ne->nextexample)
		{
			ni = ne->studysample->node;
			if (ni->proto == art_filledpolygonprim ||
				ni->proto == art_closedpolygonprim ||
				ni->proto == art_openedpolygonprim ||
				ni->proto == art_openeddottedpolygonprim ||
				ni->proto == art_openeddashedpolygonprim ||
				ni->proto == art_openedfardottepolygonprim)
			{
				var = getvalkey((INTBIG)ni, VNODEINST, VINTEGER|VISARRAY, el_trace);
			} else var = NOVARIABLE;
			if (var != NOVARIABLE)
			{
				newcount = getlength(var) / 2;
				makerot(ni, trans);
				for(i=0; i<mini(truecount, newcount); i++)
					xform((ni->geom->lowx + ni->geom->highx)/2 + ((INTBIG *)var->addr)[i*2],
						(ni->geom->lowy + ni->geom->highy)/2 +
							((INTBIG *)var->addr)[i*2+1], &cx[i], &cy[i], trans);
			} else
			{
				newcount = 2;
				if (ni->proto == art_circleprim)
				{
					var3 = getvalkey((INTBIG)ni, VNODEINST, VINTEGER, art_degreeskey);
					if (var3 != NOVARIABLE) newcount = 3;
				} else var3 = NOVARIABLE;
				if (var3 != NOVARIABLE)
				{
					cx[0] = (ni->geom->lowx + ni->geom->highx) / 2;
					cy[0] = (ni->geom->lowy + ni->geom->highy) / 2;
					makerot(ni, trans);
					dist = ni->geom->highx - cx[0];
					xform(cx[0] + mult(dist, cosine((INTSML)(var3->addr))),
						cy[0] + mult(dist, sine((INTSML)(var3->addr))), &cx[1], &cy[1], trans);
					xform(ni->geom->highx, (ni->geom->lowy + ni->geom->highy) / 2,
						&cx[2], &cy[2], trans);
				} else if (ni->proto == art_circleprim || ni->proto == art_filledcircleprim)
				{
					cx[0] = (ni->geom->lowx + ni->geom->highx) / 2;
					cy[0] = (ni->geom->lowy + ni->geom->highy) / 2;
					cx[1] = ni->geom->highx;
					cy[1] = (ni->geom->lowy + ni->geom->highy) / 2;
				} else
				{
					us_tecedgetbbox(ni, &cx[0],&cx[1], &cy[0], &cy[1]);
				}
			}
			if (newcount != truecount)
			{
				us_tecedpointout(ni, ni->parent);
				ttyputerr("Main example of %s has %d points but this has %d in %s",
					us_tecedsamplename(ne->studysample->layer),
						truecount, newcount, describenodeproto(np));
				return(1);
			}

			for(i=0; i<truecount; i++)
			{
				/* see if edges are fixed distance from example edge */
				if (leftdist[i] != cx[i] - ne->lx) factor[i] &= ~TOEDGELEFT;
				if (rightdist[i] != ne->hx - cx[i]) factor[i] &= ~TOEDGERIGHT;
				if (botdist[i] != cy[i] - ne->ly) factor[i] &= ~TOEDGEBOT;
				if (topdist[i] != ne->hy - cy[i]) factor[i] &= ~TOEDGETOP;

				/* see if edges are fixed distance from example center */
				if (centxdist[i] != cx[i] - (ne->lx+ne->hx)/2) factor[i] &= ~FROMCENTX;
				if (centydist[i] != cy[i] - (ne->ly+ne->hy)/2) factor[i] &= ~FROMCENTY;

				/* see if edges are fixed ratio from example center */
				if (ne->hx == ne->lx) r = 0; else
					r = (cx[i] - (ne->lx+ne->hx)/2) * WHOLE / (ne->hx-ne->lx);
				if (r != ratiox[i]) factor[i] &= ~RATIOCENTX;
				if (ne->hy == ne->ly) r = 0; else
					r = (cy[i] - (ne->ly+ne->hy)/2) * WHOLE / (ne->hy-ne->ly);
				if (r != ratioy[i]) factor[i] &= ~RATIOCENTY;
			}

			/* make sure port information is on the primary example */
			if (ns->layer != gen_portprim) continue;

			/* check port angle */
			var = getval((INTBIG)ns->node, VNODEINST, VINTEGER, "EDTEC_portangle");
			var2 = getval((INTBIG)ni, VNODEINST, VINTEGER, "EDTEC_portangle");
			if (var == NOVARIABLE && var2 != NOVARIABLE)
			{
				us_tecedpointout(NONODEINST, np);
				ttyputerr("Warning: moving port angle to main example of %s",
					describenodeproto(np));
				(void)setval((INTBIG)ns->node, VNODEINST, "EDTEC_portangle", var2->addr, VINTEGER);
			}

			/* check port range */
			var = getval((INTBIG)ns->node, VNODEINST, VINTEGER, "EDTEC_portrange");
			var2 = getval((INTBIG)ni, VNODEINST, VINTEGER, "EDTEC_portrange");
			if (var == NOVARIABLE && var2 != NOVARIABLE)
			{
				us_tecedpointout(NONODEINST, np);
				ttyputerr("Warning: moving port range to main example of %s", describenodeproto(np));
				(void)setval((INTBIG)ns->node, VNODEINST, "EDTEC_portrange", var2->addr, VINTEGER);
			}

			/* check connectivity */
			var = getval((INTBIG)ns->node, VNODEINST, VNODEPROTO|VISARRAY, "EDTEC_connects");
			var2 = getval((INTBIG)ni, VNODEINST, VNODEPROTO|VISARRAY, "EDTEC_connects");
			if (var == NOVARIABLE && var2 != NOVARIABLE)
			{
				us_tecedpointout(NONODEINST, np);
				ttyputerr("Warning: moving port connections to main example of %s",
					describenodeproto(np));
				(void)setval((INTBIG)ns->node, VNODEINST, "EDTEC_connects",
					var2->addr, VNODEPROTO|VISARRAY|(getlength(var2)<<VLENGTHSH));
			}
		}

		/* error check for the highlight layer */
		if (ns->layer == NONODEPROTO)
			for(i=0; i<truecount; i++)
				if ((factor[i]&(TOEDGELEFT|TOEDGERIGHT)) == 0 ||
					(factor[i]&(TOEDGETOP|TOEDGEBOT)) == 0)
		{
			us_tecedpointout(ns->node, ns->node->parent);
			ttyputerr("Highlight must be constant distance from edge in %s", describenodeproto(np));
			return(1);
		}

		/* finally, make a rule for this sample */
		newrule = us_tecedstretchpoints(px, py, count, factor, ns, np, nelist);
		if (newrule == 0) return(1);

		/* add the rule to the global list */
		var = getvalkey((INTBIG)ns->node, VNODEINST, VSTRING|VISARRAY, art_messagekey);
		if (var == NOVARIABLE) str = (char *)0; else
			str = ((char **)var->addr)[0];
		ns->rule = us_tecedaddrule(newrule, (INTSML)(count*4), 0, str);
		if (ns->rule == NORULE) return(1);
		efree((char *)newrule);
	}
	return(0);
}

/*
 * routine to ensure that the 13 arrays in the first 13 parameters are all
 * at least "want" long.  Their current size is "arrlen".  Returns nonzero
 * on error
 */
INTSML us_tecedforcearrays(INTBIG **px, INTBIG **py, INTBIG **cx, INTBIG **cy, INTBIG **factor,
	INTBIG **leftdist, INTBIG **rightdist, INTBIG **botdist, INTBIG **topdist,
	INTBIG **centxdist, INTBIG **centydist, INTBIG **ratiox, INTBIG **ratioy, INTSML *arrlen, INTSML want)
{
	if (*arrlen >= want) return(0);
	if (*arrlen != 0)
	{
		efree((char *)*px);
		efree((char *)*py);
		efree((char *)*cx);
		efree((char *)*cy);
		efree((char *)*factor);
		efree((char *)*leftdist);
		efree((char *)*rightdist);
		efree((char *)*botdist);
		efree((char *)*topdist);
		efree((char *)*centxdist);
		efree((char *)*centydist);
		efree((char *)*ratiox);
		efree((char *)*ratioy);
	}
	*arrlen = want;
	*px = emalloc((want * SIZEOFINTBIG), us_aid->cluster);
	if (*px == 0) return(1);
	*py = emalloc((want * SIZEOFINTBIG), us_aid->cluster);
	if (*py == 0) return(1);
	*cx = emalloc((want * SIZEOFINTBIG), us_aid->cluster);
	if (*cx == 0) return(1);
	*cy = emalloc((want * SIZEOFINTBIG), us_aid->cluster);
	if (*cy == 0) return(1);
	*factor = emalloc((want * SIZEOFINTBIG), us_aid->cluster);
	if (*factor == 0) return(1);
	*leftdist = emalloc((want * SIZEOFINTBIG), us_aid->cluster);
	if (*leftdist == 0) return(1);
	*rightdist = emalloc((want * SIZEOFINTBIG), us_aid->cluster);
	if (*rightdist == 0) return(1);
	*botdist = emalloc((want * SIZEOFINTBIG), us_aid->cluster);
	if (*botdist == 0) return(1);
	*topdist = emalloc((want * SIZEOFINTBIG), us_aid->cluster);
	if (*topdist == 0) return(1);
	*centxdist = emalloc((want * SIZEOFINTBIG), us_aid->cluster);
	if (*centxdist == 0) return(1);
	*centydist = emalloc((want * SIZEOFINTBIG), us_aid->cluster);
	if (*centydist == 0) return(1);
	*ratiox = emalloc((want * SIZEOFINTBIG), us_aid->cluster);
	if (*ratiox == 0) return(1);
	*ratioy = emalloc((want * SIZEOFINTBIG), us_aid->cluster);
	if (*ratioy == 0) return(1);
	return(0);
}

/*
 * routine to adjust the "count"-long array of points in "px" and "py" according
 * to the stretch factor bits in "factor" and return an array that describes
 * these points.  Returns zero on error.
 */
INTBIG *us_tecedstretchpoints(INTBIG *px, INTBIG *py, INTSML count, INTBIG *factor,
	SAMPLE *ns, NODEPROTO *np, EXAMPLE *nelist)
{
	REGISTER INTBIG *newrule;
	REGISTER INTSML i;

	newrule = emalloc((count*4*SIZEOFINTBIG), el_tempcluster);
	if (newrule == 0) return(0);

	for(i=0; i<count; i++)
	{
		/* determine the X algorithm */
		if ((factor[i]&TOEDGELEFT) != 0)
		{
			/* left edge rule */
			newrule[i*4] = -H0;
			newrule[i*4+1] = (px[i]-nelist->lx) * WHOLE/el_curtech->deflambda;
		} else if ((factor[i]&TOEDGERIGHT) != 0)
		{
			/* right edge rule */
			newrule[i*4] = H0;
			newrule[i*4+1] = (px[i]-nelist->hx) * WHOLE/el_curtech->deflambda;
		} else if ((factor[i]&FROMCENTX) != 0)
		{
			/* center rule */
			newrule[i*4] = 0;
			newrule[i*4+1] = (px[i]-(nelist->lx+nelist->hx)/2) * WHOLE/el_curtech->deflambda;
		} else if ((factor[i]&RATIOCENTX) != 0)
		{
			/* constant stretch rule */
			if (nelist->hx == nelist->lx) newrule[i*4] = 0; else
				newrule[i*4] = (px[i] - (nelist->lx+nelist->hx)/2) * WHOLE / (nelist->hx-nelist->lx);
			newrule[i*4+1] = 0;
		} else
		{
			us_tecedpointout(ns->node, ns->node->parent);
			ttyputerr("Cannot determine X stretching rule for layer %s in %s",
				us_tecedsamplename(ns->layer), describenodeproto(np));
			return(0);
		}

		/* determine the Y algorithm */
		if ((factor[i]&TOEDGEBOT) != 0)
		{
			/* bottom edge rule */
			newrule[i*4+2] = -H0;
			newrule[i*4+3] = (py[i]-nelist->ly) * WHOLE/el_curtech->deflambda;
		} else if ((factor[i]&TOEDGETOP) != 0)
		{
			/* top edge rule */
			newrule[i*4+2] = H0;
			newrule[i*4+3] = (py[i]-nelist->hy) * WHOLE/el_curtech->deflambda;
		} else if ((factor[i]&FROMCENTY) != 0)
		{
			/* center rule */
			newrule[i*4+2] = 0;
			newrule[i*4+3] = (py[i]-(nelist->ly+nelist->hy)/2) * WHOLE/el_curtech->deflambda;
		} else if ((factor[i]&RATIOCENTY) != 0)
		{
			/* constant stretch rule */
			if (nelist->hy == nelist->ly) newrule[i*4+2] = 0; else
				newrule[i*4+2] = (py[i] - (nelist->ly+nelist->hy)/2) * WHOLE /
					(nelist->hy-nelist->ly);
			newrule[i*4+3] = 0;
		} else
		{
			us_tecedpointout(ns->node, ns->node->parent);
			ttyputerr("Cannot determine Y stretching rule for layer %s in %s",
				us_tecedsamplename(ns->layer), describenodeproto(np));
			return(0);
		}
	}
	return(newrule);
}

/*
 * routine to build a rule for multiple contact-cut sample "ns" from the
 * overall example list in "nelist".  Returns nonzero on error.
 */
INTSML us_tecedmulticut(SAMPLE *ns, EXAMPLE *nelist, NODEPROTO *np)
{
	INTBIG rule[8];
	REGISTER INTSML total, i;
	REGISTER INTBIG xsep, ysep, sepx, sepy;
	REGISTER SAMPLE **nslist, *nso, *hs;
	REGISTER EXAMPLE *ne;

	rule[0] = -H0;   rule[1] = K1;
	rule[2] = -H0;   rule[3] = K1;
	rule[4] = -H0;   rule[5] = K3;
	rule[6] = -H0;   rule[7] = K3;
	ns->rule = us_tecedaddrule(rule, 8, 1, (char *)0);
	if (ns->rule == NORULE) return(1);

	/* find the highlight layer */
	for(hs = nelist->firstsample; hs != NOSAMPLE; hs = hs->nextsample)
		if (hs->layer == NONODEPROTO) break;
	if (hs == NOSAMPLE)
	{
		us_tecedpointout(NONODEINST, np);
		ttyputerr("No highlight layer on contact %s", describenodeproto(np));
		if ((us_aid->aidstate&NODETAILS) == 0)
			ttyputmsg("Use 'place-layer' option to create HIGHLIGHT");
		return(1);
	}

	/* determine size of each cut */
	ns->rule->multixs = ns->node->highx - ns->node->lowx;
	ns->rule->multiys = ns->node->highy - ns->node->lowy;

	/* determine indentation of cuts */
	ns->rule->multiindent = ns->node->lowx - hs->node->lowx;
	if (hs->node->highx - ns->node->highx != ns->rule->multiindent ||
		ns->node->lowy - hs->node->lowy != ns->rule->multiindent ||
		hs->node->highy - ns->node->highy != ns->rule->multiindent)
	{
		us_tecedpointout(ns->node, ns->node->parent);
		ttyputerr("Multiple contact cuts must be indented uniformly in %s", describenodeproto(np));
		return(1);
	}

	/* look at every example after the first */
	xsep = ysep = -1;
	for(ne = nelist->nextexample; ne != NOEXAMPLE; ne = ne->nextexample)
	{
		/* count number of samples equivalent to the main sample */
		total = 0;
		for(nso = ne->firstsample; nso != NOSAMPLE; nso = nso->nextsample)
			if (nso->assoc == ns)
		{
			/* make sure size is proper */
			if (ns->rule->multixs != nso->node->highx - nso->node->lowx ||
				ns->rule->multiys != nso->node->highy - nso->node->lowy)
			{
				us_tecedpointout(nso->node, nso->node->parent);
				ttyputerr("Multiple contact cuts must not differ in size in %s",
					describenodeproto(np));
				return(1);
			}
			total++;
		}

		/* allocate space for these samples */
		nslist = (SAMPLE **)emalloc((total * (sizeof (SAMPLE *))), el_tempcluster);
		if (nslist == 0) return(1);

		/* fill the list of samples */
		i = 0;
		for(nso = ne->firstsample; nso != NOSAMPLE; nso = nso->nextsample)
			if (nso->assoc == ns) nslist[i++] = nso;

		/* analyze the samples for separation */
		for(i=1; i<total; i++)
		{
			/* find separation */
			sepx = abs((nslist[i-1]->node->highx + nslist[i-1]->node->lowx) / 2 -
				(nslist[i]->node->highx + nslist[i]->node->lowx) / 2);
			sepy = abs((nslist[i-1]->node->highy + nslist[i-1]->node->lowy) / 2 -
				(nslist[i]->node->highy + nslist[i]->node->lowy) / 2);

			/* check for validity */
			if (sepx < ns->rule->multixs && sepy < ns->rule->multiys)
			{
				us_tecedpointout(nslist[i]->node, nslist[i]->node->parent);
				ttyputerr("Multiple contact cuts must not overlap in %s", describenodeproto(np));
				efree((char *)nslist);
				return(1);
			}

			/* accumulate minimum separation */
			if (sepx >= ns->rule->multixs)
			{
				if (xsep < 0) xsep = sepx; else
				{
					if (xsep > sepx) xsep = sepx;
				}
			}
			if (sepy >= ns->rule->multiys)
			{
				if (ysep < 0) ysep = sepy; else
				{
					if (ysep > sepy) ysep = sepy;
				}
			}
		}

		/* finally ensure that all separations are multiples of "multisep" */
		for(i=1; i<total; i++)
		{
			/* find X separation */
			sepx = abs((nslist[i-1]->node->highx + nslist[i-1]->node->lowx) / 2 -
				(nslist[i]->node->highx + nslist[i]->node->lowx) / 2);
			sepy = abs((nslist[i-1]->node->highy + nslist[i-1]->node->lowy) / 2 -
				(nslist[i]->node->highy + nslist[i]->node->lowy) / 2);
			if (sepx / xsep * xsep != sepx)
			{
				us_tecedpointout(nslist[i]->node, nslist[i]->node->parent);
				ttyputerr("Multiple contact cut X spacing must be uniform in %s",
					describenodeproto(np));
				efree((char *)nslist);
				return(1);
			}

			/* find Y separation */
			if (sepy / ysep * ysep != sepy)
			{
				us_tecedpointout(nslist[i]->node, nslist[i]->node->parent);
				ttyputerr("Multiple contact cut Y spacing must be uniform in %s",
					describenodeproto(np));
				efree((char *)nslist);
				return(1);
			}
		}
		efree((char *)nslist);
	}
	ns->rule->multisep = xsep - ns->rule->multixs;
	if (ns->rule->multisep != ysep - ns->rule->multiys)
	{
		us_tecedpointout(NONODEINST, np);
		ttyputerr("Multiple contact cut X and Y spacing must be the same in %s",
			describenodeproto(np));
		return(1);
	}
	return(0);
}

/*
 * routine to add the "len"-long list of port connections in "conlist" to
 * the list of port connections, and return the port connection entry
 * for this one.  Returns NOPCON on error
 */
PCON *us_tecedaddportlist(INTSML len, INTBIG *conlist)
{
	REGISTER PCON *pc;
	REGISTER INTSML i, j;

	/* find a port connection that is the same */
	for(pc = us_tecedfirstpcon; pc != NOPCON; pc = pc->nextpcon)
	{
		if (pc->total != len) continue;
		for(j=0; j<pc->total; j++) pc->assoc[j] = 0;
		for(i=0; i<len; i++)
		{
			for(j=0; j<pc->total; j++) if (pc->connects[j+1] == conlist[i])
			{
				pc->assoc[j]++;
				break;
			}
		}
		for(j=0; j<pc->total; j++) if (pc->assoc[j] == 0) break;
		if (j >= pc->total) return(pc);
	}

	/* not found: add to list */
	pc = (PCON *)emalloc((sizeof (PCON)), us_aid->cluster);
	if (pc == 0) return(NOPCON);
	pc->total = len;
	pc->connects = emalloc(((len+5)*SIZEOFINTBIG), us_aid->cluster);
	if (pc->connects == 0) return(NOPCON);
	pc->assoc = emalloc((len*SIZEOFINTBIG), us_aid->cluster);
	if (pc->assoc == 0) return(NOPCON);
	for(j=0; j<len; j++) pc->connects[j+1] = conlist[j];
	pc->connects[0] = -1;
	pc->connects[len+1] = AUNIV;
	pc->connects[len+2] = AINVIS;
	pc->connects[len+3] = AUNROUTED;
	pc->connects[len+4] = -1;
	pc->nextpcon = us_tecedfirstpcon;
	us_tecedfirstpcon = pc;
	return(pc);
}

/*
 * routine to get the list of libraries that are used in the construction
 * of library "lib".  Returns the number of libraries, terminated with "lib",
 * and sets the list in "liblist".
 */
INTSML us_teceditgetdependents(LIBRARY *lib, LIBRARY ***liblist)
{
	REGISTER VARIABLE *var;
	static LIBRARY **myliblist;
	static INTSML liblistsize = 0;
	REGISTER INTSML i, j, total;
	REGISTER char *pt;

	/* get list of dependent libraries */
	var = getval((INTBIG)lib, VLIBRARY, VSTRING|VISARRAY, "EDTEC_dependent_libraries");
	if (var == NOVARIABLE) j = 0; else j = getlength(var);
	if (j >= liblistsize)
	{
		if (liblistsize != 0) efree((char *)myliblist);
		myliblist = (LIBRARY **)emalloc((j+1) * (sizeof (LIBRARY *)), us_aid->cluster);
		if (myliblist == 0) return(0);
		liblistsize = j+1;
	}

	total = 0;
	for(i=0; i<j; i++)
	{
		pt = ((char **)var->addr)[i];
		myliblist[total++] = getlibrary(pt);
		if (myliblist[total-1] == NOLIBRARY)
		{
			ttyputerr("Cannot find dependent technology library %s, ignoring", pt);
			total--;
			continue;
		}
		if (myliblist[total-1] == lib)
		{
			ttyputerr("Library cannot depend on itself, ignoring dependency");
			total--;
			continue;
		}
	}
	myliblist[total++] = lib;
	*liblist = myliblist;
	return(total);
}

/*
 * general-purpose routine to scan the "dependentlibcount" libraries in "dependentlibs",
 * looking for facets that begin with the string "match".  It then uses the
 * variable "seqname" on the last library to determine an ordering of the facets.
 * Then, it returns the facets in the array "sequence" and returns the number of them.
 * Returns 0 on error.
 */
INTSML us_teceditfindsequence(LIBRARY **dependentlibs, INTSML dependentlibcount,
	char *match, char *seqname, NODEPROTO ***sequence)
{
	REGISTER INTSML total, i, j, k, l, npsize, matchcount;
	REGISTER NODEPROTO *np, *lnp, **nplist, **newnplist;
	REGISTER VARIABLE *var;
	REGISTER LIBRARY *olderlib, *laterlib;

	/* look backwards through libraries for the appropriate facets */
	matchcount = strlen(match);
	total = npsize = 0;
	for(i=dependentlibcount-1; i>=0; i--)
	{
		olderlib = dependentlibs[i];
		for(np = olderlib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
			if (namesamen(np->cell->cellname, match, matchcount) == 0)
		{
			/* see if this facet is used in a later library */
			for(j=i+1; j<dependentlibcount; j++)
			{
				laterlib = dependentlibs[j];
				for(lnp = laterlib->firstnodeproto; lnp != NONODEPROTO; lnp = lnp->nextnodeproto)
					if (namesame(lnp->cell->cellname, np->cell->cellname) == 0)
				{
					/* got older and later version of same facet: check dates */
					if (lnp->revisiondate < np->revisiondate)
						ttyputmsg("Warning: library %s has newer %s than library %s",
							olderlib->libname, np->cell->cellname, laterlib->libname);
					break;
				}
				if (lnp != NONODEPROTO) break;
			}

			/* if no later library has this, add to total */
			if (j >= dependentlibcount)
			{
				if (total >= npsize)
				{
					newnplist = (NODEPROTO **)emalloc((npsize+10) * (sizeof (NODEPROTO *)),
						el_tempcluster);
					if (newnplist == 0) return(0);
					for(k=0; k<total; k++) newnplist[k] = nplist[k];
					if (npsize != 0) efree((char *)nplist);
					nplist = newnplist;
					npsize += 10;
				}
				newnplist[total++] = np;
			}
		}
	}
	if (total <= 0) return(0);

	/* if there is no sequence, simply return the list */
	var = getval((INTBIG)dependentlibs[dependentlibcount-1], VLIBRARY, VSTRING|VISARRAY, seqname);
	if (var == NOVARIABLE)
	{
		*sequence = nplist;
		return(total);
	}

	/* allocate a new list to be built from the sequence */
	*sequence = (NODEPROTO **)emalloc(total * (sizeof (NODEPROTO *)), el_tempcluster);
	if (*sequence == 0) return(0);

	j = getlength(var);
	k = 0;
	for(i=0; i<j; i++)
	{
		for(l = 0; l < total; l++)
		{
			np = nplist[l];
			if (np == NONODEPROTO) continue;
			if (namesame(&np->cell->cellname[matchcount], ((char **)var->addr)[i]) == 0) break;
		}
		if (l >= total) continue;
		(*sequence)[k++] = np;
		nplist[l] = NONODEPROTO;
	}
	for(l = 0; l < total; l++)
	{
		np = nplist[l];
		if (np != NONODEPROTO) (*sequence)[k++] = np;
	}
	efree((char *)nplist);
	return(total);
}
