/*
 * Electric(tm) VLSI Design System
 *
 * File: usrmisc.c
 * User interface aid: miscellaneous control
 * Written by: Steven M. Rubin, Electric Editor Incorporated
 *
 * Copyright (c) 1998 Electric Editor Incorporated.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Electric Editor Incorporated
 * 23470 Sunset Drive, Suite 108
 * Los Gatos, California 95033
 * support@electriceditor.com
 */

#include "global.h"
#include "egraphics.h"
#include "usr.h"
#include "usreditemacs.h"
#include "usreditpac.h"
#include "tecgen.h"
#include "tecart.h"
#include "sim.h"
#include <ctype.h>
#include <math.h>

#define ALLTEXTWIDTH	70		/* width of side menu when it is all text */

/******************** INITIALIZATION ********************/

/*
 * default menu state: the default menu is INITIALMENUX entries across
 * and INITIALMENUY entries down at the top of the screen.  The first
 * entries of this are taken from the table of commands and arguments
 * in "us_initialmenu".  The rest are filled in with the important arcs
 * of the default technology followed by the primitive nodes of that
 * technology.  Note that the number of important arcs is computed only
 * once when the program begins and is therefore correct only for the
 * initial technology.  If technologies are changed, there will be the
 * same number of important arcs in the menu regardless of how many
 * there are in the other technology.
 */
#define	INITIALMENUX    18	/* number of entries in X of menu */
#define	INITIALMENUY    2	/* number of entries in Y of menu */
struct
{
	char  *command;
	INTSML count;
	char  *args[3];
	char  *glyph;
} us_initialmenu[] =
{
	{"library", 1, {"save", "", ""},
		"generic:Save"},
	{"node",    3, {"not", "expand", "1"},
		"generic:Node-Unexpand"},
	{"node",    2, {"expand", "1", ""},
		"generic:Node-Expand"},
	{"window",  2, {"left", "0.5", ""},
		"generic:Window-Left"},
	{"window",  2, {"right", "0.5", ""},
		"generic:Window-Right"},
	{"window",  2, {"down", "0.5", ""},
		"generic:Window-Down"},
	{"window",  2, {"up", "0.5", ""},
		"generic:Window-Up"},
	{"undo",    0, {"", "", ""},
		"generic:Undo"},
	{"rotate",  0, {"", "", ""},
		"generic:Rotate"},
	{"erase",   0, {"", "", ""},
		"generic:Erase"},
	{"window",  1, {"in-zoom", "", ""},
		"generic:Window-In-Zoom"},
	{"window",  1, {"out-zoom", "", ""},
		"generic:Window-Out-Zoom"},
	{"window",  1, {"highlight-displayed", "", ""},
		"generic:Window-Highlight-Displayed"},
	{"window",  1, {"all-displayed", "", ""},
		"generic:Window-All-Displayed"},
	{"grid",    0, {"", "", ""},
		"generic:Grid"},
	{NULL, 0, {NULL, NULL, NULL}, NULL} /* 0 */
};

/*
 * default tablet state: the table below shows the set of commands
 * that will be bound to the tablet buttons.  If there is a button
 * with the name in "us_tablet[i].but[0]" or "us_tablet[i].but[1]"
 * then that button will be bound to the command "us_tablet[i].com".
 * If no tablet button exists for an entry "i", then the command
 * there will be bound to the key "us_tablet[i].key".  Thus, tablets
 * with varying numbers of buttons can be handled by placing commands
 * on the keys if they don't fit on the buttons.
 */
struct
{
	char  *but[3];		/* the button names for this command */
	char  *key;			/* the key name for this command */
	char  *com;			/* the actual command */
	INTSML count;		/* number of parameters to the command */
	char  *args[2];		/* parameters to the command */
	INTSML used;		/* set when the command is bound */
} us_initialbutton[] =
{
	/* left/middle/right (white/yellow/blue) are basic commands */
	{{"LEFT","WHITE","BUTTON"},"f", "find",    2, {"port",          "extra-info"}, 0},
	{{"RIGHT","YELLOW",""},    "m", "move",    0, {"",              ""},           0},
	{{"MIDDLE","BLUE",""},     "n", "create",  0, {"",              ""},           0},

	/* on four-button puck, add one more basic command */
	{{"GREEN","",""},          "o", "find",    2, {"another",       "port"},       0},

	/* on mice with shift-buttons, add in others still */
	{{"SLEFT","",""},          "",  "find",    1, {"more",          ""},           0},
	{{"SRIGHT","",""},         "",  "var",     2, {"textedit",      "~"},          0},
	{{"SMIDDLE","",""},        "",  "create",  1, {"join-angle",    ""},           0},
	{{NULL, NULL, NULL}, NULL, NULL, 0, {NULL, NULL}} /* 0 */
};

/*
 * default keyboard state: the table below shows the set of commands
 * that will be bound to the keyboard keys.
 */
struct
{
	char  *key;
	char  *command;
	INTSML count;
	char  *args[2];
} us_initialkeyboard[] =
{
	{"a",    "move",      1, {"left",           ""}},
	{"A",    "move",      2, {"left",           "8"}},
	{"b",    "size",      1, {"corner-fixed",   ""}},
	{"c",    "window",    1, {"cursor-centered",""}},
	{"d",    "create",    1, {"join-angle",     ""}},
	{"e",    "erase",     0, {"",               ""}},
	{"E",    "erase",     1, {"pass-through",   ""}},
	{"g",    "grid",      0, {"",               ""}},
	{"G",    "text",      1, {"style",          ""}},
	{"i",    "show",      2, {"object",         "short"}},
	{"I",    "create",    1, {"insert",         ""}},
	{"k",    "text",      2, {"size",           "down"}},
	{"K",    "text",      2, {"size",           "up"}},
	{"l",    "move",      1, {"angle",          ""}},
	{"p",    "window",    1, {"peek",           ""}},
	{"s",    "move",      1, {"right",          ""}},
	{"S",    "move",      2, {"right",          "8"}},
	{"t",    "duplicate", 0, {"",               ""}},
	{"T",    "getproto",  1, {"this-proto",     ""}},
	{"u",    "undo",      0, {"",               ""}},
	{"v",    "window",    1, {"in-zoom",        ""}},
	{"V",    "window",    1, {"out-zoom",       ""}},
	{"w",    "move",      1, {"up",             ""}},
	{"W",    "move",      2, {"up",             "8"}},
	{"x",    "port",      1, {"export",         ""}},
	{"z",    "move",      1, {"down",           ""}},
	{"Z",    "move",      2, {"down",           "8"}},
	{" ",    "getproto",  1, {"next-proto",     ""}},
	{"^^",   "getproto",  1, {"prev-proto",     ""}},
	{"^014", "redraw",	  0, {"",               ""}},
	{"^015", "echo",	  1, {"Electric",       ""}},
	{"?",    "show",      2, {"bindings",       "short"}},
	{"&",    "iterate",   0, {"",               ""}},
	{"!",    "system",    1, {"*",              ""}},
	{"-",    "tellaid",   1, {"user",           ""}},
	{"=",    "tellaid",   2, {"network",        "highlight"}},
	{"[",    "macbegin",  1, {"macro",          ""}},
	{"]",    "macend",    0, {"",               ""}},
	{"%",    "macro",     0, {"",               ""}},
	{",",    "find",      1, {"area-move",      ""}},
	{".",    "find",      1, {"area-size",      ""}},
	{"~",    "arc",       2, {"not",            "rigid"}},
	{"|",    "arc",       1, {"rigid",          ""}},
	{"+",    "arc",       1, {"fixed-angle",    ""}},
	{"/",    "arc",       2, {"not",            "fixed-angle"}},
	{NULL, NULL, 0, {NULL, NULL}} /* 0 */
};

/*
 * When information, detected during broadcast, evokes a reaction that causes
 * change, that change must be queued until the next slice.  Examples of such
 * change are (1) facet center manipulation and (2) deletion of the variable
 * associated with a text window.
 * These routines queue the changes and then execute them when requested
 */
#define	NOUBCHANGE ((UBCHANGE *)-1)
#define	SETCC        3		/* set facet center */
#define	KILLCC       4		/* remove facet center */
#define	KILLFM       5		/* remove facet_message variable */

typedef struct Iubchange
{
	NODEPROTO *facet;		/* facet that changed */
	INTSML     change;		/* type of change */
	INTBIG     x, y;		/* facet center information */
	struct Iubchange *nextubchange;
} UBCHANGE;
static UBCHANGE *us_ubchangefree = NOUBCHANGE;
static UBCHANGE *us_ubchanges = NOUBCHANGE;

/* prototypes for local routines */
void us_splitwindownames(char*, char*, char*, char*, char*);
WINDOW *us_editdummakeeditor(WINDOW*, char*, INTSML*, INTSML*);
INTBIG us_editdumtotallines(WINDOW*);
char *us_editdumgetline(WINDOW*, INTBIG);
void us_editdumaddline(WINDOW*, INTBIG, char*);
void us_editdumreplaceline(WINDOW*, INTBIG, char*);
void us_editdumdeleteline(WINDOW*, INTBIG);
void us_editdumhighlightline(WINDOW*, INTBIG);
void us_editdumsuspendgraphics(WINDOW*);
void us_editdumresumegraphics(WINDOW*);
void us_editdumwritetextfile(WINDOW*, char*);
void us_editdumreadtextfile(WINDOW*, char*);
void us_editdumeditorterm(WINDOW*);
void us_editdumshipchanges(WINDOW*);
INTSML us_editdumgotchar(WINDOW*, INTSML);
void us_editdumcut(WINDOW*);
void us_editdumcopy(WINDOW*);
void us_editdumpaste(WINDOW*);
void us_editdumundo(WINDOW*);
void us_editdumsearch(WINDOW*, char*, INTSML);
INTBIG us_shifttextdescriptor(INTBIG, INTBIG, INTBIG, INTBIG, INTBIG, GEOM*);
INTSML us_newubchange(NODEPROTO*, INTBIG, INTBIG, INTBIG);
void us_freeubchange(UBCHANGE*);
INTSML us_pointonexparc(INTBIG cx, INTBIG cy, INTBIG sx, INTBIG sy, INTBIG ex, INTBIG ey, INTBIG x, INTBIG y);

/*
 * initialization routine to bind keys and buttons to functions
 * returns nonzero upon error
 */
INTSML us_initialbinding(void)
{
	REGISTER INTSML i, k, menusave, fixedmenu, menux, menuy;
	INTSML j;
	char si[50], sj[20], *par[MAXPARS+7];
	REGISTER char **temp;

	/* make the variables with the bindings */
	i = maxi(maxi(NUMKEYS, NUMBUTS), INITIALMENUX*INITIALMENUY);
	temp = (char **)emalloc(i * (sizeof (char *)), el_tempcluster);
	if (temp == 0) return(1);
	for(j=0; j<i; j++) temp[j] = "";
	(void)setvalkey((INTBIG)us_aid, VAID, us_binding_keys, (INTBIG)temp,
		VSTRING|VISARRAY|VDONTSAVE|(NUMKEYS<<VLENGTHSH));
	(void)setvalkey((INTBIG)us_aid, VAID, us_binding_buttons, (INTBIG)temp,
		VSTRING|VISARRAY|VDONTSAVE|(NUMBUTS<<VLENGTHSH));
	(void)setvalkey((INTBIG)us_aid, VAID, us_binding_menu, (INTBIG)temp,
		VSTRING|VISARRAY|VDONTSAVE|((INITIALMENUX*INITIALMENUY)<<VLENGTHSH));
	efree((char *)temp);

	/* bind the keys */
	for(i=0; us_initialkeyboard[i].key != 0; i++)
	{
		par[0] = "set";   par[1] = "key";
		par[2] = us_initialkeyboard[i].key;
		par[3] = us_initialkeyboard[i].command;
		for(j=0; j<us_initialkeyboard[i].count; j++)
			par[j+4] = us_initialkeyboard[i].args[j];
		us_bind((INTSML)(us_initialkeyboard[i].count+4), par);
	}

	/* bind the tablet commands that fit on the tablet puck */
	for(i=0; i<buttoncount(); i++)
	{
		(void)strcpy(si, buttonname(i, &j));
		for(j=0; us_initialbutton[j].but[0] != 0; j++)
			if (us_initialbutton[j].used == 0)
		{
			for(k=0; k<3; k++)
				if (namesame(si, us_initialbutton[j].but[k]) == 0)
			{
				par[0] = "set";   par[1] = "button";   par[2] = si;
				par[3] = us_initialbutton[j].com;
				for(k=0; k<us_initialbutton[j].count; k++)
					par[k+4] = us_initialbutton[j].args[k];
				us_bind((INTSML)(us_initialbutton[j].count+4), par);
				us_initialbutton[j].used = 1;
				break;
			}
		}
	}

	/* now bind those tablet commands that can't fit on the tablet */
	for(j=0; us_initialbutton[j].but[0] != 0; j++)
		if (us_initialbutton[j].used == 0 && *us_initialbutton[j].key != 0)
	{
		par[0] = "set";   par[1] = "key";   par[2] = us_initialbutton[j].key;
		par[3] = us_initialbutton[j].com;
		for(k=0; k<us_initialbutton[j].count; k++)
			par[k+4] = us_initialbutton[j].args[k];
		us_bind((INTSML)(us_initialbutton[j].count+4), par);
	}

	/* bind the fixed command menu entries */
	if (us_menupos <= 1)
	{
		menux = INITIALMENUX;
		menuy = INITIALMENUY;
	} else
	{
		menux = INITIALMENUY;
		menuy = INITIALMENUX;
	}
	j = us_setmenusize(menux, menuy, us_menupos, 0);
	menusave = us_aid->aidstate&MENUON;   us_aid->aidstate &= ~MENUON;
	for(fixedmenu=0; us_initialmenu[fixedmenu].command != 0; fixedmenu++)
		;
	for(i=0; i<fixedmenu; i++)
	{
		par[0] = "set";   par[1] = "menu";
		j = 2;
		if (*us_initialmenu[i].glyph != 0)
		{
			par[j++] = "glyph";
			par[j++] = us_initialmenu[i].glyph;
		}
		(void)sprintf(si, "%d", i%INITIALMENUX);
		(void)sprintf(sj, "%d", i/INITIALMENUX);
		if (us_menupos <= 1)
		{
			par[j++] = sj;  par[j++] = si;
		} else
		{
			par[j++] = si;  par[j++] = sj;
		}
		par[j++] = us_initialmenu[i].command;
		for(k=0; k<us_initialmenu[i].count; k++)
			par[j++] = us_initialmenu[i].args[k];
		us_bind(j, par);
	}

	/* initially set the rest of the menu entries to "getproto" */
	for(i=0; i<(INITIALMENUX*INITIALMENUY-fixedmenu); i++)
	{
		par[0] = "set";   par[1] = "menu";
		(void)sprintf(si, "%d", (i+fixedmenu)%INITIALMENUX);
		(void)sprintf(sj, "%d", (i+fixedmenu)/INITIALMENUX);
		if (us_menupos <= 1)
		{
			par[2] = sj;  par[3] = si;
		} else
		{
			par[2] = si;  par[3] = sj;
		}
		par[4] = "getproto";
		us_bind(5, par);
	}

	/* now fill in the "getproto" commands properly */
	us_setmenunodearcs();

	if (menusave != 0) us_aid->aidstate |= MENUON; else
		us_aid->aidstate &= ~MENUON;
	return(0);
}

/*
 * routine to determine for technology "tech" which node and arc prototypes
 * have opaque layers and set the bits in the prototype->userbits.
 * The rules for layer orderings are that the overlappable layers
 * must come first followed by the opaque layers.  The field that is
 * set in the "userbits" is then the index of the first opaque layer.
 */
void us_figuretechopaque(TECHNOLOGY *tech)
{
	REGISTER INTSML j, dispstyle;
	REGISTER INTBIG k;
	REGISTER NODEPROTO *np;
	REGISTER ARCPROTO *ap;
	REGISTER NODEINST *node;
	REGISTER ARCINST *arc;
	static POLYGON *poly = NOPOLYGON;

	/* get polygon */
	if (poly == NOPOLYGON) poly = allocpolygon(4, us_aid->cluster);

	if (el_curwindow == NOWINDOW) dispstyle = COLORMAP; else
		dispstyle = us_getdispstyle(el_curwindow);
	for(np = tech->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
	{
		np->userbits &= ~NHASOPA;
		if (dispstyle == BWRASTER) continue;
		node = dummynode();
		node->proto = np;
		node->lowx = np->lowx;   node->highx = np->highx;
		node->lowy = np->lowy;   node->highy = np->highy;
		j = nodepolys(node);
		for(k=0; k<j; k++)
		{
			shapenodepoly(node, (INTSML)k, poly);
			if (poly->desc->bits == LAYERN) continue;
			if ((poly->desc->bits & ~(LAYERT1|LAYERT2|LAYERT3|LAYERT4|LAYERT5)) == 0)
			{
				/* overlappable layer found, make sure it is at start */
				if ((np->userbits&NHASOPA) != 0)
					ttyputerr("%s: node %s has layers out of order!", tech->techname, np->primname);
				continue;
			}

			/* opaque layer found, mark its index if it is the first */
			if ((np->userbits&NHASOPA) == 0) np->userbits |= (k << NFIRSTOPASH);
			np->userbits |= NHASOPA;
		}
	}
	for(ap = tech->firstarcproto; ap != NOARCPROTO; ap = ap->nextarcproto)
	{
		ap->userbits &= ~AHASOPA;
		if (dispstyle == BWRASTER) continue;
		arc = dummyarc();
		arc->proto = ap;
		arc->userbits = ISDIRECTIONAL;
		j = arcpolys(arc);
		for(k=0; k<j; k++)
		{
			shapearcpoly(arc, (INTSML)k, poly);
			if (poly->desc->bits == LAYERN) continue;
			if ((poly->desc->bits & ~(LAYERT1|LAYERT2|LAYERT3|LAYERT4|LAYERT5)) == 0)
			{
				/* overlappable layer found, make sure it is at start */
				if ((ap->userbits&AHASOPA) != 0)
					ttyputerr("Arc %s:%s has layers out of order!", tech->techname, ap->protoname);
				continue;
			}

			/* opaque layer found, mark its index if it is the first */
			if ((ap->userbits&AHASOPA) == 0) ap->userbits |= (k << AFIRSTOPASH);
			ap->userbits |= AHASOPA;
		}
	}
}

/*
 * routine to examine the current window structure and fit their sizes
 * to the screen.  If "placemenu" is nonzero, set the menu location too.
 */
void us_windowfit(void *whichframe, INTSML placemenu)
{
	REGISTER WINDOW *w;
	REGISTER INTSML i, total, lowy, highy, lx, hx, ly, hy, drawlx, drawhx, drawly, drawhy, mtop, mleft, alltext;
	INTSML swid, shei, mwid, mhei, pwid;
	REGISTER void *frame;
	REGISTER VARIABLE *var;
	COMMANDBINDING commandbinding;

	for(frame = us_firstwindowframe(); frame != 0; frame = us_nextwindowframe(frame))
	{
		if (whichframe != 0 && whichframe != frame) continue;
		us_getwindowsize(frame, &swid, &shei);
		lowy = 0;   highy = shei - 1 - us_menubarsize;

		/* presume that there is no menu */
		drawlx = 0;      drawhx = swid-1;
		drawly = lowy;   drawhy = highy;

		/* if there is a menu, figure it out */
		if ((us_aid->aidstate&MENUON) != 0)
		{
			/* see if the menu is all text */
			alltext = 0;
			var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_binding_menu);
			if (var != NOVARIABLE)
			{
				total = us_menux*us_menuy;
				for(i=0; i<total; i++)
				{
					us_parsebinding(((char **)var->addr)[i], &commandbinding);
					if (*commandbinding.command == 0 || commandbinding.nodeglyph != NONODEPROTO ||
						commandbinding.arcglyph != NOARCPROTO)
					{
						us_freebindingparse(&commandbinding);
						break;
					}
					us_freebindingparse(&commandbinding);
				}
				if (i >= total) alltext = 1;
			}
			us_getpaletteparameters(&mwid, &mhei, &pwid);
			if (us_menuframe == 0)
			{
				/* menus come out of the only editor window */
				switch (us_menupos)
				{
					case 0:		/* menu at top */
						us_menuxsz = us_menuysz = swid / us_menux;
						us_menulx = (swid - us_menux*us_menuxsz)/2;
						us_menuhx = swid - us_menulx;
						us_menuly = highy - us_menuysz*us_menuy;
						us_menuhy = highy;
						drawlx = 0;      drawhx = swid-1;
						drawly = lowy;   drawhy = us_menuly-1;
						break;
					case 1:		/* menu at bottom */
						us_menuxsz = us_menuysz = swid / us_menux;
						us_menulx = (swid - us_menux*us_menuxsz)/2;
						us_menuhx = swid - us_menulx;
						us_menuly = lowy;
						us_menuhy = lowy + us_menuysz * us_menuy;
						drawlx = 0;             drawhx = swid-1;
						drawly = us_menuhy+1;   drawhy = highy;
						break;
					case 2:		/* menu on left */
						us_menuxsz = us_menuysz = (highy-lowy) / us_menuy;

						/* if the menu is all text, allow nonsquare menus */
						if (alltext != 0) us_menuxsz = ALLTEXTWIDTH / us_menux;

						us_menulx = 0;
						us_menuhx = us_menuxsz * us_menux;
						us_menuly = ((highy-lowy) - us_menuy*us_menuysz)/2 + lowy;
						us_menuhy = us_menuly + us_menuy*us_menuysz;
						drawlx = us_menuhx+1;   drawhx = swid-1;
						drawly = lowy;          drawhy = highy;
						break;
					case 3:		/* menu on right */
						us_menuxsz = us_menuysz = (highy-lowy) / us_menuy;

						/* if the menu is all text, allow nonsquare menus */
						if (alltext != 0) us_menuxsz = ALLTEXTWIDTH / us_menux;

						us_menulx = swid - us_menuxsz * us_menux;
						us_menuhx = swid-1;
						us_menuly = ((highy-lowy) - us_menuy*us_menuysz)/2 + lowy;
						us_menuhy = us_menuly + us_menuy*us_menuysz;
						drawlx = 0;      drawhx = us_menulx-1;
						drawly = lowy;   drawhy = highy;
						break;
				}
			} else
			{
				/* floating menu window */
				if (frame == us_menuframe && placemenu != 0)
				{
					us_getpaletteparameters(&mwid, &mhei, &pwid);
					switch (us_menupos)
					{
						case 0:		/* menu at top */
						case 1:		/* menu at bottom */
							us_menuxsz = us_menuysz = mwid / us_menux;
							if (us_menuysz * us_menuy > pwid)
								us_menuxsz = us_menuysz = pwid / us_menuy;
							us_menulx = 0;
							us_menuhx = us_menux * us_menuxsz;
							us_menuly = 0;
							us_menuhy = us_menuy * us_menuysz;
							mleft = 0;
							if (us_menupos == 0)
							{
								/* menu on the top */
								mtop = 1;
							} else
							{
								/* menu on the bottom */
								mtop = mhei - us_menuysz*us_menuy - 3;
							}
							break;

						case 2:		/* menu on left */
						case 3:		/* menu on right */
							/* determine size of menu entries */
							us_menuxsz = us_menuysz = mhei / us_menuy;
							if (us_menuxsz * us_menux > pwid)
								us_menuxsz = us_menuysz = pwid / us_menux;

							/* if the menu is all text, allow nonsquare menus */
							if (alltext != 0) us_menuxsz = ALLTEXTWIDTH / us_menux;

							/* compute menu parameters */
							us_menuly = 0;
							us_menuhy = us_menuy * us_menuysz;
							us_menulx = 0;
							us_menuhx = us_menux * us_menuxsz;
							mtop = 0;
							if (us_menupos == 2)
							{
								/* menu on the left */
								mleft = 0;
							} else
							{
								/* menu on the right */
								mleft = mwid - us_menuxsz * us_menux - 2;
							}
							break;
					}
					us_sizewindowframe(us_menuframe, (INTSML)(us_menuhx-us_menulx), (INTSML)(us_menuhy-us_menuly));
					us_movewindowframe(us_menuframe, mtop, mleft);
				}
			}
		}

		/* now fit the windows in the remaining space */
		for(w=el_topwindow; w != NOWINDOW; w = w->nextwindow)
		{
			/* this window must be on the right frame */
			if (w->frame != frame) continue;

			/* entire window is handled simply */
			if (strcmp(w->location, "entire") == 0)
			{
				lx = drawlx;              hx = drawhx;
				ly = drawly;              hy = drawhy;
			} else if (strncmp(w->location, "top", 3) == 0)
			{
				lx = drawlx;              hx = drawhx;
				ly = (drawly+drawhy)/2;   hy = drawhy;
			} else if (strncmp(w->location, "bot", 3) == 0)
			{
				lx = drawlx;              hx = drawhx;
				ly = drawly;              hy = (drawhy+drawly)/2;
			} else if (strcmp(w->location, "left") == 0)
			{
				lx = drawlx;              hx = (drawlx+drawhx)/2;
				ly = drawly;              hy = drawhy;
			} else if (strcmp(w->location, "right") == 0)
			{
				lx = (drawlx+drawhx)/2;   hx = drawhx;
				ly = drawly;              hy = drawhy;
			}

			/* subdivide for fractions of half windows */
			i = 3;
			while (w->location[i] == '-')
			{
				switch (w->location[i+1])
				{
					case 'l': hx = (lx + hx)/2;   break;
					case 'r': lx = (lx + hx)/2;   break;
					case 't': ly = (ly + hy)/2;   break;
					case 'b': hy = (ly + hy)/2;   break;
				}
				i += 2;
			}
			if (strcmp(el_topwindow->location, "entire") != 0)
			{
				lx++;   hx--;   ly++;   hy--;
			}

			/* make sure window has some size */
			if (lx >= hx) hx = lx + 1;
			if (ly >= hy) hy = ly + 1;

			/* make room for border if simulating */
			if ((w->state&WINDOWSIMULATING) != 0)
			{
				lx += SIMULATINGBORDERSIZE;   hx -= SIMULATINGBORDERSIZE;
				ly += SIMULATINGBORDERSIZE;   hy -= SIMULATINGBORDERSIZE;
			}
			w->uselx = lx;   w->usehx = hx;
			w->usely = ly;   w->usehy = hy;
		}
	}
}

/*
 * routine to tell the names of the windows that result when the window
 * with name "w" is split.  The strings "hwind1" and "hwind2" are filled
 * with the names if the window is split horizontally.  The strings "vwind1"
 * and "vwind2" are filled with the names if the window is split verticaly.
 */
void us_splitwindownames(char *w, char *hwind1, char *hwind2, char *vwind1, char *vwind2)
{
	REGISTER INTSML i;

	if (strcmp(w, "entire") == 0)
	{
		(void)strcpy(hwind1, "top");   (void)strcpy(hwind2, "bottom");
		(void)strcpy(vwind1, "left");  (void)strcpy(vwind2, "right");
		return;
	}
	if (strcmp(w, "top") == 0)
	{
		(void)strcpy(hwind1, "top-l"); (void)strcpy(hwind2, "top-r");
		(void)strcpy(vwind1, "");      (void)strcpy(vwind2, "");
		return;
	}
	if (strcmp(w, "bottom") == 0)
	{
		(void)strcpy(hwind1, "bot-l"); (void)strcpy(hwind2, "bot-r");
		(void)strcpy(vwind1, "");      (void)strcpy(vwind2, "");
		return;
	}
	if (strcmp(w, "left") == 0)
	{
		(void)strcpy(vwind1, "top-l"); (void)strcpy(vwind2, "bot-l");
		(void)strcpy(hwind1, "");      (void)strcpy(hwind2, "");
		return;
	}
	if (strcmp(w, "right") == 0)
	{
		(void)strcpy(vwind1, "top-r"); (void)strcpy(vwind2, "bot-r");
		(void)strcpy(hwind1, "");      (void)strcpy(hwind2, "");
		return;
	}
	(void)strcpy(hwind1, w);   (void)strcpy(hwind2, w);
	(void)strcpy(vwind1, w);   (void)strcpy(vwind2, w);
	i = w[strlen(w)-1];
	if (i == 'l' || i == 'r')
	{
		(void)strcat(vwind1, "-t");  (void)strcat(vwind2, "-b");
		(void)strcpy(hwind1, "");    (void)strcpy(hwind2, "");
	} else
	{
		(void)strcat(hwind1, "-l");  (void)strcat(hwind2, "-r");
		(void)strcpy(vwind1, "");    (void)strcpy(vwind2, "");
	}
}

/*
 * Routine to create a new window with whatever method is available on the
 * current machine (new window in its own frame or just a split of the current
 * window).  Prints an error and returns NOWINDOW on failure.
 */
WINDOW *us_wantnewwindow(void)
{
	REGISTER WINDOW *w;

	if (us_graphicshas(CANUSEFRAMES))
	{
		w = us_makenewwindow();
		if (w == NOWINDOW) us_abortcommand("Cannot create new window");
	} else
	{
		if (us_needwindow()) return(NOWINDOW);
		w = us_splitcurrentwindow(0, 0);
	}
	return(w);
}

WINDOW *us_makenewwindow(void)
{
	void *mw;
	REGISTER WINDOW *w;

	/* create a default window space on this frame */
	w = newewindow("entire", NOWINDOW, 1);
	if (w == NOWINDOW) return(NOWINDOW);
	w->buttonhandler = DEFAULTBUTTONHANDLER;
	w->charhandler = DEFAULTCHARHANDLER;
	w->changehandler = DEFAULTCHANGEHANDLER;
	w->termhandler = DEFAULTTERMHANDLER;
	w->redisphandler = DEFAULTREDISPHANDLER;
	(void)setvalkey((INTBIG)us_aid, VAID, us_current_window, (INTBIG)w, VWINDOW|VDONTSAVE);

	/* now draw everything */
	mw = us_getwindowframe();
	us_drawmenu(0, mw);
	return(w);
}

/*
 * routine to split the current window into two windows.  If "splitkey" is zero,
 * nature of split is unspecified.  If "splitkey" is 1, split horizontally
 * (only when splitting top window).  If "splitkey" is 2, split vertically.
 * If "fillboth" is nonzero, fill both windows with the contents of the
 * old one.  Otherwise, leave the new current window empty.  Returns the address
 * of the new current window (NOWINDOW on error).
 */
WINDOW *us_splitcurrentwindow(INTSML splitkey, INTSML fillboth)
{
	char wind1[40], wind2[40], vwind1[40], vwind2[40];
	REGISTER char *win1, *win2;
	WINDOW windowsave;
	REGISTER WINDOW *w2, *w3, *w;
	REGISTER INTSML curwx, curwy;
	REGISTER INTBIG x, y, l;
	REGISTER NODEPROTO *np;

	/* figure out new name of windows */
	us_splitwindownames(el_curwindow->location, wind1, wind2, vwind1, vwind2);

	/* use the horizontal window split unless there is none */
	if (*wind1 == 0) win1 = vwind1; else win1 = wind1;
	if (*wind2 == 0) win2 = vwind2; else win2 = wind2;

	/* special case when splitting just one window: which way to split */
	if (strcmp(el_topwindow->location, "entire") == 0)
	{
		/* see if a "horizontal" or "vertical" parameter was given */
		if (splitkey == 2)
		{
			/* vertical window specified explicitly */
			win1 = vwind1;   win2 = vwind2;
		} else if (splitkey == 0)
		{
			/* make a guess about window splitting */
			np = el_topwindow->curnodeproto;
			if (np != NONODEPROTO)
			{
				curwx = el_topwindow->usehx - el_topwindow->uselx;
				curwy = el_topwindow->usehy - el_topwindow->usely;
				x = np->highx - np->lowx;
				y = np->highy - np->lowy;
				l = el_curtech->deflambda;
				if (muldiv(x, curwy/2, l) + muldiv(y, curwx, l) >=
					muldiv(x, curwy, l) + muldiv(y, curwx/2, l))
				{
					/* vertical window makes more sense */
					win1 = vwind1;   win2 = vwind2;
				}
			}
		}
	}

	/* turn off object and window highlighting */
	us_pushhighlight();
	us_clearhighlightcount();
	w = el_curwindow;
	copywindow(&windowsave, el_curwindow);

	/* make two new windows in "w2" and "w3" to replace "el_curwindow" */
	(void)setvalkey((INTBIG)us_aid, VAID, us_current_window, (INTBIG)NOWINDOW, VWINDOW|VDONTSAVE);
	startobjectchange((INTBIG)us_aid, VAID);
	w2 = newewindow(win1, w, 0);
	w3 = newewindow(win2, w, 0);
	if (w2 == NOWINDOW || w3 == NOWINDOW)
	{
		ttyputerr("No memory for windows");
		return(NOWINDOW);
	}

	/* if splitting a simulating window, restore default handlers (turn off simulation there) */
	if ((windowsave.state&WINDOWSIMULATING) != 0)
	{
		w2->charhandler = DEFAULTCHARHANDLER;
		w2->termhandler = DEFAULTTERMHANDLER;
		w3->charhandler = DEFAULTCHARHANDLER;
		w3->termhandler = DEFAULTTERMHANDLER;
	}

	/* if splitting an editor window, move the editor structure */
	if ((w->state&WINDOWTYPE) == TEXTWINDOW || (w->state&WINDOWTYPE) == POPTEXTWINDOW)
	{
		(void)setval((INTBIG)w3, VWINDOW, "editor", (INTBIG)w->editor, VADDRESS);
		(void)setval((INTBIG)w, VWINDOW, "editor", -1, VADDRESS);
	}

	/* free the current window */
	killwindow(w);

	/* set the window extents */
	us_windowfit(w2->frame, 0);

	/* make sure that "w3" is the larger window */
	if (w2->usehx - w2->uselx > w3->usehx - w3->uselx ||
		w2->usehy - w2->usely > w3->usehy - w3->usely)
	{
		w = w2;   w2 = w3;   w3 = w;
	}

	/* zap window 2 if both are not to be filled */
	if ((windowsave.state&WINDOWTYPE) != DISPWINDOW) fillboth = 0;
	if (fillboth == 0)
	{
		w2->state = (w2->state & ~(WINDOWTYPE|GRIDON)) | DISPWINDOW;
		w2->buttonhandler = DEFAULTBUTTONHANDLER;
		w2->charhandler = DEFAULTCHARHANDLER;
		w2->changehandler = DEFAULTCHANGEHANDLER;
		w2->termhandler = DEFAULTTERMHANDLER;
		w2->redisphandler = DEFAULTREDISPHANDLER;
		w2->curnodeproto = NONODEPROTO;
		w2->editor = NOEDITOR;
	}

	/* use former window for scaling if block transfers works */
	if (us_graphicshas(CANMOVEBOX) == 0) w = NOWINDOW; else w = &windowsave;

	us_squarescreen(w2, w, 1, &w2->screenlx, &w2->screenhx, &w2->screenly, &w2->screenhy);
	computewindowscale(w2);
	us_squarescreen(w3, w, 1, &w3->screenlx, &w3->screenhx, &w3->screenly, &w3->screenhy);
	computewindowscale(w3);

	/* windows might have got bigger: see if grid can be drawn */
	if ((w2->state&GRIDTOOSMALL) != 0) us_gridset(w2, GRIDON);
	if ((w2->state&GRIDTOOSMALL) != 0) us_gridset(w3, GRIDON);

	endobjectchange((INTBIG)us_aid, VAID);
	(void)setvalkey((INTBIG)us_aid, VAID, us_current_window, (INTBIG)w2, VWINDOW|VDONTSAVE);

	/* restore all highlighting */
	(void)us_pophighlight(0);
	return(w2);
}

/*
 * routine to kill a window.  Kills the current window if "thisw" is nonzero.
 * Kills the other window, making the current one larger, if "thisw" is zero.
 */
void us_killcurrentwindow(INTSML thisw)
{
	REGISTER WINDOW *w1, *w2, *wnew, *w;
	WINDOW windowsave;
	char windcomb[40], windother[40], wind1[40], wind2[40], vwind1[40], vwind2[40];
	void *mw;

	w1 = el_curwindow;

	/* if this is the only partition, see if the window frame can be deleted */
	if (strcmp(w1->location, "entire") == 0)
	{
		if (us_graphicshas(CANUSEFRAMES) != 0)
		{
			/* save highlighting and turn it off */
			us_pushhighlight();
			us_clearhighlightcount();

			/* remember the window frame to delete it later */
			mw = w1->frame;

			/* kill the window */
			startobjectchange((INTBIG)us_aid, VAID);
			us_killwindowpickanother(w1);
			endobjectchange((INTBIG)us_aid, VAID);

			/* restore highlighting */
			(void)us_pophighlight(0);
			return;
		}
	}

	/* figure out which other window to merge this with */
	if (strcmp(w1->location, "top") == 0 || strcmp(w1->location, "bottom") == 0 ||
		strcmp(w1->location, "left") == 0 || strcmp(w1->location, "right") == 0)
			(void)strcpy(windcomb, "entire"); else
	{
		(void)strcpy(windcomb, w1->location);
		windcomb[strlen(windcomb)-2] = 0;
		if (strcmp(windcomb, "bot") == 0) (void)strcpy(windcomb, "bottom");
	}

	/* see what divisions this higher window typically makes */
	us_splitwindownames(windcomb, wind1, wind2, vwind1, vwind2);

	/* look for the other window of the typical split */
	(void)strcpy(windother, "");
	if (strcmp(wind2, w1->location) == 0) (void)strcpy(windother, wind1);
	if (strcmp(wind1, w1->location) == 0) (void)strcpy(windother, wind2);
	if (strcmp(vwind2, w1->location) == 0) (void)strcpy(windother, vwind1);
	if (strcmp(vwind1, w1->location) == 0) (void)strcpy(windother, vwind2);

	/* see if there is a window with that name */
	for(w2 = el_topwindow; w2 != NOWINDOW; w2 = w2->nextwindow)
		if (strcmp(w2->location, windother) == 0) break;

	/* if the other window can't be found, try one more hack */
	if (w2 == NOWINDOW)
	{
		/* special case for quadrants that get split strangely */
		if ((strncmp(w1->location, "top-", 4) == 0 || strncmp(w1->location, "bot-", 4) == 0) &&
			strlen(w1->location) == 5)
		{
			if (*w1->location == 't') (void)strcpy(windother, "bot-l"); else
				(void)strcpy(windother, "top-l");
			windother[4] = w1->location[4];
			if (windother[4] == 'l') (void)strcpy(windcomb, "left"); else
				(void)strcpy(windcomb, "right");
			for(w2 = el_topwindow; w2 != NOWINDOW; w2 = w2->nextwindow)
				if (strcmp(w2->location, windother) == 0) break;
		}
	}
	if (w2 == NOWINDOW)
	{
		us_abortcommand("Cannot kill the current window");
		return;
	}

	/* if the other window is to be killed, swap them */
	if (thisw == 0)
	{
		w = w1;   w1 = w2;   w2 = w;
	}

	/* turn off highlighting */
	us_pushhighlight();
	us_clearhighlightcount();

	/* create a new window to cover both */
	(void)setvalkey((INTBIG)us_aid, VAID, us_current_window, (INTBIG)NOWINDOW, VWINDOW|VDONTSAVE);
	startobjectchange((INTBIG)us_aid, VAID);
	wnew = newewindow(windcomb, w2, 0);
	if (wnew == NOWINDOW) return;

	/* save information from the old window */
	copywindow(&windowsave, w2);

	/* if merging an editor window, move the editor structure */
	if ((w2->state&WINDOWTYPE) == TEXTWINDOW || (w2->state&WINDOWTYPE) == POPTEXTWINDOW)
	{
		(void)setval((INTBIG)wnew, VWINDOW, "editor", (INTBIG)w2->editor, VADDRESS);
		(void)setval((INTBIG)w2, VWINDOW, "editor", -1, VADDRESS);
	}

	/* remove old windows */
	killwindow(w1);
	killwindow(w2);

	/* set window extents */
	us_windowfit(wnew->frame, 0);

	/* use former window for scaling if block transfers works */
	if (us_graphicshas(CANMOVEBOX) == 0) w = NOWINDOW; else w = &windowsave;

	/* re-draw the new window */
	us_squarescreen(wnew, w, 0, &wnew->screenlx, &wnew->screenhx,
		&wnew->screenly, &wnew->screenhy);
	computewindowscale(wnew);

	/* window might have got bigger: see if grid can be drawn */
	if ((wnew->state&GRIDTOOSMALL) != 0) us_gridset(wnew, GRIDON);

	(void)setval((INTBIG)el_curlib, VLIBRARY, "curnodeproto", (INTBIG)wnew->curnodeproto, VNODEPROTO);
	endobjectchange((INTBIG)us_aid, VAID);
	(void)setvalkey((INTBIG)us_aid, VAID, us_current_window, (INTBIG)wnew, VWINDOW|VDONTSAVE);

	/* restore highlighting */
	(void)us_pophighlight(0);
}

/*
 * routine to determine whether the division of window "w1" into windows
 * "w2" and "w3" can be done with block transfer.  This requires that
 * the windows have identical aspect ratios in one axis
 */
INTSML us_windowcansplit(WINDOW *w1, WINDOW *w2, WINDOW *w3)
{
	REGISTER INTBIG new2l, new2h;

	if (w2->usehx-w2->uselx == w3->usehx-w3->uselx &&
		w2->screenhx-w2->screenlx == w3->screenhx-w3->screenlx)
	{
		/* first see if it is an obvious split */
		if (w2->usehx-w2->uselx == w1->usehx-w1->uselx &&
			w2->screenhx-w2->screenlx == w1->screenhx-w1->screenlx)
				return(1);

		/* now see if it is a relative fit for changed window size */
		new2l = muldiv(((w1->usehx-w1->uselx) - (w2->usehx-w2->uselx))/2,
			w1->screenhx-w1->screenlx, w1->usehx-w1->uselx) + w1->screenlx;
		new2h = w2->screenlx + muldiv(w2->usehx-w2->uselx,
			w1->screenhx-w1->screenlx, w1->usehx-w1->uselx);
		if (new2l == w2->screenlx && new2h == w2->screenhx) return(1);
	}
	if (w2->usehy-w2->usely == w3->usehy-w3->usely &&
		w2->screenhy-w2->screenly == w3->screenhy-w3->screenly)
	{
		/* first see if it is an obvious split */
		if (w2->usehy-w2->usely == w1->usehy-w1->usely &&
			w2->screenhy-w2->screenly == w1->screenhy-w1->screenly)
				return(1);

		/* now see if it is a relative fit for changed window size */
		new2l = muldiv(((w1->usehy-w1->usely) - (w2->usehy-w2->usely))/2,
			w1->screenhy-w1->screenly, w1->usehy-w1->usely) + w1->screenly;
		new2h = w2->screenly + muldiv(w2->usehy-w2->usely,
			w1->screenhy-w1->screenly, w1->usehy-w1->usely);
		if (new2l == w2->screenly && new2h == w2->screenhy) return(1);
	}
	return(0);
}

/*
 * routine to ensure that the object pointed to by geometry module "geom"
 * is displayed somewhere on the screen.  If not, the appropriate facet
 * is displayed in the current window
 */
void us_ensurewindow(GEOM *geom)
{
	REGISTER NODEPROTO *np;
	REGISTER WINDOW *w;
	char *par[1];

	/* if the module is dummy, do nothing */
	if (geom == NOGEOM) return;

	/* determine the facet that has this geometry module */
	np = geomparent(geom);

	/* see if that facet is in a window */
	for(w = el_topwindow; w != NOWINDOW; w = w->nextwindow)
		if (w->curnodeproto == np) break;

	/* if the facet is not in a window, put it there */
	if (w == NOWINDOW)
	{
		par[0] = describenodeproto(np);
		us_editfacet(1, par);
	}
}

/*
 * Routine to kill window "w" and set the current window to some other.
 */
void us_killwindowpickanother(WINDOW *w)
{
	REGISTER NODEPROTO *np;

	killwindow(w);

	if (w != el_curwindow) return;

	w = el_topwindow;
	(void)setvalkey((INTBIG)us_aid, VAID, us_current_window, (INTBIG)w, VWINDOW|VDONTSAVE);
	if (w != NOWINDOW) np = w->curnodeproto; else np = NONODEPROTO;
	(void)setval((INTBIG)el_curlib, VLIBRARY, "curnodeproto", (INTBIG)np, VNODEPROTO);
}

/*
 * routine to adjust the coordinate values in (x, y) from screen space to
 * the space of window "w"
 */
void us_scaletowindow(INTBIG *x, INTBIG *y, WINDOW *w)
{
	*x = muldiv(*x - w->uselx, w->screenhx - w->screenlx, w->usehx - w->uselx) + w->screenlx;
	*y = muldiv(*y - w->usely, w->screenhy - w->screenly, w->usehy - w->usely) + w->screenly;
}

/******************** TEXT EDITING ********************/

/* dummy editor routines */
WINDOW *us_editdummakeeditor(WINDOW *oriwin, char *header, INTSML *chars, INTSML *lines)
{
	ttyputmsg("Sorry, this is not a working text editor");
	return(NOWINDOW);
}
INTBIG us_editdumtotallines(WINDOW *win) { return(0); }
char *us_editdumgetline(WINDOW *win, INTBIG index) { return(""); }
void us_editdumaddline(WINDOW *win, INTBIG index, char *str) {}
void us_editdumreplaceline(WINDOW *win, INTBIG index, char *str) {}
void us_editdumdeleteline(WINDOW *win, INTBIG index) {}
void us_editdumhighlightline(WINDOW *win, INTBIG index) {}
void us_editdumsuspendgraphics(WINDOW *win) {}
void us_editdumresumegraphics(WINDOW *win) {}
void us_editdumwritetextfile(WINDOW *win, char *file) {}
void us_editdumreadtextfile(WINDOW *win, char *file) {}
void us_editdumeditorterm(WINDOW *w) {}
void us_editdumshipchanges(WINDOW *w) {}
INTSML us_editdumgotchar(WINDOW *w, INTSML i) { return(0); }
void us_editdumcut(WINDOW *w) {}
void us_editdumcopy(WINDOW *w) {}
void us_editdumpaste(WINDOW *w) {}
void us_editdumundo(WINDOW *w) {}
void us_editdumsearch(WINDOW *w, char *str, INTSML fromtop) {}

EDITORTABLE us_editortable[] =
{
	/* the point-and-click editor */
	{"Point-and-click",
	us_editpacmakeeditor, us_editpactotallines, us_editpacgetline, us_editpacaddline,
	us_editpacreplaceline, us_editpacdeleteline, us_editpachighlightline,
	us_editpacsuspendgraphics, us_editpacresumegraphics,
	us_editpacwritetextfile, us_editpacreadtextfile,
	us_editpaceditorterm, us_editpacshipchanges, us_editpacgotchar,
	us_editpaccut, us_editpaccopy, us_editpacpaste,
	us_editpacundo, us_editpacsearch},

	/* the EMACS-like editor */
	{"EMACS",
	us_editemacsmakeeditor, us_editemacstotallines, us_editemacsgetline, us_editemacsaddline,
	us_editemacsreplaceline, us_editemacsdeleteline, us_editemacshighlightline,
	us_editemacssuspendgraphics, us_editemacsresumegraphics,
	us_editemacswritetextfile, us_editemacsreadtextfile,
	us_editemacseditorterm, us_editemacsshipchanges, us_editemacsgotchar,
	us_editemacscut, us_editemacscopy, us_editemacspaste,
	us_editemacsundo, us_editemacssearch},

	/* the nonexistant editor */
	{"Null",
	us_editdummakeeditor, us_editdumtotallines, us_editdumgetline, us_editdumaddline,
	us_editdumreplaceline, us_editdumdeleteline, us_editdumhighlightline,
	us_editdumsuspendgraphics, us_editdumresumegraphics,
	us_editdumwritetextfile, us_editdumreadtextfile,
	us_editdumeditorterm, us_editdumshipchanges, us_editdumgotchar,
	us_editdumcut, us_editdumcopy, us_editdumpaste,
	us_editdumundo, us_editdumsearch},

	{NULL,
    NULL, NULL, NULL, NULL,
    NULL, NULL, NULL,
    NULL, NULL,
    NULL, NULL,
    NULL, NULL, NULL,
    NULL, NULL} /* 0 */
};

/*
 * dispatch routine to describe this editor
 */
void us_describeeditor(char **name)
{
	*name = us_editortable[us_currenteditor].editorname;
}

/*
 * dispatch routine for creating a new editor
 */
WINDOW *us_makeeditor(WINDOW *oriwin, char *header, INTSML *chars, INTSML *lines)
{
	/* quit if text editing not available */
	if (checkcap(us_aid, USERHASTEXTED) == 0)
	{
		ttyputerr("Sorry, text editing is not available");
		return(NOWINDOW);
	}

	return((*us_editortable[us_currenteditor].makeeditor)(oriwin,header,chars,lines));
}

/*
 * dispatch routine to return the total number of valid lines in the edit buffer
 */
INTBIG us_totallines(WINDOW *win)
{
	return((*us_editortable[us_currenteditor].totallines)(win));
}

/*
 * dispatch routine to get the string on line "index" (0 based).  A negative line
 * returns the current line.  Returns -1 if the index is beyond the file limit
 */
char *us_getline(WINDOW *win, INTBIG index)
{
	return((*us_editortable[us_currenteditor].getline)(win, index));
}

/*
 * dispatch routine to add line "str" to the text facet to become line "index"
 */
void us_addline(WINDOW *win, INTBIG index, char *str)
{
	(*us_editortable[us_currenteditor].addline)(win, index, str);
}

/*
 * dispatch routine to replace the line number "index" with the string "str".
 */
void us_replaceline(WINDOW *win, INTBIG index, char *str)
{
	(*us_editortable[us_currenteditor].replaceline)(win, index, str);
}

/*
 * dispatch routine to delete line number "index"
 */
void us_deleteline(WINDOW *win, INTBIG index)
{
	(*us_editortable[us_currenteditor].deleteline)(win, index);
}

/*
 * dispatch routine to highlight line "index" in the text window
 */
void us_highlightline(WINDOW *win, INTBIG index)
{
	(*us_editortable[us_currenteditor].highlightline)(win, index);
}

/*
 * dispatch routine to stop the graphic display of changes (for batching)
 */
void us_suspendgraphics(WINDOW *win)
{
	(*us_editortable[us_currenteditor].suspendgraphics)(win);
}

/*
 * dispatch routine to restart the graphic display of changes and redisplay (for batching)
 */
void us_resumegraphics(WINDOW *win)
{
	(*us_editortable[us_currenteditor].resumegraphics)(win);
}

/*
 * dispatch routine to write the text file to "file"
 */
void us_writetextfile(WINDOW *win, char *file)
{
	(*us_editortable[us_currenteditor].writetextfile)(win, file);
}

/*
 * dispatch routine to read the text file "file"
 */
void us_readtextfile(WINDOW *win, char *file)
{
	(*us_editortable[us_currenteditor].readtextfile)(win, file);
}

/*
 * dispatch routine to get the next character
 */
void us_editorterm(WINDOW *w)
{
	(*us_editortable[us_currenteditor].editorterm)(w);
}

/*
 * dispatch routine to force changes from the editor in window "w"
 */
void us_shipchanges(WINDOW *w)
{
	(*us_editortable[us_currenteditor].shipchanges)(w);
}

/*
 * dispatch routine to get the next character
 */
INTSML us_gotchar(WINDOW *w, INTSML i)
{
	return((*us_editortable[us_currenteditor].gotchar)(w, i));
}

/*
 * dispatch routine to cut text
 */
void us_cuttext(WINDOW *w)
{
	(*us_editortable[us_currenteditor].cut)(w);
}

/*
 * dispatch routine to copy text
 */
void us_copytext(WINDOW *w)
{
	(*us_editortable[us_currenteditor].copy)(w);
}

/*
 * dispatch routine to paste text
 */
void us_pastetext(WINDOW *w)
{
	(*us_editortable[us_currenteditor].paste)(w);
}

/*
 * dispatch routine to undo text changes
 */
void us_undotext(WINDOW *w)
{
	(*us_editortable[us_currenteditor].undo)(w);
}

/*
 * dispatch routine to search for text
 */
void us_searchtext(WINDOW *w, char *str, INTSML fromtop)
{
	(*us_editortable[us_currenteditor].search)(w, str, fromtop);
}

/*
 * support routine to allocate a new editor from the pool (if any) or memory
 * routine returns NOEDITOR upon error
 */
EDITOR *us_alloceditor(void)
{
	REGISTER EDITOR *e;

	if (us_editorfree == NOEDITOR)
	{
		e = (EDITOR *)emalloc((sizeof (EDITOR)), us_aid->cluster);
		if (e == 0) return(NOEDITOR);
		e->state = 0;
	} else
	{
		/* take editor from free list */
		e = us_editorfree;
		us_editorfree = (EDITOR *)e->nexteditor;
	}
	e->nexteditor = NOEDITOR;
	e->editobjvar = NOVARIABLE;
	return(e);
}

/*
 * support routine to return editor "e" to the pool of free editors
 */
void us_freeeditor(EDITOR *e)
{
	if (e == 0 || e == NOEDITOR) return;
	efree(e->header);
	e->nexteditor = us_editorfree;
	us_editorfree = e;
}

/******************** SPECIAL WINDOW HANDLERS ********************/

/*
 * routine to accept changes in an edit window examining a variable.  If "nature" is:
 *  REPLACETEXTLINE  line "changed" goes from "oldline" to "newline"
 *  DELETETEXTLINE   line "changed deleted (was "oldline")
 *  INSERTTEXTLINE   line "newline" inserted before line "changed"
 *  REPLACEALLTEXT   "changed" lines "newline" replace all text
 */
void us_varchanges(WINDOW *w, INTSML nature, char *oldline, char *newline, INTBIG changed)
{
	REGISTER INTSML j, l, save, res;
	REGISTER INTBIG newval, i, len;
	REGISTER char **newlist;
	float newfloat;
	REGISTER EDITOR *ed;

	ed = w->editor;
	if (ed->editobjvar == NOVARIABLE) return;
	if ((ed->editobjvar->type&VCANTSET) != 0)
	{
		ttyputerr("This variable cannot be changed");
		ed->editobjvar = NOVARIABLE;
		return;
	}
	len = getlength(ed->editobjvar);

	/* when replacing the entire text, reduce to individual calls */
	if (nature == REPLACEALLTEXT)
	{
		newlist = (char **)newline;
		for(i=0; i<changed; i++)
			us_varchanges(w, REPLACETEXTLINE, "", newlist[i], i);
		for(i=len-1; i>=changed; i--)
			us_varchanges(w, DELETETEXTLINE, "", "", i);
		return;
	}

	if (nature == DELETETEXTLINE && len == 1)
	{
		/* delete of last entry: instead, replace it with a null */
		newline = "";
		nature = REPLACETEXTLINE;
	} else if (nature == REPLACETEXTLINE && changed >= len)
	{
		/* change of line beyond end: instead make an insert */
		nature = INSERTTEXTLINE;
	}

	/* get the value */
	if (nature == REPLACETEXTLINE || nature == INSERTTEXTLINE)
		switch (ed->editobjvar->type&VTYPE)
	{
		case VINTEGER:
		case VSHORT:
		case VADDRESS:
			newval = myatoi(newline);
			break;
		case VFRACT:
			newval = atofr(newline);
			break;
		case VFLOAT:
		case VDOUBLE:
			newfloat = atof(newline);
			newval = castint(newfloat);
			break;
		case VSTRING:
			j = l = strlen(newline);
			if (strcmp(&newline[l-3], " */") == 0)
			{
				for(j = l-5; j >= 0; j--)
					if (strncmp(&newline[j], "/* ", 3) == 0) break;
				while (j > 0 && newline[j-1] == ' ') j--;
				if (j < 0) j = l;
			}
			save = newline[j];
			newline[j] = 0;
			newval = (INTBIG)newline;
			break;
		default:
			ttyputmsg("Cannot update this type of variable (0%o)", ed->editobjvar->type);
			break;
	}

	/* make the change */
	res = 1;
	switch (nature)
	{
		case REPLACETEXTLINE:
			if (changed < 0 || changed >= len) return;
			res = setind((INTBIG)ed->editobjaddr, ed->editobjtype, ed->editobjqual, changed, newval);
			break;
		case DELETETEXTLINE:
			if (changed < 0 || changed >= len) return;
			res = delind((INTBIG)ed->editobjaddr, ed->editobjtype, ed->editobjqual, changed);
			break;
		case INSERTTEXTLINE:
			if (changed <= 0 || changed > len) return;
			res = insind((INTBIG)ed->editobjaddr, ed->editobjtype, ed->editobjqual, changed, newval);

			/* why is this next line necessary? */
			ed->editobjvar = getval((INTBIG)ed->editobjaddr, ed->editobjtype, -1, ed->editobjqual);
			break;
	}

	/* clean-up string if one was passed */
	if ((nature == REPLACETEXTLINE || nature == INSERTTEXTLINE) &&
		(ed->editobjvar->type&VTYPE) == VSTRING) newline[j] = save;

	if (res != 0)
	{
		ttyputerr("Error changing variable: ignoring further changes");
		ed->editobjvar = NOVARIABLE;
	}
}

/*
 * routine to terminate in an edit window examining a variable
 */
void us_varterm(WINDOW *w)
{
	REGISTER EDITOR *ed;
	REGISTER char *objname;

	/* save addresses of objects that must be deallocated */
	ed = w->editor;
	if (ed == NOEDITOR) return;
	objname = ed->editobjname;

	/* now do the real editor termination (this frees the "ed" object) */
	us_editorterm(w);

	/* deallocate */
	efree(objname);
}

/*
 * routine to accept changes in an edit window examining a textual facet.  If "nature" is:
 *  REPLACETEXTLINE  line "changed" goes from "oldline" to "newline"
 *  DELETETEXTLINE   line "changed" deleted (was "oldline")
 *  INSERTTEXTLINE   line "newline" inserted before line "changed"
 *  REPLACEALLTEXT   "changed" lines "newline" replace all text
 */
void us_textfacetchanges(WINDOW *w, INTSML nature, char *oldline, char *newline, INTBIG changed)
{
	REGISTER INTBIG len;
	REGISTER INTSML res;
	REGISTER EDITOR *ed;

	ed = w->editor;
	if (ed->editobjvar == NOVARIABLE) return;
	if ((ed->editobjvar->type&VTYPE) != VSTRING) return;
	if ((ed->editobjvar->type&VCANTSET) != 0) return;
	len = getlength(ed->editobjvar);

	if (nature == DELETETEXTLINE && len == 1)
	{
		/* delete of last line: instead, replace it with a blank */
		newline = "";
		nature = REPLACETEXTLINE;
	} else if (nature == REPLACETEXTLINE && changed >= len)
	{
		if (changed >= len)
		{
			/* change of line one beyond end: instead make an insert */
			nature = INSERTTEXTLINE;
		}
	}

	res = 1;
	switch (nature)
	{
		case REPLACETEXTLINE:
			if (changed < 0 || changed >= len) return;
			res = setindkey((INTBIG)ed->editobjaddr, VNODEPROTO, el_facet_message,
				changed, (INTBIG)newline);
			break;
		case DELETETEXTLINE:
			if (changed < 0 || changed >= len) return;
			res = delindkey((INTBIG)ed->editobjaddr, VNODEPROTO, el_facet_message,
				changed);
			break;
		case INSERTTEXTLINE:
			if (changed <= 0 || changed > len) return;
			res = insindkey((INTBIG)ed->editobjaddr, VNODEPROTO, el_facet_message,
				changed, (INTBIG)newline);
			break;
		case REPLACEALLTEXT:
			if (setvalkey((INTBIG)ed->editobjaddr, VNODEPROTO, el_facet_message,
				(INTBIG)newline, VSTRING | VISARRAY | (changed << VLENGTHSH)) != NOVARIABLE)
					res = 0;
			break;
	}

	if (res != 0)
	{
		ttyputerr("Error changing variable: ignoring further changes");
		ed->editobjvar = NOVARIABLE;
	}
}

/*
 * private character handler for the text window.  This routine normally
 * passes all commands to the editor's character handler.  However, it
 * interprets M(=) which is for editing the facet on the current line
 */
INTSML us_facetedithandler(WINDOW *w, INTSML ch)
{
	char *newpar[2], *str;
	REGISTER INTSML i, meta;
	REGISTER EDITOR *ed;
	extern INTSML us_lastemacschar;

	/* the EMACS text editor must be running */
	us_describeeditor(&str);
	if (namesame(str, "emacs") != 0) return(us_gotchar(w, ch));

	/* see if the meta key is held down (serious black magic) */
	meta = 0;
	if ((ch&0200) != 0) meta = 1;
	if ((us_lastemacschar&2) != 0) meta = 1;

	/* pass character on to the editor if not M(=) */
	if (meta == 0 || (ch&0177) != '=') return(us_gotchar(w, ch));

	/* M(=) typed: parse current line to edit named facet */
	ed = w->editor;
	(void)allocstring(&str, us_getline(w, ed->curline), el_tempcluster);

	/* first drop everything past the first space character */
	for(i=0; str[i] != 0; i++) if (str[i] == ' ') break;
	if (str[i] != 0) str[i] = 0;

	if (str[0] == 0) ttyputerr("No facet specified on this line"); else
	{
		/* issue the "editfacet" command */
		newpar[0] = "editfacet";
		newpar[1] = str;
		(void)tellaid(us_aid, 2, newpar);
		setactivity("FACET SELECTION");
	}

	/* clean up */
	efree(str);
	return(0);
}

/******************** COMMAND SUPPORT ********************/

INTSML us_demandxy(INTBIG *x, INTBIG *y)
{
	INTSML ret;
	ret = getxy(x, y);
	if (ret != 0) ttyputmsgf("Cursor must be in an editing window");
	return(ret);
}

static INTBIG us_curx, us_cury;

/*
 * routine to get the co-ordinates of the cursor into the reference parameters
 * "x" and "y".  If "GOTXY" is set in the global variable "us_state" then
 * this has already been done.  The routine returns nonzero if there is not a
 * valid cursor position.
 */
INTSML getxy(INTBIG *x, INTBIG *y)
{
	INTSML gx, gy;
	REGISTER INTSML ret;

	if ((us_state&GOTXY) == 0)
	{
		readtablet(&gx, &gy);
		ret = us_setxy(gx, gy);
	} else ret = 0;
	*x = us_curx;
	*y = us_cury;
	return(ret);
}

/*
 * routine to take the values (realx, realy) from the tablet and store
 * them in the variables (us_curx, us_cury) which are in design-space
 * co-ordinates.  "GOTXY" in the global variable "us_state" is set to indicate
 * that the co-ordinates are valid.  The current window is set according
 * to the cursor position.  The routine returns nonzero if the position
 * is not in a window.
 */
INTSML us_setxy(INTSML x, INTSML y)
{
	REGISTER WINDOW *w;
	REGISTER void *curframe;

	us_curx = x;   us_cury = y;

	/* figure out which window it is in */
	curframe = us_getwindowframe();
	for(w = el_topwindow; w != NOWINDOW; w = w->nextwindow)
	{
		if (w->frame != curframe) continue;
		if (x >= w->uselx && x <= w->usehx && y >= w->usely && y <= w->usehy)
		{
			/* make this window the current one */
			if (w != el_curwindow)
			{
				(void)setvalkey((INTBIG)us_aid, VAID, us_current_window, (INTBIG)w, VWINDOW|VDONTSAVE);
				(void)setval((INTBIG)el_curlib, VLIBRARY, "curnodeproto",
					(INTBIG)w->curnodeproto, VNODEPROTO);
			}
			us_scaletowindow(&us_curx, &us_cury, w);
			us_state |= GOTXY;
			return(0);
		}
	}
	return(1);
}

/*
 * routine to force the parameters "xcur" and "ycur" to align to the
 * nearest "alignment" units
 */
void gridalign(INTBIG *xcur, INTBIG *ycur, INTBIG alignment)
{
	INTBIG otheralign;
	REGISTER INTBIG val;

	val = us_alignvalue(*xcur, alignment, &otheralign);
	if (abs(*xcur-val) < abs(*xcur-otheralign)) *xcur = val; else
		*xcur = otheralign;
	val = us_alignvalue(*ycur, alignment, &otheralign);
	if (abs(*ycur-val) < abs(*ycur-otheralign)) *ycur = val; else
		*ycur = otheralign;
}

/*
 * routine to return "value", aligned to the nearest "alignment" units.
 * The next closest alignment value (if "value" is not on the grid)
 * is returned in "otheralign".
 */
INTBIG us_alignvalue(INTBIG value, INTBIG alignment, INTBIG *otheralign)
{
	REGISTER INTBIG i, v1, v2;
	REGISTER INTSML sign;

	/* determine the sign of the value */
	if (value < 0) { sign = -1; value = -value; } else sign = 1;

	/* compute the two aligned values */
	if (alignment == 0) v1 = value; else
		v1 = value / alignment * alignment;
	if (v1 == value) v2 = value; else v2 = v1 + alignment;
	v1 *= sign;   v2 *= sign;

	/* make sure "v1" is the closest aligned value */
	if (abs(v1-value) > abs(v2-value)) { i = v1;   v1 = v2;   v2 = i; }

	*otheralign = v2;
	return(v1);
}

/* routine to ensure that a current window exists */
INTSML us_needwindow(void)
{
	if (el_curwindow != NOWINDOW) return(0);
	us_abortcommand("No current window");
	return(1);
}

/* routine to ensure that a facet exists in the current window */
NODEPROTO *us_needfacet(void)
{
	REGISTER NODEPROTO *np;

	np = getcurfacet();
	if (np != NONODEPROTO) return(np);
	if (el_curwindow == NOWINDOW)
	{
		us_abortcommand("No current window (create one first)");
	} else
	{
		if ((us_aid->aidstate&NODETAILS) != 0)
			us_abortcommand("No facet in this window (create one first)"); else
				us_abortcommand("No facet in this window (use '-editfacet' to create one)");
	}
	return(NONODEPROTO);
}

/*
 * routine to ensure that NODEPROTO "np" can be edited (prints error and returns
 * nonzero if it cannot).  If NONODEPROTO is given, facet lock is checked
 */
INTSML us_protolocked(NODEPROTO *np)
{
	/* if a prototype is specified, check it */
	if (np != NONODEPROTO)
	{
		if (np->index != 0)
		{
			/* see if a primitive is locked */
			if ((np->userbits&LOCKEDPRIM) == 0 || (us_aid->aidstate&NOPRIMCHANGES) == 0) return(0);
			us_abortcommand("Locked primitive changes are currently disallowed");
			if ((us_aid->aidstate&NODETAILS) == 0)
				ttyputmsg("Use 'defnode * locked-primitives' to restore editing freedom");
			return(1);
		} else
		{
			/* see if facets are locked */
			if ((us_aid->aidstate&NOFACETCHANGES) == 0) return(0);
			us_abortcommand("Facet changes are currently disallowed");
			if ((us_aid->aidstate&NODETAILS) == 0)
				ttyputmsg("Use 'defnode * alterable' to restore editing freedom");
			return(1);
		}
	}

	/* no specific NODEPROTO mentioned: check facet lock */
	if ((us_aid->aidstate&NOFACETCHANGES) == 0) return(0);
	us_abortcommand("Changes are currently disallowed");
	return(1);
}

/*
 * routine to determine whether NODEPROTO "np" can be edited.
 * returns nonzero if not.
 */
INTSML us_islocked(NODEPROTO *np)
{
	if (np->index != 0)
	{
		/* see if a primitive is locked */
		if ((np->userbits&LOCKEDPRIM) == 0 || (us_aid->aidstate&NOPRIMCHANGES) == 0) return(0);
		return(1);
	}

	/* see if facets are locked */
	if ((us_aid->aidstate&NOFACETCHANGES) == 0) return(0);
	return(1);
}

/*
 * routine to determine the proper position of the cursor given that
 * it must adjust to the nearest "angle" tenth-degree radial coming out of
 * the point (tx,ty) and that it is currently at (nx, ny).  The
 * adjusted point is placed into (fx, fy) and the proper radial starting
 * point in "poly" is placed in (tx, ty).
 */
void us_getslide(INTSML angle, INTBIG tx, INTBIG ty, INTBIG nx, INTBIG ny, INTBIG *fx,
	INTBIG *fy)
{
	REGISTER INTSML ang;
	INTBIG ix, iy;

	/* if angle is unconstrained, use the exact cursor position */
	if (angle <= 0)
	{
		*fx = nx;   *fy = ny;
		return;
	}

	/* check all permissable angles */
	for(ang = 0; ang < 3600; ang += angle)
	{
		/* get close point to (nx,ny) on "ang" tenth-degree radial from (tx,ty) */
		(void)intersect(tx, ty, ang, nx, ny, (ang+900)%3600, &ix, &iy);

		/* accumulate the intersection closest to the cursor */
		if (ang != 0 && abs(*fx-nx) + abs(*fy-ny) < abs(ix-nx) + abs(iy-ny)) continue;
		*fx = ix;   *fy = iy;
	}
}

/*
 * routine to convert command interpreter letter "letter" to the full
 * variable name on the user aid object
 */
char *us_commandvarname(INTSML letter)
{
	static char varname[20];

	if (isupper(letter))
	{
		(void)strcpy(varname, "USER_local_capX");
		varname[14] = tolower(letter);
	} else
	{
		(void)strcpy(varname, "USER_local_X");
		varname[11] = letter;
	}
	return(varname);
}

/*
 * routine to parse the variable path in "str" and return the object address
 * and type on which this variable resides along with the variable name.
 * The object address and type are placed in "objaddr" and "objtype";
 * the variable name is placed in "varname".  "comvar" is set to nonzero
 * if the variable is a command-interpreter variable (as opposed to a
 * database variable).  If an array index specification is given (a "[]")
 * then the index value is returned in "index" (otherwise the value is set
 * to -1).  The routine returns nonzero if the variable path is invalid.
 */
INTSML us_getvar(char *str, INTBIG *objaddr, INTBIG *objtype, char **varname,
	INTSML *comvar, INTSML *index)
{
	REGISTER INTSML i;
	static char fullvarname[50];

	/* see if an array index is specified */
	*index = -1;
	i = strlen(str);
	if (str[i-1] == ']') for(i--; i >= 0; i--) if (str[i] == '[')
	{
		*index = myatoi(&str[i+1]);
		break;
	}

	/* see if this is a command interpreter variable */
	*comvar = 0;
	if (str[1] == 0 || str[1] == '[')
	{
		if (str[0] >= 'a' && str[0] <= 'z') *comvar = 1;
		if (str[0] >= 'A' && str[0] <= 'Z') *comvar = 1;
	}

	/* replace the actual name for command interpreter variables */
	if (*comvar != 0)
	{
		(void)sprintf(fullvarname, "aid:user.%s", us_commandvarname(str[0]));
		str = fullvarname;
	}

	/* pick apart the variable path */
	return(us_evaluatevariable(str, objaddr, objtype, varname));
}

void us_adjustlambda(INTBIG oldlam, INTBIG newlam)
{
	REGISTER WINDOW *w;
	REGISTER INTBIG i;

	i = muldiv(us_alignment, newlam, oldlam);
	(void)setvalkey((INTBIG)us_aid, VAID, us_alignment_obj, i, VINTEGER|VDONTSAVE);
	i = muldiv(us_edgealignment, newlam, oldlam);
	(void)setvalkey((INTBIG)us_aid, VAID, us_alignment_edge, i, VINTEGER|VDONTSAVE);
	for(w = el_topwindow; w != NOWINDOW; w = w->nextwindow)
	{
		(void)setval((INTBIG)w, VWINDOW, "gridx", muldiv(w->gridx, newlam, oldlam), VINTEGER);
		(void)setval((INTBIG)w, VWINDOW, "gridy", muldiv(w->gridy, newlam, oldlam), VINTEGER);
	}
}

/*
 * routine to determine whether the command bound to key "key" is the
 * last instance of the "tellaid user" command that is bound to a key.
 * This is important to know because if the last "tellaid user" is unbound,
 * there is no way to execute any long commands!
 */
INTSML us_islasteval(INTSML key)
{
	REGISTER INTSML j, keytotal, retval, foundanother;
	REGISTER VARIABLE *var;
	COMMANDBINDING commandbindingthis, commandbindingother;

	/* get the command on this key */
	retval = 0;
	var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_binding_keys);
	if (var == NOVARIABLE) return(0);
	us_parsebinding(((char **)var->addr)[key], &commandbindingthis);
	if (*commandbindingthis.command != 0)
	{
		/* see if it is "tellaid user" */
		if (namesame(commandbindingthis.command, "tellaid user") == 0)
		{
			/* this is "tellaid user"...check all other keys for this command */
			keytotal = getlength(var);
			foundanother = 0;
			for(j=0; j<keytotal; j++)
			{
				if (j == key) continue;
				us_parsebinding(((char **)var->addr)[j], &commandbindingother);
				if (*commandbindingother.command != 0 &&
					namesame(commandbindingother.command, "tellaid user") == 0) foundanother++;
				us_freebindingparse(&commandbindingother);
				if (foundanother != 0) break;
			}
			if (foundanother == 0) retval = 1;
		}
	}
	us_freebindingparse(&commandbindingthis);
	return(retval);
}

/*
 * routine to set the trace information in the "size" coordinate pairs in
 * "newlist" onto the node "ni".
 */
void us_settrace(NODEINST *ni, INTBIG *newlist, INTSML size)
{
	REGISTER INTBIG lx, hx, ly, hy, x, y;
	REGISTER INTSML i;
	INTBIG lxo, hxo, lyo, hyo;

	/* get the extent of the data */
	lx = hx = newlist[0];   ly = hy = newlist[1];
	for(i=1; i<size; i++)
	{
		x = newlist[i*2];
		y = newlist[i*2+1];
		lx = mini(lx, x);   hx = maxi(hx, x);
		ly = mini(ly, y);   hy = maxi(hy, y);
	}

	/* make these co-ordinates relative to the center */
	x = (hx+lx) / 2;   y = (hy+ly) / 2;
	for(i=0; i<size; i++)
	{
		newlist[i*2] = newlist[i*2] - x;
		newlist[i*2+1] = newlist[i*2+1] - y;
	}

	/* adjust size for node size offset */
	nodesizeoffset(ni->proto, &lxo, &lyo, &hxo, &hyo);
	lx -= lxo;   hx += hxo;
	ly -= lyo;   hy += hyo;

	/* erase the node instance */
	startobjectchange((INTBIG)ni, VNODEINST);

	/* change the trace data */
	(void)setvalkey((INTBIG)ni, VNODEINST, el_trace, (INTBIG)newlist,
		VINTEGER|VISARRAY|((size*2)<<VLENGTHSH));

	/* scale the node (which redraws too) */
	if (ni->proto->index != 0 && (lx != ni->lowx || hx != ni->highx || ly != ni->lowy ||
		hy != ni->highy || ni->rotation != 0 || ni->transpose != 0))
			modifynodeinst(ni, lx-ni->lowx, ly-ni->lowy, hx-ni->highx, hy-ni->highy,
				(INTSML)(-ni->rotation), ni->transpose);

	/* redisplay */
	endobjectchange((INTBIG)ni, VNODEINST);
}

/*
 * routine to scale the trace information on node "ni" given that it will change
 * in size to "nlx"->"nhx" and "nly"->"nhy".
 */
void us_scaletraceinfo(NODEINST *ni, INTBIG nlx, INTBIG nhx, INTBIG nly, INTBIG nhy)
{
	REGISTER VARIABLE *var;
	REGISTER INTBIG *newlist, oldx, oldy;
	REGISTER INTSML len, i;

	/* stop now if no trace information */
	var = gettrace(ni);
	if (var == NOVARIABLE) return;

	/* get new array for new trace */
	len = getlength(var);
	newlist = (INTBIG *)emalloc(len * SIZEOFINTBIG, el_tempcluster);
	if (newlist == 0) return;

	/* copy the data and scale it */
	for(i=0; i<len; i += 2)
	{
		oldx = ((INTBIG *)var->addr)[i];
		oldy = ((INTBIG *)var->addr)[i+1];
		oldx = muldiv(oldx, nhx - nlx, ni->highx - ni->lowx);
		oldy = muldiv(oldy, nhy - nly, ni->highy - ni->lowy);
		newlist[i] = oldx;   newlist[i+1] = oldy;
	}

	/* store the new list */
	(void)setvalkey((INTBIG)ni, VNODEINST, el_trace, (INTBIG)newlist,
		VINTEGER|VISARRAY|(len<<VLENGTHSH));
	efree((char *)newlist);
}

/*
 * Routine to fillet the two highlighted objects
 */
void us_dofillet(void)
{
	REGISTER VARIABLE *var;
	HIGHLIGHT thishigh, otherhigh;
	REGISTER NODEINST *ni1, *ni2, *swapni;
	double startoffset, endangle, srot, erot, newangle, dx, dy;
	INTBIG ix, iy, ix1, iy1, ix2, iy2;
	REGISTER INTSML ang1, ang2, size1, size2, arc1, arc2, swapsize, icount, on1, on2, newrot;
	REGISTER INTBIG *newlist1, *newlist2, *line1xs, *line1ys, *line1xe, *line1ye,
		*line2xs, *line2ys, *line2xe, *line2ye, x, y, i, *swaplist;
	XARRAY trans;

	/* must be exactly two nodes selected */
	var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_highlighted);
	if (var == NOVARIABLE)
	{
		us_abortcommand("Must select two nodes before filleting");
		return;
	}
	if (getlength(var) != 2)
	{
		us_abortcommand("Must select two nodes before filleting");
		return;
	}

	if (us_makehighlight(((char **)var->addr)[0], &thishigh) != 0 ||
		us_makehighlight(((char **)var->addr)[1], &otherhigh) != 0) return;

	/* get the two objects */
	if ((thishigh.status&HIGHTYPE) != HIGHFROM || (otherhigh.status&HIGHTYPE) != HIGHFROM)
	{
		us_abortcommand("Must select two nodes before filleting");
		return;
	}

	if (thishigh.fromgeom->entrytype != OBJNODEINST || otherhigh.fromgeom->entrytype != OBJNODEINST)
	{
		us_abortcommand("Must select two nodes before filleting");
		return;
	}

	/* get description of first node */
	ni1 = thishigh.fromgeom->entryaddr.ni;
	if (ni1->proto == art_circleprim)
	{
		getarcdegrees(ni1, &startoffset, &endangle);
		if (startoffset == 0.0 && endangle == 0.0)
		{
			us_abortcommand("Must select arcs, not circles before filleting");
			return;
		}
		newlist1 = emalloc((6*SIZEOFINTBIG), el_tempcluster);
		if (newlist1 == 0) return;
		newlist1[0] = (ni1->lowx + ni1->highx) / 2;
		newlist1[1] = (ni1->lowy + ni1->highy) / 2;
		getarcendpoints(ni1, startoffset, endangle, &newlist1[2], &newlist1[3],
			&newlist1[4], &newlist1[5]);
		arc1 = 1;
	} else if (ni1->proto == art_openedpolygonprim || ni1->proto == art_openeddottedpolygonprim ||
		ni1->proto == art_openeddashedpolygonprim || ni1->proto == art_openedfardottepolygonprim ||
		ni1->proto == art_closedpolygonprim)
	{
		var = gettrace(ni1);
		if (var == NOVARIABLE)
		{
			us_abortcommand("Must select nodes with trace information before filleting");
			return;
		}

		/* transform the traces */
		size1 = getlength(var) / 2;
		newlist1 = emalloc((size1*2*SIZEOFINTBIG), el_tempcluster);
		if (newlist1 == 0) return;
		makerot(ni1, trans);
		x = (ni1->highx + ni1->lowx) / 2;
		y = (ni1->highy + ni1->lowy) / 2;
		for(i=0; i<size1; i++)
			xform(((INTBIG *)var->addr)[i*2]+x, ((INTBIG *)var->addr)[i*2+1]+y, &newlist1[i*2],
				&newlist1[i*2+1], trans);
		arc1 = 0;
	} else
	{
		us_abortcommand("Node %s cannot be filleted", describenodeinst(ni1));
		return;
	}

	/* get description of second node */
	ni2 = otherhigh.fromgeom->entryaddr.ni;
	if (ni2->proto == art_circleprim)
	{
		getarcdegrees(ni2, &startoffset, &endangle);
		if (startoffset == 0.0 && endangle == 0.0)
		{
			us_abortcommand("Must select arcs, not circles before filleting");
			return;
		}
		newlist2 = emalloc((6*SIZEOFINTBIG), el_tempcluster);
		if (newlist2 == 0) return;
		newlist2[0] = (ni2->lowx + ni2->highx) / 2;
		newlist2[1] = (ni2->lowy + ni2->highy) / 2;
		getarcendpoints(ni2, startoffset, endangle, &newlist2[2], &newlist2[3],
			&newlist2[4], &newlist2[5]);
		arc2 = 1;
	} else if (ni2->proto == art_openedpolygonprim || ni2->proto == art_openeddottedpolygonprim ||
		ni2->proto == art_openeddashedpolygonprim || ni2->proto == art_openedfardottepolygonprim ||
		ni2->proto == art_closedpolygonprim)
	{
		var = gettrace(ni2);
		if (var == NOVARIABLE)
		{
			us_abortcommand("Must select nodes with trace information before filleting");
			return;
		}

		/* transform the traces */
		size2 = getlength(var) / 2;
		newlist2 = emalloc((size2*2*SIZEOFINTBIG), el_tempcluster);
		if (newlist2 == 0) return;
		makerot(ni2, trans);
		x = (ni2->highx + ni2->lowx) / 2;
		y = (ni2->highy + ni2->lowy) / 2;
		for(i=0; i<size2; i++)
			xform(((INTBIG *)var->addr)[i*2]+x, ((INTBIG *)var->addr)[i*2+1]+y, &newlist2[i*2],
				&newlist2[i*2+1], trans);
		arc2 = 0;
	} else
	{
		us_abortcommand("Node %s cannot be filleted", describenodeinst(ni2));
		return;
	}

	/* handle different types of filleting */
	if (arc1 != 0 && arc2 != 0)
	{
		/* cannot handle arc-to-arc filleting */
		us_abortcommand("Cannot fillet two curves");
	} else if (arc1 == 0 && arc2 == 0)
	{
		/* handle line-to-line filleting: find out which endpoints are closest */
		if (computedistance(x,y, newlist1[0],newlist1[1]) <
			computedistance(x,y, newlist1[size1*2-2],newlist1[size1*2-1]))
		{
			line1xs = &newlist1[0];
			line1ys = &newlist1[1];
			line1xe = &newlist1[2];
			line1ye = &newlist1[3];
		} else
		{
			line1xs = &newlist1[size1*2-2];
			line1ys = &newlist1[size1*2-1];
			line1xe = &newlist1[size1*2-4];
			line1ye = &newlist1[size1*2-3];
		}
		if (computedistance(*line1xs,*line1ys, newlist2[0],newlist2[1]) <
			computedistance(*line1xs,*line1ys, newlist2[size2*2-2],newlist2[size2*2-1]))
		{
			line2xs = &newlist2[0];
			line2ys = &newlist2[1];
			line2xe = &newlist2[2];
			line2ye = &newlist2[3];
		} else
		{
			line2xs = &newlist2[size2*2-2];
			line2ys = &newlist2[size2*2-1];
			line2xe = &newlist2[size2*2-4];
			line2ye = &newlist2[size2*2-3];
		}

		/* compute intersection point */
		ang1 = figureangle(*line1xs, *line1ys, *line1xe, *line1ye);
		ang2 = figureangle(*line2xs, *line2ys, *line2xe, *line2ye);
		if (intersect(*line1xs, *line1ys, ang1, *line2xs, *line2ys, ang2, &ix, &iy) != 0)
			us_abortcommand("Lines do not intersect"); else
		{
			*line1xs = ix;   *line1ys = iy;
			*line2xs = ix;   *line2ys = iy;
			us_pushhighlight();
			us_clearhighlightcount();
			us_settrace(ni1, newlist1, (INTSML)size1);
			us_settrace(ni2, newlist2, (INTSML)size2);
			(void)us_pophighlight(1);
		}
	} else
	{
		/* handle arc-to-line filleting */
		if (arc1 == 0)
		{
			swaplist = newlist1;   newlist1 = newlist2;   newlist2 = swaplist;
			swapsize = size1;      size1 = size2;         size2 = swapsize;
			swapni = ni1;          ni1 = ni2;             ni2 = swapni;
		}

		/* "newlist1" describes the arc, "newlist2" describes the line */
		if (computedistance(newlist1[0],newlist1[1], newlist2[0],newlist2[1]) <
			computedistance(newlist1[0],newlist1[1], newlist2[size2*2-2],newlist2[size2*2-1]))
		{
			line2xs = &newlist2[0];
			line2ys = &newlist2[1];
			line2xe = &newlist2[2];
			line2ye = &newlist2[3];
		} else
		{
			line2xs = &newlist2[size2*2-2];
			line2ys = &newlist2[size2*2-1];
			line2xe = &newlist2[size2*2-4];
			line2ye = &newlist2[size2*2-3];
		}
		icount = circlelineintersection(newlist1[0],newlist1[1], newlist1[2],newlist1[3],
			*line2xs, *line2ys, *line2xe, *line2ye, &ix1, &iy1, &ix2, &iy2, 0);
		if (icount == 0)
		{
			us_abortcommand("Line does not intersect arc: cannot fillet");
		} else
		{
			if (icount == 2)
			{
				on1 = us_pointonexparc(newlist1[0],newlist1[1], newlist1[2],newlist1[3],
					newlist1[4],newlist1[5], ix1, iy1);
				on2 = us_pointonexparc(newlist1[0],newlist1[1], newlist1[2],newlist1[3],
					newlist1[4],newlist1[5], ix2, iy2);
				if (on1 == 0 && on2 != 0)
				{
					icount = 1;
					ix1 = ix2;   iy1 = iy2;
				} else if (on1 != 0 && on2 == 0)
				{
					icount = 1;
				}
			}
			if (icount == 2)
			{
				x = (*line2xs + *line2xe) / 2;
				y = (*line2ys + *line2ye) / 2;
				if (computedistance(ix1,iy1, x,y) > computedistance(ix2,iy2, x,y))
				{
					ix1 = ix2;   iy1 = iy2;
				}
			}

			/* make them fillet at (ix1,iy1) */
			us_pushhighlight();
			us_clearhighlightcount();

			/* adjust the arc (node ni1) */
			dx = (double)(newlist1[2]-newlist1[0]);   dy = (double)(newlist1[3]-newlist1[1]);
			if (dx == 0.0 && dy == 0.0)
			{
				us_abortcommand("Domain error during fillet");
				return;
			}
			srot = atan2(dy, dx);
			if (srot < 0.0) srot += EPI*2.0;

			dx = (double)(newlist1[4]-newlist1[0]);   dy = (double)(newlist1[5]-newlist1[1]);
			if (dx == 0.0 && dy == 0.0)
			{
				us_abortcommand("Domain error during fillet");
				return;
			}
			erot = atan2(dy, dx);
			if (erot < 0.0) erot += EPI*2.0;

			dx = (double)(ix1-newlist1[0]);   dy = (double)(iy1-newlist1[1]);
			if (dx == 0.0 && dy == 0.0)
			{
				us_abortcommand("Domain error during fillet");
				return;
			}
			newangle = atan2(dy, dx);
			if (newangle < 0.0) newangle += EPI*2.0;
			if (computedistance(ix1,iy1, newlist1[2],newlist1[3]) <
				computedistance(ix1,iy1, newlist1[4],newlist1[5])) srot = newangle; else
					erot = newangle;
			erot -= srot;
			if (erot < 0.0) erot += EPI*2.0;
			newrot = rounddouble(srot * 1800.0 / EPI);
			srot -= ((double)newrot) * EPI / 1800.0;
			startobjectchange((INTBIG)ni1, VNODEINST);
			modifynodeinst(ni1, 0, 0, 0, 0, (INTSML)(newrot - ni1->rotation), 0);
			setarcdegrees(ni1, srot, erot);
			endobjectchange((INTBIG)ni1, VNODEINST);

			/* adjust the line (node ni2) */
			*line2xs = ix1;   *line2ys = iy1;
			us_settrace(ni2, newlist2, (INTSML)size2);

			/* restore highlighting */
			(void)us_pophighlight(1);
		}
	}
	efree((char *)newlist1);
	efree((char *)newlist2);
}

/*
 * Routine to determine whether the point (x,y) is on the arc centered at (cx,cy), starting
 * at (sx,sy), and ending at (ex,ey).  Returns nonzero if on the arc.
 */
INTSML us_pointonexparc(INTBIG cx, INTBIG cy, INTBIG sx, INTBIG sy, INTBIG ex, INTBIG ey, INTBIG x, INTBIG y)
{
	REGISTER INTSML as, ae, a;

	as = figureangle(cx, cy, sx, sy);
	ae = figureangle(cx, cy, ex, ey);
	a = figureangle(cx, cy, x, y);

	if (ae > as)
	{
		if (a >= as && a <= ae) return(1);
	} else
	{
		if (a >= as || a <= ae) return(1);
	}
	return(0);
}

/*
 * routine to recursively check sub-facet revision times
 * P. Attfield
 */
void us_check_facet_date(NODEPROTO *np, UINTBIG rev_time)
{
	REGISTER NODEPROTO *np2;
	REGISTER NODEINST *ni;

	for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		np2 = ni->proto;
		if (np2->index != 0) continue; /* ignore if primitive */
		if (np2->temp1 != 0) continue; /* ignore if already seen */
		us_check_facet_date(np2, rev_time); /* recurse */
	}

	/* check this facet */
	np->temp1++; /* flag that we have seen this one */
	if (np->revisiondate <= rev_time) return;

	/* possible error in hierarchy */
	ttyputerr("WARNING: sub-facet '%s' has been edited", describenodeproto(np));
	ttyputmsg("         since the last revision to the current facet");
}

/*
 * routine to switch to library "lib"
 */
void us_switchtolibrary(LIBRARY *lib)
{
	char *newpar[2];

	/* first remove any "getproto" menu entries to facets in lib */
	us_removegetfacets();

	/* next select the new library */
	us_clearhighlightcount();
	us_adjustlambda(el_curtech->deflambda, lib->lambda[el_curtech->index]);
	selectlibrary(lib);
	us_setlambda(0);
	if ((us_curnodeproto == NONODEPROTO || us_curnodeproto->index == 0) &&
		(us_state&NONPERSISTENTCURNODE) == 0)
			us_setnodeproto(el_curtech->firstnodeproto);

	/* edit the top facet of the new library if there is one */
	if (el_curlib->curnodeproto != NONODEPROTO)
	{
		newpar[0] = describenodeproto(el_curlib->curnodeproto);
		us_editfacet(1, newpar);
	}
}

/******************** TEXT OBJECTS ********************/

/*
 * routine to recompute the text descriptor in "descript" to change the
 * grab-point according to the location of (xcur, ycur), given that the
 * text centers at (xc, yc) and is "xw" by "yw" in size.  The new text
 * descriptor is returned.
 */
INTBIG us_figuregrabpoint(INTBIG descript, INTBIG xcur, INTBIG ycur, INTBIG xc,
	INTBIG yc, INTBIG xw, INTBIG yw)
{
	if (xcur < xc - xw/2)
	{
		/* grab-point is on the bottom left */
		if (ycur < yc - yw/2) return((descript & ~VTPOSITION) | VTPOSUPRIGHT);

		/* grab-point is on the top left */
		if (ycur > yc + yw/2) return((descript & ~VTPOSITION) | VTPOSDOWNRIGHT);

		/* grab-point is on the left */
		return((descript & ~VTPOSITION) | VTPOSRIGHT);
	}

	if (xcur > xc + xw/2)
	{
		/* grab-point is on the bottom right */
		if (ycur < yc - yw/2) return((descript & ~VTPOSITION) | VTPOSUPLEFT);

		/* grab-point is on the top right */
		if (ycur > yc + yw/2) return((descript & ~VTPOSITION) | VTPOSDOWNLEFT);

		/* grab-point is on the right */
		return((descript & ~VTPOSITION) | VTPOSLEFT);
	}

	/* grab-point is on the bottom */
	if (ycur < yc - yw/2) return((descript & ~VTPOSITION) | VTPOSUP);

	/* grab-point is on the top */
	if (ycur > yc + yw/2) return((descript & ~VTPOSITION) | VTPOSDOWN);

	/* grab-point is in the center: check for VERY center */
	if (ycur >= yc - yw/6 && ycur <= yc + yw/6 && xcur >= xc - xw/6 &&
		xcur <= xc + xw/6) return((descript & ~VTPOSITION) | VTPOSBOXED);

	/* grab-point is simply centered */
	return((descript & ~VTPOSITION) | VTPOSCENT);
}

/*
 * routine to return the new text descriptor field, given that the old is in
 * "formerdesc".  The instructions for changing this variable are in "count"
 * and "par", where "count" where the first two values are the X and Y offset,
 * and the third value is the text position.  Returns -1 on error.
 */
INTBIG us_figurevariableplace(INTBIG formerdesc, INTSML count, char *par[])
{
	INTBIG xval, yval;
	REGISTER INTBIG grab;

	if (count >= 2)
	{
		xval = atofr(par[0]) * 4 / WHOLE;
		yval = atofr(par[1]) * 4 / WHOLE;
		formerdesc = us_setdescriptoffset(formerdesc, xval, yval);
	}
	if (count >= 3)
	{
		grab = us_gettextposition(par[2]);
		if (grab < 0) return(-1);
		formerdesc = (formerdesc & ~VTPOSITION) | grab;
	}
	return(formerdesc);
}

/*
 * routine to change the X and Y offset factors in the text descriptor
 * "formerdesc" to "xval" and "yval".  Returns the new text descriptor.
 */
INTBIG us_setdescriptoffset(INTBIG formerdesc, INTBIG xval, INTBIG yval)
{
	/* make sure the range is proper */
	if (abs(xval) >= 512 || abs(yval) >= 512)
	{
		ttyputmsg("Text clipped to within 128 lambda of object");
		xval = mini(xval, 511);   xval = maxi(xval, -511);
		yval = mini(yval, 511);   yval = maxi(yval, -511);
	}

	formerdesc &= ~(VTXOFF|VTXOFFNEG|VTYOFF|VTYOFFNEG);
	formerdesc |= (abs(xval) << VTXOFFSH) & VTXOFF;
	if (xval < 0) formerdesc |= VTXOFFNEG;
	formerdesc |= (abs(yval) << VTYOFFSH) & VTYOFF;
	if (yval < 0) formerdesc |= VTYOFFNEG;
	return(formerdesc);
}

/*
 * routine to rotate the text descriptor in "descript" to account for
 * the rotation of the object on which it resides: "geom".  The new
 * text descriptor is returned.
 */
INTBIG us_rotatedescriptI(GEOM *geom, INTBIG descript)
{
	REGISTER INTBIG style;
	REGISTER NODEINST *ni;
	XARRAY trans;

	/* arcs do not rotate */
	if (geom->entrytype != OBJNODEINST) return(descript);

	switch (descript&VTPOSITION)
	{
		case VTPOSCENT:
		case VTPOSBOXED:     return(descript);
		case VTPOSUP:        style = TEXTBOT;       break;
		case VTPOSDOWN:      style = TEXTTOP;       break;
		case VTPOSLEFT:      style = TEXTRIGHT;     break;
		case VTPOSRIGHT:     style = TEXTLEFT;      break;
		case VTPOSUPLEFT:    style = TEXTBOTRIGHT;  break;
		case VTPOSUPRIGHT:   style = TEXTBOTLEFT;   break;
		case VTPOSDOWNLEFT:  style = TEXTTOPRIGHT;  break;
		case VTPOSDOWNRIGHT: style = TEXTTOPLEFT;   break;
	}
	ni = geom->entryaddr.ni;
	if (ni->transpose == 0) makeangle((INTSML)((3600 - ni->rotation)%3600), 0, trans); else
		makeangle(ni->rotation, ni->transpose, trans);
	style = rotatelabel((INTSML)style, trans);
	switch (style)
	{
		case TEXTBOT:       style = VTPOSUP;          break;
		case TEXTTOP:       style = VTPOSDOWN;        break;
		case TEXTRIGHT:     style = VTPOSLEFT;        break;
		case TEXTLEFT:      style = VTPOSRIGHT;       break;
		case TEXTBOTRIGHT:  style = VTPOSUPLEFT;      break;
		case TEXTBOTLEFT:   style = VTPOSUPRIGHT;     break;
		case TEXTTOPRIGHT:  style = VTPOSDOWNLEFT;    break;
		case TEXTTOPLEFT:   style = VTPOSDOWNRIGHT;   break;
	}
	return((descript & ~VTPOSITION) | style);
}

/*
 * routine to adjust the displayable text on node "ni" to account for new
 * size/rotation factors in the node.  This is only done for invisible pins
 * in the generic technology, where the displayable text must track the node.
 */
void us_adjustdisplayabletext(NODEINST *ni)
{
	REGISTER INTSML i;
	REGISTER INTBIG descript, halfx, halfy;
	REGISTER VARIABLE *var;

	/* make sure this is the invisible pin */
	if (ni->proto != gen_invispinprim) return;

	/* search for displayable text */
	for(i=0; i<ni->numvar; i++)
	{
		var = &ni->firstvar[i];
		if ((var->type&VDISPLAY) == 0) continue;

		/* compute the proper display offset */
		descript = var->textdescript;
		halfx = (ni->highx - ni->lowx) * 2 / ni->proto->tech->deflambda;
		halfy = (ni->highy - ni->lowy) * 2 / ni->proto->tech->deflambda;
		switch (descript&VTPOSITION)
		{
			case VTPOSCENT:
			case VTPOSBOXED:
				modifydescript(var, us_setdescriptoffset(descript, 0, 0));
				break;
			case VTPOSUP:
				modifydescript(var, us_setdescriptoffset(descript, 0, -halfy));
				break;
			case VTPOSDOWN:
				modifydescript(var, us_setdescriptoffset(descript, 0, halfy));
				break;
			case VTPOSLEFT:
				modifydescript(var, us_setdescriptoffset(descript, halfx, 0));
				break;
			case VTPOSRIGHT:
				modifydescript(var, us_setdescriptoffset(descript, -halfx, 0));
				break;
			case VTPOSUPLEFT:
				modifydescript(var, us_setdescriptoffset(descript, halfx, -halfy));
				break;
			case VTPOSUPRIGHT:
				modifydescript(var, us_setdescriptoffset(descript, -halfx, -halfy));
				break;
			case VTPOSDOWNLEFT:
				modifydescript(var, us_setdescriptoffset(descript, halfx, halfy));
				break;
			case VTPOSDOWNRIGHT:
				modifydescript(var, us_setdescriptoffset(descript, -halfx, halfy));
				break;
		}
	}
}

/*
 * routine to parse the "grab-point" specification in "pp" and return the
 * code.  Prints an error message and returns -1 on error.
 */
INTBIG us_gettextposition(char *pp)
{
	REGISTER INTSML l;

	l = strlen(pp);
	if (namesamen(pp, "centered", l) == 0 && l >= 1) return(VTPOSCENT);
	if (namesamen(pp, "boxed", l) == 0 && l >= 1) return(VTPOSBOXED);
	if (namesame(pp, "up") == 0) return(VTPOSUP);
	if (namesame(pp, "down") == 0) return(VTPOSDOWN);
	if (namesamen(pp, "left", l) == 0 && l >= 1) return(VTPOSLEFT);
	if (namesamen(pp, "right", l) == 0 && l >= 1) return(VTPOSRIGHT);
	if (namesamen(pp, "up-left", l) == 0 && l >= 4) return(VTPOSUPLEFT);
	if (namesamen(pp, "up-right", l) == 0 && l >= 4) return(VTPOSUPRIGHT);
	if (namesamen(pp, "down-left", l) == 0 && l >= 6) return(VTPOSDOWNLEFT);
	if (namesamen(pp, "down-right", l) == 0 && l >= 6) return(VTPOSDOWNRIGHT);
	us_abortcommand("Unrecognized grab-point: %s", pp);
	return(-1);
}

/*
 * routine to parse the "text size" specification in "pp" and return the
 * code.  Prints an error message and returns -1 on error.
 */
INTBIG us_gettextsize(char *pp, INTBIG old)
{
	REGISTER INTSML l;

	l = strlen(pp);
	if (namesamen(pp, "4p", l) == 0 && l >= 1) return(TXT4P);
	if (namesamen(pp, "6p", l) == 0 && l >= 1) return(TXT6P);
	if (namesamen(pp, "8p", l) == 0 && l >= 1) return(TXT8P);
	if (namesamen(pp, "10p", l) == 0 && l >= 1) return(TXT10P);
	if (namesamen(pp, "12p", l) == 0 && l >= 1) return(TXT12P);
	if (namesamen(pp, "14p", l) == 0 && l >= 1) return(TXT14P);
	if (namesamen(pp, "16p", l) == 0 && l >= 1) return(TXT16P);
	if (namesamen(pp, "18p", l) == 0 && l >= 1) return(TXT18P);
	if (namesamen(pp, "20p", l) == 0 && l >= 1) return(TXT20P);
	if (namesamen(pp, "small", l) == 0 && l >= 1) return(TXTSMALL);
	if (namesamen(pp, "medium", l) == 0 && l >= 1) return(TXTMEDIUM);
	if (namesamen(pp, "large", l) == 0 && l >= 1) return(TXTLARGE);
	if (namesamen(pp, "up", l) == 0 && l >= 1)
	{
		if (old >= TXT4P && old < TXT20P) old++; else
			if (old >= TXTSMALL && old < TXTLARGE) old++;
		return(old);
	} else if (namesamen(pp, "down", l) == 0 && l >= 1)
	{
		if (old > TXT4P && old <= TXT20P) old--; else
			if (old > TXTSMALL && old <= TXTLARGE) old--;
		return(old);
	}
	us_abortcommand("Unrecognized text size: %s", pp);
	return(-1);
}

/******************** FACET CENTER/TEXT WINDOW CODE ********************/

/*
 * routine to allocate a new ubchange from the pool (if any) or memory,
 * fill in the "facet", "change", "x", and "y" fields, and link it to the
 * global list.  Returns nonzero on error.
 */
INTSML us_newubchange(NODEPROTO *facet, INTBIG change, INTBIG x, INTBIG y)
{
	REGISTER UBCHANGE *d;

	if (us_ubchangefree == NOUBCHANGE)
	{
		d = (UBCHANGE *)emalloc((sizeof (UBCHANGE)), us_aid->cluster);
		if (d == 0) return(-1);
	} else
	{
		/* take ubchange from free list */
		d = us_ubchangefree;
		us_ubchangefree = (UBCHANGE *)d->nextubchange;
	}
	d->facet = facet;
	d->change = change;
	d->x = x;
	d->y = y;
	d->nextubchange = us_ubchanges;
	us_ubchanges = d;
	return(0);
}

/*
 * routine to return ubchange "d" to the pool of free ubchanges
 */
void us_freeubchange(UBCHANGE *d)
{
	d->nextubchange = us_ubchangefree;
	us_ubchangefree = d;
}

/*
 * routine to remove all queued user broadcast changes to facet "np"
 * because it was deleted
 */
void us_removeubchange(NODEPROTO *np)
{
	REGISTER UBCHANGE *d, *lastd, *nextd;

	lastd = NOUBCHANGE;
	for(d = us_ubchanges; d != NOUBCHANGE; d = nextd)
	{
		nextd = d->nextubchange;
		if (d->facet == np)
		{
			if (lastd == NOUBCHANGE) us_ubchanges = nextd; else
				lastd->nextubchange = nextd;
			us_freeubchange(d);
			continue;
		}
		lastd = d;
	}
}

/*
 * routine to put facet center (x, y) on facet "np".
 */
void us_setnodeprotocenter(INTBIG x, INTBIG y, NODEPROTO *np)
{
	(void)us_newubchange(np, SETCC, x, y);
}

/*
 * routine to remove facet center from facet "np".
 */
void us_delnodeprotocenter(NODEPROTO *np)
{
	(void)us_newubchange(np, KILLCC, 0, 0);
}

/*
 * routine to remove variable "FACET_message" from facet "np".
 */
void us_delfacetmessage(NODEPROTO *np)
{
	(void)us_newubchange(np, KILLFM, 0, 0);
}

/*
 * routine to implement all user broadcast changes queued during the last broadcast
 */
void us_doubchanges(void)
{
	REGISTER UBCHANGE *d, *nextd;
	INTBIG position[2];
	REGISTER WINDOW *w;
	REGISTER EDITOR *ed;
	REGISTER VARIABLE *var;

	for(d = us_ubchanges; d != NOUBCHANGE; d = nextd)
	{
		nextd = d->nextubchange;

		switch (d->change)
		{
			case SETCC:		/* set facet center */
				position[0] = d->x;   position[1] = d->y;
				(void)setvalkey((INTBIG)d->facet, VNODEPROTO, el_prototype_center,
					(INTBIG)position, VINTEGER|VISARRAY|(2<<VLENGTHSH));
				break;

			case KILLCC:	/* remove facet center */
				(void)delvalkey((INTBIG)d->facet, VNODEPROTO, el_prototype_center);
				break;

			case KILLFM:	/* remove facet_message */
				for(w = el_topwindow; w != NOWINDOW; w = w->nextwindow)
				{
					if (w->curnodeproto != d->facet) continue;
					if ((w->state&WINDOWTYPE) != TEXTWINDOW) continue;

					/* see if the window still has a valid variable */
					ed = w->editor;
					if (ed->editobjvar == NOVARIABLE) continue;
					var = getval((INTBIG)ed->editobjaddr, ed->editobjtype, -1, ed->editobjqual);
					if (var == NOVARIABLE)
					{
						(void)newewindow(w->location, w, 0);
						killwindow(w);
					}
				}
				break;
		}

		/* cleanup */
		us_freeubchange(d);
	}
	us_ubchanges = NOUBCHANGE;
}

/******************** COLOR ********************/

/*
 * setup the color map for the graphics of technology "tech".  "style" is:
 *  COLORSEXISTING  continue existing colors
 *  COLORSDEFAULT   use default nonoverlappable colors
 *  COLORSBLACK     use black background colors
 *  COLORSWHITE     use white background colors
 * A set of overlappable colors is obtained from technology "tech", combined
 * with the appropriate nonoverlappable colors, and set into the proper
 *  variables on the "user" aid (and subsequently displayed).
 * The 256 entries are organized thusly:
 * Bit 0 is for highlighting; bit 1 is an escape for
 * opaque colors, the next 5 bits are the overlappable colors (if the opaque
 * escape is off) or the opaque value (if the bit is set).
 * The last bit is for the grid, although it may not appear if there are 128 entries.
 * This routine uses the database variable "USER_color_map" on the
 * technologies.
 */
void us_getcolormap(TECHNOLOGY *tech, INTSML style, INTSML broadcast)
{
	static TECH_COLORMAP colmap[38] =
	{
		{255,255,255}, /*   4(0004) WHITE:   white                           */
		  {0,  0,  0}, /*  12(0014) BLACK:   black                           */
		{255,  0,  0}, /*  20(0024) RED:     red                             */
		  {0,  0,255}, /*  28(0034) BLUE:    blue                            */
		  {0,255,  0}, /*  36(0044) GREEN:   green                           */
		  {0,255,255}, /*  44(0054) CYAN:    cyan                            */
		{255,  0,255}, /*  52(0064) MAGENTA: magenta                         */
		{255,255,  0}, /*  60(0074) YELLOW:  yellow                          */
		  {0,  0,  0}, /*  68(0104) FACETTXT: facet and port names           */
		  {0,  0,  0}, /*  76(0114) FACETOUT: facet outline                  */
		  {0,  0,  0}, /*  84(0124) WINBOR:  window border color             */
		  {0,255,  0}, /*  92(0134) HWINBOR: highlighted window border color */
		  {0,  0,  0}, /* 100(0144) MENBOR:  menu border color               */
		{255,255,255}, /* 108(0154) HMENBOR: highlighted menu border color   */
		  {0,  0,  0}, /* 116(0164) MENTXT:  menu text color                 */
		  {0,  0,  0}, /* 124(0174) MENGLY:  menu glyph color                */
		{255,255,255}, /* 132(0204) CURSOR:  cursor color                    */
		{180,180,180}, /* 140(0214) GRAY:    gray                            */
		{255,190,  6}, /* 148(0224) ORANGE:  orange                          */
		{186,  0,255}, /* 156(0234) PURPLE:  purple                          */
		{139, 99, 46}, /* 164(0244) BROWN:   brown                           */
		{230,230,230}, /* 172(0254) LGRAY:   light gray                      */
		{100,100,100}, /* 180(0264) DGRAY:   dark gray                       */
		{255,150,150}, /* 188(0274) LRED:    light red                       */
		{159, 80, 80}, /* 196(0304) DRED:    dark red                        */
		{175,255,175}, /* 204(0314) LGREEN:  light green                     */
		 {89,159, 85}, /* 212(0324) DGREEN:  dark green                      */
		{150,150,255}, /* 220(0334) LBLUE:   light blue                      */
		  {2, 15,159}, /* 228(0344) DBLUE:   dark blue                       */
		  {0,  0,  0}, /* 236(0354)          unassigned                      */
		  {0,  0,  0}, /* 244(0364)          unassigned                      */
		  {0,  0,  0}, /* 252(0374)          unassigned                      */
		  {0,  0,  0}, /*                    grid                            */
		{255,255,255}, /*                    highlight                       */
		{255,  0,  0}, /*                    black background highlight      */
		{255,  0,  0}, /*                    white background highlight      */
		{255,255,255}, /*                    black background cursor         */
		  {0,  0,  0}  /*                    white background cursor         */
	};
	static TECH_COLORMAP default_colmap[32] =
	{                  /*     overlap4 overlap3 overlap2 overlap1 overlap0 */
		{200,200,200}, /* 0:                                               */
		{  0,  0,200}, /* 1:                                      overlap0 */
		{220,  0,120}, /* 2:                             overlap1          */
		{ 80,  0,160}, /* 3:                             overlap1+overlap0 */
		{ 70,250, 70}, /* 4:                    overlap2                   */
		{  0,140,140}, /* 5:                    overlap2+         overlap0 */
		{180,130,  0}, /* 6:                    overlap2+overlap1          */
		{ 55, 70,140}, /* 7:                    overlap2+overlap1+overlap0 */
		{250,250,  0}, /* 8:           overlap3                            */
		{ 85,105,160}, /* 9:           overlap3+                  overlap0 */
		{190, 80,100}, /* 10:          overlap3+         overlap1          */
		{ 70, 50,150}, /* 11:          overlap3+         overlap1+overlap0 */
		{ 80,210,  0}, /* 12:          overlap3+overlap2                   */
		{ 50,105,130}, /* 13:          overlap3+overlap2+         overlap0 */
		{170,110,  0}, /* 14:          overlap3+overlap2+overlap1          */
		{ 60, 60,130}, /* 15:          overlap3+overlap2+overlap1+overlap0 */
		{180,180,180}, /* 16: overlap4+                                    */
		{  0,  0,180}, /* 17: overlap4+                           overlap0 */
		{200,  0,100}, /* 18: overlap4+                  overlap1          */
		{ 60,  0,140}, /* 19: overlap4+                  overlap1+overlap0 */
		{ 50,230, 50}, /* 20: overlap4+         overlap2                   */
		{  0,120,120}, /* 21: overlap4+         overlap2+         overlap0 */
		{160,110,  0}, /* 22: overlap4+         overlap2+overlap1          */
		{ 35, 50,120}, /* 23: overlap4+         overlap2+overlap1+overlap0 */
		{230,230,  0}, /* 24: overlap4+overlap3                            */
		{ 65, 85,140}, /* 25: overlap4+overlap3+                  overlap0 */
		{170, 60, 80}, /* 26: overlap4+overlap3+         overlap1          */
		{ 50, 30,130}, /* 27: overlap4+overlap3+         overlap1+overlap0 */
		{ 60,190,  0}, /* 28: overlap4+overlap3+overlap2                   */
		{ 30, 85,110}, /* 29: overlap4+overlap3+overlap2+         overlap0 */
		{150, 90,  0}, /* 30: overlap4+overlap3+overlap2+overlap1          */
		{ 40, 40,110}, /* 31: overlap4+overlap3+overlap2+overlap1+overlap0 */
	};

	TECH_COLORMAP *mapptr, *thisptr;
	REGISTER INTSML i;
	REGISTER VARIABLE *var, *rvar, *gvar, *bvar;
	static INTBIG USER_color_map = 0;
	INTBIG red[256], green[256], blue[256];
	extern GRAPHICS us_gbox;

	/* get the technology's color information */
	if (USER_color_map == 0) USER_color_map = makekey("USER_color_map");
	var = getvalkey((INTBIG)tech, VTECHNOLOGY, VCHAR|VISARRAY, USER_color_map);
	if (var != NOVARIABLE) mapptr = (TECH_COLORMAP *)var->addr; else mapptr = 0;

	/* get existing color information */
	rvar = getvalkey((INTBIG)us_aid, VAID, VINTEGER|VISARRAY, us_colormap_red);
	gvar = getvalkey((INTBIG)us_aid, VAID, VINTEGER|VISARRAY, us_colormap_green);
	bvar = getvalkey((INTBIG)us_aid, VAID, VINTEGER|VISARRAY, us_colormap_blue);

	/* must have some colors */
	if (rvar == NOVARIABLE && gvar == NOVARIABLE && bvar == NOVARIABLE && var == NOVARIABLE)
	{
		mapptr = default_colmap;
	}

	if (style == COLORSEXISTING)
	{
		/* not resetting, get the old color values */
		for(i=0; i<256; i++)
		{
			red[i] = ((INTBIG *)rvar->addr)[i];
			green[i] = ((INTBIG *)gvar->addr)[i];
			blue[i] = ((INTBIG *)bvar->addr)[i];
			if ((i&(LAYERH|LAYERG|LAYEROE)) != 0) continue;
			if (mapptr == 0) continue;
			if (i == 0) continue;
			thisptr = &mapptr[i>>2];
			red[i]  = thisptr->red;   green[i]   = thisptr->green;
			blue[i] = thisptr->blue;
		}
	} else
	{
#if SIMAID
		/* update simulation window colors */
		switch (style)
		{
			case COLORSWHITE:
			case COLORSDEFAULT:
				sim_window_displaycolor(OFF_STRENGTH, BLUE);
				sim_window_displaycolor(NODE_STRENGTH, GREEN);
				sim_window_displaycolor(GATE_STRENGTH, MAGENTA);
				sim_window_displaycolor(VDD_STRENGTH, BLACK);
				sim_window_displaycolor(LOGIC_LOW, BLUE);
				sim_window_displaycolor(LOGIC_HIGH, MAGENTA);
				sim_window_displaycolor(LOGIC_X, BLACK);
				break;
			case COLORSBLACK:
				sim_window_displaycolor(OFF_STRENGTH, GREEN);
				sim_window_displaycolor(NODE_STRENGTH, CYAN);
				sim_window_displaycolor(GATE_STRENGTH, MAGENTA);
				sim_window_displaycolor(VDD_STRENGTH, LRED);
				sim_window_displaycolor(LOGIC_LOW, GREEN);
				sim_window_displaycolor(LOGIC_HIGH, MAGENTA);
				sim_window_displaycolor(LOGIC_X, LRED);
				break;
		}
#endif

		/* resetting: load entirely new color map */
		for(i=0; i<256; i++)
		{
			if ((i&LAYERH) != 0)
			{
				switch (style)
				{
					case COLORSDEFAULT: thisptr = &colmap[33];   break;	/* white */
					case COLORSBLACK:   thisptr = &colmap[34];   break;	/* red */
					case COLORSWHITE:   thisptr = &colmap[35];   break;	/* red */
				}
			} else if ((i&LAYERG) != 0)
			{
				switch (style)
				{
					case COLORSDEFAULT: thisptr = &colmap[32];   break;	/* black */
					case COLORSBLACK:   thisptr = &colmap[33];   break;	/* white */
					case COLORSWHITE:   thisptr = &colmap[32];   break;	/* black */
				}
			} else if ((i&LAYEROE) != 0)
			{
				thisptr = &colmap[i>>2];

				if (i == HMENBOR) switch (style)
				{
					case COLORSBLACK: thisptr = &colmap[2];   break;	/* red */
					case COLORSWHITE: thisptr = &colmap[2];   break;	/* red */
				}
				if (i == CURSOR) switch (style)
				{
					case COLORSDEFAULT: thisptr = &colmap[16];   break;	/* default */
					case COLORSBLACK:   thisptr = &colmap[36];   break;	/* white */
					case COLORSWHITE:   thisptr = &colmap[37];   break;	/* black */
				}

				/* reverse black and white when using black background */
				if (style == COLORSBLACK)
				{
					switch (i)
					{
						case FACETTXT:
						case FACETOUT:
						case WINBOR:
						case MENBOR:
						case MENTXT:
						case MENGLY:
						case BLACK:    thisptr = &colmap[33];   break;		/* white */
						case WHITE:    thisptr = &colmap[37];   break;		/* black */
					}
				}
			} else
			{
				if (rvar != NOVARIABLE) red[i] = ((INTBIG *)rvar->addr)[i];
				if (gvar != NOVARIABLE) green[i] = ((INTBIG *)gvar->addr)[i];
				if (bvar != NOVARIABLE) blue[i] = ((INTBIG *)bvar->addr)[i];
				if (mapptr != 0) thisptr = &mapptr[i>>2]; else thisptr = 0;
				if (i == ALLOFF) switch (style)
				{
					case COLORSBLACK: thisptr = &colmap[32];   break;	/* black */
					case COLORSWHITE: thisptr = &colmap[33];   break;	/* white */
				}
				if (thisptr == 0) continue;
			}
			red[i]  = thisptr->red;   green[i]   = thisptr->green;
			blue[i] = thisptr->blue;
		}

		/* also set the grid color appropriately if it doesn't have its own bitplane */
		if (el_maplength < 256)
		{
			switch (style)
			{
				case COLORSDEFAULT: us_gbox.col = BLACK;   break;	/* black */
				case COLORSBLACK:   us_gbox.col = WHITE;   break;	/* white */
				case COLORSWHITE:   us_gbox.col = BLACK;   break;	/* black */
			}
		}
	}

	/* set the color map */
	if (broadcast != 0)
		startobjectchange((INTBIG)us_aid, VAID);
	if (broadcast == 0) nextvarchangequiet();
	(void)setvalkey((INTBIG)us_aid, VAID, us_colormap_red, (INTBIG)red,
		VINTEGER|VISARRAY|(256<<VLENGTHSH)|VDONTSAVE);
	if (broadcast == 0) nextvarchangequiet();
	(void)setvalkey((INTBIG)us_aid, VAID, us_colormap_green, (INTBIG)green,
		VINTEGER|VISARRAY|(256<<VLENGTHSH)|VDONTSAVE);
	if (broadcast == 0) nextvarchangequiet();
	(void)setvalkey((INTBIG)us_aid, VAID, us_colormap_blue, (INTBIG)blue,
		VINTEGER|VISARRAY|(256<<VLENGTHSH)|VDONTSAVE);
	if (broadcast != 0)
		endobjectchange((INTBIG)us_aid, VAID);
}

/*
 * routine to load entry "entry" of the global color map entries with the value
 * (red, green, blue), letter "letter".  Handles highlight and grid layers
 * right if "spread" is nonzero.
 */
void us_setcolorentry(INTSML entry1, INTSML red, INTSML green, INTSML blue, INTSML letter,
	INTSML spread)
{
	REGISTER INTSML j;

	startobjectchange((INTBIG)us_aid, VAID);
	(void)setindkey((INTBIG)us_aid, VAID, us_colormap_red, entry1, red);
	(void)setindkey((INTBIG)us_aid, VAID, us_colormap_green, entry1, green);
	(void)setindkey((INTBIG)us_aid, VAID, us_colormap_blue, entry1, blue);

	/* place in other entries if special */
	if ((entry1&LAYERH) == LAYERH && spread != 0)
	{
		/* set all highlight colors */
		for(j=0; j<256; j++) if ((j&LAYERH) == LAYERH)
		{
			(void)setindkey((INTBIG)us_aid, VAID, us_colormap_red, j, red);
			(void)setindkey((INTBIG)us_aid, VAID, us_colormap_green, j, green);
			(void)setindkey((INTBIG)us_aid, VAID, us_colormap_blue, j, blue);
		}
	} else if ((entry1&(LAYERG|LAYERH)) == LAYERG && spread != 0)
	{
		/* set all grid colors */
		for(j=0; j<256; j++) if ((j&(LAYERG|LAYERH)) == LAYERG)
		{
			(void)setindkey((INTBIG)us_aid, VAID, us_colormap_red, j, red);
			(void)setindkey((INTBIG)us_aid, VAID, us_colormap_green, j, green);
			(void)setindkey((INTBIG)us_aid, VAID, us_colormap_blue, j, blue);
		}
	}
	endobjectchange((INTBIG)us_aid, VAID);
}

/*
 * routine to convert a red/green/blue color in (ir,ig,ib) to a hue/saturation/
 * intensity color in (h,s,i)
 */
void us_rgbtohsv(INTSML ir, INTSML ig, INTSML ib, float *h, float *s, float *i)
{
	float x, r, g, b, rdot, gdot, bdot;

	r = ir / 255.0;
	g = ig / 255.0;
	b = ib / 255.0;

	/* "i" is maximum of "r", "g", and "b" */
	if (r > g) *i = r; else *i = g;
	if (b > *i) *i = b;

	/* "x" is minimum of "r", "g", and "b" */
	if (r < g) x = r; else x = g;
	if (b < x) x = b;

	/* "saturation" is (i-x)/i */
	if (*i == 0.0) *s = 0.0; else *s = (*i - x) / *i;

	if (*s == 0.0) *h = 0.0; else
	{
		rdot = (*i - r) / (*i - x);
		gdot = (*i - g) / (*i - x);
		bdot = (*i - b) / (*i - x);
		if (b == x && r == *i) *h = (1.0 - gdot) / 6.0; else
		if (b == x && g == *i) *h = (1.0 + rdot) / 6.0; else
		if (r == x && g == *i) *h = (3.0 - bdot) / 6.0; else
		if (r == x && b == *i) *h = (3.0 + gdot) / 6.0; else
		if (g == x && b == *i) *h = (5.0 - rdot) / 6.0; else
		if (g == x && r == *i) *h = (5.0 + bdot) / 6.0; else
			ttyputmsg("Cannot convert (%d,%d,%d), for x=%f i=%f s=%f", ir, ig, ib, x, *i, *s);
	}
}

/*
 * routine to convert a hue/saturation/intensity color in (h,s,v) to a red/
 * green/blue color in (r,g,b)
 */
void us_hsvtorgb(float h, float s, float v, INTBIG *r, INTBIG *g, INTBIG *b)
{
	REGISTER INTBIG i;
	REGISTER float f, m, n, k;

	h = h * 6.0;
	i = (INTBIG)h;
	f = h - (float)i;
	m = v * (1.0 - s);
	n = v * (1.0 - s * f);
	k = v * (1.0 - s * (1.0 - f));
	switch (i)
	{
		case 0:
			*r = (INTBIG)(v*255.0); *g = (INTBIG)(k*255.0); *b = (INTBIG)(m*255.0);
			break;
		case 1:
			*r = (INTBIG)(n*255.0); *g = (INTBIG)(v*255.0); *b = (INTBIG)(m*255.0);
			break;
		case 2:
			*r = (INTBIG)(m*255.0); *g = (INTBIG)(v*255.0); *b = (INTBIG)(k*255.0);
			break;
		case 3:
			*r = (INTBIG)(m*255.0); *g = (INTBIG)(n*255.0); *b = (INTBIG)(v*255.0);
			break;
		case 4:
			*r = (INTBIG)(k*255.0); *g = (INTBIG)(m*255.0); *b = (INTBIG)(v*255.0);
			break;
		case 5:
			*r = (INTBIG)(v*255.0); *g = (INTBIG)(m*255.0); *b = (INTBIG)(n*255.0);
			break;
	}
	if (*r < 0 || *r > 255 || *g < 0 || *g > 255 || *b < 0 || *b > 255)
		ttyputmsg("(%f,%f,%f) -> (%d,%d,%d) (i=%d)",h, s, v, *r, *g, *b, i);
}

/******************** MISCELLANEOUS ********************/

void us_getlowleft(NODEINST *ni, INTBIG *x, INTBIG *y)
{
	INTBIG lx, ly, hx, hy;
	XARRAY trans;
	static POLYGON *poly = NOPOLYGON;

	if (poly == NOPOLYGON) poly = allocpolygon(4, us_aid->cluster);

	nodesizeoffset(ni->proto, &lx, &ly, &hx, &hy);
	maketruerectpoly(ni->lowx+lx, ni->highx-hx, ni->lowy+ly, ni->highy-hy, poly);
	if (ni->rotation != 0 || ni->transpose != 0)
	{
		makerot(ni, trans);
		xformpoly(poly, trans);
	}
	getbbox(poly, x, &hx, y, &hy);
}

/*
 * routine to modify the text descriptor in the highlighted object "high"
 */
void us_modifytextdescript(HIGHLIGHT *high, INTBIG descript)
{
	if (high->fromvar != NOVARIABLE)
	{
		if (descript != high->fromvar->textdescript)
			modifydescript(high->fromvar, descript);
		return;
	}
	if (high->fromport != NOPORTPROTO)
	{
		(void)setval((INTBIG)high->fromport, VPORTPROTO, "textdescript", descript, VINTEGER);
		return;
	}
	if (high->fromgeom->entrytype == OBJNODEINST)
	{
		(void)setval((INTBIG)high->fromgeom->entryaddr.ni, VNODEINST, "textdescript", descript,
			VINTEGER);
		return;
	}
}

INTSML us_describerestrictions(char ***names)
{
	static char *mynames[30];
	REGISTER INTSML count;
	extern AIDENTRY *io_aid, *net_aid;
#if COMAID
	extern AIDENTRY *com_aid;
#endif
#if COMPENAID
	extern AIDENTRY *compen_aid;
#endif
#if DRCAID
	extern AIDENTRY *dr_aid;
#endif
#if MAPPERAID
	extern AIDENTRY *map_aid;
#endif
#if PLAAID
	extern AIDENTRY *pla_aid;
#endif
#if PROJECTAID
	extern AIDENTRY *proj_aid;
#endif
#if ROUTAID
	extern AIDENTRY *ro_aid;
#endif
#if SCAID
	extern AIDENTRY *sc_aid;
#endif
#if SIMAID
	extern AIDENTRY *sim_aid;
#endif
#if VHDLAID
	extern AIDENTRY *vhdl_aid;
#endif

	count = 0;

	if (checkcap(us_aid, USERHASTEXTED) == 0) mynames[count++] = "Text edit";
	if (checkcap(us_aid, USERHASTECHED) == 0) mynames[count++] = "Technology edit";
	if (checkcap(us_aid, USERHASLANG) == 0) mynames[count++] = "Language interpreter";

	if (checkcap(io_aid, IOHASIC) == 0) mynames[count++] = "IC I/O";
	if (checkcap(io_aid, IOHASMECH) == 0) mynames[count++] = "Mechanical I/O";
	if (checkcap(io_aid, IOCANWRITE) == 0) mynames[count++] = "Write libraries";
	if (checkcap(io_aid, IOHASPRINT) == 0) mynames[count++] = "Plotting output";

#if COMAID
	if (checkcap(com_aid, COMPAVAIL) == 0) mynames[count++] = "Compaction";
#endif
#if PLAAID
	if (checkcap(pla_aid, PLAHASNMOS) == 0) mynames[count++] = "nMOS PLAs";
	if (checkcap(pla_aid, PLAHASCMOS) == 0) mynames[count++] = "CMOS PLAs";
#endif
#if ROUTAID
	if (checkcap(ro_aid, ROUTHASSTITCH) == 0) mynames[count++] = "Stitch routing";
	if (checkcap(ro_aid, ROUTHASRIVER) == 0) mynames[count++] = "River routing";
	if (checkcap(ro_aid, ROUTHASMAZE) == 0) mynames[count++] = "Maze routing";
#endif
#if SCAID
	if (checkcap(sc_aid, SILCOMPAVAIL) == 0) mynames[count++] = "Silicon compiler";
#endif
	if (checkcap(net_aid, NETHASNCC) == 0) mynames[count++] = "Network consistency checking";
#if DRCAID
	if (checkcap(dr_aid, DRCHASINC) == 0) mynames[count++] = "Incremental DRC";
	if (checkcap(dr_aid, DRCHASBATCH) == 0) mynames[count++] = "Batch DRC";
	if (checkcap(dr_aid, DRCHASDRAC) == 0) mynames[count++] = "ECAD DRC";
#endif
#if VHDLAID
	if (checkcap(vhdl_aid, VHDLHASGEN) == 0) mynames[count++] = "VHDL generation";
	if (checkcap(vhdl_aid, VHDLHASCOMP) == 0) mynames[count++] = "VHDL compilation";
#endif
#if SIMAID
	if (checkcap(sim_aid, SIMHASSPICE) == 0) mynames[count++] = "SPICE simulation";
	if (checkcap(sim_aid, SIMHASALS) == 0) mynames[count++] = "ALS simulation";
	if (checkcap(sim_aid, SIMHASCOMM) == 0) mynames[count++] = "Commercial simulation interfaces";
	if (checkcap(sim_aid, SIMHASUNIV) == 0) mynames[count++] = "University simulation interfaces";
#endif
#if COMPENAID
	if (checkcap(compen_aid, MISCHASCOMPEN) == 0) mynames[count++] = "Compensation";
#endif
#if PROJECTAID
	if (checkcap(proj_aid, MISCHASPROJ) == 0) mynames[count++] = "Project management";
#endif
#if MAPPERAID
	if (checkcap(map_aid, MISCHASMAPPER) == 0) mynames[count++] = "Mapping";
#endif
	*names = mynames;
	return(count);
}
