/*
 * Electric(tm) VLSI Design System
 *
 * File: usrnet.c
 * User interface aid: network manipulation routines
 * Written by: Steven M. Rubin, Electric Editor Incorporated
 *
 * Copyright (c) 1998 Electric Editor Incorporated.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Electric Editor Incorporated
 * 23470 Sunset Drive, Suite 108
 * Los Gatos, California 95033
 * support@electriceditor.com
 */

#include "global.h"
#include "egraphics.h"
#include "usr.h"
#include "usrtrack.h"
#include "database.h"
#include "efunction.h"
#include "edialogs.h"
#include "tecgen.h"
#include "tecart.h"
#include "tecschem.h"
#include <math.h>

#define EXACTSELECTDISTANCE 5		/* number of pixels for selection */

/* prototypes for local routines */
void us_modnodebits(NODEINST*, INTSML, INTSML, INTBIG, INTBIG);
INTSML us_copyvariablefacets(INTBIG, INTBIG);
void us_recursivelysearch(RTNODE*, INTSML, INTSML, INTSML, INTSML, HIGHLIGHT*, HIGHLIGHT*, HIGHLIGHT*, HIGHLIGHT*, HIGHLIGHT*, INTSML*, INTBIG*, INTBIG, INTBIG, WINDOW*, INTSML);
void us_checkoutobject(GEOM*, INTSML, INTSML, INTSML, HIGHLIGHT*, HIGHLIGHT*, HIGHLIGHT*, HIGHLIGHT*, HIGHLIGHT*, INTSML*, INTBIG*, INTBIG, INTBIG, WINDOW*);
void us_initnodetext(NODEINST*, INTSML);
INTSML us_getnodetext(NODEINST*, WINDOW*, POLYGON*, VARIABLE**, PORTPROTO**);
void us_initarctext(ARCINST*, INTSML);
INTSML us_getarctext(ARCINST*, WINDOW*, POLYGON*, VARIABLE**);
INTBIG us_findnewplace(INTBIG*, INTBIG, INTBIG, INTBIG);
void us_setbestsnappoint(HIGHLIGHT *best, INTBIG wantx, INTBIG wanty, INTBIG newx, INTBIG newy, INTSML tan, INTSML perp);
void us_selectsnappoly(HIGHLIGHT *best, POLYGON *poly, INTBIG wantx, INTBIG wanty);
INTSML us_pointonarc(INTBIG x, INTBIG y, POLYGON *poly);
void us_intersectsnapline(HIGHLIGHT *best, INTBIG x1, INTBIG y1, INTBIG x2, INTBIG y2,
	POLYGON *interpoly, INTBIG wantx, INTBIG wanty);
void us_intersectsnappoly(HIGHLIGHT *best, POLYGON *poly, POLYGON *interpoly, INTBIG wantx, INTBIG wanty);
void us_adjustonetangent(HIGHLIGHT *high, POLYGON *poly, INTBIG x, INTBIG y);
void us_adjustoneperpendicular(HIGHLIGHT *high, POLYGON *poly, INTBIG x, INTBIG y);
void us_xformpointtonode(INTBIG x, INTBIG y, NODEINST *ni, INTBIG *xo, INTBIG *yo);
void us_addleadtoicon(NODEPROTO *facet, ARCPROTO *wire, NODEINST *pin, NODEINST *box,
	PORTPROTO *boxport, INTBIG pinx, INTBIG piny, INTBIG boxx, INTBIG boxy);

/*
 * routine to recursively expand the facet "ni" by "amount" levels.
 * "sofar" is the number of levels that this has already been expanded.
 */
void us_doexpand(NODEINST *ni, INTSML amount, INTSML sofar)
{
	REGISTER NODEPROTO *np;
	REGISTER NODEINST *ono;

	if ((ni->userbits & NEXPAND) == 0)
	{
		/* turn the nodeinst display off */
		startobjectchange((INTBIG)ni, VNODEINST);

		/* expanded the facet */
		ni->userbits |= NEXPAND;

		/* redisplay node */
		endobjectchange((INTBIG)ni, VNODEINST);

		/* if depth limit reached, quit */
		if (++sofar >= amount) return;
	}

	/* explore insides of this one */
	np = ni->proto;
	for(ono = np->firstnodeinst; ono != NONODEINST; ono = ono->nextnodeinst)
	{
		if (ono->proto->index != 0) continue;
		us_doexpand(ono, amount, sofar);
	}
}

void us_dounexpand(NODEINST *ni)
{
	REGISTER NODEPROTO *np;
	REGISTER NODEINST *ono;

	if ((ni->userbits & NEXPAND) == 0) return;

	np = ni->proto;
	for(ono = np->firstnodeinst; ono != NONODEINST; ono = ono->nextnodeinst)
	{
		if (ono->proto->index != 0) continue;
		if ((ono->userbits & NEXPAND) != 0) us_dounexpand(ono);
	}

	if ((ni->userbits & WANTEXP) != 0)
	{
		/* turn the nodeinst display off */
		startobjectchange((INTBIG)ni, VNODEINST);

		/* expanded the facet */
		ni->userbits &= ~NEXPAND;

		/* redisplay node */
		endobjectchange((INTBIG)ni, VNODEINST);
	}
}

INTSML us_setunexpand(NODEINST *ni, INTSML amount)
{
	REGISTER INTSML depth;
	REGISTER NODEINST *ono;
	REGISTER NODEPROTO *np;

	ni->userbits &= ~WANTEXP;
	if ((ni->userbits & NEXPAND) == 0) return(0);
	np = ni->proto;
	depth = 0;
	for(ono = np->firstnodeinst; ono != NONODEINST; ono = ono->nextnodeinst)
	{
		if (ono->proto->index != 0) continue;
		if ((ono->userbits & NEXPAND) != 0)
			depth = maxi(depth, us_setunexpand(ono, amount));
	}
	if (depth < amount) ni->userbits |= WANTEXP;
	return(depth+1);
}

/*
 * routine to recursively modify the bits in the user aid
 * word of nodes below nodeinst "nodeinst".  This goes only to a depth of
 * "depth" levels below "curdepth", so initially, "curdepth" should be 0 and
 * "depth" should be the desired recursion depth.  The bits in "onbits"
 * of each nodeinst are turned on and the bits in "offbits" are turned off.
 */
void us_modnodebits(NODEINST *nodeinst, INTSML depth, INTSML curdepth, INTBIG onbits,
	INTBIG offbits)
{
	REGISTER NODEINST *ni;
	REGISTER NODEPROTO *np;

	nodeinst->userbits = (nodeinst->userbits | onbits) & ~offbits;
	if (++curdepth >= depth) return;

	np = nodeinst->proto;
	if (np->index != 0) return;
	for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
		us_modnodebits(ni, depth, curdepth, onbits,offbits);
}

/*
 * routine to modify the arcs in list "list".  The "change" field has the
 * following meaning:
 *  change=0   make arc rigid
 *  change=1   make arc un-rigid
 *  change=2   make arc fixed-angle
 *  change=3   make arc not fixed-angle
 *  change=4   make arc slidable
 *  change=5   make arc nonslidable
 *  change=6   make arc temporarily rigid
 *  change=7   make arc temporarily un-rigid
 *  change=8   make arc ends extend by half width
 *  change=9   make arc ends not extend by half width
 *  change=10  make arc directional
 *  change=11  make arc not directional
 *  change=12  make arc negated
 *  change=13  make arc not negated
 *  change=14  make arc skip the tail end
 *  change=15  make arc not skip the tail end
 *  change=16  make arc skip the head end
 *  change=17  make arc not skip the head end
 *  change=18  reverse ends of the arc
 *  change=19  clear arc constraints
 *  change=20  print arc constraints
 *  change=21  set special arc constraint from "prop"
 *  change=22  add to special arc constraint from "prop"
 *  change=23  toggle arc rigid
 *  change=24  toggle arc fixed-angle
 *  change=25  toggle arc slidable
 *  change=26  toggle arc ends extend
 *  change=27  toggle arc directional
 *  change=28  toggle arc negated
 *  change=29  toggle arc skip the tail
 *  change=30  toggle arc skip the head
 * If "redraw" is nonzero then the arcs are re-drawn to reflect the change.
 * The routine returns the number of arcs that were modified (-1 if an error
 * occurred).
 */
INTSML us_modarcbits(INTSML change, INTSML redraw, char *prop, GEOM **list)
{
	REGISTER INTSML total, affected;
	REGISTER INTBIG newvalue;
	REGISTER ARCINST *ai;

	/* must be a facet in the window */
	if (us_needfacet() == NONODEPROTO) return(-1);

	/* run through the list */
	affected = 0;
	for(total=0; list[total] != NOGEOM; total++)
	{
		ai = list[total]->entryaddr.ai;

		/* notify system that arc is to be re-drawn */
		if (redraw != 0) startobjectchange((INTBIG)ai, VARCINST);

		/* change the arc state bits */
		switch (change)
		{
			case 0:			/* arc rigid */
			case 1:			/* arc un-rigid */
			case 2:			/* arc fixed-angle */
			case 3:			/* arc not fixed-angle */
			case 4:			/* arc slidable */
			case 5:			/* arc nonslidable */
			case 6:			/* arc temporarily rigid */
			case 7:			/* arc temporarily un-rigid */
				if ((*el_curconstraint->setobject)((INTBIG)ai, VARCINST, change, 0) == 0)
					affected++;
				break;
			case 8:			/* arc ends extend by half width */
				if ((ai->userbits&NOEXTEND) != 0) affected++;
				(void)setval((INTBIG)ai, VARCINST, "userbits", ai->userbits & ~NOEXTEND, VINTEGER);
				break;
			case 9:			/* arc ends not extend by half width */
				if ((ai->userbits&NOEXTEND) == 0) affected++;
				(void)setval((INTBIG)ai, VARCINST, "userbits", ai->userbits|NOEXTEND, VINTEGER);
				break;
			case 10:			/* arc directional */
				if ((ai->userbits&ISDIRECTIONAL) == 0) affected++;
				(void)setval((INTBIG)ai, VARCINST, "userbits", ai->userbits|ISDIRECTIONAL, VINTEGER);
				break;
			case 11:			/* arc not directional */
				if ((ai->userbits&ISDIRECTIONAL) != 0) affected++;
				(void)setval((INTBIG)ai, VARCINST, "userbits", ai->userbits & ~ISDIRECTIONAL, VINTEGER);
				break;
			case 12:			/* arc negated */
				if ((ai->userbits&ISNEGATED) == 0) affected++;
				newvalue = ai->userbits | ISNEGATED;

				/* don't put the negation circle on a pin */
				if (((ai->end[0].nodeinst->proto->userbits&NFUNCTION) >> NFUNCTIONSH) == NPPIN &&
					((ai->end[1].nodeinst->proto->userbits&NFUNCTION) >> NFUNCTIONSH) != NPPIN)
						newvalue |= REVERSEEND;

				/* prefer output negation to input negation */
				if ((ai->end[0].portarcinst->proto->userbits&STATEBITS) == INPORT &&
					(ai->end[1].portarcinst->proto->userbits&STATEBITS) == OUTPORT)
						newvalue |= REVERSEEND;

				(void)setval((INTBIG)ai, VARCINST, "userbits", newvalue, VINTEGER);
				break;
			case 13:			/* arc not negated */
				if ((ai->userbits&ISNEGATED) != 0) affected++;
				(void)setval((INTBIG)ai, VARCINST, "userbits", ai->userbits & ~ISNEGATED, VINTEGER);
				break;
			case 14:			/* arc skip the tail end */
				if ((ai->userbits&NOTEND0) == 0) affected++;
				(void)setval((INTBIG)ai, VARCINST, "userbits", ai->userbits|NOTEND0, VINTEGER);
				break;
			case 15:			/* arc not skip the tail end */
				if ((ai->userbits&NOTEND0) != 0) affected++;
				(void)setval((INTBIG)ai, VARCINST, "userbits", ai->userbits & ~NOTEND0, VINTEGER);
				break;
			case 16:			/* arc skip the head end */
				if ((ai->userbits&NOTEND1) == 0) affected++;
				(void)setval((INTBIG)ai, VARCINST, "userbits", ai->userbits|NOTEND1, VINTEGER);
				break;
			case 17:			/* arc not skip the head end */
				if ((ai->userbits&NOTEND1) != 0) affected++;
				(void)setval((INTBIG)ai, VARCINST, "userbits", ai->userbits & ~NOTEND1, VINTEGER);
				break;
			case 18:			/* reverse ends of the arc */
				affected++;
				newvalue = (ai->userbits & ~REVERSEEND) | ((~ai->userbits) & REVERSEEND);
				(void)setval((INTBIG)ai, VARCINST, "userbits", newvalue, VINTEGER);
				break;
			case 19:			/* clear special arc properties */
				if ((*el_curconstraint->setobject)((INTBIG)ai, VARCINST, 2, (INTBIG)"") == 0)
					affected++;
				break;
			case 20:			/* print special arc properties */
				if ((*el_curconstraint->setobject)((INTBIG)ai, VARCINST, 3, (INTBIG)"") == 0)
					affected++;
				break;
			case 21:			/* set special arc properties */
				if ((*el_curconstraint->setobject)((INTBIG)ai, VARCINST, 0, (INTBIG)prop) == 0)
					affected++;
				break;
			case 22:			/* add to special arc properties */
				if ((*el_curconstraint->setobject)((INTBIG)ai, VARCINST, 1, (INTBIG)prop) == 0)
					affected++;
				break;
			case 23:			/* arc toggle rigid */
				if ((ai->userbits&FIXED) != 0)
					(void)(*el_curconstraint->setobject)((INTBIG)ai, VARCINST, 1, 0); else
						(void)(*el_curconstraint->setobject)((INTBIG)ai, VARCINST, 0, 0);
				affected++;
				break;
			case 24:			/* arc toggle fixed-angle */
				if ((ai->userbits&FIXANG) != 0)
					(void)(*el_curconstraint->setobject)((INTBIG)ai, VARCINST, 3, 0); else
						(void)(*el_curconstraint->setobject)((INTBIG)ai, VARCINST, 2, 0);
				affected++;
				break;
			case 25:			/* arc toggle slidable */
				if ((ai->userbits&CANTSLIDE) != 0)
					(void)(*el_curconstraint->setobject)((INTBIG)ai, VARCINST, 4, 0); else
						(void)(*el_curconstraint->setobject)((INTBIG)ai, VARCINST, 5, 0);
				affected++;
				break;
			case 26:			/* arc toggle ends extend */
				if ((ai->userbits&NOEXTEND) != 0) newvalue = ai->userbits & ~NOEXTEND; else
					newvalue = ai->userbits | ~NOEXTEND;
				(void)setval((INTBIG)ai, VARCINST, "userbits", newvalue, VINTEGER);
				affected++;
				break;
			case 27:			/* arc toggle directional */
				if ((ai->userbits&ISDIRECTIONAL) != 0) newvalue = ai->userbits & ~ISDIRECTIONAL; else
					newvalue = ai->userbits | ISDIRECTIONAL;
				(void)setval((INTBIG)ai, VARCINST, "userbits", newvalue, VINTEGER);
				affected++;
				break;
			case 28:			/* arc toggle negated */
				if ((ai->userbits&ISNEGATED) != 0) newvalue = ai->userbits & ~ISNEGATED; else
					newvalue = ai->userbits | ISNEGATED;

				if ((newvalue&ISNEGATED) != 0)
				{
					/* don't put the negation circle on a pin */
					if (((ai->end[0].nodeinst->proto->userbits&NFUNCTION) >> NFUNCTIONSH) == NPPIN &&
						((ai->end[1].nodeinst->proto->userbits&NFUNCTION) >> NFUNCTIONSH) != NPPIN)
							newvalue |= REVERSEEND;

					/* prefer output negation to input negation */
					if ((ai->end[0].portarcinst->proto->userbits&STATEBITS) == INPORT &&
						(ai->end[1].portarcinst->proto->userbits&STATEBITS) == OUTPORT)
							newvalue |= REVERSEEND;
				}

				(void)setval((INTBIG)ai, VARCINST, "userbits", newvalue, VINTEGER);
				affected++;
				break;
			case 29:			/* arc toggle skip the tail */
				if ((ai->userbits&NOTEND0) != 0) newvalue = ai->userbits & ~NOTEND0; else
					newvalue = ai->userbits | NOTEND0;
				(void)setval((INTBIG)ai, VARCINST, "userbits", newvalue, VINTEGER);
				affected++;
				break;
			case 30:			/* arc toggle skip the head */
				if ((ai->userbits&NOTEND1) != 0) newvalue = ai->userbits & ~NOTEND1; else
					newvalue = ai->userbits | NOTEND1;
				(void)setval((INTBIG)ai, VARCINST, "userbits", newvalue, VINTEGER);
				affected++;
				break;
		}

		/* notify the system that the arc is done and can be re-drawn */
		if (redraw != 0) endobjectchange((INTBIG)ai, VARCINST);
	}

	return(affected);
}

/*
 * routine to set the arc name of arc "ai" to the name "name".  If "name" is
 * zero, remove the name
 */
void us_setarcname(ARCINST *ai, char *name)
{
	REGISTER VARIABLE *var;
	REGISTER INTBIG descript;

	startobjectchange((INTBIG)ai, VARCINST);
	if (name == 0) (void)delvalkey((INTBIG)ai, VARCINST, el_arc_name); else
	{
		var = getvalkey((INTBIG)ai, VARCINST, VSTRING, el_arc_name);
		if (var != NOVARIABLE) descript = var->textdescript; else
			descript = defaulttextdescript();
		var = setvalkey((INTBIG)ai, VARCINST, el_arc_name, (INTBIG)name, VSTRING|VDISPLAY);
		if (var == NOVARIABLE) return;
		var->textdescript = descript;

		/* for zero-width arcs, adjust the location */
		if (ai->width == 0)
		{
			if (ai->end[0].ypos == ai->end[1].ypos)
			{
				/* zero-width horizontal arc has name above */
				modifydescript(var, (var->textdescript & ~VTPOSITION) | VTPOSUP);
			} else if (ai->end[0].xpos == ai->end[1].xpos)
			{
				/* zero-width vertical arc has name to right */
				modifydescript(var, (var->textdescript & ~VTPOSITION) | VTPOSRIGHT);
			}
		}
	}
	endobjectchange((INTBIG)ai, VARCINST);
}

/*
 * routine to travel through the network starting at nodeinst "ni" and set
 * "bit" in all nodes connected with arcs that do not spread.
 */
void us_nettravel(NODEINST *ni, INTBIG bit)
{
	REGISTER INTSML i;
	REGISTER PORTARCINST *pi;

	if ((ni->userbits & bit) != 0) return;
	ni->userbits |= bit;

	for(pi=ni->firstportarcinst; pi!=NOPORTARCINST; pi=pi->nextportarcinst)
	{
		if ((pi->conarcinst->userbits & SPREADA) != 0) continue;
		for(i=0; i<2; i++) us_nettravel(pi->conarcinst->end[i].nodeinst, bit);
	}
}

/*
 * routine to recursively travel along all arcs coming out of nodeinst "ni"
 * and set the SPREADN bit in the connecting nodeinst "userbits" if that node
 * is connected horizontally (if "hor" is nonzero) or connected vertically (if
 * "hor" is zero).  This is called from "spread" to propagate along manhattan
 * arcs that are in the correct orientation (along the spread line).
 */
void us_manhattantravel(NODEINST *ni, INTSML hor)
{
	REGISTER PORTARCINST *pi;
	REGISTER NODEINST *other;
	REGISTER ARCINST *ai;

	ni->userbits |= SPREADN;
	for(pi=ni->firstportarcinst; pi!=NOPORTARCINST; pi=pi->nextportarcinst)
	{
		ai = pi->conarcinst;
		if (hor != 0)
		{
			/* "hor" is nonzero: only want horizontal arcs */
			if (((ai->userbits&AANGLE)>>AANGLESH) != 0 &&
				((ai->userbits&AANGLE)>>AANGLESH) != 180) continue;
		} else
		{
			/* "hor" is zero: only want vertical arcs */
			if (((ai->userbits&AANGLE)>>AANGLESH) != 90 &&
				((ai->userbits&AANGLE)>>AANGLESH) != 270) continue;
		}
		if (ai->end[0].portarcinst == pi) other = ai->end[1].nodeinst; else
			other = ai->end[0].nodeinst;
		if ((other->userbits&SPREADN) != 0) continue;
		us_manhattantravel(other, hor);
	}
}

/*
 * Routine to determine whether facet "np" has a facet center in it.  If so,
 * an error is displayed and the routine returns nonzero.
 */
INTSML us_alreadyfacetcenter(NODEPROTO *np)
{
	REGISTER NODEINST *ni;

	for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
		if (ni->proto == gen_facetcenterprim)
	{
		us_abortcommand("This facet already has a facet-center");
		return(1);
	}
	return(0);
}

/*
 * routine to obtain details about a "port" command in "count" and "par".
 * The node under consideration is in "ni", and the port under consideration
 * if "ppt" (which is NOPORTPROTO if no particular port is under consideration).
 * The port characteristic bits are set in "bits" and the parts of these bits
 * that are set have mask bits set into "mask".  The port to be affected is
 * returned.  If "wantexp" is nonzero, the desired port should already be
 * exported (otherwise it should not be exported).  If "intendedname" is set,
 * it is the name that will be given to the port when it is exported.  The
 * routine returns NOPORTPROTO if there is an error.
 */
PORTPROTO *us_portdetails(PORTPROTO *ppt, INTSML count, char *par[], NODEINST *ni,
	INTBIG *bits, INTBIG *mask, INTSML wantexp, char *intendedname)
{
	REGISTER PORTPROTO *wantpp, *pp;
	HIGHLIGHT high;
	INTBIG x, y;
	REGISTER INTSML i, l, m, index, specify;
	REGISTER INTBIG onlx, only, onhx, onhy, bestx, besty;
	REGISTER PORTEXPINST *pe;
	REGISTER NODEPROTO *np;
	REGISTER VARIABLE *var;
	REGISTER char *pt;
	char *newpar;
	static struct
	{
		char  *name;
		INTSML  significant;
		UINTBIG bits, mask;
	} portparse[] =
	{
		{"input",         1, INPORT,           STATEBITS},
		{"output",        1, OUTPORT,          STATEBITS},
		{"bidirectional", 2, BIDIRPORT,        STATEBITS},
		{"power",         1, PWRPORT,          STATEBITS},
		{"ground",        1, GNDPORT,          STATEBITS},
		{"clock1",        6, C1PORT,           STATEBITS},
		{"clock2",        6, C2PORT,           STATEBITS},
		{"clock3",        6, C3PORT,           STATEBITS},
		{"clock4",        6, C4PORT,           STATEBITS},
		{"clock5",        6, C5PORT,           STATEBITS},
		{"clock6",        6, C6PORT,           STATEBITS},
		{"clock",         1, CLKPORT,          STATEBITS},
		{"refout",        4, REFOUTPORT,       STATEBITS},
		{"refin",         4, REFINPORT,        STATEBITS},
		{"none",          1, 0,                STATEBITS|PORTDRAWN|BODYONLY},
		{"always-drawn",  1, PORTDRAWN,        PORTDRAWN},
		{"body-only",     2, BODYONLY,         BODYONLY},
		{NULL, 0, 0, 0}
	};

	/* quick sanity check first */
	np = ni->proto;
	if (np->firstportproto == NOPORTPROTO)
	{
		us_abortcommand("This node has no ports");
		return(NOPORTPROTO);
	}

	/* prepare to parse parameters */
	wantpp = NOPORTPROTO;
	specify = *bits = *mask = 0;

	/* look at all parameters */
	for(i=0; i<count; i++)
	{
		l = strlen(pt = par[i]);

		/* check the basic characteristics from the table */
		for(m=0; portparse[m].name != 0; m++)
			if (namesamen(pt, portparse[m].name, l) == 0 && l >= portparse[m].significant)
		{
			*bits |= portparse[m].bits;
			*mask |= portparse[m].mask;
			break;
		}
		if (portparse[m].name != 0) continue;

		if (namesamen(pt, "specify", l) == 0 && l >= 1)
		{ specify++; continue; }
		if (namesamen(pt, "use", l) == 0 && l >= 1)
		{
			if (i+1 >= count)
			{
				us_abortcommand("Usage: port use PORTNAME");
				return(NOPORTPROTO);
			}
			i++;
			if (wantexp == 0)
			{
				/* want to export: look for any port on the node */
				for(wantpp = np->firstportproto; wantpp != NOPORTPROTO; wantpp = wantpp->nextportproto)
					if (namesame(wantpp->protoname, par[i]) == 0) break;
				if (wantpp == NOPORTPROTO)
				{
					us_abortcommand("No port called %s", par[i]);
					return(NOPORTPROTO);
				}
			} else
			{
				/* want exported ports: look specificially for them */
				for(pe = ni->firstportexpinst; pe != NOPORTEXPINST; pe = pe->nextportexpinst)
					if (namesame(pe->exportproto->protoname, par[i]) == 0) break;
				if (pe != NOPORTEXPINST) wantpp = pe->exportproto; else
				{
					us_abortcommand("No port exported as %s", par[i]);
					return(NOPORTPROTO);
				}
			}
			continue;
		}
		us_abortcommand("Bad PORT option: %s", pt);
		return(NOPORTPROTO);
	}

	/* if no port explicitly found, use default (if any) */
	if (wantpp == NOPORTPROTO) wantpp = ppt;

	/* if no port found and heuristics are allowed, try them */
	if (wantpp == NOPORTPROTO && specify == 0)
	{
		/* if there is only one possible port, use it */
		if (wantexp == 0)
		{
			/* look for only unexported port */
			index = 0;
			for(pp = np->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
			{
				for(pe = ni->firstportexpinst; pe != NOPORTEXPINST; pe = pe->nextportexpinst)
					if (pe->proto == pp) break;
				if (pe != NOPORTEXPINST) continue;
				index++;
				wantpp = pp;
			}
			if (index != 1) wantpp = NOPORTPROTO;
		} else
		{
			/* if there is one exported port return it */
			pe = ni->firstportexpinst;
			if (pe != NOPORTEXPINST && pe->nextportexpinst == NOPORTEXPINST)
				wantpp = pe->exportproto;
		}

		/* if a port is highlighted, use it */
		if (wantpp == NOPORTPROTO)
		{
			var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_highlighted);
			if (var != NOVARIABLE)
			{
				if (getlength(var) == 1)
				{
					(void)us_makehighlight(((char **)var->addr)[0], &high);
					if ((high.status&HIGHTYPE) == HIGHFROM && high.fromport != NOPORTPROTO)
					{
						pp = high.fromport;
						if (wantexp == 0) wantpp = pp; else
						{
							for(pe = ni->firstportexpinst; pe != NOPORTEXPINST; pe = pe->nextportexpinst)
								if (pe->proto == pp) break;
							if (pe != NOPORTEXPINST) wantpp = pe->exportproto; else
							{
								us_abortcommand("Port %s must be exported first", pp->protoname);
								return(NOPORTPROTO);
							}
						}
					}
				}
			}
		}

		/* if exporting port with the same name as the subportinst, use it */
		if (wantpp == NOPORTPROTO && *intendedname != 0 && wantexp == 0)
		{
			for(pp = np->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
				if (namesame(intendedname, pp->protoname) == 0)
			{
				wantpp = pp;
				break;
			}
		}

		/* if port is on one edge of the facet and is being exported, use it */
		if (wantpp == NOPORTPROTO && wantexp == 0)
		{
			if (ni->geom->lowx == ni->parent->lowx) onlx = 1; else onlx = 0;
			if (ni->geom->highx == ni->parent->highx) onhx = 1; else onhx = 0;
			if (ni->geom->lowy == ni->parent->lowy) only = 1; else only = 0;
			if (ni->geom->highy == ni->parent->highy) onhy = 1; else onhy = 0;
			if (onlx+onhx+only+onhy == 1)
			{
				/* look for one port on the node that is on the proper edge */
				bestx = (ni->lowx+ni->highx)/2;
				besty = (ni->lowy+ni->highy)/2;
				wantpp = NOPORTPROTO;
				for(pp = np->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
				{
					portposition(ni, pp, &x, &y);
					if (onlx != 0 && x == bestx) wantpp = NOPORTPROTO;
					if (onlx != 0 && x < bestx)
					{
						wantpp = pp;  bestx = x;
					}
					if (onhx != 0 && x == bestx) wantpp = NOPORTPROTO;
					if (onhx != 0 && x > bestx)
					{
						wantpp = pp;  bestx = x;
					}
					if (only != 0 && y == besty) wantpp = NOPORTPROTO;
					if (only != 0 && y < besty)
					{
						wantpp = pp;  besty = y;
					}
					if (onhy != 0 && y == besty) wantpp = NOPORTPROTO;
					if (onhy != 0 && y > besty)
					{
						wantpp = pp;  besty = y;
					}
				}
			}
		}
	}

	/* give up and ask the port name wanted */
	if (wantpp == NOPORTPROTO)
	{
		us_identifyports(ni, LAYERA);
		ttyputerr("Which port of node %s is to be the port:", describenodeproto(np));
		index = 0;
		for(pp = np->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
			ttyputmsg("%d: %s", ++index, pp->protoname);
		for(;;)
		{
			newpar = ttygetline("Select a number: ");
			if (*newpar == 0)
			{
				us_abortedmsg();
				break;
			}
			i = atoi(newpar);
			if (i <= 0 || i > index)
			{
				ttyputerr("Please select a number from 1 to %d (default aborts)", index);
				continue;
			}

			/* convert to a port */
			x = 0;
			for(pp = np->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
				if (++x == i) break;
			if (wantexp == 0) wantpp = pp; else
			{
				for(pe = ni->firstportexpinst; pe != NOPORTEXPINST; pe = pe->nextportexpinst)
					if (pe->proto == pp) break;
				if (pe == NOPORTEXPINST)
				{
					ttyputerr("That port is not exported");
					continue;
				}
				wantpp = pe->exportproto;
			}
			break;
		}
		us_identifyports(ni, ALLOFF);
	}

	/* finally, return the port */
	return(wantpp);
}

/*
 * routine to recursively delete ports at nodeinst "ni" and all arcs connected
 * to them anywhere.  If "spt" is not NOPORTPROTO, delete only that portproto
 * on this nodeinst (and its hierarchically related ports).  Otherwise delete
 * all portprotos on this nodeinst.
 */
void us_undoportproto(NODEINST *ni, PORTPROTO *spt)
{
	REGISTER PORTEXPINST *pe, *nextpe;

	for(pe = ni->firstportexpinst; pe != NOPORTEXPINST; pe = nextpe)
	{
		nextpe = pe->nextportexpinst;
		if (spt != NOPORTPROTO && spt != pe->exportproto) continue;
		if (killportproto(pe->exportproto->parent, pe->exportproto))
			ttyputerr("killportproto error");
	}
}

/*
 * recursive helper routine for "us_copyfacet" which copies facet "fromnp"
 * to a new facet called "toname" in the current library ("el_curlib") with the
 * new view type "nview".  All needed subfacets, shared view facets, and facets
 * referenced by variables are copied too.
 */
NODEPROTO *us_copyrecursively(NODEPROTO *fromnp, char *toname, VIEW *nview)
{
	REGISTER NODEPROTO *np, *onp, *newfromnp;
	REGISTER NODEINST *ni;
	REGISTER ARCINST *ai;
	REGISTER PORTPROTO *pp;
	REGISTER char *newname;

	/* check variables on the nodeproto that reference other facets */
	if (us_copyvariablefacets((INTBIG)fromnp, VNODEPROTO) != 0) return(NONODEPROTO);

	/* must copy subfacets */
	for(ni = fromnp->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		if (us_copyvariablefacets((INTBIG)ni, VNODEINST) != 0) return(NONODEPROTO);
		np = ni->proto;
		if (np->index != 0) continue;

		/* see if newly copied facet with this name and view already exists */
		for(onp = el_curlib->firstnodeproto; onp != NONODEPROTO; onp = onp->nextnodeproto)
			if (namesame(onp->cell->cellname, np->cell->cellname) == 0 &&
				onp->cellview == np->cellview && onp->temp1 == 0) break;
		if (onp != NONODEPROTO) continue;

		ttyputmsg("Copying subfacet %s", describenodeproto(np));
		onp = us_copyrecursively(np, np->cell->cellname, np->cellview);
		if (onp == NONODEPROTO)
		{
			ttyputerr("Copy of subfacet %s failed", describenodeproto(np));
			return(NONODEPROTO);
		}
	}

	/* copy facet variables on the arcs */
	for(ai = fromnp->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
	{
		if (us_copyvariablefacets((INTBIG)ai, VARCINST) != 0) return(NONODEPROTO);
		if (us_copyvariablefacets((INTBIG)ai->end[0].portarcinst, VPORTARCINST) != 0)
			return(NONODEPROTO);
		if (us_copyvariablefacets((INTBIG)ai->end[1].portarcinst, VPORTARCINST) != 0)
			return(NONODEPROTO);
	}

	/* copy facet variables on the ports */
	for(pp = fromnp->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
	{
		if (us_copyvariablefacets((INTBIG)pp, VPORTPROTO) != 0) return(NONODEPROTO);
		if (us_copyvariablefacets((INTBIG)pp->subportexpinst, VPORTEXPINST) != 0)
			return(NONODEPROTO);
	}

	/* copy the facet if it is not already done */
	for(newfromnp = el_curlib->firstnodeproto; newfromnp != NONODEPROTO; newfromnp = newfromnp->nextnodeproto)
		if (namesame(newfromnp->cell->cellname, toname) == 0 &&
			newfromnp->cellview == nview && newfromnp->temp1 == 0) break;
	if (newfromnp == NONODEPROTO)
	{
		if (*nview->sviewname != 0)
		{
			(void)initinfstr();
			(void)addstringtoinfstr(toname);
			(void)addtoinfstr('{');
			(void)addstringtoinfstr(nview->sviewname);
			(void)addtoinfstr('}');
			newname = returninfstr();
		} else newname = toname;
		newfromnp = copynodeproto(fromnp, el_curlib, newname);
		if (newfromnp == NONODEPROTO) return(NONODEPROTO);

		/* ensure that the copied facet is the right size */
		(*el_curconstraint->solve)(newfromnp);
		newfromnp->temp1 = 0;
	}

	/* also copy equivalent views */
	for(np = fromnp->cell->firstincell; np != NONODEPROTO; np = np->nextincell)
		if (np != fromnp)
	{
		/* see if there is a facet with this name in the new library */
		for(onp = el_curlib->firstnodeproto; onp != NONODEPROTO; onp = onp->nextnodeproto)
			if (namesame(onp->cell->cellname, np->cell->cellname) == 0 &&
				onp->cellview == np->cellview && onp->temp1 == 0) break;
		if (onp != NONODEPROTO) continue;

		ttyputmsg("Copying alternate view %s", describenodeproto(np));
		onp = us_copyrecursively(np, np->cell->cellname, np->cellview);
		if (onp == NONODEPROTO)
		{
			ttyputerr("Copy of alternate view %s failed", describenodeproto(np));
			return(NONODEPROTO);
		}
	}
	return(newfromnp);
}

/*
 * helper routine of "us_copyrecursively" which examines the variables on object
 * "fromaddr" of type "fromtype" and copies any references to facets.
 */
INTSML us_copyvariablefacets(INTBIG fromaddr, INTBIG fromtype)
{
	REGISTER INTSML i, j;
	REGISTER INTBIG addr, type, len;
	INTSML *numvar;
	REGISTER NODEPROTO *np, *onp;
	VARIABLE **firstvar;

	/* get the list of variables on this object */
	if (db_getvarptr(fromaddr, fromtype, &firstvar, &numvar) != 0) return(0);

	/* look at each variable */
	for(i=0; i<(*numvar); i++)
	{
		type = (*firstvar)[i].type;
		if ((type&VTYPE) != VNODEPROTO) continue;

		/* NODEPROTO variable found: see if a facet must be copied */
		if ((type&VISARRAY) == 0)
		{
			addr = (*firstvar)[i].addr;
			np = (NODEPROTO *)addr;
			if (np == NONODEPROTO) continue;
			if (np->index != 0) continue;

			/* see if this facet has already been copied to the library */
			for(onp = el_curlib->firstnodeproto; onp != NONODEPROTO; onp = onp->nextnodeproto)
				if (namesame(onp->cell->cellname, np->cell->cellname) == 0 &&
					onp->cellview == np->cellview) break;
			if (onp != NONODEPROTO) continue;

			ttyputmsg("Copying associated facet %s", describenodeproto(np));
			onp = us_copyrecursively(np, np->cell->cellname, np->cellview);
			if (onp == NONODEPROTO)
			{
				ttyputerr("Copy of facet %s failed", describenodeproto(np));
				return(1);
			}
		} else
		{
			len = (type&VLENGTH) >> VLENGTHSH;
			if (len != 0)
			{
				for(j=0; j<len; j++)
				{
					np = ((NODEPROTO **)addr)[j];
					if (np == NONODEPROTO) continue;
					if (np->index != 0) continue;

					/* see if this facet has already been copied to the library */
					for(onp = el_curlib->firstnodeproto; onp != NONODEPROTO; onp = onp->nextnodeproto)
						if (namesame(onp->cell->cellname, np->cell->cellname) == 0 &&
							onp->cellview == np->cellview) break;
					if (onp != NONODEPROTO) continue;

					ttyputmsg("Copying associated facet %s", describenodeproto(np));
					onp = us_copyrecursively(np, np->cell->cellname, np->cellview);
					if (onp == NONODEPROTO)
					{
						ttyputerr("Copy of facet %s failed", describenodeproto(np));
						return(1);
					}
				}
			}
		}
	}
	return(0);
}

/*
 * routine to find an object/port close to (wantx, wanty) in the current facet.
 * If there is more than one object/port under the cursor, they are returned
 * in reverse sequential order, provided that the most recently found
 * object is described in "curhigh".  The next close object is placed in
 * "curhigh".  If "exclusively" is nonzero, find only nodes or arcs of the
 * current prototype.  If "another" is nonzero, this is the second find,
 * and should not consider text objects.  If "findport" is nonzero, port selection
 * is also desired.  If "under" is nonzero, only find objects
 * exactly under the desired cursor location.
 */
void us_findobject(INTBIG wantx, INTBIG wanty, WINDOW *win, HIGHLIGHT *curhigh,
	INTSML exclusively, INTSML another, INTSML findport, INTSML under)
{
	HIGHLIGHT best, lastdirect, prevdirect, bestdirect;
	REGISTER PORTPROTO *pp;
	REGISTER NODEINST *ni;
	REGISTER INTBIG dist;
	REGISTER INTSML phase;
	INTSML looping;
	INTBIG bestdist;
	static POLYGON *poly = NOPOLYGON;

	/* get polygon */
	if (poly == NOPOLYGON) poly = allocpolygon(4, us_aid->cluster);

	/* initialize */
	bestdist = HUGEINT;
	looping = 0;
	best.fromgeom = NOGEOM;             best.status = HIGHFROM;
	bestdirect.fromgeom = NOGEOM;       bestdirect.status = HIGHFROM;
	lastdirect.fromgeom = NOGEOM;       lastdirect.status = HIGHFROM;
	prevdirect.fromgeom = NOGEOM;       prevdirect.status = HIGHFROM;

	/* search the relevant objects in the circuit */
	for(phase = 0; phase < 3; phase++)
		us_recursivelysearch(win->curnodeproto->rtree, exclusively, another, findport,
			under, curhigh, &best, &bestdirect, &lastdirect, &prevdirect, &looping,
				&bestdist, wantx, wanty, win, phase);

	/* use best direct hit if one exists, otherwise best any-kind-of-hit */
	if (bestdirect.fromgeom != NOGEOM)
	{
		curhigh->status = bestdirect.status;
		curhigh->fromgeom = bestdirect.fromgeom;
		curhigh->fromvar = bestdirect.fromvar;
		curhigh->fromport = bestdirect.fromport;
		curhigh->snapx = bestdirect.snapx;
		curhigh->snapy = bestdirect.snapy;
	} else
	{
		if (under == 0)
		{
			curhigh->status = best.status;
			curhigh->fromgeom = best.fromgeom;
			curhigh->fromvar = best.fromvar;
			curhigh->fromport = best.fromport;
			curhigh->snapx = best.snapx;
			curhigh->snapy = best.snapy;
		} else
		{
			curhigh->status = 0;
			curhigh->fromgeom = NOGEOM;
			curhigh->fromvar = NOVARIABLE;
			curhigh->fromport = NOPORTPROTO;
			curhigh->frompoint = 0;
		}
	}

	/* see if looping through direct hits */
	if (looping != 0)
	{
		/* made direct hit on previously selected object: looping through */
		if (prevdirect.fromgeom != NOGEOM)
		{
			curhigh->status = prevdirect.status;
			curhigh->fromgeom = prevdirect.fromgeom;
			curhigh->fromvar = prevdirect.fromvar;
			curhigh->fromport = prevdirect.fromport;
			curhigh->snapx = prevdirect.snapx;
			curhigh->snapy = prevdirect.snapy;
		} else if (lastdirect.fromgeom != NOGEOM)
		{
			curhigh->status = lastdirect.status;
			curhigh->fromgeom = lastdirect.fromgeom;
			curhigh->fromvar = lastdirect.fromvar;
			curhigh->fromport = lastdirect.fromport;
			curhigh->snapx = lastdirect.snapx;
			curhigh->snapy = lastdirect.snapy;
		}
	}

	/* quit now if nothing found */
	if (curhigh->fromgeom == NOGEOM) return;

	/* find the closest port if this is a nodeinst and no port hit directly */
	if ((curhigh->status&HIGHTYPE) == HIGHFROM && curhigh->fromgeom->entrytype == OBJNODEINST &&
		curhigh->fromport == NOPORTPROTO)
	{
		ni = curhigh->fromgeom->entryaddr.ni;
		for(pp = ni->proto->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
		{
			shapeportpoly(ni, pp, poly, 0);

			/* get distance of desired point to polygon */
			dist = polydistance(poly, wantx, wanty);
			if (dist < 0)
			{
				curhigh->fromport = pp;
				break;
			}
			if (curhigh->fromport == NOPORTPROTO) bestdist = dist;
			if (dist > bestdist) continue;
			bestdist = dist;   curhigh->fromport = pp;
		}
	}
}

/*
 * routine to search R-tree "rtree" for objects that are close to (wantx, wanty)
 * in window "win".  Those that are found are passed to "us_checkoutobject"
 * for proximity evaluation, along with the evaluation parameters "curhigh",
 * "best", "bestdirect", "lastdirect", "prevdirect", "looping", "bestdist",
 * "exclusively", "another", "findport", and "under".  The "phase" value ranges
 * from 0 to 2 according to the type of object desired.
 */
void us_recursivelysearch(RTNODE *rtree, INTSML exclusively, INTSML another, INTSML findport,
	INTSML under, HIGHLIGHT *curhigh, HIGHLIGHT *best, HIGHLIGHT *bestdirect,
	HIGHLIGHT *lastdirect, HIGHLIGHT *prevdirect, INTSML *looping, INTBIG *bestdist,
	INTBIG wantx, INTBIG wanty, WINDOW *win, INTSML phase)
{
	REGISTER GEOM *geom;
	REGISTER INTSML i, bestrt, found;
	REGISTER INTBIG disttort, bestdisttort, slop, directhitdist;
	INTBIG lx, hx, ly, hy;

	found = 0;
	bestdisttort = HUGEINT;
	slop = el_curtech->deflambda * 10;
	directhitdist = muldiv(EXACTSELECTDISTANCE, win->screenhx - win->screenlx, win->usehx - win->uselx);
	if (directhitdist > slop) slop = directhitdist;
	for(i=0; i<rtree->total; i++)
	{
		db_rtnbbox(rtree, i, &lx, &hx, &ly, &hy);

		/* accumulate best R-tree module in case none are direct hits */
		disttort = abs(wantx - (lx+hx)/2) + abs(wanty - (ly+hy)/2);
		if (disttort < bestdisttort)
		{
			bestdisttort = disttort;
			bestrt = i;
		}

		/* see if this R-tree node is a direct hit */
		if (exclusively == 0 &&
			(lx > wantx+slop || hx < wantx-slop || ly > wanty+slop || hy < wanty-slop)) continue;
		found++;

		/* search it */
		if (rtree->flag != 0)
		{
			geom = (GEOM *)rtree->pointers[i];
			switch (phase)
			{
				case 0:			/* only allow complex nodes */
					if (geom->entrytype != OBJNODEINST) break;
					if (geom->entryaddr.ni->proto->index != 0) break;
					us_checkoutobject(geom, exclusively, another, findport, curhigh,
						best, bestdirect, lastdirect, prevdirect, looping,
							bestdist, wantx, wanty, win);
					break;
				case 1:			/* only allow arcs */
					if (geom->entrytype != OBJARCINST) break;
					us_checkoutobject(geom, exclusively, another, findport, curhigh,
						best, bestdirect, lastdirect, prevdirect, looping,
							bestdist, wantx, wanty, win);
					break;
				case 2:			/* only allow primitive nodes */
					if (geom->entrytype != OBJNODEINST) break;
					if (geom->entryaddr.ni->proto->index == 0) break;
					us_checkoutobject(geom, exclusively, another, findport, curhigh,
						best, bestdirect, lastdirect, prevdirect, looping,
							bestdist, wantx, wanty, win);
					break;
			}
		} else us_recursivelysearch((RTNODE *)rtree->pointers[i], exclusively,
			another, findport, under, curhigh, best, bestdirect, lastdirect,
				prevdirect, looping, bestdist, wantx, wanty, win, phase);
	}

	if (found != 0) return;
	if (bestdisttort == HUGEINT) return;
	if (under != 0) return;

	/* nothing found, use the closest */
	if (rtree->flag != 0)
	{
		geom = (GEOM *)rtree->pointers[bestrt];
		switch (phase)
		{
			case 0:			/* only allow complex nodes */
				if (geom->entrytype != OBJNODEINST) break;
				if (geom->entryaddr.ni->proto->index != 0) break;
				us_checkoutobject(geom, exclusively, another, findport, curhigh,
					best, bestdirect, lastdirect, prevdirect, looping,
						bestdist, wantx, wanty, win);
				break;
			case 1:			/* only allow arcs */
				if (geom->entrytype != OBJARCINST) break;
				us_checkoutobject(geom, exclusively, another, findport, curhigh,
					best, bestdirect, lastdirect, prevdirect, looping,
						bestdist, wantx, wanty, win);
				break;
			case 2:			/* only allow primitive nodes */
				if (geom->entrytype != OBJNODEINST) break;
				if (geom->entryaddr.ni->proto->index == 0) break;
				us_checkoutobject(geom, exclusively, another, findport, curhigh,
					best, bestdirect, lastdirect, prevdirect, looping,
						bestdist, wantx, wanty, win);
				break;
		}
	} else us_recursivelysearch((RTNODE *)rtree->pointers[bestrt], exclusively,
		another, findport, under, curhigh, best, bestdirect, lastdirect, prevdirect,
			looping, bestdist, wantx, wanty, win, phase);
}

/*
 * search helper routine to include object "geom" in the search for the
 * closest object to the cursor position at (wantx, wanty) in window "win".
 * If "exclusively" is nonzero, ignore nodes or arcs that are not of the
 * current type.  If "another" is nonzero, ignore text objects.  If "findport"
 * is nonzero, ports are being selected so facet names should not.  The closest
 * object is "*bestdist" away and is described in "best".  The closest direct
 * hit is in "bestdirect".  If that direct hit is the same as the last hit
 * (kept in "curhigh") then the last direct hit (kept in "lastdirect") is
 * moved to the previous direct hit (kept in "prevdirect") and the "looping"
 * flag is set.  This indicates that the "prevdirect" object should be used
 * (if it exists) and that the "lastdirect" object should be used failing that.
 */
void us_checkoutobject(GEOM *geom, INTSML exclusively, INTSML another, INTSML findport,
	HIGHLIGHT *curhigh, HIGHLIGHT *best, HIGHLIGHT *bestdirect, HIGHLIGHT *lastdirect,
	HIGHLIGHT *prevdirect, INTSML *looping, INTBIG *bestdist, INTBIG wantx, INTBIG wanty,
	WINDOW *win)
{
	REGISTER PORTPROTO *pp;
	VARIABLE *var;
	REGISTER NODEINST *ni;
	REGISTER ARCINST *ai;
	static POLYGON *poly = NOPOLYGON;
	PORTPROTO *port;
	REGISTER INTSML dispstyle, i, j;
	REGISTER INTBIG dist, wid, lx, hx, ly, hy, directhitdist, thisdist;
	XARRAY trans;

	/* get polygon */
	if (poly == NOPOLYGON) poly = allocpolygon(4, us_aid->cluster);

	/* compute threshold for direct hits */
	directhitdist = muldiv(EXACTSELECTDISTANCE, win->screenhx - win->screenlx, win->usehx - win->uselx);

	if (geom->entrytype == OBJNODEINST)
	{
		/* examine a node object */
		ni = geom->entryaddr.ni;

		/* do not "find" Facet-Center nodes if "findport" is set */
		if (findport != 0 && ni->proto == gen_facetcenterprim) return;

		/* skip if being exclusive */
		if (exclusively != 0 && ni->proto != us_curnodeproto) return;

		/* try text on the node (if not searching for "another") */
		if (another == 0 && exclusively == 0)
		{
			us_initnodetext(ni, findport);
			for(;;)
			{
				if (us_getnodetext(ni, win, poly, &var, &port) != 0) break;

				/* get distance of desired point to polygon */
				dist = polydistance(poly, wantx, wanty);

				/* direct hit */
				if (dist < directhitdist)
				{
					if (curhigh->fromgeom == geom && (curhigh->status&HIGHTYPE) == HIGHTEXT &&
						(curhigh->fromvar == var || curhigh->fromport == port))
					{
						*looping = 1;
						prevdirect->status = lastdirect->status;
						prevdirect->fromgeom = lastdirect->fromgeom;
						prevdirect->fromvar = lastdirect->fromvar;
						prevdirect->fromport = lastdirect->fromport;
					}
					lastdirect->status = HIGHTEXT;
					lastdirect->fromgeom = geom;
					lastdirect->fromport = port;
					lastdirect->fromvar = var;
					if (dist < *bestdist)
					{
						bestdirect->status = HIGHTEXT;
						bestdirect->fromgeom = geom;
						bestdirect->fromvar = var;
						bestdirect->fromport = port;
					}
				}

				/* see if it is closer than others */
				if (dist < *bestdist)
				{
					best->status = HIGHTEXT;
					best->fromgeom = geom;
					best->fromvar = var;
					best->fromport = port;
					*bestdist = dist;
				}
			}
		}

		/* do not "find" Invisible-Pins if they have text */
		if (ni->proto == gen_invispinprim)
		{
			for(i=0; i<ni->numvar; i++)
			{
				var = &ni->firstvar[i];
				if ((var->type&VDISPLAY) != 0) return;
			}
		}

		/* get the bounds of the object in a polygon */
		makerot(ni, trans);
		lx = ni->lowx;   hx = ni->highx;
		ly = ni->lowy;   hy = ni->highy;
		if (lx == hx)
		{
			lx -= el_curtech->deflambda/2;
			hx = lx + el_curtech->deflambda;
		}
		if (ly == hy)
		{
			ly -= el_curtech->deflambda/2;
			hy = ly + el_curtech->deflambda;
		}
		maketruerectpoly(lx, hx, ly, hy, poly);
		poly->style = FILLEDRECT;
		xformpoly(poly, trans);

		/* get distance of desired point to polygon */
		dist = polydistance(poly, wantx, wanty);

		/* if there is 1 polygon in primitive node, use more precise distance */
		if (ni->proto->index != 0 && (ni->proto->userbits&NEDGESELECT) != 0)
		{
			i = nodepolys(ni);
			for(j=0; j<i; j++)
			{
				dispstyle = us_getdispstyle(win);
				shapenodepoly(ni, j, poly);
				if ((poly->desc->style[dispstyle]&INVISIBLE) == 0)
				{
					xformpoly(poly, trans);
					thisdist = polydistance(poly, wantx, wanty);
					if (j == 0 || thisdist < dist) dist = thisdist;
				}
			}
		}

		/* direct hit */
		if (dist < directhitdist)
		{
			if (curhigh->fromgeom == geom && (curhigh->status&HIGHTYPE) != HIGHTEXT)
			{
				*looping = 1;
				prevdirect->status = lastdirect->status;
				prevdirect->fromgeom = lastdirect->fromgeom;
				prevdirect->fromvar = lastdirect->fromvar;
				prevdirect->fromport = lastdirect->fromport;
				prevdirect->snapx = lastdirect->snapx;
				prevdirect->snapy = lastdirect->snapy;

				/* see if there is another port under the cursor */
				if (curhigh->fromport != NOPORTPROTO)
				{
					for(pp = curhigh->fromport->nextportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
					{
						shapeportpoly(ni, pp, poly, 0);
						if (isinside(wantx, wanty, poly) != 0)
						{
							prevdirect->status = HIGHFROM;
							prevdirect->fromgeom = geom;
							prevdirect->fromport = pp;
							break;
						}
					}
				}
			}
			lastdirect->status = HIGHFROM;
			lastdirect->fromgeom = geom;
			lastdirect->fromport = NOPORTPROTO;
			us_selectsnap(lastdirect, wantx, wanty);
			if (dist < *bestdist)
			{
				bestdirect->status = HIGHFROM;
				bestdirect->fromgeom = geom;
				bestdirect->fromport = NOPORTPROTO;
				us_selectsnap(bestdirect, wantx, wanty);
			}
		}

		/* see if it is closer than others */
		if (dist < *bestdist)
		{
			best->status = HIGHFROM;
			best->fromgeom = geom;
			best->fromport = NOPORTPROTO;
			us_selectsnap(best, wantx, wanty);
			*bestdist = dist;
		}
	} else
	{
		/* examine an arc object */
		ai = geom->entryaddr.ai;

		/* skip if being exclusive */
		if (exclusively != 0 && ai->proto != us_curarcproto) return;

		/* try text on the arc (if not searching for "another") */
		if (another == 0 && exclusively == 0)
		{
			us_initarctext(ai, findport);
			for(;;)
			{
				if (us_getarctext(ai, win, poly, &var) != 0) break;

				/* get distance of desired point to polygon */
				dist = polydistance(poly, wantx, wanty);

				/* direct hit */
				if (dist < directhitdist)
				{
					if (curhigh->fromgeom == geom && (curhigh->status&HIGHTYPE) == HIGHTEXT &&
						(curhigh->fromvar == var))
					{
						*looping = 1;
						prevdirect->status = lastdirect->status;
						prevdirect->fromgeom = lastdirect->fromgeom;
						prevdirect->fromvar = lastdirect->fromvar;
						prevdirect->fromport = lastdirect->fromport;
					}
					lastdirect->status = HIGHTEXT;
					lastdirect->fromgeom = geom;
					lastdirect->fromvar = var;
					lastdirect->fromport = NOPORTPROTO;
					if (dist < *bestdist)
					{
						bestdirect->status = HIGHTEXT;
						bestdirect->fromgeom = geom;
						bestdirect->fromvar = var;
						us_selectsnap(bestdirect, wantx, wanty);
						bestdirect->fromport = NOPORTPROTO;
					}
				}

				/* see if it is closer than others */
				if (dist < *bestdist)
				{
					best->status = HIGHTEXT;
					best->fromgeom = geom;
					best->fromvar = var;
					best->fromport = NOPORTPROTO;
					us_selectsnap(best, wantx, wanty);
					*bestdist = dist;
				}
			}
		}

		/* prepare to examine the arc */
		wid = ai->width - arcwidthoffset(ai->proto);
		if (wid == 0) wid = ai->proto->tech->deflambda;
		if (curvedarcoutline(ai, poly, FILLED, wid) != 0)
			makearcpoly(ai->length, wid, ai, poly, FILLED);

		/* get distance of desired point to polygon */
		dist = polydistance(poly, wantx, wanty);

		/* arc is selectable precisely, check distance to cursor */
		if ((ai->proto->userbits&AEDGESELECT) != 0)
		{
			i = arcpolys(ai);
			for(j=0; j<i; j++)
			{
				dispstyle = us_getdispstyle(win);
				shapearcpoly(ai, j, poly);
				if ((poly->desc->style[dispstyle]&INVISIBLE) == 0)
				{
					thisdist = polydistance(poly, wantx, wanty);
					if (j == 0 || thisdist < dist) dist = thisdist;
				}
			}
		}

		/* direct hit */
		if (dist < directhitdist)
		{
			if (curhigh->fromgeom == geom && (curhigh->status&HIGHTYPE) != HIGHTEXT)
			{
				*looping = 1;
				prevdirect->status = lastdirect->status;
				prevdirect->fromgeom = lastdirect->fromgeom;
				prevdirect->fromvar = lastdirect->fromvar;
				prevdirect->fromport = lastdirect->fromport;
				prevdirect->snapx = lastdirect->snapx;
				prevdirect->snapy = lastdirect->snapy;
			}
			lastdirect->status = HIGHFROM;
			lastdirect->fromgeom = geom;
			lastdirect->fromport = NOPORTPROTO;
			us_selectsnap(lastdirect, wantx, wanty);
			if (dist < *bestdist)
			{
				bestdirect->status = HIGHFROM;
				bestdirect->fromgeom = geom;
				bestdirect->fromvar = NOVARIABLE;
				bestdirect->fromport = NOPORTPROTO;
				us_selectsnap(bestdirect, wantx, wanty);
			}
		}

		/* see if it is closer than others */
		if (dist < *bestdist)
		{
			best->status = HIGHFROM;
			best->fromgeom = geom;
			best->fromvar = NOVARIABLE;
			best->fromport = NOPORTPROTO;
			us_selectsnap(best, wantx, wanty);
			*bestdist = dist;
		}
	}
}

/*
 * routine to determine whether the cursor (xcur, ycur) is over the object in "high".
 */
INTSML us_cursoroverhigh(HIGHLIGHT *high, INTBIG xcur, INTBIG ycur, WINDOW *win)
{
	VARIABLE *var;
	REGISTER NODEINST *ni;
	REGISTER ARCINST *ai;
	static POLYGON *poly = NOPOLYGON;
	PORTPROTO *port;
	REGISTER INTBIG wid, lx, hx, ly, hy, directhitdist;
	REGISTER INTSML dispstyle, i, j;
	XARRAY trans;

	/* must be in the same facet */
	if (high->facet != win->curnodeproto) return(0);

	/* get polygon */
	if (poly == NOPOLYGON) poly = allocpolygon(4, us_aid->cluster);

	/* compute threshold for direct hits */
	directhitdist = muldiv(EXACTSELECTDISTANCE, win->screenhx - win->screenlx, win->usehx - win->uselx);

	/* could be selected text */
	if ((high->status&HIGHTEXT) != 0)
	{
		/* examine all text on the object */
		if (high->fromgeom->entrytype == OBJNODEINST)
		{
			ni = high->fromgeom->entryaddr.ni;
			us_initnodetext(ni, 0);
		} else
		{
			ai = high->fromgeom->entryaddr.ai;
			us_initarctext(ai, 0);
		}

		for(;;)
		{
			if (high->fromgeom->entrytype == OBJNODEINST)
			{
				if (us_getnodetext(ni, win, poly, &var, &port) != 0) break;
			} else
			{
				if (us_getarctext(ai, win, poly, &var) != 0) break;
			}

			/* ignore if not on the text */
			if (polydistance(poly, xcur, ycur) > directhitdist) continue;

			if (high->fromvar == var || high->fromport == port) return(1);
		}
		return(0);
	}

	/* must be a single node or arc selected */
	if ((high->status&HIGHFROM) == 0) return(0);

	if (high->fromgeom->entrytype == OBJNODEINST)
	{
		ni = high->fromgeom->entryaddr.ni;

		/* get the bounds of the object in a polygon */
		makerot(ni, trans);

		/* if there is only 1 polygon in primitive node, check precise distance to cursor */
		if (ni->proto->index != 0 && (ni->proto->userbits&NEDGESELECT) != 0)
		{
			i = nodepolys(ni);
			for(j=0; j<i; j++)
			{
				dispstyle = us_getdispstyle(win);
				shapenodepoly(ni, j, poly);
				if ((poly->desc->style[dispstyle]&INVISIBLE) == 0)
				{
					xformpoly(poly, trans);
					if (polydistance(poly, xcur, ycur) <= directhitdist) return(1);
				}
			}
			return(0);
		}

		/* check bounding box of node */
		lx = ni->lowx;   hx = ni->highx;
		ly = ni->lowy;   hy = ni->highy;
		if (lx == hx)
		{
			lx -= el_curtech->deflambda/2;
			hx = lx + el_curtech->deflambda;
		}
		if (ly == hy)
		{
			ly -= el_curtech->deflambda/2;
			hy = ly + el_curtech->deflambda;
		}
		maketruerectpoly(lx, hx, ly, hy, poly);
		poly->style = FILLEDRECT;
		xformpoly(poly, trans);

		/* get distance of desired point to polygon */
		if (polydistance(poly, xcur, ycur) <= 0) return(1);

		return(0);
	}

	/* examine an arc object */
	ai = high->fromgeom->entryaddr.ai;

	/* arc is selectable precisely, check distance to cursor */
	if ((ai->proto->userbits&AEDGESELECT) != 0)
	{
		i = arcpolys(ai);
		for(j=0; j<i; j++)
		{
			dispstyle = us_getdispstyle(win);
			shapearcpoly(ai, j, poly);
			if ((poly->desc->style[dispstyle]&INVISIBLE) == 0)
			{
				if (polydistance(poly, xcur, ycur) <= directhitdist) return(1);
			}
		}
		return(0);
	}

	/* prepare to examine the arc */
	wid = ai->width - arcwidthoffset(ai->proto);
	if (wid == 0) wid = ai->proto->tech->deflambda;
	if (curvedarcoutline(ai, poly, FILLED, wid) != 0)
		makearcpoly(ai->length, wid, ai, poly, FILLED);

	/* get distance of desired point to polygon */
	if (polydistance(poly, xcur, ycur) > 0) return(0);
	return(1);
}

/*
 * routine to add snapping selection to the highlight in "best".
 */
void us_selectsnap(HIGHLIGHT *best, INTBIG wantx, INTBIG wanty)
{
	REGISTER NODEINST *ni;
	REGISTER ARCINST *ai;
	REGISTER INTSML j, k;
	INTBIG cx, cy;
	static POLYGON *poly = NOPOLYGON;
	XARRAY trans;

	if ((us_state&SNAPMODE) == SNAPMODENONE) return;

	/* get polygon */
	if (poly == NOPOLYGON) poly = allocpolygon(4, us_aid->cluster);

	if (best->fromgeom->entrytype == OBJNODEINST)
	{
		ni = best->fromgeom->entryaddr.ni;
		if (ni->proto->index == 0)
		{
			if ((us_state&SNAPMODE) == SNAPMODECENTER)
			{
				corneroffset(ni, ni->proto, ni->rotation, ni->transpose, &cx, &cy);
				us_setbestsnappoint(best, wantx, wanty, ni->lowx+cx, ni->lowy+cy, 0, 0);
			}
			return;
		}
		makerot(ni, trans);
		k = nodepolys(ni);
		for(j=0; j<k; j++)
		{
			shapenodepoly(ni, j, poly);
			xformpoly(poly, trans);
			us_selectsnappoly(best, poly, wantx, wanty);
		}
	} else
	{
		ai = best->fromgeom->entryaddr.ai;
		k = arcpolys(ai);
		for(j=0; j<k; j++)
		{
			shapearcpoly(ai, j, poly);
			us_selectsnappoly(best, poly, wantx, wanty);
		}
	}
}

void us_selectsnappoly(HIGHLIGHT *best, POLYGON *poly, INTBIG wantx, INTBIG wanty)
{
	REGISTER INTBIG radius, sea;
	REGISTER INTSML j, k, tan, perp;
	INTBIG testx, testy;
	REGISTER NODEINST *ni;
	REGISTER ARCINST *ai;
	REGISTER GEOM *geom;
	XARRAY trans;
	REGISTER INTSML angle, otherang, i, last;
	static POLYGON *interpoly = NOPOLYGON;

	switch (us_state&SNAPMODE)
	{
		case SNAPMODECENTER:
			if (poly->style == CIRCLE || poly->style == DISC || poly->style == CIRCLEARC)
			{
				us_setbestsnappoint(best, wantx, wanty, poly->xv[0], poly->yv[0], 0, 0);
			} else if (poly->style != OPENED && poly->style != OPENEDT1 && poly->style != OPENEDT2 &&
				poly->style != OPENEDT3 && poly->style != CLOSED)
			{
				getcenter(poly, &testx, &testy);
				us_setbestsnappoint(best, wantx, wanty, testx, testy, 0, 0);
			}
			break;
		case SNAPMODEMIDPOINT:
			if (poly->style == OPENED || poly->style == OPENEDT1 || poly->style == OPENEDT2 ||
				poly->style == OPENEDT3 || poly->style == CLOSED)
			{
				for(i=0; i<poly->count; i++)
				{
					if (i == 0)
					{
						if (poly->style != CLOSED) continue;
						last = poly->count - 1;
					} else last = i-1;
					testx = (poly->xv[last] + poly->xv[i]) / 2;
					testy = (poly->yv[last] + poly->yv[i]) / 2;
					us_setbestsnappoint(best, wantx, wanty, testx, testy, 0, 0);
				}
			} else if (poly->style == VECTORS)
			{
				for(i=0; i<poly->count; i += 2)
				{
					testx = (poly->xv[i+1] + poly->xv[i]) / 2;
					testy = (poly->yv[i+1] + poly->yv[i]) / 2;
					us_setbestsnappoint(best, wantx, wanty, testx, testy, 0, 0);
				}
			}
			break;
		case SNAPMODEENDPOINT:
			if (poly->style == OPENED || poly->style == OPENEDT1 || poly->style == OPENEDT2 ||
				poly->style == OPENEDT3 || poly->style == CLOSED)
			{
				for(i=0; i<poly->count; i++)
					us_setbestsnappoint(best, wantx, wanty, poly->xv[i], poly->yv[i], 0, 0);
			} else if (poly->style == VECTORS)
			{
				for(i=0; i<poly->count; i += 2)
				{
					us_setbestsnappoint(best, wantx, wanty, poly->xv[i], poly->yv[i], 0, 0);
					us_setbestsnappoint(best, wantx, wanty, poly->xv[i+1], poly->yv[i+1], 0, 0);
				}
			} else if (poly->style == CIRCLEARC)
			{
				us_setbestsnappoint(best, wantx, wanty, poly->xv[1], poly->yv[1], 0, 0);
				us_setbestsnappoint(best, wantx, wanty, poly->xv[2], poly->yv[2], 0, 0);
			}
			break;
		case SNAPMODETANGENT:
		case SNAPMODEPERP:
			if (poly->style == OPENED || poly->style == OPENEDT1 || poly->style == OPENEDT2 ||
				poly->style == OPENEDT3 || poly->style == CLOSED)
			{
				for(i=0; i<poly->count; i++)
				{
					if (i == 0)
					{
						if (poly->style != CLOSED) continue;
						last = poly->count - 1;
					} else last = i-1;
					angle = figureangle(poly->xv[last],poly->yv[last], poly->xv[i],poly->yv[i]);
					otherang = (angle+900) % 3600;
					if (intersect(poly->xv[last],poly->yv[last], angle, wantx, wanty, otherang,
						&testx, &testy) >= 0)
					{
						if (testx >= mini(poly->xv[last], poly->xv[i]) &&
							testx <= maxi(poly->xv[last], poly->xv[i]) &&
							testy >= mini(poly->yv[last], poly->yv[i]) &&
							testy <= maxi(poly->yv[last], poly->yv[i]))
						{
							us_setbestsnappoint(best, wantx, wanty, testx, testy, 0, 1);
						}
					}
				}
			} else if (poly->style == VECTORS)
			{
				for(i=0; i<poly->count; i += 2)
				{
					angle = figureangle(poly->xv[i],poly->yv[i], poly->xv[i+1],poly->yv[i+1]);
					otherang = (angle+900) % 3600;
					if (intersect(poly->xv[i],poly->yv[i], angle, wantx, wanty, otherang,
						&testx, &testy) >= 0)
					{
						if (testx >= mini(poly->xv[i], poly->xv[i+1]) &&
							testx <= maxi(poly->xv[i], poly->xv[i+1]) &&
							testy >= mini(poly->yv[i], poly->yv[i+1]) &&
							testy <= maxi(poly->yv[i], poly->yv[i+1]))
						{
							us_setbestsnappoint(best, wantx, wanty, testx, testy, 0, 1);
						}
					}
				}
			} else if (poly->style == CIRCLE || poly->style == DISC || poly->style == CIRCLEARC)
			{
				if (poly->xv[0] == wantx && poly->yv[0] == wanty) break;
				angle = figureangle(poly->xv[0],poly->yv[0], wantx,wanty);
				radius = computedistance(poly->xv[0],poly->yv[0], poly->xv[1],poly->yv[1]);
				testx = poly->xv[0] + mult(radius, cosine(angle));
				testy = poly->yv[0] + mult(radius, sine(angle));
				if (poly->style == CIRCLEARC && us_pointonarc(testx, testy, poly) == 0) break;
				if ((us_state&SNAPMODE) == SNAPMODETANGENT) tan = 1; else tan = 0;
				if ((us_state&SNAPMODE) == SNAPMODEPERP) perp = 1; else perp = 0;
				us_setbestsnappoint(best, wantx, wanty, testx, testy, tan, perp);
			}
			break;
		case SNAPMODEQUAD:
			if (poly->style == CIRCLE || poly->style == DISC)
			{
				radius = computedistance(poly->xv[0],poly->yv[0], poly->xv[1],poly->yv[1]);
				us_setbestsnappoint(best, wantx, wanty, poly->xv[0]+radius, poly->yv[0], 0, 0);
				us_setbestsnappoint(best, wantx, wanty, poly->xv[0]-radius, poly->yv[0], 0, 0);
				us_setbestsnappoint(best, wantx, wanty, poly->xv[0], poly->yv[0]+radius, 0, 0);
				us_setbestsnappoint(best, wantx, wanty, poly->xv[0], poly->yv[0]-radius, 0, 0);
			} else if (poly->style == CIRCLEARC)
			{
				radius = computedistance(poly->xv[0],poly->yv[0], poly->xv[1],poly->yv[1]);
				if (us_pointonarc(poly->xv[0]+radius, poly->yv[0], poly) != 0)
					us_setbestsnappoint(best, wantx, wanty, poly->xv[0]+radius, poly->yv[0], 0, 0);
				if (us_pointonarc(poly->xv[0]-radius, poly->yv[0], poly) != 0)
					us_setbestsnappoint(best, wantx, wanty, poly->xv[0]-radius, poly->yv[0], 0, 0);
				if (us_pointonarc(poly->xv[0], poly->yv[0]+radius, poly) != 0)
					us_setbestsnappoint(best, wantx, wanty, poly->xv[0], poly->yv[0]+radius, 0, 0);
				if (us_pointonarc(poly->xv[0], poly->yv[0]-radius, poly) != 0)
					us_setbestsnappoint(best, wantx, wanty, poly->xv[0], poly->yv[0]-radius, 0, 0);
			}
			break;
		case SNAPMODEINTER:
			/* get intersection polygon */
			if (interpoly == NOPOLYGON) interpoly = allocpolygon(4, us_aid->cluster);

			/* search in area around this object */
			sea = initsearch(best->fromgeom->lowx, best->fromgeom->highx, best->fromgeom->lowy,
				best->fromgeom->highy, geomparent(best->fromgeom));
			for(;;)
			{
				geom = nextobject(sea);
				if (geom == NOGEOM) break;
				if (geom == best->fromgeom) continue;
				if (geom->entrytype == OBJNODEINST)
				{
					ni = geom->entryaddr.ni;
					if (ni->proto->index == 0) continue;
					makerot(ni, trans);
					k = nodepolys(ni);
					for(j=0; j<k; j++)
					{
						shapenodepoly(ni, j, interpoly);
						xformpoly(interpoly, trans);
						us_intersectsnappoly(best, poly, interpoly, wantx, wanty);
					}
				} else
				{
					ai = geom->entryaddr.ai;
					k = arcpolys(ai);
					for(j=0; j<k; j++)
					{
						shapearcpoly(ai, j, interpoly);
						us_intersectsnappoly(best, poly, interpoly, wantx, wanty);
					}
				}
			}
			break;
	}
}

/*
 * routine to find the intersection between polygons "poly" and "interpoly" and set this as the snap
 * point in highlight "best" (the cursor is at (wantx,wanty).
 */
void us_intersectsnappoly(HIGHLIGHT *best, POLYGON *poly, POLYGON *interpoly, INTBIG wantx, INTBIG wanty)
{
	REGISTER POLYGON *swappoly;
	REGISTER INTSML i, last;

	if (interpoly->style == OPENED || interpoly->style == OPENEDT1 || interpoly->style == OPENEDT2 ||
		interpoly->style == OPENEDT3 || interpoly->style == CLOSED || interpoly->style == VECTORS)
	{
		swappoly = poly;   poly = interpoly;   interpoly = swappoly;
	}

	if (poly->style == OPENED || poly->style == OPENEDT1 || poly->style == OPENEDT2 ||
		poly->style == OPENEDT3 || poly->style == CLOSED)
	{
		for(i=0; i<poly->count; i++)
		{
			if (i == 0)
			{
				if (poly->style != CLOSED) continue;
				last = poly->count - 1;
			} else last = i-1;
			us_intersectsnapline(best, poly->xv[last],poly->yv[last], poly->xv[i],poly->yv[i],
				interpoly, wantx, wanty);
		}
		return;
	}
	if (poly->style == VECTORS)
	{
		for(i=0; i<poly->count; i += 2)
		{
			us_intersectsnapline(best, poly->xv[i],poly->yv[i], poly->xv[i+1],poly->yv[i+1],
				interpoly, wantx, wanty);
		}
		return;
	}
}

/*
 * routine to find the intersection between the line from (x1,y1) to (x2,y2) and polygon "interpoly".
 * This is set this as the snap point in highlight "best" (the cursor is at (wantx,wanty).
 */
void us_intersectsnapline(HIGHLIGHT *best, INTBIG x1, INTBIG y1, INTBIG x2, INTBIG y2,
	POLYGON *interpoly, INTBIG wantx, INTBIG wanty)
{
	REGISTER INTSML i, last, angle, interangle;
	INTBIG ix, iy, ix1, iy1, ix2, iy2;

	angle = figureangle(x1,y1, x2,y2);
	if (interpoly->style == OPENED || interpoly->style == OPENEDT1 || interpoly->style == OPENEDT2 ||
		interpoly->style == OPENEDT3 || interpoly->style == CLOSED)
	{
		for(i=0; i<interpoly->count; i++)
		{
			if (i == 0)
			{
				if (interpoly->style != CLOSED) continue;
				last = interpoly->count - 1;
			} else last = i-1;
			interangle = figureangle(interpoly->xv[last],interpoly->yv[last], interpoly->xv[i],interpoly->yv[i]);
			if (intersect(x1,y1, angle, interpoly->xv[last],interpoly->yv[last], interangle, &ix, &iy) < 0)
				continue;
			if (ix < mini(x1,x2)) continue;
			if (ix > maxi(x1,x2)) continue;
			if (iy < mini(y1,y2)) continue;
			if (iy > maxi(y1,y2)) continue;
			if (ix < mini(interpoly->xv[last],interpoly->xv[i])) continue;
			if (ix > maxi(interpoly->xv[last],interpoly->xv[i])) continue;
			if (iy < mini(interpoly->yv[last],interpoly->yv[i])) continue;
			if (iy > maxi(interpoly->yv[last],interpoly->yv[i])) continue;
			us_setbestsnappoint(best, wantx, wanty, ix, iy, 0, 0);
		}
		return;
	}
	if (interpoly->style == VECTORS)
	{
		for(i=0; i<interpoly->count; i += 2)
		{
			interangle = figureangle(interpoly->xv[i],interpoly->yv[i], interpoly->xv[i+1],interpoly->yv[i+1]);
			if (intersect(x1,y1, angle, interpoly->xv[i],interpoly->yv[i], interangle, &ix, &iy) < 0)
				continue;
			if (ix < mini(x1,x2)) continue;
			if (ix > maxi(x1,x2)) continue;
			if (iy < mini(y1,y2)) continue;
			if (iy > maxi(y1,y2)) continue;
			if (ix < mini(interpoly->xv[i],interpoly->xv[i+1])) continue;
			if (ix > maxi(interpoly->xv[i],interpoly->xv[i+1])) continue;
			if (iy < mini(interpoly->yv[i],interpoly->yv[i+1])) continue;
			if (iy > maxi(interpoly->yv[i],interpoly->yv[i+1])) continue;
			us_setbestsnappoint(best, wantx, wanty, ix, iy, 0, 0);
		}
		return;
	}
	if (interpoly->style == CIRCLEARC)
	{
		i = circlelineintersection(interpoly->xv[0], interpoly->yv[0], interpoly->xv[1], interpoly->yv[1],
			x1, y1, x2, y2, &ix1, &iy1, &ix2, &iy2, 0);
		if (i >= 1)
		{
			if (us_pointonarc(ix1, iy1, interpoly) != 0)
				us_setbestsnappoint(best, wantx, wanty, ix1, iy1, 0, 0);
			if (i >= 2)
			{
				if (us_pointonarc(ix2, iy2, interpoly) != 0)
					us_setbestsnappoint(best, wantx, wanty, ix2, iy2, 0, 0);
			}
		}
		return;
	}
	if (interpoly->style == CIRCLE)
	{
		i = circlelineintersection(interpoly->xv[0], interpoly->yv[0], interpoly->xv[1], interpoly->yv[1],
			x1, y1, x2, y2, &ix1, &iy1, &ix2, &iy2, 0);
		if (i >= 1)
		{
			us_setbestsnappoint(best, wantx, wanty, ix1, iy1, 0, 0);
			if (i >= 2)
			{
				us_setbestsnappoint(best, wantx, wanty, ix2, iy2, 0, 0);
			}
		}
		return;
	}
}

/*
 * Routine to adjust the two highlight modules "firsthigh" and "secondhigh" to account for the
 * fact that one or both has a tangent snap point that must be tangent to the other's snap point.
 */
void us_adjusttangentsnappoints(HIGHLIGHT *firsthigh, HIGHLIGHT *secondhigh)
{
	INTBIG fx, fy, sx, sy, pfx[4], pfy[4], psx[4], psy[4], ix1, iy1, ix2, iy2;
	REGISTER INTBIG frad, srad, rad, dist, bestdist;
	REGISTER INTSML j, k, dps, bestone;
	double ang, oang, dx, dy;
	static POLYGON *firstpoly = NOPOLYGON, *secondpoly = NOPOLYGON;
	POLYGON *swappoly;
	HIGHLIGHT *swaphighlight;
	REGISTER NODEINST *ni;
	XARRAY trans;

	/* get polygon describing first object */
	if ((firsthigh->status&HIGHSNAPTAN) != 0)
	{
		if (firstpoly == NOPOLYGON) firstpoly = allocpolygon(4, us_aid->cluster);
		if (firsthigh->fromgeom->entrytype != OBJNODEINST) return;
		ni = firsthigh->fromgeom->entryaddr.ni;
		if (ni->proto->index == 0) return;
		makerot(ni, trans);
		k = nodepolys(ni);
		for(j=0; j<k; j++)
		{
			shapenodepoly(ni, j, firstpoly);
			if (firstpoly->style == CIRCLEARC || firstpoly->style == CIRCLE ||
				firstpoly->style == DISC) break;
		}
		if (j >= k) return;
		xformpoly(firstpoly, trans);
	}

	/* get polygon describing second object */
	if ((secondhigh->status&HIGHSNAPTAN) != 0)
	{
		if (secondpoly == NOPOLYGON) secondpoly = allocpolygon(4, us_aid->cluster);
		if (secondhigh->fromgeom->entrytype != OBJNODEINST) return;
		ni = secondhigh->fromgeom->entryaddr.ni;
		if (ni->proto->index == 0) return;
		makerot(ni, trans);
		k = nodepolys(ni);
		for(j=0; j<k; j++)
		{
			shapenodepoly(ni, j, secondpoly);
			if (secondpoly->style == CIRCLEARC || secondpoly->style == CIRCLE ||
				secondpoly->style == DISC) break;
		}
		if (j >= k) return;
		xformpoly(secondpoly, trans);
	}

	if ((firsthigh->status&HIGHSNAPTAN) != 0)
	{
		if ((secondhigh->status&HIGHSNAPTAN) != 0)
		{
			/* tangent on both curves: find radii and make sure first is larger */
			frad = computedistance(firstpoly->xv[0], firstpoly->yv[0],
				firstpoly->xv[1], firstpoly->yv[1]);
			srad = computedistance(secondpoly->xv[0], secondpoly->yv[0],
				secondpoly->xv[1], secondpoly->yv[1]);
			if (frad < srad)
			{
				swappoly = firstpoly;       firstpoly = secondpoly;   secondpoly = swappoly;
				swaphighlight = firsthigh;  firsthigh = secondhigh;   secondhigh = swaphighlight;
				rad = frad;                 frad = srad;              srad = rad;
			}

			/* find tangent lines along outside of two circles */
			dps = 0;
			if (frad == srad)
			{
				/* special case when radii are equal: construct simple outside tangent lines */
				dx = (double)(secondpoly->xv[0]-firstpoly->xv[0]);
				dy = (double)(secondpoly->yv[0]-firstpoly->yv[0]);
				if (dx == 0.0 && dy == 0.0)
				{
					us_abortcommand("Domain error during tangent computation");
					return;
				}
				ang = atan2(dy, dx);
				oang = ang + EPI / 2.0;
				if (oang > EPI * 2.0) oang -= EPI * 2.0;
				pfx[dps] = firstpoly->xv[0] + rounddouble(cos(oang) * (double)frad);
				pfy[dps] = firstpoly->yv[0] + rounddouble(sin(oang) * (double)frad);
				psx[dps] = secondpoly->xv[0] + rounddouble(cos(oang) * (double)srad);
				psy[dps] = secondpoly->yv[0] + rounddouble(sin(oang) * (double)srad);
				dps++;

				oang = ang - EPI / 2.0;
				if (oang < -EPI * 2.0) oang += EPI * 2.0;
				pfx[dps] = firstpoly->xv[0] + rounddouble(cos(oang) * (double)frad);
				pfy[dps] = firstpoly->yv[0] + rounddouble(sin(oang) * (double)frad);
				psx[dps] = secondpoly->xv[0] + rounddouble(cos(oang) * (double)srad);
				psy[dps] = secondpoly->yv[0] + rounddouble(sin(oang) * (double)srad);
				dps++;
			} else
			{
				if (circletangents(secondpoly->xv[0], secondpoly->yv[0],
					firstpoly->xv[0], firstpoly->yv[0], firstpoly->xv[0]+frad-srad, firstpoly->yv[0],
						&ix1, &iy1, &ix2, &iy2) == 0)
				{
					dx = (double)(ix1-firstpoly->xv[0]);   dy = (double)(iy1-firstpoly->yv[0]);
					if (dx == 0.0 && dy == 0.0)
					{
						us_abortcommand("Domain error during tangent computation");
						return;
					}
					ang = atan2(dy, dx);
					pfx[dps] = firstpoly->xv[0] + rounddouble(cos(ang) * (double)frad);
					pfy[dps] = firstpoly->yv[0] + rounddouble(sin(ang) * (double)frad);
					psx[dps] = secondpoly->xv[0] + rounddouble(cos(ang) * (double)srad);
					psy[dps] = secondpoly->yv[0] + rounddouble(sin(ang) * (double)srad);
					dps++;

					dx = (double)(ix2-firstpoly->xv[0]);   dy = (double)(iy2-firstpoly->yv[0]);
					if (dx == 0.0 && dy == 0.0)
					{
						us_abortcommand("Domain error during tangent computation");
						return;
					}
					ang = atan2(dy, dx);
					pfx[dps] = firstpoly->xv[0] + rounddouble(cos(ang) * (double)frad);
					pfy[dps] = firstpoly->yv[0] + rounddouble(sin(ang) * (double)frad);
					psx[dps] = secondpoly->xv[0] + rounddouble(cos(ang) * (double)srad);
					psy[dps] = secondpoly->yv[0] + rounddouble(sin(ang) * (double)srad);
					dps++;
				}
			}

			/* find tangent lines that cross between two circles */
			if (circletangents(secondpoly->xv[0], secondpoly->yv[0],
				firstpoly->xv[0], firstpoly->yv[0], firstpoly->xv[0]+frad+srad, firstpoly->yv[0],
					&ix1, &iy1, &ix2, &iy2) == 0)
			{
				dx = (double)(ix1-firstpoly->xv[0]);   dy = (double)(iy1-firstpoly->yv[0]);
				if (dx == 0.0 && dy == 0.0)
				{
					us_abortcommand("Domain error during tangent computation");
					return;
				}
				ang = atan2(dy, dx);
				pfx[dps] = firstpoly->xv[0] + rounddouble(cos(ang) * (double)frad);
				pfy[dps] = firstpoly->yv[0] + rounddouble(sin(ang) * (double)frad);
				psx[dps] = secondpoly->xv[0] - rounddouble(cos(ang) * (double)srad);
				psy[dps] = secondpoly->yv[0] - rounddouble(sin(ang) * (double)srad);
				dps++;

				dx = (double)(ix2-firstpoly->xv[0]);   dy = (double)(iy2-firstpoly->yv[0]);
				if (dx == 0.0 && dy == 0.0)
				{
					us_abortcommand("Domain error during tangent computation");
					return;
				}
				ang = atan2(dy, dx);
				pfx[dps] = firstpoly->xv[0] + rounddouble(cos(ang) * (double)frad);
				pfy[dps] = firstpoly->yv[0] + rounddouble(sin(ang) * (double)frad);
				psx[dps] = secondpoly->xv[0] - rounddouble(cos(ang) * (double)srad);
				psy[dps] = secondpoly->yv[0] - rounddouble(sin(ang) * (double)srad);
				dps++;
			}

			/* screen out points that are not on arcs */
			k = 0;
			for(j=0; j<dps; j++)
			{
				if (firstpoly->style == CIRCLEARC && us_pointonarc(pfx[j], pfy[j], firstpoly) == 0)
					continue;
				if (secondpoly->style == CIRCLEARC && us_pointonarc(psx[j], psy[j], secondpoly) == 0)
					continue;
				pfx[k] = pfx[j];   pfy[k] = pfy[j];
				psx[k] = psx[j];   psy[k] = psy[j];
				k++;
			}
			dps = k;
			if (dps == 0) return;

			/* now find the tangent line that is closest to the snap points */
			us_getsnappoint(firsthigh, &fx, &fy);
			us_getsnappoint(secondhigh, &sx, &sy);
			for(j=0; j<dps; j++)
			{
				dist = computedistance(pfx[j],pfy[j], fx,fy) + computedistance(psx[j],psy[j], sx,sy);
				if (j == 0 || dist < bestdist)
				{
					bestdist = dist;
					bestone = j;
				}
			}

			/* set the best one */
			us_xformpointtonode(pfx[bestone], pfy[bestone], firsthigh->fromgeom->entryaddr.ni,
				&firsthigh->snapx, &firsthigh->snapy);
			us_xformpointtonode(psx[bestone], psy[bestone], secondhigh->fromgeom->entryaddr.ni,
				&secondhigh->snapx, &secondhigh->snapy);
		} else
		{
			/* compute tangent to first object */
			us_getsnappoint(secondhigh, &sx, &sy);
			us_adjustonetangent(firsthigh, firstpoly, sx, sy);
		}
	} else
	{
		if ((secondhigh->status&HIGHSNAPTAN) != 0)
		{
			us_getsnappoint(firsthigh, &fx, &fy);
			us_adjustonetangent(secondhigh, secondpoly, fx, fy);
		}
	}
}

/*
 * Routine to adjust the snap point on "high" so that it is tangent to its curved
 * polygon "poly" and runs through (x, y).
 */
void us_adjustonetangent(HIGHLIGHT *high, POLYGON *poly, INTBIG x, INTBIG y)
{
	REGISTER NODEINST *ni;
	INTBIG ix1, iy1, ix2, iy2, fx, fy;
	REGISTER INTBIG xv, yv;

	if (high->fromgeom->entrytype != OBJNODEINST) return;
	ni = high->fromgeom->entryaddr.ni;
	us_getsnappoint(high, &fx, &fy);
	if (circletangents(x, y, poly->xv[0], poly->yv[0], poly->xv[1], poly->yv[1],
		&ix1, &iy1, &ix2, &iy2) != 0) return;
	if (computedistance(fx, fy, ix1, iy1) > computedistance(fx, fy, ix2, iy2))
	{
		xv = ix1;   ix1 = ix2;   ix2 = xv;
		yv = iy1;   iy1 = iy2;   iy2 = yv;
	}

	if (poly->style != CIRCLEARC || us_pointonarc(ix1, iy1, poly) != 0)
	{
		us_xformpointtonode(ix1, iy1, ni, &high->snapx, &high->snapy);
		return;
	}
	if (poly->style != CIRCLEARC || us_pointonarc(ix2, iy2, poly) != 0)
	{
		us_xformpointtonode(ix2, iy2, ni, &high->snapx, &high->snapy);
		return;
	}
}

/*
 * Routine to adjust the two highlight modules "firsthigh" and "secondhigh" to account for the
 * fact that one or both has a perpendicular snap point that must be perpendicular
 * to the other's snap point.
 */
void us_adjustperpendicularsnappoints(HIGHLIGHT *firsthigh, HIGHLIGHT *secondhigh)
{
	INTBIG fx, fy;
	static POLYGON *secondpoly = NOPOLYGON;
	REGISTER NODEINST *ni;
	XARRAY trans;

	if ((secondhigh->status&HIGHSNAPPERP) != 0)
	{
		/* get polygon describing second object */
		if (secondpoly == NOPOLYGON) secondpoly = allocpolygon(4, us_aid->cluster);
		if (secondhigh->fromgeom->entrytype != OBJNODEINST) return;
		ni = secondhigh->fromgeom->entryaddr.ni;
		if (ni->proto->index == 0) return;
		makerot(ni, trans);
		(void)nodepolys(ni);
		shapenodepoly(ni, 0, secondpoly);
		xformpoly(secondpoly, trans);

		us_getsnappoint(firsthigh, &fx, &fy);
		us_adjustoneperpendicular(secondhigh, secondpoly, fx, fy);
	}
}

/*
 * Routine to adjust the snap point on "high" so that it is perpendicular to
 * polygon "poly" and point (x, y).
 */
void us_adjustoneperpendicular(HIGHLIGHT *high, POLYGON *poly, INTBIG x, INTBIG y)
{
	REGISTER NODEINST *ni;
	REGISTER INTBIG rad;
	INTBIG ix, iy;
	REGISTER INTSML ang;

	if (high->fromgeom->entrytype != OBJNODEINST) return;
	ni = high->fromgeom->entryaddr.ni;

	if (poly->style == CIRCLE || poly->style == CIRCLEARC)
	{
		/* compute perpendicular point */
		ang = figureangle(poly->xv[0], poly->yv[0], x, y);
		rad = computedistance(poly->xv[0], poly->yv[0], poly->xv[1], poly->yv[1]);
		ix = poly->xv[0] + mult(cosine(ang), rad);
		iy = poly->yv[0] + mult(sine(ang), rad);
		if (poly->style == CIRCLEARC || us_pointonarc(ix, iy, poly) == 0) return;
		us_xformpointtonode(ix, iy, ni, &high->snapx, &high->snapy);
		return;
	}

	/* handle straight line perpendiculars */
	ix = x;   iy = y;
	(void)closestpointtosegment(poly->xv[0], poly->yv[0], poly->xv[1], poly->yv[1], &ix, &iy);
	if (ix != x || iy != y) us_xformpointtonode(ix, iy, ni, &high->snapx, &high->snapy);
}

/*
 * routine to determine whether the point (x, y) is on the arc in "poly".
 * returns nonzero if so.
 */
INTSML us_pointonarc(INTBIG x, INTBIG y, POLYGON *poly)
{
	REGISTER INTSML angle, startangle, endangle;

	if (poly->style != CIRCLEARC) return(0);

	angle = figureangle(poly->xv[0],poly->yv[0], x,y);
	endangle = figureangle(poly->xv[0],poly->yv[0], poly->xv[1],poly->yv[1]);
	startangle = figureangle(poly->xv[0],poly->yv[0], poly->xv[2],poly->yv[2]);

	if (endangle > startangle)
	{
		if (angle >= startangle && angle <= endangle) return(1);
	} else
	{
		if (angle >= startangle || angle <= endangle) return(1);
	}
	return(0);
}

/*
 * routine to get the true coordinate of the snap point in "high" and place it in (x,y).
 */
void us_getsnappoint(HIGHLIGHT *high, INTBIG *x, INTBIG *y)
{
	REGISTER NODEINST *ni;
	XARRAY trans;
	INTBIG xt, yt;

	if (high->fromgeom->entrytype == OBJNODEINST)
	{
		ni = high->fromgeom->entryaddr.ni;
		makeangle(ni->rotation, ni->transpose, trans);
		xform(high->snapx, high->snapy, &xt, &yt, trans);
		*x = (ni->highx + ni->lowx) / 2 + xt;
		*y = (ni->highy + ni->lowy) / 2 + yt;
	} else
	{
		*x = (high->fromgeom->highx + high->fromgeom->lowx) / 2 + high->snapx;
		*y = (high->fromgeom->highy + high->fromgeom->lowy) / 2 + high->snapy;
	}
}

void us_setbestsnappoint(HIGHLIGHT *best, INTBIG wantx, INTBIG wanty, INTBIG newx, INTBIG newy, INTSML tan, INTSML perp)
{
	REGISTER INTBIG olddist, newdist;
	INTBIG oldx, oldy;

	if ((best->status & HIGHSNAP) != 0)
	{
		us_getsnappoint(best, &oldx, &oldy);
		olddist = computedistance(wantx, wanty, oldx, oldy);
		newdist = computedistance(wantx, wanty, newx, newy);
		if (newdist >= olddist) return;
	}

	/* set the snap point */
	if (best->fromgeom->entrytype == OBJNODEINST)
	{
		us_xformpointtonode(newx, newy, best->fromgeom->entryaddr.ni, &best->snapx, &best->snapy);
	} else
	{
		best->snapx = newx - (best->fromgeom->highx + best->fromgeom->lowx) / 2;
		best->snapy = newy - (best->fromgeom->highy + best->fromgeom->lowy) / 2;
	}
	best->status |= HIGHSNAP;
	if (tan != 0) best->status |= HIGHSNAPTAN;
	if (perp != 0) best->status |= HIGHSNAPPERP;
}

void us_xformpointtonode(INTBIG x, INTBIG y, NODEINST *ni, INTBIG *xo, INTBIG *yo)
{
	XARRAY trans;
	INTBIG xv, yv;

	if (ni->transpose != 0) makeangle(ni->rotation, ni->transpose, trans); else
		makeangle((INTSML)((3600 - ni->rotation)%3600), 0, trans);
	xv = x - (ni->highx + ni->lowx) / 2;
	yv = y - (ni->highy + ni->lowy) / 2;
	xform(xv, yv, xo, yo, trans);
}

static INTSML us_nodearcvarptr;
static INTSML us_nodenameflg;
static PORTEXPINST *us_nodeexpptr;

void us_initnodetext(NODEINST *ni, INTSML findport)
{
	us_nodearcvarptr = 0;
	if (findport == 0)
	{
		us_nodenameflg = 0;
	} else
	{
		us_nodenameflg = 1;

		/* if the "port" option is set and node text is disabled, skip it */
		if ((us_aid->aidstate&NOTEXTSELECT) != 0) us_nodearcvarptr = ni->numvar;
	}
	us_nodeexpptr = ni->firstportexpinst;
}

INTSML us_getnodetext(NODEINST *ni, WINDOW *win, POLYGON *poly, VARIABLE **var,
	PORTPROTO **port)
{
	INTBIG xc, yc;
	REGISTER INTSML skipch;
	REGISTER char *pt;
	REGISTER PORTEXPINST *pe;

	if (us_nodenameflg == 0)
	{
		us_nodenameflg = 1;
		if (ni->proto->index == 0 && (ni->userbits&NEXPAND) == 0)
		{
			*var = NOVARIABLE;
			*port = NOPORTPROTO;
			us_maketextpoly(describenodeproto(ni->proto), win,
				(ni->lowx + ni->highx) / 2, (ni->lowy + ni->highy) / 2,
					ni->geom, ni->textdescript, poly);
			poly->style = FILLED;
			return(0);
		}
	}
	for(;;)
	{
		if (us_nodearcvarptr >= ni->numvar) break;
		*var = &ni->firstvar[us_nodearcvarptr];
		*port = NOPORTPROTO;
		us_nodearcvarptr++;
		if (((*var)->type&VDISPLAY) == 0) continue;

		/* build polygon that surrounds text */
		if (((*var)->type&VISARRAY) != 0)
		{
			if (getlength(*var) > 1)
			{
				makerectpoly(ni->lowx, ni->highx, ni->lowy, ni->highy, poly);
				poly->style = FILLED;
				return(0);
			}
		}
		pt = describevariable(*var, 0, -1);
		skipch = ((*var)->type & VLENGTH) >> VLENGTHSH;
		if (skipch >= strlen(pt)) continue;
		us_maketextpoly(&pt[skipch], win, (ni->lowx + ni->highx) / 2,
			(ni->lowy + ni->highy) / 2, ni->geom, (*var)->textdescript, poly);
		poly->style = FILLED;
		return(0);
	}

	/* check exported ports on the node */
	if (us_nodeexpptr != NOPORTEXPINST)
	{
		pe = us_nodeexpptr;
		*port = pe->exportproto;
		*var = NOVARIABLE;
		us_nodeexpptr = pe->nextportexpinst;

		/* build polygon that surrounds text */
		portposition(ni, (*port)->subportproto, &xc, &yc);
		us_maketextpoly((*port)->protoname, win, xc, yc, ni->geom, (*port)->textdescript, poly);
		poly->style = FILLED;
		return(0);
	}

	return(1);
}

void us_initarctext(ARCINST *ai, INTSML findport)
{
	us_nodearcvarptr = 0;
	if (findport != 0)
	{
		/* if the "port" option is set and arc text is disabled, skip it */
		if ((us_aid->aidstate&NOTEXTSELECT) != 0) us_nodearcvarptr = ai->numvar;
	}
}

INTSML us_getarctext(ARCINST *ai, WINDOW *win, POLYGON *poly, VARIABLE **var)
{
	for(;;)
	{
		if (us_nodearcvarptr >= ai->numvar) break;
		*var = &ai->firstvar[us_nodearcvarptr];
		us_nodearcvarptr++;
		if (((*var)->type&VDISPLAY) == 0) continue;

		/* build polygon that surrounds text */
		us_maketextpoly(describevariable(*var, -1, -1), win,
			(ai->end[0].xpos + ai->end[1].xpos) / 2, (ai->end[0].ypos + ai->end[1].ypos) / 2,
				ai->geom, (*var)->textdescript, poly);
		poly->style = FILLED;
		return(0);
	}
	return(1);
}

/*
 * routine to return the object that is closest to point (rdx, rdy)
 * in facet "facet".  Searches nodes first.
 * This is used in the "create join-angle" command.
 */
GEOM *us_getclosest(INTBIG rdx, INTBIG rdy, NODEPROTO *facet)
{
	XARRAY trans;
	REGISTER GEOM *geom, *highgeom, *bestgeom;
	REGISTER INTBIG sea, wid, bestdist, dist;
	static POLYGON *poly = NOPOLYGON;
	REGISTER NODEINST *ni;
	REGISTER ARCINST *ai;
	REGISTER VARIABLE *var;
	HIGHLIGHT high;

	if (poly == NOPOLYGON) poly = allocpolygon(4, us_aid->cluster);

	highgeom = NOGEOM;
	var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_highlighted);
	if (var != NOVARIABLE)
	{
		if (getlength(var) == 1)
		{
			(void)us_makehighlight(((char **)var->addr)[0], &high);
			highgeom = high.fromgeom;
		}
	}

	/* see if there is a direct hit on another node */
	sea = initsearch(rdx, rdx, rdy, rdy, facet);
	bestdist = HUGEINT;
	for(;;)
	{
		geom = nextobject(sea);
		if (geom == NOGEOM) break;
		if (geom == highgeom) continue;
		if (geom->entrytype != OBJNODEINST) continue;
		ni = geom->entryaddr.ni;

		/* get the bounds of the node in a polygon */
		makerot(ni, trans);
		maketruerectpoly(ni->lowx, ni->highx, ni->lowy, ni->highy, poly);
		poly->style = FILLEDRECT;
		xformpoly(poly, trans);
		dist = polydistance(poly, rdx, rdy);
		if (dist > bestdist) continue;
		bestdist = dist;
		bestgeom = geom;
	}
	if (bestdist < 0) return(bestgeom);

	/* look at arcs second */
	bestdist = HUGEINT;
	sea = initsearch(rdx, rdx, rdy, rdy, facet);
	for(;;)
	{
		geom = nextobject(sea);
		if (geom == NOGEOM) break;
		if (geom == highgeom) continue;
		if (geom->entrytype != OBJARCINST) continue;
		ai = geom->entryaddr.ai;

		/* prepare to examine the arc */
		wid = ai->width - arcwidthoffset(ai->proto);
		if (wid == 0) wid = ai->proto->tech->deflambda;
		if (curvedarcoutline(ai, poly, FILLED, wid) != 0)
			makearcpoly(ai->length, wid, ai, poly, FILLED);

		dist = polydistance(poly, rdx, rdy);
		if (dist > bestdist) continue;
		bestdist = dist;
		bestgeom = geom;
	}
	if (bestdist < 0) return(bestgeom);
	return(NOGEOM);
}

/*
 * routine to build a polygon in "poly" that has four points describing the
 * text in "str" with descriptor "descript".  The text is in window "win"
 * on an object whose center is (xc,yc) and is on object "geom".
 */
void us_maketextpoly(char *str, WINDOW *win, INTBIG xc, INTBIG yc, GEOM *geom,
	INTBIG descript, POLYGON *poly)
{
	INTBIG xw, yw, newxc, newyc, lambda;
	XARRAY trans;
	INTSML tsx, tsy;

	/* determine size of text */
	us_settextsize(win, truefontsize((INTSML)((descript&VTSIZE)>>VTSIZESH), win, el_curtech));
	us_textsize(win, str, &tsx, &tsy);
	xw = muldiv(tsx, win->screenhx - win->screenlx, win->usehx - win->uselx);
	yw = muldiv(tsy, win->screenhy - win->screenly, win->usehy - win->usely);

	/* determine location of text */
	if (geom->entrytype == OBJNODEINST) makeangle(geom->entryaddr.ni->rotation,
		geom->entryaddr.ni->transpose, trans); else
			transid(trans);

	lambda = figurelambda(geom);
	newxc = (descript&VTXOFF)>>VTXOFFSH;
	if ((descript&VTXOFFNEG) != 0) newxc = -newxc;
	newxc = newxc * lambda / 4;
	newyc = (descript&VTYOFF)>>VTYOFFSH;
	if ((descript&VTYOFFNEG) != 0) newyc = -newyc;
	newyc = newyc * lambda / 4;
	xform(newxc, newyc, &newxc, &newyc, trans);
	xc += newxc;   yc += newyc;
	poly->style = rotatelabel(poly->style, trans);

	switch (poly->style)
	{
		case TEXTTOP:                    yc -= yw/2;   break;
		case TEXTBOT:                    yc += yw/2;   break;
		case TEXTLEFT:     xc += xw/2;                 break;
		case TEXTRIGHT:    xc -= xw/2;                 break;
		case TEXTTOPLEFT:  xc += xw/2;   yc -= yw/2;   break;
		case TEXTBOTLEFT:  xc += xw/2;   yc += yw/2;   break;
		case TEXTTOPRIGHT: xc -= xw/2;   yc -= yw/2;   break;
		case TEXTBOTRIGHT: xc -= xw/2;   yc += yw/2;   break;
	}

	/* construct polygon with actual size */
	poly->xv[0] = xc - xw/2;   poly->yv[0] = yc - yw/2;
	poly->xv[1] = xc - xw/2;   poly->yv[1] = yc + yw/2;
	poly->xv[2] = xc + xw/2;   poly->yv[2] = yc + yw/2;
	poly->xv[3] = xc + xw/2;   poly->yv[3] = yc - yw/2;
	poly->count = 4;
	poly->layer = -1;
	poly->style = CLOSED;
}

void us_erasenodeinst(NODEINST *ni)
{
	REGISTER PORTARCINST *pi, *npi;
	REGISTER ARCINST *ai;
	REGISTER NODEINST *ni1, *ni2;

	/* erase all connecting arcs to this nodeinst */
	for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = npi)
	{
		npi = pi->nextportarcinst;

		/* don't delete if already dead */
		ai = pi->conarcinst;
		if ((ai->userbits&DEADA) != 0) continue;

		/* see if nodes need to be undrawn to account for "Steiner Point" changes */
		ni1 = ai->end[0].nodeinst;   ni2 = ai->end[1].nodeinst;
		if ((ni1->proto->userbits&WIPEON1OR2) != 0) startobjectchange((INTBIG)ni1, VNODEINST);
		if ((ni2->proto->userbits&WIPEON1OR2) != 0) startobjectchange((INTBIG)ni2, VNODEINST);

		startobjectchange((INTBIG)ai, VARCINST);
		if (killarcinst(ai)) ttyputerr("Error killing arc");

		/* see if nodes need to be redrawn to account for "Steiner Point" changes */
		if ((ni1->proto->userbits&WIPEON1OR2) != 0) endobjectchange((INTBIG)ni1, VNODEINST);
		if ((ni2->proto->userbits&WIPEON1OR2) != 0) endobjectchange((INTBIG)ni2, VNODEINST);
	}

	/* see if this nodeinst is a port of the facet */
	startobjectchange((INTBIG)ni, VNODEINST);
	if (ni->firstportexpinst != NOPORTEXPINST) us_undoportproto(ni, NOPORTPROTO);

	/* now erase the nodeinst */
	if (killnodeinst(ni)) ttyputerr("Error from killnodeinst");
}

/*
 * routine to kill a node between two arcs and join the arc as one.  Returns an error
 * code according to its success.
 */
INTSML us_erasepassthru(NODEINST *ni, INTSML allowdiffs)
{
	INTSML i, j;
	PORTARCINST *pi;
	NODEINST *reconno[2];
	PORTPROTO *reconpt[2];
	INTBIG reconx[2], recony[2], wid, bits, dx[2], dy[2];
	ARCINST *ai, *reconar[2];

	/* look for two arcs that will get merged */
	j = 0;
	for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
	{
		if (j >= 2) { j = 0;   break; }
		reconar[j] = pi->conarcinst;
		for(i=0; i<2; i++) if (pi->conarcinst->end[i].nodeinst != ni)
		{
			reconno[j] = pi->conarcinst->end[i].nodeinst;
			reconpt[j] = pi->conarcinst->end[i].portarcinst->proto;
			reconx[j] = pi->conarcinst->end[i].xpos;
			dx[j] = reconx[j] - pi->conarcinst->end[i==0].xpos;
			recony[j] = pi->conarcinst->end[i].ypos;
			dy[j] = recony[j] - pi->conarcinst->end[i==0].ypos;
		}
		j++;
	}
	if (j != 2) return(j);

	/* verify that the two arcs to merge have the same type */
	if (reconar[0]->proto != reconar[1]->proto) return(-1);

	if (allowdiffs == 0)
	{
		/* verify that the two arcs to merge have the same width */
		if (reconar[0]->width != reconar[1]->width) return(-2);

		/* verify that the two arcs have the same slope */
		if ((dx[1]*dy[0]) != (dx[0]*dy[1])) return(-3);
	}

	/* remember facts about the new arcinst */
	wid = reconar[0]->width;
	bits = reconar[0]->userbits | reconar[1]->userbits;

	/* special code to handle directionality */
	if ((bits&(ISDIRECTIONAL|ISNEGATED|NOTEND0|NOTEND1|REVERSEEND)) != 0)
	{
		/* reverse ends if the arcs point the wrong way */
		for(i=0; i<2; i++)
			if (reconar[i]->end[i].nodeinst == ni)
				if ((reconar[i]->userbits&REVERSEEND) == 0)
					reconar[i]->userbits |= REVERSEEND; else
						reconar[i]->userbits &= ~REVERSEEND;
		bits = reconar[0]->userbits | reconar[1]->userbits;

		/* two negations make a positive */
		if ((reconar[0]->userbits&ISNEGATED) != 0 &&
			(reconar[1]->userbits&ISNEGATED) != 0) bits &= ~ISNEGATED;
	}

	/* erase the nodeinst, as requested (this will erase connecting arcs) */
	us_erasenodeinst(ni);

	/* make the new arcinst */
	ai = newarcinst(reconar[0]->proto, wid, bits, reconno[0], reconpt[0], reconx[0], recony[0],
		reconno[1], reconpt[1], reconx[1], recony[1], ni->parent);
	if (ai == NOARCINST) return(-5);

	(void)copyvars((INTBIG)reconar[0], VARCINST, (INTBIG)ai, VARCINST);
	(void)copyvars((INTBIG)reconar[1], VARCINST, (INTBIG)ai, VARCINST);
	endobjectchange((INTBIG)ai, VARCINST);
	ai->changed = 0;
	return(2);
}

/*
 * routine to determine a path down from the currently highlighted port.  Returns
 * the subnode and subport in "hini" and "hipp" (sets them to NONODEINST and
 * NOPORTPROTO if no lower path is defined).
 */
void us_findlowerport(NODEINST **hini, PORTPROTO **hipp)
{
	HIGHLIGHT high;
	REGISTER VARIABLE *var;
	REGISTER INTSML len;
	NODEINST *ni;
	PORTPROTO *pp;

	/* presume no lower port */
	*hini = NONODEINST;
	*hipp = NOPORTPROTO;

	/* must be 1 highlighted object */
	var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_highlighted);
	if (var == NOVARIABLE) return;
	len = getlength(var);
	if (len > 1) return;

	/* get the highlighted object */
	if (us_makehighlight(((char **)var->addr)[0], &high) != 0) return;

	/* see if it is a port */
	if ((high.status&HIGHTYPE) == HIGHTEXT)
	{
		if (high.fromvar != NOVARIABLE || high.fromport == NOPORTPROTO) return;
		pp = high.fromport->subportproto;
	} else
	{
		if ((high.status&HIGHTYPE) != HIGHFROM || high.fromport == NOPORTPROTO) return;
		pp = high.fromport;
	}

	/* see if port is on instance */
	ni = high.fromgeom->entryaddr.ni;
	if (ni->proto->index != 0) return;

	/* describe source of the port */
	*hini = pp->subnodeinst;
	*hipp = pp->subportproto;
}

/*
 * routine to determine a path from the currently highlighted node.  If an
 * exported port is selected, the path out to all instances is checked and
 * a facet with an instance is returned (the instance is returned in "hini").
 * Returns the facet to edit, or zero if the command was aborted, or NONODEPROTO
 * if this operation does not apply.
 *
 * This routine originally written by Burnie West, Schlumberger ATE,
 */
NODEPROTO *us_followexportedportup(NODEINST **hini, PORTPROTO **hipp)
{
	HIGHLIGHT high;
	REGISTER VARIABLE *var;
	NODEPROTO *np, *inp, **newfacetlist;
	NETWORK **newnetlist;
	NODEINST *ni, **newinstlist;
	PORTPROTO *pp, *ipp, **newportlist;
	PORTARCINST *pi;
	PORTEXPINST *pe;
	POPUPMENU *pm, *cpopup;
	POPUPMENUITEM *mi, *selected;
	INTSML butstate, facetcount, i, k, *newinstcount, len;
	char buf[50];
	static INTSML listsize = 0;
	static NODEPROTO **facetlist;
	static INTSML *instcount;
	static PORTPROTO **portlist;
	static NODEINST **instlist;
	static NETWORK **netlist;

	/* must be 1 highlighted object */
	var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_highlighted);
	if (var == NOVARIABLE) return(NONODEPROTO);
	len = getlength(var);
	if (len > 1) return(NONODEPROTO);

	/* get the highlighted object */
	if (us_makehighlight(((char **)var->addr)[0], &high) != 0) return(NONODEPROTO);

	if ((high.status&HIGHTYPE) == HIGHTEXT)
	{
		if (high.fromvar != NOVARIABLE || high.fromport == NOPORTPROTO) return(NONODEPROTO);
		pp = high.fromport->subportproto;
	} else
	{
		if ((high.status&HIGHTYPE) != HIGHFROM || high.fromport == NOPORTPROTO) return(NONODEPROTO);
		pp = high.fromport;
	}
	ni = high.fromgeom->entryaddr.ni;
	np = ni->proto;

	/* see if port is exported */
	for (pe = ni->firstportexpinst; pe != NOPORTEXPINST; pe = pe->nextportexpinst)
		if (pe->proto == pp) break;
	if (pe == NOPORTEXPINST) return(NONODEPROTO);

	/* find true parent */
	pp = pe->exportproto;
	if ((inp = iconview(ni->parent)) == NONODEPROTO) inp = ni->parent;
	ipp = equivalentport(ni->parent, pp, inp);
	if (ipp == NOPORTPROTO) return(NONODEPROTO);

	/* make a list of choices, up the hierarchy */
	facetcount = 0;
	for(ni = inp->firstinst; ni != NONODEINST; ni = ni->nextinst)
	{
		/* ignore this instance if it is a duplicate */
		for (i=0; i<facetcount; i++) if (facetlist[i] == ni->parent) break;
		if (i < facetcount)
		{
			instcount[i]++;
			continue;
		}

		/* ensure room in the list */
		if (facetcount >= listsize)
		{
			k = listsize + 32;
			newfacetlist = (NODEPROTO **)emalloc(k * (sizeof (NODEPROTO *)), us_aid->cluster);
			newinstlist = (NODEINST **)emalloc(k * (sizeof (NODEINST *)), us_aid->cluster);
			newinstcount = (INTSML *)emalloc(k * SIZEOFINTSML, us_aid->cluster);
			newportlist = (PORTPROTO **)emalloc(k * (sizeof (PORTPROTO *)), us_aid->cluster);
			newnetlist = (NETWORK **)emalloc(k * (sizeof (NETWORK *)), us_aid->cluster);
			if (newfacetlist == 0 || newportlist == 0 || newnetlist == 0) return(0);
			for(i=0; i<facetcount; i++)
			{
				newfacetlist[i] = facetlist[i];
				newinstlist[i] = instlist[i];
				newinstcount[i] = instcount[i];
				newportlist[i] = portlist[i];
				newnetlist[i] = netlist[i];
			}
			if (listsize != 0)
			{
				efree((char *)facetlist);
				efree((char *)instlist);
				efree((char *)instcount);
				efree((char *)portlist);
				efree((char *)netlist);
			}
			facetlist = newfacetlist;
			instlist = newinstlist;
			instcount = newinstcount;
			portlist = newportlist;
			netlist = newnetlist;
			listsize = k;
		}

		facetlist[facetcount] = ni->parent;
		instlist[facetcount]  = ni;
		instcount[facetcount] = 1;
		portlist[facetcount]  = ipp;
		for(pe = ni->firstportexpinst; pe != NOPORTEXPINST; pe = pe->nextportexpinst)
			if (pe->proto == ipp) break;
		if (pe != NOPORTEXPINST) netlist[facetcount] = pe->exportproto->network; else
		{
			for (pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
				if (pi->proto == ipp) break;
			if (pi != NOPORTARCINST) netlist[facetcount] = pi->conarcinst->network; else
				netlist[facetcount] = NONETWORK;
		}
		facetcount++;
	}

	/* if no instances of this facet found, exit */
	if (facetcount == 0) return(NONODEPROTO);

	/* if only one instance, answer is easy */
	if (facetcount == 1)
	{
		*hini = instlist[0];
		*hipp = portlist[0];
		return(facetlist[0]);
	}

	/* make a menu of all facets connected to this exported port */
	pm = (POPUPMENU *)emalloc(sizeof(POPUPMENU), el_tempcluster);
	if (pm == 0) return(0);

	mi = (POPUPMENUITEM *)emalloc(facetcount * (sizeof (POPUPMENUITEM)), el_tempcluster);
	if (mi == 0)
	{
		efree((char *)pm);
		return(0);
	}
	for (i=0; i<facetcount; i++)
	{
		(void)allocstring(&mi[i].attribute, facetlist[i]->cell->cellname, el_tempcluster);

		(void)initinfstr();
		if (netlist[i] == NONETWORK)
		{
			(void)addstringtoinfstr("No net");
		} else if (netlist[i]->namecount < 0 || netlist[i]->netname == NOSTRING)
		{
			(void)addstringtoinfstr("Unnamed net");
		} else
		{
			(void)addstringtoinfstr("Net ");
			(void)addstringtoinfstr(netlist[i]->netname);
		}
		if (instcount[i] > 1)
		{
			(void)sprintf(buf, "; %d instances", instcount[i]);
			(void)addstringtoinfstr(buf);
		}
		(void)allocstring(&mi[i].value, returninfstr(), el_tempcluster);
		mi[i].valueparse = NOCOMCOMP;
		mi[i].maxlen = -1;
		mi[i].response = NOUSERCOM;
		mi[i].changed = 0;
	}
	pm->name = "noname";
	pm->list = mi;
	pm->total = facetcount;
	pm->header = "Which facet up the hierarchy?";

	/* display and select from the menu */
	butstate = 0;
	cpopup = pm;
	selected = us_popupmenu(&cpopup, &butstate, 1, -1, -1, 0);

	/* free up allocated menu space */
	for (k=0; k<facetcount; k++)
	{
		efree(mi[k].attribute);
		efree(mi[k].value);
	}
	efree((char *)mi);
	efree((char *)pm);

	/* stop if display doesn't support popup menus */
	if (selected == 0) return(NONODEPROTO);

	if (selected == NOPOPUPMENUITEM) return(0);
	for (i=0; i<facetcount; i++) if (selected == &mi[i]) break;

	/* selected entry is invalid */
	if (i >= facetcount) return(0);

	/* return the selected entry */
	*hini = instlist[i];
	*hipp = portlist[i];
	return(facetlist[i]);
}

/*
 * routine to re-export port "pp" on nodeinst "ni".  Returns nonzero if there
 * is an error
 */
INTSML us_reexportport(PORTPROTO *pp, NODEINST *ni)
{
	REGISTER INTSML i;
	REGISTER VARIABLE *var;
	char *portname;
	REGISTER PORTPROTO *ppt;

	/* generate an initial guess for the new port name */
	i = initinfstr();
	i += addstringtoinfstr(pp->protoname);

	/* add in local node name if applicable */
	var = getvalkey((INTBIG)ni, VNODEINST, VSTRING, el_node_name);
	if (var != NOVARIABLE) i += addstringtoinfstr((char *)var->addr);

	if (i != 0)
	{
		ttyputerr("Memory low");
		return(1);
	}
	portname = us_uniqueportname(returninfstr(), ni->parent);

	/* export the port */
	ttyputmsg("Exporting port %s of node %s as %s", pp->protoname, describenodeinst(ni), portname);
	startobjectchange((INTBIG)ni, VNODEINST);
	ppt = newportproto(ni->parent, ni, pp, portname);
	if (ppt == NOPORTPROTO)
	{
		us_abortcommand("Error creating port %s", portname);
		return(1);
	}
	ppt->textdescript = pp->textdescript;
	endobjectchange((INTBIG)ni, VNODEINST);
	return(0);
}

/*
 * routine to rename port "pp" to be "pt"
 */
void us_renameport(PORTPROTO *pp, char *pt)
{
	char *ch;
	REGISTER PORTPROTO *opp;
	REGISTER NODEINST *ni;
	REGISTER NODEPROTO *np;
	extern AIDENTRY *net_aid;

	/* check for duplicate name */
	if (strcmp(pp->protoname, pt) == 0)
	{
		ttyputmsg("Port name has not changed");
		return;
	}

	np = pp->parent;
	for(opp = np->firstportproto; opp != NOPORTPROTO; opp = opp->nextportproto)
		if (opp != pp && namesame(opp->protoname, pt) == 0) break;
	if (opp == NOPORTPROTO) ch = pt; else
	{
		ch = us_uniqueportname(pt, np);
		ttyputmsg("Already a port called %s, calling this %s", pt, ch);
	}

	/* look at all instances of this nodeproto for use on display */
	startobjectchange((INTBIG)pp->subnodeinst, VNODEINST);
	for(ni = np->firstinst; ni != NONODEINST; ni = ni->nextinst)
	{
		/* no port names in expanded facets */
		if ((ni->userbits & NEXPAND) == 0) startobjectchange((INTBIG)ni, VNODEINST);
	}

	/* change the port name */
	ttyputmsgf("Port %s renamed to %s", pp->protoname, ch);
	(void)setval((INTBIG)pp, VPORTPROTO, "protoname", (INTBIG)ch, VSTRING);

	/* look at all instances of this nodeproto for use on display */
	for(ni = np->firstinst; ni != NONODEINST; ni = ni->nextinst)
	{
		/* no port names in expanded facets */
		if ((ni->userbits & NEXPAND) == 0) endobjectchange((INTBIG)ni, VNODEINST);
	}
	endobjectchange((INTBIG)pp->subnodeinst, VNODEINST);

	/* tell the network maintainter to reevaluate this facet */
	(void)askaid(net_aid, "re-number", (INTBIG)pp->parent);
}

/*
 * routine to yank the contents of complex node instance "topno" into its
 * parent facet.  If "keepports" is nonzero, keep any exported ports in the
 * facet
 */
void us_yankonenode(NODEINST *topno, INTSML keepports)
{
	REGISTER NODEINST *ni, *newni;
	REGISTER ARCINST *ai, *newar;
	REGISTER PORTARCINST *pi, *nextpi;
	REGISTER PORTEXPINST *pe, *nextpe;
	REGISTER PORTPROTO *pp, *newpp;
	REGISTER NODEPROTO *np;
	REGISTER ARCPROTO *ap;
	NODEINST *noa[2];
	PORTPROTO *pta[2];
	REGISTER INTBIG wid, i, oldbits, lowx, highx, lowy, highy;
	XARRAY localtrans, localrot, trans;
	INTBIG nox[2], noy[2], newxc, newyc, xc, yc;
	INTSML newang;
	static POLYGON *poly = NOPOLYGON;

	/* get polygon */
	if (poly == NOPOLYGON) poly = allocpolygon(4, us_aid->cluster);

	/* make transformation matrix for this facet */
	np = topno->proto;
	maketrans(topno, localtrans);
	makerot(topno, localrot);
	transmult(localtrans, localrot, trans);

	/* copy the nodes */
	for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		/* do not yank "facet center" primitives */
		if (ni->proto == gen_facetcenterprim)
		{
			ni->temp1 = (INTBIG)NONODEINST;
			continue;
		}

		/* this "center" computation is unstable for odd size nodes */
		xc = (ni->lowx + ni->highx) / 2;   yc = (ni->lowy + ni->highy) / 2;
		xform(xc, yc, &newxc, &newyc, trans);
		lowx = ni->lowx + newxc - xc;
		lowy = ni->lowy + newyc - yc;
		highx = ni->highx + newxc - xc;
		highy = ni->highy + newyc - yc;
		if (ni->transpose == 0) newang = ni->rotation + topno->rotation; else
			newang = ni->rotation + 3600 - topno->rotation;
		newang = newang % 3600;   if (newang < 0) newang += 3600;
		newni = newnodeinst(ni->proto, lowx, highx, lowy, highy,
			(INTSML)((ni->transpose+topno->transpose)&1), newang, topno->parent);
		if (newni == NONODEINST)
		{
			us_abortcommand("Cannot create node in this facet");
			return;
		}
		ni->temp1 = (INTBIG)newni;
		newni->userbits = ni->userbits;
		newni->textdescript = ni->textdescript;
		(void)copyvars((INTBIG)ni, VNODEINST, (INTBIG)newni, VNODEINST);
		endobjectchange((INTBIG)newni, VNODEINST);
	}

	/* copy the arcs */
	for(ai = np->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
	{
		/* ignore arcs connected to nodes that didn't get yanked */
		if ((NODEINST *)ai->end[0].nodeinst->temp1 == NONODEINST ||
			(NODEINST *)ai->end[1].nodeinst->temp1 == NONODEINST) continue;

		xform(ai->end[0].xpos, ai->end[0].ypos, &nox[0], &noy[0], trans);
		xform(ai->end[1].xpos, ai->end[1].ypos, &nox[1], &noy[1], trans);

		/* make sure end 0 fits in the port */
		shapeportpoly((NODEINST *)ai->end[0].nodeinst->temp1, ai->end[0].portarcinst->proto, poly, 0);
		if (isinside(nox[0], noy[0], poly) == 0)
			portposition((NODEINST *)ai->end[0].nodeinst->temp1,
				ai->end[0].portarcinst->proto, &nox[0], &noy[0]);

		/* make sure end 1 fits in the port */
		shapeportpoly((NODEINST *)ai->end[1].nodeinst->temp1, ai->end[1].portarcinst->proto, poly, 0);
		if (isinside(nox[1], noy[1], poly) == 0)
			portposition((NODEINST *)ai->end[1].nodeinst->temp1,
				ai->end[1].portarcinst->proto, &nox[1], &noy[1]);

		newar = newarcinst(ai->proto, ai->width, ai->userbits, (NODEINST *)ai->end[0].nodeinst->temp1,
			ai->end[0].portarcinst->proto, nox[0], noy[0], (NODEINST *)ai->end[1].nodeinst->temp1,
				ai->end[1].portarcinst->proto, nox[1], noy[1], topno->parent);
		if (newar == NOARCINST)
		{
			us_abortcommand("Cannot create arc in this facet");
			return;
		}
		(void)copyvars((INTBIG)ai, VARCINST, (INTBIG)newar, VARCINST);
		for(i=0; i<2; i++)
			(void)copyvars((INTBIG)ai->end[i].portarcinst, VPORTARCINST,
				(INTBIG)newar->end[i].portarcinst, VPORTARCINST);
		endobjectchange((INTBIG)newar, VARCINST);
	}

	/* replace arcs to this facet */
	for(pi = topno->firstportarcinst; pi != NOPORTARCINST; pi = nextpi)
	{
		/* remember facts about this arcinst */
		nextpi = pi->nextportarcinst;
		ai = pi->conarcinst;  ap = ai->proto;
		wid = ai->width;  oldbits = ai->userbits;
		for(i=0; i<2; i++)
		{
			noa[i] = ai->end[i].nodeinst;
			pta[i] = ai->end[i].portarcinst->proto;
			nox[i] = ai->end[i].xpos;   noy[i] = ai->end[i].ypos;
			if (noa[i] != topno) continue;
			noa[i] = (NODEINST *)ai->end[i].portarcinst->proto->subnodeinst->temp1;
			pta[i] = ai->end[i].portarcinst->proto->subportproto;
		}
		if (noa[0] == NONODEINST || noa[1] == NONODEINST) continue;
		startobjectchange((INTBIG)ai, VARCINST);
		if (killarcinst(ai)) ttyputerr("Error killing arc");
		newar = newarcinst(ap, wid, oldbits, noa[0], pta[0], nox[0], noy[0],
			noa[1], pta[1], nox[1], noy[1], topno->parent);
		if (newar == NOARCINST)
		{
			us_abortcommand("Cannot create arc to this facet");
			return;
		}

		/* copy variables (this presumes killed arc is not yet deallocated) */
		(void)copyvars((INTBIG)ai, VARCINST, (INTBIG)newar, VARCINST);
		for(i=0; i<2; i++)
			(void)copyvars((INTBIG)ai->end[i].portarcinst, VPORTARCINST,
				(INTBIG)newar->end[i].portarcinst, VPORTARCINST);
		endobjectchange((INTBIG)newar, VARCINST);
	}

	/* replace the exported ports */
	for(pe = topno->firstportexpinst; pe != NOPORTEXPINST; pe = nextpe)
	{
		nextpe = pe->nextportexpinst;
		pp = pe->proto;
		if ((NODEINST *)pp->subnodeinst->temp1 == NONODEINST) continue;
		if (moveportproto(topno->parent, pe->exportproto,
			(NODEINST *)pp->subnodeinst->temp1, pp->subportproto))
				ttyputerr("Moveportproto error");
	}

	/* copy the exported ports if requested */
	if (keepports != 0)
		for(pp=np->firstportproto; pp!=NOPORTPROTO; pp=pp->nextportproto)
	{
		if ((NODEINST *)pp->subnodeinst->temp1 == NONODEINST) continue;
		newpp = newportproto(topno->parent,
			(NODEINST *)pp->subnodeinst->temp1, pp->subportproto, pp->protoname);
		if (newpp == NOPORTPROTO)
			ttyputerr("Cannot keep port %s", pp->protoname); else
		{
			newpp->userbits = (newpp->userbits & ~STATEBITS) | (pp->userbits & STATEBITS);
			newpp->textdescript = pp->textdescript;
			(void)copyvars((INTBIG)pp, VPORTPROTO, (INTBIG)newpp, VPORTPROTO);
			(void)copyvars((INTBIG)pp->subportexpinst, VPORTEXPINST,
				(INTBIG)newpp->subportexpinst, VPORTEXPINST);
		}
	}

	/* delete the facet */
	startobjectchange((INTBIG)topno, VNODEINST);
	if (killnodeinst(topno)) ttyputerr("Killnodeinst error");
}

/*
 * routine to handle the motion of the text object in "high".  The specific
 * command for motion is in the "count" strings in "par"
 */
void us_movetext(HIGHLIGHT *high, INTSML count, char *par[])
{
	REGISTER INTSML l, len, itemHit;
	REGISTER INTBIG descript, lambda, amt;
	INTBIG xcur, ycur, xc, yc, xw, yw, dx, dy;
	XARRAY trans;
	INTSML tsx, tsy;
	REGISTER NODEINST *ni;
	REGISTER ARCINST *ai;
	REGISTER NODEPROTO *np;
	REGISTER char *str, *pp;
	extern DIALOG usr_movetodialog;

	/* turn off all highlighting */
	us_clearhighlightcount();

	/* no arguments: simple motion */
	if (count == 0)
	{
		/* move text to cursor position: get co-ordinates of cursor */
		if (us_demandxy(&xcur, &ycur)) return;
		gridalign(&xcur, &ycur, us_alignment);

		/* make sure the cursor is in the right facet */
		np = us_needfacet();
		if (np == NONODEPROTO) return;
		if (np != high->facet)
		{
			us_abortcommand("Cannot move text to another facet");
			(void)us_addhighlight(high);
			return;
		}

		/* get descriptor and text string */
		ni = NONODEINST;
		if (high->fromvar != NOVARIABLE)
		{
			if (high->fromgeom->entrytype == OBJNODEINST)
			{
				ni = high->fromgeom->entryaddr.ni;
				xc = (ni->lowx + ni->highx) / 2;
				yc = (ni->lowy + ni->highy) / 2;
			} else
			{
				ai = high->fromgeom->entryaddr.ai;
				xc = (ai->end[0].xpos + ai->end[1].xpos) / 2;
				yc = (ai->end[0].ypos + ai->end[1].ypos) / 2;
			}
			descript = high->fromvar->textdescript;
			str = describevariable(high->fromvar, -1, -1);
		} else if (high->fromport != NOPORTPROTO)
		{
			ni = high->fromgeom->entryaddr.ni;
			descript = high->fromport->textdescript;
			str = high->fromport->protoname;
			portposition(high->fromport->subnodeinst, high->fromport->subportproto, &xc, &yc);
		} else if (high->fromgeom->entrytype == OBJNODEINST)
		{
			ni = high->fromgeom->entryaddr.ni;
			descript = ni->textdescript;
			str = describenodeproto(ni->proto);
			xc = (ni->lowx + ni->highx) / 2;
			yc = (ni->lowy + ni->highy) / 2;
		}

		/* determine number of lines of text and text size */
		len = 1;
		if (high->fromvar != NOVARIABLE && (high->fromvar->type&VISARRAY) != 0)
			len = getlength(high->fromvar);
		if (len > 1)
		{
			xw = high->fromgeom->highx - high->fromgeom->lowx;
			yw = high->fromgeom->highy - high->fromgeom->lowy;
		} else
		{
			us_settextsize(el_curwindow, truefontsize((INTSML)((descript&VTSIZE)>>VTSIZESH),
				el_curwindow, el_curtech));
			us_textsize(el_curwindow, str, &tsx, &tsy);
			xw = muldiv(tsx, el_curwindow->screenhx-el_curwindow->screenlx,
				el_curwindow->usehx-el_curwindow->uselx);
			yw = muldiv(tsy, el_curwindow->screenhy-el_curwindow->screenly,
				el_curwindow->usehy-el_curwindow->usely);
		}

		/* adjust the cursor position if selecting interactively */
		if ((us_aid->aidstate&INTERACTIVE) != 0 &&
			us_graphicshas(CANTRACKCURSOR) != 0)
		{
			us_textmoveinit(xc, yc, xw, yw, descript, high->fromgeom);
			trackcursor(0, us_ignoreup, us_textmovebegin, us_textmovedown,
				us_stoponchar, us_dragup, TRACKDRAGGING);
			if (el_pleasestop != 0) return;
			if (us_demandxy(&xcur, &ycur) != 0) return;
			gridalign(&xcur, &ycur, us_alignment);
		}

		dx = xcur-xc;    dy = ycur-yc;

		/* move entire node if it is an "invisible pin" */
		if (ni != NONODEINST && ni->proto == gen_invispinprim)
		{
			startobjectchange((INTBIG)ni, VNODEINST);
			modifynodeinst(ni, dx, dy, dx, dy, 0, 0);
			endobjectchange((INTBIG)ni, VNODEINST);
			(void)us_addhighlight(high);
			return;
		}

		/* undraw the text */
		startobjectchange((INTBIG)high->fromgeom->entryaddr.blind,
			high->fromgeom->entrytype == OBJNODEINST ? VNODEINST : VARCINST);

		/* set the new descriptor on the text */
		if (ni != NONODEINST)
		{
			if (ni->transpose != 0)
				makeangle(ni->rotation, ni->transpose, trans); else
					makeangle((INTSML)((3600-ni->rotation)%3600), 0, trans);
			xform(dx, dy, &dx, &dy, trans);
		}
		lambda = figurelambda(high->fromgeom);
		descript = us_setdescriptoffset(descript, dx*4/lambda, dy*4/lambda);
		us_modifytextdescript(high, descript);

		/* redisplay the text */
		endobjectchange((INTBIG)high->fromgeom->entryaddr.blind,
			high->fromgeom->entrytype == OBJNODEINST ? VNODEINST : VARCINST);
		(void)us_addhighlight(high);

		/* modify all higher-level nodes if port moved */
		if (high->fromvar == NOVARIABLE && high->fromport != NOPORTPROTO)
		{
			for(ni = high->fromport->parent->firstinst; ni != NONODEINST; ni = ni->nextinst)
			{
				if ((ni->userbits&NEXPAND) != 0 && (high->fromport->userbits&PORTDRAWN) == 0)
					continue;
				startobjectchange((INTBIG)ni, VNODEINST);
				endobjectchange((INTBIG)ni, VNODEINST);
			}
		}
		return;
	}

	/* handle absolute motion option "move to X Y" */
	l = (INTSML)strlen(pp = par[0]);
	if (namesamen(pp, "angle", l) == 0 && l >= 1)
	{
		us_abortcommand("Cannot move text along an angle");
		return;
	}
	if (namesamen(pp, "to", l) == 0 && l >= 1)
	{
		ni = NONODEINST;
		if (high->fromvar != NOVARIABLE)
		{
			if (high->fromgeom->entrytype == OBJNODEINST) ni = high->fromgeom->entryaddr.ni;
			descript = high->fromvar->textdescript;
		} else if (high->fromport != NOPORTPROTO)
		{
			ni = high->fromgeom->entryaddr.ni;
			descript = high->fromport->textdescript;
		} else if (high->fromgeom->entrytype == OBJNODEINST)
		{
			ni = high->fromgeom->entryaddr.ni;
			descript = high->fromgeom->entryaddr.ni->textdescript;
		}

		/* get absolute position to place object */
		if (count == 2 && namesamen(par[1], "dialog", (INTSML)strlen(par[1])) == 0)
		{
			/* get coordinates from dialog */
			if (DiaInitDialog(&usr_movetodialog) != 0) return;
			DiaSetText(3, latoa(ni->lowx));
			DiaSetText(5, latoa(ni->lowy));
			for(;;)
			{
				itemHit = DiaNextHit();
				if (itemHit == CANCEL || itemHit == OK) break;
			}
			xcur = atola(DiaGetText(3));
			ycur = atola(DiaGetText(5));
			DiaDoneDialog();
			if (itemHit == CANCEL) return;
		} else
		{
			/* get coordinates from command line */
			if (count < 3)
			{
				us_abortcommand("Usage: move to X Y");
				/* (void)us_addhighlight(high); */
				return;
			}
			xcur = atola(par[1]);
			ycur = atola(par[2]);
		}

		/* move entire node if it is an "invisible pin" */
		if (ni != NONODEINST && ni->proto == gen_invispinprim)
		{
			dx = xcur - ni->lowx;   dy = ycur - ni->lowy;
			startobjectchange((INTBIG)ni, VNODEINST);
			modifynodeinst(ni, dx, dy, dx, dy, 0, 0);
			endobjectchange((INTBIG)ni, VNODEINST);
			(void)us_addhighlight(high);
			return;
		}

		/* undraw the text */
		startobjectchange((INTBIG)high->fromgeom->entryaddr.blind,
			high->fromgeom->entrytype == OBJNODEINST ? VNODEINST : VARCINST);

		if (ni != NONODEINST)
		{
			if (ni->transpose != 0)
				makeangle(ni->rotation, ni->transpose, trans); else
					makeangle((INTSML)((3600-ni->rotation)%3600), 0, trans);
			xform(xcur, ycur, &xcur, &ycur, trans);
		}

		/* set the new descriptor on the text */
		lambda = figurelambda(high->fromgeom);
		descript = us_setdescriptoffset(descript, xcur*4/lambda, ycur*4/lambda);
		us_modifytextdescript(high, descript);

		/* redisplay the text */
		endobjectchange((INTBIG)high->fromgeom->entryaddr.blind,
			high->fromgeom->entrytype == OBJNODEINST ? VNODEINST : VARCINST);
		(void)us_addhighlight(high);

		/* modify all higher-level nodes if port moved */
		if (high->fromvar == NOVARIABLE && high->fromport != NOPORTPROTO)
		{
			for(ni = high->fromport->parent->firstinst; ni != NONODEINST; ni = ni->nextinst)
			{
				if ((ni->userbits&NEXPAND) != 0 &&
					(high->fromport->userbits&PORTDRAWN) == 0) continue;
				startobjectchange((INTBIG)ni, VNODEINST);
				endobjectchange((INTBIG)ni, VNODEINST);
			}
		}
		return;
	}

	/* if direction is specified, get it */
	dx = dy = 0;
	if (namesamen(pp, "up", l) == 0 && l >= 1)
	{
		if (count < 2) amt = el_curtech->deflambda; else amt = atola(par[1]);
		dy += amt;
	} else if (namesamen(pp, "down", l) == 0 && l >= 1)
	{
		if (count < 2) amt = el_curtech->deflambda; else amt = atola(par[1]);
		dy -= amt;
	} else if (namesamen(pp, "left", l) == 0 && l >= 1)
	{
		if (count < 2) amt = el_curtech->deflambda; else amt = atola(par[1]);
		dx -= amt;
	} else if (namesamen(pp, "right", l) == 0 && l >= 1)
	{
		if (count < 2) amt = el_curtech->deflambda; else amt = atola(par[1]);
		dx += amt;
	} else
	{
		us_abortcommand("Invalid MOVE option: %s", par[0]);
		(void)us_addhighlight(high);
		return;
	}

	if (dx == 0 && dy == 0) return;

	/* now move the object */
	ni = NONODEINST;
	if (high->fromvar != NOVARIABLE)
	{
		if (high->fromgeom->entrytype == OBJNODEINST) ni = high->fromgeom->entryaddr.ni;
		descript = high->fromvar->textdescript;
	} else if (high->fromport != NOPORTPROTO)
	{
		ni = high->fromgeom->entryaddr.ni;
		descript = high->fromport->textdescript;
	} else if (high->fromgeom->entrytype == OBJNODEINST)
	{
		ni = high->fromgeom->entryaddr.ni;
		descript = high->fromgeom->entryaddr.ni->textdescript;
	}

	/* move entire node if it is an "invisible pin" */
	if (ni != NONODEINST && ni->proto == gen_invispinprim)
	{
		startobjectchange((INTBIG)ni, VNODEINST);
		modifynodeinst(ni, dx, dy, dx, dy, 0, 0);
		endobjectchange((INTBIG)ni, VNODEINST);
		(void)us_addhighlight(high);
		return;
	}

	/* make offsets be the proper scale */
	xcur = (descript&VTXOFF) >> VTXOFFSH;
	if ((descript&VTXOFFNEG) != 0) xcur = -xcur;
	ycur = (descript&VTYOFF) >> VTYOFFSH;
	if ((descript&VTYOFFNEG) != 0) ycur = -ycur;

	if (ni != NONODEINST)
	{
		if (ni->transpose != 0)
			makeangle(ni->rotation, ni->transpose, trans); else
				makeangle((INTSML)((3600-ni->rotation)%3600), 0, trans);
		xform(dx, dy, &dx, &dy, trans);
	}

	/* recompute the descriptor */
	lambda = figurelambda(high->fromgeom);
	descript = us_setdescriptoffset(descript, xcur+dx*4/lambda, ycur+dy*4/lambda);

	/* undraw the text */
	startobjectchange((INTBIG)high->fromgeom->entryaddr.blind,
		high->fromgeom->entrytype == OBJNODEINST ? VNODEINST : VARCINST);

	/* set the new descriptor on the text */
	us_modifytextdescript(high, descript);

	/* redisplay the text */
	endobjectchange((INTBIG)high->fromgeom->entryaddr.blind,
		high->fromgeom->entrytype == OBJNODEINST ? VNODEINST : VARCINST);
	(void)us_addhighlight(high);

	/* modify all higher-level nodes if port moved */
	if (high->fromvar == NOVARIABLE && high->fromport != NOPORTPROTO)
	{
		for(ni = high->fromport->parent->firstinst; ni != NONODEINST; ni = ni->nextinst)
		{
			if ((ni->userbits&NEXPAND) != 0 && (high->fromport->userbits&PORTDRAWN) == 0) continue;
			startobjectchange((INTBIG)ni, VNODEINST);
			endobjectchange((INTBIG)ni, VNODEINST);
		}
	}
}

/*
 * routine to move the arcs in the GEOM module list "list" (terminated by
 * NOGEOM) and the "total" nodes in the list "nodelist" by (dx, dy).
 */
void us_manymove(GEOM **list, NODEINST **nodelist, INTSML total, INTBIG dx, INTBIG dy)
{
	REGISTER NODEINST *ni;
	NODEINST *mynodelist[1];
	GEOM *nulllist[1];
	REGISTER ARCINST *ai;
	REGISTER INTSML i, j, again, e1, e2;
	extern CONSTRAINT *cla_constraint;

#if 0
	REGISTER INTSML fun;
	char *extra;
	REGISTER NODEINST *oni;
	REGISTER PORTARCINST *pi, *opi;

	/* for all nonpins, set temporary rigidity on all arcs to stranded pins */
	for(i=0; list[i] != NOGEOM; i++)
	{
		if (list[i]->entrytype != OBJNODEINST) continue;
		ni = list[i]->entryaddr.ni;
		fun = nodefunction(ni, &extra);
		if (fun == NPPIN) continue;

		/* nonpin node found: check all arcs */
		for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
		{
			ai = pi->conarcinst;
			if (ai->end[0].nodeinst == ni) oni = ai->end[1].nodeinst; else
				oni = ai->end[0].nodeinst;
			if (oni == ni) continue;
			fun = nodefunction(oni, &extra);
			if (fun != NPPIN) continue;

			/* found a pin at the end of this arc: make sure it has only one arc */
			j = 0;
			for(opi = oni->firstportarcinst; opi != NOPORTARCINST; opi = opi->nextportarcinst) j++;
			if (j != 1) continue;

			/* make this arc temporarily rigid */
			(void)(*el_curconstraint->setobject)((INTBIG)ai, VARCINST, 6, 0);
		}
	}
#endif

	/* special case if moving only one node */
	if (total == 1 && list[1] == NOGEOM)
	{
		ni = nodelist[0];
		startobjectchange((INTBIG)ni, VNODEINST);
		modifynodeinst(ni, dx, dy, dx, dy, 0, 0);
		endobjectchange((INTBIG)ni, VNODEINST);
		return;
	}

	/* special case if moving only one arc */
	if (list[1] == NOGEOM && list[0]->entrytype == OBJARCINST)
	{
		ai = list[0]->entryaddr.ai;

		/* see if the arc moves in its ports */
		if ((ai->userbits&(FIXED|CANTSLIDE)) != 0) e1 = e2 = 0; else
		{
			e1 = db_stillinport(ai, 0, ai->end[0].xpos+dx, ai->end[0].ypos+dy);
			e2 = db_stillinport(ai, 1, ai->end[1].xpos+dx, ai->end[1].ypos+dy);
		}

		/* if arc moves in both ports, simply adjust the arc */
		if (e1 != 0 && e2 != 0)
		{
			startobjectchange((INTBIG)ai, VARCINST);
			(void)modifyarcinst(ai, 0, dx, dy, dx, dy);
			endobjectchange((INTBIG)ai, VARCINST);
			return;
		}

		/* if arc moves in one port, only move the other node */
		if (e1 != 0 || e2 != 0)
		{
			if (e1 == 0) mynodelist[0] = ai->end[0].nodeinst; else
				mynodelist[0] = ai->end[1].nodeinst;
			nodelist = mynodelist;
			total = 1;
			nulllist[0] = NOGEOM;
			list = nulllist;
		}
	}

	if (el_curconstraint != cla_constraint)
	{
		us_abortcommand("Must use the layout constraint system to move");
		return;
	}

	/* remember initial coordinates of all nodes */
	for(i=0; i<total; i++)
	{
		ni = nodelist[i];
		ni->temp1 = ni->lowx;
		ni->temp2 = ni->lowy;
	}

	/* loop through every node to be moved */
	again = 1;
	while (again)
	{
		again = 0;
		for(i=0; i<total; i++)
		{
			if (stopping("Move") != 0) break;

			/* ignore this nodeinst if it has been moved already */
			ni = nodelist[i];
			if (ni->lowx != ni->temp1 || ni->lowy != ni->temp2) continue;

			/* set every selected arc to be temporarily rigid */
			for(j=0; list[j] != NOGEOM; j++)
				if (list[j]->entrytype == OBJARCINST)
					(void)(*el_curconstraint->setobject)((INTBIG)list[j]->entryaddr.ai, VARCINST, 6, 0);

			/* move this nodeinst in proper direction */
			startobjectchange((INTBIG)ni, VNODEINST);
			modifynodeinst(ni, dx-(ni->lowx-ni->temp1), dy-(ni->lowy-ni->temp2),
				dx-(ni->lowx-ni->temp1), dy-(ni->lowy-ni->temp2), 0, 0);
			endobjectchange((INTBIG)ni, VNODEINST);

			/* set loop iteration flag */
			again++;
			break;
		}
	}
}

/*
 * routine to create a port in facet "np" called "portname".  The port resides on
 * node "ni", port "pp" in the facet.  The userbits field will have "bits" set where
 * "mask" points.  The text descriptor is "textdescript".
 * Also, check across icon/contents boundary to create a parallel port if possible
 */
void us_makenewportproto(NODEPROTO *np, NODEINST *ni, PORTPROTO *pp,
	char *portname, INTBIG mask, INTBIG bits, INTBIG textdescript)
{
	REGISTER NODEPROTO *onp, *pinproto;
	REGISTER NODEINST *pinni, *boxni;
	INTBIG x, y, portcount, portlen, *newportlocation;
	static INTBIG portlimit = 0, *portlocation;
	REGISTER INTBIG character, boxlx, boxhx, boxly, boxhy, rangelx, rangehx,
		rangely, rangehy;
	REGISTER PORTPROTO *ppt, *opp, *bpp, *outputport, *topport, *inputport, *bidirport;

	startobjectchange((INTBIG)ni, VNODEINST);
	ppt = newportproto(np, ni, pp, portname);
	if (ppt == NOPORTPROTO)
	{
		us_abortcommand("Error creating the port");
		(void)us_pophighlight(0);
		return;
	}
	if ((mask&STATEBITS) != 0) ppt->userbits = (ppt->userbits & ~STATEBITS) | (bits & STATEBITS);
	if ((mask&PORTDRAWN) != 0) ppt->userbits = (ppt->userbits & ~PORTDRAWN) | (bits & PORTDRAWN);
	if ((mask&BODYONLY) != 0)  ppt->userbits = (ppt->userbits & ~BODYONLY) | (bits & BODYONLY);
	if (ni->proto->index == 0) ppt->textdescript = textdescript;
	endobjectchange((INTBIG)ni, VNODEINST);

	/* ignore new port if not intended for icon */
	if ((pp->userbits&BODYONLY) != 0) return;

	/* see if there is an associated icon facet */
	onp = NONODEPROTO;
	if (np->cellview != el_iconview) onp = iconview(np);
	if (onp == NONODEPROTO) return;

	/* find the box in the icon facet */
	for(boxni = onp->firstnodeinst; boxni != NONODEINST; boxni = boxni->nextnodeinst)
		if (boxni->proto == sch_bboxprim) break;
	if (boxni == NONODEINST)
	{
		boxlx = boxhx = (onp->lowx + onp->highx) / 2;
		boxly = boxhy = (onp->lowy + onp->highy) / 2;
		rangelx = onp->lowx;
		rangehx = onp->highx;
		rangely = onp->lowy;
		rangehy = onp->highy;
	} else
	{
		rangelx = boxlx = boxni->lowx;
		rangehx = boxhx = boxni->highx;
		rangely = boxly = boxni->lowy;
		rangehy = boxhy = boxni->highy;
	}

	/* icon facet found, quit if this port is already there */
	for(opp = onp->firstportproto; opp != NOPORTPROTO; opp = opp->nextportproto)
		if (namesame(opp->protoname, ppt->protoname) == 0) return;

	/* add a port to the icon */
	character = ppt->userbits & STATEBITS;

	/* special detection for power and ground ports */
	if (portispower(pp) != 0 || portisground(pp) != 0) character = GNDPORT;

	/* count the number of ports */
	portlen = 0;
	for(opp = onp->firstportproto; opp != NOPORTPROTO; opp = opp->nextportproto) portlen++;

	if (portlen > portlimit)
	{
		newportlocation = (INTBIG *)emalloc(portlen * SIZEOFINTBIG, us_aid->cluster);
		if (newportlocation == 0) return;
		if (portlimit > 0) efree((char *)portlocation);
		portlocation = newportlocation;
		portlimit = portlen;
	}
	portcount = 0;
	switch (character)
	{
		case OUTPORT:
			for(opp = onp->firstportproto; opp != NOPORTPROTO; opp = opp->nextportproto)
			{
				portposition(opp->subnodeinst, opp->subportproto, &x, &y);
				if (x > boxhx) portlocation[portcount++] = y;
			}
			y = us_findnewplace(portlocation, portcount, rangely, rangehy);
			x = onp->highx;
			if (boxni != NONODEINST && onp->highx == boxni->highx) x += el_curtech->deflambda * 4;
			break;
		case BIDIRPORT:
			for(opp = onp->firstportproto; opp != NOPORTPROTO; opp = opp->nextportproto)
			{
				portposition(opp->subnodeinst, opp->subportproto, &x, &y);
				if (y < boxly) portlocation[portcount++] = x;
			}
			x = us_findnewplace(portlocation, portcount, rangelx, rangehx);
			y = onp->lowy;
			if (boxni != NONODEINST && onp->lowy == boxni->lowy) y -= el_curtech->deflambda * 4;
			break;
		case PWRPORT:
		case GNDPORT:
			for(opp = onp->firstportproto; opp != NOPORTPROTO; opp = opp->nextportproto)
			{
				portposition(opp->subnodeinst, opp->subportproto, &x, &y);
				if (y > boxhy) portlocation[portcount++] = x;
			}
			x = us_findnewplace(portlocation, portcount, rangelx, rangehx);
			y = onp->highy;
			if (boxni != NONODEINST && onp->highy == boxni->highy) y += el_curtech->deflambda * 4;
			break;
		default:		/* INPORT, unlabeled, and all CLOCK ports */
			for(opp = onp->firstportproto; opp != NOPORTPROTO; opp = opp->nextportproto)
			{
				portposition(opp->subnodeinst, opp->subportproto, &x, &y);
				if (x < boxlx) portlocation[portcount++] = y;
			}
			y = us_findnewplace(portlocation, portcount, rangely, rangehy);
			x = onp->lowx;
			if (boxni != NONODEINST && onp->lowx == boxni->lowx) x -= el_curtech->deflambda * 4;
			break;
	}

	/* determine type of pin */
	pinproto = sch_wirepinprim;
	if (pp->subnodeinst != NONODEINST)
	{
		bpp = pp;
		while (bpp->subnodeinst->proto->index == 0) bpp = bpp->subportproto;
		if (bpp->subnodeinst->proto == sch_buspinprim) pinproto = sch_buspinprim;
	}

	/* create the pin */
	pinni = newnodeinst(pinproto, x-(pinproto->highx-pinproto->lowx)/2,
		x+(pinproto->highx-pinproto->lowx)/2, y-(pinproto->highy-pinproto->lowy)/2,
			y+(pinproto->highy-pinproto->lowy)/2, 0, 0, onp);
	if (pinni == NONODEINST) return;

	/* wire the pin if there is a center box */
	if (boxni != NONODEINST)
	{
		outputport = sch_bboxprim->firstportproto;
		topport = outputport->nextportproto;
		inputport = topport->nextportproto;
		bidirport = inputport->nextportproto;
		switch (character)
		{
			case OUTPORT:
				us_addleadtoicon(onp, sch_wirearc, pinni, boxni, outputport, x, y,
					boxhx, y);
				break;
			case BIDIRPORT:
				us_addleadtoicon(onp, sch_wirearc, pinni, boxni, bidirport, x, y,
					x, boxly);
				break;
			case PWRPORT:
			case GNDPORT:
				us_addleadtoicon(onp, sch_wirearc, pinni, boxni, topport, x, y,
					x, boxhy);
				break;
			default:		/* INPORT, unlabeled, and all CLOCK ports */
				us_addleadtoicon(onp, sch_wirearc, pinni, boxni, inputport, x, y,
					boxlx, y);
				break;
		}
	}

	/* export the port that should be on this pin */
	us_makenewportproto(onp, pinni, pinproto->firstportproto, portname, mask, bits|PORTDRAWN, 0);
}

/*
 * routine to find the largest gap in an integer array that is within the bounds
 * "low" to "high".  The array is sorted by this routine.
 */
INTBIG us_findnewplace(INTBIG *arr, INTBIG count, INTBIG low, INTBIG high)
{
	REGISTER INTBIG swap, i, gapwid, gappos;
	REGISTER INTSML sorted;

	/* easy if nothing in the array */
	if (count <= 0) return((low + high) / 2);

	/* first sort the array */
	sorted = 0;
	while (sorted == 0)
	{
		sorted = 1;
		for(i=1; i<count; i++)
		{
			if (arr[i-1] < arr[i]) continue;
			swap = arr[i-1];
			arr[i-1] = arr[i];
			arr[i] = swap;
			sorted = 0;
		}
	}

	/* now find the widest gap */
	gapwid = 0;
	gappos = (low + high) / 2;
	for(i=1; i<count; i++)
	{
		if (arr[i] - arr[i-1] > gapwid)
		{
			gapwid = arr[i] - arr[i-1];
			gappos = (arr[i-1] + arr[i]) / 2;
		}
	}
	if (arr[0] - low > gapwid)
	{
		gapwid = arr[0] - low;
		gappos = (low + arr[0]) / 2;
	}
	if (high - arr[count-1] > gapwid)
	{
		gapwid = high - arr[count-1];
		gappos = (arr[count-1] + high) / 2;
	}
	return(gappos);
}

/*
 * routine to generate an icon in library "lib" with name "iconname" from the
 * port list in "fpp".  The icon facet is called "pt" and it is constructed
 * from technology "tech".  The icon facet is returned (NONODEPROTO on error).
 */
NODEPROTO *us_makeiconfacet(PORTPROTO *fpp, char *iconname, char *pt,
	TECHNOLOGY *tech, LIBRARY *lib)
{
	REGISTER NODEPROTO *np, *pintype;
	REGISTER NODEINST *bbni, *pinni;
	REGISTER PORTPROTO *pp, *port, *inputport, *outputport, *bidirport,
		*topport, *whichport, *bpp;
	REGISTER ARCPROTO *wiretype;
	REGISTER INTSML inputside, outputside, bidirside, topside;
	REGISTER INTBIG character, index, xsize, ysize, xpos, ypos, xbbpos, ybbpos, spacing;
	REGISTER VARIABLE *var;

	/* get the necessary symbols */
	outputport = sch_bboxprim->firstportproto;
	topport = outputport->nextportproto;
	inputport = topport->nextportproto;
	bidirport = inputport->nextportproto;

	/* create the new icon facet */
	np = newnodeproto(pt, lib);
	if (np == NONODEPROTO)
	{
		us_abortcommand("Cannot create icon %s", pt);
		return(NONODEPROTO);
	}
	np->userbits |= WANTNEXPAND;

	/* determine number of inputs and outputs */
	inputside = outputside = bidirside = topside = 0;
	for(pp = fpp; pp != NOPORTPROTO; pp = pp->nextportproto)
	{
		if ((pp->userbits&BODYONLY) != 0) continue;
		character = pp->userbits & STATEBITS;

		/* special detection for power and ground ports */
		if (portispower(pp) != 0 || portisground(pp) != 0) character = GNDPORT;

		/* make a count of the types of ports and save it on the ports */
		switch (character)
		{
			case OUTPORT:
				pp->temp1 = outputside++;
				break;
			case BIDIRPORT:
				pp->temp1 = bidirside++;
				break;
			case PWRPORT:
			case GNDPORT:
				pp->temp1 = topside++;
				break;
			default:		/* INPORT, unlabeled, and all CLOCK ports */
				pp->temp1 = inputside++;
				break;
		}
	}

	/* create the Black Box with the correct size */
	ysize = maxi(maxi(inputside, outputside), 5) * 2 * tech->deflambda;
	xsize = maxi(maxi(topside, bidirside), 3) * 2 * tech->deflambda;

	/* create the Black Box instance */
	bbni = newnodeinst(sch_bboxprim, 0, xsize, 0, ysize, 0, 0, np);
	if (bbni == NONODEINST) return(NONODEPROTO);

	/* create the Facet Center instance */
	pinni = newnodeinst(gen_facetcenterprim, 0, 0, 0, 0, 0, 0, np);
	if (pinni == NONODEINST) return(NONODEPROTO);

	/* put the original cell name on the Black Box */
	var = setvalkey((INTBIG)bbni, VNODEINST, sch_functionkey, (INTBIG)iconname, VSTRING|VDISPLAY);
	if (var != NOVARIABLE) var->textdescript = defaulttextdescript();

	/* place pins around the Black Box */
	for(pp = fpp; pp != NOPORTPROTO; pp = pp->nextportproto)
	{
		if ((pp->userbits&BODYONLY) != 0) continue;
		character = pp->userbits & STATEBITS;

		/* special detection for power and ground ports */
		if (portispower(pp) != 0 || portisground(pp) != 0) character = GNDPORT;

		/* make a count of the types of ports and save it on the ports */
		switch (character)
		{
			case OUTPORT:
				xpos = xsize + 2 * tech->deflambda;
				xbbpos = xsize;
				index = pp->temp1;
				spacing = 2 * tech->deflambda;
				if (outputside*2 < inputside) spacing = 4 * tech->deflambda;
				ybbpos = ypos = ysize - ((ysize - (outputside-1)*spacing) / 2 + index * spacing);
				whichport = outputport;
				break;
			case BIDIRPORT:
				index = pp->temp1;
				spacing = 2 * tech->deflambda;
				if (bidirside*2 < topside) spacing = 4 * tech->deflambda;
				xbbpos = xpos = xsize - ((xsize - (bidirside-1)*spacing) / 2 + index * spacing);
				ypos = -2 * tech->deflambda;
				ybbpos = 0;
				whichport = bidirport;
				break;
			case PWRPORT:
			case GNDPORT:
				index = pp->temp1;
				spacing = 2 * tech->deflambda;
				if (topside*2 < bidirside) spacing = 4 * tech->deflambda;
				xbbpos = xpos = xsize - ((xsize - (topside-1)*spacing) / 2 + index * spacing);
				ypos = ysize + 2 * tech->deflambda;
				ybbpos = ysize;
				whichport = topport;
				break;
			default:		/* INPORT, unlabeled, and all CLOCK ports */
				xpos = -2 * tech->deflambda;
				xbbpos = 0;
				index = pp->temp1;
				spacing = 2 * tech->deflambda;
				if (inputside*2 < outputside) spacing = 4 * tech->deflambda;
				ybbpos = ypos = ysize - ((ysize - (inputside-1)*spacing) / 2 + index * spacing);
				whichport = inputport;
				break;
		}

		/* determine type of pin */
		pintype = sch_wirepinprim;
		wiretype = sch_wirearc;
		if (pp->subnodeinst != NONODEINST)
		{
			bpp = pp;
			while (bpp->subnodeinst->proto->index == 0) bpp = bpp->subportproto;
			if (bpp->subnodeinst->proto == sch_buspinprim)
			{
				pintype = sch_buspinprim;
				wiretype = sch_busarc;
			}
		}

		/* create the pin */
		pinni = newnodeinst(pintype, xpos-(pintype->highx-pintype->lowx)/2,
			xpos+(pintype->highx-pintype->lowx)/2, ypos-(pintype->highy-pintype->lowy)/2,
				ypos+(pintype->highy-pintype->lowy)/2, 0, 0, np);
		if (pinni == NONODEINST) return(NONODEPROTO);

		/* export the port that should be on this pin */
		port = newportproto(np, pinni, pintype->firstportproto, pp->protoname);
		if (port != NOPORTPROTO)
			port->userbits = (port->userbits & ~STATEBITS) | (pp->userbits & STATEBITS) | PORTDRAWN;

		/* wire this pin to the black box */
		us_addleadtoicon(np, wiretype, pinni, bbni, whichport, xpos, ypos,
			xbbpos, ybbpos);
	}
	return(np);
}

void us_addleadtoicon(NODEPROTO *facet, ARCPROTO *wire, NODEINST *pin, NODEINST *box,
	PORTPROTO *boxport, INTBIG pinx, INTBIG piny, INTBIG boxx, INTBIG boxy)
{
	NODEINST *ni;
	INTBIG lx, hx, ly, hy, cx, cy, data[4];

	if (wire != sch_wirearc)
	{
		/* not a simple wire: connect this pin to the black box */
		(void)newarcinst(wire, wire->nominalwidth, us_makearcuserbits(wire),
			pin, pin->proto->firstportproto, pinx, piny, box, boxport, boxx, boxy, facet);
	} else
	{
		/* simple wire: draw a line from this pin to the black box */
		lx = mini(pinx, boxx);   hx = maxi(pinx, boxx);
		ly = mini(piny, boxy);   hy = maxi(piny, boxy);
		cx = (pinx + boxx) / 2;  cy = (piny + boxy) / 2;
		ni = newnodeinst(art_openedpolygonprim, lx, hx, ly, hy, 0, 0, facet);
		if (ni == NONODEINST) return;
		data[0] = pinx - cx;   data[1] = piny - cy;
		data[2] = boxx - cx;   data[3] = boxy - cy;
		(void)setvalkey((INTBIG)ni, VNODEINST, el_trace, (INTBIG)data,
			VINTEGER|VISARRAY|(4<<VLENGTHSH));
	}
}

/*
 * routine to determine the "userbits" to use for an arc of type "ap".
 */
INTBIG us_makearcuserbits(ARCPROTO *ap)
{
	REGISTER INTBIG bits, protobits;
	REGISTER VARIABLE *var;

	var = getvalkey((INTBIG)us_aid, VAID, VINTEGER, us_arcstyle);
	if (var != NOVARIABLE) protobits = var->addr; else
		protobits = ap->userbits;
	bits = 0;
	if ((protobits&WANTFIXANG) != 0)      bits |= FIXANG;
	if ((protobits&WANTFIX) != 0)         bits |= FIXED;
	if ((protobits&WANTCANTSLIDE) != 0)   bits |= CANTSLIDE;
	if ((protobits&WANTNOEXTEND) != 0)    bits |= NOEXTEND;
	if ((protobits&WANTNEGATED) != 0)     bits |= ISNEGATED;
	if ((protobits&WANTDIRECTIONAL) != 0) bits |= ISDIRECTIONAL;
	return(bits);
}
