/*
 * Electric(tm) VLSI Design System
 *
 * File: usrpallet.c
 * User interface aid: color palette mixing code
 * Written by: Steven M. Rubin, Electric Editor Incorporated
 *
 * Copyright (c) 1998 Electric Editor Incorporated.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Electric Editor Incorporated
 * 23470 Sunset Drive, Suite 108
 * Los Gatos, California 95033
 * support@electriceditor.com
 */

#include "global.h"
#include "egraphics.h"
#include "usr.h"
#include <math.h>

#define	CENTERX	    130					/* center of hue/saturation wheel */
#define	CENTERY	    130					/* center of hue/saturation wheel */
#define	DEGTORAD    (EPI/180.0)

/* the editable color map entries */
#define	MAPEDIT       0					/* base of editable colors */
#define	MAPENDEDIT   15					/* end of editable colors */
#define MAXCOL	(MAPENDEDIT-MAPEDIT+1)	/* number of colors to mix */

/* the hue/intensity color map entries (takes 216 entries) */
#define	MAPHUEINTEN  16					/* base of editable colors */

/* the black&white saturation color map entries */
#define	MAPSATUR    232					/* base of saturation range */
#define	MAPENDSATUR 254					/* end of saturation range */
#define	MAPBLACK    MAPSATUR			/* the off color */
#define	MAPWHITE    MAPENDSATUR			/* the on color */

/* globals for this module */
static WINDOW us_palletwindow;
static GRAPHICS us_palletgraph = {LAYERA, 0, {SOLIDC, SOLIDC, SOLIDC, SOLIDC},
	{0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF}, NOVARIABLE, 0};
static POLYGON *us_palletpoly = NOPOLYGON;
static INTSML us_palcur, us_paltrueindex[MAXCOL];
static float us_palcurtheta, us_palcurinten, us_palcurr;
static char *us_pallayername[5], *us_pallayerabbrev[5];
static INTBIG us_redmap[256], us_greenmap[256], us_bluemap[256];
static INTBIG us_oredmap[256], us_ogreenmap[256], us_obluemap[256];

/* prototypes for local routines */
void us_palmakeinitcolormap(void);
void us_palredisphandler(WINDOW*);
INTSML us_palcharhandler(WINDOW*, INTSML);
void us_paltermhandler(WINDOW*);
void us_palbuttonhandler(WINDOW*, INTSML, INTSML, INTSML);
void us_filltrueentry(INTSML, INTSML, INTSML, INTSML);
void us_paldrawsplotch(INTSML, WINDOW*);
void us_palbuildindex(INTSML, INTSML, INTSML, INTSML, INTSML);
void us_palbuildprimaries(INTSML, INTSML, INTSML, INTSML, INTSML);
void us_palbuildspecials(void);
void us_paldrawarrow(INTSML, INTSML, WINDOW*);
void us_paldrawsidearrow(INTSML, INTSML, WINDOW*);
void us_paldrawhue(INTSML, INTSML, INTSML, WINDOW*);
void us_paldrawentries(WINDOW*);
char *us_palentname(INTSML);

/*
 * routine to display a hue/saturation/intensity color gamut in window "w" and
 * interactively mix colors in a palette.  The color map is in the variables
 * "USER_colormap_red/green/blue" on the "user" aid and is immediately copied
 * to the local globals "us_redmap", "us_greenmap", "us_bluemap".  The
 * overlappable layer names are in "overlayernames" (five entries), and the
 * one-letter abbreviations are in "overlayerabbrev".
 */
void us_palette(WINDOW *w, char *overlayernames[], char *overlayerabbrev[])
{
	REGISTER INTSML i;
	REGISTER INTBIG dx, dy;
	REGISTER VARIABLE *varred, *vargreen, *varblue;

	/* cannot handle screens with less than 256 colors */
	if (el_maplength < 256)
	{
		us_abortcommand("Sorry, display must support 256 colors to do mixing (this one uses %d)",
			el_maplength);
		return;
	}

	/* get polygon */
	if (us_palletpoly == NOPOLYGON)
		us_palletpoly = allocpolygon(6, us_aid->cluster);
	us_palletpoly->desc = &us_palletgraph;

	/* get the current colors */
	varred = getvalkey((INTBIG)us_aid, VAID, VINTEGER|VISARRAY, us_colormap_red);
	vargreen = getvalkey((INTBIG)us_aid, VAID, VINTEGER|VISARRAY, us_colormap_green);
	varblue = getvalkey((INTBIG)us_aid, VAID, VINTEGER|VISARRAY, us_colormap_blue);
	if (varred == NOVARIABLE || vargreen == NOVARIABLE || varblue == NOVARIABLE) return;
	for(i=0; i<256; i++)
	{
		us_oredmap[i] = us_redmap[i] = ((INTBIG *)varred->addr)[i];
		us_ogreenmap[i] = us_greenmap[i] = ((INTBIG *)vargreen->addr)[i];
		us_obluemap[i] = us_bluemap[i] = ((INTBIG *)varblue->addr)[i];
	}

	/* copy abbreviations */
	for(i=0; i<5; i++)
	{
		(void)allocstring(&us_pallayerabbrev[i], overlayerabbrev[i], el_tempcluster);
		(void)allocstring(&us_pallayername[i], overlayernames[i], el_tempcluster);
	}

	/* load initial color map */
	us_palmakeinitcolormap();

	/* save window */
	copywindow(&us_palletwindow, w);
	us_palletwindow.state &= ~WINDOWSIMULATING;

	/* initialize window */
	startobjectchange((INTBIG)w, VWINDOW);
	(void)setval((INTBIG)w, VWINDOW, "screenlx", 0, VINTEGER);
	(void)setval((INTBIG)w, VWINDOW, "screenly", 0, VINTEGER);
	dx = w->usehx - w->uselx;
	dy = w->usehy - w->usely;
	if (dx > dy)
	{
		(void)setval((INTBIG)w, VWINDOW, "screenhx", 511*dx/dy, VINTEGER);
		(void)setval((INTBIG)w, VWINDOW, "screenhy", 511, VINTEGER);
	} else
	{
		(void)setval((INTBIG)w, VWINDOW, "screenhx", 511, VINTEGER);
		(void)setval((INTBIG)w, VWINDOW, "screenhy", 511*dy/dx, VINTEGER);
	}
	computewindowscale(w);
	(void)setval((INTBIG)w, VWINDOW, "buttonhandler", (INTBIG)us_palbuttonhandler, VADDRESS);
	(void)setval((INTBIG)w, VWINDOW, "charhandler", (INTBIG)us_palcharhandler, VADDRESS);
	(void)setval((INTBIG)w, VWINDOW, "termhandler", (INTBIG)us_paltermhandler, VADDRESS);
	(void)setval((INTBIG)w, VWINDOW, "redisphandler", (INTBIG)us_palredisphandler, VADDRESS);
	(void)setval((INTBIG)w, VWINDOW, "state", (w->state & ~(WINDOWTYPE|WINDOWSIMULATING)) | PALETTEWINDOW, VINTEGER);
	(void)setval((INTBIG)w, VWINDOW, "curnodeproto", (INTBIG)NONODEPROTO, VNODEPROTO);
	endobjectchange((INTBIG)w, VWINDOW);
}

/*
 * routine to build and load the proper color map
 */
void us_palmakeinitcolormap(void)
{
	INTSML i, theta, inc, rr;
	INTBIG red[256], green[256], blue[256];

	for(i=0; i<256; i++) red[i] = green[i] = blue[i] = 255;
	for(i=MAPSATUR; i<=MAPENDSATUR; i++)
		red[i] = green[i] = blue[i] = (i-MAPSATUR) * 255 / (MAPENDSATUR-MAPSATUR);
	for(i=MAPEDIT; i<=MAPENDEDIT; i++) red[i] = green[i] = blue[i] = 0;
	us_palbuildprimaries(LAYERT1, LAYERT2, LAYERT3, LAYERT4, LAYERT5);
	for(i=0; i<MAXCOL; i++)
	{
		red[i+MAPEDIT] = us_redmap[us_paltrueindex[i]];
		green[i+MAPEDIT] = us_greenmap[us_paltrueindex[i]];
		blue[i+MAPEDIT] = us_bluemap[us_paltrueindex[i]];
	}
	i = MAPHUEINTEN;
	for(theta=0; theta<360; theta+=60)
	{
		for(inc=0; inc<60; inc+=20) for(rr=0; rr<60; rr+=15)
		{
			us_hsvtorgb(((float)(theta+inc+10)) / 360.0,
				((float)rr+7.5) / 120.0, 1.0, &red[i], &green[i], &blue[i]);
			i++;
		}
		for(inc=0; inc<60; inc+=10) for(rr=60; rr<120; rr+=15)
		{
			us_hsvtorgb(((float)(theta+inc+5)) / 360.0,
				((float)rr+7.5) / 120.0, 1.0, &red[i], &green[i], &blue[i]);
			i++;
		}
	}
	startobjectchange((INTBIG)us_aid, VAID);
	(void)setvalkey((INTBIG)us_aid, VAID, us_colormap_red, (INTBIG)red,
		VINTEGER|VISARRAY|(256<<VLENGTHSH)|VDONTSAVE);
	(void)setvalkey((INTBIG)us_aid, VAID, us_colormap_green, (INTBIG)green,
		VINTEGER|VISARRAY|(256<<VLENGTHSH)|VDONTSAVE);
	(void)setvalkey((INTBIG)us_aid, VAID, us_colormap_blue, (INTBIG)blue,
		VINTEGER|VISARRAY|(256<<VLENGTHSH)|VDONTSAVE);
	endobjectchange((INTBIG)us_aid, VAID);
}

/*
 * redisplay routine for the palette
 */
void us_palredisphandler(WINDOW *w)
{
	REGISTER INTSML theta, inc, i;
	REGISTER INTBIG rr, dx, dy;
	float intermed, intertwo;

	/* keep the palette square */
	dx = w->usehx - w->uselx;
	dy = w->usehy - w->usely;
	w->screenlx = w->screenly = 0;
	if (dx > dy)
	{
		w->screenhx = 511*dx/dy;
		w->screenhy = 511;
	} else
	{
		w->screenhx = 511;
		w->screenhy = 511*dy/dx;
	}
	computewindowscale(w);

	/* erase the window */
	us_palletgraph.col = MAPBLACK;
	maketruerectpoly(0, 511, 0, 511, us_palletpoly);
	us_palletpoly->style = FILLEDRECT;
	(void)us_showpoly(us_palletpoly, w);

	/* draw the "done" button */
	us_palletgraph.col = MAPWHITE;
	makerectpoly(450, 511, 90, 110, us_palletpoly);
	us_palletpoly->style = TEXTBOX;
	us_palletpoly->font = TXTMEDIUM;
	us_palletpoly->string = "Done";
	(void)us_showpoly(us_palletpoly, w);
	us_palletpoly->style = CLOSED;
	(void)us_showpoly(us_palletpoly, w);

	/* draw the "cancel" button */
	makerectpoly(450, 511, 190, 210, us_palletpoly);
	us_palletpoly->style = TEXTBOX;
	us_palletpoly->string = "Cancel";
	(void)us_showpoly(us_palletpoly, w);
	us_palletpoly->style = CLOSED;
	(void)us_showpoly(us_palletpoly, w);

	/* draw hue/saturation wheel */
	us_palletgraph.col = MAPWHITE;
	makerectpoly(CENTERX-100, CENTERX+100, 250, 270, us_palletpoly);
	us_palletpoly->style = TEXTBOX;
	us_palletpoly->font = TXTMEDIUM;
	us_palletpoly->string = "Mix Hue/Saturation:";
	(void)us_showpoly(us_palletpoly, w);
	i = MAPHUEINTEN;
	us_palletpoly->count = 4;
	us_palletpoly->style = FILLED;
	for(theta=0; theta<360; theta+=60)
	{
		for(inc=0; inc<60; inc+=20) for(rr=0; rr<60; rr+=15)
		{
			intermed = (float) theta + inc;
			intertwo = (float) rr + 16;
			us_palletpoly->xv[0] = (INTBIG)(((float)rr) * cos(intermed*DEGTORAD)) + CENTERX;
			us_palletpoly->yv[0] = (INTBIG)(((float)rr) * sin(intermed*DEGTORAD)) + CENTERY;
			us_palletpoly->xv[1] = (INTBIG)(intertwo * cos(intermed*DEGTORAD)) + CENTERX;
			us_palletpoly->yv[1] = (INTBIG)(intertwo * sin(intermed*DEGTORAD)) + CENTERY;
			intermed += 21.0;
			us_palletpoly->xv[2] = (INTBIG)(intertwo * cos(intermed*DEGTORAD)) + CENTERX;
			us_palletpoly->yv[2] = (INTBIG)(intertwo * sin(intermed*DEGTORAD)) + CENTERY;
			us_palletpoly->xv[3] = (INTBIG)(((float)rr) * cos(intermed*DEGTORAD)) + CENTERX;
			us_palletpoly->yv[3] = (INTBIG)(((float)rr) * sin(intermed*DEGTORAD)) + CENTERY;
			us_palletgraph.col = i++;
			(void)us_showpoly(us_palletpoly, w);
		}
		for(inc=0; inc<60; inc+=10) for(rr=60; rr<120; rr+=15)
		{
			intermed = (float) theta + inc;
			intertwo = (float) rr + 16;
			us_palletpoly->xv[0] = (INTBIG)(((float)rr)* cos(intermed*DEGTORAD)) + CENTERX;
			us_palletpoly->yv[0] = (INTBIG)(((float)rr)* sin(intermed*DEGTORAD)) + CENTERY;
			us_palletpoly->xv[1] = (INTBIG)(intertwo * cos(intermed*DEGTORAD)) + CENTERX;
			us_palletpoly->yv[1] = (INTBIG)(intertwo * sin(intermed*DEGTORAD)) + CENTERY;
			intermed += 12.0;
			us_palletpoly->xv[2] = (INTBIG)(intertwo * cos(intermed*DEGTORAD)) + CENTERX;
			us_palletpoly->yv[2] = (INTBIG)(intertwo * sin(intermed*DEGTORAD)) + CENTERY;
			us_palletpoly->xv[3] = (INTBIG)(((float)rr) * cos(intermed*DEGTORAD)) + CENTERX;
			us_palletpoly->yv[3] = (INTBIG)(((float)rr) * sin(intermed*DEGTORAD)) + CENTERY;
			us_palletgraph.col = i++;
			(void)us_showpoly(us_palletpoly, w);
		}
	}

	/* draw the intensity slider */
	us_palletgraph.col = MAPWHITE;
	makerectpoly(250, 450, 245, 265, us_palletpoly);
	us_palletpoly->style = TEXTBOX;
	us_palletpoly->font = TXTMEDIUM;
	us_palletpoly->string = "Mix Intensity:";
	(void)us_showpoly(us_palletpoly, w);
	for(i=MAPSATUR; i<=MAPENDSATUR; i++)
	{
		us_palletgraph.col = i;
		maketruerectpoly(300, 400, 9*(i-MAPSATUR)+20, 9*(i-MAPSATUR)+29, us_palletpoly);
		us_palletpoly->style = FILLEDRECT;
		(void)us_showpoly(us_palletpoly, w);
	}
	maketruerectpoly(300, 400, 20, 9*(MAPENDSATUR-MAPSATUR)+29, us_palletpoly);
	us_palletgraph.col = MAPWHITE;
	us_palletpoly->style = CLOSEDRECT;
	(void)us_showpoly(us_palletpoly, w);

	/* draw the color set selectors */
	us_palletgraph.col = MAPWHITE;
	makerectpoly(0, 511, 490, 512, us_palletpoly);
	us_palletpoly->style = TEXTBOX;
	us_palletpoly->font = TXTMEDIUM;
	us_palletpoly->string = "Select colors for palette:";
	(void)us_showpoly(us_palletpoly, w);
	for(i=0; i<7; i++)
	{
		/* draw name in the selector entry */
		us_palletgraph.col = MAPWHITE;
		makerectpoly(4+i*73, 69+i*73, 470, 490, us_palletpoly);
		us_palletpoly->style = TEXTBOX;
		us_palletpoly->font = TXTMEDIUM;
		if (i == 0) us_palletpoly->string = "PRIMARIES"; else
			if (i == 1) us_palletpoly->string = "SPECIALS"; else
		{
			(void)initinfstr();
			(void)addstringtoinfstr(us_pallayername[i-2]);
			(void)addtoinfstr('=');
			(void)addstringtoinfstr(us_pallayerabbrev[i-2]);
			us_palletpoly->string = returninfstr();
		}
		(void)us_showpoly(us_palletpoly, w);

		/* draw outline around pallet entry */
		us_palletpoly->style = CLOSED;
		(void)us_showpoly(us_palletpoly, w);
	}

	/* draw the palette boxes */
	us_palletgraph.col = MAPWHITE;
	makerectpoly(0, 511, 450, 470, us_palletpoly);
	us_palletpoly->style = TEXTBOX;
	us_palletpoly->font = TXTMEDIUM;
	us_palletpoly->string = "Fill palette:";
	(void)us_showpoly(us_palletpoly, w);
	for(i=0; i<MAXCOL; i++) us_paldrawsplotch(i, w);
	us_paldrawentries(w);

	us_palcur = 0;

	us_rgbtohsv((INTSML)us_redmap[us_paltrueindex[us_palcur]],
		(INTSML)us_greenmap[us_paltrueindex[us_palcur]],
			(INTSML)us_bluemap[us_paltrueindex[us_palcur]],
				&us_palcurtheta, &us_palcurr, &us_palcurinten);
	us_palcurtheta *= 360.0;
	us_palcurr *= 120.0;
	us_paldrawhue((INTSML)(125.0 * cos(us_palcurtheta * DEGTORAD) + CENTERX),
		(INTSML)(125.0 * sin(us_palcurtheta * DEGTORAD) + CENTERY), MAPWHITE, w);
	us_paldrawsidearrow((INTSML)(us_palcurinten * 210.0 + 20), MAPWHITE, w);
	us_paldrawarrow(us_palcur, MAPWHITE, w);
}

INTSML us_palcharhandler(WINDOW *w, INTSML chr)
{
	ttyputmsg("Can only use the mouse in this window");
	return(0);
}

void us_paltermhandler(WINDOW *w)
{
	REGISTER INTSML i;

	startobjectchange((INTBIG)us_aid, VAID);
	(void)setvalkey((INTBIG)us_aid, VAID, us_colormap_red, (INTBIG)us_redmap,
		VINTEGER|VISARRAY|(256<<VLENGTHSH)|VDONTSAVE);
	(void)setvalkey((INTBIG)us_aid, VAID, us_colormap_green, (INTBIG)us_greenmap,
		VINTEGER|VISARRAY|(256<<VLENGTHSH)|VDONTSAVE);
	(void)setvalkey((INTBIG)us_aid, VAID, us_colormap_blue, (INTBIG)us_bluemap,
		VINTEGER|VISARRAY|(256<<VLENGTHSH)|VDONTSAVE);
	endobjectchange((INTBIG)us_aid, VAID);

	startobjectchange((INTBIG)w, VWINDOW);
	(void)setval((INTBIG)w, VWINDOW, "screenlx", us_palletwindow.screenlx, VINTEGER);
	(void)setval((INTBIG)w, VWINDOW, "screenhx", us_palletwindow.screenhx, VINTEGER);
	(void)setval((INTBIG)w, VWINDOW, "screenly", us_palletwindow.screenly, VINTEGER);
	(void)setval((INTBIG)w, VWINDOW, "screenhy", us_palletwindow.screenhy, VINTEGER);
	(void)setval((INTBIG)w, VWINDOW, "buttonhandler", (INTBIG)us_palletwindow.buttonhandler, VADDRESS);
	(void)setval((INTBIG)w, VWINDOW, "charhandler", (INTBIG)us_palletwindow.charhandler, VADDRESS);
	(void)setval((INTBIG)w, VWINDOW, "termhandler", (INTBIG)us_palletwindow.termhandler, VADDRESS);
	(void)setval((INTBIG)w, VWINDOW, "redisphandler", (INTBIG)us_palletwindow.redisphandler, VADDRESS);
	(void)setval((INTBIG)w, VWINDOW, "state", us_palletwindow.state, VINTEGER);
	(void)setval((INTBIG)w, VWINDOW, "curnodeproto", (INTBIG)us_palletwindow.curnodeproto, VNODEPROTO);
	endobjectchange((INTBIG)w, VWINDOW);
	us_drawwindow(w, el_colwinbor);

	for(i=0; i<5; i++)
	{
		efree(us_pallayerabbrev[i]);
		efree(us_pallayername[i]);
	}
}

void us_palbuttonhandler(WINDOW *w, INTSML button, INTSML inx, INTSML iny)
{
	REGISTER INTSML i, j, sel, xsize, xspace, ent;
	INTBIG x, y, curred, curgreen, curblue;

	x = inx;   y = iny;
	us_scaletowindow(&x, &y, w);
	if (x >= 450 && y >= 90 && y <= 110)
	{
		/* quit button */
		us_paltermhandler(w);
		return;
	}
	if (x >= 450 && y >= 190 && y <= 210)
	{
		/* cancel button */
		for(i=0; i<256; i++)
		{
			us_redmap[i] = us_oredmap[i];
			us_greenmap[i] = us_ogreenmap[i];
			us_bluemap[i] = us_obluemap[i];
		}
		us_palmakeinitcolormap();
		us_paltermhandler(w);
		return;
	}

	/* other buttons: check for hue/saturation hit */
	if (y <= CENTERY+120 && y >= CENTERY-120 && x <= CENTERX+120 && x >= CENTERX-120)
	{
		us_paldrawhue((INTSML)(125.0 * cos(us_palcurtheta * DEGTORAD) + CENTERX),
			(INTSML)(125.0 * sin(us_palcurtheta * DEGTORAD) + CENTERY), MAPBLACK, w);
		us_palcurtheta = (float)((figureangle(CENTERX, CENTERY, x, y) + 5) / 10);
		if (us_palcurtheta < 0.0) us_palcurtheta += 360.0;
		us_paldrawhue((INTSML)(125.0 * cos(us_palcurtheta * DEGTORAD) + CENTERX),
			(INTSML)(125.0 * sin(us_palcurtheta * DEGTORAD) + CENTERY), MAPWHITE, w);
		us_palcurr = sqrt((float)((y-CENTERY)*(y-CENTERY) + (x-CENTERX)*(x-CENTERX)));
		if (us_palcurr > 125.0) return;
		us_hsvtorgb(us_palcurtheta / 360.0, us_palcurr / 125.0, us_palcurinten, &curred,
			&curgreen, &curblue);
		us_setcolorentry((INTSML)(us_palcur+MAPEDIT), (INTSML)curred, (INTSML)curgreen, (INTSML)curblue,
			0, 0);
		us_paldrawsplotch(us_palcur, w);
		us_filltrueentry(us_palcur, (INTSML)curred, (INTSML)curgreen, (INTSML)curblue);
		return;
	}

	/* check for intensity slider hit */
	if (x >= 300 && x <= 400 && y >= 20 && y <= 250)
	{
		us_paldrawsidearrow((INTSML)(us_palcurinten * 210.0 + 20), MAPBLACK, w);
		us_palcurinten = (((float)y)-20.0) / 210.0;
		if (us_palcurinten > 1.0) us_palcurinten = 1.0;
		us_paldrawsidearrow((INTSML)(us_palcurinten * 210.0 + 20), MAPWHITE, w);
		us_hsvtorgb(us_palcurtheta / 360.0, us_palcurr / 120.0,
			us_palcurinten, &curred, &curgreen, &curblue);
		us_setcolorentry((INTSML)(us_palcur+MAPEDIT), (INTSML)curred, (INTSML)curgreen, (INTSML)curblue,
			0, 0);
		us_paldrawsplotch(us_palcur, w);
		us_filltrueentry(us_palcur, (INTSML)curred, (INTSML)curgreen, (INTSML)curblue);
		return;
	}

	/* check for palette hits */
	if (y >= 280 && y <= 440)
	{
		xsize = 600 / MAXCOL;
		xspace = 1000 / MAXCOL - xsize;
		j = -xsize-xspace/2;
		for(i=0; i<MAXCOL/2; i++)
		{
			j += xsize + xspace;
			if (x >= j && x <= j+xsize)
			{
				us_paldrawarrow(us_palcur, MAPBLACK, w);
				if (y < 360) i += MAXCOL/2;
				us_palcur = i;
				us_paldrawarrow(us_palcur, MAPWHITE, w);

				us_paldrawhue((INTSML)(125.0 * cos(us_palcurtheta * DEGTORAD) + CENTERX),
					(INTSML)(125.0 * sin(us_palcurtheta * DEGTORAD) + CENTERY), MAPBLACK, w);
				us_paldrawsidearrow((INTSML)(us_palcurinten * 210.0 + 20), MAPBLACK, w);
				us_rgbtohsv((INTSML)us_redmap[us_paltrueindex[us_palcur]],
					(INTSML)us_greenmap[us_paltrueindex[us_palcur]],
						(INTSML)us_bluemap[us_paltrueindex[us_palcur]],
							&us_palcurtheta, &us_palcurr, &us_palcurinten);
				us_palcurtheta *= 360.0;
				us_palcurr *= 120.0;
				us_paldrawsidearrow((INTSML)(us_palcurinten * 210.0 + 20), MAPWHITE, w);
				us_paldrawhue((INTSML)(125.0 * cos(us_palcurtheta * DEGTORAD) + CENTERX),
					(INTSML)(125.0 * sin(us_palcurtheta * DEGTORAD) + CENTERY), MAPWHITE, w);
				break;
			}
		}
		return;
	}

	/* check for selector hits */
	if (y >= 470 && y <= 490)
	{
		us_paldrawhue((INTSML)(125.0 * cos(us_palcurtheta * DEGTORAD) + CENTERX),
			(INTSML)(125.0 * sin(us_palcurtheta * DEGTORAD) + CENTERY), MAPBLACK, w);
		us_paldrawsidearrow((INTSML)(us_palcurinten * 210.0 + 20), MAPBLACK, w);
		sel = x / 73;
		switch (sel)
		{
			case 0:				/* primaries */
				us_palbuildprimaries(LAYERT1, LAYERT2, LAYERT3, LAYERT4,LAYERT5);
				break;
			case 1:				/* primaries */
				us_palbuildspecials();
				break;
			case 2:				/* layer 1 focus */
				us_palbuildindex(LAYERT1, LAYERT2, LAYERT3, LAYERT4, LAYERT5);
				break;
			case 3:				/* layer 2 focus */
				us_palbuildindex(LAYERT2, LAYERT1, LAYERT3, LAYERT4, LAYERT5);
				break;
			case 4:				/* layer 3 focus */
				us_palbuildindex(LAYERT3, LAYERT1, LAYERT2, LAYERT4, LAYERT5);
				break;
			case 5:				/* layer 4 focus */
				us_palbuildindex(LAYERT4, LAYERT1, LAYERT2, LAYERT3, LAYERT5);
				break;
			case 6:				/* layer 5 focus */
				us_palbuildindex(LAYERT5, LAYERT1, LAYERT2, LAYERT3, LAYERT4);
				break;
		}
		startobjectchange((INTBIG)us_aid, VAID);
		for(i=0; i<MAXCOL; i++)
		{
			ent = us_paltrueindex[i];
			(void)setindkey((INTBIG)us_aid, VAID, us_colormap_red, i+MAPEDIT, us_redmap[ent]);
			(void)setindkey((INTBIG)us_aid, VAID, us_colormap_green, i+MAPEDIT, us_greenmap[ent]);
			(void)setindkey((INTBIG)us_aid, VAID, us_colormap_blue, i+MAPEDIT, us_bluemap[ent]);
		}
		endobjectchange((INTBIG)us_aid, VAID);
		for(i=0; i<MAXCOL; i++) us_paldrawsplotch(i, w);
		us_paldrawentries(w);
		us_rgbtohsv((INTSML)us_redmap[us_paltrueindex[us_palcur]],
			(INTSML)us_greenmap[us_paltrueindex[us_palcur]],
				(INTSML)us_bluemap[us_paltrueindex[us_palcur]],
					&us_palcurtheta, &us_palcurr, &us_palcurinten);
		us_palcurtheta *= 360.0;
		us_palcurr *= 120.0;
		us_paldrawsidearrow((INTSML)(us_palcurinten * 210.0 + 20), MAPWHITE, w);
		us_paldrawhue((INTSML)(125.0 * cos(us_palcurtheta * DEGTORAD) + CENTERX),
			(INTSML)(125.0 * sin(us_palcurtheta * DEGTORAD) + CENTERY), MAPWHITE, w);
		setactivity("COLOR MIX");
		return;
	}
}

void us_filltrueentry(INTSML ent, INTSML red, INTSML green, INTSML blue)
{
	REGISTER INTSML trueone, j;

	trueone = us_paltrueindex[ent];
	us_redmap[trueone] = red;
	us_greenmap[trueone] = green;
	us_bluemap[trueone] = blue;
	if ((trueone&LAYERH) == LAYERH)
	{
		/* set all highlight colors */
		for(j=0; j<256; j++) if ((j&LAYERH) == LAYERH)
		{
			us_redmap[j] = red;
			us_greenmap[j] = green;
			us_bluemap[j] = blue;
		}
	} else if ((trueone&(LAYERG|LAYERH)) == LAYERG)
	{
		/* set all grid colors */
		for(j=0; j<256; j++) if ((j&(LAYERG|LAYERH)) == LAYERG)
		{
			us_redmap[j] = red;
			us_greenmap[j] = green;
			us_bluemap[j] = blue;
		}
	}
}

void us_paldrawsplotch(INTSML which, WINDOW *w)
{
	INTSML xsize, xspace, initialx;

	xsize = 600 / MAXCOL;
	xspace = 1000 / MAXCOL - xsize;
	initialx = (xsize+xspace) * (which%(MAXCOL/2)) + xspace / 2;
	if (which >= MAXCOL/2)
		maketruerectpoly(initialx, initialx+xsize, 290, 345, us_palletpoly); else
			maketruerectpoly(initialx, initialx+xsize, 375, 430, us_palletpoly);
	us_palletgraph.col = which+MAPEDIT;
	us_palletpoly->style = FILLEDRECT;
	(void)us_showpoly(us_palletpoly, w);
}

void us_palbuildindex(INTSML layer1, INTSML layer2, INTSML layer3, INTSML layer4, INTSML layer5)
{
	us_paltrueindex[0]  = layer1;
	us_paltrueindex[1]  = layer1 | layer2;
	us_paltrueindex[2]  = layer1 |          layer3;
	us_paltrueindex[3]  = layer1 | layer2 | layer3;
	us_paltrueindex[4]  = layer1 |                   layer4;
	us_paltrueindex[5]  = layer1 | layer2 |          layer4;
	us_paltrueindex[6]  = layer1 |          layer3 | layer4;
	us_paltrueindex[7]  = layer1 | layer2 | layer3 | layer4;
	us_paltrueindex[8]  = layer1 |                            layer5;
	us_paltrueindex[9]  = layer1 | layer2 |                   layer5;
	us_paltrueindex[10] = layer1 |          layer3 |          layer5;
	us_paltrueindex[11] = layer1 | layer2 | layer3 |          layer5;
	us_paltrueindex[12] = layer1 |                   layer4 | layer5;
	us_paltrueindex[13] = layer1 | layer2 |          layer4 | layer5;
	us_paltrueindex[14] = layer1 |          layer3 | layer4 | layer5;
	us_paltrueindex[15] = layer1 | layer2 | layer3 | layer4 | layer5;
}

void us_palbuildprimaries(INTSML layer1, INTSML layer2, INTSML layer3, INTSML layer4, INTSML layer5)
{
	us_paltrueindex[0]  = ALLOFF;
	us_paltrueindex[1]  = layer1;
	us_paltrueindex[2]  = layer2;
	us_paltrueindex[3]  = layer3;
	us_paltrueindex[4]  = layer4;
	us_paltrueindex[5]  = layer5;
	us_paltrueindex[6]  = layer1 | layer2;
	us_paltrueindex[7]  = layer1 | layer3;
	us_paltrueindex[8]  = layer1 | layer4;
	us_paltrueindex[9]  = layer1 | layer5;
	us_paltrueindex[10] = layer2 | layer3;
	us_paltrueindex[11] = layer2 | layer4;
	us_paltrueindex[12] = layer2 | layer5;
	us_paltrueindex[13] = layer3 | layer4;
	us_paltrueindex[14] = layer3 | layer5;
	us_paltrueindex[15] = layer4 | layer5;
}

void us_palbuildspecials(void)
{
	us_paltrueindex[0]  = ALLOFF;
	us_paltrueindex[1]  = GRID;
	us_paltrueindex[2]  = HIGHLIT;
	us_paltrueindex[3]  = el_colfacettxt;
	us_paltrueindex[4]  = el_colfacet;
	us_paltrueindex[5]  = el_colwinbor;
	us_paltrueindex[6]  = el_colhwinbor;
	us_paltrueindex[7]  = el_colmenbor;
	us_paltrueindex[8]  = el_colhmenbor;
	us_paltrueindex[9]  = el_colmentxt;
	us_paltrueindex[10] = el_colmengly;
	us_paltrueindex[11] = el_colcursor;
	us_paltrueindex[12] = 0354;
	us_paltrueindex[13] = 0364;
	us_paltrueindex[14] = 0374;
	us_paltrueindex[15] = ALLOFF;
}

/*
 * helper routine to draw an arrow pointing to palette entry "entr"
 * in color "color"
 */
void us_paldrawarrow(INTSML entr, INTSML color, WINDOW *w)
{
	REGISTER INTSML initialx, dist;

	dist = 1000 / MAXCOL;
	if (entr >= MAXCOL/2) initialx = -500/MAXCOL + (entr+1-MAXCOL/2)*dist; else
		initialx = -500/MAXCOL + (entr+1)*dist;

	us_palletgraph.col = color;
	us_palletpoly->xv[0] = initialx;   us_palletpoly->yv[0] = 355;
	us_palletpoly->xv[1] = initialx;   us_palletpoly->yv[1] = 365;
	if (entr >= MAXCOL/2)
	{
		/* down pointing arrow */
		us_palletpoly->xv[2] = initialx;   us_palletpoly->yv[2] = 355;
		us_palletpoly->xv[3] = initialx-5; us_palletpoly->yv[3] = 360;
		us_palletpoly->xv[4] = initialx;   us_palletpoly->yv[4] = 355;
		us_palletpoly->xv[5] = initialx+5; us_palletpoly->yv[5] = 360;
	} else
	{
		/* up pointing arrow */
		us_palletpoly->xv[2] = initialx;   us_palletpoly->yv[2] = 365;
		us_palletpoly->xv[3] = initialx-5; us_palletpoly->yv[3] = 360;
		us_palletpoly->xv[4] = initialx;   us_palletpoly->yv[4] = 365;
		us_palletpoly->xv[5] = initialx+5; us_palletpoly->yv[5] = 360;
	}
	us_palletpoly->count = 6;
	us_palletpoly->style = VECTORS;
	(void)us_showpoly(us_palletpoly, w);
}

/*
 * helper routine to draw a left-pointing arrow at Y coordinate "initialy"
 * in color "color" pointing to the intensity slider
 */
void us_paldrawsidearrow(INTSML initialy, INTSML color, WINDOW *w)
{
	us_palletgraph.col = color;
	us_palletpoly->xv[0] = 402;   us_palletpoly->yv[0] = initialy;
	us_palletpoly->xv[1] = 412;   us_palletpoly->yv[1] = initialy-5;
	us_palletpoly->xv[2] = 412;   us_palletpoly->yv[2] = initialy+5;
	us_palletpoly->xv[3] = 402;   us_palletpoly->yv[3] = initialy;
	us_palletpoly->xv[4] = 402;   us_palletpoly->yv[4] = initialy;
	us_palletpoly->xv[5] = 422;   us_palletpoly->yv[5] = initialy;
	us_palletpoly->count = 6;
	us_palletpoly->style = VECTORS;
	(void)us_showpoly(us_palletpoly, w);
}

/*
 * helper routine to draw a cross at coordinate (x, y) in color "color"
 * around the circumference of the hue/saturation wheel
 */
void us_paldrawhue(INTSML x, INTSML y, INTSML color, WINDOW *w)
{
	us_palletgraph.col = color;
	us_palletpoly->xv[0] = x-2;   us_palletpoly->yv[0] = y-2;
	us_palletpoly->xv[1] = x+2;   us_palletpoly->yv[1] = y+2;
	us_palletpoly->xv[2] = x+2;   us_palletpoly->yv[2] = y-2;
	us_palletpoly->xv[3] = x-2;   us_palletpoly->yv[3] = y+2;
	us_palletpoly->count = 4;
	us_palletpoly->style = VECTORS;
	(void)us_showpoly(us_palletpoly, w);
}

void us_paldrawentries(WINDOW *w)
{
	REGISTER INTSML xsize, xspace, initialx, i;

	/* erase the labels */
	us_palletgraph.col = MAPBLACK;
	maketruerectpoly(0, 511, 430, 450, us_palletpoly);
	us_palletpoly->style = FILLEDRECT;
	(void)us_showpoly(us_palletpoly, w);
	maketruerectpoly(0, 511, 270, 290, us_palletpoly);
	(void)us_showpoly(us_palletpoly, w);

	/* redraw the labels */
	xsize = 600 / MAXCOL;
	xspace = 1000 / MAXCOL - xsize;
	initialx = -xsize - xspace / 2;
	for(i=0; i<MAXCOL/2; i++)
	{
		initialx += xsize + xspace;

		/* draw name above top pallet entry */
		us_palletgraph.col = MAPWHITE;
		makerectpoly(initialx, initialx+xsize, 430, 450, us_palletpoly);
		us_palletpoly->style = TEXTBOX;
		us_palletpoly->font = TXTMEDIUM;
		us_palletpoly->string = us_palentname(us_paltrueindex[i]);
		(void)us_showpoly(us_palletpoly, w);

		/* draw outline around top pallet entry */
		maketruerectpoly(initialx, initialx+xsize, 375, 430, us_palletpoly);
		us_palletgraph.col = MAPWHITE;
		us_palletpoly->style = CLOSEDRECT;
		(void)us_showpoly(us_palletpoly, w);

		/* draw name below bottom pallet entry */
		us_palletgraph.col = MAPWHITE;
		makerectpoly(initialx, initialx+xsize, 270, 290, us_palletpoly);
		us_palletpoly->style = TEXTBOX;
		us_palletpoly->font = TXTMEDIUM;
		us_palletpoly->string = us_palentname(us_paltrueindex[i+MAXCOL/2]);
		(void)us_showpoly(us_palletpoly, w);

		/* draw outline around bottom pallet entry */
		maketruerectpoly(initialx, initialx+xsize, 290, 345, us_palletpoly);
		us_palletgraph.col = MAPWHITE;
		us_palletpoly->style = CLOSEDRECT;
		(void)us_showpoly(us_palletpoly, w);
	}
}

char *us_palentname(INTSML ind)
{
	if (ind == ALLOFF) return("Bkgr");
	if (ind == GRID) return("Grid");
	if (ind == HIGHLIT) return("High");
	if (ind == 0354) return("Extra1");
	if (ind == 0364) return("Extra2");
	if (ind == 0374) return("Extra3");
	if (ind == el_colfacettxt) return("FName");
	if (ind == el_colfacet) return("FOutl");
	if (ind == el_colwinbor) return("Wind");
	if (ind == el_colhwinbor) return("HWind");
	if (ind == el_colmenbor) return("MnBor");
	if (ind == el_colhmenbor) return("HMnBor");
	if (ind == el_colmentxt) return("MenTxt");
	if (ind == el_colmengly) return("MenGly");
	if (ind == el_colcursor) return("Curs");
	(void)initinfstr();
	if ((ind&LAYERT1) != 0) (void)addstringtoinfstr(us_pallayerabbrev[0]);
	if ((ind&LAYERT2) != 0) (void)addstringtoinfstr(us_pallayerabbrev[1]);
	if ((ind&LAYERT3) != 0) (void)addstringtoinfstr(us_pallayerabbrev[2]);
	if ((ind&LAYERT4) != 0) (void)addstringtoinfstr(us_pallayerabbrev[3]);
	if ((ind&LAYERT5) != 0) (void)addstringtoinfstr(us_pallayerabbrev[4]);
	return(returninfstr());
}
