/*
 * Electric(tm) VLSI Design System
 *
 * File: usrtrack.c
 * User interface aid: cursor tracking routines
 * Written by: Steven M. Rubin, Electric Editor Incorporated
 *
 * Copyright (c) 1998 Electric Editor Incorporated.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Electric Editor Incorporated
 * 23470 Sunset Drive, Suite 108
 * Los Gatos, California 95033
 * support@electriceditor.com
 */

/*
 * the code in this module makes me quiver with trepidation whenever it
 * breaks ... smr
 */
#include "global.h"
#include "egraphics.h"
#include "usr.h"
#include "usrtrack.h"
#include "efunction.h"
#include "tecart.h"
#include "tecgen.h"

#define	MAXTRACE 400
static INTSML us_tracelist;
static INTBIG us_tracedata[MAXTRACE];
static GRAPHICS us_highl = {LAYERH, 0, {SOLIDC, SOLIDC, SOLIDC, SOLIDC},
	{0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF}, NOVARIABLE, 0};

static INTBIG us_dragx, us_dragy, us_dragox, us_dragoy, us_dragoffx,
	us_dragoffy, us_lastcurx, us_lastcury, us_dragpoint, us_dragjoinfactor,
	us_dragnodetotal, us_dragdescript, us_cantdrag;
static INTSML us_dragshown, us_dragangle, us_dragstate, us_dragfport, us_dragextra,
	us_dragnobox, us_dragstayonhigh, us_dragstill, us_dragcorner;
static WINDOW *us_dragwindow;
static POLYGON *us_dragpoly = NOPOLYGON;
static NODEPROTO *us_dragnodeproto;
static NODEINST *us_dragnodeinst, **us_dragnodelist;
static ARCINST *us_dragarcinst;
static GEOM *us_dragobject, **us_dragobjectlist;
static HIGHLIGHT us_draghigh;
static char us_dragmessage[100];

/* prototypes for local routines */
void us_finddoibegin(INTSML);
void us_multidragdraw(INTBIG, INTBIG, INTBIG);
void us_invertstretch(void);
void us_wanttoinvert(INTBIG fx, INTBIG fy, INTBIG tx, INTBIG ty, WINDOW *w);

/************************* NODE TRACE *************************/

/* initialization routine when reading a cursor trace */
void us_tracebegin(void)
{
	us_tracelist = 0;
	us_highl.col = HIGHLIT;
}

/* cursor advance routine when reading a cursor trace */
INTSML us_tracedown(INTBIG x, INTBIG y)
{
	if (us_tracelist >= MAXTRACE) return(1);
	if (us_tracelist != 0)
		us_drawline(el_curwindow, (INTSML)us_tracedata[us_tracelist-2], (INTSML)us_tracedata[us_tracelist-1],
			(INTSML)x, (INTSML)y, &us_highl, 0);
	us_tracedata[us_tracelist++] = x;
	us_tracedata[us_tracelist++] = y;
	return(0);
}

/* termination routine when reading a cursor trace */
void us_traceup(void)
{
	REGISTER INTSML i;

	if (us_tracelist == 0)
	{
		(void)setval((INTBIG)us_aid, VAID, us_commandvarname('T'), (INTBIG)"nothing",
			VSTRING|VDONTSAVE);
		return;
	}
	(void)setval((INTBIG)us_aid, VAID, us_commandvarname('T'), (INTBIG)us_tracedata,
		(INTBIG)(VINTEGER|VISARRAY|(us_tracelist<<VLENGTHSH)|VDONTSAVE));
	us_highl.col = 0;
	for(i=2; i<us_tracelist; i += 2)
		us_drawline(el_curwindow, (INTSML)us_tracedata[i-2], (INTSML)us_tracedata[i-1],
			(INTSML)us_tracedata[i], (INTSML)us_tracedata[i+1], &us_highl, 0);
}

/* preinitialization routine when creating or moving a point on a node */
void us_pointinit(NODEINST *node, INTSML point)
{
	us_dragnodeinst = node;
	us_dragpoint = point;
}

/* initialization routine when creating or moving a point on a node */
void us_pointbegin(void)
{
	/* initialize polygon */
	if (us_dragpoly == NOPOLYGON) us_dragpoly = allocpolygon(4, us_aid->cluster);
	us_dragpoly->desc = &us_highl;

	us_dragshown = 0;
	(void)getxy(&us_dragx, &us_dragy);
	us_dragwindow = el_curwindow;
}

/* initialization routine when finding and moving a point on a node */
void us_findpointbegin(void)
{
	REGISTER VARIABLE *highvar;
	HIGHLIGHT newhigh;

	/* initialize polygon */
	if (us_dragpoly == NOPOLYGON) us_dragpoly = allocpolygon(4, us_aid->cluster);
	us_dragpoly->desc = &us_highl;

	us_dragshown = 0;
	if (getxy(&us_dragx, &us_dragy)) return;
	gridalign(&us_dragx, &us_dragy, us_alignment);
	us_setcursorpos(0, us_dragx, us_dragy);
	us_dragwindow = el_curwindow;

	/* select the current point on the node */
	highvar = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_highlighted);
	if (highvar == NOVARIABLE) return;
	(void)us_makehighlight(((char **)highvar->addr)[0], &newhigh);
	newhigh.fromport = NOPORTPROTO;
	us_setfind(&newhigh, 1, 0, 0, 0);

	/* get the point to be moved */
	highvar = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_highlighted);
	if (highvar == NOVARIABLE) return;
	(void)us_makehighlight(((char **)highvar->addr)[0], &newhigh);
	us_dragpoint = newhigh.frompoint;
}

/* cursor advance routine when creating a point on a node */
INTSML us_addpdown(INTBIG x, INTBIG y)
{
	REGISTER INTSML p, size;
	REGISTER INTBIG cx, cy;
	XARRAY trans;
	REGISTER VARIABLE *var;

	/* grid align the cursor value */
	if (us_setxy((INTSML)x, (INTSML)y) != 0 || us_dragwindow != el_curwindow)
	{
		(void)us_setxy((INTSML)us_lastcurx, (INTSML)us_lastcury);
		return(0);
	}
	(void)getxy(&us_lastcurx, &us_lastcury);
	gridalign(&us_lastcurx, &us_lastcury, us_alignment);
	us_setcursorpos(0, us_lastcurx, us_lastcury);

	/* if the lines are already being shown, erase them */
	if (us_dragshown != 0)
	{
		/* if it didn't actually move, go no further */
		if (us_dragx == us_lastcurx && us_dragy == us_lastcury) return(0);

		/* undraw the box */
		us_highl.col = 0;
		(void)us_showpoly(us_dragpoly, us_dragwindow);
	}

	makerot(us_dragnodeinst, trans);
	var = gettrace(us_dragnodeinst);
	if (var == NOVARIABLE)
	{
		us_dragpoly->xv[0] = us_lastcurx;
		us_dragpoly->yv[0] = us_lastcury;
		us_dragpoly->count = 1;
		us_dragpoly->style = CROSS;
	} else
	{
		size = getlength(var) / 2;
		cx = (us_dragnodeinst->lowx + us_dragnodeinst->highx) / 2;
		cy = (us_dragnodeinst->lowy + us_dragnodeinst->highy) / 2;
		p = 0;
		if (us_dragpoint <= size)
		{
			xform(((INTBIG *)var->addr)[(us_dragpoint-1)*2] + cx,
				((INTBIG *)var->addr)[(us_dragpoint-1)*2+1] + cy,
					&us_dragpoly->xv[p], &us_dragpoly->yv[p], trans);
			p++;
		}
		us_dragpoly->xv[p] = us_lastcurx;
		us_dragpoly->yv[p] = us_lastcury;
		p++;
		if (us_dragpoint < size)
		{
			xform(((INTBIG *)var->addr)[us_dragpoint*2] + cx,
				((INTBIG *)var->addr)[us_dragpoint*2+1] + cy,
					&us_dragpoly->xv[p], &us_dragpoly->yv[p], trans);
			p++;
		}
		us_dragpoly->count = p;
		us_dragpoly->style = OPENED;
	}

	us_dragx = us_lastcurx;
	us_dragy = us_lastcury;

	/* draw the new box */
	us_highl.col = HIGHLIT;
	(void)us_showpoly(us_dragpoly, us_dragwindow);
	us_dragshown = 1;
	return(0);
}

/* cursor advance routine when moving a point on a node */
INTSML us_movepdown(INTBIG x, INTBIG y)
{
	REGISTER INTSML p, size;
	REGISTER INTBIG cx, cy;
	XARRAY trans;
	REGISTER VARIABLE *var;

	/* grid align the cursor value */
	if (us_setxy((INTSML)x, (INTSML)y) != 0 || us_dragwindow != el_curwindow)
	{
		(void)us_setxy((INTSML)us_lastcurx, (INTSML)us_lastcury);
		return(0);
	}
	(void)getxy(&us_lastcurx, &us_lastcury);
	gridalign(&us_lastcurx, &us_lastcury, us_alignment);
	us_setcursorpos(0, us_lastcurx, us_lastcury);

	/* if the lines are already being shown, erase them */
	if (us_dragshown != 0)
	{
		/* if it didn't actually move, go no further */
		if (us_dragx == us_lastcurx && us_dragy == us_lastcury) return(0);

		/* undraw the box */
		us_highl.col = 0;
		(void)us_showpoly(us_dragpoly, us_dragwindow);
	}

	makerot(us_dragnodeinst, trans);
	var = gettrace(us_dragnodeinst);
	if (var == NOVARIABLE) return(0);

	size = getlength(var) / 2;
	cx = (us_dragnodeinst->lowx + us_dragnodeinst->highx) / 2;
	cy = (us_dragnodeinst->lowy + us_dragnodeinst->highy) / 2;
	p = 0;
	if (us_dragpoint > 1)
	{
		xform(((INTBIG *)var->addr)[(us_dragpoint-2)*2] + cx,
			((INTBIG *)var->addr)[(us_dragpoint-2)*2+1] + cy,
				&us_dragpoly->xv[p], &us_dragpoly->yv[p], trans);
		p++;
	}
	us_dragpoly->xv[p] = us_lastcurx;
	us_dragpoly->yv[p] = us_lastcury;
	p++;
	if (us_dragpoint < size)
	{
		xform(((INTBIG *)var->addr)[us_dragpoint*2] + cx,
			((INTBIG *)var->addr)[us_dragpoint*2+1] + cy,
				&us_dragpoly->xv[p], &us_dragpoly->yv[p], trans);
		p++;
	}
	us_dragpoly->count = p;
	us_dragpoly->style = OPENED;

	us_dragx = us_lastcurx;
	us_dragy = us_lastcury;

	/* draw the new box */
	us_highl.col = HIGHLIT;
	(void)us_showpoly(us_dragpoly, us_dragwindow);
	us_dragshown = 1;
	return(0);
}

/************************* TEXT GRAB-POINT *************************/

/* preinitialization routine when positioning text's grab point */
void us_textgrabinit(INTBIG olddescript, INTBIG xw, INTBIG yw, INTBIG xc, INTBIG yc,
	GEOM *geom)
{
	/* save text extent information */
	us_dragdescript = olddescript;
	us_dragx = xc;          us_dragy = yc;
	us_dragox = xw;         us_dragoy = yw;
	us_dragobject = geom;
}

/* initialization routine when positioning text's grab-point */
void us_textgrabbegin(void)
{
	/* initialize polygon */
	if (us_dragpoly == NOPOLYGON) us_dragpoly = allocpolygon(4, us_aid->cluster);
	us_dragpoly->desc = &us_highl;

	us_dragshown = 0;
	us_dragwindow = el_curwindow;
}

/* cursor advance routine when positioning text's grab-point */
INTSML us_textgrabdown(INTBIG x, INTBIG y)
{
	REGISTER INTBIG descript;
	REGISTER INTSML style;
	XARRAY trans;

	/* get the cursor value */
	if (us_setxy((INTSML)x, (INTSML)y) != 0 || us_dragwindow != el_curwindow)
	{
		(void)us_setxy((INTSML)us_lastcurx, (INTSML)us_lastcury);
		return(0);
	}
	(void)getxy(&us_lastcurx, &us_lastcury);

	/* if the box is already being shown, erase it */
	if (us_dragshown != 0)
	{
		/* undraw the box */
		us_highl.col = 0;
		(void)us_showpoly(us_dragpoly, us_dragwindow);
	}

	/* convert the cursor position into a text descriptor */
	descript = us_figuregrabpoint(us_dragdescript, us_lastcurx, us_lastcury,
		us_dragx, us_dragy, us_dragox, us_dragoy);
	descript = us_rotatedescriptI(us_dragobject, descript);

	/* convert the text descriptor into a GRAPHICS style */
	switch (descript&VTPOSITION)
	{
		case VTPOSCENT:      style = TEXTCENT;      break;
		case VTPOSBOXED:     style = TEXTBOX;       break;
		case VTPOSUP:        style = TEXTBOT;       break;
		case VTPOSDOWN:      style = TEXTTOP;       break;
		case VTPOSLEFT:      style = TEXTRIGHT;     break;
		case VTPOSRIGHT:     style = TEXTLEFT;      break;
		case VTPOSUPLEFT:    style = TEXTBOTRIGHT;  break;
		case VTPOSUPRIGHT:   style = TEXTBOTLEFT;   break;
		case VTPOSDOWNLEFT:  style = TEXTTOPRIGHT;  break;
		case VTPOSDOWNRIGHT: style = TEXTTOPLEFT;   break;
	}
	if (us_dragobject->entrytype == OBJNODEINST)
	{
		makeangle(us_dragobject->entryaddr.ni->rotation, us_dragobject->entryaddr.ni->transpose,
			trans);
		style = rotatelabel(style, trans);
	}

	/* convert the text descriptor into a POLYGON */
	us_buildtexthighpoly(us_dragobject, us_dragx, us_dragy, us_dragox, us_dragoy,
		style, us_dragpoly);

	/* draw the new box */
	us_highl.col = HIGHLIT;
	(void)us_showpoly(us_dragpoly, us_dragwindow);
	us_dragshown = 1;
	return(0);
}

/************************* TEXT MOTION *************************/

/* preinitialization routine when positioning text */
void us_textmoveinit(INTBIG xc, INTBIG yc, INTBIG xw, INTBIG yw, INTBIG descript, GEOM *geom)
{
	/* save text extent information */
	us_dragx = xc;      us_dragy = yc;
	us_dragoffx = xc;   us_dragoffy = yc;
	us_dragox = xw;     us_dragoy = yw;
	us_dragdescript = descript;
	us_dragobject = geom;
}

/* initialization routine when positioning text */
void us_textmovebegin(void)
{
	/* initialize polygon */
	if (us_dragpoly == NOPOLYGON) us_dragpoly = allocpolygon(4, us_aid->cluster);
	us_dragpoly->desc = &us_highl;

	us_dragshown = 0;
	us_dragwindow = el_curwindow;
}

/* cursor advance routine when positioning text */
INTSML us_textmovedown(INTBIG x, INTBIG y)
{
	REGISTER INTBIG xcur, ycur, lambda;

	/* get the cursor value */
	if (us_setxy((INTSML)x, (INTSML)y) != 0 || us_dragwindow != el_curwindow)
	{
		(void)us_setxy((INTSML)us_lastcurx, (INTSML)us_lastcury);
		return(0);
	}
	(void)getxy(&us_lastcurx, &us_lastcury);
	gridalign(&us_lastcurx, &us_lastcury, us_alignment);
	us_setcursorpos(0, us_lastcurx, us_lastcury);

	/* if the box is already being shown, erase it */
	if (us_dragshown != 0)
	{
		/* undraw the box */
		us_highl.col = 0;
		(void)us_showpoly(us_dragpoly, us_dragwindow);
	}

	/* convert the text descriptor into a GRAPHICS style */
	us_dragpoly->count = 1;
	us_dragpoly->xv[0] = us_dragpoly->yv[0] = 0;
	adjustdisoffset(us_dragobject, us_dragpoly, us_dragdescript);

	/* clip range of text descriptor */
	lambda = figurelambda(us_dragobject);
	xcur = (us_lastcurx - us_dragx) * 4 / lambda;
	xcur = mini(xcur, 1023);   xcur = maxi(xcur, -1023);
	us_lastcurx = xcur * lambda / 4 + us_dragoffx;
	ycur = (us_lastcury - us_dragy) * 4 / lambda;
	ycur = mini(ycur, 1023);   ycur = maxi(ycur, -1023);
	us_lastcury = ycur * lambda / 4 + us_dragoffy;

	/* draw the descriptor */
	us_buildtexthighpoly(us_dragobject, us_lastcurx, us_lastcury,
		us_dragox, us_dragoy, us_dragpoly->style, us_dragpoly);

	/* draw the new box */
	us_highl.col = HIGHLIT;
	(void)us_showpoly(us_dragpoly, us_dragwindow);
	us_dragshown = 1;
	return(0);
}

/************************* ROTATE *************************/

/* preinitialization routine when rotating a node */
void us_rotateinit(NODEINST *node)
{
	us_dragnodeinst = node;
}

/* initialization routine when rotating a node */
void us_rotatebegin(void)
{
	/* initialize polygon */
	if (us_dragpoly == NOPOLYGON) us_dragpoly = allocpolygon(4, us_aid->cluster);
	us_dragpoly->desc = &us_highl;

	(void)getxy(&us_dragx, &us_dragy);
	us_dragshown = 0;
	us_dragwindow = el_curwindow;
	us_dragangle = figureangle((us_dragnodeinst->lowx+us_dragnodeinst->highx)/2,
		(us_dragnodeinst->lowy+us_dragnodeinst->highy)/2, us_dragx, us_dragy);
}

/* cursor advance routine when rotating the current node */
INTSML us_rotatedown(INTBIG x, INTBIG y)
{
	REGISTER INTBIG lx, hx, ly, hy;
	REGISTER INTSML newangle, saver, savet;
	XARRAY trans;
	INTBIG xl, yl, xh, yh;

	/* grid align the cursor value */
	if (us_setxy((INTSML)x, (INTSML)y) != 0 || us_dragwindow != el_curwindow)
	{
		(void)us_setxy((INTSML)us_lastcurx, (INTSML)us_lastcury);
		return(0);
	}
	(void)getxy(&us_lastcurx, &us_lastcury);

	/* if the box is already being shown, erase it */
	if (us_dragshown != 0)
	{
		/* undraw the box */
		us_highl.col = 0;
		(void)us_showpoly(us_dragpoly, us_dragwindow);
	}

	/* compute new angle from node center to cursor */
	newangle = figureangle((us_dragnodeinst->lowx+us_dragnodeinst->highx)/2,
		(us_dragnodeinst->lowy+us_dragnodeinst->highy)/2, us_lastcurx, us_lastcury);

	/* compute highlighted box about node */
	nodesizeoffset(us_dragnodeinst->proto, &xl, &yl, &xh, &yh);
	lx = us_dragnodeinst->lowx+xl;   hx = us_dragnodeinst->highx-xh;
	ly = us_dragnodeinst->lowy+yl;   hy = us_dragnodeinst->highy-yh;
	makerot(us_dragnodeinst, trans);
	maketruerectpoly(lx, hx, ly, hy, us_dragpoly);
	us_dragpoly->style = CLOSEDRECT;
	xformpoly(us_dragpoly, trans);

	/* rotate this box according to the cursor */
	saver = us_dragnodeinst->rotation;
	savet = us_dragnodeinst->transpose;
	us_dragnodeinst->transpose = 0;
	us_dragnodeinst->rotation = newangle-us_dragangle;
	while (us_dragnodeinst->rotation < 0) us_dragnodeinst->rotation += 3600;
	while (us_dragnodeinst->rotation > 3600) us_dragnodeinst->rotation -= 3600;
	makerot(us_dragnodeinst, trans);
	xformpoly(us_dragpoly, trans);
	us_dragnodeinst->rotation = saver;
	us_dragnodeinst->transpose = savet;

	/* draw the new box */
	us_highl.col = HIGHLIT;
	(void)us_showpoly(us_dragpoly, us_dragwindow);
	us_dragshown = 1;
	return(0);
}

/************************* SIZE *************************/

/* preinitialization routine when changing the size of a node */
void us_sizeinit(NODEINST *node)
{
	/* initialize polygon */
	if (us_dragpoly == NOPOLYGON) us_dragpoly = allocpolygon(4, us_aid->cluster);
	us_dragpoly->desc = &us_highl;
	us_dragnodeinst = node;
	us_dragshown = 0;
	us_dragwindow = el_curwindow;
	us_dragcorner = -2;
}

/* initialization routine when changing the size of a node */
void us_sizebegin(void)
{
	us_dragcorner = -1;
}

INTSML us_sizeterm(void)
{
	return(us_dragcorner);
}

/* preinitialization routine when changing the size of an arc */
void us_sizeainit(ARCINST *arc)
{
	us_dragarcinst = arc;
}

/* initialization routine when changing the size of an arc */
void us_sizeabegin(void)
{
	/* initialize polygon */
	if (us_dragpoly == NOPOLYGON) us_dragpoly = allocpolygon(4, us_aid->cluster);
	us_dragpoly->desc = &us_highl;

	us_dragshown = 0;
	us_dragwindow = el_curwindow;
}

/* cursor advance routine when stretching the current arc */
INTSML us_sizeadown(INTBIG x, INTBIG y)
{
	REGISTER INTBIG wid;

	/* grid align the cursor value */
	if (us_setxy((INTSML)x, (INTSML)y) != 0 || us_dragwindow != el_curwindow)
	{
		(void)us_setxy((INTSML)us_lastcurx, (INTSML)us_lastcury);
		return(0);
	}
	(void)getxy(&us_lastcurx, &us_lastcury);
	gridalign(&us_lastcurx, &us_lastcury, us_alignment/2);
	us_setcursorpos(0, us_lastcurx, us_lastcury);

	/* if the box is already being shown, erase it */
	if (us_dragshown != 0)
	{
		/* if it didn't actually move, go no further */
		if (us_dragx == us_lastcurx && us_dragy == us_lastcury) return(0);

		/* undraw the box */
		us_highl.col = 0;
		(void)us_showpoly(us_dragpoly, us_dragwindow);
	}

	/* compute new size of the arcinst */
	if (us_dragarcinst->end[0].xpos == us_dragarcinst->end[1].xpos)
		wid = abs(us_lastcurx - us_dragarcinst->end[0].xpos) * 2; else
			wid = abs(us_lastcury - us_dragarcinst->end[0].ypos) * 2;
	makearcpoly(us_dragarcinst->length, wid, us_dragarcinst, us_dragpoly,CLOSED);
	us_dragx = us_lastcurx;
	us_dragy = us_lastcury;

	/* draw the new box */
	us_highl.col = HIGHLIT;
	(void)us_showpoly(us_dragpoly, us_dragwindow);
	us_dragshown = 1;
	return(0);
}

/* cursor advance routine when stretching the current node about far corner */
INTSML us_sizedown(INTBIG x, INTBIG y)
{
	REGISTER INTBIG lx, hx, ly, hy, otherx, othery, dist, bestdist;
	INTBIG xl, xh, yl, yh, rx, ry;
	REGISTER INTSML corner;
	XARRAY trans;

	/* grid align the cursor value */
	if (us_setxy((INTSML)x, (INTSML)y) != 0 || us_dragwindow != el_curwindow)
	{
		(void)us_setxy((INTSML)us_lastcurx, (INTSML)us_lastcury);
		return(0);
	}
	(void)getxy(&us_lastcurx, &us_lastcury);
	gridalign(&us_lastcurx, &us_lastcury, us_alignment);
	us_setcursorpos(0, us_lastcurx, us_lastcury);

	/* if the box is already being shown, erase it */
	if (us_dragshown != 0)
	{
		/* if it didn't actually move, go no further */
		if (us_dragx == us_lastcurx && us_dragy == us_lastcury) return(0);

		/* undraw the box */
		us_highl.col = 0;
		(void)us_showpoly(us_dragpoly, us_dragwindow);
	}

	nodesizeoffset(us_dragnodeinst->proto, &xl, &yl, &xh, &yh);
	lx = us_dragnodeinst->lowx+xl;   hx = us_dragnodeinst->highx-xh;
	ly = us_dragnodeinst->lowy+yl;   hy = us_dragnodeinst->highy-yh;
	makerot(us_dragnodeinst, trans);

	/* determine which corner is fixed by finding farthest from cursor */
	corner = us_dragcorner;
	if (corner < 0)
	{
		xform(lx, ly, &rx, &ry, trans);
		bestdist = abs(rx - us_lastcurx) + abs(ry - us_lastcury);
		corner = 1;	/* lower-left */
		xform(hx, ly, &rx, &ry, trans);
		dist = abs(rx - us_lastcurx) + abs(ry - us_lastcury);
		if (dist < bestdist)
		{
			bestdist = dist;   corner = 2;	/* lower-right */
		}
		xform(lx, hy, &rx, &ry, trans);
		dist = abs(rx - us_lastcurx) + abs(ry - us_lastcury);
		if (dist < bestdist)
		{
			bestdist = dist;   corner = 3;	/* upper-left */
		}
		xform(hx, hy, &rx, &ry, trans);
		dist = abs(rx - us_lastcurx) + abs(ry - us_lastcury);
		if (dist < bestdist) corner = 4;	/* upper-right */
		if (us_dragcorner == -1) us_dragcorner = corner;
	}

	switch (corner)
	{
		case 1:		/* lower-left */
			otherx = hx;   othery = hy;   break;
		case 2:		/* lower-right */
			otherx = lx;   othery = hy;   break;
		case 3:		/* upper-left */
			otherx = hx;   othery = ly;   break;
		case 4:		/* upper-right */
			otherx = lx;   othery = ly;   break;
	}
	xform(otherx, othery, &rx, &ry, trans);
	maketruerectpoly(mini(rx, us_lastcurx), maxi(rx, us_lastcurx),
		mini(ry, us_lastcury), maxi(ry, us_lastcury), us_dragpoly);
	us_dragpoly->style = CLOSEDRECT;
	us_dragx = us_lastcurx;
	us_dragy = us_lastcury;

	/* draw the new box */
	us_highl.col = HIGHLIT;
	(void)us_showpoly(us_dragpoly, us_dragwindow);
	us_dragshown = 1;
	return(0);
}

/* cursor advance routine when stretching the current node about center */
INTSML us_sizecdown(INTBIG x, INTBIG y)
{
	REGISTER INTBIG dx, dy, cx, cy;

	/* grid align the cursor value */
	if (us_setxy((INTSML)x, (INTSML)y) != 0 || us_dragwindow != el_curwindow)
	{
		(void)us_setxy((INTSML)us_lastcurx, (INTSML)us_lastcury);
		return(0);
	}
	(void)getxy(&us_lastcurx, &us_lastcury);
	gridalign(&us_lastcurx, &us_lastcury, us_alignment/2);
	us_setcursorpos(0, us_lastcurx, us_lastcury);

	/* if the box is already being shown, erase it */
	if (us_dragshown != 0)
	{
		/* if it didn't actually move, go no further */
		if (us_dragx == us_lastcurx && us_dragy == us_lastcury) return(0);

		/* undraw the box */
		us_highl.col = 0;
		(void)us_showpoly(us_dragpoly, us_dragwindow);
	}

	/* compute new size of the nodeinst */
	cx = (us_dragnodeinst->lowx+us_dragnodeinst->highx) / 2;
	cy = (us_dragnodeinst->lowy+us_dragnodeinst->highy) / 2;
	dx = abs(cx - us_lastcurx);
	dy = abs(cy - us_lastcury);

	maketruerectpoly(cx-dx, cx+dx, cy-dy, cy+dy, us_dragpoly);
	us_dragpoly->style = CLOSEDRECT;
	us_dragx = us_lastcurx;
	us_dragy = us_lastcury;

	/* draw the new box */
	us_highl.col = HIGHLIT;
	(void)us_showpoly(us_dragpoly, us_dragwindow);
	us_dragshown = 1;
	return(0);
}

/************************* FIND AREA *************************/

/* pre-initialization routine when doing "find area-XXXX" */
void us_findinit(INTBIG sizex, INTBIG sizey)
{
	us_dragox = sizex;
	us_dragoy = sizey;
}

/* initialization routine when doing "find area-move" */
void us_findmbegin(void)
{
	INTBIG xcur, ycur;

	/* initialize polygon */
	if (us_dragpoly == NOPOLYGON) us_dragpoly = allocpolygon(4, us_aid->cluster);
	us_dragpoly->desc = &us_highl;

	us_dragshown = 0;
	(void)getxy(&xcur, &ycur);
	maketruerectpoly(xcur, xcur+us_dragox, ycur, ycur+us_dragoy, us_dragpoly);
	us_dragpoly->style = CLOSEDRECT;
	us_dragx = xcur;   us_dragy = ycur;
	us_dragwindow = el_curwindow;
}

/* initialization routine when doing "find area-size" */
void us_findsbegin(void)
{
	INTBIG xcur, ycur;

	/* initialize polygon */
	if (us_dragpoly == NOPOLYGON) us_dragpoly = allocpolygon(4, us_aid->cluster);
	us_dragpoly->desc = &us_highl;

	us_dragshown = 0;
	(void)getxy(&xcur, &ycur);
	maketruerectpoly(us_dragox, xcur, us_dragoy, ycur, us_dragpoly);
	us_dragpoly->style = CLOSEDRECT;
	us_dragx = xcur;   us_dragy = ycur;
	us_dragwindow = el_curwindow;
}

/* initialization routine when doing "find area-define" */
void us_finddbegin(void)
{
	INTBIG xcur, ycur;

	/* initialize polygon */
	if (us_dragpoly == NOPOLYGON) us_dragpoly = allocpolygon(4, us_aid->cluster);
	us_dragpoly->desc = &us_highl;

	us_dragshown = 0;
	(void)getxy(&xcur, &ycur);
	gridalign(&xcur, &ycur, us_alignment);
	us_setcursorpos(0, xcur, ycur);
	us_dragox = xcur;
	us_dragoy = ycur;
	maketruerectpoly(us_dragox, xcur, us_dragoy, ycur, us_dragpoly);
	us_dragpoly->style = CLOSEDRECT;
	us_dragx = xcur;   us_dragy = ycur;
	us_dragwindow = el_curwindow;
}

/* posttermination routine when doing "find area-define" */
void us_finddterm(INTBIG *x, INTBIG *y)
{
	*x = us_dragox;
	*y = us_dragoy;
}

/************************* FIND INTERACTIVE *************************/

void us_findiinit(INTSML findport, INTSML extrainfo, INTSML findangle, INTSML stayonhighlighted,
	INTSML findstill, INTSML findnobox)
{
	us_dragfport = findport;
	us_dragextra = (extrainfo != 0 ? 1 : 0);
	us_dragangle = findangle;
	us_dragstill = findstill;
	us_dragnobox = findnobox;
	us_dragpoint = -1;
	us_dragstayonhigh = stayonhighlighted;
}

void us_findcibegin(void)
{
	us_finddoibegin(1);
}

void us_findibegin(void)
{
	us_finddoibegin(0);
}

void us_finddoibegin(INTSML compl)
{
	REGISTER VARIABLE *highvar;
	REGISTER NODEINST *ni;
	REGISTER ARCINST *ai;
	REGISTER PORTPROTO *pp;
	REGISTER INTSML len;
	REGISTER char *str;
	HIGHLIGHT newhigh, oldhigh;
	INTSML i, tsx, tsy, j;
	INTBIG xc, yc, xw, yw, xcur, ycur, dist, bestdist;

	/* initialize polygon */
	if (us_dragpoly == NOPOLYGON) us_dragpoly = allocpolygon(4, us_aid->cluster);
	us_dragpoly->desc = &us_highl;
	us_dragstate = -1;
	us_dragshown = 0;
	us_dragwindow = el_curwindow;

	/* get cursor coordinates */
	if (getxy(&xcur, &ycur)) return;
	us_dragx = xcur;   us_dragy = ycur;
	gridalign(&us_dragx, &us_dragy, us_alignment);
	us_setcursorpos(0, us_dragx, us_dragy);
	us_dragox = us_dragx;   us_dragoy = us_dragy;
	us_dragnodeproto = getcurfacet();

	/* see what is highlighted */
	highvar = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_highlighted);
	if (highvar != NOVARIABLE)
	{
		len = getlength(highvar);
		(void)us_makehighlight(((char **)highvar->addr)[0], &newhigh);
	} else
	{
		len = 0;
		newhigh.status = 0;
		newhigh.fromgeom = NOGEOM;
		newhigh.fromport = NOPORTPROTO;
		newhigh.fromvar = NOVARIABLE;
		newhigh.frompoint = 0;
	}

	/* see what is under the cursor */
	if (us_dragstayonhigh != 0 && highvar != NOVARIABLE)
	{
		/* see if the cursor is over something highlighted */
		for(i=0; i<len; i++)
		{
			(void)us_makehighlight(((char **)highvar->addr)[i], &newhigh);
			if (us_cursoroverhigh(&newhigh, xcur, ycur, el_curwindow) != 0) break;
		}
		if (i < len)
		{
			/* re-find the closest port when one node is selected */
			if (len == 1 && (newhigh.status&HIGHFROM) != 0 &&
				newhigh.fromgeom->entrytype == OBJNODEINST)
			{
				ni = newhigh.fromgeom->entryaddr.ni;
				newhigh.fromport = NOPORTPROTO;
				for(pp = ni->proto->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
				{
					shapeportpoly(ni, pp, us_dragpoly, 0);
					us_dragpoly->desc = &us_highl;

					/* get distance of desired point to polygon */
					dist = polydistance(us_dragpoly, xcur, ycur);
					if (dist < 0)
					{
						newhigh.fromport = pp;
						break;
					}
					if (newhigh.fromport == NOPORTPROTO) bestdist = dist;
					if (dist > bestdist) continue;
					bestdist = dist;   newhigh.fromport = pp;
				}
			}
		} else us_findobject(xcur, ycur, el_curwindow, &newhigh, 0, 0, us_dragfport, 1);
	} else us_findobject(xcur, ycur, el_curwindow, &newhigh, 0, 0, us_dragfport, 1);
	if (newhigh.fromgeom == NOGEOM)
	{
		/* nothing under cursor: select an area */
		if (compl != 0) us_dragstate = 2; else us_dragstate = 1;
		maketruerectpoly(us_dragx, us_dragx, us_dragy, us_dragy, us_dragpoly);
		us_dragpoly->style = CLOSEDRECT;
		return;
	}

	/* clear port info if not requested or multiple highlights */
	if (us_dragfport == 0) newhigh.fromport = NOPORTPROTO;

	/* see if object under cursor is already highlighted */
	for(i=0; i<len; i++)
	{
		if (us_makehighlight(((char **)highvar->addr)[i], &oldhigh) != 0) continue;
		if (oldhigh.facet != newhigh.facet) continue;
		if ((oldhigh.status&HIGHTYPE) != (newhigh.status&HIGHTYPE)) continue;
		if (oldhigh.fromgeom != newhigh.fromgeom) continue;
		break;
	}

	/* is this a normal command or SHIFTed? */
	if (compl == 0)
	{
		/* normal find/move */
		if (i >= len || us_dragpoint >= 0)
		{
			/* object is not highlighted: select only it and move */
			us_setfind(&newhigh, (INTSML)((us_dragpoint < 0 ? 0 : 1)),
				(INTSML)((us_dragextra != 0 ? HIGHEXTRA : 0)), 0, us_dragnobox);
		} else
		{
			/* object is already highlighted: rehighlight and move */
			newhigh.facet = geomparent(newhigh.fromgeom);
			(void)us_delhighlight(&newhigh);
			(void)us_addhighlight(&newhigh);
		}
	} else
	{
		/* SHIFT find/move */
		if (i >= len)
		{
			/* object not highlighted: add it and move */
			/* newhigh.fromport = NOPORTPROTO; */
			newhigh.facet = geomparent(newhigh.fromgeom);
			us_setfind(&newhigh, 0, (INTSML)((us_dragextra != 0 ? HIGHEXTRA : 0)), 1, us_dragnobox);
		} else
		{
			/* object highlighted: unhighlight it and quit */
			(void)us_delhighlight(&newhigh);
			return;
		}
	}

	/* can only drag if in "Mac" mode */
	if (us_dragstayonhigh == 0) return;

	/* do no motion if stillness requested */
	if (us_dragstill != 0)
	{
		us_dragstate = -1;
		return;
	}

	/* get the objects to be moved */
	highvar = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_highlighted);
	if (highvar == NOVARIABLE) return;

	/* see if a text object is highlighted */
	if (getlength(highvar) == 1)
	{
		(void)us_makehighlight(((char **)highvar->addr)[0], &us_draghigh);
		if ((us_draghigh.status&HIGHTYPE) == HIGHTEXT)
		{
			/* get descriptor and text string */
			us_dragnodeinst = NONODEINST;
			if (us_draghigh.fromvar != NOVARIABLE)
			{
				if (us_draghigh.fromgeom->entrytype == OBJNODEINST)
					us_dragnodeinst = us_draghigh.fromgeom->entryaddr.ni;
				us_dragdescript = us_draghigh.fromvar->textdescript;
				str = describevariable(us_draghigh.fromvar, -1, -1);
				if ((us_draghigh.fromvar->type&VISARRAY) == 0)
				{
					j = (us_draghigh.fromvar->type&VLENGTH) >> VLENGTHSH;
					if (j < strlen(str)) str += j;
				}
			} else if (us_draghigh.fromport != NOPORTPROTO)
			{
				us_dragnodeinst = us_draghigh.fromgeom->entryaddr.ni;
				us_dragdescript = us_draghigh.fromport->textdescript;
				str = us_draghigh.fromport->protoname;
			} else if (us_draghigh.fromgeom->entrytype == OBJNODEINST)
			{
				us_dragnodeinst = us_draghigh.fromgeom->entryaddr.ni;
				us_dragdescript = us_dragnodeinst->textdescript;
				str = describenodeproto(us_dragnodeinst->proto);
			}

			/* determine number of lines of text and text size */
			len = 1;
			if (us_draghigh.fromvar != NOVARIABLE && (us_draghigh.fromvar->type&VISARRAY) != 0)
				len = getlength(us_draghigh.fromvar);
			if (len > 1)
			{
				xw = us_draghigh.fromgeom->highx - us_draghigh.fromgeom->lowx;
				yw = us_draghigh.fromgeom->highy - us_draghigh.fromgeom->lowy;
			} else
			{
				us_settextsize(el_curwindow, truefontsize((INTSML)((us_dragdescript&VTSIZE)>>VTSIZESH),
					el_curwindow, el_curtech));
				us_textsize(el_curwindow, str, &tsx, &tsy);
				xw = muldiv(tsx, el_curwindow->screenhx-el_curwindow->screenlx,
					el_curwindow->usehx-el_curwindow->uselx);
				yw = muldiv(tsy, el_curwindow->screenhy-el_curwindow->screenly,
					el_curwindow->usehy-el_curwindow->usely);
			}

			/* get offset */
			us_dragobject = us_draghigh.fromgeom;
			us_gethightextcenter(&us_draghigh, &xc, &yc, &j);
			us_dragoffx = xc;  us_dragoffy = yc;
			us_dragox = xw;    us_dragoy = yw;
			us_dragstate = 3;
			return;
		}
	}

	us_dragobjectlist = us_gethighlighted(OBJNODEINST | OBJARCINST);
	if (us_dragobjectlist[0] == NOGEOM) return;
	for(i=0; us_dragobjectlist[i] != NOGEOM; i++)
		if (us_dragnodeproto != geomparent(us_dragobjectlist[i])) return;

	/* mark all nodes (including those touched by highlighted arcs) */
	for(ni = us_dragnodeproto->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
		ni->temp1 = 0;
	for(i=0; us_dragobjectlist[i] != NOGEOM; i++)
		if (us_dragobjectlist[i]->entrytype == OBJARCINST)
	{
		ai = us_dragobjectlist[i]->entryaddr.ai;
		ai->end[0].nodeinst->temp1 = ai->end[1].nodeinst->temp1 = 1;
	}
	for(i=0; us_dragobjectlist[i] != NOGEOM; i++)
		if (us_dragobjectlist[i]->entrytype == OBJNODEINST)
			us_dragobjectlist[i]->entryaddr.ni->temp1 = 1;

	/* count the number of nodes, excluding locked ones */
	us_dragnodetotal = 0;
	for(ni = us_dragnodeproto->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		if (ni->temp1 == 0) continue;
		if (us_islocked(ni->proto) != 0)
		{
			ni->temp1 = 0;
			continue;
		}
		us_dragnodetotal++;
	}
	if (us_dragnodetotal == 0) us_cantdrag = 1; else
		us_cantdrag = 0;

	/* remove from list all arcs touching locked nodes */
	for(i=j=0; us_dragobjectlist[i] != NOGEOM; i++)
	{
		if (us_dragobjectlist[i]->entrytype == OBJARCINST)
		{
			ai = us_dragobjectlist[i]->entryaddr.ai;
			if (us_islocked(ai->end[0].nodeinst->proto) != 0 ||
				us_islocked(ai->end[1].nodeinst->proto) != 0) continue;
		}
		us_dragobjectlist[j++] = us_dragobjectlist[i];
	}
	us_dragobjectlist[j] = NOGEOM;

	/* build a list that includes all nodes touching selected arcs */
	if (us_dragnodetotal != 0)
	{
		us_dragnodelist = (NODEINST **)emalloc((us_dragnodetotal * (sizeof (NODEINST *))),
			el_tempcluster);
		if (us_dragnodelist == 0) return;
		for(i=0, ni = us_dragnodeproto->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
		{
			if (ni->temp1 == 0) continue;
			us_dragnodelist[i++] = ni;
		}
	}

	/* save and turn off all highlighting */
	us_pushhighlight();
	us_clearhighlightcount();

	/* setup for moving */
	us_multidragbegin();
	us_dragstate = 0;
}

INTSML us_findidown(INTBIG x, INTBIG y)
{
	switch (us_dragstate)
	{
		case 0: return(us_multidragdown(x, y));
		case 1:
		case 2: return(us_stretchdown(x, y));
		case 3: return(us_textmovedown(x, y));
	}
	return(1);
}

void us_findiup(void)
{
	INTBIG xcur, ycur, search, lambda, dx, dy;
	XARRAY trans;
	REGISTER INTSML i, j, dispstyle;
	REGISTER INTBIG len, k, slx, shx, sly, shy, addtotal;
	REGISTER GEOM *geom;
	REGISTER NODEINST *ni;
	REGISTER ARCINST *ai;
	REGISTER char **highlist;
	HIGHLIGHT newhigh, oldhigh;
	REGISTER VARIABLE *highvar, *var;
	static POLYGON *poly = NOPOLYGON;

	if (us_dragstate == 0)
	{
		us_multidragup();
		if (us_cantdrag == 0 && el_pleasestop == 0)
		{
			(void)getxy(&xcur, &ycur);
			gridalign(&xcur, &ycur, us_alignment);
			us_setcursorpos(0, xcur, ycur);
			us_getslide(us_dragangle, us_dragox, us_dragoy, xcur, ycur, &dx, &dy);
			if (dx != us_dragox || dy != us_dragoy)
				us_manymove(us_dragobjectlist, us_dragnodelist, (INTSML)us_dragnodetotal,
					dx-us_dragox, dy-us_dragoy);
		}
		if (us_dragnodetotal > 0) efree((char *)us_dragnodelist);
		(void)us_pophighlight(1);
	} else if (us_dragstate == 1)
	{
		/* get area of drag */
		us_invertdragup();
		slx = mini(us_dragox, us_dragx);   shx = maxi(us_dragox, us_dragx);
		sly = mini(us_dragoy, us_dragy);   shy = maxi(us_dragoy, us_dragy);
		us_clearhighlightcount();

		/* see how many objects are being highlighted */
		addtotal = 0;
		search = initsearch(slx, shx, sly, shy, us_dragnodeproto);
		while ((geom = nextobject(search)) != NOGEOM)
		{
			if ((addtotal%50) == 49)
				if (stopping("Selection")) { termsearch(search);   return; }
			addtotal++;
		}
		if (addtotal == 0) return;

		/* build a list of objects to highlight */
		highlist = (char **)emalloc(addtotal * (sizeof (char *)), el_tempcluster);
		if (highlist == 0) return;
		addtotal = 0;
		search = initsearch(slx, shx, sly, shy, us_dragnodeproto);
		while ((geom = nextobject(search)) != NOGEOM)
		{
			if ((addtotal%50) == 49)
				if (stopping("Selection")) { termsearch(search);   break; }

			/* do not include facet-center if selecting ports */
			if (us_dragfport != 0 && geom->entrytype == OBJNODEINST &&
				geom->entryaddr.ni->proto == gen_facetcenterprim) continue;

			/* do not include "edge select" primitives that are outside the area */
			if (geom->entrytype == OBJNODEINST && geom->entryaddr.ni->proto->index != 0 &&
				(geom->entryaddr.ni->proto->userbits&NEDGESELECT) != 0)
			{
				ni = geom->entryaddr.ni;
				i = nodepolys(ni);
				if (poly == NOPOLYGON) poly = allocpolygon(4, us_aid->cluster);
				makerot(ni, trans);
				dispstyle = us_getdispstyle(el_curwindow);
				for(j=0; j<i; j++)
				{
					shapenodepoly(ni, j, poly);
					if ((poly->desc->style[dispstyle]&INVISIBLE) == 0)
					{
						xformpoly(poly, trans);
						if (polyinrect(poly, slx, shx, sly, shy) == 0) break;
					}
				}
				if (j < i) continue;
			} else if (geom->entrytype == OBJARCINST &&
				(geom->entryaddr.ai->proto->userbits&AEDGESELECT) != 0)
			{
				ai = geom->entryaddr.ai;
				i = arcpolys(ai);
				if (poly == NOPOLYGON) poly = allocpolygon(4, us_aid->cluster);
				dispstyle = us_getdispstyle(el_curwindow);
				for(j=0; j<i; j++)
				{
					shapearcpoly(ai, j, poly);
					if ((poly->desc->style[dispstyle]&INVISIBLE) == 0)
					{
						if (polyinrect(poly, slx, shx, sly, shy) == 0) break;
					}
				}
				if (j < i) continue;
			}

			newhigh.status = HIGHFROM;
			if (us_dragnobox != 0) newhigh.status |= HIGHNOBOX;
			newhigh.facet = us_dragnodeproto;
			newhigh.fromgeom = geom;
			newhigh.fromport = NOPORTPROTO;
			newhigh.fromvar = NOVARIABLE;
			newhigh.frompoint = 0;

			/* add snapping information if requested */
			if ((us_state&SNAPMODE) != SNAPMODENONE)
			{
				(void)getxy(&xcur, &ycur);
				us_selectsnap(&newhigh, xcur, ycur);
			}

			/* select the first port if ports are requested and node is artwork */
			if (us_dragfport != 0 && geom->entrytype == OBJNODEINST &&
				geom->entryaddr.ni->proto->index != 0 &&
					geom->entryaddr.ni->proto->tech == art_tech)
						newhigh.fromport = geom->entryaddr.ni->proto->firstportproto;

			/* special case when an invisible pin is selected */
			if (geom->entrytype == OBJNODEINST &&
				geom->entryaddr.ni->proto == gen_invispinprim)
			{
				/* if the pin has a message, highlight the text */
				var = getvalkey((INTBIG)geom->entryaddr.ni, VNODEINST, -1, art_messagekey);
				if (var != NOVARIABLE)
				{
					newhigh.fromvar = var;
					newhigh.status = HIGHTEXT;
				}
			}
			(void)allocstring(&highlist[addtotal], us_makehighlightstring(&newhigh), el_tempcluster);
			addtotal++;
		}
		if (addtotal > 0)
			(void)setvalkey((INTBIG)us_aid, VAID, us_highlighted, (INTBIG)highlist,
				VSTRING|VISARRAY|(addtotal<<VLENGTHSH)|VDONTSAVE);
		for(k=0; k<addtotal; k++) efree(highlist[k]);
		efree((char *)highlist);
	} else if (us_dragstate == 2)
	{
		/* special case when complementing highlight */
		us_invertdragup();
		slx = mini(us_dragox, us_dragx);   shx = maxi(us_dragox, us_dragx);
		sly = mini(us_dragoy, us_dragy);   shy = maxi(us_dragoy, us_dragy);
		highvar = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_highlighted);
		if (highvar != NOVARIABLE) len = getlength(highvar); else len = 0;

		search = initsearch(slx, shx, sly, shy, us_dragnodeproto);
		while ((geom = nextobject(search)) != NOGEOM)
		{
			if (stopping("Selection")) { termsearch(search);   break; }

			/* do not include facet-center if selecting ports */
			if (us_dragfport != 0 && geom->entrytype == OBJNODEINST &&
				geom->entryaddr.ni->proto == gen_facetcenterprim) continue;

			/* do not include "edge select" primitives that are outside the area */
			if (geom->entrytype == OBJNODEINST && geom->entryaddr.ni->proto->index != 0 &&
				(geom->entryaddr.ni->proto->userbits&NEDGESELECT) != 0)
			{
				ni = geom->entryaddr.ni;
				i = nodepolys(ni);
				if (poly == NOPOLYGON) poly = allocpolygon(4, us_aid->cluster);
				makerot(ni, trans);
				dispstyle = us_getdispstyle(el_curwindow);
				for(j=0; j<i; j++)
				{
					shapenodepoly(ni, j, poly);
					if ((poly->desc->style[dispstyle]&INVISIBLE) == 0)
					{
						xformpoly(poly, trans);
						if (polyinrect(poly, slx, shx, sly, shy) == 0) break;
					}
				}
				if (j < i) continue;
			} else if (geom->entrytype == OBJARCINST &&
				(geom->entryaddr.ai->proto->userbits&AEDGESELECT) != 0)
			{
				ai = geom->entryaddr.ai;
				i = arcpolys(ai);
				if (poly == NOPOLYGON) poly = allocpolygon(4, us_aid->cluster);
				dispstyle = us_getdispstyle(el_curwindow);
				for(j=0; j<i; j++)
				{
					shapearcpoly(ai, j, poly);
					if ((poly->desc->style[dispstyle]&INVISIBLE) == 0)
					{
						if (polyinrect(poly, slx, shx, sly, shy) == 0) break;
					}
				}
				if (j < i) continue;
			}

			newhigh.status = HIGHFROM;
			if (us_dragnobox != 0) newhigh.status |= HIGHNOBOX;
			newhigh.facet = us_dragnodeproto;
			newhigh.fromgeom = geom;
			newhigh.fromport = NOPORTPROTO;
			newhigh.fromvar = NOVARIABLE;
			newhigh.frompoint = 0;

			/* add snapping information if requested */
			if ((us_state&SNAPMODE) != SNAPMODENONE)
			{
				(void)getxy(&xcur, &ycur);
				us_selectsnap(&newhigh, xcur, ycur);
			}

			/* select the first port if ports are requested and node is artwork */
			if (us_dragfport != 0 && geom->entrytype == OBJNODEINST &&
				geom->entryaddr.ni->proto->index != 0 &&
					geom->entryaddr.ni->proto->tech == art_tech)
						newhigh.fromport = geom->entryaddr.ni->proto->firstportproto;

			/* special case when an invisible pin is selected */
			if (geom->entrytype == OBJNODEINST && geom->entryaddr.ni->proto == gen_invispinprim)
			{
				/* if the pin has a message, highlight the text */
				var = getvalkey((INTBIG)geom->entryaddr.ni, VNODEINST, -1, art_messagekey);
				if (var != NOVARIABLE)
				{
					newhigh.fromvar = var;
					newhigh.status = HIGHTEXT;
				}
			}

			/* see if this object is already highlighted */
			for(k=0; k<len; k++)
			{
				if (us_makehighlight(((char **)highvar->addr)[k], &oldhigh) != 0) continue;
				if (oldhigh.facet != newhigh.facet) continue;
				if ((oldhigh.status&HIGHTYPE) != HIGHFROM) continue;
				if (oldhigh.fromgeom != newhigh.fromgeom) continue;
				break;
			}
			if (k < len) (void)us_delhighlight(&newhigh); else
				(void)us_addhighlight(&newhigh);

			/* recache what is highlighted */
			highvar = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_highlighted);
			if (highvar != NOVARIABLE) len = getlength(highvar); else len = 0;
		}
	} else if (us_dragstate == 3)
	{
		us_dragup();
		if (el_pleasestop != 0) return;
		(void)getxy(&xcur, &ycur);
		gridalign(&xcur, &ycur, us_alignment);
		us_setcursorpos(0, xcur, ycur);
		dx = xcur-us_dragx;    dy = ycur-us_dragy;
		us_clearhighlightcount();

		/* move entire node if it is an "invisible pin" */
		if (us_dragnodeinst != NONODEINST &&
			us_dragnodeinst->proto == gen_invispinprim)
		{
			startobjectchange((INTBIG)us_dragnodeinst, VNODEINST);
			modifynodeinst(us_dragnodeinst, dx, dy, dx, dy, 0, 0);
			endobjectchange((INTBIG)us_dragnodeinst, VNODEINST);
			(void)us_addhighlight(&us_draghigh);
			return;
		}

		/* undraw the text */
		startobjectchange((INTBIG)us_draghigh.fromgeom->entryaddr.blind,
			us_draghigh.fromgeom->entrytype == OBJNODEINST ? VNODEINST : VARCINST);

		/* set the new descriptor on the text */
		if (us_dragnodeinst != NONODEINST)
		{
			if (us_dragnodeinst->transpose != 0)
				makeangle(us_dragnodeinst->rotation, us_dragnodeinst->transpose, trans); else
					makeangle((INTSML)((3600-us_dragnodeinst->rotation)%3600), 0, trans);
			xform(dx, dy, &dx, &dy, trans);
		}
		lambda = figurelambda(us_draghigh.fromgeom);
		dx = dx*4/lambda;   dy = dy*4/lambda;
		if ((us_dragdescript&VTXOFFNEG) != 0) dx -= (us_dragdescript&VTXOFF) >> VTXOFFSH; else
			dx += (us_dragdescript&VTXOFF) >> VTXOFFSH;
		if ((us_dragdescript&VTYOFFNEG) != 0) dy -= (us_dragdescript&VTYOFF) >> VTYOFFSH; else
			dy += (us_dragdescript&VTYOFF) >> VTYOFFSH;
		us_dragdescript = us_setdescriptoffset(us_dragdescript, dx, dy);
		us_modifytextdescript(&us_draghigh, us_dragdescript);

		/* redisplay the text */
		endobjectchange((INTBIG)us_draghigh.fromgeom->entryaddr.blind,
			us_draghigh.fromgeom->entrytype == OBJNODEINST ? VNODEINST : VARCINST);
		(void)us_addhighlight(&us_draghigh);

		/* modify all higher-level nodes if port moved */
		if (us_draghigh.fromvar == NOVARIABLE && us_draghigh.fromport != NOPORTPROTO)
		{
			for(ni = us_draghigh.fromport->parent->firstinst; ni != NONODEINST; ni = ni->nextinst)
			{
				if ((ni->userbits&NEXPAND) != 0 &&
					(us_draghigh.fromport->userbits&PORTDRAWN) == 0) continue;
				startobjectchange((INTBIG)ni, VNODEINST);
				endobjectchange((INTBIG)ni, VNODEINST);
			}
		}
	}
}

/************************* CREATE *************************/

/* pre-initialization routine when inserting type "np" in arc "ai" */
void us_createinsinit(ARCINST *ai, NODEPROTO *np)
{
	us_dragarcinst = ai;
	us_dragnodeproto = np;
}

/* pre-initialization routine when creating an object */
void us_createinit(INTBIG cornerx, INTBIG cornery, NODEPROTO *np, INTSML angle, INTSML join)
{
	us_dragox = cornerx;
	us_dragoy = cornery;
	us_dragnodeproto = np;
	us_dragangle = angle;
	us_dragjoinfactor = join;
}

/* initialization routine when creating an object */
void us_createbegin(void)
{
	XARRAY trans;
	INTBIG cx, cy, lx, ly, hx, hy, xcur, ycur;
	REGISTER NODEINST *ni;

	/* initialize polygon */
	if (us_dragpoly == NOPOLYGON) us_dragpoly = allocpolygon(4, us_aid->cluster);
	us_dragpoly->desc = &us_highl;

	(void)getxy(&xcur, &ycur);
	us_dragshown = 0;
	ni = dummynode();
	ni->proto = us_dragnodeproto;
	ni->rotation = us_dragangle%3600;
	ni->transpose = us_dragangle/3600;
	ni->lowx = xcur;
	ni->highx = xcur + us_dragnodeproto->highx - us_dragnodeproto->lowx;
	ni->lowy = ycur;
	ni->highy = ycur + us_dragnodeproto->highy - us_dragnodeproto->lowy;
	makerot(ni, trans);
	nodesizeoffset(us_dragnodeproto, &lx, &ly, &hx, &hy);
	maketruerectpoly(ni->lowx+lx, ni->highx-hx, ni->lowy+ly, ni->highy-hy, us_dragpoly);
	us_dragpoly->style = CLOSEDRECT;
	xformpoly(us_dragpoly, trans);
	corneroffset(NONODEINST, us_dragnodeproto, (INTSML)(us_dragangle%3600),
		(INTSML)(us_dragangle/3600), &cx, &cy);
	us_dragx = xcur + cx;   us_dragy = ycur + cy;
	us_dragwindow = el_curwindow;
}

/* initialization routine when creating along an angle */
void us_createabegin(void)
{
	XARRAY trans;
	INTBIG cx, cy, lx, ly, hx, hy, xcur, ycur;
	REGISTER NODEINST *ni;

	/* initialize polygon */
	if (us_dragpoly == NOPOLYGON) us_dragpoly = allocpolygon(4, us_aid->cluster);
	us_dragpoly->desc = &us_highl;

	(void)getxy(&xcur, &ycur);
	us_dragshown = 0;
	ni = dummynode();
	ni->proto = us_dragnodeproto;
	ni->rotation = 0;
	ni->transpose = 0;
	ni->lowx = xcur;
	ni->highx = xcur + us_dragnodeproto->highx - us_dragnodeproto->lowx;
	ni->lowy = ycur;
	ni->highy = ycur + us_dragnodeproto->highy - us_dragnodeproto->lowy;
	makerot(ni, trans);
	nodesizeoffset(us_dragnodeproto, &lx, &ly, &hx, &hy);
	maketruerectpoly(ni->lowx+lx, ni->highx-hx, ni->lowy+ly, ni->highy-hy, us_dragpoly);
	us_dragpoly->style = CLOSEDRECT;
	xformpoly(us_dragpoly, trans);
	corneroffset(NONODEINST, us_dragnodeproto, 0, 0, &cx, &cy);
	us_dragoffx = cx;
	us_dragoffy = cy;
	us_dragx = xcur;   us_dragy = ycur;
	us_dragwindow = el_curwindow;
	us_dragobject = NOGEOM;
}

/* cursor advance routine when creating along an angle */
INTSML us_createadown(INTBIG x, INTBIG y)
{
	REGISTER INTBIG dx, dy;
	REGISTER INTSML i;
	INTBIG rdx, rdy;
	REGISTER GEOM *foundgeom;

	/* grid align the cursor value */
	if (us_setxy((INTSML)x, (INTSML)y) != 0 || us_dragwindow != el_curwindow)
	{
		(void)us_setxy((INTSML)us_lastcurx, (INTSML)us_lastcury);
		return(0);
	}
	(void)getxy(&us_lastcurx, &us_lastcury);
	gridalign(&us_lastcurx, &us_lastcury, us_alignment);
	us_setcursorpos(0, us_lastcurx, us_lastcury);

	us_getslide(us_dragangle, us_dragox, us_dragoy, us_lastcurx, us_lastcury, &rdx, &rdy);
	dx = rdx - us_dragoffx;
	dy = rdy - us_dragoffy;

	/* find the object under the cursor */
	if (us_dragjoinfactor == 0) foundgeom = NOGEOM; else
		foundgeom = us_getclosest(dx+(us_dragnodeproto->highx-us_dragnodeproto->lowx)/2,
			dy+(us_dragnodeproto->highy-us_dragnodeproto->lowy)/2, us_dragwindow->curnodeproto);

	/* if the box is already being shown, erase it */
	if (us_dragshown != 0)
	{
		/* if it didn't actually move, go no further */
		if (us_dragx == dx && us_dragy == dy) return(0);

		/* undraw the box */
		us_highl.col = 0;
		(void)us_showpoly(us_dragpoly, us_dragwindow);
	}

	/* if the selected other object is being shown, erase it */
	if (us_dragobject != NOGEOM)
	{
		if (us_dragobject != foundgeom)
			us_highlighteverywhere(us_dragobject, NOPORTPROTO, 0,0, ALLOFF, 0);
	}

	/* advance the box */
	for(i = 0; i < us_dragpoly->count; i++)
	{
		us_dragpoly->xv[i] += dx - us_dragx;
		us_dragpoly->yv[i] += dy - us_dragy;
	}
	us_dragx = dx;
	us_dragy = dy;

	/* draw the new box */
	us_highl.col = HIGHLIT;
	(void)us_showpoly(us_dragpoly, us_dragwindow);
	us_dragshown = 1;
	if (foundgeom != NOGEOM && foundgeom != us_dragobject)
		us_highlighteverywhere(foundgeom, NOPORTPROTO, 0, 0, HIGHLIT, 0);
	us_dragobject = foundgeom;
	return(0);
}

/* cursor advance routine when creating inside an arc */
INTSML us_createinsdown(INTBIG x, INTBIG y)
{
	REGISTER INTSML i;
	INTBIG dx, dy;

	/* grid align the cursor value */
	if (us_setxy((INTSML)x, (INTSML)y) != 0 || us_dragwindow != el_curwindow)
	{
		(void)us_setxy((INTSML)us_lastcurx, (INTSML)us_lastcury);
		return(0);
	}
	(void)getxy(&us_lastcurx, &us_lastcury);
	gridalign(&us_lastcurx, &us_lastcury, us_alignment);
	us_setcursorpos(0, us_lastcurx, us_lastcury);

	/* find closest point along arc */
	dx = us_lastcurx;   dy = us_lastcury;
	(void)closestpointtosegment(us_dragarcinst->end[0].xpos, us_dragarcinst->end[0].ypos,
		us_dragarcinst->end[1].xpos, us_dragarcinst->end[1].ypos, &dx, &dy);
	dx -= (us_dragnodeproto->highx - us_dragnodeproto->lowx)/2;
	dy -= (us_dragnodeproto->highy - us_dragnodeproto->lowy)/2;

	/* if the box is already being shown, erase it */
	if (us_dragshown != 0)
	{
		/* if it didn't actually move, go no further */
		if (us_dragx == dx && us_dragy == dy) return(0);

		/* undraw the box */
		us_highl.col = 0;
		(void)us_showpoly(us_dragpoly, us_dragwindow);
	}

	/* advance the box */
	for(i = 0; i < us_dragpoly->count; i++)
	{
		us_dragpoly->xv[i] += dx - us_dragx;
		us_dragpoly->yv[i] += dy - us_dragy;
	}
	us_dragx = dx;
	us_dragy = dy;

	/* draw the new box */
	us_highl.col = HIGHLIT;
	(void)us_showpoly(us_dragpoly, us_dragwindow);
	us_dragshown = 1;
	return(0);
}

/* termination routine when creating along an angle */
void us_createaup(void)
{
	if (us_dragshown != 0)
	{
		/* undraw the box */
		us_highl.col = 0;
		(void)us_showpoly(us_dragpoly, us_dragwindow);
	}
	if (us_dragobject != NOGEOM)
		us_highlighteverywhere(us_dragobject, NOPORTPROTO, 0, 0, ALLOFF, 0);
}

/************************* DUPLICATE and multiple MOVE *************************/

/* preinitialization routine when duplicating objects */
void us_multidraginit(INTBIG xc, INTBIG yc, GEOM **geomlist, NODEINST **nodelist,
	INTSML total, INTSML angle, INTSML extra)
{
	us_dragox = xc;
	us_dragoy = yc;
	us_dragobjectlist = geomlist;
	us_dragnodelist = nodelist;
	us_dragnodetotal = total;
	us_dragangle = angle;
	us_dragextra = extra;
}

/* initialization routine when duplicating objects */
void us_multidragbegin(void)
{
	REGISTER INTSML i;
	REGISTER PORTARCINST *pi;
	REGISTER ARCINST *ai;

	/* initialize polygon */
	if (us_dragpoly == NOPOLYGON) us_dragpoly = allocpolygon(4, us_aid->cluster);
	us_dragpoly->desc = &us_highl;

	(void)getxy(&us_dragx, &us_dragy);
	gridalign(&us_dragx, &us_dragy, us_alignment);
	us_setcursorpos(0, us_dragx, us_dragy);
	us_dragshown = 0;
	us_dragwindow = el_curwindow;

	/* preprocess all arcs that stretch to moved nodes if in verbose mode */
	if (us_dragextra != 0)
	{
		/* reset clock (temp2) and clear display nature bits (temp1) on arcs */
		for(i=0; i<us_dragnodetotal; i++)
			for(pi = us_dragnodelist[i]->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
		{
			ai = pi->conarcinst;
			ai->temp1 = ai->temp2 = 0;
		}

		/* mark number of ends that move on each arc */
		for(i=0; i<us_dragnodetotal; i++)
			for(pi = us_dragnodelist[i]->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
				pi->conarcinst->temp1++;

		/* do not move arcs that are already in move list */
		for(i=0; us_dragobjectlist[i] != NOGEOM; i++)
		{
			if (us_dragobjectlist[i]->entrytype != OBJARCINST) continue;
			ai = us_dragobjectlist[i]->entryaddr.ai;
			ai->temp1 = 0;
		}
	}
}

/* cursor advance routine when duplicating objects */
INTSML us_multidragdown(INTBIG x, INTBIG y)
{
	INTBIG nx, ny;

	/* grid align the cursor value */
	if (us_setxy((INTSML)x, (INTSML)y) != 0 || us_dragwindow != el_curwindow)
	{
		(void)us_setxy((INTSML)us_lastcurx, (INTSML)us_lastcury);
		return(0);
	}
	(void)getxy(&us_lastcurx, &us_lastcury);
	gridalign(&us_lastcurx, &us_lastcury, us_alignment);
	us_setcursorpos(0, us_lastcurx, us_lastcury);

	us_getslide(us_dragangle, us_dragox, us_dragoy, us_lastcurx, us_lastcury, &nx, &ny);
	us_lastcurx = nx;   us_lastcury = ny;

	/* warn if moving and can't */
	if (us_cantdrag != 0)
	{
		if (us_dragx != us_lastcurx || us_dragy != us_lastcury)
		{
			if (us_cantdrag == 1)
				us_abortcommand("Sorry, changes are currently disallowed");
			us_cantdrag++;
			return(0);
		}
	}

	/* if the highlighting is already being shown, erase it */
	if (us_dragshown != 0)
	{
		/* if it didn't actually move, go no further */
		if (us_dragx == us_lastcurx && us_dragy == us_lastcury) return(0);

		/* undraw highlighting */
		us_multidragdraw(0, us_dragx-us_dragox, us_dragy-us_dragoy);
	}

	us_dragx = us_lastcurx;
	us_dragy = us_lastcury;
	us_multidragdraw(HIGHLIT, us_dragx-us_dragox, us_dragy-us_dragoy);
	us_dragshown = 1;
	return(0);
}

void us_multidragup(void)
{
	if (us_dragshown != 0)
		us_multidragdraw(0, us_dragx-us_dragox, us_dragy-us_dragoy);
}

void us_multidragdraw(INTBIG col, INTBIG dx, INTBIG dy)
{
	REGISTER INTSML i, j, thisend, portcount, len, endfixed;
	REGISTER INTBIG wid, lambda, xcur, ycur, xw, yw;
	INTBIG lx, ly, hx, hy, xc, yc;
	XARRAY trans;
	INTSML tsx, tsy;
	REGISTER char *str;
	REGISTER VARIABLE *var;
	REGISTER NODEINST *ni;
	REGISTER ARCINST *ai;
	REGISTER PORTARCINST *pi;
	REGISTER PORTPROTO *pp;

	us_highl.col = col;
	if (us_dragextra != 0) us_dragextra++;
	for(i=0; us_dragobjectlist[i] != NOGEOM; i++)
	{
		if (us_dragobjectlist[i]->entrytype != OBJNODEINST) continue;
		ni = us_dragobjectlist[i]->entryaddr.ni;

		/* if node is an invisible pin with text, show the text highlight */
		if (ni->proto == gen_invispinprim)
		{
			var = getvalkey((INTBIG)ni, VNODEINST, -1, art_messagekey);
			if (var != NOVARIABLE)
			{
				/* determine number of lines of text and text size */
				str = describevariable(var, -1, -1);
				len = 1;
				if ((var->type&VISARRAY) != 0) len = getlength(var);
				if (len > 1)
				{
					xw = ni->geom->highx - ni->geom->lowx;
					yw = ni->geom->highy - ni->geom->lowy;
				} else
				{
					us_settextsize(el_curwindow, truefontsize((INTSML)((var->textdescript&VTSIZE)>>VTSIZESH),
						el_curwindow, el_curtech));
					us_textsize(el_curwindow, str, &tsx, &tsy);
					xw = muldiv(tsx, el_curwindow->screenhx-el_curwindow->screenlx,
						el_curwindow->usehx-el_curwindow->uselx);
					yw = muldiv(tsy, el_curwindow->screenhy-el_curwindow->screenly,
						el_curwindow->usehy-el_curwindow->usely);
				}

				/* get center of highlight, and also a style */
				us_dragpoly->count = 1;
				us_dragpoly->style = FILLED;
				us_dragpoly->xv[0] = (ni->lowx + ni->highx) / 2;
				us_dragpoly->yv[0] = (ni->lowy + ni->highy) / 2;
				adjustdisoffset(ni->geom, us_dragpoly, var->textdescript);
				makerot(ni, trans);
				xformpoly(us_dragpoly, trans);
				getcenter(us_dragpoly, &xc, &yc);

				/* clip range of text descriptor */
				lambda = figurelambda(ni->geom);
				xcur = (xc + dx) * 4 / lambda;
				xcur = mini(xcur, 1023);   xcur = maxi(xcur, -1023);
				xc = xcur * lambda / 4;
				ycur = (yc + dy) * 4 / lambda;
				ycur = mini(ycur, 1023);   ycur = maxi(ycur, -1023);
				yc = ycur * lambda / 4;

				/* draw the descriptor */
				us_buildtexthighpoly(ni->geom, xc, yc, xw, yw, us_dragpoly->style, us_dragpoly);

				/* draw the text highlight */
				(void)us_showpoly(us_dragpoly, us_dragwindow);
				continue;
			}
		}

		makerot(ni, trans);
		nodesizeoffset(ni->proto, &lx, &ly, &hx, &hy);
		maketruerectpoly(ni->lowx+lx, ni->highx-hx, ni->lowy+ly, ni->highy-hy, us_dragpoly);
		us_dragpoly->style = CLOSEDRECT;
		xformpoly(us_dragpoly, trans);
		for(j=0; j<us_dragpoly->count; j++)
		{
			us_dragpoly->xv[j] += dx;
			us_dragpoly->yv[j] += dy;
		}
		(void)us_showpoly(us_dragpoly, us_dragwindow);

		/* draw more if in verbose mode */
		if (us_dragextra != 0)
		{
			/* if only 1 node selected, show its ports */
			if (us_dragnodetotal == 1)
			{
				portcount = 0;
				for(pp = ni->proto->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
				{
					/* compute the port bounds */
					shapeportpoly(ni, pp, us_dragpoly, 0);
					us_dragpoly->desc = &us_highl;

					/* see if the polygon is a single point */
					for(j=1; j<us_dragpoly->count; j++)
						if (us_dragpoly->xv[j] != us_dragpoly->xv[j-1] ||
							us_dragpoly->yv[j] != us_dragpoly->yv[j-1]) break;
					if (j < us_dragpoly->count)
					{
						/* not a single point, draw its outline */
						if (us_dragpoly->style == FILLEDRECT) us_dragpoly->style = CLOSEDRECT;
						if (us_dragpoly->style == FILLED) us_dragpoly->style = CLOSED;
						if (us_dragpoly->style == DISC) us_dragpoly->style = CIRCLE;
					} else
					{
						/* single point port: make it a cross */
						us_dragpoly->count = 1;
						us_dragpoly->style = CROSS;
					}
					for(j=0; j<us_dragpoly->count; j++)
					{
						us_dragpoly->xv[j] += dx;
						us_dragpoly->yv[j] += dy;
					}

					/* draw the port */
					(void)us_showpoly(us_dragpoly, us_dragwindow);

					/* stop if interrupted */
					portcount++;
					if ((portcount%20) == 0)
					{
						if (stopping("highlight")) break;
					}
				}
			}

			/* rubber-band arcs to this node */
			for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
			{
				ai = pi->conarcinst;

				/* ignore if not drawing this arc or already drawn */
				if (ai->temp1 == 0 || ai->temp2 == us_dragextra) continue;
				ai->temp2 = us_dragextra;

				/* get original endpoint of arc */
				us_dragpoly->xv[0] = ai->end[0].xpos;
				us_dragpoly->yv[0] = ai->end[0].ypos;
				us_dragpoly->xv[1] = ai->end[1].xpos;
				us_dragpoly->yv[1] = ai->end[1].ypos;
				us_dragpoly->count = 2;
				us_dragpoly->style = OPENED;

				/* offset this end by amount node moves */
				if (ai->end[0].portarcinst == pi) thisend = 0; else thisend = 1;
				us_dragpoly->xv[thisend] += dx;
				us_dragpoly->yv[thisend] += dy;

				/* offset other end if both connect to moved nodes or arc rigid */
				endfixed = 0;
				if (ai->temp1 >= 2 || (ai->userbits&FIXED) != 0) endfixed = 1;

#if 0
				/* if arc goes to stranded pin, make the arc rigid */
				if (ai->end[0].nodeinst == ni) oni = ai->end[1].nodeinst; else
					oni = ai->end[0].nodeinst;
				if (oni == ni) continue;
				fun = nodefunction(oni, &extra);
				if (fun == NPPIN)
				{
					/* found a pin at the end of this arc: make sure it has only one arc */
					j = 0;
					for(opi = oni->firstportarcinst; opi != NOPORTARCINST; opi = opi->nextportarcinst)
						j++;
					if (j == 1) endfixed++;
				}
#endif

				if (endfixed != 0)
				{
					us_dragpoly->xv[1-thisend] += dx;
					us_dragpoly->yv[1-thisend] += dy;
				} else if ((ai->userbits&FIXANG) != 0)
				{
					if (ai->end[0].xpos == ai->end[1].xpos) us_dragpoly->xv[1-thisend] += dx;
					if (ai->end[0].ypos == ai->end[1].ypos) us_dragpoly->yv[1-thisend] += dy;
				}

				/* draw line */
				(void)us_showpoly(us_dragpoly, us_dragwindow);
			}
		}
	}

	for(i=0; us_dragobjectlist[i] != NOGEOM; i++)
	{
		if (us_dragobjectlist[i]->entrytype != OBJARCINST) continue;
		ai = us_dragobjectlist[i]->entryaddr.ai;
		wid = ai->width - arcwidthoffset(ai->proto);
		makearcpoly(ai->length, wid, ai, us_dragpoly, CLOSED);
		for(j=0; j<us_dragpoly->count; j++)
		{
			us_dragpoly->xv[j] += dx;
			us_dragpoly->yv[j] += dy;
		}
		(void)us_showpoly(us_dragpoly, us_dragwindow);
	}
}

/************************* DISTANCE TRACKING *************************/

void us_distancebegin(void)
{
	/* initialize polygon */
	if (us_dragpoly == NOPOLYGON) us_dragpoly = allocpolygon(4, us_aid->cluster);
	us_dragpoly->desc = &us_highl;

	us_dragshown = 0;
	us_dragwindow = el_curwindow;
}

INTSML us_distancedown(INTBIG x, INTBIG y)
{
	REGISTER INTBIG dist;

	/* grid align the cursor value */
	if (us_setxy((INTSML)x, (INTSML)y) != 0 || us_dragwindow != el_curwindow) return(0);
	(void)getxy(&us_dragx, &us_dragy);
	gridalign(&us_dragx, &us_dragy, us_alignment);
	us_setcursorpos(0, us_dragx, us_dragy);

	if (us_dragshown == 0)
	{
		us_dragox = us_dragx;   us_dragoy = us_dragy;
	} else
	{
		/* if it didn't actually move, go no further */
		if (us_dragx == us_lastcurx && us_dragy == us_lastcury) return(0);

		/* undraw the distance */
		us_highl.col = 0;
		us_dragpoly->xv[0] = (us_dragox+us_lastcurx) / 2;
		us_dragpoly->yv[0] = (us_dragoy+us_lastcury) / 2;
		us_dragpoly->count = 1;
		us_dragpoly->style = TEXTBOTLEFT;
		(void)us_showpoly(us_dragpoly, us_dragwindow);

		/* undraw the line */
		us_dragpoly->xv[0] = us_dragox;     us_dragpoly->yv[0] = us_dragoy;
		us_dragpoly->xv[1] = us_lastcurx;   us_dragpoly->yv[1] = us_lastcury;
		us_dragpoly->count = 2;
		us_dragpoly->style = OPENED;
		(void)us_showpoly(us_dragpoly, us_dragwindow);
	}

	/* draw the line */
	us_highl.col = HIGHLIT;
	us_dragpoly->xv[0] = us_dragox;   us_dragpoly->yv[0] = us_dragoy;
	us_dragpoly->xv[1] = us_dragx;    us_dragpoly->yv[1] = us_dragy;
	us_dragpoly->count = 2;
	us_dragpoly->style = OPENED;
	(void)us_showpoly(us_dragpoly, us_dragwindow);

	/* draw the distance */
	us_dragpoly->xv[0] = (us_dragox+us_dragx) / 2;
	us_dragpoly->yv[0] = (us_dragoy+us_dragy) / 2;
	us_dragpoly->count = 1;
	us_dragpoly->style = TEXTBOTLEFT;
	us_dragpoly->font = TXTSMALL;
	dist = computedistance(us_dragox, us_dragoy, us_dragx, us_dragy);
	(void)strcpy(us_dragmessage, "distance:");
	(void)strcat(us_dragmessage, latoa(dist));
	(void)strcat(us_dragmessage, " dX:");
	(void)strcat(us_dragmessage, latoa(us_dragx - us_dragox));
	(void)strcat(us_dragmessage, " dY:");
	(void)strcat(us_dragmessage, latoa(us_dragy - us_dragoy));
	us_dragpoly->string = us_dragmessage;
	(void)us_showpoly(us_dragpoly, us_dragwindow);

	us_dragshown = 1;
	us_lastcurx = us_dragx;   us_lastcury = us_dragy;
	return(0);
}

void us_distanceup(void)
{
	if (us_dragshown != 0)
	{
		/* undraw the distance */
		us_highl.col = 0;
		us_dragpoly->xv[0] = (us_dragox+us_lastcurx) / 2;
		us_dragpoly->yv[0] = (us_dragoy+us_lastcury) / 2;
		us_dragpoly->count = 1;
		us_dragpoly->style = TEXTBOTLEFT;
		(void)us_showpoly(us_dragpoly, us_dragwindow);

		/* undraw the line */
		us_dragpoly->xv[0] = us_dragox;     us_dragpoly->yv[0] = us_dragoy;
		us_dragpoly->xv[1] = us_lastcurx;   us_dragpoly->yv[1] = us_lastcury;
		us_dragpoly->count = 2;
		us_dragpoly->style = OPENED;
		(void)us_showpoly(us_dragpoly, us_dragwindow);
	}
}

/************************* MISCELLANEOUS *************************/

/* cursor advance routine when stretching the highlight object */
INTSML us_stretchdown(INTBIG x, INTBIG y)
{
	/* grid align the cursor value */
	if (us_setxy((INTSML)x, (INTSML)y) != 0 || us_dragwindow != el_curwindow)
	{
		(void)us_setxy((INTSML)us_lastcurx, (INTSML)us_lastcury);
		return(0);
	}
	(void)getxy(&us_lastcurx, &us_lastcury);
	gridalign(&us_lastcurx, &us_lastcury, us_alignment);
	us_setcursorpos(0, us_lastcurx, us_lastcury);

	/* if the box is already being shown, erase it */
	if (us_dragshown != 0)
	{
		/* if it didn't actually move, go no further */
		if (us_dragx == us_lastcurx && us_dragy == us_lastcury) return(0);

		/* undraw the box */
		us_invertstretch();
	}

	/* advance the box */
	maketruerectpoly(us_dragox, us_lastcurx, us_dragoy, us_lastcury, us_dragpoly);
	us_dragpoly->style = CLOSEDRECT;
	us_dragx = us_lastcurx;
	us_dragy = us_lastcury;

	/* draw the new box */
	us_invertstretch();
	us_dragshown = 1;
	return(0);
}

void us_invertstretch(void)
{
	us_wanttoinvert(us_dragox, us_dragoy, us_dragox, us_dragy,  us_dragwindow);
	us_wanttoinvert(us_dragox, us_dragy,  us_dragx,  us_dragy,  us_dragwindow);
	us_wanttoinvert(us_dragx,  us_dragy,  us_dragx,  us_dragoy, us_dragwindow);
	us_wanttoinvert(us_dragx,  us_dragoy, us_dragox, us_dragoy, us_dragwindow);
}

void us_wanttoinvert(INTBIG fx, INTBIG fy, INTBIG tx, INTBIG ty, WINDOW *w)
{
	fx = applyxscale(w, fx-w->screenlx) + w->uselx;
	fy = applyyscale(w, fy-w->screenly) + w->usely;
	tx = applyxscale(w, tx-w->screenlx) + w->uselx;
	ty = applyyscale(w, ty-w->screenly) + w->usely;
	if (clipline(&fx, &fy, &tx, &ty, w->uselx, w->usehx, w->usely, w->usehy) != 0) return;
	us_invertline(w, (INTSML)fx, (INTSML)fy, (INTSML)tx, (INTSML)ty);
}

/* cursor advance routine when creating or moving an object */
INTSML us_dragdown(INTBIG x, INTBIG y)
{
	REGISTER INTSML i;

	/* grid align the cursor value */
	if (us_setxy((INTSML)x, (INTSML)y) != 0 || us_dragwindow != el_curwindow)
	{
		(void)us_setxy((INTSML)us_lastcurx, (INTSML)us_lastcury);
		return(0);
	}
	(void)getxy(&us_lastcurx, &us_lastcury);
	gridalign(&us_lastcurx, &us_lastcury, us_alignment);
	us_setcursorpos(0, us_lastcurx, us_lastcury);

	/* if the box is already being shown, erase it */
	if (us_dragshown != 0)
	{
		/* if it didn't actually move, go no further */
		if (us_dragx == us_lastcurx && us_dragy == us_lastcury) return(0);

		/* undraw the box */
		us_highl.col = 0;
		(void)us_showpoly(us_dragpoly, us_dragwindow);
	}

	/* advance the box */
	for(i = 0; i < us_dragpoly->count; i++)
	{
		us_dragpoly->xv[i] += us_lastcurx - us_dragx;
		us_dragpoly->yv[i] += us_lastcury - us_dragy;
	}
	us_dragx = us_lastcurx;
	us_dragy = us_lastcury;

	/* draw the new box */
	us_highl.col = HIGHLIT;
	(void)us_showpoly(us_dragpoly, us_dragwindow);
	us_dragshown = 1;
	return(0);
}

/* termination routine when stretching an area */
void us_invertdragup(void)
{
	if (us_dragshown != 0) us_invertstretch();
}

/* termination routine when creating or moving an object */
void us_dragup(void)
{
	if (us_dragshown != 0)
	{
		/* undraw the box */
		us_highl.col = 0;
		(void)us_showpoly(us_dragpoly, us_dragwindow);
	}
}

/*
 * routine to ignore all cursor motion while the button is up
 */
INTSML us_ignoreup(INTBIG x, INTBIG y)
{
	gridalign(&x, &y, us_alignment);
	us_setcursorpos(0, x, y);
	return(0);
}

/*
 * routine to cause termination of tracking when a key is typed
 */
INTSML us_stoponchar(INTBIG x, INTBIG y, INTSML chr)
{
	if (chr == 'a' || chr == 'A')
	{
		el_pleasestop = 1;
		stopping("Tracking");
	}
	return(1);
}

/*
 * routine to cause termination of tracking and pop stacked highlighting
 * when a key is typed
 */
INTSML us_stopandpoponchar(INTBIG x, INTBIG y, INTSML chr)
{
	if (chr == 'a' || chr == 'A')
	{
		el_pleasestop = 1;
		stopping("Tracking");
		(void)us_pophighlight(0);
	}
	return(1);
}
