/*
 * Electric(tm) VLSI Design System
 *
 * File: usrwindow.c
 * User interface aid: public graphics routines
 * Written by: Steven M. Rubin, Electric Editor Incorporated
 *
 * Copyright (c) 1998 Electric Editor Incorporated.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Electric Editor Incorporated
 * 23470 Sunset Drive, Suite 108
 * Los Gatos, California 95033
 * support@electriceditor.com
 */

#include "global.h"
#include "egraphics.h"
#include "usr.h"

/***** REDRAW MODULES *****/

#define	NOREDRAW ((REDRAW *)-1)

typedef struct Iredraw
{
	INTSML          entrytype;			/* type of object being re-drawn */
	union us_entry
	{
		NODEINST   *ni;
		ARCINST    *ai;
		INTBIG      blind;
	} entryaddr;						/* object being re-drawn */
	struct Iredraw *nextredraw;			/* next in list */
} REDRAW;

REDRAW    *us_redrawfree = NOREDRAW;	/* free list of re-draw modules */
REDRAW    *us_firstredraw = NOREDRAW;	/* first in list of active re-draw modules */
REDRAW    *us_firstopaque = NOREDRAW;	/* first in list of active opaque re-draws */
REDRAW    *us_usedredraw = NOREDRAW;	/* first in list of used re-draw modules */

REDRAW *us_allocredraw(void);
void us_freeredraw(REDRAW*);

/***** MISCELLANEOUS *****/

#define ZSHIFT       4
#define ZUP(x)       ((x) << ZSHIFT)
#define ZDN(x)       (((x) + 1) >> ZSHIFT)

#define	CROSSSIZE      2				/* size in pixels of half a cross */
#define	BIGCROSSSIZE   5				/* size in pixels of half a big cross */
#define MINDISCSIZE    1				/* min radius of a disc */

extern GRAPHICS us_ebox, us_box;

/* prototypes for local routines */
INTSML us_showin(WINDOW*, GEOM*, NODEPROTO*, XARRAY, INTSML, INTSML);
void us_queuevicinity(GEOM*, NODEPROTO*, XARRAY, INTBIG, INTBIG, INTBIG, INTBIG, INTSML);
INTSML us_showthispoly(POLYGON*, WINDOW*);
void us_wanttodraw(INTBIG, INTBIG, INTBIG, INTBIG, WINDOW*, GRAPHICS*, INTSML);
void us_wanttodrawo(INTBIG, INTSML, INTBIG, INTSML, INTBIG, INTSML, INTBIG, INTSML, WINDOW*, GRAPHICS*, INTSML);
void us_drawextendedpolyline(POLYGON*, WINDOW*);
INTSML us_writetext(INTBIG, INTBIG, INTBIG, INTBIG, GRAPHICS*, char*, INTSML, WINDOW*);
void us_reverse(POLYGON*);

/******************** WINDOW DISPLAY ********************/

/* routine to erase the facet in window "w" */
void us_clearwindow(WINDOW *w)
{
	REGISTER INTBIG newstate;

	startobjectchange((INTBIG)w, VWINDOW);
	(void)setval((INTBIG)w, VWINDOW, "curnodeproto", (INTBIG)NONODEPROTO, VNODEPROTO);
	newstate = (w->state & ~(GRIDON|WINDOWTYPE|WINDOWSIMULATING)) | DISPWINDOW;
	(void)setval((INTBIG)w, VWINDOW, "state", newstate, VINTEGER);
	(void)setval((INTBIG)w, VWINDOW, "buttonhandler", (INTBIG)DEFAULTBUTTONHANDLER, VADDRESS);
	(void)setval((INTBIG)w, VWINDOW, "charhandler", (INTBIG)DEFAULTCHARHANDLER, VADDRESS);
	(void)setval((INTBIG)w, VWINDOW, "changehandler", (INTBIG)DEFAULTCHANGEHANDLER, VADDRESS);
	(void)setval((INTBIG)w, VWINDOW, "termhandler", (INTBIG)DEFAULTTERMHANDLER, VADDRESS);
	(void)setval((INTBIG)w, VWINDOW, "redisphandler", (INTBIG)DEFAULTREDISPHANDLER, VADDRESS);
	endobjectchange((INTBIG)w, VWINDOW);
}

/* routine to erase the display of window "w" */
void us_erasewindow(WINDOW *w)
{
	static POLYGON *poly = NOPOLYGON;

	/* get polygon */
	if (poly == NOPOLYGON) poly = allocpolygon(4, us_aid->cluster);

	maketruerectpoly(w->screenlx, w->screenhx, w->screenly, w->screenhy, poly);
	poly->desc = &us_ebox;
	poly->style = FILLEDRECT;
	(*us_displayroutine)(poly, w);
	flushscreen();
}

/*
 * routine to begin editing facet "cur" in the current window.  The screen
 * extents of the facet are "lx", "hx", "ly", and "hy".  If "focusinst"
 * is not NONODEINST, then highlight that node and port "facetinstport"
 * (if it is not NOPORTPROTO) in the new window.  If "newframe" is nonzero,
 * create a window for this facet, otherwise reuse the current one.
 */
void us_switchtofacet(NODEPROTO *cur, INTBIG lx, INTBIG hx, INTBIG ly, INTBIG hy,
	NODEINST *focusinst, PORTPROTO *facetinstport, INTSML newframe)
{
	HIGHLIGHT newhigh;
	REGISTER INTSML i, l;
	INTSML dummy;
	REGISTER VARIABLE *var;
	char *one[1];
	REGISTER INTBIG oldstate;
	REGISTER char *thisline, *location;
	REGISTER EDITOR *ed;
	WINDOW *neww, *oldw;

	us_clearhighlightcount();
	oldw = el_curwindow;
	if (oldw == NOWINDOW)
	{
		oldstate = PORTSFULL|DISPWINDOW;
		location = "Entire";
	} else
	{
		oldstate = oldw->state;
		location = oldw->location;
	}
	startobjectchange((INTBIG)us_aid, VAID);
	if (newframe == 0)
	{
		/* delete current window and create a new window for this facet */
		neww = newewindow(location, oldw, 0);
		if (neww == NOWINDOW) return;
		if (oldw != NOWINDOW) killwindow(oldw);
	} else
	{
		/* just create a new window for this facet */
		neww = us_wantnewwindow();
		if (neww == NOWINDOW)
		{
			us_abortcommand("Cannot create new window");
			return;
		}
	}

	if ((cur->cellview->viewstate&TEXTVIEW) != 0)
	{
		/* text window: make an editor */
		if (us_makeeditor(neww, describenodeproto(cur), &dummy, &dummy) == NOWINDOW)
			return;

		/* get the text that belongs here */
		var = getvalkey((INTBIG)cur, VNODEPROTO, VSTRING|VISARRAY, el_facet_message);
		if (var == NOVARIABLE)
		{
			one[0] = "";
			var = setvalkey((INTBIG)cur, VNODEPROTO, el_facet_message, (INTBIG)one,
				VSTRING|VISARRAY|(1<<VLENGTHSH));
			if (var == NOVARIABLE) return;
		}

		ed = neww->editor;
		ed->editobjaddr = (char *)cur;
		ed->editobjtype = VNODEPROTO;
		ed->editobjqual = "FACET_message";
		ed->editobjvar = var;

		/* load the text into the window */
		us_suspendgraphics(neww);
		l = getlength(var);
		for(i=0; i<l; i++)
		{
			thisline = ((char **)var->addr)[i];
			if (i == l-1 && *thisline == 0) continue;
			us_addline(neww, i, thisline);
		}
		us_resumegraphics(neww);

		/* setup for editing */
		neww->curnodeproto = cur;
		neww->changehandler = us_textfacetchanges;
	} else
	{
		neww->curnodeproto = cur;
		neww->editor = NOEDITOR;

		/* change the window type to be appropriate for editing */
		neww->state = (oldstate & ~(WINDOWTYPE|WINDOWSIMULATING)) | DISPWINDOW;

		neww->buttonhandler = DEFAULTBUTTONHANDLER;
		neww->charhandler = DEFAULTCHARHANDLER;
		neww->changehandler = DEFAULTCHANGEHANDLER;
		neww->termhandler = DEFAULTTERMHANDLER;
		neww->redisphandler = DEFAULTREDISPHANDLER;

		us_squarescreen(neww, NOWINDOW, 0, &lx, &hx, &ly, &hy);
		neww->screenlx = lx;
		neww->screenhx = hx;
		neww->screenly = ly;
		neww->screenhy = hy;
		computewindowscale(neww);

		/* window gets bigger: see if grid can be drawn */
		us_gridset(neww, neww->state);

		if (focusinst != NONODEINST)
		{
			newhigh.status = HIGHFROM;
			newhigh.facet = focusinst->parent;
			newhigh.fromgeom = focusinst->geom;
			newhigh.fromport = facetinstport;
			newhigh.frompoint = 0;
			newhigh.fromvar = NOVARIABLE;
			(void)us_addhighlight(&newhigh);
		}
	}

	(void)setval((INTBIG)el_curlib, VLIBRARY, "curnodeproto", (INTBIG)cur, VNODEPROTO);
	endobjectchange((INTBIG)us_aid, VAID);
	(void)setvalkey((INTBIG)us_aid, VAID, us_current_window, (INTBIG)neww, VWINDOW|VDONTSAVE);
}

/*
 * routine to initialize for changes to the screen
 */
void us_beginchanges(void)
{
	us_startbatching();
}

/*
 * routine to finish making changes to the screen.  If "which" is NOWINDOW,
 * changes are made to all windows.  Otherwise, "which" is the specific
 * window to update.
 */
void us_endchanges(WINDOW *which)
{
	REGISTER WINDOW *w;
	XARRAY trans;
	REGISTER REDRAW *r, *nextr;
	REGISTER NODEINST *ni;
	REGISTER ARCINST *ai;
	REGISTER INTSML stopped, ret;

	stopped = el_pleasestop;
	for(r = us_firstredraw; r != NOREDRAW; r = nextr)
	{
		/* remember the next redraw module and queue this one for deletion */
		nextr = r->nextredraw;
		r->nextredraw = us_usedredraw;
		us_usedredraw = r;

		if (stopped != 0) continue;

		if (r->entrytype == OBJNODEINST)
		{
			ni = r->entryaddr.ni;
			if ((ni->userbits & (REWANTN|RETDONN|DEADN)) == REWANTN)
			{
				if (ni->rotation == 0 && ni->transpose == 0) transid(trans); else
					makerot(ni, trans);
				if (which != NOWINDOW)
				{
					ret = us_showin(which, ni->geom, ni->parent, trans, LAYERA, 1);
					if (ret < 0) stopped = 1; else
						if (ret & 2) us_queueopaque(ni->geom, 0);
				} else
				{
					for(w = el_topwindow; w != NOWINDOW; w = w->nextwindow)
					{
						ret = us_showin(w, ni->geom, ni->parent, trans, LAYERA, 1);
						if (ret < 0) { stopped = 1;   break; }
						if (ret & 2) us_queueopaque(ni->geom, 0);
					}
				}
				ni->userbits |= RETDONN;
			}
		} else
		{
			ai = r->entryaddr.ai;
			if ((ai->userbits & (RETDONA|DEADA)) == 0)
			{
				if (which != NOWINDOW)
				{
					ret = us_showin(which, ai->geom, ai->parent, el_matid, LAYERA, 1);
					if (ret < 0) stopped = 1; else
						if (ret & 2) us_queueopaque(ai->geom, 0);
				} else
				{
					for(w = el_topwindow; w != NOWINDOW; w = w->nextwindow)
					{
						ret = us_showin(w, ai->geom, ai->parent, el_matid, LAYERA, 1);
						if (ret < 0) { stopped = 1;   break; }
						if (ret & 2) us_queueopaque(ai->geom, 0);
					}
				}
				ai->userbits |= RETDONA;
			}
		}
	}
	us_firstredraw = NOREDRAW;

	/* now re-draw the opaque objects */
	for(r = us_firstopaque; r != NOREDRAW; r = nextr)
	{
		/* remember the next redraw module and queue this one for deletion */
		nextr = r->nextredraw;
		r->nextredraw = us_usedredraw;
		us_usedredraw = r;

		if (stopped != 0) continue;

		if (r->entrytype == OBJNODEINST)
		{
			ni = r->entryaddr.ni;
			if ((ni->userbits & (REWANTN|REODONN|DEADN)) == REWANTN)
			{
				if (ni->rotation == 0 && ni->transpose == 0) transid(trans); else
					makerot(ni, trans);
				if (which != NOWINDOW)
				{
					ret = us_showin(which, ni->geom, ni->parent, trans, LAYERA, 2);
					if (ret < 0) stopped = 1;
				} else
				{
					for(w = el_topwindow; w != NOWINDOW; w = w->nextwindow)
					{
						ret = us_showin(w, ni->geom, ni->parent, trans, LAYERA, 2);
						if (ret < 0) { stopped = 1;   break; }
					}
				}
				ni->userbits |= REODONN;
			}
		} else
		{
			ai = r->entryaddr.ai;
			if ((ai->userbits & (REODONA|DEADA)) == 0)
			{
				if (which != NOWINDOW)
				{
					ret = us_showin(which, ai->geom, ai->parent, el_matid, LAYERA, 2);
					if (ret < 0) stopped = 1;
				} else
				{
					for(w = el_topwindow; w != NOWINDOW; w = w->nextwindow)
					{
						ret = us_showin(w, ai->geom, ai->parent, el_matid, LAYERA, 2);
						if (ret < 0) { stopped = 1;   break; }
					}
				}
				ai->userbits |= REODONA;
			}
		}
	}
	us_firstopaque = NOREDRAW;

	/* now free up all of the redraw modules */
	for(r = us_usedredraw; r != NOREDRAW; r = nextr)
	{
		nextr = r->nextredraw;
		if (r->entrytype == OBJNODEINST)
		{
			ni = r->entryaddr.ni;
			ni->userbits &= ~(REWANTN|RELOCLN);
		} else if (r->entrytype == OBJARCINST)
		{
			ai = r->entryaddr.ai;
			ai->userbits &= ~(REWANTA|RELOCLA);
		}
		us_freeredraw(r);
	}
	us_usedredraw = NOREDRAW;

	us_endbatching();
}

/*
 * routine to make a change to the screen (sandwiched between a "us_beginchanges"
 * and an "us_endchanges" call.  Returns an indicator of what else needs to be
 * drawn (negative to stop display).
 */
INTSML us_showin(WINDOW *w, GEOM *object, NODEPROTO *parnt, XARRAY trans, INTSML on,
	INTSML layers)
{
	REGISTER NODEINST *ni;
	XARRAY localtran, subrot, subtran;
	REGISTER INTSML moretodo, res;
	REGISTER INTBIG objlocal;

	/* if the parent is the current facet in the window, draw the instance */
	moretodo = 0;
	if (parnt == w->curnodeproto)
	{
		if (object->entrytype == OBJNODEINST)
		{
			ni = object->entryaddr.ni;
			begintraversehierarchy();
			res = us_drawnodeinst(ni, on, trans, layers, w);
			if (res == -2) return(-1);
			if (res < 0) res = 0;
			moretodo |= res;
		} else if (object->entrytype == OBJARCINST)
		{
			res = us_drawarcinst(object->entryaddr.ai, on, trans, layers, w);
			if (res < 0) return(-1);
			moretodo |= res;
		}

		/* if drawing the opaque layer, don't look for other things to draw */
		if (on && layers == 2) return(moretodo);

		/* queue re-drawing of objects in vicinity of this one */
		if (object->entrytype == OBJNODEINST)
			us_queuevicinity(object, parnt, trans, ni->lowx, ni->highx, ni->lowy, ni->highy, on); else
				us_queuevicinity(object, parnt, trans, object->lowx,
						object->highx,object->lowy,object->highy, on);
		return(moretodo);
	}

	/* look at all instances of the parent facet that are expanded */
	if (object->entrytype == OBJNODEINST)
		objlocal = object->entryaddr.ni->userbits & RELOCLN; else
			objlocal = object->entryaddr.ai->userbits & RELOCLA;

	/* Steve Holmlund of Factron suggested this next statement */
	if (on == 0) objlocal = 0;

	for(ni = parnt->firstinst; ni != NONODEINST; ni = ni->nextinst)
	{
		if ((ni->userbits & NEXPAND) == 0) continue;
		if (objlocal != 0 && (ni->userbits&(RELOCLN|REWANTN)) == 0) continue;

		/* transform nodeinst to outer instance */
		maketrans(ni, localtran);
		transmult(trans, localtran, subrot);
		if (ni->rotation == 0 && ni->transpose == 0)
		{
			res = us_showin(w, object, ni->parent, subrot, on, layers);
			if (res < 0) return(-1);
			moretodo |= res;
		} else
		{
			makerot(ni, localtran);
			transmult(subrot, localtran, subtran);
			res = us_showin(w, object, ni->parent, subtran, on, layers);
			if (res < 0) return(-1);
			moretodo |= res;
		}
	}
	return(moretodo);
}

/*
 * routine to queue object "p" to be re-drawn.  If "local" is nonzero, only
 * draw the local instance of this object, not every instance.
 */
void us_queueredraw(GEOM *p, INTSML local)
{
	REGISTER REDRAW *r;
	REGISTER NODEINST *ni;
	REGISTER ARCINST *ai;

	if (p->entrytype == OBJNODEINST)
	{
		ni = p->entryaddr.ni;
		ni->userbits = (ni->userbits & ~(RETDONN|REODONN|RELOCLN)) | REWANTN;
		if (local) ni->userbits |= RELOCLN;
	} else if (p->entrytype == OBJARCINST)
	{
		ai = p->entryaddr.ai;
		ai->userbits = (ai->userbits & ~(RETDONA|REODONA|RELOCLA)) | REWANTA;
		if (local) ai->userbits |= RELOCLA;
	}

	/* queue this object for being re-drawn */
	r = us_allocredraw();
	if (r == NOREDRAW)
	{
		ttyputerr("No memory for display!");
		return;
	}
	r->entrytype = p->entrytype;
	r->entryaddr.blind = p->entryaddr.blind;
	r->nextredraw = us_firstredraw;
	us_firstredraw = r;
}

/*
 * Routine to remove all redraw objects that are in library "lib"
 * (because the library has been deleted)
 */
void us_unqueueredraw(LIBRARY *lib)
{
	REGISTER REDRAW *r, *nextr, *lastr;
	REGISTER LIBRARY *rlib;

	lastr = NOREDRAW;
	for(r = us_firstredraw; r != NOREDRAW; r = nextr)
	{
		nextr = r->nextredraw;
		if (r->entrytype == OBJNODEINST) rlib = r->entryaddr.ni->parent->cell->lib; else
			rlib = r->entryaddr.ai->parent->cell->lib;
		if (rlib == lib)
		{
			if (lastr == NOREDRAW) us_firstredraw = nextr; else
				lastr->nextredraw = nextr;
			us_freeredraw(r);
			continue;
		}
		lastr = r;
	}
}

/*
 * routine to erase the display of "geom" in all windows (sandwiched between
 * "us_beginchanges" and "us_endchanges" calls)
 */
void us_undisplayobject(GEOM *geom)
{
	REGISTER NODEINST *ni;
	REGISTER ARCINST *ai;
	REGISTER WINDOW *w;
	XARRAY trans;

	if (geom->entrytype == OBJNODEINST)
	{
		ni = geom->entryaddr.ni;
		if (ni->rotation == 0 && ni->transpose == 0) transid(trans); else
			makerot(ni, trans);
		for(w = el_topwindow; w != NOWINDOW; w = w->nextwindow)
			(void)us_showin(w, geom, ni->parent, trans, LAYERN, 3);
	} else if (geom->entrytype == OBJARCINST)
	{
		ai = geom->entryaddr.ai;
		for(w = el_topwindow; w != NOWINDOW; w = w->nextwindow)
			(void)us_showin(w, geom, ai->parent, el_matid, LAYERN, 3);
	}
}

/******************** HELPER ROUTINES ******************/

/*
 * routine to queue the opaque layers of object "p" to be re-drawn.
 * If "local" is nonzero, only draw the local instance of this object, not
 * every instance.
 */
void us_queueopaque(GEOM *p, INTSML local)
{
	REGISTER REDRAW *r;
	REGISTER NODEINST *ni;
	REGISTER NODEPROTO *np;
	REGISTER ARCINST *ai;
	REGISTER INTSML i;

	if (p->entrytype == OBJNODEINST)
	{
		ni = p->entryaddr.ni;

		/* count displayable variables on this node */
		for(i = 0; i < ni->numvar; i++)
			if ((ni->firstvar[i].type&VDISPLAY) != 0) break;

		/* if this nodeinst has nothing on the opaque layer, don't queue it */
		np = ni->proto;
		if (np->index != 0 && (np->userbits&NHASOPA) == 0 &&
			i >= ni->numvar && ni->firstportexpinst == NOPORTEXPINST) return;

		/* set the nodeinst bits for opaque redraw */
		ni->userbits = (ni->userbits & ~REODONN) | REWANTN;
		if (local) ni->userbits |= RELOCLN;
	} else if (p->entrytype == OBJARCINST)
	{
		ai = p->entryaddr.ai;

		/* count displayable variables on this arc */
		for(i = 0; i < ai->numvar; i++)
			if ((ai->firstvar[i].type&VDISPLAY) != 0) break;

		/* if this arcinst has nothing on the opaque layer, don't queue it */
		if ((ai->proto->userbits&AHASOPA) == 0 && i >= ai->numvar) return;

		/* set the arcinst bits for opaque redraw */
		ai->userbits = (ai->userbits & ~REODONA) | REWANTA;
		if (local) ai->userbits |= RELOCLA;
	}

	/* queue the object for opaque redraw */
	r = us_allocredraw();
	if (r == NOREDRAW)
	{
		ttyputerr("No memory for display!");
		return;
	}
	r->entrytype = p->entrytype;
	r->entryaddr.blind = p->entryaddr.blind;
	r->nextredraw = us_firstopaque;
	us_firstopaque = r;
}

/*
 * routine to allocate a new redraw from the pool (if any) or memory
 * routine returns NOREDRAW upon error
 */
#define	REDRAWBLOCKSIZE 200	/* number to "malloc" at a time */
REDRAW *us_allocredraw(void)
{
	REGISTER REDRAW *r;
	REGISTER INTSML i, j;

	if (us_redrawfree == NOREDRAW)
	{
		for(i=REDRAWBLOCKSIZE; i>0; i /= 2)
		{
			r = (REDRAW *)emalloc((sizeof (REDRAW)) * i, us_aid->cluster);
			if (r != 0) break;
		}
		if (r == 0) return(NOREDRAW);
		for(j=1; j<i; j++) r[j].nextredraw = &r[j-1];
		r[0].nextredraw = NOREDRAW;   us_redrawfree = &r[i-1];
	}

	/* take module from free list */
	r = us_redrawfree;
	us_redrawfree = r->nextredraw;
	return(r);
}

/*
 * routine to return redraw module "r" to the pool of free modules
 */
void us_freeredraw(REDRAW *r)
{
	r->nextredraw = us_redrawfree;
	us_redrawfree = r;
}

/*
 * routine to queue for local re-draw anything in facet "parnt" that is
 * in the vicinity of "object" with bounding box (lx-hx, ly-hy).  The objects
 * are to be drawn on if "on" is nonzero.  The transformation matrix
 * between the bounding box and the facet is in "trans".
 */
void us_queuevicinity(GEOM *object, NODEPROTO *parnt, XARRAY trans, INTBIG lx,
	INTBIG hx, INTBIG ly, INTBIG hy, INTSML on)
{
	REGISTER GEOM *look;
	REGISTER NODEINST *ni;
	INTBIG bx, ux, by, uy;
	XARRAY localtran, xf, subrot;
	REGISTER INTSML local;
	REGISTER INTBIG i, search, former;

	xform(lx,ly, &bx,&by, trans);
	xform(hx,hy, &ux,&uy, trans);
	if (bx > ux) { i = bx;  bx = ux;  ux = i; }
	if (by > uy) { i = by;  by = uy;  uy = i; }
	search = initsearch(bx,ux,by,uy, parnt);
	if (object == NOGEOM) local = 1; else local = 0;
	for(;;)
	{
		if ((look = nextobject(search)) == NOGEOM) break;

		/* don't re-draw the current object whose vicinity is being checked */
		if (look == object) continue;

		/* get the former inside factor (if it is a nodeinst) */
		if (look->entrytype == OBJNODEINST) former = look->entryaddr.ni->userbits;

		/* redraw all if object is going off, redraw opaque if going on */
		if (on == LAYERN) us_queueredraw(look, local); else
			us_queueopaque(look, local);

		/* that's all if the object is an arcinst */
		if (look->entrytype != OBJNODEINST) continue;
		ni = look->entryaddr.ni;

		/* don't bother exploring if the nodeinst is already to be redrawn */
		if ((former & REWANTN) != 0) continue;

		/* explore the insides if the facet is expanded */
		if (ni->proto->index != 0 || (ni->userbits&NEXPAND) == 0) continue;

		/* explore the insides of this node and queue the pertinent parts */
		maketransI(ni, localtran);
		transmult(trans, localtran, xf);

		/* this is the former contents if "makerotI", which I think is wrong...smr */
		if (ni->transpose != 0) makeangle(ni->rotation, ni->transpose, localtran); else
			makeangle((INTSML)((3600 - ni->rotation)%3600), 0, localtran);
		localtran[2][0] = ni->proto->lowx + ni->proto->highx;
		localtran[2][1] = ni->proto->lowy + ni->proto->highy;
		xform(-localtran[2][0], -localtran[2][1], &localtran[2][0], &localtran[2][1], localtran);
		localtran[2][0] /= 2;    localtran[2][1] /= 2;
		transmult(xf, localtran, subrot);
		us_queuevicinity(NOGEOM, ni->proto, subrot, lx,hx,ly,hy, on);

		/* now flag this nodeinst so that it doesn't get completely redrawn */
		ni->userbits = (ni->userbits & ~REWANTN) | RELOCLN;
	}
}

/*
 * null routine for polygon display
 */
INTSML us_nulldisplayroutine(POLYGON *obj, WINDOW *w)
{
	return(1);
}

/*
 * routine to write polygon "obj" in window "w".  Returns nonzero
 * if nothing is drawn (due to clipping or otherwise)
 */
INTSML us_showpoly(POLYGON *obj, WINDOW *w)
{
	/* quit if no bits to be written */
	if (obj->desc->bits == LAYERN) return(1);

	return(us_showthispoly(obj, w));
}

INTSML us_showthispoly(POLYGON *obj, WINDOW *w)
{
	REGISTER INTBIG i;
	REGISTER INTSML pre, dispstyle, tx, ty;
	INTBIG lx, ux, ly, uy, six, siy;
	WINDOW wsc;
	REGISTER GRAPHICS *gra;
	static POLYGON *objc = NOPOLYGON;
	INTSML tsx, tsy;

	/* special case for grid display */
	if (obj->style == GRIDDOTS)
	{
		us_drawgrid(w, obj);
		return(0);
	}

	/* now draw the polygon */
	gra = obj->desc;
	dispstyle = us_getdispstyle(w);
	switch (obj->style)
	{
		case FILLED:		/* filled polygon */
		case FILLEDRECT:	/* filled rectangle */
			if (isbox(obj, &lx, &ux, &ly, &uy) != 0)
			{
				/* simple rectangular box: transform, clip, and draw */
				if (us_makescreen(&lx, &ly, &ux, &uy, w)) return(1);
				us_drawbox(w, (INTSML)lx, (INTSML)ux, (INTSML)ly, (INTSML)uy, gra);

				/* for patterned and outlined rectangles, draw the box too */
				if ((gra->style[dispstyle]&(NATURE|OUTLINEPAT)) == (PATTERNED|OUTLINEPAT))
				{
					us_drawline(w, (INTSML)lx, (INTSML)ly, (INTSML)lx, (INTSML)uy, gra, 0);
					us_drawline(w, (INTSML)lx, (INTSML)uy, (INTSML)ux, (INTSML)uy, gra, 0);
					us_drawline(w, (INTSML)ux, (INTSML)uy, (INTSML)ux, (INTSML)ly, gra, 0);
					us_drawline(w, (INTSML)ux, (INTSML)ly, (INTSML)lx, (INTSML)ly, gra, 0);
				}
				return(0);
			}

			/* copy the polygon since it will be mangled when clipped */
			if (objc == NOPOLYGON) objc = allocpolygon(obj->count, us_aid->cluster); else
				if (objc->limit < obj->count) (void)extendpolygon(objc, obj->count);
			objc->count = obj->count;
			objc->style = obj->style;
			objc->desc = gra;
			for(i=0; i<obj->count; i++)
			{
				objc->xv[i] = applyxscale(w, obj->xv[i]-w->screenlx) + w->uselx;
				objc->yv[i] = applyyscale(w, obj->yv[i]-w->screenly) + w->usely;
			}

			/* clip and draw the polygon */
			clippoly(objc, w->uselx, w->usehx, w->usely, w->usehy);
			if (objc->count <= 1) return(1);
			if (objc->count > 2)
			{
				/* always clockwise */
				if (areapoly(objc) < 0) us_reverse(objc);
				us_drawpolygon(w, objc->xv, objc->yv, objc->count, objc->desc);

				/* for patterned and outlined polygons, draw the outline too */
				if ((objc->desc->style[dispstyle]&(NATURE|OUTLINEPAT)) ==
					(PATTERNED|OUTLINEPAT))
				{
					for(i=0; i<objc->count; i++)
					{
						if (i == 0) pre = objc->count-1; else pre = i-1;
						us_drawline(w, (INTSML)objc->xv[pre], (INTSML)objc->yv[pre], (INTSML)objc->xv[i], (INTSML)objc->yv[i],
							objc->desc, 0);
					}
				}
			} else us_drawline(w, (INTSML)objc->xv[0], (INTSML)objc->yv[0], (INTSML)objc->xv[1], (INTSML)objc->yv[1],
				gra, 0);
			return(0);

		case CLOSEDRECT:		/* closed rectangle outline */
			us_wanttodraw(obj->xv[0], obj->yv[0], obj->xv[0], obj->yv[1], w, gra, 0);
			us_wanttodraw(obj->xv[0], obj->yv[1], obj->xv[1], obj->yv[1], w, gra, 0);
			us_wanttodraw(obj->xv[1], obj->yv[1], obj->xv[1], obj->yv[0], w, gra, 0);
			us_wanttodraw(obj->xv[1], obj->yv[0], obj->xv[0], obj->yv[0], w, gra, 0);
			return(0);

		case CROSSED:		/* polygon outline with cross */
			us_wanttodraw(obj->xv[0], obj->yv[0], obj->xv[2], obj->yv[2], w, gra, 0);
			us_wanttodraw(obj->xv[1], obj->yv[1], obj->xv[3], obj->yv[3], w, gra, 0);
			/* fall into the next case */

		case CLOSED:		/* closed polygon outline */
			for(i=0; i<obj->count; i++)
			{
				if (i == 0) pre = obj->count-1; else pre = i-1;
				us_wanttodraw(obj->xv[pre], obj->yv[pre], obj->xv[i], obj->yv[i], w, gra, 0);
			}
			return(0);

		case OPENED:		/* opened polygon outline */
			for(i=1; i<obj->count; i++)
				us_wanttodraw(obj->xv[i-1], obj->yv[i-1], obj->xv[i], obj->yv[i], w, gra, 0);
			return(0);

		case OPENEDT1:		/* opened polygon outline, texture 1 */
			for(i=1; i<obj->count; i++)
				us_wanttodraw(obj->xv[i-1], obj->yv[i-1], obj->xv[i], obj->yv[i], w, gra, 1);
			return(0);

		case OPENEDT2:		/* opened polygon outline, texture 2 */
			for(i=1; i<obj->count; i++)
				us_wanttodraw(obj->xv[i-1], obj->yv[i-1], obj->xv[i], obj->yv[i], w, gra, 2);
			return(0);

		case OPENEDT3:		/* opened polygon outline, texture 3 */
			for(i=1; i<obj->count; i++)
				us_wanttodraw(obj->xv[i-1], obj->yv[i-1], obj->xv[i], obj->yv[i], w, gra, 3);
			return(0);

		case OPENEDO1:		/* extended opened polygon outline */
			us_drawextendedpolyline(obj, w);
			return(0);

		case VECTORS:		/* many lines */
			if (obj->count % 2 != 0)
				ttyputmsg("us_showpoly: bad vector description");
			for(i=0; i<obj->count; i += 2)
				us_wanttodraw(obj->xv[i], obj->yv[i], obj->xv[i+1], obj->yv[i+1], w, gra, 0);
			return(0);

		case CROSS:		/* crosses (always have one point) */
		case BIGCROSS:
			getcenter(obj, &six, &siy);
			if (six < w->screenlx || six > w->screenhx || siy < w->screenly || siy > w->screenhy)
				return(1);
			if (obj->style == CROSS) i = CROSSSIZE; else i = BIGCROSSSIZE;
			us_wanttodrawo(six, (INTSML)(-i), siy, 0, six, (INTSML)i, siy, 0, w, gra, 0);
			us_wanttodrawo(six, 0, siy, (INTSML)(-i), six, 0, siy, (INTSML)i, w, gra, 0);
			return(0);

		case TEXTCENT:		/* text centered in box */
		case TEXTTOP:		/* text below top of box */
		case TEXTBOT:		/* text above bottom of box */
		case TEXTLEFT:		/* text right of left edge of box */
		case TEXTRIGHT:		/* text left of right edge of box */
		case TEXTTOPLEFT:	/* text to lower-right of upper-left corner */
		case TEXTBOTLEFT:	/* text to upper-right of lower-left corner */
		case TEXTTOPRIGHT:	/* text to lower-left of upper-right corner */
		case TEXTBOTRIGHT:	/* text to upper-left of lower-right corner */
			getbbox(obj, &lx, &ux, &ly, &uy);
			if (us_makescreen(&lx, &ly, &ux, &uy, w)) return(1);
			i = truefontsize(obj->font, w, el_curtech);
			us_settextsize(w, (INTSML)i);
			us_textsize(w, obj->string, &tsx, &tsy);
			switch (obj->style)
			{
				case TEXTCENT:
					tx = maxi(w->uselx, (lx+ux-tsx)/2);
					ty = maxi(w->usely, (ly+uy-tsy)/2);
					break;
				case TEXTTOP:
					tx = maxi(w->uselx, (lx+ux-tsx)/2);
					ty = maxi(w->usely, uy-tsy);
					break;
				case TEXTBOT:
					tx = maxi(w->uselx, (lx+ux-tsx)/2);
					ty = ly;
					break;
				case TEXTLEFT:
					tx = lx;
					ty = maxi(w->usely, (ly+uy-tsy)/2);
					break;
				case TEXTRIGHT:
					tx = maxi(w->uselx, ux-tsx);
					ty = maxi(w->usely, (ly+uy-tsy)/2);
					break;
				case TEXTTOPLEFT:
					tx = lx;
					ty = maxi(w->usely, uy-tsy);
					break;
				case TEXTBOTLEFT:
					tx = lx;
					ty = ly;
					break;
				case TEXTTOPRIGHT:
					tx = maxi(w->uselx, ux-tsx);
					ty = maxi(w->usely, uy-tsy);
					break;
				case TEXTBOTRIGHT:
					tx = maxi(w->uselx, ux-tsx);
					ty = ly;
					break;
			}
			us_puttext(w, tx, ty, obj->string, gra);
			return(0);

		case TEXTBOX:		/* text centered and contained in box */
			getbbox(obj, &lx, &ux, &ly, &uy);
			if (us_makescreen(&lx, &ly, &ux, &uy, w)) return(1);
			return(us_writetext(lx,ux, ly,uy, gra, obj->string, truefontsize(obj->font, w, el_curtech), w));

		case CIRCLE:   case DISC:   case CIRCLEARC:
			/* must scale the window for best precision when drawing curves */
			wsc.screenlx = w->screenlx;
			wsc.screenly = w->screenly;
			wsc.screenhx = w->screenhx;
			wsc.screenhy = w->screenhy;
			wsc.uselx = ZUP(w->uselx);
			wsc.usely = ZUP(w->usely);
			wsc.usehx = ZUP(w->usehx);
			wsc.usehy = ZUP(w->usehy);
			computewindowscale(&wsc);

			/* get copy polygon */
			if (objc == NOPOLYGON) objc = allocpolygon(obj->count, us_aid->cluster); else
				if (objc->limit < obj->count) (void)extendpolygon(objc, obj->count);

			/* transform and copy the polygon */
			objc->count = obj->count;
			objc->style = obj->style;
			for(i=0; i<obj->count; i++)
			{
				objc->xv[i] = applyxscale(&wsc, obj->xv[i]-wsc.screenlx) + wsc.uselx;
				objc->yv[i] = applyyscale(&wsc, obj->yv[i]-wsc.screenly) + wsc.usely;
			}

			/* clip the circle */
			cliparc(objc, wsc.uselx, wsc.usehx, wsc.usely, wsc.usehy);

			/* circle outline at [0] radius to [1] */
			if (objc->style == CIRCLE)
			{
				if (objc->count != 2) return(1);
				i = computedistance(objc->xv[0], objc->yv[0], objc->xv[1], objc->yv[1]);
				us_drawcircle(w, ZDN(objc->xv[0]), ZDN(objc->yv[0]), ZDN(i), gra);
				return(0);
			}

			/* filled circle at [0] radius to [1] */
			if (objc->style == DISC)
			{
				if (objc->count != 2) return(1);
				i = ZDN(computedistance(objc->xv[0], objc->yv[0], objc->xv[1], objc->yv[1]));
				if (i <= MINDISCSIZE) i = MINDISCSIZE;
				us_drawdisc(w, ZDN(objc->xv[0]), ZDN(objc->yv[0]), i, gra);
				return(0);
			}

			/* arcs at [i] points [1+i] [2+i] clockwise */
			if (objc->count == 0) return(1);
			if ((objc->count%3) != 0) return(1);
			for (i=0; i<objc->count; i += 3)
				us_drawcirclearc(w, ZDN(objc->xv[i]), ZDN(objc->yv[i]), ZDN(objc->xv[i+1]),
					ZDN(objc->yv[1+i]), ZDN(objc->xv[i+2]), ZDN(objc->yv[i+2]), gra);
			return(0);
	}

	/* unknown polygon type */
	return(1);
}

/*
 * routine to clip and possibly draw a line from (fx,fy) to (tx,ty) in
 * window "w" with description "desc", texture "texture"
 */
void us_wanttodraw(INTBIG fx, INTBIG fy, INTBIG tx, INTBIG ty, WINDOW *w, GRAPHICS *desc,
	INTSML texture)
{
	fx = applyxscale(w, fx-w->screenlx) + w->uselx;
	fy = applyyscale(w, fy-w->screenly) + w->usely;
	tx = applyxscale(w, tx-w->screenlx) + w->uselx;
	ty = applyyscale(w, ty-w->screenly) + w->usely;
	if (clipline(&fx, &fy, &tx, &ty, w->uselx, w->usehx, w->usely, w->usehy) != 0) return;
	us_drawline(w, (INTSML)fx, (INTSML)fy, (INTSML)tx, (INTSML)ty, desc, texture);
}

/*
 * routine to clip and possibly draw a line from (fx,fy) to (tx,ty) in
 * window "w" with description "desc", texture "texture"
 */
void us_wanttodrawo(INTBIG fx, INTSML fxo, INTBIG fy, INTSML fyo, INTBIG tx, INTSML txo,
	INTBIG ty, INTSML tyo, WINDOW *w, GRAPHICS *desc, INTSML texture)
{
	fx = applyxscale(w, fx-w->screenlx) + w->uselx + fxo;
	fy = applyyscale(w, fy-w->screenly) + w->usely + fyo;
	tx = applyxscale(w, tx-w->screenlx) + w->uselx + txo;
	ty = applyyscale(w, ty-w->screenly) + w->usely + tyo;
	if (clipline(&fx, &fy, &tx, &ty, w->uselx, w->usehx, w->usely, w->usehy) != 0) return;
	us_drawline(w, (INTSML)fx, (INTSML)fy, (INTSML)tx, (INTSML)ty, desc, texture);
}

/*
 * routine to draw an opened polygon full of lines, set out by 1 pixel.  The polygon is in "obj".
 */
void us_drawextendedpolyline(POLYGON *obj, WINDOW *w)
{
	REGISTER INTSML i;
	REGISTER INTBIG x1, y1, x2, y2, x1o, y1o, x2o, y2o, centerx, centery, diff;
	INTBIG lx, hx, ly, hy;

	/* if polygon is a line, extension is easy */
	if (isbox(obj, &lx, &hx, &ly, &hy) != 0)
	{
		if (lx == hx)
		{
			us_wanttodrawo(lx, -1, ly, 0, lx, -1, hy, 0, w, obj->desc, 0);
			us_wanttodrawo(lx, 1, ly, 0, lx, 1, hy, 0, w, obj->desc, 0);
			return;
		}
		if (ly == hy)
		{
			us_wanttodrawo(lx, -1, ly, 0, hx, -1, ly, 0, w, obj->desc, 0);
			us_wanttodrawo(lx, 1, ly, 0, hx, 1, ly, 0, w, obj->desc, 0);
			return;
		}
	}

	if (obj->count == 3 && obj->xv[0] == obj->xv[2] && obj->yv[0] == obj->yv[2])
	{
		x1 = obj->xv[0];   y1 = obj->yv[0];
		x2 = obj->xv[1];   y2 = obj->yv[1];
		if (x1 == x2)
		{
			us_wanttodrawo(x1,-1, y1, 0, x2,-1, y2, 0, w, obj->desc, 0);
			us_wanttodrawo(x1, 1, y1, 0, x2, 1, y2, 0, w, obj->desc, 0);
			return;
		}
		if (y1 == y2)
		{
			us_wanttodrawo(x1, 0, y1,-1, x2, 0, y2,-1, w, obj->desc, 0);
			us_wanttodrawo(x1, 0, y1,1, x2, 0, y2, 1, w, obj->desc, 0);
			return;
		}
		if ((x1-x2) * (y1-y2) > 0)
		{
			us_wanttodrawo(x1,1, y1,-1, x2,1, y2,-1, w, obj->desc, 0);
			us_wanttodrawo(x1,-1, y1,1, x2,-1, y2,1, w, obj->desc, 0);
		} else
		{
			us_wanttodrawo(x1,1, y1,1, x2,1, y2,1, w, obj->desc, 0);
			us_wanttodrawo(x1,-1, y1,-1, x2,-1, y2,-1, w, obj->desc, 0);
		}
		return;
	}

	/* do extension about polygon (and see if the polygon is a single point) */
	centerx = centery = diff = 0;
	for(i=0; i<obj->count; i++)
	{
		centerx += obj->xv[i];   centery += obj->yv[i];
		if (obj->xv[i] != obj->xv[0]) diff++;
		if (obj->yv[i] != obj->yv[0]) diff++;
	}
	centerx /= obj->count;   centery /= obj->count;

	/* special case if a single point */
	if (diff == 0)
	{
		us_wanttodrawo(centerx, -1, centery, -1, centerx, 1, centery, -1, w, obj->desc, 0);
		us_wanttodrawo(centerx, 1, centery, -1, centerx, 1, centery, 1, w, obj->desc, 0);
		us_wanttodrawo(centerx, 1, centery, 1, centerx, -1, centery, 1, w, obj->desc, 0);
		us_wanttodrawo(centerx, -1, centery, 1, centerx, -1, centery, -1, w, obj->desc, 0);
		return;
	}

	for(i=1; i<obj->count; i++)
	{
		x1 = obj->xv[i-1];   y1 = obj->yv[i-1];
		x2 = obj->xv[i];     y2 = obj->yv[i];
		if (x1 < centerx) x1o = -1; else
			if (x1 > centerx) x1o = 1; else x1o = 0;
		if (y1 < centery) y1o = -1; else
			if (y1 > centery) y1o = 1; else y1o = 0;
		if (x2 < centerx) x2o = -1; else
			if (x2 > centerx) x2o = 1; else x2o = 0;
		if (y2 < centery) y2o = -1; else
			if (y2 > centery) y2o = 1; else y2o = 0;
		us_wanttodrawo(x1, (INTSML)x1o, y1, (INTSML)y1o, x2, (INTSML)x2o, y2, (INTSML)y2o, w, obj->desc,
			0);
	}
}

/*
 * Write text in a box.  The box ranges from "lx" to "ux" in X and
 * from "ly" to "uy" in Y.  Draw in bit planes "desc->bits" with color
 * "desc->color".  Put "txt" there (or as much as will fit).  The value of
 * "initialfont" is the default size of text which will be reduced until it
 * can fit.  The routine returns 0 if any text is drawn, nonzero if no letters
 * will fit.
 */
INTSML us_writetext(INTBIG lx, INTBIG ux, INTBIG ly, INTBIG uy, GRAPHICS *desc, char *txt,
	INTSML initialfont, WINDOW *win)
{
	REGISTER INTSML stop, save, font;
	INTSML six, siy;

	/* on video displays, only bother using one size of text */
	if (us_getdispstyle(win) == BWVIDEO) initialfont = 0;

	/* scan for a font that fits */
	for(font=initialfont; font>=TXT4P; font--)
	{
		us_settextsize(win, font);
		us_textsize(win, txt, &six, &siy);
		if (six <= ux-lx && siy <= uy-ly) break;
	}

	/* if the text doesn't fit in Y, quit */
	if (siy > uy-ly) return(1);

	/* truncate in X if possible */
	if (six > ux-lx)
	{
		stop = (ux-lx) * strlen(txt);
		stop /= six;
		if (stop == 0) return(1);
		save = txt[stop];   txt[stop] = 0;
		us_textsize(win, txt, &six, &siy);
	} else stop = -1;

	/* draw the text */
	us_puttext(win, (INTSML)(lx+(ux-lx-six)/2), (INTSML)(ly+(uy-ly-siy)/2), txt, desc);
	if (stop >= 0) txt[stop] = save;
	return(0);
}

/******************** WINDOW PANNING ********************/

/*
 * routine to slide the contents of the current window up by "dist" lambda
 * units (slides down if "dist" is negative)
 */
void us_slideup(INTBIG dist)
{
	INTBIG schx, schy;
	REGISTER NODEPROTO *np;

	/* on windows with schematic frames, limit sliding to maximum frame extent */
	np = getcurfacet();
	if (np != NONODEPROTO && framesize(&schx, &schy, np) == 0)
	{
		if (el_curwindow->screenhy - dist > schy/2) dist = el_curwindow->screenhy - schy/2;
		if (el_curwindow->screenly - dist < -schy/2) dist = el_curwindow->screenly + schy/2;
		if (dist == 0) return;
	}

	/* save and erase highlighting */
	us_pushhighlight();
	us_clearhighlightcount();

	/* set the new window data */
	startobjectchange((INTBIG)el_curwindow, VWINDOW);
	(void)setval((INTBIG)el_curwindow, VWINDOW, "screenly", el_curwindow->screenly - dist, VINTEGER);
	(void)setval((INTBIG)el_curwindow, VWINDOW, "screenhy", el_curwindow->screenhy - dist, VINTEGER);
	endobjectchange((INTBIG)el_curwindow, VWINDOW);

	/* restore highlighting */
	(void)us_pophighlight(0);
}

/*
 * routine to slide the current window left by "dist" lambda units
 * (slides right if "dist" is negative)
 */
void us_slideleft(INTBIG dist)
{
	INTBIG schx, schy;
	REGISTER NODEPROTO *np;

	/* on windows with schematic frames, limit sliding to maximum frame extent */
	np = getcurfacet();
	if (np != NONODEPROTO && framesize(&schx, &schy, np) == 0)
	{
		if (el_curwindow->screenhx + dist > schx/2) dist = schx/2 - el_curwindow->screenhx;
		if (el_curwindow->screenlx + dist < -schx/2) dist = -schx/2 - el_curwindow->screenlx;
		if (dist == 0) return;
	}

	/* save and erase highlighting */
	us_pushhighlight();
	us_clearhighlightcount();

	startobjectchange((INTBIG)el_curwindow, VWINDOW);
	(void)setval((INTBIG)el_curwindow, VWINDOW, "screenlx", el_curwindow->screenlx + dist, VINTEGER);
	(void)setval((INTBIG)el_curwindow, VWINDOW, "screenhx", el_curwindow->screenhx + dist, VINTEGER);
	endobjectchange((INTBIG)el_curwindow, VWINDOW);

	/* restore highlighting */
	(void)us_pophighlight(0);
}

/******************** TRANSFORMATION TO SCREEN ********************/

/*
 * routine to convert the reference parameters (lx,ux, ly,uy)
 * which define a box to screen co-ordinates ready to plot.
 * The values are scaled to screen space of window "w" and clipped.
 * If the routine returns nonzero, the box is all off the screen.
 */
INTSML us_makescreen(INTBIG *lx, INTBIG *ly, INTBIG *ux, INTBIG *uy, WINDOW *w)
{
	/* transform to screen space */
	if (*ux < w->screenlx || *lx > w->screenhx) return(1);
	if (*uy < w->screenly || *ly > w->screenhy) return(1);
	*lx = applyxscale(w, *lx-w->screenlx) + w->uselx;
	*ly = applyyscale(w, *ly-w->screenly) + w->usely;
	*ux = applyxscale(w, *ux-w->screenlx) + w->uselx;
	*uy = applyyscale(w, *uy-w->screenly) + w->usely;

	/* now clip to screen bounds */
	if (*lx < w->uselx) *lx = w->uselx;
	if (*ly < w->usely) *ly = w->usely;
	if (*ux > w->usehx) *ux = w->usehx;
	if (*uy > w->usehy) *uy = w->usehy;
	return(0);
}

/*
 * reverse the edge order of a polygon
 */
void us_reverse(POLYGON *poly)
{
	REGISTER INTSML i, invi;
	REGISTER INTBIG mx, my;

	for (i = 0, invi = poly->count-1; i<invi; i++, invi--)
	{
		mx = poly->xv[invi];
		my = poly->yv[invi];
		poly->xv[invi] = poly->xv[i];
		poly->yv[invi] = poly->yv[i];
		poly->xv[i] = mx;
		poly->yv[i] = my;
	}
}
