/*******************************************************************/
/*******************************************************************/
/*                                                                 */
/*                  CALCUL DES UNITES DE STARK DE CORPS            */
/*                         TOTALEMENT REELS                        */
/*                                                                 */
/*******************************************************************/
/*******************************************************************/
/* $Id: stark.c,v 2.0.0.8 1998/05/04 12:58:03 belabas Exp belabas $ */
#include "pari.h"

/********************************************************************/
/*                    Quelques fonctions diverses                   */
/********************************************************************/

/* Calcule l'image de logelt par chi, cf. remarque partie 3 */
static GEN
ComputeImagebyChar(GEN chi, GEN logelt)
{ return gpuigs((GEN)chi[2],itos(gmul((GEN)chi[1],logelt))); }

/* Calcule les representants sur le systeme de generateurs de tous les
   elements du groupe correspondant */
static GEN
FindEltofGroup(long order, GEN cyc)
{
  long l,i,adec,j,dj;
  GEN rep,p1;
  
  l=lg(cyc)-1;
  
  rep=cgetg(order+1,t_VEC);
  for (i=1; i<=order; i++)
  {
    p1=cgetg(l+1,t_COL);
    rep[i]=(long)p1;
    adec=i;
    for (j=l; j; j--)
    {
      dj=itos((GEN)cyc[j]);
      p1[j]=lstoi(adec%dj);
      adec/=dj;
    }
  }

  return rep;
}

/* Soit dataC donne par un InitQuotient0, calcule un systeme de
   representants des elements du quotient dans le groupe au-dessus */
static GEN
ComputeLift(GEN dataC)
{
  long order,i,av=avma;
  GEN cyc,surj,eltq,elt;
  
  order= itos((GEN)dataC[1]);
  cyc  = (GEN)dataC[2];
  surj = (GEN)dataC[3];

  eltq=FindEltofGroup(order,cyc);

  elt=cgetg(order+1,t_VEC);
  for (i=1; i<=order; i++)
    elt[i]=(long)inverseimage(surj,(GEN)eltq[i]);

  return gerepileupto(av,elt);
}

/* Let bnr1, bnr2 be such that mod(bnr2) | mod(bnr1), compute the
   matrix of the surjective map Cl(bnr1) ->> Cl(bnr2) */
static GEN
GetSurjMat(GEN bnr1, GEN bnr2)
{
  long nbg,i;
  GEN gen,M;
  
  gen= gmael(bnr1,5,3);
  nbg= lg(gen)-1;

  M=cgetg(nbg+1,t_MAT);
  for (i=1; i<=nbg; i++) M[i]=(long)isprincipalray(bnr2,(GEN)gen[i]);

  return M;
}

/* Let chi a character defined over bnr and primitif over bnrc,
   compute the corresponding primitif character and the vectors of
   prime ideals dividing bnr but not bnr. */
static GEN
GetPrimChar(GEN chi, GEN bnr, GEN bnrc, long prec)
{
  long nbg,lc,i,j,l,av=avma,nd;
  GEN gen,cyc,U,d,rep,chic,M,s,p1,cond,condc,p2,nf,prdiff,Mrc;
  
  gen  = gmael(bnr,5,3);
  nbg  = lg(gen)-1;
  cyc  = gmael(bnr,5,2);
  Mrc  = diagonal(gmael(bnrc,5,2));
  nf   = gmael(bnr,1,7);
  cond = gmael(bnr,2,1);
  condc= gmael(bnrc,2,1);
 
  if (gegal(cond,condc))
    return NULL;

  cond=(GEN)cond[1];
  condc=(GEN)condc[1];
  
  M=GetSurjMat(bnr,bnrc);
  lc=lg((GEN)M[1])-1;
  p1=hnfall(concat(M,Mrc));
  U=(GEN)p1[2];
  /* l=nbg-lc; */

  chic=cgetg(lc+1,t_VEC);
  for (i=1; i<=lc; i++)
  {
    s=gzero;
    p1=(GEN)U[i+nbg];
    for (j=1; j<=nbg; j++)
    {
      p2=(GEN)cyc[j];
      p2=gdiv((GEN)p1[j],p2);
      s=gadd(s,gmul(p2,(GEN)chi[j]));
    }
    chic[i]=(long)s;
  }

  p2=(GEN)idealfactor(nf,cond)[1];
  l=lg(p2)-1; nd=0;
  prdiff=cgetg(l+1,t_COL);
  for (i=1; i<=l; i++)
    if (!idealval(nf,condc,(GEN)p2[i]))
    { nd++; prdiff[nd]=p2[i]; }
  setlg(prdiff,nd+1);

  p1=cgetg(3,t_VEC);
  rep=cgetg(4,t_VEC);
  
  d=denom(chic);
  rep[1]=lmul(d,chic);
  rep[2]=lexp(gdiv(gmul2n(gmul(gi,gpi),1),d),prec);
  rep[3]=(long)d;
  
  p1[1]=(long)rep;
  p1[2]=lcopy(prdiff);
  
  return gerepileupto(av,p1);
}

/* Soit dataCR une liste de caracteres et logelt log discret d'un
   ideal, calcule son image par tous ces caracteres */
static GEN
chiideal(GEN dataCR, GEN logelt)
{
  long j,cl;
  GEN rep;

  cl=lg(dataCR)-1;
  rep=cgetg(cl+1,t_VEC);
  for (j=1; j<=cl; j++)
    rep[j]=(long)ComputeImagebyChar(gmael(dataCR,j,5),logelt);
    
  return rep;
}

/********************************************************************/
/*                   1e partie : trouver le corps K                 */
/********************************************************************/

/* soient A un groupe abelien fini donne par sa matrice des relations
   et C definissant un sous-groupe, calcule le cardinal du quotient,
   sa structure et la matrice donnant la classe des generateurs de A
   en fonction des generateurs de Q */
static GEN
InitQuotient0(GEN DA, GEN C)
{
  long av=avma,i,l;
  GEN MQ,MrC,H,snf,snf2,rep,p1;

  l  = lg(DA)-1;
  
  if (gcmp0(C))
   H=DA;
  else
    H=C;
    
  MrC=gmul(ginv(H),DA);
  snf=smith2(hnf(MrC));
  MQ=concat(gmul(H,(GEN)snf[1]),DA);
  snf2=smith2(hnf(MQ));

  rep=cgetg(5,t_VEC);
  p1=cgetg(l+1,t_VEC);
  for (i=1; i<=l; i++) p1[i]=lcopy(gcoeff((GEN)snf2[3],i,i));
  if (lg((GEN)snf2[3])>1) rep[1] = (long)det((GEN)snf2[3]);
  else rep[1]=un;
  rep[2] = (long)p1;
  rep[3] = lcopy((GEN)snf2[1]); 
  rep[4] = lcopy(C);
   
  return gerepileupto(av,rep);
}

/* Soit m un module et C un sous-groupe modulo m, calcule les donnees
   necessaires pour calculer avec le quotient Clk(m)/C, a savoir 1)
   bnr, 2.1) cardinal quotient, 2.2) structure cyclique, 2.3) la
   matrice de surjection exprimant l'image des generateurs de Clk(m)
   en fonction de ceux du quotient et 2.4) le groupe C. */
static GEN
InitQuotient(GEN bnr, GEN C)
{
  long av=avma;
  GEN Mrm,dataquo;

  Mrm= diagonal(gmael(bnr,5,2));

  dataquo=cgetg(3,t_VEC);
  dataquo[1] = lcopy(bnr);
  dataquo[2] = (long)InitQuotient0(Mrm,C);
     
  return gerepileupto(av,dataquo);
}

/* Soient s: A->B avec s(GEN(A))=GEN(B)P, DA et DB resp. la matrice
   diag des relations de A et de B, et nbA, nbB resp. le rang de A et
   de B, calcule le noyau de s. Si DA=0 alors le groupe A est libre */
static GEN
ComputeKernel0(GEN P, GEN DA, GEN DB, long nbA, long nbB)
{
  long rk,av=avma;
  GEN herm,mask1,mask2,U;

  herm=hnfall(concat(P,DB));
  mask1=subis(shifti(gun,nbA),1);
  rk=nbA+nbB+1-lg((GEN)herm[1]);
  mask2=subis(shifti(gun,rk),1);
  U=matextract((GEN)herm[2],mask1,mask2);

  if (gcmp0(DA))
    return gerepileupto(av,hnf(U));
  else
    return gerepileupto(av,hnf(concat(U,DA)));
}

/* Soient m et n deux modules tels que n divise m, et soit C un groupe
   de congruence modulo n, calcule le groupe de congruence modulo m
   correspondant a C en calculant le noyau de Clk(m)->>Clk(n)/C */
static GEN
ComputeKernel(GEN bnrm, GEN dataC)
{
  long av=avma,i,nbm,nbq;
  GEN bnrn,Mrm,genm,Mrq,mgq,P;

  bnrn= (GEN)dataC[1];
  Mrm = diagonal(gmael(bnrm,5,2));
  genm= gmael(bnrm,5,3);
  nbm = lg(genm)-1;
  Mrq = diagonal(gmael(dataC,2,2));
  mgq = gmael(dataC,2,3);
  nbq = lg(mgq)-1;
  
  P=cgetg(nbm+1,t_MAT);
  for (i=1; i<=nbm; i++)
    P[i]=lmul(mgq,isprincipalray(bnrn,(GEN)genm[i]));

  return gerepileupto(av,ComputeKernel0(P,Mrm,Mrq,nbm,nbq));
}

/* Soit bnr un module et C un groupe de congruence modulo m, calcule
   les sous-groupes d'indice 2 de C comme sous-groupes de Clk(m) */
static GEN
ComputeIndex2Subgroup(GEN bnr, GEN C)
{
  long nbsbgrp,i,l,av=avma;
  GEN Mrc,snf,Mr,U,subgrp,rep,p1;

  Mr = diagonal(gmael(bnr,5,2));

  Mrc=gmul(ginv(C),Mr);
  snf=smith2(Mrc);
  U=ginv((GEN)snf[1]);
  l=lg((GEN)snf[3]);
  p1=cgetg(l,t_VEC);
  for (i=1; i<l; i++) p1[i]=mael3(snf,3,i,i);
  subgrp=subgrouplist0(p1,2,0,0);
  nbsbgrp=lg(subgrp)-1;
  
  rep=cgetg(nbsbgrp,t_VEC);
  for (i=2; i<=nbsbgrp; i++)
  {
    p1=hnf(concat(gmul(C,gmul(U,(GEN)subgrp[i])),diagonal(Mr)));
    rep[i-1]=(long)p1;
  }
  
  return gerepileupto(av,rep);
}

/* Let pr be a prime (pr may divide mod(bnr)), compute the indexes
   e,f of the splitting of pr in the class field nf(bnr/subgroup) */
GEN
GetIndex(GEN pr, GEN bnr, GEN subgroup, long prec)
{
  long av=avma,v,lg,i;
  GEN bnf,mod,mod0,mpr0,mpr,bnrpr,subpr,M,e,f,dtQ,p1,rep,cycpr,cycQ;
    
  bnf = (GEN)bnr[1];
  mod = gmael(bnr,2,1);
  mod0= (GEN)mod[1];
    
  /* Compute the part of mod coprime to pr */
  v=idealval(bnf,mod0,pr);
  mpr0=idealdivexact(bnf,mod0,idealpow(bnf,pr,stoi(v)));
  mpr=cgetg(3,t_VEC);
  mpr[1]=(long)mpr0;
  mpr[2]=mod[2];
  if (gegal(mpr0,mod0))
  {
    bnrpr=bnr;
    subpr=subgroup;
  }
  else
  {
    bnrpr=buchrayinitgen(bnf,mpr,prec);
    cycpr=gmael(bnrpr,5,2);
    M=GetSurjMat(bnr,bnrpr);
    M=gmul(M,subgroup);
    subpr=hnf(concat(M,diagonal(cycpr)));
  }

  /* e = #(bnr/subgroup) / #(bnrpr/subpr) */
  e=gdiv(det(subgroup),det(subpr));

  /* f = order of [pr] in bnrpr/subpr */
  dtQ=InitQuotient(bnrpr,subpr);
  p1=isprincipalray(bnrpr,pr);
  p1=gmul(gmael(dtQ,2,3),p1);
  cycQ=gmael(dtQ,2,2);
  lg=lg(cycQ)-1;
  f=gun;
  for (i=1; i<=lg; i++)
    f=glcm(f,gdiv((GEN)cycQ[i],ggcd((GEN)cycQ[i],(GEN)p1[i]))); 

  rep=cgetg(3,t_VEC);
  rep[1]=lcopy(e);
  rep[2]=lcopy(f);

  return gerepileupto(av,rep);
}

/* Soient f un module (sans ramification infinie et conducteur) et C
   un groupe de congruence modulo f donne par un dataquotient, calcule
   le couple (m,D) tel que m soit le conducteur du groupe de
   congruence D modulo m, f divise m, une seule place infinie ne
   divise pas m, D definit un sous-groupe d'indice 2 de C et m est de
   norme minimale.  Renvoie le groupe de rayon modulo m, le groupe d,
   le quotient de Clk(m)/D et le quotient de Clk(m)/C */
static GEN
FindModule(GEN dataC, long prec)
{
  long n,i,narch,nbp,maxnorm,minnorm,flag,N,nbidnn,s,c;
  long j,nbcand,av=avma,av1,limnorm;
  GEN bnr,rep,bnf,nf,f,vecun,arch,m,listid,idnormn,bnrm,ImC,candD,D;
  GEN bpr,indpr,sgp,p1;
  
  bnr= (GEN)dataC[1];
  sgp= gmael(dataC,2,4);
  bnf= (GEN)bnr[1];
  nf = (GEN)bnf[7];
  N  = degree((GEN)nf[1]);
  f  = gmael3(bnr,2,1,1);

  bpr=(GEN)idealfactor(nf,f)[1];
  nbp=lg(bpr)-1;
  indpr=cgetg(nbp+1,t_VEC);
  for (i=1; i<=nbp; i++)
  {
    p1=GetIndex((GEN)bpr[i],bnr,sgp,prec);
    indpr[i]=lmulii((GEN)p1[1],(GEN)p1[2]);
  }
    
  /* Initialisation des parties infinies du module possibles */     
  vecun=cgetg(N+1,t_VEC);
  for (i=1; i<=N; i++) vecun[i]=un;
   if (N==2) narch=1; /* si N=2, un seul cas est necessaire */
  else narch=N;
  arch=cgetg(narch+1,t_VEC);
  for (i=1; i<=narch; i++)
  {
    arch[i]=lcopy(vecun);
    mael(arch,i,N+1-i)=zero;
  }

  m=cgetg(3,t_VEC);
  maxnorm=50;
  minnorm=1;
  limnorm=200;
  flag=1;

  while(flag)
  {
    
    /* On calcule tous les ideaux de norm <= maxnorm */
    listid=ideallist(nf,maxnorm);
    av1=avma;
    for (n=minnorm; (n<=maxnorm)&&flag; n++)
    {
      if (DEBUGLEVEL>=2) fprintferr(" %ld",n);
      idnormn=(GEN)listid[n];
      nbidnn=lg(idnormn)-1;
      for (i=1; (i<=nbidnn)&&flag; i++)
      {
	avma=av1;

        /* la partie finie du comducteur */
	m[1]=(long)idealmul(nf,f,(GEN)idnormn[i]);
	for (s=1; (s<=narch)&&flag; s++)
	{
	  /* la partie infinie */
	  m[2]=arch[s];

          /* On calcule Clk(m) et on verifie que m est un conducteur */
	  bnrm=buchrayinitgen(bnf,m,prec);
	  if (gcmp1(conductor(bnr,gzero,-1,prec)))
	  {

	    /* On calcule le groupe correspodant a C dans Clk(m)... */
	    ImC=ComputeKernel(bnrm,dataC);

	    /* ...et ses sous-groupes d'indice 2 */
	    candD=ComputeIndex2Subgroup(bnrm,ImC);
	    nbcand=lg(candD)-1;
	    for (c=1; (c<=nbcand)&&flag; c++)
	    {

	      /* Est-ce que le groupe D est de conducteur m ? */
	      D=(GEN)candD[c];
	      flag=!itos(conductor(bnrm,D,-1,prec));
	      if (!flag)
		for (j=1; (j<=nbp) && !flag; j++)
		{
		  p1=GetIndex((GEN)bpr[j],bnrm,D,prec);
		  p1=mulii((GEN)p1[1],(GEN)p1[2]);
		  if (gegal(p1,(GEN)indpr[j])) flag=1;
		}
	    }
	  }
	}
      }
    }
    if (flag)
    {
      
      /* On rajoute les ideaux suivants */
      minnorm=maxnorm;
      maxnorm=maxnorm<<=1;

      if (maxnorm>limnorm)
	err(talker,"Cannot find a suitable module in FindModule");
    }
  }

  if (DEBUGLEVEL>=2) fprintferr("\n");
  
  rep=cgetg(5,t_VEC);
  rep[1]=lcopy(bnrm);
  rep[2]=lcopy(D);
  rep[3]=(long)InitQuotient((GEN)rep[1],(GEN)rep[2]);
  rep[4]=(long)InitQuotient((GEN)rep[1],ImC);
  return gerepileupto(av,rep);
}

/********************************************************************/
/*                    2e partie : calculer W(X)                     */
/********************************************************************/

/* Calcule la constante W(chi) telle que Ld(s,chi)=W(chi).Ld(1-s,chi*) */
static GEN
ComputeArtinNumber(GEN datachi, long prec)
{
  long av=avma,G,ms,j,i,nz,zcard,q,l,N;
  GEN chi,nc,dc,p1,cond0,cond1,elts,Msign,umod2,lambda,nf,sg,p2,chib,diff,vt;
  GEN z,idg,mu,idh,zid,zstruc,zgen,zchi,allclass,aclass,bnr,beta,s,tr,p3;

  chi  = (GEN)datachi[8];
  bnr  = (GEN)datachi[3];
  nf   = gmael(bnr,1,7);
  diff = gmael(nf,5,5);
  cond0= gmael3(bnr,2,1,1);
  cond1= gmael3(bnr,2,1,2);
  umod2= gmodulcp(gun,gdeux);
  N    = degree((GEN)nf[1]);
  
  /* Le cas trivial */
  if (cmpsi(2,(GEN)chi[3])>=0) return gerepileupto(av,gun);

  /* quelques initialisations supplementaires */
  G  = -bit_accuracy(prec)>>1;
  nc = idealnorm(nf,cond0);
  dc = idealmul(nf,diff,cond0);
  z = gexp(gdiv(gmul2n(gmul(gi,gpi),1),nc),prec);
  q=0; for (i=1; i<lg(cond1); i++) if (gcmp1((GEN)cond1[i])) q++;
  
  /* On calcule un systeme d'elements congru a 1 mod cond0 et donnant toutes
     les signatures possibles pour cond1 */
  p1=zarchstar(nf,cond0,cond1,q);
  elts =(GEN)p1[2];
  Msign=gmul((GEN)p1[3],umod2);
  ms=lg(elts)-1;
  /* for (i=1; i<=ms; i++) elts[i]=(long)basistoalg(nf,(GEN)elts[i]); */
  
  /* On trouve lambda dans diff.cond tel que (lambda.(diff.cond)^-1,cond0)=1
     et lambda>(cond1)> 0 */
  lambda=idealappr(nf,dc);
  sg=zsigne(nf,lambda,cond1);
  p2=lift(inverseimage(Msign,sg));
  for (j=1; j<=ms; j++)
    if (gcmp1((GEN)p2[j])) lambda=element_mul(nf,lambda,(GEN)elts[j]);
  idg=idealdivexact(nf,lambda,dc);

  /* On trouve mu dans idg tel que idh=(mu)/g est premier avec cond0
     et mu>(cond1)>0 */
  if (!gcmp1(gcoeff(idg,1,1)))
  {
    p1=idealfactor(nf,idg);
    p2=idealfactor(nf,cond0);
    l=lg((GEN)p2[1])-1;
    for (i=1; i<=l; i++) coeff(p2,i,2)=zero;
    p1=gtrans(concat(gtrans(p1),gtrans(p2)));
    mu=idealapprfact(nf,p1);
    sg=zsigne(nf,mu,cond1);
    p2=lift(inverseimage(Msign,sg));
    for (j=1; j<=ms; j++)
      if (gcmp1((GEN)p2[j])) mu=element_mul(nf,mu,(GEN)elts[j]);
    idh=idealdivexact(nf,mu,idg);
  }
  else
  {
    mu=gun;
    idh=gcopy(idg);
  }
  
  /* On calcule un systeme de generateurs de (Ok/cond0)^* qu'on rend
   positif en cond1 */
  zid=zidealstarinitgen(nf,cond0);
  zcard = itos(gmael(zid,2,1));
  zstruc= gmael(zid,2,2);
  zgen  = gmael(zid,2,3);
  nz=lg(zgen)-1;
  zchi=cgetg(nz+1,t_VEC);
  for (i=1; i<=nz; i++)
  {
    p1=(GEN)zgen[i];
    sg=zsigne(nf,p1,cond1);
    p2=lift(inverseimage(Msign,sg));
    for (j=1; j<=ms; j++)
      if (gcmp1((GEN)p2[j])) p1=element_mul(nf,p1,(GEN)elts[j]);
    aclass=isprincipalray(bnr,p1);
    zchi[i]=(long)ComputeImagebyChar(chi,aclass);
    zgen[i]=(long)p1;
  }  
    
  /* On fait la somme de chi(beta)*exp(2*i*pi*Tr(beta*mu/lambda) ou
     beta parcourt les classes de (Ok/cond0)^* avec beta >(cond1)> 0 */
  allclass=FindEltofGroup(zcard,zstruc);
  s=gzero;

  p3=cgetg(N+1,t_COL);
  for (i=1; i<=N; i++) p3[i]=zero;
  vt=cgetg(N+1,t_VEC);
  for (i=1; i<=N; i++)
  {
    p3[i]=un;
    vt[i]=ltrace(basistoalg(nf,p3));
    p3[i]=zero;
  }
    
  for (i=1; i<=zcard; i++)
  {
    beta=gun;
    chib=gun;
    p1=(GEN)allclass[i];
    for (j=1; j<=nz; j++)
    {
      p2=element_pow(nf,(GEN)zgen[j],(GEN)p1[j]);
      beta=element_mul(nf,beta,p2);
      chib=gmul(chib,gpuigs((GEN)zchi[j],itos((GEN)p1[j])));
    }
    beta=element_mul(nf,beta,mu);
    beta=element_div(nf,beta,lambda);
    tr=gmul(vt,beta);
    tr=gmod(gmul(tr,nc),nc);
    s=gadd(s,gmul(chib,gpuigs(z,itos(tr))));
  }
  aclass=isprincipalray(bnr,idh);
  s=gmul(s,ComputeImagebyChar(chi,aclass));
  s=gdiv(s,gsqrt(nc,prec));

  p1=gsubgs(gabs(s,prec),1);
  i=expo(p1);

  if (i>G) err(bugparier,"ComputeArtinNumber");
      
  return gerepileupto(av,gmul(s,gpuigs(gneg(gi),q)));
}

/* Calcule la constante W de l'equation fonctionnelle de la fonction
   Lambda associe au caractere chi defini sur bnr, si flag=1, le
   caractere est considere comme primitif */
GEN
bnrrootnumber(GEN bnr, GEN chi, long flag, long prec)
{
  long av=avma,l,i;
  GEN cond,condc,bnrc,chic,cyc,d,p1,p2,dtcr;
  
  checkbnr(bnr);
  cond = gmael(bnr,2,1);
  l = lg(gmael(bnr,5,2));
    
  if (typ(chi)!=t_VEC || lg(chi)!=l)
    err(talker,"incorrect character in bnrrootnumber");

  if (!flag)
  {
    condc=bnrconductorofchar(bnr,chi,prec);
    if (gegal(cond,condc)) flag=1;
  }
  else condc=cond;
  
  if (flag)
    bnrc=bnr;
  else
    bnrc=buchrayinitgen((GEN)bnr[1],condc,prec);

  chic= cgetg(l,t_VEC);
  cyc = gmael(bnr,5,2);
  for (i=1; i<l; i++) chic[i]=ldiv((GEN)chi[i],(GEN)cyc[i]);
  d = denom(chic);
  p2= cgetg(4,t_VEC);
  p2[1]=lmul(d,chic);
  p2[2]=lexp(gdiv(gmul2n(gmul(gi,gpi),1),d),prec); 
  if (gegal(d,gdeux)) p2[2]=lreal((GEN)p2[2]);
  p2[3]=(long)d;
  
  dtcr=cgetg(9,t_VEC);  
  dtcr[1]=(long)chi;
  dtcr[2]=zero;
  dtcr[3]=(long)bnrc;
  dtcr[4]=(long)bnr;
  dtcr[5]=(long)p2;
  dtcr[6]=zero;
  dtcr[7]=(long)condc;
  p1=GetPrimChar(chi,bnr,bnrc,prec);

  if(!p1) dtcr[8]=(long)p2;
  else dtcr[8]=p1[1];
  
  return gerepileupto(av,ComputeArtinNumber(dtcr,prec));
}


/********************************************************************/
/*               3e partie : initialiser les caracteres             */
/********************************************************************/

/* Releve le caractere chi en un caractere sur le groupe de structure
   cyc avec Mat matrice de surjection */
static GEN
LiftChar(GEN cyc, GEN Mat, GEN chi)
{
  long lm,l,i,j,av;
  GEN lchi,s;

  lm= lg(cyc)-1;
  l = lg(chi)-1;

  lchi=cgetg(lm+1,t_VEC);
  for (i=1; i<=lm; i++)
  {
    av=avma;
    s=gzero;
    for (j=1; j<=l; j++)
      s=gadd(s,gmul((GEN)chi[j],gcoeff(Mat,j,i)));
    lchi[i]=(long)gerepileupto(av,gmod(gmul(s,(GEN)cyc[i]),(GEN)cyc[i]));
  }
  
  return lchi;  
}

/* Soit chi un caractere de condcteur cond(bnr), calcule la constante
   A(chi) associe aux premiers divisant diff */
static GEN
ComputeAChi(GEN dtcr)
{
  long l,i;
  GEN ray,A,diff,chi,bnrc;

  diff= (GEN)dtcr[6];
  bnrc= (GEN)dtcr[3];
  chi = (GEN)dtcr[8];
  l   = lg(diff)-1;
  
  A=gun;
  for (i=1; i<=l; i++)
  {
    ray=isprincipalray(bnrc,(GEN)diff[i]);
    A=gmul(A,gsub(gun,ComputeImagebyChar(chi,ray))); 
  }

  return A;
}
  
/* Un caractere chi est donnee par un vecteur (c_i) de nombres entiers
   et une racine de l'unite z d'ordre d tels que si (a_i) est le
   vecteur des log discret de A dans bnr alors chi(A)=z^s ou
   s=sum(c_i*a_i) (note : ce n'est pas la reprensentation usuelle). Un
   vecteur dataCHAR contient 1) chi represente dans bnr(m) (format
   usuel), 2) constante C(chi), 3) bnr(cond(chi)), 4) bnr, 5)
   l'expression de chi mentionne ci-dessus, i.e. un vecteur
   [(c_i),z,d], 6) le vecteur des premiers qui divisent m mais pas
   cond(chi) 7) la partie finie du conducteur cond(chi), 8) le
   caractere irreductible associe a chi. */
static GEN
InitChar(GEN dataD, long prec)
{
  long nbg,hD,h,lD,i,nc,tnc,j,N,av=avma,fl,tetpil;
  GEN bnr,Mr,mod0,bnf,MrD,Surj,p1,dataCR,chi,lchi,cond,bnrc;
  GEN chic,dk,C,d,p2,p3,setchar,toto;
    
  bnr = (GEN)dataD[1];
  Mr  = gmael(bnr,5,2);
  mod0= gmael3(bnr,2,1,1); 
  nbg = lg(Mr)-1;
  bnf = (GEN)bnr[1];
  hD  = itos(gmael(dataD,2,1));
  h   = hD>>1;
  MrD = gmael(dataD,2,2);
  Surj= gmael(dataD,2,3);
  lD  = lg(MrD)-1;
  dk  = gmael(bnf,7,3);
  N   = degree(gmael(bnf,7,1));

  setchar=gtoset(cgetg(1,t_VEC));
  toto=cgetg(3,t_VEC); toto[1]=toto[2]=zero;
  C=gsqrt(gdiv(dk,gpuigs(gpi,N)),prec);
  p1=FindEltofGroup(hD,MrD);
  dataCR=cgetg(h+1,t_VEC);
  for (i=1; i<=h ;i++) dataCR[i]=lgetg(9,t_VEC);
  nc=1;
  tnc=1;

  /* tnc est le nombre de caracteres initialises */
  for (i=1; tnc<=h; i++)
  {

    /* On releve un caractere de D dans Clk(m) */
    chi=gcopy((GEN)p1[i]);
    for (j=1; j<=lD; j++) chi[j]=ldiv((GEN)chi[j],(GEN)MrD[j]);
    lchi=LiftChar(Mr,Surj,chi);

    /* On determine son conducteur */
    if (setsearch(setchar,lchi,0)) cond=toto;
    else
      cond=bnrconductorofchar(bnr,lchi,prec);

    /* si la partie infinie est non triviale... */
    if (!gcmp0((GEN)cond[2]))
    {

      /* Est-ce que les invariants de ce caractere sont deja connus ? */
      fl=0;
      for (j=1; (j<nc)&&!fl; j++)
	if (gegal((GEN)cond[1],gmael(dataCR,j,7))) fl=j;
      
      if (!fl&&gegal((GEN)cond[1],mod0))
      {
	mael(dataCR,nc,2)=lmul(C,gsqrt(det((GEN)cond[1]),prec));
	mael(dataCR,nc,3)=lcopy(bnr);
	mael(dataCR,nc,4)=(long)idmat(nbg);
	mael(dataCR,nc,6)=lgetg(1,t_VEC);
	mael(dataCR,nc,7)=lcopy(mod0);
	fl=nc;
      }

      /* Constante C(chi) */
      mael(dataCR,nc,1)=(long)lchi;
      if (!fl)
	mael(dataCR,nc,2)=lmul(C,gsqrt(det((GEN)cond[1]),prec));
      else
	mael(dataCR,nc,2)=mael(dataCR,fl,2);
      
      /* bnr du conducteur de chi */
      if (!fl)
     	bnrc=buchrayinitgen(bnf,cond,prec);
      else
	bnrc=gmael(dataCR,fl,3);

      mael(dataCR,nc,3)=(long)bnrc;

      /* Clk(m) sur lequel le caractere sera defini */
      mael(dataCR,nc,4)=(long)bnr;

      /* On calcule la representation explique ci-dessus de chi */
      chic=cgetg(nbg+1,t_VEC);
      for (j=1; j<=nbg; j++)
	chic[j]=ldiv((GEN)lchi[j],(GEN)Mr[j]);      
      d=denom(chic);

      p2=cgetg(4,t_VEC);
      p2[1]=lmul(d,chic);
      p2[2]=lexp(gdiv(gmul2n(gmul(gi,gpi),1),d),prec); 
      p2[3]=(long)d;
      mael(dataCR,nc,5)=(long)p2;

      /* Difference entre cond(bnr) et cond(chi) et caractere primitif
         associe */
      mael(dataCR,nc,7)=cond[1];
      p3=GetPrimChar(lchi,bnr,bnrc,prec);
      if (p3)
      {	
	mael(dataCR,nc,6)=p3[2];
	mael(dataCR,nc,8)=p3[1];
      }
      else
      {
	mael(dataCR,nc,6)=lgetg(1,t_VEC);
	mael(dataCR,nc,8)=(long)p2;
      }
	
      /* On ajoute le charactere conjugue si le caractere n'est pas reel */
      if (!gegal(d,gdeux))
      {
	p3=cgetg(nbg+1,t_VEC);
	for (j=1; j<=nbg; j++) p3[j]=lsub((GEN)Mr[j],(GEN)lchi[j]);
	p2=cgetg(2,t_VEC); p2[1]=(long)p3;
	setchar=setunion(setchar,p2);
	tnc++;
      }      

      nc++;tnc++;
      p2=cgetg(2,t_VEC); p2[1]=(long)lchi;
      setchar=setunion(setchar,p2);
    }
  }

  setlg(dataCR,nc);
  tetpil=avma;
  return gerepile(av,tetpil,gcopy(dataCR));    
}

/* reactualise les donnees de dataCR avec la nouvelle precision */
static GEN
CharNewPrec(GEN dataCR, GEN nf, long prec)
{
  GEN dk,C,p1;
  long av=avma,N,l,j;
  
  dk= (GEN)nf[3];
  N = degree((GEN)nf[1]);
  l = lg(dataCR)-1;
  constpi(prec+1);
  
  C=gsqrt(gdiv(dk,gpuigs(gpi,N)),prec);
  for (j=1; j<=l; j++)
  {
    mael(dataCR,j,2)=lmul(C,gsqrt(det(gmael(dataCR,j,7)),prec));
    mael4(dataCR,j,3,1,7)=lcopy(nf);
    p1=gmael(dataCR,j,5);
    p1[2]=lexp(gdiv(gmul2n(gmul(gi,gpi),1),(GEN)p1[3]),prec);
    p1=gmael(dataCR,j,8);
    p1[2]=lexp(gdiv(gmul2n(gmul(gi,gpi),1),(GEN)p1[3]),prec);
  }
    
  return gerepileupto(av,gcopy(dataCR));
}  

/********************************************************************/
/*            4e partie : Calcul des coefficients an(chi)           */
/********************************************************************/

/* Corrige les coefficients an(chi) en rajoutant les ideaux premiers
   avec cond(chi) mais pas avec cond(bnr) */
static void
CorrectCoeff(GEN dtcr, GEN an)
{
  long nmax,lg,av1,j,p,q,limk,k,av=avma;  
  GEN chi,bnrc,diff,an2,ray,ki,ki2,pr;
    
  chi = (GEN)dtcr[8];
  bnrc= (GEN)dtcr[3];
  diff= (GEN)dtcr[6];
  nmax= lg(an)-1;
  lg  = lg(diff)-1;

  if (DEBUGLEVEL>2)
  {  
    fprintferr("Correction du charactere de conducteur ");
    output((GEN)dtcr[7]);
  }

  av1=avma;
  if (!lg)
    return;
  else
  {
    for(j=1; j<=lg; j++)
    {
      an2=gcopy(an);
      pr=(GEN)diff[j];
      ray=isprincipalray(bnrc,pr);
      ki=ComputeImagebyChar(chi,ray);
      p=itos(gpui((GEN)pr[1],(GEN)pr[4],0));
      q=p;limk=nmax/q;
      ki2=gcopy(ki);
      while(q<=nmax)
      {
	for(k=1;k<=limk;k++)
	  gaffect(gadd((GEN)an[k*q],gmul((GEN)an2[k],ki2)),(GEN)an[k*q]);
	q*=p;limk/=p;
	ki2=gmul(ki2,ki);
      }
      avma=av1;
    }
  }

  avma=av;
  return;
}


/*******************************************************************/
/*******************************************************************/
/*                                                                 */
/*                   Corps de classes avec unites                  */
/*               de Stark pour les corps quadratiques reels        */
/*                                                                 */
/*******************************************************************/
/*******************************************************************/


/* Appel de la fonction FindModule pour quadhilbertreal, renvoie
   Cl(m), le groupe D, le quotient Cl(m)/D, le InitChar(Cl(m)/D) et le
   quotient Cl(m)/C */
GEN
FindModule0(GEN bnf, long prec)
{
  GEN bnr,dataC,p1,rep;

  bnr=buchrayinitgen(bnf,gun,prec);
  dataC=InitQuotient(bnr,gzero);

  rep=cgetg(6,t_VEC);
  p1=FindModule(dataC,prec);
  rep[1]=p1[1];rep[2]=p1[2];
  rep[3]=p1[3];rep[5]=p1[4];
  rep[4]=(long)InitChar((GEN)rep[3],prec);

  return rep;
}

/* Determination des fonctions L' pour QuadStark */
static GEN
getLprime(GEN dataCR, GEN V1, GEN V2, long prec)
{
  long i,cl,av=avma;
  GEN W,Lp,A;

  cl  = lg(dataCR)-1;
  
  Lp=cgetg(cl+1,t_VEC);
  for (i=1; i<=cl; i++)
  {
    W=ComputeArtinNumber((GEN)dataCR[i],prec);
    A=ComputeAChi((GEN)dataCR[i]);
    Lp[i]=lmul(A,gadd((GEN)V2[i],gmul(W,(GEN)V1[i])));
  }

  if (DEBUGLEVEL) msgtimer("Compute W");
   
  return gerepileupto(av,gcopy(Lp));
}


/* Calcule les coefficients de la serie de Dirichlet de toutes les
   fonctions L correspondant aux caracteres de dtcr */
static GEN  
computean(GEN bnrh, GEN dtcr, long nmax, long prec)
{
  long i,j,cl,q,cp,al,v1,v2,v,fldiv,av,av1;
  GEN bnf,ideal,dk,idno,matan,p1,prime,chi,qg,chi1,chi2;
  GEN chi11,chi12,bnr,pr,pr1,pr2,xray,xray1,xray2,p2;
  byteptr dp = diffptr;

  /* On prepare la matrice des coefficients */
  cl=lg(dtcr)-1;
  matan=cgetg(cl+1,t_MAT);
  for(j=1; j<=cl; j++)
  {
    p1=cgetg(nmax+1,t_COL); matan[j]=(long)p1;
    for(i=1; i<=nmax; i++)
    {
      p2=cgetg(3,t_COMPLEX);
      p2[1]=lgetr(prec);
      p2[2]=lgetr(prec);
      gaffect(gun,p2);
      p1[i]=(long)p2;
    }
  }

  av=avma;
  bnr=(GEN)bnrh[1]; 
  bnf=(GEN)bnr[1]; ideal=gmael3(bnr,2,1,1);
  idno=idealnorm(bnf,ideal);
  dk=gmael(bnf,7,3);

  prime=stoi(2); dp++;
  av1=avma;
  while (*dp && prime[2]<=nmax)
  {

    qg=prime; al=1;
    switch (krogs(dk,prime[2]))
    {
      
      /* Cas ou le premier est inerte */
      case -1:
	fldiv=divise(idno,prime);
	if (!fldiv)
	{
	  xray=isprincipalray(bnr,prime);
	  chi=chiideal(dtcr,xray);
	  chi1=gcopy(chi);
	}
       	while(cmpis(qg,nmax)<=0)
	{
	  q=qg[2];
	  for(cp=1,i=q; i<=nmax; i+=q,cp++)
	    if(cp%prime[2])
	    {
	      if (fldiv || al%2)
                for(j=1; j<=cl; j++)
		  gaffect(gzero,gcoeff(matan,i,j));
	      else
		for(j=1; j<=cl; j++) 
		  gaffect(gmul(gcoeff(matan,i,j),(GEN)chi[j]),gcoeff(matan,i,j));
	    }
	  qg=mulii(qg,prime); al++;
	  if ((al%2) && !fldiv)
	    for (j=1; j<=cl; j++) chi[j]=lmul((GEN)chi[j],(GEN)chi1[j]);
 	}
	break;
	
      /* Cas ou le premier est ramifie */
      case 0:
	fldiv=divise(idno,prime);
	if (!fldiv)
	{
	  pr=(GEN)primedec(bnf,prime)[1];
	  xray=isprincipalray(bnr,pr);
	  chi=chiideal(dtcr,xray);
	  chi2=gcopy(chi);
	}
	while(cmpis(qg,nmax)<=0)
	{
	  q=qg[2];
	  for(cp=1,i=q; i<=nmax; i+=q,cp++)
	    if(cp%prime[2])
	      {
		if (fldiv)
                  for(j=1; j<=cl; j++)
		    gaffect(gzero,gcoeff(matan,i,j));
		else
		{
		  for(j=1; j<=cl; j++)
		    gaffect(gmul((GEN)chi[j],gcoeff(matan,i,j)),gcoeff(matan,i,j));
		}
	      }
	  qg=mulii(qg,prime); al++;
	  if((cmpis(qg,nmax)<=0)&&(!fldiv))
	    for (j=1; j<=cl; j++)
	      chi[j]=lmul((GEN)chi[j],(GEN)chi2[j]);
	}
	break;
	
     /* Cas ou le premier est decompose */
      case 1:
	p1=primedec(bnf,prime);
        pr1=(GEN)p1[1]; v1=idealval(bnf,ideal,pr1);
        pr2=(GEN)p1[2]; v2=idealval(bnf,ideal,pr2);
	if (v1+v2==0)
	{
	  xray1=isprincipalray(bnr,pr1);
	  chi11=chiideal(dtcr,xray1);
	  xray2=isprincipalray(bnr,pr2);
	  chi12=chiideal(dtcr,xray2);
	  chi1=gadd(chi11,chi12); chi2=gcopy(chi12);
	  while(cmpis(qg,nmax)<=0)
	  {
	    q=qg[2];
	    for(cp=1,i=q; i<=nmax; i+=q,cp++)
	      if(cp%prime[2])
		for(j=1; j<=cl; j++)
		  gaffect(gmul((GEN)chi1[j],gcoeff(matan,i,j)),gcoeff(matan,i,j));
	    qg=mulii(qg,prime); al++;
	    if(cmpis(qg,nmax)<=0)
	      for (j=1; j<=cl; j++)
	      {
		chi2[j]=lmul((GEN)chi2[j],(GEN)chi12[j]);
		chi1[j]=ladd((GEN)chi2[j],gmul((GEN)chi1[j],(GEN)chi11[j]));
	      }
	  }
	}
	else
	{
	  if (v1) {v=v2; pr=pr2; } else {v=v1; pr=pr1; }
	  if (v==0)
	  {
	    xray=isprincipalray(bnr,pr);
	    chi1=chiideal(dtcr,xray);
	    chi=gcopy(chi1);
	  }
	  while(cmpis(qg,nmax)<=0)
	  {
	    q=qg[2];
	    for(cp=1,i=q; i<=nmax; i+=q,cp++)
	      if(cp%prime[2])
              {
		if (v)
                  for(j=1; j<=cl; j++)
		    gaffect(gzero,gcoeff(matan,i,j));
		else
		  for(j=1; j<=cl; j++)
		    gaffect(gmul((GEN)chi[j],gcoeff(matan,i,j)),gcoeff(matan,i,j));
              }
	    qg=mulii(qg,prime); al++;
	    if (!v && (cmpis(qg,nmax)<=0))
	      for (j=1; j<=cl; j++) chi[j]=lmul((GEN)chi[j],(GEN)chi1[j]);
	  }
	}
	break;
    }
    prime[2] += (*dp++);
    avma=av1;
  }

  /*  matan2=cgetg(cl+1,t_MAT); */
  for (i=1; i<=cl; i++)
    CorrectCoeff((GEN)dtcr[i],(GEN)matan[i]);
  
  return gerepileupto(av,matan);
}


/* Calcul les derivees des fonctions zeta correspondant a bnrh */
static GEN
getallzeta(GEN bnrh, long prec)
{
  long av=avma,n,i,j,nn[100],nmax,cl,h,av1,test,k;
  GEN C,p1,matan,c2,cexp,cn,cn2,veclprime,veclprime1;
  GEN veclprime2,veczeta,sig,valchi,subgroup,dtcr,cond;

  constpi(prec+1);
  subgroup=(GEN)bnrh[2]; dtcr=(GEN)bnrh[4];
  cl=lg(dtcr)-1;C=cgetg(cl+1,t_VEC);
  cond=cgetg(cl+1,t_VEC);
  h=itos(gmul2n(det(subgroup),-1));
  nmax=0;
  for (j=1; j<=cl; j++)
  {
    C[j]=mael(dtcr,j,2);
    cond[j]=mael(dtcr,j,7);
    nn[j]=(bit_accuracy(prec)*gtodouble((GEN)C[j])*0.35);
    if (nn[j]>nmax) nmax=nn[j];
  }
  if(DEBUGLEVEL>=2) fprintferr("nmax = %ld\n",nmax);
  
  /* On calcule les coefficients de L(s,chi) */
  matan=computean(bnrh,dtcr,nmax,prec);
  if (DEBUGLEVEL) msgtimer("Compute an");

  /* puis, les termes f2(C/n) */
  veclprime1=cgetg(cl+1,t_VEC);
  for (j=1; j<=cl; j++)
  {
    p1=cgetg(3,t_COMPLEX);
    p1[1]=lgetr(prec);
    p1[2]=lgetr(prec);
    gaffect(gzero,p1);
    veclprime1[j]=(long)p1;
  }
  c2=cgetg(cl+1,t_VEC); cexp=cgetg(cl+1,t_VEC);
  cn=cgetg(cl+1,t_VEC); cn2=cgetg(cl+1,t_VEC);
  for (j=1; j<=cl; j++)
  {
    c2[j]=ldivsg(2,(GEN)C[j]);
    cexp[j]=lexp(gneg((GEN)c2[j]),prec);
    cn[j]=lgetr(prec); affir(gun,(GEN)cn[j]);
    cn2[j]=lgetr(prec);
  }

  av1=avma;
  for (j=1; j<=cl; j++)
  {
    for (n=1; n<=nn[j]; n++)
    {
      affrr(gmul((GEN)cn[j],(GEN)cexp[j]),(GEN)cn[j]);
      affrr(gdivgs((GEN)cn[j],n),(GEN)cn2[j]);
      gaffect(gadd((GEN)veclprime1[j],gmul((GEN)cn2[j],gcoeff(matan,n,j))),
	      (GEN)veclprime1[j]);
      avma=av1;
    }
  }
  for (j=1; j<=cl; j++)
    gaffect(gmul2n(gconj(gmul((GEN)veclprime1[j],(GEN)C[j])),-1),
	    (GEN)veclprime1[j]);

  if (DEBUGLEVEL) msgtimer("Compute V1");

  /* et les termes f1(C/n) */
  veclprime2=cgetg(cl+1,t_VEC);
  for (j=1; j<=cl; j++)
  {
    p1=cgetg(3,t_COMPLEX);
    p1[1]=lgetr(prec);
    p1[2]=lgetr(prec);
    gaffect(gzero,p1);
    veclprime2[j]=(long)p1;
  }
  p1=cgetg(cl+1,t_VEC);
  for (j=1; j<=cl; j++)
  {
    test=0;
    for (k=1; k<j && !test; k++)
      if (gegal((GEN)cond[j],(GEN)cond[k])) test=k;
    if (test) p1[j]=p1[test];
    else p1[j]=(long)veceint1(stoi(nn[j]),(GEN)c2[j],prec);

    av1=avma;
    for (n=1; n<=nn[j]; n++)
      gaffect(gadd((GEN)veclprime2[j],gmul(gmael(p1,j,n),gcoeff(matan,n,j))),
	      (GEN)veclprime2[j]);
    avma=av1;
  }
  if (DEBUGLEVEL) msgtimer("Compute V2");

  /* On en deduit les termes L'(0,chi) */
  veclprime=getLprime(dtcr,veclprime1,veclprime2,prec);

  /* et z'(0,sigma) pour un systeme de representants */
  veczeta=cgetg(h+1,t_VEC);
  p1=ComputeLift(gmael(bnrh,5,2));
  for (i=1; i<=h; i++)
  {
    veczeta[i]=zero;
    sig=(GEN)p1[i];
    valchi=chiideal(dtcr,sig);
    for (j=1; j<=cl; j++)
    {
      if (gegal(gdeux,gmael3(dtcr,j,5,3)))
	veczeta[i]=ladd((GEN)veczeta[i],gmul((GEN)veclprime[j],(GEN)valchi[j]));
      else
	veczeta[i]=ladd((GEN)veczeta[i],gmul2n(greal(gmul((GEN)veclprime[j],(GEN)valchi[j])),1));
    }
    veczeta[i]=ldivgs((GEN)veczeta[i],2*h);
  }
  
  return gerepileupto(av,greal(veczeta));
}

/* Reconnaissance d'un entier quadratique par recherche exhaustive */
static GEN
recbeta2(GEN nf, GEN beta, GEN bound, long prec)
{
  long av=avma,av2,tetpil,i,range,G,e,m;
  GEN om,om1,om2,dom,p1,a,b,rom,bom2,*gptr[2];

  G=min(-32,-bit_accuracy(prec)>>4);

  if (DEBUGLEVEL>1)
  { fprintferr("Entree dans recbeta2 avec B = "); outerr(bound); }
  if (DEBUGLEVEL>3)
    fprintferr("\n Precision demandee : %ld",G);

  om = gmael(nf,7,2);
  rom= (GEN)nf[6];
  om1= poleval(om,(GEN)rom[1]);
  om2= poleval(om,(GEN)rom[2]);
  dom= subrr(om1,om2);

  /* b va parcourir b -> b + range */
  p1 = gaddgs(gmul2n(gceil(absr(divir(bound,dom))),1),2);
  range = VERYBIGINT;
  if (cmpis(p1, VERYBIGINT)<0) range=itos(p1);
  
  av2=avma;
  b=gdiv(gsub(bound,beta),dom);
  if (gsigne(b)<0) b=subis(negi(gcvtoi(gneg(b),&e)),1);
  else b=gcvtoi(b,&e);
  if (e>0) {avma=av; return NULL;}
  bom2 = mulir(b,om2);
  m=0;
  for (i=0; i<=range; i++)
  {

    /* pour chaque b, on construit un a candidat qu'on teste */
    a = grndtoi(gsub(beta,bom2),&e);
    if (e>0) {avma=av; return NULL;}
    p1 = gsub(mpadd(a,bom2), beta);
   
    if ( DEBUGLEVEL>3 && expo(p1)<m )
    { m=expo(p1); fprintferr("\n Precision trouvee : %ld",expo(p1)); }
    if (gcmp0(p1) || expo(p1) < G)
    {
      p1 = gadd(a, gmul(b,om));
      return gerepileupto(av,gmodulcp(p1, (GEN)nf[1]));
    }
    tetpil=avma;
    b=gaddgs(b,1); bom2=gadd(bom2,om2);
    gptr[0]=&b; gptr[1]=&bom2;
    gerepilemanysp(av2,tetpil,gptr,2);
  }

  /* si la reconnaissance a echoue */
  return NULL;
}

/* Soit polrel le polynome relatif du corps de classes de Hilbert
   de bnf, redescend ce polynome sur Q */
static GEN
makescind(GEN nf, GEN polabs, long cl, long prec)
{
  long av=avma,i,debkeep=DEBUGLEVEL,fl=0;
  GEN pol,p1,nf2,dabs,dk;

  if (DEBUGLEVEL==1) DEBUGLEVEL=0;

  /* on verifie (un peu) la construction en verifiant la signature et
     le discriminant */
  dabs=discf(polabs); dk=(GEN)nf[3];
  if (!gegal(dabs,gpuigs(dk,cl)) || sturmpart(polabs,NULL,NULL)!=2*cl)
    err(bugparier,"quadhilbert");

  /* on cherche les sous-corps par polred... */
  p1=polred(polabs,(prec<<1)-2);
  for (i=1; i<lg(p1) && !fl; i++)
  {
    pol=(GEN)p1[i];
    if (degree(pol)==cl)
    {
      if (cl%2) fl=1;
      if (!gegal(sqri(discf(pol)),dabs)) fl=1;
    }
  }
  if (debkeep) msgtimer("polred");

  /* ... puis par nfsubfields */
  if (!fl)
  {
    nf2=nfinit0(polabs,1,prec);
    p1=subfields(nf2,stoi(cl));
    
    if (debkeep) msgtimer("subfields");
    
    for (i=1; i<lg(p1) && !fl; i++)
    {
      pol=gmael(p1,i,1);
      if (cl%2) fl=1;
      if (!gegal(sqri(discf(pol)),(GEN)nf2[3])) fl=1;
    } 

    for (i=1; i<lg(p1) && !fl; i++)
    {
      pol=gmael(p1,i,1);
      if (degree(gcoeff(nffactor(nf,pol),1,1))==cl) fl=1;
    }
  }

  if (fl)
  {
    pol=polredabs(pol,prec);
    DEBUGLEVEL=debkeep;
    return gerepileupto(av,pol);
  }

  err(talker,"bug in makescind: no polynomial found");
  return NULL; /* not reached */
}

/* Calcul le corps de classes de Hilbert par la theorie du corps de
   classes de genre */
static GEN
GenusField(GEN bnf, long prec)
{
  long hk,c,l,av=avma;
  GEN disc,quat,x2,pol,div,d;
  
  hk  = itos(gmael3(bnf,8,1,1));
  disc= gmael(bnf,7,3);
  quat= stoi(4);
  x2  = gsqr(polx[0]);

  if (gcmp0(modii(disc,quat))) disc=divii(disc,quat);

  div=divisors(disc);
  c=1; l=0;
  while(l<hk)
  {
    c++;
    d=(GEN)div[c];
    if (gcmp1(modii(d,quat)))
    {
      if (!l) pol=gsub(x2,d);
      else pol=(GEN)compositum(pol,gsub(x2,d))[1];
      l=degree(pol);
    }
  }

  return gerepileupto(av,polredabs(pol,prec));
}

/* Part specific to the quadratic case, if flag = 0 returns the
   reduced polynomial, flag = 1 returns the non-reduced polynomial,
   flag = 2 returns an absolute reduced polynomial, flag = 3 returns
   the polynomial of the Stark's unit */
static GEN
QuadStark(GEN data, GEN nf, long flag, long newprec)
{
  long cl,j,cpt=0,av;
  GEN veczeta,p1,polrelnum,polrel,bound,B=stoi(500000),beta,*gptr[2];

LABDOUB:

  av=avma;
  veczeta=getallzeta(data,newprec);
  if (DEBUGLEVEL>=2) { fprintferr("zetavalues = "); outerr(veczeta); }

  cl=lg(veczeta)-1;
  polrelnum=gun;
  if (flag<3)
  {
    for (j=1; j<=cl; j++)
    {
      p1=gexp(gmul2n((GEN)veczeta[j],1),newprec);
      polrelnum=gmul(polrelnum,gsub(polx[0],gadd(p1,ginv(p1))));
    }
  }
  else
  {
    for (j=1; j<=cl; j++)
    {
      p1=gexp(gmul2n((GEN)veczeta[j],1),newprec);
      polrelnum=gmul(polrelnum,gsub(polx[0],p1));
      polrelnum=gmul(polrelnum,gsub(polx[0],ginv(p1)));
    }
    cl*=2;
  }
  if (DEBUGLEVEL>=2) { fprintferr("polrelnum = "); outerr(polrelnum); }
  if (DEBUGLEVEL) msgtimer("Compute polnum");

  polrel=gcopy(polrelnum);
  for (j=2; j<=cl+1; j++)
  {
    bound=binome(stoi(cl),j-2);
    if (flag<3) bound=shifti(bound,cl+2-j);
    beta=greal((GEN)polrelnum[j]);
    p1=NULL;
    
    if (gcmp(bound,B)>0) p1=recbeta2(nf,beta,B,newprec);
    if (!p1) p1=recbeta2(nf,beta,bound,newprec);
      
    if (!p1)
    {
      cpt++;
      if (cpt>=5) err(talker,"insufficient precision in quadhilbert");

      beta=vecmax(gabs(gtovec(polrelnum),newprec));
      p1=gceil(gdiv(mplog(beta),dbltor(2.3026)));
      newprec=max(1.2*newprec,(long)(itos(p1)*pariK1+7)); 
      
      if (DEBUGLEVEL) err(warnprec,"QuadStark",newprec);
      nf=nfnewprec(nf,newprec);
      data[4]=(long)CharNewPrec((GEN)data[4],nf,newprec);
      gptr[0]=&data; gptr[1]=&nf;
      gerepilemany(av,gptr,2);
      goto LABDOUB;
    }
    polrel[j]=(long)p1;
  }
  if (DEBUGLEVEL>=2) { fprintferr("polrel = "); outerr(polrel); }
  if (DEBUGLEVEL) msgtimer("Recpolnum");

  if (!flag) return gerepileupto(av,rnfpolredabs(nf,polrel,0,newprec));
  if (flag!=2) return gerepileupto(av,gcopy(polrel));
  return gerepileupto(av,rnfpolredabs(nf,polrel,2,newprec));
}

/********************************************************************/
/*                     Programmes principaux                        */
/********************************************************************/

/* Calcul le polynome sur Q du corps de classes de
   Hilbert de Q(sqrt(D)) ou D est un discriminant fondamental positif */
GEN
quadhilbertreal(GEN D, long prec)
{
  long av=avma,cl,debkeep=DEBUGLEVEL,newprec;
  GEN pol,bnf,bnrh,p1,nf,exp;

  DEBUGLEVEL=0; timer2(); constpi(prec+1);
  cl=itos((GEN)quadclassunit0(D,0,NULL,prec)[1]);
  if (cl==1) { avma=av; DEBUGLEVEL=debkeep; return polx[0]; }

  pol=quadpoly(D); lisexpr("y"); setvarn(pol,1);
  bnf=bnfinit0(pol,1,NULL,prec); nf=(GEN)bnf[7];
  DEBUGLEVEL=debkeep;
  if (DEBUGLEVEL) msgtimer("Compute Cl(k)");

  exp=gmael4(bnf,8,1,2,1);
  if (gegal(exp,gdeux)) return GenusField(bnf,prec);

  bnrh=FindModule0(bnf,prec);
  if (DEBUGLEVEL) msgtimer("FindModule");

  if(DEBUGLEVEL>=2)
  {
    fprintferr("\nmodule = "); outerr(gmael3(bnrh,1,2,1));
    fprintferr("subgroup = "); outerr((GEN)bnrh[2]);
  }
  p1=gsqrt(gmul(gmael(bnf,7,3),det(gmael4(bnrh,1,2,1,1))),prec); 
  p1=gceil(gmul2n(gdiv(p1,dbltor(2.177)),-TWOPOTBITS_IN_LONG));
  newprec=max(itos(p1)+3+(cl>>2),prec);
  if (DEBUGLEVEL>=2) fprintferr("precision de depart = %ld \n",newprec);
  
  if (newprec>prec)
  {
    nf=nfnewprec(nf,newprec);
    bnrh[4]=(long)CharNewPrec((GEN)bnrh[4],nf,newprec);
  }
  else newprec=prec;
  
  pol=QuadStark(bnrh,nf,2,newprec);
  return gerepileupto(av,makescind(nf,pol,cl,prec));
}

GEN
bnrstark(GEN bnr, GEN subgroup, long flag, long prec)
{
  long cl,N,newprec;
  GEN bnf,dataS,p1,bnrm,D,dataD,dataCR,Mcyc,nf,dataQ;

  bnf = (GEN)bnr[1];
  nf  = (GEN)bnf[7];
  Mcyc= diagonal(gmael(bnr,5,2));
  N   = degree(gmael(bnf,7,1));
    
  if (N>2) err(impl,"bnrstark for non-quadratic fields");

  if (!varn(gmael(bnf,7,1)))
    err(talker,"main variable in bnrstark must not be x");
  
  if (cmpis(gmael3(bnf,7,2,1),N))
    err(talker,"not a totally real ground base field in bnrstark");
  
  checkbnrgen(bnr);
  if (gcmp0(subgroup)) { subgroup=Mcyc; }
  else
  {
    p1=gauss(subgroup,Mcyc);
    if (!gcmp1(denom(p1)))
      err(talker,"incorrect subgroup in bnrstark");
  }
  cl=itos(det(subgroup));

  if (cl==1) return polx[0];
    
  if (gcmp0(conductor(bnr,subgroup,-1,prec)))
    err(impl,"bnrstark when the module is not the conductor");
  
  if (!gcmp0(gmael3(bnr,2,1,2)))
    err(talker,"not a totally real class field in bnrstark");
  
  timer2();

  dataS=InitQuotient(bnr,subgroup);
  p1=FindModule(dataS,prec);
  bnrm=(GEN)p1[1];
  D=(GEN)p1[2];
  dataD=(GEN)p1[3];
  dataQ=(GEN)p1[4];
  if (DEBUGLEVEL) msgtimer("FindModule");
  if(DEBUGLEVEL>=2)
  {
    fprintferr("\nmodule = "); outerr(gmael(bnrm,2,1));
    fprintferr("subgroup = "); outerr(D);
  }
  p1=gsqrt(gmul(gmael(bnf,7,3),det(gmael3(bnrm,2,1,1))),prec); 
  p1=gceil(gmul2n(gdiv(p1,dbltor(2.177)),-TWOPOTBITS_IN_LONG));
  newprec=max(itos(p1)+3+(cl>>2),prec);
  
  if (DEBUGLEVEL>=2) fprintferr("precision de depart = %ld \n",newprec);
  if (newprec>prec)
  {
    nf=nfnewprec(nf,newprec);
    mael3(dataD,1,1,7)=(long)nf;
  }
  else newprec=prec;
 
  dataCR=InitChar(dataD,newprec);
  if (DEBUGLEVEL) msgtimer("InitChar");

  /* partie specifique au cas quadratique */
  p1=cgetg(6,t_VEC);
  p1[1]=(long)bnrm;
  p1[2]=(long)D;
  p1[3]=(long)dataD;
  p1[4]=(long)dataCR;
  p1[5]=(long)dataQ;

  return QuadStark(p1,nf,flag,newprec);
}
