/******************************************************************************
*
*       NTRegmon - Registry Monitor for Windows NT
*		
*		Copyright (c) 1996, 1997 Mark Russinovich and Bryce Cogswell
*
*    	PROGRAM: Regmon.c
*
*    	PURPOSE: Communicates with the RegMon driver to display 
*		registry activity information.
*
******************************************************************************/
#include <windows.h>    // includes basic windows functionality
#include <windowsx.h>
#include <commctrl.h>   // includes the common control header
#include <stdio.h>
#include <string.h>
#include <winioctl.h>
#include "resource.h"
#include "../dd/ioctlcmd.h"
#include "regmon.h"

// this typedef, present in newer include files,
// supports the building regmon on older systems
typedef struct 
{
    DWORD cbSize;
    DWORD dwMajorVersion;
    DWORD dwMinorVersion;
    DWORD dwBuildNumber;
    DWORD dwPlatformID;
} DLLVERSIONINFO_, *PDLLVERSIONINFO_;

HRESULT (CALLBACK *pDllGetVersionProc)( PDLLVERSIONINFO_ pdvi );

// toolbar height plus the borders
#define TOOLBARHEIGHT	28

// Number of columns in the listview
#define NUMCOLUMNS	6

// Variables/definitions for the driver that performs the actual monitoring.
#define				SYS_FILE		TEXT("REGSYS.SYS")
#define				SYS_NAME		TEXT("REGMON")
static HANDLE		sys_handle		= INVALID_HANDLE_VALUE;

// Position settings data structure 
typedef struct {
	int		left;
	int		top;
	int		width;
	int		height;
	DWORD	column[NUMCOLUMNS];
	DWORD	historydepth;
} POSITION_SETTINGS;

// The variable that holds the position settings
POSITION_SETTINGS	PositionInfo;

// typedef for balloon popup
typedef struct {
	char	itemText[1024];
	POINT	itemPosition;
} ITEM_CLICK, *PITEM_CLICK;

// toolbar constants
#define ID_TOOLBAR         1

// defined for comtl32.dll version 4.7
#define TOOLBAR_FLAT		0x800

// button definitions

// for installations that support flat style
TBBUTTON tbButtons[] = {
	{ 0, 0, TBSTATE_ENABLED, TBSTYLE_SEP, 0L, 0},
	{ 0, IDM_SAVE, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0},
	{ 8, 0, 0, TBSTYLE_BUTTON, 0L, 0},
	{ 2, IDM_CAPTURE, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0},
	{ 4, IDM_AUTOSCROLL, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0},
	{ 6, IDM_CLEAR, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0},	
	{ 8, 0, 0, TBSTYLE_BUTTON, 0L, 0},
	{ 5, IDM_FILTER, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0},
	{ 8, 0, 0, TBSTYLE_BUTTON, 0L, 0},
	{ 7, IDM_FIND, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0},
	{ 8, 0, 0, TBSTYLE_BUTTON, 0L, 0},
};
#define NUMBUTTONS		11

// for older installations
TBBUTTON tbButtonsOld[] = {
	{ 0, 0, TBSTATE_ENABLED, TBSTYLE_SEP, 0L, 0},
	{ 0, IDM_SAVE, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0},
	{ 0, 0, TBSTATE_ENABLED, TBSTYLE_SEP, 0L, 0},
	{ 2, IDM_CAPTURE, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0},
	{ 4, IDM_AUTOSCROLL, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0},
	{ 6, IDM_CLEAR, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0},	
	{ 0, 0, TBSTATE_ENABLED, TBSTYLE_SEP, 0L, 0},
	{ 5, IDM_FILTER, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0},
	{ 0, 0, TBSTATE_ENABLED, TBSTYLE_SEP, 0L, 0},
	{ 7, IDM_FIND, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0},
};						 
#define NUMBUTTONSOLD	10

// Buffer into which driver can copy statistics
char				Stats[ MAX_STORE ];
// Current fraction of buffer filled
DWORD				StatsLen;

// Search string
TCHAR				FindString[256];
FINDREPLACE			FindTextInfo;
DWORD				FindFlags = FR_DOWN;
BOOLEAN				PrevMatch;
TCHAR				PrevMatchString[256];	

// Application instance handle
HINSTANCE			hInst;

// For info saving
TCHAR				szFileName[256];
BOOLEAN				FileChosen = FALSE;

// Misc globals
HWND				hWndFind = NULL;
UINT				findMessageID;
HWND				hWndList;
HWND				hBalloon = NULL;
BOOLEAN				Capture = TRUE;
BOOLEAN				Autoscroll = TRUE;

// General buffer for storing temporary strings
static TCHAR		msgbuf[ 257 ];

// Filter-related
FILTER				FilterDefinition;

// listview size limiting
DWORD				MaxLines = 0;
DWORD				LastRow = 0;

// General cursor manipulation
HCURSOR 			hSaveCursor;
HCURSOR 			hHourGlass;

// Procs
long APIENTRY 		MainWndProc( HWND, UINT, UINT, LONG );
BOOL APIENTRY 		About( HWND, UINT, UINT, LONG );
BOOL APIENTRY 		FilterProc( HWND, UINT, UINT, LONG );


// Functions
BOOL 				InitApplication( HANDLE );
HWND 				InitInstance( HANDLE, int );
HWND 				CreateList( HWND );
void 				UpdateStatistics( HWND hWnd, HWND hWndList, BOOL Clear );
void				SaveFile( HWND hDlg, HWND listbox, BOOLEAN SaveAs );



/******************************************************************************
*
*	FUNCTION:	Abort:
*
*	PURPOSE:	Handles emergency exit conditions.
*
*****************************************************************************/
void Abort( HWND hWnd, TCHAR * Msg )
{
	LPVOID	lpMsgBuf;
	TCHAR	errmsg[256];
 
	FormatMessage( FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,
					NULL, GetLastError(), 
					MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
					(LPTSTR) &lpMsgBuf, 0, NULL );
	UnloadDeviceDriver( SYS_NAME );
	sprintf(errmsg, TEXT("%s: %s"), Msg, lpMsgBuf );
	MessageBox( hWnd, errmsg, TEXT("NTRegmon"), MB_OK );
	PostQuitMessage( 1 );
	LocalFree( lpMsgBuf );
}

/******************************************************************************
*
*	FUNCTION:	BalloonDialog
*
*	PURPOSE:	Dialog function for home-brewed balloon help.
*
******************************************************************************/
LONG APIENTRY BalloonDialog( HWND hDlg, UINT message, UINT wParam, LONG lParam )
{
	static ITEM_CLICK	ctx;
	static RECT			rect;
	static HFONT		hfont;
	LPCREATESTRUCT		lpcs;
	HDC					hdc;
	LOGFONT				lf;

	switch (message) {
		case WM_CREATE:

			lpcs = (void *)lParam;
			ctx = *(PITEM_CLICK) lpcs->lpCreateParams;
			hdc = GetDC( hDlg );

			// Get font
			if ( hfont == NULL )  {
				GetObject( GetStockObject(SYSTEM_FONT), sizeof lf, &lf ); 
				lf.lfWeight = FW_NORMAL;
				lf.lfHeight = 8;
				lf.lfWidth  = 0;
				strcpy( lf.lfFaceName, TEXT("MS Sans Serif") );
 				hfont = CreateFontIndirect( &lf ); 
			} 

			// Compute size of required rectangle
			rect.left	= 0;
			rect.top	= 0;
			rect.right	= lpcs->cx;
			rect.bottom	= lpcs->cy;
			SelectObject( hdc, hfont );
			DrawText( hdc, ctx.itemText, -1, &rect, 
						DT_NOCLIP|DT_LEFT|DT_NOPREFIX|DT_WORDBREAK|DT_CALCRECT );

			// Move and resize window
			if( ctx.itemPosition.x - 5 + rect.right + 10 >
				 GetSystemMetrics(SM_CXFULLSCREEN) ) {

				 ctx.itemPosition.x = GetSystemMetrics(SM_CXFULLSCREEN) -
							(rect.right+10-5);
			}
			MoveWindow( hDlg, 
						ctx.itemPosition.x-5, ctx.itemPosition.y+10, 
						rect.right + 10, 
						rect.bottom + 10,
						TRUE );

			// Adjust rectangle so text is centered
			rect.left	+= 5;
			rect.right	+= 5;
			rect.top	+= 5;
			rect.bottom	+= 5;
			break;

		case WM_PAINT:
			hdc = GetDC( hDlg );

			// Set colors
			SetTextColor( hdc, 0x00000000 );
			SetBkMode( hdc, TRANSPARENT );
			SelectObject( hdc, hfont );
			DrawText( hdc, ctx.itemText, -1, &rect, 
						DT_NOCLIP|DT_LEFT|DT_NOPREFIX|DT_WORDBREAK );
			break;

		case WM_MOUSEMOVE:
		case WM_CLOSE:	
			hBalloon = NULL;
			DestroyWindow( hDlg );
			break;
	}

    return DefWindowProc( hDlg, message, wParam, lParam );
}


/******************************************************************************
*
*	FUNCTION:	FindInListview:
*
*	PURPOSE:	Searches for a string in the listview. Note: its okay if
*				items are being added to the list view or the list view
*				is cleared while this search is in progress - the effect
*				is harmless.
*
*****************************************************************************/
BOOLEAN FindInListview(HWND hWnd, LPFINDREPLACE FindInfo )
{
	int		currentItem;
	DWORD	i;
	int		subitem, numItems;
	TCHAR	fieldtext[256];
	BOOLEAN match = FALSE;
	TCHAR	errmsg[256];
	BOOLEAN	goUp;

	// get the search direction
	goUp = ((FindInfo->Flags & FR_DOWN) == FR_DOWN);

	// initialize stuff
	if( !(numItems = ListView_GetItemCount( hWndList ))) {

		MessageBox( hWnd, TEXT("No items to search"), TEXT("NTRegmon"), 
			MB_OK|MB_ICONWARNING );
		return FALSE;
	}

	// find the item with the focus
	currentItem = ListView_GetNextItem( hWndList, 0, LVNI_SELECTED );

	// if no current item, start at the top or the bottom
	if( currentItem == -1 ) {
		if( goUp )
			currentItem = 0;
		else {
			if( PrevMatch ) {
				wsprintf(errmsg, TEXT("Cannot find string \"%s\""), FindInfo->lpstrFindWhat );
				MessageBox( hWnd, errmsg, TEXT("NTRegmon"), MB_OK|MB_ICONWARNING );
				return FALSE;
			}
			currentItem = numItems;
		}
	}

	// if we're continuing a search, start with the next item
	if( PrevMatch && !strcmp( FindString, PrevMatchString ) ) {
		if( goUp ) currentItem++;
		else currentItem--;

		if( (!goUp && currentItem < 0) ||
			(goUp && currentItem >= numItems )) {

			wsprintf(errmsg, TEXT("Cannot find string \"%s\""), FindInfo->lpstrFindWhat );
			MessageBox( hWnd, errmsg, TEXT("NTRegmon"), MB_OK|MB_ICONWARNING );
			return FALSE;
		}
	}

	// loop through each item looking for the string
	while( 1 ) {

		// get the item text
		for( subitem = 0; subitem < NUMCOLUMNS; subitem++ ) {
			fieldtext[0] = 0;
			ListView_GetItemText( hWndList, currentItem, subitem, fieldtext, 256 );

			// make sure enought string for a match
			if( strlen( fieldtext ) < strlen( FindInfo->lpstrFindWhat ))
				continue;

			// do a scan all the way through for the substring
			if( FindInfo->Flags & FR_WHOLEWORD ) {

				i = 0;
				while( fieldtext[i] ) {
					while( fieldtext[i] && fieldtext[i] != ' ' ) i++;
					if( FindInfo->Flags & FR_MATCHCASE ) 
						match = !strcmp( fieldtext, FindInfo->lpstrFindWhat );
					else
						match = !stricmp( fieldtext, FindInfo->lpstrFindWhat );
					if( match) break;
					i++;
				}	
			} else {
				for( i = 0; i < strlen( fieldtext ) - strlen(FindInfo->lpstrFindWhat)+1; i++ ) {
					if( FindInfo->Flags & FR_MATCHCASE ) 
						match = !strncmp( &fieldtext[i], FindInfo->lpstrFindWhat, 
											strlen(FindInfo->lpstrFindWhat) );
					else
						match = !strnicmp( &fieldtext[i], FindInfo->lpstrFindWhat,
											strlen(FindInfo->lpstrFindWhat) );
					if( match ) break;
				}		
			}

			if( match ) {

				strcpy( PrevMatchString, FindInfo->lpstrFindWhat );
				PrevMatch = TRUE;
				ListView_SetItemState( hWndList, currentItem, 
							LVIS_SELECTED|LVIS_FOCUSED,
							LVIS_SELECTED|LVIS_FOCUSED );
				ListView_EnsureVisible( hWndList, currentItem, FALSE ); 
				SetFocus( hWndList );
				return TRUE;
			}
		}
		currentItem = currentItem + (goUp ? 1:-1);
		if( !currentItem || currentItem == numItems+1 ) {
			// end of the road
			break;
		}
	}
	wsprintf(errmsg, TEXT("Cannot find string \"%s\""), FindInfo->lpstrFindWhat );
	MessageBox( hWnd, errmsg, TEXT("NTRegmon"), MB_OK|MB_ICONWARNING );
	return FALSE;
}


/******************************************************************************
*
*	FUNCTION:	PopFindDialog:
*
*	PURPOSE:	Calls the find message dialog box.
*
*****************************************************************************/
void PopFindDialog(HWND hWnd)
{
	strcpy( FindString, PrevMatchString );
    FindTextInfo.lStructSize = sizeof( FindTextInfo );
    FindTextInfo.hwndOwner = hWnd;
    FindTextInfo.hInstance = (HANDLE)hInst;
    FindTextInfo.lpstrFindWhat = FindString;
    FindTextInfo.lpstrReplaceWith = NULL;
    FindTextInfo.wFindWhatLen = sizeof(FindString);
    FindTextInfo.wReplaceWithLen = 0;
    FindTextInfo.lCustData = 0;
    FindTextInfo.Flags =  FindFlags;
    FindTextInfo.lpfnHook = (LPFRHOOKPROC)(FARPROC)NULL;
    FindTextInfo.lpTemplateName = NULL;

    if ((hWndFind = FindText(&FindTextInfo)) == NULL)
		MessageBox( hWnd, TEXT("Unable to create Find dialog"), TEXT("NTRegmon"), MB_OK|MB_ICONERROR );      
}


/******************************************************************************
*
*	FUNCTION:	Get_Position_Settings
*
*	PURPOSE:	Reads the Registry to get the last-set window position.
*
******************************************************************************/
VOID Get_Position_Settings()
{
	HANDLE	hKey;
	DWORD	ParamSize;

	// Fist, set the default settings
	PositionInfo.top	= CW_USEDEFAULT;
	PositionInfo.left	= CW_USEDEFAULT;
	PositionInfo.width	= CW_USEDEFAULT;
	PositionInfo.height	= CW_USEDEFAULT;

	// set the default listview widths
	PositionInfo.column[0] = 35;
	PositionInfo.column[1] = 90;
	PositionInfo.column[2] = 130;
	PositionInfo.column[3] = 200;
	PositionInfo.column[4] = 70;
	PositionInfo.column[5] = 150;

	// set the default history depth (infinite)
	PositionInfo.historydepth = 0;

	// first, get the last-entered params from the registry
	RegCreateKey(HKEY_CURRENT_USER, 
			TEXT("Software\\NTInternals\\Regmon"),
			&hKey );

	// get the params and ignore errors
	ParamSize = sizeof( PositionInfo );
	RegQueryValueEx( hKey,TEXT("Settings"), NULL, NULL, (LPBYTE) &PositionInfo,
				&ParamSize );
	CloseHandle( hKey );

	// extract the history depth
	MaxLines = PositionInfo.historydepth;
}


/******************************************************************************
*
*	FUNCTION:	Save_Position_Settings
*
*	PURPOSE:	Saves the current window settings to the Registry.
*
******************************************************************************/
VOID Save_Position_Settings( HWND hWnd )
{
	RECT		rc;
	int			i;
	HANDLE		hKey;

	// get the position of the main window
	GetWindowRect( hWnd, &rc );
	PositionInfo.left = rc.left;
	PositionInfo.top = rc.top;
	PositionInfo.width = rc.right - rc.left;
	PositionInfo.height = rc.bottom - rc.top;

	// get the widths of the listview columns
	for( i = 0; i < NUMCOLUMNS; i++ ) {
		PositionInfo.column[i] = ListView_GetColumnWidth( hWndList, i );
	}

	// get the history depth
	PositionInfo.historydepth = MaxLines;

	// save connection info to registry
	RegOpenKey(HKEY_CURRENT_USER, 
			TEXT("Software\\NTInternals\\Regmon"),
			&hKey );
	RegSetValueEx( hKey, TEXT("Settings"), 0, REG_BINARY, (LPBYTE) &PositionInfo,
			sizeof( PositionInfo ) );
	CloseHandle( hKey );
}	


/******************************************************************************
*
*	FUNCTION:	Split
*
*	PURPOSE:	Split a delimited line into components
*
******************************************************************************/
int Split( char * line, char delimiter, char * items[] )
{
	int		cnt = 0;

	for (;;)  {
		// Add prefix to list of components		
		items[cnt++] = line;

		// Check for more components
		line = strchr( line, delimiter );
		if ( line == NULL )
			return cnt;

		// Terminate previous component and move to next
		*line++ = '\0';
	}		
}


/******************************************************************************
*
*	FUNCTION:	ListAppend
*
*	PURPOSE:	Add a new line to List window
*
******************************************************************************/
BOOL List_Append( HWND hWndList, DWORD seq, char * line )
{
	LV_ITEM		lvI;	// list view item structure
	int			row;
	char	*	items[20];
	int			itemcnt = 0;

	// Split line into columns
	itemcnt = Split( line, '\t', items );
	if ( itemcnt == 0 )
		return TRUE;

	// Determine row number for request
	if ( *items[0] )  {
		// Its a new request.  Put at end.
		row = 0x7FFFFFFF;
	} else {
		// Its a status.  Locate its associated request.
		lvI.mask = LVIF_PARAM;
		lvI.iSubItem = 0;
		for ( row = ListView_GetItemCount(hWndList) - 1; row >= 0; --row )  {
			lvI.iItem = row;
			if ( ListView_GetItem( hWndList, &lvI )  &&  (DWORD)lvI.lParam == seq )
				break;
		}
		if ( row == -1 )
			// No request associated with status.
			return TRUE;
	}

	// Sequence number if a new item
	if ( *items[0] )  {
		wsprintf( msgbuf, TEXT("%d"), seq );
		lvI.mask		= LVIF_TEXT | LVIF_PARAM;
		lvI.iItem		= row;
		lvI.iSubItem	= 0;
		lvI.pszText		= msgbuf;
		lvI.cchTextMax	= lstrlen( lvI.pszText ) + 1;
		lvI.lParam		= seq;
		row = ListView_InsertItem( hWndList, &lvI );
		if ( row == -1 )  {
			wsprintf( msgbuf, TEXT("Error adding item %d to list view"), seq );
			MessageBox( hWndList, msgbuf, TEXT("NTRegmon Error"), MB_OK );
			return FALSE;
		}
        LastRow = row;
	}

	// Process name
	if ( itemcnt>0 && *items[0] ) {
		OemToChar( items[0], msgbuf );
		ListView_SetItemText( hWndList, row, 1, msgbuf );
	}

	// Request type
	if ( itemcnt>1 && *items[1] )  {
		OemToChar( items[1], msgbuf );
		ListView_SetItemText( hWndList, row, 2, msgbuf );
	}

	// Path
	if ( itemcnt>2 && *items[2] )  {
		OemToChar( items[2], msgbuf );
		ListView_SetItemText( hWndList, row, 3, msgbuf );
	}

	// Result
	if ( itemcnt>4 && *items[4] )  {
		OemToChar( items[4], msgbuf );
		ListView_SetItemText( hWndList, row, 4, msgbuf );
	}

	// Additional
	if ( itemcnt>3 && *items[3] )  {
		OemToChar( items[3], msgbuf );
		ListView_SetItemText( hWndList, row, 5, msgbuf );
	}

	return TRUE;
}


/******************************************************************************
*
*	FUNCTION:	UpdateStatistics
*
*	PURPOSE:	Clear the statistics window and refill it with the current 
*				contents of the statistics buffer.  Does not refresh the 
*				buffer from the device driver.
*
******************************************************************************/
void UpdateStatistics( HWND hWnd, HWND hWndList, BOOL Clear )
{
	PENTRY	ptr;

	// Just return if nothing to do
	if ( !Clear  &&  StatsLen < sizeof(int)+2 )
		return;

	// Start with empty list
	if ( Clear ) {
		ListView_DeleteAllItems( hWndList );
		LastRow = 0;
	}

	// Add all List items from Stats[] data
	for ( ptr = (void *)Stats; (char *)ptr < min(Stats+StatsLen,Stats + sizeof (Stats)); )  {
	 	// Add to list
		ULONG len = strlen(ptr->text);
        len += 4; len &= 0xFFFFFFFC; // +1 for null-terminator +3 for 32bit alignment
		List_Append( hWndList, ptr->seq, ptr->text );
		ptr = (void *)(ptr->text + len);
	}

	// Empty the buffer
	StatsLen = 0;

	// limit number of lines saved
	if (MaxLines) {
		SendMessage(hWndList, WM_SETREDRAW, FALSE, 0);
		while ( LastRow > MaxLines ) {
			ListView_DeleteItem ( hWndList, 0 );
		    LastRow--;
		}
		SendMessage(hWndList, WM_SETREDRAW, TRUE, 0);
    }

	// Scroll so newly added items are visible
	if ( Autoscroll ) 
		ListView_EnsureVisible( hWndList, ListView_GetItemCount(hWndList)-1, FALSE ); 
}


/****************************************************************************
* 
*    FUNCTION: CreateListView(HWND)
*
*    PURPOSE:  Creates the statistics list view window and initializes it
*
****************************************************************************/
HWND CreateList( HWND hWndParent )                                     
{
	HWND		hWndList;    	  	// handle to list view window
	RECT		rc;         	  	// rectangle for setting size of window
	LV_COLUMN	lvC;				// list view column structure
	DWORD		j;
	static struct {
		TCHAR *	Label;	// title of column
		DWORD	Width;	// width of column in pixels
		DWORD	Fmt;
	} column[] = {
		{	"#"			,	35		},
		{	"Process"	,	90		},
		{	"Request"	,	90		},
		{	"Path"		,	225		},
		{	"Result"	,	90		},
		{	"Other"		,	150		},
	};

	// Ensure that the common control DLL is loaded.
	InitCommonControls();

	// Set the column widths according to the user-settings
	for( j = 0; j < NUMCOLUMNS; j++ ) {
		column[j].Width = PositionInfo.column[j];
	}

	// Get the size and position of the parent window.
	GetClientRect( hWndParent, &rc );

	// Create the list view window
	hWndList = CreateWindowEx( /* WS_EX_CLIENTEDGE */ 0L, WC_LISTVIEW, TEXT(""), 
								WS_VISIBLE | WS_CHILD | WS_BORDER | LVS_REPORT |
								LVS_SINGLESEL,
								0, TOOLBARHEIGHT, rc.right - rc.left, rc.bottom - rc.top - TOOLBARHEIGHT,
								hWndParent,	(HMENU)ID_LIST, hInst, NULL );
	if ( hWndList == NULL )
		return NULL;

	// Initialize columns
	lvC.mask	= LVCF_FMT | LVCF_WIDTH | LVCF_TEXT | LVCF_SUBITEM;
	lvC.fmt		= LVCFMT_LEFT;	// left-align column

	// Add the columns.
	for ( j = 0; j < sizeof column/sizeof column[0]; ++j )  {
		lvC.iSubItem	= j;
		lvC.cx			= column[j].Width;
	 	lvC.pszText		= column[j].Label;
		if ( ListView_InsertColumn( hWndList, j, &lvC ) == -1 )
			return NULL;
	}

	return hWndList;
}


/****************************************************************************
* 
*    FUNCTION: SaveFile()
*
*    PURPOSE:  Lets the user go select a file.
*
****************************************************************************/
void SaveFile( HWND hWnd, HWND ListBox, BOOLEAN SaveAs )
{
	OPENFILENAME	SaveFileName;
	char			szFile[256] = "", fieldtext[256], output[1024];
	FILE			*hFile;
	int				numitems;
	int				row, subitem;

	if( SaveAs || !FileChosen ) {
		SaveFileName.lStructSize       = sizeof (SaveFileName);
		SaveFileName.hwndOwner         = hWnd;
		SaveFileName.hInstance         = (HANDLE) hInst;
		SaveFileName.lpstrFilter       = "Reg Data (*.RGD)\0*.RGD\0All (*.*)\0*.*\0";
		SaveFileName.lpstrCustomFilter = (LPTSTR)NULL;
		SaveFileName.nMaxCustFilter    = 0L;
		SaveFileName.nFilterIndex      = 1L;
		SaveFileName.lpstrFile         = szFile;
		SaveFileName.nMaxFile          = 256;
		SaveFileName.lpstrFileTitle    = NULL;
		SaveFileName.nMaxFileTitle     = 0;
		SaveFileName.lpstrInitialDir   = NULL;
		SaveFileName.lpstrTitle        = "Save File Info...";
		SaveFileName.nFileOffset       = 0;
		SaveFileName.nFileExtension    = 0;
		SaveFileName.lpstrDefExt       = "*.rgd";
		SaveFileName.lpfnHook		   = NULL;
 		SaveFileName.Flags = OFN_LONGNAMES|OFN_HIDEREADONLY;

		if( !GetSaveFileName( &SaveFileName )) 
			return;
	} else 
		// open previous szFile
		strcpy( szFile, szFileName );

	// open the file
	hFile = fopen( szFile, "w" );
	if( !hFile ) {
		MessageBox(	NULL, "Create File Failed.",
				"Save Error", MB_OK|MB_ICONSTOP );
		return;
	}
	// post hourglass icon
	SetCapture(hWnd);
	hSaveCursor = SetCursor(hHourGlass);

	numitems = ListView_GetItemCount(ListBox);
	for ( row = 0; row < numitems; row++ )  {
		output[0] = 0;
		for( subitem = 0; subitem < 6; subitem++ ) {
			fieldtext[0] = 0;
			ListView_GetItemText( ListBox, row, subitem, fieldtext, 256 );
			strcat( output, fieldtext );
			strcat( output, "\t" );
		}
		fprintf( hFile, "%s\n", output );
	}
	fclose( hFile );
	strcpy( szFileName, szFile );
	FileChosen = TRUE;
	SetCursor( hSaveCursor );
	ReleaseCapture(); 
}



/****************************************************************************
*
*	FUNCTION:	FilterProc
*
*	PURPOSE:	Processes messages for "Filter" dialog box
*
****************************************************************************/
BOOL APIENTRY FilterProc( HWND hDlg, UINT message, UINT wParam, LONG lParam )
{
	int				nb;
	FILTER			upcaseFilter;
	DWORD			newMaxLines;
	char			history[64];

	switch ( message )  {
	case WM_INITDIALOG:

		// initialize the controls to reflect the current filter
		SetDlgItemText( hDlg, IDC_PROCFILTER, FilterDefinition.processfilter );
		SetDlgItemText( hDlg, IDC_PATHFILTER, FilterDefinition.pathfilter );
		SetDlgItemText( hDlg, IDC_EXCLUDEFILTER, FilterDefinition.excludefilter );
		CheckDlgButton( hDlg, IDC_SUCCESS, FilterDefinition.logsuccess );
		CheckDlgButton( hDlg, IDC_ERROR,   FilterDefinition.logerror );
		CheckDlgButton( hDlg, IDC_LOGREADS, FilterDefinition.logreads );
		CheckDlgButton( hDlg, IDC_LOGWRITES,   FilterDefinition.logwrites );
		sprintf( history, "%d", MaxLines );
		SetDlgItemTextA( hDlg, IDC_HISTORY, history );
		return TRUE;

	case WM_COMMAND:              
		if ( LOWORD( wParam ) == IDOK )	 {

			// make sure that max lines is legal
			GetDlgItemTextA( hDlg, IDC_HISTORY, history, 64 );
			if( !sscanf( history, "%d", &newMaxLines )) {

				MessageBox(	NULL, TEXT("Invalid History Depth."),
						TEXT("Filter Error"), MB_OK|MB_ICONWARNING );
				return TRUE;
			} 
			MaxLines = newMaxLines;

			// read the values that were set
			GetDlgItemText( hDlg, IDC_PROCFILTER, FilterDefinition.processfilter, 32 );
			GetDlgItemText( hDlg, IDC_PATHFILTER, FilterDefinition.pathfilter, 256 );
			GetDlgItemText( hDlg, IDC_EXCLUDEFILTER, FilterDefinition.excludefilter, 256 );
			FilterDefinition.logsuccess = IsDlgButtonChecked( hDlg, IDC_SUCCESS );
			FilterDefinition.logerror   = IsDlgButtonChecked( hDlg, IDC_ERROR );
			FilterDefinition.logreads = IsDlgButtonChecked( hDlg, IDC_LOGREADS );
			FilterDefinition.logwrites   = IsDlgButtonChecked( hDlg, IDC_LOGWRITES );

			// make an upcase version for the driver
			upcaseFilter = FilterDefinition;
			_strupr(upcaseFilter.processfilter);
			_strupr(upcaseFilter.pathfilter);
			_strupr(upcaseFilter.excludefilter);
 
			// tell the driver the new filter
			if ( ! DeviceIoControl(	sys_handle, REGMON_setfilter,
									&upcaseFilter, sizeof(FILTER), NULL, 
									0, &nb, NULL ) )
			{
				Abort( hDlg, TEXT("Couldn't access device driver") );
				return TRUE;
			}

			EndDialog( hDlg, TRUE );
			return TRUE;

		} else if( LOWORD( wParam ) == IDCANCEL ) {

			EndDialog( hDlg, TRUE );

		} else if( LOWORD( wParam ) == IDRESET ) {

			// reset filter to default of none
			sprintf( FilterDefinition.processfilter, "*" );
			sprintf( FilterDefinition.pathfilter, "*" );
			sprintf( FilterDefinition.excludefilter, "");
			FilterDefinition.logsuccess = TRUE;
			FilterDefinition.logerror = TRUE;
			FilterDefinition.logreads = TRUE;
			FilterDefinition.logwrites = TRUE;
			MaxLines = 0;
 
			// initialize the controls to reflect the current filter
			SetDlgItemText( hDlg, IDC_PROCFILTER, FilterDefinition.processfilter );
			SetDlgItemText( hDlg, IDC_PATHFILTER, FilterDefinition.pathfilter );
			SetDlgItemText( hDlg, IDC_EXCLUDEFILTER, FilterDefinition.excludefilter );
			CheckDlgButton( hDlg, IDC_SUCCESS, FilterDefinition.logsuccess );
			CheckDlgButton( hDlg, IDC_ERROR,   FilterDefinition.logerror );
			CheckDlgButton( hDlg, IDC_LOGREADS, FilterDefinition.logreads );
			CheckDlgButton( hDlg, IDC_LOGWRITES,   FilterDefinition.logwrites );
			SetDlgItemTextA( hDlg, IDC_HISTORY, "0" );
		}
		break;

	case WM_CLOSE:
		EndDialog( hDlg, TRUE );
		return TRUE;
	}
	return FALSE;   
}


/****************************************************************************
*
*	FUNCTION:	About
*
*	PURPOSE:	Processes messages for "About" dialog box
*
****************************************************************************/
BOOL APIENTRY About( HWND hDlg, UINT message, UINT wParam, LONG lParam )
{
	switch ( message )  {
	   case WM_INITDIALOG:
		  return TRUE;

	   case WM_COMMAND:              
		  if ( LOWORD( wParam ) == IDOK )	 {
			  EndDialog( hDlg, TRUE );
			  return TRUE;
		  }
		  break;

	   case WM_CLOSE:
		  EndDialog( hDlg, TRUE );
		  return TRUE;
	}
	return FALSE;   
}


/******************************************************************************
*
*	FUNCTION:	GetDLLVersion
*
*	PURPOSE:	Gets the version number of the specified DLL.
*
******************************************************************************/
HRESULT GetDLLVersion( PCHAR DllName, LPDWORD pdwMajor, LPDWORD pdwMinor)
{
	HINSTANCE			hDll;
	HRESULT				hr = S_OK;
	DLLVERSIONINFO_		dvi;

	*pdwMajor = 0;
	*pdwMinor = 0;

	//Load the DLL.
	hDll = LoadLibrary(DllName);

	if( hDll ) {

	   pDllGetVersionProc = (PVOID)GetProcAddress(hDll, TEXT("DllGetVersion"));

	   if(pDllGetVersionProc) {
  
		  ZeroMemory(&dvi, sizeof(dvi));
		  dvi.cbSize = sizeof(dvi);

		  hr = (*pDllGetVersionProc)(&dvi);
  
		  if(SUCCEEDED(hr)) {

			 *pdwMajor = dvi.dwMajorVersion;
			 *pdwMinor = dvi.dwMinorVersion;
		  }
 	  } else {

		  // If GetProcAddress failed, the DLL is a version previous to the one 
		  // shipped with IE 3.x.
		  *pdwMajor = 4;
		  *pdwMinor = 0;
      }
   
	  FreeLibrary(hDll);
	  return hr;
	}

	return E_FAIL;
}


/****************************************************************************
*
*    FUNCTION: MainWndProc(HWND, unsigned, WORD, LONG)
*
*    PURPOSE:  Processes messages for the statistics window.
*
****************************************************************************/
LONG APIENTRY MainWndProc( HWND hWnd, UINT message, UINT wParam, LONG lParam) 
{
	DWORD			nb;
	TCHAR			Path[ 256 ], err[32];
	static HWND		hWndToolbar;
	LPTOOLTIPTEXT	lpToolTipText;
	static TCHAR	szBuf[128];
	LPFINDREPLACE	findMessageInfo;
	DWORD			majorver, minorver;
	ITEM_CLICK		itemClick;
	LVHITTESTINFO	hitItem;

	switch ( message ) {

		case WM_CREATE:

			// get hourglass icon ready
			hHourGlass = LoadCursor( NULL, IDC_WAIT );

			// post hourglass icon
			SetCapture(hWnd);
			hSaveCursor = SetCursor(hHourGlass);

			// Initialize the filter
			sprintf( FilterDefinition.processfilter, TEXT("*") );
			sprintf( FilterDefinition.pathfilter,    TEXT("*") );
			sprintf( FilterDefinition.excludefilter, "");
			FilterDefinition.logsuccess = TRUE;
			FilterDefinition.logerror   = TRUE;
			FilterDefinition.logreads   = TRUE;
			FilterDefinition.logwrites  = TRUE;

			// Create the toolbar control - use modern style if available.
			GetDLLVersion( "comctl32.dll", &majorver, &minorver );
			if( majorver > 4 || (majorver == 4 && minorver >= 70) ) {
				hWndToolbar = CreateToolbarEx( 
					hWnd, TOOLBAR_FLAT | WS_CHILD | WS_BORDER | WS_VISIBLE | TBSTYLE_TOOLTIPS,  
					ID_TOOLBAR, 8, hInst, IDB_TOOLBAR, (LPCTBBUTTON)&tbButtons,
					NUMBUTTONS, 16,16,16,16, sizeof(TBBUTTON)); 
			} else {
				hWndToolbar = CreateToolbarEx( 
					hWnd, WS_CHILD | WS_BORDER | WS_VISIBLE | TBSTYLE_TOOLTIPS,  
					ID_TOOLBAR, 8, hInst, IDB_TOOLBAR, (LPCTBBUTTON)&tbButtonsOld,
					NUMBUTTONSOLD, 16,16,16,16, sizeof(TBBUTTON)); 
			}
			if (hWndToolbar == NULL )
				MessageBox (NULL, TEXT("Toolbar not created!"), NULL, MB_OK );

			// Create the ListBox within the main window
			hWndList = CreateList( hWnd );
			if ( hWndList == NULL )
				MessageBox( NULL, TEXT("List not created!"), NULL, MB_OK );

		    // open the handle to the device
			GetCurrentDirectory( sizeof Path, Path );
			wsprintf( Path+lstrlen(Path), TEXT("\\%s"), SYS_FILE );
			if ( ! LoadDeviceDriver( SYS_NAME, Path, &sys_handle ) )  {
				if( GetLastError() == 2 )
					sprintf( err, "File Not Found" );
				else
					sprintf( err, "Error %d", GetLastError() );
				wsprintf( msgbuf, TEXT("Opening %s (%s):%s"), SYS_NAME, Path,
					err);

				Abort( hWnd, msgbuf );
			}

			// Have driver zero information
			if ( ! DeviceIoControl(	sys_handle, REGMON_zerostats,
									NULL, 0, NULL, 0, &nb, NULL ) )
			{
				Abort( hWnd, TEXT("Couldn't access device driver") );
				return TRUE;
			}

			// tell the driver the initial filter
			if ( ! DeviceIoControl(	sys_handle, REGMON_setfilter,
									&FilterDefinition, sizeof(FILTER), NULL, 
									0, &nb, NULL ) )
			{
				Abort( hWnd, TEXT("Couldn't access device driver") );
			}

			// Start up timer to periodically update screen
			SetTimer( hWnd,	1, 500/*ms*/, NULL );

			// Have driver turn on hooks
			if ( ! DeviceIoControl(	sys_handle, REGMON_hook,
									NULL, 0, NULL, 0, &nb, NULL ) )
			{
				Abort( hWnd, TEXT("Couldn't access device driver") );
				return TRUE;
			}

			
			// Initialization done
			SetCursor( hSaveCursor );
			ReleaseCapture();
			return FALSE;

		case WM_PARENTNOTIFY:

			// pop-up what the user clicked on
			if( LOWORD(wParam) == WM_RBUTTONDOWN ) {  

				hitItem.pt.x = LOWORD(lParam);
				hitItem.pt.y =  HIWORD(lParam);

				ClientToScreen( hWnd, &hitItem.pt );
				itemClick.itemPosition = hitItem.pt;

				if( ScreenToClient( hWndList, &hitItem.pt ) &&
					hitItem.pt.y >= 0 && 
					ListView_SubItemHitTest( hWndList, &hitItem ) != -1 ) {

					itemClick.itemText[0] = 0;
					ListView_GetItemText( hWndList, hitItem.iItem,
							hitItem.iSubItem, itemClick.itemText, 1024 );

					// delete any existing balloon
					if( hBalloon ) DestroyWindow( hBalloon );

					if( strlen( itemClick.itemText ) ) {

						// pop-up a balloon (tool-tip like window)
						hBalloon = CreateWindowEx( 0, TEXT("BALLOON"), 
										TEXT("balloon"), 
										WS_POPUP|WS_VISIBLE|WS_BORDER,
										100, 100,
										200, 200,
										hWnd, NULL, hInst, 
										&itemClick );
					}
					return TRUE;
				}
			}
			break;

		case WM_NOTIFY:
			// Make sure its intended for us
			if ( wParam == ID_LIST )  {
				NM_LISTVIEW	* pNm = (NM_LISTVIEW *)lParam;
				switch ( pNm->hdr.code )  {

			        case LVN_BEGINLABELEDIT:
						// Don't allow editing of information
						return TRUE;
				}
			}else {

				switch (((LPNMHDR) lParam)->code) 
				{
					case TTN_NEEDTEXT:    
						// Display the ToolTip text.
						lpToolTipText = (LPTOOLTIPTEXT)lParam;
    					LoadString (hInst, lpToolTipText->hdr.idFrom, szBuf, sizeof(szBuf));
				    	lpToolTipText->lpszText = szBuf;
						break;

					default:
						return FALSE;
				}
			}
			return FALSE;

		case WM_COMMAND:

			switch ( LOWORD( wParam ) )	 {

				// stats related commands to send to driver
				case IDM_CLEAR:
					// Have driver zero information
					if ( ! DeviceIoControl(	sys_handle, REGMON_zerostats,
											NULL, 0, NULL, 0, &nb, NULL ) )
					{
						Abort( hWnd, TEXT("Couldn't access device driver") );
						return TRUE;
					}
					// Update statistics windows
					UpdateStatistics( hWnd, hWndList, TRUE );
					return FALSE;

				case IDM_HELP:
					WinHelp(hWnd, TEXT("ntregmon.hlp"), HELP_CONTENTS, 0L);
					return 0;

				case IDM_FIND:
					// search the listview
					if( !hWndFind ) {
						PrevMatch = FALSE;
						PopFindDialog( hWnd );
					} else if( PrevMatch ) {

						// treat this like a find-next
						SetCapture(hWndFind);
						hSaveCursor = SetCursor(hHourGlass);
						EnableWindow( hWndFind, FALSE );
						if (FindInListview( hWnd, &FindTextInfo ) ) {
							Autoscroll = FALSE;
							CheckMenuItem( GetMenu(hWnd), IDM_AUTOSCROLL,
											MF_BYCOMMAND|MF_UNCHECKED ); 
							SendMessage( hWndToolbar, TB_CHANGEBITMAP, IDM_AUTOSCROLL, 3 );
						}
						EnableWindow( hWndFind, TRUE );
						SetCursor( hSaveCursor );
						ReleaseCapture(); 
					}
					return 0;

				case IDM_CAPTURE:
					// Read statistics from driver
					Capture = !Capture;
					CheckMenuItem( GetMenu(hWnd), IDM_CAPTURE,
									MF_BYCOMMAND|(Capture?MF_CHECKED:MF_UNCHECKED) );

					// Have driver turn on hooks
					if ( ! DeviceIoControl(	sys_handle, Capture ? REGMON_hook : 
											REGMON_unhook,
											NULL, 0, NULL, 0, &nb, NULL ) )
					{
						Abort( hWnd, TEXT("Couldn't access device driver") );
						return TRUE;
					}
					SendMessage( hWndToolbar, TB_CHANGEBITMAP, IDM_CAPTURE, (Capture?2:1) );
					InvalidateRect( hWndToolbar, NULL, TRUE );
					return FALSE;

				case IDM_AUTOSCROLL:
					Autoscroll = !Autoscroll;
					CheckMenuItem( GetMenu(hWnd), IDM_AUTOSCROLL,
									MF_BYCOMMAND|(Autoscroll?MF_CHECKED:MF_UNCHECKED) ); 
					SendMessage( hWndToolbar, TB_CHANGEBITMAP, IDM_AUTOSCROLL, (Autoscroll?4:3) );
					InvalidateRect( hWndToolbar, NULL, TRUE );					
					return FALSE;

				case IDM_EXIT:
					// Close ourself
					SendMessage( hWnd, WM_CLOSE, 0, 0 );
					return FALSE;

				case IDM_FILTER:
					DialogBox( hInst, TEXT("Filter"), hWnd, (DLGPROC) FilterProc );
					return FALSE;

				case IDM_ABOUT:
					// Show the names of the authors
					DialogBox( hInst, TEXT("AboutBox"), hWnd, (DLGPROC)About );
					return FALSE;

				case IDM_SAVE:
					SaveFile( hWnd, hWndList, FALSE );
					return FALSE;

				case IDM_SAVEAS:
					SaveFile( hWnd, hWndList, TRUE );
					return FALSE;

				default:
					// Default behavior
					return DefWindowProc( hWnd, message, wParam, lParam );
			}
			break;

		case WM_TIMER:
			// Time to query the device driver for more data
			if ( Capture )  {
				for (;;)  {

					// Have driver fill Stats buffer with information
					if ( ! DeviceIoControl(	sys_handle, REGMON_getstats,
											NULL, 0, &Stats, sizeof Stats,
											&StatsLen, NULL ) )
					{
						Abort( hWnd, TEXT("Couldn't access device driver") );
						return TRUE;
					}
					if ( StatsLen == 0 )
						break;
					// Update statistics windows
					UpdateStatistics( hWnd, hWndList, FALSE );
				}
			}
			return FALSE;

		case WM_SIZE:
			// Move or resize the List
			MoveWindow( hWndToolbar, 0, 0, LOWORD(lParam), HIWORD(lParam), TRUE );
            MoveWindow( hWndList, 0, TOOLBARHEIGHT, LOWORD(lParam), HIWORD(lParam)-TOOLBARHEIGHT, TRUE );
			return FALSE;

		case WM_CLOSE:

			// Have driver unhook if necessary
			if( Capture ) {
				if ( ! DeviceIoControl(	sys_handle, REGMON_unhook,
									NULL, 0, NULL, 0, &nb, NULL ) )
				{
					Abort( hWnd, TEXT("Couldn't access device driver") );
					return TRUE;
				}
			}

			CloseHandle( sys_handle );			
			if ( ! UnloadDeviceDriver( SYS_NAME ) )  {
				wsprintf( msgbuf, TEXT("Error unloading \"%s\""), SYS_NAME );
				MessageBox( hWnd, msgbuf, TEXT("NTRegmon"), MB_OK );
			}
			Save_Position_Settings( hWnd );
			return DefWindowProc( hWnd, message, wParam, lParam );

		case WM_DESTROY:
			PostQuitMessage(0);
			return FALSE;

		default:
			// is it a find-string message?
			if (message == findMessageID ){ 

				// get a pointer to the find structure
				findMessageInfo = (LPFINDREPLACE)lParam;

				// If the FR_DIALOGTERM flag is set, invalidate the find window handle
				if( findMessageInfo->Flags & FR_DIALOGTERM) {
					hWndFind = NULL;
					PrevMatch = FALSE;
				    FindFlags = FindTextInfo.Flags & (FR_DOWN|FR_MATCHCASE|FR_WHOLEWORD);
					return 0;
				}

				// if the FR_FINDNEXT flag is set, go do the search
				if( findMessageInfo->Flags & FR_FINDNEXT ) {
					SetCapture(hWndFind);
					hSaveCursor = SetCursor(hHourGlass);
					EnableWindow( hWndFind, FALSE );
					if( FindInListview( hWnd, findMessageInfo ) ) {
						Autoscroll = FALSE;
						CheckMenuItem( GetMenu(hWnd), IDM_AUTOSCROLL,
										MF_BYCOMMAND|MF_UNCHECKED ); 
						SendMessage( hWndToolbar, TB_CHANGEBITMAP, IDM_AUTOSCROLL, 3 );
					}
					EnableWindow( hWndFind, TRUE );
					ReleaseCapture(); 
					return 0;
				}
				return 0;
			}

			// Default behavior
			return DefWindowProc( hWnd, message, wParam, lParam );
	}
	return FALSE;
}



/****************************************************************************
*
*    FUNCTION: InitApplication(HANDLE)
*
*    PURPOSE: Initializes window data and registers window class
*
****************************************************************************/
BOOL InitApplication( HANDLE hInstance )
{
	WNDCLASS  wc;
	
	// Fill in window class structure with parameters that describe the
	// main (statistics) window. 
	wc.style			= 0;                     
	wc.lpfnWndProc		= (WNDPROC)MainWndProc; 
	wc.cbClsExtra		= 0;              
	wc.cbWndExtra		= 0;              
	wc.hInstance		= hInstance;       
	wc.hIcon			= LoadIcon( hInstance, TEXT("ICON") );
	wc.hCursor			= LoadCursor( NULL, IDC_ARROW );
	wc.hbrBackground	= GetStockObject( LTGRAY_BRUSH ); 
	wc.lpszMenuName		= TEXT("LISTMENU");  
	wc.lpszClassName	= TEXT("RegmonClass");
	if ( ! RegisterClass( &wc ) )
		return FALSE;

	wc.lpszMenuName	  = NULL;
 	wc.lpfnWndProc    = (WNDPROC) BalloonDialog;
	wc.hbrBackground  = CreateSolidBrush( 0x00D0FFFF );
	wc.lpszClassName  = "BALLOON";
	RegisterClass( &wc );
	
	return TRUE;
}


/****************************************************************************
*
*    FUNCTION:  InitInstance(HANDLE, int)
*
*    PURPOSE:  Saves instance handle and creates main window
*
****************************************************************************/
HWND InitInstance( HANDLE hInstance, int nCmdShow )
{
	HWND hWndMain;

	// get the window position settings from the registry
	Get_Position_Settings();

	hInst = hInstance;
	hWndMain = CreateWindow( TEXT("RegmonClass"), TEXT("NT Registry Monitor"), 
							WS_OVERLAPPEDWINDOW,
							PositionInfo.left, PositionInfo.top, 
							PositionInfo.width, PositionInfo.height,
							NULL, NULL, hInstance, NULL );

	// if window could not be created, return "failure" 
	if ( ! hWndMain )
		return NULL;
	
	// make the window visible; update its client area; and return "success"
	ShowWindow( hWndMain, nCmdShow );
	UpdateWindow( hWndMain ); 
	return hWndMain;      
}


/****************************************************************************
*
*	FUNCTION: WinMain(HANDLE, HANDLE, LPSTR, int)
*
*	PURPOSE:	calls initialization function, processes message loop
*
****************************************************************************/
int APIENTRY WinMain( HINSTANCE hInstance, HINSTANCE hPrevInstance,
						LPSTR lpCmdLine, int nCmdShow )
{
	MSG 	msg;      
	HWND	hWnd;
	HACCEL	hAccel;
 	DWORD	NTVersion;
        
	if ( ! InitApplication( hInstance ) )
		return FALSE;   
	
	// get NT version
	NTVersion = GetVersion();
	if( NTVersion >= 0x80000000 ) {
		MessageBox( NULL, TEXT("Not running on Windows NT. Visit http://www.ntinternals.com to get Regmon for Windows 95"),
			TEXT("NTRegmon"), MB_OK|MB_ICONERROR );
		return TRUE;
	}  

	// initializations that apply to a specific instance 
	if ( (hWnd = InitInstance( hInstance, nCmdShow )) == NULL )
		return FALSE;

	// load accelerators
	hAccel = LoadAccelerators( hInstance, TEXT("ACCELERATORS"));

	// register for the find window message
    findMessageID = RegisterWindowMessage( FINDMSGSTRING );

	// acquire and dispatch messages until a WM_QUIT message is received.
	while ( GetMessage( &msg, NULL, 0, 0 ) )  {
		if( !TranslateAccelerator( hWnd, hAccel, &msg ) &&
			(!hWndFind || !IsWindow(hWndFind) || !IsDialogMessage( hWndFind, &msg ))) {
			TranslateMessage( &msg );
			DispatchMessage( &msg ); 
		}
	}
	return msg.wParam;										 
}
