//
// File: MKADISK.C
//   By: Jamey Kirby (jkirby@mkallc.com)
// Date: July 1, 1998
//
// Confidential Information.
//
// Limited Distribution to Authorized Persons Only.
//
// Created 1998 and Protected as an Unpublished Work.
// Copyright (c) 1998 Magnuson, Kirby & Associates, LLC.
// All Rights Reserved.
//

#include <stdio.h>

#pragma warning (disable : 4115 4214 4514 4057)
#include <ntifs.h>
#include <ntdddisk.h>
#pragma warning (default : 4115 4214)

#include "mkadisk.h"

#ifdef ALLOC_PRAGMA
// Throw these functions away.
#pragma alloc_text(INIT, DriverEntry)
#pragma alloc_text(INIT, CreateDevice)
#pragma alloc_text(INIT, GetNumberOfVolumes)

// These all should be called at PASSIVE_LEVEL from the thread or
// from a use IOCTL. There may be others, but these we know for sure.
#pragma alloc_text(PAGE, IoThread)
#pragma alloc_text(PAGE, ReadBytes)
#pragma alloc_text(PAGE, WriteBytes)
#pragma alloc_text(PAGE, OpenMKADisk)
#pragma alloc_text(PAGE, CloseMKADisk)
#endif

// Global variables.
ULONG VolumeCount = 0;
PEPROCESS SystemProcess;

#if DBG
// Used to control debug output verbosity.
ULONG MKADiskDebugLevel =
(
    MKADISKERRORS +
    MKADISKDETAILED
);
#endif

// DriverEntry()
NTSTATUS DriverEntry
(
    IN OUT PDRIVER_OBJECT DriverObject,
    IN PUNICODE_STRING RegistryPath
)
{
    NTSTATUS ntStatus = STATUS_SUCCESS;
    WCHAR nameBuffer[256];
    UNICODE_STRING unicodeName;
    OBJECT_ATTRIBUTES objectAttributes;
    HANDLE objDirHandle;
    ULONG loop;
    ULONG numberOfVolumes = 0;

    MKADiskDump
    (
        MKADISKDETAILED,
        (
            "MKADISK, Virtual Disk Device Driver, (c) 1998 MKA, LLC.\n"
            "All rights reserved world wide.\n\n"
        )
    );

    // Initialize drivers dispatch table.
    DriverObject->MajorFunction[IRP_MJ_CREATE] = MKADiskCreate;
    DriverObject->MajorFunction[IRP_MJ_CLOSE] = MKADiskClose;
    DriverObject->MajorFunction[IRP_MJ_READ] = MKADiskReadWrite;
    DriverObject->MajorFunction[IRP_MJ_WRITE] = MKADiskReadWrite;
    DriverObject->MajorFunction[IRP_MJ_FLUSH_BUFFERS] = MKADiskFlushBuffers;
    DriverObject->MajorFunction[IRP_MJ_DEVICE_CONTROL] = MKADiskIoctl;

    // Get a pointer to the system process. DriverEntry() is
    // always called in the context of the system process.
    SystemProcess = IoGetCurrentProcess();

    // Get volume count.
    GetNumberOfVolumes(RegistryPath, &numberOfVolumes);

    // Create the object directory to hold the virtual volumes.
    swprintf
    (
        nameBuffer,
        L"\\Device" MKADISK_VOLUME_DIR
    );

    RtlInitUnicodeString(&unicodeName, nameBuffer);

    InitializeObjectAttributes
    (
        &objectAttributes,
        &unicodeName,
        OBJ_CASE_INSENSITIVE | OBJ_PERMANENT,
        NULL,
        NULL
    );

    ntStatus = ZwCreateDirectoryObject
    (
        &objDirHandle,
        DIRECTORY_ALL_ACCESS,
        &objectAttributes
    );

    if (NT_SUCCESS(ntStatus))
    {
        // Loop through and create the devices.
        for (loop = 0; loop < numberOfVolumes; loop++)
            if (NT_SUCCESS(CreateDevice(loop, DriverObject)))
                VolumeCount++;

        if (VolumeCount == 0)
        {
            ZwMakeTemporaryObject(objDirHandle);
            ntStatus = STATUS_NO_SUCH_DEVICE;
        }
        else
            ntStatus = STATUS_SUCCESS;

        ZwClose(objDirHandle);
    }

    return (ntStatus);
}

// GetNumberOfVolumes()
void GetNumberOfVolumes
(
    IN PUNICODE_STRING RegistryPath,
    PULONG NumberOfVolumes
)
{
    NTSTATUS ntStatus;
    UNICODE_STRING paramPath;
    WCHAR subKeyString[] = L"\\Parameters";
    RTL_QUERY_REGISTRY_TABLE paramTable[2];
    ULONG zero = 0;

    // Build path to "\parameters" key in the registry
    paramPath.Length = (USHORT)(RegistryPath->Length + sizeof (subKeyString));
    paramPath.Buffer = ExAllocatePool(PagedPool, paramPath.Length);
    paramPath.MaximumLength = (USHORT)(paramPath.Length);

    // Build path to registry key
    if (paramPath.Buffer != NULL)
    {
        RtlMoveMemory
        (
            paramPath.Buffer,
            RegistryPath->Buffer,
            RegistryPath->Length
        );

        RtlMoveMemory
        (
            &paramPath.Buffer[RegistryPath->Length / 2],
            subKeyString,
            sizeof (subKeyString)
        );

        // Get the number of volumes to support.
        RtlZeroMemory(paramTable, (sizeof (RTL_QUERY_REGISTRY_TABLE) * 2));
        paramTable[0].Flags = RTL_QUERY_REGISTRY_DIRECT;
        paramTable[0].Name = L"NumberOfVolumes";
        paramTable[0].EntryContext = NumberOfVolumes;
        paramTable[0].DefaultType = REG_DWORD;
        paramTable[0].DefaultData = &zero;
        paramTable[0].DefaultLength = sizeof (ULONG);

        ntStatus = RtlQueryRegistryValues
        (
            RTL_REGISTRY_ABSOLUTE | RTL_REGISTRY_OPTIONAL,
            paramPath.Buffer,
            paramTable,
            NULL,
            NULL
        );

        // If no registry value, create 1 device by default.
        if (!NT_SUCCESS(ntStatus))
            *NumberOfVolumes = 1;

        ExFreePool(paramPath.Buffer);
    }
    else
        *NumberOfVolumes = 1;
}

// CreateDevice()
NTSTATUS CreateDevice
(
    ULONG VolumeNumber,
    PDRIVER_OBJECT DriverObject
)
{
    NTSTATUS ntStatus = STATUS_SUCCESS;
    WCHAR nameBuffer[MAXIMUM_FILENAME_LENGTH];
    PWCHAR unicodeBuffer;
    UNICODE_STRING unicodeName;
    OBJECT_ATTRIBUTES objectAttributes;
    HANDLE thread;
    PDEVICE_OBJECT deviceObject;
    PMKADISK_DEVICE_EXTENSION deviceExtension;

    swprintf
    (
        nameBuffer,
        L"\\Device" MKADISK_VOLUME_DIR MKADISK_VOLUME_NAME L"%d",
        VolumeNumber
    );

    RtlInitUnicodeString
    (
        &unicodeName,
        nameBuffer
    );

    // Create the device object for the volume
    ntStatus = IoCreateDevice
    (
        DriverObject,
        sizeof (MKADISK_DEVICE_EXTENSION),
        &unicodeName,
        FILE_DEVICE_DISK,
#if MKADISK_REMOVABLE
        FILE_VIRTUAL_VOLUME | FILE_REMOVABLE_MEDIA,
#else
        FILE_VIRTUAL_VOLUME,
#endif
        FALSE,
        &deviceObject
    );

    if (!NT_SUCCESS(ntStatus))
    {
        MKADiskDump(MKADISKERRORS, ("MKADISK: Error creating device.\n"));
        goto CreateDeviceExit;
    }

    // Set device I/O method.
    deviceObject->Flags |= DO_DIRECT_IO;

    // Initialize the device extension.
    deviceExtension = (PMKADISK_DEVICE_EXTENSION)deviceObject->DeviceExtension;

    deviceExtension->DeviceName.Buffer = ExAllocatePool
    (
        PagedPool,
        unicodeName.Length
    );

    deviceExtension->DeviceName.MaximumLength = unicodeName.Length;
    deviceExtension->DeviceName.Length = 0;
    RtlCopyUnicodeString(&deviceExtension->DeviceName, &unicodeName);
    deviceExtension->DeviceObject = deviceObject;
    deviceExtension->DriverObject = DriverObject;
    deviceExtension->DeviceNumber = VolumeNumber;
    deviceExtension->TargetFilename.Length = 0;
    deviceExtension->TargetFilename.MaximumLength  = 0;
    deviceExtension->TargetFilename.Buffer = NULL;
    deviceExtension->FileSize.QuadPart = 0;
    deviceExtension->FileOpen = FALSE;
    deviceExtension->PartitionType = 0;

    // Initialize list spinlock.
    KeInitializeSpinLock(&deviceExtension->ListSpinLock);

    // Initialize the request queue.
    InitializeListHead(&deviceExtension->ListHead);

    // Initialize the request semaphore.
    KeInitializeSemaphore
    (
        &deviceExtension->RequestSemaphore,
        0L,
        MAXLONG
    );

    // Build the symbolic unicode string.
    unicodeBuffer = ExAllocatePool(PagedPool, 64);
    swprintf(unicodeBuffer, L"\\??" MKADISK_VOLUME_DIR L"%d", VolumeNumber);
    RtlInitUnicodeString(&deviceExtension->SymbolicName, unicodeBuffer);

    // Create symbolic link in the Win32 namespace.
    IoCreateSymbolicLink
    (
        &deviceExtension->SymbolicName,
        &deviceExtension->DeviceName
    );

    // Create and verify system thread for this device.
    ntStatus = PsCreateSystemThread
    (
        &thread,
        (ACCESS_MASK)0L,
        NULL,
        (HANDLE)0L,
        NULL,
        IoThread,
        deviceExtension
    );

    if (!NT_SUCCESS(ntStatus))
    {
        MKADiskDump(MKADISKERRORS, ("MKADISK: Unable to create system thread\n"));
        IoDeleteDevice(deviceObject);
        goto CreateDeviceExit;
    }

    // Get a pointer to the thread object.
    ObReferenceObjectByHandle
    (
        thread,
        THREAD_ALL_ACCESS,
        NULL,
        KernelMode,
        &deviceExtension->ThreadObject,
        NULL
    );

    // Don't need this anymore.
    ZwClose(thread);

CreateDeviceExit:

    return (ntStatus);
}

// IoThread()
VOID IoThread
(
    PVOID Context
)
{
    NTSTATUS ntStatus;
    PDEVICE_OBJECT deviceObject;
    PIRP irp;
    PIO_STACK_LOCATION irpSp;
    PLIST_ENTRY request;
    PMKADISK_DEVICE_EXTENSION deviceExtension = Context;

    // Set the threads default priority.
    KeSetPriorityThread(KeGetCurrentThread(), LOW_REALTIME_PRIORITY);

    // Main request loop.
    do
    {
        MKADiskDump(MKADISKDETAILED, ("MKADISK: IoThread() about to wait.\n"));

        // Wait for the request semaphore to get released.
        KeWaitForSingleObject
        (
            &deviceExtension->RequestSemaphore,
            Executive,
            KernelMode,
            FALSE,
            NULL
        );

        // Process request.
        while
        (
            (
                // Remove request from the list.
                request = ExInterlockedRemoveHeadList
                (
                    &deviceExtension->ListHead,
                    &deviceExtension->ListSpinLock
                )
            ) != NULL
        )
        {
            MKADiskDump(MKADISKDETAILED, ("MKADISK: Processing request.\n"));

            // Get pointers to the IRP and its stack location.
            irp = CONTAINING_RECORD(request, IRP, Tail.Overlay.ListEntry);
            irpSp = IoGetCurrentIrpStackLocation(irp);

            if (irpSp->MajorFunction == IRP_MJ_READ)
            {
                // Read
                irp->IoStatus.Status = ReadBytes
                (
                    deviceExtension,
                    irpSp->Parameters.Read.ByteOffset,
                    irpSp->Parameters.Read.Length,
                    MmGetSystemAddressForMdl(irp->MdlAddress),
                    (BOOLEAN)(irp->Flags & IRP_NOCACHE ? FALSE : TRUE)
                );

                irp->IoStatus.Information = irpSp->Parameters.Read.Length;
            }
            else
            {
                // Write
                irp->IoStatus.Status = WriteBytes
                (
                    deviceExtension,
                    irpSp->Parameters.Write.ByteOffset,
                    irpSp->Parameters.Write.Length,
                    MmGetSystemAddressForMdl(irp->MdlAddress),
                    (BOOLEAN)(irp->Flags & IRP_NOCACHE ? FALSE : TRUE)
                );

                irp->IoStatus.Information = irpSp->Parameters.Write.Length;
            }

            IoCompleteRequest
            (
                irp,
                (UCHAR)
                (
                    NT_SUCCESS(irp->IoStatus.Status) ?
                        IO_DISK_INCREMENT : IO_NO_INCREMENT
                )
            );
        }
    }
    while (TRUE);
}

// ReadBytes()
NTSTATUS ReadBytes
(
    PMKADISK_DEVICE_EXTENSION DeviceExtension,
    LARGE_INTEGER ByteOffset,
    ULONG ByteCount,
    PUCHAR Buffer,
    BOOLEAN Cache
)
{
    IO_STATUS_BLOCK ioStatus;

    MKADiskDump
    (
        MKADISKDETAILED,
        (
            "MKADISK: Reading [%x] blocks from block [%x].\n",
            (ULONG)(ByteCount / MKADISK_DISK_SECTOR_SIZE),
            (ULONG)(ByteOffset.QuadPart / MKADISK_DISK_SECTOR_SIZE)
        )
    );

    // Read the requested data from the volume file.
    return
    (
        ZwReadFile
        (
            DeviceExtension->SystemTargetFileHandle,
            NULL,
            NULL,
            NULL,
            &ioStatus,
            Buffer,
            ByteCount,
            &ByteOffset,
            NULL
        )
    );
}

// WriteBytes()
NTSTATUS WriteBytes
(
    PMKADISK_DEVICE_EXTENSION DeviceExtension,
    LARGE_INTEGER ByteOffset,
    ULONG ByteCount,
    PUCHAR Buffer,
    BOOLEAN Cache
)
{
    IO_STATUS_BLOCK ioStatus;

    MKADiskDump
    (
        MKADISKDETAILED,
        (
            "MKADISK: Writing [%x] blocks to block [%x].\n",
            (ULONG)(ByteCount / MKADISK_DISK_SECTOR_SIZE),
            (ULONG)(ByteOffset.QuadPart / MKADISK_DISK_SECTOR_SIZE)
        )
    );

    // Write the requested data to the volume file.
    return
    (
        ZwWriteFile
        (
            DeviceExtension->SystemTargetFileHandle,
            NULL,
            NULL,
            NULL,
            &ioStatus,
            Buffer,
            ByteCount,
            &ByteOffset,
            NULL
        )
    );
}

// OpenMKADisk()
NTSTATUS OpenMKADisk
(
    PMKADISK_DEVICE_EXTENSION DeviceExtension
)
{
    NTSTATUS ntStatus = STATUS_SUCCESS;
    IO_STATUS_BLOCK ioStatus;
    OBJECT_ATTRIBUTES objectAttributes;
    FILE_STANDARD_INFORMATION fileInfo;
    HANDLE fileHandle;
    PVOID fileObject;

    // Make sure we're ready.
    if (DeviceExtension->FileOpen == TRUE)
    {
        MKADiskDump(MKADISKERRORS, ("MKADISK: Media already inserted.\n"));
        ntStatus = STATUS_INVALID_DEVICE_REQUEST;
        goto OpenMKADiskExit;
    }

    // Assume failure
    DeviceExtension->FileOpen = FALSE;

    // Initialize the files object attributes.
    InitializeObjectAttributes
    (
        &objectAttributes,
        &DeviceExtension->TargetFilename,
        OBJ_CASE_INSENSITIVE,
        NULL,
        NULL
    );

    // Open the file for random access in the context of the calling
    // process. This should be a service with the appropriate access
    // rights to control the device.
    ntStatus = ZwCreateFile
    (
        &fileHandle,
        GENERIC_WRITE | GENERIC_READ,
        &objectAttributes,
        &ioStatus,
        NULL,
        FILE_ATTRIBUTE_NORMAL,
        FILE_SHARE_READ | FILE_SHARE_WRITE,
        FILE_OPEN,
        (
            FILE_NO_INTERMEDIATE_BUFFERING |
            FILE_SYNCHRONOUS_IO_NONALERT |
            FILE_RANDOM_ACCESS
        ),
        NULL,
        0
    );

    if (!NT_SUCCESS(ntStatus))
        goto OpenMKADiskExit;

    // Get file size information.
    ntStatus = ZwQueryInformationFile
    (
        fileHandle,
        &ioStatus,
        &fileInfo,
        sizeof (FILE_STANDARD_INFORMATION),
        FileStandardInformation
    );

    if (!NT_SUCCESS(ntStatus))
    {
        ZwClose(fileHandle);
        goto OpenMKADiskExit;
    }

    // Initialize disk geometry.
#if MKADISK_REMOVABLE
    DeviceExtension->Geometry.MediaType = RemovableMedia;
#else
    DeviceExtension->Geometry.MediaType = FixedMedia;
#endif

    DeviceExtension->FileSize.QuadPart = fileInfo.AllocationSize.QuadPart;
    DeviceExtension->Geometry.BytesPerSector = MKADISK_DISK_SECTOR_SIZE;
    DeviceExtension->Geometry.SectorsPerTrack = MKADISK_SECTORS_PER_TRACK;
    DeviceExtension->Geometry.TracksPerCylinder = MKADISK_TRACKS_PER_CYLINDER;

    // Calculate cylinder count.
    DeviceExtension->Geometry.Cylinders.QuadPart =
        (
            DeviceExtension->FileSize.QuadPart /
                MKADISK_DISK_SECTOR_SIZE
        ) /
        (
            DeviceExtension->Geometry.TracksPerCylinder *
                DeviceExtension->Geometry.SectorsPerTrack
        );

    // At this point, we could do I/O in the context of the calling process.

    // Get an object pointer to the file (FILE_OBJECT).
    ntStatus = ObReferenceObjectByHandle
    (
        fileHandle,
        FILE_READ_DATA | FILE_WRITE_DATA | FILE_READ_ATTRIBUTES,
        NULL,
        KernelMode,
        &fileObject,
        NULL
    );

    if (!NT_SUCCESS(ntStatus))
    {
        MKADiskDump(MKADISKERRORS, ("MKADISK: Error getting file object.\n"));
        ZwClose(fileHandle);
        goto OpenMKADiskExit;
    }

    // Attach to the system process.
    KeAttachProcess(SystemProcess);

    // Create a handle in the system process to the file we have opened.
    ntStatus = ObOpenObjectByPointer
    (
        fileObject,
        objectAttributes.Attributes,
        NULL,
        FILE_READ_DATA | FILE_WRITE_DATA | FILE_READ_ATTRIBUTES,
        NULL,
        KernelMode,
        &DeviceExtension->SystemTargetFileHandle
    );

    // Detach from the system process.
    KeDetachProcess();

    // Dereference the file object and close the file handle for
    // this process.
    ObDereferenceObject(fileObject);
    ZwClose(fileHandle);

    if (!NT_SUCCESS(ntStatus))
        MKADiskDump(MKADISKERRORS, ("MKADISK: Error getting file handle.\n"));
    else
        DeviceExtension->FileOpen = TRUE;

OpenMKADiskExit:

    return (ntStatus);
}

// CloseMKADisk()
NTSTATUS CloseMKADisk
(
    PMKADISK_DEVICE_EXTENSION DeviceExtension
)
{
    NTSTATUS ntStatus = STATUS_SUCCESS;

    if (DeviceExtension->FileOpen == TRUE)
    {
        KeAttachProcess(SystemProcess);
        DeviceExtension->FileOpen = FALSE;
        ZwClose(DeviceExtension->SystemTargetFileHandle);
        ExFreePool(DeviceExtension->TargetFilename.Buffer);
        DeviceExtension->TargetFilename.Buffer = NULL;
        KeDetachProcess();
    }
    else
        ntStatus = STATUS_NO_MEDIA_IN_DEVICE;

    return (ntStatus);
}

// MKADiskCreate()
NTSTATUS MKADiskCreate
(
    IN PDEVICE_OBJECT DeviceObject,
    IN PIRP Irp
)
{
    MKADiskDump(MKADISKDETAILED, ("MKADISK: IRP_MJ_CREATE.\n"));
    Irp->IoStatus.Status = STATUS_SUCCESS;
    Irp->IoStatus.Information = FILE_OPENED;
    IoCompleteRequest(Irp, IO_NO_INCREMENT);

    return (STATUS_SUCCESS);
}

// MKADiskClose()
NTSTATUS MKADiskClose
(
    IN PDEVICE_OBJECT DeviceObject,
    IN PIRP Irp
)
{
    MKADiskDump(MKADISKDETAILED, ("MKADISK: IRP_MJ_CLOSE.\n"));
    Irp->IoStatus.Status = STATUS_SUCCESS;
    Irp->IoStatus.Information = 0;
    IoCompleteRequest(Irp, IO_NO_INCREMENT);

    return (STATUS_SUCCESS);
}

// MKADiskReadWrite()
NTSTATUS MKADiskReadWrite
(
    IN PDEVICE_OBJECT DeviceObject,
    IN PIRP Irp
)
{
    NTSTATUS ntStatus = STATUS_SUCCESS;
    PIO_STACK_LOCATION irpSp;
    PMKADISK_DEVICE_EXTENSION deviceExtension;
    ULONG length;

    MKADiskDump(MKADISKDETAILED, ("MKADISK: IRP_MJ_READ/WRITE.\n"));
    irpSp = IoGetCurrentIrpStackLocation(Irp);
    deviceExtension = DeviceObject->DeviceExtension;

    // Check if device is ready. If not, exit with an error.
    if (deviceExtension->FileOpen == FALSE)
    {
        MKADiskDump(MKADISKERRORS, ("MKADISK: Drive not ready.\n"));
        ntStatus = Irp->IoStatus.Status = STATUS_NO_MEDIA_IN_DEVICE;
        Irp->IoStatus.Information = 0;
        IoCompleteRequest(Irp, IO_NO_INCREMENT);
        goto MKADiskReadExit;
    }

    // Set pointers to the key and length values for the request.
    if (irpSp->MajorFunction == IRP_MJ_READ)
        length = irpSp->Parameters.Read.Length;
    else
        length = irpSp->Parameters.Write.Length;

    // Check 0 byte transfer. We will do the 0 check before the
    // alignment check so that on 0 byte transfers, we eliminate having
    // to do the alignment check. A 0 byte transfer will always pass
    // the alignment test [just some more trivial optimizations].
    if (length == 0)
    {
        MKADiskDump(MKADISKDETAILED, ("MKADISK: Zero byte transfer.\n"));
        Irp->IoStatus.Information = 0;
        ntStatus = Irp->IoStatus.Status = STATUS_SUCCESS;
        IoCompleteRequest(Irp, IO_NO_INCREMENT);
        goto MKADiskReadExit;
    }

    // Validate the alignment
    if (length % deviceExtension->Geometry.BytesPerSector)
    {
        MKADiskDump(MKADISKERRORS, ("MKADISK: Invalid buffer alignment.\n"));
        Irp->IoStatus.Information = 0;
        ntStatus = Irp->IoStatus.Status = STATUS_INVALID_PARAMETER;
        IoCompleteRequest(Irp, IO_NO_INCREMENT);
        goto MKADiskReadExit;
    }

    // Mark the IRP pending. This information must be marked before adding
    // the request to the queue and before releasing the semaphore. Otherwise
    // there is a chance that the request will be completed by the thread
    // before the IRP has been marked. I have seen this happen!
    IoMarkIrpPending(Irp);
    ntStatus = Irp->IoStatus.Status = STATUS_PENDING;
    Irp->IoStatus.Information = 0;

    // Queue the request.
    ExInterlockedInsertTailList
    (
        &deviceExtension->ListHead,
        &Irp->Tail.Overlay.ListEntry,
        &deviceExtension->ListSpinLock
    );

    // Release the request semaphore to start the thread.
    KeReleaseSemaphore
    (
        &deviceExtension->RequestSemaphore,
        (KPRIORITY)0,
        1,
        FALSE
    );

MKADiskReadExit:

    return (ntStatus);
}

// MKADiskFlushBuffers()
NTSTATUS MKADiskFlushBuffers
(
    IN PDEVICE_OBJECT DeviceObject,
    IN PIRP Irp
)
{
    MKADiskDump(MKADISKDETAILED, ("MKADISK: IRP_MJ_FLUSH_BUFFERS.\n"));
    Irp->IoStatus.Status = STATUS_SUCCESS;
    Irp->IoStatus.Information = 0;
    IoCompleteRequest(Irp, IO_NO_INCREMENT);

    return (STATUS_SUCCESS);
}

// MKADiskIoctl()
NTSTATUS MKADiskIoctl
(
    IN PDEVICE_OBJECT DeviceObject,
    IN PIRP Irp
)
{
    NTSTATUS ntStatus = STATUS_SUCCESS;
    PVOID verifyBuffer;
    PMKADISK_DEVICE_EXTENSION deviceExtension;
    PIO_STACK_LOCATION irpSp;
    PPREVENT_MEDIA_REMOVAL mediaRemoval;
    PULONG mediaChangeCount;
    SET_PARTITION_INFORMATION *partitionSetInfo;
    PVERIFY_INFORMATION verifyInformation;
    PPARTITION_INFORMATION partitionInfo;
    PMKADISK_INSERT_IOCTL vDiskInsertParam;
    STRING tempName;
    PDRIVE_LAYOUT_INFORMATION driveLayout;

    deviceExtension = DeviceObject->DeviceExtension;
    irpSp = IoGetCurrentIrpStackLocation(Irp);

    if
    (
        irpSp->Parameters.DeviceIoControl.IoControlCode !=
            IOCTL_MKADISK_INSERT
    )
        // Check for drive not ready.
        if (deviceExtension->FileOpen == FALSE)
        {
            MKADiskDump(MKADISKERRORS, ("MKADISK: Drive not ready.\n"));
            ntStatus = STATUS_NO_MEDIA_IN_DEVICE;
            Irp->IoStatus.Information = 0;
            goto MKADiskIoctlExit;
        }

    switch (irpSp->Parameters.DeviceIoControl.IoControlCode)
    {
        case IOCTL_MKADISK_INSERT:
            MKADiskDump(MKADISKDETAILED, ("MKADISK: IOCTL_MKADISK_INSERT.\n"));

            // Check Parameter Size
            if
            (
               irpSp->Parameters.DeviceIoControl.InputBufferLength <
                    sizeof (MKADISK_INSERT_IOCTL)
            )
            {
                Irp->IoStatus.Information = 0;
                ntStatus = STATUS_INVALID_PARAMETER;
                break;
            }

            vDiskInsertParam = Irp->AssociatedIrp.SystemBuffer;

            MKADiskDump
            (
                MKADISKDETAILED,
                ("MKADISK: Inserting [%s]\n", vDiskInsertParam->VolumeName)
            );

            // Build Unicode filename string for the container file.
            RtlInitString(&tempName, vDiskInsertParam->VolumeName);

            RtlAnsiStringToUnicodeString
            (
                &deviceExtension->TargetFilename,
                &tempName,
                TRUE
            );

            ntStatus = OpenMKADisk(deviceExtension);

            if (!NT_SUCCESS(ntStatus))
            {
                MKADiskDump(MKADISKERRORS, ("MKADISK: Insert failed.\n"));
                RtlFreeUnicodeString(&deviceExtension->TargetFilename);
                Irp->IoStatus.Information = 0;
                break;
            }

            Irp->IoStatus.Information = 0;
            break;

        case IOCTL_MKADISK_REMOVE:
#if MKADISK_REMOVABLE
        case IOCTL_DISK_EJECT_MEDIA:
#endif
            MKADiskDump
            (
                MKADISKDETAILED,
                ("MKADISK: IOCTL_MKADISK_REMOVE/IOCTL_DISK_EJECT_MEDIA.\n")
            );
            ntStatus = CloseMKADisk(deviceExtension);
            Irp->IoStatus.Information = 0;
            break;


        case IOCTL_DISK_GET_DRIVE_GEOMETRY:
            MKADiskDump
            (
                MKADISKDETAILED,
                ("MKADISK: IOCTL_DISK_GET_DRIVE_GEOMETRY.\n")
            );

            // Check buffer length.
            if
            (
                irpSp->Parameters.DeviceIoControl.OutputBufferLength <
                    sizeof (DISK_GEOMETRY)
            )

            {
                ntStatus = STATUS_INFO_LENGTH_MISMATCH;
                Irp->IoStatus.Information =0;
                break;
            }

            // Copy our geometry structure that is stored in the
            // device extension.
            RtlCopyMemory
            (
                Irp->AssociatedIrp.SystemBuffer,
                &deviceExtension->Geometry,
                sizeof (DISK_GEOMETRY)
            );

            ntStatus = STATUS_SUCCESS;
            Irp->IoStatus.Information = sizeof (DISK_GEOMETRY);

            break;

        case IOCTL_DISK_GET_PARTITION_INFO:
            MKADiskDump
            (
                MKADISKDETAILED,
                ("MKADISK: IOCTL_DISK_GET_PARTITION_INFO.\n")
            );

            // Check buffer length.
            if
            (
                irpSp->Parameters.DeviceIoControl.OutputBufferLength <
                    sizeof (PARTITION_INFORMATION)
            )
            {
                ntStatus = STATUS_INFO_LENGTH_MISMATCH;
                Irp->IoStatus.Information = 0;
                break;
            }

            //Copy our partion information in the return buffer.
            partitionInfo = Irp->AssociatedIrp.SystemBuffer;
            partitionInfo->PartitionType = deviceExtension->PartitionType;
            partitionInfo->StartingOffset.QuadPart = 0;
            partitionInfo->HiddenSectors = 0;
            partitionInfo->PartitionNumber = 0;
            partitionInfo->BootIndicator = FALSE;
            partitionInfo->RewritePartition = FALSE;
            partitionInfo->PartitionLength = deviceExtension->FileSize;

            partitionInfo->RecognizedPartition =
                (UCHAR)IsRecognizedPartition(deviceExtension->PartitionType);

            ntStatus = STATUS_SUCCESS;
            Irp->IoStatus.Information = sizeof (PARTITION_INFORMATION);
            break;

        case IOCTL_DISK_GET_DRIVE_LAYOUT:
            MKADiskDump
            (
                MKADISKDETAILED,
                ("MKADISK: IOCTL_DISK_GET_DRIVE_LAYOUT.\n")
            );

            // Check buffer length.
            if
            (
                irpSp->Parameters.DeviceIoControl.OutputBufferLength <
                    sizeof (DRIVE_LAYOUT_INFORMATION)
            )
            {
                ntStatus = STATUS_INFO_LENGTH_MISMATCH;
                Irp->IoStatus.Information = 0;
                break;
            }

            driveLayout = Irp->AssociatedIrp.SystemBuffer;

            // Fake one partition.
            driveLayout->PartitionCount = 1;
            driveLayout->Signature = 0;

            // Build up the fake partition data
            driveLayout->PartitionEntry[0].StartingOffset.QuadPart = 0;
            driveLayout->PartitionEntry[0].PartitionNumber = 0;
            driveLayout->PartitionEntry[0].BootIndicator = FALSE;
            driveLayout->PartitionEntry[0].RewritePartition = FALSE;
            driveLayout->PartitionEntry[0].HiddenSectors = 0;

            driveLayout->PartitionEntry[0].PartitionLength =
                deviceExtension->FileSize;

            driveLayout->PartitionEntry[0].PartitionType =
                deviceExtension->PartitionType;

            driveLayout->PartitionEntry[0].RecognizedPartition =
                (UCHAR)IsRecognizedPartition(deviceExtension->PartitionType);

            ntStatus = STATUS_SUCCESS;
            Irp->IoStatus.Information = sizeof (DRIVE_LAYOUT_INFORMATION);
            break;

        case IOCTL_DISK_VERIFY:
            MKADiskDump(MKADISKDETAILED, ("MKADISK: IOCTL_DISK_VERIFY.\n"));
            verifyInformation = Irp->AssociatedIrp.SystemBuffer;

            // Allocate buffer for read data.
            verifyBuffer = ExAllocatePool
            (
                PagedPool,
                verifyInformation->Length
            );

            if (verifyBuffer == NULL)
            {
                // Not really an error, just can't get memory to read.
                Irp->IoStatus.Information = verifyInformation->Length;
                ntStatus = STATUS_SUCCESS;
            }
            else
            {
                KeAttachProcess(SystemProcess);

                ntStatus = ReadBytes
                (
                    deviceExtension,
                    verifyInformation->StartingOffset,
                    verifyInformation->Length,
                    verifyBuffer,
                    FALSE
                );

                KeDetachProcess();
                Irp->IoStatus.Information = verifyInformation->Length;
                ExFreePool(verifyBuffer);
            }

            break;

        case IOCTL_DISK_IS_WRITABLE:
            MKADiskDump(MKADISKDETAILED, ("MKADISK: IOCTL_DISK_IS_WRITABLE.\n"));

            if (deviceExtension->WriteProtected == TRUE)
                ntStatus = STATUS_MEDIA_WRITE_PROTECTED;
            else
                ntStatus = STATUS_SUCCESS;

            Irp->IoStatus.Information = 0;
            break;

#if MKADISK_REMOVABLE
        case IOCTL_DISK_MEDIA_REMOVAL:
            MKADiskDump
            (
                MKADISKDETAILED,
                ("MKADISK: IOCTL_DISK_MEDIA_REMOVABLE.\n")
            );

            // Check buffer length.
            if
            (
                irpSp->Parameters.DeviceIoControl.InputBufferLength <
                    sizeof (PREVENT_MEDIA_REMOVAL)
            )
            {
                ntStatus = STATUS_BUFFER_TOO_SMALL;
                Irp->IoStatus.Information = 0;
                break;
            }

            mediaRemoval = Irp->AssociatedIrp.SystemBuffer;

            if (mediaRemoval->PreventMediaRemoval == TRUE)
            {
                MKADiskDump(MKADISKDETAILED, ("MKADISK: Media lock.\n"));
                InterlockedIncrement(&deviceExtension->LockCount);
            }
            else
            {
                MKADiskDump(MKADISKDETAILED, ("MKADISK: Media unlock.\n"));
                InterlockedDecrement(&deviceExtension->LockCount);
            }

            ntStatus = STATUS_SUCCESS;
            Irp->IoStatus.Information = 0;
            break;
#endif

        case IOCTL_DISK_INTERNAL_SET_VERIFY:
            MKADiskDump
            (
                MKADISKDETAILED,
                ("MKADISK: IOCTL_DISK_INTERNAL_SET_VERIFY.\n")
            );

            // Should only be done by other kernel-mode drivers
            if (Irp->RequestorMode == KernelMode)
                DeviceObject->Flags |= DO_VERIFY_VOLUME;

            Irp->IoStatus.Information = 0;
            ntStatus = STATUS_SUCCESS;
            break;

        case IOCTL_DISK_INTERNAL_CLEAR_VERIFY:
            MKADiskDump
            (
                MKADISKDETAILED,
                ("MKADISK: IOCTL_DISK_INTERNAL_CLEAR_VERIFY.\n")
            );

            // Should only be done by other kernel-mode drivers
            if (Irp->RequestorMode == KernelMode)
                DeviceObject->Flags &= ~DO_VERIFY_VOLUME;

            Irp->IoStatus.Information = 0;
            ntStatus = STATUS_SUCCESS;
            break;

        case IOCTL_DISK_CHECK_VERIFY:
            MKADiskDump(MKADISKDETAILED, ("MKADISK: IOCTL_DISK_CHECK_VERIFY.\n"));

            // Media has changed.
            if (deviceExtension->MediaChangeCount != 0)
            {
                MKADiskDump(MKADISKDETAILED, ("MKADISK: Media changed\n"));
                DeviceObject->Flags |= DO_VERIFY_VOLUME;
                deviceExtension->MediaChangeCount = 0;
                Irp->IoStatus.Information = 0;
                ntStatus = STATUS_VERIFY_REQUIRED;
            }
            else
            {
                MKADiskDump(MKADISKDETAILED, ("MKADISK: Media NOT changed\n"));

                if
                (
                    (
                        Irp->IoStatus.Information =
                            irpSp->Parameters.
                                DeviceIoControl.OutputBufferLength
                    ) != 0
                )
                {
                    // Return the media change count.
                    mediaChangeCount = Irp->AssociatedIrp.SystemBuffer;
                    *mediaChangeCount = deviceExtension->MediaChangeCount;
                }

                ntStatus = STATUS_SUCCESS;
            }

            break;

        case IOCTL_DISK_SET_PARTITION_INFO:
            MKADiskDump
            (
                MKADISKDETAILED,
                ("MKADISK: IOCTL_DISK_SET_PARTITION_INFO.\n")
            );

            // Ensure enough data in buffer.
            if
            (
                irpSp->Parameters.DeviceIoControl.InputBufferLength <
                    sizeof (SET_PARTITION_INFORMATION)
            )
            {
                ntStatus = STATUS_INFO_LENGTH_MISMATCH;
                Irp->IoStatus.Information = 0;
                break;
            }

            partitionSetInfo = Irp->AssociatedIrp.SystemBuffer;
            deviceExtension->PartitionType = partitionSetInfo->PartitionType;
            Irp->IoStatus.Information = 0;
            break;

        default:
            MKADiskDump
            (
                MKADISKERRORS,
                (
                    "MKADISK: IOCTL %x not supported.\n",
                    IoGetFunctionCodeFromCtlCode
                    (
                        irpSp->Parameters.DeviceIoControl.IoControlCode
                    )
                )
            );

            ntStatus = STATUS_INVALID_DEVICE_REQUEST;
            Irp->IoStatus.Information = 0;
            break;
    }

MKADiskIoctlExit:

    Irp->IoStatus.Status = ntStatus;
    IoCompleteRequest(Irp, IO_NO_INCREMENT);

    return (ntStatus);
}

