//
// File: MKADISK.H
//   By: Jamey Kirby (jkirby@mkallc.com)
// Date: July 1, 1998
//
// Confidential Information.
//
// Limited Distribution to Authorized Persons Only.
//
// Created 1998 and Protected as an Unpublished Work.
// Copyright (c) 1998 Magnuson, Kirby & Associates, LLC.
// All Rights Reserved.
//

#ifndef MKADISK_H
#define MKADISK_H

#define MKADISK_REMOVABLE 0

// Macro, variable, and defines to control verbosity of debug output
#if DBG
extern ULONG MKADiskDebugLevel;
#endif

#if DBG
#define MKADISKBUGCHECK     ((ULONG)0x80000000)
#define MKADISKERRORS       ((ULONG)0x00000001)
#define MKADISKDETAILED     ((ULONG)0x00000002)

#define MKADiskDump(LEVEL, STRING)      \
    do                                  \
    {                                   \
        if (MKADiskDebugLevel & LEVEL)  \
            DbgPrint STRING;            \
                                        \
        if (LEVEL == MKADISKBUGCHECK)   \
            ASSERT(FALSE);              \
    }                                   \
    while (0)
#else
#define MKADiskDump(LEVEL, STRING) do {NOTHING;} while (0)
#endif

#define IOCTL_MKADISK_BASE   FILE_DEVICE_DISK
#define IOCTL_MKADISK_INSERT CTL_CODE(IOCTL_MKADISK_BASE, 0x800, METHOD_BUFFERED, FILE_ANY_ACCESS)
#define IOCTL_MKADISK_REMOVE CTL_CODE(IOCTL_MKADISK_BASE, 0x801, METHOD_NEITHER, FILE_ANY_ACCESS)

// Used to define the object name space for our devices
#define MKADISK_VOLUME_NAME L"\\Volume"
#define MKADISK_VOLUME_DIR  L"\\MKAdisk"

// Some defines that are relative to blocking. We align things on
// page and transfer boundries in hopes that this will assist the
// cache manager in doing I/O. In other words, lets try and reduce
// the cache managers code path. This will also be useful if we decide
// to switch to a mapped section object.
#define MKADISK_DISK_SECTOR_SIZE  512
#define MKADISK_SECTORS_PER_TRACK (PAGE_SIZE / MKADISK_DISK_SECTOR_SIZE)

#define MKADISK_TRACKS_PER_CYLINDER                     \
(                                                       \
   MM_MAXIMUM_DISK_IO_SIZE / MKADISK_SECTORS_PER_TRACK  \
)

// Parameter block for ioctl insert requests.
typedef struct _MKADISK_INSERT_IOCTL
{
    UCHAR VolumeName[256];
    UCHAR Key[256];
}
MKADISK_INSERT_IOCTL, *PMKADISK_INSERT_IOCTL;

// MKADISK device extension.
typedef struct _MKADISK_DEVICE_EXTENSION
{
    // Basic driver and device information
    PDRIVER_OBJECT DriverObject;
    PDEVICE_OBJECT DeviceObject;
    UNICODE_STRING DeviceName;
    UNICODE_STRING SymbolicName;
    ULONG DeviceNumber;

    // Media managment
    UNICODE_STRING TargetFilename;
    HANDLE SystemTargetFileHandle;
    LARGE_INTEGER FileSize;
    BOOLEAN FileOpen;
    BOOLEAN WriteProtected;
#if MKADISK_REMOVABLE
    LONG LockCount;
#endif
    ULONG MediaChangeCount;

    // Geometry information
    DISK_GEOMETRY Geometry;
    UCHAR PartitionType;

    // Queue and thread managment
    PVOID ThreadObject;
    KSEMAPHORE RequestSemaphore;
    LIST_ENTRY ListHead;
    KSPIN_LOCK ListSpinLock;
}
MKADISK_DEVICE_EXTENSION, *PMKADISK_DEVICE_EXTENSION;

// Initilization routines.
NTSTATUS DriverEntry
(
    IN OUT PDRIVER_OBJECT DriverObject,
    IN PUNICODE_STRING RegistryPath
);

VOID GetNumberOfVolumes
(
    IN PUNICODE_STRING RegistryPath,
    PULONG NumberOfVolumes
);

NTSTATUS CreateDevice
(
    ULONG DiskDevice,
    PDRIVER_OBJECT DriverObject
);

// Internal routines
VOID IoThread
(
    PVOID Context
);

NTSTATUS ReadBytes
(
    PMKADISK_DEVICE_EXTENSION DeviceExtension,
    IN LARGE_INTEGER StartByte,
    IN ULONG ByteCount,
    IN OUT PVOID Buffer,
    IN BOOLEAN Cache
);

NTSTATUS WriteBytes
(
    PMKADISK_DEVICE_EXTENSION DeviceExtension,
    IN LARGE_INTEGER StartByte,
    IN ULONG ByteCount,
    IN OUT PVOID Buffer,
    IN BOOLEAN Cache
);

NTSTATUS OpenMKADisk
(
    PMKADISK_DEVICE_EXTENSION DeviceExtension
);

NTSTATUS CloseMKADisk
(
    PMKADISK_DEVICE_EXTENSION DeviceExtension
);

// Dispatch routines
NTSTATUS MKADiskCreate
(
    IN PDEVICE_OBJECT DeviceObject,
    IN PIRP Irp
);

NTSTATUS MKADiskClose
(
    IN PDEVICE_OBJECT DeviceObject,
    IN PIRP Irp
);

NTSTATUS MKADiskReadWrite
(
    IN PDEVICE_OBJECT DeviceObject,
    IN PIRP Irp
);

NTSTATUS MKADiskFlushBuffers
(
    IN PDEVICE_OBJECT DeviceObject,
    IN PIRP Irp
);

NTSTATUS MKADiskIoctl
(
    IN PDEVICE_OBJECT DeviceObject,
    IN PIRP Irp
);

#endif


